//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
// $Id: G4AtomicShells.hh 94016 2015-11-05 10:14:49Z gcosmo $

// class description
//
// Class consists of atomic subshell binding energies for first 100 elements.
//
// The data were given from:
//
// I	from	Janni J.F. 1982 (I for F from estimation with formula).
//
// I(i)	from	T.A.Carlson, Photoelectron and Auger Spectroscopy,
//		Plenum, New York, 1976, p.337.
//
//		a) CRC, Handbook of Chemistry and Physics, 73rd Edition,
//	        	1992-1993

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....

// 24-04-98, implementation of the first version, V. Grichine
// 16-11-98, GetBindingEnergy(Z,ShellNb), M.Maire
// 11-02-05, GetNumberOfElectrons(Z,ShellNb), V.Ivanchenko
// 30-04-10, added fIndexOfShells, V.Ivanchenko

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....

#ifndef G4ATOMICSHELLS_H
#define G4ATOMICSHELLS_H


#include "globals.hh"
#include "templates.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....

class G4AtomicShells 
{
public :  //with description

static G4int    GetNumberOfShells(G4int Z);
static G4int    GetNumberOfElectrons(G4int Z, G4int SubshellNb);
static G4double GetBindingEnergy (G4int Z, G4int SubshellNb);
       G4double GetTotalBindingEnergy (G4int Z);

private :

static const G4int    fNumberOfShells[101];
static const G4int    fIndexOfShells[101];
static const G4int    fNumberOfElectrons[1540];
static const G4double fBindingEnergies[1540];

};


/*   *********************************************************
Below is the atomic subshell binding energytable for elements Z=1-100.
The data were given from:

I	from	Janni J.F. 1982 (I for F from estimation with formula).

I(i)	from	T.A.Carlson, Photoelectron and Auger Spectroscopy,
		Plenum, New York, 1976, p.337.

		a) CRC, Handbook of Chemistry and Physics, 73rd Edition,
	        	1992-1993


	  Free Atom Subshells Binding Energies - I(i)
	----------------------------------------------

El      A  Z     I    err I     Z(i)   I(i)   comments
                eV     %               eV
----------------------------------------------------------------
H    1.01  1    20.4  11.9   K   1    13.60  1S
----------------------------------------------------------------
HE   4.00  2    38.5   9. 0  K   2    24.59  1S
----------------------------------------------------------------
LI   6.94  3    00.0   0.00  K   2    58.    1S
                             L1  1     5.39  2S
----------------------------------------------------------------
BE   9.01  4    00.0   0.00  K   2   115.    1S
                             L1  2     9.32  2S
----------------------------------------------------------------
B   10.81  5    00.0   0.00  K   2   192.    1S
                             L1  2    12.93  2S
                             L2  1     8.30  2P1/2
----------------------------------------------------------------
C   12.01  6    73.8  11.6   K   2   288.    1S
                             L1  2    16.59  2S
                             L2  2    11.26  2P1/2
----------------------------------------------------------------
N   14.01  7    00.0   0.00  K   2   403.    1S
                             L1  2    37.3   2S		a)
                             L2  2    20.33  2P1/2
                             L3  1    14.53  2P3/2
----------------------------------------------------------------
O   16.00  8   115.7   7.73  K   2   543.1   1S		a)
                             L1  2    41.6   2S		a)
                             L2  2    28.48  2P1/2
                             L3  2    13.62  2P3/2
----------------------------------------------------------------
F   19.00  9   124.8  10.00  K   2   696.7   1S		a)
                             L1  2    37.85  2S
                             L2  5    17.42  2P1/2-2P3/2
----------------------------------------------------------------
NE  20.18  10  133.8   7.53  K   2   870.1   1S
                             L1  2    48.47  2S
                             L2  2    21.66  2P1/2
                             L3  4    21.56  2P3/2
----------------------------------------------------------------
NA  22.99  11   00.0   0.00  K   2  1075.    1S
                             L1  2    66.    2S
                             L2  2    34.    2P1/2
                             L3  4    34.    2P3/2
                             M1  1     5.14  3S
----------------------------------------------------------------
MG  24.31  12   00.0   0.00  K   2  1308.    1S
                             L1  2    92.    2S
                             L2  2    54.    2P1/2
                             L3  4    54.    2P3/2
                             M1  2     7.65  3S
----------------------------------------------------------------
AL  26.98  13   00.0   0.00  K   2  1564.    1S
                             L1  2   121.    2S
                             L2  2    77.    2P1/2
                             L3  4    77.    2P3/2
                             M1  2    10.62  3S
                             M2  1     5.99  3P1/2
----------------------------------------------------------------
SI  28.08  14  174.5   6.49  K   2  1844.    1S
                             L1  2   154.    2S
                             L2  2   104.    2P1/2
                             L3  4   104.    2P3/2
                             M1  2    13.46  3S
                             M2  2     8.15  3P1/2
----------------------------------------------------------------
P   30.97  15   00.0   0.00  K   2  2148.    1S
                             L1  2   191.    2S
                             L2  2   135.    2P1/2
                             L3  4   134.    2P3/2
                             M1  2    16.15  3S
                             M2  3    10.49  3P1/2-3P3/2
----------------------------------------------------------------
S   32.06  16   00.0   0.00  K   2  2476.    1S
                             L1  2   232.    2S
                             L2  2   170.    2P1/2
                             L3  4   168.    2P3/2
                             M1  2    20.20  3S
                             M2  4    10.36  3P1/2-3P3/2
----------------------------------------------------------------
CL  35.45  17   00.0   0.00  K   2  2829.    1S
                             L1  2   277.    2S
                             L2  2   208.    2P1/2
                             L3  4   206.    2P3/2
                             M1  2    24.54  3S
                             M2  5    12.97  3P1/2-3P3/2
----------------------------------------------------------------
AR  39.95  18  181.6   6.67  K   2  3206.3   1S
                             L1  2   326.5   2S
                             L2  2   250.6   2P1/2
                             L3  4   248.5   2P3/2
                             M1  2    29.24  3S
                             M2  2    15.94  3P1/2
                             M3  4    15.76  3P3/2
----------------------------------------------------------------
K   39.10  19   00.0   0.00  K   2  3610.    1S
                             L1  2   381.    2S
                             L2  2   299.    2P1/2
                             L3  4   296.    2P3/2
                             M1  2    37.    3S
                             M2  2    19.    3P1/2
                             M3  4    18.7   3P3/2
                             N1  1     4.34  4S
----------------------------------------------------------------
CA  40.08  20   00.0   0.00  K   2  4041.    1S
                             L1  2   441.    2S
                             L2  2   353.    2P1/2
                             L3  4   349.    2P3/2
                             M1  2    46.    3S
                             M2  2    28.    3P1/2
                             M3  4    28.    3P3/2
                             N1  2     6.11  4S
----------------------------------------------------------------
SC  44.96  21   00.0   0.00  K   2  4494.    1S
                             L1  2   503.    2S
                             L2  2   408.    2P1/2
                             L3  4   403.    2P3/2
                             M1  2    55.    3S
                             M2  2    33.    3P1/2
                             M3  4    33.    3P3/2
                             M4  1     8.    3D3/2
                             N1  2     6.54  4S
----------------------------------------------------------------
TI  47.90  22   00.0   0.00  K   2  4966.    1S		a)
                             L1  2   567.    2S
                             L2  2   465.    2P1/2
                             L3  4   459.    2P3/2
                             M1  2    64.    3S
                             M2  2    39.    3P1/2
                             M3  4    38.0   3P3/2
                             M4  2     8.    3D3/2
                             N1  2     6.82  4S
----------------------------------------------------------------
V   50.94  23   00.0   0.00  K   2  5465.    1S		a)
                             L1  2   633.    2S
                             L2  2   525.    2P1/2
                             L3  4   518.    2P3/2
                             M1  2    72.    3S
                             M2  2    44.    3P1/2
                             M3  4    43.    3P3/2
                             M4  3     8.    3D3/2
                             N1  2     6.74  4S
----------------------------------------------------------------
CR  52.01  24   00.0   0.00  K   2  5989.    1S		a)
                             L1  2   702.    2S
                             L2  2   589.    2P1/2
                             L3  4   580.    2P3/2
                             M1  2    80.    3S
                             M2  2    49.    3P1/2
                             M3  4    48.    3P3/2
                             M4  4     8.25  3D3/2
                             N1  2     6.77  4S
----------------------------------------------------------------
MN  54.94  25   00.0   0.00  K   2  6539.    1S		a)
                             L1  2   755.    2S
                             L2  2   656.    2P1/2
                             L3  4   645.    2P3/2
                             M1  2    89.    3S
                             M2  2    55.    3P1/2
                             M3  4    53.    3P3/2
                             M4  5     9.    3D3/2-3D5/2
                             N1  2     7.43  4S
----------------------------------------------------------------
FE  55.85  26   00.0   0.00  K   2  7112.    1S		a)
                             L1  2   851.    2S
                             L2  2   726.    2P1/2
                             L3  4   713.    2P3/2
                             M1  2    98.    3S
                             M2  2    61.    3P1/2
                             M3  4    59.    3P3/2
                             M4  6     9.    3D3/2-3D5/2
                             N1  2     7.87  4S
----------------------------------------------------------------
CO  58.93  27   00.0   0.00  K   2  7709.    1S		a)
                             L1  2   931.    2S
                             L2  2   800.    2P1/2
                             L3  4   785.    2P3/2
                             M1  2   107.    3S
                             M2  2    68.    3P1/2
                             M3  4    66.    3P3/2
                             M4  7     9.    3D3/2-3D5/2
                             N1  2     7.86  4S
----------------------------------------------------------------
NI  58.71  28   00.0   0.00  K   2  8333.    1S		a)
                             L1  2  1015.    2S
                             L2  2   877.    2P1/2
                             L3  4   860.    2P3/2
                             M1  2   117.    3S
                             M2  2    75.    3P1/2
                             M3  4    73.    3P3/2
                             M4  4    10.    3D3/2
                             M5  4    10.    3D5/2
                             N1  2     7.64  4S
----------------------------------------------------------------
CU  63.55  29   00.0   0.00  K   2  8979.    1S		a)
                             L1  2  1103.    2S
                             L2  2   958.    2P1/2
                             L3  4   938.    2P3/2
                             M1  2   127.    3S
                             M2  2    82.    3P1/2
                             M3  4    80.    3P3/2
                             M4  4    11.    3D3/2
                             M5  5    10.4   3D5/2
                             N1  2     7.73  4S
----------------------------------------------------------------
ZN  65.37  30   00.0   0.00  K   2  9659.    1S		a)
                             L1  2  1198.    2S
                             L2  2  1047.    2P1/2
                             L3  4  1024.    2P3/2
                             M1  2   141.    3S
                             M2  2    94.    3P1/2
                             M3  4    91.    3P3/2
                             M4  4    12.    3D3/2
                             M5  6    11.2   3D5/2
                             N1  2     9.39  4S
----------------------------------------------------------------
GA  69.72  31   00.0   0.00  K   2 10367.    1S		a)
                             L1  2  1302.    2S
                             L2  2  1146.    2P1/2
                             L3  4  1119.    2P3/2
                             M1  2   162.    3S
                             M2  2   111.    3P1/2
                             M3  4   107.    3P3/2
                             M4  4    21.    3D3/2
                             M5  6    20.    3D5/2
                             N1  2    11.    4S
                             N2  1     6.    4P1/2
----------------------------------------------------------------
GE  72.59  32   00.0   0.00  K   2 11103.    1S		a)
                             L1  2  1413.    2S
                             L2  2  1251.    2P1/2
                             L3  4  1220.    2P3/2
                             M1  2   184.    3S
                             M2  2   130.    3P1/2
                             M3  4   125.    3P3/2
                             M4  4    33.    3D3/2
                             M5  6    32.    3D5/2
                             N1  2    14.3   4S
                             N2  2     7.9   4P1/2
----------------------------------------------------------------
AS  74.92  33   00.0   0.00  K   2 11867.    1S		a)
                             L1  2  1531.    2S
                             L2  2  1362.    2P1/2
                             L3  4  1327.    2P3/2
                             M1  2   208.    3S
                             M2  2   151.    3P1/2
                             M3  4   145.    3P3/2
                             M4  4    46.    3D3/2
                             M5  6    45.    3D5/2
                             N1  2    17.    4S
                             N2  3     9.81  4P1/2-4P3/2
----------------------------------------------------------------
SE  78.96  34   00.0   0.00  K   2 12658.    1S		a)
                             L1  2  1656.    2S
                             L2  2  1479.    2P1/2
                             L3  4  1439.    2P3/2
                             M1  2   234.    3S
                             M2  2   173.    3P1/2
                             M3  4   166.    3P3/2
                             M4  4    61.    3D3/2
                             M5  6    60.    3D5/2
                             N1  2    20.15  4S
                             N2  4     9.75  4P1/2-4P3/2
----------------------------------------------------------------
BR  79.90  35   00.0   0.00  K   2 13474.    1S		a)
                             L1  2  1787.    2S
                             L2  2  1602.    2P1/2
                             L3  4  1556.    2P3/2
                             M1  2   262.    3S
                             M2  2   197.    3P1/2
                             M3  4   189.    3P3/2
                             M4  4    77.    3D3/2
                             M5  6    76.    3D5/2
                             N1  2    23.8   4S
                             N2  5    11.85  4P1/2-4P3/2
----------------------------------------------------------------
KR  83.80  36  340.8   7.52  K   2 14326.    1S		a)
                             L1  2  1924.6   2S
                             L2  2  1730.9   2P1/2
                             L3  4  1678.4   2P3/2
                             M1  2   292.8   3S
                             M2  2   222.2   3P1/2
                             M3  4   214.4   3P3/2
                             M4  4    95.0   3D3/2
                             M5  6    93.8   3D5/2
                             N1  2    27.51  4S
                             N2  2    14.65  4P1/2
                             N3  4    14.    4P3/2
----------------------------------------------------------------
RB  85.47  37  00.0   0.00   K   2 15200.    1S		a)
                             L1  2  2068.    2S
                             L2  2  1867.    2P1/2
                             L3  4  1807.    2P3/2
                             M1  2   325.    3S
                             M2  2   251.    3P1/2
                             M3  4   242.    3P3/2
                             M4  4   116.    3D3/2
                             M5  6   114.    3D5/2
                             N1  2    32.    4S
                             N2  2    16.    4P1/2
                             N3  4    15.3   4P3/2
                             O1  1     4.18  5S
----------------------------------------------------------------
SR  87.62  38  00.0   0.00   K   2 16105.    1S		a)
                             L1  2  2219.    2S
                             L2  2  2010.    2P1/2
                             L3  4  1943.    2P3/2
                             M1  2   361.    3S
                             M2  2   283.    3P1/2
                             M3  4   273.    3P3/2
                             M4  4   139.    3D3/2
                             M5  6   137.    3D5/2
                             N1  2    40.    4S
                             N2  2    23.    4P1/2
                             N3  4    22.    4P3/2
                             O1  2     5.69  5S
----------------------------------------------------------------
Y   88.91  39  00.0   0.00   K   2 17038.    1S		a)
                             L1  2  2375.    2S
                             L2  2  2158.    2P1/2
                             L3  4  2083.    2P3/2
                             M1  2   397.    3S
                             M2  2   315.    3P1/2
                             M3  4   304.    3P3/2
                             M4  4   163.    3D3/2
                             M5  6   161.    3D5/2
                             N1  2    48.    4S
                             N2  2    30.    4P1/2
                             N3  4    29.    4P3/2
                             O1  2     6.48  5S
                             N4  1     6.38  4D3/2
----------------------------------------------------------------
ZR  91.22  40  00.0   0.00   K   2 17998.    1S		a)
                             L1  2  2536.    2S
                             L2  2  2311.    2P1/2
                             L3  4  2227.    2P3/2
                             M1  2   434.    3S
                             M2  2   348.    3P1/2
                             M3  4   335.    3P3/2
                             M4  4   187.    3D3/2
                             M5  6   185.    3D5/2
                             N1  2    56.    4S
                             N2  2    35.    4P1/2
                             N3  4    33.    4P3/2
                             N4  2     8.61  4D3/2
                             O1  2     6.84  5S
----------------------------------------------------------------
NB  92.91  41  00.0   0.00   K   2 18986.    1S		a)
                             L1  2  2702.    2S
                             L2  2  2469.    2P1/2
                             L3  4  2375.    2P3/2
                             M1  2   472.    3S
                             M2  2   382.    3P1/2
                             M3  4   367.    3P3/2
                             M4  4   212.    3D3/2
                             M5  6   209.    3D5/2
                             N1  2    62.    4S
                             N2  2    40.    4P1/2
                             N3  4    38.    4P3/2
                             N4  3     7.17  4D3/2
                             O1  2     6.88  5S
----------------------------------------------------------------
MO  95.95  42  00.0   0.00   K   2 20000.    1S		a)
                             L1  2  2872.    2S
                             L2  2  2632.    2P1/2
                             L3  4  2527.    2P3/2
                             M1  2   511.    3S
                             M2  2   416.    3P1/2
                             M3  4   399.    3P3/2
                             M4  4   237.    3D3/2
                             M5  6   234.    3D5/2
                             N1  2    68.    4S
                             N2  2    45.    4P1/2
                             N3  4    42.    4P3/2
                             N4  4     8.56  4D3/2
                             O1  2     7.10  5S
----------------------------------------------------------------
TC  99.    43  00.0   0.00   K   2 21044.    1S		a)
                             L1  2  3048.    2S
                             L2  2  2800.    2P1/2
                             L3  4  2683.    2P3/2
                             M1  2   551.    3S
                             M2  2   451.    3P1/2
                             M3  4   432.    3P3/2
                             M4  4   263.    3D3/2
                             M5  6   259.    3D5/2
                             N1  2    74.    4S
                             N2  2    49.    4P1/2
                             N3  4    45.    4P3/2
                             N4  5     8.6   4D3/2-4D5/2
                             O1  2     7.28  5S
----------------------------------------------------------------
RU 101.1   44  00.0   0.00   K   2 22117.    1S		a)
                             L1  2  3230.    2S
                             L2  2  2973.    2P1/2
                             L3  4  2844.    2P3/2
                             M1  2   592.    3S
                             M2  2   488.    3P1/2
                             M3  4   466.    3P3/2
                             M4  4   290.    3D3/2
                             M5  6   286.    3D5/2
                             N1  2    81.    4S
                             N2  2    53.    4P1/2
                             N3  4    49.    4P3/2
                             N4  6     8.5   4D3/2-4D5/2
                             O1  2     7.37  5S
----------------------------------------------------------------
RH 102.91  45  00.0   0.00   K   2 23220.    1S		a)
                             L1  2  3418.    2S
                             L2  2  3152.    2P1/2
                             L3  4  3010.    2P3/2
                             M1  2   634.    3S
                             M2  2   526.    3P1/2
                             M3  4   501.    3P3/2
                             M4  4   318.    3D3/2
                             M5  6   313.    3D5/2
                             N1  2    87.    4S
                             N2  2    58.    4P1/2
                             N3  4    53.    4P3/2
                             N4  7     9.56  4D3/2-4D5/2
                             O1  2     7.46  5S
----------------------------------------------------------------
PD 106.4   46  00.0   0.00   K   2 24350.    1S		a)
                             L1  2  3611.    2S
                             L2  2  3337.    2P1/2
                             L3  4  3180.    2P3/2
                             M1  2   677.    3S
                             M2  2   565.    3P1/2
                             M3  4   537.    3P3/2
                             M4  4   347.    3D3/2
                             M5  6   342.    3D5/2
                             N1  2    93.    4S
                             N2  2    63.    4P1/2
                             N3  4    57.    4P3/2
                             N4  4     8.78  4D3/2
                             N5  4     8.34  4D5/2
                             O1  2     7.52  5S
----------------------------------------------------------------
AG 107.88  47  00.0   0.00   K   2 25514.    1S		a)
                             L1  2  3812.    2S
                             L2  2  3530.    2P1/2
                             L3  4  3357.    2P3/2
                             M1  2   724.    3S
                             M2  2   608.    3P1/2
                             M3  4   577.    3P3/2
                             M4  4   379.    3D3/2
                             M5  6   373.    3D5/2
                             N1  2   101.    4S
                             N2  2    69.    4P1/2
                             N3  4    63.    4P3/2
                             N4  4    11.    4D3/2
                             N5  5    10.    4D5/2
                             O1  2     7.58  5S
----------------------------------------------------------------
CD 112.41  48  00.0   0.00   K   2 26711.    1S		a)
                             L1  2  4022.    2S
                             L2  2  3732.    2P1/2
                             L3  4  3542.    2P3/2
                             M1  2   775.    3S
                             M2  2   655.    3P1/2
                             M3  4   621.    3P3/2
                             M4  4   415.    3D3/2
                             M5  6   408.    3D5/2
                             N1  2   112.    4S
                             N2  2    78.    4P1/2
                             N3  4    71.    4P3/2
                             N4  4    14.    4D3/2
                             N5  6    13.    4D5/2
                             O1  2     8.99  5S
----------------------------------------------------------------
IN 114.82  49  00.0   0.00   K   2 27940.    1S		a)
                             L1  2  4242.    2S
                             L2  2  3943.    2P1/2
                             L3  4  3735.    2P3/2
                             M1  2   830.    3S
                             M2  2   707.    3P1/2
                             M3  4   669.    3P3/2
                             M4  4   455.    3D3/2
                             M5  6   447.    3D5/2
                             N1  2   126.    4S
                             N2  2    90.    4P1/2
                             N3  4    82.    4P3/2
                             N4  4    21.    4D3/2
                             N5  6    20.    4D5/2
                             O1  2    10.    5S
                             O2  1     5.79  5P1/2
----------------------------------------------------------------
SN 118.70  50  00.0   0.00   K   2 29200.    1S		a)
                             L1  2  4469.    2S
                             L2  2  4160.    2P1/2
                             L3  4  3933.    2P3/2
                             M1  2   888.    3S
                             M2  2   761.    3P1/2
                             M3  4   719.    3P3/2
                             M4  4   497.    3D3/2
                             M5  6   489.    3D5/2
                             N1  2   141.    4S
                             N2  2   102.    4P1/2
                             N3  4    93.    4P3/2
                             N4  4    29.    4D3/2
                             N5  6    28.    4D5/2
                             O1  2    12.    5S
                             O2  2     7.34  5P1/2
----------------------------------------------------------------
SB 121.76  51  00.0   0.00   K   2 30419.    1S		a)
                             L1  2  4698.    2S		a)
                             L2  2  4385.    2P1/2
                             L3  4  4137.    2P3/2
                             M1  2   949.    3S
                             M2  2   817.    3P1/2
                             M3  4   771.    3P3/2
                             M4  4   542.    3D3/2
                             M5  6   533.    3D5/2
                             N1  2   157.    4S
                             N2  2   114.    4P1/2
                             N3  4   104.    4P3/2
                             N4  4    38.    4D3/2
                             N5  6    37.    4D5/2
                             O1  2    15.    5S
                             O2  3     8.64  5P1/2-5P3/2
----------------------------------------------------------------
TE 127.61  52  00.0   0.00   K   2 31814.    1S		a)
                             L1  2  4939.    2S		a)
                             L2  2  4612.    2P1/2	a)
                             L3  4  4347.    2P3/2
                             M1  2  1012.    3S
                             M2  2   876.    3P1/2
                             M3  4   825.    3P3/2
                             M4  4   589.    3D3/2
                             M5  6   578.    3D5/2
                             N1  2   174.    4S
                             N2  2   127.    4P1/2
                             N3  4   117.    4P3/2
                             N4  4    48.    4D3/2
                             N5  6    46.    4D5/2
                             O1  2    17.84  5S
                             O2  4     9.01  5P1/2-5P3/2
----------------------------------------------------------------
I  126.91  53  00.0   0.00   K   2 33169.    1S		a)
                             L1  2  5188.    2S		a)
                             L2  2  4852.    2P1/2	a)
                             L3  4  4557.    2P3/2	a)
                             M1  2  1078.    3S
                             M2  2   937.    3P1/2
                             M3  4   881.    3P3/2
                             M4  4   638.    3D3/2
                             M5  6   626.    3D5/2
                             N1  2   193.    4S
                             N2  2   141.    4P1/2
                             N3  4   131.    4P3/2
                             N4  4    58.    4D3/2
                             N5  6    56.    4D5/2
                             O1  2    20.61  5S
                             O2  5    10.45  5P1/2-5P3/2
----------------------------------------------------------------
XE 131.30  54  508.8   9.61  K   2 34570.    1S		a)
                             L1  2  5460.    2S		a)
                             L2  2  5110.    2P1/2	a)
                             L3  4  4790.    2P3/2	a)
                             M1  2  1148.7   3S
                             M2  2  1002.1   3P1/2
                             M3  4   940.6   3P3/2
                             M4  4   689.0   3D3/2
                             M5  6   676.4   3D5/2
                             N1  2   213.2   4S
                             N2  2   157.    4P1/2
                             N3  4   145.5   4P3/2
                             N4  4    69.5   4D3/2
                             N5  6    67.5   4D5/2
                             O1  2    23.39  5S
                             O2  2    13.43  5P1/2
                             O3  4    12.13  5P3/2
----------------------------------------------------------------
CS 132.91  55   00.0   0.00  K   2 35985.    1S		a)
                             L1  2  5714.    2S		a)
                             L2  2  5359.    2P1/2	a)
                             L3  4  5012.    2P3/2	a)
                             M1  2  1220.    3S
                             M2  2  1068.    3P1/2
                             M3  4  1000.    3P3/2
                             M4  4   742.    3D3/2
                             M5  6   728.    3D5/2
                             N1  2   233.    4S
                             N2  2   174.    4P1/2
                             N3  4   164.    4P3/2
                             N4  4    81.    4D3/2
                             N5  6    79.    4D5/2
                             O1  2    25.    5S
                             O2  2    14.    5P1/2
                             O3  4    12.3   5P3/2
                             P1  1     3.89  6S
----------------------------------------------------------------
BA 137.36  56   00.0   0.00  K   2 37441.    1S		a)
                             L1  2  5989.    2S		a)
                             L2  2  5624.    2P1/2	a)
                             L3  4  5247.    2P3/2	a)
                             M1  2  1293.    3S
                             M2  2  1138.    3P1/2
                             M3  4  1063.    3P3/2
                             M4  4   797.    3D3/2
                             M5  6   782.    3D5/2
                             N1  2   254.    4S
                             N2  2   193.    4P1/2
                             N3  4   181.    4P3/2
                             N4  4    94.    4D3/2
                             N5  6    92.    4D5/2
                             O1  2    31.    5S
                             O2  2    18.    5P1/2
                             O3  4    16.    5P3/2
                             P1  2     5.21  6S
----------------------------------------------------------------
LA 138.92  57   00.0   0.00  K   2 38925.    1S		a)
                             L1  2  6266.    2S		a)
                             L2  2  5891.    2P1/2	a)
                             L3  4  5483.    2P3/2	a)
                             M1  2  1365.    3S
                             M2  2  1207.    3P1/2
                             M3  4  1124.    3P3/2
                             M4  4   851.    3D3/2
                             M5  6   834.    3D5/2
                             N1  2   273.    4S
                             N2  2   210.    4P1/2
                             N3  4   196.    4P3/2
                             N4  4   105.    4D3/2
                             N5  6   103.    4D5/2
                             O1  2    36.    5S
                             O2  2    22.    5P1/2
                             O3  4    19.    5P3/2
                             O4  1     5.75  5D3/2
                             P1  2     5.58  6S
----------------------------------------------------------------
CE 140.13  58   00.0   0.00  K   2 40443.    1S		a)
                             L1  2  6548.    2S		a)
                             L2  2  6164.    2P1/2	a)
                             L3  4  5723.    2P3/2	a)
                             M1  2  1437.    3S
                             M2  2  1275.    3P1/2
                             M3  4  1184.    3P3/2
                             M4  4   903.    3D3/2
                             M5  6   885.    3D5/2
                             N1  2   291.    4S
                             N2  2   225.    4P1/2
                             N3  4   209.    4P3/2
                             N4  4   114.    4D3/2
                             N5  6   111.    4D5/2
                             O1  2    39.    5S
                             O2  2    25.    5P1/2
                             O3  4    22.    5P3/2
                             N6  2     6.    4F5/2
                             P1  2     5.65  6S
----------------------------------------------------------------
PR 140.92  59   00.0   0.00  K   2 41991.    1S		a)
                             L1  2  6835.    2S		a)
                             L2  2  6440.    2P1/2	a)
                             L3  4  5964.    2P3/2	a)
                             M1  2  1509.    3S
                             M2  2  1342.    3P1/2
                             M3  4  1244.    3P3/2
                             M4  4   954.    3D3/2
                             M5  6   934.    3D5/2
                             N1  2   307.    4S
                             N2  2   238.    4P1/2
                             N3  4   220.    4P3/2
                             N4  4   121.    4D3/2
                             N5  6   117.    4D5/2
                             O1  2    41.    5S
                             O2  2    27.    5P1/2
                             O3  4    24.    5P3/2
                             N6  3     6.    4F5/2
                             P1  2     5.42  6S
----------------------------------------------------------------
ND 144.27  60   00.0   0.00  K   2 43569.    1S		a)
                             L1  2  7126.    2S		a)
                             L2  2  6722.    2P1/2	a)
                             L3  4  6208.    2P3/2	a)
                             M1  2  1580.    3S
                             M2  2  1408.    3P1/2
                             M3  4  1303.    3P3/2
                             M4  4  1005.    3D3/2
                             M5  6   983.    3D5/2
                             N1  2   321.    4S
                             N2  2   250.    4P1/2
                             N3  4   230.    4P3/2
                             N4  4   126.    4D3/2
                             N5  6   122.    4D5/2
                             O1  2    42.    5S
                             O2  2    28.    5P1/2
                             O3  4    25.    5P3/2
                             N6  4     6.    4F5/2
                             P1  2     5.49  6S
----------------------------------------------------------------
PM 145.    61   00.0   0.00  K   2 45184.    1S		a)
                             L1  2  7428.    2S		a)
                             L2  2  7013.    2P1/2	a)
                             L3  4  6459.    2P3/2	a)
                             M1  2  1653.    3S
                             M2  2  1476.    3P1/2
                             M3  4  1362.    3P3/2
                             M4  4  1057.    3D3/2
                             M5  6  1032.    3D5/2
                             N1  2   325.    4S
                             N2  2   261.    4P1/2
                             N3  4   240.    4P3/2
                             N4  4   131.    4D3/2
                             N5  6   127.    4D5/2
                             O1  2    43.    5S
                             O2  2    28.    5P1/2
                             O3  4    25.    5P3/2
                             N6  5     6.    4F5/2
                             P1  2     5.55  6S
----------------------------------------------------------------
SM 150.35  62   00.0   0.00  K   2 46834.    1S		a)
                             L1  2  7737.    2S		a)
                             L2  2  7312.    2P1/2	a)
                             L3  4  6716.    2P3/2	a)
                             M1  2  1728.    3S
                             M2  2  1546.    3P1/2
                             M3  4  1422.    3P3/2
                             M4  4  1110.    3D3/2
                             M5  6  1083.    3D5/2
                             N1  2   349.    4S
                             N2  2   273.    4P1/2
                             N3  4   251.    4P3/2
                             N4  4   137.    4D3/2
                             N5  6   132.    4D5/2
                             O1  2    44.    5S
                             O2  2    29.    5P1/2
                             O3  4    25.    5P3/2
                             N6  6     6.    4F5/2
                             P1  2     5.63  6S
----------------------------------------------------------------
EU 152.0   63   00.0   0.00  K   2 48519.    1S		a)
                             L1  2  8052.    2S		a)
                             L2  2  7617.    2P1/2	a)
                             L3  4  6977.    2P3/2	a)
                             M1  2  1805.    3S
                             M2  2  1618.    3P1/2
                             M3  4  1484.    3P3/2
                             M4  4  1164.    3D3/2
                             M5  6  1135.    3D5/2
                             N1  2   364.    4S
                             N2  2   286.    4P1/2
                             N3  4   262.    4P3/2
                             N4  4   143.    4D3/2
                             N5  6   137.    4D5/2
                             O1  2    45.    5S
                             O2  2    30.    5P1/2
                             O3  4    26.    5P3/2
                             N6  7     6.    4F5/2-4F7/2
                             P1  2     5.68  6S
----------------------------------------------------------------
GD 157.26  64   00.0   0.00  K   2 50239.    1S		a)
                             L1  2  8376.    2S		a)
                             L2  2  7930.    2P1/2	a)
                             L3  4  7243.    2P3/2	a)
                             M1  2  1884.    3S
                             M2  2  1692.    3P1/2
                             M3  4  1547.    3P3/2
                             M4  4  1220.    3D3/2
                             M5  6  1189.    3D5/2
                             N1  2   380.    4S
                             N2  2   300.    4P1/2
                             N3  4   273.    4P3/2
                             N4  4   150.    4D3/2
                             N5  6   143.    4D5/2
                             O1  2    46.    5S
                             O2  2    31.    5P1/2
                             O3  4    27.    5P3/2
                             P1  2     6.16  6S
                             N6  7     6.    4F5/2-4F7/2
                             O4  1     6.    5D3/2
----------------------------------------------------------------
TB 158.93  65   00.0   0.00  K   2 51996.    1S		a)
                             L1  2  8708.    2S		a)
                             L2  2  8252.    2P1/2	a)
                             L3  4  7514.    2P3/2	a)
                             M1  2  1965.    3S
                             M2  2  1768.    3P1/2
                             M3  4  1612.    3P3/2
                             M4  4  1277.    3D3/2
                             M5  6  1243.    3D5/2
                             N1  2   398.    4S
                             N2  2   315.    4P1/2
                             N3  4   285.    4P3/2
                             N4  4   157.    4D3/2
                             N5  6   150.    4D5/2
                             O1  2    48.    5S
                             O2  2    32.    5P1/2
                             O3  4    28.    5P3/2
                             N6  9     6.    4F5/2-4F7/2
                             P1  2     5.85  6S
----------------------------------------------------------------
DY 162.51  66   00.0   0.00  K   2 53789.    1S		a)
                             L1  2  9046.    2S		a)
                             L2  2  8581.    2P1/2	a)
                             L3  4  7790.    2P3/2	a)
                             M1  2  2048.    3S
                             M2  2  1846.    3P1/2
                             M3  4  1678.    3P3/2
                             M4  4  1335.    3D3/2
                             M5  6  1298.    3D5/2
                             N1  2   416.    4S
                             N2  2   331.    4P1/2
                             N3  4   297.    4P3/2
                             N4  4   164.    4D3/2
                             N5  6   157.    4D5/2
                             O1  2    50.    5S
                             O2  2    33.    5P1/2
                             O3  4    28.    5P3/2
                             N6 10     6.    4F5/2-4F7/2
                             P1  2     5.93  6S
----------------------------------------------------------------
HO 164.94  67   00.0   0.00  K   2 55618.    1S		a)
                             L1  2  9394.    2S		a)
                             L2  2  8918.    2P1/2	a)
                             L3  4  8071.    2P3/2	a)
                             M1  2  2133.    3S
                             M2  2  1926.    3P1/2
                             M3  4  1746.    3P3/2
                             M4  4  1395.    3D3/2
                             M5  6  1354.    3D5/2
                             N1  2   434.    4S
                             N2  2   348.    4P1/2
                             N3  4   310.    4P3/2
                             N4  4   172.    4D3/2
                             N5  6   164.    4D5/2
                             O1  2    52.    5S
                             O2  2    34.    5P1/2
                             O3  4    29.    5P3/2
                             P1  2     6.02  6S
                             N6 11     6.    4F5/2-4F7/2
----------------------------------------------------------------
ER 167.27  68   00.0   0.00  K   2 57486.    1S		a)
                             L1  2  9751.    2S		a)
                             L2  2  9264.    2P1/2	a)
                             L3  4  8358.    2P3/2	a)
                             M1  2  2220.    3S
                             M2  2  2008.    3P1/2
                             M3  4  1815.    3P3/2
                             M4  4  1456.    3D3/2
                             M5  6  1412.    3D5/2
                             N1  2   452.    4S
                             N2  2   365.    4P1/2
                             N3  4   323.    4P3/2
                             N4  4   181.    4D3/2
                             N5  6   172.    4D5/2
                             O1  2    54.    5S
                             O2  2    35.    5P1/2
                             O3  4    30.    5P3/2
                             P1  2     6.10  6S
                             N6 12     6.    4F5/2-4F7/2
----------------------------------------------------------------
TU 168.94  69   00.0   0.00  K   2 59390.    1S		a)
                             L1  2 10116.    2S		a)
                             L2  2  9617.    2P1/2	a)
                             L3  4  8648.    2P3/2	a)
                             M1  2  2309.    3S
                             M2  2  2092.    3P1/2
                             M3  4  1885.    3P3/2
                             M4  4  1518.    3D3/2
                             M5  6  1471.    3D5/2
                             N1  2   471.    4S
                             N2  2   382.    4P1/2
                             N3  4   336.    4P3/2
                             N4  4   190.    4D3/2
                             N5  6   181.    4D5/2
                             O1  2    56.    5S
                             O2  2    36.    5P1/2
                             O3  4    30.    5P3/2
                             N6 13     7.    4F5/2-4F7/2
                             P1  2     6.18  6S
----------------------------------------------------------------
YB 173.04  70   00.0   0.00  K   2 61332.    1S		a)
                             L1  2 10486.    2S		a)
                             L2  2  9978.    2P1/2	a)
                             L3  4  8944.    2P3/2	a)
                             M1  2  2401.    3S
                             M2  2  2178.    3P1/2
                             M3  4  1956.    3P3/2
                             M4  4  1580.    3D3/2
                             M5  6  1531.    3D5/2
                             N1  2   490.    4S
                             N2  2   399.    4P1/2
                             N3  4   349.    4P3/2
                             N4  4   200.    4D3/2
                             N5  6   190.    4D5/2
                             O1  2    58.    5S
                             O2  2    37.    5P1/2
                             O3  4    31.    5P3/2
                             N6  6     8.    4F5/2
                             N7  8     7.    4F7/2
                             P1  2     6.25  6S
----------------------------------------------------------------
LU 174.99  71   00.0   0.00  K   2 63314.    1S		a)
                             L1  2 10870.    2S		a)
                             L2  2 10349.    2P1/2	a)
                             L3  4  9244.    2P3/2	a)
                             M1  2  2499.    3S
                             M2  2  2270.    3P1/2
                             M3  4  2032.    3P3/2
                             M4  4  1647.    3D3/2
                             M5  6  1596.    3D5/2
                             N1  2   514.    4S
                             N2  2   420.    4P1/2
                             N3  4   366.    4P3/2
                             N4  4   213.    4D3/2
                             N5  6   202.    4D5/2
                             O1  2    62.    5S
                             O2  2    39.    5P1/2
                             O3  4    32.    5P3/2
                             N6  6    13.    4F5/2
                             N7  8    12.    4F7/2
                             P1  2     7.0   6S
                             O4  1     6.6   5D3/2
----------------------------------------------------------------
HF 178.50  72   00.0   0.00  K   2 65351.    1S		a)
                             L1  2 11271.    2S		a)
                             L2  2 10739.    2P1/2	a)
                             L3  4  9561.    2P3/2	a)
                             M1  2  2604.    3S
                             M2  2  2369.    3P1/2
                             M3  4  2113.    3P3/2
                             M4  4  1720.    3D3/2
                             M5  6  1665.    3D5/2
                             N1  2   542.    4S
                             N2  2   444.    4P1/2
                             N3  4   386.    4P3/2
                             N4  4   229.    4D3/2
                             N5  6   217.    4D5/2
                             O1  2    68.    5S
                             O2  2    43.    5P1/2
                             O3  4    35.    5P3/2
                             N6  6    21.    4F5/2
                             N7  8    20.    4F7/2
                             P1  2     7.5   6S
                             O4  2     7.0   5D3/2
----------------------------------------------------------------
TA 180.95  73   00.0   0.00  K   2 67416.    1S		a)
                             L1  2 11682.    2S		a)
                             L2  2 11136.    2P1/2	a)
                             L3  4  9881.    2P3/2	a)
                             M1  2  2712.    3S
                             M2  2  2472.    3P1/2
                             M3  4  2197.    3P3/2
                             M4  4  1796.    3D3/2
                             M5  6  1737.    3D5/2
                             N1  2   570.    4S
                             N2  2   469.    4P1/2
                             N3  4   407.    4P3/2
                             N4  4   245.    4D3/2
                             N5  6   232.    4D5/2
                             O1  2    74.    5S
                             O2  2    47.    5P1/2
                             O3  4    38.    5P3/2
                             N6  6    30.    4F5/2
                             N7  8    28.    4F7/2
                             O4  3     8.3   5D3/2
                             P1  2     7.9   6S
----------------------------------------------------------------
W  183.86  74   00.0   0. 0  K   2 69525.    1S		a)
                             L1  2 12100.    2S		a)
                             L2  2 11544.    2P1/2	a)
                             L3  4 10207.    2P3/2	a)
                             M1  2  2823.    3S
                             M2  2  2577.    3P1/2
                             M3  4  2283.    3P3/2
                             M4  4  1874.    3D3/2
                             M5  6  1811.    3D5/2
                             N1  2   599.    4S
                             N2  2   495.    4P1/2
                             N3  4   428.    4P3/2
                             N4  4   261.    4D3/2
                             N5  6   248.    4D5/2
                             O1  2    80.    5S
                             O2  2    51.    5P1/2
                             O3  4    41.    5P3/2
                             N6  6    38.    4F5/2
                             N7  8    36.    4F7/2
                             O4  4     9.0   5D3/2
                             P1  2     8.0   6S
----------------------------------------------------------------
RE 186.22  75   00.0   0.00  K   2 71676.    1S		a)
                             L1  2 12527.    2S		a)
                             L2  2 11959.    2P1/2	a)
                             L3  4 10535.    2P3/2	a)
                             M1  2  2937.    3S
                             M2  2  2686.    3P1/2
                             M3  4  2371.    3P3/2
                             M4  4  1953.    3D3/2
                             M5  6  1887.    3D5/2
                             N1  2   629.    4S
                             N2  2   522.    4P1/2
                             N3  4   450.    4P3/2
                             N4  4   278.    4D3/2
                             N5  6   264.    4D5/2
                             O1  2    86.    5S
                             O2  2    56.    5P1/2
                             N6  6    47.    4F5/2
                             N7  8    45.    4F7/2
                             O3  4    45.    5P3/2
                             O4  5     9.6   5D3/2-5D5/2
                             P1  2     7.9   6S
----------------------------------------------------------------
OS 190.2   76   00.0   0.00  K   2 73871.    1S		a)
                             L1  2 12968.    2S		a)
                             L2  2 12385.    2P1/2	a)
                             L3  4 10871.    2P3/2	a)
                             M1  2  3054.    3S
                             M2  2  2797.    3P1/2
                             M3  4  2461.    3P3/2
                             M4  4  2035.    3D3/2
                             M5  6  1964.    3D5/2
                             N1  2   660.    4S
                             N2  2   551.    4P1/2
                             N3  4   473.    4P3/2
                             N4  4   295.    4D3/2
                             N5  6   280.    4D5/2
                             O1  2    92.    5S
                             O2  2    61.    5P1/2
                             N6  6    56.    4F5/2
                             N7  8    54.    4F7/2
                             O3  4    49.    5P3/2
                             O4  6     9.6   5D3/2-5D5/2
                             P1  2     8.5   6S
----------------------------------------------------------------
IR 192.2   77   00.0   0.00  K   2 76111.    1S		a)
                             L1  2 13419.    2S		a)
                             L2  2 12824.    2P1/2	a)
                             L3  4 11215.    2P3/2	a)
                             M1  2  3175.    3S
                             M2  2  2912.    3P1/2
                             M3  4  2554.    3P3/2
                             M4  4  2119.    3D3/2
                             M5  6  2044.    3D5/2
                             N1  2   693.    4S
                             N2  2   581.    4P1/2
                             N3  4   497.    4P3/2
                             N4  4   314.    4D3/2
                             N5  6   298.    4D5/2
                             O1  2    99.    5S
                             N6  6    67.    4F5/2
                             O2  2    66.    5P1/2
                             N7  8    64.    4F7/2
                             O3  4    53.    5P3/2
                             O4  7     9.6   5D3/2-5D5/2
                             P1  2     9.1   6S
----------------------------------------------------------------
PT 195.09  78   00.0   0.00  K   2 78395.    1S		a)
                             L1  2 13880.    2S		a)
                             L2  2 13273.    2P1/2	a)
                             L3  4 11564.    2P3/2	a)
                             M1  2  3300.    3S
                             M2  2  3030.    3P1/2
                             M3  4  2649.    3P3/2
                             M4  4  2206.    3D3/2
                             M5  6  2126.    3D5/2
                             N1  2   727.    4S
                             N2  2   612.    4P1/2
                             N3  4   522.    4P3/2
                             N4  4   335.    4D3/2
                             N5  6   318.    4D5/2
                             O1  2   106.    5S
                             N6  6    78.    4F5/2
                             N7  8    75.    4F7/2
                             O2  2    71.    5P1/2
                             O3  4    57.    5P3/2
                             O4  9     9.6   5D3/2-5D5/2
                             P1  1     9.0   6S
----------------------------------------------------------------
AU 197.0   79   00.0   0.00  K   2 80725.    1S		a)
                             L1  2 14353.    2S		a)
                             L2  2 13734.    2P1/2	a)
                             L3  4 11919.    2P3/2	a)
                             M1  2  3430.    3S
                             M2  2  3153.    3P1/2
                             M3  4  2748.    3P3/2
                             M4  4  2295.    3D3/2
                             M5  6  2210.    3D5/2
                             N1  2   764.    4S
                             N2  2   645.    4P1/2
                             N3  4   548.    4P3/2
                             N4  4   357.    4D3/2
                             N5  6   339.    4D5/2
                             O1  2   114.    5S
                             N6  6    91.    4F5/2
                             N7  8    87.    4F7/2
                             O2  2    76.    5P1/2
                             O3  4    61.    5P3/2
                             O4  4    12.5   5D3/2
                             O5  6    11.1   5D5/2
                             P1  1     9.23  6S
----------------------------------------------------------------
HG 200.61  80   00.0   0.00  K   2 83102.    1S		a)
                             L1  2 14839.    2S		a)
                             L2  2 14209.    2P1/2	a)
                             L3  4 12284.    2P3/2	a)
                             M1  2  3567.    3S
                             M2  2  3283.    3P1/2
                             M3  4  2852.    3P3/2
                             M4  4  2390.    3D3/2
                             M5  6  2300.    3D5/2
                             N1  2   806.    4S
                             N2  2   683.    4P1/2
                             N3  4   579.    4P3/2
                             N4  4   382.    4D3/2
                             N5  6   363.    4D5/2
                             O1  2   125.    5S
                             N6  6   107.    4F5/2
                             N7  8   103.    4F7/2
                             O2  2    85.    5P1/2
                             O3  4    68.    5P3/2
                             O4  4    14.    5D3/2
                             O5  6    12.    5D5/2
                             P1  2    10.4   6S
----------------------------------------------------------------
TL 204.39  81   00.0   0.00  K   2 85530.    1S		a)
                             L1  2 15347.    2S		a)
                             L2  2 14698.    2P1/2	a)
                             L3  4 12658.    2P3/2	a)
                             M1  2  3710.    3S
                             M2  2  3420.    3P1/2
                             M3  4  2961.    3P3/2
                             M4  4  2490.    3D3/2
                             M5  6  2394.    3D5/2
                             N1  2   852.    4S
                             N2  2   726.    4P1/2
                             N3  4   615.    4P3/2
                             N4  4   411.    4D3/2
                             N5  6   391.    4D5/2
                             O1  2   139.    5S
                             N6  6   127.    4F5/2
                             N7  8   123.    4F7/2
                             O2  2    98.    5P1/2
                             O3  4    79.    5P3/2
                             O4  4    21.    5D3/2
                             O5  6    19.    5D5/2
                             P1  2     8.    6S
                             P2  1     6.11  6P1/2
----------------------------------------------------------------
PB 207.21  82   00.0   0.00  K   2 88005.    1S		a)
                             L1  2 15861.    2S		a)
                             L2  2 15200.    2P1/2	a)
                             L3  4 13055.    2P3/2	a)
                             M1  2  3857.    3S
                             M2  2  3560.    3P1/2
                             M3  4  3072.    3P3/2
                             M4  4  2592.    3D3/2
                             M5  6  2490.    3D5/2
                             N1  2   899.    4S
                             N2  2   769.    4P1/2
                             N3  4   651.    4P3/2
                             N4  4   441.    4D3/2
                             N5  6   419.    4D5/2
                             O1  2   153.    5S
                             N6  6   148.    4F5/2
                             N7  8   144.    4F7/2
                             O2  2   111.    5P1/2
                             O3  4    90.    5P3/2
                             O4  4    27.    5D3/2
                             O5  6    25.    5D5/2
                             P1  2    10.    6S
                             P2  2     7.42  6P1/2
----------------------------------------------------------------
BI 209.00  83   00.0   0.00  K   2 90526.    1S		a)
                             L1  2 16388.    2S		a)
                             L2  2 15711.    2P1/2	a)
                             L3  4 13419.    2P3/2	a)
                             M1  2  4007.    3S
                             M2  2  3704.    3P1/2
                             M3  4  3185.    3P3/2
                             M4  4  2696.    3D3/2
                             M5  6  2588.    3D5/2
                             N1  2   946.    4S
                             N2  2   813.    4P1/2
                             N3  4   687.    4P3/2
                             N4  4   472.    4D3/2
                             N5  6   448.    4D5/2
                             N6  6   170.    4F5/2
                             O1  2   167.    5S
                             N7  8   165.    4F7/2
                             O2  2   125.    5P1/2
                             O3  4   101.    5P3/2
                             O4  4    34.    5D3/2
                             O5  6    32.    5D5/2
                             P1  2    12.    6S
                             P2  3     7.29  6P1/2-6P3/2
----------------------------------------------------------------
PO 209.    84   00.0   0.00  K   2 93105.    1S		a)
                             L1  2 16939.    2S		a)
                             L2  2 16244.    2P1/2	a)
                             L3  4 13814.    2P3/2	a)
                             M1  2  4161.    3S
                             M2  2  3852.    3P1/2
                             M3  4  3301.    3P3/2
                             M4  4  2802.    3D3/2
                             M5  6  2687.    3D5/2
                             N1  2   994.    4S
                             N2  2   858.    4P1/2
                             N3  4   724.    4P3/2
                             N4  4   503.    4D3/2
                             N5  6   478.    4D5/2
                             N6  6   193.    4F5/2
                             N7  8   187.    4F7/2
                             O1  2   181.    5S
                             O2  2   139.    5P1/2
                             O3  4   112.    5P3/2
                             O4  4    41.    5D3/2
                             O5  6    38.    5D5/2
                             P1  2    15.    6S
                             P2  4     8.43  6P1/2-6P3/2
----------------------------------------------------------------
AT 210.    85   00.0   0.00  K   2 95730.    1S		a)
                             L1  2 17493.    2S		a)
                             L2  2 16785.    2P1/2	a)
                             L3  4 14214.    2P3/2	a)
                             M1  2  4320.    3S
                             M2  2  4005.    3P1/2
                             M3  4  3420.    3P3/2
                             M4  4  2910.    3D3/2
                             M5  6  2788.    3D5/2
                             N1  2  1044.    4S
                             N2  2   904.    4P1/2
                             N3  4   761.    4P3/2
                             N4  4   535.    4D3/2
                             N5  6   508.    4D5/2
                             N6  6   217.    4F5/2
                             N7  8   211.    4F7/2
                             O1  2   196.    5S
                             O2  2   153.    5P1/2
                             O3  4   123.    5P3/2
                             O4  4    48.    5D3/2
                             O5  6    44.    5D5/2
                             P1  2    19.    6S
                             P2  2    11.    6P1/2
                             P3  3     9.3   6P3/2
----------------------------------------------------------------
RN 222.    86   00.0   0.00  K   2 98404.    1S		a)
                             L1  2 18049.    2S		a)
                             L2  2 17337.    2P1/2	a)
                             L3  4 14619.    2P3/2	a)
                             M1  2  4483.    3S
                             M2  2  4162.    3P1/2
                             M3  4  3452.    3P3/2
                             M4  4  3109.    3D3/2
                             M5  6  2890.    3D5/2
                             N1  2  1096.    4S
                             N2  2   951.    4P1/2
                             N3  4   798.    4P3/2
                             N4  4   567.    4D3/2
                             N5  6   538.    4D5/2
                             N6  6   242.    4F5/2
                             N7  8   235.    4F7/2
                             O1  2   212.    5S
                             O2  2   167.    5P1/2
                             O3  4   134.    5P3/2
                             O4  4    55.    5D3/2
                             O5  6    51.    5D5/2
                             P1  2    24.    6S
                             P2  2    14.    6P1/2
                             P3  4    10.7   6P3/2
----------------------------------------------------------------
FR 223.    87   00.0   0.00  K   2 101137.   1S		a)
                             L1  2 18639.    2S		a)
                             L2  2 17907.    2P1/2	a)
                             L3  4 15031.    2P3/2	a)
                             M1  2  4652.    3S  	a)
                             M2  2  4324.    3P1/2
                             M3  4  3666.    3P3/2
                             M4  4  3134.    3D3/2
                             M5  6  2998.    3D5/2
                             N1  2  1153.    4S
                             N2  2  1003.    4P1/2
                             N3  4   839.    4P3/2
                             N4  4   603.    4D3/2
                             N5  6   572.    4D5/2
                             N6  6   268.    4F5/2
                             N7  8   260.    4F7/2
                             O1  2   231.    5S
                             O2  2   183.    5P1/2
                             O3  4   147.    5P3/2
                             O4  4    65.    5D3/2
                             O5  6    61.    5D5/2
                             P1  2    33.    6S
                             P2  2    19.    6P1/2
                             P3  4    14.    6P3/2
                             Q1  1     4.0   7S
----------------------------------------------------------------
RA 226.05  88   00.0   0.00  K   2 103922.   1S		a)
                             L1  2 19237.    2S		a)
                             L2  2 18484.    2P1/2	a)
                             L3  4 15444.    2P3/2	a)
                             M1  2  4822.    3S  	a)
                             M2  2  4491.    3P1/2
                             M3  4  3793.    3P3/2
                             M4  4  3254.    3D3/2
                             M5  6  3111.    3D5/2
                             N1  2  1214.    4S
                             N2  2  1060.    4P1/2
                             N3  4   884.    4P3/2
                             N4  4   642.    4D3/2
                             N5  6   609.    4D5/2
                             N6  6   296.    4F5/2
                             N7  8   287.    4F7/2
                             O1  2   253.    5S
                             O2  2   201.    5P1/2
                             O3  4   161.    5P3/2
                             O4  4    77.    5D3/2
                             O5  6    73.    5D5/2
                             P1  2    40.    6S
                             P2  2    25.    6P1/2
                             P3  4    19.    6P3/2
                             Q1  2     5.28  7S
----------------------------------------------------------------
AC 227.    89   00.0   0.00  K   2 106755.   1S		a)
                             L1  2 19840.    2S		a)
                             L2  2 19083.    2P1/2	a)
                             L3  4 15871.    2P3/2	a)
                             M1  2  5002.    3S  	a)
                             M2  2  4656.    3P1/2	a)
                             M3  4  3921.    3P3/2
                             M4  4  3374.    3D3/2
                             M5  6  3223.    3D5/2
                             N1  2  1274.    4S
                             N2  2  1116.    4P1/2
                             N3  4   928.    4P3/2
                             N4  4   680.    4D3/2
                             N5  6   645.    4D5/2
                             N6  6   322.    4F5/2
                             N7  8   313.    4F7/2
                             O1  2   274.    5S
                             O2  2   218.    5P1/2
                             O3  4   174.    5P3/2
                             O4  4    88.    5D3/2
                             O5  6    83.    5D5/2
                             P1  2    45.    6S
                             P2  2    29.    6P1/2
                             P3  4    22.    6P3/2
                             Q1  2     6.3   7S
                             P4  1     5.7   6D3/2-6D5/2
----------------------------------------------------------------
TH 232.05  90   00.0   0.00  K   2 109651.   1S		a)
                             L1  2 20472.    2S		a)
                             L2  2 19693.    2P1/2	a)
                             L3  4 16300.    2P3/2	a)
                             M1  2  5182.    3S  	a)
                             M2  2  4830.    3P1/2	a)
                             M3  4  4049.    3P3/2
                             M4  4  3494.    3D3/2
                             M5  6  3335.    3D5/2
                             N1  2  1333.    4S
                             N2  2  1171.    4P1/2
                             N3  4   970.    4P3/2
                             N4  4   717.    4D3/2
                             N5  6   679.    4D5/2
                             N6  6   347.    4F5/2
                             N7  8   338.    4F7/2
                             O1  2   293.    5S
                             O2  2   233.    5P1/2
                             O3  4   185.    5P3/2
                             O4  4    97.    5D3/2
                             O5  6    91.    5D5/2
                             P1  2    50.    6S
                             P2  2    33.    6P1/2
                             P3  4    25.    6P3/2
                             P4  2     6.    6D3/2-6D5/2
                             Q1  2     6.    7S
----------------------------------------------------------------
PA 231.    91   00.0   0.00  K   2 112601.   1S		a)
                             L1  2 21105.    2S		a)
                             L2  2 20314.    2P1/2	a)
                             L3  4 16733.    2P3/2	a)
                             M1  2  5367.    3S  	a)
                             M2  2  5001.    3P1/2	a)
                             M3  4  4178.    3P3/2
                             M4  4  3613.    3D3/2
                             M5  6  3446.    3D5/2
                             N1  2  1390.    4S
                             N2  2  1225.    4P1/2
                             N3  4  1011.    4P3/2
                             N4  4   752.    4D3/2
                             N5  6   712.    4D5/2
                             N6  6   372.    4F5/2
                             N7  8   362.    4F7/2
                             O1  2   312.    5S
                             O2  2   248.    5P1/2
                             O3  4   195.    5P3/2
                             O4  4   104.    5D3/2
                             O5  6    97.    5D5/2
                             P1  2    50.    6S
                             P2  2    32.    6P1/2
                             P3  4    24.    6P3/2
                             O6  2     6.    5F5/2
                             P4  1     6.    6D3/2-6D5/2
                             Q1  2     6.    7S
----------------------------------------------------------------
U  238.07  92   00.0   0.00  K   2 115606.   1S		a)
                             L1  2 21757.    2S		a)
                             L2  2 20948.    2P1/2	a)
                             L3  4 17166.    2P3/2	a)
                             M1  2  5548.    3S  	a)
                             M2  2  5182.    3P1/2	a)
                             M3  4  4308.    3P3/2
                             M4  4  3733.    3D3/2
                             M5  6  3557.    3D5/2
                             N1  2  1446.    4S
                             N2  2  1278.    4P1/2
                             N3  4  1050.    4P3/2
                             N4  4   785.    4D3/2
                             N5  6   743.    4D5/2
                             N6  6   396.    4F5/2
                             N7  8   386.    4F7/2
                             O1  2   329.    5S
                             O2  2   261.    5P1/2
                             O3  4   203.    5P3/2
                             O4  4   110.    5D3/2
                             O5  6   101.    5D5/2
                             P1  2    52.    6S
                             P2  2    34.    6P1/2
                             P3  4    24.    6P3/2
                             P4  1     6.1   6D3/2-6D5/2
                             O6  3     6.    5F5/2
                             Q1  2     6.    7S
----------------------------------------------------------------
NP 237.    93   00.0   0.00  K   2 118678.   1S		a)
                             L1  2 22426.    2S		a)
                             L2  2 21600.    2P1/2	a)
                             L3  4 17610.    2P3/2	a)
                             M1  2  5723.    3S  	a)
                             M2  2  5366.    3P1/2	a)
                             M3  4  4440.    3P3/2
                             M4  4  3854.    3D3/2
                             M5  6  3669.    3D5/2
                             N1  2  1504.    4S
                             N2  2  1331.    4P1/2
                             N3  4  1089.    4P3/2
                             N4  4   819.    4D3/2
                             N5  6   774.    4D5/2
                             N6  6   421.    4F5/2
                             N7  8   410.    4F7/2
                             O1  2   346.    5S
                             O2  2   274.    5P1/2
                             O3  4   211.    5P3/2
                             O4  4   116.    5D3/2
                             O5  6   106.    5D5/2
                             P1  2    54.    6S
                             P2  2    35.    6P1/2
                             P3  4    25.    6P3/2
                             O6  4     6.    5F5/2
                             P4  1     6.    6D3/2-6D5/2
                             Q1  2     6.    7S
----------------------------------------------------------------
PU 244.    94   00.0   0.00  K   2 121818.   1S		a)
                             L1  2 23097.    2S		a)
                             L2  2 22266.    2P1/2	a)
                             L3  4 18056.    2P3/2	a)
                             M1  2  5933.    3S  	a)
                             M2  2  5541.    3P1/2	a)
                             M3  4  4557.    3P3/2	a)
                             M4  4  3977.    3D3/2
                             M5  6  3783.    3D5/2
                             N1  2  1563.    4S
                             N2  2  1384.    4P1/2
                             N3  4  1128.    4P3/2
                             N4  4   853.    4D3/2
                             N5  6   805.    4D5/2
                             N6  6   446.    4F5/2
                             N7  8   434.    4F7/2
                             O1  2   356.    5S
                             O2  2   287.    5P1/2
                             O3  4   219.    5P3/2
                             O4  4   122.    5D3/2
                             O5  6   111.    5D5/2
                             P1  2    53.    6S
                             P2  2    34.    6P1/2
                             P3  4    23.    6P3/2
                             O6  6     6.    5F5/2
                             Q1  2     6.    7S
----------------------------------------------------------------
AM 243.    95   00.0   0.00  K   2 125027.   1S		a)
                             L1  2 23773.    2S		a)
                             L2  2 22944.    2P1/2	a)
                             L3  4 18504.    2P3/2	a)
                             M1  2  6121.    3S  	a)
                             M2  2  5710.    3P1/2	a)
                             M3  4  4667.    3P3/2	a)
                             M4  4  4102.    3D3/2
                             M5  6  3898.    3D5/2
                             N1  2  1623.    4S
                             N2  2  1439.    4P1/2
                             N3  4  1167.    4P3/2
                             N4  4   887.    4D3/2
                             N5  6   836.    4D5/2
                             N6  6   467.    4F5/2
                             N7  8   452.    4F7/2
                             O1  2   355.    5S
                             O2  2   301.    5P1/2
                             O3  4   220.    5P3/2
                             O4  4   123.    5D3/2
                             O5  6   112.    5D5/2
                             P1  2    54.    6S
                             P2  2    44.    6P1/2
                             P3  4    36.    6P3/2
                             O6  7     6.    5F5/2-5F7/2
                             Q1  2     6.    7S
----------------------------------------------------------------
CM 247.    96   00.0   0.00  K   2 128220.   1S		a)
                             L1  2 24460.    2S		a)
                             L2  2 23779.    2P1/2	a)
                             L3  4 18930.    2P3/2	a)
                             M1  2  6288.    3S  	a)
                             M2  2  5895.    3P1/2	a)
                             M3  4  4797.    3P3/2	a)
                             M4  4  4236.    3D3/2
                             M5  6  4014.    3D5/2
                             N1  2  1664.    4S
                             N2  2  1493.    4P1/2
                             N3  4  1194.    4P3/2
                             N4  4   919.    4D3/2
                             N5  6   864.    4D5/2
                             N6  6   494.    4F5/2
                             N7  8   479.    4F7/2
                             O1  2   384.    5S
                             O2  2   314.    5P1/2
                             O3  4   239.    5P3/2
                             O4  4   126.    5D3/2
                             O5  6   119.    5D5/2
                             P1  2    60.    6S
                             P2  2    39.    6P1/2
                             P3  4    27.    6P3/2
                             O6  7    11.    5F5/2-5F7/2
                             Q1  2     6.    7S
                             P4  1     5.    6D3/2-6D5/2
----------------------------------------------------------------
BK 249.    97   00.0   0.00  K   2 131590.   1S		a)
                             L1  2 25275.    2S		a)
                             L2  2 24385.    2P1/2	a)
                             L3  4 19452.    2P3/2	a)
                             M1  2  6556.    3S  	a)
                             M2  2  6147.    3P1/2	a)
                             M3  4  4977.    3P3/2	a)
                             M4  4  4366.    3D3/2
                             M5  6  4133.    3D5/2
                             N1  2  1729.    4S
                             N2  2  1554.    4P1/2
                             N3  4  1236.    4P3/2
                             N4  4   955.    4D3/2
                             N5  6   898.    4D5/2
                             N6  6   520.    4F5/2
                             N7  8   504.    4F7/2
                             O1  2   401.    5S
                             O2  2   329.    5P1/2
                             O3  4   248.    5P3/2
                             O4  4   142.    5D3/2
                             O5  6   124.    5D5/2
                             P1  2    63.    6S
                             P2  2    41.    6P1/2
                             P3  4    27.    6P3/2
                             O6  8    12.    5F5/2-5F7/2
                             Q1  2     6.    7S
                             P4  1     4.    6D3/2-6D5/2
----------------------------------------------------------------
CF 249.    98   00.0   0.00  K   2 135960.   1S		a)
                             L1  2 26110.    2S		a)
                             L2  2 25250.    2P1/2	a)
                             L3  4 19930.    2P3/2	a)
                             M1  2  6754.    3S  	a)
                             M2  2  6359.    3P1/2	a)
                             M3  4  5109.    3P3/2	a)
                             M4  4  4492.    3D3/2
                             M5  6  4247.    3D5/2
                             N1  2  1789.    4S
                             N2  2  1610.    4P1/2
                             N3  4  1273.    4P3/2
                             N4  4   987.    4D3/2
                             N5  6   925.    4D5/2
                             N6  6   546.    4F5/2
                             N7  8   529.    4F7/2
                             O1  2   412.    5S
                             O2  2   338.    5P1/2
                             O3  4   251.    5P3/2
                             O4  4   142.    5D3/2
                             O5  6   129.    5D5/2
                             P1  2    61.    6S
                             P2  2    39.    6P1/2
                             P3  4    25.    6P3/2
                             O6 10     9.    5F5/2-5F7/2
                             Q1  2     6.    7S
----------------------------------------------------------------
ES 254.    99   00.0   0.00  K   2 139490.   1S		a)
                             L1  2 26900.    2S		a)
                             L2  2 26020.    2P1/2	a)
                             L3  4 20410.    2P3/2	a)
                             M1  2  6977.    3S  	a)
                             M2  2  6754.    3P1/2	a)
                             M3  4  5252.    3P3/2	a)
                             M4  4  4630.    3D3/2  	a)
                             M5  6  4369.    3D5/2
                             N1  2  1857.    4S
                             N2  2  1674.    4P1/2
                             N3  4  1316.    4P3/2
                             N4  4  1024.    4D3/2
                             N5  6   959.    4D5/2
                             N6  6   573.    4F5/2
                             N7  8   554.    4F7/2
                             O1  2   429.    5S
                             O2  2   353.    5P1/2
                             O3  4   260.    5P3/2
                             O4  4   148.    5D3/2
                             O5  6   135.    5D5/2
                             P1  2    63.    6S
                             P2  2    40.    6P1/2
                             P3  4    25.    6P3/2
                             O6 11     9.    5F5/2-5F7/2
                             Q1  2     6.    7S
----------------------------------------------------------------
FM 253.   100   00.0   0.00  K   2 143090.   1S		a)
                             L1  2 27700.    2S		a)
                             L2  2 26810.    2P1/2	a)
                             L3  4 20900.    2P3/2	a)
                             M1  2  7205.    3S  	a)
                             M2  2  6793.    3P1/2	a)
                             M3  4  5397.    3P3/2	a)
                             M4  4  4766.    3D3/2 	a)
                             M5  6  4498.    3D5/2
                             N1  2  1933.    4S
                             N2  2  1746.    4P1/2
                             N3  4  1366.    4P3/2
                             N4  4  1068.    4D3/2
                             N5  6  1000.    4D5/2
                             N6  6   606.    4F5/2
                             N7  8   587.    4F7/2
                             O1  2   453.    5S
                             O2  2   375.    5P1/2
                             O3  4   275.    5P3/2
                             O4  4   160.    5D3/2
                             O5  6   145.    5D5/2
                             P1  2    69.    6S
                             P2  2    45.    6P1/2
                             P3  4    29.    6P3/2
                             O6 12    15.    5F5/2-5F7/2
                             Q1  2     7.    7S

**********************************************************       */


#endif   // end of G4AtomicShells.hh 
