(* (c) Microsoft Corporation. All rights reserved *)

(*-------------------------------------------------------------------------
!* Primary relations on types and signatures (with the exception of
 * the constraint solving engine and method overload resolution)
 *------------------------------------------------------------------------- *)

(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Il = Microsoft.Research.AbstractIL.IL 
F#*) 
open Ildiag
open List
open Range
open Ast
open Tast
open Tastops
open Env
open Il (* Abstract IL  *)
open Lib
open Infos
open Printf

(*-------------------------------------------------------------------------
!* a :> b without coercion based on finalized (no type variable) types
 *------------------------------------------------------------------------- *)


(* QUERY: This relation is barely used in the implementation and is *)
(* not part of the offical specification. It is in general only used for *)
(* optimizations and warnings. But there are a couple of minor exceptions that we *)
(* need to look into. We can probably get rid of it *)
let rec type_definitely_subsumes_type_no_coercion ndeep g amap m ty1 ty2 = 
    if ndeep > 100 then error(Error("recursive class hierarchy (detected in type_definitely_subsumes_type_no_coercion), ty1 = "^(DebugPrint.showType ty1),m));
    let ty1 = strip_tpeqns_and_tcabbrevs ty1 in 
    let ty2 = strip_tpeqns_and_tcabbrevs ty2 in
    match ty1,ty2 with 
    | TType_app (tc1,l1)  ,TType_app (tc2,l2) when g.tcref_eq tc1 tc2  ->  
        length l1 = length l2 &&
        for_all2 (type_equiv g) l1 l2
    | TType_tuple l1    ,TType_tuple l2     -> 
        length l1 = length l2 &&
        for_all2 (type_equiv g) l1 l2 
    | TType_fun (d1,r1)  ,TType_fun (d2,r2)   -> 
        type_equiv g d1 d2 && type_equiv g r1 r2
    | _ ->  
        (type_equiv g ty1 g.obj_ty && is_ref_typ g ty2) or (* F# reference types are subtypes of type 'obj' *)
        (is_stripped_tyapp_typ ty2 &&
         is_ref_typ g ty2 && 
         let tcref,tinst = dest_stripped_tyapp_typ ty2 in 
         begin match super_of_typ g amap m ty2 with 
         | None -> false
         | Some ty -> type_definitely_subsumes_type_no_coercion (ndeep+1) g amap m ty1 ty
         end or
         exists 
           (type_definitely_subsumes_type_no_coercion (ndeep+1) g amap m ty1)
           (implements_of_typ g amap m ty2)) 



(*-------------------------------------------------------------------------
!* Feasible coercion relation. Part of the language spec.
 *------------------------------------------------------------------------- *)

type canCoerce = CanCoerce | NoCoerce
let rec type_feasibly_subsumes_type ndeep g amap m ty1 canCoerce ty2 = 
    if ndeep > 100 then error(Error("recursive class hierarchy (detected in type_feasibly_subsumes_type), ty1 = "^(DebugPrint.showType ty1),m));
    let ty1 = strip_tpeqns_and_tcabbrevs ty1 in 
    let ty2 = strip_tpeqns_and_tcabbrevs ty2 in
    match ty1,ty2 with 
    | TType_var r , _  | _, TType_var r -> true
    | TType_app (tc1,l1)  ,TType_app (tc2,l2) when g.tcref_eq tc1 tc2  ->  
        length l1 = length l2 &&
        for_all2 (type_feasibly_equiv ndeep g amap m) l1 l2
    | TType_tuple l1    ,TType_tuple l2     -> 
        length l1 = length l2 &&
        for_all2 (type_feasibly_equiv ndeep g amap m) l1 l2 
    | TType_fun (d1,r1)  ,TType_fun (d2,r2)   -> 
        (type_feasibly_equiv ndeep g amap m) d1 d2 && (type_feasibly_equiv ndeep g amap m) r1 r2
    | _ -> 
        (* F# reference types are subtypes of type 'obj' *) 
        (type_equiv g ty1 g.obj_ty && (canCoerce = CanCoerce or is_ref_typ g ty2)) 
        or 
        (is_stripped_tyapp_typ ty2 &&
         (canCoerce = CanCoerce or is_ref_typ g ty2) && 
         let tcref,tinst = dest_stripped_tyapp_typ ty2 in 
         begin match super_of_typ g amap m ty2 with 
         | None -> false
         | Some ty -> type_feasibly_subsumes_type (ndeep+1) g amap m ty1 NoCoerce ty
         end or
         exists 
           (type_feasibly_subsumes_type (ndeep+1) g amap m ty1 NoCoerce)
           (implements_of_typ g amap m ty2)) 
                   
and type_feasibly_equiv ndeep g amap m ty1 ty2 = 
    (* this could be more efficient *)
    type_feasibly_subsumes_type (ndeep+1)  g amap m ty1 NoCoerce ty2 &&
    type_feasibly_subsumes_type (ndeep+1)  g amap m ty2 NoCoerce ty1


(*-------------------------------------------------------------------------
!* Choose solutions for TExpr_tchoose type "hidden" variables introduced
 * by letrec nodes. Also used by the pattern match compiler to choose type
 * variables when compiling patterns at generalized bindings.
 *     e.g. let ([],x) = ([],[])
 * Here x gets a generalized type "list<'a>".
 *------------------------------------------------------------------------- *)

let choose_typar_solution_and_range g amap tp =
     let m = range_of_typar tp in 
     if verbose then dprintf1 "choose_typar_solution, arbitrary: tp = %s\n" (Layout.showL (typarsL [tp]));
     let max,m = 
         List.fold_left (fun (maxSoFar,_) tpc -> 
             let join m x = 
               if type_feasibly_subsumes_type 0 g amap m x CanCoerce maxSoFar then maxSoFar
               else if type_feasibly_subsumes_type 0 g amap m maxSoFar CanCoerce x then x
               else (
                   errorR(Error(Printf.sprintf "The implicit instantiation of a generic construct at or near this point could not be resolved because it could resolve to multiple unrelated types, e.g. '%s' and '%s'. Consider using type annotations to resolve the ambiguity" (DebugPrint.showType x) (DebugPrint.showType maxSoFar),m)); maxSoFar
               ) in 
             (* Don't continue if an error occurred and we set the value eagerly *)
             if tpref_is_solved tp then maxSoFar,m else
             match tpc with 
             | TTyparCoercesToType(TTyparSubtypeConstraintFromFS(x),m) -> 
                 join m x,m
             | TTyparCoercesToType(TTyparSubtypeConstraintFromIL(_),m) -> 
                 warning(Error("choose_typar_solution: unexpected TTyparSubtypeConstraintFromIL",m));
                 maxSoFar,m
             | TTyparMayResolveMemberConstraint(TTrait(_,nm,_,argtys,rty),m) -> 
                 errorR(Error("Could not resolve the ambiguity inherent in the use of the overloaded operator '"^demangle_operator nm^"' at or near this program point. Consider using type annotations to resolve the ambiguity",m));
                 maxSoFar,m
             | TTyparSimpleChoice(_,m) -> 
                 errorR(Error("Could not resolve the ambiguity inherent in the use of a 'printf'-style format string",m));
                 maxSoFar,m
             | TTyparSupportsNull _ -> 
                 maxSoFar,m
             | TTyparIsEnum(_,m) -> 
                 errorR(Error("Could not resolve the ambiguitiy in the use of a generic construct with an 'enum' cosntraint at or near this position",m));
                 maxSoFar,m
             | TTyparIsDelegate(_,_,m) -> 
                 errorR(Error("Could not resolve the ambiguitiy in the use of a generic construct with a 'delegate' cosntraint at or near this position",m));
                 maxSoFar,m
             | TTyparIsNotNullableValueType m -> 
                 join m g.int_ty,m
             | TTyparRequiresDefaultConstructor m -> 
                 (* errorR(Error("Could not resolve the ambiguity inherent in the use of a generic construct at or near this program point. Consider using type annotations to resolve the ambiguity",m)); *)
                 maxSoFar,m
             | TTyparIsReferenceType _ -> 
                 maxSoFar,m
             | TTyparDefaultsToType(priority,ty,m) -> 
                 maxSoFar,m)
           (g.obj_ty,m)
           (constraints_of_typar tp) in 
    max,m

let choose_typar_solution g amap tp = fst (choose_typar_solution_and_range g amap tp)

let choose_typar_solutions_for_tchoose g amap e = 
    match e with 
    | TExpr_tchoose(tps,e1,m)  -> 
    
        (* Only make choices for variables that are actually used in the expression *)
        let ftvs = (free_in_expr e1).free_tyvars.free_loctypars in 
        let tps = tps |> List.filter (Zset.mem_of ftvs) in 
        
        let tpenv = mk_typar_inst tps (map (choose_typar_solution g amap) tps) in 
        inst_expr g tpenv e1
    | _ -> e
                 

(*-------------------------------------------------------------------------
!* Break apart lambdas. Needs choose_typar_solutions_for_tchoose because it's used in 
 * PostTypecheckSemanticChecks before we've eliminated these nodes.
 *------------------------------------------------------------------------- *)

(* Note a true error: we currently catch this at one place so give it a name *)
exception TopLambdaFailure of string * range

let try_dest_top_lambda_upto g amap (TopValInfo (ntps,_,_) as tvd) (e,ty) =
    let rec strip_lambda_upto n (e,ty) = 
        match e with 
        | TExpr_lambda (_,None,v,b,_,rty,_) when n > 0 -> 
            let (vs',b',rty') = strip_lambda_upto (n-1) (b,rty) in 
            (v :: vs', b', rty') 
        | _ -> ([],e,ty) in 

    let rec start_strip_lambda_upto n (e,ty) = 
        match e with 
        | TExpr_lambda (_,basevopt,v,b,_,rty,_) when n > 0 -> 
            let (vs',b',rty') = strip_lambda_upto (n-1) (b,rty) in 
            (basevopt, (v :: vs'), b', rty') 
        | TExpr_tchoose (tps,b,_) -> 
            start_strip_lambda_upto n (choose_typar_solutions_for_tchoose g amap e, ty)
        | _ -> (None,[],e,ty) in 

    let n = TopValData.numCurriedArgs tvd in 
    let tps,taue,tauty = match e with TExpr_tlambda (_,tps,b,_,rty,_) when ntps <> 0 -> tps,b,rty | _ -> [],e,ty in 
    let basevopt,vsl,body,rty = start_strip_lambda_upto n (taue,tauty) in
    if length vsl <> n then None else
        
    Some (tps,basevopt,vsl,body,rty)

let dest_top_lambda_upto g amap topValInfo (e,ty) = 
    match try_dest_top_lambda_upto g amap topValInfo (e,ty) with 
    | None -> error(Error("Couldn't strip to expected arity, expr = \n"^Layout.showL(exprL e), range_of_expr e));
    | Some res -> res
    
(* iterated_adjust_arity_of_lambda_body: Do adjust_arity_of_lambda_body for a series of  *)
(* iterated lambdas, producing one method.  *)

(* The required iterated function arity (length arity_info) must be identical *)
(* to the iterated function arity of the input lambda (length vsl) *)
let iterated_adjust_arity_of_lambda_body g amap niceNameGen arity_info e =
  let tps,basevopt,vsl,body,bodyty = dest_top_lambda_upto g amap arity_info (e, type_of_expr g e) in
  let arities = TopValData.aritiesOfArgs arity_info in 
  if List.length arities <> List.length vsl then (
    errorR(Error(sprintf "internal error:iterated_adjust_arity_of_lambda_body, length arities = %d, length vsl = %d" (length arities) (length vsl), range_of_expr body))
  );
  let vsl,body = 
    fold_right2 (fun arities vs (allvs,body) -> 
      let vs,body = adjust_arity_of_lambda_body g niceNameGen arities vs body in 
      vs :: allvs, body)
    arities
    vsl
    ([],body) in 
  tps,basevopt,vsl,body,bodyty



(*-------------------------------------------------------------------------
!* Containment relation for module types.
 *------------------------------------------------------------------------- *)

exception RequiredButNotSpecified of displayEnv * Tast.modul_ref * string * (Buffer.t -> unit) * range
exception ValueNotContained of displayEnv * Tast.modul_ref * val_spec * val_spec * string
exception ConstrNotContained of displayEnv * unionconstr_spec * unionconstr_spec * string
exception ExnconstrNotContained of displayEnv * tycon_spec * tycon_spec * string
exception FieldNotContained of displayEnv * recdfield_spec * recdfield_spec * string 
exception InterfaceNotRevealed of displayEnv * Tast.typ * range


(* REVIEW: we should be measuring compatibility w.r.t. to only the 'realized' *)
(* typars, i.e. the set that are involved in types AFTER the elimination of *)
(* (visible/known) type equations. The elimination of type equations may result in the *)
(* disappearance of some phantom type variables, none of which should be in the *)
(* actual realized type parameters of the compiled version of the method. *)
(* At the moment we are being inconsistent: the type parameter sets must be identical *)
(* before eliminating type equations, but the types must be identical _after_ *)
(* eliminating (visible/known) type equations. *)
let rec subtyp_forall_typars g denv m aenv atps ftps = 
    if length atps <> length ftps then  (errorR (Error("The signature and implementation are not compatible because the respective type parameter counts differ",m)); false)
    else 
      let aenv = bind_tyeq_env_typars atps ftps aenv in 
      for_all2 (fun atp ftp -> 
          let m = range_of_typar ftp in 
          if static_req_of_typar atp <> static_req_of_typar ftp then 
              errorR (Error("The signature and implementation are not compatible because the type parameter in the class/signature has a different compile-time requirement to the one in the member/implementation", m));          
          for_all (fun atpc -> 
              match atpc with 
              (* defaults can be dropped in the signature *)
              | TTyparDefaultsToType(_,acty,_) -> true
              | _ -> 
                  if not (exists  (typarConstraints_aequiv g aenv atpc) (constraints_of_typar ftp))
                  then (errorR(Error("The declaration of the type parameter '"^name_of_typar ftp^"' requires a constraint of the form "^Layout.showL(NicePrint.constraintL denv (atp,atpc)),m)); false)
                  else  true)
            (constraints_of_typar atp))
        atps
        ftps


and subtyp_tycon g amap denv aenv atc ftc =
    let m = (range_of_tycon ftc) in 
    let err s =  Error("The type definitions in the signature and implementation are not compatible because "^s,m) in 
    if name_of_tycon atc <> name_of_tycon ftc then  (errorR (err "the names differ"); false) else
    subtyp_exnc_repr g m denv  (fun s -> ExnconstrNotContained(denv,atc,ftc,s)) aenv (exn_repr_of_tycon atc) (exn_repr_of_tycon ftc) &&
    let atps = (typars_of_tycon atc) in 
    let ftps = (typars_of_tycon ftc) in 
    if length atps <> length ftps then  (errorR (err("the respective type parameter counts differ")); false)
    else if isLessAccessible (access_of_tycon atc) (access_of_tycon ftc) then (errorR(err "the accessibility specified in the signature is more than that specified in the implementation"); false)
    else 
      let aenv = bind_tyeq_env_typars atps ftps aenv in 
      let aintfs = map p13 (filter (fun (_,compgen,_) -> not compgen) (tcaug_of_tycon atc).tcaug_implements) in 
      let fintfs = map p13 (filter (fun (_,compgen,_) -> not compgen) (tcaug_of_tycon ftc).tcaug_implements) in 
      let aintfs = gen_setify (type_equiv g) (mapConcat (super_types_of_typ g amap m) aintfs) in
      let fintfs = gen_setify (type_equiv g) (mapConcat (super_types_of_typ g amap m) fintfs) in
    
      let unimpl = gen_subtract (fun fity aity -> type_aequiv g aenv aity fity) fintfs aintfs in 
      (unimpl |> for_all (fun ity -> errorR (err ("the signature requires that the type supports the interface "^NicePrint.pretty_string_of_typ denv ity^" but the interface has not been implemented")); false)) &&
      let hidden = gen_subtract (type_aequiv g aenv) aintfs fintfs in 
      hidden |> iter (fun ity -> (if is_fsobjmodel_interface_tycon atc then error else warning) (InterfaceNotRevealed(denv,ity,range_of_tycon atc)));
      let aNull = isUnionThatUsesNullAsRepresentation g atc in
      let fNull = isUnionThatUsesNullAsRepresentation g ftc in
      if aNull && not fNull then 
        errorR(err("the implementation says this type may use nulls as a representation but the signature does not"))
      else if fNull && not aNull then 
        errorR(err("the signature says this type may use nulls as a representation but the implementation does not"));
      if is_sealed_typ g (snd (generalize_tcref (mk_local_tcref atc))) <> is_sealed_typ g (snd (generalize_tcref (mk_local_tcref ftc))) then 
        errorR(err("the 'Sealed' attributes in the signature and implementation differ. Consider adding the [<Sealed>] attribute to either the signature or the implementation"));
      let aPartial = is_partially_implemented_tycon atc in
      let fPartial = is_partially_implemented_tycon ftc in
      if aPartial && not fPartial then 
        errorR(err("the implementation is an abstract class but the signature is not. Consider adding the [<AbstractClass>] attribute to the signature"));
      if not aPartial && fPartial then 
        errorR(err("the signature is an abstract class but the implementation is not. Consider adding the [<AbstractClass>] attribute to the implementation"));
      if not (type_aequiv g aenv (super_of_tycon g atc) (super_of_tycon g ftc)) then 
          errorR (err("the types have different base types"));
      subtyp_forall_typars g denv m aenv atps ftps &&
      subtyp_tycon_repr g denv err aenv (repr_of_tycon atc) (repr_of_tycon ftc) &&
      subtyp_tycon_abbrev g denv err aenv (abbrev_of_tycon atc) (abbrev_of_tycon ftc) 
    
and string_of_arity = function
  | None -> "no arity"
  | Some l -> String.concat "/" (map string_of_int l)

and subtyp_arity err (id:ident) aarity farity = 
    match aarity,farity with 
    | _,None -> true
    | None, Some _ -> err("An arity was not inferred for this value")
    | Some (TopValInfo (ntps,_,_) as n), Some (TopValInfo (mtps,_,_) as m) ->
        let n = TopValData.aritiesOfArgs n in
        let m = TopValData.aritiesOfArgs m in 
        if ntps = mtps && length m <= length n && for_all2 (fun x y -> x <= y) m (fst (chop_at (length m) n)) then true
        else if ntps <> mtps then
          err("The number of generic parameters in the signature and implementation differ (the signature declares "^string_of_int mtps^" but the implementation declares "^string_of_int ntps) 
        else 
          err("The arities in the signature and implementation differ. The signature specifies that '"^id.idText^"' is function definition or lambda expression accepting at least "^string_of_int (length m)^" argument(s), but the implementation is a computed function value. To declare that a computed function value is a permitted implementation simply parenthesize its type in the signature, e.g.\n\tval "^id.idText^": int -> (int -> int)\ninstead of\n\tval "^id.idText^": int -> int -> int")

and subtyp_val g amap denv implModRef aenv aval fval =
    (* Propagate defn location information from implementation to signature through a mutability hack. *)
    (data_of_val fval).val_defn_range <- (data_of_val aval).val_defn_range;

    if verbose then  dprintf3 "checking value %s, %d, %d\n" (name_of_val aval) (stamp_of_val aval) (stamp_of_val fval);
    let mk_err denv s = ValueNotContained(denv,implModRef,aval,fval,s) in
    let err denv s = errorR(mk_err denv s); false in 
    let m = range_of_val aval in 
    if mutability_of_val aval <> mutability_of_val fval then (err denv "The mutability attributes differ")
    else if name_of_val aval <> name_of_val fval then (err denv "The names differ")
    else if isLessAccessible (access_of_val aval) (access_of_val fval) then (err denv "The accessibility specified in the signature is more than that specified in the implementation")
    else if mustinline(inlineFlag_of_val aval) <> mustinline(inlineFlag_of_val fval) then (err denv "The inline flags differ")
    else if literal_const_of_val aval <> literal_const_of_val fval then (err denv "The literal constant values and/or attributes differ")
    else if is_tyfunc_of_val aval <> is_tyfunc_of_val fval then (err denv "One is a type function and the other is not. The signature requires explicit type parameters if they are present in the implementation")
    else 
        let atps,atau = try_dest_forall_typ (type_of_val aval) in 
        let ftps,ftau = try_dest_forall_typ (type_of_val fval) in 
        if length atps <> length ftps then (err {denv with showTyparBinding=true} "The respective type parameter counts differ") else
        let aenv = bind_tyeq_env_typars atps ftps aenv in 
        subtyp_forall_typars g denv m aenv atps ftps &&
        let res = 
          if not (type_aequiv g aenv atau ftau) then err denv "The types differ" 
          else if not (subtyp_arity (err denv) (id_of_val aval)  (arity_of_val aval) (arity_of_val fval)) then false
          else if not (isext_of_val aval = isext_of_val fval) then err denv "One is an extension member and the other is not"
          else if not (subtyp_vspr_options g (err denv) (attribs_of_val aval, aval,member_info_of_val aval) (attribs_of_val fval,fval,member_info_of_val fval)) then false
          else true in
        (* Update the arity of the value to reflect the constraint of the signature *)
        (* REVIEW; surely this isn't needed anymore *)
        (data_of_val aval).val_arity <- (data_of_val fval).val_arity;
        res

and subtyp_exnc_repr g m denv err aenv arepr frepr =
    match arepr,frepr with 
    | TExnAsmRepr _, TExnFresh _ -> 
        (errorR (err "a .NET exception mapping is being hidden by a signature. The exception mapping must be visible to other modules"); false)
    | TExnAsmRepr tcr1, TExnAsmRepr tcr2  -> 
        if tcr1 <> tcr2 then  (errorR (err "the .NET representations differ"); false) else true
    | TExnAbbrevRepr _, TExnFresh _ -> 
        (errorR (err "the exception abbreviation is being hidden by the signature. The abbreviation must be visible to other .NET languages. Consider making the abbreviation visible in the signature"); false)
    | TExnAbbrevRepr ecr1, TExnAbbrevRepr ecr2 -> 
        if not (tcref_aequiv g aenv ecr1 ecr2) then 
          (errorR (err "the exception abbreviations in the signature and implementation differ"); false)
        else true
    | TExnFresh r1, TExnFresh  r2-> subtyp_fields_rigid g denv err aenv r1 r2
    | TExnNone,TExnNone -> true
    | _ -> 
        (errorR (err "the exception declrations differ"); false)

and aequiv_constr g denv aenv c1 c2 =
    let err msg = errorR(ConstrNotContained(denv,c1,c2,msg));false in 
    if c1.uconstr_id.idText <> c2.uconstr_id.idText then  err "The names differ"
    else if length (rfields_of_uconstr c1) <> length (rfields_of_uconstr c2) then err "The respective number of data fields differ"
    else if not (for_all2 (aequiv_field g denv aenv) (rfields_of_uconstr c1) (rfields_of_uconstr c2)) then err "The types of the fields differ"
    else if isLessAccessible (access_of_uconstr c1) (access_of_uconstr c2) then err "the accessibility specified in the signature is more than that specified in the implementation"
    else true

and aequiv_field g denv aenv f1 f2 =
    let err msg = errorR(FieldNotContained(denv,f1,f2,msg)); false in 
    if f1.rfield_id.idText <> f2.rfield_id.idText then err "The names differ"
    else if isLessAccessible (access_of_rfield f1) (access_of_rfield f2) then err "the accessibility specified in the signature is more than that specified in the implementation"
    else if f1.rfield_static <> f2.rfield_static then err "The 'static' modifiers differ"
    else if literal_value_of_rfield f1 <> literal_value_of_rfield f2 then err "The 'literal' modifiers differ"
    else if not (type_aequiv g aenv f1.rfield_type f2.rfield_type) then err "The types differ"
    else true

and subtyp_vspr_options g err  (aattrs,aval,avsprOpt) (fattrs, fval,fvsprOpt)  =
    match avsprOpt,fvsprOpt with 
    | None,None -> true
    | Some avspr, Some fvspr -> subtyp_vspr g err (aattrs,aval,avspr) (fattrs,fval,fvspr)
    | _ -> false

and subtyp_vspr g err (aattrs,aval,avspr) (fattrs,fval,fvspr)  =
    if not (avspr.vspr_il_name = fvspr.vspr_il_name) then 
      err("The .NET member names differ")
    else if not (avspr.vspr_flags.memFlagsOverloadQualifier = fvspr.vspr_flags.memFlagsOverloadQualifier) then 
      err("The overload resolution identifier attributes differ")
    else if not (avspr.vspr_flags.memFlagsInstance = fvspr.vspr_flags.memFlagsInstance) then 
      err("One is static and the other isn't")
    else if not (avspr.vspr_flags.memFlagsVirtual = fvspr.vspr_flags.memFlagsVirtual) then 
      err("One is virtual and the other isn't")
    else if not (avspr.vspr_flags.memFlagsAbstract = fvspr.vspr_flags.memFlagsAbstract) then 
      err("One is abstract and the other isn't")
   (* I've weakened this check: *)
   (*     classes have non-final CompareTo/Hash methods *)
   (*     abstract have non-final CompareTo/Hash methods *)
   (*     records  have final CompareTo/Hash methods *)
   (*     unions  have final CompareTo/Hash methods *)
   (* Therefore it is OK for the signaure to say 'non-final' when the implementation says 'final' *)
    else if not avspr.vspr_flags.memFlagsFinal && fvspr.vspr_flags.memFlagsFinal then 
      err("One is final and the other isn't")
    else if not (avspr.vspr_flags.memFlagsOverride = fvspr.vspr_flags.memFlagsOverride) then 
      err("One is marked as an override and the other isn't")
    else if not (avspr.vspr_flags.memFlagsKind = fvspr.vspr_flags.memFlagsKind) then 
      err("One is a constructor/property and the other is not")
    else  
       let finstance = valCompiledAsInstance g fval in
       let ainstance = valCompiledAsInstance g aval in
       if  finstance && not ainstance then 
        err "The compiled representation of this method is as a static member but the signature indicates its compiled representation is as an instance member"
    else if not finstance && ainstance then 
        err "The compiled representation of this method is as an instance member, but the signature indicates its compiled representation is as a static member"
    else true

and subtyp_fields_rigid g denv err aenv afields ffields =
    let afields = true_rfields_of_rfield_tables afields in 
    let ffields = true_rfields_of_rfield_tables ffields in 
    let m1 = Namemap.of_keyed_list name_of_rfield afields in 
    let m2 = Namemap.of_keyed_list name_of_rfield ffields in 
    Namemap.suball2 (fun s _ -> errorR(err ("The field "^s^" was required by the signature but was not specified by the implementation")); false) (aequiv_field g denv aenv)  m1 m2 &&
    Namemap.suball2 (fun s _ -> errorR(err ("The field "^s^" was present in the implementation but not in the signature")); false) (fun x y -> aequiv_field g denv aenv y x)  m2 m1 &&
    (* This check is required because constructors etc. are externally visible *)
    (* and thus compiled representations do pick up dependencies on the field order  *)
    (if for_all2 (fun f1 f2 -> aequiv_field g denv aenv f1 f2)  afields ffields
     then true
     else (errorR(err ("The order of the fields is different in the signature and implementation")); false))

and subtyp_vslots_rigid g denv err aenv avslots fvslots =
    let m1 = Namemap.of_keyed_list name_of_vref avslots in 
    let m2 = Namemap.of_keyed_list name_of_vref fvslots in 
    Namemap.suball2 (fun s vref -> errorR(err ("the abstract member '"^ Layout.showL(NicePrint.valSpecL denv (deref_val vref)) ^"' was required by the signature but was not specified by the implementation")); false) (fun x y -> true)  m1 m2 &&
    Namemap.suball2 (fun s vref -> errorR(err ("the abstract member '"^ Layout.showL(NicePrint.valSpecL denv (deref_val vref)) ^"' was present in the implementation but not in the signature")); false) (fun x y -> true)  m2 m1

and subtyp_fields_flex g denv err aenv afields ffields =
    let afields = true_rfields_of_rfield_tables afields in 
    let ffields = true_rfields_of_rfield_tables ffields in 
    let m1 = Namemap.of_keyed_list name_of_rfield afields in 
    let m2 = Namemap.of_keyed_list name_of_rfield ffields in 
    Namemap.suball2 (fun s _ -> errorR(err ("the field "^s^" was required by the signature but was not specified by the implementation")); false) (aequiv_field g denv aenv)  m1 m2 

and subtyp_tycon_repr g denv err aenv arepr frepr =
    let reportNiceError k s1 s2 = 
      let aset = nameset_of_list s1 in 
      let fset = nameset_of_list s2 in 
      match Zset.elements (Zset.diff aset fset) with 
      | [] -> 
          begin match Zset.elements (Zset.diff fset aset) with             
          | [] -> (errorR (err ("the number of "^k^"s differ")); false)
          | l -> (errorR (err ("the signature defines the "^k^" '"^String.concat ";" l^"' but the implementation does not (or does, but not in the same order)")); false)
          end
      | l -> (errorR (err ("the implementation defines the "^k^" '"^String.concat ";" l^"' but the signature does not (or does, but not in the same order)")); false) in 

    match arepr,frepr with 
    | Some (TRecdRepr _ | TFiniteUnionRepr _ | TIlObjModelRepr _ ), None  -> true
    | Some (TFsObjModelRepr r), None  -> 
        if r.tycon_objmodel_kind = TTyconStruct or r.tycon_objmodel_kind = TTyconEnum then 
          (errorR (err "the implementation defines a struct but the signature defines a type with a hidden representation"); false)
        else true
    | Some (TAsmRepr _), None -> 
        (errorR (err "a .NET type representation is being hidden by a signature"); false)
    | Some (TFiniteUnionRepr r1), Some (TFiniteUnionRepr r2) -> 
        let uconstrs1 = uconstrs_of_funion r1 in 
        let uconstrs2 = uconstrs_of_funion r2 in 
        if length uconstrs1 <> length uconstrs2 then
          let names l = map (fun c -> c.uconstr_id.idText) l in 
          reportNiceError "union case" (names uconstrs1) (names uconstrs2) 
        else for_all2 (aequiv_constr g denv aenv) uconstrs1 uconstrs2
    | Some (TRecdRepr afields), Some (TRecdRepr ffields) -> 
        subtyp_fields_rigid g denv err aenv afields ffields
    | Some (TFsObjModelRepr r1), Some (TFsObjModelRepr r2) -> 
        if not (begin match r1.tycon_objmodel_kind,r2.tycon_objmodel_kind with 
                           | TTyconClass,TTyconClass -> true
                           | TTyconInterface,TTyconInterface -> true
                           | TTyconStruct,TTyconStruct -> true
                           | TTyconEnum, TTyconEnum -> true
                           | TTyconDelegate (TSlotSig(nm1,typ1,ctps1,mtps1,ps1, rty1)), 
                             TTyconDelegate (TSlotSig(nm2,typ2,ctps2,mtps2,ps2, rty2)) -> 
                                (type_aequiv g aenv typ1 typ2) &&
                               (length ctps1 = length ctps2) &&
                               let aenv = bind_tyeq_env_typars ctps1 ctps2 aenv in 
                               ( (typar_decls_aequiv g aenv ctps1 ctps2) &&
                                (length mtps1 = length mtps2) &&
                               let aenv = bind_tyeq_env_typars mtps1 mtps2 aenv in 
                               ( (typar_decls_aequiv g aenv mtps1 mtps2) &&
                                (length ps1 = length ps2) &&
                                (for_all2 (fun p1 p2 -> type_aequiv g aenv (typ_of_slotparam p1) (typ_of_slotparam p2)) ps1 ps2) &&
                                (type_aequiv g aenv rty1 rty2)))
                           | _,_ -> false
                      end) then 
          (errorR (err "the types are of different kinds"); false)
        else 
          subtyp_fields_flex g denv err aenv r1.fsobjmodel_rfields r2.fsobjmodel_rfields &&
          subtyp_vslots_rigid g denv err aenv r1.fsobjmodel_vslots r2.fsobjmodel_vslots
    | Some (TAsmRepr tcr1),  Some (TAsmRepr tcr2) -> 
        if tcr1 <> tcr2 then  (errorR (err "the IL representations differ"); false) else true
    | Some _, Some _ -> (errorR (err "the representations differ"); false)
    | None, Some _ -> (errorR (err "the representations differ"); false)
    | None, None -> true

and subtyp_tycon_abbrev g denv err aenv abbrev1 abbrev2 =
    match abbrev1,abbrev2 with 
    | Some ty1, Some ty2 -> if not (type_aequiv g aenv ty1 ty2) then (errorR (err "the abbreviations differ"); false) else true
    | None,None -> true
    | Some _, None -> (errorR (err "a type abbreviation is being hidden by a signature. The abbreviation must be visible to other .NET languages. Consider making the abbreviation visible in the signature"); false)
    | None, Some _ -> (errorR (err "the signature has an abbreviation while the implementation does not"); false)

and subtyp_mtype m g amap denv aenv implModRef signModType = 
    let implModType = mtyp_of_modref implModRef in 
    (if implModType.mtyp_kind <> signModType.mtyp_kind then errorR(Error("The namespace attribute of the compilation unit differs between signature and implementation",m)));
    Namemap.suball2 (fun s fx -> errorR(RequiredButNotSpecified(denv,implModRef,"type",(fun os -> Printf.bprintf os "%s" s),m)); false) (subtyp_tycon g amap denv aenv)  implModType.mtyp_tycons signModType.mtyp_tycons &&
    Namemap.suball2 (fun s fx -> errorR(RequiredButNotSpecified(denv,implModRef,"module",(fun os -> Printf.bprintf os "%s" s),m)); false) (fun x1 x2 -> subtyp_modul g amap denv aenv (mk_local_modref x1) (mtyp_of_modul x2)) (submoduls_of_mtyp implModType) (submoduls_of_mtyp signModType) &&
    Namemap.suball2 (fun s fx -> errorR(RequiredButNotSpecified(denv,implModRef,"value",(fun os -> Printf.bprintf os "%a" (NicePrint.output_qualified_val_spec denv) fx),m)); false) (subtyp_val g amap denv implModRef aenv) implModType.mtyp_vals signModType.mtyp_vals

and subtyp_modul g amap denv aenv implModRef signModType = 
    subtyp_mtype (range_of_modref implModRef) g amap denv aenv implModRef signModType


(* REVIEW: remove this duplication *)
let rec check_names_mtype m denv implModRef signModType = 
    let implModType = mtyp_of_modref implModRef in 
    Namemap.suball2 (fun s fx -> errorR(RequiredButNotSpecified(denv,implModRef,"type",(fun os -> Printf.bprintf os "%s" s),m)); false) (fun _ _ -> true)  implModType.mtyp_tycons signModType.mtyp_tycons &&
    Namemap.suball2 (fun s fx -> errorR(RequiredButNotSpecified(denv,implModRef,"module",(fun os -> Printf.bprintf os "%s" s),m)); false) (fun x1 x2 -> check_names_modul denv (mk_local_ref x1) (mtyp_of_modul x2)) (submoduls_of_mtyp implModType) (submoduls_of_mtyp signModType) &&
    Namemap.suball2 (fun s fx -> errorR(RequiredButNotSpecified(denv,implModRef,"value",(fun os -> Printf.bprintf os "%a" (NicePrint.output_qualified_val_spec denv) fx),m)); false) (fun _ _ -> true) implModType.mtyp_vals signModType.mtyp_vals

and check_names_modul denv implModRef signModType = 
    check_names_mtype (range_of_modref implModRef) denv implModRef signModType


(*-------------------------------------------------------------------------
!* Completeness of classes
 *------------------------------------------------------------------------- *)

(** The overall information about a method implementation in a class or obeject expression *)
type override_info = Override of ident * (typars * typar_inst) * Tast.typ list * Tast.typ

let boutput_override denv os (Override(id,(mtps,memberToParentInst),argtys,rty)) = 
   let denv = { denv with showTyparBinding = true } in 
   let argInfos = map (fun ty -> (ty, TopValData.unnamedTopArg1)) argtys in
   Layout.bufferL os (NicePrint.memberSigL denv (memberToParentInst,id.idText,mtps, argInfos, rty))

let boutput_minfo_sig g amap m denv os minfo =
    let denv = { denv with showTyparBinding = true } in 
    let argtys,rty,fmtps,ttpinst = sig_of_minfo g amap m minfo in 
    let argInfos = map (fun ty -> (ty, TopValData.unnamedTopArg1)) argtys in
    let nm = name_of_minfo minfo in 
    Layout.bufferL os (NicePrint.memberSigL denv (ttpinst,nm,fmtps, argInfos, rty))


let string_of_override denv d = bufs (fun buf -> boutput_override denv buf d)
let string_of_minfo_sig g amap m denv d = bufs (fun buf -> boutput_minfo_sig g amap m denv buf d)

exception OverrideDoesntOverride of displayEnv * override_info * meth_info option * tcGlobals * Import.importMap * range

let dest_minfo_override g amap m minfo = 
    let nm = name_of_minfo minfo in 
    let argtys,rty,fmtps,ttpinst = sig_of_minfo g amap m minfo in 

    Override(mksyn_id m nm, (fmtps,ttpinst),argtys,rty)

let dest_vspr_vref_override g ovref = 
    let vspr = the (member_info_of_vref ovref) in 
    let arities = 
      match arity_of_vref ovref with 
      | Some a -> a
      | None -> error(Error("no arity recorded for implementation",range_of_vref ovref)) in 
    let _,arginfosl,rty,_ = dest_top_type arities (type_of_vref ovref)  in 
    let argtysl = map (map fst) arginfosl in
    let nm = (logical_name_of_vspr_vref ovref) in 
    (* dprintf2 "dest_vspr_vref_override (1): nm = %s, #tps = %d\n" nm (length tps); *)
    let argtysl = if nonNil(argtysl) then tl argtysl else argtysl in  (* Drop the 'this' pointer *)
    let argtys = concat argtysl in 
    let argtys = if length argtys = 1 && type_equiv g g.unit_ty (hd argtys) then [] else argtys in 
    let memberMethodTypars,memberToParentInst,argtys,rty = 
        match partition_vref_typars ovref with
        | Some(_,_,memberMethodTypars,memberToParentInst,tinst) -> 
            (* dprintf2 "dest_vspr_vref_override (2): nm = %s, #tps = %d\n" nm (length tps); *)
            memberMethodTypars, memberToParentInst,map (inst_type memberToParentInst) argtys, inst_type memberToParentInst rty 
        | None -> 
            (* dprintf2 "dest_vspr_vref_override (3): nm = %s, #tps = %d\n" nm (length tps); *)
            error(Error("this method is over-constrained in its type parameters",range_of_vref ovref)) in 
    Override(mksyn_id (range_of_vref ovref) nm, (memberMethodTypars,memberToParentInst),argtys,rty)

let dest_objexpr_override g amap ((id:ident),ty,e) = 
    if verbose then  dprintf0 "--> dest_objexpr_override\n";
    let tps,tauty = try_dest_forall_typ ty in 
    let thisty,tauty = dest_fun_typ tauty in 
    (* if id.idText = "RaiseError" then dprintf2 "id = %s, #tps = %d\n" id.idText (length tps); *)
    if not (is_fun_ty tauty) then error(Error("This override must be a function",id.idRange));
    let arity_info = TopValInfo (length tps,[TopValData.unnamedTopArg;TopValData.unnamedTopArg],TopValData.unnamedRetVal) in
    let _,basevopt,vsl,body,rty = dest_top_lambda_upto g amap arity_info (e,ty) in 
    match vsl with 
    | [[thisv];vs] -> 
        let vs = if length vs = 1 && type_equiv g (type_of_val (hd vs)) g.unit_ty then [] else vs in
        let argtys = map type_of_val vs in 
        Override(id,(tps,[]),argtys,rty),(basevopt,thisv,vs,body)
    | _ -> error(Error("This override must take one tuple of arguments",id.idRange))
      
let is_name_match virt (Override(id,_,_,_)) = 
    id.idText = name_of_minfo virt
      
let is_partial_match g amap m virt (Override(id,(mtps,_),argtys,rty) as overrideMeth) = 
    is_name_match virt overrideMeth &&
    let vargtys,_,fvmtps,_ = sig_of_minfo g amap m virt in
    length mtps = length fvmtps &&
    length argtys = length vargtys 
      
let reverse_renaming tinst = 
    tinst |> map (fun (tp,ty) -> (dest_typar_typ ty, mk_typar_ty tp))

let compose_inst inst1 inst2 = 
    inst1 |> map (map2'2 (inst_type inst2)) 
 
let is_exact_match g amap m virt (Override(id,(mtps,mtpinst),argtys,rty) as overrideMeth) =
    is_partial_match g amap m virt overrideMeth &&
    let vargtys,vrty,fvmtps,ttpinst = sig_of_minfo g amap m virt in

    (* Compare the types. sig_of_minfo, dest_objexpr_override and dest_vspr_vref_override have already *)
    (* applied all relevant substitutions except the renamings from fvtmps <-> mtps *)

    let aenv = 
       tyeq_env_empty  
       |> bind_tyeq_env_typars fvmtps mtps in 

    for_all2 (type_aequiv g aenv) vargtys argtys &&
    type_aequiv g aenv vrty rty &&
    
    (* Comparing the method typars and their constraints is much trickier since the substitutions have not been applied 
       to the constraints of these babies. This is partly because constraints are directly attached to typars so it's 
       difficult to apply substitutions to them unless we separate them off at some point, which we don't as yet.        

       Given   C<ctps>
               D<dtps>
               virt :   C<ctys[dtps]>.M<fvmtps[ctps]>(...)
               overrideMeth:  parent: D<dtys[dtps]>  value: !<ttps> <mtps[ttps]>(...) 
               
           where X[dtps] indicates that X may involve free type variables dtps
           
           we have 
               ttpinst maps  ctps --> ctys[dtps] 
               mtpinst maps  ttps --> dtps
           
           compare fvtmps[ctps] and mtps[ttps] by 
              fvtmps[ctps]  @ ttpinst     -- gives fvtmps[dtps]
              fvtmps[dtps] @ rev(mtpinst) -- gives fvtmps[ttps]
              
           Now fvtmps[ttps] and mtpinst[ttps] are comparable, i.e. have sontraints w.r.t. the same set of type variables 
               
      i.e.  Compose the substitutions ttpinst and rev(mtpinst) *)

          
    (* Compose the substitutions *)
    
    let ttpinst = 
        (* check we can reverse - in some error recovery situations we can't *)
        if mtpinst |> exists (snd >> is_typar_ty >> not) then ttpinst 
        else compose_inst ttpinst (reverse_renaming mtpinst) in

    (* Compare under the composed substitutions *)
    let aenv = bind_tyeq_env_tpinst ttpinst tyeq_env_empty in 
    
    (let res = typar_decls_aequiv g aenv fvmtps mtps in 
(*
      if not res then (
          ttpinst |> List.iter (fun (tp,ty) -> dprintf2 "  %s --> %s\n" (Layout.showL (typarL tp)) (Layout.showL (typeL ty)));
          dprintf1 "absslot.parent = %s\n" (Layout.showL (typeL (typ_of_minfo virt)));
          dprintf2 "absslot.typars = %s, overrideMeth.typars = %s\n" (Layout.showL (typarsL fvmtps)) (Layout.showL (typarsL mtps));
          dprintf2 "fvtmps = %s, mtps = %s\n" (Layout.showL (typarsL fvmtps)) (Layout.showL (typarsL mtps));
      );
*)
      res) 

      
let checkAbstractMembersAreImplemented showMissingMethods denv g amap (m,ty,virts,availPriorImplSlots,overrides) = 
    (* 6a. check all interface and abstract methods are implemented *)
    let res = ref true in 
    let fail exn = res := false ; if showMissingMethods then errorR exn in
    
    let intf = is_interface_typ ty  in 
    virts |> iter (fun virt -> 
      
      let priorOverrides = map (dest_minfo_override g amap m) availPriorImplSlots in 
      
      let allOverrides = (priorOverrides@overrides) in
      
      if verbose then dprintf1 "#overrides = %d\n" (length overrides);
      if (intf or minfo_is_abstract virt) then
        begin match filter (is_exact_match g amap m virt) overrides with
        | [h] -> () 
        | [] -> 
            if not (List.exists (is_exact_match g amap m virt) priorOverrides) then begin
                (* error reporting path *)
                let vargtys,vrty,fvmtps,_ = sig_of_minfo g amap m virt in
                let noimpl() = fail(Error("No implementation was given for '"^string_of_minfo g amap m denv virt^"'"^
                                            (if intf then ". Note that interface implementations must be given under the appropriate 'interface' declaration, e.g. 'interface ... with member ... end'" 
                                             else ""),m)) in 
                begin match  filter (is_partial_match g amap m virt) overrides with 
                | [] -> 
                    begin match  filter (is_name_match virt) overrides with 
                    | [] -> 
                        noimpl()
                    | [ Override(id,(mtps,_),argtys,_) as overrideMeth ] -> 
                        (* dprintf2 "#mtps = %d, generic_arity_of_minfo(virt) = %d\n" (length mtps) (generic_arity_of_minfo virt); *)
                        fail(Error("The member '"^string_of_override denv overrideMeth^"' does not have the correct number of "^(if length argtys <> length vargtys then "arguments" else "type parameters")^". The required signature is '"^string_of_minfo_sig g amap m denv virt^"'",id.idRange))
                    | Override(id,_,_,_) :: _ -> 
                        errorR(Error("No implementations of '"^string_of_minfo_sig g amap m denv virt^"' had the correct number of arguments and type parameters. The required signature is '"^string_of_minfo_sig g amap m denv virt^"'",id.idRange))
                    end
                | [ Override(id,_,_,_) as overrideMeth ] -> 
                    ()
                    (* Error will be reported below in checkOverridesAreAllUsedOnce *)
                    (* fail(OverrideDoesntOverride(denv, overrideMeth, Some(virt), g, amap, id.idRange)) *)
                    
                | _ -> fail(Error("The override for '"^string_of_minfo_sig g amap m denv virt^"' was ambiguous",m))
                end
            end
        | _ -> fail(Error("More than one override implements '"^string_of_minfo_sig g amap m denv virt^"'",m))
        end);
    !res

let checkOverridesAreAllUsedOnce denv g amap (m,ty,virts,overrides) = 
    (* 6b. check all implementations implement some virtual method *)
    overrides |> iter  (fun (Override(id,_,argtys,rty) as overrideMeth) -> 
        let m = id.idRange in 
        match filter (fun virt -> is_exact_match g amap m virt overrideMeth) virts with 
        | [] -> 
            begin match filter (fun virt -> is_partial_match g amap m virt overrideMeth) virts with 
            | virt :: _ -> 
                errorR(OverrideDoesntOverride(denv,overrideMeth,Some(virt),g,amap,m))
            | _ -> 
                begin match filter (fun virt -> is_name_match virt overrideMeth) virts with 
                | virt :: _ -> 
                    errorR(OverrideDoesntOverride(denv,overrideMeth,Some(virt),g,amap,m))
                | _ -> 
                    errorR(OverrideDoesntOverride(denv,overrideMeth,None,g,amap,m))
                end
            end
        | [h] -> ()
        | h1 :: h2 :: _ -> errorR(Error(Printf.sprintf "The override '%s' implements more than one abstract slot, e.g. '%s' and '%s'" (string_of_override denv overrideMeth) (string_of_minfo g amap m denv h1) (string_of_minfo g amap m denv h2),m)))

(*-------------------------------------------------------------------------
!* Get the slots of a type that can or must be implemented. This depends
 * partly on the full set of interface types that are being implemented
 * simultaneously, e.g.
 *    { new C with  interface I2 = ... interface I3 = ... }
 * allImplTys = {C;I2;I3}
 *------------------------------------------------------------------------- *)

let debug_getVirtualSlots = false 

type slotImplSet = SlotImplSet of meth_info list * meth_info list * prop_info list * prop_info list

let getImplSets denv g amap allImplTys = 

    let intfSets = 
        allImplTys |> list_mapi (fun i (ity,m) -> 
            let interfaces = super_types_of_typ g amap m ity |> filter is_interface_typ in 
            (i, ity, (ity :: interfaces),m)) in
    let implSets = 
        intfSets |> map (fun (i,ity,itys,m) -> 
            let reduced = fold_left (fun acc (j,jty,jtys,m) -> 
                               if i <> j && type_feasibly_subsumes_type 0 g amap m jty CanCoerce ity 
                               then gen_subtract (type_feasibly_equiv 0 g amap m) acc jtys
                               else acc ) itys intfSets in
            (i, ity, m, reduced)) in
    implSets |> iter (fun (i, ity, m, itys) -> 
        if is_interface_typ ity && isNil itys then 
            errorR(Error("Duplicate or redundant interface",m)));

    implSets |> iter (fun (i, ity, im, itys) -> 
                    implSets |> iter (fun (j,jty,jm,jtys) -> 
                        if i > j then  begin
                            let overlap = gen_intersect (type_feasibly_equiv 0 g amap im) itys jtys in
                            overlap |> iter (fun oty -> 
                                if nonNil(immediate_intrinsic_minfos_of_typ (None,DontIncludePrivate) g amap im oty |> filter minfo_is_virt) then                                
                                    errorR(Error("The interface "^NicePrint.pretty_string_of_typ denv (hd overlap)^" is included in multiple explicitly implemented interface types. Add an explicit implementation of this interface",im)))
                        end));

    implSets |> map (fun (_,ity,im,itys) -> 

        (** Work out if a method is relevant to the implementation relation for type 'ity' *)
        (** Virtual/abstract methods are relevant to class types *)
        (** All methods are relevant to interface types *)
        let isRelevantAbstractSlot x = 
            (minfo_is_virt x && not (is_interface_typ ity)) or 
            itys |> List.exists (fun ity -> is_interface_typ ity && type_feasibly_equiv 0 g amap im (typ_of_minfo x) ity) in

        let isRelevantImplSlot x = 
            (minfo_is_virt x && not (minfo_is_abstract x) && not (is_interface_typ ity)) 
            (* TODO: add inherited interface impls here *) in

        (** Compute the relevant abstract slots *)
        let reqd                   = 
            intrinsic_minfos_of_typ (None,IncludePrivate) IgnoreOverrides g amap im ity 
            |> filter isRelevantAbstractSlot in 
            
        (** Compute the relevant methods, including any overrides that exist *)
        let availPriorImplSlots = 
            intrinsic_minfos_of_typ (None,IncludePrivate) PreferOverrides g amap im ity 
            |> filter isRelevantImplSlot in 

        let pinfo_isRelevantAbstractSlot x = 
            (pinfo_is_virt x && not (is_interface_typ ity)) or 
            itys |> List.exists (fun ity -> is_interface_typ ity && type_feasibly_equiv 0 g amap im (typ_of_pinfo x) ity) in

        let pinfo_isRelevantImplSlot x = 
            (pinfo_is_virt x && not (pinfo_is_abstract x) && not (is_interface_typ ity))  in
            
        let pinfo_reqd                   = 
            intrinsic_pinfos_of_typ (None,DontIncludePrivate) IgnoreOverrides g amap im ity 
            |> filter pinfo_isRelevantAbstractSlot in 
            
        let pinfo_availPriorImplSlots = 
            intrinsic_pinfos_of_typ (None,DontIncludePrivate) PreferOverrides g amap im ity 
            |> filter pinfo_isRelevantImplSlot in 
            
            
        SlotImplSet(reqd, availPriorImplSlots, pinfo_reqd, pinfo_availPriorImplSlots))

(*-------------------------------------------------------------------------
!* Sets of methods involved in overload resolution and trait constraint
 * satisfaction.
 *------------------------------------------------------------------------- *)

(* 'a gets instantiated to: *)
(*    1. the expression being supplied for an argument *)
(*    2. when simply checking for the existence of an overload that satisfies *)
(*       a signature, or when finding the corresponding witness. *)
(* Note the parametricity helps ensure that overload resolution doesn't depend on the *)
(* expression on the callside (though it is in some circumstances allowed *)
(* to depend on some type information inferred syntactically from that *)
(* expression, e.g. a lambda expression may be converted to a delegate as *)
(* an adhoc conversion. *)

type 'a callerArg = CallerArg of Tast.typ * range * bool * 'a  (* bool indicates if named using a '?' *) 
type 'a calledArg = CalledArg of int * bool (* isParamArray *) * optionalArgInfo * bool (* isOutArg *) * string option * Tast.typ 
type 'a assignedCalledArg = AssignedCalledArg of 'a calledArg * 'a callerArg

type assignedItemSetterTarget = 
    | AssignedPropSetter of meth_info * Tast.tinst   (* the meth_info is a non-indexer setter property *)
    | AssignedIlFieldSetter of il_field_info 
    | AssignedRecdFieldSetter of recdfield_info 

type 'a assignedItemSetter = AssignedItemSetter of assignedItemSetterTarget * 'a callerArg 

type 'a callerNamedArg = CallerNamedArg of string * 'a callerArg  
type 'a calledMeth = 
    CalledMeth of 
        (* the method we're attempting to call *)
        meth_info * 
        (* the instantiation of the method we're attempting to call *)
        Tast.tinst * 
        (* the formal instantiation of the method we're attempting to call *)
        Tast.tinst * 
        'a calledArg list * 
        'a callerArg list * 
        Tast.typ * 
        'a assignedCalledArg list * 
        'a assignedItemSetter list * 
        (* the property related to the method we're attempting to call, if any  *)
        prop_info option * 
        'a callerNamedArg list * 
        'a callerNamedArg list  * 
        'a calledArg list 
        * 'a calledArg list

let mk_calledArgs g amap m minfo minst =
    (* Mark up the arguments with their position, so we can sort them back into order later *) 
    let methodArgTys = params_of_minfo g amap m minfo minst in  
    let methodArgAttribs = param_attrs_of_minfo g amap m minfo in 
    (combine methodArgAttribs methodArgTys) 
    |> list_mapi (fun i ((isParamArrayArg,isOutArg,optArgInfo),(nmOpt,typeOfCalledArg))  -> 
        let isOptArg = optArgInfo <> NotOptional in 
        CalledArg(i,isParamArrayArg,optArgInfo,isOutArg,nmOpt,typeOfCalledArg))  

let mk_calledMeth 
      checkingAttributeCall 
      freshen_minfo (* a function to help generte fresh type variables the property setters methods in generic classes *)
      g amap m 
      minfo             (* the method we're attempting to call *)
      minst             (* the instantiation of the method we're attempting to call *)
      uminst            (* the formal instantiation of the method we're attempting to call *)
      pinfoOpt          (* the property related to the method we're attempting to call, if any  *)
      unnamedCallerArgs (* the data about any unnamed arguments supplied by the caller *)
      namedCallerArgs   (* the data about any named arguments supplied by the caller *)
    =
    let methodRetTy = ret_typ_of_minfo g amap m minfo minst in

    if verbose then dprintf1 "--> methodRetTy = %s\n" (Layout.showL (typeL methodRetTy));
    if verbose then dprintf1 "--> minfo.Type = %s\n" (Layout.showL (typeL (typ_of_minfo minfo)));

    let fullCalledArgs = mk_calledArgs g amap m minfo minst in

    (* Find the arguments not given by name *)
    let unnamedCalledArgs = 
        fullCalledArgs |> filter (function (CalledArg(_,_,_,_,Some nm,_)) -> 
            for_all (fun (CallerNamedArg(nm2,e)) -> nm <> nm2)  namedCallerArgs | _ -> true) in

    (* See if any of them are 'out' arguments being returned as part of a return tuple *)
    let unnamedCalledArgs, unnamedCalledOptArgs, unnamedCalledOutArgs = 
        let nUnnamedCallerArgs = length unnamedCallerArgs in
        if  nUnnamedCallerArgs < length unnamedCalledArgs then
            let unnamedCalledArgsTrimmed,unnamedCalledOptOrOutArgs = chop_at nUnnamedCallerArgs unnamedCalledArgs in 
            if unnamedCalledOptOrOutArgs |> List.for_all (fun (CalledArg(i,_,_,isOutArg,_,typeOfCalledArg)) -> isOutArg && is_byref_ty g typeOfCalledArg) then 
                let unnamedCalledOutArgs = unnamedCalledOptOrOutArgs |> List.map (fun (CalledArg(i,isParamArrayArg,optArgInfo,isOutArg,nmOpt,typeOfCalledArg)) -> (CalledArg(i,isParamArrayArg,optArgInfo,isOutArg,nmOpt,typeOfCalledArg))) in
                unnamedCalledArgsTrimmed,[],unnamedCalledOutArgs
            else if unnamedCalledOptOrOutArgs |> List.for_all (fun (CalledArg(i,_,optArgInfo,isOutArg,_,typeOfCalledArg)) -> optArgInfo <> NotOptional) then 
                let unnamedCalledOptArgs = unnamedCalledOptOrOutArgs in
                unnamedCalledArgsTrimmed,unnamedCalledOptArgs,[]
            else
                unnamedCalledArgs,[],[]
        else unnamedCalledArgs,[],[] in 

    let names = namedCallerArgs |> map (function CallerNamedArg(nm,_) -> nm)  in
    if length (noRepeats string_ord names) <> length names then
        errorR(Error("a named argument has been assigned more than one value",m));
        
    let assignedNamedArgs = fullCalledArgs |> chooseList (function CalledArg(_,_,_,_,Some nm,_) as arg -> choose (fun (CallerNamedArg(nm2,arg2)) -> if nm = nm2 then Some (AssignedCalledArg(arg,arg2)) else None)  namedCallerArgs | _ -> None) in
    let unassignedNamedItem = namedCallerArgs |> filter (fun (CallerNamedArg(nm,e)) -> for_all (function CalledArg(_,_,_,_,Some nm2,_) -> nm <> nm2 | _ -> true) fullCalledArgs) in
    let attributeAssignedNamedItems,unassignedNamedItem = 
        if checkingAttributeCall then 
            (* the assignment of names to properties is substantially for attribute specifications *)
            (* permits bindings of names to non-mutable fields and properties, so we do that using the old *)
            (* reliable code for this later on. *)
            unassignedNamedItem,[]
         else 
            [],unassignedNamedItem in
    let assignedNamedProps,unassignedNamedItem = 
        let returnedObjTy = if minfo_is_ctor minfo then typ_of_minfo minfo else methodRetTy in 
        unassignedNamedItem |> splitChooseList (fun (CallerNamedArg(nm,e) as arg) -> 
            let pinfos = intrinsic_pinfos_of_typ (Some(nm),DontIncludePrivate) IgnoreOverrides g amap m returnedObjTy in 
            let pinfos = pinfos |> exclude_hidden_of_pinfos g amap m  in
            match pinfos with 
            | [pinfo] when pinfo_has_setter pinfo && not (pinfo_is_indexer g pinfo) -> 
                let pminfo = setter_minfo_of_pinfo pinfo in
                let pminst = freshen_minfo m pminfo in 
                Choice1(AssignedItemSetter(AssignedPropSetter(pminfo, pminst), e))
            | _ ->
                match il_finfos_of_typ (Some(nm),DontIncludePrivate) g amap m returnedObjTy with
                | finfo :: _ -> 
                    Choice1(AssignedItemSetter(AssignedIlFieldSetter(finfo), e))
                | _ ->              
                  match rfinfo_of_typ_by_name g amap m returnedObjTy nm with
                  | Some rfinfo -> 
                      Choice1(AssignedItemSetter(AssignedRecdFieldSetter(rfinfo), e))
                  | None -> 
                      Choice2(arg)) in

    if verbose then dprintf5 "#fullCalledArgs = %d, #unnamedCalledArgs = %d, #assignedNamedArgs = %d, #residueNamedArgs = %d, #attributeAssignedNamedItems = %d\n"
                                (length fullCalledArgs) (length unnamedCalledArgs) (length assignedNamedArgs) (length unassignedNamedItem) (length attributeAssignedNamedItems);
    CalledMeth(minfo,minst,uminst,unnamedCalledArgs,unnamedCallerArgs,methodRetTy,assignedNamedArgs,assignedNamedProps,pinfoOpt,unassignedNamedItem,attributeAssignedNamedItems,unnamedCalledOptArgs,unnamedCalledOutArgs)
    
(* This is used during constraint solving for trait constraints. *)    
let mk_formalCalledMeth g amap m minfo callerArgs = 
    mk_calledMeth false (fun _ minfo -> formal_minst_of_minfo g minfo) g amap m minfo (formal_minst_of_minfo g minfo) (formal_minst_of_minfo g minfo) None callerArgs []


let namesOfCalledArgs calledArgs = 
    chooseList (fun (CalledArg(_,_,_,_,nmOpt,_)) -> nmOpt) calledArgs

let minfo_of_cmeth                 (CalledMeth(minfo,_,_,_,_,_,_,_,_,_,_,_,_)) = minfo
let calledTyArgs_of_cmeth          (CalledMeth(_,minst,_,_,_,_,_,_,_,_,_,_,_)) = minst
let callerTypeArgs_of_cmeth        (CalledMeth(_,_,userTypeArgs,_,_,_,_,_,_,_,_,_,_)) = userTypeArgs
let unnamedCalledArgs_of_cmeth     (CalledMeth(_,_,_,unnamedCalledArgs,_,_,_,_,_,_,_,_,_)) = unnamedCalledArgs 
let numUnnamedCallerArgs_of_cmeth  (CalledMeth(_,_,_,_,unnamedCallerArgs,_,_,_,_,_,_,_,_)) = length unnamedCallerArgs 
let numAssignedArgs_of_cmeth       (CalledMeth(_,_,_,_,_,_,namedArgs,_,_,_,_,_,_)) = length namedArgs
let numAssignedProps_of_cmeth      (CalledMeth(_,_,_,_,_,_,_,namedProps,_,_,_,_,_)) = length namedProps
let unassigned_named_args_of_cmeth (CalledMeth(_,_,_,_,_,_,_,_,_,unassignedNamedItems,_,_,_)) = unassignedNamedItems
let cmeth_has_out_args             (CalledMeth(_,_,_,_,_,_,_,_,_,_,_,_,unnamedCalledOutArgs)) = nonNil unnamedCalledOutArgs
let numUnnamedCalledArgs_of_cmeth cmeth = length (unnamedCalledArgs_of_cmeth cmeth)
let numCalledTyArgs_of_cmeth      cmeth = length (calledTyArgs_of_cmeth cmeth)
let numCallerTyArgs_of_cmeth      cmeth = length (callerTypeArgs_of_cmeth cmeth)

let cmethAssignsAllNamedArgs cmeth  =
    isNil (unassigned_named_args_of_cmeth cmeth)

let cmethCorrectArity (CalledMeth(_,_,_,unnamedCalledArgs,unnamedCallerArgs,_,_,_,_,_,_,_,_) as cmeth) =
    (numCalledTyArgs_of_cmeth cmeth = numCallerTyArgs_of_cmeth cmeth)  &&
    (numUnnamedCalledArgs_of_cmeth cmeth = numUnnamedCallerArgs_of_cmeth cmeth) 

let cmethIsAccessible g amap m ad cmeth =
    minfo_accessible g amap m ad (minfo_of_cmeth cmeth) 

let cmethIsCandidate g amap m ad cmeth =
    cmethIsAccessible g amap m ad cmeth &&
    cmethCorrectArity cmeth && 
    cmethAssignsAllNamedArgs cmeth

let showAccessDomain ad =
    match ad with 
    | AccessibleFromEverywhere -> "public" 
    | AccessibleFrom(_,_) -> "accessible"
    | AccessibleFromSomewhere -> "public, protected or internal" 




(*-------------------------------------------------------------------------
!* "Type Completion" inference and a few other checks at the end of the
 * inference scope
 *------------------------------------------------------------------------- *)

exception AbstractType of range

let checkAllImplemented isImplementation g amap denv tycon =

    let m = range_of_tycon tycon in 

    let tcaug = tcaug_of_tycon tycon in 
  (* Note you only have to explicitly implement 'System.IComparable' to customize structural comparison AND equality on F# types *)
    if isImplementation &&
       isNone tcaug.tcaug_compare &&
        tcaug_has_interface g tcaug g.mk_IComparable_ty && 
        not (tcaug_has_override g tcaug "Equals" [g.obj_ty])
     then (
        (* Warn when we're doing this for class types *)
        if Augment.is_augmented_with_equals g tycon then
            warning(Error("The type '"^display_name_of_tycon tycon^"' implements 'System.IComparable'. Consider also adding an explicit override for 'Object.Equals'",range_of_tycon tycon))
        else
            warning(Error("The type '"^display_name_of_tycon tycon^"' implements 'System.IComparable' explicitly but provides no corresponding override for 'Object.Equals'. An implementation of 'Object.Equals' has been automatically provided, implemented via 'System.IComparable'. Consider implementing the override 'Object.Equals' explicitly",range_of_tycon tycon))
     ) ;

    (* Check some conditions about generic comparison and hashing. We can only check this condition after we've done the augmentation *)
    if isImplementation then
       begin 
            Augment.check_augmentation_attribs g tycon;

            let tcaug = tcaug_of_tycon tycon in 
            let m = range_of_tycon tycon in 
            let hasImplicitOrExplicitObjectGetHashCode = tcaug_has_override g tcaug "GetHashCode" [] in 
            let hasExplicitObjectEqualsOverride = tcaug_has_override g tcaug "Equals" [g.obj_ty] in 
            if isSome tcaug.tcaug_compare && hasExplicitObjectEqualsOverride then warning(Error("The struct, record or union type '"^display_name_of_tycon tycon^"' implements 'System.IComparable' implicitly because it is a struct, record or discriminated union type. Apply the '[<StructuralComparison(false)>]' attribute to the type",m)); 
            if isSome tcaug.tcaug_structural_hash && hasImplicitOrExplicitObjectGetHashCode then warning(Error("The struct, record or union type '"^display_name_of_tycon tycon^"' has an explicit implementation of 'Object.GetHashCode'. Apply the '[<StructuralEquality(false)>]' attribute to the type",m)); 

            (* remember these values to ensure we don't generate these methods during codegen *)
            set_tcaug_hasObjectGetHashCode tcaug hasImplicitOrExplicitObjectGetHashCode;
      end;

    if (repr_of_tycon tycon <> None) 
       && (not (is_fsobjmodel_tycon tycon && 
                   (is_fsobjmodel_interface_tycon tycon ||
                    is_fsobjmodel_delegate_tycon tycon))) then (

        tcaug.tcaug_closed <- true;
        let interfaces = tcaug.tcaug_implements in 
        
        let interfaces = 
          interfaces |> 
          chooseList (fun (ity,compgen,_) -> 
             (* We have a minor problem checking the IComparable interface for the default generated cases *)
             (* because the method generated is deliberately not of the correct type (because *)
             (* we use IComparable and not IComparable<T>) *)
             if compgen && type_equiv g  ity g.mk_IComparable_ty then None else 
             Some(ity,m)) in 

        let thisty = (snd(generalize_tcref (mk_local_tcref tycon))) in 
        let otys = (thisty,range_of_tycon tycon) :: interfaces  in 
        let allMembers = 
            (* tcaug_adhoc cleanup: this only has to select virtual/override/abstract members *)
            (Namemap.range_multi tcaug.tcaug_adhoc @
             Option.to_list tcaug.tcaug_compare @
             (match tcaug.tcaug_equals with None -> [] | Some(a,b) -> [a;b]) @
             Option.to_list tcaug.tcaug_structural_hash)  in 

        let implTySets = getImplSets denv g amap otys in

        let allImpls = combine otys implTySets in 

        allImpls |> iter (fun ((oty,m),SlotImplSet(virts, availPriorImplSlots,_,_)) -> 
          try 
            if verbose then dprintf2 "oty = %s, #virts = %d\n" (Layout.showL (typeL oty)) (List.length virts);
            if verbose then dprintf2 "oty = %s, #availPriorImplSlots = %d\n" (Layout.showL (typeL oty)) (List.length availPriorImplSlots);

            (* Find the overriding methods relevant to the type whose implementation relation is being checked *)
            (* We check each interface impleemntation, and then check all the abstracts related to the class *)
            (* hierarchy. *)
            (* REVIEW: why not do all types at once?*)
            let overrides2 = 
                allMembers |> filter (fun ovref -> 
                    let vspr = the (member_info_of_vref ovref) in 
                    vspr.vspr_flags.memFlagsOverride &&
                    (match vspr.vspr_implements_slotsig with 
                     | None -> 
                        type_equiv g thisty oty
                     | Some (TSlotSig(_,ty,_,_,_,_)) -> 
                        let res = (is_interface_typ ty && type_equiv g ty oty) || (not (is_interface_typ ty) && not (is_interface_typ oty)) in
                        (* if logical_name_of_vspr_vref ovref = "GetHashCode" then dprintf3 "oty = %s, vref = %s, res = %b\n" (NicePrint.pretty_string_of_typ (denv_of_tenv env) oty) (bufs (fun os -> NicePrint.output_qualified_val_spec (denv_of_tenv env) os (deref_val ovref))) res; *)
                        res))  in
                        
            
            if verbose then dprintf2 "oty = %s, #overrides2 = %d\n" (Layout.showL (typeL oty)) (List.length overrides2); 

            (* Now extract the relevant information about each overriding method *)
            let overrides2 = overrides2 |> map (fun ovref -> ovref,dest_vspr_vref_override g ovref) in
            let overrides = map snd overrides2 in 
            
            (* Now check the implementation *)
            (* We don't give missing method errors for classes since we're simply determining if the thing is abstract *)
            let showMissingMethods = isImplementation && ((is_interface_typ oty) or not tcaug.tcaug_abstract) in 
            if isImplementation && not tcaug.tcaug_abstract then begin
                let allImplemented = checkAbstractMembersAreImplemented showMissingMethods denv g amap (m,oty,virts,availPriorImplSlots,overrides) in 
                if not allImplemented && not (is_interface_typ oty) then begin
                        errorR(AbstractType(m));
                end
            end;
                
                
            checkOverridesAreAllUsedOnce denv g amap (m,oty,virts,overrides);

            (* Now record the slotsigs for each implementation *)
            (* REVIEW: get rid of this mutation. *)
            overrides2 |> iter (fun (ovbyref,ovinfo) -> 
                if verbose then dprintf1 "adjust slot for %s\n" (Layout.showL (vrefL ovbyref));
                let overridden = 
                  try Some(List.find (fun virt -> is_exact_match g amap m virt ovinfo) virts) 
                  with Not_found -> None in 

                match overridden with 
                | Some ov -> 

                    (* Get the slotsig of the overriden method *)
                    let slotsig = (slotsig_of_minfo g amap m ov) in

                    (* Reverse-map the slotsig so it is in terms of the type parameters for the overriding method *)
                    let slotsig = 
                        match partition_vref_typars ovbyref with
                        | Some(_,ctps,_,_,_) -> 
                            let parentToMemberInst,_ = mk_typar_to_typar_renaming (typars_of_tcref (apparent_parent_of_vspr_vref ovbyref)) ctps in 
                            let res = inst_slotsig parentToMemberInst slotsig in 
                            if verbose then dprintf4 "adjust slot %s, #parentToMemberInst = %d, before = %s, after = %s\n" (Layout.showL (vrefL ovbyref)) (length parentToMemberInst) (Layout.showL(tslotsigL slotsig)) (Layout.showL(tslotsigL res));
                            res
                        | None -> 
                            (* Note: it appears partition_vref_typars should never return 'None' *)
                            slotsig in 
 
                   (* Record the slotsig via mutation *)
                   let vspr = the (member_info_of_vref ovbyref) in 
                    vspr.vspr_implements_slotsig <- Some slotsig;

                | None -> 
                    if verbose then dprintf0 "adjust slot, no override found\n" ;
                    ());
          with e -> errorRecoveryPoint e; ()
        ))

(*-------------------------------------------------------------------------
!* "Single Feasible Type" inference
(* Look for the unique supertype of ty2 for which ty2 :> ty1 might feasibly hold *)
(* REVIEW: eliminate this use of type_feasibly_subsumes_type *)
(* We should be able to look for identical head types. *)
 *------------------------------------------------------------------------- *)
      
let find_unique_feasible_supertype g amap m ty1 ty2 =  
    if not (is_stripped_tyapp_typ ty2) then None else
    let tcref,tinst = dest_stripped_tyapp_typ ty2 in 
    let supertypes = 
      Option.to_list (super_of_typ g amap m ty2) @
      (implements_of_typ g amap m ty2) in
    tryfind (type_feasibly_subsumes_type 0 g amap m ty1 NoCoerce) supertypes
    

(*-------------------------------------------------------------------------
!* "Dispatch Slot" inference
 *------------------------------------------------------------------------- *)

(* Determine if a uniquely-identified-override exists based on the information *)
(* at the member signature. This is used to fetch type information if it does *)
(* REVIEW: share this code with the Expr_impl case *)
let abstract_minfos_for_syn_method (memberName:ident) g amap bindm typToSearchForAbstractMembers topValSynData =
    let minfos = 
        match typToSearchForAbstractMembers with 
        | _,Some(SlotImplSet(reqdMeths, _,_,_)) -> reqdMeths |> List.filter (fun abs_minfo -> name_of_minfo abs_minfo = memberName.idText) 
        | ty, None -> intrinsic_minfos_of_typ (Some(memberName.idText),DontIncludePrivate) IgnoreOverrides g amap bindm ty in 
    let virts = filter minfo_is_virt minfos in 
    let topValSynArities = SynArgInfo.aritiesOfArgs topValSynData in 
    let topValSynArity = if List.length topValSynArities >= 1 then List.nth topValSynArities 1 else -1 in 
    let virtsArityMatch = filter (fun minfo -> arity_of_minfo g minfo = topValSynArity) virts in 
    virts,virtsArityMatch 
     
let abstract_pinfos_for_syn_property (memberName:ident) g amap bindm typToSearchForAbstractMembers k topValSynData = 
    let pinfos = 
        match typToSearchForAbstractMembers with 
        | _,Some(SlotImplSet(_,_,reqdProps,_)) -> reqdProps |> List.filter (fun abs_pinfo -> name_of_pinfo abs_pinfo = memberName.idText) 
        | ty, None -> intrinsic_pinfos_of_typ (Some(memberName.idText),DontIncludePrivate) IgnoreOverrides g amap bindm ty in 

        
    let virts = pinfos |> filter pinfo_is_virt in 
(*
    let topValSynArities = SynArgInfo.aritiesOfArgs topValSynData in 
    let topValSynArity = if List.length topValSynArities >= 1 then List.nth topValSynArities 1 else 0 in 
    let virtsArityMatch = virts |> filter (fun pinfo -> length (argtys_of_pinfo g amap bindm pinfo) = topValSynArity) in 
*)
    virts



(*-------------------------------------------------------------------------
!* redundancy of 'isinst' patterns
 *------------------------------------------------------------------------- *)

(* True if success with the second discriminator implies success with the first *)
let discrim_subsumed_by g amap m d1 d2 =
  discrim_eq g d1 d2 or
  match d1,d2 with 
  | TTest_isinst (srcty1,tgty1), TTest_isinst (srcty2,tgty2) -> type_definitely_subsumes_type_no_coercion 0 g amap m tgty2 tgty1
  | _ -> false
    