// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.NativeInterop

#nowarn "44";;
open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Core.Operators

open System
open System.Runtime.InteropServices

module NativeOps = 
    let inline add (x : 'a nativeptr) (n:int) : 'a nativeptr = NativeOps.to_nativeint x + nativeint n * (# "sizeof !0" type('a) : nativeint #) |> NativeOps.of_nativeint
    let inline get (arr : 'a nativeptr) n = (# "ldobj !0" type ('a) (add arr n) : 'a #) 
    let inline set (arr : 'a nativeptr) n (x : 'a) = (# "stobj !0" type ('a) (add arr n) x #)  

type NativeArray<'a> = 
    { len: int; ptr : 'a nativeptr }
    static member FromPtr(p,len) : NativeArray<'a> = { len=len;ptr = p }
    member x.Ptr = x.ptr
    member inline x.Item 
       with get (n) = NativeOps.get x.Ptr n
       and  set (n) v = NativeOps.set x.Ptr n v
    member x.Length = x.len

type FortranMatrix<'a> = 
    { nrows: int; ncols:int; ptr : 'a nativeptr }
    static member FromPtr(p,nrows,(ncols:int)) : FortranMatrix<'a> = { ptr = p; nrows=nrows; ncols=ncols}
    member x.NumCols = x.ncols
    member x.NumRows = x.nrows
    member x.Ptr = x.ptr
    member inline x.Item 
       with get (row,col) = NativeOps.get x.Ptr (row + col*x.NumRows)
       and  set (row,col) v = NativeOps.set x.Ptr (row + col*x.NumRows) v
    member x.NativeTranspose = CMatrix.FromPtr(x.ptr,x.ncols,x.nrows)
  
and NativeArray2<'a> = 
    { nrows:int; ncols: int; ptr : 'a nativeptr }
    static member FromPtr(p,(nrows:int),ncols) : NativeArray2<'a> = { nrows=nrows;ncols=ncols;ptr=p}
    member x.Ptr = x.ptr
    member x.NumRows = x.nrows
    member x.NumCols = x.ncols
    member inline x.Item 
       with get (row,col) = NativeOps.get x.Ptr (row*x.NumCols + col)
       and  set (row,col) v = NativeOps.set x.Ptr (row*x.NumCols + col) v
    member x.NativeTranspose = FortranMatrix.FromPtr(x.ptr,x.ncols,x.nrows)
  
and CMatrix<'a> = NativeArray2<'a> 

#if CLI_AT_MOST_1_1
#else
module Ref = 
    let inline pin (m: 'a ref) (f : 'a nativeptr -> 'b) = NativeOps.pinAny (box m) (fun gch -> f (gch.AddrOfPinnedObject() |> NativeOps.of_nativeint))

open Microsoft.FSharp.Math

type PinnedArray<'a> = 
    { mutable gch: GCHandle; narray: NativeArray<'a> }
    static member inline of_array(m: 'a[]) : PinnedArray<'a> 
                       = let ptr,gch = Array.pin_unscoped(m) in 
                         PinnedArray.FromNative(NativeArray.FromPtr(ptr,Array.length m),gch)
    static member inline of_vector(m:Vector<'a>) : PinnedArray<'a> = let ptr,gch = m.PinHandle() in PinnedArray.FromNative(NativeArray.FromPtr(ptr,m.Length),gch)
    static member inline of_rowvec(m:RowVector<'a>) : PinnedArray<'a> = let ptr,gch = m.PinHandle() in PinnedArray.FromNative(NativeArray.FromPtr(ptr,m.Length),gch)
    static member FromNative(narray,gch) : PinnedArray<'a> = { gch=gch; narray=narray}
    member x.Ptr = x.narray.Ptr
    member x.Free() = x.gch.Free()
    member x.Length = x.narray.Length
    member x.NativeArray = x.narray

type PinnedArray2<'a> = 
    { mutable gch: GCHandle; narray: NativeArray2<'a> }
    static member inline of_array2(m: 'a[,]) : PinnedArray2<'a> = let ptr,gch = Array2.pin_unscoped(m) in PinnedArray2.FromNative(NativeArray2.FromPtr(ptr,Array2.length1 m,Array2.length2 m),gch)
    static member inline of_matrix(m:Matrix<'a>) : PinnedArray2<'a> = let ptr,gch = m.PinHandle() in PinnedArray2.FromNative(NativeArray2.FromPtr(ptr,m.NumRows,m.NumCols),gch) 
    static member FromNative(narray,gch) : PinnedArray2<'a> = { gch=gch; narray=narray}
    member x.Ptr = x.narray.Ptr
    member x.Free() = x.gch.Free()
    member x.NumRows = x.narray.NumRows
    member x.NumCols = x.narray.NumCols
    member x.NativeArray = x.narray
#endif

module NativePtr = 

    let inline of_nativeint n = NativeOps.of_nativeint n
    let inline to_nativeint p = NativeOps.to_nativeint p
    let inline of_ilsigptr p = NativeOps.of_ilsigptr p
    let inline to_ilsigptr p = NativeOps.to_ilsigptr p
    let inline to_NativeArray (p : 'a nativeptr) (len:int)  = NativeArray.FromPtr(p,len)
    let inline of_NativeArray (n: NativeArray<'a>) = n.Ptr
    let inline add np n = NativeOps.add np n
    let inline get np n = NativeOps.get np n
    let inline set np n v = NativeOps.set np n v
