// (c) Microsoft Corporation 2005-2007. 

#light 

namespace Microsoft.FSharp.Collections

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Core.LanguagePrimitives
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Primitives.Basics
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module List = 

    let nonempty x = Microsoft.FSharp.Primitives.Basics.List.nonempty x
    let length l = Microsoft.FSharp.Primitives.Basics.List.length l
    let rev l = Microsoft.FSharp.Primitives.Basics.List.rev l
    let concat l = Microsoft.FSharp.Primitives.Basics.List.concat l
    let map f l = Microsoft.FSharp.Primitives.Basics.List.map f l
    let mapi f l = Microsoft.FSharp.Primitives.Basics.List.mapi f l
    let iter f l = Microsoft.FSharp.Primitives.Basics.List.iter f l


    let hd l = match l with (x:: _) -> x | [] -> invalid_arg "hd"
    let tl l = match l with (_ :: t) -> t | [] -> invalid_arg "tl"

    let rec nth l n = 
        if l = [] || n < 0 then invalid_arg "nth" 
        elif n = 0 then hd l else nth (tl l) (n - 1)

    let append l1 l2 = l1 @ l2
    let rev_append l1 l2 = Microsoft.FSharp.Primitives.Basics.List.rev_append l1 l2

    let rec choose_acc f xs acc =
        match xs with 
        | [] -> rev acc
        | h :: t -> 
             match f h with 
             | None -> choose_acc f t acc 
             | Some x -> choose_acc f t (x::acc)

    let choose f xs = choose_acc f xs []

    let rec first f xs =
        match xs with 
        | [] -> None
        | h :: t -> 
             match f h with 
             | None -> first f t
             | Some x -> Some x


    let flatten l = concat l

    let iteri f l = Microsoft.FSharp.Primitives.Basics.List.iteri f l

    let init n f = Microsoft.FSharp.Primitives.Basics.List.init n f

    let rec rev_map_acc f l acc =
        match l with 
        | [] -> acc
        | h::t -> rev_map_acc f t (f h :: acc)

    let rev_map f l = rev_map_acc f l []

    let iter2 f l1 l2 = 
        let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
        let rec loop l1 l2 = 
            match l1,l2 with
            | [],[] -> () 
            | (h1::t1), (h2::t2) -> f.Invoke(h1,h2); loop t1 t2 
            | _ -> invalid_arg "List.iter2"
        loop l1 l2

    let iteri2 f l1 l2 = 
        let f = OptimizedClosures.FastFunc3<_,_,_,_>.Adapt(f)
        let rec loop n l1 l2 = 
            match l1,l2 with
            | [],[] -> () 
            | (h1::t1), (h2::t2) -> f.Invoke(n,h1,h2); loop (n+1) t1 t2 
            | _ -> invalid_arg "List.iter2i"
        loop 0 l1 l2
      
    let rec map3a (f:OptimizedClosures.FastFunc3<_,_,_,_>) l1 l2 l3 acc = 
        match l1,l2,l3 with
        | [],[],[] -> rev acc
        | (h1::t1), (h2::t2),(h3::t3) -> let x = f.Invoke(h1,h2,h3) in map3a f t1 t2 t3 (x :: acc)
        | _ -> invalid_arg "List.map3"

    let map3 f l1 l2 l3 = 
        let f = OptimizedClosures.FastFunc3<_,_,_,_>.Adapt(f)
        map3a f l1 l2 l3 []

    let rec mapi2a n (f:OptimizedClosures.FastFunc3<_,_,_,_>) l1 l2 acc = 
        match l1,l2 with
        | [],[] -> rev acc
        | (h1::t1), (h2::t2) -> let x = f.Invoke(n,h1,h2) in mapi2a (n+1) f t1 t2 (x :: acc)
        | _ -> invalid_arg "List.mapi2"

    let mapi2 f l1 l2 = 
        let f = OptimizedClosures.FastFunc3<_,_,_,_>.Adapt(f)
        mapi2a 0 f l1 l2 []

    let rec rev_map2_acc (f:OptimizedClosures.FastFunc2<_,_,_>) l1 l2 acc =
        match l1,l2 with 
        | [],[] -> acc
        | h1::t1, h2::t2 -> rev_map2_acc f t1 t2 (f.Invoke(h1,h2) :: acc)
        | _ -> invalid_arg "List.rev_map2_acc"

    let rev_map2 f l1 l2 = 
        let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
        rev_map2_acc f l1 l2 []

    let map2 f l1 l2 = rev (rev_map2 f l1 l2)


    let fold_left f s l = 
        let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
        let rec loop s l = 
            match l with 
            | [] -> s
            | (h::t) -> loop (f.Invoke(s,h)) t
        loop s l

    let reduce_left f l = 
        match l with 
        | [] -> invalid_arg "List.reduce_left"
        | (h::t) -> fold_left f h t

    let fold1_left f l =  reduce_left f l

    let scan_left f s l = 
        let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
        let rec loop s l acc = 
            match l with 
            | [] -> rev acc
            | (h::t) -> let s = f.Invoke(s,h) in loop s t (s :: acc)
        loop s l [s]

    let scan1_left f l = 
        match l with 
        | [] -> invalid_arg "List.scan1_left"
        | (h::t) -> scan_left f h t

    let fold_left2 f acc l1 l2 = 
        let f = OptimizedClosures.FastFunc3<_,_,_,_>.Adapt(f)
        let rec loop acc l1 l2 =
            match l1,l2 with 
            | [],[] -> acc
            | (h1::t1),(h2::t2) -> loop (f.Invoke(acc,h1,h2)) t1 t2
            | _ -> invalid_arg "List.fold_left2"
        loop acc l1 l2

    (*
    let rec fold_right f l acc = 
      match l with 
        [] -> acc
      | (h::t) -> f h (fold_right f t acc)

    *)

    let fold_array_sub_right<'a,'b> (f:OptimizedClosures.FastFunc2<'a,'b,'b>) arr start fin acc = 
        let mutable state = acc in 
        for i = fin downto start do
            state <- f.Invoke(arr.[i], state)
        state

    (* this version doesn't causes stack overflow - it uses a private stack *)
    let fold_right f l acc = 
        let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
        match l with 
        | [] -> acc
        | [h] -> f.Invoke(h,acc)
        | [h1;h2] -> f.Invoke(h1,f.Invoke(h2,acc))
        | [h1;h2;h3] -> f.Invoke(h1,f.Invoke(h2,f.Invoke(h3,acc)))
        | [h1;h2;h3;h4] -> f.Invoke(h1,f.Invoke(h2,f.Invoke(h3,f.Invoke(h4,acc))))
        | _ -> 
            // It is faster to allocate and iterate an array than to create all those 
            // highly nested stacks.  It also means we won't get stack overflows here. 
            let arr = Array.of_list l in 
            let arrn = Array.length arr in
            fold_array_sub_right f arr 0 (arrn - 1) acc

    let reduce_right f l = 
        match l with 
        | [] -> invalid_arg "List.fold1_right"
        | _ -> 
            let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
            let arr = Array.of_list l in 
            let arrn = Array.length arr in
            fold_array_sub_right f arr 0 (arrn - 2) arr.[arrn - 1]

    let fold1_right f l = reduce_right f l

    let scan_array_sub_right<'a,'b> (f:OptimizedClosures.FastFunc2<'a,'b,'b>) arr start fin initState = 
        let mutable state = initState in 
        let mutable res = [state] in 
        for i = fin downto start do
          state <- f.Invoke(arr.[i], state);
          res <- state :: res
        done;
        res

    let scan_right f l s = 
        match l with 
        | [] -> [s]
        | [h] -> 
            [f h s; s]
        | _ -> 
            let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
            // It is faster to allocate and iterate an array than to create all those 
            // highly nested stacks.  It also means we won't get stack overflows here. 
            let arr = Array.of_list l in 
            let arrn = Array.length arr in
            scan_array_sub_right f arr 0 (arrn - 1) s

    let scan1_right f l = 
        match l with 
        | [] -> invalid_arg "scan1_right"
        | _ -> 
            let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
            let arr = Array.of_list l in 
            let arrn = Array.length arr in
            scan_array_sub_right f arr 0 (arrn - 2) arr.[arrn - 1]

    (*
    let rec fold_right2 f l1 l2 acc = 
      match l1,l2 with 
        [],[] -> acc
      | (h1::t1),(h2::t2) -> f h1 h2 (fold_right2 f t1 t2 acc)
      | _ -> invalid_arg "fold_right2"
    *)


    // TODO: optimize this w.r.t. OptimizedClosures
    let rec fold_right2 f l1 l2 acc = 
        match l1,l2 with 
        | [],[] -> acc
        | [h],[k] -> f h k acc
        | [h1;h2],[k1;k2] -> f h1 k1 (f h2 k2 acc)
        | [h1;h2;h3],[k1;k2;k3] -> f h1 k1 (f h2 k2 (f h3 k3 acc))
        | [h1;h2;h3;h4],[k1;k2;k3;k4] -> f h1 k1 (f h2 k2 (f h3 k3 (f h4 k4 acc)))
        | _ -> 
            let arr1 = Array.of_list l1 in 
            let arr2 = Array.of_list l2 in 
            let n1 = Array.length arr1 in 
            let n2 = Array.length arr2 in 
            if n1 <> n2 then invalid_arg "fold_right2";
            let mutable res = acc in 
            for i = n1 - 1 downto 0 do
              res <- f arr1.(i) arr2.(i) res
            done;
            res


    // TODO: optimize this w.r.t. OptimizedClosures
    let rec for_all2 f l1 l2 = 
        match l1,l2 with 
        | [],[] -> true
        | (h1::t1),(h2::t2) -> f h1 h2  && for_all2 f t1 t2
        | _ -> invalid_arg "for_all2"

    let rec for_all f l1 = Microsoft.FSharp.Primitives.Basics.List.for_all f l1

    let rec exists f l1 = Microsoft.FSharp.Primitives.Basics.List.exists f l1

    let rec exists2 f l1 l2 = 
        match l1,l2 with 
        | [],[] -> false
        | (h1::t1),(h2::t2) -> f h1 h2  || exists2 f t1 t2
        | _ -> invalid_arg "exists2"

    let rec mem x l = match l with [] -> false | h::t -> x = h || mem x t
    let rec memq x l = match l with [] -> false | h::t -> PhysicalEquality x h || memq x t
    let rec find f l = match l with [] -> not_found() | h::t -> if f h then h else find f t
    let rec tryfind f l = match l with [] -> None | h::t -> if f h then Some h else tryfind f t

    let filter f x = Microsoft.FSharp.Primitives.Basics.List.filter f x
    let find_all f x = Microsoft.FSharp.Primitives.Basics.List.filter f x
    let partition p x = Microsoft.FSharp.Primitives.Basics.List.partition p x
        
    let rec assoc x l = 
        match l with 
        | [] -> not_found()
        | ((h,r)::t) -> if x = h then r else assoc x t

    let rec try_assoc x l = 
        match l with 
        | [] -> None
        | ((h,r)::t) -> if x = h then Some(r) else try_assoc x t

    let rec mem_assoc x l = 
        match l with 
        | [] -> false
        | ((h,r)::t) -> x = h || mem_assoc x t

    let rec remove_assoc x l = 
        match l with 
        | [] -> []
        | (((h,r) as p) ::t) -> if x = h then t else p:: remove_assoc x t

    let unzip x = Microsoft.FSharp.Primitives.Basics.List.unzip x
    let unzip3 x = Microsoft.FSharp.Primitives.Basics.List.unzip3 x
    let zip x1 x2 =  Microsoft.FSharp.Primitives.Basics.List.zip x1 x2
    let zip3 x1 x2 x3 =  Microsoft.FSharp.Primitives.Basics.List.zip3 x1 x2 x3
    let split x =  unzip x
    let combine x1 x2 =  zip x1 x2

    let half x = x >>> 1 

    let rec merge cmp a b acc = 
      match a,b with 
      | [], a | a,[] -> rev_append acc a
      | x::a', y::b' -> if cmp x y > 0 then merge cmp a  b' (y::acc) else merge cmp a' b  (x::acc)

    let sort2 cmp x y = 
      if cmp x y > 0 then [y;x] else [x;y]

    let sort3 cmp x y z = 
      let cxy = cmp x y in 
      let cyz = cmp y z in 
      if cxy > 0 && cyz < 0 then 
        if cmp x z > 0 then [y;z;x] else [y;x;z]
      elif cxy < 0 && cyz > 0 then 
        if cmp x z > 0 then [z;x;y] else [x;z;y]
      elif cxy > 0 then 
        if cyz > 0 then  [z;y;x]
        else [y;z;x]
      else 
        if cyz > 0 then [z;x;y]
        else [x;y;z] 

    let trivial a = match a with [] | [_] -> true | _ -> false
        
    (* tail recursive using a ref *)

    let rec stable_sort_inner cmp la ar =
      if la < 4 then (* sort two || three new entries *)
        match !ar with 
         | x::y::b -> 
              if la = 2 then ( ar := b; sort2 cmp x y )
              else begin
                match b with 
                | z::c -> ( ar := c; sort3 cmp x y z )
                | _ -> failwith "never" 
              end
         | _ -> failwith "never"
      else (* divide *)
        let lb = half la in
        let sb = stable_sort_inner cmp lb ar in
        let sc = stable_sort_inner cmp (la - lb) ar in 
        merge cmp sb sc []

    let stable_sort cmp a =
        if trivial a then a else
        let ar = ref a in
        stable_sort_inner cmp (length a) ar

    let sort f l = stable_sort f l

    let of_array (arr:'a array) = Microsoft.FSharp.Primitives.Basics.Array.to_list arr

    let rec copy_to_array l arr i =
        match l with 
        | [] -> ()
        | h :: t -> Microsoft.FSharp.Primitives.Basics.Array.set arr i h; copy_to_array t arr (i+1)

    let to_array (l:'a list) = Microsoft.FSharp.Primitives.Basics.Array.of_list l

    let rec assq x l = 
        match l with 
        | [] -> not_found()
        | ((h,r)::t) -> if PhysicalEquality x h then r else assq x t

    let rec try_assq x l = 
        match l with 
        | [] -> None
        | ((h,r)::t) -> if PhysicalEquality x h then Some(r) else try_assq x t

    let rec mem_assq x l = 
        match l with 
        | [] -> false
        | ((h,r)::t) -> PhysicalEquality x h || mem_assq x t

    let rec remove_assq x l = 
        match l with 
        | [] -> []
        | (((h,r) as p) ::t) -> if PhysicalEquality x h then t else p:: remove_assq x t

    let get_IEnumerator s = (Seq.of_list s).GetEnumerator()
    let to_seq l = Seq.of_list l
    let of_seq ie = Seq.to_list ie
    let to_IEnumerable l = to_seq l
    let of_IEnumerable ie = of_seq ie

#if CLI_AT_MOST_1_1
#else
    let to_ICollection s = 
        { new ICollection<'a> 
            with Add(x) = raise (new System.NotSupportedException("ReadOnlyCollection"));
            and Clear() = raise (new System.NotSupportedException("ReadOnlyCollection"));
            and Remove(x) = raise (new System.NotSupportedException("ReadOnlyCollection"));
            and Contains(x) = mem x s
            and CopyTo(arr,i) = copy_to_array s arr i
            and get_IsReadOnly() = true
            and get_Count() = length s 
          interface IEnumerable<'a>
            with GetEnumerator() = get_IEnumerator s
          interface System.Collections.IEnumerable
            with GetEnumerator() = ((Seq.of_list s).GetEnumerator() :> System.Collections.IEnumerator) }

    let of_ICollection (c :> ICollection<'a>) : list<'a> = of_seq (c :> seq<'a>)
    let to_List l = new List<_>(to_seq l)
    let of_List (l : List<'a>) = of_seq(l :> seq<'a>)
      
    let to_ResizeArray l = new ResizeArray<_>(to_seq l)
    let of_ResizeArray (l : ResizeArray<'a>) = of_seq(l :> IEnumerable<'a>)
#endif      

    let find_index f l = 
        let rec loop n = function[] -> not_found() | h::t -> if f h then n else loop (n+1) t
        loop 0 l

    let find_indexi f l = 
        let rec loop n = function[] -> not_found() | h::t -> if f n h then n else loop (n+1) t
        loop 0 l

    let tryfind_index f l = 
        let rec loop n = function[] -> None | h::t -> if f h then Some n else loop (n+1) t
        loop 0 l

    let tryfind_indexi f l = 
        let rec loop n = function [] -> None | h::t -> if f n h then Some n else loop (n+1) t
        loop 0 l

    let sumByInt     f l = let rec loop acc = function [] -> acc | h::t -> loop (acc+f h) t in loop 0    l
    let sumByFloat   f l = let rec loop acc = function [] -> acc | h::t -> loop (acc+f h) t in loop 0.0  l
    let sumByFloat32 f l = let rec loop acc = function [] -> acc | h::t -> loop (acc+f h) t in loop 0.0f l
    let sumByInt64   f l = let rec loop acc = function [] -> acc | h::t -> loop (acc+f h) t in loop 0L   l

    // TODO: this implementation could use tail-cons-mutation
    let rec rev_map_concat_onto f l acc = 
        match l with 
        | [] -> acc
        | h::t -> rev_map_concat_onto f t (rev_append (f h) acc)
      
    let map_concat f l = rev (rev_map_concat_onto f l [])
