(* (c) Microsoft Corporation 2005-2007.  *)

(*-------------------------------------------------------------------------
 * Generate the hash/compare functions we add to user-defined types by default.
 *------------------------------------------------------------------------- *)

(*F# 
module Microsoft.FSharp.Compiler.Augment 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Il = Microsoft.Research.AbstractIL.IL 
F#*) 

open Il
open Tast
open Tastops
open Ast
open List
open Lib
open Env

let mk_IComparable_CompareTo_slotsig g = 
  TSlotSig("CompareTo",g.mk_IComparable_ty, [],[], [TSlotParam(Some("obj"),g.obj_ty,false,false,false,[])],g.int_ty)

let mk_GetStructuralHashCode_slotsig g = 
  TSlotSig("GetStructuralHashCode", g.mk_IStructuralHash_ty, [],[], [TSlotParam(Some("nodesRemaining"),mk_byref_typ g g.int_ty,false,false,false,[])],g.int_ty)

let mk_GetHashCode_slotsig g = 
  TSlotSig("GetHashCode", g.obj_ty, [],[], [],g.int_ty)
let mk_Equals_slotsig g = 
  TSlotSig("Equals", g.obj_ty, [],[], [TSlotParam(Some("obj"),g.obj_ty,false,false,false,[])],g.bool_ty)


let mspec_Object_GetType ilg = Il.mk_nongeneric_instance_mspec_in_nongeneric_boxed_tref(ilg.tref_Object,"GetType",[],ilg.typ_Type)
let mspec_Object_ToString ilg = Il.mk_nongeneric_instance_mspec_in_nongeneric_boxed_tref(ilg.tref_Object,"ToString",[],ilg.typ_String)
let mk_call_Object_GetType_GetString g m e1 = 
  mk_asm ([ Il.mk_normal_callvirt(mspec_Object_ToString g.ilg)   ], [], 
             [ mk_asm ([ Il.mk_normal_callvirt(mspec_Object_GetType g.ilg) ], [], [ e1; ], [ g.int_ty ], m) ], [ g.string_ty ], m)


(*-------------------------------------------------------------------------
 * Polymorphic comparison
 *------------------------------------------------------------------------- *)

let mk_rel_binop g op m e1 e2 = mk_asm ([ Il.I_arith op  ],[],  [e1; e2],[g.bool_ty],m)
let mk_clt g m e1 e2 = mk_rel_binop g Il.AI_clt m e1 e2 
let mk_cgt g m e1 e2 = mk_rel_binop g Il.AI_cgt m e1 e2

(*-------------------------------------------------------------------------
 * Structural hashing
 *
 * For each record, union and exception declaration we bake a hash
 * method "int GetStructuralHashCode(int &count)" which helps the type implements the 
 * IStructuralHash interface.  Callers provide a default hash depth.
 * The implementations of these functions cooperate to hash the structured
 * term to a maximum count of nodes.
 *
 * REVIEW: make this a .constrained call, not a virtual call.
 *------------------------------------------------------------------------- *)

let mk_il_ldind g m v1    = mk_lval_get m (mk_local_vref v1)        
let mk_il_stind g m v1 e2 = mk_lval_set m (mk_local_vref v1) e2     

(* do something if there are nodes left to hash *)
let if_nodes_then_else g m nv e e2 = mk_cond m g.unit_ty (mk_il_ldind g m nv) e e2
let mk_ind_decr g m nv = mk_il_stind g m nv (mk_decr g m (mk_il_ldind g m nv))
let decr_nodes_and_then g m nv e = mk_seq m (mk_ind_decr g m nv) e

let hash_vars g m ty = 
  let thisv,thise = mk_local m "this" ty in  
  let nv,ne = mk_local m "nodesRemaining" (mk_byref_typ g g.int_ty) in  
  (thisv,thise,nv,ne)
    
(* REVIEW: this is actually a lousy hash accumulation technique *)
let add_to_hash_acc g m e accv acce =
  mk_val_set m accv 
    (mk_asm([ Il.mk_ldc_i32 (Nums.int_to_i32 1); 
                  Il.I_arith Il.AI_shl ; 
                  Il.I_arith Il.AI_add  ],[],[e;acce],[g.int_ty],m))

let mk_combine_hash_generators g m nv exprs accv acce =
  fold_left
    (fun tm e -> if_nodes_then_else g m nv (mk_seq m (add_to_hash_acc g m e accv acce) tm) acce)
    acce
    exprs 

(*-------------------------------------------------------------------------
 * Build comparison functions for union, record and exception types.
 *------------------------------------------------------------------------- *)
let mk_thisv m ty = mk_local m "this" ty 

let mk_thisv_thatv m ty =
  let thisv,thise = mk_thisv m ty in
  let thatv,thate = mk_local m "obj" ty in 
  thisv,thatv,thise,thate

let mk_compare_test_conjs g m exprs =
  match exprs with 
  | [] -> mk_zero g m
  | [h] -> h
  | l -> 
      let a,b = frontAndBack l in 
      fold_right
        (fun e acc -> 
          let nv,ne = mk_local m "n" g.int_ty in
          mk_let m nv e
            (mk_cond m g.int_ty
               (mk_clt g m ne (mk_zero g m))
               ne
               (mk_cond m g.int_ty 
                  (mk_cgt g m ne (mk_zero g m))
                  ne
                  acc)))
        a 
        b

let mk_equals_test_conjs g m exprs =
  match exprs with 
  | [] -> mk_one g m
  | [h] -> h
  | l -> 
      let a,b = frontAndBack l in 
      fold_right (fun e acc -> mk_cond m g.bool_ty e acc (mk_false g m)) a b

(* note: 'x == y' does not imply 'x = y' for NaN *)
let mk_physical_equality_equals_test g m tycon thise thate expr = expr
(* 
    if is_struct_tycon tycon then expr else 
    mk_cond m g.bool_ty  
            (mk_ceq g m thise thate)
            (mk_true g m)
            expr 
*)

(* note: In OCaml 'x == y' does imply 'compare x y = 0', even for NaN *)
(* note: However do not rely on this spec for F# since we are honest and call IComparable instead *)
let mk_physical_equality_compare_test g m tycon thise thate expr = expr

(*
    if is_struct_tycon tycon then expr else 
    mk_cond m g.int_ty  
            (mk_ceq g m thise thate)
            (mk_zero g m)
            expr 
*)

let minimal_type g tcref = 
    if tycon_is_exnc (deref_tycon tcref) then [], g.exn_ty 
    else generalize_tcref tcref


(* Build the comparison expression for a record type *)
let mk_recd_compare g tcref tycon = 
  let m = range_of_tycon tycon in 
  let fields = instance_rfields_of_tycon tycon in 
  let tinst,ty = minimal_type g tcref in
  let thisv,thatv,thise,thate = mk_thisv_thatv m (if is_struct_tycon tycon then mk_byref_typ g ty else ty) in  
  let mk_test fspec = 
    let fty = fspec.rfield_type in 
    let fref = rfref_of_rfield tcref fspec in 
    let m = range_of_rfref fref in 
    mk_call_poly_compare_outer g m fty
      (mk_recd_field_get_via_expra(thise, fref, tinst, m))
      (mk_recd_field_get_via_expra(thate, fref, tinst, m))  in
  let expr = mk_compare_test_conjs g m (map mk_test fields) in 
  let expr = mk_physical_equality_compare_test g m tycon thise thate expr in
  let thatv,expr = 
    if is_struct_tycon tycon then 
      let thatv2,thatvref2,thate2 = mk_mut_local m "obj" ty in 
      thatv2,mk_let m thatv (mk_val_addr m thatvref2) expr
    else thatv,expr in 
  thisv,thatv, expr


(* Build the equals expression for a record type *)
let mk_recd_equals g tcref tycon = 
  let m = range_of_tycon tycon in 
  let fields = instance_rfields_of_tycon tycon in 
  let tinst,ty = minimal_type g tcref in
  let thisv,thatv,thise,thate = mk_thisv_thatv m (if is_struct_tycon tycon then mk_byref_typ g ty else ty) in  
  let mk_test fspec = 
    let fty = fspec.rfield_type in 
    let fref = rfref_of_rfield tcref fspec in 
    let m = range_of_rfref fref in 
    mk_call_poly_equals_outer g m fty
      (mk_recd_field_get_via_expra(thise, fref, tinst, m))
      (mk_recd_field_get_via_expra(thate, fref, tinst, m))  in
  let expr = mk_equals_test_conjs g m (map mk_test fields) in 
  let expr = mk_physical_equality_equals_test g m tycon thise thate expr in
  let thatv,expr = 
    if is_struct_tycon tycon then 
      let thatv2,thatvref2,thate2 = mk_mut_local m "obj" ty in 
      thatv2,mk_let m thatv (mk_val_addr m thatvref2) expr
    else thatv,expr in 
  thisv,thatv, expr

(* Build the comparison expression for an exception constructor *)
let mk_exnconstr_compare g exnref exnc = 
  let m = range_of_tycon exnc in 
  let thatv,thate = mk_local m "obj" g.exn_ty in  
  let thisv,thise = mk_thisv m g.exn_ty in  
  let mk_test i fty = 
    mk_call_poly_compare_outer g m fty.rfield_type
      (mk_exnconstr_field_get(thise, exnref, i, m))
      (mk_exnconstr_field_get(thate, exnref, i, m))  in
  let expr = mk_compare_test_conjs g m (list_mapi mk_test (instance_rfields_of_tycon exnc)) in 
  let existential_tested =
    let mbuilder = MatchBuilder.create m in 
    let dtree = 
      TDSwitch(thate,
              [ mk_case(TTest_isinst(g.exn_ty,mk_tyapp_ty exnref []),
                        MatchBuilder.add_and_mk_result_target mbuilder expr) ],
               (* OK, this is gross - we are comparing types by comparing strings.  We should be able to do this another way. *)
                Some(MatchBuilder.add_and_mk_result_target mbuilder 
                     (mk_call_string_compare g m 
                        (mk_call_Object_GetType_GetString g m thise) 
                        (mk_call_Object_GetType_GetString g m thate))),
              m) in
    MatchBuilder.close dtree mbuilder m g.int_ty in
  let eq_tested = mk_physical_equality_compare_test g m exnc thise thate existential_tested in
  thisv,thatv, eq_tested


(* Build the equality expression for an exception constructor *)
let mk_exnconstr_equals g exnref exnc = 
  let m = range_of_tycon exnc in 
  let thatv,thate = mk_local m "obj" g.exn_ty in  
  let thisv,thise = mk_thisv m g.exn_ty in  
  let mk_test i fty = 
    mk_call_poly_equals_outer g m fty.rfield_type
      (mk_exnconstr_field_get(thise, exnref, i, m))
      (mk_exnconstr_field_get(thate, exnref, i, m))  in
  let expr = mk_equals_test_conjs g m (list_mapi mk_test (instance_rfields_of_tycon exnc)) in 
  let existential_tested =
    let mbuilder = MatchBuilder.create m in 
    let dtree = 
      TDSwitch(thate,
              [ mk_case(TTest_isinst(g.exn_ty,mk_tyapp_ty exnref []),
                        MatchBuilder.add_and_mk_result_target mbuilder expr) ],
                Some(MatchBuilder.add_and_mk_result_target mbuilder (mk_false g m)),
              m) in
    MatchBuilder.close dtree mbuilder m g.bool_ty in
  let eq_tested = mk_physical_equality_equals_test g m exnc thise thate existential_tested in
  thisv,thatv, eq_tested

(* Build the comparison expression for a union type *)
let mk_union_compare g tcref tycon = 
  let m = range_of_tycon tycon in 
  let constrs = uconstrs_of_tycon tycon in 
  let tinst,ty = minimal_type g tcref in
  let thisv,thise = mk_local m "this" ty in  
  let thatv,thate = mk_local m "obj" ty in  
  let thistagv,thistage = mk_local m "this_tag" g.int_ty in  
  let thattagv,thattage = mk_local m "that_tag" g.int_ty in  

  let expr1 = 
    let mbuilder = MatchBuilder.create m in 
    let mk_constr_case constr =
      let cref = ucref_of_uconstr tcref constr in 
      let m = range_of_ucref cref in 
      let mk_test j argty = 
        mk_call_poly_compare_outer g m argty.rfield_type
          (mk_uconstr_field_get(thise, cref, tinst, j, m))
          (mk_uconstr_field_get(thate, cref, tinst, j, m))  in
      let rfields = rfields_of_uconstr constr in 
      if isNil rfields then None else
      Some (mk_case(TTest_unionconstr(cref,tinst),
                    MatchBuilder.add_and_mk_result_target mbuilder (mk_compare_test_conjs g m (list_mapi mk_test rfields)))) in
    
    let nullary,non_nullary = partition (fun x -> x = None) (map mk_constr_case constrs) in  
    if isNil non_nullary then mk_zero g m else 
    let dtree = 
      TDSwitch(thise,map (function (Some c) -> c | None -> failwith "mk_union_compare") non_nullary, 
              (if isNil nullary then None 
              else Some (MatchBuilder.add_and_mk_result_target mbuilder (mk_zero g m))),
              m) in 
    MatchBuilder.close dtree mbuilder m g.int_ty in
  let get_tags = 
    if length constrs = 1 then expr1 else
    let tags_eq_tested = 
      mk_cond m g.int_ty  
        (mk_ceq g m thistage thattage)
        expr1
        (mk_asm ([ Il.I_arith Il.AI_sub  ],[],  [thistage; thattage],[g.int_ty],m))in 
    mk_let m thistagv
      (mk_uconstr_tag_get (thise,tcref,tinst,m))
      (mk_let m thattagv
           (mk_uconstr_tag_get (thate,tcref,tinst,m))
           tags_eq_tested) in 
  let nulltested_that = mk_nonnull_cond g m g.int_ty thate get_tags (mk_one g m) in 
  let nulltested_this = mk_nonnull_cond g m g.int_ty thise nulltested_that (mk_minus_one g m) in
  let eq_tested = mk_physical_equality_compare_test g m tycon thise thate nulltested_this in
  thisv,thatv, eq_tested

(* Build the equals expression for a union type *)
let mk_union_equals g tcref tycon = 
  let m = range_of_tycon tycon in 
  let constrs = uconstrs_of_tycon tycon in 
  let tinst,ty = minimal_type g tcref in
  let thisv,thise = mk_local m "this" ty in  
  let thatv,thate = mk_local m "obj" ty in  
  let thistagv,thistage = mk_local m "this_tag" g.int_ty in  
  let thattagv,thattage = mk_local m "that_tag" g.int_ty in  

  let expr1 = 
    let mbuilder = MatchBuilder.create m in 
    let mk_constr_case constr =
      let cref = ucref_of_uconstr tcref constr in 
      let m = range_of_ucref cref in 
      let mk_test j argty = 
        mk_call_poly_equals_outer g m argty.rfield_type
          (mk_uconstr_field_get(thise, cref, tinst, j, m))
          (mk_uconstr_field_get(thate, cref, tinst, j, m))  in
      let rfields = rfields_of_uconstr constr in 
      if isNil rfields then None else
      Some (mk_case(TTest_unionconstr(cref,tinst),
                    MatchBuilder.add_and_mk_result_target mbuilder (mk_equals_test_conjs g m (list_mapi mk_test rfields)))) in
    
    let nullary,non_nullary = partition (fun x -> x = None) (map mk_constr_case constrs) in  
    if isNil non_nullary then mk_true g m else 
    let dtree = 
      TDSwitch(thise,map (function (Some c) -> c | None -> failwith "mk_union_equals") non_nullary, 
              (if isNil nullary then None else Some (MatchBuilder.add_and_mk_result_target mbuilder (mk_true g m))),
              m) in 
    MatchBuilder.close dtree mbuilder m g.bool_ty in
  let get_tags = 
    if length constrs = 1 then expr1 else
    let tags_eq_tested = 
      mk_cond m g.bool_ty  
        (mk_ceq g m thistage thattage)
        expr1
        (mk_false g m)in 
    mk_let m thistagv
      (mk_uconstr_tag_get (thise,tcref,tinst,m))
      (mk_let m thattagv
           (mk_uconstr_tag_get (thate,tcref,tinst,m))
           tags_eq_tested) in 
  let nulltested_that = mk_nonnull_cond g m g.bool_ty thate get_tags (mk_false g m) in 
  let nulltested_this = mk_nonnull_cond g m g.bool_ty thise nulltested_that (mk_false g m) in
  let eq_tested = 
    if is_unit_typ g ty then mk_true g m else
    mk_physical_equality_equals_test g m tycon thise thate nulltested_this in
  thisv,thatv, eq_tested



(*-------------------------------------------------------------------------
 * Build hashing functions for union, record and exception types.
 * Hashing functions must respect the "=" and comparison operators.
 *------------------------------------------------------------------------- *)

let mk_return_acce m stmt acce = mk_seq stmt acce m

let mk_recd_hash g tcref tycon = 
  let m = range_of_tycon tycon in 
  let fields = instance_rfields_of_tycon tycon in 
  let tinst,ty = minimal_type g tcref in
  let (thisv,thise,nv,ne) = hash_vars g m ty in 
  let mk_field_hash fspec = 
    let fty = fspec.rfield_type in 
    let fref = rfref_of_rfield tcref fspec in 
    let m = range_of_rfref fref in 
    
    (* REVIEW: add type specializations into defs. of poly_hash and poly_compare *)
    (* This can result in better code for these calls when types are known (as the often are) *)
    mk_call_poly_hash_param_outer g m fty (mk_recd_field_get_via_expra(thise, fref, tinst, m))  ne in
  let accv,accvr,acce = mk_mut_local m "i" g.int_ty in                  
  let stmt = 
    decr_nodes_and_then g m nv
      (mk_combine_hash_generators g m nv (map mk_field_hash fields) accvr acce) in 
  let expr = mk_let m accv (mk_zero g m) stmt in 
  thisv,nv, expr

let mk_exnconstr_hash g exnref exnc = 
  let m = range_of_tycon exnc in 
  let (thisv,thise,nv,ne) = hash_vars g m g.exn_ty in 
  let mk_hash i fty = 
     mk_call_poly_hash_param_outer g m fty.rfield_type (mk_exnconstr_field_get(thise, exnref, i, m))  ne in
  let accv,accvr,acce = mk_mut_local m "i" g.int_ty in                  
  let stmt = 
    decr_nodes_and_then g m nv
      (mk_combine_hash_generators g m nv (list_mapi mk_hash (instance_rfields_of_tycon exnc)) accvr acce) in 
  let expr = mk_let m accv (mk_zero g m) stmt in 
  thisv, nv,expr

let mk_union_hash g tcref tycon = 
  let m = range_of_tycon tycon in 
  let constrs = uconstrs_of_tycon tycon in 
  let tinst,ty = minimal_type g tcref in
  let (thisv,thise,nv,ne) = hash_vars g m ty in 
  let mbuilder = MatchBuilder.create m in 
  let accv,accvr,acce = mk_mut_local m "i" g.int_ty in                  
  let mk_constr_case i constr1 = 
    let c1ref = ucref_of_uconstr tcref constr1 in 
    let m = range_of_ucref c1ref in 
    let mk_hash j argty =  mk_call_poly_hash_param_outer g m argty.rfield_type (mk_uconstr_field_get(thise, c1ref, tinst, j, m)) ne in
    mk_case(TTest_unionconstr(c1ref,tinst),
            MatchBuilder.add_and_mk_result_target mbuilder 
              (mk_seq m (mk_val_set m accvr (mk_int g m i)) 
                        (mk_combine_hash_generators g m nv (list_mapi mk_hash (rfields_of_uconstr constr1)) accvr acce))) in
  let dtree = TDSwitch(thise,list_mapi mk_constr_case constrs, None,m) in 
  let stmt = 
    decr_nodes_and_then g m nv
      (MatchBuilder.close dtree mbuilder m g.unit_ty) in 
  let expr = mk_let m accv (mk_zero g m) stmt in 
  thisv, nv,expr

(*-------------------------------------------------------------------------
 * The predicate that determines which types implement the 
 * pre-baked IStructuralHash and IComparable semantics associated with F#
 * types.  Note abstract types are not _known_ to implement these interfaces,
 * though the interfaces may be discoverable via type tests.
 *------------------------------------------------------------------------- *)

let is_nominal_exnc exnc = 
      match exn_repr_of_tycon exnc with 
      | TExnAbbrevRepr _ | TExnNone | TExnAsmRepr _ -> false
      | TExnFresh _ -> true

let is_true_struct_tycon g tycon = 
  (is_fsobjmodel_struct_tycon tycon  && not (is_fsobjmodel_enum_tycon tycon))

let can_be_augmented g tycon = 
  is_union_tycon tycon or 
  is_recd_tycon tycon or 
  (tycon_is_exnc tycon && is_nominal_exnc tycon) or
  is_true_struct_tycon g tycon

let augmentation_attribs g tycon = 
        can_be_augmented g tycon,
        fsthing_bool_attrib g g.attrib_ReferenceEqualityAttribute (attribs_of_tycon tycon),
        fsthing_bool_attrib g g.attrib_StructuralEqualityAttribute (attribs_of_tycon tycon),
        fsthing_bool_attrib g g.attrib_StructuralComparisonAttribute (attribs_of_tycon tycon) 

let check_augmentation_attribs g tycon = 
  let m = range_of_tycon tycon in
  let attribs = augmentation_attribs g tycon in
  begin match attribs with 
  
  (* THESE ARE THE LEGITIMATE CASES *)

  | true, Some(true), None      , None ->
      if is_true_struct_tycon g tycon then 
          errorR(Error("The 'ReferenceEquality' attribute may not be used on structs. Consider using the 'StructuralEquality' attribute instead, or implement an override for 'System.Object.Equals(obj)'", m))
      else ()

  (* [< >] *)
  | _,  None,      None, None
  (* [<ReferenceEquality(true)>] *)
  (* [<StructuralEquality(true); StructuralComparison(true)>] *)
  | true, None      , Some(true), Some(true) 
  (* [<StructuralEquality(false); StructuralComparison(false)>] *)
  | true, None      , Some(false), Some(false) 
  (* [<StructuralEquality(true); StructuralComparison(false)>] *)
  | true, None      , Some(true), Some(false) ->
      () 

  (* THESE ARE THE ERROR CASES *)

  (* [<ReferenceEquality(false); ...>] *)
  | _, Some(false), _      , _ ->
      errorR(Error("The 'ReferenceEquality' attribute may not be 'false'. Consider using the 'StructuralEquality' attribute instead, or implement an override for 'System.Object.Equals(obj)'", m))
  (* [<StructuralEquality(false); ...>] *)
  | _,           _, Some(false), _ ->
      errorR(Error("The 'StructuralEquality' attribute may not be 'false'. Consider using the 'ReferenceEquality' attribute instead, or implement an override for 'System.Object.Equals(obj)'", m))
        
  (* [<StructuralComparison(_)>] *)
  | true, None      , None      , Some(_) ->
      errorR(Error("The 'StructuralComparison' attribute must be used in conjunction with the 'StructuralEquality' attribute", m))
  (* [<StructuralEquality(_)>] *)
  | true, None      , Some(true), None ->
      errorR(Error("The 'StructuralEquality' attribute must be used in conjunction with the 'StructuralComparison' attribute", m))

  (* [<ReferenceEquality; StructuralEquality>] *)
  | true, Some(_)  , Some(_)    ,      _
  (* [<ReferenceEquality; StructuralComparison(_) >] *)
  | true, Some(_),             _, Some(_) -> 
      errorR(Error("A type may not have both the 'ReferenceEquality' and 'StructuralEquality' or 'StructuralComparison' attributes", m))

  (* non augmented type, [<ReferenceEquality; ... >] *)
  | false,  Some(_),           _, _
  (* non augmented type, [<StructuralEquality; ... >] *)
  | false,        _, Some(_)    , _
  (* non augmented type, [<StructuralComparison(_); ... >] *)
  | false,        _,      _     , Some(_) ->
      errorR(Error("Only record, union, exception and struct types may be augmented with the 'ReferenceEquality', 'StructuralEquality' and 'StructuralComparison' attributes", m))
  end;
  let tcaug = tcaug_of_tycon tycon in
  
  let hasExplicitIStructuralHash = 
      isNone tcaug.tcaug_compare && tcaug_has_interface g tcaug g.mk_IStructuralHash_ty in 
  let hasExplicitICompare = 
      isNone tcaug.tcaug_compare && tcaug_has_interface g tcaug g.mk_IStructuralHash_ty in 
  let hasExplicitEquals = 
      isNone tcaug.tcaug_equals && tcaug_has_override g tcaug "Equals" [g.obj_ty] in

  begin match attribs with 
  (* [<ReferrnceEquality(true)>] *)
  | _, Some(true), _, _ when hasExplicitEquals -> 
      errorR(Error("A type with attribute 'ReferenceEquality' may not have an explicit implementation of 'Object.Equals(obj)'", m))
  | _, Some(true), _, _ when hasExplicitIStructuralHash -> 
      errorR(Error("A type with attribute 'ReferenceEquality' may not have an explicit implementation of 'IStructuralHash'", m))
  | _,          _, Some(true), _ when hasExplicitEquals -> 
      errorR(Error("A type with attribute 'StructuralEquality' may not have an explicit implementation of 'Object.Equals(obj)'", m))
  | _,          _, _,  Some(true) when hasExplicitICompare -> 
      errorR(Error("A type with attribute 'StructuralComparison' may not have an explicit implementation of 'System.IComparable'", m))
  | _ -> ()
  end  

let is_augmented_with_compare g tycon = 
  match augmentation_attribs g tycon with 
  (* [< >] *)
  | true, None, None      , None
  (* [<StructuralEquality(true); StructuralComparison(true)>] *)
  | true, None, Some(true), Some(true) -> true
  (* other cases *)
  | _ -> false

let is_augmented_with_equals g tycon = 
  match augmentation_attribs g tycon with 
  (* [< >] *)
  | true, None, None      , _
  (* [<StructuralEquality(true); _ >] *)
  (* [<StructuralEquality(true); StructuralComparison(true)>] *)
  | true, None, Some(true), _ -> true
  (* other cases *)
  | _ -> false

let is_augmented_with_hash g tycon = is_augmented_with_equals g tycon
      
(*-------------------------------------------------------------------------
 * Make values that represent the implementations of the 
 * IStructuralHash and IComparable semantics associated with F#
 * types.  
 *------------------------------------------------------------------------- *)

let slotImplMethod (final,ilnm,c,slotsig) = 
  { vspr_implements_slotsig=Some(slotsig);
    vspr_flags={ memFlagsOverloadQualifier=None;
                 memFlagsInstance=true; 
                 memFlagsVirtual=false;
                 memFlagsAbstract=false;
                 memFlagsFinal=final;
                 memFlagsOverride=true;
                 memFlagsKind=MemberKindMember};
    vspr_implemented=false;
    vspr_il_name=ilnm; 
    vspr_apparent_parent=c} 

let nonVirtualMethod (ilnm,c) = 
  { vspr_implements_slotsig=None;
    vspr_flags={ memFlagsOverloadQualifier=None;
                 memFlagsInstance=true; 
                 memFlagsVirtual=false;
                 memFlagsAbstract=false;
                 memFlagsFinal=false;
                 memFlagsOverride=false;
                 memFlagsKind=MemberKindMember};
    vspr_implemented=false;
    vspr_il_name=ilnm; 
    vspr_apparent_parent=c} 


let mk_vspec tcref tmty cpath vis methn slotsig ilnm ty = 
  let tname = name_of_tcref tcref in 
  let m = range_of_tcref tcref in 
  let tps = typars_of_tcref tcref in 
  let id = mksyn_id m methn in
  let final = is_union_ty tmty or is_recd_ty tmty or is_struct_typ tmty in 
  let membInfo = match slotsig with None -> nonVirtualMethod (ilnm,tcref) | Some(slotsig) -> slotImplMethod(final,ilnm,tcref,slotsig) in 
  let inl = 
    match tname with 
    | "Tuple`2" | "Tuple`3" | "Tuple`4" | "Tuple`5"  -> AlwaysInline
    | _ -> OptionalInline in 
  let topValInfo = Some (TopValInfo (length tps,[TopValData.unnamedTopArg;TopValData.unnamedTopArg],TopValData.unnamedRetVal)) in 
  new_vspec (id,ty,Immutable,true,topValInfo,cpath,vis,ValNotInRecScope,Some(membInfo),NormalVal,[],inl,emptyXMLDoc,true,false,false,false,None,Parent(tcref)) 

let mk_vspecs_for_compare_augmentation g tcref = 
    let tycon = deref_tycon tcref in 
    let _,tmty = minimal_type g tcref in
    let tname = name_of_tcref tcref in 
    let tps = typars_of_tcref tcref in 
    let vis = repr_access_of_tcref tcref in
    let cpath = cpathopt_of_tycon tycon in 
    let cvspec = mk_vspec tcref tmty cpath vis (tname^".CompareTo" ) (Some(mk_IComparable_CompareTo_slotsig g)) "CompareTo" (tps +-> (mk_compare_ty g tmty)) in 
    cvspec

let mk_vspecs_for_hash_augmentation g tcref = 
    let tycon = deref_tycon tcref in 
    let _,tmty = minimal_type g tcref in
    let tname = name_of_tcref tcref in 
    let tps = typars_of_tcref tcref in 
    let vis = repr_access_of_tcref tcref in
    let cpath = cpathopt_of_tycon tycon in 
    let hvspec = mk_vspec tcref tmty cpath vis (tname^".GetStructuralHashCode" ) (Some(mk_GetStructuralHashCode_slotsig g)) "GetStructuralHashCode" (tps +-> (mk_hash_param_ty g tmty)) in 
    hvspec

let mk_vspecs_for_equals_augmentation g tcref = 
    let tycon = deref_tycon tcref in 
    let _,tmty = minimal_type g tcref in
    let tname = name_of_tcref tcref in 
    let tps = typars_of_tcref tcref in 
    let vis = repr_access_of_tcref tcref in
    let cpath = cpathopt_of_tycon tycon in 
    let obj_vspec = mk_vspec tcref tmty cpath vis (tname^".EqualsOverride" ) (Some(mk_Equals_slotsig g)) "Equals" (tps +-> (mk_equals_obj_ty g tmty)) in
    let vspec = mk_vspec tcref tmty cpath vis (tname^".Equals" ) None "Equals" (tps +-> (mk_rel_ty g tmty)) in
    obj_vspec,vspec

let mk_bindings_for_compare_augmentation g tcref tycon = 
  let m = range_of_tycon tycon in 
  let tps = typars_of_tycon tycon in
  let mk_compare comparef =
    match (tcaug_of_tycon tycon).tcaug_compare with 
    | None ->  []
    | Some vref -> 
        let vspec = deref_val vref in 
        let rhs = 
          let thisv,thatv,comparee = comparef g tcref tycon in 
          mk_lambdas m tps [thisv;thatv] (comparee,g.int_ty)  in 
        [ mk_bind vspec rhs ] in 
  if is_union_tycon tycon then mk_compare mk_union_compare 
  else if is_recd_tycon tycon or is_struct_tycon tycon then mk_compare mk_recd_compare 
  else if tycon_is_exnc tycon then mk_compare mk_exnconstr_compare 
  else []

let mk_bindings_for_hash_augmentation g tcref tycon = 
  let m = range_of_tycon tycon in 
  let tps = typars_of_tycon tycon in
  let mk_hash hashf = 
    match (tcaug_of_tycon tycon).tcaug_structural_hash with 
    | None ->  []
    | Some vref -> 
        let rhs = 
          let thisv,nv,hashe = hashf g tcref tycon in 
          mk_lambdas m tps [thisv;nv] (hashe,g.int_ty) in
        [ mk_bind (deref_val vref) rhs ] in 
  if is_union_tycon tycon then mk_hash mk_union_hash
  else if is_recd_tycon tycon or is_struct_tycon tycon then mk_hash mk_recd_hash
  else if tycon_is_exnc tycon then mk_hash mk_exnconstr_hash
  else []

let mk_bindings_for_equals_augmentation g tcref tycon = 
  let m = range_of_tycon tycon in 
  let tps = typars_of_tycon tycon in
  let mk_equals equalsf =
    match (tcaug_of_tycon tycon).tcaug_equals with 
    | None ->  []
    | Some (vref1,vref2) -> 
        (* this is the body of the override *)
        let rhs1 = 
          let tinst,ty = minimal_type g tcref in
          
          let thisv,thise = mk_thisv m ty in  
          let thatobjv,thatobje = mk_local m "obj" g.obj_ty in  
          let equalse = 
              if is_unit_typ g ty then mk_true g m else

              let thatv,thate = mk_local m "that" ty in  
              mk_isinst_cond g m ty (thatobjv,thatobje) thatv 
                  (mk_appl((expr_for_vref m vref2,type_of_vref vref1), (if isNil tinst then [] else [tinst]), [thise;thate], m))
                  (mk_false g m) in
          
          mk_lambdas m tps [thisv;thatobjv] (equalse,g.bool_ty)  in 
        (* this is the body of the real strongly typed implementation *)
        let rhs2 = 
          let thisv,thatv,equalse = equalsf g tcref tycon in 
          mk_lambdas m tps [thisv;thatv] (equalse,g.bool_ty)  in 
          
        [ mk_bind (deref_val vref2) rhs2;
          mk_bind (deref_val vref1) rhs1;   ] in 
  if tycon_is_exnc tycon then mk_equals mk_exnconstr_equals 
  else if is_union_tycon tycon then mk_equals mk_union_equals 
  else if is_recd_tycon tycon or is_struct_tycon tycon then mk_equals mk_recd_equals 
  else []



