(*F# 
module Microsoft.Research.AbstractIL.Internal.Support 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
F#*) 
open Ildiag
open Bytes
open System
open Compatibility
open System.Text

module IlsuppSupport = begin
    open System
    open System.Text
    open System.Runtime.InteropServices  
    open System.Reflection
    open System.Diagnostics

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>] 
    extern int resLinkNativeResource(byte[] unlinkedResource, 
                                     int unlinkedResourceSize,
                                     bool sscli, 
                                     int rva, 
                                     byte[] resBuffer, 
                                     int &resSize)

    [<DllImport("absilsuppc.dll", CharSet = CharSet.Auto)>]
    extern int resUnlinkNativeResource(int n, 
                                       byte[] linkedResource, 
                                       int linkedResourceSize, 
                                       byte[] unlinkedResource, 
                                       int &unlinkedResourceSize) 

    [<DllImport("absilsuppc.dll", CharSet = CharSet.Auto)>]
    extern int pdbWriteInitialize(string binaryName, 
                                  string pdbName, 
                                  bool sscli, 
                                  int versionHint, 
                                  IntPtr &pWriter, 
                                  IntPtr &pImport)

    [<DllImport("absilsuppc.dll")>]   
    let pdbWriteClose((pWriter: IntPtr), 
                      (pImport: IntPtr)) : int = failwith ""

    [<DllImport("absilsuppc.dll")>]   
    let pdbWriteSetUserEntryPoint((pWriter: IntPtr), 
                                  (token:int)) : int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]
    let pdbWriteDefineDocument((pWriter: IntPtr), 
                               (docName:string), 
                               (pdbDocumentWriter: IntPtr byref)) : int = failwith ""

    [<DllImport("absilsuppc.dll")>]   
    let pdbWriteOpenMethod((pWriter: IntPtr), 
                           (n : int)) : int = failwith ""

    [<DllImport("absilsuppc.dll")>]   
    let pdbWriteCloseMethod((pWriter: IntPtr)): int = failwith ""

    [<DllImport("absilsuppc.dll")>]   
    let pdbWriteOpenScope((pWriter: IntPtr), 
                          (n : int)): int = failwith ""

    [<DllImport("absilsuppc.dll")>]   
    let pdbWriteCloseScope((pWriter: IntPtr), 
                           (n : int)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbWriteDefineLocalVariable((pWriter: IntPtr), 
                                    (varName:string), 
                                    (sgn:byte[]), 
                                    (sigSize:int), 
                                    (attrib:int)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbWriteSetMethodRange((pWriter: IntPtr), 
                               (pdbDocumentWriter: IntPtr), 
                               (n1 : int), 
                               (n2 : int), 
                               (pdbDocumentWriter2: IntPtr), 
                               (n3 : int), 
                               (n4 : int)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbWriteDefineSequencePoints((pWriter: IntPtr), 
                                     (pdbDocumentWriter: IntPtr), 
                                     (nPoints : int), 
                                     (offsets: int[]), 
                                     (lines: int[]), 
                                     (columns: int[]),
                                     (endlines:  int[]), 
                                     (endColumns: int[])): int = failwith ""


    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbWriteGetDebugInfo((pWriter: IntPtr), 
                                 (pCharacteristics: int byref), 
                                 (pMajorVersion : int byref), 
                                 (pMinorVersion : int byref), 
                                 (pType: int byref),
                                 (iddDataBuffer: byte[]),
                             (iddDataSize : int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let signerReadBinaryFile((sscli : bool), 
                             (snkFile:string), 
                             (publicKey : byte[]), 
                             (publicKeySize:int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let signerGetPublicKeyForKeyPair((sscli : bool), 
                                     (keyPair:byte[]), 
                                     (keyPairSize:int), 
                                     (publicKey : byte[]), 
                                     (publicKeySize:int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let signerGetPublicKeyForKeyContainer((sscli : bool), 
                                          (keyContainer: string), 
                                          (publicKey : byte[]), 
                                          (publicKeySize:int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]  
    let  signerCloseKeyContainer((sscli : bool), 
                                 (keyContainer: string)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]  
    let  signerSignatureSize((sscli : bool), 
                             (publicKey : byte[]), 
                             (publicKeySize:int), 
                             (size : int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]
    let signerSignFileWithKeyPair((sscli : bool), 
                                  (n : string), 
                                  (keyPair:byte[]), 
                                  (keyPairSize:int)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]
    let signerSignFileWithKeyContainer((sscli : bool), 
                                       (n : string), 
                                       (keyContainer: string)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbReadOpen((moduleName : string), 
                    (path : string), 
                    (pdbReader : IntPtr byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbReadClose((pdbReader : IntPtr)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbReaderGetMethod((pdbReader : IntPtr), 
                           (token:int), 
                           (pdbMethod : IntPtr byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]
    let pdbReaderGetMethodFromDocumentPosition((pdbReader : IntPtr), 
                                               (pdbDoc : IntPtr), 
                                               (line:int), 
                                               (col:int), 
                                               (pdbMethod : IntPtr byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]
    let pdbReaderGetDocuments((pdbReader : IntPtr), 
                              (pDocuments:IntPtr[]), 
                              (numDocuments : int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]  
    let pdbReaderGetDocument((pdbReader : IntPtr), 
                             (url:string), 
                             (guid1:byte[]), 
                             (guid2:byte[]), 
                             (guid3:byte[]), 
                             (pdbDoc : IntPtr byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbDocumentGetURL((pdbDoc : IntPtr), 
                          (resBuffer:StringBuilder), 
                          (resSize: int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbDocumentGetType((pdbDoc : IntPtr), 
                           (guid:byte[])): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbDocumentGetLanguage((pdbDoc : IntPtr), 
                               (guid:byte[])): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbDocumentGetLanguageVendor((pdbDoc : IntPtr), 
                                     (guid:byte[])): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet = CharSet.Auto)>]
    let pdbDocumentFindClosestLine((pdbDoc : IntPtr), 
                                   (n1 : int), 
                                   (n2 : int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbMethodGetToken((pMethod:IntPtr), 
                          (token:int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbMethodGetRootScope((pMethod:IntPtr), 
                              (pdbMethodScope : IntPtr byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbMethodGetSequencePoints((pMethod:IntPtr), 
                                   (nPoints : int byref), 
                                   (offsets:int[]),
                       (documents:IntPtr[]),
                       (lines:int[]),
                       (columns:int[]),
                       (endLines:int[]),
                       (endColumns:int[])): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]
    let pdbScopeGetChildren((pMethodScope:IntPtr), 
                            (childScopes:IntPtr[]), 
                            (nChildScopes : int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]  
    let  pdbScopeGetOffsets((pMethodScope:IntPtr), 
                            (offset1: int byref), 
                            (offset2: int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]
    let pdbScopeGetLocals((pMethodScope:IntPtr), 
                          (localsBuffer:IntPtr[]), 
                          (nLocals : int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbVariableGetName((pdbVariable:IntPtr),  
                           (pName:StringBuilder), 
                           (nName : int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]
    let pdbVariableGetSignature((pdbVariable:IntPtr), 
                                (pRes:byte[]), 
                                (nRes : int byref)): int = failwith ""

    [<DllImport("absilsuppc.dll", CharSet=CharSet.Auto)>]   
    let pdbVariableGetAddressAttributes((pdbVariable:IntPtr), 
                                        (attr1:int byref), 
                                        (attr2:int byref)): int = failwith ""



end

type clr_kind = | SSCLI | Microsoft | Mono | Neutral
let supported_configurations = [SSCLI;Microsoft;Mono;Neutral]

let current_configuration () = Microsoft (* REVIEW: this should detect which CLR we are running *)
let configure k = 
  if k <> current_configuration() then 
    dprint_endline ("configure: cannot change the current configuration for the Abstract IL managed libraries - it is determined by the CLR you are using to run your application")

let sscli() = (current_configuration() = SSCLI)

let check action (hresult) = 
  if Int32.to_uint32 hresult >= 0x80000000ul then 
    System.Runtime.InteropServices.Marshal.ThrowExceptionForHR(hresult)
  //Printf.printf "action = %s, hresult = 0x%nx \n" action hresult

let pdb_suffix_for_configuration = function | Microsoft -> "pdb" | _ -> "ildb"

let linkNativeResource (s:bytes) (rva:int32) (buf:bytes) = 
  let mutable pSize = 0 in 
  check "action" (IlsuppSupport.resLinkNativeResource(s,Bytes.length s,sscli(),rva,buf,&pSize));
  pSize
  
type pdb_writer = 
  { pWriter: nativeint; 
    pImport: nativeint }  (* pointer to pWriter COM object *)
type pdb_document_writer = { pDocumentWriter: nativeint }  (* pointer to pDocumentWriter COM object *)

type idd =
    { iddCharacteristics: int32;
      iddMajorVersion: int32; (* actually u16 in IMAGE_DEBUG_DIRECTORY *)
      iddMinorVersion: int32; (* acutally u16 in IMAGE_DEBUG_DIRECTORY *)
      iddType: int32;
      iddData: Bytes.bytes;}

let pdbInitialize (s1:string) (s2:string) (versionHint:int) = 
  let mutable pRes1 = (new IntPtr(0)) in
  let mutable pRes2 = (new IntPtr(0)) in
  check "action" (IlsuppSupport.pdbWriteInitialize(s1,s2,sscli(),versionHint,&pRes1, &pRes2));
  {pWriter = pRes1; pImport = pRes2 }
  
let pdbClose (pdbw:pdb_writer) = 
  check "action" (IlsuppSupport.pdbWriteClose(pdbw.pWriter, pdbw.pImport))
  
let pdbSetUserEntryPoint (pdbw:pdb_writer) (n:int32) = 
  check "action" (IlsuppSupport.pdbWriteSetUserEntryPoint(pdbw.pWriter, n))
let pdbDefineDocument (pdbw:pdb_writer) (s:string) = 
  let mutable pRes = (new IntPtr(0)) in
  check "action" (IlsuppSupport.pdbWriteDefineDocument(pdbw.pWriter, s,&pRes));
  {pDocumentWriter = pRes }

let pdbOpenMethod (pdbw:pdb_writer) (n:int32) = 
  check "action" (IlsuppSupport.pdbWriteOpenMethod(pdbw.pWriter, n))

let pdbCloseMethod (pdbw:pdb_writer) = 
  check "action" (IlsuppSupport.pdbWriteCloseMethod(pdbw.pWriter))

let pdbOpenScope (pdbw:pdb_writer) (n:int32) = 
  check "action" (IlsuppSupport.pdbWriteOpenScope(pdbw.pWriter,n))

let pdbCloseScope (pdbw:pdb_writer) (n:int32) = 
  check "action" (IlsuppSupport.pdbWriteCloseScope(pdbw.pWriter,n))

let pdbDefineLocalVariable (pdbw:pdb_writer) (s:string) (signature:bytes) (n:int32) = 
  check "action" (IlsuppSupport.pdbWriteDefineLocalVariable(pdbw.pWriter,s,signature,Bytes.length signature, n))

let pdbSetMethodRange (pdbw:pdb_writer) (pdbww: pdb_document_writer) (n:int) (m:int) (pdbww2: pdb_document_writer) (n2:int) (m2:int) = 
  check "action" (IlsuppSupport.pdbWriteSetMethodRange(pdbw.pWriter,pdbww.pDocumentWriter,n,m,pdbww2.pDocumentWriter, n2,m2))

let pdbDefineSequencePoints (pdbw:pdb_writer) (pdbww: pdb_document_writer) (pts: (int * int * int * int * int) array)  = 
  let arr1 = CompatArray.of_array (Array.map (fun (x,_,_,_,_) -> x) pts) in 
  let arr2 = CompatArray.of_array (Array.map (fun (_,x,_,_,_) -> x) pts) in 
  let arr3 = CompatArray.of_array (Array.map (fun (_,_,x,_,_) -> x) pts) in 
  let arr4 = CompatArray.of_array (Array.map (fun (_,_,_,x,_) -> x) pts) in 
  let arr5 = CompatArray.of_array (Array.map (fun (_,_,_,_,x) -> x) pts) in 
  check "action" (IlsuppSupport.pdbWriteDefineSequencePoints(pdbw.pWriter,pdbww.pDocumentWriter,Array.length pts, arr1,arr2,arr3,arr4,arr5))

let pdbGetDebugInfo (pdbw: pdb_writer) = 
  let mutable iddCharacteristics = 0 in 
  let mutable iddMajorVersion =  0 in 
  let mutable iddMinorVersion = 0 in  
  let mutable iddType = 0 in 
  let mutable pDataSize = 0 in 
  check "action" (IlsuppSupport.pdbWriteGetDebugInfo(pdbw.pWriter,&iddCharacteristics,&iddMajorVersion,&iddMinorVersion,&iddType,null,&pDataSize));
  let iddData = Bytes.zero_create pDataSize in 
  check "action" (IlsuppSupport.pdbWriteGetDebugInfo(pdbw.pWriter,&iddCharacteristics,&iddMajorVersion,&iddMinorVersion,&iddType,iddData,&pDataSize));
  { iddCharacteristics= iddCharacteristics;
    iddMajorVersion= iddMajorVersion;
    iddMinorVersion= iddMinorVersion;
    iddType= iddType;
    iddData= iddData}

type keyContainerName = string
type keyPair = bytes
type pubkey = bytes

let signerOpenPublicKeyFile f = 
  let mutable pSize = 0 in 
  check "action" (IlsuppSupport.signerReadBinaryFile(sscli(),f,null,&pSize));
  let res = Bytes.zero_create pSize in 
  check "action" (IlsuppSupport.signerReadBinaryFile(sscli(),f,res,&pSize));
  res

  
let signerOpenKeyPairFile f = 
  let mutable pSize = 0 in 
  check "action" (IlsuppSupport.signerReadBinaryFile(sscli(),f,null,&pSize));
  let res = Bytes.zero_create pSize in 
  check "action" (IlsuppSupport.signerReadBinaryFile(sscli(),f,res,&pSize));
  res
  
let signerGetPublicKeyForKeyPair kp = 
  let mutable pSize = 0 in 
  check "action" (IlsuppSupport.signerGetPublicKeyForKeyPair(sscli(),kp,Bytes.length kp,null,&pSize));
  let res = Bytes.zero_create pSize in 
  check "action" (IlsuppSupport.signerGetPublicKeyForKeyPair(sscli(),kp,Bytes.length kp,res,&pSize));
  res

let signerGetPublicKeyForKeyContainer kc =
  let mutable pSize = 0 in 
  check "action" (IlsuppSupport.signerGetPublicKeyForKeyContainer(sscli(),kc,null,&pSize));
  let res = Bytes.zero_create pSize in 
  check "action" (IlsuppSupport.signerGetPublicKeyForKeyContainer(sscli(),kc,res,&pSize));
  res
 
let signerCloseKeyContainer kc = 
  check "signerCloseKeyContainer" (IlsuppSupport.signerCloseKeyContainer(sscli(),kc))

let signerSignatureSize pk = 
  let mutable pSize =  0 in 
  check "signerSignatureSize" (IlsuppSupport.signerSignatureSize(sscli(),pk,Bytes.length pk,&pSize));
  pSize

let signerSignFileWithKeyPair n kp = 
  check "signerSignFileWithKeyPair" (IlsuppSupport.signerSignFileWithKeyPair(sscli(),n,kp,Bytes.length kp))

let signerSignFileWithKeyContainer n kc = 
  check "action" (IlsuppSupport.signerSignFileWithKeyContainer(sscli(),n,kc))

let absilWriteGetTimeStamp () = Int64.to_int32 (System.DateTime.Now.Ticks)
let absilWriteGetMetadataVersion () = System.Runtime.InteropServices.RuntimeEnvironment.GetSystemVersion()

let fsharpServiceGetFileTimeStamp (s:string) = 
  let d = System.IO.File.GetLastWriteTime(s) in 
  d.Ticks


let clrInstallationDirectory () = System.Runtime.InteropServices.RuntimeEnvironment.GetRuntimeDirectory()
let clrVersion () = System.Runtime.InteropServices.RuntimeEnvironment.GetSystemVersion()

let unlinkResource (n:int32) (s:bytes) = 
  let mutable pSize = 0 in 
  check "action" (IlsuppSupport.resUnlinkNativeResource(n,s,Bytes.length s,null,&pSize));
  let bytes = Bytes.zero_create (pSize) in
  check "action" (IlsuppSupport.resUnlinkNativeResource(n,s,Bytes.length s,bytes,&pSize));
  bytes

type pdb_reader  = { pReader: nativeint }
type pdb_document  = { pDocument: nativeint }
type pdb_method  = { pMethod: nativeint }
type pdb_variable = { pVariable: nativeint }
type pdb_method_scope = { pScope: nativeint }

type pdb_sequence_point = 
    { pdbSeqPointOffset: int;
      pdbSeqPointDocument: pdb_document;
      pdbSeqPointLine: int;
      pdbSeqPointColumn: int;
      pdbSeqPointEndLine: int;
      pdbSeqPointEndColumn: int; }

let pdbReadOpen (m:string) (path:string) :  pdb_reader =  
  let mutable pRes = (new IntPtr(0)) in
  check "pdbReadOpen" (IlsuppSupport.pdbReadOpen(m,path,&pRes));
  { pReader = pRes }

let pdbReadClose (x:pdb_reader) : unit =  
  check "pdbReadClose" (IlsuppSupport.pdbReadClose(x.pReader))

let pdbReaderGetMethod (x:pdb_reader) (y:int32) : pdb_method = 
  let mutable pRes = (new IntPtr(0)) in
  check "pdbReaderGetMethod" (IlsuppSupport.pdbReaderGetMethod(x.pReader,y,&pRes));
  { pMethod = pRes }

let pdbReaderGetMethodFromDocumentPosition (r:pdb_reader)  (x:pdb_document) (y:int) (z:int) : pdb_method = 
  let mutable pRes = (new IntPtr(0)) in
  check "pdbReaderGetMethodFromDocumentPosition" (IlsuppSupport.pdbReaderGetMethodFromDocumentPosition(r.pReader,x.pDocument,y,z,&pRes));
  { pMethod = pRes }

let pdbReaderGetDocuments (x:pdb_reader) : pdb_document array = 
  let mutable pSize = 0 in 
  check "pdbReaderGetDocuments" (IlsuppSupport.pdbReaderGetDocuments(x.pReader, null, &pSize));
  let arr = CompatArray.zero_create pSize in
  check "pdbReaderGetDocuments" (IlsuppSupport.pdbReaderGetDocuments(x.pReader, arr, &pSize));
  Array.map (fun i -> { pDocument=i }) (CompatArray.to_array(arr))

  
let pdbReaderGetDocument (x:pdb_reader) (y:string) (z:bytes) (w:bytes) (u:bytes) : pdb_document = 
  let mutable pRes = (new IntPtr(0)) in
  check "pdbReaderGetDocument" (IlsuppSupport.pdbReaderGetDocument(x.pReader,y,z,w,u,&pRes));
  { pDocument = pRes }

let pdbDocumentGetURL (x:pdb_document) : string = 
  let mutable pSize = 0 in 
  check "pdbDocumentGetURL" (IlsuppSupport.pdbDocumentGetURL(x.pDocument,null,&pSize));
  let buff = new StringBuilder(pSize) in
  check "pdbDocumentGetURL" (IlsuppSupport.pdbDocumentGetURL(x.pDocument,buff,&pSize));
  buff.ToString()

let pdbDocumentGetType (x:pdb_document) : bytes (* guid *) = 
  let size = (Idioms.sizeof() : Idioms.sizeof<System.Guid>).result  in 
  let guid = Bytes.zero_create size in
  check "pdbDocumentGetType" (IlsuppSupport.pdbDocumentGetType(x.pDocument,guid));
  guid

let pdbDocumentGetLanguage (x:pdb_document) : bytes (* guid *) = 
  let size = (Idioms.sizeof() : Idioms.sizeof<System.Guid>).result  in 
  let guid = Bytes.zero_create size in
  check "pdbDocumentGetLanguage" (IlsuppSupport.pdbDocumentGetLanguage(x.pDocument,guid));
  guid
let pdbDocumentGetLanguageVendor (x:pdb_document) : bytes = 
  let size = (Idioms.sizeof() : Idioms.sizeof<System.Guid>).result  in 
  let guid = Bytes.zero_create size in
  check "pdbDocumentGetLanguageVendor" (IlsuppSupport.pdbDocumentGetLanguageVendor(x.pDocument,guid));
  guid
  
let pdbDocumentFindClosestLine (x:pdb_document) (y:int) : int = 
  let mutable pRes = 0 in
  check "pdbDocumentFindClosestLine" (IlsuppSupport.pdbDocumentFindClosestLine(x.pDocument,y,&pRes));
  pRes

let pdbMethodGetToken (x:pdb_method) : int32 = 
  let mutable pRes =  0 in
  check "pdbMethodGetToken" (IlsuppSupport.pdbMethodGetToken(x.pMethod,&pRes));
  pRes
let pdbMethodGetRootScope (x:pdb_method) : pdb_method_scope = 
  let mutable pRes = (new IntPtr(0)) in
  check "pdbMethodGetRootScope" (IlsuppSupport.pdbMethodGetRootScope(x.pMethod,&pRes));
  { pScope= pRes }

let pdbMethodGetSequencePoints (x:pdb_method) : pdb_sequence_point array =
  let mutable pSize = 0 in 
  check "pdbMethodGetSequencePoints" (IlsuppSupport.pdbMethodGetSequencePoints(x.pMethod,&pSize,null,null,null,null,null,null));
  let offsets = CompatArray.zero_create pSize in
  let docs = CompatArray.zero_create pSize in
  let lines = CompatArray.zero_create pSize in
  let cols = CompatArray.zero_create pSize in
  let endLines = CompatArray.zero_create pSize in
  let endColumns = CompatArray.zero_create pSize in
  check "pdbMethodGetSequencePoints" (IlsuppSupport.pdbMethodGetSequencePoints(x.pMethod, &pSize, offsets, docs, lines, cols, endLines, endColumns));
  Array.init pSize (fun i -> 
      { pdbSeqPointOffset = CompatArray.get offsets i;
        pdbSeqPointDocument = { pDocument = CompatArray.get docs i };
        pdbSeqPointLine = CompatArray.get  lines i;
        pdbSeqPointColumn = CompatArray.get  cols i;
        pdbSeqPointEndLine = CompatArray.get endLines i;
        pdbSeqPointEndColumn = CompatArray.get endColumns i; }) 

let pdbScopeGetChildren (x:pdb_method_scope) : pdb_method_scope array = 
  let mutable pSize = 0 in 
  check "pdbScopeGetChildren" (IlsuppSupport.pdbScopeGetChildren(x.pScope,null,&pSize));
  let arr = CompatArray.zero_create pSize in
  check "pdbScopeGetChildren" (IlsuppSupport.pdbScopeGetChildren(x.pScope, arr, &pSize));
  Array.map (fun i -> { pScope=i }) (CompatArray.to_array(arr))

let pdbScopeGetOffsets (x:pdb_method_scope) : int * int =  
  let mutable pRes1 = 0 in
  let mutable pRes2 = 0 in
  check "pdbScopeGetOffsets" (IlsuppSupport.pdbScopeGetOffsets(x.pScope,&pRes1, &pRes2));
  (pRes1, pRes2)

let pdbScopeGetLocals (x:pdb_method_scope) : pdb_variable array = 
  let mutable pSize = 0 in 
  check "pdbScopeGetLocals" (IlsuppSupport.pdbScopeGetLocals(x.pScope,null,&pSize));
  let arr = CompatArray.zero_create pSize in
  check "pdbScopeGetLocals" (IlsuppSupport.pdbScopeGetLocals(x.pScope, arr, &pSize));
  Array.map (fun i -> { pVariable=i }) (CompatArray.to_array(arr))

let pdbVariableGetName (x:pdb_variable) : string = 
  let mutable pSize = 0 in 
  check "pdbVariableGetName" (IlsuppSupport.pdbVariableGetName(x.pVariable,null,&pSize));
  let buff = new StringBuilder(pSize) in
  check "action" (IlsuppSupport.pdbVariableGetName(x.pVariable,buff,&pSize));
  buff.ToString()

let pdbVariableGetSignature (x:pdb_variable) :  bytes = 
  let mutable pSize = 0 in 
  check "pdbVariableGetSignature" (IlsuppSupport.pdbVariableGetSignature(x.pVariable,null,&pSize));
  let buff = Bytes.zero_create pSize in
  check "pdbVariableGetSignature" (IlsuppSupport.pdbVariableGetSignature(x.pVariable,buff,&pSize));
  buff

let pdbVariableGetAddressAttributes (x:pdb_variable) :  (int32 * int32) = 
  let mutable pRes1 = 0 in
  let mutable pRes2 = 0 in
  check "pdbVariableGetAddressAttributes" (IlsuppSupport.pdbVariableGetAddressAttributes(x.pVariable,&pRes1, &pRes2));
  (pRes1, pRes2)

