(* (c) Microsoft Corporation 2005-2007.  *)

(*F# 
module Microsoft.Research.AbstractIL.BinaryReader 

open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 

module Ilsupp = Microsoft.Research.AbstractIL.Internal.Support 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilbinary = Microsoft.Research.AbstractIL.Internal.BinaryConstants 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Il = Microsoft.Research.AbstractIL.IL  
module Illib = Microsoft.Research.AbstractIL.Internal.Library 
F#*)  

open Illib
open Ildiag
open Il
open Nums
open Ilbinary
open Ilsupp

type readerOptions =
 { pdbPath: string option;
   manager: Il.manager option;
   mscorlib: mscorlib_refs;
   optimizeForMemory: bool }

(*---------------------------------------------------------------------
 * The big binary reader
 *
 * Flags and a sink for profile reporting
 *---------------------------------------------------------------------*)

let report_ref = ref (fun oc -> ()) 
let add_report f = let old = !report_ref in report_ref := (fun oc -> old oc; f oc) 
let report oc = !report_ref oc

let logging = false
let checking = false  let _ = if checking then dprint_endline "warning : Ilread.checking is on"

let empty_custom_attrs = mk_custom_attrs []

(*---------------------------------------------------------------------
 * Read file from memory mapped files
 *---------------------------------------------------------------------*)

(*F#
#if CLI_AT_LEAST_2_0  // GENERICS <-> WIN32
module MMap = begin

open System
open System.IO
open System.Runtime.InteropServices
open Printf

type HANDLE = nativeint
type ADDR   = nativeint

[<DllImport("kernel32", SetLastError=true)>]
extern bool CloseHandle (HANDLE handler)

[<DllImport("kernel32", SetLastError=true, CharSet=CharSet.Auto)>]
extern HANDLE CreateFile (string lpFileName, 
                          int dwDesiredAccess, 
                          int dwShareMode,
                          HANDLE lpSecurityAttributes, 
                          int dwCreationDisposition,
                          int dwFlagsAndAttributes, 
                          HANDLE hTemplateFile)
         
[<DllImport("kernel32", SetLastError=true, CharSet=CharSet.Auto)>]
extern HANDLE CreateFileMapping (HANDLE hFile, 
                                 HANDLE lpAttributes, 
                                 int flProtect, 
                                 int dwMaximumSizeLow, 
                                 int dwMaximumSizeHigh,
                                 string lpName) 

[<DllImport("kernel32", SetLastError=true)>]
extern ADDR MapViewOfFile (HANDLE hFileMappingObject, 
                           int dwDesiredAccess, 
                           int dwFileOffsetHigh,
                           int dwFileOffsetLow, 
                           int dwNumBytesToMap)

[<DllImport("kernel32", SetLastError=true, CharSet=CharSet.Auto)>]
extern HANDLE OpenFileMapping (int dwDesiredAccess, 
                               bool bInheritHandle, 
                               string lpName)

[<DllImport("kernel32", SetLastError=true)>]
extern bool UnmapViewOfFile (ADDR lpBaseAddress)

let INVALID_HANDLE = new IntPtr(-1)
let MAP_READ    = 0x0004
let GENERIC_READ = 0x80000000
let NULL_HANDLE = IntPtr.Zero
let FILE_SHARE_NONE = 0x0000
let FILE_SHARE_READ = 0x0001
let FILE_SHARE_WRITE = 0x0002
let FILE_SHARE_READ_WRITE = 0x0003
let CREATE_ALWAYS  = 0x0002
let OPEN_EXISTING   = 0x0003
let OPEN_ALWAYS  = 0x0004

type mmap = { hMap: HANDLE; start:nativeint }

let create fileName  =
  //printf "fileName = %s\n" fileName; 
  let hFile = CreateFile (fileName, GENERIC_READ, FILE_SHARE_READ_WRITE, IntPtr.Zero, OPEN_EXISTING, 0, IntPtr.Zero  ) in 
  //printf "hFile = %Lx\n" (hFile.ToInt64()); 
  if ( hFile.Equals(INVALID_HANDLE) ) then
      failwithf "CreateFile(0x%08x)" ( Marshal.GetHRForLastWin32Error() );
  let protection = 0x00000002 (* ReadOnly *) in 
  //printf "OK! hFile = %Lx\n" (hFile.ToInt64()); 
  let hMap = CreateFileMapping (hFile, IntPtr.Zero, protection, 0,0, null ) in
  ignore(CloseHandle(hFile));
  if hMap.Equals(NULL_HANDLE) then
      failwithf "CreateFileMapping(0x%08x)" ( Marshal.GetHRForLastWin32Error() );

  let start = MapViewOfFile (hMap, MAP_READ,0,0,0) in

  if ( start.Equals(IntPtr.Zero) ) then
     failwithf "MapViewOfFile(0x%08x)" ( Marshal.GetHRForLastWin32Error() );
  { hMap = hMap; start = start }

let addr m i = 
  new IntPtr(m.start.ToInt64()+Int64.of_int i)

let read_byte m i = 
   (# "ldind.u1" (addr m i) : int #)

let read_bytes m i len = 
  let res = Bytes.zero_create len in 
  Marshal.Copy(addr m i, res, 0,len);
  res
  
let read_i32 m i = 
  (# "ldind.i4" (new IntPtr(m.start.ToInt64()+Int64.of_int i)) : int32 #)

let close m = 
   ignore(UnmapViewOfFile ( m.start ));
   ignore(CloseHandle ( m.hMap ));;

let rec count_utf8_string m i n = 
  let c = read_byte m (i+n) in 
  if c = 0 then n 
  else count_utf8_string m i (n+1)

let read_utf8_bytes m i = 
    let n = count_utf8_string m i 0 in 
    read_bytes m i n

end

module MMapChannel = begin

type t = 
  { mutable mmPos: int;
    mmMap: MMap.mmap }

let open_in f = 
  let mmap = MMap.create f in 
  { mmPos = 0; mmMap = mmap }

let input_byte mc = 
  let res = MMap.read_byte mc.mmMap mc.mmPos in 
  mc.mmPos <- mc.mmPos + 1;
  res

let input_i32 mc = 
  let res = MMap.read_i32 mc.mmMap mc.mmPos in 
  mc.mmPos <- mc.mmPos + 4;
  res

let input_bytes mc len = 
  let res = MMap.read_bytes mc.mmMap mc.mmPos len in 
  mc.mmPos <- mc.mmPos + len;
  res

let input_utf8_bytes mc = 
  let res = MMap.read_utf8_bytes mc.mmMap mc.mmPos in 
  mc.mmPos <- mc.mmPos + Bytes.length res;
  res

let seek mc addr = mc.mmPos <- addr
let close mc = MMap.close mc.mmMap
end

#endif  // GENERICS <-> WIN32
F#*)

(*---------------------------------------------------------------------
 * Read file into cached memory blocks to avoid taking any kind of a 
 * lock on the file, and avoid needing any cleanup of files.
 *---------------------------------------------------------------------*)

module MemChannel = struct

type mem_in_channel = 
  { mutable mcPos: int;
    mcBlocks: Bytes.bytes array }

let mcBlockSize = 4000 

let open_in f = 
  let is = open_in_bin f in 
  let len = in_channel_length is in 
  let nblocks = (len-1) / mcBlockSize + 1 in
  let mcBlocks = Array.create nblocks (Bytes.zero_create 0) in 
  for i = 0 to nblocks - 2 do
    mcBlocks.(i) <- Bytes.really_input is mcBlockSize;
  done;
  mcBlocks.(nblocks - 1) <- Bytes.really_input is (len - (nblocks-1)*mcBlockSize);
  close_in is;
  { mcPos = 0; mcBlocks = mcBlocks }


let input_byte mc = 
  let p = mc.mcPos in 
  let res = Bytes.get (mc.mcBlocks.(p / mcBlockSize)) (p mod mcBlockSize) in 
  mc.mcPos <- p + 1;
  res

let input_bytes mc len = 
  let res = Bytes.zero_create len in 
  for i = 0 to len - 1 do 
    Bytes.set res i (input_byte mc)
  done;
  res


let seek mc addr = 
  mc.mcPos <- addr

end

(*---------------------------------------------------------------------
 * Read file from cached memory blocks or via 'seek'
 *---------------------------------------------------------------------*)

type input = 
  | Chan of string * in_channel option ref
  | Mem of MemChannel.mem_in_channel
(*F#
#if CLI_AT_LEAST_2_0 // GENERICS <-> WIN32
  | MMap of MMapChannel.t 
#endif
F#*)

let input_byte is = 
  match is with 
  | Chan (_,{contents=Some ic}) -> Pervasives.input_byte ic
  | Chan (s,_) -> failwith ("input_byte: input channel "^s^" was closed" )
  | Mem mc -> MemChannel.input_byte mc
(*F#
#if CLI_AT_LEAST_2_0 // GENERICS <-> WIN32
   | MMap mc -> MMapChannel.input_byte mc 
#endif
F#*)

let seek is addr = 
  match is with 
  | Chan (_,{contents=Some ic}) -> Pervasives.seek_in ic (Int32.to_int addr)
  | Chan (s,_) -> failwith ("seek: input channel "^s^" was closed" )
  | Mem mc -> MemChannel.seek mc (Int32.to_int addr)
(*F#
#if CLI_AT_LEAST_2_0 // GENERICS <-> WIN32
  | MMap mc -> MMapChannel.seek mc (Int32.to_int addr) 
#endif 
F#*)

let read_bytes is len = 
  match is with 
  | Chan (_,{contents=Some ic}) -> Bytes.really_input ic (Int32.to_int len)
  | Chan (s,_) -> failwith ("read_bytes: input channel "^s^" was closed" )
  | Mem mc -> MemChannel.input_bytes mc (Int32.to_int len)
(*F#
#if CLI_AT_LEAST_2_0 // GENERICS <-> WIN32
  | MMap mc -> MMapChannel.input_bytes mc (Int32.to_int len) 
#endif
F#*)


let read_i64 is = 
  let b0 = input_byte is in
  let b1 = input_byte is in
  let b2 = input_byte is in
  let b3 = input_byte is in
  let b4 = input_byte is in
  let b5 = input_byte is in
  let b6 = input_byte is in
  let b7 = input_byte is in
  !!!!b0 |||| (!!!!b1 <<<< 8) |||| (!!!!b2 <<<< 16) |||| (!!!!b3 <<<< 24) ||||
  (!!!!b4 <<<< 32) |||| (!!!!b5 <<<< 40) |||| (!!!!b6 <<<< 48) |||| (!!!!b7 <<<< 56)

let seek_read_i64 is addr = 
  seek is addr;
  read_i64 is

let read_i32 is = 
(*F#
  match is with 
 | Chan (_,{contents=Some ic}) -> Pervasives.input_binary_int ic
 | Chan (s,_) -> failwith ("read_bytes: input channel "^s^" was closed" )
#if CLI_AT_LEAST_2_0 // GENERICS <-> WIN32
 | MMap mm -> MMapChannel.input_i32 mm 
#endif
 | Mem _ ->
 F#*) 
     let b0 = input_byte is in
     let b1 = input_byte is in
     let b2 = input_byte is in
     let b3 = input_byte is in
     !!!b0 ||| (!!!b1 <<< 8) ||| (!!!b2 <<< 16) ||| (!!!b3 <<< 24)

(*---------------------------------------------------------------------
 * Derived reading
 *---------------------------------------------------------------------*)

let seek_read_i32 is addr = 
  seek is addr;
  read_i32 is

let read_u16_as_i32 is = 
  let b0 = input_byte is in
  let b1 = input_byte is in
  !!!b0 ||| (!!!b1 <<< 8) 
    
let seek_read_u16_as_i32 is addr = 
  seek is addr;
  read_u16_as_i32 is
    
let read_u16 is = i32_to_u16 (read_u16_as_i32 is)
let seek_read_u16 is addr =  seek is addr; read_u16 is
    
let read_u8_as_i32 is = 
  let b0 = input_byte is in
  !!!b0
let seek_read_u8_as_i32 is addr = 
  seek is addr;
  read_u8_as_i32 is 

let read_u8_as_u16 is = i32_to_u16 (read_u8_as_i32 is)
let seek_read_u8_as_u16 is addr =  seek is addr; read_u8_as_u16 is
    
let read_i8 is = 
  let b0 = input_byte is in
  u8_to_i8 (int_to_u8 b0)
let seek_read_i8 is addr = 
  seek is addr;
  read_i8 is 

let read_i8_as_i32 is = i8_to_i32 (read_i8 is)
let seek_read_i8_as_i32 is addr =  seek is addr; read_i8_as_i32 is
    

let read_ieee32 is = bits_to_ieee32 (read_i32 is)
let seek_read_ieee32 is addr =  seek is addr; read_ieee32 is
let read_ieee64 is = bits_to_ieee64 (read_i64 is)
let seek_read_ieee64 is addr =  seek is addr; read_ieee64 is
    
let read_z_unsigned_int32 is = 
  let b0 = read_u8_as_i32 is in 
  if b0 <= 0x7Fl then b0 
  else if b0 <= !!!0xbf then 
    let b0 = b0 &&& !!!0x7f in 
    let b1 = read_u8_as_i32 is in 
    (b0 <<< 8) ||| b1
  else 
    let b0 = b0 &&& !!!0x3f in 
    let b1 = read_u8_as_i32 is in 
    let b2 = read_u8_as_i32 is in 
    let b3 = read_u8_as_i32 is in 
    (b0 <<< 24) ||| (b1 <<< 16) ||| (b2 <<< 8) ||| b3

let seek_read_bytes is addr len = 
  seek is addr;
  read_bytes is len
    
let rec count_utf8_string is n = 
  let c = input_byte is in 
  if c = 0 then n 
  else count_utf8_string is (n+1)

let seek_read_utf8_string is addr = 
    seek is addr;
    let res = 
(*F#
#if CLI_AT_LEAST_2_0 // GENERICS <-> WIN32
      match is with 
      | MMap mc -> 
        (* optimized implementation *) 
        MMapChannel.input_utf8_bytes mc
      | _ -> 
#endif
F#*)
        let n = count_utf8_string is 0 in 
        seek_read_bytes is addr (Int32.of_int n) in 
    Bytes.utf8_bytes_as_string res

let read_blob is = 
  let len = read_z_unsigned_int32 is in 
  read_bytes is len
    
let seek_read_blob is addr = 
  seek is addr;
  read_blob is
    
let read_user_string is = 
  let len = read_z_unsigned_int32 is in 
  read_bytes is (len --- !!!1)
    
let seek_read_user_string is addr = 
  seek is addr;
  read_user_string is
    
let read_guid is = 
  let len = !!!0x10 in 
  read_bytes is len
    
let seek_read_guid is addr = 
  seek is addr;
  read_guid is
    
(*---------------------------------------------------------------------
 * Utilities.  
 *---------------------------------------------------------------------*)

let align alignment n = ((n +++ alignment --- !!!0x1) /./ alignment) *** alignment

let uncoded_token tab idx = ((!!!(tag_of_table tab) <<< 24) ||| !!!idx)

let i32_to_uncoded_token tok  = 
  let idx = tok &&& !!!0xffffff in 
  let tab = tok lsr 24 in 
  (Table (Int32.to_int tab), Int32.to_int idx)

let read_uncoded_token is  = i32_to_uncoded_token (read_i32 is)

let uncoded_token_to_tdor (tab,tok) = 
  let tag =
    if tab = tab_TypeDef then tdor_TypeDef 
    else if tab = tab_TypeRef then tdor_TypeRef
    else if tab = tab_TypeSpec then tdor_TypeSpec
    else failwith "bad table in uncoded_token_to_tdor"  in 
  (tag,tok)

let uncoded_token_to_mdor (tab,tok) = 
  let tag =
    if tab = tab_Method then mdor_MethodDef 
    else if tab = tab_MemberRef then mdor_MemberRef
    else failwith "bad table in uncoded_token_to_mdor"  in 
  (tag,tok)

let seek_read_uncoded_token is addr  = 
  seek is addr;
  read_uncoded_token is

let read_z_tagged_idx f nbits big is = 
  let tok = if big then read_i32 is else read_u16_as_i32 is in
  let tagmask = 
    if nbits = 1 then 1 
    else if nbits = 2 then 3 
    else if nbits = 3 then 7 
    else if nbits = 4 then 15 
       else if nbits = 5 then 31 
       else failwith "too many nbits" in 
  let tag = tok &&& !!!tagmask in 
  let idx = tok lsr nbits in
  (f tag,Int32.to_int idx) 
       
(*---------------------------------------------------------------------
 * Primitives to help read signatures.  These do not use the file cursor, but
 * pass ar
 *---------------------------------------------------------------------*)


let sigptr_get_byte bytes sigptr = 
  if checking && sigptr >= Bytes.length bytes then (dprint_endline "read past end of sig. "; 0x0, sigptr)
  else Bytes.get bytes sigptr, sigptr + 1

let sigptr_get_bool bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  (b0 = 0x01) ,sigptr

let sigptr_get_u8 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  int_to_u8 b0,sigptr

let sigptr_get_i8 bytes sigptr = 
  let i,sigptr = sigptr_get_u8 bytes sigptr in 
  u8_to_i8 i,sigptr

let sigptr_get_u16 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  let b1,sigptr = sigptr_get_byte bytes sigptr in 
  i32_to_u16 (!!!b0 ||| (!!!b1 <<< 8)),sigptr

let sigptr_get_i16 bytes sigptr = 
  let u,sigptr = sigptr_get_u16 bytes sigptr in 
  u16_to_i16 u,sigptr

let sigptr_get_i32 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  let b1,sigptr = sigptr_get_byte bytes sigptr in 
  let b2,sigptr = sigptr_get_byte bytes sigptr in 
  let b3,sigptr = sigptr_get_byte bytes sigptr in 
  !!!b0 ||| (!!!b1 <<< 8) ||| (!!!b2 <<< 16) ||| (!!!b3 <<< 24),sigptr

let sigptr_get_u32 bytes sigptr = 
  let u,sigptr = sigptr_get_i32 bytes sigptr in 
  i32_to_u32 u,sigptr

let sigptr_get_i64 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  let b1,sigptr = sigptr_get_byte bytes sigptr in 
  let b2,sigptr = sigptr_get_byte bytes sigptr in 
  let b3,sigptr = sigptr_get_byte bytes sigptr in 
  let b4,sigptr = sigptr_get_byte bytes sigptr in 
  let b5,sigptr = sigptr_get_byte bytes sigptr in 
  let b6,sigptr = sigptr_get_byte bytes sigptr in 
  let b7,sigptr = sigptr_get_byte bytes sigptr in 
  !!!!b0 |||| (!!!!b1 <<<< 8) |||| (!!!!b2 <<<< 16) |||| (!!!!b3 <<<< 24) ||||
  (!!!!b4 <<<< 32) |||| (!!!!b5 <<<< 40) |||| (!!!!b6 <<<< 48) |||| (!!!!b7 <<<< 56),
  sigptr

let sigptr_get_u64 bytes sigptr = 
  let u,sigptr = sigptr_get_i64 bytes sigptr in 
  i64_to_u64 u,sigptr

let sigptr_get_ieee32 bytes sigptr = 
  let u,sigptr = sigptr_get_i32 bytes sigptr in 
  bits_to_ieee32 u,sigptr

let sigptr_get_ieee64 bytes sigptr = 
  let u,sigptr = sigptr_get_i64 bytes sigptr in 
  bits_to_ieee64 u,sigptr

let sigptr_get_z_i32 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  if b0 <= 0x7F then !!!b0, sigptr
  else if b0 <= 0xbf then 
    let b0 = !!!b0 &&& !!!0x7f in 
    let b1,sigptr = sigptr_get_byte bytes sigptr in 
       (b0 <<< 8) ||| !!!b1, sigptr
  else 
    let b0 = !!!b0 &&& !!!0x3f in 
    let b1,sigptr = sigptr_get_byte bytes sigptr in 
    let b2,sigptr = sigptr_get_byte bytes sigptr in 
    let b3,sigptr = sigptr_get_byte bytes sigptr in 
    (b0 <<< 24) ||| (!!!b1 <<< 16) ||| (!!!b2 <<< 8) ||| !!!b3, sigptr
         

let rec sigptr_foldi_acc f n (bytes:Bytes.bytes) (sigptr:int) i acc = 
  if i < n then 
    let x,sp = f bytes sigptr in 
    sigptr_foldi_acc f n bytes sp (i+1) (x::acc)
  else 
    List.rev acc, sigptr

let sigptr_foldi f n (bytes:Bytes.bytes) (sigptr:int) = 
  sigptr_foldi_acc f n bytes sigptr 0 []


(*let sigptr_foldi f n (bytes:Bytes.bytes) (sigptr:int) = 
  let res = ref [] in 
  let sigptr2 = ref sigptr in 
  for i = 0 to n - 1 do
    let x,sp = f bytes !sigptr2 in 
    res := x :: !res;
    sigptr2 := sp
  done;
  List.rev !res, !sigptr2
*)

let sigptr_get_bytes n bytes sigptr = 
  if checking && sigptr + n >= Bytes.length bytes then begin
    dprint_endline "read past end of sig. in sigptr_get_string"; 
    Bytes.zero_create 0, sigptr
  end
  else 
    let res = Bytes.zero_create n in 
    for i = 0 to (n - 1) do 
      Bytes.set res i (Bytes.get bytes (sigptr + i))
    done;
    res, sigptr + n

let sigptr_get_string n bytes sigptr = 
  let bytearray,sigptr = sigptr_get_bytes n bytes sigptr in 
  Bytes.utf8_bytes_as_string bytearray,sigptr
   

(* -------------------------------------------------------------------- 
 * Now the tables of instructions
 * -------------------------------------------------------------------- *)

(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type prefixes = 
 { mutable al:alignment; 
   mutable tl:tailness;
   mutable vol:volatility;
   mutable ro:readonlyByref;
   mutable constrained: typ option}
 
let no_prefixes mk prefixes = 
  if prefixes.al <> Aligned then failwith "an unaligned prefix is not allowed here";
  if prefixes.vol <> Nonvolatile then failwith "a volatile prefix is not allowed here";
  if prefixes.tl <> Normalcall then failwith "a tailcall prefix is not allowed here";
  if prefixes.ro <> NormalAddress then failwith "a readonly prefix is not allowed here";
  if prefixes.constrained <> None then failwith "a constrained prefix is not allowed here";
  mk 

let volatile_unaligned_prefix mk prefixes = 
  if prefixes.tl <> Normalcall then failwith "a tailcall prefix is not allowed here";
  if prefixes.constrained <> None then failwith "a constrained prefix is not allowed here";
  if prefixes.ro <> NormalAddress then failwith "a readonly prefix is not allowed here";
  mk (prefixes.al,prefixes.vol) 

let volatile_prefix mk prefixes = 
  if prefixes.al <> Aligned then failwith "an unaligned prefix is not allowed here";
  if prefixes.tl <> Normalcall then failwith "a tailcall prefix is not allowed here";
  if prefixes.constrained <> None then failwith "a constrained prefix is not allowed here";
  if prefixes.ro <> NormalAddress then failwith "a readonly prefix is not allowed here";
  mk prefixes.vol

let tail_prefix mk prefixes = 
  if prefixes.al <> Aligned then failwith "an unaligned prefix is not allowed here";
  if prefixes.vol <> Nonvolatile then failwith "a volatile prefix is not allowed here";
  if prefixes.constrained <> None then failwith "a constrained prefix is not allowed here";
  if prefixes.ro <> NormalAddress then failwith "a readonly prefix is not allowed here";
  mk prefixes.tl 

let constraint_tail_prefix mk prefixes = 
  if prefixes.al <> Aligned then failwith "an unaligned prefix is not allowed here";
  if prefixes.vol <> Nonvolatile then failwith "a volatile prefix is not allowed here";
  if prefixes.ro <> NormalAddress then failwith "a readonly prefix is not allowed here";
  mk (prefixes.constrained,prefixes.tl )

let readonly_prefix mk prefixes = 
  if prefixes.al <> Aligned then failwith "an unaligned prefix is not allowed here";
  if prefixes.vol <> Nonvolatile then failwith "a volatile prefix is not allowed here";
  if prefixes.tl <> Normalcall then failwith "a tailcall prefix is not allowed here";
  if prefixes.constrained <> None then failwith "a constrained prefix is not allowed here";
  mk prefixes.ro


(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type instr_decoder = 
  | I_u16_u8_instr of (prefixes -> u16 -> instr)
  | I_u16_u16_instr of (prefixes -> u16 -> instr)
  | I_none_instr of (prefixes -> instr)
  | I_i64_instr of (prefixes -> i64 -> instr)
  | I_i32_i32_instr of (prefixes -> i32 -> instr)
  | I_i32_i8_instr of (prefixes -> i32 -> instr)
  | I_r4_instr of (prefixes -> ieee32 -> instr)
  | I_r8_instr of (prefixes -> ieee64 -> instr)
  | I_field_instr of (prefixes -> field_spec -> instr)
  | I_method_instr of (prefixes -> method_spec * varargs -> instr)
  | I_unconditional_i32_instr of (prefixes -> code_label  -> instr)
  | I_unconditional_i8_instr of (prefixes -> code_label  -> instr)
  | I_conditional_i32_instr of (prefixes -> code_label * code_label -> instr)
  | I_conditional_i8_instr of (prefixes -> code_label * code_label -> instr)
  | I_string_instr of (prefixes -> Bytes.bytes -> instr)
  | I_switch_instr of (prefixes -> code_label list * code_label -> instr)
  | I_tok_instr of (prefixes -> ldtoken_info -> instr)
  | I_sig_instr of (prefixes -> callsig * varargs -> instr)
  | I_type_instr of (prefixes -> typ -> instr)
  | I_invalid_instr

let mk_stind dt = volatile_unaligned_prefix (fun (x,y) -> I_stind(x,y,dt))
let mk_ldind dt = volatile_unaligned_prefix (fun (x,y) -> I_ldind(x,y,dt))

let instrs () = 
 [ i_ldarg_s, I_u16_u8_instr (no_prefixes (fun x -> I_ldarg x));
   i_starg_s, I_u16_u8_instr (no_prefixes (fun x -> I_starg x));
   i_ldarga_s, I_u16_u8_instr (no_prefixes (fun x -> I_ldarga x));
   i_stloc_s, I_u16_u8_instr (no_prefixes (fun x -> I_stloc x));
   i_ldloc_s, I_u16_u8_instr (no_prefixes (fun x -> I_ldloc x));
   i_ldloca_s, I_u16_u8_instr (no_prefixes (fun x -> I_ldloca x));
   i_ldarg, I_u16_u16_instr (no_prefixes (fun x -> I_ldarg x));
   i_starg, I_u16_u16_instr (no_prefixes (fun x -> I_starg x));
   i_ldarga, I_u16_u16_instr (no_prefixes (fun x -> I_ldarga x));
   i_stloc, I_u16_u16_instr (no_prefixes (fun x -> I_stloc x));
   i_ldloc, I_u16_u16_instr (no_prefixes (fun x -> I_ldloc x));
   i_ldloca, I_u16_u16_instr (no_prefixes (fun x -> I_ldloca x)); 
   i_stind_i, I_none_instr (mk_stind            DT_I);
   i_stind_i1, I_none_instr (mk_stind           DT_I1);
   i_stind_i2, I_none_instr (mk_stind           DT_I2);
   i_stind_i4, I_none_instr (mk_stind           DT_I4);
   i_stind_i8, I_none_instr (mk_stind           DT_I8);
   i_stind_r4, I_none_instr (mk_stind           DT_R4);
   i_stind_r8, I_none_instr (mk_stind           DT_R8);
   i_stind_ref, I_none_instr (mk_stind          DT_REF);
   i_ldind_i, I_none_instr (mk_ldind            DT_I);
   i_ldind_i1, I_none_instr (mk_ldind           DT_I1);
   i_ldind_i2, I_none_instr (mk_ldind           DT_I2);
   i_ldind_i4, I_none_instr (mk_ldind           DT_I4);
   i_ldind_i8, I_none_instr (mk_ldind           DT_I8);
   i_ldind_u1, I_none_instr (mk_ldind           DT_U1);
   i_ldind_u2, I_none_instr (mk_ldind           DT_U2);
   i_ldind_u4, I_none_instr (mk_ldind           DT_U4);
   i_ldind_r4, I_none_instr (mk_ldind           DT_R4);
   i_ldind_r8, I_none_instr (mk_ldind           DT_R8);
   i_ldind_ref, I_none_instr (mk_ldind          DT_REF);
   i_cpblk, I_none_instr (volatile_unaligned_prefix (fun (x,y) -> I_cpblk(x,y)));
   i_initblk, I_none_instr (volatile_unaligned_prefix (fun (x,y) -> I_initblk(x,y))); 
   i_ldc_i8, I_i64_instr (no_prefixes (fun x ->I_arith (AI_ldc (DT_I8, NUM_I8 x)))); 
   i_ldc_i4, I_i32_i32_instr (no_prefixes (fun x -> ((mk_ldc_i32 x))));
   i_ldc_i4_s, I_i32_i8_instr (no_prefixes (fun x -> ((mk_ldc_i32 x))));
   i_ldc_r4, I_r4_instr (no_prefixes (fun x -> I_arith (AI_ldc (DT_R4, NUM_R4 x)))); 
   i_ldc_r8, I_r8_instr (no_prefixes (fun x -> I_arith (AI_ldc (DT_R8, NUM_R8 x))));
   i_ldfld, I_field_instr (volatile_unaligned_prefix(fun (x,y) fspec -> I_ldfld(x,y,fspec)));
   i_stfld, I_field_instr (volatile_unaligned_prefix(fun  (x,y) fspec -> I_stfld(x,y,fspec)));
   i_ldsfld, I_field_instr (volatile_prefix (fun x fspec -> I_ldsfld (x, fspec)));
   i_stsfld, I_field_instr (volatile_prefix (fun x fspec -> I_stsfld (x, fspec)));
   i_ldflda, I_field_instr (no_prefixes (fun fspec -> I_ldflda fspec));
   i_ldsflda, I_field_instr (no_prefixes (fun fspec -> I_ldsflda fspec)); 
   i_call, I_method_instr (tail_prefix (fun tl (mspec,y) -> I_call (tl,mspec,y)));
   i_ldftn, I_method_instr (no_prefixes (fun (mspec,y) -> I_ldftn mspec));
   i_ldvirtftn, I_method_instr (no_prefixes (fun (mspec,y) -> I_ldvirtftn mspec));
   i_newobj, I_method_instr (no_prefixes (fun (mspec,y) -> I_newobj (mspec,y)));
   i_callvirt, I_method_instr (constraint_tail_prefix (fun (c,tl) (mspec,y) -> match c with Some ty -> I_callconstraint(tl,ty,mspec,y) | None -> I_callvirt (tl,mspec,y))); 
   i_leave_s, I_unconditional_i8_instr (no_prefixes (fun x -> I_leave x));
   i_br_s, I_unconditional_i8_instr (no_prefixes (fun x -> I_br x)); 
   i_leave, I_unconditional_i32_instr (no_prefixes (fun x -> I_leave x));
   i_br, I_unconditional_i32_instr (no_prefixes (fun x -> I_br x)); 
   i_brtrue_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_brtrue,x,y)));
   i_brfalse_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_brfalse,x,y)));
   i_beq_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_beq,x,y)));
   i_blt_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_blt,x,y)));
   i_blt_un_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_blt_un,x,y)));
   i_ble_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_ble,x,y)));
   i_ble_un_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_ble_un,x,y)));
   i_bgt_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bgt,x,y)));
   i_bgt_un_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bgt_un,x,y)));
   i_bge_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bge,x,y)));
   i_bge_un_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bge_un,x,y)));
   i_bne_un_s, I_conditional_i8_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bne_un,x,y)));   
   i_brtrue, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_brtrue,x,y)));
   i_brfalse, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_brfalse,x,y)));
   i_beq, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_beq,x,y)));
   i_blt, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_blt,x,y)));
   i_blt_un, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_blt_un,x,y)));
   i_ble, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_ble,x,y)));
   i_ble_un, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_ble_un,x,y)));
   i_bgt, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bgt,x,y)));
   i_bgt_un, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bgt_un,x,y)));
   i_bge, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bge,x,y)));
   i_bge_un, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bge_un,x,y)));
   i_bne_un, I_conditional_i32_instr (no_prefixes (fun (x,y) -> I_brcmp (BI_bne_un,x,y))); 
   i_ldstr, I_string_instr (no_prefixes (fun x -> I_ldstr x)); 
   i_switch, I_switch_instr (no_prefixes (fun x -> I_switch x));
   i_ldtoken, I_tok_instr (no_prefixes (fun x -> I_ldtoken x));
   i_calli, I_sig_instr (tail_prefix (fun tl (x,y) -> I_calli (tl, x, y)));
   i_mkrefany, I_type_instr (no_prefixes (fun x -> I_mkrefany x));
   i_refanyval, I_type_instr (no_prefixes (fun x -> I_refanyval x));
   i_ldelema, I_type_instr (readonly_prefix (fun ro x -> I_ldelema (ro,sdshape,x)));
   i_ldelem_any, I_type_instr (no_prefixes (fun x -> I_ldelem_any (sdshape,x)));
   i_stelem_any, I_type_instr (no_prefixes (fun x -> I_stelem_any (sdshape,x)));
   i_newarr, I_type_instr (no_prefixes (fun x -> I_newarr (sdshape,x)));  
   i_castclass, I_type_instr (no_prefixes (fun x -> I_castclass x));
   i_isinst, I_type_instr (no_prefixes (fun x -> I_isinst x));
   i_unbox_any, I_type_instr (no_prefixes (fun x -> I_unbox_any x));
   i_cpobj, I_type_instr (no_prefixes (fun x -> I_cpobj x));
   i_initobj, I_type_instr (no_prefixes (fun x -> I_initobj x));
   i_ldobj, I_type_instr (volatile_unaligned_prefix (fun (x,y) z -> I_ldobj (x,y,z)));
   i_stobj, I_type_instr (volatile_unaligned_prefix (fun (x,y) z -> I_stobj (x,y,z)));
   i_sizeof, I_type_instr (no_prefixes (fun x -> I_sizeof x));
   i_box, I_type_instr (no_prefixes (fun x -> I_box x));
   i_unbox, I_type_instr (no_prefixes (fun x -> I_unbox x)); ] 

(* The tables are delayed to avoid building them unnecessarily at startup *)
(* Many applications of AbsIL (e.g. a compiler) don't need to read instructions. *)
let one_byte_instrs = ref None
let two_byte_instrs = ref None
let fill_instrs () = 
  let one_byte_tab = Array.create 256 I_invalid_instr in 
  let two_byte_tab = Array.create 256 I_invalid_instr in 
  let add_instr (i,f) =  
    if !!!i > !!!0xff then begin
      assert (!!!i lsr 8 = !!!0xfe); 
      let i = Int32.to_int (!!!i &&& !!!0xff) in
      if (two_byte_tab.(i) <> I_invalid_instr) then 
        dprint_endline ("warning: duplicate decode entries for "^string_of_int i);
      two_byte_tab.(i) <- f
    end else begin 
      if (one_byte_tab.(i) <> I_invalid_instr) then
        dprint_endline ("warning: duplicate decode entries for "^string_of_int i);
      one_byte_tab.(i) <- f 
    end in
  List.iter add_instr (instrs());
  List.iter (fun (x,mk) -> add_instr (x,I_none_instr (no_prefixes mk))) (noarg_instrs ());
  one_byte_instrs := Some one_byte_tab;
  two_byte_instrs := Some two_byte_tab

let rec get_one_byte_instr i = 
  match !one_byte_instrs with 
  | None -> fill_instrs(); get_one_byte_instr i
  | Some t -> t.(i)

let rec get_two_byte_instr i = 
  match !two_byte_instrs with 
  | None -> fill_instrs(); get_two_byte_instr i
  | Some t -> t.(i)
  
(*---------------------------------------------------------------------
 * 
 *---------------------------------------------------------------------*)

type chunk = { size: int32; addr: int32 }

let chunk sz next = ({addr=next; size=sz},next +++ sz) 
let nochunk next = ({addr= !!!0x0;size= !!!0x0; } ,next)

type row_element_kind = 
  | UShort 
  | ULong 
  | Byte 
  | Data 
  | GGuid 
  | Blob 
  | SString 
  | SimpleIndex of table
  | TypeDefOrRefOrSpec
  | TypeOrMethodDef
  | HasConstant 
  | HasCustomAttribute
  | HasFieldMarshal 
  | HasDeclSecurity 
  | MemberRefParent 
  | HasSemantics 
  | MethodDefOrRef
  | MemberForwarded
  | Implementation 
  | CustomAttributeType
  | ResolutionScope
type row_kind = RowKind of row_element_kind list

let kind_AssemblyRef = RowKind [ UShort; UShort; UShort; UShort; ULong; Blob; SString; SString; Blob; ]
let kind_ModuleRef =   RowKind [ SString ]
let kind_FileRef = RowKind [ ULong; SString; Blob ]
let kind_TypeRef = RowKind [ ResolutionScope; SString; SString ]
let kind_TypeSpec = RowKind [ Blob ]
let kind_TypeDef = RowKind [ ULong; SString; SString; TypeDefOrRefOrSpec; SimpleIndex tab_Field; SimpleIndex tab_Method ]
let kind_PropertyMap = RowKind [ SimpleIndex tab_TypeDef; SimpleIndex tab_Property ]
let kind_EventMap = RowKind [ SimpleIndex tab_TypeDef; SimpleIndex tab_Event ]
let kind_InterfaceImpl = RowKind [ SimpleIndex tab_TypeDef; TypeDefOrRefOrSpec ]
let kind_Nested = RowKind [ SimpleIndex tab_TypeDef; SimpleIndex tab_TypeDef ]
let kind_CustomAttribute = RowKind [HasCustomAttribute; CustomAttributeType; Blob ]
let kind_DeclSecurity = RowKind [ UShort; HasDeclSecurity; Blob ]
let kind_MemberRef = RowKind [MemberRefParent; SString; Blob ]
let kind_StandAloneSig = RowKind [Blob ]
let kind_FieldDef = RowKind [UShort; SString; Blob ]
let kind_FieldRVA = RowKind [Data; SimpleIndex tab_Field ]
let kind_FieldMarshal = RowKind [HasFieldMarshal; Blob ]
let kind_Constant = RowKind [ UShort;HasConstant; Blob ]
let kind_FieldLayout = RowKind [ULong; SimpleIndex tab_Field ]
let kind_Param = RowKind [ UShort; UShort; SString ]
let kind_MethodDef = RowKind [ULong;  UShort; UShort; SString; Blob; SimpleIndex tab_Param ]
let kind_MethodImpl = RowKind [SimpleIndex tab_TypeDef; MethodDefOrRef; MethodDefOrRef ]
let kind_ImplMap = RowKind [UShort; MemberForwarded; SString; SimpleIndex tab_ModuleRef ]
let kind_MethodSemantics = RowKind [UShort; SimpleIndex tab_Method; HasSemantics ]
let kind_Property =RowKind [ UShort; SString; Blob ]
let kind_Event =RowKind [ UShort; SString; TypeDefOrRefOrSpec ]
let kind_ManifestResource =RowKind [ ULong; ULong; SString; Implementation ]
let kind_ClassLayout = RowKind [ UShort; ULong; SimpleIndex tab_TypeDef ]
let kind_ExportedType = RowKind [  ULong; ULong; SString; SString; Implementation ]
let kind_Assembly = RowKind [  ULong; UShort; UShort; UShort; UShort; ULong; Blob; SString; SString ]
let kind_GenericParam_v1_1 = RowKind [  UShort; UShort; TypeOrMethodDef; SString; TypeDefOrRefOrSpec ]
let kind_GenericParam_v2_0 = RowKind [  UShort; UShort; TypeOrMethodDef; SString ]
let kind_MethodSpec = RowKind [ MethodDefOrRef; Blob ]
let kind_GenericParamConstraint = RowKind [  SimpleIndex tab_GenericParam; TypeDefOrRefOrSpec ]
let kind_Module = RowKind [  UShort; SString; GGuid; GGuid; GGuid ]
let kind_Illegal = RowKind []

(*---------------------------------------------------------------------
 * Used for binary searches of sorted tables.  Each function that reads
 * a table row returns a tuple that contains the elements of the row.
 * One of these elements may be a key for a sorted table.  These
 * keys can be compared using the functions below depending on the
 * kind of element in that column.
 *---------------------------------------------------------------------*)

let hc_compare (HasConstantTag t1, (idx1:int)) (HasConstantTag t2, idx2) = 
  if idx1 < idx2 then -1 else if idx1 > idx2 then 1 else compare t1 t2

let hs_compare (HasSemanticsTag t1, (idx1:int)) (HasSemanticsTag t2, idx2) = 
  if idx1 < idx2 then -1 else if idx1 > idx2 then 1 else compare t1 t2

let hca_compare (HasCustomAttributeTag t1, (idx1:int)) (HasCustomAttributeTag t2, idx2) = 
  if idx1 < idx2 then -1 else if idx1 > idx2 then 1 else compare t1 t2

let mf_compare (MemberForwardedTag t1, (idx1:int)) (MemberForwardedTag t2, idx2) = 
  if idx1 < idx2 then -1 else if idx1 > idx2 then 1 else compare t1 t2

let hds_compare (HasDeclSecurityTag t1, (idx1:int)) (HasDeclSecurityTag t2, idx2) = 
  if idx1 < idx2 then -1 else if idx1 > idx2 then 1 else compare t1 t2

let hfm_compare (HasFieldMarshalTag t1, (idx1:int) ) (HasFieldMarshalTag t2, idx2) = 
  if idx1 < idx2 then -1 else if idx1 > idx2 then 1 else compare t1 t2

let tomd_compare (TypeOrMethodDefTag t1, (idx1:int) ) (TypeOrMethodDefTag t2, idx2) = 
  if idx1 < idx2 then -1 else if idx1 > idx2 then 1 else compare t1 t2

let simpleindex_compare (idx1:int) (idx2:int) = 
  compare idx1 idx2

(*---------------------------------------------------------------------
 * The various keys for the various caches.  We avoid using polymorphic
 * types within keys becuase F# is not crash hot at hashing and compare 
 * them.  In other words, the types in the indexes below should not
 * use (real) tuples, lists, options etc.  We also add a couple of 
 * hashtables indexed by integers since these are also much faster
 * for F#.
 *---------------------------------------------------------------------*)

type typeDefAsTypIdx = TypeDefAsTypIdx of boxity * genactuals * int
type typeRefAsTypIdx = TypeRefAsTypIdx of boxity * genactuals * int
type blobAsMethodSigIdx = BlobAsMethodSigIdx of int * int32
type blobAsFieldSigIdx = BlobAsFieldSigIdx of int * int32
type blobAsPropSigIdx = BlobAsPropSigIdx of int * int32
type blobAsLocalSigIdx = BlobAsLocalSigIdx of int * int32
type memberRefAsMspecIdx =  MemberRefAsMspecIdx of int * int
type methodSpecAsMspecIdx =  MethodSpecAsMspecIdx of int * int
type memberRefAsFspecIdx = MemberRefAsFspecIdx of int * int
type customAttrIdx = CustomAttrIdx of customAttributeType_tag * int * int32
type securityDeclIdx   = SecurityDeclIdx of u16 * int32
type genericParsIdx = GenericParamsIdx of int * typeOrMethodDef_tag * int

(*---------------------------------------------------------------------
 * Polymorphic caches for row and heap readers
 *---------------------------------------------------------------------*)

(*IF-OCAML*) module I32hashtbl = Hashtbl (*ENDIF-OCAML*)
(*F# module I32hashtbl = struct 

#if CLI_AT_LEAST_2_0
       type 'a t = System.Collections.Generic.Dictionary<int32,'a> 
       let create (n:int) : 'a t = new System.Collections.Generic.Dictionary<_,_>(n)
       let add (t: 'a t) x y = t.Add(x,y)
       let find (t: 'a t) x = t.Item(x)
       let mem (t: 'a t) x = t.ContainsKey(x) 
#else
       type 'a t = Collections.HashTable<int32,'a> 
       let create n : 'a t = Collections.HashTable.Create(Collections.HashIdentity.Structural,n)
       let add (t: 'a t) x y = t.Add(x,y)
       let find (t: 'a t) x = t.Find(x)
       let mem (t: 'a t) x = t.Contains(x)
#endif
       
     end 
F#*)

(*IF-OCAML*) module Ihashtbl = Hashtbl (*ENDIF-OCAML*)
(*F# module Ihashtbl = I32hashtbl F#*)

let mk_cache_int32 lowMem inbase nm sz  =
  if lowMem then (fun f x -> f x) else
  let cache = I32hashtbl.create sz in 
  let count = ref 0 in 
  add_report (fun oc -> if !count <> 0 then output_string oc (inbase^string_of_int !count ^ " "^nm^" cache hits"^"\n"));
  fun f (idx:int32) ->
    if I32hashtbl.mem cache idx then (incr count; I32hashtbl.find cache idx)
    else let res = f idx in I32hashtbl.add cache idx res; res 
let mk_cache_int lowMem inbase nm sz  =
  if lowMem then (fun f x -> f x) else
  let cache = Ihashtbl.create sz in 
  let count = ref 0 in 
  add_report (fun oc -> if !count <> 0 then output_string oc (inbase^string_of_int !count ^ " "^nm^" cache hits"^"\n"));
  fun f (idx:int) ->
    if Ihashtbl.mem cache idx then (incr count; Ihashtbl.find cache idx)
    else let res = f idx in Ihashtbl.add cache idx res; res 
let mk_cache_gen lowMem inbase nm sz  =
  if lowMem then (fun f x -> f x) else
  let cache = Hashtbl.create sz in 
  let count = ref 0 in 
  add_report (fun oc -> if !count <> 0 then output_string oc (inbase^string_of_int !count ^ " "^nm^" cache hits"^"\n"));
  fun f (idx :'a) ->
    if Hashtbl.mem cache idx then (incr count; Hashtbl.find cache idx)
    else let res = f idx in Hashtbl.add cache idx res; res 

(*-----------------------------------------------------------------------
 * Polymorphic general helpers for searching for particular rows.
 * ----------------------------------------------------------------------*)

let seek_find_row nrows row_chooser =
  let res = ref [] in 
  let i = ref 1 in 
  while (!i <= nrows &&  not (row_chooser !i)) do incr i done;
  if !i > nrows then dprint_endline "warning: seek_find_row: row not found";
  !i  

let seek_read_indexed_rows (nrows,row_reader,keyfunc,keycomparef,binchop, row_converter) =
  if binchop then
         (* if logging then dprint_endline (infile ^ ": table "^string_of_int (tag_of_table tab)^" is sorted"); *)
         (* search for any row satisfying predicate *)
    let low = ref 0 in 
    let high = ref (nrows + 1) in
    begin 
      let fin = ref false in 
      while not !fin do 
             (* if logging then dprint_endline (infile ^ ": binary search of table "^string_of_int (tag_of_table tab)^", low = "^string_of_int !low^", high = "^string_of_int !high);*)
        if !high - !low <= 1  then fin := true 
        else begin
          let mid = (!low + !high) / 2 in 
          let midrow = row_reader mid in
          let c = keycomparef (keyfunc midrow) in 
          if c > 0 then 
            low := mid
          else if c < 0 then 
            high := mid 
          else 
            fin := true
        end
      done;
    end;
         (* if logging then dprint_endline (infile ^ ": finished binary search of table "^string_of_int (tag_of_table tab));*)
    let res = ref [] in 
    if !high - !low > 1 then begin
           (* now read off rows, forward and backwards *)
      let mid = (!low + !high) / 2 in 
           (* read forward *)
      begin 
        let fin = ref false in 
        let curr = ref mid in 
        while not !fin do 
          if !curr > nrows then begin
            fin := true;
          end else begin
            let currrow = row_reader !curr in
            if keycomparef (keyfunc currrow) = 0 then begin
              res := row_converter currrow :: !res;
            end else begin
              fin := true;
            end;
            incr curr;
          end
        done;
      end;
      res := List.rev !res;
           (* read backwards *)
      begin 
        let fin = ref false in 
        let curr = ref (mid - 1) in 
        while not !fin do 
          if !curr = 0 then begin
            fin := true
          end else  begin
            let currrow = row_reader !curr in
            if keycomparef (keyfunc currrow) = 0 then begin
              res := row_converter currrow :: !res;
            end else begin
              fin := true;
            end;
            decr curr;
          end;
        done;
      end;
    end;
         (* sanity check *)
    if checking then begin 
      let res2 = ref [] in 
      for i = 1 to nrows do
        let rowinfo = row_reader i in
        if keycomparef (keyfunc rowinfo) = 0 then 
          res2 := row_converter rowinfo :: !res2;
      done;
      if (List.rev !res2 <> !res) then 
        failwith ("results of binary search did not match results of linear search: linear search produced "^string_of_int (List.length !res2)^", binary search produced "^string_of_int (List.length !res));
    end;
    
    !res
  else 
    let res = ref [] in 
    for i = 1 to nrows do
      let rowinfo = row_reader i in
      if keycomparef (keyfunc rowinfo) = 0 then 
        res := row_converter rowinfo :: !res;
    done;
    List.rev !res  


let seek_read_optional_indexed_row ((nrows,row_reader,keyfunc,keycomparer,binchop,row_converter) as info) =
  match seek_read_indexed_rows info with 
    [k] -> Some k
  | [] -> None
  | h::t -> 
      dprint_endline ("multiple rows found when indexing table"); 
      Some h 
        
let seek_read_indexed_row ((nrows,row_reader,keyfunc,keycomparer,binchop,row_converter) as info) =
  match seek_read_optional_indexed_row info with 
  | Some row -> row
  | None -> failwith ("no row found for key when indexing table")

(*---------------------------------------------------------------------
 * The big fat reader.
 *---------------------------------------------------------------------*)

type binary_reader = { modul: Il.modul; dispose: unit -> unit }
 
type method_data = typ * callconv * string * typ list * typ * typ list 
type vararg_method_data = typ * callconv * string * typ list * typ list option * typ * typ list 
(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type ctxt = 
  { ilg: mscorlib_refs;
    data_end_points: int32 list Lazy.t;
    sorted: int64;
    pdb: (pdb_reader * (string -> source_document)) option;
    eptoken: table * int;
    nrows: table -> int; 
    text_phys_loc : int32; 
    text_phys_size : int32;
    data_phys_loc : int32;
    data_phys_size : int32;
    anyV2P : (string * int32) -> int32;
    metadata_addr: int32;
    native_resources_addr:int32;
    native_resources_size:int32;
    resources_addr:int32;
    strongname_addr:int32;
    vtable_fixups_addr:int32;
    is:input;
    infile:string;
    user_strings_stream_phys_loc: int32;
    strings_stream_phys_loc: int32;
    blobs_stream_phys_loc: int32;
    seek_read_user_string_heap: (int32 -> bytes);
    seek_read_string_heap: (int32 -> string);
    seek_read_blob_heap: (int32 -> bytes);
    guids_stream_phys_loc : int32;
    row_addr : (table -> int -> int32);
    table_bignesses : bool array;
    rs_bigness : bool;  
    tdor_bigness : bool;
    tomd_bigness : bool;   
    hc_bigness : bool;   
    hca_bigness : bool;   
    hfm_bigness : bool;   
    hds_bigness : bool;   
    mrp_bigness : bool;   
    hs_bigness : bool;   
    mdor_bigness : bool;   
    mf_bigness : bool;   
    i_bigness : bool;   
    cat_bigness : bool;   
    strings_big: bool;   
    guids_big: bool;   
    blobs_big: bool;   
    count_TypeRef : int ref;
    count_TypeDef : int ref;     
    count_Field : int ref;      
    count_Method : int ref;     
    count_Param : int ref;          
    count_InterfaceImpl : int ref;  
    count_MemberRef : int ref;        
    count_Constant : int ref;         
    count_CustomAttribute : int ref;  
    count_FieldMarshal: int ref;    
    count_Permission : int ref;      
    count_ClassLayout : int ref;     
    count_FieldLayout : int ref;       
    count_StandAloneSig : int ref;    
    count_EventMap : int ref;         
    count_Event : int ref;            
    count_PropertyMap : int ref;       
    count_Property : int ref;           
    count_MethodSemantics : int ref;    
    count_MethodImpl : int ref;  
    count_ModuleRef : int ref;       
    count_TypeSpec : int ref;         
    count_ImplMap : int ref;      
    count_FieldRVA : int ref;   
    count_Assembly : int ref;        
    count_AssemblyRef : int ref;    
    count_File : int ref;           
    count_ExportedType : int ref;  
     count_ManifestResource : int ref;
     count_Nested : int ref;         
     count_GenericParam : int ref;       
     count_GenericParamConstraint : int ref;     
     count_MethodSpec : int ref;        
     seek_read_Nested_row  : int -> int * int;
     seek_read_Constant_row  : int -> int32 * (hasConstant_tag * int) * int32;
     seek_read_MethodSemantics_row  : int -> int32 * int * (hasSemantics_tag * int);
     seek_read_TypeDef_row : int -> int32 * int32 * int32 * (typeDefOrRef_tag * int) * int * int;
     seek_read_InterfaceImpl_row  : int -> int * (typeDefOrRef_tag * int);
     seek_read_FieldMarshal_row  : int -> (hasFieldMarshal_tag * int) * int32;
     seek_read_PropertyMap_row  : int -> int * int; 
   seek_read_AssemblyRef : int -> assembly_ref;
   seek_read_MethodSpec_as_mdata : methodSpecAsMspecIdx -> vararg_method_data;
   seek_read_MemberRef_as_mdata : memberRefAsMspecIdx -> vararg_method_data;
   seek_read_MemberRef_as_fspec : memberRefAsFspecIdx -> field_spec;
   seek_read_CustomAttr : customAttrIdx -> custom_attr;
   seek_read_SecurityDecl : securityDeclIdx -> permission;
   seek_read_TypeRef : int ->type_ref;
   seek_read_TypeRef_as_typ : typeRefAsTypIdx -> typ;
   seek_read_blob_heap_as_property_sig : blobAsPropSigIdx -> hasthis * typ * typ list;
   seek_read_blob_heap_as_field_sig : blobAsFieldSigIdx -> typ;
   seek_read_blob_heap_as_method_sig : blobAsMethodSigIdx -> bool * int32 * callconv * typ * typ list * typ list option;

   seek_read_blob_heap_as_locals_sig : blobAsLocalSigIdx -> local list;
   seek_read_TypeDef_as_typ : typeDefAsTypIdx -> typ;
   seek_read_MethodDef_as_mdata : int -> method_data;
   seek_read_GenericParams : genericParsIdx -> genparam list;
   seek_read_FieldDef_as_fspec : int -> field_spec;

}
  
let rec gen_open_binary_reader infile is opts = 

  (* MSDOS HEADER *)
  let pe_signature_phys_loc = seek_read_i32 is !!!0x3c in 

  (* PE HEADER *)
  let pe_file_header_phys_loc = pe_signature_phys_loc +++ !!!0x04 in 
  let pe_optional_header_phys_loc = pe_file_header_phys_loc +++ !!!0x14 in
  let pe_signature = seek_read_i32 is (pe_signature_phys_loc +++ 0l) in 
  if pe_signature <>  !!!0x4550 then failwith "not a PE file - bad magic PE number";


  (* PE SIGNATURE *)
  let machine = seek_read_u16_as_i32 is (pe_file_header_phys_loc +++ 0l) in 
  let num_sections = seek_read_u16_as_i32 is (pe_file_header_phys_loc +++ !!!2) in 
  let opt_header_size = seek_read_u16_as_i32 is (pe_file_header_phys_loc +++ !!!16) in 
  if opt_header_size <>  !!!0xe0 &&
     opt_header_size <> !!!0xf0 then failwith "not a PE file - bad optional header size";
  let x64adjust = opt_header_size --- !!!0xe0 in 
  let section_headers_start_phys_loc = pe_optional_header_phys_loc +++ opt_header_size in 

  let flags = seek_read_u16_as_i32 is (pe_file_header_phys_loc +++ !!!18) in 
  let is_dll = (flags &&& !!!0x2000) <> !!!0x0 in 

 (* OPTIONAL PE HEADER *)
  let text_phys_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!4) in (* Size of the code (text) section, or the sum of all code sections if there are multiple sections. *)
   (* x86: 000000a0 *) 
  let initdata_phys_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!8) in (* Size of the initialized data section, or the sum of all such sections if there are multiple data sections. *)
  let uninitdata_phys_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!12) in (* Size of the uninitialized data section, or the sum of all such sections if there are multiple data sections. *)
  let entrypoint_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!16) in (* RVA of entry point , needs to point to bytes 0xFFl !!!0x25 followed by the RVA+!0x4000000 in a section marked execute/read for EXEs or 0 for DLLs e.g. !!!0x0000b57e *)
  let text_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!20) in (* e.g. !!!0x0002000 *)
   (* x86: 000000b0 *) 
  let data_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!24) in (* e.g. !!!0x0000c000 *)
  let image_base_real = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!28) in (* Image Base Always !!!0x400000 (see Section 23.1). - QUERY : no it's not always !!!0x400000, e.g. !!!0x034f0000 *)
  let align_virt = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!32) in  (*  Section Alignment Always !!!0x2000 (see Section 23.1). *)
  let align_phys = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!36) in (* File Alignment Either !!!0x200 or !!!0x1000. *)
   (* x86: 000000c0 *) 
  let os_major = seek_read_u16 is (pe_optional_header_phys_loc +++ !!!40) in  (*  OS Major Always 4 (see Section 23.1). *)
  let os_minor = seek_read_u16 is (pe_optional_header_phys_loc +++ !!!42) in  (* OS Minor Always 0 (see Section 23.1). *)
  let user_major = seek_read_u16 is (pe_optional_header_phys_loc +++ !!!44) in  (* User Major Always 0 (see Section 23.1). *)
  let user_minor = seek_read_u16 is (pe_optional_header_phys_loc +++ !!!46) in  (* User Minor Always 0 (see Section 23.1). *)
  let subsys_major = seek_read_u16 is (pe_optional_header_phys_loc +++ !!!48) in  (* SubSys Major Always 4 (see Section 23.1). *)
  let subsys_minor = seek_read_u16 is (pe_optional_header_phys_loc +++ !!!50) in  (* SubSys Minor Always 0 (see Section 23.1). *)
   (* x86: 000000d0 *) 
  let image_end_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!56) in (* Image Size: Size, in bytes, of image, including all headers and padding; shall be a multiple of Section Alignment. e.g. !!!0x0000e000 *)
  let header_phys_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!60) in (* Header Size Combined size of MS-DOS Header, PE Header, PE Optional Header and padding; shall be a multiple of the file alignment. *)
  let subsys = seek_read_u16 is (pe_optional_header_phys_loc +++ !!!68) in  (* SubSystem Subsystem required to run this image. Shall be either IMAGE_SUBSYSTEM_WINDOWS_CE_GUI (!0x3) or IMAGE_SUBSYSTEM_WINDOWS_GUI (!0x2). QUERY: Why is this 3 on the images ILASM produces??? *)
   (* x86: 000000e0 *) 

(* WARNING: THESE ARE 64 bit ON x64/ia64 *)
(*  let stack_reserve = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!72) in *)  (* Stack Reserve Size Always !!!0x100000 (1Mb) (see Section 23.1). *)
(*   let stack_commit = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!76) in  *) (* Stack Commit Size Always !!!0x1000 (4Kb) (see Section 23.1). *)
(*   let heap_reserve = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!80) in *)  (* Heap Reserve Size Always !!!0x100000 (1Mb) (see Section 23.1). *)
(*   let heap_commit = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!84) in *)  (* Heap Commit Size Always !!!0x1000 (4Kb) (see Section 23.1). *)
   (* x86: 000000f0, x64: 00000100 *) 
  let num_data_directories = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!92 +++ x64adjust) in  (* Number of Data Directories: Always !!!0x10 (see Section 23.1). *)
   (* 00000100 *) 
  let import_tab_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!104 +++ x64adjust) in  (* Import Table RVA of Import Table, (see clause 24.3.1). e.g. 0000b530 *) 
  let import_tab_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!108 +++ x64adjust) in (* Size of Import Table, (see clause 24.3.1).  *)
  let native_resources_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!112 +++ x64adjust) in  
  let native_resources_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!116 +++ x64adjust) in 
   (* 00000110 *) 
   (* 00000120 *) 
(*   let base_reloc_tab_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!136) in 
  let base_reloc_tab_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!140) in  *)
   (* 00000130 *) 
   (* 00000140 *) 
   (* 00000150 *) 
  let import_addr_tab_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!192 +++ x64adjust) in  (* RVA of Import Addr Table, (see clause 24.3.1). e.g. !!!0x00002000 *) 
  let import_addr_tab_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!196 +++ x64adjust) in (* Size of Import Addr Table, (see clause 24.3.1). e.g. !!!0x00002000 *) 
   (* 00000160 *) 
  let cli_header_addr = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!208 +++ x64adjust) in 
  let cli_header_size = seek_read_i32 is (pe_optional_header_phys_loc +++ !!!212 +++ x64adjust) in 
   (* 00000170 *) 


(* Crack section headers *)

  let find_section_header addr = 
    let rec look i pos = 
      if !!!i >= num_sections then !!!0x0 
      else
        let virt_size = seek_read_i32 is (pos +++ !!!8) in  
        let virt_addr = seek_read_i32 is (pos +++ !!!12) in  
        if (addr >= virt_addr && addr < virt_addr +++ virt_size) then pos 
        else look (i+1) (pos +++ !!!0x28) in 
    look 0 section_headers_start_phys_loc in 
  
  let text_header_start = find_section_header cli_header_addr in 
  let data_header_start = find_section_header data_addr in  
(*  let reloc_header_start = find_section_header base_reloc_tab_addr in  *)

  let text_size = if text_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (text_header_start +++ !!!8) in  
  let text_addr = if text_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (text_header_start +++ !!!12) in  
  let text_phys_size = if text_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (text_header_start +++ !!!16) in 
  let text_phys_loc = if text_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (text_header_start +++ !!!20) in 

(*
  let reloc_size = if reloc_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (reloc_header_start +++ !!!8) in  
  let reloc_addr = if reloc_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (reloc_header_start +++ !!!12) in  
  let reloc_phys_size = if reloc_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (reloc_header_start +++ !!!16) in 
  let reloc_phys_loc = if reloc_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (reloc_header_start +++ !!!20) in 
*)

   if logging then dprint_endline (infile ^ ": text_header_start = "^Int32.to_string text_header_start);
   if logging then dprint_endline (infile ^ ": data_header_start = "^Int32.to_string data_header_start);
   if logging then  dprint_endline (infile ^ ": data_addr (pre section crack) = "^Int32.to_string data_addr);

  let data_size = if data_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (data_header_start +++ !!!8) in  
  let data_addr = if data_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (data_header_start +++ !!!12) in  
  let data_phys_size = if data_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (data_header_start +++ !!!16) in 
  let data_phys_loc = if data_header_start = !!!0x0 then !!!0x0 else seek_read_i32 is (data_header_start +++ !!!20) in 

  if logging then dprint_endline (infile ^ ": data_addr (post section crack) = "^Int32.to_string data_addr);

  let anyV2P (n,v) = 
    let rec look i pos = 
      if !!!i >= num_sections then (failwith (infile ^ ": bad "^n^", rva "^Int32.to_string v); !!!0x0)
      else
        let virt_size = seek_read_i32 is (pos +++ !!!8) in  
        let virt_addr = seek_read_i32 is (pos +++ !!!12) in  
        let phys_loc = seek_read_i32 is (pos +++ !!!20) in  
        if (v >= virt_addr && (v < virt_addr +++ virt_size)) then (v --- virt_addr) +++ phys_loc 
        else look (i+1) (pos +++ !!!0x28) in 
    look 0 section_headers_start_phys_loc in 

(*  let relocV2P v = v --- reloc_addr +++ reloc_phys_loc in  *)

  if logging then dprint_endline (infile ^ ": num_sections = "^Int32.to_string num_sections); 
  if logging then dprint_endline (infile ^ ": cli_header_addr = "^Int32.to_string cli_header_addr); 
  if logging then dprint_endline (infile ^ ": cli_header_phys = "^Int32.to_string (anyV2P ("cli header",cli_header_addr))); 
  if logging then dprint_endline (infile ^ ": data_size = "^Int32.to_string data_size); 
  if logging then dprint_endline (infile ^ ": data_addr = "^Int32.to_string data_addr); 

  let cli_header_phys_loc = anyV2P ("cli header",cli_header_addr) in

  let major_runtime_version = seek_read_u16 is (cli_header_phys_loc +++ !!!4) in 
  let minor_runtime_version = seek_read_u16 is (cli_header_phys_loc +++ !!!6) in 
  let metadata_addr = seek_read_i32 is (cli_header_phys_loc +++ !!!8) in 
  let metadata_size = seek_read_i32 is (cli_header_phys_loc +++ !!!12) in 
  let cli_flags = seek_read_i32 is (cli_header_phys_loc +++ !!!16) in 
  let ilonly = (cli_flags &&& !!!0x01) <> !!!0x00 in 
  let only32  = (cli_flags &&& !!!0x02) <> !!!0x00 in 
  let strongname_signed  = (cli_flags &&& !!!0x08) <> !!!0x00 in 
  let trackdebugdata  = (cli_flags &&& !!!0x010000) <> !!!0x00 in 
  let eptoken = seek_read_uncoded_token is (cli_header_phys_loc +++ !!!20) in 
  let resources_addr = seek_read_i32 is (cli_header_phys_loc +++ !!!24) in 
  let resources_size = seek_read_i32 is (cli_header_phys_loc +++ !!!28) in 
  let strongname_addr = seek_read_i32 is (cli_header_phys_loc +++ !!!32) in 
  let strongname_size = seek_read_i32 is (cli_header_phys_loc +++ !!!36) in 
  let vtable_fixups_addr = seek_read_i32 is (cli_header_phys_loc +++ !!!40) in 
  let vtable_fixups_size = seek_read_i32 is (cli_header_phys_loc +++ !!!44) in 

  if logging then dprint_endline (infile ^ ": metadata_addr = "^Int32.to_string metadata_addr); 
  if logging then dprint_endline (infile ^ ": resources_addr = "^Int32.to_string resources_addr); 
  if logging then dprint_endline (infile ^ ": resources_size = "^Int32.to_string resources_size); 
  if logging then dprint_endline (infile ^ ": native_resources_addr = "^Int32.to_string native_resources_addr); 
  if logging then dprint_endline (infile ^ ": native_resources_size = "^Int32.to_string native_resources_size); 

  let metadata_phys_loc = anyV2P ("metadata",metadata_addr) in 
  let magic = seek_read_u16_as_i32 is metadata_phys_loc in 
  if magic <> !!!0x5342 then failwith (infile ^ ": bad metadata magic number: " ^ Int32.to_string magic);
  let magic2 = seek_read_u16_as_i32 is (metadata_phys_loc +++ !!!2) in 
  if magic2 <> !!!0x424a then failwith "bad metadata magic number";
  let major_metadata_version = seek_read_u16 is (metadata_phys_loc +++ !!!4) in 
  let minor_metadata_version = seek_read_u16 is (metadata_phys_loc +++ !!!6) in 

  let version_length = seek_read_i32 is (metadata_phys_loc +++ !!!12) in 
  let x = align !!!0x04 (!!!16 +++ version_length) in 
  let num_streams = seek_read_u16_as_i32 is (metadata_phys_loc +++ x +++ !!!2) in 
  let stream_headers_start = (metadata_phys_loc +++ x +++ !!!4) in 

  if logging then dprint_endline (infile ^ ": num_streams = "^Int32.to_string num_streams); 
  if logging then dprint_endline (infile ^ ": stream_headers_start = "^Int32.to_string stream_headers_start); 

(* Crack stream headers *)

  let try_find_stream name = 
    let rec look i pos = 
      if !!!i >= num_streams then raise Not_found
      else
        let offset = seek_read_i32 is (pos +++ 0l) in 
        let length = seek_read_i32 is (pos +++ !!!4) in 
        let res = ref true in 
        let fin = ref false in 
        let n = ref 0 in 
      (* read and compare the stream name byte by byte *)
        while (not !fin) do 
          let c= seek_read_u8_as_i32 is (pos +++ !!!8 +++ !!!(!n)) in 
          if c = 0l then fin := true
          else if !n >= Array.length name or c <> name.(!n) then res := false;
          incr n
        done;
        if !res then (offset +++ metadata_phys_loc,length) 
        else look (i+1) (align !!!0x04 (pos +++ !!!8 +++ !!!(!n))) in 
    look 0 stream_headers_start in 
  let find_stream name = try try_find_stream name with Not_found -> (!!!0x0, !!!0x0) in

  let (tables_stream_phys_loc, tables_stream_size) = 
    try try_find_stream [| !!!0x23; !!!0x7e |] (* #~ *) 
    with Not_found -> 
    try try_find_stream [| !!!0x23; !!!0x2d |] (* #-: at least one DLL I've seen uses this! *)  
    with Not_found -> 
      dprintf0 "no metadata tables found under stream names '#~' or '#-', please report this\n";
     let first_stream_offset = seek_read_i32 is (stream_headers_start +++ 0l) in 
     let first_stream_length = seek_read_i32 is (stream_headers_start +++ !!!4) in 
     first_stream_offset,first_stream_length in 
  let (strings_stream_phys_loc, strings_stream_size) = find_stream [| !!!0x23; !!!0x53; !!!0x74; !!!0x72; !!!0x69; !!!0x6e; !!!0x67; !!!0x73; |] (* #Strings *) in 
  let (user_strings_stream_phys_loc, user_strings_stream_size) = find_stream [| !!!0x23; !!!0x55; !!!0x53; |] (* #US *) in 
  let (guids_stream_phys_loc, guids_stream_size) = find_stream [| !!!0x23; !!!0x47; !!!0x55; !!!0x49; !!!0x44; |] (* #GUID *) in 
  let (blobs_stream_phys_loc, blobs_stream_size) = find_stream [| !!!0x23; !!!0x42; !!!0x6c; !!!0x6f; !!!0x62; |] (* #Blob *) in 

  if logging then dprint_endline (infile ^ ": tables_addr = "^Int32.to_string tables_stream_phys_loc); 
  if logging then dprint_endline (infile ^ ": tables_size = "^Int32.to_string tables_stream_size); 
  if logging then dprint_endline (infile ^ ": strings_addr = "^Int32.to_string strings_stream_phys_loc);
  if logging then dprint_endline (infile ^ ": strings_size = "^Int32.to_string strings_stream_size); 
  if logging then dprint_endline (infile ^ ": user_strings_addr = "^Int32.to_string user_strings_stream_phys_loc); 
  if logging then dprint_endline (infile ^ ": guids_addr = "^Int32.to_string guids_stream_phys_loc); 
  if logging then dprint_endline (infile ^ ": blobs_addr = "^Int32.to_string blobs_stream_phys_loc); 

  let tables_stream_major_version = seek_read_u8_as_i32 is (tables_stream_phys_loc +++ !!!4) in 
  let tables_stream_minor_version = seek_read_u8_as_i32 is (tables_stream_phys_loc +++ !!!5) in 

  let usingWhidbeyBeta1TableSchemeForGenericParam = (i32_to_int tables_stream_major_version = 1) && (i32_to_int tables_stream_minor_version = 1) in

  let table_kinds = 
  [|kind_Module               (* Table 0  *); 
    kind_TypeRef              (* Table 1  *);
    kind_TypeDef              (* Table 2  *);
    kind_Illegal (* kind_FieldPtr *)             (* Table 3  *);
    kind_FieldDef                (* Table 4  *);
    kind_Illegal (* kind_MethodPtr *)            (* Table 5  *);
    kind_MethodDef               (* Table 6  *);
    kind_Illegal (* kind_ParamPtr *)             (* Table 7  *);
    kind_Param                (* Table 8  *);
    kind_InterfaceImpl        (* Table 9  *);
    kind_MemberRef            (* Table 10 *);
    kind_Constant             (* Table 11 *);
    kind_CustomAttribute      (* Table 12 *);
    kind_FieldMarshal         (* Table 13 *);
    kind_DeclSecurity         (* Table 14 *);
    kind_ClassLayout          (* Table 15 *);
    kind_FieldLayout          (* Table 16 *);
    kind_StandAloneSig        (* Table 17 *);
    kind_EventMap             (* Table 18 *);
    kind_Illegal (* kind_EventPtr *)             (* Table 19 *);
    kind_Event                (* Table 20 *);
    kind_PropertyMap          (* Table 21 *);
    kind_Illegal (* kind_PropertyPtr *)          (* Table 22 *);
    kind_Property             (* Table 23 *);
    kind_MethodSemantics      (* Table 24 *);
    kind_MethodImpl           (* Table 25 *);
    kind_ModuleRef            (* Table 26 *);
    kind_TypeSpec             (* Table 27 *);
    kind_ImplMap              (* Table 28 *);
    kind_FieldRVA             (* Table 29 *);
    kind_Illegal (* kind_ENCLog *)               (* Table 30 *);
    kind_Illegal (* kind_ENCMap *)               (* Table 31 *);
    kind_Assembly             (* Table 32 *);
    kind_Illegal (* kind_AssemblyProcessor *)    (* Table 33 *);
    kind_Illegal (* kind_AssemblyOS *)           (* Table 34 *);
    kind_AssemblyRef          (* Table 35 *);
    kind_Illegal (* kind_AssemblyRefProcessor *) (* Table 36 *);
    kind_Illegal (* kind_AssemblyRefOS *)        (* Table 37 *);
    kind_FileRef                 (* Table 38 *);
    kind_ExportedType         (* Table 39 *);
    kind_ManifestResource     (* Table 40 *);
    kind_Nested               (* Table 41 *);
   (if usingWhidbeyBeta1TableSchemeForGenericParam then kind_GenericParam_v1_1 else  kind_GenericParam_v2_0);        (* Table 42 *)
    kind_MethodSpec         (* Table 43 *);
    kind_GenericParamConstraint         (* Table 44 *);
    kind_Illegal         (* Table 45 *);
    kind_Illegal         (* Table 46 *);
    kind_Illegal         (* Table 47 *);
    kind_Illegal         (* Table 48 *);
    kind_Illegal         (* Table 49 *);
    kind_Illegal         (* Table 50 *);
    kind_Illegal         (* Table 51 *);
    kind_Illegal         (* Table 52 *);
    kind_Illegal         (* Table 53 *);
    kind_Illegal         (* Table 54 *);
    kind_Illegal         (* Table 55 *);
    kind_Illegal         (* Table 56 *);
    kind_Illegal         (* Table 57 *);
    kind_Illegal         (* Table 58 *);
    kind_Illegal         (* Table 59 *);
    kind_Illegal         (* Table 60 *);
    kind_Illegal         (* Table 61 *);
    kind_Illegal         (* Table 62 *);
    kind_Illegal         (* Table 63 *);
  |] in

  let heap_sizes = seek_read_u8_as_i32 is (tables_stream_phys_loc +++ !!!6) in 
  let valid = seek_read_i64 is (tables_stream_phys_loc +++ !!!8) in 
  let sorted = seek_read_i64 is (tables_stream_phys_loc +++ !!!16) in 
  let tables_present, table_num_rows, start_of_tables = 
    let present = ref [] in 
    let num_rows = Array.create 64 0 in 
    let prev_numrow_idx = ref (tables_stream_phys_loc +++ !!!24) in 
    for i = 0 to 63 do 
      if (valid &&&& (!!!!1 <<<< i)) <> !!!! 0 then begin 
        present := i :: !present;
        num_rows.(i) <- Int32.to_int (seek_read_i32 is !prev_numrow_idx);
        prev_numrow_idx := !prev_numrow_idx +++ !!!4
      end
    done;
    List.rev !present, num_rows, !prev_numrow_idx in 
  let nrows t = table_num_rows.(tag_of_table t) in 
  let num_tables = List.length tables_present in 
  let strings_big = (heap_sizes &&& !!!1) <> 0l in 
  let guids_big = (heap_sizes &&& !!!2) <> 0l in 
  let blobs_big = (heap_sizes &&& !!!4) <> 0l in 

  if logging then dprint_endline (infile ^ ": num_tables = "^string_of_int num_tables);
  if logging && strings_big then dprint_endline (infile ^ ": strings are big");
  if logging && blobs_big then dprint_endline (infile ^ ": blobs are big");

  let table_bignesses = Array.map (fun n -> n >= 0x10000) table_num_rows in  
    
  let coded_bigness nbits tab =
    let rows = nrows tab in 
    !!!rows >= (!!!0x10000 lsr nbits) in 
  
  let tdor_bigness = 
    coded_bigness 2 tab_TypeDef || 
    coded_bigness 2 tab_TypeRef || 
    coded_bigness 2 tab_TypeSpec in 
  
  let tomd_bigness = 
    coded_bigness 1 tab_TypeDef || 
    coded_bigness 1 tab_Method in 
  
  let hc_bigness = 
    coded_bigness 2 tab_Field ||
    coded_bigness 2 tab_Param ||
    coded_bigness 2 tab_Property in 
  
    let hca_bigness = 
      coded_bigness 5 tab_Method ||
      coded_bigness 5 tab_Field ||
      coded_bigness 5 tab_TypeRef  ||
      coded_bigness 5 tab_TypeDef ||
      coded_bigness 5 tab_Param ||
      coded_bigness 5 tab_InterfaceImpl ||
      coded_bigness 5 tab_MemberRef ||
      coded_bigness 5 tab_Module ||
      coded_bigness 5 tab_Permission ||
      coded_bigness 5 tab_Property ||
      coded_bigness 5 tab_Event ||
      coded_bigness 5 tab_StandAloneSig ||
      coded_bigness 5 tab_ModuleRef ||
      coded_bigness 5 tab_TypeSpec ||
      coded_bigness 5 tab_Assembly ||
      coded_bigness 5 tab_AssemblyRef ||
      coded_bigness 5 tab_File ||
      coded_bigness 5 tab_ExportedType ||
      coded_bigness 5 tab_ManifestResource  in
    
    let hfm_bigness = 
      coded_bigness 1 tab_Field || 
      coded_bigness 1 tab_Param in 
    
    let hds_bigness = 
      coded_bigness 2 tab_TypeDef || 
      coded_bigness 2 tab_Method ||
      coded_bigness 2 tab_Assembly in 
    
    let mrp_bigness = 
      coded_bigness 3 tab_TypeRef ||
      coded_bigness 3 tab_ModuleRef ||
      coded_bigness 3 tab_Method ||
      coded_bigness 3 tab_TypeSpec in 
    
    let hs_bigness = 
      coded_bigness 1 tab_Event || 
      coded_bigness 1 tab_Property  in 
    
    let mdor_bigness =
      coded_bigness 1 tab_Method ||    
      coded_bigness 1 tab_MemberRef  in 
    
    let mf_bigness =
      coded_bigness 1 tab_Field ||
      coded_bigness 1 tab_Method  in 
    
    let i_bigness =
      coded_bigness 2 tab_File || 
      coded_bigness 2 tab_AssemblyRef ||    
      coded_bigness 2 tab_ExportedType  in
    
    let cat_bigness =  
      coded_bigness 3 tab_Method ||    
      coded_bigness 3 tab_MemberRef  in 
    
    let rs_bigness = 
      coded_bigness 2 tab_Module ||    
      coded_bigness 2 tab_ModuleRef || 
      coded_bigness 2 tab_AssemblyRef  ||
      coded_bigness 2 tab_TypeRef in
    
  let row_kind_size (RowKind kinds) = 
    List.fold_left 
      (fun sofar x -> 
        sofar +
          begin match x with 
          | UShort -> 2
          | ULong -> 4
          | Byte -> 1
          | Data -> 4
          | GGuid -> (if guids_big then 4 else 2)
          | Blob  -> (if blobs_big then 4 else 2)
          | SString  -> (if strings_big then 4 else 2)
          | SimpleIndex (Table tab) -> (if table_bignesses.(tab) then 4 else 2)
          | TypeDefOrRefOrSpec -> (if tdor_bigness then 4 else 2)
          | TypeOrMethodDef -> (if tomd_bigness then 4 else 2)
          | HasConstant  -> (if hc_bigness then 4 else 2)
          | HasCustomAttribute -> (if hca_bigness then 4 else 2)
          | HasFieldMarshal  -> (if hfm_bigness then 4 else 2)
          | HasDeclSecurity  -> (if hds_bigness then 4 else 2)
          | MemberRefParent  -> (if mrp_bigness then 4 else 2)
          | HasSemantics  -> (if hs_bigness then 4 else 2)
          | MethodDefOrRef -> (if mdor_bigness then 4 else 2)
          | MemberForwarded -> (if mf_bigness then 4 else 2)
          | Implementation  -> (if i_bigness then 4 else 2)
          | CustomAttributeType -> (if cat_bigness then 4 else 2)
          | ResolutionScope -> (if rs_bigness then 4 else 2)
          end) 0 kinds in 

   let table_row_sizes = 
     let res = Array.create 64 !!!0x0 in 
     for i = 0 to 63 do 
       res.(i) <- !!!(row_kind_size (table_kinds.(i)));
      (* dprintf2 "table_row_sizes.(%d) = %ld\n" i res.(i); *)
     done; 
     res in 

   let table_phys_locs = 
     let res = Array.create 64 !!!0x0 in 
     let prev_table_phys_loc = ref start_of_tables in 
     for i = 0 to 63 do 
       res.(i) <- !prev_table_phys_loc;
       prev_table_phys_loc := !prev_table_phys_loc +++ (!!!(table_num_rows.(i)) *** table_row_sizes.(i));
       if logging then dprintf3 "table_phys_locs.(%d) = %ld, offset from start_of_tables = 0x%08lx\n" i res.(i) (res.(i) ---  start_of_tables);
     done;
     res in 
  
   let inbase = Filename.basename infile^": " in 

(* All the caches.  The sizes are guesstimates for the rough sharing-density of the assembly *)
(* We should also take a parameter that indicates how much of the assembly we actually *)
(* expect to get read at all *)
   let cache_AssemblyRef = mk_cache_int opts.optimizeForMemory inbase "AssemblyRef"  (nrows (tab_AssemblyRef)) in 
   let cache_MethodSpec_as_mdata = mk_cache_gen opts.optimizeForMemory inbase "MethodSpec_as_mdata" (nrows (tab_MethodSpec) / 20 + 1) in 
   let cache_MemberRef_as_mdata = mk_cache_gen opts.optimizeForMemory inbase "MemberRef_as_mdata" (nrows (tab_MemberRef) / 20 + 1) in 
   let cache_MemberRef_as_fspec = mk_cache_gen opts.optimizeForMemory inbase "MemberRef_as_fspec" (nrows (tab_MemberRef) / 40 + 1) in 
   let cache_CustomAttr = mk_cache_gen opts.optimizeForMemory inbase "CustomAttr" (nrows (tab_CustomAttribute) / 10 + 1) in 
   let cache_SecurityDecl = mk_cache_gen opts.optimizeForMemory inbase "SecurityDecl" (nrows (tab_Permission) / 20 + 1) in 
   let cache_TypeRef = mk_cache_int opts.optimizeForMemory inbase "TypeRef" (nrows (tab_TypeRef) / 3 + 1) in 
   let cache_TypeRef_as_typ = mk_cache_gen opts.optimizeForMemory inbase "TypeRef_as_typ" (nrows (tab_TypeRef) / 20 + 1) in 
   let cache_blob_heap_as_property_sig = mk_cache_gen opts.optimizeForMemory inbase "blob_heap_as_property_sig" (nrows (tab_Property) / 20 + 1) in 
   let cache_blob_heap_as_field_sig = mk_cache_gen opts.optimizeForMemory inbase "blob_heap_as_field_sig" (nrows (tab_Field) / 20 + 1) in 
   let cache_blob_heap_as_method_sig = mk_cache_gen opts.optimizeForMemory inbase "blob_heap_as_method_sig" (nrows (tab_Method) / 20 + 1) in 
   let cache_blob_heap_as_locals_sig = mk_cache_gen opts.optimizeForMemory inbase "blob_heap_as_locals_sig" (nrows (tab_Method) / 20 + 1) in 
   let cache_TypeDef_as_typ = mk_cache_gen opts.optimizeForMemory inbase "TypeDef_as_typ" (nrows (tab_TypeDef) / 4 + 1) in 
   let cache_MethodDef_as_mdata = mk_cache_int opts.optimizeForMemory inbase "MethodDef_as_mdata" (nrows (tab_Method) / 20 + 1) in 
   let cache_GenericParams = mk_cache_gen opts.optimizeForMemory inbase "GenericParams" (nrows (tab_GenericParam) / 20 + 1) in 
   let cache_GenericParamConstraints = mk_cache_gen opts.optimizeForMemory inbase "GenericParamConstraints" (nrows (tab_GenericParamConstraint) / 8 + 1) in 
   let cache_FieldDef_as_fspec = mk_cache_int opts.optimizeForMemory inbase "FieldDef_as_fspec" (nrows (tab_Field) / 20 + 1) in 
   let cache_user_string_heap = mk_cache_int32 opts.optimizeForMemory inbase "user_string heap" (Int32.to_int user_strings_stream_size / 20 + 1) in 
   (* nb. Lots and lots of cache hits on this cache, hence never optimize cache away *)
   let cache_string_heap = mk_cache_int32 false inbase "string heap" (Int32.to_int strings_stream_size / 50 + 1) in 
   let cache_blob_heap = mk_cache_int32 opts.optimizeForMemory inbase "blob heap" (Int32.to_int blobs_stream_size / 50 + 1)  in 

   (* These tables are not required to enforce sharing fo the final data *)
   (* structure, but are very useful as searching these tables gives rise to many reads *)
   (* in standard applications.  It is very possible we should cache all reading of all rows. *)
   
   let cache_Nested_row = mk_cache_int opts.optimizeForMemory  inbase "Nested Table Rows" (nrows (tab_Nested) / 20 + 1) in 
   let cache_Constant_row = mk_cache_int opts.optimizeForMemory inbase "Constant Rows" (nrows (tab_Constant) / 20 + 1) in 
   let cache_MethodSemantics_row = mk_cache_int opts.optimizeForMemory inbase "MethodSemantics Rows" (nrows (tab_MethodSemantics) / 20 + 1) in 
   let cache_TypeDef_row = mk_cache_int opts.optimizeForMemory inbase "TypeDef Rows" (nrows (tab_TypeDef) / 20 + 1) in 
   let cache_InterfaceImpl_row = mk_cache_int opts.optimizeForMemory inbase "InterfaceImpl Rows" (nrows (tab_InterfaceImpl) / 20 + 1) in 
   let cache_FieldMarshal_row = mk_cache_int opts.optimizeForMemory inbase "FieldMarshal Rows" (nrows (tab_FieldMarshal) / 20 + 1) in 
   let cache_PropertyMap_row = mk_cache_int opts.optimizeForMemory inbase "PropertyMap Rows" (nrows (tab_PropertyMap) / 20 + 1) in 

   let mk_row_counter nm  =
     let count = ref 0 in 
     add_report (fun oc -> if !count <> 0 then output_string oc (inbase^string_of_int !count ^ " "^nm^" rows read"^"\n"));
     count in 

   let count_TypeRef = mk_row_counter               "TypeRef" in             
   let count_TypeDef = mk_row_counter               "TypeDef" in             
   let count_Field = mk_row_counter                 "Field" in               
   let count_Method = mk_row_counter                "Method" in              
   let count_Param = mk_row_counter                 "Param" in               
   let count_InterfaceImpl = mk_row_counter         "InterfaceImpl" in       
   let count_MemberRef = mk_row_counter             "MemberRef" in           
   let count_Constant = mk_row_counter              "Constant" in            
   let count_CustomAttribute = mk_row_counter       "CustomAttribute" in     
   let count_FieldMarshal = mk_row_counter          "FieldMarshal" in        
   let count_Permission = mk_row_counter           "Permission" in         
   let count_ClassLayout = mk_row_counter          "ClassLayout" in        
   let count_FieldLayout = mk_row_counter           "FieldLayout" in         
   let count_StandAloneSig = mk_row_counter         "StandAloneSig" in       
   let count_EventMap = mk_row_counter              "EventMap" in            
   let count_Event = mk_row_counter                 "Event" in               
   let count_PropertyMap = mk_row_counter           "PropertyMap" in         
   let count_Property = mk_row_counter              "Property" in            
   let count_MethodSemantics = mk_row_counter       "MethodSemantics" in     
   let count_MethodImpl = mk_row_counter            "MethodImpl" in          
   let count_ModuleRef = mk_row_counter             "ModuleRef" in           
   let count_TypeSpec = mk_row_counter              "TypeSpec" in            
   let count_ImplMap = mk_row_counter               "ImplMap" in             
   let count_FieldRVA = mk_row_counter              "FieldRVA" in            
   let count_Assembly = mk_row_counter              "Assembly" in            
   let count_AssemblyRef = mk_row_counter           "AssemblyRef" in         
   let count_File = mk_row_counter                  "File" in                
   let count_ExportedType = mk_row_counter          "ExportedType" in        
   let count_ManifestResource = mk_row_counter      "ManifestResource" in    
   let count_Nested = mk_row_counter                "Nested" in              
   let count_GenericParam = mk_row_counter            "GenericParam" in              
   let count_GenericParamConstraint = mk_row_counter            "GenericParamConstraint" in              
   let count_MethodSpec = mk_row_counter            "MethodSpec" in              


 (*-----------------------------------------------------------------------
  * Set up the PDB reader so we can read debug info for methods.
  * ----------------------------------------------------------------------*)

   let pdb = 
     match opts.pdbPath with 
     | None -> None
     | Some pdbpath -> 
         try 

           let pdbr = pdbReadOpen infile pdbpath in 
           let pdbdocs = pdbReaderGetDocuments pdbr in
           let tab = Hashtbl.create (Array.length pdbdocs) in 
           Array.iter 
             (fun pdbdoc -> 
               let url = pdbDocumentGetURL pdbdoc in 
               Hashtbl.add tab url 
                 { sourceLanguage = Some (pdbDocumentGetLanguage pdbdoc);
                   sourceVendor = Some (pdbDocumentGetLanguageVendor pdbdoc);
                   sourceDocType = Some (pdbDocumentGetType pdbdoc);
                   sourceFile = pdbDocumentGetURL pdbdoc })
             pdbdocs;
           let docfun url = if Hashtbl.mem tab url then Hashtbl.find tab url else failwith ("Document with URL "^url^" not found in list of documents in the PDB file") in
           Some (pdbr, docfun)
         with e -> dprint_endline ("*** Warning: PDB file could not be read and will be ignored: "^(Printexc.to_string e)); None in 

  let row_addr tab idx = table_phys_locs.(tag_of_table tab) +++ !!!(idx - 1) *** table_row_sizes.(tag_of_table tab) in


 (*-----------------------------------------------------------------------
  * Build the reader context
  * ----------------------------------------------------------------------*)

  (* use an initialization hole *)
  let ctxtH = ref None in
  let ctxt = { ilg=opts.mscorlib; 
               data_end_points = data_end_points ctxtH;
               pdb=pdb;
               sorted=sorted;
               nrows=nrows; 
               text_phys_loc=text_phys_loc; 
               text_phys_size=text_phys_size;
               data_phys_loc=data_phys_loc;
               data_phys_size=data_phys_size;
               anyV2P=anyV2P;
               metadata_addr=metadata_addr;
               native_resources_addr=native_resources_addr;
               native_resources_size=native_resources_size;
               resources_addr=resources_addr;
               strongname_addr=strongname_addr;
               vtable_fixups_addr=vtable_fixups_addr;
               is=is;
               infile=infile;
               user_strings_stream_phys_loc=user_strings_stream_phys_loc;
               strings_stream_phys_loc=strings_stream_phys_loc;
               blobs_stream_phys_loc=blobs_stream_phys_loc;
               seek_read_user_string_heap = cache_user_string_heap (seek_read_user_string_heap_uncached ctxtH);
               seek_read_string_heap = cache_string_heap (seek_read_string_heap_uncached ctxtH);
               seek_read_blob_heap = cache_blob_heap (seek_read_blob_heap_uncached ctxtH);
               seek_read_Nested_row  = cache_Nested_row  (seek_read_Nested_row_uncached ctxtH);
               seek_read_Constant_row  = cache_Constant_row  (seek_read_Constant_row_uncached ctxtH);
               seek_read_MethodSemantics_row  = cache_MethodSemantics_row  (seek_read_MethodSemantics_row_uncached ctxtH);
               seek_read_TypeDef_row  = cache_TypeDef_row  (seek_read_TypeDef_row_uncached ctxtH);
               seek_read_InterfaceImpl_row  = cache_InterfaceImpl_row  (seek_read_InterfaceImpl_row_uncached ctxtH);
               seek_read_FieldMarshal_row  = cache_FieldMarshal_row  (seek_read_FieldMarshal_row_uncached ctxtH);
               seek_read_PropertyMap_row = cache_PropertyMap_row  (seek_read_PropertyMap_row_uncached ctxtH);
               seek_read_AssemblyRef = cache_AssemblyRef  (seek_read_AssemblyRef_uncached ctxtH);
               seek_read_MethodSpec_as_mdata = cache_MethodSpec_as_mdata  (seek_read_MethodSpec_as_mdata_uncached ctxtH);
               seek_read_MemberRef_as_mdata = cache_MemberRef_as_mdata  (seek_read_MemberRef_as_mdata_uncached ctxtH);
               seek_read_MemberRef_as_fspec = cache_MemberRef_as_fspec  (seek_read_MemberRef_as_fspec_uncached ctxtH);
               seek_read_CustomAttr = cache_CustomAttr  (seek_read_CustomAttr_uncached ctxtH);
               seek_read_SecurityDecl = cache_SecurityDecl (seek_read_SecurityDecl_uncached ctxtH);
               seek_read_TypeRef = cache_TypeRef (seek_read_TypeRef_uncached ctxtH);
               seek_read_blob_heap_as_property_sig = cache_blob_heap_as_property_sig (seek_read_blob_heap_as_property_sig_uncached ctxtH);
               seek_read_blob_heap_as_field_sig = cache_blob_heap_as_field_sig (seek_read_blob_heap_as_field_sig_uncached ctxtH);
               seek_read_blob_heap_as_method_sig = cache_blob_heap_as_method_sig (seek_read_blob_heap_as_method_sig_uncached ctxtH);
               seek_read_blob_heap_as_locals_sig = cache_blob_heap_as_locals_sig (seek_read_blob_heap_as_locals_sig_uncached ctxtH);
               seek_read_TypeDef_as_typ = cache_TypeDef_as_typ (seek_read_TypeDef_as_typ_uncached ctxtH);
               seek_read_TypeRef_as_typ = cache_TypeRef_as_typ (seek_read_TypeRef_as_typ_uncached ctxtH);
               seek_read_MethodDef_as_mdata = cache_MethodDef_as_mdata (seek_read_MethodDef_as_mdata_uncached ctxtH);
               seek_read_GenericParams = cache_GenericParams (seek_read_GenericParams_uncached ctxtH);
               seek_read_FieldDef_as_fspec = cache_FieldDef_as_fspec (seek_read_FieldDef_as_fspec_uncached ctxtH);
               guids_stream_phys_loc = guids_stream_phys_loc;
               row_addr=row_addr;
               eptoken=eptoken; 
               rs_bigness =rs_bigness;
               tdor_bigness =tdor_bigness;
               tomd_bigness =tomd_bigness;   
               hc_bigness =hc_bigness;   
               hca_bigness =hca_bigness;   
               hfm_bigness =hfm_bigness;   
               hds_bigness =hds_bigness;
               mrp_bigness =mrp_bigness;
               hs_bigness =hs_bigness;
               mdor_bigness =mdor_bigness;
               mf_bigness =mf_bigness;
               i_bigness =i_bigness;
               cat_bigness =cat_bigness; 
               strings_big=strings_big;
               guids_big=guids_big;
               blobs_big=blobs_big;
               table_bignesses=table_bignesses;
               count_TypeRef = count_TypeRef;             
               count_TypeDef = count_TypeDef;             
               count_Field = count_Field;               
               count_Method = count_Method;              
               count_Param = count_Param;               
               count_InterfaceImpl = count_InterfaceImpl;       
               count_MemberRef = count_MemberRef;           
               count_Constant = count_Constant;            
               count_CustomAttribute = count_CustomAttribute;     
               count_FieldMarshal = count_FieldMarshal;        
               count_Permission = count_Permission;         
               count_ClassLayout = count_ClassLayout;        
               count_FieldLayout = count_FieldLayout;         
               count_StandAloneSig = count_StandAloneSig;       
               count_EventMap = count_EventMap;            
               count_Event = count_Event;               
               count_PropertyMap = count_PropertyMap;         
               count_Property = count_Property;            
               count_MethodSemantics = count_MethodSemantics;     
               count_MethodImpl = count_MethodImpl;          
               count_ModuleRef = count_ModuleRef;           
               count_TypeSpec = count_TypeSpec;            
               count_ImplMap = count_ImplMap;             
               count_FieldRVA = count_FieldRVA;            
               count_Assembly = count_Assembly;            
               count_AssemblyRef = count_AssemblyRef;         
               count_File = count_File;                
               count_ExportedType = count_ExportedType;        
               count_ManifestResource = count_ManifestResource;    
               count_Nested = count_Nested;              
               count_GenericParam = count_GenericParam;              
               count_GenericParamConstraint = count_GenericParamConstraint;              
               count_MethodSpec = count_MethodSpec;  }  in 
          ctxtH := Some ctxt;
   
 (*-----------------------------------------------------------------------
  * This is gross.  Some binaries have raw data embedded in 
  * their text sections, e.g. mscorlib, for field inits.  And there is no 
  * information that definitively tells us the extent of 
  * the text section that may be interesting data.  
  * But we certainly don't want to duplicate the entire 
  * text section as data! 
  *  
  * So, we assume: 
  *   1. no part of the metadata is double-used for raw data  
  *   2. the data bits are all the bits of the text section 
  *      that stretch from a Field or Resource RVA to one of 
  *        (a) the next Field or resource RVA 
  *        (b) a MethodRVA 
  *        (c) the start of the metadata 
  *        (d) the end of a section 
  *        (e) the start of the native resources attached to the binary if any
  * ----------------------------------------------------------------------*)


   seek_read_Module ctxt (subsys,ilonly,only32,is_dll, align_virt,align_phys,image_base_real) 1,pdb

and read_z_untagged_idx tab ctxt = Int32.to_int (if ctxt.table_bignesses.(tag_of_table tab) then read_i32 ctxt.is else read_u16_as_i32 ctxt.is) 


and read_rs_idx    ctxt = read_z_tagged_idx mkResolutionScopeTag     2 ctxt.rs_bigness ctxt.is   
and read_tdor_idx  ctxt = read_z_tagged_idx mkTypeDefOrRefOrSpecTag  2 ctxt.tdor_bigness ctxt.is   
and read_tomd_idx  ctxt = read_z_tagged_idx mkTypeOrMethodDefTag     1 ctxt.tomd_bigness ctxt.is   
and read_hc_idx    ctxt = read_z_tagged_idx mkHasConstantTag         2 ctxt.hc_bigness ctxt.is   
and read_hca_idx   ctxt = read_z_tagged_idx mkHasCustomAttributeTag  5 ctxt.hca_bigness ctxt.is   
and read_hfm_idx   ctxt = read_z_tagged_idx mkHasFieldMarshalTag     1 ctxt.hfm_bigness ctxt.is   
and read_hds_idx   ctxt = read_z_tagged_idx mkHasDeclSecurityTag     2 ctxt.hds_bigness ctxt.is   
and read_mrp_idx   ctxt = read_z_tagged_idx mkMemberRefParentTag     3 ctxt.mrp_bigness ctxt.is   
and read_hs_idx    ctxt = read_z_tagged_idx mkHasSemanticsTag        1 ctxt.hs_bigness ctxt.is   
and read_mdor_idx  ctxt = read_z_tagged_idx mkMethodDefOrRefTag      1 ctxt.mdor_bigness ctxt.is   
and read_mf_idx    ctxt = read_z_tagged_idx mkMemberForwardedTag     1 ctxt.mf_bigness ctxt.is   
and read_i_idx     ctxt = read_z_tagged_idx mkImplementationTag      2 ctxt.i_bigness ctxt.is   
and read_cat_idx   ctxt = read_z_tagged_idx mkCustomAttributeTypeTag 3 ctxt.cat_bigness ctxt.is   
and read_string_idx ctxt = if ctxt.strings_big then read_i32 ctxt.is else read_u16_as_i32 ctxt.is 
and read_guid_idx ctxt = if ctxt.guids_big then read_i32 ctxt.is else read_u16_as_i32 ctxt.is
and read_blob_idx ctxt = if ctxt.blobs_big then read_i32 ctxt.is else read_u16_as_i32 ctxt.is 

and read_native_resources ctxt = 

  let native_resources = 
    if logging then dprint_endline (ctxt.infile ^ ": native_resources_size = "^Int32.to_string ctxt.native_resources_size);
    if logging then dprint_endline (ctxt.infile ^ ": native_resources_addr = "^Int32.to_string ctxt.native_resources_addr);
    if ctxt.native_resources_size = !!!0x0 or ctxt.native_resources_addr = !!!0x0 then 
      None
    else
        Some
             (lazy 
              begin
                if logging then dprint_endline (ctxt.infile ^ ": reading linked resource...");
                let linkedResource = seek_read_bytes ctxt.is (ctxt.anyV2P (ctxt.infile ^ ": native resources",ctxt.native_resources_addr)) ctxt.native_resources_size in 
                if logging then dprint_endline (ctxt.infile ^ ": size = "^string_of_int (Bytes.length linkedResource));
                if logging then dprint_endline (ctxt.infile ^ ": unlinking resource...");
                unlinkResource ctxt.native_resources_addr linkedResource
              end) in 
  native_resources
   
and data_end_points ctxtH = 
    lazy
      begin 
        let ctxt = getH ctxtH in 
        let data_start_points = 
          let res = ref [] in 
          for i = 1 to ctxt.nrows (tab_FieldRVA) do
            let rva,fidx = seek_read_FieldRVA_row ctxt i in 
            res := ("field",rva) :: !res;
          done;
          for i = 1 to ctxt.nrows (tab_ManifestResource) do
            let (offset,_,_,(tag,idx)) = seek_read_ManifestResource_row ctxt i in 
            if idx = 0 then 
              let rva = ctxt.resources_addr +++ offset in 
              res := ("manifest resource", rva) :: !res;
          done;
          !res in
        if isNil data_start_points then [] 
        else
          let method_rvas = 
            let res = ref [] in 
            for i = 1 to ctxt.nrows (tab_Method) do
              let (rva, _, _, name_idx, _, _) = seek_read_Method_row ctxt i in
              if rva <> 0l then begin
                 let nm = seek_read_string_heap ctxt name_idx in
                 res := (nm,rva) :: !res;
              end;
            done;
            !res in
          List.sort compare
            ([ ctxt.text_phys_loc +++ ctxt.text_phys_size; 
              ctxt.data_phys_loc +++ ctxt.data_phys_size; ] 
             @ 
            (List.map ctxt.anyV2P (data_start_points 
                              @ [("md",ctxt.metadata_addr)]
                              @ (if ctxt.native_resources_addr = !!!0x0 then [] else [("native resources",ctxt.native_resources_addr); ])
                              @ (if ctxt.resources_addr = !!!0x0 then [] else [("managed resources",ctxt.resources_addr); ])
                              @ (if ctxt.strongname_addr = !!!0x0 then [] else [("managed strongname",ctxt.strongname_addr); ])
                              @ (if ctxt.vtable_fixups_addr = !!!0x0 then [] else [("managed vtable_fixups",ctxt.vtable_fixups_addr); ])
                              @ method_rvas)));
      end 
      

and rva_to_data ctxt nm rva = 
    if rva = !!!0x0 then failwith "rva is zero";
    let start = ctxt.anyV2P (nm, rva) in 
    let rec look l = 
      match l with 
      | [] -> 
          failwith ("find_text_data_extent: none found for "^nm^" rva "^Int32.to_string rva); 
          Bytes.of_intarray [| |]
      | e::t -> 
         if start < e then 
           (seek_read_bytes ctxt.is start (e --- start)) 
         else look t in 
    look (Lazy.force ctxt.data_end_points)

and seek_read_user_string_heap_uncached ctxtH idx = 
     let ctxt = getH ctxtH in 
     if logging then dprint_endline (ctxt.infile ^ ": reading user string heap "^Int32.to_string idx);
     let res = seek_read_user_string ctxt.is (ctxt.user_strings_stream_phys_loc +++ idx) in 
     (* if logging then dprint_endline (ctxt.infile ^ ": read string '"^res^"'"); *)
     res 
and seek_read_user_string_heap        ctxt idx = ctxt.seek_read_user_string_heap  idx 
and seek_read_user_string_heap_option ctxt idx = if idx = 0l then None else Some (seek_read_user_string_heap ctxt idx) 

and seek_read_string_heap_uncached ctxtH idx = 
     let ctxt = getH ctxtH in 
     seek_read_utf8_string ctxt.is (ctxt.strings_stream_phys_loc +++ idx) 
and seek_read_string_heap          ctxt idx = ctxt.seek_read_string_heap idx 
and seek_read_string_heap_option   ctxt idx = if idx = 0l then None else Some (seek_read_string_heap ctxt idx) 

and seek_read_blob_heap_uncached ctxtH idx = 
     let ctxt = getH ctxtH in 
     if logging then dprint_endline (ctxt.infile ^ ": reading blob heap "^Int32.to_string idx);
     seek_read_blob ctxt.is (ctxt.blobs_stream_phys_loc +++ idx) 
and seek_read_blob_heap        ctxt idx = ctxt.seek_read_blob_heap idx 
and seek_read_blob_heap_option ctxt idx = if idx = 0l then None else Some (seek_read_blob_heap ctxt idx) 

and seek_read_guids_heap ctxt idx = seek_read_guid ctxt.is (ctxt.guids_stream_phys_loc +++ idx) 

   (* read a single value out of a blob heap using the given function *)
and seek_read_blob_heap_as_bool   ctxt vidx = fst (sigptr_get_bool   (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_i8     ctxt vidx = fst (sigptr_get_i8     (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_i16    ctxt vidx = fst (sigptr_get_i16    (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_i32    ctxt vidx = fst (sigptr_get_i32    (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_i64    ctxt vidx = fst (sigptr_get_i64    (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_u8     ctxt vidx = fst (sigptr_get_u8     (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_u16    ctxt vidx = fst (sigptr_get_u16    (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_u32    ctxt vidx = fst (sigptr_get_u32    (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_u64    ctxt vidx = fst (sigptr_get_u64    (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_ieee32 ctxt vidx = fst (sigptr_get_ieee32 (seek_read_blob_heap ctxt vidx) 0) 
and seek_read_blob_heap_as_ieee64 ctxt vidx = fst (sigptr_get_ieee64 (seek_read_blob_heap ctxt vidx) 0) 
   
and seek_read_Module_row ctxt idx =
     if idx = 0 then failwith "cannot read Module table row 0";
     let addr = ctxt.row_addr tab_Module idx in 
     if logging then dprint_endline (ctxt.infile ^ ": module row addr = " ^ Int32.to_string addr);

     seek ctxt.is addr;
     let generation = read_u16 ctxt.is in 
     let name_idx = read_string_idx ctxt in 
     let mvid_idx = read_guid_idx ctxt in 
     let encid_idx = read_guid_idx ctxt in 
     let encbaseid_idx = read_guid_idx ctxt in 
     (generation, name_idx, mvid_idx, encid_idx, encbaseid_idx) 

   (* Read Table TypeRef *)
and seek_read_TypeRef_row ctxt idx =
     incr ctxt.count_TypeRef;
     let addr = ctxt.row_addr tab_TypeRef idx in 
     seek ctxt.is addr;
     let scope_idx = read_rs_idx ctxt in 
     let name_idx = read_string_idx ctxt in 
     let namespace_idx = read_string_idx ctxt in 
     (scope_idx,name_idx,namespace_idx) 

   (* Read Table TypeDef *)
and seek_read_TypeDef_row ctxt idx = ctxt.seek_read_TypeDef_row idx
and seek_read_TypeDef_row_uncached ctxtH idx =
    let ctxt = getH ctxtH in
    incr ctxt.count_TypeDef;
    let addr = ctxt.row_addr tab_TypeDef idx in 
    seek ctxt.is addr;
    let flags = read_i32 ctxt.is in 
    let name_idx = read_string_idx ctxt in 
    let namespace_idx = read_string_idx ctxt in 
    let extends_idx = read_tdor_idx ctxt in 
    let fields_idx = read_z_untagged_idx tab_Field ctxt in 
    let methods_idx = read_z_untagged_idx tab_Method ctxt in 
    (flags, name_idx, namespace_idx, extends_idx, fields_idx, methods_idx) 

   (* Read Table Field *)
and seek_read_Field_row ctxt idx =
     incr ctxt.count_Field;
     let addr = ctxt.row_addr tab_Field idx in 
     seek ctxt.is addr;
     let flags = read_u16_as_i32 ctxt.is in 
     let name_idx = read_string_idx ctxt in 
     let type_idx = read_blob_idx ctxt in 
     (flags,name_idx,type_idx)  

    (* Read Table Method *)
and seek_read_Method_row ctxt idx =
     incr ctxt.count_Method;
     let addr = ctxt.row_addr tab_Method idx in 
     seek ctxt.is addr;
     let code_rva = read_i32 ctxt.is in 
     let implflags = read_u16_as_i32 ctxt.is in 
     let flags = read_u16_as_i32 ctxt.is in 
     let name_idx = read_string_idx ctxt in 
     let type_idx = read_blob_idx ctxt in 
     let param_idx = read_z_untagged_idx tab_Param ctxt in 
     (code_rva, implflags, flags, name_idx, type_idx, param_idx) 

    (* Read Table Param *)
and seek_read_Param_row ctxt idx =
     incr ctxt.count_Param;
     let addr = ctxt.row_addr tab_Param idx in 
     seek ctxt.is addr;
     let flags = read_u16_as_i32 ctxt.is in 
     let seq = Int32.to_int (read_u16_as_i32 ctxt.is) in 
     let name_idx = read_string_idx ctxt in 
     (flags,seq,name_idx) 

    (* Read Table InterfaceImpl *)
and seek_read_InterfaceImpl_row ctxt idx = ctxt.seek_read_InterfaceImpl_row idx
and seek_read_InterfaceImpl_row_uncached ctxtH idx =
     let ctxt = getH ctxtH in
      incr ctxt.count_InterfaceImpl;
       let addr = ctxt.row_addr tab_InterfaceImpl idx in 
       seek ctxt.is addr;
       let tidx = read_z_untagged_idx tab_TypeDef ctxt in 
       let intf_idx = read_tdor_idx ctxt in 
       (tidx,intf_idx)

    (* Read Table MemberRef *)
and seek_read_MemberRef_row ctxt idx =
     incr ctxt.count_MemberRef;
     let addr = ctxt.row_addr tab_MemberRef idx in 
     seek ctxt.is addr;
     let mrp_idx = read_mrp_idx ctxt in 
     let name_idx = read_string_idx ctxt in 
     let type_idx = read_blob_idx ctxt in 
     (mrp_idx,name_idx,type_idx) 

    (* Read Table Constant *)
and seek_read_Constant_row ctxt idx = ctxt.seek_read_Constant_row idx
and seek_read_Constant_row_uncached ctxtH idx =
     let ctxt = getH ctxtH in
       incr ctxt.count_Constant;
       let addr = ctxt.row_addr tab_Constant idx in 
       seek ctxt.is addr;
       let kind = read_u16_as_i32 ctxt.is in 
       let parent_idx = read_hc_idx ctxt in 
       let val_idx = read_blob_idx ctxt in 
       (kind, parent_idx, val_idx)

    (* Read Table CustomAttribute *)
and seek_read_CustomAttribute_row ctxt idx =
     incr ctxt.count_CustomAttribute;
     let addr = ctxt.row_addr tab_CustomAttribute idx in 
     seek ctxt.is addr;
     let parent_idx = read_hca_idx ctxt in 
     let type_idx = read_cat_idx ctxt in 
     let val_idx = read_blob_idx ctxt in 
     (parent_idx, type_idx, val_idx)  

    (* Read Table FieldMarshal *)
and seek_read_FieldMarshal_row ctxt idx = ctxt.seek_read_FieldMarshal_row idx
and seek_read_FieldMarshal_row_uncached ctxtH idx =
     let ctxt = getH ctxtH in
       incr ctxt.count_FieldMarshal;
       let addr = ctxt.row_addr tab_FieldMarshal idx in 
       seek ctxt.is addr;
       let parent_idx = read_hfm_idx ctxt in 
       let type_idx = read_blob_idx ctxt in 
       (parent_idx, type_idx)

    (* Read Table Permission *)
and seek_read_Permission_row ctxt idx =
      incr ctxt.count_Permission;
    (* if logging then dprint_endline (ctxt.infile ^ ": reading Permission row "^string_of_int idx); *)
     let addr = ctxt.row_addr tab_Permission idx in 
     seek ctxt.is addr;
     let action = read_u16 ctxt.is in 
     let parent_idx = read_hds_idx ctxt in 
     let type_idx = read_blob_idx ctxt in 
     (* if logging then dprint_endline "finished read of Permission row";*)
     (action,parent_idx, type_idx) 

    (* Read Table ClassLayout *)
and seek_read_ClassLayout_row ctxt idx =
      incr ctxt.count_ClassLayout;
     let addr = ctxt.row_addr tab_ClassLayout idx in 
     seek ctxt.is addr;
     let pack = read_u16 ctxt.is in 
     let size = read_i32 ctxt.is in 
     let tidx = read_z_untagged_idx tab_TypeDef ctxt in 
     (pack,size,tidx)  

    (* Read Table FieldLayout *)
and seek_read_FieldLayout_row ctxt idx =
     incr ctxt.count_FieldLayout;
     let addr = ctxt.row_addr tab_FieldLayout idx in 
     seek ctxt.is addr;
     let offset = read_i32 ctxt.is in 
     let fidx = read_z_untagged_idx tab_Field ctxt in 
     (offset,fidx)  

    (* Read Table StandAloneSig *)
and seek_read_StandAloneSig_row ctxt idx =
     incr ctxt.count_StandAloneSig;
     let addr = ctxt.row_addr tab_StandAloneSig idx in 
     seek ctxt.is addr;
     let sig_idx = read_blob_idx ctxt in 
     (sig_idx)  

    (* Read Table EventMap *)
and seek_read_EventMap_row ctxt idx =
     incr ctxt.count_EventMap;
     let addr = ctxt.row_addr tab_EventMap idx in 
     seek ctxt.is addr;
     let tidx = read_z_untagged_idx tab_TypeDef ctxt in 
     let events_idx = read_z_untagged_idx tab_Event ctxt in 
     (tidx,events_idx) 

    (* Read Table Event *)
and seek_read_Event_row ctxt idx =
     incr ctxt.count_Event;
     let addr = ctxt.row_addr tab_Event idx in 
     seek ctxt.is addr;
     let flags = read_u16_as_i32 ctxt.is in 
     let name_idx = read_string_idx ctxt in 
     let typ_idx = read_tdor_idx ctxt in 
     (flags,name_idx,typ_idx) 
   
    (* Read Table PropertyMap *)
and seek_read_PropertyMap_row ctxt idx = ctxt.seek_read_PropertyMap_row idx
and seek_read_PropertyMap_row_uncached ctxtH idx =
     let ctxt = getH ctxtH in
       incr ctxt.count_PropertyMap;
       let addr = ctxt.row_addr tab_PropertyMap idx in 
       seek ctxt.is addr;
       let tidx = read_z_untagged_idx tab_TypeDef ctxt in 
       let props_idx = read_z_untagged_idx tab_Property ctxt in 
       (tidx,props_idx)

    (* Read Table Property *)
and seek_read_Property_row ctxt idx =
     incr ctxt.count_Property;
     let addr = ctxt.row_addr tab_Property idx in 
     seek ctxt.is addr;
     let flags = read_u16_as_i32 ctxt.is in 
     let name_idx = read_string_idx ctxt in 
     let typ_idx = read_blob_idx ctxt in 
     (flags,name_idx,typ_idx) 

    (* Read Table MethodSemantics *)
and seek_read_MethodSemantics_row ctxt idx = ctxt.seek_read_MethodSemantics_row idx
and seek_read_MethodSemantics_row_uncached ctxtH idx =
     let ctxt = getH ctxtH in
       incr ctxt.count_MethodSemantics;
       let addr = ctxt.row_addr tab_MethodSemantics idx in 
       seek ctxt.is addr;
       let flags = read_u16_as_i32 ctxt.is in 
       let midx = read_z_untagged_idx tab_Method ctxt in 
       let assoc_idx = read_hs_idx ctxt in 
       (flags,midx,assoc_idx)

    (* Read Table MethodImpl *)
and seek_read_MethodImpl_row ctxt idx =
     incr ctxt.count_MethodImpl;
     let addr = ctxt.row_addr tab_MethodImpl idx in 
     seek ctxt.is addr;
     let tidx = read_z_untagged_idx tab_TypeDef ctxt in 
     let mbody_idx = read_mdor_idx ctxt in 
     let mdecl_idx = read_mdor_idx ctxt in 
     (tidx,mbody_idx,mdecl_idx) 

    (* Read Table ModuleRef *)
and seek_read_ModuleRef_row ctxt idx =
     incr ctxt.count_ModuleRef;
     let addr = ctxt.row_addr tab_ModuleRef idx in 
     seek ctxt.is addr;
     let name_idx = read_string_idx ctxt in 
     name_idx  

    (* Read Table TypeSpec *)
and seek_read_TypeSpec_row ctxt idx =
     incr ctxt.count_TypeSpec;
     let addr = ctxt.row_addr tab_TypeSpec idx in 
     seek ctxt.is addr;
     let blob_idx = read_blob_idx ctxt in 
     blob_idx  

    (* Read Table ImplMap *)
and seek_read_ImplMap_row ctxt idx =
     incr ctxt.count_ImplMap;
     let addr = ctxt.row_addr tab_ImplMap idx in 
     seek ctxt.is addr;
     let flags = read_u16_as_i32 ctxt.is in 
     let forwrded_idx = read_mf_idx ctxt in 
     let name_idx = read_string_idx ctxt in 
     let scope_idx = read_z_untagged_idx tab_ModuleRef ctxt in 
     (flags, forwrded_idx, name_idx, scope_idx) 

    (* Read Table FieldRVA *)
and seek_read_FieldRVA_row ctxt idx =
     incr ctxt.count_FieldRVA;
     let addr = ctxt.row_addr tab_FieldRVA idx in 
     seek ctxt.is addr;
     let rva = read_i32 ctxt.is in 
     let fidx = read_z_untagged_idx tab_Field ctxt in 
     (rva,fidx) 

  (* Read Table Assembly *)
and seek_read_Assembly_row ctxt idx =
     incr ctxt.count_Assembly;
     let addr = ctxt.row_addr tab_Assembly idx in 
     seek ctxt.is addr;
     let hash = read_i32 ctxt.is in 
     let v1 = read_u16 ctxt.is in 
     let v2 = read_u16 ctxt.is in 
     let v3 = read_u16 ctxt.is in 
     let v4 = read_u16 ctxt.is in 
     let flags = read_i32 ctxt.is in 
     let public_key_idx = read_blob_idx ctxt in 
     let name_idx = read_string_idx ctxt in 
     let locale_idx = read_string_idx ctxt in 
     (hash,v1,v2,v3,v4,flags,public_key_idx, name_idx, locale_idx)

    (* Read Table AssemblyRef *)
and seek_read_AssemblyRef_row ctxt idx =
     incr ctxt.count_AssemblyRef;
     let addr = ctxt.row_addr tab_AssemblyRef idx in 
     seek ctxt.is addr;
     let v1 = read_u16 ctxt.is in 
     let v2 = read_u16 ctxt.is in 
     let v3 = read_u16 ctxt.is in 
     let v4 = read_u16 ctxt.is in 
     let flags = read_i32 ctxt.is in 
     let public_key_or_token_idx = read_blob_idx ctxt in 
     let name_idx = read_string_idx ctxt in 
     let locale_idx = read_string_idx ctxt in 
     let hash_value_idx = read_blob_idx ctxt in 
     (v1,v2,v3,v4,flags,public_key_or_token_idx, name_idx, locale_idx,hash_value_idx) 

       (* Read Table File *)
and seek_read_File_row ctxt idx =
     incr ctxt.count_File;
     let addr = ctxt.row_addr tab_File idx in 
     seek ctxt.is addr;
     let flags = read_i32 ctxt.is in 
     let name_idx = read_string_idx ctxt in 
     let hash_value_idx = read_blob_idx ctxt in 
     (flags, name_idx, hash_value_idx) 

       (* Read Table ExportedType *)
and seek_read_ExportedType_row ctxt idx =
     incr ctxt.count_ExportedType;
     let addr = ctxt.row_addr tab_ExportedType idx in 
     seek ctxt.is addr;
     let flags = read_i32 ctxt.is in 
     let tok = read_i32 ctxt.is in 
     let name_idx = read_string_idx ctxt in 
     let namespace_idx = read_string_idx ctxt in 
     let impl_idx = read_i_idx ctxt in 
     (flags,tok,name_idx,namespace_idx,impl_idx) 

       (* Read Table ManifestResource *)
and seek_read_ManifestResource_row ctxt idx =
     incr ctxt.count_ManifestResource;
     let addr = ctxt.row_addr tab_ManifestResource idx in 
     seek ctxt.is addr;
     let offset = read_i32 ctxt.is in 
     let flags = read_i32 ctxt.is in 
     let name_idx = read_string_idx ctxt in 
     let impl_idx = read_i_idx ctxt in 
     (offset,flags,name_idx,impl_idx) 

       (* Read Table Nested *) 
and seek_read_Nested_row ctxt idx = ctxt.seek_read_Nested_row idx
and seek_read_Nested_row_uncached ctxtH idx =
     let ctxt = getH ctxtH in
       incr ctxt.count_Nested;
       let addr = ctxt.row_addr tab_Nested idx in 
       seek ctxt.is addr;
       let nested_idx = read_z_untagged_idx tab_TypeDef ctxt in 
       let encl_idx = read_z_untagged_idx tab_TypeDef ctxt in 
       (nested_idx,encl_idx)

       (* Read Table GenericParam *)
and seek_read_GenericParam_row ctxt idx =
     incr ctxt.count_GenericParam;
     let addr = ctxt.row_addr tab_GenericParam idx in 
     seek ctxt.is addr;
     let seq = read_u16 ctxt.is in 
     let flags = read_u16 ctxt.is in 
     let owner_idx = read_tomd_idx ctxt in 
     let name_idx = read_string_idx ctxt in 
     (idx,seq,flags,owner_idx,name_idx) 

       (* Read Table GenericParamConstraint *)
and seek_read_GenericParamConstraint_row ctxt idx =
     incr ctxt.count_GenericParamConstraint;
     let addr = ctxt.row_addr tab_GenericParamConstraint idx in 
     seek ctxt.is addr;
     let pidx = read_z_untagged_idx tab_GenericParam ctxt in 
     let constraint_idx = read_tdor_idx ctxt in 
     (pidx,constraint_idx) 

       (* Read Table MethodSpec *)
and seek_read_MethodSpec_row ctxt idx =
     incr ctxt.count_MethodSpec;
     let addr = ctxt.row_addr tab_MethodSpec idx in 
     seek ctxt.is addr;
     let mdor_idx = read_mdor_idx ctxt in 
     let inst_idx = read_blob_idx ctxt in 
     (mdor_idx,inst_idx) 


  
 (*-----------------------------------------------------------------------
  * Read the AbsIL structure (lazily) by reading off the relevant rows.
  * ----------------------------------------------------------------------*)

and is_sorted ctxt tab = ((ctxt.sorted &&&& (!!!!1 <<<< tag_of_table tab)) <> !!!!0x0) 

and seek_read_Module ctxt (subsys,ilonly,only32,is_dll, align_virt,align_phys,image_base_real) idx =
     let (generation, name_idx, mvid_idx, encid_idx, encbaseid_idx) = seek_read_Module_row ctxt idx in 
     let mname = seek_read_string_heap ctxt name_idx in 
     let native_resources = read_native_resources ctxt in 

     { modulManifest =      
          if ctxt.nrows (tab_Assembly) > 0 then Some (seek_read_Assembly ctxt 1) 
          else None;
       modulCustomAttrs = seek_read_CustomAttrs ctxt (hca_Module,idx);
       modulName = mname;
       modulNativeResources=native_resources;
       modulTypeDefs = mk_lazy_tdefs (lazy (seek_read_top_TypeDefs ctxt ()));
       modulSubSystem = u16_to_i32 subsys;
       modulILonly = ilonly;
       modul32bit = only32;
       modulDLL=is_dll;

       modulVirtAlignment = align_virt;
       modulPhysAlignment = align_phys;
       modulImageBase = image_base_real;
       modulResources = seek_read_ManifestResources ctxt ();
       (* modulFixups = [] (* @todo:VIP *) *) }  

and seek_read_Assembly ctxt idx =
     let (hash,v1,v2,v3,v4,flags,public_key_idx, name_idx, locale_idx) = seek_read_Assembly_row ctxt idx in 
     let name = seek_read_string_heap ctxt name_idx in
     let pubkey = seek_read_blob_heap_option ctxt public_key_idx in 
     { manifestName= name; 
       manifestAuxModuleHashAlgorithm=hash;
       manifestSecurityDecls= seek_read_SecurityDecls ctxt (hds_Assembly,idx);
       manifestPublicKey= pubkey;  
       manifestVersion= Some (v1,v2,v3,v4);
       manifestLocale= seek_read_string_heap_option ctxt locale_idx;
       manifestCustomAttrs = seek_read_CustomAttrs ctxt (hca_Assembly,idx);
       manifestLongevity= 
         begin let masked = flags &&& !!!0x000e in 
           if masked = !!!0x0000 then LongevityUnspecified
           else if masked = !!!0x0002 then LongevityLibrary
           else if masked = !!!0x0004 then LongevityPlatformAppDomain
           else if masked = !!!0x0006 then LongevityPlatformProcess
           else if masked = !!!0x0008 then LongevityPlatformSystem
           else LongevityUnspecified
         end;
       manifestExportedTypes= seek_read_top_ExportedTypes ctxt ();
       manifestEntrypointElsewhere=(if fst ctxt.eptoken = tab_File then Some (seek_read_File ctxt (snd ctxt.eptoken)) else None);
       manifestDisableJitOptimizations= false;
       manifestJitTracking = false;
       } 
     
and seek_read_AssemblyRef ctxt idx = ctxt.seek_read_AssemblyRef idx
and seek_read_AssemblyRef_uncached ctxtH idx = let ctxt = getH ctxtH in
    let (v1,v2,v3,v4,flags,public_key_or_token_idx, name_idx, locale_idx,hash_value_idx) = seek_read_AssemblyRef_row ctxt idx in 
    let nm = seek_read_string_heap ctxt name_idx in 
    { assemRefName= nm;
      assemRefHash = seek_read_blob_heap_option ctxt hash_value_idx;
      assemRefPublicKeyInfo = 
      begin match seek_read_blob_heap_option ctxt public_key_or_token_idx with 
      | None -> None
      | Some blob -> Some (if (flags &&& !!!0x0001) <> !!!0x0 then PublicKey blob else PublicKeyToken blob)
      end;
      assemRefRetargetable =  (flags &&& !!!0x0100) <> !!!0x0;
      assemRefVersion=Some(v1,v2,v3,v4);
      assemRefLocale= seek_read_string_heap_option ctxt locale_idx; }

and seek_read_ModuleRef ctxt idx =
     let (name_idx) = seek_read_ModuleRef_row ctxt idx in 
     { modulRefName =  seek_read_string_heap ctxt name_idx;
       modulRefNoMetadata=false;
       modulRefHash=None }

and seek_read_File ctxt idx =
     let (flags, name_idx, hash_value_idx) = seek_read_File_row ctxt idx in 
     { modulRefName =  seek_read_string_heap ctxt name_idx;
       modulRefNoMetadata= (flags &&& !!!0x0001) <> !!!0x0;
       modulRefHash= seek_read_blob_heap_option ctxt hash_value_idx }

and seek_read_ClassLayout ctxt idx =
    match seek_read_optional_indexed_row (ctxt.nrows tab_ClassLayout,seek_read_ClassLayout_row ctxt,(fun (_,_,tidx) -> tidx),simpleindex_compare idx,is_sorted ctxt tab_ClassLayout,(fun (pack,size,_) -> pack,size)) with 
    | None -> { typeSize = None; typePack = None }
    | Some (pack,size) -> { typeSize = Some size; 
                           typePack = Some pack; }

and member_access_of_flags flags =
     let f = (flags &&& !!!0x00000007) in 
      if f = !!!0x00000001 then  MemAccess_private 
     else if f = !!!0x00000006 then  MemAccess_public 
     else if f = !!!0x00000004 then  MemAccess_family 
     else if f = !!!0x00000002 then  MemAccess_famandassem 
     else if f = !!!0x00000005 then  MemAccess_famorassem 
     else if f = !!!0x00000003 then  MemAccess_assembly 
     else MemAccess_compilercontrolled

and type_access_of_flags flags =
     let f = (flags &&& !!!0x00000007) in 
     if f = !!!0x00000001 then TypeAccess_public 
     else if f = !!!0x00000002 then TypeAccess_nested MemAccess_public 
     else if f = !!!0x00000003 then TypeAccess_nested MemAccess_private 
     else if f = !!!0x00000004 then TypeAccess_nested MemAccess_family 
     else if f = !!!0x00000006 then TypeAccess_nested MemAccess_famandassem 
     else if f = !!!0x00000007 then TypeAccess_nested MemAccess_famorassem 
     else if f = !!!0x00000005 then TypeAccess_nested MemAccess_assembly 
     else TypeAccess_private

and type_layout_of_flags ctxt flags tidx = 
     let f = (flags &&& !!!0x00000018) in 
     if f = !!!0x00000008 then TypeLayout_sequential (seek_read_ClassLayout ctxt tidx)
     else if f = !!!0x00000010 then  TypeLayout_explicit (seek_read_ClassLayout ctxt tidx)
     else TypeLayout_auto

and type_kind_of_flags nm mdefs fdefs super flags =
     if (flags &&& !!!0x00000020) <> !!!0x0 then TypeDef_interface 
     else 
          let is_enum = (match super with None -> false | Some ty -> tname_of_tspec (tspec_of_typ ty) = "System.Enum") in
          let is_delegate = (match super with None -> false | Some ty -> tname_of_tspec (tspec_of_typ ty) = "System.Delegate") in
          let is_multicast_delegate = (match super with None -> false | Some ty -> tname_of_tspec (tspec_of_typ ty) = "System.MulticastDelegate") in
          let self_is_multicast_delegate = nm = "System.MulticastDelegate" in
          let is_valuetype = (match super with None -> false | Some ty -> tname_of_tspec (tspec_of_typ ty) = "System.ValueType" && nm <> "System.Enum") in
          if is_enum then TypeDef_enum 
          else if  (is_delegate && not self_is_multicast_delegate) or is_multicast_delegate then TypeDef_delegate
          else if is_valuetype then TypeDef_valuetype 
          else TypeDef_class 

and type_encoding_of_flags flags = 
     let f = (flags &&& !!!0x00030000) in 
     if f = !!!0x00020000 then TypeEncoding_autochar 
     else if f = !!!0x00010000 then TypeEncoding_unicode 
     else TypeEncoding_ansi

and seek_is_wanted_TypeDef flags = true
       
and seek_is_top_TypeDef flags =
     (type_access_of_flags flags =  TypeAccess_private) or
     type_access_of_flags flags =  TypeAccess_public
       
and seek_is_top_TypeDef_idx ctxt idx =
    let (flags,_,_, _, _,_) = seek_read_TypeDef_row ctxt idx in 
    seek_is_top_TypeDef flags
       
and seek_read_blob_heap_as_split_type_name ctxt (name_idx,namespace_idx) = 
     let name = seek_read_string_heap ctxt name_idx in 
     let nspace = seek_read_string_heap_option ctxt namespace_idx in 
     (match nspace with Some nspace -> split_namespace_memoized nspace,name  | None -> [],name)

and seek_read_blob_heap_as_type_name ctxt (name_idx,namespace_idx) = 
     let name = seek_read_string_heap ctxt name_idx in 
     let nspace = seek_read_string_heap_option ctxt namespace_idx in 
     (match nspace with 
      | None -> name  
      | Some ns -> 
       (* ALLOC SORE POINT: Abstract IL stores type names in references in *)
       (* concatenated form, which means namespace names get repeated endlessly in various strings. *)
       ns^"."^name)

and seek_read_TypeDef_row_extents ctxt ((_,_, _, _, fields_idx, methods_idx) as info ) (idx:int) =
    if idx >= ctxt.nrows (tab_TypeDef) then 
      ctxt.nrows (tab_Field) + 1,
      ctxt.nrows (tab_Method) + 1
    else
      let (_, _, _, _, fields_idx, methods_idx) = seek_read_TypeDef_row ctxt (idx + 1) in 
      fields_idx, methods_idx 

and seek_read_TypeDef_row_with_extents ctxt (idx:int) =
     let info= seek_read_TypeDef_row ctxt idx in 
     info,seek_read_TypeDef_row_extents ctxt info idx

and seek_read_TypeDef ctxt toponly acc (idx:int) =
    let (flags,name_idx,namespace_idx, _, _, _) = seek_read_TypeDef_row ctxt idx in 
    if toponly && not (seek_is_top_TypeDef flags) then acc
    else if not (seek_is_wanted_TypeDef flags) then acc
    else
     let ns,n = seek_read_blob_heap_as_split_type_name ctxt (name_idx,namespace_idx) in 
     let cas = seek_read_CustomAttrs ctxt (hca_TypeDef,idx) in 

     let rest = 
       lazy
         begin 
           (* Re-read so as not to save all these in the lazy closure - this suspension ctxt.is the largest *)
           (* heavily allocated one in all of AbsIL*)
           let ((flags,name_idx,namespace_idx, extends_idx, fields_idx, methods_idx) as info) = seek_read_TypeDef_row ctxt idx in 
           let nm = seek_read_blob_heap_as_type_name ctxt (name_idx,namespace_idx) in 
           let cas = seek_read_CustomAttrs ctxt (hca_TypeDef,idx) in 

           if logging then dprint_endline ("reading remainder of type "^nm);       
           let (end_fields_idx, end_methods_idx) = seek_read_TypeDef_row_extents ctxt info idx in
           let typars = seek_read_GenericParams ctxt 0 (tomd_TypeDef,idx) in
           let numtypars = List.length typars in
           if logging then dprint_endline ("reading supertype of type "^nm);       
           let super = seek_read_optional_tdor ctxt numtypars AsObject extends_idx in
           let layout = type_layout_of_flags ctxt flags idx in 
           let has_layout = (match layout with TypeLayout_explicit _ -> true | _ -> false) in 
           if logging then dprint_endline ("setting up reading of methods and fields for type "^nm);       
           let mdefs = seek_read_Methods ctxt numtypars methods_idx end_methods_idx in 
           let fdefs = seek_read_Fields ctxt (numtypars,has_layout) fields_idx end_fields_idx in
           if logging then dprint_endline ("determining kind of type "^nm);        
           let kind = type_kind_of_flags nm mdefs fdefs super flags in 
           if logging then dprint_endline ("setting up read of nested types for type "^nm);        
           let nested = seek_read_nested_TypeDefs ctxt idx in 
           if logging then dprint_endline ("setting up read of interface impls for type "^nm);     
           let impls = seek_read_InterfaceImpls ctxt numtypars idx in
           if logging then dprint_endline ("setting up read of security decls for type "^nm);   
           let sdecls =  seek_read_SecurityDecls ctxt (hds_TypeDef,idx) in 
           if logging then dprint_endline ("setting up read of mimpls for type "^nm);   
           let mimpls = seek_read_MethodImpls ctxt numtypars idx in 
           if logging then dprint_endline ("setting up read of properties for type "^nm);       
           let props = seek_read_Properties ctxt numtypars idx in 
           if logging then dprint_endline ("setting up read of custom attributes for type "^nm);        
           let events = seek_read_Events ctxt numtypars idx in 
           if logging then dprint_endline ("preparing results for type "^nm);   
           let res = 
             { tdKind= kind;
               tdName=nm;
               tdGenericParams=typars; 
               tdAccess= type_access_of_flags flags;
               tdAbstract= (flags &&& !!!0x00000080) <> !!!0x0;
               tdSealed= (flags &&& !!!0x00000100) <> !!!0x0; 
               tdSerializable= (flags &&& !!!0x00002000) <> !!!0x0; 
               tdComInterop= (flags &&& !!!0x00001000) <> !!!0x0; 
               tdLayout = layout;
               tdSpecialName= (flags &&& !!!0x00000400) <> !!!0x0;
               tdEncoding=type_encoding_of_flags flags;
               tdNested= nested;
               tdImplements = impls;  
               tdExtends = super; 
               tdMethodDefs = mdefs; 
               tdSecurityDecls = sdecls;
               tdHasSecurity=(flags &&& !!!0x00040000) <> !!!0x0;
               tdFieldDefs=fdefs;
               tdMethodImpls=mimpls;
               tdInitSemantics=
               if kind = TypeDef_interface then TypeInit_beforeany
               else if (flags &&& !!!0x00100000) <> !!!0x0 then TypeInit_beforefield
               else TypeInit_beforeany; 
               tdEvents= events;
               tdProperties=props;
               tdCustomAttrs=cas; } in 
           if logging then dprint_endline ("done reading remainder of type "^nm);          
           res
         end in 
     (ns,n,cas,rest) :: acc

and seek_read_top_TypeDefs ctxt () =
    let res = ref [] in 
    for i = 1 to ctxt.nrows (tab_TypeDef) do
      res := seek_read_TypeDef ctxt true !res i;
    done;
    List.rev !res 

and seek_read_nested_TypeDefs ctxt tidx =
    mk_lazy_tdefs 
      (lazy 
         begin
           if logging then dprint_endline ("reading nested TypeDefs for tidx "^string_of_int tidx);        
           let nested_idxs = seek_read_indexed_rows (ctxt.nrows tab_Nested,seek_read_Nested_row ctxt,snd,simpleindex_compare tidx,false,fst) in 
           List.rev (List.fold_left (seek_read_TypeDef ctxt false) [] nested_idxs)
         end)

and seek_read_InterfaceImpls ctxt numtypars tidx =
    seek_read_indexed_rows (ctxt.nrows tab_InterfaceImpl,seek_read_InterfaceImpl_row ctxt,fst,simpleindex_compare tidx,is_sorted ctxt tab_InterfaceImpl,(snd >> seek_read_tdor ctxt numtypars AsObject (*ok*) [])) 

and seek_read_GenericParams ctxt numtypars (a,b) = ctxt.seek_read_GenericParams (GenericParamsIdx (numtypars,a,b))

and seek_read_GenericParams_uncached ctxtH (GenericParamsIdx (numtypars,a,b)) =
    let ctxt = getH ctxtH in 
    let pars =
      seek_read_indexed_rows
            (ctxt.nrows tab_GenericParam,seek_read_GenericParam_row ctxt,
             (fun (_,_,_,tomd,_) -> tomd),
             tomd_compare (a,b),
             is_sorted ctxt tab_GenericParam,
             (fun (gpidx,seq,flags,_,name_idx) -> 
             let flags = u16_to_i32 flags in
             let variance_flags = flags &&& !!!0x0003 in
             let variance = 
                 if variance_flags = !!!0x0000 then NonVariant
                 else if variance_flags = !!!0x0001 then CoVariant
                 else if variance_flags = !!!0x0002 then ContraVariant 
                 else NonVariant in 
                 let constraints = seek_read_GenericParamConstraints_uncached ctxt numtypars gpidx in
                 seq, {gpName=seek_read_string_heap ctxt name_idx;
                   gpConstraints=constraints;
                   gpVariance=variance;
                   gpReferenceTypeConstraint= (flags &&& !!!0x0004) <> 0l;
                   gpNotNullableValueTypeConstraint= (flags &&& !!!0x0008) <> 0l;
                   gpDefaultConstructorConstraint=(flags &&& !!!0x0010) <> 0l; })) in 
    (* (*F# printf "--------------\n"; F#*) *)
    (* (*F# List.iter (fun (seq,_) -> printf "eq = %O\n" seq) pars; F#*) *)
    List.map snd (List.sort (fun (a1,_) (a2,_) -> compare a1 a2) pars)

and seek_read_GenericParamConstraints_uncached ctxt numtypars gpidx =
    seek_read_indexed_rows (ctxt.nrows tab_GenericParamConstraint,
                            seek_read_GenericParamConstraint_row ctxt,
                            fst,
                            simpleindex_compare gpidx,
                            is_sorted ctxt tab_GenericParamConstraint,
                            (snd >>  seek_read_tdor ctxt numtypars AsObject (*ok*) []))

and seek_read_TypeDef_as_typ ctxt boxity ginst idx =
      ctxt.seek_read_TypeDef_as_typ (TypeDefAsTypIdx (boxity,ginst,idx))

and seek_read_TypeDef_as_typ_uncached ctxtH (TypeDefAsTypIdx (boxity,ginst,idx)) =
    let ctxt = getH ctxtH in 
    mk_typ boxity { tspecTypeRef=seek_read_TypeDef_as_tref ctxt idx;
                    tspecInst=ginst}

and seek_read_TypeDef_as_tref ctxt idx =
     if logging then dprint_endline ("reading TypeDef "^string_of_int idx^" as type ref"); 
     let enc = 
       if seek_is_top_TypeDef_idx ctxt idx then [] 
       else 
         let encl_idx = seek_read_indexed_row (ctxt.nrows tab_Nested,seek_read_Nested_row ctxt,fst,simpleindex_compare idx,is_sorted ctxt tab_Nested,snd) in
         let tref = seek_read_TypeDef_as_tref ctxt encl_idx in 
         enclosing_tnames_of_tref tref@[tname_of_tref tref] in 
     let (_, name_idx, namespace_idx, _, _, _) = seek_read_TypeDef_row ctxt idx in 
     { trefScope=ScopeRef_local;
       trefNested=enc;
       trefName=seek_read_blob_heap_as_type_name ctxt (name_idx,namespace_idx) }

and seek_read_TypeRef ctxt idx = ctxt.seek_read_TypeRef idx
and seek_read_TypeRef_uncached ctxtH idx =
     let ctxt = getH ctxtH in 
     if logging then dprint_endline ("reading TypeRef "^string_of_int idx); 
     let scope_idx,name_idx,namespace_idx = seek_read_TypeRef_row ctxt idx in 
     let scope,enc = seek_read_rs ctxt scope_idx in 
     let nm = seek_read_blob_heap_as_type_name ctxt (name_idx,namespace_idx) in
(*     if nm = "System.Object" then 
       (Ilprint.output_scoref stderr scope; dprint_endline ""); *)
     { trefScope=scope;
       trefNested=enc;
       trefName=nm }

and seek_read_TypeRef_as_typ ctxt boxity ginst idx = ctxt.seek_read_TypeRef_as_typ (TypeRefAsTypIdx (boxity,ginst,idx))
and seek_read_TypeRef_as_typ_uncached ctxtH (TypeRefAsTypIdx (boxity,ginst,idx)) =
     let ctxt = getH ctxtH in 
     mk_typ boxity { tspecTypeRef=seek_read_TypeRef ctxt idx;
                     tspecInst=ginst}

and seek_read_tdor ctxt numtypars boxity ginst (tag,idx) =
     match tag with 
    | tag when tag = tdor_TypeDef -> seek_read_TypeDef_as_typ ctxt boxity ginst idx
    | tag when tag = tdor_TypeRef -> seek_read_TypeRef_as_typ ctxt boxity ginst idx
    | tag when tag = tdor_TypeSpec -> 
        if ginst <> [] then dprint_endline ("type spec used as type constructor for a generic instantiation: ignoring instantiation");
        seek_read_blob_heap_as_typ ctxt numtypars (seek_read_TypeSpec_row ctxt idx)
    | _ -> failwith "seek_read_tdor ctxt"

and seek_read_tdor_as_tref ctxt (tag,idx) =
     match tag with 
    | tag when tag = tdor_TypeDef -> seek_read_TypeDef_as_tref ctxt idx
    | tag when tag = tdor_TypeRef -> seek_read_TypeRef ctxt idx
    | tag when tag = tdor_TypeSpec -> 
        dprint_endline ("type spec used where a type ref or def ctxt.is required");
        ctxt.ilg.tref_Object
    | _ -> failwith "seek_read_tdor_as_tref_read_tdor"

and seek_read_mrp ctxt numtypars (tag,idx) =
     match tag with 
    | tag when tag = mrp_TypeRef -> seek_read_TypeRef_as_typ ctxt AsObject (* not ok - no way to tell if a member ref parent ctxt.is a value type or not *) [] idx
    | tag when tag = mrp_ModuleRef -> typ_for_toplevel (ScopeRef_module (seek_read_ModuleRef ctxt idx))
    | tag when tag = mrp_MethodDef -> 
        let mspec = mk_mspec_in_typ(seek_read_MethodDef_as_mdata ctxt idx) in
        enclosing_typ_of_mspec mspec
    | tag when tag = mrp_TypeSpec -> seek_read_blob_heap_as_typ ctxt numtypars (seek_read_TypeSpec_row ctxt idx)
    | _ -> failwith "seek_read_mrp ctxt"

and seek_read_mdor ctxt numtypars (tag,idx) =
     match tag with 
    | tag when tag = mdor_MethodDef -> 
        let (encl_typ, cc, nm, argtys, retty,minst) = seek_read_MethodDef_as_mdata ctxt idx in 
        (encl_typ, cc, nm, argtys, None,retty,minst)
    | tag when tag = mdor_MemberRef -> seek_read_MemberRef_as_mdata ctxt numtypars idx
    | _ -> failwith "seek_read_mdor ctxt"

and seek_read_mdor_no_varargs ctxt numtypars x =
     let (encl_typ, cc, nm, argtys, varargs,retty,minst)=     seek_read_mdor ctxt numtypars x  in
     if varargs <> None then dprintf0 "ignoring sentinel and varargs in MethodDef token signature";
     (encl_typ, cc, nm, argtys, retty,minst)

and seek_read_cat ctxt (tag,idx) =
     match tag with 
    | tag when tag = cat_MethodDef -> (* intern_mspec manager *) (mk_mspec_in_typ (seek_read_MethodDef_as_mdata ctxt idx))
    | tag when tag = cat_MemberRef -> (* intern_mspec manager *) (mk_mspec_in_typ (seek_read_MemberRef_as_mdata_no_varargs ctxt 0 idx))
    | _ -> failwith "seek_read_cat ctxt"
    
and seek_read_i_as_scoref ctxt (tag,idx) =
     if idx = 0 then ScopeRef_local
     else 
       match tag with 
       | tag when tag = i_File -> ScopeRef_module (seek_read_File ctxt idx)
       | tag when tag = i_AssemblyRef -> ScopeRef_assembly (seek_read_AssemblyRef ctxt idx)
       | tag when tag = i_ExportedType -> failwith "seek_read_i_as_scoref ctxt"
       | _ -> failwith "seek_read_i_as_scoref ctxt"

and seek_read_rs ctxt (tag,idx) =
    match tag with 
    | tag when tag = rs_Module -> ScopeRef_local,[]
    | tag when tag = rs_ModuleRef -> ScopeRef_module (seek_read_ModuleRef ctxt idx),[]
    | tag when tag = rs_AssemblyRef -> ScopeRef_assembly (seek_read_AssemblyRef ctxt idx),[]
    | tag when tag = rs_TypeRef -> 
        let tref = seek_read_TypeRef ctxt idx in 
        scoref_of_tref tref,(enclosing_tnames_of_tref tref@[tname_of_tref tref])
    | _ -> failwith "seek_read_rs ctxt"

and seek_read_optional_tdor ctxt numtypars boxity idx = 
      if idx = (tdor_TypeDef, 0) then None
      else Some (seek_read_tdor ctxt numtypars boxity [] idx)

and seek_is_wanted_FieldDef idx = true

and seek_read_Field ctxt (numtypars, has_layout) (idx:int) =
     let (flags,name_idx,type_idx) = seek_read_Field_row ctxt idx in
     let nm = seek_read_string_heap ctxt name_idx in
     if logging then dprint_endline ("reading field "^nm);         
     let isStatic = (flags &&& !!!0x0010) <> 0l in 
     let fd = 
       { fdName = nm;
         fdType= seek_read_blob_heap_as_field_sig ctxt numtypars type_idx;
         fdAccess = member_access_of_flags flags;
         fdStatic = isStatic;
         fdInitOnly = (flags &&& !!!0x0020) <> 0l;
         fdLiteral = (flags &&& !!!0x0040) <> 0l;
         fdNotSerialized = (flags &&& !!!0x0080) <> 0l;
         fdSpecialName = (flags &&& !!!0x0200) <> 0l or (flags &&& !!!0x0400) <> 0l; (* @todo: RTSpecialName *)
         fdInit = if (flags &&& !!!0x8000) = 0l then None else Some (seek_read_Constant ctxt (hc_FieldDef,idx));
         fdMarshal = if (flags &&& !!!0x1000) = 0l then None else Some (seek_read_indexed_row (ctxt.nrows tab_FieldMarshal,seek_read_FieldMarshal_row ctxt,fst,hfm_compare (hfm_FieldDef,idx),is_sorted ctxt tab_FieldMarshal,(snd >> seek_read_blob_heap_as_native_type ctxt)));
         fdData = 
         begin 
           if (flags &&& !!!0x0100) = 0l then None 
           else 
             let rva = seek_read_indexed_row (ctxt.nrows tab_FieldRVA,seek_read_FieldRVA_row ctxt,snd,simpleindex_compare idx,is_sorted ctxt tab_FieldRVA,fst)  in 
             Some (rva_to_data ctxt "field" rva)
         end;
         fdOffset = if has_layout && not isStatic then Some (seek_read_indexed_row (ctxt.nrows tab_FieldLayout,seek_read_FieldLayout_row ctxt,snd,simpleindex_compare idx,is_sorted ctxt tab_FieldLayout,fst)) else None; 
         fdCustomAttrs=seek_read_CustomAttrs ctxt (hca_FieldDef,idx); } in
     if logging then dprint_endline ("done reading field "^nm);    
     fd
     
and seek_read_Fields ctxt (numtypars, has_layout) fidx1 fidx2 =
    mk_lazy_fdefs 
      (lazy 
         begin 
           if logging then dprint_endline ("reading fields "^string_of_int fidx1^" to "^string_of_int fidx2);      
           let res = ref [] in
           for i = fidx1 to fidx2 - 1 do
             if seek_is_wanted_FieldDef i then 
               res := seek_read_Field ctxt (numtypars, has_layout) i :: !res;
           done;
           let res2 = List.rev !res in 
           if logging then dprint_endline ("done reading fields "^string_of_int fidx1^" to "^string_of_int fidx2);         
           res2
         end)

and seek_read_Methods ctxt numtypars midx1 midx2 =
    mk_lazy_mdefs 
      (lazy 
         begin
           if logging then dprint_endline ("reading methods "^string_of_int midx1^" to "^string_of_int midx2);     
           let res = ref [] in
           for i = midx1 to midx2 - 1 do
             if seek_is_wanted_MethodDef i then 
               res := seek_read_Method ctxt numtypars i :: !res;
           done;
           List.rev !res
         end)

and sigptr_get_tdor_idx bytes sigptr = 
     let n, sigptr = sigptr_get_z_i32 bytes sigptr in 
     if (n &&& !!!0x01) = !!!0x0 then (* Type Def *)
       (tdor_TypeDef, Int32.to_int (n lsr 2)), sigptr
     else (* Type Ref *)
       (tdor_TypeRef, Int32.to_int (n lsr 2)), sigptr
         

and sigptr_get_typ ctxt numtypars bytes sigptr = 
     let b0,sigptr = sigptr_get_byte bytes sigptr in 
     if logging then dprint_endline ("reading type from sig at "^string_of_int sigptr^", et = "^string_of_int b0); 
     if b0 = et_OBJECT then ctxt.ilg.typ_Object , sigptr
     else if b0 = et_STRING then ctxt.ilg.typ_String, sigptr
     else if b0 = et_I1 then ctxt.ilg.typ_int8, sigptr
     else if b0 = et_I2 then ctxt.ilg.typ_int16, sigptr
     else if b0 = et_I4 then ctxt.ilg.typ_int32, sigptr
     else if b0 = et_I8 then ctxt.ilg.typ_int64, sigptr
     else if b0 = et_I then ctxt.ilg.typ_int, sigptr
     else if b0 = et_U1 then ctxt.ilg.typ_uint8, sigptr
     else if b0 = et_U2 then ctxt.ilg.typ_uint16, sigptr
     else if b0 = et_U4 then ctxt.ilg.typ_uint32, sigptr
     else if b0 = et_U8 then ctxt.ilg.typ_uint64, sigptr
     else if b0 = et_U then ctxt.ilg.typ_uint, sigptr
     else if b0 = et_R4 then ctxt.ilg.typ_float32, sigptr
     else if b0 = et_R8 then ctxt.ilg.typ_float64, sigptr
     else if b0 = et_CHAR then ctxt.ilg.typ_char, sigptr
     else if b0 = et_BOOLEAN then ctxt.ilg.typ_bool, sigptr
     else if b0 = et_WITH then 
       let b0,sigptr = sigptr_get_byte bytes sigptr in 
       let tdor_idx, sigptr = sigptr_get_tdor_idx bytes sigptr in 
       let n, sigptr = sigptr_get_z_i32 bytes sigptr in 
       let argtys,sigptr = sigptr_foldi (sigptr_get_typ ctxt numtypars) (Int32.to_int n) bytes sigptr in 
       seek_read_tdor ctxt numtypars (if b0 = et_CLASS then AsObject else AsValue) (List.map mk_genactual argtys) tdor_idx,
       sigptr
         
     else if b0 = et_CLASS then 
       let tdor_idx, sigptr = sigptr_get_tdor_idx bytes sigptr in 
         seek_read_tdor ctxt numtypars AsObject [] tdor_idx, sigptr
     else if b0 = et_VALUETYPE then 
       let tdor_idx, sigptr = sigptr_get_tdor_idx bytes sigptr in 
       seek_read_tdor ctxt numtypars AsValue [] tdor_idx, sigptr
     else if b0 = et_VAR then 
       let n, sigptr = sigptr_get_z_i32 bytes sigptr in 
       Type_tyvar (i32_to_u16 n),sigptr
     else if b0 = et_MVAR then 
       let n, sigptr = sigptr_get_z_i32 bytes sigptr in 
       Type_tyvar (int_to_u16 (i32_to_int n + numtypars)), sigptr
     else if b0 = et_BYREF then 
       let typ, sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
       Type_byref typ, sigptr
     else if b0 = et_PTR then 
       let typ, sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
       Type_ptr typ, sigptr
     else if b0 = et_SZARRAY then 
       let typ, sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
       mk_sdarray_ty typ, sigptr
     else if b0 = et_ARRAY then
       let typ, sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
       let rank, sigptr = sigptr_get_z_i32 bytes sigptr in 
       let num_sized, sigptr = sigptr_get_z_i32 bytes sigptr in 
       let sizes, sigptr = sigptr_foldi sigptr_get_z_i32 (Int32.to_int num_sized) bytes sigptr in 
       let num_lobounded, sigptr = sigptr_get_z_i32 bytes sigptr in 
       let lobounds, sigptr = sigptr_foldi sigptr_get_z_i32 (Int32.to_int num_lobounded) bytes sigptr in 
       let shape = 
         let dim i =
           (if i < Int32.to_int num_lobounded then Some (List.nth lobounds i) else None),
           (if i < Int32.to_int num_sized then Some (List.nth sizes i) else None) in 
         ArrayShape (Array.to_list (Array.init (Int32.to_int rank) dim)) in
       mk_array_ty (typ, shape), sigptr
         
     else if b0 = et_VOID then Type_void, sigptr
     else if b0 = et_TYPEDBYREF then ctxt.ilg.typ_TypedReference, sigptr
     else if b0 = et_CMOD_REQD or b0 = et_CMOD_OPT  then 
       let tdor_idx, sigptr = sigptr_get_tdor_idx bytes sigptr in 
       let typ, sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
       Type_modified((b0 = et_CMOD_REQD), seek_read_tdor_as_tref ctxt tdor_idx, typ), sigptr
     else if b0 = et_FNPTR then
       begin
         if logging then dprint_endline ("reading fptr sig "); 
         let cc_byte,sigptr = sigptr_get_byte bytes sigptr in 
         let generic,cc = byte_as_callconv cc_byte in 
         if generic then failwith "fptr sig may not be generic";
         let numparams,sigptr = sigptr_get_z_i32 bytes sigptr in 
         let retty,sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
         let argtys,sigptr = sigptr_foldi (sigptr_get_typ ctxt numtypars) (Int32.to_int numparams) bytes sigptr in 
         Type_fptr
           { callsigCallconv=cc;
             callsigArgs=argtys;
             callsigReturn=retty }
           ,sigptr
       end 
     else if b0 = et_SENTINEL then failwith "varargs NYI"
     else Type_void , sigptr
         
and sigptr_get_vararg_typs ctxt n numtypars bytes sigptr = 
     sigptr_foldi (sigptr_get_typ ctxt numtypars) n bytes sigptr 

and sigptr_get_arg_typs ctxt n numtypars bytes sigptr acc = 
     if n <= 0 then (List.rev acc,None),sigptr 
     else
       let b0,sigptr2 = sigptr_get_byte bytes sigptr in 
       if b0 = et_SENTINEL then 
         let varargs,sigptr = sigptr_get_vararg_typs ctxt n numtypars bytes sigptr2 in 
         (List.rev acc,Some(varargs)),sigptr
       else
         let x,sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
         sigptr_get_arg_typs ctxt (n-1) numtypars bytes sigptr (x::acc)
         
and sigptr_get_local ctxt numtypars bytes sigptr = 
     let pinned,sigptr = 
       let b0, sigptr' = sigptr_get_byte bytes sigptr in 
       if b0 = et_PINNED then 
         true, sigptr'
       else 
         false, sigptr in 
     let typ, sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
     { localPinned = pinned;
       localType = typ }, sigptr
         
and seek_read_blob_heap_as_method_sig ctxt numtypars blob_idx  =
    ctxt.seek_read_blob_heap_as_method_sig (BlobAsMethodSigIdx (numtypars,blob_idx))
and seek_read_blob_heap_as_method_sig_uncached ctxtH (BlobAsMethodSigIdx (numtypars,blob_idx)) =
     let ctxt = getH ctxtH in 
     if logging then dprint_endline ("reading method sig at "^Int32.to_string blob_idx); 
    let bytes = seek_read_blob_heap ctxt blob_idx in 
    let sigptr = 0 in 
    let cc_byte,sigptr = sigptr_get_byte bytes sigptr in 
    let generic,cc = byte_as_callconv cc_byte in 
    let genarity,sigptr = if generic then sigptr_get_z_i32 bytes sigptr else !!!0x0,sigptr in 
    let numparams,sigptr = sigptr_get_z_i32 bytes sigptr in 
    let retty,sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
    let (argtys,varargs),sigptr = sigptr_get_arg_typs ctxt  (Int32.to_int numparams) numtypars bytes sigptr [] in 
    generic,genarity,cc,retty,argtys,varargs
      
and seek_read_blob_heap_as_typ ctxt numtypars blob_idx = 
    let bytes = seek_read_blob_heap ctxt blob_idx in 
    let ty,sigptr = sigptr_get_typ ctxt numtypars bytes 0 in 
    ty

and seek_read_blob_heap_as_field_sig ctxt numtypars blob_idx  =
    ctxt.seek_read_blob_heap_as_field_sig (BlobAsFieldSigIdx (numtypars,blob_idx))
and seek_read_blob_heap_as_field_sig_uncached ctxtH (BlobAsFieldSigIdx (numtypars,blob_idx)) =
     let ctxt = getH ctxtH in 
     if logging then dprint_endline ("reading field sig at "^Int32.to_string blob_idx); 
    let bytes = seek_read_blob_heap ctxt blob_idx in 
    let sigptr = 0 in 
    let cc_byte,sigptr = sigptr_get_byte bytes sigptr in 
    if cc_byte <> e_IMAGE_CEE_CS_CALLCONV_FIELD then dprint_endline "warning: field sig was not CC_FIELD";
    let retty,sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
    retty

      
and seek_read_blob_heap_as_property_sig ctxt numtypars blob_idx  =
    ctxt.seek_read_blob_heap_as_property_sig (BlobAsPropSigIdx (numtypars,blob_idx))
and seek_read_blob_heap_as_property_sig_uncached ctxtH (BlobAsPropSigIdx (numtypars,blob_idx))  =
     let ctxt = getH ctxtH in 
    let bytes = seek_read_blob_heap ctxt blob_idx in 
    let sigptr = 0 in 
    let cc_byte,sigptr = sigptr_get_byte bytes sigptr in 
    let hasthis = byte_as_hasthis cc_byte in 
    let cc_masked = (cc_byte land 0x0f) in 
    if cc_masked <> e_IMAGE_CEE_CS_CALLCONV_PROPERTY then dprint_endline ("warning: property sig was "^string_of_int cc_masked^" instead of CC_PROPERTY");
    let numparams,sigptr = sigptr_get_z_i32 bytes sigptr in 
    let retty,sigptr = sigptr_get_typ ctxt numtypars bytes sigptr in 
    let argtys,sigptr = sigptr_foldi (sigptr_get_typ ctxt numtypars) (Int32.to_int numparams) bytes sigptr in 
    hasthis,retty,argtys
      
and seek_read_blob_heap_as_locals_sig ctxt numtypars blob_idx  =
    ctxt.seek_read_blob_heap_as_locals_sig (BlobAsLocalSigIdx (numtypars,blob_idx))
and seek_read_blob_heap_as_locals_sig_uncached ctxtH (BlobAsLocalSigIdx (numtypars,blob_idx)) =
     let ctxt = getH ctxtH in 
    let bytes = seek_read_blob_heap ctxt blob_idx in 
    let sigptr = 0 in 
    let cc_byte,sigptr = sigptr_get_byte bytes sigptr in 
    if cc_byte <> e_IMAGE_CEE_CS_CALLCONV_LOCAL_SIG then dprint_endline "warning: local sig was not CC_LOCAL";
    let numlocals,sigptr = sigptr_get_z_i32 bytes sigptr in 
    let localtys,sigptr = sigptr_foldi (sigptr_get_local ctxt numtypars) (Int32.to_int numlocals) bytes sigptr in 
    localtys
      
and byte_as_hasthis b = 
    let hasthis_masked = (b land 0x60) in
    if hasthis_masked = e_IMAGE_CEE_CS_CALLCONV_INSTANCE then CC_instance
    else if hasthis_masked = e_IMAGE_CEE_CS_CALLCONV_INSTANCE_EXPLICIT then CC_instance_explicit 
    else CC_static 

and byte_as_callconv b = 
    let cc = 
      let cc_masked = (b land 0x0f) in 
      if cc_masked =  e_IMAGE_CEE_CS_CALLCONV_FASTCALL then CC_fastcall 
      else if cc_masked = e_IMAGE_CEE_CS_CALLCONV_STDCALL then CC_stdcall 
      else if cc_masked = e_IMAGE_CEE_CS_CALLCONV_THISCALL then CC_thiscall 
      else if cc_masked = e_IMAGE_CEE_CS_CALLCONV_CDECL then CC_cdecl 
      else if cc_masked = e_IMAGE_CEE_CS_CALLCONV_VARARG then CC_vararg 
      else  CC_default in
    let generic = (b land e_IMAGE_CEE_CS_CALLCONV_GENERIC) <> 0x0 in 
    generic, Callconv (byte_as_hasthis b,cc) 
      
   and seek_read_MemberRef_as_mdata ctxt numtypars idx = 
      ctxt.seek_read_MemberRef_as_mdata (MemberRefAsMspecIdx (numtypars,idx))
   and seek_read_MemberRef_as_mdata_uncached ctxtH (MemberRefAsMspecIdx (numtypars,idx)) = 
     let ctxt = getH ctxtH in 
     let (mrp_idx,name_idx,type_idx) = seek_read_MemberRef_row ctxt idx in
     let nm = seek_read_string_heap ctxt name_idx in
     let encl_typ = seek_read_mrp ctxt numtypars mrp_idx in
     let generic,genarity,cc,retty,argtys,varargs = seek_read_blob_heap_as_method_sig ctxt (List.length (inst_of_typ encl_typ)) type_idx in 
     let minst =  collecti (fun n -> mk_tyvar_ty (int_to_u16 (numtypars+n))) (Int32.to_int genarity) in 
     (encl_typ, cc, nm, argtys, varargs,retty,minst)

   and seek_read_MemberRef_as_mdata_no_varargs ctxt numtypars idx =
     let (encl_typ, cc, nm, argtys,varargs, retty,minst) =  seek_read_MemberRef_as_mdata ctxt numtypars idx in 
     if varargs <> None then dprintf0 "ignoring sentinel and varargs in MethodDef token signature";
     (encl_typ, cc, nm, argtys, retty,minst)

   and seek_read_MethodSpec_as_mdata ctxt numtypars idx =  
      ctxt.seek_read_MethodSpec_as_mdata (MethodSpecAsMspecIdx (numtypars,idx))
   and seek_read_MethodSpec_as_mdata_uncached ctxtH (MethodSpecAsMspecIdx (numtypars,idx)) = 
     let ctxt = getH ctxtH in 
     let (mdor_idx,inst_idx) = seek_read_MethodSpec_row ctxt idx in
     let (encl_typ, cc, nm, argtys, varargs,retty,_) = seek_read_mdor ctxt numtypars mdor_idx in 
     let minst = 
       let bytes = seek_read_blob_heap ctxt inst_idx in 
       let sigptr = 0 in 
       let cc_byte,sigptr = sigptr_get_byte bytes sigptr in 
       if logging then dprint_endline (ctxt.infile ^ ": seek_read_MethodSpec_as_mdata_uncached ctxt, cc_byte = "^string_of_int cc_byte); 
       if cc_byte <> e_IMAGE_CEE_CS_CALLCONV_GENERICINST then dprint_endline ("warning: method inst callconv was "^string_of_int cc_byte^" instead of CC_GENERICINST");
       let numgpars,sigptr = sigptr_get_z_i32 bytes sigptr in 
       let argtys,sigptr = sigptr_foldi (sigptr_get_typ ctxt numtypars) (Int32.to_int numgpars) bytes sigptr in 
       argtys in 
     (encl_typ, cc, nm, argtys, varargs,retty, minst)

   and seek_read_MemberRef_as_fspec ctxt numtypars idx = 
     ctxt.seek_read_MemberRef_as_fspec (MemberRefAsFspecIdx (numtypars,idx))
   and seek_read_MemberRef_as_fspec_uncached ctxtH (MemberRefAsFspecIdx (numtypars,idx)) = 
     let ctxt = getH ctxtH in 
     let (mrp_idx,name_idx,type_idx) = seek_read_MemberRef_row ctxt idx in
     let nm = seek_read_string_heap ctxt name_idx in
     let encl_typ = seek_read_mrp ctxt numtypars mrp_idx in
     let retty = seek_read_blob_heap_as_field_sig ctxt numtypars type_idx in 
     mk_fspec_in_typ(encl_typ, nm, retty)

   (* One extremely annoying aspect of the MD format is that given a *)
   (* MethodDef token it is non-trivial to find which TypeDef it belongs *)
   (* to.  So we do a binary chop through the TypeDef table *)
   (* looking for which TypeDef has the MethodDef within its range.  *)
   (* Although the TypeDef table is not "sorted", it is effectively sorted by *)
   (* method-range and field-range start/finish indexes  *)
   and seek_read_MethodDef_as_mdata ctxt idx =
     ctxt.seek_read_MethodDef_as_mdata idx
   and seek_read_MethodDef_as_mdata_uncached ctxtH idx =
     let ctxt = getH ctxtH in 
     let (code_rva, implflags, flags, name_idx, type_idx, param_idx) = seek_read_Method_row ctxt idx in
     let nm = seek_read_string_heap ctxt name_idx in
     (* Look for the method def parent. *)
     let tidx = 
       seek_read_indexed_row (ctxt.nrows 
                              tab_TypeDef,
                               (fun i -> i, seek_read_TypeDef_row_with_extents ctxt i),
                               (fun r -> r),
                               (fun (_,((_, _, _, _, _, methods_idx),
                                        (_, end_methods_idx)))  -> 
                                          if end_methods_idx <= idx then 1 
                                          else if methods_idx <= idx && idx < end_methods_idx then 0 
                                          else -1),
                               true,fst) in 
     (* Read the method def signature. *)
     let generic,genarity,cc,retty,argtys,varargs = seek_read_blob_heap_as_method_sig ctxt 0 type_idx in 
     if varargs <> None then dprintf0 "ignoring sentinel and varargs in MethodDef token signature";
     (* Create a formal instantiation if needed *)
     let finst = generalize_gparams (seek_read_GenericParams ctxt 0 (tomd_TypeDef,tidx)) in
     let minst = generalize_gparams (seek_read_GenericParams ctxt (List.length finst) (tomd_MethodDef,idx)) in
     (* Read the method def parent. *)
     let encl_typ = seek_read_TypeDef_as_typ ctxt AsObject (* not ok: see note *) finst tidx in 
     (* Return the constituent parts: put it together at the place where this is called. *)
     (encl_typ, cc, nm, argtys, retty,minst)


   (* Similarly for fields. *)
   and seek_read_FieldDef_as_fspec ctxt idx =
     ctxt.seek_read_FieldDef_as_fspec idx
   and seek_read_FieldDef_as_fspec_uncached ctxtH idx =
     let ctxt = getH ctxtH in 
     let (flags, name_idx, type_idx) = seek_read_Field_row ctxt idx in
     let nm = seek_read_string_heap ctxt name_idx in
     (* Look for the field def parent. *)
     let tidx = 
       seek_read_indexed_row (ctxt.nrows 
                              tab_TypeDef,
                               (fun i -> i, seek_read_TypeDef_row_with_extents ctxt i),
                               (fun r -> r),
                               (fun (_,((_, _, _, _, fields_idx, _),
                                        (end_fields_idx, _)))  -> 
                                          if end_fields_idx <= idx then 1 
                                          else if fields_idx <= idx && idx < end_fields_idx then 0 
                                          else -1),
                               true,fst) in 
     (* Read the field signature. *)
     let retty = seek_read_blob_heap_as_field_sig ctxt 0 type_idx in 
     (* Create a formal instantiation if needed *)
     let finst = generalize_gparams (seek_read_GenericParams ctxt 0 (tomd_TypeDef,tidx)) in
     (* Read the field def parent. *)
     let encl_typ = seek_read_TypeDef_as_typ ctxt AsObject (* not ok: see note *) finst tidx in 
     (* Put it together. *)
     mk_fspec_in_typ(encl_typ, nm, retty)

   and seek_is_wanted_MethodDef idx = true

   and seek_read_Method ctxt numtypars (idx:int) =
     if logging then dprint_endline ("reading method "^string_of_int idx); 
     let (code_rva, implflags, flags, name_idx, type_idx, param_idx) = seek_read_Method_row ctxt idx in
     let nm = seek_read_string_heap ctxt name_idx in
     if logging then dprint_endline ("  method name = " ^ nm); 
     let isStatic = (flags &&& !!!0x0010) <> !!!0x0 in 
     let final = (flags &&& !!!0x0020) <> !!!0x0 in 
     let virt = (flags &&& !!!0x0040) <> !!!0x0 in 
     let strict = (flags &&& !!!0x0200) <> !!!0x0 in 
     let hidebysig = (flags &&& !!!0x0080) <> !!!0x0 in 
     let newslot = (flags &&& !!!0x0100) <> !!!0x0 in 
     let abstr = (flags &&& !!!0x0400) <> !!!0x0 in 
     let specialname = (flags &&& !!!0x0800) <> !!!0x0 in 
     let pinvoke = (flags &&& !!!0x2000) <> !!!0x0 in 
     let export = (flags &&& !!!0x0008) <> !!!0x0 in 
     let rtspecialname = (flags &&& !!!0x1000) <> !!!0x0 in 
     let reqsecobj = (flags &&& !!!0x8000) <> !!!0x0 in 
     let hassec = (flags &&& !!!0x4000) <> !!!0x0 in 
     let codetype = implflags &&& !!!0x0003 in 
     let unmanaged = (implflags &&& !!!0x0004) <> !!!0x0 in 
     let forwardref = (implflags &&& !!!0x0010) <> !!!0x0 in 
     let preservesig = (implflags &&& !!!0x0080) <> !!!0x0 in 
     let internalcall = (implflags &&& !!!0x1000) <> !!!0x0 in 
     let synchronized = (implflags &&& !!!0x0020) <> !!!0x0 in 
     let noinline = (implflags &&& !!!0x0008) <> !!!0x0 in 
     let mustrun = (implflags &&& !!!0x0040) <> !!!0x0 in 
     let cctor = (nm = ".cctor") in 
     let ctor = (nm = ".ctor") in 
     let generic,genarity,cc,retty,argtys,varargs = seek_read_blob_heap_as_method_sig ctxt numtypars type_idx in 
     if varargs <> None then dprintf0 "ignoring sentinel and varargs in MethodDef signature";
     
     if logging then dprint_endline ("finding end param idx"); 
     let end_param_idx =
       if idx >= ctxt.nrows (tab_Method) then 
         ctxt.nrows (tab_Param) + 1
       else
         let (_,_,_,_,_, param_idx) = seek_read_Method_row ctxt (idx + 1) in 
         param_idx in 
     
     if logging then dprint_endline ("found param range: "^string_of_int param_idx^" - "^string_of_int end_param_idx); 
     let ret,ilParams = seek_read_Params ctxt (retty,argtys) param_idx end_param_idx in
     if logging then dprint_endline ("read param range: "^string_of_int param_idx^" - "^string_of_int end_param_idx); 

     let res = 
       { mdName=nm;
         mdKind = 
         (if cctor then MethodKind_cctor 
         else if ctor then MethodKind_ctor 
         else if isStatic then MethodKind_static 
         else if virt then 
           MethodKind_virtual 
             { virtFinal=final; 
               virtNewslot=newslot; 
               virtStrict=strict;
               virtAbstract=abstr;
               virtOverrides=None; }
         else MethodKind_nonvirtual);
         mdAccess = member_access_of_flags flags;
         mdSecurityDecls=seek_read_SecurityDecls ctxt (hds_MethodDef,idx);
         mdHasSecurity=hassec;
     mdEntrypoint= (fst ctxt.eptoken = tab_Method && snd ctxt.eptoken = idx);
         mdReqSecObj=reqsecobj;
         mdHideBySig=hidebysig;
         mdSpecialName=specialname;
         mdUnmanagedExport=export;
         mdSynchronized=synchronized;
         mdMustRun=mustrun;
         mdPreserveSig=preservesig;
         mdManaged = not unmanaged;
         mdInternalCall = internalcall;
         mdForwardRef = forwardref;
         mdCodeKind = (if (codetype = !!!0x00) then MethodCodeKind_il else if (codetype = !!!0x01) then MethodCodeKind_native else if (codetype = !!!0x03) then MethodCodeKind_runtime else (dprint_endline  "unsupported code type"; MethodCodeKind_native));
         mdExport=None; (* @todo:VIP *)
         mdVtableEntry=None; (* @todo:VIP *)
         mdGenericParams=seek_read_GenericParams ctxt numtypars (tomd_MethodDef,idx);
         mdCustomAttrs=seek_read_CustomAttrs ctxt (hca_MethodDef,idx); 
         mdParams= ilParams;
         mdCallconv=cc;
         mdReturn=ret;
         mdBody=
           if (codetype = !!!0x01) && pinvoke then begin
             mk_lazy_mbody (notlazy MethodBody_native)
           end else if pinvoke then begin 
             seek_read_ImplMap ctxt nm  idx
           end else if internalcall or abstr or unmanaged or (codetype <> !!!0x00) then begin 
             if code_rva <> !!!0x0 then dprint_endline "non-IL or abstract method with non-zero RVA";
             mk_lazy_mbody (notlazy MethodBody_abstract)  
           end else 
             seek_read_MethodRVA ctxt (idx,nm,internalcall,noinline,numtypars) code_rva;   
       } in 
     if logging then dprint_endline ("  done method = " ^ nm); 
     res
       
       
   and seek_read_Params ctxt (retty,argtys) pidx1 pidx2 =
    let ret_res = ref { returnMarshal=None;
                        returnType=retty;
                        returnCustomAttrs=empty_custom_attrs; } in
    let params_res = Array.map (fun ty ->  { paramName=None;
                                             paramDefault=None;
                                             paramMarshal=None;
                                             paramIn=false;
                                             paramOut=false;
                                             paramOptional=false;
                                             paramType=ty;
                                             paramCustomAttrs=empty_custom_attrs }) (Array.of_list argtys)  in
    for i = pidx1 to pidx2 - 1 do
      seek_read_Param_extras ctxt (ret_res,params_res) i
    done;
    !ret_res, Array.to_list params_res

   and seek_read_Param_extras ctxt (ret_res,params_res) (idx:int) =
     let (flags,seq,name_idx) = seek_read_Param_row ctxt idx in
     if logging then dprint_endline ("reading param "^string_of_int idx^", seq = "^string_of_int seq); 
     let inout_masked = (flags &&& !!!0x00FF) in 
     let has_marshal = (flags &&& !!!0x2000) <> !!!0x0 in 
     let has_default = (flags &&& !!!0x1000) <> !!!0x0 in 
     let fm_reader idx = seek_read_indexed_row (ctxt.nrows tab_FieldMarshal,seek_read_FieldMarshal_row ctxt,fst,hfm_compare idx,is_sorted ctxt tab_FieldMarshal,(snd >> seek_read_blob_heap_as_native_type ctxt)) in 
     let cas = seek_read_CustomAttrs ctxt (hca_ParamDef,idx) in
     if seq = 0 then

       ret_res := 
           { !ret_res with 
                   returnMarshal=(if has_marshal then Some (fm_reader (hfm_ParamDef,idx)) else None);
                   returnCustomAttrs =cas }
     else if seq > Array.length params_res then dprint_endline "bad seq num. for param"
     else 
       params_res.(seq - 1) <- 
          { params_res.(seq - 1) with 
               paramMarshal=(if has_marshal then Some (fm_reader (hfm_ParamDef,idx)) else None);
               paramDefault = if has_default then Some (seek_read_Constant ctxt (hc_ParamDef,idx)) else None;
               paramName = seek_read_string_heap_option ctxt name_idx;
               paramIn = ((inout_masked &&& !!!0x0001) <> !!!0x0);
               paramOut = ((inout_masked &&& !!!0x0002) <> !!!0x0);
               paramOptional = ((inout_masked &&& !!!0x0010) <> !!!0x0);
               paramCustomAttrs =cas }
            
   and seek_read_MethodImpls ctxt numtypars tidx =
     mk_lazy_mimpls 
        (lazy 
          begin 
           if logging then dprint_endline ("reading method impls for tidx "^string_of_int tidx);           
            let mimpls = seek_read_indexed_rows (ctxt.nrows tab_MethodImpl,seek_read_MethodImpl_row ctxt,(fun (a,_,_) -> a),simpleindex_compare tidx,is_sorted ctxt tab_MethodImpl,(fun (_,b,c) -> b,c)) in 
            List.map 
              (fun (b,c) -> 
                { mimplOverrideBy=(* intern_mspec manager *) (mk_mspec_in_typ (seek_read_mdor_no_varargs ctxt numtypars b));
                  mimplOverrides=
                  let mspec = (* intern_mspec manager *) (mk_mspec_in_typ (seek_read_mdor_no_varargs ctxt numtypars c)) in
                  OverridesSpec(formal_mref_of_mspec mspec, enclosing_typ_of_mspec mspec) })
              mimpls 
          end)

   and seek_read_multiple_MethodSemantics ctxt (flags,id) =
     List.map snd 
      (List.filter (fun (flags2,_) -> flags = flags2) 
         (seek_read_indexed_rows 
            (ctxt.nrows tab_MethodSemantics ,
             seek_read_MethodSemantics_row ctxt,
             (fun (flags,_,c) -> c),
             hs_compare id,is_sorted ctxt tab_MethodSemantics,
             (fun (a,b,c) -> a, formal_mref_of_mspec (mk_mspec_in_typ (seek_read_MethodDef_as_mdata ctxt b))))))

   and seek_read_optional_MethodSemantics ctxt id =
    match seek_read_multiple_MethodSemantics ctxt id with 
      [] -> None
    | [h] -> Some h
    | h::t -> dprint_endline "multiple method semantics found"; Some h

   and seek_read_MethodSemantics ctxt id =
     match seek_read_optional_MethodSemantics ctxt id with 
       None -> failwith "seek_read_MethodSemantics ctxt: no method found"
     | Some x -> x

   and seek_read_Event ctxt numtypars idx =
     let (flags,name_idx,typ_idx) = seek_read_Event_row ctxt idx in 
      { eventName = seek_read_string_heap ctxt name_idx;
        eventType = seek_read_optional_tdor ctxt numtypars AsObject typ_idx;
        eventSpecialName  = (flags &&& !!!0x0200) <> !!!0x0; 
        eventRTSpecialName = (flags &&& !!!0x0400) <> !!!0x0;
        eventAddOn= seek_read_MethodSemantics ctxt (!!!0x0008,(hs_Event, idx));
        eventRemoveOn=seek_read_MethodSemantics ctxt (!!!0x0010,(hs_Event,idx));
        eventFire=seek_read_optional_MethodSemantics ctxt (!!!0x0020,(hs_Event,idx));
        eventOther = seek_read_multiple_MethodSemantics ctxt (!!!0x0004, (hs_Event, idx));
        eventCustomAttrs=seek_read_CustomAttrs ctxt (hca_Event,idx) }
     
    (* @todo: can substantially reduce numbers of EventMap and PropertyMap reads by first checking if the whole table is sorted according to TypeDef tokens and then doing a binary chop *)
   and seek_read_Events ctxt numtypars tidx =
     mk_lazy_events 
        (lazy 
           begin 
             if logging then dprint_endline ("reading events for tidx "^string_of_int tidx);       
             let res = 
               match seek_read_optional_indexed_row (ctxt.nrows tab_EventMap,(fun i -> i, seek_read_EventMap_row ctxt i),(fun (_,row) -> fst row),compare tidx,false,(fun (i,row) -> (i,snd row))) with 
               | None -> []
               | Some (row_num,begin_event_idx) ->
                   let end_event_idx =
                     if row_num >= ctxt.nrows (tab_EventMap) then 
                       ctxt.nrows (tab_Event) + 1
                     else
                       let (_, end_event_idx) = seek_read_EventMap_row ctxt (row_num + 1) in 
                       end_event_idx in 
                   let res = ref [] in
                   for i = begin_event_idx to end_event_idx - 1 do
                     res := seek_read_Event ctxt numtypars i :: !res;
                   done;
                   List.rev !res in 
             if logging then dprint_endline ("found "^string_of_int (List.length res)^" events for tidx "^string_of_int tidx);
             res
           end)

   and seek_read_Property ctxt numtypars idx =
     let (flags,name_idx,typ_idx) = seek_read_Property_row ctxt idx in 
     let cc,retty,argtys = seek_read_blob_heap_as_property_sig ctxt numtypars typ_idx in
     let setter= seek_read_optional_MethodSemantics ctxt (!!!0x0001,(hs_Property,idx)) in 
     let getter = seek_read_optional_MethodSemantics ctxt (!!!0x0002,(hs_Property,idx)) in 
(* NOTE: the "hasthis" value on the property is not reliable: better to look on the getter/setter *)
(* NOTE: e.g. tlbimp on Office msword.olb seems to set this incorrectly *)
     let hasthis_of_callconv (Callconv (a,b)) = a in
     let cc2 =
       match getter with Some mref -> hasthis_of_callconv (callconv_of_mref mref) | None -> 
         match setter with Some mref ->  hasthis_of_callconv (callconv_of_mref mref) | None -> 
           cc in 
     { propName=seek_read_string_heap ctxt name_idx;
       propCallconv = cc2;
       propRTSpecialName=(flags &&& !!!0x0400) <> !!!0x0; 
       propSpecialName= (flags &&& !!!0x0200) <> !!!0x0; 
       propSet=setter;
       propGet=getter;
       propType=retty;
       propInit= if (flags &&& !!!0x1000) = 0l then None else Some (seek_read_Constant ctxt (hc_Property,idx));
       propArgs=argtys;
       propCustomAttrs=seek_read_CustomAttrs ctxt (hca_Property,idx) }
     
   and seek_read_Properties ctxt numtypars tidx =
     mk_lazy_properties
        (lazy 
           begin 
             if logging then dprint_endline ("reading properties for tidx "^string_of_int tidx);           
             let res = 
               match seek_read_optional_indexed_row (ctxt.nrows tab_PropertyMap,(fun i -> i, seek_read_PropertyMap_row ctxt i),(fun (_,row) -> fst row),compare tidx,false,(fun (i,row) -> (i,snd row))) with 
               | None -> []
               | Some (row_num,begin_prop_idx) ->
                   let end_prop_idx =
                     if row_num >= ctxt.nrows (tab_PropertyMap) then 
                       ctxt.nrows (tab_Property) + 1
                     else
                       let (_, end_prop_idx) = seek_read_PropertyMap_row ctxt (row_num + 1) in 
                       end_prop_idx in 
                   let res = ref [] in
                   for i = begin_prop_idx to end_prop_idx - 1 do
                     res := seek_read_Property ctxt numtypars i :: !res;
                   done;
                   List.rev !res in 
             if logging then dprint_endline ("found "^string_of_int (List.length res)^" properties for tidx "^string_of_int tidx);
             res
           end)


   and seek_read_CustomAttrs ctxt idx = 
     mk_computed_custom_attrs
      (fun () ->
           if logging then dprint_endline ("reading custom attrs for token "^string_of_int (snd idx));     
           seek_read_indexed_rows (ctxt.nrows tab_CustomAttribute,seek_read_CustomAttribute_row ctxt,(fun (a,_,_) -> a),hca_compare idx,is_sorted ctxt tab_CustomAttribute,(fun (_,b,c) -> seek_read_CustomAttr ctxt (b,c))))

   and seek_read_CustomAttr ctxt ((cat,idx),b) = 
     ctxt.seek_read_CustomAttr (CustomAttrIdx (cat,idx,b))
   and seek_read_CustomAttr_uncached ctxtH (CustomAttrIdx (cat,idx,val_idx)) = 
     let ctxt = getH ctxtH in 
      { customMethod=seek_read_cat ctxt (cat,idx);
        customData=
          match seek_read_blob_heap_option ctxt val_idx with
          | Some bytes -> bytes
          | None -> Bytes.of_intarray [| |] }

   and seek_read_SecurityDecls ctxt idx = 
     mk_lazy_security_decls
      (lazy
         begin
           if logging then dprint_endline ("reading security decls for token "^string_of_int (snd idx));           
           seek_read_indexed_rows (ctxt.nrows tab_Permission,seek_read_Permission_row ctxt,(fun (_,par,_) -> par),hds_compare idx,is_sorted ctxt tab_Permission,(fun (act,_,ty) -> seek_read_SecurityDecl ctxt (act,ty)))
         end)

   and seek_read_SecurityDecl ctxt (a,b) = 
     ctxt.seek_read_SecurityDecl (SecurityDeclIdx (a,b))
   and seek_read_SecurityDecl_uncached ctxtH (SecurityDeclIdx (act,ty)) = 
     let ctxt = getH ctxtH in 
     if logging then dprint_endline "reading SecurityDecl";
     PermissionSet ((if List.mem_assoc (u16_to_int act) (Lazy.force secaction_rmap) then List.assoc (u16_to_int act) (Lazy.force secaction_rmap) else failwith "unknown security action"),
                    seek_read_blob_heap ctxt ty)


  and seek_read_Constant ctxt idx =
    let kind,vidx = seek_read_indexed_row (ctxt.nrows tab_Constant,seek_read_Constant_row ctxt,(fun (_,key,_) -> key), hc_compare idx,is_sorted ctxt tab_Constant,(fun (kind,_,v) -> kind,v)) in
    match kind with 
    | x when x = !!!et_STRING -> FieldInit_bytes (seek_read_blob_heap ctxt vidx)
    | x when x = !!!et_BOOLEAN -> FieldInit_bool (seek_read_blob_heap_as_bool ctxt vidx) 
    | x when x = !!!et_CHAR -> FieldInit_char (seek_read_blob_heap_as_u16 ctxt vidx) 
    | x when x = !!!et_I1 -> FieldInit_int8 (seek_read_blob_heap_as_i8 ctxt vidx) 
    | x when x = !!!et_I2 -> FieldInit_int16 (seek_read_blob_heap_as_i16 ctxt vidx) 
    | x when x = !!!et_I4 -> FieldInit_int32 (seek_read_blob_heap_as_i32 ctxt vidx) 
    | x when x = !!!et_I8 -> FieldInit_int64 (seek_read_blob_heap_as_i64 ctxt vidx) 
    | x when x = !!!et_U1 -> FieldInit_uint8 (seek_read_blob_heap_as_u8 ctxt vidx) 
    | x when x = !!!et_U2 -> FieldInit_uint16 (seek_read_blob_heap_as_u16 ctxt vidx) 
    | x when x = !!!et_U4 -> FieldInit_uint32 (seek_read_blob_heap_as_u32 ctxt vidx) 
    | x when x = !!!et_U8 -> FieldInit_uint64 (seek_read_blob_heap_as_u64 ctxt vidx) 
    | x when x = !!!et_R4 -> FieldInit_float32 (seek_read_blob_heap_as_ieee32 ctxt vidx) 
    | x when x = !!!et_R8 -> FieldInit_float64 (seek_read_blob_heap_as_ieee64 ctxt vidx) 
    | x when x = !!!et_CLASS or x = !!!et_OBJECT ->  FieldInit_ref
    | _ -> FieldInit_ref

   and seek_read_ImplMap ctxt nm midx = 
     mk_lazy_mbody 
        (lazy 
            begin 
              if logging then dprint_endline ("reading pinvoke map for method "^string_of_int midx);       
              let (flags,name_idx, scope_idx) = seek_read_indexed_row (ctxt.nrows tab_ImplMap,seek_read_ImplMap_row ctxt,(fun (_,m,_,_) -> m),mf_compare (mf_MethodDef,midx),is_sorted ctxt tab_ImplMap,(fun (a,_,c,d) -> a,c,d)) in 
              let cc = 
                let masked = flags &&& !!!0x0700 in 
                if masked = !!!0x0000 then PInvokeCallConvNone 
                else if masked = !!!0x0200 then PInvokeCallConvCdecl 
                else if masked = !!!0x0300 then PInvokeCallConvStdcall 
                else if masked = !!!0x0400 then PInvokeCallConvThiscall 
                else if masked = !!!0x0500 then PInvokeCallConvFastcall 
                else if masked = !!!0x0100 then PInvokeCallConvWinapi 
                else (dprint_endline "strange pinvokeCallconv"; PInvokeCallConvNone) in
              let enc = 
                let masked = flags &&& !!!0x0006 in 
                if masked = !!!0x0000 then PInvokeEncodingNone 
                else if masked = !!!0x0002 then PInvokeEncodingAnsi 
                else if masked = !!!0x0004 then PInvokeEncodingUnicode 
                else if masked = !!!0x0006 then PInvokeEncodingAuto 
                else (dprint_endline "strange pinvokeEncoding"; PInvokeEncodingNone) in
              let bestfit = 
                let masked = flags &&& !!!0x0030 in 
                if masked = !!!0x0000 then PInvokeBestFitUseAssem 
                else if masked = !!!0x0010 then PInvokeBestFitEnabled 
                else if masked = !!!0x0020 then PInvokeBestFitDisabled 
                else (dprint_endline "strange pinvokeBestFit"; PInvokeBestFitUseAssem) in
              let unmap = 
                let masked = flags &&& !!!0x3000 in 
                if masked = !!!0x0000 then PInvokeThrowOnUnmappableCharUseAssem 
                else if masked = !!!0x1000 then PInvokeThrowOnUnmappableCharEnabled 
                else if masked = !!!0x2000 then PInvokeThrowOnUnmappableCharDisabled 
                else (dprint_endline "strange pinvokeThrowOnUnmappableChar"; PInvokeThrowOnUnmappableCharUseAssem) in

              MethodBody_pinvoke { pinvokeCallconv = cc; 
                                   pinvokeEncoding = enc;
                                   pinvokeBestFit=bestfit;
                                   pinvokeThrowOnUnmappableChar=unmap;
                                   pinvokeNoMangle = (flags &&& !!!0x0001) <> !!!0x0;
                                   pinvokeLastErr = (flags &&& !!!0x0040) <> !!!0x0;
                                   pinvokeName = 
                                   begin match seek_read_string_heap_option ctxt name_idx with 
                                     None -> nm
                                   | Some nm2 -> nm2
                                   end;
                                   pinvokeWhere = seek_read_ModuleRef ctxt scope_idx }
            end)

   and seek_read_topcode ctxt nm numtypars sz start seqpoints = 
     let labels_of_raw_offsets = Hashtbl.create (sz/2) in 
     let il_offsets_of_labels = Hashtbl.create (sz/2) in 
     let try_raw2lab raw_offset = 
       if Hashtbl.mem labels_of_raw_offsets raw_offset then 
         Some(Hashtbl.find labels_of_raw_offsets raw_offset)
       else 
         None in 
     let raw2lab raw_offset = 
       match try_raw2lab raw_offset with 
       | Some l -> l
       | None -> 
         let lab = generate_code_label() in 
         Hashtbl.add labels_of_raw_offsets raw_offset lab;
         lab in 
     let mark_as_instruction_start raw_offset il_offset = 
       let lab = raw2lab raw_offset in 
       Hashtbl.add il_offsets_of_labels lab il_offset in 

     let ibuf = new_buf "code stream" (sz/2) I_ret in 
     let curr = ref 0 in 
     let prefixes = { al=Aligned; tl= Normalcall; vol= Nonvolatile;ro=NormalAddress;constrained=None } in 
     let lastb = ref !!!0x0 in 
     let lastb2 = ref !!!0x0 in 
     let b = ref !!!0x0 in 
     let get () = 
         lastb := seek_read_u8_as_i32 ctxt.is (start +++ !!!(!curr));
         incr curr;
         b := 
           if !lastb = !!!0xfe && !curr < sz then begin
             lastb2 := seek_read_u8_as_i32 ctxt.is (start +++ !!!(!curr));
             incr curr;
             !lastb2
           end else begin
             !lastb
           end in 

     let seqpoints_remaining = ref seqpoints in 

     while !curr < sz do
       if logging then dprint_endline (ctxt.infile ^ ", "^nm^": registering "^string_of_int !curr^" as start of an instruction"); 
       mark_as_instruction_start !curr ibuf.current;

       (* Insert any sequence points into the instruction sequence *)
         
       if logging then dprint_endline ("** #remaining sequence points @ "^string_of_int !curr ^ " = "^string_of_int (List.length !seqpoints_remaining)); 


       while 
         begin
           match !seqpoints_remaining with 
           |  (i,tag) :: rest when i <= !curr -> true
           | _ -> false
         end 
       do
         begin
           if logging then dprint_endline ("** Emitting one sequence point ** "); 
           let (_,tag) = List.hd !seqpoints_remaining in
           seqpoints_remaining := List.tl !seqpoints_remaining;
           buf_emit_one ibuf (I_seqpoint tag)
         end
       done;

       if logging then dprint_endline (ctxt.infile ^ ", "^nm^ ": instruction begins at "^string_of_int !curr); 
       (* Read the prefixes.  Leave lastb and lastb2 holding the instruction byte(s) *)
       begin 
         prefixes.al <- Aligned;
         prefixes.tl <- Normalcall;
         prefixes.vol <- Nonvolatile;
         prefixes.ro<-NormalAddress;
         prefixes.constrained<-None;
         get ();
         while !curr < sz && 
           !lastb = !!!0xfe &&
           (!b = (!!!i_constrained &&& !!!0xff) or
            !b = (!!!i_readonly &&& !!!0xff) or
            !b = (!!!i_unaligned &&& !!!0xff) or
            !b = (!!!i_volatile &&& !!!0xff) or
            !b = (!!!i_tail &&& !!!0xff)) do
           begin 
             if !b = (!!!i_unaligned &&& !!!0xff) then
               let unal = seek_read_u8_as_i32 ctxt.is (start +++ !!!(!curr)) in
               incr curr;
               prefixes.al <-
                  if unal = !!!0x1 then Unaligned_1 
                  else if unal = !!!0x2 then Unaligned_2
                  else if unal = !!!0x4 then Unaligned_4 
                  else (dprint_endline "bad alignment for unaligned";  Aligned)
             else if !b = (!!!i_volatile &&& !!!0xff) then prefixes.vol <- Volatile
             else if !b = (!!!i_readonly &&& !!!0xff) then prefixes.ro <- ReadonlyAddress
             else if !b = (!!!i_constrained &&& !!!0xff) then 
               begin
                 let uncoded = seek_read_uncoded_token ctxt.is (start +++ !!!(!curr)) in 
                 curr := !curr + 4;
                 let typ = seek_read_tdor ctxt numtypars AsObject [] (uncoded_token_to_tdor uncoded) in 
                 prefixes.constrained <- Some typ
               end
             else prefixes.tl <- Tailcall;
           end;
           get ();
         done;
       end;

       if logging then dprint_endline (ctxt.infile ^ ": data for instruction begins at "^string_of_int !curr); 
       (* Read and decode the instruction *)
       if (!curr <= sz) then begin
         let idecoder = 
           if !lastb = !!!0xfe then get_two_byte_instr (Int32.to_int !lastb2)
           else get_one_byte_instr (Int32.to_int !lastb) in
         buf_emit_one ibuf
           begin 
             match idecoder with 
           | I_u16_u8_instr f -> 
               let x = seek_read_u8_as_u16 ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 1;
               f prefixes x
           | I_u16_u16_instr f -> 
               let x = seek_read_u16 ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 2;
               f prefixes x
           | I_none_instr f -> 
               f prefixes 
           | I_i64_instr f ->
               let x = seek_read_i64 ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 8;
               f prefixes x
           | I_i32_i8_instr f ->
               let x = seek_read_i8_as_i32 ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 1;
               f prefixes x
           | I_i32_i32_instr f ->
               let x = seek_read_i32 ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 4;
               f prefixes x
           | I_r4_instr f ->
               let x = seek_read_ieee32 ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 4;
               f prefixes x
           | I_r8_instr f ->
               let x = seek_read_ieee64 ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 8;
               f prefixes x
           | I_field_instr f ->
               let (tab,tok) = seek_read_uncoded_token ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 4;
               let fspec = 
                 if tab = tab_Field then 
                   seek_read_FieldDef_as_fspec ctxt tok
                 else if tab = tab_MemberRef then
                   seek_read_MemberRef_as_fspec ctxt numtypars tok
                 else failwith "bad table in FieldDefOrRef" in 
               f prefixes fspec
           | I_method_instr f ->
             if logging then dprint_endline (ctxt.infile ^ ": method instruction, curr = "^string_of_int !curr); 
         
               let (tab,idx) = seek_read_uncoded_token ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 4;
               let  (encl_typ, cc, nm, argtys,varargs, retty, minst) =
                 if tab = tab_Method then 
                   seek_read_mdor ctxt numtypars (mdor_MethodDef, idx)
                 else if tab = tab_MemberRef then 
                   seek_read_mdor ctxt numtypars (mdor_MemberRef, idx)
                 else if tab = tab_MethodSpec then 
                   seek_read_MethodSpec_as_mdata ctxt numtypars idx  
                 else failwith "bad table in MethodDefOrRefOrSpec"  in 
               if is_array_ty encl_typ then 
                 let (shape,ty) = dest_array_ty encl_typ in 
                 begin match nm with
                 | "Get" -> I_ldelem_any(shape,ty)
                 | "Set" ->  I_stelem_any(shape,ty)
                 | "Address" ->  I_ldelema(prefixes.ro, shape,ty)
                 | ".ctor" ->  I_newarr(shape,ty)
                 | _ -> failwith "bad method on array type"
                 end
               else 
                 let mspec = (* intern_mspec manager *) (mk_mspec_in_typ (encl_typ, cc, nm, argtys, retty, minst)) in 
                 f prefixes (mspec,varargs)
           | I_type_instr f ->
               let uncoded = seek_read_uncoded_token ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 4;
               let typ = seek_read_tdor ctxt numtypars AsObject [] (uncoded_token_to_tdor uncoded) in 
               f prefixes typ
           | I_string_instr f ->
               let (tab,idx) = seek_read_uncoded_token ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 4;
               if tab <> tab_UserStrings then dprint_endline "warning: bad table in user string for ldstr";
               f prefixes (seek_read_user_string_heap ctxt (Int32.of_int idx))

           | I_conditional_i32_instr f ->
               let offs_dest = Int32.to_int (seek_read_i32 ctxt.is (start +++ !!!(!curr))) in 
               curr := !curr + 4;
               let dest = !curr + offs_dest in 
               let next = !curr in 
               f prefixes (raw2lab dest, raw2lab next)
           | I_conditional_i8_instr f ->
               let offs_dest = i8_to_int (seek_read_i8 ctxt.is (start +++ !!!(!curr))) in 
               curr := !curr + 1;
               let dest = !curr + offs_dest in 
               let next = !curr in 
               f prefixes (raw2lab dest, raw2lab next)
           | I_unconditional_i32_instr f ->
               let offs_dest = Int32.to_int (seek_read_i32 ctxt.is (start +++ !!!(!curr))) in 
               curr := !curr + 4;
               let dest = !curr + offs_dest in 
               f prefixes (raw2lab dest)
           | I_unconditional_i8_instr f ->
               let offs_dest = i8_to_int (seek_read_i8 ctxt.is (start +++ !!!(!curr))) in 
               curr := !curr + 1;
               let dest = !curr + offs_dest in 
               f prefixes (raw2lab dest)
           | I_invalid_instr -> dprint_endline ("invalid instruction: "^Int32.to_string !lastb^ (if !lastb = !!!0xfe then ","^Int32.to_string !lastb2 else "")); I_ret
           | I_tok_instr f ->  
               let (tab,idx) = seek_read_uncoded_token ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 4;
               (* @todo: this incorrectly labels all MemberRef tokens as Token_method's: we should go look at the MemberRef sig to determine if it ctxt.is a field or method *)        
               let token_info = 
                 if tab = tab_Method or tab = tab_MemberRef (* @todo:generics or tab = tab_MethodSpec *) then 
                   Token_method ((* intern_mspec manager *) (mk_mspec_in_typ (seek_read_mdor_no_varargs ctxt numtypars (uncoded_token_to_mdor (tab,idx)))))
                 else if tab = tab_Field then 
                   Token_field (seek_read_FieldDef_as_fspec ctxt idx)
                 else if tab = tab_TypeDef or tab = tab_TypeRef or tab = tab_TypeSpec  then 
                   Token_type (seek_read_tdor ctxt numtypars AsObject [] (uncoded_token_to_tdor (tab,idx))) 
                 else failwith "bad token for ldtoken"  in 
               f prefixes token_info
           | I_sig_instr f ->  
               let (tab,idx) = seek_read_uncoded_token ctxt.is (start +++ !!!(!curr)) in 
               curr := !curr + 4;
               if tab <> tab_StandAloneSig then dprint_endline "strange table for callsig token";
               let generic,genarity,cc,retty,argtys,varargs = seek_read_blob_heap_as_method_sig ctxt numtypars (seek_read_StandAloneSig_row ctxt idx) in 
               if generic then failwith "bad image: a generic method signature ctxt.is begin used at a calli instruction";
               f prefixes (mk_callsig (cc,argtys,retty), varargs)
           | I_switch_instr f ->  
               let n = Int32.to_int (seek_read_i32 ctxt.is (start +++ !!!(!curr))) in 
               curr := !curr + 4;
               let offsets = 
                 collecti 
                   (fun _ -> 
                     let i = Int32.to_int (seek_read_i32 ctxt.is (start +++ !!!(!curr))) in 
                     curr := !curr + 4; 
                     i) 
                   n in 
               let dests = List.map (fun offs -> raw2lab (!curr + offs)) offsets in 
               let next = raw2lab !curr in 
               f prefixes (dests,next)
           end
       end
     done;
     (* Finished reading instructions - mark the end of the instruction stream in case the PDB information refers to it. *)
     mark_as_instruction_start !curr ibuf.current;
     (* Build the function that maps from raw labels (offsets into the bytecode stream) to indexes in the AbsIL instruction stream *)
     let lab2pc lab = 
       try
          Hashtbl.find il_offsets_of_labels lab 
       with Not_found -> failwith ("branch destination "^string_of_code_label lab^" not found in code") in 

     (* Some offsets used in debug info refer to the end of an instruction, rather than the *)
     (* start of the subsequent instruction.  But all labels refer to instruction starts, *)
     (* apart from a final label which refers to the end of the method.  This function finds *)
     (* the start of the next instruction referred to by the raw offset. *)
     let raw2nextLab raw_offset = 
       let is_instr_start x = 
         match try_raw2lab x with 
         | None -> false
         | Some lab -> Hashtbl.mem il_offsets_of_labels lab in
       if  is_instr_start raw_offset then raw2lab raw_offset 
       else if  is_instr_start (raw_offset+1) then raw2lab (raw_offset+1)
       else failwith ("the bytecode raw offset "^string_of_int raw_offset^" did not refer either to the start or end of an instruction") in 
     let instrs = buf_close ibuf in 
     instrs,raw2lab, lab2pc, raw2nextLab

   and seek_read_MethodRVA ctxt (idx,nm,internalcall,noinline,numtypars) rva = 
    mk_lazy_mbody 
     (lazy
       begin 

         (* Read any debug information for this method into temporary data structures *)
         (*    -- a list of locals, marked with the raw offsets (actually closures which accept the resolution function that maps raw offsets to labels) *)
         (*    -- an overall range for the method *)
         (*    -- the sequence points for the method *)
         let local_pdb_infos, mrange_pdb_info, seqpoints = 
           match ctxt.pdb with 
             None -> 
               [], None, []
           | Some (pdbr, get_doc) -> 
               begin 
                 try 

                   let pdbm = pdbReaderGetMethod pdbr (uncoded_token tab_Method idx) in 
                   let rootScope = pdbMethodGetRootScope pdbm  in
                   let sps = pdbMethodGetSequencePoints pdbm in
                   (*dprintf2 "#sps for 0x%lx = %d\n" (uncoded_token tab_Method idx) (Array.length sps);  *)
                   (* let roota,rootb = pdbScopeGetOffsets rootScope in  *)
                   let seqpoints =
                    let arr = 
                       Array.map 
                         (fun sp -> 
                           (* It is VERY annoying to have to call GetURL for the document for each sequence point.  This appears to be a short coming of the PDB reader API.  They should return an index into the array of documents for the reader *)
                           let sourcedoc = get_doc (pdbDocumentGetURL sp.pdbSeqPointDocument) in 
                           let source = 
                             { sourceDocument = sourcedoc;
                               sourceLine = sp.pdbSeqPointLine;
                               sourceColumn = sp.pdbSeqPointColumn;
                               sourceEndLine = sp.pdbSeqPointEndLine;
                               sourceEndColumn = sp.pdbSeqPointEndColumn } in 
                           (sp.pdbSeqPointOffset,source))
                         sps in
                     Array.sort (fun (x,_) (y,_) -> compare x y) arr;
                     Array.to_list arr in
                let rec scopes scp = 
                     let a,b = pdbScopeGetOffsets scp in 
                     let lvs =  pdbScopeGetLocals scp in 
                     let ilvs = 
                       List.filter
                         (fun l -> 
                           let k,idx = pdbVariableGetAddressAttributes l in
                           Int32.to_int k = 1 (* ADDR_IL_OFFSET *)) 
                         (Array.to_list lvs) in 
                     let ilinfos =
                       List.map 
                         (fun ilv -> 
                           let k,idx = pdbVariableGetAddressAttributes ilv in
                         let n = pdbVariableGetName ilv in 
                             if logging then dprint_endline ("local variable debug info: name="^n^", kind = "^Int32.to_string k^", localNum = "^Int32.to_string idx); 
                           { localNum= Int32.to_int idx; 
                             localName=n})
                         ilvs in 
                     let this_one = 
                       (fun raw2nextLab ->
                         { locRange= (raw2nextLab a,raw2nextLab b); 
                           locInfos = ilinfos }) in
                     if logging then dprint_endline ("this scope covers IL range: "^string_of_int a^"-"^string_of_int b); 
                     let others = List.fold_right (scopes >> (@)) (Array.to_list (pdbScopeGetChildren scp)) [] in
                     this_one :: others in
                   let local_pdb_infos = [] (* <TODO> scopes fail for mscorlib </TODO> scopes rootScope  *) in 
                   if logging then dprint_endline ("done local_pdb_infos"); 
                   local_pdb_infos,
                   None (* @todo: look through sps to get ranges?  Use GetRanges?? Change AbsIL?? *),
                   seqpoints
         with e -> 
           if logging then dprint_endline ("*** Warning: PDB info for method "^nm^" could not be read and will be ignored: "^(Printexc.to_string e));
                 [],None,[]
               end   in 
         
         
         let base = ctxt.anyV2P("method rva",rva) in 
         if logging then dprint_endline (ctxt.infile ^ ": reading body of method "^nm^" at rva "^Int32.to_string rva^", phys "^Int32.to_string base); 
         let b = seek_read_u8_as_i32 ctxt.is base in 
         if (b &&& e_CorILMethod_FormatMask) = e_CorILMethod_TinyFormat then begin
           let code_base = base +++ !!!1 in 
           let code_size = Int32.to_int (b lsr 2) in 
           if logging then dprint_endline (ctxt.infile ^ ": tiny format for "^nm^", code size = " ^ string_of_int code_size);
           let instrs,_,lab2pc,raw2nextLab = seek_read_topcode ctxt nm numtypars code_size code_base seqpoints in 
           (* Convert the linear code format to the nested code format *)
           if logging then dprint_endline ("doing local_pdb_infos2 (tiny format)"); 
           let local_pdb_infos2 = List.map (fun f -> f raw2nextLab) local_pdb_infos in
           if logging then dprint_endline ("done local_pdb_infos2 (tiny format), checking code..."); 
           let code = check_code (build_code nm lab2pc instrs [] local_pdb_infos2) in
           if logging then dprint_endline ("done checking code (tiny format)."); 
           MethodBody_il
             { ilZeroInit=false;
               ilMaxStack= int_to_i32 8;
               ilNoInlining=noinline;
               ilLocals=[];
               ilSource=mrange_pdb_info; 
               ilCode=code }

         end else if (b &&& e_CorILMethod_FormatMask) = e_CorILMethod_FatFormat then begin
           let has_more_sects = (b &&& e_CorILMethod_MoreSects) <> !!!0x0 in 
           let initlocals = (b &&& e_CorILMethod_InitLocals) <> !!!0x0 in 
           let maxstack = seek_read_u16_as_i32 ctxt.is (base +++ !!!2) in
           let code_size = seek_read_i32 ctxt.is (base +++ !!!4) in
           let locals_tab,localtoken = seek_read_uncoded_token ctxt.is (base +++ !!!8) in
           let code_base = base +++ !!!12 in 
           let locals = 
             if localtoken = 0x0 then [] 
             else begin 
               if locals_tab <> tab_StandAloneSig then dprint_endline "strange table for locals token";
               seek_read_blob_heap_as_locals_sig ctxt numtypars (seek_read_StandAloneSig_row ctxt localtoken) 
             end in 

           if logging then dprint_endline (ctxt.infile ^ ": fat format for "^nm^", code size = " ^ Int32.to_string code_size^", has_more_sects = "^(if has_more_sects then "true" else "false")^",b = "^Int32.to_string b);
           
           (* Read the method body *)
           let instrs,raw2lab,lab2pc,raw2nextLab = seek_read_topcode ctxt nm numtypars (Int32.to_int code_size) code_base seqpoints in

           (* Read all the sections that follow the method body. *)
           (* These contain the exception clauses. *)
           let next_sect_base = ref (align !!!4 (code_base +++ code_size)) in 
           let more_sects = ref has_more_sects in 
           let seh = ref [] in 
           while !more_sects do
             let sect_base = !next_sect_base in 
             let sect_flag = seek_read_u8_as_i32 ctxt.is sect_base in 
             if logging then dprint_endline (ctxt.infile ^ ": fat format for "^nm^", sect_flag = " ^ Int32.to_string sect_flag);
             let sect_size, clauses = 
               if (sect_flag &&& e_CorILMethod_Sect_FatFormat) <> !!!0x0 then 
                 let big_size = (seek_read_i32 ctxt.is sect_base) lsr 8 in 
                 if logging then dprint_endline (nm^": one more section");
                 if logging then dprint_endline (ctxt.infile ^ ": big_size = "^Int32.to_string big_size);
                 let clauses = 
                   if (sect_flag &&& e_CorILMethod_Sect_EHTable) <> !!!0x0 then 
                     (* WORKAROUND: The ECMA spec says this should be  *)
                     (* let num_clauses = Int32.to_int ((big_size --- !!!4)  /./ !!!24) in  *)
                     (* but the CCI IL generator generates multiples of 24 *)
                     let num_clauses = Int32.to_int (big_size  /./ !!!24) in 
                     if logging then dprint_endline (nm^" has "^string_of_int num_clauses ^" fat seh clauses");
                     
                     collecti
                       (fun i -> 
                         let clause_base = sect_base +++ !!!4 +++ (!!!i *** !!!24) in 
                         let kind = seek_read_i32 ctxt.is (clause_base +++ 0l) in 
                         if logging then dprint_endline ("One fat SEH clause, kind = "^Int32.to_string kind);
                         let st1 = seek_read_i32 ctxt.is (clause_base +++ !!!4) in 
                         let sz1 = seek_read_i32 ctxt.is (clause_base +++ !!!8) in 
                         let st2 = seek_read_i32 ctxt.is (clause_base +++ !!!12) in 
                         let sz2 = seek_read_i32 ctxt.is (clause_base +++ !!!16) in 
                         let extra = seek_read_i32 ctxt.is (clause_base +++ !!!20) in 
                         (kind,st1,sz1,st2,sz2,extra))
                       num_clauses 
                   else [] in 
                 big_size, clauses
               else 
                 let small_size = seek_read_u8_as_i32 ctxt.is (sect_base +++ !!!0x01) in 
                 let clauses = 
                   if (sect_flag &&& e_CorILMethod_Sect_EHTable) <> !!!0x0 then begin
                     if logging then dprint_endline (nm^": small_size = "^Int32.to_string small_size);
                     (* WORKAROUND: The ECMA spec says this should be  *)
                     (* let num_clauses = Int32.to_int ((small_size --- !!!4)  /./ !!!12) in  *)
                     (* but the C# compiler (or some IL generator) generates multiples of 12 *)
                     let num_clauses = Int32.to_int (small_size  /./ !!!12) in 
                     if logging then dprint_endline (nm^" has "^string_of_int num_clauses ^" tiny seh clauses");
                     collecti
                       (fun i -> 

                         let clause_base = sect_base +++ !!!4 +++ (!!!i *** !!!12) in 
                         let kind = seek_read_u16_as_i32 ctxt.is (clause_base +++ 0l) in 
                         if logging then dprint_endline ("One tiny SEH clause, kind = "^Int32.to_string kind);
                         let st1 = seek_read_u16_as_i32 ctxt.is (clause_base +++ !!!2) in 
                         let sz1 = seek_read_u8_as_i32 ctxt.is (clause_base +++ !!!4) in 
                         let st2 = seek_read_u16_as_i32 ctxt.is (clause_base +++ !!!5) in 
                         let sz2 = seek_read_u8_as_i32 ctxt.is (clause_base +++ !!!7) in 
                         let extra = seek_read_i32 ctxt.is (clause_base +++ !!!8) in 
                         (kind,st1,sz1,st2,sz2,extra))
                       num_clauses 
                   end else [] in 
                 small_size, clauses in 

             (* Morph together clauses that cover the same range *)
             let seh_clauses = 
                let seh_map = Hashtbl.create (List.length clauses) in 
        
                List.iter
                  (fun (kind,st1,sz1,st2,sz2,extra) ->
                    let try_start = raw2lab (Int32.to_int st1) in 
                    let try_finish = raw2lab (Int32.to_int (st1 +++ sz1)) in 
                    let handler_start = raw2lab (Int32.to_int st2) in 
                    let handler_finish = raw2lab (Int32.to_int (st2 +++ sz2)) in 
                    let clause = 
                      if kind = e_COR_ILEXCEPTION_CLAUSE_EXCEPTION then 
                        SEH_type_catch(seek_read_tdor ctxt numtypars AsObject [] (uncoded_token_to_tdor (i32_to_uncoded_token extra)), (handler_start, handler_finish) )
                      else if kind = e_COR_ILEXCEPTION_CLAUSE_FILTER then 
                        let filter_start = raw2lab (Int32.to_int extra) in 
                        let filter_finish = handler_start in 
                        SEH_filter_catch((filter_start, filter_finish), (handler_start, handler_finish))
                      else if kind = e_COR_ILEXCEPTION_CLAUSE_FINALLY then 
                        SEH_finally(handler_start, handler_finish)
                      else if kind = e_COR_ILEXCEPTION_CLAUSE_FAULT then 
                        SEH_fault(handler_start, handler_finish)
                      else begin
                        dprint_endline (ctxt.infile ^ ": unknown exception handler kind: "^Int32.to_string kind);
                        SEH_finally(handler_start, handler_finish)
                      end
                    in 
                    let key =  (try_start, try_finish) in 
                    if Hashtbl.mem seh_map key then 
                      let prev = Hashtbl.find seh_map key in
                      Hashtbl.replace seh_map key (prev @ [clause])
                    else 
                      Hashtbl.add seh_map key [clause])
                  clauses;
                Hashtbl.fold  (fun key bs acc -> {exnRange=key; exnClauses=bs} :: acc) seh_map [] in 
             seh := seh_clauses;
             more_sects := (sect_flag &&& e_CorILMethod_Sect_MoreSects) <> !!!0x0;
             next_sect_base := sect_base +++ sect_size;
           done; (* while *)

           (* Convert the linear code format to the nested code format *)
           if logging then dprint_endline ("doing local_pdb_infos2"); 
           let local_pdb_infos2 = List.map (fun f -> f raw2nextLab) local_pdb_infos in
           if logging then dprint_endline ("done local_pdb_infos2, checking code..."); 
           let code = check_code (build_code nm lab2pc instrs !seh local_pdb_infos2) in
           if logging then dprint_endline ("done checking code."); 
           MethodBody_il
             { ilZeroInit=initlocals;
               ilMaxStack= maxstack;
               ilNoInlining=noinline;
               ilLocals=locals;
               ilCode=code;
               ilSource=mrange_pdb_info}
         end else begin
           if logging then failwith "unknown format";
           MethodBody_abstract
         end
       end)

  and i32_as_variant_typ ctxt (n:i32) = 
    if List.mem_assoc n (Lazy.force variant_type_rmap) then 
      List.assoc n (Lazy.force variant_type_rmap)
    else if (n &&& vt_ARRAY) <> !!!0x0 then VariantType_array (i32_as_variant_typ ctxt (n &&& (Int32.lognot vt_ARRAY)))
    else if (n &&& vt_VECTOR) <> !!!0x0 then VariantType_vector (i32_as_variant_typ ctxt (n &&& (Int32.lognot vt_VECTOR)))
    else if (n &&& vt_BYREF) <> !!!0x0 then VariantType_byref (i32_as_variant_typ ctxt (n &&& (Int32.lognot vt_BYREF)))
    else (dprint_endline (ctxt.infile ^ ": i32_as_variant_typ ctxt: unexpected variant type, n = "^Int32.to_string n) ; VariantType_empty)

  and seek_read_blob_heap_as_native_type ctxt blob_idx = 
    if logging then dprint_endline (ctxt.infile ^ ": reading native type blob "^Int32.to_string blob_idx); 
    let bytes = seek_read_blob_heap ctxt blob_idx in 
    let res,_ = sigptr_get_native_typ ctxt bytes 0 in 
    res
  and sigptr_get_native_typ ctxt bytes sigptr = 
     if logging then dprint_endline (ctxt.infile ^ ": reading native type blob, sigptr= "^string_of_int sigptr); 
    let ntbyte,sigptr = sigptr_get_byte bytes sigptr in 
    if List.mem_assoc ntbyte (Lazy.force native_type_map) then 
      List.assoc ntbyte (Lazy.force native_type_map), sigptr
    else if ntbyte = 0x0 then NativeType_empty, sigptr
    else if ntbyte = nt_CUSTOMMARSHALER then  
      begin
        if logging then
          for i = 0 to Bytes.length bytes - 1 do
            if logging then dprint_endline (ctxt.infile ^ ": byte "^string_of_int i^" = "^string_of_int(Bytes.get bytes i));
          done;
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM1) , sigptr= "^string_of_int sigptr^ ", Bytes.length bytes = "^string_of_int(Bytes.length bytes)); 
        let guidLen,sigptr = sigptr_get_z_i32 bytes sigptr in 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM2) , sigptr= "^string_of_int sigptr^", guidLen = "^string_of_int (Int32.to_int guidLen)); 
        let guid,sigptr = sigptr_get_bytes (Int32.to_int guidLen) bytes sigptr in 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM3) , sigptr= "^string_of_int sigptr); 
        let nativeTypeNameLen,sigptr = sigptr_get_z_i32 bytes sigptr in 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM4) , sigptr= "^string_of_int sigptr^", nativeTypeNameLen = "^string_of_int (Int32.to_int nativeTypeNameLen)); 
        let nativeTypeName,sigptr = sigptr_get_string (Int32.to_int nativeTypeNameLen) bytes sigptr in 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM4) , sigptr= "^string_of_int sigptr^", nativeTypeName = "^nativeTypeName); 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM5) , sigptr= "^string_of_int sigptr); 
        let custMarshallerNameLen,sigptr = sigptr_get_z_i32 bytes sigptr in 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM6) , sigptr= "^string_of_int sigptr^", custMarshallerNameLen = "^string_of_int (Int32.to_int custMarshallerNameLen)); 
        let custMarshallerName,sigptr = sigptr_get_string (Int32.to_int custMarshallerNameLen) bytes sigptr in 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM7) , sigptr= "^string_of_int sigptr^", custMarshallerName = "^custMarshallerName); 
        let cookieStringLen,sigptr = sigptr_get_z_i32 bytes sigptr in 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM8) , sigptr= "^string_of_int sigptr^", cookieStringLen = "^string_of_int (Int32.to_int cookieStringLen)); 
        let cookieString,sigptr = sigptr_get_bytes (Int32.to_int cookieStringLen) bytes sigptr in 
        if logging then dprint_endline (ctxt.infile ^ ": reading native type blob (CM9) , sigptr= "^string_of_int sigptr); 
        NativeType_custom (guid,nativeTypeName,custMarshallerName,cookieString), sigptr
      end 
    else if ntbyte = nt_FIXEDSYSSTRING then 
      let i,sigptr = sigptr_get_z_i32 bytes sigptr in 
      NativeType_fixed_sysstring i, sigptr
    else if ntbyte = nt_FIXEDARRAY then 
      let i,sigptr = sigptr_get_z_i32 bytes sigptr in 
      NativeType_fixed_array i, sigptr
    else if ntbyte = nt_SAFEARRAY then 
      (if sigptr >= Bytes.length bytes then
         NativeType_safe_array(VariantType_empty, None),sigptr
       else 
         let i,sigptr = sigptr_get_z_i32 bytes sigptr in 
         if sigptr >= Bytes.length bytes then
           NativeType_safe_array (i32_as_variant_typ ctxt i, None), sigptr
         else 
           let len,sigptr = sigptr_get_z_i32 bytes sigptr in 
           let s,sigptr = sigptr_get_string (Int32.to_int len) bytes sigptr in 
           NativeType_safe_array (i32_as_variant_typ ctxt i, Some s), sigptr)
    else if ntbyte = nt_ARRAY then 
      (if sigptr >= Bytes.length bytes then
         NativeType_array(None,None),sigptr
       else 
         let nt,sigptr = 
           let u,sigptr' = sigptr_get_z_i32 bytes sigptr in 
           if (u = !!!nt_MAX) then 
             NativeType_empty, sigptr'
           else
           (* note: go back to start and read native type *)
             sigptr_get_native_typ ctxt bytes sigptr in 
         if sigptr >= Bytes.length bytes then
           NativeType_array (Some nt,None), sigptr
         else
           let pnum,sigptr = sigptr_get_z_i32 bytes sigptr in 
           if sigptr >= Bytes.length bytes then
             NativeType_array (Some nt,Some(pnum,None)), sigptr
           else 
             let additive,sigptr = 
               if sigptr >= Bytes.length bytes then 0l, sigptr
               else sigptr_get_z_i32 bytes sigptr in 
             NativeType_array (Some nt,Some(pnum,Some(additive))), sigptr)
    else (dprint_endline (ctxt.infile ^ ": unexpected native type, nt = "^string_of_int ntbyte); NativeType_empty, sigptr)
        
   and seek_read_ManifestResources ctxt () = 
    mk_lazy_resources 
      (lazy
         begin 
           let res = ref [] in 
           for i = 1 to ctxt.nrows (tab_ManifestResource) do
             let (offset,flags,name_idx,impl_idx) = seek_read_ManifestResource_row ctxt i in 
             let scoref = seek_read_i_as_scoref ctxt impl_idx in 
             let datalab = 
               match scoref with
               | ScopeRef_local -> 
                  let start = ctxt.anyV2P ("resource",offset +++ ctxt.resources_addr) in
                  let len = seek_read_i32 ctxt.is start in
                  Resource_local (fun () -> seek_read_bytes ctxt.is (start+++ !!!4) len)
               | ScopeRef_module mref -> Resource_file (mref,offset)
               | ScopeRef_assembly aref -> Resource_assembly aref in 

             let r = 
               { resourceName= seek_read_string_heap ctxt name_idx;
                 resourceWhere = datalab;
                 resourceAccess = (if (flags &&& !!!0x01) <> !!!0x0 then Resource_public else Resource_private);
                 resourceCustomAttrs =  seek_read_CustomAttrs ctxt (hca_ManifestResource, i) } in 
             res := r :: !res;
           done;
           List.rev !res 
         end)


   and seek_read_top_NestedExportedTypes ctxt parent_idx = 
    mk_lazy_nested_exported_types
      (lazy
         begin 
           let res = ref [] in 
           for i = 1 to ctxt.nrows tab_ExportedType do
             let (flags,tok,name_idx,namespace_idx,impl_idx) = seek_read_ExportedType_row ctxt i in 
             if not (seek_is_top_TypeDef flags) && 
               begin 
                 let (tag,idx) = impl_idx in 
                 match tag with 
                 | tag when tag = i_File -> false
                 | tag when tag = i_AssemblyRef -> false
                 | tag when tag = i_ExportedType -> idx = parent_idx 
                 | _ -> false
               end
             then 
               let nm = seek_read_blob_heap_as_type_name ctxt (name_idx,namespace_idx) in
               let entry = 
                 { nestedExportedTypeName=nm;
                   nestedExportedTypeAccess=(match type_access_of_flags flags with TypeAccess_nested n -> n | _ -> failwith "non-nested access for a nested type described as being in an auxiliary module");
                   nestedExportedTypeNested=seek_read_top_NestedExportedTypes ctxt i;
                   nestedExportedTypeCustomAttrs=seek_read_CustomAttrs ctxt (hca_ExportedType, i) }  in
               res := entry :: !res;
           done;
           List.rev !res
         end)
      
  and seek_read_top_ExportedTypes ctxt () = 
    mk_lazy_exported_types 
      (lazy
         begin 
           let res = ref [] in 
           for i = 1 to ctxt.nrows tab_ExportedType do
             let (flags,tok,name_idx,namespace_idx,impl_idx) = seek_read_ExportedType_row ctxt i in 
             if seek_is_top_TypeDef flags then 
               let nm = seek_read_blob_heap_as_type_name ctxt (name_idx,namespace_idx) in
               let scoref = seek_read_i_as_scoref ctxt impl_idx in
               let entry = 
                 { exportedTypeScope=scoref;
                   exportedTypeName=nm;
                   exportedTypeForwarder =   ((flags &&& 0x00200000l) <> 0l);
                   exportedTypeAccess=type_access_of_flags flags;
                   exportedTypeNested=seek_read_top_NestedExportedTypes ctxt i;
                   exportedTypeCustomAttrs=seek_read_CustomAttrs ctxt (hca_ExportedType, i) }  in
               res := entry :: !res;
           done;
           List.rev !res
         end)
      
  
let close_binary_reader x = x.dispose()
let modul_of_binary_reader x = x.modul

let defaults = 
  { optimizeForMemory=false; 
    pdbPath= None; 
    manager = None; 
    mscorlib=ecma_mscorlib_refs } 

let open_binary_reader infile opts = 

(*F#
#if CLI_AT_LEAST_2_0 // GENERICS <-> WIN32
 try 
    let mmap = MMapChannel.open_in infile in 
    let modul,pdb = gen_open_binary_reader infile (MMap mmap) opts in 
    { modul = modul; 
      dispose = (fun () -> 
        MMapChannel.close mmap;
        begin match pdb with 
        | Some (pdbr,_) -> pdbReadClose pdbr
        | None -> ()
        end) }
  with :? System.DllNotFoundException ->
#endif
F#*)
  let is = open_in_bin infile in 
  let cell = ref (Some is) in 
  let modul,pdb = gen_open_binary_reader infile (Chan (infile,cell)) opts in 
  { modul = modul; 
    dispose = (fun () -> 
      cell := None;
      close_in is;
      begin match pdb with 
      | Some (pdbr,_) -> pdbReadClose pdbr
      | None -> ()
      end) }

let open_binary_reader_in_memory infile opts = 
  let mc = MemChannel.open_in infile in 
  let modul,pdb = gen_open_binary_reader infile (Mem mc) opts in 
  { modul = modul; 
    dispose = (fun () -> 
      begin match pdb with 
      | Some (pdbr,_) -> pdbReadClose pdbr
      | None -> ()
      end) }


let read_binary a b = modul_of_binary_reader (open_binary_reader a {defaults with pdbPath=b})

