// (c) Microsoft Corporation 2005-2007.

#light

/// Immutable sets implemented via binary trees
namespace Microsoft.FSharp.Compatibility.OCaml

open System
open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

module Set = 

    //--------------------------------------------------------------------------
    // Set.Make functor
    
    ///A collection of operations for creating and using sets based on a particular comparison function.
    ///The 'tag' type parameter is used to track information about the comparison function.
    type Provider<'a,'tag> when 'tag :> IComparer<'a> =
       { empty: Tagged.Set<'a,'tag>;
         is_empty: Tagged.Set<'a,'tag> -> bool;
         mem: 'a -> Tagged.Set<'a,'tag> -> bool;
         add: 'a -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         singleton: 'a -> Tagged.Set<'a,'tag>;
         remove: 'a -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         union: Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         inter: Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         diff: Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         iter: ('a -> unit) -> Tagged.Set<'a,'tag> -> unit;
         elements: Tagged.Set<'a,'tag> -> 'a list;
         equal: Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> bool;
         subset: Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> bool;
         compare: Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> int;
         for_all: ('a -> bool) -> Tagged.Set<'a,'tag> -> bool;
         exists: ('a -> bool) -> Tagged.Set<'a,'tag> -> bool;
         filter: ('a -> bool) -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         partition: ('a -> bool) -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> * Tagged.Set<'a,'tag>;
         fold: 'b. ('a -> 'b -> 'b) -> Tagged.Set<'a,'tag> -> 'b -> 'b;
         cardinal: Tagged.Set<'a,'tag> -> int;
         min_elt: Tagged.Set<'a,'tag> -> 'a;
         max_elt: Tagged.Set<'a,'tag> -> 'a;
         choose: Tagged.Set<'a,'tag> -> 'a }

    type Provider<'a> = Provider<'a, IComparer<'a>>

    /// Build a collection of operations for creating and using 
    /// maps based on a single consistent comparison function. This returns a record
    /// that contains the functions you use to create and manipulate maps all of which 
    /// use this comparison function.  The returned value is much like an ML module. 
    ///
    /// Use MakeTagged if you want additional type safety that guarantees that two sets
    /// based on different comparison functions can never be combined in inconsistent ways.
    
    val Make: ('a -> 'a -> int) -> Provider<'a>

    ///A functor to build a collection of operations for creating and using 
    /// sets based on the given comparison function. This returns a record that 
    /// contains the functions you use to create and manipulate maps of
    /// this kind.  The returned value is much like an ML module. 
    ///
    /// To use this function you need to define a new named class that implements IComparer and
    /// pass an instance of that class as the first argument. For example:
    ///      type MyComparer() = 
    ///          interface IComparer&lt;string&gt; with 
    ///            member self.Compare(x,y) = ...
    ///
    /// let MyStringSetProvider = Set.MakeTagged(new MyComparer())
    val MakeTagged: ('tag :> IComparer<'a>) -> Provider<'a,'tag>

    /// Sets with a user-defined comparison function.
    [<Obsolete("This type has been renamed. Consider using Tagged.Set<'a> instead")>]
    type CSet<'a> = Tagged.Set<'a>
    [<Obsolete("This type has been renamed. Consider using Set.Provider<'a> instead")>]
    type CSetOps<'a> = Provider<'a>


