<?php

namespace {

	/**
	 * <p>This class or <code>CURLStringFile</code> should be used to upload a file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
	 * <p>Unserialization of <b>CURLFile</b> instances is not allowed. As of PHP 7.4.0, serialization is forbidden in the first place.</p>
	 * @link https://php.net/manual/en/class.curlfile.php
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	class CURLFile {

		/**
		 * @var string <p>Name of the file to be uploaded.</p>
		 * @link https://php.net/manual/en/class.curlfile.php#curlfile.props.name
		 */
		public $name = "";

		/**
		 * @var string <p>MIME type of the file (default is <code>application/octet-stream</code>).</p>
		 * @link https://php.net/manual/en/class.curlfile.php#curlfile.props.mime
		 */
		public $mime = "";

		/**
		 * @var string <p>The name of the file in the upload data (defaults to the name property).</p>
		 * @link https://php.net/manual/en/class.curlfile.php#curlfile.props.postname
		 */
		public $postname = "";

		/**
		 * Create a CURLFile object
		 * <p>Object-oriented style</p><p>Creates a <code>CURLFile</code> object, used to upload a file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
		 * @param string $filename <p>Path to the file which will be uploaded.</p>
		 * @param ?string $mime_type <p>Mimetype of the file.</p>
		 * @param ?string $posted_filename <p>Name of the file to be used in the upload data.</p>
		 * @return CURLFile <p>Returns a <code>CURLFile</code> object.</p>
		 * @link https://php.net/manual/en/curlfile.construct.php
		 * @see curl_setopt()
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function __construct(string $filename, ?string $mime_type = null, ?string $posted_filename = null) {}

		/**
		 * Get file name
		 * @return string <p>Returns file name.</p>
		 * @link https://php.net/manual/en/curlfile.getfilename.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function getFilename(): string {}

		/**
		 * Get MIME type
		 * @return string <p>Returns MIME type.</p>
		 * @link https://php.net/manual/en/curlfile.getmimetype.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function getMimeType(): string {}

		/**
		 * Get file name for POST
		 * @return string <p>Returns file name for POST.</p>
		 * @link https://php.net/manual/en/curlfile.getpostfilename.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function getPostFilename(): string {}

		/**
		 * Set MIME type
		 * @param string $mime_type <p>MIME type to be used in POST data.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/curlfile.setmimetype.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function setMimeType(string $mime_type): void {}

		/**
		 * Set file name for POST
		 * @param string $posted_filename <p>Filename to be used in POST data.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/curlfile.setpostfilename.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function setPostFilename(string $posted_filename): void {}
	}

	/**
	 * <p>A fully opaque class which replaces <code>curl</code> resources as of PHP 8.0.0.</p>
	 * @link https://php.net/manual/en/class.curlhandle.php
	 * @since PHP 8
	 */
	final class CurlHandle {
	}

	/**
	 * <p>A fully opaque class which replaces <code>curl_multi</code> resources as of PHP 8.0.0.</p>
	 * @link https://php.net/manual/en/class.curlmultihandle.php
	 * @since PHP 8
	 */
	final class CurlMultiHandle {
	}

	/**
	 * <p>A fully opaque class which replaces <code>curl_share</code> resources as of PHP 8.0.0.</p>
	 * @link https://php.net/manual/en/class.curlsharehandle.php
	 * @since PHP 8
	 */
	final class CurlShareHandle {
	}

	/**
	 * <p><b>CURLStringFile</b> makes it possible to upload a file directly from a variable. This is similar to <code>CURLFile</code>, but works with the contents of the file, not filename. This class or <code>CURLFile</code> should be used to upload the contents of the file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
	 * @link https://php.net/manual/en/class.curlstringfile.php
	 * @since PHP 8 >= 8.1.0
	 */
	class CURLStringFile {

		/**
		 * @var string <p>The contents to be uploaded.</p>
		 * @link https://php.net/manual/en/class.curlstringfile.php#curlstringfile.props.data
		 */
		public $data;

		/**
		 * @var string <p>The name of the file to be used in the upload data.</p>
		 * @link https://php.net/manual/en/class.curlstringfile.php#curlstringfile.props.postname
		 */
		public $postname;

		/**
		 * @var string <p>MIME type of the file (default is <code>application/octet-stream</code>).</p>
		 * @link https://php.net/manual/en/class.curlstringfile.php#curlstringfile.props.mime
		 */
		public $mime;

		/**
		 * Create a CURLStringFile object
		 * <p>Creates a <code>CURLStringFile</code> object, used to upload a file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
		 * @param string $data <p>The contents to be uploaded.</p>
		 * @param string $postname <p>The name of the file to be used in the upload data.</p>
		 * @param string $mime <p>MIME type of the file (default is <code>application/octet-stream</code>).</p>
		 * @return self
		 * @link https://php.net/manual/en/curlstringfile.construct.php
		 * @see curl_setopt()
		 * @since PHP 8 >= 8.1.0
		 */
		public function __construct(string $data, string $postname, string $mime = "application/octet-stream") {}
	}

	/**
	 * Close a cURL session
	 * <p><b>Note</b>:</p><p>This function has no effect. Prior to PHP 8.0.0, this function was used to close the resource.</p><p>Closes a cURL session and frees all resources. The cURL handle, <code>handle</code>, is also deleted.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.curl-close.php
	 * @see curl_init(), curl_multi_close()
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_close(\CurlHandle $handle): void {}

	/**
	 * Copy a cURL handle along with all of its preferences
	 * <p>Copies a cURL handle keeping the same preferences.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return CurlHandle|false <p>Returns a new cURL handle, or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-copy-handle.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_copy_handle(\CurlHandle $handle): \CurlHandle|false {}

	/**
	 * Return the last error number
	 * <p>Returns the error number for the last cURL operation.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return int <p>Returns the error number or <code>0</code> (zero) if no error occurred.</p>
	 * @link https://php.net/manual/en/function.curl-errno.php
	 * @see curl_error()
	 * @since PHP 4 >= 4.0.3, PHP 5, PHP 7, PHP 8
	 */
	function curl_errno(\CurlHandle $handle): int {}

	/**
	 * Return a string containing the last error for the current session
	 * <p>Returns a clear text error message for the last cURL operation.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return string <p>Returns the error message or <code>''</code> (the empty string) if no error occurred.</p>
	 * @link https://php.net/manual/en/function.curl-error.php
	 * @see curl_errno()
	 * @since PHP 4 >= 4.0.3, PHP 5, PHP 7, PHP 8
	 */
	function curl_error(\CurlHandle $handle): string {}

	/**
	 * URL encodes the given string
	 * <p>This function URL encodes the given string according to &#xBB;&#xA0;RFC 3986.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param string $string <p>The string to be encoded.</p>
	 * @return string|false <p>Returns escaped string or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-escape.php
	 * @see curl_unescape(), urlencode(), rawurlencode()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_escape(\CurlHandle $handle, string $string): string|false {}

	/**
	 * Perform a cURL session
	 * <p>Execute the given cURL session.</p><p>This function should be called after initializing a cURL session and all the options for the session are set.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return string|bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. However, if the <b><code>CURLOPT_RETURNTRANSFER</code></b> option is set, it will return the result on success, <b><code>false</code></b> on failure.</p><p><b>Warning</b></p><p>This function may return Boolean <b><code>false</code></b>, but may also return a non-Boolean value which evaluates to <b><code>false</code></b>. Please read the section on Booleans for more information. Use the === operator for testing the return value of this function.</p><p><b>Note</b>:</p><p>Note that response status codes which indicate errors (such as <code>404 Not found</code>) are not regarded as failure. <code>curl_getinfo()</code> can be used to check for these.</p>
	 * @link https://php.net/manual/en/function.curl-exec.php
	 * @see curl_multi_exec()
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_exec(\CurlHandle $handle): string|bool {}

	/**
	 * Create a CURLFile object
	 * <p>Object-oriented style</p><p>Creates a <code>CURLFile</code> object, used to upload a file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
	 * @param string $filename <p>Path to the file which will be uploaded.</p>
	 * @param ?string $mime_type <p>Mimetype of the file.</p>
	 * @param ?string $posted_filename <p>Name of the file to be used in the upload data.</p>
	 * @return CURLFile <p>Returns a <code>CURLFile</code> object.</p>
	 * @link https://php.net/manual/en/curlfile.construct.php
	 * @see curl_setopt()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_file_create(string $filename, ?string $mime_type = null, ?string $posted_filename = null): \CURLFile {}

	/**
	 * Get information regarding a specific transfer
	 * <p>Gets information about the last transfer.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param ?int $option <p>This may be one of the following constants:</p>   Option Description      <b><code>CURLINFO_CAINFO</code></b>   Default built-in CA certificate path     <b><code>CURLINFO_CAPATH</code></b>   Default built-in CA path string     <b><code>CURLINFO_EFFECTIVE_URL</code></b>   Last effective URL     <b><code>CURLINFO_HTTP_CODE</code></b>   The last response code. As of cURL 7.10.8, this is a legacy alias of CURLINFO_RESPONSE_CODE     <b><code>CURLINFO_FILETIME</code></b>   Remote time of the retrieved document, with the CURLOPT_FILETIME enabled; if -1 is returned the time of the document is unknown     <b><code>CURLINFO_TOTAL_TIME</code></b>   Total transaction time in seconds for last transfer     <b><code>CURLINFO_NAMELOOKUP_TIME</code></b>   Time in seconds until name resolving was complete     <b><code>CURLINFO_CONNECT_TIME</code></b>   Time in seconds it took to establish the connection     <b><code>CURLINFO_PRETRANSFER_TIME</code></b>   Time in seconds from start until just before file transfer begins     <b><code>CURLINFO_STARTTRANSFER_TIME</code></b>   Time in seconds until the first byte is about to be transferred     <b><code>CURLINFO_REDIRECT_COUNT</code></b>   Number of redirects, with the CURLOPT_FOLLOWLOCATION option enabled     <b><code>CURLINFO_REDIRECT_TIME</code></b>   Time in seconds of all redirection steps before final transaction was started, with the CURLOPT_FOLLOWLOCATION option enabled     <b><code>CURLINFO_REDIRECT_URL</code></b>   With the CURLOPT_FOLLOWLOCATION option disabled: redirect URL found in the last transaction, that should be requested manually next. With the CURLOPT_FOLLOWLOCATION option enabled: this is empty. The redirect URL in this case is available in CURLINFO_EFFECTIVE_URL     <b><code>CURLINFO_PRIMARY_IP</code></b>   IP address of the most recent connection     <b><code>CURLINFO_PRIMARY_PORT</code></b>   Destination port of the most recent connection     <b><code>CURLINFO_LOCAL_IP</code></b>   Local (source) IP address of the most recent connection     <b><code>CURLINFO_LOCAL_PORT</code></b>   Local (source) port of the most recent connection     <b><code>CURLINFO_SIZE_UPLOAD</code></b>   Total number of bytes uploaded     <b><code>CURLINFO_SIZE_DOWNLOAD</code></b>   Total number of bytes downloaded     <b><code>CURLINFO_SPEED_DOWNLOAD</code></b>   Average download speed     <b><code>CURLINFO_SPEED_UPLOAD</code></b>   Average upload speed     <b><code>CURLINFO_HEADER_SIZE</code></b>   Total size of all headers received     <b><code>CURLINFO_HEADER_OUT</code></b>   The request string sent. For this to work, add the CURLINFO_HEADER_OUT option to the handle by calling curl_setopt()     <b><code>CURLINFO_REFERER</code></b>   The referrer header     <b><code>CURLINFO_REQUEST_SIZE</code></b>   Total size of issued requests, currently only for HTTP requests     <b><code>CURLINFO_RETRY_AFTER</code></b>   The information from the Retry-After: header, or zero if there was no valid header.     <b><code>CURLINFO_SSL_VERIFYRESULT</code></b>   Result of SSL certification verification requested by setting CURLOPT_SSL_VERIFYPEER     <b><code>CURLINFO_CONTENT_LENGTH_DOWNLOAD</code></b>   Content length of download, read from Content-Length: field     <b><code>CURLINFO_CONTENT_LENGTH_UPLOAD</code></b>   Specified size of upload     <b><code>CURLINFO_CONTENT_TYPE</code></b>   Content-Type: of the requested document. NULL indicates server did not send valid Content-Type: header     <b><code>CURLINFO_PRIVATE</code></b>   Private data associated with this cURL handle, previously set with the CURLOPT_PRIVATE option of curl_setopt()     <b><code>CURLINFO_PROXY_ERROR</code></b>   The detailed (SOCKS) proxy error code when the most recent transfer returned a CURLE_PROXY error. The returned value will be exactly one of the CURLPX_&#42; values. The error code will be CURLPX_OK if no response code was available.     <b><code>CURLINFO_RESPONSE_CODE</code></b>   The last response code     <b><code>CURLINFO_HTTP_CONNECTCODE</code></b>   The CONNECT response code     <b><code>CURLINFO_HTTPAUTH_AVAIL</code></b>   Bitmask indicating the authentication method(s) available according to the previous response     <b><code>CURLINFO_PROXYAUTH_AVAIL</code></b>   Bitmask indicating the proxy authentication method(s) available according to the previous response     <b><code>CURLINFO_OS_ERRNO</code></b>   Errno from a connect failure. The number is OS and system specific.     <b><code>CURLINFO_NUM_CONNECTS</code></b>   Number of connections curl had to create to achieve the previous transfer     <b><code>CURLINFO_SSL_ENGINES</code></b>   OpenSSL crypto-engines supported     <b><code>CURLINFO_COOKIELIST</code></b>   All known cookies     <b><code>CURLINFO_FTP_ENTRY_PATH</code></b>   Entry path in FTP server     <b><code>CURLINFO_APPCONNECT_TIME</code></b>   Time in seconds it took from the start until the SSL/SSH connect/handshake to the remote host was completed     <b><code>CURLINFO_CERTINFO</code></b>   TLS certificate chain     <b><code>CURLINFO_CONDITION_UNMET</code></b>   Info on unmet time conditional     <b><code>CURLINFO_RTSP_CLIENT_CSEQ</code></b>   Next RTSP client CSeq     <b><code>CURLINFO_RTSP_CSEQ_RECV</code></b>   Recently received CSeq     <b><code>CURLINFO_RTSP_SERVER_CSEQ</code></b>   Next RTSP server CSeq     <b><code>CURLINFO_RTSP_SESSION_ID</code></b>   RTSP session ID     <b><code>CURLINFO_CONTENT_LENGTH_DOWNLOAD_T</code></b>   The content-length of the download. This is the value read from the Content-Length: field. -1 if the size isn't known     <b><code>CURLINFO_CONTENT_LENGTH_UPLOAD_T</code></b>   The specified size of the upload. -1 if the size isn't known     <b><code>CURLINFO_HTTP_VERSION</code></b>   The version used in the last HTTP connection. The return value will be one of the defined CURL_HTTP_VERSION_&#42; constants or 0 if the version can't be determined     <b><code>CURLINFO_PROTOCOL</code></b>   The protocol used in the last HTTP connection. The returned value will be exactly one of the CURLPROTO_&#42; values     <b><code>CURLINFO_PROXY_SSL_VERIFYRESULT</code></b>   The result of the certificate verification that was requested (using the CURLOPT_PROXY_SSL_VERIFYPEER option). Only used for HTTPS proxies     <b><code>CURLINFO_SCHEME</code></b>   The URL scheme used for the most recent connection     <b><code>CURLINFO_SIZE_DOWNLOAD_T</code></b>   Total number of bytes that were downloaded. The number is only for the latest transfer and will be reset again for each new transfer     <b><code>CURLINFO_SIZE_UPLOAD_T</code></b>   Total number of bytes that were uploaded     <b><code>CURLINFO_SPEED_DOWNLOAD_T</code></b>   The average download speed in bytes/second that curl measured for the complete download     <b><code>CURLINFO_SPEED_UPLOAD_T</code></b>   The average upload speed in bytes/second that curl measured for the complete upload     <b><code>CURLINFO_APPCONNECT_TIME_T</code></b>   Time, in microseconds, it took from the start until the SSL/SSH connect/handshake to the remote host was completed     <b><code>CURLINFO_CONNECT_TIME_T</code></b>   Total time taken, in microseconds, from the start until the connection to the remote host (or proxy) was completed     <b><code>CURLINFO_FILETIME_T</code></b>   Remote time of the retrieved document (as Unix timestamp), an alternative to CURLINFO_FILETIME to allow systems with 32 bit long variables to extract dates outside of the 32bit timestamp range     <b><code>CURLINFO_NAMELOOKUP_TIME_T</code></b>   Time in microseconds from the start until the name resolving was completed     <b><code>CURLINFO_PRETRANSFER_TIME_T</code></b>   Time taken from the start until the file transfer is just about to begin, in microseconds     <b><code>CURLINFO_REDIRECT_TIME_T</code></b>   Total time, in microseconds, it took for all redirection steps include name lookup, connect, pretransfer and transfer before final transaction was started     <b><code>CURLINFO_STARTTRANSFER_TIME_T</code></b>   Time, in microseconds, it took from the start until the first byte is received     <b><code>CURLINFO_TOTAL_TIME_T</code></b>   Total time in microseconds for the previous transfer, including name resolving, TCP connect etc.
	 * @return mixed <p>If <code>option</code> is given, returns its value. Otherwise, returns an associative array with the following elements (which correspond to <code>option</code>), or <b><code>false</code></b> on failure:</p><ul> <li>  "url"  </li> <li>  "content_type"  </li> <li>  "http_code"  </li> <li>  "header_size"  </li> <li>  "request_size"  </li> <li>  "filetime"  </li> <li>  "ssl_verify_result"  </li> <li>  "redirect_count"  </li> <li>  "total_time"  </li> <li>  "namelookup_time"  </li> <li>  "connect_time"  </li> <li>  "pretransfer_time"  </li> <li>  "size_upload"  </li> <li>  "size_download"  </li> <li>  "speed_download"  </li> <li>  "speed_upload"  </li> <li>  "download_content_length"  </li> <li>  "upload_content_length"  </li> <li>  "starttransfer_time"  </li> <li>  "redirect_time"  </li> <li>  "certinfo"  </li> <li>  "primary_ip"  </li> <li>  "primary_port"  </li> <li>  "local_ip"  </li> <li>  "local_port"  </li> <li>  "redirect_url"  </li> <li>  "request_header" (This is only set if the <b><code>CURLINFO_HEADER_OUT</code></b> is set by a previous call to <code>curl_setopt()</code>)  </li> </ul> Note that private data is not included in the associative array and must be retrieved individually with the <b><code>CURLINFO_PRIVATE</code></b> option.
	 * @link https://php.net/manual/en/function.curl-getinfo.php
	 * @since PHP 4 >= 4.0.4, PHP 5, PHP 7, PHP 8
	 */
	function curl_getinfo(\CurlHandle $handle, ?int $option = null): mixed {}

	/**
	 * Initialize a cURL session
	 * <p>Initializes a new session and return a cURL handle for use with the <code>curl_setopt()</code>, <code>curl_exec()</code>, and <code>curl_close()</code> functions.</p>
	 * @param ?string $url <p>If provided, the <b><code>CURLOPT_URL</code></b> option will be set to its value. You can manually set this using the <code>curl_setopt()</code> function.</p> <p><b>Note</b>:</p><p>The <code>file</code> protocol is disabled by cURL if open_basedir is set.</p>
	 * @return CurlHandle|false <p>Returns a cURL handle on success, <b><code>false</code></b> on errors.</p>
	 * @link https://php.net/manual/en/function.curl-init.php
	 * @see curl_close(), curl_multi_init()
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_init(?string $url = null): \CurlHandle|false {}

	/**
	 * Add a normal cURL handle to a cURL multi handle
	 * <p>Adds the <code>handle</code> handle to the multi handle <code>multi_handle</code></p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return int <p>Returns 0 on success, or one of the <b><code>CURLM_XXX</code></b> errors code.</p>
	 * @link https://php.net/manual/en/function.curl-multi-add-handle.php
	 * @see curl_multi_remove_handle(), curl_multi_init(), curl_init()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_add_handle(\CurlMultiHandle $multi_handle, \CurlHandle $handle): int {}

	/**
	 * Close a set of cURL handles
	 * <p><b>Note</b>:</p><p>This function has no effect. Prior to PHP 8.0.0, this function was used to close the resource.</p><p>Closes a set of cURL handles.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.curl-multi-close.php
	 * @see curl_multi_init(), curl_close()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_close(\CurlMultiHandle $multi_handle): void {}

	/**
	 * Return the last multi curl error number
	 * <p>Return an integer containing the last multi curl error number.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @return int <p>Return an integer containing the last multi curl error number.</p>
	 * @link https://php.net/manual/en/function.curl-multi-errno.php
	 * @see curl_errno()
	 * @since PHP 7 >= 7.1.0, PHP 8
	 */
	function curl_multi_errno(\CurlMultiHandle $multi_handle): int {}

	/**
	 * Run the sub-connections of the current cURL handle
	 * <p>Processes each of the handles in the stack. This method can be called whether or not a handle needs to read or write data.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param int $still_running <p>A reference to a flag to tell whether the operations are still running.</p>
	 * @return int <p>A cURL code defined in the cURL Predefined Constants.</p><p><b>Note</b>:</p><p>This only returns errors regarding the whole multi stack. There might still have occurred problems on individual transfers even when this function returns <b><code>CURLM_OK</code></b>.</p>
	 * @link https://php.net/manual/en/function.curl-multi-exec.php
	 * @see curl_multi_init(), curl_multi_select(), curl_exec()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_exec(\CurlMultiHandle $multi_handle, int &$still_running): int {}

	/**
	 * Return the content of a cURL handle if CURLOPT_RETURNTRANSFER is set
	 * <p>If <b><code>CURLOPT_RETURNTRANSFER</code></b> is an option that is set for a specific handle, then this function will return the content of that cURL handle in the form of a string.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return ?string <p>Return the content of a cURL handle if <b><code>CURLOPT_RETURNTRANSFER</code></b> is set or <b><code>null</code></b> if not set.</p>
	 * @link https://php.net/manual/en/function.curl-multi-getcontent.php
	 * @see curl_multi_init()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_getcontent(\CurlHandle $handle): ?string {}

	/**
	 * Get information about the current transfers
	 * <p>Ask the multi handle if there are any messages or information from the individual transfers. Messages may include information such as an error code from the transfer or just the fact that a transfer is completed.</p><p>Repeated calls to this function will return a new result each time, until a <b><code>false</code></b> is returned as a signal that there is no more to get at this point. The integer pointed to with <code>queued_messages</code> will contain the number of remaining messages after this function was called.</p><p>The data the returned resource points to will not survive calling <code>curl_multi_remove_handle()</code>.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param int $queued_messages <p>Number of messages that are still in the queue</p>
	 * @return array|false <p>On success, returns an associative array for the message, <b><code>false</code></b> on failure.</p> <b>Contents of the returned array</b>   Key: Value:     <code>msg</code> The <b><code>CURLMSG_DONE</code></b> constant. Other return values are currently not available.   <code>result</code> One of the <b><code>CURLE_&#42;</code></b> constants. If everything is OK, the <b><code>CURLE_OK</code></b> will be the result.   <code>handle</code> Resource of type curl indicates the handle which it concerns.
	 * @link https://php.net/manual/en/function.curl-multi-info-read.php
	 * @see curl_multi_init()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_info_read(\CurlMultiHandle $multi_handle, int &$queued_messages = null): array|false {}

	/**
	 * Returns a new cURL multi handle
	 * <p>Allows the processing of multiple cURL handles asynchronously.</p>
	 * @return CurlMultiHandle <p>Returns a cURL multi handle.</p>
	 * @link https://php.net/manual/en/function.curl-multi-init.php
	 * @see curl_init(), curl_multi_close()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_init(): \CurlMultiHandle {}

	/**
	 * Remove a multi handle from a set of cURL handles
	 * <p>Removes a given <code>handle</code> handle from the given <code>multi_handle</code> handle. When the <code>handle</code> handle has been removed, it is again perfectly legal to run <code>curl_exec()</code> on this handle. Removing the <code>handle</code> handle while being used, will effectively halt the transfer in progress involving that handle.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return int <p>Returns 0 on success, or one of the <b><code>CURLM_XXX</code></b> error codes.</p>
	 * @link https://php.net/manual/en/function.curl-multi-remove-handle.php
	 * @see curl_init(), curl_multi_init(), curl_multi_add_handle()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_remove_handle(\CurlMultiHandle $multi_handle, \CurlHandle $handle): int {}

	/**
	 * Wait for activity on any curl_multi connection
	 * <p>Blocks until there is activity on any of the curl_multi connections.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param float $timeout <p>Time, in seconds, to wait for a response.</p>
	 * @return int <p>On success, returns the number of descriptors contained in the descriptor sets. This may be 0 if there was no activity on any of the descriptors. On failure, this function will return -1 on a select failure (from the underlying select system call).</p>
	 * @link https://php.net/manual/en/function.curl-multi-select.php
	 * @see curl_multi_init()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_select(\CurlMultiHandle $multi_handle, float $timeout = 1.0): int {}

	/**
	 * Set a cURL multi option
	 * <p>Sets an option on the given cURL multi handle.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param int $option <p>One of the <b><code>CURLMOPT_&#42;</code></b> constants.</p>
	 * @param mixed $value <p>The value to be set on <code>option</code>.</p> <p><code>value</code> should be an <code>int</code> for the following values of the <code>option</code> parameter:</p>   Option Set <code>value</code> to     <b><code>CURLMOPT_PIPELINING</code></b>  Pass 1 to enable or 0 to disable. Enabling pipelining on a multi handle will make it attempt to perform HTTP Pipelining as far as possible for transfers using this handle. This means that if you add a second request that can use an already existing connection, the second request will be "piped" on the same connection. As of cURL 7.43.0, the value is a bitmask, and you can also pass 2 to try to multiplex the new transfer over an existing HTTP/2 connection if possible. Passing 3 instructs cURL to ask for pipelining and multiplexing independently of each other. As of cURL 7.62.0, setting the pipelining bit has no effect. Instead of integer literals, you can also use the CURLPIPE_&#42; constants if available.    <b><code>CURLMOPT_MAXCONNECTS</code></b>  Pass a number that will be used as the maximum amount of simultaneously open connections that libcurl may cache. By default the size will be enlarged to fit four times the number of handles added via <code>curl_multi_add_handle()</code>. When the cache is full, curl closes the oldest one in the cache to prevent the number of open connections from increasing.    <b><code>CURLMOPT_CHUNK_LENGTH_PENALTY_SIZE</code></b>  Pass a number that specifies the chunk length threshold for pipelining in bytes.    <b><code>CURLMOPT_CONTENT_LENGTH_PENALTY_SIZE</code></b>  Pass a number that specifies the size threshold for pipelining penalty in bytes.    <b><code>CURLMOPT_MAX_CONCURRENT_STREAMS</code></b>  The set number will be used as the maximum number of concurrent streams for a connections that cURL should support on connections done using HTTP/2. Valid values range from <code>1</code> to <code>2147483647</code> (<code>2^31 - 1</code>). The value passed here would be honored based on other system resources properties. Default is <code>100</code>.    <b><code>CURLMOPT_MAX_HOST_CONNECTIONS</code></b>  Pass a number that specifies the maximum number of connections to a single host.    <b><code>CURLMOPT_MAX_PIPELINE_LENGTH</code></b>  Pass a number that specifies the maximum number of requests in a pipeline.    <b><code>CURLMOPT_MAX_TOTAL_CONNECTIONS</code></b>  Pass a number that specifies the maximum number of simultaneously open connections.    <b><code>CURLMOPT_PUSHFUNCTION</code></b>  Pass a <code>callable</code> that will be registered to handle server pushes and should have the following signature:  pushfunction(<code>resource</code> <code>$parent_ch</code>, <code>resource</code> <code>$pushed_ch</code>, <code>array</code> <code>$headers</code>): <code>int</code>   <code>parent_ch</code>  <p>The parent cURL handle (the request the client made).</p>   <code>pushed_ch</code>  <p>A new cURL handle for the pushed request.</p>   <code>headers</code>  <p>The push promise headers.</p>   The push function is supposed to return either <b><code>CURL_PUSH_OK</code></b> if it can handle the push, or <b><code>CURL_PUSH_DENY</code></b> to reject it.
	 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-multi-setopt.php
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_multi_setopt(\CurlMultiHandle $multi_handle, int $option, mixed $value): bool {}

	/**
	 * Return string describing error code
	 * <p>Returns a text error message describing the given CURLM error code.</p>
	 * @param int $error_code <p>One of the &#xBB;&#xA0;CURLM error codes constants.</p>
	 * @return ?string <p>Returns error string for valid error code, <b><code>null</code></b> otherwise.</p>
	 * @link https://php.net/manual/en/function.curl-multi-strerror.php
	 * @see curl_strerror()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_multi_strerror(int $error_code): ?string {}

	/**
	 * Pause and unpause a connection
	 * <p>Pause or unpause a cURL session. A session can be paused while a transfer is in progress, in either the read, write or both directions, by calling this function from a callback registered with <code>curl_setopt()</code>.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param int $flags <p>One of <b><code>CURLPAUSE_&#42;</code></b> constants.</p>
	 * @return int <p>Returns an error code (<b><code>CURLE_OK</code></b> for no error).</p>
	 * @link https://php.net/manual/en/function.curl-pause.php
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_pause(\CurlHandle $handle, int $flags): int {}

	/**
	 * Reset all options of a libcurl session handle
	 * <p>This function re-initializes all options set on the given cURL handle to the default values.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.curl-reset.php
	 * @see curl_setopt()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_reset(\CurlHandle $handle): void {}

	/**
	 * Set an option for a cURL transfer
	 * <p>Sets an option on the given cURL session handle.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param int $option <p>The <code>CURLOPT_XXX</code> option to set.</p>
	 * @param mixed $value <p>The value to be set on <code>option</code>.</p> <p><code>value</code> should be a <code>bool</code> for the following values of the <code>option</code> parameter:</p>   Option Set <code>value</code> to Notes     <b><code>CURLOPT_AUTOREFERER</code></b>  <b><code>true</code></b> to automatically set the <code>Referer:</code> field in requests where it follows a <code>Location:</code> redirect.      <b><code>CURLOPT_COOKIESESSION</code></b>  <b><code>true</code></b> to mark this as a new cookie "session". It will force libcurl to ignore all cookies it is about to load that are "session cookies" from the previous session. By default, libcurl always stores and loads all cookies, independent if they are session cookies or not. Session cookies are cookies without expiry date and they are meant to be alive and existing for this "session" only.      <b><code>CURLOPT_CERTINFO</code></b>  <b><code>true</code></b> to output SSL certification information to <code>STDERR</code> on secure transfers.   Added in cURL 7.19.1. Requires <b><code>CURLOPT_VERBOSE</code></b> to be on to have an effect.    <b><code>CURLOPT_CONNECT_ONLY</code></b>  <b><code>true</code></b> tells the library to perform all the required proxy authentication and connection setup, but no data transfer. This option is implemented for HTTP, SMTP and POP3.   Added in 7.15.2.    <b><code>CURLOPT_CRLF</code></b>  <b><code>true</code></b> to convert Unix newlines to CRLF newlines on transfers.      <b><code>CURLOPT_DISALLOW_USERNAME_IN_URL</code></b>  <b><code>true</code></b> to not allow URLs that include a username. Usernames are allowed by default (0).   Added in cURL 7.61.0. Available since PHP 7.3.0.    <b><code>CURLOPT_DNS_SHUFFLE_ADDRESSES</code></b>  <b><code>true</code></b> to shuffle the order of all returned addresses so that they will be used in a random order, when a name is resolved and more than one IP address is returned. This may cause IPv4 to be used before IPv6 or vice versa.   Added in cURL 7.60.0. Available since PHP 7.3.0.    <b><code>CURLOPT_HAPROXYPROTOCOL</code></b>  <b><code>true</code></b> to send an HAProxy PROXY protocol v1 header at the start of the connection. The default action is not to send this header.   Added in cURL 7.60.0. Available since PHP 7.3.0.    <b><code>CURLOPT_SSH_COMPRESSION</code></b>  <b><code>true</code></b> to enable built-in SSH compression. This is a request, not an order; the server may or may not do it.   Added in cURL 7.56.0. Available since PHP 7.3.0.    <b><code>CURLOPT_DNS_USE_GLOBAL_CACHE</code></b>  <b><code>true</code></b> to use a global DNS cache. This option is not thread-safe. It is conditionally enabled by default if PHP is built for non-threaded use (CLI, FCGI, Apache2-Prefork, etc.).      <b><code>CURLOPT_FAILONERROR</code></b>  <b><code>true</code></b> to fail verbosely if the HTTP code returned is greater than or equal to 400. The default behavior is to return the page normally, ignoring the code.      <b><code>CURLOPT_SSL_FALSESTART</code></b>  <b><code>true</code></b> to enable TLS false start.   Added in cURL 7.42.0. Available since PHP 7.0.7.    <b><code>CURLOPT_FILETIME</code></b>  <b><code>true</code></b> to attempt to retrieve the modification date of the remote document. This value can be retrieved using the <b><code>CURLINFO_FILETIME</code></b> option with <code>curl_getinfo()</code>.      <b><code>CURLOPT_FOLLOWLOCATION</code></b>  <b><code>true</code></b> to follow any <code>"Location: "</code> header that the server sends as part of the HTTP header. See also <b><code>CURLOPT_MAXREDIRS</code></b>.      <b><code>CURLOPT_FORBID_REUSE</code></b>  <b><code>true</code></b> to force the connection to explicitly close when it has finished processing, and not be pooled for reuse.      <b><code>CURLOPT_FRESH_CONNECT</code></b>  <b><code>true</code></b> to force the use of a new connection instead of a cached one.      <b><code>CURLOPT_FTP_USE_EPRT</code></b>  <b><code>true</code></b> to use EPRT (and LPRT) when doing active FTP downloads. Use <b><code>false</code></b> to disable EPRT and LPRT and use PORT only.      <b><code>CURLOPT_FTP_USE_EPSV</code></b>  <b><code>true</code></b> to first try an EPSV command for FTP transfers before reverting back to PASV. Set to <b><code>false</code></b> to disable EPSV.      <b><code>CURLOPT_FTP_CREATE_MISSING_DIRS</code></b>  <b><code>true</code></b> to create missing directories when an FTP operation encounters a path that currently doesn't exist.      <b><code>CURLOPT_FTPAPPEND</code></b>  <b><code>true</code></b> to append to the remote file instead of overwriting it.      <b><code>CURLOPT_TCP_NODELAY</code></b>  <b><code>true</code></b> to disable TCP's Nagle algorithm, which tries to minimize the number of small packets on the network.   Available for versions compiled with libcurl 7.11.2 or greater.    <b><code>CURLOPT_FTPASCII</code></b>  An alias of <b><code>CURLOPT_TRANSFERTEXT</code></b>. Use that instead.      <b><code>CURLOPT_FTPLISTONLY</code></b>  <b><code>true</code></b> to only list the names of an FTP directory.      <b><code>CURLOPT_HEADER</code></b>  <b><code>true</code></b> to include the header in the output.      <b><code>CURLINFO_HEADER_OUT</code></b>  <b><code>true</code></b> to track the handle's request string.   The <b><code>CURLINFO_</code></b> prefix is intentional.    <b><code>CURLOPT_HTTP09_ALLOWED</code></b>  Whether to allow HTTP/0.9 responses. Defaults to <b><code>false</code></b> as of libcurl 7.66.0; formerly it defaulted to <b><code>true</code></b>.   Available since PHP 7.3.15 and 7.4.3, respectively, if built against libcurl &gt;= 7.64.0    <b><code>CURLOPT_HTTPGET</code></b>  <b><code>true</code></b> to reset the HTTP request method to GET. Since GET is the default, this is only necessary if the request method has been changed.      <b><code>CURLOPT_HTTPPROXYTUNNEL</code></b>  <b><code>true</code></b> to tunnel through a given HTTP proxy.      <b><code>CURLOPT_HTTP_CONTENT_DECODING</code></b>  <b><code>false</code></b> to get the raw HTTP response body.   Available if built against libcurl &gt;= 7.16.2.    <b><code>CURLOPT_KEEP_SENDING_ON_ERROR</code></b>  <b><code>true</code></b> to keep sending the request body if the HTTP code returned is equal to or larger than 300. The default action would be to stop sending and close the stream or connection. Suitable for manual NTLM authentication. Most applications do not need this option.   Available as of PHP 7.3.0 if built against libcurl &gt;= 7.51.0.    <b><code>CURLOPT_MUTE</code></b>  <b><code>true</code></b> to be completely silent with regards to the cURL functions.   Removed in cURL 7.15.5 (You can use CURLOPT_RETURNTRANSFER instead)    <b><code>CURLOPT_NETRC</code></b>  <b><code>true</code></b> to scan the ~/.netrc file to find a username and password for the remote site that a connection is being established with.      <b><code>CURLOPT_NOBODY</code></b>  <b><code>true</code></b> to exclude the body from the output. Request method is then set to HEAD. Changing this to <b><code>false</code></b> does not change it to GET.      <b><code>CURLOPT_NOPROGRESS</code></b> <p><b><code>true</code></b> to disable the progress meter for cURL transfers.</p><p><b>Note</b>:</p><p>PHP automatically sets this option to <b><code>true</code></b>, this should only be changed for debugging purposes.</p>       <b><code>CURLOPT_NOSIGNAL</code></b>  <b><code>true</code></b> to ignore any cURL function that causes a signal to be sent to the PHP process. This is turned on by default in multi-threaded SAPIs so timeout options can still be used.   Added in cURL 7.10.    <b><code>CURLOPT_PATH_AS_IS</code></b>  <b><code>true</code></b> to not handle dot dot sequences.   Added in cURL 7.42.0. Available since PHP 7.0.7.    <b><code>CURLOPT_PIPEWAIT</code></b>  <b><code>true</code></b> to wait for pipelining/multiplexing.   Added in cURL 7.43.0. Available since PHP 7.0.7.    <b><code>CURLOPT_POST</code></b>  <b><code>true</code></b> to do a regular HTTP POST. This POST is the normal <code>application/x-www-form-urlencoded</code> kind, most commonly used by HTML forms.      <b><code>CURLOPT_PUT</code></b>  <b><code>true</code></b> to HTTP PUT a file. The file to PUT must be set with <b><code>CURLOPT_INFILE</code></b> and <b><code>CURLOPT_INFILESIZE</code></b>.      <b><code>CURLOPT_RETURNTRANSFER</code></b>  <b><code>true</code></b> to return the transfer as a string of the return value of <code>curl_exec()</code> instead of outputting it directly.      <b><code>CURLOPT_SASL_IR</code></b>  <b><code>true</code></b> to enable sending the initial response in the first packet.   Added in cURL 7.31.10. Available since PHP 7.0.7.    <b><code>CURLOPT_SSL_ENABLE_ALPN</code></b>  <b><code>false</code></b> to disable ALPN in the SSL handshake (if the SSL backend libcurl is built to use supports it), which can be used to negotiate http2.   Added in cURL 7.36.0. Available since PHP 7.0.7.    <b><code>CURLOPT_SSL_ENABLE_NPN</code></b>  <b><code>false</code></b> to disable NPN in the SSL handshake (if the SSL backend libcurl is built to use supports it), which can be used to negotiate http2.   Added in cURL 7.36.0. Available since PHP 7.0.7.    <b><code>CURLOPT_SSL_VERIFYPEER</code></b>  <b><code>false</code></b> to stop cURL from verifying the peer's certificate. Alternate certificates to verify against can be specified with the <b><code>CURLOPT_CAINFO</code></b> option or a certificate directory can be specified with the <b><code>CURLOPT_CAPATH</code></b> option.   <b><code>true</code></b> by default as of cURL 7.10. Default bundle installed as of cURL 7.10.    <b><code>CURLOPT_SSL_VERIFYSTATUS</code></b>  <b><code>true</code></b> to verify the certificate's status.   Added in cURL 7.41.0. Available since PHP 7.0.7.    <b><code>CURLOPT_PROXY_SSL_VERIFYPEER</code></b>  <b><code>false</code></b> to stop cURL from verifying the peer's certificate. Alternate certificates to verify against can be specified with the <b><code>CURLOPT_CAINFO</code></b> option or a certificate directory can be specified with the <b><code>CURLOPT_CAPATH</code></b> option. When set to false, the peer certificate verification succeeds regardless.   <b><code>true</code></b> by default. Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_SAFE_UPLOAD</code></b>  Always <b><code>true</code></b>, what disables support for the <code>@</code> prefix for uploading files in <b><code>CURLOPT_POSTFIELDS</code></b>, which means that values starting with <code>@</code> can be safely passed as fields. <code>CURLFile</code> may be used for uploads instead.      <b><code>CURLOPT_SUPPRESS_CONNECT_HEADERS</code></b>  <b><code>true</code></b> to suppress proxy CONNECT response headers from the user callback functions <b><code>CURLOPT_HEADERFUNCTION</code></b> and <b><code>CURLOPT_WRITEFUNCTION</code></b>, when <b><code>CURLOPT_HTTPPROXYTUNNEL</code></b> is used and a CONNECT request is made.   Added in cURL 7.54.0. Available since PHP 7.3.0.    <b><code>CURLOPT_TCP_FASTOPEN</code></b>  <b><code>true</code></b> to enable TCP Fast Open.   Added in cURL 7.49.0. Available since PHP 7.0.7.    <b><code>CURLOPT_TFTP_NO_OPTIONS</code></b>  <b><code>true</code></b> to not send TFTP options requests.   Added in cURL 7.48.0. Available since PHP 7.0.7.    <b><code>CURLOPT_TRANSFERTEXT</code></b>  <b><code>true</code></b> to use ASCII mode for FTP transfers. For LDAP, it retrieves data in plain text instead of HTML. On Windows systems, it will not set <code>STDOUT</code> to binary mode.      <b><code>CURLOPT_UNRESTRICTED_AUTH</code></b>  <b><code>true</code></b> to keep sending the username and password when following locations (using <b><code>CURLOPT_FOLLOWLOCATION</code></b>), even when the hostname has changed.      <b><code>CURLOPT_UPLOAD</code></b>  <b><code>true</code></b> to prepare for an upload.      <b><code>CURLOPT_VERBOSE</code></b>  <b><code>true</code></b> to output verbose information. Writes output to <code>STDERR</code>, or the file specified using <b><code>CURLOPT_STDERR</code></b>.       <p><code>value</code> should be an <code>int</code> for the following values of the <code>option</code> parameter:</p>   Option Set <code>value</code> to Notes     <b><code>CURLOPT_BUFFERSIZE</code></b>  The size of the buffer to use for each read. There is no guarantee this request will be fulfilled, however.   Added in cURL 7.10.    <b><code>CURLOPT_CONNECTTIMEOUT</code></b>  The number of seconds to wait while trying to connect. Use 0 to wait indefinitely.      <b><code>CURLOPT_CONNECTTIMEOUT_MS</code></b>  The number of milliseconds to wait while trying to connect. Use 0 to wait indefinitely. If libcurl is built to use the standard system name resolver, that portion of the connect will still use full-second resolution for timeouts with a minimum timeout allowed of one second.   Added in cURL 7.16.2.    <b><code>CURLOPT_DNS_CACHE_TIMEOUT</code></b>  The number of seconds to keep DNS entries in memory. This option is set to 120 (2 minutes) by default.      <b><code>CURLOPT_EXPECT_100_TIMEOUT_MS</code></b>  The timeout for <code>Expect: 100-continue</code> responses in milliseconds. Defaults to 1000 milliseconds.   Added in cURL 7.36.0. Available since PHP 7.0.7.    <b><code>CURLOPT_HAPPY_EYEBALLS_TIMEOUT_MS</code></b>  Head start for ipv6 for the happy eyeballs algorithm. Happy eyeballs attempts to connect to both IPv4 and IPv6 addresses for dual-stack hosts, preferring IPv6 first for timeout milliseconds. Defaults to CURL_HET_DEFAULT, which is currently 200 milliseconds.   Added in cURL 7.59.0. Available since PHP 7.3.0.    <b><code>CURLOPT_FTPSSLAUTH</code></b>  The FTP authentication method (when is activated): <code>CURLFTPAUTH_SSL</code> (try SSL first), <code>CURLFTPAUTH_TLS</code> (try TLS first), or <code>CURLFTPAUTH_DEFAULT</code> (let cURL decide).   Added in cURL 7.12.2.    <b><code>CURLOPT_HEADEROPT</code></b>  How to deal with headers. One of the following constants:  <b><code>CURLHEADER_UNIFIED</code></b>: the headers specified in <b><code>CURLOPT_HTTPHEADER</code></b> will be used in requests both to servers and proxies. With this option enabled, <b><code>CURLOPT_PROXYHEADER</code></b> will not have any effect.   <b><code>CURLHEADER_SEPARATE</code></b>: makes <b><code>CURLOPT_HTTPHEADER</code></b> headers only get sent to a server and not to a proxy. Proxy headers must be set with <b><code>CURLOPT_PROXYHEADER</code></b> to get used. Note that if a non-CONNECT request is sent to a proxy, libcurl will send both server headers and proxy headers. When doing CONNECT, libcurl will send <b><code>CURLOPT_PROXYHEADER</code></b> headers only to the proxy and then <b><code>CURLOPT_HTTPHEADER</code></b> headers only to the server.   Defaults to <b><code>CURLHEADER_SEPARATE</code></b> as of cURL 7.42.1, and <b><code>CURLHEADER_UNIFIED</code></b> before.    Added in cURL 7.37.0. Available since PHP 7.0.7.    <b><code>CURLOPT_HTTP_VERSION</code></b>  <b><code>CURL_HTTP_VERSION_NONE</code></b> (default, lets CURL decide which version to use), <b><code>CURL_HTTP_VERSION_1_0</code></b> (forces HTTP/1.0), <b><code>CURL_HTTP_VERSION_1_1</code></b> (forces HTTP/1.1), <b><code>CURL_HTTP_VERSION_2_0</code></b> (attempts HTTP 2), <b><code>CURL_HTTP_VERSION_2</code></b> (alias of <b><code>CURL_HTTP_VERSION_2_0</code></b>), <b><code>CURL_HTTP_VERSION_2TLS</code></b> (attempts HTTP 2 over TLS (HTTPS) only) or <b><code>CURL_HTTP_VERSION_2_PRIOR_KNOWLEDGE</code></b> (issues non-TLS HTTP requests using HTTP/2 without HTTP/1.1 Upgrade).      <b><code>CURLOPT_HTTPAUTH</code></b>  <p>The HTTP authentication method(s) to use. The options are: <b><code>CURLAUTH_BASIC</code></b>, <b><code>CURLAUTH_DIGEST</code></b>, <b><code>CURLAUTH_GSSNEGOTIATE</code></b>, <b><code>CURLAUTH_NTLM</code></b>, <b><code>CURLAUTH_AWS_SIGV4</code></b>, <b><code>CURLAUTH_ANY</code></b>, and <b><code>CURLAUTH_ANYSAFE</code></b>.</p> <p>The bitwise <code>|</code> (or) operator can be used to combine more than one method. If this is done, cURL will poll the server to see what methods it supports and pick the best one.</p> <p><b><code>CURLAUTH_ANY</code></b> sets all bits. cURL will automatically select the one it finds most secure.</p> <p><b><code>CURLAUTH_ANYSAFE</code></b> sets all bits except <code>CURLAUTH_BASIC</code>. cURL will automatically select the one it finds most secure.</p>      <b><code>CURLOPT_INFILESIZE</code></b>  The expected size, in bytes, of the file when uploading a file to a remote site. Note that using this option will not stop libcurl from sending more data, as exactly what is sent depends on <b><code>CURLOPT_READFUNCTION</code></b>.      <b><code>CURLOPT_LOW_SPEED_LIMIT</code></b>  The transfer speed, in bytes per second, that the transfer should be below during the count of <b><code>CURLOPT_LOW_SPEED_TIME</code></b> seconds before PHP considers the transfer too slow and aborts.      <b><code>CURLOPT_LOW_SPEED_TIME</code></b>  The number of seconds the transfer speed should be below <b><code>CURLOPT_LOW_SPEED_LIMIT</code></b> before PHP considers the transfer too slow and aborts.      <b><code>CURLOPT_MAIL_RCPT_ALLLOWFAILS</code></b>  Allow RCPT TO command to fail for some recipients.   When sending data to multiple recipients, by default cURL will abort SMTP conversation if at least one of the recipients causes RCPT TO command to return an error. This option tells cURL to ignore errors and proceed with the remaining valid recipients. If all recipients trigger RCPT TO failures and this flag is set, cURL will abort the SMTP conversation and return the error received from the last RCPT TO command.      <b><code>CURLOPT_MAXAGE_CONN</code></b>  The maximum idle time allowed for an existing connection to be considered for reuse. Default maximum age is set to <code>118</code> seconds.      <b><code>CURLOPT_MAXFILESIZE_LARGE</code></b>  The maximum file size in bytes allowed to download. If the file requested is found larger than this value, the transfer will not start and <b><code>CURLE_FILESIZE_EXCEEDED</code></b> will be returned. The file size is not always known prior to download, and for such files this option has no effect even if the file transfer ends up being larger than this given limit.      <b><code>CURLOPT_MAXLIFETIME_CONN</code></b>  The maximum time in seconds, since the creation of the connection, that is allowed for an existing connection to have for it to be considered for reuse. If a connection is found in the cache that is older than this value, it will instead be closed once any in-progress transfers are complete. Default is 0 seconds, meaning the option is disabled and all connections are eligible for reuse.      <b><code>CURLOPT_MAXCONNECTS</code></b>  The maximum amount of persistent connections that are allowed. When the limit is reached, the oldest one in the cache is closed to prevent increasing the number of open connections.      <b><code>CURLOPT_MAXREDIRS</code></b>  The maximum amount of HTTP redirections to follow. Use this option alongside <b><code>CURLOPT_FOLLOWLOCATION</code></b>. Default value of <code>20</code> is set to prevent infinite redirects. Setting to <code>-1</code> allows inifinite redirects, and <code>0</code> refuses all redirects.      <b><code>CURLOPT_PORT</code></b>  An alternative port number to connect to.      <b><code>CURLOPT_POSTREDIR</code></b>  A bitmask of 1 (301 Moved Permanently), 2 (302 Found) and 4 (303 See Other) if the HTTP POST method should be maintained when <b><code>CURLOPT_FOLLOWLOCATION</code></b> is set and a specific type of redirect occurs.   Added in cURL 7.19.1.    <b><code>CURLOPT_PROTOCOLS</code></b>  <p>Bitmask of <b><code>CURLPROTO_&#42;</code></b> values. If used, this bitmask limits what protocols libcurl may use in the transfer. This allows you to have a libcurl built to support a wide range of protocols but still limit specific transfers to only be allowed to use a subset of them. By default libcurl will accept all protocols it supports. See also <b><code>CURLOPT_REDIR_PROTOCOLS</code></b>.</p> <p>Valid protocol options are: <b><code>CURLPROTO_HTTP</code></b>, <b><code>CURLPROTO_HTTPS</code></b>, <b><code>CURLPROTO_FTP</code></b>, <b><code>CURLPROTO_FTPS</code></b>, <b><code>CURLPROTO_SCP</code></b>, <b><code>CURLPROTO_SFTP</code></b>, <b><code>CURLPROTO_TELNET</code></b>, <b><code>CURLPROTO_LDAP</code></b>, <b><code>CURLPROTO_LDAPS</code></b>, <b><code>CURLPROTO_DICT</code></b>, <b><code>CURLPROTO_FILE</code></b>, <b><code>CURLPROTO_TFTP</code></b>, <b><code>CURLPROTO_MQTT</code></b>, <b><code>CURLPROTO_ALL</code></b></p>   Added in cURL 7.19.4.    <b><code>CURLOPT_PROXYAUTH</code></b>  The HTTP authentication method(s) to use for the proxy connection. Use the same bitmasks as described in <b><code>CURLOPT_HTTPAUTH</code></b>. For proxy authentication, only <b><code>CURLAUTH_BASIC</code></b> and <b><code>CURLAUTH_NTLM</code></b> are currently supported.   Added in cURL 7.10.7.    <b><code>CURLOPT_PROXYPORT</code></b>  The port number of the proxy to connect to. This port number can also be set in <b><code>CURLOPT_PROXY</code></b>.      <b><code>CURLOPT_PROXYTYPE</code></b>  Either <b><code>CURLPROXY_HTTP</code></b> (default), <b><code>CURLPROXY_SOCKS4</code></b>, <b><code>CURLPROXY_SOCKS5</code></b>, <b><code>CURLPROXY_SOCKS4A</code></b> or <b><code>CURLPROXY_SOCKS5_HOSTNAME</code></b>.   Added in cURL 7.10.    <b><code>CURLOPT_REDIR_PROTOCOLS</code></b>  Bitmask of <b><code>CURLPROTO_&#42;</code></b> values. If used, this bitmask limits what protocols libcurl may use in a transfer that it follows to in a redirect when <b><code>CURLOPT_FOLLOWLOCATION</code></b> is enabled. This allows you to limit specific transfers to only be allowed to use a subset of protocols in redirections. By default libcurl will allow all protocols except for FILE and SCP. This is a difference compared to pre-7.19.4 versions which unconditionally would follow to all protocols supported. See also <b><code>CURLOPT_PROTOCOLS</code></b> for protocol constant values.   Added in cURL 7.19.4.    <b><code>CURLOPT_RESUME_FROM</code></b>  The offset, in bytes, to resume a transfer from.      <b><code>CURLOPT_SOCKS5_AUTH</code></b>  <p>The SOCKS5 authentication method(s) to use. The options are: <b><code>CURLAUTH_BASIC</code></b>, <b><code>CURLAUTH_GSSAPI</code></b>, <b><code>CURLAUTH_NONE</code></b>.</p> <p>The bitwise <code>|</code> (or) operator can be used to combine more than one method. If this is done, cURL will poll the server to see what methods it supports and pick the best one.</p> <p><b><code>CURLAUTH_BASIC</code></b> allows username/password authentication.</p> <p><b><code>CURLAUTH_GSSAPI</code></b> allows GSS-API authentication.</p> <p><b><code>CURLAUTH_NONE</code></b> allows no authentication.</p> <p>Defaults to <code>CURLAUTH_BASIC|CURLAUTH_GSSAPI</code>. Set the actual username and password with the <b><code>CURLOPT_PROXYUSERPWD</code></b> option.</p>   Available as of 7.3.0 and curl &gt;= 7.55.0.    <b><code>CURLOPT_SSL_OPTIONS</code></b>  Set SSL behavior options, which is a bitmask of any of the following constants:  <b><code>CURLSSLOPT_ALLOW_BEAST</code></b>: do not attempt to use any workarounds for a security flaw in the SSL3 and TLS1.0 protocols.   <b><code>CURLSSLOPT_NO_REVOKE</code></b>: disable certificate revocation checks for those SSL backends where such behavior is present.   <b><code>CURLSSLOPT_AUTO_CLIENT_CERT</code></b>: automatically locate and use a client certificate for authentication, when requested by the server. This option is only supported for Schannel (the native Windows SSL library).   <b><code>CURLSSLOPT_NATIVE_CA</code></b>: use the operating system's native CA store for certificate verification. Works only on Windows when built to use OpenSSL. This option is experimental and behavior is subject to change.   <b><code>CURLSSLOPT_NO_PARTIALCHAIN</code></b>: do not accept "partial" certificate chains, which cURL otherwise does by default. This option is only supported for OpenSSL and will fail the certificate verification if the chain ends with an intermediate certificate and not with a root certificate.   <b><code>CURLSSLOPT_REVOKE_BEST_EFFORT</code></b>: ignore certificate revocation checks in case of missing or offline distribution points for those SSL backends where such behavior is present. This option is only supported for Schannel (the native Windows SSL library). If combined with <b><code>CURLSSLOPT_NO_REVOKE</code></b>, the latter takes precedence.    Added in cURL 7.25.0. Available since PHP 7.0.7.    <b><code>CURLOPT_SSL_VERIFYHOST</code></b>  <code>2</code> to verify that a Common Name field or a Subject Alternate Name field in the SSL peer certificate matches the provided hostname. <code>0</code> to not check the names. <code>1</code> should not be used. In production environments the value of this option should be kept at <code>2</code> (default value).   Support for value <code>1</code> removed in cURL 7.28.1.    <b><code>CURLOPT_SSLVERSION</code></b>  One of <b><code>CURL_SSLVERSION_DEFAULT</code></b> (0), <b><code>CURL_SSLVERSION_TLSv1</code></b> (1), <b><code>CURL_SSLVERSION_SSLv2</code></b> (2), <b><code>CURL_SSLVERSION_SSLv3</code></b> (3), <b><code>CURL_SSLVERSION_TLSv1_0</code></b> (4), <b><code>CURL_SSLVERSION_TLSv1_1</code></b> (5), <b><code>CURL_SSLVERSION_TLSv1_2</code></b> (6) or <b><code>CURL_SSLVERSION_TLSv1_3</code></b> (7). The maximum TLS version can be set by using one of the <b><code>CURL_SSLVERSION_MAX_&#42;</code></b> constants. It is also possible to OR one of the <b><code>CURL_SSLVERSION_&#42;</code></b> constants with one of the <b><code>CURL_SSLVERSION_MAX_&#42;</code></b> constants. <b><code>CURL_SSLVERSION_MAX_DEFAULT</code></b> (the maximum version supported by the library), <b><code>CURL_SSLVERSION_MAX_TLSv1_0</code></b>, <b><code>CURL_SSLVERSION_MAX_TLSv1_1</code></b>, <b><code>CURL_SSLVERSION_MAX_TLSv1_2</code></b>, or <b><code>CURL_SSLVERSION_MAX_TLSv1_3</code></b>. <p><b>Note</b>:</p><p>Your best bet is to not set this and let it use the default. Setting it to 2 or 3 is very dangerous given the known vulnerabilities in SSLv2 and SSLv3.</p>       <b><code>CURLOPT_PROXY_SSL_OPTIONS</code></b>  Set proxy SSL behavior options, which is a bitmask of any of the following constants:  <b><code>CURLSSLOPT_ALLOW_BEAST</code></b>: do not attempt to use any workarounds for a security flaw in the SSL3 and TLS1.0 protocols.   <b><code>CURLSSLOPT_NO_REVOKE</code></b>: disable certificate revocation checks for those SSL backends where such behavior is present. (curl &gt;= 7.44.0)   <b><code>CURLSSLOPT_NO_PARTIALCHAIN</code></b>: do not accept "partial" certificate chains, which it otherwise does by default. (curl &gt;= 7.68.0)    Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_SSL_VERIFYHOST</code></b>  Set to <code>2</code> to verify in the HTTPS proxy's certificate name fields against the proxy name. When set to <code>0</code> the connection succeeds regardless of the names used in the certificate. Use that ability with caution! <code>1</code> treated as a debug option in curl 7.28.0 and earlier. From curl 7.28.1 to 7.65.3 <b><code>CURLE_BAD_FUNCTION_ARGUMENT</code></b> is returned. From curl 7.66.0 onwards <code>1</code> and <code>2</code> is treated as the same value. In production environments the value of this option should be kept at <code>2</code> (default value).   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_SSLVERSION</code></b>  One of <b><code>CURL_SSLVERSION_DEFAULT</code></b>, <b><code>CURL_SSLVERSION_TLSv1</code></b>, <b><code>CURL_SSLVERSION_TLSv1_0</code></b>, <b><code>CURL_SSLVERSION_TLSv1_1</code></b>, <b><code>CURL_SSLVERSION_TLSv1_2</code></b>, <b><code>CURL_SSLVERSION_TLSv1_3</code></b>, <b><code>CURL_SSLVERSION_MAX_DEFAULT</code></b>, <b><code>CURL_SSLVERSION_MAX_TLSv1_0</code></b>, <b><code>CURL_SSLVERSION_MAX_TLSv1_1</code></b>, <b><code>CURL_SSLVERSION_MAX_TLSv1_2</code></b>, <b><code>CURL_SSLVERSION_MAX_TLSv1_3</code></b> or <b><code>CURL_SSLVERSION_SSLv3</code></b>. <p><b>Note</b>:</p><p>Your best bet is to not set this and let it use the default <b><code>CURL_SSLVERSION_DEFAULT</code></b> which will attempt to figure out the remote SSL protocol version.</p>    Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_STREAM_WEIGHT</code></b>  Set the numerical stream weight (a number between 1 and 256).   Added in cURL 7.46.0. Available since PHP 7.0.7.    <b><code>CURLOPT_TCP_KEEPALIVE</code></b>  If set to <code>1</code>, TCP keepalive probes will be sent. The delay and frequency of these probes can be controlled by the <b><code>CURLOPT_TCP_KEEPIDLE</code></b> and <b><code>CURLOPT_TCP_KEEPINTVL</code></b> options, provided the operating system supports them. If set to <code>0</code> (default) keepalive probes are disabled.   Added in cURL 7.25.0.    <b><code>CURLOPT_TCP_KEEPIDLE</code></b>  Sets the delay, in seconds, that the operating system will wait while the connection is idle before sending keepalive probes, if <b><code>CURLOPT_TCP_KEEPALIVE</code></b> is enabled. Not all operating systems support this option. The default is <code>60</code>.   Added in cURL 7.25.0.    <b><code>CURLOPT_TCP_KEEPINTVL</code></b>  Sets the interval, in seconds, that the operating system will wait between sending keepalive probes, if <b><code>CURLOPT_TCP_KEEPALIVE</code></b> is enabled. Not all operating systems support this option. The default is <code>60</code>.   Added in cURL 7.25.0.    <b><code>CURLOPT_TIMECONDITION</code></b>  How <b><code>CURLOPT_TIMEVALUE</code></b> is treated. Use <b><code>CURL_TIMECOND_IFMODSINCE</code></b> to return the page only if it has been modified since the time specified in <b><code>CURLOPT_TIMEVALUE</code></b>. If it hasn't been modified, a <code>"304 Not Modified"</code> header will be returned assuming <b><code>CURLOPT_HEADER</code></b> is <b><code>true</code></b>. Use <b><code>CURL_TIMECOND_IFUNMODSINCE</code></b> for the reverse effect. Use <b><code>CURL_TIMECOND_NONE</code></b> to ignore <b><code>CURLOPT_TIMEVALUE</code></b> and always return the page. <b><code>CURL_TIMECOND_NONE</code></b> is the default.   Before cURL 7.46.0 the default was <b><code>CURL_TIMECOND_IFMODSINCE</code></b>.    <b><code>CURLOPT_TIMEOUT</code></b>  The maximum number of seconds to allow cURL functions to execute.      <b><code>CURLOPT_TIMEOUT_MS</code></b>  The maximum number of milliseconds to allow cURL functions to execute. If libcurl is built to use the standard system name resolver, that portion of the connect will still use full-second resolution for timeouts with a minimum timeout allowed of one second.   Added in cURL 7.16.2.    <b><code>CURLOPT_TIMEVALUE</code></b>  The time in seconds since January 1st, 1970. The time will be used by <b><code>CURLOPT_TIMECONDITION</code></b>.      <b><code>CURLOPT_TIMEVALUE_LARGE</code></b>  The time in seconds since January 1st, 1970. The time will be used by <b><code>CURLOPT_TIMECONDITION</code></b>. Defaults to zero. The difference between this option and <b><code>CURLOPT_TIMEVALUE</code></b> is the type of the argument. On systems where 'long' is only 32 bit wide, this option has to be used to set dates beyond the year 2038.   Added in cURL 7.59.0. Available since PHP 7.3.0.    <b><code>CURLOPT_UPKEEP_INTERVAL_MS</code></b>  Some protocols have "connection upkeep" mechanisms. These mechanisms usually send some traffic on existing connections in order to keep them alive. This option defines the connection upkeep interval. Currently, the only protocol with a connection upkeep mechanism is HTTP/2. When the connection upkeep interval is exceeded, an HTTP/2 PING frame is sent on the connection. Default is <code>60</code> seconds.      <b><code>CURLOPT_UPLOAD_BUFFERSIZE</code></b>  Preferred buffer size in bytes for the cURL upload buffer. The upload buffer size by default is 64 kilobytes. The maximum buffer size allowed to be set is 2 megabytes. The minimum buffer size allowed to be set is 16 kilobytes.      <b><code>CURLOPT_MAX_RECV_SPEED_LARGE</code></b>  If a download exceeds this speed (counted in bytes per second) on cumulative average during the transfer, the transfer will pause to keep the average rate less than or equal to the parameter value. Defaults to unlimited speed.   Added in cURL 7.15.5.    <b><code>CURLOPT_MAX_SEND_SPEED_LARGE</code></b>  If an upload exceeds this speed (counted in bytes per second) on cumulative average during the transfer, the transfer will pause to keep the average rate less than or equal to the parameter value. Defaults to unlimited speed.   Added in cURL 7.15.5.    <b><code>CURLOPT_SSH_AUTH_TYPES</code></b>  A bitmask consisting of one or more of <b><code>CURLSSH_AUTH_PUBLICKEY</code></b>, <b><code>CURLSSH_AUTH_PASSWORD</code></b>, <b><code>CURLSSH_AUTH_HOST</code></b>, <b><code>CURLSSH_AUTH_KEYBOARD</code></b>. Set to <b><code>CURLSSH_AUTH_ANY</code></b> to let libcurl pick one.   Added in cURL 7.16.1.    <b><code>CURLOPT_IPRESOLVE</code></b>  Allows an application to select what kind of IP addresses to use when resolving host names. This is only interesting when using host names that resolve addresses using more than one version of IP, possible values are <b><code>CURL_IPRESOLVE_WHATEVER</code></b>, <b><code>CURL_IPRESOLVE_V4</code></b>, <b><code>CURL_IPRESOLVE_V6</code></b>, by default <b><code>CURL_IPRESOLVE_WHATEVER</code></b>.   Added in cURL 7.10.8.    <b><code>CURLOPT_FTP_FILEMETHOD</code></b>  Tell curl which method to use to reach a file on a FTP(S) server. Possible values are <b><code>CURLFTPMETHOD_DEFAULT</code></b>, <b><code>CURLFTPMETHOD_MULTICWD</code></b>, <b><code>CURLFTPMETHOD_NOCWD</code></b>, and <b><code>CURLFTPMETHOD_SINGLECWD</code></b>.   Added in cURL 7.15.1.     <p><code>value</code> should be a <code>string</code> for the following values of the <code>option</code> parameter:</p>   Option Set <code>value</code> to Notes     <b><code>CURLOPT_ABSTRACT_UNIX_SOCKET</code></b>  Enables the use of an abstract Unix domain socket instead of establishing a TCP connection to a host and sets the path to the given <code>string</code>. This option shares the same semantics as <b><code>CURLOPT_UNIX_SOCKET_PATH</code></b>. These two options share the same storage and therefore only one of them can be set per handle.   Available since PHP 7.3.0 and cURL 7.53.0    <b><code>CURLOPT_ALTSVC</code></b>  Pass the filename for cURL to use as the Alt-Svc cache file to read existing cache contents from and possibly also write it back to a after a transfer, unless <b><code>CURLALTSVC_READONLYFILE</code></b> is set via <b><code>CURLOPT_ALTSVC_CTRL</code></b>.      <b><code>CURLOPT_ALTSVC_CTRL</code></b>  Populate the bitmask with the correct set of features to instruct cURL how to handle Alt-Svc for the transfers using this handle. cURL only accepts Alt-Svc headers over HTTPS. It will also only complete a request to an alternative origin if that origin is properly hosted over HTTPS. Setting any bit will enable the alt-svc engine. The options are: <b><code>CURLALTSVC_H1</code></b>, <b><code>CURLALTSVC_H2</code></b>, <b><code>CURLALTSVC_H3</code></b>, and <b><code>CURLALTSVC_READONLYFILE</code></b>.      <b><code>CURLOPT_AWS_SIGV4</code></b>  <p>Provides AWS V4 signature authentication on HTTP(S) header.</p> <p>This option overrides any other authentication types that have been set in <b><code>CURLOPT_HTTPAUTH</code></b>. This method cannot be combined with other authentication types.</p>      <b><code>CURLOPT_CAINFO</code></b>  The name of a file holding one or more certificates to verify the peer with. This only makes sense when used in combination with <b><code>CURLOPT_SSL_VERIFYPEER</code></b>.   Might require an absolute path.    <b><code>CURLOPT_CAINFO_BLOB</code></b>  The name of a PEM file holding one or more certificates to verify the peer with. This option overrides <b><code>CURLOPT_CAINFO</code></b>.   Available as of PHP 8.2.0 and cURL 7.77.0    <b><code>CURLOPT_CAPATH</code></b>  A directory that holds multiple CA certificates. Use this option alongside <b><code>CURLOPT_SSL_VERIFYPEER</code></b>.      <b><code>CURLOPT_COOKIE</code></b>  The contents of the <code>"Cookie: "</code> header to be used in the HTTP request. Note that multiple cookies are separated with a semicolon followed by a space (e.g., "<code>fruit=apple; colour=red</code>")      <b><code>CURLOPT_COOKIEFILE</code></b>  The name of the file containing the cookie data. The cookie file can be in Netscape format, or just plain HTTP-style headers dumped into a file. If the name is an empty string, no cookies are loaded, but cookie handling is still enabled.      <b><code>CURLOPT_COOKIEJAR</code></b>  The name of a file to save all internal cookies to when the handle is closed, e.g. after a call to curl_close.      <b><code>CURLOPT_COOKIELIST</code></b>  A cookie string (i.e. a single line in Netscape/Mozilla format, or a regular HTTP-style Set-Cookie header) adds that single cookie to the internal cookie store. <code>"ALL"</code> erases all cookies held in memory. <code>"SESS"</code> erases all session cookies held in memory. <code>"FLUSH"</code> writes all known cookies to the file specified by <b><code>CURLOPT_COOKIEJAR</code></b>. <code>"RELOAD"</code> loads all cookies from the files specified by <b><code>CURLOPT_COOKIEFILE</code></b>.   Available since cURL 7.14.1.    <b><code>CURLOPT_CUSTOMREQUEST</code></b> <p>A custom request method to use instead of <code>"GET"</code> or <code>"HEAD"</code> when doing a HTTP request. This is useful for doing <code>"DELETE"</code> or other, more obscure HTTP requests. Valid values are things like <code>"GET"</code>, <code>"POST"</code>, <code>"CONNECT"</code> and so on; i.e. Do not enter a whole HTTP request line here. For instance, entering <code>"GET /index.html HTTP/1.0\r\n\r\n"</code> would be incorrect.</p><p><b>Note</b>:</p><p>Don't do this without making sure the server supports the custom request method first.</p>       <b><code>CURLOPT_DEFAULT_PROTOCOL</code></b> <p>The default protocol to use if the URL is missing a scheme name.</p>  Added in cURL 7.45.0. Available since PHP 7.0.7.    <b><code>CURLOPT_DNS_INTERFACE</code></b> <p>Set the name of the network interface that the DNS resolver should bind to. This must be an interface name (not an address).</p>  Added in cURL 7.33.0. Available since PHP 7.0.7.    <b><code>CURLOPT_DNS_LOCAL_IP4</code></b> <p>Set the local IPv4 address that the resolver should bind to. The argument should contain a single numerical IPv4 address as a string.</p>  Added in cURL 7.33.0. Available since PHP 7.0.7.    <b><code>CURLOPT_DNS_LOCAL_IP6</code></b> <p>Set the local IPv6 address that the resolver should bind to. The argument should contain a single numerical IPv6 address as a string.</p>  Added in cURL 7.33.0. Available since PHP 7.0.7.    <b><code>CURLOPT_EGDSOCKET</code></b>  Like <b><code>CURLOPT_RANDOM_FILE</code></b>, except a filename to an Entropy Gathering Daemon socket.      <b><code>CURLOPT_ENCODING</code></b>  The contents of the <code>"Accept-Encoding: "</code> header. This enables decoding of the response. Supported encodings are <code>"identity"</code>, <code>"deflate"</code>, and <code>"gzip"</code>. If an empty string, <code>""</code>, is set, a header containing all supported encoding types is sent.   Added in cURL 7.10.    <b><code>CURLOPT_FTPPORT</code></b>  The value which will be used to get the IP address to use for the FTP "PORT" instruction. The "PORT" instruction tells the remote server to connect to our specified IP address. The string may be a plain IP address, a hostname, a network interface name (under Unix), or just a plain '-' to use the systems default IP address.      <b><code>CURLOPT_HSTS</code></b>  <p>HSTS (HTTP Strict Transport Security) cache file name.</p>      <b><code>CURLOPT_HSTS_CTRL</code></b>  <p>Controls HSTS (HTTP Strict Transport Security) behavior. Populate the bitmask with the correct set of features to instruct cURL how to handle HSTS for the transfers using this handle. <b><code>CURLHSTS_ENABLE</code></b> enables the in-memory HSTS cache. If the HSTS cache file is defined, set <b><code>CURLHSTS_READONLYFILE</code></b> to make the file read-only.</p>      <b><code>CURLOPT_INTERFACE</code></b>  The name of the outgoing network interface to use. This can be an interface name, an IP address or a host name.      <b><code>CURLOPT_KEYPASSWD</code></b>  The password required to use the <b><code>CURLOPT_SSLKEY</code></b> or <b><code>CURLOPT_SSH_PRIVATE_KEYFILE</code></b> private key.   Added in cURL 7.16.1.    <b><code>CURLOPT_KRB4LEVEL</code></b>  The KRB4 (Kerberos 4) security level. Any of the following values (in order from least to most powerful) are valid: <code>"clear"</code>, <code>"safe"</code>, <code>"confidential"</code>, <code>"private".</code>. If the string does not match one of these, <code>"private"</code> is used. Setting this option to <b><code>null</code></b> will disable KRB4 security. Currently KRB4 security only works with FTP transactions.      <b><code>CURLOPT_LOGIN_OPTIONS</code></b>  Can be used to set protocol specific login options, such as the preferred authentication mechanism via "AUTH=NTLM" or "AUTH=&#42;", and should be used in conjunction with the <b><code>CURLOPT_USERNAME</code></b> option.   Added in cURL 7.34.0. Available since PHP 7.0.7.    <b><code>CURLOPT_PINNEDPUBLICKEY</code></b>  Set the pinned public key. The string can be the file name of your pinned public key. The file format expected is "PEM" or "DER". The string can also be any number of base64 encoded sha256 hashes preceded by "sha256//" and separated by ";".   Added in cURL 7.39.0. Available since PHP 7.0.7.    <b><code>CURLOPT_POSTFIELDS</code></b>   The full data to post in a HTTP "POST" operation. This parameter can either be passed as a urlencoded string like '<code>para1=val1&amp;para2=val2&amp;...</code>' or as an array with the field name as key and field data as value. If <code>value</code> is an array, the <code>Content-Type</code> header will be set to <code>multipart/form-data</code>.   Files can be sent using <code>CURLFile</code> or <code>CURLStringFile</code>, in which case <code>value</code> must be an array.       <b><code>CURLOPT_PRIVATE</code></b>  Any data that should be associated with this cURL handle. This data can subsequently be retrieved with the <b><code>CURLINFO_PRIVATE</code></b> option of <code>curl_getinfo()</code>. cURL does nothing with this data. When using a cURL multi handle, this private data is typically a unique key to identify a standard cURL handle.   Added in cURL 7.10.3.    <b><code>CURLOPT_PRE_PROXY</code></b>  Set a <code>string</code> holding the host name or dotted numerical IP address to be used as the preproxy that curl connects to before it connects to the HTTP(S) proxy specified in the <b><code>CURLOPT_PROXY</code></b> option for the upcoming request. The preproxy can only be a SOCKS proxy and it should be prefixed with <code>[scheme]://</code> to specify which kind of socks is used. A numerical IPv6 address must be written within [brackets]. Setting the preproxy to an empty string explicitly disables the use of a preproxy. To specify port number in this string, append <code>:[port]</code> to the end of the host name. The proxy's port number may optionally be specified with the separate option <b><code>CURLOPT_PROXYPORT</code></b>. Defaults to using port 1080 for proxies if a port is not specified.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY</code></b>  The HTTP proxy to tunnel requests through.      <b><code>CURLOPT_PROXY_SERVICE_NAME</code></b>  The proxy authentication service name.   Added in cURL 7.43.0 for HTTP proxies, and in cURL 7.49.0 for SOCKS5 proxies. Available since PHP 7.0.7.    <b><code>CURLOPT_PROXY_CAINFO</code></b>  The path to proxy Certificate Authority (CA) bundle. Set the path as a <code>string</code> naming a file holding one or more certificates to verify the HTTPS proxy with. This option is for connecting to an HTTPS proxy, not an HTTPS server. Defaults set to the system path where libcurl's cacert bundle is assumed to be stored.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_CAINFO_BLOB</code></b>  The name of a PEM file holding one or more certificates to verify the HTTPS proxy with. This option is for connecting to an HTTPS proxy, not an HTTPS server. Defaults set to the system path where libcurl's cacert bundle is assumed to be stored.   Available as of PHP 8.2.0 and libcurl &gt;= cURL 7.77.0.    <b><code>CURLOPT_PROXY_CAPATH</code></b>  The directory holding multiple CA certificates to verify the HTTPS proxy with.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_CRLFILE</code></b>  Set the file name with the concatenation of CRL (Certificate Revocation List) in PEM format to use in the certificate validation that occurs during the SSL exchange.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_KEYPASSWD</code></b>  Set the string be used as the password required to use the <b><code>CURLOPT_PROXY_SSLKEY</code></b> private key. You never needed a passphrase to load a certificate but you need one to load your private key. This option is for connecting to an HTTPS proxy, not an HTTPS server.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_PINNEDPUBLICKEY</code></b>  Set the pinned public key for HTTPS proxy. The string can be the file name of your pinned public key. The file format expected is "PEM" or "DER". The string can also be any number of base64 encoded sha256 hashes preceded by "sha256//" and separated by ";"   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_SSLCERT</code></b>  The file name of your client certificate used to connect to the HTTPS proxy. The default format is "P12" on Secure Transport and "PEM" on other engines, and can be changed with <b><code>CURLOPT_PROXY_SSLCERTTYPE</code></b>. With NSS or Secure Transport, this can also be the nickname of the certificate you wish to authenticate with as it is named in the security database. If you want to use a file from the current directory, please precede it with "./" prefix, in order to avoid confusion with a nickname.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_SSLCERTTYPE</code></b>  The format of your client certificate used when connecting to an HTTPS proxy. Supported formats are "PEM" and "DER", except with Secure Transport. OpenSSL (versions 0.9.3 and later) and Secure Transport (on iOS 5 or later, or OS X 10.7 or later) also support "P12" for PKCS#12-encoded files. Defaults to "PEM".   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_SSL_CIPHER_LIST</code></b>  The list of ciphers to use for the connection to the HTTPS proxy. The list must be syntactically correct, it consists of one or more cipher strings separated by colons. Commas or spaces are also acceptable separators but colons are normally used, !, - and + can be used as operators.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_TLS13_CIPHERS</code></b>  The list of cipher suites to use for the TLS 1.3 connection to a proxy. The list must be syntactically correct, it consists of one or more cipher suite strings separated by colons. This option is currently used only when curl is built to use OpenSSL 1.1.1 or later. If you are using a different SSL backend you can try setting TLS 1.3 cipher suites by using the <b><code>CURLOPT_PROXY_SSL_CIPHER_LIST</code></b> option.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.61.0. Available when built with OpenSSL &gt;= 1.1.1.    <b><code>CURLOPT_PROXY_SSLKEY</code></b>  The file name of your private key used for connecting to the HTTPS proxy. The default format is "PEM" and can be changed with <b><code>CURLOPT_PROXY_SSLKEYTYPE</code></b>. (iOS and Mac OS X only) This option is ignored if curl was built against Secure Transport.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0. Available if built TLS enabled.    <b><code>CURLOPT_PROXY_SSLKEYTYPE</code></b>  The format of your private key. Supported formats are "PEM", "DER" and "ENG".   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_TLSAUTH_PASSWORD</code></b>  The password to use for the TLS authentication method specified with the <b><code>CURLOPT_PROXY_TLSAUTH_TYPE</code></b> option. Requires that the <b><code>CURLOPT_PROXY_TLSAUTH_USERNAME</code></b> option to also be set.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_TLSAUTH_TYPE</code></b>  The method of the TLS authentication used for the HTTPS connection. Supported method is "SRP". <p><b>Note</b>:</p><p>Secure Remote Password (SRP) authentication for TLS provides mutual authentication if both sides have a shared secret. To use TLS-SRP, you must also set the <b><code>CURLOPT_PROXY_TLSAUTH_USERNAME</code></b> and <b><code>CURLOPT_PROXY_TLSAUTH_PASSWORD</code></b> options.</p>    Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXY_TLSAUTH_USERNAME</code></b>  The username to use for the HTTPS proxy TLS authentication method specified with the <b><code>CURLOPT_PROXY_TLSAUTH_TYPE</code></b> option. Requires that the <b><code>CURLOPT_PROXY_TLSAUTH_PASSWORD</code></b> option to also be set.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.52.0.    <b><code>CURLOPT_PROXYUSERPWD</code></b>  A username and password formatted as <code>"[username]:[password]"</code> to use for the connection to the proxy.      <b><code>CURLOPT_RANDOM_FILE</code></b>  A filename to be used to seed the random number generator for SSL.      <b><code>CURLOPT_RANGE</code></b>  Range(s) of data to retrieve in the format <code>"X-Y"</code> where X or Y are optional. HTTP transfers also support several intervals, separated with commas in the format <code>"X-Y,N-M"</code>.      <b><code>CURLOPT_REFERER</code></b>  The contents of the <code>"Referer: "</code> header to be used in a HTTP request.      <b><code>CURLOPT_SASL_AUTHZID</code></b>  The authorization identity (authzid) for the transfer. Only applicable to the PLAIN SASL authentication mechanism where it is optional. When not specified, only the authentication identity (authcid) as specified by the username will be sent to the server, along with the password. The server will derive the authzid from the authcid when not provided, which it will then use internally.      <b><code>CURLOPT_SERVICE_NAME</code></b>  The authentication service name.   Added in cURL 7.43.0. Available since PHP 7.0.7.    <b><code>CURLOPT_SSH_HOST_PUBLIC_KEY_MD5</code></b>  A string containing 32 hexadecimal digits. The string should be the MD5 checksum of the remote host's public key, and libcurl will reject the connection to the host unless the md5sums match. This option is only for SCP and SFTP transfers.   Added in cURL 7.17.1.    <b><code>CURLOPT_SSH_HOST_PUBLIC_KEY_SHA256</code></b>  Base64-encoded SHA256 hash of the remote host's public key. The transfer will fail if the given hash does not match the hash the remote host provides.      <b><code>CURLOPT_SSH_PUBLIC_KEYFILE</code></b>  The file name for your public key. If not used, libcurl defaults to $HOME/.ssh/id_dsa.pub if the HOME environment variable is set, and just "id_dsa.pub" in the current directory if HOME is not set.   Added in cURL 7.16.1.    <b><code>CURLOPT_SSH_PRIVATE_KEYFILE</code></b>  The file name for your private key. If not used, libcurl defaults to $HOME/.ssh/id_dsa if the HOME environment variable is set, and just "id_dsa" in the current directory if HOME is not set. If the file is password-protected, set the password with <b><code>CURLOPT_KEYPASSWD</code></b>.   Added in cURL 7.16.1.    <b><code>CURLOPT_SSL_CIPHER_LIST</code></b>  A list of ciphers to use for SSL. For example, <code>RC4-SHA</code> and <code>TLSv1</code> are valid cipher lists.      <b><code>CURLOPT_SSL_EC_CURVES</code></b>  A colon delimited list of elliptic curve algorithms. For example, <code>X25519:P-521</code> is a valid list of two elliptic curves. This option defines the client's key exchange algorithms in the SSL handshake, if the SSL backend cURL is built to use supports it.      <b><code>CURLOPT_SSLCERT</code></b>  The name of a file containing a PEM formatted certificate.      <b><code>CURLOPT_SSLCERTPASSWD</code></b>  The password required to use the <b><code>CURLOPT_SSLCERT</code></b> certificate.      <b><code>CURLOPT_SSLCERTTYPE</code></b>  The format of the certificate. Supported formats are <code>"PEM"</code> (default), <code>"DER"</code>, and <code>"ENG"</code>. As of OpenSSL 0.9.3, <code>"P12"</code> (for PKCS#12-encoded files) is also supported.   Added in cURL 7.9.3.    <b><code>CURLOPT_SSLENGINE</code></b>  The identifier for the crypto engine of the private SSL key specified in <b><code>CURLOPT_SSLKEY</code></b>.      <b><code>CURLOPT_SSLENGINE_DEFAULT</code></b>  The identifier for the crypto engine used for asymmetric crypto operations.      <b><code>CURLOPT_SSLKEY</code></b>  The name of a file containing a private SSL key.      <b><code>CURLOPT_SSLKEYPASSWD</code></b> <p>The secret password needed to use the private SSL key specified in <b><code>CURLOPT_SSLKEY</code></b>.</p><p><b>Note</b>:</p><p>Since this option contains a sensitive password, remember to keep the PHP script it is contained within safe.</p>       <b><code>CURLOPT_SSLKEYTYPE</code></b>  The key type of the private SSL key specified in <b><code>CURLOPT_SSLKEY</code></b>. Supported key types are <code>"PEM"</code> (default), <code>"DER"</code>, and <code>"ENG"</code>.      <b><code>CURLOPT_TLS13_CIPHERS</code></b>  The list of cipher suites to use for the TLS 1.3 connection. The list must be syntactically correct, it consists of one or more cipher suite strings separated by colons. This option is currently used only when curl is built to use OpenSSL 1.1.1 or later. If you are using a different SSL backend you can try setting TLS 1.3 cipher suites by using the <b><code>CURLOPT_SSL_CIPHER_LIST</code></b> option.   Available since PHP 7.3.0 and libcurl &gt;= cURL 7.61.0. Available when built with OpenSSL &gt;= 1.1.1.    <b><code>CURLOPT_UNIX_SOCKET_PATH</code></b>  Enables the use of Unix domain sockets as connection endpoint and sets the path to the given <code>string</code>.   Added in cURL 7.40.0. Available since PHP 7.0.7.    <b><code>CURLOPT_URL</code></b>  The URL to fetch. This can also be set when initializing a session with <code>curl_init()</code>.      <b><code>CURLOPT_USERAGENT</code></b>  The contents of the <code>"User-Agent: "</code> header to be used in a HTTP request.      <b><code>CURLOPT_USERNAME</code></b>  The user name to use in authentication.   Added in cURL 7.19.1.    <b><code>CURLOPT_PASSWORD</code></b>  The password to use in authentication.   Added in cURL 7.19.1.    <b><code>CURLOPT_USERPWD</code></b>  A username and password formatted as <code>"[username]:[password]"</code> to use for the connection.      <b><code>CURLOPT_XOAUTH2_BEARER</code></b>  Specifies the OAuth 2.0 access token.   Added in cURL 7.33.0. Available since PHP 7.0.7.     <p><code>value</code> should be an array for the following values of the <code>option</code> parameter:</p>   Option Set <code>value</code> to Notes     <b><code>CURLOPT_CONNECT_TO</code></b>  Connect to a specific host and port instead of the URL's host and port. Accepts an array of strings with the format <code>HOST:PORT:CONNECT-TO-HOST:CONNECT-TO-PORT</code>.   Added in cURL 7.49.0. Available since PHP 7.0.7.    <b><code>CURLOPT_HTTP200ALIASES</code></b>  An array of HTTP 200 responses that will be treated as valid responses and not as errors.   Added in cURL 7.10.3.    <b><code>CURLOPT_HTTPHEADER</code></b>  An array of HTTP header fields to set, in the format <code> array('Content-type: text/plain', 'Content-length: 100') </code>      <b><code>CURLOPT_POSTQUOTE</code></b>  An array of FTP commands to execute on the server after the FTP request has been performed.      <b><code>CURLOPT_PROXYHEADER</code></b>  An array of custom HTTP headers to pass to proxies.   Added in cURL 7.37.0. Available since PHP 7.0.7.    <b><code>CURLOPT_QUOTE</code></b>  An array of FTP commands to execute on the server prior to the FTP request.      <b><code>CURLOPT_RESOLVE</code></b>  Provide a custom address for a specific host and port pair. An array of hostname, port, and IP address strings, each element separated by a colon. In the format: <code> array("example.com:80:127.0.0.1") </code>   Added in cURL 7.21.3.     <p><code>value</code> should be a stream resource (using <code>fopen()</code>, for example) for the following values of the <code>option</code> parameter:</p>   Option Set <code>value</code> to     <b><code>CURLOPT_FILE</code></b>  The file that the transfer should be written to. The default is <code>STDOUT</code> (the browser window).    <b><code>CURLOPT_INFILE</code></b>  The file that the transfer should be read from when uploading.    <b><code>CURLOPT_STDERR</code></b>  An alternative location to output errors to instead of <code>STDERR</code>.    <b><code>CURLOPT_WRITEHEADER</code></b>  The file that the header part of the transfer is written to.     <p><code>value</code> should be the name of a valid function or a Closure for the following values of the <code>option</code> parameter:</p>   Option Set <code>value</code> to Notes     <b><code>CURLOPT_HEADERFUNCTION</code></b>  A callback accepting two parameters. The first is the cURL resource, the second is a string with the header data to be written. The header data must be written by this callback. Return the number of bytes written.  &#xA0;   <b><code>CURLOPT_PASSWDFUNCTION</code></b>  A callback accepting three parameters. The first is the cURL resource, the second is a string containing a password prompt, and the third is the maximum password length. Return the string containing the password.  Removed as of PHP 7.3.0.   <b><code>CURLOPT_PROGRESSFUNCTION</code></b>  <p>A callback accepting five parameters. The first is the cURL resource, the second is the total number of bytes expected to be downloaded in this transfer, the third is the number of bytes downloaded so far, the fourth is the total number of bytes expected to be uploaded in this transfer, and the fifth is the number of bytes uploaded so far.</p> <p><b>Note</b>:</p><p>The callback is only called when the <b><code>CURLOPT_NOPROGRESS</code></b> option is set to <b><code>false</code></b>.</p>  <p>Return a non-zero value to abort the transfer. In which case, the transfer will set a <b><code>CURLE_ABORTED_BY_CALLBACK</code></b> error.</p>  &#xA0;   <b><code>CURLOPT_READFUNCTION</code></b>  A callback accepting three parameters. The first is the cURL resource, the second is a stream resource provided to cURL through the option <b><code>CURLOPT_INFILE</code></b>, and the third is the maximum amount of data to be read. The callback must return a string with a length equal or smaller than the amount of data requested, typically by reading it from the passed stream resource. It should return an empty string to signal <code>EOF</code>.  &#xA0;   <b><code>CURLOPT_WRITEFUNCTION</code></b>  A callback accepting two parameters. The first is the cURL resource, and the second is a string with the data to be written. The data must be saved by this callback. It must return the exact number of bytes written or the transfer will be aborted with an error.  &#xA0;   <b><code>CURLOPT_XFERINFOFUNCTION</code></b>  A callback accepting two parameters. Has a similar purpose as <b><code>CURLOPT_PROGRESSFUNCTION</code></b> but is more modern and the preferred option from cURL.   Added in 7.32.0. Available as of PHP 8.2.0.     <p>Other values:</p>   Option Set <code>value</code> to     <b><code>CURLOPT_SHARE</code></b>  A result of <code>curl_share_init()</code>. Makes the cURL handle to use the data from the shared handle.
	 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-setopt.php
	 * @see curl_setopt_array()
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_setopt(\CurlHandle $handle, int $option, mixed $value): bool {}

	/**
	 * Set multiple options for a cURL transfer
	 * <p>Sets multiple options for a cURL session. This function is useful for setting a large number of cURL options without repetitively calling <code>curl_setopt()</code>.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param array $options <p>An <code>array</code> specifying which options to set and their values. The keys should be valid <code>curl_setopt()</code> constants or their integer equivalents.</p>
	 * @return bool <p>Returns <b><code>true</code></b> if all options were successfully set. If an option could not be successfully set, <b><code>false</code></b> is immediately returned, ignoring any future options in the <code>options</code> array.</p>
	 * @link https://php.net/manual/en/function.curl-setopt-array.php
	 * @see curl_setopt()
	 * @since PHP 5 >= 5.1.3, PHP 7, PHP 8
	 */
	function curl_setopt_array(\CurlHandle $handle, array $options): bool {}

	/**
	 * Close a cURL share handle
	 * <p><b>Note</b>:</p><p>This function has no effect. Prior to PHP 8.0.0, this function was used to close the resource.</p><p>Closes a cURL share handle and frees all resources.</p>
	 * @param \CurlShareHandle $share_handle <p>A cURL share handle returned by <code>curl_share_init()</code>.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.curl-share-close.php
	 * @see curl_share_init()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_share_close(\CurlShareHandle $share_handle): void {}

	/**
	 * Return the last share curl error number
	 * <p>Return an integer containing the last share curl error number.</p>
	 * @param \CurlShareHandle $share_handle <p>A cURL share handle returned by <code>curl_share_init()</code>.</p>
	 * @return int <p>Returns an integer containing the last share curl error number.</p>
	 * @link https://php.net/manual/en/function.curl-share-errno.php
	 * @see curl_errno()
	 * @since PHP 7 >= 7.1.0, PHP 8
	 */
	function curl_share_errno(\CurlShareHandle $share_handle): int {}

	/**
	 * Initialize a cURL share handle
	 * <p>Allows to share data between cURL handles.</p>
	 * @return CurlShareHandle <p>Returns a cURL share handle.</p>
	 * @link https://php.net/manual/en/function.curl-share-init.php
	 * @see curl_share_setopt(), curl_share_close()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_share_init(): \CurlShareHandle {}

	/**
	 * Set an option for a cURL share handle
	 * <p>Sets an option on the given cURL share handle.</p>
	 * @param \CurlShareHandle $share_handle <p>A cURL share handle returned by <code>curl_share_init()</code>.</p>
	 * @param int $option <p></p>   Option Description     <b><code>CURLSHOPT_SHARE</code></b>  Specifies a type of data that should be shared.    <b><code>CURLSHOPT_UNSHARE</code></b>  Specifies a type of data that will be no longer shared.
	 * @param mixed $value <p></p>   Value Description     <b><code>CURL_LOCK_DATA_COOKIE</code></b>  Shares cookie data.    <b><code>CURL_LOCK_DATA_DNS</code></b>  Shares DNS cache. Note that when you use cURL multi handles, all handles added to the same multi handle will share DNS cache by default.    <b><code>CURL_LOCK_DATA_SSL_SESSION</code></b>  Shares SSL session IDs, reducing the time spent on the SSL handshake when reconnecting to the same server. Note that SSL session IDs are reused within the same handle by default.
	 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-share-setopt.php
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_share_setopt(\CurlShareHandle $share_handle, int $option, mixed $value): bool {}

	/**
	 * Return string describing the given error code
	 * <p>Returns a text error message describing the given error code.</p>
	 * @param int $error_code <p>One of the &#xBB;&#xA0;cURL error codes constants.</p>
	 * @return ?string <p>Returns error description or <b><code>null</code></b> for invalid error code.</p>
	 * @link https://php.net/manual/en/function.curl-share-strerror.php
	 * @see curl_share_errno(), curl_strerror()
	 * @since PHP 7 >= 7.1.0, PHP 8
	 */
	function curl_share_strerror(int $error_code): ?string {}

	/**
	 * Return string describing the given error code
	 * <p>Returns a text error message describing the given error code.</p>
	 * @param int $error_code <p>One of the &#xBB;&#xA0;cURL error codes constants.</p>
	 * @return ?string <p>Returns error description or <b><code>null</code></b> for invalid error code.</p>
	 * @link https://php.net/manual/en/function.curl-strerror.php
	 * @see curl_errno(), curl_error()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_strerror(int $error_code): ?string {}

	/**
	 * Decodes the given URL encoded string
	 * <p>This function decodes the given URL encoded string.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param string $string <p>The URL encoded string to be decoded.</p>
	 * @return string|false <p>Returns decoded string or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-unescape.php
	 * @see curl_escape(), urlencode(), urldecode(), rawurlencode(), rawurldecode()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_unescape(\CurlHandle $handle, string $string): string|false {}

	/**
	 * Performs any connection upkeep checks
	 * <p>Available if built against libcurl &gt;= 7.62.0.</p><p>Some protocols have "connection upkeep" mechanisms. These mechanisms usually send some traffic on existing connections in order to keep them alive; this can prevent connections from being closed due to overzealous firewalls, for example.</p><p>Connection upkeep is currently available only for HTTP/2 connections. A small amount of traffic is usually sent to keep a connection alive. HTTP/2 maintains its connection by sending a HTTP/2 PING frame.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl_upkeep.php
	 * @see curl_init()
	 * @since PHP 8 >= 8.2.0
	 */
	function curl_upkeep(\CurlHandle $handle): bool {}

	/**
	 * Gets cURL version information
	 * <p>Returns information about the cURL version.</p>
	 * @return array|false <p>Returns an associative array with the following elements:</p>   Key Value description     version_number cURL 24 bit version number   version cURL version number, as a string   ssl_version_number OpenSSL 24 bit version number   ssl_version OpenSSL version number, as a string   libz_version zlib version number, as a string   host Information about the host where cURL was built   age &#xA0;   features A bitmask of the <code>CURL_VERSION_XXX</code> constants   protocols An array of protocols names supported by cURL
	 * @link https://php.net/manual/en/function.curl-version.php
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_version(): array|false {}

	define('CURL_HTTP_VERSION_1_0', 1);

	define('CURL_HTTP_VERSION_1_1', 2);

	/**
	 * Available since PHP 7.0.7 and cURL 7.43.0
	 */
	define('CURL_HTTP_VERSION_2', 3);

	/**
	 * Available since cURL 7.33.0
	 */
	define('CURL_HTTP_VERSION_2_0', 3);

	/**
	 * Available since PHP 7.0.7 and cURL 7.49.0
	 */
	define('CURL_HTTP_VERSION_2_PRIOR_KNOWLEDGE', 5);

	/**
	 * Available since PHP 7.0.7 and cURL 7.47.0
	 */
	define('CURL_HTTP_VERSION_2TLS', 4);

	define('CURL_HTTP_VERSION_NONE', 0);

	/**
	 * Available since PHP 7.3.0 and cURL 7.10.0
	 */
	define('CURL_LOCK_DATA_CONNECT', 5);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURL_LOCK_DATA_PSL', 6);

	/**
	 * Available since PHP 7.3.0 and cURL 7.53.0
	 */
	define('CURL_MAX_READ_SIZE', 524288);

	define('CURL_NETRC_IGNORED', 0);

	define('CURL_NETRC_OPTIONAL', 1);

	define('CURL_NETRC_REQUIRED', 2);

	/**
	 * Available since PHP 7.1.0 and cURL 7.44.0
	 */
	define('CURL_PUSH_DENY', 1);

	/**
	 * Available since PHP 7.1.0 and cURL 7.44.0
	 */
	define('CURL_PUSH_OK', 0);

	/**
	 * Available since PHP 7.0.7 and cURL 7.18.2
	 */
	define('CURL_REDIR_POST_301', 1);

	/**
	 * Available since PHP 7.0.7 and cURL 7.18.2
	 */
	define('CURL_REDIR_POST_302', 2);

	/**
	 * Available since PHP 7.0.7 and cURL 7.25.1
	 */
	define('CURL_REDIR_POST_303', 4);

	/**
	 * Available since PHP 7.0.7 and cURL 7.18.2
	 */
	define('CURL_REDIR_POST_ALL', 7);

	define('CURL_SSLVERSION_DEFAULT', 0);

	/**
	 * Available since PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_DEFAULT', 65536);

	/**
	 * Available since PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_NONE', 0);

	/**
	 * Available since PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_TLSv1_0', 262144);

	/**
	 * Available since PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_TLSv1_1', 327680);

	/**
	 * Available since PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_TLSv1_2', 393216);

	/**
	 * Available since PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_TLSv1_3', 458752);

	define('CURL_SSLVERSION_SSLv2', 2);

	define('CURL_SSLVERSION_SSLv3', 3);

	define('CURL_SSLVERSION_TLSv1', 1);

	define('CURL_SSLVERSION_TLSv1_0', 4);

	define('CURL_SSLVERSION_TLSv1_1', 5);

	define('CURL_SSLVERSION_TLSv1_2', 6);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURL_SSLVERSION_TLSv1_3', 7);

	define('CURL_TIMECOND_IFMODSINCE', 1);

	define('CURL_TIMECOND_IFUNMODSINCE', 2);

	define('CURL_TIMECOND_LASTMOD', 3);

	define('CURL_TIMECOND_NONE', 0);

	/**
	 * Available since PHP 7.3.6 and cURL 7.64.1
	 */
	define('CURL_VERSION_ALTSVC', 16777216);

	/**
	 * Asynchronous DNS resolves. Available since PHP 7.3.0 and cURL 7.10.7
	 */
	define('CURL_VERSION_ASYNCHDNS', 128);

	/**
	 * Available since PHP 7.3.0 and cURL 7.57.0
	 */
	define('CURL_VERSION_BROTLI', 8388608);

	/**
	 * Character conversions supported. Available since PHP 7.3.0 and cURL 7.15.4
	 */
	define('CURL_VERSION_CONV', 4096);

	/**
	 * Debug memory tracking supported. Available since PHP 7.3.6 and cURL 7.19.6
	 */
	define('CURL_VERSION_CURLDEBUG', 8192);

	/**
	 * Built with debug capabilities. Available since PHP 7.3.0 and cURL 7.10.6
	 */
	define('CURL_VERSION_DEBUG', 64);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.76.0
	 */
	define('CURL_VERSION_GSASL', 536870912);

	/**
	 * Built against a GSS-API library. Available since PHP 7.3.0 and cURL 7.38.0
	 */
	define('CURL_VERSION_GSSAPI', 131072);

	/**
	 * Negotiate auth is supported. Available since PHP 7.3.0 and cURL 7.10.6 (deprecated since 7.38.0)
	 */
	define('CURL_VERSION_GSSNEGOTIATE', 32);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.74.0
	 */
	define('CURL_VERSION_HSTS', 268435456);

	/**
	 * HTTP2 support built-in. Available since cURL 7.33.0
	 */
	define('CURL_VERSION_HTTP2', 65536);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.66.0
	 */
	define('CURL_VERSION_HTTP3', 33554432);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURL_VERSION_HTTPS_PROXY', 2097152);

	/**
	 * Internationized Domain Names are supported. Available since PHP 7.3.0 and cURL 7.12.0
	 */
	define('CURL_VERSION_IDN', 1024);

	/**
	 * IPv6-enabled.
	 */
	define('CURL_VERSION_IPV6', 1);

	/**
	 * Kerberos V4 auth is supported.
	 */
	define('CURL_VERSION_KERBEROS4', 2);

	/**
	 * Kerberos V5 auth is supported. Available since PHP 7.0.7 and cURL 7.40.0
	 */
	define('CURL_VERSION_KERBEROS5', 262144);

	/**
	 * Supports files larger than 2GB. Available since cURL 7.33.0
	 */
	define('CURL_VERSION_LARGEFILE', 512);

	/**
	 * libz features are present.
	 */
	define('CURL_VERSION_LIBZ', 8);

	/**
	 * Available since PHP 7.3.0 and cURL 7.56.0
	 */
	define('CURL_VERSION_MULTI_SSL', 4194304);

	/**
	 * NTLM auth is supported. Available since PHP 7.3.0 and cURL 7.10.6
	 */
	define('CURL_VERSION_NTLM', 16);

	/**
	 * NTLM delegation to winbind helper is supported. Available since PHP 7.3.0 and cURL 7.22.0
	 */
	define('CURL_VERSION_NTLM_WB', 32768);

	/**
	 * Mozilla's Public Suffix List, used for cookie domain verification. Available since PHP 7.3.6 and cURL 7.47.0
	 */
	define('CURL_VERSION_PSL', 1048576);

	/**
	 * SPNEGO auth is supported. Available since PHP 7.3.0 and cURL 7.10.8
	 */
	define('CURL_VERSION_SPNEGO', 256);

	/**
	 * SSL options are present.
	 */
	define('CURL_VERSION_SSL', 4);

	/**
	 * Built against Windows SSPI. Available since PHP 7.3.0 and cURL 7.13.2
	 */
	define('CURL_VERSION_SSPI', 2048);

	/**
	 * TLS-SRP auth is supported. Available since PHP 7.3.0 and cURL 7.21.4
	 */
	define('CURL_VERSION_TLSAUTH_SRP', 16384);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.72.0
	 */
	define('CURL_VERSION_UNICODE', 134217728);

	/**
	 * Unix domain sockets support. Available since PHP 7.0.7 and cURL 7.40.0
	 */
	define('CURL_VERSION_UNIX_SOCKETS', 524288);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.72.0
	 */
	define('CURL_VERSION_ZSTD', 67108864);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLALTSVC_H1', 8);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLALTSVC_H2', 16);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLALTSVC_H3', 32);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLALTSVC_READONLYFILE', 4);

	define('CURLAUTH_ANY', -17);

	define('CURLAUTH_ANYSAFE', -18);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.75.0.
	 */
	define('CURLAUTH_AWS_SIGV4', 128);

	define('CURLAUTH_BASIC', 1);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0.
	 */
	define('CURLAUTH_BEARER', 64);

	define('CURLAUTH_DIGEST', 2);

	/**
	 * Available since PHP 7.3.0 and cURL 7.54.1
	 */
	define('CURLAUTH_GSSAPI', 4);

	define('CURLAUTH_GSSNEGOTIATE', 4);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.38.0.
	 */
	define('CURLAUTH_NEGOTIATE', 4);

	define('CURLAUTH_NTLM', 8);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.22.0
	 */
	define('CURLAUTH_NTLM_WB', 32);

	define('CURLE_ABORTED_BY_CALLBACK', 42);

	define('CURLE_BAD_CALLING_ORDER', 44);

	define('CURLE_BAD_CONTENT_ENCODING', 61);

	define('CURLE_BAD_FUNCTION_ARGUMENT', 43);

	define('CURLE_BAD_PASSWORD_ENTERED', 46);

	define('CURLE_COULDNT_CONNECT', 7);

	define('CURLE_COULDNT_RESOLVE_HOST', 6);

	define('CURLE_COULDNT_RESOLVE_PROXY', 5);

	define('CURLE_FAILED_INIT', 2);

	define('CURLE_FILE_COULDNT_READ_FILE', 37);

	define('CURLE_FILESIZE_EXCEEDED', 63);

	define('CURLE_FTP_ACCESS_DENIED', 9);

	define('CURLE_FTP_BAD_DOWNLOAD_RESUME', 36);

	define('CURLE_FTP_CANT_GET_HOST', 15);

	define('CURLE_FTP_CANT_RECONNECT', 16);

	define('CURLE_FTP_COULDNT_GET_SIZE', 32);

	define('CURLE_FTP_COULDNT_RETR_FILE', 19);

	define('CURLE_FTP_COULDNT_SET_ASCII', 29);

	define('CURLE_FTP_COULDNT_SET_BINARY', 17);

	define('CURLE_FTP_COULDNT_STOR_FILE', 25);

	define('CURLE_FTP_COULDNT_USE_REST', 31);

	define('CURLE_FTP_PORT_FAILED', 30);

	define('CURLE_FTP_QUOTE_ERROR', 21);

	define('CURLE_FTP_SSL_FAILED', 64);

	define('CURLE_FTP_USER_PASSWORD_INCORRECT', 10);

	define('CURLE_FTP_WEIRD_227_FORMAT', 14);

	define('CURLE_FTP_WEIRD_PASS_REPLY', 11);

	define('CURLE_FTP_WEIRD_PASV_REPLY', 13);

	define('CURLE_FTP_WEIRD_SERVER_REPLY', 8);

	define('CURLE_FTP_WEIRD_USER_REPLY', 12);

	define('CURLE_FTP_WRITE_ERROR', 20);

	define('CURLE_FUNCTION_NOT_FOUND', 41);

	define('CURLE_GOT_NOTHING', 52);

	define('CURLE_HTTP_NOT_FOUND', 22);

	define('CURLE_HTTP_PORT_FAILED', 45);

	define('CURLE_HTTP_POST_ERROR', 34);

	define('CURLE_HTTP_RANGE_ERROR', 33);

	define('CURLE_LDAP_CANNOT_BIND', 38);

	define('CURLE_LDAP_INVALID_URL', 62);

	define('CURLE_LDAP_SEARCH_FAILED', 39);

	define('CURLE_LIBRARY_NOT_FOUND', 40);

	define('CURLE_MALFORMAT_USER', 24);

	define('CURLE_OBSOLETE', 50);

	define('CURLE_OK', 0);

	define('CURLE_OPERATION_TIMEOUTED', 28);

	define('CURLE_OUT_OF_MEMORY', 27);

	define('CURLE_PARTIAL_FILE', 18);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLE_PROXY', 97);

	define('CURLE_READ_ERROR', 26);

	define('CURLE_RECV_ERROR', 56);

	define('CURLE_SEND_ERROR', 55);

	define('CURLE_SHARE_IN_USE', 57);

	/**
	 * Available since cURL 7.16.1.
	 */
	define('CURLE_SSH', 79);

	define('CURLE_SSL_CACERT', 60);

	define('CURLE_SSL_CERTPROBLEM', 58);

	define('CURLE_SSL_CIPHER', 59);

	define('CURLE_SSL_CONNECT_ERROR', 35);

	define('CURLE_SSL_ENGINE_NOTFOUND', 53);

	define('CURLE_SSL_ENGINE_SETFAILED', 54);

	define('CURLE_SSL_PEER_CERTIFICATE', 60);

	define('CURLE_TELNET_OPTION_SYNTAX', 49);

	define('CURLE_TOO_MANY_REDIRECTS', 47);

	define('CURLE_UNKNOWN_TELNET_OPTION', 48);

	define('CURLE_UNSUPPORTED_PROTOCOL', 1);

	define('CURLE_URL_MALFORMAT', 3);

	define('CURLE_URL_MALFORMAT_USER', 4);

	/**
	 * Available since PHP 7.3.0 and cURL 7.51.0
	 */
	define('CURLE_WEIRD_SERVER_REPLY', 8);

	define('CURLE_WRITE_ERROR', 23);

	/**
	 * Available since PHP 7.0.7 and cURL 7.19.3
	 */
	define('CURLFTP_CREATE_DIR', 1);

	/**
	 * Available since PHP 7.0.7 and cURL 7.19.3
	 */
	define('CURLFTP_CREATE_DIR_NONE', 0);

	/**
	 * Available since PHP 7.0.7 and cURL 7.19.3
	 */
	define('CURLFTP_CREATE_DIR_RETRY', 2);

	define('CURLFTPAUTH_DEFAULT', 0);

	define('CURLFTPAUTH_SSL', 1);

	define('CURLFTPAUTH_TLS', 2);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.15.3.
	 */
	define('CURLFTPMETHOD_DEFAULT', 0);

	define('CURLFTPSSL_ALL', 3);

	define('CURLFTPSSL_CONTROL', 2);

	define('CURLFTPSSL_NONE', 0);

	define('CURLFTPSSL_TRY', 1);

	/**
	 * Available since PHP 7.0.7 and cURL 7.37.0.
	 */
	define('CURLHEADER_SEPARATE', 1);

	/**
	 * Available since PHP 7.0.7 and cURL 7.37.0.
	 */
	define('CURLHEADER_UNIFIED', 0);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.74.0
	 */
	define('CURLHSTS_ENABLE', 1);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.74.0
	 */
	define('CURLHSTS_READONLYFILE', 2);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_APPCONNECT_TIME_T', 6291512);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.84.0
	 */
	define('CURLINFO_CAINFO', null);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.84.0
	 */
	define('CURLINFO_CAPATH', null);

	define('CURLINFO_CONNECT_TIME', 3145733);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_CONNECT_TIME_T', 6291508);

	define('CURLINFO_CONTENT_LENGTH_DOWNLOAD', 3145743);

	/**
	 * Available since PHP 7.3.0 and cURL 7.55.0
	 */
	define('CURLINFO_CONTENT_LENGTH_DOWNLOAD_T', 6291471);

	define('CURLINFO_CONTENT_LENGTH_UPLOAD', 3145744);

	/**
	 * Available since PHP 7.3.0 and cURL 7.55.0
	 */
	define('CURLINFO_CONTENT_LENGTH_UPLOAD_T', 6291472);

	define('CURLINFO_CONTENT_TYPE', 1048594);

	define('CURLINFO_EFFECTIVE_URL', 1048577);

	define('CURLINFO_FILETIME', 2097166);

	/**
	 * Available since PHP 7.3.0 and cURL 7.59.0
	 */
	define('CURLINFO_FILETIME_T', 6291470);

	define('CURLINFO_HEADER_OUT', 2);

	define('CURLINFO_HEADER_SIZE', 2097163);

	/**
	 * As of cURL 7.10.8, this is a legacy alias of <b><code>CURLINFO_RESPONSE_CODE</code></b>
	 */
	define('CURLINFO_HTTP_CODE', 2097154);

	/**
	 * Available since PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_HTTP_VERSION', 2097198);

	define('CURLINFO_LOCAL_IP', 1048617);

	define('CURLINFO_LOCAL_PORT', 2097194);

	define('CURLINFO_NAMELOOKUP_TIME', 3145732);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_NAMELOOKUP_TIME_T', 6291507);

	define('CURLINFO_PRETRANSFER_TIME', 3145734);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_PRETRANSFER_TIME_T', 6291509);

	define('CURLINFO_PRIMARY_IP', 1048608);

	define('CURLINFO_PRIMARY_PORT', 2097192);

	define('CURLINFO_PRIVATE', 1048597);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLINFO_PROTOCOL', 2097200);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLINFO_PROXY_ERROR', 2097211);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLINFO_PROXY_SSL_VERIFYRESULT', 2097199);

	define('CURLINFO_REDIRECT_COUNT', 2097172);

	define('CURLINFO_REDIRECT_TIME', 3145747);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_REDIRECT_TIME_T', 6291511);

	define('CURLINFO_REDIRECT_URL', 1048607);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.76.0
	 */
	define('CURLINFO_REFERER', 1048636);

	define('CURLINFO_REQUEST_SIZE', 2097164);

	/**
	 * Available since cURL 7.10.8
	 */
	define('CURLINFO_RESPONSE_CODE', 2097154);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.66.0
	 */
	define('CURLINFO_RETRY_AFTER', 6291513);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLINFO_SCHEME', 1048625);

	define('CURLINFO_SIZE_DOWNLOAD', 3145736);

	/**
	 * Available since PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_SIZE_DOWNLOAD_T', 6291464);

	define('CURLINFO_SIZE_UPLOAD', 3145735);

	/**
	 * Available since PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_SIZE_UPLOAD_T', 6291463);

	define('CURLINFO_SPEED_DOWNLOAD', 3145737);

	/**
	 * Available since PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_SPEED_DOWNLOAD_T', 6291465);

	define('CURLINFO_SPEED_UPLOAD', 3145738);

	/**
	 * Available since PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_SPEED_UPLOAD_T', 6291466);

	define('CURLINFO_SSL_VERIFYRESULT', 2097165);

	define('CURLINFO_STARTTRANSFER_TIME', 3145745);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_STARTTRANSFER_TIME_T', 6291510);

	define('CURLINFO_TOTAL_TIME', 3145731);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_TOTAL_TIME_T', 6291506);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.19.6
	 */
	define('CURLKHMATCH_LAST', 3);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.19.6
	 */
	define('CURLKHMATCH_MISMATCH', 1);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.19.6
	 */
	define('CURLKHMATCH_MISSING', 2);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.19.6
	 */
	define('CURLKHMATCH_OK', 0);

	define('CURLM_BAD_EASY_HANDLE', 2);

	define('CURLM_BAD_HANDLE', 1);

	define('CURLM_CALL_MULTI_PERFORM', -1);

	define('CURLM_INTERNAL_ERROR', 4);

	define('CURLM_OK', 0);

	define('CURLM_OUT_OF_MEMORY', 3);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.81.0
	 */
	define('CURLMIMEOPT_FORMESCAPE', 1);

	/**
	 * Available since PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_CHUNK_LENGTH_PENALTY_SIZE', 30010);

	/**
	 * Available since PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_CONTENT_LENGTH_PENALTY_SIZE', 30009);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.67.0.
	 */
	define('CURLMOPT_MAX_CONCURRENT_STREAMS', 16);

	/**
	 * Available since PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_MAX_HOST_CONNECTIONS', 7);

	/**
	 * Available since PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_MAX_PIPELINE_LENGTH', 8);

	/**
	 * Available since PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_MAX_TOTAL_CONNECTIONS', 13);

	/**
	 * Available since cURL 7.16.3.
	 */
	define('CURLMOPT_MAXCONNECTS', 6);

	/**
	 * Available since cURL 7.16.0.
	 */
	define('CURLMOPT_PIPELINING', 3);

	/**
	 * Available since PHP 7.1.0 and cURL 7.44.0
	 */
	define('CURLMOPT_PUSHFUNCTION', 20014);

	define('CURLMSG_DONE', 1);

	/**
	 * Available since PHP 7.3.0 and cURL 7.53.0
	 */
	define('CURLOPT_ABSTRACT_UNIX_SOCKET', 10264);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLOPT_ALTSVC', 10287);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLOPT_ALTSVC_CTRL', 286);

	define('CURLOPT_AUTOREFERER', 58);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.75.0
	 */
	define('CURLOPT_AWS_SIGV4', 10305);

	define('CURLOPT_BINARYTRANSFER', 19914);

	define('CURLOPT_BUFFERSIZE', 98);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.87.0
	 */
	define('CURLOPT_CA_CACHE_TIMEOUT', null);

	define('CURLOPT_CAINFO', 10065);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.77.0
	 */
	define('CURLOPT_CAINFO_BLOB', 40309);

	define('CURLOPT_CAPATH', 10097);

	/**
	 * Available since PHP 7.0.7 and cURL 7.49.0
	 */
	define('CURLOPT_CONNECT_TO', 10243);

	define('CURLOPT_CONNECTTIMEOUT', 78);

	define('CURLOPT_CONNECTTIMEOUT_MS', 156);

	define('CURLOPT_COOKIE', 10022);

	define('CURLOPT_COOKIEFILE', 10031);

	define('CURLOPT_COOKIEJAR', 10082);

	/**
	 * Available since cURL 7.14.1
	 */
	define('CURLOPT_COOKIELIST', 10135);

	define('CURLOPT_COOKIESESSION', 96);

	define('CURLOPT_CRLF', 27);

	define('CURLOPT_CUSTOMREQUEST', 10036);

	/**
	 * Available since PHP 7.0.7 and cURL 7.45.0
	 */
	define('CURLOPT_DEFAULT_PROTOCOL', 10238);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLOPT_DISALLOW_USERNAME_IN_URL', 278);

	define('CURLOPT_DNS_CACHE_TIMEOUT', 92);

	/**
	 * Available since PHP 7.0.7 and cURL 7.33.0
	 */
	define('CURLOPT_DNS_INTERFACE', 10221);

	/**
	 * Available since PHP 7.0.7 and cURL 7.33.0
	 */
	define('CURLOPT_DNS_LOCAL_IP4', 10222);

	/**
	 * Available since PHP 7.0.7 and cURL 7.33.0
	 */
	define('CURLOPT_DNS_LOCAL_IP6', 10223);

	/**
	 * Available since PHP 7.3.0 and cURL 7.60.0
	 */
	define('CURLOPT_DNS_SHUFFLE_ADDRESSES', 275);

	define('CURLOPT_DNS_USE_GLOBAL_CACHE', 91);

	/**
	 * Verify the DNS-over-HTTPS server's SSL certificate name fields against the host name. Available as of PHP 8.2.0 and cURL 7.76.0.
	 */
	define('CURLOPT_DOH_SSL_VERIFYHOST', 307);

	/**
	 * Verify the authenticity of the DNS-over-HTTPS server's SSL certificate. Available as of PHP 8.2.0 and cURL 7.76.0.
	 */
	define('CURLOPT_DOH_SSL_VERIFYPEER', 306);

	/**
	 * Tell cURL to verify the status of the DNS-over-HTTPS server certificate using the "Certificate Status Request" TLS extension (OCSP stapling). Available as of PHP 8.2.0 and cURL 7.76.0.
	 */
	define('CURLOPT_DOH_SSL_VERIFYSTATUS', 308);

	/**
	 * Provides the DNS-over-HTTPS URL. Available as of PHP 8.1.0 and cURL 7.62.0.
	 */
	define('CURLOPT_DOH_URL', 10279);

	define('CURLOPT_EGDSOCKET', 10077);

	define('CURLOPT_ENCODING', 10102);

	/**
	 * Available since PHP 7.0.7 and cURL 7.36.0
	 */
	define('CURLOPT_EXPECT_100_TIMEOUT_MS', 227);

	define('CURLOPT_FAILONERROR', 45);

	define('CURLOPT_FILE', 10001);

	define('CURLOPT_FILETIME', 69);

	/**
	 * This constant is not available when open_basedir is enabled.
	 */
	define('CURLOPT_FOLLOWLOCATION', 52);

	define('CURLOPT_FORBID_REUSE', 75);

	define('CURLOPT_FRESH_CONNECT', 74);

	define('CURLOPT_FTP_CREATE_MISSING_DIRS', 110);

	define('CURLOPT_FTP_SSL', 119);

	define('CURLOPT_FTP_USE_EPRT', 106);

	define('CURLOPT_FTP_USE_EPSV', 85);

	define('CURLOPT_FTPAPPEND', 50);

	define('CURLOPT_FTPASCII', null);

	define('CURLOPT_FTPLISTONLY', 48);

	define('CURLOPT_FTPPORT', 10017);

	define('CURLOPT_FTPSSLAUTH', 129);

	/**
	 * Available since PHP 7.3.0 and cURL 7.59.0
	 */
	define('CURLOPT_HAPPY_EYEBALLS_TIMEOUT_MS', 271);

	/**
	 * Available since PHP 7.3.0 and cURL 7.60.0
	 */
	define('CURLOPT_HAPROXYPROTOCOL', 274);

	define('CURLOPT_HEADER', 42);

	define('CURLOPT_HEADERFUNCTION', 20079);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.37.0
	 */
	define('CURLOPT_HEADEROPT', 229);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.74.0
	 */
	define('CURLOPT_HSTS', 10300);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.74.0
	 */
	define('CURLOPT_HSTS_CTRL', 299);

	/**
	 * Available since PHP 7.3.15 and 7.4.3, respectively, and cURL 7.64.0
	 */
	define('CURLOPT_HTTP09_ALLOWED', 285);

	define('CURLOPT_HTTP200ALIASES', 10104);

	define('CURLOPT_HTTP_VERSION', 84);

	define('CURLOPT_HTTPAUTH', 107);

	define('CURLOPT_HTTPGET', 80);

	define('CURLOPT_HTTPHEADER', 10023);

	define('CURLOPT_HTTPPROXYTUNNEL', 61);

	define('CURLOPT_INFILE', 10009);

	define('CURLOPT_INFILESIZE', 14);

	define('CURLOPT_INTERFACE', 10062);

	/**
	 * Issuer SSL certificate from memory blob. Available as of PHP 8.1.0 and cURL 7.71.0.
	 */
	define('CURLOPT_ISSUERCERT_BLOB', 40295);

	/**
	 * Available since PHP 7.3.0 and cURL 7.51.0
	 */
	define('CURLOPT_KEEP_SENDING_ON_ERROR', 245);

	define('CURLOPT_KEYPASSWD', 10026);

	define('CURLOPT_KRB4LEVEL', 10063);

	/**
	 * Available since PHP 7.0.7 and cURL 7.34.0
	 */
	define('CURLOPT_LOGIN_OPTIONS', 10224);

	define('CURLOPT_LOW_SPEED_LIMIT', 19);

	define('CURLOPT_LOW_SPEED_TIME', 20);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.69.0
	 */
	define('CURLOPT_MAIL_RCPT_ALLLOWFAILS', 290);

	/**
	 * Available as of cURL 7.15.5
	 */
	define('CURLOPT_MAX_RECV_SPEED_LARGE', 30146);

	/**
	 * Available as of cURL 7.15.5
	 */
	define('CURLOPT_MAX_SEND_SPEED_LARGE', 30145);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.65.0
	 */
	define('CURLOPT_MAXAGE_CONN', 288);

	define('CURLOPT_MAXCONNECTS', 71);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.11.0
	 */
	define('CURLOPT_MAXFILESIZE_LARGE', 30117);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.80.0
	 */
	define('CURLOPT_MAXLIFETIME_CONN', 314);

	define('CURLOPT_MAXREDIRS', 68);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.81.0
	 */
	define('CURLOPT_MIME_OPTIONS', 315);

	/**
	 * Removed as of cURL 7.15.5; use <b><code>CURLOPT_RETURNTRANSFER</code></b> instead.
	 */
	define('CURLOPT_MUTE', null);

	define('CURLOPT_NETRC', 51);

	define('CURLOPT_NOBODY', 44);

	define('CURLOPT_NOPROGRESS', 43);

	define('CURLOPT_NOSIGNAL', 99);

	/**
	 * Removed as of PHP 7.3.0.
	 */
	define('CURLOPT_PASSWDFUNCTION', null);

	/**
	 * Available since PHP 7.0.7 and cURL 7.42.0
	 */
	define('CURLOPT_PATH_AS_IS', 234);

	/**
	 * Available since PHP 7.0.7 and cURL 7.39.0
	 */
	define('CURLOPT_PINNEDPUBLICKEY', 10230);

	/**
	 * Available since PHP 7.0.7 and cURL 7.43.0
	 */
	define('CURLOPT_PIPEWAIT', 237);

	define('CURLOPT_PORT', 3);

	define('CURLOPT_POST', 47);

	define('CURLOPT_POSTFIELDS', 10015);

	define('CURLOPT_POSTQUOTE', 10039);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PRE_PROXY', 10262);

	define('CURLOPT_PRIVATE', 10103);

	define('CURLOPT_PROGRESSFUNCTION', 20056);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.85.0
	 */
	define('CURLOPT_PROTOCOLS_STR', null);

	define('CURLOPT_PROXY', 10004);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_CAINFO', 10246);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.77.0
	 */
	define('CURLOPT_PROXY_CAINFO_BLOB', 40310);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_CAPATH', 10247);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_CRLFILE', 10260);

	/**
	 * Proxy issuer SSL certificate filename. Available as of PHP 8.1.0 and cURL 7.71.0.
	 */
	define('CURLOPT_PROXY_ISSUERCERT', 10296);

	/**
	 * Proxy issuer SSL certificate from memory blob. Available as of PHP 8.1.0 and cURL 7.71.0.
	 */
	define('CURLOPT_PROXY_ISSUERCERT_BLOB', 40297);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_KEYPASSWD', 10258);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_PINNEDPUBLICKEY', 10263);

	/**
	 * Available since PHP 7.0.7 and cURL 7.43.0
	 */
	define('CURLOPT_PROXY_SERVICE_NAME', 10235);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSL_CIPHER_LIST', 10259);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSL_OPTIONS', 261);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSL_VERIFYHOST', 249);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSL_VERIFYPEER', 248);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSLCERT', 10254);

	/**
	 * SSL proxy client certificate from memory blob. Available as of PHP 8.1.0 and cURL 7.71.0.
	 */
	define('CURLOPT_PROXY_SSLCERT_BLOB', 40293);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSLCERTTYPE', 10255);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSLKEY', 10256);

	/**
	 * Private key for proxy cert from memory blob. Available as of PHP 8.1.0 and cURL 7.71.0.
	 */
	define('CURLOPT_PROXY_SSLKEY_BLOB', 40294);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSLKEYTYPE', 10257);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_SSLVERSION', 250);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLOPT_PROXY_TLS13_CIPHERS', 10277);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_TLSAUTH_PASSWORD', 10252);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_TLSAUTH_TYPE', 10253);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLOPT_PROXY_TLSAUTH_USERNAME', 10251);

	define('CURLOPT_PROXYAUTH', 111);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.37.0
	 */
	define('CURLOPT_PROXYHEADER', 10228);

	define('CURLOPT_PROXYPORT', 59);

	/**
	 * Available as of cURL 7.10.
	 */
	define('CURLOPT_PROXYTYPE', 101);

	define('CURLOPT_PROXYUSERPWD', 10006);

	define('CURLOPT_PUT', 54);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.87.0
	 */
	define('CURLOPT_QUICK_EXIT', null);

	define('CURLOPT_QUOTE', 10028);

	define('CURLOPT_RANDOM_FILE', 10076);

	define('CURLOPT_RANGE', 10007);

	define('CURLOPT_READFUNCTION', 20012);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.85.0.
	 */
	define('CURLOPT_REDIR_PROTOCOLS_STR', null);

	define('CURLOPT_REFERER', 10016);

	/**
	 * Available since PHP 7.3.0 and cURL 7.55.0.
	 */
	define('CURLOPT_REQUEST_TARGET', 10266);

	define('CURLOPT_RESUME_FROM', 21);

	define('CURLOPT_RETURNTRANSFER', 19913);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.66.0
	 */
	define('CURLOPT_SASL_AUTHZID', 10289);

	/**
	 * Available since PHP 7.0.7 and cURL 7.31.0
	 */
	define('CURLOPT_SASL_IR', 218);

	/**
	 * Available since PHP 7.0.7 and cURL 7.43.0
	 */
	define('CURLOPT_SERVICE_NAME', 10236);

	/**
	 * Available since PHP 7.3.0 and cURL 7.55.0
	 */
	define('CURLOPT_SOCKS5_AUTH', 267);

	define('CURLOPT_SSH_AUTH_TYPES', 151);

	/**
	 * Available since PHP 7.3.0 and cURL 7.56.0
	 */
	define('CURLOPT_SSH_COMPRESSION', 268);

	define('CURLOPT_SSH_HOST_PUBLIC_KEY_MD5', 10162);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.80.0
	 */
	define('CURLOPT_SSH_HOST_PUBLIC_KEY_SHA256', 10311);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.84.0
	 */
	define('CURLOPT_SSH_HOSTKEYFUNCTION', null);

	define('CURLOPT_SSH_PRIVATE_KEYFILE', 10153);

	define('CURLOPT_SSH_PUBLIC_KEYFILE', 10152);

	define('CURLOPT_SSL_CIPHER_LIST', 10083);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLOPT_SSL_EC_CURVES', 10298);

	/**
	 * Available since PHP 7.0.7 and cURL 7.36.0
	 */
	define('CURLOPT_SSL_ENABLE_ALPN', 226);

	/**
	 * Available since PHP 7.0.7 and cURL 7.36.0
	 */
	define('CURLOPT_SSL_ENABLE_NPN', 225);

	/**
	 * Available since PHP 7.0.7 and cURL 7.42.0
	 */
	define('CURLOPT_SSL_FALSESTART', 233);

	/**
	 * Available since cURL 7.25.0
	 */
	define('CURLOPT_SSL_OPTIONS', 216);

	define('CURLOPT_SSL_VERIFYHOST', 81);

	define('CURLOPT_SSL_VERIFYPEER', 64);

	/**
	 * Available since PHP 7.0.7 and cURL 7.41.0
	 */
	define('CURLOPT_SSL_VERIFYSTATUS', 232);

	define('CURLOPT_SSLCERT', 10025);

	/**
	 * SSL client certificate from memory blob. Available as of PHP 8.1.0 and cURL 7.71.0.
	 */
	define('CURLOPT_SSLCERT_BLOB', 40291);

	define('CURLOPT_SSLCERTPASSWD', 10026);

	define('CURLOPT_SSLCERTTYPE', 10086);

	define('CURLOPT_SSLENGINE', 10089);

	define('CURLOPT_SSLENGINE_DEFAULT', 90);

	define('CURLOPT_SSLKEY', 10087);

	/**
	 * Private key for client cert from memory blob. Available as of PHP 8.1.0 and cURL 7.71.0.
	 */
	define('CURLOPT_SSLKEY_BLOB', 40292);

	define('CURLOPT_SSLKEYPASSWD', 10026);

	define('CURLOPT_SSLKEYTYPE', 10088);

	define('CURLOPT_SSLVERSION', 32);

	define('CURLOPT_STDERR', 10037);

	/**
	 * Available since PHP 7.0.7 and cURL 7.46.0
	 */
	define('CURLOPT_STREAM_WEIGHT', 239);

	/**
	 * Available since PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURLOPT_SUPPRESS_CONNECT_HEADERS', 265);

	/**
	 * Available since PHP 7.0.7 and cURL 7.49.0
	 */
	define('CURLOPT_TCP_FASTOPEN', 244);

	/**
	 * Available since cURL 7.25.0
	 */
	define('CURLOPT_TCP_KEEPALIVE', 213);

	/**
	 * Available since cURL 7.25.0
	 */
	define('CURLOPT_TCP_KEEPIDLE', 214);

	/**
	 * Available since cURL 7.25.0
	 */
	define('CURLOPT_TCP_KEEPINTVL', 215);

	define('CURLOPT_TCP_NODELAY', 121);

	/**
	 * Available since PHP 7.0.7 and cURL 7.48.0
	 */
	define('CURLOPT_TFTP_NO_OPTIONS', 242);

	define('CURLOPT_TIMECONDITION', 33);

	define('CURLOPT_TIMEOUT', 13);

	define('CURLOPT_TIMEOUT_MS', 155);

	define('CURLOPT_TIMEVALUE', 34);

	/**
	 * Available since PHP 7.3.0 and cURL 7.59.0
	 */
	define('CURLOPT_TIMEVALUE_LARGE', 30270);

	/**
	 * Available since PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLOPT_TLS13_CIPHERS', 10276);

	define('CURLOPT_TRANSFERTEXT', 53);

	/**
	 * Available since PHP 7.0.7 and cURL 7.40.0
	 */
	define('CURLOPT_UNIX_SOCKET_PATH', 10231);

	define('CURLOPT_UNRESTRICTED_AUTH', 105);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.62.0
	 */
	define('CURLOPT_UPKEEP_INTERVAL_MS', 281);

	define('CURLOPT_UPLOAD', 46);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.62.0
	 */
	define('CURLOPT_UPLOAD_BUFFERSIZE', 280);

	define('CURLOPT_URL', 10002);

	define('CURLOPT_USERAGENT', 10018);

	/**
	 * Available since cURL 7.19.1
	 */
	define('CURLOPT_USERNAME', 10173);

	define('CURLOPT_USERPWD', 10005);

	define('CURLOPT_VERBOSE', 41);

	define('CURLOPT_WRITEFUNCTION', 20011);

	define('CURLOPT_WRITEHEADER', 10029);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.86.0
	 */
	define('CURLOPT_WS_OPTIONS', null);

	/**
	 * Available since PHP 7.0.7 and cURL 7.33.0
	 */
	define('CURLOPT_XOAUTH2_BEARER', 10220);

	/**
	 * Available since cURL 7.18.0.
	 */
	define('CURLPAUSE_ALL', 5);

	/**
	 * Available since cURL 7.18.0.
	 */
	define('CURLPAUSE_CONT', 0);

	/**
	 * Available since cURL 7.18.0.
	 */
	define('CURLPAUSE_RECV', 1);

	/**
	 * Available since cURL 7.18.0.
	 */
	define('CURLPAUSE_RECV_CONT', 0);

	/**
	 * Available since cURL 7.18.0.
	 */
	define('CURLPAUSE_SEND', 4);

	/**
	 * Available since cURL 7.18.0.
	 */
	define('CURLPAUSE_SEND_CONT', 0);

	/**
	 * Available since cURL 7.43.0.
	 */
	define('CURLPIPE_HTTP1', 1);

	/**
	 * Available since cURL 7.43.0.
	 */
	define('CURLPIPE_MULTIPLEX', 2);

	/**
	 * Available since cURL 7.43.0.
	 */
	define('CURLPIPE_NOTHING', 0);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.71.0.
	 */
	define('CURLPROTO_MQTT', 268435456);

	/**
	 * Available since PHP 7.0.7 and cURL 7.40.0.
	 */
	define('CURLPROTO_SMB', 67108864);

	/**
	 * Available since PHP 7.0.7 and cURL 7.40.0.
	 */
	define('CURLPROTO_SMBS', 134217728);

	/**
	 * Available since cURL 7.10.
	 */
	define('CURLPROXY_HTTP', 0);

	/**
	 * Available since PHP 7.0.7 and cURL 7.19.3
	 */
	define('CURLPROXY_HTTP_1_0', 1);

	/**
	 * Available since PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLPROXY_HTTPS', 2);

	/**
	 * Available since cURL 7.10.
	 */
	define('CURLPROXY_SOCKS4', 4);

	/**
	 * Available since cURL 7.18.0.
	 */
	define('CURLPROXY_SOCKS4A', 6);

	/**
	 * Available since cURL 7.10.
	 */
	define('CURLPROXY_SOCKS5', 5);

	/**
	 * Available since cURL 7.18.0.
	 */
	define('CURLPROXY_SOCKS5_HOSTNAME', 7);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_BAD_ADDRESS_TYPE', 1);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_BAD_VERSION', 2);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_CLOSED', 3);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_GSSAPI', 4);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_GSSAPI_PERMSG', 5);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_GSSAPI_PROTECTION', 6);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_IDENTD', 7);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_IDENTD_DIFFER', 8);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_LONG_HOSTNAME', 9);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_LONG_PASSWD', 10);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_LONG_USER', 11);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_NO_AUTH', 12);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_OK', 0);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RECV_ADDRESS', 13);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RECV_AUTH', 14);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RECV_CONNECT', 15);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RECV_REQACK', 16);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_ADDRESS_TYPE_NOT_SUPPORTED', 17);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_COMMAND_NOT_SUPPORTED', 18);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_CONNECTION_REFUSED', 19);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_GENERAL_SERVER_FAILURE', 20);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_HOST_UNREACHABLE', 21);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_NETWORK_UNREACHABLE', 22);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_NOT_ALLOWED', 23);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_TTL_EXPIRED', 24);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_UNASSIGNED', 25);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REQUEST_FAILED', 26);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RESOLVE_HOST', 27);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_SEND_AUTH', 28);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_SEND_CONNECT', 29);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_SEND_REQUEST', 30);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_UNKNOWN_FAIL', 31);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_UNKNOWN_MODE', 32);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_USER_REJECTED', 33);

	/**
	 * Available since PHP 7.0.7 and cURL 7.28.0
	 */
	define('CURLSSH_AUTH_AGENT', 16);

	define('CURLSSH_AUTH_ANY', -1);

	define('CURLSSH_AUTH_DEFAULT', -1);

	/**
	 * Available since PHP 7.3.0 and cURL 7.58.0
	 */
	define('CURLSSH_AUTH_GSSAPI', 32);

	define('CURLSSH_AUTH_HOST', 4);

	define('CURLSSH_AUTH_KEYBOARD', 8);

	define('CURLSSH_AUTH_NONE', 0);

	define('CURLSSH_AUTH_PASSWORD', 2);

	define('CURLSSH_AUTH_PUBLICKEY', 1);

	/**
	 * Available since cURL 7.25.0
	 */
	define('CURLSSLOPT_ALLOW_BEAST', 1);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.77.0
	 */
	define('CURLSSLOPT_AUTO_CLIENT_CERT', 32);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.71.0
	 */
	define('CURLSSLOPT_NATIVE_CA', 16);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.68.0
	 */
	define('CURLSSLOPT_NO_PARTIALCHAIN', 4);

	/**
	 * Available since PHP 7.0.7 and cURL 7.44.0
	 */
	define('CURLSSLOPT_NO_REVOKE', 2);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.70.0
	 */
	define('CURLSSLOPT_REVOKE_BEST_EFFORT', 8);

	define('CURLVERSION_NOW', 9);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.86.0
	 */
	define('CURLWS_RAW_MODE', null);

}
