
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <mjpeg_types.h>

#include "global.h"
#include "motionsearch.h"

static void set_pic_params( int decode,
							int b_index,
							pict_data_s *picture )
{
	picture->decode = decode;
	picture->dc_prec = opt_dc_prec;
	picture->secondfield = 0;
	picture->ipflag = 0;

	switch ( picture->pict_type )
	{
		case I_TYPE :
			picture->forw_hor_f_code = 15;
			picture->forw_vert_f_code = 15;
			picture->back_hor_f_code = 15;
			picture->back_vert_f_code = 15;
			picture->sxf = opt_motion_data[0].sxf;
			picture->syf = opt_motion_data[0].syf;
			break;
			
		case P_TYPE :
			picture->forw_hor_f_code = opt_motion_data[0].forw_hor_f_code;
			picture->forw_vert_f_code = opt_motion_data[0].forw_vert_f_code;
			picture->back_hor_f_code = 15;
			picture->back_vert_f_code = 15;
			picture->sxf = opt_motion_data[0].sxf;
			picture->syf = opt_motion_data[0].syf;
			break;
			
		case B_TYPE :
			picture->forw_hor_f_code = opt_motion_data[b_index].forw_hor_f_code;
			picture->forw_vert_f_code = opt_motion_data[b_index].forw_vert_f_code;
			picture->back_hor_f_code = opt_motion_data[b_index].back_hor_f_code;
			picture->back_vert_f_code = opt_motion_data[b_index].back_vert_f_code;
			picture->sxf = opt_motion_data[b_index].sxf;
			picture->syf = opt_motion_data[b_index].syf;
			picture->sxb = opt_motion_data[b_index].sxb;
			picture->syb = opt_motion_data[b_index].syb;
			break;
	}


	picture->prog_frame = 1;
	picture->q_scale_type = 1;
	picture->intravlc = 0; /**/
	picture->altscan = 0;
}

static void create_threads( pthread_t *threads, int num, void *(*start_routine)(void *) )
{
	int i;
	pthread_attr_t *pattr = NULL;

	/* For some Unixen we get a ridiculously small default stack size.
	   Hence we need to beef this up if we can.
	*/
#ifdef HAVE_PTHREADSTACKSIZE
#define MINSTACKSIZE 200000
	pthread_attr_t attr;
	size_t stacksize;

	pthread_attr_init(&attr);
	pthread_attr_getstacksize(&attr, &stacksize);

	if (stacksize < MINSTACKSIZE) pthread_attr_setstacksize(&attr, MINSTACKSIZE);

	pattr = &attr;
#endif

	for(i = 0; i < num; ++i )
		if( pthread_create( &threads[i], pattr, start_routine, NULL ) != 0 )
			mjpeg_error_exit1( "worker thread creation failed: %s", strerror(errno) );


}

static void init_pict_data( pict_data_s *picture )
{
	int i;

	picture->qblocks = (int16_t (*)[64])bufalloc(mb_per_pict*block_count*sizeof(int16_t [64]));
	picture->mbinfo = (struct mbinfo *)bufalloc(mb_per_pict*sizeof(struct mbinfo));
	picture->blocks = (int16_t (*)[64])bufalloc(mb_per_pict*block_count*sizeof(int16_t [64]));
	picture->curref = (uint8_t**)bufalloc( sizeof( uint8_t[3] ) );
	picture->curorg = (uint8_t**)bufalloc( sizeof( uint8_t[3] ) );
	picture->pred = (uint8_t**)bufalloc( sizeof( uint8_t[3] ) );

	for( i = 0 ; i<3; i++)
	{
		int size =  (i==0) ? lum_buffer_size : chrom_buffer_size;
		picture->curref[i] = bufalloc(size);
		picture->curorg[i] = NULL;
		picture->pred[i]   = bufalloc(size);
	}

	sync_guard_init( &picture->completion, 1 );
}

#define R_PICS (MAX_WORKER_THREADS+2)
#define B_PICS (MAX_WORKER_THREADS+2)

static void init_pictures( pict_data_s *ref_pictures, pict_data_s *b_pictures )
{
	int i,j;
	
	for( i = 0; i < R_PICS; ++i ) init_pict_data( &ref_pictures[i] );

	for( i = 0; i < R_PICS; ++i )
	{
		j = (i + 1) % R_PICS;

		ref_pictures[j].oldorg = ref_pictures[i].curorg;
		ref_pictures[j].oldref = ref_pictures[i].curref;
		ref_pictures[j].neworg = ref_pictures[j].curorg;
		ref_pictures[j].newref = ref_pictures[j].curref;
	}

	for( i = 0; i < B_PICS; ++i ) init_pict_data( &b_pictures[i]);
}


static void reconstruct( pict_data_s *picture)
{
	if( picture->pict_type!=B_TYPE)
	{
		iquantize(picture);
		itransform(picture);
	}
}

static mp_semaphore_t worker_available =  SEMAPHORE_INITIALIZER;
static mp_semaphore_t picture_available = SEMAPHORE_INITIALIZER;
static mp_semaphore_t picture_started = SEMAPHORE_INITIALIZER;

static void stencodeworker(pict_data_s *picture)
{
	motion_subsampled_lum(picture);
	motion_estimation(picture);
	predict(picture);
	transform(picture);
	putpict(picture);
	reconstruct(picture);
}

typedef pict_data_s * pict_data_ptr;
volatile static pict_data_ptr picture_to_encode;

static void *parencodeworker(void *start_arg)
{
	pict_data_ptr picture;

	for(;;)
	{
		mp_semaphore_signal( &worker_available, 1);
		mp_semaphore_wait( &picture_available );
		picture = (pict_data_s *)picture_to_encode;
		mp_semaphore_signal( &picture_started, 1);

		motion_subsampled_lum(picture);

		if( ctl_refine_from_rec )
		{
			sync_guard_test( picture->ref_frame_completion );
			motion_estimation(picture);
		}
		else
		{
			motion_estimation(picture);
			sync_guard_test( picture->ref_frame_completion );
		}
		predict(picture);
		transform(picture);
		sync_guard_test( picture->prev_frame_completion );
		putpict(picture);
		reconstruct(picture);

		sync_guard_update( &picture->completion, 1 );
	}

	return NULL;
}


static void parencodepict( pict_data_s *picture )
{
	mp_semaphore_wait( &worker_available );
	picture_to_encode = picture;
	mp_semaphore_signal( &picture_available, 1 );
	mp_semaphore_wait( &picture_started );
}

 
void putseq(void)
{
	int cur_ref_idx = 0;
	int cur_b_idx = 0;
	pict_data_s b_pictures[B_PICS];
	pict_data_s ref_pictures[R_PICS];
	pthread_t worker_threads[MAX_WORKER_THREADS];
	pict_data_s *cur_picture, *old_picture;
	pict_data_s *new_ref_picture, *old_ref_picture;
	int frame_seq = 0;
	int frame_gop = 0;
	int frame_b = 0;

	init_pictures( ref_pictures, b_pictures );
	if( ctl_max_encoding_frames > 1 ) create_threads( worker_threads, 2, parencodeworker );

	old_ref_picture = &ref_pictures[R_PICS-1];
	new_ref_picture = &ref_pictures[cur_ref_idx];
	cur_picture = old_ref_picture;

	/* loop through all frames in encoding/decoding order */
	while( frame_seq<istrm_nframes )
	{
		int type, tempref, prog, tff, repeat, pictSize;
		char seqHead;
		//double quantiser;
		int64_t bitcount;

		frame_pinfo(frame_seq, &type, &tempref, &prog, &tff, &repeat, /*&quantiser,*/ &bitcount, &pictSize, &seqHead );
		if( tempref == 0 ) frame_gop = 0;
		old_picture = cur_picture;
		
		if ( type != 3) //I or P frame
		{
			cur_ref_idx = (cur_ref_idx + 1) % R_PICS;
			old_ref_picture = new_ref_picture;
			new_ref_picture = &ref_pictures[cur_ref_idx];
			new_ref_picture->ref_frame_completion = &old_ref_picture->completion;
			new_ref_picture->prev_frame_completion = &cur_picture->completion;
			
			new_ref_picture->present = frame_seq - frame_gop + tempref;
			
			if (type == 1) new_ref_picture->pict_type = I_TYPE;	
			else new_ref_picture->pict_type = P_TYPE;
			
			cur_picture = new_ref_picture;
			frame_b = 0;
		}
		else
		{
			pict_data_s *new_b_picture;
			cur_b_idx = ( cur_b_idx + 1) % B_PICS;
			new_b_picture = &b_pictures[cur_b_idx];
			new_b_picture->oldorg = new_ref_picture->oldorg;
			new_b_picture->oldref = new_ref_picture->oldref;
			new_b_picture->neworg = new_ref_picture->neworg;
			new_b_picture->newref = new_ref_picture->newref;
			new_b_picture->ref_frame_completion = &new_ref_picture->completion;
			new_b_picture->prev_frame_completion = &cur_picture->completion;
			
			new_b_picture->present = frame_seq - 1;
			new_b_picture->pict_type = B_TYPE;

			cur_picture = new_b_picture;
			frame_b++;
		}
		cur_picture->temp_ref = tempref;
		if( type == 1 ) cur_picture->gop_start = 1;
		else cur_picture->gop_start = 0;
		cur_picture->pict_struct = FRAME_PICTURE;
		cur_picture->repeatfirst = repeat;
		cur_picture->topfirst = tff;
		//cur_picture->quantiser = quantiser;
		cur_picture->bitcount = bitcount;
		cur_picture->pictSize = pictSize;
		cur_picture->seqHead = seqHead;
		
		sync_guard_update( &cur_picture->completion, 0 );
		
//		fprintf(stderr,	"frame: %7d type: %1d temp_ref: %2d ",
//						frame_seq, cur_picture->pict_type, cur_picture->temp_ref);
		
		if( readframe(frame_seq, cur_picture->curorg) )
		    mjpeg_error_exit1("Corrupt frame data in frame %d aborting!", frame_seq );

		set_pic_params( frame_seq, frame_b, cur_picture );
		if( ctl_max_encoding_frames > 1 ) parencodepict( cur_picture );
		else stencodeworker( cur_picture );

		frame_seq++;
		frame_gop++;
	}
	
	if( ctl_max_encoding_frames > 1 ) sync_guard_test( &cur_picture->completion );
	putseqend();
	writeBuffer();
}

