RCS_ID("$Id: FFThumbView.m 558 2006-08-20 14:12:46Z ravemax $")

#import "FFThumbView.h"
#import <math.h>
#import "FFPreferences.h"
#import "NSBezierPath_Additions.h"

@implementation FFThumbView

// Design constants
#define SPACE_TO_BORDER			10
#define SPACE_BETWEEN_THUMBS	10
#define VIEW_BG_COLOR			[NSColor colorWithCalibratedWhite:(230.0f/256.0f) alpha:1.0f]
#define MARK_BORDER				4.0f
#define MARK_OVERLAP			(MARK_BORDER+3)
#define MARK_RADIUS				8.0f
#define MARK_FILL_COLOR			[NSColor colorWithDeviceRed:0.2f green:0.2f blue:0.1f alpha:0.4f]
#define MARK_STROKE_COLOR		[NSColor selectedControlColor]

// Static global variables
static NSImage* loadingImage = nil;

#pragma mark -
#pragma mark Initialization & cleanup

+ (void)initialize {
	if (loadingImage == nil)
		loadingImage = [[NSImage imageNamed:@"loading"] retain];
}

- (void)_createMarkImage {
	NSBezierPath*	path;
	NSSize			markSize;

	// Create the rectangle path
	markSize.width	= m_thumbWidth + 2*MARK_OVERLAP;
	markSize.height	= m_thumbHeight + 2*MARK_OVERLAP;
	path			= [NSBezierPath roundedRectangleWithWidth:markSize.width
													   height:markSize.height
												andEdgeRadius:MARK_RADIUS];
	[path setLineWidth:MARK_BORDER]; 

	// Create the image
	m_markImg = [[NSImage alloc] initWithSize:markSize];
	[m_markImg lockFocus];
	[MARK_FILL_COLOR set];
	[MARK_STROKE_COLOR setStroke];
	[path fill];
	[path stroke];
	[m_markImg unlockFocus];
};


- (id)initWithFrame:(NSRect)frameRect {
	self = [super initWithFrame:frameRect];
	if (self != nil) {
		m_dataSrc			= nil;
		m_thumbWidth		= (float)[[FFPreferences instance] thumbWidth];
		m_thumbHeight		= (float)[[FFPreferences instance] thumbHeight];
		m_adjustY			= SPACE_TO_BORDER + m_thumbHeight;
		m_numRows			= 0;
		m_numCols			= 0;
		m_bgCol				= [VIEW_BG_COLOR retain]; // Never trust autorelease
		m_prevSelectedImg	= -1;
		
		m_markedThumbIndex	= NSNotFound;
		[self _createMarkImage];
		
		[[self window] makeFirstResponder:self];
	}
	return self;
}

- (void)dealloc {
	[m_selectionImg release];
	[m_markImg release];
	[m_bgCol release];
	
	[super dealloc];
}

#pragma mark -
#pragma mark Draw & calc it

- (void)drawRect:(NSRect)rect {
	unsigned	numTb, tbIdx;
	int			selImg;
	FFThumb*	tb;
	NSRect		tbRect;

	[m_bgCol set];
	NSRectFill(rect);

	if (m_dataSrc == nil)
		return;

	FFLOG(6, @"drawRect: %4.1f, %4.1f - %4.1f x %4.1f", 
			rect.origin.x, rect.origin.y, rect.size.width, rect.size.height)
	
	tbRect.size	= NSMakeSize(m_thumbWidth, m_thumbHeight);
	numTb		= [m_dataSrc numberOfThumbsInThumbView:self];
	selImg		= [m_dataSrc selectedImageIndex];
	
	for (tbIdx = 0; tbIdx < numTb; tbIdx++) {
		tb = [m_dataSrc thumbView:self thumbAtIndex:tbIdx];

		// In the to be drawn rectangle?
		tbRect.origin = tb->origin;
		if (!NSIntersectsRect(tbRect, rect))
			continue;
		
		[(THUMB_NOT_LOADED(tb) ? loadingImage : tb->img) compositeToPoint:tb->origin operation:2];
		
		if (tbIdx == selImg)
			[m_selectionImg compositeToPoint:tb->origin operation:NSCompositeSourceOver];
			
		if (tbIdx == m_markedThumbIndex) 
			[m_markImg compositeToPoint:m_markRect.origin operation:NSCompositeSourceOver];
	}
}

- (void)_calc {
	float		frameWd, frameHt;
	unsigned	numTb, tbIdx; 
	FFThumb*	tb;
	
	// Frame width & number of thumbs
	frameWd = [self frame].size.width;
	numTb	= [m_dataSrc numberOfThumbsInThumbView:self];
	
	// Number of rows and columns
	m_numCols = (frameWd - (SPACE_TO_BORDER - SPACE_BETWEEN_THUMBS) - SPACE_TO_BORDER) / (m_thumbWidth + SPACE_BETWEEN_THUMBS);
	m_numRows = ceil((float)numTb / m_numCols);

	// Real (adjusted) space between the thumbs
	m_spaceBtwThumbs = (m_numCols > 1) ?  (frameWd - 2*SPACE_TO_BORDER - m_numCols*m_thumbWidth) / (m_numCols - 1) : 0;

	// Resize the frame
	frameHt	= m_numRows*(m_thumbHeight + SPACE_BETWEEN_THUMBS) - SPACE_BETWEEN_THUMBS + 2*SPACE_TO_BORDER;
	[self setFrameSize:NSMakeSize(frameWd, frameHt)];
	
	FFLOG(3, @"frame-wd=%4.1f, ht=%4.1f : numCols=%d, numRows=%d, realSpace=%2.1f", frameWd, frameHt, m_numCols, m_numRows, m_spaceBtwThumbs)
	
	// Thumb origins
	for (tbIdx = 0; tbIdx < numTb; tbIdx++) {
		 tb = [m_dataSrc thumbView:self thumbAtIndex:tbIdx];
		
		tb->origin.x = (float)SPACE_TO_BORDER + (tbIdx % m_numCols)*(m_spaceBtwThumbs + m_thumbWidth);
		tb->origin.y = frameHt - m_adjustY - (tbIdx / m_numCols)*(SPACE_BETWEEN_THUMBS + m_thumbHeight);
		
		FFLOG(3, @"%d (%p)> x=%4.1f, y=%4.1f", tbIdx, tb, tb->origin.x, tb->origin.y)
	}
	
	// Redraw
	[self setNeedsDisplay:TRUE];	
}

- (void)resizeWithOldSuperviewSize:(NSSize)old {
	[super resizeWithOldSuperviewSize:old];
	if (m_dataSrc != nil)
		[self _calc];
}

#pragma mark -
#pragma mark Mouse events

- (BOOL)acceptsFirstMouse		{ return TRUE; }
- (BOOL)acceptsFirstResponder	{ return TRUE; }

- (BOOL)becomeFirstResponder {
	[[self window] setAcceptsMouseMovedEvents:TRUE];
	return TRUE;
}

- (BOOL)resignFirstResponder {
	[[self window] setAcceptsMouseMovedEvents:FALSE];
	//	[self _removeMark];
	return TRUE;
}

- (unsigned)_thumbIndexForPoint:(NSPoint)pt {
	int			col, row;
	unsigned	tbIdx;
	NSPoint*	tbOrigin;
		
	// Outside of the window)
	if ((pt.x < 0) || (pt.y < 0) || ![self mouse:pt inRect:[self visibleRect]])
		return NSNotFound;
	
	// Col & row
	row = pt.y / (m_thumbHeight + SPACE_BETWEEN_THUMBS);
	col = pt.x / (m_thumbWidth + m_spaceBtwThumbs);
		
	if ((col >= m_numCols) || (row >= m_numRows)) // happens when the view < scrollview
		return NSNotFound;

	// Thumb index
	tbIdx = (m_numRows - row - 1)*m_numCols + col;
	if (tbIdx >= [m_dataSrc numberOfThumbsInThumbView:self])
		return NSNotFound;

	// Space between the thumbs or border
	tbOrigin = &([m_dataSrc thumbView:self thumbAtIndex:tbIdx]->origin);
	if ((pt.x < tbOrigin->x) ||	(pt.x >= tbOrigin->x + m_thumbWidth) || 
		(pt.y < tbOrigin->y))
		return NSNotFound;

	return tbIdx;
}

- (void)mouseDown:(NSEvent*)ev {
	m_mouseDownIndex = [self _thumbIndexForPoint:[self convertPoint:
								   [ev locationInWindow] fromView:nil]];
	m_mouseDrag	= FALSE;
}

- (void)mouseDragged:(NSEvent*)ev {
	if (m_mouseDrag || (m_mouseDownIndex == NSNotFound)) // Drag already init. or invalid index
		return;
	
	m_mouseDrag	= TRUE;

	if ([m_delegate thumbViewWillBeginDrag:m_mouseDownIndex])
		[self dragImage:[m_dataSrc thumbView:self thumbAtIndex:m_mouseDownIndex]->img
					 at:[self convertPoint:[ev locationInWindow] fromView:nil]
				 offset:NSZeroSize
				  event:ev
			 pasteboard:[NSPasteboard pasteboardWithName:NSDragPboard]
				 source:self
			  slideBack:YES];
}

- (NSArray*)namesOfPromisedFilesDroppedAtDestination:(NSURL*)dropDestination {
	return [m_delegate thumbViewNamesOfFilesDroppedAtDestination:dropDestination];
}

- (void)mouseUp:(NSEvent*)ev {
	if (!m_mouseDrag && (m_mouseDownIndex != NSNotFound))
		[m_delegate thumbViewDidClickThumbWithIndex:m_mouseDownIndex];
}

- (void)_removeMark {
	m_markedThumbIndex = NSNotFound;
	[self setNeedsDisplayInRect:m_markRect];
	[self setNeedsDisplay:TRUE];
	[m_delegate thumbViewDidUnmarkThumb];
}

- (void)_updateMark:(unsigned)tbIdx {
	NSPoint*	tbOrigin;
	
	tbOrigin			= &([m_dataSrc thumbView:self thumbAtIndex:tbIdx]->origin);
	m_markedThumbIndex	= tbIdx;
	m_markRect.origin.x	= tbOrigin->x - MARK_OVERLAP;
	m_markRect.origin.y	= tbOrigin->y - MARK_OVERLAP;	
	
	[self setNeedsDisplay:TRUE];
//	[self setNeedsDisplayInRect:m_markRect]; // To fast, old is sometimes not removed

	NSPoint tbCenter = [[self window] convertBaseToScreen:				// view on screen
							[self convertPoint:*tbOrigin toView:nil]];	// inside the view
	tbCenter.x += m_thumbWidth / 2;
	tbCenter.y += m_thumbHeight / 2;

	[m_delegate thumbViewDidMarkThumbWithIndex:tbIdx withCenter:tbCenter];
}

- (void)mouseMoved:(NSEvent*)ev {
	unsigned tbIdx = [self _thumbIndexForPoint:[self convertPoint:
								   [ev locationInWindow] fromView:nil]];
	
	if (tbIdx != m_markedThumbIndex) { // Mark needs to be changed
		if (tbIdx != NSNotFound)
			[self _updateMark:tbIdx];
		else
			[self _removeMark];
	}
}

#pragma mark -
#pragma mark Exported methods

- (void)setDataSource:(id)dataSrc {
	m_dataSrc = dataSrc;
}

- (void)setSelectionImage:(NSImage*)img {
	m_selectionImg = [img retain];
}

- (void)reload {
	m_prevSelectedImg = [m_dataSrc selectedImageIndex];
	[self _calc];
}

- (void)updateSelection {
	// Calc the rect that both (previous and the new) thumb cover
	int			selImg	= [m_dataSrc selectedImageIndex];
	FFThumb*	prevTb, *selTb;
	float		xl, xr, yb, yt;

	prevTb	= [m_dataSrc thumbView:self thumbAtIndex:m_prevSelectedImg],
	selTb	= [m_dataSrc thumbView:self thumbAtIndex:selImg];
	
	if (prevTb->origin.x <= selTb->origin.x) {
		xl	= prevTb->origin.x;
		xr	= selTb->origin.x;
	} else {
		xl	= selTb->origin.x;
		xr	= prevTb->origin.x;
	}
	
	if (prevTb->origin.y <= selTb->origin.y) {
		yb	= prevTb->origin.y;
		yt	= selTb->origin.y;
	} else {
		yb	= selTb->origin.y;
		yt	= prevTb->origin.y;
	}
	
	// Store the selected image index & update both
	m_prevSelectedImg = selImg;
	[self setNeedsDisplayInRect:NSMakeRect(xl, yb, xr - xl + m_thumbWidth,
										   yt - yb + m_thumbHeight)];
}

- (void)thumbLoaded:(FFThumb*)thumb {
	[self setNeedsDisplayInRect:NSMakeRect(thumb->origin.x, thumb->origin.y,
										   m_thumbWidth, m_thumbHeight)];
}

- (NSColor*)backgroundColor {
	return m_bgCol;
}

@end
