RCS_ID("$Id: FFPageView.m 595 2006-09-30 19:11:14Z ravemax $")

#import "FFPageView.h"
#import <OpenGL/glext.h>
#import "FFGL.h"
#import "FFPage.h"
#import "FFOptions.h"

#ifdef FFVIEW_APPLICATION
	#import "FFOSD.h"
	#import "FFShadowRectImage.h"
#endif


@implementation FFPageView

// Enum -> angle values
const static GLfloat RotationAsAngle[OPT_NUM_ROTATION] = { 0.0, 270.0, 180.0, 90.0 };

// OSD padding - space between OSD border and view border
#define OSD_PADDING	20

// Color index
#define RED		0
#define GREEN	1
#define BLUE	2

#pragma mark -
#pragma mark Initialisation & Cleanup

- (void)_setupOpenGLContext {
	[[self openGLContext] makeCurrentContext];
	
	GL(glPixelStorei(GL_UNPACK_ALIGNMENT, 1));
	GL(glEnable(GL_RASTER_POSITION_UNCLIPPED_IBM));
	GL(glDisable(GL_DITHER));

	// Disable unnecessary tests
	GL(glDisable(GL_ALPHA_TEST));
    GL(glDisable(GL_DEPTH_TEST));
	GL(glDisable(GL_SCISSOR_TEST));

	// Enable culling - discard back
	GL(glEnable(GL_CULL_FACE));
	GL(glCullFace(GL_BACK));
	GL(glPolygonMode(GL_FRONT, GL_FILL));
	GL(glPolygonMode(GL_BACK, GL_FILL));
	
	GL(glHint(GL_TRANSFORM_HINT_APPLE, GL_FASTEST));
}

- (void)_setupMaxTileSizeUsing:(int)userSize {
	if (userSize == 0) {
		GLint texMax;
		
		[[self openGLContext] makeCurrentContext];
		GL(glGetIntegerv(GL_MAX_TEXTURE_SIZE, &texMax));
		m_maxTileSize = texMax >> 1; // /2
	} else
		m_maxTileSize = userSize;
	
	FFLOG(8, @"max.tile size = %d", m_maxTileSize);
}

#ifdef FFVIEW_APPLICATION
	- (id)initWithOptions:(FFOptions*)opts
		  backgroundColor:(NSColor*)col
				 lensSize:(NSSize)lensSize
				 lensZoom:(GLfloat)lensZoom
	  doublePageAlignment:(FFDoublePageAlignment)dpa
			  maxTileSize:(int)maxTileSize
			 eventHandler:(id)handler
				   andOSD:(FFOSD*)OSD
#else
	- (id)initWithOptions:(FFOptions*)opts
		  backgroundColor:(NSColor*)col
	  doublePageAlignment:(FFDoublePageAlignment)dpa
			  maxTileSize:(int)maxTileSize
		  andEventHandler:(id)handler
#endif
{
	// Pixel attributes
	NSOpenGLPixelFormatAttribute attrs[] = {
		NSOpenGLPFADoubleBuffer,
		NSOpenGLPFAColorSize, (NSOpenGLPixelFormatAttribute)32,
		NSOpenGLPFAAlphaSize, (NSOpenGLPixelFormatAttribute)8,
		NSOpenGLPFADepthSize, (NSOpenGLPixelFormatAttribute)32,
		NSOpenGLPFAWindow,
		(NSOpenGLPixelFormatAttribute)nil
    };
	
	// Get pixel format from OpenGL
    NSOpenGLPixelFormat* pixelFormat = [[NSOpenGLPixelFormat alloc] initWithAttributes:attrs];
	if (pixelFormat == nil)
		return nil;
	
	// Create the view
    self = [super initWithFrame:NSMakeRect(0.0f, 0.0f, 0.0f, 0.0f) pixelFormat:pixelFormat];
	[pixelFormat release];

	if (self != nil) {
		[self _setupOpenGLContext];
		[self _setupMaxTileSizeUsing:maxTileSize];
		
		// Member variables
		m_dpAlignment	= dpa;
		m_eventHandler	= handler;
		m_cursor		= nil;
		m_numPages		= 0;
		m_opts			= [opts retain];
		m_needsLayout	= FALSE;
		
#ifdef FFVIEW_APPLICATION
		m_OSD			= OSD;
				
		m_lensWidth		= lensSize.width;
		m_lensHeight	= lensSize.height;		
		m_lensRectImg	= [[FFShadowRectImage alloc] initWithWidth:m_lensWidth andHeight:m_lensHeight];
		m_lensEnabled	= FALSE;
		m_lensVisible	= FALSE;
		m_lensZoom		= lensZoom;
#endif
		
		[self setBackgroundColor:col];
	}
	return self;
}

- (void)dealloc {
#ifdef FFVIEW_APPLICATION
	[m_lensRectImg release];
#endif
	[m_opts release];
	
	[super dealloc];
}


#pragma mark -
#pragma mark Page managment

- (void)addPage:(FFPage*)page {
	if ((page != nil) && (m_numPages < MAX_NUM_PAGES)) {
		m_pages[m_numPages] = page;
		m_numPages++;
	}
}

- (void)removeAllPages {
	m_numPages = 0;
}

- (void)finishedAdding {
	// Calculate the total size of the combined, unmodifed pages
	if ((m_numPages == 1) || ([m_opts pageMode] == OPT_MODE_SINGLE)) {
		m_totalWd	= m_pages[0]->width;
		m_totalHt	= m_pages[0]->height;
	} else {
		m_totalWd	= m_pages[0]->width + m_pages[1]->width;
		m_totalHt	= MAX(m_pages[0]->height, m_pages[1]->height);
	}
	
	// Update the background color if auto
	if (m_bgAuto) 
		[self _autoBackgroundColor];
}

- (NSSize)preferredSize {
	if (m_numPages == 0)
		return NSMakeSize(0.0f, 0.0f);

	NSSize psize;

	if (([m_opts rotation] == OPT_NO_ROTATION) || ([m_opts rotation] == OPT_ROTATE_180)) {
		psize.width		= m_totalWd;
		psize.height	= m_totalHt;
	} else {
		psize.width		= m_totalHt;
		psize.height	= m_totalWd;
	}
		
	if ([m_opts scaling] == OPT_NO_SCALING) {
		psize.width		*= [m_opts zoomMultiplier];
		psize.height	*= [m_opts zoomMultiplier];
	}

	return psize;
}

- (void)doLayout {
	FFLOG(6, @"FFGLView: DoLayout - numPages=%d", m_numPages);
	
	if (m_numPages == 0)
		return;
	
	// 1. X,Y of the pages
	if (m_totalWd == m_pages[0]->width) { // Single page: Fast comparism
		m_pages[0]->x	= 0.0f;
		m_pages[0]->y	= 0.0f;
	} else {
		m_pages[0]->x	= ceilf((float)(m_pages[0]->width	- m_totalWd) / 2);
		m_pages[1]->x	= ceilf((float)m_pages[0]->width / 2);
		
		if (m_dpAlignment == ALIGNMENT_TOP) {
			m_pages[0]->y	= ceilf((m_totalHt - m_pages[0]->height) / 2);
			m_pages[1]->y	= ceilf((m_totalHt - m_pages[1]->height) / 2);
		} else if (m_dpAlignment == ALIGNMENT_CENTER) {
			m_pages[0]->y	= 0.0f;
			m_pages[1]->y	= 0.0f;
		} else { // ALIGNMENT_BOTTOM
			m_pages[0]->y	= ceilf((m_pages[0]->height - m_totalHt) / 2); // Top negative
			m_pages[1]->y	= ceilf((m_pages[1]->height - m_totalHt) / 2);		
		}
	}
	
	// 2. Apply rotation mode
	if (([m_opts rotation] == OPT_ROTATE_RIGHT) || ([m_opts rotation] == OPT_ROTATE_LEFT)) {
		m_totalWdSR	= m_totalHt;
		m_totalHtSR	= m_totalWd;
	} else {
		m_totalWdSR	= m_totalWd;
		m_totalHtSR	= m_totalHt;
	}
	m_angle	= RotationAsAngle[[m_opts rotation]];

	// 3. Calculate the internal zoom factor
	if ([m_opts scaling] == OPT_NO_SCALING) {
		m_scaleFactor = [m_opts zoomMultiplier];
	}
	else if ([m_opts scaling] == OPT_FIT_TO_WINDOW) {
		m_scaleFactor = MIN(width / m_totalWdSR, height / m_totalHtSR);

		if ((m_scaleFactor > 1.0f) && ([m_opts noBlowUp]))
			m_scaleFactor = 1.0f;
	} else // OPT_PAGEWIDTH
		m_scaleFactor = width / m_totalWdSR;		
	
	// 4. Correct the combined size
	m_totalWdSR	*= m_scaleFactor;
	m_totalHtSR	*= m_scaleFactor;
	
	// 5. Panning possible (X, Y direction)
	m_panX	= m_totalWdSR > width;
	m_panY	= m_totalHtSR > height;
	
	// 6. X, Y within the view & min/max position
	if (m_panX) {
		m_maxX		= (m_totalWdSR - width) / 2;
		m_minX		= - m_maxX;
		m_relativeX = (m_totalWdSR - width) / width;
		m_x			= m_maxX; // left
	} else
		m_x	= 0.0f;
	
	if (m_panY) {
		m_maxY		= (m_totalHtSR - height) / 2;
		m_minY		= - m_maxY;
		m_relativeY	= (m_totalHtSR - height) / height;
		m_y			= m_minY; // top
	} else
		m_y	= 0.0f;
	
	// No more
	m_needsLayout = FALSE;
	
	FFLOG(6, @"after:doLayout:%@", [self description])
}

- (void)doLayoutAndDisplay {
	[self doLayout];
	[self setNeedsDisplay:TRUE];
}

#pragma mark -
#pragma mark Overloaded view methods (especially the drawing code)

- (void)_drawAxis {
	glBegin(GL_LINES);
	glEnable(GL_LINE_STIPPLE);
	glLineStipple(1, 0xAAAA);
	glLineWidth (2.0);
	
	glColor3f(1.0, 0.0, 0.0);
	glVertex3i(0, 0, 0);
	glVertex3i(200, 0, 0);
	
	glColor3f(0.0, 1.0, 0.0);
	glVertex3i(0, 0, 0);
	glVertex3i(0, 200, 0);
	
	glColor3f(0.0, 0.0, 1.0);
	glVertex3i(0, 0, 0);
	glVertex3i(0, 0, 200);
	
	GL(glEnd());
	glDisable(GL_LINE_STIPPLE);
}

static inline void _setViewport(GLint x, GLint y, float wd, float ht) {
	GL(glViewport(x, y, wd, ht));
	GL(glMatrixMode(GL_PROJECTION));
	GL(glLoadIdentity());
	GL(glOrtho(- (wd / 2), wd / 2,
			   - (ht / 2), ht / 2, 0, 1)); 
	GL(glMatrixMode(GL_MODELVIEW));
}

static inline void _drawImage(GLfloat x, GLfloat y, float wd, float ht, const GLvoid* data) {
	lockGLPixelPipeline();
	GL(glEnable(GL_BLEND));
	GL(glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA)); // Don't ignore the image alpha
	GL(glPixelZoom(1.0f, -1.0f)); // flip image
	GL(glRasterPos2f(x, y));
	GL(glDrawPixels(wd, ht, GL_RGBA, GL_UNSIGNED_BYTE, data));
	GL(glDisable(GL_BLEND));
	unlockGLPixelPipeline();
}

- (void)drawRect:(NSRect)rect {
	int	i; 
	
	FFLOG(6, @"FFGLView: DrawRect:%@", NSStringFromRect(rect));
	
	[[self openGLContext] makeCurrentContext];
	
	// Clear the screen & initialize projection matrix
	GL(glClearColor(m_bgColor[RED], m_bgColor[GREEN], m_bgColor[BLUE], 1.0f));
	GL(glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT));

	if (m_numPages == 0)
		return;

	// Texture stuff
	GL(glEnable(GL_TEXTURE_RECTANGLE_EXT));
	GL(glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE));
	
	if ([m_opts antialiasing]) {
		GL(glTexParameteri(GL_TEXTURE_RECTANGLE_EXT, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
		GL(glTexParameteri(GL_TEXTURE_RECTANGLE_EXT, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
	} else {
		GL(glTexParameteri(GL_TEXTURE_RECTANGLE_EXT, GL_TEXTURE_MIN_FILTER, GL_NEAREST));
		GL(glTexParameteri(GL_TEXTURE_RECTANGLE_EXT, GL_TEXTURE_MAG_FILTER, GL_NEAREST));
	}

	// Draw the image
	_setViewport(0, 0, width, height);	
	
	GL(glLoadIdentity());
	glPushMatrix();
	
	// Panning as well as the options & zoom
	glTranslatef(ceilf(m_x), ceilf(m_y), 0.0f);
	glRotatef(m_angle, 0, 0, 1);
	glScalef(m_scaleFactor, m_scaleFactor, 1.0f);
	
	for (i = 0; i < m_numPages; i++)
		[m_pages[i] draw];

	GL(glDisable(GL_TEXTURE_RECTANGLE_EXT));
	
	glPopMatrix();
	
#ifdef FFVIEW_APPLICATION
	// Draw the OSD if visible
	if ([m_OSD isVisible])
		_drawImage(m_OSDX, m_OSDY, [m_OSD width], [m_OSD height], [m_OSD dataRGBA]);
	
	// Draw the magnifying lens if visible
	if (m_lensVisible) {
		_drawImage(m_lensFrameX, m_lensFrameY, m_lensRectImg->width, m_lensRectImg->height, m_lensRectImg->dataRGBA);

		// Lens as new viewport
		GL(glEnable(GL_TEXTURE_RECTANGLE_EXT));
	
		_setViewport(m_lensViewportX, m_lensViewportY, m_lensWidth, m_lensHeight);

		glTranslatef(m_lensX, m_lensY, 0.0f);
		glRotatef(m_angle, 0, 0, 1);
		glScalef(m_lensZoom, m_lensZoom, 1.0f);

		for (i = 0; i < m_numPages; i++)
			[m_pages[i] draw];	

		GL(glDisable(GL_TEXTURE_RECTANGLE_EXT));
	}
#endif	
	
	[[self openGLContext] flushBuffer];
}

- (void)reshape {
	NSSize bsize = [self bounds].size;

	// No clue why OS X calls reshape w/ the same size
	if ((width == bsize.width) && (height == bsize.height))
		return;
	
	// Store the new size
	width	= bsize.width;
	height	= bsize.height;
	
	// Inform OpenGL about the resizing
	[[self openGLContext] makeCurrentContext];
	[[self openGLContext] update];

#ifdef FFVIEW_APPLICATION	
	// Calculate OSD coordinates
	m_OSDX	= - width / 2 +	OSD_PADDING;
	m_OSDY	= height / 2 - OSD_PADDING;
#endif
	
	// Relayout
	[self doLayout];
	
#ifdef FFVIEW_APPLICATION
	if (m_lensEnabled)
		[self _calcMagnifyingLens];
#endif
}

#pragma mark -
#pragma mark Misc

- (GLint)maxTileSize {
	return m_maxTileSize;
}

- (void)readBufferWithWidth:(GLint)wd andHeight:(GLint)ht intoBuffer:(GLubyte*)buffer {
	[[self openGLContext] makeCurrentContext];

	glReadBuffer(GL_FRONT);
	glReadPixels((width - wd) / 2, (height - ht) / 2, wd, ht,
				 GL_RGBA, GL_UNSIGNED_BYTE, buffer);
}

#pragma mark -
#pragma mark Responder - mostly forwarded

- (BOOL)acceptsFirstResponder   { return TRUE; }

- (BOOL)becomeFirstResponder {
	[[self window] setAcceptsMouseMovedEvents:TRUE];
	return TRUE;
}

- (BOOL)resignFirstResponder {
	[[self window] setAcceptsMouseMovedEvents:FALSE];
	return TRUE;
}

- (void)mouseMoved:(NSEvent*)event			{ [m_eventHandler mouseMoved:event]; }
- (void)flagsChanged:(NSEvent*)event		{ [m_eventHandler flagsChanged:event]; }
- (void)scrollWheel:(NSEvent*)event			{ [m_eventHandler scrollWheel:event]; }
- (void)rightMouseDown:(NSEvent*)event		{ [m_eventHandler rightMouseDown:event]; }
- (void)rightMouseUp:(NSEvent*)event		{ [m_eventHandler rightMouseUp:event]; }
- (void)rightMouseDragged:(NSEvent*)event   { [m_eventHandler rightMouseDragged:event]; }

- (void)mouseDown:(NSEvent*)event {
	m_mouseDrag = FALSE;
	[super mouseDown:event];
}

- (void)mouseDragged:(NSEvent*)event {
	m_mouseDrag = TRUE;
	[super mouseDragged:event];
}

- (void)mouseUp:(NSEvent*)event {
	if (!m_mouseDrag && (m_numPages >= 1)) {
		// Page click
		if (m_totalWd == m_pages[0]->width)
			[m_eventHandler pageClicked:0];
		else {
			float sepPos = ((float)m_pages[0]->width / 2 + m_pages[0]->x) * m_scaleFactor;
			if (m_angle >= 180.0f) // 180, 270
				sepPos *= -1.0f;
			
			BOOL first;
			if ((m_angle == 0.0f) || (m_angle == 180.0f))
				first = ([event locationInWindow].x - width / 2 < sepPos);
			else
				first = ([event locationInWindow].y - height / 2 < sepPos);

			first ^= (m_angle >= 180.0f);				
			[m_eventHandler pageClicked:first ? 0 : 1];
		}
	}
	
	[super mouseUp:event];
}	

#pragma mark -
#pragma mark Magnifying lens (FFVIEW_APPLICATION)

#ifdef FFVIEW_APPLICATION

// TRUE = needs display
- (BOOL)_calcMagnifyingLens {
	// Coordinates of the scaled, rotated image
	float	cx	= m_lensMouseX - (width - m_totalWdSR) / 2;
	float	cy	= m_lensMouseY - (height - m_totalHtSR) / 2;
	
	// Outside the image
	if ((cx < 0) || (cx >= m_totalWdSR) || (cy < 0) || (cy > m_totalHtSR)) {
		BOOL displ = m_lensVisible;
		m_lensVisible = FALSE;
		return displ;
	}
	m_lensVisible = TRUE;

	// Center zoom box
	m_lensViewportX	= m_lensMouseX - m_lensWidth / 2;
	m_lensViewportY	= m_lensMouseY - m_lensHeight / 2;
	
	// Calculate the x,y coord
	if (([m_opts rotation] == OPT_NO_ROTATION) || ([m_opts rotation] == OPT_ROTATE_180)) {
		m_lensX = m_totalWd / 2;
		m_lensY = m_totalHt / 2;
	} else {
		m_lensX = m_totalHt / 2;
		m_lensY = m_totalWd / 2;
	}
	m_lensX -= (cx - m_x) / m_scaleFactor;
	m_lensY -= (cy - m_y) / m_scaleFactor;
	m_lensX	*= m_lensZoom;
	m_lensY	*= m_lensZoom;	
	
	// Calculate the frame coordinates
	m_lensFrameX = m_lensViewportX - width / 2 - m_lensRectImg->padding;
	m_lensFrameY = m_lensViewportY - height / 2 + m_lensRectImg->padding
					+ m_lensHeight; // + padding, +m_lensHeight due to the image flipping
	
	return TRUE;
}

- (void)updateMagnifyingLensPoint:(NSPoint)pt {
	m_lensMouseX	= pt.x;
	m_lensMouseY	= pt.y;
	
	// Calculate frame + lensViewport only here?
	if ([self _calcMagnifyingLens])
		[self setNeedsDisplay:TRUE];
}

- (void)showMagnifyingLensAtPoint:(NSPoint)pt {
	m_lensEnabled = TRUE;
	[self updateMagnifyingLensPoint:pt];
}

- (void)hideMagnifyingLens {
	m_lensEnabled	= FALSE;
	m_lensVisible	= FALSE;
}

#endif // FFVIEW_APPLICATION

#pragma mark -
#pragma mark Panning

- (BOOL)panWithDeltaX:(float)x andDeltaY:(float)y relative:(BOOL)relative {
	BOOL needsDisplay = FALSE;

	// Horizontal
	if (m_panX && (x != 0.0f)) {
		if (relative)
			x *= m_relativeX;
		
		if ((x < 0) && (m_x > m_minX)) { // Left
			m_x += x;
			if (m_x < m_minX)
				m_x = m_minX;
			needsDisplay = TRUE;
		} else if ((x > 0) && (m_x < m_maxX)) { // Right
			m_x += x;
			if (m_x > m_maxX)
				m_x = m_maxX;
			needsDisplay = TRUE;
		}
	}
	
	// Vertical
	if (m_panY && (y != 0.0f)) {
		if (relative)
			y *= m_relativeY;
		
		if ((y < 0) && (m_y < m_maxY)) { // Up
			m_y -= y; //+
			if (m_y > m_maxY)
				m_y = m_maxY;
			needsDisplay = TRUE;
		} else if ((y > 0) && (m_y > m_minY)) { // Down
			m_y -= y;
			if (m_y < m_minY)
				m_y = m_minY;
			needsDisplay = TRUE;
		}
	}	
	
	if (needsDisplay) {
#ifdef FFVIEW_APPLICATION
		if (m_lensVisible)
			(void)[self _calcMagnifyingLens];
#endif		
		
		[self setNeedsDisplay:TRUE];
	}
	
	return needsDisplay;
}

- (BOOL)pageUp {
	return [self panWithDeltaX:0.0f andDeltaY:height relative:FALSE];
}

- (BOOL)pageDown {
	return [self panWithDeltaX:0.0f andDeltaY:-height relative:FALSE];
}

- (BOOL)canPageUp		{ return m_panY && (m_y > m_minY); }
- (BOOL)canPageDown		{ return m_panY && (m_y < m_maxY); }
- (BOOL)canMoveLeft		{ return m_panX && (m_x < m_maxX); }
- (BOOL)canMoveRight	{ return m_panX && (m_x > m_minX); }

#pragma mark -
#pragma mark Background color & double page alignment

- (void)_autoBackgroundColor {
	if (m_numPages > 0) {
		m_bgColor[RED]		= (float)m_pages[0]->data[1] / 255; // Simply, stupid
		m_bgColor[GREEN]	= (float)m_pages[0]->data[2] / 255;
		m_bgColor[BLUE]		= (float)m_pages[0]->data[3] / 255;
		
		FFLOG(8, @"AutoBG: %1.3f,%1.3f,%1.3f",
			  m_bgColor[RED], m_bgColor[GREEN], m_bgColor[BLUE]);
	} else {
		m_bgColor[RED]		= 0.0f;
		m_bgColor[GREEN]	= 0.0f;
		m_bgColor[BLUE]		= 0.0f;
	}
}

- (void)setBackgroundColor:(NSColor*)col {
	if (col == nil) {
		m_bgAuto = TRUE;
		[self _autoBackgroundColor];
	} else {
		m_bgAuto = FALSE;
		m_bgColor[RED]		= [col redComponent];
		m_bgColor[GREEN]	= [col greenComponent];
		m_bgColor[BLUE]		= [col blueComponent];
	}
}

- (void)setDoublePageAlignment:(FFDoublePageAlignment)dpa {
	m_dpAlignment = dpa;
	if (m_numPages > 0) {
		[self doLayout];
		[self setNeedsDisplay:TRUE];
	}
}

#pragma mark -
#pragma mark Cursor

- (void)resetCursorRects {
	if (m_cursor != nil)
		[self addCursorRect:[self frame] cursor:m_cursor];
}

- (void)setCursor:(NSCursor*)cursor {
	if (cursor == nil) { // normal cursor
		if (m_cursor != nil) {
			[self removeCursorRect:[self frame]
							cursor:m_cursor];
			m_cursor = cursor;
			[[self window] invalidateCursorRectsForView:self];
		}
	} else { // special cursor
		m_cursor = cursor;
		[self resetCursorRects];
		[[self window] invalidateCursorRectsForView:self];
	}
}

// Own refresh validation code 
- (void)setNeedsLayout:(BOOL)flag {
	m_needsLayout = flag;
}

- (BOOL)needsLayout {
	return m_needsLayout;
}

- (void)doLayoutAndDisplayIfNeeded {
	if (m_needsLayout)
		[self doLayoutAndDisplay];
}

#pragma mark -
#pragma mark Development

- (NSString*)description {
	NSMutableString* dm;
	
	dm = [NSMutableString stringWithFormat:@"FFPageView (%p): %.1fx%.1f @ %.1f,%.1f - max.Tile %d ",
		self, [self frame].size.width, [self frame].size.height, 
		[self frame].origin.x, [self frame].origin.y, [self maxTileSize]]; 
	
	if ([self window])
		[dm appendFormat:@"- win frame: %@ ", NSStringFromRect([[self window] frame])];
	
	if (m_numPages > 0) {
		[dm appendFormat:@"- total %.1fx%.1f (scale=%.4f)",
			m_totalWd, m_totalHt, m_scaleFactor];
		int i;
		for (i = 0; i < m_numPages; i++)
			[dm appendFormat:@"\n- %d: %p", i, m_pages[i]];
		
	} else
		[dm appendString:@"<no pages>"];
	
	return dm;
}

@end
