RCS_ID("$Id: FFImageFile.m 473 2005-09-25 14:52:06Z ravemax $")

#import "FFImageFile.h"
#import <FFArchive/FFArchive.h>
#import "NSArray_Additions.h"
#import "NSString_Additions.h"

@implementation FFImageFile

// Dict keys
static NSString* KeyFileType	= @"filetype";
static NSString* KeyFullPath	= @"fullpath";
static NSString* KeyArchive		= @"archive";
static NSString* KeyFileSize	= @"filesize";
static NSString* KeyDate		= @"date";

// Always increased - for user sort order
static unsigned			highestFileIndex	= FILE_INDEX_UNDEFINED + 1; 
static NSDictionary*	displayNameAttr		= nil;
static NSFileManager*	fileManager;

#pragma mark -
#pragma mark Initialisation and cleanup

+ (void)initialize {
	if (displayNameAttr == nil) {
		NSMutableParagraphStyle* style = [[NSParagraphStyle defaultParagraphStyle] mutableCopy];

        [style setLineBreakMode:NSLineBreakByTruncatingMiddle];
        displayNameAttr = [[NSDictionary alloc] initWithObjectsAndKeys:
			style, NSParagraphStyleAttributeName,
			nil];
        [style release];
		
		//
		fileManager = [NSFileManager defaultManager];
	}
}

- (id)initWithNormalFile:(NSString*)fullPath {
	if ([super init]) {
		m_fullPath		= [fullPath retain];
		m_archive		= nil;
		m_fileType		= IMG_NORMAL_FILE;
		m_displayName   = [[NSAttributedString alloc] initWithString:[fullPath lastPathComponent] 
														  attributes:displayNameAttr];
		m_fileIndex		= highestFileIndex++;
		m_userSortIndex	= m_fileIndex;
		m_filtered		= FALSE;
		m_thumb.img		= NULL;
		
		// Filesize and date
		NSDictionary* fa = [fileManager fileAttributesAtPath:fullPath traverseLink:TRUE];
		m_fileSize		= [[fa objectForKey:NSFileSize] unsignedLongValue];
		m_date			= [[fa fileCreationDate] retain];
	}
	return self;
}

- (id)initWithFile:(NSString*)fullPath
	   fromArchive:(FFArchive*)archive 
	  withFileSize:(unsigned long)fsize
		   andDate:(NSDate*)date {
	
	if ([super init]) {
		m_fullPath		= [fullPath retain];
		m_archive		= [archive retain];
		m_fileType		= IMG_PART_OF_ARCHIVE;
		m_displayName   = [[NSAttributedString alloc] initWithString:
							[NSString stringWithFormat:@"%@:%@", [[archive filePath] lastPathComponent], 
								[fullPath lastPathComponent]]
							attributes:displayNameAttr];
		m_fileIndex		= highestFileIndex++;
		m_userSortIndex	= m_fileIndex;
		m_fileSize		= fsize;
		m_date			= [date retain];
		m_filtered		= FALSE;
		m_thumb.img		= NULL;
	}
	return self;
}

- (void)dealloc {
	[m_fullPath release];
	if (m_archive != nil)
		[m_archive release];
	[m_displayName release];
	[m_date release];
	if (m_thumb.img != NULL) {
		[m_thumb.img release];
		[m_thumb.name release];
		[m_thumb.desc release];
	}
	
	[super dealloc];
}

+ (FFImageFile*)normalFile:(NSString*)fullPath {
	return [[[self alloc] initWithNormalFile:fullPath] autorelease];
}

+ (FFImageFile*)file:(NSString*)fullPath fromArchive:(FFArchive*)archive 
		withFileSize:(unsigned long)fsize andDate:(NSDate*)date {
	return [[[self alloc] initWithFile:fullPath fromArchive:archive
						  withFileSize:fsize andDate:date] autorelease];
}

#pragma mark -
#pragma mark Storage

- (NSDictionary*)asDictRelativeToPath:(NSString*)path {
	NSMutableDictionary* dict = [NSMutableDictionary dictionaryWithCapacity:2];
	
	[dict setObject:[NSNumber numberWithInt:m_fileType] forKey:KeyFileType];
	if (m_fileType == IMG_PART_OF_ARCHIVE) {
		[dict setObject:[[m_archive filePath] stringRelativeToPath:path] forKey:KeyArchive];
		[dict setObject:m_fullPath forKey:KeyFullPath];
		[dict setObject:[NSNumber numberWithUnsignedLong:m_fileSize] forKey:KeyFileSize];
		[dict setObject:m_date forKey:KeyDate];
	} else 
		[dict setObject:[m_fullPath stringRelativeToPath:path] forKey:KeyFullPath];

	return dict;
}

+ (FFImageFileType)typeFromDict:(NSDictionary*)dict {
	return (FFImageFileType)[[dict objectForKey:KeyFileType] intValue];
}

+ (NSString*)absolutFullpathFromDict:(NSDictionary*)dict relativeToPath:(NSString*)path {
	if ([self typeFromDict:dict] == IMG_PART_OF_ARCHIVE)
		return [[dict objectForKey:KeyArchive] absolutWithRootPath:path];
	return [[dict objectForKey:KeyFullPath] absolutWithRootPath:path];
}


+ (FFImageFile*)normalFileFromDict:(NSDictionary*)dict relativeToPath:(NSString*)path {
	return [[[self alloc] initWithNormalFile:[[dict objectForKey:KeyFullPath] absolutWithRootPath:path]] autorelease];
}

+ (FFImageFile*)fileFromDict:(NSDictionary*)dict fromArchive:(FFArchive*)archive {
	NSNumber* fs = [dict objectForKey:KeyFileSize];
	if (fs != nil) // 0.95+ format
		return [[[self alloc] initWithFile:[dict objectForKey:KeyFullPath]
							   fromArchive:archive
							  withFileSize:[fs unsignedLongValue]
								   andDate:[dict objectForKey:KeyDate]] autorelease];
	// till 0.9
	return [[[self alloc] initWithFile:[dict objectForKey:KeyFullPath]
						   fromArchive:archive
						  withFileSize:0 andDate:[NSDate date]] autorelease];
}


#pragma mark -
#pragma mark Getters

- (NSString*)fullPath				{ return m_fullPath; }
- (FFArchive*)archive				{ return m_archive; }
- (FFImageFileType)fileType			{ return m_fileType; }
- (NSAttributedString*)displayName	{ return m_displayName; }
- (unsigned)fileIndex				{ return m_fileIndex; }
- (unsigned)userSortIndex			{ return m_userSortIndex; }
- (unsigned long)fileSize		{ return m_fileSize; }
- (NSDate*)date						{ return m_date; }
- (BOOL)filtered					{ return m_filtered; }
- (FFThumb*)thumb					{ return &m_thumb; }

#pragma mark -
#pragma mark Setters

- (void)setUserSortIndex:(unsigned)idx {
	m_userSortIndex = idx;
}

- (void)setFiltered:(BOOL)filtered {
	m_filtered = filtered;
}

#pragma mark -
#pragma mark Comparators

- (NSComparisonResult)compareFileIndexAsc:(FFImageFile*)other { 
	// NSOrderedSame can never happen
	return ((m_userSortIndex < [other userSortIndex]) ? NSOrderedAscending : NSOrderedDescending);
}

- (NSComparisonResult)compareFileIndexDesc:(FFImageFile*)other { 
	// NSOrderedSame can never happen
	return ((m_userSortIndex > [other userSortIndex]) ? NSOrderedAscending : NSOrderedDescending);
}

- (NSComparisonResult)compareNumericAsc:(FFImageFile*)other {
	return [[m_displayName string] caseInsensitiveAndNumericCompare:[[other displayName] string]];
}

#define RETURN_SWAPPED_COMPARE_RES(RES) \
	if (RES == NSOrderedAscending) \
		return NSOrderedDescending; \
	if (RES == NSOrderedDescending) \
		return NSOrderedAscending; \
	return NSOrderedSame

- (NSComparisonResult)compareNumericDesc:(FFImageFile*)other {
	NSComparisonResult r = [[m_displayName string] caseInsensitiveAndNumericCompare:[[other displayName] string]];
	RETURN_SWAPPED_COMPARE_RES(r);
}

- (NSComparisonResult)compareAlphaAsc:(FFImageFile*)other {
	return [[m_displayName string] caseInsensitiveCompare:[[other displayName] string]]; 
}

- (NSComparisonResult)compareAlphaDesc:(FFImageFile*)other {
	NSComparisonResult r = [[m_displayName string] caseInsensitiveCompare:[[other displayName] string]];
	RETURN_SWAPPED_COMPARE_RES(r);
}

- (NSComparisonResult)comparePathAsc:(FFImageFile*)other {
	// normal:normal/archive
	if (m_fileType == IMG_NORMAL_FILE) {
		if ([other fileType] == IMG_NORMAL_FILE)
			return [m_fullPath caseInsensitiveAndNumericCompare:[other fullPath]];
		return  [m_fullPath caseInsensitiveAndNumericCompare:[[other archive] filePath]];
	}
	// archive:normal
	if ([other fileType] == IMG_NORMAL_FILE)
		return [[m_archive filePath] caseInsensitiveAndNumericCompare:[other fullPath]];
	
	// archive:archive
	NSComparisonResult res = [[m_archive filePath] caseInsensitiveAndNumericCompare:[[other archive] filePath]];
	if (res != NSOrderedSame)
		return res;	
	return  [m_fullPath caseInsensitiveAndNumericCompare:[other fullPath]];
}

- (NSComparisonResult)comparePathDesc:(FFImageFile*)other {
	NSComparisonResult r = [self comparePathAsc:other];
	RETURN_SWAPPED_COMPARE_RES(r);
}

- (NSComparisonResult)compareDateAsc:(FFImageFile*)other {
	return [m_date compare:[other date]];
}

- (NSComparisonResult)compareDateDesc:(FFImageFile*)other {
	NSComparisonResult r = [m_date compare:[other date]];
	RETURN_SWAPPED_COMPARE_RES(r);
}

- (NSComparisonResult)compareFileSizeAsc:(FFImageFile*)other {
	if (m_fileSize < [other fileSize])
		return NSOrderedAscending;
	if (m_fileSize > [other fileSize])
		return NSOrderedDescending;
	return NSOrderedSame;
}

- (NSComparisonResult)compareFileSizeDesc:(FFImageFile*)other {
	if (m_fileSize < [other fileSize])
		return NSOrderedDescending;
	if (m_fileSize > [other fileSize])
		return NSOrderedAscending;
	return NSOrderedSame;
}

@end
