/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <string.h>
#include <stdlib.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <gdk/gdkprivate.h>
#include <GL/glx.h>
#include "gtkglwindow.h"

#define CHILD_SPACING     1
#define DEFAULT_LEFT_POS  4
#define DEFAULT_TOP_POS   4
#define DEFAULT_SPACING   7


enum {
    GLINIT,
    GLDRAW,
    GLRESIZE,
    LAST_SIGNAL
};


static void gtk_glwindow_class_init    ( GtkGLWindowClass  *klass );
static void gtk_glwindow_init          ( GtkGLWindow       *glwindow );
static void gtk_glwindow_realize       ( GtkWidget         *widget );
static void gtk_glwindow_size_request  ( GtkWidget         *widget,
					 GtkRequisition    *requisition );
static void gtk_glwindow_size_allocate ( GtkWidget         *widget,
					 GtkAllocation     *allocation );
static gint gtk_glwindow_expose        ( GtkWidget         *widget,
					 GdkEventExpose    *event );
static gint gtk_glwindow_resize        ( GtkWidget         *widget,
					 GtkAllocation     *alloc );
static void gtk_glwindow_marshal_signal1( GtkObject *object,
					  GtkSignalFunc func,
					  gpointer func_data,
					  GtkArg *args );

static gint glwindow_signals[LAST_SIGNAL] = { 0 };


guint
gtk_glwindow_get_type()
{
    static guint glwindow_type = 0;

    if( !glwindow_type )
    {
	GtkTypeInfo glwindow_info =
	{
	    "GtkGLWindow",
	    sizeof( GtkGLWindow ),
	    sizeof( GtkGLWindowClass ),
	    (GtkClassInitFunc) gtk_glwindow_class_init,
	    (GtkObjectInitFunc) gtk_glwindow_init,
	    (GtkArgFunc) NULL,
	};

	glwindow_type = gtk_type_unique( gtk_widget_get_type (),
					 &glwindow_info );
    }

    return glwindow_type;
}

static void
gtk_glwindow_class_init( GtkGLWindowClass *klass )
{
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;

    object_class = (GtkObjectClass*) klass;
    widget_class = (GtkWidgetClass*) klass;

    glwindow_signals[GLINIT] =
	gtk_signal_new( "glinit",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET(GtkGLWindowClass,glinit),
			gtk_signal_default_marshaller,
			GTK_TYPE_NONE, 0 );
    glwindow_signals[GLDRAW] =
	gtk_signal_new( "gldraw",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET(GtkGLWindowClass,gldraw),
			gtk_signal_default_marshaller,
			GTK_TYPE_NONE, 0 );
    glwindow_signals[GLRESIZE] =
	gtk_signal_new( "glresize",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET(GtkGLWindowClass,glresize),
			gtk_glwindow_marshal_signal1,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_POINTER );

    gtk_object_class_add_signals( object_class, glwindow_signals,
				  LAST_SIGNAL );

    widget_class->realize = gtk_glwindow_realize;
    widget_class->size_request = gtk_glwindow_size_request;
    widget_class->size_allocate = gtk_glwindow_size_allocate;
    widget_class->expose_event = gtk_glwindow_expose;

    /* default handlers */
    klass->glinit = NULL;
    klass->gldraw = NULL;
    klass->glresize = NULL;
}

static void
gtk_glwindow_init( GtkGLWindow *glwindow )
{
    GTK_WIDGET_SET_FLAGS( glwindow, GTK_CAN_FOCUS );
}

GtkWidget*
gtk_glwindow_new( int *glattrib, gint attribLength, guint flags )
{
    GtkGLWindow *glwin;
    glwin = (GtkGLWindow *) gtk_type_new( gtk_glwindow_get_type() );
    
    glwin->glattrib = g_new( int, attribLength );
    memcpy( glwin->glattrib, glattrib, attribLength*sizeof(int) );

    /* I don't want to parse the glattrib's to find out if we're using
       double buffering */
    glwin->swapbuffers = !( flags & GTK_GLWIN_NO_DOUBLE_BUFFER );
    return GTK_WIDGET( glwin );
}

void
gtk_glwindow_draw( GtkGLWindow *glwindow )
{
    if( !glwindow || !GTK_WIDGET_REALIZED(GTK_WIDGET(glwindow)) )
	return;
	
    /* set context first */
    gtk_glwindow_make_current( glwindow );
    gtk_signal_emit( GTK_OBJECT(glwindow), glwindow_signals[GLDRAW] );

    if( glwindow->swapbuffers )
    {
	GdkWindowPrivate *w = (GdkWindowPrivate *) glwindow->widget.window;
	gtk_glwindow_make_current( glwindow );
	glXSwapBuffers( w->xdisplay, w->xwindow );
    }
}

void
gtk_glwindow_make_current( GtkGLWindow *glwindow )
{
    GdkWindowPrivate *w = (GdkWindowPrivate *) glwindow->widget.window;
    glXMakeCurrent( w->xdisplay, w->xwindow, glwindow->context );
}

static void
gtk_glwindow_realize( GtkWidget *widget )
{
    GtkGLWindow *glwindow;
    GdkWindowPrivate *pparent;
    XVisualInfo *visInfo;
    int scrnum;
    GdkWindowAttr gdkAttr;
    gint attrib_mask;

    g_return_if_fail( widget != NULL );
    g_return_if_fail( GTK_IS_GLWINDOW(widget) );

    glwindow = GTK_GLWINDOW( widget );
    GTK_WIDGET_SET_FLAGS( widget, GTK_REALIZED );

    pparent = (GdkWindowPrivate *) widget->parent->window;
    scrnum = DefaultScreen( pparent->xdisplay );
    visInfo = glXChooseVisual( pparent->xdisplay, scrnum, glwindow->glattrib );
    if( !visInfo )
    {
	g_print( "Cannot create OpenGL visual\n" );
	gtk_exit( 1 );
    }
    g_free( glwindow->glattrib );
    glwindow->glattrib = NULL;

    glwindow->context = glXCreateContext( pparent->xdisplay, visInfo, 0, 1 );
    
    gdkAttr.window_type = GDK_WINDOW_CHILD;
    gdkAttr.x = widget->allocation.x;
    gdkAttr.y = widget->allocation.y;
    gdkAttr.width = widget->allocation.width;
    gdkAttr.height = widget->allocation.height;
    gdkAttr.wclass = GDK_INPUT_OUTPUT;
    gdkAttr.visual = gdk_visual_get_best_with_both( visInfo->depth,
						    GDK_VISUAL_TRUE_COLOR );
    gdkAttr.colormap = gtk_widget_get_colormap( widget );
    gdkAttr.event_mask = ( gtk_widget_get_events(widget) |
			   GDK_EXPOSURE_MASK |
			   GDK_BUTTON_PRESS_MASK |
			   GDK_BUTTON_RELEASE_MASK |
			   GDK_ENTER_NOTIFY_MASK |
			   GDK_LEAVE_NOTIFY_MASK );
    attrib_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

    widget->window = gdk_window_new( widget->parent->window, &gdkAttr,
				     attrib_mask );
    gdk_window_set_user_data( widget->window, glwindow );
    widget->style = gtk_style_attach( widget->style, widget->window );
    gtk_style_set_background( widget->style, widget->window,
			      GTK_STATE_NORMAL );

    gtk_glwindow_make_current( glwindow );
    gtk_signal_emit( GTK_OBJECT(glwindow), glwindow_signals[GLINIT] );
}

static void
gtk_glwindow_size_request( GtkWidget      *widget,
			   GtkRequisition *requisition )
{
    requisition->width = 100;
    requisition->height = 100;
}

static void
gtk_glwindow_size_allocate( GtkWidget     *widget,
			    GtkAllocation *allocation )
{
    GtkGLWindow *glwindow;
    GtkAllocation child_allocation;
    gint border_width;

    g_return_if_fail( widget != NULL );
    g_return_if_fail( GTK_IS_GLWINDOW(widget) );
    g_return_if_fail( allocation != NULL );

    widget->allocation = *allocation;
    if( GTK_WIDGET_REALIZED(widget) )
    {
	glwindow = GTK_GLWINDOW( widget );

	gdk_window_move_resize( widget->window,
				allocation->x, allocation->y,
				allocation->width, allocation->height );
	gtk_glwindow_resize( widget, allocation );
    }
}

static gint
gtk_glwindow_expose( GtkWidget *widget,
		     GdkEventExpose *event )
{
    g_return_val_if_fail( widget != NULL, 0 );
    g_return_val_if_fail( GTK_IS_GLWINDOW(widget), 0 );
    gtk_glwindow_draw( GTK_GLWINDOW(widget) );
    return 0;
}

static gint
gtk_glwindow_resize( GtkWidget *widget, GtkAllocation *a )
{
    GdkWindowPrivate *priv;
    GtkGLWindow *glwindow;
    
    g_return_val_if_fail( widget != NULL, 0);
    g_return_val_if_fail( GTK_IS_GLWINDOW(widget), 0 );

    priv = (GdkWindowPrivate *) widget->window;
    glwindow = GTK_GLWINDOW( widget );

    /* set context first */
    glXMakeCurrent( priv->xdisplay, priv->xwindow, glwindow->context );
    gtk_signal_emit( GTK_OBJECT(widget), glwindow_signals[GLRESIZE], a );
    return 0;
}


void
gtk_glwindow_marshal_signal1( GtkObject *object,
			      GtkSignalFunc func,
			      gpointer func_data,
			      GtkArg *args )
{
    void (*rfunc)( GtkObject *, gpointer, gpointer );

    rfunc = (void(*)(GtkObject *,gpointer,gpointer)) func;
    (*rfunc)( object, GTK_VALUE_POINTER(args[0]), func_data );
}
