// Copyright 2012-2022 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

// NOTE: The following code was generated by "scripts/unicode.py", do not edit directly

use core::cmp::Ordering;

#[derive(Clone, Copy, Debug, PartialEq, Eq)]
struct WidthInfo(u16);

impl WidthInfo {
    /// No special handling necessary
    const DEFAULT: Self = Self(0);
    const LINE_FEED: Self = Self(0b0000000000000001);
    const EMOJI_MODIFIER: Self = Self(0b0000000000000010);
    const REGIONAL_INDICATOR: Self = Self(0b0000000000000011);
    const SEVERAL_REGIONAL_INDICATOR: Self = Self(0b0000000000000100);
    const EMOJI_PRESENTATION: Self = Self(0b0000000000000101);
    const ZWJ_EMOJI_PRESENTATION: Self = Self(0b0001000000000110);
    const VS16_ZWJ_EMOJI_PRESENTATION: Self = Self(0b1001000000000110);
    const KEYCAP_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0001000000000111);
    const VS16_KEYCAP_ZWJ_EMOJI_PRESENTATION: Self = Self(0b1001000000000111);
    const REGIONAL_INDICATOR_ZWJ_PRESENTATION: Self = Self(0b0000000000001001);
    const EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION: Self = Self(0b0000000000001010);
    const ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION: Self = Self(0b0000000000001011);
    const TAG_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000010000);
    const TAG_D1_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000010001);
    const TAG_D2_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000010010);
    const TAG_D3_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000010011);
    const TAG_A1_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000011001);
    const TAG_A2_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000011010);
    const TAG_A3_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000011011);
    const TAG_A4_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000011100);
    const TAG_A5_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000011101);
    const TAG_A6_END_ZWJ_EMOJI_PRESENTATION: Self = Self(0b0000000000011110);
    const VARIATION_SELECTOR_15: Self = Self(0b0100000000000000);
    const VARIATION_SELECTOR_16: Self = Self(0b1000000000000000);
    const JOINING_GROUP_ALEF: Self = Self(0b0011000011111111);
    #[cfg(feature = "cjk")]
    const COMBINING_LONG_SOLIDUS_OVERLAY: Self = Self(0b0011110011111111);
    #[cfg(feature = "cjk")]
    const SOLIDUS_OVERLAY_ALEF: Self = Self(0b0011100011111111);
    const HEBREW_LETTER_LAMED: Self = Self(0b0011100000000000);
    const ZWJ_HEBREW_LETTER_LAMED: Self = Self(0b0011110000000000);
    const BUGINESE_LETTER_YA: Self = Self(0b0011100000000001);
    const ZWJ_BUGINESE_LETTER_YA: Self = Self(0b0011110000000001);
    const BUGINESE_VOWEL_SIGN_I_ZWJ_LETTER_YA: Self = Self(0b0011110000000010);
    const TIFINAGH_CONSONANT: Self = Self(0b0011100000000011);
    const ZWJ_TIFINAGH_CONSONANT: Self = Self(0b0011110000000011);
    const TIFINAGH_JOINER_CONSONANT: Self = Self(0b0011110000000100);
    const LISU_TONE_LETTER_MYA_NA_JEU: Self = Self(0b0011110000000101);
    const OLD_TURKIC_LETTER_ORKHON_I: Self = Self(0b0011100000000110);
    const ZWJ_OLD_TURKIC_LETTER_ORKHON_I: Self = Self(0b0011110000000110);
    const KHMER_COENG_ELIGIBLE_LETTER: Self = Self(0b0011110000000111);

    /// Whether this width mode is ligature_transparent
    /// (has 5th MSB set.)
    fn is_ligature_transparent(self) -> bool {
        (self.0 & 0b0000_1000_0000_0000) == 0b0000_1000_0000_0000
    }

    /// Sets 6th MSB.
    fn set_zwj_bit(self) -> Self {
        Self(self.0 | 0b0000_0100_0000_0000)
    }

    /// Has top bit set
    fn is_emoji_presentation(self) -> bool {
        (self.0 & 0b1000_0000_0000_0000) == 0b1000_0000_0000_0000
    }

    /// Has top bit set
    fn is_zwj_emoji_presentation(self) -> bool {
        (self.0 & 0b1011_0000_0000_0000) == 0b1001_0000_0000_0000
    }

    /// Set top bit
    fn set_emoji_presentation(self) -> Self {
        if (self.0 & 0b0010_0000_0000_0000) == 0b0010_0000_0000_0000
            || (self.0 & 0b1001_0000_0000_0000) == 0b0001_0000_0000_0000
        {
            Self(self.0 | 0b1000_0000_0000_0000)
        } else {
            Self::VARIATION_SELECTOR_16
        }
    }

    /// Clear top bit
    fn unset_emoji_presentation(self) -> Self {
        if (self.0 & 0b0010_0000_0000_0000) == 0b0010_0000_0000_0000 {
            Self(self.0 & 0b0111_1111_1111_1111)
        } else {
            Self::DEFAULT
        }
    }

    /// Has 2nd bit set
    fn is_text_presentation(self) -> bool {
        (self.0 & 0b0100_0000_0000_0000) == 0b0100_0000_0000_0000
    }

    /// Set 2nd bit
    fn set_text_presentation(self) -> Self {
        if (self.0 & 0b0010_0000_0000_0000) == 0b0010_0000_0000_0000 {
            Self(self.0 | 0b0100_0000_0000_0000)
        } else {
            Self(0b0100_0000_0000_0000)
        }
    }

    /// Clear 2nd bit
    fn unset_text_presentation(self) -> Self {
        Self(self.0 & 0b1011_1111_1111_1111)
    }
}

/// The version of [Unicode](http://www.unicode.org/)
/// that this version of unicode-width is based on.
pub const UNICODE_VERSION: (u8, u8, u8) = (15, 1, 0);

/// Returns the [UAX #11](https://www.unicode.org/reports/tr11/) based width of `c` by
/// consulting a multi-level lookup table.
///
/// # Maintenance
/// The tables themselves are autogenerated but this function is hardcoded. You should have
/// nothing to worry about if you re-run `unicode.py` (for example, when updating Unicode.)
/// However, if you change the *actual structure* of the lookup tables (perhaps by editing the
/// `make_tables` function in `unicode.py`) you must ensure that this code reflects those changes.
#[inline]
fn lookup_width(c: char) -> (u8, WidthInfo) {
    let cp = c as usize;

    let t1_offset = WIDTH_ROOT.0[cp >> 13];

    // Each sub-table in WIDTH_MIDDLE is 7 bits, and each stored entry is a byte,
    // so each sub-table is 128 bytes in size.
    // (Sub-tables are selected using the computed offset from the previous table.)
    let t2_offset = WIDTH_MIDDLE.0[usize::from(t1_offset)][cp >> 7 & 0x3F];

    // Each sub-table in WIDTH_LEAVES is 6 bits, but each stored entry is 2 bits.
    // This is accomplished by packing four stored entries into one byte.
    // So each sub-table is 2**(7-2) == 32 bytes in size.
    // Since this is the last table, each entry represents an encoded width.
    let packed_widths = WIDTH_LEAVES.0[usize::from(t2_offset)][cp >> 2 & 0x1F];

    // Extract the packed width
    let width = packed_widths >> (2 * (cp & 0b11)) & 0b11;

    if width < 3 {
        (width, WidthInfo::DEFAULT)
    } else {
        match c {
            '\u{A}' => (1, WidthInfo::LINE_FEED),
            '\u{5DC}' => (1, WidthInfo::HEBREW_LETTER_LAMED),
            '\u{622}'..='\u{882}' => (1, WidthInfo::JOINING_GROUP_ALEF),
            '\u{1780}'..='\u{17AF}' => (1, WidthInfo::KHMER_COENG_ELIGIBLE_LETTER),
            '\u{17D8}' => (3, WidthInfo::DEFAULT),
            '\u{1A10}' => (1, WidthInfo::BUGINESE_LETTER_YA),
            '\u{2D31}'..='\u{2D6F}' => (1, WidthInfo::TIFINAGH_CONSONANT),
            '\u{A4FC}'..='\u{A4FD}' => (1, WidthInfo::LISU_TONE_LETTER_MYA_NA_JEU),
            '\u{FE0E}' => (0, WidthInfo::VARIATION_SELECTOR_15),
            '\u{FE0F}' => (0, WidthInfo::VARIATION_SELECTOR_16),
            '\u{10C03}' => (1, WidthInfo::OLD_TURKIC_LETTER_ORKHON_I),
            '\u{1F1E6}'..='\u{1F1FF}' => (1, WidthInfo::REGIONAL_INDICATOR),
            '\u{1F3FB}'..='\u{1F3FF}' => (2, WidthInfo::EMOJI_MODIFIER),
            _ => (2, WidthInfo::EMOJI_PRESENTATION),
        }
    }
}

/// Returns the [UAX #11](https://www.unicode.org/reports/tr11/) based width of `c`, or
/// `None` if `c` is a control character.
/// Ambiguous width characters are treated as narrow.
#[inline]
pub fn single_char_width(c: char) -> Option<usize> {
    if c < '\u{7F}' {
        if c >= '\u{20}' {
            // U+0020 to U+007F (exclusive) are single-width ASCII codepoints
            Some(1)
        } else {
            // U+0000 to U+0020 (exclusive) are control codes
            None
        }
    } else if c >= '\u{A0}' {
        // No characters >= U+00A0 are control codes, so we can consult the lookup tables
        Some(lookup_width(c).0.into())
    } else {
        // U+007F to U+00A0 (exclusive) are control codes
        None
    }
}

/// Returns the [UAX #11](https://www.unicode.org/reports/tr11/) based width of `c`.
/// Ambiguous width characters are treated as narrow.
#[inline]
fn width_in_str(c: char, mut next_info: WidthInfo) -> (i8, WidthInfo) {
    if next_info.is_emoji_presentation() {
        if starts_emoji_presentation_seq(c) {
            let width = if next_info.is_zwj_emoji_presentation() {
                0
            } else {
                2
            };
            return (width, WidthInfo::EMOJI_PRESENTATION);
        } else {
            next_info = next_info.unset_emoji_presentation();
        }
    }
    if c <= '\u{A0}' {
        match c {
            // According to the spec, LF should be width 1, which is how it is often rendered when it is forced to have a single-line rendering
            // However, this makes it harder to use this crate to calculate line breaks, and breaks assumptions of downstream crates.
            // https://github.com/unicode-rs/unicode-width/issues/60
            '\n' => (0, WidthInfo::LINE_FEED),
            '\r' if next_info == WidthInfo::LINE_FEED => (0, WidthInfo::DEFAULT),
            _ => (1, WidthInfo::DEFAULT),
        }
    } else {
        // Fast path
        if next_info != WidthInfo::DEFAULT {
            if c == '\u{FE0F}' {
                return (0, next_info.set_emoji_presentation());
            }
            if c == '\u{FE0E}' {
                return (0, next_info.set_text_presentation());
            }
            if next_info.is_text_presentation() {
                if starts_non_ideographic_text_presentation_seq(c) {
                    return (1, WidthInfo::DEFAULT);
                } else {
                    next_info = next_info.unset_text_presentation();
                }
            }
            if next_info.is_ligature_transparent() {
                if c == '\u{200D}' {
                    return (0, next_info.set_zwj_bit());
                } else if is_ligature_transparent(c) {
                    return (0, next_info);
                }
            }

            match (next_info, c) {
                // Arabic Lam-Alef ligature
                (
                    WidthInfo::JOINING_GROUP_ALEF,
                    '\u{644}' | '\u{6B5}'..='\u{6B8}' | '\u{76A}' | '\u{8A6}' | '\u{8C7}',
                ) => return (0, WidthInfo::DEFAULT),
                (WidthInfo::JOINING_GROUP_ALEF, _) if is_transparent_zero_width(c) => {
                    return (0, WidthInfo::JOINING_GROUP_ALEF);
                }

                // Hebrew Alef-ZWJ-Lamed ligature
                (WidthInfo::ZWJ_HEBREW_LETTER_LAMED, '\u{05D0}') => {
                    return (0, WidthInfo::DEFAULT);
                }

                // Khmer coeng signs
                (WidthInfo::KHMER_COENG_ELIGIBLE_LETTER, '\u{17D2}') => {
                    return (-1, WidthInfo::DEFAULT);
                }

                // Buginese <a, -i> ZWJ ya ligature
                (WidthInfo::ZWJ_BUGINESE_LETTER_YA, '\u{1A17}') => {
                    return (0, WidthInfo::BUGINESE_VOWEL_SIGN_I_ZWJ_LETTER_YA)
                }
                (WidthInfo::BUGINESE_VOWEL_SIGN_I_ZWJ_LETTER_YA, '\u{1A15}') => {
                    return (0, WidthInfo::DEFAULT)
                }

                // Tifinagh bi-consonants
                (WidthInfo::TIFINAGH_CONSONANT | WidthInfo::ZWJ_TIFINAGH_CONSONANT, '\u{2D7F}') => {
                    return (1, WidthInfo::TIFINAGH_JOINER_CONSONANT);
                }
                (WidthInfo::ZWJ_TIFINAGH_CONSONANT, '\u{2D31}'..='\u{2D65}' | '\u{2D6F}') => {
                    return (0, WidthInfo::DEFAULT);
                }
                (WidthInfo::TIFINAGH_JOINER_CONSONANT, '\u{2D31}'..='\u{2D65}' | '\u{2D6F}') => {
                    return (-1, WidthInfo::DEFAULT);
                }

                // Lisu tone letter combinations
                (WidthInfo::LISU_TONE_LETTER_MYA_NA_JEU, '\u{A4F8}'..='\u{A4FB}') => {
                    return (0, WidthInfo::DEFAULT);
                }

                // Old Turkic ligature
                (WidthInfo::ZWJ_OLD_TURKIC_LETTER_ORKHON_I, '\u{10C32}') => {
                    return (0, WidthInfo::DEFAULT);
                }
                // Emoji modifier
                (WidthInfo::EMOJI_MODIFIER, _) if is_emoji_modifier_base(c) => {
                    return (0, WidthInfo::EMOJI_PRESENTATION);
                }

                // Regional indicator
                (
                    WidthInfo::REGIONAL_INDICATOR | WidthInfo::SEVERAL_REGIONAL_INDICATOR,
                    '\u{1F1E6}'..='\u{1F1FF}',
                ) => return (1, WidthInfo::SEVERAL_REGIONAL_INDICATOR),

                // ZWJ emoji
                (
                    WidthInfo::EMOJI_PRESENTATION
                    | WidthInfo::SEVERAL_REGIONAL_INDICATOR
                    | WidthInfo::EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION
                    | WidthInfo::ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION
                    | WidthInfo::EMOJI_MODIFIER,
                    '\u{200D}',
                ) => return (0, WidthInfo::ZWJ_EMOJI_PRESENTATION),
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, '\u{20E3}') => {
                    return (0, WidthInfo::KEYCAP_ZWJ_EMOJI_PRESENTATION);
                }
                (WidthInfo::VS16_ZWJ_EMOJI_PRESENTATION, _) if starts_emoji_presentation_seq(c) => {
                    return (0, WidthInfo::EMOJI_PRESENTATION)
                }
                (WidthInfo::VS16_KEYCAP_ZWJ_EMOJI_PRESENTATION, '0'..='9' | '#' | '*') => {
                    return (0, WidthInfo::EMOJI_PRESENTATION)
                }
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, '\u{1F1E6}'..='\u{1F1FF}') => {
                    return (1, WidthInfo::REGIONAL_INDICATOR_ZWJ_PRESENTATION);
                }
                (
                    WidthInfo::REGIONAL_INDICATOR_ZWJ_PRESENTATION
                    | WidthInfo::ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION,
                    '\u{1F1E6}'..='\u{1F1FF}',
                ) => return (-1, WidthInfo::EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION),
                (
                    WidthInfo::EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION,
                    '\u{1F1E6}'..='\u{1F1FF}',
                ) => return (3, WidthInfo::ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION),
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, '\u{1F3FB}'..='\u{1F3FF}') => {
                    return (0, WidthInfo::EMOJI_MODIFIER);
                }
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, '\u{E007F}') => {
                    return (0, WidthInfo::TAG_END_ZWJ_EMOJI_PRESENTATION);
                }
                (WidthInfo::TAG_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A1_END_ZWJ_EMOJI_PRESENTATION);
                }
                (WidthInfo::TAG_A1_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A2_END_ZWJ_EMOJI_PRESENTATION)
                }
                (WidthInfo::TAG_A2_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION)
                }
                (WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION)
                }
                (WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A5_END_ZWJ_EMOJI_PRESENTATION)
                }
                (WidthInfo::TAG_A5_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A6_END_ZWJ_EMOJI_PRESENTATION)
                }
                (
                    WidthInfo::TAG_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A1_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A2_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION,
                    '\u{E0030}'..='\u{E0039}',
                ) => return (0, WidthInfo::TAG_D1_END_ZWJ_EMOJI_PRESENTATION),
                (WidthInfo::TAG_D1_END_ZWJ_EMOJI_PRESENTATION, '\u{E0030}'..='\u{E0039}') => {
                    return (0, WidthInfo::TAG_D2_END_ZWJ_EMOJI_PRESENTATION);
                }
                (WidthInfo::TAG_D2_END_ZWJ_EMOJI_PRESENTATION, '\u{E0030}'..='\u{E0039}') => {
                    return (0, WidthInfo::TAG_D3_END_ZWJ_EMOJI_PRESENTATION);
                }
                (
                    WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A5_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A6_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_D3_END_ZWJ_EMOJI_PRESENTATION,
                    '\u{1F3F4}',
                ) => return (0, WidthInfo::EMOJI_PRESENTATION),
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, _)
                    if lookup_width(c).1 == WidthInfo::EMOJI_PRESENTATION =>
                {
                    return (0, WidthInfo::EMOJI_PRESENTATION)
                }

                // Fallback
                _ => {}
            }
        }

        let ret = lookup_width(c);
        (ret.0 as i8, ret.1)
    }
}

#[inline]
pub fn str_width(s: &str) -> usize {
    s.chars()
        .rfold(
            (0, WidthInfo::DEFAULT),
            |(sum, next_info), c| -> (usize, WidthInfo) {
                let (add, info) = width_in_str(c, next_info);
                (sum.wrapping_add_signed(isize::from(add)), info)
            },
        )
        .0
}

/// Returns the [UAX #11](https://www.unicode.org/reports/tr11/) based width of `c` by
/// consulting a multi-level lookup table.
///
/// # Maintenance
/// The tables themselves are autogenerated but this function is hardcoded. You should have
/// nothing to worry about if you re-run `unicode.py` (for example, when updating Unicode.)
/// However, if you change the *actual structure* of the lookup tables (perhaps by editing the
/// `make_tables` function in `unicode.py`) you must ensure that this code reflects those changes.
#[cfg(feature = "cjk")]
#[inline]
fn lookup_width_cjk(c: char) -> (u8, WidthInfo) {
    let cp = c as usize;

    let t1_offset = WIDTH_ROOT_CJK.0[cp >> 13];

    // Each sub-table in WIDTH_MIDDLE is 7 bits, and each stored entry is a byte,
    // so each sub-table is 128 bytes in size.
    // (Sub-tables are selected using the computed offset from the previous table.)
    let t2_offset = WIDTH_MIDDLE.0[usize::from(t1_offset)][cp >> 7 & 0x3F];

    // Each sub-table in WIDTH_LEAVES is 6 bits, but each stored entry is 2 bits.
    // This is accomplished by packing four stored entries into one byte.
    // So each sub-table is 2**(7-2) == 32 bytes in size.
    // Since this is the last table, each entry represents an encoded width.
    let packed_widths = WIDTH_LEAVES.0[usize::from(t2_offset)][cp >> 2 & 0x1F];

    // Extract the packed width
    let width = packed_widths >> (2 * (cp & 0b11)) & 0b11;

    if width < 3 {
        (width, WidthInfo::DEFAULT)
    } else {
        match c {
            '\u{A}' => (1, WidthInfo::LINE_FEED),
            '\u{338}' => (0, WidthInfo::COMBINING_LONG_SOLIDUS_OVERLAY),
            '\u{5DC}' => (1, WidthInfo::HEBREW_LETTER_LAMED),
            '\u{622}'..='\u{882}' => (1, WidthInfo::JOINING_GROUP_ALEF),
            '\u{1780}'..='\u{17AF}' => (1, WidthInfo::KHMER_COENG_ELIGIBLE_LETTER),
            '\u{17D8}' => (3, WidthInfo::DEFAULT),
            '\u{1A10}' => (1, WidthInfo::BUGINESE_LETTER_YA),
            '\u{2D31}'..='\u{2D6F}' => (1, WidthInfo::TIFINAGH_CONSONANT),
            '\u{A4FC}'..='\u{A4FD}' => (1, WidthInfo::LISU_TONE_LETTER_MYA_NA_JEU),
            '\u{FE0F}' => (0, WidthInfo::VARIATION_SELECTOR_16),
            '\u{10C03}' => (1, WidthInfo::OLD_TURKIC_LETTER_ORKHON_I),
            '\u{1F1E6}'..='\u{1F1FF}' => (1, WidthInfo::REGIONAL_INDICATOR),
            '\u{1F3FB}'..='\u{1F3FF}' => (2, WidthInfo::EMOJI_MODIFIER),
            _ => (2, WidthInfo::EMOJI_PRESENTATION),
        }
    }
}

/// Returns the [UAX #11](https://www.unicode.org/reports/tr11/) based width of `c`, or
/// `None` if `c` is a control character.
/// Ambiguous width characters are treated as wide.
#[cfg(feature = "cjk")]
#[inline]
pub fn single_char_width_cjk(c: char) -> Option<usize> {
    if c < '\u{7F}' {
        if c >= '\u{20}' {
            // U+0020 to U+007F (exclusive) are single-width ASCII codepoints
            Some(1)
        } else {
            // U+0000 to U+0020 (exclusive) are control codes
            None
        }
    } else if c >= '\u{A0}' {
        // No characters >= U+00A0 are control codes, so we can consult the lookup tables
        Some(lookup_width_cjk(c).0.into())
    } else {
        // U+007F to U+00A0 (exclusive) are control codes
        None
    }
}

/// Returns the [UAX #11](https://www.unicode.org/reports/tr11/) based width of `c`.
/// Ambiguous width characters are treated as wide.
#[cfg(feature = "cjk")]
#[inline]
fn width_in_str_cjk(c: char, mut next_info: WidthInfo) -> (i8, WidthInfo) {
    if next_info.is_emoji_presentation() {
        if starts_emoji_presentation_seq(c) {
            let width = if next_info.is_zwj_emoji_presentation() {
                0
            } else {
                2
            };
            return (width, WidthInfo::EMOJI_PRESENTATION);
        } else {
            next_info = next_info.unset_emoji_presentation();
        }
    }
    if (matches!(
        next_info,
        WidthInfo::COMBINING_LONG_SOLIDUS_OVERLAY | WidthInfo::SOLIDUS_OVERLAY_ALEF
    ) && matches!(c, '<' | '=' | '>'))
    {
        return (2, WidthInfo::DEFAULT);
    }
    if c <= '\u{A0}' {
        match c {
            // According to the spec, LF should be width 1, which is how it is often rendered when it is forced to have a single-line rendering
            // However, this makes it harder to use this crate to calculate line breaks, and breaks assumptions of downstream crates.
            // https://github.com/unicode-rs/unicode-width/issues/60
            '\n' => (0, WidthInfo::LINE_FEED),
            '\r' if next_info == WidthInfo::LINE_FEED => (0, WidthInfo::DEFAULT),
            _ => (1, WidthInfo::DEFAULT),
        }
    } else {
        // Fast path
        if next_info != WidthInfo::DEFAULT {
            if c == '\u{FE0F}' {
                return (0, next_info.set_emoji_presentation());
            }
            if next_info.is_ligature_transparent() {
                if c == '\u{200D}' {
                    return (0, next_info.set_zwj_bit());
                } else if is_ligature_transparent(c) {
                    return (0, next_info);
                }
            }

            match (next_info, c) {
                (WidthInfo::COMBINING_LONG_SOLIDUS_OVERLAY, _) if is_solidus_transparent(c) => {
                    return (
                        lookup_width_cjk(c).0 as i8,
                        WidthInfo::COMBINING_LONG_SOLIDUS_OVERLAY,
                    );
                }
                (WidthInfo::JOINING_GROUP_ALEF, '\u{0338}') => {
                    return (0, WidthInfo::SOLIDUS_OVERLAY_ALEF);
                }
                // Arabic Lam-Alef ligature
                (
                    WidthInfo::JOINING_GROUP_ALEF | WidthInfo::SOLIDUS_OVERLAY_ALEF,
                    '\u{644}' | '\u{6B5}'..='\u{6B8}' | '\u{76A}' | '\u{8A6}' | '\u{8C7}',
                ) => return (0, WidthInfo::DEFAULT),
                (WidthInfo::JOINING_GROUP_ALEF, _) if is_transparent_zero_width(c) => {
                    return (0, WidthInfo::JOINING_GROUP_ALEF);
                }

                // Hebrew Alef-ZWJ-Lamed ligature
                (WidthInfo::ZWJ_HEBREW_LETTER_LAMED, '\u{05D0}') => {
                    return (0, WidthInfo::DEFAULT);
                }

                // Khmer coeng signs
                (WidthInfo::KHMER_COENG_ELIGIBLE_LETTER, '\u{17D2}') => {
                    return (-1, WidthInfo::DEFAULT);
                }

                // Buginese <a, -i> ZWJ ya ligature
                (WidthInfo::ZWJ_BUGINESE_LETTER_YA, '\u{1A17}') => {
                    return (0, WidthInfo::BUGINESE_VOWEL_SIGN_I_ZWJ_LETTER_YA)
                }
                (WidthInfo::BUGINESE_VOWEL_SIGN_I_ZWJ_LETTER_YA, '\u{1A15}') => {
                    return (0, WidthInfo::DEFAULT)
                }

                // Tifinagh bi-consonants
                (WidthInfo::TIFINAGH_CONSONANT | WidthInfo::ZWJ_TIFINAGH_CONSONANT, '\u{2D7F}') => {
                    return (1, WidthInfo::TIFINAGH_JOINER_CONSONANT);
                }
                (WidthInfo::ZWJ_TIFINAGH_CONSONANT, '\u{2D31}'..='\u{2D65}' | '\u{2D6F}') => {
                    return (0, WidthInfo::DEFAULT);
                }
                (WidthInfo::TIFINAGH_JOINER_CONSONANT, '\u{2D31}'..='\u{2D65}' | '\u{2D6F}') => {
                    return (-1, WidthInfo::DEFAULT);
                }

                // Lisu tone letter combinations
                (WidthInfo::LISU_TONE_LETTER_MYA_NA_JEU, '\u{A4F8}'..='\u{A4FB}') => {
                    return (0, WidthInfo::DEFAULT);
                }

                // Old Turkic ligature
                (WidthInfo::ZWJ_OLD_TURKIC_LETTER_ORKHON_I, '\u{10C32}') => {
                    return (0, WidthInfo::DEFAULT);
                }
                // Emoji modifier
                (WidthInfo::EMOJI_MODIFIER, _) if is_emoji_modifier_base(c) => {
                    return (0, WidthInfo::EMOJI_PRESENTATION);
                }

                // Regional indicator
                (
                    WidthInfo::REGIONAL_INDICATOR | WidthInfo::SEVERAL_REGIONAL_INDICATOR,
                    '\u{1F1E6}'..='\u{1F1FF}',
                ) => return (1, WidthInfo::SEVERAL_REGIONAL_INDICATOR),

                // ZWJ emoji
                (
                    WidthInfo::EMOJI_PRESENTATION
                    | WidthInfo::SEVERAL_REGIONAL_INDICATOR
                    | WidthInfo::EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION
                    | WidthInfo::ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION
                    | WidthInfo::EMOJI_MODIFIER,
                    '\u{200D}',
                ) => return (0, WidthInfo::ZWJ_EMOJI_PRESENTATION),
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, '\u{20E3}') => {
                    return (0, WidthInfo::KEYCAP_ZWJ_EMOJI_PRESENTATION);
                }
                (WidthInfo::VS16_ZWJ_EMOJI_PRESENTATION, _) if starts_emoji_presentation_seq(c) => {
                    return (0, WidthInfo::EMOJI_PRESENTATION)
                }
                (WidthInfo::VS16_KEYCAP_ZWJ_EMOJI_PRESENTATION, '0'..='9' | '#' | '*') => {
                    return (0, WidthInfo::EMOJI_PRESENTATION)
                }
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, '\u{1F1E6}'..='\u{1F1FF}') => {
                    return (1, WidthInfo::REGIONAL_INDICATOR_ZWJ_PRESENTATION);
                }
                (
                    WidthInfo::REGIONAL_INDICATOR_ZWJ_PRESENTATION
                    | WidthInfo::ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION,
                    '\u{1F1E6}'..='\u{1F1FF}',
                ) => return (-1, WidthInfo::EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION),
                (
                    WidthInfo::EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION,
                    '\u{1F1E6}'..='\u{1F1FF}',
                ) => return (3, WidthInfo::ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION),
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, '\u{1F3FB}'..='\u{1F3FF}') => {
                    return (0, WidthInfo::EMOJI_MODIFIER);
                }
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, '\u{E007F}') => {
                    return (0, WidthInfo::TAG_END_ZWJ_EMOJI_PRESENTATION);
                }
                (WidthInfo::TAG_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A1_END_ZWJ_EMOJI_PRESENTATION);
                }
                (WidthInfo::TAG_A1_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A2_END_ZWJ_EMOJI_PRESENTATION)
                }
                (WidthInfo::TAG_A2_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION)
                }
                (WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION)
                }
                (WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A5_END_ZWJ_EMOJI_PRESENTATION)
                }
                (WidthInfo::TAG_A5_END_ZWJ_EMOJI_PRESENTATION, '\u{E0061}'..='\u{E007A}') => {
                    return (0, WidthInfo::TAG_A6_END_ZWJ_EMOJI_PRESENTATION)
                }
                (
                    WidthInfo::TAG_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A1_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A2_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION,
                    '\u{E0030}'..='\u{E0039}',
                ) => return (0, WidthInfo::TAG_D1_END_ZWJ_EMOJI_PRESENTATION),
                (WidthInfo::TAG_D1_END_ZWJ_EMOJI_PRESENTATION, '\u{E0030}'..='\u{E0039}') => {
                    return (0, WidthInfo::TAG_D2_END_ZWJ_EMOJI_PRESENTATION);
                }
                (WidthInfo::TAG_D2_END_ZWJ_EMOJI_PRESENTATION, '\u{E0030}'..='\u{E0039}') => {
                    return (0, WidthInfo::TAG_D3_END_ZWJ_EMOJI_PRESENTATION);
                }
                (
                    WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A5_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_A6_END_ZWJ_EMOJI_PRESENTATION
                    | WidthInfo::TAG_D3_END_ZWJ_EMOJI_PRESENTATION,
                    '\u{1F3F4}',
                ) => return (0, WidthInfo::EMOJI_PRESENTATION),
                (WidthInfo::ZWJ_EMOJI_PRESENTATION, _)
                    if lookup_width_cjk(c).1 == WidthInfo::EMOJI_PRESENTATION =>
                {
                    return (0, WidthInfo::EMOJI_PRESENTATION)
                }

                // Fallback
                _ => {}
            }
        }

        let ret = lookup_width_cjk(c);
        (ret.0 as i8, ret.1)
    }
}

#[cfg(feature = "cjk")]
#[inline]
pub fn str_width_cjk(s: &str) -> usize {
    s.chars()
        .rfold(
            (0, WidthInfo::DEFAULT),
            |(sum, next_info), c| -> (usize, WidthInfo) {
                let (add, info) = width_in_str_cjk(c, next_info);
                (sum.wrapping_add_signed(isize::from(add)), info)
            },
        )
        .0
}

/// Whether this character is a zero-width character with
/// `Joining_Type=Transparent`. Used by the Alef-Lamed ligatures.
/// See also [`is_ligature_transparent`], a near-subset of this (only ZWJ is excepted)
/// which is transparent for non-Arabic ligatures.
fn is_transparent_zero_width(c: char) -> bool {
    if lookup_width(c).0 != 0 {
        // Not zero-width
        false
    } else {
        let cp: u32 = c.into();
        NON_TRANSPARENT_ZERO_WIDTHS
            .binary_search_by(|&(lo, hi)| {
                let lo = u32::from_le_bytes([lo[0], lo[1], lo[2], 0]);
                let hi = u32::from_le_bytes([hi[0], hi[1], hi[2], 0]);
                if cp < lo {
                    Ordering::Greater
                } else if cp > hi {
                    Ordering::Less
                } else {
                    Ordering::Equal
                }
            })
            .is_err()
    }
}

/// Whether this character is a default-ignorable combining mark
/// or ZWJ. These characters won't interrupt non-Arabic ligatures.
fn is_ligature_transparent(c: char) -> bool {
    matches!(c, '\u{34F}' | '\u{17B4}'..='\u{17B5}' | '\u{180B}'..='\u{180D}' | '\u{180F}' | '\u{200D}' | '\u{FE00}'..='\u{FE0F}' | '\u{E0100}'..='\u{E01EF}')
}

/// Whether this character is transparent wrt the effect of
/// U+0338 COMBINING LONG SOLIDUS OVERLAY
/// on its base character.
#[cfg(feature = "cjk")]
fn is_solidus_transparent(c: char) -> bool {
    let cp: u32 = c.into();
    is_ligature_transparent(c)
        || SOLIDUS_TRANSPARENT
            .binary_search_by(|&(lo, hi)| {
                let lo = u32::from_le_bytes([lo[0], lo[1], lo[2], 0]);
                let hi = u32::from_le_bytes([hi[0], hi[1], hi[2], 0]);
                if cp < lo {
                    Ordering::Greater
                } else if cp > hi {
                    Ordering::Less
                } else {
                    Ordering::Equal
                }
            })
            .is_ok()
}

/// Whether this character forms an [emoji presentation sequence]
/// (https://www.unicode.org/reports/tr51/#def_emoji_presentation_sequence)
/// when followed by `'\u{FEOF}'`.
/// Emoji presentation sequences are considered to have width 2.
#[inline]
pub fn starts_emoji_presentation_seq(c: char) -> bool {
    let cp: u32 = c.into();
    // First level of lookup uses all but 10 LSB
    let top_bits = cp >> 10;
    let idx_of_leaf: usize = match top_bits {
        0x0 => 0,
        0x8 => 1,
        0x9 => 2,
        0xA => 3,
        0xC => 4,
        0x7C => 5,
        0x7D => 6,
        _ => return false,
    };
    // Extract the 3-9th (0-indexed) least significant bits of `cp`,
    // and use them to index into `leaf_row`.
    let idx_within_leaf = usize::try_from((cp >> 3) & 0x7F).unwrap();
    let leaf_byte = EMOJI_PRESENTATION_LEAVES.0[idx_of_leaf][idx_within_leaf];
    // Use the 3 LSB of `cp` to index into `leaf_byte`.
    ((leaf_byte >> (cp & 7)) & 1) == 1
}

/// Returns `true` if `c` has default emoji presentation, but forms a [text presentation sequence]
/// (https://www.unicode.org/reports/tr51/#def_text_presentation_sequence)
/// when followed by `'\u{FEOE}'`, and is not ideographic.
/// Such sequences are considered to have width 1.
#[inline]
pub fn starts_non_ideographic_text_presentation_seq(c: char) -> bool {
    let cp: u32 = c.into();
    // First level of lookup uses all but 8 LSB
    let top_bits = cp >> 8;
    let leaf: &[(u8, u8)] = match top_bits {
        0x23 => &TEXT_PRESENTATION_LEAF_0,
        0x25 => &TEXT_PRESENTATION_LEAF_1,
        0x26 => &TEXT_PRESENTATION_LEAF_2,
        0x27 => &TEXT_PRESENTATION_LEAF_3,
        0x2B => &TEXT_PRESENTATION_LEAF_4,
        0x1F0 => &TEXT_PRESENTATION_LEAF_5,
        0x1F3 => &TEXT_PRESENTATION_LEAF_6,
        0x1F4 => &TEXT_PRESENTATION_LEAF_7,
        0x1F5 => &TEXT_PRESENTATION_LEAF_8,
        0x1F6 => &TEXT_PRESENTATION_LEAF_9,
        _ => return false,
    };

    let bottom_bits = (cp & 0xFF) as u8;
    leaf.binary_search_by(|&(lo, hi)| {
        if bottom_bits < lo {
            Ordering::Greater
        } else if bottom_bits > hi {
            Ordering::Less
        } else {
            Ordering::Equal
        }
    })
    .is_ok()
}

/// Returns `true` if `c` is an `Emoji_Modifier_Base`.
#[inline]
pub fn is_emoji_modifier_base(c: char) -> bool {
    let cp: u32 = c.into();
    // First level of lookup uses all but 8 LSB
    let top_bits = cp >> 8;
    let leaf: &[(u8, u8)] = match top_bits {
        0x26 => &EMOJI_MODIFIER_LEAF_0,
        0x27 => &EMOJI_MODIFIER_LEAF_1,
        0x1F3 => &EMOJI_MODIFIER_LEAF_2,
        0x1F4 => &EMOJI_MODIFIER_LEAF_3,
        0x1F5 => &EMOJI_MODIFIER_LEAF_4,
        0x1F6 => &EMOJI_MODIFIER_LEAF_5,
        0x1F9 => &EMOJI_MODIFIER_LEAF_6,
        0x1FA => &EMOJI_MODIFIER_LEAF_7,
        _ => return false,
    };

    let bottom_bits = (cp & 0xFF) as u8;
    leaf.binary_search_by(|&(lo, hi)| {
        if bottom_bits < lo {
            Ordering::Greater
        } else if bottom_bits > hi {
            Ordering::Less
        } else {
            Ordering::Equal
        }
    })
    .is_ok()
}

#[repr(align(32))]
struct Align32<T>(T);

#[repr(align(64))]
struct Align64<T>(T);

#[repr(align(128))]
struct Align128<T>(T);
/// Autogenerated. 1 sub-table(s). Consult [`lookup_width`] for layout info.)
static WIDTH_ROOT: Align128<[u8; 256]> = Align128([
    0x00, 0x01, 0x02, 0x03, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x0F, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x0F,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x10, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]);
/// Autogenerated. 17 sub-table(s). Consult [`lookup_width`] for layout info.)
#[cfg(feature = "cjk")]
static WIDTH_ROOT_CJK: Align128<[u8; 256]> = Align128([
    0x11, 0x12, 0x02, 0x03, 0x03, 0x04, 0x05, 0x13, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x14,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x0F, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x0F,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x10, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x0F,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]);

#[cfg(feature = "cjk")]
const WIDTH_MIDDLE_LEN: usize = 21;
#[cfg(not(feature = "cjk"))]
const WIDTH_MIDDLE_LEN: usize = 17;
/// Autogenerated. 4 sub-table(s). Consult [`lookup_width`] for layout info.
static WIDTH_MIDDLE: Align64<[[u8; 64]; WIDTH_MIDDLE_LEN]> = Align64([
    [
        0x00, 0x01, 0x02, 0x02, 0x02, 0x02, 0x03, 0x02, 0x02, 0x04, 0x02, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x02, 0x02, 0x1E, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x02, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x02, 0x2A,
        0x02, 0x02, 0x02, 0x02,
    ],
    [
        0x2B, 0x2C, 0x02, 0x02, 0x02, 0x02, 0x2D, 0x2E, 0x02, 0x02, 0x02, 0x2F, 0x30, 0x31, 0x32,
        0x33, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x34, 0x02, 0x02, 0x35, 0x36, 0x37, 0x02, 0x38,
        0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39,
    ],
    [
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x40, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39,
    ],
    [
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39,
    ],
    [
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x41, 0x02, 0x02, 0x42, 0x43, 0x02,
        0x02, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x02, 0x4A, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39,
    ],
    [
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x4B, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
    [
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x39, 0x39, 0x39, 0x39, 0x4C, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x4D, 0x4E, 0x4F, 0x50,
    ],
    [
        0x02, 0x02, 0x02, 0x51, 0x02, 0x52, 0x53, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x54, 0x55, 0x02, 0x02, 0x56, 0x02, 0x57, 0x02, 0x02, 0x58,
        0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 0x60, 0x61, 0x02, 0x62, 0x63, 0x02, 0x64, 0x65,
        0x66, 0x67, 0x02, 0x68, 0x02, 0x69, 0x6A, 0x6B, 0x6C, 0x02, 0x02, 0x6D, 0x6E, 0x6F, 0x70,
        0x02, 0x71, 0x72, 0x02,
    ],
    [
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x73, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
    [
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
    [
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x74, 0x75, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x76, 0x77, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39,
    ],
    [
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x78, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x79, 0x7A, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
    [
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x7B, 0x39, 0x39, 0x7C, 0x39, 0x39, 0x7D, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x7E, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
    [
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x7F, 0x02, 0x02, 0x02, 0x80, 0x81, 0x82, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x83, 0x84, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
    [
        0x85, 0x86, 0x75, 0x02, 0x02, 0x87, 0x02, 0x02, 0x02, 0x88, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x89, 0x8A, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x8B, 0x8C, 0x02, 0x8D, 0x8E, 0x02, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95,
        0x96, 0x02, 0x97, 0x02, 0x02, 0x98, 0x99, 0x9A, 0x9B, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
    [
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x9C,
    ],
    [
        0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D,
        0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D,
        0x1D, 0x1D, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
    #[cfg(feature = "cjk")]
    [
        0x00, 0x9D, 0x02, 0x02, 0x02, 0x02, 0x9E, 0x9F, 0x02, 0x04, 0x02, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x02, 0x02, 0x1E, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x02, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x02, 0x2A,
        0x02, 0x02, 0x02, 0x02,
    ],
    #[cfg(feature = "cjk")]
    [
        0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0x2E, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD,
        0x33, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0xAE, 0x02, 0x02, 0x35, 0x36, 0x37, 0x02, 0x38,
        0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0xAF, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39,
    ],
    #[cfg(feature = "cjk")]
    [
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
        0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x4C, 0x02, 0x02, 0x02, 0x02, 0x02,
        0xB0, 0x4E, 0x4F, 0xB1,
    ],
    #[cfg(feature = "cjk")]
    [
        0x85, 0x86, 0x75, 0x02, 0x02, 0x87, 0x02, 0x02, 0x02, 0x88, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x89, 0x8A, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x8B, 0x8C, 0xB2, 0xB3, 0x8E, 0x02, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95,
        0x96, 0x02, 0x97, 0x02, 0x02, 0x98, 0x99, 0x9A, 0x9B, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
        0x02, 0x02, 0x02, 0x02,
    ],
]);

#[cfg(feature = "cjk")]
const WIDTH_LEAVES_LEN: usize = 180;
#[cfg(not(feature = "cjk"))]
const WIDTH_LEAVES_LEN: usize = 157;
/// Autogenerated. 180 sub-table(s). Consult [`lookup_width`] for layout info.
static WIDTH_LEAVES: Align32<[[u8; 32]; WIDTH_LEAVES_LEN]> = Align32([
    [
        0x55, 0x55, 0x75, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x51, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x15, 0x00, 0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x10, 0x41, 0x10, 0x55, 0x55, 0x55, 0x55, 0x55, 0x57, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x51, 0x55, 0x55, 0x00, 0x00, 0x40, 0x54, 0xF5, 0xDD, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x55, 0x55, 0x55, 0xFC, 0x5D,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x00, 0x14, 0x00, 0x14, 0x04, 0x50, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x15, 0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xD5, 0x57,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x00, 0x00, 0x54, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x00, 0x55,
        0x55, 0x51,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x10, 0x00, 0x00, 0x01, 0x01, 0x50, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01, 0x55, 0x55, 0x55, 0x55, 0x55, 0xFF, 0xFF,
        0xFF, 0xFF,
    ],
    [
        0x7F, 0x55, 0x55, 0x55, 0x50, 0x55, 0x00, 0x00, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00,
    ],
    [
        0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x45,
        0x54, 0x01, 0x00, 0x54, 0x51, 0x01, 0x00, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x44, 0x01, 0x54, 0x55, 0x51, 0x55, 0x15, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x45,
    ],
    [
        0x41, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x54, 0x41, 0x15, 0x14, 0x50, 0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x50, 0x51,
        0x55, 0x55,
    ],
    [
        0x41, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x54, 0x01, 0x10, 0x54, 0x51, 0x55, 0x55, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x05, 0x00,
    ],
    [
        0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x04, 0x01, 0x54, 0x55, 0x51, 0x55, 0x01, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x45, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x45, 0x54, 0x55, 0x55, 0x51, 0x55, 0x15, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x54, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x04, 0x54, 0x05, 0x04, 0x50, 0x55, 0x41, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x14, 0x44, 0x05, 0x04, 0x50, 0x55, 0x41, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15,
        0x44, 0x01, 0x54, 0x55, 0x41, 0x55, 0x15, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x45, 0x15, 0x05, 0x44, 0x55, 0x15, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x51, 0x00, 0x40,
        0x55, 0x55, 0x15, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x51, 0x00, 0x00,
        0x54, 0x55, 0x55, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x11, 0x51,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01, 0x00,
        0x00, 0x40,
    ],
    [
        0x00, 0x04, 0x55, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x54, 0x55, 0x45, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01, 0x04, 0x00, 0x41,
        0x41, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x50, 0x05, 0x54, 0x55, 0x55, 0x55, 0x01, 0x54,
        0x55, 0x55,
    ],
    [
        0x45, 0x41, 0x55, 0x51, 0x55, 0x55, 0x55, 0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00,
    ],
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x05, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x55,
        0x55, 0x55,
    ],
    [
        0x7F, 0xFF, 0xFD, 0xF7, 0xFF, 0xFD, 0xD7, 0x5F, 0x77, 0xD6, 0xD5, 0xD7, 0x55, 0x10, 0x00,
        0x50, 0x55, 0x45, 0x01, 0x00, 0x00, 0x55, 0x57, 0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x15, 0x00, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x41, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x51, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x40, 0x15, 0x54, 0x55, 0x45, 0x55, 0x01,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x57, 0x15, 0x14, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x45, 0x00, 0x40, 0x44, 0x01, 0x00, 0x54, 0x15, 0x00,
        0x00, 0x14,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x00, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00,
        0x50, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x00, 0x55,
        0x55, 0x55,
    ],
    [
        0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x50, 0x10, 0x50, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x45, 0x50, 0x11, 0x50, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x05, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x40, 0x00, 0x00, 0x00, 0x04, 0x00, 0x54, 0x51, 0x55, 0x54,
        0x50, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00,
    ],
    [
        0x55, 0x55, 0x15, 0x00, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x40, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xF5, 0x55, 0x55, 0x55, 0x69, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xFD, 0x57, 0xD7, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x7D,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x5F, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0xFF, 0xFF, 0xFF, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0xD5,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0xD5, 0x55, 0x55, 0x55, 0x5D, 0x55, 0xF5, 0x55, 0x55, 0x55, 0x55,
        0x7D, 0x55, 0x5F, 0x55, 0x75, 0x55, 0x57, 0x55, 0x55, 0x55, 0x55, 0x75, 0x55, 0xF5, 0x5D,
        0x75, 0x5D,
    ],
    [
        0x55, 0x5D, 0xF5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x57, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x77, 0xD5, 0xDF, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0xFD, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x57, 0x55, 0x55,
        0xD5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xD5, 0x57, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x57, 0x5D, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x50, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xFD, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x5F, 0x55, 0xD5, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x9A, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x55,
        0x55, 0x55,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x5A, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x0A, 0x00, 0xAA, 0xAA, 0xAA,
        0x6A, 0xA9, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x6A, 0x81, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    [
        0x55, 0xA9, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xA9, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xA8, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    [
        0xAA, 0xAA, 0xAA, 0x6A, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x95, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x6A, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0x55, 0x55, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0xAA, 0xAA, 0xAA, 0x56, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0x6A, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x5F,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x40, 0x00,
        0x00, 0x50,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x50, 0x55,
        0x55, 0x55,
    ],
    [
        0x45, 0x45, 0x15, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x41, 0x55, 0x54, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x50, 0x55,
        0x45, 0x15,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x00, 0x50, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x15, 0x00, 0x00, 0x50, 0x55, 0x55, 0x55, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0x56,
    ],
    [
        0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x05, 0x50,
        0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x51, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01, 0x40, 0x41, 0x41, 0x55,
        0x55, 0x15, 0x55, 0x55, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x54,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x04, 0x14, 0x54,
        0x05, 0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x50, 0x55, 0x45,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x51, 0x54, 0x51, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x45, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x00, 0x00, 0x00, 0xF0, 0xAA, 0xAA, 0x5A, 0x55, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x6A, 0xAA, 0xAA, 0xAA, 0xAA, 0x6A, 0xAA, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x15,
    ],
    [
        0xA9, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x56, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xAA, 0x6A, 0x55, 0x55, 0x00, 0x00,
        0x54, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x51,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05,
        0x40, 0x55,
    ],
    [
        0x01, 0x41, 0x55, 0x00, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x40,
        0x15, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x41, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0xD5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x54, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x01,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x05, 0x00, 0x00, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x05, 0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00,
        0x00, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x14, 0x54,
        0x55, 0x15,
    ],
    [
        0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x40, 0x41,
        0x55, 0x45, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x01, 0x00, 0x54, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x55,
        0x55, 0x55,
    ],
    [
        0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x00,
        0x40, 0x05, 0x55, 0x01, 0x14, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x50, 0x04, 0x55,
        0x45, 0x51, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x15, 0x00, 0x40, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15,
        0x44, 0x54, 0x55, 0x55, 0x55, 0x55, 0x15, 0x55, 0x55, 0x55, 0x05, 0x00, 0x54, 0x00, 0x54,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00,
        0x00, 0x05, 0x44, 0x55, 0x55, 0x55, 0x55, 0x55, 0x45, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x14, 0x00, 0x44,
        0x11, 0x04, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x05, 0x50, 0x55,
        0x10, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x40,
        0x11, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x51, 0x00, 0x10, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01, 0x05, 0x10, 0x00, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x00, 0x41,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x54, 0x55, 0x15,
        0x04, 0x11, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x05, 0x55, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x01, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x04,
        0x40, 0x55, 0x15, 0x55, 0x55, 0x01, 0x40, 0x01, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x00, 0x00, 0x00, 0x00, 0x40, 0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x40, 0x00,
        0x10, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x04, 0x41, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01, 0x40, 0x45,
        0x10, 0x00, 0x00, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x50, 0x11, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x15, 0x54,
        0x55, 0x55,
    ],
    [
        0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x40,
        0x55, 0x44, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x54, 0x15, 0x00, 0x00, 0x00, 0x50, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x54,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x40, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x15, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x15, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xAA, 0x54, 0x55, 0x55, 0x5A, 0x55,
        0x55, 0x55,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0x55, 0x55,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x5A, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0xAA, 0xAA, 0x56, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xAA, 0xA9,
        0xAA, 0x69,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x6A, 0x55, 0x55, 0x55, 0x65, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x6A, 0x59, 0x55, 0x55, 0x55, 0xAA, 0x55, 0x55, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x41, 0x00, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x11, 0x50, 0x05, 0x00, 0x00,
        0x00, 0x00,
    ],
    [
        0x40, 0x01, 0x00, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x50, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x05, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x15,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x55, 0x51,
        0x55, 0x55,
    ],
    [
        0x55, 0x54, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x01, 0x00, 0x00, 0x00, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x10, 0x04, 0x40, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x15, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x45, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x00, 0x40, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x57, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0xD5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x75, 0xFD, 0xFF, 0x7F, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xF5, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF,
    ],
    [
        0x6E, 0x55, 0x55, 0x55, 0xAA, 0xAA, 0xBA, 0xAA, 0xAA, 0xAA, 0xAA, 0xEA, 0xFA, 0xBF, 0xBF,
        0x55, 0xAA, 0xAA, 0x56, 0x55, 0x5F, 0x55, 0x55, 0x55, 0xAA, 0x5A, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x57, 0x55, 0x55, 0xFD, 0xFF, 0xDF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xF7,
    ],
    [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x55, 0x55, 0x55, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0x7F, 0xD5, 0xFF, 0x55, 0x55, 0x55, 0xFF, 0xFF, 0xFF, 0xFF, 0x57, 0x57,
        0xFF, 0xFF,
    ],
    [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0x7F, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF,
    ],
    [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xD7,
    ],
    [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0x5F, 0x55, 0x55, 0xD5, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x55, 0x55, 0x55, 0x55,
        0x75, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x7D, 0x55, 0x55, 0x55, 0x57, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0xD5, 0xFF,
    ],
    [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0x5F, 0x55, 0x57, 0x7F, 0xFD, 0x55, 0xFF, 0x55, 0x55, 0xD5, 0x57, 0x55, 0xFF,
        0xFF, 0x57,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xFF, 0xFF, 0xFF, 0x55, 0x57, 0x55,
        0x55, 0x55,
    ],
    [
        0x55, 0x55, 0x55, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F,
        0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF,
    ],
    [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF,
    ],
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xFF, 0xFF,
        0xFF, 0x57,
    ],
    [
        0xFF, 0xFF, 0x57, 0x55, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xDF, 0xFF, 0x5F, 0x55, 0xF5, 0xFF, 0xFF, 0xFF, 0x55, 0xFF, 0xFF, 0x57, 0x55, 0xFF, 0xFF,
        0x57, 0x55,
    ],
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0x5A,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x59, 0x96, 0x55, 0x61, 0xAA, 0xA5, 0x59,
        0xAA, 0x55, 0x55, 0x55, 0x55, 0x55, 0x95, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x95,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x95, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x55, 0x15, 0x00, 0x96, 0x6A, 0x5A, 0x5A, 0x6A, 0xAA, 0x05, 0x40, 0xA6, 0x59, 0x95,
        0x65, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x55, 0x56,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0xA9, 0x56, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x56, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0x95, 0x59, 0x59, 0x55, 0x55, 0x65, 0x55, 0x55, 0x69, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x95, 0x56, 0x95, 0x6A, 0xAA, 0xAA, 0xAA, 0x55, 0xAA, 0xAA,
        0x5A, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x55, 0x59, 0x55, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55, 0x65, 0x55, 0x55, 0x5A,
        0x55, 0x55, 0x55, 0x55, 0xA5, 0x65, 0x56, 0x55, 0x55, 0x55, 0x95, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0xA6, 0x96, 0x9A, 0x96, 0x59, 0x59, 0x65, 0xA9, 0x96, 0xAA, 0xAA, 0x66, 0x55, 0xAA, 0x55,
        0x5A, 0x59, 0x55, 0x5A, 0x56, 0x65, 0x55, 0x55, 0x55, 0x6A, 0xAA, 0xA5, 0xA5, 0x5A, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0xA5, 0xAA, 0x5A, 0x55, 0x55, 0x59, 0x59, 0x55, 0x55, 0x59, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x95, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x55, 0x55, 0x55, 0x65, 0x55, 0xF5, 0x55, 0x55, 0x55, 0x69, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    #[cfg(feature = "cjk")]
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x9A, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    #[cfg(feature = "cjk")]
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xA5, 0x5A, 0x55, 0x55, 0x9A, 0xAA, 0x5A, 0x55, 0xA5, 0xA5, 0x55,
        0x5A, 0x5A, 0xA5, 0x96, 0xA5, 0x5A, 0x55, 0x55, 0x55, 0xA5, 0x5A, 0x55, 0x95, 0x55, 0x55,
        0x55, 0x7D,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x69, 0x59, 0xA5, 0x55, 0x5F, 0x55, 0x66, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x66, 0x55, 0xFF, 0xFF, 0xFF, 0x55, 0x55, 0x55, 0x9A, 0x9A, 0x6A, 0x9A, 0x55, 0x55,
        0x55, 0xD5,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x55, 0x55, 0x55, 0xD5, 0x55, 0x55, 0xA5, 0x5D, 0x55, 0xF5, 0x55, 0x55, 0x55, 0x55,
        0xBD, 0x55, 0xAF, 0xAA, 0xBA, 0xAA, 0xAB, 0xAA, 0xAA, 0x9A, 0x55, 0xBA, 0xAA, 0xFA, 0xAE,
        0xBA, 0xAE,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x5D, 0xF5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x57, 0x55, 0x55, 0x55, 0x55,
        0x59, 0x55, 0x55, 0x55, 0x77, 0xD5, 0xDF, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xA5,
        0xAA, 0xAA,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xD5, 0x57, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x57, 0xAD, 0x5A, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x6A, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    #[cfg(feature = "cjk")]
    [
        0x00, 0x00, 0x00, 0xC0, 0xAA, 0xAA, 0x5A, 0x55, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x6A, 0xAA, 0xAA, 0xAA, 0xAA, 0x6A, 0xAA, 0x55, 0x55, 0x55,
        0x55, 0x55,
    ],
    #[cfg(feature = "cjk")]
    [
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x05, 0x54, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xAA, 0x6A, 0x55, 0x55, 0x00, 0x00,
        0x54, 0x59,
    ],
    #[cfg(feature = "cjk")]
    [
        0xAA, 0xAA, 0x6A, 0x55, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x5A, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x5A, 0x55, 0xAA, 0xAA,
        0xAA, 0xAA,
    ],
    #[cfg(feature = "cjk")]
    [
        0xAA, 0xAA, 0xAA, 0xBA, 0xFE, 0xFF, 0xBF, 0xAA, 0xAA, 0xAA, 0xAA, 0x56, 0x55, 0x55, 0x55,
        0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xF5, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF,
    ],
]);

/// Sorted list of codepoint ranges (inclusive)
/// that are zero-width but not `Joining_Type=Transparent`
/// FIXME: can we get better compression?
static NON_TRANSPARENT_ZERO_WIDTHS: [([u8; 3], [u8; 3]); 53] = [
    ([0x05, 0x06, 0x00], [0x05, 0x06, 0x00]),
    ([0x90, 0x08, 0x00], [0x91, 0x08, 0x00]),
    ([0xE2, 0x08, 0x00], [0xE2, 0x08, 0x00]),
    ([0xBE, 0x09, 0x00], [0xBE, 0x09, 0x00]),
    ([0xD7, 0x09, 0x00], [0xD7, 0x09, 0x00]),
    ([0x3E, 0x0B, 0x00], [0x3E, 0x0B, 0x00]),
    ([0x57, 0x0B, 0x00], [0x57, 0x0B, 0x00]),
    ([0xBE, 0x0B, 0x00], [0xBE, 0x0B, 0x00]),
    ([0xD7, 0x0B, 0x00], [0xD7, 0x0B, 0x00]),
    ([0xC0, 0x0C, 0x00], [0xC0, 0x0C, 0x00]),
    ([0xC2, 0x0C, 0x00], [0xC2, 0x0C, 0x00]),
    ([0xC7, 0x0C, 0x00], [0xC8, 0x0C, 0x00]),
    ([0xCA, 0x0C, 0x00], [0xCB, 0x0C, 0x00]),
    ([0xD5, 0x0C, 0x00], [0xD6, 0x0C, 0x00]),
    ([0x3E, 0x0D, 0x00], [0x3E, 0x0D, 0x00]),
    ([0x4E, 0x0D, 0x00], [0x4E, 0x0D, 0x00]),
    ([0x57, 0x0D, 0x00], [0x57, 0x0D, 0x00]),
    ([0xCF, 0x0D, 0x00], [0xCF, 0x0D, 0x00]),
    ([0xDF, 0x0D, 0x00], [0xDF, 0x0D, 0x00]),
    ([0x60, 0x11, 0x00], [0xFF, 0x11, 0x00]),
    ([0x0E, 0x18, 0x00], [0x0E, 0x18, 0x00]),
    ([0x35, 0x1B, 0x00], [0x35, 0x1B, 0x00]),
    ([0x3B, 0x1B, 0x00], [0x3B, 0x1B, 0x00]),
    ([0x3D, 0x1B, 0x00], [0x3D, 0x1B, 0x00]),
    ([0x43, 0x1B, 0x00], [0x43, 0x1B, 0x00]),
    ([0x0C, 0x20, 0x00], [0x0D, 0x20, 0x00]),
    ([0x65, 0x20, 0x00], [0x69, 0x20, 0x00]),
    ([0x2E, 0x30, 0x00], [0x2F, 0x30, 0x00]),
    ([0x64, 0x31, 0x00], [0x64, 0x31, 0x00]),
    ([0xFA, 0xA8, 0x00], [0xFA, 0xA8, 0x00]),
    ([0xB0, 0xD7, 0x00], [0xC6, 0xD7, 0x00]),
    ([0xCB, 0xD7, 0x00], [0xFB, 0xD7, 0x00]),
    ([0x9E, 0xFF, 0x00], [0xA0, 0xFF, 0x00]),
    ([0xF0, 0xFF, 0x00], [0xF8, 0xFF, 0x00]),
    ([0xC2, 0x11, 0x01], [0xC3, 0x11, 0x01]),
    ([0x3E, 0x13, 0x01], [0x3E, 0x13, 0x01]),
    ([0x57, 0x13, 0x01], [0x57, 0x13, 0x01]),
    ([0xB0, 0x14, 0x01], [0xB0, 0x14, 0x01]),
    ([0xBD, 0x14, 0x01], [0xBD, 0x14, 0x01]),
    ([0xAF, 0x15, 0x01], [0xAF, 0x15, 0x01]),
    ([0x30, 0x19, 0x01], [0x30, 0x19, 0x01]),
    ([0x3F, 0x19, 0x01], [0x3F, 0x19, 0x01]),
    ([0x41, 0x19, 0x01], [0x41, 0x19, 0x01]),
    ([0x3A, 0x1A, 0x01], [0x3A, 0x1A, 0x01]),
    ([0x84, 0x1A, 0x01], [0x89, 0x1A, 0x01]),
    ([0x46, 0x1D, 0x01], [0x46, 0x1D, 0x01]),
    ([0x02, 0x1F, 0x01], [0x02, 0x1F, 0x01]),
    ([0x65, 0xD1, 0x01], [0x65, 0xD1, 0x01]),
    ([0x6E, 0xD1, 0x01], [0x72, 0xD1, 0x01]),
    ([0x00, 0x00, 0x0E], [0x00, 0x00, 0x0E]),
    ([0x02, 0x00, 0x0E], [0x1F, 0x00, 0x0E]),
    ([0x80, 0x00, 0x0E], [0xFF, 0x00, 0x0E]),
    ([0xF0, 0x01, 0x0E], [0xFF, 0x0F, 0x0E]),
];

/// Sorted list of codepoint ranges (inclusive)
/// that don't affect how the combining solidus applies
/// (mostly ccc > 1).
/// FIXME: can we get better compression?
#[cfg(feature = "cjk")]
static SOLIDUS_TRANSPARENT: [([u8; 3], [u8; 3]); 198] = [
    ([0x00, 0x03, 0x00], [0x33, 0x03, 0x00]),
    ([0x39, 0x03, 0x00], [0x6F, 0x03, 0x00]),
    ([0x83, 0x04, 0x00], [0x87, 0x04, 0x00]),
    ([0x91, 0x05, 0x00], [0xBD, 0x05, 0x00]),
    ([0xBF, 0x05, 0x00], [0xBF, 0x05, 0x00]),
    ([0xC1, 0x05, 0x00], [0xC2, 0x05, 0x00]),
    ([0xC4, 0x05, 0x00], [0xC5, 0x05, 0x00]),
    ([0xC7, 0x05, 0x00], [0xC7, 0x05, 0x00]),
    ([0x10, 0x06, 0x00], [0x1A, 0x06, 0x00]),
    ([0x4B, 0x06, 0x00], [0x5F, 0x06, 0x00]),
    ([0x70, 0x06, 0x00], [0x70, 0x06, 0x00]),
    ([0xD6, 0x06, 0x00], [0xDC, 0x06, 0x00]),
    ([0xDF, 0x06, 0x00], [0xE4, 0x06, 0x00]),
    ([0xE7, 0x06, 0x00], [0xE8, 0x06, 0x00]),
    ([0xEA, 0x06, 0x00], [0xED, 0x06, 0x00]),
    ([0x11, 0x07, 0x00], [0x11, 0x07, 0x00]),
    ([0x30, 0x07, 0x00], [0x4A, 0x07, 0x00]),
    ([0xEB, 0x07, 0x00], [0xF3, 0x07, 0x00]),
    ([0xFD, 0x07, 0x00], [0xFD, 0x07, 0x00]),
    ([0x16, 0x08, 0x00], [0x19, 0x08, 0x00]),
    ([0x1B, 0x08, 0x00], [0x23, 0x08, 0x00]),
    ([0x25, 0x08, 0x00], [0x27, 0x08, 0x00]),
    ([0x29, 0x08, 0x00], [0x2D, 0x08, 0x00]),
    ([0x59, 0x08, 0x00], [0x5B, 0x08, 0x00]),
    ([0x98, 0x08, 0x00], [0x9F, 0x08, 0x00]),
    ([0xCA, 0x08, 0x00], [0xE1, 0x08, 0x00]),
    ([0xE3, 0x08, 0x00], [0xFF, 0x08, 0x00]),
    ([0x3C, 0x09, 0x00], [0x3C, 0x09, 0x00]),
    ([0x4D, 0x09, 0x00], [0x4D, 0x09, 0x00]),
    ([0x51, 0x09, 0x00], [0x54, 0x09, 0x00]),
    ([0xBC, 0x09, 0x00], [0xBC, 0x09, 0x00]),
    ([0xCD, 0x09, 0x00], [0xCD, 0x09, 0x00]),
    ([0xFE, 0x09, 0x00], [0xFE, 0x09, 0x00]),
    ([0x3C, 0x0A, 0x00], [0x3C, 0x0A, 0x00]),
    ([0x4D, 0x0A, 0x00], [0x4D, 0x0A, 0x00]),
    ([0xBC, 0x0A, 0x00], [0xBC, 0x0A, 0x00]),
    ([0xCD, 0x0A, 0x00], [0xCD, 0x0A, 0x00]),
    ([0x3C, 0x0B, 0x00], [0x3C, 0x0B, 0x00]),
    ([0x4D, 0x0B, 0x00], [0x4D, 0x0B, 0x00]),
    ([0xCD, 0x0B, 0x00], [0xCD, 0x0B, 0x00]),
    ([0x3C, 0x0C, 0x00], [0x3C, 0x0C, 0x00]),
    ([0x4D, 0x0C, 0x00], [0x4D, 0x0C, 0x00]),
    ([0x55, 0x0C, 0x00], [0x56, 0x0C, 0x00]),
    ([0xBC, 0x0C, 0x00], [0xBC, 0x0C, 0x00]),
    ([0xCD, 0x0C, 0x00], [0xCD, 0x0C, 0x00]),
    ([0x3B, 0x0D, 0x00], [0x3C, 0x0D, 0x00]),
    ([0x4D, 0x0D, 0x00], [0x4D, 0x0D, 0x00]),
    ([0xCA, 0x0D, 0x00], [0xCA, 0x0D, 0x00]),
    ([0x38, 0x0E, 0x00], [0x3A, 0x0E, 0x00]),
    ([0x48, 0x0E, 0x00], [0x4B, 0x0E, 0x00]),
    ([0xB8, 0x0E, 0x00], [0xBA, 0x0E, 0x00]),
    ([0xC8, 0x0E, 0x00], [0xCB, 0x0E, 0x00]),
    ([0x18, 0x0F, 0x00], [0x19, 0x0F, 0x00]),
    ([0x35, 0x0F, 0x00], [0x35, 0x0F, 0x00]),
    ([0x37, 0x0F, 0x00], [0x37, 0x0F, 0x00]),
    ([0x39, 0x0F, 0x00], [0x39, 0x0F, 0x00]),
    ([0x71, 0x0F, 0x00], [0x75, 0x0F, 0x00]),
    ([0x7A, 0x0F, 0x00], [0x7D, 0x0F, 0x00]),
    ([0x80, 0x0F, 0x00], [0x84, 0x0F, 0x00]),
    ([0x86, 0x0F, 0x00], [0x87, 0x0F, 0x00]),
    ([0xC6, 0x0F, 0x00], [0xC6, 0x0F, 0x00]),
    ([0x37, 0x10, 0x00], [0x37, 0x10, 0x00]),
    ([0x39, 0x10, 0x00], [0x3A, 0x10, 0x00]),
    ([0x8D, 0x10, 0x00], [0x8D, 0x10, 0x00]),
    ([0x5D, 0x13, 0x00], [0x5F, 0x13, 0x00]),
    ([0x14, 0x17, 0x00], [0x15, 0x17, 0x00]),
    ([0x34, 0x17, 0x00], [0x34, 0x17, 0x00]),
    ([0xD2, 0x17, 0x00], [0xD2, 0x17, 0x00]),
    ([0xDD, 0x17, 0x00], [0xDD, 0x17, 0x00]),
    ([0xA9, 0x18, 0x00], [0xA9, 0x18, 0x00]),
    ([0x39, 0x19, 0x00], [0x3B, 0x19, 0x00]),
    ([0x17, 0x1A, 0x00], [0x18, 0x1A, 0x00]),
    ([0x60, 0x1A, 0x00], [0x60, 0x1A, 0x00]),
    ([0x75, 0x1A, 0x00], [0x7C, 0x1A, 0x00]),
    ([0x7F, 0x1A, 0x00], [0x7F, 0x1A, 0x00]),
    ([0xB0, 0x1A, 0x00], [0xBD, 0x1A, 0x00]),
    ([0xBF, 0x1A, 0x00], [0xCE, 0x1A, 0x00]),
    ([0x34, 0x1B, 0x00], [0x34, 0x1B, 0x00]),
    ([0x44, 0x1B, 0x00], [0x44, 0x1B, 0x00]),
    ([0x6B, 0x1B, 0x00], [0x73, 0x1B, 0x00]),
    ([0xAA, 0x1B, 0x00], [0xAB, 0x1B, 0x00]),
    ([0xE6, 0x1B, 0x00], [0xE6, 0x1B, 0x00]),
    ([0xF2, 0x1B, 0x00], [0xF3, 0x1B, 0x00]),
    ([0x37, 0x1C, 0x00], [0x37, 0x1C, 0x00]),
    ([0xD0, 0x1C, 0x00], [0xD2, 0x1C, 0x00]),
    ([0xD5, 0x1C, 0x00], [0xE0, 0x1C, 0x00]),
    ([0xED, 0x1C, 0x00], [0xED, 0x1C, 0x00]),
    ([0xF4, 0x1C, 0x00], [0xF4, 0x1C, 0x00]),
    ([0xF8, 0x1C, 0x00], [0xF9, 0x1C, 0x00]),
    ([0xC0, 0x1D, 0x00], [0xFF, 0x1D, 0x00]),
    ([0xD0, 0x20, 0x00], [0xD1, 0x20, 0x00]),
    ([0xD4, 0x20, 0x00], [0xD7, 0x20, 0x00]),
    ([0xDB, 0x20, 0x00], [0xDC, 0x20, 0x00]),
    ([0xE1, 0x20, 0x00], [0xE1, 0x20, 0x00]),
    ([0xE7, 0x20, 0x00], [0xE9, 0x20, 0x00]),
    ([0xEC, 0x20, 0x00], [0xF0, 0x20, 0x00]),
    ([0xEF, 0x2C, 0x00], [0xF1, 0x2C, 0x00]),
    ([0x7F, 0x2D, 0x00], [0x7F, 0x2D, 0x00]),
    ([0xE0, 0x2D, 0x00], [0xFF, 0x2D, 0x00]),
    ([0x2A, 0x30, 0x00], [0x2F, 0x30, 0x00]),
    ([0x99, 0x30, 0x00], [0x9A, 0x30, 0x00]),
    ([0x6F, 0xA6, 0x00], [0x6F, 0xA6, 0x00]),
    ([0x74, 0xA6, 0x00], [0x7D, 0xA6, 0x00]),
    ([0x9E, 0xA6, 0x00], [0x9F, 0xA6, 0x00]),
    ([0xF0, 0xA6, 0x00], [0xF1, 0xA6, 0x00]),
    ([0x06, 0xA8, 0x00], [0x06, 0xA8, 0x00]),
    ([0x2C, 0xA8, 0x00], [0x2C, 0xA8, 0x00]),
    ([0xC4, 0xA8, 0x00], [0xC4, 0xA8, 0x00]),
    ([0xE0, 0xA8, 0x00], [0xF1, 0xA8, 0x00]),
    ([0x2B, 0xA9, 0x00], [0x2D, 0xA9, 0x00]),
    ([0x53, 0xA9, 0x00], [0x53, 0xA9, 0x00]),
    ([0xB3, 0xA9, 0x00], [0xB3, 0xA9, 0x00]),
    ([0xC0, 0xA9, 0x00], [0xC0, 0xA9, 0x00]),
    ([0xB0, 0xAA, 0x00], [0xB0, 0xAA, 0x00]),
    ([0xB2, 0xAA, 0x00], [0xB4, 0xAA, 0x00]),
    ([0xB7, 0xAA, 0x00], [0xB8, 0xAA, 0x00]),
    ([0xBE, 0xAA, 0x00], [0xBF, 0xAA, 0x00]),
    ([0xC1, 0xAA, 0x00], [0xC1, 0xAA, 0x00]),
    ([0xF6, 0xAA, 0x00], [0xF6, 0xAA, 0x00]),
    ([0xED, 0xAB, 0x00], [0xED, 0xAB, 0x00]),
    ([0x1E, 0xFB, 0x00], [0x1E, 0xFB, 0x00]),
    ([0x20, 0xFE, 0x00], [0x2F, 0xFE, 0x00]),
    ([0x1B, 0x01, 0x01], [0x1B, 0x01, 0x01]),
    ([0x6B, 0x01, 0x01], [0x6B, 0x01, 0x01]),
    ([0xFD, 0x01, 0x01], [0xFD, 0x01, 0x01]),
    ([0xE0, 0x02, 0x01], [0xE0, 0x02, 0x01]),
    ([0xF5, 0x02, 0x01], [0xF5, 0x02, 0x01]),
    ([0x76, 0x03, 0x01], [0x7A, 0x03, 0x01]),
    ([0xD4, 0x09, 0x01], [0xD4, 0x09, 0x01]),
    ([0x0D, 0x0A, 0x01], [0x0D, 0x0A, 0x01]),
    ([0x0F, 0x0A, 0x01], [0x0F, 0x0A, 0x01]),
    ([0x38, 0x0A, 0x01], [0x38, 0x0A, 0x01]),
    ([0x3A, 0x0A, 0x01], [0x3A, 0x0A, 0x01]),
    ([0x3F, 0x0A, 0x01], [0x3F, 0x0A, 0x01]),
    ([0xE5, 0x0A, 0x01], [0xE6, 0x0A, 0x01]),
    ([0x24, 0x0D, 0x01], [0x27, 0x0D, 0x01]),
    ([0x74, 0x0E, 0x01], [0x74, 0x0E, 0x01]),
    ([0xAB, 0x0E, 0x01], [0xAC, 0x0E, 0x01]),
    ([0xFD, 0x0E, 0x01], [0xFF, 0x0E, 0x01]),
    ([0x46, 0x0F, 0x01], [0x50, 0x0F, 0x01]),
    ([0x82, 0x0F, 0x01], [0x85, 0x0F, 0x01]),
    ([0x46, 0x10, 0x01], [0x46, 0x10, 0x01]),
    ([0x70, 0x10, 0x01], [0x70, 0x10, 0x01]),
    ([0x7F, 0x10, 0x01], [0x7F, 0x10, 0x01]),
    ([0xB9, 0x10, 0x01], [0xBA, 0x10, 0x01]),
    ([0x00, 0x11, 0x01], [0x02, 0x11, 0x01]),
    ([0x33, 0x11, 0x01], [0x34, 0x11, 0x01]),
    ([0x73, 0x11, 0x01], [0x73, 0x11, 0x01]),
    ([0xC0, 0x11, 0x01], [0xC0, 0x11, 0x01]),
    ([0xCA, 0x11, 0x01], [0xCA, 0x11, 0x01]),
    ([0x35, 0x12, 0x01], [0x36, 0x12, 0x01]),
    ([0xE9, 0x12, 0x01], [0xEA, 0x12, 0x01]),
    ([0x3B, 0x13, 0x01], [0x3C, 0x13, 0x01]),
    ([0x4D, 0x13, 0x01], [0x4D, 0x13, 0x01]),
    ([0x66, 0x13, 0x01], [0x6C, 0x13, 0x01]),
    ([0x70, 0x13, 0x01], [0x74, 0x13, 0x01]),
    ([0x42, 0x14, 0x01], [0x42, 0x14, 0x01]),
    ([0x46, 0x14, 0x01], [0x46, 0x14, 0x01]),
    ([0x5E, 0x14, 0x01], [0x5E, 0x14, 0x01]),
    ([0xC2, 0x14, 0x01], [0xC3, 0x14, 0x01]),
    ([0xBF, 0x15, 0x01], [0xC0, 0x15, 0x01]),
    ([0x3F, 0x16, 0x01], [0x3F, 0x16, 0x01]),
    ([0xB6, 0x16, 0x01], [0xB7, 0x16, 0x01]),
    ([0x2B, 0x17, 0x01], [0x2B, 0x17, 0x01]),
    ([0x39, 0x18, 0x01], [0x3A, 0x18, 0x01]),
    ([0x3D, 0x19, 0x01], [0x3E, 0x19, 0x01]),
    ([0x43, 0x19, 0x01], [0x43, 0x19, 0x01]),
    ([0xE0, 0x19, 0x01], [0xE0, 0x19, 0x01]),
    ([0x34, 0x1A, 0x01], [0x34, 0x1A, 0x01]),
    ([0x47, 0x1A, 0x01], [0x47, 0x1A, 0x01]),
    ([0x99, 0x1A, 0x01], [0x99, 0x1A, 0x01]),
    ([0x3F, 0x1C, 0x01], [0x3F, 0x1C, 0x01]),
    ([0x42, 0x1D, 0x01], [0x42, 0x1D, 0x01]),
    ([0x44, 0x1D, 0x01], [0x45, 0x1D, 0x01]),
    ([0x97, 0x1D, 0x01], [0x97, 0x1D, 0x01]),
    ([0x41, 0x1F, 0x01], [0x42, 0x1F, 0x01]),
    ([0x30, 0x6B, 0x01], [0x36, 0x6B, 0x01]),
    ([0xF0, 0x6F, 0x01], [0xF1, 0x6F, 0x01]),
    ([0x65, 0xD1, 0x01], [0x66, 0xD1, 0x01]),
    ([0x6D, 0xD1, 0x01], [0x72, 0xD1, 0x01]),
    ([0x7B, 0xD1, 0x01], [0x82, 0xD1, 0x01]),
    ([0x85, 0xD1, 0x01], [0x8B, 0xD1, 0x01]),
    ([0xAA, 0xD1, 0x01], [0xAD, 0xD1, 0x01]),
    ([0x42, 0xD2, 0x01], [0x44, 0xD2, 0x01]),
    ([0x00, 0xE0, 0x01], [0x06, 0xE0, 0x01]),
    ([0x08, 0xE0, 0x01], [0x18, 0xE0, 0x01]),
    ([0x1B, 0xE0, 0x01], [0x21, 0xE0, 0x01]),
    ([0x23, 0xE0, 0x01], [0x24, 0xE0, 0x01]),
    ([0x26, 0xE0, 0x01], [0x2A, 0xE0, 0x01]),
    ([0x8F, 0xE0, 0x01], [0x8F, 0xE0, 0x01]),
    ([0x30, 0xE1, 0x01], [0x36, 0xE1, 0x01]),
    ([0xAE, 0xE2, 0x01], [0xAE, 0xE2, 0x01]),
    ([0xEC, 0xE2, 0x01], [0xEF, 0xE2, 0x01]),
    ([0xEC, 0xE4, 0x01], [0xEF, 0xE4, 0x01]),
    ([0xD0, 0xE8, 0x01], [0xD6, 0xE8, 0x01]),
    ([0x44, 0xE9, 0x01], [0x4A, 0xE9, 0x01]),
    ([0x85, 0xEC, 0x01], [0x85, 0xEC, 0x01]),
    ([0x15, 0xED, 0x01], [0x15, 0xED, 0x01]),
];

/// Array of 1024-bit bitmaps. Index into the correct bitmap with the 10 LSB of your codepoint
/// to get whether it can start an emoji presentation sequence.
static EMOJI_PRESENTATION_LEAVES: Align128<[[u8; 128]; 7]> = Align128([
    [
        0x00, 0x00, 0x00, 0x00, 0x08, 0x04, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    ],
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0x03, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x80, 0x00, 0x00, 0x00, 0xFE, 0x0F, 0x07,
    ],
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x78, 0x1F, 0x40, 0x32, 0x21, 0x4D, 0xC4, 0x00, 0x07, 0x05, 0xFF, 0x0F,
        0x80, 0x69, 0x01, 0x00, 0xC8, 0x00, 0x00, 0xFC, 0x1A, 0x83, 0x0C, 0x03, 0x60, 0x30, 0xC1,
        0x1A, 0x00, 0x00, 0x06, 0xBF, 0x27, 0x24, 0xBF, 0x54, 0x20, 0x02, 0x01, 0x18, 0x00, 0x90,
        0x50, 0xB8, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x00, 0x02, 0x00, 0x01, 0x80,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    ],
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    ],
    [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    ],
    [
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x03, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x04, 0x00, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x20, 0x10, 0xF2, 0x1F, 0x40, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0xC8, 0xCE, 0x80, 0x70, 0x00, 0x00,
        0x54, 0x7C, 0xF0, 0xFF, 0x01, 0x20, 0xA8, 0x00,
    ],
    [
        0x00, 0x01, 0x20, 0x80, 0x40, 0x00, 0x00, 0x80, 0xC6, 0x63, 0x08, 0x00, 0x00, 0x04, 0x00,
        0x20, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x09, 0x88, 0x00, 0x08, 0x00, 0x84, 0x70, 0x3C,
        0x80, 0x2E, 0x00, 0x21, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0xFF, 0xFF, 0xFF,
        0x80, 0xF9, 0x03, 0x80, 0x3C, 0x01, 0x00, 0x20, 0x01, 0x06, 0x10, 0x1C, 0x00, 0x0E, 0x70,
        0x0A, 0x81, 0x08, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x20, 0x12, 0x01, 0x00, 0x20, 0x04, 0x16, 0x00, 0xE8,
        0x00, 0x00, 0x3F, 0x02, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    ],
]);

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_0: [(u8, u8); 4] = [
    (0x1A, 0x1B),
    (0xE9, 0xEC),
    (0xF0, 0xF0),
    (0xF3, 0xF3),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_1: [(u8, u8); 1] = [
    (0xFD, 0xFE),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_2: [(u8, u8); 15] = [
    (0x14, 0x15),
    (0x48, 0x53),
    (0x7F, 0x7F),
    (0x93, 0x93),
    (0xA1, 0xA1),
    (0xAA, 0xAB),
    (0xBD, 0xBE),
    (0xC4, 0xC5),
    (0xCE, 0xCE),
    (0xD4, 0xD4),
    (0xEA, 0xEA),
    (0xF2, 0xF3),
    (0xF5, 0xF5),
    (0xFA, 0xFA),
    (0xFD, 0xFD),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_3: [(u8, u8); 10] = [
    (0x05, 0x05),
    (0x0A, 0x0B),
    (0x28, 0x28),
    (0x4C, 0x4C),
    (0x4E, 0x4E),
    (0x53, 0x55),
    (0x57, 0x57),
    (0x95, 0x97),
    (0xB0, 0xB0),
    (0xBF, 0xBF),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_4: [(u8, u8); 3] = [
    (0x1B, 0x1C),
    (0x50, 0x50),
    (0x55, 0x55),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_5: [(u8, u8); 1] = [
    (0x04, 0x04),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_6: [(u8, u8); 13] = [
    (0x0D, 0x0F),
    (0x15, 0x15),
    (0x1C, 0x1C),
    (0x78, 0x78),
    (0x93, 0x93),
    (0xA7, 0xA7),
    (0xAC, 0xAE),
    (0xC2, 0xC2),
    (0xC4, 0xC4),
    (0xC6, 0xC6),
    (0xCA, 0xCA),
    (0xE0, 0xE0),
    (0xED, 0xED),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_7: [(u8, u8); 22] = [
    (0x08, 0x08),
    (0x15, 0x15),
    (0x1F, 0x1F),
    (0x26, 0x26),
    (0x42, 0x42),
    (0x46, 0x49),
    (0x4D, 0x4E),
    (0x53, 0x53),
    (0x6A, 0x6A),
    (0x7D, 0x7D),
    (0xA3, 0xA3),
    (0xB0, 0xB0),
    (0xB3, 0xB3),
    (0xBB, 0xBB),
    (0xBF, 0xBF),
    (0xCB, 0xCB),
    (0xDA, 0xDA),
    (0xDF, 0xDF),
    (0xE4, 0xE6),
    (0xEA, 0xED),
    (0xF7, 0xF7),
    (0xF9, 0xFB),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_8: [(u8, u8); 4] = [
    (0x08, 0x08),
    (0x0D, 0x0D),
    (0x12, 0x13),
    (0x50, 0x67),
];

#[rustfmt::skip]
static TEXT_PRESENTATION_LEAF_9: [(u8, u8); 10] = [
    (0x10, 0x10),
    (0x87, 0x87),
    (0x8D, 0x8D),
    (0x91, 0x91),
    (0x94, 0x94),
    (0x98, 0x98),
    (0xAD, 0xAD),
    (0xB2, 0xB2),
    (0xB9, 0xBA),
    (0xBC, 0xBC),
];

#[rustfmt::skip]
static EMOJI_MODIFIER_LEAF_0: [(u8, u8); 2] = [
    (0x1D, 0x1D),
    (0xF9, 0xF9),
];

#[rustfmt::skip]
static EMOJI_MODIFIER_LEAF_1: [(u8, u8); 1] = [
    (0x0A, 0x0D),
];

#[rustfmt::skip]
static EMOJI_MODIFIER_LEAF_2: [(u8, u8); 4] = [
    (0x85, 0x85),
    (0xC2, 0xC4),
    (0xC7, 0xC7),
    (0xCA, 0xCC),
];

#[rustfmt::skip]
static EMOJI_MODIFIER_LEAF_3: [(u8, u8); 9] = [
    (0x42, 0x43),
    (0x46, 0x50),
    (0x66, 0x78),
    (0x7C, 0x7C),
    (0x81, 0x83),
    (0x85, 0x87),
    (0x8F, 0x8F),
    (0x91, 0x91),
    (0xAA, 0xAA),
];

#[rustfmt::skip]
static EMOJI_MODIFIER_LEAF_4: [(u8, u8); 4] = [
    (0x74, 0x75),
    (0x7A, 0x7A),
    (0x90, 0x90),
    (0x95, 0x96),
];

#[rustfmt::skip]
static EMOJI_MODIFIER_LEAF_5: [(u8, u8); 6] = [
    (0x45, 0x47),
    (0x4B, 0x4F),
    (0xA3, 0xA3),
    (0xB4, 0xB6),
    (0xC0, 0xC0),
    (0xCC, 0xCC),
];

#[rustfmt::skip]
static EMOJI_MODIFIER_LEAF_6: [(u8, u8); 12] = [
    (0x0C, 0x0C),
    (0x0F, 0x0F),
    (0x18, 0x1F),
    (0x26, 0x26),
    (0x30, 0x39),
    (0x3C, 0x3E),
    (0x77, 0x77),
    (0xB5, 0xB6),
    (0xB8, 0xB9),
    (0xBB, 0xBB),
    (0xCD, 0xCF),
    (0xD1, 0xDD),
];

#[rustfmt::skip]
static EMOJI_MODIFIER_LEAF_7: [(u8, u8); 2] = [
    (0xC3, 0xC5),
    (0xF0, 0xF8),
];

#[cfg(test)]
mod tests {
    use super::*;

    fn str_width_test(s: &str, init: WidthInfo) -> isize {
        s.chars()
            .rfold((0, init), |(sum, next_info), c| -> (isize, WidthInfo) {
                let (add, info) = width_in_str(c, next_info);
                (sum.checked_add(isize::from(add)).unwrap(), info)
            })
            .0
    }

    #[cfg(feature = "cjk")]
    fn str_width_test_cjk(s: &str, init: WidthInfo) -> isize {
        s.chars()
            .rfold((0, init), |(sum, next_info), c| -> (isize, WidthInfo) {
                let (add, info) = width_in_str_cjk(c, next_info);
                (sum.checked_add(isize::from(add)).unwrap(), info)
            })
            .0
    }

    #[test]
    fn test_normalization() {
        for &(orig, nfc, nfd, nfkc, nfkd) in &NORMALIZATION_TEST {
            for init in NORMALIZATION_TEST_WIDTHS {
                assert_eq!(
                    str_width_test(orig, init),
                    str_width_test(nfc, init),
                    "width of X = {orig:?} differs from toNFC(X) = {nfc:?} with mode {init:X?}",
                );
                assert_eq!(
                    str_width_test(orig, init),
                    str_width_test(nfd, init),
                    "width of X = {orig:?} differs from toNFD(X) = {nfd:?} with mode {init:X?}",
                );
                assert_eq!(
                    str_width_test(nfkc, init),
                    str_width_test(nfkd, init),
                    "width of toNFKC(X) = {nfkc:?} differs from toNFKD(X) = {nfkd:?} with mode {init:X?}",
                );
            }

            #[cfg(feature = "cjk")]
            for init in NORMALIZATION_TEST_WIDTHS_CJK {
                assert_eq!(
                    str_width_test_cjk(orig, init),
                    str_width_test_cjk(nfc, init),
                    "CJK width of X = {orig:?} differs from toNFC(X) = {nfc:?} with mode {init:X?}",
                );
                assert_eq!(
                    str_width_test_cjk(orig, init),
                    str_width_test_cjk(nfd, init),
                    "CJK width of X = {orig:?} differs from toNFD(X) = {nfd:?} with mode {init:X?}",
                );
                assert_eq!(
                    str_width_test_cjk(nfkc, init),
                    str_width_test_cjk(nfkd, init),
                    "CJK width of toNFKC(X) = {nfkc:?} differs from toNFKD(X) = {nfkd:?} with mode {init:?}",
                );
            }
        }
    }

    static NORMALIZATION_TEST_WIDTHS: [WidthInfo; 38] = [
        WidthInfo::DEFAULT,
        WidthInfo::LINE_FEED,
        WidthInfo::EMOJI_MODIFIER,
        WidthInfo::REGIONAL_INDICATOR,
        WidthInfo::SEVERAL_REGIONAL_INDICATOR,
        WidthInfo::EMOJI_PRESENTATION,
        WidthInfo::ZWJ_EMOJI_PRESENTATION,
        WidthInfo::VS16_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::KEYCAP_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::VS16_KEYCAP_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::REGIONAL_INDICATOR_ZWJ_PRESENTATION,
        WidthInfo::EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION,
        WidthInfo::ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION,
        WidthInfo::TAG_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_D1_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_D2_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_D3_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A1_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A2_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A5_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A6_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::VARIATION_SELECTOR_15,
        WidthInfo::VARIATION_SELECTOR_16,
        WidthInfo::JOINING_GROUP_ALEF,
        WidthInfo::HEBREW_LETTER_LAMED,
        WidthInfo::ZWJ_HEBREW_LETTER_LAMED,
        WidthInfo::BUGINESE_LETTER_YA,
        WidthInfo::ZWJ_BUGINESE_LETTER_YA,
        WidthInfo::BUGINESE_VOWEL_SIGN_I_ZWJ_LETTER_YA,
        WidthInfo::TIFINAGH_CONSONANT,
        WidthInfo::ZWJ_TIFINAGH_CONSONANT,
        WidthInfo::TIFINAGH_JOINER_CONSONANT,
        WidthInfo::LISU_TONE_LETTER_MYA_NA_JEU,
        WidthInfo::OLD_TURKIC_LETTER_ORKHON_I,
        WidthInfo::ZWJ_OLD_TURKIC_LETTER_ORKHON_I,
        WidthInfo::KHMER_COENG_ELIGIBLE_LETTER,
    ];

    #[cfg(feature = "cjk")]
    static NORMALIZATION_TEST_WIDTHS_CJK: [WidthInfo; 39] = [
        WidthInfo::DEFAULT,
        WidthInfo::LINE_FEED,
        WidthInfo::EMOJI_MODIFIER,
        WidthInfo::REGIONAL_INDICATOR,
        WidthInfo::SEVERAL_REGIONAL_INDICATOR,
        WidthInfo::EMOJI_PRESENTATION,
        WidthInfo::ZWJ_EMOJI_PRESENTATION,
        WidthInfo::VS16_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::KEYCAP_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::VS16_KEYCAP_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::REGIONAL_INDICATOR_ZWJ_PRESENTATION,
        WidthInfo::EVEN_REGIONAL_INDICATOR_ZWJ_PRESENTATION,
        WidthInfo::ODD_REGIONAL_INDICATOR_ZWJ_PRESENTATION,
        WidthInfo::TAG_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_D1_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_D2_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_D3_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A1_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A2_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A3_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A4_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A5_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::TAG_A6_END_ZWJ_EMOJI_PRESENTATION,
        WidthInfo::VARIATION_SELECTOR_16,
        WidthInfo::JOINING_GROUP_ALEF,
        WidthInfo::COMBINING_LONG_SOLIDUS_OVERLAY,
        WidthInfo::SOLIDUS_OVERLAY_ALEF,
        WidthInfo::HEBREW_LETTER_LAMED,
        WidthInfo::ZWJ_HEBREW_LETTER_LAMED,
        WidthInfo::BUGINESE_LETTER_YA,
        WidthInfo::ZWJ_BUGINESE_LETTER_YA,
        WidthInfo::BUGINESE_VOWEL_SIGN_I_ZWJ_LETTER_YA,
        WidthInfo::TIFINAGH_CONSONANT,
        WidthInfo::ZWJ_TIFINAGH_CONSONANT,
        WidthInfo::TIFINAGH_JOINER_CONSONANT,
        WidthInfo::LISU_TONE_LETTER_MYA_NA_JEU,
        WidthInfo::OLD_TURKIC_LETTER_ORKHON_I,
        WidthInfo::ZWJ_OLD_TURKIC_LETTER_ORKHON_I,
        WidthInfo::KHMER_COENG_ELIGIBLE_LETTER,
    ];

    #[rustfmt::skip]
    static NORMALIZATION_TEST: [(&str, &str, &str, &str, &str); 19074] = [
        (r#"Ḋ"#, r#"Ḋ"#, r#"Ḋ"#, r#"Ḋ"#, r#"Ḋ"#),
        (r#"Ḍ"#, r#"Ḍ"#, r#"Ḍ"#, r#"Ḍ"#, r#"Ḍ"#),
        (r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#),
        (r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#),
        (r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#),
        (r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#, r#"Ḍ̇"#),
        (r#"Ḋ̛"#, r#"Ḋ̛"#, r#"Ḋ̛"#, r#"Ḋ̛"#, r#"Ḋ̛"#),
        (r#"Ḍ̛"#, r#"Ḍ̛"#, r#"Ḍ̛"#, r#"Ḍ̛"#, r#"Ḍ̛"#),
        (r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#),
        (r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#),
        (r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#),
        (r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#, r#"Ḍ̛̇"#),
        (r#"È"#, r#"È"#, r#"È"#, r#"È"#, r#"È"#),
        (r#"Ē"#, r#"Ē"#, r#"Ē"#, r#"Ē"#, r#"Ē"#),
        (r#"È"#, r#"È"#, r#"È"#, r#"È"#, r#"È"#),
        (r#"Ē"#, r#"Ē"#, r#"Ē"#, r#"Ē"#, r#"Ē"#),
        (r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#),
        (r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#),
        (r#"Ḕ̄"#, r#"Ḕ̄"#, r#"Ḕ̄"#, r#"Ḕ̄"#, r#"Ḕ̄"#),
        (r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#),
        (r#"È̄"#, r#"È̄"#, r#"È̄"#, r#"È̄"#, r#"È̄"#),
        (r#"ֱָֹ֑׃ְ֬֟"#, r#"ֱָֹ֑׃ְ֬֟"#, r#"ֱָֹ֑׃ְ֬֟"#, r#"ֱָֹ֑׃ְ֬֟"#, r#"ֱָֹ֑׃ְ֬֟"#),
        (r#"ְַּ֥֒׀֭ׄ"#, r#"ְַּ֥֒׀֭ׄ"#, r#"ְַּ֥֒׀֭ׄ"#, r#"ְַּ֥֒׀֭ׄ"#, r#"ְַּ֥֒׀֭ׄ"#),
        (r#"ᄀ각"#, r#"ᄀ각"#, r#"ᄀ각"#, r#"ᄀ각"#, r#"ᄀ각"#),
        (r#"ᄀ각ᆨ"#, r#"ᄀ각ᆨ"#, r#"ᄀ각ᆨ"#, r#"ᄀ각ᆨ"#, r#"ᄀ각ᆨ"#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#"¨"#, r#"¨"#, r#"¨"#, r#" ̈"#, r#" ̈"#),
        (r#"ª"#, r#"ª"#, r#"ª"#, r#"a"#, r#"a"#),
        (r#"¯"#, r#"¯"#, r#"¯"#, r#" ̄"#, r#" ̄"#),
        (r#"²"#, r#"²"#, r#"²"#, r#"2"#, r#"2"#),
        (r#"³"#, r#"³"#, r#"³"#, r#"3"#, r#"3"#),
        (r#"´"#, r#"´"#, r#"´"#, r#" ́"#, r#" ́"#),
        (r#"µ"#, r#"µ"#, r#"µ"#, r#"μ"#, r#"μ"#),
        (r#"¸"#, r#"¸"#, r#"¸"#, r#" ̧"#, r#" ̧"#),
        (r#"¹"#, r#"¹"#, r#"¹"#, r#"1"#, r#"1"#),
        (r#"º"#, r#"º"#, r#"º"#, r#"o"#, r#"o"#),
        (r#"¼"#, r#"¼"#, r#"¼"#, r#"1⁄4"#, r#"1⁄4"#),
        (r#"½"#, r#"½"#, r#"½"#, r#"1⁄2"#, r#"1⁄2"#),
        (r#"¾"#, r#"¾"#, r#"¾"#, r#"3⁄4"#, r#"3⁄4"#),
        (r#"À"#, r#"À"#, r#"À"#, r#"À"#, r#"À"#),
        (r#"Á"#, r#"Á"#, r#"Á"#, r#"Á"#, r#"Á"#),
        (r#"Â"#, r#"Â"#, r#"Â"#, r#"Â"#, r#"Â"#),
        (r#"Ã"#, r#"Ã"#, r#"Ã"#, r#"Ã"#, r#"Ã"#),
        (r#"Ä"#, r#"Ä"#, r#"Ä"#, r#"Ä"#, r#"Ä"#),
        (r#"Å"#, r#"Å"#, r#"Å"#, r#"Å"#, r#"Å"#),
        (r#"Ç"#, r#"Ç"#, r#"Ç"#, r#"Ç"#, r#"Ç"#),
        (r#"È"#, r#"È"#, r#"È"#, r#"È"#, r#"È"#),
        (r#"É"#, r#"É"#, r#"É"#, r#"É"#, r#"É"#),
        (r#"Ê"#, r#"Ê"#, r#"Ê"#, r#"Ê"#, r#"Ê"#),
        (r#"Ë"#, r#"Ë"#, r#"Ë"#, r#"Ë"#, r#"Ë"#),
        (r#"Ì"#, r#"Ì"#, r#"Ì"#, r#"Ì"#, r#"Ì"#),
        (r#"Í"#, r#"Í"#, r#"Í"#, r#"Í"#, r#"Í"#),
        (r#"Î"#, r#"Î"#, r#"Î"#, r#"Î"#, r#"Î"#),
        (r#"Ï"#, r#"Ï"#, r#"Ï"#, r#"Ï"#, r#"Ï"#),
        (r#"Ñ"#, r#"Ñ"#, r#"Ñ"#, r#"Ñ"#, r#"Ñ"#),
        (r#"Ò"#, r#"Ò"#, r#"Ò"#, r#"Ò"#, r#"Ò"#),
        (r#"Ó"#, r#"Ó"#, r#"Ó"#, r#"Ó"#, r#"Ó"#),
        (r#"Ô"#, r#"Ô"#, r#"Ô"#, r#"Ô"#, r#"Ô"#),
        (r#"Õ"#, r#"Õ"#, r#"Õ"#, r#"Õ"#, r#"Õ"#),
        (r#"Ö"#, r#"Ö"#, r#"Ö"#, r#"Ö"#, r#"Ö"#),
        (r#"Ù"#, r#"Ù"#, r#"Ù"#, r#"Ù"#, r#"Ù"#),
        (r#"Ú"#, r#"Ú"#, r#"Ú"#, r#"Ú"#, r#"Ú"#),
        (r#"Û"#, r#"Û"#, r#"Û"#, r#"Û"#, r#"Û"#),
        (r#"Ü"#, r#"Ü"#, r#"Ü"#, r#"Ü"#, r#"Ü"#),
        (r#"Ý"#, r#"Ý"#, r#"Ý"#, r#"Ý"#, r#"Ý"#),
        (r#"à"#, r#"à"#, r#"à"#, r#"à"#, r#"à"#),
        (r#"á"#, r#"á"#, r#"á"#, r#"á"#, r#"á"#),
        (r#"â"#, r#"â"#, r#"â"#, r#"â"#, r#"â"#),
        (r#"ã"#, r#"ã"#, r#"ã"#, r#"ã"#, r#"ã"#),
        (r#"ä"#, r#"ä"#, r#"ä"#, r#"ä"#, r#"ä"#),
        (r#"å"#, r#"å"#, r#"å"#, r#"å"#, r#"å"#),
        (r#"ç"#, r#"ç"#, r#"ç"#, r#"ç"#, r#"ç"#),
        (r#"è"#, r#"è"#, r#"è"#, r#"è"#, r#"è"#),
        (r#"é"#, r#"é"#, r#"é"#, r#"é"#, r#"é"#),
        (r#"ê"#, r#"ê"#, r#"ê"#, r#"ê"#, r#"ê"#),
        (r#"ë"#, r#"ë"#, r#"ë"#, r#"ë"#, r#"ë"#),
        (r#"ì"#, r#"ì"#, r#"ì"#, r#"ì"#, r#"ì"#),
        (r#"í"#, r#"í"#, r#"í"#, r#"í"#, r#"í"#),
        (r#"î"#, r#"î"#, r#"î"#, r#"î"#, r#"î"#),
        (r#"ï"#, r#"ï"#, r#"ï"#, r#"ï"#, r#"ï"#),
        (r#"ñ"#, r#"ñ"#, r#"ñ"#, r#"ñ"#, r#"ñ"#),
        (r#"ò"#, r#"ò"#, r#"ò"#, r#"ò"#, r#"ò"#),
        (r#"ó"#, r#"ó"#, r#"ó"#, r#"ó"#, r#"ó"#),
        (r#"ô"#, r#"ô"#, r#"ô"#, r#"ô"#, r#"ô"#),
        (r#"õ"#, r#"õ"#, r#"õ"#, r#"õ"#, r#"õ"#),
        (r#"ö"#, r#"ö"#, r#"ö"#, r#"ö"#, r#"ö"#),
        (r#"ù"#, r#"ù"#, r#"ù"#, r#"ù"#, r#"ù"#),
        (r#"ú"#, r#"ú"#, r#"ú"#, r#"ú"#, r#"ú"#),
        (r#"û"#, r#"û"#, r#"û"#, r#"û"#, r#"û"#),
        (r#"ü"#, r#"ü"#, r#"ü"#, r#"ü"#, r#"ü"#),
        (r#"ý"#, r#"ý"#, r#"ý"#, r#"ý"#, r#"ý"#),
        (r#"ÿ"#, r#"ÿ"#, r#"ÿ"#, r#"ÿ"#, r#"ÿ"#),
        (r#"Ā"#, r#"Ā"#, r#"Ā"#, r#"Ā"#, r#"Ā"#),
        (r#"ā"#, r#"ā"#, r#"ā"#, r#"ā"#, r#"ā"#),
        (r#"Ă"#, r#"Ă"#, r#"Ă"#, r#"Ă"#, r#"Ă"#),
        (r#"ă"#, r#"ă"#, r#"ă"#, r#"ă"#, r#"ă"#),
        (r#"Ą"#, r#"Ą"#, r#"Ą"#, r#"Ą"#, r#"Ą"#),
        (r#"ą"#, r#"ą"#, r#"ą"#, r#"ą"#, r#"ą"#),
        (r#"Ć"#, r#"Ć"#, r#"Ć"#, r#"Ć"#, r#"Ć"#),
        (r#"ć"#, r#"ć"#, r#"ć"#, r#"ć"#, r#"ć"#),
        (r#"Ĉ"#, r#"Ĉ"#, r#"Ĉ"#, r#"Ĉ"#, r#"Ĉ"#),
        (r#"ĉ"#, r#"ĉ"#, r#"ĉ"#, r#"ĉ"#, r#"ĉ"#),
        (r#"Ċ"#, r#"Ċ"#, r#"Ċ"#, r#"Ċ"#, r#"Ċ"#),
        (r#"ċ"#, r#"ċ"#, r#"ċ"#, r#"ċ"#, r#"ċ"#),
        (r#"Č"#, r#"Č"#, r#"Č"#, r#"Č"#, r#"Č"#),
        (r#"č"#, r#"č"#, r#"č"#, r#"č"#, r#"č"#),
        (r#"Ď"#, r#"Ď"#, r#"Ď"#, r#"Ď"#, r#"Ď"#),
        (r#"ď"#, r#"ď"#, r#"ď"#, r#"ď"#, r#"ď"#),
        (r#"Ē"#, r#"Ē"#, r#"Ē"#, r#"Ē"#, r#"Ē"#),
        (r#"ē"#, r#"ē"#, r#"ē"#, r#"ē"#, r#"ē"#),
        (r#"Ĕ"#, r#"Ĕ"#, r#"Ĕ"#, r#"Ĕ"#, r#"Ĕ"#),
        (r#"ĕ"#, r#"ĕ"#, r#"ĕ"#, r#"ĕ"#, r#"ĕ"#),
        (r#"Ė"#, r#"Ė"#, r#"Ė"#, r#"Ė"#, r#"Ė"#),
        (r#"ė"#, r#"ė"#, r#"ė"#, r#"ė"#, r#"ė"#),
        (r#"Ę"#, r#"Ę"#, r#"Ę"#, r#"Ę"#, r#"Ę"#),
        (r#"ę"#, r#"ę"#, r#"ę"#, r#"ę"#, r#"ę"#),
        (r#"Ě"#, r#"Ě"#, r#"Ě"#, r#"Ě"#, r#"Ě"#),
        (r#"ě"#, r#"ě"#, r#"ě"#, r#"ě"#, r#"ě"#),
        (r#"Ĝ"#, r#"Ĝ"#, r#"Ĝ"#, r#"Ĝ"#, r#"Ĝ"#),
        (r#"ĝ"#, r#"ĝ"#, r#"ĝ"#, r#"ĝ"#, r#"ĝ"#),
        (r#"Ğ"#, r#"Ğ"#, r#"Ğ"#, r#"Ğ"#, r#"Ğ"#),
        (r#"ğ"#, r#"ğ"#, r#"ğ"#, r#"ğ"#, r#"ğ"#),
        (r#"Ġ"#, r#"Ġ"#, r#"Ġ"#, r#"Ġ"#, r#"Ġ"#),
        (r#"ġ"#, r#"ġ"#, r#"ġ"#, r#"ġ"#, r#"ġ"#),
        (r#"Ģ"#, r#"Ģ"#, r#"Ģ"#, r#"Ģ"#, r#"Ģ"#),
        (r#"ģ"#, r#"ģ"#, r#"ģ"#, r#"ģ"#, r#"ģ"#),
        (r#"Ĥ"#, r#"Ĥ"#, r#"Ĥ"#, r#"Ĥ"#, r#"Ĥ"#),
        (r#"ĥ"#, r#"ĥ"#, r#"ĥ"#, r#"ĥ"#, r#"ĥ"#),
        (r#"Ĩ"#, r#"Ĩ"#, r#"Ĩ"#, r#"Ĩ"#, r#"Ĩ"#),
        (r#"ĩ"#, r#"ĩ"#, r#"ĩ"#, r#"ĩ"#, r#"ĩ"#),
        (r#"Ī"#, r#"Ī"#, r#"Ī"#, r#"Ī"#, r#"Ī"#),
        (r#"ī"#, r#"ī"#, r#"ī"#, r#"ī"#, r#"ī"#),
        (r#"Ĭ"#, r#"Ĭ"#, r#"Ĭ"#, r#"Ĭ"#, r#"Ĭ"#),
        (r#"ĭ"#, r#"ĭ"#, r#"ĭ"#, r#"ĭ"#, r#"ĭ"#),
        (r#"Į"#, r#"Į"#, r#"Į"#, r#"Į"#, r#"Į"#),
        (r#"į"#, r#"į"#, r#"į"#, r#"į"#, r#"į"#),
        (r#"İ"#, r#"İ"#, r#"İ"#, r#"İ"#, r#"İ"#),
        (r#"Ĳ"#, r#"Ĳ"#, r#"Ĳ"#, r#"IJ"#, r#"IJ"#),
        (r#"ĳ"#, r#"ĳ"#, r#"ĳ"#, r#"ij"#, r#"ij"#),
        (r#"Ĵ"#, r#"Ĵ"#, r#"Ĵ"#, r#"Ĵ"#, r#"Ĵ"#),
        (r#"ĵ"#, r#"ĵ"#, r#"ĵ"#, r#"ĵ"#, r#"ĵ"#),
        (r#"Ķ"#, r#"Ķ"#, r#"Ķ"#, r#"Ķ"#, r#"Ķ"#),
        (r#"ķ"#, r#"ķ"#, r#"ķ"#, r#"ķ"#, r#"ķ"#),
        (r#"Ĺ"#, r#"Ĺ"#, r#"Ĺ"#, r#"Ĺ"#, r#"Ĺ"#),
        (r#"ĺ"#, r#"ĺ"#, r#"ĺ"#, r#"ĺ"#, r#"ĺ"#),
        (r#"Ļ"#, r#"Ļ"#, r#"Ļ"#, r#"Ļ"#, r#"Ļ"#),
        (r#"ļ"#, r#"ļ"#, r#"ļ"#, r#"ļ"#, r#"ļ"#),
        (r#"Ľ"#, r#"Ľ"#, r#"Ľ"#, r#"Ľ"#, r#"Ľ"#),
        (r#"ľ"#, r#"ľ"#, r#"ľ"#, r#"ľ"#, r#"ľ"#),
        (r#"Ŀ"#, r#"Ŀ"#, r#"Ŀ"#, r#"L·"#, r#"L·"#),
        (r#"ŀ"#, r#"ŀ"#, r#"ŀ"#, r#"l·"#, r#"l·"#),
        (r#"Ń"#, r#"Ń"#, r#"Ń"#, r#"Ń"#, r#"Ń"#),
        (r#"ń"#, r#"ń"#, r#"ń"#, r#"ń"#, r#"ń"#),
        (r#"Ņ"#, r#"Ņ"#, r#"Ņ"#, r#"Ņ"#, r#"Ņ"#),
        (r#"ņ"#, r#"ņ"#, r#"ņ"#, r#"ņ"#, r#"ņ"#),
        (r#"Ň"#, r#"Ň"#, r#"Ň"#, r#"Ň"#, r#"Ň"#),
        (r#"ň"#, r#"ň"#, r#"ň"#, r#"ň"#, r#"ň"#),
        (r#"ŉ"#, r#"ŉ"#, r#"ŉ"#, r#"ʼn"#, r#"ʼn"#),
        (r#"Ō"#, r#"Ō"#, r#"Ō"#, r#"Ō"#, r#"Ō"#),
        (r#"ō"#, r#"ō"#, r#"ō"#, r#"ō"#, r#"ō"#),
        (r#"Ŏ"#, r#"Ŏ"#, r#"Ŏ"#, r#"Ŏ"#, r#"Ŏ"#),
        (r#"ŏ"#, r#"ŏ"#, r#"ŏ"#, r#"ŏ"#, r#"ŏ"#),
        (r#"Ő"#, r#"Ő"#, r#"Ő"#, r#"Ő"#, r#"Ő"#),
        (r#"ő"#, r#"ő"#, r#"ő"#, r#"ő"#, r#"ő"#),
        (r#"Ŕ"#, r#"Ŕ"#, r#"Ŕ"#, r#"Ŕ"#, r#"Ŕ"#),
        (r#"ŕ"#, r#"ŕ"#, r#"ŕ"#, r#"ŕ"#, r#"ŕ"#),
        (r#"Ŗ"#, r#"Ŗ"#, r#"Ŗ"#, r#"Ŗ"#, r#"Ŗ"#),
        (r#"ŗ"#, r#"ŗ"#, r#"ŗ"#, r#"ŗ"#, r#"ŗ"#),
        (r#"Ř"#, r#"Ř"#, r#"Ř"#, r#"Ř"#, r#"Ř"#),
        (r#"ř"#, r#"ř"#, r#"ř"#, r#"ř"#, r#"ř"#),
        (r#"Ś"#, r#"Ś"#, r#"Ś"#, r#"Ś"#, r#"Ś"#),
        (r#"ś"#, r#"ś"#, r#"ś"#, r#"ś"#, r#"ś"#),
        (r#"Ŝ"#, r#"Ŝ"#, r#"Ŝ"#, r#"Ŝ"#, r#"Ŝ"#),
        (r#"ŝ"#, r#"ŝ"#, r#"ŝ"#, r#"ŝ"#, r#"ŝ"#),
        (r#"Ş"#, r#"Ş"#, r#"Ş"#, r#"Ş"#, r#"Ş"#),
        (r#"ş"#, r#"ş"#, r#"ş"#, r#"ş"#, r#"ş"#),
        (r#"Š"#, r#"Š"#, r#"Š"#, r#"Š"#, r#"Š"#),
        (r#"š"#, r#"š"#, r#"š"#, r#"š"#, r#"š"#),
        (r#"Ţ"#, r#"Ţ"#, r#"Ţ"#, r#"Ţ"#, r#"Ţ"#),
        (r#"ţ"#, r#"ţ"#, r#"ţ"#, r#"ţ"#, r#"ţ"#),
        (r#"Ť"#, r#"Ť"#, r#"Ť"#, r#"Ť"#, r#"Ť"#),
        (r#"ť"#, r#"ť"#, r#"ť"#, r#"ť"#, r#"ť"#),
        (r#"Ũ"#, r#"Ũ"#, r#"Ũ"#, r#"Ũ"#, r#"Ũ"#),
        (r#"ũ"#, r#"ũ"#, r#"ũ"#, r#"ũ"#, r#"ũ"#),
        (r#"Ū"#, r#"Ū"#, r#"Ū"#, r#"Ū"#, r#"Ū"#),
        (r#"ū"#, r#"ū"#, r#"ū"#, r#"ū"#, r#"ū"#),
        (r#"Ŭ"#, r#"Ŭ"#, r#"Ŭ"#, r#"Ŭ"#, r#"Ŭ"#),
        (r#"ŭ"#, r#"ŭ"#, r#"ŭ"#, r#"ŭ"#, r#"ŭ"#),
        (r#"Ů"#, r#"Ů"#, r#"Ů"#, r#"Ů"#, r#"Ů"#),
        (r#"ů"#, r#"ů"#, r#"ů"#, r#"ů"#, r#"ů"#),
        (r#"Ű"#, r#"Ű"#, r#"Ű"#, r#"Ű"#, r#"Ű"#),
        (r#"ű"#, r#"ű"#, r#"ű"#, r#"ű"#, r#"ű"#),
        (r#"Ų"#, r#"Ų"#, r#"Ų"#, r#"Ų"#, r#"Ų"#),
        (r#"ų"#, r#"ų"#, r#"ų"#, r#"ų"#, r#"ų"#),
        (r#"Ŵ"#, r#"Ŵ"#, r#"Ŵ"#, r#"Ŵ"#, r#"Ŵ"#),
        (r#"ŵ"#, r#"ŵ"#, r#"ŵ"#, r#"ŵ"#, r#"ŵ"#),
        (r#"Ŷ"#, r#"Ŷ"#, r#"Ŷ"#, r#"Ŷ"#, r#"Ŷ"#),
        (r#"ŷ"#, r#"ŷ"#, r#"ŷ"#, r#"ŷ"#, r#"ŷ"#),
        (r#"Ÿ"#, r#"Ÿ"#, r#"Ÿ"#, r#"Ÿ"#, r#"Ÿ"#),
        (r#"Ź"#, r#"Ź"#, r#"Ź"#, r#"Ź"#, r#"Ź"#),
        (r#"ź"#, r#"ź"#, r#"ź"#, r#"ź"#, r#"ź"#),
        (r#"Ż"#, r#"Ż"#, r#"Ż"#, r#"Ż"#, r#"Ż"#),
        (r#"ż"#, r#"ż"#, r#"ż"#, r#"ż"#, r#"ż"#),
        (r#"Ž"#, r#"Ž"#, r#"Ž"#, r#"Ž"#, r#"Ž"#),
        (r#"ž"#, r#"ž"#, r#"ž"#, r#"ž"#, r#"ž"#),
        (r#"ſ"#, r#"ſ"#, r#"ſ"#, r#"s"#, r#"s"#),
        (r#"Ơ"#, r#"Ơ"#, r#"Ơ"#, r#"Ơ"#, r#"Ơ"#),
        (r#"ơ"#, r#"ơ"#, r#"ơ"#, r#"ơ"#, r#"ơ"#),
        (r#"Ư"#, r#"Ư"#, r#"Ư"#, r#"Ư"#, r#"Ư"#),
        (r#"ư"#, r#"ư"#, r#"ư"#, r#"ư"#, r#"ư"#),
        (r#"Ǆ"#, r#"Ǆ"#, r#"Ǆ"#, r#"DŽ"#, r#"DŽ"#),
        (r#"ǅ"#, r#"ǅ"#, r#"ǅ"#, r#"Dž"#, r#"Dž"#),
        (r#"ǆ"#, r#"ǆ"#, r#"ǆ"#, r#"dž"#, r#"dž"#),
        (r#"Ǉ"#, r#"Ǉ"#, r#"Ǉ"#, r#"LJ"#, r#"LJ"#),
        (r#"ǈ"#, r#"ǈ"#, r#"ǈ"#, r#"Lj"#, r#"Lj"#),
        (r#"ǉ"#, r#"ǉ"#, r#"ǉ"#, r#"lj"#, r#"lj"#),
        (r#"Ǌ"#, r#"Ǌ"#, r#"Ǌ"#, r#"NJ"#, r#"NJ"#),
        (r#"ǋ"#, r#"ǋ"#, r#"ǋ"#, r#"Nj"#, r#"Nj"#),
        (r#"ǌ"#, r#"ǌ"#, r#"ǌ"#, r#"nj"#, r#"nj"#),
        (r#"Ǎ"#, r#"Ǎ"#, r#"Ǎ"#, r#"Ǎ"#, r#"Ǎ"#),
        (r#"ǎ"#, r#"ǎ"#, r#"ǎ"#, r#"ǎ"#, r#"ǎ"#),
        (r#"Ǐ"#, r#"Ǐ"#, r#"Ǐ"#, r#"Ǐ"#, r#"Ǐ"#),
        (r#"ǐ"#, r#"ǐ"#, r#"ǐ"#, r#"ǐ"#, r#"ǐ"#),
        (r#"Ǒ"#, r#"Ǒ"#, r#"Ǒ"#, r#"Ǒ"#, r#"Ǒ"#),
        (r#"ǒ"#, r#"ǒ"#, r#"ǒ"#, r#"ǒ"#, r#"ǒ"#),
        (r#"Ǔ"#, r#"Ǔ"#, r#"Ǔ"#, r#"Ǔ"#, r#"Ǔ"#),
        (r#"ǔ"#, r#"ǔ"#, r#"ǔ"#, r#"ǔ"#, r#"ǔ"#),
        (r#"Ǖ"#, r#"Ǖ"#, r#"Ǖ"#, r#"Ǖ"#, r#"Ǖ"#),
        (r#"ǖ"#, r#"ǖ"#, r#"ǖ"#, r#"ǖ"#, r#"ǖ"#),
        (r#"Ǘ"#, r#"Ǘ"#, r#"Ǘ"#, r#"Ǘ"#, r#"Ǘ"#),
        (r#"ǘ"#, r#"ǘ"#, r#"ǘ"#, r#"ǘ"#, r#"ǘ"#),
        (r#"Ǚ"#, r#"Ǚ"#, r#"Ǚ"#, r#"Ǚ"#, r#"Ǚ"#),
        (r#"ǚ"#, r#"ǚ"#, r#"ǚ"#, r#"ǚ"#, r#"ǚ"#),
        (r#"Ǜ"#, r#"Ǜ"#, r#"Ǜ"#, r#"Ǜ"#, r#"Ǜ"#),
        (r#"ǜ"#, r#"ǜ"#, r#"ǜ"#, r#"ǜ"#, r#"ǜ"#),
        (r#"Ǟ"#, r#"Ǟ"#, r#"Ǟ"#, r#"Ǟ"#, r#"Ǟ"#),
        (r#"ǟ"#, r#"ǟ"#, r#"ǟ"#, r#"ǟ"#, r#"ǟ"#),
        (r#"Ǡ"#, r#"Ǡ"#, r#"Ǡ"#, r#"Ǡ"#, r#"Ǡ"#),
        (r#"ǡ"#, r#"ǡ"#, r#"ǡ"#, r#"ǡ"#, r#"ǡ"#),
        (r#"Ǣ"#, r#"Ǣ"#, r#"Ǣ"#, r#"Ǣ"#, r#"Ǣ"#),
        (r#"ǣ"#, r#"ǣ"#, r#"ǣ"#, r#"ǣ"#, r#"ǣ"#),
        (r#"Ǧ"#, r#"Ǧ"#, r#"Ǧ"#, r#"Ǧ"#, r#"Ǧ"#),
        (r#"ǧ"#, r#"ǧ"#, r#"ǧ"#, r#"ǧ"#, r#"ǧ"#),
        (r#"Ǩ"#, r#"Ǩ"#, r#"Ǩ"#, r#"Ǩ"#, r#"Ǩ"#),
        (r#"ǩ"#, r#"ǩ"#, r#"ǩ"#, r#"ǩ"#, r#"ǩ"#),
        (r#"Ǫ"#, r#"Ǫ"#, r#"Ǫ"#, r#"Ǫ"#, r#"Ǫ"#),
        (r#"ǫ"#, r#"ǫ"#, r#"ǫ"#, r#"ǫ"#, r#"ǫ"#),
        (r#"Ǭ"#, r#"Ǭ"#, r#"Ǭ"#, r#"Ǭ"#, r#"Ǭ"#),
        (r#"ǭ"#, r#"ǭ"#, r#"ǭ"#, r#"ǭ"#, r#"ǭ"#),
        (r#"Ǯ"#, r#"Ǯ"#, r#"Ǯ"#, r#"Ǯ"#, r#"Ǯ"#),
        (r#"ǯ"#, r#"ǯ"#, r#"ǯ"#, r#"ǯ"#, r#"ǯ"#),
        (r#"ǰ"#, r#"ǰ"#, r#"ǰ"#, r#"ǰ"#, r#"ǰ"#),
        (r#"Ǳ"#, r#"Ǳ"#, r#"Ǳ"#, r#"DZ"#, r#"DZ"#),
        (r#"ǲ"#, r#"ǲ"#, r#"ǲ"#, r#"Dz"#, r#"Dz"#),
        (r#"ǳ"#, r#"ǳ"#, r#"ǳ"#, r#"dz"#, r#"dz"#),
        (r#"Ǵ"#, r#"Ǵ"#, r#"Ǵ"#, r#"Ǵ"#, r#"Ǵ"#),
        (r#"ǵ"#, r#"ǵ"#, r#"ǵ"#, r#"ǵ"#, r#"ǵ"#),
        (r#"Ǹ"#, r#"Ǹ"#, r#"Ǹ"#, r#"Ǹ"#, r#"Ǹ"#),
        (r#"ǹ"#, r#"ǹ"#, r#"ǹ"#, r#"ǹ"#, r#"ǹ"#),
        (r#"Ǻ"#, r#"Ǻ"#, r#"Ǻ"#, r#"Ǻ"#, r#"Ǻ"#),
        (r#"ǻ"#, r#"ǻ"#, r#"ǻ"#, r#"ǻ"#, r#"ǻ"#),
        (r#"Ǽ"#, r#"Ǽ"#, r#"Ǽ"#, r#"Ǽ"#, r#"Ǽ"#),
        (r#"ǽ"#, r#"ǽ"#, r#"ǽ"#, r#"ǽ"#, r#"ǽ"#),
        (r#"Ǿ"#, r#"Ǿ"#, r#"Ǿ"#, r#"Ǿ"#, r#"Ǿ"#),
        (r#"ǿ"#, r#"ǿ"#, r#"ǿ"#, r#"ǿ"#, r#"ǿ"#),
        (r#"Ȁ"#, r#"Ȁ"#, r#"Ȁ"#, r#"Ȁ"#, r#"Ȁ"#),
        (r#"ȁ"#, r#"ȁ"#, r#"ȁ"#, r#"ȁ"#, r#"ȁ"#),
        (r#"Ȃ"#, r#"Ȃ"#, r#"Ȃ"#, r#"Ȃ"#, r#"Ȃ"#),
        (r#"ȃ"#, r#"ȃ"#, r#"ȃ"#, r#"ȃ"#, r#"ȃ"#),
        (r#"Ȅ"#, r#"Ȅ"#, r#"Ȅ"#, r#"Ȅ"#, r#"Ȅ"#),
        (r#"ȅ"#, r#"ȅ"#, r#"ȅ"#, r#"ȅ"#, r#"ȅ"#),
        (r#"Ȇ"#, r#"Ȇ"#, r#"Ȇ"#, r#"Ȇ"#, r#"Ȇ"#),
        (r#"ȇ"#, r#"ȇ"#, r#"ȇ"#, r#"ȇ"#, r#"ȇ"#),
        (r#"Ȉ"#, r#"Ȉ"#, r#"Ȉ"#, r#"Ȉ"#, r#"Ȉ"#),
        (r#"ȉ"#, r#"ȉ"#, r#"ȉ"#, r#"ȉ"#, r#"ȉ"#),
        (r#"Ȋ"#, r#"Ȋ"#, r#"Ȋ"#, r#"Ȋ"#, r#"Ȋ"#),
        (r#"ȋ"#, r#"ȋ"#, r#"ȋ"#, r#"ȋ"#, r#"ȋ"#),
        (r#"Ȍ"#, r#"Ȍ"#, r#"Ȍ"#, r#"Ȍ"#, r#"Ȍ"#),
        (r#"ȍ"#, r#"ȍ"#, r#"ȍ"#, r#"ȍ"#, r#"ȍ"#),
        (r#"Ȏ"#, r#"Ȏ"#, r#"Ȏ"#, r#"Ȏ"#, r#"Ȏ"#),
        (r#"ȏ"#, r#"ȏ"#, r#"ȏ"#, r#"ȏ"#, r#"ȏ"#),
        (r#"Ȑ"#, r#"Ȑ"#, r#"Ȑ"#, r#"Ȑ"#, r#"Ȑ"#),
        (r#"ȑ"#, r#"ȑ"#, r#"ȑ"#, r#"ȑ"#, r#"ȑ"#),
        (r#"Ȓ"#, r#"Ȓ"#, r#"Ȓ"#, r#"Ȓ"#, r#"Ȓ"#),
        (r#"ȓ"#, r#"ȓ"#, r#"ȓ"#, r#"ȓ"#, r#"ȓ"#),
        (r#"Ȕ"#, r#"Ȕ"#, r#"Ȕ"#, r#"Ȕ"#, r#"Ȕ"#),
        (r#"ȕ"#, r#"ȕ"#, r#"ȕ"#, r#"ȕ"#, r#"ȕ"#),
        (r#"Ȗ"#, r#"Ȗ"#, r#"Ȗ"#, r#"Ȗ"#, r#"Ȗ"#),
        (r#"ȗ"#, r#"ȗ"#, r#"ȗ"#, r#"ȗ"#, r#"ȗ"#),
        (r#"Ș"#, r#"Ș"#, r#"Ș"#, r#"Ș"#, r#"Ș"#),
        (r#"ș"#, r#"ș"#, r#"ș"#, r#"ș"#, r#"ș"#),
        (r#"Ț"#, r#"Ț"#, r#"Ț"#, r#"Ț"#, r#"Ț"#),
        (r#"ț"#, r#"ț"#, r#"ț"#, r#"ț"#, r#"ț"#),
        (r#"Ȟ"#, r#"Ȟ"#, r#"Ȟ"#, r#"Ȟ"#, r#"Ȟ"#),
        (r#"ȟ"#, r#"ȟ"#, r#"ȟ"#, r#"ȟ"#, r#"ȟ"#),
        (r#"Ȧ"#, r#"Ȧ"#, r#"Ȧ"#, r#"Ȧ"#, r#"Ȧ"#),
        (r#"ȧ"#, r#"ȧ"#, r#"ȧ"#, r#"ȧ"#, r#"ȧ"#),
        (r#"Ȩ"#, r#"Ȩ"#, r#"Ȩ"#, r#"Ȩ"#, r#"Ȩ"#),
        (r#"ȩ"#, r#"ȩ"#, r#"ȩ"#, r#"ȩ"#, r#"ȩ"#),
        (r#"Ȫ"#, r#"Ȫ"#, r#"Ȫ"#, r#"Ȫ"#, r#"Ȫ"#),
        (r#"ȫ"#, r#"ȫ"#, r#"ȫ"#, r#"ȫ"#, r#"ȫ"#),
        (r#"Ȭ"#, r#"Ȭ"#, r#"Ȭ"#, r#"Ȭ"#, r#"Ȭ"#),
        (r#"ȭ"#, r#"ȭ"#, r#"ȭ"#, r#"ȭ"#, r#"ȭ"#),
        (r#"Ȯ"#, r#"Ȯ"#, r#"Ȯ"#, r#"Ȯ"#, r#"Ȯ"#),
        (r#"ȯ"#, r#"ȯ"#, r#"ȯ"#, r#"ȯ"#, r#"ȯ"#),
        (r#"Ȱ"#, r#"Ȱ"#, r#"Ȱ"#, r#"Ȱ"#, r#"Ȱ"#),
        (r#"ȱ"#, r#"ȱ"#, r#"ȱ"#, r#"ȱ"#, r#"ȱ"#),
        (r#"Ȳ"#, r#"Ȳ"#, r#"Ȳ"#, r#"Ȳ"#, r#"Ȳ"#),
        (r#"ȳ"#, r#"ȳ"#, r#"ȳ"#, r#"ȳ"#, r#"ȳ"#),
        (r#"ʰ"#, r#"ʰ"#, r#"ʰ"#, r#"h"#, r#"h"#),
        (r#"ʱ"#, r#"ʱ"#, r#"ʱ"#, r#"ɦ"#, r#"ɦ"#),
        (r#"ʲ"#, r#"ʲ"#, r#"ʲ"#, r#"j"#, r#"j"#),
        (r#"ʳ"#, r#"ʳ"#, r#"ʳ"#, r#"r"#, r#"r"#),
        (r#"ʴ"#, r#"ʴ"#, r#"ʴ"#, r#"ɹ"#, r#"ɹ"#),
        (r#"ʵ"#, r#"ʵ"#, r#"ʵ"#, r#"ɻ"#, r#"ɻ"#),
        (r#"ʶ"#, r#"ʶ"#, r#"ʶ"#, r#"ʁ"#, r#"ʁ"#),
        (r#"ʷ"#, r#"ʷ"#, r#"ʷ"#, r#"w"#, r#"w"#),
        (r#"ʸ"#, r#"ʸ"#, r#"ʸ"#, r#"y"#, r#"y"#),
        (r#"˘"#, r#"˘"#, r#"˘"#, r#" ̆"#, r#" ̆"#),
        (r#"˙"#, r#"˙"#, r#"˙"#, r#" ̇"#, r#" ̇"#),
        (r#"˚"#, r#"˚"#, r#"˚"#, r#" ̊"#, r#" ̊"#),
        (r#"˛"#, r#"˛"#, r#"˛"#, r#" ̨"#, r#" ̨"#),
        (r#"˜"#, r#"˜"#, r#"˜"#, r#" ̃"#, r#" ̃"#),
        (r#"˝"#, r#"˝"#, r#"˝"#, r#" ̋"#, r#" ̋"#),
        (r#"ˠ"#, r#"ˠ"#, r#"ˠ"#, r#"ɣ"#, r#"ɣ"#),
        (r#"ˡ"#, r#"ˡ"#, r#"ˡ"#, r#"l"#, r#"l"#),
        (r#"ˢ"#, r#"ˢ"#, r#"ˢ"#, r#"s"#, r#"s"#),
        (r#"ˣ"#, r#"ˣ"#, r#"ˣ"#, r#"x"#, r#"x"#),
        (r#"ˤ"#, r#"ˤ"#, r#"ˤ"#, r#"ʕ"#, r#"ʕ"#),
        (r#"̀"#, r#"̀"#, r#"̀"#, r#"̀"#, r#"̀"#),
        (r#"́"#, r#"́"#, r#"́"#, r#"́"#, r#"́"#),
        (r#"̓"#, r#"̓"#, r#"̓"#, r#"̓"#, r#"̓"#),
        (r#"̈́"#, r#"̈́"#, r#"̈́"#, r#"̈́"#, r#"̈́"#),
        (r#"ʹ"#, r#"ʹ"#, r#"ʹ"#, r#"ʹ"#, r#"ʹ"#),
        (r#"ͺ"#, r#"ͺ"#, r#"ͺ"#, r#" ͅ"#, r#" ͅ"#),
        (r#";"#, r#";"#, r#";"#, r#";"#, r#";"#),
        (r#"΄"#, r#"΄"#, r#"΄"#, r#" ́"#, r#" ́"#),
        (r#"΅"#, r#"΅"#, r#"΅"#, r#" ̈́"#, r#" ̈́"#),
        (r#"Ά"#, r#"Ά"#, r#"Ά"#, r#"Ά"#, r#"Ά"#),
        (r#"·"#, r#"·"#, r#"·"#, r#"·"#, r#"·"#),
        (r#"Έ"#, r#"Έ"#, r#"Έ"#, r#"Έ"#, r#"Έ"#),
        (r#"Ή"#, r#"Ή"#, r#"Ή"#, r#"Ή"#, r#"Ή"#),
        (r#"Ί"#, r#"Ί"#, r#"Ί"#, r#"Ί"#, r#"Ί"#),
        (r#"Ό"#, r#"Ό"#, r#"Ό"#, r#"Ό"#, r#"Ό"#),
        (r#"Ύ"#, r#"Ύ"#, r#"Ύ"#, r#"Ύ"#, r#"Ύ"#),
        (r#"Ώ"#, r#"Ώ"#, r#"Ώ"#, r#"Ώ"#, r#"Ώ"#),
        (r#"ΐ"#, r#"ΐ"#, r#"ΐ"#, r#"ΐ"#, r#"ΐ"#),
        (r#"Ϊ"#, r#"Ϊ"#, r#"Ϊ"#, r#"Ϊ"#, r#"Ϊ"#),
        (r#"Ϋ"#, r#"Ϋ"#, r#"Ϋ"#, r#"Ϋ"#, r#"Ϋ"#),
        (r#"ά"#, r#"ά"#, r#"ά"#, r#"ά"#, r#"ά"#),
        (r#"έ"#, r#"έ"#, r#"έ"#, r#"έ"#, r#"έ"#),
        (r#"ή"#, r#"ή"#, r#"ή"#, r#"ή"#, r#"ή"#),
        (r#"ί"#, r#"ί"#, r#"ί"#, r#"ί"#, r#"ί"#),
        (r#"ΰ"#, r#"ΰ"#, r#"ΰ"#, r#"ΰ"#, r#"ΰ"#),
        (r#"ϊ"#, r#"ϊ"#, r#"ϊ"#, r#"ϊ"#, r#"ϊ"#),
        (r#"ϋ"#, r#"ϋ"#, r#"ϋ"#, r#"ϋ"#, r#"ϋ"#),
        (r#"ό"#, r#"ό"#, r#"ό"#, r#"ό"#, r#"ό"#),
        (r#"ύ"#, r#"ύ"#, r#"ύ"#, r#"ύ"#, r#"ύ"#),
        (r#"ώ"#, r#"ώ"#, r#"ώ"#, r#"ώ"#, r#"ώ"#),
        (r#"ϐ"#, r#"ϐ"#, r#"ϐ"#, r#"β"#, r#"β"#),
        (r#"ϑ"#, r#"ϑ"#, r#"ϑ"#, r#"θ"#, r#"θ"#),
        (r#"ϒ"#, r#"ϒ"#, r#"ϒ"#, r#"Υ"#, r#"Υ"#),
        (r#"ϓ"#, r#"ϓ"#, r#"ϓ"#, r#"Ύ"#, r#"Ύ"#),
        (r#"ϔ"#, r#"ϔ"#, r#"ϔ"#, r#"Ϋ"#, r#"Ϋ"#),
        (r#"ϕ"#, r#"ϕ"#, r#"ϕ"#, r#"φ"#, r#"φ"#),
        (r#"ϖ"#, r#"ϖ"#, r#"ϖ"#, r#"π"#, r#"π"#),
        (r#"ϰ"#, r#"ϰ"#, r#"ϰ"#, r#"κ"#, r#"κ"#),
        (r#"ϱ"#, r#"ϱ"#, r#"ϱ"#, r#"ρ"#, r#"ρ"#),
        (r#"ϲ"#, r#"ϲ"#, r#"ϲ"#, r#"ς"#, r#"ς"#),
        (r#"ϴ"#, r#"ϴ"#, r#"ϴ"#, r#"Θ"#, r#"Θ"#),
        (r#"ϵ"#, r#"ϵ"#, r#"ϵ"#, r#"ε"#, r#"ε"#),
        (r#"Ϲ"#, r#"Ϲ"#, r#"Ϲ"#, r#"Σ"#, r#"Σ"#),
        (r#"Ѐ"#, r#"Ѐ"#, r#"Ѐ"#, r#"Ѐ"#, r#"Ѐ"#),
        (r#"Ё"#, r#"Ё"#, r#"Ё"#, r#"Ё"#, r#"Ё"#),
        (r#"Ѓ"#, r#"Ѓ"#, r#"Ѓ"#, r#"Ѓ"#, r#"Ѓ"#),
        (r#"Ї"#, r#"Ї"#, r#"Ї"#, r#"Ї"#, r#"Ї"#),
        (r#"Ќ"#, r#"Ќ"#, r#"Ќ"#, r#"Ќ"#, r#"Ќ"#),
        (r#"Ѝ"#, r#"Ѝ"#, r#"Ѝ"#, r#"Ѝ"#, r#"Ѝ"#),
        (r#"Ў"#, r#"Ў"#, r#"Ў"#, r#"Ў"#, r#"Ў"#),
        (r#"Й"#, r#"Й"#, r#"Й"#, r#"Й"#, r#"Й"#),
        (r#"й"#, r#"й"#, r#"й"#, r#"й"#, r#"й"#),
        (r#"ѐ"#, r#"ѐ"#, r#"ѐ"#, r#"ѐ"#, r#"ѐ"#),
        (r#"ё"#, r#"ё"#, r#"ё"#, r#"ё"#, r#"ё"#),
        (r#"ѓ"#, r#"ѓ"#, r#"ѓ"#, r#"ѓ"#, r#"ѓ"#),
        (r#"ї"#, r#"ї"#, r#"ї"#, r#"ї"#, r#"ї"#),
        (r#"ќ"#, r#"ќ"#, r#"ќ"#, r#"ќ"#, r#"ќ"#),
        (r#"ѝ"#, r#"ѝ"#, r#"ѝ"#, r#"ѝ"#, r#"ѝ"#),
        (r#"ў"#, r#"ў"#, r#"ў"#, r#"ў"#, r#"ў"#),
        (r#"Ѷ"#, r#"Ѷ"#, r#"Ѷ"#, r#"Ѷ"#, r#"Ѷ"#),
        (r#"ѷ"#, r#"ѷ"#, r#"ѷ"#, r#"ѷ"#, r#"ѷ"#),
        (r#"Ӂ"#, r#"Ӂ"#, r#"Ӂ"#, r#"Ӂ"#, r#"Ӂ"#),
        (r#"ӂ"#, r#"ӂ"#, r#"ӂ"#, r#"ӂ"#, r#"ӂ"#),
        (r#"Ӑ"#, r#"Ӑ"#, r#"Ӑ"#, r#"Ӑ"#, r#"Ӑ"#),
        (r#"ӑ"#, r#"ӑ"#, r#"ӑ"#, r#"ӑ"#, r#"ӑ"#),
        (r#"Ӓ"#, r#"Ӓ"#, r#"Ӓ"#, r#"Ӓ"#, r#"Ӓ"#),
        (r#"ӓ"#, r#"ӓ"#, r#"ӓ"#, r#"ӓ"#, r#"ӓ"#),
        (r#"Ӗ"#, r#"Ӗ"#, r#"Ӗ"#, r#"Ӗ"#, r#"Ӗ"#),
        (r#"ӗ"#, r#"ӗ"#, r#"ӗ"#, r#"ӗ"#, r#"ӗ"#),
        (r#"Ӛ"#, r#"Ӛ"#, r#"Ӛ"#, r#"Ӛ"#, r#"Ӛ"#),
        (r#"ӛ"#, r#"ӛ"#, r#"ӛ"#, r#"ӛ"#, r#"ӛ"#),
        (r#"Ӝ"#, r#"Ӝ"#, r#"Ӝ"#, r#"Ӝ"#, r#"Ӝ"#),
        (r#"ӝ"#, r#"ӝ"#, r#"ӝ"#, r#"ӝ"#, r#"ӝ"#),
        (r#"Ӟ"#, r#"Ӟ"#, r#"Ӟ"#, r#"Ӟ"#, r#"Ӟ"#),
        (r#"ӟ"#, r#"ӟ"#, r#"ӟ"#, r#"ӟ"#, r#"ӟ"#),
        (r#"Ӣ"#, r#"Ӣ"#, r#"Ӣ"#, r#"Ӣ"#, r#"Ӣ"#),
        (r#"ӣ"#, r#"ӣ"#, r#"ӣ"#, r#"ӣ"#, r#"ӣ"#),
        (r#"Ӥ"#, r#"Ӥ"#, r#"Ӥ"#, r#"Ӥ"#, r#"Ӥ"#),
        (r#"ӥ"#, r#"ӥ"#, r#"ӥ"#, r#"ӥ"#, r#"ӥ"#),
        (r#"Ӧ"#, r#"Ӧ"#, r#"Ӧ"#, r#"Ӧ"#, r#"Ӧ"#),
        (r#"ӧ"#, r#"ӧ"#, r#"ӧ"#, r#"ӧ"#, r#"ӧ"#),
        (r#"Ӫ"#, r#"Ӫ"#, r#"Ӫ"#, r#"Ӫ"#, r#"Ӫ"#),
        (r#"ӫ"#, r#"ӫ"#, r#"ӫ"#, r#"ӫ"#, r#"ӫ"#),
        (r#"Ӭ"#, r#"Ӭ"#, r#"Ӭ"#, r#"Ӭ"#, r#"Ӭ"#),
        (r#"ӭ"#, r#"ӭ"#, r#"ӭ"#, r#"ӭ"#, r#"ӭ"#),
        (r#"Ӯ"#, r#"Ӯ"#, r#"Ӯ"#, r#"Ӯ"#, r#"Ӯ"#),
        (r#"ӯ"#, r#"ӯ"#, r#"ӯ"#, r#"ӯ"#, r#"ӯ"#),
        (r#"Ӱ"#, r#"Ӱ"#, r#"Ӱ"#, r#"Ӱ"#, r#"Ӱ"#),
        (r#"ӱ"#, r#"ӱ"#, r#"ӱ"#, r#"ӱ"#, r#"ӱ"#),
        (r#"Ӳ"#, r#"Ӳ"#, r#"Ӳ"#, r#"Ӳ"#, r#"Ӳ"#),
        (r#"ӳ"#, r#"ӳ"#, r#"ӳ"#, r#"ӳ"#, r#"ӳ"#),
        (r#"Ӵ"#, r#"Ӵ"#, r#"Ӵ"#, r#"Ӵ"#, r#"Ӵ"#),
        (r#"ӵ"#, r#"ӵ"#, r#"ӵ"#, r#"ӵ"#, r#"ӵ"#),
        (r#"Ӹ"#, r#"Ӹ"#, r#"Ӹ"#, r#"Ӹ"#, r#"Ӹ"#),
        (r#"ӹ"#, r#"ӹ"#, r#"ӹ"#, r#"ӹ"#, r#"ӹ"#),
        (r#"և"#, r#"և"#, r#"և"#, r#"եւ"#, r#"եւ"#),
        (r#"آ"#, r#"آ"#, r#"آ"#, r#"آ"#, r#"آ"#),
        (r#"أ"#, r#"أ"#, r#"أ"#, r#"أ"#, r#"أ"#),
        (r#"ؤ"#, r#"ؤ"#, r#"ؤ"#, r#"ؤ"#, r#"ؤ"#),
        (r#"إ"#, r#"إ"#, r#"إ"#, r#"إ"#, r#"إ"#),
        (r#"ئ"#, r#"ئ"#, r#"ئ"#, r#"ئ"#, r#"ئ"#),
        (r#"ٵ"#, r#"ٵ"#, r#"ٵ"#, r#"اٴ"#, r#"اٴ"#),
        (r#"ٶ"#, r#"ٶ"#, r#"ٶ"#, r#"وٴ"#, r#"وٴ"#),
        (r#"ٷ"#, r#"ٷ"#, r#"ٷ"#, r#"ۇٴ"#, r#"ۇٴ"#),
        (r#"ٸ"#, r#"ٸ"#, r#"ٸ"#, r#"يٴ"#, r#"يٴ"#),
        (r#"ۀ"#, r#"ۀ"#, r#"ۀ"#, r#"ۀ"#, r#"ۀ"#),
        (r#"ۂ"#, r#"ۂ"#, r#"ۂ"#, r#"ۂ"#, r#"ۂ"#),
        (r#"ۓ"#, r#"ۓ"#, r#"ۓ"#, r#"ۓ"#, r#"ۓ"#),
        (r#"ऩ"#, r#"ऩ"#, r#"ऩ"#, r#"ऩ"#, r#"ऩ"#),
        (r#"ऱ"#, r#"ऱ"#, r#"ऱ"#, r#"ऱ"#, r#"ऱ"#),
        (r#"ऴ"#, r#"ऴ"#, r#"ऴ"#, r#"ऴ"#, r#"ऴ"#),
        (r#"क़"#, r#"क़"#, r#"क़"#, r#"क़"#, r#"क़"#),
        (r#"ख़"#, r#"ख़"#, r#"ख़"#, r#"ख़"#, r#"ख़"#),
        (r#"ग़"#, r#"ग़"#, r#"ग़"#, r#"ग़"#, r#"ग़"#),
        (r#"ज़"#, r#"ज़"#, r#"ज़"#, r#"ज़"#, r#"ज़"#),
        (r#"ड़"#, r#"ड़"#, r#"ड़"#, r#"ड़"#, r#"ड़"#),
        (r#"ढ़"#, r#"ढ़"#, r#"ढ़"#, r#"ढ़"#, r#"ढ़"#),
        (r#"फ़"#, r#"फ़"#, r#"फ़"#, r#"फ़"#, r#"फ़"#),
        (r#"य़"#, r#"य़"#, r#"य़"#, r#"य़"#, r#"य़"#),
        (r#"ো"#, r#"ো"#, r#"ো"#, r#"ো"#, r#"ো"#),
        (r#"ৌ"#, r#"ৌ"#, r#"ৌ"#, r#"ৌ"#, r#"ৌ"#),
        (r#"ড়"#, r#"ড়"#, r#"ড়"#, r#"ড়"#, r#"ড়"#),
        (r#"ঢ়"#, r#"ঢ়"#, r#"ঢ়"#, r#"ঢ়"#, r#"ঢ়"#),
        (r#"য়"#, r#"য়"#, r#"য়"#, r#"য়"#, r#"য়"#),
        (r#"ਲ਼"#, r#"ਲ਼"#, r#"ਲ਼"#, r#"ਲ਼"#, r#"ਲ਼"#),
        (r#"ਸ਼"#, r#"ਸ਼"#, r#"ਸ਼"#, r#"ਸ਼"#, r#"ਸ਼"#),
        (r#"ਖ਼"#, r#"ਖ਼"#, r#"ਖ਼"#, r#"ਖ਼"#, r#"ਖ਼"#),
        (r#"ਗ਼"#, r#"ਗ਼"#, r#"ਗ਼"#, r#"ਗ਼"#, r#"ਗ਼"#),
        (r#"ਜ਼"#, r#"ਜ਼"#, r#"ਜ਼"#, r#"ਜ਼"#, r#"ਜ਼"#),
        (r#"ਫ਼"#, r#"ਫ਼"#, r#"ਫ਼"#, r#"ਫ਼"#, r#"ਫ਼"#),
        (r#"ୈ"#, r#"ୈ"#, r#"ୈ"#, r#"ୈ"#, r#"ୈ"#),
        (r#"ୋ"#, r#"ୋ"#, r#"ୋ"#, r#"ୋ"#, r#"ୋ"#),
        (r#"ୌ"#, r#"ୌ"#, r#"ୌ"#, r#"ୌ"#, r#"ୌ"#),
        (r#"ଡ଼"#, r#"ଡ଼"#, r#"ଡ଼"#, r#"ଡ଼"#, r#"ଡ଼"#),
        (r#"ଢ଼"#, r#"ଢ଼"#, r#"ଢ଼"#, r#"ଢ଼"#, r#"ଢ଼"#),
        (r#"ஔ"#, r#"ஔ"#, r#"ஔ"#, r#"ஔ"#, r#"ஔ"#),
        (r#"ொ"#, r#"ொ"#, r#"ொ"#, r#"ொ"#, r#"ொ"#),
        (r#"ோ"#, r#"ோ"#, r#"ோ"#, r#"ோ"#, r#"ோ"#),
        (r#"ௌ"#, r#"ௌ"#, r#"ௌ"#, r#"ௌ"#, r#"ௌ"#),
        (r#"ై"#, r#"ై"#, r#"ై"#, r#"ై"#, r#"ై"#),
        (r#"ೀ"#, r#"ೀ"#, r#"ೀ"#, r#"ೀ"#, r#"ೀ"#),
        (r#"ೇ"#, r#"ೇ"#, r#"ೇ"#, r#"ೇ"#, r#"ೇ"#),
        (r#"ೈ"#, r#"ೈ"#, r#"ೈ"#, r#"ೈ"#, r#"ೈ"#),
        (r#"ೊ"#, r#"ೊ"#, r#"ೊ"#, r#"ೊ"#, r#"ೊ"#),
        (r#"ೋ"#, r#"ೋ"#, r#"ೋ"#, r#"ೋ"#, r#"ೋ"#),
        (r#"ൊ"#, r#"ൊ"#, r#"ൊ"#, r#"ൊ"#, r#"ൊ"#),
        (r#"ോ"#, r#"ോ"#, r#"ോ"#, r#"ോ"#, r#"ോ"#),
        (r#"ൌ"#, r#"ൌ"#, r#"ൌ"#, r#"ൌ"#, r#"ൌ"#),
        (r#"ේ"#, r#"ේ"#, r#"ේ"#, r#"ේ"#, r#"ේ"#),
        (r#"ො"#, r#"ො"#, r#"ො"#, r#"ො"#, r#"ො"#),
        (r#"ෝ"#, r#"ෝ"#, r#"ෝ"#, r#"ෝ"#, r#"ෝ"#),
        (r#"ෞ"#, r#"ෞ"#, r#"ෞ"#, r#"ෞ"#, r#"ෞ"#),
        (r#"ำ"#, r#"ำ"#, r#"ำ"#, r#"ํา"#, r#"ํา"#),
        (r#"ຳ"#, r#"ຳ"#, r#"ຳ"#, r#"ໍາ"#, r#"ໍາ"#),
        (r#"ໜ"#, r#"ໜ"#, r#"ໜ"#, r#"ຫນ"#, r#"ຫນ"#),
        (r#"ໝ"#, r#"ໝ"#, r#"ໝ"#, r#"ຫມ"#, r#"ຫມ"#),
        (r#"༌"#, r#"༌"#, r#"༌"#, r#"་"#, r#"་"#),
        (r#"གྷ"#, r#"གྷ"#, r#"གྷ"#, r#"གྷ"#, r#"གྷ"#),
        (r#"ཌྷ"#, r#"ཌྷ"#, r#"ཌྷ"#, r#"ཌྷ"#, r#"ཌྷ"#),
        (r#"དྷ"#, r#"དྷ"#, r#"དྷ"#, r#"དྷ"#, r#"དྷ"#),
        (r#"བྷ"#, r#"བྷ"#, r#"བྷ"#, r#"བྷ"#, r#"བྷ"#),
        (r#"ཛྷ"#, r#"ཛྷ"#, r#"ཛྷ"#, r#"ཛྷ"#, r#"ཛྷ"#),
        (r#"ཀྵ"#, r#"ཀྵ"#, r#"ཀྵ"#, r#"ཀྵ"#, r#"ཀྵ"#),
        (r#"ཱི"#, r#"ཱི"#, r#"ཱི"#, r#"ཱི"#, r#"ཱི"#),
        (r#"ཱུ"#, r#"ཱུ"#, r#"ཱུ"#, r#"ཱུ"#, r#"ཱུ"#),
        (r#"ྲྀ"#, r#"ྲྀ"#, r#"ྲྀ"#, r#"ྲྀ"#, r#"ྲྀ"#),
        (r#"ཷ"#, r#"ཷ"#, r#"ཷ"#, r#"ྲཱྀ"#, r#"ྲཱྀ"#),
        (r#"ླྀ"#, r#"ླྀ"#, r#"ླྀ"#, r#"ླྀ"#, r#"ླྀ"#),
        (r#"ཹ"#, r#"ཹ"#, r#"ཹ"#, r#"ླཱྀ"#, r#"ླཱྀ"#),
        (r#"ཱྀ"#, r#"ཱྀ"#, r#"ཱྀ"#, r#"ཱྀ"#, r#"ཱྀ"#),
        (r#"ྒྷ"#, r#"ྒྷ"#, r#"ྒྷ"#, r#"ྒྷ"#, r#"ྒྷ"#),
        (r#"ྜྷ"#, r#"ྜྷ"#, r#"ྜྷ"#, r#"ྜྷ"#, r#"ྜྷ"#),
        (r#"ྡྷ"#, r#"ྡྷ"#, r#"ྡྷ"#, r#"ྡྷ"#, r#"ྡྷ"#),
        (r#"ྦྷ"#, r#"ྦྷ"#, r#"ྦྷ"#, r#"ྦྷ"#, r#"ྦྷ"#),
        (r#"ྫྷ"#, r#"ྫྷ"#, r#"ྫྷ"#, r#"ྫྷ"#, r#"ྫྷ"#),
        (r#"ྐྵ"#, r#"ྐྵ"#, r#"ྐྵ"#, r#"ྐྵ"#, r#"ྐྵ"#),
        (r#"ဦ"#, r#"ဦ"#, r#"ဦ"#, r#"ဦ"#, r#"ဦ"#),
        (r#"ჼ"#, r#"ჼ"#, r#"ჼ"#, r#"ნ"#, r#"ნ"#),
        (r#"ᬆ"#, r#"ᬆ"#, r#"ᬆ"#, r#"ᬆ"#, r#"ᬆ"#),
        (r#"ᬈ"#, r#"ᬈ"#, r#"ᬈ"#, r#"ᬈ"#, r#"ᬈ"#),
        (r#"ᬊ"#, r#"ᬊ"#, r#"ᬊ"#, r#"ᬊ"#, r#"ᬊ"#),
        (r#"ᬌ"#, r#"ᬌ"#, r#"ᬌ"#, r#"ᬌ"#, r#"ᬌ"#),
        (r#"ᬎ"#, r#"ᬎ"#, r#"ᬎ"#, r#"ᬎ"#, r#"ᬎ"#),
        (r#"ᬒ"#, r#"ᬒ"#, r#"ᬒ"#, r#"ᬒ"#, r#"ᬒ"#),
        (r#"ᬻ"#, r#"ᬻ"#, r#"ᬻ"#, r#"ᬻ"#, r#"ᬻ"#),
        (r#"ᬽ"#, r#"ᬽ"#, r#"ᬽ"#, r#"ᬽ"#, r#"ᬽ"#),
        (r#"ᭀ"#, r#"ᭀ"#, r#"ᭀ"#, r#"ᭀ"#, r#"ᭀ"#),
        (r#"ᭁ"#, r#"ᭁ"#, r#"ᭁ"#, r#"ᭁ"#, r#"ᭁ"#),
        (r#"ᭃ"#, r#"ᭃ"#, r#"ᭃ"#, r#"ᭃ"#, r#"ᭃ"#),
        (r#"ᴬ"#, r#"ᴬ"#, r#"ᴬ"#, r#"A"#, r#"A"#),
        (r#"ᴭ"#, r#"ᴭ"#, r#"ᴭ"#, r#"Æ"#, r#"Æ"#),
        (r#"ᴮ"#, r#"ᴮ"#, r#"ᴮ"#, r#"B"#, r#"B"#),
        (r#"ᴰ"#, r#"ᴰ"#, r#"ᴰ"#, r#"D"#, r#"D"#),
        (r#"ᴱ"#, r#"ᴱ"#, r#"ᴱ"#, r#"E"#, r#"E"#),
        (r#"ᴲ"#, r#"ᴲ"#, r#"ᴲ"#, r#"Ǝ"#, r#"Ǝ"#),
        (r#"ᴳ"#, r#"ᴳ"#, r#"ᴳ"#, r#"G"#, r#"G"#),
        (r#"ᴴ"#, r#"ᴴ"#, r#"ᴴ"#, r#"H"#, r#"H"#),
        (r#"ᴵ"#, r#"ᴵ"#, r#"ᴵ"#, r#"I"#, r#"I"#),
        (r#"ᴶ"#, r#"ᴶ"#, r#"ᴶ"#, r#"J"#, r#"J"#),
        (r#"ᴷ"#, r#"ᴷ"#, r#"ᴷ"#, r#"K"#, r#"K"#),
        (r#"ᴸ"#, r#"ᴸ"#, r#"ᴸ"#, r#"L"#, r#"L"#),
        (r#"ᴹ"#, r#"ᴹ"#, r#"ᴹ"#, r#"M"#, r#"M"#),
        (r#"ᴺ"#, r#"ᴺ"#, r#"ᴺ"#, r#"N"#, r#"N"#),
        (r#"ᴼ"#, r#"ᴼ"#, r#"ᴼ"#, r#"O"#, r#"O"#),
        (r#"ᴽ"#, r#"ᴽ"#, r#"ᴽ"#, r#"Ȣ"#, r#"Ȣ"#),
        (r#"ᴾ"#, r#"ᴾ"#, r#"ᴾ"#, r#"P"#, r#"P"#),
        (r#"ᴿ"#, r#"ᴿ"#, r#"ᴿ"#, r#"R"#, r#"R"#),
        (r#"ᵀ"#, r#"ᵀ"#, r#"ᵀ"#, r#"T"#, r#"T"#),
        (r#"ᵁ"#, r#"ᵁ"#, r#"ᵁ"#, r#"U"#, r#"U"#),
        (r#"ᵂ"#, r#"ᵂ"#, r#"ᵂ"#, r#"W"#, r#"W"#),
        (r#"ᵃ"#, r#"ᵃ"#, r#"ᵃ"#, r#"a"#, r#"a"#),
        (r#"ᵄ"#, r#"ᵄ"#, r#"ᵄ"#, r#"ɐ"#, r#"ɐ"#),
        (r#"ᵅ"#, r#"ᵅ"#, r#"ᵅ"#, r#"ɑ"#, r#"ɑ"#),
        (r#"ᵆ"#, r#"ᵆ"#, r#"ᵆ"#, r#"ᴂ"#, r#"ᴂ"#),
        (r#"ᵇ"#, r#"ᵇ"#, r#"ᵇ"#, r#"b"#, r#"b"#),
        (r#"ᵈ"#, r#"ᵈ"#, r#"ᵈ"#, r#"d"#, r#"d"#),
        (r#"ᵉ"#, r#"ᵉ"#, r#"ᵉ"#, r#"e"#, r#"e"#),
        (r#"ᵊ"#, r#"ᵊ"#, r#"ᵊ"#, r#"ə"#, r#"ə"#),
        (r#"ᵋ"#, r#"ᵋ"#, r#"ᵋ"#, r#"ɛ"#, r#"ɛ"#),
        (r#"ᵌ"#, r#"ᵌ"#, r#"ᵌ"#, r#"ɜ"#, r#"ɜ"#),
        (r#"ᵍ"#, r#"ᵍ"#, r#"ᵍ"#, r#"g"#, r#"g"#),
        (r#"ᵏ"#, r#"ᵏ"#, r#"ᵏ"#, r#"k"#, r#"k"#),
        (r#"ᵐ"#, r#"ᵐ"#, r#"ᵐ"#, r#"m"#, r#"m"#),
        (r#"ᵑ"#, r#"ᵑ"#, r#"ᵑ"#, r#"ŋ"#, r#"ŋ"#),
        (r#"ᵒ"#, r#"ᵒ"#, r#"ᵒ"#, r#"o"#, r#"o"#),
        (r#"ᵓ"#, r#"ᵓ"#, r#"ᵓ"#, r#"ɔ"#, r#"ɔ"#),
        (r#"ᵔ"#, r#"ᵔ"#, r#"ᵔ"#, r#"ᴖ"#, r#"ᴖ"#),
        (r#"ᵕ"#, r#"ᵕ"#, r#"ᵕ"#, r#"ᴗ"#, r#"ᴗ"#),
        (r#"ᵖ"#, r#"ᵖ"#, r#"ᵖ"#, r#"p"#, r#"p"#),
        (r#"ᵗ"#, r#"ᵗ"#, r#"ᵗ"#, r#"t"#, r#"t"#),
        (r#"ᵘ"#, r#"ᵘ"#, r#"ᵘ"#, r#"u"#, r#"u"#),
        (r#"ᵙ"#, r#"ᵙ"#, r#"ᵙ"#, r#"ᴝ"#, r#"ᴝ"#),
        (r#"ᵚ"#, r#"ᵚ"#, r#"ᵚ"#, r#"ɯ"#, r#"ɯ"#),
        (r#"ᵛ"#, r#"ᵛ"#, r#"ᵛ"#, r#"v"#, r#"v"#),
        (r#"ᵜ"#, r#"ᵜ"#, r#"ᵜ"#, r#"ᴥ"#, r#"ᴥ"#),
        (r#"ᵝ"#, r#"ᵝ"#, r#"ᵝ"#, r#"β"#, r#"β"#),
        (r#"ᵞ"#, r#"ᵞ"#, r#"ᵞ"#, r#"γ"#, r#"γ"#),
        (r#"ᵟ"#, r#"ᵟ"#, r#"ᵟ"#, r#"δ"#, r#"δ"#),
        (r#"ᵠ"#, r#"ᵠ"#, r#"ᵠ"#, r#"φ"#, r#"φ"#),
        (r#"ᵡ"#, r#"ᵡ"#, r#"ᵡ"#, r#"χ"#, r#"χ"#),
        (r#"ᵢ"#, r#"ᵢ"#, r#"ᵢ"#, r#"i"#, r#"i"#),
        (r#"ᵣ"#, r#"ᵣ"#, r#"ᵣ"#, r#"r"#, r#"r"#),
        (r#"ᵤ"#, r#"ᵤ"#, r#"ᵤ"#, r#"u"#, r#"u"#),
        (r#"ᵥ"#, r#"ᵥ"#, r#"ᵥ"#, r#"v"#, r#"v"#),
        (r#"ᵦ"#, r#"ᵦ"#, r#"ᵦ"#, r#"β"#, r#"β"#),
        (r#"ᵧ"#, r#"ᵧ"#, r#"ᵧ"#, r#"γ"#, r#"γ"#),
        (r#"ᵨ"#, r#"ᵨ"#, r#"ᵨ"#, r#"ρ"#, r#"ρ"#),
        (r#"ᵩ"#, r#"ᵩ"#, r#"ᵩ"#, r#"φ"#, r#"φ"#),
        (r#"ᵪ"#, r#"ᵪ"#, r#"ᵪ"#, r#"χ"#, r#"χ"#),
        (r#"ᵸ"#, r#"ᵸ"#, r#"ᵸ"#, r#"н"#, r#"н"#),
        (r#"ᶛ"#, r#"ᶛ"#, r#"ᶛ"#, r#"ɒ"#, r#"ɒ"#),
        (r#"ᶜ"#, r#"ᶜ"#, r#"ᶜ"#, r#"c"#, r#"c"#),
        (r#"ᶝ"#, r#"ᶝ"#, r#"ᶝ"#, r#"ɕ"#, r#"ɕ"#),
        (r#"ᶞ"#, r#"ᶞ"#, r#"ᶞ"#, r#"ð"#, r#"ð"#),
        (r#"ᶟ"#, r#"ᶟ"#, r#"ᶟ"#, r#"ɜ"#, r#"ɜ"#),
        (r#"ᶠ"#, r#"ᶠ"#, r#"ᶠ"#, r#"f"#, r#"f"#),
        (r#"ᶡ"#, r#"ᶡ"#, r#"ᶡ"#, r#"ɟ"#, r#"ɟ"#),
        (r#"ᶢ"#, r#"ᶢ"#, r#"ᶢ"#, r#"ɡ"#, r#"ɡ"#),
        (r#"ᶣ"#, r#"ᶣ"#, r#"ᶣ"#, r#"ɥ"#, r#"ɥ"#),
        (r#"ᶤ"#, r#"ᶤ"#, r#"ᶤ"#, r#"ɨ"#, r#"ɨ"#),
        (r#"ᶥ"#, r#"ᶥ"#, r#"ᶥ"#, r#"ɩ"#, r#"ɩ"#),
        (r#"ᶦ"#, r#"ᶦ"#, r#"ᶦ"#, r#"ɪ"#, r#"ɪ"#),
        (r#"ᶧ"#, r#"ᶧ"#, r#"ᶧ"#, r#"ᵻ"#, r#"ᵻ"#),
        (r#"ᶨ"#, r#"ᶨ"#, r#"ᶨ"#, r#"ʝ"#, r#"ʝ"#),
        (r#"ᶩ"#, r#"ᶩ"#, r#"ᶩ"#, r#"ɭ"#, r#"ɭ"#),
        (r#"ᶪ"#, r#"ᶪ"#, r#"ᶪ"#, r#"ᶅ"#, r#"ᶅ"#),
        (r#"ᶫ"#, r#"ᶫ"#, r#"ᶫ"#, r#"ʟ"#, r#"ʟ"#),
        (r#"ᶬ"#, r#"ᶬ"#, r#"ᶬ"#, r#"ɱ"#, r#"ɱ"#),
        (r#"ᶭ"#, r#"ᶭ"#, r#"ᶭ"#, r#"ɰ"#, r#"ɰ"#),
        (r#"ᶮ"#, r#"ᶮ"#, r#"ᶮ"#, r#"ɲ"#, r#"ɲ"#),
        (r#"ᶯ"#, r#"ᶯ"#, r#"ᶯ"#, r#"ɳ"#, r#"ɳ"#),
        (r#"ᶰ"#, r#"ᶰ"#, r#"ᶰ"#, r#"ɴ"#, r#"ɴ"#),
        (r#"ᶱ"#, r#"ᶱ"#, r#"ᶱ"#, r#"ɵ"#, r#"ɵ"#),
        (r#"ᶲ"#, r#"ᶲ"#, r#"ᶲ"#, r#"ɸ"#, r#"ɸ"#),
        (r#"ᶳ"#, r#"ᶳ"#, r#"ᶳ"#, r#"ʂ"#, r#"ʂ"#),
        (r#"ᶴ"#, r#"ᶴ"#, r#"ᶴ"#, r#"ʃ"#, r#"ʃ"#),
        (r#"ᶵ"#, r#"ᶵ"#, r#"ᶵ"#, r#"ƫ"#, r#"ƫ"#),
        (r#"ᶶ"#, r#"ᶶ"#, r#"ᶶ"#, r#"ʉ"#, r#"ʉ"#),
        (r#"ᶷ"#, r#"ᶷ"#, r#"ᶷ"#, r#"ʊ"#, r#"ʊ"#),
        (r#"ᶸ"#, r#"ᶸ"#, r#"ᶸ"#, r#"ᴜ"#, r#"ᴜ"#),
        (r#"ᶹ"#, r#"ᶹ"#, r#"ᶹ"#, r#"ʋ"#, r#"ʋ"#),
        (r#"ᶺ"#, r#"ᶺ"#, r#"ᶺ"#, r#"ʌ"#, r#"ʌ"#),
        (r#"ᶻ"#, r#"ᶻ"#, r#"ᶻ"#, r#"z"#, r#"z"#),
        (r#"ᶼ"#, r#"ᶼ"#, r#"ᶼ"#, r#"ʐ"#, r#"ʐ"#),
        (r#"ᶽ"#, r#"ᶽ"#, r#"ᶽ"#, r#"ʑ"#, r#"ʑ"#),
        (r#"ᶾ"#, r#"ᶾ"#, r#"ᶾ"#, r#"ʒ"#, r#"ʒ"#),
        (r#"ᶿ"#, r#"ᶿ"#, r#"ᶿ"#, r#"θ"#, r#"θ"#),
        (r#"Ḁ"#, r#"Ḁ"#, r#"Ḁ"#, r#"Ḁ"#, r#"Ḁ"#),
        (r#"ḁ"#, r#"ḁ"#, r#"ḁ"#, r#"ḁ"#, r#"ḁ"#),
        (r#"Ḃ"#, r#"Ḃ"#, r#"Ḃ"#, r#"Ḃ"#, r#"Ḃ"#),
        (r#"ḃ"#, r#"ḃ"#, r#"ḃ"#, r#"ḃ"#, r#"ḃ"#),
        (r#"Ḅ"#, r#"Ḅ"#, r#"Ḅ"#, r#"Ḅ"#, r#"Ḅ"#),
        (r#"ḅ"#, r#"ḅ"#, r#"ḅ"#, r#"ḅ"#, r#"ḅ"#),
        (r#"Ḇ"#, r#"Ḇ"#, r#"Ḇ"#, r#"Ḇ"#, r#"Ḇ"#),
        (r#"ḇ"#, r#"ḇ"#, r#"ḇ"#, r#"ḇ"#, r#"ḇ"#),
        (r#"Ḉ"#, r#"Ḉ"#, r#"Ḉ"#, r#"Ḉ"#, r#"Ḉ"#),
        (r#"ḉ"#, r#"ḉ"#, r#"ḉ"#, r#"ḉ"#, r#"ḉ"#),
        (r#"Ḋ"#, r#"Ḋ"#, r#"Ḋ"#, r#"Ḋ"#, r#"Ḋ"#),
        (r#"ḋ"#, r#"ḋ"#, r#"ḋ"#, r#"ḋ"#, r#"ḋ"#),
        (r#"Ḍ"#, r#"Ḍ"#, r#"Ḍ"#, r#"Ḍ"#, r#"Ḍ"#),
        (r#"ḍ"#, r#"ḍ"#, r#"ḍ"#, r#"ḍ"#, r#"ḍ"#),
        (r#"Ḏ"#, r#"Ḏ"#, r#"Ḏ"#, r#"Ḏ"#, r#"Ḏ"#),
        (r#"ḏ"#, r#"ḏ"#, r#"ḏ"#, r#"ḏ"#, r#"ḏ"#),
        (r#"Ḑ"#, r#"Ḑ"#, r#"Ḑ"#, r#"Ḑ"#, r#"Ḑ"#),
        (r#"ḑ"#, r#"ḑ"#, r#"ḑ"#, r#"ḑ"#, r#"ḑ"#),
        (r#"Ḓ"#, r#"Ḓ"#, r#"Ḓ"#, r#"Ḓ"#, r#"Ḓ"#),
        (r#"ḓ"#, r#"ḓ"#, r#"ḓ"#, r#"ḓ"#, r#"ḓ"#),
        (r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#, r#"Ḕ"#),
        (r#"ḕ"#, r#"ḕ"#, r#"ḕ"#, r#"ḕ"#, r#"ḕ"#),
        (r#"Ḗ"#, r#"Ḗ"#, r#"Ḗ"#, r#"Ḗ"#, r#"Ḗ"#),
        (r#"ḗ"#, r#"ḗ"#, r#"ḗ"#, r#"ḗ"#, r#"ḗ"#),
        (r#"Ḙ"#, r#"Ḙ"#, r#"Ḙ"#, r#"Ḙ"#, r#"Ḙ"#),
        (r#"ḙ"#, r#"ḙ"#, r#"ḙ"#, r#"ḙ"#, r#"ḙ"#),
        (r#"Ḛ"#, r#"Ḛ"#, r#"Ḛ"#, r#"Ḛ"#, r#"Ḛ"#),
        (r#"ḛ"#, r#"ḛ"#, r#"ḛ"#, r#"ḛ"#, r#"ḛ"#),
        (r#"Ḝ"#, r#"Ḝ"#, r#"Ḝ"#, r#"Ḝ"#, r#"Ḝ"#),
        (r#"ḝ"#, r#"ḝ"#, r#"ḝ"#, r#"ḝ"#, r#"ḝ"#),
        (r#"Ḟ"#, r#"Ḟ"#, r#"Ḟ"#, r#"Ḟ"#, r#"Ḟ"#),
        (r#"ḟ"#, r#"ḟ"#, r#"ḟ"#, r#"ḟ"#, r#"ḟ"#),
        (r#"Ḡ"#, r#"Ḡ"#, r#"Ḡ"#, r#"Ḡ"#, r#"Ḡ"#),
        (r#"ḡ"#, r#"ḡ"#, r#"ḡ"#, r#"ḡ"#, r#"ḡ"#),
        (r#"Ḣ"#, r#"Ḣ"#, r#"Ḣ"#, r#"Ḣ"#, r#"Ḣ"#),
        (r#"ḣ"#, r#"ḣ"#, r#"ḣ"#, r#"ḣ"#, r#"ḣ"#),
        (r#"Ḥ"#, r#"Ḥ"#, r#"Ḥ"#, r#"Ḥ"#, r#"Ḥ"#),
        (r#"ḥ"#, r#"ḥ"#, r#"ḥ"#, r#"ḥ"#, r#"ḥ"#),
        (r#"Ḧ"#, r#"Ḧ"#, r#"Ḧ"#, r#"Ḧ"#, r#"Ḧ"#),
        (r#"ḧ"#, r#"ḧ"#, r#"ḧ"#, r#"ḧ"#, r#"ḧ"#),
        (r#"Ḩ"#, r#"Ḩ"#, r#"Ḩ"#, r#"Ḩ"#, r#"Ḩ"#),
        (r#"ḩ"#, r#"ḩ"#, r#"ḩ"#, r#"ḩ"#, r#"ḩ"#),
        (r#"Ḫ"#, r#"Ḫ"#, r#"Ḫ"#, r#"Ḫ"#, r#"Ḫ"#),
        (r#"ḫ"#, r#"ḫ"#, r#"ḫ"#, r#"ḫ"#, r#"ḫ"#),
        (r#"Ḭ"#, r#"Ḭ"#, r#"Ḭ"#, r#"Ḭ"#, r#"Ḭ"#),
        (r#"ḭ"#, r#"ḭ"#, r#"ḭ"#, r#"ḭ"#, r#"ḭ"#),
        (r#"Ḯ"#, r#"Ḯ"#, r#"Ḯ"#, r#"Ḯ"#, r#"Ḯ"#),
        (r#"ḯ"#, r#"ḯ"#, r#"ḯ"#, r#"ḯ"#, r#"ḯ"#),
        (r#"Ḱ"#, r#"Ḱ"#, r#"Ḱ"#, r#"Ḱ"#, r#"Ḱ"#),
        (r#"ḱ"#, r#"ḱ"#, r#"ḱ"#, r#"ḱ"#, r#"ḱ"#),
        (r#"Ḳ"#, r#"Ḳ"#, r#"Ḳ"#, r#"Ḳ"#, r#"Ḳ"#),
        (r#"ḳ"#, r#"ḳ"#, r#"ḳ"#, r#"ḳ"#, r#"ḳ"#),
        (r#"Ḵ"#, r#"Ḵ"#, r#"Ḵ"#, r#"Ḵ"#, r#"Ḵ"#),
        (r#"ḵ"#, r#"ḵ"#, r#"ḵ"#, r#"ḵ"#, r#"ḵ"#),
        (r#"Ḷ"#, r#"Ḷ"#, r#"Ḷ"#, r#"Ḷ"#, r#"Ḷ"#),
        (r#"ḷ"#, r#"ḷ"#, r#"ḷ"#, r#"ḷ"#, r#"ḷ"#),
        (r#"Ḹ"#, r#"Ḹ"#, r#"Ḹ"#, r#"Ḹ"#, r#"Ḹ"#),
        (r#"ḹ"#, r#"ḹ"#, r#"ḹ"#, r#"ḹ"#, r#"ḹ"#),
        (r#"Ḻ"#, r#"Ḻ"#, r#"Ḻ"#, r#"Ḻ"#, r#"Ḻ"#),
        (r#"ḻ"#, r#"ḻ"#, r#"ḻ"#, r#"ḻ"#, r#"ḻ"#),
        (r#"Ḽ"#, r#"Ḽ"#, r#"Ḽ"#, r#"Ḽ"#, r#"Ḽ"#),
        (r#"ḽ"#, r#"ḽ"#, r#"ḽ"#, r#"ḽ"#, r#"ḽ"#),
        (r#"Ḿ"#, r#"Ḿ"#, r#"Ḿ"#, r#"Ḿ"#, r#"Ḿ"#),
        (r#"ḿ"#, r#"ḿ"#, r#"ḿ"#, r#"ḿ"#, r#"ḿ"#),
        (r#"Ṁ"#, r#"Ṁ"#, r#"Ṁ"#, r#"Ṁ"#, r#"Ṁ"#),
        (r#"ṁ"#, r#"ṁ"#, r#"ṁ"#, r#"ṁ"#, r#"ṁ"#),
        (r#"Ṃ"#, r#"Ṃ"#, r#"Ṃ"#, r#"Ṃ"#, r#"Ṃ"#),
        (r#"ṃ"#, r#"ṃ"#, r#"ṃ"#, r#"ṃ"#, r#"ṃ"#),
        (r#"Ṅ"#, r#"Ṅ"#, r#"Ṅ"#, r#"Ṅ"#, r#"Ṅ"#),
        (r#"ṅ"#, r#"ṅ"#, r#"ṅ"#, r#"ṅ"#, r#"ṅ"#),
        (r#"Ṇ"#, r#"Ṇ"#, r#"Ṇ"#, r#"Ṇ"#, r#"Ṇ"#),
        (r#"ṇ"#, r#"ṇ"#, r#"ṇ"#, r#"ṇ"#, r#"ṇ"#),
        (r#"Ṉ"#, r#"Ṉ"#, r#"Ṉ"#, r#"Ṉ"#, r#"Ṉ"#),
        (r#"ṉ"#, r#"ṉ"#, r#"ṉ"#, r#"ṉ"#, r#"ṉ"#),
        (r#"Ṋ"#, r#"Ṋ"#, r#"Ṋ"#, r#"Ṋ"#, r#"Ṋ"#),
        (r#"ṋ"#, r#"ṋ"#, r#"ṋ"#, r#"ṋ"#, r#"ṋ"#),
        (r#"Ṍ"#, r#"Ṍ"#, r#"Ṍ"#, r#"Ṍ"#, r#"Ṍ"#),
        (r#"ṍ"#, r#"ṍ"#, r#"ṍ"#, r#"ṍ"#, r#"ṍ"#),
        (r#"Ṏ"#, r#"Ṏ"#, r#"Ṏ"#, r#"Ṏ"#, r#"Ṏ"#),
        (r#"ṏ"#, r#"ṏ"#, r#"ṏ"#, r#"ṏ"#, r#"ṏ"#),
        (r#"Ṑ"#, r#"Ṑ"#, r#"Ṑ"#, r#"Ṑ"#, r#"Ṑ"#),
        (r#"ṑ"#, r#"ṑ"#, r#"ṑ"#, r#"ṑ"#, r#"ṑ"#),
        (r#"Ṓ"#, r#"Ṓ"#, r#"Ṓ"#, r#"Ṓ"#, r#"Ṓ"#),
        (r#"ṓ"#, r#"ṓ"#, r#"ṓ"#, r#"ṓ"#, r#"ṓ"#),
        (r#"Ṕ"#, r#"Ṕ"#, r#"Ṕ"#, r#"Ṕ"#, r#"Ṕ"#),
        (r#"ṕ"#, r#"ṕ"#, r#"ṕ"#, r#"ṕ"#, r#"ṕ"#),
        (r#"Ṗ"#, r#"Ṗ"#, r#"Ṗ"#, r#"Ṗ"#, r#"Ṗ"#),
        (r#"ṗ"#, r#"ṗ"#, r#"ṗ"#, r#"ṗ"#, r#"ṗ"#),
        (r#"Ṙ"#, r#"Ṙ"#, r#"Ṙ"#, r#"Ṙ"#, r#"Ṙ"#),
        (r#"ṙ"#, r#"ṙ"#, r#"ṙ"#, r#"ṙ"#, r#"ṙ"#),
        (r#"Ṛ"#, r#"Ṛ"#, r#"Ṛ"#, r#"Ṛ"#, r#"Ṛ"#),
        (r#"ṛ"#, r#"ṛ"#, r#"ṛ"#, r#"ṛ"#, r#"ṛ"#),
        (r#"Ṝ"#, r#"Ṝ"#, r#"Ṝ"#, r#"Ṝ"#, r#"Ṝ"#),
        (r#"ṝ"#, r#"ṝ"#, r#"ṝ"#, r#"ṝ"#, r#"ṝ"#),
        (r#"Ṟ"#, r#"Ṟ"#, r#"Ṟ"#, r#"Ṟ"#, r#"Ṟ"#),
        (r#"ṟ"#, r#"ṟ"#, r#"ṟ"#, r#"ṟ"#, r#"ṟ"#),
        (r#"Ṡ"#, r#"Ṡ"#, r#"Ṡ"#, r#"Ṡ"#, r#"Ṡ"#),
        (r#"ṡ"#, r#"ṡ"#, r#"ṡ"#, r#"ṡ"#, r#"ṡ"#),
        (r#"Ṣ"#, r#"Ṣ"#, r#"Ṣ"#, r#"Ṣ"#, r#"Ṣ"#),
        (r#"ṣ"#, r#"ṣ"#, r#"ṣ"#, r#"ṣ"#, r#"ṣ"#),
        (r#"Ṥ"#, r#"Ṥ"#, r#"Ṥ"#, r#"Ṥ"#, r#"Ṥ"#),
        (r#"ṥ"#, r#"ṥ"#, r#"ṥ"#, r#"ṥ"#, r#"ṥ"#),
        (r#"Ṧ"#, r#"Ṧ"#, r#"Ṧ"#, r#"Ṧ"#, r#"Ṧ"#),
        (r#"ṧ"#, r#"ṧ"#, r#"ṧ"#, r#"ṧ"#, r#"ṧ"#),
        (r#"Ṩ"#, r#"Ṩ"#, r#"Ṩ"#, r#"Ṩ"#, r#"Ṩ"#),
        (r#"ṩ"#, r#"ṩ"#, r#"ṩ"#, r#"ṩ"#, r#"ṩ"#),
        (r#"Ṫ"#, r#"Ṫ"#, r#"Ṫ"#, r#"Ṫ"#, r#"Ṫ"#),
        (r#"ṫ"#, r#"ṫ"#, r#"ṫ"#, r#"ṫ"#, r#"ṫ"#),
        (r#"Ṭ"#, r#"Ṭ"#, r#"Ṭ"#, r#"Ṭ"#, r#"Ṭ"#),
        (r#"ṭ"#, r#"ṭ"#, r#"ṭ"#, r#"ṭ"#, r#"ṭ"#),
        (r#"Ṯ"#, r#"Ṯ"#, r#"Ṯ"#, r#"Ṯ"#, r#"Ṯ"#),
        (r#"ṯ"#, r#"ṯ"#, r#"ṯ"#, r#"ṯ"#, r#"ṯ"#),
        (r#"Ṱ"#, r#"Ṱ"#, r#"Ṱ"#, r#"Ṱ"#, r#"Ṱ"#),
        (r#"ṱ"#, r#"ṱ"#, r#"ṱ"#, r#"ṱ"#, r#"ṱ"#),
        (r#"Ṳ"#, r#"Ṳ"#, r#"Ṳ"#, r#"Ṳ"#, r#"Ṳ"#),
        (r#"ṳ"#, r#"ṳ"#, r#"ṳ"#, r#"ṳ"#, r#"ṳ"#),
        (r#"Ṵ"#, r#"Ṵ"#, r#"Ṵ"#, r#"Ṵ"#, r#"Ṵ"#),
        (r#"ṵ"#, r#"ṵ"#, r#"ṵ"#, r#"ṵ"#, r#"ṵ"#),
        (r#"Ṷ"#, r#"Ṷ"#, r#"Ṷ"#, r#"Ṷ"#, r#"Ṷ"#),
        (r#"ṷ"#, r#"ṷ"#, r#"ṷ"#, r#"ṷ"#, r#"ṷ"#),
        (r#"Ṹ"#, r#"Ṹ"#, r#"Ṹ"#, r#"Ṹ"#, r#"Ṹ"#),
        (r#"ṹ"#, r#"ṹ"#, r#"ṹ"#, r#"ṹ"#, r#"ṹ"#),
        (r#"Ṻ"#, r#"Ṻ"#, r#"Ṻ"#, r#"Ṻ"#, r#"Ṻ"#),
        (r#"ṻ"#, r#"ṻ"#, r#"ṻ"#, r#"ṻ"#, r#"ṻ"#),
        (r#"Ṽ"#, r#"Ṽ"#, r#"Ṽ"#, r#"Ṽ"#, r#"Ṽ"#),
        (r#"ṽ"#, r#"ṽ"#, r#"ṽ"#, r#"ṽ"#, r#"ṽ"#),
        (r#"Ṿ"#, r#"Ṿ"#, r#"Ṿ"#, r#"Ṿ"#, r#"Ṿ"#),
        (r#"ṿ"#, r#"ṿ"#, r#"ṿ"#, r#"ṿ"#, r#"ṿ"#),
        (r#"Ẁ"#, r#"Ẁ"#, r#"Ẁ"#, r#"Ẁ"#, r#"Ẁ"#),
        (r#"ẁ"#, r#"ẁ"#, r#"ẁ"#, r#"ẁ"#, r#"ẁ"#),
        (r#"Ẃ"#, r#"Ẃ"#, r#"Ẃ"#, r#"Ẃ"#, r#"Ẃ"#),
        (r#"ẃ"#, r#"ẃ"#, r#"ẃ"#, r#"ẃ"#, r#"ẃ"#),
        (r#"Ẅ"#, r#"Ẅ"#, r#"Ẅ"#, r#"Ẅ"#, r#"Ẅ"#),
        (r#"ẅ"#, r#"ẅ"#, r#"ẅ"#, r#"ẅ"#, r#"ẅ"#),
        (r#"Ẇ"#, r#"Ẇ"#, r#"Ẇ"#, r#"Ẇ"#, r#"Ẇ"#),
        (r#"ẇ"#, r#"ẇ"#, r#"ẇ"#, r#"ẇ"#, r#"ẇ"#),
        (r#"Ẉ"#, r#"Ẉ"#, r#"Ẉ"#, r#"Ẉ"#, r#"Ẉ"#),
        (r#"ẉ"#, r#"ẉ"#, r#"ẉ"#, r#"ẉ"#, r#"ẉ"#),
        (r#"Ẋ"#, r#"Ẋ"#, r#"Ẋ"#, r#"Ẋ"#, r#"Ẋ"#),
        (r#"ẋ"#, r#"ẋ"#, r#"ẋ"#, r#"ẋ"#, r#"ẋ"#),
        (r#"Ẍ"#, r#"Ẍ"#, r#"Ẍ"#, r#"Ẍ"#, r#"Ẍ"#),
        (r#"ẍ"#, r#"ẍ"#, r#"ẍ"#, r#"ẍ"#, r#"ẍ"#),
        (r#"Ẏ"#, r#"Ẏ"#, r#"Ẏ"#, r#"Ẏ"#, r#"Ẏ"#),
        (r#"ẏ"#, r#"ẏ"#, r#"ẏ"#, r#"ẏ"#, r#"ẏ"#),
        (r#"Ẑ"#, r#"Ẑ"#, r#"Ẑ"#, r#"Ẑ"#, r#"Ẑ"#),
        (r#"ẑ"#, r#"ẑ"#, r#"ẑ"#, r#"ẑ"#, r#"ẑ"#),
        (r#"Ẓ"#, r#"Ẓ"#, r#"Ẓ"#, r#"Ẓ"#, r#"Ẓ"#),
        (r#"ẓ"#, r#"ẓ"#, r#"ẓ"#, r#"ẓ"#, r#"ẓ"#),
        (r#"Ẕ"#, r#"Ẕ"#, r#"Ẕ"#, r#"Ẕ"#, r#"Ẕ"#),
        (r#"ẕ"#, r#"ẕ"#, r#"ẕ"#, r#"ẕ"#, r#"ẕ"#),
        (r#"ẖ"#, r#"ẖ"#, r#"ẖ"#, r#"ẖ"#, r#"ẖ"#),
        (r#"ẗ"#, r#"ẗ"#, r#"ẗ"#, r#"ẗ"#, r#"ẗ"#),
        (r#"ẘ"#, r#"ẘ"#, r#"ẘ"#, r#"ẘ"#, r#"ẘ"#),
        (r#"ẙ"#, r#"ẙ"#, r#"ẙ"#, r#"ẙ"#, r#"ẙ"#),
        (r#"ẚ"#, r#"ẚ"#, r#"ẚ"#, r#"aʾ"#, r#"aʾ"#),
        (r#"ẛ"#, r#"ẛ"#, r#"ẛ"#, r#"ṡ"#, r#"ṡ"#),
        (r#"Ạ"#, r#"Ạ"#, r#"Ạ"#, r#"Ạ"#, r#"Ạ"#),
        (r#"ạ"#, r#"ạ"#, r#"ạ"#, r#"ạ"#, r#"ạ"#),
        (r#"Ả"#, r#"Ả"#, r#"Ả"#, r#"Ả"#, r#"Ả"#),
        (r#"ả"#, r#"ả"#, r#"ả"#, r#"ả"#, r#"ả"#),
        (r#"Ấ"#, r#"Ấ"#, r#"Ấ"#, r#"Ấ"#, r#"Ấ"#),
        (r#"ấ"#, r#"ấ"#, r#"ấ"#, r#"ấ"#, r#"ấ"#),
        (r#"Ầ"#, r#"Ầ"#, r#"Ầ"#, r#"Ầ"#, r#"Ầ"#),
        (r#"ầ"#, r#"ầ"#, r#"ầ"#, r#"ầ"#, r#"ầ"#),
        (r#"Ẩ"#, r#"Ẩ"#, r#"Ẩ"#, r#"Ẩ"#, r#"Ẩ"#),
        (r#"ẩ"#, r#"ẩ"#, r#"ẩ"#, r#"ẩ"#, r#"ẩ"#),
        (r#"Ẫ"#, r#"Ẫ"#, r#"Ẫ"#, r#"Ẫ"#, r#"Ẫ"#),
        (r#"ẫ"#, r#"ẫ"#, r#"ẫ"#, r#"ẫ"#, r#"ẫ"#),
        (r#"Ậ"#, r#"Ậ"#, r#"Ậ"#, r#"Ậ"#, r#"Ậ"#),
        (r#"ậ"#, r#"ậ"#, r#"ậ"#, r#"ậ"#, r#"ậ"#),
        (r#"Ắ"#, r#"Ắ"#, r#"Ắ"#, r#"Ắ"#, r#"Ắ"#),
        (r#"ắ"#, r#"ắ"#, r#"ắ"#, r#"ắ"#, r#"ắ"#),
        (r#"Ằ"#, r#"Ằ"#, r#"Ằ"#, r#"Ằ"#, r#"Ằ"#),
        (r#"ằ"#, r#"ằ"#, r#"ằ"#, r#"ằ"#, r#"ằ"#),
        (r#"Ẳ"#, r#"Ẳ"#, r#"Ẳ"#, r#"Ẳ"#, r#"Ẳ"#),
        (r#"ẳ"#, r#"ẳ"#, r#"ẳ"#, r#"ẳ"#, r#"ẳ"#),
        (r#"Ẵ"#, r#"Ẵ"#, r#"Ẵ"#, r#"Ẵ"#, r#"Ẵ"#),
        (r#"ẵ"#, r#"ẵ"#, r#"ẵ"#, r#"ẵ"#, r#"ẵ"#),
        (r#"Ặ"#, r#"Ặ"#, r#"Ặ"#, r#"Ặ"#, r#"Ặ"#),
        (r#"ặ"#, r#"ặ"#, r#"ặ"#, r#"ặ"#, r#"ặ"#),
        (r#"Ẹ"#, r#"Ẹ"#, r#"Ẹ"#, r#"Ẹ"#, r#"Ẹ"#),
        (r#"ẹ"#, r#"ẹ"#, r#"ẹ"#, r#"ẹ"#, r#"ẹ"#),
        (r#"Ẻ"#, r#"Ẻ"#, r#"Ẻ"#, r#"Ẻ"#, r#"Ẻ"#),
        (r#"ẻ"#, r#"ẻ"#, r#"ẻ"#, r#"ẻ"#, r#"ẻ"#),
        (r#"Ẽ"#, r#"Ẽ"#, r#"Ẽ"#, r#"Ẽ"#, r#"Ẽ"#),
        (r#"ẽ"#, r#"ẽ"#, r#"ẽ"#, r#"ẽ"#, r#"ẽ"#),
        (r#"Ế"#, r#"Ế"#, r#"Ế"#, r#"Ế"#, r#"Ế"#),
        (r#"ế"#, r#"ế"#, r#"ế"#, r#"ế"#, r#"ế"#),
        (r#"Ề"#, r#"Ề"#, r#"Ề"#, r#"Ề"#, r#"Ề"#),
        (r#"ề"#, r#"ề"#, r#"ề"#, r#"ề"#, r#"ề"#),
        (r#"Ể"#, r#"Ể"#, r#"Ể"#, r#"Ể"#, r#"Ể"#),
        (r#"ể"#, r#"ể"#, r#"ể"#, r#"ể"#, r#"ể"#),
        (r#"Ễ"#, r#"Ễ"#, r#"Ễ"#, r#"Ễ"#, r#"Ễ"#),
        (r#"ễ"#, r#"ễ"#, r#"ễ"#, r#"ễ"#, r#"ễ"#),
        (r#"Ệ"#, r#"Ệ"#, r#"Ệ"#, r#"Ệ"#, r#"Ệ"#),
        (r#"ệ"#, r#"ệ"#, r#"ệ"#, r#"ệ"#, r#"ệ"#),
        (r#"Ỉ"#, r#"Ỉ"#, r#"Ỉ"#, r#"Ỉ"#, r#"Ỉ"#),
        (r#"ỉ"#, r#"ỉ"#, r#"ỉ"#, r#"ỉ"#, r#"ỉ"#),
        (r#"Ị"#, r#"Ị"#, r#"Ị"#, r#"Ị"#, r#"Ị"#),
        (r#"ị"#, r#"ị"#, r#"ị"#, r#"ị"#, r#"ị"#),
        (r#"Ọ"#, r#"Ọ"#, r#"Ọ"#, r#"Ọ"#, r#"Ọ"#),
        (r#"ọ"#, r#"ọ"#, r#"ọ"#, r#"ọ"#, r#"ọ"#),
        (r#"Ỏ"#, r#"Ỏ"#, r#"Ỏ"#, r#"Ỏ"#, r#"Ỏ"#),
        (r#"ỏ"#, r#"ỏ"#, r#"ỏ"#, r#"ỏ"#, r#"ỏ"#),
        (r#"Ố"#, r#"Ố"#, r#"Ố"#, r#"Ố"#, r#"Ố"#),
        (r#"ố"#, r#"ố"#, r#"ố"#, r#"ố"#, r#"ố"#),
        (r#"Ồ"#, r#"Ồ"#, r#"Ồ"#, r#"Ồ"#, r#"Ồ"#),
        (r#"ồ"#, r#"ồ"#, r#"ồ"#, r#"ồ"#, r#"ồ"#),
        (r#"Ổ"#, r#"Ổ"#, r#"Ổ"#, r#"Ổ"#, r#"Ổ"#),
        (r#"ổ"#, r#"ổ"#, r#"ổ"#, r#"ổ"#, r#"ổ"#),
        (r#"Ỗ"#, r#"Ỗ"#, r#"Ỗ"#, r#"Ỗ"#, r#"Ỗ"#),
        (r#"ỗ"#, r#"ỗ"#, r#"ỗ"#, r#"ỗ"#, r#"ỗ"#),
        (r#"Ộ"#, r#"Ộ"#, r#"Ộ"#, r#"Ộ"#, r#"Ộ"#),
        (r#"ộ"#, r#"ộ"#, r#"ộ"#, r#"ộ"#, r#"ộ"#),
        (r#"Ớ"#, r#"Ớ"#, r#"Ớ"#, r#"Ớ"#, r#"Ớ"#),
        (r#"ớ"#, r#"ớ"#, r#"ớ"#, r#"ớ"#, r#"ớ"#),
        (r#"Ờ"#, r#"Ờ"#, r#"Ờ"#, r#"Ờ"#, r#"Ờ"#),
        (r#"ờ"#, r#"ờ"#, r#"ờ"#, r#"ờ"#, r#"ờ"#),
        (r#"Ở"#, r#"Ở"#, r#"Ở"#, r#"Ở"#, r#"Ở"#),
        (r#"ở"#, r#"ở"#, r#"ở"#, r#"ở"#, r#"ở"#),
        (r#"Ỡ"#, r#"Ỡ"#, r#"Ỡ"#, r#"Ỡ"#, r#"Ỡ"#),
        (r#"ỡ"#, r#"ỡ"#, r#"ỡ"#, r#"ỡ"#, r#"ỡ"#),
        (r#"Ợ"#, r#"Ợ"#, r#"Ợ"#, r#"Ợ"#, r#"Ợ"#),
        (r#"ợ"#, r#"ợ"#, r#"ợ"#, r#"ợ"#, r#"ợ"#),
        (r#"Ụ"#, r#"Ụ"#, r#"Ụ"#, r#"Ụ"#, r#"Ụ"#),
        (r#"ụ"#, r#"ụ"#, r#"ụ"#, r#"ụ"#, r#"ụ"#),
        (r#"Ủ"#, r#"Ủ"#, r#"Ủ"#, r#"Ủ"#, r#"Ủ"#),
        (r#"ủ"#, r#"ủ"#, r#"ủ"#, r#"ủ"#, r#"ủ"#),
        (r#"Ứ"#, r#"Ứ"#, r#"Ứ"#, r#"Ứ"#, r#"Ứ"#),
        (r#"ứ"#, r#"ứ"#, r#"ứ"#, r#"ứ"#, r#"ứ"#),
        (r#"Ừ"#, r#"Ừ"#, r#"Ừ"#, r#"Ừ"#, r#"Ừ"#),
        (r#"ừ"#, r#"ừ"#, r#"ừ"#, r#"ừ"#, r#"ừ"#),
        (r#"Ử"#, r#"Ử"#, r#"Ử"#, r#"Ử"#, r#"Ử"#),
        (r#"ử"#, r#"ử"#, r#"ử"#, r#"ử"#, r#"ử"#),
        (r#"Ữ"#, r#"Ữ"#, r#"Ữ"#, r#"Ữ"#, r#"Ữ"#),
        (r#"ữ"#, r#"ữ"#, r#"ữ"#, r#"ữ"#, r#"ữ"#),
        (r#"Ự"#, r#"Ự"#, r#"Ự"#, r#"Ự"#, r#"Ự"#),
        (r#"ự"#, r#"ự"#, r#"ự"#, r#"ự"#, r#"ự"#),
        (r#"Ỳ"#, r#"Ỳ"#, r#"Ỳ"#, r#"Ỳ"#, r#"Ỳ"#),
        (r#"ỳ"#, r#"ỳ"#, r#"ỳ"#, r#"ỳ"#, r#"ỳ"#),
        (r#"Ỵ"#, r#"Ỵ"#, r#"Ỵ"#, r#"Ỵ"#, r#"Ỵ"#),
        (r#"ỵ"#, r#"ỵ"#, r#"ỵ"#, r#"ỵ"#, r#"ỵ"#),
        (r#"Ỷ"#, r#"Ỷ"#, r#"Ỷ"#, r#"Ỷ"#, r#"Ỷ"#),
        (r#"ỷ"#, r#"ỷ"#, r#"ỷ"#, r#"ỷ"#, r#"ỷ"#),
        (r#"Ỹ"#, r#"Ỹ"#, r#"Ỹ"#, r#"Ỹ"#, r#"Ỹ"#),
        (r#"ỹ"#, r#"ỹ"#, r#"ỹ"#, r#"ỹ"#, r#"ỹ"#),
        (r#"ἀ"#, r#"ἀ"#, r#"ἀ"#, r#"ἀ"#, r#"ἀ"#),
        (r#"ἁ"#, r#"ἁ"#, r#"ἁ"#, r#"ἁ"#, r#"ἁ"#),
        (r#"ἂ"#, r#"ἂ"#, r#"ἂ"#, r#"ἂ"#, r#"ἂ"#),
        (r#"ἃ"#, r#"ἃ"#, r#"ἃ"#, r#"ἃ"#, r#"ἃ"#),
        (r#"ἄ"#, r#"ἄ"#, r#"ἄ"#, r#"ἄ"#, r#"ἄ"#),
        (r#"ἅ"#, r#"ἅ"#, r#"ἅ"#, r#"ἅ"#, r#"ἅ"#),
        (r#"ἆ"#, r#"ἆ"#, r#"ἆ"#, r#"ἆ"#, r#"ἆ"#),
        (r#"ἇ"#, r#"ἇ"#, r#"ἇ"#, r#"ἇ"#, r#"ἇ"#),
        (r#"Ἀ"#, r#"Ἀ"#, r#"Ἀ"#, r#"Ἀ"#, r#"Ἀ"#),
        (r#"Ἁ"#, r#"Ἁ"#, r#"Ἁ"#, r#"Ἁ"#, r#"Ἁ"#),
        (r#"Ἂ"#, r#"Ἂ"#, r#"Ἂ"#, r#"Ἂ"#, r#"Ἂ"#),
        (r#"Ἃ"#, r#"Ἃ"#, r#"Ἃ"#, r#"Ἃ"#, r#"Ἃ"#),
        (r#"Ἄ"#, r#"Ἄ"#, r#"Ἄ"#, r#"Ἄ"#, r#"Ἄ"#),
        (r#"Ἅ"#, r#"Ἅ"#, r#"Ἅ"#, r#"Ἅ"#, r#"Ἅ"#),
        (r#"Ἆ"#, r#"Ἆ"#, r#"Ἆ"#, r#"Ἆ"#, r#"Ἆ"#),
        (r#"Ἇ"#, r#"Ἇ"#, r#"Ἇ"#, r#"Ἇ"#, r#"Ἇ"#),
        (r#"ἐ"#, r#"ἐ"#, r#"ἐ"#, r#"ἐ"#, r#"ἐ"#),
        (r#"ἑ"#, r#"ἑ"#, r#"ἑ"#, r#"ἑ"#, r#"ἑ"#),
        (r#"ἒ"#, r#"ἒ"#, r#"ἒ"#, r#"ἒ"#, r#"ἒ"#),
        (r#"ἓ"#, r#"ἓ"#, r#"ἓ"#, r#"ἓ"#, r#"ἓ"#),
        (r#"ἔ"#, r#"ἔ"#, r#"ἔ"#, r#"ἔ"#, r#"ἔ"#),
        (r#"ἕ"#, r#"ἕ"#, r#"ἕ"#, r#"ἕ"#, r#"ἕ"#),
        (r#"Ἐ"#, r#"Ἐ"#, r#"Ἐ"#, r#"Ἐ"#, r#"Ἐ"#),
        (r#"Ἑ"#, r#"Ἑ"#, r#"Ἑ"#, r#"Ἑ"#, r#"Ἑ"#),
        (r#"Ἒ"#, r#"Ἒ"#, r#"Ἒ"#, r#"Ἒ"#, r#"Ἒ"#),
        (r#"Ἓ"#, r#"Ἓ"#, r#"Ἓ"#, r#"Ἓ"#, r#"Ἓ"#),
        (r#"Ἔ"#, r#"Ἔ"#, r#"Ἔ"#, r#"Ἔ"#, r#"Ἔ"#),
        (r#"Ἕ"#, r#"Ἕ"#, r#"Ἕ"#, r#"Ἕ"#, r#"Ἕ"#),
        (r#"ἠ"#, r#"ἠ"#, r#"ἠ"#, r#"ἠ"#, r#"ἠ"#),
        (r#"ἡ"#, r#"ἡ"#, r#"ἡ"#, r#"ἡ"#, r#"ἡ"#),
        (r#"ἢ"#, r#"ἢ"#, r#"ἢ"#, r#"ἢ"#, r#"ἢ"#),
        (r#"ἣ"#, r#"ἣ"#, r#"ἣ"#, r#"ἣ"#, r#"ἣ"#),
        (r#"ἤ"#, r#"ἤ"#, r#"ἤ"#, r#"ἤ"#, r#"ἤ"#),
        (r#"ἥ"#, r#"ἥ"#, r#"ἥ"#, r#"ἥ"#, r#"ἥ"#),
        (r#"ἦ"#, r#"ἦ"#, r#"ἦ"#, r#"ἦ"#, r#"ἦ"#),
        (r#"ἧ"#, r#"ἧ"#, r#"ἧ"#, r#"ἧ"#, r#"ἧ"#),
        (r#"Ἠ"#, r#"Ἠ"#, r#"Ἠ"#, r#"Ἠ"#, r#"Ἠ"#),
        (r#"Ἡ"#, r#"Ἡ"#, r#"Ἡ"#, r#"Ἡ"#, r#"Ἡ"#),
        (r#"Ἢ"#, r#"Ἢ"#, r#"Ἢ"#, r#"Ἢ"#, r#"Ἢ"#),
        (r#"Ἣ"#, r#"Ἣ"#, r#"Ἣ"#, r#"Ἣ"#, r#"Ἣ"#),
        (r#"Ἤ"#, r#"Ἤ"#, r#"Ἤ"#, r#"Ἤ"#, r#"Ἤ"#),
        (r#"Ἥ"#, r#"Ἥ"#, r#"Ἥ"#, r#"Ἥ"#, r#"Ἥ"#),
        (r#"Ἦ"#, r#"Ἦ"#, r#"Ἦ"#, r#"Ἦ"#, r#"Ἦ"#),
        (r#"Ἧ"#, r#"Ἧ"#, r#"Ἧ"#, r#"Ἧ"#, r#"Ἧ"#),
        (r#"ἰ"#, r#"ἰ"#, r#"ἰ"#, r#"ἰ"#, r#"ἰ"#),
        (r#"ἱ"#, r#"ἱ"#, r#"ἱ"#, r#"ἱ"#, r#"ἱ"#),
        (r#"ἲ"#, r#"ἲ"#, r#"ἲ"#, r#"ἲ"#, r#"ἲ"#),
        (r#"ἳ"#, r#"ἳ"#, r#"ἳ"#, r#"ἳ"#, r#"ἳ"#),
        (r#"ἴ"#, r#"ἴ"#, r#"ἴ"#, r#"ἴ"#, r#"ἴ"#),
        (r#"ἵ"#, r#"ἵ"#, r#"ἵ"#, r#"ἵ"#, r#"ἵ"#),
        (r#"ἶ"#, r#"ἶ"#, r#"ἶ"#, r#"ἶ"#, r#"ἶ"#),
        (r#"ἷ"#, r#"ἷ"#, r#"ἷ"#, r#"ἷ"#, r#"ἷ"#),
        (r#"Ἰ"#, r#"Ἰ"#, r#"Ἰ"#, r#"Ἰ"#, r#"Ἰ"#),
        (r#"Ἱ"#, r#"Ἱ"#, r#"Ἱ"#, r#"Ἱ"#, r#"Ἱ"#),
        (r#"Ἲ"#, r#"Ἲ"#, r#"Ἲ"#, r#"Ἲ"#, r#"Ἲ"#),
        (r#"Ἳ"#, r#"Ἳ"#, r#"Ἳ"#, r#"Ἳ"#, r#"Ἳ"#),
        (r#"Ἴ"#, r#"Ἴ"#, r#"Ἴ"#, r#"Ἴ"#, r#"Ἴ"#),
        (r#"Ἵ"#, r#"Ἵ"#, r#"Ἵ"#, r#"Ἵ"#, r#"Ἵ"#),
        (r#"Ἶ"#, r#"Ἶ"#, r#"Ἶ"#, r#"Ἶ"#, r#"Ἶ"#),
        (r#"Ἷ"#, r#"Ἷ"#, r#"Ἷ"#, r#"Ἷ"#, r#"Ἷ"#),
        (r#"ὀ"#, r#"ὀ"#, r#"ὀ"#, r#"ὀ"#, r#"ὀ"#),
        (r#"ὁ"#, r#"ὁ"#, r#"ὁ"#, r#"ὁ"#, r#"ὁ"#),
        (r#"ὂ"#, r#"ὂ"#, r#"ὂ"#, r#"ὂ"#, r#"ὂ"#),
        (r#"ὃ"#, r#"ὃ"#, r#"ὃ"#, r#"ὃ"#, r#"ὃ"#),
        (r#"ὄ"#, r#"ὄ"#, r#"ὄ"#, r#"ὄ"#, r#"ὄ"#),
        (r#"ὅ"#, r#"ὅ"#, r#"ὅ"#, r#"ὅ"#, r#"ὅ"#),
        (r#"Ὀ"#, r#"Ὀ"#, r#"Ὀ"#, r#"Ὀ"#, r#"Ὀ"#),
        (r#"Ὁ"#, r#"Ὁ"#, r#"Ὁ"#, r#"Ὁ"#, r#"Ὁ"#),
        (r#"Ὂ"#, r#"Ὂ"#, r#"Ὂ"#, r#"Ὂ"#, r#"Ὂ"#),
        (r#"Ὃ"#, r#"Ὃ"#, r#"Ὃ"#, r#"Ὃ"#, r#"Ὃ"#),
        (r#"Ὄ"#, r#"Ὄ"#, r#"Ὄ"#, r#"Ὄ"#, r#"Ὄ"#),
        (r#"Ὅ"#, r#"Ὅ"#, r#"Ὅ"#, r#"Ὅ"#, r#"Ὅ"#),
        (r#"ὐ"#, r#"ὐ"#, r#"ὐ"#, r#"ὐ"#, r#"ὐ"#),
        (r#"ὑ"#, r#"ὑ"#, r#"ὑ"#, r#"ὑ"#, r#"ὑ"#),
        (r#"ὒ"#, r#"ὒ"#, r#"ὒ"#, r#"ὒ"#, r#"ὒ"#),
        (r#"ὓ"#, r#"ὓ"#, r#"ὓ"#, r#"ὓ"#, r#"ὓ"#),
        (r#"ὔ"#, r#"ὔ"#, r#"ὔ"#, r#"ὔ"#, r#"ὔ"#),
        (r#"ὕ"#, r#"ὕ"#, r#"ὕ"#, r#"ὕ"#, r#"ὕ"#),
        (r#"ὖ"#, r#"ὖ"#, r#"ὖ"#, r#"ὖ"#, r#"ὖ"#),
        (r#"ὗ"#, r#"ὗ"#, r#"ὗ"#, r#"ὗ"#, r#"ὗ"#),
        (r#"Ὑ"#, r#"Ὑ"#, r#"Ὑ"#, r#"Ὑ"#, r#"Ὑ"#),
        (r#"Ὓ"#, r#"Ὓ"#, r#"Ὓ"#, r#"Ὓ"#, r#"Ὓ"#),
        (r#"Ὕ"#, r#"Ὕ"#, r#"Ὕ"#, r#"Ὕ"#, r#"Ὕ"#),
        (r#"Ὗ"#, r#"Ὗ"#, r#"Ὗ"#, r#"Ὗ"#, r#"Ὗ"#),
        (r#"ὠ"#, r#"ὠ"#, r#"ὠ"#, r#"ὠ"#, r#"ὠ"#),
        (r#"ὡ"#, r#"ὡ"#, r#"ὡ"#, r#"ὡ"#, r#"ὡ"#),
        (r#"ὢ"#, r#"ὢ"#, r#"ὢ"#, r#"ὢ"#, r#"ὢ"#),
        (r#"ὣ"#, r#"ὣ"#, r#"ὣ"#, r#"ὣ"#, r#"ὣ"#),
        (r#"ὤ"#, r#"ὤ"#, r#"ὤ"#, r#"ὤ"#, r#"ὤ"#),
        (r#"ὥ"#, r#"ὥ"#, r#"ὥ"#, r#"ὥ"#, r#"ὥ"#),
        (r#"ὦ"#, r#"ὦ"#, r#"ὦ"#, r#"ὦ"#, r#"ὦ"#),
        (r#"ὧ"#, r#"ὧ"#, r#"ὧ"#, r#"ὧ"#, r#"ὧ"#),
        (r#"Ὠ"#, r#"Ὠ"#, r#"Ὠ"#, r#"Ὠ"#, r#"Ὠ"#),
        (r#"Ὡ"#, r#"Ὡ"#, r#"Ὡ"#, r#"Ὡ"#, r#"Ὡ"#),
        (r#"Ὢ"#, r#"Ὢ"#, r#"Ὢ"#, r#"Ὢ"#, r#"Ὢ"#),
        (r#"Ὣ"#, r#"Ὣ"#, r#"Ὣ"#, r#"Ὣ"#, r#"Ὣ"#),
        (r#"Ὤ"#, r#"Ὤ"#, r#"Ὤ"#, r#"Ὤ"#, r#"Ὤ"#),
        (r#"Ὥ"#, r#"Ὥ"#, r#"Ὥ"#, r#"Ὥ"#, r#"Ὥ"#),
        (r#"Ὦ"#, r#"Ὦ"#, r#"Ὦ"#, r#"Ὦ"#, r#"Ὦ"#),
        (r#"Ὧ"#, r#"Ὧ"#, r#"Ὧ"#, r#"Ὧ"#, r#"Ὧ"#),
        (r#"ὰ"#, r#"ὰ"#, r#"ὰ"#, r#"ὰ"#, r#"ὰ"#),
        (r#"ά"#, r#"ά"#, r#"ά"#, r#"ά"#, r#"ά"#),
        (r#"ὲ"#, r#"ὲ"#, r#"ὲ"#, r#"ὲ"#, r#"ὲ"#),
        (r#"έ"#, r#"έ"#, r#"έ"#, r#"έ"#, r#"έ"#),
        (r#"ὴ"#, r#"ὴ"#, r#"ὴ"#, r#"ὴ"#, r#"ὴ"#),
        (r#"ή"#, r#"ή"#, r#"ή"#, r#"ή"#, r#"ή"#),
        (r#"ὶ"#, r#"ὶ"#, r#"ὶ"#, r#"ὶ"#, r#"ὶ"#),
        (r#"ί"#, r#"ί"#, r#"ί"#, r#"ί"#, r#"ί"#),
        (r#"ὸ"#, r#"ὸ"#, r#"ὸ"#, r#"ὸ"#, r#"ὸ"#),
        (r#"ό"#, r#"ό"#, r#"ό"#, r#"ό"#, r#"ό"#),
        (r#"ὺ"#, r#"ὺ"#, r#"ὺ"#, r#"ὺ"#, r#"ὺ"#),
        (r#"ύ"#, r#"ύ"#, r#"ύ"#, r#"ύ"#, r#"ύ"#),
        (r#"ὼ"#, r#"ὼ"#, r#"ὼ"#, r#"ὼ"#, r#"ὼ"#),
        (r#"ώ"#, r#"ώ"#, r#"ώ"#, r#"ώ"#, r#"ώ"#),
        (r#"ᾀ"#, r#"ᾀ"#, r#"ᾀ"#, r#"ᾀ"#, r#"ᾀ"#),
        (r#"ᾁ"#, r#"ᾁ"#, r#"ᾁ"#, r#"ᾁ"#, r#"ᾁ"#),
        (r#"ᾂ"#, r#"ᾂ"#, r#"ᾂ"#, r#"ᾂ"#, r#"ᾂ"#),
        (r#"ᾃ"#, r#"ᾃ"#, r#"ᾃ"#, r#"ᾃ"#, r#"ᾃ"#),
        (r#"ᾄ"#, r#"ᾄ"#, r#"ᾄ"#, r#"ᾄ"#, r#"ᾄ"#),
        (r#"ᾅ"#, r#"ᾅ"#, r#"ᾅ"#, r#"ᾅ"#, r#"ᾅ"#),
        (r#"ᾆ"#, r#"ᾆ"#, r#"ᾆ"#, r#"ᾆ"#, r#"ᾆ"#),
        (r#"ᾇ"#, r#"ᾇ"#, r#"ᾇ"#, r#"ᾇ"#, r#"ᾇ"#),
        (r#"ᾈ"#, r#"ᾈ"#, r#"ᾈ"#, r#"ᾈ"#, r#"ᾈ"#),
        (r#"ᾉ"#, r#"ᾉ"#, r#"ᾉ"#, r#"ᾉ"#, r#"ᾉ"#),
        (r#"ᾊ"#, r#"ᾊ"#, r#"ᾊ"#, r#"ᾊ"#, r#"ᾊ"#),
        (r#"ᾋ"#, r#"ᾋ"#, r#"ᾋ"#, r#"ᾋ"#, r#"ᾋ"#),
        (r#"ᾌ"#, r#"ᾌ"#, r#"ᾌ"#, r#"ᾌ"#, r#"ᾌ"#),
        (r#"ᾍ"#, r#"ᾍ"#, r#"ᾍ"#, r#"ᾍ"#, r#"ᾍ"#),
        (r#"ᾎ"#, r#"ᾎ"#, r#"ᾎ"#, r#"ᾎ"#, r#"ᾎ"#),
        (r#"ᾏ"#, r#"ᾏ"#, r#"ᾏ"#, r#"ᾏ"#, r#"ᾏ"#),
        (r#"ᾐ"#, r#"ᾐ"#, r#"ᾐ"#, r#"ᾐ"#, r#"ᾐ"#),
        (r#"ᾑ"#, r#"ᾑ"#, r#"ᾑ"#, r#"ᾑ"#, r#"ᾑ"#),
        (r#"ᾒ"#, r#"ᾒ"#, r#"ᾒ"#, r#"ᾒ"#, r#"ᾒ"#),
        (r#"ᾓ"#, r#"ᾓ"#, r#"ᾓ"#, r#"ᾓ"#, r#"ᾓ"#),
        (r#"ᾔ"#, r#"ᾔ"#, r#"ᾔ"#, r#"ᾔ"#, r#"ᾔ"#),
        (r#"ᾕ"#, r#"ᾕ"#, r#"ᾕ"#, r#"ᾕ"#, r#"ᾕ"#),
        (r#"ᾖ"#, r#"ᾖ"#, r#"ᾖ"#, r#"ᾖ"#, r#"ᾖ"#),
        (r#"ᾗ"#, r#"ᾗ"#, r#"ᾗ"#, r#"ᾗ"#, r#"ᾗ"#),
        (r#"ᾘ"#, r#"ᾘ"#, r#"ᾘ"#, r#"ᾘ"#, r#"ᾘ"#),
        (r#"ᾙ"#, r#"ᾙ"#, r#"ᾙ"#, r#"ᾙ"#, r#"ᾙ"#),
        (r#"ᾚ"#, r#"ᾚ"#, r#"ᾚ"#, r#"ᾚ"#, r#"ᾚ"#),
        (r#"ᾛ"#, r#"ᾛ"#, r#"ᾛ"#, r#"ᾛ"#, r#"ᾛ"#),
        (r#"ᾜ"#, r#"ᾜ"#, r#"ᾜ"#, r#"ᾜ"#, r#"ᾜ"#),
        (r#"ᾝ"#, r#"ᾝ"#, r#"ᾝ"#, r#"ᾝ"#, r#"ᾝ"#),
        (r#"ᾞ"#, r#"ᾞ"#, r#"ᾞ"#, r#"ᾞ"#, r#"ᾞ"#),
        (r#"ᾟ"#, r#"ᾟ"#, r#"ᾟ"#, r#"ᾟ"#, r#"ᾟ"#),
        (r#"ᾠ"#, r#"ᾠ"#, r#"ᾠ"#, r#"ᾠ"#, r#"ᾠ"#),
        (r#"ᾡ"#, r#"ᾡ"#, r#"ᾡ"#, r#"ᾡ"#, r#"ᾡ"#),
        (r#"ᾢ"#, r#"ᾢ"#, r#"ᾢ"#, r#"ᾢ"#, r#"ᾢ"#),
        (r#"ᾣ"#, r#"ᾣ"#, r#"ᾣ"#, r#"ᾣ"#, r#"ᾣ"#),
        (r#"ᾤ"#, r#"ᾤ"#, r#"ᾤ"#, r#"ᾤ"#, r#"ᾤ"#),
        (r#"ᾥ"#, r#"ᾥ"#, r#"ᾥ"#, r#"ᾥ"#, r#"ᾥ"#),
        (r#"ᾦ"#, r#"ᾦ"#, r#"ᾦ"#, r#"ᾦ"#, r#"ᾦ"#),
        (r#"ᾧ"#, r#"ᾧ"#, r#"ᾧ"#, r#"ᾧ"#, r#"ᾧ"#),
        (r#"ᾨ"#, r#"ᾨ"#, r#"ᾨ"#, r#"ᾨ"#, r#"ᾨ"#),
        (r#"ᾩ"#, r#"ᾩ"#, r#"ᾩ"#, r#"ᾩ"#, r#"ᾩ"#),
        (r#"ᾪ"#, r#"ᾪ"#, r#"ᾪ"#, r#"ᾪ"#, r#"ᾪ"#),
        (r#"ᾫ"#, r#"ᾫ"#, r#"ᾫ"#, r#"ᾫ"#, r#"ᾫ"#),
        (r#"ᾬ"#, r#"ᾬ"#, r#"ᾬ"#, r#"ᾬ"#, r#"ᾬ"#),
        (r#"ᾭ"#, r#"ᾭ"#, r#"ᾭ"#, r#"ᾭ"#, r#"ᾭ"#),
        (r#"ᾮ"#, r#"ᾮ"#, r#"ᾮ"#, r#"ᾮ"#, r#"ᾮ"#),
        (r#"ᾯ"#, r#"ᾯ"#, r#"ᾯ"#, r#"ᾯ"#, r#"ᾯ"#),
        (r#"ᾰ"#, r#"ᾰ"#, r#"ᾰ"#, r#"ᾰ"#, r#"ᾰ"#),
        (r#"ᾱ"#, r#"ᾱ"#, r#"ᾱ"#, r#"ᾱ"#, r#"ᾱ"#),
        (r#"ᾲ"#, r#"ᾲ"#, r#"ᾲ"#, r#"ᾲ"#, r#"ᾲ"#),
        (r#"ᾳ"#, r#"ᾳ"#, r#"ᾳ"#, r#"ᾳ"#, r#"ᾳ"#),
        (r#"ᾴ"#, r#"ᾴ"#, r#"ᾴ"#, r#"ᾴ"#, r#"ᾴ"#),
        (r#"ᾶ"#, r#"ᾶ"#, r#"ᾶ"#, r#"ᾶ"#, r#"ᾶ"#),
        (r#"ᾷ"#, r#"ᾷ"#, r#"ᾷ"#, r#"ᾷ"#, r#"ᾷ"#),
        (r#"Ᾰ"#, r#"Ᾰ"#, r#"Ᾰ"#, r#"Ᾰ"#, r#"Ᾰ"#),
        (r#"Ᾱ"#, r#"Ᾱ"#, r#"Ᾱ"#, r#"Ᾱ"#, r#"Ᾱ"#),
        (r#"Ὰ"#, r#"Ὰ"#, r#"Ὰ"#, r#"Ὰ"#, r#"Ὰ"#),
        (r#"Ά"#, r#"Ά"#, r#"Ά"#, r#"Ά"#, r#"Ά"#),
        (r#"ᾼ"#, r#"ᾼ"#, r#"ᾼ"#, r#"ᾼ"#, r#"ᾼ"#),
        (r#"᾽"#, r#"᾽"#, r#"᾽"#, r#" ̓"#, r#" ̓"#),
        (r#"ι"#, r#"ι"#, r#"ι"#, r#"ι"#, r#"ι"#),
        (r#"᾿"#, r#"᾿"#, r#"᾿"#, r#" ̓"#, r#" ̓"#),
        (r#"῀"#, r#"῀"#, r#"῀"#, r#" ͂"#, r#" ͂"#),
        (r#"῁"#, r#"῁"#, r#"῁"#, r#" ̈͂"#, r#" ̈͂"#),
        (r#"ῂ"#, r#"ῂ"#, r#"ῂ"#, r#"ῂ"#, r#"ῂ"#),
        (r#"ῃ"#, r#"ῃ"#, r#"ῃ"#, r#"ῃ"#, r#"ῃ"#),
        (r#"ῄ"#, r#"ῄ"#, r#"ῄ"#, r#"ῄ"#, r#"ῄ"#),
        (r#"ῆ"#, r#"ῆ"#, r#"ῆ"#, r#"ῆ"#, r#"ῆ"#),
        (r#"ῇ"#, r#"ῇ"#, r#"ῇ"#, r#"ῇ"#, r#"ῇ"#),
        (r#"Ὲ"#, r#"Ὲ"#, r#"Ὲ"#, r#"Ὲ"#, r#"Ὲ"#),
        (r#"Έ"#, r#"Έ"#, r#"Έ"#, r#"Έ"#, r#"Έ"#),
        (r#"Ὴ"#, r#"Ὴ"#, r#"Ὴ"#, r#"Ὴ"#, r#"Ὴ"#),
        (r#"Ή"#, r#"Ή"#, r#"Ή"#, r#"Ή"#, r#"Ή"#),
        (r#"ῌ"#, r#"ῌ"#, r#"ῌ"#, r#"ῌ"#, r#"ῌ"#),
        (r#"῍"#, r#"῍"#, r#"῍"#, r#" ̓̀"#, r#" ̓̀"#),
        (r#"῎"#, r#"῎"#, r#"῎"#, r#" ̓́"#, r#" ̓́"#),
        (r#"῏"#, r#"῏"#, r#"῏"#, r#" ̓͂"#, r#" ̓͂"#),
        (r#"ῐ"#, r#"ῐ"#, r#"ῐ"#, r#"ῐ"#, r#"ῐ"#),
        (r#"ῑ"#, r#"ῑ"#, r#"ῑ"#, r#"ῑ"#, r#"ῑ"#),
        (r#"ῒ"#, r#"ῒ"#, r#"ῒ"#, r#"ῒ"#, r#"ῒ"#),
        (r#"ΐ"#, r#"ΐ"#, r#"ΐ"#, r#"ΐ"#, r#"ΐ"#),
        (r#"ῖ"#, r#"ῖ"#, r#"ῖ"#, r#"ῖ"#, r#"ῖ"#),
        (r#"ῗ"#, r#"ῗ"#, r#"ῗ"#, r#"ῗ"#, r#"ῗ"#),
        (r#"Ῐ"#, r#"Ῐ"#, r#"Ῐ"#, r#"Ῐ"#, r#"Ῐ"#),
        (r#"Ῑ"#, r#"Ῑ"#, r#"Ῑ"#, r#"Ῑ"#, r#"Ῑ"#),
        (r#"Ὶ"#, r#"Ὶ"#, r#"Ὶ"#, r#"Ὶ"#, r#"Ὶ"#),
        (r#"Ί"#, r#"Ί"#, r#"Ί"#, r#"Ί"#, r#"Ί"#),
        (r#"῝"#, r#"῝"#, r#"῝"#, r#" ̔̀"#, r#" ̔̀"#),
        (r#"῞"#, r#"῞"#, r#"῞"#, r#" ̔́"#, r#" ̔́"#),
        (r#"῟"#, r#"῟"#, r#"῟"#, r#" ̔͂"#, r#" ̔͂"#),
        (r#"ῠ"#, r#"ῠ"#, r#"ῠ"#, r#"ῠ"#, r#"ῠ"#),
        (r#"ῡ"#, r#"ῡ"#, r#"ῡ"#, r#"ῡ"#, r#"ῡ"#),
        (r#"ῢ"#, r#"ῢ"#, r#"ῢ"#, r#"ῢ"#, r#"ῢ"#),
        (r#"ΰ"#, r#"ΰ"#, r#"ΰ"#, r#"ΰ"#, r#"ΰ"#),
        (r#"ῤ"#, r#"ῤ"#, r#"ῤ"#, r#"ῤ"#, r#"ῤ"#),
        (r#"ῥ"#, r#"ῥ"#, r#"ῥ"#, r#"ῥ"#, r#"ῥ"#),
        (r#"ῦ"#, r#"ῦ"#, r#"ῦ"#, r#"ῦ"#, r#"ῦ"#),
        (r#"ῧ"#, r#"ῧ"#, r#"ῧ"#, r#"ῧ"#, r#"ῧ"#),
        (r#"Ῠ"#, r#"Ῠ"#, r#"Ῠ"#, r#"Ῠ"#, r#"Ῠ"#),
        (r#"Ῡ"#, r#"Ῡ"#, r#"Ῡ"#, r#"Ῡ"#, r#"Ῡ"#),
        (r#"Ὺ"#, r#"Ὺ"#, r#"Ὺ"#, r#"Ὺ"#, r#"Ὺ"#),
        (r#"Ύ"#, r#"Ύ"#, r#"Ύ"#, r#"Ύ"#, r#"Ύ"#),
        (r#"Ῥ"#, r#"Ῥ"#, r#"Ῥ"#, r#"Ῥ"#, r#"Ῥ"#),
        (r#"῭"#, r#"῭"#, r#"῭"#, r#" ̈̀"#, r#" ̈̀"#),
        (r#"΅"#, r#"΅"#, r#"΅"#, r#" ̈́"#, r#" ̈́"#),
        (r#"`"#, r#"`"#, r#"`"#, r#"`"#, r#"`"#),
        (r#"ῲ"#, r#"ῲ"#, r#"ῲ"#, r#"ῲ"#, r#"ῲ"#),
        (r#"ῳ"#, r#"ῳ"#, r#"ῳ"#, r#"ῳ"#, r#"ῳ"#),
        (r#"ῴ"#, r#"ῴ"#, r#"ῴ"#, r#"ῴ"#, r#"ῴ"#),
        (r#"ῶ"#, r#"ῶ"#, r#"ῶ"#, r#"ῶ"#, r#"ῶ"#),
        (r#"ῷ"#, r#"ῷ"#, r#"ῷ"#, r#"ῷ"#, r#"ῷ"#),
        (r#"Ὸ"#, r#"Ὸ"#, r#"Ὸ"#, r#"Ὸ"#, r#"Ὸ"#),
        (r#"Ό"#, r#"Ό"#, r#"Ό"#, r#"Ό"#, r#"Ό"#),
        (r#"Ὼ"#, r#"Ὼ"#, r#"Ὼ"#, r#"Ὼ"#, r#"Ὼ"#),
        (r#"Ώ"#, r#"Ώ"#, r#"Ώ"#, r#"Ώ"#, r#"Ώ"#),
        (r#"ῼ"#, r#"ῼ"#, r#"ῼ"#, r#"ῼ"#, r#"ῼ"#),
        (r#"´"#, r#"´"#, r#"´"#, r#" ́"#, r#" ́"#),
        (r#"῾"#, r#"῾"#, r#"῾"#, r#" ̔"#, r#" ̔"#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#"‑"#, r#"‑"#, r#"‑"#, r#"‐"#, r#"‐"#),
        (r#"‗"#, r#"‗"#, r#"‗"#, r#" ̳"#, r#" ̳"#),
        (r#"․"#, r#"․"#, r#"․"#, r#"."#, r#"."#),
        (r#"‥"#, r#"‥"#, r#"‥"#, r#".."#, r#".."#),
        (r#"…"#, r#"…"#, r#"…"#, r#"..."#, r#"..."#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#"″"#, r#"″"#, r#"″"#, r#"′′"#, r#"′′"#),
        (r#"‴"#, r#"‴"#, r#"‴"#, r#"′′′"#, r#"′′′"#),
        (r#"‶"#, r#"‶"#, r#"‶"#, r#"‵‵"#, r#"‵‵"#),
        (r#"‷"#, r#"‷"#, r#"‷"#, r#"‵‵‵"#, r#"‵‵‵"#),
        (r#"‼"#, r#"‼"#, r#"‼"#, r#"!!"#, r#"!!"#),
        (r#"‾"#, r#"‾"#, r#"‾"#, r#" ̅"#, r#" ̅"#),
        (r#"⁇"#, r#"⁇"#, r#"⁇"#, r#"??"#, r#"??"#),
        (r#"⁈"#, r#"⁈"#, r#"⁈"#, r#"?!"#, r#"?!"#),
        (r#"⁉"#, r#"⁉"#, r#"⁉"#, r#"!?"#, r#"!?"#),
        (r#"⁗"#, r#"⁗"#, r#"⁗"#, r#"′′′′"#, r#"′′′′"#),
        (r#" "#, r#" "#, r#" "#, r#" "#, r#" "#),
        (r#"⁰"#, r#"⁰"#, r#"⁰"#, r#"0"#, r#"0"#),
        (r#"ⁱ"#, r#"ⁱ"#, r#"ⁱ"#, r#"i"#, r#"i"#),
        (r#"⁴"#, r#"⁴"#, r#"⁴"#, r#"4"#, r#"4"#),
        (r#"⁵"#, r#"⁵"#, r#"⁵"#, r#"5"#, r#"5"#),
        (r#"⁶"#, r#"⁶"#, r#"⁶"#, r#"6"#, r#"6"#),
        (r#"⁷"#, r#"⁷"#, r#"⁷"#, r#"7"#, r#"7"#),
        (r#"⁸"#, r#"⁸"#, r#"⁸"#, r#"8"#, r#"8"#),
        (r#"⁹"#, r#"⁹"#, r#"⁹"#, r#"9"#, r#"9"#),
        (r#"⁺"#, r#"⁺"#, r#"⁺"#, r#"+"#, r#"+"#),
        (r#"⁻"#, r#"⁻"#, r#"⁻"#, r#"−"#, r#"−"#),
        (r#"⁼"#, r#"⁼"#, r#"⁼"#, r#"="#, r#"="#),
        (r#"⁽"#, r#"⁽"#, r#"⁽"#, r#"("#, r#"("#),
        (r#"⁾"#, r#"⁾"#, r#"⁾"#, r#")"#, r#")"#),
        (r#"ⁿ"#, r#"ⁿ"#, r#"ⁿ"#, r#"n"#, r#"n"#),
        (r#"₀"#, r#"₀"#, r#"₀"#, r#"0"#, r#"0"#),
        (r#"₁"#, r#"₁"#, r#"₁"#, r#"1"#, r#"1"#),
        (r#"₂"#, r#"₂"#, r#"₂"#, r#"2"#, r#"2"#),
        (r#"₃"#, r#"₃"#, r#"₃"#, r#"3"#, r#"3"#),
        (r#"₄"#, r#"₄"#, r#"₄"#, r#"4"#, r#"4"#),
        (r#"₅"#, r#"₅"#, r#"₅"#, r#"5"#, r#"5"#),
        (r#"₆"#, r#"₆"#, r#"₆"#, r#"6"#, r#"6"#),
        (r#"₇"#, r#"₇"#, r#"₇"#, r#"7"#, r#"7"#),
        (r#"₈"#, r#"₈"#, r#"₈"#, r#"8"#, r#"8"#),
        (r#"₉"#, r#"₉"#, r#"₉"#, r#"9"#, r#"9"#),
        (r#"₊"#, r#"₊"#, r#"₊"#, r#"+"#, r#"+"#),
        (r#"₋"#, r#"₋"#, r#"₋"#, r#"−"#, r#"−"#),
        (r#"₌"#, r#"₌"#, r#"₌"#, r#"="#, r#"="#),
        (r#"₍"#, r#"₍"#, r#"₍"#, r#"("#, r#"("#),
        (r#"₎"#, r#"₎"#, r#"₎"#, r#")"#, r#")"#),
        (r#"ₐ"#, r#"ₐ"#, r#"ₐ"#, r#"a"#, r#"a"#),
        (r#"ₑ"#, r#"ₑ"#, r#"ₑ"#, r#"e"#, r#"e"#),
        (r#"ₒ"#, r#"ₒ"#, r#"ₒ"#, r#"o"#, r#"o"#),
        (r#"ₓ"#, r#"ₓ"#, r#"ₓ"#, r#"x"#, r#"x"#),
        (r#"ₔ"#, r#"ₔ"#, r#"ₔ"#, r#"ə"#, r#"ə"#),
        (r#"ₕ"#, r#"ₕ"#, r#"ₕ"#, r#"h"#, r#"h"#),
        (r#"ₖ"#, r#"ₖ"#, r#"ₖ"#, r#"k"#, r#"k"#),
        (r#"ₗ"#, r#"ₗ"#, r#"ₗ"#, r#"l"#, r#"l"#),
        (r#"ₘ"#, r#"ₘ"#, r#"ₘ"#, r#"m"#, r#"m"#),
        (r#"ₙ"#, r#"ₙ"#, r#"ₙ"#, r#"n"#, r#"n"#),
        (r#"ₚ"#, r#"ₚ"#, r#"ₚ"#, r#"p"#, r#"p"#),
        (r#"ₛ"#, r#"ₛ"#, r#"ₛ"#, r#"s"#, r#"s"#),
        (r#"ₜ"#, r#"ₜ"#, r#"ₜ"#, r#"t"#, r#"t"#),
        (r#"₨"#, r#"₨"#, r#"₨"#, r#"Rs"#, r#"Rs"#),
        (r#"℀"#, r#"℀"#, r#"℀"#, r#"a/c"#, r#"a/c"#),
        (r#"℁"#, r#"℁"#, r#"℁"#, r#"a/s"#, r#"a/s"#),
        (r#"ℂ"#, r#"ℂ"#, r#"ℂ"#, r#"C"#, r#"C"#),
        (r#"℃"#, r#"℃"#, r#"℃"#, r#"°C"#, r#"°C"#),
        (r#"℅"#, r#"℅"#, r#"℅"#, r#"c/o"#, r#"c/o"#),
        (r#"℆"#, r#"℆"#, r#"℆"#, r#"c/u"#, r#"c/u"#),
        (r#"ℇ"#, r#"ℇ"#, r#"ℇ"#, r#"Ɛ"#, r#"Ɛ"#),
        (r#"℉"#, r#"℉"#, r#"℉"#, r#"°F"#, r#"°F"#),
        (r#"ℊ"#, r#"ℊ"#, r#"ℊ"#, r#"g"#, r#"g"#),
        (r#"ℋ"#, r#"ℋ"#, r#"ℋ"#, r#"H"#, r#"H"#),
        (r#"ℌ"#, r#"ℌ"#, r#"ℌ"#, r#"H"#, r#"H"#),
        (r#"ℍ"#, r#"ℍ"#, r#"ℍ"#, r#"H"#, r#"H"#),
        (r#"ℎ"#, r#"ℎ"#, r#"ℎ"#, r#"h"#, r#"h"#),
        (r#"ℏ"#, r#"ℏ"#, r#"ℏ"#, r#"ħ"#, r#"ħ"#),
        (r#"ℐ"#, r#"ℐ"#, r#"ℐ"#, r#"I"#, r#"I"#),
        (r#"ℑ"#, r#"ℑ"#, r#"ℑ"#, r#"I"#, r#"I"#),
        (r#"ℒ"#, r#"ℒ"#, r#"ℒ"#, r#"L"#, r#"L"#),
        (r#"ℓ"#, r#"ℓ"#, r#"ℓ"#, r#"l"#, r#"l"#),
        (r#"ℕ"#, r#"ℕ"#, r#"ℕ"#, r#"N"#, r#"N"#),
        (r#"№"#, r#"№"#, r#"№"#, r#"No"#, r#"No"#),
        (r#"ℙ"#, r#"ℙ"#, r#"ℙ"#, r#"P"#, r#"P"#),
        (r#"ℚ"#, r#"ℚ"#, r#"ℚ"#, r#"Q"#, r#"Q"#),
        (r#"ℛ"#, r#"ℛ"#, r#"ℛ"#, r#"R"#, r#"R"#),
        (r#"ℜ"#, r#"ℜ"#, r#"ℜ"#, r#"R"#, r#"R"#),
        (r#"ℝ"#, r#"ℝ"#, r#"ℝ"#, r#"R"#, r#"R"#),
        (r#"℠"#, r#"℠"#, r#"℠"#, r#"SM"#, r#"SM"#),
        (r#"℡"#, r#"℡"#, r#"℡"#, r#"TEL"#, r#"TEL"#),
        (r#"™"#, r#"™"#, r#"™"#, r#"TM"#, r#"TM"#),
        (r#"ℤ"#, r#"ℤ"#, r#"ℤ"#, r#"Z"#, r#"Z"#),
        (r#"Ω"#, r#"Ω"#, r#"Ω"#, r#"Ω"#, r#"Ω"#),
        (r#"ℨ"#, r#"ℨ"#, r#"ℨ"#, r#"Z"#, r#"Z"#),
        (r#"K"#, r#"K"#, r#"K"#, r#"K"#, r#"K"#),
        (r#"Å"#, r#"Å"#, r#"Å"#, r#"Å"#, r#"Å"#),
        (r#"ℬ"#, r#"ℬ"#, r#"ℬ"#, r#"B"#, r#"B"#),
        (r#"ℭ"#, r#"ℭ"#, r#"ℭ"#, r#"C"#, r#"C"#),
        (r#"ℯ"#, r#"ℯ"#, r#"ℯ"#, r#"e"#, r#"e"#),
        (r#"ℰ"#, r#"ℰ"#, r#"ℰ"#, r#"E"#, r#"E"#),
        (r#"ℱ"#, r#"ℱ"#, r#"ℱ"#, r#"F"#, r#"F"#),
        (r#"ℳ"#, r#"ℳ"#, r#"ℳ"#, r#"M"#, r#"M"#),
        (r#"ℴ"#, r#"ℴ"#, r#"ℴ"#, r#"o"#, r#"o"#),
        (r#"ℵ"#, r#"ℵ"#, r#"ℵ"#, r#"א"#, r#"א"#),
        (r#"ℶ"#, r#"ℶ"#, r#"ℶ"#, r#"ב"#, r#"ב"#),
        (r#"ℷ"#, r#"ℷ"#, r#"ℷ"#, r#"ג"#, r#"ג"#),
        (r#"ℸ"#, r#"ℸ"#, r#"ℸ"#, r#"ד"#, r#"ד"#),
        (r#"ℹ"#, r#"ℹ"#, r#"ℹ"#, r#"i"#, r#"i"#),
        (r#"℻"#, r#"℻"#, r#"℻"#, r#"FAX"#, r#"FAX"#),
        (r#"ℼ"#, r#"ℼ"#, r#"ℼ"#, r#"π"#, r#"π"#),
        (r#"ℽ"#, r#"ℽ"#, r#"ℽ"#, r#"γ"#, r#"γ"#),
        (r#"ℾ"#, r#"ℾ"#, r#"ℾ"#, r#"Γ"#, r#"Γ"#),
        (r#"ℿ"#, r#"ℿ"#, r#"ℿ"#, r#"Π"#, r#"Π"#),
        (r#"⅀"#, r#"⅀"#, r#"⅀"#, r#"∑"#, r#"∑"#),
        (r#"ⅅ"#, r#"ⅅ"#, r#"ⅅ"#, r#"D"#, r#"D"#),
        (r#"ⅆ"#, r#"ⅆ"#, r#"ⅆ"#, r#"d"#, r#"d"#),
        (r#"ⅇ"#, r#"ⅇ"#, r#"ⅇ"#, r#"e"#, r#"e"#),
        (r#"ⅈ"#, r#"ⅈ"#, r#"ⅈ"#, r#"i"#, r#"i"#),
        (r#"ⅉ"#, r#"ⅉ"#, r#"ⅉ"#, r#"j"#, r#"j"#),
        (r#"⅐"#, r#"⅐"#, r#"⅐"#, r#"1⁄7"#, r#"1⁄7"#),
        (r#"⅑"#, r#"⅑"#, r#"⅑"#, r#"1⁄9"#, r#"1⁄9"#),
        (r#"⅒"#, r#"⅒"#, r#"⅒"#, r#"1⁄10"#, r#"1⁄10"#),
        (r#"⅓"#, r#"⅓"#, r#"⅓"#, r#"1⁄3"#, r#"1⁄3"#),
        (r#"⅔"#, r#"⅔"#, r#"⅔"#, r#"2⁄3"#, r#"2⁄3"#),
        (r#"⅕"#, r#"⅕"#, r#"⅕"#, r#"1⁄5"#, r#"1⁄5"#),
        (r#"⅖"#, r#"⅖"#, r#"⅖"#, r#"2⁄5"#, r#"2⁄5"#),
        (r#"⅗"#, r#"⅗"#, r#"⅗"#, r#"3⁄5"#, r#"3⁄5"#),
        (r#"⅘"#, r#"⅘"#, r#"⅘"#, r#"4⁄5"#, r#"4⁄5"#),
        (r#"⅙"#, r#"⅙"#, r#"⅙"#, r#"1⁄6"#, r#"1⁄6"#),
        (r#"⅚"#, r#"⅚"#, r#"⅚"#, r#"5⁄6"#, r#"5⁄6"#),
        (r#"⅛"#, r#"⅛"#, r#"⅛"#, r#"1⁄8"#, r#"1⁄8"#),
        (r#"⅜"#, r#"⅜"#, r#"⅜"#, r#"3⁄8"#, r#"3⁄8"#),
        (r#"⅝"#, r#"⅝"#, r#"⅝"#, r#"5⁄8"#, r#"5⁄8"#),
        (r#"⅞"#, r#"⅞"#, r#"⅞"#, r#"7⁄8"#, r#"7⁄8"#),
        (r#"⅟"#, r#"⅟"#, r#"⅟"#, r#"1⁄"#, r#"1⁄"#),
        (r#"Ⅰ"#, r#"Ⅰ"#, r#"Ⅰ"#, r#"I"#, r#"I"#),
        (r#"Ⅱ"#, r#"Ⅱ"#, r#"Ⅱ"#, r#"II"#, r#"II"#),
        (r#"Ⅲ"#, r#"Ⅲ"#, r#"Ⅲ"#, r#"III"#, r#"III"#),
        (r#"Ⅳ"#, r#"Ⅳ"#, r#"Ⅳ"#, r#"IV"#, r#"IV"#),
        (r#"Ⅴ"#, r#"Ⅴ"#, r#"Ⅴ"#, r#"V"#, r#"V"#),
        (r#"Ⅵ"#, r#"Ⅵ"#, r#"Ⅵ"#, r#"VI"#, r#"VI"#),
        (r#"Ⅶ"#, r#"Ⅶ"#, r#"Ⅶ"#, r#"VII"#, r#"VII"#),
        (r#"Ⅷ"#, r#"Ⅷ"#, r#"Ⅷ"#, r#"VIII"#, r#"VIII"#),
        (r#"Ⅸ"#, r#"Ⅸ"#, r#"Ⅸ"#, r#"IX"#, r#"IX"#),
        (r#"Ⅹ"#, r#"Ⅹ"#, r#"Ⅹ"#, r#"X"#, r#"X"#),
        (r#"Ⅺ"#, r#"Ⅺ"#, r#"Ⅺ"#, r#"XI"#, r#"XI"#),
        (r#"Ⅻ"#, r#"Ⅻ"#, r#"Ⅻ"#, r#"XII"#, r#"XII"#),
        (r#"Ⅼ"#, r#"Ⅼ"#, r#"Ⅼ"#, r#"L"#, r#"L"#),
        (r#"Ⅽ"#, r#"Ⅽ"#, r#"Ⅽ"#, r#"C"#, r#"C"#),
        (r#"Ⅾ"#, r#"Ⅾ"#, r#"Ⅾ"#, r#"D"#, r#"D"#),
        (r#"Ⅿ"#, r#"Ⅿ"#, r#"Ⅿ"#, r#"M"#, r#"M"#),
        (r#"ⅰ"#, r#"ⅰ"#, r#"ⅰ"#, r#"i"#, r#"i"#),
        (r#"ⅱ"#, r#"ⅱ"#, r#"ⅱ"#, r#"ii"#, r#"ii"#),
        (r#"ⅲ"#, r#"ⅲ"#, r#"ⅲ"#, r#"iii"#, r#"iii"#),
        (r#"ⅳ"#, r#"ⅳ"#, r#"ⅳ"#, r#"iv"#, r#"iv"#),
        (r#"ⅴ"#, r#"ⅴ"#, r#"ⅴ"#, r#"v"#, r#"v"#),
        (r#"ⅵ"#, r#"ⅵ"#, r#"ⅵ"#, r#"vi"#, r#"vi"#),
        (r#"ⅶ"#, r#"ⅶ"#, r#"ⅶ"#, r#"vii"#, r#"vii"#),
        (r#"ⅷ"#, r#"ⅷ"#, r#"ⅷ"#, r#"viii"#, r#"viii"#),
        (r#"ⅸ"#, r#"ⅸ"#, r#"ⅸ"#, r#"ix"#, r#"ix"#),
        (r#"ⅹ"#, r#"ⅹ"#, r#"ⅹ"#, r#"x"#, r#"x"#),
        (r#"ⅺ"#, r#"ⅺ"#, r#"ⅺ"#, r#"xi"#, r#"xi"#),
        (r#"ⅻ"#, r#"ⅻ"#, r#"ⅻ"#, r#"xii"#, r#"xii"#),
        (r#"ⅼ"#, r#"ⅼ"#, r#"ⅼ"#, r#"l"#, r#"l"#),
        (r#"ⅽ"#, r#"ⅽ"#, r#"ⅽ"#, r#"c"#, r#"c"#),
        (r#"ⅾ"#, r#"ⅾ"#, r#"ⅾ"#, r#"d"#, r#"d"#),
        (r#"ⅿ"#, r#"ⅿ"#, r#"ⅿ"#, r#"m"#, r#"m"#),
        (r#"↉"#, r#"↉"#, r#"↉"#, r#"0⁄3"#, r#"0⁄3"#),
        (r#"↚"#, r#"↚"#, r#"↚"#, r#"↚"#, r#"↚"#),
        (r#"↛"#, r#"↛"#, r#"↛"#, r#"↛"#, r#"↛"#),
        (r#"↮"#, r#"↮"#, r#"↮"#, r#"↮"#, r#"↮"#),
        (r#"⇍"#, r#"⇍"#, r#"⇍"#, r#"⇍"#, r#"⇍"#),
        (r#"⇎"#, r#"⇎"#, r#"⇎"#, r#"⇎"#, r#"⇎"#),
        (r#"⇏"#, r#"⇏"#, r#"⇏"#, r#"⇏"#, r#"⇏"#),
        (r#"∄"#, r#"∄"#, r#"∄"#, r#"∄"#, r#"∄"#),
        (r#"∉"#, r#"∉"#, r#"∉"#, r#"∉"#, r#"∉"#),
        (r#"∌"#, r#"∌"#, r#"∌"#, r#"∌"#, r#"∌"#),
        (r#"∤"#, r#"∤"#, r#"∤"#, r#"∤"#, r#"∤"#),
        (r#"∦"#, r#"∦"#, r#"∦"#, r#"∦"#, r#"∦"#),
        (r#"∬"#, r#"∬"#, r#"∬"#, r#"∫∫"#, r#"∫∫"#),
        (r#"∭"#, r#"∭"#, r#"∭"#, r#"∫∫∫"#, r#"∫∫∫"#),
        (r#"∯"#, r#"∯"#, r#"∯"#, r#"∮∮"#, r#"∮∮"#),
        (r#"∰"#, r#"∰"#, r#"∰"#, r#"∮∮∮"#, r#"∮∮∮"#),
        (r#"≁"#, r#"≁"#, r#"≁"#, r#"≁"#, r#"≁"#),
        (r#"≄"#, r#"≄"#, r#"≄"#, r#"≄"#, r#"≄"#),
        (r#"≇"#, r#"≇"#, r#"≇"#, r#"≇"#, r#"≇"#),
        (r#"≉"#, r#"≉"#, r#"≉"#, r#"≉"#, r#"≉"#),
        (r#"≠"#, r#"≠"#, r#"≠"#, r#"≠"#, r#"≠"#),
        (r#"≢"#, r#"≢"#, r#"≢"#, r#"≢"#, r#"≢"#),
        (r#"≭"#, r#"≭"#, r#"≭"#, r#"≭"#, r#"≭"#),
        (r#"≮"#, r#"≮"#, r#"≮"#, r#"≮"#, r#"≮"#),
        (r#"≯"#, r#"≯"#, r#"≯"#, r#"≯"#, r#"≯"#),
        (r#"≰"#, r#"≰"#, r#"≰"#, r#"≰"#, r#"≰"#),
        (r#"≱"#, r#"≱"#, r#"≱"#, r#"≱"#, r#"≱"#),
        (r#"≴"#, r#"≴"#, r#"≴"#, r#"≴"#, r#"≴"#),
        (r#"≵"#, r#"≵"#, r#"≵"#, r#"≵"#, r#"≵"#),
        (r#"≸"#, r#"≸"#, r#"≸"#, r#"≸"#, r#"≸"#),
        (r#"≹"#, r#"≹"#, r#"≹"#, r#"≹"#, r#"≹"#),
        (r#"⊀"#, r#"⊀"#, r#"⊀"#, r#"⊀"#, r#"⊀"#),
        (r#"⊁"#, r#"⊁"#, r#"⊁"#, r#"⊁"#, r#"⊁"#),
        (r#"⊄"#, r#"⊄"#, r#"⊄"#, r#"⊄"#, r#"⊄"#),
        (r#"⊅"#, r#"⊅"#, r#"⊅"#, r#"⊅"#, r#"⊅"#),
        (r#"⊈"#, r#"⊈"#, r#"⊈"#, r#"⊈"#, r#"⊈"#),
        (r#"⊉"#, r#"⊉"#, r#"⊉"#, r#"⊉"#, r#"⊉"#),
        (r#"⊬"#, r#"⊬"#, r#"⊬"#, r#"⊬"#, r#"⊬"#),
        (r#"⊭"#, r#"⊭"#, r#"⊭"#, r#"⊭"#, r#"⊭"#),
        (r#"⊮"#, r#"⊮"#, r#"⊮"#, r#"⊮"#, r#"⊮"#),
        (r#"⊯"#, r#"⊯"#, r#"⊯"#, r#"⊯"#, r#"⊯"#),
        (r#"⋠"#, r#"⋠"#, r#"⋠"#, r#"⋠"#, r#"⋠"#),
        (r#"⋡"#, r#"⋡"#, r#"⋡"#, r#"⋡"#, r#"⋡"#),
        (r#"⋢"#, r#"⋢"#, r#"⋢"#, r#"⋢"#, r#"⋢"#),
        (r#"⋣"#, r#"⋣"#, r#"⋣"#, r#"⋣"#, r#"⋣"#),
        (r#"⋪"#, r#"⋪"#, r#"⋪"#, r#"⋪"#, r#"⋪"#),
        (r#"⋫"#, r#"⋫"#, r#"⋫"#, r#"⋫"#, r#"⋫"#),
        (r#"⋬"#, r#"⋬"#, r#"⋬"#, r#"⋬"#, r#"⋬"#),
        (r#"⋭"#, r#"⋭"#, r#"⋭"#, r#"⋭"#, r#"⋭"#),
        (r#"〈"#, r#"〈"#, r#"〈"#, r#"〈"#, r#"〈"#),
        (r#"〉"#, r#"〉"#, r#"〉"#, r#"〉"#, r#"〉"#),
        (r#"①"#, r#"①"#, r#"①"#, r#"1"#, r#"1"#),
        (r#"②"#, r#"②"#, r#"②"#, r#"2"#, r#"2"#),
        (r#"③"#, r#"③"#, r#"③"#, r#"3"#, r#"3"#),
        (r#"④"#, r#"④"#, r#"④"#, r#"4"#, r#"4"#),
        (r#"⑤"#, r#"⑤"#, r#"⑤"#, r#"5"#, r#"5"#),
        (r#"⑥"#, r#"⑥"#, r#"⑥"#, r#"6"#, r#"6"#),
        (r#"⑦"#, r#"⑦"#, r#"⑦"#, r#"7"#, r#"7"#),
        (r#"⑧"#, r#"⑧"#, r#"⑧"#, r#"8"#, r#"8"#),
        (r#"⑨"#, r#"⑨"#, r#"⑨"#, r#"9"#, r#"9"#),
        (r#"⑩"#, r#"⑩"#, r#"⑩"#, r#"10"#, r#"10"#),
        (r#"⑪"#, r#"⑪"#, r#"⑪"#, r#"11"#, r#"11"#),
        (r#"⑫"#, r#"⑫"#, r#"⑫"#, r#"12"#, r#"12"#),
        (r#"⑬"#, r#"⑬"#, r#"⑬"#, r#"13"#, r#"13"#),
        (r#"⑭"#, r#"⑭"#, r#"⑭"#, r#"14"#, r#"14"#),
        (r#"⑮"#, r#"⑮"#, r#"⑮"#, r#"15"#, r#"15"#),
        (r#"⑯"#, r#"⑯"#, r#"⑯"#, r#"16"#, r#"16"#),
        (r#"⑰"#, r#"⑰"#, r#"⑰"#, r#"17"#, r#"17"#),
        (r#"⑱"#, r#"⑱"#, r#"⑱"#, r#"18"#, r#"18"#),
        (r#"⑲"#, r#"⑲"#, r#"⑲"#, r#"19"#, r#"19"#),
        (r#"⑳"#, r#"⑳"#, r#"⑳"#, r#"20"#, r#"20"#),
        (r#"⑴"#, r#"⑴"#, r#"⑴"#, r#"(1)"#, r#"(1)"#),
        (r#"⑵"#, r#"⑵"#, r#"⑵"#, r#"(2)"#, r#"(2)"#),
        (r#"⑶"#, r#"⑶"#, r#"⑶"#, r#"(3)"#, r#"(3)"#),
        (r#"⑷"#, r#"⑷"#, r#"⑷"#, r#"(4)"#, r#"(4)"#),
        (r#"⑸"#, r#"⑸"#, r#"⑸"#, r#"(5)"#, r#"(5)"#),
        (r#"⑹"#, r#"⑹"#, r#"⑹"#, r#"(6)"#, r#"(6)"#),
        (r#"⑺"#, r#"⑺"#, r#"⑺"#, r#"(7)"#, r#"(7)"#),
        (r#"⑻"#, r#"⑻"#, r#"⑻"#, r#"(8)"#, r#"(8)"#),
        (r#"⑼"#, r#"⑼"#, r#"⑼"#, r#"(9)"#, r#"(9)"#),
        (r#"⑽"#, r#"⑽"#, r#"⑽"#, r#"(10)"#, r#"(10)"#),
        (r#"⑾"#, r#"⑾"#, r#"⑾"#, r#"(11)"#, r#"(11)"#),
        (r#"⑿"#, r#"⑿"#, r#"⑿"#, r#"(12)"#, r#"(12)"#),
        (r#"⒀"#, r#"⒀"#, r#"⒀"#, r#"(13)"#, r#"(13)"#),
        (r#"⒁"#, r#"⒁"#, r#"⒁"#, r#"(14)"#, r#"(14)"#),
        (r#"⒂"#, r#"⒂"#, r#"⒂"#, r#"(15)"#, r#"(15)"#),
        (r#"⒃"#, r#"⒃"#, r#"⒃"#, r#"(16)"#, r#"(16)"#),
        (r#"⒄"#, r#"⒄"#, r#"⒄"#, r#"(17)"#, r#"(17)"#),
        (r#"⒅"#, r#"⒅"#, r#"⒅"#, r#"(18)"#, r#"(18)"#),
        (r#"⒆"#, r#"⒆"#, r#"⒆"#, r#"(19)"#, r#"(19)"#),
        (r#"⒇"#, r#"⒇"#, r#"⒇"#, r#"(20)"#, r#"(20)"#),
        (r#"⒈"#, r#"⒈"#, r#"⒈"#, r#"1."#, r#"1."#),
        (r#"⒉"#, r#"⒉"#, r#"⒉"#, r#"2."#, r#"2."#),
        (r#"⒊"#, r#"⒊"#, r#"⒊"#, r#"3."#, r#"3."#),
        (r#"⒋"#, r#"⒋"#, r#"⒋"#, r#"4."#, r#"4."#),
        (r#"⒌"#, r#"⒌"#, r#"⒌"#, r#"5."#, r#"5."#),
        (r#"⒍"#, r#"⒍"#, r#"⒍"#, r#"6."#, r#"6."#),
        (r#"⒎"#, r#"⒎"#, r#"⒎"#, r#"7."#, r#"7."#),
        (r#"⒏"#, r#"⒏"#, r#"⒏"#, r#"8."#, r#"8."#),
        (r#"⒐"#, r#"⒐"#, r#"⒐"#, r#"9."#, r#"9."#),
        (r#"⒑"#, r#"⒑"#, r#"⒑"#, r#"10."#, r#"10."#),
        (r#"⒒"#, r#"⒒"#, r#"⒒"#, r#"11."#, r#"11."#),
        (r#"⒓"#, r#"⒓"#, r#"⒓"#, r#"12."#, r#"12."#),
        (r#"⒔"#, r#"⒔"#, r#"⒔"#, r#"13."#, r#"13."#),
        (r#"⒕"#, r#"⒕"#, r#"⒕"#, r#"14."#, r#"14."#),
        (r#"⒖"#, r#"⒖"#, r#"⒖"#, r#"15."#, r#"15."#),
        (r#"⒗"#, r#"⒗"#, r#"⒗"#, r#"16."#, r#"16."#),
        (r#"⒘"#, r#"⒘"#, r#"⒘"#, r#"17."#, r#"17."#),
        (r#"⒙"#, r#"⒙"#, r#"⒙"#, r#"18."#, r#"18."#),
        (r#"⒚"#, r#"⒚"#, r#"⒚"#, r#"19."#, r#"19."#),
        (r#"⒛"#, r#"⒛"#, r#"⒛"#, r#"20."#, r#"20."#),
        (r#"⒜"#, r#"⒜"#, r#"⒜"#, r#"(a)"#, r#"(a)"#),
        (r#"⒝"#, r#"⒝"#, r#"⒝"#, r#"(b)"#, r#"(b)"#),
        (r#"⒞"#, r#"⒞"#, r#"⒞"#, r#"(c)"#, r#"(c)"#),
        (r#"⒟"#, r#"⒟"#, r#"⒟"#, r#"(d)"#, r#"(d)"#),
        (r#"⒠"#, r#"⒠"#, r#"⒠"#, r#"(e)"#, r#"(e)"#),
        (r#"⒡"#, r#"⒡"#, r#"⒡"#, r#"(f)"#, r#"(f)"#),
        (r#"⒢"#, r#"⒢"#, r#"⒢"#, r#"(g)"#, r#"(g)"#),
        (r#"⒣"#, r#"⒣"#, r#"⒣"#, r#"(h)"#, r#"(h)"#),
        (r#"⒤"#, r#"⒤"#, r#"⒤"#, r#"(i)"#, r#"(i)"#),
        (r#"⒥"#, r#"⒥"#, r#"⒥"#, r#"(j)"#, r#"(j)"#),
        (r#"⒦"#, r#"⒦"#, r#"⒦"#, r#"(k)"#, r#"(k)"#),
        (r#"⒧"#, r#"⒧"#, r#"⒧"#, r#"(l)"#, r#"(l)"#),
        (r#"⒨"#, r#"⒨"#, r#"⒨"#, r#"(m)"#, r#"(m)"#),
        (r#"⒩"#, r#"⒩"#, r#"⒩"#, r#"(n)"#, r#"(n)"#),
        (r#"⒪"#, r#"⒪"#, r#"⒪"#, r#"(o)"#, r#"(o)"#),
        (r#"⒫"#, r#"⒫"#, r#"⒫"#, r#"(p)"#, r#"(p)"#),
        (r#"⒬"#, r#"⒬"#, r#"⒬"#, r#"(q)"#, r#"(q)"#),
        (r#"⒭"#, r#"⒭"#, r#"⒭"#, r#"(r)"#, r#"(r)"#),
        (r#"⒮"#, r#"⒮"#, r#"⒮"#, r#"(s)"#, r#"(s)"#),
        (r#"⒯"#, r#"⒯"#, r#"⒯"#, r#"(t)"#, r#"(t)"#),
        (r#"⒰"#, r#"⒰"#, r#"⒰"#, r#"(u)"#, r#"(u)"#),
        (r#"⒱"#, r#"⒱"#, r#"⒱"#, r#"(v)"#, r#"(v)"#),
        (r#"⒲"#, r#"⒲"#, r#"⒲"#, r#"(w)"#, r#"(w)"#),
        (r#"⒳"#, r#"⒳"#, r#"⒳"#, r#"(x)"#, r#"(x)"#),
        (r#"⒴"#, r#"⒴"#, r#"⒴"#, r#"(y)"#, r#"(y)"#),
        (r#"⒵"#, r#"⒵"#, r#"⒵"#, r#"(z)"#, r#"(z)"#),
        (r#"Ⓐ"#, r#"Ⓐ"#, r#"Ⓐ"#, r#"A"#, r#"A"#),
        (r#"Ⓑ"#, r#"Ⓑ"#, r#"Ⓑ"#, r#"B"#, r#"B"#),
        (r#"Ⓒ"#, r#"Ⓒ"#, r#"Ⓒ"#, r#"C"#, r#"C"#),
        (r#"Ⓓ"#, r#"Ⓓ"#, r#"Ⓓ"#, r#"D"#, r#"D"#),
        (r#"Ⓔ"#, r#"Ⓔ"#, r#"Ⓔ"#, r#"E"#, r#"E"#),
        (r#"Ⓕ"#, r#"Ⓕ"#, r#"Ⓕ"#, r#"F"#, r#"F"#),
        (r#"Ⓖ"#, r#"Ⓖ"#, r#"Ⓖ"#, r#"G"#, r#"G"#),
        (r#"Ⓗ"#, r#"Ⓗ"#, r#"Ⓗ"#, r#"H"#, r#"H"#),
        (r#"Ⓘ"#, r#"Ⓘ"#, r#"Ⓘ"#, r#"I"#, r#"I"#),
        (r#"Ⓙ"#, r#"Ⓙ"#, r#"Ⓙ"#, r#"J"#, r#"J"#),
        (r#"Ⓚ"#, r#"Ⓚ"#, r#"Ⓚ"#, r#"K"#, r#"K"#),
        (r#"Ⓛ"#, r#"Ⓛ"#, r#"Ⓛ"#, r#"L"#, r#"L"#),
        (r#"Ⓜ"#, r#"Ⓜ"#, r#"Ⓜ"#, r#"M"#, r#"M"#),
        (r#"Ⓝ"#, r#"Ⓝ"#, r#"Ⓝ"#, r#"N"#, r#"N"#),
        (r#"Ⓞ"#, r#"Ⓞ"#, r#"Ⓞ"#, r#"O"#, r#"O"#),
        (r#"Ⓟ"#, r#"Ⓟ"#, r#"Ⓟ"#, r#"P"#, r#"P"#),
        (r#"Ⓠ"#, r#"Ⓠ"#, r#"Ⓠ"#, r#"Q"#, r#"Q"#),
        (r#"Ⓡ"#, r#"Ⓡ"#, r#"Ⓡ"#, r#"R"#, r#"R"#),
        (r#"Ⓢ"#, r#"Ⓢ"#, r#"Ⓢ"#, r#"S"#, r#"S"#),
        (r#"Ⓣ"#, r#"Ⓣ"#, r#"Ⓣ"#, r#"T"#, r#"T"#),
        (r#"Ⓤ"#, r#"Ⓤ"#, r#"Ⓤ"#, r#"U"#, r#"U"#),
        (r#"Ⓥ"#, r#"Ⓥ"#, r#"Ⓥ"#, r#"V"#, r#"V"#),
        (r#"Ⓦ"#, r#"Ⓦ"#, r#"Ⓦ"#, r#"W"#, r#"W"#),
        (r#"Ⓧ"#, r#"Ⓧ"#, r#"Ⓧ"#, r#"X"#, r#"X"#),
        (r#"Ⓨ"#, r#"Ⓨ"#, r#"Ⓨ"#, r#"Y"#, r#"Y"#),
        (r#"Ⓩ"#, r#"Ⓩ"#, r#"Ⓩ"#, r#"Z"#, r#"Z"#),
        (r#"ⓐ"#, r#"ⓐ"#, r#"ⓐ"#, r#"a"#, r#"a"#),
        (r#"ⓑ"#, r#"ⓑ"#, r#"ⓑ"#, r#"b"#, r#"b"#),
        (r#"ⓒ"#, r#"ⓒ"#, r#"ⓒ"#, r#"c"#, r#"c"#),
        (r#"ⓓ"#, r#"ⓓ"#, r#"ⓓ"#, r#"d"#, r#"d"#),
        (r#"ⓔ"#, r#"ⓔ"#, r#"ⓔ"#, r#"e"#, r#"e"#),
        (r#"ⓕ"#, r#"ⓕ"#, r#"ⓕ"#, r#"f"#, r#"f"#),
        (r#"ⓖ"#, r#"ⓖ"#, r#"ⓖ"#, r#"g"#, r#"g"#),
        (r#"ⓗ"#, r#"ⓗ"#, r#"ⓗ"#, r#"h"#, r#"h"#),
        (r#"ⓘ"#, r#"ⓘ"#, r#"ⓘ"#, r#"i"#, r#"i"#),
        (r#"ⓙ"#, r#"ⓙ"#, r#"ⓙ"#, r#"j"#, r#"j"#),
        (r#"ⓚ"#, r#"ⓚ"#, r#"ⓚ"#, r#"k"#, r#"k"#),
        (r#"ⓛ"#, r#"ⓛ"#, r#"ⓛ"#, r#"l"#, r#"l"#),
        (r#"ⓜ"#, r#"ⓜ"#, r#"ⓜ"#, r#"m"#, r#"m"#),
        (r#"ⓝ"#, r#"ⓝ"#, r#"ⓝ"#, r#"n"#, r#"n"#),
        (r#"ⓞ"#, r#"ⓞ"#, r#"ⓞ"#, r#"o"#, r#"o"#),
        (r#"ⓟ"#, r#"ⓟ"#, r#"ⓟ"#, r#"p"#, r#"p"#),
        (r#"ⓠ"#, r#"ⓠ"#, r#"ⓠ"#, r#"q"#, r#"q"#),
        (r#"ⓡ"#, r#"ⓡ"#, r#"ⓡ"#, r#"r"#, r#"r"#),
        (r#"ⓢ"#, r#"ⓢ"#, r#"ⓢ"#, r#"s"#, r#"s"#),
        (r#"ⓣ"#, r#"ⓣ"#, r#"ⓣ"#, r#"t"#, r#"t"#),
        (r#"ⓤ"#, r#"ⓤ"#, r#"ⓤ"#, r#"u"#, r#"u"#),
        (r#"ⓥ"#, r#"ⓥ"#, r#"ⓥ"#, r#"v"#, r#"v"#),
        (r#"ⓦ"#, r#"ⓦ"#, r#"ⓦ"#, r#"w"#, r#"w"#),
        (r#"ⓧ"#, r#"ⓧ"#, r#"ⓧ"#, r#"x"#, r#"x"#),
        (r#"ⓨ"#, r#"ⓨ"#, r#"ⓨ"#, r#"y"#, r#"y"#),
        (r#"ⓩ"#, r#"ⓩ"#, r#"ⓩ"#, r#"z"#, r#"z"#),
        (r#"⓪"#, r#"⓪"#, r#"⓪"#, r#"0"#, r#"0"#),
        (r#"⨌"#, r#"⨌"#, r#"⨌"#, r#"∫∫∫∫"#, r#"∫∫∫∫"#),
        (r#"⩴"#, r#"⩴"#, r#"⩴"#, r#"::="#, r#"::="#),
        (r#"⩵"#, r#"⩵"#, r#"⩵"#, r#"=="#, r#"=="#),
        (r#"⩶"#, r#"⩶"#, r#"⩶"#, r#"==="#, r#"==="#),
        (r#"⫝̸"#, r#"⫝̸"#, r#"⫝̸"#, r#"⫝̸"#, r#"⫝̸"#),
        (r#"ⱼ"#, r#"ⱼ"#, r#"ⱼ"#, r#"j"#, r#"j"#),
        (r#"ⱽ"#, r#"ⱽ"#, r#"ⱽ"#, r#"V"#, r#"V"#),
        (r#"ⵯ"#, r#"ⵯ"#, r#"ⵯ"#, r#"ⵡ"#, r#"ⵡ"#),
        (r#"⺟"#, r#"⺟"#, r#"⺟"#, r#"母"#, r#"母"#),
        (r#"⻳"#, r#"⻳"#, r#"⻳"#, r#"龟"#, r#"龟"#),
        (r#"⼀"#, r#"⼀"#, r#"⼀"#, r#"一"#, r#"一"#),
        (r#"⼁"#, r#"⼁"#, r#"⼁"#, r#"丨"#, r#"丨"#),
        (r#"⼂"#, r#"⼂"#, r#"⼂"#, r#"丶"#, r#"丶"#),
        (r#"⼃"#, r#"⼃"#, r#"⼃"#, r#"丿"#, r#"丿"#),
        (r#"⼄"#, r#"⼄"#, r#"⼄"#, r#"乙"#, r#"乙"#),
        (r#"⼅"#, r#"⼅"#, r#"⼅"#, r#"亅"#, r#"亅"#),
        (r#"⼆"#, r#"⼆"#, r#"⼆"#, r#"二"#, r#"二"#),
        (r#"⼇"#, r#"⼇"#, r#"⼇"#, r#"亠"#, r#"亠"#),
        (r#"⼈"#, r#"⼈"#, r#"⼈"#, r#"人"#, r#"人"#),
        (r#"⼉"#, r#"⼉"#, r#"⼉"#, r#"儿"#, r#"儿"#),
        (r#"⼊"#, r#"⼊"#, r#"⼊"#, r#"入"#, r#"入"#),
        (r#"⼋"#, r#"⼋"#, r#"⼋"#, r#"八"#, r#"八"#),
        (r#"⼌"#, r#"⼌"#, r#"⼌"#, r#"冂"#, r#"冂"#),
        (r#"⼍"#, r#"⼍"#, r#"⼍"#, r#"冖"#, r#"冖"#),
        (r#"⼎"#, r#"⼎"#, r#"⼎"#, r#"冫"#, r#"冫"#),
        (r#"⼏"#, r#"⼏"#, r#"⼏"#, r#"几"#, r#"几"#),
        (r#"⼐"#, r#"⼐"#, r#"⼐"#, r#"凵"#, r#"凵"#),
        (r#"⼑"#, r#"⼑"#, r#"⼑"#, r#"刀"#, r#"刀"#),
        (r#"⼒"#, r#"⼒"#, r#"⼒"#, r#"力"#, r#"力"#),
        (r#"⼓"#, r#"⼓"#, r#"⼓"#, r#"勹"#, r#"勹"#),
        (r#"⼔"#, r#"⼔"#, r#"⼔"#, r#"匕"#, r#"匕"#),
        (r#"⼕"#, r#"⼕"#, r#"⼕"#, r#"匚"#, r#"匚"#),
        (r#"⼖"#, r#"⼖"#, r#"⼖"#, r#"匸"#, r#"匸"#),
        (r#"⼗"#, r#"⼗"#, r#"⼗"#, r#"十"#, r#"十"#),
        (r#"⼘"#, r#"⼘"#, r#"⼘"#, r#"卜"#, r#"卜"#),
        (r#"⼙"#, r#"⼙"#, r#"⼙"#, r#"卩"#, r#"卩"#),
        (r#"⼚"#, r#"⼚"#, r#"⼚"#, r#"厂"#, r#"厂"#),
        (r#"⼛"#, r#"⼛"#, r#"⼛"#, r#"厶"#, r#"厶"#),
        (r#"⼜"#, r#"⼜"#, r#"⼜"#, r#"又"#, r#"又"#),
        (r#"⼝"#, r#"⼝"#, r#"⼝"#, r#"口"#, r#"口"#),
        (r#"⼞"#, r#"⼞"#, r#"⼞"#, r#"囗"#, r#"囗"#),
        (r#"⼟"#, r#"⼟"#, r#"⼟"#, r#"土"#, r#"土"#),
        (r#"⼠"#, r#"⼠"#, r#"⼠"#, r#"士"#, r#"士"#),
        (r#"⼡"#, r#"⼡"#, r#"⼡"#, r#"夂"#, r#"夂"#),
        (r#"⼢"#, r#"⼢"#, r#"⼢"#, r#"夊"#, r#"夊"#),
        (r#"⼣"#, r#"⼣"#, r#"⼣"#, r#"夕"#, r#"夕"#),
        (r#"⼤"#, r#"⼤"#, r#"⼤"#, r#"大"#, r#"大"#),
        (r#"⼥"#, r#"⼥"#, r#"⼥"#, r#"女"#, r#"女"#),
        (r#"⼦"#, r#"⼦"#, r#"⼦"#, r#"子"#, r#"子"#),
        (r#"⼧"#, r#"⼧"#, r#"⼧"#, r#"宀"#, r#"宀"#),
        (r#"⼨"#, r#"⼨"#, r#"⼨"#, r#"寸"#, r#"寸"#),
        (r#"⼩"#, r#"⼩"#, r#"⼩"#, r#"小"#, r#"小"#),
        (r#"⼪"#, r#"⼪"#, r#"⼪"#, r#"尢"#, r#"尢"#),
        (r#"⼫"#, r#"⼫"#, r#"⼫"#, r#"尸"#, r#"尸"#),
        (r#"⼬"#, r#"⼬"#, r#"⼬"#, r#"屮"#, r#"屮"#),
        (r#"⼭"#, r#"⼭"#, r#"⼭"#, r#"山"#, r#"山"#),
        (r#"⼮"#, r#"⼮"#, r#"⼮"#, r#"巛"#, r#"巛"#),
        (r#"⼯"#, r#"⼯"#, r#"⼯"#, r#"工"#, r#"工"#),
        (r#"⼰"#, r#"⼰"#, r#"⼰"#, r#"己"#, r#"己"#),
        (r#"⼱"#, r#"⼱"#, r#"⼱"#, r#"巾"#, r#"巾"#),
        (r#"⼲"#, r#"⼲"#, r#"⼲"#, r#"干"#, r#"干"#),
        (r#"⼳"#, r#"⼳"#, r#"⼳"#, r#"幺"#, r#"幺"#),
        (r#"⼴"#, r#"⼴"#, r#"⼴"#, r#"广"#, r#"广"#),
        (r#"⼵"#, r#"⼵"#, r#"⼵"#, r#"廴"#, r#"廴"#),
        (r#"⼶"#, r#"⼶"#, r#"⼶"#, r#"廾"#, r#"廾"#),
        (r#"⼷"#, r#"⼷"#, r#"⼷"#, r#"弋"#, r#"弋"#),
        (r#"⼸"#, r#"⼸"#, r#"⼸"#, r#"弓"#, r#"弓"#),
        (r#"⼹"#, r#"⼹"#, r#"⼹"#, r#"彐"#, r#"彐"#),
        (r#"⼺"#, r#"⼺"#, r#"⼺"#, r#"彡"#, r#"彡"#),
        (r#"⼻"#, r#"⼻"#, r#"⼻"#, r#"彳"#, r#"彳"#),
        (r#"⼼"#, r#"⼼"#, r#"⼼"#, r#"心"#, r#"心"#),
        (r#"⼽"#, r#"⼽"#, r#"⼽"#, r#"戈"#, r#"戈"#),
        (r#"⼾"#, r#"⼾"#, r#"⼾"#, r#"戶"#, r#"戶"#),
        (r#"⼿"#, r#"⼿"#, r#"⼿"#, r#"手"#, r#"手"#),
        (r#"⽀"#, r#"⽀"#, r#"⽀"#, r#"支"#, r#"支"#),
        (r#"⽁"#, r#"⽁"#, r#"⽁"#, r#"攴"#, r#"攴"#),
        (r#"⽂"#, r#"⽂"#, r#"⽂"#, r#"文"#, r#"文"#),
        (r#"⽃"#, r#"⽃"#, r#"⽃"#, r#"斗"#, r#"斗"#),
        (r#"⽄"#, r#"⽄"#, r#"⽄"#, r#"斤"#, r#"斤"#),
        (r#"⽅"#, r#"⽅"#, r#"⽅"#, r#"方"#, r#"方"#),
        (r#"⽆"#, r#"⽆"#, r#"⽆"#, r#"无"#, r#"无"#),
        (r#"⽇"#, r#"⽇"#, r#"⽇"#, r#"日"#, r#"日"#),
        (r#"⽈"#, r#"⽈"#, r#"⽈"#, r#"曰"#, r#"曰"#),
        (r#"⽉"#, r#"⽉"#, r#"⽉"#, r#"月"#, r#"月"#),
        (r#"⽊"#, r#"⽊"#, r#"⽊"#, r#"木"#, r#"木"#),
        (r#"⽋"#, r#"⽋"#, r#"⽋"#, r#"欠"#, r#"欠"#),
        (r#"⽌"#, r#"⽌"#, r#"⽌"#, r#"止"#, r#"止"#),
        (r#"⽍"#, r#"⽍"#, r#"⽍"#, r#"歹"#, r#"歹"#),
        (r#"⽎"#, r#"⽎"#, r#"⽎"#, r#"殳"#, r#"殳"#),
        (r#"⽏"#, r#"⽏"#, r#"⽏"#, r#"毋"#, r#"毋"#),
        (r#"⽐"#, r#"⽐"#, r#"⽐"#, r#"比"#, r#"比"#),
        (r#"⽑"#, r#"⽑"#, r#"⽑"#, r#"毛"#, r#"毛"#),
        (r#"⽒"#, r#"⽒"#, r#"⽒"#, r#"氏"#, r#"氏"#),
        (r#"⽓"#, r#"⽓"#, r#"⽓"#, r#"气"#, r#"气"#),
        (r#"⽔"#, r#"⽔"#, r#"⽔"#, r#"水"#, r#"水"#),
        (r#"⽕"#, r#"⽕"#, r#"⽕"#, r#"火"#, r#"火"#),
        (r#"⽖"#, r#"⽖"#, r#"⽖"#, r#"爪"#, r#"爪"#),
        (r#"⽗"#, r#"⽗"#, r#"⽗"#, r#"父"#, r#"父"#),
        (r#"⽘"#, r#"⽘"#, r#"⽘"#, r#"爻"#, r#"爻"#),
        (r#"⽙"#, r#"⽙"#, r#"⽙"#, r#"爿"#, r#"爿"#),
        (r#"⽚"#, r#"⽚"#, r#"⽚"#, r#"片"#, r#"片"#),
        (r#"⽛"#, r#"⽛"#, r#"⽛"#, r#"牙"#, r#"牙"#),
        (r#"⽜"#, r#"⽜"#, r#"⽜"#, r#"牛"#, r#"牛"#),
        (r#"⽝"#, r#"⽝"#, r#"⽝"#, r#"犬"#, r#"犬"#),
        (r#"⽞"#, r#"⽞"#, r#"⽞"#, r#"玄"#, r#"玄"#),
        (r#"⽟"#, r#"⽟"#, r#"⽟"#, r#"玉"#, r#"玉"#),
        (r#"⽠"#, r#"⽠"#, r#"⽠"#, r#"瓜"#, r#"瓜"#),
        (r#"⽡"#, r#"⽡"#, r#"⽡"#, r#"瓦"#, r#"瓦"#),
        (r#"⽢"#, r#"⽢"#, r#"⽢"#, r#"甘"#, r#"甘"#),
        (r#"⽣"#, r#"⽣"#, r#"⽣"#, r#"生"#, r#"生"#),
        (r#"⽤"#, r#"⽤"#, r#"⽤"#, r#"用"#, r#"用"#),
        (r#"⽥"#, r#"⽥"#, r#"⽥"#, r#"田"#, r#"田"#),
        (r#"⽦"#, r#"⽦"#, r#"⽦"#, r#"疋"#, r#"疋"#),
        (r#"⽧"#, r#"⽧"#, r#"⽧"#, r#"疒"#, r#"疒"#),
        (r#"⽨"#, r#"⽨"#, r#"⽨"#, r#"癶"#, r#"癶"#),
        (r#"⽩"#, r#"⽩"#, r#"⽩"#, r#"白"#, r#"白"#),
        (r#"⽪"#, r#"⽪"#, r#"⽪"#, r#"皮"#, r#"皮"#),
        (r#"⽫"#, r#"⽫"#, r#"⽫"#, r#"皿"#, r#"皿"#),
        (r#"⽬"#, r#"⽬"#, r#"⽬"#, r#"目"#, r#"目"#),
        (r#"⽭"#, r#"⽭"#, r#"⽭"#, r#"矛"#, r#"矛"#),
        (r#"⽮"#, r#"⽮"#, r#"⽮"#, r#"矢"#, r#"矢"#),
        (r#"⽯"#, r#"⽯"#, r#"⽯"#, r#"石"#, r#"石"#),
        (r#"⽰"#, r#"⽰"#, r#"⽰"#, r#"示"#, r#"示"#),
        (r#"⽱"#, r#"⽱"#, r#"⽱"#, r#"禸"#, r#"禸"#),
        (r#"⽲"#, r#"⽲"#, r#"⽲"#, r#"禾"#, r#"禾"#),
        (r#"⽳"#, r#"⽳"#, r#"⽳"#, r#"穴"#, r#"穴"#),
        (r#"⽴"#, r#"⽴"#, r#"⽴"#, r#"立"#, r#"立"#),
        (r#"⽵"#, r#"⽵"#, r#"⽵"#, r#"竹"#, r#"竹"#),
        (r#"⽶"#, r#"⽶"#, r#"⽶"#, r#"米"#, r#"米"#),
        (r#"⽷"#, r#"⽷"#, r#"⽷"#, r#"糸"#, r#"糸"#),
        (r#"⽸"#, r#"⽸"#, r#"⽸"#, r#"缶"#, r#"缶"#),
        (r#"⽹"#, r#"⽹"#, r#"⽹"#, r#"网"#, r#"网"#),
        (r#"⽺"#, r#"⽺"#, r#"⽺"#, r#"羊"#, r#"羊"#),
        (r#"⽻"#, r#"⽻"#, r#"⽻"#, r#"羽"#, r#"羽"#),
        (r#"⽼"#, r#"⽼"#, r#"⽼"#, r#"老"#, r#"老"#),
        (r#"⽽"#, r#"⽽"#, r#"⽽"#, r#"而"#, r#"而"#),
        (r#"⽾"#, r#"⽾"#, r#"⽾"#, r#"耒"#, r#"耒"#),
        (r#"⽿"#, r#"⽿"#, r#"⽿"#, r#"耳"#, r#"耳"#),
        (r#"⾀"#, r#"⾀"#, r#"⾀"#, r#"聿"#, r#"聿"#),
        (r#"⾁"#, r#"⾁"#, r#"⾁"#, r#"肉"#, r#"肉"#),
        (r#"⾂"#, r#"⾂"#, r#"⾂"#, r#"臣"#, r#"臣"#),
        (r#"⾃"#, r#"⾃"#, r#"⾃"#, r#"自"#, r#"自"#),
        (r#"⾄"#, r#"⾄"#, r#"⾄"#, r#"至"#, r#"至"#),
        (r#"⾅"#, r#"⾅"#, r#"⾅"#, r#"臼"#, r#"臼"#),
        (r#"⾆"#, r#"⾆"#, r#"⾆"#, r#"舌"#, r#"舌"#),
        (r#"⾇"#, r#"⾇"#, r#"⾇"#, r#"舛"#, r#"舛"#),
        (r#"⾈"#, r#"⾈"#, r#"⾈"#, r#"舟"#, r#"舟"#),
        (r#"⾉"#, r#"⾉"#, r#"⾉"#, r#"艮"#, r#"艮"#),
        (r#"⾊"#, r#"⾊"#, r#"⾊"#, r#"色"#, r#"色"#),
        (r#"⾋"#, r#"⾋"#, r#"⾋"#, r#"艸"#, r#"艸"#),
        (r#"⾌"#, r#"⾌"#, r#"⾌"#, r#"虍"#, r#"虍"#),
        (r#"⾍"#, r#"⾍"#, r#"⾍"#, r#"虫"#, r#"虫"#),
        (r#"⾎"#, r#"⾎"#, r#"⾎"#, r#"血"#, r#"血"#),
        (r#"⾏"#, r#"⾏"#, r#"⾏"#, r#"行"#, r#"行"#),
        (r#"⾐"#, r#"⾐"#, r#"⾐"#, r#"衣"#, r#"衣"#),
        (r#"⾑"#, r#"⾑"#, r#"⾑"#, r#"襾"#, r#"襾"#),
        (r#"⾒"#, r#"⾒"#, r#"⾒"#, r#"見"#, r#"見"#),
        (r#"⾓"#, r#"⾓"#, r#"⾓"#, r#"角"#, r#"角"#),
        (r#"⾔"#, r#"⾔"#, r#"⾔"#, r#"言"#, r#"言"#),
        (r#"⾕"#, r#"⾕"#, r#"⾕"#, r#"谷"#, r#"谷"#),
        (r#"⾖"#, r#"⾖"#, r#"⾖"#, r#"豆"#, r#"豆"#),
        (r#"⾗"#, r#"⾗"#, r#"⾗"#, r#"豕"#, r#"豕"#),
        (r#"⾘"#, r#"⾘"#, r#"⾘"#, r#"豸"#, r#"豸"#),
        (r#"⾙"#, r#"⾙"#, r#"⾙"#, r#"貝"#, r#"貝"#),
        (r#"⾚"#, r#"⾚"#, r#"⾚"#, r#"赤"#, r#"赤"#),
        (r#"⾛"#, r#"⾛"#, r#"⾛"#, r#"走"#, r#"走"#),
        (r#"⾜"#, r#"⾜"#, r#"⾜"#, r#"足"#, r#"足"#),
        (r#"⾝"#, r#"⾝"#, r#"⾝"#, r#"身"#, r#"身"#),
        (r#"⾞"#, r#"⾞"#, r#"⾞"#, r#"車"#, r#"車"#),
        (r#"⾟"#, r#"⾟"#, r#"⾟"#, r#"辛"#, r#"辛"#),
        (r#"⾠"#, r#"⾠"#, r#"⾠"#, r#"辰"#, r#"辰"#),
        (r#"⾡"#, r#"⾡"#, r#"⾡"#, r#"辵"#, r#"辵"#),
        (r#"⾢"#, r#"⾢"#, r#"⾢"#, r#"邑"#, r#"邑"#),
        (r#"⾣"#, r#"⾣"#, r#"⾣"#, r#"酉"#, r#"酉"#),
        (r#"⾤"#, r#"⾤"#, r#"⾤"#, r#"釆"#, r#"釆"#),
        (r#"⾥"#, r#"⾥"#, r#"⾥"#, r#"里"#, r#"里"#),
        (r#"⾦"#, r#"⾦"#, r#"⾦"#, r#"金"#, r#"金"#),
        (r#"⾧"#, r#"⾧"#, r#"⾧"#, r#"長"#, r#"長"#),
        (r#"⾨"#, r#"⾨"#, r#"⾨"#, r#"門"#, r#"門"#),
        (r#"⾩"#, r#"⾩"#, r#"⾩"#, r#"阜"#, r#"阜"#),
        (r#"⾪"#, r#"⾪"#, r#"⾪"#, r#"隶"#, r#"隶"#),
        (r#"⾫"#, r#"⾫"#, r#"⾫"#, r#"隹"#, r#"隹"#),
        (r#"⾬"#, r#"⾬"#, r#"⾬"#, r#"雨"#, r#"雨"#),
        (r#"⾭"#, r#"⾭"#, r#"⾭"#, r#"靑"#, r#"靑"#),
        (r#"⾮"#, r#"⾮"#, r#"⾮"#, r#"非"#, r#"非"#),
        (r#"⾯"#, r#"⾯"#, r#"⾯"#, r#"面"#, r#"面"#),
        (r#"⾰"#, r#"⾰"#, r#"⾰"#, r#"革"#, r#"革"#),
        (r#"⾱"#, r#"⾱"#, r#"⾱"#, r#"韋"#, r#"韋"#),
        (r#"⾲"#, r#"⾲"#, r#"⾲"#, r#"韭"#, r#"韭"#),
        (r#"⾳"#, r#"⾳"#, r#"⾳"#, r#"音"#, r#"音"#),
        (r#"⾴"#, r#"⾴"#, r#"⾴"#, r#"頁"#, r#"頁"#),
        (r#"⾵"#, r#"⾵"#, r#"⾵"#, r#"風"#, r#"風"#),
        (r#"⾶"#, r#"⾶"#, r#"⾶"#, r#"飛"#, r#"飛"#),
        (r#"⾷"#, r#"⾷"#, r#"⾷"#, r#"食"#, r#"食"#),
        (r#"⾸"#, r#"⾸"#, r#"⾸"#, r#"首"#, r#"首"#),
        (r#"⾹"#, r#"⾹"#, r#"⾹"#, r#"香"#, r#"香"#),
        (r#"⾺"#, r#"⾺"#, r#"⾺"#, r#"馬"#, r#"馬"#),
        (r#"⾻"#, r#"⾻"#, r#"⾻"#, r#"骨"#, r#"骨"#),
        (r#"⾼"#, r#"⾼"#, r#"⾼"#, r#"高"#, r#"高"#),
        (r#"⾽"#, r#"⾽"#, r#"⾽"#, r#"髟"#, r#"髟"#),
        (r#"⾾"#, r#"⾾"#, r#"⾾"#, r#"鬥"#, r#"鬥"#),
        (r#"⾿"#, r#"⾿"#, r#"⾿"#, r#"鬯"#, r#"鬯"#),
        (r#"⿀"#, r#"⿀"#, r#"⿀"#, r#"鬲"#, r#"鬲"#),
        (r#"⿁"#, r#"⿁"#, r#"⿁"#, r#"鬼"#, r#"鬼"#),
        (r#"⿂"#, r#"⿂"#, r#"⿂"#, r#"魚"#, r#"魚"#),
        (r#"⿃"#, r#"⿃"#, r#"⿃"#, r#"鳥"#, r#"鳥"#),
        (r#"⿄"#, r#"⿄"#, r#"⿄"#, r#"鹵"#, r#"鹵"#),
        (r#"⿅"#, r#"⿅"#, r#"⿅"#, r#"鹿"#, r#"鹿"#),
        (r#"⿆"#, r#"⿆"#, r#"⿆"#, r#"麥"#, r#"麥"#),
        (r#"⿇"#, r#"⿇"#, r#"⿇"#, r#"麻"#, r#"麻"#),
        (r#"⿈"#, r#"⿈"#, r#"⿈"#, r#"黃"#, r#"黃"#),
        (r#"⿉"#, r#"⿉"#, r#"⿉"#, r#"黍"#, r#"黍"#),
        (r#"⿊"#, r#"⿊"#, r#"⿊"#, r#"黑"#, r#"黑"#),
        (r#"⿋"#, r#"⿋"#, r#"⿋"#, r#"黹"#, r#"黹"#),
        (r#"⿌"#, r#"⿌"#, r#"⿌"#, r#"黽"#, r#"黽"#),
        (r#"⿍"#, r#"⿍"#, r#"⿍"#, r#"鼎"#, r#"鼎"#),
        (r#"⿎"#, r#"⿎"#, r#"⿎"#, r#"鼓"#, r#"鼓"#),
        (r#"⿏"#, r#"⿏"#, r#"⿏"#, r#"鼠"#, r#"鼠"#),
        (r#"⿐"#, r#"⿐"#, r#"⿐"#, r#"鼻"#, r#"鼻"#),
        (r#"⿑"#, r#"⿑"#, r#"⿑"#, r#"齊"#, r#"齊"#),
        (r#"⿒"#, r#"⿒"#, r#"⿒"#, r#"齒"#, r#"齒"#),
        (r#"⿓"#, r#"⿓"#, r#"⿓"#, r#"龍"#, r#"龍"#),
        (r#"⿔"#, r#"⿔"#, r#"⿔"#, r#"龜"#, r#"龜"#),
        (r#"⿕"#, r#"⿕"#, r#"⿕"#, r#"龠"#, r#"龠"#),
        (r#"　"#, r#"　"#, r#"　"#, r#" "#, r#" "#),
        (r#"〶"#, r#"〶"#, r#"〶"#, r#"〒"#, r#"〒"#),
        (r#"〸"#, r#"〸"#, r#"〸"#, r#"十"#, r#"十"#),
        (r#"〹"#, r#"〹"#, r#"〹"#, r#"卄"#, r#"卄"#),
        (r#"〺"#, r#"〺"#, r#"〺"#, r#"卅"#, r#"卅"#),
        (r#"が"#, r#"が"#, r#"が"#, r#"が"#, r#"が"#),
        (r#"ぎ"#, r#"ぎ"#, r#"ぎ"#, r#"ぎ"#, r#"ぎ"#),
        (r#"ぐ"#, r#"ぐ"#, r#"ぐ"#, r#"ぐ"#, r#"ぐ"#),
        (r#"げ"#, r#"げ"#, r#"げ"#, r#"げ"#, r#"げ"#),
        (r#"ご"#, r#"ご"#, r#"ご"#, r#"ご"#, r#"ご"#),
        (r#"ざ"#, r#"ざ"#, r#"ざ"#, r#"ざ"#, r#"ざ"#),
        (r#"じ"#, r#"じ"#, r#"じ"#, r#"じ"#, r#"じ"#),
        (r#"ず"#, r#"ず"#, r#"ず"#, r#"ず"#, r#"ず"#),
        (r#"ぜ"#, r#"ぜ"#, r#"ぜ"#, r#"ぜ"#, r#"ぜ"#),
        (r#"ぞ"#, r#"ぞ"#, r#"ぞ"#, r#"ぞ"#, r#"ぞ"#),
        (r#"だ"#, r#"だ"#, r#"だ"#, r#"だ"#, r#"だ"#),
        (r#"ぢ"#, r#"ぢ"#, r#"ぢ"#, r#"ぢ"#, r#"ぢ"#),
        (r#"づ"#, r#"づ"#, r#"づ"#, r#"づ"#, r#"づ"#),
        (r#"で"#, r#"で"#, r#"で"#, r#"で"#, r#"で"#),
        (r#"ど"#, r#"ど"#, r#"ど"#, r#"ど"#, r#"ど"#),
        (r#"ば"#, r#"ば"#, r#"ば"#, r#"ば"#, r#"ば"#),
        (r#"ぱ"#, r#"ぱ"#, r#"ぱ"#, r#"ぱ"#, r#"ぱ"#),
        (r#"び"#, r#"び"#, r#"び"#, r#"び"#, r#"び"#),
        (r#"ぴ"#, r#"ぴ"#, r#"ぴ"#, r#"ぴ"#, r#"ぴ"#),
        (r#"ぶ"#, r#"ぶ"#, r#"ぶ"#, r#"ぶ"#, r#"ぶ"#),
        (r#"ぷ"#, r#"ぷ"#, r#"ぷ"#, r#"ぷ"#, r#"ぷ"#),
        (r#"べ"#, r#"べ"#, r#"べ"#, r#"べ"#, r#"べ"#),
        (r#"ぺ"#, r#"ぺ"#, r#"ぺ"#, r#"ぺ"#, r#"ぺ"#),
        (r#"ぼ"#, r#"ぼ"#, r#"ぼ"#, r#"ぼ"#, r#"ぼ"#),
        (r#"ぽ"#, r#"ぽ"#, r#"ぽ"#, r#"ぽ"#, r#"ぽ"#),
        (r#"ゔ"#, r#"ゔ"#, r#"ゔ"#, r#"ゔ"#, r#"ゔ"#),
        (r#"゛"#, r#"゛"#, r#"゛"#, r#" ゙"#, r#" ゙"#),
        (r#"゜"#, r#"゜"#, r#"゜"#, r#" ゚"#, r#" ゚"#),
        (r#"ゞ"#, r#"ゞ"#, r#"ゞ"#, r#"ゞ"#, r#"ゞ"#),
        (r#"ゟ"#, r#"ゟ"#, r#"ゟ"#, r#"より"#, r#"より"#),
        (r#"ガ"#, r#"ガ"#, r#"ガ"#, r#"ガ"#, r#"ガ"#),
        (r#"ギ"#, r#"ギ"#, r#"ギ"#, r#"ギ"#, r#"ギ"#),
        (r#"グ"#, r#"グ"#, r#"グ"#, r#"グ"#, r#"グ"#),
        (r#"ゲ"#, r#"ゲ"#, r#"ゲ"#, r#"ゲ"#, r#"ゲ"#),
        (r#"ゴ"#, r#"ゴ"#, r#"ゴ"#, r#"ゴ"#, r#"ゴ"#),
        (r#"ザ"#, r#"ザ"#, r#"ザ"#, r#"ザ"#, r#"ザ"#),
        (r#"ジ"#, r#"ジ"#, r#"ジ"#, r#"ジ"#, r#"ジ"#),
        (r#"ズ"#, r#"ズ"#, r#"ズ"#, r#"ズ"#, r#"ズ"#),
        (r#"ゼ"#, r#"ゼ"#, r#"ゼ"#, r#"ゼ"#, r#"ゼ"#),
        (r#"ゾ"#, r#"ゾ"#, r#"ゾ"#, r#"ゾ"#, r#"ゾ"#),
        (r#"ダ"#, r#"ダ"#, r#"ダ"#, r#"ダ"#, r#"ダ"#),
        (r#"ヂ"#, r#"ヂ"#, r#"ヂ"#, r#"ヂ"#, r#"ヂ"#),
        (r#"ヅ"#, r#"ヅ"#, r#"ヅ"#, r#"ヅ"#, r#"ヅ"#),
        (r#"デ"#, r#"デ"#, r#"デ"#, r#"デ"#, r#"デ"#),
        (r#"ド"#, r#"ド"#, r#"ド"#, r#"ド"#, r#"ド"#),
        (r#"バ"#, r#"バ"#, r#"バ"#, r#"バ"#, r#"バ"#),
        (r#"パ"#, r#"パ"#, r#"パ"#, r#"パ"#, r#"パ"#),
        (r#"ビ"#, r#"ビ"#, r#"ビ"#, r#"ビ"#, r#"ビ"#),
        (r#"ピ"#, r#"ピ"#, r#"ピ"#, r#"ピ"#, r#"ピ"#),
        (r#"ブ"#, r#"ブ"#, r#"ブ"#, r#"ブ"#, r#"ブ"#),
        (r#"プ"#, r#"プ"#, r#"プ"#, r#"プ"#, r#"プ"#),
        (r#"ベ"#, r#"ベ"#, r#"ベ"#, r#"ベ"#, r#"ベ"#),
        (r#"ペ"#, r#"ペ"#, r#"ペ"#, r#"ペ"#, r#"ペ"#),
        (r#"ボ"#, r#"ボ"#, r#"ボ"#, r#"ボ"#, r#"ボ"#),
        (r#"ポ"#, r#"ポ"#, r#"ポ"#, r#"ポ"#, r#"ポ"#),
        (r#"ヴ"#, r#"ヴ"#, r#"ヴ"#, r#"ヴ"#, r#"ヴ"#),
        (r#"ヷ"#, r#"ヷ"#, r#"ヷ"#, r#"ヷ"#, r#"ヷ"#),
        (r#"ヸ"#, r#"ヸ"#, r#"ヸ"#, r#"ヸ"#, r#"ヸ"#),
        (r#"ヹ"#, r#"ヹ"#, r#"ヹ"#, r#"ヹ"#, r#"ヹ"#),
        (r#"ヺ"#, r#"ヺ"#, r#"ヺ"#, r#"ヺ"#, r#"ヺ"#),
        (r#"ヾ"#, r#"ヾ"#, r#"ヾ"#, r#"ヾ"#, r#"ヾ"#),
        (r#"ヿ"#, r#"ヿ"#, r#"ヿ"#, r#"コト"#, r#"コト"#),
        (r#"ㄱ"#, r#"ㄱ"#, r#"ㄱ"#, r#"ᄀ"#, r#"ᄀ"#),
        (r#"ㄲ"#, r#"ㄲ"#, r#"ㄲ"#, r#"ᄁ"#, r#"ᄁ"#),
        (r#"ㄳ"#, r#"ㄳ"#, r#"ㄳ"#, r#"ᆪ"#, r#"ᆪ"#),
        (r#"ㄴ"#, r#"ㄴ"#, r#"ㄴ"#, r#"ᄂ"#, r#"ᄂ"#),
        (r#"ㄵ"#, r#"ㄵ"#, r#"ㄵ"#, r#"ᆬ"#, r#"ᆬ"#),
        (r#"ㄶ"#, r#"ㄶ"#, r#"ㄶ"#, r#"ᆭ"#, r#"ᆭ"#),
        (r#"ㄷ"#, r#"ㄷ"#, r#"ㄷ"#, r#"ᄃ"#, r#"ᄃ"#),
        (r#"ㄸ"#, r#"ㄸ"#, r#"ㄸ"#, r#"ᄄ"#, r#"ᄄ"#),
        (r#"ㄹ"#, r#"ㄹ"#, r#"ㄹ"#, r#"ᄅ"#, r#"ᄅ"#),
        (r#"ㄺ"#, r#"ㄺ"#, r#"ㄺ"#, r#"ᆰ"#, r#"ᆰ"#),
        (r#"ㄻ"#, r#"ㄻ"#, r#"ㄻ"#, r#"ᆱ"#, r#"ᆱ"#),
        (r#"ㄼ"#, r#"ㄼ"#, r#"ㄼ"#, r#"ᆲ"#, r#"ᆲ"#),
        (r#"ㄽ"#, r#"ㄽ"#, r#"ㄽ"#, r#"ᆳ"#, r#"ᆳ"#),
        (r#"ㄾ"#, r#"ㄾ"#, r#"ㄾ"#, r#"ᆴ"#, r#"ᆴ"#),
        (r#"ㄿ"#, r#"ㄿ"#, r#"ㄿ"#, r#"ᆵ"#, r#"ᆵ"#),
        (r#"ㅀ"#, r#"ㅀ"#, r#"ㅀ"#, r#"ᄚ"#, r#"ᄚ"#),
        (r#"ㅁ"#, r#"ㅁ"#, r#"ㅁ"#, r#"ᄆ"#, r#"ᄆ"#),
        (r#"ㅂ"#, r#"ㅂ"#, r#"ㅂ"#, r#"ᄇ"#, r#"ᄇ"#),
        (r#"ㅃ"#, r#"ㅃ"#, r#"ㅃ"#, r#"ᄈ"#, r#"ᄈ"#),
        (r#"ㅄ"#, r#"ㅄ"#, r#"ㅄ"#, r#"ᄡ"#, r#"ᄡ"#),
        (r#"ㅅ"#, r#"ㅅ"#, r#"ㅅ"#, r#"ᄉ"#, r#"ᄉ"#),
        (r#"ㅆ"#, r#"ㅆ"#, r#"ㅆ"#, r#"ᄊ"#, r#"ᄊ"#),
        (r#"ㅇ"#, r#"ㅇ"#, r#"ㅇ"#, r#"ᄋ"#, r#"ᄋ"#),
        (r#"ㅈ"#, r#"ㅈ"#, r#"ㅈ"#, r#"ᄌ"#, r#"ᄌ"#),
        (r#"ㅉ"#, r#"ㅉ"#, r#"ㅉ"#, r#"ᄍ"#, r#"ᄍ"#),
        (r#"ㅊ"#, r#"ㅊ"#, r#"ㅊ"#, r#"ᄎ"#, r#"ᄎ"#),
        (r#"ㅋ"#, r#"ㅋ"#, r#"ㅋ"#, r#"ᄏ"#, r#"ᄏ"#),
        (r#"ㅌ"#, r#"ㅌ"#, r#"ㅌ"#, r#"ᄐ"#, r#"ᄐ"#),
        (r#"ㅍ"#, r#"ㅍ"#, r#"ㅍ"#, r#"ᄑ"#, r#"ᄑ"#),
        (r#"ㅎ"#, r#"ㅎ"#, r#"ㅎ"#, r#"ᄒ"#, r#"ᄒ"#),
        (r#"ㅏ"#, r#"ㅏ"#, r#"ㅏ"#, r#"ᅡ"#, r#"ᅡ"#),
        (r#"ㅐ"#, r#"ㅐ"#, r#"ㅐ"#, r#"ᅢ"#, r#"ᅢ"#),
        (r#"ㅑ"#, r#"ㅑ"#, r#"ㅑ"#, r#"ᅣ"#, r#"ᅣ"#),
        (r#"ㅒ"#, r#"ㅒ"#, r#"ㅒ"#, r#"ᅤ"#, r#"ᅤ"#),
        (r#"ㅓ"#, r#"ㅓ"#, r#"ㅓ"#, r#"ᅥ"#, r#"ᅥ"#),
        (r#"ㅔ"#, r#"ㅔ"#, r#"ㅔ"#, r#"ᅦ"#, r#"ᅦ"#),
        (r#"ㅕ"#, r#"ㅕ"#, r#"ㅕ"#, r#"ᅧ"#, r#"ᅧ"#),
        (r#"ㅖ"#, r#"ㅖ"#, r#"ㅖ"#, r#"ᅨ"#, r#"ᅨ"#),
        (r#"ㅗ"#, r#"ㅗ"#, r#"ㅗ"#, r#"ᅩ"#, r#"ᅩ"#),
        (r#"ㅘ"#, r#"ㅘ"#, r#"ㅘ"#, r#"ᅪ"#, r#"ᅪ"#),
        (r#"ㅙ"#, r#"ㅙ"#, r#"ㅙ"#, r#"ᅫ"#, r#"ᅫ"#),
        (r#"ㅚ"#, r#"ㅚ"#, r#"ㅚ"#, r#"ᅬ"#, r#"ᅬ"#),
        (r#"ㅛ"#, r#"ㅛ"#, r#"ㅛ"#, r#"ᅭ"#, r#"ᅭ"#),
        (r#"ㅜ"#, r#"ㅜ"#, r#"ㅜ"#, r#"ᅮ"#, r#"ᅮ"#),
        (r#"ㅝ"#, r#"ㅝ"#, r#"ㅝ"#, r#"ᅯ"#, r#"ᅯ"#),
        (r#"ㅞ"#, r#"ㅞ"#, r#"ㅞ"#, r#"ᅰ"#, r#"ᅰ"#),
        (r#"ㅟ"#, r#"ㅟ"#, r#"ㅟ"#, r#"ᅱ"#, r#"ᅱ"#),
        (r#"ㅠ"#, r#"ㅠ"#, r#"ㅠ"#, r#"ᅲ"#, r#"ᅲ"#),
        (r#"ㅡ"#, r#"ㅡ"#, r#"ㅡ"#, r#"ᅳ"#, r#"ᅳ"#),
        (r#"ㅢ"#, r#"ㅢ"#, r#"ㅢ"#, r#"ᅴ"#, r#"ᅴ"#),
        (r#"ㅣ"#, r#"ㅣ"#, r#"ㅣ"#, r#"ᅵ"#, r#"ᅵ"#),
        (r#"ㅤ"#, r#"ㅤ"#, r#"ㅤ"#, r#"ᅠ"#, r#"ᅠ"#),
        (r#"ㅥ"#, r#"ㅥ"#, r#"ㅥ"#, r#"ᄔ"#, r#"ᄔ"#),
        (r#"ㅦ"#, r#"ㅦ"#, r#"ㅦ"#, r#"ᄕ"#, r#"ᄕ"#),
        (r#"ㅧ"#, r#"ㅧ"#, r#"ㅧ"#, r#"ᇇ"#, r#"ᇇ"#),
        (r#"ㅨ"#, r#"ㅨ"#, r#"ㅨ"#, r#"ᇈ"#, r#"ᇈ"#),
        (r#"ㅩ"#, r#"ㅩ"#, r#"ㅩ"#, r#"ᇌ"#, r#"ᇌ"#),
        (r#"ㅪ"#, r#"ㅪ"#, r#"ㅪ"#, r#"ᇎ"#, r#"ᇎ"#),
        (r#"ㅫ"#, r#"ㅫ"#, r#"ㅫ"#, r#"ᇓ"#, r#"ᇓ"#),
        (r#"ㅬ"#, r#"ㅬ"#, r#"ㅬ"#, r#"ᇗ"#, r#"ᇗ"#),
        (r#"ㅭ"#, r#"ㅭ"#, r#"ㅭ"#, r#"ᇙ"#, r#"ᇙ"#),
        (r#"ㅮ"#, r#"ㅮ"#, r#"ㅮ"#, r#"ᄜ"#, r#"ᄜ"#),
        (r#"ㅯ"#, r#"ㅯ"#, r#"ㅯ"#, r#"ᇝ"#, r#"ᇝ"#),
        (r#"ㅰ"#, r#"ㅰ"#, r#"ㅰ"#, r#"ᇟ"#, r#"ᇟ"#),
        (r#"ㅱ"#, r#"ㅱ"#, r#"ㅱ"#, r#"ᄝ"#, r#"ᄝ"#),
        (r#"ㅲ"#, r#"ㅲ"#, r#"ㅲ"#, r#"ᄞ"#, r#"ᄞ"#),
        (r#"ㅳ"#, r#"ㅳ"#, r#"ㅳ"#, r#"ᄠ"#, r#"ᄠ"#),
        (r#"ㅴ"#, r#"ㅴ"#, r#"ㅴ"#, r#"ᄢ"#, r#"ᄢ"#),
        (r#"ㅵ"#, r#"ㅵ"#, r#"ㅵ"#, r#"ᄣ"#, r#"ᄣ"#),
        (r#"ㅶ"#, r#"ㅶ"#, r#"ㅶ"#, r#"ᄧ"#, r#"ᄧ"#),
        (r#"ㅷ"#, r#"ㅷ"#, r#"ㅷ"#, r#"ᄩ"#, r#"ᄩ"#),
        (r#"ㅸ"#, r#"ㅸ"#, r#"ㅸ"#, r#"ᄫ"#, r#"ᄫ"#),
        (r#"ㅹ"#, r#"ㅹ"#, r#"ㅹ"#, r#"ᄬ"#, r#"ᄬ"#),
        (r#"ㅺ"#, r#"ㅺ"#, r#"ㅺ"#, r#"ᄭ"#, r#"ᄭ"#),
        (r#"ㅻ"#, r#"ㅻ"#, r#"ㅻ"#, r#"ᄮ"#, r#"ᄮ"#),
        (r#"ㅼ"#, r#"ㅼ"#, r#"ㅼ"#, r#"ᄯ"#, r#"ᄯ"#),
        (r#"ㅽ"#, r#"ㅽ"#, r#"ㅽ"#, r#"ᄲ"#, r#"ᄲ"#),
        (r#"ㅾ"#, r#"ㅾ"#, r#"ㅾ"#, r#"ᄶ"#, r#"ᄶ"#),
        (r#"ㅿ"#, r#"ㅿ"#, r#"ㅿ"#, r#"ᅀ"#, r#"ᅀ"#),
        (r#"ㆀ"#, r#"ㆀ"#, r#"ㆀ"#, r#"ᅇ"#, r#"ᅇ"#),
        (r#"ㆁ"#, r#"ㆁ"#, r#"ㆁ"#, r#"ᅌ"#, r#"ᅌ"#),
        (r#"ㆂ"#, r#"ㆂ"#, r#"ㆂ"#, r#"ᇱ"#, r#"ᇱ"#),
        (r#"ㆃ"#, r#"ㆃ"#, r#"ㆃ"#, r#"ᇲ"#, r#"ᇲ"#),
        (r#"ㆄ"#, r#"ㆄ"#, r#"ㆄ"#, r#"ᅗ"#, r#"ᅗ"#),
        (r#"ㆅ"#, r#"ㆅ"#, r#"ㆅ"#, r#"ᅘ"#, r#"ᅘ"#),
        (r#"ㆆ"#, r#"ㆆ"#, r#"ㆆ"#, r#"ᅙ"#, r#"ᅙ"#),
        (r#"ㆇ"#, r#"ㆇ"#, r#"ㆇ"#, r#"ᆄ"#, r#"ᆄ"#),
        (r#"ㆈ"#, r#"ㆈ"#, r#"ㆈ"#, r#"ᆅ"#, r#"ᆅ"#),
        (r#"ㆉ"#, r#"ㆉ"#, r#"ㆉ"#, r#"ᆈ"#, r#"ᆈ"#),
        (r#"ㆊ"#, r#"ㆊ"#, r#"ㆊ"#, r#"ᆑ"#, r#"ᆑ"#),
        (r#"ㆋ"#, r#"ㆋ"#, r#"ㆋ"#, r#"ᆒ"#, r#"ᆒ"#),
        (r#"ㆌ"#, r#"ㆌ"#, r#"ㆌ"#, r#"ᆔ"#, r#"ᆔ"#),
        (r#"ㆍ"#, r#"ㆍ"#, r#"ㆍ"#, r#"ᆞ"#, r#"ᆞ"#),
        (r#"ㆎ"#, r#"ㆎ"#, r#"ㆎ"#, r#"ᆡ"#, r#"ᆡ"#),
        (r#"㆒"#, r#"㆒"#, r#"㆒"#, r#"一"#, r#"一"#),
        (r#"㆓"#, r#"㆓"#, r#"㆓"#, r#"二"#, r#"二"#),
        (r#"㆔"#, r#"㆔"#, r#"㆔"#, r#"三"#, r#"三"#),
        (r#"㆕"#, r#"㆕"#, r#"㆕"#, r#"四"#, r#"四"#),
        (r#"㆖"#, r#"㆖"#, r#"㆖"#, r#"上"#, r#"上"#),
        (r#"㆗"#, r#"㆗"#, r#"㆗"#, r#"中"#, r#"中"#),
        (r#"㆘"#, r#"㆘"#, r#"㆘"#, r#"下"#, r#"下"#),
        (r#"㆙"#, r#"㆙"#, r#"㆙"#, r#"甲"#, r#"甲"#),
        (r#"㆚"#, r#"㆚"#, r#"㆚"#, r#"乙"#, r#"乙"#),
        (r#"㆛"#, r#"㆛"#, r#"㆛"#, r#"丙"#, r#"丙"#),
        (r#"㆜"#, r#"㆜"#, r#"㆜"#, r#"丁"#, r#"丁"#),
        (r#"㆝"#, r#"㆝"#, r#"㆝"#, r#"天"#, r#"天"#),
        (r#"㆞"#, r#"㆞"#, r#"㆞"#, r#"地"#, r#"地"#),
        (r#"㆟"#, r#"㆟"#, r#"㆟"#, r#"人"#, r#"人"#),
        (r#"㈀"#, r#"㈀"#, r#"㈀"#, r#"(ᄀ)"#, r#"(ᄀ)"#),
        (r#"㈁"#, r#"㈁"#, r#"㈁"#, r#"(ᄂ)"#, r#"(ᄂ)"#),
        (r#"㈂"#, r#"㈂"#, r#"㈂"#, r#"(ᄃ)"#, r#"(ᄃ)"#),
        (r#"㈃"#, r#"㈃"#, r#"㈃"#, r#"(ᄅ)"#, r#"(ᄅ)"#),
        (r#"㈄"#, r#"㈄"#, r#"㈄"#, r#"(ᄆ)"#, r#"(ᄆ)"#),
        (r#"㈅"#, r#"㈅"#, r#"㈅"#, r#"(ᄇ)"#, r#"(ᄇ)"#),
        (r#"㈆"#, r#"㈆"#, r#"㈆"#, r#"(ᄉ)"#, r#"(ᄉ)"#),
        (r#"㈇"#, r#"㈇"#, r#"㈇"#, r#"(ᄋ)"#, r#"(ᄋ)"#),
        (r#"㈈"#, r#"㈈"#, r#"㈈"#, r#"(ᄌ)"#, r#"(ᄌ)"#),
        (r#"㈉"#, r#"㈉"#, r#"㈉"#, r#"(ᄎ)"#, r#"(ᄎ)"#),
        (r#"㈊"#, r#"㈊"#, r#"㈊"#, r#"(ᄏ)"#, r#"(ᄏ)"#),
        (r#"㈋"#, r#"㈋"#, r#"㈋"#, r#"(ᄐ)"#, r#"(ᄐ)"#),
        (r#"㈌"#, r#"㈌"#, r#"㈌"#, r#"(ᄑ)"#, r#"(ᄑ)"#),
        (r#"㈍"#, r#"㈍"#, r#"㈍"#, r#"(ᄒ)"#, r#"(ᄒ)"#),
        (r#"㈎"#, r#"㈎"#, r#"㈎"#, r#"(가)"#, r#"(가)"#),
        (r#"㈏"#, r#"㈏"#, r#"㈏"#, r#"(나)"#, r#"(나)"#),
        (r#"㈐"#, r#"㈐"#, r#"㈐"#, r#"(다)"#, r#"(다)"#),
        (r#"㈑"#, r#"㈑"#, r#"㈑"#, r#"(라)"#, r#"(라)"#),
        (r#"㈒"#, r#"㈒"#, r#"㈒"#, r#"(마)"#, r#"(마)"#),
        (r#"㈓"#, r#"㈓"#, r#"㈓"#, r#"(바)"#, r#"(바)"#),
        (r#"㈔"#, r#"㈔"#, r#"㈔"#, r#"(사)"#, r#"(사)"#),
        (r#"㈕"#, r#"㈕"#, r#"㈕"#, r#"(아)"#, r#"(아)"#),
        (r#"㈖"#, r#"㈖"#, r#"㈖"#, r#"(자)"#, r#"(자)"#),
        (r#"㈗"#, r#"㈗"#, r#"㈗"#, r#"(차)"#, r#"(차)"#),
        (r#"㈘"#, r#"㈘"#, r#"㈘"#, r#"(카)"#, r#"(카)"#),
        (r#"㈙"#, r#"㈙"#, r#"㈙"#, r#"(타)"#, r#"(타)"#),
        (r#"㈚"#, r#"㈚"#, r#"㈚"#, r#"(파)"#, r#"(파)"#),
        (r#"㈛"#, r#"㈛"#, r#"㈛"#, r#"(하)"#, r#"(하)"#),
        (r#"㈜"#, r#"㈜"#, r#"㈜"#, r#"(주)"#, r#"(주)"#),
        (r#"㈝"#, r#"㈝"#, r#"㈝"#, r#"(오전)"#, r#"(오전)"#),
        (r#"㈞"#, r#"㈞"#, r#"㈞"#, r#"(오후)"#, r#"(오후)"#),
        (r#"㈠"#, r#"㈠"#, r#"㈠"#, r#"(一)"#, r#"(一)"#),
        (r#"㈡"#, r#"㈡"#, r#"㈡"#, r#"(二)"#, r#"(二)"#),
        (r#"㈢"#, r#"㈢"#, r#"㈢"#, r#"(三)"#, r#"(三)"#),
        (r#"㈣"#, r#"㈣"#, r#"㈣"#, r#"(四)"#, r#"(四)"#),
        (r#"㈤"#, r#"㈤"#, r#"㈤"#, r#"(五)"#, r#"(五)"#),
        (r#"㈥"#, r#"㈥"#, r#"㈥"#, r#"(六)"#, r#"(六)"#),
        (r#"㈦"#, r#"㈦"#, r#"㈦"#, r#"(七)"#, r#"(七)"#),
        (r#"㈧"#, r#"㈧"#, r#"㈧"#, r#"(八)"#, r#"(八)"#),
        (r#"㈨"#, r#"㈨"#, r#"㈨"#, r#"(九)"#, r#"(九)"#),
        (r#"㈩"#, r#"㈩"#, r#"㈩"#, r#"(十)"#, r#"(十)"#),
        (r#"㈪"#, r#"㈪"#, r#"㈪"#, r#"(月)"#, r#"(月)"#),
        (r#"㈫"#, r#"㈫"#, r#"㈫"#, r#"(火)"#, r#"(火)"#),
        (r#"㈬"#, r#"㈬"#, r#"㈬"#, r#"(水)"#, r#"(水)"#),
        (r#"㈭"#, r#"㈭"#, r#"㈭"#, r#"(木)"#, r#"(木)"#),
        (r#"㈮"#, r#"㈮"#, r#"㈮"#, r#"(金)"#, r#"(金)"#),
        (r#"㈯"#, r#"㈯"#, r#"㈯"#, r#"(土)"#, r#"(土)"#),
        (r#"㈰"#, r#"㈰"#, r#"㈰"#, r#"(日)"#, r#"(日)"#),
        (r#"㈱"#, r#"㈱"#, r#"㈱"#, r#"(株)"#, r#"(株)"#),
        (r#"㈲"#, r#"㈲"#, r#"㈲"#, r#"(有)"#, r#"(有)"#),
        (r#"㈳"#, r#"㈳"#, r#"㈳"#, r#"(社)"#, r#"(社)"#),
        (r#"㈴"#, r#"㈴"#, r#"㈴"#, r#"(名)"#, r#"(名)"#),
        (r#"㈵"#, r#"㈵"#, r#"㈵"#, r#"(特)"#, r#"(特)"#),
        (r#"㈶"#, r#"㈶"#, r#"㈶"#, r#"(財)"#, r#"(財)"#),
        (r#"㈷"#, r#"㈷"#, r#"㈷"#, r#"(祝)"#, r#"(祝)"#),
        (r#"㈸"#, r#"㈸"#, r#"㈸"#, r#"(労)"#, r#"(労)"#),
        (r#"㈹"#, r#"㈹"#, r#"㈹"#, r#"(代)"#, r#"(代)"#),
        (r#"㈺"#, r#"㈺"#, r#"㈺"#, r#"(呼)"#, r#"(呼)"#),
        (r#"㈻"#, r#"㈻"#, r#"㈻"#, r#"(学)"#, r#"(学)"#),
        (r#"㈼"#, r#"㈼"#, r#"㈼"#, r#"(監)"#, r#"(監)"#),
        (r#"㈽"#, r#"㈽"#, r#"㈽"#, r#"(企)"#, r#"(企)"#),
        (r#"㈾"#, r#"㈾"#, r#"㈾"#, r#"(資)"#, r#"(資)"#),
        (r#"㈿"#, r#"㈿"#, r#"㈿"#, r#"(協)"#, r#"(協)"#),
        (r#"㉀"#, r#"㉀"#, r#"㉀"#, r#"(祭)"#, r#"(祭)"#),
        (r#"㉁"#, r#"㉁"#, r#"㉁"#, r#"(休)"#, r#"(休)"#),
        (r#"㉂"#, r#"㉂"#, r#"㉂"#, r#"(自)"#, r#"(自)"#),
        (r#"㉃"#, r#"㉃"#, r#"㉃"#, r#"(至)"#, r#"(至)"#),
        (r#"㉄"#, r#"㉄"#, r#"㉄"#, r#"問"#, r#"問"#),
        (r#"㉅"#, r#"㉅"#, r#"㉅"#, r#"幼"#, r#"幼"#),
        (r#"㉆"#, r#"㉆"#, r#"㉆"#, r#"文"#, r#"文"#),
        (r#"㉇"#, r#"㉇"#, r#"㉇"#, r#"箏"#, r#"箏"#),
        (r#"㉐"#, r#"㉐"#, r#"㉐"#, r#"PTE"#, r#"PTE"#),
        (r#"㉑"#, r#"㉑"#, r#"㉑"#, r#"21"#, r#"21"#),
        (r#"㉒"#, r#"㉒"#, r#"㉒"#, r#"22"#, r#"22"#),
        (r#"㉓"#, r#"㉓"#, r#"㉓"#, r#"23"#, r#"23"#),
        (r#"㉔"#, r#"㉔"#, r#"㉔"#, r#"24"#, r#"24"#),
        (r#"㉕"#, r#"㉕"#, r#"㉕"#, r#"25"#, r#"25"#),
        (r#"㉖"#, r#"㉖"#, r#"㉖"#, r#"26"#, r#"26"#),
        (r#"㉗"#, r#"㉗"#, r#"㉗"#, r#"27"#, r#"27"#),
        (r#"㉘"#, r#"㉘"#, r#"㉘"#, r#"28"#, r#"28"#),
        (r#"㉙"#, r#"㉙"#, r#"㉙"#, r#"29"#, r#"29"#),
        (r#"㉚"#, r#"㉚"#, r#"㉚"#, r#"30"#, r#"30"#),
        (r#"㉛"#, r#"㉛"#, r#"㉛"#, r#"31"#, r#"31"#),
        (r#"㉜"#, r#"㉜"#, r#"㉜"#, r#"32"#, r#"32"#),
        (r#"㉝"#, r#"㉝"#, r#"㉝"#, r#"33"#, r#"33"#),
        (r#"㉞"#, r#"㉞"#, r#"㉞"#, r#"34"#, r#"34"#),
        (r#"㉟"#, r#"㉟"#, r#"㉟"#, r#"35"#, r#"35"#),
        (r#"㉠"#, r#"㉠"#, r#"㉠"#, r#"ᄀ"#, r#"ᄀ"#),
        (r#"㉡"#, r#"㉡"#, r#"㉡"#, r#"ᄂ"#, r#"ᄂ"#),
        (r#"㉢"#, r#"㉢"#, r#"㉢"#, r#"ᄃ"#, r#"ᄃ"#),
        (r#"㉣"#, r#"㉣"#, r#"㉣"#, r#"ᄅ"#, r#"ᄅ"#),
        (r#"㉤"#, r#"㉤"#, r#"㉤"#, r#"ᄆ"#, r#"ᄆ"#),
        (r#"㉥"#, r#"㉥"#, r#"㉥"#, r#"ᄇ"#, r#"ᄇ"#),
        (r#"㉦"#, r#"㉦"#, r#"㉦"#, r#"ᄉ"#, r#"ᄉ"#),
        (r#"㉧"#, r#"㉧"#, r#"㉧"#, r#"ᄋ"#, r#"ᄋ"#),
        (r#"㉨"#, r#"㉨"#, r#"㉨"#, r#"ᄌ"#, r#"ᄌ"#),
        (r#"㉩"#, r#"㉩"#, r#"㉩"#, r#"ᄎ"#, r#"ᄎ"#),
        (r#"㉪"#, r#"㉪"#, r#"㉪"#, r#"ᄏ"#, r#"ᄏ"#),
        (r#"㉫"#, r#"㉫"#, r#"㉫"#, r#"ᄐ"#, r#"ᄐ"#),
        (r#"㉬"#, r#"㉬"#, r#"㉬"#, r#"ᄑ"#, r#"ᄑ"#),
        (r#"㉭"#, r#"㉭"#, r#"㉭"#, r#"ᄒ"#, r#"ᄒ"#),
        (r#"㉮"#, r#"㉮"#, r#"㉮"#, r#"가"#, r#"가"#),
        (r#"㉯"#, r#"㉯"#, r#"㉯"#, r#"나"#, r#"나"#),
        (r#"㉰"#, r#"㉰"#, r#"㉰"#, r#"다"#, r#"다"#),
        (r#"㉱"#, r#"㉱"#, r#"㉱"#, r#"라"#, r#"라"#),
        (r#"㉲"#, r#"㉲"#, r#"㉲"#, r#"마"#, r#"마"#),
        (r#"㉳"#, r#"㉳"#, r#"㉳"#, r#"바"#, r#"바"#),
        (r#"㉴"#, r#"㉴"#, r#"㉴"#, r#"사"#, r#"사"#),
        (r#"㉵"#, r#"㉵"#, r#"㉵"#, r#"아"#, r#"아"#),
        (r#"㉶"#, r#"㉶"#, r#"㉶"#, r#"자"#, r#"자"#),
        (r#"㉷"#, r#"㉷"#, r#"㉷"#, r#"차"#, r#"차"#),
        (r#"㉸"#, r#"㉸"#, r#"㉸"#, r#"카"#, r#"카"#),
        (r#"㉹"#, r#"㉹"#, r#"㉹"#, r#"타"#, r#"타"#),
        (r#"㉺"#, r#"㉺"#, r#"㉺"#, r#"파"#, r#"파"#),
        (r#"㉻"#, r#"㉻"#, r#"㉻"#, r#"하"#, r#"하"#),
        (r#"㉼"#, r#"㉼"#, r#"㉼"#, r#"참고"#, r#"참고"#),
        (r#"㉽"#, r#"㉽"#, r#"㉽"#, r#"주의"#, r#"주의"#),
        (r#"㉾"#, r#"㉾"#, r#"㉾"#, r#"우"#, r#"우"#),
        (r#"㊀"#, r#"㊀"#, r#"㊀"#, r#"一"#, r#"一"#),
        (r#"㊁"#, r#"㊁"#, r#"㊁"#, r#"二"#, r#"二"#),
        (r#"㊂"#, r#"㊂"#, r#"㊂"#, r#"三"#, r#"三"#),
        (r#"㊃"#, r#"㊃"#, r#"㊃"#, r#"四"#, r#"四"#),
        (r#"㊄"#, r#"㊄"#, r#"㊄"#, r#"五"#, r#"五"#),
        (r#"㊅"#, r#"㊅"#, r#"㊅"#, r#"六"#, r#"六"#),
        (r#"㊆"#, r#"㊆"#, r#"㊆"#, r#"七"#, r#"七"#),
        (r#"㊇"#, r#"㊇"#, r#"㊇"#, r#"八"#, r#"八"#),
        (r#"㊈"#, r#"㊈"#, r#"㊈"#, r#"九"#, r#"九"#),
        (r#"㊉"#, r#"㊉"#, r#"㊉"#, r#"十"#, r#"十"#),
        (r#"㊊"#, r#"㊊"#, r#"㊊"#, r#"月"#, r#"月"#),
        (r#"㊋"#, r#"㊋"#, r#"㊋"#, r#"火"#, r#"火"#),
        (r#"㊌"#, r#"㊌"#, r#"㊌"#, r#"水"#, r#"水"#),
        (r#"㊍"#, r#"㊍"#, r#"㊍"#, r#"木"#, r#"木"#),
        (r#"㊎"#, r#"㊎"#, r#"㊎"#, r#"金"#, r#"金"#),
        (r#"㊏"#, r#"㊏"#, r#"㊏"#, r#"土"#, r#"土"#),
        (r#"㊐"#, r#"㊐"#, r#"㊐"#, r#"日"#, r#"日"#),
        (r#"㊑"#, r#"㊑"#, r#"㊑"#, r#"株"#, r#"株"#),
        (r#"㊒"#, r#"㊒"#, r#"㊒"#, r#"有"#, r#"有"#),
        (r#"㊓"#, r#"㊓"#, r#"㊓"#, r#"社"#, r#"社"#),
        (r#"㊔"#, r#"㊔"#, r#"㊔"#, r#"名"#, r#"名"#),
        (r#"㊕"#, r#"㊕"#, r#"㊕"#, r#"特"#, r#"特"#),
        (r#"㊖"#, r#"㊖"#, r#"㊖"#, r#"財"#, r#"財"#),
        (r#"㊗"#, r#"㊗"#, r#"㊗"#, r#"祝"#, r#"祝"#),
        (r#"㊘"#, r#"㊘"#, r#"㊘"#, r#"労"#, r#"労"#),
        (r#"㊙"#, r#"㊙"#, r#"㊙"#, r#"秘"#, r#"秘"#),
        (r#"㊚"#, r#"㊚"#, r#"㊚"#, r#"男"#, r#"男"#),
        (r#"㊛"#, r#"㊛"#, r#"㊛"#, r#"女"#, r#"女"#),
        (r#"㊜"#, r#"㊜"#, r#"㊜"#, r#"適"#, r#"適"#),
        (r#"㊝"#, r#"㊝"#, r#"㊝"#, r#"優"#, r#"優"#),
        (r#"㊞"#, r#"㊞"#, r#"㊞"#, r#"印"#, r#"印"#),
        (r#"㊟"#, r#"㊟"#, r#"㊟"#, r#"注"#, r#"注"#),
        (r#"㊠"#, r#"㊠"#, r#"㊠"#, r#"項"#, r#"項"#),
        (r#"㊡"#, r#"㊡"#, r#"㊡"#, r#"休"#, r#"休"#),
        (r#"㊢"#, r#"㊢"#, r#"㊢"#, r#"写"#, r#"写"#),
        (r#"㊣"#, r#"㊣"#, r#"㊣"#, r#"正"#, r#"正"#),
        (r#"㊤"#, r#"㊤"#, r#"㊤"#, r#"上"#, r#"上"#),
        (r#"㊥"#, r#"㊥"#, r#"㊥"#, r#"中"#, r#"中"#),
        (r#"㊦"#, r#"㊦"#, r#"㊦"#, r#"下"#, r#"下"#),
        (r#"㊧"#, r#"㊧"#, r#"㊧"#, r#"左"#, r#"左"#),
        (r#"㊨"#, r#"㊨"#, r#"㊨"#, r#"右"#, r#"右"#),
        (r#"㊩"#, r#"㊩"#, r#"㊩"#, r#"医"#, r#"医"#),
        (r#"㊪"#, r#"㊪"#, r#"㊪"#, r#"宗"#, r#"宗"#),
        (r#"㊫"#, r#"㊫"#, r#"㊫"#, r#"学"#, r#"学"#),
        (r#"㊬"#, r#"㊬"#, r#"㊬"#, r#"監"#, r#"監"#),
        (r#"㊭"#, r#"㊭"#, r#"㊭"#, r#"企"#, r#"企"#),
        (r#"㊮"#, r#"㊮"#, r#"㊮"#, r#"資"#, r#"資"#),
        (r#"㊯"#, r#"㊯"#, r#"㊯"#, r#"協"#, r#"協"#),
        (r#"㊰"#, r#"㊰"#, r#"㊰"#, r#"夜"#, r#"夜"#),
        (r#"㊱"#, r#"㊱"#, r#"㊱"#, r#"36"#, r#"36"#),
        (r#"㊲"#, r#"㊲"#, r#"㊲"#, r#"37"#, r#"37"#),
        (r#"㊳"#, r#"㊳"#, r#"㊳"#, r#"38"#, r#"38"#),
        (r#"㊴"#, r#"㊴"#, r#"㊴"#, r#"39"#, r#"39"#),
        (r#"㊵"#, r#"㊵"#, r#"㊵"#, r#"40"#, r#"40"#),
        (r#"㊶"#, r#"㊶"#, r#"㊶"#, r#"41"#, r#"41"#),
        (r#"㊷"#, r#"㊷"#, r#"㊷"#, r#"42"#, r#"42"#),
        (r#"㊸"#, r#"㊸"#, r#"㊸"#, r#"43"#, r#"43"#),
        (r#"㊹"#, r#"㊹"#, r#"㊹"#, r#"44"#, r#"44"#),
        (r#"㊺"#, r#"㊺"#, r#"㊺"#, r#"45"#, r#"45"#),
        (r#"㊻"#, r#"㊻"#, r#"㊻"#, r#"46"#, r#"46"#),
        (r#"㊼"#, r#"㊼"#, r#"㊼"#, r#"47"#, r#"47"#),
        (r#"㊽"#, r#"㊽"#, r#"㊽"#, r#"48"#, r#"48"#),
        (r#"㊾"#, r#"㊾"#, r#"㊾"#, r#"49"#, r#"49"#),
        (r#"㊿"#, r#"㊿"#, r#"㊿"#, r#"50"#, r#"50"#),
        (r#"㋀"#, r#"㋀"#, r#"㋀"#, r#"1月"#, r#"1月"#),
        (r#"㋁"#, r#"㋁"#, r#"㋁"#, r#"2月"#, r#"2月"#),
        (r#"㋂"#, r#"㋂"#, r#"㋂"#, r#"3月"#, r#"3月"#),
        (r#"㋃"#, r#"㋃"#, r#"㋃"#, r#"4月"#, r#"4月"#),
        (r#"㋄"#, r#"㋄"#, r#"㋄"#, r#"5月"#, r#"5月"#),
        (r#"㋅"#, r#"㋅"#, r#"㋅"#, r#"6月"#, r#"6月"#),
        (r#"㋆"#, r#"㋆"#, r#"㋆"#, r#"7月"#, r#"7月"#),
        (r#"㋇"#, r#"㋇"#, r#"㋇"#, r#"8月"#, r#"8月"#),
        (r#"㋈"#, r#"㋈"#, r#"㋈"#, r#"9月"#, r#"9月"#),
        (r#"㋉"#, r#"㋉"#, r#"㋉"#, r#"10月"#, r#"10月"#),
        (r#"㋊"#, r#"㋊"#, r#"㋊"#, r#"11月"#, r#"11月"#),
        (r#"㋋"#, r#"㋋"#, r#"㋋"#, r#"12月"#, r#"12月"#),
        (r#"㋌"#, r#"㋌"#, r#"㋌"#, r#"Hg"#, r#"Hg"#),
        (r#"㋍"#, r#"㋍"#, r#"㋍"#, r#"erg"#, r#"erg"#),
        (r#"㋎"#, r#"㋎"#, r#"㋎"#, r#"eV"#, r#"eV"#),
        (r#"㋏"#, r#"㋏"#, r#"㋏"#, r#"LTD"#, r#"LTD"#),
        (r#"㋐"#, r#"㋐"#, r#"㋐"#, r#"ア"#, r#"ア"#),
        (r#"㋑"#, r#"㋑"#, r#"㋑"#, r#"イ"#, r#"イ"#),
        (r#"㋒"#, r#"㋒"#, r#"㋒"#, r#"ウ"#, r#"ウ"#),
        (r#"㋓"#, r#"㋓"#, r#"㋓"#, r#"エ"#, r#"エ"#),
        (r#"㋔"#, r#"㋔"#, r#"㋔"#, r#"オ"#, r#"オ"#),
        (r#"㋕"#, r#"㋕"#, r#"㋕"#, r#"カ"#, r#"カ"#),
        (r#"㋖"#, r#"㋖"#, r#"㋖"#, r#"キ"#, r#"キ"#),
        (r#"㋗"#, r#"㋗"#, r#"㋗"#, r#"ク"#, r#"ク"#),
        (r#"㋘"#, r#"㋘"#, r#"㋘"#, r#"ケ"#, r#"ケ"#),
        (r#"㋙"#, r#"㋙"#, r#"㋙"#, r#"コ"#, r#"コ"#),
        (r#"㋚"#, r#"㋚"#, r#"㋚"#, r#"サ"#, r#"サ"#),
        (r#"㋛"#, r#"㋛"#, r#"㋛"#, r#"シ"#, r#"シ"#),
        (r#"㋜"#, r#"㋜"#, r#"㋜"#, r#"ス"#, r#"ス"#),
        (r#"㋝"#, r#"㋝"#, r#"㋝"#, r#"セ"#, r#"セ"#),
        (r#"㋞"#, r#"㋞"#, r#"㋞"#, r#"ソ"#, r#"ソ"#),
        (r#"㋟"#, r#"㋟"#, r#"㋟"#, r#"タ"#, r#"タ"#),
        (r#"㋠"#, r#"㋠"#, r#"㋠"#, r#"チ"#, r#"チ"#),
        (r#"㋡"#, r#"㋡"#, r#"㋡"#, r#"ツ"#, r#"ツ"#),
        (r#"㋢"#, r#"㋢"#, r#"㋢"#, r#"テ"#, r#"テ"#),
        (r#"㋣"#, r#"㋣"#, r#"㋣"#, r#"ト"#, r#"ト"#),
        (r#"㋤"#, r#"㋤"#, r#"㋤"#, r#"ナ"#, r#"ナ"#),
        (r#"㋥"#, r#"㋥"#, r#"㋥"#, r#"ニ"#, r#"ニ"#),
        (r#"㋦"#, r#"㋦"#, r#"㋦"#, r#"ヌ"#, r#"ヌ"#),
        (r#"㋧"#, r#"㋧"#, r#"㋧"#, r#"ネ"#, r#"ネ"#),
        (r#"㋨"#, r#"㋨"#, r#"㋨"#, r#"ノ"#, r#"ノ"#),
        (r#"㋩"#, r#"㋩"#, r#"㋩"#, r#"ハ"#, r#"ハ"#),
        (r#"㋪"#, r#"㋪"#, r#"㋪"#, r#"ヒ"#, r#"ヒ"#),
        (r#"㋫"#, r#"㋫"#, r#"㋫"#, r#"フ"#, r#"フ"#),
        (r#"㋬"#, r#"㋬"#, r#"㋬"#, r#"ヘ"#, r#"ヘ"#),
        (r#"㋭"#, r#"㋭"#, r#"㋭"#, r#"ホ"#, r#"ホ"#),
        (r#"㋮"#, r#"㋮"#, r#"㋮"#, r#"マ"#, r#"マ"#),
        (r#"㋯"#, r#"㋯"#, r#"㋯"#, r#"ミ"#, r#"ミ"#),
        (r#"㋰"#, r#"㋰"#, r#"㋰"#, r#"ム"#, r#"ム"#),
        (r#"㋱"#, r#"㋱"#, r#"㋱"#, r#"メ"#, r#"メ"#),
        (r#"㋲"#, r#"㋲"#, r#"㋲"#, r#"モ"#, r#"モ"#),
        (r#"㋳"#, r#"㋳"#, r#"㋳"#, r#"ヤ"#, r#"ヤ"#),
        (r#"㋴"#, r#"㋴"#, r#"㋴"#, r#"ユ"#, r#"ユ"#),
        (r#"㋵"#, r#"㋵"#, r#"㋵"#, r#"ヨ"#, r#"ヨ"#),
        (r#"㋶"#, r#"㋶"#, r#"㋶"#, r#"ラ"#, r#"ラ"#),
        (r#"㋷"#, r#"㋷"#, r#"㋷"#, r#"リ"#, r#"リ"#),
        (r#"㋸"#, r#"㋸"#, r#"㋸"#, r#"ル"#, r#"ル"#),
        (r#"㋹"#, r#"㋹"#, r#"㋹"#, r#"レ"#, r#"レ"#),
        (r#"㋺"#, r#"㋺"#, r#"㋺"#, r#"ロ"#, r#"ロ"#),
        (r#"㋻"#, r#"㋻"#, r#"㋻"#, r#"ワ"#, r#"ワ"#),
        (r#"㋼"#, r#"㋼"#, r#"㋼"#, r#"ヰ"#, r#"ヰ"#),
        (r#"㋽"#, r#"㋽"#, r#"㋽"#, r#"ヱ"#, r#"ヱ"#),
        (r#"㋾"#, r#"㋾"#, r#"㋾"#, r#"ヲ"#, r#"ヲ"#),
        (r#"㋿"#, r#"㋿"#, r#"㋿"#, r#"令和"#, r#"令和"#),
        (r#"㌀"#, r#"㌀"#, r#"㌀"#, r#"アパート"#, r#"アパート"#),
        (r#"㌁"#, r#"㌁"#, r#"㌁"#, r#"アルファ"#, r#"アルファ"#),
        (r#"㌂"#, r#"㌂"#, r#"㌂"#, r#"アンペア"#, r#"アンペア"#),
        (r#"㌃"#, r#"㌃"#, r#"㌃"#, r#"アール"#, r#"アール"#),
        (r#"㌄"#, r#"㌄"#, r#"㌄"#, r#"イニング"#, r#"イニング"#),
        (r#"㌅"#, r#"㌅"#, r#"㌅"#, r#"インチ"#, r#"インチ"#),
        (r#"㌆"#, r#"㌆"#, r#"㌆"#, r#"ウォン"#, r#"ウォン"#),
        (r#"㌇"#, r#"㌇"#, r#"㌇"#, r#"エスクード"#, r#"エスクード"#),
        (r#"㌈"#, r#"㌈"#, r#"㌈"#, r#"エーカー"#, r#"エーカー"#),
        (r#"㌉"#, r#"㌉"#, r#"㌉"#, r#"オンス"#, r#"オンス"#),
        (r#"㌊"#, r#"㌊"#, r#"㌊"#, r#"オーム"#, r#"オーム"#),
        (r#"㌋"#, r#"㌋"#, r#"㌋"#, r#"カイリ"#, r#"カイリ"#),
        (r#"㌌"#, r#"㌌"#, r#"㌌"#, r#"カラット"#, r#"カラット"#),
        (r#"㌍"#, r#"㌍"#, r#"㌍"#, r#"カロリー"#, r#"カロリー"#),
        (r#"㌎"#, r#"㌎"#, r#"㌎"#, r#"ガロン"#, r#"ガロン"#),
        (r#"㌏"#, r#"㌏"#, r#"㌏"#, r#"ガンマ"#, r#"ガンマ"#),
        (r#"㌐"#, r#"㌐"#, r#"㌐"#, r#"ギガ"#, r#"ギガ"#),
        (r#"㌑"#, r#"㌑"#, r#"㌑"#, r#"ギニー"#, r#"ギニー"#),
        (r#"㌒"#, r#"㌒"#, r#"㌒"#, r#"キュリー"#, r#"キュリー"#),
        (r#"㌓"#, r#"㌓"#, r#"㌓"#, r#"ギルダー"#, r#"ギルダー"#),
        (r#"㌔"#, r#"㌔"#, r#"㌔"#, r#"キロ"#, r#"キロ"#),
        (r#"㌕"#, r#"㌕"#, r#"㌕"#, r#"キログラム"#, r#"キログラム"#),
        (r#"㌖"#, r#"㌖"#, r#"㌖"#, r#"キロメートル"#, r#"キロメートル"#),
        (r#"㌗"#, r#"㌗"#, r#"㌗"#, r#"キロワット"#, r#"キロワット"#),
        (r#"㌘"#, r#"㌘"#, r#"㌘"#, r#"グラム"#, r#"グラム"#),
        (r#"㌙"#, r#"㌙"#, r#"㌙"#, r#"グラムトン"#, r#"グラムトン"#),
        (r#"㌚"#, r#"㌚"#, r#"㌚"#, r#"クルゼイロ"#, r#"クルゼイロ"#),
        (r#"㌛"#, r#"㌛"#, r#"㌛"#, r#"クローネ"#, r#"クローネ"#),
        (r#"㌜"#, r#"㌜"#, r#"㌜"#, r#"ケース"#, r#"ケース"#),
        (r#"㌝"#, r#"㌝"#, r#"㌝"#, r#"コルナ"#, r#"コルナ"#),
        (r#"㌞"#, r#"㌞"#, r#"㌞"#, r#"コーポ"#, r#"コーポ"#),
        (r#"㌟"#, r#"㌟"#, r#"㌟"#, r#"サイクル"#, r#"サイクル"#),
        (r#"㌠"#, r#"㌠"#, r#"㌠"#, r#"サンチーム"#, r#"サンチーム"#),
        (r#"㌡"#, r#"㌡"#, r#"㌡"#, r#"シリング"#, r#"シリング"#),
        (r#"㌢"#, r#"㌢"#, r#"㌢"#, r#"センチ"#, r#"センチ"#),
        (r#"㌣"#, r#"㌣"#, r#"㌣"#, r#"セント"#, r#"セント"#),
        (r#"㌤"#, r#"㌤"#, r#"㌤"#, r#"ダース"#, r#"ダース"#),
        (r#"㌥"#, r#"㌥"#, r#"㌥"#, r#"デシ"#, r#"デシ"#),
        (r#"㌦"#, r#"㌦"#, r#"㌦"#, r#"ドル"#, r#"ドル"#),
        (r#"㌧"#, r#"㌧"#, r#"㌧"#, r#"トン"#, r#"トン"#),
        (r#"㌨"#, r#"㌨"#, r#"㌨"#, r#"ナノ"#, r#"ナノ"#),
        (r#"㌩"#, r#"㌩"#, r#"㌩"#, r#"ノット"#, r#"ノット"#),
        (r#"㌪"#, r#"㌪"#, r#"㌪"#, r#"ハイツ"#, r#"ハイツ"#),
        (r#"㌫"#, r#"㌫"#, r#"㌫"#, r#"パーセント"#, r#"パーセント"#),
        (r#"㌬"#, r#"㌬"#, r#"㌬"#, r#"パーツ"#, r#"パーツ"#),
        (r#"㌭"#, r#"㌭"#, r#"㌭"#, r#"バーレル"#, r#"バーレル"#),
        (r#"㌮"#, r#"㌮"#, r#"㌮"#, r#"ピアストル"#, r#"ピアストル"#),
        (r#"㌯"#, r#"㌯"#, r#"㌯"#, r#"ピクル"#, r#"ピクル"#),
        (r#"㌰"#, r#"㌰"#, r#"㌰"#, r#"ピコ"#, r#"ピコ"#),
        (r#"㌱"#, r#"㌱"#, r#"㌱"#, r#"ビル"#, r#"ビル"#),
        (r#"㌲"#, r#"㌲"#, r#"㌲"#, r#"ファラッド"#, r#"ファラッド"#),
        (r#"㌳"#, r#"㌳"#, r#"㌳"#, r#"フィート"#, r#"フィート"#),
        (r#"㌴"#, r#"㌴"#, r#"㌴"#, r#"ブッシェル"#, r#"ブッシェル"#),
        (r#"㌵"#, r#"㌵"#, r#"㌵"#, r#"フラン"#, r#"フラン"#),
        (r#"㌶"#, r#"㌶"#, r#"㌶"#, r#"ヘクタール"#, r#"ヘクタール"#),
        (r#"㌷"#, r#"㌷"#, r#"㌷"#, r#"ペソ"#, r#"ペソ"#),
        (r#"㌸"#, r#"㌸"#, r#"㌸"#, r#"ペニヒ"#, r#"ペニヒ"#),
        (r#"㌹"#, r#"㌹"#, r#"㌹"#, r#"ヘルツ"#, r#"ヘルツ"#),
        (r#"㌺"#, r#"㌺"#, r#"㌺"#, r#"ペンス"#, r#"ペンス"#),
        (r#"㌻"#, r#"㌻"#, r#"㌻"#, r#"ページ"#, r#"ページ"#),
        (r#"㌼"#, r#"㌼"#, r#"㌼"#, r#"ベータ"#, r#"ベータ"#),
        (r#"㌽"#, r#"㌽"#, r#"㌽"#, r#"ポイント"#, r#"ポイント"#),
        (r#"㌾"#, r#"㌾"#, r#"㌾"#, r#"ボルト"#, r#"ボルト"#),
        (r#"㌿"#, r#"㌿"#, r#"㌿"#, r#"ホン"#, r#"ホン"#),
        (r#"㍀"#, r#"㍀"#, r#"㍀"#, r#"ポンド"#, r#"ポンド"#),
        (r#"㍁"#, r#"㍁"#, r#"㍁"#, r#"ホール"#, r#"ホール"#),
        (r#"㍂"#, r#"㍂"#, r#"㍂"#, r#"ホーン"#, r#"ホーン"#),
        (r#"㍃"#, r#"㍃"#, r#"㍃"#, r#"マイクロ"#, r#"マイクロ"#),
        (r#"㍄"#, r#"㍄"#, r#"㍄"#, r#"マイル"#, r#"マイル"#),
        (r#"㍅"#, r#"㍅"#, r#"㍅"#, r#"マッハ"#, r#"マッハ"#),
        (r#"㍆"#, r#"㍆"#, r#"㍆"#, r#"マルク"#, r#"マルク"#),
        (r#"㍇"#, r#"㍇"#, r#"㍇"#, r#"マンション"#, r#"マンション"#),
        (r#"㍈"#, r#"㍈"#, r#"㍈"#, r#"ミクロン"#, r#"ミクロン"#),
        (r#"㍉"#, r#"㍉"#, r#"㍉"#, r#"ミリ"#, r#"ミリ"#),
        (r#"㍊"#, r#"㍊"#, r#"㍊"#, r#"ミリバール"#, r#"ミリバール"#),
        (r#"㍋"#, r#"㍋"#, r#"㍋"#, r#"メガ"#, r#"メガ"#),
        (r#"㍌"#, r#"㍌"#, r#"㍌"#, r#"メガトン"#, r#"メガトン"#),
        (r#"㍍"#, r#"㍍"#, r#"㍍"#, r#"メートル"#, r#"メートル"#),
        (r#"㍎"#, r#"㍎"#, r#"㍎"#, r#"ヤード"#, r#"ヤード"#),
        (r#"㍏"#, r#"㍏"#, r#"㍏"#, r#"ヤール"#, r#"ヤール"#),
        (r#"㍐"#, r#"㍐"#, r#"㍐"#, r#"ユアン"#, r#"ユアン"#),
        (r#"㍑"#, r#"㍑"#, r#"㍑"#, r#"リットル"#, r#"リットル"#),
        (r#"㍒"#, r#"㍒"#, r#"㍒"#, r#"リラ"#, r#"リラ"#),
        (r#"㍓"#, r#"㍓"#, r#"㍓"#, r#"ルピー"#, r#"ルピー"#),
        (r#"㍔"#, r#"㍔"#, r#"㍔"#, r#"ルーブル"#, r#"ルーブル"#),
        (r#"㍕"#, r#"㍕"#, r#"㍕"#, r#"レム"#, r#"レム"#),
        (r#"㍖"#, r#"㍖"#, r#"㍖"#, r#"レントゲン"#, r#"レントゲン"#),
        (r#"㍗"#, r#"㍗"#, r#"㍗"#, r#"ワット"#, r#"ワット"#),
        (r#"㍘"#, r#"㍘"#, r#"㍘"#, r#"0点"#, r#"0点"#),
        (r#"㍙"#, r#"㍙"#, r#"㍙"#, r#"1点"#, r#"1点"#),
        (r#"㍚"#, r#"㍚"#, r#"㍚"#, r#"2点"#, r#"2点"#),
        (r#"㍛"#, r#"㍛"#, r#"㍛"#, r#"3点"#, r#"3点"#),
        (r#"㍜"#, r#"㍜"#, r#"㍜"#, r#"4点"#, r#"4点"#),
        (r#"㍝"#, r#"㍝"#, r#"㍝"#, r#"5点"#, r#"5点"#),
        (r#"㍞"#, r#"㍞"#, r#"㍞"#, r#"6点"#, r#"6点"#),
        (r#"㍟"#, r#"㍟"#, r#"㍟"#, r#"7点"#, r#"7点"#),
        (r#"㍠"#, r#"㍠"#, r#"㍠"#, r#"8点"#, r#"8点"#),
        (r#"㍡"#, r#"㍡"#, r#"㍡"#, r#"9点"#, r#"9点"#),
        (r#"㍢"#, r#"㍢"#, r#"㍢"#, r#"10点"#, r#"10点"#),
        (r#"㍣"#, r#"㍣"#, r#"㍣"#, r#"11点"#, r#"11点"#),
        (r#"㍤"#, r#"㍤"#, r#"㍤"#, r#"12点"#, r#"12点"#),
        (r#"㍥"#, r#"㍥"#, r#"㍥"#, r#"13点"#, r#"13点"#),
        (r#"㍦"#, r#"㍦"#, r#"㍦"#, r#"14点"#, r#"14点"#),
        (r#"㍧"#, r#"㍧"#, r#"㍧"#, r#"15点"#, r#"15点"#),
        (r#"㍨"#, r#"㍨"#, r#"㍨"#, r#"16点"#, r#"16点"#),
        (r#"㍩"#, r#"㍩"#, r#"㍩"#, r#"17点"#, r#"17点"#),
        (r#"㍪"#, r#"㍪"#, r#"㍪"#, r#"18点"#, r#"18点"#),
        (r#"㍫"#, r#"㍫"#, r#"㍫"#, r#"19点"#, r#"19点"#),
        (r#"㍬"#, r#"㍬"#, r#"㍬"#, r#"20点"#, r#"20点"#),
        (r#"㍭"#, r#"㍭"#, r#"㍭"#, r#"21点"#, r#"21点"#),
        (r#"㍮"#, r#"㍮"#, r#"㍮"#, r#"22点"#, r#"22点"#),
        (r#"㍯"#, r#"㍯"#, r#"㍯"#, r#"23点"#, r#"23点"#),
        (r#"㍰"#, r#"㍰"#, r#"㍰"#, r#"24点"#, r#"24点"#),
        (r#"㍱"#, r#"㍱"#, r#"㍱"#, r#"hPa"#, r#"hPa"#),
        (r#"㍲"#, r#"㍲"#, r#"㍲"#, r#"da"#, r#"da"#),
        (r#"㍳"#, r#"㍳"#, r#"㍳"#, r#"AU"#, r#"AU"#),
        (r#"㍴"#, r#"㍴"#, r#"㍴"#, r#"bar"#, r#"bar"#),
        (r#"㍵"#, r#"㍵"#, r#"㍵"#, r#"oV"#, r#"oV"#),
        (r#"㍶"#, r#"㍶"#, r#"㍶"#, r#"pc"#, r#"pc"#),
        (r#"㍷"#, r#"㍷"#, r#"㍷"#, r#"dm"#, r#"dm"#),
        (r#"㍸"#, r#"㍸"#, r#"㍸"#, r#"dm2"#, r#"dm2"#),
        (r#"㍹"#, r#"㍹"#, r#"㍹"#, r#"dm3"#, r#"dm3"#),
        (r#"㍺"#, r#"㍺"#, r#"㍺"#, r#"IU"#, r#"IU"#),
        (r#"㍻"#, r#"㍻"#, r#"㍻"#, r#"平成"#, r#"平成"#),
        (r#"㍼"#, r#"㍼"#, r#"㍼"#, r#"昭和"#, r#"昭和"#),
        (r#"㍽"#, r#"㍽"#, r#"㍽"#, r#"大正"#, r#"大正"#),
        (r#"㍾"#, r#"㍾"#, r#"㍾"#, r#"明治"#, r#"明治"#),
        (r#"㍿"#, r#"㍿"#, r#"㍿"#, r#"株式会社"#, r#"株式会社"#),
        (r#"㎀"#, r#"㎀"#, r#"㎀"#, r#"pA"#, r#"pA"#),
        (r#"㎁"#, r#"㎁"#, r#"㎁"#, r#"nA"#, r#"nA"#),
        (r#"㎂"#, r#"㎂"#, r#"㎂"#, r#"μA"#, r#"μA"#),
        (r#"㎃"#, r#"㎃"#, r#"㎃"#, r#"mA"#, r#"mA"#),
        (r#"㎄"#, r#"㎄"#, r#"㎄"#, r#"kA"#, r#"kA"#),
        (r#"㎅"#, r#"㎅"#, r#"㎅"#, r#"KB"#, r#"KB"#),
        (r#"㎆"#, r#"㎆"#, r#"㎆"#, r#"MB"#, r#"MB"#),
        (r#"㎇"#, r#"㎇"#, r#"㎇"#, r#"GB"#, r#"GB"#),
        (r#"㎈"#, r#"㎈"#, r#"㎈"#, r#"cal"#, r#"cal"#),
        (r#"㎉"#, r#"㎉"#, r#"㎉"#, r#"kcal"#, r#"kcal"#),
        (r#"㎊"#, r#"㎊"#, r#"㎊"#, r#"pF"#, r#"pF"#),
        (r#"㎋"#, r#"㎋"#, r#"㎋"#, r#"nF"#, r#"nF"#),
        (r#"㎌"#, r#"㎌"#, r#"㎌"#, r#"μF"#, r#"μF"#),
        (r#"㎍"#, r#"㎍"#, r#"㎍"#, r#"μg"#, r#"μg"#),
        (r#"㎎"#, r#"㎎"#, r#"㎎"#, r#"mg"#, r#"mg"#),
        (r#"㎏"#, r#"㎏"#, r#"㎏"#, r#"kg"#, r#"kg"#),
        (r#"㎐"#, r#"㎐"#, r#"㎐"#, r#"Hz"#, r#"Hz"#),
        (r#"㎑"#, r#"㎑"#, r#"㎑"#, r#"kHz"#, r#"kHz"#),
        (r#"㎒"#, r#"㎒"#, r#"㎒"#, r#"MHz"#, r#"MHz"#),
        (r#"㎓"#, r#"㎓"#, r#"㎓"#, r#"GHz"#, r#"GHz"#),
        (r#"㎔"#, r#"㎔"#, r#"㎔"#, r#"THz"#, r#"THz"#),
        (r#"㎕"#, r#"㎕"#, r#"㎕"#, r#"μl"#, r#"μl"#),
        (r#"㎖"#, r#"㎖"#, r#"㎖"#, r#"ml"#, r#"ml"#),
        (r#"㎗"#, r#"㎗"#, r#"㎗"#, r#"dl"#, r#"dl"#),
        (r#"㎘"#, r#"㎘"#, r#"㎘"#, r#"kl"#, r#"kl"#),
        (r#"㎙"#, r#"㎙"#, r#"㎙"#, r#"fm"#, r#"fm"#),
        (r#"㎚"#, r#"㎚"#, r#"㎚"#, r#"nm"#, r#"nm"#),
        (r#"㎛"#, r#"㎛"#, r#"㎛"#, r#"μm"#, r#"μm"#),
        (r#"㎜"#, r#"㎜"#, r#"㎜"#, r#"mm"#, r#"mm"#),
        (r#"㎝"#, r#"㎝"#, r#"㎝"#, r#"cm"#, r#"cm"#),
        (r#"㎞"#, r#"㎞"#, r#"㎞"#, r#"km"#, r#"km"#),
        (r#"㎟"#, r#"㎟"#, r#"㎟"#, r#"mm2"#, r#"mm2"#),
        (r#"㎠"#, r#"㎠"#, r#"㎠"#, r#"cm2"#, r#"cm2"#),
        (r#"㎡"#, r#"㎡"#, r#"㎡"#, r#"m2"#, r#"m2"#),
        (r#"㎢"#, r#"㎢"#, r#"㎢"#, r#"km2"#, r#"km2"#),
        (r#"㎣"#, r#"㎣"#, r#"㎣"#, r#"mm3"#, r#"mm3"#),
        (r#"㎤"#, r#"㎤"#, r#"㎤"#, r#"cm3"#, r#"cm3"#),
        (r#"㎥"#, r#"㎥"#, r#"㎥"#, r#"m3"#, r#"m3"#),
        (r#"㎦"#, r#"㎦"#, r#"㎦"#, r#"km3"#, r#"km3"#),
        (r#"㎧"#, r#"㎧"#, r#"㎧"#, r#"m∕s"#, r#"m∕s"#),
        (r#"㎨"#, r#"㎨"#, r#"㎨"#, r#"m∕s2"#, r#"m∕s2"#),
        (r#"㎩"#, r#"㎩"#, r#"㎩"#, r#"Pa"#, r#"Pa"#),
        (r#"㎪"#, r#"㎪"#, r#"㎪"#, r#"kPa"#, r#"kPa"#),
        (r#"㎫"#, r#"㎫"#, r#"㎫"#, r#"MPa"#, r#"MPa"#),
        (r#"㎬"#, r#"㎬"#, r#"㎬"#, r#"GPa"#, r#"GPa"#),
        (r#"㎭"#, r#"㎭"#, r#"㎭"#, r#"rad"#, r#"rad"#),
        (r#"㎮"#, r#"㎮"#, r#"㎮"#, r#"rad∕s"#, r#"rad∕s"#),
        (r#"㎯"#, r#"㎯"#, r#"㎯"#, r#"rad∕s2"#, r#"rad∕s2"#),
        (r#"㎰"#, r#"㎰"#, r#"㎰"#, r#"ps"#, r#"ps"#),
        (r#"㎱"#, r#"㎱"#, r#"㎱"#, r#"ns"#, r#"ns"#),
        (r#"㎲"#, r#"㎲"#, r#"㎲"#, r#"μs"#, r#"μs"#),
        (r#"㎳"#, r#"㎳"#, r#"㎳"#, r#"ms"#, r#"ms"#),
        (r#"㎴"#, r#"㎴"#, r#"㎴"#, r#"pV"#, r#"pV"#),
        (r#"㎵"#, r#"㎵"#, r#"㎵"#, r#"nV"#, r#"nV"#),
        (r#"㎶"#, r#"㎶"#, r#"㎶"#, r#"μV"#, r#"μV"#),
        (r#"㎷"#, r#"㎷"#, r#"㎷"#, r#"mV"#, r#"mV"#),
        (r#"㎸"#, r#"㎸"#, r#"㎸"#, r#"kV"#, r#"kV"#),
        (r#"㎹"#, r#"㎹"#, r#"㎹"#, r#"MV"#, r#"MV"#),
        (r#"㎺"#, r#"㎺"#, r#"㎺"#, r#"pW"#, r#"pW"#),
        (r#"㎻"#, r#"㎻"#, r#"㎻"#, r#"nW"#, r#"nW"#),
        (r#"㎼"#, r#"㎼"#, r#"㎼"#, r#"μW"#, r#"μW"#),
        (r#"㎽"#, r#"㎽"#, r#"㎽"#, r#"mW"#, r#"mW"#),
        (r#"㎾"#, r#"㎾"#, r#"㎾"#, r#"kW"#, r#"kW"#),
        (r#"㎿"#, r#"㎿"#, r#"㎿"#, r#"MW"#, r#"MW"#),
        (r#"㏀"#, r#"㏀"#, r#"㏀"#, r#"kΩ"#, r#"kΩ"#),
        (r#"㏁"#, r#"㏁"#, r#"㏁"#, r#"MΩ"#, r#"MΩ"#),
        (r#"㏂"#, r#"㏂"#, r#"㏂"#, r#"a.m."#, r#"a.m."#),
        (r#"㏃"#, r#"㏃"#, r#"㏃"#, r#"Bq"#, r#"Bq"#),
        (r#"㏄"#, r#"㏄"#, r#"㏄"#, r#"cc"#, r#"cc"#),
        (r#"㏅"#, r#"㏅"#, r#"㏅"#, r#"cd"#, r#"cd"#),
        (r#"㏆"#, r#"㏆"#, r#"㏆"#, r#"C∕kg"#, r#"C∕kg"#),
        (r#"㏇"#, r#"㏇"#, r#"㏇"#, r#"Co."#, r#"Co."#),
        (r#"㏈"#, r#"㏈"#, r#"㏈"#, r#"dB"#, r#"dB"#),
        (r#"㏉"#, r#"㏉"#, r#"㏉"#, r#"Gy"#, r#"Gy"#),
        (r#"㏊"#, r#"㏊"#, r#"㏊"#, r#"ha"#, r#"ha"#),
        (r#"㏋"#, r#"㏋"#, r#"㏋"#, r#"HP"#, r#"HP"#),
        (r#"㏌"#, r#"㏌"#, r#"㏌"#, r#"in"#, r#"in"#),
        (r#"㏍"#, r#"㏍"#, r#"㏍"#, r#"KK"#, r#"KK"#),
        (r#"㏎"#, r#"㏎"#, r#"㏎"#, r#"KM"#, r#"KM"#),
        (r#"㏏"#, r#"㏏"#, r#"㏏"#, r#"kt"#, r#"kt"#),
        (r#"㏐"#, r#"㏐"#, r#"㏐"#, r#"lm"#, r#"lm"#),
        (r#"㏑"#, r#"㏑"#, r#"㏑"#, r#"ln"#, r#"ln"#),
        (r#"㏒"#, r#"㏒"#, r#"㏒"#, r#"log"#, r#"log"#),
        (r#"㏓"#, r#"㏓"#, r#"㏓"#, r#"lx"#, r#"lx"#),
        (r#"㏔"#, r#"㏔"#, r#"㏔"#, r#"mb"#, r#"mb"#),
        (r#"㏕"#, r#"㏕"#, r#"㏕"#, r#"mil"#, r#"mil"#),
        (r#"㏖"#, r#"㏖"#, r#"㏖"#, r#"mol"#, r#"mol"#),
        (r#"㏗"#, r#"㏗"#, r#"㏗"#, r#"PH"#, r#"PH"#),
        (r#"㏘"#, r#"㏘"#, r#"㏘"#, r#"p.m."#, r#"p.m."#),
        (r#"㏙"#, r#"㏙"#, r#"㏙"#, r#"PPM"#, r#"PPM"#),
        (r#"㏚"#, r#"㏚"#, r#"㏚"#, r#"PR"#, r#"PR"#),
        (r#"㏛"#, r#"㏛"#, r#"㏛"#, r#"sr"#, r#"sr"#),
        (r#"㏜"#, r#"㏜"#, r#"㏜"#, r#"Sv"#, r#"Sv"#),
        (r#"㏝"#, r#"㏝"#, r#"㏝"#, r#"Wb"#, r#"Wb"#),
        (r#"㏞"#, r#"㏞"#, r#"㏞"#, r#"V∕m"#, r#"V∕m"#),
        (r#"㏟"#, r#"㏟"#, r#"㏟"#, r#"A∕m"#, r#"A∕m"#),
        (r#"㏠"#, r#"㏠"#, r#"㏠"#, r#"1日"#, r#"1日"#),
        (r#"㏡"#, r#"㏡"#, r#"㏡"#, r#"2日"#, r#"2日"#),
        (r#"㏢"#, r#"㏢"#, r#"㏢"#, r#"3日"#, r#"3日"#),
        (r#"㏣"#, r#"㏣"#, r#"㏣"#, r#"4日"#, r#"4日"#),
        (r#"㏤"#, r#"㏤"#, r#"㏤"#, r#"5日"#, r#"5日"#),
        (r#"㏥"#, r#"㏥"#, r#"㏥"#, r#"6日"#, r#"6日"#),
        (r#"㏦"#, r#"㏦"#, r#"㏦"#, r#"7日"#, r#"7日"#),
        (r#"㏧"#, r#"㏧"#, r#"㏧"#, r#"8日"#, r#"8日"#),
        (r#"㏨"#, r#"㏨"#, r#"㏨"#, r#"9日"#, r#"9日"#),
        (r#"㏩"#, r#"㏩"#, r#"㏩"#, r#"10日"#, r#"10日"#),
        (r#"㏪"#, r#"㏪"#, r#"㏪"#, r#"11日"#, r#"11日"#),
        (r#"㏫"#, r#"㏫"#, r#"㏫"#, r#"12日"#, r#"12日"#),
        (r#"㏬"#, r#"㏬"#, r#"㏬"#, r#"13日"#, r#"13日"#),
        (r#"㏭"#, r#"㏭"#, r#"㏭"#, r#"14日"#, r#"14日"#),
        (r#"㏮"#, r#"㏮"#, r#"㏮"#, r#"15日"#, r#"15日"#),
        (r#"㏯"#, r#"㏯"#, r#"㏯"#, r#"16日"#, r#"16日"#),
        (r#"㏰"#, r#"㏰"#, r#"㏰"#, r#"17日"#, r#"17日"#),
        (r#"㏱"#, r#"㏱"#, r#"㏱"#, r#"18日"#, r#"18日"#),
        (r#"㏲"#, r#"㏲"#, r#"㏲"#, r#"19日"#, r#"19日"#),
        (r#"㏳"#, r#"㏳"#, r#"㏳"#, r#"20日"#, r#"20日"#),
        (r#"㏴"#, r#"㏴"#, r#"㏴"#, r#"21日"#, r#"21日"#),
        (r#"㏵"#, r#"㏵"#, r#"㏵"#, r#"22日"#, r#"22日"#),
        (r#"㏶"#, r#"㏶"#, r#"㏶"#, r#"23日"#, r#"23日"#),
        (r#"㏷"#, r#"㏷"#, r#"㏷"#, r#"24日"#, r#"24日"#),
        (r#"㏸"#, r#"㏸"#, r#"㏸"#, r#"25日"#, r#"25日"#),
        (r#"㏹"#, r#"㏹"#, r#"㏹"#, r#"26日"#, r#"26日"#),
        (r#"㏺"#, r#"㏺"#, r#"㏺"#, r#"27日"#, r#"27日"#),
        (r#"㏻"#, r#"㏻"#, r#"㏻"#, r#"28日"#, r#"28日"#),
        (r#"㏼"#, r#"㏼"#, r#"㏼"#, r#"29日"#, r#"29日"#),
        (r#"㏽"#, r#"㏽"#, r#"㏽"#, r#"30日"#, r#"30日"#),
        (r#"㏾"#, r#"㏾"#, r#"㏾"#, r#"31日"#, r#"31日"#),
        (r#"㏿"#, r#"㏿"#, r#"㏿"#, r#"gal"#, r#"gal"#),
        (r#"ꚜ"#, r#"ꚜ"#, r#"ꚜ"#, r#"ъ"#, r#"ъ"#),
        (r#"ꚝ"#, r#"ꚝ"#, r#"ꚝ"#, r#"ь"#, r#"ь"#),
        (r#"ꝰ"#, r#"ꝰ"#, r#"ꝰ"#, r#"ꝯ"#, r#"ꝯ"#),
        (r#"ꟲ"#, r#"ꟲ"#, r#"ꟲ"#, r#"C"#, r#"C"#),
        (r#"ꟳ"#, r#"ꟳ"#, r#"ꟳ"#, r#"F"#, r#"F"#),
        (r#"ꟴ"#, r#"ꟴ"#, r#"ꟴ"#, r#"Q"#, r#"Q"#),
        (r#"ꟸ"#, r#"ꟸ"#, r#"ꟸ"#, r#"Ħ"#, r#"Ħ"#),
        (r#"ꟹ"#, r#"ꟹ"#, r#"ꟹ"#, r#"œ"#, r#"œ"#),
        (r#"ꭜ"#, r#"ꭜ"#, r#"ꭜ"#, r#"ꜧ"#, r#"ꜧ"#),
        (r#"ꭝ"#, r#"ꭝ"#, r#"ꭝ"#, r#"ꬷ"#, r#"ꬷ"#),
        (r#"ꭞ"#, r#"ꭞ"#, r#"ꭞ"#, r#"ɫ"#, r#"ɫ"#),
        (r#"ꭟ"#, r#"ꭟ"#, r#"ꭟ"#, r#"ꭒ"#, r#"ꭒ"#),
        (r#"ꭩ"#, r#"ꭩ"#, r#"ꭩ"#, r#"ʍ"#, r#"ʍ"#),
        (r#"가"#, r#"가"#, r#"가"#, r#"가"#, r#"가"#),
        (r#"각"#, r#"각"#, r#"각"#, r#"각"#, r#"각"#),
        (r#"갂"#, r#"갂"#, r#"갂"#, r#"갂"#, r#"갂"#),
        (r#"갃"#, r#"갃"#, r#"갃"#, r#"갃"#, r#"갃"#),
        (r#"간"#, r#"간"#, r#"간"#, r#"간"#, r#"간"#),
        (r#"갅"#, r#"갅"#, r#"갅"#, r#"갅"#, r#"갅"#),
        (r#"갆"#, r#"갆"#, r#"갆"#, r#"갆"#, r#"갆"#),
        (r#"갇"#, r#"갇"#, r#"갇"#, r#"갇"#, r#"갇"#),
        (r#"갈"#, r#"갈"#, r#"갈"#, r#"갈"#, r#"갈"#),
        (r#"갉"#, r#"갉"#, r#"갉"#, r#"갉"#, r#"갉"#),
        (r#"갊"#, r#"갊"#, r#"갊"#, r#"갊"#, r#"갊"#),
        (r#"갋"#, r#"갋"#, r#"갋"#, r#"갋"#, r#"갋"#),
        (r#"갌"#, r#"갌"#, r#"갌"#, r#"갌"#, r#"갌"#),
        (r#"갍"#, r#"갍"#, r#"갍"#, r#"갍"#, r#"갍"#),
        (r#"갎"#, r#"갎"#, r#"갎"#, r#"갎"#, r#"갎"#),
        (r#"갏"#, r#"갏"#, r#"갏"#, r#"갏"#, r#"갏"#),
        (r#"감"#, r#"감"#, r#"감"#, r#"감"#, r#"감"#),
        (r#"갑"#, r#"갑"#, r#"갑"#, r#"갑"#, r#"갑"#),
        (r#"값"#, r#"값"#, r#"값"#, r#"값"#, r#"값"#),
        (r#"갓"#, r#"갓"#, r#"갓"#, r#"갓"#, r#"갓"#),
        (r#"갔"#, r#"갔"#, r#"갔"#, r#"갔"#, r#"갔"#),
        (r#"강"#, r#"강"#, r#"강"#, r#"강"#, r#"강"#),
        (r#"갖"#, r#"갖"#, r#"갖"#, r#"갖"#, r#"갖"#),
        (r#"갗"#, r#"갗"#, r#"갗"#, r#"갗"#, r#"갗"#),
        (r#"갘"#, r#"갘"#, r#"갘"#, r#"갘"#, r#"갘"#),
        (r#"같"#, r#"같"#, r#"같"#, r#"같"#, r#"같"#),
        (r#"갚"#, r#"갚"#, r#"갚"#, r#"갚"#, r#"갚"#),
        (r#"갛"#, r#"갛"#, r#"갛"#, r#"갛"#, r#"갛"#),
        (r#"개"#, r#"개"#, r#"개"#, r#"개"#, r#"개"#),
        (r#"객"#, r#"객"#, r#"객"#, r#"객"#, r#"객"#),
        (r#"갞"#, r#"갞"#, r#"갞"#, r#"갞"#, r#"갞"#),
        (r#"갟"#, r#"갟"#, r#"갟"#, r#"갟"#, r#"갟"#),
        (r#"갠"#, r#"갠"#, r#"갠"#, r#"갠"#, r#"갠"#),
        (r#"갡"#, r#"갡"#, r#"갡"#, r#"갡"#, r#"갡"#),
        (r#"갢"#, r#"갢"#, r#"갢"#, r#"갢"#, r#"갢"#),
        (r#"갣"#, r#"갣"#, r#"갣"#, r#"갣"#, r#"갣"#),
        (r#"갤"#, r#"갤"#, r#"갤"#, r#"갤"#, r#"갤"#),
        (r#"갥"#, r#"갥"#, r#"갥"#, r#"갥"#, r#"갥"#),
        (r#"갦"#, r#"갦"#, r#"갦"#, r#"갦"#, r#"갦"#),
        (r#"갧"#, r#"갧"#, r#"갧"#, r#"갧"#, r#"갧"#),
        (r#"갨"#, r#"갨"#, r#"갨"#, r#"갨"#, r#"갨"#),
        (r#"갩"#, r#"갩"#, r#"갩"#, r#"갩"#, r#"갩"#),
        (r#"갪"#, r#"갪"#, r#"갪"#, r#"갪"#, r#"갪"#),
        (r#"갫"#, r#"갫"#, r#"갫"#, r#"갫"#, r#"갫"#),
        (r#"갬"#, r#"갬"#, r#"갬"#, r#"갬"#, r#"갬"#),
        (r#"갭"#, r#"갭"#, r#"갭"#, r#"갭"#, r#"갭"#),
        (r#"갮"#, r#"갮"#, r#"갮"#, r#"갮"#, r#"갮"#),
        (r#"갯"#, r#"갯"#, r#"갯"#, r#"갯"#, r#"갯"#),
        (r#"갰"#, r#"갰"#, r#"갰"#, r#"갰"#, r#"갰"#),
        (r#"갱"#, r#"갱"#, r#"갱"#, r#"갱"#, r#"갱"#),
        (r#"갲"#, r#"갲"#, r#"갲"#, r#"갲"#, r#"갲"#),
        (r#"갳"#, r#"갳"#, r#"갳"#, r#"갳"#, r#"갳"#),
        (r#"갴"#, r#"갴"#, r#"갴"#, r#"갴"#, r#"갴"#),
        (r#"갵"#, r#"갵"#, r#"갵"#, r#"갵"#, r#"갵"#),
        (r#"갶"#, r#"갶"#, r#"갶"#, r#"갶"#, r#"갶"#),
        (r#"갷"#, r#"갷"#, r#"갷"#, r#"갷"#, r#"갷"#),
        (r#"갸"#, r#"갸"#, r#"갸"#, r#"갸"#, r#"갸"#),
        (r#"갹"#, r#"갹"#, r#"갹"#, r#"갹"#, r#"갹"#),
        (r#"갺"#, r#"갺"#, r#"갺"#, r#"갺"#, r#"갺"#),
        (r#"갻"#, r#"갻"#, r#"갻"#, r#"갻"#, r#"갻"#),
        (r#"갼"#, r#"갼"#, r#"갼"#, r#"갼"#, r#"갼"#),
        (r#"갽"#, r#"갽"#, r#"갽"#, r#"갽"#, r#"갽"#),
        (r#"갾"#, r#"갾"#, r#"갾"#, r#"갾"#, r#"갾"#),
        (r#"갿"#, r#"갿"#, r#"갿"#, r#"갿"#, r#"갿"#),
        (r#"걀"#, r#"걀"#, r#"걀"#, r#"걀"#, r#"걀"#),
        (r#"걁"#, r#"걁"#, r#"걁"#, r#"걁"#, r#"걁"#),
        (r#"걂"#, r#"걂"#, r#"걂"#, r#"걂"#, r#"걂"#),
        (r#"걃"#, r#"걃"#, r#"걃"#, r#"걃"#, r#"걃"#),
        (r#"걄"#, r#"걄"#, r#"걄"#, r#"걄"#, r#"걄"#),
        (r#"걅"#, r#"걅"#, r#"걅"#, r#"걅"#, r#"걅"#),
        (r#"걆"#, r#"걆"#, r#"걆"#, r#"걆"#, r#"걆"#),
        (r#"걇"#, r#"걇"#, r#"걇"#, r#"걇"#, r#"걇"#),
        (r#"걈"#, r#"걈"#, r#"걈"#, r#"걈"#, r#"걈"#),
        (r#"걉"#, r#"걉"#, r#"걉"#, r#"걉"#, r#"걉"#),
        (r#"걊"#, r#"걊"#, r#"걊"#, r#"걊"#, r#"걊"#),
        (r#"걋"#, r#"걋"#, r#"걋"#, r#"걋"#, r#"걋"#),
        (r#"걌"#, r#"걌"#, r#"걌"#, r#"걌"#, r#"걌"#),
        (r#"걍"#, r#"걍"#, r#"걍"#, r#"걍"#, r#"걍"#),
        (r#"걎"#, r#"걎"#, r#"걎"#, r#"걎"#, r#"걎"#),
        (r#"걏"#, r#"걏"#, r#"걏"#, r#"걏"#, r#"걏"#),
        (r#"걐"#, r#"걐"#, r#"걐"#, r#"걐"#, r#"걐"#),
        (r#"걑"#, r#"걑"#, r#"걑"#, r#"걑"#, r#"걑"#),
        (r#"걒"#, r#"걒"#, r#"걒"#, r#"걒"#, r#"걒"#),
        (r#"걓"#, r#"걓"#, r#"걓"#, r#"걓"#, r#"걓"#),
        (r#"걔"#, r#"걔"#, r#"걔"#, r#"걔"#, r#"걔"#),
        (r#"걕"#, r#"걕"#, r#"걕"#, r#"걕"#, r#"걕"#),
        (r#"걖"#, r#"걖"#, r#"걖"#, r#"걖"#, r#"걖"#),
        (r#"걗"#, r#"걗"#, r#"걗"#, r#"걗"#, r#"걗"#),
        (r#"걘"#, r#"걘"#, r#"걘"#, r#"걘"#, r#"걘"#),
        (r#"걙"#, r#"걙"#, r#"걙"#, r#"걙"#, r#"걙"#),
        (r#"걚"#, r#"걚"#, r#"걚"#, r#"걚"#, r#"걚"#),
        (r#"걛"#, r#"걛"#, r#"걛"#, r#"걛"#, r#"걛"#),
        (r#"걜"#, r#"걜"#, r#"걜"#, r#"걜"#, r#"걜"#),
        (r#"걝"#, r#"걝"#, r#"걝"#, r#"걝"#, r#"걝"#),
        (r#"걞"#, r#"걞"#, r#"걞"#, r#"걞"#, r#"걞"#),
        (r#"걟"#, r#"걟"#, r#"걟"#, r#"걟"#, r#"걟"#),
        (r#"걠"#, r#"걠"#, r#"걠"#, r#"걠"#, r#"걠"#),
        (r#"걡"#, r#"걡"#, r#"걡"#, r#"걡"#, r#"걡"#),
        (r#"걢"#, r#"걢"#, r#"걢"#, r#"걢"#, r#"걢"#),
        (r#"걣"#, r#"걣"#, r#"걣"#, r#"걣"#, r#"걣"#),
        (r#"걤"#, r#"걤"#, r#"걤"#, r#"걤"#, r#"걤"#),
        (r#"걥"#, r#"걥"#, r#"걥"#, r#"걥"#, r#"걥"#),
        (r#"걦"#, r#"걦"#, r#"걦"#, r#"걦"#, r#"걦"#),
        (r#"걧"#, r#"걧"#, r#"걧"#, r#"걧"#, r#"걧"#),
        (r#"걨"#, r#"걨"#, r#"걨"#, r#"걨"#, r#"걨"#),
        (r#"걩"#, r#"걩"#, r#"걩"#, r#"걩"#, r#"걩"#),
        (r#"걪"#, r#"걪"#, r#"걪"#, r#"걪"#, r#"걪"#),
        (r#"걫"#, r#"걫"#, r#"걫"#, r#"걫"#, r#"걫"#),
        (r#"걬"#, r#"걬"#, r#"걬"#, r#"걬"#, r#"걬"#),
        (r#"걭"#, r#"걭"#, r#"걭"#, r#"걭"#, r#"걭"#),
        (r#"걮"#, r#"걮"#, r#"걮"#, r#"걮"#, r#"걮"#),
        (r#"걯"#, r#"걯"#, r#"걯"#, r#"걯"#, r#"걯"#),
        (r#"거"#, r#"거"#, r#"거"#, r#"거"#, r#"거"#),
        (r#"걱"#, r#"걱"#, r#"걱"#, r#"걱"#, r#"걱"#),
        (r#"걲"#, r#"걲"#, r#"걲"#, r#"걲"#, r#"걲"#),
        (r#"걳"#, r#"걳"#, r#"걳"#, r#"걳"#, r#"걳"#),
        (r#"건"#, r#"건"#, r#"건"#, r#"건"#, r#"건"#),
        (r#"걵"#, r#"걵"#, r#"걵"#, r#"걵"#, r#"걵"#),
        (r#"걶"#, r#"걶"#, r#"걶"#, r#"걶"#, r#"걶"#),
        (r#"걷"#, r#"걷"#, r#"걷"#, r#"걷"#, r#"걷"#),
        (r#"걸"#, r#"걸"#, r#"걸"#, r#"걸"#, r#"걸"#),
        (r#"걹"#, r#"걹"#, r#"걹"#, r#"걹"#, r#"걹"#),
        (r#"걺"#, r#"걺"#, r#"걺"#, r#"걺"#, r#"걺"#),
        (r#"걻"#, r#"걻"#, r#"걻"#, r#"걻"#, r#"걻"#),
        (r#"걼"#, r#"걼"#, r#"걼"#, r#"걼"#, r#"걼"#),
        (r#"걽"#, r#"걽"#, r#"걽"#, r#"걽"#, r#"걽"#),
        (r#"걾"#, r#"걾"#, r#"걾"#, r#"걾"#, r#"걾"#),
        (r#"걿"#, r#"걿"#, r#"걿"#, r#"걿"#, r#"걿"#),
        (r#"검"#, r#"검"#, r#"검"#, r#"검"#, r#"검"#),
        (r#"겁"#, r#"겁"#, r#"겁"#, r#"겁"#, r#"겁"#),
        (r#"겂"#, r#"겂"#, r#"겂"#, r#"겂"#, r#"겂"#),
        (r#"것"#, r#"것"#, r#"것"#, r#"것"#, r#"것"#),
        (r#"겄"#, r#"겄"#, r#"겄"#, r#"겄"#, r#"겄"#),
        (r#"겅"#, r#"겅"#, r#"겅"#, r#"겅"#, r#"겅"#),
        (r#"겆"#, r#"겆"#, r#"겆"#, r#"겆"#, r#"겆"#),
        (r#"겇"#, r#"겇"#, r#"겇"#, r#"겇"#, r#"겇"#),
        (r#"겈"#, r#"겈"#, r#"겈"#, r#"겈"#, r#"겈"#),
        (r#"겉"#, r#"겉"#, r#"겉"#, r#"겉"#, r#"겉"#),
        (r#"겊"#, r#"겊"#, r#"겊"#, r#"겊"#, r#"겊"#),
        (r#"겋"#, r#"겋"#, r#"겋"#, r#"겋"#, r#"겋"#),
        (r#"게"#, r#"게"#, r#"게"#, r#"게"#, r#"게"#),
        (r#"겍"#, r#"겍"#, r#"겍"#, r#"겍"#, r#"겍"#),
        (r#"겎"#, r#"겎"#, r#"겎"#, r#"겎"#, r#"겎"#),
        (r#"겏"#, r#"겏"#, r#"겏"#, r#"겏"#, r#"겏"#),
        (r#"겐"#, r#"겐"#, r#"겐"#, r#"겐"#, r#"겐"#),
        (r#"겑"#, r#"겑"#, r#"겑"#, r#"겑"#, r#"겑"#),
        (r#"겒"#, r#"겒"#, r#"겒"#, r#"겒"#, r#"겒"#),
        (r#"겓"#, r#"겓"#, r#"겓"#, r#"겓"#, r#"겓"#),
        (r#"겔"#, r#"겔"#, r#"겔"#, r#"겔"#, r#"겔"#),
        (r#"겕"#, r#"겕"#, r#"겕"#, r#"겕"#, r#"겕"#),
        (r#"겖"#, r#"겖"#, r#"겖"#, r#"겖"#, r#"겖"#),
        (r#"겗"#, r#"겗"#, r#"겗"#, r#"겗"#, r#"겗"#),
        (r#"겘"#, r#"겘"#, r#"겘"#, r#"겘"#, r#"겘"#),
        (r#"겙"#, r#"겙"#, r#"겙"#, r#"겙"#, r#"겙"#),
        (r#"겚"#, r#"겚"#, r#"겚"#, r#"겚"#, r#"겚"#),
        (r#"겛"#, r#"겛"#, r#"겛"#, r#"겛"#, r#"겛"#),
        (r#"겜"#, r#"겜"#, r#"겜"#, r#"겜"#, r#"겜"#),
        (r#"겝"#, r#"겝"#, r#"겝"#, r#"겝"#, r#"겝"#),
        (r#"겞"#, r#"겞"#, r#"겞"#, r#"겞"#, r#"겞"#),
        (r#"겟"#, r#"겟"#, r#"겟"#, r#"겟"#, r#"겟"#),
        (r#"겠"#, r#"겠"#, r#"겠"#, r#"겠"#, r#"겠"#),
        (r#"겡"#, r#"겡"#, r#"겡"#, r#"겡"#, r#"겡"#),
        (r#"겢"#, r#"겢"#, r#"겢"#, r#"겢"#, r#"겢"#),
        (r#"겣"#, r#"겣"#, r#"겣"#, r#"겣"#, r#"겣"#),
        (r#"겤"#, r#"겤"#, r#"겤"#, r#"겤"#, r#"겤"#),
        (r#"겥"#, r#"겥"#, r#"겥"#, r#"겥"#, r#"겥"#),
        (r#"겦"#, r#"겦"#, r#"겦"#, r#"겦"#, r#"겦"#),
        (r#"겧"#, r#"겧"#, r#"겧"#, r#"겧"#, r#"겧"#),
        (r#"겨"#, r#"겨"#, r#"겨"#, r#"겨"#, r#"겨"#),
        (r#"격"#, r#"격"#, r#"격"#, r#"격"#, r#"격"#),
        (r#"겪"#, r#"겪"#, r#"겪"#, r#"겪"#, r#"겪"#),
        (r#"겫"#, r#"겫"#, r#"겫"#, r#"겫"#, r#"겫"#),
        (r#"견"#, r#"견"#, r#"견"#, r#"견"#, r#"견"#),
        (r#"겭"#, r#"겭"#, r#"겭"#, r#"겭"#, r#"겭"#),
        (r#"겮"#, r#"겮"#, r#"겮"#, r#"겮"#, r#"겮"#),
        (r#"겯"#, r#"겯"#, r#"겯"#, r#"겯"#, r#"겯"#),
        (r#"결"#, r#"결"#, r#"결"#, r#"결"#, r#"결"#),
        (r#"겱"#, r#"겱"#, r#"겱"#, r#"겱"#, r#"겱"#),
        (r#"겲"#, r#"겲"#, r#"겲"#, r#"겲"#, r#"겲"#),
        (r#"겳"#, r#"겳"#, r#"겳"#, r#"겳"#, r#"겳"#),
        (r#"겴"#, r#"겴"#, r#"겴"#, r#"겴"#, r#"겴"#),
        (r#"겵"#, r#"겵"#, r#"겵"#, r#"겵"#, r#"겵"#),
        (r#"겶"#, r#"겶"#, r#"겶"#, r#"겶"#, r#"겶"#),
        (r#"겷"#, r#"겷"#, r#"겷"#, r#"겷"#, r#"겷"#),
        (r#"겸"#, r#"겸"#, r#"겸"#, r#"겸"#, r#"겸"#),
        (r#"겹"#, r#"겹"#, r#"겹"#, r#"겹"#, r#"겹"#),
        (r#"겺"#, r#"겺"#, r#"겺"#, r#"겺"#, r#"겺"#),
        (r#"겻"#, r#"겻"#, r#"겻"#, r#"겻"#, r#"겻"#),
        (r#"겼"#, r#"겼"#, r#"겼"#, r#"겼"#, r#"겼"#),
        (r#"경"#, r#"경"#, r#"경"#, r#"경"#, r#"경"#),
        (r#"겾"#, r#"겾"#, r#"겾"#, r#"겾"#, r#"겾"#),
        (r#"겿"#, r#"겿"#, r#"겿"#, r#"겿"#, r#"겿"#),
        (r#"곀"#, r#"곀"#, r#"곀"#, r#"곀"#, r#"곀"#),
        (r#"곁"#, r#"곁"#, r#"곁"#, r#"곁"#, r#"곁"#),
        (r#"곂"#, r#"곂"#, r#"곂"#, r#"곂"#, r#"곂"#),
        (r#"곃"#, r#"곃"#, r#"곃"#, r#"곃"#, r#"곃"#),
        (r#"계"#, r#"계"#, r#"계"#, r#"계"#, r#"계"#),
        (r#"곅"#, r#"곅"#, r#"곅"#, r#"곅"#, r#"곅"#),
        (r#"곆"#, r#"곆"#, r#"곆"#, r#"곆"#, r#"곆"#),
        (r#"곇"#, r#"곇"#, r#"곇"#, r#"곇"#, r#"곇"#),
        (r#"곈"#, r#"곈"#, r#"곈"#, r#"곈"#, r#"곈"#),
        (r#"곉"#, r#"곉"#, r#"곉"#, r#"곉"#, r#"곉"#),
        (r#"곊"#, r#"곊"#, r#"곊"#, r#"곊"#, r#"곊"#),
        (r#"곋"#, r#"곋"#, r#"곋"#, r#"곋"#, r#"곋"#),
        (r#"곌"#, r#"곌"#, r#"곌"#, r#"곌"#, r#"곌"#),
        (r#"곍"#, r#"곍"#, r#"곍"#, r#"곍"#, r#"곍"#),
        (r#"곎"#, r#"곎"#, r#"곎"#, r#"곎"#, r#"곎"#),
        (r#"곏"#, r#"곏"#, r#"곏"#, r#"곏"#, r#"곏"#),
        (r#"곐"#, r#"곐"#, r#"곐"#, r#"곐"#, r#"곐"#),
        (r#"곑"#, r#"곑"#, r#"곑"#, r#"곑"#, r#"곑"#),
        (r#"곒"#, r#"곒"#, r#"곒"#, r#"곒"#, r#"곒"#),
        (r#"곓"#, r#"곓"#, r#"곓"#, r#"곓"#, r#"곓"#),
        (r#"곔"#, r#"곔"#, r#"곔"#, r#"곔"#, r#"곔"#),
        (r#"곕"#, r#"곕"#, r#"곕"#, r#"곕"#, r#"곕"#),
        (r#"곖"#, r#"곖"#, r#"곖"#, r#"곖"#, r#"곖"#),
        (r#"곗"#, r#"곗"#, r#"곗"#, r#"곗"#, r#"곗"#),
        (r#"곘"#, r#"곘"#, r#"곘"#, r#"곘"#, r#"곘"#),
        (r#"곙"#, r#"곙"#, r#"곙"#, r#"곙"#, r#"곙"#),
        (r#"곚"#, r#"곚"#, r#"곚"#, r#"곚"#, r#"곚"#),
        (r#"곛"#, r#"곛"#, r#"곛"#, r#"곛"#, r#"곛"#),
        (r#"곜"#, r#"곜"#, r#"곜"#, r#"곜"#, r#"곜"#),
        (r#"곝"#, r#"곝"#, r#"곝"#, r#"곝"#, r#"곝"#),
        (r#"곞"#, r#"곞"#, r#"곞"#, r#"곞"#, r#"곞"#),
        (r#"곟"#, r#"곟"#, r#"곟"#, r#"곟"#, r#"곟"#),
        (r#"고"#, r#"고"#, r#"고"#, r#"고"#, r#"고"#),
        (r#"곡"#, r#"곡"#, r#"곡"#, r#"곡"#, r#"곡"#),
        (r#"곢"#, r#"곢"#, r#"곢"#, r#"곢"#, r#"곢"#),
        (r#"곣"#, r#"곣"#, r#"곣"#, r#"곣"#, r#"곣"#),
        (r#"곤"#, r#"곤"#, r#"곤"#, r#"곤"#, r#"곤"#),
        (r#"곥"#, r#"곥"#, r#"곥"#, r#"곥"#, r#"곥"#),
        (r#"곦"#, r#"곦"#, r#"곦"#, r#"곦"#, r#"곦"#),
        (r#"곧"#, r#"곧"#, r#"곧"#, r#"곧"#, r#"곧"#),
        (r#"골"#, r#"골"#, r#"골"#, r#"골"#, r#"골"#),
        (r#"곩"#, r#"곩"#, r#"곩"#, r#"곩"#, r#"곩"#),
        (r#"곪"#, r#"곪"#, r#"곪"#, r#"곪"#, r#"곪"#),
        (r#"곫"#, r#"곫"#, r#"곫"#, r#"곫"#, r#"곫"#),
        (r#"곬"#, r#"곬"#, r#"곬"#, r#"곬"#, r#"곬"#),
        (r#"곭"#, r#"곭"#, r#"곭"#, r#"곭"#, r#"곭"#),
        (r#"곮"#, r#"곮"#, r#"곮"#, r#"곮"#, r#"곮"#),
        (r#"곯"#, r#"곯"#, r#"곯"#, r#"곯"#, r#"곯"#),
        (r#"곰"#, r#"곰"#, r#"곰"#, r#"곰"#, r#"곰"#),
        (r#"곱"#, r#"곱"#, r#"곱"#, r#"곱"#, r#"곱"#),
        (r#"곲"#, r#"곲"#, r#"곲"#, r#"곲"#, r#"곲"#),
        (r#"곳"#, r#"곳"#, r#"곳"#, r#"곳"#, r#"곳"#),
        (r#"곴"#, r#"곴"#, r#"곴"#, r#"곴"#, r#"곴"#),
        (r#"공"#, r#"공"#, r#"공"#, r#"공"#, r#"공"#),
        (r#"곶"#, r#"곶"#, r#"곶"#, r#"곶"#, r#"곶"#),
        (r#"곷"#, r#"곷"#, r#"곷"#, r#"곷"#, r#"곷"#),
        (r#"곸"#, r#"곸"#, r#"곸"#, r#"곸"#, r#"곸"#),
        (r#"곹"#, r#"곹"#, r#"곹"#, r#"곹"#, r#"곹"#),
        (r#"곺"#, r#"곺"#, r#"곺"#, r#"곺"#, r#"곺"#),
        (r#"곻"#, r#"곻"#, r#"곻"#, r#"곻"#, r#"곻"#),
        (r#"과"#, r#"과"#, r#"과"#, r#"과"#, r#"과"#),
        (r#"곽"#, r#"곽"#, r#"곽"#, r#"곽"#, r#"곽"#),
        (r#"곾"#, r#"곾"#, r#"곾"#, r#"곾"#, r#"곾"#),
        (r#"곿"#, r#"곿"#, r#"곿"#, r#"곿"#, r#"곿"#),
        (r#"관"#, r#"관"#, r#"관"#, r#"관"#, r#"관"#),
        (r#"괁"#, r#"괁"#, r#"괁"#, r#"괁"#, r#"괁"#),
        (r#"괂"#, r#"괂"#, r#"괂"#, r#"괂"#, r#"괂"#),
        (r#"괃"#, r#"괃"#, r#"괃"#, r#"괃"#, r#"괃"#),
        (r#"괄"#, r#"괄"#, r#"괄"#, r#"괄"#, r#"괄"#),
        (r#"괅"#, r#"괅"#, r#"괅"#, r#"괅"#, r#"괅"#),
        (r#"괆"#, r#"괆"#, r#"괆"#, r#"괆"#, r#"괆"#),
        (r#"괇"#, r#"괇"#, r#"괇"#, r#"괇"#, r#"괇"#),
        (r#"괈"#, r#"괈"#, r#"괈"#, r#"괈"#, r#"괈"#),
        (r#"괉"#, r#"괉"#, r#"괉"#, r#"괉"#, r#"괉"#),
        (r#"괊"#, r#"괊"#, r#"괊"#, r#"괊"#, r#"괊"#),
        (r#"괋"#, r#"괋"#, r#"괋"#, r#"괋"#, r#"괋"#),
        (r#"괌"#, r#"괌"#, r#"괌"#, r#"괌"#, r#"괌"#),
        (r#"괍"#, r#"괍"#, r#"괍"#, r#"괍"#, r#"괍"#),
        (r#"괎"#, r#"괎"#, r#"괎"#, r#"괎"#, r#"괎"#),
        (r#"괏"#, r#"괏"#, r#"괏"#, r#"괏"#, r#"괏"#),
        (r#"괐"#, r#"괐"#, r#"괐"#, r#"괐"#, r#"괐"#),
        (r#"광"#, r#"광"#, r#"광"#, r#"광"#, r#"광"#),
        (r#"괒"#, r#"괒"#, r#"괒"#, r#"괒"#, r#"괒"#),
        (r#"괓"#, r#"괓"#, r#"괓"#, r#"괓"#, r#"괓"#),
        (r#"괔"#, r#"괔"#, r#"괔"#, r#"괔"#, r#"괔"#),
        (r#"괕"#, r#"괕"#, r#"괕"#, r#"괕"#, r#"괕"#),
        (r#"괖"#, r#"괖"#, r#"괖"#, r#"괖"#, r#"괖"#),
        (r#"괗"#, r#"괗"#, r#"괗"#, r#"괗"#, r#"괗"#),
        (r#"괘"#, r#"괘"#, r#"괘"#, r#"괘"#, r#"괘"#),
        (r#"괙"#, r#"괙"#, r#"괙"#, r#"괙"#, r#"괙"#),
        (r#"괚"#, r#"괚"#, r#"괚"#, r#"괚"#, r#"괚"#),
        (r#"괛"#, r#"괛"#, r#"괛"#, r#"괛"#, r#"괛"#),
        (r#"괜"#, r#"괜"#, r#"괜"#, r#"괜"#, r#"괜"#),
        (r#"괝"#, r#"괝"#, r#"괝"#, r#"괝"#, r#"괝"#),
        (r#"괞"#, r#"괞"#, r#"괞"#, r#"괞"#, r#"괞"#),
        (r#"괟"#, r#"괟"#, r#"괟"#, r#"괟"#, r#"괟"#),
        (r#"괠"#, r#"괠"#, r#"괠"#, r#"괠"#, r#"괠"#),
        (r#"괡"#, r#"괡"#, r#"괡"#, r#"괡"#, r#"괡"#),
        (r#"괢"#, r#"괢"#, r#"괢"#, r#"괢"#, r#"괢"#),
        (r#"괣"#, r#"괣"#, r#"괣"#, r#"괣"#, r#"괣"#),
        (r#"괤"#, r#"괤"#, r#"괤"#, r#"괤"#, r#"괤"#),
        (r#"괥"#, r#"괥"#, r#"괥"#, r#"괥"#, r#"괥"#),
        (r#"괦"#, r#"괦"#, r#"괦"#, r#"괦"#, r#"괦"#),
        (r#"괧"#, r#"괧"#, r#"괧"#, r#"괧"#, r#"괧"#),
        (r#"괨"#, r#"괨"#, r#"괨"#, r#"괨"#, r#"괨"#),
        (r#"괩"#, r#"괩"#, r#"괩"#, r#"괩"#, r#"괩"#),
        (r#"괪"#, r#"괪"#, r#"괪"#, r#"괪"#, r#"괪"#),
        (r#"괫"#, r#"괫"#, r#"괫"#, r#"괫"#, r#"괫"#),
        (r#"괬"#, r#"괬"#, r#"괬"#, r#"괬"#, r#"괬"#),
        (r#"괭"#, r#"괭"#, r#"괭"#, r#"괭"#, r#"괭"#),
        (r#"괮"#, r#"괮"#, r#"괮"#, r#"괮"#, r#"괮"#),
        (r#"괯"#, r#"괯"#, r#"괯"#, r#"괯"#, r#"괯"#),
        (r#"괰"#, r#"괰"#, r#"괰"#, r#"괰"#, r#"괰"#),
        (r#"괱"#, r#"괱"#, r#"괱"#, r#"괱"#, r#"괱"#),
        (r#"괲"#, r#"괲"#, r#"괲"#, r#"괲"#, r#"괲"#),
        (r#"괳"#, r#"괳"#, r#"괳"#, r#"괳"#, r#"괳"#),
        (r#"괴"#, r#"괴"#, r#"괴"#, r#"괴"#, r#"괴"#),
        (r#"괵"#, r#"괵"#, r#"괵"#, r#"괵"#, r#"괵"#),
        (r#"괶"#, r#"괶"#, r#"괶"#, r#"괶"#, r#"괶"#),
        (r#"괷"#, r#"괷"#, r#"괷"#, r#"괷"#, r#"괷"#),
        (r#"괸"#, r#"괸"#, r#"괸"#, r#"괸"#, r#"괸"#),
        (r#"괹"#, r#"괹"#, r#"괹"#, r#"괹"#, r#"괹"#),
        (r#"괺"#, r#"괺"#, r#"괺"#, r#"괺"#, r#"괺"#),
        (r#"괻"#, r#"괻"#, r#"괻"#, r#"괻"#, r#"괻"#),
        (r#"괼"#, r#"괼"#, r#"괼"#, r#"괼"#, r#"괼"#),
        (r#"괽"#, r#"괽"#, r#"괽"#, r#"괽"#, r#"괽"#),
        (r#"괾"#, r#"괾"#, r#"괾"#, r#"괾"#, r#"괾"#),
        (r#"괿"#, r#"괿"#, r#"괿"#, r#"괿"#, r#"괿"#),
        (r#"굀"#, r#"굀"#, r#"굀"#, r#"굀"#, r#"굀"#),
        (r#"굁"#, r#"굁"#, r#"굁"#, r#"굁"#, r#"굁"#),
        (r#"굂"#, r#"굂"#, r#"굂"#, r#"굂"#, r#"굂"#),
        (r#"굃"#, r#"굃"#, r#"굃"#, r#"굃"#, r#"굃"#),
        (r#"굄"#, r#"굄"#, r#"굄"#, r#"굄"#, r#"굄"#),
        (r#"굅"#, r#"굅"#, r#"굅"#, r#"굅"#, r#"굅"#),
        (r#"굆"#, r#"굆"#, r#"굆"#, r#"굆"#, r#"굆"#),
        (r#"굇"#, r#"굇"#, r#"굇"#, r#"굇"#, r#"굇"#),
        (r#"굈"#, r#"굈"#, r#"굈"#, r#"굈"#, r#"굈"#),
        (r#"굉"#, r#"굉"#, r#"굉"#, r#"굉"#, r#"굉"#),
        (r#"굊"#, r#"굊"#, r#"굊"#, r#"굊"#, r#"굊"#),
        (r#"굋"#, r#"굋"#, r#"굋"#, r#"굋"#, r#"굋"#),
        (r#"굌"#, r#"굌"#, r#"굌"#, r#"굌"#, r#"굌"#),
        (r#"굍"#, r#"굍"#, r#"굍"#, r#"굍"#, r#"굍"#),
        (r#"굎"#, r#"굎"#, r#"굎"#, r#"굎"#, r#"굎"#),
        (r#"굏"#, r#"굏"#, r#"굏"#, r#"굏"#, r#"굏"#),
        (r#"교"#, r#"교"#, r#"교"#, r#"교"#, r#"교"#),
        (r#"굑"#, r#"굑"#, r#"굑"#, r#"굑"#, r#"굑"#),
        (r#"굒"#, r#"굒"#, r#"굒"#, r#"굒"#, r#"굒"#),
        (r#"굓"#, r#"굓"#, r#"굓"#, r#"굓"#, r#"굓"#),
        (r#"굔"#, r#"굔"#, r#"굔"#, r#"굔"#, r#"굔"#),
        (r#"굕"#, r#"굕"#, r#"굕"#, r#"굕"#, r#"굕"#),
        (r#"굖"#, r#"굖"#, r#"굖"#, r#"굖"#, r#"굖"#),
        (r#"굗"#, r#"굗"#, r#"굗"#, r#"굗"#, r#"굗"#),
        (r#"굘"#, r#"굘"#, r#"굘"#, r#"굘"#, r#"굘"#),
        (r#"굙"#, r#"굙"#, r#"굙"#, r#"굙"#, r#"굙"#),
        (r#"굚"#, r#"굚"#, r#"굚"#, r#"굚"#, r#"굚"#),
        (r#"굛"#, r#"굛"#, r#"굛"#, r#"굛"#, r#"굛"#),
        (r#"굜"#, r#"굜"#, r#"굜"#, r#"굜"#, r#"굜"#),
        (r#"굝"#, r#"굝"#, r#"굝"#, r#"굝"#, r#"굝"#),
        (r#"굞"#, r#"굞"#, r#"굞"#, r#"굞"#, r#"굞"#),
        (r#"굟"#, r#"굟"#, r#"굟"#, r#"굟"#, r#"굟"#),
        (r#"굠"#, r#"굠"#, r#"굠"#, r#"굠"#, r#"굠"#),
        (r#"굡"#, r#"굡"#, r#"굡"#, r#"굡"#, r#"굡"#),
        (r#"굢"#, r#"굢"#, r#"굢"#, r#"굢"#, r#"굢"#),
        (r#"굣"#, r#"굣"#, r#"굣"#, r#"굣"#, r#"굣"#),
        (r#"굤"#, r#"굤"#, r#"굤"#, r#"굤"#, r#"굤"#),
        (r#"굥"#, r#"굥"#, r#"굥"#, r#"굥"#, r#"굥"#),
        (r#"굦"#, r#"굦"#, r#"굦"#, r#"굦"#, r#"굦"#),
        (r#"굧"#, r#"굧"#, r#"굧"#, r#"굧"#, r#"굧"#),
        (r#"굨"#, r#"굨"#, r#"굨"#, r#"굨"#, r#"굨"#),
        (r#"굩"#, r#"굩"#, r#"굩"#, r#"굩"#, r#"굩"#),
        (r#"굪"#, r#"굪"#, r#"굪"#, r#"굪"#, r#"굪"#),
        (r#"굫"#, r#"굫"#, r#"굫"#, r#"굫"#, r#"굫"#),
        (r#"구"#, r#"구"#, r#"구"#, r#"구"#, r#"구"#),
        (r#"국"#, r#"국"#, r#"국"#, r#"국"#, r#"국"#),
        (r#"굮"#, r#"굮"#, r#"굮"#, r#"굮"#, r#"굮"#),
        (r#"굯"#, r#"굯"#, r#"굯"#, r#"굯"#, r#"굯"#),
        (r#"군"#, r#"군"#, r#"군"#, r#"군"#, r#"군"#),
        (r#"굱"#, r#"굱"#, r#"굱"#, r#"굱"#, r#"굱"#),
        (r#"굲"#, r#"굲"#, r#"굲"#, r#"굲"#, r#"굲"#),
        (r#"굳"#, r#"굳"#, r#"굳"#, r#"굳"#, r#"굳"#),
        (r#"굴"#, r#"굴"#, r#"굴"#, r#"굴"#, r#"굴"#),
        (r#"굵"#, r#"굵"#, r#"굵"#, r#"굵"#, r#"굵"#),
        (r#"굶"#, r#"굶"#, r#"굶"#, r#"굶"#, r#"굶"#),
        (r#"굷"#, r#"굷"#, r#"굷"#, r#"굷"#, r#"굷"#),
        (r#"굸"#, r#"굸"#, r#"굸"#, r#"굸"#, r#"굸"#),
        (r#"굹"#, r#"굹"#, r#"굹"#, r#"굹"#, r#"굹"#),
        (r#"굺"#, r#"굺"#, r#"굺"#, r#"굺"#, r#"굺"#),
        (r#"굻"#, r#"굻"#, r#"굻"#, r#"굻"#, r#"굻"#),
        (r#"굼"#, r#"굼"#, r#"굼"#, r#"굼"#, r#"굼"#),
        (r#"굽"#, r#"굽"#, r#"굽"#, r#"굽"#, r#"굽"#),
        (r#"굾"#, r#"굾"#, r#"굾"#, r#"굾"#, r#"굾"#),
        (r#"굿"#, r#"굿"#, r#"굿"#, r#"굿"#, r#"굿"#),
        (r#"궀"#, r#"궀"#, r#"궀"#, r#"궀"#, r#"궀"#),
        (r#"궁"#, r#"궁"#, r#"궁"#, r#"궁"#, r#"궁"#),
        (r#"궂"#, r#"궂"#, r#"궂"#, r#"궂"#, r#"궂"#),
        (r#"궃"#, r#"궃"#, r#"궃"#, r#"궃"#, r#"궃"#),
        (r#"궄"#, r#"궄"#, r#"궄"#, r#"궄"#, r#"궄"#),
        (r#"궅"#, r#"궅"#, r#"궅"#, r#"궅"#, r#"궅"#),
        (r#"궆"#, r#"궆"#, r#"궆"#, r#"궆"#, r#"궆"#),
        (r#"궇"#, r#"궇"#, r#"궇"#, r#"궇"#, r#"궇"#),
        (r#"궈"#, r#"궈"#, r#"궈"#, r#"궈"#, r#"궈"#),
        (r#"궉"#, r#"궉"#, r#"궉"#, r#"궉"#, r#"궉"#),
        (r#"궊"#, r#"궊"#, r#"궊"#, r#"궊"#, r#"궊"#),
        (r#"궋"#, r#"궋"#, r#"궋"#, r#"궋"#, r#"궋"#),
        (r#"권"#, r#"권"#, r#"권"#, r#"권"#, r#"권"#),
        (r#"궍"#, r#"궍"#, r#"궍"#, r#"궍"#, r#"궍"#),
        (r#"궎"#, r#"궎"#, r#"궎"#, r#"궎"#, r#"궎"#),
        (r#"궏"#, r#"궏"#, r#"궏"#, r#"궏"#, r#"궏"#),
        (r#"궐"#, r#"궐"#, r#"궐"#, r#"궐"#, r#"궐"#),
        (r#"궑"#, r#"궑"#, r#"궑"#, r#"궑"#, r#"궑"#),
        (r#"궒"#, r#"궒"#, r#"궒"#, r#"궒"#, r#"궒"#),
        (r#"궓"#, r#"궓"#, r#"궓"#, r#"궓"#, r#"궓"#),
        (r#"궔"#, r#"궔"#, r#"궔"#, r#"궔"#, r#"궔"#),
        (r#"궕"#, r#"궕"#, r#"궕"#, r#"궕"#, r#"궕"#),
        (r#"궖"#, r#"궖"#, r#"궖"#, r#"궖"#, r#"궖"#),
        (r#"궗"#, r#"궗"#, r#"궗"#, r#"궗"#, r#"궗"#),
        (r#"궘"#, r#"궘"#, r#"궘"#, r#"궘"#, r#"궘"#),
        (r#"궙"#, r#"궙"#, r#"궙"#, r#"궙"#, r#"궙"#),
        (r#"궚"#, r#"궚"#, r#"궚"#, r#"궚"#, r#"궚"#),
        (r#"궛"#, r#"궛"#, r#"궛"#, r#"궛"#, r#"궛"#),
        (r#"궜"#, r#"궜"#, r#"궜"#, r#"궜"#, r#"궜"#),
        (r#"궝"#, r#"궝"#, r#"궝"#, r#"궝"#, r#"궝"#),
        (r#"궞"#, r#"궞"#, r#"궞"#, r#"궞"#, r#"궞"#),
        (r#"궟"#, r#"궟"#, r#"궟"#, r#"궟"#, r#"궟"#),
        (r#"궠"#, r#"궠"#, r#"궠"#, r#"궠"#, r#"궠"#),
        (r#"궡"#, r#"궡"#, r#"궡"#, r#"궡"#, r#"궡"#),
        (r#"궢"#, r#"궢"#, r#"궢"#, r#"궢"#, r#"궢"#),
        (r#"궣"#, r#"궣"#, r#"궣"#, r#"궣"#, r#"궣"#),
        (r#"궤"#, r#"궤"#, r#"궤"#, r#"궤"#, r#"궤"#),
        (r#"궥"#, r#"궥"#, r#"궥"#, r#"궥"#, r#"궥"#),
        (r#"궦"#, r#"궦"#, r#"궦"#, r#"궦"#, r#"궦"#),
        (r#"궧"#, r#"궧"#, r#"궧"#, r#"궧"#, r#"궧"#),
        (r#"궨"#, r#"궨"#, r#"궨"#, r#"궨"#, r#"궨"#),
        (r#"궩"#, r#"궩"#, r#"궩"#, r#"궩"#, r#"궩"#),
        (r#"궪"#, r#"궪"#, r#"궪"#, r#"궪"#, r#"궪"#),
        (r#"궫"#, r#"궫"#, r#"궫"#, r#"궫"#, r#"궫"#),
        (r#"궬"#, r#"궬"#, r#"궬"#, r#"궬"#, r#"궬"#),
        (r#"궭"#, r#"궭"#, r#"궭"#, r#"궭"#, r#"궭"#),
        (r#"궮"#, r#"궮"#, r#"궮"#, r#"궮"#, r#"궮"#),
        (r#"궯"#, r#"궯"#, r#"궯"#, r#"궯"#, r#"궯"#),
        (r#"궰"#, r#"궰"#, r#"궰"#, r#"궰"#, r#"궰"#),
        (r#"궱"#, r#"궱"#, r#"궱"#, r#"궱"#, r#"궱"#),
        (r#"궲"#, r#"궲"#, r#"궲"#, r#"궲"#, r#"궲"#),
        (r#"궳"#, r#"궳"#, r#"궳"#, r#"궳"#, r#"궳"#),
        (r#"궴"#, r#"궴"#, r#"궴"#, r#"궴"#, r#"궴"#),
        (r#"궵"#, r#"궵"#, r#"궵"#, r#"궵"#, r#"궵"#),
        (r#"궶"#, r#"궶"#, r#"궶"#, r#"궶"#, r#"궶"#),
        (r#"궷"#, r#"궷"#, r#"궷"#, r#"궷"#, r#"궷"#),
        (r#"궸"#, r#"궸"#, r#"궸"#, r#"궸"#, r#"궸"#),
        (r#"궹"#, r#"궹"#, r#"궹"#, r#"궹"#, r#"궹"#),
        (r#"궺"#, r#"궺"#, r#"궺"#, r#"궺"#, r#"궺"#),
        (r#"궻"#, r#"궻"#, r#"궻"#, r#"궻"#, r#"궻"#),
        (r#"궼"#, r#"궼"#, r#"궼"#, r#"궼"#, r#"궼"#),
        (r#"궽"#, r#"궽"#, r#"궽"#, r#"궽"#, r#"궽"#),
        (r#"궾"#, r#"궾"#, r#"궾"#, r#"궾"#, r#"궾"#),
        (r#"궿"#, r#"궿"#, r#"궿"#, r#"궿"#, r#"궿"#),
        (r#"귀"#, r#"귀"#, r#"귀"#, r#"귀"#, r#"귀"#),
        (r#"귁"#, r#"귁"#, r#"귁"#, r#"귁"#, r#"귁"#),
        (r#"귂"#, r#"귂"#, r#"귂"#, r#"귂"#, r#"귂"#),
        (r#"귃"#, r#"귃"#, r#"귃"#, r#"귃"#, r#"귃"#),
        (r#"귄"#, r#"귄"#, r#"귄"#, r#"귄"#, r#"귄"#),
        (r#"귅"#, r#"귅"#, r#"귅"#, r#"귅"#, r#"귅"#),
        (r#"귆"#, r#"귆"#, r#"귆"#, r#"귆"#, r#"귆"#),
        (r#"귇"#, r#"귇"#, r#"귇"#, r#"귇"#, r#"귇"#),
        (r#"귈"#, r#"귈"#, r#"귈"#, r#"귈"#, r#"귈"#),
        (r#"귉"#, r#"귉"#, r#"귉"#, r#"귉"#, r#"귉"#),
        (r#"귊"#, r#"귊"#, r#"귊"#, r#"귊"#, r#"귊"#),
        (r#"귋"#, r#"귋"#, r#"귋"#, r#"귋"#, r#"귋"#),
        (r#"귌"#, r#"귌"#, r#"귌"#, r#"귌"#, r#"귌"#),
        (r#"귍"#, r#"귍"#, r#"귍"#, r#"귍"#, r#"귍"#),
        (r#"귎"#, r#"귎"#, r#"귎"#, r#"귎"#, r#"귎"#),
        (r#"귏"#, r#"귏"#, r#"귏"#, r#"귏"#, r#"귏"#),
        (r#"귐"#, r#"귐"#, r#"귐"#, r#"귐"#, r#"귐"#),
        (r#"귑"#, r#"귑"#, r#"귑"#, r#"귑"#, r#"귑"#),
        (r#"귒"#, r#"귒"#, r#"귒"#, r#"귒"#, r#"귒"#),
        (r#"귓"#, r#"귓"#, r#"귓"#, r#"귓"#, r#"귓"#),
        (r#"귔"#, r#"귔"#, r#"귔"#, r#"귔"#, r#"귔"#),
        (r#"귕"#, r#"귕"#, r#"귕"#, r#"귕"#, r#"귕"#),
        (r#"귖"#, r#"귖"#, r#"귖"#, r#"귖"#, r#"귖"#),
        (r#"귗"#, r#"귗"#, r#"귗"#, r#"귗"#, r#"귗"#),
        (r#"귘"#, r#"귘"#, r#"귘"#, r#"귘"#, r#"귘"#),
        (r#"귙"#, r#"귙"#, r#"귙"#, r#"귙"#, r#"귙"#),
        (r#"귚"#, r#"귚"#, r#"귚"#, r#"귚"#, r#"귚"#),
        (r#"귛"#, r#"귛"#, r#"귛"#, r#"귛"#, r#"귛"#),
        (r#"규"#, r#"규"#, r#"규"#, r#"규"#, r#"규"#),
        (r#"귝"#, r#"귝"#, r#"귝"#, r#"귝"#, r#"귝"#),
        (r#"귞"#, r#"귞"#, r#"귞"#, r#"귞"#, r#"귞"#),
        (r#"귟"#, r#"귟"#, r#"귟"#, r#"귟"#, r#"귟"#),
        (r#"균"#, r#"균"#, r#"균"#, r#"균"#, r#"균"#),
        (r#"귡"#, r#"귡"#, r#"귡"#, r#"귡"#, r#"귡"#),
        (r#"귢"#, r#"귢"#, r#"귢"#, r#"귢"#, r#"귢"#),
        (r#"귣"#, r#"귣"#, r#"귣"#, r#"귣"#, r#"귣"#),
        (r#"귤"#, r#"귤"#, r#"귤"#, r#"귤"#, r#"귤"#),
        (r#"귥"#, r#"귥"#, r#"귥"#, r#"귥"#, r#"귥"#),
        (r#"귦"#, r#"귦"#, r#"귦"#, r#"귦"#, r#"귦"#),
        (r#"귧"#, r#"귧"#, r#"귧"#, r#"귧"#, r#"귧"#),
        (r#"귨"#, r#"귨"#, r#"귨"#, r#"귨"#, r#"귨"#),
        (r#"귩"#, r#"귩"#, r#"귩"#, r#"귩"#, r#"귩"#),
        (r#"귪"#, r#"귪"#, r#"귪"#, r#"귪"#, r#"귪"#),
        (r#"귫"#, r#"귫"#, r#"귫"#, r#"귫"#, r#"귫"#),
        (r#"귬"#, r#"귬"#, r#"귬"#, r#"귬"#, r#"귬"#),
        (r#"귭"#, r#"귭"#, r#"귭"#, r#"귭"#, r#"귭"#),
        (r#"귮"#, r#"귮"#, r#"귮"#, r#"귮"#, r#"귮"#),
        (r#"귯"#, r#"귯"#, r#"귯"#, r#"귯"#, r#"귯"#),
        (r#"귰"#, r#"귰"#, r#"귰"#, r#"귰"#, r#"귰"#),
        (r#"귱"#, r#"귱"#, r#"귱"#, r#"귱"#, r#"귱"#),
        (r#"귲"#, r#"귲"#, r#"귲"#, r#"귲"#, r#"귲"#),
        (r#"귳"#, r#"귳"#, r#"귳"#, r#"귳"#, r#"귳"#),
        (r#"귴"#, r#"귴"#, r#"귴"#, r#"귴"#, r#"귴"#),
        (r#"귵"#, r#"귵"#, r#"귵"#, r#"귵"#, r#"귵"#),
        (r#"귶"#, r#"귶"#, r#"귶"#, r#"귶"#, r#"귶"#),
        (r#"귷"#, r#"귷"#, r#"귷"#, r#"귷"#, r#"귷"#),
        (r#"그"#, r#"그"#, r#"그"#, r#"그"#, r#"그"#),
        (r#"극"#, r#"극"#, r#"극"#, r#"극"#, r#"극"#),
        (r#"귺"#, r#"귺"#, r#"귺"#, r#"귺"#, r#"귺"#),
        (r#"귻"#, r#"귻"#, r#"귻"#, r#"귻"#, r#"귻"#),
        (r#"근"#, r#"근"#, r#"근"#, r#"근"#, r#"근"#),
        (r#"귽"#, r#"귽"#, r#"귽"#, r#"귽"#, r#"귽"#),
        (r#"귾"#, r#"귾"#, r#"귾"#, r#"귾"#, r#"귾"#),
        (r#"귿"#, r#"귿"#, r#"귿"#, r#"귿"#, r#"귿"#),
        (r#"글"#, r#"글"#, r#"글"#, r#"글"#, r#"글"#),
        (r#"긁"#, r#"긁"#, r#"긁"#, r#"긁"#, r#"긁"#),
        (r#"긂"#, r#"긂"#, r#"긂"#, r#"긂"#, r#"긂"#),
        (r#"긃"#, r#"긃"#, r#"긃"#, r#"긃"#, r#"긃"#),
        (r#"긄"#, r#"긄"#, r#"긄"#, r#"긄"#, r#"긄"#),
        (r#"긅"#, r#"긅"#, r#"긅"#, r#"긅"#, r#"긅"#),
        (r#"긆"#, r#"긆"#, r#"긆"#, r#"긆"#, r#"긆"#),
        (r#"긇"#, r#"긇"#, r#"긇"#, r#"긇"#, r#"긇"#),
        (r#"금"#, r#"금"#, r#"금"#, r#"금"#, r#"금"#),
        (r#"급"#, r#"급"#, r#"급"#, r#"급"#, r#"급"#),
        (r#"긊"#, r#"긊"#, r#"긊"#, r#"긊"#, r#"긊"#),
        (r#"긋"#, r#"긋"#, r#"긋"#, r#"긋"#, r#"긋"#),
        (r#"긌"#, r#"긌"#, r#"긌"#, r#"긌"#, r#"긌"#),
        (r#"긍"#, r#"긍"#, r#"긍"#, r#"긍"#, r#"긍"#),
        (r#"긎"#, r#"긎"#, r#"긎"#, r#"긎"#, r#"긎"#),
        (r#"긏"#, r#"긏"#, r#"긏"#, r#"긏"#, r#"긏"#),
        (r#"긐"#, r#"긐"#, r#"긐"#, r#"긐"#, r#"긐"#),
        (r#"긑"#, r#"긑"#, r#"긑"#, r#"긑"#, r#"긑"#),
        (r#"긒"#, r#"긒"#, r#"긒"#, r#"긒"#, r#"긒"#),
        (r#"긓"#, r#"긓"#, r#"긓"#, r#"긓"#, r#"긓"#),
        (r#"긔"#, r#"긔"#, r#"긔"#, r#"긔"#, r#"긔"#),
        (r#"긕"#, r#"긕"#, r#"긕"#, r#"긕"#, r#"긕"#),
        (r#"긖"#, r#"긖"#, r#"긖"#, r#"긖"#, r#"긖"#),
        (r#"긗"#, r#"긗"#, r#"긗"#, r#"긗"#, r#"긗"#),
        (r#"긘"#, r#"긘"#, r#"긘"#, r#"긘"#, r#"긘"#),
        (r#"긙"#, r#"긙"#, r#"긙"#, r#"긙"#, r#"긙"#),
        (r#"긚"#, r#"긚"#, r#"긚"#, r#"긚"#, r#"긚"#),
        (r#"긛"#, r#"긛"#, r#"긛"#, r#"긛"#, r#"긛"#),
        (r#"긜"#, r#"긜"#, r#"긜"#, r#"긜"#, r#"긜"#),
        (r#"긝"#, r#"긝"#, r#"긝"#, r#"긝"#, r#"긝"#),
        (r#"긞"#, r#"긞"#, r#"긞"#, r#"긞"#, r#"긞"#),
        (r#"긟"#, r#"긟"#, r#"긟"#, r#"긟"#, r#"긟"#),
        (r#"긠"#, r#"긠"#, r#"긠"#, r#"긠"#, r#"긠"#),
        (r#"긡"#, r#"긡"#, r#"긡"#, r#"긡"#, r#"긡"#),
        (r#"긢"#, r#"긢"#, r#"긢"#, r#"긢"#, r#"긢"#),
        (r#"긣"#, r#"긣"#, r#"긣"#, r#"긣"#, r#"긣"#),
        (r#"긤"#, r#"긤"#, r#"긤"#, r#"긤"#, r#"긤"#),
        (r#"긥"#, r#"긥"#, r#"긥"#, r#"긥"#, r#"긥"#),
        (r#"긦"#, r#"긦"#, r#"긦"#, r#"긦"#, r#"긦"#),
        (r#"긧"#, r#"긧"#, r#"긧"#, r#"긧"#, r#"긧"#),
        (r#"긨"#, r#"긨"#, r#"긨"#, r#"긨"#, r#"긨"#),
        (r#"긩"#, r#"긩"#, r#"긩"#, r#"긩"#, r#"긩"#),
        (r#"긪"#, r#"긪"#, r#"긪"#, r#"긪"#, r#"긪"#),
        (r#"긫"#, r#"긫"#, r#"긫"#, r#"긫"#, r#"긫"#),
        (r#"긬"#, r#"긬"#, r#"긬"#, r#"긬"#, r#"긬"#),
        (r#"긭"#, r#"긭"#, r#"긭"#, r#"긭"#, r#"긭"#),
        (r#"긮"#, r#"긮"#, r#"긮"#, r#"긮"#, r#"긮"#),
        (r#"긯"#, r#"긯"#, r#"긯"#, r#"긯"#, r#"긯"#),
        (r#"기"#, r#"기"#, r#"기"#, r#"기"#, r#"기"#),
        (r#"긱"#, r#"긱"#, r#"긱"#, r#"긱"#, r#"긱"#),
        (r#"긲"#, r#"긲"#, r#"긲"#, r#"긲"#, r#"긲"#),
        (r#"긳"#, r#"긳"#, r#"긳"#, r#"긳"#, r#"긳"#),
        (r#"긴"#, r#"긴"#, r#"긴"#, r#"긴"#, r#"긴"#),
        (r#"긵"#, r#"긵"#, r#"긵"#, r#"긵"#, r#"긵"#),
        (r#"긶"#, r#"긶"#, r#"긶"#, r#"긶"#, r#"긶"#),
        (r#"긷"#, r#"긷"#, r#"긷"#, r#"긷"#, r#"긷"#),
        (r#"길"#, r#"길"#, r#"길"#, r#"길"#, r#"길"#),
        (r#"긹"#, r#"긹"#, r#"긹"#, r#"긹"#, r#"긹"#),
        (r#"긺"#, r#"긺"#, r#"긺"#, r#"긺"#, r#"긺"#),
        (r#"긻"#, r#"긻"#, r#"긻"#, r#"긻"#, r#"긻"#),
        (r#"긼"#, r#"긼"#, r#"긼"#, r#"긼"#, r#"긼"#),
        (r#"긽"#, r#"긽"#, r#"긽"#, r#"긽"#, r#"긽"#),
        (r#"긾"#, r#"긾"#, r#"긾"#, r#"긾"#, r#"긾"#),
        (r#"긿"#, r#"긿"#, r#"긿"#, r#"긿"#, r#"긿"#),
        (r#"김"#, r#"김"#, r#"김"#, r#"김"#, r#"김"#),
        (r#"깁"#, r#"깁"#, r#"깁"#, r#"깁"#, r#"깁"#),
        (r#"깂"#, r#"깂"#, r#"깂"#, r#"깂"#, r#"깂"#),
        (r#"깃"#, r#"깃"#, r#"깃"#, r#"깃"#, r#"깃"#),
        (r#"깄"#, r#"깄"#, r#"깄"#, r#"깄"#, r#"깄"#),
        (r#"깅"#, r#"깅"#, r#"깅"#, r#"깅"#, r#"깅"#),
        (r#"깆"#, r#"깆"#, r#"깆"#, r#"깆"#, r#"깆"#),
        (r#"깇"#, r#"깇"#, r#"깇"#, r#"깇"#, r#"깇"#),
        (r#"깈"#, r#"깈"#, r#"깈"#, r#"깈"#, r#"깈"#),
        (r#"깉"#, r#"깉"#, r#"깉"#, r#"깉"#, r#"깉"#),
        (r#"깊"#, r#"깊"#, r#"깊"#, r#"깊"#, r#"깊"#),
        (r#"깋"#, r#"깋"#, r#"깋"#, r#"깋"#, r#"깋"#),
        (r#"까"#, r#"까"#, r#"까"#, r#"까"#, r#"까"#),
        (r#"깍"#, r#"깍"#, r#"깍"#, r#"깍"#, r#"깍"#),
        (r#"깎"#, r#"깎"#, r#"깎"#, r#"깎"#, r#"깎"#),
        (r#"깏"#, r#"깏"#, r#"깏"#, r#"깏"#, r#"깏"#),
        (r#"깐"#, r#"깐"#, r#"깐"#, r#"깐"#, r#"깐"#),
        (r#"깑"#, r#"깑"#, r#"깑"#, r#"깑"#, r#"깑"#),
        (r#"깒"#, r#"깒"#, r#"깒"#, r#"깒"#, r#"깒"#),
        (r#"깓"#, r#"깓"#, r#"깓"#, r#"깓"#, r#"깓"#),
        (r#"깔"#, r#"깔"#, r#"깔"#, r#"깔"#, r#"깔"#),
        (r#"깕"#, r#"깕"#, r#"깕"#, r#"깕"#, r#"깕"#),
        (r#"깖"#, r#"깖"#, r#"깖"#, r#"깖"#, r#"깖"#),
        (r#"깗"#, r#"깗"#, r#"깗"#, r#"깗"#, r#"깗"#),
        (r#"깘"#, r#"깘"#, r#"깘"#, r#"깘"#, r#"깘"#),
        (r#"깙"#, r#"깙"#, r#"깙"#, r#"깙"#, r#"깙"#),
        (r#"깚"#, r#"깚"#, r#"깚"#, r#"깚"#, r#"깚"#),
        (r#"깛"#, r#"깛"#, r#"깛"#, r#"깛"#, r#"깛"#),
        (r#"깜"#, r#"깜"#, r#"깜"#, r#"깜"#, r#"깜"#),
        (r#"깝"#, r#"깝"#, r#"깝"#, r#"깝"#, r#"깝"#),
        (r#"깞"#, r#"깞"#, r#"깞"#, r#"깞"#, r#"깞"#),
        (r#"깟"#, r#"깟"#, r#"깟"#, r#"깟"#, r#"깟"#),
        (r#"깠"#, r#"깠"#, r#"깠"#, r#"깠"#, r#"깠"#),
        (r#"깡"#, r#"깡"#, r#"깡"#, r#"깡"#, r#"깡"#),
        (r#"깢"#, r#"깢"#, r#"깢"#, r#"깢"#, r#"깢"#),
        (r#"깣"#, r#"깣"#, r#"깣"#, r#"깣"#, r#"깣"#),
        (r#"깤"#, r#"깤"#, r#"깤"#, r#"깤"#, r#"깤"#),
        (r#"깥"#, r#"깥"#, r#"깥"#, r#"깥"#, r#"깥"#),
        (r#"깦"#, r#"깦"#, r#"깦"#, r#"깦"#, r#"깦"#),
        (r#"깧"#, r#"깧"#, r#"깧"#, r#"깧"#, r#"깧"#),
        (r#"깨"#, r#"깨"#, r#"깨"#, r#"깨"#, r#"깨"#),
        (r#"깩"#, r#"깩"#, r#"깩"#, r#"깩"#, r#"깩"#),
        (r#"깪"#, r#"깪"#, r#"깪"#, r#"깪"#, r#"깪"#),
        (r#"깫"#, r#"깫"#, r#"깫"#, r#"깫"#, r#"깫"#),
        (r#"깬"#, r#"깬"#, r#"깬"#, r#"깬"#, r#"깬"#),
        (r#"깭"#, r#"깭"#, r#"깭"#, r#"깭"#, r#"깭"#),
        (r#"깮"#, r#"깮"#, r#"깮"#, r#"깮"#, r#"깮"#),
        (r#"깯"#, r#"깯"#, r#"깯"#, r#"깯"#, r#"깯"#),
        (r#"깰"#, r#"깰"#, r#"깰"#, r#"깰"#, r#"깰"#),
        (r#"깱"#, r#"깱"#, r#"깱"#, r#"깱"#, r#"깱"#),
        (r#"깲"#, r#"깲"#, r#"깲"#, r#"깲"#, r#"깲"#),
        (r#"깳"#, r#"깳"#, r#"깳"#, r#"깳"#, r#"깳"#),
        (r#"깴"#, r#"깴"#, r#"깴"#, r#"깴"#, r#"깴"#),
        (r#"깵"#, r#"깵"#, r#"깵"#, r#"깵"#, r#"깵"#),
        (r#"깶"#, r#"깶"#, r#"깶"#, r#"깶"#, r#"깶"#),
        (r#"깷"#, r#"깷"#, r#"깷"#, r#"깷"#, r#"깷"#),
        (r#"깸"#, r#"깸"#, r#"깸"#, r#"깸"#, r#"깸"#),
        (r#"깹"#, r#"깹"#, r#"깹"#, r#"깹"#, r#"깹"#),
        (r#"깺"#, r#"깺"#, r#"깺"#, r#"깺"#, r#"깺"#),
        (r#"깻"#, r#"깻"#, r#"깻"#, r#"깻"#, r#"깻"#),
        (r#"깼"#, r#"깼"#, r#"깼"#, r#"깼"#, r#"깼"#),
        (r#"깽"#, r#"깽"#, r#"깽"#, r#"깽"#, r#"깽"#),
        (r#"깾"#, r#"깾"#, r#"깾"#, r#"깾"#, r#"깾"#),
        (r#"깿"#, r#"깿"#, r#"깿"#, r#"깿"#, r#"깿"#),
        (r#"꺀"#, r#"꺀"#, r#"꺀"#, r#"꺀"#, r#"꺀"#),
        (r#"꺁"#, r#"꺁"#, r#"꺁"#, r#"꺁"#, r#"꺁"#),
        (r#"꺂"#, r#"꺂"#, r#"꺂"#, r#"꺂"#, r#"꺂"#),
        (r#"꺃"#, r#"꺃"#, r#"꺃"#, r#"꺃"#, r#"꺃"#),
        (r#"꺄"#, r#"꺄"#, r#"꺄"#, r#"꺄"#, r#"꺄"#),
        (r#"꺅"#, r#"꺅"#, r#"꺅"#, r#"꺅"#, r#"꺅"#),
        (r#"꺆"#, r#"꺆"#, r#"꺆"#, r#"꺆"#, r#"꺆"#),
        (r#"꺇"#, r#"꺇"#, r#"꺇"#, r#"꺇"#, r#"꺇"#),
        (r#"꺈"#, r#"꺈"#, r#"꺈"#, r#"꺈"#, r#"꺈"#),
        (r#"꺉"#, r#"꺉"#, r#"꺉"#, r#"꺉"#, r#"꺉"#),
        (r#"꺊"#, r#"꺊"#, r#"꺊"#, r#"꺊"#, r#"꺊"#),
        (r#"꺋"#, r#"꺋"#, r#"꺋"#, r#"꺋"#, r#"꺋"#),
        (r#"꺌"#, r#"꺌"#, r#"꺌"#, r#"꺌"#, r#"꺌"#),
        (r#"꺍"#, r#"꺍"#, r#"꺍"#, r#"꺍"#, r#"꺍"#),
        (r#"꺎"#, r#"꺎"#, r#"꺎"#, r#"꺎"#, r#"꺎"#),
        (r#"꺏"#, r#"꺏"#, r#"꺏"#, r#"꺏"#, r#"꺏"#),
        (r#"꺐"#, r#"꺐"#, r#"꺐"#, r#"꺐"#, r#"꺐"#),
        (r#"꺑"#, r#"꺑"#, r#"꺑"#, r#"꺑"#, r#"꺑"#),
        (r#"꺒"#, r#"꺒"#, r#"꺒"#, r#"꺒"#, r#"꺒"#),
        (r#"꺓"#, r#"꺓"#, r#"꺓"#, r#"꺓"#, r#"꺓"#),
        (r#"꺔"#, r#"꺔"#, r#"꺔"#, r#"꺔"#, r#"꺔"#),
        (r#"꺕"#, r#"꺕"#, r#"꺕"#, r#"꺕"#, r#"꺕"#),
        (r#"꺖"#, r#"꺖"#, r#"꺖"#, r#"꺖"#, r#"꺖"#),
        (r#"꺗"#, r#"꺗"#, r#"꺗"#, r#"꺗"#, r#"꺗"#),
        (r#"꺘"#, r#"꺘"#, r#"꺘"#, r#"꺘"#, r#"꺘"#),
        (r#"꺙"#, r#"꺙"#, r#"꺙"#, r#"꺙"#, r#"꺙"#),
        (r#"꺚"#, r#"꺚"#, r#"꺚"#, r#"꺚"#, r#"꺚"#),
        (r#"꺛"#, r#"꺛"#, r#"꺛"#, r#"꺛"#, r#"꺛"#),
        (r#"꺜"#, r#"꺜"#, r#"꺜"#, r#"꺜"#, r#"꺜"#),
        (r#"꺝"#, r#"꺝"#, r#"꺝"#, r#"꺝"#, r#"꺝"#),
        (r#"꺞"#, r#"꺞"#, r#"꺞"#, r#"꺞"#, r#"꺞"#),
        (r#"꺟"#, r#"꺟"#, r#"꺟"#, r#"꺟"#, r#"꺟"#),
        (r#"꺠"#, r#"꺠"#, r#"꺠"#, r#"꺠"#, r#"꺠"#),
        (r#"꺡"#, r#"꺡"#, r#"꺡"#, r#"꺡"#, r#"꺡"#),
        (r#"꺢"#, r#"꺢"#, r#"꺢"#, r#"꺢"#, r#"꺢"#),
        (r#"꺣"#, r#"꺣"#, r#"꺣"#, r#"꺣"#, r#"꺣"#),
        (r#"꺤"#, r#"꺤"#, r#"꺤"#, r#"꺤"#, r#"꺤"#),
        (r#"꺥"#, r#"꺥"#, r#"꺥"#, r#"꺥"#, r#"꺥"#),
        (r#"꺦"#, r#"꺦"#, r#"꺦"#, r#"꺦"#, r#"꺦"#),
        (r#"꺧"#, r#"꺧"#, r#"꺧"#, r#"꺧"#, r#"꺧"#),
        (r#"꺨"#, r#"꺨"#, r#"꺨"#, r#"꺨"#, r#"꺨"#),
        (r#"꺩"#, r#"꺩"#, r#"꺩"#, r#"꺩"#, r#"꺩"#),
        (r#"꺪"#, r#"꺪"#, r#"꺪"#, r#"꺪"#, r#"꺪"#),
        (r#"꺫"#, r#"꺫"#, r#"꺫"#, r#"꺫"#, r#"꺫"#),
        (r#"꺬"#, r#"꺬"#, r#"꺬"#, r#"꺬"#, r#"꺬"#),
        (r#"꺭"#, r#"꺭"#, r#"꺭"#, r#"꺭"#, r#"꺭"#),
        (r#"꺮"#, r#"꺮"#, r#"꺮"#, r#"꺮"#, r#"꺮"#),
        (r#"꺯"#, r#"꺯"#, r#"꺯"#, r#"꺯"#, r#"꺯"#),
        (r#"꺰"#, r#"꺰"#, r#"꺰"#, r#"꺰"#, r#"꺰"#),
        (r#"꺱"#, r#"꺱"#, r#"꺱"#, r#"꺱"#, r#"꺱"#),
        (r#"꺲"#, r#"꺲"#, r#"꺲"#, r#"꺲"#, r#"꺲"#),
        (r#"꺳"#, r#"꺳"#, r#"꺳"#, r#"꺳"#, r#"꺳"#),
        (r#"꺴"#, r#"꺴"#, r#"꺴"#, r#"꺴"#, r#"꺴"#),
        (r#"꺵"#, r#"꺵"#, r#"꺵"#, r#"꺵"#, r#"꺵"#),
        (r#"꺶"#, r#"꺶"#, r#"꺶"#, r#"꺶"#, r#"꺶"#),
        (r#"꺷"#, r#"꺷"#, r#"꺷"#, r#"꺷"#, r#"꺷"#),
        (r#"꺸"#, r#"꺸"#, r#"꺸"#, r#"꺸"#, r#"꺸"#),
        (r#"꺹"#, r#"꺹"#, r#"꺹"#, r#"꺹"#, r#"꺹"#),
        (r#"꺺"#, r#"꺺"#, r#"꺺"#, r#"꺺"#, r#"꺺"#),
        (r#"꺻"#, r#"꺻"#, r#"꺻"#, r#"꺻"#, r#"꺻"#),
        (r#"꺼"#, r#"꺼"#, r#"꺼"#, r#"꺼"#, r#"꺼"#),
        (r#"꺽"#, r#"꺽"#, r#"꺽"#, r#"꺽"#, r#"꺽"#),
        (r#"꺾"#, r#"꺾"#, r#"꺾"#, r#"꺾"#, r#"꺾"#),
        (r#"꺿"#, r#"꺿"#, r#"꺿"#, r#"꺿"#, r#"꺿"#),
        (r#"껀"#, r#"껀"#, r#"껀"#, r#"껀"#, r#"껀"#),
        (r#"껁"#, r#"껁"#, r#"껁"#, r#"껁"#, r#"껁"#),
        (r#"껂"#, r#"껂"#, r#"껂"#, r#"껂"#, r#"껂"#),
        (r#"껃"#, r#"껃"#, r#"껃"#, r#"껃"#, r#"껃"#),
        (r#"껄"#, r#"껄"#, r#"껄"#, r#"껄"#, r#"껄"#),
        (r#"껅"#, r#"껅"#, r#"껅"#, r#"껅"#, r#"껅"#),
        (r#"껆"#, r#"껆"#, r#"껆"#, r#"껆"#, r#"껆"#),
        (r#"껇"#, r#"껇"#, r#"껇"#, r#"껇"#, r#"껇"#),
        (r#"껈"#, r#"껈"#, r#"껈"#, r#"껈"#, r#"껈"#),
        (r#"껉"#, r#"껉"#, r#"껉"#, r#"껉"#, r#"껉"#),
        (r#"껊"#, r#"껊"#, r#"껊"#, r#"껊"#, r#"껊"#),
        (r#"껋"#, r#"껋"#, r#"껋"#, r#"껋"#, r#"껋"#),
        (r#"껌"#, r#"껌"#, r#"껌"#, r#"껌"#, r#"껌"#),
        (r#"껍"#, r#"껍"#, r#"껍"#, r#"껍"#, r#"껍"#),
        (r#"껎"#, r#"껎"#, r#"껎"#, r#"껎"#, r#"껎"#),
        (r#"껏"#, r#"껏"#, r#"껏"#, r#"껏"#, r#"껏"#),
        (r#"껐"#, r#"껐"#, r#"껐"#, r#"껐"#, r#"껐"#),
        (r#"껑"#, r#"껑"#, r#"껑"#, r#"껑"#, r#"껑"#),
        (r#"껒"#, r#"껒"#, r#"껒"#, r#"껒"#, r#"껒"#),
        (r#"껓"#, r#"껓"#, r#"껓"#, r#"껓"#, r#"껓"#),
        (r#"껔"#, r#"껔"#, r#"껔"#, r#"껔"#, r#"껔"#),
        (r#"껕"#, r#"껕"#, r#"껕"#, r#"껕"#, r#"껕"#),
        (r#"껖"#, r#"껖"#, r#"껖"#, r#"껖"#, r#"껖"#),
        (r#"껗"#, r#"껗"#, r#"껗"#, r#"껗"#, r#"껗"#),
        (r#"께"#, r#"께"#, r#"께"#, r#"께"#, r#"께"#),
        (r#"껙"#, r#"껙"#, r#"껙"#, r#"껙"#, r#"껙"#),
        (r#"껚"#, r#"껚"#, r#"껚"#, r#"껚"#, r#"껚"#),
        (r#"껛"#, r#"껛"#, r#"껛"#, r#"껛"#, r#"껛"#),
        (r#"껜"#, r#"껜"#, r#"껜"#, r#"껜"#, r#"껜"#),
        (r#"껝"#, r#"껝"#, r#"껝"#, r#"껝"#, r#"껝"#),
        (r#"껞"#, r#"껞"#, r#"껞"#, r#"껞"#, r#"껞"#),
        (r#"껟"#, r#"껟"#, r#"껟"#, r#"껟"#, r#"껟"#),
        (r#"껠"#, r#"껠"#, r#"껠"#, r#"껠"#, r#"껠"#),
        (r#"껡"#, r#"껡"#, r#"껡"#, r#"껡"#, r#"껡"#),
        (r#"껢"#, r#"껢"#, r#"껢"#, r#"껢"#, r#"껢"#),
        (r#"껣"#, r#"껣"#, r#"껣"#, r#"껣"#, r#"껣"#),
        (r#"껤"#, r#"껤"#, r#"껤"#, r#"껤"#, r#"껤"#),
        (r#"껥"#, r#"껥"#, r#"껥"#, r#"껥"#, r#"껥"#),
        (r#"껦"#, r#"껦"#, r#"껦"#, r#"껦"#, r#"껦"#),
        (r#"껧"#, r#"껧"#, r#"껧"#, r#"껧"#, r#"껧"#),
        (r#"껨"#, r#"껨"#, r#"껨"#, r#"껨"#, r#"껨"#),
        (r#"껩"#, r#"껩"#, r#"껩"#, r#"껩"#, r#"껩"#),
        (r#"껪"#, r#"껪"#, r#"껪"#, r#"껪"#, r#"껪"#),
        (r#"껫"#, r#"껫"#, r#"껫"#, r#"껫"#, r#"껫"#),
        (r#"껬"#, r#"껬"#, r#"껬"#, r#"껬"#, r#"껬"#),
        (r#"껭"#, r#"껭"#, r#"껭"#, r#"껭"#, r#"껭"#),
        (r#"껮"#, r#"껮"#, r#"껮"#, r#"껮"#, r#"껮"#),
        (r#"껯"#, r#"껯"#, r#"껯"#, r#"껯"#, r#"껯"#),
        (r#"껰"#, r#"껰"#, r#"껰"#, r#"껰"#, r#"껰"#),
        (r#"껱"#, r#"껱"#, r#"껱"#, r#"껱"#, r#"껱"#),
        (r#"껲"#, r#"껲"#, r#"껲"#, r#"껲"#, r#"껲"#),
        (r#"껳"#, r#"껳"#, r#"껳"#, r#"껳"#, r#"껳"#),
        (r#"껴"#, r#"껴"#, r#"껴"#, r#"껴"#, r#"껴"#),
        (r#"껵"#, r#"껵"#, r#"껵"#, r#"껵"#, r#"껵"#),
        (r#"껶"#, r#"껶"#, r#"껶"#, r#"껶"#, r#"껶"#),
        (r#"껷"#, r#"껷"#, r#"껷"#, r#"껷"#, r#"껷"#),
        (r#"껸"#, r#"껸"#, r#"껸"#, r#"껸"#, r#"껸"#),
        (r#"껹"#, r#"껹"#, r#"껹"#, r#"껹"#, r#"껹"#),
        (r#"껺"#, r#"껺"#, r#"껺"#, r#"껺"#, r#"껺"#),
        (r#"껻"#, r#"껻"#, r#"껻"#, r#"껻"#, r#"껻"#),
        (r#"껼"#, r#"껼"#, r#"껼"#, r#"껼"#, r#"껼"#),
        (r#"껽"#, r#"껽"#, r#"껽"#, r#"껽"#, r#"껽"#),
        (r#"껾"#, r#"껾"#, r#"껾"#, r#"껾"#, r#"껾"#),
        (r#"껿"#, r#"껿"#, r#"껿"#, r#"껿"#, r#"껿"#),
        (r#"꼀"#, r#"꼀"#, r#"꼀"#, r#"꼀"#, r#"꼀"#),
        (r#"꼁"#, r#"꼁"#, r#"꼁"#, r#"꼁"#, r#"꼁"#),
        (r#"꼂"#, r#"꼂"#, r#"꼂"#, r#"꼂"#, r#"꼂"#),
        (r#"꼃"#, r#"꼃"#, r#"꼃"#, r#"꼃"#, r#"꼃"#),
        (r#"꼄"#, r#"꼄"#, r#"꼄"#, r#"꼄"#, r#"꼄"#),
        (r#"꼅"#, r#"꼅"#, r#"꼅"#, r#"꼅"#, r#"꼅"#),
        (r#"꼆"#, r#"꼆"#, r#"꼆"#, r#"꼆"#, r#"꼆"#),
        (r#"꼇"#, r#"꼇"#, r#"꼇"#, r#"꼇"#, r#"꼇"#),
        (r#"꼈"#, r#"꼈"#, r#"꼈"#, r#"꼈"#, r#"꼈"#),
        (r#"꼉"#, r#"꼉"#, r#"꼉"#, r#"꼉"#, r#"꼉"#),
        (r#"꼊"#, r#"꼊"#, r#"꼊"#, r#"꼊"#, r#"꼊"#),
        (r#"꼋"#, r#"꼋"#, r#"꼋"#, r#"꼋"#, r#"꼋"#),
        (r#"꼌"#, r#"꼌"#, r#"꼌"#, r#"꼌"#, r#"꼌"#),
        (r#"꼍"#, r#"꼍"#, r#"꼍"#, r#"꼍"#, r#"꼍"#),
        (r#"꼎"#, r#"꼎"#, r#"꼎"#, r#"꼎"#, r#"꼎"#),
        (r#"꼏"#, r#"꼏"#, r#"꼏"#, r#"꼏"#, r#"꼏"#),
        (r#"꼐"#, r#"꼐"#, r#"꼐"#, r#"꼐"#, r#"꼐"#),
        (r#"꼑"#, r#"꼑"#, r#"꼑"#, r#"꼑"#, r#"꼑"#),
        (r#"꼒"#, r#"꼒"#, r#"꼒"#, r#"꼒"#, r#"꼒"#),
        (r#"꼓"#, r#"꼓"#, r#"꼓"#, r#"꼓"#, r#"꼓"#),
        (r#"꼔"#, r#"꼔"#, r#"꼔"#, r#"꼔"#, r#"꼔"#),
        (r#"꼕"#, r#"꼕"#, r#"꼕"#, r#"꼕"#, r#"꼕"#),
        (r#"꼖"#, r#"꼖"#, r#"꼖"#, r#"꼖"#, r#"꼖"#),
        (r#"꼗"#, r#"꼗"#, r#"꼗"#, r#"꼗"#, r#"꼗"#),
        (r#"꼘"#, r#"꼘"#, r#"꼘"#, r#"꼘"#, r#"꼘"#),
        (r#"꼙"#, r#"꼙"#, r#"꼙"#, r#"꼙"#, r#"꼙"#),
        (r#"꼚"#, r#"꼚"#, r#"꼚"#, r#"꼚"#, r#"꼚"#),
        (r#"꼛"#, r#"꼛"#, r#"꼛"#, r#"꼛"#, r#"꼛"#),
        (r#"꼜"#, r#"꼜"#, r#"꼜"#, r#"꼜"#, r#"꼜"#),
        (r#"꼝"#, r#"꼝"#, r#"꼝"#, r#"꼝"#, r#"꼝"#),
        (r#"꼞"#, r#"꼞"#, r#"꼞"#, r#"꼞"#, r#"꼞"#),
        (r#"꼟"#, r#"꼟"#, r#"꼟"#, r#"꼟"#, r#"꼟"#),
        (r#"꼠"#, r#"꼠"#, r#"꼠"#, r#"꼠"#, r#"꼠"#),
        (r#"꼡"#, r#"꼡"#, r#"꼡"#, r#"꼡"#, r#"꼡"#),
        (r#"꼢"#, r#"꼢"#, r#"꼢"#, r#"꼢"#, r#"꼢"#),
        (r#"꼣"#, r#"꼣"#, r#"꼣"#, r#"꼣"#, r#"꼣"#),
        (r#"꼤"#, r#"꼤"#, r#"꼤"#, r#"꼤"#, r#"꼤"#),
        (r#"꼥"#, r#"꼥"#, r#"꼥"#, r#"꼥"#, r#"꼥"#),
        (r#"꼦"#, r#"꼦"#, r#"꼦"#, r#"꼦"#, r#"꼦"#),
        (r#"꼧"#, r#"꼧"#, r#"꼧"#, r#"꼧"#, r#"꼧"#),
        (r#"꼨"#, r#"꼨"#, r#"꼨"#, r#"꼨"#, r#"꼨"#),
        (r#"꼩"#, r#"꼩"#, r#"꼩"#, r#"꼩"#, r#"꼩"#),
        (r#"꼪"#, r#"꼪"#, r#"꼪"#, r#"꼪"#, r#"꼪"#),
        (r#"꼫"#, r#"꼫"#, r#"꼫"#, r#"꼫"#, r#"꼫"#),
        (r#"꼬"#, r#"꼬"#, r#"꼬"#, r#"꼬"#, r#"꼬"#),
        (r#"꼭"#, r#"꼭"#, r#"꼭"#, r#"꼭"#, r#"꼭"#),
        (r#"꼮"#, r#"꼮"#, r#"꼮"#, r#"꼮"#, r#"꼮"#),
        (r#"꼯"#, r#"꼯"#, r#"꼯"#, r#"꼯"#, r#"꼯"#),
        (r#"꼰"#, r#"꼰"#, r#"꼰"#, r#"꼰"#, r#"꼰"#),
        (r#"꼱"#, r#"꼱"#, r#"꼱"#, r#"꼱"#, r#"꼱"#),
        (r#"꼲"#, r#"꼲"#, r#"꼲"#, r#"꼲"#, r#"꼲"#),
        (r#"꼳"#, r#"꼳"#, r#"꼳"#, r#"꼳"#, r#"꼳"#),
        (r#"꼴"#, r#"꼴"#, r#"꼴"#, r#"꼴"#, r#"꼴"#),
        (r#"꼵"#, r#"꼵"#, r#"꼵"#, r#"꼵"#, r#"꼵"#),
        (r#"꼶"#, r#"꼶"#, r#"꼶"#, r#"꼶"#, r#"꼶"#),
        (r#"꼷"#, r#"꼷"#, r#"꼷"#, r#"꼷"#, r#"꼷"#),
        (r#"꼸"#, r#"꼸"#, r#"꼸"#, r#"꼸"#, r#"꼸"#),
        (r#"꼹"#, r#"꼹"#, r#"꼹"#, r#"꼹"#, r#"꼹"#),
        (r#"꼺"#, r#"꼺"#, r#"꼺"#, r#"꼺"#, r#"꼺"#),
        (r#"꼻"#, r#"꼻"#, r#"꼻"#, r#"꼻"#, r#"꼻"#),
        (r#"꼼"#, r#"꼼"#, r#"꼼"#, r#"꼼"#, r#"꼼"#),
        (r#"꼽"#, r#"꼽"#, r#"꼽"#, r#"꼽"#, r#"꼽"#),
        (r#"꼾"#, r#"꼾"#, r#"꼾"#, r#"꼾"#, r#"꼾"#),
        (r#"꼿"#, r#"꼿"#, r#"꼿"#, r#"꼿"#, r#"꼿"#),
        (r#"꽀"#, r#"꽀"#, r#"꽀"#, r#"꽀"#, r#"꽀"#),
        (r#"꽁"#, r#"꽁"#, r#"꽁"#, r#"꽁"#, r#"꽁"#),
        (r#"꽂"#, r#"꽂"#, r#"꽂"#, r#"꽂"#, r#"꽂"#),
        (r#"꽃"#, r#"꽃"#, r#"꽃"#, r#"꽃"#, r#"꽃"#),
        (r#"꽄"#, r#"꽄"#, r#"꽄"#, r#"꽄"#, r#"꽄"#),
        (r#"꽅"#, r#"꽅"#, r#"꽅"#, r#"꽅"#, r#"꽅"#),
        (r#"꽆"#, r#"꽆"#, r#"꽆"#, r#"꽆"#, r#"꽆"#),
        (r#"꽇"#, r#"꽇"#, r#"꽇"#, r#"꽇"#, r#"꽇"#),
        (r#"꽈"#, r#"꽈"#, r#"꽈"#, r#"꽈"#, r#"꽈"#),
        (r#"꽉"#, r#"꽉"#, r#"꽉"#, r#"꽉"#, r#"꽉"#),
        (r#"꽊"#, r#"꽊"#, r#"꽊"#, r#"꽊"#, r#"꽊"#),
        (r#"꽋"#, r#"꽋"#, r#"꽋"#, r#"꽋"#, r#"꽋"#),
        (r#"꽌"#, r#"꽌"#, r#"꽌"#, r#"꽌"#, r#"꽌"#),
        (r#"꽍"#, r#"꽍"#, r#"꽍"#, r#"꽍"#, r#"꽍"#),
        (r#"꽎"#, r#"꽎"#, r#"꽎"#, r#"꽎"#, r#"꽎"#),
        (r#"꽏"#, r#"꽏"#, r#"꽏"#, r#"꽏"#, r#"꽏"#),
        (r#"꽐"#, r#"꽐"#, r#"꽐"#, r#"꽐"#, r#"꽐"#),
        (r#"꽑"#, r#"꽑"#, r#"꽑"#, r#"꽑"#, r#"꽑"#),
        (r#"꽒"#, r#"꽒"#, r#"꽒"#, r#"꽒"#, r#"꽒"#),
        (r#"꽓"#, r#"꽓"#, r#"꽓"#, r#"꽓"#, r#"꽓"#),
        (r#"꽔"#, r#"꽔"#, r#"꽔"#, r#"꽔"#, r#"꽔"#),
        (r#"꽕"#, r#"꽕"#, r#"꽕"#, r#"꽕"#, r#"꽕"#),
        (r#"꽖"#, r#"꽖"#, r#"꽖"#, r#"꽖"#, r#"꽖"#),
        (r#"꽗"#, r#"꽗"#, r#"꽗"#, r#"꽗"#, r#"꽗"#),
        (r#"꽘"#, r#"꽘"#, r#"꽘"#, r#"꽘"#, r#"꽘"#),
        (r#"꽙"#, r#"꽙"#, r#"꽙"#, r#"꽙"#, r#"꽙"#),
        (r#"꽚"#, r#"꽚"#, r#"꽚"#, r#"꽚"#, r#"꽚"#),
        (r#"꽛"#, r#"꽛"#, r#"꽛"#, r#"꽛"#, r#"꽛"#),
        (r#"꽜"#, r#"꽜"#, r#"꽜"#, r#"꽜"#, r#"꽜"#),
        (r#"꽝"#, r#"꽝"#, r#"꽝"#, r#"꽝"#, r#"꽝"#),
        (r#"꽞"#, r#"꽞"#, r#"꽞"#, r#"꽞"#, r#"꽞"#),
        (r#"꽟"#, r#"꽟"#, r#"꽟"#, r#"꽟"#, r#"꽟"#),
        (r#"꽠"#, r#"꽠"#, r#"꽠"#, r#"꽠"#, r#"꽠"#),
        (r#"꽡"#, r#"꽡"#, r#"꽡"#, r#"꽡"#, r#"꽡"#),
        (r#"꽢"#, r#"꽢"#, r#"꽢"#, r#"꽢"#, r#"꽢"#),
        (r#"꽣"#, r#"꽣"#, r#"꽣"#, r#"꽣"#, r#"꽣"#),
        (r#"꽤"#, r#"꽤"#, r#"꽤"#, r#"꽤"#, r#"꽤"#),
        (r#"꽥"#, r#"꽥"#, r#"꽥"#, r#"꽥"#, r#"꽥"#),
        (r#"꽦"#, r#"꽦"#, r#"꽦"#, r#"꽦"#, r#"꽦"#),
        (r#"꽧"#, r#"꽧"#, r#"꽧"#, r#"꽧"#, r#"꽧"#),
        (r#"꽨"#, r#"꽨"#, r#"꽨"#, r#"꽨"#, r#"꽨"#),
        (r#"꽩"#, r#"꽩"#, r#"꽩"#, r#"꽩"#, r#"꽩"#),
        (r#"꽪"#, r#"꽪"#, r#"꽪"#, r#"꽪"#, r#"꽪"#),
        (r#"꽫"#, r#"꽫"#, r#"꽫"#, r#"꽫"#, r#"꽫"#),
        (r#"꽬"#, r#"꽬"#, r#"꽬"#, r#"꽬"#, r#"꽬"#),
        (r#"꽭"#, r#"꽭"#, r#"꽭"#, r#"꽭"#, r#"꽭"#),
        (r#"꽮"#, r#"꽮"#, r#"꽮"#, r#"꽮"#, r#"꽮"#),
        (r#"꽯"#, r#"꽯"#, r#"꽯"#, r#"꽯"#, r#"꽯"#),
        (r#"꽰"#, r#"꽰"#, r#"꽰"#, r#"꽰"#, r#"꽰"#),
        (r#"꽱"#, r#"꽱"#, r#"꽱"#, r#"꽱"#, r#"꽱"#),
        (r#"꽲"#, r#"꽲"#, r#"꽲"#, r#"꽲"#, r#"꽲"#),
        (r#"꽳"#, r#"꽳"#, r#"꽳"#, r#"꽳"#, r#"꽳"#),
        (r#"꽴"#, r#"꽴"#, r#"꽴"#, r#"꽴"#, r#"꽴"#),
        (r#"꽵"#, r#"꽵"#, r#"꽵"#, r#"꽵"#, r#"꽵"#),
        (r#"꽶"#, r#"꽶"#, r#"꽶"#, r#"꽶"#, r#"꽶"#),
        (r#"꽷"#, r#"꽷"#, r#"꽷"#, r#"꽷"#, r#"꽷"#),
        (r#"꽸"#, r#"꽸"#, r#"꽸"#, r#"꽸"#, r#"꽸"#),
        (r#"꽹"#, r#"꽹"#, r#"꽹"#, r#"꽹"#, r#"꽹"#),
        (r#"꽺"#, r#"꽺"#, r#"꽺"#, r#"꽺"#, r#"꽺"#),
        (r#"꽻"#, r#"꽻"#, r#"꽻"#, r#"꽻"#, r#"꽻"#),
        (r#"꽼"#, r#"꽼"#, r#"꽼"#, r#"꽼"#, r#"꽼"#),
        (r#"꽽"#, r#"꽽"#, r#"꽽"#, r#"꽽"#, r#"꽽"#),
        (r#"꽾"#, r#"꽾"#, r#"꽾"#, r#"꽾"#, r#"꽾"#),
        (r#"꽿"#, r#"꽿"#, r#"꽿"#, r#"꽿"#, r#"꽿"#),
        (r#"꾀"#, r#"꾀"#, r#"꾀"#, r#"꾀"#, r#"꾀"#),
        (r#"꾁"#, r#"꾁"#, r#"꾁"#, r#"꾁"#, r#"꾁"#),
        (r#"꾂"#, r#"꾂"#, r#"꾂"#, r#"꾂"#, r#"꾂"#),
        (r#"꾃"#, r#"꾃"#, r#"꾃"#, r#"꾃"#, r#"꾃"#),
        (r#"꾄"#, r#"꾄"#, r#"꾄"#, r#"꾄"#, r#"꾄"#),
        (r#"꾅"#, r#"꾅"#, r#"꾅"#, r#"꾅"#, r#"꾅"#),
        (r#"꾆"#, r#"꾆"#, r#"꾆"#, r#"꾆"#, r#"꾆"#),
        (r#"꾇"#, r#"꾇"#, r#"꾇"#, r#"꾇"#, r#"꾇"#),
        (r#"꾈"#, r#"꾈"#, r#"꾈"#, r#"꾈"#, r#"꾈"#),
        (r#"꾉"#, r#"꾉"#, r#"꾉"#, r#"꾉"#, r#"꾉"#),
        (r#"꾊"#, r#"꾊"#, r#"꾊"#, r#"꾊"#, r#"꾊"#),
        (r#"꾋"#, r#"꾋"#, r#"꾋"#, r#"꾋"#, r#"꾋"#),
        (r#"꾌"#, r#"꾌"#, r#"꾌"#, r#"꾌"#, r#"꾌"#),
        (r#"꾍"#, r#"꾍"#, r#"꾍"#, r#"꾍"#, r#"꾍"#),
        (r#"꾎"#, r#"꾎"#, r#"꾎"#, r#"꾎"#, r#"꾎"#),
        (r#"꾏"#, r#"꾏"#, r#"꾏"#, r#"꾏"#, r#"꾏"#),
        (r#"꾐"#, r#"꾐"#, r#"꾐"#, r#"꾐"#, r#"꾐"#),
        (r#"꾑"#, r#"꾑"#, r#"꾑"#, r#"꾑"#, r#"꾑"#),
        (r#"꾒"#, r#"꾒"#, r#"꾒"#, r#"꾒"#, r#"꾒"#),
        (r#"꾓"#, r#"꾓"#, r#"꾓"#, r#"꾓"#, r#"꾓"#),
        (r#"꾔"#, r#"꾔"#, r#"꾔"#, r#"꾔"#, r#"꾔"#),
        (r#"꾕"#, r#"꾕"#, r#"꾕"#, r#"꾕"#, r#"꾕"#),
        (r#"꾖"#, r#"꾖"#, r#"꾖"#, r#"꾖"#, r#"꾖"#),
        (r#"꾗"#, r#"꾗"#, r#"꾗"#, r#"꾗"#, r#"꾗"#),
        (r#"꾘"#, r#"꾘"#, r#"꾘"#, r#"꾘"#, r#"꾘"#),
        (r#"꾙"#, r#"꾙"#, r#"꾙"#, r#"꾙"#, r#"꾙"#),
        (r#"꾚"#, r#"꾚"#, r#"꾚"#, r#"꾚"#, r#"꾚"#),
        (r#"꾛"#, r#"꾛"#, r#"꾛"#, r#"꾛"#, r#"꾛"#),
        (r#"꾜"#, r#"꾜"#, r#"꾜"#, r#"꾜"#, r#"꾜"#),
        (r#"꾝"#, r#"꾝"#, r#"꾝"#, r#"꾝"#, r#"꾝"#),
        (r#"꾞"#, r#"꾞"#, r#"꾞"#, r#"꾞"#, r#"꾞"#),
        (r#"꾟"#, r#"꾟"#, r#"꾟"#, r#"꾟"#, r#"꾟"#),
        (r#"꾠"#, r#"꾠"#, r#"꾠"#, r#"꾠"#, r#"꾠"#),
        (r#"꾡"#, r#"꾡"#, r#"꾡"#, r#"꾡"#, r#"꾡"#),
        (r#"꾢"#, r#"꾢"#, r#"꾢"#, r#"꾢"#, r#"꾢"#),
        (r#"꾣"#, r#"꾣"#, r#"꾣"#, r#"꾣"#, r#"꾣"#),
        (r#"꾤"#, r#"꾤"#, r#"꾤"#, r#"꾤"#, r#"꾤"#),
        (r#"꾥"#, r#"꾥"#, r#"꾥"#, r#"꾥"#, r#"꾥"#),
        (r#"꾦"#, r#"꾦"#, r#"꾦"#, r#"꾦"#, r#"꾦"#),
        (r#"꾧"#, r#"꾧"#, r#"꾧"#, r#"꾧"#, r#"꾧"#),
        (r#"꾨"#, r#"꾨"#, r#"꾨"#, r#"꾨"#, r#"꾨"#),
        (r#"꾩"#, r#"꾩"#, r#"꾩"#, r#"꾩"#, r#"꾩"#),
        (r#"꾪"#, r#"꾪"#, r#"꾪"#, r#"꾪"#, r#"꾪"#),
        (r#"꾫"#, r#"꾫"#, r#"꾫"#, r#"꾫"#, r#"꾫"#),
        (r#"꾬"#, r#"꾬"#, r#"꾬"#, r#"꾬"#, r#"꾬"#),
        (r#"꾭"#, r#"꾭"#, r#"꾭"#, r#"꾭"#, r#"꾭"#),
        (r#"꾮"#, r#"꾮"#, r#"꾮"#, r#"꾮"#, r#"꾮"#),
        (r#"꾯"#, r#"꾯"#, r#"꾯"#, r#"꾯"#, r#"꾯"#),
        (r#"꾰"#, r#"꾰"#, r#"꾰"#, r#"꾰"#, r#"꾰"#),
        (r#"꾱"#, r#"꾱"#, r#"꾱"#, r#"꾱"#, r#"꾱"#),
        (r#"꾲"#, r#"꾲"#, r#"꾲"#, r#"꾲"#, r#"꾲"#),
        (r#"꾳"#, r#"꾳"#, r#"꾳"#, r#"꾳"#, r#"꾳"#),
        (r#"꾴"#, r#"꾴"#, r#"꾴"#, r#"꾴"#, r#"꾴"#),
        (r#"꾵"#, r#"꾵"#, r#"꾵"#, r#"꾵"#, r#"꾵"#),
        (r#"꾶"#, r#"꾶"#, r#"꾶"#, r#"꾶"#, r#"꾶"#),
        (r#"꾷"#, r#"꾷"#, r#"꾷"#, r#"꾷"#, r#"꾷"#),
        (r#"꾸"#, r#"꾸"#, r#"꾸"#, r#"꾸"#, r#"꾸"#),
        (r#"꾹"#, r#"꾹"#, r#"꾹"#, r#"꾹"#, r#"꾹"#),
        (r#"꾺"#, r#"꾺"#, r#"꾺"#, r#"꾺"#, r#"꾺"#),
        (r#"꾻"#, r#"꾻"#, r#"꾻"#, r#"꾻"#, r#"꾻"#),
        (r#"꾼"#, r#"꾼"#, r#"꾼"#, r#"꾼"#, r#"꾼"#),
        (r#"꾽"#, r#"꾽"#, r#"꾽"#, r#"꾽"#, r#"꾽"#),
        (r#"꾾"#, r#"꾾"#, r#"꾾"#, r#"꾾"#, r#"꾾"#),
        (r#"꾿"#, r#"꾿"#, r#"꾿"#, r#"꾿"#, r#"꾿"#),
        (r#"꿀"#, r#"꿀"#, r#"꿀"#, r#"꿀"#, r#"꿀"#),
        (r#"꿁"#, r#"꿁"#, r#"꿁"#, r#"꿁"#, r#"꿁"#),
        (r#"꿂"#, r#"꿂"#, r#"꿂"#, r#"꿂"#, r#"꿂"#),
        (r#"꿃"#, r#"꿃"#, r#"꿃"#, r#"꿃"#, r#"꿃"#),
        (r#"꿄"#, r#"꿄"#, r#"꿄"#, r#"꿄"#, r#"꿄"#),
        (r#"꿅"#, r#"꿅"#, r#"꿅"#, r#"꿅"#, r#"꿅"#),
        (r#"꿆"#, r#"꿆"#, r#"꿆"#, r#"꿆"#, r#"꿆"#),
        (r#"꿇"#, r#"꿇"#, r#"꿇"#, r#"꿇"#, r#"꿇"#),
        (r#"꿈"#, r#"꿈"#, r#"꿈"#, r#"꿈"#, r#"꿈"#),
        (r#"꿉"#, r#"꿉"#, r#"꿉"#, r#"꿉"#, r#"꿉"#),
        (r#"꿊"#, r#"꿊"#, r#"꿊"#, r#"꿊"#, r#"꿊"#),
        (r#"꿋"#, r#"꿋"#, r#"꿋"#, r#"꿋"#, r#"꿋"#),
        (r#"꿌"#, r#"꿌"#, r#"꿌"#, r#"꿌"#, r#"꿌"#),
        (r#"꿍"#, r#"꿍"#, r#"꿍"#, r#"꿍"#, r#"꿍"#),
        (r#"꿎"#, r#"꿎"#, r#"꿎"#, r#"꿎"#, r#"꿎"#),
        (r#"꿏"#, r#"꿏"#, r#"꿏"#, r#"꿏"#, r#"꿏"#),
        (r#"꿐"#, r#"꿐"#, r#"꿐"#, r#"꿐"#, r#"꿐"#),
        (r#"꿑"#, r#"꿑"#, r#"꿑"#, r#"꿑"#, r#"꿑"#),
        (r#"꿒"#, r#"꿒"#, r#"꿒"#, r#"꿒"#, r#"꿒"#),
        (r#"꿓"#, r#"꿓"#, r#"꿓"#, r#"꿓"#, r#"꿓"#),
        (r#"꿔"#, r#"꿔"#, r#"꿔"#, r#"꿔"#, r#"꿔"#),
        (r#"꿕"#, r#"꿕"#, r#"꿕"#, r#"꿕"#, r#"꿕"#),
        (r#"꿖"#, r#"꿖"#, r#"꿖"#, r#"꿖"#, r#"꿖"#),
        (r#"꿗"#, r#"꿗"#, r#"꿗"#, r#"꿗"#, r#"꿗"#),
        (r#"꿘"#, r#"꿘"#, r#"꿘"#, r#"꿘"#, r#"꿘"#),
        (r#"꿙"#, r#"꿙"#, r#"꿙"#, r#"꿙"#, r#"꿙"#),
        (r#"꿚"#, r#"꿚"#, r#"꿚"#, r#"꿚"#, r#"꿚"#),
        (r#"꿛"#, r#"꿛"#, r#"꿛"#, r#"꿛"#, r#"꿛"#),
        (r#"꿜"#, r#"꿜"#, r#"꿜"#, r#"꿜"#, r#"꿜"#),
        (r#"꿝"#, r#"꿝"#, r#"꿝"#, r#"꿝"#, r#"꿝"#),
        (r#"꿞"#, r#"꿞"#, r#"꿞"#, r#"꿞"#, r#"꿞"#),
        (r#"꿟"#, r#"꿟"#, r#"꿟"#, r#"꿟"#, r#"꿟"#),
        (r#"꿠"#, r#"꿠"#, r#"꿠"#, r#"꿠"#, r#"꿠"#),
        (r#"꿡"#, r#"꿡"#, r#"꿡"#, r#"꿡"#, r#"꿡"#),
        (r#"꿢"#, r#"꿢"#, r#"꿢"#, r#"꿢"#, r#"꿢"#),
        (r#"꿣"#, r#"꿣"#, r#"꿣"#, r#"꿣"#, r#"꿣"#),
        (r#"꿤"#, r#"꿤"#, r#"꿤"#, r#"꿤"#, r#"꿤"#),
        (r#"꿥"#, r#"꿥"#, r#"꿥"#, r#"꿥"#, r#"꿥"#),
        (r#"꿦"#, r#"꿦"#, r#"꿦"#, r#"꿦"#, r#"꿦"#),
        (r#"꿧"#, r#"꿧"#, r#"꿧"#, r#"꿧"#, r#"꿧"#),
        (r#"꿨"#, r#"꿨"#, r#"꿨"#, r#"꿨"#, r#"꿨"#),
        (r#"꿩"#, r#"꿩"#, r#"꿩"#, r#"꿩"#, r#"꿩"#),
        (r#"꿪"#, r#"꿪"#, r#"꿪"#, r#"꿪"#, r#"꿪"#),
        (r#"꿫"#, r#"꿫"#, r#"꿫"#, r#"꿫"#, r#"꿫"#),
        (r#"꿬"#, r#"꿬"#, r#"꿬"#, r#"꿬"#, r#"꿬"#),
        (r#"꿭"#, r#"꿭"#, r#"꿭"#, r#"꿭"#, r#"꿭"#),
        (r#"꿮"#, r#"꿮"#, r#"꿮"#, r#"꿮"#, r#"꿮"#),
        (r#"꿯"#, r#"꿯"#, r#"꿯"#, r#"꿯"#, r#"꿯"#),
        (r#"꿰"#, r#"꿰"#, r#"꿰"#, r#"꿰"#, r#"꿰"#),
        (r#"꿱"#, r#"꿱"#, r#"꿱"#, r#"꿱"#, r#"꿱"#),
        (r#"꿲"#, r#"꿲"#, r#"꿲"#, r#"꿲"#, r#"꿲"#),
        (r#"꿳"#, r#"꿳"#, r#"꿳"#, r#"꿳"#, r#"꿳"#),
        (r#"꿴"#, r#"꿴"#, r#"꿴"#, r#"꿴"#, r#"꿴"#),
        (r#"꿵"#, r#"꿵"#, r#"꿵"#, r#"꿵"#, r#"꿵"#),
        (r#"꿶"#, r#"꿶"#, r#"꿶"#, r#"꿶"#, r#"꿶"#),
        (r#"꿷"#, r#"꿷"#, r#"꿷"#, r#"꿷"#, r#"꿷"#),
        (r#"꿸"#, r#"꿸"#, r#"꿸"#, r#"꿸"#, r#"꿸"#),
        (r#"꿹"#, r#"꿹"#, r#"꿹"#, r#"꿹"#, r#"꿹"#),
        (r#"꿺"#, r#"꿺"#, r#"꿺"#, r#"꿺"#, r#"꿺"#),
        (r#"꿻"#, r#"꿻"#, r#"꿻"#, r#"꿻"#, r#"꿻"#),
        (r#"꿼"#, r#"꿼"#, r#"꿼"#, r#"꿼"#, r#"꿼"#),
        (r#"꿽"#, r#"꿽"#, r#"꿽"#, r#"꿽"#, r#"꿽"#),
        (r#"꿾"#, r#"꿾"#, r#"꿾"#, r#"꿾"#, r#"꿾"#),
        (r#"꿿"#, r#"꿿"#, r#"꿿"#, r#"꿿"#, r#"꿿"#),
        (r#"뀀"#, r#"뀀"#, r#"뀀"#, r#"뀀"#, r#"뀀"#),
        (r#"뀁"#, r#"뀁"#, r#"뀁"#, r#"뀁"#, r#"뀁"#),
        (r#"뀂"#, r#"뀂"#, r#"뀂"#, r#"뀂"#, r#"뀂"#),
        (r#"뀃"#, r#"뀃"#, r#"뀃"#, r#"뀃"#, r#"뀃"#),
        (r#"뀄"#, r#"뀄"#, r#"뀄"#, r#"뀄"#, r#"뀄"#),
        (r#"뀅"#, r#"뀅"#, r#"뀅"#, r#"뀅"#, r#"뀅"#),
        (r#"뀆"#, r#"뀆"#, r#"뀆"#, r#"뀆"#, r#"뀆"#),
        (r#"뀇"#, r#"뀇"#, r#"뀇"#, r#"뀇"#, r#"뀇"#),
        (r#"뀈"#, r#"뀈"#, r#"뀈"#, r#"뀈"#, r#"뀈"#),
        (r#"뀉"#, r#"뀉"#, r#"뀉"#, r#"뀉"#, r#"뀉"#),
        (r#"뀊"#, r#"뀊"#, r#"뀊"#, r#"뀊"#, r#"뀊"#),
        (r#"뀋"#, r#"뀋"#, r#"뀋"#, r#"뀋"#, r#"뀋"#),
        (r#"뀌"#, r#"뀌"#, r#"뀌"#, r#"뀌"#, r#"뀌"#),
        (r#"뀍"#, r#"뀍"#, r#"뀍"#, r#"뀍"#, r#"뀍"#),
        (r#"뀎"#, r#"뀎"#, r#"뀎"#, r#"뀎"#, r#"뀎"#),
        (r#"뀏"#, r#"뀏"#, r#"뀏"#, r#"뀏"#, r#"뀏"#),
        (r#"뀐"#, r#"뀐"#, r#"뀐"#, r#"뀐"#, r#"뀐"#),
        (r#"뀑"#, r#"뀑"#, r#"뀑"#, r#"뀑"#, r#"뀑"#),
        (r#"뀒"#, r#"뀒"#, r#"뀒"#, r#"뀒"#, r#"뀒"#),
        (r#"뀓"#, r#"뀓"#, r#"뀓"#, r#"뀓"#, r#"뀓"#),
        (r#"뀔"#, r#"뀔"#, r#"뀔"#, r#"뀔"#, r#"뀔"#),
        (r#"뀕"#, r#"뀕"#, r#"뀕"#, r#"뀕"#, r#"뀕"#),
        (r#"뀖"#, r#"뀖"#, r#"뀖"#, r#"뀖"#, r#"뀖"#),
        (r#"뀗"#, r#"뀗"#, r#"뀗"#, r#"뀗"#, r#"뀗"#),
        (r#"뀘"#, r#"뀘"#, r#"뀘"#, r#"뀘"#, r#"뀘"#),
        (r#"뀙"#, r#"뀙"#, r#"뀙"#, r#"뀙"#, r#"뀙"#),
        (r#"뀚"#, r#"뀚"#, r#"뀚"#, r#"뀚"#, r#"뀚"#),
        (r#"뀛"#, r#"뀛"#, r#"뀛"#, r#"뀛"#, r#"뀛"#),
        (r#"뀜"#, r#"뀜"#, r#"뀜"#, r#"뀜"#, r#"뀜"#),
        (r#"뀝"#, r#"뀝"#, r#"뀝"#, r#"뀝"#, r#"뀝"#),
        (r#"뀞"#, r#"뀞"#, r#"뀞"#, r#"뀞"#, r#"뀞"#),
        (r#"뀟"#, r#"뀟"#, r#"뀟"#, r#"뀟"#, r#"뀟"#),
        (r#"뀠"#, r#"뀠"#, r#"뀠"#, r#"뀠"#, r#"뀠"#),
        (r#"뀡"#, r#"뀡"#, r#"뀡"#, r#"뀡"#, r#"뀡"#),
        (r#"뀢"#, r#"뀢"#, r#"뀢"#, r#"뀢"#, r#"뀢"#),
        (r#"뀣"#, r#"뀣"#, r#"뀣"#, r#"뀣"#, r#"뀣"#),
        (r#"뀤"#, r#"뀤"#, r#"뀤"#, r#"뀤"#, r#"뀤"#),
        (r#"뀥"#, r#"뀥"#, r#"뀥"#, r#"뀥"#, r#"뀥"#),
        (r#"뀦"#, r#"뀦"#, r#"뀦"#, r#"뀦"#, r#"뀦"#),
        (r#"뀧"#, r#"뀧"#, r#"뀧"#, r#"뀧"#, r#"뀧"#),
        (r#"뀨"#, r#"뀨"#, r#"뀨"#, r#"뀨"#, r#"뀨"#),
        (r#"뀩"#, r#"뀩"#, r#"뀩"#, r#"뀩"#, r#"뀩"#),
        (r#"뀪"#, r#"뀪"#, r#"뀪"#, r#"뀪"#, r#"뀪"#),
        (r#"뀫"#, r#"뀫"#, r#"뀫"#, r#"뀫"#, r#"뀫"#),
        (r#"뀬"#, r#"뀬"#, r#"뀬"#, r#"뀬"#, r#"뀬"#),
        (r#"뀭"#, r#"뀭"#, r#"뀭"#, r#"뀭"#, r#"뀭"#),
        (r#"뀮"#, r#"뀮"#, r#"뀮"#, r#"뀮"#, r#"뀮"#),
        (r#"뀯"#, r#"뀯"#, r#"뀯"#, r#"뀯"#, r#"뀯"#),
        (r#"뀰"#, r#"뀰"#, r#"뀰"#, r#"뀰"#, r#"뀰"#),
        (r#"뀱"#, r#"뀱"#, r#"뀱"#, r#"뀱"#, r#"뀱"#),
        (r#"뀲"#, r#"뀲"#, r#"뀲"#, r#"뀲"#, r#"뀲"#),
        (r#"뀳"#, r#"뀳"#, r#"뀳"#, r#"뀳"#, r#"뀳"#),
        (r#"뀴"#, r#"뀴"#, r#"뀴"#, r#"뀴"#, r#"뀴"#),
        (r#"뀵"#, r#"뀵"#, r#"뀵"#, r#"뀵"#, r#"뀵"#),
        (r#"뀶"#, r#"뀶"#, r#"뀶"#, r#"뀶"#, r#"뀶"#),
        (r#"뀷"#, r#"뀷"#, r#"뀷"#, r#"뀷"#, r#"뀷"#),
        (r#"뀸"#, r#"뀸"#, r#"뀸"#, r#"뀸"#, r#"뀸"#),
        (r#"뀹"#, r#"뀹"#, r#"뀹"#, r#"뀹"#, r#"뀹"#),
        (r#"뀺"#, r#"뀺"#, r#"뀺"#, r#"뀺"#, r#"뀺"#),
        (r#"뀻"#, r#"뀻"#, r#"뀻"#, r#"뀻"#, r#"뀻"#),
        (r#"뀼"#, r#"뀼"#, r#"뀼"#, r#"뀼"#, r#"뀼"#),
        (r#"뀽"#, r#"뀽"#, r#"뀽"#, r#"뀽"#, r#"뀽"#),
        (r#"뀾"#, r#"뀾"#, r#"뀾"#, r#"뀾"#, r#"뀾"#),
        (r#"뀿"#, r#"뀿"#, r#"뀿"#, r#"뀿"#, r#"뀿"#),
        (r#"끀"#, r#"끀"#, r#"끀"#, r#"끀"#, r#"끀"#),
        (r#"끁"#, r#"끁"#, r#"끁"#, r#"끁"#, r#"끁"#),
        (r#"끂"#, r#"끂"#, r#"끂"#, r#"끂"#, r#"끂"#),
        (r#"끃"#, r#"끃"#, r#"끃"#, r#"끃"#, r#"끃"#),
        (r#"끄"#, r#"끄"#, r#"끄"#, r#"끄"#, r#"끄"#),
        (r#"끅"#, r#"끅"#, r#"끅"#, r#"끅"#, r#"끅"#),
        (r#"끆"#, r#"끆"#, r#"끆"#, r#"끆"#, r#"끆"#),
        (r#"끇"#, r#"끇"#, r#"끇"#, r#"끇"#, r#"끇"#),
        (r#"끈"#, r#"끈"#, r#"끈"#, r#"끈"#, r#"끈"#),
        (r#"끉"#, r#"끉"#, r#"끉"#, r#"끉"#, r#"끉"#),
        (r#"끊"#, r#"끊"#, r#"끊"#, r#"끊"#, r#"끊"#),
        (r#"끋"#, r#"끋"#, r#"끋"#, r#"끋"#, r#"끋"#),
        (r#"끌"#, r#"끌"#, r#"끌"#, r#"끌"#, r#"끌"#),
        (r#"끍"#, r#"끍"#, r#"끍"#, r#"끍"#, r#"끍"#),
        (r#"끎"#, r#"끎"#, r#"끎"#, r#"끎"#, r#"끎"#),
        (r#"끏"#, r#"끏"#, r#"끏"#, r#"끏"#, r#"끏"#),
        (r#"끐"#, r#"끐"#, r#"끐"#, r#"끐"#, r#"끐"#),
        (r#"끑"#, r#"끑"#, r#"끑"#, r#"끑"#, r#"끑"#),
        (r#"끒"#, r#"끒"#, r#"끒"#, r#"끒"#, r#"끒"#),
        (r#"끓"#, r#"끓"#, r#"끓"#, r#"끓"#, r#"끓"#),
        (r#"끔"#, r#"끔"#, r#"끔"#, r#"끔"#, r#"끔"#),
        (r#"끕"#, r#"끕"#, r#"끕"#, r#"끕"#, r#"끕"#),
        (r#"끖"#, r#"끖"#, r#"끖"#, r#"끖"#, r#"끖"#),
        (r#"끗"#, r#"끗"#, r#"끗"#, r#"끗"#, r#"끗"#),
        (r#"끘"#, r#"끘"#, r#"끘"#, r#"끘"#, r#"끘"#),
        (r#"끙"#, r#"끙"#, r#"끙"#, r#"끙"#, r#"끙"#),
        (r#"끚"#, r#"끚"#, r#"끚"#, r#"끚"#, r#"끚"#),
        (r#"끛"#, r#"끛"#, r#"끛"#, r#"끛"#, r#"끛"#),
        (r#"끜"#, r#"끜"#, r#"끜"#, r#"끜"#, r#"끜"#),
        (r#"끝"#, r#"끝"#, r#"끝"#, r#"끝"#, r#"끝"#),
        (r#"끞"#, r#"끞"#, r#"끞"#, r#"끞"#, r#"끞"#),
        (r#"끟"#, r#"끟"#, r#"끟"#, r#"끟"#, r#"끟"#),
        (r#"끠"#, r#"끠"#, r#"끠"#, r#"끠"#, r#"끠"#),
        (r#"끡"#, r#"끡"#, r#"끡"#, r#"끡"#, r#"끡"#),
        (r#"끢"#, r#"끢"#, r#"끢"#, r#"끢"#, r#"끢"#),
        (r#"끣"#, r#"끣"#, r#"끣"#, r#"끣"#, r#"끣"#),
        (r#"끤"#, r#"끤"#, r#"끤"#, r#"끤"#, r#"끤"#),
        (r#"끥"#, r#"끥"#, r#"끥"#, r#"끥"#, r#"끥"#),
        (r#"끦"#, r#"끦"#, r#"끦"#, r#"끦"#, r#"끦"#),
        (r#"끧"#, r#"끧"#, r#"끧"#, r#"끧"#, r#"끧"#),
        (r#"끨"#, r#"끨"#, r#"끨"#, r#"끨"#, r#"끨"#),
        (r#"끩"#, r#"끩"#, r#"끩"#, r#"끩"#, r#"끩"#),
        (r#"끪"#, r#"끪"#, r#"끪"#, r#"끪"#, r#"끪"#),
        (r#"끫"#, r#"끫"#, r#"끫"#, r#"끫"#, r#"끫"#),
        (r#"끬"#, r#"끬"#, r#"끬"#, r#"끬"#, r#"끬"#),
        (r#"끭"#, r#"끭"#, r#"끭"#, r#"끭"#, r#"끭"#),
        (r#"끮"#, r#"끮"#, r#"끮"#, r#"끮"#, r#"끮"#),
        (r#"끯"#, r#"끯"#, r#"끯"#, r#"끯"#, r#"끯"#),
        (r#"끰"#, r#"끰"#, r#"끰"#, r#"끰"#, r#"끰"#),
        (r#"끱"#, r#"끱"#, r#"끱"#, r#"끱"#, r#"끱"#),
        (r#"끲"#, r#"끲"#, r#"끲"#, r#"끲"#, r#"끲"#),
        (r#"끳"#, r#"끳"#, r#"끳"#, r#"끳"#, r#"끳"#),
        (r#"끴"#, r#"끴"#, r#"끴"#, r#"끴"#, r#"끴"#),
        (r#"끵"#, r#"끵"#, r#"끵"#, r#"끵"#, r#"끵"#),
        (r#"끶"#, r#"끶"#, r#"끶"#, r#"끶"#, r#"끶"#),
        (r#"끷"#, r#"끷"#, r#"끷"#, r#"끷"#, r#"끷"#),
        (r#"끸"#, r#"끸"#, r#"끸"#, r#"끸"#, r#"끸"#),
        (r#"끹"#, r#"끹"#, r#"끹"#, r#"끹"#, r#"끹"#),
        (r#"끺"#, r#"끺"#, r#"끺"#, r#"끺"#, r#"끺"#),
        (r#"끻"#, r#"끻"#, r#"끻"#, r#"끻"#, r#"끻"#),
        (r#"끼"#, r#"끼"#, r#"끼"#, r#"끼"#, r#"끼"#),
        (r#"끽"#, r#"끽"#, r#"끽"#, r#"끽"#, r#"끽"#),
        (r#"끾"#, r#"끾"#, r#"끾"#, r#"끾"#, r#"끾"#),
        (r#"끿"#, r#"끿"#, r#"끿"#, r#"끿"#, r#"끿"#),
        (r#"낀"#, r#"낀"#, r#"낀"#, r#"낀"#, r#"낀"#),
        (r#"낁"#, r#"낁"#, r#"낁"#, r#"낁"#, r#"낁"#),
        (r#"낂"#, r#"낂"#, r#"낂"#, r#"낂"#, r#"낂"#),
        (r#"낃"#, r#"낃"#, r#"낃"#, r#"낃"#, r#"낃"#),
        (r#"낄"#, r#"낄"#, r#"낄"#, r#"낄"#, r#"낄"#),
        (r#"낅"#, r#"낅"#, r#"낅"#, r#"낅"#, r#"낅"#),
        (r#"낆"#, r#"낆"#, r#"낆"#, r#"낆"#, r#"낆"#),
        (r#"낇"#, r#"낇"#, r#"낇"#, r#"낇"#, r#"낇"#),
        (r#"낈"#, r#"낈"#, r#"낈"#, r#"낈"#, r#"낈"#),
        (r#"낉"#, r#"낉"#, r#"낉"#, r#"낉"#, r#"낉"#),
        (r#"낊"#, r#"낊"#, r#"낊"#, r#"낊"#, r#"낊"#),
        (r#"낋"#, r#"낋"#, r#"낋"#, r#"낋"#, r#"낋"#),
        (r#"낌"#, r#"낌"#, r#"낌"#, r#"낌"#, r#"낌"#),
        (r#"낍"#, r#"낍"#, r#"낍"#, r#"낍"#, r#"낍"#),
        (r#"낎"#, r#"낎"#, r#"낎"#, r#"낎"#, r#"낎"#),
        (r#"낏"#, r#"낏"#, r#"낏"#, r#"낏"#, r#"낏"#),
        (r#"낐"#, r#"낐"#, r#"낐"#, r#"낐"#, r#"낐"#),
        (r#"낑"#, r#"낑"#, r#"낑"#, r#"낑"#, r#"낑"#),
        (r#"낒"#, r#"낒"#, r#"낒"#, r#"낒"#, r#"낒"#),
        (r#"낓"#, r#"낓"#, r#"낓"#, r#"낓"#, r#"낓"#),
        (r#"낔"#, r#"낔"#, r#"낔"#, r#"낔"#, r#"낔"#),
        (r#"낕"#, r#"낕"#, r#"낕"#, r#"낕"#, r#"낕"#),
        (r#"낖"#, r#"낖"#, r#"낖"#, r#"낖"#, r#"낖"#),
        (r#"낗"#, r#"낗"#, r#"낗"#, r#"낗"#, r#"낗"#),
        (r#"나"#, r#"나"#, r#"나"#, r#"나"#, r#"나"#),
        (r#"낙"#, r#"낙"#, r#"낙"#, r#"낙"#, r#"낙"#),
        (r#"낚"#, r#"낚"#, r#"낚"#, r#"낚"#, r#"낚"#),
        (r#"낛"#, r#"낛"#, r#"낛"#, r#"낛"#, r#"낛"#),
        (r#"난"#, r#"난"#, r#"난"#, r#"난"#, r#"난"#),
        (r#"낝"#, r#"낝"#, r#"낝"#, r#"낝"#, r#"낝"#),
        (r#"낞"#, r#"낞"#, r#"낞"#, r#"낞"#, r#"낞"#),
        (r#"낟"#, r#"낟"#, r#"낟"#, r#"낟"#, r#"낟"#),
        (r#"날"#, r#"날"#, r#"날"#, r#"날"#, r#"날"#),
        (r#"낡"#, r#"낡"#, r#"낡"#, r#"낡"#, r#"낡"#),
        (r#"낢"#, r#"낢"#, r#"낢"#, r#"낢"#, r#"낢"#),
        (r#"낣"#, r#"낣"#, r#"낣"#, r#"낣"#, r#"낣"#),
        (r#"낤"#, r#"낤"#, r#"낤"#, r#"낤"#, r#"낤"#),
        (r#"낥"#, r#"낥"#, r#"낥"#, r#"낥"#, r#"낥"#),
        (r#"낦"#, r#"낦"#, r#"낦"#, r#"낦"#, r#"낦"#),
        (r#"낧"#, r#"낧"#, r#"낧"#, r#"낧"#, r#"낧"#),
        (r#"남"#, r#"남"#, r#"남"#, r#"남"#, r#"남"#),
        (r#"납"#, r#"납"#, r#"납"#, r#"납"#, r#"납"#),
        (r#"낪"#, r#"낪"#, r#"낪"#, r#"낪"#, r#"낪"#),
        (r#"낫"#, r#"낫"#, r#"낫"#, r#"낫"#, r#"낫"#),
        (r#"났"#, r#"났"#, r#"났"#, r#"났"#, r#"났"#),
        (r#"낭"#, r#"낭"#, r#"낭"#, r#"낭"#, r#"낭"#),
        (r#"낮"#, r#"낮"#, r#"낮"#, r#"낮"#, r#"낮"#),
        (r#"낯"#, r#"낯"#, r#"낯"#, r#"낯"#, r#"낯"#),
        (r#"낰"#, r#"낰"#, r#"낰"#, r#"낰"#, r#"낰"#),
        (r#"낱"#, r#"낱"#, r#"낱"#, r#"낱"#, r#"낱"#),
        (r#"낲"#, r#"낲"#, r#"낲"#, r#"낲"#, r#"낲"#),
        (r#"낳"#, r#"낳"#, r#"낳"#, r#"낳"#, r#"낳"#),
        (r#"내"#, r#"내"#, r#"내"#, r#"내"#, r#"내"#),
        (r#"낵"#, r#"낵"#, r#"낵"#, r#"낵"#, r#"낵"#),
        (r#"낶"#, r#"낶"#, r#"낶"#, r#"낶"#, r#"낶"#),
        (r#"낷"#, r#"낷"#, r#"낷"#, r#"낷"#, r#"낷"#),
        (r#"낸"#, r#"낸"#, r#"낸"#, r#"낸"#, r#"낸"#),
        (r#"낹"#, r#"낹"#, r#"낹"#, r#"낹"#, r#"낹"#),
        (r#"낺"#, r#"낺"#, r#"낺"#, r#"낺"#, r#"낺"#),
        (r#"낻"#, r#"낻"#, r#"낻"#, r#"낻"#, r#"낻"#),
        (r#"낼"#, r#"낼"#, r#"낼"#, r#"낼"#, r#"낼"#),
        (r#"낽"#, r#"낽"#, r#"낽"#, r#"낽"#, r#"낽"#),
        (r#"낾"#, r#"낾"#, r#"낾"#, r#"낾"#, r#"낾"#),
        (r#"낿"#, r#"낿"#, r#"낿"#, r#"낿"#, r#"낿"#),
        (r#"냀"#, r#"냀"#, r#"냀"#, r#"냀"#, r#"냀"#),
        (r#"냁"#, r#"냁"#, r#"냁"#, r#"냁"#, r#"냁"#),
        (r#"냂"#, r#"냂"#, r#"냂"#, r#"냂"#, r#"냂"#),
        (r#"냃"#, r#"냃"#, r#"냃"#, r#"냃"#, r#"냃"#),
        (r#"냄"#, r#"냄"#, r#"냄"#, r#"냄"#, r#"냄"#),
        (r#"냅"#, r#"냅"#, r#"냅"#, r#"냅"#, r#"냅"#),
        (r#"냆"#, r#"냆"#, r#"냆"#, r#"냆"#, r#"냆"#),
        (r#"냇"#, r#"냇"#, r#"냇"#, r#"냇"#, r#"냇"#),
        (r#"냈"#, r#"냈"#, r#"냈"#, r#"냈"#, r#"냈"#),
        (r#"냉"#, r#"냉"#, r#"냉"#, r#"냉"#, r#"냉"#),
        (r#"냊"#, r#"냊"#, r#"냊"#, r#"냊"#, r#"냊"#),
        (r#"냋"#, r#"냋"#, r#"냋"#, r#"냋"#, r#"냋"#),
        (r#"냌"#, r#"냌"#, r#"냌"#, r#"냌"#, r#"냌"#),
        (r#"냍"#, r#"냍"#, r#"냍"#, r#"냍"#, r#"냍"#),
        (r#"냎"#, r#"냎"#, r#"냎"#, r#"냎"#, r#"냎"#),
        (r#"냏"#, r#"냏"#, r#"냏"#, r#"냏"#, r#"냏"#),
        (r#"냐"#, r#"냐"#, r#"냐"#, r#"냐"#, r#"냐"#),
        (r#"냑"#, r#"냑"#, r#"냑"#, r#"냑"#, r#"냑"#),
        (r#"냒"#, r#"냒"#, r#"냒"#, r#"냒"#, r#"냒"#),
        (r#"냓"#, r#"냓"#, r#"냓"#, r#"냓"#, r#"냓"#),
        (r#"냔"#, r#"냔"#, r#"냔"#, r#"냔"#, r#"냔"#),
        (r#"냕"#, r#"냕"#, r#"냕"#, r#"냕"#, r#"냕"#),
        (r#"냖"#, r#"냖"#, r#"냖"#, r#"냖"#, r#"냖"#),
        (r#"냗"#, r#"냗"#, r#"냗"#, r#"냗"#, r#"냗"#),
        (r#"냘"#, r#"냘"#, r#"냘"#, r#"냘"#, r#"냘"#),
        (r#"냙"#, r#"냙"#, r#"냙"#, r#"냙"#, r#"냙"#),
        (r#"냚"#, r#"냚"#, r#"냚"#, r#"냚"#, r#"냚"#),
        (r#"냛"#, r#"냛"#, r#"냛"#, r#"냛"#, r#"냛"#),
        (r#"냜"#, r#"냜"#, r#"냜"#, r#"냜"#, r#"냜"#),
        (r#"냝"#, r#"냝"#, r#"냝"#, r#"냝"#, r#"냝"#),
        (r#"냞"#, r#"냞"#, r#"냞"#, r#"냞"#, r#"냞"#),
        (r#"냟"#, r#"냟"#, r#"냟"#, r#"냟"#, r#"냟"#),
        (r#"냠"#, r#"냠"#, r#"냠"#, r#"냠"#, r#"냠"#),
        (r#"냡"#, r#"냡"#, r#"냡"#, r#"냡"#, r#"냡"#),
        (r#"냢"#, r#"냢"#, r#"냢"#, r#"냢"#, r#"냢"#),
        (r#"냣"#, r#"냣"#, r#"냣"#, r#"냣"#, r#"냣"#),
        (r#"냤"#, r#"냤"#, r#"냤"#, r#"냤"#, r#"냤"#),
        (r#"냥"#, r#"냥"#, r#"냥"#, r#"냥"#, r#"냥"#),
        (r#"냦"#, r#"냦"#, r#"냦"#, r#"냦"#, r#"냦"#),
        (r#"냧"#, r#"냧"#, r#"냧"#, r#"냧"#, r#"냧"#),
        (r#"냨"#, r#"냨"#, r#"냨"#, r#"냨"#, r#"냨"#),
        (r#"냩"#, r#"냩"#, r#"냩"#, r#"냩"#, r#"냩"#),
        (r#"냪"#, r#"냪"#, r#"냪"#, r#"냪"#, r#"냪"#),
        (r#"냫"#, r#"냫"#, r#"냫"#, r#"냫"#, r#"냫"#),
        (r#"냬"#, r#"냬"#, r#"냬"#, r#"냬"#, r#"냬"#),
        (r#"냭"#, r#"냭"#, r#"냭"#, r#"냭"#, r#"냭"#),
        (r#"냮"#, r#"냮"#, r#"냮"#, r#"냮"#, r#"냮"#),
        (r#"냯"#, r#"냯"#, r#"냯"#, r#"냯"#, r#"냯"#),
        (r#"냰"#, r#"냰"#, r#"냰"#, r#"냰"#, r#"냰"#),
        (r#"냱"#, r#"냱"#, r#"냱"#, r#"냱"#, r#"냱"#),
        (r#"냲"#, r#"냲"#, r#"냲"#, r#"냲"#, r#"냲"#),
        (r#"냳"#, r#"냳"#, r#"냳"#, r#"냳"#, r#"냳"#),
        (r#"냴"#, r#"냴"#, r#"냴"#, r#"냴"#, r#"냴"#),
        (r#"냵"#, r#"냵"#, r#"냵"#, r#"냵"#, r#"냵"#),
        (r#"냶"#, r#"냶"#, r#"냶"#, r#"냶"#, r#"냶"#),
        (r#"냷"#, r#"냷"#, r#"냷"#, r#"냷"#, r#"냷"#),
        (r#"냸"#, r#"냸"#, r#"냸"#, r#"냸"#, r#"냸"#),
        (r#"냹"#, r#"냹"#, r#"냹"#, r#"냹"#, r#"냹"#),
        (r#"냺"#, r#"냺"#, r#"냺"#, r#"냺"#, r#"냺"#),
        (r#"냻"#, r#"냻"#, r#"냻"#, r#"냻"#, r#"냻"#),
        (r#"냼"#, r#"냼"#, r#"냼"#, r#"냼"#, r#"냼"#),
        (r#"냽"#, r#"냽"#, r#"냽"#, r#"냽"#, r#"냽"#),
        (r#"냾"#, r#"냾"#, r#"냾"#, r#"냾"#, r#"냾"#),
        (r#"냿"#, r#"냿"#, r#"냿"#, r#"냿"#, r#"냿"#),
        (r#"넀"#, r#"넀"#, r#"넀"#, r#"넀"#, r#"넀"#),
        (r#"넁"#, r#"넁"#, r#"넁"#, r#"넁"#, r#"넁"#),
        (r#"넂"#, r#"넂"#, r#"넂"#, r#"넂"#, r#"넂"#),
        (r#"넃"#, r#"넃"#, r#"넃"#, r#"넃"#, r#"넃"#),
        (r#"넄"#, r#"넄"#, r#"넄"#, r#"넄"#, r#"넄"#),
        (r#"넅"#, r#"넅"#, r#"넅"#, r#"넅"#, r#"넅"#),
        (r#"넆"#, r#"넆"#, r#"넆"#, r#"넆"#, r#"넆"#),
        (r#"넇"#, r#"넇"#, r#"넇"#, r#"넇"#, r#"넇"#),
        (r#"너"#, r#"너"#, r#"너"#, r#"너"#, r#"너"#),
        (r#"넉"#, r#"넉"#, r#"넉"#, r#"넉"#, r#"넉"#),
        (r#"넊"#, r#"넊"#, r#"넊"#, r#"넊"#, r#"넊"#),
        (r#"넋"#, r#"넋"#, r#"넋"#, r#"넋"#, r#"넋"#),
        (r#"넌"#, r#"넌"#, r#"넌"#, r#"넌"#, r#"넌"#),
        (r#"넍"#, r#"넍"#, r#"넍"#, r#"넍"#, r#"넍"#),
        (r#"넎"#, r#"넎"#, r#"넎"#, r#"넎"#, r#"넎"#),
        (r#"넏"#, r#"넏"#, r#"넏"#, r#"넏"#, r#"넏"#),
        (r#"널"#, r#"널"#, r#"널"#, r#"널"#, r#"널"#),
        (r#"넑"#, r#"넑"#, r#"넑"#, r#"넑"#, r#"넑"#),
        (r#"넒"#, r#"넒"#, r#"넒"#, r#"넒"#, r#"넒"#),
        (r#"넓"#, r#"넓"#, r#"넓"#, r#"넓"#, r#"넓"#),
        (r#"넔"#, r#"넔"#, r#"넔"#, r#"넔"#, r#"넔"#),
        (r#"넕"#, r#"넕"#, r#"넕"#, r#"넕"#, r#"넕"#),
        (r#"넖"#, r#"넖"#, r#"넖"#, r#"넖"#, r#"넖"#),
        (r#"넗"#, r#"넗"#, r#"넗"#, r#"넗"#, r#"넗"#),
        (r#"넘"#, r#"넘"#, r#"넘"#, r#"넘"#, r#"넘"#),
        (r#"넙"#, r#"넙"#, r#"넙"#, r#"넙"#, r#"넙"#),
        (r#"넚"#, r#"넚"#, r#"넚"#, r#"넚"#, r#"넚"#),
        (r#"넛"#, r#"넛"#, r#"넛"#, r#"넛"#, r#"넛"#),
        (r#"넜"#, r#"넜"#, r#"넜"#, r#"넜"#, r#"넜"#),
        (r#"넝"#, r#"넝"#, r#"넝"#, r#"넝"#, r#"넝"#),
        (r#"넞"#, r#"넞"#, r#"넞"#, r#"넞"#, r#"넞"#),
        (r#"넟"#, r#"넟"#, r#"넟"#, r#"넟"#, r#"넟"#),
        (r#"넠"#, r#"넠"#, r#"넠"#, r#"넠"#, r#"넠"#),
        (r#"넡"#, r#"넡"#, r#"넡"#, r#"넡"#, r#"넡"#),
        (r#"넢"#, r#"넢"#, r#"넢"#, r#"넢"#, r#"넢"#),
        (r#"넣"#, r#"넣"#, r#"넣"#, r#"넣"#, r#"넣"#),
        (r#"네"#, r#"네"#, r#"네"#, r#"네"#, r#"네"#),
        (r#"넥"#, r#"넥"#, r#"넥"#, r#"넥"#, r#"넥"#),
        (r#"넦"#, r#"넦"#, r#"넦"#, r#"넦"#, r#"넦"#),
        (r#"넧"#, r#"넧"#, r#"넧"#, r#"넧"#, r#"넧"#),
        (r#"넨"#, r#"넨"#, r#"넨"#, r#"넨"#, r#"넨"#),
        (r#"넩"#, r#"넩"#, r#"넩"#, r#"넩"#, r#"넩"#),
        (r#"넪"#, r#"넪"#, r#"넪"#, r#"넪"#, r#"넪"#),
        (r#"넫"#, r#"넫"#, r#"넫"#, r#"넫"#, r#"넫"#),
        (r#"넬"#, r#"넬"#, r#"넬"#, r#"넬"#, r#"넬"#),
        (r#"넭"#, r#"넭"#, r#"넭"#, r#"넭"#, r#"넭"#),
        (r#"넮"#, r#"넮"#, r#"넮"#, r#"넮"#, r#"넮"#),
        (r#"넯"#, r#"넯"#, r#"넯"#, r#"넯"#, r#"넯"#),
        (r#"넰"#, r#"넰"#, r#"넰"#, r#"넰"#, r#"넰"#),
        (r#"넱"#, r#"넱"#, r#"넱"#, r#"넱"#, r#"넱"#),
        (r#"넲"#, r#"넲"#, r#"넲"#, r#"넲"#, r#"넲"#),
        (r#"넳"#, r#"넳"#, r#"넳"#, r#"넳"#, r#"넳"#),
        (r#"넴"#, r#"넴"#, r#"넴"#, r#"넴"#, r#"넴"#),
        (r#"넵"#, r#"넵"#, r#"넵"#, r#"넵"#, r#"넵"#),
        (r#"넶"#, r#"넶"#, r#"넶"#, r#"넶"#, r#"넶"#),
        (r#"넷"#, r#"넷"#, r#"넷"#, r#"넷"#, r#"넷"#),
        (r#"넸"#, r#"넸"#, r#"넸"#, r#"넸"#, r#"넸"#),
        (r#"넹"#, r#"넹"#, r#"넹"#, r#"넹"#, r#"넹"#),
        (r#"넺"#, r#"넺"#, r#"넺"#, r#"넺"#, r#"넺"#),
        (r#"넻"#, r#"넻"#, r#"넻"#, r#"넻"#, r#"넻"#),
        (r#"넼"#, r#"넼"#, r#"넼"#, r#"넼"#, r#"넼"#),
        (r#"넽"#, r#"넽"#, r#"넽"#, r#"넽"#, r#"넽"#),
        (r#"넾"#, r#"넾"#, r#"넾"#, r#"넾"#, r#"넾"#),
        (r#"넿"#, r#"넿"#, r#"넿"#, r#"넿"#, r#"넿"#),
        (r#"녀"#, r#"녀"#, r#"녀"#, r#"녀"#, r#"녀"#),
        (r#"녁"#, r#"녁"#, r#"녁"#, r#"녁"#, r#"녁"#),
        (r#"녂"#, r#"녂"#, r#"녂"#, r#"녂"#, r#"녂"#),
        (r#"녃"#, r#"녃"#, r#"녃"#, r#"녃"#, r#"녃"#),
        (r#"년"#, r#"년"#, r#"년"#, r#"년"#, r#"년"#),
        (r#"녅"#, r#"녅"#, r#"녅"#, r#"녅"#, r#"녅"#),
        (r#"녆"#, r#"녆"#, r#"녆"#, r#"녆"#, r#"녆"#),
        (r#"녇"#, r#"녇"#, r#"녇"#, r#"녇"#, r#"녇"#),
        (r#"녈"#, r#"녈"#, r#"녈"#, r#"녈"#, r#"녈"#),
        (r#"녉"#, r#"녉"#, r#"녉"#, r#"녉"#, r#"녉"#),
        (r#"녊"#, r#"녊"#, r#"녊"#, r#"녊"#, r#"녊"#),
        (r#"녋"#, r#"녋"#, r#"녋"#, r#"녋"#, r#"녋"#),
        (r#"녌"#, r#"녌"#, r#"녌"#, r#"녌"#, r#"녌"#),
        (r#"녍"#, r#"녍"#, r#"녍"#, r#"녍"#, r#"녍"#),
        (r#"녎"#, r#"녎"#, r#"녎"#, r#"녎"#, r#"녎"#),
        (r#"녏"#, r#"녏"#, r#"녏"#, r#"녏"#, r#"녏"#),
        (r#"념"#, r#"념"#, r#"념"#, r#"념"#, r#"념"#),
        (r#"녑"#, r#"녑"#, r#"녑"#, r#"녑"#, r#"녑"#),
        (r#"녒"#, r#"녒"#, r#"녒"#, r#"녒"#, r#"녒"#),
        (r#"녓"#, r#"녓"#, r#"녓"#, r#"녓"#, r#"녓"#),
        (r#"녔"#, r#"녔"#, r#"녔"#, r#"녔"#, r#"녔"#),
        (r#"녕"#, r#"녕"#, r#"녕"#, r#"녕"#, r#"녕"#),
        (r#"녖"#, r#"녖"#, r#"녖"#, r#"녖"#, r#"녖"#),
        (r#"녗"#, r#"녗"#, r#"녗"#, r#"녗"#, r#"녗"#),
        (r#"녘"#, r#"녘"#, r#"녘"#, r#"녘"#, r#"녘"#),
        (r#"녙"#, r#"녙"#, r#"녙"#, r#"녙"#, r#"녙"#),
        (r#"녚"#, r#"녚"#, r#"녚"#, r#"녚"#, r#"녚"#),
        (r#"녛"#, r#"녛"#, r#"녛"#, r#"녛"#, r#"녛"#),
        (r#"녜"#, r#"녜"#, r#"녜"#, r#"녜"#, r#"녜"#),
        (r#"녝"#, r#"녝"#, r#"녝"#, r#"녝"#, r#"녝"#),
        (r#"녞"#, r#"녞"#, r#"녞"#, r#"녞"#, r#"녞"#),
        (r#"녟"#, r#"녟"#, r#"녟"#, r#"녟"#, r#"녟"#),
        (r#"녠"#, r#"녠"#, r#"녠"#, r#"녠"#, r#"녠"#),
        (r#"녡"#, r#"녡"#, r#"녡"#, r#"녡"#, r#"녡"#),
        (r#"녢"#, r#"녢"#, r#"녢"#, r#"녢"#, r#"녢"#),
        (r#"녣"#, r#"녣"#, r#"녣"#, r#"녣"#, r#"녣"#),
        (r#"녤"#, r#"녤"#, r#"녤"#, r#"녤"#, r#"녤"#),
        (r#"녥"#, r#"녥"#, r#"녥"#, r#"녥"#, r#"녥"#),
        (r#"녦"#, r#"녦"#, r#"녦"#, r#"녦"#, r#"녦"#),
        (r#"녧"#, r#"녧"#, r#"녧"#, r#"녧"#, r#"녧"#),
        (r#"녨"#, r#"녨"#, r#"녨"#, r#"녨"#, r#"녨"#),
        (r#"녩"#, r#"녩"#, r#"녩"#, r#"녩"#, r#"녩"#),
        (r#"녪"#, r#"녪"#, r#"녪"#, r#"녪"#, r#"녪"#),
        (r#"녫"#, r#"녫"#, r#"녫"#, r#"녫"#, r#"녫"#),
        (r#"녬"#, r#"녬"#, r#"녬"#, r#"녬"#, r#"녬"#),
        (r#"녭"#, r#"녭"#, r#"녭"#, r#"녭"#, r#"녭"#),
        (r#"녮"#, r#"녮"#, r#"녮"#, r#"녮"#, r#"녮"#),
        (r#"녯"#, r#"녯"#, r#"녯"#, r#"녯"#, r#"녯"#),
        (r#"녰"#, r#"녰"#, r#"녰"#, r#"녰"#, r#"녰"#),
        (r#"녱"#, r#"녱"#, r#"녱"#, r#"녱"#, r#"녱"#),
        (r#"녲"#, r#"녲"#, r#"녲"#, r#"녲"#, r#"녲"#),
        (r#"녳"#, r#"녳"#, r#"녳"#, r#"녳"#, r#"녳"#),
        (r#"녴"#, r#"녴"#, r#"녴"#, r#"녴"#, r#"녴"#),
        (r#"녵"#, r#"녵"#, r#"녵"#, r#"녵"#, r#"녵"#),
        (r#"녶"#, r#"녶"#, r#"녶"#, r#"녶"#, r#"녶"#),
        (r#"녷"#, r#"녷"#, r#"녷"#, r#"녷"#, r#"녷"#),
        (r#"노"#, r#"노"#, r#"노"#, r#"노"#, r#"노"#),
        (r#"녹"#, r#"녹"#, r#"녹"#, r#"녹"#, r#"녹"#),
        (r#"녺"#, r#"녺"#, r#"녺"#, r#"녺"#, r#"녺"#),
        (r#"녻"#, r#"녻"#, r#"녻"#, r#"녻"#, r#"녻"#),
        (r#"논"#, r#"논"#, r#"논"#, r#"논"#, r#"논"#),
        (r#"녽"#, r#"녽"#, r#"녽"#, r#"녽"#, r#"녽"#),
        (r#"녾"#, r#"녾"#, r#"녾"#, r#"녾"#, r#"녾"#),
        (r#"녿"#, r#"녿"#, r#"녿"#, r#"녿"#, r#"녿"#),
        (r#"놀"#, r#"놀"#, r#"놀"#, r#"놀"#, r#"놀"#),
        (r#"놁"#, r#"놁"#, r#"놁"#, r#"놁"#, r#"놁"#),
        (r#"놂"#, r#"놂"#, r#"놂"#, r#"놂"#, r#"놂"#),
        (r#"놃"#, r#"놃"#, r#"놃"#, r#"놃"#, r#"놃"#),
        (r#"놄"#, r#"놄"#, r#"놄"#, r#"놄"#, r#"놄"#),
        (r#"놅"#, r#"놅"#, r#"놅"#, r#"놅"#, r#"놅"#),
        (r#"놆"#, r#"놆"#, r#"놆"#, r#"놆"#, r#"놆"#),
        (r#"놇"#, r#"놇"#, r#"놇"#, r#"놇"#, r#"놇"#),
        (r#"놈"#, r#"놈"#, r#"놈"#, r#"놈"#, r#"놈"#),
        (r#"놉"#, r#"놉"#, r#"놉"#, r#"놉"#, r#"놉"#),
        (r#"놊"#, r#"놊"#, r#"놊"#, r#"놊"#, r#"놊"#),
        (r#"놋"#, r#"놋"#, r#"놋"#, r#"놋"#, r#"놋"#),
        (r#"놌"#, r#"놌"#, r#"놌"#, r#"놌"#, r#"놌"#),
        (r#"농"#, r#"농"#, r#"농"#, r#"농"#, r#"농"#),
        (r#"놎"#, r#"놎"#, r#"놎"#, r#"놎"#, r#"놎"#),
        (r#"놏"#, r#"놏"#, r#"놏"#, r#"놏"#, r#"놏"#),
        (r#"놐"#, r#"놐"#, r#"놐"#, r#"놐"#, r#"놐"#),
        (r#"놑"#, r#"놑"#, r#"놑"#, r#"놑"#, r#"놑"#),
        (r#"높"#, r#"높"#, r#"높"#, r#"높"#, r#"높"#),
        (r#"놓"#, r#"놓"#, r#"놓"#, r#"놓"#, r#"놓"#),
        (r#"놔"#, r#"놔"#, r#"놔"#, r#"놔"#, r#"놔"#),
        (r#"놕"#, r#"놕"#, r#"놕"#, r#"놕"#, r#"놕"#),
        (r#"놖"#, r#"놖"#, r#"놖"#, r#"놖"#, r#"놖"#),
        (r#"놗"#, r#"놗"#, r#"놗"#, r#"놗"#, r#"놗"#),
        (r#"놘"#, r#"놘"#, r#"놘"#, r#"놘"#, r#"놘"#),
        (r#"놙"#, r#"놙"#, r#"놙"#, r#"놙"#, r#"놙"#),
        (r#"놚"#, r#"놚"#, r#"놚"#, r#"놚"#, r#"놚"#),
        (r#"놛"#, r#"놛"#, r#"놛"#, r#"놛"#, r#"놛"#),
        (r#"놜"#, r#"놜"#, r#"놜"#, r#"놜"#, r#"놜"#),
        (r#"놝"#, r#"놝"#, r#"놝"#, r#"놝"#, r#"놝"#),
        (r#"놞"#, r#"놞"#, r#"놞"#, r#"놞"#, r#"놞"#),
        (r#"놟"#, r#"놟"#, r#"놟"#, r#"놟"#, r#"놟"#),
        (r#"놠"#, r#"놠"#, r#"놠"#, r#"놠"#, r#"놠"#),
        (r#"놡"#, r#"놡"#, r#"놡"#, r#"놡"#, r#"놡"#),
        (r#"놢"#, r#"놢"#, r#"놢"#, r#"놢"#, r#"놢"#),
        (r#"놣"#, r#"놣"#, r#"놣"#, r#"놣"#, r#"놣"#),
        (r#"놤"#, r#"놤"#, r#"놤"#, r#"놤"#, r#"놤"#),
        (r#"놥"#, r#"놥"#, r#"놥"#, r#"놥"#, r#"놥"#),
        (r#"놦"#, r#"놦"#, r#"놦"#, r#"놦"#, r#"놦"#),
        (r#"놧"#, r#"놧"#, r#"놧"#, r#"놧"#, r#"놧"#),
        (r#"놨"#, r#"놨"#, r#"놨"#, r#"놨"#, r#"놨"#),
        (r#"놩"#, r#"놩"#, r#"놩"#, r#"놩"#, r#"놩"#),
        (r#"놪"#, r#"놪"#, r#"놪"#, r#"놪"#, r#"놪"#),
        (r#"놫"#, r#"놫"#, r#"놫"#, r#"놫"#, r#"놫"#),
        (r#"놬"#, r#"놬"#, r#"놬"#, r#"놬"#, r#"놬"#),
        (r#"놭"#, r#"놭"#, r#"놭"#, r#"놭"#, r#"놭"#),
        (r#"놮"#, r#"놮"#, r#"놮"#, r#"놮"#, r#"놮"#),
        (r#"놯"#, r#"놯"#, r#"놯"#, r#"놯"#, r#"놯"#),
        (r#"놰"#, r#"놰"#, r#"놰"#, r#"놰"#, r#"놰"#),
        (r#"놱"#, r#"놱"#, r#"놱"#, r#"놱"#, r#"놱"#),
        (r#"놲"#, r#"놲"#, r#"놲"#, r#"놲"#, r#"놲"#),
        (r#"놳"#, r#"놳"#, r#"놳"#, r#"놳"#, r#"놳"#),
        (r#"놴"#, r#"놴"#, r#"놴"#, r#"놴"#, r#"놴"#),
        (r#"놵"#, r#"놵"#, r#"놵"#, r#"놵"#, r#"놵"#),
        (r#"놶"#, r#"놶"#, r#"놶"#, r#"놶"#, r#"놶"#),
        (r#"놷"#, r#"놷"#, r#"놷"#, r#"놷"#, r#"놷"#),
        (r#"놸"#, r#"놸"#, r#"놸"#, r#"놸"#, r#"놸"#),
        (r#"놹"#, r#"놹"#, r#"놹"#, r#"놹"#, r#"놹"#),
        (r#"놺"#, r#"놺"#, r#"놺"#, r#"놺"#, r#"놺"#),
        (r#"놻"#, r#"놻"#, r#"놻"#, r#"놻"#, r#"놻"#),
        (r#"놼"#, r#"놼"#, r#"놼"#, r#"놼"#, r#"놼"#),
        (r#"놽"#, r#"놽"#, r#"놽"#, r#"놽"#, r#"놽"#),
        (r#"놾"#, r#"놾"#, r#"놾"#, r#"놾"#, r#"놾"#),
        (r#"놿"#, r#"놿"#, r#"놿"#, r#"놿"#, r#"놿"#),
        (r#"뇀"#, r#"뇀"#, r#"뇀"#, r#"뇀"#, r#"뇀"#),
        (r#"뇁"#, r#"뇁"#, r#"뇁"#, r#"뇁"#, r#"뇁"#),
        (r#"뇂"#, r#"뇂"#, r#"뇂"#, r#"뇂"#, r#"뇂"#),
        (r#"뇃"#, r#"뇃"#, r#"뇃"#, r#"뇃"#, r#"뇃"#),
        (r#"뇄"#, r#"뇄"#, r#"뇄"#, r#"뇄"#, r#"뇄"#),
        (r#"뇅"#, r#"뇅"#, r#"뇅"#, r#"뇅"#, r#"뇅"#),
        (r#"뇆"#, r#"뇆"#, r#"뇆"#, r#"뇆"#, r#"뇆"#),
        (r#"뇇"#, r#"뇇"#, r#"뇇"#, r#"뇇"#, r#"뇇"#),
        (r#"뇈"#, r#"뇈"#, r#"뇈"#, r#"뇈"#, r#"뇈"#),
        (r#"뇉"#, r#"뇉"#, r#"뇉"#, r#"뇉"#, r#"뇉"#),
        (r#"뇊"#, r#"뇊"#, r#"뇊"#, r#"뇊"#, r#"뇊"#),
        (r#"뇋"#, r#"뇋"#, r#"뇋"#, r#"뇋"#, r#"뇋"#),
        (r#"뇌"#, r#"뇌"#, r#"뇌"#, r#"뇌"#, r#"뇌"#),
        (r#"뇍"#, r#"뇍"#, r#"뇍"#, r#"뇍"#, r#"뇍"#),
        (r#"뇎"#, r#"뇎"#, r#"뇎"#, r#"뇎"#, r#"뇎"#),
        (r#"뇏"#, r#"뇏"#, r#"뇏"#, r#"뇏"#, r#"뇏"#),
        (r#"뇐"#, r#"뇐"#, r#"뇐"#, r#"뇐"#, r#"뇐"#),
        (r#"뇑"#, r#"뇑"#, r#"뇑"#, r#"뇑"#, r#"뇑"#),
        (r#"뇒"#, r#"뇒"#, r#"뇒"#, r#"뇒"#, r#"뇒"#),
        (r#"뇓"#, r#"뇓"#, r#"뇓"#, r#"뇓"#, r#"뇓"#),
        (r#"뇔"#, r#"뇔"#, r#"뇔"#, r#"뇔"#, r#"뇔"#),
        (r#"뇕"#, r#"뇕"#, r#"뇕"#, r#"뇕"#, r#"뇕"#),
        (r#"뇖"#, r#"뇖"#, r#"뇖"#, r#"뇖"#, r#"뇖"#),
        (r#"뇗"#, r#"뇗"#, r#"뇗"#, r#"뇗"#, r#"뇗"#),
        (r#"뇘"#, r#"뇘"#, r#"뇘"#, r#"뇘"#, r#"뇘"#),
        (r#"뇙"#, r#"뇙"#, r#"뇙"#, r#"뇙"#, r#"뇙"#),
        (r#"뇚"#, r#"뇚"#, r#"뇚"#, r#"뇚"#, r#"뇚"#),
        (r#"뇛"#, r#"뇛"#, r#"뇛"#, r#"뇛"#, r#"뇛"#),
        (r#"뇜"#, r#"뇜"#, r#"뇜"#, r#"뇜"#, r#"뇜"#),
        (r#"뇝"#, r#"뇝"#, r#"뇝"#, r#"뇝"#, r#"뇝"#),
        (r#"뇞"#, r#"뇞"#, r#"뇞"#, r#"뇞"#, r#"뇞"#),
        (r#"뇟"#, r#"뇟"#, r#"뇟"#, r#"뇟"#, r#"뇟"#),
        (r#"뇠"#, r#"뇠"#, r#"뇠"#, r#"뇠"#, r#"뇠"#),
        (r#"뇡"#, r#"뇡"#, r#"뇡"#, r#"뇡"#, r#"뇡"#),
        (r#"뇢"#, r#"뇢"#, r#"뇢"#, r#"뇢"#, r#"뇢"#),
        (r#"뇣"#, r#"뇣"#, r#"뇣"#, r#"뇣"#, r#"뇣"#),
        (r#"뇤"#, r#"뇤"#, r#"뇤"#, r#"뇤"#, r#"뇤"#),
        (r#"뇥"#, r#"뇥"#, r#"뇥"#, r#"뇥"#, r#"뇥"#),
        (r#"뇦"#, r#"뇦"#, r#"뇦"#, r#"뇦"#, r#"뇦"#),
        (r#"뇧"#, r#"뇧"#, r#"뇧"#, r#"뇧"#, r#"뇧"#),
        (r#"뇨"#, r#"뇨"#, r#"뇨"#, r#"뇨"#, r#"뇨"#),
        (r#"뇩"#, r#"뇩"#, r#"뇩"#, r#"뇩"#, r#"뇩"#),
        (r#"뇪"#, r#"뇪"#, r#"뇪"#, r#"뇪"#, r#"뇪"#),
        (r#"뇫"#, r#"뇫"#, r#"뇫"#, r#"뇫"#, r#"뇫"#),
        (r#"뇬"#, r#"뇬"#, r#"뇬"#, r#"뇬"#, r#"뇬"#),
        (r#"뇭"#, r#"뇭"#, r#"뇭"#, r#"뇭"#, r#"뇭"#),
        (r#"뇮"#, r#"뇮"#, r#"뇮"#, r#"뇮"#, r#"뇮"#),
        (r#"뇯"#, r#"뇯"#, r#"뇯"#, r#"뇯"#, r#"뇯"#),
        (r#"뇰"#, r#"뇰"#, r#"뇰"#, r#"뇰"#, r#"뇰"#),
        (r#"뇱"#, r#"뇱"#, r#"뇱"#, r#"뇱"#, r#"뇱"#),
        (r#"뇲"#, r#"뇲"#, r#"뇲"#, r#"뇲"#, r#"뇲"#),
        (r#"뇳"#, r#"뇳"#, r#"뇳"#, r#"뇳"#, r#"뇳"#),
        (r#"뇴"#, r#"뇴"#, r#"뇴"#, r#"뇴"#, r#"뇴"#),
        (r#"뇵"#, r#"뇵"#, r#"뇵"#, r#"뇵"#, r#"뇵"#),
        (r#"뇶"#, r#"뇶"#, r#"뇶"#, r#"뇶"#, r#"뇶"#),
        (r#"뇷"#, r#"뇷"#, r#"뇷"#, r#"뇷"#, r#"뇷"#),
        (r#"뇸"#, r#"뇸"#, r#"뇸"#, r#"뇸"#, r#"뇸"#),
        (r#"뇹"#, r#"뇹"#, r#"뇹"#, r#"뇹"#, r#"뇹"#),
        (r#"뇺"#, r#"뇺"#, r#"뇺"#, r#"뇺"#, r#"뇺"#),
        (r#"뇻"#, r#"뇻"#, r#"뇻"#, r#"뇻"#, r#"뇻"#),
        (r#"뇼"#, r#"뇼"#, r#"뇼"#, r#"뇼"#, r#"뇼"#),
        (r#"뇽"#, r#"뇽"#, r#"뇽"#, r#"뇽"#, r#"뇽"#),
        (r#"뇾"#, r#"뇾"#, r#"뇾"#, r#"뇾"#, r#"뇾"#),
        (r#"뇿"#, r#"뇿"#, r#"뇿"#, r#"뇿"#, r#"뇿"#),
        (r#"눀"#, r#"눀"#, r#"눀"#, r#"눀"#, r#"눀"#),
        (r#"눁"#, r#"눁"#, r#"눁"#, r#"눁"#, r#"눁"#),
        (r#"눂"#, r#"눂"#, r#"눂"#, r#"눂"#, r#"눂"#),
        (r#"눃"#, r#"눃"#, r#"눃"#, r#"눃"#, r#"눃"#),
        (r#"누"#, r#"누"#, r#"누"#, r#"누"#, r#"누"#),
        (r#"눅"#, r#"눅"#, r#"눅"#, r#"눅"#, r#"눅"#),
        (r#"눆"#, r#"눆"#, r#"눆"#, r#"눆"#, r#"눆"#),
        (r#"눇"#, r#"눇"#, r#"눇"#, r#"눇"#, r#"눇"#),
        (r#"눈"#, r#"눈"#, r#"눈"#, r#"눈"#, r#"눈"#),
        (r#"눉"#, r#"눉"#, r#"눉"#, r#"눉"#, r#"눉"#),
        (r#"눊"#, r#"눊"#, r#"눊"#, r#"눊"#, r#"눊"#),
        (r#"눋"#, r#"눋"#, r#"눋"#, r#"눋"#, r#"눋"#),
        (r#"눌"#, r#"눌"#, r#"눌"#, r#"눌"#, r#"눌"#),
        (r#"눍"#, r#"눍"#, r#"눍"#, r#"눍"#, r#"눍"#),
        (r#"눎"#, r#"눎"#, r#"눎"#, r#"눎"#, r#"눎"#),
        (r#"눏"#, r#"눏"#, r#"눏"#, r#"눏"#, r#"눏"#),
        (r#"눐"#, r#"눐"#, r#"눐"#, r#"눐"#, r#"눐"#),
        (r#"눑"#, r#"눑"#, r#"눑"#, r#"눑"#, r#"눑"#),
        (r#"눒"#, r#"눒"#, r#"눒"#, r#"눒"#, r#"눒"#),
        (r#"눓"#, r#"눓"#, r#"눓"#, r#"눓"#, r#"눓"#),
        (r#"눔"#, r#"눔"#, r#"눔"#, r#"눔"#, r#"눔"#),
        (r#"눕"#, r#"눕"#, r#"눕"#, r#"눕"#, r#"눕"#),
        (r#"눖"#, r#"눖"#, r#"눖"#, r#"눖"#, r#"눖"#),
        (r#"눗"#, r#"눗"#, r#"눗"#, r#"눗"#, r#"눗"#),
        (r#"눘"#, r#"눘"#, r#"눘"#, r#"눘"#, r#"눘"#),
        (r#"눙"#, r#"눙"#, r#"눙"#, r#"눙"#, r#"눙"#),
        (r#"눚"#, r#"눚"#, r#"눚"#, r#"눚"#, r#"눚"#),
        (r#"눛"#, r#"눛"#, r#"눛"#, r#"눛"#, r#"눛"#),
        (r#"눜"#, r#"눜"#, r#"눜"#, r#"눜"#, r#"눜"#),
        (r#"눝"#, r#"눝"#, r#"눝"#, r#"눝"#, r#"눝"#),
        (r#"눞"#, r#"눞"#, r#"눞"#, r#"눞"#, r#"눞"#),
        (r#"눟"#, r#"눟"#, r#"눟"#, r#"눟"#, r#"눟"#),
        (r#"눠"#, r#"눠"#, r#"눠"#, r#"눠"#, r#"눠"#),
        (r#"눡"#, r#"눡"#, r#"눡"#, r#"눡"#, r#"눡"#),
        (r#"눢"#, r#"눢"#, r#"눢"#, r#"눢"#, r#"눢"#),
        (r#"눣"#, r#"눣"#, r#"눣"#, r#"눣"#, r#"눣"#),
        (r#"눤"#, r#"눤"#, r#"눤"#, r#"눤"#, r#"눤"#),
        (r#"눥"#, r#"눥"#, r#"눥"#, r#"눥"#, r#"눥"#),
        (r#"눦"#, r#"눦"#, r#"눦"#, r#"눦"#, r#"눦"#),
        (r#"눧"#, r#"눧"#, r#"눧"#, r#"눧"#, r#"눧"#),
        (r#"눨"#, r#"눨"#, r#"눨"#, r#"눨"#, r#"눨"#),
        (r#"눩"#, r#"눩"#, r#"눩"#, r#"눩"#, r#"눩"#),
        (r#"눪"#, r#"눪"#, r#"눪"#, r#"눪"#, r#"눪"#),
        (r#"눫"#, r#"눫"#, r#"눫"#, r#"눫"#, r#"눫"#),
        (r#"눬"#, r#"눬"#, r#"눬"#, r#"눬"#, r#"눬"#),
        (r#"눭"#, r#"눭"#, r#"눭"#, r#"눭"#, r#"눭"#),
        (r#"눮"#, r#"눮"#, r#"눮"#, r#"눮"#, r#"눮"#),
        (r#"눯"#, r#"눯"#, r#"눯"#, r#"눯"#, r#"눯"#),
        (r#"눰"#, r#"눰"#, r#"눰"#, r#"눰"#, r#"눰"#),
        (r#"눱"#, r#"눱"#, r#"눱"#, r#"눱"#, r#"눱"#),
        (r#"눲"#, r#"눲"#, r#"눲"#, r#"눲"#, r#"눲"#),
        (r#"눳"#, r#"눳"#, r#"눳"#, r#"눳"#, r#"눳"#),
        (r#"눴"#, r#"눴"#, r#"눴"#, r#"눴"#, r#"눴"#),
        (r#"눵"#, r#"눵"#, r#"눵"#, r#"눵"#, r#"눵"#),
        (r#"눶"#, r#"눶"#, r#"눶"#, r#"눶"#, r#"눶"#),
        (r#"눷"#, r#"눷"#, r#"눷"#, r#"눷"#, r#"눷"#),
        (r#"눸"#, r#"눸"#, r#"눸"#, r#"눸"#, r#"눸"#),
        (r#"눹"#, r#"눹"#, r#"눹"#, r#"눹"#, r#"눹"#),
        (r#"눺"#, r#"눺"#, r#"눺"#, r#"눺"#, r#"눺"#),
        (r#"눻"#, r#"눻"#, r#"눻"#, r#"눻"#, r#"눻"#),
        (r#"눼"#, r#"눼"#, r#"눼"#, r#"눼"#, r#"눼"#),
        (r#"눽"#, r#"눽"#, r#"눽"#, r#"눽"#, r#"눽"#),
        (r#"눾"#, r#"눾"#, r#"눾"#, r#"눾"#, r#"눾"#),
        (r#"눿"#, r#"눿"#, r#"눿"#, r#"눿"#, r#"눿"#),
        (r#"뉀"#, r#"뉀"#, r#"뉀"#, r#"뉀"#, r#"뉀"#),
        (r#"뉁"#, r#"뉁"#, r#"뉁"#, r#"뉁"#, r#"뉁"#),
        (r#"뉂"#, r#"뉂"#, r#"뉂"#, r#"뉂"#, r#"뉂"#),
        (r#"뉃"#, r#"뉃"#, r#"뉃"#, r#"뉃"#, r#"뉃"#),
        (r#"뉄"#, r#"뉄"#, r#"뉄"#, r#"뉄"#, r#"뉄"#),
        (r#"뉅"#, r#"뉅"#, r#"뉅"#, r#"뉅"#, r#"뉅"#),
        (r#"뉆"#, r#"뉆"#, r#"뉆"#, r#"뉆"#, r#"뉆"#),
        (r#"뉇"#, r#"뉇"#, r#"뉇"#, r#"뉇"#, r#"뉇"#),
        (r#"뉈"#, r#"뉈"#, r#"뉈"#, r#"뉈"#, r#"뉈"#),
        (r#"뉉"#, r#"뉉"#, r#"뉉"#, r#"뉉"#, r#"뉉"#),
        (r#"뉊"#, r#"뉊"#, r#"뉊"#, r#"뉊"#, r#"뉊"#),
        (r#"뉋"#, r#"뉋"#, r#"뉋"#, r#"뉋"#, r#"뉋"#),
        (r#"뉌"#, r#"뉌"#, r#"뉌"#, r#"뉌"#, r#"뉌"#),
        (r#"뉍"#, r#"뉍"#, r#"뉍"#, r#"뉍"#, r#"뉍"#),
        (r#"뉎"#, r#"뉎"#, r#"뉎"#, r#"뉎"#, r#"뉎"#),
        (r#"뉏"#, r#"뉏"#, r#"뉏"#, r#"뉏"#, r#"뉏"#),
        (r#"뉐"#, r#"뉐"#, r#"뉐"#, r#"뉐"#, r#"뉐"#),
        (r#"뉑"#, r#"뉑"#, r#"뉑"#, r#"뉑"#, r#"뉑"#),
        (r#"뉒"#, r#"뉒"#, r#"뉒"#, r#"뉒"#, r#"뉒"#),
        (r#"뉓"#, r#"뉓"#, r#"뉓"#, r#"뉓"#, r#"뉓"#),
        (r#"뉔"#, r#"뉔"#, r#"뉔"#, r#"뉔"#, r#"뉔"#),
        (r#"뉕"#, r#"뉕"#, r#"뉕"#, r#"뉕"#, r#"뉕"#),
        (r#"뉖"#, r#"뉖"#, r#"뉖"#, r#"뉖"#, r#"뉖"#),
        (r#"뉗"#, r#"뉗"#, r#"뉗"#, r#"뉗"#, r#"뉗"#),
        (r#"뉘"#, r#"뉘"#, r#"뉘"#, r#"뉘"#, r#"뉘"#),
        (r#"뉙"#, r#"뉙"#, r#"뉙"#, r#"뉙"#, r#"뉙"#),
        (r#"뉚"#, r#"뉚"#, r#"뉚"#, r#"뉚"#, r#"뉚"#),
        (r#"뉛"#, r#"뉛"#, r#"뉛"#, r#"뉛"#, r#"뉛"#),
        (r#"뉜"#, r#"뉜"#, r#"뉜"#, r#"뉜"#, r#"뉜"#),
        (r#"뉝"#, r#"뉝"#, r#"뉝"#, r#"뉝"#, r#"뉝"#),
        (r#"뉞"#, r#"뉞"#, r#"뉞"#, r#"뉞"#, r#"뉞"#),
        (r#"뉟"#, r#"뉟"#, r#"뉟"#, r#"뉟"#, r#"뉟"#),
        (r#"뉠"#, r#"뉠"#, r#"뉠"#, r#"뉠"#, r#"뉠"#),
        (r#"뉡"#, r#"뉡"#, r#"뉡"#, r#"뉡"#, r#"뉡"#),
        (r#"뉢"#, r#"뉢"#, r#"뉢"#, r#"뉢"#, r#"뉢"#),
        (r#"뉣"#, r#"뉣"#, r#"뉣"#, r#"뉣"#, r#"뉣"#),
        (r#"뉤"#, r#"뉤"#, r#"뉤"#, r#"뉤"#, r#"뉤"#),
        (r#"뉥"#, r#"뉥"#, r#"뉥"#, r#"뉥"#, r#"뉥"#),
        (r#"뉦"#, r#"뉦"#, r#"뉦"#, r#"뉦"#, r#"뉦"#),
        (r#"뉧"#, r#"뉧"#, r#"뉧"#, r#"뉧"#, r#"뉧"#),
        (r#"뉨"#, r#"뉨"#, r#"뉨"#, r#"뉨"#, r#"뉨"#),
        (r#"뉩"#, r#"뉩"#, r#"뉩"#, r#"뉩"#, r#"뉩"#),
        (r#"뉪"#, r#"뉪"#, r#"뉪"#, r#"뉪"#, r#"뉪"#),
        (r#"뉫"#, r#"뉫"#, r#"뉫"#, r#"뉫"#, r#"뉫"#),
        (r#"뉬"#, r#"뉬"#, r#"뉬"#, r#"뉬"#, r#"뉬"#),
        (r#"뉭"#, r#"뉭"#, r#"뉭"#, r#"뉭"#, r#"뉭"#),
        (r#"뉮"#, r#"뉮"#, r#"뉮"#, r#"뉮"#, r#"뉮"#),
        (r#"뉯"#, r#"뉯"#, r#"뉯"#, r#"뉯"#, r#"뉯"#),
        (r#"뉰"#, r#"뉰"#, r#"뉰"#, r#"뉰"#, r#"뉰"#),
        (r#"뉱"#, r#"뉱"#, r#"뉱"#, r#"뉱"#, r#"뉱"#),
        (r#"뉲"#, r#"뉲"#, r#"뉲"#, r#"뉲"#, r#"뉲"#),
        (r#"뉳"#, r#"뉳"#, r#"뉳"#, r#"뉳"#, r#"뉳"#),
        (r#"뉴"#, r#"뉴"#, r#"뉴"#, r#"뉴"#, r#"뉴"#),
        (r#"뉵"#, r#"뉵"#, r#"뉵"#, r#"뉵"#, r#"뉵"#),
        (r#"뉶"#, r#"뉶"#, r#"뉶"#, r#"뉶"#, r#"뉶"#),
        (r#"뉷"#, r#"뉷"#, r#"뉷"#, r#"뉷"#, r#"뉷"#),
        (r#"뉸"#, r#"뉸"#, r#"뉸"#, r#"뉸"#, r#"뉸"#),
        (r#"뉹"#, r#"뉹"#, r#"뉹"#, r#"뉹"#, r#"뉹"#),
        (r#"뉺"#, r#"뉺"#, r#"뉺"#, r#"뉺"#, r#"뉺"#),
        (r#"뉻"#, r#"뉻"#, r#"뉻"#, r#"뉻"#, r#"뉻"#),
        (r#"뉼"#, r#"뉼"#, r#"뉼"#, r#"뉼"#, r#"뉼"#),
        (r#"뉽"#, r#"뉽"#, r#"뉽"#, r#"뉽"#, r#"뉽"#),
        (r#"뉾"#, r#"뉾"#, r#"뉾"#, r#"뉾"#, r#"뉾"#),
        (r#"뉿"#, r#"뉿"#, r#"뉿"#, r#"뉿"#, r#"뉿"#),
        (r#"늀"#, r#"늀"#, r#"늀"#, r#"늀"#, r#"늀"#),
        (r#"늁"#, r#"늁"#, r#"늁"#, r#"늁"#, r#"늁"#),
        (r#"늂"#, r#"늂"#, r#"늂"#, r#"늂"#, r#"늂"#),
        (r#"늃"#, r#"늃"#, r#"늃"#, r#"늃"#, r#"늃"#),
        (r#"늄"#, r#"늄"#, r#"늄"#, r#"늄"#, r#"늄"#),
        (r#"늅"#, r#"늅"#, r#"늅"#, r#"늅"#, r#"늅"#),
        (r#"늆"#, r#"늆"#, r#"늆"#, r#"늆"#, r#"늆"#),
        (r#"늇"#, r#"늇"#, r#"늇"#, r#"늇"#, r#"늇"#),
        (r#"늈"#, r#"늈"#, r#"늈"#, r#"늈"#, r#"늈"#),
        (r#"늉"#, r#"늉"#, r#"늉"#, r#"늉"#, r#"늉"#),
        (r#"늊"#, r#"늊"#, r#"늊"#, r#"늊"#, r#"늊"#),
        (r#"늋"#, r#"늋"#, r#"늋"#, r#"늋"#, r#"늋"#),
        (r#"늌"#, r#"늌"#, r#"늌"#, r#"늌"#, r#"늌"#),
        (r#"늍"#, r#"늍"#, r#"늍"#, r#"늍"#, r#"늍"#),
        (r#"늎"#, r#"늎"#, r#"늎"#, r#"늎"#, r#"늎"#),
        (r#"늏"#, r#"늏"#, r#"늏"#, r#"늏"#, r#"늏"#),
        (r#"느"#, r#"느"#, r#"느"#, r#"느"#, r#"느"#),
        (r#"늑"#, r#"늑"#, r#"늑"#, r#"늑"#, r#"늑"#),
        (r#"늒"#, r#"늒"#, r#"늒"#, r#"늒"#, r#"늒"#),
        (r#"늓"#, r#"늓"#, r#"늓"#, r#"늓"#, r#"늓"#),
        (r#"는"#, r#"는"#, r#"는"#, r#"는"#, r#"는"#),
        (r#"늕"#, r#"늕"#, r#"늕"#, r#"늕"#, r#"늕"#),
        (r#"늖"#, r#"늖"#, r#"늖"#, r#"늖"#, r#"늖"#),
        (r#"늗"#, r#"늗"#, r#"늗"#, r#"늗"#, r#"늗"#),
        (r#"늘"#, r#"늘"#, r#"늘"#, r#"늘"#, r#"늘"#),
        (r#"늙"#, r#"늙"#, r#"늙"#, r#"늙"#, r#"늙"#),
        (r#"늚"#, r#"늚"#, r#"늚"#, r#"늚"#, r#"늚"#),
        (r#"늛"#, r#"늛"#, r#"늛"#, r#"늛"#, r#"늛"#),
        (r#"늜"#, r#"늜"#, r#"늜"#, r#"늜"#, r#"늜"#),
        (r#"늝"#, r#"늝"#, r#"늝"#, r#"늝"#, r#"늝"#),
        (r#"늞"#, r#"늞"#, r#"늞"#, r#"늞"#, r#"늞"#),
        (r#"늟"#, r#"늟"#, r#"늟"#, r#"늟"#, r#"늟"#),
        (r#"늠"#, r#"늠"#, r#"늠"#, r#"늠"#, r#"늠"#),
        (r#"늡"#, r#"늡"#, r#"늡"#, r#"늡"#, r#"늡"#),
        (r#"늢"#, r#"늢"#, r#"늢"#, r#"늢"#, r#"늢"#),
        (r#"늣"#, r#"늣"#, r#"늣"#, r#"늣"#, r#"늣"#),
        (r#"늤"#, r#"늤"#, r#"늤"#, r#"늤"#, r#"늤"#),
        (r#"능"#, r#"능"#, r#"능"#, r#"능"#, r#"능"#),
        (r#"늦"#, r#"늦"#, r#"늦"#, r#"늦"#, r#"늦"#),
        (r#"늧"#, r#"늧"#, r#"늧"#, r#"늧"#, r#"늧"#),
        (r#"늨"#, r#"늨"#, r#"늨"#, r#"늨"#, r#"늨"#),
        (r#"늩"#, r#"늩"#, r#"늩"#, r#"늩"#, r#"늩"#),
        (r#"늪"#, r#"늪"#, r#"늪"#, r#"늪"#, r#"늪"#),
        (r#"늫"#, r#"늫"#, r#"늫"#, r#"늫"#, r#"늫"#),
        (r#"늬"#, r#"늬"#, r#"늬"#, r#"늬"#, r#"늬"#),
        (r#"늭"#, r#"늭"#, r#"늭"#, r#"늭"#, r#"늭"#),
        (r#"늮"#, r#"늮"#, r#"늮"#, r#"늮"#, r#"늮"#),
        (r#"늯"#, r#"늯"#, r#"늯"#, r#"늯"#, r#"늯"#),
        (r#"늰"#, r#"늰"#, r#"늰"#, r#"늰"#, r#"늰"#),
        (r#"늱"#, r#"늱"#, r#"늱"#, r#"늱"#, r#"늱"#),
        (r#"늲"#, r#"늲"#, r#"늲"#, r#"늲"#, r#"늲"#),
        (r#"늳"#, r#"늳"#, r#"늳"#, r#"늳"#, r#"늳"#),
        (r#"늴"#, r#"늴"#, r#"늴"#, r#"늴"#, r#"늴"#),
        (r#"늵"#, r#"늵"#, r#"늵"#, r#"늵"#, r#"늵"#),
        (r#"늶"#, r#"늶"#, r#"늶"#, r#"늶"#, r#"늶"#),
        (r#"늷"#, r#"늷"#, r#"늷"#, r#"늷"#, r#"늷"#),
        (r#"늸"#, r#"늸"#, r#"늸"#, r#"늸"#, r#"늸"#),
        (r#"늹"#, r#"늹"#, r#"늹"#, r#"늹"#, r#"늹"#),
        (r#"늺"#, r#"늺"#, r#"늺"#, r#"늺"#, r#"늺"#),
        (r#"늻"#, r#"늻"#, r#"늻"#, r#"늻"#, r#"늻"#),
        (r#"늼"#, r#"늼"#, r#"늼"#, r#"늼"#, r#"늼"#),
        (r#"늽"#, r#"늽"#, r#"늽"#, r#"늽"#, r#"늽"#),
        (r#"늾"#, r#"늾"#, r#"늾"#, r#"늾"#, r#"늾"#),
        (r#"늿"#, r#"늿"#, r#"늿"#, r#"늿"#, r#"늿"#),
        (r#"닀"#, r#"닀"#, r#"닀"#, r#"닀"#, r#"닀"#),
        (r#"닁"#, r#"닁"#, r#"닁"#, r#"닁"#, r#"닁"#),
        (r#"닂"#, r#"닂"#, r#"닂"#, r#"닂"#, r#"닂"#),
        (r#"닃"#, r#"닃"#, r#"닃"#, r#"닃"#, r#"닃"#),
        (r#"닄"#, r#"닄"#, r#"닄"#, r#"닄"#, r#"닄"#),
        (r#"닅"#, r#"닅"#, r#"닅"#, r#"닅"#, r#"닅"#),
        (r#"닆"#, r#"닆"#, r#"닆"#, r#"닆"#, r#"닆"#),
        (r#"닇"#, r#"닇"#, r#"닇"#, r#"닇"#, r#"닇"#),
        (r#"니"#, r#"니"#, r#"니"#, r#"니"#, r#"니"#),
        (r#"닉"#, r#"닉"#, r#"닉"#, r#"닉"#, r#"닉"#),
        (r#"닊"#, r#"닊"#, r#"닊"#, r#"닊"#, r#"닊"#),
        (r#"닋"#, r#"닋"#, r#"닋"#, r#"닋"#, r#"닋"#),
        (r#"닌"#, r#"닌"#, r#"닌"#, r#"닌"#, r#"닌"#),
        (r#"닍"#, r#"닍"#, r#"닍"#, r#"닍"#, r#"닍"#),
        (r#"닎"#, r#"닎"#, r#"닎"#, r#"닎"#, r#"닎"#),
        (r#"닏"#, r#"닏"#, r#"닏"#, r#"닏"#, r#"닏"#),
        (r#"닐"#, r#"닐"#, r#"닐"#, r#"닐"#, r#"닐"#),
        (r#"닑"#, r#"닑"#, r#"닑"#, r#"닑"#, r#"닑"#),
        (r#"닒"#, r#"닒"#, r#"닒"#, r#"닒"#, r#"닒"#),
        (r#"닓"#, r#"닓"#, r#"닓"#, r#"닓"#, r#"닓"#),
        (r#"닔"#, r#"닔"#, r#"닔"#, r#"닔"#, r#"닔"#),
        (r#"닕"#, r#"닕"#, r#"닕"#, r#"닕"#, r#"닕"#),
        (r#"닖"#, r#"닖"#, r#"닖"#, r#"닖"#, r#"닖"#),
        (r#"닗"#, r#"닗"#, r#"닗"#, r#"닗"#, r#"닗"#),
        (r#"님"#, r#"님"#, r#"님"#, r#"님"#, r#"님"#),
        (r#"닙"#, r#"닙"#, r#"닙"#, r#"닙"#, r#"닙"#),
        (r#"닚"#, r#"닚"#, r#"닚"#, r#"닚"#, r#"닚"#),
        (r#"닛"#, r#"닛"#, r#"닛"#, r#"닛"#, r#"닛"#),
        (r#"닜"#, r#"닜"#, r#"닜"#, r#"닜"#, r#"닜"#),
        (r#"닝"#, r#"닝"#, r#"닝"#, r#"닝"#, r#"닝"#),
        (r#"닞"#, r#"닞"#, r#"닞"#, r#"닞"#, r#"닞"#),
        (r#"닟"#, r#"닟"#, r#"닟"#, r#"닟"#, r#"닟"#),
        (r#"닠"#, r#"닠"#, r#"닠"#, r#"닠"#, r#"닠"#),
        (r#"닡"#, r#"닡"#, r#"닡"#, r#"닡"#, r#"닡"#),
        (r#"닢"#, r#"닢"#, r#"닢"#, r#"닢"#, r#"닢"#),
        (r#"닣"#, r#"닣"#, r#"닣"#, r#"닣"#, r#"닣"#),
        (r#"다"#, r#"다"#, r#"다"#, r#"다"#, r#"다"#),
        (r#"닥"#, r#"닥"#, r#"닥"#, r#"닥"#, r#"닥"#),
        (r#"닦"#, r#"닦"#, r#"닦"#, r#"닦"#, r#"닦"#),
        (r#"닧"#, r#"닧"#, r#"닧"#, r#"닧"#, r#"닧"#),
        (r#"단"#, r#"단"#, r#"단"#, r#"단"#, r#"단"#),
        (r#"닩"#, r#"닩"#, r#"닩"#, r#"닩"#, r#"닩"#),
        (r#"닪"#, r#"닪"#, r#"닪"#, r#"닪"#, r#"닪"#),
        (r#"닫"#, r#"닫"#, r#"닫"#, r#"닫"#, r#"닫"#),
        (r#"달"#, r#"달"#, r#"달"#, r#"달"#, r#"달"#),
        (r#"닭"#, r#"닭"#, r#"닭"#, r#"닭"#, r#"닭"#),
        (r#"닮"#, r#"닮"#, r#"닮"#, r#"닮"#, r#"닮"#),
        (r#"닯"#, r#"닯"#, r#"닯"#, r#"닯"#, r#"닯"#),
        (r#"닰"#, r#"닰"#, r#"닰"#, r#"닰"#, r#"닰"#),
        (r#"닱"#, r#"닱"#, r#"닱"#, r#"닱"#, r#"닱"#),
        (r#"닲"#, r#"닲"#, r#"닲"#, r#"닲"#, r#"닲"#),
        (r#"닳"#, r#"닳"#, r#"닳"#, r#"닳"#, r#"닳"#),
        (r#"담"#, r#"담"#, r#"담"#, r#"담"#, r#"담"#),
        (r#"답"#, r#"답"#, r#"답"#, r#"답"#, r#"답"#),
        (r#"닶"#, r#"닶"#, r#"닶"#, r#"닶"#, r#"닶"#),
        (r#"닷"#, r#"닷"#, r#"닷"#, r#"닷"#, r#"닷"#),
        (r#"닸"#, r#"닸"#, r#"닸"#, r#"닸"#, r#"닸"#),
        (r#"당"#, r#"당"#, r#"당"#, r#"당"#, r#"당"#),
        (r#"닺"#, r#"닺"#, r#"닺"#, r#"닺"#, r#"닺"#),
        (r#"닻"#, r#"닻"#, r#"닻"#, r#"닻"#, r#"닻"#),
        (r#"닼"#, r#"닼"#, r#"닼"#, r#"닼"#, r#"닼"#),
        (r#"닽"#, r#"닽"#, r#"닽"#, r#"닽"#, r#"닽"#),
        (r#"닾"#, r#"닾"#, r#"닾"#, r#"닾"#, r#"닾"#),
        (r#"닿"#, r#"닿"#, r#"닿"#, r#"닿"#, r#"닿"#),
        (r#"대"#, r#"대"#, r#"대"#, r#"대"#, r#"대"#),
        (r#"댁"#, r#"댁"#, r#"댁"#, r#"댁"#, r#"댁"#),
        (r#"댂"#, r#"댂"#, r#"댂"#, r#"댂"#, r#"댂"#),
        (r#"댃"#, r#"댃"#, r#"댃"#, r#"댃"#, r#"댃"#),
        (r#"댄"#, r#"댄"#, r#"댄"#, r#"댄"#, r#"댄"#),
        (r#"댅"#, r#"댅"#, r#"댅"#, r#"댅"#, r#"댅"#),
        (r#"댆"#, r#"댆"#, r#"댆"#, r#"댆"#, r#"댆"#),
        (r#"댇"#, r#"댇"#, r#"댇"#, r#"댇"#, r#"댇"#),
        (r#"댈"#, r#"댈"#, r#"댈"#, r#"댈"#, r#"댈"#),
        (r#"댉"#, r#"댉"#, r#"댉"#, r#"댉"#, r#"댉"#),
        (r#"댊"#, r#"댊"#, r#"댊"#, r#"댊"#, r#"댊"#),
        (r#"댋"#, r#"댋"#, r#"댋"#, r#"댋"#, r#"댋"#),
        (r#"댌"#, r#"댌"#, r#"댌"#, r#"댌"#, r#"댌"#),
        (r#"댍"#, r#"댍"#, r#"댍"#, r#"댍"#, r#"댍"#),
        (r#"댎"#, r#"댎"#, r#"댎"#, r#"댎"#, r#"댎"#),
        (r#"댏"#, r#"댏"#, r#"댏"#, r#"댏"#, r#"댏"#),
        (r#"댐"#, r#"댐"#, r#"댐"#, r#"댐"#, r#"댐"#),
        (r#"댑"#, r#"댑"#, r#"댑"#, r#"댑"#, r#"댑"#),
        (r#"댒"#, r#"댒"#, r#"댒"#, r#"댒"#, r#"댒"#),
        (r#"댓"#, r#"댓"#, r#"댓"#, r#"댓"#, r#"댓"#),
        (r#"댔"#, r#"댔"#, r#"댔"#, r#"댔"#, r#"댔"#),
        (r#"댕"#, r#"댕"#, r#"댕"#, r#"댕"#, r#"댕"#),
        (r#"댖"#, r#"댖"#, r#"댖"#, r#"댖"#, r#"댖"#),
        (r#"댗"#, r#"댗"#, r#"댗"#, r#"댗"#, r#"댗"#),
        (r#"댘"#, r#"댘"#, r#"댘"#, r#"댘"#, r#"댘"#),
        (r#"댙"#, r#"댙"#, r#"댙"#, r#"댙"#, r#"댙"#),
        (r#"댚"#, r#"댚"#, r#"댚"#, r#"댚"#, r#"댚"#),
        (r#"댛"#, r#"댛"#, r#"댛"#, r#"댛"#, r#"댛"#),
        (r#"댜"#, r#"댜"#, r#"댜"#, r#"댜"#, r#"댜"#),
        (r#"댝"#, r#"댝"#, r#"댝"#, r#"댝"#, r#"댝"#),
        (r#"댞"#, r#"댞"#, r#"댞"#, r#"댞"#, r#"댞"#),
        (r#"댟"#, r#"댟"#, r#"댟"#, r#"댟"#, r#"댟"#),
        (r#"댠"#, r#"댠"#, r#"댠"#, r#"댠"#, r#"댠"#),
        (r#"댡"#, r#"댡"#, r#"댡"#, r#"댡"#, r#"댡"#),
        (r#"댢"#, r#"댢"#, r#"댢"#, r#"댢"#, r#"댢"#),
        (r#"댣"#, r#"댣"#, r#"댣"#, r#"댣"#, r#"댣"#),
        (r#"댤"#, r#"댤"#, r#"댤"#, r#"댤"#, r#"댤"#),
        (r#"댥"#, r#"댥"#, r#"댥"#, r#"댥"#, r#"댥"#),
        (r#"댦"#, r#"댦"#, r#"댦"#, r#"댦"#, r#"댦"#),
        (r#"댧"#, r#"댧"#, r#"댧"#, r#"댧"#, r#"댧"#),
        (r#"댨"#, r#"댨"#, r#"댨"#, r#"댨"#, r#"댨"#),
        (r#"댩"#, r#"댩"#, r#"댩"#, r#"댩"#, r#"댩"#),
        (r#"댪"#, r#"댪"#, r#"댪"#, r#"댪"#, r#"댪"#),
        (r#"댫"#, r#"댫"#, r#"댫"#, r#"댫"#, r#"댫"#),
        (r#"댬"#, r#"댬"#, r#"댬"#, r#"댬"#, r#"댬"#),
        (r#"댭"#, r#"댭"#, r#"댭"#, r#"댭"#, r#"댭"#),
        (r#"댮"#, r#"댮"#, r#"댮"#, r#"댮"#, r#"댮"#),
        (r#"댯"#, r#"댯"#, r#"댯"#, r#"댯"#, r#"댯"#),
        (r#"댰"#, r#"댰"#, r#"댰"#, r#"댰"#, r#"댰"#),
        (r#"댱"#, r#"댱"#, r#"댱"#, r#"댱"#, r#"댱"#),
        (r#"댲"#, r#"댲"#, r#"댲"#, r#"댲"#, r#"댲"#),
        (r#"댳"#, r#"댳"#, r#"댳"#, r#"댳"#, r#"댳"#),
        (r#"댴"#, r#"댴"#, r#"댴"#, r#"댴"#, r#"댴"#),
        (r#"댵"#, r#"댵"#, r#"댵"#, r#"댵"#, r#"댵"#),
        (r#"댶"#, r#"댶"#, r#"댶"#, r#"댶"#, r#"댶"#),
        (r#"댷"#, r#"댷"#, r#"댷"#, r#"댷"#, r#"댷"#),
        (r#"댸"#, r#"댸"#, r#"댸"#, r#"댸"#, r#"댸"#),
        (r#"댹"#, r#"댹"#, r#"댹"#, r#"댹"#, r#"댹"#),
        (r#"댺"#, r#"댺"#, r#"댺"#, r#"댺"#, r#"댺"#),
        (r#"댻"#, r#"댻"#, r#"댻"#, r#"댻"#, r#"댻"#),
        (r#"댼"#, r#"댼"#, r#"댼"#, r#"댼"#, r#"댼"#),
        (r#"댽"#, r#"댽"#, r#"댽"#, r#"댽"#, r#"댽"#),
        (r#"댾"#, r#"댾"#, r#"댾"#, r#"댾"#, r#"댾"#),
        (r#"댿"#, r#"댿"#, r#"댿"#, r#"댿"#, r#"댿"#),
        (r#"덀"#, r#"덀"#, r#"덀"#, r#"덀"#, r#"덀"#),
        (r#"덁"#, r#"덁"#, r#"덁"#, r#"덁"#, r#"덁"#),
        (r#"덂"#, r#"덂"#, r#"덂"#, r#"덂"#, r#"덂"#),
        (r#"덃"#, r#"덃"#, r#"덃"#, r#"덃"#, r#"덃"#),
        (r#"덄"#, r#"덄"#, r#"덄"#, r#"덄"#, r#"덄"#),
        (r#"덅"#, r#"덅"#, r#"덅"#, r#"덅"#, r#"덅"#),
        (r#"덆"#, r#"덆"#, r#"덆"#, r#"덆"#, r#"덆"#),
        (r#"덇"#, r#"덇"#, r#"덇"#, r#"덇"#, r#"덇"#),
        (r#"덈"#, r#"덈"#, r#"덈"#, r#"덈"#, r#"덈"#),
        (r#"덉"#, r#"덉"#, r#"덉"#, r#"덉"#, r#"덉"#),
        (r#"덊"#, r#"덊"#, r#"덊"#, r#"덊"#, r#"덊"#),
        (r#"덋"#, r#"덋"#, r#"덋"#, r#"덋"#, r#"덋"#),
        (r#"덌"#, r#"덌"#, r#"덌"#, r#"덌"#, r#"덌"#),
        (r#"덍"#, r#"덍"#, r#"덍"#, r#"덍"#, r#"덍"#),
        (r#"덎"#, r#"덎"#, r#"덎"#, r#"덎"#, r#"덎"#),
        (r#"덏"#, r#"덏"#, r#"덏"#, r#"덏"#, r#"덏"#),
        (r#"덐"#, r#"덐"#, r#"덐"#, r#"덐"#, r#"덐"#),
        (r#"덑"#, r#"덑"#, r#"덑"#, r#"덑"#, r#"덑"#),
        (r#"덒"#, r#"덒"#, r#"덒"#, r#"덒"#, r#"덒"#),
        (r#"덓"#, r#"덓"#, r#"덓"#, r#"덓"#, r#"덓"#),
        (r#"더"#, r#"더"#, r#"더"#, r#"더"#, r#"더"#),
        (r#"덕"#, r#"덕"#, r#"덕"#, r#"덕"#, r#"덕"#),
        (r#"덖"#, r#"덖"#, r#"덖"#, r#"덖"#, r#"덖"#),
        (r#"덗"#, r#"덗"#, r#"덗"#, r#"덗"#, r#"덗"#),
        (r#"던"#, r#"던"#, r#"던"#, r#"던"#, r#"던"#),
        (r#"덙"#, r#"덙"#, r#"덙"#, r#"덙"#, r#"덙"#),
        (r#"덚"#, r#"덚"#, r#"덚"#, r#"덚"#, r#"덚"#),
        (r#"덛"#, r#"덛"#, r#"덛"#, r#"덛"#, r#"덛"#),
        (r#"덜"#, r#"덜"#, r#"덜"#, r#"덜"#, r#"덜"#),
        (r#"덝"#, r#"덝"#, r#"덝"#, r#"덝"#, r#"덝"#),
        (r#"덞"#, r#"덞"#, r#"덞"#, r#"덞"#, r#"덞"#),
        (r#"덟"#, r#"덟"#, r#"덟"#, r#"덟"#, r#"덟"#),
        (r#"덠"#, r#"덠"#, r#"덠"#, r#"덠"#, r#"덠"#),
        (r#"덡"#, r#"덡"#, r#"덡"#, r#"덡"#, r#"덡"#),
        (r#"덢"#, r#"덢"#, r#"덢"#, r#"덢"#, r#"덢"#),
        (r#"덣"#, r#"덣"#, r#"덣"#, r#"덣"#, r#"덣"#),
        (r#"덤"#, r#"덤"#, r#"덤"#, r#"덤"#, r#"덤"#),
        (r#"덥"#, r#"덥"#, r#"덥"#, r#"덥"#, r#"덥"#),
        (r#"덦"#, r#"덦"#, r#"덦"#, r#"덦"#, r#"덦"#),
        (r#"덧"#, r#"덧"#, r#"덧"#, r#"덧"#, r#"덧"#),
        (r#"덨"#, r#"덨"#, r#"덨"#, r#"덨"#, r#"덨"#),
        (r#"덩"#, r#"덩"#, r#"덩"#, r#"덩"#, r#"덩"#),
        (r#"덪"#, r#"덪"#, r#"덪"#, r#"덪"#, r#"덪"#),
        (r#"덫"#, r#"덫"#, r#"덫"#, r#"덫"#, r#"덫"#),
        (r#"덬"#, r#"덬"#, r#"덬"#, r#"덬"#, r#"덬"#),
        (r#"덭"#, r#"덭"#, r#"덭"#, r#"덭"#, r#"덭"#),
        (r#"덮"#, r#"덮"#, r#"덮"#, r#"덮"#, r#"덮"#),
        (r#"덯"#, r#"덯"#, r#"덯"#, r#"덯"#, r#"덯"#),
        (r#"데"#, r#"데"#, r#"데"#, r#"데"#, r#"데"#),
        (r#"덱"#, r#"덱"#, r#"덱"#, r#"덱"#, r#"덱"#),
        (r#"덲"#, r#"덲"#, r#"덲"#, r#"덲"#, r#"덲"#),
        (r#"덳"#, r#"덳"#, r#"덳"#, r#"덳"#, r#"덳"#),
        (r#"덴"#, r#"덴"#, r#"덴"#, r#"덴"#, r#"덴"#),
        (r#"덵"#, r#"덵"#, r#"덵"#, r#"덵"#, r#"덵"#),
        (r#"덶"#, r#"덶"#, r#"덶"#, r#"덶"#, r#"덶"#),
        (r#"덷"#, r#"덷"#, r#"덷"#, r#"덷"#, r#"덷"#),
        (r#"델"#, r#"델"#, r#"델"#, r#"델"#, r#"델"#),
        (r#"덹"#, r#"덹"#, r#"덹"#, r#"덹"#, r#"덹"#),
        (r#"덺"#, r#"덺"#, r#"덺"#, r#"덺"#, r#"덺"#),
        (r#"덻"#, r#"덻"#, r#"덻"#, r#"덻"#, r#"덻"#),
        (r#"덼"#, r#"덼"#, r#"덼"#, r#"덼"#, r#"덼"#),
        (r#"덽"#, r#"덽"#, r#"덽"#, r#"덽"#, r#"덽"#),
        (r#"덾"#, r#"덾"#, r#"덾"#, r#"덾"#, r#"덾"#),
        (r#"덿"#, r#"덿"#, r#"덿"#, r#"덿"#, r#"덿"#),
        (r#"뎀"#, r#"뎀"#, r#"뎀"#, r#"뎀"#, r#"뎀"#),
        (r#"뎁"#, r#"뎁"#, r#"뎁"#, r#"뎁"#, r#"뎁"#),
        (r#"뎂"#, r#"뎂"#, r#"뎂"#, r#"뎂"#, r#"뎂"#),
        (r#"뎃"#, r#"뎃"#, r#"뎃"#, r#"뎃"#, r#"뎃"#),
        (r#"뎄"#, r#"뎄"#, r#"뎄"#, r#"뎄"#, r#"뎄"#),
        (r#"뎅"#, r#"뎅"#, r#"뎅"#, r#"뎅"#, r#"뎅"#),
        (r#"뎆"#, r#"뎆"#, r#"뎆"#, r#"뎆"#, r#"뎆"#),
        (r#"뎇"#, r#"뎇"#, r#"뎇"#, r#"뎇"#, r#"뎇"#),
        (r#"뎈"#, r#"뎈"#, r#"뎈"#, r#"뎈"#, r#"뎈"#),
        (r#"뎉"#, r#"뎉"#, r#"뎉"#, r#"뎉"#, r#"뎉"#),
        (r#"뎊"#, r#"뎊"#, r#"뎊"#, r#"뎊"#, r#"뎊"#),
        (r#"뎋"#, r#"뎋"#, r#"뎋"#, r#"뎋"#, r#"뎋"#),
        (r#"뎌"#, r#"뎌"#, r#"뎌"#, r#"뎌"#, r#"뎌"#),
        (r#"뎍"#, r#"뎍"#, r#"뎍"#, r#"뎍"#, r#"뎍"#),
        (r#"뎎"#, r#"뎎"#, r#"뎎"#, r#"뎎"#, r#"뎎"#),
        (r#"뎏"#, r#"뎏"#, r#"뎏"#, r#"뎏"#, r#"뎏"#),
        (r#"뎐"#, r#"뎐"#, r#"뎐"#, r#"뎐"#, r#"뎐"#),
        (r#"뎑"#, r#"뎑"#, r#"뎑"#, r#"뎑"#, r#"뎑"#),
        (r#"뎒"#, r#"뎒"#, r#"뎒"#, r#"뎒"#, r#"뎒"#),
        (r#"뎓"#, r#"뎓"#, r#"뎓"#, r#"뎓"#, r#"뎓"#),
        (r#"뎔"#, r#"뎔"#, r#"뎔"#, r#"뎔"#, r#"뎔"#),
        (r#"뎕"#, r#"뎕"#, r#"뎕"#, r#"뎕"#, r#"뎕"#),
        (r#"뎖"#, r#"뎖"#, r#"뎖"#, r#"뎖"#, r#"뎖"#),
        (r#"뎗"#, r#"뎗"#, r#"뎗"#, r#"뎗"#, r#"뎗"#),
        (r#"뎘"#, r#"뎘"#, r#"뎘"#, r#"뎘"#, r#"뎘"#),
        (r#"뎙"#, r#"뎙"#, r#"뎙"#, r#"뎙"#, r#"뎙"#),
        (r#"뎚"#, r#"뎚"#, r#"뎚"#, r#"뎚"#, r#"뎚"#),
        (r#"뎛"#, r#"뎛"#, r#"뎛"#, r#"뎛"#, r#"뎛"#),
        (r#"뎜"#, r#"뎜"#, r#"뎜"#, r#"뎜"#, r#"뎜"#),
        (r#"뎝"#, r#"뎝"#, r#"뎝"#, r#"뎝"#, r#"뎝"#),
        (r#"뎞"#, r#"뎞"#, r#"뎞"#, r#"뎞"#, r#"뎞"#),
        (r#"뎟"#, r#"뎟"#, r#"뎟"#, r#"뎟"#, r#"뎟"#),
        (r#"뎠"#, r#"뎠"#, r#"뎠"#, r#"뎠"#, r#"뎠"#),
        (r#"뎡"#, r#"뎡"#, r#"뎡"#, r#"뎡"#, r#"뎡"#),
        (r#"뎢"#, r#"뎢"#, r#"뎢"#, r#"뎢"#, r#"뎢"#),
        (r#"뎣"#, r#"뎣"#, r#"뎣"#, r#"뎣"#, r#"뎣"#),
        (r#"뎤"#, r#"뎤"#, r#"뎤"#, r#"뎤"#, r#"뎤"#),
        (r#"뎥"#, r#"뎥"#, r#"뎥"#, r#"뎥"#, r#"뎥"#),
        (r#"뎦"#, r#"뎦"#, r#"뎦"#, r#"뎦"#, r#"뎦"#),
        (r#"뎧"#, r#"뎧"#, r#"뎧"#, r#"뎧"#, r#"뎧"#),
        (r#"뎨"#, r#"뎨"#, r#"뎨"#, r#"뎨"#, r#"뎨"#),
        (r#"뎩"#, r#"뎩"#, r#"뎩"#, r#"뎩"#, r#"뎩"#),
        (r#"뎪"#, r#"뎪"#, r#"뎪"#, r#"뎪"#, r#"뎪"#),
        (r#"뎫"#, r#"뎫"#, r#"뎫"#, r#"뎫"#, r#"뎫"#),
        (r#"뎬"#, r#"뎬"#, r#"뎬"#, r#"뎬"#, r#"뎬"#),
        (r#"뎭"#, r#"뎭"#, r#"뎭"#, r#"뎭"#, r#"뎭"#),
        (r#"뎮"#, r#"뎮"#, r#"뎮"#, r#"뎮"#, r#"뎮"#),
        (r#"뎯"#, r#"뎯"#, r#"뎯"#, r#"뎯"#, r#"뎯"#),
        (r#"뎰"#, r#"뎰"#, r#"뎰"#, r#"뎰"#, r#"뎰"#),
        (r#"뎱"#, r#"뎱"#, r#"뎱"#, r#"뎱"#, r#"뎱"#),
        (r#"뎲"#, r#"뎲"#, r#"뎲"#, r#"뎲"#, r#"뎲"#),
        (r#"뎳"#, r#"뎳"#, r#"뎳"#, r#"뎳"#, r#"뎳"#),
        (r#"뎴"#, r#"뎴"#, r#"뎴"#, r#"뎴"#, r#"뎴"#),
        (r#"뎵"#, r#"뎵"#, r#"뎵"#, r#"뎵"#, r#"뎵"#),
        (r#"뎶"#, r#"뎶"#, r#"뎶"#, r#"뎶"#, r#"뎶"#),
        (r#"뎷"#, r#"뎷"#, r#"뎷"#, r#"뎷"#, r#"뎷"#),
        (r#"뎸"#, r#"뎸"#, r#"뎸"#, r#"뎸"#, r#"뎸"#),
        (r#"뎹"#, r#"뎹"#, r#"뎹"#, r#"뎹"#, r#"뎹"#),
        (r#"뎺"#, r#"뎺"#, r#"뎺"#, r#"뎺"#, r#"뎺"#),
        (r#"뎻"#, r#"뎻"#, r#"뎻"#, r#"뎻"#, r#"뎻"#),
        (r#"뎼"#, r#"뎼"#, r#"뎼"#, r#"뎼"#, r#"뎼"#),
        (r#"뎽"#, r#"뎽"#, r#"뎽"#, r#"뎽"#, r#"뎽"#),
        (r#"뎾"#, r#"뎾"#, r#"뎾"#, r#"뎾"#, r#"뎾"#),
        (r#"뎿"#, r#"뎿"#, r#"뎿"#, r#"뎿"#, r#"뎿"#),
        (r#"돀"#, r#"돀"#, r#"돀"#, r#"돀"#, r#"돀"#),
        (r#"돁"#, r#"돁"#, r#"돁"#, r#"돁"#, r#"돁"#),
        (r#"돂"#, r#"돂"#, r#"돂"#, r#"돂"#, r#"돂"#),
        (r#"돃"#, r#"돃"#, r#"돃"#, r#"돃"#, r#"돃"#),
        (r#"도"#, r#"도"#, r#"도"#, r#"도"#, r#"도"#),
        (r#"독"#, r#"독"#, r#"독"#, r#"독"#, r#"독"#),
        (r#"돆"#, r#"돆"#, r#"돆"#, r#"돆"#, r#"돆"#),
        (r#"돇"#, r#"돇"#, r#"돇"#, r#"돇"#, r#"돇"#),
        (r#"돈"#, r#"돈"#, r#"돈"#, r#"돈"#, r#"돈"#),
        (r#"돉"#, r#"돉"#, r#"돉"#, r#"돉"#, r#"돉"#),
        (r#"돊"#, r#"돊"#, r#"돊"#, r#"돊"#, r#"돊"#),
        (r#"돋"#, r#"돋"#, r#"돋"#, r#"돋"#, r#"돋"#),
        (r#"돌"#, r#"돌"#, r#"돌"#, r#"돌"#, r#"돌"#),
        (r#"돍"#, r#"돍"#, r#"돍"#, r#"돍"#, r#"돍"#),
        (r#"돎"#, r#"돎"#, r#"돎"#, r#"돎"#, r#"돎"#),
        (r#"돏"#, r#"돏"#, r#"돏"#, r#"돏"#, r#"돏"#),
        (r#"돐"#, r#"돐"#, r#"돐"#, r#"돐"#, r#"돐"#),
        (r#"돑"#, r#"돑"#, r#"돑"#, r#"돑"#, r#"돑"#),
        (r#"돒"#, r#"돒"#, r#"돒"#, r#"돒"#, r#"돒"#),
        (r#"돓"#, r#"돓"#, r#"돓"#, r#"돓"#, r#"돓"#),
        (r#"돔"#, r#"돔"#, r#"돔"#, r#"돔"#, r#"돔"#),
        (r#"돕"#, r#"돕"#, r#"돕"#, r#"돕"#, r#"돕"#),
        (r#"돖"#, r#"돖"#, r#"돖"#, r#"돖"#, r#"돖"#),
        (r#"돗"#, r#"돗"#, r#"돗"#, r#"돗"#, r#"돗"#),
        (r#"돘"#, r#"돘"#, r#"돘"#, r#"돘"#, r#"돘"#),
        (r#"동"#, r#"동"#, r#"동"#, r#"동"#, r#"동"#),
        (r#"돚"#, r#"돚"#, r#"돚"#, r#"돚"#, r#"돚"#),
        (r#"돛"#, r#"돛"#, r#"돛"#, r#"돛"#, r#"돛"#),
        (r#"돜"#, r#"돜"#, r#"돜"#, r#"돜"#, r#"돜"#),
        (r#"돝"#, r#"돝"#, r#"돝"#, r#"돝"#, r#"돝"#),
        (r#"돞"#, r#"돞"#, r#"돞"#, r#"돞"#, r#"돞"#),
        (r#"돟"#, r#"돟"#, r#"돟"#, r#"돟"#, r#"돟"#),
        (r#"돠"#, r#"돠"#, r#"돠"#, r#"돠"#, r#"돠"#),
        (r#"돡"#, r#"돡"#, r#"돡"#, r#"돡"#, r#"돡"#),
        (r#"돢"#, r#"돢"#, r#"돢"#, r#"돢"#, r#"돢"#),
        (r#"돣"#, r#"돣"#, r#"돣"#, r#"돣"#, r#"돣"#),
        (r#"돤"#, r#"돤"#, r#"돤"#, r#"돤"#, r#"돤"#),
        (r#"돥"#, r#"돥"#, r#"돥"#, r#"돥"#, r#"돥"#),
        (r#"돦"#, r#"돦"#, r#"돦"#, r#"돦"#, r#"돦"#),
        (r#"돧"#, r#"돧"#, r#"돧"#, r#"돧"#, r#"돧"#),
        (r#"돨"#, r#"돨"#, r#"돨"#, r#"돨"#, r#"돨"#),
        (r#"돩"#, r#"돩"#, r#"돩"#, r#"돩"#, r#"돩"#),
        (r#"돪"#, r#"돪"#, r#"돪"#, r#"돪"#, r#"돪"#),
        (r#"돫"#, r#"돫"#, r#"돫"#, r#"돫"#, r#"돫"#),
        (r#"돬"#, r#"돬"#, r#"돬"#, r#"돬"#, r#"돬"#),
        (r#"돭"#, r#"돭"#, r#"돭"#, r#"돭"#, r#"돭"#),
        (r#"돮"#, r#"돮"#, r#"돮"#, r#"돮"#, r#"돮"#),
        (r#"돯"#, r#"돯"#, r#"돯"#, r#"돯"#, r#"돯"#),
        (r#"돰"#, r#"돰"#, r#"돰"#, r#"돰"#, r#"돰"#),
        (r#"돱"#, r#"돱"#, r#"돱"#, r#"돱"#, r#"돱"#),
        (r#"돲"#, r#"돲"#, r#"돲"#, r#"돲"#, r#"돲"#),
        (r#"돳"#, r#"돳"#, r#"돳"#, r#"돳"#, r#"돳"#),
        (r#"돴"#, r#"돴"#, r#"돴"#, r#"돴"#, r#"돴"#),
        (r#"돵"#, r#"돵"#, r#"돵"#, r#"돵"#, r#"돵"#),
        (r#"돶"#, r#"돶"#, r#"돶"#, r#"돶"#, r#"돶"#),
        (r#"돷"#, r#"돷"#, r#"돷"#, r#"돷"#, r#"돷"#),
        (r#"돸"#, r#"돸"#, r#"돸"#, r#"돸"#, r#"돸"#),
        (r#"돹"#, r#"돹"#, r#"돹"#, r#"돹"#, r#"돹"#),
        (r#"돺"#, r#"돺"#, r#"돺"#, r#"돺"#, r#"돺"#),
        (r#"돻"#, r#"돻"#, r#"돻"#, r#"돻"#, r#"돻"#),
        (r#"돼"#, r#"돼"#, r#"돼"#, r#"돼"#, r#"돼"#),
        (r#"돽"#, r#"돽"#, r#"돽"#, r#"돽"#, r#"돽"#),
        (r#"돾"#, r#"돾"#, r#"돾"#, r#"돾"#, r#"돾"#),
        (r#"돿"#, r#"돿"#, r#"돿"#, r#"돿"#, r#"돿"#),
        (r#"됀"#, r#"됀"#, r#"됀"#, r#"됀"#, r#"됀"#),
        (r#"됁"#, r#"됁"#, r#"됁"#, r#"됁"#, r#"됁"#),
        (r#"됂"#, r#"됂"#, r#"됂"#, r#"됂"#, r#"됂"#),
        (r#"됃"#, r#"됃"#, r#"됃"#, r#"됃"#, r#"됃"#),
        (r#"됄"#, r#"됄"#, r#"됄"#, r#"됄"#, r#"됄"#),
        (r#"됅"#, r#"됅"#, r#"됅"#, r#"됅"#, r#"됅"#),
        (r#"됆"#, r#"됆"#, r#"됆"#, r#"됆"#, r#"됆"#),
        (r#"됇"#, r#"됇"#, r#"됇"#, r#"됇"#, r#"됇"#),
        (r#"됈"#, r#"됈"#, r#"됈"#, r#"됈"#, r#"됈"#),
        (r#"됉"#, r#"됉"#, r#"됉"#, r#"됉"#, r#"됉"#),
        (r#"됊"#, r#"됊"#, r#"됊"#, r#"됊"#, r#"됊"#),
        (r#"됋"#, r#"됋"#, r#"됋"#, r#"됋"#, r#"됋"#),
        (r#"됌"#, r#"됌"#, r#"됌"#, r#"됌"#, r#"됌"#),
        (r#"됍"#, r#"됍"#, r#"됍"#, r#"됍"#, r#"됍"#),
        (r#"됎"#, r#"됎"#, r#"됎"#, r#"됎"#, r#"됎"#),
        (r#"됏"#, r#"됏"#, r#"됏"#, r#"됏"#, r#"됏"#),
        (r#"됐"#, r#"됐"#, r#"됐"#, r#"됐"#, r#"됐"#),
        (r#"됑"#, r#"됑"#, r#"됑"#, r#"됑"#, r#"됑"#),
        (r#"됒"#, r#"됒"#, r#"됒"#, r#"됒"#, r#"됒"#),
        (r#"됓"#, r#"됓"#, r#"됓"#, r#"됓"#, r#"됓"#),
        (r#"됔"#, r#"됔"#, r#"됔"#, r#"됔"#, r#"됔"#),
        (r#"됕"#, r#"됕"#, r#"됕"#, r#"됕"#, r#"됕"#),
        (r#"됖"#, r#"됖"#, r#"됖"#, r#"됖"#, r#"됖"#),
        (r#"됗"#, r#"됗"#, r#"됗"#, r#"됗"#, r#"됗"#),
        (r#"되"#, r#"되"#, r#"되"#, r#"되"#, r#"되"#),
        (r#"됙"#, r#"됙"#, r#"됙"#, r#"됙"#, r#"됙"#),
        (r#"됚"#, r#"됚"#, r#"됚"#, r#"됚"#, r#"됚"#),
        (r#"됛"#, r#"됛"#, r#"됛"#, r#"됛"#, r#"됛"#),
        (r#"된"#, r#"된"#, r#"된"#, r#"된"#, r#"된"#),
        (r#"됝"#, r#"됝"#, r#"됝"#, r#"됝"#, r#"됝"#),
        (r#"됞"#, r#"됞"#, r#"됞"#, r#"됞"#, r#"됞"#),
        (r#"됟"#, r#"됟"#, r#"됟"#, r#"됟"#, r#"됟"#),
        (r#"될"#, r#"될"#, r#"될"#, r#"될"#, r#"될"#),
        (r#"됡"#, r#"됡"#, r#"됡"#, r#"됡"#, r#"됡"#),
        (r#"됢"#, r#"됢"#, r#"됢"#, r#"됢"#, r#"됢"#),
        (r#"됣"#, r#"됣"#, r#"됣"#, r#"됣"#, r#"됣"#),
        (r#"됤"#, r#"됤"#, r#"됤"#, r#"됤"#, r#"됤"#),
        (r#"됥"#, r#"됥"#, r#"됥"#, r#"됥"#, r#"됥"#),
        (r#"됦"#, r#"됦"#, r#"됦"#, r#"됦"#, r#"됦"#),
        (r#"됧"#, r#"됧"#, r#"됧"#, r#"됧"#, r#"됧"#),
        (r#"됨"#, r#"됨"#, r#"됨"#, r#"됨"#, r#"됨"#),
        (r#"됩"#, r#"됩"#, r#"됩"#, r#"됩"#, r#"됩"#),
        (r#"됪"#, r#"됪"#, r#"됪"#, r#"됪"#, r#"됪"#),
        (r#"됫"#, r#"됫"#, r#"됫"#, r#"됫"#, r#"됫"#),
        (r#"됬"#, r#"됬"#, r#"됬"#, r#"됬"#, r#"됬"#),
        (r#"됭"#, r#"됭"#, r#"됭"#, r#"됭"#, r#"됭"#),
        (r#"됮"#, r#"됮"#, r#"됮"#, r#"됮"#, r#"됮"#),
        (r#"됯"#, r#"됯"#, r#"됯"#, r#"됯"#, r#"됯"#),
        (r#"됰"#, r#"됰"#, r#"됰"#, r#"됰"#, r#"됰"#),
        (r#"됱"#, r#"됱"#, r#"됱"#, r#"됱"#, r#"됱"#),
        (r#"됲"#, r#"됲"#, r#"됲"#, r#"됲"#, r#"됲"#),
        (r#"됳"#, r#"됳"#, r#"됳"#, r#"됳"#, r#"됳"#),
        (r#"됴"#, r#"됴"#, r#"됴"#, r#"됴"#, r#"됴"#),
        (r#"됵"#, r#"됵"#, r#"됵"#, r#"됵"#, r#"됵"#),
        (r#"됶"#, r#"됶"#, r#"됶"#, r#"됶"#, r#"됶"#),
        (r#"됷"#, r#"됷"#, r#"됷"#, r#"됷"#, r#"됷"#),
        (r#"됸"#, r#"됸"#, r#"됸"#, r#"됸"#, r#"됸"#),
        (r#"됹"#, r#"됹"#, r#"됹"#, r#"됹"#, r#"됹"#),
        (r#"됺"#, r#"됺"#, r#"됺"#, r#"됺"#, r#"됺"#),
        (r#"됻"#, r#"됻"#, r#"됻"#, r#"됻"#, r#"됻"#),
        (r#"됼"#, r#"됼"#, r#"됼"#, r#"됼"#, r#"됼"#),
        (r#"됽"#, r#"됽"#, r#"됽"#, r#"됽"#, r#"됽"#),
        (r#"됾"#, r#"됾"#, r#"됾"#, r#"됾"#, r#"됾"#),
        (r#"됿"#, r#"됿"#, r#"됿"#, r#"됿"#, r#"됿"#),
        (r#"둀"#, r#"둀"#, r#"둀"#, r#"둀"#, r#"둀"#),
        (r#"둁"#, r#"둁"#, r#"둁"#, r#"둁"#, r#"둁"#),
        (r#"둂"#, r#"둂"#, r#"둂"#, r#"둂"#, r#"둂"#),
        (r#"둃"#, r#"둃"#, r#"둃"#, r#"둃"#, r#"둃"#),
        (r#"둄"#, r#"둄"#, r#"둄"#, r#"둄"#, r#"둄"#),
        (r#"둅"#, r#"둅"#, r#"둅"#, r#"둅"#, r#"둅"#),
        (r#"둆"#, r#"둆"#, r#"둆"#, r#"둆"#, r#"둆"#),
        (r#"둇"#, r#"둇"#, r#"둇"#, r#"둇"#, r#"둇"#),
        (r#"둈"#, r#"둈"#, r#"둈"#, r#"둈"#, r#"둈"#),
        (r#"둉"#, r#"둉"#, r#"둉"#, r#"둉"#, r#"둉"#),
        (r#"둊"#, r#"둊"#, r#"둊"#, r#"둊"#, r#"둊"#),
        (r#"둋"#, r#"둋"#, r#"둋"#, r#"둋"#, r#"둋"#),
        (r#"둌"#, r#"둌"#, r#"둌"#, r#"둌"#, r#"둌"#),
        (r#"둍"#, r#"둍"#, r#"둍"#, r#"둍"#, r#"둍"#),
        (r#"둎"#, r#"둎"#, r#"둎"#, r#"둎"#, r#"둎"#),
        (r#"둏"#, r#"둏"#, r#"둏"#, r#"둏"#, r#"둏"#),
        (r#"두"#, r#"두"#, r#"두"#, r#"두"#, r#"두"#),
        (r#"둑"#, r#"둑"#, r#"둑"#, r#"둑"#, r#"둑"#),
        (r#"둒"#, r#"둒"#, r#"둒"#, r#"둒"#, r#"둒"#),
        (r#"둓"#, r#"둓"#, r#"둓"#, r#"둓"#, r#"둓"#),
        (r#"둔"#, r#"둔"#, r#"둔"#, r#"둔"#, r#"둔"#),
        (r#"둕"#, r#"둕"#, r#"둕"#, r#"둕"#, r#"둕"#),
        (r#"둖"#, r#"둖"#, r#"둖"#, r#"둖"#, r#"둖"#),
        (r#"둗"#, r#"둗"#, r#"둗"#, r#"둗"#, r#"둗"#),
        (r#"둘"#, r#"둘"#, r#"둘"#, r#"둘"#, r#"둘"#),
        (r#"둙"#, r#"둙"#, r#"둙"#, r#"둙"#, r#"둙"#),
        (r#"둚"#, r#"둚"#, r#"둚"#, r#"둚"#, r#"둚"#),
        (r#"둛"#, r#"둛"#, r#"둛"#, r#"둛"#, r#"둛"#),
        (r#"둜"#, r#"둜"#, r#"둜"#, r#"둜"#, r#"둜"#),
        (r#"둝"#, r#"둝"#, r#"둝"#, r#"둝"#, r#"둝"#),
        (r#"둞"#, r#"둞"#, r#"둞"#, r#"둞"#, r#"둞"#),
        (r#"둟"#, r#"둟"#, r#"둟"#, r#"둟"#, r#"둟"#),
        (r#"둠"#, r#"둠"#, r#"둠"#, r#"둠"#, r#"둠"#),
        (r#"둡"#, r#"둡"#, r#"둡"#, r#"둡"#, r#"둡"#),
        (r#"둢"#, r#"둢"#, r#"둢"#, r#"둢"#, r#"둢"#),
        (r#"둣"#, r#"둣"#, r#"둣"#, r#"둣"#, r#"둣"#),
        (r#"둤"#, r#"둤"#, r#"둤"#, r#"둤"#, r#"둤"#),
        (r#"둥"#, r#"둥"#, r#"둥"#, r#"둥"#, r#"둥"#),
        (r#"둦"#, r#"둦"#, r#"둦"#, r#"둦"#, r#"둦"#),
        (r#"둧"#, r#"둧"#, r#"둧"#, r#"둧"#, r#"둧"#),
        (r#"둨"#, r#"둨"#, r#"둨"#, r#"둨"#, r#"둨"#),
        (r#"둩"#, r#"둩"#, r#"둩"#, r#"둩"#, r#"둩"#),
        (r#"둪"#, r#"둪"#, r#"둪"#, r#"둪"#, r#"둪"#),
        (r#"둫"#, r#"둫"#, r#"둫"#, r#"둫"#, r#"둫"#),
        (r#"둬"#, r#"둬"#, r#"둬"#, r#"둬"#, r#"둬"#),
        (r#"둭"#, r#"둭"#, r#"둭"#, r#"둭"#, r#"둭"#),
        (r#"둮"#, r#"둮"#, r#"둮"#, r#"둮"#, r#"둮"#),
        (r#"둯"#, r#"둯"#, r#"둯"#, r#"둯"#, r#"둯"#),
        (r#"둰"#, r#"둰"#, r#"둰"#, r#"둰"#, r#"둰"#),
        (r#"둱"#, r#"둱"#, r#"둱"#, r#"둱"#, r#"둱"#),
        (r#"둲"#, r#"둲"#, r#"둲"#, r#"둲"#, r#"둲"#),
        (r#"둳"#, r#"둳"#, r#"둳"#, r#"둳"#, r#"둳"#),
        (r#"둴"#, r#"둴"#, r#"둴"#, r#"둴"#, r#"둴"#),
        (r#"둵"#, r#"둵"#, r#"둵"#, r#"둵"#, r#"둵"#),
        (r#"둶"#, r#"둶"#, r#"둶"#, r#"둶"#, r#"둶"#),
        (r#"둷"#, r#"둷"#, r#"둷"#, r#"둷"#, r#"둷"#),
        (r#"둸"#, r#"둸"#, r#"둸"#, r#"둸"#, r#"둸"#),
        (r#"둹"#, r#"둹"#, r#"둹"#, r#"둹"#, r#"둹"#),
        (r#"둺"#, r#"둺"#, r#"둺"#, r#"둺"#, r#"둺"#),
        (r#"둻"#, r#"둻"#, r#"둻"#, r#"둻"#, r#"둻"#),
        (r#"둼"#, r#"둼"#, r#"둼"#, r#"둼"#, r#"둼"#),
        (r#"둽"#, r#"둽"#, r#"둽"#, r#"둽"#, r#"둽"#),
        (r#"둾"#, r#"둾"#, r#"둾"#, r#"둾"#, r#"둾"#),
        (r#"둿"#, r#"둿"#, r#"둿"#, r#"둿"#, r#"둿"#),
        (r#"뒀"#, r#"뒀"#, r#"뒀"#, r#"뒀"#, r#"뒀"#),
        (r#"뒁"#, r#"뒁"#, r#"뒁"#, r#"뒁"#, r#"뒁"#),
        (r#"뒂"#, r#"뒂"#, r#"뒂"#, r#"뒂"#, r#"뒂"#),
        (r#"뒃"#, r#"뒃"#, r#"뒃"#, r#"뒃"#, r#"뒃"#),
        (r#"뒄"#, r#"뒄"#, r#"뒄"#, r#"뒄"#, r#"뒄"#),
        (r#"뒅"#, r#"뒅"#, r#"뒅"#, r#"뒅"#, r#"뒅"#),
        (r#"뒆"#, r#"뒆"#, r#"뒆"#, r#"뒆"#, r#"뒆"#),
        (r#"뒇"#, r#"뒇"#, r#"뒇"#, r#"뒇"#, r#"뒇"#),
        (r#"뒈"#, r#"뒈"#, r#"뒈"#, r#"뒈"#, r#"뒈"#),
        (r#"뒉"#, r#"뒉"#, r#"뒉"#, r#"뒉"#, r#"뒉"#),
        (r#"뒊"#, r#"뒊"#, r#"뒊"#, r#"뒊"#, r#"뒊"#),
        (r#"뒋"#, r#"뒋"#, r#"뒋"#, r#"뒋"#, r#"뒋"#),
        (r#"뒌"#, r#"뒌"#, r#"뒌"#, r#"뒌"#, r#"뒌"#),
        (r#"뒍"#, r#"뒍"#, r#"뒍"#, r#"뒍"#, r#"뒍"#),
        (r#"뒎"#, r#"뒎"#, r#"뒎"#, r#"뒎"#, r#"뒎"#),
        (r#"뒏"#, r#"뒏"#, r#"뒏"#, r#"뒏"#, r#"뒏"#),
        (r#"뒐"#, r#"뒐"#, r#"뒐"#, r#"뒐"#, r#"뒐"#),
        (r#"뒑"#, r#"뒑"#, r#"뒑"#, r#"뒑"#, r#"뒑"#),
        (r#"뒒"#, r#"뒒"#, r#"뒒"#, r#"뒒"#, r#"뒒"#),
        (r#"뒓"#, r#"뒓"#, r#"뒓"#, r#"뒓"#, r#"뒓"#),
        (r#"뒔"#, r#"뒔"#, r#"뒔"#, r#"뒔"#, r#"뒔"#),
        (r#"뒕"#, r#"뒕"#, r#"뒕"#, r#"뒕"#, r#"뒕"#),
        (r#"뒖"#, r#"뒖"#, r#"뒖"#, r#"뒖"#, r#"뒖"#),
        (r#"뒗"#, r#"뒗"#, r#"뒗"#, r#"뒗"#, r#"뒗"#),
        (r#"뒘"#, r#"뒘"#, r#"뒘"#, r#"뒘"#, r#"뒘"#),
        (r#"뒙"#, r#"뒙"#, r#"뒙"#, r#"뒙"#, r#"뒙"#),
        (r#"뒚"#, r#"뒚"#, r#"뒚"#, r#"뒚"#, r#"뒚"#),
        (r#"뒛"#, r#"뒛"#, r#"뒛"#, r#"뒛"#, r#"뒛"#),
        (r#"뒜"#, r#"뒜"#, r#"뒜"#, r#"뒜"#, r#"뒜"#),
        (r#"뒝"#, r#"뒝"#, r#"뒝"#, r#"뒝"#, r#"뒝"#),
        (r#"뒞"#, r#"뒞"#, r#"뒞"#, r#"뒞"#, r#"뒞"#),
        (r#"뒟"#, r#"뒟"#, r#"뒟"#, r#"뒟"#, r#"뒟"#),
        (r#"뒠"#, r#"뒠"#, r#"뒠"#, r#"뒠"#, r#"뒠"#),
        (r#"뒡"#, r#"뒡"#, r#"뒡"#, r#"뒡"#, r#"뒡"#),
        (r#"뒢"#, r#"뒢"#, r#"뒢"#, r#"뒢"#, r#"뒢"#),
        (r#"뒣"#, r#"뒣"#, r#"뒣"#, r#"뒣"#, r#"뒣"#),
        (r#"뒤"#, r#"뒤"#, r#"뒤"#, r#"뒤"#, r#"뒤"#),
        (r#"뒥"#, r#"뒥"#, r#"뒥"#, r#"뒥"#, r#"뒥"#),
        (r#"뒦"#, r#"뒦"#, r#"뒦"#, r#"뒦"#, r#"뒦"#),
        (r#"뒧"#, r#"뒧"#, r#"뒧"#, r#"뒧"#, r#"뒧"#),
        (r#"뒨"#, r#"뒨"#, r#"뒨"#, r#"뒨"#, r#"뒨"#),
        (r#"뒩"#, r#"뒩"#, r#"뒩"#, r#"뒩"#, r#"뒩"#),
        (r#"뒪"#, r#"뒪"#, r#"뒪"#, r#"뒪"#, r#"뒪"#),
        (r#"뒫"#, r#"뒫"#, r#"뒫"#, r#"뒫"#, r#"뒫"#),
        (r#"뒬"#, r#"뒬"#, r#"뒬"#, r#"뒬"#, r#"뒬"#),
        (r#"뒭"#, r#"뒭"#, r#"뒭"#, r#"뒭"#, r#"뒭"#),
        (r#"뒮"#, r#"뒮"#, r#"뒮"#, r#"뒮"#, r#"뒮"#),
        (r#"뒯"#, r#"뒯"#, r#"뒯"#, r#"뒯"#, r#"뒯"#),
        (r#"뒰"#, r#"뒰"#, r#"뒰"#, r#"뒰"#, r#"뒰"#),
        (r#"뒱"#, r#"뒱"#, r#"뒱"#, r#"뒱"#, r#"뒱"#),
        (r#"뒲"#, r#"뒲"#, r#"뒲"#, r#"뒲"#, r#"뒲"#),
        (r#"뒳"#, r#"뒳"#, r#"뒳"#, r#"뒳"#, r#"뒳"#),
        (r#"뒴"#, r#"뒴"#, r#"뒴"#, r#"뒴"#, r#"뒴"#),
        (r#"뒵"#, r#"뒵"#, r#"뒵"#, r#"뒵"#, r#"뒵"#),
        (r#"뒶"#, r#"뒶"#, r#"뒶"#, r#"뒶"#, r#"뒶"#),
        (r#"뒷"#, r#"뒷"#, r#"뒷"#, r#"뒷"#, r#"뒷"#),
        (r#"뒸"#, r#"뒸"#, r#"뒸"#, r#"뒸"#, r#"뒸"#),
        (r#"뒹"#, r#"뒹"#, r#"뒹"#, r#"뒹"#, r#"뒹"#),
        (r#"뒺"#, r#"뒺"#, r#"뒺"#, r#"뒺"#, r#"뒺"#),
        (r#"뒻"#, r#"뒻"#, r#"뒻"#, r#"뒻"#, r#"뒻"#),
        (r#"뒼"#, r#"뒼"#, r#"뒼"#, r#"뒼"#, r#"뒼"#),
        (r#"뒽"#, r#"뒽"#, r#"뒽"#, r#"뒽"#, r#"뒽"#),
        (r#"뒾"#, r#"뒾"#, r#"뒾"#, r#"뒾"#, r#"뒾"#),
        (r#"뒿"#, r#"뒿"#, r#"뒿"#, r#"뒿"#, r#"뒿"#),
        (r#"듀"#, r#"듀"#, r#"듀"#, r#"듀"#, r#"듀"#),
        (r#"듁"#, r#"듁"#, r#"듁"#, r#"듁"#, r#"듁"#),
        (r#"듂"#, r#"듂"#, r#"듂"#, r#"듂"#, r#"듂"#),
        (r#"듃"#, r#"듃"#, r#"듃"#, r#"듃"#, r#"듃"#),
        (r#"듄"#, r#"듄"#, r#"듄"#, r#"듄"#, r#"듄"#),
        (r#"듅"#, r#"듅"#, r#"듅"#, r#"듅"#, r#"듅"#),
        (r#"듆"#, r#"듆"#, r#"듆"#, r#"듆"#, r#"듆"#),
        (r#"듇"#, r#"듇"#, r#"듇"#, r#"듇"#, r#"듇"#),
        (r#"듈"#, r#"듈"#, r#"듈"#, r#"듈"#, r#"듈"#),
        (r#"듉"#, r#"듉"#, r#"듉"#, r#"듉"#, r#"듉"#),
        (r#"듊"#, r#"듊"#, r#"듊"#, r#"듊"#, r#"듊"#),
        (r#"듋"#, r#"듋"#, r#"듋"#, r#"듋"#, r#"듋"#),
        (r#"듌"#, r#"듌"#, r#"듌"#, r#"듌"#, r#"듌"#),
        (r#"듍"#, r#"듍"#, r#"듍"#, r#"듍"#, r#"듍"#),
        (r#"듎"#, r#"듎"#, r#"듎"#, r#"듎"#, r#"듎"#),
        (r#"듏"#, r#"듏"#, r#"듏"#, r#"듏"#, r#"듏"#),
        (r#"듐"#, r#"듐"#, r#"듐"#, r#"듐"#, r#"듐"#),
        (r#"듑"#, r#"듑"#, r#"듑"#, r#"듑"#, r#"듑"#),
        (r#"듒"#, r#"듒"#, r#"듒"#, r#"듒"#, r#"듒"#),
        (r#"듓"#, r#"듓"#, r#"듓"#, r#"듓"#, r#"듓"#),
        (r#"듔"#, r#"듔"#, r#"듔"#, r#"듔"#, r#"듔"#),
        (r#"듕"#, r#"듕"#, r#"듕"#, r#"듕"#, r#"듕"#),
        (r#"듖"#, r#"듖"#, r#"듖"#, r#"듖"#, r#"듖"#),
        (r#"듗"#, r#"듗"#, r#"듗"#, r#"듗"#, r#"듗"#),
        (r#"듘"#, r#"듘"#, r#"듘"#, r#"듘"#, r#"듘"#),
        (r#"듙"#, r#"듙"#, r#"듙"#, r#"듙"#, r#"듙"#),
        (r#"듚"#, r#"듚"#, r#"듚"#, r#"듚"#, r#"듚"#),
        (r#"듛"#, r#"듛"#, r#"듛"#, r#"듛"#, r#"듛"#),
        (r#"드"#, r#"드"#, r#"드"#, r#"드"#, r#"드"#),
        (r#"득"#, r#"득"#, r#"득"#, r#"득"#, r#"득"#),
        (r#"듞"#, r#"듞"#, r#"듞"#, r#"듞"#, r#"듞"#),
        (r#"듟"#, r#"듟"#, r#"듟"#, r#"듟"#, r#"듟"#),
        (r#"든"#, r#"든"#, r#"든"#, r#"든"#, r#"든"#),
        (r#"듡"#, r#"듡"#, r#"듡"#, r#"듡"#, r#"듡"#),
        (r#"듢"#, r#"듢"#, r#"듢"#, r#"듢"#, r#"듢"#),
        (r#"듣"#, r#"듣"#, r#"듣"#, r#"듣"#, r#"듣"#),
        (r#"들"#, r#"들"#, r#"들"#, r#"들"#, r#"들"#),
        (r#"듥"#, r#"듥"#, r#"듥"#, r#"듥"#, r#"듥"#),
        (r#"듦"#, r#"듦"#, r#"듦"#, r#"듦"#, r#"듦"#),
        (r#"듧"#, r#"듧"#, r#"듧"#, r#"듧"#, r#"듧"#),
        (r#"듨"#, r#"듨"#, r#"듨"#, r#"듨"#, r#"듨"#),
        (r#"듩"#, r#"듩"#, r#"듩"#, r#"듩"#, r#"듩"#),
        (r#"듪"#, r#"듪"#, r#"듪"#, r#"듪"#, r#"듪"#),
        (r#"듫"#, r#"듫"#, r#"듫"#, r#"듫"#, r#"듫"#),
        (r#"듬"#, r#"듬"#, r#"듬"#, r#"듬"#, r#"듬"#),
        (r#"듭"#, r#"듭"#, r#"듭"#, r#"듭"#, r#"듭"#),
        (r#"듮"#, r#"듮"#, r#"듮"#, r#"듮"#, r#"듮"#),
        (r#"듯"#, r#"듯"#, r#"듯"#, r#"듯"#, r#"듯"#),
        (r#"듰"#, r#"듰"#, r#"듰"#, r#"듰"#, r#"듰"#),
        (r#"등"#, r#"등"#, r#"등"#, r#"등"#, r#"등"#),
        (r#"듲"#, r#"듲"#, r#"듲"#, r#"듲"#, r#"듲"#),
        (r#"듳"#, r#"듳"#, r#"듳"#, r#"듳"#, r#"듳"#),
        (r#"듴"#, r#"듴"#, r#"듴"#, r#"듴"#, r#"듴"#),
        (r#"듵"#, r#"듵"#, r#"듵"#, r#"듵"#, r#"듵"#),
        (r#"듶"#, r#"듶"#, r#"듶"#, r#"듶"#, r#"듶"#),
        (r#"듷"#, r#"듷"#, r#"듷"#, r#"듷"#, r#"듷"#),
        (r#"듸"#, r#"듸"#, r#"듸"#, r#"듸"#, r#"듸"#),
        (r#"듹"#, r#"듹"#, r#"듹"#, r#"듹"#, r#"듹"#),
        (r#"듺"#, r#"듺"#, r#"듺"#, r#"듺"#, r#"듺"#),
        (r#"듻"#, r#"듻"#, r#"듻"#, r#"듻"#, r#"듻"#),
        (r#"듼"#, r#"듼"#, r#"듼"#, r#"듼"#, r#"듼"#),
        (r#"듽"#, r#"듽"#, r#"듽"#, r#"듽"#, r#"듽"#),
        (r#"듾"#, r#"듾"#, r#"듾"#, r#"듾"#, r#"듾"#),
        (r#"듿"#, r#"듿"#, r#"듿"#, r#"듿"#, r#"듿"#),
        (r#"딀"#, r#"딀"#, r#"딀"#, r#"딀"#, r#"딀"#),
        (r#"딁"#, r#"딁"#, r#"딁"#, r#"딁"#, r#"딁"#),
        (r#"딂"#, r#"딂"#, r#"딂"#, r#"딂"#, r#"딂"#),
        (r#"딃"#, r#"딃"#, r#"딃"#, r#"딃"#, r#"딃"#),
        (r#"딄"#, r#"딄"#, r#"딄"#, r#"딄"#, r#"딄"#),
        (r#"딅"#, r#"딅"#, r#"딅"#, r#"딅"#, r#"딅"#),
        (r#"딆"#, r#"딆"#, r#"딆"#, r#"딆"#, r#"딆"#),
        (r#"딇"#, r#"딇"#, r#"딇"#, r#"딇"#, r#"딇"#),
        (r#"딈"#, r#"딈"#, r#"딈"#, r#"딈"#, r#"딈"#),
        (r#"딉"#, r#"딉"#, r#"딉"#, r#"딉"#, r#"딉"#),
        (r#"딊"#, r#"딊"#, r#"딊"#, r#"딊"#, r#"딊"#),
        (r#"딋"#, r#"딋"#, r#"딋"#, r#"딋"#, r#"딋"#),
        (r#"딌"#, r#"딌"#, r#"딌"#, r#"딌"#, r#"딌"#),
        (r#"딍"#, r#"딍"#, r#"딍"#, r#"딍"#, r#"딍"#),
        (r#"딎"#, r#"딎"#, r#"딎"#, r#"딎"#, r#"딎"#),
        (r#"딏"#, r#"딏"#, r#"딏"#, r#"딏"#, r#"딏"#),
        (r#"딐"#, r#"딐"#, r#"딐"#, r#"딐"#, r#"딐"#),
        (r#"딑"#, r#"딑"#, r#"딑"#, r#"딑"#, r#"딑"#),
        (r#"딒"#, r#"딒"#, r#"딒"#, r#"딒"#, r#"딒"#),
        (r#"딓"#, r#"딓"#, r#"딓"#, r#"딓"#, r#"딓"#),
        (r#"디"#, r#"디"#, r#"디"#, r#"디"#, r#"디"#),
        (r#"딕"#, r#"딕"#, r#"딕"#, r#"딕"#, r#"딕"#),
        (r#"딖"#, r#"딖"#, r#"딖"#, r#"딖"#, r#"딖"#),
        (r#"딗"#, r#"딗"#, r#"딗"#, r#"딗"#, r#"딗"#),
        (r#"딘"#, r#"딘"#, r#"딘"#, r#"딘"#, r#"딘"#),
        (r#"딙"#, r#"딙"#, r#"딙"#, r#"딙"#, r#"딙"#),
        (r#"딚"#, r#"딚"#, r#"딚"#, r#"딚"#, r#"딚"#),
        (r#"딛"#, r#"딛"#, r#"딛"#, r#"딛"#, r#"딛"#),
        (r#"딜"#, r#"딜"#, r#"딜"#, r#"딜"#, r#"딜"#),
        (r#"딝"#, r#"딝"#, r#"딝"#, r#"딝"#, r#"딝"#),
        (r#"딞"#, r#"딞"#, r#"딞"#, r#"딞"#, r#"딞"#),
        (r#"딟"#, r#"딟"#, r#"딟"#, r#"딟"#, r#"딟"#),
        (r#"딠"#, r#"딠"#, r#"딠"#, r#"딠"#, r#"딠"#),
        (r#"딡"#, r#"딡"#, r#"딡"#, r#"딡"#, r#"딡"#),
        (r#"딢"#, r#"딢"#, r#"딢"#, r#"딢"#, r#"딢"#),
        (r#"딣"#, r#"딣"#, r#"딣"#, r#"딣"#, r#"딣"#),
        (r#"딤"#, r#"딤"#, r#"딤"#, r#"딤"#, r#"딤"#),
        (r#"딥"#, r#"딥"#, r#"딥"#, r#"딥"#, r#"딥"#),
        (r#"딦"#, r#"딦"#, r#"딦"#, r#"딦"#, r#"딦"#),
        (r#"딧"#, r#"딧"#, r#"딧"#, r#"딧"#, r#"딧"#),
        (r#"딨"#, r#"딨"#, r#"딨"#, r#"딨"#, r#"딨"#),
        (r#"딩"#, r#"딩"#, r#"딩"#, r#"딩"#, r#"딩"#),
        (r#"딪"#, r#"딪"#, r#"딪"#, r#"딪"#, r#"딪"#),
        (r#"딫"#, r#"딫"#, r#"딫"#, r#"딫"#, r#"딫"#),
        (r#"딬"#, r#"딬"#, r#"딬"#, r#"딬"#, r#"딬"#),
        (r#"딭"#, r#"딭"#, r#"딭"#, r#"딭"#, r#"딭"#),
        (r#"딮"#, r#"딮"#, r#"딮"#, r#"딮"#, r#"딮"#),
        (r#"딯"#, r#"딯"#, r#"딯"#, r#"딯"#, r#"딯"#),
        (r#"따"#, r#"따"#, r#"따"#, r#"따"#, r#"따"#),
        (r#"딱"#, r#"딱"#, r#"딱"#, r#"딱"#, r#"딱"#),
        (r#"딲"#, r#"딲"#, r#"딲"#, r#"딲"#, r#"딲"#),
        (r#"딳"#, r#"딳"#, r#"딳"#, r#"딳"#, r#"딳"#),
        (r#"딴"#, r#"딴"#, r#"딴"#, r#"딴"#, r#"딴"#),
        (r#"딵"#, r#"딵"#, r#"딵"#, r#"딵"#, r#"딵"#),
        (r#"딶"#, r#"딶"#, r#"딶"#, r#"딶"#, r#"딶"#),
        (r#"딷"#, r#"딷"#, r#"딷"#, r#"딷"#, r#"딷"#),
        (r#"딸"#, r#"딸"#, r#"딸"#, r#"딸"#, r#"딸"#),
        (r#"딹"#, r#"딹"#, r#"딹"#, r#"딹"#, r#"딹"#),
        (r#"딺"#, r#"딺"#, r#"딺"#, r#"딺"#, r#"딺"#),
        (r#"딻"#, r#"딻"#, r#"딻"#, r#"딻"#, r#"딻"#),
        (r#"딼"#, r#"딼"#, r#"딼"#, r#"딼"#, r#"딼"#),
        (r#"딽"#, r#"딽"#, r#"딽"#, r#"딽"#, r#"딽"#),
        (r#"딾"#, r#"딾"#, r#"딾"#, r#"딾"#, r#"딾"#),
        (r#"딿"#, r#"딿"#, r#"딿"#, r#"딿"#, r#"딿"#),
        (r#"땀"#, r#"땀"#, r#"땀"#, r#"땀"#, r#"땀"#),
        (r#"땁"#, r#"땁"#, r#"땁"#, r#"땁"#, r#"땁"#),
        (r#"땂"#, r#"땂"#, r#"땂"#, r#"땂"#, r#"땂"#),
        (r#"땃"#, r#"땃"#, r#"땃"#, r#"땃"#, r#"땃"#),
        (r#"땄"#, r#"땄"#, r#"땄"#, r#"땄"#, r#"땄"#),
        (r#"땅"#, r#"땅"#, r#"땅"#, r#"땅"#, r#"땅"#),
        (r#"땆"#, r#"땆"#, r#"땆"#, r#"땆"#, r#"땆"#),
        (r#"땇"#, r#"땇"#, r#"땇"#, r#"땇"#, r#"땇"#),
        (r#"땈"#, r#"땈"#, r#"땈"#, r#"땈"#, r#"땈"#),
        (r#"땉"#, r#"땉"#, r#"땉"#, r#"땉"#, r#"땉"#),
        (r#"땊"#, r#"땊"#, r#"땊"#, r#"땊"#, r#"땊"#),
        (r#"땋"#, r#"땋"#, r#"땋"#, r#"땋"#, r#"땋"#),
        (r#"때"#, r#"때"#, r#"때"#, r#"때"#, r#"때"#),
        (r#"땍"#, r#"땍"#, r#"땍"#, r#"땍"#, r#"땍"#),
        (r#"땎"#, r#"땎"#, r#"땎"#, r#"땎"#, r#"땎"#),
        (r#"땏"#, r#"땏"#, r#"땏"#, r#"땏"#, r#"땏"#),
        (r#"땐"#, r#"땐"#, r#"땐"#, r#"땐"#, r#"땐"#),
        (r#"땑"#, r#"땑"#, r#"땑"#, r#"땑"#, r#"땑"#),
        (r#"땒"#, r#"땒"#, r#"땒"#, r#"땒"#, r#"땒"#),
        (r#"땓"#, r#"땓"#, r#"땓"#, r#"땓"#, r#"땓"#),
        (r#"땔"#, r#"땔"#, r#"땔"#, r#"땔"#, r#"땔"#),
        (r#"땕"#, r#"땕"#, r#"땕"#, r#"땕"#, r#"땕"#),
        (r#"땖"#, r#"땖"#, r#"땖"#, r#"땖"#, r#"땖"#),
        (r#"땗"#, r#"땗"#, r#"땗"#, r#"땗"#, r#"땗"#),
        (r#"땘"#, r#"땘"#, r#"땘"#, r#"땘"#, r#"땘"#),
        (r#"땙"#, r#"땙"#, r#"땙"#, r#"땙"#, r#"땙"#),
        (r#"땚"#, r#"땚"#, r#"땚"#, r#"땚"#, r#"땚"#),
        (r#"땛"#, r#"땛"#, r#"땛"#, r#"땛"#, r#"땛"#),
        (r#"땜"#, r#"땜"#, r#"땜"#, r#"땜"#, r#"땜"#),
        (r#"땝"#, r#"땝"#, r#"땝"#, r#"땝"#, r#"땝"#),
        (r#"땞"#, r#"땞"#, r#"땞"#, r#"땞"#, r#"땞"#),
        (r#"땟"#, r#"땟"#, r#"땟"#, r#"땟"#, r#"땟"#),
        (r#"땠"#, r#"땠"#, r#"땠"#, r#"땠"#, r#"땠"#),
        (r#"땡"#, r#"땡"#, r#"땡"#, r#"땡"#, r#"땡"#),
        (r#"땢"#, r#"땢"#, r#"땢"#, r#"땢"#, r#"땢"#),
        (r#"땣"#, r#"땣"#, r#"땣"#, r#"땣"#, r#"땣"#),
        (r#"땤"#, r#"땤"#, r#"땤"#, r#"땤"#, r#"땤"#),
        (r#"땥"#, r#"땥"#, r#"땥"#, r#"땥"#, r#"땥"#),
        (r#"땦"#, r#"땦"#, r#"땦"#, r#"땦"#, r#"땦"#),
        (r#"땧"#, r#"땧"#, r#"땧"#, r#"땧"#, r#"땧"#),
        (r#"땨"#, r#"땨"#, r#"땨"#, r#"땨"#, r#"땨"#),
        (r#"땩"#, r#"땩"#, r#"땩"#, r#"땩"#, r#"땩"#),
        (r#"땪"#, r#"땪"#, r#"땪"#, r#"땪"#, r#"땪"#),
        (r#"땫"#, r#"땫"#, r#"땫"#, r#"땫"#, r#"땫"#),
        (r#"땬"#, r#"땬"#, r#"땬"#, r#"땬"#, r#"땬"#),
        (r#"땭"#, r#"땭"#, r#"땭"#, r#"땭"#, r#"땭"#),
        (r#"땮"#, r#"땮"#, r#"땮"#, r#"땮"#, r#"땮"#),
        (r#"땯"#, r#"땯"#, r#"땯"#, r#"땯"#, r#"땯"#),
        (r#"땰"#, r#"땰"#, r#"땰"#, r#"땰"#, r#"땰"#),
        (r#"땱"#, r#"땱"#, r#"땱"#, r#"땱"#, r#"땱"#),
        (r#"땲"#, r#"땲"#, r#"땲"#, r#"땲"#, r#"땲"#),
        (r#"땳"#, r#"땳"#, r#"땳"#, r#"땳"#, r#"땳"#),
        (r#"땴"#, r#"땴"#, r#"땴"#, r#"땴"#, r#"땴"#),
        (r#"땵"#, r#"땵"#, r#"땵"#, r#"땵"#, r#"땵"#),
        (r#"땶"#, r#"땶"#, r#"땶"#, r#"땶"#, r#"땶"#),
        (r#"땷"#, r#"땷"#, r#"땷"#, r#"땷"#, r#"땷"#),
        (r#"땸"#, r#"땸"#, r#"땸"#, r#"땸"#, r#"땸"#),
        (r#"땹"#, r#"땹"#, r#"땹"#, r#"땹"#, r#"땹"#),
        (r#"땺"#, r#"땺"#, r#"땺"#, r#"땺"#, r#"땺"#),
        (r#"땻"#, r#"땻"#, r#"땻"#, r#"땻"#, r#"땻"#),
        (r#"땼"#, r#"땼"#, r#"땼"#, r#"땼"#, r#"땼"#),
        (r#"땽"#, r#"땽"#, r#"땽"#, r#"땽"#, r#"땽"#),
        (r#"땾"#, r#"땾"#, r#"땾"#, r#"땾"#, r#"땾"#),
        (r#"땿"#, r#"땿"#, r#"땿"#, r#"땿"#, r#"땿"#),
        (r#"떀"#, r#"떀"#, r#"떀"#, r#"떀"#, r#"떀"#),
        (r#"떁"#, r#"떁"#, r#"떁"#, r#"떁"#, r#"떁"#),
        (r#"떂"#, r#"떂"#, r#"떂"#, r#"떂"#, r#"떂"#),
        (r#"떃"#, r#"떃"#, r#"떃"#, r#"떃"#, r#"떃"#),
        (r#"떄"#, r#"떄"#, r#"떄"#, r#"떄"#, r#"떄"#),
        (r#"떅"#, r#"떅"#, r#"떅"#, r#"떅"#, r#"떅"#),
        (r#"떆"#, r#"떆"#, r#"떆"#, r#"떆"#, r#"떆"#),
        (r#"떇"#, r#"떇"#, r#"떇"#, r#"떇"#, r#"떇"#),
        (r#"떈"#, r#"떈"#, r#"떈"#, r#"떈"#, r#"떈"#),
        (r#"떉"#, r#"떉"#, r#"떉"#, r#"떉"#, r#"떉"#),
        (r#"떊"#, r#"떊"#, r#"떊"#, r#"떊"#, r#"떊"#),
        (r#"떋"#, r#"떋"#, r#"떋"#, r#"떋"#, r#"떋"#),
        (r#"떌"#, r#"떌"#, r#"떌"#, r#"떌"#, r#"떌"#),
        (r#"떍"#, r#"떍"#, r#"떍"#, r#"떍"#, r#"떍"#),
        (r#"떎"#, r#"떎"#, r#"떎"#, r#"떎"#, r#"떎"#),
        (r#"떏"#, r#"떏"#, r#"떏"#, r#"떏"#, r#"떏"#),
        (r#"떐"#, r#"떐"#, r#"떐"#, r#"떐"#, r#"떐"#),
        (r#"떑"#, r#"떑"#, r#"떑"#, r#"떑"#, r#"떑"#),
        (r#"떒"#, r#"떒"#, r#"떒"#, r#"떒"#, r#"떒"#),
        (r#"떓"#, r#"떓"#, r#"떓"#, r#"떓"#, r#"떓"#),
        (r#"떔"#, r#"떔"#, r#"떔"#, r#"떔"#, r#"떔"#),
        (r#"떕"#, r#"떕"#, r#"떕"#, r#"떕"#, r#"떕"#),
        (r#"떖"#, r#"떖"#, r#"떖"#, r#"떖"#, r#"떖"#),
        (r#"떗"#, r#"떗"#, r#"떗"#, r#"떗"#, r#"떗"#),
        (r#"떘"#, r#"떘"#, r#"떘"#, r#"떘"#, r#"떘"#),
        (r#"떙"#, r#"떙"#, r#"떙"#, r#"떙"#, r#"떙"#),
        (r#"떚"#, r#"떚"#, r#"떚"#, r#"떚"#, r#"떚"#),
        (r#"떛"#, r#"떛"#, r#"떛"#, r#"떛"#, r#"떛"#),
        (r#"떜"#, r#"떜"#, r#"떜"#, r#"떜"#, r#"떜"#),
        (r#"떝"#, r#"떝"#, r#"떝"#, r#"떝"#, r#"떝"#),
        (r#"떞"#, r#"떞"#, r#"떞"#, r#"떞"#, r#"떞"#),
        (r#"떟"#, r#"떟"#, r#"떟"#, r#"떟"#, r#"떟"#),
        (r#"떠"#, r#"떠"#, r#"떠"#, r#"떠"#, r#"떠"#),
        (r#"떡"#, r#"떡"#, r#"떡"#, r#"떡"#, r#"떡"#),
        (r#"떢"#, r#"떢"#, r#"떢"#, r#"떢"#, r#"떢"#),
        (r#"떣"#, r#"떣"#, r#"떣"#, r#"떣"#, r#"떣"#),
        (r#"떤"#, r#"떤"#, r#"떤"#, r#"떤"#, r#"떤"#),
        (r#"떥"#, r#"떥"#, r#"떥"#, r#"떥"#, r#"떥"#),
        (r#"떦"#, r#"떦"#, r#"떦"#, r#"떦"#, r#"떦"#),
        (r#"떧"#, r#"떧"#, r#"떧"#, r#"떧"#, r#"떧"#),
        (r#"떨"#, r#"떨"#, r#"떨"#, r#"떨"#, r#"떨"#),
        (r#"떩"#, r#"떩"#, r#"떩"#, r#"떩"#, r#"떩"#),
        (r#"떪"#, r#"떪"#, r#"떪"#, r#"떪"#, r#"떪"#),
        (r#"떫"#, r#"떫"#, r#"떫"#, r#"떫"#, r#"떫"#),
        (r#"떬"#, r#"떬"#, r#"떬"#, r#"떬"#, r#"떬"#),
        (r#"떭"#, r#"떭"#, r#"떭"#, r#"떭"#, r#"떭"#),
        (r#"떮"#, r#"떮"#, r#"떮"#, r#"떮"#, r#"떮"#),
        (r#"떯"#, r#"떯"#, r#"떯"#, r#"떯"#, r#"떯"#),
        (r#"떰"#, r#"떰"#, r#"떰"#, r#"떰"#, r#"떰"#),
        (r#"떱"#, r#"떱"#, r#"떱"#, r#"떱"#, r#"떱"#),
        (r#"떲"#, r#"떲"#, r#"떲"#, r#"떲"#, r#"떲"#),
        (r#"떳"#, r#"떳"#, r#"떳"#, r#"떳"#, r#"떳"#),
        (r#"떴"#, r#"떴"#, r#"떴"#, r#"떴"#, r#"떴"#),
        (r#"떵"#, r#"떵"#, r#"떵"#, r#"떵"#, r#"떵"#),
        (r#"떶"#, r#"떶"#, r#"떶"#, r#"떶"#, r#"떶"#),
        (r#"떷"#, r#"떷"#, r#"떷"#, r#"떷"#, r#"떷"#),
        (r#"떸"#, r#"떸"#, r#"떸"#, r#"떸"#, r#"떸"#),
        (r#"떹"#, r#"떹"#, r#"떹"#, r#"떹"#, r#"떹"#),
        (r#"떺"#, r#"떺"#, r#"떺"#, r#"떺"#, r#"떺"#),
        (r#"떻"#, r#"떻"#, r#"떻"#, r#"떻"#, r#"떻"#),
        (r#"떼"#, r#"떼"#, r#"떼"#, r#"떼"#, r#"떼"#),
        (r#"떽"#, r#"떽"#, r#"떽"#, r#"떽"#, r#"떽"#),
        (r#"떾"#, r#"떾"#, r#"떾"#, r#"떾"#, r#"떾"#),
        (r#"떿"#, r#"떿"#, r#"떿"#, r#"떿"#, r#"떿"#),
        (r#"뗀"#, r#"뗀"#, r#"뗀"#, r#"뗀"#, r#"뗀"#),
        (r#"뗁"#, r#"뗁"#, r#"뗁"#, r#"뗁"#, r#"뗁"#),
        (r#"뗂"#, r#"뗂"#, r#"뗂"#, r#"뗂"#, r#"뗂"#),
        (r#"뗃"#, r#"뗃"#, r#"뗃"#, r#"뗃"#, r#"뗃"#),
        (r#"뗄"#, r#"뗄"#, r#"뗄"#, r#"뗄"#, r#"뗄"#),
        (r#"뗅"#, r#"뗅"#, r#"뗅"#, r#"뗅"#, r#"뗅"#),
        (r#"뗆"#, r#"뗆"#, r#"뗆"#, r#"뗆"#, r#"뗆"#),
        (r#"뗇"#, r#"뗇"#, r#"뗇"#, r#"뗇"#, r#"뗇"#),
        (r#"뗈"#, r#"뗈"#, r#"뗈"#, r#"뗈"#, r#"뗈"#),
        (r#"뗉"#, r#"뗉"#, r#"뗉"#, r#"뗉"#, r#"뗉"#),
        (r#"뗊"#, r#"뗊"#, r#"뗊"#, r#"뗊"#, r#"뗊"#),
        (r#"뗋"#, r#"뗋"#, r#"뗋"#, r#"뗋"#, r#"뗋"#),
        (r#"뗌"#, r#"뗌"#, r#"뗌"#, r#"뗌"#, r#"뗌"#),
        (r#"뗍"#, r#"뗍"#, r#"뗍"#, r#"뗍"#, r#"뗍"#),
        (r#"뗎"#, r#"뗎"#, r#"뗎"#, r#"뗎"#, r#"뗎"#),
        (r#"뗏"#, r#"뗏"#, r#"뗏"#, r#"뗏"#, r#"뗏"#),
        (r#"뗐"#, r#"뗐"#, r#"뗐"#, r#"뗐"#, r#"뗐"#),
        (r#"뗑"#, r#"뗑"#, r#"뗑"#, r#"뗑"#, r#"뗑"#),
        (r#"뗒"#, r#"뗒"#, r#"뗒"#, r#"뗒"#, r#"뗒"#),
        (r#"뗓"#, r#"뗓"#, r#"뗓"#, r#"뗓"#, r#"뗓"#),
        (r#"뗔"#, r#"뗔"#, r#"뗔"#, r#"뗔"#, r#"뗔"#),
        (r#"뗕"#, r#"뗕"#, r#"뗕"#, r#"뗕"#, r#"뗕"#),
        (r#"뗖"#, r#"뗖"#, r#"뗖"#, r#"뗖"#, r#"뗖"#),
        (r#"뗗"#, r#"뗗"#, r#"뗗"#, r#"뗗"#, r#"뗗"#),
        (r#"뗘"#, r#"뗘"#, r#"뗘"#, r#"뗘"#, r#"뗘"#),
        (r#"뗙"#, r#"뗙"#, r#"뗙"#, r#"뗙"#, r#"뗙"#),
        (r#"뗚"#, r#"뗚"#, r#"뗚"#, r#"뗚"#, r#"뗚"#),
        (r#"뗛"#, r#"뗛"#, r#"뗛"#, r#"뗛"#, r#"뗛"#),
        (r#"뗜"#, r#"뗜"#, r#"뗜"#, r#"뗜"#, r#"뗜"#),
        (r#"뗝"#, r#"뗝"#, r#"뗝"#, r#"뗝"#, r#"뗝"#),
        (r#"뗞"#, r#"뗞"#, r#"뗞"#, r#"뗞"#, r#"뗞"#),
        (r#"뗟"#, r#"뗟"#, r#"뗟"#, r#"뗟"#, r#"뗟"#),
        (r#"뗠"#, r#"뗠"#, r#"뗠"#, r#"뗠"#, r#"뗠"#),
        (r#"뗡"#, r#"뗡"#, r#"뗡"#, r#"뗡"#, r#"뗡"#),
        (r#"뗢"#, r#"뗢"#, r#"뗢"#, r#"뗢"#, r#"뗢"#),
        (r#"뗣"#, r#"뗣"#, r#"뗣"#, r#"뗣"#, r#"뗣"#),
        (r#"뗤"#, r#"뗤"#, r#"뗤"#, r#"뗤"#, r#"뗤"#),
        (r#"뗥"#, r#"뗥"#, r#"뗥"#, r#"뗥"#, r#"뗥"#),
        (r#"뗦"#, r#"뗦"#, r#"뗦"#, r#"뗦"#, r#"뗦"#),
        (r#"뗧"#, r#"뗧"#, r#"뗧"#, r#"뗧"#, r#"뗧"#),
        (r#"뗨"#, r#"뗨"#, r#"뗨"#, r#"뗨"#, r#"뗨"#),
        (r#"뗩"#, r#"뗩"#, r#"뗩"#, r#"뗩"#, r#"뗩"#),
        (r#"뗪"#, r#"뗪"#, r#"뗪"#, r#"뗪"#, r#"뗪"#),
        (r#"뗫"#, r#"뗫"#, r#"뗫"#, r#"뗫"#, r#"뗫"#),
        (r#"뗬"#, r#"뗬"#, r#"뗬"#, r#"뗬"#, r#"뗬"#),
        (r#"뗭"#, r#"뗭"#, r#"뗭"#, r#"뗭"#, r#"뗭"#),
        (r#"뗮"#, r#"뗮"#, r#"뗮"#, r#"뗮"#, r#"뗮"#),
        (r#"뗯"#, r#"뗯"#, r#"뗯"#, r#"뗯"#, r#"뗯"#),
        (r#"뗰"#, r#"뗰"#, r#"뗰"#, r#"뗰"#, r#"뗰"#),
        (r#"뗱"#, r#"뗱"#, r#"뗱"#, r#"뗱"#, r#"뗱"#),
        (r#"뗲"#, r#"뗲"#, r#"뗲"#, r#"뗲"#, r#"뗲"#),
        (r#"뗳"#, r#"뗳"#, r#"뗳"#, r#"뗳"#, r#"뗳"#),
        (r#"뗴"#, r#"뗴"#, r#"뗴"#, r#"뗴"#, r#"뗴"#),
        (r#"뗵"#, r#"뗵"#, r#"뗵"#, r#"뗵"#, r#"뗵"#),
        (r#"뗶"#, r#"뗶"#, r#"뗶"#, r#"뗶"#, r#"뗶"#),
        (r#"뗷"#, r#"뗷"#, r#"뗷"#, r#"뗷"#, r#"뗷"#),
        (r#"뗸"#, r#"뗸"#, r#"뗸"#, r#"뗸"#, r#"뗸"#),
        (r#"뗹"#, r#"뗹"#, r#"뗹"#, r#"뗹"#, r#"뗹"#),
        (r#"뗺"#, r#"뗺"#, r#"뗺"#, r#"뗺"#, r#"뗺"#),
        (r#"뗻"#, r#"뗻"#, r#"뗻"#, r#"뗻"#, r#"뗻"#),
        (r#"뗼"#, r#"뗼"#, r#"뗼"#, r#"뗼"#, r#"뗼"#),
        (r#"뗽"#, r#"뗽"#, r#"뗽"#, r#"뗽"#, r#"뗽"#),
        (r#"뗾"#, r#"뗾"#, r#"뗾"#, r#"뗾"#, r#"뗾"#),
        (r#"뗿"#, r#"뗿"#, r#"뗿"#, r#"뗿"#, r#"뗿"#),
        (r#"똀"#, r#"똀"#, r#"똀"#, r#"똀"#, r#"똀"#),
        (r#"똁"#, r#"똁"#, r#"똁"#, r#"똁"#, r#"똁"#),
        (r#"똂"#, r#"똂"#, r#"똂"#, r#"똂"#, r#"똂"#),
        (r#"똃"#, r#"똃"#, r#"똃"#, r#"똃"#, r#"똃"#),
        (r#"똄"#, r#"똄"#, r#"똄"#, r#"똄"#, r#"똄"#),
        (r#"똅"#, r#"똅"#, r#"똅"#, r#"똅"#, r#"똅"#),
        (r#"똆"#, r#"똆"#, r#"똆"#, r#"똆"#, r#"똆"#),
        (r#"똇"#, r#"똇"#, r#"똇"#, r#"똇"#, r#"똇"#),
        (r#"똈"#, r#"똈"#, r#"똈"#, r#"똈"#, r#"똈"#),
        (r#"똉"#, r#"똉"#, r#"똉"#, r#"똉"#, r#"똉"#),
        (r#"똊"#, r#"똊"#, r#"똊"#, r#"똊"#, r#"똊"#),
        (r#"똋"#, r#"똋"#, r#"똋"#, r#"똋"#, r#"똋"#),
        (r#"똌"#, r#"똌"#, r#"똌"#, r#"똌"#, r#"똌"#),
        (r#"똍"#, r#"똍"#, r#"똍"#, r#"똍"#, r#"똍"#),
        (r#"똎"#, r#"똎"#, r#"똎"#, r#"똎"#, r#"똎"#),
        (r#"똏"#, r#"똏"#, r#"똏"#, r#"똏"#, r#"똏"#),
        (r#"또"#, r#"또"#, r#"또"#, r#"또"#, r#"또"#),
        (r#"똑"#, r#"똑"#, r#"똑"#, r#"똑"#, r#"똑"#),
        (r#"똒"#, r#"똒"#, r#"똒"#, r#"똒"#, r#"똒"#),
        (r#"똓"#, r#"똓"#, r#"똓"#, r#"똓"#, r#"똓"#),
        (r#"똔"#, r#"똔"#, r#"똔"#, r#"똔"#, r#"똔"#),
        (r#"똕"#, r#"똕"#, r#"똕"#, r#"똕"#, r#"똕"#),
        (r#"똖"#, r#"똖"#, r#"똖"#, r#"똖"#, r#"똖"#),
        (r#"똗"#, r#"똗"#, r#"똗"#, r#"똗"#, r#"똗"#),
        (r#"똘"#, r#"똘"#, r#"똘"#, r#"똘"#, r#"똘"#),
        (r#"똙"#, r#"똙"#, r#"똙"#, r#"똙"#, r#"똙"#),
        (r#"똚"#, r#"똚"#, r#"똚"#, r#"똚"#, r#"똚"#),
        (r#"똛"#, r#"똛"#, r#"똛"#, r#"똛"#, r#"똛"#),
        (r#"똜"#, r#"똜"#, r#"똜"#, r#"똜"#, r#"똜"#),
        (r#"똝"#, r#"똝"#, r#"똝"#, r#"똝"#, r#"똝"#),
        (r#"똞"#, r#"똞"#, r#"똞"#, r#"똞"#, r#"똞"#),
        (r#"똟"#, r#"똟"#, r#"똟"#, r#"똟"#, r#"똟"#),
        (r#"똠"#, r#"똠"#, r#"똠"#, r#"똠"#, r#"똠"#),
        (r#"똡"#, r#"똡"#, r#"똡"#, r#"똡"#, r#"똡"#),
        (r#"똢"#, r#"똢"#, r#"똢"#, r#"똢"#, r#"똢"#),
        (r#"똣"#, r#"똣"#, r#"똣"#, r#"똣"#, r#"똣"#),
        (r#"똤"#, r#"똤"#, r#"똤"#, r#"똤"#, r#"똤"#),
        (r#"똥"#, r#"똥"#, r#"똥"#, r#"똥"#, r#"똥"#),
        (r#"똦"#, r#"똦"#, r#"똦"#, r#"똦"#, r#"똦"#),
        (r#"똧"#, r#"똧"#, r#"똧"#, r#"똧"#, r#"똧"#),
        (r#"똨"#, r#"똨"#, r#"똨"#, r#"똨"#, r#"똨"#),
        (r#"똩"#, r#"똩"#, r#"똩"#, r#"똩"#, r#"똩"#),
        (r#"똪"#, r#"똪"#, r#"똪"#, r#"똪"#, r#"똪"#),
        (r#"똫"#, r#"똫"#, r#"똫"#, r#"똫"#, r#"똫"#),
        (r#"똬"#, r#"똬"#, r#"똬"#, r#"똬"#, r#"똬"#),
        (r#"똭"#, r#"똭"#, r#"똭"#, r#"똭"#, r#"똭"#),
        (r#"똮"#, r#"똮"#, r#"똮"#, r#"똮"#, r#"똮"#),
        (r#"똯"#, r#"똯"#, r#"똯"#, r#"똯"#, r#"똯"#),
        (r#"똰"#, r#"똰"#, r#"똰"#, r#"똰"#, r#"똰"#),
        (r#"똱"#, r#"똱"#, r#"똱"#, r#"똱"#, r#"똱"#),
        (r#"똲"#, r#"똲"#, r#"똲"#, r#"똲"#, r#"똲"#),
        (r#"똳"#, r#"똳"#, r#"똳"#, r#"똳"#, r#"똳"#),
        (r#"똴"#, r#"똴"#, r#"똴"#, r#"똴"#, r#"똴"#),
        (r#"똵"#, r#"똵"#, r#"똵"#, r#"똵"#, r#"똵"#),
        (r#"똶"#, r#"똶"#, r#"똶"#, r#"똶"#, r#"똶"#),
        (r#"똷"#, r#"똷"#, r#"똷"#, r#"똷"#, r#"똷"#),
        (r#"똸"#, r#"똸"#, r#"똸"#, r#"똸"#, r#"똸"#),
        (r#"똹"#, r#"똹"#, r#"똹"#, r#"똹"#, r#"똹"#),
        (r#"똺"#, r#"똺"#, r#"똺"#, r#"똺"#, r#"똺"#),
        (r#"똻"#, r#"똻"#, r#"똻"#, r#"똻"#, r#"똻"#),
        (r#"똼"#, r#"똼"#, r#"똼"#, r#"똼"#, r#"똼"#),
        (r#"똽"#, r#"똽"#, r#"똽"#, r#"똽"#, r#"똽"#),
        (r#"똾"#, r#"똾"#, r#"똾"#, r#"똾"#, r#"똾"#),
        (r#"똿"#, r#"똿"#, r#"똿"#, r#"똿"#, r#"똿"#),
        (r#"뙀"#, r#"뙀"#, r#"뙀"#, r#"뙀"#, r#"뙀"#),
        (r#"뙁"#, r#"뙁"#, r#"뙁"#, r#"뙁"#, r#"뙁"#),
        (r#"뙂"#, r#"뙂"#, r#"뙂"#, r#"뙂"#, r#"뙂"#),
        (r#"뙃"#, r#"뙃"#, r#"뙃"#, r#"뙃"#, r#"뙃"#),
        (r#"뙄"#, r#"뙄"#, r#"뙄"#, r#"뙄"#, r#"뙄"#),
        (r#"뙅"#, r#"뙅"#, r#"뙅"#, r#"뙅"#, r#"뙅"#),
        (r#"뙆"#, r#"뙆"#, r#"뙆"#, r#"뙆"#, r#"뙆"#),
        (r#"뙇"#, r#"뙇"#, r#"뙇"#, r#"뙇"#, r#"뙇"#),
        (r#"뙈"#, r#"뙈"#, r#"뙈"#, r#"뙈"#, r#"뙈"#),
        (r#"뙉"#, r#"뙉"#, r#"뙉"#, r#"뙉"#, r#"뙉"#),
        (r#"뙊"#, r#"뙊"#, r#"뙊"#, r#"뙊"#, r#"뙊"#),
        (r#"뙋"#, r#"뙋"#, r#"뙋"#, r#"뙋"#, r#"뙋"#),
        (r#"뙌"#, r#"뙌"#, r#"뙌"#, r#"뙌"#, r#"뙌"#),
        (r#"뙍"#, r#"뙍"#, r#"뙍"#, r#"뙍"#, r#"뙍"#),
        (r#"뙎"#, r#"뙎"#, r#"뙎"#, r#"뙎"#, r#"뙎"#),
        (r#"뙏"#, r#"뙏"#, r#"뙏"#, r#"뙏"#, r#"뙏"#),
        (r#"뙐"#, r#"뙐"#, r#"뙐"#, r#"뙐"#, r#"뙐"#),
        (r#"뙑"#, r#"뙑"#, r#"뙑"#, r#"뙑"#, r#"뙑"#),
        (r#"뙒"#, r#"뙒"#, r#"뙒"#, r#"뙒"#, r#"뙒"#),
        (r#"뙓"#, r#"뙓"#, r#"뙓"#, r#"뙓"#, r#"뙓"#),
        (r#"뙔"#, r#"뙔"#, r#"뙔"#, r#"뙔"#, r#"뙔"#),
        (r#"뙕"#, r#"뙕"#, r#"뙕"#, r#"뙕"#, r#"뙕"#),
        (r#"뙖"#, r#"뙖"#, r#"뙖"#, r#"뙖"#, r#"뙖"#),
        (r#"뙗"#, r#"뙗"#, r#"뙗"#, r#"뙗"#, r#"뙗"#),
        (r#"뙘"#, r#"뙘"#, r#"뙘"#, r#"뙘"#, r#"뙘"#),
        (r#"뙙"#, r#"뙙"#, r#"뙙"#, r#"뙙"#, r#"뙙"#),
        (r#"뙚"#, r#"뙚"#, r#"뙚"#, r#"뙚"#, r#"뙚"#),
        (r#"뙛"#, r#"뙛"#, r#"뙛"#, r#"뙛"#, r#"뙛"#),
        (r#"뙜"#, r#"뙜"#, r#"뙜"#, r#"뙜"#, r#"뙜"#),
        (r#"뙝"#, r#"뙝"#, r#"뙝"#, r#"뙝"#, r#"뙝"#),
        (r#"뙞"#, r#"뙞"#, r#"뙞"#, r#"뙞"#, r#"뙞"#),
        (r#"뙟"#, r#"뙟"#, r#"뙟"#, r#"뙟"#, r#"뙟"#),
        (r#"뙠"#, r#"뙠"#, r#"뙠"#, r#"뙠"#, r#"뙠"#),
        (r#"뙡"#, r#"뙡"#, r#"뙡"#, r#"뙡"#, r#"뙡"#),
        (r#"뙢"#, r#"뙢"#, r#"뙢"#, r#"뙢"#, r#"뙢"#),
        (r#"뙣"#, r#"뙣"#, r#"뙣"#, r#"뙣"#, r#"뙣"#),
        (r#"뙤"#, r#"뙤"#, r#"뙤"#, r#"뙤"#, r#"뙤"#),
        (r#"뙥"#, r#"뙥"#, r#"뙥"#, r#"뙥"#, r#"뙥"#),
        (r#"뙦"#, r#"뙦"#, r#"뙦"#, r#"뙦"#, r#"뙦"#),
        (r#"뙧"#, r#"뙧"#, r#"뙧"#, r#"뙧"#, r#"뙧"#),
        (r#"뙨"#, r#"뙨"#, r#"뙨"#, r#"뙨"#, r#"뙨"#),
        (r#"뙩"#, r#"뙩"#, r#"뙩"#, r#"뙩"#, r#"뙩"#),
        (r#"뙪"#, r#"뙪"#, r#"뙪"#, r#"뙪"#, r#"뙪"#),
        (r#"뙫"#, r#"뙫"#, r#"뙫"#, r#"뙫"#, r#"뙫"#),
        (r#"뙬"#, r#"뙬"#, r#"뙬"#, r#"뙬"#, r#"뙬"#),
        (r#"뙭"#, r#"뙭"#, r#"뙭"#, r#"뙭"#, r#"뙭"#),
        (r#"뙮"#, r#"뙮"#, r#"뙮"#, r#"뙮"#, r#"뙮"#),
        (r#"뙯"#, r#"뙯"#, r#"뙯"#, r#"뙯"#, r#"뙯"#),
        (r#"뙰"#, r#"뙰"#, r#"뙰"#, r#"뙰"#, r#"뙰"#),
        (r#"뙱"#, r#"뙱"#, r#"뙱"#, r#"뙱"#, r#"뙱"#),
        (r#"뙲"#, r#"뙲"#, r#"뙲"#, r#"뙲"#, r#"뙲"#),
        (r#"뙳"#, r#"뙳"#, r#"뙳"#, r#"뙳"#, r#"뙳"#),
        (r#"뙴"#, r#"뙴"#, r#"뙴"#, r#"뙴"#, r#"뙴"#),
        (r#"뙵"#, r#"뙵"#, r#"뙵"#, r#"뙵"#, r#"뙵"#),
        (r#"뙶"#, r#"뙶"#, r#"뙶"#, r#"뙶"#, r#"뙶"#),
        (r#"뙷"#, r#"뙷"#, r#"뙷"#, r#"뙷"#, r#"뙷"#),
        (r#"뙸"#, r#"뙸"#, r#"뙸"#, r#"뙸"#, r#"뙸"#),
        (r#"뙹"#, r#"뙹"#, r#"뙹"#, r#"뙹"#, r#"뙹"#),
        (r#"뙺"#, r#"뙺"#, r#"뙺"#, r#"뙺"#, r#"뙺"#),
        (r#"뙻"#, r#"뙻"#, r#"뙻"#, r#"뙻"#, r#"뙻"#),
        (r#"뙼"#, r#"뙼"#, r#"뙼"#, r#"뙼"#, r#"뙼"#),
        (r#"뙽"#, r#"뙽"#, r#"뙽"#, r#"뙽"#, r#"뙽"#),
        (r#"뙾"#, r#"뙾"#, r#"뙾"#, r#"뙾"#, r#"뙾"#),
        (r#"뙿"#, r#"뙿"#, r#"뙿"#, r#"뙿"#, r#"뙿"#),
        (r#"뚀"#, r#"뚀"#, r#"뚀"#, r#"뚀"#, r#"뚀"#),
        (r#"뚁"#, r#"뚁"#, r#"뚁"#, r#"뚁"#, r#"뚁"#),
        (r#"뚂"#, r#"뚂"#, r#"뚂"#, r#"뚂"#, r#"뚂"#),
        (r#"뚃"#, r#"뚃"#, r#"뚃"#, r#"뚃"#, r#"뚃"#),
        (r#"뚄"#, r#"뚄"#, r#"뚄"#, r#"뚄"#, r#"뚄"#),
        (r#"뚅"#, r#"뚅"#, r#"뚅"#, r#"뚅"#, r#"뚅"#),
        (r#"뚆"#, r#"뚆"#, r#"뚆"#, r#"뚆"#, r#"뚆"#),
        (r#"뚇"#, r#"뚇"#, r#"뚇"#, r#"뚇"#, r#"뚇"#),
        (r#"뚈"#, r#"뚈"#, r#"뚈"#, r#"뚈"#, r#"뚈"#),
        (r#"뚉"#, r#"뚉"#, r#"뚉"#, r#"뚉"#, r#"뚉"#),
        (r#"뚊"#, r#"뚊"#, r#"뚊"#, r#"뚊"#, r#"뚊"#),
        (r#"뚋"#, r#"뚋"#, r#"뚋"#, r#"뚋"#, r#"뚋"#),
        (r#"뚌"#, r#"뚌"#, r#"뚌"#, r#"뚌"#, r#"뚌"#),
        (r#"뚍"#, r#"뚍"#, r#"뚍"#, r#"뚍"#, r#"뚍"#),
        (r#"뚎"#, r#"뚎"#, r#"뚎"#, r#"뚎"#, r#"뚎"#),
        (r#"뚏"#, r#"뚏"#, r#"뚏"#, r#"뚏"#, r#"뚏"#),
        (r#"뚐"#, r#"뚐"#, r#"뚐"#, r#"뚐"#, r#"뚐"#),
        (r#"뚑"#, r#"뚑"#, r#"뚑"#, r#"뚑"#, r#"뚑"#),
        (r#"뚒"#, r#"뚒"#, r#"뚒"#, r#"뚒"#, r#"뚒"#),
        (r#"뚓"#, r#"뚓"#, r#"뚓"#, r#"뚓"#, r#"뚓"#),
        (r#"뚔"#, r#"뚔"#, r#"뚔"#, r#"뚔"#, r#"뚔"#),
        (r#"뚕"#, r#"뚕"#, r#"뚕"#, r#"뚕"#, r#"뚕"#),
        (r#"뚖"#, r#"뚖"#, r#"뚖"#, r#"뚖"#, r#"뚖"#),
        (r#"뚗"#, r#"뚗"#, r#"뚗"#, r#"뚗"#, r#"뚗"#),
        (r#"뚘"#, r#"뚘"#, r#"뚘"#, r#"뚘"#, r#"뚘"#),
        (r#"뚙"#, r#"뚙"#, r#"뚙"#, r#"뚙"#, r#"뚙"#),
        (r#"뚚"#, r#"뚚"#, r#"뚚"#, r#"뚚"#, r#"뚚"#),
        (r#"뚛"#, r#"뚛"#, r#"뚛"#, r#"뚛"#, r#"뚛"#),
        (r#"뚜"#, r#"뚜"#, r#"뚜"#, r#"뚜"#, r#"뚜"#),
        (r#"뚝"#, r#"뚝"#, r#"뚝"#, r#"뚝"#, r#"뚝"#),
        (r#"뚞"#, r#"뚞"#, r#"뚞"#, r#"뚞"#, r#"뚞"#),
        (r#"뚟"#, r#"뚟"#, r#"뚟"#, r#"뚟"#, r#"뚟"#),
        (r#"뚠"#, r#"뚠"#, r#"뚠"#, r#"뚠"#, r#"뚠"#),
        (r#"뚡"#, r#"뚡"#, r#"뚡"#, r#"뚡"#, r#"뚡"#),
        (r#"뚢"#, r#"뚢"#, r#"뚢"#, r#"뚢"#, r#"뚢"#),
        (r#"뚣"#, r#"뚣"#, r#"뚣"#, r#"뚣"#, r#"뚣"#),
        (r#"뚤"#, r#"뚤"#, r#"뚤"#, r#"뚤"#, r#"뚤"#),
        (r#"뚥"#, r#"뚥"#, r#"뚥"#, r#"뚥"#, r#"뚥"#),
        (r#"뚦"#, r#"뚦"#, r#"뚦"#, r#"뚦"#, r#"뚦"#),
        (r#"뚧"#, r#"뚧"#, r#"뚧"#, r#"뚧"#, r#"뚧"#),
        (r#"뚨"#, r#"뚨"#, r#"뚨"#, r#"뚨"#, r#"뚨"#),
        (r#"뚩"#, r#"뚩"#, r#"뚩"#, r#"뚩"#, r#"뚩"#),
        (r#"뚪"#, r#"뚪"#, r#"뚪"#, r#"뚪"#, r#"뚪"#),
        (r#"뚫"#, r#"뚫"#, r#"뚫"#, r#"뚫"#, r#"뚫"#),
        (r#"뚬"#, r#"뚬"#, r#"뚬"#, r#"뚬"#, r#"뚬"#),
        (r#"뚭"#, r#"뚭"#, r#"뚭"#, r#"뚭"#, r#"뚭"#),
        (r#"뚮"#, r#"뚮"#, r#"뚮"#, r#"뚮"#, r#"뚮"#),
        (r#"뚯"#, r#"뚯"#, r#"뚯"#, r#"뚯"#, r#"뚯"#),
        (r#"뚰"#, r#"뚰"#, r#"뚰"#, r#"뚰"#, r#"뚰"#),
        (r#"뚱"#, r#"뚱"#, r#"뚱"#, r#"뚱"#, r#"뚱"#),
        (r#"뚲"#, r#"뚲"#, r#"뚲"#, r#"뚲"#, r#"뚲"#),
        (r#"뚳"#, r#"뚳"#, r#"뚳"#, r#"뚳"#, r#"뚳"#),
        (r#"뚴"#, r#"뚴"#, r#"뚴"#, r#"뚴"#, r#"뚴"#),
        (r#"뚵"#, r#"뚵"#, r#"뚵"#, r#"뚵"#, r#"뚵"#),
        (r#"뚶"#, r#"뚶"#, r#"뚶"#, r#"뚶"#, r#"뚶"#),
        (r#"뚷"#, r#"뚷"#, r#"뚷"#, r#"뚷"#, r#"뚷"#),
        (r#"뚸"#, r#"뚸"#, r#"뚸"#, r#"뚸"#, r#"뚸"#),
        (r#"뚹"#, r#"뚹"#, r#"뚹"#, r#"뚹"#, r#"뚹"#),
        (r#"뚺"#, r#"뚺"#, r#"뚺"#, r#"뚺"#, r#"뚺"#),
        (r#"뚻"#, r#"뚻"#, r#"뚻"#, r#"뚻"#, r#"뚻"#),
        (r#"뚼"#, r#"뚼"#, r#"뚼"#, r#"뚼"#, r#"뚼"#),
        (r#"뚽"#, r#"뚽"#, r#"뚽"#, r#"뚽"#, r#"뚽"#),
        (r#"뚾"#, r#"뚾"#, r#"뚾"#, r#"뚾"#, r#"뚾"#),
        (r#"뚿"#, r#"뚿"#, r#"뚿"#, r#"뚿"#, r#"뚿"#),
        (r#"뛀"#, r#"뛀"#, r#"뛀"#, r#"뛀"#, r#"뛀"#),
        (r#"뛁"#, r#"뛁"#, r#"뛁"#, r#"뛁"#, r#"뛁"#),
        (r#"뛂"#, r#"뛂"#, r#"뛂"#, r#"뛂"#, r#"뛂"#),
        (r#"뛃"#, r#"뛃"#, r#"뛃"#, r#"뛃"#, r#"뛃"#),
        (r#"뛄"#, r#"뛄"#, r#"뛄"#, r#"뛄"#, r#"뛄"#),
        (r#"뛅"#, r#"뛅"#, r#"뛅"#, r#"뛅"#, r#"뛅"#),
        (r#"뛆"#, r#"뛆"#, r#"뛆"#, r#"뛆"#, r#"뛆"#),
        (r#"뛇"#, r#"뛇"#, r#"뛇"#, r#"뛇"#, r#"뛇"#),
        (r#"뛈"#, r#"뛈"#, r#"뛈"#, r#"뛈"#, r#"뛈"#),
        (r#"뛉"#, r#"뛉"#, r#"뛉"#, r#"뛉"#, r#"뛉"#),
        (r#"뛊"#, r#"뛊"#, r#"뛊"#, r#"뛊"#, r#"뛊"#),
        (r#"뛋"#, r#"뛋"#, r#"뛋"#, r#"뛋"#, r#"뛋"#),
        (r#"뛌"#, r#"뛌"#, r#"뛌"#, r#"뛌"#, r#"뛌"#),
        (r#"뛍"#, r#"뛍"#, r#"뛍"#, r#"뛍"#, r#"뛍"#),
        (r#"뛎"#, r#"뛎"#, r#"뛎"#, r#"뛎"#, r#"뛎"#),
        (r#"뛏"#, r#"뛏"#, r#"뛏"#, r#"뛏"#, r#"뛏"#),
        (r#"뛐"#, r#"뛐"#, r#"뛐"#, r#"뛐"#, r#"뛐"#),
        (r#"뛑"#, r#"뛑"#, r#"뛑"#, r#"뛑"#, r#"뛑"#),
        (r#"뛒"#, r#"뛒"#, r#"뛒"#, r#"뛒"#, r#"뛒"#),
        (r#"뛓"#, r#"뛓"#, r#"뛓"#, r#"뛓"#, r#"뛓"#),
        (r#"뛔"#, r#"뛔"#, r#"뛔"#, r#"뛔"#, r#"뛔"#),
        (r#"뛕"#, r#"뛕"#, r#"뛕"#, r#"뛕"#, r#"뛕"#),
        (r#"뛖"#, r#"뛖"#, r#"뛖"#, r#"뛖"#, r#"뛖"#),
        (r#"뛗"#, r#"뛗"#, r#"뛗"#, r#"뛗"#, r#"뛗"#),
        (r#"뛘"#, r#"뛘"#, r#"뛘"#, r#"뛘"#, r#"뛘"#),
        (r#"뛙"#, r#"뛙"#, r#"뛙"#, r#"뛙"#, r#"뛙"#),
        (r#"뛚"#, r#"뛚"#, r#"뛚"#, r#"뛚"#, r#"뛚"#),
        (r#"뛛"#, r#"뛛"#, r#"뛛"#, r#"뛛"#, r#"뛛"#),
        (r#"뛜"#, r#"뛜"#, r#"뛜"#, r#"뛜"#, r#"뛜"#),
        (r#"뛝"#, r#"뛝"#, r#"뛝"#, r#"뛝"#, r#"뛝"#),
        (r#"뛞"#, r#"뛞"#, r#"뛞"#, r#"뛞"#, r#"뛞"#),
        (r#"뛟"#, r#"뛟"#, r#"뛟"#, r#"뛟"#, r#"뛟"#),
        (r#"뛠"#, r#"뛠"#, r#"뛠"#, r#"뛠"#, r#"뛠"#),
        (r#"뛡"#, r#"뛡"#, r#"뛡"#, r#"뛡"#, r#"뛡"#),
        (r#"뛢"#, r#"뛢"#, r#"뛢"#, r#"뛢"#, r#"뛢"#),
        (r#"뛣"#, r#"뛣"#, r#"뛣"#, r#"뛣"#, r#"뛣"#),
        (r#"뛤"#, r#"뛤"#, r#"뛤"#, r#"뛤"#, r#"뛤"#),
        (r#"뛥"#, r#"뛥"#, r#"뛥"#, r#"뛥"#, r#"뛥"#),
        (r#"뛦"#, r#"뛦"#, r#"뛦"#, r#"뛦"#, r#"뛦"#),
        (r#"뛧"#, r#"뛧"#, r#"뛧"#, r#"뛧"#, r#"뛧"#),
        (r#"뛨"#, r#"뛨"#, r#"뛨"#, r#"뛨"#, r#"뛨"#),
        (r#"뛩"#, r#"뛩"#, r#"뛩"#, r#"뛩"#, r#"뛩"#),
        (r#"뛪"#, r#"뛪"#, r#"뛪"#, r#"뛪"#, r#"뛪"#),
        (r#"뛫"#, r#"뛫"#, r#"뛫"#, r#"뛫"#, r#"뛫"#),
        (r#"뛬"#, r#"뛬"#, r#"뛬"#, r#"뛬"#, r#"뛬"#),
        (r#"뛭"#, r#"뛭"#, r#"뛭"#, r#"뛭"#, r#"뛭"#),
        (r#"뛮"#, r#"뛮"#, r#"뛮"#, r#"뛮"#, r#"뛮"#),
        (r#"뛯"#, r#"뛯"#, r#"뛯"#, r#"뛯"#, r#"뛯"#),
        (r#"뛰"#, r#"뛰"#, r#"뛰"#, r#"뛰"#, r#"뛰"#),
        (r#"뛱"#, r#"뛱"#, r#"뛱"#, r#"뛱"#, r#"뛱"#),
        (r#"뛲"#, r#"뛲"#, r#"뛲"#, r#"뛲"#, r#"뛲"#),
        (r#"뛳"#, r#"뛳"#, r#"뛳"#, r#"뛳"#, r#"뛳"#),
        (r#"뛴"#, r#"뛴"#, r#"뛴"#, r#"뛴"#, r#"뛴"#),
        (r#"뛵"#, r#"뛵"#, r#"뛵"#, r#"뛵"#, r#"뛵"#),
        (r#"뛶"#, r#"뛶"#, r#"뛶"#, r#"뛶"#, r#"뛶"#),
        (r#"뛷"#, r#"뛷"#, r#"뛷"#, r#"뛷"#, r#"뛷"#),
        (r#"뛸"#, r#"뛸"#, r#"뛸"#, r#"뛸"#, r#"뛸"#),
        (r#"뛹"#, r#"뛹"#, r#"뛹"#, r#"뛹"#, r#"뛹"#),
        (r#"뛺"#, r#"뛺"#, r#"뛺"#, r#"뛺"#, r#"뛺"#),
        (r#"뛻"#, r#"뛻"#, r#"뛻"#, r#"뛻"#, r#"뛻"#),
        (r#"뛼"#, r#"뛼"#, r#"뛼"#, r#"뛼"#, r#"뛼"#),
        (r#"뛽"#, r#"뛽"#, r#"뛽"#, r#"뛽"#, r#"뛽"#),
        (r#"뛾"#, r#"뛾"#, r#"뛾"#, r#"뛾"#, r#"뛾"#),
        (r#"뛿"#, r#"뛿"#, r#"뛿"#, r#"뛿"#, r#"뛿"#),
        (r#"뜀"#, r#"뜀"#, r#"뜀"#, r#"뜀"#, r#"뜀"#),
        (r#"뜁"#, r#"뜁"#, r#"뜁"#, r#"뜁"#, r#"뜁"#),
        (r#"뜂"#, r#"뜂"#, r#"뜂"#, r#"뜂"#, r#"뜂"#),
        (r#"뜃"#, r#"뜃"#, r#"뜃"#, r#"뜃"#, r#"뜃"#),
        (r#"뜄"#, r#"뜄"#, r#"뜄"#, r#"뜄"#, r#"뜄"#),
        (r#"뜅"#, r#"뜅"#, r#"뜅"#, r#"뜅"#, r#"뜅"#),
        (r#"뜆"#, r#"뜆"#, r#"뜆"#, r#"뜆"#, r#"뜆"#),
        (r#"뜇"#, r#"뜇"#, r#"뜇"#, r#"뜇"#, r#"뜇"#),
        (r#"뜈"#, r#"뜈"#, r#"뜈"#, r#"뜈"#, r#"뜈"#),
        (r#"뜉"#, r#"뜉"#, r#"뜉"#, r#"뜉"#, r#"뜉"#),
        (r#"뜊"#, r#"뜊"#, r#"뜊"#, r#"뜊"#, r#"뜊"#),
        (r#"뜋"#, r#"뜋"#, r#"뜋"#, r#"뜋"#, r#"뜋"#),
        (r#"뜌"#, r#"뜌"#, r#"뜌"#, r#"뜌"#, r#"뜌"#),
        (r#"뜍"#, r#"뜍"#, r#"뜍"#, r#"뜍"#, r#"뜍"#),
        (r#"뜎"#, r#"뜎"#, r#"뜎"#, r#"뜎"#, r#"뜎"#),
        (r#"뜏"#, r#"뜏"#, r#"뜏"#, r#"뜏"#, r#"뜏"#),
        (r#"뜐"#, r#"뜐"#, r#"뜐"#, r#"뜐"#, r#"뜐"#),
        (r#"뜑"#, r#"뜑"#, r#"뜑"#, r#"뜑"#, r#"뜑"#),
        (r#"뜒"#, r#"뜒"#, r#"뜒"#, r#"뜒"#, r#"뜒"#),
        (r#"뜓"#, r#"뜓"#, r#"뜓"#, r#"뜓"#, r#"뜓"#),
        (r#"뜔"#, r#"뜔"#, r#"뜔"#, r#"뜔"#, r#"뜔"#),
        (r#"뜕"#, r#"뜕"#, r#"뜕"#, r#"뜕"#, r#"뜕"#),
        (r#"뜖"#, r#"뜖"#, r#"뜖"#, r#"뜖"#, r#"뜖"#),
        (r#"뜗"#, r#"뜗"#, r#"뜗"#, r#"뜗"#, r#"뜗"#),
        (r#"뜘"#, r#"뜘"#, r#"뜘"#, r#"뜘"#, r#"뜘"#),
        (r#"뜙"#, r#"뜙"#, r#"뜙"#, r#"뜙"#, r#"뜙"#),
        (r#"뜚"#, r#"뜚"#, r#"뜚"#, r#"뜚"#, r#"뜚"#),
        (r#"뜛"#, r#"뜛"#, r#"뜛"#, r#"뜛"#, r#"뜛"#),
        (r#"뜜"#, r#"뜜"#, r#"뜜"#, r#"뜜"#, r#"뜜"#),
        (r#"뜝"#, r#"뜝"#, r#"뜝"#, r#"뜝"#, r#"뜝"#),
        (r#"뜞"#, r#"뜞"#, r#"뜞"#, r#"뜞"#, r#"뜞"#),
        (r#"뜟"#, r#"뜟"#, r#"뜟"#, r#"뜟"#, r#"뜟"#),
        (r#"뜠"#, r#"뜠"#, r#"뜠"#, r#"뜠"#, r#"뜠"#),
        (r#"뜡"#, r#"뜡"#, r#"뜡"#, r#"뜡"#, r#"뜡"#),
        (r#"뜢"#, r#"뜢"#, r#"뜢"#, r#"뜢"#, r#"뜢"#),
        (r#"뜣"#, r#"뜣"#, r#"뜣"#, r#"뜣"#, r#"뜣"#),
        (r#"뜤"#, r#"뜤"#, r#"뜤"#, r#"뜤"#, r#"뜤"#),
        (r#"뜥"#, r#"뜥"#, r#"뜥"#, r#"뜥"#, r#"뜥"#),
        (r#"뜦"#, r#"뜦"#, r#"뜦"#, r#"뜦"#, r#"뜦"#),
        (r#"뜧"#, r#"뜧"#, r#"뜧"#, r#"뜧"#, r#"뜧"#),
        (r#"뜨"#, r#"뜨"#, r#"뜨"#, r#"뜨"#, r#"뜨"#),
        (r#"뜩"#, r#"뜩"#, r#"뜩"#, r#"뜩"#, r#"뜩"#),
        (r#"뜪"#, r#"뜪"#, r#"뜪"#, r#"뜪"#, r#"뜪"#),
        (r#"뜫"#, r#"뜫"#, r#"뜫"#, r#"뜫"#, r#"뜫"#),
        (r#"뜬"#, r#"뜬"#, r#"뜬"#, r#"뜬"#, r#"뜬"#),
        (r#"뜭"#, r#"뜭"#, r#"뜭"#, r#"뜭"#, r#"뜭"#),
        (r#"뜮"#, r#"뜮"#, r#"뜮"#, r#"뜮"#, r#"뜮"#),
        (r#"뜯"#, r#"뜯"#, r#"뜯"#, r#"뜯"#, r#"뜯"#),
        (r#"뜰"#, r#"뜰"#, r#"뜰"#, r#"뜰"#, r#"뜰"#),
        (r#"뜱"#, r#"뜱"#, r#"뜱"#, r#"뜱"#, r#"뜱"#),
        (r#"뜲"#, r#"뜲"#, r#"뜲"#, r#"뜲"#, r#"뜲"#),
        (r#"뜳"#, r#"뜳"#, r#"뜳"#, r#"뜳"#, r#"뜳"#),
        (r#"뜴"#, r#"뜴"#, r#"뜴"#, r#"뜴"#, r#"뜴"#),
        (r#"뜵"#, r#"뜵"#, r#"뜵"#, r#"뜵"#, r#"뜵"#),
        (r#"뜶"#, r#"뜶"#, r#"뜶"#, r#"뜶"#, r#"뜶"#),
        (r#"뜷"#, r#"뜷"#, r#"뜷"#, r#"뜷"#, r#"뜷"#),
        (r#"뜸"#, r#"뜸"#, r#"뜸"#, r#"뜸"#, r#"뜸"#),
        (r#"뜹"#, r#"뜹"#, r#"뜹"#, r#"뜹"#, r#"뜹"#),
        (r#"뜺"#, r#"뜺"#, r#"뜺"#, r#"뜺"#, r#"뜺"#),
        (r#"뜻"#, r#"뜻"#, r#"뜻"#, r#"뜻"#, r#"뜻"#),
        (r#"뜼"#, r#"뜼"#, r#"뜼"#, r#"뜼"#, r#"뜼"#),
        (r#"뜽"#, r#"뜽"#, r#"뜽"#, r#"뜽"#, r#"뜽"#),
        (r#"뜾"#, r#"뜾"#, r#"뜾"#, r#"뜾"#, r#"뜾"#),
        (r#"뜿"#, r#"뜿"#, r#"뜿"#, r#"뜿"#, r#"뜿"#),
        (r#"띀"#, r#"띀"#, r#"띀"#, r#"띀"#, r#"띀"#),
        (r#"띁"#, r#"띁"#, r#"띁"#, r#"띁"#, r#"띁"#),
        (r#"띂"#, r#"띂"#, r#"띂"#, r#"띂"#, r#"띂"#),
        (r#"띃"#, r#"띃"#, r#"띃"#, r#"띃"#, r#"띃"#),
        (r#"띄"#, r#"띄"#, r#"띄"#, r#"띄"#, r#"띄"#),
        (r#"띅"#, r#"띅"#, r#"띅"#, r#"띅"#, r#"띅"#),
        (r#"띆"#, r#"띆"#, r#"띆"#, r#"띆"#, r#"띆"#),
        (r#"띇"#, r#"띇"#, r#"띇"#, r#"띇"#, r#"띇"#),
        (r#"띈"#, r#"띈"#, r#"띈"#, r#"띈"#, r#"띈"#),
        (r#"띉"#, r#"띉"#, r#"띉"#, r#"띉"#, r#"띉"#),
        (r#"띊"#, r#"띊"#, r#"띊"#, r#"띊"#, r#"띊"#),
        (r#"띋"#, r#"띋"#, r#"띋"#, r#"띋"#, r#"띋"#),
        (r#"띌"#, r#"띌"#, r#"띌"#, r#"띌"#, r#"띌"#),
        (r#"띍"#, r#"띍"#, r#"띍"#, r#"띍"#, r#"띍"#),
        (r#"띎"#, r#"띎"#, r#"띎"#, r#"띎"#, r#"띎"#),
        (r#"띏"#, r#"띏"#, r#"띏"#, r#"띏"#, r#"띏"#),
        (r#"띐"#, r#"띐"#, r#"띐"#, r#"띐"#, r#"띐"#),
        (r#"띑"#, r#"띑"#, r#"띑"#, r#"띑"#, r#"띑"#),
        (r#"띒"#, r#"띒"#, r#"띒"#, r#"띒"#, r#"띒"#),
        (r#"띓"#, r#"띓"#, r#"띓"#, r#"띓"#, r#"띓"#),
        (r#"띔"#, r#"띔"#, r#"띔"#, r#"띔"#, r#"띔"#),
        (r#"띕"#, r#"띕"#, r#"띕"#, r#"띕"#, r#"띕"#),
        (r#"띖"#, r#"띖"#, r#"띖"#, r#"띖"#, r#"띖"#),
        (r#"띗"#, r#"띗"#, r#"띗"#, r#"띗"#, r#"띗"#),
        (r#"띘"#, r#"띘"#, r#"띘"#, r#"띘"#, r#"띘"#),
        (r#"띙"#, r#"띙"#, r#"띙"#, r#"띙"#, r#"띙"#),
        (r#"띚"#, r#"띚"#, r#"띚"#, r#"띚"#, r#"띚"#),
        (r#"띛"#, r#"띛"#, r#"띛"#, r#"띛"#, r#"띛"#),
        (r#"띜"#, r#"띜"#, r#"띜"#, r#"띜"#, r#"띜"#),
        (r#"띝"#, r#"띝"#, r#"띝"#, r#"띝"#, r#"띝"#),
        (r#"띞"#, r#"띞"#, r#"띞"#, r#"띞"#, r#"띞"#),
        (r#"띟"#, r#"띟"#, r#"띟"#, r#"띟"#, r#"띟"#),
        (r#"띠"#, r#"띠"#, r#"띠"#, r#"띠"#, r#"띠"#),
        (r#"띡"#, r#"띡"#, r#"띡"#, r#"띡"#, r#"띡"#),
        (r#"띢"#, r#"띢"#, r#"띢"#, r#"띢"#, r#"띢"#),
        (r#"띣"#, r#"띣"#, r#"띣"#, r#"띣"#, r#"띣"#),
        (r#"띤"#, r#"띤"#, r#"띤"#, r#"띤"#, r#"띤"#),
        (r#"띥"#, r#"띥"#, r#"띥"#, r#"띥"#, r#"띥"#),
        (r#"띦"#, r#"띦"#, r#"띦"#, r#"띦"#, r#"띦"#),
        (r#"띧"#, r#"띧"#, r#"띧"#, r#"띧"#, r#"띧"#),
        (r#"띨"#, r#"띨"#, r#"띨"#, r#"띨"#, r#"띨"#),
        (r#"띩"#, r#"띩"#, r#"띩"#, r#"띩"#, r#"띩"#),
        (r#"띪"#, r#"띪"#, r#"띪"#, r#"띪"#, r#"띪"#),
        (r#"띫"#, r#"띫"#, r#"띫"#, r#"띫"#, r#"띫"#),
        (r#"띬"#, r#"띬"#, r#"띬"#, r#"띬"#, r#"띬"#),
        (r#"띭"#, r#"띭"#, r#"띭"#, r#"띭"#, r#"띭"#),
        (r#"띮"#, r#"띮"#, r#"띮"#, r#"띮"#, r#"띮"#),
        (r#"띯"#, r#"띯"#, r#"띯"#, r#"띯"#, r#"띯"#),
        (r#"띰"#, r#"띰"#, r#"띰"#, r#"띰"#, r#"띰"#),
        (r#"띱"#, r#"띱"#, r#"띱"#, r#"띱"#, r#"띱"#),
        (r#"띲"#, r#"띲"#, r#"띲"#, r#"띲"#, r#"띲"#),
        (r#"띳"#, r#"띳"#, r#"띳"#, r#"띳"#, r#"띳"#),
        (r#"띴"#, r#"띴"#, r#"띴"#, r#"띴"#, r#"띴"#),
        (r#"띵"#, r#"띵"#, r#"띵"#, r#"띵"#, r#"띵"#),
        (r#"띶"#, r#"띶"#, r#"띶"#, r#"띶"#, r#"띶"#),
        (r#"띷"#, r#"띷"#, r#"띷"#, r#"띷"#, r#"띷"#),
        (r#"띸"#, r#"띸"#, r#"띸"#, r#"띸"#, r#"띸"#),
        (r#"띹"#, r#"띹"#, r#"띹"#, r#"띹"#, r#"띹"#),
        (r#"띺"#, r#"띺"#, r#"띺"#, r#"띺"#, r#"띺"#),
        (r#"띻"#, r#"띻"#, r#"띻"#, r#"띻"#, r#"띻"#),
        (r#"라"#, r#"라"#, r#"라"#, r#"라"#, r#"라"#),
        (r#"락"#, r#"락"#, r#"락"#, r#"락"#, r#"락"#),
        (r#"띾"#, r#"띾"#, r#"띾"#, r#"띾"#, r#"띾"#),
        (r#"띿"#, r#"띿"#, r#"띿"#, r#"띿"#, r#"띿"#),
        (r#"란"#, r#"란"#, r#"란"#, r#"란"#, r#"란"#),
        (r#"랁"#, r#"랁"#, r#"랁"#, r#"랁"#, r#"랁"#),
        (r#"랂"#, r#"랂"#, r#"랂"#, r#"랂"#, r#"랂"#),
        (r#"랃"#, r#"랃"#, r#"랃"#, r#"랃"#, r#"랃"#),
        (r#"랄"#, r#"랄"#, r#"랄"#, r#"랄"#, r#"랄"#),
        (r#"랅"#, r#"랅"#, r#"랅"#, r#"랅"#, r#"랅"#),
        (r#"랆"#, r#"랆"#, r#"랆"#, r#"랆"#, r#"랆"#),
        (r#"랇"#, r#"랇"#, r#"랇"#, r#"랇"#, r#"랇"#),
        (r#"랈"#, r#"랈"#, r#"랈"#, r#"랈"#, r#"랈"#),
        (r#"랉"#, r#"랉"#, r#"랉"#, r#"랉"#, r#"랉"#),
        (r#"랊"#, r#"랊"#, r#"랊"#, r#"랊"#, r#"랊"#),
        (r#"랋"#, r#"랋"#, r#"랋"#, r#"랋"#, r#"랋"#),
        (r#"람"#, r#"람"#, r#"람"#, r#"람"#, r#"람"#),
        (r#"랍"#, r#"랍"#, r#"랍"#, r#"랍"#, r#"랍"#),
        (r#"랎"#, r#"랎"#, r#"랎"#, r#"랎"#, r#"랎"#),
        (r#"랏"#, r#"랏"#, r#"랏"#, r#"랏"#, r#"랏"#),
        (r#"랐"#, r#"랐"#, r#"랐"#, r#"랐"#, r#"랐"#),
        (r#"랑"#, r#"랑"#, r#"랑"#, r#"랑"#, r#"랑"#),
        (r#"랒"#, r#"랒"#, r#"랒"#, r#"랒"#, r#"랒"#),
        (r#"랓"#, r#"랓"#, r#"랓"#, r#"랓"#, r#"랓"#),
        (r#"랔"#, r#"랔"#, r#"랔"#, r#"랔"#, r#"랔"#),
        (r#"랕"#, r#"랕"#, r#"랕"#, r#"랕"#, r#"랕"#),
        (r#"랖"#, r#"랖"#, r#"랖"#, r#"랖"#, r#"랖"#),
        (r#"랗"#, r#"랗"#, r#"랗"#, r#"랗"#, r#"랗"#),
        (r#"래"#, r#"래"#, r#"래"#, r#"래"#, r#"래"#),
        (r#"랙"#, r#"랙"#, r#"랙"#, r#"랙"#, r#"랙"#),
        (r#"랚"#, r#"랚"#, r#"랚"#, r#"랚"#, r#"랚"#),
        (r#"랛"#, r#"랛"#, r#"랛"#, r#"랛"#, r#"랛"#),
        (r#"랜"#, r#"랜"#, r#"랜"#, r#"랜"#, r#"랜"#),
        (r#"랝"#, r#"랝"#, r#"랝"#, r#"랝"#, r#"랝"#),
        (r#"랞"#, r#"랞"#, r#"랞"#, r#"랞"#, r#"랞"#),
        (r#"랟"#, r#"랟"#, r#"랟"#, r#"랟"#, r#"랟"#),
        (r#"랠"#, r#"랠"#, r#"랠"#, r#"랠"#, r#"랠"#),
        (r#"랡"#, r#"랡"#, r#"랡"#, r#"랡"#, r#"랡"#),
        (r#"랢"#, r#"랢"#, r#"랢"#, r#"랢"#, r#"랢"#),
        (r#"랣"#, r#"랣"#, r#"랣"#, r#"랣"#, r#"랣"#),
        (r#"랤"#, r#"랤"#, r#"랤"#, r#"랤"#, r#"랤"#),
        (r#"랥"#, r#"랥"#, r#"랥"#, r#"랥"#, r#"랥"#),
        (r#"랦"#, r#"랦"#, r#"랦"#, r#"랦"#, r#"랦"#),
        (r#"랧"#, r#"랧"#, r#"랧"#, r#"랧"#, r#"랧"#),
        (r#"램"#, r#"램"#, r#"램"#, r#"램"#, r#"램"#),
        (r#"랩"#, r#"랩"#, r#"랩"#, r#"랩"#, r#"랩"#),
        (r#"랪"#, r#"랪"#, r#"랪"#, r#"랪"#, r#"랪"#),
        (r#"랫"#, r#"랫"#, r#"랫"#, r#"랫"#, r#"랫"#),
        (r#"랬"#, r#"랬"#, r#"랬"#, r#"랬"#, r#"랬"#),
        (r#"랭"#, r#"랭"#, r#"랭"#, r#"랭"#, r#"랭"#),
        (r#"랮"#, r#"랮"#, r#"랮"#, r#"랮"#, r#"랮"#),
        (r#"랯"#, r#"랯"#, r#"랯"#, r#"랯"#, r#"랯"#),
        (r#"랰"#, r#"랰"#, r#"랰"#, r#"랰"#, r#"랰"#),
        (r#"랱"#, r#"랱"#, r#"랱"#, r#"랱"#, r#"랱"#),
        (r#"랲"#, r#"랲"#, r#"랲"#, r#"랲"#, r#"랲"#),
        (r#"랳"#, r#"랳"#, r#"랳"#, r#"랳"#, r#"랳"#),
        (r#"랴"#, r#"랴"#, r#"랴"#, r#"랴"#, r#"랴"#),
        (r#"략"#, r#"략"#, r#"략"#, r#"략"#, r#"략"#),
        (r#"랶"#, r#"랶"#, r#"랶"#, r#"랶"#, r#"랶"#),
        (r#"랷"#, r#"랷"#, r#"랷"#, r#"랷"#, r#"랷"#),
        (r#"랸"#, r#"랸"#, r#"랸"#, r#"랸"#, r#"랸"#),
        (r#"랹"#, r#"랹"#, r#"랹"#, r#"랹"#, r#"랹"#),
        (r#"랺"#, r#"랺"#, r#"랺"#, r#"랺"#, r#"랺"#),
        (r#"랻"#, r#"랻"#, r#"랻"#, r#"랻"#, r#"랻"#),
        (r#"랼"#, r#"랼"#, r#"랼"#, r#"랼"#, r#"랼"#),
        (r#"랽"#, r#"랽"#, r#"랽"#, r#"랽"#, r#"랽"#),
        (r#"랾"#, r#"랾"#, r#"랾"#, r#"랾"#, r#"랾"#),
        (r#"랿"#, r#"랿"#, r#"랿"#, r#"랿"#, r#"랿"#),
        (r#"럀"#, r#"럀"#, r#"럀"#, r#"럀"#, r#"럀"#),
        (r#"럁"#, r#"럁"#, r#"럁"#, r#"럁"#, r#"럁"#),
        (r#"럂"#, r#"럂"#, r#"럂"#, r#"럂"#, r#"럂"#),
        (r#"럃"#, r#"럃"#, r#"럃"#, r#"럃"#, r#"럃"#),
        (r#"럄"#, r#"럄"#, r#"럄"#, r#"럄"#, r#"럄"#),
        (r#"럅"#, r#"럅"#, r#"럅"#, r#"럅"#, r#"럅"#),
        (r#"럆"#, r#"럆"#, r#"럆"#, r#"럆"#, r#"럆"#),
        (r#"럇"#, r#"럇"#, r#"럇"#, r#"럇"#, r#"럇"#),
        (r#"럈"#, r#"럈"#, r#"럈"#, r#"럈"#, r#"럈"#),
        (r#"량"#, r#"량"#, r#"량"#, r#"량"#, r#"량"#),
        (r#"럊"#, r#"럊"#, r#"럊"#, r#"럊"#, r#"럊"#),
        (r#"럋"#, r#"럋"#, r#"럋"#, r#"럋"#, r#"럋"#),
        (r#"럌"#, r#"럌"#, r#"럌"#, r#"럌"#, r#"럌"#),
        (r#"럍"#, r#"럍"#, r#"럍"#, r#"럍"#, r#"럍"#),
        (r#"럎"#, r#"럎"#, r#"럎"#, r#"럎"#, r#"럎"#),
        (r#"럏"#, r#"럏"#, r#"럏"#, r#"럏"#, r#"럏"#),
        (r#"럐"#, r#"럐"#, r#"럐"#, r#"럐"#, r#"럐"#),
        (r#"럑"#, r#"럑"#, r#"럑"#, r#"럑"#, r#"럑"#),
        (r#"럒"#, r#"럒"#, r#"럒"#, r#"럒"#, r#"럒"#),
        (r#"럓"#, r#"럓"#, r#"럓"#, r#"럓"#, r#"럓"#),
        (r#"럔"#, r#"럔"#, r#"럔"#, r#"럔"#, r#"럔"#),
        (r#"럕"#, r#"럕"#, r#"럕"#, r#"럕"#, r#"럕"#),
        (r#"럖"#, r#"럖"#, r#"럖"#, r#"럖"#, r#"럖"#),
        (r#"럗"#, r#"럗"#, r#"럗"#, r#"럗"#, r#"럗"#),
        (r#"럘"#, r#"럘"#, r#"럘"#, r#"럘"#, r#"럘"#),
        (r#"럙"#, r#"럙"#, r#"럙"#, r#"럙"#, r#"럙"#),
        (r#"럚"#, r#"럚"#, r#"럚"#, r#"럚"#, r#"럚"#),
        (r#"럛"#, r#"럛"#, r#"럛"#, r#"럛"#, r#"럛"#),
        (r#"럜"#, r#"럜"#, r#"럜"#, r#"럜"#, r#"럜"#),
        (r#"럝"#, r#"럝"#, r#"럝"#, r#"럝"#, r#"럝"#),
        (r#"럞"#, r#"럞"#, r#"럞"#, r#"럞"#, r#"럞"#),
        (r#"럟"#, r#"럟"#, r#"럟"#, r#"럟"#, r#"럟"#),
        (r#"럠"#, r#"럠"#, r#"럠"#, r#"럠"#, r#"럠"#),
        (r#"럡"#, r#"럡"#, r#"럡"#, r#"럡"#, r#"럡"#),
        (r#"럢"#, r#"럢"#, r#"럢"#, r#"럢"#, r#"럢"#),
        (r#"럣"#, r#"럣"#, r#"럣"#, r#"럣"#, r#"럣"#),
        (r#"럤"#, r#"럤"#, r#"럤"#, r#"럤"#, r#"럤"#),
        (r#"럥"#, r#"럥"#, r#"럥"#, r#"럥"#, r#"럥"#),
        (r#"럦"#, r#"럦"#, r#"럦"#, r#"럦"#, r#"럦"#),
        (r#"럧"#, r#"럧"#, r#"럧"#, r#"럧"#, r#"럧"#),
        (r#"럨"#, r#"럨"#, r#"럨"#, r#"럨"#, r#"럨"#),
        (r#"럩"#, r#"럩"#, r#"럩"#, r#"럩"#, r#"럩"#),
        (r#"럪"#, r#"럪"#, r#"럪"#, r#"럪"#, r#"럪"#),
        (r#"럫"#, r#"럫"#, r#"럫"#, r#"럫"#, r#"럫"#),
        (r#"러"#, r#"러"#, r#"러"#, r#"러"#, r#"러"#),
        (r#"럭"#, r#"럭"#, r#"럭"#, r#"럭"#, r#"럭"#),
        (r#"럮"#, r#"럮"#, r#"럮"#, r#"럮"#, r#"럮"#),
        (r#"럯"#, r#"럯"#, r#"럯"#, r#"럯"#, r#"럯"#),
        (r#"런"#, r#"런"#, r#"런"#, r#"런"#, r#"런"#),
        (r#"럱"#, r#"럱"#, r#"럱"#, r#"럱"#, r#"럱"#),
        (r#"럲"#, r#"럲"#, r#"럲"#, r#"럲"#, r#"럲"#),
        (r#"럳"#, r#"럳"#, r#"럳"#, r#"럳"#, r#"럳"#),
        (r#"럴"#, r#"럴"#, r#"럴"#, r#"럴"#, r#"럴"#),
        (r#"럵"#, r#"럵"#, r#"럵"#, r#"럵"#, r#"럵"#),
        (r#"럶"#, r#"럶"#, r#"럶"#, r#"럶"#, r#"럶"#),
        (r#"럷"#, r#"럷"#, r#"럷"#, r#"럷"#, r#"럷"#),
        (r#"럸"#, r#"럸"#, r#"럸"#, r#"럸"#, r#"럸"#),
        (r#"럹"#, r#"럹"#, r#"럹"#, r#"럹"#, r#"럹"#),
        (r#"럺"#, r#"럺"#, r#"럺"#, r#"럺"#, r#"럺"#),
        (r#"럻"#, r#"럻"#, r#"럻"#, r#"럻"#, r#"럻"#),
        (r#"럼"#, r#"럼"#, r#"럼"#, r#"럼"#, r#"럼"#),
        (r#"럽"#, r#"럽"#, r#"럽"#, r#"럽"#, r#"럽"#),
        (r#"럾"#, r#"럾"#, r#"럾"#, r#"럾"#, r#"럾"#),
        (r#"럿"#, r#"럿"#, r#"럿"#, r#"럿"#, r#"럿"#),
        (r#"렀"#, r#"렀"#, r#"렀"#, r#"렀"#, r#"렀"#),
        (r#"렁"#, r#"렁"#, r#"렁"#, r#"렁"#, r#"렁"#),
        (r#"렂"#, r#"렂"#, r#"렂"#, r#"렂"#, r#"렂"#),
        (r#"렃"#, r#"렃"#, r#"렃"#, r#"렃"#, r#"렃"#),
        (r#"렄"#, r#"렄"#, r#"렄"#, r#"렄"#, r#"렄"#),
        (r#"렅"#, r#"렅"#, r#"렅"#, r#"렅"#, r#"렅"#),
        (r#"렆"#, r#"렆"#, r#"렆"#, r#"렆"#, r#"렆"#),
        (r#"렇"#, r#"렇"#, r#"렇"#, r#"렇"#, r#"렇"#),
        (r#"레"#, r#"레"#, r#"레"#, r#"레"#, r#"레"#),
        (r#"렉"#, r#"렉"#, r#"렉"#, r#"렉"#, r#"렉"#),
        (r#"렊"#, r#"렊"#, r#"렊"#, r#"렊"#, r#"렊"#),
        (r#"렋"#, r#"렋"#, r#"렋"#, r#"렋"#, r#"렋"#),
        (r#"렌"#, r#"렌"#, r#"렌"#, r#"렌"#, r#"렌"#),
        (r#"렍"#, r#"렍"#, r#"렍"#, r#"렍"#, r#"렍"#),
        (r#"렎"#, r#"렎"#, r#"렎"#, r#"렎"#, r#"렎"#),
        (r#"렏"#, r#"렏"#, r#"렏"#, r#"렏"#, r#"렏"#),
        (r#"렐"#, r#"렐"#, r#"렐"#, r#"렐"#, r#"렐"#),
        (r#"렑"#, r#"렑"#, r#"렑"#, r#"렑"#, r#"렑"#),
        (r#"렒"#, r#"렒"#, r#"렒"#, r#"렒"#, r#"렒"#),
        (r#"렓"#, r#"렓"#, r#"렓"#, r#"렓"#, r#"렓"#),
        (r#"렔"#, r#"렔"#, r#"렔"#, r#"렔"#, r#"렔"#),
        (r#"렕"#, r#"렕"#, r#"렕"#, r#"렕"#, r#"렕"#),
        (r#"렖"#, r#"렖"#, r#"렖"#, r#"렖"#, r#"렖"#),
        (r#"렗"#, r#"렗"#, r#"렗"#, r#"렗"#, r#"렗"#),
        (r#"렘"#, r#"렘"#, r#"렘"#, r#"렘"#, r#"렘"#),
        (r#"렙"#, r#"렙"#, r#"렙"#, r#"렙"#, r#"렙"#),
        (r#"렚"#, r#"렚"#, r#"렚"#, r#"렚"#, r#"렚"#),
        (r#"렛"#, r#"렛"#, r#"렛"#, r#"렛"#, r#"렛"#),
        (r#"렜"#, r#"렜"#, r#"렜"#, r#"렜"#, r#"렜"#),
        (r#"렝"#, r#"렝"#, r#"렝"#, r#"렝"#, r#"렝"#),
        (r#"렞"#, r#"렞"#, r#"렞"#, r#"렞"#, r#"렞"#),
        (r#"렟"#, r#"렟"#, r#"렟"#, r#"렟"#, r#"렟"#),
        (r#"렠"#, r#"렠"#, r#"렠"#, r#"렠"#, r#"렠"#),
        (r#"렡"#, r#"렡"#, r#"렡"#, r#"렡"#, r#"렡"#),
        (r#"렢"#, r#"렢"#, r#"렢"#, r#"렢"#, r#"렢"#),
        (r#"렣"#, r#"렣"#, r#"렣"#, r#"렣"#, r#"렣"#),
        (r#"려"#, r#"려"#, r#"려"#, r#"려"#, r#"려"#),
        (r#"력"#, r#"력"#, r#"력"#, r#"력"#, r#"력"#),
        (r#"렦"#, r#"렦"#, r#"렦"#, r#"렦"#, r#"렦"#),
        (r#"렧"#, r#"렧"#, r#"렧"#, r#"렧"#, r#"렧"#),
        (r#"련"#, r#"련"#, r#"련"#, r#"련"#, r#"련"#),
        (r#"렩"#, r#"렩"#, r#"렩"#, r#"렩"#, r#"렩"#),
        (r#"렪"#, r#"렪"#, r#"렪"#, r#"렪"#, r#"렪"#),
        (r#"렫"#, r#"렫"#, r#"렫"#, r#"렫"#, r#"렫"#),
        (r#"렬"#, r#"렬"#, r#"렬"#, r#"렬"#, r#"렬"#),
        (r#"렭"#, r#"렭"#, r#"렭"#, r#"렭"#, r#"렭"#),
        (r#"렮"#, r#"렮"#, r#"렮"#, r#"렮"#, r#"렮"#),
        (r#"렯"#, r#"렯"#, r#"렯"#, r#"렯"#, r#"렯"#),
        (r#"렰"#, r#"렰"#, r#"렰"#, r#"렰"#, r#"렰"#),
        (r#"렱"#, r#"렱"#, r#"렱"#, r#"렱"#, r#"렱"#),
        (r#"렲"#, r#"렲"#, r#"렲"#, r#"렲"#, r#"렲"#),
        (r#"렳"#, r#"렳"#, r#"렳"#, r#"렳"#, r#"렳"#),
        (r#"렴"#, r#"렴"#, r#"렴"#, r#"렴"#, r#"렴"#),
        (r#"렵"#, r#"렵"#, r#"렵"#, r#"렵"#, r#"렵"#),
        (r#"렶"#, r#"렶"#, r#"렶"#, r#"렶"#, r#"렶"#),
        (r#"렷"#, r#"렷"#, r#"렷"#, r#"렷"#, r#"렷"#),
        (r#"렸"#, r#"렸"#, r#"렸"#, r#"렸"#, r#"렸"#),
        (r#"령"#, r#"령"#, r#"령"#, r#"령"#, r#"령"#),
        (r#"렺"#, r#"렺"#, r#"렺"#, r#"렺"#, r#"렺"#),
        (r#"렻"#, r#"렻"#, r#"렻"#, r#"렻"#, r#"렻"#),
        (r#"렼"#, r#"렼"#, r#"렼"#, r#"렼"#, r#"렼"#),
        (r#"렽"#, r#"렽"#, r#"렽"#, r#"렽"#, r#"렽"#),
        (r#"렾"#, r#"렾"#, r#"렾"#, r#"렾"#, r#"렾"#),
        (r#"렿"#, r#"렿"#, r#"렿"#, r#"렿"#, r#"렿"#),
        (r#"례"#, r#"례"#, r#"례"#, r#"례"#, r#"례"#),
        (r#"롁"#, r#"롁"#, r#"롁"#, r#"롁"#, r#"롁"#),
        (r#"롂"#, r#"롂"#, r#"롂"#, r#"롂"#, r#"롂"#),
        (r#"롃"#, r#"롃"#, r#"롃"#, r#"롃"#, r#"롃"#),
        (r#"롄"#, r#"롄"#, r#"롄"#, r#"롄"#, r#"롄"#),
        (r#"롅"#, r#"롅"#, r#"롅"#, r#"롅"#, r#"롅"#),
        (r#"롆"#, r#"롆"#, r#"롆"#, r#"롆"#, r#"롆"#),
        (r#"롇"#, r#"롇"#, r#"롇"#, r#"롇"#, r#"롇"#),
        (r#"롈"#, r#"롈"#, r#"롈"#, r#"롈"#, r#"롈"#),
        (r#"롉"#, r#"롉"#, r#"롉"#, r#"롉"#, r#"롉"#),
        (r#"롊"#, r#"롊"#, r#"롊"#, r#"롊"#, r#"롊"#),
        (r#"롋"#, r#"롋"#, r#"롋"#, r#"롋"#, r#"롋"#),
        (r#"롌"#, r#"롌"#, r#"롌"#, r#"롌"#, r#"롌"#),
        (r#"롍"#, r#"롍"#, r#"롍"#, r#"롍"#, r#"롍"#),
        (r#"롎"#, r#"롎"#, r#"롎"#, r#"롎"#, r#"롎"#),
        (r#"롏"#, r#"롏"#, r#"롏"#, r#"롏"#, r#"롏"#),
        (r#"롐"#, r#"롐"#, r#"롐"#, r#"롐"#, r#"롐"#),
        (r#"롑"#, r#"롑"#, r#"롑"#, r#"롑"#, r#"롑"#),
        (r#"롒"#, r#"롒"#, r#"롒"#, r#"롒"#, r#"롒"#),
        (r#"롓"#, r#"롓"#, r#"롓"#, r#"롓"#, r#"롓"#),
        (r#"롔"#, r#"롔"#, r#"롔"#, r#"롔"#, r#"롔"#),
        (r#"롕"#, r#"롕"#, r#"롕"#, r#"롕"#, r#"롕"#),
        (r#"롖"#, r#"롖"#, r#"롖"#, r#"롖"#, r#"롖"#),
        (r#"롗"#, r#"롗"#, r#"롗"#, r#"롗"#, r#"롗"#),
        (r#"롘"#, r#"롘"#, r#"롘"#, r#"롘"#, r#"롘"#),
        (r#"롙"#, r#"롙"#, r#"롙"#, r#"롙"#, r#"롙"#),
        (r#"롚"#, r#"롚"#, r#"롚"#, r#"롚"#, r#"롚"#),
        (r#"롛"#, r#"롛"#, r#"롛"#, r#"롛"#, r#"롛"#),
        (r#"로"#, r#"로"#, r#"로"#, r#"로"#, r#"로"#),
        (r#"록"#, r#"록"#, r#"록"#, r#"록"#, r#"록"#),
        (r#"롞"#, r#"롞"#, r#"롞"#, r#"롞"#, r#"롞"#),
        (r#"롟"#, r#"롟"#, r#"롟"#, r#"롟"#, r#"롟"#),
        (r#"론"#, r#"론"#, r#"론"#, r#"론"#, r#"론"#),
        (r#"롡"#, r#"롡"#, r#"롡"#, r#"롡"#, r#"롡"#),
        (r#"롢"#, r#"롢"#, r#"롢"#, r#"롢"#, r#"롢"#),
        (r#"롣"#, r#"롣"#, r#"롣"#, r#"롣"#, r#"롣"#),
        (r#"롤"#, r#"롤"#, r#"롤"#, r#"롤"#, r#"롤"#),
        (r#"롥"#, r#"롥"#, r#"롥"#, r#"롥"#, r#"롥"#),
        (r#"롦"#, r#"롦"#, r#"롦"#, r#"롦"#, r#"롦"#),
        (r#"롧"#, r#"롧"#, r#"롧"#, r#"롧"#, r#"롧"#),
        (r#"롨"#, r#"롨"#, r#"롨"#, r#"롨"#, r#"롨"#),
        (r#"롩"#, r#"롩"#, r#"롩"#, r#"롩"#, r#"롩"#),
        (r#"롪"#, r#"롪"#, r#"롪"#, r#"롪"#, r#"롪"#),
        (r#"롫"#, r#"롫"#, r#"롫"#, r#"롫"#, r#"롫"#),
        (r#"롬"#, r#"롬"#, r#"롬"#, r#"롬"#, r#"롬"#),
        (r#"롭"#, r#"롭"#, r#"롭"#, r#"롭"#, r#"롭"#),
        (r#"롮"#, r#"롮"#, r#"롮"#, r#"롮"#, r#"롮"#),
        (r#"롯"#, r#"롯"#, r#"롯"#, r#"롯"#, r#"롯"#),
        (r#"롰"#, r#"롰"#, r#"롰"#, r#"롰"#, r#"롰"#),
        (r#"롱"#, r#"롱"#, r#"롱"#, r#"롱"#, r#"롱"#),
        (r#"롲"#, r#"롲"#, r#"롲"#, r#"롲"#, r#"롲"#),
        (r#"롳"#, r#"롳"#, r#"롳"#, r#"롳"#, r#"롳"#),
        (r#"롴"#, r#"롴"#, r#"롴"#, r#"롴"#, r#"롴"#),
        (r#"롵"#, r#"롵"#, r#"롵"#, r#"롵"#, r#"롵"#),
        (r#"롶"#, r#"롶"#, r#"롶"#, r#"롶"#, r#"롶"#),
        (r#"롷"#, r#"롷"#, r#"롷"#, r#"롷"#, r#"롷"#),
        (r#"롸"#, r#"롸"#, r#"롸"#, r#"롸"#, r#"롸"#),
        (r#"롹"#, r#"롹"#, r#"롹"#, r#"롹"#, r#"롹"#),
        (r#"롺"#, r#"롺"#, r#"롺"#, r#"롺"#, r#"롺"#),
        (r#"롻"#, r#"롻"#, r#"롻"#, r#"롻"#, r#"롻"#),
        (r#"롼"#, r#"롼"#, r#"롼"#, r#"롼"#, r#"롼"#),
        (r#"롽"#, r#"롽"#, r#"롽"#, r#"롽"#, r#"롽"#),
        (r#"롾"#, r#"롾"#, r#"롾"#, r#"롾"#, r#"롾"#),
        (r#"롿"#, r#"롿"#, r#"롿"#, r#"롿"#, r#"롿"#),
        (r#"뢀"#, r#"뢀"#, r#"뢀"#, r#"뢀"#, r#"뢀"#),
        (r#"뢁"#, r#"뢁"#, r#"뢁"#, r#"뢁"#, r#"뢁"#),
        (r#"뢂"#, r#"뢂"#, r#"뢂"#, r#"뢂"#, r#"뢂"#),
        (r#"뢃"#, r#"뢃"#, r#"뢃"#, r#"뢃"#, r#"뢃"#),
        (r#"뢄"#, r#"뢄"#, r#"뢄"#, r#"뢄"#, r#"뢄"#),
        (r#"뢅"#, r#"뢅"#, r#"뢅"#, r#"뢅"#, r#"뢅"#),
        (r#"뢆"#, r#"뢆"#, r#"뢆"#, r#"뢆"#, r#"뢆"#),
        (r#"뢇"#, r#"뢇"#, r#"뢇"#, r#"뢇"#, r#"뢇"#),
        (r#"뢈"#, r#"뢈"#, r#"뢈"#, r#"뢈"#, r#"뢈"#),
        (r#"뢉"#, r#"뢉"#, r#"뢉"#, r#"뢉"#, r#"뢉"#),
        (r#"뢊"#, r#"뢊"#, r#"뢊"#, r#"뢊"#, r#"뢊"#),
        (r#"뢋"#, r#"뢋"#, r#"뢋"#, r#"뢋"#, r#"뢋"#),
        (r#"뢌"#, r#"뢌"#, r#"뢌"#, r#"뢌"#, r#"뢌"#),
        (r#"뢍"#, r#"뢍"#, r#"뢍"#, r#"뢍"#, r#"뢍"#),
        (r#"뢎"#, r#"뢎"#, r#"뢎"#, r#"뢎"#, r#"뢎"#),
        (r#"뢏"#, r#"뢏"#, r#"뢏"#, r#"뢏"#, r#"뢏"#),
        (r#"뢐"#, r#"뢐"#, r#"뢐"#, r#"뢐"#, r#"뢐"#),
        (r#"뢑"#, r#"뢑"#, r#"뢑"#, r#"뢑"#, r#"뢑"#),
        (r#"뢒"#, r#"뢒"#, r#"뢒"#, r#"뢒"#, r#"뢒"#),
        (r#"뢓"#, r#"뢓"#, r#"뢓"#, r#"뢓"#, r#"뢓"#),
        (r#"뢔"#, r#"뢔"#, r#"뢔"#, r#"뢔"#, r#"뢔"#),
        (r#"뢕"#, r#"뢕"#, r#"뢕"#, r#"뢕"#, r#"뢕"#),
        (r#"뢖"#, r#"뢖"#, r#"뢖"#, r#"뢖"#, r#"뢖"#),
        (r#"뢗"#, r#"뢗"#, r#"뢗"#, r#"뢗"#, r#"뢗"#),
        (r#"뢘"#, r#"뢘"#, r#"뢘"#, r#"뢘"#, r#"뢘"#),
        (r#"뢙"#, r#"뢙"#, r#"뢙"#, r#"뢙"#, r#"뢙"#),
        (r#"뢚"#, r#"뢚"#, r#"뢚"#, r#"뢚"#, r#"뢚"#),
        (r#"뢛"#, r#"뢛"#, r#"뢛"#, r#"뢛"#, r#"뢛"#),
        (r#"뢜"#, r#"뢜"#, r#"뢜"#, r#"뢜"#, r#"뢜"#),
        (r#"뢝"#, r#"뢝"#, r#"뢝"#, r#"뢝"#, r#"뢝"#),
        (r#"뢞"#, r#"뢞"#, r#"뢞"#, r#"뢞"#, r#"뢞"#),
        (r#"뢟"#, r#"뢟"#, r#"뢟"#, r#"뢟"#, r#"뢟"#),
        (r#"뢠"#, r#"뢠"#, r#"뢠"#, r#"뢠"#, r#"뢠"#),
        (r#"뢡"#, r#"뢡"#, r#"뢡"#, r#"뢡"#, r#"뢡"#),
        (r#"뢢"#, r#"뢢"#, r#"뢢"#, r#"뢢"#, r#"뢢"#),
        (r#"뢣"#, r#"뢣"#, r#"뢣"#, r#"뢣"#, r#"뢣"#),
        (r#"뢤"#, r#"뢤"#, r#"뢤"#, r#"뢤"#, r#"뢤"#),
        (r#"뢥"#, r#"뢥"#, r#"뢥"#, r#"뢥"#, r#"뢥"#),
        (r#"뢦"#, r#"뢦"#, r#"뢦"#, r#"뢦"#, r#"뢦"#),
        (r#"뢧"#, r#"뢧"#, r#"뢧"#, r#"뢧"#, r#"뢧"#),
        (r#"뢨"#, r#"뢨"#, r#"뢨"#, r#"뢨"#, r#"뢨"#),
        (r#"뢩"#, r#"뢩"#, r#"뢩"#, r#"뢩"#, r#"뢩"#),
        (r#"뢪"#, r#"뢪"#, r#"뢪"#, r#"뢪"#, r#"뢪"#),
        (r#"뢫"#, r#"뢫"#, r#"뢫"#, r#"뢫"#, r#"뢫"#),
        (r#"뢬"#, r#"뢬"#, r#"뢬"#, r#"뢬"#, r#"뢬"#),
        (r#"뢭"#, r#"뢭"#, r#"뢭"#, r#"뢭"#, r#"뢭"#),
        (r#"뢮"#, r#"뢮"#, r#"뢮"#, r#"뢮"#, r#"뢮"#),
        (r#"뢯"#, r#"뢯"#, r#"뢯"#, r#"뢯"#, r#"뢯"#),
        (r#"뢰"#, r#"뢰"#, r#"뢰"#, r#"뢰"#, r#"뢰"#),
        (r#"뢱"#, r#"뢱"#, r#"뢱"#, r#"뢱"#, r#"뢱"#),
        (r#"뢲"#, r#"뢲"#, r#"뢲"#, r#"뢲"#, r#"뢲"#),
        (r#"뢳"#, r#"뢳"#, r#"뢳"#, r#"뢳"#, r#"뢳"#),
        (r#"뢴"#, r#"뢴"#, r#"뢴"#, r#"뢴"#, r#"뢴"#),
        (r#"뢵"#, r#"뢵"#, r#"뢵"#, r#"뢵"#, r#"뢵"#),
        (r#"뢶"#, r#"뢶"#, r#"뢶"#, r#"뢶"#, r#"뢶"#),
        (r#"뢷"#, r#"뢷"#, r#"뢷"#, r#"뢷"#, r#"뢷"#),
        (r#"뢸"#, r#"뢸"#, r#"뢸"#, r#"뢸"#, r#"뢸"#),
        (r#"뢹"#, r#"뢹"#, r#"뢹"#, r#"뢹"#, r#"뢹"#),
        (r#"뢺"#, r#"뢺"#, r#"뢺"#, r#"뢺"#, r#"뢺"#),
        (r#"뢻"#, r#"뢻"#, r#"뢻"#, r#"뢻"#, r#"뢻"#),
        (r#"뢼"#, r#"뢼"#, r#"뢼"#, r#"뢼"#, r#"뢼"#),
        (r#"뢽"#, r#"뢽"#, r#"뢽"#, r#"뢽"#, r#"뢽"#),
        (r#"뢾"#, r#"뢾"#, r#"뢾"#, r#"뢾"#, r#"뢾"#),
        (r#"뢿"#, r#"뢿"#, r#"뢿"#, r#"뢿"#, r#"뢿"#),
        (r#"룀"#, r#"룀"#, r#"룀"#, r#"룀"#, r#"룀"#),
        (r#"룁"#, r#"룁"#, r#"룁"#, r#"룁"#, r#"룁"#),
        (r#"룂"#, r#"룂"#, r#"룂"#, r#"룂"#, r#"룂"#),
        (r#"룃"#, r#"룃"#, r#"룃"#, r#"룃"#, r#"룃"#),
        (r#"룄"#, r#"룄"#, r#"룄"#, r#"룄"#, r#"룄"#),
        (r#"룅"#, r#"룅"#, r#"룅"#, r#"룅"#, r#"룅"#),
        (r#"룆"#, r#"룆"#, r#"룆"#, r#"룆"#, r#"룆"#),
        (r#"룇"#, r#"룇"#, r#"룇"#, r#"룇"#, r#"룇"#),
        (r#"룈"#, r#"룈"#, r#"룈"#, r#"룈"#, r#"룈"#),
        (r#"룉"#, r#"룉"#, r#"룉"#, r#"룉"#, r#"룉"#),
        (r#"룊"#, r#"룊"#, r#"룊"#, r#"룊"#, r#"룊"#),
        (r#"룋"#, r#"룋"#, r#"룋"#, r#"룋"#, r#"룋"#),
        (r#"료"#, r#"료"#, r#"료"#, r#"료"#, r#"료"#),
        (r#"룍"#, r#"룍"#, r#"룍"#, r#"룍"#, r#"룍"#),
        (r#"룎"#, r#"룎"#, r#"룎"#, r#"룎"#, r#"룎"#),
        (r#"룏"#, r#"룏"#, r#"룏"#, r#"룏"#, r#"룏"#),
        (r#"룐"#, r#"룐"#, r#"룐"#, r#"룐"#, r#"룐"#),
        (r#"룑"#, r#"룑"#, r#"룑"#, r#"룑"#, r#"룑"#),
        (r#"룒"#, r#"룒"#, r#"룒"#, r#"룒"#, r#"룒"#),
        (r#"룓"#, r#"룓"#, r#"룓"#, r#"룓"#, r#"룓"#),
        (r#"룔"#, r#"룔"#, r#"룔"#, r#"룔"#, r#"룔"#),
        (r#"룕"#, r#"룕"#, r#"룕"#, r#"룕"#, r#"룕"#),
        (r#"룖"#, r#"룖"#, r#"룖"#, r#"룖"#, r#"룖"#),
        (r#"룗"#, r#"룗"#, r#"룗"#, r#"룗"#, r#"룗"#),
        (r#"룘"#, r#"룘"#, r#"룘"#, r#"룘"#, r#"룘"#),
        (r#"룙"#, r#"룙"#, r#"룙"#, r#"룙"#, r#"룙"#),
        (r#"룚"#, r#"룚"#, r#"룚"#, r#"룚"#, r#"룚"#),
        (r#"룛"#, r#"룛"#, r#"룛"#, r#"룛"#, r#"룛"#),
        (r#"룜"#, r#"룜"#, r#"룜"#, r#"룜"#, r#"룜"#),
        (r#"룝"#, r#"룝"#, r#"룝"#, r#"룝"#, r#"룝"#),
        (r#"룞"#, r#"룞"#, r#"룞"#, r#"룞"#, r#"룞"#),
        (r#"룟"#, r#"룟"#, r#"룟"#, r#"룟"#, r#"룟"#),
        (r#"룠"#, r#"룠"#, r#"룠"#, r#"룠"#, r#"룠"#),
        (r#"룡"#, r#"룡"#, r#"룡"#, r#"룡"#, r#"룡"#),
        (r#"룢"#, r#"룢"#, r#"룢"#, r#"룢"#, r#"룢"#),
        (r#"룣"#, r#"룣"#, r#"룣"#, r#"룣"#, r#"룣"#),
        (r#"룤"#, r#"룤"#, r#"룤"#, r#"룤"#, r#"룤"#),
        (r#"룥"#, r#"룥"#, r#"룥"#, r#"룥"#, r#"룥"#),
        (r#"룦"#, r#"룦"#, r#"룦"#, r#"룦"#, r#"룦"#),
        (r#"룧"#, r#"룧"#, r#"룧"#, r#"룧"#, r#"룧"#),
        (r#"루"#, r#"루"#, r#"루"#, r#"루"#, r#"루"#),
        (r#"룩"#, r#"룩"#, r#"룩"#, r#"룩"#, r#"룩"#),
        (r#"룪"#, r#"룪"#, r#"룪"#, r#"룪"#, r#"룪"#),
        (r#"룫"#, r#"룫"#, r#"룫"#, r#"룫"#, r#"룫"#),
        (r#"룬"#, r#"룬"#, r#"룬"#, r#"룬"#, r#"룬"#),
        (r#"룭"#, r#"룭"#, r#"룭"#, r#"룭"#, r#"룭"#),
        (r#"룮"#, r#"룮"#, r#"룮"#, r#"룮"#, r#"룮"#),
        (r#"룯"#, r#"룯"#, r#"룯"#, r#"룯"#, r#"룯"#),
        (r#"룰"#, r#"룰"#, r#"룰"#, r#"룰"#, r#"룰"#),
        (r#"룱"#, r#"룱"#, r#"룱"#, r#"룱"#, r#"룱"#),
        (r#"룲"#, r#"룲"#, r#"룲"#, r#"룲"#, r#"룲"#),
        (r#"룳"#, r#"룳"#, r#"룳"#, r#"룳"#, r#"룳"#),
        (r#"룴"#, r#"룴"#, r#"룴"#, r#"룴"#, r#"룴"#),
        (r#"룵"#, r#"룵"#, r#"룵"#, r#"룵"#, r#"룵"#),
        (r#"룶"#, r#"룶"#, r#"룶"#, r#"룶"#, r#"룶"#),
        (r#"룷"#, r#"룷"#, r#"룷"#, r#"룷"#, r#"룷"#),
        (r#"룸"#, r#"룸"#, r#"룸"#, r#"룸"#, r#"룸"#),
        (r#"룹"#, r#"룹"#, r#"룹"#, r#"룹"#, r#"룹"#),
        (r#"룺"#, r#"룺"#, r#"룺"#, r#"룺"#, r#"룺"#),
        (r#"룻"#, r#"룻"#, r#"룻"#, r#"룻"#, r#"룻"#),
        (r#"룼"#, r#"룼"#, r#"룼"#, r#"룼"#, r#"룼"#),
        (r#"룽"#, r#"룽"#, r#"룽"#, r#"룽"#, r#"룽"#),
        (r#"룾"#, r#"룾"#, r#"룾"#, r#"룾"#, r#"룾"#),
        (r#"룿"#, r#"룿"#, r#"룿"#, r#"룿"#, r#"룿"#),
        (r#"뤀"#, r#"뤀"#, r#"뤀"#, r#"뤀"#, r#"뤀"#),
        (r#"뤁"#, r#"뤁"#, r#"뤁"#, r#"뤁"#, r#"뤁"#),
        (r#"뤂"#, r#"뤂"#, r#"뤂"#, r#"뤂"#, r#"뤂"#),
        (r#"뤃"#, r#"뤃"#, r#"뤃"#, r#"뤃"#, r#"뤃"#),
        (r#"뤄"#, r#"뤄"#, r#"뤄"#, r#"뤄"#, r#"뤄"#),
        (r#"뤅"#, r#"뤅"#, r#"뤅"#, r#"뤅"#, r#"뤅"#),
        (r#"뤆"#, r#"뤆"#, r#"뤆"#, r#"뤆"#, r#"뤆"#),
        (r#"뤇"#, r#"뤇"#, r#"뤇"#, r#"뤇"#, r#"뤇"#),
        (r#"뤈"#, r#"뤈"#, r#"뤈"#, r#"뤈"#, r#"뤈"#),
        (r#"뤉"#, r#"뤉"#, r#"뤉"#, r#"뤉"#, r#"뤉"#),
        (r#"뤊"#, r#"뤊"#, r#"뤊"#, r#"뤊"#, r#"뤊"#),
        (r#"뤋"#, r#"뤋"#, r#"뤋"#, r#"뤋"#, r#"뤋"#),
        (r#"뤌"#, r#"뤌"#, r#"뤌"#, r#"뤌"#, r#"뤌"#),
        (r#"뤍"#, r#"뤍"#, r#"뤍"#, r#"뤍"#, r#"뤍"#),
        (r#"뤎"#, r#"뤎"#, r#"뤎"#, r#"뤎"#, r#"뤎"#),
        (r#"뤏"#, r#"뤏"#, r#"뤏"#, r#"뤏"#, r#"뤏"#),
        (r#"뤐"#, r#"뤐"#, r#"뤐"#, r#"뤐"#, r#"뤐"#),
        (r#"뤑"#, r#"뤑"#, r#"뤑"#, r#"뤑"#, r#"뤑"#),
        (r#"뤒"#, r#"뤒"#, r#"뤒"#, r#"뤒"#, r#"뤒"#),
        (r#"뤓"#, r#"뤓"#, r#"뤓"#, r#"뤓"#, r#"뤓"#),
        (r#"뤔"#, r#"뤔"#, r#"뤔"#, r#"뤔"#, r#"뤔"#),
        (r#"뤕"#, r#"뤕"#, r#"뤕"#, r#"뤕"#, r#"뤕"#),
        (r#"뤖"#, r#"뤖"#, r#"뤖"#, r#"뤖"#, r#"뤖"#),
        (r#"뤗"#, r#"뤗"#, r#"뤗"#, r#"뤗"#, r#"뤗"#),
        (r#"뤘"#, r#"뤘"#, r#"뤘"#, r#"뤘"#, r#"뤘"#),
        (r#"뤙"#, r#"뤙"#, r#"뤙"#, r#"뤙"#, r#"뤙"#),
        (r#"뤚"#, r#"뤚"#, r#"뤚"#, r#"뤚"#, r#"뤚"#),
        (r#"뤛"#, r#"뤛"#, r#"뤛"#, r#"뤛"#, r#"뤛"#),
        (r#"뤜"#, r#"뤜"#, r#"뤜"#, r#"뤜"#, r#"뤜"#),
        (r#"뤝"#, r#"뤝"#, r#"뤝"#, r#"뤝"#, r#"뤝"#),
        (r#"뤞"#, r#"뤞"#, r#"뤞"#, r#"뤞"#, r#"뤞"#),
        (r#"뤟"#, r#"뤟"#, r#"뤟"#, r#"뤟"#, r#"뤟"#),
        (r#"뤠"#, r#"뤠"#, r#"뤠"#, r#"뤠"#, r#"뤠"#),
        (r#"뤡"#, r#"뤡"#, r#"뤡"#, r#"뤡"#, r#"뤡"#),
        (r#"뤢"#, r#"뤢"#, r#"뤢"#, r#"뤢"#, r#"뤢"#),
        (r#"뤣"#, r#"뤣"#, r#"뤣"#, r#"뤣"#, r#"뤣"#),
        (r#"뤤"#, r#"뤤"#, r#"뤤"#, r#"뤤"#, r#"뤤"#),
        (r#"뤥"#, r#"뤥"#, r#"뤥"#, r#"뤥"#, r#"뤥"#),
        (r#"뤦"#, r#"뤦"#, r#"뤦"#, r#"뤦"#, r#"뤦"#),
        (r#"뤧"#, r#"뤧"#, r#"뤧"#, r#"뤧"#, r#"뤧"#),
        (r#"뤨"#, r#"뤨"#, r#"뤨"#, r#"뤨"#, r#"뤨"#),
        (r#"뤩"#, r#"뤩"#, r#"뤩"#, r#"뤩"#, r#"뤩"#),
        (r#"뤪"#, r#"뤪"#, r#"뤪"#, r#"뤪"#, r#"뤪"#),
        (r#"뤫"#, r#"뤫"#, r#"뤫"#, r#"뤫"#, r#"뤫"#),
        (r#"뤬"#, r#"뤬"#, r#"뤬"#, r#"뤬"#, r#"뤬"#),
        (r#"뤭"#, r#"뤭"#, r#"뤭"#, r#"뤭"#, r#"뤭"#),
        (r#"뤮"#, r#"뤮"#, r#"뤮"#, r#"뤮"#, r#"뤮"#),
        (r#"뤯"#, r#"뤯"#, r#"뤯"#, r#"뤯"#, r#"뤯"#),
        (r#"뤰"#, r#"뤰"#, r#"뤰"#, r#"뤰"#, r#"뤰"#),
        (r#"뤱"#, r#"뤱"#, r#"뤱"#, r#"뤱"#, r#"뤱"#),
        (r#"뤲"#, r#"뤲"#, r#"뤲"#, r#"뤲"#, r#"뤲"#),
        (r#"뤳"#, r#"뤳"#, r#"뤳"#, r#"뤳"#, r#"뤳"#),
        (r#"뤴"#, r#"뤴"#, r#"뤴"#, r#"뤴"#, r#"뤴"#),
        (r#"뤵"#, r#"뤵"#, r#"뤵"#, r#"뤵"#, r#"뤵"#),
        (r#"뤶"#, r#"뤶"#, r#"뤶"#, r#"뤶"#, r#"뤶"#),
        (r#"뤷"#, r#"뤷"#, r#"뤷"#, r#"뤷"#, r#"뤷"#),
        (r#"뤸"#, r#"뤸"#, r#"뤸"#, r#"뤸"#, r#"뤸"#),
        (r#"뤹"#, r#"뤹"#, r#"뤹"#, r#"뤹"#, r#"뤹"#),
        (r#"뤺"#, r#"뤺"#, r#"뤺"#, r#"뤺"#, r#"뤺"#),
        (r#"뤻"#, r#"뤻"#, r#"뤻"#, r#"뤻"#, r#"뤻"#),
        (r#"뤼"#, r#"뤼"#, r#"뤼"#, r#"뤼"#, r#"뤼"#),
        (r#"뤽"#, r#"뤽"#, r#"뤽"#, r#"뤽"#, r#"뤽"#),
        (r#"뤾"#, r#"뤾"#, r#"뤾"#, r#"뤾"#, r#"뤾"#),
        (r#"뤿"#, r#"뤿"#, r#"뤿"#, r#"뤿"#, r#"뤿"#),
        (r#"륀"#, r#"륀"#, r#"륀"#, r#"륀"#, r#"륀"#),
        (r#"륁"#, r#"륁"#, r#"륁"#, r#"륁"#, r#"륁"#),
        (r#"륂"#, r#"륂"#, r#"륂"#, r#"륂"#, r#"륂"#),
        (r#"륃"#, r#"륃"#, r#"륃"#, r#"륃"#, r#"륃"#),
        (r#"륄"#, r#"륄"#, r#"륄"#, r#"륄"#, r#"륄"#),
        (r#"륅"#, r#"륅"#, r#"륅"#, r#"륅"#, r#"륅"#),
        (r#"륆"#, r#"륆"#, r#"륆"#, r#"륆"#, r#"륆"#),
        (r#"륇"#, r#"륇"#, r#"륇"#, r#"륇"#, r#"륇"#),
        (r#"륈"#, r#"륈"#, r#"륈"#, r#"륈"#, r#"륈"#),
        (r#"륉"#, r#"륉"#, r#"륉"#, r#"륉"#, r#"륉"#),
        (r#"륊"#, r#"륊"#, r#"륊"#, r#"륊"#, r#"륊"#),
        (r#"륋"#, r#"륋"#, r#"륋"#, r#"륋"#, r#"륋"#),
        (r#"륌"#, r#"륌"#, r#"륌"#, r#"륌"#, r#"륌"#),
        (r#"륍"#, r#"륍"#, r#"륍"#, r#"륍"#, r#"륍"#),
        (r#"륎"#, r#"륎"#, r#"륎"#, r#"륎"#, r#"륎"#),
        (r#"륏"#, r#"륏"#, r#"륏"#, r#"륏"#, r#"륏"#),
        (r#"륐"#, r#"륐"#, r#"륐"#, r#"륐"#, r#"륐"#),
        (r#"륑"#, r#"륑"#, r#"륑"#, r#"륑"#, r#"륑"#),
        (r#"륒"#, r#"륒"#, r#"륒"#, r#"륒"#, r#"륒"#),
        (r#"륓"#, r#"륓"#, r#"륓"#, r#"륓"#, r#"륓"#),
        (r#"륔"#, r#"륔"#, r#"륔"#, r#"륔"#, r#"륔"#),
        (r#"륕"#, r#"륕"#, r#"륕"#, r#"륕"#, r#"륕"#),
        (r#"륖"#, r#"륖"#, r#"륖"#, r#"륖"#, r#"륖"#),
        (r#"륗"#, r#"륗"#, r#"륗"#, r#"륗"#, r#"륗"#),
        (r#"류"#, r#"류"#, r#"류"#, r#"류"#, r#"류"#),
        (r#"륙"#, r#"륙"#, r#"륙"#, r#"륙"#, r#"륙"#),
        (r#"륚"#, r#"륚"#, r#"륚"#, r#"륚"#, r#"륚"#),
        (r#"륛"#, r#"륛"#, r#"륛"#, r#"륛"#, r#"륛"#),
        (r#"륜"#, r#"륜"#, r#"륜"#, r#"륜"#, r#"륜"#),
        (r#"륝"#, r#"륝"#, r#"륝"#, r#"륝"#, r#"륝"#),
        (r#"륞"#, r#"륞"#, r#"륞"#, r#"륞"#, r#"륞"#),
        (r#"륟"#, r#"륟"#, r#"륟"#, r#"륟"#, r#"륟"#),
        (r#"률"#, r#"률"#, r#"률"#, r#"률"#, r#"률"#),
        (r#"륡"#, r#"륡"#, r#"륡"#, r#"륡"#, r#"륡"#),
        (r#"륢"#, r#"륢"#, r#"륢"#, r#"륢"#, r#"륢"#),
        (r#"륣"#, r#"륣"#, r#"륣"#, r#"륣"#, r#"륣"#),
        (r#"륤"#, r#"륤"#, r#"륤"#, r#"륤"#, r#"륤"#),
        (r#"륥"#, r#"륥"#, r#"륥"#, r#"륥"#, r#"륥"#),
        (r#"륦"#, r#"륦"#, r#"륦"#, r#"륦"#, r#"륦"#),
        (r#"륧"#, r#"륧"#, r#"륧"#, r#"륧"#, r#"륧"#),
        (r#"륨"#, r#"륨"#, r#"륨"#, r#"륨"#, r#"륨"#),
        (r#"륩"#, r#"륩"#, r#"륩"#, r#"륩"#, r#"륩"#),
        (r#"륪"#, r#"륪"#, r#"륪"#, r#"륪"#, r#"륪"#),
        (r#"륫"#, r#"륫"#, r#"륫"#, r#"륫"#, r#"륫"#),
        (r#"륬"#, r#"륬"#, r#"륬"#, r#"륬"#, r#"륬"#),
        (r#"륭"#, r#"륭"#, r#"륭"#, r#"륭"#, r#"륭"#),
        (r#"륮"#, r#"륮"#, r#"륮"#, r#"륮"#, r#"륮"#),
        (r#"륯"#, r#"륯"#, r#"륯"#, r#"륯"#, r#"륯"#),
        (r#"륰"#, r#"륰"#, r#"륰"#, r#"륰"#, r#"륰"#),
        (r#"륱"#, r#"륱"#, r#"륱"#, r#"륱"#, r#"륱"#),
        (r#"륲"#, r#"륲"#, r#"륲"#, r#"륲"#, r#"륲"#),
        (r#"륳"#, r#"륳"#, r#"륳"#, r#"륳"#, r#"륳"#),
        (r#"르"#, r#"르"#, r#"르"#, r#"르"#, r#"르"#),
        (r#"륵"#, r#"륵"#, r#"륵"#, r#"륵"#, r#"륵"#),
        (r#"륶"#, r#"륶"#, r#"륶"#, r#"륶"#, r#"륶"#),
        (r#"륷"#, r#"륷"#, r#"륷"#, r#"륷"#, r#"륷"#),
        (r#"른"#, r#"른"#, r#"른"#, r#"른"#, r#"른"#),
        (r#"륹"#, r#"륹"#, r#"륹"#, r#"륹"#, r#"륹"#),
        (r#"륺"#, r#"륺"#, r#"륺"#, r#"륺"#, r#"륺"#),
        (r#"륻"#, r#"륻"#, r#"륻"#, r#"륻"#, r#"륻"#),
        (r#"를"#, r#"를"#, r#"를"#, r#"를"#, r#"를"#),
        (r#"륽"#, r#"륽"#, r#"륽"#, r#"륽"#, r#"륽"#),
        (r#"륾"#, r#"륾"#, r#"륾"#, r#"륾"#, r#"륾"#),
        (r#"륿"#, r#"륿"#, r#"륿"#, r#"륿"#, r#"륿"#),
        (r#"릀"#, r#"릀"#, r#"릀"#, r#"릀"#, r#"릀"#),
        (r#"릁"#, r#"릁"#, r#"릁"#, r#"릁"#, r#"릁"#),
        (r#"릂"#, r#"릂"#, r#"릂"#, r#"릂"#, r#"릂"#),
        (r#"릃"#, r#"릃"#, r#"릃"#, r#"릃"#, r#"릃"#),
        (r#"름"#, r#"름"#, r#"름"#, r#"름"#, r#"름"#),
        (r#"릅"#, r#"릅"#, r#"릅"#, r#"릅"#, r#"릅"#),
        (r#"릆"#, r#"릆"#, r#"릆"#, r#"릆"#, r#"릆"#),
        (r#"릇"#, r#"릇"#, r#"릇"#, r#"릇"#, r#"릇"#),
        (r#"릈"#, r#"릈"#, r#"릈"#, r#"릈"#, r#"릈"#),
        (r#"릉"#, r#"릉"#, r#"릉"#, r#"릉"#, r#"릉"#),
        (r#"릊"#, r#"릊"#, r#"릊"#, r#"릊"#, r#"릊"#),
        (r#"릋"#, r#"릋"#, r#"릋"#, r#"릋"#, r#"릋"#),
        (r#"릌"#, r#"릌"#, r#"릌"#, r#"릌"#, r#"릌"#),
        (r#"릍"#, r#"릍"#, r#"릍"#, r#"릍"#, r#"릍"#),
        (r#"릎"#, r#"릎"#, r#"릎"#, r#"릎"#, r#"릎"#),
        (r#"릏"#, r#"릏"#, r#"릏"#, r#"릏"#, r#"릏"#),
        (r#"릐"#, r#"릐"#, r#"릐"#, r#"릐"#, r#"릐"#),
        (r#"릑"#, r#"릑"#, r#"릑"#, r#"릑"#, r#"릑"#),
        (r#"릒"#, r#"릒"#, r#"릒"#, r#"릒"#, r#"릒"#),
        (r#"릓"#, r#"릓"#, r#"릓"#, r#"릓"#, r#"릓"#),
        (r#"릔"#, r#"릔"#, r#"릔"#, r#"릔"#, r#"릔"#),
        (r#"릕"#, r#"릕"#, r#"릕"#, r#"릕"#, r#"릕"#),
        (r#"릖"#, r#"릖"#, r#"릖"#, r#"릖"#, r#"릖"#),
        (r#"릗"#, r#"릗"#, r#"릗"#, r#"릗"#, r#"릗"#),
        (r#"릘"#, r#"릘"#, r#"릘"#, r#"릘"#, r#"릘"#),
        (r#"릙"#, r#"릙"#, r#"릙"#, r#"릙"#, r#"릙"#),
        (r#"릚"#, r#"릚"#, r#"릚"#, r#"릚"#, r#"릚"#),
        (r#"릛"#, r#"릛"#, r#"릛"#, r#"릛"#, r#"릛"#),
        (r#"릜"#, r#"릜"#, r#"릜"#, r#"릜"#, r#"릜"#),
        (r#"릝"#, r#"릝"#, r#"릝"#, r#"릝"#, r#"릝"#),
        (r#"릞"#, r#"릞"#, r#"릞"#, r#"릞"#, r#"릞"#),
        (r#"릟"#, r#"릟"#, r#"릟"#, r#"릟"#, r#"릟"#),
        (r#"릠"#, r#"릠"#, r#"릠"#, r#"릠"#, r#"릠"#),
        (r#"릡"#, r#"릡"#, r#"릡"#, r#"릡"#, r#"릡"#),
        (r#"릢"#, r#"릢"#, r#"릢"#, r#"릢"#, r#"릢"#),
        (r#"릣"#, r#"릣"#, r#"릣"#, r#"릣"#, r#"릣"#),
        (r#"릤"#, r#"릤"#, r#"릤"#, r#"릤"#, r#"릤"#),
        (r#"릥"#, r#"릥"#, r#"릥"#, r#"릥"#, r#"릥"#),
        (r#"릦"#, r#"릦"#, r#"릦"#, r#"릦"#, r#"릦"#),
        (r#"릧"#, r#"릧"#, r#"릧"#, r#"릧"#, r#"릧"#),
        (r#"릨"#, r#"릨"#, r#"릨"#, r#"릨"#, r#"릨"#),
        (r#"릩"#, r#"릩"#, r#"릩"#, r#"릩"#, r#"릩"#),
        (r#"릪"#, r#"릪"#, r#"릪"#, r#"릪"#, r#"릪"#),
        (r#"릫"#, r#"릫"#, r#"릫"#, r#"릫"#, r#"릫"#),
        (r#"리"#, r#"리"#, r#"리"#, r#"리"#, r#"리"#),
        (r#"릭"#, r#"릭"#, r#"릭"#, r#"릭"#, r#"릭"#),
        (r#"릮"#, r#"릮"#, r#"릮"#, r#"릮"#, r#"릮"#),
        (r#"릯"#, r#"릯"#, r#"릯"#, r#"릯"#, r#"릯"#),
        (r#"린"#, r#"린"#, r#"린"#, r#"린"#, r#"린"#),
        (r#"릱"#, r#"릱"#, r#"릱"#, r#"릱"#, r#"릱"#),
        (r#"릲"#, r#"릲"#, r#"릲"#, r#"릲"#, r#"릲"#),
        (r#"릳"#, r#"릳"#, r#"릳"#, r#"릳"#, r#"릳"#),
        (r#"릴"#, r#"릴"#, r#"릴"#, r#"릴"#, r#"릴"#),
        (r#"릵"#, r#"릵"#, r#"릵"#, r#"릵"#, r#"릵"#),
        (r#"릶"#, r#"릶"#, r#"릶"#, r#"릶"#, r#"릶"#),
        (r#"릷"#, r#"릷"#, r#"릷"#, r#"릷"#, r#"릷"#),
        (r#"릸"#, r#"릸"#, r#"릸"#, r#"릸"#, r#"릸"#),
        (r#"릹"#, r#"릹"#, r#"릹"#, r#"릹"#, r#"릹"#),
        (r#"릺"#, r#"릺"#, r#"릺"#, r#"릺"#, r#"릺"#),
        (r#"릻"#, r#"릻"#, r#"릻"#, r#"릻"#, r#"릻"#),
        (r#"림"#, r#"림"#, r#"림"#, r#"림"#, r#"림"#),
        (r#"립"#, r#"립"#, r#"립"#, r#"립"#, r#"립"#),
        (r#"릾"#, r#"릾"#, r#"릾"#, r#"릾"#, r#"릾"#),
        (r#"릿"#, r#"릿"#, r#"릿"#, r#"릿"#, r#"릿"#),
        (r#"맀"#, r#"맀"#, r#"맀"#, r#"맀"#, r#"맀"#),
        (r#"링"#, r#"링"#, r#"링"#, r#"링"#, r#"링"#),
        (r#"맂"#, r#"맂"#, r#"맂"#, r#"맂"#, r#"맂"#),
        (r#"맃"#, r#"맃"#, r#"맃"#, r#"맃"#, r#"맃"#),
        (r#"맄"#, r#"맄"#, r#"맄"#, r#"맄"#, r#"맄"#),
        (r#"맅"#, r#"맅"#, r#"맅"#, r#"맅"#, r#"맅"#),
        (r#"맆"#, r#"맆"#, r#"맆"#, r#"맆"#, r#"맆"#),
        (r#"맇"#, r#"맇"#, r#"맇"#, r#"맇"#, r#"맇"#),
        (r#"마"#, r#"마"#, r#"마"#, r#"마"#, r#"마"#),
        (r#"막"#, r#"막"#, r#"막"#, r#"막"#, r#"막"#),
        (r#"맊"#, r#"맊"#, r#"맊"#, r#"맊"#, r#"맊"#),
        (r#"맋"#, r#"맋"#, r#"맋"#, r#"맋"#, r#"맋"#),
        (r#"만"#, r#"만"#, r#"만"#, r#"만"#, r#"만"#),
        (r#"맍"#, r#"맍"#, r#"맍"#, r#"맍"#, r#"맍"#),
        (r#"많"#, r#"많"#, r#"많"#, r#"많"#, r#"많"#),
        (r#"맏"#, r#"맏"#, r#"맏"#, r#"맏"#, r#"맏"#),
        (r#"말"#, r#"말"#, r#"말"#, r#"말"#, r#"말"#),
        (r#"맑"#, r#"맑"#, r#"맑"#, r#"맑"#, r#"맑"#),
        (r#"맒"#, r#"맒"#, r#"맒"#, r#"맒"#, r#"맒"#),
        (r#"맓"#, r#"맓"#, r#"맓"#, r#"맓"#, r#"맓"#),
        (r#"맔"#, r#"맔"#, r#"맔"#, r#"맔"#, r#"맔"#),
        (r#"맕"#, r#"맕"#, r#"맕"#, r#"맕"#, r#"맕"#),
        (r#"맖"#, r#"맖"#, r#"맖"#, r#"맖"#, r#"맖"#),
        (r#"맗"#, r#"맗"#, r#"맗"#, r#"맗"#, r#"맗"#),
        (r#"맘"#, r#"맘"#, r#"맘"#, r#"맘"#, r#"맘"#),
        (r#"맙"#, r#"맙"#, r#"맙"#, r#"맙"#, r#"맙"#),
        (r#"맚"#, r#"맚"#, r#"맚"#, r#"맚"#, r#"맚"#),
        (r#"맛"#, r#"맛"#, r#"맛"#, r#"맛"#, r#"맛"#),
        (r#"맜"#, r#"맜"#, r#"맜"#, r#"맜"#, r#"맜"#),
        (r#"망"#, r#"망"#, r#"망"#, r#"망"#, r#"망"#),
        (r#"맞"#, r#"맞"#, r#"맞"#, r#"맞"#, r#"맞"#),
        (r#"맟"#, r#"맟"#, r#"맟"#, r#"맟"#, r#"맟"#),
        (r#"맠"#, r#"맠"#, r#"맠"#, r#"맠"#, r#"맠"#),
        (r#"맡"#, r#"맡"#, r#"맡"#, r#"맡"#, r#"맡"#),
        (r#"맢"#, r#"맢"#, r#"맢"#, r#"맢"#, r#"맢"#),
        (r#"맣"#, r#"맣"#, r#"맣"#, r#"맣"#, r#"맣"#),
        (r#"매"#, r#"매"#, r#"매"#, r#"매"#, r#"매"#),
        (r#"맥"#, r#"맥"#, r#"맥"#, r#"맥"#, r#"맥"#),
        (r#"맦"#, r#"맦"#, r#"맦"#, r#"맦"#, r#"맦"#),
        (r#"맧"#, r#"맧"#, r#"맧"#, r#"맧"#, r#"맧"#),
        (r#"맨"#, r#"맨"#, r#"맨"#, r#"맨"#, r#"맨"#),
        (r#"맩"#, r#"맩"#, r#"맩"#, r#"맩"#, r#"맩"#),
        (r#"맪"#, r#"맪"#, r#"맪"#, r#"맪"#, r#"맪"#),
        (r#"맫"#, r#"맫"#, r#"맫"#, r#"맫"#, r#"맫"#),
        (r#"맬"#, r#"맬"#, r#"맬"#, r#"맬"#, r#"맬"#),
        (r#"맭"#, r#"맭"#, r#"맭"#, r#"맭"#, r#"맭"#),
        (r#"맮"#, r#"맮"#, r#"맮"#, r#"맮"#, r#"맮"#),
        (r#"맯"#, r#"맯"#, r#"맯"#, r#"맯"#, r#"맯"#),
        (r#"맰"#, r#"맰"#, r#"맰"#, r#"맰"#, r#"맰"#),
        (r#"맱"#, r#"맱"#, r#"맱"#, r#"맱"#, r#"맱"#),
        (r#"맲"#, r#"맲"#, r#"맲"#, r#"맲"#, r#"맲"#),
        (r#"맳"#, r#"맳"#, r#"맳"#, r#"맳"#, r#"맳"#),
        (r#"맴"#, r#"맴"#, r#"맴"#, r#"맴"#, r#"맴"#),
        (r#"맵"#, r#"맵"#, r#"맵"#, r#"맵"#, r#"맵"#),
        (r#"맶"#, r#"맶"#, r#"맶"#, r#"맶"#, r#"맶"#),
        (r#"맷"#, r#"맷"#, r#"맷"#, r#"맷"#, r#"맷"#),
        (r#"맸"#, r#"맸"#, r#"맸"#, r#"맸"#, r#"맸"#),
        (r#"맹"#, r#"맹"#, r#"맹"#, r#"맹"#, r#"맹"#),
        (r#"맺"#, r#"맺"#, r#"맺"#, r#"맺"#, r#"맺"#),
        (r#"맻"#, r#"맻"#, r#"맻"#, r#"맻"#, r#"맻"#),
        (r#"맼"#, r#"맼"#, r#"맼"#, r#"맼"#, r#"맼"#),
        (r#"맽"#, r#"맽"#, r#"맽"#, r#"맽"#, r#"맽"#),
        (r#"맾"#, r#"맾"#, r#"맾"#, r#"맾"#, r#"맾"#),
        (r#"맿"#, r#"맿"#, r#"맿"#, r#"맿"#, r#"맿"#),
        (r#"먀"#, r#"먀"#, r#"먀"#, r#"먀"#, r#"먀"#),
        (r#"먁"#, r#"먁"#, r#"먁"#, r#"먁"#, r#"먁"#),
        (r#"먂"#, r#"먂"#, r#"먂"#, r#"먂"#, r#"먂"#),
        (r#"먃"#, r#"먃"#, r#"먃"#, r#"먃"#, r#"먃"#),
        (r#"먄"#, r#"먄"#, r#"먄"#, r#"먄"#, r#"먄"#),
        (r#"먅"#, r#"먅"#, r#"먅"#, r#"먅"#, r#"먅"#),
        (r#"먆"#, r#"먆"#, r#"먆"#, r#"먆"#, r#"먆"#),
        (r#"먇"#, r#"먇"#, r#"먇"#, r#"먇"#, r#"먇"#),
        (r#"먈"#, r#"먈"#, r#"먈"#, r#"먈"#, r#"먈"#),
        (r#"먉"#, r#"먉"#, r#"먉"#, r#"먉"#, r#"먉"#),
        (r#"먊"#, r#"먊"#, r#"먊"#, r#"먊"#, r#"먊"#),
        (r#"먋"#, r#"먋"#, r#"먋"#, r#"먋"#, r#"먋"#),
        (r#"먌"#, r#"먌"#, r#"먌"#, r#"먌"#, r#"먌"#),
        (r#"먍"#, r#"먍"#, r#"먍"#, r#"먍"#, r#"먍"#),
        (r#"먎"#, r#"먎"#, r#"먎"#, r#"먎"#, r#"먎"#),
        (r#"먏"#, r#"먏"#, r#"먏"#, r#"먏"#, r#"먏"#),
        (r#"먐"#, r#"먐"#, r#"먐"#, r#"먐"#, r#"먐"#),
        (r#"먑"#, r#"먑"#, r#"먑"#, r#"먑"#, r#"먑"#),
        (r#"먒"#, r#"먒"#, r#"먒"#, r#"먒"#, r#"먒"#),
        (r#"먓"#, r#"먓"#, r#"먓"#, r#"먓"#, r#"먓"#),
        (r#"먔"#, r#"먔"#, r#"먔"#, r#"먔"#, r#"먔"#),
        (r#"먕"#, r#"먕"#, r#"먕"#, r#"먕"#, r#"먕"#),
        (r#"먖"#, r#"먖"#, r#"먖"#, r#"먖"#, r#"먖"#),
        (r#"먗"#, r#"먗"#, r#"먗"#, r#"먗"#, r#"먗"#),
        (r#"먘"#, r#"먘"#, r#"먘"#, r#"먘"#, r#"먘"#),
        (r#"먙"#, r#"먙"#, r#"먙"#, r#"먙"#, r#"먙"#),
        (r#"먚"#, r#"먚"#, r#"먚"#, r#"먚"#, r#"먚"#),
        (r#"먛"#, r#"먛"#, r#"먛"#, r#"먛"#, r#"먛"#),
        (r#"먜"#, r#"먜"#, r#"먜"#, r#"먜"#, r#"먜"#),
        (r#"먝"#, r#"먝"#, r#"먝"#, r#"먝"#, r#"먝"#),
        (r#"먞"#, r#"먞"#, r#"먞"#, r#"먞"#, r#"먞"#),
        (r#"먟"#, r#"먟"#, r#"먟"#, r#"먟"#, r#"먟"#),
        (r#"먠"#, r#"먠"#, r#"먠"#, r#"먠"#, r#"먠"#),
        (r#"먡"#, r#"먡"#, r#"먡"#, r#"먡"#, r#"먡"#),
        (r#"먢"#, r#"먢"#, r#"먢"#, r#"먢"#, r#"먢"#),
        (r#"먣"#, r#"먣"#, r#"먣"#, r#"먣"#, r#"먣"#),
        (r#"먤"#, r#"먤"#, r#"먤"#, r#"먤"#, r#"먤"#),
        (r#"먥"#, r#"먥"#, r#"먥"#, r#"먥"#, r#"먥"#),
        (r#"먦"#, r#"먦"#, r#"먦"#, r#"먦"#, r#"먦"#),
        (r#"먧"#, r#"먧"#, r#"먧"#, r#"먧"#, r#"먧"#),
        (r#"먨"#, r#"먨"#, r#"먨"#, r#"먨"#, r#"먨"#),
        (r#"먩"#, r#"먩"#, r#"먩"#, r#"먩"#, r#"먩"#),
        (r#"먪"#, r#"먪"#, r#"먪"#, r#"먪"#, r#"먪"#),
        (r#"먫"#, r#"먫"#, r#"먫"#, r#"먫"#, r#"먫"#),
        (r#"먬"#, r#"먬"#, r#"먬"#, r#"먬"#, r#"먬"#),
        (r#"먭"#, r#"먭"#, r#"먭"#, r#"먭"#, r#"먭"#),
        (r#"먮"#, r#"먮"#, r#"먮"#, r#"먮"#, r#"먮"#),
        (r#"먯"#, r#"먯"#, r#"먯"#, r#"먯"#, r#"먯"#),
        (r#"먰"#, r#"먰"#, r#"먰"#, r#"먰"#, r#"먰"#),
        (r#"먱"#, r#"먱"#, r#"먱"#, r#"먱"#, r#"먱"#),
        (r#"먲"#, r#"먲"#, r#"먲"#, r#"먲"#, r#"먲"#),
        (r#"먳"#, r#"먳"#, r#"먳"#, r#"먳"#, r#"먳"#),
        (r#"먴"#, r#"먴"#, r#"먴"#, r#"먴"#, r#"먴"#),
        (r#"먵"#, r#"먵"#, r#"먵"#, r#"먵"#, r#"먵"#),
        (r#"먶"#, r#"먶"#, r#"먶"#, r#"먶"#, r#"먶"#),
        (r#"먷"#, r#"먷"#, r#"먷"#, r#"먷"#, r#"먷"#),
        (r#"머"#, r#"머"#, r#"머"#, r#"머"#, r#"머"#),
        (r#"먹"#, r#"먹"#, r#"먹"#, r#"먹"#, r#"먹"#),
        (r#"먺"#, r#"먺"#, r#"먺"#, r#"먺"#, r#"먺"#),
        (r#"먻"#, r#"먻"#, r#"먻"#, r#"먻"#, r#"먻"#),
        (r#"먼"#, r#"먼"#, r#"먼"#, r#"먼"#, r#"먼"#),
        (r#"먽"#, r#"먽"#, r#"먽"#, r#"먽"#, r#"먽"#),
        (r#"먾"#, r#"먾"#, r#"먾"#, r#"먾"#, r#"먾"#),
        (r#"먿"#, r#"먿"#, r#"먿"#, r#"먿"#, r#"먿"#),
        (r#"멀"#, r#"멀"#, r#"멀"#, r#"멀"#, r#"멀"#),
        (r#"멁"#, r#"멁"#, r#"멁"#, r#"멁"#, r#"멁"#),
        (r#"멂"#, r#"멂"#, r#"멂"#, r#"멂"#, r#"멂"#),
        (r#"멃"#, r#"멃"#, r#"멃"#, r#"멃"#, r#"멃"#),
        (r#"멄"#, r#"멄"#, r#"멄"#, r#"멄"#, r#"멄"#),
        (r#"멅"#, r#"멅"#, r#"멅"#, r#"멅"#, r#"멅"#),
        (r#"멆"#, r#"멆"#, r#"멆"#, r#"멆"#, r#"멆"#),
        (r#"멇"#, r#"멇"#, r#"멇"#, r#"멇"#, r#"멇"#),
        (r#"멈"#, r#"멈"#, r#"멈"#, r#"멈"#, r#"멈"#),
        (r#"멉"#, r#"멉"#, r#"멉"#, r#"멉"#, r#"멉"#),
        (r#"멊"#, r#"멊"#, r#"멊"#, r#"멊"#, r#"멊"#),
        (r#"멋"#, r#"멋"#, r#"멋"#, r#"멋"#, r#"멋"#),
        (r#"멌"#, r#"멌"#, r#"멌"#, r#"멌"#, r#"멌"#),
        (r#"멍"#, r#"멍"#, r#"멍"#, r#"멍"#, r#"멍"#),
        (r#"멎"#, r#"멎"#, r#"멎"#, r#"멎"#, r#"멎"#),
        (r#"멏"#, r#"멏"#, r#"멏"#, r#"멏"#, r#"멏"#),
        (r#"멐"#, r#"멐"#, r#"멐"#, r#"멐"#, r#"멐"#),
        (r#"멑"#, r#"멑"#, r#"멑"#, r#"멑"#, r#"멑"#),
        (r#"멒"#, r#"멒"#, r#"멒"#, r#"멒"#, r#"멒"#),
        (r#"멓"#, r#"멓"#, r#"멓"#, r#"멓"#, r#"멓"#),
        (r#"메"#, r#"메"#, r#"메"#, r#"메"#, r#"메"#),
        (r#"멕"#, r#"멕"#, r#"멕"#, r#"멕"#, r#"멕"#),
        (r#"멖"#, r#"멖"#, r#"멖"#, r#"멖"#, r#"멖"#),
        (r#"멗"#, r#"멗"#, r#"멗"#, r#"멗"#, r#"멗"#),
        (r#"멘"#, r#"멘"#, r#"멘"#, r#"멘"#, r#"멘"#),
        (r#"멙"#, r#"멙"#, r#"멙"#, r#"멙"#, r#"멙"#),
        (r#"멚"#, r#"멚"#, r#"멚"#, r#"멚"#, r#"멚"#),
        (r#"멛"#, r#"멛"#, r#"멛"#, r#"멛"#, r#"멛"#),
        (r#"멜"#, r#"멜"#, r#"멜"#, r#"멜"#, r#"멜"#),
        (r#"멝"#, r#"멝"#, r#"멝"#, r#"멝"#, r#"멝"#),
        (r#"멞"#, r#"멞"#, r#"멞"#, r#"멞"#, r#"멞"#),
        (r#"멟"#, r#"멟"#, r#"멟"#, r#"멟"#, r#"멟"#),
        (r#"멠"#, r#"멠"#, r#"멠"#, r#"멠"#, r#"멠"#),
        (r#"멡"#, r#"멡"#, r#"멡"#, r#"멡"#, r#"멡"#),
        (r#"멢"#, r#"멢"#, r#"멢"#, r#"멢"#, r#"멢"#),
        (r#"멣"#, r#"멣"#, r#"멣"#, r#"멣"#, r#"멣"#),
        (r#"멤"#, r#"멤"#, r#"멤"#, r#"멤"#, r#"멤"#),
        (r#"멥"#, r#"멥"#, r#"멥"#, r#"멥"#, r#"멥"#),
        (r#"멦"#, r#"멦"#, r#"멦"#, r#"멦"#, r#"멦"#),
        (r#"멧"#, r#"멧"#, r#"멧"#, r#"멧"#, r#"멧"#),
        (r#"멨"#, r#"멨"#, r#"멨"#, r#"멨"#, r#"멨"#),
        (r#"멩"#, r#"멩"#, r#"멩"#, r#"멩"#, r#"멩"#),
        (r#"멪"#, r#"멪"#, r#"멪"#, r#"멪"#, r#"멪"#),
        (r#"멫"#, r#"멫"#, r#"멫"#, r#"멫"#, r#"멫"#),
        (r#"멬"#, r#"멬"#, r#"멬"#, r#"멬"#, r#"멬"#),
        (r#"멭"#, r#"멭"#, r#"멭"#, r#"멭"#, r#"멭"#),
        (r#"멮"#, r#"멮"#, r#"멮"#, r#"멮"#, r#"멮"#),
        (r#"멯"#, r#"멯"#, r#"멯"#, r#"멯"#, r#"멯"#),
        (r#"며"#, r#"며"#, r#"며"#, r#"며"#, r#"며"#),
        (r#"멱"#, r#"멱"#, r#"멱"#, r#"멱"#, r#"멱"#),
        (r#"멲"#, r#"멲"#, r#"멲"#, r#"멲"#, r#"멲"#),
        (r#"멳"#, r#"멳"#, r#"멳"#, r#"멳"#, r#"멳"#),
        (r#"면"#, r#"면"#, r#"면"#, r#"면"#, r#"면"#),
        (r#"멵"#, r#"멵"#, r#"멵"#, r#"멵"#, r#"멵"#),
        (r#"멶"#, r#"멶"#, r#"멶"#, r#"멶"#, r#"멶"#),
        (r#"멷"#, r#"멷"#, r#"멷"#, r#"멷"#, r#"멷"#),
        (r#"멸"#, r#"멸"#, r#"멸"#, r#"멸"#, r#"멸"#),
        (r#"멹"#, r#"멹"#, r#"멹"#, r#"멹"#, r#"멹"#),
        (r#"멺"#, r#"멺"#, r#"멺"#, r#"멺"#, r#"멺"#),
        (r#"멻"#, r#"멻"#, r#"멻"#, r#"멻"#, r#"멻"#),
        (r#"멼"#, r#"멼"#, r#"멼"#, r#"멼"#, r#"멼"#),
        (r#"멽"#, r#"멽"#, r#"멽"#, r#"멽"#, r#"멽"#),
        (r#"멾"#, r#"멾"#, r#"멾"#, r#"멾"#, r#"멾"#),
        (r#"멿"#, r#"멿"#, r#"멿"#, r#"멿"#, r#"멿"#),
        (r#"몀"#, r#"몀"#, r#"몀"#, r#"몀"#, r#"몀"#),
        (r#"몁"#, r#"몁"#, r#"몁"#, r#"몁"#, r#"몁"#),
        (r#"몂"#, r#"몂"#, r#"몂"#, r#"몂"#, r#"몂"#),
        (r#"몃"#, r#"몃"#, r#"몃"#, r#"몃"#, r#"몃"#),
        (r#"몄"#, r#"몄"#, r#"몄"#, r#"몄"#, r#"몄"#),
        (r#"명"#, r#"명"#, r#"명"#, r#"명"#, r#"명"#),
        (r#"몆"#, r#"몆"#, r#"몆"#, r#"몆"#, r#"몆"#),
        (r#"몇"#, r#"몇"#, r#"몇"#, r#"몇"#, r#"몇"#),
        (r#"몈"#, r#"몈"#, r#"몈"#, r#"몈"#, r#"몈"#),
        (r#"몉"#, r#"몉"#, r#"몉"#, r#"몉"#, r#"몉"#),
        (r#"몊"#, r#"몊"#, r#"몊"#, r#"몊"#, r#"몊"#),
        (r#"몋"#, r#"몋"#, r#"몋"#, r#"몋"#, r#"몋"#),
        (r#"몌"#, r#"몌"#, r#"몌"#, r#"몌"#, r#"몌"#),
        (r#"몍"#, r#"몍"#, r#"몍"#, r#"몍"#, r#"몍"#),
        (r#"몎"#, r#"몎"#, r#"몎"#, r#"몎"#, r#"몎"#),
        (r#"몏"#, r#"몏"#, r#"몏"#, r#"몏"#, r#"몏"#),
        (r#"몐"#, r#"몐"#, r#"몐"#, r#"몐"#, r#"몐"#),
        (r#"몑"#, r#"몑"#, r#"몑"#, r#"몑"#, r#"몑"#),
        (r#"몒"#, r#"몒"#, r#"몒"#, r#"몒"#, r#"몒"#),
        (r#"몓"#, r#"몓"#, r#"몓"#, r#"몓"#, r#"몓"#),
        (r#"몔"#, r#"몔"#, r#"몔"#, r#"몔"#, r#"몔"#),
        (r#"몕"#, r#"몕"#, r#"몕"#, r#"몕"#, r#"몕"#),
        (r#"몖"#, r#"몖"#, r#"몖"#, r#"몖"#, r#"몖"#),
        (r#"몗"#, r#"몗"#, r#"몗"#, r#"몗"#, r#"몗"#),
        (r#"몘"#, r#"몘"#, r#"몘"#, r#"몘"#, r#"몘"#),
        (r#"몙"#, r#"몙"#, r#"몙"#, r#"몙"#, r#"몙"#),
        (r#"몚"#, r#"몚"#, r#"몚"#, r#"몚"#, r#"몚"#),
        (r#"몛"#, r#"몛"#, r#"몛"#, r#"몛"#, r#"몛"#),
        (r#"몜"#, r#"몜"#, r#"몜"#, r#"몜"#, r#"몜"#),
        (r#"몝"#, r#"몝"#, r#"몝"#, r#"몝"#, r#"몝"#),
        (r#"몞"#, r#"몞"#, r#"몞"#, r#"몞"#, r#"몞"#),
        (r#"몟"#, r#"몟"#, r#"몟"#, r#"몟"#, r#"몟"#),
        (r#"몠"#, r#"몠"#, r#"몠"#, r#"몠"#, r#"몠"#),
        (r#"몡"#, r#"몡"#, r#"몡"#, r#"몡"#, r#"몡"#),
        (r#"몢"#, r#"몢"#, r#"몢"#, r#"몢"#, r#"몢"#),
        (r#"몣"#, r#"몣"#, r#"몣"#, r#"몣"#, r#"몣"#),
        (r#"몤"#, r#"몤"#, r#"몤"#, r#"몤"#, r#"몤"#),
        (r#"몥"#, r#"몥"#, r#"몥"#, r#"몥"#, r#"몥"#),
        (r#"몦"#, r#"몦"#, r#"몦"#, r#"몦"#, r#"몦"#),
        (r#"몧"#, r#"몧"#, r#"몧"#, r#"몧"#, r#"몧"#),
        (r#"모"#, r#"모"#, r#"모"#, r#"모"#, r#"모"#),
        (r#"목"#, r#"목"#, r#"목"#, r#"목"#, r#"목"#),
        (r#"몪"#, r#"몪"#, r#"몪"#, r#"몪"#, r#"몪"#),
        (r#"몫"#, r#"몫"#, r#"몫"#, r#"몫"#, r#"몫"#),
        (r#"몬"#, r#"몬"#, r#"몬"#, r#"몬"#, r#"몬"#),
        (r#"몭"#, r#"몭"#, r#"몭"#, r#"몭"#, r#"몭"#),
        (r#"몮"#, r#"몮"#, r#"몮"#, r#"몮"#, r#"몮"#),
        (r#"몯"#, r#"몯"#, r#"몯"#, r#"몯"#, r#"몯"#),
        (r#"몰"#, r#"몰"#, r#"몰"#, r#"몰"#, r#"몰"#),
        (r#"몱"#, r#"몱"#, r#"몱"#, r#"몱"#, r#"몱"#),
        (r#"몲"#, r#"몲"#, r#"몲"#, r#"몲"#, r#"몲"#),
        (r#"몳"#, r#"몳"#, r#"몳"#, r#"몳"#, r#"몳"#),
        (r#"몴"#, r#"몴"#, r#"몴"#, r#"몴"#, r#"몴"#),
        (r#"몵"#, r#"몵"#, r#"몵"#, r#"몵"#, r#"몵"#),
        (r#"몶"#, r#"몶"#, r#"몶"#, r#"몶"#, r#"몶"#),
        (r#"몷"#, r#"몷"#, r#"몷"#, r#"몷"#, r#"몷"#),
        (r#"몸"#, r#"몸"#, r#"몸"#, r#"몸"#, r#"몸"#),
        (r#"몹"#, r#"몹"#, r#"몹"#, r#"몹"#, r#"몹"#),
        (r#"몺"#, r#"몺"#, r#"몺"#, r#"몺"#, r#"몺"#),
        (r#"못"#, r#"못"#, r#"못"#, r#"못"#, r#"못"#),
        (r#"몼"#, r#"몼"#, r#"몼"#, r#"몼"#, r#"몼"#),
        (r#"몽"#, r#"몽"#, r#"몽"#, r#"몽"#, r#"몽"#),
        (r#"몾"#, r#"몾"#, r#"몾"#, r#"몾"#, r#"몾"#),
        (r#"몿"#, r#"몿"#, r#"몿"#, r#"몿"#, r#"몿"#),
        (r#"뫀"#, r#"뫀"#, r#"뫀"#, r#"뫀"#, r#"뫀"#),
        (r#"뫁"#, r#"뫁"#, r#"뫁"#, r#"뫁"#, r#"뫁"#),
        (r#"뫂"#, r#"뫂"#, r#"뫂"#, r#"뫂"#, r#"뫂"#),
        (r#"뫃"#, r#"뫃"#, r#"뫃"#, r#"뫃"#, r#"뫃"#),
        (r#"뫄"#, r#"뫄"#, r#"뫄"#, r#"뫄"#, r#"뫄"#),
        (r#"뫅"#, r#"뫅"#, r#"뫅"#, r#"뫅"#, r#"뫅"#),
        (r#"뫆"#, r#"뫆"#, r#"뫆"#, r#"뫆"#, r#"뫆"#),
        (r#"뫇"#, r#"뫇"#, r#"뫇"#, r#"뫇"#, r#"뫇"#),
        (r#"뫈"#, r#"뫈"#, r#"뫈"#, r#"뫈"#, r#"뫈"#),
        (r#"뫉"#, r#"뫉"#, r#"뫉"#, r#"뫉"#, r#"뫉"#),
        (r#"뫊"#, r#"뫊"#, r#"뫊"#, r#"뫊"#, r#"뫊"#),
        (r#"뫋"#, r#"뫋"#, r#"뫋"#, r#"뫋"#, r#"뫋"#),
        (r#"뫌"#, r#"뫌"#, r#"뫌"#, r#"뫌"#, r#"뫌"#),
        (r#"뫍"#, r#"뫍"#, r#"뫍"#, r#"뫍"#, r#"뫍"#),
        (r#"뫎"#, r#"뫎"#, r#"뫎"#, r#"뫎"#, r#"뫎"#),
        (r#"뫏"#, r#"뫏"#, r#"뫏"#, r#"뫏"#, r#"뫏"#),
        (r#"뫐"#, r#"뫐"#, r#"뫐"#, r#"뫐"#, r#"뫐"#),
        (r#"뫑"#, r#"뫑"#, r#"뫑"#, r#"뫑"#, r#"뫑"#),
        (r#"뫒"#, r#"뫒"#, r#"뫒"#, r#"뫒"#, r#"뫒"#),
        (r#"뫓"#, r#"뫓"#, r#"뫓"#, r#"뫓"#, r#"뫓"#),
        (r#"뫔"#, r#"뫔"#, r#"뫔"#, r#"뫔"#, r#"뫔"#),
        (r#"뫕"#, r#"뫕"#, r#"뫕"#, r#"뫕"#, r#"뫕"#),
        (r#"뫖"#, r#"뫖"#, r#"뫖"#, r#"뫖"#, r#"뫖"#),
        (r#"뫗"#, r#"뫗"#, r#"뫗"#, r#"뫗"#, r#"뫗"#),
        (r#"뫘"#, r#"뫘"#, r#"뫘"#, r#"뫘"#, r#"뫘"#),
        (r#"뫙"#, r#"뫙"#, r#"뫙"#, r#"뫙"#, r#"뫙"#),
        (r#"뫚"#, r#"뫚"#, r#"뫚"#, r#"뫚"#, r#"뫚"#),
        (r#"뫛"#, r#"뫛"#, r#"뫛"#, r#"뫛"#, r#"뫛"#),
        (r#"뫜"#, r#"뫜"#, r#"뫜"#, r#"뫜"#, r#"뫜"#),
        (r#"뫝"#, r#"뫝"#, r#"뫝"#, r#"뫝"#, r#"뫝"#),
        (r#"뫞"#, r#"뫞"#, r#"뫞"#, r#"뫞"#, r#"뫞"#),
        (r#"뫟"#, r#"뫟"#, r#"뫟"#, r#"뫟"#, r#"뫟"#),
        (r#"뫠"#, r#"뫠"#, r#"뫠"#, r#"뫠"#, r#"뫠"#),
        (r#"뫡"#, r#"뫡"#, r#"뫡"#, r#"뫡"#, r#"뫡"#),
        (r#"뫢"#, r#"뫢"#, r#"뫢"#, r#"뫢"#, r#"뫢"#),
        (r#"뫣"#, r#"뫣"#, r#"뫣"#, r#"뫣"#, r#"뫣"#),
        (r#"뫤"#, r#"뫤"#, r#"뫤"#, r#"뫤"#, r#"뫤"#),
        (r#"뫥"#, r#"뫥"#, r#"뫥"#, r#"뫥"#, r#"뫥"#),
        (r#"뫦"#, r#"뫦"#, r#"뫦"#, r#"뫦"#, r#"뫦"#),
        (r#"뫧"#, r#"뫧"#, r#"뫧"#, r#"뫧"#, r#"뫧"#),
        (r#"뫨"#, r#"뫨"#, r#"뫨"#, r#"뫨"#, r#"뫨"#),
        (r#"뫩"#, r#"뫩"#, r#"뫩"#, r#"뫩"#, r#"뫩"#),
        (r#"뫪"#, r#"뫪"#, r#"뫪"#, r#"뫪"#, r#"뫪"#),
        (r#"뫫"#, r#"뫫"#, r#"뫫"#, r#"뫫"#, r#"뫫"#),
        (r#"뫬"#, r#"뫬"#, r#"뫬"#, r#"뫬"#, r#"뫬"#),
        (r#"뫭"#, r#"뫭"#, r#"뫭"#, r#"뫭"#, r#"뫭"#),
        (r#"뫮"#, r#"뫮"#, r#"뫮"#, r#"뫮"#, r#"뫮"#),
        (r#"뫯"#, r#"뫯"#, r#"뫯"#, r#"뫯"#, r#"뫯"#),
        (r#"뫰"#, r#"뫰"#, r#"뫰"#, r#"뫰"#, r#"뫰"#),
        (r#"뫱"#, r#"뫱"#, r#"뫱"#, r#"뫱"#, r#"뫱"#),
        (r#"뫲"#, r#"뫲"#, r#"뫲"#, r#"뫲"#, r#"뫲"#),
        (r#"뫳"#, r#"뫳"#, r#"뫳"#, r#"뫳"#, r#"뫳"#),
        (r#"뫴"#, r#"뫴"#, r#"뫴"#, r#"뫴"#, r#"뫴"#),
        (r#"뫵"#, r#"뫵"#, r#"뫵"#, r#"뫵"#, r#"뫵"#),
        (r#"뫶"#, r#"뫶"#, r#"뫶"#, r#"뫶"#, r#"뫶"#),
        (r#"뫷"#, r#"뫷"#, r#"뫷"#, r#"뫷"#, r#"뫷"#),
        (r#"뫸"#, r#"뫸"#, r#"뫸"#, r#"뫸"#, r#"뫸"#),
        (r#"뫹"#, r#"뫹"#, r#"뫹"#, r#"뫹"#, r#"뫹"#),
        (r#"뫺"#, r#"뫺"#, r#"뫺"#, r#"뫺"#, r#"뫺"#),
        (r#"뫻"#, r#"뫻"#, r#"뫻"#, r#"뫻"#, r#"뫻"#),
        (r#"뫼"#, r#"뫼"#, r#"뫼"#, r#"뫼"#, r#"뫼"#),
        (r#"뫽"#, r#"뫽"#, r#"뫽"#, r#"뫽"#, r#"뫽"#),
        (r#"뫾"#, r#"뫾"#, r#"뫾"#, r#"뫾"#, r#"뫾"#),
        (r#"뫿"#, r#"뫿"#, r#"뫿"#, r#"뫿"#, r#"뫿"#),
        (r#"묀"#, r#"묀"#, r#"묀"#, r#"묀"#, r#"묀"#),
        (r#"묁"#, r#"묁"#, r#"묁"#, r#"묁"#, r#"묁"#),
        (r#"묂"#, r#"묂"#, r#"묂"#, r#"묂"#, r#"묂"#),
        (r#"묃"#, r#"묃"#, r#"묃"#, r#"묃"#, r#"묃"#),
        (r#"묄"#, r#"묄"#, r#"묄"#, r#"묄"#, r#"묄"#),
        (r#"묅"#, r#"묅"#, r#"묅"#, r#"묅"#, r#"묅"#),
        (r#"묆"#, r#"묆"#, r#"묆"#, r#"묆"#, r#"묆"#),
        (r#"묇"#, r#"묇"#, r#"묇"#, r#"묇"#, r#"묇"#),
        (r#"묈"#, r#"묈"#, r#"묈"#, r#"묈"#, r#"묈"#),
        (r#"묉"#, r#"묉"#, r#"묉"#, r#"묉"#, r#"묉"#),
        (r#"묊"#, r#"묊"#, r#"묊"#, r#"묊"#, r#"묊"#),
        (r#"묋"#, r#"묋"#, r#"묋"#, r#"묋"#, r#"묋"#),
        (r#"묌"#, r#"묌"#, r#"묌"#, r#"묌"#, r#"묌"#),
        (r#"묍"#, r#"묍"#, r#"묍"#, r#"묍"#, r#"묍"#),
        (r#"묎"#, r#"묎"#, r#"묎"#, r#"묎"#, r#"묎"#),
        (r#"묏"#, r#"묏"#, r#"묏"#, r#"묏"#, r#"묏"#),
        (r#"묐"#, r#"묐"#, r#"묐"#, r#"묐"#, r#"묐"#),
        (r#"묑"#, r#"묑"#, r#"묑"#, r#"묑"#, r#"묑"#),
        (r#"묒"#, r#"묒"#, r#"묒"#, r#"묒"#, r#"묒"#),
        (r#"묓"#, r#"묓"#, r#"묓"#, r#"묓"#, r#"묓"#),
        (r#"묔"#, r#"묔"#, r#"묔"#, r#"묔"#, r#"묔"#),
        (r#"묕"#, r#"묕"#, r#"묕"#, r#"묕"#, r#"묕"#),
        (r#"묖"#, r#"묖"#, r#"묖"#, r#"묖"#, r#"묖"#),
        (r#"묗"#, r#"묗"#, r#"묗"#, r#"묗"#, r#"묗"#),
        (r#"묘"#, r#"묘"#, r#"묘"#, r#"묘"#, r#"묘"#),
        (r#"묙"#, r#"묙"#, r#"묙"#, r#"묙"#, r#"묙"#),
        (r#"묚"#, r#"묚"#, r#"묚"#, r#"묚"#, r#"묚"#),
        (r#"묛"#, r#"묛"#, r#"묛"#, r#"묛"#, r#"묛"#),
        (r#"묜"#, r#"묜"#, r#"묜"#, r#"묜"#, r#"묜"#),
        (r#"묝"#, r#"묝"#, r#"묝"#, r#"묝"#, r#"묝"#),
        (r#"묞"#, r#"묞"#, r#"묞"#, r#"묞"#, r#"묞"#),
        (r#"묟"#, r#"묟"#, r#"묟"#, r#"묟"#, r#"묟"#),
        (r#"묠"#, r#"묠"#, r#"묠"#, r#"묠"#, r#"묠"#),
        (r#"묡"#, r#"묡"#, r#"묡"#, r#"묡"#, r#"묡"#),
        (r#"묢"#, r#"묢"#, r#"묢"#, r#"묢"#, r#"묢"#),
        (r#"묣"#, r#"묣"#, r#"묣"#, r#"묣"#, r#"묣"#),
        (r#"묤"#, r#"묤"#, r#"묤"#, r#"묤"#, r#"묤"#),
        (r#"묥"#, r#"묥"#, r#"묥"#, r#"묥"#, r#"묥"#),
        (r#"묦"#, r#"묦"#, r#"묦"#, r#"묦"#, r#"묦"#),
        (r#"묧"#, r#"묧"#, r#"묧"#, r#"묧"#, r#"묧"#),
        (r#"묨"#, r#"묨"#, r#"묨"#, r#"묨"#, r#"묨"#),
        (r#"묩"#, r#"묩"#, r#"묩"#, r#"묩"#, r#"묩"#),
        (r#"묪"#, r#"묪"#, r#"묪"#, r#"묪"#, r#"묪"#),
        (r#"묫"#, r#"묫"#, r#"묫"#, r#"묫"#, r#"묫"#),
        (r#"묬"#, r#"묬"#, r#"묬"#, r#"묬"#, r#"묬"#),
        (r#"묭"#, r#"묭"#, r#"묭"#, r#"묭"#, r#"묭"#),
        (r#"묮"#, r#"묮"#, r#"묮"#, r#"묮"#, r#"묮"#),
        (r#"묯"#, r#"묯"#, r#"묯"#, r#"묯"#, r#"묯"#),
        (r#"묰"#, r#"묰"#, r#"묰"#, r#"묰"#, r#"묰"#),
        (r#"묱"#, r#"묱"#, r#"묱"#, r#"묱"#, r#"묱"#),
        (r#"묲"#, r#"묲"#, r#"묲"#, r#"묲"#, r#"묲"#),
        (r#"묳"#, r#"묳"#, r#"묳"#, r#"묳"#, r#"묳"#),
        (r#"무"#, r#"무"#, r#"무"#, r#"무"#, r#"무"#),
        (r#"묵"#, r#"묵"#, r#"묵"#, r#"묵"#, r#"묵"#),
        (r#"묶"#, r#"묶"#, r#"묶"#, r#"묶"#, r#"묶"#),
        (r#"묷"#, r#"묷"#, r#"묷"#, r#"묷"#, r#"묷"#),
        (r#"문"#, r#"문"#, r#"문"#, r#"문"#, r#"문"#),
        (r#"묹"#, r#"묹"#, r#"묹"#, r#"묹"#, r#"묹"#),
        (r#"묺"#, r#"묺"#, r#"묺"#, r#"묺"#, r#"묺"#),
        (r#"묻"#, r#"묻"#, r#"묻"#, r#"묻"#, r#"묻"#),
        (r#"물"#, r#"물"#, r#"물"#, r#"물"#, r#"물"#),
        (r#"묽"#, r#"묽"#, r#"묽"#, r#"묽"#, r#"묽"#),
        (r#"묾"#, r#"묾"#, r#"묾"#, r#"묾"#, r#"묾"#),
        (r#"묿"#, r#"묿"#, r#"묿"#, r#"묿"#, r#"묿"#),
        (r#"뭀"#, r#"뭀"#, r#"뭀"#, r#"뭀"#, r#"뭀"#),
        (r#"뭁"#, r#"뭁"#, r#"뭁"#, r#"뭁"#, r#"뭁"#),
        (r#"뭂"#, r#"뭂"#, r#"뭂"#, r#"뭂"#, r#"뭂"#),
        (r#"뭃"#, r#"뭃"#, r#"뭃"#, r#"뭃"#, r#"뭃"#),
        (r#"뭄"#, r#"뭄"#, r#"뭄"#, r#"뭄"#, r#"뭄"#),
        (r#"뭅"#, r#"뭅"#, r#"뭅"#, r#"뭅"#, r#"뭅"#),
        (r#"뭆"#, r#"뭆"#, r#"뭆"#, r#"뭆"#, r#"뭆"#),
        (r#"뭇"#, r#"뭇"#, r#"뭇"#, r#"뭇"#, r#"뭇"#),
        (r#"뭈"#, r#"뭈"#, r#"뭈"#, r#"뭈"#, r#"뭈"#),
        (r#"뭉"#, r#"뭉"#, r#"뭉"#, r#"뭉"#, r#"뭉"#),
        (r#"뭊"#, r#"뭊"#, r#"뭊"#, r#"뭊"#, r#"뭊"#),
        (r#"뭋"#, r#"뭋"#, r#"뭋"#, r#"뭋"#, r#"뭋"#),
        (r#"뭌"#, r#"뭌"#, r#"뭌"#, r#"뭌"#, r#"뭌"#),
        (r#"뭍"#, r#"뭍"#, r#"뭍"#, r#"뭍"#, r#"뭍"#),
        (r#"뭎"#, r#"뭎"#, r#"뭎"#, r#"뭎"#, r#"뭎"#),
        (r#"뭏"#, r#"뭏"#, r#"뭏"#, r#"뭏"#, r#"뭏"#),
        (r#"뭐"#, r#"뭐"#, r#"뭐"#, r#"뭐"#, r#"뭐"#),
        (r#"뭑"#, r#"뭑"#, r#"뭑"#, r#"뭑"#, r#"뭑"#),
        (r#"뭒"#, r#"뭒"#, r#"뭒"#, r#"뭒"#, r#"뭒"#),
        (r#"뭓"#, r#"뭓"#, r#"뭓"#, r#"뭓"#, r#"뭓"#),
        (r#"뭔"#, r#"뭔"#, r#"뭔"#, r#"뭔"#, r#"뭔"#),
        (r#"뭕"#, r#"뭕"#, r#"뭕"#, r#"뭕"#, r#"뭕"#),
        (r#"뭖"#, r#"뭖"#, r#"뭖"#, r#"뭖"#, r#"뭖"#),
        (r#"뭗"#, r#"뭗"#, r#"뭗"#, r#"뭗"#, r#"뭗"#),
        (r#"뭘"#, r#"뭘"#, r#"뭘"#, r#"뭘"#, r#"뭘"#),
        (r#"뭙"#, r#"뭙"#, r#"뭙"#, r#"뭙"#, r#"뭙"#),
        (r#"뭚"#, r#"뭚"#, r#"뭚"#, r#"뭚"#, r#"뭚"#),
        (r#"뭛"#, r#"뭛"#, r#"뭛"#, r#"뭛"#, r#"뭛"#),
        (r#"뭜"#, r#"뭜"#, r#"뭜"#, r#"뭜"#, r#"뭜"#),
        (r#"뭝"#, r#"뭝"#, r#"뭝"#, r#"뭝"#, r#"뭝"#),
        (r#"뭞"#, r#"뭞"#, r#"뭞"#, r#"뭞"#, r#"뭞"#),
        (r#"뭟"#, r#"뭟"#, r#"뭟"#, r#"뭟"#, r#"뭟"#),
        (r#"뭠"#, r#"뭠"#, r#"뭠"#, r#"뭠"#, r#"뭠"#),
        (r#"뭡"#, r#"뭡"#, r#"뭡"#, r#"뭡"#, r#"뭡"#),
        (r#"뭢"#, r#"뭢"#, r#"뭢"#, r#"뭢"#, r#"뭢"#),
        (r#"뭣"#, r#"뭣"#, r#"뭣"#, r#"뭣"#, r#"뭣"#),
        (r#"뭤"#, r#"뭤"#, r#"뭤"#, r#"뭤"#, r#"뭤"#),
        (r#"뭥"#, r#"뭥"#, r#"뭥"#, r#"뭥"#, r#"뭥"#),
        (r#"뭦"#, r#"뭦"#, r#"뭦"#, r#"뭦"#, r#"뭦"#),
        (r#"뭧"#, r#"뭧"#, r#"뭧"#, r#"뭧"#, r#"뭧"#),
        (r#"뭨"#, r#"뭨"#, r#"뭨"#, r#"뭨"#, r#"뭨"#),
        (r#"뭩"#, r#"뭩"#, r#"뭩"#, r#"뭩"#, r#"뭩"#),
        (r#"뭪"#, r#"뭪"#, r#"뭪"#, r#"뭪"#, r#"뭪"#),
        (r#"뭫"#, r#"뭫"#, r#"뭫"#, r#"뭫"#, r#"뭫"#),
        (r#"뭬"#, r#"뭬"#, r#"뭬"#, r#"뭬"#, r#"뭬"#),
        (r#"뭭"#, r#"뭭"#, r#"뭭"#, r#"뭭"#, r#"뭭"#),
        (r#"뭮"#, r#"뭮"#, r#"뭮"#, r#"뭮"#, r#"뭮"#),
        (r#"뭯"#, r#"뭯"#, r#"뭯"#, r#"뭯"#, r#"뭯"#),
        (r#"뭰"#, r#"뭰"#, r#"뭰"#, r#"뭰"#, r#"뭰"#),
        (r#"뭱"#, r#"뭱"#, r#"뭱"#, r#"뭱"#, r#"뭱"#),
        (r#"뭲"#, r#"뭲"#, r#"뭲"#, r#"뭲"#, r#"뭲"#),
        (r#"뭳"#, r#"뭳"#, r#"뭳"#, r#"뭳"#, r#"뭳"#),
        (r#"뭴"#, r#"뭴"#, r#"뭴"#, r#"뭴"#, r#"뭴"#),
        (r#"뭵"#, r#"뭵"#, r#"뭵"#, r#"뭵"#, r#"뭵"#),
        (r#"뭶"#, r#"뭶"#, r#"뭶"#, r#"뭶"#, r#"뭶"#),
        (r#"뭷"#, r#"뭷"#, r#"뭷"#, r#"뭷"#, r#"뭷"#),
        (r#"뭸"#, r#"뭸"#, r#"뭸"#, r#"뭸"#, r#"뭸"#),
        (r#"뭹"#, r#"뭹"#, r#"뭹"#, r#"뭹"#, r#"뭹"#),
        (r#"뭺"#, r#"뭺"#, r#"뭺"#, r#"뭺"#, r#"뭺"#),
        (r#"뭻"#, r#"뭻"#, r#"뭻"#, r#"뭻"#, r#"뭻"#),
        (r#"뭼"#, r#"뭼"#, r#"뭼"#, r#"뭼"#, r#"뭼"#),
        (r#"뭽"#, r#"뭽"#, r#"뭽"#, r#"뭽"#, r#"뭽"#),
        (r#"뭾"#, r#"뭾"#, r#"뭾"#, r#"뭾"#, r#"뭾"#),
        (r#"뭿"#, r#"뭿"#, r#"뭿"#, r#"뭿"#, r#"뭿"#),
        (r#"뮀"#, r#"뮀"#, r#"뮀"#, r#"뮀"#, r#"뮀"#),
        (r#"뮁"#, r#"뮁"#, r#"뮁"#, r#"뮁"#, r#"뮁"#),
        (r#"뮂"#, r#"뮂"#, r#"뮂"#, r#"뮂"#, r#"뮂"#),
        (r#"뮃"#, r#"뮃"#, r#"뮃"#, r#"뮃"#, r#"뮃"#),
        (r#"뮄"#, r#"뮄"#, r#"뮄"#, r#"뮄"#, r#"뮄"#),
        (r#"뮅"#, r#"뮅"#, r#"뮅"#, r#"뮅"#, r#"뮅"#),
        (r#"뮆"#, r#"뮆"#, r#"뮆"#, r#"뮆"#, r#"뮆"#),
        (r#"뮇"#, r#"뮇"#, r#"뮇"#, r#"뮇"#, r#"뮇"#),
        (r#"뮈"#, r#"뮈"#, r#"뮈"#, r#"뮈"#, r#"뮈"#),
        (r#"뮉"#, r#"뮉"#, r#"뮉"#, r#"뮉"#, r#"뮉"#),
        (r#"뮊"#, r#"뮊"#, r#"뮊"#, r#"뮊"#, r#"뮊"#),
        (r#"뮋"#, r#"뮋"#, r#"뮋"#, r#"뮋"#, r#"뮋"#),
        (r#"뮌"#, r#"뮌"#, r#"뮌"#, r#"뮌"#, r#"뮌"#),
        (r#"뮍"#, r#"뮍"#, r#"뮍"#, r#"뮍"#, r#"뮍"#),
        (r#"뮎"#, r#"뮎"#, r#"뮎"#, r#"뮎"#, r#"뮎"#),
        (r#"뮏"#, r#"뮏"#, r#"뮏"#, r#"뮏"#, r#"뮏"#),
        (r#"뮐"#, r#"뮐"#, r#"뮐"#, r#"뮐"#, r#"뮐"#),
        (r#"뮑"#, r#"뮑"#, r#"뮑"#, r#"뮑"#, r#"뮑"#),
        (r#"뮒"#, r#"뮒"#, r#"뮒"#, r#"뮒"#, r#"뮒"#),
        (r#"뮓"#, r#"뮓"#, r#"뮓"#, r#"뮓"#, r#"뮓"#),
        (r#"뮔"#, r#"뮔"#, r#"뮔"#, r#"뮔"#, r#"뮔"#),
        (r#"뮕"#, r#"뮕"#, r#"뮕"#, r#"뮕"#, r#"뮕"#),
        (r#"뮖"#, r#"뮖"#, r#"뮖"#, r#"뮖"#, r#"뮖"#),
        (r#"뮗"#, r#"뮗"#, r#"뮗"#, r#"뮗"#, r#"뮗"#),
        (r#"뮘"#, r#"뮘"#, r#"뮘"#, r#"뮘"#, r#"뮘"#),
        (r#"뮙"#, r#"뮙"#, r#"뮙"#, r#"뮙"#, r#"뮙"#),
        (r#"뮚"#, r#"뮚"#, r#"뮚"#, r#"뮚"#, r#"뮚"#),
        (r#"뮛"#, r#"뮛"#, r#"뮛"#, r#"뮛"#, r#"뮛"#),
        (r#"뮜"#, r#"뮜"#, r#"뮜"#, r#"뮜"#, r#"뮜"#),
        (r#"뮝"#, r#"뮝"#, r#"뮝"#, r#"뮝"#, r#"뮝"#),
        (r#"뮞"#, r#"뮞"#, r#"뮞"#, r#"뮞"#, r#"뮞"#),
        (r#"뮟"#, r#"뮟"#, r#"뮟"#, r#"뮟"#, r#"뮟"#),
        (r#"뮠"#, r#"뮠"#, r#"뮠"#, r#"뮠"#, r#"뮠"#),
        (r#"뮡"#, r#"뮡"#, r#"뮡"#, r#"뮡"#, r#"뮡"#),
        (r#"뮢"#, r#"뮢"#, r#"뮢"#, r#"뮢"#, r#"뮢"#),
        (r#"뮣"#, r#"뮣"#, r#"뮣"#, r#"뮣"#, r#"뮣"#),
        (r#"뮤"#, r#"뮤"#, r#"뮤"#, r#"뮤"#, r#"뮤"#),
        (r#"뮥"#, r#"뮥"#, r#"뮥"#, r#"뮥"#, r#"뮥"#),
        (r#"뮦"#, r#"뮦"#, r#"뮦"#, r#"뮦"#, r#"뮦"#),
        (r#"뮧"#, r#"뮧"#, r#"뮧"#, r#"뮧"#, r#"뮧"#),
        (r#"뮨"#, r#"뮨"#, r#"뮨"#, r#"뮨"#, r#"뮨"#),
        (r#"뮩"#, r#"뮩"#, r#"뮩"#, r#"뮩"#, r#"뮩"#),
        (r#"뮪"#, r#"뮪"#, r#"뮪"#, r#"뮪"#, r#"뮪"#),
        (r#"뮫"#, r#"뮫"#, r#"뮫"#, r#"뮫"#, r#"뮫"#),
        (r#"뮬"#, r#"뮬"#, r#"뮬"#, r#"뮬"#, r#"뮬"#),
        (r#"뮭"#, r#"뮭"#, r#"뮭"#, r#"뮭"#, r#"뮭"#),
        (r#"뮮"#, r#"뮮"#, r#"뮮"#, r#"뮮"#, r#"뮮"#),
        (r#"뮯"#, r#"뮯"#, r#"뮯"#, r#"뮯"#, r#"뮯"#),
        (r#"뮰"#, r#"뮰"#, r#"뮰"#, r#"뮰"#, r#"뮰"#),
        (r#"뮱"#, r#"뮱"#, r#"뮱"#, r#"뮱"#, r#"뮱"#),
        (r#"뮲"#, r#"뮲"#, r#"뮲"#, r#"뮲"#, r#"뮲"#),
        (r#"뮳"#, r#"뮳"#, r#"뮳"#, r#"뮳"#, r#"뮳"#),
        (r#"뮴"#, r#"뮴"#, r#"뮴"#, r#"뮴"#, r#"뮴"#),
        (r#"뮵"#, r#"뮵"#, r#"뮵"#, r#"뮵"#, r#"뮵"#),
        (r#"뮶"#, r#"뮶"#, r#"뮶"#, r#"뮶"#, r#"뮶"#),
        (r#"뮷"#, r#"뮷"#, r#"뮷"#, r#"뮷"#, r#"뮷"#),
        (r#"뮸"#, r#"뮸"#, r#"뮸"#, r#"뮸"#, r#"뮸"#),
        (r#"뮹"#, r#"뮹"#, r#"뮹"#, r#"뮹"#, r#"뮹"#),
        (r#"뮺"#, r#"뮺"#, r#"뮺"#, r#"뮺"#, r#"뮺"#),
        (r#"뮻"#, r#"뮻"#, r#"뮻"#, r#"뮻"#, r#"뮻"#),
        (r#"뮼"#, r#"뮼"#, r#"뮼"#, r#"뮼"#, r#"뮼"#),
        (r#"뮽"#, r#"뮽"#, r#"뮽"#, r#"뮽"#, r#"뮽"#),
        (r#"뮾"#, r#"뮾"#, r#"뮾"#, r#"뮾"#, r#"뮾"#),
        (r#"뮿"#, r#"뮿"#, r#"뮿"#, r#"뮿"#, r#"뮿"#),
        (r#"므"#, r#"므"#, r#"므"#, r#"므"#, r#"므"#),
        (r#"믁"#, r#"믁"#, r#"믁"#, r#"믁"#, r#"믁"#),
        (r#"믂"#, r#"믂"#, r#"믂"#, r#"믂"#, r#"믂"#),
        (r#"믃"#, r#"믃"#, r#"믃"#, r#"믃"#, r#"믃"#),
        (r#"믄"#, r#"믄"#, r#"믄"#, r#"믄"#, r#"믄"#),
        (r#"믅"#, r#"믅"#, r#"믅"#, r#"믅"#, r#"믅"#),
        (r#"믆"#, r#"믆"#, r#"믆"#, r#"믆"#, r#"믆"#),
        (r#"믇"#, r#"믇"#, r#"믇"#, r#"믇"#, r#"믇"#),
        (r#"믈"#, r#"믈"#, r#"믈"#, r#"믈"#, r#"믈"#),
        (r#"믉"#, r#"믉"#, r#"믉"#, r#"믉"#, r#"믉"#),
        (r#"믊"#, r#"믊"#, r#"믊"#, r#"믊"#, r#"믊"#),
        (r#"믋"#, r#"믋"#, r#"믋"#, r#"믋"#, r#"믋"#),
        (r#"믌"#, r#"믌"#, r#"믌"#, r#"믌"#, r#"믌"#),
        (r#"믍"#, r#"믍"#, r#"믍"#, r#"믍"#, r#"믍"#),
        (r#"믎"#, r#"믎"#, r#"믎"#, r#"믎"#, r#"믎"#),
        (r#"믏"#, r#"믏"#, r#"믏"#, r#"믏"#, r#"믏"#),
        (r#"믐"#, r#"믐"#, r#"믐"#, r#"믐"#, r#"믐"#),
        (r#"믑"#, r#"믑"#, r#"믑"#, r#"믑"#, r#"믑"#),
        (r#"믒"#, r#"믒"#, r#"믒"#, r#"믒"#, r#"믒"#),
        (r#"믓"#, r#"믓"#, r#"믓"#, r#"믓"#, r#"믓"#),
        (r#"믔"#, r#"믔"#, r#"믔"#, r#"믔"#, r#"믔"#),
        (r#"믕"#, r#"믕"#, r#"믕"#, r#"믕"#, r#"믕"#),
        (r#"믖"#, r#"믖"#, r#"믖"#, r#"믖"#, r#"믖"#),
        (r#"믗"#, r#"믗"#, r#"믗"#, r#"믗"#, r#"믗"#),
        (r#"믘"#, r#"믘"#, r#"믘"#, r#"믘"#, r#"믘"#),
        (r#"믙"#, r#"믙"#, r#"믙"#, r#"믙"#, r#"믙"#),
        (r#"믚"#, r#"믚"#, r#"믚"#, r#"믚"#, r#"믚"#),
        (r#"믛"#, r#"믛"#, r#"믛"#, r#"믛"#, r#"믛"#),
        (r#"믜"#, r#"믜"#, r#"믜"#, r#"믜"#, r#"믜"#),
        (r#"믝"#, r#"믝"#, r#"믝"#, r#"믝"#, r#"믝"#),
        (r#"믞"#, r#"믞"#, r#"믞"#, r#"믞"#, r#"믞"#),
        (r#"믟"#, r#"믟"#, r#"믟"#, r#"믟"#, r#"믟"#),
        (r#"믠"#, r#"믠"#, r#"믠"#, r#"믠"#, r#"믠"#),
        (r#"믡"#, r#"믡"#, r#"믡"#, r#"믡"#, r#"믡"#),
        (r#"믢"#, r#"믢"#, r#"믢"#, r#"믢"#, r#"믢"#),
        (r#"믣"#, r#"믣"#, r#"믣"#, r#"믣"#, r#"믣"#),
        (r#"믤"#, r#"믤"#, r#"믤"#, r#"믤"#, r#"믤"#),
        (r#"믥"#, r#"믥"#, r#"믥"#, r#"믥"#, r#"믥"#),
        (r#"믦"#, r#"믦"#, r#"믦"#, r#"믦"#, r#"믦"#),
        (r#"믧"#, r#"믧"#, r#"믧"#, r#"믧"#, r#"믧"#),
        (r#"믨"#, r#"믨"#, r#"믨"#, r#"믨"#, r#"믨"#),
        (r#"믩"#, r#"믩"#, r#"믩"#, r#"믩"#, r#"믩"#),
        (r#"믪"#, r#"믪"#, r#"믪"#, r#"믪"#, r#"믪"#),
        (r#"믫"#, r#"믫"#, r#"믫"#, r#"믫"#, r#"믫"#),
        (r#"믬"#, r#"믬"#, r#"믬"#, r#"믬"#, r#"믬"#),
        (r#"믭"#, r#"믭"#, r#"믭"#, r#"믭"#, r#"믭"#),
        (r#"믮"#, r#"믮"#, r#"믮"#, r#"믮"#, r#"믮"#),
        (r#"믯"#, r#"믯"#, r#"믯"#, r#"믯"#, r#"믯"#),
        (r#"믰"#, r#"믰"#, r#"믰"#, r#"믰"#, r#"믰"#),
        (r#"믱"#, r#"믱"#, r#"믱"#, r#"믱"#, r#"믱"#),
        (r#"믲"#, r#"믲"#, r#"믲"#, r#"믲"#, r#"믲"#),
        (r#"믳"#, r#"믳"#, r#"믳"#, r#"믳"#, r#"믳"#),
        (r#"믴"#, r#"믴"#, r#"믴"#, r#"믴"#, r#"믴"#),
        (r#"믵"#, r#"믵"#, r#"믵"#, r#"믵"#, r#"믵"#),
        (r#"믶"#, r#"믶"#, r#"믶"#, r#"믶"#, r#"믶"#),
        (r#"믷"#, r#"믷"#, r#"믷"#, r#"믷"#, r#"믷"#),
        (r#"미"#, r#"미"#, r#"미"#, r#"미"#, r#"미"#),
        (r#"믹"#, r#"믹"#, r#"믹"#, r#"믹"#, r#"믹"#),
        (r#"믺"#, r#"믺"#, r#"믺"#, r#"믺"#, r#"믺"#),
        (r#"믻"#, r#"믻"#, r#"믻"#, r#"믻"#, r#"믻"#),
        (r#"민"#, r#"민"#, r#"민"#, r#"민"#, r#"민"#),
        (r#"믽"#, r#"믽"#, r#"믽"#, r#"믽"#, r#"믽"#),
        (r#"믾"#, r#"믾"#, r#"믾"#, r#"믾"#, r#"믾"#),
        (r#"믿"#, r#"믿"#, r#"믿"#, r#"믿"#, r#"믿"#),
        (r#"밀"#, r#"밀"#, r#"밀"#, r#"밀"#, r#"밀"#),
        (r#"밁"#, r#"밁"#, r#"밁"#, r#"밁"#, r#"밁"#),
        (r#"밂"#, r#"밂"#, r#"밂"#, r#"밂"#, r#"밂"#),
        (r#"밃"#, r#"밃"#, r#"밃"#, r#"밃"#, r#"밃"#),
        (r#"밄"#, r#"밄"#, r#"밄"#, r#"밄"#, r#"밄"#),
        (r#"밅"#, r#"밅"#, r#"밅"#, r#"밅"#, r#"밅"#),
        (r#"밆"#, r#"밆"#, r#"밆"#, r#"밆"#, r#"밆"#),
        (r#"밇"#, r#"밇"#, r#"밇"#, r#"밇"#, r#"밇"#),
        (r#"밈"#, r#"밈"#, r#"밈"#, r#"밈"#, r#"밈"#),
        (r#"밉"#, r#"밉"#, r#"밉"#, r#"밉"#, r#"밉"#),
        (r#"밊"#, r#"밊"#, r#"밊"#, r#"밊"#, r#"밊"#),
        (r#"밋"#, r#"밋"#, r#"밋"#, r#"밋"#, r#"밋"#),
        (r#"밌"#, r#"밌"#, r#"밌"#, r#"밌"#, r#"밌"#),
        (r#"밍"#, r#"밍"#, r#"밍"#, r#"밍"#, r#"밍"#),
        (r#"밎"#, r#"밎"#, r#"밎"#, r#"밎"#, r#"밎"#),
        (r#"및"#, r#"및"#, r#"및"#, r#"및"#, r#"및"#),
        (r#"밐"#, r#"밐"#, r#"밐"#, r#"밐"#, r#"밐"#),
        (r#"밑"#, r#"밑"#, r#"밑"#, r#"밑"#, r#"밑"#),
        (r#"밒"#, r#"밒"#, r#"밒"#, r#"밒"#, r#"밒"#),
        (r#"밓"#, r#"밓"#, r#"밓"#, r#"밓"#, r#"밓"#),
        (r#"바"#, r#"바"#, r#"바"#, r#"바"#, r#"바"#),
        (r#"박"#, r#"박"#, r#"박"#, r#"박"#, r#"박"#),
        (r#"밖"#, r#"밖"#, r#"밖"#, r#"밖"#, r#"밖"#),
        (r#"밗"#, r#"밗"#, r#"밗"#, r#"밗"#, r#"밗"#),
        (r#"반"#, r#"반"#, r#"반"#, r#"반"#, r#"반"#),
        (r#"밙"#, r#"밙"#, r#"밙"#, r#"밙"#, r#"밙"#),
        (r#"밚"#, r#"밚"#, r#"밚"#, r#"밚"#, r#"밚"#),
        (r#"받"#, r#"받"#, r#"받"#, r#"받"#, r#"받"#),
        (r#"발"#, r#"발"#, r#"발"#, r#"발"#, r#"발"#),
        (r#"밝"#, r#"밝"#, r#"밝"#, r#"밝"#, r#"밝"#),
        (r#"밞"#, r#"밞"#, r#"밞"#, r#"밞"#, r#"밞"#),
        (r#"밟"#, r#"밟"#, r#"밟"#, r#"밟"#, r#"밟"#),
        (r#"밠"#, r#"밠"#, r#"밠"#, r#"밠"#, r#"밠"#),
        (r#"밡"#, r#"밡"#, r#"밡"#, r#"밡"#, r#"밡"#),
        (r#"밢"#, r#"밢"#, r#"밢"#, r#"밢"#, r#"밢"#),
        (r#"밣"#, r#"밣"#, r#"밣"#, r#"밣"#, r#"밣"#),
        (r#"밤"#, r#"밤"#, r#"밤"#, r#"밤"#, r#"밤"#),
        (r#"밥"#, r#"밥"#, r#"밥"#, r#"밥"#, r#"밥"#),
        (r#"밦"#, r#"밦"#, r#"밦"#, r#"밦"#, r#"밦"#),
        (r#"밧"#, r#"밧"#, r#"밧"#, r#"밧"#, r#"밧"#),
        (r#"밨"#, r#"밨"#, r#"밨"#, r#"밨"#, r#"밨"#),
        (r#"방"#, r#"방"#, r#"방"#, r#"방"#, r#"방"#),
        (r#"밪"#, r#"밪"#, r#"밪"#, r#"밪"#, r#"밪"#),
        (r#"밫"#, r#"밫"#, r#"밫"#, r#"밫"#, r#"밫"#),
        (r#"밬"#, r#"밬"#, r#"밬"#, r#"밬"#, r#"밬"#),
        (r#"밭"#, r#"밭"#, r#"밭"#, r#"밭"#, r#"밭"#),
        (r#"밮"#, r#"밮"#, r#"밮"#, r#"밮"#, r#"밮"#),
        (r#"밯"#, r#"밯"#, r#"밯"#, r#"밯"#, r#"밯"#),
        (r#"배"#, r#"배"#, r#"배"#, r#"배"#, r#"배"#),
        (r#"백"#, r#"백"#, r#"백"#, r#"백"#, r#"백"#),
        (r#"밲"#, r#"밲"#, r#"밲"#, r#"밲"#, r#"밲"#),
        (r#"밳"#, r#"밳"#, r#"밳"#, r#"밳"#, r#"밳"#),
        (r#"밴"#, r#"밴"#, r#"밴"#, r#"밴"#, r#"밴"#),
        (r#"밵"#, r#"밵"#, r#"밵"#, r#"밵"#, r#"밵"#),
        (r#"밶"#, r#"밶"#, r#"밶"#, r#"밶"#, r#"밶"#),
        (r#"밷"#, r#"밷"#, r#"밷"#, r#"밷"#, r#"밷"#),
        (r#"밸"#, r#"밸"#, r#"밸"#, r#"밸"#, r#"밸"#),
        (r#"밹"#, r#"밹"#, r#"밹"#, r#"밹"#, r#"밹"#),
        (r#"밺"#, r#"밺"#, r#"밺"#, r#"밺"#, r#"밺"#),
        (r#"밻"#, r#"밻"#, r#"밻"#, r#"밻"#, r#"밻"#),
        (r#"밼"#, r#"밼"#, r#"밼"#, r#"밼"#, r#"밼"#),
        (r#"밽"#, r#"밽"#, r#"밽"#, r#"밽"#, r#"밽"#),
        (r#"밾"#, r#"밾"#, r#"밾"#, r#"밾"#, r#"밾"#),
        (r#"밿"#, r#"밿"#, r#"밿"#, r#"밿"#, r#"밿"#),
        (r#"뱀"#, r#"뱀"#, r#"뱀"#, r#"뱀"#, r#"뱀"#),
        (r#"뱁"#, r#"뱁"#, r#"뱁"#, r#"뱁"#, r#"뱁"#),
        (r#"뱂"#, r#"뱂"#, r#"뱂"#, r#"뱂"#, r#"뱂"#),
        (r#"뱃"#, r#"뱃"#, r#"뱃"#, r#"뱃"#, r#"뱃"#),
        (r#"뱄"#, r#"뱄"#, r#"뱄"#, r#"뱄"#, r#"뱄"#),
        (r#"뱅"#, r#"뱅"#, r#"뱅"#, r#"뱅"#, r#"뱅"#),
        (r#"뱆"#, r#"뱆"#, r#"뱆"#, r#"뱆"#, r#"뱆"#),
        (r#"뱇"#, r#"뱇"#, r#"뱇"#, r#"뱇"#, r#"뱇"#),
        (r#"뱈"#, r#"뱈"#, r#"뱈"#, r#"뱈"#, r#"뱈"#),
        (r#"뱉"#, r#"뱉"#, r#"뱉"#, r#"뱉"#, r#"뱉"#),
        (r#"뱊"#, r#"뱊"#, r#"뱊"#, r#"뱊"#, r#"뱊"#),
        (r#"뱋"#, r#"뱋"#, r#"뱋"#, r#"뱋"#, r#"뱋"#),
        (r#"뱌"#, r#"뱌"#, r#"뱌"#, r#"뱌"#, r#"뱌"#),
        (r#"뱍"#, r#"뱍"#, r#"뱍"#, r#"뱍"#, r#"뱍"#),
        (r#"뱎"#, r#"뱎"#, r#"뱎"#, r#"뱎"#, r#"뱎"#),
        (r#"뱏"#, r#"뱏"#, r#"뱏"#, r#"뱏"#, r#"뱏"#),
        (r#"뱐"#, r#"뱐"#, r#"뱐"#, r#"뱐"#, r#"뱐"#),
        (r#"뱑"#, r#"뱑"#, r#"뱑"#, r#"뱑"#, r#"뱑"#),
        (r#"뱒"#, r#"뱒"#, r#"뱒"#, r#"뱒"#, r#"뱒"#),
        (r#"뱓"#, r#"뱓"#, r#"뱓"#, r#"뱓"#, r#"뱓"#),
        (r#"뱔"#, r#"뱔"#, r#"뱔"#, r#"뱔"#, r#"뱔"#),
        (r#"뱕"#, r#"뱕"#, r#"뱕"#, r#"뱕"#, r#"뱕"#),
        (r#"뱖"#, r#"뱖"#, r#"뱖"#, r#"뱖"#, r#"뱖"#),
        (r#"뱗"#, r#"뱗"#, r#"뱗"#, r#"뱗"#, r#"뱗"#),
        (r#"뱘"#, r#"뱘"#, r#"뱘"#, r#"뱘"#, r#"뱘"#),
        (r#"뱙"#, r#"뱙"#, r#"뱙"#, r#"뱙"#, r#"뱙"#),
        (r#"뱚"#, r#"뱚"#, r#"뱚"#, r#"뱚"#, r#"뱚"#),
        (r#"뱛"#, r#"뱛"#, r#"뱛"#, r#"뱛"#, r#"뱛"#),
        (r#"뱜"#, r#"뱜"#, r#"뱜"#, r#"뱜"#, r#"뱜"#),
        (r#"뱝"#, r#"뱝"#, r#"뱝"#, r#"뱝"#, r#"뱝"#),
        (r#"뱞"#, r#"뱞"#, r#"뱞"#, r#"뱞"#, r#"뱞"#),
        (r#"뱟"#, r#"뱟"#, r#"뱟"#, r#"뱟"#, r#"뱟"#),
        (r#"뱠"#, r#"뱠"#, r#"뱠"#, r#"뱠"#, r#"뱠"#),
        (r#"뱡"#, r#"뱡"#, r#"뱡"#, r#"뱡"#, r#"뱡"#),
        (r#"뱢"#, r#"뱢"#, r#"뱢"#, r#"뱢"#, r#"뱢"#),
        (r#"뱣"#, r#"뱣"#, r#"뱣"#, r#"뱣"#, r#"뱣"#),
        (r#"뱤"#, r#"뱤"#, r#"뱤"#, r#"뱤"#, r#"뱤"#),
        (r#"뱥"#, r#"뱥"#, r#"뱥"#, r#"뱥"#, r#"뱥"#),
        (r#"뱦"#, r#"뱦"#, r#"뱦"#, r#"뱦"#, r#"뱦"#),
        (r#"뱧"#, r#"뱧"#, r#"뱧"#, r#"뱧"#, r#"뱧"#),
        (r#"뱨"#, r#"뱨"#, r#"뱨"#, r#"뱨"#, r#"뱨"#),
        (r#"뱩"#, r#"뱩"#, r#"뱩"#, r#"뱩"#, r#"뱩"#),
        (r#"뱪"#, r#"뱪"#, r#"뱪"#, r#"뱪"#, r#"뱪"#),
        (r#"뱫"#, r#"뱫"#, r#"뱫"#, r#"뱫"#, r#"뱫"#),
        (r#"뱬"#, r#"뱬"#, r#"뱬"#, r#"뱬"#, r#"뱬"#),
        (r#"뱭"#, r#"뱭"#, r#"뱭"#, r#"뱭"#, r#"뱭"#),
        (r#"뱮"#, r#"뱮"#, r#"뱮"#, r#"뱮"#, r#"뱮"#),
        (r#"뱯"#, r#"뱯"#, r#"뱯"#, r#"뱯"#, r#"뱯"#),
        (r#"뱰"#, r#"뱰"#, r#"뱰"#, r#"뱰"#, r#"뱰"#),
        (r#"뱱"#, r#"뱱"#, r#"뱱"#, r#"뱱"#, r#"뱱"#),
        (r#"뱲"#, r#"뱲"#, r#"뱲"#, r#"뱲"#, r#"뱲"#),
        (r#"뱳"#, r#"뱳"#, r#"뱳"#, r#"뱳"#, r#"뱳"#),
        (r#"뱴"#, r#"뱴"#, r#"뱴"#, r#"뱴"#, r#"뱴"#),
        (r#"뱵"#, r#"뱵"#, r#"뱵"#, r#"뱵"#, r#"뱵"#),
        (r#"뱶"#, r#"뱶"#, r#"뱶"#, r#"뱶"#, r#"뱶"#),
        (r#"뱷"#, r#"뱷"#, r#"뱷"#, r#"뱷"#, r#"뱷"#),
        (r#"뱸"#, r#"뱸"#, r#"뱸"#, r#"뱸"#, r#"뱸"#),
        (r#"뱹"#, r#"뱹"#, r#"뱹"#, r#"뱹"#, r#"뱹"#),
        (r#"뱺"#, r#"뱺"#, r#"뱺"#, r#"뱺"#, r#"뱺"#),
        (r#"뱻"#, r#"뱻"#, r#"뱻"#, r#"뱻"#, r#"뱻"#),
        (r#"뱼"#, r#"뱼"#, r#"뱼"#, r#"뱼"#, r#"뱼"#),
        (r#"뱽"#, r#"뱽"#, r#"뱽"#, r#"뱽"#, r#"뱽"#),
        (r#"뱾"#, r#"뱾"#, r#"뱾"#, r#"뱾"#, r#"뱾"#),
        (r#"뱿"#, r#"뱿"#, r#"뱿"#, r#"뱿"#, r#"뱿"#),
        (r#"벀"#, r#"벀"#, r#"벀"#, r#"벀"#, r#"벀"#),
        (r#"벁"#, r#"벁"#, r#"벁"#, r#"벁"#, r#"벁"#),
        (r#"벂"#, r#"벂"#, r#"벂"#, r#"벂"#, r#"벂"#),
        (r#"벃"#, r#"벃"#, r#"벃"#, r#"벃"#, r#"벃"#),
        (r#"버"#, r#"버"#, r#"버"#, r#"버"#, r#"버"#),
        (r#"벅"#, r#"벅"#, r#"벅"#, r#"벅"#, r#"벅"#),
        (r#"벆"#, r#"벆"#, r#"벆"#, r#"벆"#, r#"벆"#),
        (r#"벇"#, r#"벇"#, r#"벇"#, r#"벇"#, r#"벇"#),
        (r#"번"#, r#"번"#, r#"번"#, r#"번"#, r#"번"#),
        (r#"벉"#, r#"벉"#, r#"벉"#, r#"벉"#, r#"벉"#),
        (r#"벊"#, r#"벊"#, r#"벊"#, r#"벊"#, r#"벊"#),
        (r#"벋"#, r#"벋"#, r#"벋"#, r#"벋"#, r#"벋"#),
        (r#"벌"#, r#"벌"#, r#"벌"#, r#"벌"#, r#"벌"#),
        (r#"벍"#, r#"벍"#, r#"벍"#, r#"벍"#, r#"벍"#),
        (r#"벎"#, r#"벎"#, r#"벎"#, r#"벎"#, r#"벎"#),
        (r#"벏"#, r#"벏"#, r#"벏"#, r#"벏"#, r#"벏"#),
        (r#"벐"#, r#"벐"#, r#"벐"#, r#"벐"#, r#"벐"#),
        (r#"벑"#, r#"벑"#, r#"벑"#, r#"벑"#, r#"벑"#),
        (r#"벒"#, r#"벒"#, r#"벒"#, r#"벒"#, r#"벒"#),
        (r#"벓"#, r#"벓"#, r#"벓"#, r#"벓"#, r#"벓"#),
        (r#"범"#, r#"범"#, r#"범"#, r#"범"#, r#"범"#),
        (r#"법"#, r#"법"#, r#"법"#, r#"법"#, r#"법"#),
        (r#"벖"#, r#"벖"#, r#"벖"#, r#"벖"#, r#"벖"#),
        (r#"벗"#, r#"벗"#, r#"벗"#, r#"벗"#, r#"벗"#),
        (r#"벘"#, r#"벘"#, r#"벘"#, r#"벘"#, r#"벘"#),
        (r#"벙"#, r#"벙"#, r#"벙"#, r#"벙"#, r#"벙"#),
        (r#"벚"#, r#"벚"#, r#"벚"#, r#"벚"#, r#"벚"#),
        (r#"벛"#, r#"벛"#, r#"벛"#, r#"벛"#, r#"벛"#),
        (r#"벜"#, r#"벜"#, r#"벜"#, r#"벜"#, r#"벜"#),
        (r#"벝"#, r#"벝"#, r#"벝"#, r#"벝"#, r#"벝"#),
        (r#"벞"#, r#"벞"#, r#"벞"#, r#"벞"#, r#"벞"#),
        (r#"벟"#, r#"벟"#, r#"벟"#, r#"벟"#, r#"벟"#),
        (r#"베"#, r#"베"#, r#"베"#, r#"베"#, r#"베"#),
        (r#"벡"#, r#"벡"#, r#"벡"#, r#"벡"#, r#"벡"#),
        (r#"벢"#, r#"벢"#, r#"벢"#, r#"벢"#, r#"벢"#),
        (r#"벣"#, r#"벣"#, r#"벣"#, r#"벣"#, r#"벣"#),
        (r#"벤"#, r#"벤"#, r#"벤"#, r#"벤"#, r#"벤"#),
        (r#"벥"#, r#"벥"#, r#"벥"#, r#"벥"#, r#"벥"#),
        (r#"벦"#, r#"벦"#, r#"벦"#, r#"벦"#, r#"벦"#),
        (r#"벧"#, r#"벧"#, r#"벧"#, r#"벧"#, r#"벧"#),
        (r#"벨"#, r#"벨"#, r#"벨"#, r#"벨"#, r#"벨"#),
        (r#"벩"#, r#"벩"#, r#"벩"#, r#"벩"#, r#"벩"#),
        (r#"벪"#, r#"벪"#, r#"벪"#, r#"벪"#, r#"벪"#),
        (r#"벫"#, r#"벫"#, r#"벫"#, r#"벫"#, r#"벫"#),
        (r#"벬"#, r#"벬"#, r#"벬"#, r#"벬"#, r#"벬"#),
        (r#"벭"#, r#"벭"#, r#"벭"#, r#"벭"#, r#"벭"#),
        (r#"벮"#, r#"벮"#, r#"벮"#, r#"벮"#, r#"벮"#),
        (r#"벯"#, r#"벯"#, r#"벯"#, r#"벯"#, r#"벯"#),
        (r#"벰"#, r#"벰"#, r#"벰"#, r#"벰"#, r#"벰"#),
        (r#"벱"#, r#"벱"#, r#"벱"#, r#"벱"#, r#"벱"#),
        (r#"벲"#, r#"벲"#, r#"벲"#, r#"벲"#, r#"벲"#),
        (r#"벳"#, r#"벳"#, r#"벳"#, r#"벳"#, r#"벳"#),
        (r#"벴"#, r#"벴"#, r#"벴"#, r#"벴"#, r#"벴"#),
        (r#"벵"#, r#"벵"#, r#"벵"#, r#"벵"#, r#"벵"#),
        (r#"벶"#, r#"벶"#, r#"벶"#, r#"벶"#, r#"벶"#),
        (r#"벷"#, r#"벷"#, r#"벷"#, r#"벷"#, r#"벷"#),
        (r#"벸"#, r#"벸"#, r#"벸"#, r#"벸"#, r#"벸"#),
        (r#"벹"#, r#"벹"#, r#"벹"#, r#"벹"#, r#"벹"#),
        (r#"벺"#, r#"벺"#, r#"벺"#, r#"벺"#, r#"벺"#),
        (r#"벻"#, r#"벻"#, r#"벻"#, r#"벻"#, r#"벻"#),
        (r#"벼"#, r#"벼"#, r#"벼"#, r#"벼"#, r#"벼"#),
        (r#"벽"#, r#"벽"#, r#"벽"#, r#"벽"#, r#"벽"#),
        (r#"벾"#, r#"벾"#, r#"벾"#, r#"벾"#, r#"벾"#),
        (r#"벿"#, r#"벿"#, r#"벿"#, r#"벿"#, r#"벿"#),
        (r#"변"#, r#"변"#, r#"변"#, r#"변"#, r#"변"#),
        (r#"볁"#, r#"볁"#, r#"볁"#, r#"볁"#, r#"볁"#),
        (r#"볂"#, r#"볂"#, r#"볂"#, r#"볂"#, r#"볂"#),
        (r#"볃"#, r#"볃"#, r#"볃"#, r#"볃"#, r#"볃"#),
        (r#"별"#, r#"별"#, r#"별"#, r#"별"#, r#"별"#),
        (r#"볅"#, r#"볅"#, r#"볅"#, r#"볅"#, r#"볅"#),
        (r#"볆"#, r#"볆"#, r#"볆"#, r#"볆"#, r#"볆"#),
        (r#"볇"#, r#"볇"#, r#"볇"#, r#"볇"#, r#"볇"#),
        (r#"볈"#, r#"볈"#, r#"볈"#, r#"볈"#, r#"볈"#),
        (r#"볉"#, r#"볉"#, r#"볉"#, r#"볉"#, r#"볉"#),
        (r#"볊"#, r#"볊"#, r#"볊"#, r#"볊"#, r#"볊"#),
        (r#"볋"#, r#"볋"#, r#"볋"#, r#"볋"#, r#"볋"#),
        (r#"볌"#, r#"볌"#, r#"볌"#, r#"볌"#, r#"볌"#),
        (r#"볍"#, r#"볍"#, r#"볍"#, r#"볍"#, r#"볍"#),
        (r#"볎"#, r#"볎"#, r#"볎"#, r#"볎"#, r#"볎"#),
        (r#"볏"#, r#"볏"#, r#"볏"#, r#"볏"#, r#"볏"#),
        (r#"볐"#, r#"볐"#, r#"볐"#, r#"볐"#, r#"볐"#),
        (r#"병"#, r#"병"#, r#"병"#, r#"병"#, r#"병"#),
        (r#"볒"#, r#"볒"#, r#"볒"#, r#"볒"#, r#"볒"#),
        (r#"볓"#, r#"볓"#, r#"볓"#, r#"볓"#, r#"볓"#),
        (r#"볔"#, r#"볔"#, r#"볔"#, r#"볔"#, r#"볔"#),
        (r#"볕"#, r#"볕"#, r#"볕"#, r#"볕"#, r#"볕"#),
        (r#"볖"#, r#"볖"#, r#"볖"#, r#"볖"#, r#"볖"#),
        (r#"볗"#, r#"볗"#, r#"볗"#, r#"볗"#, r#"볗"#),
        (r#"볘"#, r#"볘"#, r#"볘"#, r#"볘"#, r#"볘"#),
        (r#"볙"#, r#"볙"#, r#"볙"#, r#"볙"#, r#"볙"#),
        (r#"볚"#, r#"볚"#, r#"볚"#, r#"볚"#, r#"볚"#),
        (r#"볛"#, r#"볛"#, r#"볛"#, r#"볛"#, r#"볛"#),
        (r#"볜"#, r#"볜"#, r#"볜"#, r#"볜"#, r#"볜"#),
        (r#"볝"#, r#"볝"#, r#"볝"#, r#"볝"#, r#"볝"#),
        (r#"볞"#, r#"볞"#, r#"볞"#, r#"볞"#, r#"볞"#),
        (r#"볟"#, r#"볟"#, r#"볟"#, r#"볟"#, r#"볟"#),
        (r#"볠"#, r#"볠"#, r#"볠"#, r#"볠"#, r#"볠"#),
        (r#"볡"#, r#"볡"#, r#"볡"#, r#"볡"#, r#"볡"#),
        (r#"볢"#, r#"볢"#, r#"볢"#, r#"볢"#, r#"볢"#),
        (r#"볣"#, r#"볣"#, r#"볣"#, r#"볣"#, r#"볣"#),
        (r#"볤"#, r#"볤"#, r#"볤"#, r#"볤"#, r#"볤"#),
        (r#"볥"#, r#"볥"#, r#"볥"#, r#"볥"#, r#"볥"#),
        (r#"볦"#, r#"볦"#, r#"볦"#, r#"볦"#, r#"볦"#),
        (r#"볧"#, r#"볧"#, r#"볧"#, r#"볧"#, r#"볧"#),
        (r#"볨"#, r#"볨"#, r#"볨"#, r#"볨"#, r#"볨"#),
        (r#"볩"#, r#"볩"#, r#"볩"#, r#"볩"#, r#"볩"#),
        (r#"볪"#, r#"볪"#, r#"볪"#, r#"볪"#, r#"볪"#),
        (r#"볫"#, r#"볫"#, r#"볫"#, r#"볫"#, r#"볫"#),
        (r#"볬"#, r#"볬"#, r#"볬"#, r#"볬"#, r#"볬"#),
        (r#"볭"#, r#"볭"#, r#"볭"#, r#"볭"#, r#"볭"#),
        (r#"볮"#, r#"볮"#, r#"볮"#, r#"볮"#, r#"볮"#),
        (r#"볯"#, r#"볯"#, r#"볯"#, r#"볯"#, r#"볯"#),
        (r#"볰"#, r#"볰"#, r#"볰"#, r#"볰"#, r#"볰"#),
        (r#"볱"#, r#"볱"#, r#"볱"#, r#"볱"#, r#"볱"#),
        (r#"볲"#, r#"볲"#, r#"볲"#, r#"볲"#, r#"볲"#),
        (r#"볳"#, r#"볳"#, r#"볳"#, r#"볳"#, r#"볳"#),
        (r#"보"#, r#"보"#, r#"보"#, r#"보"#, r#"보"#),
        (r#"복"#, r#"복"#, r#"복"#, r#"복"#, r#"복"#),
        (r#"볶"#, r#"볶"#, r#"볶"#, r#"볶"#, r#"볶"#),
        (r#"볷"#, r#"볷"#, r#"볷"#, r#"볷"#, r#"볷"#),
        (r#"본"#, r#"본"#, r#"본"#, r#"본"#, r#"본"#),
        (r#"볹"#, r#"볹"#, r#"볹"#, r#"볹"#, r#"볹"#),
        (r#"볺"#, r#"볺"#, r#"볺"#, r#"볺"#, r#"볺"#),
        (r#"볻"#, r#"볻"#, r#"볻"#, r#"볻"#, r#"볻"#),
        (r#"볼"#, r#"볼"#, r#"볼"#, r#"볼"#, r#"볼"#),
        (r#"볽"#, r#"볽"#, r#"볽"#, r#"볽"#, r#"볽"#),
        (r#"볾"#, r#"볾"#, r#"볾"#, r#"볾"#, r#"볾"#),
        (r#"볿"#, r#"볿"#, r#"볿"#, r#"볿"#, r#"볿"#),
        (r#"봀"#, r#"봀"#, r#"봀"#, r#"봀"#, r#"봀"#),
        (r#"봁"#, r#"봁"#, r#"봁"#, r#"봁"#, r#"봁"#),
        (r#"봂"#, r#"봂"#, r#"봂"#, r#"봂"#, r#"봂"#),
        (r#"봃"#, r#"봃"#, r#"봃"#, r#"봃"#, r#"봃"#),
        (r#"봄"#, r#"봄"#, r#"봄"#, r#"봄"#, r#"봄"#),
        (r#"봅"#, r#"봅"#, r#"봅"#, r#"봅"#, r#"봅"#),
        (r#"봆"#, r#"봆"#, r#"봆"#, r#"봆"#, r#"봆"#),
        (r#"봇"#, r#"봇"#, r#"봇"#, r#"봇"#, r#"봇"#),
        (r#"봈"#, r#"봈"#, r#"봈"#, r#"봈"#, r#"봈"#),
        (r#"봉"#, r#"봉"#, r#"봉"#, r#"봉"#, r#"봉"#),
        (r#"봊"#, r#"봊"#, r#"봊"#, r#"봊"#, r#"봊"#),
        (r#"봋"#, r#"봋"#, r#"봋"#, r#"봋"#, r#"봋"#),
        (r#"봌"#, r#"봌"#, r#"봌"#, r#"봌"#, r#"봌"#),
        (r#"봍"#, r#"봍"#, r#"봍"#, r#"봍"#, r#"봍"#),
        (r#"봎"#, r#"봎"#, r#"봎"#, r#"봎"#, r#"봎"#),
        (r#"봏"#, r#"봏"#, r#"봏"#, r#"봏"#, r#"봏"#),
        (r#"봐"#, r#"봐"#, r#"봐"#, r#"봐"#, r#"봐"#),
        (r#"봑"#, r#"봑"#, r#"봑"#, r#"봑"#, r#"봑"#),
        (r#"봒"#, r#"봒"#, r#"봒"#, r#"봒"#, r#"봒"#),
        (r#"봓"#, r#"봓"#, r#"봓"#, r#"봓"#, r#"봓"#),
        (r#"봔"#, r#"봔"#, r#"봔"#, r#"봔"#, r#"봔"#),
        (r#"봕"#, r#"봕"#, r#"봕"#, r#"봕"#, r#"봕"#),
        (r#"봖"#, r#"봖"#, r#"봖"#, r#"봖"#, r#"봖"#),
        (r#"봗"#, r#"봗"#, r#"봗"#, r#"봗"#, r#"봗"#),
        (r#"봘"#, r#"봘"#, r#"봘"#, r#"봘"#, r#"봘"#),
        (r#"봙"#, r#"봙"#, r#"봙"#, r#"봙"#, r#"봙"#),
        (r#"봚"#, r#"봚"#, r#"봚"#, r#"봚"#, r#"봚"#),
        (r#"봛"#, r#"봛"#, r#"봛"#, r#"봛"#, r#"봛"#),
        (r#"봜"#, r#"봜"#, r#"봜"#, r#"봜"#, r#"봜"#),
        (r#"봝"#, r#"봝"#, r#"봝"#, r#"봝"#, r#"봝"#),
        (r#"봞"#, r#"봞"#, r#"봞"#, r#"봞"#, r#"봞"#),
        (r#"봟"#, r#"봟"#, r#"봟"#, r#"봟"#, r#"봟"#),
        (r#"봠"#, r#"봠"#, r#"봠"#, r#"봠"#, r#"봠"#),
        (r#"봡"#, r#"봡"#, r#"봡"#, r#"봡"#, r#"봡"#),
        (r#"봢"#, r#"봢"#, r#"봢"#, r#"봢"#, r#"봢"#),
        (r#"봣"#, r#"봣"#, r#"봣"#, r#"봣"#, r#"봣"#),
        (r#"봤"#, r#"봤"#, r#"봤"#, r#"봤"#, r#"봤"#),
        (r#"봥"#, r#"봥"#, r#"봥"#, r#"봥"#, r#"봥"#),
        (r#"봦"#, r#"봦"#, r#"봦"#, r#"봦"#, r#"봦"#),
        (r#"봧"#, r#"봧"#, r#"봧"#, r#"봧"#, r#"봧"#),
        (r#"봨"#, r#"봨"#, r#"봨"#, r#"봨"#, r#"봨"#),
        (r#"봩"#, r#"봩"#, r#"봩"#, r#"봩"#, r#"봩"#),
        (r#"봪"#, r#"봪"#, r#"봪"#, r#"봪"#, r#"봪"#),
        (r#"봫"#, r#"봫"#, r#"봫"#, r#"봫"#, r#"봫"#),
        (r#"봬"#, r#"봬"#, r#"봬"#, r#"봬"#, r#"봬"#),
        (r#"봭"#, r#"봭"#, r#"봭"#, r#"봭"#, r#"봭"#),
        (r#"봮"#, r#"봮"#, r#"봮"#, r#"봮"#, r#"봮"#),
        (r#"봯"#, r#"봯"#, r#"봯"#, r#"봯"#, r#"봯"#),
        (r#"봰"#, r#"봰"#, r#"봰"#, r#"봰"#, r#"봰"#),
        (r#"봱"#, r#"봱"#, r#"봱"#, r#"봱"#, r#"봱"#),
        (r#"봲"#, r#"봲"#, r#"봲"#, r#"봲"#, r#"봲"#),
        (r#"봳"#, r#"봳"#, r#"봳"#, r#"봳"#, r#"봳"#),
        (r#"봴"#, r#"봴"#, r#"봴"#, r#"봴"#, r#"봴"#),
        (r#"봵"#, r#"봵"#, r#"봵"#, r#"봵"#, r#"봵"#),
        (r#"봶"#, r#"봶"#, r#"봶"#, r#"봶"#, r#"봶"#),
        (r#"봷"#, r#"봷"#, r#"봷"#, r#"봷"#, r#"봷"#),
        (r#"봸"#, r#"봸"#, r#"봸"#, r#"봸"#, r#"봸"#),
        (r#"봹"#, r#"봹"#, r#"봹"#, r#"봹"#, r#"봹"#),
        (r#"봺"#, r#"봺"#, r#"봺"#, r#"봺"#, r#"봺"#),
        (r#"봻"#, r#"봻"#, r#"봻"#, r#"봻"#, r#"봻"#),
        (r#"봼"#, r#"봼"#, r#"봼"#, r#"봼"#, r#"봼"#),
        (r#"봽"#, r#"봽"#, r#"봽"#, r#"봽"#, r#"봽"#),
        (r#"봾"#, r#"봾"#, r#"봾"#, r#"봾"#, r#"봾"#),
        (r#"봿"#, r#"봿"#, r#"봿"#, r#"봿"#, r#"봿"#),
        (r#"뵀"#, r#"뵀"#, r#"뵀"#, r#"뵀"#, r#"뵀"#),
        (r#"뵁"#, r#"뵁"#, r#"뵁"#, r#"뵁"#, r#"뵁"#),
        (r#"뵂"#, r#"뵂"#, r#"뵂"#, r#"뵂"#, r#"뵂"#),
        (r#"뵃"#, r#"뵃"#, r#"뵃"#, r#"뵃"#, r#"뵃"#),
        (r#"뵄"#, r#"뵄"#, r#"뵄"#, r#"뵄"#, r#"뵄"#),
        (r#"뵅"#, r#"뵅"#, r#"뵅"#, r#"뵅"#, r#"뵅"#),
        (r#"뵆"#, r#"뵆"#, r#"뵆"#, r#"뵆"#, r#"뵆"#),
        (r#"뵇"#, r#"뵇"#, r#"뵇"#, r#"뵇"#, r#"뵇"#),
        (r#"뵈"#, r#"뵈"#, r#"뵈"#, r#"뵈"#, r#"뵈"#),
        (r#"뵉"#, r#"뵉"#, r#"뵉"#, r#"뵉"#, r#"뵉"#),
        (r#"뵊"#, r#"뵊"#, r#"뵊"#, r#"뵊"#, r#"뵊"#),
        (r#"뵋"#, r#"뵋"#, r#"뵋"#, r#"뵋"#, r#"뵋"#),
        (r#"뵌"#, r#"뵌"#, r#"뵌"#, r#"뵌"#, r#"뵌"#),
        (r#"뵍"#, r#"뵍"#, r#"뵍"#, r#"뵍"#, r#"뵍"#),
        (r#"뵎"#, r#"뵎"#, r#"뵎"#, r#"뵎"#, r#"뵎"#),
        (r#"뵏"#, r#"뵏"#, r#"뵏"#, r#"뵏"#, r#"뵏"#),
        (r#"뵐"#, r#"뵐"#, r#"뵐"#, r#"뵐"#, r#"뵐"#),
        (r#"뵑"#, r#"뵑"#, r#"뵑"#, r#"뵑"#, r#"뵑"#),
        (r#"뵒"#, r#"뵒"#, r#"뵒"#, r#"뵒"#, r#"뵒"#),
        (r#"뵓"#, r#"뵓"#, r#"뵓"#, r#"뵓"#, r#"뵓"#),
        (r#"뵔"#, r#"뵔"#, r#"뵔"#, r#"뵔"#, r#"뵔"#),
        (r#"뵕"#, r#"뵕"#, r#"뵕"#, r#"뵕"#, r#"뵕"#),
        (r#"뵖"#, r#"뵖"#, r#"뵖"#, r#"뵖"#, r#"뵖"#),
        (r#"뵗"#, r#"뵗"#, r#"뵗"#, r#"뵗"#, r#"뵗"#),
        (r#"뵘"#, r#"뵘"#, r#"뵘"#, r#"뵘"#, r#"뵘"#),
        (r#"뵙"#, r#"뵙"#, r#"뵙"#, r#"뵙"#, r#"뵙"#),
        (r#"뵚"#, r#"뵚"#, r#"뵚"#, r#"뵚"#, r#"뵚"#),
        (r#"뵛"#, r#"뵛"#, r#"뵛"#, r#"뵛"#, r#"뵛"#),
        (r#"뵜"#, r#"뵜"#, r#"뵜"#, r#"뵜"#, r#"뵜"#),
        (r#"뵝"#, r#"뵝"#, r#"뵝"#, r#"뵝"#, r#"뵝"#),
        (r#"뵞"#, r#"뵞"#, r#"뵞"#, r#"뵞"#, r#"뵞"#),
        (r#"뵟"#, r#"뵟"#, r#"뵟"#, r#"뵟"#, r#"뵟"#),
        (r#"뵠"#, r#"뵠"#, r#"뵠"#, r#"뵠"#, r#"뵠"#),
        (r#"뵡"#, r#"뵡"#, r#"뵡"#, r#"뵡"#, r#"뵡"#),
        (r#"뵢"#, r#"뵢"#, r#"뵢"#, r#"뵢"#, r#"뵢"#),
        (r#"뵣"#, r#"뵣"#, r#"뵣"#, r#"뵣"#, r#"뵣"#),
        (r#"뵤"#, r#"뵤"#, r#"뵤"#, r#"뵤"#, r#"뵤"#),
        (r#"뵥"#, r#"뵥"#, r#"뵥"#, r#"뵥"#, r#"뵥"#),
        (r#"뵦"#, r#"뵦"#, r#"뵦"#, r#"뵦"#, r#"뵦"#),
        (r#"뵧"#, r#"뵧"#, r#"뵧"#, r#"뵧"#, r#"뵧"#),
        (r#"뵨"#, r#"뵨"#, r#"뵨"#, r#"뵨"#, r#"뵨"#),
        (r#"뵩"#, r#"뵩"#, r#"뵩"#, r#"뵩"#, r#"뵩"#),
        (r#"뵪"#, r#"뵪"#, r#"뵪"#, r#"뵪"#, r#"뵪"#),
        (r#"뵫"#, r#"뵫"#, r#"뵫"#, r#"뵫"#, r#"뵫"#),
        (r#"뵬"#, r#"뵬"#, r#"뵬"#, r#"뵬"#, r#"뵬"#),
        (r#"뵭"#, r#"뵭"#, r#"뵭"#, r#"뵭"#, r#"뵭"#),
        (r#"뵮"#, r#"뵮"#, r#"뵮"#, r#"뵮"#, r#"뵮"#),
        (r#"뵯"#, r#"뵯"#, r#"뵯"#, r#"뵯"#, r#"뵯"#),
        (r#"뵰"#, r#"뵰"#, r#"뵰"#, r#"뵰"#, r#"뵰"#),
        (r#"뵱"#, r#"뵱"#, r#"뵱"#, r#"뵱"#, r#"뵱"#),
        (r#"뵲"#, r#"뵲"#, r#"뵲"#, r#"뵲"#, r#"뵲"#),
        (r#"뵳"#, r#"뵳"#, r#"뵳"#, r#"뵳"#, r#"뵳"#),
        (r#"뵴"#, r#"뵴"#, r#"뵴"#, r#"뵴"#, r#"뵴"#),
        (r#"뵵"#, r#"뵵"#, r#"뵵"#, r#"뵵"#, r#"뵵"#),
        (r#"뵶"#, r#"뵶"#, r#"뵶"#, r#"뵶"#, r#"뵶"#),
        (r#"뵷"#, r#"뵷"#, r#"뵷"#, r#"뵷"#, r#"뵷"#),
        (r#"뵸"#, r#"뵸"#, r#"뵸"#, r#"뵸"#, r#"뵸"#),
        (r#"뵹"#, r#"뵹"#, r#"뵹"#, r#"뵹"#, r#"뵹"#),
        (r#"뵺"#, r#"뵺"#, r#"뵺"#, r#"뵺"#, r#"뵺"#),
        (r#"뵻"#, r#"뵻"#, r#"뵻"#, r#"뵻"#, r#"뵻"#),
        (r#"뵼"#, r#"뵼"#, r#"뵼"#, r#"뵼"#, r#"뵼"#),
        (r#"뵽"#, r#"뵽"#, r#"뵽"#, r#"뵽"#, r#"뵽"#),
        (r#"뵾"#, r#"뵾"#, r#"뵾"#, r#"뵾"#, r#"뵾"#),
        (r#"뵿"#, r#"뵿"#, r#"뵿"#, r#"뵿"#, r#"뵿"#),
        (r#"부"#, r#"부"#, r#"부"#, r#"부"#, r#"부"#),
        (r#"북"#, r#"북"#, r#"북"#, r#"북"#, r#"북"#),
        (r#"붂"#, r#"붂"#, r#"붂"#, r#"붂"#, r#"붂"#),
        (r#"붃"#, r#"붃"#, r#"붃"#, r#"붃"#, r#"붃"#),
        (r#"분"#, r#"분"#, r#"분"#, r#"분"#, r#"분"#),
        (r#"붅"#, r#"붅"#, r#"붅"#, r#"붅"#, r#"붅"#),
        (r#"붆"#, r#"붆"#, r#"붆"#, r#"붆"#, r#"붆"#),
        (r#"붇"#, r#"붇"#, r#"붇"#, r#"붇"#, r#"붇"#),
        (r#"불"#, r#"불"#, r#"불"#, r#"불"#, r#"불"#),
        (r#"붉"#, r#"붉"#, r#"붉"#, r#"붉"#, r#"붉"#),
        (r#"붊"#, r#"붊"#, r#"붊"#, r#"붊"#, r#"붊"#),
        (r#"붋"#, r#"붋"#, r#"붋"#, r#"붋"#, r#"붋"#),
        (r#"붌"#, r#"붌"#, r#"붌"#, r#"붌"#, r#"붌"#),
        (r#"붍"#, r#"붍"#, r#"붍"#, r#"붍"#, r#"붍"#),
        (r#"붎"#, r#"붎"#, r#"붎"#, r#"붎"#, r#"붎"#),
        (r#"붏"#, r#"붏"#, r#"붏"#, r#"붏"#, r#"붏"#),
        (r#"붐"#, r#"붐"#, r#"붐"#, r#"붐"#, r#"붐"#),
        (r#"붑"#, r#"붑"#, r#"붑"#, r#"붑"#, r#"붑"#),
        (r#"붒"#, r#"붒"#, r#"붒"#, r#"붒"#, r#"붒"#),
        (r#"붓"#, r#"붓"#, r#"붓"#, r#"붓"#, r#"붓"#),
        (r#"붔"#, r#"붔"#, r#"붔"#, r#"붔"#, r#"붔"#),
        (r#"붕"#, r#"붕"#, r#"붕"#, r#"붕"#, r#"붕"#),
        (r#"붖"#, r#"붖"#, r#"붖"#, r#"붖"#, r#"붖"#),
        (r#"붗"#, r#"붗"#, r#"붗"#, r#"붗"#, r#"붗"#),
        (r#"붘"#, r#"붘"#, r#"붘"#, r#"붘"#, r#"붘"#),
        (r#"붙"#, r#"붙"#, r#"붙"#, r#"붙"#, r#"붙"#),
        (r#"붚"#, r#"붚"#, r#"붚"#, r#"붚"#, r#"붚"#),
        (r#"붛"#, r#"붛"#, r#"붛"#, r#"붛"#, r#"붛"#),
        (r#"붜"#, r#"붜"#, r#"붜"#, r#"붜"#, r#"붜"#),
        (r#"붝"#, r#"붝"#, r#"붝"#, r#"붝"#, r#"붝"#),
        (r#"붞"#, r#"붞"#, r#"붞"#, r#"붞"#, r#"붞"#),
        (r#"붟"#, r#"붟"#, r#"붟"#, r#"붟"#, r#"붟"#),
        (r#"붠"#, r#"붠"#, r#"붠"#, r#"붠"#, r#"붠"#),
        (r#"붡"#, r#"붡"#, r#"붡"#, r#"붡"#, r#"붡"#),
        (r#"붢"#, r#"붢"#, r#"붢"#, r#"붢"#, r#"붢"#),
        (r#"붣"#, r#"붣"#, r#"붣"#, r#"붣"#, r#"붣"#),
        (r#"붤"#, r#"붤"#, r#"붤"#, r#"붤"#, r#"붤"#),
        (r#"붥"#, r#"붥"#, r#"붥"#, r#"붥"#, r#"붥"#),
        (r#"붦"#, r#"붦"#, r#"붦"#, r#"붦"#, r#"붦"#),
        (r#"붧"#, r#"붧"#, r#"붧"#, r#"붧"#, r#"붧"#),
        (r#"붨"#, r#"붨"#, r#"붨"#, r#"붨"#, r#"붨"#),
        (r#"붩"#, r#"붩"#, r#"붩"#, r#"붩"#, r#"붩"#),
        (r#"붪"#, r#"붪"#, r#"붪"#, r#"붪"#, r#"붪"#),
        (r#"붫"#, r#"붫"#, r#"붫"#, r#"붫"#, r#"붫"#),
        (r#"붬"#, r#"붬"#, r#"붬"#, r#"붬"#, r#"붬"#),
        (r#"붭"#, r#"붭"#, r#"붭"#, r#"붭"#, r#"붭"#),
        (r#"붮"#, r#"붮"#, r#"붮"#, r#"붮"#, r#"붮"#),
        (r#"붯"#, r#"붯"#, r#"붯"#, r#"붯"#, r#"붯"#),
        (r#"붰"#, r#"붰"#, r#"붰"#, r#"붰"#, r#"붰"#),
        (r#"붱"#, r#"붱"#, r#"붱"#, r#"붱"#, r#"붱"#),
        (r#"붲"#, r#"붲"#, r#"붲"#, r#"붲"#, r#"붲"#),
        (r#"붳"#, r#"붳"#, r#"붳"#, r#"붳"#, r#"붳"#),
        (r#"붴"#, r#"붴"#, r#"붴"#, r#"붴"#, r#"붴"#),
        (r#"붵"#, r#"붵"#, r#"붵"#, r#"붵"#, r#"붵"#),
        (r#"붶"#, r#"붶"#, r#"붶"#, r#"붶"#, r#"붶"#),
        (r#"붷"#, r#"붷"#, r#"붷"#, r#"붷"#, r#"붷"#),
        (r#"붸"#, r#"붸"#, r#"붸"#, r#"붸"#, r#"붸"#),
        (r#"붹"#, r#"붹"#, r#"붹"#, r#"붹"#, r#"붹"#),
        (r#"붺"#, r#"붺"#, r#"붺"#, r#"붺"#, r#"붺"#),
        (r#"붻"#, r#"붻"#, r#"붻"#, r#"붻"#, r#"붻"#),
        (r#"붼"#, r#"붼"#, r#"붼"#, r#"붼"#, r#"붼"#),
        (r#"붽"#, r#"붽"#, r#"붽"#, r#"붽"#, r#"붽"#),
        (r#"붾"#, r#"붾"#, r#"붾"#, r#"붾"#, r#"붾"#),
        (r#"붿"#, r#"붿"#, r#"붿"#, r#"붿"#, r#"붿"#),
        (r#"뷀"#, r#"뷀"#, r#"뷀"#, r#"뷀"#, r#"뷀"#),
        (r#"뷁"#, r#"뷁"#, r#"뷁"#, r#"뷁"#, r#"뷁"#),
        (r#"뷂"#, r#"뷂"#, r#"뷂"#, r#"뷂"#, r#"뷂"#),
        (r#"뷃"#, r#"뷃"#, r#"뷃"#, r#"뷃"#, r#"뷃"#),
        (r#"뷄"#, r#"뷄"#, r#"뷄"#, r#"뷄"#, r#"뷄"#),
        (r#"뷅"#, r#"뷅"#, r#"뷅"#, r#"뷅"#, r#"뷅"#),
        (r#"뷆"#, r#"뷆"#, r#"뷆"#, r#"뷆"#, r#"뷆"#),
        (r#"뷇"#, r#"뷇"#, r#"뷇"#, r#"뷇"#, r#"뷇"#),
        (r#"뷈"#, r#"뷈"#, r#"뷈"#, r#"뷈"#, r#"뷈"#),
        (r#"뷉"#, r#"뷉"#, r#"뷉"#, r#"뷉"#, r#"뷉"#),
        (r#"뷊"#, r#"뷊"#, r#"뷊"#, r#"뷊"#, r#"뷊"#),
        (r#"뷋"#, r#"뷋"#, r#"뷋"#, r#"뷋"#, r#"뷋"#),
        (r#"뷌"#, r#"뷌"#, r#"뷌"#, r#"뷌"#, r#"뷌"#),
        (r#"뷍"#, r#"뷍"#, r#"뷍"#, r#"뷍"#, r#"뷍"#),
        (r#"뷎"#, r#"뷎"#, r#"뷎"#, r#"뷎"#, r#"뷎"#),
        (r#"뷏"#, r#"뷏"#, r#"뷏"#, r#"뷏"#, r#"뷏"#),
        (r#"뷐"#, r#"뷐"#, r#"뷐"#, r#"뷐"#, r#"뷐"#),
        (r#"뷑"#, r#"뷑"#, r#"뷑"#, r#"뷑"#, r#"뷑"#),
        (r#"뷒"#, r#"뷒"#, r#"뷒"#, r#"뷒"#, r#"뷒"#),
        (r#"뷓"#, r#"뷓"#, r#"뷓"#, r#"뷓"#, r#"뷓"#),
        (r#"뷔"#, r#"뷔"#, r#"뷔"#, r#"뷔"#, r#"뷔"#),
        (r#"뷕"#, r#"뷕"#, r#"뷕"#, r#"뷕"#, r#"뷕"#),
        (r#"뷖"#, r#"뷖"#, r#"뷖"#, r#"뷖"#, r#"뷖"#),
        (r#"뷗"#, r#"뷗"#, r#"뷗"#, r#"뷗"#, r#"뷗"#),
        (r#"뷘"#, r#"뷘"#, r#"뷘"#, r#"뷘"#, r#"뷘"#),
        (r#"뷙"#, r#"뷙"#, r#"뷙"#, r#"뷙"#, r#"뷙"#),
        (r#"뷚"#, r#"뷚"#, r#"뷚"#, r#"뷚"#, r#"뷚"#),
        (r#"뷛"#, r#"뷛"#, r#"뷛"#, r#"뷛"#, r#"뷛"#),
        (r#"뷜"#, r#"뷜"#, r#"뷜"#, r#"뷜"#, r#"뷜"#),
        (r#"뷝"#, r#"뷝"#, r#"뷝"#, r#"뷝"#, r#"뷝"#),
        (r#"뷞"#, r#"뷞"#, r#"뷞"#, r#"뷞"#, r#"뷞"#),
        (r#"뷟"#, r#"뷟"#, r#"뷟"#, r#"뷟"#, r#"뷟"#),
        (r#"뷠"#, r#"뷠"#, r#"뷠"#, r#"뷠"#, r#"뷠"#),
        (r#"뷡"#, r#"뷡"#, r#"뷡"#, r#"뷡"#, r#"뷡"#),
        (r#"뷢"#, r#"뷢"#, r#"뷢"#, r#"뷢"#, r#"뷢"#),
        (r#"뷣"#, r#"뷣"#, r#"뷣"#, r#"뷣"#, r#"뷣"#),
        (r#"뷤"#, r#"뷤"#, r#"뷤"#, r#"뷤"#, r#"뷤"#),
        (r#"뷥"#, r#"뷥"#, r#"뷥"#, r#"뷥"#, r#"뷥"#),
        (r#"뷦"#, r#"뷦"#, r#"뷦"#, r#"뷦"#, r#"뷦"#),
        (r#"뷧"#, r#"뷧"#, r#"뷧"#, r#"뷧"#, r#"뷧"#),
        (r#"뷨"#, r#"뷨"#, r#"뷨"#, r#"뷨"#, r#"뷨"#),
        (r#"뷩"#, r#"뷩"#, r#"뷩"#, r#"뷩"#, r#"뷩"#),
        (r#"뷪"#, r#"뷪"#, r#"뷪"#, r#"뷪"#, r#"뷪"#),
        (r#"뷫"#, r#"뷫"#, r#"뷫"#, r#"뷫"#, r#"뷫"#),
        (r#"뷬"#, r#"뷬"#, r#"뷬"#, r#"뷬"#, r#"뷬"#),
        (r#"뷭"#, r#"뷭"#, r#"뷭"#, r#"뷭"#, r#"뷭"#),
        (r#"뷮"#, r#"뷮"#, r#"뷮"#, r#"뷮"#, r#"뷮"#),
        (r#"뷯"#, r#"뷯"#, r#"뷯"#, r#"뷯"#, r#"뷯"#),
        (r#"뷰"#, r#"뷰"#, r#"뷰"#, r#"뷰"#, r#"뷰"#),
        (r#"뷱"#, r#"뷱"#, r#"뷱"#, r#"뷱"#, r#"뷱"#),
        (r#"뷲"#, r#"뷲"#, r#"뷲"#, r#"뷲"#, r#"뷲"#),
        (r#"뷳"#, r#"뷳"#, r#"뷳"#, r#"뷳"#, r#"뷳"#),
        (r#"뷴"#, r#"뷴"#, r#"뷴"#, r#"뷴"#, r#"뷴"#),
        (r#"뷵"#, r#"뷵"#, r#"뷵"#, r#"뷵"#, r#"뷵"#),
        (r#"뷶"#, r#"뷶"#, r#"뷶"#, r#"뷶"#, r#"뷶"#),
        (r#"뷷"#, r#"뷷"#, r#"뷷"#, r#"뷷"#, r#"뷷"#),
        (r#"뷸"#, r#"뷸"#, r#"뷸"#, r#"뷸"#, r#"뷸"#),
        (r#"뷹"#, r#"뷹"#, r#"뷹"#, r#"뷹"#, r#"뷹"#),
        (r#"뷺"#, r#"뷺"#, r#"뷺"#, r#"뷺"#, r#"뷺"#),
        (r#"뷻"#, r#"뷻"#, r#"뷻"#, r#"뷻"#, r#"뷻"#),
        (r#"뷼"#, r#"뷼"#, r#"뷼"#, r#"뷼"#, r#"뷼"#),
        (r#"뷽"#, r#"뷽"#, r#"뷽"#, r#"뷽"#, r#"뷽"#),
        (r#"뷾"#, r#"뷾"#, r#"뷾"#, r#"뷾"#, r#"뷾"#),
        (r#"뷿"#, r#"뷿"#, r#"뷿"#, r#"뷿"#, r#"뷿"#),
        (r#"븀"#, r#"븀"#, r#"븀"#, r#"븀"#, r#"븀"#),
        (r#"븁"#, r#"븁"#, r#"븁"#, r#"븁"#, r#"븁"#),
        (r#"븂"#, r#"븂"#, r#"븂"#, r#"븂"#, r#"븂"#),
        (r#"븃"#, r#"븃"#, r#"븃"#, r#"븃"#, r#"븃"#),
        (r#"븄"#, r#"븄"#, r#"븄"#, r#"븄"#, r#"븄"#),
        (r#"븅"#, r#"븅"#, r#"븅"#, r#"븅"#, r#"븅"#),
        (r#"븆"#, r#"븆"#, r#"븆"#, r#"븆"#, r#"븆"#),
        (r#"븇"#, r#"븇"#, r#"븇"#, r#"븇"#, r#"븇"#),
        (r#"븈"#, r#"븈"#, r#"븈"#, r#"븈"#, r#"븈"#),
        (r#"븉"#, r#"븉"#, r#"븉"#, r#"븉"#, r#"븉"#),
        (r#"븊"#, r#"븊"#, r#"븊"#, r#"븊"#, r#"븊"#),
        (r#"븋"#, r#"븋"#, r#"븋"#, r#"븋"#, r#"븋"#),
        (r#"브"#, r#"브"#, r#"브"#, r#"브"#, r#"브"#),
        (r#"븍"#, r#"븍"#, r#"븍"#, r#"븍"#, r#"븍"#),
        (r#"븎"#, r#"븎"#, r#"븎"#, r#"븎"#, r#"븎"#),
        (r#"븏"#, r#"븏"#, r#"븏"#, r#"븏"#, r#"븏"#),
        (r#"븐"#, r#"븐"#, r#"븐"#, r#"븐"#, r#"븐"#),
        (r#"븑"#, r#"븑"#, r#"븑"#, r#"븑"#, r#"븑"#),
        (r#"븒"#, r#"븒"#, r#"븒"#, r#"븒"#, r#"븒"#),
        (r#"븓"#, r#"븓"#, r#"븓"#, r#"븓"#, r#"븓"#),
        (r#"블"#, r#"블"#, r#"블"#, r#"블"#, r#"블"#),
        (r#"븕"#, r#"븕"#, r#"븕"#, r#"븕"#, r#"븕"#),
        (r#"븖"#, r#"븖"#, r#"븖"#, r#"븖"#, r#"븖"#),
        (r#"븗"#, r#"븗"#, r#"븗"#, r#"븗"#, r#"븗"#),
        (r#"븘"#, r#"븘"#, r#"븘"#, r#"븘"#, r#"븘"#),
        (r#"븙"#, r#"븙"#, r#"븙"#, r#"븙"#, r#"븙"#),
        (r#"븚"#, r#"븚"#, r#"븚"#, r#"븚"#, r#"븚"#),
        (r#"븛"#, r#"븛"#, r#"븛"#, r#"븛"#, r#"븛"#),
        (r#"븜"#, r#"븜"#, r#"븜"#, r#"븜"#, r#"븜"#),
        (r#"븝"#, r#"븝"#, r#"븝"#, r#"븝"#, r#"븝"#),
        (r#"븞"#, r#"븞"#, r#"븞"#, r#"븞"#, r#"븞"#),
        (r#"븟"#, r#"븟"#, r#"븟"#, r#"븟"#, r#"븟"#),
        (r#"븠"#, r#"븠"#, r#"븠"#, r#"븠"#, r#"븠"#),
        (r#"븡"#, r#"븡"#, r#"븡"#, r#"븡"#, r#"븡"#),
        (r#"븢"#, r#"븢"#, r#"븢"#, r#"븢"#, r#"븢"#),
        (r#"븣"#, r#"븣"#, r#"븣"#, r#"븣"#, r#"븣"#),
        (r#"븤"#, r#"븤"#, r#"븤"#, r#"븤"#, r#"븤"#),
        (r#"븥"#, r#"븥"#, r#"븥"#, r#"븥"#, r#"븥"#),
        (r#"븦"#, r#"븦"#, r#"븦"#, r#"븦"#, r#"븦"#),
        (r#"븧"#, r#"븧"#, r#"븧"#, r#"븧"#, r#"븧"#),
        (r#"븨"#, r#"븨"#, r#"븨"#, r#"븨"#, r#"븨"#),
        (r#"븩"#, r#"븩"#, r#"븩"#, r#"븩"#, r#"븩"#),
        (r#"븪"#, r#"븪"#, r#"븪"#, r#"븪"#, r#"븪"#),
        (r#"븫"#, r#"븫"#, r#"븫"#, r#"븫"#, r#"븫"#),
        (r#"븬"#, r#"븬"#, r#"븬"#, r#"븬"#, r#"븬"#),
        (r#"븭"#, r#"븭"#, r#"븭"#, r#"븭"#, r#"븭"#),
        (r#"븮"#, r#"븮"#, r#"븮"#, r#"븮"#, r#"븮"#),
        (r#"븯"#, r#"븯"#, r#"븯"#, r#"븯"#, r#"븯"#),
        (r#"븰"#, r#"븰"#, r#"븰"#, r#"븰"#, r#"븰"#),
        (r#"븱"#, r#"븱"#, r#"븱"#, r#"븱"#, r#"븱"#),
        (r#"븲"#, r#"븲"#, r#"븲"#, r#"븲"#, r#"븲"#),
        (r#"븳"#, r#"븳"#, r#"븳"#, r#"븳"#, r#"븳"#),
        (r#"븴"#, r#"븴"#, r#"븴"#, r#"븴"#, r#"븴"#),
        (r#"븵"#, r#"븵"#, r#"븵"#, r#"븵"#, r#"븵"#),
        (r#"븶"#, r#"븶"#, r#"븶"#, r#"븶"#, r#"븶"#),
        (r#"븷"#, r#"븷"#, r#"븷"#, r#"븷"#, r#"븷"#),
        (r#"븸"#, r#"븸"#, r#"븸"#, r#"븸"#, r#"븸"#),
        (r#"븹"#, r#"븹"#, r#"븹"#, r#"븹"#, r#"븹"#),
        (r#"븺"#, r#"븺"#, r#"븺"#, r#"븺"#, r#"븺"#),
        (r#"븻"#, r#"븻"#, r#"븻"#, r#"븻"#, r#"븻"#),
        (r#"븼"#, r#"븼"#, r#"븼"#, r#"븼"#, r#"븼"#),
        (r#"븽"#, r#"븽"#, r#"븽"#, r#"븽"#, r#"븽"#),
        (r#"븾"#, r#"븾"#, r#"븾"#, r#"븾"#, r#"븾"#),
        (r#"븿"#, r#"븿"#, r#"븿"#, r#"븿"#, r#"븿"#),
        (r#"빀"#, r#"빀"#, r#"빀"#, r#"빀"#, r#"빀"#),
        (r#"빁"#, r#"빁"#, r#"빁"#, r#"빁"#, r#"빁"#),
        (r#"빂"#, r#"빂"#, r#"빂"#, r#"빂"#, r#"빂"#),
        (r#"빃"#, r#"빃"#, r#"빃"#, r#"빃"#, r#"빃"#),
        (r#"비"#, r#"비"#, r#"비"#, r#"비"#, r#"비"#),
        (r#"빅"#, r#"빅"#, r#"빅"#, r#"빅"#, r#"빅"#),
        (r#"빆"#, r#"빆"#, r#"빆"#, r#"빆"#, r#"빆"#),
        (r#"빇"#, r#"빇"#, r#"빇"#, r#"빇"#, r#"빇"#),
        (r#"빈"#, r#"빈"#, r#"빈"#, r#"빈"#, r#"빈"#),
        (r#"빉"#, r#"빉"#, r#"빉"#, r#"빉"#, r#"빉"#),
        (r#"빊"#, r#"빊"#, r#"빊"#, r#"빊"#, r#"빊"#),
        (r#"빋"#, r#"빋"#, r#"빋"#, r#"빋"#, r#"빋"#),
        (r#"빌"#, r#"빌"#, r#"빌"#, r#"빌"#, r#"빌"#),
        (r#"빍"#, r#"빍"#, r#"빍"#, r#"빍"#, r#"빍"#),
        (r#"빎"#, r#"빎"#, r#"빎"#, r#"빎"#, r#"빎"#),
        (r#"빏"#, r#"빏"#, r#"빏"#, r#"빏"#, r#"빏"#),
        (r#"빐"#, r#"빐"#, r#"빐"#, r#"빐"#, r#"빐"#),
        (r#"빑"#, r#"빑"#, r#"빑"#, r#"빑"#, r#"빑"#),
        (r#"빒"#, r#"빒"#, r#"빒"#, r#"빒"#, r#"빒"#),
        (r#"빓"#, r#"빓"#, r#"빓"#, r#"빓"#, r#"빓"#),
        (r#"빔"#, r#"빔"#, r#"빔"#, r#"빔"#, r#"빔"#),
        (r#"빕"#, r#"빕"#, r#"빕"#, r#"빕"#, r#"빕"#),
        (r#"빖"#, r#"빖"#, r#"빖"#, r#"빖"#, r#"빖"#),
        (r#"빗"#, r#"빗"#, r#"빗"#, r#"빗"#, r#"빗"#),
        (r#"빘"#, r#"빘"#, r#"빘"#, r#"빘"#, r#"빘"#),
        (r#"빙"#, r#"빙"#, r#"빙"#, r#"빙"#, r#"빙"#),
        (r#"빚"#, r#"빚"#, r#"빚"#, r#"빚"#, r#"빚"#),
        (r#"빛"#, r#"빛"#, r#"빛"#, r#"빛"#, r#"빛"#),
        (r#"빜"#, r#"빜"#, r#"빜"#, r#"빜"#, r#"빜"#),
        (r#"빝"#, r#"빝"#, r#"빝"#, r#"빝"#, r#"빝"#),
        (r#"빞"#, r#"빞"#, r#"빞"#, r#"빞"#, r#"빞"#),
        (r#"빟"#, r#"빟"#, r#"빟"#, r#"빟"#, r#"빟"#),
        (r#"빠"#, r#"빠"#, r#"빠"#, r#"빠"#, r#"빠"#),
        (r#"빡"#, r#"빡"#, r#"빡"#, r#"빡"#, r#"빡"#),
        (r#"빢"#, r#"빢"#, r#"빢"#, r#"빢"#, r#"빢"#),
        (r#"빣"#, r#"빣"#, r#"빣"#, r#"빣"#, r#"빣"#),
        (r#"빤"#, r#"빤"#, r#"빤"#, r#"빤"#, r#"빤"#),
        (r#"빥"#, r#"빥"#, r#"빥"#, r#"빥"#, r#"빥"#),
        (r#"빦"#, r#"빦"#, r#"빦"#, r#"빦"#, r#"빦"#),
        (r#"빧"#, r#"빧"#, r#"빧"#, r#"빧"#, r#"빧"#),
        (r#"빨"#, r#"빨"#, r#"빨"#, r#"빨"#, r#"빨"#),
        (r#"빩"#, r#"빩"#, r#"빩"#, r#"빩"#, r#"빩"#),
        (r#"빪"#, r#"빪"#, r#"빪"#, r#"빪"#, r#"빪"#),
        (r#"빫"#, r#"빫"#, r#"빫"#, r#"빫"#, r#"빫"#),
        (r#"빬"#, r#"빬"#, r#"빬"#, r#"빬"#, r#"빬"#),
        (r#"빭"#, r#"빭"#, r#"빭"#, r#"빭"#, r#"빭"#),
        (r#"빮"#, r#"빮"#, r#"빮"#, r#"빮"#, r#"빮"#),
        (r#"빯"#, r#"빯"#, r#"빯"#, r#"빯"#, r#"빯"#),
        (r#"빰"#, r#"빰"#, r#"빰"#, r#"빰"#, r#"빰"#),
        (r#"빱"#, r#"빱"#, r#"빱"#, r#"빱"#, r#"빱"#),
        (r#"빲"#, r#"빲"#, r#"빲"#, r#"빲"#, r#"빲"#),
        (r#"빳"#, r#"빳"#, r#"빳"#, r#"빳"#, r#"빳"#),
        (r#"빴"#, r#"빴"#, r#"빴"#, r#"빴"#, r#"빴"#),
        (r#"빵"#, r#"빵"#, r#"빵"#, r#"빵"#, r#"빵"#),
        (r#"빶"#, r#"빶"#, r#"빶"#, r#"빶"#, r#"빶"#),
        (r#"빷"#, r#"빷"#, r#"빷"#, r#"빷"#, r#"빷"#),
        (r#"빸"#, r#"빸"#, r#"빸"#, r#"빸"#, r#"빸"#),
        (r#"빹"#, r#"빹"#, r#"빹"#, r#"빹"#, r#"빹"#),
        (r#"빺"#, r#"빺"#, r#"빺"#, r#"빺"#, r#"빺"#),
        (r#"빻"#, r#"빻"#, r#"빻"#, r#"빻"#, r#"빻"#),
        (r#"빼"#, r#"빼"#, r#"빼"#, r#"빼"#, r#"빼"#),
        (r#"빽"#, r#"빽"#, r#"빽"#, r#"빽"#, r#"빽"#),
        (r#"빾"#, r#"빾"#, r#"빾"#, r#"빾"#, r#"빾"#),
        (r#"빿"#, r#"빿"#, r#"빿"#, r#"빿"#, r#"빿"#),
        (r#"뺀"#, r#"뺀"#, r#"뺀"#, r#"뺀"#, r#"뺀"#),
        (r#"뺁"#, r#"뺁"#, r#"뺁"#, r#"뺁"#, r#"뺁"#),
        (r#"뺂"#, r#"뺂"#, r#"뺂"#, r#"뺂"#, r#"뺂"#),
        (r#"뺃"#, r#"뺃"#, r#"뺃"#, r#"뺃"#, r#"뺃"#),
        (r#"뺄"#, r#"뺄"#, r#"뺄"#, r#"뺄"#, r#"뺄"#),
        (r#"뺅"#, r#"뺅"#, r#"뺅"#, r#"뺅"#, r#"뺅"#),
        (r#"뺆"#, r#"뺆"#, r#"뺆"#, r#"뺆"#, r#"뺆"#),
        (r#"뺇"#, r#"뺇"#, r#"뺇"#, r#"뺇"#, r#"뺇"#),
        (r#"뺈"#, r#"뺈"#, r#"뺈"#, r#"뺈"#, r#"뺈"#),
        (r#"뺉"#, r#"뺉"#, r#"뺉"#, r#"뺉"#, r#"뺉"#),
        (r#"뺊"#, r#"뺊"#, r#"뺊"#, r#"뺊"#, r#"뺊"#),
        (r#"뺋"#, r#"뺋"#, r#"뺋"#, r#"뺋"#, r#"뺋"#),
        (r#"뺌"#, r#"뺌"#, r#"뺌"#, r#"뺌"#, r#"뺌"#),
        (r#"뺍"#, r#"뺍"#, r#"뺍"#, r#"뺍"#, r#"뺍"#),
        (r#"뺎"#, r#"뺎"#, r#"뺎"#, r#"뺎"#, r#"뺎"#),
        (r#"뺏"#, r#"뺏"#, r#"뺏"#, r#"뺏"#, r#"뺏"#),
        (r#"뺐"#, r#"뺐"#, r#"뺐"#, r#"뺐"#, r#"뺐"#),
        (r#"뺑"#, r#"뺑"#, r#"뺑"#, r#"뺑"#, r#"뺑"#),
        (r#"뺒"#, r#"뺒"#, r#"뺒"#, r#"뺒"#, r#"뺒"#),
        (r#"뺓"#, r#"뺓"#, r#"뺓"#, r#"뺓"#, r#"뺓"#),
        (r#"뺔"#, r#"뺔"#, r#"뺔"#, r#"뺔"#, r#"뺔"#),
        (r#"뺕"#, r#"뺕"#, r#"뺕"#, r#"뺕"#, r#"뺕"#),
        (r#"뺖"#, r#"뺖"#, r#"뺖"#, r#"뺖"#, r#"뺖"#),
        (r#"뺗"#, r#"뺗"#, r#"뺗"#, r#"뺗"#, r#"뺗"#),
        (r#"뺘"#, r#"뺘"#, r#"뺘"#, r#"뺘"#, r#"뺘"#),
        (r#"뺙"#, r#"뺙"#, r#"뺙"#, r#"뺙"#, r#"뺙"#),
        (r#"뺚"#, r#"뺚"#, r#"뺚"#, r#"뺚"#, r#"뺚"#),
        (r#"뺛"#, r#"뺛"#, r#"뺛"#, r#"뺛"#, r#"뺛"#),
        (r#"뺜"#, r#"뺜"#, r#"뺜"#, r#"뺜"#, r#"뺜"#),
        (r#"뺝"#, r#"뺝"#, r#"뺝"#, r#"뺝"#, r#"뺝"#),
        (r#"뺞"#, r#"뺞"#, r#"뺞"#, r#"뺞"#, r#"뺞"#),
        (r#"뺟"#, r#"뺟"#, r#"뺟"#, r#"뺟"#, r#"뺟"#),
        (r#"뺠"#, r#"뺠"#, r#"뺠"#, r#"뺠"#, r#"뺠"#),
        (r#"뺡"#, r#"뺡"#, r#"뺡"#, r#"뺡"#, r#"뺡"#),
        (r#"뺢"#, r#"뺢"#, r#"뺢"#, r#"뺢"#, r#"뺢"#),
        (r#"뺣"#, r#"뺣"#, r#"뺣"#, r#"뺣"#, r#"뺣"#),
        (r#"뺤"#, r#"뺤"#, r#"뺤"#, r#"뺤"#, r#"뺤"#),
        (r#"뺥"#, r#"뺥"#, r#"뺥"#, r#"뺥"#, r#"뺥"#),
        (r#"뺦"#, r#"뺦"#, r#"뺦"#, r#"뺦"#, r#"뺦"#),
        (r#"뺧"#, r#"뺧"#, r#"뺧"#, r#"뺧"#, r#"뺧"#),
        (r#"뺨"#, r#"뺨"#, r#"뺨"#, r#"뺨"#, r#"뺨"#),
        (r#"뺩"#, r#"뺩"#, r#"뺩"#, r#"뺩"#, r#"뺩"#),
        (r#"뺪"#, r#"뺪"#, r#"뺪"#, r#"뺪"#, r#"뺪"#),
        (r#"뺫"#, r#"뺫"#, r#"뺫"#, r#"뺫"#, r#"뺫"#),
        (r#"뺬"#, r#"뺬"#, r#"뺬"#, r#"뺬"#, r#"뺬"#),
        (r#"뺭"#, r#"뺭"#, r#"뺭"#, r#"뺭"#, r#"뺭"#),
        (r#"뺮"#, r#"뺮"#, r#"뺮"#, r#"뺮"#, r#"뺮"#),
        (r#"뺯"#, r#"뺯"#, r#"뺯"#, r#"뺯"#, r#"뺯"#),
        (r#"뺰"#, r#"뺰"#, r#"뺰"#, r#"뺰"#, r#"뺰"#),
        (r#"뺱"#, r#"뺱"#, r#"뺱"#, r#"뺱"#, r#"뺱"#),
        (r#"뺲"#, r#"뺲"#, r#"뺲"#, r#"뺲"#, r#"뺲"#),
        (r#"뺳"#, r#"뺳"#, r#"뺳"#, r#"뺳"#, r#"뺳"#),
        (r#"뺴"#, r#"뺴"#, r#"뺴"#, r#"뺴"#, r#"뺴"#),
        (r#"뺵"#, r#"뺵"#, r#"뺵"#, r#"뺵"#, r#"뺵"#),
        (r#"뺶"#, r#"뺶"#, r#"뺶"#, r#"뺶"#, r#"뺶"#),
        (r#"뺷"#, r#"뺷"#, r#"뺷"#, r#"뺷"#, r#"뺷"#),
        (r#"뺸"#, r#"뺸"#, r#"뺸"#, r#"뺸"#, r#"뺸"#),
        (r#"뺹"#, r#"뺹"#, r#"뺹"#, r#"뺹"#, r#"뺹"#),
        (r#"뺺"#, r#"뺺"#, r#"뺺"#, r#"뺺"#, r#"뺺"#),
        (r#"뺻"#, r#"뺻"#, r#"뺻"#, r#"뺻"#, r#"뺻"#),
        (r#"뺼"#, r#"뺼"#, r#"뺼"#, r#"뺼"#, r#"뺼"#),
        (r#"뺽"#, r#"뺽"#, r#"뺽"#, r#"뺽"#, r#"뺽"#),
        (r#"뺾"#, r#"뺾"#, r#"뺾"#, r#"뺾"#, r#"뺾"#),
        (r#"뺿"#, r#"뺿"#, r#"뺿"#, r#"뺿"#, r#"뺿"#),
        (r#"뻀"#, r#"뻀"#, r#"뻀"#, r#"뻀"#, r#"뻀"#),
        (r#"뻁"#, r#"뻁"#, r#"뻁"#, r#"뻁"#, r#"뻁"#),
        (r#"뻂"#, r#"뻂"#, r#"뻂"#, r#"뻂"#, r#"뻂"#),
        (r#"뻃"#, r#"뻃"#, r#"뻃"#, r#"뻃"#, r#"뻃"#),
        (r#"뻄"#, r#"뻄"#, r#"뻄"#, r#"뻄"#, r#"뻄"#),
        (r#"뻅"#, r#"뻅"#, r#"뻅"#, r#"뻅"#, r#"뻅"#),
        (r#"뻆"#, r#"뻆"#, r#"뻆"#, r#"뻆"#, r#"뻆"#),
        (r#"뻇"#, r#"뻇"#, r#"뻇"#, r#"뻇"#, r#"뻇"#),
        (r#"뻈"#, r#"뻈"#, r#"뻈"#, r#"뻈"#, r#"뻈"#),
        (r#"뻉"#, r#"뻉"#, r#"뻉"#, r#"뻉"#, r#"뻉"#),
        (r#"뻊"#, r#"뻊"#, r#"뻊"#, r#"뻊"#, r#"뻊"#),
        (r#"뻋"#, r#"뻋"#, r#"뻋"#, r#"뻋"#, r#"뻋"#),
        (r#"뻌"#, r#"뻌"#, r#"뻌"#, r#"뻌"#, r#"뻌"#),
        (r#"뻍"#, r#"뻍"#, r#"뻍"#, r#"뻍"#, r#"뻍"#),
        (r#"뻎"#, r#"뻎"#, r#"뻎"#, r#"뻎"#, r#"뻎"#),
        (r#"뻏"#, r#"뻏"#, r#"뻏"#, r#"뻏"#, r#"뻏"#),
        (r#"뻐"#, r#"뻐"#, r#"뻐"#, r#"뻐"#, r#"뻐"#),
        (r#"뻑"#, r#"뻑"#, r#"뻑"#, r#"뻑"#, r#"뻑"#),
        (r#"뻒"#, r#"뻒"#, r#"뻒"#, r#"뻒"#, r#"뻒"#),
        (r#"뻓"#, r#"뻓"#, r#"뻓"#, r#"뻓"#, r#"뻓"#),
        (r#"뻔"#, r#"뻔"#, r#"뻔"#, r#"뻔"#, r#"뻔"#),
        (r#"뻕"#, r#"뻕"#, r#"뻕"#, r#"뻕"#, r#"뻕"#),
        (r#"뻖"#, r#"뻖"#, r#"뻖"#, r#"뻖"#, r#"뻖"#),
        (r#"뻗"#, r#"뻗"#, r#"뻗"#, r#"뻗"#, r#"뻗"#),
        (r#"뻘"#, r#"뻘"#, r#"뻘"#, r#"뻘"#, r#"뻘"#),
        (r#"뻙"#, r#"뻙"#, r#"뻙"#, r#"뻙"#, r#"뻙"#),
        (r#"뻚"#, r#"뻚"#, r#"뻚"#, r#"뻚"#, r#"뻚"#),
        (r#"뻛"#, r#"뻛"#, r#"뻛"#, r#"뻛"#, r#"뻛"#),
        (r#"뻜"#, r#"뻜"#, r#"뻜"#, r#"뻜"#, r#"뻜"#),
        (r#"뻝"#, r#"뻝"#, r#"뻝"#, r#"뻝"#, r#"뻝"#),
        (r#"뻞"#, r#"뻞"#, r#"뻞"#, r#"뻞"#, r#"뻞"#),
        (r#"뻟"#, r#"뻟"#, r#"뻟"#, r#"뻟"#, r#"뻟"#),
        (r#"뻠"#, r#"뻠"#, r#"뻠"#, r#"뻠"#, r#"뻠"#),
        (r#"뻡"#, r#"뻡"#, r#"뻡"#, r#"뻡"#, r#"뻡"#),
        (r#"뻢"#, r#"뻢"#, r#"뻢"#, r#"뻢"#, r#"뻢"#),
        (r#"뻣"#, r#"뻣"#, r#"뻣"#, r#"뻣"#, r#"뻣"#),
        (r#"뻤"#, r#"뻤"#, r#"뻤"#, r#"뻤"#, r#"뻤"#),
        (r#"뻥"#, r#"뻥"#, r#"뻥"#, r#"뻥"#, r#"뻥"#),
        (r#"뻦"#, r#"뻦"#, r#"뻦"#, r#"뻦"#, r#"뻦"#),
        (r#"뻧"#, r#"뻧"#, r#"뻧"#, r#"뻧"#, r#"뻧"#),
        (r#"뻨"#, r#"뻨"#, r#"뻨"#, r#"뻨"#, r#"뻨"#),
        (r#"뻩"#, r#"뻩"#, r#"뻩"#, r#"뻩"#, r#"뻩"#),
        (r#"뻪"#, r#"뻪"#, r#"뻪"#, r#"뻪"#, r#"뻪"#),
        (r#"뻫"#, r#"뻫"#, r#"뻫"#, r#"뻫"#, r#"뻫"#),
        (r#"뻬"#, r#"뻬"#, r#"뻬"#, r#"뻬"#, r#"뻬"#),
        (r#"뻭"#, r#"뻭"#, r#"뻭"#, r#"뻭"#, r#"뻭"#),
        (r#"뻮"#, r#"뻮"#, r#"뻮"#, r#"뻮"#, r#"뻮"#),
        (r#"뻯"#, r#"뻯"#, r#"뻯"#, r#"뻯"#, r#"뻯"#),
        (r#"뻰"#, r#"뻰"#, r#"뻰"#, r#"뻰"#, r#"뻰"#),
        (r#"뻱"#, r#"뻱"#, r#"뻱"#, r#"뻱"#, r#"뻱"#),
        (r#"뻲"#, r#"뻲"#, r#"뻲"#, r#"뻲"#, r#"뻲"#),
        (r#"뻳"#, r#"뻳"#, r#"뻳"#, r#"뻳"#, r#"뻳"#),
        (r#"뻴"#, r#"뻴"#, r#"뻴"#, r#"뻴"#, r#"뻴"#),
        (r#"뻵"#, r#"뻵"#, r#"뻵"#, r#"뻵"#, r#"뻵"#),
        (r#"뻶"#, r#"뻶"#, r#"뻶"#, r#"뻶"#, r#"뻶"#),
        (r#"뻷"#, r#"뻷"#, r#"뻷"#, r#"뻷"#, r#"뻷"#),
        (r#"뻸"#, r#"뻸"#, r#"뻸"#, r#"뻸"#, r#"뻸"#),
        (r#"뻹"#, r#"뻹"#, r#"뻹"#, r#"뻹"#, r#"뻹"#),
        (r#"뻺"#, r#"뻺"#, r#"뻺"#, r#"뻺"#, r#"뻺"#),
        (r#"뻻"#, r#"뻻"#, r#"뻻"#, r#"뻻"#, r#"뻻"#),
        (r#"뻼"#, r#"뻼"#, r#"뻼"#, r#"뻼"#, r#"뻼"#),
        (r#"뻽"#, r#"뻽"#, r#"뻽"#, r#"뻽"#, r#"뻽"#),
        (r#"뻾"#, r#"뻾"#, r#"뻾"#, r#"뻾"#, r#"뻾"#),
        (r#"뻿"#, r#"뻿"#, r#"뻿"#, r#"뻿"#, r#"뻿"#),
        (r#"뼀"#, r#"뼀"#, r#"뼀"#, r#"뼀"#, r#"뼀"#),
        (r#"뼁"#, r#"뼁"#, r#"뼁"#, r#"뼁"#, r#"뼁"#),
        (r#"뼂"#, r#"뼂"#, r#"뼂"#, r#"뼂"#, r#"뼂"#),
        (r#"뼃"#, r#"뼃"#, r#"뼃"#, r#"뼃"#, r#"뼃"#),
        (r#"뼄"#, r#"뼄"#, r#"뼄"#, r#"뼄"#, r#"뼄"#),
        (r#"뼅"#, r#"뼅"#, r#"뼅"#, r#"뼅"#, r#"뼅"#),
        (r#"뼆"#, r#"뼆"#, r#"뼆"#, r#"뼆"#, r#"뼆"#),
        (r#"뼇"#, r#"뼇"#, r#"뼇"#, r#"뼇"#, r#"뼇"#),
        (r#"뼈"#, r#"뼈"#, r#"뼈"#, r#"뼈"#, r#"뼈"#),
        (r#"뼉"#, r#"뼉"#, r#"뼉"#, r#"뼉"#, r#"뼉"#),
        (r#"뼊"#, r#"뼊"#, r#"뼊"#, r#"뼊"#, r#"뼊"#),
        (r#"뼋"#, r#"뼋"#, r#"뼋"#, r#"뼋"#, r#"뼋"#),
        (r#"뼌"#, r#"뼌"#, r#"뼌"#, r#"뼌"#, r#"뼌"#),
        (r#"뼍"#, r#"뼍"#, r#"뼍"#, r#"뼍"#, r#"뼍"#),
        (r#"뼎"#, r#"뼎"#, r#"뼎"#, r#"뼎"#, r#"뼎"#),
        (r#"뼏"#, r#"뼏"#, r#"뼏"#, r#"뼏"#, r#"뼏"#),
        (r#"뼐"#, r#"뼐"#, r#"뼐"#, r#"뼐"#, r#"뼐"#),
        (r#"뼑"#, r#"뼑"#, r#"뼑"#, r#"뼑"#, r#"뼑"#),
        (r#"뼒"#, r#"뼒"#, r#"뼒"#, r#"뼒"#, r#"뼒"#),
        (r#"뼓"#, r#"뼓"#, r#"뼓"#, r#"뼓"#, r#"뼓"#),
        (r#"뼔"#, r#"뼔"#, r#"뼔"#, r#"뼔"#, r#"뼔"#),
        (r#"뼕"#, r#"뼕"#, r#"뼕"#, r#"뼕"#, r#"뼕"#),
        (r#"뼖"#, r#"뼖"#, r#"뼖"#, r#"뼖"#, r#"뼖"#),
        (r#"뼗"#, r#"뼗"#, r#"뼗"#, r#"뼗"#, r#"뼗"#),
        (r#"뼘"#, r#"뼘"#, r#"뼘"#, r#"뼘"#, r#"뼘"#),
        (r#"뼙"#, r#"뼙"#, r#"뼙"#, r#"뼙"#, r#"뼙"#),
        (r#"뼚"#, r#"뼚"#, r#"뼚"#, r#"뼚"#, r#"뼚"#),
        (r#"뼛"#, r#"뼛"#, r#"뼛"#, r#"뼛"#, r#"뼛"#),
        (r#"뼜"#, r#"뼜"#, r#"뼜"#, r#"뼜"#, r#"뼜"#),
        (r#"뼝"#, r#"뼝"#, r#"뼝"#, r#"뼝"#, r#"뼝"#),
        (r#"뼞"#, r#"뼞"#, r#"뼞"#, r#"뼞"#, r#"뼞"#),
        (r#"뼟"#, r#"뼟"#, r#"뼟"#, r#"뼟"#, r#"뼟"#),
        (r#"뼠"#, r#"뼠"#, r#"뼠"#, r#"뼠"#, r#"뼠"#),
        (r#"뼡"#, r#"뼡"#, r#"뼡"#, r#"뼡"#, r#"뼡"#),
        (r#"뼢"#, r#"뼢"#, r#"뼢"#, r#"뼢"#, r#"뼢"#),
        (r#"뼣"#, r#"뼣"#, r#"뼣"#, r#"뼣"#, r#"뼣"#),
        (r#"뼤"#, r#"뼤"#, r#"뼤"#, r#"뼤"#, r#"뼤"#),
        (r#"뼥"#, r#"뼥"#, r#"뼥"#, r#"뼥"#, r#"뼥"#),
        (r#"뼦"#, r#"뼦"#, r#"뼦"#, r#"뼦"#, r#"뼦"#),
        (r#"뼧"#, r#"뼧"#, r#"뼧"#, r#"뼧"#, r#"뼧"#),
        (r#"뼨"#, r#"뼨"#, r#"뼨"#, r#"뼨"#, r#"뼨"#),
        (r#"뼩"#, r#"뼩"#, r#"뼩"#, r#"뼩"#, r#"뼩"#),
        (r#"뼪"#, r#"뼪"#, r#"뼪"#, r#"뼪"#, r#"뼪"#),
        (r#"뼫"#, r#"뼫"#, r#"뼫"#, r#"뼫"#, r#"뼫"#),
        (r#"뼬"#, r#"뼬"#, r#"뼬"#, r#"뼬"#, r#"뼬"#),
        (r#"뼭"#, r#"뼭"#, r#"뼭"#, r#"뼭"#, r#"뼭"#),
        (r#"뼮"#, r#"뼮"#, r#"뼮"#, r#"뼮"#, r#"뼮"#),
        (r#"뼯"#, r#"뼯"#, r#"뼯"#, r#"뼯"#, r#"뼯"#),
        (r#"뼰"#, r#"뼰"#, r#"뼰"#, r#"뼰"#, r#"뼰"#),
        (r#"뼱"#, r#"뼱"#, r#"뼱"#, r#"뼱"#, r#"뼱"#),
        (r#"뼲"#, r#"뼲"#, r#"뼲"#, r#"뼲"#, r#"뼲"#),
        (r#"뼳"#, r#"뼳"#, r#"뼳"#, r#"뼳"#, r#"뼳"#),
        (r#"뼴"#, r#"뼴"#, r#"뼴"#, r#"뼴"#, r#"뼴"#),
        (r#"뼵"#, r#"뼵"#, r#"뼵"#, r#"뼵"#, r#"뼵"#),
        (r#"뼶"#, r#"뼶"#, r#"뼶"#, r#"뼶"#, r#"뼶"#),
        (r#"뼷"#, r#"뼷"#, r#"뼷"#, r#"뼷"#, r#"뼷"#),
        (r#"뼸"#, r#"뼸"#, r#"뼸"#, r#"뼸"#, r#"뼸"#),
        (r#"뼹"#, r#"뼹"#, r#"뼹"#, r#"뼹"#, r#"뼹"#),
        (r#"뼺"#, r#"뼺"#, r#"뼺"#, r#"뼺"#, r#"뼺"#),
        (r#"뼻"#, r#"뼻"#, r#"뼻"#, r#"뼻"#, r#"뼻"#),
        (r#"뼼"#, r#"뼼"#, r#"뼼"#, r#"뼼"#, r#"뼼"#),
        (r#"뼽"#, r#"뼽"#, r#"뼽"#, r#"뼽"#, r#"뼽"#),
        (r#"뼾"#, r#"뼾"#, r#"뼾"#, r#"뼾"#, r#"뼾"#),
        (r#"뼿"#, r#"뼿"#, r#"뼿"#, r#"뼿"#, r#"뼿"#),
        (r#"뽀"#, r#"뽀"#, r#"뽀"#, r#"뽀"#, r#"뽀"#),
        (r#"뽁"#, r#"뽁"#, r#"뽁"#, r#"뽁"#, r#"뽁"#),
        (r#"뽂"#, r#"뽂"#, r#"뽂"#, r#"뽂"#, r#"뽂"#),
        (r#"뽃"#, r#"뽃"#, r#"뽃"#, r#"뽃"#, r#"뽃"#),
        (r#"뽄"#, r#"뽄"#, r#"뽄"#, r#"뽄"#, r#"뽄"#),
        (r#"뽅"#, r#"뽅"#, r#"뽅"#, r#"뽅"#, r#"뽅"#),
        (r#"뽆"#, r#"뽆"#, r#"뽆"#, r#"뽆"#, r#"뽆"#),
        (r#"뽇"#, r#"뽇"#, r#"뽇"#, r#"뽇"#, r#"뽇"#),
        (r#"뽈"#, r#"뽈"#, r#"뽈"#, r#"뽈"#, r#"뽈"#),
        (r#"뽉"#, r#"뽉"#, r#"뽉"#, r#"뽉"#, r#"뽉"#),
        (r#"뽊"#, r#"뽊"#, r#"뽊"#, r#"뽊"#, r#"뽊"#),
        (r#"뽋"#, r#"뽋"#, r#"뽋"#, r#"뽋"#, r#"뽋"#),
        (r#"뽌"#, r#"뽌"#, r#"뽌"#, r#"뽌"#, r#"뽌"#),
        (r#"뽍"#, r#"뽍"#, r#"뽍"#, r#"뽍"#, r#"뽍"#),
        (r#"뽎"#, r#"뽎"#, r#"뽎"#, r#"뽎"#, r#"뽎"#),
        (r#"뽏"#, r#"뽏"#, r#"뽏"#, r#"뽏"#, r#"뽏"#),
        (r#"뽐"#, r#"뽐"#, r#"뽐"#, r#"뽐"#, r#"뽐"#),
        (r#"뽑"#, r#"뽑"#, r#"뽑"#, r#"뽑"#, r#"뽑"#),
        (r#"뽒"#, r#"뽒"#, r#"뽒"#, r#"뽒"#, r#"뽒"#),
        (r#"뽓"#, r#"뽓"#, r#"뽓"#, r#"뽓"#, r#"뽓"#),
        (r#"뽔"#, r#"뽔"#, r#"뽔"#, r#"뽔"#, r#"뽔"#),
        (r#"뽕"#, r#"뽕"#, r#"뽕"#, r#"뽕"#, r#"뽕"#),
        (r#"뽖"#, r#"뽖"#, r#"뽖"#, r#"뽖"#, r#"뽖"#),
        (r#"뽗"#, r#"뽗"#, r#"뽗"#, r#"뽗"#, r#"뽗"#),
        (r#"뽘"#, r#"뽘"#, r#"뽘"#, r#"뽘"#, r#"뽘"#),
        (r#"뽙"#, r#"뽙"#, r#"뽙"#, r#"뽙"#, r#"뽙"#),
        (r#"뽚"#, r#"뽚"#, r#"뽚"#, r#"뽚"#, r#"뽚"#),
        (r#"뽛"#, r#"뽛"#, r#"뽛"#, r#"뽛"#, r#"뽛"#),
        (r#"뽜"#, r#"뽜"#, r#"뽜"#, r#"뽜"#, r#"뽜"#),
        (r#"뽝"#, r#"뽝"#, r#"뽝"#, r#"뽝"#, r#"뽝"#),
        (r#"뽞"#, r#"뽞"#, r#"뽞"#, r#"뽞"#, r#"뽞"#),
        (r#"뽟"#, r#"뽟"#, r#"뽟"#, r#"뽟"#, r#"뽟"#),
        (r#"뽠"#, r#"뽠"#, r#"뽠"#, r#"뽠"#, r#"뽠"#),
        (r#"뽡"#, r#"뽡"#, r#"뽡"#, r#"뽡"#, r#"뽡"#),
        (r#"뽢"#, r#"뽢"#, r#"뽢"#, r#"뽢"#, r#"뽢"#),
        (r#"뽣"#, r#"뽣"#, r#"뽣"#, r#"뽣"#, r#"뽣"#),
        (r#"뽤"#, r#"뽤"#, r#"뽤"#, r#"뽤"#, r#"뽤"#),
        (r#"뽥"#, r#"뽥"#, r#"뽥"#, r#"뽥"#, r#"뽥"#),
        (r#"뽦"#, r#"뽦"#, r#"뽦"#, r#"뽦"#, r#"뽦"#),
        (r#"뽧"#, r#"뽧"#, r#"뽧"#, r#"뽧"#, r#"뽧"#),
        (r#"뽨"#, r#"뽨"#, r#"뽨"#, r#"뽨"#, r#"뽨"#),
        (r#"뽩"#, r#"뽩"#, r#"뽩"#, r#"뽩"#, r#"뽩"#),
        (r#"뽪"#, r#"뽪"#, r#"뽪"#, r#"뽪"#, r#"뽪"#),
        (r#"뽫"#, r#"뽫"#, r#"뽫"#, r#"뽫"#, r#"뽫"#),
        (r#"뽬"#, r#"뽬"#, r#"뽬"#, r#"뽬"#, r#"뽬"#),
        (r#"뽭"#, r#"뽭"#, r#"뽭"#, r#"뽭"#, r#"뽭"#),
        (r#"뽮"#, r#"뽮"#, r#"뽮"#, r#"뽮"#, r#"뽮"#),
        (r#"뽯"#, r#"뽯"#, r#"뽯"#, r#"뽯"#, r#"뽯"#),
        (r#"뽰"#, r#"뽰"#, r#"뽰"#, r#"뽰"#, r#"뽰"#),
        (r#"뽱"#, r#"뽱"#, r#"뽱"#, r#"뽱"#, r#"뽱"#),
        (r#"뽲"#, r#"뽲"#, r#"뽲"#, r#"뽲"#, r#"뽲"#),
        (r#"뽳"#, r#"뽳"#, r#"뽳"#, r#"뽳"#, r#"뽳"#),
        (r#"뽴"#, r#"뽴"#, r#"뽴"#, r#"뽴"#, r#"뽴"#),
        (r#"뽵"#, r#"뽵"#, r#"뽵"#, r#"뽵"#, r#"뽵"#),
        (r#"뽶"#, r#"뽶"#, r#"뽶"#, r#"뽶"#, r#"뽶"#),
        (r#"뽷"#, r#"뽷"#, r#"뽷"#, r#"뽷"#, r#"뽷"#),
        (r#"뽸"#, r#"뽸"#, r#"뽸"#, r#"뽸"#, r#"뽸"#),
        (r#"뽹"#, r#"뽹"#, r#"뽹"#, r#"뽹"#, r#"뽹"#),
        (r#"뽺"#, r#"뽺"#, r#"뽺"#, r#"뽺"#, r#"뽺"#),
        (r#"뽻"#, r#"뽻"#, r#"뽻"#, r#"뽻"#, r#"뽻"#),
        (r#"뽼"#, r#"뽼"#, r#"뽼"#, r#"뽼"#, r#"뽼"#),
        (r#"뽽"#, r#"뽽"#, r#"뽽"#, r#"뽽"#, r#"뽽"#),
        (r#"뽾"#, r#"뽾"#, r#"뽾"#, r#"뽾"#, r#"뽾"#),
        (r#"뽿"#, r#"뽿"#, r#"뽿"#, r#"뽿"#, r#"뽿"#),
        (r#"뾀"#, r#"뾀"#, r#"뾀"#, r#"뾀"#, r#"뾀"#),
        (r#"뾁"#, r#"뾁"#, r#"뾁"#, r#"뾁"#, r#"뾁"#),
        (r#"뾂"#, r#"뾂"#, r#"뾂"#, r#"뾂"#, r#"뾂"#),
        (r#"뾃"#, r#"뾃"#, r#"뾃"#, r#"뾃"#, r#"뾃"#),
        (r#"뾄"#, r#"뾄"#, r#"뾄"#, r#"뾄"#, r#"뾄"#),
        (r#"뾅"#, r#"뾅"#, r#"뾅"#, r#"뾅"#, r#"뾅"#),
        (r#"뾆"#, r#"뾆"#, r#"뾆"#, r#"뾆"#, r#"뾆"#),
        (r#"뾇"#, r#"뾇"#, r#"뾇"#, r#"뾇"#, r#"뾇"#),
        (r#"뾈"#, r#"뾈"#, r#"뾈"#, r#"뾈"#, r#"뾈"#),
        (r#"뾉"#, r#"뾉"#, r#"뾉"#, r#"뾉"#, r#"뾉"#),
        (r#"뾊"#, r#"뾊"#, r#"뾊"#, r#"뾊"#, r#"뾊"#),
        (r#"뾋"#, r#"뾋"#, r#"뾋"#, r#"뾋"#, r#"뾋"#),
        (r#"뾌"#, r#"뾌"#, r#"뾌"#, r#"뾌"#, r#"뾌"#),
        (r#"뾍"#, r#"뾍"#, r#"뾍"#, r#"뾍"#, r#"뾍"#),
        (r#"뾎"#, r#"뾎"#, r#"뾎"#, r#"뾎"#, r#"뾎"#),
        (r#"뾏"#, r#"뾏"#, r#"뾏"#, r#"뾏"#, r#"뾏"#),
        (r#"뾐"#, r#"뾐"#, r#"뾐"#, r#"뾐"#, r#"뾐"#),
        (r#"뾑"#, r#"뾑"#, r#"뾑"#, r#"뾑"#, r#"뾑"#),
        (r#"뾒"#, r#"뾒"#, r#"뾒"#, r#"뾒"#, r#"뾒"#),
        (r#"뾓"#, r#"뾓"#, r#"뾓"#, r#"뾓"#, r#"뾓"#),
        (r#"뾔"#, r#"뾔"#, r#"뾔"#, r#"뾔"#, r#"뾔"#),
        (r#"뾕"#, r#"뾕"#, r#"뾕"#, r#"뾕"#, r#"뾕"#),
        (r#"뾖"#, r#"뾖"#, r#"뾖"#, r#"뾖"#, r#"뾖"#),
        (r#"뾗"#, r#"뾗"#, r#"뾗"#, r#"뾗"#, r#"뾗"#),
        (r#"뾘"#, r#"뾘"#, r#"뾘"#, r#"뾘"#, r#"뾘"#),
        (r#"뾙"#, r#"뾙"#, r#"뾙"#, r#"뾙"#, r#"뾙"#),
        (r#"뾚"#, r#"뾚"#, r#"뾚"#, r#"뾚"#, r#"뾚"#),
        (r#"뾛"#, r#"뾛"#, r#"뾛"#, r#"뾛"#, r#"뾛"#),
        (r#"뾜"#, r#"뾜"#, r#"뾜"#, r#"뾜"#, r#"뾜"#),
        (r#"뾝"#, r#"뾝"#, r#"뾝"#, r#"뾝"#, r#"뾝"#),
        (r#"뾞"#, r#"뾞"#, r#"뾞"#, r#"뾞"#, r#"뾞"#),
        (r#"뾟"#, r#"뾟"#, r#"뾟"#, r#"뾟"#, r#"뾟"#),
        (r#"뾠"#, r#"뾠"#, r#"뾠"#, r#"뾠"#, r#"뾠"#),
        (r#"뾡"#, r#"뾡"#, r#"뾡"#, r#"뾡"#, r#"뾡"#),
        (r#"뾢"#, r#"뾢"#, r#"뾢"#, r#"뾢"#, r#"뾢"#),
        (r#"뾣"#, r#"뾣"#, r#"뾣"#, r#"뾣"#, r#"뾣"#),
        (r#"뾤"#, r#"뾤"#, r#"뾤"#, r#"뾤"#, r#"뾤"#),
        (r#"뾥"#, r#"뾥"#, r#"뾥"#, r#"뾥"#, r#"뾥"#),
        (r#"뾦"#, r#"뾦"#, r#"뾦"#, r#"뾦"#, r#"뾦"#),
        (r#"뾧"#, r#"뾧"#, r#"뾧"#, r#"뾧"#, r#"뾧"#),
        (r#"뾨"#, r#"뾨"#, r#"뾨"#, r#"뾨"#, r#"뾨"#),
        (r#"뾩"#, r#"뾩"#, r#"뾩"#, r#"뾩"#, r#"뾩"#),
        (r#"뾪"#, r#"뾪"#, r#"뾪"#, r#"뾪"#, r#"뾪"#),
        (r#"뾫"#, r#"뾫"#, r#"뾫"#, r#"뾫"#, r#"뾫"#),
        (r#"뾬"#, r#"뾬"#, r#"뾬"#, r#"뾬"#, r#"뾬"#),
        (r#"뾭"#, r#"뾭"#, r#"뾭"#, r#"뾭"#, r#"뾭"#),
        (r#"뾮"#, r#"뾮"#, r#"뾮"#, r#"뾮"#, r#"뾮"#),
        (r#"뾯"#, r#"뾯"#, r#"뾯"#, r#"뾯"#, r#"뾯"#),
        (r#"뾰"#, r#"뾰"#, r#"뾰"#, r#"뾰"#, r#"뾰"#),
        (r#"뾱"#, r#"뾱"#, r#"뾱"#, r#"뾱"#, r#"뾱"#),
        (r#"뾲"#, r#"뾲"#, r#"뾲"#, r#"뾲"#, r#"뾲"#),
        (r#"뾳"#, r#"뾳"#, r#"뾳"#, r#"뾳"#, r#"뾳"#),
        (r#"뾴"#, r#"뾴"#, r#"뾴"#, r#"뾴"#, r#"뾴"#),
        (r#"뾵"#, r#"뾵"#, r#"뾵"#, r#"뾵"#, r#"뾵"#),
        (r#"뾶"#, r#"뾶"#, r#"뾶"#, r#"뾶"#, r#"뾶"#),
        (r#"뾷"#, r#"뾷"#, r#"뾷"#, r#"뾷"#, r#"뾷"#),
        (r#"뾸"#, r#"뾸"#, r#"뾸"#, r#"뾸"#, r#"뾸"#),
        (r#"뾹"#, r#"뾹"#, r#"뾹"#, r#"뾹"#, r#"뾹"#),
        (r#"뾺"#, r#"뾺"#, r#"뾺"#, r#"뾺"#, r#"뾺"#),
        (r#"뾻"#, r#"뾻"#, r#"뾻"#, r#"뾻"#, r#"뾻"#),
        (r#"뾼"#, r#"뾼"#, r#"뾼"#, r#"뾼"#, r#"뾼"#),
        (r#"뾽"#, r#"뾽"#, r#"뾽"#, r#"뾽"#, r#"뾽"#),
        (r#"뾾"#, r#"뾾"#, r#"뾾"#, r#"뾾"#, r#"뾾"#),
        (r#"뾿"#, r#"뾿"#, r#"뾿"#, r#"뾿"#, r#"뾿"#),
        (r#"뿀"#, r#"뿀"#, r#"뿀"#, r#"뿀"#, r#"뿀"#),
        (r#"뿁"#, r#"뿁"#, r#"뿁"#, r#"뿁"#, r#"뿁"#),
        (r#"뿂"#, r#"뿂"#, r#"뿂"#, r#"뿂"#, r#"뿂"#),
        (r#"뿃"#, r#"뿃"#, r#"뿃"#, r#"뿃"#, r#"뿃"#),
        (r#"뿄"#, r#"뿄"#, r#"뿄"#, r#"뿄"#, r#"뿄"#),
        (r#"뿅"#, r#"뿅"#, r#"뿅"#, r#"뿅"#, r#"뿅"#),
        (r#"뿆"#, r#"뿆"#, r#"뿆"#, r#"뿆"#, r#"뿆"#),
        (r#"뿇"#, r#"뿇"#, r#"뿇"#, r#"뿇"#, r#"뿇"#),
        (r#"뿈"#, r#"뿈"#, r#"뿈"#, r#"뿈"#, r#"뿈"#),
        (r#"뿉"#, r#"뿉"#, r#"뿉"#, r#"뿉"#, r#"뿉"#),
        (r#"뿊"#, r#"뿊"#, r#"뿊"#, r#"뿊"#, r#"뿊"#),
        (r#"뿋"#, r#"뿋"#, r#"뿋"#, r#"뿋"#, r#"뿋"#),
        (r#"뿌"#, r#"뿌"#, r#"뿌"#, r#"뿌"#, r#"뿌"#),
        (r#"뿍"#, r#"뿍"#, r#"뿍"#, r#"뿍"#, r#"뿍"#),
        (r#"뿎"#, r#"뿎"#, r#"뿎"#, r#"뿎"#, r#"뿎"#),
        (r#"뿏"#, r#"뿏"#, r#"뿏"#, r#"뿏"#, r#"뿏"#),
        (r#"뿐"#, r#"뿐"#, r#"뿐"#, r#"뿐"#, r#"뿐"#),
        (r#"뿑"#, r#"뿑"#, r#"뿑"#, r#"뿑"#, r#"뿑"#),
        (r#"뿒"#, r#"뿒"#, r#"뿒"#, r#"뿒"#, r#"뿒"#),
        (r#"뿓"#, r#"뿓"#, r#"뿓"#, r#"뿓"#, r#"뿓"#),
        (r#"뿔"#, r#"뿔"#, r#"뿔"#, r#"뿔"#, r#"뿔"#),
        (r#"뿕"#, r#"뿕"#, r#"뿕"#, r#"뿕"#, r#"뿕"#),
        (r#"뿖"#, r#"뿖"#, r#"뿖"#, r#"뿖"#, r#"뿖"#),
        (r#"뿗"#, r#"뿗"#, r#"뿗"#, r#"뿗"#, r#"뿗"#),
        (r#"뿘"#, r#"뿘"#, r#"뿘"#, r#"뿘"#, r#"뿘"#),
        (r#"뿙"#, r#"뿙"#, r#"뿙"#, r#"뿙"#, r#"뿙"#),
        (r#"뿚"#, r#"뿚"#, r#"뿚"#, r#"뿚"#, r#"뿚"#),
        (r#"뿛"#, r#"뿛"#, r#"뿛"#, r#"뿛"#, r#"뿛"#),
        (r#"뿜"#, r#"뿜"#, r#"뿜"#, r#"뿜"#, r#"뿜"#),
        (r#"뿝"#, r#"뿝"#, r#"뿝"#, r#"뿝"#, r#"뿝"#),
        (r#"뿞"#, r#"뿞"#, r#"뿞"#, r#"뿞"#, r#"뿞"#),
        (r#"뿟"#, r#"뿟"#, r#"뿟"#, r#"뿟"#, r#"뿟"#),
        (r#"뿠"#, r#"뿠"#, r#"뿠"#, r#"뿠"#, r#"뿠"#),
        (r#"뿡"#, r#"뿡"#, r#"뿡"#, r#"뿡"#, r#"뿡"#),
        (r#"뿢"#, r#"뿢"#, r#"뿢"#, r#"뿢"#, r#"뿢"#),
        (r#"뿣"#, r#"뿣"#, r#"뿣"#, r#"뿣"#, r#"뿣"#),
        (r#"뿤"#, r#"뿤"#, r#"뿤"#, r#"뿤"#, r#"뿤"#),
        (r#"뿥"#, r#"뿥"#, r#"뿥"#, r#"뿥"#, r#"뿥"#),
        (r#"뿦"#, r#"뿦"#, r#"뿦"#, r#"뿦"#, r#"뿦"#),
        (r#"뿧"#, r#"뿧"#, r#"뿧"#, r#"뿧"#, r#"뿧"#),
        (r#"뿨"#, r#"뿨"#, r#"뿨"#, r#"뿨"#, r#"뿨"#),
        (r#"뿩"#, r#"뿩"#, r#"뿩"#, r#"뿩"#, r#"뿩"#),
        (r#"뿪"#, r#"뿪"#, r#"뿪"#, r#"뿪"#, r#"뿪"#),
        (r#"뿫"#, r#"뿫"#, r#"뿫"#, r#"뿫"#, r#"뿫"#),
        (r#"뿬"#, r#"뿬"#, r#"뿬"#, r#"뿬"#, r#"뿬"#),
        (r#"뿭"#, r#"뿭"#, r#"뿭"#, r#"뿭"#, r#"뿭"#),
        (r#"뿮"#, r#"뿮"#, r#"뿮"#, r#"뿮"#, r#"뿮"#),
        (r#"뿯"#, r#"뿯"#, r#"뿯"#, r#"뿯"#, r#"뿯"#),
        (r#"뿰"#, r#"뿰"#, r#"뿰"#, r#"뿰"#, r#"뿰"#),
        (r#"뿱"#, r#"뿱"#, r#"뿱"#, r#"뿱"#, r#"뿱"#),
        (r#"뿲"#, r#"뿲"#, r#"뿲"#, r#"뿲"#, r#"뿲"#),
        (r#"뿳"#, r#"뿳"#, r#"뿳"#, r#"뿳"#, r#"뿳"#),
        (r#"뿴"#, r#"뿴"#, r#"뿴"#, r#"뿴"#, r#"뿴"#),
        (r#"뿵"#, r#"뿵"#, r#"뿵"#, r#"뿵"#, r#"뿵"#),
        (r#"뿶"#, r#"뿶"#, r#"뿶"#, r#"뿶"#, r#"뿶"#),
        (r#"뿷"#, r#"뿷"#, r#"뿷"#, r#"뿷"#, r#"뿷"#),
        (r#"뿸"#, r#"뿸"#, r#"뿸"#, r#"뿸"#, r#"뿸"#),
        (r#"뿹"#, r#"뿹"#, r#"뿹"#, r#"뿹"#, r#"뿹"#),
        (r#"뿺"#, r#"뿺"#, r#"뿺"#, r#"뿺"#, r#"뿺"#),
        (r#"뿻"#, r#"뿻"#, r#"뿻"#, r#"뿻"#, r#"뿻"#),
        (r#"뿼"#, r#"뿼"#, r#"뿼"#, r#"뿼"#, r#"뿼"#),
        (r#"뿽"#, r#"뿽"#, r#"뿽"#, r#"뿽"#, r#"뿽"#),
        (r#"뿾"#, r#"뿾"#, r#"뿾"#, r#"뿾"#, r#"뿾"#),
        (r#"뿿"#, r#"뿿"#, r#"뿿"#, r#"뿿"#, r#"뿿"#),
        (r#"쀀"#, r#"쀀"#, r#"쀀"#, r#"쀀"#, r#"쀀"#),
        (r#"쀁"#, r#"쀁"#, r#"쀁"#, r#"쀁"#, r#"쀁"#),
        (r#"쀂"#, r#"쀂"#, r#"쀂"#, r#"쀂"#, r#"쀂"#),
        (r#"쀃"#, r#"쀃"#, r#"쀃"#, r#"쀃"#, r#"쀃"#),
        (r#"쀄"#, r#"쀄"#, r#"쀄"#, r#"쀄"#, r#"쀄"#),
        (r#"쀅"#, r#"쀅"#, r#"쀅"#, r#"쀅"#, r#"쀅"#),
        (r#"쀆"#, r#"쀆"#, r#"쀆"#, r#"쀆"#, r#"쀆"#),
        (r#"쀇"#, r#"쀇"#, r#"쀇"#, r#"쀇"#, r#"쀇"#),
        (r#"쀈"#, r#"쀈"#, r#"쀈"#, r#"쀈"#, r#"쀈"#),
        (r#"쀉"#, r#"쀉"#, r#"쀉"#, r#"쀉"#, r#"쀉"#),
        (r#"쀊"#, r#"쀊"#, r#"쀊"#, r#"쀊"#, r#"쀊"#),
        (r#"쀋"#, r#"쀋"#, r#"쀋"#, r#"쀋"#, r#"쀋"#),
        (r#"쀌"#, r#"쀌"#, r#"쀌"#, r#"쀌"#, r#"쀌"#),
        (r#"쀍"#, r#"쀍"#, r#"쀍"#, r#"쀍"#, r#"쀍"#),
        (r#"쀎"#, r#"쀎"#, r#"쀎"#, r#"쀎"#, r#"쀎"#),
        (r#"쀏"#, r#"쀏"#, r#"쀏"#, r#"쀏"#, r#"쀏"#),
        (r#"쀐"#, r#"쀐"#, r#"쀐"#, r#"쀐"#, r#"쀐"#),
        (r#"쀑"#, r#"쀑"#, r#"쀑"#, r#"쀑"#, r#"쀑"#),
        (r#"쀒"#, r#"쀒"#, r#"쀒"#, r#"쀒"#, r#"쀒"#),
        (r#"쀓"#, r#"쀓"#, r#"쀓"#, r#"쀓"#, r#"쀓"#),
        (r#"쀔"#, r#"쀔"#, r#"쀔"#, r#"쀔"#, r#"쀔"#),
        (r#"쀕"#, r#"쀕"#, r#"쀕"#, r#"쀕"#, r#"쀕"#),
        (r#"쀖"#, r#"쀖"#, r#"쀖"#, r#"쀖"#, r#"쀖"#),
        (r#"쀗"#, r#"쀗"#, r#"쀗"#, r#"쀗"#, r#"쀗"#),
        (r#"쀘"#, r#"쀘"#, r#"쀘"#, r#"쀘"#, r#"쀘"#),
        (r#"쀙"#, r#"쀙"#, r#"쀙"#, r#"쀙"#, r#"쀙"#),
        (r#"쀚"#, r#"쀚"#, r#"쀚"#, r#"쀚"#, r#"쀚"#),
        (r#"쀛"#, r#"쀛"#, r#"쀛"#, r#"쀛"#, r#"쀛"#),
        (r#"쀜"#, r#"쀜"#, r#"쀜"#, r#"쀜"#, r#"쀜"#),
        (r#"쀝"#, r#"쀝"#, r#"쀝"#, r#"쀝"#, r#"쀝"#),
        (r#"쀞"#, r#"쀞"#, r#"쀞"#, r#"쀞"#, r#"쀞"#),
        (r#"쀟"#, r#"쀟"#, r#"쀟"#, r#"쀟"#, r#"쀟"#),
        (r#"쀠"#, r#"쀠"#, r#"쀠"#, r#"쀠"#, r#"쀠"#),
        (r#"쀡"#, r#"쀡"#, r#"쀡"#, r#"쀡"#, r#"쀡"#),
        (r#"쀢"#, r#"쀢"#, r#"쀢"#, r#"쀢"#, r#"쀢"#),
        (r#"쀣"#, r#"쀣"#, r#"쀣"#, r#"쀣"#, r#"쀣"#),
        (r#"쀤"#, r#"쀤"#, r#"쀤"#, r#"쀤"#, r#"쀤"#),
        (r#"쀥"#, r#"쀥"#, r#"쀥"#, r#"쀥"#, r#"쀥"#),
        (r#"쀦"#, r#"쀦"#, r#"쀦"#, r#"쀦"#, r#"쀦"#),
        (r#"쀧"#, r#"쀧"#, r#"쀧"#, r#"쀧"#, r#"쀧"#),
        (r#"쀨"#, r#"쀨"#, r#"쀨"#, r#"쀨"#, r#"쀨"#),
        (r#"쀩"#, r#"쀩"#, r#"쀩"#, r#"쀩"#, r#"쀩"#),
        (r#"쀪"#, r#"쀪"#, r#"쀪"#, r#"쀪"#, r#"쀪"#),
        (r#"쀫"#, r#"쀫"#, r#"쀫"#, r#"쀫"#, r#"쀫"#),
        (r#"쀬"#, r#"쀬"#, r#"쀬"#, r#"쀬"#, r#"쀬"#),
        (r#"쀭"#, r#"쀭"#, r#"쀭"#, r#"쀭"#, r#"쀭"#),
        (r#"쀮"#, r#"쀮"#, r#"쀮"#, r#"쀮"#, r#"쀮"#),
        (r#"쀯"#, r#"쀯"#, r#"쀯"#, r#"쀯"#, r#"쀯"#),
        (r#"쀰"#, r#"쀰"#, r#"쀰"#, r#"쀰"#, r#"쀰"#),
        (r#"쀱"#, r#"쀱"#, r#"쀱"#, r#"쀱"#, r#"쀱"#),
        (r#"쀲"#, r#"쀲"#, r#"쀲"#, r#"쀲"#, r#"쀲"#),
        (r#"쀳"#, r#"쀳"#, r#"쀳"#, r#"쀳"#, r#"쀳"#),
        (r#"쀴"#, r#"쀴"#, r#"쀴"#, r#"쀴"#, r#"쀴"#),
        (r#"쀵"#, r#"쀵"#, r#"쀵"#, r#"쀵"#, r#"쀵"#),
        (r#"쀶"#, r#"쀶"#, r#"쀶"#, r#"쀶"#, r#"쀶"#),
        (r#"쀷"#, r#"쀷"#, r#"쀷"#, r#"쀷"#, r#"쀷"#),
        (r#"쀸"#, r#"쀸"#, r#"쀸"#, r#"쀸"#, r#"쀸"#),
        (r#"쀹"#, r#"쀹"#, r#"쀹"#, r#"쀹"#, r#"쀹"#),
        (r#"쀺"#, r#"쀺"#, r#"쀺"#, r#"쀺"#, r#"쀺"#),
        (r#"쀻"#, r#"쀻"#, r#"쀻"#, r#"쀻"#, r#"쀻"#),
        (r#"쀼"#, r#"쀼"#, r#"쀼"#, r#"쀼"#, r#"쀼"#),
        (r#"쀽"#, r#"쀽"#, r#"쀽"#, r#"쀽"#, r#"쀽"#),
        (r#"쀾"#, r#"쀾"#, r#"쀾"#, r#"쀾"#, r#"쀾"#),
        (r#"쀿"#, r#"쀿"#, r#"쀿"#, r#"쀿"#, r#"쀿"#),
        (r#"쁀"#, r#"쁀"#, r#"쁀"#, r#"쁀"#, r#"쁀"#),
        (r#"쁁"#, r#"쁁"#, r#"쁁"#, r#"쁁"#, r#"쁁"#),
        (r#"쁂"#, r#"쁂"#, r#"쁂"#, r#"쁂"#, r#"쁂"#),
        (r#"쁃"#, r#"쁃"#, r#"쁃"#, r#"쁃"#, r#"쁃"#),
        (r#"쁄"#, r#"쁄"#, r#"쁄"#, r#"쁄"#, r#"쁄"#),
        (r#"쁅"#, r#"쁅"#, r#"쁅"#, r#"쁅"#, r#"쁅"#),
        (r#"쁆"#, r#"쁆"#, r#"쁆"#, r#"쁆"#, r#"쁆"#),
        (r#"쁇"#, r#"쁇"#, r#"쁇"#, r#"쁇"#, r#"쁇"#),
        (r#"쁈"#, r#"쁈"#, r#"쁈"#, r#"쁈"#, r#"쁈"#),
        (r#"쁉"#, r#"쁉"#, r#"쁉"#, r#"쁉"#, r#"쁉"#),
        (r#"쁊"#, r#"쁊"#, r#"쁊"#, r#"쁊"#, r#"쁊"#),
        (r#"쁋"#, r#"쁋"#, r#"쁋"#, r#"쁋"#, r#"쁋"#),
        (r#"쁌"#, r#"쁌"#, r#"쁌"#, r#"쁌"#, r#"쁌"#),
        (r#"쁍"#, r#"쁍"#, r#"쁍"#, r#"쁍"#, r#"쁍"#),
        (r#"쁎"#, r#"쁎"#, r#"쁎"#, r#"쁎"#, r#"쁎"#),
        (r#"쁏"#, r#"쁏"#, r#"쁏"#, r#"쁏"#, r#"쁏"#),
        (r#"쁐"#, r#"쁐"#, r#"쁐"#, r#"쁐"#, r#"쁐"#),
        (r#"쁑"#, r#"쁑"#, r#"쁑"#, r#"쁑"#, r#"쁑"#),
        (r#"쁒"#, r#"쁒"#, r#"쁒"#, r#"쁒"#, r#"쁒"#),
        (r#"쁓"#, r#"쁓"#, r#"쁓"#, r#"쁓"#, r#"쁓"#),
        (r#"쁔"#, r#"쁔"#, r#"쁔"#, r#"쁔"#, r#"쁔"#),
        (r#"쁕"#, r#"쁕"#, r#"쁕"#, r#"쁕"#, r#"쁕"#),
        (r#"쁖"#, r#"쁖"#, r#"쁖"#, r#"쁖"#, r#"쁖"#),
        (r#"쁗"#, r#"쁗"#, r#"쁗"#, r#"쁗"#, r#"쁗"#),
        (r#"쁘"#, r#"쁘"#, r#"쁘"#, r#"쁘"#, r#"쁘"#),
        (r#"쁙"#, r#"쁙"#, r#"쁙"#, r#"쁙"#, r#"쁙"#),
        (r#"쁚"#, r#"쁚"#, r#"쁚"#, r#"쁚"#, r#"쁚"#),
        (r#"쁛"#, r#"쁛"#, r#"쁛"#, r#"쁛"#, r#"쁛"#),
        (r#"쁜"#, r#"쁜"#, r#"쁜"#, r#"쁜"#, r#"쁜"#),
        (r#"쁝"#, r#"쁝"#, r#"쁝"#, r#"쁝"#, r#"쁝"#),
        (r#"쁞"#, r#"쁞"#, r#"쁞"#, r#"쁞"#, r#"쁞"#),
        (r#"쁟"#, r#"쁟"#, r#"쁟"#, r#"쁟"#, r#"쁟"#),
        (r#"쁠"#, r#"쁠"#, r#"쁠"#, r#"쁠"#, r#"쁠"#),
        (r#"쁡"#, r#"쁡"#, r#"쁡"#, r#"쁡"#, r#"쁡"#),
        (r#"쁢"#, r#"쁢"#, r#"쁢"#, r#"쁢"#, r#"쁢"#),
        (r#"쁣"#, r#"쁣"#, r#"쁣"#, r#"쁣"#, r#"쁣"#),
        (r#"쁤"#, r#"쁤"#, r#"쁤"#, r#"쁤"#, r#"쁤"#),
        (r#"쁥"#, r#"쁥"#, r#"쁥"#, r#"쁥"#, r#"쁥"#),
        (r#"쁦"#, r#"쁦"#, r#"쁦"#, r#"쁦"#, r#"쁦"#),
        (r#"쁧"#, r#"쁧"#, r#"쁧"#, r#"쁧"#, r#"쁧"#),
        (r#"쁨"#, r#"쁨"#, r#"쁨"#, r#"쁨"#, r#"쁨"#),
        (r#"쁩"#, r#"쁩"#, r#"쁩"#, r#"쁩"#, r#"쁩"#),
        (r#"쁪"#, r#"쁪"#, r#"쁪"#, r#"쁪"#, r#"쁪"#),
        (r#"쁫"#, r#"쁫"#, r#"쁫"#, r#"쁫"#, r#"쁫"#),
        (r#"쁬"#, r#"쁬"#, r#"쁬"#, r#"쁬"#, r#"쁬"#),
        (r#"쁭"#, r#"쁭"#, r#"쁭"#, r#"쁭"#, r#"쁭"#),
        (r#"쁮"#, r#"쁮"#, r#"쁮"#, r#"쁮"#, r#"쁮"#),
        (r#"쁯"#, r#"쁯"#, r#"쁯"#, r#"쁯"#, r#"쁯"#),
        (r#"쁰"#, r#"쁰"#, r#"쁰"#, r#"쁰"#, r#"쁰"#),
        (r#"쁱"#, r#"쁱"#, r#"쁱"#, r#"쁱"#, r#"쁱"#),
        (r#"쁲"#, r#"쁲"#, r#"쁲"#, r#"쁲"#, r#"쁲"#),
        (r#"쁳"#, r#"쁳"#, r#"쁳"#, r#"쁳"#, r#"쁳"#),
        (r#"쁴"#, r#"쁴"#, r#"쁴"#, r#"쁴"#, r#"쁴"#),
        (r#"쁵"#, r#"쁵"#, r#"쁵"#, r#"쁵"#, r#"쁵"#),
        (r#"쁶"#, r#"쁶"#, r#"쁶"#, r#"쁶"#, r#"쁶"#),
        (r#"쁷"#, r#"쁷"#, r#"쁷"#, r#"쁷"#, r#"쁷"#),
        (r#"쁸"#, r#"쁸"#, r#"쁸"#, r#"쁸"#, r#"쁸"#),
        (r#"쁹"#, r#"쁹"#, r#"쁹"#, r#"쁹"#, r#"쁹"#),
        (r#"쁺"#, r#"쁺"#, r#"쁺"#, r#"쁺"#, r#"쁺"#),
        (r#"쁻"#, r#"쁻"#, r#"쁻"#, r#"쁻"#, r#"쁻"#),
        (r#"쁼"#, r#"쁼"#, r#"쁼"#, r#"쁼"#, r#"쁼"#),
        (r#"쁽"#, r#"쁽"#, r#"쁽"#, r#"쁽"#, r#"쁽"#),
        (r#"쁾"#, r#"쁾"#, r#"쁾"#, r#"쁾"#, r#"쁾"#),
        (r#"쁿"#, r#"쁿"#, r#"쁿"#, r#"쁿"#, r#"쁿"#),
        (r#"삀"#, r#"삀"#, r#"삀"#, r#"삀"#, r#"삀"#),
        (r#"삁"#, r#"삁"#, r#"삁"#, r#"삁"#, r#"삁"#),
        (r#"삂"#, r#"삂"#, r#"삂"#, r#"삂"#, r#"삂"#),
        (r#"삃"#, r#"삃"#, r#"삃"#, r#"삃"#, r#"삃"#),
        (r#"삄"#, r#"삄"#, r#"삄"#, r#"삄"#, r#"삄"#),
        (r#"삅"#, r#"삅"#, r#"삅"#, r#"삅"#, r#"삅"#),
        (r#"삆"#, r#"삆"#, r#"삆"#, r#"삆"#, r#"삆"#),
        (r#"삇"#, r#"삇"#, r#"삇"#, r#"삇"#, r#"삇"#),
        (r#"삈"#, r#"삈"#, r#"삈"#, r#"삈"#, r#"삈"#),
        (r#"삉"#, r#"삉"#, r#"삉"#, r#"삉"#, r#"삉"#),
        (r#"삊"#, r#"삊"#, r#"삊"#, r#"삊"#, r#"삊"#),
        (r#"삋"#, r#"삋"#, r#"삋"#, r#"삋"#, r#"삋"#),
        (r#"삌"#, r#"삌"#, r#"삌"#, r#"삌"#, r#"삌"#),
        (r#"삍"#, r#"삍"#, r#"삍"#, r#"삍"#, r#"삍"#),
        (r#"삎"#, r#"삎"#, r#"삎"#, r#"삎"#, r#"삎"#),
        (r#"삏"#, r#"삏"#, r#"삏"#, r#"삏"#, r#"삏"#),
        (r#"삐"#, r#"삐"#, r#"삐"#, r#"삐"#, r#"삐"#),
        (r#"삑"#, r#"삑"#, r#"삑"#, r#"삑"#, r#"삑"#),
        (r#"삒"#, r#"삒"#, r#"삒"#, r#"삒"#, r#"삒"#),
        (r#"삓"#, r#"삓"#, r#"삓"#, r#"삓"#, r#"삓"#),
        (r#"삔"#, r#"삔"#, r#"삔"#, r#"삔"#, r#"삔"#),
        (r#"삕"#, r#"삕"#, r#"삕"#, r#"삕"#, r#"삕"#),
        (r#"삖"#, r#"삖"#, r#"삖"#, r#"삖"#, r#"삖"#),
        (r#"삗"#, r#"삗"#, r#"삗"#, r#"삗"#, r#"삗"#),
        (r#"삘"#, r#"삘"#, r#"삘"#, r#"삘"#, r#"삘"#),
        (r#"삙"#, r#"삙"#, r#"삙"#, r#"삙"#, r#"삙"#),
        (r#"삚"#, r#"삚"#, r#"삚"#, r#"삚"#, r#"삚"#),
        (r#"삛"#, r#"삛"#, r#"삛"#, r#"삛"#, r#"삛"#),
        (r#"삜"#, r#"삜"#, r#"삜"#, r#"삜"#, r#"삜"#),
        (r#"삝"#, r#"삝"#, r#"삝"#, r#"삝"#, r#"삝"#),
        (r#"삞"#, r#"삞"#, r#"삞"#, r#"삞"#, r#"삞"#),
        (r#"삟"#, r#"삟"#, r#"삟"#, r#"삟"#, r#"삟"#),
        (r#"삠"#, r#"삠"#, r#"삠"#, r#"삠"#, r#"삠"#),
        (r#"삡"#, r#"삡"#, r#"삡"#, r#"삡"#, r#"삡"#),
        (r#"삢"#, r#"삢"#, r#"삢"#, r#"삢"#, r#"삢"#),
        (r#"삣"#, r#"삣"#, r#"삣"#, r#"삣"#, r#"삣"#),
        (r#"삤"#, r#"삤"#, r#"삤"#, r#"삤"#, r#"삤"#),
        (r#"삥"#, r#"삥"#, r#"삥"#, r#"삥"#, r#"삥"#),
        (r#"삦"#, r#"삦"#, r#"삦"#, r#"삦"#, r#"삦"#),
        (r#"삧"#, r#"삧"#, r#"삧"#, r#"삧"#, r#"삧"#),
        (r#"삨"#, r#"삨"#, r#"삨"#, r#"삨"#, r#"삨"#),
        (r#"삩"#, r#"삩"#, r#"삩"#, r#"삩"#, r#"삩"#),
        (r#"삪"#, r#"삪"#, r#"삪"#, r#"삪"#, r#"삪"#),
        (r#"삫"#, r#"삫"#, r#"삫"#, r#"삫"#, r#"삫"#),
        (r#"사"#, r#"사"#, r#"사"#, r#"사"#, r#"사"#),
        (r#"삭"#, r#"삭"#, r#"삭"#, r#"삭"#, r#"삭"#),
        (r#"삮"#, r#"삮"#, r#"삮"#, r#"삮"#, r#"삮"#),
        (r#"삯"#, r#"삯"#, r#"삯"#, r#"삯"#, r#"삯"#),
        (r#"산"#, r#"산"#, r#"산"#, r#"산"#, r#"산"#),
        (r#"삱"#, r#"삱"#, r#"삱"#, r#"삱"#, r#"삱"#),
        (r#"삲"#, r#"삲"#, r#"삲"#, r#"삲"#, r#"삲"#),
        (r#"삳"#, r#"삳"#, r#"삳"#, r#"삳"#, r#"삳"#),
        (r#"살"#, r#"살"#, r#"살"#, r#"살"#, r#"살"#),
        (r#"삵"#, r#"삵"#, r#"삵"#, r#"삵"#, r#"삵"#),
        (r#"삶"#, r#"삶"#, r#"삶"#, r#"삶"#, r#"삶"#),
        (r#"삷"#, r#"삷"#, r#"삷"#, r#"삷"#, r#"삷"#),
        (r#"삸"#, r#"삸"#, r#"삸"#, r#"삸"#, r#"삸"#),
        (r#"삹"#, r#"삹"#, r#"삹"#, r#"삹"#, r#"삹"#),
        (r#"삺"#, r#"삺"#, r#"삺"#, r#"삺"#, r#"삺"#),
        (r#"삻"#, r#"삻"#, r#"삻"#, r#"삻"#, r#"삻"#),
        (r#"삼"#, r#"삼"#, r#"삼"#, r#"삼"#, r#"삼"#),
        (r#"삽"#, r#"삽"#, r#"삽"#, r#"삽"#, r#"삽"#),
        (r#"삾"#, r#"삾"#, r#"삾"#, r#"삾"#, r#"삾"#),
        (r#"삿"#, r#"삿"#, r#"삿"#, r#"삿"#, r#"삿"#),
        (r#"샀"#, r#"샀"#, r#"샀"#, r#"샀"#, r#"샀"#),
        (r#"상"#, r#"상"#, r#"상"#, r#"상"#, r#"상"#),
        (r#"샂"#, r#"샂"#, r#"샂"#, r#"샂"#, r#"샂"#),
        (r#"샃"#, r#"샃"#, r#"샃"#, r#"샃"#, r#"샃"#),
        (r#"샄"#, r#"샄"#, r#"샄"#, r#"샄"#, r#"샄"#),
        (r#"샅"#, r#"샅"#, r#"샅"#, r#"샅"#, r#"샅"#),
        (r#"샆"#, r#"샆"#, r#"샆"#, r#"샆"#, r#"샆"#),
        (r#"샇"#, r#"샇"#, r#"샇"#, r#"샇"#, r#"샇"#),
        (r#"새"#, r#"새"#, r#"새"#, r#"새"#, r#"새"#),
        (r#"색"#, r#"색"#, r#"색"#, r#"색"#, r#"색"#),
        (r#"샊"#, r#"샊"#, r#"샊"#, r#"샊"#, r#"샊"#),
        (r#"샋"#, r#"샋"#, r#"샋"#, r#"샋"#, r#"샋"#),
        (r#"샌"#, r#"샌"#, r#"샌"#, r#"샌"#, r#"샌"#),
        (r#"샍"#, r#"샍"#, r#"샍"#, r#"샍"#, r#"샍"#),
        (r#"샎"#, r#"샎"#, r#"샎"#, r#"샎"#, r#"샎"#),
        (r#"샏"#, r#"샏"#, r#"샏"#, r#"샏"#, r#"샏"#),
        (r#"샐"#, r#"샐"#, r#"샐"#, r#"샐"#, r#"샐"#),
        (r#"샑"#, r#"샑"#, r#"샑"#, r#"샑"#, r#"샑"#),
        (r#"샒"#, r#"샒"#, r#"샒"#, r#"샒"#, r#"샒"#),
        (r#"샓"#, r#"샓"#, r#"샓"#, r#"샓"#, r#"샓"#),
        (r#"샔"#, r#"샔"#, r#"샔"#, r#"샔"#, r#"샔"#),
        (r#"샕"#, r#"샕"#, r#"샕"#, r#"샕"#, r#"샕"#),
        (r#"샖"#, r#"샖"#, r#"샖"#, r#"샖"#, r#"샖"#),
        (r#"샗"#, r#"샗"#, r#"샗"#, r#"샗"#, r#"샗"#),
        (r#"샘"#, r#"샘"#, r#"샘"#, r#"샘"#, r#"샘"#),
        (r#"샙"#, r#"샙"#, r#"샙"#, r#"샙"#, r#"샙"#),
        (r#"샚"#, r#"샚"#, r#"샚"#, r#"샚"#, r#"샚"#),
        (r#"샛"#, r#"샛"#, r#"샛"#, r#"샛"#, r#"샛"#),
        (r#"샜"#, r#"샜"#, r#"샜"#, r#"샜"#, r#"샜"#),
        (r#"생"#, r#"생"#, r#"생"#, r#"생"#, r#"생"#),
        (r#"샞"#, r#"샞"#, r#"샞"#, r#"샞"#, r#"샞"#),
        (r#"샟"#, r#"샟"#, r#"샟"#, r#"샟"#, r#"샟"#),
        (r#"샠"#, r#"샠"#, r#"샠"#, r#"샠"#, r#"샠"#),
        (r#"샡"#, r#"샡"#, r#"샡"#, r#"샡"#, r#"샡"#),
        (r#"샢"#, r#"샢"#, r#"샢"#, r#"샢"#, r#"샢"#),
        (r#"샣"#, r#"샣"#, r#"샣"#, r#"샣"#, r#"샣"#),
        (r#"샤"#, r#"샤"#, r#"샤"#, r#"샤"#, r#"샤"#),
        (r#"샥"#, r#"샥"#, r#"샥"#, r#"샥"#, r#"샥"#),
        (r#"샦"#, r#"샦"#, r#"샦"#, r#"샦"#, r#"샦"#),
        (r#"샧"#, r#"샧"#, r#"샧"#, r#"샧"#, r#"샧"#),
        (r#"샨"#, r#"샨"#, r#"샨"#, r#"샨"#, r#"샨"#),
        (r#"샩"#, r#"샩"#, r#"샩"#, r#"샩"#, r#"샩"#),
        (r#"샪"#, r#"샪"#, r#"샪"#, r#"샪"#, r#"샪"#),
        (r#"샫"#, r#"샫"#, r#"샫"#, r#"샫"#, r#"샫"#),
        (r#"샬"#, r#"샬"#, r#"샬"#, r#"샬"#, r#"샬"#),
        (r#"샭"#, r#"샭"#, r#"샭"#, r#"샭"#, r#"샭"#),
        (r#"샮"#, r#"샮"#, r#"샮"#, r#"샮"#, r#"샮"#),
        (r#"샯"#, r#"샯"#, r#"샯"#, r#"샯"#, r#"샯"#),
        (r#"샰"#, r#"샰"#, r#"샰"#, r#"샰"#, r#"샰"#),
        (r#"샱"#, r#"샱"#, r#"샱"#, r#"샱"#, r#"샱"#),
        (r#"샲"#, r#"샲"#, r#"샲"#, r#"샲"#, r#"샲"#),
        (r#"샳"#, r#"샳"#, r#"샳"#, r#"샳"#, r#"샳"#),
        (r#"샴"#, r#"샴"#, r#"샴"#, r#"샴"#, r#"샴"#),
        (r#"샵"#, r#"샵"#, r#"샵"#, r#"샵"#, r#"샵"#),
        (r#"샶"#, r#"샶"#, r#"샶"#, r#"샶"#, r#"샶"#),
        (r#"샷"#, r#"샷"#, r#"샷"#, r#"샷"#, r#"샷"#),
        (r#"샸"#, r#"샸"#, r#"샸"#, r#"샸"#, r#"샸"#),
        (r#"샹"#, r#"샹"#, r#"샹"#, r#"샹"#, r#"샹"#),
        (r#"샺"#, r#"샺"#, r#"샺"#, r#"샺"#, r#"샺"#),
        (r#"샻"#, r#"샻"#, r#"샻"#, r#"샻"#, r#"샻"#),
        (r#"샼"#, r#"샼"#, r#"샼"#, r#"샼"#, r#"샼"#),
        (r#"샽"#, r#"샽"#, r#"샽"#, r#"샽"#, r#"샽"#),
        (r#"샾"#, r#"샾"#, r#"샾"#, r#"샾"#, r#"샾"#),
        (r#"샿"#, r#"샿"#, r#"샿"#, r#"샿"#, r#"샿"#),
        (r#"섀"#, r#"섀"#, r#"섀"#, r#"섀"#, r#"섀"#),
        (r#"섁"#, r#"섁"#, r#"섁"#, r#"섁"#, r#"섁"#),
        (r#"섂"#, r#"섂"#, r#"섂"#, r#"섂"#, r#"섂"#),
        (r#"섃"#, r#"섃"#, r#"섃"#, r#"섃"#, r#"섃"#),
        (r#"섄"#, r#"섄"#, r#"섄"#, r#"섄"#, r#"섄"#),
        (r#"섅"#, r#"섅"#, r#"섅"#, r#"섅"#, r#"섅"#),
        (r#"섆"#, r#"섆"#, r#"섆"#, r#"섆"#, r#"섆"#),
        (r#"섇"#, r#"섇"#, r#"섇"#, r#"섇"#, r#"섇"#),
        (r#"섈"#, r#"섈"#, r#"섈"#, r#"섈"#, r#"섈"#),
        (r#"섉"#, r#"섉"#, r#"섉"#, r#"섉"#, r#"섉"#),
        (r#"섊"#, r#"섊"#, r#"섊"#, r#"섊"#, r#"섊"#),
        (r#"섋"#, r#"섋"#, r#"섋"#, r#"섋"#, r#"섋"#),
        (r#"섌"#, r#"섌"#, r#"섌"#, r#"섌"#, r#"섌"#),
        (r#"섍"#, r#"섍"#, r#"섍"#, r#"섍"#, r#"섍"#),
        (r#"섎"#, r#"섎"#, r#"섎"#, r#"섎"#, r#"섎"#),
        (r#"섏"#, r#"섏"#, r#"섏"#, r#"섏"#, r#"섏"#),
        (r#"섐"#, r#"섐"#, r#"섐"#, r#"섐"#, r#"섐"#),
        (r#"섑"#, r#"섑"#, r#"섑"#, r#"섑"#, r#"섑"#),
        (r#"섒"#, r#"섒"#, r#"섒"#, r#"섒"#, r#"섒"#),
        (r#"섓"#, r#"섓"#, r#"섓"#, r#"섓"#, r#"섓"#),
        (r#"섔"#, r#"섔"#, r#"섔"#, r#"섔"#, r#"섔"#),
        (r#"섕"#, r#"섕"#, r#"섕"#, r#"섕"#, r#"섕"#),
        (r#"섖"#, r#"섖"#, r#"섖"#, r#"섖"#, r#"섖"#),
        (r#"섗"#, r#"섗"#, r#"섗"#, r#"섗"#, r#"섗"#),
        (r#"섘"#, r#"섘"#, r#"섘"#, r#"섘"#, r#"섘"#),
        (r#"섙"#, r#"섙"#, r#"섙"#, r#"섙"#, r#"섙"#),
        (r#"섚"#, r#"섚"#, r#"섚"#, r#"섚"#, r#"섚"#),
        (r#"섛"#, r#"섛"#, r#"섛"#, r#"섛"#, r#"섛"#),
        (r#"서"#, r#"서"#, r#"서"#, r#"서"#, r#"서"#),
        (r#"석"#, r#"석"#, r#"석"#, r#"석"#, r#"석"#),
        (r#"섞"#, r#"섞"#, r#"섞"#, r#"섞"#, r#"섞"#),
        (r#"섟"#, r#"섟"#, r#"섟"#, r#"섟"#, r#"섟"#),
        (r#"선"#, r#"선"#, r#"선"#, r#"선"#, r#"선"#),
        (r#"섡"#, r#"섡"#, r#"섡"#, r#"섡"#, r#"섡"#),
        (r#"섢"#, r#"섢"#, r#"섢"#, r#"섢"#, r#"섢"#),
        (r#"섣"#, r#"섣"#, r#"섣"#, r#"섣"#, r#"섣"#),
        (r#"설"#, r#"설"#, r#"설"#, r#"설"#, r#"설"#),
        (r#"섥"#, r#"섥"#, r#"섥"#, r#"섥"#, r#"섥"#),
        (r#"섦"#, r#"섦"#, r#"섦"#, r#"섦"#, r#"섦"#),
        (r#"섧"#, r#"섧"#, r#"섧"#, r#"섧"#, r#"섧"#),
        (r#"섨"#, r#"섨"#, r#"섨"#, r#"섨"#, r#"섨"#),
        (r#"섩"#, r#"섩"#, r#"섩"#, r#"섩"#, r#"섩"#),
        (r#"섪"#, r#"섪"#, r#"섪"#, r#"섪"#, r#"섪"#),
        (r#"섫"#, r#"섫"#, r#"섫"#, r#"섫"#, r#"섫"#),
        (r#"섬"#, r#"섬"#, r#"섬"#, r#"섬"#, r#"섬"#),
        (r#"섭"#, r#"섭"#, r#"섭"#, r#"섭"#, r#"섭"#),
        (r#"섮"#, r#"섮"#, r#"섮"#, r#"섮"#, r#"섮"#),
        (r#"섯"#, r#"섯"#, r#"섯"#, r#"섯"#, r#"섯"#),
        (r#"섰"#, r#"섰"#, r#"섰"#, r#"섰"#, r#"섰"#),
        (r#"성"#, r#"성"#, r#"성"#, r#"성"#, r#"성"#),
        (r#"섲"#, r#"섲"#, r#"섲"#, r#"섲"#, r#"섲"#),
        (r#"섳"#, r#"섳"#, r#"섳"#, r#"섳"#, r#"섳"#),
        (r#"섴"#, r#"섴"#, r#"섴"#, r#"섴"#, r#"섴"#),
        (r#"섵"#, r#"섵"#, r#"섵"#, r#"섵"#, r#"섵"#),
        (r#"섶"#, r#"섶"#, r#"섶"#, r#"섶"#, r#"섶"#),
        (r#"섷"#, r#"섷"#, r#"섷"#, r#"섷"#, r#"섷"#),
        (r#"세"#, r#"세"#, r#"세"#, r#"세"#, r#"세"#),
        (r#"섹"#, r#"섹"#, r#"섹"#, r#"섹"#, r#"섹"#),
        (r#"섺"#, r#"섺"#, r#"섺"#, r#"섺"#, r#"섺"#),
        (r#"섻"#, r#"섻"#, r#"섻"#, r#"섻"#, r#"섻"#),
        (r#"센"#, r#"센"#, r#"센"#, r#"센"#, r#"센"#),
        (r#"섽"#, r#"섽"#, r#"섽"#, r#"섽"#, r#"섽"#),
        (r#"섾"#, r#"섾"#, r#"섾"#, r#"섾"#, r#"섾"#),
        (r#"섿"#, r#"섿"#, r#"섿"#, r#"섿"#, r#"섿"#),
        (r#"셀"#, r#"셀"#, r#"셀"#, r#"셀"#, r#"셀"#),
        (r#"셁"#, r#"셁"#, r#"셁"#, r#"셁"#, r#"셁"#),
        (r#"셂"#, r#"셂"#, r#"셂"#, r#"셂"#, r#"셂"#),
        (r#"셃"#, r#"셃"#, r#"셃"#, r#"셃"#, r#"셃"#),
        (r#"셄"#, r#"셄"#, r#"셄"#, r#"셄"#, r#"셄"#),
        (r#"셅"#, r#"셅"#, r#"셅"#, r#"셅"#, r#"셅"#),
        (r#"셆"#, r#"셆"#, r#"셆"#, r#"셆"#, r#"셆"#),
        (r#"셇"#, r#"셇"#, r#"셇"#, r#"셇"#, r#"셇"#),
        (r#"셈"#, r#"셈"#, r#"셈"#, r#"셈"#, r#"셈"#),
        (r#"셉"#, r#"셉"#, r#"셉"#, r#"셉"#, r#"셉"#),
        (r#"셊"#, r#"셊"#, r#"셊"#, r#"셊"#, r#"셊"#),
        (r#"셋"#, r#"셋"#, r#"셋"#, r#"셋"#, r#"셋"#),
        (r#"셌"#, r#"셌"#, r#"셌"#, r#"셌"#, r#"셌"#),
        (r#"셍"#, r#"셍"#, r#"셍"#, r#"셍"#, r#"셍"#),
        (r#"셎"#, r#"셎"#, r#"셎"#, r#"셎"#, r#"셎"#),
        (r#"셏"#, r#"셏"#, r#"셏"#, r#"셏"#, r#"셏"#),
        (r#"셐"#, r#"셐"#, r#"셐"#, r#"셐"#, r#"셐"#),
        (r#"셑"#, r#"셑"#, r#"셑"#, r#"셑"#, r#"셑"#),
        (r#"셒"#, r#"셒"#, r#"셒"#, r#"셒"#, r#"셒"#),
        (r#"셓"#, r#"셓"#, r#"셓"#, r#"셓"#, r#"셓"#),
        (r#"셔"#, r#"셔"#, r#"셔"#, r#"셔"#, r#"셔"#),
        (r#"셕"#, r#"셕"#, r#"셕"#, r#"셕"#, r#"셕"#),
        (r#"셖"#, r#"셖"#, r#"셖"#, r#"셖"#, r#"셖"#),
        (r#"셗"#, r#"셗"#, r#"셗"#, r#"셗"#, r#"셗"#),
        (r#"션"#, r#"션"#, r#"션"#, r#"션"#, r#"션"#),
        (r#"셙"#, r#"셙"#, r#"셙"#, r#"셙"#, r#"셙"#),
        (r#"셚"#, r#"셚"#, r#"셚"#, r#"셚"#, r#"셚"#),
        (r#"셛"#, r#"셛"#, r#"셛"#, r#"셛"#, r#"셛"#),
        (r#"셜"#, r#"셜"#, r#"셜"#, r#"셜"#, r#"셜"#),
        (r#"셝"#, r#"셝"#, r#"셝"#, r#"셝"#, r#"셝"#),
        (r#"셞"#, r#"셞"#, r#"셞"#, r#"셞"#, r#"셞"#),
        (r#"셟"#, r#"셟"#, r#"셟"#, r#"셟"#, r#"셟"#),
        (r#"셠"#, r#"셠"#, r#"셠"#, r#"셠"#, r#"셠"#),
        (r#"셡"#, r#"셡"#, r#"셡"#, r#"셡"#, r#"셡"#),
        (r#"셢"#, r#"셢"#, r#"셢"#, r#"셢"#, r#"셢"#),
        (r#"셣"#, r#"셣"#, r#"셣"#, r#"셣"#, r#"셣"#),
        (r#"셤"#, r#"셤"#, r#"셤"#, r#"셤"#, r#"셤"#),
        (r#"셥"#, r#"셥"#, r#"셥"#, r#"셥"#, r#"셥"#),
        (r#"셦"#, r#"셦"#, r#"셦"#, r#"셦"#, r#"셦"#),
        (r#"셧"#, r#"셧"#, r#"셧"#, r#"셧"#, r#"셧"#),
        (r#"셨"#, r#"셨"#, r#"셨"#, r#"셨"#, r#"셨"#),
        (r#"셩"#, r#"셩"#, r#"셩"#, r#"셩"#, r#"셩"#),
        (r#"셪"#, r#"셪"#, r#"셪"#, r#"셪"#, r#"셪"#),
        (r#"셫"#, r#"셫"#, r#"셫"#, r#"셫"#, r#"셫"#),
        (r#"셬"#, r#"셬"#, r#"셬"#, r#"셬"#, r#"셬"#),
        (r#"셭"#, r#"셭"#, r#"셭"#, r#"셭"#, r#"셭"#),
        (r#"셮"#, r#"셮"#, r#"셮"#, r#"셮"#, r#"셮"#),
        (r#"셯"#, r#"셯"#, r#"셯"#, r#"셯"#, r#"셯"#),
        (r#"셰"#, r#"셰"#, r#"셰"#, r#"셰"#, r#"셰"#),
        (r#"셱"#, r#"셱"#, r#"셱"#, r#"셱"#, r#"셱"#),
        (r#"셲"#, r#"셲"#, r#"셲"#, r#"셲"#, r#"셲"#),
        (r#"셳"#, r#"셳"#, r#"셳"#, r#"셳"#, r#"셳"#),
        (r#"셴"#, r#"셴"#, r#"셴"#, r#"셴"#, r#"셴"#),
        (r#"셵"#, r#"셵"#, r#"셵"#, r#"셵"#, r#"셵"#),
        (r#"셶"#, r#"셶"#, r#"셶"#, r#"셶"#, r#"셶"#),
        (r#"셷"#, r#"셷"#, r#"셷"#, r#"셷"#, r#"셷"#),
        (r#"셸"#, r#"셸"#, r#"셸"#, r#"셸"#, r#"셸"#),
        (r#"셹"#, r#"셹"#, r#"셹"#, r#"셹"#, r#"셹"#),
        (r#"셺"#, r#"셺"#, r#"셺"#, r#"셺"#, r#"셺"#),
        (r#"셻"#, r#"셻"#, r#"셻"#, r#"셻"#, r#"셻"#),
        (r#"셼"#, r#"셼"#, r#"셼"#, r#"셼"#, r#"셼"#),
        (r#"셽"#, r#"셽"#, r#"셽"#, r#"셽"#, r#"셽"#),
        (r#"셾"#, r#"셾"#, r#"셾"#, r#"셾"#, r#"셾"#),
        (r#"셿"#, r#"셿"#, r#"셿"#, r#"셿"#, r#"셿"#),
        (r#"솀"#, r#"솀"#, r#"솀"#, r#"솀"#, r#"솀"#),
        (r#"솁"#, r#"솁"#, r#"솁"#, r#"솁"#, r#"솁"#),
        (r#"솂"#, r#"솂"#, r#"솂"#, r#"솂"#, r#"솂"#),
        (r#"솃"#, r#"솃"#, r#"솃"#, r#"솃"#, r#"솃"#),
        (r#"솄"#, r#"솄"#, r#"솄"#, r#"솄"#, r#"솄"#),
        (r#"솅"#, r#"솅"#, r#"솅"#, r#"솅"#, r#"솅"#),
        (r#"솆"#, r#"솆"#, r#"솆"#, r#"솆"#, r#"솆"#),
        (r#"솇"#, r#"솇"#, r#"솇"#, r#"솇"#, r#"솇"#),
        (r#"솈"#, r#"솈"#, r#"솈"#, r#"솈"#, r#"솈"#),
        (r#"솉"#, r#"솉"#, r#"솉"#, r#"솉"#, r#"솉"#),
        (r#"솊"#, r#"솊"#, r#"솊"#, r#"솊"#, r#"솊"#),
        (r#"솋"#, r#"솋"#, r#"솋"#, r#"솋"#, r#"솋"#),
        (r#"소"#, r#"소"#, r#"소"#, r#"소"#, r#"소"#),
        (r#"속"#, r#"속"#, r#"속"#, r#"속"#, r#"속"#),
        (r#"솎"#, r#"솎"#, r#"솎"#, r#"솎"#, r#"솎"#),
        (r#"솏"#, r#"솏"#, r#"솏"#, r#"솏"#, r#"솏"#),
        (r#"손"#, r#"손"#, r#"손"#, r#"손"#, r#"손"#),
        (r#"솑"#, r#"솑"#, r#"솑"#, r#"솑"#, r#"솑"#),
        (r#"솒"#, r#"솒"#, r#"솒"#, r#"솒"#, r#"솒"#),
        (r#"솓"#, r#"솓"#, r#"솓"#, r#"솓"#, r#"솓"#),
        (r#"솔"#, r#"솔"#, r#"솔"#, r#"솔"#, r#"솔"#),
        (r#"솕"#, r#"솕"#, r#"솕"#, r#"솕"#, r#"솕"#),
        (r#"솖"#, r#"솖"#, r#"솖"#, r#"솖"#, r#"솖"#),
        (r#"솗"#, r#"솗"#, r#"솗"#, r#"솗"#, r#"솗"#),
        (r#"솘"#, r#"솘"#, r#"솘"#, r#"솘"#, r#"솘"#),
        (r#"솙"#, r#"솙"#, r#"솙"#, r#"솙"#, r#"솙"#),
        (r#"솚"#, r#"솚"#, r#"솚"#, r#"솚"#, r#"솚"#),
        (r#"솛"#, r#"솛"#, r#"솛"#, r#"솛"#, r#"솛"#),
        (r#"솜"#, r#"솜"#, r#"솜"#, r#"솜"#, r#"솜"#),
        (r#"솝"#, r#"솝"#, r#"솝"#, r#"솝"#, r#"솝"#),
        (r#"솞"#, r#"솞"#, r#"솞"#, r#"솞"#, r#"솞"#),
        (r#"솟"#, r#"솟"#, r#"솟"#, r#"솟"#, r#"솟"#),
        (r#"솠"#, r#"솠"#, r#"솠"#, r#"솠"#, r#"솠"#),
        (r#"송"#, r#"송"#, r#"송"#, r#"송"#, r#"송"#),
        (r#"솢"#, r#"솢"#, r#"솢"#, r#"솢"#, r#"솢"#),
        (r#"솣"#, r#"솣"#, r#"솣"#, r#"솣"#, r#"솣"#),
        (r#"솤"#, r#"솤"#, r#"솤"#, r#"솤"#, r#"솤"#),
        (r#"솥"#, r#"솥"#, r#"솥"#, r#"솥"#, r#"솥"#),
        (r#"솦"#, r#"솦"#, r#"솦"#, r#"솦"#, r#"솦"#),
        (r#"솧"#, r#"솧"#, r#"솧"#, r#"솧"#, r#"솧"#),
        (r#"솨"#, r#"솨"#, r#"솨"#, r#"솨"#, r#"솨"#),
        (r#"솩"#, r#"솩"#, r#"솩"#, r#"솩"#, r#"솩"#),
        (r#"솪"#, r#"솪"#, r#"솪"#, r#"솪"#, r#"솪"#),
        (r#"솫"#, r#"솫"#, r#"솫"#, r#"솫"#, r#"솫"#),
        (r#"솬"#, r#"솬"#, r#"솬"#, r#"솬"#, r#"솬"#),
        (r#"솭"#, r#"솭"#, r#"솭"#, r#"솭"#, r#"솭"#),
        (r#"솮"#, r#"솮"#, r#"솮"#, r#"솮"#, r#"솮"#),
        (r#"솯"#, r#"솯"#, r#"솯"#, r#"솯"#, r#"솯"#),
        (r#"솰"#, r#"솰"#, r#"솰"#, r#"솰"#, r#"솰"#),
        (r#"솱"#, r#"솱"#, r#"솱"#, r#"솱"#, r#"솱"#),
        (r#"솲"#, r#"솲"#, r#"솲"#, r#"솲"#, r#"솲"#),
        (r#"솳"#, r#"솳"#, r#"솳"#, r#"솳"#, r#"솳"#),
        (r#"솴"#, r#"솴"#, r#"솴"#, r#"솴"#, r#"솴"#),
        (r#"솵"#, r#"솵"#, r#"솵"#, r#"솵"#, r#"솵"#),
        (r#"솶"#, r#"솶"#, r#"솶"#, r#"솶"#, r#"솶"#),
        (r#"솷"#, r#"솷"#, r#"솷"#, r#"솷"#, r#"솷"#),
        (r#"솸"#, r#"솸"#, r#"솸"#, r#"솸"#, r#"솸"#),
        (r#"솹"#, r#"솹"#, r#"솹"#, r#"솹"#, r#"솹"#),
        (r#"솺"#, r#"솺"#, r#"솺"#, r#"솺"#, r#"솺"#),
        (r#"솻"#, r#"솻"#, r#"솻"#, r#"솻"#, r#"솻"#),
        (r#"솼"#, r#"솼"#, r#"솼"#, r#"솼"#, r#"솼"#),
        (r#"솽"#, r#"솽"#, r#"솽"#, r#"솽"#, r#"솽"#),
        (r#"솾"#, r#"솾"#, r#"솾"#, r#"솾"#, r#"솾"#),
        (r#"솿"#, r#"솿"#, r#"솿"#, r#"솿"#, r#"솿"#),
        (r#"쇀"#, r#"쇀"#, r#"쇀"#, r#"쇀"#, r#"쇀"#),
        (r#"쇁"#, r#"쇁"#, r#"쇁"#, r#"쇁"#, r#"쇁"#),
        (r#"쇂"#, r#"쇂"#, r#"쇂"#, r#"쇂"#, r#"쇂"#),
        (r#"쇃"#, r#"쇃"#, r#"쇃"#, r#"쇃"#, r#"쇃"#),
        (r#"쇄"#, r#"쇄"#, r#"쇄"#, r#"쇄"#, r#"쇄"#),
        (r#"쇅"#, r#"쇅"#, r#"쇅"#, r#"쇅"#, r#"쇅"#),
        (r#"쇆"#, r#"쇆"#, r#"쇆"#, r#"쇆"#, r#"쇆"#),
        (r#"쇇"#, r#"쇇"#, r#"쇇"#, r#"쇇"#, r#"쇇"#),
        (r#"쇈"#, r#"쇈"#, r#"쇈"#, r#"쇈"#, r#"쇈"#),
        (r#"쇉"#, r#"쇉"#, r#"쇉"#, r#"쇉"#, r#"쇉"#),
        (r#"쇊"#, r#"쇊"#, r#"쇊"#, r#"쇊"#, r#"쇊"#),
        (r#"쇋"#, r#"쇋"#, r#"쇋"#, r#"쇋"#, r#"쇋"#),
        (r#"쇌"#, r#"쇌"#, r#"쇌"#, r#"쇌"#, r#"쇌"#),
        (r#"쇍"#, r#"쇍"#, r#"쇍"#, r#"쇍"#, r#"쇍"#),
        (r#"쇎"#, r#"쇎"#, r#"쇎"#, r#"쇎"#, r#"쇎"#),
        (r#"쇏"#, r#"쇏"#, r#"쇏"#, r#"쇏"#, r#"쇏"#),
        (r#"쇐"#, r#"쇐"#, r#"쇐"#, r#"쇐"#, r#"쇐"#),
        (r#"쇑"#, r#"쇑"#, r#"쇑"#, r#"쇑"#, r#"쇑"#),
        (r#"쇒"#, r#"쇒"#, r#"쇒"#, r#"쇒"#, r#"쇒"#),
        (r#"쇓"#, r#"쇓"#, r#"쇓"#, r#"쇓"#, r#"쇓"#),
        (r#"쇔"#, r#"쇔"#, r#"쇔"#, r#"쇔"#, r#"쇔"#),
        (r#"쇕"#, r#"쇕"#, r#"쇕"#, r#"쇕"#, r#"쇕"#),
        (r#"쇖"#, r#"쇖"#, r#"쇖"#, r#"쇖"#, r#"쇖"#),
        (r#"쇗"#, r#"쇗"#, r#"쇗"#, r#"쇗"#, r#"쇗"#),
        (r#"쇘"#, r#"쇘"#, r#"쇘"#, r#"쇘"#, r#"쇘"#),
        (r#"쇙"#, r#"쇙"#, r#"쇙"#, r#"쇙"#, r#"쇙"#),
        (r#"쇚"#, r#"쇚"#, r#"쇚"#, r#"쇚"#, r#"쇚"#),
        (r#"쇛"#, r#"쇛"#, r#"쇛"#, r#"쇛"#, r#"쇛"#),
        (r#"쇜"#, r#"쇜"#, r#"쇜"#, r#"쇜"#, r#"쇜"#),
        (r#"쇝"#, r#"쇝"#, r#"쇝"#, r#"쇝"#, r#"쇝"#),
        (r#"쇞"#, r#"쇞"#, r#"쇞"#, r#"쇞"#, r#"쇞"#),
        (r#"쇟"#, r#"쇟"#, r#"쇟"#, r#"쇟"#, r#"쇟"#),
        (r#"쇠"#, r#"쇠"#, r#"쇠"#, r#"쇠"#, r#"쇠"#),
        (r#"쇡"#, r#"쇡"#, r#"쇡"#, r#"쇡"#, r#"쇡"#),
        (r#"쇢"#, r#"쇢"#, r#"쇢"#, r#"쇢"#, r#"쇢"#),
        (r#"쇣"#, r#"쇣"#, r#"쇣"#, r#"쇣"#, r#"쇣"#),
        (r#"쇤"#, r#"쇤"#, r#"쇤"#, r#"쇤"#, r#"쇤"#),
        (r#"쇥"#, r#"쇥"#, r#"쇥"#, r#"쇥"#, r#"쇥"#),
        (r#"쇦"#, r#"쇦"#, r#"쇦"#, r#"쇦"#, r#"쇦"#),
        (r#"쇧"#, r#"쇧"#, r#"쇧"#, r#"쇧"#, r#"쇧"#),
        (r#"쇨"#, r#"쇨"#, r#"쇨"#, r#"쇨"#, r#"쇨"#),
        (r#"쇩"#, r#"쇩"#, r#"쇩"#, r#"쇩"#, r#"쇩"#),
        (r#"쇪"#, r#"쇪"#, r#"쇪"#, r#"쇪"#, r#"쇪"#),
        (r#"쇫"#, r#"쇫"#, r#"쇫"#, r#"쇫"#, r#"쇫"#),
        (r#"쇬"#, r#"쇬"#, r#"쇬"#, r#"쇬"#, r#"쇬"#),
        (r#"쇭"#, r#"쇭"#, r#"쇭"#, r#"쇭"#, r#"쇭"#),
        (r#"쇮"#, r#"쇮"#, r#"쇮"#, r#"쇮"#, r#"쇮"#),
        (r#"쇯"#, r#"쇯"#, r#"쇯"#, r#"쇯"#, r#"쇯"#),
        (r#"쇰"#, r#"쇰"#, r#"쇰"#, r#"쇰"#, r#"쇰"#),
        (r#"쇱"#, r#"쇱"#, r#"쇱"#, r#"쇱"#, r#"쇱"#),
        (r#"쇲"#, r#"쇲"#, r#"쇲"#, r#"쇲"#, r#"쇲"#),
        (r#"쇳"#, r#"쇳"#, r#"쇳"#, r#"쇳"#, r#"쇳"#),
        (r#"쇴"#, r#"쇴"#, r#"쇴"#, r#"쇴"#, r#"쇴"#),
        (r#"쇵"#, r#"쇵"#, r#"쇵"#, r#"쇵"#, r#"쇵"#),
        (r#"쇶"#, r#"쇶"#, r#"쇶"#, r#"쇶"#, r#"쇶"#),
        (r#"쇷"#, r#"쇷"#, r#"쇷"#, r#"쇷"#, r#"쇷"#),
        (r#"쇸"#, r#"쇸"#, r#"쇸"#, r#"쇸"#, r#"쇸"#),
        (r#"쇹"#, r#"쇹"#, r#"쇹"#, r#"쇹"#, r#"쇹"#),
        (r#"쇺"#, r#"쇺"#, r#"쇺"#, r#"쇺"#, r#"쇺"#),
        (r#"쇻"#, r#"쇻"#, r#"쇻"#, r#"쇻"#, r#"쇻"#),
        (r#"쇼"#, r#"쇼"#, r#"쇼"#, r#"쇼"#, r#"쇼"#),
        (r#"쇽"#, r#"쇽"#, r#"쇽"#, r#"쇽"#, r#"쇽"#),
        (r#"쇾"#, r#"쇾"#, r#"쇾"#, r#"쇾"#, r#"쇾"#),
        (r#"쇿"#, r#"쇿"#, r#"쇿"#, r#"쇿"#, r#"쇿"#),
        (r#"숀"#, r#"숀"#, r#"숀"#, r#"숀"#, r#"숀"#),
        (r#"숁"#, r#"숁"#, r#"숁"#, r#"숁"#, r#"숁"#),
        (r#"숂"#, r#"숂"#, r#"숂"#, r#"숂"#, r#"숂"#),
        (r#"숃"#, r#"숃"#, r#"숃"#, r#"숃"#, r#"숃"#),
        (r#"숄"#, r#"숄"#, r#"숄"#, r#"숄"#, r#"숄"#),
        (r#"숅"#, r#"숅"#, r#"숅"#, r#"숅"#, r#"숅"#),
        (r#"숆"#, r#"숆"#, r#"숆"#, r#"숆"#, r#"숆"#),
        (r#"숇"#, r#"숇"#, r#"숇"#, r#"숇"#, r#"숇"#),
        (r#"숈"#, r#"숈"#, r#"숈"#, r#"숈"#, r#"숈"#),
        (r#"숉"#, r#"숉"#, r#"숉"#, r#"숉"#, r#"숉"#),
        (r#"숊"#, r#"숊"#, r#"숊"#, r#"숊"#, r#"숊"#),
        (r#"숋"#, r#"숋"#, r#"숋"#, r#"숋"#, r#"숋"#),
        (r#"숌"#, r#"숌"#, r#"숌"#, r#"숌"#, r#"숌"#),
        (r#"숍"#, r#"숍"#, r#"숍"#, r#"숍"#, r#"숍"#),
        (r#"숎"#, r#"숎"#, r#"숎"#, r#"숎"#, r#"숎"#),
        (r#"숏"#, r#"숏"#, r#"숏"#, r#"숏"#, r#"숏"#),
        (r#"숐"#, r#"숐"#, r#"숐"#, r#"숐"#, r#"숐"#),
        (r#"숑"#, r#"숑"#, r#"숑"#, r#"숑"#, r#"숑"#),
        (r#"숒"#, r#"숒"#, r#"숒"#, r#"숒"#, r#"숒"#),
        (r#"숓"#, r#"숓"#, r#"숓"#, r#"숓"#, r#"숓"#),
        (r#"숔"#, r#"숔"#, r#"숔"#, r#"숔"#, r#"숔"#),
        (r#"숕"#, r#"숕"#, r#"숕"#, r#"숕"#, r#"숕"#),
        (r#"숖"#, r#"숖"#, r#"숖"#, r#"숖"#, r#"숖"#),
        (r#"숗"#, r#"숗"#, r#"숗"#, r#"숗"#, r#"숗"#),
        (r#"수"#, r#"수"#, r#"수"#, r#"수"#, r#"수"#),
        (r#"숙"#, r#"숙"#, r#"숙"#, r#"숙"#, r#"숙"#),
        (r#"숚"#, r#"숚"#, r#"숚"#, r#"숚"#, r#"숚"#),
        (r#"숛"#, r#"숛"#, r#"숛"#, r#"숛"#, r#"숛"#),
        (r#"순"#, r#"순"#, r#"순"#, r#"순"#, r#"순"#),
        (r#"숝"#, r#"숝"#, r#"숝"#, r#"숝"#, r#"숝"#),
        (r#"숞"#, r#"숞"#, r#"숞"#, r#"숞"#, r#"숞"#),
        (r#"숟"#, r#"숟"#, r#"숟"#, r#"숟"#, r#"숟"#),
        (r#"술"#, r#"술"#, r#"술"#, r#"술"#, r#"술"#),
        (r#"숡"#, r#"숡"#, r#"숡"#, r#"숡"#, r#"숡"#),
        (r#"숢"#, r#"숢"#, r#"숢"#, r#"숢"#, r#"숢"#),
        (r#"숣"#, r#"숣"#, r#"숣"#, r#"숣"#, r#"숣"#),
        (r#"숤"#, r#"숤"#, r#"숤"#, r#"숤"#, r#"숤"#),
        (r#"숥"#, r#"숥"#, r#"숥"#, r#"숥"#, r#"숥"#),
        (r#"숦"#, r#"숦"#, r#"숦"#, r#"숦"#, r#"숦"#),
        (r#"숧"#, r#"숧"#, r#"숧"#, r#"숧"#, r#"숧"#),
        (r#"숨"#, r#"숨"#, r#"숨"#, r#"숨"#, r#"숨"#),
        (r#"숩"#, r#"숩"#, r#"숩"#, r#"숩"#, r#"숩"#),
        (r#"숪"#, r#"숪"#, r#"숪"#, r#"숪"#, r#"숪"#),
        (r#"숫"#, r#"숫"#, r#"숫"#, r#"숫"#, r#"숫"#),
        (r#"숬"#, r#"숬"#, r#"숬"#, r#"숬"#, r#"숬"#),
        (r#"숭"#, r#"숭"#, r#"숭"#, r#"숭"#, r#"숭"#),
        (r#"숮"#, r#"숮"#, r#"숮"#, r#"숮"#, r#"숮"#),
        (r#"숯"#, r#"숯"#, r#"숯"#, r#"숯"#, r#"숯"#),
        (r#"숰"#, r#"숰"#, r#"숰"#, r#"숰"#, r#"숰"#),
        (r#"숱"#, r#"숱"#, r#"숱"#, r#"숱"#, r#"숱"#),
        (r#"숲"#, r#"숲"#, r#"숲"#, r#"숲"#, r#"숲"#),
        (r#"숳"#, r#"숳"#, r#"숳"#, r#"숳"#, r#"숳"#),
        (r#"숴"#, r#"숴"#, r#"숴"#, r#"숴"#, r#"숴"#),
        (r#"숵"#, r#"숵"#, r#"숵"#, r#"숵"#, r#"숵"#),
        (r#"숶"#, r#"숶"#, r#"숶"#, r#"숶"#, r#"숶"#),
        (r#"숷"#, r#"숷"#, r#"숷"#, r#"숷"#, r#"숷"#),
        (r#"숸"#, r#"숸"#, r#"숸"#, r#"숸"#, r#"숸"#),
        (r#"숹"#, r#"숹"#, r#"숹"#, r#"숹"#, r#"숹"#),
        (r#"숺"#, r#"숺"#, r#"숺"#, r#"숺"#, r#"숺"#),
        (r#"숻"#, r#"숻"#, r#"숻"#, r#"숻"#, r#"숻"#),
        (r#"숼"#, r#"숼"#, r#"숼"#, r#"숼"#, r#"숼"#),
        (r#"숽"#, r#"숽"#, r#"숽"#, r#"숽"#, r#"숽"#),
        (r#"숾"#, r#"숾"#, r#"숾"#, r#"숾"#, r#"숾"#),
        (r#"숿"#, r#"숿"#, r#"숿"#, r#"숿"#, r#"숿"#),
        (r#"쉀"#, r#"쉀"#, r#"쉀"#, r#"쉀"#, r#"쉀"#),
        (r#"쉁"#, r#"쉁"#, r#"쉁"#, r#"쉁"#, r#"쉁"#),
        (r#"쉂"#, r#"쉂"#, r#"쉂"#, r#"쉂"#, r#"쉂"#),
        (r#"쉃"#, r#"쉃"#, r#"쉃"#, r#"쉃"#, r#"쉃"#),
        (r#"쉄"#, r#"쉄"#, r#"쉄"#, r#"쉄"#, r#"쉄"#),
        (r#"쉅"#, r#"쉅"#, r#"쉅"#, r#"쉅"#, r#"쉅"#),
        (r#"쉆"#, r#"쉆"#, r#"쉆"#, r#"쉆"#, r#"쉆"#),
        (r#"쉇"#, r#"쉇"#, r#"쉇"#, r#"쉇"#, r#"쉇"#),
        (r#"쉈"#, r#"쉈"#, r#"쉈"#, r#"쉈"#, r#"쉈"#),
        (r#"쉉"#, r#"쉉"#, r#"쉉"#, r#"쉉"#, r#"쉉"#),
        (r#"쉊"#, r#"쉊"#, r#"쉊"#, r#"쉊"#, r#"쉊"#),
        (r#"쉋"#, r#"쉋"#, r#"쉋"#, r#"쉋"#, r#"쉋"#),
        (r#"쉌"#, r#"쉌"#, r#"쉌"#, r#"쉌"#, r#"쉌"#),
        (r#"쉍"#, r#"쉍"#, r#"쉍"#, r#"쉍"#, r#"쉍"#),
        (r#"쉎"#, r#"쉎"#, r#"쉎"#, r#"쉎"#, r#"쉎"#),
        (r#"쉏"#, r#"쉏"#, r#"쉏"#, r#"쉏"#, r#"쉏"#),
        (r#"쉐"#, r#"쉐"#, r#"쉐"#, r#"쉐"#, r#"쉐"#),
        (r#"쉑"#, r#"쉑"#, r#"쉑"#, r#"쉑"#, r#"쉑"#),
        (r#"쉒"#, r#"쉒"#, r#"쉒"#, r#"쉒"#, r#"쉒"#),
        (r#"쉓"#, r#"쉓"#, r#"쉓"#, r#"쉓"#, r#"쉓"#),
        (r#"쉔"#, r#"쉔"#, r#"쉔"#, r#"쉔"#, r#"쉔"#),
        (r#"쉕"#, r#"쉕"#, r#"쉕"#, r#"쉕"#, r#"쉕"#),
        (r#"쉖"#, r#"쉖"#, r#"쉖"#, r#"쉖"#, r#"쉖"#),
        (r#"쉗"#, r#"쉗"#, r#"쉗"#, r#"쉗"#, r#"쉗"#),
        (r#"쉘"#, r#"쉘"#, r#"쉘"#, r#"쉘"#, r#"쉘"#),
        (r#"쉙"#, r#"쉙"#, r#"쉙"#, r#"쉙"#, r#"쉙"#),
        (r#"쉚"#, r#"쉚"#, r#"쉚"#, r#"쉚"#, r#"쉚"#),
        (r#"쉛"#, r#"쉛"#, r#"쉛"#, r#"쉛"#, r#"쉛"#),
        (r#"쉜"#, r#"쉜"#, r#"쉜"#, r#"쉜"#, r#"쉜"#),
        (r#"쉝"#, r#"쉝"#, r#"쉝"#, r#"쉝"#, r#"쉝"#),
        (r#"쉞"#, r#"쉞"#, r#"쉞"#, r#"쉞"#, r#"쉞"#),
        (r#"쉟"#, r#"쉟"#, r#"쉟"#, r#"쉟"#, r#"쉟"#),
        (r#"쉠"#, r#"쉠"#, r#"쉠"#, r#"쉠"#, r#"쉠"#),
        (r#"쉡"#, r#"쉡"#, r#"쉡"#, r#"쉡"#, r#"쉡"#),
        (r#"쉢"#, r#"쉢"#, r#"쉢"#, r#"쉢"#, r#"쉢"#),
        (r#"쉣"#, r#"쉣"#, r#"쉣"#, r#"쉣"#, r#"쉣"#),
        (r#"쉤"#, r#"쉤"#, r#"쉤"#, r#"쉤"#, r#"쉤"#),
        (r#"쉥"#, r#"쉥"#, r#"쉥"#, r#"쉥"#, r#"쉥"#),
        (r#"쉦"#, r#"쉦"#, r#"쉦"#, r#"쉦"#, r#"쉦"#),
        (r#"쉧"#, r#"쉧"#, r#"쉧"#, r#"쉧"#, r#"쉧"#),
        (r#"쉨"#, r#"쉨"#, r#"쉨"#, r#"쉨"#, r#"쉨"#),
        (r#"쉩"#, r#"쉩"#, r#"쉩"#, r#"쉩"#, r#"쉩"#),
        (r#"쉪"#, r#"쉪"#, r#"쉪"#, r#"쉪"#, r#"쉪"#),
        (r#"쉫"#, r#"쉫"#, r#"쉫"#, r#"쉫"#, r#"쉫"#),
        (r#"쉬"#, r#"쉬"#, r#"쉬"#, r#"쉬"#, r#"쉬"#),
        (r#"쉭"#, r#"쉭"#, r#"쉭"#, r#"쉭"#, r#"쉭"#),
        (r#"쉮"#, r#"쉮"#, r#"쉮"#, r#"쉮"#, r#"쉮"#),
        (r#"쉯"#, r#"쉯"#, r#"쉯"#, r#"쉯"#, r#"쉯"#),
        (r#"쉰"#, r#"쉰"#, r#"쉰"#, r#"쉰"#, r#"쉰"#),
        (r#"쉱"#, r#"쉱"#, r#"쉱"#, r#"쉱"#, r#"쉱"#),
        (r#"쉲"#, r#"쉲"#, r#"쉲"#, r#"쉲"#, r#"쉲"#),
        (r#"쉳"#, r#"쉳"#, r#"쉳"#, r#"쉳"#, r#"쉳"#),
        (r#"쉴"#, r#"쉴"#, r#"쉴"#, r#"쉴"#, r#"쉴"#),
        (r#"쉵"#, r#"쉵"#, r#"쉵"#, r#"쉵"#, r#"쉵"#),
        (r#"쉶"#, r#"쉶"#, r#"쉶"#, r#"쉶"#, r#"쉶"#),
        (r#"쉷"#, r#"쉷"#, r#"쉷"#, r#"쉷"#, r#"쉷"#),
        (r#"쉸"#, r#"쉸"#, r#"쉸"#, r#"쉸"#, r#"쉸"#),
        (r#"쉹"#, r#"쉹"#, r#"쉹"#, r#"쉹"#, r#"쉹"#),
        (r#"쉺"#, r#"쉺"#, r#"쉺"#, r#"쉺"#, r#"쉺"#),
        (r#"쉻"#, r#"쉻"#, r#"쉻"#, r#"쉻"#, r#"쉻"#),
        (r#"쉼"#, r#"쉼"#, r#"쉼"#, r#"쉼"#, r#"쉼"#),
        (r#"쉽"#, r#"쉽"#, r#"쉽"#, r#"쉽"#, r#"쉽"#),
        (r#"쉾"#, r#"쉾"#, r#"쉾"#, r#"쉾"#, r#"쉾"#),
        (r#"쉿"#, r#"쉿"#, r#"쉿"#, r#"쉿"#, r#"쉿"#),
        (r#"슀"#, r#"슀"#, r#"슀"#, r#"슀"#, r#"슀"#),
        (r#"슁"#, r#"슁"#, r#"슁"#, r#"슁"#, r#"슁"#),
        (r#"슂"#, r#"슂"#, r#"슂"#, r#"슂"#, r#"슂"#),
        (r#"슃"#, r#"슃"#, r#"슃"#, r#"슃"#, r#"슃"#),
        (r#"슄"#, r#"슄"#, r#"슄"#, r#"슄"#, r#"슄"#),
        (r#"슅"#, r#"슅"#, r#"슅"#, r#"슅"#, r#"슅"#),
        (r#"슆"#, r#"슆"#, r#"슆"#, r#"슆"#, r#"슆"#),
        (r#"슇"#, r#"슇"#, r#"슇"#, r#"슇"#, r#"슇"#),
        (r#"슈"#, r#"슈"#, r#"슈"#, r#"슈"#, r#"슈"#),
        (r#"슉"#, r#"슉"#, r#"슉"#, r#"슉"#, r#"슉"#),
        (r#"슊"#, r#"슊"#, r#"슊"#, r#"슊"#, r#"슊"#),
        (r#"슋"#, r#"슋"#, r#"슋"#, r#"슋"#, r#"슋"#),
        (r#"슌"#, r#"슌"#, r#"슌"#, r#"슌"#, r#"슌"#),
        (r#"슍"#, r#"슍"#, r#"슍"#, r#"슍"#, r#"슍"#),
        (r#"슎"#, r#"슎"#, r#"슎"#, r#"슎"#, r#"슎"#),
        (r#"슏"#, r#"슏"#, r#"슏"#, r#"슏"#, r#"슏"#),
        (r#"슐"#, r#"슐"#, r#"슐"#, r#"슐"#, r#"슐"#),
        (r#"슑"#, r#"슑"#, r#"슑"#, r#"슑"#, r#"슑"#),
        (r#"슒"#, r#"슒"#, r#"슒"#, r#"슒"#, r#"슒"#),
        (r#"슓"#, r#"슓"#, r#"슓"#, r#"슓"#, r#"슓"#),
        (r#"슔"#, r#"슔"#, r#"슔"#, r#"슔"#, r#"슔"#),
        (r#"슕"#, r#"슕"#, r#"슕"#, r#"슕"#, r#"슕"#),
        (r#"슖"#, r#"슖"#, r#"슖"#, r#"슖"#, r#"슖"#),
        (r#"슗"#, r#"슗"#, r#"슗"#, r#"슗"#, r#"슗"#),
        (r#"슘"#, r#"슘"#, r#"슘"#, r#"슘"#, r#"슘"#),
        (r#"슙"#, r#"슙"#, r#"슙"#, r#"슙"#, r#"슙"#),
        (r#"슚"#, r#"슚"#, r#"슚"#, r#"슚"#, r#"슚"#),
        (r#"슛"#, r#"슛"#, r#"슛"#, r#"슛"#, r#"슛"#),
        (r#"슜"#, r#"슜"#, r#"슜"#, r#"슜"#, r#"슜"#),
        (r#"슝"#, r#"슝"#, r#"슝"#, r#"슝"#, r#"슝"#),
        (r#"슞"#, r#"슞"#, r#"슞"#, r#"슞"#, r#"슞"#),
        (r#"슟"#, r#"슟"#, r#"슟"#, r#"슟"#, r#"슟"#),
        (r#"슠"#, r#"슠"#, r#"슠"#, r#"슠"#, r#"슠"#),
        (r#"슡"#, r#"슡"#, r#"슡"#, r#"슡"#, r#"슡"#),
        (r#"슢"#, r#"슢"#, r#"슢"#, r#"슢"#, r#"슢"#),
        (r#"슣"#, r#"슣"#, r#"슣"#, r#"슣"#, r#"슣"#),
        (r#"스"#, r#"스"#, r#"스"#, r#"스"#, r#"스"#),
        (r#"슥"#, r#"슥"#, r#"슥"#, r#"슥"#, r#"슥"#),
        (r#"슦"#, r#"슦"#, r#"슦"#, r#"슦"#, r#"슦"#),
        (r#"슧"#, r#"슧"#, r#"슧"#, r#"슧"#, r#"슧"#),
        (r#"슨"#, r#"슨"#, r#"슨"#, r#"슨"#, r#"슨"#),
        (r#"슩"#, r#"슩"#, r#"슩"#, r#"슩"#, r#"슩"#),
        (r#"슪"#, r#"슪"#, r#"슪"#, r#"슪"#, r#"슪"#),
        (r#"슫"#, r#"슫"#, r#"슫"#, r#"슫"#, r#"슫"#),
        (r#"슬"#, r#"슬"#, r#"슬"#, r#"슬"#, r#"슬"#),
        (r#"슭"#, r#"슭"#, r#"슭"#, r#"슭"#, r#"슭"#),
        (r#"슮"#, r#"슮"#, r#"슮"#, r#"슮"#, r#"슮"#),
        (r#"슯"#, r#"슯"#, r#"슯"#, r#"슯"#, r#"슯"#),
        (r#"슰"#, r#"슰"#, r#"슰"#, r#"슰"#, r#"슰"#),
        (r#"슱"#, r#"슱"#, r#"슱"#, r#"슱"#, r#"슱"#),
        (r#"슲"#, r#"슲"#, r#"슲"#, r#"슲"#, r#"슲"#),
        (r#"슳"#, r#"슳"#, r#"슳"#, r#"슳"#, r#"슳"#),
        (r#"슴"#, r#"슴"#, r#"슴"#, r#"슴"#, r#"슴"#),
        (r#"습"#, r#"습"#, r#"습"#, r#"습"#, r#"습"#),
        (r#"슶"#, r#"슶"#, r#"슶"#, r#"슶"#, r#"슶"#),
        (r#"슷"#, r#"슷"#, r#"슷"#, r#"슷"#, r#"슷"#),
        (r#"슸"#, r#"슸"#, r#"슸"#, r#"슸"#, r#"슸"#),
        (r#"승"#, r#"승"#, r#"승"#, r#"승"#, r#"승"#),
        (r#"슺"#, r#"슺"#, r#"슺"#, r#"슺"#, r#"슺"#),
        (r#"슻"#, r#"슻"#, r#"슻"#, r#"슻"#, r#"슻"#),
        (r#"슼"#, r#"슼"#, r#"슼"#, r#"슼"#, r#"슼"#),
        (r#"슽"#, r#"슽"#, r#"슽"#, r#"슽"#, r#"슽"#),
        (r#"슾"#, r#"슾"#, r#"슾"#, r#"슾"#, r#"슾"#),
        (r#"슿"#, r#"슿"#, r#"슿"#, r#"슿"#, r#"슿"#),
        (r#"싀"#, r#"싀"#, r#"싀"#, r#"싀"#, r#"싀"#),
        (r#"싁"#, r#"싁"#, r#"싁"#, r#"싁"#, r#"싁"#),
        (r#"싂"#, r#"싂"#, r#"싂"#, r#"싂"#, r#"싂"#),
        (r#"싃"#, r#"싃"#, r#"싃"#, r#"싃"#, r#"싃"#),
        (r#"싄"#, r#"싄"#, r#"싄"#, r#"싄"#, r#"싄"#),
        (r#"싅"#, r#"싅"#, r#"싅"#, r#"싅"#, r#"싅"#),
        (r#"싆"#, r#"싆"#, r#"싆"#, r#"싆"#, r#"싆"#),
        (r#"싇"#, r#"싇"#, r#"싇"#, r#"싇"#, r#"싇"#),
        (r#"싈"#, r#"싈"#, r#"싈"#, r#"싈"#, r#"싈"#),
        (r#"싉"#, r#"싉"#, r#"싉"#, r#"싉"#, r#"싉"#),
        (r#"싊"#, r#"싊"#, r#"싊"#, r#"싊"#, r#"싊"#),
        (r#"싋"#, r#"싋"#, r#"싋"#, r#"싋"#, r#"싋"#),
        (r#"싌"#, r#"싌"#, r#"싌"#, r#"싌"#, r#"싌"#),
        (r#"싍"#, r#"싍"#, r#"싍"#, r#"싍"#, r#"싍"#),
        (r#"싎"#, r#"싎"#, r#"싎"#, r#"싎"#, r#"싎"#),
        (r#"싏"#, r#"싏"#, r#"싏"#, r#"싏"#, r#"싏"#),
        (r#"싐"#, r#"싐"#, r#"싐"#, r#"싐"#, r#"싐"#),
        (r#"싑"#, r#"싑"#, r#"싑"#, r#"싑"#, r#"싑"#),
        (r#"싒"#, r#"싒"#, r#"싒"#, r#"싒"#, r#"싒"#),
        (r#"싓"#, r#"싓"#, r#"싓"#, r#"싓"#, r#"싓"#),
        (r#"싔"#, r#"싔"#, r#"싔"#, r#"싔"#, r#"싔"#),
        (r#"싕"#, r#"싕"#, r#"싕"#, r#"싕"#, r#"싕"#),
        (r#"싖"#, r#"싖"#, r#"싖"#, r#"싖"#, r#"싖"#),
        (r#"싗"#, r#"싗"#, r#"싗"#, r#"싗"#, r#"싗"#),
        (r#"싘"#, r#"싘"#, r#"싘"#, r#"싘"#, r#"싘"#),
        (r#"싙"#, r#"싙"#, r#"싙"#, r#"싙"#, r#"싙"#),
        (r#"싚"#, r#"싚"#, r#"싚"#, r#"싚"#, r#"싚"#),
        (r#"싛"#, r#"싛"#, r#"싛"#, r#"싛"#, r#"싛"#),
        (r#"시"#, r#"시"#, r#"시"#, r#"시"#, r#"시"#),
        (r#"식"#, r#"식"#, r#"식"#, r#"식"#, r#"식"#),
        (r#"싞"#, r#"싞"#, r#"싞"#, r#"싞"#, r#"싞"#),
        (r#"싟"#, r#"싟"#, r#"싟"#, r#"싟"#, r#"싟"#),
        (r#"신"#, r#"신"#, r#"신"#, r#"신"#, r#"신"#),
        (r#"싡"#, r#"싡"#, r#"싡"#, r#"싡"#, r#"싡"#),
        (r#"싢"#, r#"싢"#, r#"싢"#, r#"싢"#, r#"싢"#),
        (r#"싣"#, r#"싣"#, r#"싣"#, r#"싣"#, r#"싣"#),
        (r#"실"#, r#"실"#, r#"실"#, r#"실"#, r#"실"#),
        (r#"싥"#, r#"싥"#, r#"싥"#, r#"싥"#, r#"싥"#),
        (r#"싦"#, r#"싦"#, r#"싦"#, r#"싦"#, r#"싦"#),
        (r#"싧"#, r#"싧"#, r#"싧"#, r#"싧"#, r#"싧"#),
        (r#"싨"#, r#"싨"#, r#"싨"#, r#"싨"#, r#"싨"#),
        (r#"싩"#, r#"싩"#, r#"싩"#, r#"싩"#, r#"싩"#),
        (r#"싪"#, r#"싪"#, r#"싪"#, r#"싪"#, r#"싪"#),
        (r#"싫"#, r#"싫"#, r#"싫"#, r#"싫"#, r#"싫"#),
        (r#"심"#, r#"심"#, r#"심"#, r#"심"#, r#"심"#),
        (r#"십"#, r#"십"#, r#"십"#, r#"십"#, r#"십"#),
        (r#"싮"#, r#"싮"#, r#"싮"#, r#"싮"#, r#"싮"#),
        (r#"싯"#, r#"싯"#, r#"싯"#, r#"싯"#, r#"싯"#),
        (r#"싰"#, r#"싰"#, r#"싰"#, r#"싰"#, r#"싰"#),
        (r#"싱"#, r#"싱"#, r#"싱"#, r#"싱"#, r#"싱"#),
        (r#"싲"#, r#"싲"#, r#"싲"#, r#"싲"#, r#"싲"#),
        (r#"싳"#, r#"싳"#, r#"싳"#, r#"싳"#, r#"싳"#),
        (r#"싴"#, r#"싴"#, r#"싴"#, r#"싴"#, r#"싴"#),
        (r#"싵"#, r#"싵"#, r#"싵"#, r#"싵"#, r#"싵"#),
        (r#"싶"#, r#"싶"#, r#"싶"#, r#"싶"#, r#"싶"#),
        (r#"싷"#, r#"싷"#, r#"싷"#, r#"싷"#, r#"싷"#),
        (r#"싸"#, r#"싸"#, r#"싸"#, r#"싸"#, r#"싸"#),
        (r#"싹"#, r#"싹"#, r#"싹"#, r#"싹"#, r#"싹"#),
        (r#"싺"#, r#"싺"#, r#"싺"#, r#"싺"#, r#"싺"#),
        (r#"싻"#, r#"싻"#, r#"싻"#, r#"싻"#, r#"싻"#),
        (r#"싼"#, r#"싼"#, r#"싼"#, r#"싼"#, r#"싼"#),
        (r#"싽"#, r#"싽"#, r#"싽"#, r#"싽"#, r#"싽"#),
        (r#"싾"#, r#"싾"#, r#"싾"#, r#"싾"#, r#"싾"#),
        (r#"싿"#, r#"싿"#, r#"싿"#, r#"싿"#, r#"싿"#),
        (r#"쌀"#, r#"쌀"#, r#"쌀"#, r#"쌀"#, r#"쌀"#),
        (r#"쌁"#, r#"쌁"#, r#"쌁"#, r#"쌁"#, r#"쌁"#),
        (r#"쌂"#, r#"쌂"#, r#"쌂"#, r#"쌂"#, r#"쌂"#),
        (r#"쌃"#, r#"쌃"#, r#"쌃"#, r#"쌃"#, r#"쌃"#),
        (r#"쌄"#, r#"쌄"#, r#"쌄"#, r#"쌄"#, r#"쌄"#),
        (r#"쌅"#, r#"쌅"#, r#"쌅"#, r#"쌅"#, r#"쌅"#),
        (r#"쌆"#, r#"쌆"#, r#"쌆"#, r#"쌆"#, r#"쌆"#),
        (r#"쌇"#, r#"쌇"#, r#"쌇"#, r#"쌇"#, r#"쌇"#),
        (r#"쌈"#, r#"쌈"#, r#"쌈"#, r#"쌈"#, r#"쌈"#),
        (r#"쌉"#, r#"쌉"#, r#"쌉"#, r#"쌉"#, r#"쌉"#),
        (r#"쌊"#, r#"쌊"#, r#"쌊"#, r#"쌊"#, r#"쌊"#),
        (r#"쌋"#, r#"쌋"#, r#"쌋"#, r#"쌋"#, r#"쌋"#),
        (r#"쌌"#, r#"쌌"#, r#"쌌"#, r#"쌌"#, r#"쌌"#),
        (r#"쌍"#, r#"쌍"#, r#"쌍"#, r#"쌍"#, r#"쌍"#),
        (r#"쌎"#, r#"쌎"#, r#"쌎"#, r#"쌎"#, r#"쌎"#),
        (r#"쌏"#, r#"쌏"#, r#"쌏"#, r#"쌏"#, r#"쌏"#),
        (r#"쌐"#, r#"쌐"#, r#"쌐"#, r#"쌐"#, r#"쌐"#),
        (r#"쌑"#, r#"쌑"#, r#"쌑"#, r#"쌑"#, r#"쌑"#),
        (r#"쌒"#, r#"쌒"#, r#"쌒"#, r#"쌒"#, r#"쌒"#),
        (r#"쌓"#, r#"쌓"#, r#"쌓"#, r#"쌓"#, r#"쌓"#),
        (r#"쌔"#, r#"쌔"#, r#"쌔"#, r#"쌔"#, r#"쌔"#),
        (r#"쌕"#, r#"쌕"#, r#"쌕"#, r#"쌕"#, r#"쌕"#),
        (r#"쌖"#, r#"쌖"#, r#"쌖"#, r#"쌖"#, r#"쌖"#),
        (r#"쌗"#, r#"쌗"#, r#"쌗"#, r#"쌗"#, r#"쌗"#),
        (r#"쌘"#, r#"쌘"#, r#"쌘"#, r#"쌘"#, r#"쌘"#),
        (r#"쌙"#, r#"쌙"#, r#"쌙"#, r#"쌙"#, r#"쌙"#),
        (r#"쌚"#, r#"쌚"#, r#"쌚"#, r#"쌚"#, r#"쌚"#),
        (r#"쌛"#, r#"쌛"#, r#"쌛"#, r#"쌛"#, r#"쌛"#),
        (r#"쌜"#, r#"쌜"#, r#"쌜"#, r#"쌜"#, r#"쌜"#),
        (r#"쌝"#, r#"쌝"#, r#"쌝"#, r#"쌝"#, r#"쌝"#),
        (r#"쌞"#, r#"쌞"#, r#"쌞"#, r#"쌞"#, r#"쌞"#),
        (r#"쌟"#, r#"쌟"#, r#"쌟"#, r#"쌟"#, r#"쌟"#),
        (r#"쌠"#, r#"쌠"#, r#"쌠"#, r#"쌠"#, r#"쌠"#),
        (r#"쌡"#, r#"쌡"#, r#"쌡"#, r#"쌡"#, r#"쌡"#),
        (r#"쌢"#, r#"쌢"#, r#"쌢"#, r#"쌢"#, r#"쌢"#),
        (r#"쌣"#, r#"쌣"#, r#"쌣"#, r#"쌣"#, r#"쌣"#),
        (r#"쌤"#, r#"쌤"#, r#"쌤"#, r#"쌤"#, r#"쌤"#),
        (r#"쌥"#, r#"쌥"#, r#"쌥"#, r#"쌥"#, r#"쌥"#),
        (r#"쌦"#, r#"쌦"#, r#"쌦"#, r#"쌦"#, r#"쌦"#),
        (r#"쌧"#, r#"쌧"#, r#"쌧"#, r#"쌧"#, r#"쌧"#),
        (r#"쌨"#, r#"쌨"#, r#"쌨"#, r#"쌨"#, r#"쌨"#),
        (r#"쌩"#, r#"쌩"#, r#"쌩"#, r#"쌩"#, r#"쌩"#),
        (r#"쌪"#, r#"쌪"#, r#"쌪"#, r#"쌪"#, r#"쌪"#),
        (r#"쌫"#, r#"쌫"#, r#"쌫"#, r#"쌫"#, r#"쌫"#),
        (r#"쌬"#, r#"쌬"#, r#"쌬"#, r#"쌬"#, r#"쌬"#),
        (r#"쌭"#, r#"쌭"#, r#"쌭"#, r#"쌭"#, r#"쌭"#),
        (r#"쌮"#, r#"쌮"#, r#"쌮"#, r#"쌮"#, r#"쌮"#),
        (r#"쌯"#, r#"쌯"#, r#"쌯"#, r#"쌯"#, r#"쌯"#),
        (r#"쌰"#, r#"쌰"#, r#"쌰"#, r#"쌰"#, r#"쌰"#),
        (r#"쌱"#, r#"쌱"#, r#"쌱"#, r#"쌱"#, r#"쌱"#),
        (r#"쌲"#, r#"쌲"#, r#"쌲"#, r#"쌲"#, r#"쌲"#),
        (r#"쌳"#, r#"쌳"#, r#"쌳"#, r#"쌳"#, r#"쌳"#),
        (r#"쌴"#, r#"쌴"#, r#"쌴"#, r#"쌴"#, r#"쌴"#),
        (r#"쌵"#, r#"쌵"#, r#"쌵"#, r#"쌵"#, r#"쌵"#),
        (r#"쌶"#, r#"쌶"#, r#"쌶"#, r#"쌶"#, r#"쌶"#),
        (r#"쌷"#, r#"쌷"#, r#"쌷"#, r#"쌷"#, r#"쌷"#),
        (r#"쌸"#, r#"쌸"#, r#"쌸"#, r#"쌸"#, r#"쌸"#),
        (r#"쌹"#, r#"쌹"#, r#"쌹"#, r#"쌹"#, r#"쌹"#),
        (r#"쌺"#, r#"쌺"#, r#"쌺"#, r#"쌺"#, r#"쌺"#),
        (r#"쌻"#, r#"쌻"#, r#"쌻"#, r#"쌻"#, r#"쌻"#),
        (r#"쌼"#, r#"쌼"#, r#"쌼"#, r#"쌼"#, r#"쌼"#),
        (r#"쌽"#, r#"쌽"#, r#"쌽"#, r#"쌽"#, r#"쌽"#),
        (r#"쌾"#, r#"쌾"#, r#"쌾"#, r#"쌾"#, r#"쌾"#),
        (r#"쌿"#, r#"쌿"#, r#"쌿"#, r#"쌿"#, r#"쌿"#),
        (r#"썀"#, r#"썀"#, r#"썀"#, r#"썀"#, r#"썀"#),
        (r#"썁"#, r#"썁"#, r#"썁"#, r#"썁"#, r#"썁"#),
        (r#"썂"#, r#"썂"#, r#"썂"#, r#"썂"#, r#"썂"#),
        (r#"썃"#, r#"썃"#, r#"썃"#, r#"썃"#, r#"썃"#),
        (r#"썄"#, r#"썄"#, r#"썄"#, r#"썄"#, r#"썄"#),
        (r#"썅"#, r#"썅"#, r#"썅"#, r#"썅"#, r#"썅"#),
        (r#"썆"#, r#"썆"#, r#"썆"#, r#"썆"#, r#"썆"#),
        (r#"썇"#, r#"썇"#, r#"썇"#, r#"썇"#, r#"썇"#),
        (r#"썈"#, r#"썈"#, r#"썈"#, r#"썈"#, r#"썈"#),
        (r#"썉"#, r#"썉"#, r#"썉"#, r#"썉"#, r#"썉"#),
        (r#"썊"#, r#"썊"#, r#"썊"#, r#"썊"#, r#"썊"#),
        (r#"썋"#, r#"썋"#, r#"썋"#, r#"썋"#, r#"썋"#),
        (r#"썌"#, r#"썌"#, r#"썌"#, r#"썌"#, r#"썌"#),
        (r#"썍"#, r#"썍"#, r#"썍"#, r#"썍"#, r#"썍"#),
        (r#"썎"#, r#"썎"#, r#"썎"#, r#"썎"#, r#"썎"#),
        (r#"썏"#, r#"썏"#, r#"썏"#, r#"썏"#, r#"썏"#),
        (r#"썐"#, r#"썐"#, r#"썐"#, r#"썐"#, r#"썐"#),
        (r#"썑"#, r#"썑"#, r#"썑"#, r#"썑"#, r#"썑"#),
        (r#"썒"#, r#"썒"#, r#"썒"#, r#"썒"#, r#"썒"#),
        (r#"썓"#, r#"썓"#, r#"썓"#, r#"썓"#, r#"썓"#),
        (r#"썔"#, r#"썔"#, r#"썔"#, r#"썔"#, r#"썔"#),
        (r#"썕"#, r#"썕"#, r#"썕"#, r#"썕"#, r#"썕"#),
        (r#"썖"#, r#"썖"#, r#"썖"#, r#"썖"#, r#"썖"#),
        (r#"썗"#, r#"썗"#, r#"썗"#, r#"썗"#, r#"썗"#),
        (r#"썘"#, r#"썘"#, r#"썘"#, r#"썘"#, r#"썘"#),
        (r#"썙"#, r#"썙"#, r#"썙"#, r#"썙"#, r#"썙"#),
        (r#"썚"#, r#"썚"#, r#"썚"#, r#"썚"#, r#"썚"#),
        (r#"썛"#, r#"썛"#, r#"썛"#, r#"썛"#, r#"썛"#),
        (r#"썜"#, r#"썜"#, r#"썜"#, r#"썜"#, r#"썜"#),
        (r#"썝"#, r#"썝"#, r#"썝"#, r#"썝"#, r#"썝"#),
        (r#"썞"#, r#"썞"#, r#"썞"#, r#"썞"#, r#"썞"#),
        (r#"썟"#, r#"썟"#, r#"썟"#, r#"썟"#, r#"썟"#),
        (r#"썠"#, r#"썠"#, r#"썠"#, r#"썠"#, r#"썠"#),
        (r#"썡"#, r#"썡"#, r#"썡"#, r#"썡"#, r#"썡"#),
        (r#"썢"#, r#"썢"#, r#"썢"#, r#"썢"#, r#"썢"#),
        (r#"썣"#, r#"썣"#, r#"썣"#, r#"썣"#, r#"썣"#),
        (r#"썤"#, r#"썤"#, r#"썤"#, r#"썤"#, r#"썤"#),
        (r#"썥"#, r#"썥"#, r#"썥"#, r#"썥"#, r#"썥"#),
        (r#"썦"#, r#"썦"#, r#"썦"#, r#"썦"#, r#"썦"#),
        (r#"썧"#, r#"썧"#, r#"썧"#, r#"썧"#, r#"썧"#),
        (r#"써"#, r#"써"#, r#"써"#, r#"써"#, r#"써"#),
        (r#"썩"#, r#"썩"#, r#"썩"#, r#"썩"#, r#"썩"#),
        (r#"썪"#, r#"썪"#, r#"썪"#, r#"썪"#, r#"썪"#),
        (r#"썫"#, r#"썫"#, r#"썫"#, r#"썫"#, r#"썫"#),
        (r#"썬"#, r#"썬"#, r#"썬"#, r#"썬"#, r#"썬"#),
        (r#"썭"#, r#"썭"#, r#"썭"#, r#"썭"#, r#"썭"#),
        (r#"썮"#, r#"썮"#, r#"썮"#, r#"썮"#, r#"썮"#),
        (r#"썯"#, r#"썯"#, r#"썯"#, r#"썯"#, r#"썯"#),
        (r#"썰"#, r#"썰"#, r#"썰"#, r#"썰"#, r#"썰"#),
        (r#"썱"#, r#"썱"#, r#"썱"#, r#"썱"#, r#"썱"#),
        (r#"썲"#, r#"썲"#, r#"썲"#, r#"썲"#, r#"썲"#),
        (r#"썳"#, r#"썳"#, r#"썳"#, r#"썳"#, r#"썳"#),
        (r#"썴"#, r#"썴"#, r#"썴"#, r#"썴"#, r#"썴"#),
        (r#"썵"#, r#"썵"#, r#"썵"#, r#"썵"#, r#"썵"#),
        (r#"썶"#, r#"썶"#, r#"썶"#, r#"썶"#, r#"썶"#),
        (r#"썷"#, r#"썷"#, r#"썷"#, r#"썷"#, r#"썷"#),
        (r#"썸"#, r#"썸"#, r#"썸"#, r#"썸"#, r#"썸"#),
        (r#"썹"#, r#"썹"#, r#"썹"#, r#"썹"#, r#"썹"#),
        (r#"썺"#, r#"썺"#, r#"썺"#, r#"썺"#, r#"썺"#),
        (r#"썻"#, r#"썻"#, r#"썻"#, r#"썻"#, r#"썻"#),
        (r#"썼"#, r#"썼"#, r#"썼"#, r#"썼"#, r#"썼"#),
        (r#"썽"#, r#"썽"#, r#"썽"#, r#"썽"#, r#"썽"#),
        (r#"썾"#, r#"썾"#, r#"썾"#, r#"썾"#, r#"썾"#),
        (r#"썿"#, r#"썿"#, r#"썿"#, r#"썿"#, r#"썿"#),
        (r#"쎀"#, r#"쎀"#, r#"쎀"#, r#"쎀"#, r#"쎀"#),
        (r#"쎁"#, r#"쎁"#, r#"쎁"#, r#"쎁"#, r#"쎁"#),
        (r#"쎂"#, r#"쎂"#, r#"쎂"#, r#"쎂"#, r#"쎂"#),
        (r#"쎃"#, r#"쎃"#, r#"쎃"#, r#"쎃"#, r#"쎃"#),
        (r#"쎄"#, r#"쎄"#, r#"쎄"#, r#"쎄"#, r#"쎄"#),
        (r#"쎅"#, r#"쎅"#, r#"쎅"#, r#"쎅"#, r#"쎅"#),
        (r#"쎆"#, r#"쎆"#, r#"쎆"#, r#"쎆"#, r#"쎆"#),
        (r#"쎇"#, r#"쎇"#, r#"쎇"#, r#"쎇"#, r#"쎇"#),
        (r#"쎈"#, r#"쎈"#, r#"쎈"#, r#"쎈"#, r#"쎈"#),
        (r#"쎉"#, r#"쎉"#, r#"쎉"#, r#"쎉"#, r#"쎉"#),
        (r#"쎊"#, r#"쎊"#, r#"쎊"#, r#"쎊"#, r#"쎊"#),
        (r#"쎋"#, r#"쎋"#, r#"쎋"#, r#"쎋"#, r#"쎋"#),
        (r#"쎌"#, r#"쎌"#, r#"쎌"#, r#"쎌"#, r#"쎌"#),
        (r#"쎍"#, r#"쎍"#, r#"쎍"#, r#"쎍"#, r#"쎍"#),
        (r#"쎎"#, r#"쎎"#, r#"쎎"#, r#"쎎"#, r#"쎎"#),
        (r#"쎏"#, r#"쎏"#, r#"쎏"#, r#"쎏"#, r#"쎏"#),
        (r#"쎐"#, r#"쎐"#, r#"쎐"#, r#"쎐"#, r#"쎐"#),
        (r#"쎑"#, r#"쎑"#, r#"쎑"#, r#"쎑"#, r#"쎑"#),
        (r#"쎒"#, r#"쎒"#, r#"쎒"#, r#"쎒"#, r#"쎒"#),
        (r#"쎓"#, r#"쎓"#, r#"쎓"#, r#"쎓"#, r#"쎓"#),
        (r#"쎔"#, r#"쎔"#, r#"쎔"#, r#"쎔"#, r#"쎔"#),
        (r#"쎕"#, r#"쎕"#, r#"쎕"#, r#"쎕"#, r#"쎕"#),
        (r#"쎖"#, r#"쎖"#, r#"쎖"#, r#"쎖"#, r#"쎖"#),
        (r#"쎗"#, r#"쎗"#, r#"쎗"#, r#"쎗"#, r#"쎗"#),
        (r#"쎘"#, r#"쎘"#, r#"쎘"#, r#"쎘"#, r#"쎘"#),
        (r#"쎙"#, r#"쎙"#, r#"쎙"#, r#"쎙"#, r#"쎙"#),
        (r#"쎚"#, r#"쎚"#, r#"쎚"#, r#"쎚"#, r#"쎚"#),
        (r#"쎛"#, r#"쎛"#, r#"쎛"#, r#"쎛"#, r#"쎛"#),
        (r#"쎜"#, r#"쎜"#, r#"쎜"#, r#"쎜"#, r#"쎜"#),
        (r#"쎝"#, r#"쎝"#, r#"쎝"#, r#"쎝"#, r#"쎝"#),
        (r#"쎞"#, r#"쎞"#, r#"쎞"#, r#"쎞"#, r#"쎞"#),
        (r#"쎟"#, r#"쎟"#, r#"쎟"#, r#"쎟"#, r#"쎟"#),
        (r#"쎠"#, r#"쎠"#, r#"쎠"#, r#"쎠"#, r#"쎠"#),
        (r#"쎡"#, r#"쎡"#, r#"쎡"#, r#"쎡"#, r#"쎡"#),
        (r#"쎢"#, r#"쎢"#, r#"쎢"#, r#"쎢"#, r#"쎢"#),
        (r#"쎣"#, r#"쎣"#, r#"쎣"#, r#"쎣"#, r#"쎣"#),
        (r#"쎤"#, r#"쎤"#, r#"쎤"#, r#"쎤"#, r#"쎤"#),
        (r#"쎥"#, r#"쎥"#, r#"쎥"#, r#"쎥"#, r#"쎥"#),
        (r#"쎦"#, r#"쎦"#, r#"쎦"#, r#"쎦"#, r#"쎦"#),
        (r#"쎧"#, r#"쎧"#, r#"쎧"#, r#"쎧"#, r#"쎧"#),
        (r#"쎨"#, r#"쎨"#, r#"쎨"#, r#"쎨"#, r#"쎨"#),
        (r#"쎩"#, r#"쎩"#, r#"쎩"#, r#"쎩"#, r#"쎩"#),
        (r#"쎪"#, r#"쎪"#, r#"쎪"#, r#"쎪"#, r#"쎪"#),
        (r#"쎫"#, r#"쎫"#, r#"쎫"#, r#"쎫"#, r#"쎫"#),
        (r#"쎬"#, r#"쎬"#, r#"쎬"#, r#"쎬"#, r#"쎬"#),
        (r#"쎭"#, r#"쎭"#, r#"쎭"#, r#"쎭"#, r#"쎭"#),
        (r#"쎮"#, r#"쎮"#, r#"쎮"#, r#"쎮"#, r#"쎮"#),
        (r#"쎯"#, r#"쎯"#, r#"쎯"#, r#"쎯"#, r#"쎯"#),
        (r#"쎰"#, r#"쎰"#, r#"쎰"#, r#"쎰"#, r#"쎰"#),
        (r#"쎱"#, r#"쎱"#, r#"쎱"#, r#"쎱"#, r#"쎱"#),
        (r#"쎲"#, r#"쎲"#, r#"쎲"#, r#"쎲"#, r#"쎲"#),
        (r#"쎳"#, r#"쎳"#, r#"쎳"#, r#"쎳"#, r#"쎳"#),
        (r#"쎴"#, r#"쎴"#, r#"쎴"#, r#"쎴"#, r#"쎴"#),
        (r#"쎵"#, r#"쎵"#, r#"쎵"#, r#"쎵"#, r#"쎵"#),
        (r#"쎶"#, r#"쎶"#, r#"쎶"#, r#"쎶"#, r#"쎶"#),
        (r#"쎷"#, r#"쎷"#, r#"쎷"#, r#"쎷"#, r#"쎷"#),
        (r#"쎸"#, r#"쎸"#, r#"쎸"#, r#"쎸"#, r#"쎸"#),
        (r#"쎹"#, r#"쎹"#, r#"쎹"#, r#"쎹"#, r#"쎹"#),
        (r#"쎺"#, r#"쎺"#, r#"쎺"#, r#"쎺"#, r#"쎺"#),
        (r#"쎻"#, r#"쎻"#, r#"쎻"#, r#"쎻"#, r#"쎻"#),
        (r#"쎼"#, r#"쎼"#, r#"쎼"#, r#"쎼"#, r#"쎼"#),
        (r#"쎽"#, r#"쎽"#, r#"쎽"#, r#"쎽"#, r#"쎽"#),
        (r#"쎾"#, r#"쎾"#, r#"쎾"#, r#"쎾"#, r#"쎾"#),
        (r#"쎿"#, r#"쎿"#, r#"쎿"#, r#"쎿"#, r#"쎿"#),
        (r#"쏀"#, r#"쏀"#, r#"쏀"#, r#"쏀"#, r#"쏀"#),
        (r#"쏁"#, r#"쏁"#, r#"쏁"#, r#"쏁"#, r#"쏁"#),
        (r#"쏂"#, r#"쏂"#, r#"쏂"#, r#"쏂"#, r#"쏂"#),
        (r#"쏃"#, r#"쏃"#, r#"쏃"#, r#"쏃"#, r#"쏃"#),
        (r#"쏄"#, r#"쏄"#, r#"쏄"#, r#"쏄"#, r#"쏄"#),
        (r#"쏅"#, r#"쏅"#, r#"쏅"#, r#"쏅"#, r#"쏅"#),
        (r#"쏆"#, r#"쏆"#, r#"쏆"#, r#"쏆"#, r#"쏆"#),
        (r#"쏇"#, r#"쏇"#, r#"쏇"#, r#"쏇"#, r#"쏇"#),
        (r#"쏈"#, r#"쏈"#, r#"쏈"#, r#"쏈"#, r#"쏈"#),
        (r#"쏉"#, r#"쏉"#, r#"쏉"#, r#"쏉"#, r#"쏉"#),
        (r#"쏊"#, r#"쏊"#, r#"쏊"#, r#"쏊"#, r#"쏊"#),
        (r#"쏋"#, r#"쏋"#, r#"쏋"#, r#"쏋"#, r#"쏋"#),
        (r#"쏌"#, r#"쏌"#, r#"쏌"#, r#"쏌"#, r#"쏌"#),
        (r#"쏍"#, r#"쏍"#, r#"쏍"#, r#"쏍"#, r#"쏍"#),
        (r#"쏎"#, r#"쏎"#, r#"쏎"#, r#"쏎"#, r#"쏎"#),
        (r#"쏏"#, r#"쏏"#, r#"쏏"#, r#"쏏"#, r#"쏏"#),
        (r#"쏐"#, r#"쏐"#, r#"쏐"#, r#"쏐"#, r#"쏐"#),
        (r#"쏑"#, r#"쏑"#, r#"쏑"#, r#"쏑"#, r#"쏑"#),
        (r#"쏒"#, r#"쏒"#, r#"쏒"#, r#"쏒"#, r#"쏒"#),
        (r#"쏓"#, r#"쏓"#, r#"쏓"#, r#"쏓"#, r#"쏓"#),
        (r#"쏔"#, r#"쏔"#, r#"쏔"#, r#"쏔"#, r#"쏔"#),
        (r#"쏕"#, r#"쏕"#, r#"쏕"#, r#"쏕"#, r#"쏕"#),
        (r#"쏖"#, r#"쏖"#, r#"쏖"#, r#"쏖"#, r#"쏖"#),
        (r#"쏗"#, r#"쏗"#, r#"쏗"#, r#"쏗"#, r#"쏗"#),
        (r#"쏘"#, r#"쏘"#, r#"쏘"#, r#"쏘"#, r#"쏘"#),
        (r#"쏙"#, r#"쏙"#, r#"쏙"#, r#"쏙"#, r#"쏙"#),
        (r#"쏚"#, r#"쏚"#, r#"쏚"#, r#"쏚"#, r#"쏚"#),
        (r#"쏛"#, r#"쏛"#, r#"쏛"#, r#"쏛"#, r#"쏛"#),
        (r#"쏜"#, r#"쏜"#, r#"쏜"#, r#"쏜"#, r#"쏜"#),
        (r#"쏝"#, r#"쏝"#, r#"쏝"#, r#"쏝"#, r#"쏝"#),
        (r#"쏞"#, r#"쏞"#, r#"쏞"#, r#"쏞"#, r#"쏞"#),
        (r#"쏟"#, r#"쏟"#, r#"쏟"#, r#"쏟"#, r#"쏟"#),
        (r#"쏠"#, r#"쏠"#, r#"쏠"#, r#"쏠"#, r#"쏠"#),
        (r#"쏡"#, r#"쏡"#, r#"쏡"#, r#"쏡"#, r#"쏡"#),
        (r#"쏢"#, r#"쏢"#, r#"쏢"#, r#"쏢"#, r#"쏢"#),
        (r#"쏣"#, r#"쏣"#, r#"쏣"#, r#"쏣"#, r#"쏣"#),
        (r#"쏤"#, r#"쏤"#, r#"쏤"#, r#"쏤"#, r#"쏤"#),
        (r#"쏥"#, r#"쏥"#, r#"쏥"#, r#"쏥"#, r#"쏥"#),
        (r#"쏦"#, r#"쏦"#, r#"쏦"#, r#"쏦"#, r#"쏦"#),
        (r#"쏧"#, r#"쏧"#, r#"쏧"#, r#"쏧"#, r#"쏧"#),
        (r#"쏨"#, r#"쏨"#, r#"쏨"#, r#"쏨"#, r#"쏨"#),
        (r#"쏩"#, r#"쏩"#, r#"쏩"#, r#"쏩"#, r#"쏩"#),
        (r#"쏪"#, r#"쏪"#, r#"쏪"#, r#"쏪"#, r#"쏪"#),
        (r#"쏫"#, r#"쏫"#, r#"쏫"#, r#"쏫"#, r#"쏫"#),
        (r#"쏬"#, r#"쏬"#, r#"쏬"#, r#"쏬"#, r#"쏬"#),
        (r#"쏭"#, r#"쏭"#, r#"쏭"#, r#"쏭"#, r#"쏭"#),
        (r#"쏮"#, r#"쏮"#, r#"쏮"#, r#"쏮"#, r#"쏮"#),
        (r#"쏯"#, r#"쏯"#, r#"쏯"#, r#"쏯"#, r#"쏯"#),
        (r#"쏰"#, r#"쏰"#, r#"쏰"#, r#"쏰"#, r#"쏰"#),
        (r#"쏱"#, r#"쏱"#, r#"쏱"#, r#"쏱"#, r#"쏱"#),
        (r#"쏲"#, r#"쏲"#, r#"쏲"#, r#"쏲"#, r#"쏲"#),
        (r#"쏳"#, r#"쏳"#, r#"쏳"#, r#"쏳"#, r#"쏳"#),
        (r#"쏴"#, r#"쏴"#, r#"쏴"#, r#"쏴"#, r#"쏴"#),
        (r#"쏵"#, r#"쏵"#, r#"쏵"#, r#"쏵"#, r#"쏵"#),
        (r#"쏶"#, r#"쏶"#, r#"쏶"#, r#"쏶"#, r#"쏶"#),
        (r#"쏷"#, r#"쏷"#, r#"쏷"#, r#"쏷"#, r#"쏷"#),
        (r#"쏸"#, r#"쏸"#, r#"쏸"#, r#"쏸"#, r#"쏸"#),
        (r#"쏹"#, r#"쏹"#, r#"쏹"#, r#"쏹"#, r#"쏹"#),
        (r#"쏺"#, r#"쏺"#, r#"쏺"#, r#"쏺"#, r#"쏺"#),
        (r#"쏻"#, r#"쏻"#, r#"쏻"#, r#"쏻"#, r#"쏻"#),
        (r#"쏼"#, r#"쏼"#, r#"쏼"#, r#"쏼"#, r#"쏼"#),
        (r#"쏽"#, r#"쏽"#, r#"쏽"#, r#"쏽"#, r#"쏽"#),
        (r#"쏾"#, r#"쏾"#, r#"쏾"#, r#"쏾"#, r#"쏾"#),
        (r#"쏿"#, r#"쏿"#, r#"쏿"#, r#"쏿"#, r#"쏿"#),
        (r#"쐀"#, r#"쐀"#, r#"쐀"#, r#"쐀"#, r#"쐀"#),
        (r#"쐁"#, r#"쐁"#, r#"쐁"#, r#"쐁"#, r#"쐁"#),
        (r#"쐂"#, r#"쐂"#, r#"쐂"#, r#"쐂"#, r#"쐂"#),
        (r#"쐃"#, r#"쐃"#, r#"쐃"#, r#"쐃"#, r#"쐃"#),
        (r#"쐄"#, r#"쐄"#, r#"쐄"#, r#"쐄"#, r#"쐄"#),
        (r#"쐅"#, r#"쐅"#, r#"쐅"#, r#"쐅"#, r#"쐅"#),
        (r#"쐆"#, r#"쐆"#, r#"쐆"#, r#"쐆"#, r#"쐆"#),
        (r#"쐇"#, r#"쐇"#, r#"쐇"#, r#"쐇"#, r#"쐇"#),
        (r#"쐈"#, r#"쐈"#, r#"쐈"#, r#"쐈"#, r#"쐈"#),
        (r#"쐉"#, r#"쐉"#, r#"쐉"#, r#"쐉"#, r#"쐉"#),
        (r#"쐊"#, r#"쐊"#, r#"쐊"#, r#"쐊"#, r#"쐊"#),
        (r#"쐋"#, r#"쐋"#, r#"쐋"#, r#"쐋"#, r#"쐋"#),
        (r#"쐌"#, r#"쐌"#, r#"쐌"#, r#"쐌"#, r#"쐌"#),
        (r#"쐍"#, r#"쐍"#, r#"쐍"#, r#"쐍"#, r#"쐍"#),
        (r#"쐎"#, r#"쐎"#, r#"쐎"#, r#"쐎"#, r#"쐎"#),
        (r#"쐏"#, r#"쐏"#, r#"쐏"#, r#"쐏"#, r#"쐏"#),
        (r#"쐐"#, r#"쐐"#, r#"쐐"#, r#"쐐"#, r#"쐐"#),
        (r#"쐑"#, r#"쐑"#, r#"쐑"#, r#"쐑"#, r#"쐑"#),
        (r#"쐒"#, r#"쐒"#, r#"쐒"#, r#"쐒"#, r#"쐒"#),
        (r#"쐓"#, r#"쐓"#, r#"쐓"#, r#"쐓"#, r#"쐓"#),
        (r#"쐔"#, r#"쐔"#, r#"쐔"#, r#"쐔"#, r#"쐔"#),
        (r#"쐕"#, r#"쐕"#, r#"쐕"#, r#"쐕"#, r#"쐕"#),
        (r#"쐖"#, r#"쐖"#, r#"쐖"#, r#"쐖"#, r#"쐖"#),
        (r#"쐗"#, r#"쐗"#, r#"쐗"#, r#"쐗"#, r#"쐗"#),
        (r#"쐘"#, r#"쐘"#, r#"쐘"#, r#"쐘"#, r#"쐘"#),
        (r#"쐙"#, r#"쐙"#, r#"쐙"#, r#"쐙"#, r#"쐙"#),
        (r#"쐚"#, r#"쐚"#, r#"쐚"#, r#"쐚"#, r#"쐚"#),
        (r#"쐛"#, r#"쐛"#, r#"쐛"#, r#"쐛"#, r#"쐛"#),
        (r#"쐜"#, r#"쐜"#, r#"쐜"#, r#"쐜"#, r#"쐜"#),
        (r#"쐝"#, r#"쐝"#, r#"쐝"#, r#"쐝"#, r#"쐝"#),
        (r#"쐞"#, r#"쐞"#, r#"쐞"#, r#"쐞"#, r#"쐞"#),
        (r#"쐟"#, r#"쐟"#, r#"쐟"#, r#"쐟"#, r#"쐟"#),
        (r#"쐠"#, r#"쐠"#, r#"쐠"#, r#"쐠"#, r#"쐠"#),
        (r#"쐡"#, r#"쐡"#, r#"쐡"#, r#"쐡"#, r#"쐡"#),
        (r#"쐢"#, r#"쐢"#, r#"쐢"#, r#"쐢"#, r#"쐢"#),
        (r#"쐣"#, r#"쐣"#, r#"쐣"#, r#"쐣"#, r#"쐣"#),
        (r#"쐤"#, r#"쐤"#, r#"쐤"#, r#"쐤"#, r#"쐤"#),
        (r#"쐥"#, r#"쐥"#, r#"쐥"#, r#"쐥"#, r#"쐥"#),
        (r#"쐦"#, r#"쐦"#, r#"쐦"#, r#"쐦"#, r#"쐦"#),
        (r#"쐧"#, r#"쐧"#, r#"쐧"#, r#"쐧"#, r#"쐧"#),
        (r#"쐨"#, r#"쐨"#, r#"쐨"#, r#"쐨"#, r#"쐨"#),
        (r#"쐩"#, r#"쐩"#, r#"쐩"#, r#"쐩"#, r#"쐩"#),
        (r#"쐪"#, r#"쐪"#, r#"쐪"#, r#"쐪"#, r#"쐪"#),
        (r#"쐫"#, r#"쐫"#, r#"쐫"#, r#"쐫"#, r#"쐫"#),
        (r#"쐬"#, r#"쐬"#, r#"쐬"#, r#"쐬"#, r#"쐬"#),
        (r#"쐭"#, r#"쐭"#, r#"쐭"#, r#"쐭"#, r#"쐭"#),
        (r#"쐮"#, r#"쐮"#, r#"쐮"#, r#"쐮"#, r#"쐮"#),
        (r#"쐯"#, r#"쐯"#, r#"쐯"#, r#"쐯"#, r#"쐯"#),
        (r#"쐰"#, r#"쐰"#, r#"쐰"#, r#"쐰"#, r#"쐰"#),
        (r#"쐱"#, r#"쐱"#, r#"쐱"#, r#"쐱"#, r#"쐱"#),
        (r#"쐲"#, r#"쐲"#, r#"쐲"#, r#"쐲"#, r#"쐲"#),
        (r#"쐳"#, r#"쐳"#, r#"쐳"#, r#"쐳"#, r#"쐳"#),
        (r#"쐴"#, r#"쐴"#, r#"쐴"#, r#"쐴"#, r#"쐴"#),
        (r#"쐵"#, r#"쐵"#, r#"쐵"#, r#"쐵"#, r#"쐵"#),
        (r#"쐶"#, r#"쐶"#, r#"쐶"#, r#"쐶"#, r#"쐶"#),
        (r#"쐷"#, r#"쐷"#, r#"쐷"#, r#"쐷"#, r#"쐷"#),
        (r#"쐸"#, r#"쐸"#, r#"쐸"#, r#"쐸"#, r#"쐸"#),
        (r#"쐹"#, r#"쐹"#, r#"쐹"#, r#"쐹"#, r#"쐹"#),
        (r#"쐺"#, r#"쐺"#, r#"쐺"#, r#"쐺"#, r#"쐺"#),
        (r#"쐻"#, r#"쐻"#, r#"쐻"#, r#"쐻"#, r#"쐻"#),
        (r#"쐼"#, r#"쐼"#, r#"쐼"#, r#"쐼"#, r#"쐼"#),
        (r#"쐽"#, r#"쐽"#, r#"쐽"#, r#"쐽"#, r#"쐽"#),
        (r#"쐾"#, r#"쐾"#, r#"쐾"#, r#"쐾"#, r#"쐾"#),
        (r#"쐿"#, r#"쐿"#, r#"쐿"#, r#"쐿"#, r#"쐿"#),
        (r#"쑀"#, r#"쑀"#, r#"쑀"#, r#"쑀"#, r#"쑀"#),
        (r#"쑁"#, r#"쑁"#, r#"쑁"#, r#"쑁"#, r#"쑁"#),
        (r#"쑂"#, r#"쑂"#, r#"쑂"#, r#"쑂"#, r#"쑂"#),
        (r#"쑃"#, r#"쑃"#, r#"쑃"#, r#"쑃"#, r#"쑃"#),
        (r#"쑄"#, r#"쑄"#, r#"쑄"#, r#"쑄"#, r#"쑄"#),
        (r#"쑅"#, r#"쑅"#, r#"쑅"#, r#"쑅"#, r#"쑅"#),
        (r#"쑆"#, r#"쑆"#, r#"쑆"#, r#"쑆"#, r#"쑆"#),
        (r#"쑇"#, r#"쑇"#, r#"쑇"#, r#"쑇"#, r#"쑇"#),
        (r#"쑈"#, r#"쑈"#, r#"쑈"#, r#"쑈"#, r#"쑈"#),
        (r#"쑉"#, r#"쑉"#, r#"쑉"#, r#"쑉"#, r#"쑉"#),
        (r#"쑊"#, r#"쑊"#, r#"쑊"#, r#"쑊"#, r#"쑊"#),
        (r#"쑋"#, r#"쑋"#, r#"쑋"#, r#"쑋"#, r#"쑋"#),
        (r#"쑌"#, r#"쑌"#, r#"쑌"#, r#"쑌"#, r#"쑌"#),
        (r#"쑍"#, r#"쑍"#, r#"쑍"#, r#"쑍"#, r#"쑍"#),
        (r#"쑎"#, r#"쑎"#, r#"쑎"#, r#"쑎"#, r#"쑎"#),
        (r#"쑏"#, r#"쑏"#, r#"쑏"#, r#"쑏"#, r#"쑏"#),
        (r#"쑐"#, r#"쑐"#, r#"쑐"#, r#"쑐"#, r#"쑐"#),
        (r#"쑑"#, r#"쑑"#, r#"쑑"#, r#"쑑"#, r#"쑑"#),
        (r#"쑒"#, r#"쑒"#, r#"쑒"#, r#"쑒"#, r#"쑒"#),
        (r#"쑓"#, r#"쑓"#, r#"쑓"#, r#"쑓"#, r#"쑓"#),
        (r#"쑔"#, r#"쑔"#, r#"쑔"#, r#"쑔"#, r#"쑔"#),
        (r#"쑕"#, r#"쑕"#, r#"쑕"#, r#"쑕"#, r#"쑕"#),
        (r#"쑖"#, r#"쑖"#, r#"쑖"#, r#"쑖"#, r#"쑖"#),
        (r#"쑗"#, r#"쑗"#, r#"쑗"#, r#"쑗"#, r#"쑗"#),
        (r#"쑘"#, r#"쑘"#, r#"쑘"#, r#"쑘"#, r#"쑘"#),
        (r#"쑙"#, r#"쑙"#, r#"쑙"#, r#"쑙"#, r#"쑙"#),
        (r#"쑚"#, r#"쑚"#, r#"쑚"#, r#"쑚"#, r#"쑚"#),
        (r#"쑛"#, r#"쑛"#, r#"쑛"#, r#"쑛"#, r#"쑛"#),
        (r#"쑜"#, r#"쑜"#, r#"쑜"#, r#"쑜"#, r#"쑜"#),
        (r#"쑝"#, r#"쑝"#, r#"쑝"#, r#"쑝"#, r#"쑝"#),
        (r#"쑞"#, r#"쑞"#, r#"쑞"#, r#"쑞"#, r#"쑞"#),
        (r#"쑟"#, r#"쑟"#, r#"쑟"#, r#"쑟"#, r#"쑟"#),
        (r#"쑠"#, r#"쑠"#, r#"쑠"#, r#"쑠"#, r#"쑠"#),
        (r#"쑡"#, r#"쑡"#, r#"쑡"#, r#"쑡"#, r#"쑡"#),
        (r#"쑢"#, r#"쑢"#, r#"쑢"#, r#"쑢"#, r#"쑢"#),
        (r#"쑣"#, r#"쑣"#, r#"쑣"#, r#"쑣"#, r#"쑣"#),
        (r#"쑤"#, r#"쑤"#, r#"쑤"#, r#"쑤"#, r#"쑤"#),
        (r#"쑥"#, r#"쑥"#, r#"쑥"#, r#"쑥"#, r#"쑥"#),
        (r#"쑦"#, r#"쑦"#, r#"쑦"#, r#"쑦"#, r#"쑦"#),
        (r#"쑧"#, r#"쑧"#, r#"쑧"#, r#"쑧"#, r#"쑧"#),
        (r#"쑨"#, r#"쑨"#, r#"쑨"#, r#"쑨"#, r#"쑨"#),
        (r#"쑩"#, r#"쑩"#, r#"쑩"#, r#"쑩"#, r#"쑩"#),
        (r#"쑪"#, r#"쑪"#, r#"쑪"#, r#"쑪"#, r#"쑪"#),
        (r#"쑫"#, r#"쑫"#, r#"쑫"#, r#"쑫"#, r#"쑫"#),
        (r#"쑬"#, r#"쑬"#, r#"쑬"#, r#"쑬"#, r#"쑬"#),
        (r#"쑭"#, r#"쑭"#, r#"쑭"#, r#"쑭"#, r#"쑭"#),
        (r#"쑮"#, r#"쑮"#, r#"쑮"#, r#"쑮"#, r#"쑮"#),
        (r#"쑯"#, r#"쑯"#, r#"쑯"#, r#"쑯"#, r#"쑯"#),
        (r#"쑰"#, r#"쑰"#, r#"쑰"#, r#"쑰"#, r#"쑰"#),
        (r#"쑱"#, r#"쑱"#, r#"쑱"#, r#"쑱"#, r#"쑱"#),
        (r#"쑲"#, r#"쑲"#, r#"쑲"#, r#"쑲"#, r#"쑲"#),
        (r#"쑳"#, r#"쑳"#, r#"쑳"#, r#"쑳"#, r#"쑳"#),
        (r#"쑴"#, r#"쑴"#, r#"쑴"#, r#"쑴"#, r#"쑴"#),
        (r#"쑵"#, r#"쑵"#, r#"쑵"#, r#"쑵"#, r#"쑵"#),
        (r#"쑶"#, r#"쑶"#, r#"쑶"#, r#"쑶"#, r#"쑶"#),
        (r#"쑷"#, r#"쑷"#, r#"쑷"#, r#"쑷"#, r#"쑷"#),
        (r#"쑸"#, r#"쑸"#, r#"쑸"#, r#"쑸"#, r#"쑸"#),
        (r#"쑹"#, r#"쑹"#, r#"쑹"#, r#"쑹"#, r#"쑹"#),
        (r#"쑺"#, r#"쑺"#, r#"쑺"#, r#"쑺"#, r#"쑺"#),
        (r#"쑻"#, r#"쑻"#, r#"쑻"#, r#"쑻"#, r#"쑻"#),
        (r#"쑼"#, r#"쑼"#, r#"쑼"#, r#"쑼"#, r#"쑼"#),
        (r#"쑽"#, r#"쑽"#, r#"쑽"#, r#"쑽"#, r#"쑽"#),
        (r#"쑾"#, r#"쑾"#, r#"쑾"#, r#"쑾"#, r#"쑾"#),
        (r#"쑿"#, r#"쑿"#, r#"쑿"#, r#"쑿"#, r#"쑿"#),
        (r#"쒀"#, r#"쒀"#, r#"쒀"#, r#"쒀"#, r#"쒀"#),
        (r#"쒁"#, r#"쒁"#, r#"쒁"#, r#"쒁"#, r#"쒁"#),
        (r#"쒂"#, r#"쒂"#, r#"쒂"#, r#"쒂"#, r#"쒂"#),
        (r#"쒃"#, r#"쒃"#, r#"쒃"#, r#"쒃"#, r#"쒃"#),
        (r#"쒄"#, r#"쒄"#, r#"쒄"#, r#"쒄"#, r#"쒄"#),
        (r#"쒅"#, r#"쒅"#, r#"쒅"#, r#"쒅"#, r#"쒅"#),
        (r#"쒆"#, r#"쒆"#, r#"쒆"#, r#"쒆"#, r#"쒆"#),
        (r#"쒇"#, r#"쒇"#, r#"쒇"#, r#"쒇"#, r#"쒇"#),
        (r#"쒈"#, r#"쒈"#, r#"쒈"#, r#"쒈"#, r#"쒈"#),
        (r#"쒉"#, r#"쒉"#, r#"쒉"#, r#"쒉"#, r#"쒉"#),
        (r#"쒊"#, r#"쒊"#, r#"쒊"#, r#"쒊"#, r#"쒊"#),
        (r#"쒋"#, r#"쒋"#, r#"쒋"#, r#"쒋"#, r#"쒋"#),
        (r#"쒌"#, r#"쒌"#, r#"쒌"#, r#"쒌"#, r#"쒌"#),
        (r#"쒍"#, r#"쒍"#, r#"쒍"#, r#"쒍"#, r#"쒍"#),
        (r#"쒎"#, r#"쒎"#, r#"쒎"#, r#"쒎"#, r#"쒎"#),
        (r#"쒏"#, r#"쒏"#, r#"쒏"#, r#"쒏"#, r#"쒏"#),
        (r#"쒐"#, r#"쒐"#, r#"쒐"#, r#"쒐"#, r#"쒐"#),
        (r#"쒑"#, r#"쒑"#, r#"쒑"#, r#"쒑"#, r#"쒑"#),
        (r#"쒒"#, r#"쒒"#, r#"쒒"#, r#"쒒"#, r#"쒒"#),
        (r#"쒓"#, r#"쒓"#, r#"쒓"#, r#"쒓"#, r#"쒓"#),
        (r#"쒔"#, r#"쒔"#, r#"쒔"#, r#"쒔"#, r#"쒔"#),
        (r#"쒕"#, r#"쒕"#, r#"쒕"#, r#"쒕"#, r#"쒕"#),
        (r#"쒖"#, r#"쒖"#, r#"쒖"#, r#"쒖"#, r#"쒖"#),
        (r#"쒗"#, r#"쒗"#, r#"쒗"#, r#"쒗"#, r#"쒗"#),
        (r#"쒘"#, r#"쒘"#, r#"쒘"#, r#"쒘"#, r#"쒘"#),
        (r#"쒙"#, r#"쒙"#, r#"쒙"#, r#"쒙"#, r#"쒙"#),
        (r#"쒚"#, r#"쒚"#, r#"쒚"#, r#"쒚"#, r#"쒚"#),
        (r#"쒛"#, r#"쒛"#, r#"쒛"#, r#"쒛"#, r#"쒛"#),
        (r#"쒜"#, r#"쒜"#, r#"쒜"#, r#"쒜"#, r#"쒜"#),
        (r#"쒝"#, r#"쒝"#, r#"쒝"#, r#"쒝"#, r#"쒝"#),
        (r#"쒞"#, r#"쒞"#, r#"쒞"#, r#"쒞"#, r#"쒞"#),
        (r#"쒟"#, r#"쒟"#, r#"쒟"#, r#"쒟"#, r#"쒟"#),
        (r#"쒠"#, r#"쒠"#, r#"쒠"#, r#"쒠"#, r#"쒠"#),
        (r#"쒡"#, r#"쒡"#, r#"쒡"#, r#"쒡"#, r#"쒡"#),
        (r#"쒢"#, r#"쒢"#, r#"쒢"#, r#"쒢"#, r#"쒢"#),
        (r#"쒣"#, r#"쒣"#, r#"쒣"#, r#"쒣"#, r#"쒣"#),
        (r#"쒤"#, r#"쒤"#, r#"쒤"#, r#"쒤"#, r#"쒤"#),
        (r#"쒥"#, r#"쒥"#, r#"쒥"#, r#"쒥"#, r#"쒥"#),
        (r#"쒦"#, r#"쒦"#, r#"쒦"#, r#"쒦"#, r#"쒦"#),
        (r#"쒧"#, r#"쒧"#, r#"쒧"#, r#"쒧"#, r#"쒧"#),
        (r#"쒨"#, r#"쒨"#, r#"쒨"#, r#"쒨"#, r#"쒨"#),
        (r#"쒩"#, r#"쒩"#, r#"쒩"#, r#"쒩"#, r#"쒩"#),
        (r#"쒪"#, r#"쒪"#, r#"쒪"#, r#"쒪"#, r#"쒪"#),
        (r#"쒫"#, r#"쒫"#, r#"쒫"#, r#"쒫"#, r#"쒫"#),
        (r#"쒬"#, r#"쒬"#, r#"쒬"#, r#"쒬"#, r#"쒬"#),
        (r#"쒭"#, r#"쒭"#, r#"쒭"#, r#"쒭"#, r#"쒭"#),
        (r#"쒮"#, r#"쒮"#, r#"쒮"#, r#"쒮"#, r#"쒮"#),
        (r#"쒯"#, r#"쒯"#, r#"쒯"#, r#"쒯"#, r#"쒯"#),
        (r#"쒰"#, r#"쒰"#, r#"쒰"#, r#"쒰"#, r#"쒰"#),
        (r#"쒱"#, r#"쒱"#, r#"쒱"#, r#"쒱"#, r#"쒱"#),
        (r#"쒲"#, r#"쒲"#, r#"쒲"#, r#"쒲"#, r#"쒲"#),
        (r#"쒳"#, r#"쒳"#, r#"쒳"#, r#"쒳"#, r#"쒳"#),
        (r#"쒴"#, r#"쒴"#, r#"쒴"#, r#"쒴"#, r#"쒴"#),
        (r#"쒵"#, r#"쒵"#, r#"쒵"#, r#"쒵"#, r#"쒵"#),
        (r#"쒶"#, r#"쒶"#, r#"쒶"#, r#"쒶"#, r#"쒶"#),
        (r#"쒷"#, r#"쒷"#, r#"쒷"#, r#"쒷"#, r#"쒷"#),
        (r#"쒸"#, r#"쒸"#, r#"쒸"#, r#"쒸"#, r#"쒸"#),
        (r#"쒹"#, r#"쒹"#, r#"쒹"#, r#"쒹"#, r#"쒹"#),
        (r#"쒺"#, r#"쒺"#, r#"쒺"#, r#"쒺"#, r#"쒺"#),
        (r#"쒻"#, r#"쒻"#, r#"쒻"#, r#"쒻"#, r#"쒻"#),
        (r#"쒼"#, r#"쒼"#, r#"쒼"#, r#"쒼"#, r#"쒼"#),
        (r#"쒽"#, r#"쒽"#, r#"쒽"#, r#"쒽"#, r#"쒽"#),
        (r#"쒾"#, r#"쒾"#, r#"쒾"#, r#"쒾"#, r#"쒾"#),
        (r#"쒿"#, r#"쒿"#, r#"쒿"#, r#"쒿"#, r#"쒿"#),
        (r#"쓀"#, r#"쓀"#, r#"쓀"#, r#"쓀"#, r#"쓀"#),
        (r#"쓁"#, r#"쓁"#, r#"쓁"#, r#"쓁"#, r#"쓁"#),
        (r#"쓂"#, r#"쓂"#, r#"쓂"#, r#"쓂"#, r#"쓂"#),
        (r#"쓃"#, r#"쓃"#, r#"쓃"#, r#"쓃"#, r#"쓃"#),
        (r#"쓄"#, r#"쓄"#, r#"쓄"#, r#"쓄"#, r#"쓄"#),
        (r#"쓅"#, r#"쓅"#, r#"쓅"#, r#"쓅"#, r#"쓅"#),
        (r#"쓆"#, r#"쓆"#, r#"쓆"#, r#"쓆"#, r#"쓆"#),
        (r#"쓇"#, r#"쓇"#, r#"쓇"#, r#"쓇"#, r#"쓇"#),
        (r#"쓈"#, r#"쓈"#, r#"쓈"#, r#"쓈"#, r#"쓈"#),
        (r#"쓉"#, r#"쓉"#, r#"쓉"#, r#"쓉"#, r#"쓉"#),
        (r#"쓊"#, r#"쓊"#, r#"쓊"#, r#"쓊"#, r#"쓊"#),
        (r#"쓋"#, r#"쓋"#, r#"쓋"#, r#"쓋"#, r#"쓋"#),
        (r#"쓌"#, r#"쓌"#, r#"쓌"#, r#"쓌"#, r#"쓌"#),
        (r#"쓍"#, r#"쓍"#, r#"쓍"#, r#"쓍"#, r#"쓍"#),
        (r#"쓎"#, r#"쓎"#, r#"쓎"#, r#"쓎"#, r#"쓎"#),
        (r#"쓏"#, r#"쓏"#, r#"쓏"#, r#"쓏"#, r#"쓏"#),
        (r#"쓐"#, r#"쓐"#, r#"쓐"#, r#"쓐"#, r#"쓐"#),
        (r#"쓑"#, r#"쓑"#, r#"쓑"#, r#"쓑"#, r#"쓑"#),
        (r#"쓒"#, r#"쓒"#, r#"쓒"#, r#"쓒"#, r#"쓒"#),
        (r#"쓓"#, r#"쓓"#, r#"쓓"#, r#"쓓"#, r#"쓓"#),
        (r#"쓔"#, r#"쓔"#, r#"쓔"#, r#"쓔"#, r#"쓔"#),
        (r#"쓕"#, r#"쓕"#, r#"쓕"#, r#"쓕"#, r#"쓕"#),
        (r#"쓖"#, r#"쓖"#, r#"쓖"#, r#"쓖"#, r#"쓖"#),
        (r#"쓗"#, r#"쓗"#, r#"쓗"#, r#"쓗"#, r#"쓗"#),
        (r#"쓘"#, r#"쓘"#, r#"쓘"#, r#"쓘"#, r#"쓘"#),
        (r#"쓙"#, r#"쓙"#, r#"쓙"#, r#"쓙"#, r#"쓙"#),
        (r#"쓚"#, r#"쓚"#, r#"쓚"#, r#"쓚"#, r#"쓚"#),
        (r#"쓛"#, r#"쓛"#, r#"쓛"#, r#"쓛"#, r#"쓛"#),
        (r#"쓜"#, r#"쓜"#, r#"쓜"#, r#"쓜"#, r#"쓜"#),
        (r#"쓝"#, r#"쓝"#, r#"쓝"#, r#"쓝"#, r#"쓝"#),
        (r#"쓞"#, r#"쓞"#, r#"쓞"#, r#"쓞"#, r#"쓞"#),
        (r#"쓟"#, r#"쓟"#, r#"쓟"#, r#"쓟"#, r#"쓟"#),
        (r#"쓠"#, r#"쓠"#, r#"쓠"#, r#"쓠"#, r#"쓠"#),
        (r#"쓡"#, r#"쓡"#, r#"쓡"#, r#"쓡"#, r#"쓡"#),
        (r#"쓢"#, r#"쓢"#, r#"쓢"#, r#"쓢"#, r#"쓢"#),
        (r#"쓣"#, r#"쓣"#, r#"쓣"#, r#"쓣"#, r#"쓣"#),
        (r#"쓤"#, r#"쓤"#, r#"쓤"#, r#"쓤"#, r#"쓤"#),
        (r#"쓥"#, r#"쓥"#, r#"쓥"#, r#"쓥"#, r#"쓥"#),
        (r#"쓦"#, r#"쓦"#, r#"쓦"#, r#"쓦"#, r#"쓦"#),
        (r#"쓧"#, r#"쓧"#, r#"쓧"#, r#"쓧"#, r#"쓧"#),
        (r#"쓨"#, r#"쓨"#, r#"쓨"#, r#"쓨"#, r#"쓨"#),
        (r#"쓩"#, r#"쓩"#, r#"쓩"#, r#"쓩"#, r#"쓩"#),
        (r#"쓪"#, r#"쓪"#, r#"쓪"#, r#"쓪"#, r#"쓪"#),
        (r#"쓫"#, r#"쓫"#, r#"쓫"#, r#"쓫"#, r#"쓫"#),
        (r#"쓬"#, r#"쓬"#, r#"쓬"#, r#"쓬"#, r#"쓬"#),
        (r#"쓭"#, r#"쓭"#, r#"쓭"#, r#"쓭"#, r#"쓭"#),
        (r#"쓮"#, r#"쓮"#, r#"쓮"#, r#"쓮"#, r#"쓮"#),
        (r#"쓯"#, r#"쓯"#, r#"쓯"#, r#"쓯"#, r#"쓯"#),
        (r#"쓰"#, r#"쓰"#, r#"쓰"#, r#"쓰"#, r#"쓰"#),
        (r#"쓱"#, r#"쓱"#, r#"쓱"#, r#"쓱"#, r#"쓱"#),
        (r#"쓲"#, r#"쓲"#, r#"쓲"#, r#"쓲"#, r#"쓲"#),
        (r#"쓳"#, r#"쓳"#, r#"쓳"#, r#"쓳"#, r#"쓳"#),
        (r#"쓴"#, r#"쓴"#, r#"쓴"#, r#"쓴"#, r#"쓴"#),
        (r#"쓵"#, r#"쓵"#, r#"쓵"#, r#"쓵"#, r#"쓵"#),
        (r#"쓶"#, r#"쓶"#, r#"쓶"#, r#"쓶"#, r#"쓶"#),
        (r#"쓷"#, r#"쓷"#, r#"쓷"#, r#"쓷"#, r#"쓷"#),
        (r#"쓸"#, r#"쓸"#, r#"쓸"#, r#"쓸"#, r#"쓸"#),
        (r#"쓹"#, r#"쓹"#, r#"쓹"#, r#"쓹"#, r#"쓹"#),
        (r#"쓺"#, r#"쓺"#, r#"쓺"#, r#"쓺"#, r#"쓺"#),
        (r#"쓻"#, r#"쓻"#, r#"쓻"#, r#"쓻"#, r#"쓻"#),
        (r#"쓼"#, r#"쓼"#, r#"쓼"#, r#"쓼"#, r#"쓼"#),
        (r#"쓽"#, r#"쓽"#, r#"쓽"#, r#"쓽"#, r#"쓽"#),
        (r#"쓾"#, r#"쓾"#, r#"쓾"#, r#"쓾"#, r#"쓾"#),
        (r#"쓿"#, r#"쓿"#, r#"쓿"#, r#"쓿"#, r#"쓿"#),
        (r#"씀"#, r#"씀"#, r#"씀"#, r#"씀"#, r#"씀"#),
        (r#"씁"#, r#"씁"#, r#"씁"#, r#"씁"#, r#"씁"#),
        (r#"씂"#, r#"씂"#, r#"씂"#, r#"씂"#, r#"씂"#),
        (r#"씃"#, r#"씃"#, r#"씃"#, r#"씃"#, r#"씃"#),
        (r#"씄"#, r#"씄"#, r#"씄"#, r#"씄"#, r#"씄"#),
        (r#"씅"#, r#"씅"#, r#"씅"#, r#"씅"#, r#"씅"#),
        (r#"씆"#, r#"씆"#, r#"씆"#, r#"씆"#, r#"씆"#),
        (r#"씇"#, r#"씇"#, r#"씇"#, r#"씇"#, r#"씇"#),
        (r#"씈"#, r#"씈"#, r#"씈"#, r#"씈"#, r#"씈"#),
        (r#"씉"#, r#"씉"#, r#"씉"#, r#"씉"#, r#"씉"#),
        (r#"씊"#, r#"씊"#, r#"씊"#, r#"씊"#, r#"씊"#),
        (r#"씋"#, r#"씋"#, r#"씋"#, r#"씋"#, r#"씋"#),
        (r#"씌"#, r#"씌"#, r#"씌"#, r#"씌"#, r#"씌"#),
        (r#"씍"#, r#"씍"#, r#"씍"#, r#"씍"#, r#"씍"#),
        (r#"씎"#, r#"씎"#, r#"씎"#, r#"씎"#, r#"씎"#),
        (r#"씏"#, r#"씏"#, r#"씏"#, r#"씏"#, r#"씏"#),
        (r#"씐"#, r#"씐"#, r#"씐"#, r#"씐"#, r#"씐"#),
        (r#"씑"#, r#"씑"#, r#"씑"#, r#"씑"#, r#"씑"#),
        (r#"씒"#, r#"씒"#, r#"씒"#, r#"씒"#, r#"씒"#),
        (r#"씓"#, r#"씓"#, r#"씓"#, r#"씓"#, r#"씓"#),
        (r#"씔"#, r#"씔"#, r#"씔"#, r#"씔"#, r#"씔"#),
        (r#"씕"#, r#"씕"#, r#"씕"#, r#"씕"#, r#"씕"#),
        (r#"씖"#, r#"씖"#, r#"씖"#, r#"씖"#, r#"씖"#),
        (r#"씗"#, r#"씗"#, r#"씗"#, r#"씗"#, r#"씗"#),
        (r#"씘"#, r#"씘"#, r#"씘"#, r#"씘"#, r#"씘"#),
        (r#"씙"#, r#"씙"#, r#"씙"#, r#"씙"#, r#"씙"#),
        (r#"씚"#, r#"씚"#, r#"씚"#, r#"씚"#, r#"씚"#),
        (r#"씛"#, r#"씛"#, r#"씛"#, r#"씛"#, r#"씛"#),
        (r#"씜"#, r#"씜"#, r#"씜"#, r#"씜"#, r#"씜"#),
        (r#"씝"#, r#"씝"#, r#"씝"#, r#"씝"#, r#"씝"#),
        (r#"씞"#, r#"씞"#, r#"씞"#, r#"씞"#, r#"씞"#),
        (r#"씟"#, r#"씟"#, r#"씟"#, r#"씟"#, r#"씟"#),
        (r#"씠"#, r#"씠"#, r#"씠"#, r#"씠"#, r#"씠"#),
        (r#"씡"#, r#"씡"#, r#"씡"#, r#"씡"#, r#"씡"#),
        (r#"씢"#, r#"씢"#, r#"씢"#, r#"씢"#, r#"씢"#),
        (r#"씣"#, r#"씣"#, r#"씣"#, r#"씣"#, r#"씣"#),
        (r#"씤"#, r#"씤"#, r#"씤"#, r#"씤"#, r#"씤"#),
        (r#"씥"#, r#"씥"#, r#"씥"#, r#"씥"#, r#"씥"#),
        (r#"씦"#, r#"씦"#, r#"씦"#, r#"씦"#, r#"씦"#),
        (r#"씧"#, r#"씧"#, r#"씧"#, r#"씧"#, r#"씧"#),
        (r#"씨"#, r#"씨"#, r#"씨"#, r#"씨"#, r#"씨"#),
        (r#"씩"#, r#"씩"#, r#"씩"#, r#"씩"#, r#"씩"#),
        (r#"씪"#, r#"씪"#, r#"씪"#, r#"씪"#, r#"씪"#),
        (r#"씫"#, r#"씫"#, r#"씫"#, r#"씫"#, r#"씫"#),
        (r#"씬"#, r#"씬"#, r#"씬"#, r#"씬"#, r#"씬"#),
        (r#"씭"#, r#"씭"#, r#"씭"#, r#"씭"#, r#"씭"#),
        (r#"씮"#, r#"씮"#, r#"씮"#, r#"씮"#, r#"씮"#),
        (r#"씯"#, r#"씯"#, r#"씯"#, r#"씯"#, r#"씯"#),
        (r#"씰"#, r#"씰"#, r#"씰"#, r#"씰"#, r#"씰"#),
        (r#"씱"#, r#"씱"#, r#"씱"#, r#"씱"#, r#"씱"#),
        (r#"씲"#, r#"씲"#, r#"씲"#, r#"씲"#, r#"씲"#),
        (r#"씳"#, r#"씳"#, r#"씳"#, r#"씳"#, r#"씳"#),
        (r#"씴"#, r#"씴"#, r#"씴"#, r#"씴"#, r#"씴"#),
        (r#"씵"#, r#"씵"#, r#"씵"#, r#"씵"#, r#"씵"#),
        (r#"씶"#, r#"씶"#, r#"씶"#, r#"씶"#, r#"씶"#),
        (r#"씷"#, r#"씷"#, r#"씷"#, r#"씷"#, r#"씷"#),
        (r#"씸"#, r#"씸"#, r#"씸"#, r#"씸"#, r#"씸"#),
        (r#"씹"#, r#"씹"#, r#"씹"#, r#"씹"#, r#"씹"#),
        (r#"씺"#, r#"씺"#, r#"씺"#, r#"씺"#, r#"씺"#),
        (r#"씻"#, r#"씻"#, r#"씻"#, r#"씻"#, r#"씻"#),
        (r#"씼"#, r#"씼"#, r#"씼"#, r#"씼"#, r#"씼"#),
        (r#"씽"#, r#"씽"#, r#"씽"#, r#"씽"#, r#"씽"#),
        (r#"씾"#, r#"씾"#, r#"씾"#, r#"씾"#, r#"씾"#),
        (r#"씿"#, r#"씿"#, r#"씿"#, r#"씿"#, r#"씿"#),
        (r#"앀"#, r#"앀"#, r#"앀"#, r#"앀"#, r#"앀"#),
        (r#"앁"#, r#"앁"#, r#"앁"#, r#"앁"#, r#"앁"#),
        (r#"앂"#, r#"앂"#, r#"앂"#, r#"앂"#, r#"앂"#),
        (r#"앃"#, r#"앃"#, r#"앃"#, r#"앃"#, r#"앃"#),
        (r#"아"#, r#"아"#, r#"아"#, r#"아"#, r#"아"#),
        (r#"악"#, r#"악"#, r#"악"#, r#"악"#, r#"악"#),
        (r#"앆"#, r#"앆"#, r#"앆"#, r#"앆"#, r#"앆"#),
        (r#"앇"#, r#"앇"#, r#"앇"#, r#"앇"#, r#"앇"#),
        (r#"안"#, r#"안"#, r#"안"#, r#"안"#, r#"안"#),
        (r#"앉"#, r#"앉"#, r#"앉"#, r#"앉"#, r#"앉"#),
        (r#"않"#, r#"않"#, r#"않"#, r#"않"#, r#"않"#),
        (r#"앋"#, r#"앋"#, r#"앋"#, r#"앋"#, r#"앋"#),
        (r#"알"#, r#"알"#, r#"알"#, r#"알"#, r#"알"#),
        (r#"앍"#, r#"앍"#, r#"앍"#, r#"앍"#, r#"앍"#),
        (r#"앎"#, r#"앎"#, r#"앎"#, r#"앎"#, r#"앎"#),
        (r#"앏"#, r#"앏"#, r#"앏"#, r#"앏"#, r#"앏"#),
        (r#"앐"#, r#"앐"#, r#"앐"#, r#"앐"#, r#"앐"#),
        (r#"앑"#, r#"앑"#, r#"앑"#, r#"앑"#, r#"앑"#),
        (r#"앒"#, r#"앒"#, r#"앒"#, r#"앒"#, r#"앒"#),
        (r#"앓"#, r#"앓"#, r#"앓"#, r#"앓"#, r#"앓"#),
        (r#"암"#, r#"암"#, r#"암"#, r#"암"#, r#"암"#),
        (r#"압"#, r#"압"#, r#"압"#, r#"압"#, r#"압"#),
        (r#"앖"#, r#"앖"#, r#"앖"#, r#"앖"#, r#"앖"#),
        (r#"앗"#, r#"앗"#, r#"앗"#, r#"앗"#, r#"앗"#),
        (r#"았"#, r#"았"#, r#"았"#, r#"았"#, r#"았"#),
        (r#"앙"#, r#"앙"#, r#"앙"#, r#"앙"#, r#"앙"#),
        (r#"앚"#, r#"앚"#, r#"앚"#, r#"앚"#, r#"앚"#),
        (r#"앛"#, r#"앛"#, r#"앛"#, r#"앛"#, r#"앛"#),
        (r#"앜"#, r#"앜"#, r#"앜"#, r#"앜"#, r#"앜"#),
        (r#"앝"#, r#"앝"#, r#"앝"#, r#"앝"#, r#"앝"#),
        (r#"앞"#, r#"앞"#, r#"앞"#, r#"앞"#, r#"앞"#),
        (r#"앟"#, r#"앟"#, r#"앟"#, r#"앟"#, r#"앟"#),
        (r#"애"#, r#"애"#, r#"애"#, r#"애"#, r#"애"#),
        (r#"액"#, r#"액"#, r#"액"#, r#"액"#, r#"액"#),
        (r#"앢"#, r#"앢"#, r#"앢"#, r#"앢"#, r#"앢"#),
        (r#"앣"#, r#"앣"#, r#"앣"#, r#"앣"#, r#"앣"#),
        (r#"앤"#, r#"앤"#, r#"앤"#, r#"앤"#, r#"앤"#),
        (r#"앥"#, r#"앥"#, r#"앥"#, r#"앥"#, r#"앥"#),
        (r#"앦"#, r#"앦"#, r#"앦"#, r#"앦"#, r#"앦"#),
        (r#"앧"#, r#"앧"#, r#"앧"#, r#"앧"#, r#"앧"#),
        (r#"앨"#, r#"앨"#, r#"앨"#, r#"앨"#, r#"앨"#),
        (r#"앩"#, r#"앩"#, r#"앩"#, r#"앩"#, r#"앩"#),
        (r#"앪"#, r#"앪"#, r#"앪"#, r#"앪"#, r#"앪"#),
        (r#"앫"#, r#"앫"#, r#"앫"#, r#"앫"#, r#"앫"#),
        (r#"앬"#, r#"앬"#, r#"앬"#, r#"앬"#, r#"앬"#),
        (r#"앭"#, r#"앭"#, r#"앭"#, r#"앭"#, r#"앭"#),
        (r#"앮"#, r#"앮"#, r#"앮"#, r#"앮"#, r#"앮"#),
        (r#"앯"#, r#"앯"#, r#"앯"#, r#"앯"#, r#"앯"#),
        (r#"앰"#, r#"앰"#, r#"앰"#, r#"앰"#, r#"앰"#),
        (r#"앱"#, r#"앱"#, r#"앱"#, r#"앱"#, r#"앱"#),
        (r#"앲"#, r#"앲"#, r#"앲"#, r#"앲"#, r#"앲"#),
        (r#"앳"#, r#"앳"#, r#"앳"#, r#"앳"#, r#"앳"#),
        (r#"앴"#, r#"앴"#, r#"앴"#, r#"앴"#, r#"앴"#),
        (r#"앵"#, r#"앵"#, r#"앵"#, r#"앵"#, r#"앵"#),
        (r#"앶"#, r#"앶"#, r#"앶"#, r#"앶"#, r#"앶"#),
        (r#"앷"#, r#"앷"#, r#"앷"#, r#"앷"#, r#"앷"#),
        (r#"앸"#, r#"앸"#, r#"앸"#, r#"앸"#, r#"앸"#),
        (r#"앹"#, r#"앹"#, r#"앹"#, r#"앹"#, r#"앹"#),
        (r#"앺"#, r#"앺"#, r#"앺"#, r#"앺"#, r#"앺"#),
        (r#"앻"#, r#"앻"#, r#"앻"#, r#"앻"#, r#"앻"#),
        (r#"야"#, r#"야"#, r#"야"#, r#"야"#, r#"야"#),
        (r#"약"#, r#"약"#, r#"약"#, r#"약"#, r#"약"#),
        (r#"앾"#, r#"앾"#, r#"앾"#, r#"앾"#, r#"앾"#),
        (r#"앿"#, r#"앿"#, r#"앿"#, r#"앿"#, r#"앿"#),
        (r#"얀"#, r#"얀"#, r#"얀"#, r#"얀"#, r#"얀"#),
        (r#"얁"#, r#"얁"#, r#"얁"#, r#"얁"#, r#"얁"#),
        (r#"얂"#, r#"얂"#, r#"얂"#, r#"얂"#, r#"얂"#),
        (r#"얃"#, r#"얃"#, r#"얃"#, r#"얃"#, r#"얃"#),
        (r#"얄"#, r#"얄"#, r#"얄"#, r#"얄"#, r#"얄"#),
        (r#"얅"#, r#"얅"#, r#"얅"#, r#"얅"#, r#"얅"#),
        (r#"얆"#, r#"얆"#, r#"얆"#, r#"얆"#, r#"얆"#),
        (r#"얇"#, r#"얇"#, r#"얇"#, r#"얇"#, r#"얇"#),
        (r#"얈"#, r#"얈"#, r#"얈"#, r#"얈"#, r#"얈"#),
        (r#"얉"#, r#"얉"#, r#"얉"#, r#"얉"#, r#"얉"#),
        (r#"얊"#, r#"얊"#, r#"얊"#, r#"얊"#, r#"얊"#),
        (r#"얋"#, r#"얋"#, r#"얋"#, r#"얋"#, r#"얋"#),
        (r#"얌"#, r#"얌"#, r#"얌"#, r#"얌"#, r#"얌"#),
        (r#"얍"#, r#"얍"#, r#"얍"#, r#"얍"#, r#"얍"#),
        (r#"얎"#, r#"얎"#, r#"얎"#, r#"얎"#, r#"얎"#),
        (r#"얏"#, r#"얏"#, r#"얏"#, r#"얏"#, r#"얏"#),
        (r#"얐"#, r#"얐"#, r#"얐"#, r#"얐"#, r#"얐"#),
        (r#"양"#, r#"양"#, r#"양"#, r#"양"#, r#"양"#),
        (r#"얒"#, r#"얒"#, r#"얒"#, r#"얒"#, r#"얒"#),
        (r#"얓"#, r#"얓"#, r#"얓"#, r#"얓"#, r#"얓"#),
        (r#"얔"#, r#"얔"#, r#"얔"#, r#"얔"#, r#"얔"#),
        (r#"얕"#, r#"얕"#, r#"얕"#, r#"얕"#, r#"얕"#),
        (r#"얖"#, r#"얖"#, r#"얖"#, r#"얖"#, r#"얖"#),
        (r#"얗"#, r#"얗"#, r#"얗"#, r#"얗"#, r#"얗"#),
        (r#"얘"#, r#"얘"#, r#"얘"#, r#"얘"#, r#"얘"#),
        (r#"얙"#, r#"얙"#, r#"얙"#, r#"얙"#, r#"얙"#),
        (r#"얚"#, r#"얚"#, r#"얚"#, r#"얚"#, r#"얚"#),
        (r#"얛"#, r#"얛"#, r#"얛"#, r#"얛"#, r#"얛"#),
        (r#"얜"#, r#"얜"#, r#"얜"#, r#"얜"#, r#"얜"#),
        (r#"얝"#, r#"얝"#, r#"얝"#, r#"얝"#, r#"얝"#),
        (r#"얞"#, r#"얞"#, r#"얞"#, r#"얞"#, r#"얞"#),
        (r#"얟"#, r#"얟"#, r#"얟"#, r#"얟"#, r#"얟"#),
        (r#"얠"#, r#"얠"#, r#"얠"#, r#"얠"#, r#"얠"#),
        (r#"얡"#, r#"얡"#, r#"얡"#, r#"얡"#, r#"얡"#),
        (r#"얢"#, r#"얢"#, r#"얢"#, r#"얢"#, r#"얢"#),
        (r#"얣"#, r#"얣"#, r#"얣"#, r#"얣"#, r#"얣"#),
        (r#"얤"#, r#"얤"#, r#"얤"#, r#"얤"#, r#"얤"#),
        (r#"얥"#, r#"얥"#, r#"얥"#, r#"얥"#, r#"얥"#),
        (r#"얦"#, r#"얦"#, r#"얦"#, r#"얦"#, r#"얦"#),
        (r#"얧"#, r#"얧"#, r#"얧"#, r#"얧"#, r#"얧"#),
        (r#"얨"#, r#"얨"#, r#"얨"#, r#"얨"#, r#"얨"#),
        (r#"얩"#, r#"얩"#, r#"얩"#, r#"얩"#, r#"얩"#),
        (r#"얪"#, r#"얪"#, r#"얪"#, r#"얪"#, r#"얪"#),
        (r#"얫"#, r#"얫"#, r#"얫"#, r#"얫"#, r#"얫"#),
        (r#"얬"#, r#"얬"#, r#"얬"#, r#"얬"#, r#"얬"#),
        (r#"얭"#, r#"얭"#, r#"얭"#, r#"얭"#, r#"얭"#),
        (r#"얮"#, r#"얮"#, r#"얮"#, r#"얮"#, r#"얮"#),
        (r#"얯"#, r#"얯"#, r#"얯"#, r#"얯"#, r#"얯"#),
        (r#"얰"#, r#"얰"#, r#"얰"#, r#"얰"#, r#"얰"#),
        (r#"얱"#, r#"얱"#, r#"얱"#, r#"얱"#, r#"얱"#),
        (r#"얲"#, r#"얲"#, r#"얲"#, r#"얲"#, r#"얲"#),
        (r#"얳"#, r#"얳"#, r#"얳"#, r#"얳"#, r#"얳"#),
        (r#"어"#, r#"어"#, r#"어"#, r#"어"#, r#"어"#),
        (r#"억"#, r#"억"#, r#"억"#, r#"억"#, r#"억"#),
        (r#"얶"#, r#"얶"#, r#"얶"#, r#"얶"#, r#"얶"#),
        (r#"얷"#, r#"얷"#, r#"얷"#, r#"얷"#, r#"얷"#),
        (r#"언"#, r#"언"#, r#"언"#, r#"언"#, r#"언"#),
        (r#"얹"#, r#"얹"#, r#"얹"#, r#"얹"#, r#"얹"#),
        (r#"얺"#, r#"얺"#, r#"얺"#, r#"얺"#, r#"얺"#),
        (r#"얻"#, r#"얻"#, r#"얻"#, r#"얻"#, r#"얻"#),
        (r#"얼"#, r#"얼"#, r#"얼"#, r#"얼"#, r#"얼"#),
        (r#"얽"#, r#"얽"#, r#"얽"#, r#"얽"#, r#"얽"#),
        (r#"얾"#, r#"얾"#, r#"얾"#, r#"얾"#, r#"얾"#),
        (r#"얿"#, r#"얿"#, r#"얿"#, r#"얿"#, r#"얿"#),
        (r#"엀"#, r#"엀"#, r#"엀"#, r#"엀"#, r#"엀"#),
        (r#"엁"#, r#"엁"#, r#"엁"#, r#"엁"#, r#"엁"#),
        (r#"엂"#, r#"엂"#, r#"엂"#, r#"엂"#, r#"엂"#),
        (r#"엃"#, r#"엃"#, r#"엃"#, r#"엃"#, r#"엃"#),
        (r#"엄"#, r#"엄"#, r#"엄"#, r#"엄"#, r#"엄"#),
        (r#"업"#, r#"업"#, r#"업"#, r#"업"#, r#"업"#),
        (r#"없"#, r#"없"#, r#"없"#, r#"없"#, r#"없"#),
        (r#"엇"#, r#"엇"#, r#"엇"#, r#"엇"#, r#"엇"#),
        (r#"었"#, r#"었"#, r#"었"#, r#"었"#, r#"었"#),
        (r#"엉"#, r#"엉"#, r#"엉"#, r#"엉"#, r#"엉"#),
        (r#"엊"#, r#"엊"#, r#"엊"#, r#"엊"#, r#"엊"#),
        (r#"엋"#, r#"엋"#, r#"엋"#, r#"엋"#, r#"엋"#),
        (r#"엌"#, r#"엌"#, r#"엌"#, r#"엌"#, r#"엌"#),
        (r#"엍"#, r#"엍"#, r#"엍"#, r#"엍"#, r#"엍"#),
        (r#"엎"#, r#"엎"#, r#"엎"#, r#"엎"#, r#"엎"#),
        (r#"엏"#, r#"엏"#, r#"엏"#, r#"엏"#, r#"엏"#),
        (r#"에"#, r#"에"#, r#"에"#, r#"에"#, r#"에"#),
        (r#"엑"#, r#"엑"#, r#"엑"#, r#"엑"#, r#"엑"#),
        (r#"엒"#, r#"엒"#, r#"엒"#, r#"엒"#, r#"엒"#),
        (r#"엓"#, r#"엓"#, r#"엓"#, r#"엓"#, r#"엓"#),
        (r#"엔"#, r#"엔"#, r#"엔"#, r#"엔"#, r#"엔"#),
        (r#"엕"#, r#"엕"#, r#"엕"#, r#"엕"#, r#"엕"#),
        (r#"엖"#, r#"엖"#, r#"엖"#, r#"엖"#, r#"엖"#),
        (r#"엗"#, r#"엗"#, r#"엗"#, r#"엗"#, r#"엗"#),
        (r#"엘"#, r#"엘"#, r#"엘"#, r#"엘"#, r#"엘"#),
        (r#"엙"#, r#"엙"#, r#"엙"#, r#"엙"#, r#"엙"#),
        (r#"엚"#, r#"엚"#, r#"엚"#, r#"엚"#, r#"엚"#),
        (r#"엛"#, r#"엛"#, r#"엛"#, r#"엛"#, r#"엛"#),
        (r#"엜"#, r#"엜"#, r#"엜"#, r#"엜"#, r#"엜"#),
        (r#"엝"#, r#"엝"#, r#"엝"#, r#"엝"#, r#"엝"#),
        (r#"엞"#, r#"엞"#, r#"엞"#, r#"엞"#, r#"엞"#),
        (r#"엟"#, r#"엟"#, r#"엟"#, r#"엟"#, r#"엟"#),
        (r#"엠"#, r#"엠"#, r#"엠"#, r#"엠"#, r#"엠"#),
        (r#"엡"#, r#"엡"#, r#"엡"#, r#"엡"#, r#"엡"#),
        (r#"엢"#, r#"엢"#, r#"엢"#, r#"엢"#, r#"엢"#),
        (r#"엣"#, r#"엣"#, r#"엣"#, r#"엣"#, r#"엣"#),
        (r#"엤"#, r#"엤"#, r#"엤"#, r#"엤"#, r#"엤"#),
        (r#"엥"#, r#"엥"#, r#"엥"#, r#"엥"#, r#"엥"#),
        (r#"엦"#, r#"엦"#, r#"엦"#, r#"엦"#, r#"엦"#),
        (r#"엧"#, r#"엧"#, r#"엧"#, r#"엧"#, r#"엧"#),
        (r#"엨"#, r#"엨"#, r#"엨"#, r#"엨"#, r#"엨"#),
        (r#"엩"#, r#"엩"#, r#"엩"#, r#"엩"#, r#"엩"#),
        (r#"엪"#, r#"엪"#, r#"엪"#, r#"엪"#, r#"엪"#),
        (r#"엫"#, r#"엫"#, r#"엫"#, r#"엫"#, r#"엫"#),
        (r#"여"#, r#"여"#, r#"여"#, r#"여"#, r#"여"#),
        (r#"역"#, r#"역"#, r#"역"#, r#"역"#, r#"역"#),
        (r#"엮"#, r#"엮"#, r#"엮"#, r#"엮"#, r#"엮"#),
        (r#"엯"#, r#"엯"#, r#"엯"#, r#"엯"#, r#"엯"#),
        (r#"연"#, r#"연"#, r#"연"#, r#"연"#, r#"연"#),
        (r#"엱"#, r#"엱"#, r#"엱"#, r#"엱"#, r#"엱"#),
        (r#"엲"#, r#"엲"#, r#"엲"#, r#"엲"#, r#"엲"#),
        (r#"엳"#, r#"엳"#, r#"엳"#, r#"엳"#, r#"엳"#),
        (r#"열"#, r#"열"#, r#"열"#, r#"열"#, r#"열"#),
        (r#"엵"#, r#"엵"#, r#"엵"#, r#"엵"#, r#"엵"#),
        (r#"엶"#, r#"엶"#, r#"엶"#, r#"엶"#, r#"엶"#),
        (r#"엷"#, r#"엷"#, r#"엷"#, r#"엷"#, r#"엷"#),
        (r#"엸"#, r#"엸"#, r#"엸"#, r#"엸"#, r#"엸"#),
        (r#"엹"#, r#"엹"#, r#"엹"#, r#"엹"#, r#"엹"#),
        (r#"엺"#, r#"엺"#, r#"엺"#, r#"엺"#, r#"엺"#),
        (r#"엻"#, r#"엻"#, r#"엻"#, r#"엻"#, r#"엻"#),
        (r#"염"#, r#"염"#, r#"염"#, r#"염"#, r#"염"#),
        (r#"엽"#, r#"엽"#, r#"엽"#, r#"엽"#, r#"엽"#),
        (r#"엾"#, r#"엾"#, r#"엾"#, r#"엾"#, r#"엾"#),
        (r#"엿"#, r#"엿"#, r#"엿"#, r#"엿"#, r#"엿"#),
        (r#"였"#, r#"였"#, r#"였"#, r#"였"#, r#"였"#),
        (r#"영"#, r#"영"#, r#"영"#, r#"영"#, r#"영"#),
        (r#"옂"#, r#"옂"#, r#"옂"#, r#"옂"#, r#"옂"#),
        (r#"옃"#, r#"옃"#, r#"옃"#, r#"옃"#, r#"옃"#),
        (r#"옄"#, r#"옄"#, r#"옄"#, r#"옄"#, r#"옄"#),
        (r#"옅"#, r#"옅"#, r#"옅"#, r#"옅"#, r#"옅"#),
        (r#"옆"#, r#"옆"#, r#"옆"#, r#"옆"#, r#"옆"#),
        (r#"옇"#, r#"옇"#, r#"옇"#, r#"옇"#, r#"옇"#),
        (r#"예"#, r#"예"#, r#"예"#, r#"예"#, r#"예"#),
        (r#"옉"#, r#"옉"#, r#"옉"#, r#"옉"#, r#"옉"#),
        (r#"옊"#, r#"옊"#, r#"옊"#, r#"옊"#, r#"옊"#),
        (r#"옋"#, r#"옋"#, r#"옋"#, r#"옋"#, r#"옋"#),
        (r#"옌"#, r#"옌"#, r#"옌"#, r#"옌"#, r#"옌"#),
        (r#"옍"#, r#"옍"#, r#"옍"#, r#"옍"#, r#"옍"#),
        (r#"옎"#, r#"옎"#, r#"옎"#, r#"옎"#, r#"옎"#),
        (r#"옏"#, r#"옏"#, r#"옏"#, r#"옏"#, r#"옏"#),
        (r#"옐"#, r#"옐"#, r#"옐"#, r#"옐"#, r#"옐"#),
        (r#"옑"#, r#"옑"#, r#"옑"#, r#"옑"#, r#"옑"#),
        (r#"옒"#, r#"옒"#, r#"옒"#, r#"옒"#, r#"옒"#),
        (r#"옓"#, r#"옓"#, r#"옓"#, r#"옓"#, r#"옓"#),
        (r#"옔"#, r#"옔"#, r#"옔"#, r#"옔"#, r#"옔"#),
        (r#"옕"#, r#"옕"#, r#"옕"#, r#"옕"#, r#"옕"#),
        (r#"옖"#, r#"옖"#, r#"옖"#, r#"옖"#, r#"옖"#),
        (r#"옗"#, r#"옗"#, r#"옗"#, r#"옗"#, r#"옗"#),
        (r#"옘"#, r#"옘"#, r#"옘"#, r#"옘"#, r#"옘"#),
        (r#"옙"#, r#"옙"#, r#"옙"#, r#"옙"#, r#"옙"#),
        (r#"옚"#, r#"옚"#, r#"옚"#, r#"옚"#, r#"옚"#),
        (r#"옛"#, r#"옛"#, r#"옛"#, r#"옛"#, r#"옛"#),
        (r#"옜"#, r#"옜"#, r#"옜"#, r#"옜"#, r#"옜"#),
        (r#"옝"#, r#"옝"#, r#"옝"#, r#"옝"#, r#"옝"#),
        (r#"옞"#, r#"옞"#, r#"옞"#, r#"옞"#, r#"옞"#),
        (r#"옟"#, r#"옟"#, r#"옟"#, r#"옟"#, r#"옟"#),
        (r#"옠"#, r#"옠"#, r#"옠"#, r#"옠"#, r#"옠"#),
        (r#"옡"#, r#"옡"#, r#"옡"#, r#"옡"#, r#"옡"#),
        (r#"옢"#, r#"옢"#, r#"옢"#, r#"옢"#, r#"옢"#),
        (r#"옣"#, r#"옣"#, r#"옣"#, r#"옣"#, r#"옣"#),
        (r#"오"#, r#"오"#, r#"오"#, r#"오"#, r#"오"#),
        (r#"옥"#, r#"옥"#, r#"옥"#, r#"옥"#, r#"옥"#),
        (r#"옦"#, r#"옦"#, r#"옦"#, r#"옦"#, r#"옦"#),
        (r#"옧"#, r#"옧"#, r#"옧"#, r#"옧"#, r#"옧"#),
        (r#"온"#, r#"온"#, r#"온"#, r#"온"#, r#"온"#),
        (r#"옩"#, r#"옩"#, r#"옩"#, r#"옩"#, r#"옩"#),
        (r#"옪"#, r#"옪"#, r#"옪"#, r#"옪"#, r#"옪"#),
        (r#"옫"#, r#"옫"#, r#"옫"#, r#"옫"#, r#"옫"#),
        (r#"올"#, r#"올"#, r#"올"#, r#"올"#, r#"올"#),
        (r#"옭"#, r#"옭"#, r#"옭"#, r#"옭"#, r#"옭"#),
        (r#"옮"#, r#"옮"#, r#"옮"#, r#"옮"#, r#"옮"#),
        (r#"옯"#, r#"옯"#, r#"옯"#, r#"옯"#, r#"옯"#),
        (r#"옰"#, r#"옰"#, r#"옰"#, r#"옰"#, r#"옰"#),
        (r#"옱"#, r#"옱"#, r#"옱"#, r#"옱"#, r#"옱"#),
        (r#"옲"#, r#"옲"#, r#"옲"#, r#"옲"#, r#"옲"#),
        (r#"옳"#, r#"옳"#, r#"옳"#, r#"옳"#, r#"옳"#),
        (r#"옴"#, r#"옴"#, r#"옴"#, r#"옴"#, r#"옴"#),
        (r#"옵"#, r#"옵"#, r#"옵"#, r#"옵"#, r#"옵"#),
        (r#"옶"#, r#"옶"#, r#"옶"#, r#"옶"#, r#"옶"#),
        (r#"옷"#, r#"옷"#, r#"옷"#, r#"옷"#, r#"옷"#),
        (r#"옸"#, r#"옸"#, r#"옸"#, r#"옸"#, r#"옸"#),
        (r#"옹"#, r#"옹"#, r#"옹"#, r#"옹"#, r#"옹"#),
        (r#"옺"#, r#"옺"#, r#"옺"#, r#"옺"#, r#"옺"#),
        (r#"옻"#, r#"옻"#, r#"옻"#, r#"옻"#, r#"옻"#),
        (r#"옼"#, r#"옼"#, r#"옼"#, r#"옼"#, r#"옼"#),
        (r#"옽"#, r#"옽"#, r#"옽"#, r#"옽"#, r#"옽"#),
        (r#"옾"#, r#"옾"#, r#"옾"#, r#"옾"#, r#"옾"#),
        (r#"옿"#, r#"옿"#, r#"옿"#, r#"옿"#, r#"옿"#),
        (r#"와"#, r#"와"#, r#"와"#, r#"와"#, r#"와"#),
        (r#"왁"#, r#"왁"#, r#"왁"#, r#"왁"#, r#"왁"#),
        (r#"왂"#, r#"왂"#, r#"왂"#, r#"왂"#, r#"왂"#),
        (r#"왃"#, r#"왃"#, r#"왃"#, r#"왃"#, r#"왃"#),
        (r#"완"#, r#"완"#, r#"완"#, r#"완"#, r#"완"#),
        (r#"왅"#, r#"왅"#, r#"왅"#, r#"왅"#, r#"왅"#),
        (r#"왆"#, r#"왆"#, r#"왆"#, r#"왆"#, r#"왆"#),
        (r#"왇"#, r#"왇"#, r#"왇"#, r#"왇"#, r#"왇"#),
        (r#"왈"#, r#"왈"#, r#"왈"#, r#"왈"#, r#"왈"#),
        (r#"왉"#, r#"왉"#, r#"왉"#, r#"왉"#, r#"왉"#),
        (r#"왊"#, r#"왊"#, r#"왊"#, r#"왊"#, r#"왊"#),
        (r#"왋"#, r#"왋"#, r#"왋"#, r#"왋"#, r#"왋"#),
        (r#"왌"#, r#"왌"#, r#"왌"#, r#"왌"#, r#"왌"#),
        (r#"왍"#, r#"왍"#, r#"왍"#, r#"왍"#, r#"왍"#),
        (r#"왎"#, r#"왎"#, r#"왎"#, r#"왎"#, r#"왎"#),
        (r#"왏"#, r#"왏"#, r#"왏"#, r#"왏"#, r#"왏"#),
        (r#"왐"#, r#"왐"#, r#"왐"#, r#"왐"#, r#"왐"#),
        (r#"왑"#, r#"왑"#, r#"왑"#, r#"왑"#, r#"왑"#),
        (r#"왒"#, r#"왒"#, r#"왒"#, r#"왒"#, r#"왒"#),
        (r#"왓"#, r#"왓"#, r#"왓"#, r#"왓"#, r#"왓"#),
        (r#"왔"#, r#"왔"#, r#"왔"#, r#"왔"#, r#"왔"#),
        (r#"왕"#, r#"왕"#, r#"왕"#, r#"왕"#, r#"왕"#),
        (r#"왖"#, r#"왖"#, r#"왖"#, r#"왖"#, r#"왖"#),
        (r#"왗"#, r#"왗"#, r#"왗"#, r#"왗"#, r#"왗"#),
        (r#"왘"#, r#"왘"#, r#"왘"#, r#"왘"#, r#"왘"#),
        (r#"왙"#, r#"왙"#, r#"왙"#, r#"왙"#, r#"왙"#),
        (r#"왚"#, r#"왚"#, r#"왚"#, r#"왚"#, r#"왚"#),
        (r#"왛"#, r#"왛"#, r#"왛"#, r#"왛"#, r#"왛"#),
        (r#"왜"#, r#"왜"#, r#"왜"#, r#"왜"#, r#"왜"#),
        (r#"왝"#, r#"왝"#, r#"왝"#, r#"왝"#, r#"왝"#),
        (r#"왞"#, r#"왞"#, r#"왞"#, r#"왞"#, r#"왞"#),
        (r#"왟"#, r#"왟"#, r#"왟"#, r#"왟"#, r#"왟"#),
        (r#"왠"#, r#"왠"#, r#"왠"#, r#"왠"#, r#"왠"#),
        (r#"왡"#, r#"왡"#, r#"왡"#, r#"왡"#, r#"왡"#),
        (r#"왢"#, r#"왢"#, r#"왢"#, r#"왢"#, r#"왢"#),
        (r#"왣"#, r#"왣"#, r#"왣"#, r#"왣"#, r#"왣"#),
        (r#"왤"#, r#"왤"#, r#"왤"#, r#"왤"#, r#"왤"#),
        (r#"왥"#, r#"왥"#, r#"왥"#, r#"왥"#, r#"왥"#),
        (r#"왦"#, r#"왦"#, r#"왦"#, r#"왦"#, r#"왦"#),
        (r#"왧"#, r#"왧"#, r#"왧"#, r#"왧"#, r#"왧"#),
        (r#"왨"#, r#"왨"#, r#"왨"#, r#"왨"#, r#"왨"#),
        (r#"왩"#, r#"왩"#, r#"왩"#, r#"왩"#, r#"왩"#),
        (r#"왪"#, r#"왪"#, r#"왪"#, r#"왪"#, r#"왪"#),
        (r#"왫"#, r#"왫"#, r#"왫"#, r#"왫"#, r#"왫"#),
        (r#"왬"#, r#"왬"#, r#"왬"#, r#"왬"#, r#"왬"#),
        (r#"왭"#, r#"왭"#, r#"왭"#, r#"왭"#, r#"왭"#),
        (r#"왮"#, r#"왮"#, r#"왮"#, r#"왮"#, r#"왮"#),
        (r#"왯"#, r#"왯"#, r#"왯"#, r#"왯"#, r#"왯"#),
        (r#"왰"#, r#"왰"#, r#"왰"#, r#"왰"#, r#"왰"#),
        (r#"왱"#, r#"왱"#, r#"왱"#, r#"왱"#, r#"왱"#),
        (r#"왲"#, r#"왲"#, r#"왲"#, r#"왲"#, r#"왲"#),
        (r#"왳"#, r#"왳"#, r#"왳"#, r#"왳"#, r#"왳"#),
        (r#"왴"#, r#"왴"#, r#"왴"#, r#"왴"#, r#"왴"#),
        (r#"왵"#, r#"왵"#, r#"왵"#, r#"왵"#, r#"왵"#),
        (r#"왶"#, r#"왶"#, r#"왶"#, r#"왶"#, r#"왶"#),
        (r#"왷"#, r#"왷"#, r#"왷"#, r#"왷"#, r#"왷"#),
        (r#"외"#, r#"외"#, r#"외"#, r#"외"#, r#"외"#),
        (r#"왹"#, r#"왹"#, r#"왹"#, r#"왹"#, r#"왹"#),
        (r#"왺"#, r#"왺"#, r#"왺"#, r#"왺"#, r#"왺"#),
        (r#"왻"#, r#"왻"#, r#"왻"#, r#"왻"#, r#"왻"#),
        (r#"왼"#, r#"왼"#, r#"왼"#, r#"왼"#, r#"왼"#),
        (r#"왽"#, r#"왽"#, r#"왽"#, r#"왽"#, r#"왽"#),
        (r#"왾"#, r#"왾"#, r#"왾"#, r#"왾"#, r#"왾"#),
        (r#"왿"#, r#"왿"#, r#"왿"#, r#"왿"#, r#"왿"#),
        (r#"욀"#, r#"욀"#, r#"욀"#, r#"욀"#, r#"욀"#),
        (r#"욁"#, r#"욁"#, r#"욁"#, r#"욁"#, r#"욁"#),
        (r#"욂"#, r#"욂"#, r#"욂"#, r#"욂"#, r#"욂"#),
        (r#"욃"#, r#"욃"#, r#"욃"#, r#"욃"#, r#"욃"#),
        (r#"욄"#, r#"욄"#, r#"욄"#, r#"욄"#, r#"욄"#),
        (r#"욅"#, r#"욅"#, r#"욅"#, r#"욅"#, r#"욅"#),
        (r#"욆"#, r#"욆"#, r#"욆"#, r#"욆"#, r#"욆"#),
        (r#"욇"#, r#"욇"#, r#"욇"#, r#"욇"#, r#"욇"#),
        (r#"욈"#, r#"욈"#, r#"욈"#, r#"욈"#, r#"욈"#),
        (r#"욉"#, r#"욉"#, r#"욉"#, r#"욉"#, r#"욉"#),
        (r#"욊"#, r#"욊"#, r#"욊"#, r#"욊"#, r#"욊"#),
        (r#"욋"#, r#"욋"#, r#"욋"#, r#"욋"#, r#"욋"#),
        (r#"욌"#, r#"욌"#, r#"욌"#, r#"욌"#, r#"욌"#),
        (r#"욍"#, r#"욍"#, r#"욍"#, r#"욍"#, r#"욍"#),
        (r#"욎"#, r#"욎"#, r#"욎"#, r#"욎"#, r#"욎"#),
        (r#"욏"#, r#"욏"#, r#"욏"#, r#"욏"#, r#"욏"#),
        (r#"욐"#, r#"욐"#, r#"욐"#, r#"욐"#, r#"욐"#),
        (r#"욑"#, r#"욑"#, r#"욑"#, r#"욑"#, r#"욑"#),
        (r#"욒"#, r#"욒"#, r#"욒"#, r#"욒"#, r#"욒"#),
        (r#"욓"#, r#"욓"#, r#"욓"#, r#"욓"#, r#"욓"#),
        (r#"요"#, r#"요"#, r#"요"#, r#"요"#, r#"요"#),
        (r#"욕"#, r#"욕"#, r#"욕"#, r#"욕"#, r#"욕"#),
        (r#"욖"#, r#"욖"#, r#"욖"#, r#"욖"#, r#"욖"#),
        (r#"욗"#, r#"욗"#, r#"욗"#, r#"욗"#, r#"욗"#),
        (r#"욘"#, r#"욘"#, r#"욘"#, r#"욘"#, r#"욘"#),
        (r#"욙"#, r#"욙"#, r#"욙"#, r#"욙"#, r#"욙"#),
        (r#"욚"#, r#"욚"#, r#"욚"#, r#"욚"#, r#"욚"#),
        (r#"욛"#, r#"욛"#, r#"욛"#, r#"욛"#, r#"욛"#),
        (r#"욜"#, r#"욜"#, r#"욜"#, r#"욜"#, r#"욜"#),
        (r#"욝"#, r#"욝"#, r#"욝"#, r#"욝"#, r#"욝"#),
        (r#"욞"#, r#"욞"#, r#"욞"#, r#"욞"#, r#"욞"#),
        (r#"욟"#, r#"욟"#, r#"욟"#, r#"욟"#, r#"욟"#),
        (r#"욠"#, r#"욠"#, r#"욠"#, r#"욠"#, r#"욠"#),
        (r#"욡"#, r#"욡"#, r#"욡"#, r#"욡"#, r#"욡"#),
        (r#"욢"#, r#"욢"#, r#"욢"#, r#"욢"#, r#"욢"#),
        (r#"욣"#, r#"욣"#, r#"욣"#, r#"욣"#, r#"욣"#),
        (r#"욤"#, r#"욤"#, r#"욤"#, r#"욤"#, r#"욤"#),
        (r#"욥"#, r#"욥"#, r#"욥"#, r#"욥"#, r#"욥"#),
        (r#"욦"#, r#"욦"#, r#"욦"#, r#"욦"#, r#"욦"#),
        (r#"욧"#, r#"욧"#, r#"욧"#, r#"욧"#, r#"욧"#),
        (r#"욨"#, r#"욨"#, r#"욨"#, r#"욨"#, r#"욨"#),
        (r#"용"#, r#"용"#, r#"용"#, r#"용"#, r#"용"#),
        (r#"욪"#, r#"욪"#, r#"욪"#, r#"욪"#, r#"욪"#),
        (r#"욫"#, r#"욫"#, r#"욫"#, r#"욫"#, r#"욫"#),
        (r#"욬"#, r#"욬"#, r#"욬"#, r#"욬"#, r#"욬"#),
        (r#"욭"#, r#"욭"#, r#"욭"#, r#"욭"#, r#"욭"#),
        (r#"욮"#, r#"욮"#, r#"욮"#, r#"욮"#, r#"욮"#),
        (r#"욯"#, r#"욯"#, r#"욯"#, r#"욯"#, r#"욯"#),
        (r#"우"#, r#"우"#, r#"우"#, r#"우"#, r#"우"#),
        (r#"욱"#, r#"욱"#, r#"욱"#, r#"욱"#, r#"욱"#),
        (r#"욲"#, r#"욲"#, r#"욲"#, r#"욲"#, r#"욲"#),
        (r#"욳"#, r#"욳"#, r#"욳"#, r#"욳"#, r#"욳"#),
        (r#"운"#, r#"운"#, r#"운"#, r#"운"#, r#"운"#),
        (r#"욵"#, r#"욵"#, r#"욵"#, r#"욵"#, r#"욵"#),
        (r#"욶"#, r#"욶"#, r#"욶"#, r#"욶"#, r#"욶"#),
        (r#"욷"#, r#"욷"#, r#"욷"#, r#"욷"#, r#"욷"#),
        (r#"울"#, r#"울"#, r#"울"#, r#"울"#, r#"울"#),
        (r#"욹"#, r#"욹"#, r#"욹"#, r#"욹"#, r#"욹"#),
        (r#"욺"#, r#"욺"#, r#"욺"#, r#"욺"#, r#"욺"#),
        (r#"욻"#, r#"욻"#, r#"욻"#, r#"욻"#, r#"욻"#),
        (r#"욼"#, r#"욼"#, r#"욼"#, r#"욼"#, r#"욼"#),
        (r#"욽"#, r#"욽"#, r#"욽"#, r#"욽"#, r#"욽"#),
        (r#"욾"#, r#"욾"#, r#"욾"#, r#"욾"#, r#"욾"#),
        (r#"욿"#, r#"욿"#, r#"욿"#, r#"욿"#, r#"욿"#),
        (r#"움"#, r#"움"#, r#"움"#, r#"움"#, r#"움"#),
        (r#"웁"#, r#"웁"#, r#"웁"#, r#"웁"#, r#"웁"#),
        (r#"웂"#, r#"웂"#, r#"웂"#, r#"웂"#, r#"웂"#),
        (r#"웃"#, r#"웃"#, r#"웃"#, r#"웃"#, r#"웃"#),
        (r#"웄"#, r#"웄"#, r#"웄"#, r#"웄"#, r#"웄"#),
        (r#"웅"#, r#"웅"#, r#"웅"#, r#"웅"#, r#"웅"#),
        (r#"웆"#, r#"웆"#, r#"웆"#, r#"웆"#, r#"웆"#),
        (r#"웇"#, r#"웇"#, r#"웇"#, r#"웇"#, r#"웇"#),
        (r#"웈"#, r#"웈"#, r#"웈"#, r#"웈"#, r#"웈"#),
        (r#"웉"#, r#"웉"#, r#"웉"#, r#"웉"#, r#"웉"#),
        (r#"웊"#, r#"웊"#, r#"웊"#, r#"웊"#, r#"웊"#),
        (r#"웋"#, r#"웋"#, r#"웋"#, r#"웋"#, r#"웋"#),
        (r#"워"#, r#"워"#, r#"워"#, r#"워"#, r#"워"#),
        (r#"웍"#, r#"웍"#, r#"웍"#, r#"웍"#, r#"웍"#),
        (r#"웎"#, r#"웎"#, r#"웎"#, r#"웎"#, r#"웎"#),
        (r#"웏"#, r#"웏"#, r#"웏"#, r#"웏"#, r#"웏"#),
        (r#"원"#, r#"원"#, r#"원"#, r#"원"#, r#"원"#),
        (r#"웑"#, r#"웑"#, r#"웑"#, r#"웑"#, r#"웑"#),
        (r#"웒"#, r#"웒"#, r#"웒"#, r#"웒"#, r#"웒"#),
        (r#"웓"#, r#"웓"#, r#"웓"#, r#"웓"#, r#"웓"#),
        (r#"월"#, r#"월"#, r#"월"#, r#"월"#, r#"월"#),
        (r#"웕"#, r#"웕"#, r#"웕"#, r#"웕"#, r#"웕"#),
        (r#"웖"#, r#"웖"#, r#"웖"#, r#"웖"#, r#"웖"#),
        (r#"웗"#, r#"웗"#, r#"웗"#, r#"웗"#, r#"웗"#),
        (r#"웘"#, r#"웘"#, r#"웘"#, r#"웘"#, r#"웘"#),
        (r#"웙"#, r#"웙"#, r#"웙"#, r#"웙"#, r#"웙"#),
        (r#"웚"#, r#"웚"#, r#"웚"#, r#"웚"#, r#"웚"#),
        (r#"웛"#, r#"웛"#, r#"웛"#, r#"웛"#, r#"웛"#),
        (r#"웜"#, r#"웜"#, r#"웜"#, r#"웜"#, r#"웜"#),
        (r#"웝"#, r#"웝"#, r#"웝"#, r#"웝"#, r#"웝"#),
        (r#"웞"#, r#"웞"#, r#"웞"#, r#"웞"#, r#"웞"#),
        (r#"웟"#, r#"웟"#, r#"웟"#, r#"웟"#, r#"웟"#),
        (r#"웠"#, r#"웠"#, r#"웠"#, r#"웠"#, r#"웠"#),
        (r#"웡"#, r#"웡"#, r#"웡"#, r#"웡"#, r#"웡"#),
        (r#"웢"#, r#"웢"#, r#"웢"#, r#"웢"#, r#"웢"#),
        (r#"웣"#, r#"웣"#, r#"웣"#, r#"웣"#, r#"웣"#),
        (r#"웤"#, r#"웤"#, r#"웤"#, r#"웤"#, r#"웤"#),
        (r#"웥"#, r#"웥"#, r#"웥"#, r#"웥"#, r#"웥"#),
        (r#"웦"#, r#"웦"#, r#"웦"#, r#"웦"#, r#"웦"#),
        (r#"웧"#, r#"웧"#, r#"웧"#, r#"웧"#, r#"웧"#),
        (r#"웨"#, r#"웨"#, r#"웨"#, r#"웨"#, r#"웨"#),
        (r#"웩"#, r#"웩"#, r#"웩"#, r#"웩"#, r#"웩"#),
        (r#"웪"#, r#"웪"#, r#"웪"#, r#"웪"#, r#"웪"#),
        (r#"웫"#, r#"웫"#, r#"웫"#, r#"웫"#, r#"웫"#),
        (r#"웬"#, r#"웬"#, r#"웬"#, r#"웬"#, r#"웬"#),
        (r#"웭"#, r#"웭"#, r#"웭"#, r#"웭"#, r#"웭"#),
        (r#"웮"#, r#"웮"#, r#"웮"#, r#"웮"#, r#"웮"#),
        (r#"웯"#, r#"웯"#, r#"웯"#, r#"웯"#, r#"웯"#),
        (r#"웰"#, r#"웰"#, r#"웰"#, r#"웰"#, r#"웰"#),
        (r#"웱"#, r#"웱"#, r#"웱"#, r#"웱"#, r#"웱"#),
        (r#"웲"#, r#"웲"#, r#"웲"#, r#"웲"#, r#"웲"#),
        (r#"웳"#, r#"웳"#, r#"웳"#, r#"웳"#, r#"웳"#),
        (r#"웴"#, r#"웴"#, r#"웴"#, r#"웴"#, r#"웴"#),
        (r#"웵"#, r#"웵"#, r#"웵"#, r#"웵"#, r#"웵"#),
        (r#"웶"#, r#"웶"#, r#"웶"#, r#"웶"#, r#"웶"#),
        (r#"웷"#, r#"웷"#, r#"웷"#, r#"웷"#, r#"웷"#),
        (r#"웸"#, r#"웸"#, r#"웸"#, r#"웸"#, r#"웸"#),
        (r#"웹"#, r#"웹"#, r#"웹"#, r#"웹"#, r#"웹"#),
        (r#"웺"#, r#"웺"#, r#"웺"#, r#"웺"#, r#"웺"#),
        (r#"웻"#, r#"웻"#, r#"웻"#, r#"웻"#, r#"웻"#),
        (r#"웼"#, r#"웼"#, r#"웼"#, r#"웼"#, r#"웼"#),
        (r#"웽"#, r#"웽"#, r#"웽"#, r#"웽"#, r#"웽"#),
        (r#"웾"#, r#"웾"#, r#"웾"#, r#"웾"#, r#"웾"#),
        (r#"웿"#, r#"웿"#, r#"웿"#, r#"웿"#, r#"웿"#),
        (r#"윀"#, r#"윀"#, r#"윀"#, r#"윀"#, r#"윀"#),
        (r#"윁"#, r#"윁"#, r#"윁"#, r#"윁"#, r#"윁"#),
        (r#"윂"#, r#"윂"#, r#"윂"#, r#"윂"#, r#"윂"#),
        (r#"윃"#, r#"윃"#, r#"윃"#, r#"윃"#, r#"윃"#),
        (r#"위"#, r#"위"#, r#"위"#, r#"위"#, r#"위"#),
        (r#"윅"#, r#"윅"#, r#"윅"#, r#"윅"#, r#"윅"#),
        (r#"윆"#, r#"윆"#, r#"윆"#, r#"윆"#, r#"윆"#),
        (r#"윇"#, r#"윇"#, r#"윇"#, r#"윇"#, r#"윇"#),
        (r#"윈"#, r#"윈"#, r#"윈"#, r#"윈"#, r#"윈"#),
        (r#"윉"#, r#"윉"#, r#"윉"#, r#"윉"#, r#"윉"#),
        (r#"윊"#, r#"윊"#, r#"윊"#, r#"윊"#, r#"윊"#),
        (r#"윋"#, r#"윋"#, r#"윋"#, r#"윋"#, r#"윋"#),
        (r#"윌"#, r#"윌"#, r#"윌"#, r#"윌"#, r#"윌"#),
        (r#"윍"#, r#"윍"#, r#"윍"#, r#"윍"#, r#"윍"#),
        (r#"윎"#, r#"윎"#, r#"윎"#, r#"윎"#, r#"윎"#),
        (r#"윏"#, r#"윏"#, r#"윏"#, r#"윏"#, r#"윏"#),
        (r#"윐"#, r#"윐"#, r#"윐"#, r#"윐"#, r#"윐"#),
        (r#"윑"#, r#"윑"#, r#"윑"#, r#"윑"#, r#"윑"#),
        (r#"윒"#, r#"윒"#, r#"윒"#, r#"윒"#, r#"윒"#),
        (r#"윓"#, r#"윓"#, r#"윓"#, r#"윓"#, r#"윓"#),
        (r#"윔"#, r#"윔"#, r#"윔"#, r#"윔"#, r#"윔"#),
        (r#"윕"#, r#"윕"#, r#"윕"#, r#"윕"#, r#"윕"#),
        (r#"윖"#, r#"윖"#, r#"윖"#, r#"윖"#, r#"윖"#),
        (r#"윗"#, r#"윗"#, r#"윗"#, r#"윗"#, r#"윗"#),
        (r#"윘"#, r#"윘"#, r#"윘"#, r#"윘"#, r#"윘"#),
        (r#"윙"#, r#"윙"#, r#"윙"#, r#"윙"#, r#"윙"#),
        (r#"윚"#, r#"윚"#, r#"윚"#, r#"윚"#, r#"윚"#),
        (r#"윛"#, r#"윛"#, r#"윛"#, r#"윛"#, r#"윛"#),
        (r#"윜"#, r#"윜"#, r#"윜"#, r#"윜"#, r#"윜"#),
        (r#"윝"#, r#"윝"#, r#"윝"#, r#"윝"#, r#"윝"#),
        (r#"윞"#, r#"윞"#, r#"윞"#, r#"윞"#, r#"윞"#),
        (r#"윟"#, r#"윟"#, r#"윟"#, r#"윟"#, r#"윟"#),
        (r#"유"#, r#"유"#, r#"유"#, r#"유"#, r#"유"#),
        (r#"육"#, r#"육"#, r#"육"#, r#"육"#, r#"육"#),
        (r#"윢"#, r#"윢"#, r#"윢"#, r#"윢"#, r#"윢"#),
        (r#"윣"#, r#"윣"#, r#"윣"#, r#"윣"#, r#"윣"#),
        (r#"윤"#, r#"윤"#, r#"윤"#, r#"윤"#, r#"윤"#),
        (r#"윥"#, r#"윥"#, r#"윥"#, r#"윥"#, r#"윥"#),
        (r#"윦"#, r#"윦"#, r#"윦"#, r#"윦"#, r#"윦"#),
        (r#"윧"#, r#"윧"#, r#"윧"#, r#"윧"#, r#"윧"#),
        (r#"율"#, r#"율"#, r#"율"#, r#"율"#, r#"율"#),
        (r#"윩"#, r#"윩"#, r#"윩"#, r#"윩"#, r#"윩"#),
        (r#"윪"#, r#"윪"#, r#"윪"#, r#"윪"#, r#"윪"#),
        (r#"윫"#, r#"윫"#, r#"윫"#, r#"윫"#, r#"윫"#),
        (r#"윬"#, r#"윬"#, r#"윬"#, r#"윬"#, r#"윬"#),
        (r#"윭"#, r#"윭"#, r#"윭"#, r#"윭"#, r#"윭"#),
        (r#"윮"#, r#"윮"#, r#"윮"#, r#"윮"#, r#"윮"#),
        (r#"윯"#, r#"윯"#, r#"윯"#, r#"윯"#, r#"윯"#),
        (r#"윰"#, r#"윰"#, r#"윰"#, r#"윰"#, r#"윰"#),
        (r#"윱"#, r#"윱"#, r#"윱"#, r#"윱"#, r#"윱"#),
        (r#"윲"#, r#"윲"#, r#"윲"#, r#"윲"#, r#"윲"#),
        (r#"윳"#, r#"윳"#, r#"윳"#, r#"윳"#, r#"윳"#),
        (r#"윴"#, r#"윴"#, r#"윴"#, r#"윴"#, r#"윴"#),
        (r#"융"#, r#"융"#, r#"융"#, r#"융"#, r#"융"#),
        (r#"윶"#, r#"윶"#, r#"윶"#, r#"윶"#, r#"윶"#),
        (r#"윷"#, r#"윷"#, r#"윷"#, r#"윷"#, r#"윷"#),
        (r#"윸"#, r#"윸"#, r#"윸"#, r#"윸"#, r#"윸"#),
        (r#"윹"#, r#"윹"#, r#"윹"#, r#"윹"#, r#"윹"#),
        (r#"윺"#, r#"윺"#, r#"윺"#, r#"윺"#, r#"윺"#),
        (r#"윻"#, r#"윻"#, r#"윻"#, r#"윻"#, r#"윻"#),
        (r#"으"#, r#"으"#, r#"으"#, r#"으"#, r#"으"#),
        (r#"윽"#, r#"윽"#, r#"윽"#, r#"윽"#, r#"윽"#),
        (r#"윾"#, r#"윾"#, r#"윾"#, r#"윾"#, r#"윾"#),
        (r#"윿"#, r#"윿"#, r#"윿"#, r#"윿"#, r#"윿"#),
        (r#"은"#, r#"은"#, r#"은"#, r#"은"#, r#"은"#),
        (r#"읁"#, r#"읁"#, r#"읁"#, r#"읁"#, r#"읁"#),
        (r#"읂"#, r#"읂"#, r#"읂"#, r#"읂"#, r#"읂"#),
        (r#"읃"#, r#"읃"#, r#"읃"#, r#"읃"#, r#"읃"#),
        (r#"을"#, r#"을"#, r#"을"#, r#"을"#, r#"을"#),
        (r#"읅"#, r#"읅"#, r#"읅"#, r#"읅"#, r#"읅"#),
        (r#"읆"#, r#"읆"#, r#"읆"#, r#"읆"#, r#"읆"#),
        (r#"읇"#, r#"읇"#, r#"읇"#, r#"읇"#, r#"읇"#),
        (r#"읈"#, r#"읈"#, r#"읈"#, r#"읈"#, r#"읈"#),
        (r#"읉"#, r#"읉"#, r#"읉"#, r#"읉"#, r#"읉"#),
        (r#"읊"#, r#"읊"#, r#"읊"#, r#"읊"#, r#"읊"#),
        (r#"읋"#, r#"읋"#, r#"읋"#, r#"읋"#, r#"읋"#),
        (r#"음"#, r#"음"#, r#"음"#, r#"음"#, r#"음"#),
        (r#"읍"#, r#"읍"#, r#"읍"#, r#"읍"#, r#"읍"#),
        (r#"읎"#, r#"읎"#, r#"읎"#, r#"읎"#, r#"읎"#),
        (r#"읏"#, r#"읏"#, r#"읏"#, r#"읏"#, r#"읏"#),
        (r#"읐"#, r#"읐"#, r#"읐"#, r#"읐"#, r#"읐"#),
        (r#"응"#, r#"응"#, r#"응"#, r#"응"#, r#"응"#),
        (r#"읒"#, r#"읒"#, r#"읒"#, r#"읒"#, r#"읒"#),
        (r#"읓"#, r#"읓"#, r#"읓"#, r#"읓"#, r#"읓"#),
        (r#"읔"#, r#"읔"#, r#"읔"#, r#"읔"#, r#"읔"#),
        (r#"읕"#, r#"읕"#, r#"읕"#, r#"읕"#, r#"읕"#),
        (r#"읖"#, r#"읖"#, r#"읖"#, r#"읖"#, r#"읖"#),
        (r#"읗"#, r#"읗"#, r#"읗"#, r#"읗"#, r#"읗"#),
        (r#"의"#, r#"의"#, r#"의"#, r#"의"#, r#"의"#),
        (r#"읙"#, r#"읙"#, r#"읙"#, r#"읙"#, r#"읙"#),
        (r#"읚"#, r#"읚"#, r#"읚"#, r#"읚"#, r#"읚"#),
        (r#"읛"#, r#"읛"#, r#"읛"#, r#"읛"#, r#"읛"#),
        (r#"읜"#, r#"읜"#, r#"읜"#, r#"읜"#, r#"읜"#),
        (r#"읝"#, r#"읝"#, r#"읝"#, r#"읝"#, r#"읝"#),
        (r#"읞"#, r#"읞"#, r#"읞"#, r#"읞"#, r#"읞"#),
        (r#"읟"#, r#"읟"#, r#"읟"#, r#"읟"#, r#"읟"#),
        (r#"읠"#, r#"읠"#, r#"읠"#, r#"읠"#, r#"읠"#),
        (r#"읡"#, r#"읡"#, r#"읡"#, r#"읡"#, r#"읡"#),
        (r#"읢"#, r#"읢"#, r#"읢"#, r#"읢"#, r#"읢"#),
        (r#"읣"#, r#"읣"#, r#"읣"#, r#"읣"#, r#"읣"#),
        (r#"읤"#, r#"읤"#, r#"읤"#, r#"읤"#, r#"읤"#),
        (r#"읥"#, r#"읥"#, r#"읥"#, r#"읥"#, r#"읥"#),
        (r#"읦"#, r#"읦"#, r#"읦"#, r#"읦"#, r#"읦"#),
        (r#"읧"#, r#"읧"#, r#"읧"#, r#"읧"#, r#"읧"#),
        (r#"읨"#, r#"읨"#, r#"읨"#, r#"읨"#, r#"읨"#),
        (r#"읩"#, r#"읩"#, r#"읩"#, r#"읩"#, r#"읩"#),
        (r#"읪"#, r#"읪"#, r#"읪"#, r#"읪"#, r#"읪"#),
        (r#"읫"#, r#"읫"#, r#"읫"#, r#"읫"#, r#"읫"#),
        (r#"읬"#, r#"읬"#, r#"읬"#, r#"읬"#, r#"읬"#),
        (r#"읭"#, r#"읭"#, r#"읭"#, r#"읭"#, r#"읭"#),
        (r#"읮"#, r#"읮"#, r#"읮"#, r#"읮"#, r#"읮"#),
        (r#"읯"#, r#"읯"#, r#"읯"#, r#"읯"#, r#"읯"#),
        (r#"읰"#, r#"읰"#, r#"읰"#, r#"읰"#, r#"읰"#),
        (r#"읱"#, r#"읱"#, r#"읱"#, r#"읱"#, r#"읱"#),
        (r#"읲"#, r#"읲"#, r#"읲"#, r#"읲"#, r#"읲"#),
        (r#"읳"#, r#"읳"#, r#"읳"#, r#"읳"#, r#"읳"#),
        (r#"이"#, r#"이"#, r#"이"#, r#"이"#, r#"이"#),
        (r#"익"#, r#"익"#, r#"익"#, r#"익"#, r#"익"#),
        (r#"읶"#, r#"읶"#, r#"읶"#, r#"읶"#, r#"읶"#),
        (r#"읷"#, r#"읷"#, r#"읷"#, r#"읷"#, r#"읷"#),
        (r#"인"#, r#"인"#, r#"인"#, r#"인"#, r#"인"#),
        (r#"읹"#, r#"읹"#, r#"읹"#, r#"읹"#, r#"읹"#),
        (r#"읺"#, r#"읺"#, r#"읺"#, r#"읺"#, r#"읺"#),
        (r#"읻"#, r#"읻"#, r#"읻"#, r#"읻"#, r#"읻"#),
        (r#"일"#, r#"일"#, r#"일"#, r#"일"#, r#"일"#),
        (r#"읽"#, r#"읽"#, r#"읽"#, r#"읽"#, r#"읽"#),
        (r#"읾"#, r#"읾"#, r#"읾"#, r#"읾"#, r#"읾"#),
        (r#"읿"#, r#"읿"#, r#"읿"#, r#"읿"#, r#"읿"#),
        (r#"잀"#, r#"잀"#, r#"잀"#, r#"잀"#, r#"잀"#),
        (r#"잁"#, r#"잁"#, r#"잁"#, r#"잁"#, r#"잁"#),
        (r#"잂"#, r#"잂"#, r#"잂"#, r#"잂"#, r#"잂"#),
        (r#"잃"#, r#"잃"#, r#"잃"#, r#"잃"#, r#"잃"#),
        (r#"임"#, r#"임"#, r#"임"#, r#"임"#, r#"임"#),
        (r#"입"#, r#"입"#, r#"입"#, r#"입"#, r#"입"#),
        (r#"잆"#, r#"잆"#, r#"잆"#, r#"잆"#, r#"잆"#),
        (r#"잇"#, r#"잇"#, r#"잇"#, r#"잇"#, r#"잇"#),
        (r#"있"#, r#"있"#, r#"있"#, r#"있"#, r#"있"#),
        (r#"잉"#, r#"잉"#, r#"잉"#, r#"잉"#, r#"잉"#),
        (r#"잊"#, r#"잊"#, r#"잊"#, r#"잊"#, r#"잊"#),
        (r#"잋"#, r#"잋"#, r#"잋"#, r#"잋"#, r#"잋"#),
        (r#"잌"#, r#"잌"#, r#"잌"#, r#"잌"#, r#"잌"#),
        (r#"잍"#, r#"잍"#, r#"잍"#, r#"잍"#, r#"잍"#),
        (r#"잎"#, r#"잎"#, r#"잎"#, r#"잎"#, r#"잎"#),
        (r#"잏"#, r#"잏"#, r#"잏"#, r#"잏"#, r#"잏"#),
        (r#"자"#, r#"자"#, r#"자"#, r#"자"#, r#"자"#),
        (r#"작"#, r#"작"#, r#"작"#, r#"작"#, r#"작"#),
        (r#"잒"#, r#"잒"#, r#"잒"#, r#"잒"#, r#"잒"#),
        (r#"잓"#, r#"잓"#, r#"잓"#, r#"잓"#, r#"잓"#),
        (r#"잔"#, r#"잔"#, r#"잔"#, r#"잔"#, r#"잔"#),
        (r#"잕"#, r#"잕"#, r#"잕"#, r#"잕"#, r#"잕"#),
        (r#"잖"#, r#"잖"#, r#"잖"#, r#"잖"#, r#"잖"#),
        (r#"잗"#, r#"잗"#, r#"잗"#, r#"잗"#, r#"잗"#),
        (r#"잘"#, r#"잘"#, r#"잘"#, r#"잘"#, r#"잘"#),
        (r#"잙"#, r#"잙"#, r#"잙"#, r#"잙"#, r#"잙"#),
        (r#"잚"#, r#"잚"#, r#"잚"#, r#"잚"#, r#"잚"#),
        (r#"잛"#, r#"잛"#, r#"잛"#, r#"잛"#, r#"잛"#),
        (r#"잜"#, r#"잜"#, r#"잜"#, r#"잜"#, r#"잜"#),
        (r#"잝"#, r#"잝"#, r#"잝"#, r#"잝"#, r#"잝"#),
        (r#"잞"#, r#"잞"#, r#"잞"#, r#"잞"#, r#"잞"#),
        (r#"잟"#, r#"잟"#, r#"잟"#, r#"잟"#, r#"잟"#),
        (r#"잠"#, r#"잠"#, r#"잠"#, r#"잠"#, r#"잠"#),
        (r#"잡"#, r#"잡"#, r#"잡"#, r#"잡"#, r#"잡"#),
        (r#"잢"#, r#"잢"#, r#"잢"#, r#"잢"#, r#"잢"#),
        (r#"잣"#, r#"잣"#, r#"잣"#, r#"잣"#, r#"잣"#),
        (r#"잤"#, r#"잤"#, r#"잤"#, r#"잤"#, r#"잤"#),
        (r#"장"#, r#"장"#, r#"장"#, r#"장"#, r#"장"#),
        (r#"잦"#, r#"잦"#, r#"잦"#, r#"잦"#, r#"잦"#),
        (r#"잧"#, r#"잧"#, r#"잧"#, r#"잧"#, r#"잧"#),
        (r#"잨"#, r#"잨"#, r#"잨"#, r#"잨"#, r#"잨"#),
        (r#"잩"#, r#"잩"#, r#"잩"#, r#"잩"#, r#"잩"#),
        (r#"잪"#, r#"잪"#, r#"잪"#, r#"잪"#, r#"잪"#),
        (r#"잫"#, r#"잫"#, r#"잫"#, r#"잫"#, r#"잫"#),
        (r#"재"#, r#"재"#, r#"재"#, r#"재"#, r#"재"#),
        (r#"잭"#, r#"잭"#, r#"잭"#, r#"잭"#, r#"잭"#),
        (r#"잮"#, r#"잮"#, r#"잮"#, r#"잮"#, r#"잮"#),
        (r#"잯"#, r#"잯"#, r#"잯"#, r#"잯"#, r#"잯"#),
        (r#"잰"#, r#"잰"#, r#"잰"#, r#"잰"#, r#"잰"#),
        (r#"잱"#, r#"잱"#, r#"잱"#, r#"잱"#, r#"잱"#),
        (r#"잲"#, r#"잲"#, r#"잲"#, r#"잲"#, r#"잲"#),
        (r#"잳"#, r#"잳"#, r#"잳"#, r#"잳"#, r#"잳"#),
        (r#"잴"#, r#"잴"#, r#"잴"#, r#"잴"#, r#"잴"#),
        (r#"잵"#, r#"잵"#, r#"잵"#, r#"잵"#, r#"잵"#),
        (r#"잶"#, r#"잶"#, r#"잶"#, r#"잶"#, r#"잶"#),
        (r#"잷"#, r#"잷"#, r#"잷"#, r#"잷"#, r#"잷"#),
        (r#"잸"#, r#"잸"#, r#"잸"#, r#"잸"#, r#"잸"#),
        (r#"잹"#, r#"잹"#, r#"잹"#, r#"잹"#, r#"잹"#),
        (r#"잺"#, r#"잺"#, r#"잺"#, r#"잺"#, r#"잺"#),
        (r#"잻"#, r#"잻"#, r#"잻"#, r#"잻"#, r#"잻"#),
        (r#"잼"#, r#"잼"#, r#"잼"#, r#"잼"#, r#"잼"#),
        (r#"잽"#, r#"잽"#, r#"잽"#, r#"잽"#, r#"잽"#),
        (r#"잾"#, r#"잾"#, r#"잾"#, r#"잾"#, r#"잾"#),
        (r#"잿"#, r#"잿"#, r#"잿"#, r#"잿"#, r#"잿"#),
        (r#"쟀"#, r#"쟀"#, r#"쟀"#, r#"쟀"#, r#"쟀"#),
        (r#"쟁"#, r#"쟁"#, r#"쟁"#, r#"쟁"#, r#"쟁"#),
        (r#"쟂"#, r#"쟂"#, r#"쟂"#, r#"쟂"#, r#"쟂"#),
        (r#"쟃"#, r#"쟃"#, r#"쟃"#, r#"쟃"#, r#"쟃"#),
        (r#"쟄"#, r#"쟄"#, r#"쟄"#, r#"쟄"#, r#"쟄"#),
        (r#"쟅"#, r#"쟅"#, r#"쟅"#, r#"쟅"#, r#"쟅"#),
        (r#"쟆"#, r#"쟆"#, r#"쟆"#, r#"쟆"#, r#"쟆"#),
        (r#"쟇"#, r#"쟇"#, r#"쟇"#, r#"쟇"#, r#"쟇"#),
        (r#"쟈"#, r#"쟈"#, r#"쟈"#, r#"쟈"#, r#"쟈"#),
        (r#"쟉"#, r#"쟉"#, r#"쟉"#, r#"쟉"#, r#"쟉"#),
        (r#"쟊"#, r#"쟊"#, r#"쟊"#, r#"쟊"#, r#"쟊"#),
        (r#"쟋"#, r#"쟋"#, r#"쟋"#, r#"쟋"#, r#"쟋"#),
        (r#"쟌"#, r#"쟌"#, r#"쟌"#, r#"쟌"#, r#"쟌"#),
        (r#"쟍"#, r#"쟍"#, r#"쟍"#, r#"쟍"#, r#"쟍"#),
        (r#"쟎"#, r#"쟎"#, r#"쟎"#, r#"쟎"#, r#"쟎"#),
        (r#"쟏"#, r#"쟏"#, r#"쟏"#, r#"쟏"#, r#"쟏"#),
        (r#"쟐"#, r#"쟐"#, r#"쟐"#, r#"쟐"#, r#"쟐"#),
        (r#"쟑"#, r#"쟑"#, r#"쟑"#, r#"쟑"#, r#"쟑"#),
        (r#"쟒"#, r#"쟒"#, r#"쟒"#, r#"쟒"#, r#"쟒"#),
        (r#"쟓"#, r#"쟓"#, r#"쟓"#, r#"쟓"#, r#"쟓"#),
        (r#"쟔"#, r#"쟔"#, r#"쟔"#, r#"쟔"#, r#"쟔"#),
        (r#"쟕"#, r#"쟕"#, r#"쟕"#, r#"쟕"#, r#"쟕"#),
        (r#"쟖"#, r#"쟖"#, r#"쟖"#, r#"쟖"#, r#"쟖"#),
        (r#"쟗"#, r#"쟗"#, r#"쟗"#, r#"쟗"#, r#"쟗"#),
        (r#"쟘"#, r#"쟘"#, r#"쟘"#, r#"쟘"#, r#"쟘"#),
        (r#"쟙"#, r#"쟙"#, r#"쟙"#, r#"쟙"#, r#"쟙"#),
        (r#"쟚"#, r#"쟚"#, r#"쟚"#, r#"쟚"#, r#"쟚"#),
        (r#"쟛"#, r#"쟛"#, r#"쟛"#, r#"쟛"#, r#"쟛"#),
        (r#"쟜"#, r#"쟜"#, r#"쟜"#, r#"쟜"#, r#"쟜"#),
        (r#"쟝"#, r#"쟝"#, r#"쟝"#, r#"쟝"#, r#"쟝"#),
        (r#"쟞"#, r#"쟞"#, r#"쟞"#, r#"쟞"#, r#"쟞"#),
        (r#"쟟"#, r#"쟟"#, r#"쟟"#, r#"쟟"#, r#"쟟"#),
        (r#"쟠"#, r#"쟠"#, r#"쟠"#, r#"쟠"#, r#"쟠"#),
        (r#"쟡"#, r#"쟡"#, r#"쟡"#, r#"쟡"#, r#"쟡"#),
        (r#"쟢"#, r#"쟢"#, r#"쟢"#, r#"쟢"#, r#"쟢"#),
        (r#"쟣"#, r#"쟣"#, r#"쟣"#, r#"쟣"#, r#"쟣"#),
        (r#"쟤"#, r#"쟤"#, r#"쟤"#, r#"쟤"#, r#"쟤"#),
        (r#"쟥"#, r#"쟥"#, r#"쟥"#, r#"쟥"#, r#"쟥"#),
        (r#"쟦"#, r#"쟦"#, r#"쟦"#, r#"쟦"#, r#"쟦"#),
        (r#"쟧"#, r#"쟧"#, r#"쟧"#, r#"쟧"#, r#"쟧"#),
        (r#"쟨"#, r#"쟨"#, r#"쟨"#, r#"쟨"#, r#"쟨"#),
        (r#"쟩"#, r#"쟩"#, r#"쟩"#, r#"쟩"#, r#"쟩"#),
        (r#"쟪"#, r#"쟪"#, r#"쟪"#, r#"쟪"#, r#"쟪"#),
        (r#"쟫"#, r#"쟫"#, r#"쟫"#, r#"쟫"#, r#"쟫"#),
        (r#"쟬"#, r#"쟬"#, r#"쟬"#, r#"쟬"#, r#"쟬"#),
        (r#"쟭"#, r#"쟭"#, r#"쟭"#, r#"쟭"#, r#"쟭"#),
        (r#"쟮"#, r#"쟮"#, r#"쟮"#, r#"쟮"#, r#"쟮"#),
        (r#"쟯"#, r#"쟯"#, r#"쟯"#, r#"쟯"#, r#"쟯"#),
        (r#"쟰"#, r#"쟰"#, r#"쟰"#, r#"쟰"#, r#"쟰"#),
        (r#"쟱"#, r#"쟱"#, r#"쟱"#, r#"쟱"#, r#"쟱"#),
        (r#"쟲"#, r#"쟲"#, r#"쟲"#, r#"쟲"#, r#"쟲"#),
        (r#"쟳"#, r#"쟳"#, r#"쟳"#, r#"쟳"#, r#"쟳"#),
        (r#"쟴"#, r#"쟴"#, r#"쟴"#, r#"쟴"#, r#"쟴"#),
        (r#"쟵"#, r#"쟵"#, r#"쟵"#, r#"쟵"#, r#"쟵"#),
        (r#"쟶"#, r#"쟶"#, r#"쟶"#, r#"쟶"#, r#"쟶"#),
        (r#"쟷"#, r#"쟷"#, r#"쟷"#, r#"쟷"#, r#"쟷"#),
        (r#"쟸"#, r#"쟸"#, r#"쟸"#, r#"쟸"#, r#"쟸"#),
        (r#"쟹"#, r#"쟹"#, r#"쟹"#, r#"쟹"#, r#"쟹"#),
        (r#"쟺"#, r#"쟺"#, r#"쟺"#, r#"쟺"#, r#"쟺"#),
        (r#"쟻"#, r#"쟻"#, r#"쟻"#, r#"쟻"#, r#"쟻"#),
        (r#"쟼"#, r#"쟼"#, r#"쟼"#, r#"쟼"#, r#"쟼"#),
        (r#"쟽"#, r#"쟽"#, r#"쟽"#, r#"쟽"#, r#"쟽"#),
        (r#"쟾"#, r#"쟾"#, r#"쟾"#, r#"쟾"#, r#"쟾"#),
        (r#"쟿"#, r#"쟿"#, r#"쟿"#, r#"쟿"#, r#"쟿"#),
        (r#"저"#, r#"저"#, r#"저"#, r#"저"#, r#"저"#),
        (r#"적"#, r#"적"#, r#"적"#, r#"적"#, r#"적"#),
        (r#"젂"#, r#"젂"#, r#"젂"#, r#"젂"#, r#"젂"#),
        (r#"젃"#, r#"젃"#, r#"젃"#, r#"젃"#, r#"젃"#),
        (r#"전"#, r#"전"#, r#"전"#, r#"전"#, r#"전"#),
        (r#"젅"#, r#"젅"#, r#"젅"#, r#"젅"#, r#"젅"#),
        (r#"젆"#, r#"젆"#, r#"젆"#, r#"젆"#, r#"젆"#),
        (r#"젇"#, r#"젇"#, r#"젇"#, r#"젇"#, r#"젇"#),
        (r#"절"#, r#"절"#, r#"절"#, r#"절"#, r#"절"#),
        (r#"젉"#, r#"젉"#, r#"젉"#, r#"젉"#, r#"젉"#),
        (r#"젊"#, r#"젊"#, r#"젊"#, r#"젊"#, r#"젊"#),
        (r#"젋"#, r#"젋"#, r#"젋"#, r#"젋"#, r#"젋"#),
        (r#"젌"#, r#"젌"#, r#"젌"#, r#"젌"#, r#"젌"#),
        (r#"젍"#, r#"젍"#, r#"젍"#, r#"젍"#, r#"젍"#),
        (r#"젎"#, r#"젎"#, r#"젎"#, r#"젎"#, r#"젎"#),
        (r#"젏"#, r#"젏"#, r#"젏"#, r#"젏"#, r#"젏"#),
        (r#"점"#, r#"점"#, r#"점"#, r#"점"#, r#"점"#),
        (r#"접"#, r#"접"#, r#"접"#, r#"접"#, r#"접"#),
        (r#"젒"#, r#"젒"#, r#"젒"#, r#"젒"#, r#"젒"#),
        (r#"젓"#, r#"젓"#, r#"젓"#, r#"젓"#, r#"젓"#),
        (r#"젔"#, r#"젔"#, r#"젔"#, r#"젔"#, r#"젔"#),
        (r#"정"#, r#"정"#, r#"정"#, r#"정"#, r#"정"#),
        (r#"젖"#, r#"젖"#, r#"젖"#, r#"젖"#, r#"젖"#),
        (r#"젗"#, r#"젗"#, r#"젗"#, r#"젗"#, r#"젗"#),
        (r#"젘"#, r#"젘"#, r#"젘"#, r#"젘"#, r#"젘"#),
        (r#"젙"#, r#"젙"#, r#"젙"#, r#"젙"#, r#"젙"#),
        (r#"젚"#, r#"젚"#, r#"젚"#, r#"젚"#, r#"젚"#),
        (r#"젛"#, r#"젛"#, r#"젛"#, r#"젛"#, r#"젛"#),
        (r#"제"#, r#"제"#, r#"제"#, r#"제"#, r#"제"#),
        (r#"젝"#, r#"젝"#, r#"젝"#, r#"젝"#, r#"젝"#),
        (r#"젞"#, r#"젞"#, r#"젞"#, r#"젞"#, r#"젞"#),
        (r#"젟"#, r#"젟"#, r#"젟"#, r#"젟"#, r#"젟"#),
        (r#"젠"#, r#"젠"#, r#"젠"#, r#"젠"#, r#"젠"#),
        (r#"젡"#, r#"젡"#, r#"젡"#, r#"젡"#, r#"젡"#),
        (r#"젢"#, r#"젢"#, r#"젢"#, r#"젢"#, r#"젢"#),
        (r#"젣"#, r#"젣"#, r#"젣"#, r#"젣"#, r#"젣"#),
        (r#"젤"#, r#"젤"#, r#"젤"#, r#"젤"#, r#"젤"#),
        (r#"젥"#, r#"젥"#, r#"젥"#, r#"젥"#, r#"젥"#),
        (r#"젦"#, r#"젦"#, r#"젦"#, r#"젦"#, r#"젦"#),
        (r#"젧"#, r#"젧"#, r#"젧"#, r#"젧"#, r#"젧"#),
        (r#"젨"#, r#"젨"#, r#"젨"#, r#"젨"#, r#"젨"#),
        (r#"젩"#, r#"젩"#, r#"젩"#, r#"젩"#, r#"젩"#),
        (r#"젪"#, r#"젪"#, r#"젪"#, r#"젪"#, r#"젪"#),
        (r#"젫"#, r#"젫"#, r#"젫"#, r#"젫"#, r#"젫"#),
        (r#"젬"#, r#"젬"#, r#"젬"#, r#"젬"#, r#"젬"#),
        (r#"젭"#, r#"젭"#, r#"젭"#, r#"젭"#, r#"젭"#),
        (r#"젮"#, r#"젮"#, r#"젮"#, r#"젮"#, r#"젮"#),
        (r#"젯"#, r#"젯"#, r#"젯"#, r#"젯"#, r#"젯"#),
        (r#"젰"#, r#"젰"#, r#"젰"#, r#"젰"#, r#"젰"#),
        (r#"젱"#, r#"젱"#, r#"젱"#, r#"젱"#, r#"젱"#),
        (r#"젲"#, r#"젲"#, r#"젲"#, r#"젲"#, r#"젲"#),
        (r#"젳"#, r#"젳"#, r#"젳"#, r#"젳"#, r#"젳"#),
        (r#"젴"#, r#"젴"#, r#"젴"#, r#"젴"#, r#"젴"#),
        (r#"젵"#, r#"젵"#, r#"젵"#, r#"젵"#, r#"젵"#),
        (r#"젶"#, r#"젶"#, r#"젶"#, r#"젶"#, r#"젶"#),
        (r#"젷"#, r#"젷"#, r#"젷"#, r#"젷"#, r#"젷"#),
        (r#"져"#, r#"져"#, r#"져"#, r#"져"#, r#"져"#),
        (r#"젹"#, r#"젹"#, r#"젹"#, r#"젹"#, r#"젹"#),
        (r#"젺"#, r#"젺"#, r#"젺"#, r#"젺"#, r#"젺"#),
        (r#"젻"#, r#"젻"#, r#"젻"#, r#"젻"#, r#"젻"#),
        (r#"젼"#, r#"젼"#, r#"젼"#, r#"젼"#, r#"젼"#),
        (r#"젽"#, r#"젽"#, r#"젽"#, r#"젽"#, r#"젽"#),
        (r#"젾"#, r#"젾"#, r#"젾"#, r#"젾"#, r#"젾"#),
        (r#"젿"#, r#"젿"#, r#"젿"#, r#"젿"#, r#"젿"#),
        (r#"졀"#, r#"졀"#, r#"졀"#, r#"졀"#, r#"졀"#),
        (r#"졁"#, r#"졁"#, r#"졁"#, r#"졁"#, r#"졁"#),
        (r#"졂"#, r#"졂"#, r#"졂"#, r#"졂"#, r#"졂"#),
        (r#"졃"#, r#"졃"#, r#"졃"#, r#"졃"#, r#"졃"#),
        (r#"졄"#, r#"졄"#, r#"졄"#, r#"졄"#, r#"졄"#),
        (r#"졅"#, r#"졅"#, r#"졅"#, r#"졅"#, r#"졅"#),
        (r#"졆"#, r#"졆"#, r#"졆"#, r#"졆"#, r#"졆"#),
        (r#"졇"#, r#"졇"#, r#"졇"#, r#"졇"#, r#"졇"#),
        (r#"졈"#, r#"졈"#, r#"졈"#, r#"졈"#, r#"졈"#),
        (r#"졉"#, r#"졉"#, r#"졉"#, r#"졉"#, r#"졉"#),
        (r#"졊"#, r#"졊"#, r#"졊"#, r#"졊"#, r#"졊"#),
        (r#"졋"#, r#"졋"#, r#"졋"#, r#"졋"#, r#"졋"#),
        (r#"졌"#, r#"졌"#, r#"졌"#, r#"졌"#, r#"졌"#),
        (r#"졍"#, r#"졍"#, r#"졍"#, r#"졍"#, r#"졍"#),
        (r#"졎"#, r#"졎"#, r#"졎"#, r#"졎"#, r#"졎"#),
        (r#"졏"#, r#"졏"#, r#"졏"#, r#"졏"#, r#"졏"#),
        (r#"졐"#, r#"졐"#, r#"졐"#, r#"졐"#, r#"졐"#),
        (r#"졑"#, r#"졑"#, r#"졑"#, r#"졑"#, r#"졑"#),
        (r#"졒"#, r#"졒"#, r#"졒"#, r#"졒"#, r#"졒"#),
        (r#"졓"#, r#"졓"#, r#"졓"#, r#"졓"#, r#"졓"#),
        (r#"졔"#, r#"졔"#, r#"졔"#, r#"졔"#, r#"졔"#),
        (r#"졕"#, r#"졕"#, r#"졕"#, r#"졕"#, r#"졕"#),
        (r#"졖"#, r#"졖"#, r#"졖"#, r#"졖"#, r#"졖"#),
        (r#"졗"#, r#"졗"#, r#"졗"#, r#"졗"#, r#"졗"#),
        (r#"졘"#, r#"졘"#, r#"졘"#, r#"졘"#, r#"졘"#),
        (r#"졙"#, r#"졙"#, r#"졙"#, r#"졙"#, r#"졙"#),
        (r#"졚"#, r#"졚"#, r#"졚"#, r#"졚"#, r#"졚"#),
        (r#"졛"#, r#"졛"#, r#"졛"#, r#"졛"#, r#"졛"#),
        (r#"졜"#, r#"졜"#, r#"졜"#, r#"졜"#, r#"졜"#),
        (r#"졝"#, r#"졝"#, r#"졝"#, r#"졝"#, r#"졝"#),
        (r#"졞"#, r#"졞"#, r#"졞"#, r#"졞"#, r#"졞"#),
        (r#"졟"#, r#"졟"#, r#"졟"#, r#"졟"#, r#"졟"#),
        (r#"졠"#, r#"졠"#, r#"졠"#, r#"졠"#, r#"졠"#),
        (r#"졡"#, r#"졡"#, r#"졡"#, r#"졡"#, r#"졡"#),
        (r#"졢"#, r#"졢"#, r#"졢"#, r#"졢"#, r#"졢"#),
        (r#"졣"#, r#"졣"#, r#"졣"#, r#"졣"#, r#"졣"#),
        (r#"졤"#, r#"졤"#, r#"졤"#, r#"졤"#, r#"졤"#),
        (r#"졥"#, r#"졥"#, r#"졥"#, r#"졥"#, r#"졥"#),
        (r#"졦"#, r#"졦"#, r#"졦"#, r#"졦"#, r#"졦"#),
        (r#"졧"#, r#"졧"#, r#"졧"#, r#"졧"#, r#"졧"#),
        (r#"졨"#, r#"졨"#, r#"졨"#, r#"졨"#, r#"졨"#),
        (r#"졩"#, r#"졩"#, r#"졩"#, r#"졩"#, r#"졩"#),
        (r#"졪"#, r#"졪"#, r#"졪"#, r#"졪"#, r#"졪"#),
        (r#"졫"#, r#"졫"#, r#"졫"#, r#"졫"#, r#"졫"#),
        (r#"졬"#, r#"졬"#, r#"졬"#, r#"졬"#, r#"졬"#),
        (r#"졭"#, r#"졭"#, r#"졭"#, r#"졭"#, r#"졭"#),
        (r#"졮"#, r#"졮"#, r#"졮"#, r#"졮"#, r#"졮"#),
        (r#"졯"#, r#"졯"#, r#"졯"#, r#"졯"#, r#"졯"#),
        (r#"조"#, r#"조"#, r#"조"#, r#"조"#, r#"조"#),
        (r#"족"#, r#"족"#, r#"족"#, r#"족"#, r#"족"#),
        (r#"졲"#, r#"졲"#, r#"졲"#, r#"졲"#, r#"졲"#),
        (r#"졳"#, r#"졳"#, r#"졳"#, r#"졳"#, r#"졳"#),
        (r#"존"#, r#"존"#, r#"존"#, r#"존"#, r#"존"#),
        (r#"졵"#, r#"졵"#, r#"졵"#, r#"졵"#, r#"졵"#),
        (r#"졶"#, r#"졶"#, r#"졶"#, r#"졶"#, r#"졶"#),
        (r#"졷"#, r#"졷"#, r#"졷"#, r#"졷"#, r#"졷"#),
        (r#"졸"#, r#"졸"#, r#"졸"#, r#"졸"#, r#"졸"#),
        (r#"졹"#, r#"졹"#, r#"졹"#, r#"졹"#, r#"졹"#),
        (r#"졺"#, r#"졺"#, r#"졺"#, r#"졺"#, r#"졺"#),
        (r#"졻"#, r#"졻"#, r#"졻"#, r#"졻"#, r#"졻"#),
        (r#"졼"#, r#"졼"#, r#"졼"#, r#"졼"#, r#"졼"#),
        (r#"졽"#, r#"졽"#, r#"졽"#, r#"졽"#, r#"졽"#),
        (r#"졾"#, r#"졾"#, r#"졾"#, r#"졾"#, r#"졾"#),
        (r#"졿"#, r#"졿"#, r#"졿"#, r#"졿"#, r#"졿"#),
        (r#"좀"#, r#"좀"#, r#"좀"#, r#"좀"#, r#"좀"#),
        (r#"좁"#, r#"좁"#, r#"좁"#, r#"좁"#, r#"좁"#),
        (r#"좂"#, r#"좂"#, r#"좂"#, r#"좂"#, r#"좂"#),
        (r#"좃"#, r#"좃"#, r#"좃"#, r#"좃"#, r#"좃"#),
        (r#"좄"#, r#"좄"#, r#"좄"#, r#"좄"#, r#"좄"#),
        (r#"종"#, r#"종"#, r#"종"#, r#"종"#, r#"종"#),
        (r#"좆"#, r#"좆"#, r#"좆"#, r#"좆"#, r#"좆"#),
        (r#"좇"#, r#"좇"#, r#"좇"#, r#"좇"#, r#"좇"#),
        (r#"좈"#, r#"좈"#, r#"좈"#, r#"좈"#, r#"좈"#),
        (r#"좉"#, r#"좉"#, r#"좉"#, r#"좉"#, r#"좉"#),
        (r#"좊"#, r#"좊"#, r#"좊"#, r#"좊"#, r#"좊"#),
        (r#"좋"#, r#"좋"#, r#"좋"#, r#"좋"#, r#"좋"#),
        (r#"좌"#, r#"좌"#, r#"좌"#, r#"좌"#, r#"좌"#),
        (r#"좍"#, r#"좍"#, r#"좍"#, r#"좍"#, r#"좍"#),
        (r#"좎"#, r#"좎"#, r#"좎"#, r#"좎"#, r#"좎"#),
        (r#"좏"#, r#"좏"#, r#"좏"#, r#"좏"#, r#"좏"#),
        (r#"좐"#, r#"좐"#, r#"좐"#, r#"좐"#, r#"좐"#),
        (r#"좑"#, r#"좑"#, r#"좑"#, r#"좑"#, r#"좑"#),
        (r#"좒"#, r#"좒"#, r#"좒"#, r#"좒"#, r#"좒"#),
        (r#"좓"#, r#"좓"#, r#"좓"#, r#"좓"#, r#"좓"#),
        (r#"좔"#, r#"좔"#, r#"좔"#, r#"좔"#, r#"좔"#),
        (r#"좕"#, r#"좕"#, r#"좕"#, r#"좕"#, r#"좕"#),
        (r#"좖"#, r#"좖"#, r#"좖"#, r#"좖"#, r#"좖"#),
        (r#"좗"#, r#"좗"#, r#"좗"#, r#"좗"#, r#"좗"#),
        (r#"좘"#, r#"좘"#, r#"좘"#, r#"좘"#, r#"좘"#),
        (r#"좙"#, r#"좙"#, r#"좙"#, r#"좙"#, r#"좙"#),
        (r#"좚"#, r#"좚"#, r#"좚"#, r#"좚"#, r#"좚"#),
        (r#"좛"#, r#"좛"#, r#"좛"#, r#"좛"#, r#"좛"#),
        (r#"좜"#, r#"좜"#, r#"좜"#, r#"좜"#, r#"좜"#),
        (r#"좝"#, r#"좝"#, r#"좝"#, r#"좝"#, r#"좝"#),
        (r#"좞"#, r#"좞"#, r#"좞"#, r#"좞"#, r#"좞"#),
        (r#"좟"#, r#"좟"#, r#"좟"#, r#"좟"#, r#"좟"#),
        (r#"좠"#, r#"좠"#, r#"좠"#, r#"좠"#, r#"좠"#),
        (r#"좡"#, r#"좡"#, r#"좡"#, r#"좡"#, r#"좡"#),
        (r#"좢"#, r#"좢"#, r#"좢"#, r#"좢"#, r#"좢"#),
        (r#"좣"#, r#"좣"#, r#"좣"#, r#"좣"#, r#"좣"#),
        (r#"좤"#, r#"좤"#, r#"좤"#, r#"좤"#, r#"좤"#),
        (r#"좥"#, r#"좥"#, r#"좥"#, r#"좥"#, r#"좥"#),
        (r#"좦"#, r#"좦"#, r#"좦"#, r#"좦"#, r#"좦"#),
        (r#"좧"#, r#"좧"#, r#"좧"#, r#"좧"#, r#"좧"#),
        (r#"좨"#, r#"좨"#, r#"좨"#, r#"좨"#, r#"좨"#),
        (r#"좩"#, r#"좩"#, r#"좩"#, r#"좩"#, r#"좩"#),
        (r#"좪"#, r#"좪"#, r#"좪"#, r#"좪"#, r#"좪"#),
        (r#"좫"#, r#"좫"#, r#"좫"#, r#"좫"#, r#"좫"#),
        (r#"좬"#, r#"좬"#, r#"좬"#, r#"좬"#, r#"좬"#),
        (r#"좭"#, r#"좭"#, r#"좭"#, r#"좭"#, r#"좭"#),
        (r#"좮"#, r#"좮"#, r#"좮"#, r#"좮"#, r#"좮"#),
        (r#"좯"#, r#"좯"#, r#"좯"#, r#"좯"#, r#"좯"#),
        (r#"좰"#, r#"좰"#, r#"좰"#, r#"좰"#, r#"좰"#),
        (r#"좱"#, r#"좱"#, r#"좱"#, r#"좱"#, r#"좱"#),
        (r#"좲"#, r#"좲"#, r#"좲"#, r#"좲"#, r#"좲"#),
        (r#"좳"#, r#"좳"#, r#"좳"#, r#"좳"#, r#"좳"#),
        (r#"좴"#, r#"좴"#, r#"좴"#, r#"좴"#, r#"좴"#),
        (r#"좵"#, r#"좵"#, r#"좵"#, r#"좵"#, r#"좵"#),
        (r#"좶"#, r#"좶"#, r#"좶"#, r#"좶"#, r#"좶"#),
        (r#"좷"#, r#"좷"#, r#"좷"#, r#"좷"#, r#"좷"#),
        (r#"좸"#, r#"좸"#, r#"좸"#, r#"좸"#, r#"좸"#),
        (r#"좹"#, r#"좹"#, r#"좹"#, r#"좹"#, r#"좹"#),
        (r#"좺"#, r#"좺"#, r#"좺"#, r#"좺"#, r#"좺"#),
        (r#"좻"#, r#"좻"#, r#"좻"#, r#"좻"#, r#"좻"#),
        (r#"좼"#, r#"좼"#, r#"좼"#, r#"좼"#, r#"좼"#),
        (r#"좽"#, r#"좽"#, r#"좽"#, r#"좽"#, r#"좽"#),
        (r#"좾"#, r#"좾"#, r#"좾"#, r#"좾"#, r#"좾"#),
        (r#"좿"#, r#"좿"#, r#"좿"#, r#"좿"#, r#"좿"#),
        (r#"죀"#, r#"죀"#, r#"죀"#, r#"죀"#, r#"죀"#),
        (r#"죁"#, r#"죁"#, r#"죁"#, r#"죁"#, r#"죁"#),
        (r#"죂"#, r#"죂"#, r#"죂"#, r#"죂"#, r#"죂"#),
        (r#"죃"#, r#"죃"#, r#"죃"#, r#"죃"#, r#"죃"#),
        (r#"죄"#, r#"죄"#, r#"죄"#, r#"죄"#, r#"죄"#),
        (r#"죅"#, r#"죅"#, r#"죅"#, r#"죅"#, r#"죅"#),
        (r#"죆"#, r#"죆"#, r#"죆"#, r#"죆"#, r#"죆"#),
        (r#"죇"#, r#"죇"#, r#"죇"#, r#"죇"#, r#"죇"#),
        (r#"죈"#, r#"죈"#, r#"죈"#, r#"죈"#, r#"죈"#),
        (r#"죉"#, r#"죉"#, r#"죉"#, r#"죉"#, r#"죉"#),
        (r#"죊"#, r#"죊"#, r#"죊"#, r#"죊"#, r#"죊"#),
        (r#"죋"#, r#"죋"#, r#"죋"#, r#"죋"#, r#"죋"#),
        (r#"죌"#, r#"죌"#, r#"죌"#, r#"죌"#, r#"죌"#),
        (r#"죍"#, r#"죍"#, r#"죍"#, r#"죍"#, r#"죍"#),
        (r#"죎"#, r#"죎"#, r#"죎"#, r#"죎"#, r#"죎"#),
        (r#"죏"#, r#"죏"#, r#"죏"#, r#"죏"#, r#"죏"#),
        (r#"죐"#, r#"죐"#, r#"죐"#, r#"죐"#, r#"죐"#),
        (r#"죑"#, r#"죑"#, r#"죑"#, r#"죑"#, r#"죑"#),
        (r#"죒"#, r#"죒"#, r#"죒"#, r#"죒"#, r#"죒"#),
        (r#"죓"#, r#"죓"#, r#"죓"#, r#"죓"#, r#"죓"#),
        (r#"죔"#, r#"죔"#, r#"죔"#, r#"죔"#, r#"죔"#),
        (r#"죕"#, r#"죕"#, r#"죕"#, r#"죕"#, r#"죕"#),
        (r#"죖"#, r#"죖"#, r#"죖"#, r#"죖"#, r#"죖"#),
        (r#"죗"#, r#"죗"#, r#"죗"#, r#"죗"#, r#"죗"#),
        (r#"죘"#, r#"죘"#, r#"죘"#, r#"죘"#, r#"죘"#),
        (r#"죙"#, r#"죙"#, r#"죙"#, r#"죙"#, r#"죙"#),
        (r#"죚"#, r#"죚"#, r#"죚"#, r#"죚"#, r#"죚"#),
        (r#"죛"#, r#"죛"#, r#"죛"#, r#"죛"#, r#"죛"#),
        (r#"죜"#, r#"죜"#, r#"죜"#, r#"죜"#, r#"죜"#),
        (r#"죝"#, r#"죝"#, r#"죝"#, r#"죝"#, r#"죝"#),
        (r#"죞"#, r#"죞"#, r#"죞"#, r#"죞"#, r#"죞"#),
        (r#"죟"#, r#"죟"#, r#"죟"#, r#"죟"#, r#"죟"#),
        (r#"죠"#, r#"죠"#, r#"죠"#, r#"죠"#, r#"죠"#),
        (r#"죡"#, r#"죡"#, r#"죡"#, r#"죡"#, r#"죡"#),
        (r#"죢"#, r#"죢"#, r#"죢"#, r#"죢"#, r#"죢"#),
        (r#"죣"#, r#"죣"#, r#"죣"#, r#"죣"#, r#"죣"#),
        (r#"죤"#, r#"죤"#, r#"죤"#, r#"죤"#, r#"죤"#),
        (r#"죥"#, r#"죥"#, r#"죥"#, r#"죥"#, r#"죥"#),
        (r#"죦"#, r#"죦"#, r#"죦"#, r#"죦"#, r#"죦"#),
        (r#"죧"#, r#"죧"#, r#"죧"#, r#"죧"#, r#"죧"#),
        (r#"죨"#, r#"죨"#, r#"죨"#, r#"죨"#, r#"죨"#),
        (r#"죩"#, r#"죩"#, r#"죩"#, r#"죩"#, r#"죩"#),
        (r#"죪"#, r#"죪"#, r#"죪"#, r#"죪"#, r#"죪"#),
        (r#"죫"#, r#"죫"#, r#"죫"#, r#"죫"#, r#"죫"#),
        (r#"죬"#, r#"죬"#, r#"죬"#, r#"죬"#, r#"죬"#),
        (r#"죭"#, r#"죭"#, r#"죭"#, r#"죭"#, r#"죭"#),
        (r#"죮"#, r#"죮"#, r#"죮"#, r#"죮"#, r#"죮"#),
        (r#"죯"#, r#"죯"#, r#"죯"#, r#"죯"#, r#"죯"#),
        (r#"죰"#, r#"죰"#, r#"죰"#, r#"죰"#, r#"죰"#),
        (r#"죱"#, r#"죱"#, r#"죱"#, r#"죱"#, r#"죱"#),
        (r#"죲"#, r#"죲"#, r#"죲"#, r#"죲"#, r#"죲"#),
        (r#"죳"#, r#"죳"#, r#"죳"#, r#"죳"#, r#"죳"#),
        (r#"죴"#, r#"죴"#, r#"죴"#, r#"죴"#, r#"죴"#),
        (r#"죵"#, r#"죵"#, r#"죵"#, r#"죵"#, r#"죵"#),
        (r#"죶"#, r#"죶"#, r#"죶"#, r#"죶"#, r#"죶"#),
        (r#"죷"#, r#"죷"#, r#"죷"#, r#"죷"#, r#"죷"#),
        (r#"죸"#, r#"죸"#, r#"죸"#, r#"죸"#, r#"죸"#),
        (r#"죹"#, r#"죹"#, r#"죹"#, r#"죹"#, r#"죹"#),
        (r#"죺"#, r#"죺"#, r#"죺"#, r#"죺"#, r#"죺"#),
        (r#"죻"#, r#"죻"#, r#"죻"#, r#"죻"#, r#"죻"#),
        (r#"주"#, r#"주"#, r#"주"#, r#"주"#, r#"주"#),
        (r#"죽"#, r#"죽"#, r#"죽"#, r#"죽"#, r#"죽"#),
        (r#"죾"#, r#"죾"#, r#"죾"#, r#"죾"#, r#"죾"#),
        (r#"죿"#, r#"죿"#, r#"죿"#, r#"죿"#, r#"죿"#),
        (r#"준"#, r#"준"#, r#"준"#, r#"준"#, r#"준"#),
        (r#"줁"#, r#"줁"#, r#"줁"#, r#"줁"#, r#"줁"#),
        (r#"줂"#, r#"줂"#, r#"줂"#, r#"줂"#, r#"줂"#),
        (r#"줃"#, r#"줃"#, r#"줃"#, r#"줃"#, r#"줃"#),
        (r#"줄"#, r#"줄"#, r#"줄"#, r#"줄"#, r#"줄"#),
        (r#"줅"#, r#"줅"#, r#"줅"#, r#"줅"#, r#"줅"#),
        (r#"줆"#, r#"줆"#, r#"줆"#, r#"줆"#, r#"줆"#),
        (r#"줇"#, r#"줇"#, r#"줇"#, r#"줇"#, r#"줇"#),
        (r#"줈"#, r#"줈"#, r#"줈"#, r#"줈"#, r#"줈"#),
        (r#"줉"#, r#"줉"#, r#"줉"#, r#"줉"#, r#"줉"#),
        (r#"줊"#, r#"줊"#, r#"줊"#, r#"줊"#, r#"줊"#),
        (r#"줋"#, r#"줋"#, r#"줋"#, r#"줋"#, r#"줋"#),
        (r#"줌"#, r#"줌"#, r#"줌"#, r#"줌"#, r#"줌"#),
        (r#"줍"#, r#"줍"#, r#"줍"#, r#"줍"#, r#"줍"#),
        (r#"줎"#, r#"줎"#, r#"줎"#, r#"줎"#, r#"줎"#),
        (r#"줏"#, r#"줏"#, r#"줏"#, r#"줏"#, r#"줏"#),
        (r#"줐"#, r#"줐"#, r#"줐"#, r#"줐"#, r#"줐"#),
        (r#"중"#, r#"중"#, r#"중"#, r#"중"#, r#"중"#),
        (r#"줒"#, r#"줒"#, r#"줒"#, r#"줒"#, r#"줒"#),
        (r#"줓"#, r#"줓"#, r#"줓"#, r#"줓"#, r#"줓"#),
        (r#"줔"#, r#"줔"#, r#"줔"#, r#"줔"#, r#"줔"#),
        (r#"줕"#, r#"줕"#, r#"줕"#, r#"줕"#, r#"줕"#),
        (r#"줖"#, r#"줖"#, r#"줖"#, r#"줖"#, r#"줖"#),
        (r#"줗"#, r#"줗"#, r#"줗"#, r#"줗"#, r#"줗"#),
        (r#"줘"#, r#"줘"#, r#"줘"#, r#"줘"#, r#"줘"#),
        (r#"줙"#, r#"줙"#, r#"줙"#, r#"줙"#, r#"줙"#),
        (r#"줚"#, r#"줚"#, r#"줚"#, r#"줚"#, r#"줚"#),
        (r#"줛"#, r#"줛"#, r#"줛"#, r#"줛"#, r#"줛"#),
        (r#"줜"#, r#"줜"#, r#"줜"#, r#"줜"#, r#"줜"#),
        (r#"줝"#, r#"줝"#, r#"줝"#, r#"줝"#, r#"줝"#),
        (r#"줞"#, r#"줞"#, r#"줞"#, r#"줞"#, r#"줞"#),
        (r#"줟"#, r#"줟"#, r#"줟"#, r#"줟"#, r#"줟"#),
        (r#"줠"#, r#"줠"#, r#"줠"#, r#"줠"#, r#"줠"#),
        (r#"줡"#, r#"줡"#, r#"줡"#, r#"줡"#, r#"줡"#),
        (r#"줢"#, r#"줢"#, r#"줢"#, r#"줢"#, r#"줢"#),
        (r#"줣"#, r#"줣"#, r#"줣"#, r#"줣"#, r#"줣"#),
        (r#"줤"#, r#"줤"#, r#"줤"#, r#"줤"#, r#"줤"#),
        (r#"줥"#, r#"줥"#, r#"줥"#, r#"줥"#, r#"줥"#),
        (r#"줦"#, r#"줦"#, r#"줦"#, r#"줦"#, r#"줦"#),
        (r#"줧"#, r#"줧"#, r#"줧"#, r#"줧"#, r#"줧"#),
        (r#"줨"#, r#"줨"#, r#"줨"#, r#"줨"#, r#"줨"#),
        (r#"줩"#, r#"줩"#, r#"줩"#, r#"줩"#, r#"줩"#),
        (r#"줪"#, r#"줪"#, r#"줪"#, r#"줪"#, r#"줪"#),
        (r#"줫"#, r#"줫"#, r#"줫"#, r#"줫"#, r#"줫"#),
        (r#"줬"#, r#"줬"#, r#"줬"#, r#"줬"#, r#"줬"#),
        (r#"줭"#, r#"줭"#, r#"줭"#, r#"줭"#, r#"줭"#),
        (r#"줮"#, r#"줮"#, r#"줮"#, r#"줮"#, r#"줮"#),
        (r#"줯"#, r#"줯"#, r#"줯"#, r#"줯"#, r#"줯"#),
        (r#"줰"#, r#"줰"#, r#"줰"#, r#"줰"#, r#"줰"#),
        (r#"줱"#, r#"줱"#, r#"줱"#, r#"줱"#, r#"줱"#),
        (r#"줲"#, r#"줲"#, r#"줲"#, r#"줲"#, r#"줲"#),
        (r#"줳"#, r#"줳"#, r#"줳"#, r#"줳"#, r#"줳"#),
        (r#"줴"#, r#"줴"#, r#"줴"#, r#"줴"#, r#"줴"#),
        (r#"줵"#, r#"줵"#, r#"줵"#, r#"줵"#, r#"줵"#),
        (r#"줶"#, r#"줶"#, r#"줶"#, r#"줶"#, r#"줶"#),
        (r#"줷"#, r#"줷"#, r#"줷"#, r#"줷"#, r#"줷"#),
        (r#"줸"#, r#"줸"#, r#"줸"#, r#"줸"#, r#"줸"#),
        (r#"줹"#, r#"줹"#, r#"줹"#, r#"줹"#, r#"줹"#),
        (r#"줺"#, r#"줺"#, r#"줺"#, r#"줺"#, r#"줺"#),
        (r#"줻"#, r#"줻"#, r#"줻"#, r#"줻"#, r#"줻"#),
        (r#"줼"#, r#"줼"#, r#"줼"#, r#"줼"#, r#"줼"#),
        (r#"줽"#, r#"줽"#, r#"줽"#, r#"줽"#, r#"줽"#),
        (r#"줾"#, r#"줾"#, r#"줾"#, r#"줾"#, r#"줾"#),
        (r#"줿"#, r#"줿"#, r#"줿"#, r#"줿"#, r#"줿"#),
        (r#"쥀"#, r#"쥀"#, r#"쥀"#, r#"쥀"#, r#"쥀"#),
        (r#"쥁"#, r#"쥁"#, r#"쥁"#, r#"쥁"#, r#"쥁"#),
        (r#"쥂"#, r#"쥂"#, r#"쥂"#, r#"쥂"#, r#"쥂"#),
        (r#"쥃"#, r#"쥃"#, r#"쥃"#, r#"쥃"#, r#"쥃"#),
        (r#"쥄"#, r#"쥄"#, r#"쥄"#, r#"쥄"#, r#"쥄"#),
        (r#"쥅"#, r#"쥅"#, r#"쥅"#, r#"쥅"#, r#"쥅"#),
        (r#"쥆"#, r#"쥆"#, r#"쥆"#, r#"쥆"#, r#"쥆"#),
        (r#"쥇"#, r#"쥇"#, r#"쥇"#, r#"쥇"#, r#"쥇"#),
        (r#"쥈"#, r#"쥈"#, r#"쥈"#, r#"쥈"#, r#"쥈"#),
        (r#"쥉"#, r#"쥉"#, r#"쥉"#, r#"쥉"#, r#"쥉"#),
        (r#"쥊"#, r#"쥊"#, r#"쥊"#, r#"쥊"#, r#"쥊"#),
        (r#"쥋"#, r#"쥋"#, r#"쥋"#, r#"쥋"#, r#"쥋"#),
        (r#"쥌"#, r#"쥌"#, r#"쥌"#, r#"쥌"#, r#"쥌"#),
        (r#"쥍"#, r#"쥍"#, r#"쥍"#, r#"쥍"#, r#"쥍"#),
        (r#"쥎"#, r#"쥎"#, r#"쥎"#, r#"쥎"#, r#"쥎"#),
        (r#"쥏"#, r#"쥏"#, r#"쥏"#, r#"쥏"#, r#"쥏"#),
        (r#"쥐"#, r#"쥐"#, r#"쥐"#, r#"쥐"#, r#"쥐"#),
        (r#"쥑"#, r#"쥑"#, r#"쥑"#, r#"쥑"#, r#"쥑"#),
        (r#"쥒"#, r#"쥒"#, r#"쥒"#, r#"쥒"#, r#"쥒"#),
        (r#"쥓"#, r#"쥓"#, r#"쥓"#, r#"쥓"#, r#"쥓"#),
        (r#"쥔"#, r#"쥔"#, r#"쥔"#, r#"쥔"#, r#"쥔"#),
        (r#"쥕"#, r#"쥕"#, r#"쥕"#, r#"쥕"#, r#"쥕"#),
        (r#"쥖"#, r#"쥖"#, r#"쥖"#, r#"쥖"#, r#"쥖"#),
        (r#"쥗"#, r#"쥗"#, r#"쥗"#, r#"쥗"#, r#"쥗"#),
        (r#"쥘"#, r#"쥘"#, r#"쥘"#, r#"쥘"#, r#"쥘"#),
        (r#"쥙"#, r#"쥙"#, r#"쥙"#, r#"쥙"#, r#"쥙"#),
        (r#"쥚"#, r#"쥚"#, r#"쥚"#, r#"쥚"#, r#"쥚"#),
        (r#"쥛"#, r#"쥛"#, r#"쥛"#, r#"쥛"#, r#"쥛"#),
        (r#"쥜"#, r#"쥜"#, r#"쥜"#, r#"쥜"#, r#"쥜"#),
        (r#"쥝"#, r#"쥝"#, r#"쥝"#, r#"쥝"#, r#"쥝"#),
        (r#"쥞"#, r#"쥞"#, r#"쥞"#, r#"쥞"#, r#"쥞"#),
        (r#"쥟"#, r#"쥟"#, r#"쥟"#, r#"쥟"#, r#"쥟"#),
        (r#"쥠"#, r#"쥠"#, r#"쥠"#, r#"쥠"#, r#"쥠"#),
        (r#"쥡"#, r#"쥡"#, r#"쥡"#, r#"쥡"#, r#"쥡"#),
        (r#"쥢"#, r#"쥢"#, r#"쥢"#, r#"쥢"#, r#"쥢"#),
        (r#"쥣"#, r#"쥣"#, r#"쥣"#, r#"쥣"#, r#"쥣"#),
        (r#"쥤"#, r#"쥤"#, r#"쥤"#, r#"쥤"#, r#"쥤"#),
        (r#"쥥"#, r#"쥥"#, r#"쥥"#, r#"쥥"#, r#"쥥"#),
        (r#"쥦"#, r#"쥦"#, r#"쥦"#, r#"쥦"#, r#"쥦"#),
        (r#"쥧"#, r#"쥧"#, r#"쥧"#, r#"쥧"#, r#"쥧"#),
        (r#"쥨"#, r#"쥨"#, r#"쥨"#, r#"쥨"#, r#"쥨"#),
        (r#"쥩"#, r#"쥩"#, r#"쥩"#, r#"쥩"#, r#"쥩"#),
        (r#"쥪"#, r#"쥪"#, r#"쥪"#, r#"쥪"#, r#"쥪"#),
        (r#"쥫"#, r#"쥫"#, r#"쥫"#, r#"쥫"#, r#"쥫"#),
        (r#"쥬"#, r#"쥬"#, r#"쥬"#, r#"쥬"#, r#"쥬"#),
        (r#"쥭"#, r#"쥭"#, r#"쥭"#, r#"쥭"#, r#"쥭"#),
        (r#"쥮"#, r#"쥮"#, r#"쥮"#, r#"쥮"#, r#"쥮"#),
        (r#"쥯"#, r#"쥯"#, r#"쥯"#, r#"쥯"#, r#"쥯"#),
        (r#"쥰"#, r#"쥰"#, r#"쥰"#, r#"쥰"#, r#"쥰"#),
        (r#"쥱"#, r#"쥱"#, r#"쥱"#, r#"쥱"#, r#"쥱"#),
        (r#"쥲"#, r#"쥲"#, r#"쥲"#, r#"쥲"#, r#"쥲"#),
        (r#"쥳"#, r#"쥳"#, r#"쥳"#, r#"쥳"#, r#"쥳"#),
        (r#"쥴"#, r#"쥴"#, r#"쥴"#, r#"쥴"#, r#"쥴"#),
        (r#"쥵"#, r#"쥵"#, r#"쥵"#, r#"쥵"#, r#"쥵"#),
        (r#"쥶"#, r#"쥶"#, r#"쥶"#, r#"쥶"#, r#"쥶"#),
        (r#"쥷"#, r#"쥷"#, r#"쥷"#, r#"쥷"#, r#"쥷"#),
        (r#"쥸"#, r#"쥸"#, r#"쥸"#, r#"쥸"#, r#"쥸"#),
        (r#"쥹"#, r#"쥹"#, r#"쥹"#, r#"쥹"#, r#"쥹"#),
        (r#"쥺"#, r#"쥺"#, r#"쥺"#, r#"쥺"#, r#"쥺"#),
        (r#"쥻"#, r#"쥻"#, r#"쥻"#, r#"쥻"#, r#"쥻"#),
        (r#"쥼"#, r#"쥼"#, r#"쥼"#, r#"쥼"#, r#"쥼"#),
        (r#"쥽"#, r#"쥽"#, r#"쥽"#, r#"쥽"#, r#"쥽"#),
        (r#"쥾"#, r#"쥾"#, r#"쥾"#, r#"쥾"#, r#"쥾"#),
        (r#"쥿"#, r#"쥿"#, r#"쥿"#, r#"쥿"#, r#"쥿"#),
        (r#"즀"#, r#"즀"#, r#"즀"#, r#"즀"#, r#"즀"#),
        (r#"즁"#, r#"즁"#, r#"즁"#, r#"즁"#, r#"즁"#),
        (r#"즂"#, r#"즂"#, r#"즂"#, r#"즂"#, r#"즂"#),
        (r#"즃"#, r#"즃"#, r#"즃"#, r#"즃"#, r#"즃"#),
        (r#"즄"#, r#"즄"#, r#"즄"#, r#"즄"#, r#"즄"#),
        (r#"즅"#, r#"즅"#, r#"즅"#, r#"즅"#, r#"즅"#),
        (r#"즆"#, r#"즆"#, r#"즆"#, r#"즆"#, r#"즆"#),
        (r#"즇"#, r#"즇"#, r#"즇"#, r#"즇"#, r#"즇"#),
        (r#"즈"#, r#"즈"#, r#"즈"#, r#"즈"#, r#"즈"#),
        (r#"즉"#, r#"즉"#, r#"즉"#, r#"즉"#, r#"즉"#),
        (r#"즊"#, r#"즊"#, r#"즊"#, r#"즊"#, r#"즊"#),
        (r#"즋"#, r#"즋"#, r#"즋"#, r#"즋"#, r#"즋"#),
        (r#"즌"#, r#"즌"#, r#"즌"#, r#"즌"#, r#"즌"#),
        (r#"즍"#, r#"즍"#, r#"즍"#, r#"즍"#, r#"즍"#),
        (r#"즎"#, r#"즎"#, r#"즎"#, r#"즎"#, r#"즎"#),
        (r#"즏"#, r#"즏"#, r#"즏"#, r#"즏"#, r#"즏"#),
        (r#"즐"#, r#"즐"#, r#"즐"#, r#"즐"#, r#"즐"#),
        (r#"즑"#, r#"즑"#, r#"즑"#, r#"즑"#, r#"즑"#),
        (r#"즒"#, r#"즒"#, r#"즒"#, r#"즒"#, r#"즒"#),
        (r#"즓"#, r#"즓"#, r#"즓"#, r#"즓"#, r#"즓"#),
        (r#"즔"#, r#"즔"#, r#"즔"#, r#"즔"#, r#"즔"#),
        (r#"즕"#, r#"즕"#, r#"즕"#, r#"즕"#, r#"즕"#),
        (r#"즖"#, r#"즖"#, r#"즖"#, r#"즖"#, r#"즖"#),
        (r#"즗"#, r#"즗"#, r#"즗"#, r#"즗"#, r#"즗"#),
        (r#"즘"#, r#"즘"#, r#"즘"#, r#"즘"#, r#"즘"#),
        (r#"즙"#, r#"즙"#, r#"즙"#, r#"즙"#, r#"즙"#),
        (r#"즚"#, r#"즚"#, r#"즚"#, r#"즚"#, r#"즚"#),
        (r#"즛"#, r#"즛"#, r#"즛"#, r#"즛"#, r#"즛"#),
        (r#"즜"#, r#"즜"#, r#"즜"#, r#"즜"#, r#"즜"#),
        (r#"증"#, r#"증"#, r#"증"#, r#"증"#, r#"증"#),
        (r#"즞"#, r#"즞"#, r#"즞"#, r#"즞"#, r#"즞"#),
        (r#"즟"#, r#"즟"#, r#"즟"#, r#"즟"#, r#"즟"#),
        (r#"즠"#, r#"즠"#, r#"즠"#, r#"즠"#, r#"즠"#),
        (r#"즡"#, r#"즡"#, r#"즡"#, r#"즡"#, r#"즡"#),
        (r#"즢"#, r#"즢"#, r#"즢"#, r#"즢"#, r#"즢"#),
        (r#"즣"#, r#"즣"#, r#"즣"#, r#"즣"#, r#"즣"#),
        (r#"즤"#, r#"즤"#, r#"즤"#, r#"즤"#, r#"즤"#),
        (r#"즥"#, r#"즥"#, r#"즥"#, r#"즥"#, r#"즥"#),
        (r#"즦"#, r#"즦"#, r#"즦"#, r#"즦"#, r#"즦"#),
        (r#"즧"#, r#"즧"#, r#"즧"#, r#"즧"#, r#"즧"#),
        (r#"즨"#, r#"즨"#, r#"즨"#, r#"즨"#, r#"즨"#),
        (r#"즩"#, r#"즩"#, r#"즩"#, r#"즩"#, r#"즩"#),
        (r#"즪"#, r#"즪"#, r#"즪"#, r#"즪"#, r#"즪"#),
        (r#"즫"#, r#"즫"#, r#"즫"#, r#"즫"#, r#"즫"#),
        (r#"즬"#, r#"즬"#, r#"즬"#, r#"즬"#, r#"즬"#),
        (r#"즭"#, r#"즭"#, r#"즭"#, r#"즭"#, r#"즭"#),
        (r#"즮"#, r#"즮"#, r#"즮"#, r#"즮"#, r#"즮"#),
        (r#"즯"#, r#"즯"#, r#"즯"#, r#"즯"#, r#"즯"#),
        (r#"즰"#, r#"즰"#, r#"즰"#, r#"즰"#, r#"즰"#),
        (r#"즱"#, r#"즱"#, r#"즱"#, r#"즱"#, r#"즱"#),
        (r#"즲"#, r#"즲"#, r#"즲"#, r#"즲"#, r#"즲"#),
        (r#"즳"#, r#"즳"#, r#"즳"#, r#"즳"#, r#"즳"#),
        (r#"즴"#, r#"즴"#, r#"즴"#, r#"즴"#, r#"즴"#),
        (r#"즵"#, r#"즵"#, r#"즵"#, r#"즵"#, r#"즵"#),
        (r#"즶"#, r#"즶"#, r#"즶"#, r#"즶"#, r#"즶"#),
        (r#"즷"#, r#"즷"#, r#"즷"#, r#"즷"#, r#"즷"#),
        (r#"즸"#, r#"즸"#, r#"즸"#, r#"즸"#, r#"즸"#),
        (r#"즹"#, r#"즹"#, r#"즹"#, r#"즹"#, r#"즹"#),
        (r#"즺"#, r#"즺"#, r#"즺"#, r#"즺"#, r#"즺"#),
        (r#"즻"#, r#"즻"#, r#"즻"#, r#"즻"#, r#"즻"#),
        (r#"즼"#, r#"즼"#, r#"즼"#, r#"즼"#, r#"즼"#),
        (r#"즽"#, r#"즽"#, r#"즽"#, r#"즽"#, r#"즽"#),
        (r#"즾"#, r#"즾"#, r#"즾"#, r#"즾"#, r#"즾"#),
        (r#"즿"#, r#"즿"#, r#"즿"#, r#"즿"#, r#"즿"#),
        (r#"지"#, r#"지"#, r#"지"#, r#"지"#, r#"지"#),
        (r#"직"#, r#"직"#, r#"직"#, r#"직"#, r#"직"#),
        (r#"짂"#, r#"짂"#, r#"짂"#, r#"짂"#, r#"짂"#),
        (r#"짃"#, r#"짃"#, r#"짃"#, r#"짃"#, r#"짃"#),
        (r#"진"#, r#"진"#, r#"진"#, r#"진"#, r#"진"#),
        (r#"짅"#, r#"짅"#, r#"짅"#, r#"짅"#, r#"짅"#),
        (r#"짆"#, r#"짆"#, r#"짆"#, r#"짆"#, r#"짆"#),
        (r#"짇"#, r#"짇"#, r#"짇"#, r#"짇"#, r#"짇"#),
        (r#"질"#, r#"질"#, r#"질"#, r#"질"#, r#"질"#),
        (r#"짉"#, r#"짉"#, r#"짉"#, r#"짉"#, r#"짉"#),
        (r#"짊"#, r#"짊"#, r#"짊"#, r#"짊"#, r#"짊"#),
        (r#"짋"#, r#"짋"#, r#"짋"#, r#"짋"#, r#"짋"#),
        (r#"짌"#, r#"짌"#, r#"짌"#, r#"짌"#, r#"짌"#),
        (r#"짍"#, r#"짍"#, r#"짍"#, r#"짍"#, r#"짍"#),
        (r#"짎"#, r#"짎"#, r#"짎"#, r#"짎"#, r#"짎"#),
        (r#"짏"#, r#"짏"#, r#"짏"#, r#"짏"#, r#"짏"#),
        (r#"짐"#, r#"짐"#, r#"짐"#, r#"짐"#, r#"짐"#),
        (r#"집"#, r#"집"#, r#"집"#, r#"집"#, r#"집"#),
        (r#"짒"#, r#"짒"#, r#"짒"#, r#"짒"#, r#"짒"#),
        (r#"짓"#, r#"짓"#, r#"짓"#, r#"짓"#, r#"짓"#),
        (r#"짔"#, r#"짔"#, r#"짔"#, r#"짔"#, r#"짔"#),
        (r#"징"#, r#"징"#, r#"징"#, r#"징"#, r#"징"#),
        (r#"짖"#, r#"짖"#, r#"짖"#, r#"짖"#, r#"짖"#),
        (r#"짗"#, r#"짗"#, r#"짗"#, r#"짗"#, r#"짗"#),
        (r#"짘"#, r#"짘"#, r#"짘"#, r#"짘"#, r#"짘"#),
        (r#"짙"#, r#"짙"#, r#"짙"#, r#"짙"#, r#"짙"#),
        (r#"짚"#, r#"짚"#, r#"짚"#, r#"짚"#, r#"짚"#),
        (r#"짛"#, r#"짛"#, r#"짛"#, r#"짛"#, r#"짛"#),
        (r#"짜"#, r#"짜"#, r#"짜"#, r#"짜"#, r#"짜"#),
        (r#"짝"#, r#"짝"#, r#"짝"#, r#"짝"#, r#"짝"#),
        (r#"짞"#, r#"짞"#, r#"짞"#, r#"짞"#, r#"짞"#),
        (r#"짟"#, r#"짟"#, r#"짟"#, r#"짟"#, r#"짟"#),
        (r#"짠"#, r#"짠"#, r#"짠"#, r#"짠"#, r#"짠"#),
        (r#"짡"#, r#"짡"#, r#"짡"#, r#"짡"#, r#"짡"#),
        (r#"짢"#, r#"짢"#, r#"짢"#, r#"짢"#, r#"짢"#),
        (r#"짣"#, r#"짣"#, r#"짣"#, r#"짣"#, r#"짣"#),
        (r#"짤"#, r#"짤"#, r#"짤"#, r#"짤"#, r#"짤"#),
        (r#"짥"#, r#"짥"#, r#"짥"#, r#"짥"#, r#"짥"#),
        (r#"짦"#, r#"짦"#, r#"짦"#, r#"짦"#, r#"짦"#),
        (r#"짧"#, r#"짧"#, r#"짧"#, r#"짧"#, r#"짧"#),
        (r#"짨"#, r#"짨"#, r#"짨"#, r#"짨"#, r#"짨"#),
        (r#"짩"#, r#"짩"#, r#"짩"#, r#"짩"#, r#"짩"#),
        (r#"짪"#, r#"짪"#, r#"짪"#, r#"짪"#, r#"짪"#),
        (r#"짫"#, r#"짫"#, r#"짫"#, r#"짫"#, r#"짫"#),
        (r#"짬"#, r#"짬"#, r#"짬"#, r#"짬"#, r#"짬"#),
        (r#"짭"#, r#"짭"#, r#"짭"#, r#"짭"#, r#"짭"#),
        (r#"짮"#, r#"짮"#, r#"짮"#, r#"짮"#, r#"짮"#),
        (r#"짯"#, r#"짯"#, r#"짯"#, r#"짯"#, r#"짯"#),
        (r#"짰"#, r#"짰"#, r#"짰"#, r#"짰"#, r#"짰"#),
        (r#"짱"#, r#"짱"#, r#"짱"#, r#"짱"#, r#"짱"#),
        (r#"짲"#, r#"짲"#, r#"짲"#, r#"짲"#, r#"짲"#),
        (r#"짳"#, r#"짳"#, r#"짳"#, r#"짳"#, r#"짳"#),
        (r#"짴"#, r#"짴"#, r#"짴"#, r#"짴"#, r#"짴"#),
        (r#"짵"#, r#"짵"#, r#"짵"#, r#"짵"#, r#"짵"#),
        (r#"짶"#, r#"짶"#, r#"짶"#, r#"짶"#, r#"짶"#),
        (r#"짷"#, r#"짷"#, r#"짷"#, r#"짷"#, r#"짷"#),
        (r#"째"#, r#"째"#, r#"째"#, r#"째"#, r#"째"#),
        (r#"짹"#, r#"짹"#, r#"짹"#, r#"짹"#, r#"짹"#),
        (r#"짺"#, r#"짺"#, r#"짺"#, r#"짺"#, r#"짺"#),
        (r#"짻"#, r#"짻"#, r#"짻"#, r#"짻"#, r#"짻"#),
        (r#"짼"#, r#"짼"#, r#"짼"#, r#"짼"#, r#"짼"#),
        (r#"짽"#, r#"짽"#, r#"짽"#, r#"짽"#, r#"짽"#),
        (r#"짾"#, r#"짾"#, r#"짾"#, r#"짾"#, r#"짾"#),
        (r#"짿"#, r#"짿"#, r#"짿"#, r#"짿"#, r#"짿"#),
        (r#"쨀"#, r#"쨀"#, r#"쨀"#, r#"쨀"#, r#"쨀"#),
        (r#"쨁"#, r#"쨁"#, r#"쨁"#, r#"쨁"#, r#"쨁"#),
        (r#"쨂"#, r#"쨂"#, r#"쨂"#, r#"쨂"#, r#"쨂"#),
        (r#"쨃"#, r#"쨃"#, r#"쨃"#, r#"쨃"#, r#"쨃"#),
        (r#"쨄"#, r#"쨄"#, r#"쨄"#, r#"쨄"#, r#"쨄"#),
        (r#"쨅"#, r#"쨅"#, r#"쨅"#, r#"쨅"#, r#"쨅"#),
        (r#"쨆"#, r#"쨆"#, r#"쨆"#, r#"쨆"#, r#"쨆"#),
        (r#"쨇"#, r#"쨇"#, r#"쨇"#, r#"쨇"#, r#"쨇"#),
        (r#"쨈"#, r#"쨈"#, r#"쨈"#, r#"쨈"#, r#"쨈"#),
        (r#"쨉"#, r#"쨉"#, r#"쨉"#, r#"쨉"#, r#"쨉"#),
        (r#"쨊"#, r#"쨊"#, r#"쨊"#, r#"쨊"#, r#"쨊"#),
        (r#"쨋"#, r#"쨋"#, r#"쨋"#, r#"쨋"#, r#"쨋"#),
        (r#"쨌"#, r#"쨌"#, r#"쨌"#, r#"쨌"#, r#"쨌"#),
        (r#"쨍"#, r#"쨍"#, r#"쨍"#, r#"쨍"#, r#"쨍"#),
        (r#"쨎"#, r#"쨎"#, r#"쨎"#, r#"쨎"#, r#"쨎"#),
        (r#"쨏"#, r#"쨏"#, r#"쨏"#, r#"쨏"#, r#"쨏"#),
        (r#"쨐"#, r#"쨐"#, r#"쨐"#, r#"쨐"#, r#"쨐"#),
        (r#"쨑"#, r#"쨑"#, r#"쨑"#, r#"쨑"#, r#"쨑"#),
        (r#"쨒"#, r#"쨒"#, r#"쨒"#, r#"쨒"#, r#"쨒"#),
        (r#"쨓"#, r#"쨓"#, r#"쨓"#, r#"쨓"#, r#"쨓"#),
        (r#"쨔"#, r#"쨔"#, r#"쨔"#, r#"쨔"#, r#"쨔"#),
        (r#"쨕"#, r#"쨕"#, r#"쨕"#, r#"쨕"#, r#"쨕"#),
        (r#"쨖"#, r#"쨖"#, r#"쨖"#, r#"쨖"#, r#"쨖"#),
        (r#"쨗"#, r#"쨗"#, r#"쨗"#, r#"쨗"#, r#"쨗"#),
        (r#"쨘"#, r#"쨘"#, r#"쨘"#, r#"쨘"#, r#"쨘"#),
        (r#"쨙"#, r#"쨙"#, r#"쨙"#, r#"쨙"#, r#"쨙"#),
        (r#"쨚"#, r#"쨚"#, r#"쨚"#, r#"쨚"#, r#"쨚"#),
        (r#"쨛"#, r#"쨛"#, r#"쨛"#, r#"쨛"#, r#"쨛"#),
        (r#"쨜"#, r#"쨜"#, r#"쨜"#, r#"쨜"#, r#"쨜"#),
        (r#"쨝"#, r#"쨝"#, r#"쨝"#, r#"쨝"#, r#"쨝"#),
        (r#"쨞"#, r#"쨞"#, r#"쨞"#, r#"쨞"#, r#"쨞"#),
        (r#"쨟"#, r#"쨟"#, r#"쨟"#, r#"쨟"#, r#"쨟"#),
        (r#"쨠"#, r#"쨠"#, r#"쨠"#, r#"쨠"#, r#"쨠"#),
        (r#"쨡"#, r#"쨡"#, r#"쨡"#, r#"쨡"#, r#"쨡"#),
        (r#"쨢"#, r#"쨢"#, r#"쨢"#, r#"쨢"#, r#"쨢"#),
        (r#"쨣"#, r#"쨣"#, r#"쨣"#, r#"쨣"#, r#"쨣"#),
        (r#"쨤"#, r#"쨤"#, r#"쨤"#, r#"쨤"#, r#"쨤"#),
        (r#"쨥"#, r#"쨥"#, r#"쨥"#, r#"쨥"#, r#"쨥"#),
        (r#"쨦"#, r#"쨦"#, r#"쨦"#, r#"쨦"#, r#"쨦"#),
        (r#"쨧"#, r#"쨧"#, r#"쨧"#, r#"쨧"#, r#"쨧"#),
        (r#"쨨"#, r#"쨨"#, r#"쨨"#, r#"쨨"#, r#"쨨"#),
        (r#"쨩"#, r#"쨩"#, r#"쨩"#, r#"쨩"#, r#"쨩"#),
        (r#"쨪"#, r#"쨪"#, r#"쨪"#, r#"쨪"#, r#"쨪"#),
        (r#"쨫"#, r#"쨫"#, r#"쨫"#, r#"쨫"#, r#"쨫"#),
        (r#"쨬"#, r#"쨬"#, r#"쨬"#, r#"쨬"#, r#"쨬"#),
        (r#"쨭"#, r#"쨭"#, r#"쨭"#, r#"쨭"#, r#"쨭"#),
        (r#"쨮"#, r#"쨮"#, r#"쨮"#, r#"쨮"#, r#"쨮"#),
        (r#"쨯"#, r#"쨯"#, r#"쨯"#, r#"쨯"#, r#"쨯"#),
        (r#"쨰"#, r#"쨰"#, r#"쨰"#, r#"쨰"#, r#"쨰"#),
        (r#"쨱"#, r#"쨱"#, r#"쨱"#, r#"쨱"#, r#"쨱"#),
        (r#"쨲"#, r#"쨲"#, r#"쨲"#, r#"쨲"#, r#"쨲"#),
        (r#"쨳"#, r#"쨳"#, r#"쨳"#, r#"쨳"#, r#"쨳"#),
        (r#"쨴"#, r#"쨴"#, r#"쨴"#, r#"쨴"#, r#"쨴"#),
        (r#"쨵"#, r#"쨵"#, r#"쨵"#, r#"쨵"#, r#"쨵"#),
        (r#"쨶"#, r#"쨶"#, r#"쨶"#, r#"쨶"#, r#"쨶"#),
        (r#"쨷"#, r#"쨷"#, r#"쨷"#, r#"쨷"#, r#"쨷"#),
        (r#"쨸"#, r#"쨸"#, r#"쨸"#, r#"쨸"#, r#"쨸"#),
        (r#"쨹"#, r#"쨹"#, r#"쨹"#, r#"쨹"#, r#"쨹"#),
        (r#"쨺"#, r#"쨺"#, r#"쨺"#, r#"쨺"#, r#"쨺"#),
        (r#"쨻"#, r#"쨻"#, r#"쨻"#, r#"쨻"#, r#"쨻"#),
        (r#"쨼"#, r#"쨼"#, r#"쨼"#, r#"쨼"#, r#"쨼"#),
        (r#"쨽"#, r#"쨽"#, r#"쨽"#, r#"쨽"#, r#"쨽"#),
        (r#"쨾"#, r#"쨾"#, r#"쨾"#, r#"쨾"#, r#"쨾"#),
        (r#"쨿"#, r#"쨿"#, r#"쨿"#, r#"쨿"#, r#"쨿"#),
        (r#"쩀"#, r#"쩀"#, r#"쩀"#, r#"쩀"#, r#"쩀"#),
        (r#"쩁"#, r#"쩁"#, r#"쩁"#, r#"쩁"#, r#"쩁"#),
        (r#"쩂"#, r#"쩂"#, r#"쩂"#, r#"쩂"#, r#"쩂"#),
        (r#"쩃"#, r#"쩃"#, r#"쩃"#, r#"쩃"#, r#"쩃"#),
        (r#"쩄"#, r#"쩄"#, r#"쩄"#, r#"쩄"#, r#"쩄"#),
        (r#"쩅"#, r#"쩅"#, r#"쩅"#, r#"쩅"#, r#"쩅"#),
        (r#"쩆"#, r#"쩆"#, r#"쩆"#, r#"쩆"#, r#"쩆"#),
        (r#"쩇"#, r#"쩇"#, r#"쩇"#, r#"쩇"#, r#"쩇"#),
        (r#"쩈"#, r#"쩈"#, r#"쩈"#, r#"쩈"#, r#"쩈"#),
        (r#"쩉"#, r#"쩉"#, r#"쩉"#, r#"쩉"#, r#"쩉"#),
        (r#"쩊"#, r#"쩊"#, r#"쩊"#, r#"쩊"#, r#"쩊"#),
        (r#"쩋"#, r#"쩋"#, r#"쩋"#, r#"쩋"#, r#"쩋"#),
        (r#"쩌"#, r#"쩌"#, r#"쩌"#, r#"쩌"#, r#"쩌"#),
        (r#"쩍"#, r#"쩍"#, r#"쩍"#, r#"쩍"#, r#"쩍"#),
        (r#"쩎"#, r#"쩎"#, r#"쩎"#, r#"쩎"#, r#"쩎"#),
        (r#"쩏"#, r#"쩏"#, r#"쩏"#, r#"쩏"#, r#"쩏"#),
        (r#"쩐"#, r#"쩐"#, r#"쩐"#, r#"쩐"#, r#"쩐"#),
        (r#"쩑"#, r#"쩑"#, r#"쩑"#, r#"쩑"#, r#"쩑"#),
        (r#"쩒"#, r#"쩒"#, r#"쩒"#, r#"쩒"#, r#"쩒"#),
        (r#"쩓"#, r#"쩓"#, r#"쩓"#, r#"쩓"#, r#"쩓"#),
        (r#"쩔"#, r#"쩔"#, r#"쩔"#, r#"쩔"#, r#"쩔"#),
        (r#"쩕"#, r#"쩕"#, r#"쩕"#, r#"쩕"#, r#"쩕"#),
        (r#"쩖"#, r#"쩖"#, r#"쩖"#, r#"쩖"#, r#"쩖"#),
        (r#"쩗"#, r#"쩗"#, r#"쩗"#, r#"쩗"#, r#"쩗"#),
        (r#"쩘"#, r#"쩘"#, r#"쩘"#, r#"쩘"#, r#"쩘"#),
        (r#"쩙"#, r#"쩙"#, r#"쩙"#, r#"쩙"#, r#"쩙"#),
        (r#"쩚"#, r#"쩚"#, r#"쩚"#, r#"쩚"#, r#"쩚"#),
        (r#"쩛"#, r#"쩛"#, r#"쩛"#, r#"쩛"#, r#"쩛"#),
        (r#"쩜"#, r#"쩜"#, r#"쩜"#, r#"쩜"#, r#"쩜"#),
        (r#"쩝"#, r#"쩝"#, r#"쩝"#, r#"쩝"#, r#"쩝"#),
        (r#"쩞"#, r#"쩞"#, r#"쩞"#, r#"쩞"#, r#"쩞"#),
        (r#"쩟"#, r#"쩟"#, r#"쩟"#, r#"쩟"#, r#"쩟"#),
        (r#"쩠"#, r#"쩠"#, r#"쩠"#, r#"쩠"#, r#"쩠"#),
        (r#"쩡"#, r#"쩡"#, r#"쩡"#, r#"쩡"#, r#"쩡"#),
        (r#"쩢"#, r#"쩢"#, r#"쩢"#, r#"쩢"#, r#"쩢"#),
        (r#"쩣"#, r#"쩣"#, r#"쩣"#, r#"쩣"#, r#"쩣"#),
        (r#"쩤"#, r#"쩤"#, r#"쩤"#, r#"쩤"#, r#"쩤"#),
        (r#"쩥"#, r#"쩥"#, r#"쩥"#, r#"쩥"#, r#"쩥"#),
        (r#"쩦"#, r#"쩦"#, r#"쩦"#, r#"쩦"#, r#"쩦"#),
        (r#"쩧"#, r#"쩧"#, r#"쩧"#, r#"쩧"#, r#"쩧"#),
        (r#"쩨"#, r#"쩨"#, r#"쩨"#, r#"쩨"#, r#"쩨"#),
        (r#"쩩"#, r#"쩩"#, r#"쩩"#, r#"쩩"#, r#"쩩"#),
        (r#"쩪"#, r#"쩪"#, r#"쩪"#, r#"쩪"#, r#"쩪"#),
        (r#"쩫"#, r#"쩫"#, r#"쩫"#, r#"쩫"#, r#"쩫"#),
        (r#"쩬"#, r#"쩬"#, r#"쩬"#, r#"쩬"#, r#"쩬"#),
        (r#"쩭"#, r#"쩭"#, r#"쩭"#, r#"쩭"#, r#"쩭"#),
        (r#"쩮"#, r#"쩮"#, r#"쩮"#, r#"쩮"#, r#"쩮"#),
        (r#"쩯"#, r#"쩯"#, r#"쩯"#, r#"쩯"#, r#"쩯"#),
        (r#"쩰"#, r#"쩰"#, r#"쩰"#, r#"쩰"#, r#"쩰"#),
        (r#"쩱"#, r#"쩱"#, r#"쩱"#, r#"쩱"#, r#"쩱"#),
        (r#"쩲"#, r#"쩲"#, r#"쩲"#, r#"쩲"#, r#"쩲"#),
        (r#"쩳"#, r#"쩳"#, r#"쩳"#, r#"쩳"#, r#"쩳"#),
        (r#"쩴"#, r#"쩴"#, r#"쩴"#, r#"쩴"#, r#"쩴"#),
        (r#"쩵"#, r#"쩵"#, r#"쩵"#, r#"쩵"#, r#"쩵"#),
        (r#"쩶"#, r#"쩶"#, r#"쩶"#, r#"쩶"#, r#"쩶"#),
        (r#"쩷"#, r#"쩷"#, r#"쩷"#, r#"쩷"#, r#"쩷"#),
        (r#"쩸"#, r#"쩸"#, r#"쩸"#, r#"쩸"#, r#"쩸"#),
        (r#"쩹"#, r#"쩹"#, r#"쩹"#, r#"쩹"#, r#"쩹"#),
        (r#"쩺"#, r#"쩺"#, r#"쩺"#, r#"쩺"#, r#"쩺"#),
        (r#"쩻"#, r#"쩻"#, r#"쩻"#, r#"쩻"#, r#"쩻"#),
        (r#"쩼"#, r#"쩼"#, r#"쩼"#, r#"쩼"#, r#"쩼"#),
        (r#"쩽"#, r#"쩽"#, r#"쩽"#, r#"쩽"#, r#"쩽"#),
        (r#"쩾"#, r#"쩾"#, r#"쩾"#, r#"쩾"#, r#"쩾"#),
        (r#"쩿"#, r#"쩿"#, r#"쩿"#, r#"쩿"#, r#"쩿"#),
        (r#"쪀"#, r#"쪀"#, r#"쪀"#, r#"쪀"#, r#"쪀"#),
        (r#"쪁"#, r#"쪁"#, r#"쪁"#, r#"쪁"#, r#"쪁"#),
        (r#"쪂"#, r#"쪂"#, r#"쪂"#, r#"쪂"#, r#"쪂"#),
        (r#"쪃"#, r#"쪃"#, r#"쪃"#, r#"쪃"#, r#"쪃"#),
        (r#"쪄"#, r#"쪄"#, r#"쪄"#, r#"쪄"#, r#"쪄"#),
        (r#"쪅"#, r#"쪅"#, r#"쪅"#, r#"쪅"#, r#"쪅"#),
        (r#"쪆"#, r#"쪆"#, r#"쪆"#, r#"쪆"#, r#"쪆"#),
        (r#"쪇"#, r#"쪇"#, r#"쪇"#, r#"쪇"#, r#"쪇"#),
        (r#"쪈"#, r#"쪈"#, r#"쪈"#, r#"쪈"#, r#"쪈"#),
        (r#"쪉"#, r#"쪉"#, r#"쪉"#, r#"쪉"#, r#"쪉"#),
        (r#"쪊"#, r#"쪊"#, r#"쪊"#, r#"쪊"#, r#"쪊"#),
        (r#"쪋"#, r#"쪋"#, r#"쪋"#, r#"쪋"#, r#"쪋"#),
        (r#"쪌"#, r#"쪌"#, r#"쪌"#, r#"쪌"#, r#"쪌"#),
        (r#"쪍"#, r#"쪍"#, r#"쪍"#, r#"쪍"#, r#"쪍"#),
        (r#"쪎"#, r#"쪎"#, r#"쪎"#, r#"쪎"#, r#"쪎"#),
        (r#"쪏"#, r#"쪏"#, r#"쪏"#, r#"쪏"#, r#"쪏"#),
        (r#"쪐"#, r#"쪐"#, r#"쪐"#, r#"쪐"#, r#"쪐"#),
        (r#"쪑"#, r#"쪑"#, r#"쪑"#, r#"쪑"#, r#"쪑"#),
        (r#"쪒"#, r#"쪒"#, r#"쪒"#, r#"쪒"#, r#"쪒"#),
        (r#"쪓"#, r#"쪓"#, r#"쪓"#, r#"쪓"#, r#"쪓"#),
        (r#"쪔"#, r#"쪔"#, r#"쪔"#, r#"쪔"#, r#"쪔"#),
        (r#"쪕"#, r#"쪕"#, r#"쪕"#, r#"쪕"#, r#"쪕"#),
        (r#"쪖"#, r#"쪖"#, r#"쪖"#, r#"쪖"#, r#"쪖"#),
        (r#"쪗"#, r#"쪗"#, r#"쪗"#, r#"쪗"#, r#"쪗"#),
        (r#"쪘"#, r#"쪘"#, r#"쪘"#, r#"쪘"#, r#"쪘"#),
        (r#"쪙"#, r#"쪙"#, r#"쪙"#, r#"쪙"#, r#"쪙"#),
        (r#"쪚"#, r#"쪚"#, r#"쪚"#, r#"쪚"#, r#"쪚"#),
        (r#"쪛"#, r#"쪛"#, r#"쪛"#, r#"쪛"#, r#"쪛"#),
        (r#"쪜"#, r#"쪜"#, r#"쪜"#, r#"쪜"#, r#"쪜"#),
        (r#"쪝"#, r#"쪝"#, r#"쪝"#, r#"쪝"#, r#"쪝"#),
        (r#"쪞"#, r#"쪞"#, r#"쪞"#, r#"쪞"#, r#"쪞"#),
        (r#"쪟"#, r#"쪟"#, r#"쪟"#, r#"쪟"#, r#"쪟"#),
        (r#"쪠"#, r#"쪠"#, r#"쪠"#, r#"쪠"#, r#"쪠"#),
        (r#"쪡"#, r#"쪡"#, r#"쪡"#, r#"쪡"#, r#"쪡"#),
        (r#"쪢"#, r#"쪢"#, r#"쪢"#, r#"쪢"#, r#"쪢"#),
        (r#"쪣"#, r#"쪣"#, r#"쪣"#, r#"쪣"#, r#"쪣"#),
        (r#"쪤"#, r#"쪤"#, r#"쪤"#, r#"쪤"#, r#"쪤"#),
        (r#"쪥"#, r#"쪥"#, r#"쪥"#, r#"쪥"#, r#"쪥"#),
        (r#"쪦"#, r#"쪦"#, r#"쪦"#, r#"쪦"#, r#"쪦"#),
        (r#"쪧"#, r#"쪧"#, r#"쪧"#, r#"쪧"#, r#"쪧"#),
        (r#"쪨"#, r#"쪨"#, r#"쪨"#, r#"쪨"#, r#"쪨"#),
        (r#"쪩"#, r#"쪩"#, r#"쪩"#, r#"쪩"#, r#"쪩"#),
        (r#"쪪"#, r#"쪪"#, r#"쪪"#, r#"쪪"#, r#"쪪"#),
        (r#"쪫"#, r#"쪫"#, r#"쪫"#, r#"쪫"#, r#"쪫"#),
        (r#"쪬"#, r#"쪬"#, r#"쪬"#, r#"쪬"#, r#"쪬"#),
        (r#"쪭"#, r#"쪭"#, r#"쪭"#, r#"쪭"#, r#"쪭"#),
        (r#"쪮"#, r#"쪮"#, r#"쪮"#, r#"쪮"#, r#"쪮"#),
        (r#"쪯"#, r#"쪯"#, r#"쪯"#, r#"쪯"#, r#"쪯"#),
        (r#"쪰"#, r#"쪰"#, r#"쪰"#, r#"쪰"#, r#"쪰"#),
        (r#"쪱"#, r#"쪱"#, r#"쪱"#, r#"쪱"#, r#"쪱"#),
        (r#"쪲"#, r#"쪲"#, r#"쪲"#, r#"쪲"#, r#"쪲"#),
        (r#"쪳"#, r#"쪳"#, r#"쪳"#, r#"쪳"#, r#"쪳"#),
        (r#"쪴"#, r#"쪴"#, r#"쪴"#, r#"쪴"#, r#"쪴"#),
        (r#"쪵"#, r#"쪵"#, r#"쪵"#, r#"쪵"#, r#"쪵"#),
        (r#"쪶"#, r#"쪶"#, r#"쪶"#, r#"쪶"#, r#"쪶"#),
        (r#"쪷"#, r#"쪷"#, r#"쪷"#, r#"쪷"#, r#"쪷"#),
        (r#"쪸"#, r#"쪸"#, r#"쪸"#, r#"쪸"#, r#"쪸"#),
        (r#"쪹"#, r#"쪹"#, r#"쪹"#, r#"쪹"#, r#"쪹"#),
        (r#"쪺"#, r#"쪺"#, r#"쪺"#, r#"쪺"#, r#"쪺"#),
        (r#"쪻"#, r#"쪻"#, r#"쪻"#, r#"쪻"#, r#"쪻"#),
        (r#"쪼"#, r#"쪼"#, r#"쪼"#, r#"쪼"#, r#"쪼"#),
        (r#"쪽"#, r#"쪽"#, r#"쪽"#, r#"쪽"#, r#"쪽"#),
        (r#"쪾"#, r#"쪾"#, r#"쪾"#, r#"쪾"#, r#"쪾"#),
        (r#"쪿"#, r#"쪿"#, r#"쪿"#, r#"쪿"#, r#"쪿"#),
        (r#"쫀"#, r#"쫀"#, r#"쫀"#, r#"쫀"#, r#"쫀"#),
        (r#"쫁"#, r#"쫁"#, r#"쫁"#, r#"쫁"#, r#"쫁"#),
        (r#"쫂"#, r#"쫂"#, r#"쫂"#, r#"쫂"#, r#"쫂"#),
        (r#"쫃"#, r#"쫃"#, r#"쫃"#, r#"쫃"#, r#"쫃"#),
        (r#"쫄"#, r#"쫄"#, r#"쫄"#, r#"쫄"#, r#"쫄"#),
        (r#"쫅"#, r#"쫅"#, r#"쫅"#, r#"쫅"#, r#"쫅"#),
        (r#"쫆"#, r#"쫆"#, r#"쫆"#, r#"쫆"#, r#"쫆"#),
        (r#"쫇"#, r#"쫇"#, r#"쫇"#, r#"쫇"#, r#"쫇"#),
        (r#"쫈"#, r#"쫈"#, r#"쫈"#, r#"쫈"#, r#"쫈"#),
        (r#"쫉"#, r#"쫉"#, r#"쫉"#, r#"쫉"#, r#"쫉"#),
        (r#"쫊"#, r#"쫊"#, r#"쫊"#, r#"쫊"#, r#"쫊"#),
        (r#"쫋"#, r#"쫋"#, r#"쫋"#, r#"쫋"#, r#"쫋"#),
        (r#"쫌"#, r#"쫌"#, r#"쫌"#, r#"쫌"#, r#"쫌"#),
        (r#"쫍"#, r#"쫍"#, r#"쫍"#, r#"쫍"#, r#"쫍"#),
        (r#"쫎"#, r#"쫎"#, r#"쫎"#, r#"쫎"#, r#"쫎"#),
        (r#"쫏"#, r#"쫏"#, r#"쫏"#, r#"쫏"#, r#"쫏"#),
        (r#"쫐"#, r#"쫐"#, r#"쫐"#, r#"쫐"#, r#"쫐"#),
        (r#"쫑"#, r#"쫑"#, r#"쫑"#, r#"쫑"#, r#"쫑"#),
        (r#"쫒"#, r#"쫒"#, r#"쫒"#, r#"쫒"#, r#"쫒"#),
        (r#"쫓"#, r#"쫓"#, r#"쫓"#, r#"쫓"#, r#"쫓"#),
        (r#"쫔"#, r#"쫔"#, r#"쫔"#, r#"쫔"#, r#"쫔"#),
        (r#"쫕"#, r#"쫕"#, r#"쫕"#, r#"쫕"#, r#"쫕"#),
        (r#"쫖"#, r#"쫖"#, r#"쫖"#, r#"쫖"#, r#"쫖"#),
        (r#"쫗"#, r#"쫗"#, r#"쫗"#, r#"쫗"#, r#"쫗"#),
        (r#"쫘"#, r#"쫘"#, r#"쫘"#, r#"쫘"#, r#"쫘"#),
        (r#"쫙"#, r#"쫙"#, r#"쫙"#, r#"쫙"#, r#"쫙"#),
        (r#"쫚"#, r#"쫚"#, r#"쫚"#, r#"쫚"#, r#"쫚"#),
        (r#"쫛"#, r#"쫛"#, r#"쫛"#, r#"쫛"#, r#"쫛"#),
        (r#"쫜"#, r#"쫜"#, r#"쫜"#, r#"쫜"#, r#"쫜"#),
        (r#"쫝"#, r#"쫝"#, r#"쫝"#, r#"쫝"#, r#"쫝"#),
        (r#"쫞"#, r#"쫞"#, r#"쫞"#, r#"쫞"#, r#"쫞"#),
        (r#"쫟"#, r#"쫟"#, r#"쫟"#, r#"쫟"#, r#"쫟"#),
        (r#"쫠"#, r#"쫠"#, r#"쫠"#, r#"쫠"#, r#"쫠"#),
        (r#"쫡"#, r#"쫡"#, r#"쫡"#, r#"쫡"#, r#"쫡"#),
        (r#"쫢"#, r#"쫢"#, r#"쫢"#, r#"쫢"#, r#"쫢"#),
        (r#"쫣"#, r#"쫣"#, r#"쫣"#, r#"쫣"#, r#"쫣"#),
        (r#"쫤"#, r#"쫤"#, r#"쫤"#, r#"쫤"#, r#"쫤"#),
        (r#"쫥"#, r#"쫥"#, r#"쫥"#, r#"쫥"#, r#"쫥"#),
        (r#"쫦"#, r#"쫦"#, r#"쫦"#, r#"쫦"#, r#"쫦"#),
        (r#"쫧"#, r#"쫧"#, r#"쫧"#, r#"쫧"#, r#"쫧"#),
        (r#"쫨"#, r#"쫨"#, r#"쫨"#, r#"쫨"#, r#"쫨"#),
        (r#"쫩"#, r#"쫩"#, r#"쫩"#, r#"쫩"#, r#"쫩"#),
        (r#"쫪"#, r#"쫪"#, r#"쫪"#, r#"쫪"#, r#"쫪"#),
        (r#"쫫"#, r#"쫫"#, r#"쫫"#, r#"쫫"#, r#"쫫"#),
        (r#"쫬"#, r#"쫬"#, r#"쫬"#, r#"쫬"#, r#"쫬"#),
        (r#"쫭"#, r#"쫭"#, r#"쫭"#, r#"쫭"#, r#"쫭"#),
        (r#"쫮"#, r#"쫮"#, r#"쫮"#, r#"쫮"#, r#"쫮"#),
        (r#"쫯"#, r#"쫯"#, r#"쫯"#, r#"쫯"#, r#"쫯"#),
        (r#"쫰"#, r#"쫰"#, r#"쫰"#, r#"쫰"#, r#"쫰"#),
        (r#"쫱"#, r#"쫱"#, r#"쫱"#, r#"쫱"#, r#"쫱"#),
        (r#"쫲"#, r#"쫲"#, r#"쫲"#, r#"쫲"#, r#"쫲"#),
        (r#"쫳"#, r#"쫳"#, r#"쫳"#, r#"쫳"#, r#"쫳"#),
        (r#"쫴"#, r#"쫴"#, r#"쫴"#, r#"쫴"#, r#"쫴"#),
        (r#"쫵"#, r#"쫵"#, r#"쫵"#, r#"쫵"#, r#"쫵"#),
        (r#"쫶"#, r#"쫶"#, r#"쫶"#, r#"쫶"#, r#"쫶"#),
        (r#"쫷"#, r#"쫷"#, r#"쫷"#, r#"쫷"#, r#"쫷"#),
        (r#"쫸"#, r#"쫸"#, r#"쫸"#, r#"쫸"#, r#"쫸"#),
        (r#"쫹"#, r#"쫹"#, r#"쫹"#, r#"쫹"#, r#"쫹"#),
        (r#"쫺"#, r#"쫺"#, r#"쫺"#, r#"쫺"#, r#"쫺"#),
        (r#"쫻"#, r#"쫻"#, r#"쫻"#, r#"쫻"#, r#"쫻"#),
        (r#"쫼"#, r#"쫼"#, r#"쫼"#, r#"쫼"#, r#"쫼"#),
        (r#"쫽"#, r#"쫽"#, r#"쫽"#, r#"쫽"#, r#"쫽"#),
        (r#"쫾"#, r#"쫾"#, r#"쫾"#, r#"쫾"#, r#"쫾"#),
        (r#"쫿"#, r#"쫿"#, r#"쫿"#, r#"쫿"#, r#"쫿"#),
        (r#"쬀"#, r#"쬀"#, r#"쬀"#, r#"쬀"#, r#"쬀"#),
        (r#"쬁"#, r#"쬁"#, r#"쬁"#, r#"쬁"#, r#"쬁"#),
        (r#"쬂"#, r#"쬂"#, r#"쬂"#, r#"쬂"#, r#"쬂"#),
        (r#"쬃"#, r#"쬃"#, r#"쬃"#, r#"쬃"#, r#"쬃"#),
        (r#"쬄"#, r#"쬄"#, r#"쬄"#, r#"쬄"#, r#"쬄"#),
        (r#"쬅"#, r#"쬅"#, r#"쬅"#, r#"쬅"#, r#"쬅"#),
        (r#"쬆"#, r#"쬆"#, r#"쬆"#, r#"쬆"#, r#"쬆"#),
        (r#"쬇"#, r#"쬇"#, r#"쬇"#, r#"쬇"#, r#"쬇"#),
        (r#"쬈"#, r#"쬈"#, r#"쬈"#, r#"쬈"#, r#"쬈"#),
        (r#"쬉"#, r#"쬉"#, r#"쬉"#, r#"쬉"#, r#"쬉"#),
        (r#"쬊"#, r#"쬊"#, r#"쬊"#, r#"쬊"#, r#"쬊"#),
        (r#"쬋"#, r#"쬋"#, r#"쬋"#, r#"쬋"#, r#"쬋"#),
        (r#"쬌"#, r#"쬌"#, r#"쬌"#, r#"쬌"#, r#"쬌"#),
        (r#"쬍"#, r#"쬍"#, r#"쬍"#, r#"쬍"#, r#"쬍"#),
        (r#"쬎"#, r#"쬎"#, r#"쬎"#, r#"쬎"#, r#"쬎"#),
        (r#"쬏"#, r#"쬏"#, r#"쬏"#, r#"쬏"#, r#"쬏"#),
        (r#"쬐"#, r#"쬐"#, r#"쬐"#, r#"쬐"#, r#"쬐"#),
        (r#"쬑"#, r#"쬑"#, r#"쬑"#, r#"쬑"#, r#"쬑"#),
        (r#"쬒"#, r#"쬒"#, r#"쬒"#, r#"쬒"#, r#"쬒"#),
        (r#"쬓"#, r#"쬓"#, r#"쬓"#, r#"쬓"#, r#"쬓"#),
        (r#"쬔"#, r#"쬔"#, r#"쬔"#, r#"쬔"#, r#"쬔"#),
        (r#"쬕"#, r#"쬕"#, r#"쬕"#, r#"쬕"#, r#"쬕"#),
        (r#"쬖"#, r#"쬖"#, r#"쬖"#, r#"쬖"#, r#"쬖"#),
        (r#"쬗"#, r#"쬗"#, r#"쬗"#, r#"쬗"#, r#"쬗"#),
        (r#"쬘"#, r#"쬘"#, r#"쬘"#, r#"쬘"#, r#"쬘"#),
        (r#"쬙"#, r#"쬙"#, r#"쬙"#, r#"쬙"#, r#"쬙"#),
        (r#"쬚"#, r#"쬚"#, r#"쬚"#, r#"쬚"#, r#"쬚"#),
        (r#"쬛"#, r#"쬛"#, r#"쬛"#, r#"쬛"#, r#"쬛"#),
        (r#"쬜"#, r#"쬜"#, r#"쬜"#, r#"쬜"#, r#"쬜"#),
        (r#"쬝"#, r#"쬝"#, r#"쬝"#, r#"쬝"#, r#"쬝"#),
        (r#"쬞"#, r#"쬞"#, r#"쬞"#, r#"쬞"#, r#"쬞"#),
        (r#"쬟"#, r#"쬟"#, r#"쬟"#, r#"쬟"#, r#"쬟"#),
        (r#"쬠"#, r#"쬠"#, r#"쬠"#, r#"쬠"#, r#"쬠"#),
        (r#"쬡"#, r#"쬡"#, r#"쬡"#, r#"쬡"#, r#"쬡"#),
        (r#"쬢"#, r#"쬢"#, r#"쬢"#, r#"쬢"#, r#"쬢"#),
        (r#"쬣"#, r#"쬣"#, r#"쬣"#, r#"쬣"#, r#"쬣"#),
        (r#"쬤"#, r#"쬤"#, r#"쬤"#, r#"쬤"#, r#"쬤"#),
        (r#"쬥"#, r#"쬥"#, r#"쬥"#, r#"쬥"#, r#"쬥"#),
        (r#"쬦"#, r#"쬦"#, r#"쬦"#, r#"쬦"#, r#"쬦"#),
        (r#"쬧"#, r#"쬧"#, r#"쬧"#, r#"쬧"#, r#"쬧"#),
        (r#"쬨"#, r#"쬨"#, r#"쬨"#, r#"쬨"#, r#"쬨"#),
        (r#"쬩"#, r#"쬩"#, r#"쬩"#, r#"쬩"#, r#"쬩"#),
        (r#"쬪"#, r#"쬪"#, r#"쬪"#, r#"쬪"#, r#"쬪"#),
        (r#"쬫"#, r#"쬫"#, r#"쬫"#, r#"쬫"#, r#"쬫"#),
        (r#"쬬"#, r#"쬬"#, r#"쬬"#, r#"쬬"#, r#"쬬"#),
        (r#"쬭"#, r#"쬭"#, r#"쬭"#, r#"쬭"#, r#"쬭"#),
        (r#"쬮"#, r#"쬮"#, r#"쬮"#, r#"쬮"#, r#"쬮"#),
        (r#"쬯"#, r#"쬯"#, r#"쬯"#, r#"쬯"#, r#"쬯"#),
        (r#"쬰"#, r#"쬰"#, r#"쬰"#, r#"쬰"#, r#"쬰"#),
        (r#"쬱"#, r#"쬱"#, r#"쬱"#, r#"쬱"#, r#"쬱"#),
        (r#"쬲"#, r#"쬲"#, r#"쬲"#, r#"쬲"#, r#"쬲"#),
        (r#"쬳"#, r#"쬳"#, r#"쬳"#, r#"쬳"#, r#"쬳"#),
        (r#"쬴"#, r#"쬴"#, r#"쬴"#, r#"쬴"#, r#"쬴"#),
        (r#"쬵"#, r#"쬵"#, r#"쬵"#, r#"쬵"#, r#"쬵"#),
        (r#"쬶"#, r#"쬶"#, r#"쬶"#, r#"쬶"#, r#"쬶"#),
        (r#"쬷"#, r#"쬷"#, r#"쬷"#, r#"쬷"#, r#"쬷"#),
        (r#"쬸"#, r#"쬸"#, r#"쬸"#, r#"쬸"#, r#"쬸"#),
        (r#"쬹"#, r#"쬹"#, r#"쬹"#, r#"쬹"#, r#"쬹"#),
        (r#"쬺"#, r#"쬺"#, r#"쬺"#, r#"쬺"#, r#"쬺"#),
        (r#"쬻"#, r#"쬻"#, r#"쬻"#, r#"쬻"#, r#"쬻"#),
        (r#"쬼"#, r#"쬼"#, r#"쬼"#, r#"쬼"#, r#"쬼"#),
        (r#"쬽"#, r#"쬽"#, r#"쬽"#, r#"쬽"#, r#"쬽"#),
        (r#"쬾"#, r#"쬾"#, r#"쬾"#, r#"쬾"#, r#"쬾"#),
        (r#"쬿"#, r#"쬿"#, r#"쬿"#, r#"쬿"#, r#"쬿"#),
        (r#"쭀"#, r#"쭀"#, r#"쭀"#, r#"쭀"#, r#"쭀"#),
        (r#"쭁"#, r#"쭁"#, r#"쭁"#, r#"쭁"#, r#"쭁"#),
        (r#"쭂"#, r#"쭂"#, r#"쭂"#, r#"쭂"#, r#"쭂"#),
        (r#"쭃"#, r#"쭃"#, r#"쭃"#, r#"쭃"#, r#"쭃"#),
        (r#"쭄"#, r#"쭄"#, r#"쭄"#, r#"쭄"#, r#"쭄"#),
        (r#"쭅"#, r#"쭅"#, r#"쭅"#, r#"쭅"#, r#"쭅"#),
        (r#"쭆"#, r#"쭆"#, r#"쭆"#, r#"쭆"#, r#"쭆"#),
        (r#"쭇"#, r#"쭇"#, r#"쭇"#, r#"쭇"#, r#"쭇"#),
        (r#"쭈"#, r#"쭈"#, r#"쭈"#, r#"쭈"#, r#"쭈"#),
        (r#"쭉"#, r#"쭉"#, r#"쭉"#, r#"쭉"#, r#"쭉"#),
        (r#"쭊"#, r#"쭊"#, r#"쭊"#, r#"쭊"#, r#"쭊"#),
        (r#"쭋"#, r#"쭋"#, r#"쭋"#, r#"쭋"#, r#"쭋"#),
        (r#"쭌"#, r#"쭌"#, r#"쭌"#, r#"쭌"#, r#"쭌"#),
        (r#"쭍"#, r#"쭍"#, r#"쭍"#, r#"쭍"#, r#"쭍"#),
        (r#"쭎"#, r#"쭎"#, r#"쭎"#, r#"쭎"#, r#"쭎"#),
        (r#"쭏"#, r#"쭏"#, r#"쭏"#, r#"쭏"#, r#"쭏"#),
        (r#"쭐"#, r#"쭐"#, r#"쭐"#, r#"쭐"#, r#"쭐"#),
        (r#"쭑"#, r#"쭑"#, r#"쭑"#, r#"쭑"#, r#"쭑"#),
        (r#"쭒"#, r#"쭒"#, r#"쭒"#, r#"쭒"#, r#"쭒"#),
        (r#"쭓"#, r#"쭓"#, r#"쭓"#, r#"쭓"#, r#"쭓"#),
        (r#"쭔"#, r#"쭔"#, r#"쭔"#, r#"쭔"#, r#"쭔"#),
        (r#"쭕"#, r#"쭕"#, r#"쭕"#, r#"쭕"#, r#"쭕"#),
        (r#"쭖"#, r#"쭖"#, r#"쭖"#, r#"쭖"#, r#"쭖"#),
        (r#"쭗"#, r#"쭗"#, r#"쭗"#, r#"쭗"#, r#"쭗"#),
        (r#"쭘"#, r#"쭘"#, r#"쭘"#, r#"쭘"#, r#"쭘"#),
        (r#"쭙"#, r#"쭙"#, r#"쭙"#, r#"쭙"#, r#"쭙"#),
        (r#"쭚"#, r#"쭚"#, r#"쭚"#, r#"쭚"#, r#"쭚"#),
        (r#"쭛"#, r#"쭛"#, r#"쭛"#, r#"쭛"#, r#"쭛"#),
        (r#"쭜"#, r#"쭜"#, r#"쭜"#, r#"쭜"#, r#"쭜"#),
        (r#"쭝"#, r#"쭝"#, r#"쭝"#, r#"쭝"#, r#"쭝"#),
        (r#"쭞"#, r#"쭞"#, r#"쭞"#, r#"쭞"#, r#"쭞"#),
        (r#"쭟"#, r#"쭟"#, r#"쭟"#, r#"쭟"#, r#"쭟"#),
        (r#"쭠"#, r#"쭠"#, r#"쭠"#, r#"쭠"#, r#"쭠"#),
        (r#"쭡"#, r#"쭡"#, r#"쭡"#, r#"쭡"#, r#"쭡"#),
        (r#"쭢"#, r#"쭢"#, r#"쭢"#, r#"쭢"#, r#"쭢"#),
        (r#"쭣"#, r#"쭣"#, r#"쭣"#, r#"쭣"#, r#"쭣"#),
        (r#"쭤"#, r#"쭤"#, r#"쭤"#, r#"쭤"#, r#"쭤"#),
        (r#"쭥"#, r#"쭥"#, r#"쭥"#, r#"쭥"#, r#"쭥"#),
        (r#"쭦"#, r#"쭦"#, r#"쭦"#, r#"쭦"#, r#"쭦"#),
        (r#"쭧"#, r#"쭧"#, r#"쭧"#, r#"쭧"#, r#"쭧"#),
        (r#"쭨"#, r#"쭨"#, r#"쭨"#, r#"쭨"#, r#"쭨"#),
        (r#"쭩"#, r#"쭩"#, r#"쭩"#, r#"쭩"#, r#"쭩"#),
        (r#"쭪"#, r#"쭪"#, r#"쭪"#, r#"쭪"#, r#"쭪"#),
        (r#"쭫"#, r#"쭫"#, r#"쭫"#, r#"쭫"#, r#"쭫"#),
        (r#"쭬"#, r#"쭬"#, r#"쭬"#, r#"쭬"#, r#"쭬"#),
        (r#"쭭"#, r#"쭭"#, r#"쭭"#, r#"쭭"#, r#"쭭"#),
        (r#"쭮"#, r#"쭮"#, r#"쭮"#, r#"쭮"#, r#"쭮"#),
        (r#"쭯"#, r#"쭯"#, r#"쭯"#, r#"쭯"#, r#"쭯"#),
        (r#"쭰"#, r#"쭰"#, r#"쭰"#, r#"쭰"#, r#"쭰"#),
        (r#"쭱"#, r#"쭱"#, r#"쭱"#, r#"쭱"#, r#"쭱"#),
        (r#"쭲"#, r#"쭲"#, r#"쭲"#, r#"쭲"#, r#"쭲"#),
        (r#"쭳"#, r#"쭳"#, r#"쭳"#, r#"쭳"#, r#"쭳"#),
        (r#"쭴"#, r#"쭴"#, r#"쭴"#, r#"쭴"#, r#"쭴"#),
        (r#"쭵"#, r#"쭵"#, r#"쭵"#, r#"쭵"#, r#"쭵"#),
        (r#"쭶"#, r#"쭶"#, r#"쭶"#, r#"쭶"#, r#"쭶"#),
        (r#"쭷"#, r#"쭷"#, r#"쭷"#, r#"쭷"#, r#"쭷"#),
        (r#"쭸"#, r#"쭸"#, r#"쭸"#, r#"쭸"#, r#"쭸"#),
        (r#"쭹"#, r#"쭹"#, r#"쭹"#, r#"쭹"#, r#"쭹"#),
        (r#"쭺"#, r#"쭺"#, r#"쭺"#, r#"쭺"#, r#"쭺"#),
        (r#"쭻"#, r#"쭻"#, r#"쭻"#, r#"쭻"#, r#"쭻"#),
        (r#"쭼"#, r#"쭼"#, r#"쭼"#, r#"쭼"#, r#"쭼"#),
        (r#"쭽"#, r#"쭽"#, r#"쭽"#, r#"쭽"#, r#"쭽"#),
        (r#"쭾"#, r#"쭾"#, r#"쭾"#, r#"쭾"#, r#"쭾"#),
        (r#"쭿"#, r#"쭿"#, r#"쭿"#, r#"쭿"#, r#"쭿"#),
        (r#"쮀"#, r#"쮀"#, r#"쮀"#, r#"쮀"#, r#"쮀"#),
        (r#"쮁"#, r#"쮁"#, r#"쮁"#, r#"쮁"#, r#"쮁"#),
        (r#"쮂"#, r#"쮂"#, r#"쮂"#, r#"쮂"#, r#"쮂"#),
        (r#"쮃"#, r#"쮃"#, r#"쮃"#, r#"쮃"#, r#"쮃"#),
        (r#"쮄"#, r#"쮄"#, r#"쮄"#, r#"쮄"#, r#"쮄"#),
        (r#"쮅"#, r#"쮅"#, r#"쮅"#, r#"쮅"#, r#"쮅"#),
        (r#"쮆"#, r#"쮆"#, r#"쮆"#, r#"쮆"#, r#"쮆"#),
        (r#"쮇"#, r#"쮇"#, r#"쮇"#, r#"쮇"#, r#"쮇"#),
        (r#"쮈"#, r#"쮈"#, r#"쮈"#, r#"쮈"#, r#"쮈"#),
        (r#"쮉"#, r#"쮉"#, r#"쮉"#, r#"쮉"#, r#"쮉"#),
        (r#"쮊"#, r#"쮊"#, r#"쮊"#, r#"쮊"#, r#"쮊"#),
        (r#"쮋"#, r#"쮋"#, r#"쮋"#, r#"쮋"#, r#"쮋"#),
        (r#"쮌"#, r#"쮌"#, r#"쮌"#, r#"쮌"#, r#"쮌"#),
        (r#"쮍"#, r#"쮍"#, r#"쮍"#, r#"쮍"#, r#"쮍"#),
        (r#"쮎"#, r#"쮎"#, r#"쮎"#, r#"쮎"#, r#"쮎"#),
        (r#"쮏"#, r#"쮏"#, r#"쮏"#, r#"쮏"#, r#"쮏"#),
        (r#"쮐"#, r#"쮐"#, r#"쮐"#, r#"쮐"#, r#"쮐"#),
        (r#"쮑"#, r#"쮑"#, r#"쮑"#, r#"쮑"#, r#"쮑"#),
        (r#"쮒"#, r#"쮒"#, r#"쮒"#, r#"쮒"#, r#"쮒"#),
        (r#"쮓"#, r#"쮓"#, r#"쮓"#, r#"쮓"#, r#"쮓"#),
        (r#"쮔"#, r#"쮔"#, r#"쮔"#, r#"쮔"#, r#"쮔"#),
        (r#"쮕"#, r#"쮕"#, r#"쮕"#, r#"쮕"#, r#"쮕"#),
        (r#"쮖"#, r#"쮖"#, r#"쮖"#, r#"쮖"#, r#"쮖"#),
        (r#"쮗"#, r#"쮗"#, r#"쮗"#, r#"쮗"#, r#"쮗"#),
        (r#"쮘"#, r#"쮘"#, r#"쮘"#, r#"쮘"#, r#"쮘"#),
        (r#"쮙"#, r#"쮙"#, r#"쮙"#, r#"쮙"#, r#"쮙"#),
        (r#"쮚"#, r#"쮚"#, r#"쮚"#, r#"쮚"#, r#"쮚"#),
        (r#"쮛"#, r#"쮛"#, r#"쮛"#, r#"쮛"#, r#"쮛"#),
        (r#"쮜"#, r#"쮜"#, r#"쮜"#, r#"쮜"#, r#"쮜"#),
        (r#"쮝"#, r#"쮝"#, r#"쮝"#, r#"쮝"#, r#"쮝"#),
        (r#"쮞"#, r#"쮞"#, r#"쮞"#, r#"쮞"#, r#"쮞"#),
        (r#"쮟"#, r#"쮟"#, r#"쮟"#, r#"쮟"#, r#"쮟"#),
        (r#"쮠"#, r#"쮠"#, r#"쮠"#, r#"쮠"#, r#"쮠"#),
        (r#"쮡"#, r#"쮡"#, r#"쮡"#, r#"쮡"#, r#"쮡"#),
        (r#"쮢"#, r#"쮢"#, r#"쮢"#, r#"쮢"#, r#"쮢"#),
        (r#"쮣"#, r#"쮣"#, r#"쮣"#, r#"쮣"#, r#"쮣"#),
        (r#"쮤"#, r#"쮤"#, r#"쮤"#, r#"쮤"#, r#"쮤"#),
        (r#"쮥"#, r#"쮥"#, r#"쮥"#, r#"쮥"#, r#"쮥"#),
        (r#"쮦"#, r#"쮦"#, r#"쮦"#, r#"쮦"#, r#"쮦"#),
        (r#"쮧"#, r#"쮧"#, r#"쮧"#, r#"쮧"#, r#"쮧"#),
        (r#"쮨"#, r#"쮨"#, r#"쮨"#, r#"쮨"#, r#"쮨"#),
        (r#"쮩"#, r#"쮩"#, r#"쮩"#, r#"쮩"#, r#"쮩"#),
        (r#"쮪"#, r#"쮪"#, r#"쮪"#, r#"쮪"#, r#"쮪"#),
        (r#"쮫"#, r#"쮫"#, r#"쮫"#, r#"쮫"#, r#"쮫"#),
        (r#"쮬"#, r#"쮬"#, r#"쮬"#, r#"쮬"#, r#"쮬"#),
        (r#"쮭"#, r#"쮭"#, r#"쮭"#, r#"쮭"#, r#"쮭"#),
        (r#"쮮"#, r#"쮮"#, r#"쮮"#, r#"쮮"#, r#"쮮"#),
        (r#"쮯"#, r#"쮯"#, r#"쮯"#, r#"쮯"#, r#"쮯"#),
        (r#"쮰"#, r#"쮰"#, r#"쮰"#, r#"쮰"#, r#"쮰"#),
        (r#"쮱"#, r#"쮱"#, r#"쮱"#, r#"쮱"#, r#"쮱"#),
        (r#"쮲"#, r#"쮲"#, r#"쮲"#, r#"쮲"#, r#"쮲"#),
        (r#"쮳"#, r#"쮳"#, r#"쮳"#, r#"쮳"#, r#"쮳"#),
        (r#"쮴"#, r#"쮴"#, r#"쮴"#, r#"쮴"#, r#"쮴"#),
        (r#"쮵"#, r#"쮵"#, r#"쮵"#, r#"쮵"#, r#"쮵"#),
        (r#"쮶"#, r#"쮶"#, r#"쮶"#, r#"쮶"#, r#"쮶"#),
        (r#"쮷"#, r#"쮷"#, r#"쮷"#, r#"쮷"#, r#"쮷"#),
        (r#"쮸"#, r#"쮸"#, r#"쮸"#, r#"쮸"#, r#"쮸"#),
        (r#"쮹"#, r#"쮹"#, r#"쮹"#, r#"쮹"#, r#"쮹"#),
        (r#"쮺"#, r#"쮺"#, r#"쮺"#, r#"쮺"#, r#"쮺"#),
        (r#"쮻"#, r#"쮻"#, r#"쮻"#, r#"쮻"#, r#"쮻"#),
        (r#"쮼"#, r#"쮼"#, r#"쮼"#, r#"쮼"#, r#"쮼"#),
        (r#"쮽"#, r#"쮽"#, r#"쮽"#, r#"쮽"#, r#"쮽"#),
        (r#"쮾"#, r#"쮾"#, r#"쮾"#, r#"쮾"#, r#"쮾"#),
        (r#"쮿"#, r#"쮿"#, r#"쮿"#, r#"쮿"#, r#"쮿"#),
        (r#"쯀"#, r#"쯀"#, r#"쯀"#, r#"쯀"#, r#"쯀"#),
        (r#"쯁"#, r#"쯁"#, r#"쯁"#, r#"쯁"#, r#"쯁"#),
        (r#"쯂"#, r#"쯂"#, r#"쯂"#, r#"쯂"#, r#"쯂"#),
        (r#"쯃"#, r#"쯃"#, r#"쯃"#, r#"쯃"#, r#"쯃"#),
        (r#"쯄"#, r#"쯄"#, r#"쯄"#, r#"쯄"#, r#"쯄"#),
        (r#"쯅"#, r#"쯅"#, r#"쯅"#, r#"쯅"#, r#"쯅"#),
        (r#"쯆"#, r#"쯆"#, r#"쯆"#, r#"쯆"#, r#"쯆"#),
        (r#"쯇"#, r#"쯇"#, r#"쯇"#, r#"쯇"#, r#"쯇"#),
        (r#"쯈"#, r#"쯈"#, r#"쯈"#, r#"쯈"#, r#"쯈"#),
        (r#"쯉"#, r#"쯉"#, r#"쯉"#, r#"쯉"#, r#"쯉"#),
        (r#"쯊"#, r#"쯊"#, r#"쯊"#, r#"쯊"#, r#"쯊"#),
        (r#"쯋"#, r#"쯋"#, r#"쯋"#, r#"쯋"#, r#"쯋"#),
        (r#"쯌"#, r#"쯌"#, r#"쯌"#, r#"쯌"#, r#"쯌"#),
        (r#"쯍"#, r#"쯍"#, r#"쯍"#, r#"쯍"#, r#"쯍"#),
        (r#"쯎"#, r#"쯎"#, r#"쯎"#, r#"쯎"#, r#"쯎"#),
        (r#"쯏"#, r#"쯏"#, r#"쯏"#, r#"쯏"#, r#"쯏"#),
        (r#"쯐"#, r#"쯐"#, r#"쯐"#, r#"쯐"#, r#"쯐"#),
        (r#"쯑"#, r#"쯑"#, r#"쯑"#, r#"쯑"#, r#"쯑"#),
        (r#"쯒"#, r#"쯒"#, r#"쯒"#, r#"쯒"#, r#"쯒"#),
        (r#"쯓"#, r#"쯓"#, r#"쯓"#, r#"쯓"#, r#"쯓"#),
        (r#"쯔"#, r#"쯔"#, r#"쯔"#, r#"쯔"#, r#"쯔"#),
        (r#"쯕"#, r#"쯕"#, r#"쯕"#, r#"쯕"#, r#"쯕"#),
        (r#"쯖"#, r#"쯖"#, r#"쯖"#, r#"쯖"#, r#"쯖"#),
        (r#"쯗"#, r#"쯗"#, r#"쯗"#, r#"쯗"#, r#"쯗"#),
        (r#"쯘"#, r#"쯘"#, r#"쯘"#, r#"쯘"#, r#"쯘"#),
        (r#"쯙"#, r#"쯙"#, r#"쯙"#, r#"쯙"#, r#"쯙"#),
        (r#"쯚"#, r#"쯚"#, r#"쯚"#, r#"쯚"#, r#"쯚"#),
        (r#"쯛"#, r#"쯛"#, r#"쯛"#, r#"쯛"#, r#"쯛"#),
        (r#"쯜"#, r#"쯜"#, r#"쯜"#, r#"쯜"#, r#"쯜"#),
        (r#"쯝"#, r#"쯝"#, r#"쯝"#, r#"쯝"#, r#"쯝"#),
        (r#"쯞"#, r#"쯞"#, r#"쯞"#, r#"쯞"#, r#"쯞"#),
        (r#"쯟"#, r#"쯟"#, r#"쯟"#, r#"쯟"#, r#"쯟"#),
        (r#"쯠"#, r#"쯠"#, r#"쯠"#, r#"쯠"#, r#"쯠"#),
        (r#"쯡"#, r#"쯡"#, r#"쯡"#, r#"쯡"#, r#"쯡"#),
        (r#"쯢"#, r#"쯢"#, r#"쯢"#, r#"쯢"#, r#"쯢"#),
        (r#"쯣"#, r#"쯣"#, r#"쯣"#, r#"쯣"#, r#"쯣"#),
        (r#"쯤"#, r#"쯤"#, r#"쯤"#, r#"쯤"#, r#"쯤"#),
        (r#"쯥"#, r#"쯥"#, r#"쯥"#, r#"쯥"#, r#"쯥"#),
        (r#"쯦"#, r#"쯦"#, r#"쯦"#, r#"쯦"#, r#"쯦"#),
        (r#"쯧"#, r#"쯧"#, r#"쯧"#, r#"쯧"#, r#"쯧"#),
        (r#"쯨"#, r#"쯨"#, r#"쯨"#, r#"쯨"#, r#"쯨"#),
        (r#"쯩"#, r#"쯩"#, r#"쯩"#, r#"쯩"#, r#"쯩"#),
        (r#"쯪"#, r#"쯪"#, r#"쯪"#, r#"쯪"#, r#"쯪"#),
        (r#"쯫"#, r#"쯫"#, r#"쯫"#, r#"쯫"#, r#"쯫"#),
        (r#"쯬"#, r#"쯬"#, r#"쯬"#, r#"쯬"#, r#"쯬"#),
        (r#"쯭"#, r#"쯭"#, r#"쯭"#, r#"쯭"#, r#"쯭"#),
        (r#"쯮"#, r#"쯮"#, r#"쯮"#, r#"쯮"#, r#"쯮"#),
        (r#"쯯"#, r#"쯯"#, r#"쯯"#, r#"쯯"#, r#"쯯"#),
        (r#"쯰"#, r#"쯰"#, r#"쯰"#, r#"쯰"#, r#"쯰"#),
        (r#"쯱"#, r#"쯱"#, r#"쯱"#, r#"쯱"#, r#"쯱"#),
        (r#"쯲"#, r#"쯲"#, r#"쯲"#, r#"쯲"#, r#"쯲"#),
        (r#"쯳"#, r#"쯳"#, r#"쯳"#, r#"쯳"#, r#"쯳"#),
        (r#"쯴"#, r#"쯴"#, r#"쯴"#, r#"쯴"#, r#"쯴"#),
        (r#"쯵"#, r#"쯵"#, r#"쯵"#, r#"쯵"#, r#"쯵"#),
        (r#"쯶"#, r#"쯶"#, r#"쯶"#, r#"쯶"#, r#"쯶"#),
        (r#"쯷"#, r#"쯷"#, r#"쯷"#, r#"쯷"#, r#"쯷"#),
        (r#"쯸"#, r#"쯸"#, r#"쯸"#, r#"쯸"#, r#"쯸"#),
        (r#"쯹"#, r#"쯹"#, r#"쯹"#, r#"쯹"#, r#"쯹"#),
        (r#"쯺"#, r#"쯺"#, r#"쯺"#, r#"쯺"#, r#"쯺"#),
        (r#"쯻"#, r#"쯻"#, r#"쯻"#, r#"쯻"#, r#"쯻"#),
        (r#"쯼"#, r#"쯼"#, r#"쯼"#, r#"쯼"#, r#"쯼"#),
        (r#"쯽"#, r#"쯽"#, r#"쯽"#, r#"쯽"#, r#"쯽"#),
        (r#"쯾"#, r#"쯾"#, r#"쯾"#, r#"쯾"#, r#"쯾"#),
        (r#"쯿"#, r#"쯿"#, r#"쯿"#, r#"쯿"#, r#"쯿"#),
        (r#"찀"#, r#"찀"#, r#"찀"#, r#"찀"#, r#"찀"#),
        (r#"찁"#, r#"찁"#, r#"찁"#, r#"찁"#, r#"찁"#),
        (r#"찂"#, r#"찂"#, r#"찂"#, r#"찂"#, r#"찂"#),
        (r#"찃"#, r#"찃"#, r#"찃"#, r#"찃"#, r#"찃"#),
        (r#"찄"#, r#"찄"#, r#"찄"#, r#"찄"#, r#"찄"#),
        (r#"찅"#, r#"찅"#, r#"찅"#, r#"찅"#, r#"찅"#),
        (r#"찆"#, r#"찆"#, r#"찆"#, r#"찆"#, r#"찆"#),
        (r#"찇"#, r#"찇"#, r#"찇"#, r#"찇"#, r#"찇"#),
        (r#"찈"#, r#"찈"#, r#"찈"#, r#"찈"#, r#"찈"#),
        (r#"찉"#, r#"찉"#, r#"찉"#, r#"찉"#, r#"찉"#),
        (r#"찊"#, r#"찊"#, r#"찊"#, r#"찊"#, r#"찊"#),
        (r#"찋"#, r#"찋"#, r#"찋"#, r#"찋"#, r#"찋"#),
        (r#"찌"#, r#"찌"#, r#"찌"#, r#"찌"#, r#"찌"#),
        (r#"찍"#, r#"찍"#, r#"찍"#, r#"찍"#, r#"찍"#),
        (r#"찎"#, r#"찎"#, r#"찎"#, r#"찎"#, r#"찎"#),
        (r#"찏"#, r#"찏"#, r#"찏"#, r#"찏"#, r#"찏"#),
        (r#"찐"#, r#"찐"#, r#"찐"#, r#"찐"#, r#"찐"#),
        (r#"찑"#, r#"찑"#, r#"찑"#, r#"찑"#, r#"찑"#),
        (r#"찒"#, r#"찒"#, r#"찒"#, r#"찒"#, r#"찒"#),
        (r#"찓"#, r#"찓"#, r#"찓"#, r#"찓"#, r#"찓"#),
        (r#"찔"#, r#"찔"#, r#"찔"#, r#"찔"#, r#"찔"#),
        (r#"찕"#, r#"찕"#, r#"찕"#, r#"찕"#, r#"찕"#),
        (r#"찖"#, r#"찖"#, r#"찖"#, r#"찖"#, r#"찖"#),
        (r#"찗"#, r#"찗"#, r#"찗"#, r#"찗"#, r#"찗"#),
        (r#"찘"#, r#"찘"#, r#"찘"#, r#"찘"#, r#"찘"#),
        (r#"찙"#, r#"찙"#, r#"찙"#, r#"찙"#, r#"찙"#),
        (r#"찚"#, r#"찚"#, r#"찚"#, r#"찚"#, r#"찚"#),
        (r#"찛"#, r#"찛"#, r#"찛"#, r#"찛"#, r#"찛"#),
        (r#"찜"#, r#"찜"#, r#"찜"#, r#"찜"#, r#"찜"#),
        (r#"찝"#, r#"찝"#, r#"찝"#, r#"찝"#, r#"찝"#),
        (r#"찞"#, r#"찞"#, r#"찞"#, r#"찞"#, r#"찞"#),
        (r#"찟"#, r#"찟"#, r#"찟"#, r#"찟"#, r#"찟"#),
        (r#"찠"#, r#"찠"#, r#"찠"#, r#"찠"#, r#"찠"#),
        (r#"찡"#, r#"찡"#, r#"찡"#, r#"찡"#, r#"찡"#),
        (r#"찢"#, r#"찢"#, r#"찢"#, r#"찢"#, r#"찢"#),
        (r#"찣"#, r#"찣"#, r#"찣"#, r#"찣"#, r#"찣"#),
        (r#"찤"#, r#"찤"#, r#"찤"#, r#"찤"#, r#"찤"#),
        (r#"찥"#, r#"찥"#, r#"찥"#, r#"찥"#, r#"찥"#),
        (r#"찦"#, r#"찦"#, r#"찦"#, r#"찦"#, r#"찦"#),
        (r#"찧"#, r#"찧"#, r#"찧"#, r#"찧"#, r#"찧"#),
        (r#"차"#, r#"차"#, r#"차"#, r#"차"#, r#"차"#),
        (r#"착"#, r#"착"#, r#"착"#, r#"착"#, r#"착"#),
        (r#"찪"#, r#"찪"#, r#"찪"#, r#"찪"#, r#"찪"#),
        (r#"찫"#, r#"찫"#, r#"찫"#, r#"찫"#, r#"찫"#),
        (r#"찬"#, r#"찬"#, r#"찬"#, r#"찬"#, r#"찬"#),
        (r#"찭"#, r#"찭"#, r#"찭"#, r#"찭"#, r#"찭"#),
        (r#"찮"#, r#"찮"#, r#"찮"#, r#"찮"#, r#"찮"#),
        (r#"찯"#, r#"찯"#, r#"찯"#, r#"찯"#, r#"찯"#),
        (r#"찰"#, r#"찰"#, r#"찰"#, r#"찰"#, r#"찰"#),
        (r#"찱"#, r#"찱"#, r#"찱"#, r#"찱"#, r#"찱"#),
        (r#"찲"#, r#"찲"#, r#"찲"#, r#"찲"#, r#"찲"#),
        (r#"찳"#, r#"찳"#, r#"찳"#, r#"찳"#, r#"찳"#),
        (r#"찴"#, r#"찴"#, r#"찴"#, r#"찴"#, r#"찴"#),
        (r#"찵"#, r#"찵"#, r#"찵"#, r#"찵"#, r#"찵"#),
        (r#"찶"#, r#"찶"#, r#"찶"#, r#"찶"#, r#"찶"#),
        (r#"찷"#, r#"찷"#, r#"찷"#, r#"찷"#, r#"찷"#),
        (r#"참"#, r#"참"#, r#"참"#, r#"참"#, r#"참"#),
        (r#"찹"#, r#"찹"#, r#"찹"#, r#"찹"#, r#"찹"#),
        (r#"찺"#, r#"찺"#, r#"찺"#, r#"찺"#, r#"찺"#),
        (r#"찻"#, r#"찻"#, r#"찻"#, r#"찻"#, r#"찻"#),
        (r#"찼"#, r#"찼"#, r#"찼"#, r#"찼"#, r#"찼"#),
        (r#"창"#, r#"창"#, r#"창"#, r#"창"#, r#"창"#),
        (r#"찾"#, r#"찾"#, r#"찾"#, r#"찾"#, r#"찾"#),
        (r#"찿"#, r#"찿"#, r#"찿"#, r#"찿"#, r#"찿"#),
        (r#"챀"#, r#"챀"#, r#"챀"#, r#"챀"#, r#"챀"#),
        (r#"챁"#, r#"챁"#, r#"챁"#, r#"챁"#, r#"챁"#),
        (r#"챂"#, r#"챂"#, r#"챂"#, r#"챂"#, r#"챂"#),
        (r#"챃"#, r#"챃"#, r#"챃"#, r#"챃"#, r#"챃"#),
        (r#"채"#, r#"채"#, r#"채"#, r#"채"#, r#"채"#),
        (r#"책"#, r#"책"#, r#"책"#, r#"책"#, r#"책"#),
        (r#"챆"#, r#"챆"#, r#"챆"#, r#"챆"#, r#"챆"#),
        (r#"챇"#, r#"챇"#, r#"챇"#, r#"챇"#, r#"챇"#),
        (r#"챈"#, r#"챈"#, r#"챈"#, r#"챈"#, r#"챈"#),
        (r#"챉"#, r#"챉"#, r#"챉"#, r#"챉"#, r#"챉"#),
        (r#"챊"#, r#"챊"#, r#"챊"#, r#"챊"#, r#"챊"#),
        (r#"챋"#, r#"챋"#, r#"챋"#, r#"챋"#, r#"챋"#),
        (r#"챌"#, r#"챌"#, r#"챌"#, r#"챌"#, r#"챌"#),
        (r#"챍"#, r#"챍"#, r#"챍"#, r#"챍"#, r#"챍"#),
        (r#"챎"#, r#"챎"#, r#"챎"#, r#"챎"#, r#"챎"#),
        (r#"챏"#, r#"챏"#, r#"챏"#, r#"챏"#, r#"챏"#),
        (r#"챐"#, r#"챐"#, r#"챐"#, r#"챐"#, r#"챐"#),
        (r#"챑"#, r#"챑"#, r#"챑"#, r#"챑"#, r#"챑"#),
        (r#"챒"#, r#"챒"#, r#"챒"#, r#"챒"#, r#"챒"#),
        (r#"챓"#, r#"챓"#, r#"챓"#, r#"챓"#, r#"챓"#),
        (r#"챔"#, r#"챔"#, r#"챔"#, r#"챔"#, r#"챔"#),
        (r#"챕"#, r#"챕"#, r#"챕"#, r#"챕"#, r#"챕"#),
        (r#"챖"#, r#"챖"#, r#"챖"#, r#"챖"#, r#"챖"#),
        (r#"챗"#, r#"챗"#, r#"챗"#, r#"챗"#, r#"챗"#),
        (r#"챘"#, r#"챘"#, r#"챘"#, r#"챘"#, r#"챘"#),
        (r#"챙"#, r#"챙"#, r#"챙"#, r#"챙"#, r#"챙"#),
        (r#"챚"#, r#"챚"#, r#"챚"#, r#"챚"#, r#"챚"#),
        (r#"챛"#, r#"챛"#, r#"챛"#, r#"챛"#, r#"챛"#),
        (r#"챜"#, r#"챜"#, r#"챜"#, r#"챜"#, r#"챜"#),
        (r#"챝"#, r#"챝"#, r#"챝"#, r#"챝"#, r#"챝"#),
        (r#"챞"#, r#"챞"#, r#"챞"#, r#"챞"#, r#"챞"#),
        (r#"챟"#, r#"챟"#, r#"챟"#, r#"챟"#, r#"챟"#),
        (r#"챠"#, r#"챠"#, r#"챠"#, r#"챠"#, r#"챠"#),
        (r#"챡"#, r#"챡"#, r#"챡"#, r#"챡"#, r#"챡"#),
        (r#"챢"#, r#"챢"#, r#"챢"#, r#"챢"#, r#"챢"#),
        (r#"챣"#, r#"챣"#, r#"챣"#, r#"챣"#, r#"챣"#),
        (r#"챤"#, r#"챤"#, r#"챤"#, r#"챤"#, r#"챤"#),
        (r#"챥"#, r#"챥"#, r#"챥"#, r#"챥"#, r#"챥"#),
        (r#"챦"#, r#"챦"#, r#"챦"#, r#"챦"#, r#"챦"#),
        (r#"챧"#, r#"챧"#, r#"챧"#, r#"챧"#, r#"챧"#),
        (r#"챨"#, r#"챨"#, r#"챨"#, r#"챨"#, r#"챨"#),
        (r#"챩"#, r#"챩"#, r#"챩"#, r#"챩"#, r#"챩"#),
        (r#"챪"#, r#"챪"#, r#"챪"#, r#"챪"#, r#"챪"#),
        (r#"챫"#, r#"챫"#, r#"챫"#, r#"챫"#, r#"챫"#),
        (r#"챬"#, r#"챬"#, r#"챬"#, r#"챬"#, r#"챬"#),
        (r#"챭"#, r#"챭"#, r#"챭"#, r#"챭"#, r#"챭"#),
        (r#"챮"#, r#"챮"#, r#"챮"#, r#"챮"#, r#"챮"#),
        (r#"챯"#, r#"챯"#, r#"챯"#, r#"챯"#, r#"챯"#),
        (r#"챰"#, r#"챰"#, r#"챰"#, r#"챰"#, r#"챰"#),
        (r#"챱"#, r#"챱"#, r#"챱"#, r#"챱"#, r#"챱"#),
        (r#"챲"#, r#"챲"#, r#"챲"#, r#"챲"#, r#"챲"#),
        (r#"챳"#, r#"챳"#, r#"챳"#, r#"챳"#, r#"챳"#),
        (r#"챴"#, r#"챴"#, r#"챴"#, r#"챴"#, r#"챴"#),
        (r#"챵"#, r#"챵"#, r#"챵"#, r#"챵"#, r#"챵"#),
        (r#"챶"#, r#"챶"#, r#"챶"#, r#"챶"#, r#"챶"#),
        (r#"챷"#, r#"챷"#, r#"챷"#, r#"챷"#, r#"챷"#),
        (r#"챸"#, r#"챸"#, r#"챸"#, r#"챸"#, r#"챸"#),
        (r#"챹"#, r#"챹"#, r#"챹"#, r#"챹"#, r#"챹"#),
        (r#"챺"#, r#"챺"#, r#"챺"#, r#"챺"#, r#"챺"#),
        (r#"챻"#, r#"챻"#, r#"챻"#, r#"챻"#, r#"챻"#),
        (r#"챼"#, r#"챼"#, r#"챼"#, r#"챼"#, r#"챼"#),
        (r#"챽"#, r#"챽"#, r#"챽"#, r#"챽"#, r#"챽"#),
        (r#"챾"#, r#"챾"#, r#"챾"#, r#"챾"#, r#"챾"#),
        (r#"챿"#, r#"챿"#, r#"챿"#, r#"챿"#, r#"챿"#),
        (r#"첀"#, r#"첀"#, r#"첀"#, r#"첀"#, r#"첀"#),
        (r#"첁"#, r#"첁"#, r#"첁"#, r#"첁"#, r#"첁"#),
        (r#"첂"#, r#"첂"#, r#"첂"#, r#"첂"#, r#"첂"#),
        (r#"첃"#, r#"첃"#, r#"첃"#, r#"첃"#, r#"첃"#),
        (r#"첄"#, r#"첄"#, r#"첄"#, r#"첄"#, r#"첄"#),
        (r#"첅"#, r#"첅"#, r#"첅"#, r#"첅"#, r#"첅"#),
        (r#"첆"#, r#"첆"#, r#"첆"#, r#"첆"#, r#"첆"#),
        (r#"첇"#, r#"첇"#, r#"첇"#, r#"첇"#, r#"첇"#),
        (r#"첈"#, r#"첈"#, r#"첈"#, r#"첈"#, r#"첈"#),
        (r#"첉"#, r#"첉"#, r#"첉"#, r#"첉"#, r#"첉"#),
        (r#"첊"#, r#"첊"#, r#"첊"#, r#"첊"#, r#"첊"#),
        (r#"첋"#, r#"첋"#, r#"첋"#, r#"첋"#, r#"첋"#),
        (r#"첌"#, r#"첌"#, r#"첌"#, r#"첌"#, r#"첌"#),
        (r#"첍"#, r#"첍"#, r#"첍"#, r#"첍"#, r#"첍"#),
        (r#"첎"#, r#"첎"#, r#"첎"#, r#"첎"#, r#"첎"#),
        (r#"첏"#, r#"첏"#, r#"첏"#, r#"첏"#, r#"첏"#),
        (r#"첐"#, r#"첐"#, r#"첐"#, r#"첐"#, r#"첐"#),
        (r#"첑"#, r#"첑"#, r#"첑"#, r#"첑"#, r#"첑"#),
        (r#"첒"#, r#"첒"#, r#"첒"#, r#"첒"#, r#"첒"#),
        (r#"첓"#, r#"첓"#, r#"첓"#, r#"첓"#, r#"첓"#),
        (r#"첔"#, r#"첔"#, r#"첔"#, r#"첔"#, r#"첔"#),
        (r#"첕"#, r#"첕"#, r#"첕"#, r#"첕"#, r#"첕"#),
        (r#"첖"#, r#"첖"#, r#"첖"#, r#"첖"#, r#"첖"#),
        (r#"첗"#, r#"첗"#, r#"첗"#, r#"첗"#, r#"첗"#),
        (r#"처"#, r#"처"#, r#"처"#, r#"처"#, r#"처"#),
        (r#"척"#, r#"척"#, r#"척"#, r#"척"#, r#"척"#),
        (r#"첚"#, r#"첚"#, r#"첚"#, r#"첚"#, r#"첚"#),
        (r#"첛"#, r#"첛"#, r#"첛"#, r#"첛"#, r#"첛"#),
        (r#"천"#, r#"천"#, r#"천"#, r#"천"#, r#"천"#),
        (r#"첝"#, r#"첝"#, r#"첝"#, r#"첝"#, r#"첝"#),
        (r#"첞"#, r#"첞"#, r#"첞"#, r#"첞"#, r#"첞"#),
        (r#"첟"#, r#"첟"#, r#"첟"#, r#"첟"#, r#"첟"#),
        (r#"철"#, r#"철"#, r#"철"#, r#"철"#, r#"철"#),
        (r#"첡"#, r#"첡"#, r#"첡"#, r#"첡"#, r#"첡"#),
        (r#"첢"#, r#"첢"#, r#"첢"#, r#"첢"#, r#"첢"#),
        (r#"첣"#, r#"첣"#, r#"첣"#, r#"첣"#, r#"첣"#),
        (r#"첤"#, r#"첤"#, r#"첤"#, r#"첤"#, r#"첤"#),
        (r#"첥"#, r#"첥"#, r#"첥"#, r#"첥"#, r#"첥"#),
        (r#"첦"#, r#"첦"#, r#"첦"#, r#"첦"#, r#"첦"#),
        (r#"첧"#, r#"첧"#, r#"첧"#, r#"첧"#, r#"첧"#),
        (r#"첨"#, r#"첨"#, r#"첨"#, r#"첨"#, r#"첨"#),
        (r#"첩"#, r#"첩"#, r#"첩"#, r#"첩"#, r#"첩"#),
        (r#"첪"#, r#"첪"#, r#"첪"#, r#"첪"#, r#"첪"#),
        (r#"첫"#, r#"첫"#, r#"첫"#, r#"첫"#, r#"첫"#),
        (r#"첬"#, r#"첬"#, r#"첬"#, r#"첬"#, r#"첬"#),
        (r#"청"#, r#"청"#, r#"청"#, r#"청"#, r#"청"#),
        (r#"첮"#, r#"첮"#, r#"첮"#, r#"첮"#, r#"첮"#),
        (r#"첯"#, r#"첯"#, r#"첯"#, r#"첯"#, r#"첯"#),
        (r#"첰"#, r#"첰"#, r#"첰"#, r#"첰"#, r#"첰"#),
        (r#"첱"#, r#"첱"#, r#"첱"#, r#"첱"#, r#"첱"#),
        (r#"첲"#, r#"첲"#, r#"첲"#, r#"첲"#, r#"첲"#),
        (r#"첳"#, r#"첳"#, r#"첳"#, r#"첳"#, r#"첳"#),
        (r#"체"#, r#"체"#, r#"체"#, r#"체"#, r#"체"#),
        (r#"첵"#, r#"첵"#, r#"첵"#, r#"첵"#, r#"첵"#),
        (r#"첶"#, r#"첶"#, r#"첶"#, r#"첶"#, r#"첶"#),
        (r#"첷"#, r#"첷"#, r#"첷"#, r#"첷"#, r#"첷"#),
        (r#"첸"#, r#"첸"#, r#"첸"#, r#"첸"#, r#"첸"#),
        (r#"첹"#, r#"첹"#, r#"첹"#, r#"첹"#, r#"첹"#),
        (r#"첺"#, r#"첺"#, r#"첺"#, r#"첺"#, r#"첺"#),
        (r#"첻"#, r#"첻"#, r#"첻"#, r#"첻"#, r#"첻"#),
        (r#"첼"#, r#"첼"#, r#"첼"#, r#"첼"#, r#"첼"#),
        (r#"첽"#, r#"첽"#, r#"첽"#, r#"첽"#, r#"첽"#),
        (r#"첾"#, r#"첾"#, r#"첾"#, r#"첾"#, r#"첾"#),
        (r#"첿"#, r#"첿"#, r#"첿"#, r#"첿"#, r#"첿"#),
        (r#"쳀"#, r#"쳀"#, r#"쳀"#, r#"쳀"#, r#"쳀"#),
        (r#"쳁"#, r#"쳁"#, r#"쳁"#, r#"쳁"#, r#"쳁"#),
        (r#"쳂"#, r#"쳂"#, r#"쳂"#, r#"쳂"#, r#"쳂"#),
        (r#"쳃"#, r#"쳃"#, r#"쳃"#, r#"쳃"#, r#"쳃"#),
        (r#"쳄"#, r#"쳄"#, r#"쳄"#, r#"쳄"#, r#"쳄"#),
        (r#"쳅"#, r#"쳅"#, r#"쳅"#, r#"쳅"#, r#"쳅"#),
        (r#"쳆"#, r#"쳆"#, r#"쳆"#, r#"쳆"#, r#"쳆"#),
        (r#"쳇"#, r#"쳇"#, r#"쳇"#, r#"쳇"#, r#"쳇"#),
        (r#"쳈"#, r#"쳈"#, r#"쳈"#, r#"쳈"#, r#"쳈"#),
        (r#"쳉"#, r#"쳉"#, r#"쳉"#, r#"쳉"#, r#"쳉"#),
        (r#"쳊"#, r#"쳊"#, r#"쳊"#, r#"쳊"#, r#"쳊"#),
        (r#"쳋"#, r#"쳋"#, r#"쳋"#, r#"쳋"#, r#"쳋"#),
        (r#"쳌"#, r#"쳌"#, r#"쳌"#, r#"쳌"#, r#"쳌"#),
        (r#"쳍"#, r#"쳍"#, r#"쳍"#, r#"쳍"#, r#"쳍"#),
        (r#"쳎"#, r#"쳎"#, r#"쳎"#, r#"쳎"#, r#"쳎"#),
        (r#"쳏"#, r#"쳏"#, r#"쳏"#, r#"쳏"#, r#"쳏"#),
        (r#"쳐"#, r#"쳐"#, r#"쳐"#, r#"쳐"#, r#"쳐"#),
        (r#"쳑"#, r#"쳑"#, r#"쳑"#, r#"쳑"#, r#"쳑"#),
        (r#"쳒"#, r#"쳒"#, r#"쳒"#, r#"쳒"#, r#"쳒"#),
        (r#"쳓"#, r#"쳓"#, r#"쳓"#, r#"쳓"#, r#"쳓"#),
        (r#"쳔"#, r#"쳔"#, r#"쳔"#, r#"쳔"#, r#"쳔"#),
        (r#"쳕"#, r#"쳕"#, r#"쳕"#, r#"쳕"#, r#"쳕"#),
        (r#"쳖"#, r#"쳖"#, r#"쳖"#, r#"쳖"#, r#"쳖"#),
        (r#"쳗"#, r#"쳗"#, r#"쳗"#, r#"쳗"#, r#"쳗"#),
        (r#"쳘"#, r#"쳘"#, r#"쳘"#, r#"쳘"#, r#"쳘"#),
        (r#"쳙"#, r#"쳙"#, r#"쳙"#, r#"쳙"#, r#"쳙"#),
        (r#"쳚"#, r#"쳚"#, r#"쳚"#, r#"쳚"#, r#"쳚"#),
        (r#"쳛"#, r#"쳛"#, r#"쳛"#, r#"쳛"#, r#"쳛"#),
        (r#"쳜"#, r#"쳜"#, r#"쳜"#, r#"쳜"#, r#"쳜"#),
        (r#"쳝"#, r#"쳝"#, r#"쳝"#, r#"쳝"#, r#"쳝"#),
        (r#"쳞"#, r#"쳞"#, r#"쳞"#, r#"쳞"#, r#"쳞"#),
        (r#"쳟"#, r#"쳟"#, r#"쳟"#, r#"쳟"#, r#"쳟"#),
        (r#"쳠"#, r#"쳠"#, r#"쳠"#, r#"쳠"#, r#"쳠"#),
        (r#"쳡"#, r#"쳡"#, r#"쳡"#, r#"쳡"#, r#"쳡"#),
        (r#"쳢"#, r#"쳢"#, r#"쳢"#, r#"쳢"#, r#"쳢"#),
        (r#"쳣"#, r#"쳣"#, r#"쳣"#, r#"쳣"#, r#"쳣"#),
        (r#"쳤"#, r#"쳤"#, r#"쳤"#, r#"쳤"#, r#"쳤"#),
        (r#"쳥"#, r#"쳥"#, r#"쳥"#, r#"쳥"#, r#"쳥"#),
        (r#"쳦"#, r#"쳦"#, r#"쳦"#, r#"쳦"#, r#"쳦"#),
        (r#"쳧"#, r#"쳧"#, r#"쳧"#, r#"쳧"#, r#"쳧"#),
        (r#"쳨"#, r#"쳨"#, r#"쳨"#, r#"쳨"#, r#"쳨"#),
        (r#"쳩"#, r#"쳩"#, r#"쳩"#, r#"쳩"#, r#"쳩"#),
        (r#"쳪"#, r#"쳪"#, r#"쳪"#, r#"쳪"#, r#"쳪"#),
        (r#"쳫"#, r#"쳫"#, r#"쳫"#, r#"쳫"#, r#"쳫"#),
        (r#"쳬"#, r#"쳬"#, r#"쳬"#, r#"쳬"#, r#"쳬"#),
        (r#"쳭"#, r#"쳭"#, r#"쳭"#, r#"쳭"#, r#"쳭"#),
        (r#"쳮"#, r#"쳮"#, r#"쳮"#, r#"쳮"#, r#"쳮"#),
        (r#"쳯"#, r#"쳯"#, r#"쳯"#, r#"쳯"#, r#"쳯"#),
        (r#"쳰"#, r#"쳰"#, r#"쳰"#, r#"쳰"#, r#"쳰"#),
        (r#"쳱"#, r#"쳱"#, r#"쳱"#, r#"쳱"#, r#"쳱"#),
        (r#"쳲"#, r#"쳲"#, r#"쳲"#, r#"쳲"#, r#"쳲"#),
        (r#"쳳"#, r#"쳳"#, r#"쳳"#, r#"쳳"#, r#"쳳"#),
        (r#"쳴"#, r#"쳴"#, r#"쳴"#, r#"쳴"#, r#"쳴"#),
        (r#"쳵"#, r#"쳵"#, r#"쳵"#, r#"쳵"#, r#"쳵"#),
        (r#"쳶"#, r#"쳶"#, r#"쳶"#, r#"쳶"#, r#"쳶"#),
        (r#"쳷"#, r#"쳷"#, r#"쳷"#, r#"쳷"#, r#"쳷"#),
        (r#"쳸"#, r#"쳸"#, r#"쳸"#, r#"쳸"#, r#"쳸"#),
        (r#"쳹"#, r#"쳹"#, r#"쳹"#, r#"쳹"#, r#"쳹"#),
        (r#"쳺"#, r#"쳺"#, r#"쳺"#, r#"쳺"#, r#"쳺"#),
        (r#"쳻"#, r#"쳻"#, r#"쳻"#, r#"쳻"#, r#"쳻"#),
        (r#"쳼"#, r#"쳼"#, r#"쳼"#, r#"쳼"#, r#"쳼"#),
        (r#"쳽"#, r#"쳽"#, r#"쳽"#, r#"쳽"#, r#"쳽"#),
        (r#"쳾"#, r#"쳾"#, r#"쳾"#, r#"쳾"#, r#"쳾"#),
        (r#"쳿"#, r#"쳿"#, r#"쳿"#, r#"쳿"#, r#"쳿"#),
        (r#"촀"#, r#"촀"#, r#"촀"#, r#"촀"#, r#"촀"#),
        (r#"촁"#, r#"촁"#, r#"촁"#, r#"촁"#, r#"촁"#),
        (r#"촂"#, r#"촂"#, r#"촂"#, r#"촂"#, r#"촂"#),
        (r#"촃"#, r#"촃"#, r#"촃"#, r#"촃"#, r#"촃"#),
        (r#"촄"#, r#"촄"#, r#"촄"#, r#"촄"#, r#"촄"#),
        (r#"촅"#, r#"촅"#, r#"촅"#, r#"촅"#, r#"촅"#),
        (r#"촆"#, r#"촆"#, r#"촆"#, r#"촆"#, r#"촆"#),
        (r#"촇"#, r#"촇"#, r#"촇"#, r#"촇"#, r#"촇"#),
        (r#"초"#, r#"초"#, r#"초"#, r#"초"#, r#"초"#),
        (r#"촉"#, r#"촉"#, r#"촉"#, r#"촉"#, r#"촉"#),
        (r#"촊"#, r#"촊"#, r#"촊"#, r#"촊"#, r#"촊"#),
        (r#"촋"#, r#"촋"#, r#"촋"#, r#"촋"#, r#"촋"#),
        (r#"촌"#, r#"촌"#, r#"촌"#, r#"촌"#, r#"촌"#),
        (r#"촍"#, r#"촍"#, r#"촍"#, r#"촍"#, r#"촍"#),
        (r#"촎"#, r#"촎"#, r#"촎"#, r#"촎"#, r#"촎"#),
        (r#"촏"#, r#"촏"#, r#"촏"#, r#"촏"#, r#"촏"#),
        (r#"촐"#, r#"촐"#, r#"촐"#, r#"촐"#, r#"촐"#),
        (r#"촑"#, r#"촑"#, r#"촑"#, r#"촑"#, r#"촑"#),
        (r#"촒"#, r#"촒"#, r#"촒"#, r#"촒"#, r#"촒"#),
        (r#"촓"#, r#"촓"#, r#"촓"#, r#"촓"#, r#"촓"#),
        (r#"촔"#, r#"촔"#, r#"촔"#, r#"촔"#, r#"촔"#),
        (r#"촕"#, r#"촕"#, r#"촕"#, r#"촕"#, r#"촕"#),
        (r#"촖"#, r#"촖"#, r#"촖"#, r#"촖"#, r#"촖"#),
        (r#"촗"#, r#"촗"#, r#"촗"#, r#"촗"#, r#"촗"#),
        (r#"촘"#, r#"촘"#, r#"촘"#, r#"촘"#, r#"촘"#),
        (r#"촙"#, r#"촙"#, r#"촙"#, r#"촙"#, r#"촙"#),
        (r#"촚"#, r#"촚"#, r#"촚"#, r#"촚"#, r#"촚"#),
        (r#"촛"#, r#"촛"#, r#"촛"#, r#"촛"#, r#"촛"#),
        (r#"촜"#, r#"촜"#, r#"촜"#, r#"촜"#, r#"촜"#),
        (r#"총"#, r#"총"#, r#"총"#, r#"총"#, r#"총"#),
        (r#"촞"#, r#"촞"#, r#"촞"#, r#"촞"#, r#"촞"#),
        (r#"촟"#, r#"촟"#, r#"촟"#, r#"촟"#, r#"촟"#),
        (r#"촠"#, r#"촠"#, r#"촠"#, r#"촠"#, r#"촠"#),
        (r#"촡"#, r#"촡"#, r#"촡"#, r#"촡"#, r#"촡"#),
        (r#"촢"#, r#"촢"#, r#"촢"#, r#"촢"#, r#"촢"#),
        (r#"촣"#, r#"촣"#, r#"촣"#, r#"촣"#, r#"촣"#),
        (r#"촤"#, r#"촤"#, r#"촤"#, r#"촤"#, r#"촤"#),
        (r#"촥"#, r#"촥"#, r#"촥"#, r#"촥"#, r#"촥"#),
        (r#"촦"#, r#"촦"#, r#"촦"#, r#"촦"#, r#"촦"#),
        (r#"촧"#, r#"촧"#, r#"촧"#, r#"촧"#, r#"촧"#),
        (r#"촨"#, r#"촨"#, r#"촨"#, r#"촨"#, r#"촨"#),
        (r#"촩"#, r#"촩"#, r#"촩"#, r#"촩"#, r#"촩"#),
        (r#"촪"#, r#"촪"#, r#"촪"#, r#"촪"#, r#"촪"#),
        (r#"촫"#, r#"촫"#, r#"촫"#, r#"촫"#, r#"촫"#),
        (r#"촬"#, r#"촬"#, r#"촬"#, r#"촬"#, r#"촬"#),
        (r#"촭"#, r#"촭"#, r#"촭"#, r#"촭"#, r#"촭"#),
        (r#"촮"#, r#"촮"#, r#"촮"#, r#"촮"#, r#"촮"#),
        (r#"촯"#, r#"촯"#, r#"촯"#, r#"촯"#, r#"촯"#),
        (r#"촰"#, r#"촰"#, r#"촰"#, r#"촰"#, r#"촰"#),
        (r#"촱"#, r#"촱"#, r#"촱"#, r#"촱"#, r#"촱"#),
        (r#"촲"#, r#"촲"#, r#"촲"#, r#"촲"#, r#"촲"#),
        (r#"촳"#, r#"촳"#, r#"촳"#, r#"촳"#, r#"촳"#),
        (r#"촴"#, r#"촴"#, r#"촴"#, r#"촴"#, r#"촴"#),
        (r#"촵"#, r#"촵"#, r#"촵"#, r#"촵"#, r#"촵"#),
        (r#"촶"#, r#"촶"#, r#"촶"#, r#"촶"#, r#"촶"#),
        (r#"촷"#, r#"촷"#, r#"촷"#, r#"촷"#, r#"촷"#),
        (r#"촸"#, r#"촸"#, r#"촸"#, r#"촸"#, r#"촸"#),
        (r#"촹"#, r#"촹"#, r#"촹"#, r#"촹"#, r#"촹"#),
        (r#"촺"#, r#"촺"#, r#"촺"#, r#"촺"#, r#"촺"#),
        (r#"촻"#, r#"촻"#, r#"촻"#, r#"촻"#, r#"촻"#),
        (r#"촼"#, r#"촼"#, r#"촼"#, r#"촼"#, r#"촼"#),
        (r#"촽"#, r#"촽"#, r#"촽"#, r#"촽"#, r#"촽"#),
        (r#"촾"#, r#"촾"#, r#"촾"#, r#"촾"#, r#"촾"#),
        (r#"촿"#, r#"촿"#, r#"촿"#, r#"촿"#, r#"촿"#),
        (r#"쵀"#, r#"쵀"#, r#"쵀"#, r#"쵀"#, r#"쵀"#),
        (r#"쵁"#, r#"쵁"#, r#"쵁"#, r#"쵁"#, r#"쵁"#),
        (r#"쵂"#, r#"쵂"#, r#"쵂"#, r#"쵂"#, r#"쵂"#),
        (r#"쵃"#, r#"쵃"#, r#"쵃"#, r#"쵃"#, r#"쵃"#),
        (r#"쵄"#, r#"쵄"#, r#"쵄"#, r#"쵄"#, r#"쵄"#),
        (r#"쵅"#, r#"쵅"#, r#"쵅"#, r#"쵅"#, r#"쵅"#),
        (r#"쵆"#, r#"쵆"#, r#"쵆"#, r#"쵆"#, r#"쵆"#),
        (r#"쵇"#, r#"쵇"#, r#"쵇"#, r#"쵇"#, r#"쵇"#),
        (r#"쵈"#, r#"쵈"#, r#"쵈"#, r#"쵈"#, r#"쵈"#),
        (r#"쵉"#, r#"쵉"#, r#"쵉"#, r#"쵉"#, r#"쵉"#),
        (r#"쵊"#, r#"쵊"#, r#"쵊"#, r#"쵊"#, r#"쵊"#),
        (r#"쵋"#, r#"쵋"#, r#"쵋"#, r#"쵋"#, r#"쵋"#),
        (r#"쵌"#, r#"쵌"#, r#"쵌"#, r#"쵌"#, r#"쵌"#),
        (r#"쵍"#, r#"쵍"#, r#"쵍"#, r#"쵍"#, r#"쵍"#),
        (r#"쵎"#, r#"쵎"#, r#"쵎"#, r#"쵎"#, r#"쵎"#),
        (r#"쵏"#, r#"쵏"#, r#"쵏"#, r#"쵏"#, r#"쵏"#),
        (r#"쵐"#, r#"쵐"#, r#"쵐"#, r#"쵐"#, r#"쵐"#),
        (r#"쵑"#, r#"쵑"#, r#"쵑"#, r#"쵑"#, r#"쵑"#),
        (r#"쵒"#, r#"쵒"#, r#"쵒"#, r#"쵒"#, r#"쵒"#),
        (r#"쵓"#, r#"쵓"#, r#"쵓"#, r#"쵓"#, r#"쵓"#),
        (r#"쵔"#, r#"쵔"#, r#"쵔"#, r#"쵔"#, r#"쵔"#),
        (r#"쵕"#, r#"쵕"#, r#"쵕"#, r#"쵕"#, r#"쵕"#),
        (r#"쵖"#, r#"쵖"#, r#"쵖"#, r#"쵖"#, r#"쵖"#),
        (r#"쵗"#, r#"쵗"#, r#"쵗"#, r#"쵗"#, r#"쵗"#),
        (r#"쵘"#, r#"쵘"#, r#"쵘"#, r#"쵘"#, r#"쵘"#),
        (r#"쵙"#, r#"쵙"#, r#"쵙"#, r#"쵙"#, r#"쵙"#),
        (r#"쵚"#, r#"쵚"#, r#"쵚"#, r#"쵚"#, r#"쵚"#),
        (r#"쵛"#, r#"쵛"#, r#"쵛"#, r#"쵛"#, r#"쵛"#),
        (r#"최"#, r#"최"#, r#"최"#, r#"최"#, r#"최"#),
        (r#"쵝"#, r#"쵝"#, r#"쵝"#, r#"쵝"#, r#"쵝"#),
        (r#"쵞"#, r#"쵞"#, r#"쵞"#, r#"쵞"#, r#"쵞"#),
        (r#"쵟"#, r#"쵟"#, r#"쵟"#, r#"쵟"#, r#"쵟"#),
        (r#"쵠"#, r#"쵠"#, r#"쵠"#, r#"쵠"#, r#"쵠"#),
        (r#"쵡"#, r#"쵡"#, r#"쵡"#, r#"쵡"#, r#"쵡"#),
        (r#"쵢"#, r#"쵢"#, r#"쵢"#, r#"쵢"#, r#"쵢"#),
        (r#"쵣"#, r#"쵣"#, r#"쵣"#, r#"쵣"#, r#"쵣"#),
        (r#"쵤"#, r#"쵤"#, r#"쵤"#, r#"쵤"#, r#"쵤"#),
        (r#"쵥"#, r#"쵥"#, r#"쵥"#, r#"쵥"#, r#"쵥"#),
        (r#"쵦"#, r#"쵦"#, r#"쵦"#, r#"쵦"#, r#"쵦"#),
        (r#"쵧"#, r#"쵧"#, r#"쵧"#, r#"쵧"#, r#"쵧"#),
        (r#"쵨"#, r#"쵨"#, r#"쵨"#, r#"쵨"#, r#"쵨"#),
        (r#"쵩"#, r#"쵩"#, r#"쵩"#, r#"쵩"#, r#"쵩"#),
        (r#"쵪"#, r#"쵪"#, r#"쵪"#, r#"쵪"#, r#"쵪"#),
        (r#"쵫"#, r#"쵫"#, r#"쵫"#, r#"쵫"#, r#"쵫"#),
        (r#"쵬"#, r#"쵬"#, r#"쵬"#, r#"쵬"#, r#"쵬"#),
        (r#"쵭"#, r#"쵭"#, r#"쵭"#, r#"쵭"#, r#"쵭"#),
        (r#"쵮"#, r#"쵮"#, r#"쵮"#, r#"쵮"#, r#"쵮"#),
        (r#"쵯"#, r#"쵯"#, r#"쵯"#, r#"쵯"#, r#"쵯"#),
        (r#"쵰"#, r#"쵰"#, r#"쵰"#, r#"쵰"#, r#"쵰"#),
        (r#"쵱"#, r#"쵱"#, r#"쵱"#, r#"쵱"#, r#"쵱"#),
        (r#"쵲"#, r#"쵲"#, r#"쵲"#, r#"쵲"#, r#"쵲"#),
        (r#"쵳"#, r#"쵳"#, r#"쵳"#, r#"쵳"#, r#"쵳"#),
        (r#"쵴"#, r#"쵴"#, r#"쵴"#, r#"쵴"#, r#"쵴"#),
        (r#"쵵"#, r#"쵵"#, r#"쵵"#, r#"쵵"#, r#"쵵"#),
        (r#"쵶"#, r#"쵶"#, r#"쵶"#, r#"쵶"#, r#"쵶"#),
        (r#"쵷"#, r#"쵷"#, r#"쵷"#, r#"쵷"#, r#"쵷"#),
        (r#"쵸"#, r#"쵸"#, r#"쵸"#, r#"쵸"#, r#"쵸"#),
        (r#"쵹"#, r#"쵹"#, r#"쵹"#, r#"쵹"#, r#"쵹"#),
        (r#"쵺"#, r#"쵺"#, r#"쵺"#, r#"쵺"#, r#"쵺"#),
        (r#"쵻"#, r#"쵻"#, r#"쵻"#, r#"쵻"#, r#"쵻"#),
        (r#"쵼"#, r#"쵼"#, r#"쵼"#, r#"쵼"#, r#"쵼"#),
        (r#"쵽"#, r#"쵽"#, r#"쵽"#, r#"쵽"#, r#"쵽"#),
        (r#"쵾"#, r#"쵾"#, r#"쵾"#, r#"쵾"#, r#"쵾"#),
        (r#"쵿"#, r#"쵿"#, r#"쵿"#, r#"쵿"#, r#"쵿"#),
        (r#"춀"#, r#"춀"#, r#"춀"#, r#"춀"#, r#"춀"#),
        (r#"춁"#, r#"춁"#, r#"춁"#, r#"춁"#, r#"춁"#),
        (r#"춂"#, r#"춂"#, r#"춂"#, r#"춂"#, r#"춂"#),
        (r#"춃"#, r#"춃"#, r#"춃"#, r#"춃"#, r#"춃"#),
        (r#"춄"#, r#"춄"#, r#"춄"#, r#"춄"#, r#"춄"#),
        (r#"춅"#, r#"춅"#, r#"춅"#, r#"춅"#, r#"춅"#),
        (r#"춆"#, r#"춆"#, r#"춆"#, r#"춆"#, r#"춆"#),
        (r#"춇"#, r#"춇"#, r#"춇"#, r#"춇"#, r#"춇"#),
        (r#"춈"#, r#"춈"#, r#"춈"#, r#"춈"#, r#"춈"#),
        (r#"춉"#, r#"춉"#, r#"춉"#, r#"춉"#, r#"춉"#),
        (r#"춊"#, r#"춊"#, r#"춊"#, r#"춊"#, r#"춊"#),
        (r#"춋"#, r#"춋"#, r#"춋"#, r#"춋"#, r#"춋"#),
        (r#"춌"#, r#"춌"#, r#"춌"#, r#"춌"#, r#"춌"#),
        (r#"춍"#, r#"춍"#, r#"춍"#, r#"춍"#, r#"춍"#),
        (r#"춎"#, r#"춎"#, r#"춎"#, r#"춎"#, r#"춎"#),
        (r#"춏"#, r#"춏"#, r#"춏"#, r#"춏"#, r#"춏"#),
        (r#"춐"#, r#"춐"#, r#"춐"#, r#"춐"#, r#"춐"#),
        (r#"춑"#, r#"춑"#, r#"춑"#, r#"춑"#, r#"춑"#),
        (r#"춒"#, r#"춒"#, r#"춒"#, r#"춒"#, r#"춒"#),
        (r#"춓"#, r#"춓"#, r#"춓"#, r#"춓"#, r#"춓"#),
        (r#"추"#, r#"추"#, r#"추"#, r#"추"#, r#"추"#),
        (r#"축"#, r#"축"#, r#"축"#, r#"축"#, r#"축"#),
        (r#"춖"#, r#"춖"#, r#"춖"#, r#"춖"#, r#"춖"#),
        (r#"춗"#, r#"춗"#, r#"춗"#, r#"춗"#, r#"춗"#),
        (r#"춘"#, r#"춘"#, r#"춘"#, r#"춘"#, r#"춘"#),
        (r#"춙"#, r#"춙"#, r#"춙"#, r#"춙"#, r#"춙"#),
        (r#"춚"#, r#"춚"#, r#"춚"#, r#"춚"#, r#"춚"#),
        (r#"춛"#, r#"춛"#, r#"춛"#, r#"춛"#, r#"춛"#),
        (r#"출"#, r#"출"#, r#"출"#, r#"출"#, r#"출"#),
        (r#"춝"#, r#"춝"#, r#"춝"#, r#"춝"#, r#"춝"#),
        (r#"춞"#, r#"춞"#, r#"춞"#, r#"춞"#, r#"춞"#),
        (r#"춟"#, r#"춟"#, r#"춟"#, r#"춟"#, r#"춟"#),
        (r#"춠"#, r#"춠"#, r#"춠"#, r#"춠"#, r#"춠"#),
        (r#"춡"#, r#"춡"#, r#"춡"#, r#"춡"#, r#"춡"#),
        (r#"춢"#, r#"춢"#, r#"춢"#, r#"춢"#, r#"춢"#),
        (r#"춣"#, r#"춣"#, r#"춣"#, r#"춣"#, r#"춣"#),
        (r#"춤"#, r#"춤"#, r#"춤"#, r#"춤"#, r#"춤"#),
        (r#"춥"#, r#"춥"#, r#"춥"#, r#"춥"#, r#"춥"#),
        (r#"춦"#, r#"춦"#, r#"춦"#, r#"춦"#, r#"춦"#),
        (r#"춧"#, r#"춧"#, r#"춧"#, r#"춧"#, r#"춧"#),
        (r#"춨"#, r#"춨"#, r#"춨"#, r#"춨"#, r#"춨"#),
        (r#"충"#, r#"충"#, r#"충"#, r#"충"#, r#"충"#),
        (r#"춪"#, r#"춪"#, r#"춪"#, r#"춪"#, r#"춪"#),
        (r#"춫"#, r#"춫"#, r#"춫"#, r#"춫"#, r#"춫"#),
        (r#"춬"#, r#"춬"#, r#"춬"#, r#"춬"#, r#"춬"#),
        (r#"춭"#, r#"춭"#, r#"춭"#, r#"춭"#, r#"춭"#),
        (r#"춮"#, r#"춮"#, r#"춮"#, r#"춮"#, r#"춮"#),
        (r#"춯"#, r#"춯"#, r#"춯"#, r#"춯"#, r#"춯"#),
        (r#"춰"#, r#"춰"#, r#"춰"#, r#"춰"#, r#"춰"#),
        (r#"춱"#, r#"춱"#, r#"춱"#, r#"춱"#, r#"춱"#),
        (r#"춲"#, r#"춲"#, r#"춲"#, r#"춲"#, r#"춲"#),
        (r#"춳"#, r#"춳"#, r#"춳"#, r#"춳"#, r#"춳"#),
        (r#"춴"#, r#"춴"#, r#"춴"#, r#"춴"#, r#"춴"#),
        (r#"춵"#, r#"춵"#, r#"춵"#, r#"춵"#, r#"춵"#),
        (r#"춶"#, r#"춶"#, r#"춶"#, r#"춶"#, r#"춶"#),
        (r#"춷"#, r#"춷"#, r#"춷"#, r#"춷"#, r#"춷"#),
        (r#"춸"#, r#"춸"#, r#"춸"#, r#"춸"#, r#"춸"#),
        (r#"춹"#, r#"춹"#, r#"춹"#, r#"춹"#, r#"춹"#),
        (r#"춺"#, r#"춺"#, r#"춺"#, r#"춺"#, r#"춺"#),
        (r#"춻"#, r#"춻"#, r#"춻"#, r#"춻"#, r#"춻"#),
        (r#"춼"#, r#"춼"#, r#"춼"#, r#"춼"#, r#"춼"#),
        (r#"춽"#, r#"춽"#, r#"춽"#, r#"춽"#, r#"춽"#),
        (r#"춾"#, r#"춾"#, r#"춾"#, r#"춾"#, r#"춾"#),
        (r#"춿"#, r#"춿"#, r#"춿"#, r#"춿"#, r#"춿"#),
        (r#"췀"#, r#"췀"#, r#"췀"#, r#"췀"#, r#"췀"#),
        (r#"췁"#, r#"췁"#, r#"췁"#, r#"췁"#, r#"췁"#),
        (r#"췂"#, r#"췂"#, r#"췂"#, r#"췂"#, r#"췂"#),
        (r#"췃"#, r#"췃"#, r#"췃"#, r#"췃"#, r#"췃"#),
        (r#"췄"#, r#"췄"#, r#"췄"#, r#"췄"#, r#"췄"#),
        (r#"췅"#, r#"췅"#, r#"췅"#, r#"췅"#, r#"췅"#),
        (r#"췆"#, r#"췆"#, r#"췆"#, r#"췆"#, r#"췆"#),
        (r#"췇"#, r#"췇"#, r#"췇"#, r#"췇"#, r#"췇"#),
        (r#"췈"#, r#"췈"#, r#"췈"#, r#"췈"#, r#"췈"#),
        (r#"췉"#, r#"췉"#, r#"췉"#, r#"췉"#, r#"췉"#),
        (r#"췊"#, r#"췊"#, r#"췊"#, r#"췊"#, r#"췊"#),
        (r#"췋"#, r#"췋"#, r#"췋"#, r#"췋"#, r#"췋"#),
        (r#"췌"#, r#"췌"#, r#"췌"#, r#"췌"#, r#"췌"#),
        (r#"췍"#, r#"췍"#, r#"췍"#, r#"췍"#, r#"췍"#),
        (r#"췎"#, r#"췎"#, r#"췎"#, r#"췎"#, r#"췎"#),
        (r#"췏"#, r#"췏"#, r#"췏"#, r#"췏"#, r#"췏"#),
        (r#"췐"#, r#"췐"#, r#"췐"#, r#"췐"#, r#"췐"#),
        (r#"췑"#, r#"췑"#, r#"췑"#, r#"췑"#, r#"췑"#),
        (r#"췒"#, r#"췒"#, r#"췒"#, r#"췒"#, r#"췒"#),
        (r#"췓"#, r#"췓"#, r#"췓"#, r#"췓"#, r#"췓"#),
        (r#"췔"#, r#"췔"#, r#"췔"#, r#"췔"#, r#"췔"#),
        (r#"췕"#, r#"췕"#, r#"췕"#, r#"췕"#, r#"췕"#),
        (r#"췖"#, r#"췖"#, r#"췖"#, r#"췖"#, r#"췖"#),
        (r#"췗"#, r#"췗"#, r#"췗"#, r#"췗"#, r#"췗"#),
        (r#"췘"#, r#"췘"#, r#"췘"#, r#"췘"#, r#"췘"#),
        (r#"췙"#, r#"췙"#, r#"췙"#, r#"췙"#, r#"췙"#),
        (r#"췚"#, r#"췚"#, r#"췚"#, r#"췚"#, r#"췚"#),
        (r#"췛"#, r#"췛"#, r#"췛"#, r#"췛"#, r#"췛"#),
        (r#"췜"#, r#"췜"#, r#"췜"#, r#"췜"#, r#"췜"#),
        (r#"췝"#, r#"췝"#, r#"췝"#, r#"췝"#, r#"췝"#),
        (r#"췞"#, r#"췞"#, r#"췞"#, r#"췞"#, r#"췞"#),
        (r#"췟"#, r#"췟"#, r#"췟"#, r#"췟"#, r#"췟"#),
        (r#"췠"#, r#"췠"#, r#"췠"#, r#"췠"#, r#"췠"#),
        (r#"췡"#, r#"췡"#, r#"췡"#, r#"췡"#, r#"췡"#),
        (r#"췢"#, r#"췢"#, r#"췢"#, r#"췢"#, r#"췢"#),
        (r#"췣"#, r#"췣"#, r#"췣"#, r#"췣"#, r#"췣"#),
        (r#"췤"#, r#"췤"#, r#"췤"#, r#"췤"#, r#"췤"#),
        (r#"췥"#, r#"췥"#, r#"췥"#, r#"췥"#, r#"췥"#),
        (r#"췦"#, r#"췦"#, r#"췦"#, r#"췦"#, r#"췦"#),
        (r#"췧"#, r#"췧"#, r#"췧"#, r#"췧"#, r#"췧"#),
        (r#"취"#, r#"취"#, r#"취"#, r#"취"#, r#"취"#),
        (r#"췩"#, r#"췩"#, r#"췩"#, r#"췩"#, r#"췩"#),
        (r#"췪"#, r#"췪"#, r#"췪"#, r#"췪"#, r#"췪"#),
        (r#"췫"#, r#"췫"#, r#"췫"#, r#"췫"#, r#"췫"#),
        (r#"췬"#, r#"췬"#, r#"췬"#, r#"췬"#, r#"췬"#),
        (r#"췭"#, r#"췭"#, r#"췭"#, r#"췭"#, r#"췭"#),
        (r#"췮"#, r#"췮"#, r#"췮"#, r#"췮"#, r#"췮"#),
        (r#"췯"#, r#"췯"#, r#"췯"#, r#"췯"#, r#"췯"#),
        (r#"췰"#, r#"췰"#, r#"췰"#, r#"췰"#, r#"췰"#),
        (r#"췱"#, r#"췱"#, r#"췱"#, r#"췱"#, r#"췱"#),
        (r#"췲"#, r#"췲"#, r#"췲"#, r#"췲"#, r#"췲"#),
        (r#"췳"#, r#"췳"#, r#"췳"#, r#"췳"#, r#"췳"#),
        (r#"췴"#, r#"췴"#, r#"췴"#, r#"췴"#, r#"췴"#),
        (r#"췵"#, r#"췵"#, r#"췵"#, r#"췵"#, r#"췵"#),
        (r#"췶"#, r#"췶"#, r#"췶"#, r#"췶"#, r#"췶"#),
        (r#"췷"#, r#"췷"#, r#"췷"#, r#"췷"#, r#"췷"#),
        (r#"췸"#, r#"췸"#, r#"췸"#, r#"췸"#, r#"췸"#),
        (r#"췹"#, r#"췹"#, r#"췹"#, r#"췹"#, r#"췹"#),
        (r#"췺"#, r#"췺"#, r#"췺"#, r#"췺"#, r#"췺"#),
        (r#"췻"#, r#"췻"#, r#"췻"#, r#"췻"#, r#"췻"#),
        (r#"췼"#, r#"췼"#, r#"췼"#, r#"췼"#, r#"췼"#),
        (r#"췽"#, r#"췽"#, r#"췽"#, r#"췽"#, r#"췽"#),
        (r#"췾"#, r#"췾"#, r#"췾"#, r#"췾"#, r#"췾"#),
        (r#"췿"#, r#"췿"#, r#"췿"#, r#"췿"#, r#"췿"#),
        (r#"츀"#, r#"츀"#, r#"츀"#, r#"츀"#, r#"츀"#),
        (r#"츁"#, r#"츁"#, r#"츁"#, r#"츁"#, r#"츁"#),
        (r#"츂"#, r#"츂"#, r#"츂"#, r#"츂"#, r#"츂"#),
        (r#"츃"#, r#"츃"#, r#"츃"#, r#"츃"#, r#"츃"#),
        (r#"츄"#, r#"츄"#, r#"츄"#, r#"츄"#, r#"츄"#),
        (r#"츅"#, r#"츅"#, r#"츅"#, r#"츅"#, r#"츅"#),
        (r#"츆"#, r#"츆"#, r#"츆"#, r#"츆"#, r#"츆"#),
        (r#"츇"#, r#"츇"#, r#"츇"#, r#"츇"#, r#"츇"#),
        (r#"츈"#, r#"츈"#, r#"츈"#, r#"츈"#, r#"츈"#),
        (r#"츉"#, r#"츉"#, r#"츉"#, r#"츉"#, r#"츉"#),
        (r#"츊"#, r#"츊"#, r#"츊"#, r#"츊"#, r#"츊"#),
        (r#"츋"#, r#"츋"#, r#"츋"#, r#"츋"#, r#"츋"#),
        (r#"츌"#, r#"츌"#, r#"츌"#, r#"츌"#, r#"츌"#),
        (r#"츍"#, r#"츍"#, r#"츍"#, r#"츍"#, r#"츍"#),
        (r#"츎"#, r#"츎"#, r#"츎"#, r#"츎"#, r#"츎"#),
        (r#"츏"#, r#"츏"#, r#"츏"#, r#"츏"#, r#"츏"#),
        (r#"츐"#, r#"츐"#, r#"츐"#, r#"츐"#, r#"츐"#),
        (r#"츑"#, r#"츑"#, r#"츑"#, r#"츑"#, r#"츑"#),
        (r#"츒"#, r#"츒"#, r#"츒"#, r#"츒"#, r#"츒"#),
        (r#"츓"#, r#"츓"#, r#"츓"#, r#"츓"#, r#"츓"#),
        (r#"츔"#, r#"츔"#, r#"츔"#, r#"츔"#, r#"츔"#),
        (r#"츕"#, r#"츕"#, r#"츕"#, r#"츕"#, r#"츕"#),
        (r#"츖"#, r#"츖"#, r#"츖"#, r#"츖"#, r#"츖"#),
        (r#"츗"#, r#"츗"#, r#"츗"#, r#"츗"#, r#"츗"#),
        (r#"츘"#, r#"츘"#, r#"츘"#, r#"츘"#, r#"츘"#),
        (r#"츙"#, r#"츙"#, r#"츙"#, r#"츙"#, r#"츙"#),
        (r#"츚"#, r#"츚"#, r#"츚"#, r#"츚"#, r#"츚"#),
        (r#"츛"#, r#"츛"#, r#"츛"#, r#"츛"#, r#"츛"#),
        (r#"츜"#, r#"츜"#, r#"츜"#, r#"츜"#, r#"츜"#),
        (r#"츝"#, r#"츝"#, r#"츝"#, r#"츝"#, r#"츝"#),
        (r#"츞"#, r#"츞"#, r#"츞"#, r#"츞"#, r#"츞"#),
        (r#"츟"#, r#"츟"#, r#"츟"#, r#"츟"#, r#"츟"#),
        (r#"츠"#, r#"츠"#, r#"츠"#, r#"츠"#, r#"츠"#),
        (r#"측"#, r#"측"#, r#"측"#, r#"측"#, r#"측"#),
        (r#"츢"#, r#"츢"#, r#"츢"#, r#"츢"#, r#"츢"#),
        (r#"츣"#, r#"츣"#, r#"츣"#, r#"츣"#, r#"츣"#),
        (r#"츤"#, r#"츤"#, r#"츤"#, r#"츤"#, r#"츤"#),
        (r#"츥"#, r#"츥"#, r#"츥"#, r#"츥"#, r#"츥"#),
        (r#"츦"#, r#"츦"#, r#"츦"#, r#"츦"#, r#"츦"#),
        (r#"츧"#, r#"츧"#, r#"츧"#, r#"츧"#, r#"츧"#),
        (r#"츨"#, r#"츨"#, r#"츨"#, r#"츨"#, r#"츨"#),
        (r#"츩"#, r#"츩"#, r#"츩"#, r#"츩"#, r#"츩"#),
        (r#"츪"#, r#"츪"#, r#"츪"#, r#"츪"#, r#"츪"#),
        (r#"츫"#, r#"츫"#, r#"츫"#, r#"츫"#, r#"츫"#),
        (r#"츬"#, r#"츬"#, r#"츬"#, r#"츬"#, r#"츬"#),
        (r#"츭"#, r#"츭"#, r#"츭"#, r#"츭"#, r#"츭"#),
        (r#"츮"#, r#"츮"#, r#"츮"#, r#"츮"#, r#"츮"#),
        (r#"츯"#, r#"츯"#, r#"츯"#, r#"츯"#, r#"츯"#),
        (r#"츰"#, r#"츰"#, r#"츰"#, r#"츰"#, r#"츰"#),
        (r#"츱"#, r#"츱"#, r#"츱"#, r#"츱"#, r#"츱"#),
        (r#"츲"#, r#"츲"#, r#"츲"#, r#"츲"#, r#"츲"#),
        (r#"츳"#, r#"츳"#, r#"츳"#, r#"츳"#, r#"츳"#),
        (r#"츴"#, r#"츴"#, r#"츴"#, r#"츴"#, r#"츴"#),
        (r#"층"#, r#"층"#, r#"층"#, r#"층"#, r#"층"#),
        (r#"츶"#, r#"츶"#, r#"츶"#, r#"츶"#, r#"츶"#),
        (r#"츷"#, r#"츷"#, r#"츷"#, r#"츷"#, r#"츷"#),
        (r#"츸"#, r#"츸"#, r#"츸"#, r#"츸"#, r#"츸"#),
        (r#"츹"#, r#"츹"#, r#"츹"#, r#"츹"#, r#"츹"#),
        (r#"츺"#, r#"츺"#, r#"츺"#, r#"츺"#, r#"츺"#),
        (r#"츻"#, r#"츻"#, r#"츻"#, r#"츻"#, r#"츻"#),
        (r#"츼"#, r#"츼"#, r#"츼"#, r#"츼"#, r#"츼"#),
        (r#"츽"#, r#"츽"#, r#"츽"#, r#"츽"#, r#"츽"#),
        (r#"츾"#, r#"츾"#, r#"츾"#, r#"츾"#, r#"츾"#),
        (r#"츿"#, r#"츿"#, r#"츿"#, r#"츿"#, r#"츿"#),
        (r#"칀"#, r#"칀"#, r#"칀"#, r#"칀"#, r#"칀"#),
        (r#"칁"#, r#"칁"#, r#"칁"#, r#"칁"#, r#"칁"#),
        (r#"칂"#, r#"칂"#, r#"칂"#, r#"칂"#, r#"칂"#),
        (r#"칃"#, r#"칃"#, r#"칃"#, r#"칃"#, r#"칃"#),
        (r#"칄"#, r#"칄"#, r#"칄"#, r#"칄"#, r#"칄"#),
        (r#"칅"#, r#"칅"#, r#"칅"#, r#"칅"#, r#"칅"#),
        (r#"칆"#, r#"칆"#, r#"칆"#, r#"칆"#, r#"칆"#),
        (r#"칇"#, r#"칇"#, r#"칇"#, r#"칇"#, r#"칇"#),
        (r#"칈"#, r#"칈"#, r#"칈"#, r#"칈"#, r#"칈"#),
        (r#"칉"#, r#"칉"#, r#"칉"#, r#"칉"#, r#"칉"#),
        (r#"칊"#, r#"칊"#, r#"칊"#, r#"칊"#, r#"칊"#),
        (r#"칋"#, r#"칋"#, r#"칋"#, r#"칋"#, r#"칋"#),
        (r#"칌"#, r#"칌"#, r#"칌"#, r#"칌"#, r#"칌"#),
        (r#"칍"#, r#"칍"#, r#"칍"#, r#"칍"#, r#"칍"#),
        (r#"칎"#, r#"칎"#, r#"칎"#, r#"칎"#, r#"칎"#),
        (r#"칏"#, r#"칏"#, r#"칏"#, r#"칏"#, r#"칏"#),
        (r#"칐"#, r#"칐"#, r#"칐"#, r#"칐"#, r#"칐"#),
        (r#"칑"#, r#"칑"#, r#"칑"#, r#"칑"#, r#"칑"#),
        (r#"칒"#, r#"칒"#, r#"칒"#, r#"칒"#, r#"칒"#),
        (r#"칓"#, r#"칓"#, r#"칓"#, r#"칓"#, r#"칓"#),
        (r#"칔"#, r#"칔"#, r#"칔"#, r#"칔"#, r#"칔"#),
        (r#"칕"#, r#"칕"#, r#"칕"#, r#"칕"#, r#"칕"#),
        (r#"칖"#, r#"칖"#, r#"칖"#, r#"칖"#, r#"칖"#),
        (r#"칗"#, r#"칗"#, r#"칗"#, r#"칗"#, r#"칗"#),
        (r#"치"#, r#"치"#, r#"치"#, r#"치"#, r#"치"#),
        (r#"칙"#, r#"칙"#, r#"칙"#, r#"칙"#, r#"칙"#),
        (r#"칚"#, r#"칚"#, r#"칚"#, r#"칚"#, r#"칚"#),
        (r#"칛"#, r#"칛"#, r#"칛"#, r#"칛"#, r#"칛"#),
        (r#"친"#, r#"친"#, r#"친"#, r#"친"#, r#"친"#),
        (r#"칝"#, r#"칝"#, r#"칝"#, r#"칝"#, r#"칝"#),
        (r#"칞"#, r#"칞"#, r#"칞"#, r#"칞"#, r#"칞"#),
        (r#"칟"#, r#"칟"#, r#"칟"#, r#"칟"#, r#"칟"#),
        (r#"칠"#, r#"칠"#, r#"칠"#, r#"칠"#, r#"칠"#),
        (r#"칡"#, r#"칡"#, r#"칡"#, r#"칡"#, r#"칡"#),
        (r#"칢"#, r#"칢"#, r#"칢"#, r#"칢"#, r#"칢"#),
        (r#"칣"#, r#"칣"#, r#"칣"#, r#"칣"#, r#"칣"#),
        (r#"칤"#, r#"칤"#, r#"칤"#, r#"칤"#, r#"칤"#),
        (r#"칥"#, r#"칥"#, r#"칥"#, r#"칥"#, r#"칥"#),
        (r#"칦"#, r#"칦"#, r#"칦"#, r#"칦"#, r#"칦"#),
        (r#"칧"#, r#"칧"#, r#"칧"#, r#"칧"#, r#"칧"#),
        (r#"침"#, r#"침"#, r#"침"#, r#"침"#, r#"침"#),
        (r#"칩"#, r#"칩"#, r#"칩"#, r#"칩"#, r#"칩"#),
        (r#"칪"#, r#"칪"#, r#"칪"#, r#"칪"#, r#"칪"#),
        (r#"칫"#, r#"칫"#, r#"칫"#, r#"칫"#, r#"칫"#),
        (r#"칬"#, r#"칬"#, r#"칬"#, r#"칬"#, r#"칬"#),
        (r#"칭"#, r#"칭"#, r#"칭"#, r#"칭"#, r#"칭"#),
        (r#"칮"#, r#"칮"#, r#"칮"#, r#"칮"#, r#"칮"#),
        (r#"칯"#, r#"칯"#, r#"칯"#, r#"칯"#, r#"칯"#),
        (r#"칰"#, r#"칰"#, r#"칰"#, r#"칰"#, r#"칰"#),
        (r#"칱"#, r#"칱"#, r#"칱"#, r#"칱"#, r#"칱"#),
        (r#"칲"#, r#"칲"#, r#"칲"#, r#"칲"#, r#"칲"#),
        (r#"칳"#, r#"칳"#, r#"칳"#, r#"칳"#, r#"칳"#),
        (r#"카"#, r#"카"#, r#"카"#, r#"카"#, r#"카"#),
        (r#"칵"#, r#"칵"#, r#"칵"#, r#"칵"#, r#"칵"#),
        (r#"칶"#, r#"칶"#, r#"칶"#, r#"칶"#, r#"칶"#),
        (r#"칷"#, r#"칷"#, r#"칷"#, r#"칷"#, r#"칷"#),
        (r#"칸"#, r#"칸"#, r#"칸"#, r#"칸"#, r#"칸"#),
        (r#"칹"#, r#"칹"#, r#"칹"#, r#"칹"#, r#"칹"#),
        (r#"칺"#, r#"칺"#, r#"칺"#, r#"칺"#, r#"칺"#),
        (r#"칻"#, r#"칻"#, r#"칻"#, r#"칻"#, r#"칻"#),
        (r#"칼"#, r#"칼"#, r#"칼"#, r#"칼"#, r#"칼"#),
        (r#"칽"#, r#"칽"#, r#"칽"#, r#"칽"#, r#"칽"#),
        (r#"칾"#, r#"칾"#, r#"칾"#, r#"칾"#, r#"칾"#),
        (r#"칿"#, r#"칿"#, r#"칿"#, r#"칿"#, r#"칿"#),
        (r#"캀"#, r#"캀"#, r#"캀"#, r#"캀"#, r#"캀"#),
        (r#"캁"#, r#"캁"#, r#"캁"#, r#"캁"#, r#"캁"#),
        (r#"캂"#, r#"캂"#, r#"캂"#, r#"캂"#, r#"캂"#),
        (r#"캃"#, r#"캃"#, r#"캃"#, r#"캃"#, r#"캃"#),
        (r#"캄"#, r#"캄"#, r#"캄"#, r#"캄"#, r#"캄"#),
        (r#"캅"#, r#"캅"#, r#"캅"#, r#"캅"#, r#"캅"#),
        (r#"캆"#, r#"캆"#, r#"캆"#, r#"캆"#, r#"캆"#),
        (r#"캇"#, r#"캇"#, r#"캇"#, r#"캇"#, r#"캇"#),
        (r#"캈"#, r#"캈"#, r#"캈"#, r#"캈"#, r#"캈"#),
        (r#"캉"#, r#"캉"#, r#"캉"#, r#"캉"#, r#"캉"#),
        (r#"캊"#, r#"캊"#, r#"캊"#, r#"캊"#, r#"캊"#),
        (r#"캋"#, r#"캋"#, r#"캋"#, r#"캋"#, r#"캋"#),
        (r#"캌"#, r#"캌"#, r#"캌"#, r#"캌"#, r#"캌"#),
        (r#"캍"#, r#"캍"#, r#"캍"#, r#"캍"#, r#"캍"#),
        (r#"캎"#, r#"캎"#, r#"캎"#, r#"캎"#, r#"캎"#),
        (r#"캏"#, r#"캏"#, r#"캏"#, r#"캏"#, r#"캏"#),
        (r#"캐"#, r#"캐"#, r#"캐"#, r#"캐"#, r#"캐"#),
        (r#"캑"#, r#"캑"#, r#"캑"#, r#"캑"#, r#"캑"#),
        (r#"캒"#, r#"캒"#, r#"캒"#, r#"캒"#, r#"캒"#),
        (r#"캓"#, r#"캓"#, r#"캓"#, r#"캓"#, r#"캓"#),
        (r#"캔"#, r#"캔"#, r#"캔"#, r#"캔"#, r#"캔"#),
        (r#"캕"#, r#"캕"#, r#"캕"#, r#"캕"#, r#"캕"#),
        (r#"캖"#, r#"캖"#, r#"캖"#, r#"캖"#, r#"캖"#),
        (r#"캗"#, r#"캗"#, r#"캗"#, r#"캗"#, r#"캗"#),
        (r#"캘"#, r#"캘"#, r#"캘"#, r#"캘"#, r#"캘"#),
        (r#"캙"#, r#"캙"#, r#"캙"#, r#"캙"#, r#"캙"#),
        (r#"캚"#, r#"캚"#, r#"캚"#, r#"캚"#, r#"캚"#),
        (r#"캛"#, r#"캛"#, r#"캛"#, r#"캛"#, r#"캛"#),
        (r#"캜"#, r#"캜"#, r#"캜"#, r#"캜"#, r#"캜"#),
        (r#"캝"#, r#"캝"#, r#"캝"#, r#"캝"#, r#"캝"#),
        (r#"캞"#, r#"캞"#, r#"캞"#, r#"캞"#, r#"캞"#),
        (r#"캟"#, r#"캟"#, r#"캟"#, r#"캟"#, r#"캟"#),
        (r#"캠"#, r#"캠"#, r#"캠"#, r#"캠"#, r#"캠"#),
        (r#"캡"#, r#"캡"#, r#"캡"#, r#"캡"#, r#"캡"#),
        (r#"캢"#, r#"캢"#, r#"캢"#, r#"캢"#, r#"캢"#),
        (r#"캣"#, r#"캣"#, r#"캣"#, r#"캣"#, r#"캣"#),
        (r#"캤"#, r#"캤"#, r#"캤"#, r#"캤"#, r#"캤"#),
        (r#"캥"#, r#"캥"#, r#"캥"#, r#"캥"#, r#"캥"#),
        (r#"캦"#, r#"캦"#, r#"캦"#, r#"캦"#, r#"캦"#),
        (r#"캧"#, r#"캧"#, r#"캧"#, r#"캧"#, r#"캧"#),
        (r#"캨"#, r#"캨"#, r#"캨"#, r#"캨"#, r#"캨"#),
        (r#"캩"#, r#"캩"#, r#"캩"#, r#"캩"#, r#"캩"#),
        (r#"캪"#, r#"캪"#, r#"캪"#, r#"캪"#, r#"캪"#),
        (r#"캫"#, r#"캫"#, r#"캫"#, r#"캫"#, r#"캫"#),
        (r#"캬"#, r#"캬"#, r#"캬"#, r#"캬"#, r#"캬"#),
        (r#"캭"#, r#"캭"#, r#"캭"#, r#"캭"#, r#"캭"#),
        (r#"캮"#, r#"캮"#, r#"캮"#, r#"캮"#, r#"캮"#),
        (r#"캯"#, r#"캯"#, r#"캯"#, r#"캯"#, r#"캯"#),
        (r#"캰"#, r#"캰"#, r#"캰"#, r#"캰"#, r#"캰"#),
        (r#"캱"#, r#"캱"#, r#"캱"#, r#"캱"#, r#"캱"#),
        (r#"캲"#, r#"캲"#, r#"캲"#, r#"캲"#, r#"캲"#),
        (r#"캳"#, r#"캳"#, r#"캳"#, r#"캳"#, r#"캳"#),
        (r#"캴"#, r#"캴"#, r#"캴"#, r#"캴"#, r#"캴"#),
        (r#"캵"#, r#"캵"#, r#"캵"#, r#"캵"#, r#"캵"#),
        (r#"캶"#, r#"캶"#, r#"캶"#, r#"캶"#, r#"캶"#),
        (r#"캷"#, r#"캷"#, r#"캷"#, r#"캷"#, r#"캷"#),
        (r#"캸"#, r#"캸"#, r#"캸"#, r#"캸"#, r#"캸"#),
        (r#"캹"#, r#"캹"#, r#"캹"#, r#"캹"#, r#"캹"#),
        (r#"캺"#, r#"캺"#, r#"캺"#, r#"캺"#, r#"캺"#),
        (r#"캻"#, r#"캻"#, r#"캻"#, r#"캻"#, r#"캻"#),
        (r#"캼"#, r#"캼"#, r#"캼"#, r#"캼"#, r#"캼"#),
        (r#"캽"#, r#"캽"#, r#"캽"#, r#"캽"#, r#"캽"#),
        (r#"캾"#, r#"캾"#, r#"캾"#, r#"캾"#, r#"캾"#),
        (r#"캿"#, r#"캿"#, r#"캿"#, r#"캿"#, r#"캿"#),
        (r#"컀"#, r#"컀"#, r#"컀"#, r#"컀"#, r#"컀"#),
        (r#"컁"#, r#"컁"#, r#"컁"#, r#"컁"#, r#"컁"#),
        (r#"컂"#, r#"컂"#, r#"컂"#, r#"컂"#, r#"컂"#),
        (r#"컃"#, r#"컃"#, r#"컃"#, r#"컃"#, r#"컃"#),
        (r#"컄"#, r#"컄"#, r#"컄"#, r#"컄"#, r#"컄"#),
        (r#"컅"#, r#"컅"#, r#"컅"#, r#"컅"#, r#"컅"#),
        (r#"컆"#, r#"컆"#, r#"컆"#, r#"컆"#, r#"컆"#),
        (r#"컇"#, r#"컇"#, r#"컇"#, r#"컇"#, r#"컇"#),
        (r#"컈"#, r#"컈"#, r#"컈"#, r#"컈"#, r#"컈"#),
        (r#"컉"#, r#"컉"#, r#"컉"#, r#"컉"#, r#"컉"#),
        (r#"컊"#, r#"컊"#, r#"컊"#, r#"컊"#, r#"컊"#),
        (r#"컋"#, r#"컋"#, r#"컋"#, r#"컋"#, r#"컋"#),
        (r#"컌"#, r#"컌"#, r#"컌"#, r#"컌"#, r#"컌"#),
        (r#"컍"#, r#"컍"#, r#"컍"#, r#"컍"#, r#"컍"#),
        (r#"컎"#, r#"컎"#, r#"컎"#, r#"컎"#, r#"컎"#),
        (r#"컏"#, r#"컏"#, r#"컏"#, r#"컏"#, r#"컏"#),
        (r#"컐"#, r#"컐"#, r#"컐"#, r#"컐"#, r#"컐"#),
        (r#"컑"#, r#"컑"#, r#"컑"#, r#"컑"#, r#"컑"#),
        (r#"컒"#, r#"컒"#, r#"컒"#, r#"컒"#, r#"컒"#),
        (r#"컓"#, r#"컓"#, r#"컓"#, r#"컓"#, r#"컓"#),
        (r#"컔"#, r#"컔"#, r#"컔"#, r#"컔"#, r#"컔"#),
        (r#"컕"#, r#"컕"#, r#"컕"#, r#"컕"#, r#"컕"#),
        (r#"컖"#, r#"컖"#, r#"컖"#, r#"컖"#, r#"컖"#),
        (r#"컗"#, r#"컗"#, r#"컗"#, r#"컗"#, r#"컗"#),
        (r#"컘"#, r#"컘"#, r#"컘"#, r#"컘"#, r#"컘"#),
        (r#"컙"#, r#"컙"#, r#"컙"#, r#"컙"#, r#"컙"#),
        (r#"컚"#, r#"컚"#, r#"컚"#, r#"컚"#, r#"컚"#),
        (r#"컛"#, r#"컛"#, r#"컛"#, r#"컛"#, r#"컛"#),
        (r#"컜"#, r#"컜"#, r#"컜"#, r#"컜"#, r#"컜"#),
        (r#"컝"#, r#"컝"#, r#"컝"#, r#"컝"#, r#"컝"#),
        (r#"컞"#, r#"컞"#, r#"컞"#, r#"컞"#, r#"컞"#),
        (r#"컟"#, r#"컟"#, r#"컟"#, r#"컟"#, r#"컟"#),
        (r#"컠"#, r#"컠"#, r#"컠"#, r#"컠"#, r#"컠"#),
        (r#"컡"#, r#"컡"#, r#"컡"#, r#"컡"#, r#"컡"#),
        (r#"컢"#, r#"컢"#, r#"컢"#, r#"컢"#, r#"컢"#),
        (r#"컣"#, r#"컣"#, r#"컣"#, r#"컣"#, r#"컣"#),
        (r#"커"#, r#"커"#, r#"커"#, r#"커"#, r#"커"#),
        (r#"컥"#, r#"컥"#, r#"컥"#, r#"컥"#, r#"컥"#),
        (r#"컦"#, r#"컦"#, r#"컦"#, r#"컦"#, r#"컦"#),
        (r#"컧"#, r#"컧"#, r#"컧"#, r#"컧"#, r#"컧"#),
        (r#"컨"#, r#"컨"#, r#"컨"#, r#"컨"#, r#"컨"#),
        (r#"컩"#, r#"컩"#, r#"컩"#, r#"컩"#, r#"컩"#),
        (r#"컪"#, r#"컪"#, r#"컪"#, r#"컪"#, r#"컪"#),
        (r#"컫"#, r#"컫"#, r#"컫"#, r#"컫"#, r#"컫"#),
        (r#"컬"#, r#"컬"#, r#"컬"#, r#"컬"#, r#"컬"#),
        (r#"컭"#, r#"컭"#, r#"컭"#, r#"컭"#, r#"컭"#),
        (r#"컮"#, r#"컮"#, r#"컮"#, r#"컮"#, r#"컮"#),
        (r#"컯"#, r#"컯"#, r#"컯"#, r#"컯"#, r#"컯"#),
        (r#"컰"#, r#"컰"#, r#"컰"#, r#"컰"#, r#"컰"#),
        (r#"컱"#, r#"컱"#, r#"컱"#, r#"컱"#, r#"컱"#),
        (r#"컲"#, r#"컲"#, r#"컲"#, r#"컲"#, r#"컲"#),
        (r#"컳"#, r#"컳"#, r#"컳"#, r#"컳"#, r#"컳"#),
        (r#"컴"#, r#"컴"#, r#"컴"#, r#"컴"#, r#"컴"#),
        (r#"컵"#, r#"컵"#, r#"컵"#, r#"컵"#, r#"컵"#),
        (r#"컶"#, r#"컶"#, r#"컶"#, r#"컶"#, r#"컶"#),
        (r#"컷"#, r#"컷"#, r#"컷"#, r#"컷"#, r#"컷"#),
        (r#"컸"#, r#"컸"#, r#"컸"#, r#"컸"#, r#"컸"#),
        (r#"컹"#, r#"컹"#, r#"컹"#, r#"컹"#, r#"컹"#),
        (r#"컺"#, r#"컺"#, r#"컺"#, r#"컺"#, r#"컺"#),
        (r#"컻"#, r#"컻"#, r#"컻"#, r#"컻"#, r#"컻"#),
        (r#"컼"#, r#"컼"#, r#"컼"#, r#"컼"#, r#"컼"#),
        (r#"컽"#, r#"컽"#, r#"컽"#, r#"컽"#, r#"컽"#),
        (r#"컾"#, r#"컾"#, r#"컾"#, r#"컾"#, r#"컾"#),
        (r#"컿"#, r#"컿"#, r#"컿"#, r#"컿"#, r#"컿"#),
        (r#"케"#, r#"케"#, r#"케"#, r#"케"#, r#"케"#),
        (r#"켁"#, r#"켁"#, r#"켁"#, r#"켁"#, r#"켁"#),
        (r#"켂"#, r#"켂"#, r#"켂"#, r#"켂"#, r#"켂"#),
        (r#"켃"#, r#"켃"#, r#"켃"#, r#"켃"#, r#"켃"#),
        (r#"켄"#, r#"켄"#, r#"켄"#, r#"켄"#, r#"켄"#),
        (r#"켅"#, r#"켅"#, r#"켅"#, r#"켅"#, r#"켅"#),
        (r#"켆"#, r#"켆"#, r#"켆"#, r#"켆"#, r#"켆"#),
        (r#"켇"#, r#"켇"#, r#"켇"#, r#"켇"#, r#"켇"#),
        (r#"켈"#, r#"켈"#, r#"켈"#, r#"켈"#, r#"켈"#),
        (r#"켉"#, r#"켉"#, r#"켉"#, r#"켉"#, r#"켉"#),
        (r#"켊"#, r#"켊"#, r#"켊"#, r#"켊"#, r#"켊"#),
        (r#"켋"#, r#"켋"#, r#"켋"#, r#"켋"#, r#"켋"#),
        (r#"켌"#, r#"켌"#, r#"켌"#, r#"켌"#, r#"켌"#),
        (r#"켍"#, r#"켍"#, r#"켍"#, r#"켍"#, r#"켍"#),
        (r#"켎"#, r#"켎"#, r#"켎"#, r#"켎"#, r#"켎"#),
        (r#"켏"#, r#"켏"#, r#"켏"#, r#"켏"#, r#"켏"#),
        (r#"켐"#, r#"켐"#, r#"켐"#, r#"켐"#, r#"켐"#),
        (r#"켑"#, r#"켑"#, r#"켑"#, r#"켑"#, r#"켑"#),
        (r#"켒"#, r#"켒"#, r#"켒"#, r#"켒"#, r#"켒"#),
        (r#"켓"#, r#"켓"#, r#"켓"#, r#"켓"#, r#"켓"#),
        (r#"켔"#, r#"켔"#, r#"켔"#, r#"켔"#, r#"켔"#),
        (r#"켕"#, r#"켕"#, r#"켕"#, r#"켕"#, r#"켕"#),
        (r#"켖"#, r#"켖"#, r#"켖"#, r#"켖"#, r#"켖"#),
        (r#"켗"#, r#"켗"#, r#"켗"#, r#"켗"#, r#"켗"#),
        (r#"켘"#, r#"켘"#, r#"켘"#, r#"켘"#, r#"켘"#),
        (r#"켙"#, r#"켙"#, r#"켙"#, r#"켙"#, r#"켙"#),
        (r#"켚"#, r#"켚"#, r#"켚"#, r#"켚"#, r#"켚"#),
        (r#"켛"#, r#"켛"#, r#"켛"#, r#"켛"#, r#"켛"#),
        (r#"켜"#, r#"켜"#, r#"켜"#, r#"켜"#, r#"켜"#),
        (r#"켝"#, r#"켝"#, r#"켝"#, r#"켝"#, r#"켝"#),
        (r#"켞"#, r#"켞"#, r#"켞"#, r#"켞"#, r#"켞"#),
        (r#"켟"#, r#"켟"#, r#"켟"#, r#"켟"#, r#"켟"#),
        (r#"켠"#, r#"켠"#, r#"켠"#, r#"켠"#, r#"켠"#),
        (r#"켡"#, r#"켡"#, r#"켡"#, r#"켡"#, r#"켡"#),
        (r#"켢"#, r#"켢"#, r#"켢"#, r#"켢"#, r#"켢"#),
        (r#"켣"#, r#"켣"#, r#"켣"#, r#"켣"#, r#"켣"#),
        (r#"켤"#, r#"켤"#, r#"켤"#, r#"켤"#, r#"켤"#),
        (r#"켥"#, r#"켥"#, r#"켥"#, r#"켥"#, r#"켥"#),
        (r#"켦"#, r#"켦"#, r#"켦"#, r#"켦"#, r#"켦"#),
        (r#"켧"#, r#"켧"#, r#"켧"#, r#"켧"#, r#"켧"#),
        (r#"켨"#, r#"켨"#, r#"켨"#, r#"켨"#, r#"켨"#),
        (r#"켩"#, r#"켩"#, r#"켩"#, r#"켩"#, r#"켩"#),
        (r#"켪"#, r#"켪"#, r#"켪"#, r#"켪"#, r#"켪"#),
        (r#"켫"#, r#"켫"#, r#"켫"#, r#"켫"#, r#"켫"#),
        (r#"켬"#, r#"켬"#, r#"켬"#, r#"켬"#, r#"켬"#),
        (r#"켭"#, r#"켭"#, r#"켭"#, r#"켭"#, r#"켭"#),
        (r#"켮"#, r#"켮"#, r#"켮"#, r#"켮"#, r#"켮"#),
        (r#"켯"#, r#"켯"#, r#"켯"#, r#"켯"#, r#"켯"#),
        (r#"켰"#, r#"켰"#, r#"켰"#, r#"켰"#, r#"켰"#),
        (r#"켱"#, r#"켱"#, r#"켱"#, r#"켱"#, r#"켱"#),
        (r#"켲"#, r#"켲"#, r#"켲"#, r#"켲"#, r#"켲"#),
        (r#"켳"#, r#"켳"#, r#"켳"#, r#"켳"#, r#"켳"#),
        (r#"켴"#, r#"켴"#, r#"켴"#, r#"켴"#, r#"켴"#),
        (r#"켵"#, r#"켵"#, r#"켵"#, r#"켵"#, r#"켵"#),
        (r#"켶"#, r#"켶"#, r#"켶"#, r#"켶"#, r#"켶"#),
        (r#"켷"#, r#"켷"#, r#"켷"#, r#"켷"#, r#"켷"#),
        (r#"켸"#, r#"켸"#, r#"켸"#, r#"켸"#, r#"켸"#),
        (r#"켹"#, r#"켹"#, r#"켹"#, r#"켹"#, r#"켹"#),
        (r#"켺"#, r#"켺"#, r#"켺"#, r#"켺"#, r#"켺"#),
        (r#"켻"#, r#"켻"#, r#"켻"#, r#"켻"#, r#"켻"#),
        (r#"켼"#, r#"켼"#, r#"켼"#, r#"켼"#, r#"켼"#),
        (r#"켽"#, r#"켽"#, r#"켽"#, r#"켽"#, r#"켽"#),
        (r#"켾"#, r#"켾"#, r#"켾"#, r#"켾"#, r#"켾"#),
        (r#"켿"#, r#"켿"#, r#"켿"#, r#"켿"#, r#"켿"#),
        (r#"콀"#, r#"콀"#, r#"콀"#, r#"콀"#, r#"콀"#),
        (r#"콁"#, r#"콁"#, r#"콁"#, r#"콁"#, r#"콁"#),
        (r#"콂"#, r#"콂"#, r#"콂"#, r#"콂"#, r#"콂"#),
        (r#"콃"#, r#"콃"#, r#"콃"#, r#"콃"#, r#"콃"#),
        (r#"콄"#, r#"콄"#, r#"콄"#, r#"콄"#, r#"콄"#),
        (r#"콅"#, r#"콅"#, r#"콅"#, r#"콅"#, r#"콅"#),
        (r#"콆"#, r#"콆"#, r#"콆"#, r#"콆"#, r#"콆"#),
        (r#"콇"#, r#"콇"#, r#"콇"#, r#"콇"#, r#"콇"#),
        (r#"콈"#, r#"콈"#, r#"콈"#, r#"콈"#, r#"콈"#),
        (r#"콉"#, r#"콉"#, r#"콉"#, r#"콉"#, r#"콉"#),
        (r#"콊"#, r#"콊"#, r#"콊"#, r#"콊"#, r#"콊"#),
        (r#"콋"#, r#"콋"#, r#"콋"#, r#"콋"#, r#"콋"#),
        (r#"콌"#, r#"콌"#, r#"콌"#, r#"콌"#, r#"콌"#),
        (r#"콍"#, r#"콍"#, r#"콍"#, r#"콍"#, r#"콍"#),
        (r#"콎"#, r#"콎"#, r#"콎"#, r#"콎"#, r#"콎"#),
        (r#"콏"#, r#"콏"#, r#"콏"#, r#"콏"#, r#"콏"#),
        (r#"콐"#, r#"콐"#, r#"콐"#, r#"콐"#, r#"콐"#),
        (r#"콑"#, r#"콑"#, r#"콑"#, r#"콑"#, r#"콑"#),
        (r#"콒"#, r#"콒"#, r#"콒"#, r#"콒"#, r#"콒"#),
        (r#"콓"#, r#"콓"#, r#"콓"#, r#"콓"#, r#"콓"#),
        (r#"코"#, r#"코"#, r#"코"#, r#"코"#, r#"코"#),
        (r#"콕"#, r#"콕"#, r#"콕"#, r#"콕"#, r#"콕"#),
        (r#"콖"#, r#"콖"#, r#"콖"#, r#"콖"#, r#"콖"#),
        (r#"콗"#, r#"콗"#, r#"콗"#, r#"콗"#, r#"콗"#),
        (r#"콘"#, r#"콘"#, r#"콘"#, r#"콘"#, r#"콘"#),
        (r#"콙"#, r#"콙"#, r#"콙"#, r#"콙"#, r#"콙"#),
        (r#"콚"#, r#"콚"#, r#"콚"#, r#"콚"#, r#"콚"#),
        (r#"콛"#, r#"콛"#, r#"콛"#, r#"콛"#, r#"콛"#),
        (r#"콜"#, r#"콜"#, r#"콜"#, r#"콜"#, r#"콜"#),
        (r#"콝"#, r#"콝"#, r#"콝"#, r#"콝"#, r#"콝"#),
        (r#"콞"#, r#"콞"#, r#"콞"#, r#"콞"#, r#"콞"#),
        (r#"콟"#, r#"콟"#, r#"콟"#, r#"콟"#, r#"콟"#),
        (r#"콠"#, r#"콠"#, r#"콠"#, r#"콠"#, r#"콠"#),
        (r#"콡"#, r#"콡"#, r#"콡"#, r#"콡"#, r#"콡"#),
        (r#"콢"#, r#"콢"#, r#"콢"#, r#"콢"#, r#"콢"#),
        (r#"콣"#, r#"콣"#, r#"콣"#, r#"콣"#, r#"콣"#),
        (r#"콤"#, r#"콤"#, r#"콤"#, r#"콤"#, r#"콤"#),
        (r#"콥"#, r#"콥"#, r#"콥"#, r#"콥"#, r#"콥"#),
        (r#"콦"#, r#"콦"#, r#"콦"#, r#"콦"#, r#"콦"#),
        (r#"콧"#, r#"콧"#, r#"콧"#, r#"콧"#, r#"콧"#),
        (r#"콨"#, r#"콨"#, r#"콨"#, r#"콨"#, r#"콨"#),
        (r#"콩"#, r#"콩"#, r#"콩"#, r#"콩"#, r#"콩"#),
        (r#"콪"#, r#"콪"#, r#"콪"#, r#"콪"#, r#"콪"#),
        (r#"콫"#, r#"콫"#, r#"콫"#, r#"콫"#, r#"콫"#),
        (r#"콬"#, r#"콬"#, r#"콬"#, r#"콬"#, r#"콬"#),
        (r#"콭"#, r#"콭"#, r#"콭"#, r#"콭"#, r#"콭"#),
        (r#"콮"#, r#"콮"#, r#"콮"#, r#"콮"#, r#"콮"#),
        (r#"콯"#, r#"콯"#, r#"콯"#, r#"콯"#, r#"콯"#),
        (r#"콰"#, r#"콰"#, r#"콰"#, r#"콰"#, r#"콰"#),
        (r#"콱"#, r#"콱"#, r#"콱"#, r#"콱"#, r#"콱"#),
        (r#"콲"#, r#"콲"#, r#"콲"#, r#"콲"#, r#"콲"#),
        (r#"콳"#, r#"콳"#, r#"콳"#, r#"콳"#, r#"콳"#),
        (r#"콴"#, r#"콴"#, r#"콴"#, r#"콴"#, r#"콴"#),
        (r#"콵"#, r#"콵"#, r#"콵"#, r#"콵"#, r#"콵"#),
        (r#"콶"#, r#"콶"#, r#"콶"#, r#"콶"#, r#"콶"#),
        (r#"콷"#, r#"콷"#, r#"콷"#, r#"콷"#, r#"콷"#),
        (r#"콸"#, r#"콸"#, r#"콸"#, r#"콸"#, r#"콸"#),
        (r#"콹"#, r#"콹"#, r#"콹"#, r#"콹"#, r#"콹"#),
        (r#"콺"#, r#"콺"#, r#"콺"#, r#"콺"#, r#"콺"#),
        (r#"콻"#, r#"콻"#, r#"콻"#, r#"콻"#, r#"콻"#),
        (r#"콼"#, r#"콼"#, r#"콼"#, r#"콼"#, r#"콼"#),
        (r#"콽"#, r#"콽"#, r#"콽"#, r#"콽"#, r#"콽"#),
        (r#"콾"#, r#"콾"#, r#"콾"#, r#"콾"#, r#"콾"#),
        (r#"콿"#, r#"콿"#, r#"콿"#, r#"콿"#, r#"콿"#),
        (r#"쾀"#, r#"쾀"#, r#"쾀"#, r#"쾀"#, r#"쾀"#),
        (r#"쾁"#, r#"쾁"#, r#"쾁"#, r#"쾁"#, r#"쾁"#),
        (r#"쾂"#, r#"쾂"#, r#"쾂"#, r#"쾂"#, r#"쾂"#),
        (r#"쾃"#, r#"쾃"#, r#"쾃"#, r#"쾃"#, r#"쾃"#),
        (r#"쾄"#, r#"쾄"#, r#"쾄"#, r#"쾄"#, r#"쾄"#),
        (r#"쾅"#, r#"쾅"#, r#"쾅"#, r#"쾅"#, r#"쾅"#),
        (r#"쾆"#, r#"쾆"#, r#"쾆"#, r#"쾆"#, r#"쾆"#),
        (r#"쾇"#, r#"쾇"#, r#"쾇"#, r#"쾇"#, r#"쾇"#),
        (r#"쾈"#, r#"쾈"#, r#"쾈"#, r#"쾈"#, r#"쾈"#),
        (r#"쾉"#, r#"쾉"#, r#"쾉"#, r#"쾉"#, r#"쾉"#),
        (r#"쾊"#, r#"쾊"#, r#"쾊"#, r#"쾊"#, r#"쾊"#),
        (r#"쾋"#, r#"쾋"#, r#"쾋"#, r#"쾋"#, r#"쾋"#),
        (r#"쾌"#, r#"쾌"#, r#"쾌"#, r#"쾌"#, r#"쾌"#),
        (r#"쾍"#, r#"쾍"#, r#"쾍"#, r#"쾍"#, r#"쾍"#),
        (r#"쾎"#, r#"쾎"#, r#"쾎"#, r#"쾎"#, r#"쾎"#),
        (r#"쾏"#, r#"쾏"#, r#"쾏"#, r#"쾏"#, r#"쾏"#),
        (r#"쾐"#, r#"쾐"#, r#"쾐"#, r#"쾐"#, r#"쾐"#),
        (r#"쾑"#, r#"쾑"#, r#"쾑"#, r#"쾑"#, r#"쾑"#),
        (r#"쾒"#, r#"쾒"#, r#"쾒"#, r#"쾒"#, r#"쾒"#),
        (r#"쾓"#, r#"쾓"#, r#"쾓"#, r#"쾓"#, r#"쾓"#),
        (r#"쾔"#, r#"쾔"#, r#"쾔"#, r#"쾔"#, r#"쾔"#),
        (r#"쾕"#, r#"쾕"#, r#"쾕"#, r#"쾕"#, r#"쾕"#),
        (r#"쾖"#, r#"쾖"#, r#"쾖"#, r#"쾖"#, r#"쾖"#),
        (r#"쾗"#, r#"쾗"#, r#"쾗"#, r#"쾗"#, r#"쾗"#),
        (r#"쾘"#, r#"쾘"#, r#"쾘"#, r#"쾘"#, r#"쾘"#),
        (r#"쾙"#, r#"쾙"#, r#"쾙"#, r#"쾙"#, r#"쾙"#),
        (r#"쾚"#, r#"쾚"#, r#"쾚"#, r#"쾚"#, r#"쾚"#),
        (r#"쾛"#, r#"쾛"#, r#"쾛"#, r#"쾛"#, r#"쾛"#),
        (r#"쾜"#, r#"쾜"#, r#"쾜"#, r#"쾜"#, r#"쾜"#),
        (r#"쾝"#, r#"쾝"#, r#"쾝"#, r#"쾝"#, r#"쾝"#),
        (r#"쾞"#, r#"쾞"#, r#"쾞"#, r#"쾞"#, r#"쾞"#),
        (r#"쾟"#, r#"쾟"#, r#"쾟"#, r#"쾟"#, r#"쾟"#),
        (r#"쾠"#, r#"쾠"#, r#"쾠"#, r#"쾠"#, r#"쾠"#),
        (r#"쾡"#, r#"쾡"#, r#"쾡"#, r#"쾡"#, r#"쾡"#),
        (r#"쾢"#, r#"쾢"#, r#"쾢"#, r#"쾢"#, r#"쾢"#),
        (r#"쾣"#, r#"쾣"#, r#"쾣"#, r#"쾣"#, r#"쾣"#),
        (r#"쾤"#, r#"쾤"#, r#"쾤"#, r#"쾤"#, r#"쾤"#),
        (r#"쾥"#, r#"쾥"#, r#"쾥"#, r#"쾥"#, r#"쾥"#),
        (r#"쾦"#, r#"쾦"#, r#"쾦"#, r#"쾦"#, r#"쾦"#),
        (r#"쾧"#, r#"쾧"#, r#"쾧"#, r#"쾧"#, r#"쾧"#),
        (r#"쾨"#, r#"쾨"#, r#"쾨"#, r#"쾨"#, r#"쾨"#),
        (r#"쾩"#, r#"쾩"#, r#"쾩"#, r#"쾩"#, r#"쾩"#),
        (r#"쾪"#, r#"쾪"#, r#"쾪"#, r#"쾪"#, r#"쾪"#),
        (r#"쾫"#, r#"쾫"#, r#"쾫"#, r#"쾫"#, r#"쾫"#),
        (r#"쾬"#, r#"쾬"#, r#"쾬"#, r#"쾬"#, r#"쾬"#),
        (r#"쾭"#, r#"쾭"#, r#"쾭"#, r#"쾭"#, r#"쾭"#),
        (r#"쾮"#, r#"쾮"#, r#"쾮"#, r#"쾮"#, r#"쾮"#),
        (r#"쾯"#, r#"쾯"#, r#"쾯"#, r#"쾯"#, r#"쾯"#),
        (r#"쾰"#, r#"쾰"#, r#"쾰"#, r#"쾰"#, r#"쾰"#),
        (r#"쾱"#, r#"쾱"#, r#"쾱"#, r#"쾱"#, r#"쾱"#),
        (r#"쾲"#, r#"쾲"#, r#"쾲"#, r#"쾲"#, r#"쾲"#),
        (r#"쾳"#, r#"쾳"#, r#"쾳"#, r#"쾳"#, r#"쾳"#),
        (r#"쾴"#, r#"쾴"#, r#"쾴"#, r#"쾴"#, r#"쾴"#),
        (r#"쾵"#, r#"쾵"#, r#"쾵"#, r#"쾵"#, r#"쾵"#),
        (r#"쾶"#, r#"쾶"#, r#"쾶"#, r#"쾶"#, r#"쾶"#),
        (r#"쾷"#, r#"쾷"#, r#"쾷"#, r#"쾷"#, r#"쾷"#),
        (r#"쾸"#, r#"쾸"#, r#"쾸"#, r#"쾸"#, r#"쾸"#),
        (r#"쾹"#, r#"쾹"#, r#"쾹"#, r#"쾹"#, r#"쾹"#),
        (r#"쾺"#, r#"쾺"#, r#"쾺"#, r#"쾺"#, r#"쾺"#),
        (r#"쾻"#, r#"쾻"#, r#"쾻"#, r#"쾻"#, r#"쾻"#),
        (r#"쾼"#, r#"쾼"#, r#"쾼"#, r#"쾼"#, r#"쾼"#),
        (r#"쾽"#, r#"쾽"#, r#"쾽"#, r#"쾽"#, r#"쾽"#),
        (r#"쾾"#, r#"쾾"#, r#"쾾"#, r#"쾾"#, r#"쾾"#),
        (r#"쾿"#, r#"쾿"#, r#"쾿"#, r#"쾿"#, r#"쾿"#),
        (r#"쿀"#, r#"쿀"#, r#"쿀"#, r#"쿀"#, r#"쿀"#),
        (r#"쿁"#, r#"쿁"#, r#"쿁"#, r#"쿁"#, r#"쿁"#),
        (r#"쿂"#, r#"쿂"#, r#"쿂"#, r#"쿂"#, r#"쿂"#),
        (r#"쿃"#, r#"쿃"#, r#"쿃"#, r#"쿃"#, r#"쿃"#),
        (r#"쿄"#, r#"쿄"#, r#"쿄"#, r#"쿄"#, r#"쿄"#),
        (r#"쿅"#, r#"쿅"#, r#"쿅"#, r#"쿅"#, r#"쿅"#),
        (r#"쿆"#, r#"쿆"#, r#"쿆"#, r#"쿆"#, r#"쿆"#),
        (r#"쿇"#, r#"쿇"#, r#"쿇"#, r#"쿇"#, r#"쿇"#),
        (r#"쿈"#, r#"쿈"#, r#"쿈"#, r#"쿈"#, r#"쿈"#),
        (r#"쿉"#, r#"쿉"#, r#"쿉"#, r#"쿉"#, r#"쿉"#),
        (r#"쿊"#, r#"쿊"#, r#"쿊"#, r#"쿊"#, r#"쿊"#),
        (r#"쿋"#, r#"쿋"#, r#"쿋"#, r#"쿋"#, r#"쿋"#),
        (r#"쿌"#, r#"쿌"#, r#"쿌"#, r#"쿌"#, r#"쿌"#),
        (r#"쿍"#, r#"쿍"#, r#"쿍"#, r#"쿍"#, r#"쿍"#),
        (r#"쿎"#, r#"쿎"#, r#"쿎"#, r#"쿎"#, r#"쿎"#),
        (r#"쿏"#, r#"쿏"#, r#"쿏"#, r#"쿏"#, r#"쿏"#),
        (r#"쿐"#, r#"쿐"#, r#"쿐"#, r#"쿐"#, r#"쿐"#),
        (r#"쿑"#, r#"쿑"#, r#"쿑"#, r#"쿑"#, r#"쿑"#),
        (r#"쿒"#, r#"쿒"#, r#"쿒"#, r#"쿒"#, r#"쿒"#),
        (r#"쿓"#, r#"쿓"#, r#"쿓"#, r#"쿓"#, r#"쿓"#),
        (r#"쿔"#, r#"쿔"#, r#"쿔"#, r#"쿔"#, r#"쿔"#),
        (r#"쿕"#, r#"쿕"#, r#"쿕"#, r#"쿕"#, r#"쿕"#),
        (r#"쿖"#, r#"쿖"#, r#"쿖"#, r#"쿖"#, r#"쿖"#),
        (r#"쿗"#, r#"쿗"#, r#"쿗"#, r#"쿗"#, r#"쿗"#),
        (r#"쿘"#, r#"쿘"#, r#"쿘"#, r#"쿘"#, r#"쿘"#),
        (r#"쿙"#, r#"쿙"#, r#"쿙"#, r#"쿙"#, r#"쿙"#),
        (r#"쿚"#, r#"쿚"#, r#"쿚"#, r#"쿚"#, r#"쿚"#),
        (r#"쿛"#, r#"쿛"#, r#"쿛"#, r#"쿛"#, r#"쿛"#),
        (r#"쿜"#, r#"쿜"#, r#"쿜"#, r#"쿜"#, r#"쿜"#),
        (r#"쿝"#, r#"쿝"#, r#"쿝"#, r#"쿝"#, r#"쿝"#),
        (r#"쿞"#, r#"쿞"#, r#"쿞"#, r#"쿞"#, r#"쿞"#),
        (r#"쿟"#, r#"쿟"#, r#"쿟"#, r#"쿟"#, r#"쿟"#),
        (r#"쿠"#, r#"쿠"#, r#"쿠"#, r#"쿠"#, r#"쿠"#),
        (r#"쿡"#, r#"쿡"#, r#"쿡"#, r#"쿡"#, r#"쿡"#),
        (r#"쿢"#, r#"쿢"#, r#"쿢"#, r#"쿢"#, r#"쿢"#),
        (r#"쿣"#, r#"쿣"#, r#"쿣"#, r#"쿣"#, r#"쿣"#),
        (r#"쿤"#, r#"쿤"#, r#"쿤"#, r#"쿤"#, r#"쿤"#),
        (r#"쿥"#, r#"쿥"#, r#"쿥"#, r#"쿥"#, r#"쿥"#),
        (r#"쿦"#, r#"쿦"#, r#"쿦"#, r#"쿦"#, r#"쿦"#),
        (r#"쿧"#, r#"쿧"#, r#"쿧"#, r#"쿧"#, r#"쿧"#),
        (r#"쿨"#, r#"쿨"#, r#"쿨"#, r#"쿨"#, r#"쿨"#),
        (r#"쿩"#, r#"쿩"#, r#"쿩"#, r#"쿩"#, r#"쿩"#),
        (r#"쿪"#, r#"쿪"#, r#"쿪"#, r#"쿪"#, r#"쿪"#),
        (r#"쿫"#, r#"쿫"#, r#"쿫"#, r#"쿫"#, r#"쿫"#),
        (r#"쿬"#, r#"쿬"#, r#"쿬"#, r#"쿬"#, r#"쿬"#),
        (r#"쿭"#, r#"쿭"#, r#"쿭"#, r#"쿭"#, r#"쿭"#),
        (r#"쿮"#, r#"쿮"#, r#"쿮"#, r#"쿮"#, r#"쿮"#),
        (r#"쿯"#, r#"쿯"#, r#"쿯"#, r#"쿯"#, r#"쿯"#),
        (r#"쿰"#, r#"쿰"#, r#"쿰"#, r#"쿰"#, r#"쿰"#),
        (r#"쿱"#, r#"쿱"#, r#"쿱"#, r#"쿱"#, r#"쿱"#),
        (r#"쿲"#, r#"쿲"#, r#"쿲"#, r#"쿲"#, r#"쿲"#),
        (r#"쿳"#, r#"쿳"#, r#"쿳"#, r#"쿳"#, r#"쿳"#),
        (r#"쿴"#, r#"쿴"#, r#"쿴"#, r#"쿴"#, r#"쿴"#),
        (r#"쿵"#, r#"쿵"#, r#"쿵"#, r#"쿵"#, r#"쿵"#),
        (r#"쿶"#, r#"쿶"#, r#"쿶"#, r#"쿶"#, r#"쿶"#),
        (r#"쿷"#, r#"쿷"#, r#"쿷"#, r#"쿷"#, r#"쿷"#),
        (r#"쿸"#, r#"쿸"#, r#"쿸"#, r#"쿸"#, r#"쿸"#),
        (r#"쿹"#, r#"쿹"#, r#"쿹"#, r#"쿹"#, r#"쿹"#),
        (r#"쿺"#, r#"쿺"#, r#"쿺"#, r#"쿺"#, r#"쿺"#),
        (r#"쿻"#, r#"쿻"#, r#"쿻"#, r#"쿻"#, r#"쿻"#),
        (r#"쿼"#, r#"쿼"#, r#"쿼"#, r#"쿼"#, r#"쿼"#),
        (r#"쿽"#, r#"쿽"#, r#"쿽"#, r#"쿽"#, r#"쿽"#),
        (r#"쿾"#, r#"쿾"#, r#"쿾"#, r#"쿾"#, r#"쿾"#),
        (r#"쿿"#, r#"쿿"#, r#"쿿"#, r#"쿿"#, r#"쿿"#),
        (r#"퀀"#, r#"퀀"#, r#"퀀"#, r#"퀀"#, r#"퀀"#),
        (r#"퀁"#, r#"퀁"#, r#"퀁"#, r#"퀁"#, r#"퀁"#),
        (r#"퀂"#, r#"퀂"#, r#"퀂"#, r#"퀂"#, r#"퀂"#),
        (r#"퀃"#, r#"퀃"#, r#"퀃"#, r#"퀃"#, r#"퀃"#),
        (r#"퀄"#, r#"퀄"#, r#"퀄"#, r#"퀄"#, r#"퀄"#),
        (r#"퀅"#, r#"퀅"#, r#"퀅"#, r#"퀅"#, r#"퀅"#),
        (r#"퀆"#, r#"퀆"#, r#"퀆"#, r#"퀆"#, r#"퀆"#),
        (r#"퀇"#, r#"퀇"#, r#"퀇"#, r#"퀇"#, r#"퀇"#),
        (r#"퀈"#, r#"퀈"#, r#"퀈"#, r#"퀈"#, r#"퀈"#),
        (r#"퀉"#, r#"퀉"#, r#"퀉"#, r#"퀉"#, r#"퀉"#),
        (r#"퀊"#, r#"퀊"#, r#"퀊"#, r#"퀊"#, r#"퀊"#),
        (r#"퀋"#, r#"퀋"#, r#"퀋"#, r#"퀋"#, r#"퀋"#),
        (r#"퀌"#, r#"퀌"#, r#"퀌"#, r#"퀌"#, r#"퀌"#),
        (r#"퀍"#, r#"퀍"#, r#"퀍"#, r#"퀍"#, r#"퀍"#),
        (r#"퀎"#, r#"퀎"#, r#"퀎"#, r#"퀎"#, r#"퀎"#),
        (r#"퀏"#, r#"퀏"#, r#"퀏"#, r#"퀏"#, r#"퀏"#),
        (r#"퀐"#, r#"퀐"#, r#"퀐"#, r#"퀐"#, r#"퀐"#),
        (r#"퀑"#, r#"퀑"#, r#"퀑"#, r#"퀑"#, r#"퀑"#),
        (r#"퀒"#, r#"퀒"#, r#"퀒"#, r#"퀒"#, r#"퀒"#),
        (r#"퀓"#, r#"퀓"#, r#"퀓"#, r#"퀓"#, r#"퀓"#),
        (r#"퀔"#, r#"퀔"#, r#"퀔"#, r#"퀔"#, r#"퀔"#),
        (r#"퀕"#, r#"퀕"#, r#"퀕"#, r#"퀕"#, r#"퀕"#),
        (r#"퀖"#, r#"퀖"#, r#"퀖"#, r#"퀖"#, r#"퀖"#),
        (r#"퀗"#, r#"퀗"#, r#"퀗"#, r#"퀗"#, r#"퀗"#),
        (r#"퀘"#, r#"퀘"#, r#"퀘"#, r#"퀘"#, r#"퀘"#),
        (r#"퀙"#, r#"퀙"#, r#"퀙"#, r#"퀙"#, r#"퀙"#),
        (r#"퀚"#, r#"퀚"#, r#"퀚"#, r#"퀚"#, r#"퀚"#),
        (r#"퀛"#, r#"퀛"#, r#"퀛"#, r#"퀛"#, r#"퀛"#),
        (r#"퀜"#, r#"퀜"#, r#"퀜"#, r#"퀜"#, r#"퀜"#),
        (r#"퀝"#, r#"퀝"#, r#"퀝"#, r#"퀝"#, r#"퀝"#),
        (r#"퀞"#, r#"퀞"#, r#"퀞"#, r#"퀞"#, r#"퀞"#),
        (r#"퀟"#, r#"퀟"#, r#"퀟"#, r#"퀟"#, r#"퀟"#),
        (r#"퀠"#, r#"퀠"#, r#"퀠"#, r#"퀠"#, r#"퀠"#),
        (r#"퀡"#, r#"퀡"#, r#"퀡"#, r#"퀡"#, r#"퀡"#),
        (r#"퀢"#, r#"퀢"#, r#"퀢"#, r#"퀢"#, r#"퀢"#),
        (r#"퀣"#, r#"퀣"#, r#"퀣"#, r#"퀣"#, r#"퀣"#),
        (r#"퀤"#, r#"퀤"#, r#"퀤"#, r#"퀤"#, r#"퀤"#),
        (r#"퀥"#, r#"퀥"#, r#"퀥"#, r#"퀥"#, r#"퀥"#),
        (r#"퀦"#, r#"퀦"#, r#"퀦"#, r#"퀦"#, r#"퀦"#),
        (r#"퀧"#, r#"퀧"#, r#"퀧"#, r#"퀧"#, r#"퀧"#),
        (r#"퀨"#, r#"퀨"#, r#"퀨"#, r#"퀨"#, r#"퀨"#),
        (r#"퀩"#, r#"퀩"#, r#"퀩"#, r#"퀩"#, r#"퀩"#),
        (r#"퀪"#, r#"퀪"#, r#"퀪"#, r#"퀪"#, r#"퀪"#),
        (r#"퀫"#, r#"퀫"#, r#"퀫"#, r#"퀫"#, r#"퀫"#),
        (r#"퀬"#, r#"퀬"#, r#"퀬"#, r#"퀬"#, r#"퀬"#),
        (r#"퀭"#, r#"퀭"#, r#"퀭"#, r#"퀭"#, r#"퀭"#),
        (r#"퀮"#, r#"퀮"#, r#"퀮"#, r#"퀮"#, r#"퀮"#),
        (r#"퀯"#, r#"퀯"#, r#"퀯"#, r#"퀯"#, r#"퀯"#),
        (r#"퀰"#, r#"퀰"#, r#"퀰"#, r#"퀰"#, r#"퀰"#),
        (r#"퀱"#, r#"퀱"#, r#"퀱"#, r#"퀱"#, r#"퀱"#),
        (r#"퀲"#, r#"퀲"#, r#"퀲"#, r#"퀲"#, r#"퀲"#),
        (r#"퀳"#, r#"퀳"#, r#"퀳"#, r#"퀳"#, r#"퀳"#),
        (r#"퀴"#, r#"퀴"#, r#"퀴"#, r#"퀴"#, r#"퀴"#),
        (r#"퀵"#, r#"퀵"#, r#"퀵"#, r#"퀵"#, r#"퀵"#),
        (r#"퀶"#, r#"퀶"#, r#"퀶"#, r#"퀶"#, r#"퀶"#),
        (r#"퀷"#, r#"퀷"#, r#"퀷"#, r#"퀷"#, r#"퀷"#),
        (r#"퀸"#, r#"퀸"#, r#"퀸"#, r#"퀸"#, r#"퀸"#),
        (r#"퀹"#, r#"퀹"#, r#"퀹"#, r#"퀹"#, r#"퀹"#),
        (r#"퀺"#, r#"퀺"#, r#"퀺"#, r#"퀺"#, r#"퀺"#),
        (r#"퀻"#, r#"퀻"#, r#"퀻"#, r#"퀻"#, r#"퀻"#),
        (r#"퀼"#, r#"퀼"#, r#"퀼"#, r#"퀼"#, r#"퀼"#),
        (r#"퀽"#, r#"퀽"#, r#"퀽"#, r#"퀽"#, r#"퀽"#),
        (r#"퀾"#, r#"퀾"#, r#"퀾"#, r#"퀾"#, r#"퀾"#),
        (r#"퀿"#, r#"퀿"#, r#"퀿"#, r#"퀿"#, r#"퀿"#),
        (r#"큀"#, r#"큀"#, r#"큀"#, r#"큀"#, r#"큀"#),
        (r#"큁"#, r#"큁"#, r#"큁"#, r#"큁"#, r#"큁"#),
        (r#"큂"#, r#"큂"#, r#"큂"#, r#"큂"#, r#"큂"#),
        (r#"큃"#, r#"큃"#, r#"큃"#, r#"큃"#, r#"큃"#),
        (r#"큄"#, r#"큄"#, r#"큄"#, r#"큄"#, r#"큄"#),
        (r#"큅"#, r#"큅"#, r#"큅"#, r#"큅"#, r#"큅"#),
        (r#"큆"#, r#"큆"#, r#"큆"#, r#"큆"#, r#"큆"#),
        (r#"큇"#, r#"큇"#, r#"큇"#, r#"큇"#, r#"큇"#),
        (r#"큈"#, r#"큈"#, r#"큈"#, r#"큈"#, r#"큈"#),
        (r#"큉"#, r#"큉"#, r#"큉"#, r#"큉"#, r#"큉"#),
        (r#"큊"#, r#"큊"#, r#"큊"#, r#"큊"#, r#"큊"#),
        (r#"큋"#, r#"큋"#, r#"큋"#, r#"큋"#, r#"큋"#),
        (r#"큌"#, r#"큌"#, r#"큌"#, r#"큌"#, r#"큌"#),
        (r#"큍"#, r#"큍"#, r#"큍"#, r#"큍"#, r#"큍"#),
        (r#"큎"#, r#"큎"#, r#"큎"#, r#"큎"#, r#"큎"#),
        (r#"큏"#, r#"큏"#, r#"큏"#, r#"큏"#, r#"큏"#),
        (r#"큐"#, r#"큐"#, r#"큐"#, r#"큐"#, r#"큐"#),
        (r#"큑"#, r#"큑"#, r#"큑"#, r#"큑"#, r#"큑"#),
        (r#"큒"#, r#"큒"#, r#"큒"#, r#"큒"#, r#"큒"#),
        (r#"큓"#, r#"큓"#, r#"큓"#, r#"큓"#, r#"큓"#),
        (r#"큔"#, r#"큔"#, r#"큔"#, r#"큔"#, r#"큔"#),
        (r#"큕"#, r#"큕"#, r#"큕"#, r#"큕"#, r#"큕"#),
        (r#"큖"#, r#"큖"#, r#"큖"#, r#"큖"#, r#"큖"#),
        (r#"큗"#, r#"큗"#, r#"큗"#, r#"큗"#, r#"큗"#),
        (r#"큘"#, r#"큘"#, r#"큘"#, r#"큘"#, r#"큘"#),
        (r#"큙"#, r#"큙"#, r#"큙"#, r#"큙"#, r#"큙"#),
        (r#"큚"#, r#"큚"#, r#"큚"#, r#"큚"#, r#"큚"#),
        (r#"큛"#, r#"큛"#, r#"큛"#, r#"큛"#, r#"큛"#),
        (r#"큜"#, r#"큜"#, r#"큜"#, r#"큜"#, r#"큜"#),
        (r#"큝"#, r#"큝"#, r#"큝"#, r#"큝"#, r#"큝"#),
        (r#"큞"#, r#"큞"#, r#"큞"#, r#"큞"#, r#"큞"#),
        (r#"큟"#, r#"큟"#, r#"큟"#, r#"큟"#, r#"큟"#),
        (r#"큠"#, r#"큠"#, r#"큠"#, r#"큠"#, r#"큠"#),
        (r#"큡"#, r#"큡"#, r#"큡"#, r#"큡"#, r#"큡"#),
        (r#"큢"#, r#"큢"#, r#"큢"#, r#"큢"#, r#"큢"#),
        (r#"큣"#, r#"큣"#, r#"큣"#, r#"큣"#, r#"큣"#),
        (r#"큤"#, r#"큤"#, r#"큤"#, r#"큤"#, r#"큤"#),
        (r#"큥"#, r#"큥"#, r#"큥"#, r#"큥"#, r#"큥"#),
        (r#"큦"#, r#"큦"#, r#"큦"#, r#"큦"#, r#"큦"#),
        (r#"큧"#, r#"큧"#, r#"큧"#, r#"큧"#, r#"큧"#),
        (r#"큨"#, r#"큨"#, r#"큨"#, r#"큨"#, r#"큨"#),
        (r#"큩"#, r#"큩"#, r#"큩"#, r#"큩"#, r#"큩"#),
        (r#"큪"#, r#"큪"#, r#"큪"#, r#"큪"#, r#"큪"#),
        (r#"큫"#, r#"큫"#, r#"큫"#, r#"큫"#, r#"큫"#),
        (r#"크"#, r#"크"#, r#"크"#, r#"크"#, r#"크"#),
        (r#"큭"#, r#"큭"#, r#"큭"#, r#"큭"#, r#"큭"#),
        (r#"큮"#, r#"큮"#, r#"큮"#, r#"큮"#, r#"큮"#),
        (r#"큯"#, r#"큯"#, r#"큯"#, r#"큯"#, r#"큯"#),
        (r#"큰"#, r#"큰"#, r#"큰"#, r#"큰"#, r#"큰"#),
        (r#"큱"#, r#"큱"#, r#"큱"#, r#"큱"#, r#"큱"#),
        (r#"큲"#, r#"큲"#, r#"큲"#, r#"큲"#, r#"큲"#),
        (r#"큳"#, r#"큳"#, r#"큳"#, r#"큳"#, r#"큳"#),
        (r#"클"#, r#"클"#, r#"클"#, r#"클"#, r#"클"#),
        (r#"큵"#, r#"큵"#, r#"큵"#, r#"큵"#, r#"큵"#),
        (r#"큶"#, r#"큶"#, r#"큶"#, r#"큶"#, r#"큶"#),
        (r#"큷"#, r#"큷"#, r#"큷"#, r#"큷"#, r#"큷"#),
        (r#"큸"#, r#"큸"#, r#"큸"#, r#"큸"#, r#"큸"#),
        (r#"큹"#, r#"큹"#, r#"큹"#, r#"큹"#, r#"큹"#),
        (r#"큺"#, r#"큺"#, r#"큺"#, r#"큺"#, r#"큺"#),
        (r#"큻"#, r#"큻"#, r#"큻"#, r#"큻"#, r#"큻"#),
        (r#"큼"#, r#"큼"#, r#"큼"#, r#"큼"#, r#"큼"#),
        (r#"큽"#, r#"큽"#, r#"큽"#, r#"큽"#, r#"큽"#),
        (r#"큾"#, r#"큾"#, r#"큾"#, r#"큾"#, r#"큾"#),
        (r#"큿"#, r#"큿"#, r#"큿"#, r#"큿"#, r#"큿"#),
        (r#"킀"#, r#"킀"#, r#"킀"#, r#"킀"#, r#"킀"#),
        (r#"킁"#, r#"킁"#, r#"킁"#, r#"킁"#, r#"킁"#),
        (r#"킂"#, r#"킂"#, r#"킂"#, r#"킂"#, r#"킂"#),
        (r#"킃"#, r#"킃"#, r#"킃"#, r#"킃"#, r#"킃"#),
        (r#"킄"#, r#"킄"#, r#"킄"#, r#"킄"#, r#"킄"#),
        (r#"킅"#, r#"킅"#, r#"킅"#, r#"킅"#, r#"킅"#),
        (r#"킆"#, r#"킆"#, r#"킆"#, r#"킆"#, r#"킆"#),
        (r#"킇"#, r#"킇"#, r#"킇"#, r#"킇"#, r#"킇"#),
        (r#"킈"#, r#"킈"#, r#"킈"#, r#"킈"#, r#"킈"#),
        (r#"킉"#, r#"킉"#, r#"킉"#, r#"킉"#, r#"킉"#),
        (r#"킊"#, r#"킊"#, r#"킊"#, r#"킊"#, r#"킊"#),
        (r#"킋"#, r#"킋"#, r#"킋"#, r#"킋"#, r#"킋"#),
        (r#"킌"#, r#"킌"#, r#"킌"#, r#"킌"#, r#"킌"#),
        (r#"킍"#, r#"킍"#, r#"킍"#, r#"킍"#, r#"킍"#),
        (r#"킎"#, r#"킎"#, r#"킎"#, r#"킎"#, r#"킎"#),
        (r#"킏"#, r#"킏"#, r#"킏"#, r#"킏"#, r#"킏"#),
        (r#"킐"#, r#"킐"#, r#"킐"#, r#"킐"#, r#"킐"#),
        (r#"킑"#, r#"킑"#, r#"킑"#, r#"킑"#, r#"킑"#),
        (r#"킒"#, r#"킒"#, r#"킒"#, r#"킒"#, r#"킒"#),
        (r#"킓"#, r#"킓"#, r#"킓"#, r#"킓"#, r#"킓"#),
        (r#"킔"#, r#"킔"#, r#"킔"#, r#"킔"#, r#"킔"#),
        (r#"킕"#, r#"킕"#, r#"킕"#, r#"킕"#, r#"킕"#),
        (r#"킖"#, r#"킖"#, r#"킖"#, r#"킖"#, r#"킖"#),
        (r#"킗"#, r#"킗"#, r#"킗"#, r#"킗"#, r#"킗"#),
        (r#"킘"#, r#"킘"#, r#"킘"#, r#"킘"#, r#"킘"#),
        (r#"킙"#, r#"킙"#, r#"킙"#, r#"킙"#, r#"킙"#),
        (r#"킚"#, r#"킚"#, r#"킚"#, r#"킚"#, r#"킚"#),
        (r#"킛"#, r#"킛"#, r#"킛"#, r#"킛"#, r#"킛"#),
        (r#"킜"#, r#"킜"#, r#"킜"#, r#"킜"#, r#"킜"#),
        (r#"킝"#, r#"킝"#, r#"킝"#, r#"킝"#, r#"킝"#),
        (r#"킞"#, r#"킞"#, r#"킞"#, r#"킞"#, r#"킞"#),
        (r#"킟"#, r#"킟"#, r#"킟"#, r#"킟"#, r#"킟"#),
        (r#"킠"#, r#"킠"#, r#"킠"#, r#"킠"#, r#"킠"#),
        (r#"킡"#, r#"킡"#, r#"킡"#, r#"킡"#, r#"킡"#),
        (r#"킢"#, r#"킢"#, r#"킢"#, r#"킢"#, r#"킢"#),
        (r#"킣"#, r#"킣"#, r#"킣"#, r#"킣"#, r#"킣"#),
        (r#"키"#, r#"키"#, r#"키"#, r#"키"#, r#"키"#),
        (r#"킥"#, r#"킥"#, r#"킥"#, r#"킥"#, r#"킥"#),
        (r#"킦"#, r#"킦"#, r#"킦"#, r#"킦"#, r#"킦"#),
        (r#"킧"#, r#"킧"#, r#"킧"#, r#"킧"#, r#"킧"#),
        (r#"킨"#, r#"킨"#, r#"킨"#, r#"킨"#, r#"킨"#),
        (r#"킩"#, r#"킩"#, r#"킩"#, r#"킩"#, r#"킩"#),
        (r#"킪"#, r#"킪"#, r#"킪"#, r#"킪"#, r#"킪"#),
        (r#"킫"#, r#"킫"#, r#"킫"#, r#"킫"#, r#"킫"#),
        (r#"킬"#, r#"킬"#, r#"킬"#, r#"킬"#, r#"킬"#),
        (r#"킭"#, r#"킭"#, r#"킭"#, r#"킭"#, r#"킭"#),
        (r#"킮"#, r#"킮"#, r#"킮"#, r#"킮"#, r#"킮"#),
        (r#"킯"#, r#"킯"#, r#"킯"#, r#"킯"#, r#"킯"#),
        (r#"킰"#, r#"킰"#, r#"킰"#, r#"킰"#, r#"킰"#),
        (r#"킱"#, r#"킱"#, r#"킱"#, r#"킱"#, r#"킱"#),
        (r#"킲"#, r#"킲"#, r#"킲"#, r#"킲"#, r#"킲"#),
        (r#"킳"#, r#"킳"#, r#"킳"#, r#"킳"#, r#"킳"#),
        (r#"킴"#, r#"킴"#, r#"킴"#, r#"킴"#, r#"킴"#),
        (r#"킵"#, r#"킵"#, r#"킵"#, r#"킵"#, r#"킵"#),
        (r#"킶"#, r#"킶"#, r#"킶"#, r#"킶"#, r#"킶"#),
        (r#"킷"#, r#"킷"#, r#"킷"#, r#"킷"#, r#"킷"#),
        (r#"킸"#, r#"킸"#, r#"킸"#, r#"킸"#, r#"킸"#),
        (r#"킹"#, r#"킹"#, r#"킹"#, r#"킹"#, r#"킹"#),
        (r#"킺"#, r#"킺"#, r#"킺"#, r#"킺"#, r#"킺"#),
        (r#"킻"#, r#"킻"#, r#"킻"#, r#"킻"#, r#"킻"#),
        (r#"킼"#, r#"킼"#, r#"킼"#, r#"킼"#, r#"킼"#),
        (r#"킽"#, r#"킽"#, r#"킽"#, r#"킽"#, r#"킽"#),
        (r#"킾"#, r#"킾"#, r#"킾"#, r#"킾"#, r#"킾"#),
        (r#"킿"#, r#"킿"#, r#"킿"#, r#"킿"#, r#"킿"#),
        (r#"타"#, r#"타"#, r#"타"#, r#"타"#, r#"타"#),
        (r#"탁"#, r#"탁"#, r#"탁"#, r#"탁"#, r#"탁"#),
        (r#"탂"#, r#"탂"#, r#"탂"#, r#"탂"#, r#"탂"#),
        (r#"탃"#, r#"탃"#, r#"탃"#, r#"탃"#, r#"탃"#),
        (r#"탄"#, r#"탄"#, r#"탄"#, r#"탄"#, r#"탄"#),
        (r#"탅"#, r#"탅"#, r#"탅"#, r#"탅"#, r#"탅"#),
        (r#"탆"#, r#"탆"#, r#"탆"#, r#"탆"#, r#"탆"#),
        (r#"탇"#, r#"탇"#, r#"탇"#, r#"탇"#, r#"탇"#),
        (r#"탈"#, r#"탈"#, r#"탈"#, r#"탈"#, r#"탈"#),
        (r#"탉"#, r#"탉"#, r#"탉"#, r#"탉"#, r#"탉"#),
        (r#"탊"#, r#"탊"#, r#"탊"#, r#"탊"#, r#"탊"#),
        (r#"탋"#, r#"탋"#, r#"탋"#, r#"탋"#, r#"탋"#),
        (r#"탌"#, r#"탌"#, r#"탌"#, r#"탌"#, r#"탌"#),
        (r#"탍"#, r#"탍"#, r#"탍"#, r#"탍"#, r#"탍"#),
        (r#"탎"#, r#"탎"#, r#"탎"#, r#"탎"#, r#"탎"#),
        (r#"탏"#, r#"탏"#, r#"탏"#, r#"탏"#, r#"탏"#),
        (r#"탐"#, r#"탐"#, r#"탐"#, r#"탐"#, r#"탐"#),
        (r#"탑"#, r#"탑"#, r#"탑"#, r#"탑"#, r#"탑"#),
        (r#"탒"#, r#"탒"#, r#"탒"#, r#"탒"#, r#"탒"#),
        (r#"탓"#, r#"탓"#, r#"탓"#, r#"탓"#, r#"탓"#),
        (r#"탔"#, r#"탔"#, r#"탔"#, r#"탔"#, r#"탔"#),
        (r#"탕"#, r#"탕"#, r#"탕"#, r#"탕"#, r#"탕"#),
        (r#"탖"#, r#"탖"#, r#"탖"#, r#"탖"#, r#"탖"#),
        (r#"탗"#, r#"탗"#, r#"탗"#, r#"탗"#, r#"탗"#),
        (r#"탘"#, r#"탘"#, r#"탘"#, r#"탘"#, r#"탘"#),
        (r#"탙"#, r#"탙"#, r#"탙"#, r#"탙"#, r#"탙"#),
        (r#"탚"#, r#"탚"#, r#"탚"#, r#"탚"#, r#"탚"#),
        (r#"탛"#, r#"탛"#, r#"탛"#, r#"탛"#, r#"탛"#),
        (r#"태"#, r#"태"#, r#"태"#, r#"태"#, r#"태"#),
        (r#"택"#, r#"택"#, r#"택"#, r#"택"#, r#"택"#),
        (r#"탞"#, r#"탞"#, r#"탞"#, r#"탞"#, r#"탞"#),
        (r#"탟"#, r#"탟"#, r#"탟"#, r#"탟"#, r#"탟"#),
        (r#"탠"#, r#"탠"#, r#"탠"#, r#"탠"#, r#"탠"#),
        (r#"탡"#, r#"탡"#, r#"탡"#, r#"탡"#, r#"탡"#),
        (r#"탢"#, r#"탢"#, r#"탢"#, r#"탢"#, r#"탢"#),
        (r#"탣"#, r#"탣"#, r#"탣"#, r#"탣"#, r#"탣"#),
        (r#"탤"#, r#"탤"#, r#"탤"#, r#"탤"#, r#"탤"#),
        (r#"탥"#, r#"탥"#, r#"탥"#, r#"탥"#, r#"탥"#),
        (r#"탦"#, r#"탦"#, r#"탦"#, r#"탦"#, r#"탦"#),
        (r#"탧"#, r#"탧"#, r#"탧"#, r#"탧"#, r#"탧"#),
        (r#"탨"#, r#"탨"#, r#"탨"#, r#"탨"#, r#"탨"#),
        (r#"탩"#, r#"탩"#, r#"탩"#, r#"탩"#, r#"탩"#),
        (r#"탪"#, r#"탪"#, r#"탪"#, r#"탪"#, r#"탪"#),
        (r#"탫"#, r#"탫"#, r#"탫"#, r#"탫"#, r#"탫"#),
        (r#"탬"#, r#"탬"#, r#"탬"#, r#"탬"#, r#"탬"#),
        (r#"탭"#, r#"탭"#, r#"탭"#, r#"탭"#, r#"탭"#),
        (r#"탮"#, r#"탮"#, r#"탮"#, r#"탮"#, r#"탮"#),
        (r#"탯"#, r#"탯"#, r#"탯"#, r#"탯"#, r#"탯"#),
        (r#"탰"#, r#"탰"#, r#"탰"#, r#"탰"#, r#"탰"#),
        (r#"탱"#, r#"탱"#, r#"탱"#, r#"탱"#, r#"탱"#),
        (r#"탲"#, r#"탲"#, r#"탲"#, r#"탲"#, r#"탲"#),
        (r#"탳"#, r#"탳"#, r#"탳"#, r#"탳"#, r#"탳"#),
        (r#"탴"#, r#"탴"#, r#"탴"#, r#"탴"#, r#"탴"#),
        (r#"탵"#, r#"탵"#, r#"탵"#, r#"탵"#, r#"탵"#),
        (r#"탶"#, r#"탶"#, r#"탶"#, r#"탶"#, r#"탶"#),
        (r#"탷"#, r#"탷"#, r#"탷"#, r#"탷"#, r#"탷"#),
        (r#"탸"#, r#"탸"#, r#"탸"#, r#"탸"#, r#"탸"#),
        (r#"탹"#, r#"탹"#, r#"탹"#, r#"탹"#, r#"탹"#),
        (r#"탺"#, r#"탺"#, r#"탺"#, r#"탺"#, r#"탺"#),
        (r#"탻"#, r#"탻"#, r#"탻"#, r#"탻"#, r#"탻"#),
        (r#"탼"#, r#"탼"#, r#"탼"#, r#"탼"#, r#"탼"#),
        (r#"탽"#, r#"탽"#, r#"탽"#, r#"탽"#, r#"탽"#),
        (r#"탾"#, r#"탾"#, r#"탾"#, r#"탾"#, r#"탾"#),
        (r#"탿"#, r#"탿"#, r#"탿"#, r#"탿"#, r#"탿"#),
        (r#"턀"#, r#"턀"#, r#"턀"#, r#"턀"#, r#"턀"#),
        (r#"턁"#, r#"턁"#, r#"턁"#, r#"턁"#, r#"턁"#),
        (r#"턂"#, r#"턂"#, r#"턂"#, r#"턂"#, r#"턂"#),
        (r#"턃"#, r#"턃"#, r#"턃"#, r#"턃"#, r#"턃"#),
        (r#"턄"#, r#"턄"#, r#"턄"#, r#"턄"#, r#"턄"#),
        (r#"턅"#, r#"턅"#, r#"턅"#, r#"턅"#, r#"턅"#),
        (r#"턆"#, r#"턆"#, r#"턆"#, r#"턆"#, r#"턆"#),
        (r#"턇"#, r#"턇"#, r#"턇"#, r#"턇"#, r#"턇"#),
        (r#"턈"#, r#"턈"#, r#"턈"#, r#"턈"#, r#"턈"#),
        (r#"턉"#, r#"턉"#, r#"턉"#, r#"턉"#, r#"턉"#),
        (r#"턊"#, r#"턊"#, r#"턊"#, r#"턊"#, r#"턊"#),
        (r#"턋"#, r#"턋"#, r#"턋"#, r#"턋"#, r#"턋"#),
        (r#"턌"#, r#"턌"#, r#"턌"#, r#"턌"#, r#"턌"#),
        (r#"턍"#, r#"턍"#, r#"턍"#, r#"턍"#, r#"턍"#),
        (r#"턎"#, r#"턎"#, r#"턎"#, r#"턎"#, r#"턎"#),
        (r#"턏"#, r#"턏"#, r#"턏"#, r#"턏"#, r#"턏"#),
        (r#"턐"#, r#"턐"#, r#"턐"#, r#"턐"#, r#"턐"#),
        (r#"턑"#, r#"턑"#, r#"턑"#, r#"턑"#, r#"턑"#),
        (r#"턒"#, r#"턒"#, r#"턒"#, r#"턒"#, r#"턒"#),
        (r#"턓"#, r#"턓"#, r#"턓"#, r#"턓"#, r#"턓"#),
        (r#"턔"#, r#"턔"#, r#"턔"#, r#"턔"#, r#"턔"#),
        (r#"턕"#, r#"턕"#, r#"턕"#, r#"턕"#, r#"턕"#),
        (r#"턖"#, r#"턖"#, r#"턖"#, r#"턖"#, r#"턖"#),
        (r#"턗"#, r#"턗"#, r#"턗"#, r#"턗"#, r#"턗"#),
        (r#"턘"#, r#"턘"#, r#"턘"#, r#"턘"#, r#"턘"#),
        (r#"턙"#, r#"턙"#, r#"턙"#, r#"턙"#, r#"턙"#),
        (r#"턚"#, r#"턚"#, r#"턚"#, r#"턚"#, r#"턚"#),
        (r#"턛"#, r#"턛"#, r#"턛"#, r#"턛"#, r#"턛"#),
        (r#"턜"#, r#"턜"#, r#"턜"#, r#"턜"#, r#"턜"#),
        (r#"턝"#, r#"턝"#, r#"턝"#, r#"턝"#, r#"턝"#),
        (r#"턞"#, r#"턞"#, r#"턞"#, r#"턞"#, r#"턞"#),
        (r#"턟"#, r#"턟"#, r#"턟"#, r#"턟"#, r#"턟"#),
        (r#"턠"#, r#"턠"#, r#"턠"#, r#"턠"#, r#"턠"#),
        (r#"턡"#, r#"턡"#, r#"턡"#, r#"턡"#, r#"턡"#),
        (r#"턢"#, r#"턢"#, r#"턢"#, r#"턢"#, r#"턢"#),
        (r#"턣"#, r#"턣"#, r#"턣"#, r#"턣"#, r#"턣"#),
        (r#"턤"#, r#"턤"#, r#"턤"#, r#"턤"#, r#"턤"#),
        (r#"턥"#, r#"턥"#, r#"턥"#, r#"턥"#, r#"턥"#),
        (r#"턦"#, r#"턦"#, r#"턦"#, r#"턦"#, r#"턦"#),
        (r#"턧"#, r#"턧"#, r#"턧"#, r#"턧"#, r#"턧"#),
        (r#"턨"#, r#"턨"#, r#"턨"#, r#"턨"#, r#"턨"#),
        (r#"턩"#, r#"턩"#, r#"턩"#, r#"턩"#, r#"턩"#),
        (r#"턪"#, r#"턪"#, r#"턪"#, r#"턪"#, r#"턪"#),
        (r#"턫"#, r#"턫"#, r#"턫"#, r#"턫"#, r#"턫"#),
        (r#"턬"#, r#"턬"#, r#"턬"#, r#"턬"#, r#"턬"#),
        (r#"턭"#, r#"턭"#, r#"턭"#, r#"턭"#, r#"턭"#),
        (r#"턮"#, r#"턮"#, r#"턮"#, r#"턮"#, r#"턮"#),
        (r#"턯"#, r#"턯"#, r#"턯"#, r#"턯"#, r#"턯"#),
        (r#"터"#, r#"터"#, r#"터"#, r#"터"#, r#"터"#),
        (r#"턱"#, r#"턱"#, r#"턱"#, r#"턱"#, r#"턱"#),
        (r#"턲"#, r#"턲"#, r#"턲"#, r#"턲"#, r#"턲"#),
        (r#"턳"#, r#"턳"#, r#"턳"#, r#"턳"#, r#"턳"#),
        (r#"턴"#, r#"턴"#, r#"턴"#, r#"턴"#, r#"턴"#),
        (r#"턵"#, r#"턵"#, r#"턵"#, r#"턵"#, r#"턵"#),
        (r#"턶"#, r#"턶"#, r#"턶"#, r#"턶"#, r#"턶"#),
        (r#"턷"#, r#"턷"#, r#"턷"#, r#"턷"#, r#"턷"#),
        (r#"털"#, r#"털"#, r#"털"#, r#"털"#, r#"털"#),
        (r#"턹"#, r#"턹"#, r#"턹"#, r#"턹"#, r#"턹"#),
        (r#"턺"#, r#"턺"#, r#"턺"#, r#"턺"#, r#"턺"#),
        (r#"턻"#, r#"턻"#, r#"턻"#, r#"턻"#, r#"턻"#),
        (r#"턼"#, r#"턼"#, r#"턼"#, r#"턼"#, r#"턼"#),
        (r#"턽"#, r#"턽"#, r#"턽"#, r#"턽"#, r#"턽"#),
        (r#"턾"#, r#"턾"#, r#"턾"#, r#"턾"#, r#"턾"#),
        (r#"턿"#, r#"턿"#, r#"턿"#, r#"턿"#, r#"턿"#),
        (r#"텀"#, r#"텀"#, r#"텀"#, r#"텀"#, r#"텀"#),
        (r#"텁"#, r#"텁"#, r#"텁"#, r#"텁"#, r#"텁"#),
        (r#"텂"#, r#"텂"#, r#"텂"#, r#"텂"#, r#"텂"#),
        (r#"텃"#, r#"텃"#, r#"텃"#, r#"텃"#, r#"텃"#),
        (r#"텄"#, r#"텄"#, r#"텄"#, r#"텄"#, r#"텄"#),
        (r#"텅"#, r#"텅"#, r#"텅"#, r#"텅"#, r#"텅"#),
        (r#"텆"#, r#"텆"#, r#"텆"#, r#"텆"#, r#"텆"#),
        (r#"텇"#, r#"텇"#, r#"텇"#, r#"텇"#, r#"텇"#),
        (r#"텈"#, r#"텈"#, r#"텈"#, r#"텈"#, r#"텈"#),
        (r#"텉"#, r#"텉"#, r#"텉"#, r#"텉"#, r#"텉"#),
        (r#"텊"#, r#"텊"#, r#"텊"#, r#"텊"#, r#"텊"#),
        (r#"텋"#, r#"텋"#, r#"텋"#, r#"텋"#, r#"텋"#),
        (r#"테"#, r#"테"#, r#"테"#, r#"테"#, r#"테"#),
        (r#"텍"#, r#"텍"#, r#"텍"#, r#"텍"#, r#"텍"#),
        (r#"텎"#, r#"텎"#, r#"텎"#, r#"텎"#, r#"텎"#),
        (r#"텏"#, r#"텏"#, r#"텏"#, r#"텏"#, r#"텏"#),
        (r#"텐"#, r#"텐"#, r#"텐"#, r#"텐"#, r#"텐"#),
        (r#"텑"#, r#"텑"#, r#"텑"#, r#"텑"#, r#"텑"#),
        (r#"텒"#, r#"텒"#, r#"텒"#, r#"텒"#, r#"텒"#),
        (r#"텓"#, r#"텓"#, r#"텓"#, r#"텓"#, r#"텓"#),
        (r#"텔"#, r#"텔"#, r#"텔"#, r#"텔"#, r#"텔"#),
        (r#"텕"#, r#"텕"#, r#"텕"#, r#"텕"#, r#"텕"#),
        (r#"텖"#, r#"텖"#, r#"텖"#, r#"텖"#, r#"텖"#),
        (r#"텗"#, r#"텗"#, r#"텗"#, r#"텗"#, r#"텗"#),
        (r#"텘"#, r#"텘"#, r#"텘"#, r#"텘"#, r#"텘"#),
        (r#"텙"#, r#"텙"#, r#"텙"#, r#"텙"#, r#"텙"#),
        (r#"텚"#, r#"텚"#, r#"텚"#, r#"텚"#, r#"텚"#),
        (r#"텛"#, r#"텛"#, r#"텛"#, r#"텛"#, r#"텛"#),
        (r#"템"#, r#"템"#, r#"템"#, r#"템"#, r#"템"#),
        (r#"텝"#, r#"텝"#, r#"텝"#, r#"텝"#, r#"텝"#),
        (r#"텞"#, r#"텞"#, r#"텞"#, r#"텞"#, r#"텞"#),
        (r#"텟"#, r#"텟"#, r#"텟"#, r#"텟"#, r#"텟"#),
        (r#"텠"#, r#"텠"#, r#"텠"#, r#"텠"#, r#"텠"#),
        (r#"텡"#, r#"텡"#, r#"텡"#, r#"텡"#, r#"텡"#),
        (r#"텢"#, r#"텢"#, r#"텢"#, r#"텢"#, r#"텢"#),
        (r#"텣"#, r#"텣"#, r#"텣"#, r#"텣"#, r#"텣"#),
        (r#"텤"#, r#"텤"#, r#"텤"#, r#"텤"#, r#"텤"#),
        (r#"텥"#, r#"텥"#, r#"텥"#, r#"텥"#, r#"텥"#),
        (r#"텦"#, r#"텦"#, r#"텦"#, r#"텦"#, r#"텦"#),
        (r#"텧"#, r#"텧"#, r#"텧"#, r#"텧"#, r#"텧"#),
        (r#"텨"#, r#"텨"#, r#"텨"#, r#"텨"#, r#"텨"#),
        (r#"텩"#, r#"텩"#, r#"텩"#, r#"텩"#, r#"텩"#),
        (r#"텪"#, r#"텪"#, r#"텪"#, r#"텪"#, r#"텪"#),
        (r#"텫"#, r#"텫"#, r#"텫"#, r#"텫"#, r#"텫"#),
        (r#"텬"#, r#"텬"#, r#"텬"#, r#"텬"#, r#"텬"#),
        (r#"텭"#, r#"텭"#, r#"텭"#, r#"텭"#, r#"텭"#),
        (r#"텮"#, r#"텮"#, r#"텮"#, r#"텮"#, r#"텮"#),
        (r#"텯"#, r#"텯"#, r#"텯"#, r#"텯"#, r#"텯"#),
        (r#"텰"#, r#"텰"#, r#"텰"#, r#"텰"#, r#"텰"#),
        (r#"텱"#, r#"텱"#, r#"텱"#, r#"텱"#, r#"텱"#),
        (r#"텲"#, r#"텲"#, r#"텲"#, r#"텲"#, r#"텲"#),
        (r#"텳"#, r#"텳"#, r#"텳"#, r#"텳"#, r#"텳"#),
        (r#"텴"#, r#"텴"#, r#"텴"#, r#"텴"#, r#"텴"#),
        (r#"텵"#, r#"텵"#, r#"텵"#, r#"텵"#, r#"텵"#),
        (r#"텶"#, r#"텶"#, r#"텶"#, r#"텶"#, r#"텶"#),
        (r#"텷"#, r#"텷"#, r#"텷"#, r#"텷"#, r#"텷"#),
        (r#"텸"#, r#"텸"#, r#"텸"#, r#"텸"#, r#"텸"#),
        (r#"텹"#, r#"텹"#, r#"텹"#, r#"텹"#, r#"텹"#),
        (r#"텺"#, r#"텺"#, r#"텺"#, r#"텺"#, r#"텺"#),
        (r#"텻"#, r#"텻"#, r#"텻"#, r#"텻"#, r#"텻"#),
        (r#"텼"#, r#"텼"#, r#"텼"#, r#"텼"#, r#"텼"#),
        (r#"텽"#, r#"텽"#, r#"텽"#, r#"텽"#, r#"텽"#),
        (r#"텾"#, r#"텾"#, r#"텾"#, r#"텾"#, r#"텾"#),
        (r#"텿"#, r#"텿"#, r#"텿"#, r#"텿"#, r#"텿"#),
        (r#"톀"#, r#"톀"#, r#"톀"#, r#"톀"#, r#"톀"#),
        (r#"톁"#, r#"톁"#, r#"톁"#, r#"톁"#, r#"톁"#),
        (r#"톂"#, r#"톂"#, r#"톂"#, r#"톂"#, r#"톂"#),
        (r#"톃"#, r#"톃"#, r#"톃"#, r#"톃"#, r#"톃"#),
        (r#"톄"#, r#"톄"#, r#"톄"#, r#"톄"#, r#"톄"#),
        (r#"톅"#, r#"톅"#, r#"톅"#, r#"톅"#, r#"톅"#),
        (r#"톆"#, r#"톆"#, r#"톆"#, r#"톆"#, r#"톆"#),
        (r#"톇"#, r#"톇"#, r#"톇"#, r#"톇"#, r#"톇"#),
        (r#"톈"#, r#"톈"#, r#"톈"#, r#"톈"#, r#"톈"#),
        (r#"톉"#, r#"톉"#, r#"톉"#, r#"톉"#, r#"톉"#),
        (r#"톊"#, r#"톊"#, r#"톊"#, r#"톊"#, r#"톊"#),
        (r#"톋"#, r#"톋"#, r#"톋"#, r#"톋"#, r#"톋"#),
        (r#"톌"#, r#"톌"#, r#"톌"#, r#"톌"#, r#"톌"#),
        (r#"톍"#, r#"톍"#, r#"톍"#, r#"톍"#, r#"톍"#),
        (r#"톎"#, r#"톎"#, r#"톎"#, r#"톎"#, r#"톎"#),
        (r#"톏"#, r#"톏"#, r#"톏"#, r#"톏"#, r#"톏"#),
        (r#"톐"#, r#"톐"#, r#"톐"#, r#"톐"#, r#"톐"#),
        (r#"톑"#, r#"톑"#, r#"톑"#, r#"톑"#, r#"톑"#),
        (r#"톒"#, r#"톒"#, r#"톒"#, r#"톒"#, r#"톒"#),
        (r#"톓"#, r#"톓"#, r#"톓"#, r#"톓"#, r#"톓"#),
        (r#"톔"#, r#"톔"#, r#"톔"#, r#"톔"#, r#"톔"#),
        (r#"톕"#, r#"톕"#, r#"톕"#, r#"톕"#, r#"톕"#),
        (r#"톖"#, r#"톖"#, r#"톖"#, r#"톖"#, r#"톖"#),
        (r#"톗"#, r#"톗"#, r#"톗"#, r#"톗"#, r#"톗"#),
        (r#"톘"#, r#"톘"#, r#"톘"#, r#"톘"#, r#"톘"#),
        (r#"톙"#, r#"톙"#, r#"톙"#, r#"톙"#, r#"톙"#),
        (r#"톚"#, r#"톚"#, r#"톚"#, r#"톚"#, r#"톚"#),
        (r#"톛"#, r#"톛"#, r#"톛"#, r#"톛"#, r#"톛"#),
        (r#"톜"#, r#"톜"#, r#"톜"#, r#"톜"#, r#"톜"#),
        (r#"톝"#, r#"톝"#, r#"톝"#, r#"톝"#, r#"톝"#),
        (r#"톞"#, r#"톞"#, r#"톞"#, r#"톞"#, r#"톞"#),
        (r#"톟"#, r#"톟"#, r#"톟"#, r#"톟"#, r#"톟"#),
        (r#"토"#, r#"토"#, r#"토"#, r#"토"#, r#"토"#),
        (r#"톡"#, r#"톡"#, r#"톡"#, r#"톡"#, r#"톡"#),
        (r#"톢"#, r#"톢"#, r#"톢"#, r#"톢"#, r#"톢"#),
        (r#"톣"#, r#"톣"#, r#"톣"#, r#"톣"#, r#"톣"#),
        (r#"톤"#, r#"톤"#, r#"톤"#, r#"톤"#, r#"톤"#),
        (r#"톥"#, r#"톥"#, r#"톥"#, r#"톥"#, r#"톥"#),
        (r#"톦"#, r#"톦"#, r#"톦"#, r#"톦"#, r#"톦"#),
        (r#"톧"#, r#"톧"#, r#"톧"#, r#"톧"#, r#"톧"#),
        (r#"톨"#, r#"톨"#, r#"톨"#, r#"톨"#, r#"톨"#),
        (r#"톩"#, r#"톩"#, r#"톩"#, r#"톩"#, r#"톩"#),
        (r#"톪"#, r#"톪"#, r#"톪"#, r#"톪"#, r#"톪"#),
        (r#"톫"#, r#"톫"#, r#"톫"#, r#"톫"#, r#"톫"#),
        (r#"톬"#, r#"톬"#, r#"톬"#, r#"톬"#, r#"톬"#),
        (r#"톭"#, r#"톭"#, r#"톭"#, r#"톭"#, r#"톭"#),
        (r#"톮"#, r#"톮"#, r#"톮"#, r#"톮"#, r#"톮"#),
        (r#"톯"#, r#"톯"#, r#"톯"#, r#"톯"#, r#"톯"#),
        (r#"톰"#, r#"톰"#, r#"톰"#, r#"톰"#, r#"톰"#),
        (r#"톱"#, r#"톱"#, r#"톱"#, r#"톱"#, r#"톱"#),
        (r#"톲"#, r#"톲"#, r#"톲"#, r#"톲"#, r#"톲"#),
        (r#"톳"#, r#"톳"#, r#"톳"#, r#"톳"#, r#"톳"#),
        (r#"톴"#, r#"톴"#, r#"톴"#, r#"톴"#, r#"톴"#),
        (r#"통"#, r#"통"#, r#"통"#, r#"통"#, r#"통"#),
        (r#"톶"#, r#"톶"#, r#"톶"#, r#"톶"#, r#"톶"#),
        (r#"톷"#, r#"톷"#, r#"톷"#, r#"톷"#, r#"톷"#),
        (r#"톸"#, r#"톸"#, r#"톸"#, r#"톸"#, r#"톸"#),
        (r#"톹"#, r#"톹"#, r#"톹"#, r#"톹"#, r#"톹"#),
        (r#"톺"#, r#"톺"#, r#"톺"#, r#"톺"#, r#"톺"#),
        (r#"톻"#, r#"톻"#, r#"톻"#, r#"톻"#, r#"톻"#),
        (r#"톼"#, r#"톼"#, r#"톼"#, r#"톼"#, r#"톼"#),
        (r#"톽"#, r#"톽"#, r#"톽"#, r#"톽"#, r#"톽"#),
        (r#"톾"#, r#"톾"#, r#"톾"#, r#"톾"#, r#"톾"#),
        (r#"톿"#, r#"톿"#, r#"톿"#, r#"톿"#, r#"톿"#),
        (r#"퇀"#, r#"퇀"#, r#"퇀"#, r#"퇀"#, r#"퇀"#),
        (r#"퇁"#, r#"퇁"#, r#"퇁"#, r#"퇁"#, r#"퇁"#),
        (r#"퇂"#, r#"퇂"#, r#"퇂"#, r#"퇂"#, r#"퇂"#),
        (r#"퇃"#, r#"퇃"#, r#"퇃"#, r#"퇃"#, r#"퇃"#),
        (r#"퇄"#, r#"퇄"#, r#"퇄"#, r#"퇄"#, r#"퇄"#),
        (r#"퇅"#, r#"퇅"#, r#"퇅"#, r#"퇅"#, r#"퇅"#),
        (r#"퇆"#, r#"퇆"#, r#"퇆"#, r#"퇆"#, r#"퇆"#),
        (r#"퇇"#, r#"퇇"#, r#"퇇"#, r#"퇇"#, r#"퇇"#),
        (r#"퇈"#, r#"퇈"#, r#"퇈"#, r#"퇈"#, r#"퇈"#),
        (r#"퇉"#, r#"퇉"#, r#"퇉"#, r#"퇉"#, r#"퇉"#),
        (r#"퇊"#, r#"퇊"#, r#"퇊"#, r#"퇊"#, r#"퇊"#),
        (r#"퇋"#, r#"퇋"#, r#"퇋"#, r#"퇋"#, r#"퇋"#),
        (r#"퇌"#, r#"퇌"#, r#"퇌"#, r#"퇌"#, r#"퇌"#),
        (r#"퇍"#, r#"퇍"#, r#"퇍"#, r#"퇍"#, r#"퇍"#),
        (r#"퇎"#, r#"퇎"#, r#"퇎"#, r#"퇎"#, r#"퇎"#),
        (r#"퇏"#, r#"퇏"#, r#"퇏"#, r#"퇏"#, r#"퇏"#),
        (r#"퇐"#, r#"퇐"#, r#"퇐"#, r#"퇐"#, r#"퇐"#),
        (r#"퇑"#, r#"퇑"#, r#"퇑"#, r#"퇑"#, r#"퇑"#),
        (r#"퇒"#, r#"퇒"#, r#"퇒"#, r#"퇒"#, r#"퇒"#),
        (r#"퇓"#, r#"퇓"#, r#"퇓"#, r#"퇓"#, r#"퇓"#),
        (r#"퇔"#, r#"퇔"#, r#"퇔"#, r#"퇔"#, r#"퇔"#),
        (r#"퇕"#, r#"퇕"#, r#"퇕"#, r#"퇕"#, r#"퇕"#),
        (r#"퇖"#, r#"퇖"#, r#"퇖"#, r#"퇖"#, r#"퇖"#),
        (r#"퇗"#, r#"퇗"#, r#"퇗"#, r#"퇗"#, r#"퇗"#),
        (r#"퇘"#, r#"퇘"#, r#"퇘"#, r#"퇘"#, r#"퇘"#),
        (r#"퇙"#, r#"퇙"#, r#"퇙"#, r#"퇙"#, r#"퇙"#),
        (r#"퇚"#, r#"퇚"#, r#"퇚"#, r#"퇚"#, r#"퇚"#),
        (r#"퇛"#, r#"퇛"#, r#"퇛"#, r#"퇛"#, r#"퇛"#),
        (r#"퇜"#, r#"퇜"#, r#"퇜"#, r#"퇜"#, r#"퇜"#),
        (r#"퇝"#, r#"퇝"#, r#"퇝"#, r#"퇝"#, r#"퇝"#),
        (r#"퇞"#, r#"퇞"#, r#"퇞"#, r#"퇞"#, r#"퇞"#),
        (r#"퇟"#, r#"퇟"#, r#"퇟"#, r#"퇟"#, r#"퇟"#),
        (r#"퇠"#, r#"퇠"#, r#"퇠"#, r#"퇠"#, r#"퇠"#),
        (r#"퇡"#, r#"퇡"#, r#"퇡"#, r#"퇡"#, r#"퇡"#),
        (r#"퇢"#, r#"퇢"#, r#"퇢"#, r#"퇢"#, r#"퇢"#),
        (r#"퇣"#, r#"퇣"#, r#"퇣"#, r#"퇣"#, r#"퇣"#),
        (r#"퇤"#, r#"퇤"#, r#"퇤"#, r#"퇤"#, r#"퇤"#),
        (r#"퇥"#, r#"퇥"#, r#"퇥"#, r#"퇥"#, r#"퇥"#),
        (r#"퇦"#, r#"퇦"#, r#"퇦"#, r#"퇦"#, r#"퇦"#),
        (r#"퇧"#, r#"퇧"#, r#"퇧"#, r#"퇧"#, r#"퇧"#),
        (r#"퇨"#, r#"퇨"#, r#"퇨"#, r#"퇨"#, r#"퇨"#),
        (r#"퇩"#, r#"퇩"#, r#"퇩"#, r#"퇩"#, r#"퇩"#),
        (r#"퇪"#, r#"퇪"#, r#"퇪"#, r#"퇪"#, r#"퇪"#),
        (r#"퇫"#, r#"퇫"#, r#"퇫"#, r#"퇫"#, r#"퇫"#),
        (r#"퇬"#, r#"퇬"#, r#"퇬"#, r#"퇬"#, r#"퇬"#),
        (r#"퇭"#, r#"퇭"#, r#"퇭"#, r#"퇭"#, r#"퇭"#),
        (r#"퇮"#, r#"퇮"#, r#"퇮"#, r#"퇮"#, r#"퇮"#),
        (r#"퇯"#, r#"퇯"#, r#"퇯"#, r#"퇯"#, r#"퇯"#),
        (r#"퇰"#, r#"퇰"#, r#"퇰"#, r#"퇰"#, r#"퇰"#),
        (r#"퇱"#, r#"퇱"#, r#"퇱"#, r#"퇱"#, r#"퇱"#),
        (r#"퇲"#, r#"퇲"#, r#"퇲"#, r#"퇲"#, r#"퇲"#),
        (r#"퇳"#, r#"퇳"#, r#"퇳"#, r#"퇳"#, r#"퇳"#),
        (r#"퇴"#, r#"퇴"#, r#"퇴"#, r#"퇴"#, r#"퇴"#),
        (r#"퇵"#, r#"퇵"#, r#"퇵"#, r#"퇵"#, r#"퇵"#),
        (r#"퇶"#, r#"퇶"#, r#"퇶"#, r#"퇶"#, r#"퇶"#),
        (r#"퇷"#, r#"퇷"#, r#"퇷"#, r#"퇷"#, r#"퇷"#),
        (r#"퇸"#, r#"퇸"#, r#"퇸"#, r#"퇸"#, r#"퇸"#),
        (r#"퇹"#, r#"퇹"#, r#"퇹"#, r#"퇹"#, r#"퇹"#),
        (r#"퇺"#, r#"퇺"#, r#"퇺"#, r#"퇺"#, r#"퇺"#),
        (r#"퇻"#, r#"퇻"#, r#"퇻"#, r#"퇻"#, r#"퇻"#),
        (r#"퇼"#, r#"퇼"#, r#"퇼"#, r#"퇼"#, r#"퇼"#),
        (r#"퇽"#, r#"퇽"#, r#"퇽"#, r#"퇽"#, r#"퇽"#),
        (r#"퇾"#, r#"퇾"#, r#"퇾"#, r#"퇾"#, r#"퇾"#),
        (r#"퇿"#, r#"퇿"#, r#"퇿"#, r#"퇿"#, r#"퇿"#),
        (r#"툀"#, r#"툀"#, r#"툀"#, r#"툀"#, r#"툀"#),
        (r#"툁"#, r#"툁"#, r#"툁"#, r#"툁"#, r#"툁"#),
        (r#"툂"#, r#"툂"#, r#"툂"#, r#"툂"#, r#"툂"#),
        (r#"툃"#, r#"툃"#, r#"툃"#, r#"툃"#, r#"툃"#),
        (r#"툄"#, r#"툄"#, r#"툄"#, r#"툄"#, r#"툄"#),
        (r#"툅"#, r#"툅"#, r#"툅"#, r#"툅"#, r#"툅"#),
        (r#"툆"#, r#"툆"#, r#"툆"#, r#"툆"#, r#"툆"#),
        (r#"툇"#, r#"툇"#, r#"툇"#, r#"툇"#, r#"툇"#),
        (r#"툈"#, r#"툈"#, r#"툈"#, r#"툈"#, r#"툈"#),
        (r#"툉"#, r#"툉"#, r#"툉"#, r#"툉"#, r#"툉"#),
        (r#"툊"#, r#"툊"#, r#"툊"#, r#"툊"#, r#"툊"#),
        (r#"툋"#, r#"툋"#, r#"툋"#, r#"툋"#, r#"툋"#),
        (r#"툌"#, r#"툌"#, r#"툌"#, r#"툌"#, r#"툌"#),
        (r#"툍"#, r#"툍"#, r#"툍"#, r#"툍"#, r#"툍"#),
        (r#"툎"#, r#"툎"#, r#"툎"#, r#"툎"#, r#"툎"#),
        (r#"툏"#, r#"툏"#, r#"툏"#, r#"툏"#, r#"툏"#),
        (r#"툐"#, r#"툐"#, r#"툐"#, r#"툐"#, r#"툐"#),
        (r#"툑"#, r#"툑"#, r#"툑"#, r#"툑"#, r#"툑"#),
        (r#"툒"#, r#"툒"#, r#"툒"#, r#"툒"#, r#"툒"#),
        (r#"툓"#, r#"툓"#, r#"툓"#, r#"툓"#, r#"툓"#),
        (r#"툔"#, r#"툔"#, r#"툔"#, r#"툔"#, r#"툔"#),
        (r#"툕"#, r#"툕"#, r#"툕"#, r#"툕"#, r#"툕"#),
        (r#"툖"#, r#"툖"#, r#"툖"#, r#"툖"#, r#"툖"#),
        (r#"툗"#, r#"툗"#, r#"툗"#, r#"툗"#, r#"툗"#),
        (r#"툘"#, r#"툘"#, r#"툘"#, r#"툘"#, r#"툘"#),
        (r#"툙"#, r#"툙"#, r#"툙"#, r#"툙"#, r#"툙"#),
        (r#"툚"#, r#"툚"#, r#"툚"#, r#"툚"#, r#"툚"#),
        (r#"툛"#, r#"툛"#, r#"툛"#, r#"툛"#, r#"툛"#),
        (r#"툜"#, r#"툜"#, r#"툜"#, r#"툜"#, r#"툜"#),
        (r#"툝"#, r#"툝"#, r#"툝"#, r#"툝"#, r#"툝"#),
        (r#"툞"#, r#"툞"#, r#"툞"#, r#"툞"#, r#"툞"#),
        (r#"툟"#, r#"툟"#, r#"툟"#, r#"툟"#, r#"툟"#),
        (r#"툠"#, r#"툠"#, r#"툠"#, r#"툠"#, r#"툠"#),
        (r#"툡"#, r#"툡"#, r#"툡"#, r#"툡"#, r#"툡"#),
        (r#"툢"#, r#"툢"#, r#"툢"#, r#"툢"#, r#"툢"#),
        (r#"툣"#, r#"툣"#, r#"툣"#, r#"툣"#, r#"툣"#),
        (r#"툤"#, r#"툤"#, r#"툤"#, r#"툤"#, r#"툤"#),
        (r#"툥"#, r#"툥"#, r#"툥"#, r#"툥"#, r#"툥"#),
        (r#"툦"#, r#"툦"#, r#"툦"#, r#"툦"#, r#"툦"#),
        (r#"툧"#, r#"툧"#, r#"툧"#, r#"툧"#, r#"툧"#),
        (r#"툨"#, r#"툨"#, r#"툨"#, r#"툨"#, r#"툨"#),
        (r#"툩"#, r#"툩"#, r#"툩"#, r#"툩"#, r#"툩"#),
        (r#"툪"#, r#"툪"#, r#"툪"#, r#"툪"#, r#"툪"#),
        (r#"툫"#, r#"툫"#, r#"툫"#, r#"툫"#, r#"툫"#),
        (r#"투"#, r#"투"#, r#"투"#, r#"투"#, r#"투"#),
        (r#"툭"#, r#"툭"#, r#"툭"#, r#"툭"#, r#"툭"#),
        (r#"툮"#, r#"툮"#, r#"툮"#, r#"툮"#, r#"툮"#),
        (r#"툯"#, r#"툯"#, r#"툯"#, r#"툯"#, r#"툯"#),
        (r#"툰"#, r#"툰"#, r#"툰"#, r#"툰"#, r#"툰"#),
        (r#"툱"#, r#"툱"#, r#"툱"#, r#"툱"#, r#"툱"#),
        (r#"툲"#, r#"툲"#, r#"툲"#, r#"툲"#, r#"툲"#),
        (r#"툳"#, r#"툳"#, r#"툳"#, r#"툳"#, r#"툳"#),
        (r#"툴"#, r#"툴"#, r#"툴"#, r#"툴"#, r#"툴"#),
        (r#"툵"#, r#"툵"#, r#"툵"#, r#"툵"#, r#"툵"#),
        (r#"툶"#, r#"툶"#, r#"툶"#, r#"툶"#, r#"툶"#),
        (r#"툷"#, r#"툷"#, r#"툷"#, r#"툷"#, r#"툷"#),
        (r#"툸"#, r#"툸"#, r#"툸"#, r#"툸"#, r#"툸"#),
        (r#"툹"#, r#"툹"#, r#"툹"#, r#"툹"#, r#"툹"#),
        (r#"툺"#, r#"툺"#, r#"툺"#, r#"툺"#, r#"툺"#),
        (r#"툻"#, r#"툻"#, r#"툻"#, r#"툻"#, r#"툻"#),
        (r#"툼"#, r#"툼"#, r#"툼"#, r#"툼"#, r#"툼"#),
        (r#"툽"#, r#"툽"#, r#"툽"#, r#"툽"#, r#"툽"#),
        (r#"툾"#, r#"툾"#, r#"툾"#, r#"툾"#, r#"툾"#),
        (r#"툿"#, r#"툿"#, r#"툿"#, r#"툿"#, r#"툿"#),
        (r#"퉀"#, r#"퉀"#, r#"퉀"#, r#"퉀"#, r#"퉀"#),
        (r#"퉁"#, r#"퉁"#, r#"퉁"#, r#"퉁"#, r#"퉁"#),
        (r#"퉂"#, r#"퉂"#, r#"퉂"#, r#"퉂"#, r#"퉂"#),
        (r#"퉃"#, r#"퉃"#, r#"퉃"#, r#"퉃"#, r#"퉃"#),
        (r#"퉄"#, r#"퉄"#, r#"퉄"#, r#"퉄"#, r#"퉄"#),
        (r#"퉅"#, r#"퉅"#, r#"퉅"#, r#"퉅"#, r#"퉅"#),
        (r#"퉆"#, r#"퉆"#, r#"퉆"#, r#"퉆"#, r#"퉆"#),
        (r#"퉇"#, r#"퉇"#, r#"퉇"#, r#"퉇"#, r#"퉇"#),
        (r#"퉈"#, r#"퉈"#, r#"퉈"#, r#"퉈"#, r#"퉈"#),
        (r#"퉉"#, r#"퉉"#, r#"퉉"#, r#"퉉"#, r#"퉉"#),
        (r#"퉊"#, r#"퉊"#, r#"퉊"#, r#"퉊"#, r#"퉊"#),
        (r#"퉋"#, r#"퉋"#, r#"퉋"#, r#"퉋"#, r#"퉋"#),
        (r#"퉌"#, r#"퉌"#, r#"퉌"#, r#"퉌"#, r#"퉌"#),
        (r#"퉍"#, r#"퉍"#, r#"퉍"#, r#"퉍"#, r#"퉍"#),
        (r#"퉎"#, r#"퉎"#, r#"퉎"#, r#"퉎"#, r#"퉎"#),
        (r#"퉏"#, r#"퉏"#, r#"퉏"#, r#"퉏"#, r#"퉏"#),
        (r#"퉐"#, r#"퉐"#, r#"퉐"#, r#"퉐"#, r#"퉐"#),
        (r#"퉑"#, r#"퉑"#, r#"퉑"#, r#"퉑"#, r#"퉑"#),
        (r#"퉒"#, r#"퉒"#, r#"퉒"#, r#"퉒"#, r#"퉒"#),
        (r#"퉓"#, r#"퉓"#, r#"퉓"#, r#"퉓"#, r#"퉓"#),
        (r#"퉔"#, r#"퉔"#, r#"퉔"#, r#"퉔"#, r#"퉔"#),
        (r#"퉕"#, r#"퉕"#, r#"퉕"#, r#"퉕"#, r#"퉕"#),
        (r#"퉖"#, r#"퉖"#, r#"퉖"#, r#"퉖"#, r#"퉖"#),
        (r#"퉗"#, r#"퉗"#, r#"퉗"#, r#"퉗"#, r#"퉗"#),
        (r#"퉘"#, r#"퉘"#, r#"퉘"#, r#"퉘"#, r#"퉘"#),
        (r#"퉙"#, r#"퉙"#, r#"퉙"#, r#"퉙"#, r#"퉙"#),
        (r#"퉚"#, r#"퉚"#, r#"퉚"#, r#"퉚"#, r#"퉚"#),
        (r#"퉛"#, r#"퉛"#, r#"퉛"#, r#"퉛"#, r#"퉛"#),
        (r#"퉜"#, r#"퉜"#, r#"퉜"#, r#"퉜"#, r#"퉜"#),
        (r#"퉝"#, r#"퉝"#, r#"퉝"#, r#"퉝"#, r#"퉝"#),
        (r#"퉞"#, r#"퉞"#, r#"퉞"#, r#"퉞"#, r#"퉞"#),
        (r#"퉟"#, r#"퉟"#, r#"퉟"#, r#"퉟"#, r#"퉟"#),
        (r#"퉠"#, r#"퉠"#, r#"퉠"#, r#"퉠"#, r#"퉠"#),
        (r#"퉡"#, r#"퉡"#, r#"퉡"#, r#"퉡"#, r#"퉡"#),
        (r#"퉢"#, r#"퉢"#, r#"퉢"#, r#"퉢"#, r#"퉢"#),
        (r#"퉣"#, r#"퉣"#, r#"퉣"#, r#"퉣"#, r#"퉣"#),
        (r#"퉤"#, r#"퉤"#, r#"퉤"#, r#"퉤"#, r#"퉤"#),
        (r#"퉥"#, r#"퉥"#, r#"퉥"#, r#"퉥"#, r#"퉥"#),
        (r#"퉦"#, r#"퉦"#, r#"퉦"#, r#"퉦"#, r#"퉦"#),
        (r#"퉧"#, r#"퉧"#, r#"퉧"#, r#"퉧"#, r#"퉧"#),
        (r#"퉨"#, r#"퉨"#, r#"퉨"#, r#"퉨"#, r#"퉨"#),
        (r#"퉩"#, r#"퉩"#, r#"퉩"#, r#"퉩"#, r#"퉩"#),
        (r#"퉪"#, r#"퉪"#, r#"퉪"#, r#"퉪"#, r#"퉪"#),
        (r#"퉫"#, r#"퉫"#, r#"퉫"#, r#"퉫"#, r#"퉫"#),
        (r#"퉬"#, r#"퉬"#, r#"퉬"#, r#"퉬"#, r#"퉬"#),
        (r#"퉭"#, r#"퉭"#, r#"퉭"#, r#"퉭"#, r#"퉭"#),
        (r#"퉮"#, r#"퉮"#, r#"퉮"#, r#"퉮"#, r#"퉮"#),
        (r#"퉯"#, r#"퉯"#, r#"퉯"#, r#"퉯"#, r#"퉯"#),
        (r#"퉰"#, r#"퉰"#, r#"퉰"#, r#"퉰"#, r#"퉰"#),
        (r#"퉱"#, r#"퉱"#, r#"퉱"#, r#"퉱"#, r#"퉱"#),
        (r#"퉲"#, r#"퉲"#, r#"퉲"#, r#"퉲"#, r#"퉲"#),
        (r#"퉳"#, r#"퉳"#, r#"퉳"#, r#"퉳"#, r#"퉳"#),
        (r#"퉴"#, r#"퉴"#, r#"퉴"#, r#"퉴"#, r#"퉴"#),
        (r#"퉵"#, r#"퉵"#, r#"퉵"#, r#"퉵"#, r#"퉵"#),
        (r#"퉶"#, r#"퉶"#, r#"퉶"#, r#"퉶"#, r#"퉶"#),
        (r#"퉷"#, r#"퉷"#, r#"퉷"#, r#"퉷"#, r#"퉷"#),
        (r#"퉸"#, r#"퉸"#, r#"퉸"#, r#"퉸"#, r#"퉸"#),
        (r#"퉹"#, r#"퉹"#, r#"퉹"#, r#"퉹"#, r#"퉹"#),
        (r#"퉺"#, r#"퉺"#, r#"퉺"#, r#"퉺"#, r#"퉺"#),
        (r#"퉻"#, r#"퉻"#, r#"퉻"#, r#"퉻"#, r#"퉻"#),
        (r#"퉼"#, r#"퉼"#, r#"퉼"#, r#"퉼"#, r#"퉼"#),
        (r#"퉽"#, r#"퉽"#, r#"퉽"#, r#"퉽"#, r#"퉽"#),
        (r#"퉾"#, r#"퉾"#, r#"퉾"#, r#"퉾"#, r#"퉾"#),
        (r#"퉿"#, r#"퉿"#, r#"퉿"#, r#"퉿"#, r#"퉿"#),
        (r#"튀"#, r#"튀"#, r#"튀"#, r#"튀"#, r#"튀"#),
        (r#"튁"#, r#"튁"#, r#"튁"#, r#"튁"#, r#"튁"#),
        (r#"튂"#, r#"튂"#, r#"튂"#, r#"튂"#, r#"튂"#),
        (r#"튃"#, r#"튃"#, r#"튃"#, r#"튃"#, r#"튃"#),
        (r#"튄"#, r#"튄"#, r#"튄"#, r#"튄"#, r#"튄"#),
        (r#"튅"#, r#"튅"#, r#"튅"#, r#"튅"#, r#"튅"#),
        (r#"튆"#, r#"튆"#, r#"튆"#, r#"튆"#, r#"튆"#),
        (r#"튇"#, r#"튇"#, r#"튇"#, r#"튇"#, r#"튇"#),
        (r#"튈"#, r#"튈"#, r#"튈"#, r#"튈"#, r#"튈"#),
        (r#"튉"#, r#"튉"#, r#"튉"#, r#"튉"#, r#"튉"#),
        (r#"튊"#, r#"튊"#, r#"튊"#, r#"튊"#, r#"튊"#),
        (r#"튋"#, r#"튋"#, r#"튋"#, r#"튋"#, r#"튋"#),
        (r#"튌"#, r#"튌"#, r#"튌"#, r#"튌"#, r#"튌"#),
        (r#"튍"#, r#"튍"#, r#"튍"#, r#"튍"#, r#"튍"#),
        (r#"튎"#, r#"튎"#, r#"튎"#, r#"튎"#, r#"튎"#),
        (r#"튏"#, r#"튏"#, r#"튏"#, r#"튏"#, r#"튏"#),
        (r#"튐"#, r#"튐"#, r#"튐"#, r#"튐"#, r#"튐"#),
        (r#"튑"#, r#"튑"#, r#"튑"#, r#"튑"#, r#"튑"#),
        (r#"튒"#, r#"튒"#, r#"튒"#, r#"튒"#, r#"튒"#),
        (r#"튓"#, r#"튓"#, r#"튓"#, r#"튓"#, r#"튓"#),
        (r#"튔"#, r#"튔"#, r#"튔"#, r#"튔"#, r#"튔"#),
        (r#"튕"#, r#"튕"#, r#"튕"#, r#"튕"#, r#"튕"#),
        (r#"튖"#, r#"튖"#, r#"튖"#, r#"튖"#, r#"튖"#),
        (r#"튗"#, r#"튗"#, r#"튗"#, r#"튗"#, r#"튗"#),
        (r#"튘"#, r#"튘"#, r#"튘"#, r#"튘"#, r#"튘"#),
        (r#"튙"#, r#"튙"#, r#"튙"#, r#"튙"#, r#"튙"#),
        (r#"튚"#, r#"튚"#, r#"튚"#, r#"튚"#, r#"튚"#),
        (r#"튛"#, r#"튛"#, r#"튛"#, r#"튛"#, r#"튛"#),
        (r#"튜"#, r#"튜"#, r#"튜"#, r#"튜"#, r#"튜"#),
        (r#"튝"#, r#"튝"#, r#"튝"#, r#"튝"#, r#"튝"#),
        (r#"튞"#, r#"튞"#, r#"튞"#, r#"튞"#, r#"튞"#),
        (r#"튟"#, r#"튟"#, r#"튟"#, r#"튟"#, r#"튟"#),
        (r#"튠"#, r#"튠"#, r#"튠"#, r#"튠"#, r#"튠"#),
        (r#"튡"#, r#"튡"#, r#"튡"#, r#"튡"#, r#"튡"#),
        (r#"튢"#, r#"튢"#, r#"튢"#, r#"튢"#, r#"튢"#),
        (r#"튣"#, r#"튣"#, r#"튣"#, r#"튣"#, r#"튣"#),
        (r#"튤"#, r#"튤"#, r#"튤"#, r#"튤"#, r#"튤"#),
        (r#"튥"#, r#"튥"#, r#"튥"#, r#"튥"#, r#"튥"#),
        (r#"튦"#, r#"튦"#, r#"튦"#, r#"튦"#, r#"튦"#),
        (r#"튧"#, r#"튧"#, r#"튧"#, r#"튧"#, r#"튧"#),
        (r#"튨"#, r#"튨"#, r#"튨"#, r#"튨"#, r#"튨"#),
        (r#"튩"#, r#"튩"#, r#"튩"#, r#"튩"#, r#"튩"#),
        (r#"튪"#, r#"튪"#, r#"튪"#, r#"튪"#, r#"튪"#),
        (r#"튫"#, r#"튫"#, r#"튫"#, r#"튫"#, r#"튫"#),
        (r#"튬"#, r#"튬"#, r#"튬"#, r#"튬"#, r#"튬"#),
        (r#"튭"#, r#"튭"#, r#"튭"#, r#"튭"#, r#"튭"#),
        (r#"튮"#, r#"튮"#, r#"튮"#, r#"튮"#, r#"튮"#),
        (r#"튯"#, r#"튯"#, r#"튯"#, r#"튯"#, r#"튯"#),
        (r#"튰"#, r#"튰"#, r#"튰"#, r#"튰"#, r#"튰"#),
        (r#"튱"#, r#"튱"#, r#"튱"#, r#"튱"#, r#"튱"#),
        (r#"튲"#, r#"튲"#, r#"튲"#, r#"튲"#, r#"튲"#),
        (r#"튳"#, r#"튳"#, r#"튳"#, r#"튳"#, r#"튳"#),
        (r#"튴"#, r#"튴"#, r#"튴"#, r#"튴"#, r#"튴"#),
        (r#"튵"#, r#"튵"#, r#"튵"#, r#"튵"#, r#"튵"#),
        (r#"튶"#, r#"튶"#, r#"튶"#, r#"튶"#, r#"튶"#),
        (r#"튷"#, r#"튷"#, r#"튷"#, r#"튷"#, r#"튷"#),
        (r#"트"#, r#"트"#, r#"트"#, r#"트"#, r#"트"#),
        (r#"특"#, r#"특"#, r#"특"#, r#"특"#, r#"특"#),
        (r#"튺"#, r#"튺"#, r#"튺"#, r#"튺"#, r#"튺"#),
        (r#"튻"#, r#"튻"#, r#"튻"#, r#"튻"#, r#"튻"#),
        (r#"튼"#, r#"튼"#, r#"튼"#, r#"튼"#, r#"튼"#),
        (r#"튽"#, r#"튽"#, r#"튽"#, r#"튽"#, r#"튽"#),
        (r#"튾"#, r#"튾"#, r#"튾"#, r#"튾"#, r#"튾"#),
        (r#"튿"#, r#"튿"#, r#"튿"#, r#"튿"#, r#"튿"#),
        (r#"틀"#, r#"틀"#, r#"틀"#, r#"틀"#, r#"틀"#),
        (r#"틁"#, r#"틁"#, r#"틁"#, r#"틁"#, r#"틁"#),
        (r#"틂"#, r#"틂"#, r#"틂"#, r#"틂"#, r#"틂"#),
        (r#"틃"#, r#"틃"#, r#"틃"#, r#"틃"#, r#"틃"#),
        (r#"틄"#, r#"틄"#, r#"틄"#, r#"틄"#, r#"틄"#),
        (r#"틅"#, r#"틅"#, r#"틅"#, r#"틅"#, r#"틅"#),
        (r#"틆"#, r#"틆"#, r#"틆"#, r#"틆"#, r#"틆"#),
        (r#"틇"#, r#"틇"#, r#"틇"#, r#"틇"#, r#"틇"#),
        (r#"틈"#, r#"틈"#, r#"틈"#, r#"틈"#, r#"틈"#),
        (r#"틉"#, r#"틉"#, r#"틉"#, r#"틉"#, r#"틉"#),
        (r#"틊"#, r#"틊"#, r#"틊"#, r#"틊"#, r#"틊"#),
        (r#"틋"#, r#"틋"#, r#"틋"#, r#"틋"#, r#"틋"#),
        (r#"틌"#, r#"틌"#, r#"틌"#, r#"틌"#, r#"틌"#),
        (r#"틍"#, r#"틍"#, r#"틍"#, r#"틍"#, r#"틍"#),
        (r#"틎"#, r#"틎"#, r#"틎"#, r#"틎"#, r#"틎"#),
        (r#"틏"#, r#"틏"#, r#"틏"#, r#"틏"#, r#"틏"#),
        (r#"틐"#, r#"틐"#, r#"틐"#, r#"틐"#, r#"틐"#),
        (r#"틑"#, r#"틑"#, r#"틑"#, r#"틑"#, r#"틑"#),
        (r#"틒"#, r#"틒"#, r#"틒"#, r#"틒"#, r#"틒"#),
        (r#"틓"#, r#"틓"#, r#"틓"#, r#"틓"#, r#"틓"#),
        (r#"틔"#, r#"틔"#, r#"틔"#, r#"틔"#, r#"틔"#),
        (r#"틕"#, r#"틕"#, r#"틕"#, r#"틕"#, r#"틕"#),
        (r#"틖"#, r#"틖"#, r#"틖"#, r#"틖"#, r#"틖"#),
        (r#"틗"#, r#"틗"#, r#"틗"#, r#"틗"#, r#"틗"#),
        (r#"틘"#, r#"틘"#, r#"틘"#, r#"틘"#, r#"틘"#),
        (r#"틙"#, r#"틙"#, r#"틙"#, r#"틙"#, r#"틙"#),
        (r#"틚"#, r#"틚"#, r#"틚"#, r#"틚"#, r#"틚"#),
        (r#"틛"#, r#"틛"#, r#"틛"#, r#"틛"#, r#"틛"#),
        (r#"틜"#, r#"틜"#, r#"틜"#, r#"틜"#, r#"틜"#),
        (r#"틝"#, r#"틝"#, r#"틝"#, r#"틝"#, r#"틝"#),
        (r#"틞"#, r#"틞"#, r#"틞"#, r#"틞"#, r#"틞"#),
        (r#"틟"#, r#"틟"#, r#"틟"#, r#"틟"#, r#"틟"#),
        (r#"틠"#, r#"틠"#, r#"틠"#, r#"틠"#, r#"틠"#),
        (r#"틡"#, r#"틡"#, r#"틡"#, r#"틡"#, r#"틡"#),
        (r#"틢"#, r#"틢"#, r#"틢"#, r#"틢"#, r#"틢"#),
        (r#"틣"#, r#"틣"#, r#"틣"#, r#"틣"#, r#"틣"#),
        (r#"틤"#, r#"틤"#, r#"틤"#, r#"틤"#, r#"틤"#),
        (r#"틥"#, r#"틥"#, r#"틥"#, r#"틥"#, r#"틥"#),
        (r#"틦"#, r#"틦"#, r#"틦"#, r#"틦"#, r#"틦"#),
        (r#"틧"#, r#"틧"#, r#"틧"#, r#"틧"#, r#"틧"#),
        (r#"틨"#, r#"틨"#, r#"틨"#, r#"틨"#, r#"틨"#),
        (r#"틩"#, r#"틩"#, r#"틩"#, r#"틩"#, r#"틩"#),
        (r#"틪"#, r#"틪"#, r#"틪"#, r#"틪"#, r#"틪"#),
        (r#"틫"#, r#"틫"#, r#"틫"#, r#"틫"#, r#"틫"#),
        (r#"틬"#, r#"틬"#, r#"틬"#, r#"틬"#, r#"틬"#),
        (r#"틭"#, r#"틭"#, r#"틭"#, r#"틭"#, r#"틭"#),
        (r#"틮"#, r#"틮"#, r#"틮"#, r#"틮"#, r#"틮"#),
        (r#"틯"#, r#"틯"#, r#"틯"#, r#"틯"#, r#"틯"#),
        (r#"티"#, r#"티"#, r#"티"#, r#"티"#, r#"티"#),
        (r#"틱"#, r#"틱"#, r#"틱"#, r#"틱"#, r#"틱"#),
        (r#"틲"#, r#"틲"#, r#"틲"#, r#"틲"#, r#"틲"#),
        (r#"틳"#, r#"틳"#, r#"틳"#, r#"틳"#, r#"틳"#),
        (r#"틴"#, r#"틴"#, r#"틴"#, r#"틴"#, r#"틴"#),
        (r#"틵"#, r#"틵"#, r#"틵"#, r#"틵"#, r#"틵"#),
        (r#"틶"#, r#"틶"#, r#"틶"#, r#"틶"#, r#"틶"#),
        (r#"틷"#, r#"틷"#, r#"틷"#, r#"틷"#, r#"틷"#),
        (r#"틸"#, r#"틸"#, r#"틸"#, r#"틸"#, r#"틸"#),
        (r#"틹"#, r#"틹"#, r#"틹"#, r#"틹"#, r#"틹"#),
        (r#"틺"#, r#"틺"#, r#"틺"#, r#"틺"#, r#"틺"#),
        (r#"틻"#, r#"틻"#, r#"틻"#, r#"틻"#, r#"틻"#),
        (r#"틼"#, r#"틼"#, r#"틼"#, r#"틼"#, r#"틼"#),
        (r#"틽"#, r#"틽"#, r#"틽"#, r#"틽"#, r#"틽"#),
        (r#"틾"#, r#"틾"#, r#"틾"#, r#"틾"#, r#"틾"#),
        (r#"틿"#, r#"틿"#, r#"틿"#, r#"틿"#, r#"틿"#),
        (r#"팀"#, r#"팀"#, r#"팀"#, r#"팀"#, r#"팀"#),
        (r#"팁"#, r#"팁"#, r#"팁"#, r#"팁"#, r#"팁"#),
        (r#"팂"#, r#"팂"#, r#"팂"#, r#"팂"#, r#"팂"#),
        (r#"팃"#, r#"팃"#, r#"팃"#, r#"팃"#, r#"팃"#),
        (r#"팄"#, r#"팄"#, r#"팄"#, r#"팄"#, r#"팄"#),
        (r#"팅"#, r#"팅"#, r#"팅"#, r#"팅"#, r#"팅"#),
        (r#"팆"#, r#"팆"#, r#"팆"#, r#"팆"#, r#"팆"#),
        (r#"팇"#, r#"팇"#, r#"팇"#, r#"팇"#, r#"팇"#),
        (r#"팈"#, r#"팈"#, r#"팈"#, r#"팈"#, r#"팈"#),
        (r#"팉"#, r#"팉"#, r#"팉"#, r#"팉"#, r#"팉"#),
        (r#"팊"#, r#"팊"#, r#"팊"#, r#"팊"#, r#"팊"#),
        (r#"팋"#, r#"팋"#, r#"팋"#, r#"팋"#, r#"팋"#),
        (r#"파"#, r#"파"#, r#"파"#, r#"파"#, r#"파"#),
        (r#"팍"#, r#"팍"#, r#"팍"#, r#"팍"#, r#"팍"#),
        (r#"팎"#, r#"팎"#, r#"팎"#, r#"팎"#, r#"팎"#),
        (r#"팏"#, r#"팏"#, r#"팏"#, r#"팏"#, r#"팏"#),
        (r#"판"#, r#"판"#, r#"판"#, r#"판"#, r#"판"#),
        (r#"팑"#, r#"팑"#, r#"팑"#, r#"팑"#, r#"팑"#),
        (r#"팒"#, r#"팒"#, r#"팒"#, r#"팒"#, r#"팒"#),
        (r#"팓"#, r#"팓"#, r#"팓"#, r#"팓"#, r#"팓"#),
        (r#"팔"#, r#"팔"#, r#"팔"#, r#"팔"#, r#"팔"#),
        (r#"팕"#, r#"팕"#, r#"팕"#, r#"팕"#, r#"팕"#),
        (r#"팖"#, r#"팖"#, r#"팖"#, r#"팖"#, r#"팖"#),
        (r#"팗"#, r#"팗"#, r#"팗"#, r#"팗"#, r#"팗"#),
        (r#"팘"#, r#"팘"#, r#"팘"#, r#"팘"#, r#"팘"#),
        (r#"팙"#, r#"팙"#, r#"팙"#, r#"팙"#, r#"팙"#),
        (r#"팚"#, r#"팚"#, r#"팚"#, r#"팚"#, r#"팚"#),
        (r#"팛"#, r#"팛"#, r#"팛"#, r#"팛"#, r#"팛"#),
        (r#"팜"#, r#"팜"#, r#"팜"#, r#"팜"#, r#"팜"#),
        (r#"팝"#, r#"팝"#, r#"팝"#, r#"팝"#, r#"팝"#),
        (r#"팞"#, r#"팞"#, r#"팞"#, r#"팞"#, r#"팞"#),
        (r#"팟"#, r#"팟"#, r#"팟"#, r#"팟"#, r#"팟"#),
        (r#"팠"#, r#"팠"#, r#"팠"#, r#"팠"#, r#"팠"#),
        (r#"팡"#, r#"팡"#, r#"팡"#, r#"팡"#, r#"팡"#),
        (r#"팢"#, r#"팢"#, r#"팢"#, r#"팢"#, r#"팢"#),
        (r#"팣"#, r#"팣"#, r#"팣"#, r#"팣"#, r#"팣"#),
        (r#"팤"#, r#"팤"#, r#"팤"#, r#"팤"#, r#"팤"#),
        (r#"팥"#, r#"팥"#, r#"팥"#, r#"팥"#, r#"팥"#),
        (r#"팦"#, r#"팦"#, r#"팦"#, r#"팦"#, r#"팦"#),
        (r#"팧"#, r#"팧"#, r#"팧"#, r#"팧"#, r#"팧"#),
        (r#"패"#, r#"패"#, r#"패"#, r#"패"#, r#"패"#),
        (r#"팩"#, r#"팩"#, r#"팩"#, r#"팩"#, r#"팩"#),
        (r#"팪"#, r#"팪"#, r#"팪"#, r#"팪"#, r#"팪"#),
        (r#"팫"#, r#"팫"#, r#"팫"#, r#"팫"#, r#"팫"#),
        (r#"팬"#, r#"팬"#, r#"팬"#, r#"팬"#, r#"팬"#),
        (r#"팭"#, r#"팭"#, r#"팭"#, r#"팭"#, r#"팭"#),
        (r#"팮"#, r#"팮"#, r#"팮"#, r#"팮"#, r#"팮"#),
        (r#"팯"#, r#"팯"#, r#"팯"#, r#"팯"#, r#"팯"#),
        (r#"팰"#, r#"팰"#, r#"팰"#, r#"팰"#, r#"팰"#),
        (r#"팱"#, r#"팱"#, r#"팱"#, r#"팱"#, r#"팱"#),
        (r#"팲"#, r#"팲"#, r#"팲"#, r#"팲"#, r#"팲"#),
        (r#"팳"#, r#"팳"#, r#"팳"#, r#"팳"#, r#"팳"#),
        (r#"팴"#, r#"팴"#, r#"팴"#, r#"팴"#, r#"팴"#),
        (r#"팵"#, r#"팵"#, r#"팵"#, r#"팵"#, r#"팵"#),
        (r#"팶"#, r#"팶"#, r#"팶"#, r#"팶"#, r#"팶"#),
        (r#"팷"#, r#"팷"#, r#"팷"#, r#"팷"#, r#"팷"#),
        (r#"팸"#, r#"팸"#, r#"팸"#, r#"팸"#, r#"팸"#),
        (r#"팹"#, r#"팹"#, r#"팹"#, r#"팹"#, r#"팹"#),
        (r#"팺"#, r#"팺"#, r#"팺"#, r#"팺"#, r#"팺"#),
        (r#"팻"#, r#"팻"#, r#"팻"#, r#"팻"#, r#"팻"#),
        (r#"팼"#, r#"팼"#, r#"팼"#, r#"팼"#, r#"팼"#),
        (r#"팽"#, r#"팽"#, r#"팽"#, r#"팽"#, r#"팽"#),
        (r#"팾"#, r#"팾"#, r#"팾"#, r#"팾"#, r#"팾"#),
        (r#"팿"#, r#"팿"#, r#"팿"#, r#"팿"#, r#"팿"#),
        (r#"퍀"#, r#"퍀"#, r#"퍀"#, r#"퍀"#, r#"퍀"#),
        (r#"퍁"#, r#"퍁"#, r#"퍁"#, r#"퍁"#, r#"퍁"#),
        (r#"퍂"#, r#"퍂"#, r#"퍂"#, r#"퍂"#, r#"퍂"#),
        (r#"퍃"#, r#"퍃"#, r#"퍃"#, r#"퍃"#, r#"퍃"#),
        (r#"퍄"#, r#"퍄"#, r#"퍄"#, r#"퍄"#, r#"퍄"#),
        (r#"퍅"#, r#"퍅"#, r#"퍅"#, r#"퍅"#, r#"퍅"#),
        (r#"퍆"#, r#"퍆"#, r#"퍆"#, r#"퍆"#, r#"퍆"#),
        (r#"퍇"#, r#"퍇"#, r#"퍇"#, r#"퍇"#, r#"퍇"#),
        (r#"퍈"#, r#"퍈"#, r#"퍈"#, r#"퍈"#, r#"퍈"#),
        (r#"퍉"#, r#"퍉"#, r#"퍉"#, r#"퍉"#, r#"퍉"#),
        (r#"퍊"#, r#"퍊"#, r#"퍊"#, r#"퍊"#, r#"퍊"#),
        (r#"퍋"#, r#"퍋"#, r#"퍋"#, r#"퍋"#, r#"퍋"#),
        (r#"퍌"#, r#"퍌"#, r#"퍌"#, r#"퍌"#, r#"퍌"#),
        (r#"퍍"#, r#"퍍"#, r#"퍍"#, r#"퍍"#, r#"퍍"#),
        (r#"퍎"#, r#"퍎"#, r#"퍎"#, r#"퍎"#, r#"퍎"#),
        (r#"퍏"#, r#"퍏"#, r#"퍏"#, r#"퍏"#, r#"퍏"#),
        (r#"퍐"#, r#"퍐"#, r#"퍐"#, r#"퍐"#, r#"퍐"#),
        (r#"퍑"#, r#"퍑"#, r#"퍑"#, r#"퍑"#, r#"퍑"#),
        (r#"퍒"#, r#"퍒"#, r#"퍒"#, r#"퍒"#, r#"퍒"#),
        (r#"퍓"#, r#"퍓"#, r#"퍓"#, r#"퍓"#, r#"퍓"#),
        (r#"퍔"#, r#"퍔"#, r#"퍔"#, r#"퍔"#, r#"퍔"#),
        (r#"퍕"#, r#"퍕"#, r#"퍕"#, r#"퍕"#, r#"퍕"#),
        (r#"퍖"#, r#"퍖"#, r#"퍖"#, r#"퍖"#, r#"퍖"#),
        (r#"퍗"#, r#"퍗"#, r#"퍗"#, r#"퍗"#, r#"퍗"#),
        (r#"퍘"#, r#"퍘"#, r#"퍘"#, r#"퍘"#, r#"퍘"#),
        (r#"퍙"#, r#"퍙"#, r#"퍙"#, r#"퍙"#, r#"퍙"#),
        (r#"퍚"#, r#"퍚"#, r#"퍚"#, r#"퍚"#, r#"퍚"#),
        (r#"퍛"#, r#"퍛"#, r#"퍛"#, r#"퍛"#, r#"퍛"#),
        (r#"퍜"#, r#"퍜"#, r#"퍜"#, r#"퍜"#, r#"퍜"#),
        (r#"퍝"#, r#"퍝"#, r#"퍝"#, r#"퍝"#, r#"퍝"#),
        (r#"퍞"#, r#"퍞"#, r#"퍞"#, r#"퍞"#, r#"퍞"#),
        (r#"퍟"#, r#"퍟"#, r#"퍟"#, r#"퍟"#, r#"퍟"#),
        (r#"퍠"#, r#"퍠"#, r#"퍠"#, r#"퍠"#, r#"퍠"#),
        (r#"퍡"#, r#"퍡"#, r#"퍡"#, r#"퍡"#, r#"퍡"#),
        (r#"퍢"#, r#"퍢"#, r#"퍢"#, r#"퍢"#, r#"퍢"#),
        (r#"퍣"#, r#"퍣"#, r#"퍣"#, r#"퍣"#, r#"퍣"#),
        (r#"퍤"#, r#"퍤"#, r#"퍤"#, r#"퍤"#, r#"퍤"#),
        (r#"퍥"#, r#"퍥"#, r#"퍥"#, r#"퍥"#, r#"퍥"#),
        (r#"퍦"#, r#"퍦"#, r#"퍦"#, r#"퍦"#, r#"퍦"#),
        (r#"퍧"#, r#"퍧"#, r#"퍧"#, r#"퍧"#, r#"퍧"#),
        (r#"퍨"#, r#"퍨"#, r#"퍨"#, r#"퍨"#, r#"퍨"#),
        (r#"퍩"#, r#"퍩"#, r#"퍩"#, r#"퍩"#, r#"퍩"#),
        (r#"퍪"#, r#"퍪"#, r#"퍪"#, r#"퍪"#, r#"퍪"#),
        (r#"퍫"#, r#"퍫"#, r#"퍫"#, r#"퍫"#, r#"퍫"#),
        (r#"퍬"#, r#"퍬"#, r#"퍬"#, r#"퍬"#, r#"퍬"#),
        (r#"퍭"#, r#"퍭"#, r#"퍭"#, r#"퍭"#, r#"퍭"#),
        (r#"퍮"#, r#"퍮"#, r#"퍮"#, r#"퍮"#, r#"퍮"#),
        (r#"퍯"#, r#"퍯"#, r#"퍯"#, r#"퍯"#, r#"퍯"#),
        (r#"퍰"#, r#"퍰"#, r#"퍰"#, r#"퍰"#, r#"퍰"#),
        (r#"퍱"#, r#"퍱"#, r#"퍱"#, r#"퍱"#, r#"퍱"#),
        (r#"퍲"#, r#"퍲"#, r#"퍲"#, r#"퍲"#, r#"퍲"#),
        (r#"퍳"#, r#"퍳"#, r#"퍳"#, r#"퍳"#, r#"퍳"#),
        (r#"퍴"#, r#"퍴"#, r#"퍴"#, r#"퍴"#, r#"퍴"#),
        (r#"퍵"#, r#"퍵"#, r#"퍵"#, r#"퍵"#, r#"퍵"#),
        (r#"퍶"#, r#"퍶"#, r#"퍶"#, r#"퍶"#, r#"퍶"#),
        (r#"퍷"#, r#"퍷"#, r#"퍷"#, r#"퍷"#, r#"퍷"#),
        (r#"퍸"#, r#"퍸"#, r#"퍸"#, r#"퍸"#, r#"퍸"#),
        (r#"퍹"#, r#"퍹"#, r#"퍹"#, r#"퍹"#, r#"퍹"#),
        (r#"퍺"#, r#"퍺"#, r#"퍺"#, r#"퍺"#, r#"퍺"#),
        (r#"퍻"#, r#"퍻"#, r#"퍻"#, r#"퍻"#, r#"퍻"#),
        (r#"퍼"#, r#"퍼"#, r#"퍼"#, r#"퍼"#, r#"퍼"#),
        (r#"퍽"#, r#"퍽"#, r#"퍽"#, r#"퍽"#, r#"퍽"#),
        (r#"퍾"#, r#"퍾"#, r#"퍾"#, r#"퍾"#, r#"퍾"#),
        (r#"퍿"#, r#"퍿"#, r#"퍿"#, r#"퍿"#, r#"퍿"#),
        (r#"펀"#, r#"펀"#, r#"펀"#, r#"펀"#, r#"펀"#),
        (r#"펁"#, r#"펁"#, r#"펁"#, r#"펁"#, r#"펁"#),
        (r#"펂"#, r#"펂"#, r#"펂"#, r#"펂"#, r#"펂"#),
        (r#"펃"#, r#"펃"#, r#"펃"#, r#"펃"#, r#"펃"#),
        (r#"펄"#, r#"펄"#, r#"펄"#, r#"펄"#, r#"펄"#),
        (r#"펅"#, r#"펅"#, r#"펅"#, r#"펅"#, r#"펅"#),
        (r#"펆"#, r#"펆"#, r#"펆"#, r#"펆"#, r#"펆"#),
        (r#"펇"#, r#"펇"#, r#"펇"#, r#"펇"#, r#"펇"#),
        (r#"펈"#, r#"펈"#, r#"펈"#, r#"펈"#, r#"펈"#),
        (r#"펉"#, r#"펉"#, r#"펉"#, r#"펉"#, r#"펉"#),
        (r#"펊"#, r#"펊"#, r#"펊"#, r#"펊"#, r#"펊"#),
        (r#"펋"#, r#"펋"#, r#"펋"#, r#"펋"#, r#"펋"#),
        (r#"펌"#, r#"펌"#, r#"펌"#, r#"펌"#, r#"펌"#),
        (r#"펍"#, r#"펍"#, r#"펍"#, r#"펍"#, r#"펍"#),
        (r#"펎"#, r#"펎"#, r#"펎"#, r#"펎"#, r#"펎"#),
        (r#"펏"#, r#"펏"#, r#"펏"#, r#"펏"#, r#"펏"#),
        (r#"펐"#, r#"펐"#, r#"펐"#, r#"펐"#, r#"펐"#),
        (r#"펑"#, r#"펑"#, r#"펑"#, r#"펑"#, r#"펑"#),
        (r#"펒"#, r#"펒"#, r#"펒"#, r#"펒"#, r#"펒"#),
        (r#"펓"#, r#"펓"#, r#"펓"#, r#"펓"#, r#"펓"#),
        (r#"펔"#, r#"펔"#, r#"펔"#, r#"펔"#, r#"펔"#),
        (r#"펕"#, r#"펕"#, r#"펕"#, r#"펕"#, r#"펕"#),
        (r#"펖"#, r#"펖"#, r#"펖"#, r#"펖"#, r#"펖"#),
        (r#"펗"#, r#"펗"#, r#"펗"#, r#"펗"#, r#"펗"#),
        (r#"페"#, r#"페"#, r#"페"#, r#"페"#, r#"페"#),
        (r#"펙"#, r#"펙"#, r#"펙"#, r#"펙"#, r#"펙"#),
        (r#"펚"#, r#"펚"#, r#"펚"#, r#"펚"#, r#"펚"#),
        (r#"펛"#, r#"펛"#, r#"펛"#, r#"펛"#, r#"펛"#),
        (r#"펜"#, r#"펜"#, r#"펜"#, r#"펜"#, r#"펜"#),
        (r#"펝"#, r#"펝"#, r#"펝"#, r#"펝"#, r#"펝"#),
        (r#"펞"#, r#"펞"#, r#"펞"#, r#"펞"#, r#"펞"#),
        (r#"펟"#, r#"펟"#, r#"펟"#, r#"펟"#, r#"펟"#),
        (r#"펠"#, r#"펠"#, r#"펠"#, r#"펠"#, r#"펠"#),
        (r#"펡"#, r#"펡"#, r#"펡"#, r#"펡"#, r#"펡"#),
        (r#"펢"#, r#"펢"#, r#"펢"#, r#"펢"#, r#"펢"#),
        (r#"펣"#, r#"펣"#, r#"펣"#, r#"펣"#, r#"펣"#),
        (r#"펤"#, r#"펤"#, r#"펤"#, r#"펤"#, r#"펤"#),
        (r#"펥"#, r#"펥"#, r#"펥"#, r#"펥"#, r#"펥"#),
        (r#"펦"#, r#"펦"#, r#"펦"#, r#"펦"#, r#"펦"#),
        (r#"펧"#, r#"펧"#, r#"펧"#, r#"펧"#, r#"펧"#),
        (r#"펨"#, r#"펨"#, r#"펨"#, r#"펨"#, r#"펨"#),
        (r#"펩"#, r#"펩"#, r#"펩"#, r#"펩"#, r#"펩"#),
        (r#"펪"#, r#"펪"#, r#"펪"#, r#"펪"#, r#"펪"#),
        (r#"펫"#, r#"펫"#, r#"펫"#, r#"펫"#, r#"펫"#),
        (r#"펬"#, r#"펬"#, r#"펬"#, r#"펬"#, r#"펬"#),
        (r#"펭"#, r#"펭"#, r#"펭"#, r#"펭"#, r#"펭"#),
        (r#"펮"#, r#"펮"#, r#"펮"#, r#"펮"#, r#"펮"#),
        (r#"펯"#, r#"펯"#, r#"펯"#, r#"펯"#, r#"펯"#),
        (r#"펰"#, r#"펰"#, r#"펰"#, r#"펰"#, r#"펰"#),
        (r#"펱"#, r#"펱"#, r#"펱"#, r#"펱"#, r#"펱"#),
        (r#"펲"#, r#"펲"#, r#"펲"#, r#"펲"#, r#"펲"#),
        (r#"펳"#, r#"펳"#, r#"펳"#, r#"펳"#, r#"펳"#),
        (r#"펴"#, r#"펴"#, r#"펴"#, r#"펴"#, r#"펴"#),
        (r#"펵"#, r#"펵"#, r#"펵"#, r#"펵"#, r#"펵"#),
        (r#"펶"#, r#"펶"#, r#"펶"#, r#"펶"#, r#"펶"#),
        (r#"펷"#, r#"펷"#, r#"펷"#, r#"펷"#, r#"펷"#),
        (r#"편"#, r#"편"#, r#"편"#, r#"편"#, r#"편"#),
        (r#"펹"#, r#"펹"#, r#"펹"#, r#"펹"#, r#"펹"#),
        (r#"펺"#, r#"펺"#, r#"펺"#, r#"펺"#, r#"펺"#),
        (r#"펻"#, r#"펻"#, r#"펻"#, r#"펻"#, r#"펻"#),
        (r#"펼"#, r#"펼"#, r#"펼"#, r#"펼"#, r#"펼"#),
        (r#"펽"#, r#"펽"#, r#"펽"#, r#"펽"#, r#"펽"#),
        (r#"펾"#, r#"펾"#, r#"펾"#, r#"펾"#, r#"펾"#),
        (r#"펿"#, r#"펿"#, r#"펿"#, r#"펿"#, r#"펿"#),
        (r#"폀"#, r#"폀"#, r#"폀"#, r#"폀"#, r#"폀"#),
        (r#"폁"#, r#"폁"#, r#"폁"#, r#"폁"#, r#"폁"#),
        (r#"폂"#, r#"폂"#, r#"폂"#, r#"폂"#, r#"폂"#),
        (r#"폃"#, r#"폃"#, r#"폃"#, r#"폃"#, r#"폃"#),
        (r#"폄"#, r#"폄"#, r#"폄"#, r#"폄"#, r#"폄"#),
        (r#"폅"#, r#"폅"#, r#"폅"#, r#"폅"#, r#"폅"#),
        (r#"폆"#, r#"폆"#, r#"폆"#, r#"폆"#, r#"폆"#),
        (r#"폇"#, r#"폇"#, r#"폇"#, r#"폇"#, r#"폇"#),
        (r#"폈"#, r#"폈"#, r#"폈"#, r#"폈"#, r#"폈"#),
        (r#"평"#, r#"평"#, r#"평"#, r#"평"#, r#"평"#),
        (r#"폊"#, r#"폊"#, r#"폊"#, r#"폊"#, r#"폊"#),
        (r#"폋"#, r#"폋"#, r#"폋"#, r#"폋"#, r#"폋"#),
        (r#"폌"#, r#"폌"#, r#"폌"#, r#"폌"#, r#"폌"#),
        (r#"폍"#, r#"폍"#, r#"폍"#, r#"폍"#, r#"폍"#),
        (r#"폎"#, r#"폎"#, r#"폎"#, r#"폎"#, r#"폎"#),
        (r#"폏"#, r#"폏"#, r#"폏"#, r#"폏"#, r#"폏"#),
        (r#"폐"#, r#"폐"#, r#"폐"#, r#"폐"#, r#"폐"#),
        (r#"폑"#, r#"폑"#, r#"폑"#, r#"폑"#, r#"폑"#),
        (r#"폒"#, r#"폒"#, r#"폒"#, r#"폒"#, r#"폒"#),
        (r#"폓"#, r#"폓"#, r#"폓"#, r#"폓"#, r#"폓"#),
        (r#"폔"#, r#"폔"#, r#"폔"#, r#"폔"#, r#"폔"#),
        (r#"폕"#, r#"폕"#, r#"폕"#, r#"폕"#, r#"폕"#),
        (r#"폖"#, r#"폖"#, r#"폖"#, r#"폖"#, r#"폖"#),
        (r#"폗"#, r#"폗"#, r#"폗"#, r#"폗"#, r#"폗"#),
        (r#"폘"#, r#"폘"#, r#"폘"#, r#"폘"#, r#"폘"#),
        (r#"폙"#, r#"폙"#, r#"폙"#, r#"폙"#, r#"폙"#),
        (r#"폚"#, r#"폚"#, r#"폚"#, r#"폚"#, r#"폚"#),
        (r#"폛"#, r#"폛"#, r#"폛"#, r#"폛"#, r#"폛"#),
        (r#"폜"#, r#"폜"#, r#"폜"#, r#"폜"#, r#"폜"#),
        (r#"폝"#, r#"폝"#, r#"폝"#, r#"폝"#, r#"폝"#),
        (r#"폞"#, r#"폞"#, r#"폞"#, r#"폞"#, r#"폞"#),
        (r#"폟"#, r#"폟"#, r#"폟"#, r#"폟"#, r#"폟"#),
        (r#"폠"#, r#"폠"#, r#"폠"#, r#"폠"#, r#"폠"#),
        (r#"폡"#, r#"폡"#, r#"폡"#, r#"폡"#, r#"폡"#),
        (r#"폢"#, r#"폢"#, r#"폢"#, r#"폢"#, r#"폢"#),
        (r#"폣"#, r#"폣"#, r#"폣"#, r#"폣"#, r#"폣"#),
        (r#"폤"#, r#"폤"#, r#"폤"#, r#"폤"#, r#"폤"#),
        (r#"폥"#, r#"폥"#, r#"폥"#, r#"폥"#, r#"폥"#),
        (r#"폦"#, r#"폦"#, r#"폦"#, r#"폦"#, r#"폦"#),
        (r#"폧"#, r#"폧"#, r#"폧"#, r#"폧"#, r#"폧"#),
        (r#"폨"#, r#"폨"#, r#"폨"#, r#"폨"#, r#"폨"#),
        (r#"폩"#, r#"폩"#, r#"폩"#, r#"폩"#, r#"폩"#),
        (r#"폪"#, r#"폪"#, r#"폪"#, r#"폪"#, r#"폪"#),
        (r#"폫"#, r#"폫"#, r#"폫"#, r#"폫"#, r#"폫"#),
        (r#"포"#, r#"포"#, r#"포"#, r#"포"#, r#"포"#),
        (r#"폭"#, r#"폭"#, r#"폭"#, r#"폭"#, r#"폭"#),
        (r#"폮"#, r#"폮"#, r#"폮"#, r#"폮"#, r#"폮"#),
        (r#"폯"#, r#"폯"#, r#"폯"#, r#"폯"#, r#"폯"#),
        (r#"폰"#, r#"폰"#, r#"폰"#, r#"폰"#, r#"폰"#),
        (r#"폱"#, r#"폱"#, r#"폱"#, r#"폱"#, r#"폱"#),
        (r#"폲"#, r#"폲"#, r#"폲"#, r#"폲"#, r#"폲"#),
        (r#"폳"#, r#"폳"#, r#"폳"#, r#"폳"#, r#"폳"#),
        (r#"폴"#, r#"폴"#, r#"폴"#, r#"폴"#, r#"폴"#),
        (r#"폵"#, r#"폵"#, r#"폵"#, r#"폵"#, r#"폵"#),
        (r#"폶"#, r#"폶"#, r#"폶"#, r#"폶"#, r#"폶"#),
        (r#"폷"#, r#"폷"#, r#"폷"#, r#"폷"#, r#"폷"#),
        (r#"폸"#, r#"폸"#, r#"폸"#, r#"폸"#, r#"폸"#),
        (r#"폹"#, r#"폹"#, r#"폹"#, r#"폹"#, r#"폹"#),
        (r#"폺"#, r#"폺"#, r#"폺"#, r#"폺"#, r#"폺"#),
        (r#"폻"#, r#"폻"#, r#"폻"#, r#"폻"#, r#"폻"#),
        (r#"폼"#, r#"폼"#, r#"폼"#, r#"폼"#, r#"폼"#),
        (r#"폽"#, r#"폽"#, r#"폽"#, r#"폽"#, r#"폽"#),
        (r#"폾"#, r#"폾"#, r#"폾"#, r#"폾"#, r#"폾"#),
        (r#"폿"#, r#"폿"#, r#"폿"#, r#"폿"#, r#"폿"#),
        (r#"퐀"#, r#"퐀"#, r#"퐀"#, r#"퐀"#, r#"퐀"#),
        (r#"퐁"#, r#"퐁"#, r#"퐁"#, r#"퐁"#, r#"퐁"#),
        (r#"퐂"#, r#"퐂"#, r#"퐂"#, r#"퐂"#, r#"퐂"#),
        (r#"퐃"#, r#"퐃"#, r#"퐃"#, r#"퐃"#, r#"퐃"#),
        (r#"퐄"#, r#"퐄"#, r#"퐄"#, r#"퐄"#, r#"퐄"#),
        (r#"퐅"#, r#"퐅"#, r#"퐅"#, r#"퐅"#, r#"퐅"#),
        (r#"퐆"#, r#"퐆"#, r#"퐆"#, r#"퐆"#, r#"퐆"#),
        (r#"퐇"#, r#"퐇"#, r#"퐇"#, r#"퐇"#, r#"퐇"#),
        (r#"퐈"#, r#"퐈"#, r#"퐈"#, r#"퐈"#, r#"퐈"#),
        (r#"퐉"#, r#"퐉"#, r#"퐉"#, r#"퐉"#, r#"퐉"#),
        (r#"퐊"#, r#"퐊"#, r#"퐊"#, r#"퐊"#, r#"퐊"#),
        (r#"퐋"#, r#"퐋"#, r#"퐋"#, r#"퐋"#, r#"퐋"#),
        (r#"퐌"#, r#"퐌"#, r#"퐌"#, r#"퐌"#, r#"퐌"#),
        (r#"퐍"#, r#"퐍"#, r#"퐍"#, r#"퐍"#, r#"퐍"#),
        (r#"퐎"#, r#"퐎"#, r#"퐎"#, r#"퐎"#, r#"퐎"#),
        (r#"퐏"#, r#"퐏"#, r#"퐏"#, r#"퐏"#, r#"퐏"#),
        (r#"퐐"#, r#"퐐"#, r#"퐐"#, r#"퐐"#, r#"퐐"#),
        (r#"퐑"#, r#"퐑"#, r#"퐑"#, r#"퐑"#, r#"퐑"#),
        (r#"퐒"#, r#"퐒"#, r#"퐒"#, r#"퐒"#, r#"퐒"#),
        (r#"퐓"#, r#"퐓"#, r#"퐓"#, r#"퐓"#, r#"퐓"#),
        (r#"퐔"#, r#"퐔"#, r#"퐔"#, r#"퐔"#, r#"퐔"#),
        (r#"퐕"#, r#"퐕"#, r#"퐕"#, r#"퐕"#, r#"퐕"#),
        (r#"퐖"#, r#"퐖"#, r#"퐖"#, r#"퐖"#, r#"퐖"#),
        (r#"퐗"#, r#"퐗"#, r#"퐗"#, r#"퐗"#, r#"퐗"#),
        (r#"퐘"#, r#"퐘"#, r#"퐘"#, r#"퐘"#, r#"퐘"#),
        (r#"퐙"#, r#"퐙"#, r#"퐙"#, r#"퐙"#, r#"퐙"#),
        (r#"퐚"#, r#"퐚"#, r#"퐚"#, r#"퐚"#, r#"퐚"#),
        (r#"퐛"#, r#"퐛"#, r#"퐛"#, r#"퐛"#, r#"퐛"#),
        (r#"퐜"#, r#"퐜"#, r#"퐜"#, r#"퐜"#, r#"퐜"#),
        (r#"퐝"#, r#"퐝"#, r#"퐝"#, r#"퐝"#, r#"퐝"#),
        (r#"퐞"#, r#"퐞"#, r#"퐞"#, r#"퐞"#, r#"퐞"#),
        (r#"퐟"#, r#"퐟"#, r#"퐟"#, r#"퐟"#, r#"퐟"#),
        (r#"퐠"#, r#"퐠"#, r#"퐠"#, r#"퐠"#, r#"퐠"#),
        (r#"퐡"#, r#"퐡"#, r#"퐡"#, r#"퐡"#, r#"퐡"#),
        (r#"퐢"#, r#"퐢"#, r#"퐢"#, r#"퐢"#, r#"퐢"#),
        (r#"퐣"#, r#"퐣"#, r#"퐣"#, r#"퐣"#, r#"퐣"#),
        (r#"퐤"#, r#"퐤"#, r#"퐤"#, r#"퐤"#, r#"퐤"#),
        (r#"퐥"#, r#"퐥"#, r#"퐥"#, r#"퐥"#, r#"퐥"#),
        (r#"퐦"#, r#"퐦"#, r#"퐦"#, r#"퐦"#, r#"퐦"#),
        (r#"퐧"#, r#"퐧"#, r#"퐧"#, r#"퐧"#, r#"퐧"#),
        (r#"퐨"#, r#"퐨"#, r#"퐨"#, r#"퐨"#, r#"퐨"#),
        (r#"퐩"#, r#"퐩"#, r#"퐩"#, r#"퐩"#, r#"퐩"#),
        (r#"퐪"#, r#"퐪"#, r#"퐪"#, r#"퐪"#, r#"퐪"#),
        (r#"퐫"#, r#"퐫"#, r#"퐫"#, r#"퐫"#, r#"퐫"#),
        (r#"퐬"#, r#"퐬"#, r#"퐬"#, r#"퐬"#, r#"퐬"#),
        (r#"퐭"#, r#"퐭"#, r#"퐭"#, r#"퐭"#, r#"퐭"#),
        (r#"퐮"#, r#"퐮"#, r#"퐮"#, r#"퐮"#, r#"퐮"#),
        (r#"퐯"#, r#"퐯"#, r#"퐯"#, r#"퐯"#, r#"퐯"#),
        (r#"퐰"#, r#"퐰"#, r#"퐰"#, r#"퐰"#, r#"퐰"#),
        (r#"퐱"#, r#"퐱"#, r#"퐱"#, r#"퐱"#, r#"퐱"#),
        (r#"퐲"#, r#"퐲"#, r#"퐲"#, r#"퐲"#, r#"퐲"#),
        (r#"퐳"#, r#"퐳"#, r#"퐳"#, r#"퐳"#, r#"퐳"#),
        (r#"퐴"#, r#"퐴"#, r#"퐴"#, r#"퐴"#, r#"퐴"#),
        (r#"퐵"#, r#"퐵"#, r#"퐵"#, r#"퐵"#, r#"퐵"#),
        (r#"퐶"#, r#"퐶"#, r#"퐶"#, r#"퐶"#, r#"퐶"#),
        (r#"퐷"#, r#"퐷"#, r#"퐷"#, r#"퐷"#, r#"퐷"#),
        (r#"퐸"#, r#"퐸"#, r#"퐸"#, r#"퐸"#, r#"퐸"#),
        (r#"퐹"#, r#"퐹"#, r#"퐹"#, r#"퐹"#, r#"퐹"#),
        (r#"퐺"#, r#"퐺"#, r#"퐺"#, r#"퐺"#, r#"퐺"#),
        (r#"퐻"#, r#"퐻"#, r#"퐻"#, r#"퐻"#, r#"퐻"#),
        (r#"퐼"#, r#"퐼"#, r#"퐼"#, r#"퐼"#, r#"퐼"#),
        (r#"퐽"#, r#"퐽"#, r#"퐽"#, r#"퐽"#, r#"퐽"#),
        (r#"퐾"#, r#"퐾"#, r#"퐾"#, r#"퐾"#, r#"퐾"#),
        (r#"퐿"#, r#"퐿"#, r#"퐿"#, r#"퐿"#, r#"퐿"#),
        (r#"푀"#, r#"푀"#, r#"푀"#, r#"푀"#, r#"푀"#),
        (r#"푁"#, r#"푁"#, r#"푁"#, r#"푁"#, r#"푁"#),
        (r#"푂"#, r#"푂"#, r#"푂"#, r#"푂"#, r#"푂"#),
        (r#"푃"#, r#"푃"#, r#"푃"#, r#"푃"#, r#"푃"#),
        (r#"푄"#, r#"푄"#, r#"푄"#, r#"푄"#, r#"푄"#),
        (r#"푅"#, r#"푅"#, r#"푅"#, r#"푅"#, r#"푅"#),
        (r#"푆"#, r#"푆"#, r#"푆"#, r#"푆"#, r#"푆"#),
        (r#"푇"#, r#"푇"#, r#"푇"#, r#"푇"#, r#"푇"#),
        (r#"푈"#, r#"푈"#, r#"푈"#, r#"푈"#, r#"푈"#),
        (r#"푉"#, r#"푉"#, r#"푉"#, r#"푉"#, r#"푉"#),
        (r#"푊"#, r#"푊"#, r#"푊"#, r#"푊"#, r#"푊"#),
        (r#"푋"#, r#"푋"#, r#"푋"#, r#"푋"#, r#"푋"#),
        (r#"푌"#, r#"푌"#, r#"푌"#, r#"푌"#, r#"푌"#),
        (r#"푍"#, r#"푍"#, r#"푍"#, r#"푍"#, r#"푍"#),
        (r#"푎"#, r#"푎"#, r#"푎"#, r#"푎"#, r#"푎"#),
        (r#"푏"#, r#"푏"#, r#"푏"#, r#"푏"#, r#"푏"#),
        (r#"푐"#, r#"푐"#, r#"푐"#, r#"푐"#, r#"푐"#),
        (r#"푑"#, r#"푑"#, r#"푑"#, r#"푑"#, r#"푑"#),
        (r#"푒"#, r#"푒"#, r#"푒"#, r#"푒"#, r#"푒"#),
        (r#"푓"#, r#"푓"#, r#"푓"#, r#"푓"#, r#"푓"#),
        (r#"푔"#, r#"푔"#, r#"푔"#, r#"푔"#, r#"푔"#),
        (r#"푕"#, r#"푕"#, r#"푕"#, r#"푕"#, r#"푕"#),
        (r#"푖"#, r#"푖"#, r#"푖"#, r#"푖"#, r#"푖"#),
        (r#"푗"#, r#"푗"#, r#"푗"#, r#"푗"#, r#"푗"#),
        (r#"푘"#, r#"푘"#, r#"푘"#, r#"푘"#, r#"푘"#),
        (r#"푙"#, r#"푙"#, r#"푙"#, r#"푙"#, r#"푙"#),
        (r#"푚"#, r#"푚"#, r#"푚"#, r#"푚"#, r#"푚"#),
        (r#"푛"#, r#"푛"#, r#"푛"#, r#"푛"#, r#"푛"#),
        (r#"표"#, r#"표"#, r#"표"#, r#"표"#, r#"표"#),
        (r#"푝"#, r#"푝"#, r#"푝"#, r#"푝"#, r#"푝"#),
        (r#"푞"#, r#"푞"#, r#"푞"#, r#"푞"#, r#"푞"#),
        (r#"푟"#, r#"푟"#, r#"푟"#, r#"푟"#, r#"푟"#),
        (r#"푠"#, r#"푠"#, r#"푠"#, r#"푠"#, r#"푠"#),
        (r#"푡"#, r#"푡"#, r#"푡"#, r#"푡"#, r#"푡"#),
        (r#"푢"#, r#"푢"#, r#"푢"#, r#"푢"#, r#"푢"#),
        (r#"푣"#, r#"푣"#, r#"푣"#, r#"푣"#, r#"푣"#),
        (r#"푤"#, r#"푤"#, r#"푤"#, r#"푤"#, r#"푤"#),
        (r#"푥"#, r#"푥"#, r#"푥"#, r#"푥"#, r#"푥"#),
        (r#"푦"#, r#"푦"#, r#"푦"#, r#"푦"#, r#"푦"#),
        (r#"푧"#, r#"푧"#, r#"푧"#, r#"푧"#, r#"푧"#),
        (r#"푨"#, r#"푨"#, r#"푨"#, r#"푨"#, r#"푨"#),
        (r#"푩"#, r#"푩"#, r#"푩"#, r#"푩"#, r#"푩"#),
        (r#"푪"#, r#"푪"#, r#"푪"#, r#"푪"#, r#"푪"#),
        (r#"푫"#, r#"푫"#, r#"푫"#, r#"푫"#, r#"푫"#),
        (r#"푬"#, r#"푬"#, r#"푬"#, r#"푬"#, r#"푬"#),
        (r#"푭"#, r#"푭"#, r#"푭"#, r#"푭"#, r#"푭"#),
        (r#"푮"#, r#"푮"#, r#"푮"#, r#"푮"#, r#"푮"#),
        (r#"푯"#, r#"푯"#, r#"푯"#, r#"푯"#, r#"푯"#),
        (r#"푰"#, r#"푰"#, r#"푰"#, r#"푰"#, r#"푰"#),
        (r#"푱"#, r#"푱"#, r#"푱"#, r#"푱"#, r#"푱"#),
        (r#"푲"#, r#"푲"#, r#"푲"#, r#"푲"#, r#"푲"#),
        (r#"푳"#, r#"푳"#, r#"푳"#, r#"푳"#, r#"푳"#),
        (r#"푴"#, r#"푴"#, r#"푴"#, r#"푴"#, r#"푴"#),
        (r#"푵"#, r#"푵"#, r#"푵"#, r#"푵"#, r#"푵"#),
        (r#"푶"#, r#"푶"#, r#"푶"#, r#"푶"#, r#"푶"#),
        (r#"푷"#, r#"푷"#, r#"푷"#, r#"푷"#, r#"푷"#),
        (r#"푸"#, r#"푸"#, r#"푸"#, r#"푸"#, r#"푸"#),
        (r#"푹"#, r#"푹"#, r#"푹"#, r#"푹"#, r#"푹"#),
        (r#"푺"#, r#"푺"#, r#"푺"#, r#"푺"#, r#"푺"#),
        (r#"푻"#, r#"푻"#, r#"푻"#, r#"푻"#, r#"푻"#),
        (r#"푼"#, r#"푼"#, r#"푼"#, r#"푼"#, r#"푼"#),
        (r#"푽"#, r#"푽"#, r#"푽"#, r#"푽"#, r#"푽"#),
        (r#"푾"#, r#"푾"#, r#"푾"#, r#"푾"#, r#"푾"#),
        (r#"푿"#, r#"푿"#, r#"푿"#, r#"푿"#, r#"푿"#),
        (r#"풀"#, r#"풀"#, r#"풀"#, r#"풀"#, r#"풀"#),
        (r#"풁"#, r#"풁"#, r#"풁"#, r#"풁"#, r#"풁"#),
        (r#"풂"#, r#"풂"#, r#"풂"#, r#"풂"#, r#"풂"#),
        (r#"풃"#, r#"풃"#, r#"풃"#, r#"풃"#, r#"풃"#),
        (r#"풄"#, r#"풄"#, r#"풄"#, r#"풄"#, r#"풄"#),
        (r#"풅"#, r#"풅"#, r#"풅"#, r#"풅"#, r#"풅"#),
        (r#"풆"#, r#"풆"#, r#"풆"#, r#"풆"#, r#"풆"#),
        (r#"풇"#, r#"풇"#, r#"풇"#, r#"풇"#, r#"풇"#),
        (r#"품"#, r#"품"#, r#"품"#, r#"품"#, r#"품"#),
        (r#"풉"#, r#"풉"#, r#"풉"#, r#"풉"#, r#"풉"#),
        (r#"풊"#, r#"풊"#, r#"풊"#, r#"풊"#, r#"풊"#),
        (r#"풋"#, r#"풋"#, r#"풋"#, r#"풋"#, r#"풋"#),
        (r#"풌"#, r#"풌"#, r#"풌"#, r#"풌"#, r#"풌"#),
        (r#"풍"#, r#"풍"#, r#"풍"#, r#"풍"#, r#"풍"#),
        (r#"풎"#, r#"풎"#, r#"풎"#, r#"풎"#, r#"풎"#),
        (r#"풏"#, r#"풏"#, r#"풏"#, r#"풏"#, r#"풏"#),
        (r#"풐"#, r#"풐"#, r#"풐"#, r#"풐"#, r#"풐"#),
        (r#"풑"#, r#"풑"#, r#"풑"#, r#"풑"#, r#"풑"#),
        (r#"풒"#, r#"풒"#, r#"풒"#, r#"풒"#, r#"풒"#),
        (r#"풓"#, r#"풓"#, r#"풓"#, r#"풓"#, r#"풓"#),
        (r#"풔"#, r#"풔"#, r#"풔"#, r#"풔"#, r#"풔"#),
        (r#"풕"#, r#"풕"#, r#"풕"#, r#"풕"#, r#"풕"#),
        (r#"풖"#, r#"풖"#, r#"풖"#, r#"풖"#, r#"풖"#),
        (r#"풗"#, r#"풗"#, r#"풗"#, r#"풗"#, r#"풗"#),
        (r#"풘"#, r#"풘"#, r#"풘"#, r#"풘"#, r#"풘"#),
        (r#"풙"#, r#"풙"#, r#"풙"#, r#"풙"#, r#"풙"#),
        (r#"풚"#, r#"풚"#, r#"풚"#, r#"풚"#, r#"풚"#),
        (r#"풛"#, r#"풛"#, r#"풛"#, r#"풛"#, r#"풛"#),
        (r#"풜"#, r#"풜"#, r#"풜"#, r#"풜"#, r#"풜"#),
        (r#"풝"#, r#"풝"#, r#"풝"#, r#"풝"#, r#"풝"#),
        (r#"풞"#, r#"풞"#, r#"풞"#, r#"풞"#, r#"풞"#),
        (r#"풟"#, r#"풟"#, r#"풟"#, r#"풟"#, r#"풟"#),
        (r#"풠"#, r#"풠"#, r#"풠"#, r#"풠"#, r#"풠"#),
        (r#"풡"#, r#"풡"#, r#"풡"#, r#"풡"#, r#"풡"#),
        (r#"풢"#, r#"풢"#, r#"풢"#, r#"풢"#, r#"풢"#),
        (r#"풣"#, r#"풣"#, r#"풣"#, r#"풣"#, r#"풣"#),
        (r#"풤"#, r#"풤"#, r#"풤"#, r#"풤"#, r#"풤"#),
        (r#"풥"#, r#"풥"#, r#"풥"#, r#"풥"#, r#"풥"#),
        (r#"풦"#, r#"풦"#, r#"풦"#, r#"풦"#, r#"풦"#),
        (r#"풧"#, r#"풧"#, r#"풧"#, r#"풧"#, r#"풧"#),
        (r#"풨"#, r#"풨"#, r#"풨"#, r#"풨"#, r#"풨"#),
        (r#"풩"#, r#"풩"#, r#"풩"#, r#"풩"#, r#"풩"#),
        (r#"풪"#, r#"풪"#, r#"풪"#, r#"풪"#, r#"풪"#),
        (r#"풫"#, r#"풫"#, r#"풫"#, r#"풫"#, r#"풫"#),
        (r#"풬"#, r#"풬"#, r#"풬"#, r#"풬"#, r#"풬"#),
        (r#"풭"#, r#"풭"#, r#"풭"#, r#"풭"#, r#"풭"#),
        (r#"풮"#, r#"풮"#, r#"풮"#, r#"풮"#, r#"풮"#),
        (r#"풯"#, r#"풯"#, r#"풯"#, r#"풯"#, r#"풯"#),
        (r#"풰"#, r#"풰"#, r#"풰"#, r#"풰"#, r#"풰"#),
        (r#"풱"#, r#"풱"#, r#"풱"#, r#"풱"#, r#"풱"#),
        (r#"풲"#, r#"풲"#, r#"풲"#, r#"풲"#, r#"풲"#),
        (r#"풳"#, r#"풳"#, r#"풳"#, r#"풳"#, r#"풳"#),
        (r#"풴"#, r#"풴"#, r#"풴"#, r#"풴"#, r#"풴"#),
        (r#"풵"#, r#"풵"#, r#"풵"#, r#"풵"#, r#"풵"#),
        (r#"풶"#, r#"풶"#, r#"풶"#, r#"풶"#, r#"풶"#),
        (r#"풷"#, r#"풷"#, r#"풷"#, r#"풷"#, r#"풷"#),
        (r#"풸"#, r#"풸"#, r#"풸"#, r#"풸"#, r#"풸"#),
        (r#"풹"#, r#"풹"#, r#"풹"#, r#"풹"#, r#"풹"#),
        (r#"풺"#, r#"풺"#, r#"풺"#, r#"풺"#, r#"풺"#),
        (r#"풻"#, r#"풻"#, r#"풻"#, r#"풻"#, r#"풻"#),
        (r#"풼"#, r#"풼"#, r#"풼"#, r#"풼"#, r#"풼"#),
        (r#"풽"#, r#"풽"#, r#"풽"#, r#"풽"#, r#"풽"#),
        (r#"풾"#, r#"풾"#, r#"풾"#, r#"풾"#, r#"풾"#),
        (r#"풿"#, r#"풿"#, r#"풿"#, r#"풿"#, r#"풿"#),
        (r#"퓀"#, r#"퓀"#, r#"퓀"#, r#"퓀"#, r#"퓀"#),
        (r#"퓁"#, r#"퓁"#, r#"퓁"#, r#"퓁"#, r#"퓁"#),
        (r#"퓂"#, r#"퓂"#, r#"퓂"#, r#"퓂"#, r#"퓂"#),
        (r#"퓃"#, r#"퓃"#, r#"퓃"#, r#"퓃"#, r#"퓃"#),
        (r#"퓄"#, r#"퓄"#, r#"퓄"#, r#"퓄"#, r#"퓄"#),
        (r#"퓅"#, r#"퓅"#, r#"퓅"#, r#"퓅"#, r#"퓅"#),
        (r#"퓆"#, r#"퓆"#, r#"퓆"#, r#"퓆"#, r#"퓆"#),
        (r#"퓇"#, r#"퓇"#, r#"퓇"#, r#"퓇"#, r#"퓇"#),
        (r#"퓈"#, r#"퓈"#, r#"퓈"#, r#"퓈"#, r#"퓈"#),
        (r#"퓉"#, r#"퓉"#, r#"퓉"#, r#"퓉"#, r#"퓉"#),
        (r#"퓊"#, r#"퓊"#, r#"퓊"#, r#"퓊"#, r#"퓊"#),
        (r#"퓋"#, r#"퓋"#, r#"퓋"#, r#"퓋"#, r#"퓋"#),
        (r#"퓌"#, r#"퓌"#, r#"퓌"#, r#"퓌"#, r#"퓌"#),
        (r#"퓍"#, r#"퓍"#, r#"퓍"#, r#"퓍"#, r#"퓍"#),
        (r#"퓎"#, r#"퓎"#, r#"퓎"#, r#"퓎"#, r#"퓎"#),
        (r#"퓏"#, r#"퓏"#, r#"퓏"#, r#"퓏"#, r#"퓏"#),
        (r#"퓐"#, r#"퓐"#, r#"퓐"#, r#"퓐"#, r#"퓐"#),
        (r#"퓑"#, r#"퓑"#, r#"퓑"#, r#"퓑"#, r#"퓑"#),
        (r#"퓒"#, r#"퓒"#, r#"퓒"#, r#"퓒"#, r#"퓒"#),
        (r#"퓓"#, r#"퓓"#, r#"퓓"#, r#"퓓"#, r#"퓓"#),
        (r#"퓔"#, r#"퓔"#, r#"퓔"#, r#"퓔"#, r#"퓔"#),
        (r#"퓕"#, r#"퓕"#, r#"퓕"#, r#"퓕"#, r#"퓕"#),
        (r#"퓖"#, r#"퓖"#, r#"퓖"#, r#"퓖"#, r#"퓖"#),
        (r#"퓗"#, r#"퓗"#, r#"퓗"#, r#"퓗"#, r#"퓗"#),
        (r#"퓘"#, r#"퓘"#, r#"퓘"#, r#"퓘"#, r#"퓘"#),
        (r#"퓙"#, r#"퓙"#, r#"퓙"#, r#"퓙"#, r#"퓙"#),
        (r#"퓚"#, r#"퓚"#, r#"퓚"#, r#"퓚"#, r#"퓚"#),
        (r#"퓛"#, r#"퓛"#, r#"퓛"#, r#"퓛"#, r#"퓛"#),
        (r#"퓜"#, r#"퓜"#, r#"퓜"#, r#"퓜"#, r#"퓜"#),
        (r#"퓝"#, r#"퓝"#, r#"퓝"#, r#"퓝"#, r#"퓝"#),
        (r#"퓞"#, r#"퓞"#, r#"퓞"#, r#"퓞"#, r#"퓞"#),
        (r#"퓟"#, r#"퓟"#, r#"퓟"#, r#"퓟"#, r#"퓟"#),
        (r#"퓠"#, r#"퓠"#, r#"퓠"#, r#"퓠"#, r#"퓠"#),
        (r#"퓡"#, r#"퓡"#, r#"퓡"#, r#"퓡"#, r#"퓡"#),
        (r#"퓢"#, r#"퓢"#, r#"퓢"#, r#"퓢"#, r#"퓢"#),
        (r#"퓣"#, r#"퓣"#, r#"퓣"#, r#"퓣"#, r#"퓣"#),
        (r#"퓤"#, r#"퓤"#, r#"퓤"#, r#"퓤"#, r#"퓤"#),
        (r#"퓥"#, r#"퓥"#, r#"퓥"#, r#"퓥"#, r#"퓥"#),
        (r#"퓦"#, r#"퓦"#, r#"퓦"#, r#"퓦"#, r#"퓦"#),
        (r#"퓧"#, r#"퓧"#, r#"퓧"#, r#"퓧"#, r#"퓧"#),
        (r#"퓨"#, r#"퓨"#, r#"퓨"#, r#"퓨"#, r#"퓨"#),
        (r#"퓩"#, r#"퓩"#, r#"퓩"#, r#"퓩"#, r#"퓩"#),
        (r#"퓪"#, r#"퓪"#, r#"퓪"#, r#"퓪"#, r#"퓪"#),
        (r#"퓫"#, r#"퓫"#, r#"퓫"#, r#"퓫"#, r#"퓫"#),
        (r#"퓬"#, r#"퓬"#, r#"퓬"#, r#"퓬"#, r#"퓬"#),
        (r#"퓭"#, r#"퓭"#, r#"퓭"#, r#"퓭"#, r#"퓭"#),
        (r#"퓮"#, r#"퓮"#, r#"퓮"#, r#"퓮"#, r#"퓮"#),
        (r#"퓯"#, r#"퓯"#, r#"퓯"#, r#"퓯"#, r#"퓯"#),
        (r#"퓰"#, r#"퓰"#, r#"퓰"#, r#"퓰"#, r#"퓰"#),
        (r#"퓱"#, r#"퓱"#, r#"퓱"#, r#"퓱"#, r#"퓱"#),
        (r#"퓲"#, r#"퓲"#, r#"퓲"#, r#"퓲"#, r#"퓲"#),
        (r#"퓳"#, r#"퓳"#, r#"퓳"#, r#"퓳"#, r#"퓳"#),
        (r#"퓴"#, r#"퓴"#, r#"퓴"#, r#"퓴"#, r#"퓴"#),
        (r#"퓵"#, r#"퓵"#, r#"퓵"#, r#"퓵"#, r#"퓵"#),
        (r#"퓶"#, r#"퓶"#, r#"퓶"#, r#"퓶"#, r#"퓶"#),
        (r#"퓷"#, r#"퓷"#, r#"퓷"#, r#"퓷"#, r#"퓷"#),
        (r#"퓸"#, r#"퓸"#, r#"퓸"#, r#"퓸"#, r#"퓸"#),
        (r#"퓹"#, r#"퓹"#, r#"퓹"#, r#"퓹"#, r#"퓹"#),
        (r#"퓺"#, r#"퓺"#, r#"퓺"#, r#"퓺"#, r#"퓺"#),
        (r#"퓻"#, r#"퓻"#, r#"퓻"#, r#"퓻"#, r#"퓻"#),
        (r#"퓼"#, r#"퓼"#, r#"퓼"#, r#"퓼"#, r#"퓼"#),
        (r#"퓽"#, r#"퓽"#, r#"퓽"#, r#"퓽"#, r#"퓽"#),
        (r#"퓾"#, r#"퓾"#, r#"퓾"#, r#"퓾"#, r#"퓾"#),
        (r#"퓿"#, r#"퓿"#, r#"퓿"#, r#"퓿"#, r#"퓿"#),
        (r#"픀"#, r#"픀"#, r#"픀"#, r#"픀"#, r#"픀"#),
        (r#"픁"#, r#"픁"#, r#"픁"#, r#"픁"#, r#"픁"#),
        (r#"픂"#, r#"픂"#, r#"픂"#, r#"픂"#, r#"픂"#),
        (r#"픃"#, r#"픃"#, r#"픃"#, r#"픃"#, r#"픃"#),
        (r#"프"#, r#"프"#, r#"프"#, r#"프"#, r#"프"#),
        (r#"픅"#, r#"픅"#, r#"픅"#, r#"픅"#, r#"픅"#),
        (r#"픆"#, r#"픆"#, r#"픆"#, r#"픆"#, r#"픆"#),
        (r#"픇"#, r#"픇"#, r#"픇"#, r#"픇"#, r#"픇"#),
        (r#"픈"#, r#"픈"#, r#"픈"#, r#"픈"#, r#"픈"#),
        (r#"픉"#, r#"픉"#, r#"픉"#, r#"픉"#, r#"픉"#),
        (r#"픊"#, r#"픊"#, r#"픊"#, r#"픊"#, r#"픊"#),
        (r#"픋"#, r#"픋"#, r#"픋"#, r#"픋"#, r#"픋"#),
        (r#"플"#, r#"플"#, r#"플"#, r#"플"#, r#"플"#),
        (r#"픍"#, r#"픍"#, r#"픍"#, r#"픍"#, r#"픍"#),
        (r#"픎"#, r#"픎"#, r#"픎"#, r#"픎"#, r#"픎"#),
        (r#"픏"#, r#"픏"#, r#"픏"#, r#"픏"#, r#"픏"#),
        (r#"픐"#, r#"픐"#, r#"픐"#, r#"픐"#, r#"픐"#),
        (r#"픑"#, r#"픑"#, r#"픑"#, r#"픑"#, r#"픑"#),
        (r#"픒"#, r#"픒"#, r#"픒"#, r#"픒"#, r#"픒"#),
        (r#"픓"#, r#"픓"#, r#"픓"#, r#"픓"#, r#"픓"#),
        (r#"픔"#, r#"픔"#, r#"픔"#, r#"픔"#, r#"픔"#),
        (r#"픕"#, r#"픕"#, r#"픕"#, r#"픕"#, r#"픕"#),
        (r#"픖"#, r#"픖"#, r#"픖"#, r#"픖"#, r#"픖"#),
        (r#"픗"#, r#"픗"#, r#"픗"#, r#"픗"#, r#"픗"#),
        (r#"픘"#, r#"픘"#, r#"픘"#, r#"픘"#, r#"픘"#),
        (r#"픙"#, r#"픙"#, r#"픙"#, r#"픙"#, r#"픙"#),
        (r#"픚"#, r#"픚"#, r#"픚"#, r#"픚"#, r#"픚"#),
        (r#"픛"#, r#"픛"#, r#"픛"#, r#"픛"#, r#"픛"#),
        (r#"픜"#, r#"픜"#, r#"픜"#, r#"픜"#, r#"픜"#),
        (r#"픝"#, r#"픝"#, r#"픝"#, r#"픝"#, r#"픝"#),
        (r#"픞"#, r#"픞"#, r#"픞"#, r#"픞"#, r#"픞"#),
        (r#"픟"#, r#"픟"#, r#"픟"#, r#"픟"#, r#"픟"#),
        (r#"픠"#, r#"픠"#, r#"픠"#, r#"픠"#, r#"픠"#),
        (r#"픡"#, r#"픡"#, r#"픡"#, r#"픡"#, r#"픡"#),
        (r#"픢"#, r#"픢"#, r#"픢"#, r#"픢"#, r#"픢"#),
        (r#"픣"#, r#"픣"#, r#"픣"#, r#"픣"#, r#"픣"#),
        (r#"픤"#, r#"픤"#, r#"픤"#, r#"픤"#, r#"픤"#),
        (r#"픥"#, r#"픥"#, r#"픥"#, r#"픥"#, r#"픥"#),
        (r#"픦"#, r#"픦"#, r#"픦"#, r#"픦"#, r#"픦"#),
        (r#"픧"#, r#"픧"#, r#"픧"#, r#"픧"#, r#"픧"#),
        (r#"픨"#, r#"픨"#, r#"픨"#, r#"픨"#, r#"픨"#),
        (r#"픩"#, r#"픩"#, r#"픩"#, r#"픩"#, r#"픩"#),
        (r#"픪"#, r#"픪"#, r#"픪"#, r#"픪"#, r#"픪"#),
        (r#"픫"#, r#"픫"#, r#"픫"#, r#"픫"#, r#"픫"#),
        (r#"픬"#, r#"픬"#, r#"픬"#, r#"픬"#, r#"픬"#),
        (r#"픭"#, r#"픭"#, r#"픭"#, r#"픭"#, r#"픭"#),
        (r#"픮"#, r#"픮"#, r#"픮"#, r#"픮"#, r#"픮"#),
        (r#"픯"#, r#"픯"#, r#"픯"#, r#"픯"#, r#"픯"#),
        (r#"픰"#, r#"픰"#, r#"픰"#, r#"픰"#, r#"픰"#),
        (r#"픱"#, r#"픱"#, r#"픱"#, r#"픱"#, r#"픱"#),
        (r#"픲"#, r#"픲"#, r#"픲"#, r#"픲"#, r#"픲"#),
        (r#"픳"#, r#"픳"#, r#"픳"#, r#"픳"#, r#"픳"#),
        (r#"픴"#, r#"픴"#, r#"픴"#, r#"픴"#, r#"픴"#),
        (r#"픵"#, r#"픵"#, r#"픵"#, r#"픵"#, r#"픵"#),
        (r#"픶"#, r#"픶"#, r#"픶"#, r#"픶"#, r#"픶"#),
        (r#"픷"#, r#"픷"#, r#"픷"#, r#"픷"#, r#"픷"#),
        (r#"픸"#, r#"픸"#, r#"픸"#, r#"픸"#, r#"픸"#),
        (r#"픹"#, r#"픹"#, r#"픹"#, r#"픹"#, r#"픹"#),
        (r#"픺"#, r#"픺"#, r#"픺"#, r#"픺"#, r#"픺"#),
        (r#"픻"#, r#"픻"#, r#"픻"#, r#"픻"#, r#"픻"#),
        (r#"피"#, r#"피"#, r#"피"#, r#"피"#, r#"피"#),
        (r#"픽"#, r#"픽"#, r#"픽"#, r#"픽"#, r#"픽"#),
        (r#"픾"#, r#"픾"#, r#"픾"#, r#"픾"#, r#"픾"#),
        (r#"픿"#, r#"픿"#, r#"픿"#, r#"픿"#, r#"픿"#),
        (r#"핀"#, r#"핀"#, r#"핀"#, r#"핀"#, r#"핀"#),
        (r#"핁"#, r#"핁"#, r#"핁"#, r#"핁"#, r#"핁"#),
        (r#"핂"#, r#"핂"#, r#"핂"#, r#"핂"#, r#"핂"#),
        (r#"핃"#, r#"핃"#, r#"핃"#, r#"핃"#, r#"핃"#),
        (r#"필"#, r#"필"#, r#"필"#, r#"필"#, r#"필"#),
        (r#"핅"#, r#"핅"#, r#"핅"#, r#"핅"#, r#"핅"#),
        (r#"핆"#, r#"핆"#, r#"핆"#, r#"핆"#, r#"핆"#),
        (r#"핇"#, r#"핇"#, r#"핇"#, r#"핇"#, r#"핇"#),
        (r#"핈"#, r#"핈"#, r#"핈"#, r#"핈"#, r#"핈"#),
        (r#"핉"#, r#"핉"#, r#"핉"#, r#"핉"#, r#"핉"#),
        (r#"핊"#, r#"핊"#, r#"핊"#, r#"핊"#, r#"핊"#),
        (r#"핋"#, r#"핋"#, r#"핋"#, r#"핋"#, r#"핋"#),
        (r#"핌"#, r#"핌"#, r#"핌"#, r#"핌"#, r#"핌"#),
        (r#"핍"#, r#"핍"#, r#"핍"#, r#"핍"#, r#"핍"#),
        (r#"핎"#, r#"핎"#, r#"핎"#, r#"핎"#, r#"핎"#),
        (r#"핏"#, r#"핏"#, r#"핏"#, r#"핏"#, r#"핏"#),
        (r#"핐"#, r#"핐"#, r#"핐"#, r#"핐"#, r#"핐"#),
        (r#"핑"#, r#"핑"#, r#"핑"#, r#"핑"#, r#"핑"#),
        (r#"핒"#, r#"핒"#, r#"핒"#, r#"핒"#, r#"핒"#),
        (r#"핓"#, r#"핓"#, r#"핓"#, r#"핓"#, r#"핓"#),
        (r#"핔"#, r#"핔"#, r#"핔"#, r#"핔"#, r#"핔"#),
        (r#"핕"#, r#"핕"#, r#"핕"#, r#"핕"#, r#"핕"#),
        (r#"핖"#, r#"핖"#, r#"핖"#, r#"핖"#, r#"핖"#),
        (r#"핗"#, r#"핗"#, r#"핗"#, r#"핗"#, r#"핗"#),
        (r#"하"#, r#"하"#, r#"하"#, r#"하"#, r#"하"#),
        (r#"학"#, r#"학"#, r#"학"#, r#"학"#, r#"학"#),
        (r#"핚"#, r#"핚"#, r#"핚"#, r#"핚"#, r#"핚"#),
        (r#"핛"#, r#"핛"#, r#"핛"#, r#"핛"#, r#"핛"#),
        (r#"한"#, r#"한"#, r#"한"#, r#"한"#, r#"한"#),
        (r#"핝"#, r#"핝"#, r#"핝"#, r#"핝"#, r#"핝"#),
        (r#"핞"#, r#"핞"#, r#"핞"#, r#"핞"#, r#"핞"#),
        (r#"핟"#, r#"핟"#, r#"핟"#, r#"핟"#, r#"핟"#),
        (r#"할"#, r#"할"#, r#"할"#, r#"할"#, r#"할"#),
        (r#"핡"#, r#"핡"#, r#"핡"#, r#"핡"#, r#"핡"#),
        (r#"핢"#, r#"핢"#, r#"핢"#, r#"핢"#, r#"핢"#),
        (r#"핣"#, r#"핣"#, r#"핣"#, r#"핣"#, r#"핣"#),
        (r#"핤"#, r#"핤"#, r#"핤"#, r#"핤"#, r#"핤"#),
        (r#"핥"#, r#"핥"#, r#"핥"#, r#"핥"#, r#"핥"#),
        (r#"핦"#, r#"핦"#, r#"핦"#, r#"핦"#, r#"핦"#),
        (r#"핧"#, r#"핧"#, r#"핧"#, r#"핧"#, r#"핧"#),
        (r#"함"#, r#"함"#, r#"함"#, r#"함"#, r#"함"#),
        (r#"합"#, r#"합"#, r#"합"#, r#"합"#, r#"합"#),
        (r#"핪"#, r#"핪"#, r#"핪"#, r#"핪"#, r#"핪"#),
        (r#"핫"#, r#"핫"#, r#"핫"#, r#"핫"#, r#"핫"#),
        (r#"핬"#, r#"핬"#, r#"핬"#, r#"핬"#, r#"핬"#),
        (r#"항"#, r#"항"#, r#"항"#, r#"항"#, r#"항"#),
        (r#"핮"#, r#"핮"#, r#"핮"#, r#"핮"#, r#"핮"#),
        (r#"핯"#, r#"핯"#, r#"핯"#, r#"핯"#, r#"핯"#),
        (r#"핰"#, r#"핰"#, r#"핰"#, r#"핰"#, r#"핰"#),
        (r#"핱"#, r#"핱"#, r#"핱"#, r#"핱"#, r#"핱"#),
        (r#"핲"#, r#"핲"#, r#"핲"#, r#"핲"#, r#"핲"#),
        (r#"핳"#, r#"핳"#, r#"핳"#, r#"핳"#, r#"핳"#),
        (r#"해"#, r#"해"#, r#"해"#, r#"해"#, r#"해"#),
        (r#"핵"#, r#"핵"#, r#"핵"#, r#"핵"#, r#"핵"#),
        (r#"핶"#, r#"핶"#, r#"핶"#, r#"핶"#, r#"핶"#),
        (r#"핷"#, r#"핷"#, r#"핷"#, r#"핷"#, r#"핷"#),
        (r#"핸"#, r#"핸"#, r#"핸"#, r#"핸"#, r#"핸"#),
        (r#"핹"#, r#"핹"#, r#"핹"#, r#"핹"#, r#"핹"#),
        (r#"핺"#, r#"핺"#, r#"핺"#, r#"핺"#, r#"핺"#),
        (r#"핻"#, r#"핻"#, r#"핻"#, r#"핻"#, r#"핻"#),
        (r#"핼"#, r#"핼"#, r#"핼"#, r#"핼"#, r#"핼"#),
        (r#"핽"#, r#"핽"#, r#"핽"#, r#"핽"#, r#"핽"#),
        (r#"핾"#, r#"핾"#, r#"핾"#, r#"핾"#, r#"핾"#),
        (r#"핿"#, r#"핿"#, r#"핿"#, r#"핿"#, r#"핿"#),
        (r#"햀"#, r#"햀"#, r#"햀"#, r#"햀"#, r#"햀"#),
        (r#"햁"#, r#"햁"#, r#"햁"#, r#"햁"#, r#"햁"#),
        (r#"햂"#, r#"햂"#, r#"햂"#, r#"햂"#, r#"햂"#),
        (r#"햃"#, r#"햃"#, r#"햃"#, r#"햃"#, r#"햃"#),
        (r#"햄"#, r#"햄"#, r#"햄"#, r#"햄"#, r#"햄"#),
        (r#"햅"#, r#"햅"#, r#"햅"#, r#"햅"#, r#"햅"#),
        (r#"햆"#, r#"햆"#, r#"햆"#, r#"햆"#, r#"햆"#),
        (r#"햇"#, r#"햇"#, r#"햇"#, r#"햇"#, r#"햇"#),
        (r#"했"#, r#"했"#, r#"했"#, r#"했"#, r#"했"#),
        (r#"행"#, r#"행"#, r#"행"#, r#"행"#, r#"행"#),
        (r#"햊"#, r#"햊"#, r#"햊"#, r#"햊"#, r#"햊"#),
        (r#"햋"#, r#"햋"#, r#"햋"#, r#"햋"#, r#"햋"#),
        (r#"햌"#, r#"햌"#, r#"햌"#, r#"햌"#, r#"햌"#),
        (r#"햍"#, r#"햍"#, r#"햍"#, r#"햍"#, r#"햍"#),
        (r#"햎"#, r#"햎"#, r#"햎"#, r#"햎"#, r#"햎"#),
        (r#"햏"#, r#"햏"#, r#"햏"#, r#"햏"#, r#"햏"#),
        (r#"햐"#, r#"햐"#, r#"햐"#, r#"햐"#, r#"햐"#),
        (r#"햑"#, r#"햑"#, r#"햑"#, r#"햑"#, r#"햑"#),
        (r#"햒"#, r#"햒"#, r#"햒"#, r#"햒"#, r#"햒"#),
        (r#"햓"#, r#"햓"#, r#"햓"#, r#"햓"#, r#"햓"#),
        (r#"햔"#, r#"햔"#, r#"햔"#, r#"햔"#, r#"햔"#),
        (r#"햕"#, r#"햕"#, r#"햕"#, r#"햕"#, r#"햕"#),
        (r#"햖"#, r#"햖"#, r#"햖"#, r#"햖"#, r#"햖"#),
        (r#"햗"#, r#"햗"#, r#"햗"#, r#"햗"#, r#"햗"#),
        (r#"햘"#, r#"햘"#, r#"햘"#, r#"햘"#, r#"햘"#),
        (r#"햙"#, r#"햙"#, r#"햙"#, r#"햙"#, r#"햙"#),
        (r#"햚"#, r#"햚"#, r#"햚"#, r#"햚"#, r#"햚"#),
        (r#"햛"#, r#"햛"#, r#"햛"#, r#"햛"#, r#"햛"#),
        (r#"햜"#, r#"햜"#, r#"햜"#, r#"햜"#, r#"햜"#),
        (r#"햝"#, r#"햝"#, r#"햝"#, r#"햝"#, r#"햝"#),
        (r#"햞"#, r#"햞"#, r#"햞"#, r#"햞"#, r#"햞"#),
        (r#"햟"#, r#"햟"#, r#"햟"#, r#"햟"#, r#"햟"#),
        (r#"햠"#, r#"햠"#, r#"햠"#, r#"햠"#, r#"햠"#),
        (r#"햡"#, r#"햡"#, r#"햡"#, r#"햡"#, r#"햡"#),
        (r#"햢"#, r#"햢"#, r#"햢"#, r#"햢"#, r#"햢"#),
        (r#"햣"#, r#"햣"#, r#"햣"#, r#"햣"#, r#"햣"#),
        (r#"햤"#, r#"햤"#, r#"햤"#, r#"햤"#, r#"햤"#),
        (r#"향"#, r#"향"#, r#"향"#, r#"향"#, r#"향"#),
        (r#"햦"#, r#"햦"#, r#"햦"#, r#"햦"#, r#"햦"#),
        (r#"햧"#, r#"햧"#, r#"햧"#, r#"햧"#, r#"햧"#),
        (r#"햨"#, r#"햨"#, r#"햨"#, r#"햨"#, r#"햨"#),
        (r#"햩"#, r#"햩"#, r#"햩"#, r#"햩"#, r#"햩"#),
        (r#"햪"#, r#"햪"#, r#"햪"#, r#"햪"#, r#"햪"#),
        (r#"햫"#, r#"햫"#, r#"햫"#, r#"햫"#, r#"햫"#),
        (r#"햬"#, r#"햬"#, r#"햬"#, r#"햬"#, r#"햬"#),
        (r#"햭"#, r#"햭"#, r#"햭"#, r#"햭"#, r#"햭"#),
        (r#"햮"#, r#"햮"#, r#"햮"#, r#"햮"#, r#"햮"#),
        (r#"햯"#, r#"햯"#, r#"햯"#, r#"햯"#, r#"햯"#),
        (r#"햰"#, r#"햰"#, r#"햰"#, r#"햰"#, r#"햰"#),
        (r#"햱"#, r#"햱"#, r#"햱"#, r#"햱"#, r#"햱"#),
        (r#"햲"#, r#"햲"#, r#"햲"#, r#"햲"#, r#"햲"#),
        (r#"햳"#, r#"햳"#, r#"햳"#, r#"햳"#, r#"햳"#),
        (r#"햴"#, r#"햴"#, r#"햴"#, r#"햴"#, r#"햴"#),
        (r#"햵"#, r#"햵"#, r#"햵"#, r#"햵"#, r#"햵"#),
        (r#"햶"#, r#"햶"#, r#"햶"#, r#"햶"#, r#"햶"#),
        (r#"햷"#, r#"햷"#, r#"햷"#, r#"햷"#, r#"햷"#),
        (r#"햸"#, r#"햸"#, r#"햸"#, r#"햸"#, r#"햸"#),
        (r#"햹"#, r#"햹"#, r#"햹"#, r#"햹"#, r#"햹"#),
        (r#"햺"#, r#"햺"#, r#"햺"#, r#"햺"#, r#"햺"#),
        (r#"햻"#, r#"햻"#, r#"햻"#, r#"햻"#, r#"햻"#),
        (r#"햼"#, r#"햼"#, r#"햼"#, r#"햼"#, r#"햼"#),
        (r#"햽"#, r#"햽"#, r#"햽"#, r#"햽"#, r#"햽"#),
        (r#"햾"#, r#"햾"#, r#"햾"#, r#"햾"#, r#"햾"#),
        (r#"햿"#, r#"햿"#, r#"햿"#, r#"햿"#, r#"햿"#),
        (r#"헀"#, r#"헀"#, r#"헀"#, r#"헀"#, r#"헀"#),
        (r#"헁"#, r#"헁"#, r#"헁"#, r#"헁"#, r#"헁"#),
        (r#"헂"#, r#"헂"#, r#"헂"#, r#"헂"#, r#"헂"#),
        (r#"헃"#, r#"헃"#, r#"헃"#, r#"헃"#, r#"헃"#),
        (r#"헄"#, r#"헄"#, r#"헄"#, r#"헄"#, r#"헄"#),
        (r#"헅"#, r#"헅"#, r#"헅"#, r#"헅"#, r#"헅"#),
        (r#"헆"#, r#"헆"#, r#"헆"#, r#"헆"#, r#"헆"#),
        (r#"헇"#, r#"헇"#, r#"헇"#, r#"헇"#, r#"헇"#),
        (r#"허"#, r#"허"#, r#"허"#, r#"허"#, r#"허"#),
        (r#"헉"#, r#"헉"#, r#"헉"#, r#"헉"#, r#"헉"#),
        (r#"헊"#, r#"헊"#, r#"헊"#, r#"헊"#, r#"헊"#),
        (r#"헋"#, r#"헋"#, r#"헋"#, r#"헋"#, r#"헋"#),
        (r#"헌"#, r#"헌"#, r#"헌"#, r#"헌"#, r#"헌"#),
        (r#"헍"#, r#"헍"#, r#"헍"#, r#"헍"#, r#"헍"#),
        (r#"헎"#, r#"헎"#, r#"헎"#, r#"헎"#, r#"헎"#),
        (r#"헏"#, r#"헏"#, r#"헏"#, r#"헏"#, r#"헏"#),
        (r#"헐"#, r#"헐"#, r#"헐"#, r#"헐"#, r#"헐"#),
        (r#"헑"#, r#"헑"#, r#"헑"#, r#"헑"#, r#"헑"#),
        (r#"헒"#, r#"헒"#, r#"헒"#, r#"헒"#, r#"헒"#),
        (r#"헓"#, r#"헓"#, r#"헓"#, r#"헓"#, r#"헓"#),
        (r#"헔"#, r#"헔"#, r#"헔"#, r#"헔"#, r#"헔"#),
        (r#"헕"#, r#"헕"#, r#"헕"#, r#"헕"#, r#"헕"#),
        (r#"헖"#, r#"헖"#, r#"헖"#, r#"헖"#, r#"헖"#),
        (r#"헗"#, r#"헗"#, r#"헗"#, r#"헗"#, r#"헗"#),
        (r#"험"#, r#"험"#, r#"험"#, r#"험"#, r#"험"#),
        (r#"헙"#, r#"헙"#, r#"헙"#, r#"헙"#, r#"헙"#),
        (r#"헚"#, r#"헚"#, r#"헚"#, r#"헚"#, r#"헚"#),
        (r#"헛"#, r#"헛"#, r#"헛"#, r#"헛"#, r#"헛"#),
        (r#"헜"#, r#"헜"#, r#"헜"#, r#"헜"#, r#"헜"#),
        (r#"헝"#, r#"헝"#, r#"헝"#, r#"헝"#, r#"헝"#),
        (r#"헞"#, r#"헞"#, r#"헞"#, r#"헞"#, r#"헞"#),
        (r#"헟"#, r#"헟"#, r#"헟"#, r#"헟"#, r#"헟"#),
        (r#"헠"#, r#"헠"#, r#"헠"#, r#"헠"#, r#"헠"#),
        (r#"헡"#, r#"헡"#, r#"헡"#, r#"헡"#, r#"헡"#),
        (r#"헢"#, r#"헢"#, r#"헢"#, r#"헢"#, r#"헢"#),
        (r#"헣"#, r#"헣"#, r#"헣"#, r#"헣"#, r#"헣"#),
        (r#"헤"#, r#"헤"#, r#"헤"#, r#"헤"#, r#"헤"#),
        (r#"헥"#, r#"헥"#, r#"헥"#, r#"헥"#, r#"헥"#),
        (r#"헦"#, r#"헦"#, r#"헦"#, r#"헦"#, r#"헦"#),
        (r#"헧"#, r#"헧"#, r#"헧"#, r#"헧"#, r#"헧"#),
        (r#"헨"#, r#"헨"#, r#"헨"#, r#"헨"#, r#"헨"#),
        (r#"헩"#, r#"헩"#, r#"헩"#, r#"헩"#, r#"헩"#),
        (r#"헪"#, r#"헪"#, r#"헪"#, r#"헪"#, r#"헪"#),
        (r#"헫"#, r#"헫"#, r#"헫"#, r#"헫"#, r#"헫"#),
        (r#"헬"#, r#"헬"#, r#"헬"#, r#"헬"#, r#"헬"#),
        (r#"헭"#, r#"헭"#, r#"헭"#, r#"헭"#, r#"헭"#),
        (r#"헮"#, r#"헮"#, r#"헮"#, r#"헮"#, r#"헮"#),
        (r#"헯"#, r#"헯"#, r#"헯"#, r#"헯"#, r#"헯"#),
        (r#"헰"#, r#"헰"#, r#"헰"#, r#"헰"#, r#"헰"#),
        (r#"헱"#, r#"헱"#, r#"헱"#, r#"헱"#, r#"헱"#),
        (r#"헲"#, r#"헲"#, r#"헲"#, r#"헲"#, r#"헲"#),
        (r#"헳"#, r#"헳"#, r#"헳"#, r#"헳"#, r#"헳"#),
        (r#"헴"#, r#"헴"#, r#"헴"#, r#"헴"#, r#"헴"#),
        (r#"헵"#, r#"헵"#, r#"헵"#, r#"헵"#, r#"헵"#),
        (r#"헶"#, r#"헶"#, r#"헶"#, r#"헶"#, r#"헶"#),
        (r#"헷"#, r#"헷"#, r#"헷"#, r#"헷"#, r#"헷"#),
        (r#"헸"#, r#"헸"#, r#"헸"#, r#"헸"#, r#"헸"#),
        (r#"헹"#, r#"헹"#, r#"헹"#, r#"헹"#, r#"헹"#),
        (r#"헺"#, r#"헺"#, r#"헺"#, r#"헺"#, r#"헺"#),
        (r#"헻"#, r#"헻"#, r#"헻"#, r#"헻"#, r#"헻"#),
        (r#"헼"#, r#"헼"#, r#"헼"#, r#"헼"#, r#"헼"#),
        (r#"헽"#, r#"헽"#, r#"헽"#, r#"헽"#, r#"헽"#),
        (r#"헾"#, r#"헾"#, r#"헾"#, r#"헾"#, r#"헾"#),
        (r#"헿"#, r#"헿"#, r#"헿"#, r#"헿"#, r#"헿"#),
        (r#"혀"#, r#"혀"#, r#"혀"#, r#"혀"#, r#"혀"#),
        (r#"혁"#, r#"혁"#, r#"혁"#, r#"혁"#, r#"혁"#),
        (r#"혂"#, r#"혂"#, r#"혂"#, r#"혂"#, r#"혂"#),
        (r#"혃"#, r#"혃"#, r#"혃"#, r#"혃"#, r#"혃"#),
        (r#"현"#, r#"현"#, r#"현"#, r#"현"#, r#"현"#),
        (r#"혅"#, r#"혅"#, r#"혅"#, r#"혅"#, r#"혅"#),
        (r#"혆"#, r#"혆"#, r#"혆"#, r#"혆"#, r#"혆"#),
        (r#"혇"#, r#"혇"#, r#"혇"#, r#"혇"#, r#"혇"#),
        (r#"혈"#, r#"혈"#, r#"혈"#, r#"혈"#, r#"혈"#),
        (r#"혉"#, r#"혉"#, r#"혉"#, r#"혉"#, r#"혉"#),
        (r#"혊"#, r#"혊"#, r#"혊"#, r#"혊"#, r#"혊"#),
        (r#"혋"#, r#"혋"#, r#"혋"#, r#"혋"#, r#"혋"#),
        (r#"혌"#, r#"혌"#, r#"혌"#, r#"혌"#, r#"혌"#),
        (r#"혍"#, r#"혍"#, r#"혍"#, r#"혍"#, r#"혍"#),
        (r#"혎"#, r#"혎"#, r#"혎"#, r#"혎"#, r#"혎"#),
        (r#"혏"#, r#"혏"#, r#"혏"#, r#"혏"#, r#"혏"#),
        (r#"혐"#, r#"혐"#, r#"혐"#, r#"혐"#, r#"혐"#),
        (r#"협"#, r#"협"#, r#"협"#, r#"협"#, r#"협"#),
        (r#"혒"#, r#"혒"#, r#"혒"#, r#"혒"#, r#"혒"#),
        (r#"혓"#, r#"혓"#, r#"혓"#, r#"혓"#, r#"혓"#),
        (r#"혔"#, r#"혔"#, r#"혔"#, r#"혔"#, r#"혔"#),
        (r#"형"#, r#"형"#, r#"형"#, r#"형"#, r#"형"#),
        (r#"혖"#, r#"혖"#, r#"혖"#, r#"혖"#, r#"혖"#),
        (r#"혗"#, r#"혗"#, r#"혗"#, r#"혗"#, r#"혗"#),
        (r#"혘"#, r#"혘"#, r#"혘"#, r#"혘"#, r#"혘"#),
        (r#"혙"#, r#"혙"#, r#"혙"#, r#"혙"#, r#"혙"#),
        (r#"혚"#, r#"혚"#, r#"혚"#, r#"혚"#, r#"혚"#),
        (r#"혛"#, r#"혛"#, r#"혛"#, r#"혛"#, r#"혛"#),
        (r#"혜"#, r#"혜"#, r#"혜"#, r#"혜"#, r#"혜"#),
        (r#"혝"#, r#"혝"#, r#"혝"#, r#"혝"#, r#"혝"#),
        (r#"혞"#, r#"혞"#, r#"혞"#, r#"혞"#, r#"혞"#),
        (r#"혟"#, r#"혟"#, r#"혟"#, r#"혟"#, r#"혟"#),
        (r#"혠"#, r#"혠"#, r#"혠"#, r#"혠"#, r#"혠"#),
        (r#"혡"#, r#"혡"#, r#"혡"#, r#"혡"#, r#"혡"#),
        (r#"혢"#, r#"혢"#, r#"혢"#, r#"혢"#, r#"혢"#),
        (r#"혣"#, r#"혣"#, r#"혣"#, r#"혣"#, r#"혣"#),
        (r#"혤"#, r#"혤"#, r#"혤"#, r#"혤"#, r#"혤"#),
        (r#"혥"#, r#"혥"#, r#"혥"#, r#"혥"#, r#"혥"#),
        (r#"혦"#, r#"혦"#, r#"혦"#, r#"혦"#, r#"혦"#),
        (r#"혧"#, r#"혧"#, r#"혧"#, r#"혧"#, r#"혧"#),
        (r#"혨"#, r#"혨"#, r#"혨"#, r#"혨"#, r#"혨"#),
        (r#"혩"#, r#"혩"#, r#"혩"#, r#"혩"#, r#"혩"#),
        (r#"혪"#, r#"혪"#, r#"혪"#, r#"혪"#, r#"혪"#),
        (r#"혫"#, r#"혫"#, r#"혫"#, r#"혫"#, r#"혫"#),
        (r#"혬"#, r#"혬"#, r#"혬"#, r#"혬"#, r#"혬"#),
        (r#"혭"#, r#"혭"#, r#"혭"#, r#"혭"#, r#"혭"#),
        (r#"혮"#, r#"혮"#, r#"혮"#, r#"혮"#, r#"혮"#),
        (r#"혯"#, r#"혯"#, r#"혯"#, r#"혯"#, r#"혯"#),
        (r#"혰"#, r#"혰"#, r#"혰"#, r#"혰"#, r#"혰"#),
        (r#"혱"#, r#"혱"#, r#"혱"#, r#"혱"#, r#"혱"#),
        (r#"혲"#, r#"혲"#, r#"혲"#, r#"혲"#, r#"혲"#),
        (r#"혳"#, r#"혳"#, r#"혳"#, r#"혳"#, r#"혳"#),
        (r#"혴"#, r#"혴"#, r#"혴"#, r#"혴"#, r#"혴"#),
        (r#"혵"#, r#"혵"#, r#"혵"#, r#"혵"#, r#"혵"#),
        (r#"혶"#, r#"혶"#, r#"혶"#, r#"혶"#, r#"혶"#),
        (r#"혷"#, r#"혷"#, r#"혷"#, r#"혷"#, r#"혷"#),
        (r#"호"#, r#"호"#, r#"호"#, r#"호"#, r#"호"#),
        (r#"혹"#, r#"혹"#, r#"혹"#, r#"혹"#, r#"혹"#),
        (r#"혺"#, r#"혺"#, r#"혺"#, r#"혺"#, r#"혺"#),
        (r#"혻"#, r#"혻"#, r#"혻"#, r#"혻"#, r#"혻"#),
        (r#"혼"#, r#"혼"#, r#"혼"#, r#"혼"#, r#"혼"#),
        (r#"혽"#, r#"혽"#, r#"혽"#, r#"혽"#, r#"혽"#),
        (r#"혾"#, r#"혾"#, r#"혾"#, r#"혾"#, r#"혾"#),
        (r#"혿"#, r#"혿"#, r#"혿"#, r#"혿"#, r#"혿"#),
        (r#"홀"#, r#"홀"#, r#"홀"#, r#"홀"#, r#"홀"#),
        (r#"홁"#, r#"홁"#, r#"홁"#, r#"홁"#, r#"홁"#),
        (r#"홂"#, r#"홂"#, r#"홂"#, r#"홂"#, r#"홂"#),
        (r#"홃"#, r#"홃"#, r#"홃"#, r#"홃"#, r#"홃"#),
        (r#"홄"#, r#"홄"#, r#"홄"#, r#"홄"#, r#"홄"#),
        (r#"홅"#, r#"홅"#, r#"홅"#, r#"홅"#, r#"홅"#),
        (r#"홆"#, r#"홆"#, r#"홆"#, r#"홆"#, r#"홆"#),
        (r#"홇"#, r#"홇"#, r#"홇"#, r#"홇"#, r#"홇"#),
        (r#"홈"#, r#"홈"#, r#"홈"#, r#"홈"#, r#"홈"#),
        (r#"홉"#, r#"홉"#, r#"홉"#, r#"홉"#, r#"홉"#),
        (r#"홊"#, r#"홊"#, r#"홊"#, r#"홊"#, r#"홊"#),
        (r#"홋"#, r#"홋"#, r#"홋"#, r#"홋"#, r#"홋"#),
        (r#"홌"#, r#"홌"#, r#"홌"#, r#"홌"#, r#"홌"#),
        (r#"홍"#, r#"홍"#, r#"홍"#, r#"홍"#, r#"홍"#),
        (r#"홎"#, r#"홎"#, r#"홎"#, r#"홎"#, r#"홎"#),
        (r#"홏"#, r#"홏"#, r#"홏"#, r#"홏"#, r#"홏"#),
        (r#"홐"#, r#"홐"#, r#"홐"#, r#"홐"#, r#"홐"#),
        (r#"홑"#, r#"홑"#, r#"홑"#, r#"홑"#, r#"홑"#),
        (r#"홒"#, r#"홒"#, r#"홒"#, r#"홒"#, r#"홒"#),
        (r#"홓"#, r#"홓"#, r#"홓"#, r#"홓"#, r#"홓"#),
        (r#"화"#, r#"화"#, r#"화"#, r#"화"#, r#"화"#),
        (r#"확"#, r#"확"#, r#"확"#, r#"확"#, r#"확"#),
        (r#"홖"#, r#"홖"#, r#"홖"#, r#"홖"#, r#"홖"#),
        (r#"홗"#, r#"홗"#, r#"홗"#, r#"홗"#, r#"홗"#),
        (r#"환"#, r#"환"#, r#"환"#, r#"환"#, r#"환"#),
        (r#"홙"#, r#"홙"#, r#"홙"#, r#"홙"#, r#"홙"#),
        (r#"홚"#, r#"홚"#, r#"홚"#, r#"홚"#, r#"홚"#),
        (r#"홛"#, r#"홛"#, r#"홛"#, r#"홛"#, r#"홛"#),
        (r#"활"#, r#"활"#, r#"활"#, r#"활"#, r#"활"#),
        (r#"홝"#, r#"홝"#, r#"홝"#, r#"홝"#, r#"홝"#),
        (r#"홞"#, r#"홞"#, r#"홞"#, r#"홞"#, r#"홞"#),
        (r#"홟"#, r#"홟"#, r#"홟"#, r#"홟"#, r#"홟"#),
        (r#"홠"#, r#"홠"#, r#"홠"#, r#"홠"#, r#"홠"#),
        (r#"홡"#, r#"홡"#, r#"홡"#, r#"홡"#, r#"홡"#),
        (r#"홢"#, r#"홢"#, r#"홢"#, r#"홢"#, r#"홢"#),
        (r#"홣"#, r#"홣"#, r#"홣"#, r#"홣"#, r#"홣"#),
        (r#"홤"#, r#"홤"#, r#"홤"#, r#"홤"#, r#"홤"#),
        (r#"홥"#, r#"홥"#, r#"홥"#, r#"홥"#, r#"홥"#),
        (r#"홦"#, r#"홦"#, r#"홦"#, r#"홦"#, r#"홦"#),
        (r#"홧"#, r#"홧"#, r#"홧"#, r#"홧"#, r#"홧"#),
        (r#"홨"#, r#"홨"#, r#"홨"#, r#"홨"#, r#"홨"#),
        (r#"황"#, r#"황"#, r#"황"#, r#"황"#, r#"황"#),
        (r#"홪"#, r#"홪"#, r#"홪"#, r#"홪"#, r#"홪"#),
        (r#"홫"#, r#"홫"#, r#"홫"#, r#"홫"#, r#"홫"#),
        (r#"홬"#, r#"홬"#, r#"홬"#, r#"홬"#, r#"홬"#),
        (r#"홭"#, r#"홭"#, r#"홭"#, r#"홭"#, r#"홭"#),
        (r#"홮"#, r#"홮"#, r#"홮"#, r#"홮"#, r#"홮"#),
        (r#"홯"#, r#"홯"#, r#"홯"#, r#"홯"#, r#"홯"#),
        (r#"홰"#, r#"홰"#, r#"홰"#, r#"홰"#, r#"홰"#),
        (r#"홱"#, r#"홱"#, r#"홱"#, r#"홱"#, r#"홱"#),
        (r#"홲"#, r#"홲"#, r#"홲"#, r#"홲"#, r#"홲"#),
        (r#"홳"#, r#"홳"#, r#"홳"#, r#"홳"#, r#"홳"#),
        (r#"홴"#, r#"홴"#, r#"홴"#, r#"홴"#, r#"홴"#),
        (r#"홵"#, r#"홵"#, r#"홵"#, r#"홵"#, r#"홵"#),
        (r#"홶"#, r#"홶"#, r#"홶"#, r#"홶"#, r#"홶"#),
        (r#"홷"#, r#"홷"#, r#"홷"#, r#"홷"#, r#"홷"#),
        (r#"홸"#, r#"홸"#, r#"홸"#, r#"홸"#, r#"홸"#),
        (r#"홹"#, r#"홹"#, r#"홹"#, r#"홹"#, r#"홹"#),
        (r#"홺"#, r#"홺"#, r#"홺"#, r#"홺"#, r#"홺"#),
        (r#"홻"#, r#"홻"#, r#"홻"#, r#"홻"#, r#"홻"#),
        (r#"홼"#, r#"홼"#, r#"홼"#, r#"홼"#, r#"홼"#),
        (r#"홽"#, r#"홽"#, r#"홽"#, r#"홽"#, r#"홽"#),
        (r#"홾"#, r#"홾"#, r#"홾"#, r#"홾"#, r#"홾"#),
        (r#"홿"#, r#"홿"#, r#"홿"#, r#"홿"#, r#"홿"#),
        (r#"횀"#, r#"횀"#, r#"횀"#, r#"횀"#, r#"횀"#),
        (r#"횁"#, r#"횁"#, r#"횁"#, r#"횁"#, r#"횁"#),
        (r#"횂"#, r#"횂"#, r#"횂"#, r#"횂"#, r#"횂"#),
        (r#"횃"#, r#"횃"#, r#"횃"#, r#"횃"#, r#"횃"#),
        (r#"횄"#, r#"횄"#, r#"횄"#, r#"횄"#, r#"횄"#),
        (r#"횅"#, r#"횅"#, r#"횅"#, r#"횅"#, r#"횅"#),
        (r#"횆"#, r#"횆"#, r#"횆"#, r#"횆"#, r#"횆"#),
        (r#"횇"#, r#"횇"#, r#"횇"#, r#"횇"#, r#"횇"#),
        (r#"횈"#, r#"횈"#, r#"횈"#, r#"횈"#, r#"횈"#),
        (r#"횉"#, r#"횉"#, r#"횉"#, r#"횉"#, r#"횉"#),
        (r#"횊"#, r#"횊"#, r#"횊"#, r#"횊"#, r#"횊"#),
        (r#"횋"#, r#"횋"#, r#"횋"#, r#"횋"#, r#"횋"#),
        (r#"회"#, r#"회"#, r#"회"#, r#"회"#, r#"회"#),
        (r#"획"#, r#"획"#, r#"획"#, r#"획"#, r#"획"#),
        (r#"횎"#, r#"횎"#, r#"횎"#, r#"횎"#, r#"횎"#),
        (r#"횏"#, r#"횏"#, r#"횏"#, r#"횏"#, r#"횏"#),
        (r#"횐"#, r#"횐"#, r#"횐"#, r#"횐"#, r#"횐"#),
        (r#"횑"#, r#"횑"#, r#"횑"#, r#"횑"#, r#"횑"#),
        (r#"횒"#, r#"횒"#, r#"횒"#, r#"횒"#, r#"횒"#),
        (r#"횓"#, r#"횓"#, r#"횓"#, r#"횓"#, r#"횓"#),
        (r#"횔"#, r#"횔"#, r#"횔"#, r#"횔"#, r#"횔"#),
        (r#"횕"#, r#"횕"#, r#"횕"#, r#"횕"#, r#"횕"#),
        (r#"횖"#, r#"횖"#, r#"횖"#, r#"횖"#, r#"횖"#),
        (r#"횗"#, r#"횗"#, r#"횗"#, r#"횗"#, r#"횗"#),
        (r#"횘"#, r#"횘"#, r#"횘"#, r#"횘"#, r#"횘"#),
        (r#"횙"#, r#"횙"#, r#"횙"#, r#"횙"#, r#"횙"#),
        (r#"횚"#, r#"횚"#, r#"횚"#, r#"횚"#, r#"횚"#),
        (r#"횛"#, r#"횛"#, r#"횛"#, r#"횛"#, r#"횛"#),
        (r#"횜"#, r#"횜"#, r#"횜"#, r#"횜"#, r#"횜"#),
        (r#"횝"#, r#"횝"#, r#"횝"#, r#"횝"#, r#"횝"#),
        (r#"횞"#, r#"횞"#, r#"횞"#, r#"횞"#, r#"횞"#),
        (r#"횟"#, r#"횟"#, r#"횟"#, r#"횟"#, r#"횟"#),
        (r#"횠"#, r#"횠"#, r#"횠"#, r#"횠"#, r#"횠"#),
        (r#"횡"#, r#"횡"#, r#"횡"#, r#"횡"#, r#"횡"#),
        (r#"횢"#, r#"횢"#, r#"횢"#, r#"횢"#, r#"횢"#),
        (r#"횣"#, r#"횣"#, r#"횣"#, r#"횣"#, r#"횣"#),
        (r#"횤"#, r#"횤"#, r#"횤"#, r#"횤"#, r#"횤"#),
        (r#"횥"#, r#"횥"#, r#"횥"#, r#"횥"#, r#"횥"#),
        (r#"횦"#, r#"횦"#, r#"횦"#, r#"횦"#, r#"횦"#),
        (r#"횧"#, r#"횧"#, r#"횧"#, r#"횧"#, r#"횧"#),
        (r#"효"#, r#"효"#, r#"효"#, r#"효"#, r#"효"#),
        (r#"횩"#, r#"횩"#, r#"횩"#, r#"횩"#, r#"횩"#),
        (r#"횪"#, r#"횪"#, r#"횪"#, r#"횪"#, r#"횪"#),
        (r#"횫"#, r#"횫"#, r#"횫"#, r#"횫"#, r#"횫"#),
        (r#"횬"#, r#"횬"#, r#"횬"#, r#"횬"#, r#"횬"#),
        (r#"횭"#, r#"횭"#, r#"횭"#, r#"횭"#, r#"횭"#),
        (r#"횮"#, r#"횮"#, r#"횮"#, r#"횮"#, r#"횮"#),
        (r#"횯"#, r#"횯"#, r#"횯"#, r#"횯"#, r#"횯"#),
        (r#"횰"#, r#"횰"#, r#"횰"#, r#"횰"#, r#"횰"#),
        (r#"횱"#, r#"횱"#, r#"횱"#, r#"횱"#, r#"횱"#),
        (r#"횲"#, r#"횲"#, r#"횲"#, r#"횲"#, r#"횲"#),
        (r#"횳"#, r#"횳"#, r#"횳"#, r#"횳"#, r#"횳"#),
        (r#"횴"#, r#"횴"#, r#"횴"#, r#"횴"#, r#"횴"#),
        (r#"횵"#, r#"횵"#, r#"횵"#, r#"횵"#, r#"횵"#),
        (r#"횶"#, r#"횶"#, r#"횶"#, r#"횶"#, r#"횶"#),
        (r#"횷"#, r#"횷"#, r#"횷"#, r#"횷"#, r#"횷"#),
        (r#"횸"#, r#"횸"#, r#"횸"#, r#"횸"#, r#"횸"#),
        (r#"횹"#, r#"횹"#, r#"횹"#, r#"횹"#, r#"횹"#),
        (r#"횺"#, r#"횺"#, r#"횺"#, r#"횺"#, r#"횺"#),
        (r#"횻"#, r#"횻"#, r#"횻"#, r#"횻"#, r#"횻"#),
        (r#"횼"#, r#"횼"#, r#"횼"#, r#"횼"#, r#"횼"#),
        (r#"횽"#, r#"횽"#, r#"횽"#, r#"횽"#, r#"횽"#),
        (r#"횾"#, r#"횾"#, r#"횾"#, r#"횾"#, r#"횾"#),
        (r#"횿"#, r#"횿"#, r#"횿"#, r#"횿"#, r#"횿"#),
        (r#"훀"#, r#"훀"#, r#"훀"#, r#"훀"#, r#"훀"#),
        (r#"훁"#, r#"훁"#, r#"훁"#, r#"훁"#, r#"훁"#),
        (r#"훂"#, r#"훂"#, r#"훂"#, r#"훂"#, r#"훂"#),
        (r#"훃"#, r#"훃"#, r#"훃"#, r#"훃"#, r#"훃"#),
        (r#"후"#, r#"후"#, r#"후"#, r#"후"#, r#"후"#),
        (r#"훅"#, r#"훅"#, r#"훅"#, r#"훅"#, r#"훅"#),
        (r#"훆"#, r#"훆"#, r#"훆"#, r#"훆"#, r#"훆"#),
        (r#"훇"#, r#"훇"#, r#"훇"#, r#"훇"#, r#"훇"#),
        (r#"훈"#, r#"훈"#, r#"훈"#, r#"훈"#, r#"훈"#),
        (r#"훉"#, r#"훉"#, r#"훉"#, r#"훉"#, r#"훉"#),
        (r#"훊"#, r#"훊"#, r#"훊"#, r#"훊"#, r#"훊"#),
        (r#"훋"#, r#"훋"#, r#"훋"#, r#"훋"#, r#"훋"#),
        (r#"훌"#, r#"훌"#, r#"훌"#, r#"훌"#, r#"훌"#),
        (r#"훍"#, r#"훍"#, r#"훍"#, r#"훍"#, r#"훍"#),
        (r#"훎"#, r#"훎"#, r#"훎"#, r#"훎"#, r#"훎"#),
        (r#"훏"#, r#"훏"#, r#"훏"#, r#"훏"#, r#"훏"#),
        (r#"훐"#, r#"훐"#, r#"훐"#, r#"훐"#, r#"훐"#),
        (r#"훑"#, r#"훑"#, r#"훑"#, r#"훑"#, r#"훑"#),
        (r#"훒"#, r#"훒"#, r#"훒"#, r#"훒"#, r#"훒"#),
        (r#"훓"#, r#"훓"#, r#"훓"#, r#"훓"#, r#"훓"#),
        (r#"훔"#, r#"훔"#, r#"훔"#, r#"훔"#, r#"훔"#),
        (r#"훕"#, r#"훕"#, r#"훕"#, r#"훕"#, r#"훕"#),
        (r#"훖"#, r#"훖"#, r#"훖"#, r#"훖"#, r#"훖"#),
        (r#"훗"#, r#"훗"#, r#"훗"#, r#"훗"#, r#"훗"#),
        (r#"훘"#, r#"훘"#, r#"훘"#, r#"훘"#, r#"훘"#),
        (r#"훙"#, r#"훙"#, r#"훙"#, r#"훙"#, r#"훙"#),
        (r#"훚"#, r#"훚"#, r#"훚"#, r#"훚"#, r#"훚"#),
        (r#"훛"#, r#"훛"#, r#"훛"#, r#"훛"#, r#"훛"#),
        (r#"훜"#, r#"훜"#, r#"훜"#, r#"훜"#, r#"훜"#),
        (r#"훝"#, r#"훝"#, r#"훝"#, r#"훝"#, r#"훝"#),
        (r#"훞"#, r#"훞"#, r#"훞"#, r#"훞"#, r#"훞"#),
        (r#"훟"#, r#"훟"#, r#"훟"#, r#"훟"#, r#"훟"#),
        (r#"훠"#, r#"훠"#, r#"훠"#, r#"훠"#, r#"훠"#),
        (r#"훡"#, r#"훡"#, r#"훡"#, r#"훡"#, r#"훡"#),
        (r#"훢"#, r#"훢"#, r#"훢"#, r#"훢"#, r#"훢"#),
        (r#"훣"#, r#"훣"#, r#"훣"#, r#"훣"#, r#"훣"#),
        (r#"훤"#, r#"훤"#, r#"훤"#, r#"훤"#, r#"훤"#),
        (r#"훥"#, r#"훥"#, r#"훥"#, r#"훥"#, r#"훥"#),
        (r#"훦"#, r#"훦"#, r#"훦"#, r#"훦"#, r#"훦"#),
        (r#"훧"#, r#"훧"#, r#"훧"#, r#"훧"#, r#"훧"#),
        (r#"훨"#, r#"훨"#, r#"훨"#, r#"훨"#, r#"훨"#),
        (r#"훩"#, r#"훩"#, r#"훩"#, r#"훩"#, r#"훩"#),
        (r#"훪"#, r#"훪"#, r#"훪"#, r#"훪"#, r#"훪"#),
        (r#"훫"#, r#"훫"#, r#"훫"#, r#"훫"#, r#"훫"#),
        (r#"훬"#, r#"훬"#, r#"훬"#, r#"훬"#, r#"훬"#),
        (r#"훭"#, r#"훭"#, r#"훭"#, r#"훭"#, r#"훭"#),
        (r#"훮"#, r#"훮"#, r#"훮"#, r#"훮"#, r#"훮"#),
        (r#"훯"#, r#"훯"#, r#"훯"#, r#"훯"#, r#"훯"#),
        (r#"훰"#, r#"훰"#, r#"훰"#, r#"훰"#, r#"훰"#),
        (r#"훱"#, r#"훱"#, r#"훱"#, r#"훱"#, r#"훱"#),
        (r#"훲"#, r#"훲"#, r#"훲"#, r#"훲"#, r#"훲"#),
        (r#"훳"#, r#"훳"#, r#"훳"#, r#"훳"#, r#"훳"#),
        (r#"훴"#, r#"훴"#, r#"훴"#, r#"훴"#, r#"훴"#),
        (r#"훵"#, r#"훵"#, r#"훵"#, r#"훵"#, r#"훵"#),
        (r#"훶"#, r#"훶"#, r#"훶"#, r#"훶"#, r#"훶"#),
        (r#"훷"#, r#"훷"#, r#"훷"#, r#"훷"#, r#"훷"#),
        (r#"훸"#, r#"훸"#, r#"훸"#, r#"훸"#, r#"훸"#),
        (r#"훹"#, r#"훹"#, r#"훹"#, r#"훹"#, r#"훹"#),
        (r#"훺"#, r#"훺"#, r#"훺"#, r#"훺"#, r#"훺"#),
        (r#"훻"#, r#"훻"#, r#"훻"#, r#"훻"#, r#"훻"#),
        (r#"훼"#, r#"훼"#, r#"훼"#, r#"훼"#, r#"훼"#),
        (r#"훽"#, r#"훽"#, r#"훽"#, r#"훽"#, r#"훽"#),
        (r#"훾"#, r#"훾"#, r#"훾"#, r#"훾"#, r#"훾"#),
        (r#"훿"#, r#"훿"#, r#"훿"#, r#"훿"#, r#"훿"#),
        (r#"휀"#, r#"휀"#, r#"휀"#, r#"휀"#, r#"휀"#),
        (r#"휁"#, r#"휁"#, r#"휁"#, r#"휁"#, r#"휁"#),
        (r#"휂"#, r#"휂"#, r#"휂"#, r#"휂"#, r#"휂"#),
        (r#"휃"#, r#"휃"#, r#"휃"#, r#"휃"#, r#"휃"#),
        (r#"휄"#, r#"휄"#, r#"휄"#, r#"휄"#, r#"휄"#),
        (r#"휅"#, r#"휅"#, r#"휅"#, r#"휅"#, r#"휅"#),
        (r#"휆"#, r#"휆"#, r#"휆"#, r#"휆"#, r#"휆"#),
        (r#"휇"#, r#"휇"#, r#"휇"#, r#"휇"#, r#"휇"#),
        (r#"휈"#, r#"휈"#, r#"휈"#, r#"휈"#, r#"휈"#),
        (r#"휉"#, r#"휉"#, r#"휉"#, r#"휉"#, r#"휉"#),
        (r#"휊"#, r#"휊"#, r#"휊"#, r#"휊"#, r#"휊"#),
        (r#"휋"#, r#"휋"#, r#"휋"#, r#"휋"#, r#"휋"#),
        (r#"휌"#, r#"휌"#, r#"휌"#, r#"휌"#, r#"휌"#),
        (r#"휍"#, r#"휍"#, r#"휍"#, r#"휍"#, r#"휍"#),
        (r#"휎"#, r#"휎"#, r#"휎"#, r#"휎"#, r#"휎"#),
        (r#"휏"#, r#"휏"#, r#"휏"#, r#"휏"#, r#"휏"#),
        (r#"휐"#, r#"휐"#, r#"휐"#, r#"휐"#, r#"휐"#),
        (r#"휑"#, r#"휑"#, r#"휑"#, r#"휑"#, r#"휑"#),
        (r#"휒"#, r#"휒"#, r#"휒"#, r#"휒"#, r#"휒"#),
        (r#"휓"#, r#"휓"#, r#"휓"#, r#"휓"#, r#"휓"#),
        (r#"휔"#, r#"휔"#, r#"휔"#, r#"휔"#, r#"휔"#),
        (r#"휕"#, r#"휕"#, r#"휕"#, r#"휕"#, r#"휕"#),
        (r#"휖"#, r#"휖"#, r#"휖"#, r#"휖"#, r#"휖"#),
        (r#"휗"#, r#"휗"#, r#"휗"#, r#"휗"#, r#"휗"#),
        (r#"휘"#, r#"휘"#, r#"휘"#, r#"휘"#, r#"휘"#),
        (r#"휙"#, r#"휙"#, r#"휙"#, r#"휙"#, r#"휙"#),
        (r#"휚"#, r#"휚"#, r#"휚"#, r#"휚"#, r#"휚"#),
        (r#"휛"#, r#"휛"#, r#"휛"#, r#"휛"#, r#"휛"#),
        (r#"휜"#, r#"휜"#, r#"휜"#, r#"휜"#, r#"휜"#),
        (r#"휝"#, r#"휝"#, r#"휝"#, r#"휝"#, r#"휝"#),
        (r#"휞"#, r#"휞"#, r#"휞"#, r#"휞"#, r#"휞"#),
        (r#"휟"#, r#"휟"#, r#"휟"#, r#"휟"#, r#"휟"#),
        (r#"휠"#, r#"휠"#, r#"휠"#, r#"휠"#, r#"휠"#),
        (r#"휡"#, r#"휡"#, r#"휡"#, r#"휡"#, r#"휡"#),
        (r#"휢"#, r#"휢"#, r#"휢"#, r#"휢"#, r#"휢"#),
        (r#"휣"#, r#"휣"#, r#"휣"#, r#"휣"#, r#"휣"#),
        (r#"휤"#, r#"휤"#, r#"휤"#, r#"휤"#, r#"휤"#),
        (r#"휥"#, r#"휥"#, r#"휥"#, r#"휥"#, r#"휥"#),
        (r#"휦"#, r#"휦"#, r#"휦"#, r#"휦"#, r#"휦"#),
        (r#"휧"#, r#"휧"#, r#"휧"#, r#"휧"#, r#"휧"#),
        (r#"휨"#, r#"휨"#, r#"휨"#, r#"휨"#, r#"휨"#),
        (r#"휩"#, r#"휩"#, r#"휩"#, r#"휩"#, r#"휩"#),
        (r#"휪"#, r#"휪"#, r#"휪"#, r#"휪"#, r#"휪"#),
        (r#"휫"#, r#"휫"#, r#"휫"#, r#"휫"#, r#"휫"#),
        (r#"휬"#, r#"휬"#, r#"휬"#, r#"휬"#, r#"휬"#),
        (r#"휭"#, r#"휭"#, r#"휭"#, r#"휭"#, r#"휭"#),
        (r#"휮"#, r#"휮"#, r#"휮"#, r#"휮"#, r#"휮"#),
        (r#"휯"#, r#"휯"#, r#"휯"#, r#"휯"#, r#"휯"#),
        (r#"휰"#, r#"휰"#, r#"휰"#, r#"휰"#, r#"휰"#),
        (r#"휱"#, r#"휱"#, r#"휱"#, r#"휱"#, r#"휱"#),
        (r#"휲"#, r#"휲"#, r#"휲"#, r#"휲"#, r#"휲"#),
        (r#"휳"#, r#"휳"#, r#"휳"#, r#"휳"#, r#"휳"#),
        (r#"휴"#, r#"휴"#, r#"휴"#, r#"휴"#, r#"휴"#),
        (r#"휵"#, r#"휵"#, r#"휵"#, r#"휵"#, r#"휵"#),
        (r#"휶"#, r#"휶"#, r#"휶"#, r#"휶"#, r#"휶"#),
        (r#"휷"#, r#"휷"#, r#"휷"#, r#"휷"#, r#"휷"#),
        (r#"휸"#, r#"휸"#, r#"휸"#, r#"휸"#, r#"휸"#),
        (r#"휹"#, r#"휹"#, r#"휹"#, r#"휹"#, r#"휹"#),
        (r#"휺"#, r#"휺"#, r#"휺"#, r#"휺"#, r#"휺"#),
        (r#"휻"#, r#"휻"#, r#"휻"#, r#"휻"#, r#"휻"#),
        (r#"휼"#, r#"휼"#, r#"휼"#, r#"휼"#, r#"휼"#),
        (r#"휽"#, r#"휽"#, r#"휽"#, r#"휽"#, r#"휽"#),
        (r#"휾"#, r#"휾"#, r#"휾"#, r#"휾"#, r#"휾"#),
        (r#"휿"#, r#"휿"#, r#"휿"#, r#"휿"#, r#"휿"#),
        (r#"흀"#, r#"흀"#, r#"흀"#, r#"흀"#, r#"흀"#),
        (r#"흁"#, r#"흁"#, r#"흁"#, r#"흁"#, r#"흁"#),
        (r#"흂"#, r#"흂"#, r#"흂"#, r#"흂"#, r#"흂"#),
        (r#"흃"#, r#"흃"#, r#"흃"#, r#"흃"#, r#"흃"#),
        (r#"흄"#, r#"흄"#, r#"흄"#, r#"흄"#, r#"흄"#),
        (r#"흅"#, r#"흅"#, r#"흅"#, r#"흅"#, r#"흅"#),
        (r#"흆"#, r#"흆"#, r#"흆"#, r#"흆"#, r#"흆"#),
        (r#"흇"#, r#"흇"#, r#"흇"#, r#"흇"#, r#"흇"#),
        (r#"흈"#, r#"흈"#, r#"흈"#, r#"흈"#, r#"흈"#),
        (r#"흉"#, r#"흉"#, r#"흉"#, r#"흉"#, r#"흉"#),
        (r#"흊"#, r#"흊"#, r#"흊"#, r#"흊"#, r#"흊"#),
        (r#"흋"#, r#"흋"#, r#"흋"#, r#"흋"#, r#"흋"#),
        (r#"흌"#, r#"흌"#, r#"흌"#, r#"흌"#, r#"흌"#),
        (r#"흍"#, r#"흍"#, r#"흍"#, r#"흍"#, r#"흍"#),
        (r#"흎"#, r#"흎"#, r#"흎"#, r#"흎"#, r#"흎"#),
        (r#"흏"#, r#"흏"#, r#"흏"#, r#"흏"#, r#"흏"#),
        (r#"흐"#, r#"흐"#, r#"흐"#, r#"흐"#, r#"흐"#),
        (r#"흑"#, r#"흑"#, r#"흑"#, r#"흑"#, r#"흑"#),
        (r#"흒"#, r#"흒"#, r#"흒"#, r#"흒"#, r#"흒"#),
        (r#"흓"#, r#"흓"#, r#"흓"#, r#"흓"#, r#"흓"#),
        (r#"흔"#, r#"흔"#, r#"흔"#, r#"흔"#, r#"흔"#),
        (r#"흕"#, r#"흕"#, r#"흕"#, r#"흕"#, r#"흕"#),
        (r#"흖"#, r#"흖"#, r#"흖"#, r#"흖"#, r#"흖"#),
        (r#"흗"#, r#"흗"#, r#"흗"#, r#"흗"#, r#"흗"#),
        (r#"흘"#, r#"흘"#, r#"흘"#, r#"흘"#, r#"흘"#),
        (r#"흙"#, r#"흙"#, r#"흙"#, r#"흙"#, r#"흙"#),
        (r#"흚"#, r#"흚"#, r#"흚"#, r#"흚"#, r#"흚"#),
        (r#"흛"#, r#"흛"#, r#"흛"#, r#"흛"#, r#"흛"#),
        (r#"흜"#, r#"흜"#, r#"흜"#, r#"흜"#, r#"흜"#),
        (r#"흝"#, r#"흝"#, r#"흝"#, r#"흝"#, r#"흝"#),
        (r#"흞"#, r#"흞"#, r#"흞"#, r#"흞"#, r#"흞"#),
        (r#"흟"#, r#"흟"#, r#"흟"#, r#"흟"#, r#"흟"#),
        (r#"흠"#, r#"흠"#, r#"흠"#, r#"흠"#, r#"흠"#),
        (r#"흡"#, r#"흡"#, r#"흡"#, r#"흡"#, r#"흡"#),
        (r#"흢"#, r#"흢"#, r#"흢"#, r#"흢"#, r#"흢"#),
        (r#"흣"#, r#"흣"#, r#"흣"#, r#"흣"#, r#"흣"#),
        (r#"흤"#, r#"흤"#, r#"흤"#, r#"흤"#, r#"흤"#),
        (r#"흥"#, r#"흥"#, r#"흥"#, r#"흥"#, r#"흥"#),
        (r#"흦"#, r#"흦"#, r#"흦"#, r#"흦"#, r#"흦"#),
        (r#"흧"#, r#"흧"#, r#"흧"#, r#"흧"#, r#"흧"#),
        (r#"흨"#, r#"흨"#, r#"흨"#, r#"흨"#, r#"흨"#),
        (r#"흩"#, r#"흩"#, r#"흩"#, r#"흩"#, r#"흩"#),
        (r#"흪"#, r#"흪"#, r#"흪"#, r#"흪"#, r#"흪"#),
        (r#"흫"#, r#"흫"#, r#"흫"#, r#"흫"#, r#"흫"#),
        (r#"희"#, r#"희"#, r#"희"#, r#"희"#, r#"희"#),
        (r#"흭"#, r#"흭"#, r#"흭"#, r#"흭"#, r#"흭"#),
        (r#"흮"#, r#"흮"#, r#"흮"#, r#"흮"#, r#"흮"#),
        (r#"흯"#, r#"흯"#, r#"흯"#, r#"흯"#, r#"흯"#),
        (r#"흰"#, r#"흰"#, r#"흰"#, r#"흰"#, r#"흰"#),
        (r#"흱"#, r#"흱"#, r#"흱"#, r#"흱"#, r#"흱"#),
        (r#"흲"#, r#"흲"#, r#"흲"#, r#"흲"#, r#"흲"#),
        (r#"흳"#, r#"흳"#, r#"흳"#, r#"흳"#, r#"흳"#),
        (r#"흴"#, r#"흴"#, r#"흴"#, r#"흴"#, r#"흴"#),
        (r#"흵"#, r#"흵"#, r#"흵"#, r#"흵"#, r#"흵"#),
        (r#"흶"#, r#"흶"#, r#"흶"#, r#"흶"#, r#"흶"#),
        (r#"흷"#, r#"흷"#, r#"흷"#, r#"흷"#, r#"흷"#),
        (r#"흸"#, r#"흸"#, r#"흸"#, r#"흸"#, r#"흸"#),
        (r#"흹"#, r#"흹"#, r#"흹"#, r#"흹"#, r#"흹"#),
        (r#"흺"#, r#"흺"#, r#"흺"#, r#"흺"#, r#"흺"#),
        (r#"흻"#, r#"흻"#, r#"흻"#, r#"흻"#, r#"흻"#),
        (r#"흼"#, r#"흼"#, r#"흼"#, r#"흼"#, r#"흼"#),
        (r#"흽"#, r#"흽"#, r#"흽"#, r#"흽"#, r#"흽"#),
        (r#"흾"#, r#"흾"#, r#"흾"#, r#"흾"#, r#"흾"#),
        (r#"흿"#, r#"흿"#, r#"흿"#, r#"흿"#, r#"흿"#),
        (r#"힀"#, r#"힀"#, r#"힀"#, r#"힀"#, r#"힀"#),
        (r#"힁"#, r#"힁"#, r#"힁"#, r#"힁"#, r#"힁"#),
        (r#"힂"#, r#"힂"#, r#"힂"#, r#"힂"#, r#"힂"#),
        (r#"힃"#, r#"힃"#, r#"힃"#, r#"힃"#, r#"힃"#),
        (r#"힄"#, r#"힄"#, r#"힄"#, r#"힄"#, r#"힄"#),
        (r#"힅"#, r#"힅"#, r#"힅"#, r#"힅"#, r#"힅"#),
        (r#"힆"#, r#"힆"#, r#"힆"#, r#"힆"#, r#"힆"#),
        (r#"힇"#, r#"힇"#, r#"힇"#, r#"힇"#, r#"힇"#),
        (r#"히"#, r#"히"#, r#"히"#, r#"히"#, r#"히"#),
        (r#"힉"#, r#"힉"#, r#"힉"#, r#"힉"#, r#"힉"#),
        (r#"힊"#, r#"힊"#, r#"힊"#, r#"힊"#, r#"힊"#),
        (r#"힋"#, r#"힋"#, r#"힋"#, r#"힋"#, r#"힋"#),
        (r#"힌"#, r#"힌"#, r#"힌"#, r#"힌"#, r#"힌"#),
        (r#"힍"#, r#"힍"#, r#"힍"#, r#"힍"#, r#"힍"#),
        (r#"힎"#, r#"힎"#, r#"힎"#, r#"힎"#, r#"힎"#),
        (r#"힏"#, r#"힏"#, r#"힏"#, r#"힏"#, r#"힏"#),
        (r#"힐"#, r#"힐"#, r#"힐"#, r#"힐"#, r#"힐"#),
        (r#"힑"#, r#"힑"#, r#"힑"#, r#"힑"#, r#"힑"#),
        (r#"힒"#, r#"힒"#, r#"힒"#, r#"힒"#, r#"힒"#),
        (r#"힓"#, r#"힓"#, r#"힓"#, r#"힓"#, r#"힓"#),
        (r#"힔"#, r#"힔"#, r#"힔"#, r#"힔"#, r#"힔"#),
        (r#"힕"#, r#"힕"#, r#"힕"#, r#"힕"#, r#"힕"#),
        (r#"힖"#, r#"힖"#, r#"힖"#, r#"힖"#, r#"힖"#),
        (r#"힗"#, r#"힗"#, r#"힗"#, r#"힗"#, r#"힗"#),
        (r#"힘"#, r#"힘"#, r#"힘"#, r#"힘"#, r#"힘"#),
        (r#"힙"#, r#"힙"#, r#"힙"#, r#"힙"#, r#"힙"#),
        (r#"힚"#, r#"힚"#, r#"힚"#, r#"힚"#, r#"힚"#),
        (r#"힛"#, r#"힛"#, r#"힛"#, r#"힛"#, r#"힛"#),
        (r#"힜"#, r#"힜"#, r#"힜"#, r#"힜"#, r#"힜"#),
        (r#"힝"#, r#"힝"#, r#"힝"#, r#"힝"#, r#"힝"#),
        (r#"힞"#, r#"힞"#, r#"힞"#, r#"힞"#, r#"힞"#),
        (r#"힟"#, r#"힟"#, r#"힟"#, r#"힟"#, r#"힟"#),
        (r#"힠"#, r#"힠"#, r#"힠"#, r#"힠"#, r#"힠"#),
        (r#"힡"#, r#"힡"#, r#"힡"#, r#"힡"#, r#"힡"#),
        (r#"힢"#, r#"힢"#, r#"힢"#, r#"힢"#, r#"힢"#),
        (r#"힣"#, r#"힣"#, r#"힣"#, r#"힣"#, r#"힣"#),
        (r#"豈"#, r#"豈"#, r#"豈"#, r#"豈"#, r#"豈"#),
        (r#"更"#, r#"更"#, r#"更"#, r#"更"#, r#"更"#),
        (r#"車"#, r#"車"#, r#"車"#, r#"車"#, r#"車"#),
        (r#"賈"#, r#"賈"#, r#"賈"#, r#"賈"#, r#"賈"#),
        (r#"滑"#, r#"滑"#, r#"滑"#, r#"滑"#, r#"滑"#),
        (r#"串"#, r#"串"#, r#"串"#, r#"串"#, r#"串"#),
        (r#"句"#, r#"句"#, r#"句"#, r#"句"#, r#"句"#),
        (r#"龜"#, r#"龜"#, r#"龜"#, r#"龜"#, r#"龜"#),
        (r#"龜"#, r#"龜"#, r#"龜"#, r#"龜"#, r#"龜"#),
        (r#"契"#, r#"契"#, r#"契"#, r#"契"#, r#"契"#),
        (r#"金"#, r#"金"#, r#"金"#, r#"金"#, r#"金"#),
        (r#"喇"#, r#"喇"#, r#"喇"#, r#"喇"#, r#"喇"#),
        (r#"奈"#, r#"奈"#, r#"奈"#, r#"奈"#, r#"奈"#),
        (r#"懶"#, r#"懶"#, r#"懶"#, r#"懶"#, r#"懶"#),
        (r#"癩"#, r#"癩"#, r#"癩"#, r#"癩"#, r#"癩"#),
        (r#"羅"#, r#"羅"#, r#"羅"#, r#"羅"#, r#"羅"#),
        (r#"蘿"#, r#"蘿"#, r#"蘿"#, r#"蘿"#, r#"蘿"#),
        (r#"螺"#, r#"螺"#, r#"螺"#, r#"螺"#, r#"螺"#),
        (r#"裸"#, r#"裸"#, r#"裸"#, r#"裸"#, r#"裸"#),
        (r#"邏"#, r#"邏"#, r#"邏"#, r#"邏"#, r#"邏"#),
        (r#"樂"#, r#"樂"#, r#"樂"#, r#"樂"#, r#"樂"#),
        (r#"洛"#, r#"洛"#, r#"洛"#, r#"洛"#, r#"洛"#),
        (r#"烙"#, r#"烙"#, r#"烙"#, r#"烙"#, r#"烙"#),
        (r#"珞"#, r#"珞"#, r#"珞"#, r#"珞"#, r#"珞"#),
        (r#"落"#, r#"落"#, r#"落"#, r#"落"#, r#"落"#),
        (r#"酪"#, r#"酪"#, r#"酪"#, r#"酪"#, r#"酪"#),
        (r#"駱"#, r#"駱"#, r#"駱"#, r#"駱"#, r#"駱"#),
        (r#"亂"#, r#"亂"#, r#"亂"#, r#"亂"#, r#"亂"#),
        (r#"卵"#, r#"卵"#, r#"卵"#, r#"卵"#, r#"卵"#),
        (r#"欄"#, r#"欄"#, r#"欄"#, r#"欄"#, r#"欄"#),
        (r#"爛"#, r#"爛"#, r#"爛"#, r#"爛"#, r#"爛"#),
        (r#"蘭"#, r#"蘭"#, r#"蘭"#, r#"蘭"#, r#"蘭"#),
        (r#"鸞"#, r#"鸞"#, r#"鸞"#, r#"鸞"#, r#"鸞"#),
        (r#"嵐"#, r#"嵐"#, r#"嵐"#, r#"嵐"#, r#"嵐"#),
        (r#"濫"#, r#"濫"#, r#"濫"#, r#"濫"#, r#"濫"#),
        (r#"藍"#, r#"藍"#, r#"藍"#, r#"藍"#, r#"藍"#),
        (r#"襤"#, r#"襤"#, r#"襤"#, r#"襤"#, r#"襤"#),
        (r#"拉"#, r#"拉"#, r#"拉"#, r#"拉"#, r#"拉"#),
        (r#"臘"#, r#"臘"#, r#"臘"#, r#"臘"#, r#"臘"#),
        (r#"蠟"#, r#"蠟"#, r#"蠟"#, r#"蠟"#, r#"蠟"#),
        (r#"廊"#, r#"廊"#, r#"廊"#, r#"廊"#, r#"廊"#),
        (r#"朗"#, r#"朗"#, r#"朗"#, r#"朗"#, r#"朗"#),
        (r#"浪"#, r#"浪"#, r#"浪"#, r#"浪"#, r#"浪"#),
        (r#"狼"#, r#"狼"#, r#"狼"#, r#"狼"#, r#"狼"#),
        (r#"郎"#, r#"郎"#, r#"郎"#, r#"郎"#, r#"郎"#),
        (r#"來"#, r#"來"#, r#"來"#, r#"來"#, r#"來"#),
        (r#"冷"#, r#"冷"#, r#"冷"#, r#"冷"#, r#"冷"#),
        (r#"勞"#, r#"勞"#, r#"勞"#, r#"勞"#, r#"勞"#),
        (r#"擄"#, r#"擄"#, r#"擄"#, r#"擄"#, r#"擄"#),
        (r#"櫓"#, r#"櫓"#, r#"櫓"#, r#"櫓"#, r#"櫓"#),
        (r#"爐"#, r#"爐"#, r#"爐"#, r#"爐"#, r#"爐"#),
        (r#"盧"#, r#"盧"#, r#"盧"#, r#"盧"#, r#"盧"#),
        (r#"老"#, r#"老"#, r#"老"#, r#"老"#, r#"老"#),
        (r#"蘆"#, r#"蘆"#, r#"蘆"#, r#"蘆"#, r#"蘆"#),
        (r#"虜"#, r#"虜"#, r#"虜"#, r#"虜"#, r#"虜"#),
        (r#"路"#, r#"路"#, r#"路"#, r#"路"#, r#"路"#),
        (r#"露"#, r#"露"#, r#"露"#, r#"露"#, r#"露"#),
        (r#"魯"#, r#"魯"#, r#"魯"#, r#"魯"#, r#"魯"#),
        (r#"鷺"#, r#"鷺"#, r#"鷺"#, r#"鷺"#, r#"鷺"#),
        (r#"碌"#, r#"碌"#, r#"碌"#, r#"碌"#, r#"碌"#),
        (r#"祿"#, r#"祿"#, r#"祿"#, r#"祿"#, r#"祿"#),
        (r#"綠"#, r#"綠"#, r#"綠"#, r#"綠"#, r#"綠"#),
        (r#"菉"#, r#"菉"#, r#"菉"#, r#"菉"#, r#"菉"#),
        (r#"錄"#, r#"錄"#, r#"錄"#, r#"錄"#, r#"錄"#),
        (r#"鹿"#, r#"鹿"#, r#"鹿"#, r#"鹿"#, r#"鹿"#),
        (r#"論"#, r#"論"#, r#"論"#, r#"論"#, r#"論"#),
        (r#"壟"#, r#"壟"#, r#"壟"#, r#"壟"#, r#"壟"#),
        (r#"弄"#, r#"弄"#, r#"弄"#, r#"弄"#, r#"弄"#),
        (r#"籠"#, r#"籠"#, r#"籠"#, r#"籠"#, r#"籠"#),
        (r#"聾"#, r#"聾"#, r#"聾"#, r#"聾"#, r#"聾"#),
        (r#"牢"#, r#"牢"#, r#"牢"#, r#"牢"#, r#"牢"#),
        (r#"磊"#, r#"磊"#, r#"磊"#, r#"磊"#, r#"磊"#),
        (r#"賂"#, r#"賂"#, r#"賂"#, r#"賂"#, r#"賂"#),
        (r#"雷"#, r#"雷"#, r#"雷"#, r#"雷"#, r#"雷"#),
        (r#"壘"#, r#"壘"#, r#"壘"#, r#"壘"#, r#"壘"#),
        (r#"屢"#, r#"屢"#, r#"屢"#, r#"屢"#, r#"屢"#),
        (r#"樓"#, r#"樓"#, r#"樓"#, r#"樓"#, r#"樓"#),
        (r#"淚"#, r#"淚"#, r#"淚"#, r#"淚"#, r#"淚"#),
        (r#"漏"#, r#"漏"#, r#"漏"#, r#"漏"#, r#"漏"#),
        (r#"累"#, r#"累"#, r#"累"#, r#"累"#, r#"累"#),
        (r#"縷"#, r#"縷"#, r#"縷"#, r#"縷"#, r#"縷"#),
        (r#"陋"#, r#"陋"#, r#"陋"#, r#"陋"#, r#"陋"#),
        (r#"勒"#, r#"勒"#, r#"勒"#, r#"勒"#, r#"勒"#),
        (r#"肋"#, r#"肋"#, r#"肋"#, r#"肋"#, r#"肋"#),
        (r#"凜"#, r#"凜"#, r#"凜"#, r#"凜"#, r#"凜"#),
        (r#"凌"#, r#"凌"#, r#"凌"#, r#"凌"#, r#"凌"#),
        (r#"稜"#, r#"稜"#, r#"稜"#, r#"稜"#, r#"稜"#),
        (r#"綾"#, r#"綾"#, r#"綾"#, r#"綾"#, r#"綾"#),
        (r#"菱"#, r#"菱"#, r#"菱"#, r#"菱"#, r#"菱"#),
        (r#"陵"#, r#"陵"#, r#"陵"#, r#"陵"#, r#"陵"#),
        (r#"讀"#, r#"讀"#, r#"讀"#, r#"讀"#, r#"讀"#),
        (r#"拏"#, r#"拏"#, r#"拏"#, r#"拏"#, r#"拏"#),
        (r#"樂"#, r#"樂"#, r#"樂"#, r#"樂"#, r#"樂"#),
        (r#"諾"#, r#"諾"#, r#"諾"#, r#"諾"#, r#"諾"#),
        (r#"丹"#, r#"丹"#, r#"丹"#, r#"丹"#, r#"丹"#),
        (r#"寧"#, r#"寧"#, r#"寧"#, r#"寧"#, r#"寧"#),
        (r#"怒"#, r#"怒"#, r#"怒"#, r#"怒"#, r#"怒"#),
        (r#"率"#, r#"率"#, r#"率"#, r#"率"#, r#"率"#),
        (r#"異"#, r#"異"#, r#"異"#, r#"異"#, r#"異"#),
        (r#"北"#, r#"北"#, r#"北"#, r#"北"#, r#"北"#),
        (r#"磻"#, r#"磻"#, r#"磻"#, r#"磻"#, r#"磻"#),
        (r#"便"#, r#"便"#, r#"便"#, r#"便"#, r#"便"#),
        (r#"復"#, r#"復"#, r#"復"#, r#"復"#, r#"復"#),
        (r#"不"#, r#"不"#, r#"不"#, r#"不"#, r#"不"#),
        (r#"泌"#, r#"泌"#, r#"泌"#, r#"泌"#, r#"泌"#),
        (r#"數"#, r#"數"#, r#"數"#, r#"數"#, r#"數"#),
        (r#"索"#, r#"索"#, r#"索"#, r#"索"#, r#"索"#),
        (r#"參"#, r#"參"#, r#"參"#, r#"參"#, r#"參"#),
        (r#"塞"#, r#"塞"#, r#"塞"#, r#"塞"#, r#"塞"#),
        (r#"省"#, r#"省"#, r#"省"#, r#"省"#, r#"省"#),
        (r#"葉"#, r#"葉"#, r#"葉"#, r#"葉"#, r#"葉"#),
        (r#"說"#, r#"說"#, r#"說"#, r#"說"#, r#"說"#),
        (r#"殺"#, r#"殺"#, r#"殺"#, r#"殺"#, r#"殺"#),
        (r#"辰"#, r#"辰"#, r#"辰"#, r#"辰"#, r#"辰"#),
        (r#"沈"#, r#"沈"#, r#"沈"#, r#"沈"#, r#"沈"#),
        (r#"拾"#, r#"拾"#, r#"拾"#, r#"拾"#, r#"拾"#),
        (r#"若"#, r#"若"#, r#"若"#, r#"若"#, r#"若"#),
        (r#"掠"#, r#"掠"#, r#"掠"#, r#"掠"#, r#"掠"#),
        (r#"略"#, r#"略"#, r#"略"#, r#"略"#, r#"略"#),
        (r#"亮"#, r#"亮"#, r#"亮"#, r#"亮"#, r#"亮"#),
        (r#"兩"#, r#"兩"#, r#"兩"#, r#"兩"#, r#"兩"#),
        (r#"凉"#, r#"凉"#, r#"凉"#, r#"凉"#, r#"凉"#),
        (r#"梁"#, r#"梁"#, r#"梁"#, r#"梁"#, r#"梁"#),
        (r#"糧"#, r#"糧"#, r#"糧"#, r#"糧"#, r#"糧"#),
        (r#"良"#, r#"良"#, r#"良"#, r#"良"#, r#"良"#),
        (r#"諒"#, r#"諒"#, r#"諒"#, r#"諒"#, r#"諒"#),
        (r#"量"#, r#"量"#, r#"量"#, r#"量"#, r#"量"#),
        (r#"勵"#, r#"勵"#, r#"勵"#, r#"勵"#, r#"勵"#),
        (r#"呂"#, r#"呂"#, r#"呂"#, r#"呂"#, r#"呂"#),
        (r#"女"#, r#"女"#, r#"女"#, r#"女"#, r#"女"#),
        (r#"廬"#, r#"廬"#, r#"廬"#, r#"廬"#, r#"廬"#),
        (r#"旅"#, r#"旅"#, r#"旅"#, r#"旅"#, r#"旅"#),
        (r#"濾"#, r#"濾"#, r#"濾"#, r#"濾"#, r#"濾"#),
        (r#"礪"#, r#"礪"#, r#"礪"#, r#"礪"#, r#"礪"#),
        (r#"閭"#, r#"閭"#, r#"閭"#, r#"閭"#, r#"閭"#),
        (r#"驪"#, r#"驪"#, r#"驪"#, r#"驪"#, r#"驪"#),
        (r#"麗"#, r#"麗"#, r#"麗"#, r#"麗"#, r#"麗"#),
        (r#"黎"#, r#"黎"#, r#"黎"#, r#"黎"#, r#"黎"#),
        (r#"力"#, r#"力"#, r#"力"#, r#"力"#, r#"力"#),
        (r#"曆"#, r#"曆"#, r#"曆"#, r#"曆"#, r#"曆"#),
        (r#"歷"#, r#"歷"#, r#"歷"#, r#"歷"#, r#"歷"#),
        (r#"轢"#, r#"轢"#, r#"轢"#, r#"轢"#, r#"轢"#),
        (r#"年"#, r#"年"#, r#"年"#, r#"年"#, r#"年"#),
        (r#"憐"#, r#"憐"#, r#"憐"#, r#"憐"#, r#"憐"#),
        (r#"戀"#, r#"戀"#, r#"戀"#, r#"戀"#, r#"戀"#),
        (r#"撚"#, r#"撚"#, r#"撚"#, r#"撚"#, r#"撚"#),
        (r#"漣"#, r#"漣"#, r#"漣"#, r#"漣"#, r#"漣"#),
        (r#"煉"#, r#"煉"#, r#"煉"#, r#"煉"#, r#"煉"#),
        (r#"璉"#, r#"璉"#, r#"璉"#, r#"璉"#, r#"璉"#),
        (r#"秊"#, r#"秊"#, r#"秊"#, r#"秊"#, r#"秊"#),
        (r#"練"#, r#"練"#, r#"練"#, r#"練"#, r#"練"#),
        (r#"聯"#, r#"聯"#, r#"聯"#, r#"聯"#, r#"聯"#),
        (r#"輦"#, r#"輦"#, r#"輦"#, r#"輦"#, r#"輦"#),
        (r#"蓮"#, r#"蓮"#, r#"蓮"#, r#"蓮"#, r#"蓮"#),
        (r#"連"#, r#"連"#, r#"連"#, r#"連"#, r#"連"#),
        (r#"鍊"#, r#"鍊"#, r#"鍊"#, r#"鍊"#, r#"鍊"#),
        (r#"列"#, r#"列"#, r#"列"#, r#"列"#, r#"列"#),
        (r#"劣"#, r#"劣"#, r#"劣"#, r#"劣"#, r#"劣"#),
        (r#"咽"#, r#"咽"#, r#"咽"#, r#"咽"#, r#"咽"#),
        (r#"烈"#, r#"烈"#, r#"烈"#, r#"烈"#, r#"烈"#),
        (r#"裂"#, r#"裂"#, r#"裂"#, r#"裂"#, r#"裂"#),
        (r#"說"#, r#"說"#, r#"說"#, r#"說"#, r#"說"#),
        (r#"廉"#, r#"廉"#, r#"廉"#, r#"廉"#, r#"廉"#),
        (r#"念"#, r#"念"#, r#"念"#, r#"念"#, r#"念"#),
        (r#"捻"#, r#"捻"#, r#"捻"#, r#"捻"#, r#"捻"#),
        (r#"殮"#, r#"殮"#, r#"殮"#, r#"殮"#, r#"殮"#),
        (r#"簾"#, r#"簾"#, r#"簾"#, r#"簾"#, r#"簾"#),
        (r#"獵"#, r#"獵"#, r#"獵"#, r#"獵"#, r#"獵"#),
        (r#"令"#, r#"令"#, r#"令"#, r#"令"#, r#"令"#),
        (r#"囹"#, r#"囹"#, r#"囹"#, r#"囹"#, r#"囹"#),
        (r#"寧"#, r#"寧"#, r#"寧"#, r#"寧"#, r#"寧"#),
        (r#"嶺"#, r#"嶺"#, r#"嶺"#, r#"嶺"#, r#"嶺"#),
        (r#"怜"#, r#"怜"#, r#"怜"#, r#"怜"#, r#"怜"#),
        (r#"玲"#, r#"玲"#, r#"玲"#, r#"玲"#, r#"玲"#),
        (r#"瑩"#, r#"瑩"#, r#"瑩"#, r#"瑩"#, r#"瑩"#),
        (r#"羚"#, r#"羚"#, r#"羚"#, r#"羚"#, r#"羚"#),
        (r#"聆"#, r#"聆"#, r#"聆"#, r#"聆"#, r#"聆"#),
        (r#"鈴"#, r#"鈴"#, r#"鈴"#, r#"鈴"#, r#"鈴"#),
        (r#"零"#, r#"零"#, r#"零"#, r#"零"#, r#"零"#),
        (r#"靈"#, r#"靈"#, r#"靈"#, r#"靈"#, r#"靈"#),
        (r#"領"#, r#"領"#, r#"領"#, r#"領"#, r#"領"#),
        (r#"例"#, r#"例"#, r#"例"#, r#"例"#, r#"例"#),
        (r#"禮"#, r#"禮"#, r#"禮"#, r#"禮"#, r#"禮"#),
        (r#"醴"#, r#"醴"#, r#"醴"#, r#"醴"#, r#"醴"#),
        (r#"隸"#, r#"隸"#, r#"隸"#, r#"隸"#, r#"隸"#),
        (r#"惡"#, r#"惡"#, r#"惡"#, r#"惡"#, r#"惡"#),
        (r#"了"#, r#"了"#, r#"了"#, r#"了"#, r#"了"#),
        (r#"僚"#, r#"僚"#, r#"僚"#, r#"僚"#, r#"僚"#),
        (r#"寮"#, r#"寮"#, r#"寮"#, r#"寮"#, r#"寮"#),
        (r#"尿"#, r#"尿"#, r#"尿"#, r#"尿"#, r#"尿"#),
        (r#"料"#, r#"料"#, r#"料"#, r#"料"#, r#"料"#),
        (r#"樂"#, r#"樂"#, r#"樂"#, r#"樂"#, r#"樂"#),
        (r#"燎"#, r#"燎"#, r#"燎"#, r#"燎"#, r#"燎"#),
        (r#"療"#, r#"療"#, r#"療"#, r#"療"#, r#"療"#),
        (r#"蓼"#, r#"蓼"#, r#"蓼"#, r#"蓼"#, r#"蓼"#),
        (r#"遼"#, r#"遼"#, r#"遼"#, r#"遼"#, r#"遼"#),
        (r#"龍"#, r#"龍"#, r#"龍"#, r#"龍"#, r#"龍"#),
        (r#"暈"#, r#"暈"#, r#"暈"#, r#"暈"#, r#"暈"#),
        (r#"阮"#, r#"阮"#, r#"阮"#, r#"阮"#, r#"阮"#),
        (r#"劉"#, r#"劉"#, r#"劉"#, r#"劉"#, r#"劉"#),
        (r#"杻"#, r#"杻"#, r#"杻"#, r#"杻"#, r#"杻"#),
        (r#"柳"#, r#"柳"#, r#"柳"#, r#"柳"#, r#"柳"#),
        (r#"流"#, r#"流"#, r#"流"#, r#"流"#, r#"流"#),
        (r#"溜"#, r#"溜"#, r#"溜"#, r#"溜"#, r#"溜"#),
        (r#"琉"#, r#"琉"#, r#"琉"#, r#"琉"#, r#"琉"#),
        (r#"留"#, r#"留"#, r#"留"#, r#"留"#, r#"留"#),
        (r#"硫"#, r#"硫"#, r#"硫"#, r#"硫"#, r#"硫"#),
        (r#"紐"#, r#"紐"#, r#"紐"#, r#"紐"#, r#"紐"#),
        (r#"類"#, r#"類"#, r#"類"#, r#"類"#, r#"類"#),
        (r#"六"#, r#"六"#, r#"六"#, r#"六"#, r#"六"#),
        (r#"戮"#, r#"戮"#, r#"戮"#, r#"戮"#, r#"戮"#),
        (r#"陸"#, r#"陸"#, r#"陸"#, r#"陸"#, r#"陸"#),
        (r#"倫"#, r#"倫"#, r#"倫"#, r#"倫"#, r#"倫"#),
        (r#"崙"#, r#"崙"#, r#"崙"#, r#"崙"#, r#"崙"#),
        (r#"淪"#, r#"淪"#, r#"淪"#, r#"淪"#, r#"淪"#),
        (r#"輪"#, r#"輪"#, r#"輪"#, r#"輪"#, r#"輪"#),
        (r#"律"#, r#"律"#, r#"律"#, r#"律"#, r#"律"#),
        (r#"慄"#, r#"慄"#, r#"慄"#, r#"慄"#, r#"慄"#),
        (r#"栗"#, r#"栗"#, r#"栗"#, r#"栗"#, r#"栗"#),
        (r#"率"#, r#"率"#, r#"率"#, r#"率"#, r#"率"#),
        (r#"隆"#, r#"隆"#, r#"隆"#, r#"隆"#, r#"隆"#),
        (r#"利"#, r#"利"#, r#"利"#, r#"利"#, r#"利"#),
        (r#"吏"#, r#"吏"#, r#"吏"#, r#"吏"#, r#"吏"#),
        (r#"履"#, r#"履"#, r#"履"#, r#"履"#, r#"履"#),
        (r#"易"#, r#"易"#, r#"易"#, r#"易"#, r#"易"#),
        (r#"李"#, r#"李"#, r#"李"#, r#"李"#, r#"李"#),
        (r#"梨"#, r#"梨"#, r#"梨"#, r#"梨"#, r#"梨"#),
        (r#"泥"#, r#"泥"#, r#"泥"#, r#"泥"#, r#"泥"#),
        (r#"理"#, r#"理"#, r#"理"#, r#"理"#, r#"理"#),
        (r#"痢"#, r#"痢"#, r#"痢"#, r#"痢"#, r#"痢"#),
        (r#"罹"#, r#"罹"#, r#"罹"#, r#"罹"#, r#"罹"#),
        (r#"裏"#, r#"裏"#, r#"裏"#, r#"裏"#, r#"裏"#),
        (r#"裡"#, r#"裡"#, r#"裡"#, r#"裡"#, r#"裡"#),
        (r#"里"#, r#"里"#, r#"里"#, r#"里"#, r#"里"#),
        (r#"離"#, r#"離"#, r#"離"#, r#"離"#, r#"離"#),
        (r#"匿"#, r#"匿"#, r#"匿"#, r#"匿"#, r#"匿"#),
        (r#"溺"#, r#"溺"#, r#"溺"#, r#"溺"#, r#"溺"#),
        (r#"吝"#, r#"吝"#, r#"吝"#, r#"吝"#, r#"吝"#),
        (r#"燐"#, r#"燐"#, r#"燐"#, r#"燐"#, r#"燐"#),
        (r#"璘"#, r#"璘"#, r#"璘"#, r#"璘"#, r#"璘"#),
        (r#"藺"#, r#"藺"#, r#"藺"#, r#"藺"#, r#"藺"#),
        (r#"隣"#, r#"隣"#, r#"隣"#, r#"隣"#, r#"隣"#),
        (r#"鱗"#, r#"鱗"#, r#"鱗"#, r#"鱗"#, r#"鱗"#),
        (r#"麟"#, r#"麟"#, r#"麟"#, r#"麟"#, r#"麟"#),
        (r#"林"#, r#"林"#, r#"林"#, r#"林"#, r#"林"#),
        (r#"淋"#, r#"淋"#, r#"淋"#, r#"淋"#, r#"淋"#),
        (r#"臨"#, r#"臨"#, r#"臨"#, r#"臨"#, r#"臨"#),
        (r#"立"#, r#"立"#, r#"立"#, r#"立"#, r#"立"#),
        (r#"笠"#, r#"笠"#, r#"笠"#, r#"笠"#, r#"笠"#),
        (r#"粒"#, r#"粒"#, r#"粒"#, r#"粒"#, r#"粒"#),
        (r#"狀"#, r#"狀"#, r#"狀"#, r#"狀"#, r#"狀"#),
        (r#"炙"#, r#"炙"#, r#"炙"#, r#"炙"#, r#"炙"#),
        (r#"識"#, r#"識"#, r#"識"#, r#"識"#, r#"識"#),
        (r#"什"#, r#"什"#, r#"什"#, r#"什"#, r#"什"#),
        (r#"茶"#, r#"茶"#, r#"茶"#, r#"茶"#, r#"茶"#),
        (r#"刺"#, r#"刺"#, r#"刺"#, r#"刺"#, r#"刺"#),
        (r#"切"#, r#"切"#, r#"切"#, r#"切"#, r#"切"#),
        (r#"度"#, r#"度"#, r#"度"#, r#"度"#, r#"度"#),
        (r#"拓"#, r#"拓"#, r#"拓"#, r#"拓"#, r#"拓"#),
        (r#"糖"#, r#"糖"#, r#"糖"#, r#"糖"#, r#"糖"#),
        (r#"宅"#, r#"宅"#, r#"宅"#, r#"宅"#, r#"宅"#),
        (r#"洞"#, r#"洞"#, r#"洞"#, r#"洞"#, r#"洞"#),
        (r#"暴"#, r#"暴"#, r#"暴"#, r#"暴"#, r#"暴"#),
        (r#"輻"#, r#"輻"#, r#"輻"#, r#"輻"#, r#"輻"#),
        (r#"行"#, r#"行"#, r#"行"#, r#"行"#, r#"行"#),
        (r#"降"#, r#"降"#, r#"降"#, r#"降"#, r#"降"#),
        (r#"見"#, r#"見"#, r#"見"#, r#"見"#, r#"見"#),
        (r#"廓"#, r#"廓"#, r#"廓"#, r#"廓"#, r#"廓"#),
        (r#"兀"#, r#"兀"#, r#"兀"#, r#"兀"#, r#"兀"#),
        (r#"嗀"#, r#"嗀"#, r#"嗀"#, r#"嗀"#, r#"嗀"#),
        (r#"塚"#, r#"塚"#, r#"塚"#, r#"塚"#, r#"塚"#),
        (r#"晴"#, r#"晴"#, r#"晴"#, r#"晴"#, r#"晴"#),
        (r#"凞"#, r#"凞"#, r#"凞"#, r#"凞"#, r#"凞"#),
        (r#"猪"#, r#"猪"#, r#"猪"#, r#"猪"#, r#"猪"#),
        (r#"益"#, r#"益"#, r#"益"#, r#"益"#, r#"益"#),
        (r#"礼"#, r#"礼"#, r#"礼"#, r#"礼"#, r#"礼"#),
        (r#"神"#, r#"神"#, r#"神"#, r#"神"#, r#"神"#),
        (r#"祥"#, r#"祥"#, r#"祥"#, r#"祥"#, r#"祥"#),
        (r#"福"#, r#"福"#, r#"福"#, r#"福"#, r#"福"#),
        (r#"靖"#, r#"靖"#, r#"靖"#, r#"靖"#, r#"靖"#),
        (r#"精"#, r#"精"#, r#"精"#, r#"精"#, r#"精"#),
        (r#"羽"#, r#"羽"#, r#"羽"#, r#"羽"#, r#"羽"#),
        (r#"蘒"#, r#"蘒"#, r#"蘒"#, r#"蘒"#, r#"蘒"#),
        (r#"諸"#, r#"諸"#, r#"諸"#, r#"諸"#, r#"諸"#),
        (r#"逸"#, r#"逸"#, r#"逸"#, r#"逸"#, r#"逸"#),
        (r#"都"#, r#"都"#, r#"都"#, r#"都"#, r#"都"#),
        (r#"飯"#, r#"飯"#, r#"飯"#, r#"飯"#, r#"飯"#),
        (r#"飼"#, r#"飼"#, r#"飼"#, r#"飼"#, r#"飼"#),
        (r#"館"#, r#"館"#, r#"館"#, r#"館"#, r#"館"#),
        (r#"鶴"#, r#"鶴"#, r#"鶴"#, r#"鶴"#, r#"鶴"#),
        (r#"郞"#, r#"郞"#, r#"郞"#, r#"郞"#, r#"郞"#),
        (r#"隷"#, r#"隷"#, r#"隷"#, r#"隷"#, r#"隷"#),
        (r#"侮"#, r#"侮"#, r#"侮"#, r#"侮"#, r#"侮"#),
        (r#"僧"#, r#"僧"#, r#"僧"#, r#"僧"#, r#"僧"#),
        (r#"免"#, r#"免"#, r#"免"#, r#"免"#, r#"免"#),
        (r#"勉"#, r#"勉"#, r#"勉"#, r#"勉"#, r#"勉"#),
        (r#"勤"#, r#"勤"#, r#"勤"#, r#"勤"#, r#"勤"#),
        (r#"卑"#, r#"卑"#, r#"卑"#, r#"卑"#, r#"卑"#),
        (r#"喝"#, r#"喝"#, r#"喝"#, r#"喝"#, r#"喝"#),
        (r#"嘆"#, r#"嘆"#, r#"嘆"#, r#"嘆"#, r#"嘆"#),
        (r#"器"#, r#"器"#, r#"器"#, r#"器"#, r#"器"#),
        (r#"塀"#, r#"塀"#, r#"塀"#, r#"塀"#, r#"塀"#),
        (r#"墨"#, r#"墨"#, r#"墨"#, r#"墨"#, r#"墨"#),
        (r#"層"#, r#"層"#, r#"層"#, r#"層"#, r#"層"#),
        (r#"屮"#, r#"屮"#, r#"屮"#, r#"屮"#, r#"屮"#),
        (r#"悔"#, r#"悔"#, r#"悔"#, r#"悔"#, r#"悔"#),
        (r#"慨"#, r#"慨"#, r#"慨"#, r#"慨"#, r#"慨"#),
        (r#"憎"#, r#"憎"#, r#"憎"#, r#"憎"#, r#"憎"#),
        (r#"懲"#, r#"懲"#, r#"懲"#, r#"懲"#, r#"懲"#),
        (r#"敏"#, r#"敏"#, r#"敏"#, r#"敏"#, r#"敏"#),
        (r#"既"#, r#"既"#, r#"既"#, r#"既"#, r#"既"#),
        (r#"暑"#, r#"暑"#, r#"暑"#, r#"暑"#, r#"暑"#),
        (r#"梅"#, r#"梅"#, r#"梅"#, r#"梅"#, r#"梅"#),
        (r#"海"#, r#"海"#, r#"海"#, r#"海"#, r#"海"#),
        (r#"渚"#, r#"渚"#, r#"渚"#, r#"渚"#, r#"渚"#),
        (r#"漢"#, r#"漢"#, r#"漢"#, r#"漢"#, r#"漢"#),
        (r#"煮"#, r#"煮"#, r#"煮"#, r#"煮"#, r#"煮"#),
        (r#"爫"#, r#"爫"#, r#"爫"#, r#"爫"#, r#"爫"#),
        (r#"琢"#, r#"琢"#, r#"琢"#, r#"琢"#, r#"琢"#),
        (r#"碑"#, r#"碑"#, r#"碑"#, r#"碑"#, r#"碑"#),
        (r#"社"#, r#"社"#, r#"社"#, r#"社"#, r#"社"#),
        (r#"祉"#, r#"祉"#, r#"祉"#, r#"祉"#, r#"祉"#),
        (r#"祈"#, r#"祈"#, r#"祈"#, r#"祈"#, r#"祈"#),
        (r#"祐"#, r#"祐"#, r#"祐"#, r#"祐"#, r#"祐"#),
        (r#"祖"#, r#"祖"#, r#"祖"#, r#"祖"#, r#"祖"#),
        (r#"祝"#, r#"祝"#, r#"祝"#, r#"祝"#, r#"祝"#),
        (r#"禍"#, r#"禍"#, r#"禍"#, r#"禍"#, r#"禍"#),
        (r#"禎"#, r#"禎"#, r#"禎"#, r#"禎"#, r#"禎"#),
        (r#"穀"#, r#"穀"#, r#"穀"#, r#"穀"#, r#"穀"#),
        (r#"突"#, r#"突"#, r#"突"#, r#"突"#, r#"突"#),
        (r#"節"#, r#"節"#, r#"節"#, r#"節"#, r#"節"#),
        (r#"練"#, r#"練"#, r#"練"#, r#"練"#, r#"練"#),
        (r#"縉"#, r#"縉"#, r#"縉"#, r#"縉"#, r#"縉"#),
        (r#"繁"#, r#"繁"#, r#"繁"#, r#"繁"#, r#"繁"#),
        (r#"署"#, r#"署"#, r#"署"#, r#"署"#, r#"署"#),
        (r#"者"#, r#"者"#, r#"者"#, r#"者"#, r#"者"#),
        (r#"臭"#, r#"臭"#, r#"臭"#, r#"臭"#, r#"臭"#),
        (r#"艹"#, r#"艹"#, r#"艹"#, r#"艹"#, r#"艹"#),
        (r#"艹"#, r#"艹"#, r#"艹"#, r#"艹"#, r#"艹"#),
        (r#"著"#, r#"著"#, r#"著"#, r#"著"#, r#"著"#),
        (r#"褐"#, r#"褐"#, r#"褐"#, r#"褐"#, r#"褐"#),
        (r#"視"#, r#"視"#, r#"視"#, r#"視"#, r#"視"#),
        (r#"謁"#, r#"謁"#, r#"謁"#, r#"謁"#, r#"謁"#),
        (r#"謹"#, r#"謹"#, r#"謹"#, r#"謹"#, r#"謹"#),
        (r#"賓"#, r#"賓"#, r#"賓"#, r#"賓"#, r#"賓"#),
        (r#"贈"#, r#"贈"#, r#"贈"#, r#"贈"#, r#"贈"#),
        (r#"辶"#, r#"辶"#, r#"辶"#, r#"辶"#, r#"辶"#),
        (r#"逸"#, r#"逸"#, r#"逸"#, r#"逸"#, r#"逸"#),
        (r#"難"#, r#"難"#, r#"難"#, r#"難"#, r#"難"#),
        (r#"響"#, r#"響"#, r#"響"#, r#"響"#, r#"響"#),
        (r#"頻"#, r#"頻"#, r#"頻"#, r#"頻"#, r#"頻"#),
        (r#"恵"#, r#"恵"#, r#"恵"#, r#"恵"#, r#"恵"#),
        (r#"𤋮"#, r#"𤋮"#, r#"𤋮"#, r#"𤋮"#, r#"𤋮"#),
        (r#"舘"#, r#"舘"#, r#"舘"#, r#"舘"#, r#"舘"#),
        (r#"並"#, r#"並"#, r#"並"#, r#"並"#, r#"並"#),
        (r#"况"#, r#"况"#, r#"况"#, r#"况"#, r#"况"#),
        (r#"全"#, r#"全"#, r#"全"#, r#"全"#, r#"全"#),
        (r#"侀"#, r#"侀"#, r#"侀"#, r#"侀"#, r#"侀"#),
        (r#"充"#, r#"充"#, r#"充"#, r#"充"#, r#"充"#),
        (r#"冀"#, r#"冀"#, r#"冀"#, r#"冀"#, r#"冀"#),
        (r#"勇"#, r#"勇"#, r#"勇"#, r#"勇"#, r#"勇"#),
        (r#"勺"#, r#"勺"#, r#"勺"#, r#"勺"#, r#"勺"#),
        (r#"喝"#, r#"喝"#, r#"喝"#, r#"喝"#, r#"喝"#),
        (r#"啕"#, r#"啕"#, r#"啕"#, r#"啕"#, r#"啕"#),
        (r#"喙"#, r#"喙"#, r#"喙"#, r#"喙"#, r#"喙"#),
        (r#"嗢"#, r#"嗢"#, r#"嗢"#, r#"嗢"#, r#"嗢"#),
        (r#"塚"#, r#"塚"#, r#"塚"#, r#"塚"#, r#"塚"#),
        (r#"墳"#, r#"墳"#, r#"墳"#, r#"墳"#, r#"墳"#),
        (r#"奄"#, r#"奄"#, r#"奄"#, r#"奄"#, r#"奄"#),
        (r#"奔"#, r#"奔"#, r#"奔"#, r#"奔"#, r#"奔"#),
        (r#"婢"#, r#"婢"#, r#"婢"#, r#"婢"#, r#"婢"#),
        (r#"嬨"#, r#"嬨"#, r#"嬨"#, r#"嬨"#, r#"嬨"#),
        (r#"廒"#, r#"廒"#, r#"廒"#, r#"廒"#, r#"廒"#),
        (r#"廙"#, r#"廙"#, r#"廙"#, r#"廙"#, r#"廙"#),
        (r#"彩"#, r#"彩"#, r#"彩"#, r#"彩"#, r#"彩"#),
        (r#"徭"#, r#"徭"#, r#"徭"#, r#"徭"#, r#"徭"#),
        (r#"惘"#, r#"惘"#, r#"惘"#, r#"惘"#, r#"惘"#),
        (r#"慎"#, r#"慎"#, r#"慎"#, r#"慎"#, r#"慎"#),
        (r#"愈"#, r#"愈"#, r#"愈"#, r#"愈"#, r#"愈"#),
        (r#"憎"#, r#"憎"#, r#"憎"#, r#"憎"#, r#"憎"#),
        (r#"慠"#, r#"慠"#, r#"慠"#, r#"慠"#, r#"慠"#),
        (r#"懲"#, r#"懲"#, r#"懲"#, r#"懲"#, r#"懲"#),
        (r#"戴"#, r#"戴"#, r#"戴"#, r#"戴"#, r#"戴"#),
        (r#"揄"#, r#"揄"#, r#"揄"#, r#"揄"#, r#"揄"#),
        (r#"搜"#, r#"搜"#, r#"搜"#, r#"搜"#, r#"搜"#),
        (r#"摒"#, r#"摒"#, r#"摒"#, r#"摒"#, r#"摒"#),
        (r#"敖"#, r#"敖"#, r#"敖"#, r#"敖"#, r#"敖"#),
        (r#"晴"#, r#"晴"#, r#"晴"#, r#"晴"#, r#"晴"#),
        (r#"朗"#, r#"朗"#, r#"朗"#, r#"朗"#, r#"朗"#),
        (r#"望"#, r#"望"#, r#"望"#, r#"望"#, r#"望"#),
        (r#"杖"#, r#"杖"#, r#"杖"#, r#"杖"#, r#"杖"#),
        (r#"歹"#, r#"歹"#, r#"歹"#, r#"歹"#, r#"歹"#),
        (r#"殺"#, r#"殺"#, r#"殺"#, r#"殺"#, r#"殺"#),
        (r#"流"#, r#"流"#, r#"流"#, r#"流"#, r#"流"#),
        (r#"滛"#, r#"滛"#, r#"滛"#, r#"滛"#, r#"滛"#),
        (r#"滋"#, r#"滋"#, r#"滋"#, r#"滋"#, r#"滋"#),
        (r#"漢"#, r#"漢"#, r#"漢"#, r#"漢"#, r#"漢"#),
        (r#"瀞"#, r#"瀞"#, r#"瀞"#, r#"瀞"#, r#"瀞"#),
        (r#"煮"#, r#"煮"#, r#"煮"#, r#"煮"#, r#"煮"#),
        (r#"瞧"#, r#"瞧"#, r#"瞧"#, r#"瞧"#, r#"瞧"#),
        (r#"爵"#, r#"爵"#, r#"爵"#, r#"爵"#, r#"爵"#),
        (r#"犯"#, r#"犯"#, r#"犯"#, r#"犯"#, r#"犯"#),
        (r#"猪"#, r#"猪"#, r#"猪"#, r#"猪"#, r#"猪"#),
        (r#"瑱"#, r#"瑱"#, r#"瑱"#, r#"瑱"#, r#"瑱"#),
        (r#"甆"#, r#"甆"#, r#"甆"#, r#"甆"#, r#"甆"#),
        (r#"画"#, r#"画"#, r#"画"#, r#"画"#, r#"画"#),
        (r#"瘝"#, r#"瘝"#, r#"瘝"#, r#"瘝"#, r#"瘝"#),
        (r#"瘟"#, r#"瘟"#, r#"瘟"#, r#"瘟"#, r#"瘟"#),
        (r#"益"#, r#"益"#, r#"益"#, r#"益"#, r#"益"#),
        (r#"盛"#, r#"盛"#, r#"盛"#, r#"盛"#, r#"盛"#),
        (r#"直"#, r#"直"#, r#"直"#, r#"直"#, r#"直"#),
        (r#"睊"#, r#"睊"#, r#"睊"#, r#"睊"#, r#"睊"#),
        (r#"着"#, r#"着"#, r#"着"#, r#"着"#, r#"着"#),
        (r#"磌"#, r#"磌"#, r#"磌"#, r#"磌"#, r#"磌"#),
        (r#"窱"#, r#"窱"#, r#"窱"#, r#"窱"#, r#"窱"#),
        (r#"節"#, r#"節"#, r#"節"#, r#"節"#, r#"節"#),
        (r#"类"#, r#"类"#, r#"类"#, r#"类"#, r#"类"#),
        (r#"絛"#, r#"絛"#, r#"絛"#, r#"絛"#, r#"絛"#),
        (r#"練"#, r#"練"#, r#"練"#, r#"練"#, r#"練"#),
        (r#"缾"#, r#"缾"#, r#"缾"#, r#"缾"#, r#"缾"#),
        (r#"者"#, r#"者"#, r#"者"#, r#"者"#, r#"者"#),
        (r#"荒"#, r#"荒"#, r#"荒"#, r#"荒"#, r#"荒"#),
        (r#"華"#, r#"華"#, r#"華"#, r#"華"#, r#"華"#),
        (r#"蝹"#, r#"蝹"#, r#"蝹"#, r#"蝹"#, r#"蝹"#),
        (r#"襁"#, r#"襁"#, r#"襁"#, r#"襁"#, r#"襁"#),
        (r#"覆"#, r#"覆"#, r#"覆"#, r#"覆"#, r#"覆"#),
        (r#"視"#, r#"視"#, r#"視"#, r#"視"#, r#"視"#),
        (r#"調"#, r#"調"#, r#"調"#, r#"調"#, r#"調"#),
        (r#"諸"#, r#"諸"#, r#"諸"#, r#"諸"#, r#"諸"#),
        (r#"請"#, r#"請"#, r#"請"#, r#"請"#, r#"請"#),
        (r#"謁"#, r#"謁"#, r#"謁"#, r#"謁"#, r#"謁"#),
        (r#"諾"#, r#"諾"#, r#"諾"#, r#"諾"#, r#"諾"#),
        (r#"諭"#, r#"諭"#, r#"諭"#, r#"諭"#, r#"諭"#),
        (r#"謹"#, r#"謹"#, r#"謹"#, r#"謹"#, r#"謹"#),
        (r#"變"#, r#"變"#, r#"變"#, r#"變"#, r#"變"#),
        (r#"贈"#, r#"贈"#, r#"贈"#, r#"贈"#, r#"贈"#),
        (r#"輸"#, r#"輸"#, r#"輸"#, r#"輸"#, r#"輸"#),
        (r#"遲"#, r#"遲"#, r#"遲"#, r#"遲"#, r#"遲"#),
        (r#"醙"#, r#"醙"#, r#"醙"#, r#"醙"#, r#"醙"#),
        (r#"鉶"#, r#"鉶"#, r#"鉶"#, r#"鉶"#, r#"鉶"#),
        (r#"陼"#, r#"陼"#, r#"陼"#, r#"陼"#, r#"陼"#),
        (r#"難"#, r#"難"#, r#"難"#, r#"難"#, r#"難"#),
        (r#"靖"#, r#"靖"#, r#"靖"#, r#"靖"#, r#"靖"#),
        (r#"韛"#, r#"韛"#, r#"韛"#, r#"韛"#, r#"韛"#),
        (r#"響"#, r#"響"#, r#"響"#, r#"響"#, r#"響"#),
        (r#"頋"#, r#"頋"#, r#"頋"#, r#"頋"#, r#"頋"#),
        (r#"頻"#, r#"頻"#, r#"頻"#, r#"頻"#, r#"頻"#),
        (r#"鬒"#, r#"鬒"#, r#"鬒"#, r#"鬒"#, r#"鬒"#),
        (r#"龜"#, r#"龜"#, r#"龜"#, r#"龜"#, r#"龜"#),
        (r#"𢡊"#, r#"𢡊"#, r#"𢡊"#, r#"𢡊"#, r#"𢡊"#),
        (r#"𢡄"#, r#"𢡄"#, r#"𢡄"#, r#"𢡄"#, r#"𢡄"#),
        (r#"𣏕"#, r#"𣏕"#, r#"𣏕"#, r#"𣏕"#, r#"𣏕"#),
        (r#"㮝"#, r#"㮝"#, r#"㮝"#, r#"㮝"#, r#"㮝"#),
        (r#"䀘"#, r#"䀘"#, r#"䀘"#, r#"䀘"#, r#"䀘"#),
        (r#"䀹"#, r#"䀹"#, r#"䀹"#, r#"䀹"#, r#"䀹"#),
        (r#"𥉉"#, r#"𥉉"#, r#"𥉉"#, r#"𥉉"#, r#"𥉉"#),
        (r#"𥳐"#, r#"𥳐"#, r#"𥳐"#, r#"𥳐"#, r#"𥳐"#),
        (r#"𧻓"#, r#"𧻓"#, r#"𧻓"#, r#"𧻓"#, r#"𧻓"#),
        (r#"齃"#, r#"齃"#, r#"齃"#, r#"齃"#, r#"齃"#),
        (r#"龎"#, r#"龎"#, r#"龎"#, r#"龎"#, r#"龎"#),
        (r#"ﬀ"#, r#"ﬀ"#, r#"ﬀ"#, r#"ff"#, r#"ff"#),
        (r#"ﬁ"#, r#"ﬁ"#, r#"ﬁ"#, r#"fi"#, r#"fi"#),
        (r#"ﬂ"#, r#"ﬂ"#, r#"ﬂ"#, r#"fl"#, r#"fl"#),
        (r#"ﬃ"#, r#"ﬃ"#, r#"ﬃ"#, r#"ffi"#, r#"ffi"#),
        (r#"ﬄ"#, r#"ﬄ"#, r#"ﬄ"#, r#"ffl"#, r#"ffl"#),
        (r#"ﬅ"#, r#"ﬅ"#, r#"ﬅ"#, r#"st"#, r#"st"#),
        (r#"ﬆ"#, r#"ﬆ"#, r#"ﬆ"#, r#"st"#, r#"st"#),
        (r#"ﬓ"#, r#"ﬓ"#, r#"ﬓ"#, r#"մն"#, r#"մն"#),
        (r#"ﬔ"#, r#"ﬔ"#, r#"ﬔ"#, r#"մե"#, r#"մե"#),
        (r#"ﬕ"#, r#"ﬕ"#, r#"ﬕ"#, r#"մի"#, r#"մի"#),
        (r#"ﬖ"#, r#"ﬖ"#, r#"ﬖ"#, r#"վն"#, r#"վն"#),
        (r#"ﬗ"#, r#"ﬗ"#, r#"ﬗ"#, r#"մխ"#, r#"մխ"#),
        (r#"יִ"#, r#"יִ"#, r#"יִ"#, r#"יִ"#, r#"יִ"#),
        (r#"ײַ"#, r#"ײַ"#, r#"ײַ"#, r#"ײַ"#, r#"ײַ"#),
        (r#"ﬠ"#, r#"ﬠ"#, r#"ﬠ"#, r#"ע"#, r#"ע"#),
        (r#"ﬡ"#, r#"ﬡ"#, r#"ﬡ"#, r#"א"#, r#"א"#),
        (r#"ﬢ"#, r#"ﬢ"#, r#"ﬢ"#, r#"ד"#, r#"ד"#),
        (r#"ﬣ"#, r#"ﬣ"#, r#"ﬣ"#, r#"ה"#, r#"ה"#),
        (r#"ﬤ"#, r#"ﬤ"#, r#"ﬤ"#, r#"כ"#, r#"כ"#),
        (r#"ﬥ"#, r#"ﬥ"#, r#"ﬥ"#, r#"ל"#, r#"ל"#),
        (r#"ﬦ"#, r#"ﬦ"#, r#"ﬦ"#, r#"ם"#, r#"ם"#),
        (r#"ﬧ"#, r#"ﬧ"#, r#"ﬧ"#, r#"ר"#, r#"ר"#),
        (r#"ﬨ"#, r#"ﬨ"#, r#"ﬨ"#, r#"ת"#, r#"ת"#),
        (r#"﬩"#, r#"﬩"#, r#"﬩"#, r#"+"#, r#"+"#),
        (r#"שׁ"#, r#"שׁ"#, r#"שׁ"#, r#"שׁ"#, r#"שׁ"#),
        (r#"שׂ"#, r#"שׂ"#, r#"שׂ"#, r#"שׂ"#, r#"שׂ"#),
        (r#"שּׁ"#, r#"שּׁ"#, r#"שּׁ"#, r#"שּׁ"#, r#"שּׁ"#),
        (r#"שּׂ"#, r#"שּׂ"#, r#"שּׂ"#, r#"שּׂ"#, r#"שּׂ"#),
        (r#"אַ"#, r#"אַ"#, r#"אַ"#, r#"אַ"#, r#"אַ"#),
        (r#"אָ"#, r#"אָ"#, r#"אָ"#, r#"אָ"#, r#"אָ"#),
        (r#"אּ"#, r#"אּ"#, r#"אּ"#, r#"אּ"#, r#"אּ"#),
        (r#"בּ"#, r#"בּ"#, r#"בּ"#, r#"בּ"#, r#"בּ"#),
        (r#"גּ"#, r#"גּ"#, r#"גּ"#, r#"גּ"#, r#"גּ"#),
        (r#"דּ"#, r#"דּ"#, r#"דּ"#, r#"דּ"#, r#"דּ"#),
        (r#"הּ"#, r#"הּ"#, r#"הּ"#, r#"הּ"#, r#"הּ"#),
        (r#"וּ"#, r#"וּ"#, r#"וּ"#, r#"וּ"#, r#"וּ"#),
        (r#"זּ"#, r#"זּ"#, r#"זּ"#, r#"זּ"#, r#"זּ"#),
        (r#"טּ"#, r#"טּ"#, r#"טּ"#, r#"טּ"#, r#"טּ"#),
        (r#"יּ"#, r#"יּ"#, r#"יּ"#, r#"יּ"#, r#"יּ"#),
        (r#"ךּ"#, r#"ךּ"#, r#"ךּ"#, r#"ךּ"#, r#"ךּ"#),
        (r#"כּ"#, r#"כּ"#, r#"כּ"#, r#"כּ"#, r#"כּ"#),
        (r#"לּ"#, r#"לּ"#, r#"לּ"#, r#"לּ"#, r#"לּ"#),
        (r#"מּ"#, r#"מּ"#, r#"מּ"#, r#"מּ"#, r#"מּ"#),
        (r#"נּ"#, r#"נּ"#, r#"נּ"#, r#"נּ"#, r#"נּ"#),
        (r#"סּ"#, r#"סּ"#, r#"סּ"#, r#"סּ"#, r#"סּ"#),
        (r#"ףּ"#, r#"ףּ"#, r#"ףּ"#, r#"ףּ"#, r#"ףּ"#),
        (r#"פּ"#, r#"פּ"#, r#"פּ"#, r#"פּ"#, r#"פּ"#),
        (r#"צּ"#, r#"צּ"#, r#"צּ"#, r#"צּ"#, r#"צּ"#),
        (r#"קּ"#, r#"קּ"#, r#"קּ"#, r#"קּ"#, r#"קּ"#),
        (r#"רּ"#, r#"רּ"#, r#"רּ"#, r#"רּ"#, r#"רּ"#),
        (r#"שּ"#, r#"שּ"#, r#"שּ"#, r#"שּ"#, r#"שּ"#),
        (r#"תּ"#, r#"תּ"#, r#"תּ"#, r#"תּ"#, r#"תּ"#),
        (r#"וֹ"#, r#"וֹ"#, r#"וֹ"#, r#"וֹ"#, r#"וֹ"#),
        (r#"בֿ"#, r#"בֿ"#, r#"בֿ"#, r#"בֿ"#, r#"בֿ"#),
        (r#"כֿ"#, r#"כֿ"#, r#"כֿ"#, r#"כֿ"#, r#"כֿ"#),
        (r#"פֿ"#, r#"פֿ"#, r#"פֿ"#, r#"פֿ"#, r#"פֿ"#),
        (r#"ﭏ"#, r#"ﭏ"#, r#"ﭏ"#, r#"אל"#, r#"אל"#),
        (r#"ﭐ"#, r#"ﭐ"#, r#"ﭐ"#, r#"ٱ"#, r#"ٱ"#),
        (r#"ﭑ"#, r#"ﭑ"#, r#"ﭑ"#, r#"ٱ"#, r#"ٱ"#),
        (r#"ﭒ"#, r#"ﭒ"#, r#"ﭒ"#, r#"ٻ"#, r#"ٻ"#),
        (r#"ﭓ"#, r#"ﭓ"#, r#"ﭓ"#, r#"ٻ"#, r#"ٻ"#),
        (r#"ﭔ"#, r#"ﭔ"#, r#"ﭔ"#, r#"ٻ"#, r#"ٻ"#),
        (r#"ﭕ"#, r#"ﭕ"#, r#"ﭕ"#, r#"ٻ"#, r#"ٻ"#),
        (r#"ﭖ"#, r#"ﭖ"#, r#"ﭖ"#, r#"پ"#, r#"پ"#),
        (r#"ﭗ"#, r#"ﭗ"#, r#"ﭗ"#, r#"پ"#, r#"پ"#),
        (r#"ﭘ"#, r#"ﭘ"#, r#"ﭘ"#, r#"پ"#, r#"پ"#),
        (r#"ﭙ"#, r#"ﭙ"#, r#"ﭙ"#, r#"پ"#, r#"پ"#),
        (r#"ﭚ"#, r#"ﭚ"#, r#"ﭚ"#, r#"ڀ"#, r#"ڀ"#),
        (r#"ﭛ"#, r#"ﭛ"#, r#"ﭛ"#, r#"ڀ"#, r#"ڀ"#),
        (r#"ﭜ"#, r#"ﭜ"#, r#"ﭜ"#, r#"ڀ"#, r#"ڀ"#),
        (r#"ﭝ"#, r#"ﭝ"#, r#"ﭝ"#, r#"ڀ"#, r#"ڀ"#),
        (r#"ﭞ"#, r#"ﭞ"#, r#"ﭞ"#, r#"ٺ"#, r#"ٺ"#),
        (r#"ﭟ"#, r#"ﭟ"#, r#"ﭟ"#, r#"ٺ"#, r#"ٺ"#),
        (r#"ﭠ"#, r#"ﭠ"#, r#"ﭠ"#, r#"ٺ"#, r#"ٺ"#),
        (r#"ﭡ"#, r#"ﭡ"#, r#"ﭡ"#, r#"ٺ"#, r#"ٺ"#),
        (r#"ﭢ"#, r#"ﭢ"#, r#"ﭢ"#, r#"ٿ"#, r#"ٿ"#),
        (r#"ﭣ"#, r#"ﭣ"#, r#"ﭣ"#, r#"ٿ"#, r#"ٿ"#),
        (r#"ﭤ"#, r#"ﭤ"#, r#"ﭤ"#, r#"ٿ"#, r#"ٿ"#),
        (r#"ﭥ"#, r#"ﭥ"#, r#"ﭥ"#, r#"ٿ"#, r#"ٿ"#),
        (r#"ﭦ"#, r#"ﭦ"#, r#"ﭦ"#, r#"ٹ"#, r#"ٹ"#),
        (r#"ﭧ"#, r#"ﭧ"#, r#"ﭧ"#, r#"ٹ"#, r#"ٹ"#),
        (r#"ﭨ"#, r#"ﭨ"#, r#"ﭨ"#, r#"ٹ"#, r#"ٹ"#),
        (r#"ﭩ"#, r#"ﭩ"#, r#"ﭩ"#, r#"ٹ"#, r#"ٹ"#),
        (r#"ﭪ"#, r#"ﭪ"#, r#"ﭪ"#, r#"ڤ"#, r#"ڤ"#),
        (r#"ﭫ"#, r#"ﭫ"#, r#"ﭫ"#, r#"ڤ"#, r#"ڤ"#),
        (r#"ﭬ"#, r#"ﭬ"#, r#"ﭬ"#, r#"ڤ"#, r#"ڤ"#),
        (r#"ﭭ"#, r#"ﭭ"#, r#"ﭭ"#, r#"ڤ"#, r#"ڤ"#),
        (r#"ﭮ"#, r#"ﭮ"#, r#"ﭮ"#, r#"ڦ"#, r#"ڦ"#),
        (r#"ﭯ"#, r#"ﭯ"#, r#"ﭯ"#, r#"ڦ"#, r#"ڦ"#),
        (r#"ﭰ"#, r#"ﭰ"#, r#"ﭰ"#, r#"ڦ"#, r#"ڦ"#),
        (r#"ﭱ"#, r#"ﭱ"#, r#"ﭱ"#, r#"ڦ"#, r#"ڦ"#),
        (r#"ﭲ"#, r#"ﭲ"#, r#"ﭲ"#, r#"ڄ"#, r#"ڄ"#),
        (r#"ﭳ"#, r#"ﭳ"#, r#"ﭳ"#, r#"ڄ"#, r#"ڄ"#),
        (r#"ﭴ"#, r#"ﭴ"#, r#"ﭴ"#, r#"ڄ"#, r#"ڄ"#),
        (r#"ﭵ"#, r#"ﭵ"#, r#"ﭵ"#, r#"ڄ"#, r#"ڄ"#),
        (r#"ﭶ"#, r#"ﭶ"#, r#"ﭶ"#, r#"ڃ"#, r#"ڃ"#),
        (r#"ﭷ"#, r#"ﭷ"#, r#"ﭷ"#, r#"ڃ"#, r#"ڃ"#),
        (r#"ﭸ"#, r#"ﭸ"#, r#"ﭸ"#, r#"ڃ"#, r#"ڃ"#),
        (r#"ﭹ"#, r#"ﭹ"#, r#"ﭹ"#, r#"ڃ"#, r#"ڃ"#),
        (r#"ﭺ"#, r#"ﭺ"#, r#"ﭺ"#, r#"چ"#, r#"چ"#),
        (r#"ﭻ"#, r#"ﭻ"#, r#"ﭻ"#, r#"چ"#, r#"چ"#),
        (r#"ﭼ"#, r#"ﭼ"#, r#"ﭼ"#, r#"چ"#, r#"چ"#),
        (r#"ﭽ"#, r#"ﭽ"#, r#"ﭽ"#, r#"چ"#, r#"چ"#),
        (r#"ﭾ"#, r#"ﭾ"#, r#"ﭾ"#, r#"ڇ"#, r#"ڇ"#),
        (r#"ﭿ"#, r#"ﭿ"#, r#"ﭿ"#, r#"ڇ"#, r#"ڇ"#),
        (r#"ﮀ"#, r#"ﮀ"#, r#"ﮀ"#, r#"ڇ"#, r#"ڇ"#),
        (r#"ﮁ"#, r#"ﮁ"#, r#"ﮁ"#, r#"ڇ"#, r#"ڇ"#),
        (r#"ﮂ"#, r#"ﮂ"#, r#"ﮂ"#, r#"ڍ"#, r#"ڍ"#),
        (r#"ﮃ"#, r#"ﮃ"#, r#"ﮃ"#, r#"ڍ"#, r#"ڍ"#),
        (r#"ﮄ"#, r#"ﮄ"#, r#"ﮄ"#, r#"ڌ"#, r#"ڌ"#),
        (r#"ﮅ"#, r#"ﮅ"#, r#"ﮅ"#, r#"ڌ"#, r#"ڌ"#),
        (r#"ﮆ"#, r#"ﮆ"#, r#"ﮆ"#, r#"ڎ"#, r#"ڎ"#),
        (r#"ﮇ"#, r#"ﮇ"#, r#"ﮇ"#, r#"ڎ"#, r#"ڎ"#),
        (r#"ﮈ"#, r#"ﮈ"#, r#"ﮈ"#, r#"ڈ"#, r#"ڈ"#),
        (r#"ﮉ"#, r#"ﮉ"#, r#"ﮉ"#, r#"ڈ"#, r#"ڈ"#),
        (r#"ﮊ"#, r#"ﮊ"#, r#"ﮊ"#, r#"ژ"#, r#"ژ"#),
        (r#"ﮋ"#, r#"ﮋ"#, r#"ﮋ"#, r#"ژ"#, r#"ژ"#),
        (r#"ﮌ"#, r#"ﮌ"#, r#"ﮌ"#, r#"ڑ"#, r#"ڑ"#),
        (r#"ﮍ"#, r#"ﮍ"#, r#"ﮍ"#, r#"ڑ"#, r#"ڑ"#),
        (r#"ﮎ"#, r#"ﮎ"#, r#"ﮎ"#, r#"ک"#, r#"ک"#),
        (r#"ﮏ"#, r#"ﮏ"#, r#"ﮏ"#, r#"ک"#, r#"ک"#),
        (r#"ﮐ"#, r#"ﮐ"#, r#"ﮐ"#, r#"ک"#, r#"ک"#),
        (r#"ﮑ"#, r#"ﮑ"#, r#"ﮑ"#, r#"ک"#, r#"ک"#),
        (r#"ﮒ"#, r#"ﮒ"#, r#"ﮒ"#, r#"گ"#, r#"گ"#),
        (r#"ﮓ"#, r#"ﮓ"#, r#"ﮓ"#, r#"گ"#, r#"گ"#),
        (r#"ﮔ"#, r#"ﮔ"#, r#"ﮔ"#, r#"گ"#, r#"گ"#),
        (r#"ﮕ"#, r#"ﮕ"#, r#"ﮕ"#, r#"گ"#, r#"گ"#),
        (r#"ﮖ"#, r#"ﮖ"#, r#"ﮖ"#, r#"ڳ"#, r#"ڳ"#),
        (r#"ﮗ"#, r#"ﮗ"#, r#"ﮗ"#, r#"ڳ"#, r#"ڳ"#),
        (r#"ﮘ"#, r#"ﮘ"#, r#"ﮘ"#, r#"ڳ"#, r#"ڳ"#),
        (r#"ﮙ"#, r#"ﮙ"#, r#"ﮙ"#, r#"ڳ"#, r#"ڳ"#),
        (r#"ﮚ"#, r#"ﮚ"#, r#"ﮚ"#, r#"ڱ"#, r#"ڱ"#),
        (r#"ﮛ"#, r#"ﮛ"#, r#"ﮛ"#, r#"ڱ"#, r#"ڱ"#),
        (r#"ﮜ"#, r#"ﮜ"#, r#"ﮜ"#, r#"ڱ"#, r#"ڱ"#),
        (r#"ﮝ"#, r#"ﮝ"#, r#"ﮝ"#, r#"ڱ"#, r#"ڱ"#),
        (r#"ﮞ"#, r#"ﮞ"#, r#"ﮞ"#, r#"ں"#, r#"ں"#),
        (r#"ﮟ"#, r#"ﮟ"#, r#"ﮟ"#, r#"ں"#, r#"ں"#),
        (r#"ﮠ"#, r#"ﮠ"#, r#"ﮠ"#, r#"ڻ"#, r#"ڻ"#),
        (r#"ﮡ"#, r#"ﮡ"#, r#"ﮡ"#, r#"ڻ"#, r#"ڻ"#),
        (r#"ﮢ"#, r#"ﮢ"#, r#"ﮢ"#, r#"ڻ"#, r#"ڻ"#),
        (r#"ﮣ"#, r#"ﮣ"#, r#"ﮣ"#, r#"ڻ"#, r#"ڻ"#),
        (r#"ﮤ"#, r#"ﮤ"#, r#"ﮤ"#, r#"ۀ"#, r#"ۀ"#),
        (r#"ﮥ"#, r#"ﮥ"#, r#"ﮥ"#, r#"ۀ"#, r#"ۀ"#),
        (r#"ﮦ"#, r#"ﮦ"#, r#"ﮦ"#, r#"ہ"#, r#"ہ"#),
        (r#"ﮧ"#, r#"ﮧ"#, r#"ﮧ"#, r#"ہ"#, r#"ہ"#),
        (r#"ﮨ"#, r#"ﮨ"#, r#"ﮨ"#, r#"ہ"#, r#"ہ"#),
        (r#"ﮩ"#, r#"ﮩ"#, r#"ﮩ"#, r#"ہ"#, r#"ہ"#),
        (r#"ﮪ"#, r#"ﮪ"#, r#"ﮪ"#, r#"ھ"#, r#"ھ"#),
        (r#"ﮫ"#, r#"ﮫ"#, r#"ﮫ"#, r#"ھ"#, r#"ھ"#),
        (r#"ﮬ"#, r#"ﮬ"#, r#"ﮬ"#, r#"ھ"#, r#"ھ"#),
        (r#"ﮭ"#, r#"ﮭ"#, r#"ﮭ"#, r#"ھ"#, r#"ھ"#),
        (r#"ﮮ"#, r#"ﮮ"#, r#"ﮮ"#, r#"ے"#, r#"ے"#),
        (r#"ﮯ"#, r#"ﮯ"#, r#"ﮯ"#, r#"ے"#, r#"ے"#),
        (r#"ﮰ"#, r#"ﮰ"#, r#"ﮰ"#, r#"ۓ"#, r#"ۓ"#),
        (r#"ﮱ"#, r#"ﮱ"#, r#"ﮱ"#, r#"ۓ"#, r#"ۓ"#),
        (r#"ﯓ"#, r#"ﯓ"#, r#"ﯓ"#, r#"ڭ"#, r#"ڭ"#),
        (r#"ﯔ"#, r#"ﯔ"#, r#"ﯔ"#, r#"ڭ"#, r#"ڭ"#),
        (r#"ﯕ"#, r#"ﯕ"#, r#"ﯕ"#, r#"ڭ"#, r#"ڭ"#),
        (r#"ﯖ"#, r#"ﯖ"#, r#"ﯖ"#, r#"ڭ"#, r#"ڭ"#),
        (r#"ﯗ"#, r#"ﯗ"#, r#"ﯗ"#, r#"ۇ"#, r#"ۇ"#),
        (r#"ﯘ"#, r#"ﯘ"#, r#"ﯘ"#, r#"ۇ"#, r#"ۇ"#),
        (r#"ﯙ"#, r#"ﯙ"#, r#"ﯙ"#, r#"ۆ"#, r#"ۆ"#),
        (r#"ﯚ"#, r#"ﯚ"#, r#"ﯚ"#, r#"ۆ"#, r#"ۆ"#),
        (r#"ﯛ"#, r#"ﯛ"#, r#"ﯛ"#, r#"ۈ"#, r#"ۈ"#),
        (r#"ﯜ"#, r#"ﯜ"#, r#"ﯜ"#, r#"ۈ"#, r#"ۈ"#),
        (r#"ﯝ"#, r#"ﯝ"#, r#"ﯝ"#, r#"ۇٴ"#, r#"ۇٴ"#),
        (r#"ﯞ"#, r#"ﯞ"#, r#"ﯞ"#, r#"ۋ"#, r#"ۋ"#),
        (r#"ﯟ"#, r#"ﯟ"#, r#"ﯟ"#, r#"ۋ"#, r#"ۋ"#),
        (r#"ﯠ"#, r#"ﯠ"#, r#"ﯠ"#, r#"ۅ"#, r#"ۅ"#),
        (r#"ﯡ"#, r#"ﯡ"#, r#"ﯡ"#, r#"ۅ"#, r#"ۅ"#),
        (r#"ﯢ"#, r#"ﯢ"#, r#"ﯢ"#, r#"ۉ"#, r#"ۉ"#),
        (r#"ﯣ"#, r#"ﯣ"#, r#"ﯣ"#, r#"ۉ"#, r#"ۉ"#),
        (r#"ﯤ"#, r#"ﯤ"#, r#"ﯤ"#, r#"ې"#, r#"ې"#),
        (r#"ﯥ"#, r#"ﯥ"#, r#"ﯥ"#, r#"ې"#, r#"ې"#),
        (r#"ﯦ"#, r#"ﯦ"#, r#"ﯦ"#, r#"ې"#, r#"ې"#),
        (r#"ﯧ"#, r#"ﯧ"#, r#"ﯧ"#, r#"ې"#, r#"ې"#),
        (r#"ﯨ"#, r#"ﯨ"#, r#"ﯨ"#, r#"ى"#, r#"ى"#),
        (r#"ﯩ"#, r#"ﯩ"#, r#"ﯩ"#, r#"ى"#, r#"ى"#),
        (r#"ﯪ"#, r#"ﯪ"#, r#"ﯪ"#, r#"ئا"#, r#"ئا"#),
        (r#"ﯫ"#, r#"ﯫ"#, r#"ﯫ"#, r#"ئا"#, r#"ئا"#),
        (r#"ﯬ"#, r#"ﯬ"#, r#"ﯬ"#, r#"ئە"#, r#"ئە"#),
        (r#"ﯭ"#, r#"ﯭ"#, r#"ﯭ"#, r#"ئە"#, r#"ئە"#),
        (r#"ﯮ"#, r#"ﯮ"#, r#"ﯮ"#, r#"ئو"#, r#"ئو"#),
        (r#"ﯯ"#, r#"ﯯ"#, r#"ﯯ"#, r#"ئو"#, r#"ئو"#),
        (r#"ﯰ"#, r#"ﯰ"#, r#"ﯰ"#, r#"ئۇ"#, r#"ئۇ"#),
        (r#"ﯱ"#, r#"ﯱ"#, r#"ﯱ"#, r#"ئۇ"#, r#"ئۇ"#),
        (r#"ﯲ"#, r#"ﯲ"#, r#"ﯲ"#, r#"ئۆ"#, r#"ئۆ"#),
        (r#"ﯳ"#, r#"ﯳ"#, r#"ﯳ"#, r#"ئۆ"#, r#"ئۆ"#),
        (r#"ﯴ"#, r#"ﯴ"#, r#"ﯴ"#, r#"ئۈ"#, r#"ئۈ"#),
        (r#"ﯵ"#, r#"ﯵ"#, r#"ﯵ"#, r#"ئۈ"#, r#"ئۈ"#),
        (r#"ﯶ"#, r#"ﯶ"#, r#"ﯶ"#, r#"ئې"#, r#"ئې"#),
        (r#"ﯷ"#, r#"ﯷ"#, r#"ﯷ"#, r#"ئې"#, r#"ئې"#),
        (r#"ﯸ"#, r#"ﯸ"#, r#"ﯸ"#, r#"ئې"#, r#"ئې"#),
        (r#"ﯹ"#, r#"ﯹ"#, r#"ﯹ"#, r#"ئى"#, r#"ئى"#),
        (r#"ﯺ"#, r#"ﯺ"#, r#"ﯺ"#, r#"ئى"#, r#"ئى"#),
        (r#"ﯻ"#, r#"ﯻ"#, r#"ﯻ"#, r#"ئى"#, r#"ئى"#),
        (r#"ﯼ"#, r#"ﯼ"#, r#"ﯼ"#, r#"ی"#, r#"ی"#),
        (r#"ﯽ"#, r#"ﯽ"#, r#"ﯽ"#, r#"ی"#, r#"ی"#),
        (r#"ﯾ"#, r#"ﯾ"#, r#"ﯾ"#, r#"ی"#, r#"ی"#),
        (r#"ﯿ"#, r#"ﯿ"#, r#"ﯿ"#, r#"ی"#, r#"ی"#),
        (r#"ﰀ"#, r#"ﰀ"#, r#"ﰀ"#, r#"ئج"#, r#"ئج"#),
        (r#"ﰁ"#, r#"ﰁ"#, r#"ﰁ"#, r#"ئح"#, r#"ئح"#),
        (r#"ﰂ"#, r#"ﰂ"#, r#"ﰂ"#, r#"ئم"#, r#"ئم"#),
        (r#"ﰃ"#, r#"ﰃ"#, r#"ﰃ"#, r#"ئى"#, r#"ئى"#),
        (r#"ﰄ"#, r#"ﰄ"#, r#"ﰄ"#, r#"ئي"#, r#"ئي"#),
        (r#"ﰅ"#, r#"ﰅ"#, r#"ﰅ"#, r#"بج"#, r#"بج"#),
        (r#"ﰆ"#, r#"ﰆ"#, r#"ﰆ"#, r#"بح"#, r#"بح"#),
        (r#"ﰇ"#, r#"ﰇ"#, r#"ﰇ"#, r#"بخ"#, r#"بخ"#),
        (r#"ﰈ"#, r#"ﰈ"#, r#"ﰈ"#, r#"بم"#, r#"بم"#),
        (r#"ﰉ"#, r#"ﰉ"#, r#"ﰉ"#, r#"بى"#, r#"بى"#),
        (r#"ﰊ"#, r#"ﰊ"#, r#"ﰊ"#, r#"بي"#, r#"بي"#),
        (r#"ﰋ"#, r#"ﰋ"#, r#"ﰋ"#, r#"تج"#, r#"تج"#),
        (r#"ﰌ"#, r#"ﰌ"#, r#"ﰌ"#, r#"تح"#, r#"تح"#),
        (r#"ﰍ"#, r#"ﰍ"#, r#"ﰍ"#, r#"تخ"#, r#"تخ"#),
        (r#"ﰎ"#, r#"ﰎ"#, r#"ﰎ"#, r#"تم"#, r#"تم"#),
        (r#"ﰏ"#, r#"ﰏ"#, r#"ﰏ"#, r#"تى"#, r#"تى"#),
        (r#"ﰐ"#, r#"ﰐ"#, r#"ﰐ"#, r#"تي"#, r#"تي"#),
        (r#"ﰑ"#, r#"ﰑ"#, r#"ﰑ"#, r#"ثج"#, r#"ثج"#),
        (r#"ﰒ"#, r#"ﰒ"#, r#"ﰒ"#, r#"ثم"#, r#"ثم"#),
        (r#"ﰓ"#, r#"ﰓ"#, r#"ﰓ"#, r#"ثى"#, r#"ثى"#),
        (r#"ﰔ"#, r#"ﰔ"#, r#"ﰔ"#, r#"ثي"#, r#"ثي"#),
        (r#"ﰕ"#, r#"ﰕ"#, r#"ﰕ"#, r#"جح"#, r#"جح"#),
        (r#"ﰖ"#, r#"ﰖ"#, r#"ﰖ"#, r#"جم"#, r#"جم"#),
        (r#"ﰗ"#, r#"ﰗ"#, r#"ﰗ"#, r#"حج"#, r#"حج"#),
        (r#"ﰘ"#, r#"ﰘ"#, r#"ﰘ"#, r#"حم"#, r#"حم"#),
        (r#"ﰙ"#, r#"ﰙ"#, r#"ﰙ"#, r#"خج"#, r#"خج"#),
        (r#"ﰚ"#, r#"ﰚ"#, r#"ﰚ"#, r#"خح"#, r#"خح"#),
        (r#"ﰛ"#, r#"ﰛ"#, r#"ﰛ"#, r#"خم"#, r#"خم"#),
        (r#"ﰜ"#, r#"ﰜ"#, r#"ﰜ"#, r#"سج"#, r#"سج"#),
        (r#"ﰝ"#, r#"ﰝ"#, r#"ﰝ"#, r#"سح"#, r#"سح"#),
        (r#"ﰞ"#, r#"ﰞ"#, r#"ﰞ"#, r#"سخ"#, r#"سخ"#),
        (r#"ﰟ"#, r#"ﰟ"#, r#"ﰟ"#, r#"سم"#, r#"سم"#),
        (r#"ﰠ"#, r#"ﰠ"#, r#"ﰠ"#, r#"صح"#, r#"صح"#),
        (r#"ﰡ"#, r#"ﰡ"#, r#"ﰡ"#, r#"صم"#, r#"صم"#),
        (r#"ﰢ"#, r#"ﰢ"#, r#"ﰢ"#, r#"ضج"#, r#"ضج"#),
        (r#"ﰣ"#, r#"ﰣ"#, r#"ﰣ"#, r#"ضح"#, r#"ضح"#),
        (r#"ﰤ"#, r#"ﰤ"#, r#"ﰤ"#, r#"ضخ"#, r#"ضخ"#),
        (r#"ﰥ"#, r#"ﰥ"#, r#"ﰥ"#, r#"ضم"#, r#"ضم"#),
        (r#"ﰦ"#, r#"ﰦ"#, r#"ﰦ"#, r#"طح"#, r#"طح"#),
        (r#"ﰧ"#, r#"ﰧ"#, r#"ﰧ"#, r#"طم"#, r#"طم"#),
        (r#"ﰨ"#, r#"ﰨ"#, r#"ﰨ"#, r#"ظم"#, r#"ظم"#),
        (r#"ﰩ"#, r#"ﰩ"#, r#"ﰩ"#, r#"عج"#, r#"عج"#),
        (r#"ﰪ"#, r#"ﰪ"#, r#"ﰪ"#, r#"عم"#, r#"عم"#),
        (r#"ﰫ"#, r#"ﰫ"#, r#"ﰫ"#, r#"غج"#, r#"غج"#),
        (r#"ﰬ"#, r#"ﰬ"#, r#"ﰬ"#, r#"غم"#, r#"غم"#),
        (r#"ﰭ"#, r#"ﰭ"#, r#"ﰭ"#, r#"فج"#, r#"فج"#),
        (r#"ﰮ"#, r#"ﰮ"#, r#"ﰮ"#, r#"فح"#, r#"فح"#),
        (r#"ﰯ"#, r#"ﰯ"#, r#"ﰯ"#, r#"فخ"#, r#"فخ"#),
        (r#"ﰰ"#, r#"ﰰ"#, r#"ﰰ"#, r#"فم"#, r#"فم"#),
        (r#"ﰱ"#, r#"ﰱ"#, r#"ﰱ"#, r#"فى"#, r#"فى"#),
        (r#"ﰲ"#, r#"ﰲ"#, r#"ﰲ"#, r#"في"#, r#"في"#),
        (r#"ﰳ"#, r#"ﰳ"#, r#"ﰳ"#, r#"قح"#, r#"قح"#),
        (r#"ﰴ"#, r#"ﰴ"#, r#"ﰴ"#, r#"قم"#, r#"قم"#),
        (r#"ﰵ"#, r#"ﰵ"#, r#"ﰵ"#, r#"قى"#, r#"قى"#),
        (r#"ﰶ"#, r#"ﰶ"#, r#"ﰶ"#, r#"قي"#, r#"قي"#),
        (r#"ﰷ"#, r#"ﰷ"#, r#"ﰷ"#, r#"كا"#, r#"كا"#),
        (r#"ﰸ"#, r#"ﰸ"#, r#"ﰸ"#, r#"كج"#, r#"كج"#),
        (r#"ﰹ"#, r#"ﰹ"#, r#"ﰹ"#, r#"كح"#, r#"كح"#),
        (r#"ﰺ"#, r#"ﰺ"#, r#"ﰺ"#, r#"كخ"#, r#"كخ"#),
        (r#"ﰻ"#, r#"ﰻ"#, r#"ﰻ"#, r#"كل"#, r#"كل"#),
        (r#"ﰼ"#, r#"ﰼ"#, r#"ﰼ"#, r#"كم"#, r#"كم"#),
        (r#"ﰽ"#, r#"ﰽ"#, r#"ﰽ"#, r#"كى"#, r#"كى"#),
        (r#"ﰾ"#, r#"ﰾ"#, r#"ﰾ"#, r#"كي"#, r#"كي"#),
        (r#"ﰿ"#, r#"ﰿ"#, r#"ﰿ"#, r#"لج"#, r#"لج"#),
        (r#"ﱀ"#, r#"ﱀ"#, r#"ﱀ"#, r#"لح"#, r#"لح"#),
        (r#"ﱁ"#, r#"ﱁ"#, r#"ﱁ"#, r#"لخ"#, r#"لخ"#),
        (r#"ﱂ"#, r#"ﱂ"#, r#"ﱂ"#, r#"لم"#, r#"لم"#),
        (r#"ﱃ"#, r#"ﱃ"#, r#"ﱃ"#, r#"لى"#, r#"لى"#),
        (r#"ﱄ"#, r#"ﱄ"#, r#"ﱄ"#, r#"لي"#, r#"لي"#),
        (r#"ﱅ"#, r#"ﱅ"#, r#"ﱅ"#, r#"مج"#, r#"مج"#),
        (r#"ﱆ"#, r#"ﱆ"#, r#"ﱆ"#, r#"مح"#, r#"مح"#),
        (r#"ﱇ"#, r#"ﱇ"#, r#"ﱇ"#, r#"مخ"#, r#"مخ"#),
        (r#"ﱈ"#, r#"ﱈ"#, r#"ﱈ"#, r#"مم"#, r#"مم"#),
        (r#"ﱉ"#, r#"ﱉ"#, r#"ﱉ"#, r#"مى"#, r#"مى"#),
        (r#"ﱊ"#, r#"ﱊ"#, r#"ﱊ"#, r#"مي"#, r#"مي"#),
        (r#"ﱋ"#, r#"ﱋ"#, r#"ﱋ"#, r#"نج"#, r#"نج"#),
        (r#"ﱌ"#, r#"ﱌ"#, r#"ﱌ"#, r#"نح"#, r#"نح"#),
        (r#"ﱍ"#, r#"ﱍ"#, r#"ﱍ"#, r#"نخ"#, r#"نخ"#),
        (r#"ﱎ"#, r#"ﱎ"#, r#"ﱎ"#, r#"نم"#, r#"نم"#),
        (r#"ﱏ"#, r#"ﱏ"#, r#"ﱏ"#, r#"نى"#, r#"نى"#),
        (r#"ﱐ"#, r#"ﱐ"#, r#"ﱐ"#, r#"ني"#, r#"ني"#),
        (r#"ﱑ"#, r#"ﱑ"#, r#"ﱑ"#, r#"هج"#, r#"هج"#),
        (r#"ﱒ"#, r#"ﱒ"#, r#"ﱒ"#, r#"هم"#, r#"هم"#),
        (r#"ﱓ"#, r#"ﱓ"#, r#"ﱓ"#, r#"هى"#, r#"هى"#),
        (r#"ﱔ"#, r#"ﱔ"#, r#"ﱔ"#, r#"هي"#, r#"هي"#),
        (r#"ﱕ"#, r#"ﱕ"#, r#"ﱕ"#, r#"يج"#, r#"يج"#),
        (r#"ﱖ"#, r#"ﱖ"#, r#"ﱖ"#, r#"يح"#, r#"يح"#),
        (r#"ﱗ"#, r#"ﱗ"#, r#"ﱗ"#, r#"يخ"#, r#"يخ"#),
        (r#"ﱘ"#, r#"ﱘ"#, r#"ﱘ"#, r#"يم"#, r#"يم"#),
        (r#"ﱙ"#, r#"ﱙ"#, r#"ﱙ"#, r#"يى"#, r#"يى"#),
        (r#"ﱚ"#, r#"ﱚ"#, r#"ﱚ"#, r#"يي"#, r#"يي"#),
        (r#"ﱛ"#, r#"ﱛ"#, r#"ﱛ"#, r#"ذٰ"#, r#"ذٰ"#),
        (r#"ﱜ"#, r#"ﱜ"#, r#"ﱜ"#, r#"رٰ"#, r#"رٰ"#),
        (r#"ﱝ"#, r#"ﱝ"#, r#"ﱝ"#, r#"ىٰ"#, r#"ىٰ"#),
        (r#"ﱞ"#, r#"ﱞ"#, r#"ﱞ"#, r#" ٌّ"#, r#" ٌّ"#),
        (r#"ﱟ"#, r#"ﱟ"#, r#"ﱟ"#, r#" ٍّ"#, r#" ٍّ"#),
        (r#"ﱠ"#, r#"ﱠ"#, r#"ﱠ"#, r#" َّ"#, r#" َّ"#),
        (r#"ﱡ"#, r#"ﱡ"#, r#"ﱡ"#, r#" ُّ"#, r#" ُّ"#),
        (r#"ﱢ"#, r#"ﱢ"#, r#"ﱢ"#, r#" ِّ"#, r#" ِّ"#),
        (r#"ﱣ"#, r#"ﱣ"#, r#"ﱣ"#, r#" ّٰ"#, r#" ّٰ"#),
        (r#"ﱤ"#, r#"ﱤ"#, r#"ﱤ"#, r#"ئر"#, r#"ئر"#),
        (r#"ﱥ"#, r#"ﱥ"#, r#"ﱥ"#, r#"ئز"#, r#"ئز"#),
        (r#"ﱦ"#, r#"ﱦ"#, r#"ﱦ"#, r#"ئم"#, r#"ئم"#),
        (r#"ﱧ"#, r#"ﱧ"#, r#"ﱧ"#, r#"ئن"#, r#"ئن"#),
        (r#"ﱨ"#, r#"ﱨ"#, r#"ﱨ"#, r#"ئى"#, r#"ئى"#),
        (r#"ﱩ"#, r#"ﱩ"#, r#"ﱩ"#, r#"ئي"#, r#"ئي"#),
        (r#"ﱪ"#, r#"ﱪ"#, r#"ﱪ"#, r#"بر"#, r#"بر"#),
        (r#"ﱫ"#, r#"ﱫ"#, r#"ﱫ"#, r#"بز"#, r#"بز"#),
        (r#"ﱬ"#, r#"ﱬ"#, r#"ﱬ"#, r#"بم"#, r#"بم"#),
        (r#"ﱭ"#, r#"ﱭ"#, r#"ﱭ"#, r#"بن"#, r#"بن"#),
        (r#"ﱮ"#, r#"ﱮ"#, r#"ﱮ"#, r#"بى"#, r#"بى"#),
        (r#"ﱯ"#, r#"ﱯ"#, r#"ﱯ"#, r#"بي"#, r#"بي"#),
        (r#"ﱰ"#, r#"ﱰ"#, r#"ﱰ"#, r#"تر"#, r#"تر"#),
        (r#"ﱱ"#, r#"ﱱ"#, r#"ﱱ"#, r#"تز"#, r#"تز"#),
        (r#"ﱲ"#, r#"ﱲ"#, r#"ﱲ"#, r#"تم"#, r#"تم"#),
        (r#"ﱳ"#, r#"ﱳ"#, r#"ﱳ"#, r#"تن"#, r#"تن"#),
        (r#"ﱴ"#, r#"ﱴ"#, r#"ﱴ"#, r#"تى"#, r#"تى"#),
        (r#"ﱵ"#, r#"ﱵ"#, r#"ﱵ"#, r#"تي"#, r#"تي"#),
        (r#"ﱶ"#, r#"ﱶ"#, r#"ﱶ"#, r#"ثر"#, r#"ثر"#),
        (r#"ﱷ"#, r#"ﱷ"#, r#"ﱷ"#, r#"ثز"#, r#"ثز"#),
        (r#"ﱸ"#, r#"ﱸ"#, r#"ﱸ"#, r#"ثم"#, r#"ثم"#),
        (r#"ﱹ"#, r#"ﱹ"#, r#"ﱹ"#, r#"ثن"#, r#"ثن"#),
        (r#"ﱺ"#, r#"ﱺ"#, r#"ﱺ"#, r#"ثى"#, r#"ثى"#),
        (r#"ﱻ"#, r#"ﱻ"#, r#"ﱻ"#, r#"ثي"#, r#"ثي"#),
        (r#"ﱼ"#, r#"ﱼ"#, r#"ﱼ"#, r#"فى"#, r#"فى"#),
        (r#"ﱽ"#, r#"ﱽ"#, r#"ﱽ"#, r#"في"#, r#"في"#),
        (r#"ﱾ"#, r#"ﱾ"#, r#"ﱾ"#, r#"قى"#, r#"قى"#),
        (r#"ﱿ"#, r#"ﱿ"#, r#"ﱿ"#, r#"قي"#, r#"قي"#),
        (r#"ﲀ"#, r#"ﲀ"#, r#"ﲀ"#, r#"كا"#, r#"كا"#),
        (r#"ﲁ"#, r#"ﲁ"#, r#"ﲁ"#, r#"كل"#, r#"كل"#),
        (r#"ﲂ"#, r#"ﲂ"#, r#"ﲂ"#, r#"كم"#, r#"كم"#),
        (r#"ﲃ"#, r#"ﲃ"#, r#"ﲃ"#, r#"كى"#, r#"كى"#),
        (r#"ﲄ"#, r#"ﲄ"#, r#"ﲄ"#, r#"كي"#, r#"كي"#),
        (r#"ﲅ"#, r#"ﲅ"#, r#"ﲅ"#, r#"لم"#, r#"لم"#),
        (r#"ﲆ"#, r#"ﲆ"#, r#"ﲆ"#, r#"لى"#, r#"لى"#),
        (r#"ﲇ"#, r#"ﲇ"#, r#"ﲇ"#, r#"لي"#, r#"لي"#),
        (r#"ﲈ"#, r#"ﲈ"#, r#"ﲈ"#, r#"ما"#, r#"ما"#),
        (r#"ﲉ"#, r#"ﲉ"#, r#"ﲉ"#, r#"مم"#, r#"مم"#),
        (r#"ﲊ"#, r#"ﲊ"#, r#"ﲊ"#, r#"نر"#, r#"نر"#),
        (r#"ﲋ"#, r#"ﲋ"#, r#"ﲋ"#, r#"نز"#, r#"نز"#),
        (r#"ﲌ"#, r#"ﲌ"#, r#"ﲌ"#, r#"نم"#, r#"نم"#),
        (r#"ﲍ"#, r#"ﲍ"#, r#"ﲍ"#, r#"نن"#, r#"نن"#),
        (r#"ﲎ"#, r#"ﲎ"#, r#"ﲎ"#, r#"نى"#, r#"نى"#),
        (r#"ﲏ"#, r#"ﲏ"#, r#"ﲏ"#, r#"ني"#, r#"ني"#),
        (r#"ﲐ"#, r#"ﲐ"#, r#"ﲐ"#, r#"ىٰ"#, r#"ىٰ"#),
        (r#"ﲑ"#, r#"ﲑ"#, r#"ﲑ"#, r#"ير"#, r#"ير"#),
        (r#"ﲒ"#, r#"ﲒ"#, r#"ﲒ"#, r#"يز"#, r#"يز"#),
        (r#"ﲓ"#, r#"ﲓ"#, r#"ﲓ"#, r#"يم"#, r#"يم"#),
        (r#"ﲔ"#, r#"ﲔ"#, r#"ﲔ"#, r#"ين"#, r#"ين"#),
        (r#"ﲕ"#, r#"ﲕ"#, r#"ﲕ"#, r#"يى"#, r#"يى"#),
        (r#"ﲖ"#, r#"ﲖ"#, r#"ﲖ"#, r#"يي"#, r#"يي"#),
        (r#"ﲗ"#, r#"ﲗ"#, r#"ﲗ"#, r#"ئج"#, r#"ئج"#),
        (r#"ﲘ"#, r#"ﲘ"#, r#"ﲘ"#, r#"ئح"#, r#"ئح"#),
        (r#"ﲙ"#, r#"ﲙ"#, r#"ﲙ"#, r#"ئخ"#, r#"ئخ"#),
        (r#"ﲚ"#, r#"ﲚ"#, r#"ﲚ"#, r#"ئم"#, r#"ئم"#),
        (r#"ﲛ"#, r#"ﲛ"#, r#"ﲛ"#, r#"ئه"#, r#"ئه"#),
        (r#"ﲜ"#, r#"ﲜ"#, r#"ﲜ"#, r#"بج"#, r#"بج"#),
        (r#"ﲝ"#, r#"ﲝ"#, r#"ﲝ"#, r#"بح"#, r#"بح"#),
        (r#"ﲞ"#, r#"ﲞ"#, r#"ﲞ"#, r#"بخ"#, r#"بخ"#),
        (r#"ﲟ"#, r#"ﲟ"#, r#"ﲟ"#, r#"بم"#, r#"بم"#),
        (r#"ﲠ"#, r#"ﲠ"#, r#"ﲠ"#, r#"به"#, r#"به"#),
        (r#"ﲡ"#, r#"ﲡ"#, r#"ﲡ"#, r#"تج"#, r#"تج"#),
        (r#"ﲢ"#, r#"ﲢ"#, r#"ﲢ"#, r#"تح"#, r#"تح"#),
        (r#"ﲣ"#, r#"ﲣ"#, r#"ﲣ"#, r#"تخ"#, r#"تخ"#),
        (r#"ﲤ"#, r#"ﲤ"#, r#"ﲤ"#, r#"تم"#, r#"تم"#),
        (r#"ﲥ"#, r#"ﲥ"#, r#"ﲥ"#, r#"ته"#, r#"ته"#),
        (r#"ﲦ"#, r#"ﲦ"#, r#"ﲦ"#, r#"ثم"#, r#"ثم"#),
        (r#"ﲧ"#, r#"ﲧ"#, r#"ﲧ"#, r#"جح"#, r#"جح"#),
        (r#"ﲨ"#, r#"ﲨ"#, r#"ﲨ"#, r#"جم"#, r#"جم"#),
        (r#"ﲩ"#, r#"ﲩ"#, r#"ﲩ"#, r#"حج"#, r#"حج"#),
        (r#"ﲪ"#, r#"ﲪ"#, r#"ﲪ"#, r#"حم"#, r#"حم"#),
        (r#"ﲫ"#, r#"ﲫ"#, r#"ﲫ"#, r#"خج"#, r#"خج"#),
        (r#"ﲬ"#, r#"ﲬ"#, r#"ﲬ"#, r#"خم"#, r#"خم"#),
        (r#"ﲭ"#, r#"ﲭ"#, r#"ﲭ"#, r#"سج"#, r#"سج"#),
        (r#"ﲮ"#, r#"ﲮ"#, r#"ﲮ"#, r#"سح"#, r#"سح"#),
        (r#"ﲯ"#, r#"ﲯ"#, r#"ﲯ"#, r#"سخ"#, r#"سخ"#),
        (r#"ﲰ"#, r#"ﲰ"#, r#"ﲰ"#, r#"سم"#, r#"سم"#),
        (r#"ﲱ"#, r#"ﲱ"#, r#"ﲱ"#, r#"صح"#, r#"صح"#),
        (r#"ﲲ"#, r#"ﲲ"#, r#"ﲲ"#, r#"صخ"#, r#"صخ"#),
        (r#"ﲳ"#, r#"ﲳ"#, r#"ﲳ"#, r#"صم"#, r#"صم"#),
        (r#"ﲴ"#, r#"ﲴ"#, r#"ﲴ"#, r#"ضج"#, r#"ضج"#),
        (r#"ﲵ"#, r#"ﲵ"#, r#"ﲵ"#, r#"ضح"#, r#"ضح"#),
        (r#"ﲶ"#, r#"ﲶ"#, r#"ﲶ"#, r#"ضخ"#, r#"ضخ"#),
        (r#"ﲷ"#, r#"ﲷ"#, r#"ﲷ"#, r#"ضم"#, r#"ضم"#),
        (r#"ﲸ"#, r#"ﲸ"#, r#"ﲸ"#, r#"طح"#, r#"طح"#),
        (r#"ﲹ"#, r#"ﲹ"#, r#"ﲹ"#, r#"ظم"#, r#"ظم"#),
        (r#"ﲺ"#, r#"ﲺ"#, r#"ﲺ"#, r#"عج"#, r#"عج"#),
        (r#"ﲻ"#, r#"ﲻ"#, r#"ﲻ"#, r#"عم"#, r#"عم"#),
        (r#"ﲼ"#, r#"ﲼ"#, r#"ﲼ"#, r#"غج"#, r#"غج"#),
        (r#"ﲽ"#, r#"ﲽ"#, r#"ﲽ"#, r#"غم"#, r#"غم"#),
        (r#"ﲾ"#, r#"ﲾ"#, r#"ﲾ"#, r#"فج"#, r#"فج"#),
        (r#"ﲿ"#, r#"ﲿ"#, r#"ﲿ"#, r#"فح"#, r#"فح"#),
        (r#"ﳀ"#, r#"ﳀ"#, r#"ﳀ"#, r#"فخ"#, r#"فخ"#),
        (r#"ﳁ"#, r#"ﳁ"#, r#"ﳁ"#, r#"فم"#, r#"فم"#),
        (r#"ﳂ"#, r#"ﳂ"#, r#"ﳂ"#, r#"قح"#, r#"قح"#),
        (r#"ﳃ"#, r#"ﳃ"#, r#"ﳃ"#, r#"قم"#, r#"قم"#),
        (r#"ﳄ"#, r#"ﳄ"#, r#"ﳄ"#, r#"كج"#, r#"كج"#),
        (r#"ﳅ"#, r#"ﳅ"#, r#"ﳅ"#, r#"كح"#, r#"كح"#),
        (r#"ﳆ"#, r#"ﳆ"#, r#"ﳆ"#, r#"كخ"#, r#"كخ"#),
        (r#"ﳇ"#, r#"ﳇ"#, r#"ﳇ"#, r#"كل"#, r#"كل"#),
        (r#"ﳈ"#, r#"ﳈ"#, r#"ﳈ"#, r#"كم"#, r#"كم"#),
        (r#"ﳉ"#, r#"ﳉ"#, r#"ﳉ"#, r#"لج"#, r#"لج"#),
        (r#"ﳊ"#, r#"ﳊ"#, r#"ﳊ"#, r#"لح"#, r#"لح"#),
        (r#"ﳋ"#, r#"ﳋ"#, r#"ﳋ"#, r#"لخ"#, r#"لخ"#),
        (r#"ﳌ"#, r#"ﳌ"#, r#"ﳌ"#, r#"لم"#, r#"لم"#),
        (r#"ﳍ"#, r#"ﳍ"#, r#"ﳍ"#, r#"له"#, r#"له"#),
        (r#"ﳎ"#, r#"ﳎ"#, r#"ﳎ"#, r#"مج"#, r#"مج"#),
        (r#"ﳏ"#, r#"ﳏ"#, r#"ﳏ"#, r#"مح"#, r#"مح"#),
        (r#"ﳐ"#, r#"ﳐ"#, r#"ﳐ"#, r#"مخ"#, r#"مخ"#),
        (r#"ﳑ"#, r#"ﳑ"#, r#"ﳑ"#, r#"مم"#, r#"مم"#),
        (r#"ﳒ"#, r#"ﳒ"#, r#"ﳒ"#, r#"نج"#, r#"نج"#),
        (r#"ﳓ"#, r#"ﳓ"#, r#"ﳓ"#, r#"نح"#, r#"نح"#),
        (r#"ﳔ"#, r#"ﳔ"#, r#"ﳔ"#, r#"نخ"#, r#"نخ"#),
        (r#"ﳕ"#, r#"ﳕ"#, r#"ﳕ"#, r#"نم"#, r#"نم"#),
        (r#"ﳖ"#, r#"ﳖ"#, r#"ﳖ"#, r#"نه"#, r#"نه"#),
        (r#"ﳗ"#, r#"ﳗ"#, r#"ﳗ"#, r#"هج"#, r#"هج"#),
        (r#"ﳘ"#, r#"ﳘ"#, r#"ﳘ"#, r#"هم"#, r#"هم"#),
        (r#"ﳙ"#, r#"ﳙ"#, r#"ﳙ"#, r#"هٰ"#, r#"هٰ"#),
        (r#"ﳚ"#, r#"ﳚ"#, r#"ﳚ"#, r#"يج"#, r#"يج"#),
        (r#"ﳛ"#, r#"ﳛ"#, r#"ﳛ"#, r#"يح"#, r#"يح"#),
        (r#"ﳜ"#, r#"ﳜ"#, r#"ﳜ"#, r#"يخ"#, r#"يخ"#),
        (r#"ﳝ"#, r#"ﳝ"#, r#"ﳝ"#, r#"يم"#, r#"يم"#),
        (r#"ﳞ"#, r#"ﳞ"#, r#"ﳞ"#, r#"يه"#, r#"يه"#),
        (r#"ﳟ"#, r#"ﳟ"#, r#"ﳟ"#, r#"ئم"#, r#"ئم"#),
        (r#"ﳠ"#, r#"ﳠ"#, r#"ﳠ"#, r#"ئه"#, r#"ئه"#),
        (r#"ﳡ"#, r#"ﳡ"#, r#"ﳡ"#, r#"بم"#, r#"بم"#),
        (r#"ﳢ"#, r#"ﳢ"#, r#"ﳢ"#, r#"به"#, r#"به"#),
        (r#"ﳣ"#, r#"ﳣ"#, r#"ﳣ"#, r#"تم"#, r#"تم"#),
        (r#"ﳤ"#, r#"ﳤ"#, r#"ﳤ"#, r#"ته"#, r#"ته"#),
        (r#"ﳥ"#, r#"ﳥ"#, r#"ﳥ"#, r#"ثم"#, r#"ثم"#),
        (r#"ﳦ"#, r#"ﳦ"#, r#"ﳦ"#, r#"ثه"#, r#"ثه"#),
        (r#"ﳧ"#, r#"ﳧ"#, r#"ﳧ"#, r#"سم"#, r#"سم"#),
        (r#"ﳨ"#, r#"ﳨ"#, r#"ﳨ"#, r#"سه"#, r#"سه"#),
        (r#"ﳩ"#, r#"ﳩ"#, r#"ﳩ"#, r#"شم"#, r#"شم"#),
        (r#"ﳪ"#, r#"ﳪ"#, r#"ﳪ"#, r#"شه"#, r#"شه"#),
        (r#"ﳫ"#, r#"ﳫ"#, r#"ﳫ"#, r#"كل"#, r#"كل"#),
        (r#"ﳬ"#, r#"ﳬ"#, r#"ﳬ"#, r#"كم"#, r#"كم"#),
        (r#"ﳭ"#, r#"ﳭ"#, r#"ﳭ"#, r#"لم"#, r#"لم"#),
        (r#"ﳮ"#, r#"ﳮ"#, r#"ﳮ"#, r#"نم"#, r#"نم"#),
        (r#"ﳯ"#, r#"ﳯ"#, r#"ﳯ"#, r#"نه"#, r#"نه"#),
        (r#"ﳰ"#, r#"ﳰ"#, r#"ﳰ"#, r#"يم"#, r#"يم"#),
        (r#"ﳱ"#, r#"ﳱ"#, r#"ﳱ"#, r#"يه"#, r#"يه"#),
        (r#"ﳲ"#, r#"ﳲ"#, r#"ﳲ"#, r#"ـَّ"#, r#"ـَّ"#),
        (r#"ﳳ"#, r#"ﳳ"#, r#"ﳳ"#, r#"ـُّ"#, r#"ـُّ"#),
        (r#"ﳴ"#, r#"ﳴ"#, r#"ﳴ"#, r#"ـِّ"#, r#"ـِّ"#),
        (r#"ﳵ"#, r#"ﳵ"#, r#"ﳵ"#, r#"طى"#, r#"طى"#),
        (r#"ﳶ"#, r#"ﳶ"#, r#"ﳶ"#, r#"طي"#, r#"طي"#),
        (r#"ﳷ"#, r#"ﳷ"#, r#"ﳷ"#, r#"عى"#, r#"عى"#),
        (r#"ﳸ"#, r#"ﳸ"#, r#"ﳸ"#, r#"عي"#, r#"عي"#),
        (r#"ﳹ"#, r#"ﳹ"#, r#"ﳹ"#, r#"غى"#, r#"غى"#),
        (r#"ﳺ"#, r#"ﳺ"#, r#"ﳺ"#, r#"غي"#, r#"غي"#),
        (r#"ﳻ"#, r#"ﳻ"#, r#"ﳻ"#, r#"سى"#, r#"سى"#),
        (r#"ﳼ"#, r#"ﳼ"#, r#"ﳼ"#, r#"سي"#, r#"سي"#),
        (r#"ﳽ"#, r#"ﳽ"#, r#"ﳽ"#, r#"شى"#, r#"شى"#),
        (r#"ﳾ"#, r#"ﳾ"#, r#"ﳾ"#, r#"شي"#, r#"شي"#),
        (r#"ﳿ"#, r#"ﳿ"#, r#"ﳿ"#, r#"حى"#, r#"حى"#),
        (r#"ﴀ"#, r#"ﴀ"#, r#"ﴀ"#, r#"حي"#, r#"حي"#),
        (r#"ﴁ"#, r#"ﴁ"#, r#"ﴁ"#, r#"جى"#, r#"جى"#),
        (r#"ﴂ"#, r#"ﴂ"#, r#"ﴂ"#, r#"جي"#, r#"جي"#),
        (r#"ﴃ"#, r#"ﴃ"#, r#"ﴃ"#, r#"خى"#, r#"خى"#),
        (r#"ﴄ"#, r#"ﴄ"#, r#"ﴄ"#, r#"خي"#, r#"خي"#),
        (r#"ﴅ"#, r#"ﴅ"#, r#"ﴅ"#, r#"صى"#, r#"صى"#),
        (r#"ﴆ"#, r#"ﴆ"#, r#"ﴆ"#, r#"صي"#, r#"صي"#),
        (r#"ﴇ"#, r#"ﴇ"#, r#"ﴇ"#, r#"ضى"#, r#"ضى"#),
        (r#"ﴈ"#, r#"ﴈ"#, r#"ﴈ"#, r#"ضي"#, r#"ضي"#),
        (r#"ﴉ"#, r#"ﴉ"#, r#"ﴉ"#, r#"شج"#, r#"شج"#),
        (r#"ﴊ"#, r#"ﴊ"#, r#"ﴊ"#, r#"شح"#, r#"شح"#),
        (r#"ﴋ"#, r#"ﴋ"#, r#"ﴋ"#, r#"شخ"#, r#"شخ"#),
        (r#"ﴌ"#, r#"ﴌ"#, r#"ﴌ"#, r#"شم"#, r#"شم"#),
        (r#"ﴍ"#, r#"ﴍ"#, r#"ﴍ"#, r#"شر"#, r#"شر"#),
        (r#"ﴎ"#, r#"ﴎ"#, r#"ﴎ"#, r#"سر"#, r#"سر"#),
        (r#"ﴏ"#, r#"ﴏ"#, r#"ﴏ"#, r#"صر"#, r#"صر"#),
        (r#"ﴐ"#, r#"ﴐ"#, r#"ﴐ"#, r#"ضر"#, r#"ضر"#),
        (r#"ﴑ"#, r#"ﴑ"#, r#"ﴑ"#, r#"طى"#, r#"طى"#),
        (r#"ﴒ"#, r#"ﴒ"#, r#"ﴒ"#, r#"طي"#, r#"طي"#),
        (r#"ﴓ"#, r#"ﴓ"#, r#"ﴓ"#, r#"عى"#, r#"عى"#),
        (r#"ﴔ"#, r#"ﴔ"#, r#"ﴔ"#, r#"عي"#, r#"عي"#),
        (r#"ﴕ"#, r#"ﴕ"#, r#"ﴕ"#, r#"غى"#, r#"غى"#),
        (r#"ﴖ"#, r#"ﴖ"#, r#"ﴖ"#, r#"غي"#, r#"غي"#),
        (r#"ﴗ"#, r#"ﴗ"#, r#"ﴗ"#, r#"سى"#, r#"سى"#),
        (r#"ﴘ"#, r#"ﴘ"#, r#"ﴘ"#, r#"سي"#, r#"سي"#),
        (r#"ﴙ"#, r#"ﴙ"#, r#"ﴙ"#, r#"شى"#, r#"شى"#),
        (r#"ﴚ"#, r#"ﴚ"#, r#"ﴚ"#, r#"شي"#, r#"شي"#),
        (r#"ﴛ"#, r#"ﴛ"#, r#"ﴛ"#, r#"حى"#, r#"حى"#),
        (r#"ﴜ"#, r#"ﴜ"#, r#"ﴜ"#, r#"حي"#, r#"حي"#),
        (r#"ﴝ"#, r#"ﴝ"#, r#"ﴝ"#, r#"جى"#, r#"جى"#),
        (r#"ﴞ"#, r#"ﴞ"#, r#"ﴞ"#, r#"جي"#, r#"جي"#),
        (r#"ﴟ"#, r#"ﴟ"#, r#"ﴟ"#, r#"خى"#, r#"خى"#),
        (r#"ﴠ"#, r#"ﴠ"#, r#"ﴠ"#, r#"خي"#, r#"خي"#),
        (r#"ﴡ"#, r#"ﴡ"#, r#"ﴡ"#, r#"صى"#, r#"صى"#),
        (r#"ﴢ"#, r#"ﴢ"#, r#"ﴢ"#, r#"صي"#, r#"صي"#),
        (r#"ﴣ"#, r#"ﴣ"#, r#"ﴣ"#, r#"ضى"#, r#"ضى"#),
        (r#"ﴤ"#, r#"ﴤ"#, r#"ﴤ"#, r#"ضي"#, r#"ضي"#),
        (r#"ﴥ"#, r#"ﴥ"#, r#"ﴥ"#, r#"شج"#, r#"شج"#),
        (r#"ﴦ"#, r#"ﴦ"#, r#"ﴦ"#, r#"شح"#, r#"شح"#),
        (r#"ﴧ"#, r#"ﴧ"#, r#"ﴧ"#, r#"شخ"#, r#"شخ"#),
        (r#"ﴨ"#, r#"ﴨ"#, r#"ﴨ"#, r#"شم"#, r#"شم"#),
        (r#"ﴩ"#, r#"ﴩ"#, r#"ﴩ"#, r#"شر"#, r#"شر"#),
        (r#"ﴪ"#, r#"ﴪ"#, r#"ﴪ"#, r#"سر"#, r#"سر"#),
        (r#"ﴫ"#, r#"ﴫ"#, r#"ﴫ"#, r#"صر"#, r#"صر"#),
        (r#"ﴬ"#, r#"ﴬ"#, r#"ﴬ"#, r#"ضر"#, r#"ضر"#),
        (r#"ﴭ"#, r#"ﴭ"#, r#"ﴭ"#, r#"شج"#, r#"شج"#),
        (r#"ﴮ"#, r#"ﴮ"#, r#"ﴮ"#, r#"شح"#, r#"شح"#),
        (r#"ﴯ"#, r#"ﴯ"#, r#"ﴯ"#, r#"شخ"#, r#"شخ"#),
        (r#"ﴰ"#, r#"ﴰ"#, r#"ﴰ"#, r#"شم"#, r#"شم"#),
        (r#"ﴱ"#, r#"ﴱ"#, r#"ﴱ"#, r#"سه"#, r#"سه"#),
        (r#"ﴲ"#, r#"ﴲ"#, r#"ﴲ"#, r#"شه"#, r#"شه"#),
        (r#"ﴳ"#, r#"ﴳ"#, r#"ﴳ"#, r#"طم"#, r#"طم"#),
        (r#"ﴴ"#, r#"ﴴ"#, r#"ﴴ"#, r#"سج"#, r#"سج"#),
        (r#"ﴵ"#, r#"ﴵ"#, r#"ﴵ"#, r#"سح"#, r#"سح"#),
        (r#"ﴶ"#, r#"ﴶ"#, r#"ﴶ"#, r#"سخ"#, r#"سخ"#),
        (r#"ﴷ"#, r#"ﴷ"#, r#"ﴷ"#, r#"شج"#, r#"شج"#),
        (r#"ﴸ"#, r#"ﴸ"#, r#"ﴸ"#, r#"شح"#, r#"شح"#),
        (r#"ﴹ"#, r#"ﴹ"#, r#"ﴹ"#, r#"شخ"#, r#"شخ"#),
        (r#"ﴺ"#, r#"ﴺ"#, r#"ﴺ"#, r#"طم"#, r#"طم"#),
        (r#"ﴻ"#, r#"ﴻ"#, r#"ﴻ"#, r#"ظم"#, r#"ظم"#),
        (r#"ﴼ"#, r#"ﴼ"#, r#"ﴼ"#, r#"اً"#, r#"اً"#),
        (r#"ﴽ"#, r#"ﴽ"#, r#"ﴽ"#, r#"اً"#, r#"اً"#),
        (r#"ﵐ"#, r#"ﵐ"#, r#"ﵐ"#, r#"تجم"#, r#"تجم"#),
        (r#"ﵑ"#, r#"ﵑ"#, r#"ﵑ"#, r#"تحج"#, r#"تحج"#),
        (r#"ﵒ"#, r#"ﵒ"#, r#"ﵒ"#, r#"تحج"#, r#"تحج"#),
        (r#"ﵓ"#, r#"ﵓ"#, r#"ﵓ"#, r#"تحم"#, r#"تحم"#),
        (r#"ﵔ"#, r#"ﵔ"#, r#"ﵔ"#, r#"تخم"#, r#"تخم"#),
        (r#"ﵕ"#, r#"ﵕ"#, r#"ﵕ"#, r#"تمج"#, r#"تمج"#),
        (r#"ﵖ"#, r#"ﵖ"#, r#"ﵖ"#, r#"تمح"#, r#"تمح"#),
        (r#"ﵗ"#, r#"ﵗ"#, r#"ﵗ"#, r#"تمخ"#, r#"تمخ"#),
        (r#"ﵘ"#, r#"ﵘ"#, r#"ﵘ"#, r#"جمح"#, r#"جمح"#),
        (r#"ﵙ"#, r#"ﵙ"#, r#"ﵙ"#, r#"جمح"#, r#"جمح"#),
        (r#"ﵚ"#, r#"ﵚ"#, r#"ﵚ"#, r#"حمي"#, r#"حمي"#),
        (r#"ﵛ"#, r#"ﵛ"#, r#"ﵛ"#, r#"حمى"#, r#"حمى"#),
        (r#"ﵜ"#, r#"ﵜ"#, r#"ﵜ"#, r#"سحج"#, r#"سحج"#),
        (r#"ﵝ"#, r#"ﵝ"#, r#"ﵝ"#, r#"سجح"#, r#"سجح"#),
        (r#"ﵞ"#, r#"ﵞ"#, r#"ﵞ"#, r#"سجى"#, r#"سجى"#),
        (r#"ﵟ"#, r#"ﵟ"#, r#"ﵟ"#, r#"سمح"#, r#"سمح"#),
        (r#"ﵠ"#, r#"ﵠ"#, r#"ﵠ"#, r#"سمح"#, r#"سمح"#),
        (r#"ﵡ"#, r#"ﵡ"#, r#"ﵡ"#, r#"سمج"#, r#"سمج"#),
        (r#"ﵢ"#, r#"ﵢ"#, r#"ﵢ"#, r#"سمم"#, r#"سمم"#),
        (r#"ﵣ"#, r#"ﵣ"#, r#"ﵣ"#, r#"سمم"#, r#"سمم"#),
        (r#"ﵤ"#, r#"ﵤ"#, r#"ﵤ"#, r#"صحح"#, r#"صحح"#),
        (r#"ﵥ"#, r#"ﵥ"#, r#"ﵥ"#, r#"صحح"#, r#"صحح"#),
        (r#"ﵦ"#, r#"ﵦ"#, r#"ﵦ"#, r#"صمم"#, r#"صمم"#),
        (r#"ﵧ"#, r#"ﵧ"#, r#"ﵧ"#, r#"شحم"#, r#"شحم"#),
        (r#"ﵨ"#, r#"ﵨ"#, r#"ﵨ"#, r#"شحم"#, r#"شحم"#),
        (r#"ﵩ"#, r#"ﵩ"#, r#"ﵩ"#, r#"شجي"#, r#"شجي"#),
        (r#"ﵪ"#, r#"ﵪ"#, r#"ﵪ"#, r#"شمخ"#, r#"شمخ"#),
        (r#"ﵫ"#, r#"ﵫ"#, r#"ﵫ"#, r#"شمخ"#, r#"شمخ"#),
        (r#"ﵬ"#, r#"ﵬ"#, r#"ﵬ"#, r#"شمم"#, r#"شمم"#),
        (r#"ﵭ"#, r#"ﵭ"#, r#"ﵭ"#, r#"شمم"#, r#"شمم"#),
        (r#"ﵮ"#, r#"ﵮ"#, r#"ﵮ"#, r#"ضحى"#, r#"ضحى"#),
        (r#"ﵯ"#, r#"ﵯ"#, r#"ﵯ"#, r#"ضخم"#, r#"ضخم"#),
        (r#"ﵰ"#, r#"ﵰ"#, r#"ﵰ"#, r#"ضخم"#, r#"ضخم"#),
        (r#"ﵱ"#, r#"ﵱ"#, r#"ﵱ"#, r#"طمح"#, r#"طمح"#),
        (r#"ﵲ"#, r#"ﵲ"#, r#"ﵲ"#, r#"طمح"#, r#"طمح"#),
        (r#"ﵳ"#, r#"ﵳ"#, r#"ﵳ"#, r#"طمم"#, r#"طمم"#),
        (r#"ﵴ"#, r#"ﵴ"#, r#"ﵴ"#, r#"طمي"#, r#"طمي"#),
        (r#"ﵵ"#, r#"ﵵ"#, r#"ﵵ"#, r#"عجم"#, r#"عجم"#),
        (r#"ﵶ"#, r#"ﵶ"#, r#"ﵶ"#, r#"عمم"#, r#"عمم"#),
        (r#"ﵷ"#, r#"ﵷ"#, r#"ﵷ"#, r#"عمم"#, r#"عمم"#),
        (r#"ﵸ"#, r#"ﵸ"#, r#"ﵸ"#, r#"عمى"#, r#"عمى"#),
        (r#"ﵹ"#, r#"ﵹ"#, r#"ﵹ"#, r#"غمم"#, r#"غمم"#),
        (r#"ﵺ"#, r#"ﵺ"#, r#"ﵺ"#, r#"غمي"#, r#"غمي"#),
        (r#"ﵻ"#, r#"ﵻ"#, r#"ﵻ"#, r#"غمى"#, r#"غمى"#),
        (r#"ﵼ"#, r#"ﵼ"#, r#"ﵼ"#, r#"فخم"#, r#"فخم"#),
        (r#"ﵽ"#, r#"ﵽ"#, r#"ﵽ"#, r#"فخم"#, r#"فخم"#),
        (r#"ﵾ"#, r#"ﵾ"#, r#"ﵾ"#, r#"قمح"#, r#"قمح"#),
        (r#"ﵿ"#, r#"ﵿ"#, r#"ﵿ"#, r#"قمم"#, r#"قمم"#),
        (r#"ﶀ"#, r#"ﶀ"#, r#"ﶀ"#, r#"لحم"#, r#"لحم"#),
        (r#"ﶁ"#, r#"ﶁ"#, r#"ﶁ"#, r#"لحي"#, r#"لحي"#),
        (r#"ﶂ"#, r#"ﶂ"#, r#"ﶂ"#, r#"لحى"#, r#"لحى"#),
        (r#"ﶃ"#, r#"ﶃ"#, r#"ﶃ"#, r#"لجج"#, r#"لجج"#),
        (r#"ﶄ"#, r#"ﶄ"#, r#"ﶄ"#, r#"لجج"#, r#"لجج"#),
        (r#"ﶅ"#, r#"ﶅ"#, r#"ﶅ"#, r#"لخم"#, r#"لخم"#),
        (r#"ﶆ"#, r#"ﶆ"#, r#"ﶆ"#, r#"لخم"#, r#"لخم"#),
        (r#"ﶇ"#, r#"ﶇ"#, r#"ﶇ"#, r#"لمح"#, r#"لمح"#),
        (r#"ﶈ"#, r#"ﶈ"#, r#"ﶈ"#, r#"لمح"#, r#"لمح"#),
        (r#"ﶉ"#, r#"ﶉ"#, r#"ﶉ"#, r#"محج"#, r#"محج"#),
        (r#"ﶊ"#, r#"ﶊ"#, r#"ﶊ"#, r#"محم"#, r#"محم"#),
        (r#"ﶋ"#, r#"ﶋ"#, r#"ﶋ"#, r#"محي"#, r#"محي"#),
        (r#"ﶌ"#, r#"ﶌ"#, r#"ﶌ"#, r#"مجح"#, r#"مجح"#),
        (r#"ﶍ"#, r#"ﶍ"#, r#"ﶍ"#, r#"مجم"#, r#"مجم"#),
        (r#"ﶎ"#, r#"ﶎ"#, r#"ﶎ"#, r#"مخج"#, r#"مخج"#),
        (r#"ﶏ"#, r#"ﶏ"#, r#"ﶏ"#, r#"مخم"#, r#"مخم"#),
        (r#"ﶒ"#, r#"ﶒ"#, r#"ﶒ"#, r#"مجخ"#, r#"مجخ"#),
        (r#"ﶓ"#, r#"ﶓ"#, r#"ﶓ"#, r#"همج"#, r#"همج"#),
        (r#"ﶔ"#, r#"ﶔ"#, r#"ﶔ"#, r#"همم"#, r#"همم"#),
        (r#"ﶕ"#, r#"ﶕ"#, r#"ﶕ"#, r#"نحم"#, r#"نحم"#),
        (r#"ﶖ"#, r#"ﶖ"#, r#"ﶖ"#, r#"نحى"#, r#"نحى"#),
        (r#"ﶗ"#, r#"ﶗ"#, r#"ﶗ"#, r#"نجم"#, r#"نجم"#),
        (r#"ﶘ"#, r#"ﶘ"#, r#"ﶘ"#, r#"نجم"#, r#"نجم"#),
        (r#"ﶙ"#, r#"ﶙ"#, r#"ﶙ"#, r#"نجى"#, r#"نجى"#),
        (r#"ﶚ"#, r#"ﶚ"#, r#"ﶚ"#, r#"نمي"#, r#"نمي"#),
        (r#"ﶛ"#, r#"ﶛ"#, r#"ﶛ"#, r#"نمى"#, r#"نمى"#),
        (r#"ﶜ"#, r#"ﶜ"#, r#"ﶜ"#, r#"يمم"#, r#"يمم"#),
        (r#"ﶝ"#, r#"ﶝ"#, r#"ﶝ"#, r#"يمم"#, r#"يمم"#),
        (r#"ﶞ"#, r#"ﶞ"#, r#"ﶞ"#, r#"بخي"#, r#"بخي"#),
        (r#"ﶟ"#, r#"ﶟ"#, r#"ﶟ"#, r#"تجي"#, r#"تجي"#),
        (r#"ﶠ"#, r#"ﶠ"#, r#"ﶠ"#, r#"تجى"#, r#"تجى"#),
        (r#"ﶡ"#, r#"ﶡ"#, r#"ﶡ"#, r#"تخي"#, r#"تخي"#),
        (r#"ﶢ"#, r#"ﶢ"#, r#"ﶢ"#, r#"تخى"#, r#"تخى"#),
        (r#"ﶣ"#, r#"ﶣ"#, r#"ﶣ"#, r#"تمي"#, r#"تمي"#),
        (r#"ﶤ"#, r#"ﶤ"#, r#"ﶤ"#, r#"تمى"#, r#"تمى"#),
        (r#"ﶥ"#, r#"ﶥ"#, r#"ﶥ"#, r#"جمي"#, r#"جمي"#),
        (r#"ﶦ"#, r#"ﶦ"#, r#"ﶦ"#, r#"جحى"#, r#"جحى"#),
        (r#"ﶧ"#, r#"ﶧ"#, r#"ﶧ"#, r#"جمى"#, r#"جمى"#),
        (r#"ﶨ"#, r#"ﶨ"#, r#"ﶨ"#, r#"سخى"#, r#"سخى"#),
        (r#"ﶩ"#, r#"ﶩ"#, r#"ﶩ"#, r#"صحي"#, r#"صحي"#),
        (r#"ﶪ"#, r#"ﶪ"#, r#"ﶪ"#, r#"شحي"#, r#"شحي"#),
        (r#"ﶫ"#, r#"ﶫ"#, r#"ﶫ"#, r#"ضحي"#, r#"ضحي"#),
        (r#"ﶬ"#, r#"ﶬ"#, r#"ﶬ"#, r#"لجي"#, r#"لجي"#),
        (r#"ﶭ"#, r#"ﶭ"#, r#"ﶭ"#, r#"لمي"#, r#"لمي"#),
        (r#"ﶮ"#, r#"ﶮ"#, r#"ﶮ"#, r#"يحي"#, r#"يحي"#),
        (r#"ﶯ"#, r#"ﶯ"#, r#"ﶯ"#, r#"يجي"#, r#"يجي"#),
        (r#"ﶰ"#, r#"ﶰ"#, r#"ﶰ"#, r#"يمي"#, r#"يمي"#),
        (r#"ﶱ"#, r#"ﶱ"#, r#"ﶱ"#, r#"ممي"#, r#"ممي"#),
        (r#"ﶲ"#, r#"ﶲ"#, r#"ﶲ"#, r#"قمي"#, r#"قمي"#),
        (r#"ﶳ"#, r#"ﶳ"#, r#"ﶳ"#, r#"نحي"#, r#"نحي"#),
        (r#"ﶴ"#, r#"ﶴ"#, r#"ﶴ"#, r#"قمح"#, r#"قمح"#),
        (r#"ﶵ"#, r#"ﶵ"#, r#"ﶵ"#, r#"لحم"#, r#"لحم"#),
        (r#"ﶶ"#, r#"ﶶ"#, r#"ﶶ"#, r#"عمي"#, r#"عمي"#),
        (r#"ﶷ"#, r#"ﶷ"#, r#"ﶷ"#, r#"كمي"#, r#"كمي"#),
        (r#"ﶸ"#, r#"ﶸ"#, r#"ﶸ"#, r#"نجح"#, r#"نجح"#),
        (r#"ﶹ"#, r#"ﶹ"#, r#"ﶹ"#, r#"مخي"#, r#"مخي"#),
        (r#"ﶺ"#, r#"ﶺ"#, r#"ﶺ"#, r#"لجم"#, r#"لجم"#),
        (r#"ﶻ"#, r#"ﶻ"#, r#"ﶻ"#, r#"كمم"#, r#"كمم"#),
        (r#"ﶼ"#, r#"ﶼ"#, r#"ﶼ"#, r#"لجم"#, r#"لجم"#),
        (r#"ﶽ"#, r#"ﶽ"#, r#"ﶽ"#, r#"نجح"#, r#"نجح"#),
        (r#"ﶾ"#, r#"ﶾ"#, r#"ﶾ"#, r#"جحي"#, r#"جحي"#),
        (r#"ﶿ"#, r#"ﶿ"#, r#"ﶿ"#, r#"حجي"#, r#"حجي"#),
        (r#"ﷀ"#, r#"ﷀ"#, r#"ﷀ"#, r#"مجي"#, r#"مجي"#),
        (r#"ﷁ"#, r#"ﷁ"#, r#"ﷁ"#, r#"فمي"#, r#"فمي"#),
        (r#"ﷂ"#, r#"ﷂ"#, r#"ﷂ"#, r#"بحي"#, r#"بحي"#),
        (r#"ﷃ"#, r#"ﷃ"#, r#"ﷃ"#, r#"كمم"#, r#"كمم"#),
        (r#"ﷄ"#, r#"ﷄ"#, r#"ﷄ"#, r#"عجم"#, r#"عجم"#),
        (r#"ﷅ"#, r#"ﷅ"#, r#"ﷅ"#, r#"صمم"#, r#"صمم"#),
        (r#"ﷆ"#, r#"ﷆ"#, r#"ﷆ"#, r#"سخي"#, r#"سخي"#),
        (r#"ﷇ"#, r#"ﷇ"#, r#"ﷇ"#, r#"نجي"#, r#"نجي"#),
        (r#"ﷰ"#, r#"ﷰ"#, r#"ﷰ"#, r#"صلے"#, r#"صلے"#),
        (r#"ﷱ"#, r#"ﷱ"#, r#"ﷱ"#, r#"قلے"#, r#"قلے"#),
        (r#"ﷲ"#, r#"ﷲ"#, r#"ﷲ"#, r#"الله"#, r#"الله"#),
        (r#"ﷳ"#, r#"ﷳ"#, r#"ﷳ"#, r#"اكبر"#, r#"اكبر"#),
        (r#"ﷴ"#, r#"ﷴ"#, r#"ﷴ"#, r#"محمد"#, r#"محمد"#),
        (r#"ﷵ"#, r#"ﷵ"#, r#"ﷵ"#, r#"صلعم"#, r#"صلعم"#),
        (r#"ﷶ"#, r#"ﷶ"#, r#"ﷶ"#, r#"رسول"#, r#"رسول"#),
        (r#"ﷷ"#, r#"ﷷ"#, r#"ﷷ"#, r#"عليه"#, r#"عليه"#),
        (r#"ﷸ"#, r#"ﷸ"#, r#"ﷸ"#, r#"وسلم"#, r#"وسلم"#),
        (r#"ﷹ"#, r#"ﷹ"#, r#"ﷹ"#, r#"صلى"#, r#"صلى"#),
        (r#"ﷺ"#, r#"ﷺ"#, r#"ﷺ"#, r#"صلى الله عليه وسلم"#, r#"صلى الله عليه وسلم"#),
        (r#"ﷻ"#, r#"ﷻ"#, r#"ﷻ"#, r#"جل جلاله"#, r#"جل جلاله"#),
        (r#"﷼"#, r#"﷼"#, r#"﷼"#, r#"ریال"#, r#"ریال"#),
        (r#"︐"#, r#"︐"#, r#"︐"#, r#","#, r#","#),
        (r#"︑"#, r#"︑"#, r#"︑"#, r#"、"#, r#"、"#),
        (r#"︒"#, r#"︒"#, r#"︒"#, r#"。"#, r#"。"#),
        (r#"︓"#, r#"︓"#, r#"︓"#, r#":"#, r#":"#),
        (r#"︔"#, r#"︔"#, r#"︔"#, r#";"#, r#";"#),
        (r#"︕"#, r#"︕"#, r#"︕"#, r#"!"#, r#"!"#),
        (r#"︖"#, r#"︖"#, r#"︖"#, r#"?"#, r#"?"#),
        (r#"︗"#, r#"︗"#, r#"︗"#, r#"〖"#, r#"〖"#),
        (r#"︘"#, r#"︘"#, r#"︘"#, r#"〗"#, r#"〗"#),
        (r#"︙"#, r#"︙"#, r#"︙"#, r#"..."#, r#"..."#),
        (r#"︰"#, r#"︰"#, r#"︰"#, r#".."#, r#".."#),
        (r#"︱"#, r#"︱"#, r#"︱"#, r#"—"#, r#"—"#),
        (r#"︲"#, r#"︲"#, r#"︲"#, r#"–"#, r#"–"#),
        (r#"︳"#, r#"︳"#, r#"︳"#, r#"_"#, r#"_"#),
        (r#"︴"#, r#"︴"#, r#"︴"#, r#"_"#, r#"_"#),
        (r#"︵"#, r#"︵"#, r#"︵"#, r#"("#, r#"("#),
        (r#"︶"#, r#"︶"#, r#"︶"#, r#")"#, r#")"#),
        (r#"︷"#, r#"︷"#, r#"︷"#, r#"{"#, r#"{"#),
        (r#"︸"#, r#"︸"#, r#"︸"#, r#"}"#, r#"}"#),
        (r#"︹"#, r#"︹"#, r#"︹"#, r#"〔"#, r#"〔"#),
        (r#"︺"#, r#"︺"#, r#"︺"#, r#"〕"#, r#"〕"#),
        (r#"︻"#, r#"︻"#, r#"︻"#, r#"【"#, r#"【"#),
        (r#"︼"#, r#"︼"#, r#"︼"#, r#"】"#, r#"】"#),
        (r#"︽"#, r#"︽"#, r#"︽"#, r#"《"#, r#"《"#),
        (r#"︾"#, r#"︾"#, r#"︾"#, r#"》"#, r#"》"#),
        (r#"︿"#, r#"︿"#, r#"︿"#, r#"〈"#, r#"〈"#),
        (r#"﹀"#, r#"﹀"#, r#"﹀"#, r#"〉"#, r#"〉"#),
        (r#"﹁"#, r#"﹁"#, r#"﹁"#, r#"「"#, r#"「"#),
        (r#"﹂"#, r#"﹂"#, r#"﹂"#, r#"」"#, r#"」"#),
        (r#"﹃"#, r#"﹃"#, r#"﹃"#, r#"『"#, r#"『"#),
        (r#"﹄"#, r#"﹄"#, r#"﹄"#, r#"』"#, r#"』"#),
        (r#"﹇"#, r#"﹇"#, r#"﹇"#, r#"["#, r#"["#),
        (r#"﹈"#, r#"﹈"#, r#"﹈"#, r#"]"#, r#"]"#),
        (r#"﹉"#, r#"﹉"#, r#"﹉"#, r#" ̅"#, r#" ̅"#),
        (r#"﹊"#, r#"﹊"#, r#"﹊"#, r#" ̅"#, r#" ̅"#),
        (r#"﹋"#, r#"﹋"#, r#"﹋"#, r#" ̅"#, r#" ̅"#),
        (r#"﹌"#, r#"﹌"#, r#"﹌"#, r#" ̅"#, r#" ̅"#),
        (r#"﹍"#, r#"﹍"#, r#"﹍"#, r#"_"#, r#"_"#),
        (r#"﹎"#, r#"﹎"#, r#"﹎"#, r#"_"#, r#"_"#),
        (r#"﹏"#, r#"﹏"#, r#"﹏"#, r#"_"#, r#"_"#),
        (r#"﹐"#, r#"﹐"#, r#"﹐"#, r#","#, r#","#),
        (r#"﹑"#, r#"﹑"#, r#"﹑"#, r#"、"#, r#"、"#),
        (r#"﹒"#, r#"﹒"#, r#"﹒"#, r#"."#, r#"."#),
        (r#"﹔"#, r#"﹔"#, r#"﹔"#, r#";"#, r#";"#),
        (r#"﹕"#, r#"﹕"#, r#"﹕"#, r#":"#, r#":"#),
        (r#"﹖"#, r#"﹖"#, r#"﹖"#, r#"?"#, r#"?"#),
        (r#"﹗"#, r#"﹗"#, r#"﹗"#, r#"!"#, r#"!"#),
        (r#"﹘"#, r#"﹘"#, r#"﹘"#, r#"—"#, r#"—"#),
        (r#"﹙"#, r#"﹙"#, r#"﹙"#, r#"("#, r#"("#),
        (r#"﹚"#, r#"﹚"#, r#"﹚"#, r#")"#, r#")"#),
        (r#"﹛"#, r#"﹛"#, r#"﹛"#, r#"{"#, r#"{"#),
        (r#"﹜"#, r#"﹜"#, r#"﹜"#, r#"}"#, r#"}"#),
        (r#"﹝"#, r#"﹝"#, r#"﹝"#, r#"〔"#, r#"〔"#),
        (r#"﹞"#, r#"﹞"#, r#"﹞"#, r#"〕"#, r#"〕"#),
        (r#"﹟"#, r#"﹟"#, r#"﹟"#, r#"#"#, r#"#"#),
        (r#"﹠"#, r#"﹠"#, r#"﹠"#, r#"&"#, r#"&"#),
        (r#"﹡"#, r#"﹡"#, r#"﹡"#, r#"*"#, r#"*"#),
        (r#"﹢"#, r#"﹢"#, r#"﹢"#, r#"+"#, r#"+"#),
        (r#"﹣"#, r#"﹣"#, r#"﹣"#, r#"-"#, r#"-"#),
        (r#"﹤"#, r#"﹤"#, r#"﹤"#, r#"<"#, r#"<"#),
        (r#"﹥"#, r#"﹥"#, r#"﹥"#, r#">"#, r#">"#),
        (r#"﹦"#, r#"﹦"#, r#"﹦"#, r#"="#, r#"="#),
        (r#"﹨"#, r#"﹨"#, r#"﹨"#, r#"\"#, r#"\"#),
        (r#"﹩"#, r#"﹩"#, r#"﹩"#, r#"$"#, r#"$"#),
        (r#"﹪"#, r#"﹪"#, r#"﹪"#, r#"%"#, r#"%"#),
        (r#"﹫"#, r#"﹫"#, r#"﹫"#, r#"@"#, r#"@"#),
        (r#"ﹰ"#, r#"ﹰ"#, r#"ﹰ"#, r#" ً"#, r#" ً"#),
        (r#"ﹱ"#, r#"ﹱ"#, r#"ﹱ"#, r#"ـً"#, r#"ـً"#),
        (r#"ﹲ"#, r#"ﹲ"#, r#"ﹲ"#, r#" ٌ"#, r#" ٌ"#),
        (r#"ﹴ"#, r#"ﹴ"#, r#"ﹴ"#, r#" ٍ"#, r#" ٍ"#),
        (r#"ﹶ"#, r#"ﹶ"#, r#"ﹶ"#, r#" َ"#, r#" َ"#),
        (r#"ﹷ"#, r#"ﹷ"#, r#"ﹷ"#, r#"ـَ"#, r#"ـَ"#),
        (r#"ﹸ"#, r#"ﹸ"#, r#"ﹸ"#, r#" ُ"#, r#" ُ"#),
        (r#"ﹹ"#, r#"ﹹ"#, r#"ﹹ"#, r#"ـُ"#, r#"ـُ"#),
        (r#"ﹺ"#, r#"ﹺ"#, r#"ﹺ"#, r#" ِ"#, r#" ِ"#),
        (r#"ﹻ"#, r#"ﹻ"#, r#"ﹻ"#, r#"ـِ"#, r#"ـِ"#),
        (r#"ﹼ"#, r#"ﹼ"#, r#"ﹼ"#, r#" ّ"#, r#" ّ"#),
        (r#"ﹽ"#, r#"ﹽ"#, r#"ﹽ"#, r#"ـّ"#, r#"ـّ"#),
        (r#"ﹾ"#, r#"ﹾ"#, r#"ﹾ"#, r#" ْ"#, r#" ْ"#),
        (r#"ﹿ"#, r#"ﹿ"#, r#"ﹿ"#, r#"ـْ"#, r#"ـْ"#),
        (r#"ﺀ"#, r#"ﺀ"#, r#"ﺀ"#, r#"ء"#, r#"ء"#),
        (r#"ﺁ"#, r#"ﺁ"#, r#"ﺁ"#, r#"آ"#, r#"آ"#),
        (r#"ﺂ"#, r#"ﺂ"#, r#"ﺂ"#, r#"آ"#, r#"آ"#),
        (r#"ﺃ"#, r#"ﺃ"#, r#"ﺃ"#, r#"أ"#, r#"أ"#),
        (r#"ﺄ"#, r#"ﺄ"#, r#"ﺄ"#, r#"أ"#, r#"أ"#),
        (r#"ﺅ"#, r#"ﺅ"#, r#"ﺅ"#, r#"ؤ"#, r#"ؤ"#),
        (r#"ﺆ"#, r#"ﺆ"#, r#"ﺆ"#, r#"ؤ"#, r#"ؤ"#),
        (r#"ﺇ"#, r#"ﺇ"#, r#"ﺇ"#, r#"إ"#, r#"إ"#),
        (r#"ﺈ"#, r#"ﺈ"#, r#"ﺈ"#, r#"إ"#, r#"إ"#),
        (r#"ﺉ"#, r#"ﺉ"#, r#"ﺉ"#, r#"ئ"#, r#"ئ"#),
        (r#"ﺊ"#, r#"ﺊ"#, r#"ﺊ"#, r#"ئ"#, r#"ئ"#),
        (r#"ﺋ"#, r#"ﺋ"#, r#"ﺋ"#, r#"ئ"#, r#"ئ"#),
        (r#"ﺌ"#, r#"ﺌ"#, r#"ﺌ"#, r#"ئ"#, r#"ئ"#),
        (r#"ﺍ"#, r#"ﺍ"#, r#"ﺍ"#, r#"ا"#, r#"ا"#),
        (r#"ﺎ"#, r#"ﺎ"#, r#"ﺎ"#, r#"ا"#, r#"ا"#),
        (r#"ﺏ"#, r#"ﺏ"#, r#"ﺏ"#, r#"ب"#, r#"ب"#),
        (r#"ﺐ"#, r#"ﺐ"#, r#"ﺐ"#, r#"ب"#, r#"ب"#),
        (r#"ﺑ"#, r#"ﺑ"#, r#"ﺑ"#, r#"ب"#, r#"ب"#),
        (r#"ﺒ"#, r#"ﺒ"#, r#"ﺒ"#, r#"ب"#, r#"ب"#),
        (r#"ﺓ"#, r#"ﺓ"#, r#"ﺓ"#, r#"ة"#, r#"ة"#),
        (r#"ﺔ"#, r#"ﺔ"#, r#"ﺔ"#, r#"ة"#, r#"ة"#),
        (r#"ﺕ"#, r#"ﺕ"#, r#"ﺕ"#, r#"ت"#, r#"ت"#),
        (r#"ﺖ"#, r#"ﺖ"#, r#"ﺖ"#, r#"ت"#, r#"ت"#),
        (r#"ﺗ"#, r#"ﺗ"#, r#"ﺗ"#, r#"ت"#, r#"ت"#),
        (r#"ﺘ"#, r#"ﺘ"#, r#"ﺘ"#, r#"ت"#, r#"ت"#),
        (r#"ﺙ"#, r#"ﺙ"#, r#"ﺙ"#, r#"ث"#, r#"ث"#),
        (r#"ﺚ"#, r#"ﺚ"#, r#"ﺚ"#, r#"ث"#, r#"ث"#),
        (r#"ﺛ"#, r#"ﺛ"#, r#"ﺛ"#, r#"ث"#, r#"ث"#),
        (r#"ﺜ"#, r#"ﺜ"#, r#"ﺜ"#, r#"ث"#, r#"ث"#),
        (r#"ﺝ"#, r#"ﺝ"#, r#"ﺝ"#, r#"ج"#, r#"ج"#),
        (r#"ﺞ"#, r#"ﺞ"#, r#"ﺞ"#, r#"ج"#, r#"ج"#),
        (r#"ﺟ"#, r#"ﺟ"#, r#"ﺟ"#, r#"ج"#, r#"ج"#),
        (r#"ﺠ"#, r#"ﺠ"#, r#"ﺠ"#, r#"ج"#, r#"ج"#),
        (r#"ﺡ"#, r#"ﺡ"#, r#"ﺡ"#, r#"ح"#, r#"ح"#),
        (r#"ﺢ"#, r#"ﺢ"#, r#"ﺢ"#, r#"ح"#, r#"ح"#),
        (r#"ﺣ"#, r#"ﺣ"#, r#"ﺣ"#, r#"ح"#, r#"ح"#),
        (r#"ﺤ"#, r#"ﺤ"#, r#"ﺤ"#, r#"ح"#, r#"ح"#),
        (r#"ﺥ"#, r#"ﺥ"#, r#"ﺥ"#, r#"خ"#, r#"خ"#),
        (r#"ﺦ"#, r#"ﺦ"#, r#"ﺦ"#, r#"خ"#, r#"خ"#),
        (r#"ﺧ"#, r#"ﺧ"#, r#"ﺧ"#, r#"خ"#, r#"خ"#),
        (r#"ﺨ"#, r#"ﺨ"#, r#"ﺨ"#, r#"خ"#, r#"خ"#),
        (r#"ﺩ"#, r#"ﺩ"#, r#"ﺩ"#, r#"د"#, r#"د"#),
        (r#"ﺪ"#, r#"ﺪ"#, r#"ﺪ"#, r#"د"#, r#"د"#),
        (r#"ﺫ"#, r#"ﺫ"#, r#"ﺫ"#, r#"ذ"#, r#"ذ"#),
        (r#"ﺬ"#, r#"ﺬ"#, r#"ﺬ"#, r#"ذ"#, r#"ذ"#),
        (r#"ﺭ"#, r#"ﺭ"#, r#"ﺭ"#, r#"ر"#, r#"ر"#),
        (r#"ﺮ"#, r#"ﺮ"#, r#"ﺮ"#, r#"ر"#, r#"ر"#),
        (r#"ﺯ"#, r#"ﺯ"#, r#"ﺯ"#, r#"ز"#, r#"ز"#),
        (r#"ﺰ"#, r#"ﺰ"#, r#"ﺰ"#, r#"ز"#, r#"ز"#),
        (r#"ﺱ"#, r#"ﺱ"#, r#"ﺱ"#, r#"س"#, r#"س"#),
        (r#"ﺲ"#, r#"ﺲ"#, r#"ﺲ"#, r#"س"#, r#"س"#),
        (r#"ﺳ"#, r#"ﺳ"#, r#"ﺳ"#, r#"س"#, r#"س"#),
        (r#"ﺴ"#, r#"ﺴ"#, r#"ﺴ"#, r#"س"#, r#"س"#),
        (r#"ﺵ"#, r#"ﺵ"#, r#"ﺵ"#, r#"ش"#, r#"ش"#),
        (r#"ﺶ"#, r#"ﺶ"#, r#"ﺶ"#, r#"ش"#, r#"ش"#),
        (r#"ﺷ"#, r#"ﺷ"#, r#"ﺷ"#, r#"ش"#, r#"ش"#),
        (r#"ﺸ"#, r#"ﺸ"#, r#"ﺸ"#, r#"ش"#, r#"ش"#),
        (r#"ﺹ"#, r#"ﺹ"#, r#"ﺹ"#, r#"ص"#, r#"ص"#),
        (r#"ﺺ"#, r#"ﺺ"#, r#"ﺺ"#, r#"ص"#, r#"ص"#),
        (r#"ﺻ"#, r#"ﺻ"#, r#"ﺻ"#, r#"ص"#, r#"ص"#),
        (r#"ﺼ"#, r#"ﺼ"#, r#"ﺼ"#, r#"ص"#, r#"ص"#),
        (r#"ﺽ"#, r#"ﺽ"#, r#"ﺽ"#, r#"ض"#, r#"ض"#),
        (r#"ﺾ"#, r#"ﺾ"#, r#"ﺾ"#, r#"ض"#, r#"ض"#),
        (r#"ﺿ"#, r#"ﺿ"#, r#"ﺿ"#, r#"ض"#, r#"ض"#),
        (r#"ﻀ"#, r#"ﻀ"#, r#"ﻀ"#, r#"ض"#, r#"ض"#),
        (r#"ﻁ"#, r#"ﻁ"#, r#"ﻁ"#, r#"ط"#, r#"ط"#),
        (r#"ﻂ"#, r#"ﻂ"#, r#"ﻂ"#, r#"ط"#, r#"ط"#),
        (r#"ﻃ"#, r#"ﻃ"#, r#"ﻃ"#, r#"ط"#, r#"ط"#),
        (r#"ﻄ"#, r#"ﻄ"#, r#"ﻄ"#, r#"ط"#, r#"ط"#),
        (r#"ﻅ"#, r#"ﻅ"#, r#"ﻅ"#, r#"ظ"#, r#"ظ"#),
        (r#"ﻆ"#, r#"ﻆ"#, r#"ﻆ"#, r#"ظ"#, r#"ظ"#),
        (r#"ﻇ"#, r#"ﻇ"#, r#"ﻇ"#, r#"ظ"#, r#"ظ"#),
        (r#"ﻈ"#, r#"ﻈ"#, r#"ﻈ"#, r#"ظ"#, r#"ظ"#),
        (r#"ﻉ"#, r#"ﻉ"#, r#"ﻉ"#, r#"ع"#, r#"ع"#),
        (r#"ﻊ"#, r#"ﻊ"#, r#"ﻊ"#, r#"ع"#, r#"ع"#),
        (r#"ﻋ"#, r#"ﻋ"#, r#"ﻋ"#, r#"ع"#, r#"ع"#),
        (r#"ﻌ"#, r#"ﻌ"#, r#"ﻌ"#, r#"ع"#, r#"ع"#),
        (r#"ﻍ"#, r#"ﻍ"#, r#"ﻍ"#, r#"غ"#, r#"غ"#),
        (r#"ﻎ"#, r#"ﻎ"#, r#"ﻎ"#, r#"غ"#, r#"غ"#),
        (r#"ﻏ"#, r#"ﻏ"#, r#"ﻏ"#, r#"غ"#, r#"غ"#),
        (r#"ﻐ"#, r#"ﻐ"#, r#"ﻐ"#, r#"غ"#, r#"غ"#),
        (r#"ﻑ"#, r#"ﻑ"#, r#"ﻑ"#, r#"ف"#, r#"ف"#),
        (r#"ﻒ"#, r#"ﻒ"#, r#"ﻒ"#, r#"ف"#, r#"ف"#),
        (r#"ﻓ"#, r#"ﻓ"#, r#"ﻓ"#, r#"ف"#, r#"ف"#),
        (r#"ﻔ"#, r#"ﻔ"#, r#"ﻔ"#, r#"ف"#, r#"ف"#),
        (r#"ﻕ"#, r#"ﻕ"#, r#"ﻕ"#, r#"ق"#, r#"ق"#),
        (r#"ﻖ"#, r#"ﻖ"#, r#"ﻖ"#, r#"ق"#, r#"ق"#),
        (r#"ﻗ"#, r#"ﻗ"#, r#"ﻗ"#, r#"ق"#, r#"ق"#),
        (r#"ﻘ"#, r#"ﻘ"#, r#"ﻘ"#, r#"ق"#, r#"ق"#),
        (r#"ﻙ"#, r#"ﻙ"#, r#"ﻙ"#, r#"ك"#, r#"ك"#),
        (r#"ﻚ"#, r#"ﻚ"#, r#"ﻚ"#, r#"ك"#, r#"ك"#),
        (r#"ﻛ"#, r#"ﻛ"#, r#"ﻛ"#, r#"ك"#, r#"ك"#),
        (r#"ﻜ"#, r#"ﻜ"#, r#"ﻜ"#, r#"ك"#, r#"ك"#),
        (r#"ﻝ"#, r#"ﻝ"#, r#"ﻝ"#, r#"ل"#, r#"ل"#),
        (r#"ﻞ"#, r#"ﻞ"#, r#"ﻞ"#, r#"ل"#, r#"ل"#),
        (r#"ﻟ"#, r#"ﻟ"#, r#"ﻟ"#, r#"ل"#, r#"ل"#),
        (r#"ﻠ"#, r#"ﻠ"#, r#"ﻠ"#, r#"ل"#, r#"ل"#),
        (r#"ﻡ"#, r#"ﻡ"#, r#"ﻡ"#, r#"م"#, r#"م"#),
        (r#"ﻢ"#, r#"ﻢ"#, r#"ﻢ"#, r#"م"#, r#"م"#),
        (r#"ﻣ"#, r#"ﻣ"#, r#"ﻣ"#, r#"م"#, r#"م"#),
        (r#"ﻤ"#, r#"ﻤ"#, r#"ﻤ"#, r#"م"#, r#"م"#),
        (r#"ﻥ"#, r#"ﻥ"#, r#"ﻥ"#, r#"ن"#, r#"ن"#),
        (r#"ﻦ"#, r#"ﻦ"#, r#"ﻦ"#, r#"ن"#, r#"ن"#),
        (r#"ﻧ"#, r#"ﻧ"#, r#"ﻧ"#, r#"ن"#, r#"ن"#),
        (r#"ﻨ"#, r#"ﻨ"#, r#"ﻨ"#, r#"ن"#, r#"ن"#),
        (r#"ﻩ"#, r#"ﻩ"#, r#"ﻩ"#, r#"ه"#, r#"ه"#),
        (r#"ﻪ"#, r#"ﻪ"#, r#"ﻪ"#, r#"ه"#, r#"ه"#),
        (r#"ﻫ"#, r#"ﻫ"#, r#"ﻫ"#, r#"ه"#, r#"ه"#),
        (r#"ﻬ"#, r#"ﻬ"#, r#"ﻬ"#, r#"ه"#, r#"ه"#),
        (r#"ﻭ"#, r#"ﻭ"#, r#"ﻭ"#, r#"و"#, r#"و"#),
        (r#"ﻮ"#, r#"ﻮ"#, r#"ﻮ"#, r#"و"#, r#"و"#),
        (r#"ﻯ"#, r#"ﻯ"#, r#"ﻯ"#, r#"ى"#, r#"ى"#),
        (r#"ﻰ"#, r#"ﻰ"#, r#"ﻰ"#, r#"ى"#, r#"ى"#),
        (r#"ﻱ"#, r#"ﻱ"#, r#"ﻱ"#, r#"ي"#, r#"ي"#),
        (r#"ﻲ"#, r#"ﻲ"#, r#"ﻲ"#, r#"ي"#, r#"ي"#),
        (r#"ﻳ"#, r#"ﻳ"#, r#"ﻳ"#, r#"ي"#, r#"ي"#),
        (r#"ﻴ"#, r#"ﻴ"#, r#"ﻴ"#, r#"ي"#, r#"ي"#),
        (r#"ﻵ"#, r#"ﻵ"#, r#"ﻵ"#, r#"لآ"#, r#"لآ"#),
        (r#"ﻶ"#, r#"ﻶ"#, r#"ﻶ"#, r#"لآ"#, r#"لآ"#),
        (r#"ﻷ"#, r#"ﻷ"#, r#"ﻷ"#, r#"لأ"#, r#"لأ"#),
        (r#"ﻸ"#, r#"ﻸ"#, r#"ﻸ"#, r#"لأ"#, r#"لأ"#),
        (r#"ﻹ"#, r#"ﻹ"#, r#"ﻹ"#, r#"لإ"#, r#"لإ"#),
        (r#"ﻺ"#, r#"ﻺ"#, r#"ﻺ"#, r#"لإ"#, r#"لإ"#),
        (r#"ﻻ"#, r#"ﻻ"#, r#"ﻻ"#, r#"لا"#, r#"لا"#),
        (r#"ﻼ"#, r#"ﻼ"#, r#"ﻼ"#, r#"لا"#, r#"لا"#),
        (r#"！"#, r#"！"#, r#"！"#, r#"!"#, r#"!"#),
        (r#"＂"#, r#"＂"#, r#"＂"#, r#"""#, r#"""#),
        (r#"＃"#, r#"＃"#, r#"＃"#, r#"#"#, r#"#"#),
        (r#"＄"#, r#"＄"#, r#"＄"#, r#"$"#, r#"$"#),
        (r#"％"#, r#"％"#, r#"％"#, r#"%"#, r#"%"#),
        (r#"＆"#, r#"＆"#, r#"＆"#, r#"&"#, r#"&"#),
        (r#"＇"#, r#"＇"#, r#"＇"#, r#"'"#, r#"'"#),
        (r#"（"#, r#"（"#, r#"（"#, r#"("#, r#"("#),
        (r#"）"#, r#"）"#, r#"）"#, r#")"#, r#")"#),
        (r#"＊"#, r#"＊"#, r#"＊"#, r#"*"#, r#"*"#),
        (r#"＋"#, r#"＋"#, r#"＋"#, r#"+"#, r#"+"#),
        (r#"，"#, r#"，"#, r#"，"#, r#","#, r#","#),
        (r#"－"#, r#"－"#, r#"－"#, r#"-"#, r#"-"#),
        (r#"．"#, r#"．"#, r#"．"#, r#"."#, r#"."#),
        (r#"／"#, r#"／"#, r#"／"#, r#"/"#, r#"/"#),
        (r#"０"#, r#"０"#, r#"０"#, r#"0"#, r#"0"#),
        (r#"１"#, r#"１"#, r#"１"#, r#"1"#, r#"1"#),
        (r#"２"#, r#"２"#, r#"２"#, r#"2"#, r#"2"#),
        (r#"３"#, r#"３"#, r#"３"#, r#"3"#, r#"3"#),
        (r#"４"#, r#"４"#, r#"４"#, r#"4"#, r#"4"#),
        (r#"５"#, r#"５"#, r#"５"#, r#"5"#, r#"5"#),
        (r#"６"#, r#"６"#, r#"６"#, r#"6"#, r#"6"#),
        (r#"７"#, r#"７"#, r#"７"#, r#"7"#, r#"7"#),
        (r#"８"#, r#"８"#, r#"８"#, r#"8"#, r#"8"#),
        (r#"９"#, r#"９"#, r#"９"#, r#"9"#, r#"9"#),
        (r#"："#, r#"："#, r#"："#, r#":"#, r#":"#),
        (r#"；"#, r#"；"#, r#"；"#, r#";"#, r#";"#),
        (r#"＜"#, r#"＜"#, r#"＜"#, r#"<"#, r#"<"#),
        (r#"＝"#, r#"＝"#, r#"＝"#, r#"="#, r#"="#),
        (r#"＞"#, r#"＞"#, r#"＞"#, r#">"#, r#">"#),
        (r#"？"#, r#"？"#, r#"？"#, r#"?"#, r#"?"#),
        (r#"＠"#, r#"＠"#, r#"＠"#, r#"@"#, r#"@"#),
        (r#"Ａ"#, r#"Ａ"#, r#"Ａ"#, r#"A"#, r#"A"#),
        (r#"Ｂ"#, r#"Ｂ"#, r#"Ｂ"#, r#"B"#, r#"B"#),
        (r#"Ｃ"#, r#"Ｃ"#, r#"Ｃ"#, r#"C"#, r#"C"#),
        (r#"Ｄ"#, r#"Ｄ"#, r#"Ｄ"#, r#"D"#, r#"D"#),
        (r#"Ｅ"#, r#"Ｅ"#, r#"Ｅ"#, r#"E"#, r#"E"#),
        (r#"Ｆ"#, r#"Ｆ"#, r#"Ｆ"#, r#"F"#, r#"F"#),
        (r#"Ｇ"#, r#"Ｇ"#, r#"Ｇ"#, r#"G"#, r#"G"#),
        (r#"Ｈ"#, r#"Ｈ"#, r#"Ｈ"#, r#"H"#, r#"H"#),
        (r#"Ｉ"#, r#"Ｉ"#, r#"Ｉ"#, r#"I"#, r#"I"#),
        (r#"Ｊ"#, r#"Ｊ"#, r#"Ｊ"#, r#"J"#, r#"J"#),
        (r#"Ｋ"#, r#"Ｋ"#, r#"Ｋ"#, r#"K"#, r#"K"#),
        (r#"Ｌ"#, r#"Ｌ"#, r#"Ｌ"#, r#"L"#, r#"L"#),
        (r#"Ｍ"#, r#"Ｍ"#, r#"Ｍ"#, r#"M"#, r#"M"#),
        (r#"Ｎ"#, r#"Ｎ"#, r#"Ｎ"#, r#"N"#, r#"N"#),
        (r#"Ｏ"#, r#"Ｏ"#, r#"Ｏ"#, r#"O"#, r#"O"#),
        (r#"Ｐ"#, r#"Ｐ"#, r#"Ｐ"#, r#"P"#, r#"P"#),
        (r#"Ｑ"#, r#"Ｑ"#, r#"Ｑ"#, r#"Q"#, r#"Q"#),
        (r#"Ｒ"#, r#"Ｒ"#, r#"Ｒ"#, r#"R"#, r#"R"#),
        (r#"Ｓ"#, r#"Ｓ"#, r#"Ｓ"#, r#"S"#, r#"S"#),
        (r#"Ｔ"#, r#"Ｔ"#, r#"Ｔ"#, r#"T"#, r#"T"#),
        (r#"Ｕ"#, r#"Ｕ"#, r#"Ｕ"#, r#"U"#, r#"U"#),
        (r#"Ｖ"#, r#"Ｖ"#, r#"Ｖ"#, r#"V"#, r#"V"#),
        (r#"Ｗ"#, r#"Ｗ"#, r#"Ｗ"#, r#"W"#, r#"W"#),
        (r#"Ｘ"#, r#"Ｘ"#, r#"Ｘ"#, r#"X"#, r#"X"#),
        (r#"Ｙ"#, r#"Ｙ"#, r#"Ｙ"#, r#"Y"#, r#"Y"#),
        (r#"Ｚ"#, r#"Ｚ"#, r#"Ｚ"#, r#"Z"#, r#"Z"#),
        (r#"［"#, r#"［"#, r#"［"#, r#"["#, r#"["#),
        (r#"＼"#, r#"＼"#, r#"＼"#, r#"\"#, r#"\"#),
        (r#"］"#, r#"］"#, r#"］"#, r#"]"#, r#"]"#),
        (r#"＾"#, r#"＾"#, r#"＾"#, r#"^"#, r#"^"#),
        (r#"＿"#, r#"＿"#, r#"＿"#, r#"_"#, r#"_"#),
        (r#"｀"#, r#"｀"#, r#"｀"#, r#"`"#, r#"`"#),
        (r#"ａ"#, r#"ａ"#, r#"ａ"#, r#"a"#, r#"a"#),
        (r#"ｂ"#, r#"ｂ"#, r#"ｂ"#, r#"b"#, r#"b"#),
        (r#"ｃ"#, r#"ｃ"#, r#"ｃ"#, r#"c"#, r#"c"#),
        (r#"ｄ"#, r#"ｄ"#, r#"ｄ"#, r#"d"#, r#"d"#),
        (r#"ｅ"#, r#"ｅ"#, r#"ｅ"#, r#"e"#, r#"e"#),
        (r#"ｆ"#, r#"ｆ"#, r#"ｆ"#, r#"f"#, r#"f"#),
        (r#"ｇ"#, r#"ｇ"#, r#"ｇ"#, r#"g"#, r#"g"#),
        (r#"ｈ"#, r#"ｈ"#, r#"ｈ"#, r#"h"#, r#"h"#),
        (r#"ｉ"#, r#"ｉ"#, r#"ｉ"#, r#"i"#, r#"i"#),
        (r#"ｊ"#, r#"ｊ"#, r#"ｊ"#, r#"j"#, r#"j"#),
        (r#"ｋ"#, r#"ｋ"#, r#"ｋ"#, r#"k"#, r#"k"#),
        (r#"ｌ"#, r#"ｌ"#, r#"ｌ"#, r#"l"#, r#"l"#),
        (r#"ｍ"#, r#"ｍ"#, r#"ｍ"#, r#"m"#, r#"m"#),
        (r#"ｎ"#, r#"ｎ"#, r#"ｎ"#, r#"n"#, r#"n"#),
        (r#"ｏ"#, r#"ｏ"#, r#"ｏ"#, r#"o"#, r#"o"#),
        (r#"ｐ"#, r#"ｐ"#, r#"ｐ"#, r#"p"#, r#"p"#),
        (r#"ｑ"#, r#"ｑ"#, r#"ｑ"#, r#"q"#, r#"q"#),
        (r#"ｒ"#, r#"ｒ"#, r#"ｒ"#, r#"r"#, r#"r"#),
        (r#"ｓ"#, r#"ｓ"#, r#"ｓ"#, r#"s"#, r#"s"#),
        (r#"ｔ"#, r#"ｔ"#, r#"ｔ"#, r#"t"#, r#"t"#),
        (r#"ｕ"#, r#"ｕ"#, r#"ｕ"#, r#"u"#, r#"u"#),
        (r#"ｖ"#, r#"ｖ"#, r#"ｖ"#, r#"v"#, r#"v"#),
        (r#"ｗ"#, r#"ｗ"#, r#"ｗ"#, r#"w"#, r#"w"#),
        (r#"ｘ"#, r#"ｘ"#, r#"ｘ"#, r#"x"#, r#"x"#),
        (r#"ｙ"#, r#"ｙ"#, r#"ｙ"#, r#"y"#, r#"y"#),
        (r#"ｚ"#, r#"ｚ"#, r#"ｚ"#, r#"z"#, r#"z"#),
        (r#"｛"#, r#"｛"#, r#"｛"#, r#"{"#, r#"{"#),
        (r#"｜"#, r#"｜"#, r#"｜"#, r#"|"#, r#"|"#),
        (r#"｝"#, r#"｝"#, r#"｝"#, r#"}"#, r#"}"#),
        (r#"～"#, r#"～"#, r#"～"#, r#"~"#, r#"~"#),
        (r#"｟"#, r#"｟"#, r#"｟"#, r#"⦅"#, r#"⦅"#),
        (r#"｠"#, r#"｠"#, r#"｠"#, r#"⦆"#, r#"⦆"#),
        (r#"｡"#, r#"｡"#, r#"｡"#, r#"。"#, r#"。"#),
        (r#"｢"#, r#"｢"#, r#"｢"#, r#"「"#, r#"「"#),
        (r#"｣"#, r#"｣"#, r#"｣"#, r#"」"#, r#"」"#),
        (r#"､"#, r#"､"#, r#"､"#, r#"、"#, r#"、"#),
        (r#"･"#, r#"･"#, r#"･"#, r#"・"#, r#"・"#),
        (r#"ｦ"#, r#"ｦ"#, r#"ｦ"#, r#"ヲ"#, r#"ヲ"#),
        (r#"ｧ"#, r#"ｧ"#, r#"ｧ"#, r#"ァ"#, r#"ァ"#),
        (r#"ｨ"#, r#"ｨ"#, r#"ｨ"#, r#"ィ"#, r#"ィ"#),
        (r#"ｩ"#, r#"ｩ"#, r#"ｩ"#, r#"ゥ"#, r#"ゥ"#),
        (r#"ｪ"#, r#"ｪ"#, r#"ｪ"#, r#"ェ"#, r#"ェ"#),
        (r#"ｫ"#, r#"ｫ"#, r#"ｫ"#, r#"ォ"#, r#"ォ"#),
        (r#"ｬ"#, r#"ｬ"#, r#"ｬ"#, r#"ャ"#, r#"ャ"#),
        (r#"ｭ"#, r#"ｭ"#, r#"ｭ"#, r#"ュ"#, r#"ュ"#),
        (r#"ｮ"#, r#"ｮ"#, r#"ｮ"#, r#"ョ"#, r#"ョ"#),
        (r#"ｯ"#, r#"ｯ"#, r#"ｯ"#, r#"ッ"#, r#"ッ"#),
        (r#"ｰ"#, r#"ｰ"#, r#"ｰ"#, r#"ー"#, r#"ー"#),
        (r#"ｱ"#, r#"ｱ"#, r#"ｱ"#, r#"ア"#, r#"ア"#),
        (r#"ｲ"#, r#"ｲ"#, r#"ｲ"#, r#"イ"#, r#"イ"#),
        (r#"ｳ"#, r#"ｳ"#, r#"ｳ"#, r#"ウ"#, r#"ウ"#),
        (r#"ｴ"#, r#"ｴ"#, r#"ｴ"#, r#"エ"#, r#"エ"#),
        (r#"ｵ"#, r#"ｵ"#, r#"ｵ"#, r#"オ"#, r#"オ"#),
        (r#"ｶ"#, r#"ｶ"#, r#"ｶ"#, r#"カ"#, r#"カ"#),
        (r#"ｷ"#, r#"ｷ"#, r#"ｷ"#, r#"キ"#, r#"キ"#),
        (r#"ｸ"#, r#"ｸ"#, r#"ｸ"#, r#"ク"#, r#"ク"#),
        (r#"ｹ"#, r#"ｹ"#, r#"ｹ"#, r#"ケ"#, r#"ケ"#),
        (r#"ｺ"#, r#"ｺ"#, r#"ｺ"#, r#"コ"#, r#"コ"#),
        (r#"ｻ"#, r#"ｻ"#, r#"ｻ"#, r#"サ"#, r#"サ"#),
        (r#"ｼ"#, r#"ｼ"#, r#"ｼ"#, r#"シ"#, r#"シ"#),
        (r#"ｽ"#, r#"ｽ"#, r#"ｽ"#, r#"ス"#, r#"ス"#),
        (r#"ｾ"#, r#"ｾ"#, r#"ｾ"#, r#"セ"#, r#"セ"#),
        (r#"ｿ"#, r#"ｿ"#, r#"ｿ"#, r#"ソ"#, r#"ソ"#),
        (r#"ﾀ"#, r#"ﾀ"#, r#"ﾀ"#, r#"タ"#, r#"タ"#),
        (r#"ﾁ"#, r#"ﾁ"#, r#"ﾁ"#, r#"チ"#, r#"チ"#),
        (r#"ﾂ"#, r#"ﾂ"#, r#"ﾂ"#, r#"ツ"#, r#"ツ"#),
        (r#"ﾃ"#, r#"ﾃ"#, r#"ﾃ"#, r#"テ"#, r#"テ"#),
        (r#"ﾄ"#, r#"ﾄ"#, r#"ﾄ"#, r#"ト"#, r#"ト"#),
        (r#"ﾅ"#, r#"ﾅ"#, r#"ﾅ"#, r#"ナ"#, r#"ナ"#),
        (r#"ﾆ"#, r#"ﾆ"#, r#"ﾆ"#, r#"ニ"#, r#"ニ"#),
        (r#"ﾇ"#, r#"ﾇ"#, r#"ﾇ"#, r#"ヌ"#, r#"ヌ"#),
        (r#"ﾈ"#, r#"ﾈ"#, r#"ﾈ"#, r#"ネ"#, r#"ネ"#),
        (r#"ﾉ"#, r#"ﾉ"#, r#"ﾉ"#, r#"ノ"#, r#"ノ"#),
        (r#"ﾊ"#, r#"ﾊ"#, r#"ﾊ"#, r#"ハ"#, r#"ハ"#),
        (r#"ﾋ"#, r#"ﾋ"#, r#"ﾋ"#, r#"ヒ"#, r#"ヒ"#),
        (r#"ﾌ"#, r#"ﾌ"#, r#"ﾌ"#, r#"フ"#, r#"フ"#),
        (r#"ﾍ"#, r#"ﾍ"#, r#"ﾍ"#, r#"ヘ"#, r#"ヘ"#),
        (r#"ﾎ"#, r#"ﾎ"#, r#"ﾎ"#, r#"ホ"#, r#"ホ"#),
        (r#"ﾏ"#, r#"ﾏ"#, r#"ﾏ"#, r#"マ"#, r#"マ"#),
        (r#"ﾐ"#, r#"ﾐ"#, r#"ﾐ"#, r#"ミ"#, r#"ミ"#),
        (r#"ﾑ"#, r#"ﾑ"#, r#"ﾑ"#, r#"ム"#, r#"ム"#),
        (r#"ﾒ"#, r#"ﾒ"#, r#"ﾒ"#, r#"メ"#, r#"メ"#),
        (r#"ﾓ"#, r#"ﾓ"#, r#"ﾓ"#, r#"モ"#, r#"モ"#),
        (r#"ﾔ"#, r#"ﾔ"#, r#"ﾔ"#, r#"ヤ"#, r#"ヤ"#),
        (r#"ﾕ"#, r#"ﾕ"#, r#"ﾕ"#, r#"ユ"#, r#"ユ"#),
        (r#"ﾖ"#, r#"ﾖ"#, r#"ﾖ"#, r#"ヨ"#, r#"ヨ"#),
        (r#"ﾗ"#, r#"ﾗ"#, r#"ﾗ"#, r#"ラ"#, r#"ラ"#),
        (r#"ﾘ"#, r#"ﾘ"#, r#"ﾘ"#, r#"リ"#, r#"リ"#),
        (r#"ﾙ"#, r#"ﾙ"#, r#"ﾙ"#, r#"ル"#, r#"ル"#),
        (r#"ﾚ"#, r#"ﾚ"#, r#"ﾚ"#, r#"レ"#, r#"レ"#),
        (r#"ﾛ"#, r#"ﾛ"#, r#"ﾛ"#, r#"ロ"#, r#"ロ"#),
        (r#"ﾜ"#, r#"ﾜ"#, r#"ﾜ"#, r#"ワ"#, r#"ワ"#),
        (r#"ﾝ"#, r#"ﾝ"#, r#"ﾝ"#, r#"ン"#, r#"ン"#),
        (r#"ﾞ"#, r#"ﾞ"#, r#"ﾞ"#, r#"゙"#, r#"゙"#),
        (r#"ﾟ"#, r#"ﾟ"#, r#"ﾟ"#, r#"゚"#, r#"゚"#),
        (r#"ﾠ"#, r#"ﾠ"#, r#"ﾠ"#, r#"ᅠ"#, r#"ᅠ"#),
        (r#"ﾡ"#, r#"ﾡ"#, r#"ﾡ"#, r#"ᄀ"#, r#"ᄀ"#),
        (r#"ﾢ"#, r#"ﾢ"#, r#"ﾢ"#, r#"ᄁ"#, r#"ᄁ"#),
        (r#"ﾣ"#, r#"ﾣ"#, r#"ﾣ"#, r#"ᆪ"#, r#"ᆪ"#),
        (r#"ﾤ"#, r#"ﾤ"#, r#"ﾤ"#, r#"ᄂ"#, r#"ᄂ"#),
        (r#"ﾥ"#, r#"ﾥ"#, r#"ﾥ"#, r#"ᆬ"#, r#"ᆬ"#),
        (r#"ﾦ"#, r#"ﾦ"#, r#"ﾦ"#, r#"ᆭ"#, r#"ᆭ"#),
        (r#"ﾧ"#, r#"ﾧ"#, r#"ﾧ"#, r#"ᄃ"#, r#"ᄃ"#),
        (r#"ﾨ"#, r#"ﾨ"#, r#"ﾨ"#, r#"ᄄ"#, r#"ᄄ"#),
        (r#"ﾩ"#, r#"ﾩ"#, r#"ﾩ"#, r#"ᄅ"#, r#"ᄅ"#),
        (r#"ﾪ"#, r#"ﾪ"#, r#"ﾪ"#, r#"ᆰ"#, r#"ᆰ"#),
        (r#"ﾫ"#, r#"ﾫ"#, r#"ﾫ"#, r#"ᆱ"#, r#"ᆱ"#),
        (r#"ﾬ"#, r#"ﾬ"#, r#"ﾬ"#, r#"ᆲ"#, r#"ᆲ"#),
        (r#"ﾭ"#, r#"ﾭ"#, r#"ﾭ"#, r#"ᆳ"#, r#"ᆳ"#),
        (r#"ﾮ"#, r#"ﾮ"#, r#"ﾮ"#, r#"ᆴ"#, r#"ᆴ"#),
        (r#"ﾯ"#, r#"ﾯ"#, r#"ﾯ"#, r#"ᆵ"#, r#"ᆵ"#),
        (r#"ﾰ"#, r#"ﾰ"#, r#"ﾰ"#, r#"ᄚ"#, r#"ᄚ"#),
        (r#"ﾱ"#, r#"ﾱ"#, r#"ﾱ"#, r#"ᄆ"#, r#"ᄆ"#),
        (r#"ﾲ"#, r#"ﾲ"#, r#"ﾲ"#, r#"ᄇ"#, r#"ᄇ"#),
        (r#"ﾳ"#, r#"ﾳ"#, r#"ﾳ"#, r#"ᄈ"#, r#"ᄈ"#),
        (r#"ﾴ"#, r#"ﾴ"#, r#"ﾴ"#, r#"ᄡ"#, r#"ᄡ"#),
        (r#"ﾵ"#, r#"ﾵ"#, r#"ﾵ"#, r#"ᄉ"#, r#"ᄉ"#),
        (r#"ﾶ"#, r#"ﾶ"#, r#"ﾶ"#, r#"ᄊ"#, r#"ᄊ"#),
        (r#"ﾷ"#, r#"ﾷ"#, r#"ﾷ"#, r#"ᄋ"#, r#"ᄋ"#),
        (r#"ﾸ"#, r#"ﾸ"#, r#"ﾸ"#, r#"ᄌ"#, r#"ᄌ"#),
        (r#"ﾹ"#, r#"ﾹ"#, r#"ﾹ"#, r#"ᄍ"#, r#"ᄍ"#),
        (r#"ﾺ"#, r#"ﾺ"#, r#"ﾺ"#, r#"ᄎ"#, r#"ᄎ"#),
        (r#"ﾻ"#, r#"ﾻ"#, r#"ﾻ"#, r#"ᄏ"#, r#"ᄏ"#),
        (r#"ﾼ"#, r#"ﾼ"#, r#"ﾼ"#, r#"ᄐ"#, r#"ᄐ"#),
        (r#"ﾽ"#, r#"ﾽ"#, r#"ﾽ"#, r#"ᄑ"#, r#"ᄑ"#),
        (r#"ﾾ"#, r#"ﾾ"#, r#"ﾾ"#, r#"ᄒ"#, r#"ᄒ"#),
        (r#"ￂ"#, r#"ￂ"#, r#"ￂ"#, r#"ᅡ"#, r#"ᅡ"#),
        (r#"ￃ"#, r#"ￃ"#, r#"ￃ"#, r#"ᅢ"#, r#"ᅢ"#),
        (r#"ￄ"#, r#"ￄ"#, r#"ￄ"#, r#"ᅣ"#, r#"ᅣ"#),
        (r#"ￅ"#, r#"ￅ"#, r#"ￅ"#, r#"ᅤ"#, r#"ᅤ"#),
        (r#"ￆ"#, r#"ￆ"#, r#"ￆ"#, r#"ᅥ"#, r#"ᅥ"#),
        (r#"ￇ"#, r#"ￇ"#, r#"ￇ"#, r#"ᅦ"#, r#"ᅦ"#),
        (r#"ￊ"#, r#"ￊ"#, r#"ￊ"#, r#"ᅧ"#, r#"ᅧ"#),
        (r#"ￋ"#, r#"ￋ"#, r#"ￋ"#, r#"ᅨ"#, r#"ᅨ"#),
        (r#"ￌ"#, r#"ￌ"#, r#"ￌ"#, r#"ᅩ"#, r#"ᅩ"#),
        (r#"ￍ"#, r#"ￍ"#, r#"ￍ"#, r#"ᅪ"#, r#"ᅪ"#),
        (r#"ￎ"#, r#"ￎ"#, r#"ￎ"#, r#"ᅫ"#, r#"ᅫ"#),
        (r#"ￏ"#, r#"ￏ"#, r#"ￏ"#, r#"ᅬ"#, r#"ᅬ"#),
        (r#"ￒ"#, r#"ￒ"#, r#"ￒ"#, r#"ᅭ"#, r#"ᅭ"#),
        (r#"ￓ"#, r#"ￓ"#, r#"ￓ"#, r#"ᅮ"#, r#"ᅮ"#),
        (r#"ￔ"#, r#"ￔ"#, r#"ￔ"#, r#"ᅯ"#, r#"ᅯ"#),
        (r#"ￕ"#, r#"ￕ"#, r#"ￕ"#, r#"ᅰ"#, r#"ᅰ"#),
        (r#"ￖ"#, r#"ￖ"#, r#"ￖ"#, r#"ᅱ"#, r#"ᅱ"#),
        (r#"ￗ"#, r#"ￗ"#, r#"ￗ"#, r#"ᅲ"#, r#"ᅲ"#),
        (r#"ￚ"#, r#"ￚ"#, r#"ￚ"#, r#"ᅳ"#, r#"ᅳ"#),
        (r#"ￛ"#, r#"ￛ"#, r#"ￛ"#, r#"ᅴ"#, r#"ᅴ"#),
        (r#"ￜ"#, r#"ￜ"#, r#"ￜ"#, r#"ᅵ"#, r#"ᅵ"#),
        (r#"￠"#, r#"￠"#, r#"￠"#, r#"¢"#, r#"¢"#),
        (r#"￡"#, r#"￡"#, r#"￡"#, r#"£"#, r#"£"#),
        (r#"￢"#, r#"￢"#, r#"￢"#, r#"¬"#, r#"¬"#),
        (r#"￣"#, r#"￣"#, r#"￣"#, r#" ̄"#, r#" ̄"#),
        (r#"￤"#, r#"￤"#, r#"￤"#, r#"¦"#, r#"¦"#),
        (r#"￥"#, r#"￥"#, r#"￥"#, r#"¥"#, r#"¥"#),
        (r#"￦"#, r#"￦"#, r#"￦"#, r#"₩"#, r#"₩"#),
        (r#"￨"#, r#"￨"#, r#"￨"#, r#"│"#, r#"│"#),
        (r#"￩"#, r#"￩"#, r#"￩"#, r#"←"#, r#"←"#),
        (r#"￪"#, r#"￪"#, r#"￪"#, r#"↑"#, r#"↑"#),
        (r#"￫"#, r#"￫"#, r#"￫"#, r#"→"#, r#"→"#),
        (r#"￬"#, r#"￬"#, r#"￬"#, r#"↓"#, r#"↓"#),
        (r#"￭"#, r#"￭"#, r#"￭"#, r#"■"#, r#"■"#),
        (r#"￮"#, r#"￮"#, r#"￮"#, r#"○"#, r#"○"#),
        (r#"𐞁"#, r#"𐞁"#, r#"𐞁"#, r#"ː"#, r#"ː"#),
        (r#"𐞂"#, r#"𐞂"#, r#"𐞂"#, r#"ˑ"#, r#"ˑ"#),
        (r#"𐞃"#, r#"𐞃"#, r#"𐞃"#, r#"æ"#, r#"æ"#),
        (r#"𐞄"#, r#"𐞄"#, r#"𐞄"#, r#"ʙ"#, r#"ʙ"#),
        (r#"𐞅"#, r#"𐞅"#, r#"𐞅"#, r#"ɓ"#, r#"ɓ"#),
        (r#"𐞇"#, r#"𐞇"#, r#"𐞇"#, r#"ʣ"#, r#"ʣ"#),
        (r#"𐞈"#, r#"𐞈"#, r#"𐞈"#, r#"ꭦ"#, r#"ꭦ"#),
        (r#"𐞉"#, r#"𐞉"#, r#"𐞉"#, r#"ʥ"#, r#"ʥ"#),
        (r#"𐞊"#, r#"𐞊"#, r#"𐞊"#, r#"ʤ"#, r#"ʤ"#),
        (r#"𐞋"#, r#"𐞋"#, r#"𐞋"#, r#"ɖ"#, r#"ɖ"#),
        (r#"𐞌"#, r#"𐞌"#, r#"𐞌"#, r#"ɗ"#, r#"ɗ"#),
        (r#"𐞍"#, r#"𐞍"#, r#"𐞍"#, r#"ᶑ"#, r#"ᶑ"#),
        (r#"𐞎"#, r#"𐞎"#, r#"𐞎"#, r#"ɘ"#, r#"ɘ"#),
        (r#"𐞏"#, r#"𐞏"#, r#"𐞏"#, r#"ɞ"#, r#"ɞ"#),
        (r#"𐞐"#, r#"𐞐"#, r#"𐞐"#, r#"ʩ"#, r#"ʩ"#),
        (r#"𐞑"#, r#"𐞑"#, r#"𐞑"#, r#"ɤ"#, r#"ɤ"#),
        (r#"𐞒"#, r#"𐞒"#, r#"𐞒"#, r#"ɢ"#, r#"ɢ"#),
        (r#"𐞓"#, r#"𐞓"#, r#"𐞓"#, r#"ɠ"#, r#"ɠ"#),
        (r#"𐞔"#, r#"𐞔"#, r#"𐞔"#, r#"ʛ"#, r#"ʛ"#),
        (r#"𐞕"#, r#"𐞕"#, r#"𐞕"#, r#"ħ"#, r#"ħ"#),
        (r#"𐞖"#, r#"𐞖"#, r#"𐞖"#, r#"ʜ"#, r#"ʜ"#),
        (r#"𐞗"#, r#"𐞗"#, r#"𐞗"#, r#"ɧ"#, r#"ɧ"#),
        (r#"𐞘"#, r#"𐞘"#, r#"𐞘"#, r#"ʄ"#, r#"ʄ"#),
        (r#"𐞙"#, r#"𐞙"#, r#"𐞙"#, r#"ʪ"#, r#"ʪ"#),
        (r#"𐞚"#, r#"𐞚"#, r#"𐞚"#, r#"ʫ"#, r#"ʫ"#),
        (r#"𐞛"#, r#"𐞛"#, r#"𐞛"#, r#"ɬ"#, r#"ɬ"#),
        (r#"𐞜"#, r#"𐞜"#, r#"𐞜"#, r#"𝼄"#, r#"𝼄"#),
        (r#"𐞝"#, r#"𐞝"#, r#"𐞝"#, r#"ꞎ"#, r#"ꞎ"#),
        (r#"𐞞"#, r#"𐞞"#, r#"𐞞"#, r#"ɮ"#, r#"ɮ"#),
        (r#"𐞟"#, r#"𐞟"#, r#"𐞟"#, r#"𝼅"#, r#"𝼅"#),
        (r#"𐞠"#, r#"𐞠"#, r#"𐞠"#, r#"ʎ"#, r#"ʎ"#),
        (r#"𐞡"#, r#"𐞡"#, r#"𐞡"#, r#"𝼆"#, r#"𝼆"#),
        (r#"𐞢"#, r#"𐞢"#, r#"𐞢"#, r#"ø"#, r#"ø"#),
        (r#"𐞣"#, r#"𐞣"#, r#"𐞣"#, r#"ɶ"#, r#"ɶ"#),
        (r#"𐞤"#, r#"𐞤"#, r#"𐞤"#, r#"ɷ"#, r#"ɷ"#),
        (r#"𐞥"#, r#"𐞥"#, r#"𐞥"#, r#"q"#, r#"q"#),
        (r#"𐞦"#, r#"𐞦"#, r#"𐞦"#, r#"ɺ"#, r#"ɺ"#),
        (r#"𐞧"#, r#"𐞧"#, r#"𐞧"#, r#"𝼈"#, r#"𝼈"#),
        (r#"𐞨"#, r#"𐞨"#, r#"𐞨"#, r#"ɽ"#, r#"ɽ"#),
        (r#"𐞩"#, r#"𐞩"#, r#"𐞩"#, r#"ɾ"#, r#"ɾ"#),
        (r#"𐞪"#, r#"𐞪"#, r#"𐞪"#, r#"ʀ"#, r#"ʀ"#),
        (r#"𐞫"#, r#"𐞫"#, r#"𐞫"#, r#"ʨ"#, r#"ʨ"#),
        (r#"𐞬"#, r#"𐞬"#, r#"𐞬"#, r#"ʦ"#, r#"ʦ"#),
        (r#"𐞭"#, r#"𐞭"#, r#"𐞭"#, r#"ꭧ"#, r#"ꭧ"#),
        (r#"𐞮"#, r#"𐞮"#, r#"𐞮"#, r#"ʧ"#, r#"ʧ"#),
        (r#"𐞯"#, r#"𐞯"#, r#"𐞯"#, r#"ʈ"#, r#"ʈ"#),
        (r#"𐞰"#, r#"𐞰"#, r#"𐞰"#, r#"ⱱ"#, r#"ⱱ"#),
        (r#"𐞲"#, r#"𐞲"#, r#"𐞲"#, r#"ʏ"#, r#"ʏ"#),
        (r#"𐞳"#, r#"𐞳"#, r#"𐞳"#, r#"ʡ"#, r#"ʡ"#),
        (r#"𐞴"#, r#"𐞴"#, r#"𐞴"#, r#"ʢ"#, r#"ʢ"#),
        (r#"𐞵"#, r#"𐞵"#, r#"𐞵"#, r#"ʘ"#, r#"ʘ"#),
        (r#"𐞶"#, r#"𐞶"#, r#"𐞶"#, r#"ǀ"#, r#"ǀ"#),
        (r#"𐞷"#, r#"𐞷"#, r#"𐞷"#, r#"ǁ"#, r#"ǁ"#),
        (r#"𐞸"#, r#"𐞸"#, r#"𐞸"#, r#"ǂ"#, r#"ǂ"#),
        (r#"𐞹"#, r#"𐞹"#, r#"𐞹"#, r#"𝼊"#, r#"𝼊"#),
        (r#"𐞺"#, r#"𐞺"#, r#"𐞺"#, r#"𝼞"#, r#"𝼞"#),
        (r#"𑂚"#, r#"𑂚"#, r#"𑂚"#, r#"𑂚"#, r#"𑂚"#),
        (r#"𑂜"#, r#"𑂜"#, r#"𑂜"#, r#"𑂜"#, r#"𑂜"#),
        (r#"𑂫"#, r#"𑂫"#, r#"𑂫"#, r#"𑂫"#, r#"𑂫"#),
        (r#"𑄮"#, r#"𑄮"#, r#"𑄮"#, r#"𑄮"#, r#"𑄮"#),
        (r#"𑄯"#, r#"𑄯"#, r#"𑄯"#, r#"𑄯"#, r#"𑄯"#),
        (r#"𑍋"#, r#"𑍋"#, r#"𑍋"#, r#"𑍋"#, r#"𑍋"#),
        (r#"𑍌"#, r#"𑍌"#, r#"𑍌"#, r#"𑍌"#, r#"𑍌"#),
        (r#"𑒻"#, r#"𑒻"#, r#"𑒻"#, r#"𑒻"#, r#"𑒻"#),
        (r#"𑒼"#, r#"𑒼"#, r#"𑒼"#, r#"𑒼"#, r#"𑒼"#),
        (r#"𑒾"#, r#"𑒾"#, r#"𑒾"#, r#"𑒾"#, r#"𑒾"#),
        (r#"𑖺"#, r#"𑖺"#, r#"𑖺"#, r#"𑖺"#, r#"𑖺"#),
        (r#"𑖻"#, r#"𑖻"#, r#"𑖻"#, r#"𑖻"#, r#"𑖻"#),
        (r#"𑤸"#, r#"𑤸"#, r#"𑤸"#, r#"𑤸"#, r#"𑤸"#),
        (r#"𝅗𝅥"#, r#"𝅗𝅥"#, r#"𝅗𝅥"#, r#"𝅗𝅥"#, r#"𝅗𝅥"#),
        (r#"𝅘𝅥"#, r#"𝅘𝅥"#, r#"𝅘𝅥"#, r#"𝅘𝅥"#, r#"𝅘𝅥"#),
        (r#"𝅘𝅥𝅮"#, r#"𝅘𝅥𝅮"#, r#"𝅘𝅥𝅮"#, r#"𝅘𝅥𝅮"#, r#"𝅘𝅥𝅮"#),
        (r#"𝅘𝅥𝅯"#, r#"𝅘𝅥𝅯"#, r#"𝅘𝅥𝅯"#, r#"𝅘𝅥𝅯"#, r#"𝅘𝅥𝅯"#),
        (r#"𝅘𝅥𝅰"#, r#"𝅘𝅥𝅰"#, r#"𝅘𝅥𝅰"#, r#"𝅘𝅥𝅰"#, r#"𝅘𝅥𝅰"#),
        (r#"𝅘𝅥𝅱"#, r#"𝅘𝅥𝅱"#, r#"𝅘𝅥𝅱"#, r#"𝅘𝅥𝅱"#, r#"𝅘𝅥𝅱"#),
        (r#"𝅘𝅥𝅲"#, r#"𝅘𝅥𝅲"#, r#"𝅘𝅥𝅲"#, r#"𝅘𝅥𝅲"#, r#"𝅘𝅥𝅲"#),
        (r#"𝆹𝅥"#, r#"𝆹𝅥"#, r#"𝆹𝅥"#, r#"𝆹𝅥"#, r#"𝆹𝅥"#),
        (r#"𝆺𝅥"#, r#"𝆺𝅥"#, r#"𝆺𝅥"#, r#"𝆺𝅥"#, r#"𝆺𝅥"#),
        (r#"𝆹𝅥𝅮"#, r#"𝆹𝅥𝅮"#, r#"𝆹𝅥𝅮"#, r#"𝆹𝅥𝅮"#, r#"𝆹𝅥𝅮"#),
        (r#"𝆺𝅥𝅮"#, r#"𝆺𝅥𝅮"#, r#"𝆺𝅥𝅮"#, r#"𝆺𝅥𝅮"#, r#"𝆺𝅥𝅮"#),
        (r#"𝆹𝅥𝅯"#, r#"𝆹𝅥𝅯"#, r#"𝆹𝅥𝅯"#, r#"𝆹𝅥𝅯"#, r#"𝆹𝅥𝅯"#),
        (r#"𝆺𝅥𝅯"#, r#"𝆺𝅥𝅯"#, r#"𝆺𝅥𝅯"#, r#"𝆺𝅥𝅯"#, r#"𝆺𝅥𝅯"#),
        (r#"𝐀"#, r#"𝐀"#, r#"𝐀"#, r#"A"#, r#"A"#),
        (r#"𝐁"#, r#"𝐁"#, r#"𝐁"#, r#"B"#, r#"B"#),
        (r#"𝐂"#, r#"𝐂"#, r#"𝐂"#, r#"C"#, r#"C"#),
        (r#"𝐃"#, r#"𝐃"#, r#"𝐃"#, r#"D"#, r#"D"#),
        (r#"𝐄"#, r#"𝐄"#, r#"𝐄"#, r#"E"#, r#"E"#),
        (r#"𝐅"#, r#"𝐅"#, r#"𝐅"#, r#"F"#, r#"F"#),
        (r#"𝐆"#, r#"𝐆"#, r#"𝐆"#, r#"G"#, r#"G"#),
        (r#"𝐇"#, r#"𝐇"#, r#"𝐇"#, r#"H"#, r#"H"#),
        (r#"𝐈"#, r#"𝐈"#, r#"𝐈"#, r#"I"#, r#"I"#),
        (r#"𝐉"#, r#"𝐉"#, r#"𝐉"#, r#"J"#, r#"J"#),
        (r#"𝐊"#, r#"𝐊"#, r#"𝐊"#, r#"K"#, r#"K"#),
        (r#"𝐋"#, r#"𝐋"#, r#"𝐋"#, r#"L"#, r#"L"#),
        (r#"𝐌"#, r#"𝐌"#, r#"𝐌"#, r#"M"#, r#"M"#),
        (r#"𝐍"#, r#"𝐍"#, r#"𝐍"#, r#"N"#, r#"N"#),
        (r#"𝐎"#, r#"𝐎"#, r#"𝐎"#, r#"O"#, r#"O"#),
        (r#"𝐏"#, r#"𝐏"#, r#"𝐏"#, r#"P"#, r#"P"#),
        (r#"𝐐"#, r#"𝐐"#, r#"𝐐"#, r#"Q"#, r#"Q"#),
        (r#"𝐑"#, r#"𝐑"#, r#"𝐑"#, r#"R"#, r#"R"#),
        (r#"𝐒"#, r#"𝐒"#, r#"𝐒"#, r#"S"#, r#"S"#),
        (r#"𝐓"#, r#"𝐓"#, r#"𝐓"#, r#"T"#, r#"T"#),
        (r#"𝐔"#, r#"𝐔"#, r#"𝐔"#, r#"U"#, r#"U"#),
        (r#"𝐕"#, r#"𝐕"#, r#"𝐕"#, r#"V"#, r#"V"#),
        (r#"𝐖"#, r#"𝐖"#, r#"𝐖"#, r#"W"#, r#"W"#),
        (r#"𝐗"#, r#"𝐗"#, r#"𝐗"#, r#"X"#, r#"X"#),
        (r#"𝐘"#, r#"𝐘"#, r#"𝐘"#, r#"Y"#, r#"Y"#),
        (r#"𝐙"#, r#"𝐙"#, r#"𝐙"#, r#"Z"#, r#"Z"#),
        (r#"𝐚"#, r#"𝐚"#, r#"𝐚"#, r#"a"#, r#"a"#),
        (r#"𝐛"#, r#"𝐛"#, r#"𝐛"#, r#"b"#, r#"b"#),
        (r#"𝐜"#, r#"𝐜"#, r#"𝐜"#, r#"c"#, r#"c"#),
        (r#"𝐝"#, r#"𝐝"#, r#"𝐝"#, r#"d"#, r#"d"#),
        (r#"𝐞"#, r#"𝐞"#, r#"𝐞"#, r#"e"#, r#"e"#),
        (r#"𝐟"#, r#"𝐟"#, r#"𝐟"#, r#"f"#, r#"f"#),
        (r#"𝐠"#, r#"𝐠"#, r#"𝐠"#, r#"g"#, r#"g"#),
        (r#"𝐡"#, r#"𝐡"#, r#"𝐡"#, r#"h"#, r#"h"#),
        (r#"𝐢"#, r#"𝐢"#, r#"𝐢"#, r#"i"#, r#"i"#),
        (r#"𝐣"#, r#"𝐣"#, r#"𝐣"#, r#"j"#, r#"j"#),
        (r#"𝐤"#, r#"𝐤"#, r#"𝐤"#, r#"k"#, r#"k"#),
        (r#"𝐥"#, r#"𝐥"#, r#"𝐥"#, r#"l"#, r#"l"#),
        (r#"𝐦"#, r#"𝐦"#, r#"𝐦"#, r#"m"#, r#"m"#),
        (r#"𝐧"#, r#"𝐧"#, r#"𝐧"#, r#"n"#, r#"n"#),
        (r#"𝐨"#, r#"𝐨"#, r#"𝐨"#, r#"o"#, r#"o"#),
        (r#"𝐩"#, r#"𝐩"#, r#"𝐩"#, r#"p"#, r#"p"#),
        (r#"𝐪"#, r#"𝐪"#, r#"𝐪"#, r#"q"#, r#"q"#),
        (r#"𝐫"#, r#"𝐫"#, r#"𝐫"#, r#"r"#, r#"r"#),
        (r#"𝐬"#, r#"𝐬"#, r#"𝐬"#, r#"s"#, r#"s"#),
        (r#"𝐭"#, r#"𝐭"#, r#"𝐭"#, r#"t"#, r#"t"#),
        (r#"𝐮"#, r#"𝐮"#, r#"𝐮"#, r#"u"#, r#"u"#),
        (r#"𝐯"#, r#"𝐯"#, r#"𝐯"#, r#"v"#, r#"v"#),
        (r#"𝐰"#, r#"𝐰"#, r#"𝐰"#, r#"w"#, r#"w"#),
        (r#"𝐱"#, r#"𝐱"#, r#"𝐱"#, r#"x"#, r#"x"#),
        (r#"𝐲"#, r#"𝐲"#, r#"𝐲"#, r#"y"#, r#"y"#),
        (r#"𝐳"#, r#"𝐳"#, r#"𝐳"#, r#"z"#, r#"z"#),
        (r#"𝐴"#, r#"𝐴"#, r#"𝐴"#, r#"A"#, r#"A"#),
        (r#"𝐵"#, r#"𝐵"#, r#"𝐵"#, r#"B"#, r#"B"#),
        (r#"𝐶"#, r#"𝐶"#, r#"𝐶"#, r#"C"#, r#"C"#),
        (r#"𝐷"#, r#"𝐷"#, r#"𝐷"#, r#"D"#, r#"D"#),
        (r#"𝐸"#, r#"𝐸"#, r#"𝐸"#, r#"E"#, r#"E"#),
        (r#"𝐹"#, r#"𝐹"#, r#"𝐹"#, r#"F"#, r#"F"#),
        (r#"𝐺"#, r#"𝐺"#, r#"𝐺"#, r#"G"#, r#"G"#),
        (r#"𝐻"#, r#"𝐻"#, r#"𝐻"#, r#"H"#, r#"H"#),
        (r#"𝐼"#, r#"𝐼"#, r#"𝐼"#, r#"I"#, r#"I"#),
        (r#"𝐽"#, r#"𝐽"#, r#"𝐽"#, r#"J"#, r#"J"#),
        (r#"𝐾"#, r#"𝐾"#, r#"𝐾"#, r#"K"#, r#"K"#),
        (r#"𝐿"#, r#"𝐿"#, r#"𝐿"#, r#"L"#, r#"L"#),
        (r#"𝑀"#, r#"𝑀"#, r#"𝑀"#, r#"M"#, r#"M"#),
        (r#"𝑁"#, r#"𝑁"#, r#"𝑁"#, r#"N"#, r#"N"#),
        (r#"𝑂"#, r#"𝑂"#, r#"𝑂"#, r#"O"#, r#"O"#),
        (r#"𝑃"#, r#"𝑃"#, r#"𝑃"#, r#"P"#, r#"P"#),
        (r#"𝑄"#, r#"𝑄"#, r#"𝑄"#, r#"Q"#, r#"Q"#),
        (r#"𝑅"#, r#"𝑅"#, r#"𝑅"#, r#"R"#, r#"R"#),
        (r#"𝑆"#, r#"𝑆"#, r#"𝑆"#, r#"S"#, r#"S"#),
        (r#"𝑇"#, r#"𝑇"#, r#"𝑇"#, r#"T"#, r#"T"#),
        (r#"𝑈"#, r#"𝑈"#, r#"𝑈"#, r#"U"#, r#"U"#),
        (r#"𝑉"#, r#"𝑉"#, r#"𝑉"#, r#"V"#, r#"V"#),
        (r#"𝑊"#, r#"𝑊"#, r#"𝑊"#, r#"W"#, r#"W"#),
        (r#"𝑋"#, r#"𝑋"#, r#"𝑋"#, r#"X"#, r#"X"#),
        (r#"𝑌"#, r#"𝑌"#, r#"𝑌"#, r#"Y"#, r#"Y"#),
        (r#"𝑍"#, r#"𝑍"#, r#"𝑍"#, r#"Z"#, r#"Z"#),
        (r#"𝑎"#, r#"𝑎"#, r#"𝑎"#, r#"a"#, r#"a"#),
        (r#"𝑏"#, r#"𝑏"#, r#"𝑏"#, r#"b"#, r#"b"#),
        (r#"𝑐"#, r#"𝑐"#, r#"𝑐"#, r#"c"#, r#"c"#),
        (r#"𝑑"#, r#"𝑑"#, r#"𝑑"#, r#"d"#, r#"d"#),
        (r#"𝑒"#, r#"𝑒"#, r#"𝑒"#, r#"e"#, r#"e"#),
        (r#"𝑓"#, r#"𝑓"#, r#"𝑓"#, r#"f"#, r#"f"#),
        (r#"𝑔"#, r#"𝑔"#, r#"𝑔"#, r#"g"#, r#"g"#),
        (r#"𝑖"#, r#"𝑖"#, r#"𝑖"#, r#"i"#, r#"i"#),
        (r#"𝑗"#, r#"𝑗"#, r#"𝑗"#, r#"j"#, r#"j"#),
        (r#"𝑘"#, r#"𝑘"#, r#"𝑘"#, r#"k"#, r#"k"#),
        (r#"𝑙"#, r#"𝑙"#, r#"𝑙"#, r#"l"#, r#"l"#),
        (r#"𝑚"#, r#"𝑚"#, r#"𝑚"#, r#"m"#, r#"m"#),
        (r#"𝑛"#, r#"𝑛"#, r#"𝑛"#, r#"n"#, r#"n"#),
        (r#"𝑜"#, r#"𝑜"#, r#"𝑜"#, r#"o"#, r#"o"#),
        (r#"𝑝"#, r#"𝑝"#, r#"𝑝"#, r#"p"#, r#"p"#),
        (r#"𝑞"#, r#"𝑞"#, r#"𝑞"#, r#"q"#, r#"q"#),
        (r#"𝑟"#, r#"𝑟"#, r#"𝑟"#, r#"r"#, r#"r"#),
        (r#"𝑠"#, r#"𝑠"#, r#"𝑠"#, r#"s"#, r#"s"#),
        (r#"𝑡"#, r#"𝑡"#, r#"𝑡"#, r#"t"#, r#"t"#),
        (r#"𝑢"#, r#"𝑢"#, r#"𝑢"#, r#"u"#, r#"u"#),
        (r#"𝑣"#, r#"𝑣"#, r#"𝑣"#, r#"v"#, r#"v"#),
        (r#"𝑤"#, r#"𝑤"#, r#"𝑤"#, r#"w"#, r#"w"#),
        (r#"𝑥"#, r#"𝑥"#, r#"𝑥"#, r#"x"#, r#"x"#),
        (r#"𝑦"#, r#"𝑦"#, r#"𝑦"#, r#"y"#, r#"y"#),
        (r#"𝑧"#, r#"𝑧"#, r#"𝑧"#, r#"z"#, r#"z"#),
        (r#"𝑨"#, r#"𝑨"#, r#"𝑨"#, r#"A"#, r#"A"#),
        (r#"𝑩"#, r#"𝑩"#, r#"𝑩"#, r#"B"#, r#"B"#),
        (r#"𝑪"#, r#"𝑪"#, r#"𝑪"#, r#"C"#, r#"C"#),
        (r#"𝑫"#, r#"𝑫"#, r#"𝑫"#, r#"D"#, r#"D"#),
        (r#"𝑬"#, r#"𝑬"#, r#"𝑬"#, r#"E"#, r#"E"#),
        (r#"𝑭"#, r#"𝑭"#, r#"𝑭"#, r#"F"#, r#"F"#),
        (r#"𝑮"#, r#"𝑮"#, r#"𝑮"#, r#"G"#, r#"G"#),
        (r#"𝑯"#, r#"𝑯"#, r#"𝑯"#, r#"H"#, r#"H"#),
        (r#"𝑰"#, r#"𝑰"#, r#"𝑰"#, r#"I"#, r#"I"#),
        (r#"𝑱"#, r#"𝑱"#, r#"𝑱"#, r#"J"#, r#"J"#),
        (r#"𝑲"#, r#"𝑲"#, r#"𝑲"#, r#"K"#, r#"K"#),
        (r#"𝑳"#, r#"𝑳"#, r#"𝑳"#, r#"L"#, r#"L"#),
        (r#"𝑴"#, r#"𝑴"#, r#"𝑴"#, r#"M"#, r#"M"#),
        (r#"𝑵"#, r#"𝑵"#, r#"𝑵"#, r#"N"#, r#"N"#),
        (r#"𝑶"#, r#"𝑶"#, r#"𝑶"#, r#"O"#, r#"O"#),
        (r#"𝑷"#, r#"𝑷"#, r#"𝑷"#, r#"P"#, r#"P"#),
        (r#"𝑸"#, r#"𝑸"#, r#"𝑸"#, r#"Q"#, r#"Q"#),
        (r#"𝑹"#, r#"𝑹"#, r#"𝑹"#, r#"R"#, r#"R"#),
        (r#"𝑺"#, r#"𝑺"#, r#"𝑺"#, r#"S"#, r#"S"#),
        (r#"𝑻"#, r#"𝑻"#, r#"𝑻"#, r#"T"#, r#"T"#),
        (r#"𝑼"#, r#"𝑼"#, r#"𝑼"#, r#"U"#, r#"U"#),
        (r#"𝑽"#, r#"𝑽"#, r#"𝑽"#, r#"V"#, r#"V"#),
        (r#"𝑾"#, r#"𝑾"#, r#"𝑾"#, r#"W"#, r#"W"#),
        (r#"𝑿"#, r#"𝑿"#, r#"𝑿"#, r#"X"#, r#"X"#),
        (r#"𝒀"#, r#"𝒀"#, r#"𝒀"#, r#"Y"#, r#"Y"#),
        (r#"𝒁"#, r#"𝒁"#, r#"𝒁"#, r#"Z"#, r#"Z"#),
        (r#"𝒂"#, r#"𝒂"#, r#"𝒂"#, r#"a"#, r#"a"#),
        (r#"𝒃"#, r#"𝒃"#, r#"𝒃"#, r#"b"#, r#"b"#),
        (r#"𝒄"#, r#"𝒄"#, r#"𝒄"#, r#"c"#, r#"c"#),
        (r#"𝒅"#, r#"𝒅"#, r#"𝒅"#, r#"d"#, r#"d"#),
        (r#"𝒆"#, r#"𝒆"#, r#"𝒆"#, r#"e"#, r#"e"#),
        (r#"𝒇"#, r#"𝒇"#, r#"𝒇"#, r#"f"#, r#"f"#),
        (r#"𝒈"#, r#"𝒈"#, r#"𝒈"#, r#"g"#, r#"g"#),
        (r#"𝒉"#, r#"𝒉"#, r#"𝒉"#, r#"h"#, r#"h"#),
        (r#"𝒊"#, r#"𝒊"#, r#"𝒊"#, r#"i"#, r#"i"#),
        (r#"𝒋"#, r#"𝒋"#, r#"𝒋"#, r#"j"#, r#"j"#),
        (r#"𝒌"#, r#"𝒌"#, r#"𝒌"#, r#"k"#, r#"k"#),
        (r#"𝒍"#, r#"𝒍"#, r#"𝒍"#, r#"l"#, r#"l"#),
        (r#"𝒎"#, r#"𝒎"#, r#"𝒎"#, r#"m"#, r#"m"#),
        (r#"𝒏"#, r#"𝒏"#, r#"𝒏"#, r#"n"#, r#"n"#),
        (r#"𝒐"#, r#"𝒐"#, r#"𝒐"#, r#"o"#, r#"o"#),
        (r#"𝒑"#, r#"𝒑"#, r#"𝒑"#, r#"p"#, r#"p"#),
        (r#"𝒒"#, r#"𝒒"#, r#"𝒒"#, r#"q"#, r#"q"#),
        (r#"𝒓"#, r#"𝒓"#, r#"𝒓"#, r#"r"#, r#"r"#),
        (r#"𝒔"#, r#"𝒔"#, r#"𝒔"#, r#"s"#, r#"s"#),
        (r#"𝒕"#, r#"𝒕"#, r#"𝒕"#, r#"t"#, r#"t"#),
        (r#"𝒖"#, r#"𝒖"#, r#"𝒖"#, r#"u"#, r#"u"#),
        (r#"𝒗"#, r#"𝒗"#, r#"𝒗"#, r#"v"#, r#"v"#),
        (r#"𝒘"#, r#"𝒘"#, r#"𝒘"#, r#"w"#, r#"w"#),
        (r#"𝒙"#, r#"𝒙"#, r#"𝒙"#, r#"x"#, r#"x"#),
        (r#"𝒚"#, r#"𝒚"#, r#"𝒚"#, r#"y"#, r#"y"#),
        (r#"𝒛"#, r#"𝒛"#, r#"𝒛"#, r#"z"#, r#"z"#),
        (r#"𝒜"#, r#"𝒜"#, r#"𝒜"#, r#"A"#, r#"A"#),
        (r#"𝒞"#, r#"𝒞"#, r#"𝒞"#, r#"C"#, r#"C"#),
        (r#"𝒟"#, r#"𝒟"#, r#"𝒟"#, r#"D"#, r#"D"#),
        (r#"𝒢"#, r#"𝒢"#, r#"𝒢"#, r#"G"#, r#"G"#),
        (r#"𝒥"#, r#"𝒥"#, r#"𝒥"#, r#"J"#, r#"J"#),
        (r#"𝒦"#, r#"𝒦"#, r#"𝒦"#, r#"K"#, r#"K"#),
        (r#"𝒩"#, r#"𝒩"#, r#"𝒩"#, r#"N"#, r#"N"#),
        (r#"𝒪"#, r#"𝒪"#, r#"𝒪"#, r#"O"#, r#"O"#),
        (r#"𝒫"#, r#"𝒫"#, r#"𝒫"#, r#"P"#, r#"P"#),
        (r#"𝒬"#, r#"𝒬"#, r#"𝒬"#, r#"Q"#, r#"Q"#),
        (r#"𝒮"#, r#"𝒮"#, r#"𝒮"#, r#"S"#, r#"S"#),
        (r#"𝒯"#, r#"𝒯"#, r#"𝒯"#, r#"T"#, r#"T"#),
        (r#"𝒰"#, r#"𝒰"#, r#"𝒰"#, r#"U"#, r#"U"#),
        (r#"𝒱"#, r#"𝒱"#, r#"𝒱"#, r#"V"#, r#"V"#),
        (r#"𝒲"#, r#"𝒲"#, r#"𝒲"#, r#"W"#, r#"W"#),
        (r#"𝒳"#, r#"𝒳"#, r#"𝒳"#, r#"X"#, r#"X"#),
        (r#"𝒴"#, r#"𝒴"#, r#"𝒴"#, r#"Y"#, r#"Y"#),
        (r#"𝒵"#, r#"𝒵"#, r#"𝒵"#, r#"Z"#, r#"Z"#),
        (r#"𝒶"#, r#"𝒶"#, r#"𝒶"#, r#"a"#, r#"a"#),
        (r#"𝒷"#, r#"𝒷"#, r#"𝒷"#, r#"b"#, r#"b"#),
        (r#"𝒸"#, r#"𝒸"#, r#"𝒸"#, r#"c"#, r#"c"#),
        (r#"𝒹"#, r#"𝒹"#, r#"𝒹"#, r#"d"#, r#"d"#),
        (r#"𝒻"#, r#"𝒻"#, r#"𝒻"#, r#"f"#, r#"f"#),
        (r#"𝒽"#, r#"𝒽"#, r#"𝒽"#, r#"h"#, r#"h"#),
        (r#"𝒾"#, r#"𝒾"#, r#"𝒾"#, r#"i"#, r#"i"#),
        (r#"𝒿"#, r#"𝒿"#, r#"𝒿"#, r#"j"#, r#"j"#),
        (r#"𝓀"#, r#"𝓀"#, r#"𝓀"#, r#"k"#, r#"k"#),
        (r#"𝓁"#, r#"𝓁"#, r#"𝓁"#, r#"l"#, r#"l"#),
        (r#"𝓂"#, r#"𝓂"#, r#"𝓂"#, r#"m"#, r#"m"#),
        (r#"𝓃"#, r#"𝓃"#, r#"𝓃"#, r#"n"#, r#"n"#),
        (r#"𝓅"#, r#"𝓅"#, r#"𝓅"#, r#"p"#, r#"p"#),
        (r#"𝓆"#, r#"𝓆"#, r#"𝓆"#, r#"q"#, r#"q"#),
        (r#"𝓇"#, r#"𝓇"#, r#"𝓇"#, r#"r"#, r#"r"#),
        (r#"𝓈"#, r#"𝓈"#, r#"𝓈"#, r#"s"#, r#"s"#),
        (r#"𝓉"#, r#"𝓉"#, r#"𝓉"#, r#"t"#, r#"t"#),
        (r#"𝓊"#, r#"𝓊"#, r#"𝓊"#, r#"u"#, r#"u"#),
        (r#"𝓋"#, r#"𝓋"#, r#"𝓋"#, r#"v"#, r#"v"#),
        (r#"𝓌"#, r#"𝓌"#, r#"𝓌"#, r#"w"#, r#"w"#),
        (r#"𝓍"#, r#"𝓍"#, r#"𝓍"#, r#"x"#, r#"x"#),
        (r#"𝓎"#, r#"𝓎"#, r#"𝓎"#, r#"y"#, r#"y"#),
        (r#"𝓏"#, r#"𝓏"#, r#"𝓏"#, r#"z"#, r#"z"#),
        (r#"𝓐"#, r#"𝓐"#, r#"𝓐"#, r#"A"#, r#"A"#),
        (r#"𝓑"#, r#"𝓑"#, r#"𝓑"#, r#"B"#, r#"B"#),
        (r#"𝓒"#, r#"𝓒"#, r#"𝓒"#, r#"C"#, r#"C"#),
        (r#"𝓓"#, r#"𝓓"#, r#"𝓓"#, r#"D"#, r#"D"#),
        (r#"𝓔"#, r#"𝓔"#, r#"𝓔"#, r#"E"#, r#"E"#),
        (r#"𝓕"#, r#"𝓕"#, r#"𝓕"#, r#"F"#, r#"F"#),
        (r#"𝓖"#, r#"𝓖"#, r#"𝓖"#, r#"G"#, r#"G"#),
        (r#"𝓗"#, r#"𝓗"#, r#"𝓗"#, r#"H"#, r#"H"#),
        (r#"𝓘"#, r#"𝓘"#, r#"𝓘"#, r#"I"#, r#"I"#),
        (r#"𝓙"#, r#"𝓙"#, r#"𝓙"#, r#"J"#, r#"J"#),
        (r#"𝓚"#, r#"𝓚"#, r#"𝓚"#, r#"K"#, r#"K"#),
        (r#"𝓛"#, r#"𝓛"#, r#"𝓛"#, r#"L"#, r#"L"#),
        (r#"𝓜"#, r#"𝓜"#, r#"𝓜"#, r#"M"#, r#"M"#),
        (r#"𝓝"#, r#"𝓝"#, r#"𝓝"#, r#"N"#, r#"N"#),
        (r#"𝓞"#, r#"𝓞"#, r#"𝓞"#, r#"O"#, r#"O"#),
        (r#"𝓟"#, r#"𝓟"#, r#"𝓟"#, r#"P"#, r#"P"#),
        (r#"𝓠"#, r#"𝓠"#, r#"𝓠"#, r#"Q"#, r#"Q"#),
        (r#"𝓡"#, r#"𝓡"#, r#"𝓡"#, r#"R"#, r#"R"#),
        (r#"𝓢"#, r#"𝓢"#, r#"𝓢"#, r#"S"#, r#"S"#),
        (r#"𝓣"#, r#"𝓣"#, r#"𝓣"#, r#"T"#, r#"T"#),
        (r#"𝓤"#, r#"𝓤"#, r#"𝓤"#, r#"U"#, r#"U"#),
        (r#"𝓥"#, r#"𝓥"#, r#"𝓥"#, r#"V"#, r#"V"#),
        (r#"𝓦"#, r#"𝓦"#, r#"𝓦"#, r#"W"#, r#"W"#),
        (r#"𝓧"#, r#"𝓧"#, r#"𝓧"#, r#"X"#, r#"X"#),
        (r#"𝓨"#, r#"𝓨"#, r#"𝓨"#, r#"Y"#, r#"Y"#),
        (r#"𝓩"#, r#"𝓩"#, r#"𝓩"#, r#"Z"#, r#"Z"#),
        (r#"𝓪"#, r#"𝓪"#, r#"𝓪"#, r#"a"#, r#"a"#),
        (r#"𝓫"#, r#"𝓫"#, r#"𝓫"#, r#"b"#, r#"b"#),
        (r#"𝓬"#, r#"𝓬"#, r#"𝓬"#, r#"c"#, r#"c"#),
        (r#"𝓭"#, r#"𝓭"#, r#"𝓭"#, r#"d"#, r#"d"#),
        (r#"𝓮"#, r#"𝓮"#, r#"𝓮"#, r#"e"#, r#"e"#),
        (r#"𝓯"#, r#"𝓯"#, r#"𝓯"#, r#"f"#, r#"f"#),
        (r#"𝓰"#, r#"𝓰"#, r#"𝓰"#, r#"g"#, r#"g"#),
        (r#"𝓱"#, r#"𝓱"#, r#"𝓱"#, r#"h"#, r#"h"#),
        (r#"𝓲"#, r#"𝓲"#, r#"𝓲"#, r#"i"#, r#"i"#),
        (r#"𝓳"#, r#"𝓳"#, r#"𝓳"#, r#"j"#, r#"j"#),
        (r#"𝓴"#, r#"𝓴"#, r#"𝓴"#, r#"k"#, r#"k"#),
        (r#"𝓵"#, r#"𝓵"#, r#"𝓵"#, r#"l"#, r#"l"#),
        (r#"𝓶"#, r#"𝓶"#, r#"𝓶"#, r#"m"#, r#"m"#),
        (r#"𝓷"#, r#"𝓷"#, r#"𝓷"#, r#"n"#, r#"n"#),
        (r#"𝓸"#, r#"𝓸"#, r#"𝓸"#, r#"o"#, r#"o"#),
        (r#"𝓹"#, r#"𝓹"#, r#"𝓹"#, r#"p"#, r#"p"#),
        (r#"𝓺"#, r#"𝓺"#, r#"𝓺"#, r#"q"#, r#"q"#),
        (r#"𝓻"#, r#"𝓻"#, r#"𝓻"#, r#"r"#, r#"r"#),
        (r#"𝓼"#, r#"𝓼"#, r#"𝓼"#, r#"s"#, r#"s"#),
        (r#"𝓽"#, r#"𝓽"#, r#"𝓽"#, r#"t"#, r#"t"#),
        (r#"𝓾"#, r#"𝓾"#, r#"𝓾"#, r#"u"#, r#"u"#),
        (r#"𝓿"#, r#"𝓿"#, r#"𝓿"#, r#"v"#, r#"v"#),
        (r#"𝔀"#, r#"𝔀"#, r#"𝔀"#, r#"w"#, r#"w"#),
        (r#"𝔁"#, r#"𝔁"#, r#"𝔁"#, r#"x"#, r#"x"#),
        (r#"𝔂"#, r#"𝔂"#, r#"𝔂"#, r#"y"#, r#"y"#),
        (r#"𝔃"#, r#"𝔃"#, r#"𝔃"#, r#"z"#, r#"z"#),
        (r#"𝔄"#, r#"𝔄"#, r#"𝔄"#, r#"A"#, r#"A"#),
        (r#"𝔅"#, r#"𝔅"#, r#"𝔅"#, r#"B"#, r#"B"#),
        (r#"𝔇"#, r#"𝔇"#, r#"𝔇"#, r#"D"#, r#"D"#),
        (r#"𝔈"#, r#"𝔈"#, r#"𝔈"#, r#"E"#, r#"E"#),
        (r#"𝔉"#, r#"𝔉"#, r#"𝔉"#, r#"F"#, r#"F"#),
        (r#"𝔊"#, r#"𝔊"#, r#"𝔊"#, r#"G"#, r#"G"#),
        (r#"𝔍"#, r#"𝔍"#, r#"𝔍"#, r#"J"#, r#"J"#),
        (r#"𝔎"#, r#"𝔎"#, r#"𝔎"#, r#"K"#, r#"K"#),
        (r#"𝔏"#, r#"𝔏"#, r#"𝔏"#, r#"L"#, r#"L"#),
        (r#"𝔐"#, r#"𝔐"#, r#"𝔐"#, r#"M"#, r#"M"#),
        (r#"𝔑"#, r#"𝔑"#, r#"𝔑"#, r#"N"#, r#"N"#),
        (r#"𝔒"#, r#"𝔒"#, r#"𝔒"#, r#"O"#, r#"O"#),
        (r#"𝔓"#, r#"𝔓"#, r#"𝔓"#, r#"P"#, r#"P"#),
        (r#"𝔔"#, r#"𝔔"#, r#"𝔔"#, r#"Q"#, r#"Q"#),
        (r#"𝔖"#, r#"𝔖"#, r#"𝔖"#, r#"S"#, r#"S"#),
        (r#"𝔗"#, r#"𝔗"#, r#"𝔗"#, r#"T"#, r#"T"#),
        (r#"𝔘"#, r#"𝔘"#, r#"𝔘"#, r#"U"#, r#"U"#),
        (r#"𝔙"#, r#"𝔙"#, r#"𝔙"#, r#"V"#, r#"V"#),
        (r#"𝔚"#, r#"𝔚"#, r#"𝔚"#, r#"W"#, r#"W"#),
        (r#"𝔛"#, r#"𝔛"#, r#"𝔛"#, r#"X"#, r#"X"#),
        (r#"𝔜"#, r#"𝔜"#, r#"𝔜"#, r#"Y"#, r#"Y"#),
        (r#"𝔞"#, r#"𝔞"#, r#"𝔞"#, r#"a"#, r#"a"#),
        (r#"𝔟"#, r#"𝔟"#, r#"𝔟"#, r#"b"#, r#"b"#),
        (r#"𝔠"#, r#"𝔠"#, r#"𝔠"#, r#"c"#, r#"c"#),
        (r#"𝔡"#, r#"𝔡"#, r#"𝔡"#, r#"d"#, r#"d"#),
        (r#"𝔢"#, r#"𝔢"#, r#"𝔢"#, r#"e"#, r#"e"#),
        (r#"𝔣"#, r#"𝔣"#, r#"𝔣"#, r#"f"#, r#"f"#),
        (r#"𝔤"#, r#"𝔤"#, r#"𝔤"#, r#"g"#, r#"g"#),
        (r#"𝔥"#, r#"𝔥"#, r#"𝔥"#, r#"h"#, r#"h"#),
        (r#"𝔦"#, r#"𝔦"#, r#"𝔦"#, r#"i"#, r#"i"#),
        (r#"𝔧"#, r#"𝔧"#, r#"𝔧"#, r#"j"#, r#"j"#),
        (r#"𝔨"#, r#"𝔨"#, r#"𝔨"#, r#"k"#, r#"k"#),
        (r#"𝔩"#, r#"𝔩"#, r#"𝔩"#, r#"l"#, r#"l"#),
        (r#"𝔪"#, r#"𝔪"#, r#"𝔪"#, r#"m"#, r#"m"#),
        (r#"𝔫"#, r#"𝔫"#, r#"𝔫"#, r#"n"#, r#"n"#),
        (r#"𝔬"#, r#"𝔬"#, r#"𝔬"#, r#"o"#, r#"o"#),
        (r#"𝔭"#, r#"𝔭"#, r#"𝔭"#, r#"p"#, r#"p"#),
        (r#"𝔮"#, r#"𝔮"#, r#"𝔮"#, r#"q"#, r#"q"#),
        (r#"𝔯"#, r#"𝔯"#, r#"𝔯"#, r#"r"#, r#"r"#),
        (r#"𝔰"#, r#"𝔰"#, r#"𝔰"#, r#"s"#, r#"s"#),
        (r#"𝔱"#, r#"𝔱"#, r#"𝔱"#, r#"t"#, r#"t"#),
        (r#"𝔲"#, r#"𝔲"#, r#"𝔲"#, r#"u"#, r#"u"#),
        (r#"𝔳"#, r#"𝔳"#, r#"𝔳"#, r#"v"#, r#"v"#),
        (r#"𝔴"#, r#"𝔴"#, r#"𝔴"#, r#"w"#, r#"w"#),
        (r#"𝔵"#, r#"𝔵"#, r#"𝔵"#, r#"x"#, r#"x"#),
        (r#"𝔶"#, r#"𝔶"#, r#"𝔶"#, r#"y"#, r#"y"#),
        (r#"𝔷"#, r#"𝔷"#, r#"𝔷"#, r#"z"#, r#"z"#),
        (r#"𝔸"#, r#"𝔸"#, r#"𝔸"#, r#"A"#, r#"A"#),
        (r#"𝔹"#, r#"𝔹"#, r#"𝔹"#, r#"B"#, r#"B"#),
        (r#"𝔻"#, r#"𝔻"#, r#"𝔻"#, r#"D"#, r#"D"#),
        (r#"𝔼"#, r#"𝔼"#, r#"𝔼"#, r#"E"#, r#"E"#),
        (r#"𝔽"#, r#"𝔽"#, r#"𝔽"#, r#"F"#, r#"F"#),
        (r#"𝔾"#, r#"𝔾"#, r#"𝔾"#, r#"G"#, r#"G"#),
        (r#"𝕀"#, r#"𝕀"#, r#"𝕀"#, r#"I"#, r#"I"#),
        (r#"𝕁"#, r#"𝕁"#, r#"𝕁"#, r#"J"#, r#"J"#),
        (r#"𝕂"#, r#"𝕂"#, r#"𝕂"#, r#"K"#, r#"K"#),
        (r#"𝕃"#, r#"𝕃"#, r#"𝕃"#, r#"L"#, r#"L"#),
        (r#"𝕄"#, r#"𝕄"#, r#"𝕄"#, r#"M"#, r#"M"#),
        (r#"𝕆"#, r#"𝕆"#, r#"𝕆"#, r#"O"#, r#"O"#),
        (r#"𝕊"#, r#"𝕊"#, r#"𝕊"#, r#"S"#, r#"S"#),
        (r#"𝕋"#, r#"𝕋"#, r#"𝕋"#, r#"T"#, r#"T"#),
        (r#"𝕌"#, r#"𝕌"#, r#"𝕌"#, r#"U"#, r#"U"#),
        (r#"𝕍"#, r#"𝕍"#, r#"𝕍"#, r#"V"#, r#"V"#),
        (r#"𝕎"#, r#"𝕎"#, r#"𝕎"#, r#"W"#, r#"W"#),
        (r#"𝕏"#, r#"𝕏"#, r#"𝕏"#, r#"X"#, r#"X"#),
        (r#"𝕐"#, r#"𝕐"#, r#"𝕐"#, r#"Y"#, r#"Y"#),
        (r#"𝕒"#, r#"𝕒"#, r#"𝕒"#, r#"a"#, r#"a"#),
        (r#"𝕓"#, r#"𝕓"#, r#"𝕓"#, r#"b"#, r#"b"#),
        (r#"𝕔"#, r#"𝕔"#, r#"𝕔"#, r#"c"#, r#"c"#),
        (r#"𝕕"#, r#"𝕕"#, r#"𝕕"#, r#"d"#, r#"d"#),
        (r#"𝕖"#, r#"𝕖"#, r#"𝕖"#, r#"e"#, r#"e"#),
        (r#"𝕗"#, r#"𝕗"#, r#"𝕗"#, r#"f"#, r#"f"#),
        (r#"𝕘"#, r#"𝕘"#, r#"𝕘"#, r#"g"#, r#"g"#),
        (r#"𝕙"#, r#"𝕙"#, r#"𝕙"#, r#"h"#, r#"h"#),
        (r#"𝕚"#, r#"𝕚"#, r#"𝕚"#, r#"i"#, r#"i"#),
        (r#"𝕛"#, r#"𝕛"#, r#"𝕛"#, r#"j"#, r#"j"#),
        (r#"𝕜"#, r#"𝕜"#, r#"𝕜"#, r#"k"#, r#"k"#),
        (r#"𝕝"#, r#"𝕝"#, r#"𝕝"#, r#"l"#, r#"l"#),
        (r#"𝕞"#, r#"𝕞"#, r#"𝕞"#, r#"m"#, r#"m"#),
        (r#"𝕟"#, r#"𝕟"#, r#"𝕟"#, r#"n"#, r#"n"#),
        (r#"𝕠"#, r#"𝕠"#, r#"𝕠"#, r#"o"#, r#"o"#),
        (r#"𝕡"#, r#"𝕡"#, r#"𝕡"#, r#"p"#, r#"p"#),
        (r#"𝕢"#, r#"𝕢"#, r#"𝕢"#, r#"q"#, r#"q"#),
        (r#"𝕣"#, r#"𝕣"#, r#"𝕣"#, r#"r"#, r#"r"#),
        (r#"𝕤"#, r#"𝕤"#, r#"𝕤"#, r#"s"#, r#"s"#),
        (r#"𝕥"#, r#"𝕥"#, r#"𝕥"#, r#"t"#, r#"t"#),
        (r#"𝕦"#, r#"𝕦"#, r#"𝕦"#, r#"u"#, r#"u"#),
        (r#"𝕧"#, r#"𝕧"#, r#"𝕧"#, r#"v"#, r#"v"#),
        (r#"𝕨"#, r#"𝕨"#, r#"𝕨"#, r#"w"#, r#"w"#),
        (r#"𝕩"#, r#"𝕩"#, r#"𝕩"#, r#"x"#, r#"x"#),
        (r#"𝕪"#, r#"𝕪"#, r#"𝕪"#, r#"y"#, r#"y"#),
        (r#"𝕫"#, r#"𝕫"#, r#"𝕫"#, r#"z"#, r#"z"#),
        (r#"𝕬"#, r#"𝕬"#, r#"𝕬"#, r#"A"#, r#"A"#),
        (r#"𝕭"#, r#"𝕭"#, r#"𝕭"#, r#"B"#, r#"B"#),
        (r#"𝕮"#, r#"𝕮"#, r#"𝕮"#, r#"C"#, r#"C"#),
        (r#"𝕯"#, r#"𝕯"#, r#"𝕯"#, r#"D"#, r#"D"#),
        (r#"𝕰"#, r#"𝕰"#, r#"𝕰"#, r#"E"#, r#"E"#),
        (r#"𝕱"#, r#"𝕱"#, r#"𝕱"#, r#"F"#, r#"F"#),
        (r#"𝕲"#, r#"𝕲"#, r#"𝕲"#, r#"G"#, r#"G"#),
        (r#"𝕳"#, r#"𝕳"#, r#"𝕳"#, r#"H"#, r#"H"#),
        (r#"𝕴"#, r#"𝕴"#, r#"𝕴"#, r#"I"#, r#"I"#),
        (r#"𝕵"#, r#"𝕵"#, r#"𝕵"#, r#"J"#, r#"J"#),
        (r#"𝕶"#, r#"𝕶"#, r#"𝕶"#, r#"K"#, r#"K"#),
        (r#"𝕷"#, r#"𝕷"#, r#"𝕷"#, r#"L"#, r#"L"#),
        (r#"𝕸"#, r#"𝕸"#, r#"𝕸"#, r#"M"#, r#"M"#),
        (r#"𝕹"#, r#"𝕹"#, r#"𝕹"#, r#"N"#, r#"N"#),
        (r#"𝕺"#, r#"𝕺"#, r#"𝕺"#, r#"O"#, r#"O"#),
        (r#"𝕻"#, r#"𝕻"#, r#"𝕻"#, r#"P"#, r#"P"#),
        (r#"𝕼"#, r#"𝕼"#, r#"𝕼"#, r#"Q"#, r#"Q"#),
        (r#"𝕽"#, r#"𝕽"#, r#"𝕽"#, r#"R"#, r#"R"#),
        (r#"𝕾"#, r#"𝕾"#, r#"𝕾"#, r#"S"#, r#"S"#),
        (r#"𝕿"#, r#"𝕿"#, r#"𝕿"#, r#"T"#, r#"T"#),
        (r#"𝖀"#, r#"𝖀"#, r#"𝖀"#, r#"U"#, r#"U"#),
        (r#"𝖁"#, r#"𝖁"#, r#"𝖁"#, r#"V"#, r#"V"#),
        (r#"𝖂"#, r#"𝖂"#, r#"𝖂"#, r#"W"#, r#"W"#),
        (r#"𝖃"#, r#"𝖃"#, r#"𝖃"#, r#"X"#, r#"X"#),
        (r#"𝖄"#, r#"𝖄"#, r#"𝖄"#, r#"Y"#, r#"Y"#),
        (r#"𝖅"#, r#"𝖅"#, r#"𝖅"#, r#"Z"#, r#"Z"#),
        (r#"𝖆"#, r#"𝖆"#, r#"𝖆"#, r#"a"#, r#"a"#),
        (r#"𝖇"#, r#"𝖇"#, r#"𝖇"#, r#"b"#, r#"b"#),
        (r#"𝖈"#, r#"𝖈"#, r#"𝖈"#, r#"c"#, r#"c"#),
        (r#"𝖉"#, r#"𝖉"#, r#"𝖉"#, r#"d"#, r#"d"#),
        (r#"𝖊"#, r#"𝖊"#, r#"𝖊"#, r#"e"#, r#"e"#),
        (r#"𝖋"#, r#"𝖋"#, r#"𝖋"#, r#"f"#, r#"f"#),
        (r#"𝖌"#, r#"𝖌"#, r#"𝖌"#, r#"g"#, r#"g"#),
        (r#"𝖍"#, r#"𝖍"#, r#"𝖍"#, r#"h"#, r#"h"#),
        (r#"𝖎"#, r#"𝖎"#, r#"𝖎"#, r#"i"#, r#"i"#),
        (r#"𝖏"#, r#"𝖏"#, r#"𝖏"#, r#"j"#, r#"j"#),
        (r#"𝖐"#, r#"𝖐"#, r#"𝖐"#, r#"k"#, r#"k"#),
        (r#"𝖑"#, r#"𝖑"#, r#"𝖑"#, r#"l"#, r#"l"#),
        (r#"𝖒"#, r#"𝖒"#, r#"𝖒"#, r#"m"#, r#"m"#),
        (r#"𝖓"#, r#"𝖓"#, r#"𝖓"#, r#"n"#, r#"n"#),
        (r#"𝖔"#, r#"𝖔"#, r#"𝖔"#, r#"o"#, r#"o"#),
        (r#"𝖕"#, r#"𝖕"#, r#"𝖕"#, r#"p"#, r#"p"#),
        (r#"𝖖"#, r#"𝖖"#, r#"𝖖"#, r#"q"#, r#"q"#),
        (r#"𝖗"#, r#"𝖗"#, r#"𝖗"#, r#"r"#, r#"r"#),
        (r#"𝖘"#, r#"𝖘"#, r#"𝖘"#, r#"s"#, r#"s"#),
        (r#"𝖙"#, r#"𝖙"#, r#"𝖙"#, r#"t"#, r#"t"#),
        (r#"𝖚"#, r#"𝖚"#, r#"𝖚"#, r#"u"#, r#"u"#),
        (r#"𝖛"#, r#"𝖛"#, r#"𝖛"#, r#"v"#, r#"v"#),
        (r#"𝖜"#, r#"𝖜"#, r#"𝖜"#, r#"w"#, r#"w"#),
        (r#"𝖝"#, r#"𝖝"#, r#"𝖝"#, r#"x"#, r#"x"#),
        (r#"𝖞"#, r#"𝖞"#, r#"𝖞"#, r#"y"#, r#"y"#),
        (r#"𝖟"#, r#"𝖟"#, r#"𝖟"#, r#"z"#, r#"z"#),
        (r#"𝖠"#, r#"𝖠"#, r#"𝖠"#, r#"A"#, r#"A"#),
        (r#"𝖡"#, r#"𝖡"#, r#"𝖡"#, r#"B"#, r#"B"#),
        (r#"𝖢"#, r#"𝖢"#, r#"𝖢"#, r#"C"#, r#"C"#),
        (r#"𝖣"#, r#"𝖣"#, r#"𝖣"#, r#"D"#, r#"D"#),
        (r#"𝖤"#, r#"𝖤"#, r#"𝖤"#, r#"E"#, r#"E"#),
        (r#"𝖥"#, r#"𝖥"#, r#"𝖥"#, r#"F"#, r#"F"#),
        (r#"𝖦"#, r#"𝖦"#, r#"𝖦"#, r#"G"#, r#"G"#),
        (r#"𝖧"#, r#"𝖧"#, r#"𝖧"#, r#"H"#, r#"H"#),
        (r#"𝖨"#, r#"𝖨"#, r#"𝖨"#, r#"I"#, r#"I"#),
        (r#"𝖩"#, r#"𝖩"#, r#"𝖩"#, r#"J"#, r#"J"#),
        (r#"𝖪"#, r#"𝖪"#, r#"𝖪"#, r#"K"#, r#"K"#),
        (r#"𝖫"#, r#"𝖫"#, r#"𝖫"#, r#"L"#, r#"L"#),
        (r#"𝖬"#, r#"𝖬"#, r#"𝖬"#, r#"M"#, r#"M"#),
        (r#"𝖭"#, r#"𝖭"#, r#"𝖭"#, r#"N"#, r#"N"#),
        (r#"𝖮"#, r#"𝖮"#, r#"𝖮"#, r#"O"#, r#"O"#),
        (r#"𝖯"#, r#"𝖯"#, r#"𝖯"#, r#"P"#, r#"P"#),
        (r#"𝖰"#, r#"𝖰"#, r#"𝖰"#, r#"Q"#, r#"Q"#),
        (r#"𝖱"#, r#"𝖱"#, r#"𝖱"#, r#"R"#, r#"R"#),
        (r#"𝖲"#, r#"𝖲"#, r#"𝖲"#, r#"S"#, r#"S"#),
        (r#"𝖳"#, r#"𝖳"#, r#"𝖳"#, r#"T"#, r#"T"#),
        (r#"𝖴"#, r#"𝖴"#, r#"𝖴"#, r#"U"#, r#"U"#),
        (r#"𝖵"#, r#"𝖵"#, r#"𝖵"#, r#"V"#, r#"V"#),
        (r#"𝖶"#, r#"𝖶"#, r#"𝖶"#, r#"W"#, r#"W"#),
        (r#"𝖷"#, r#"𝖷"#, r#"𝖷"#, r#"X"#, r#"X"#),
        (r#"𝖸"#, r#"𝖸"#, r#"𝖸"#, r#"Y"#, r#"Y"#),
        (r#"𝖹"#, r#"𝖹"#, r#"𝖹"#, r#"Z"#, r#"Z"#),
        (r#"𝖺"#, r#"𝖺"#, r#"𝖺"#, r#"a"#, r#"a"#),
        (r#"𝖻"#, r#"𝖻"#, r#"𝖻"#, r#"b"#, r#"b"#),
        (r#"𝖼"#, r#"𝖼"#, r#"𝖼"#, r#"c"#, r#"c"#),
        (r#"𝖽"#, r#"𝖽"#, r#"𝖽"#, r#"d"#, r#"d"#),
        (r#"𝖾"#, r#"𝖾"#, r#"𝖾"#, r#"e"#, r#"e"#),
        (r#"𝖿"#, r#"𝖿"#, r#"𝖿"#, r#"f"#, r#"f"#),
        (r#"𝗀"#, r#"𝗀"#, r#"𝗀"#, r#"g"#, r#"g"#),
        (r#"𝗁"#, r#"𝗁"#, r#"𝗁"#, r#"h"#, r#"h"#),
        (r#"𝗂"#, r#"𝗂"#, r#"𝗂"#, r#"i"#, r#"i"#),
        (r#"𝗃"#, r#"𝗃"#, r#"𝗃"#, r#"j"#, r#"j"#),
        (r#"𝗄"#, r#"𝗄"#, r#"𝗄"#, r#"k"#, r#"k"#),
        (r#"𝗅"#, r#"𝗅"#, r#"𝗅"#, r#"l"#, r#"l"#),
        (r#"𝗆"#, r#"𝗆"#, r#"𝗆"#, r#"m"#, r#"m"#),
        (r#"𝗇"#, r#"𝗇"#, r#"𝗇"#, r#"n"#, r#"n"#),
        (r#"𝗈"#, r#"𝗈"#, r#"𝗈"#, r#"o"#, r#"o"#),
        (r#"𝗉"#, r#"𝗉"#, r#"𝗉"#, r#"p"#, r#"p"#),
        (r#"𝗊"#, r#"𝗊"#, r#"𝗊"#, r#"q"#, r#"q"#),
        (r#"𝗋"#, r#"𝗋"#, r#"𝗋"#, r#"r"#, r#"r"#),
        (r#"𝗌"#, r#"𝗌"#, r#"𝗌"#, r#"s"#, r#"s"#),
        (r#"𝗍"#, r#"𝗍"#, r#"𝗍"#, r#"t"#, r#"t"#),
        (r#"𝗎"#, r#"𝗎"#, r#"𝗎"#, r#"u"#, r#"u"#),
        (r#"𝗏"#, r#"𝗏"#, r#"𝗏"#, r#"v"#, r#"v"#),
        (r#"𝗐"#, r#"𝗐"#, r#"𝗐"#, r#"w"#, r#"w"#),
        (r#"𝗑"#, r#"𝗑"#, r#"𝗑"#, r#"x"#, r#"x"#),
        (r#"𝗒"#, r#"𝗒"#, r#"𝗒"#, r#"y"#, r#"y"#),
        (r#"𝗓"#, r#"𝗓"#, r#"𝗓"#, r#"z"#, r#"z"#),
        (r#"𝗔"#, r#"𝗔"#, r#"𝗔"#, r#"A"#, r#"A"#),
        (r#"𝗕"#, r#"𝗕"#, r#"𝗕"#, r#"B"#, r#"B"#),
        (r#"𝗖"#, r#"𝗖"#, r#"𝗖"#, r#"C"#, r#"C"#),
        (r#"𝗗"#, r#"𝗗"#, r#"𝗗"#, r#"D"#, r#"D"#),
        (r#"𝗘"#, r#"𝗘"#, r#"𝗘"#, r#"E"#, r#"E"#),
        (r#"𝗙"#, r#"𝗙"#, r#"𝗙"#, r#"F"#, r#"F"#),
        (r#"𝗚"#, r#"𝗚"#, r#"𝗚"#, r#"G"#, r#"G"#),
        (r#"𝗛"#, r#"𝗛"#, r#"𝗛"#, r#"H"#, r#"H"#),
        (r#"𝗜"#, r#"𝗜"#, r#"𝗜"#, r#"I"#, r#"I"#),
        (r#"𝗝"#, r#"𝗝"#, r#"𝗝"#, r#"J"#, r#"J"#),
        (r#"𝗞"#, r#"𝗞"#, r#"𝗞"#, r#"K"#, r#"K"#),
        (r#"𝗟"#, r#"𝗟"#, r#"𝗟"#, r#"L"#, r#"L"#),
        (r#"𝗠"#, r#"𝗠"#, r#"𝗠"#, r#"M"#, r#"M"#),
        (r#"𝗡"#, r#"𝗡"#, r#"𝗡"#, r#"N"#, r#"N"#),
        (r#"𝗢"#, r#"𝗢"#, r#"𝗢"#, r#"O"#, r#"O"#),
        (r#"𝗣"#, r#"𝗣"#, r#"𝗣"#, r#"P"#, r#"P"#),
        (r#"𝗤"#, r#"𝗤"#, r#"𝗤"#, r#"Q"#, r#"Q"#),
        (r#"𝗥"#, r#"𝗥"#, r#"𝗥"#, r#"R"#, r#"R"#),
        (r#"𝗦"#, r#"𝗦"#, r#"𝗦"#, r#"S"#, r#"S"#),
        (r#"𝗧"#, r#"𝗧"#, r#"𝗧"#, r#"T"#, r#"T"#),
        (r#"𝗨"#, r#"𝗨"#, r#"𝗨"#, r#"U"#, r#"U"#),
        (r#"𝗩"#, r#"𝗩"#, r#"𝗩"#, r#"V"#, r#"V"#),
        (r#"𝗪"#, r#"𝗪"#, r#"𝗪"#, r#"W"#, r#"W"#),
        (r#"𝗫"#, r#"𝗫"#, r#"𝗫"#, r#"X"#, r#"X"#),
        (r#"𝗬"#, r#"𝗬"#, r#"𝗬"#, r#"Y"#, r#"Y"#),
        (r#"𝗭"#, r#"𝗭"#, r#"𝗭"#, r#"Z"#, r#"Z"#),
        (r#"𝗮"#, r#"𝗮"#, r#"𝗮"#, r#"a"#, r#"a"#),
        (r#"𝗯"#, r#"𝗯"#, r#"𝗯"#, r#"b"#, r#"b"#),
        (r#"𝗰"#, r#"𝗰"#, r#"𝗰"#, r#"c"#, r#"c"#),
        (r#"𝗱"#, r#"𝗱"#, r#"𝗱"#, r#"d"#, r#"d"#),
        (r#"𝗲"#, r#"𝗲"#, r#"𝗲"#, r#"e"#, r#"e"#),
        (r#"𝗳"#, r#"𝗳"#, r#"𝗳"#, r#"f"#, r#"f"#),
        (r#"𝗴"#, r#"𝗴"#, r#"𝗴"#, r#"g"#, r#"g"#),
        (r#"𝗵"#, r#"𝗵"#, r#"𝗵"#, r#"h"#, r#"h"#),
        (r#"𝗶"#, r#"𝗶"#, r#"𝗶"#, r#"i"#, r#"i"#),
        (r#"𝗷"#, r#"𝗷"#, r#"𝗷"#, r#"j"#, r#"j"#),
        (r#"𝗸"#, r#"𝗸"#, r#"𝗸"#, r#"k"#, r#"k"#),
        (r#"𝗹"#, r#"𝗹"#, r#"𝗹"#, r#"l"#, r#"l"#),
        (r#"𝗺"#, r#"𝗺"#, r#"𝗺"#, r#"m"#, r#"m"#),
        (r#"𝗻"#, r#"𝗻"#, r#"𝗻"#, r#"n"#, r#"n"#),
        (r#"𝗼"#, r#"𝗼"#, r#"𝗼"#, r#"o"#, r#"o"#),
        (r#"𝗽"#, r#"𝗽"#, r#"𝗽"#, r#"p"#, r#"p"#),
        (r#"𝗾"#, r#"𝗾"#, r#"𝗾"#, r#"q"#, r#"q"#),
        (r#"𝗿"#, r#"𝗿"#, r#"𝗿"#, r#"r"#, r#"r"#),
        (r#"𝘀"#, r#"𝘀"#, r#"𝘀"#, r#"s"#, r#"s"#),
        (r#"𝘁"#, r#"𝘁"#, r#"𝘁"#, r#"t"#, r#"t"#),
        (r#"𝘂"#, r#"𝘂"#, r#"𝘂"#, r#"u"#, r#"u"#),
        (r#"𝘃"#, r#"𝘃"#, r#"𝘃"#, r#"v"#, r#"v"#),
        (r#"𝘄"#, r#"𝘄"#, r#"𝘄"#, r#"w"#, r#"w"#),
        (r#"𝘅"#, r#"𝘅"#, r#"𝘅"#, r#"x"#, r#"x"#),
        (r#"𝘆"#, r#"𝘆"#, r#"𝘆"#, r#"y"#, r#"y"#),
        (r#"𝘇"#, r#"𝘇"#, r#"𝘇"#, r#"z"#, r#"z"#),
        (r#"𝘈"#, r#"𝘈"#, r#"𝘈"#, r#"A"#, r#"A"#),
        (r#"𝘉"#, r#"𝘉"#, r#"𝘉"#, r#"B"#, r#"B"#),
        (r#"𝘊"#, r#"𝘊"#, r#"𝘊"#, r#"C"#, r#"C"#),
        (r#"𝘋"#, r#"𝘋"#, r#"𝘋"#, r#"D"#, r#"D"#),
        (r#"𝘌"#, r#"𝘌"#, r#"𝘌"#, r#"E"#, r#"E"#),
        (r#"𝘍"#, r#"𝘍"#, r#"𝘍"#, r#"F"#, r#"F"#),
        (r#"𝘎"#, r#"𝘎"#, r#"𝘎"#, r#"G"#, r#"G"#),
        (r#"𝘏"#, r#"𝘏"#, r#"𝘏"#, r#"H"#, r#"H"#),
        (r#"𝘐"#, r#"𝘐"#, r#"𝘐"#, r#"I"#, r#"I"#),
        (r#"𝘑"#, r#"𝘑"#, r#"𝘑"#, r#"J"#, r#"J"#),
        (r#"𝘒"#, r#"𝘒"#, r#"𝘒"#, r#"K"#, r#"K"#),
        (r#"𝘓"#, r#"𝘓"#, r#"𝘓"#, r#"L"#, r#"L"#),
        (r#"𝘔"#, r#"𝘔"#, r#"𝘔"#, r#"M"#, r#"M"#),
        (r#"𝘕"#, r#"𝘕"#, r#"𝘕"#, r#"N"#, r#"N"#),
        (r#"𝘖"#, r#"𝘖"#, r#"𝘖"#, r#"O"#, r#"O"#),
        (r#"𝘗"#, r#"𝘗"#, r#"𝘗"#, r#"P"#, r#"P"#),
        (r#"𝘘"#, r#"𝘘"#, r#"𝘘"#, r#"Q"#, r#"Q"#),
        (r#"𝘙"#, r#"𝘙"#, r#"𝘙"#, r#"R"#, r#"R"#),
        (r#"𝘚"#, r#"𝘚"#, r#"𝘚"#, r#"S"#, r#"S"#),
        (r#"𝘛"#, r#"𝘛"#, r#"𝘛"#, r#"T"#, r#"T"#),
        (r#"𝘜"#, r#"𝘜"#, r#"𝘜"#, r#"U"#, r#"U"#),
        (r#"𝘝"#, r#"𝘝"#, r#"𝘝"#, r#"V"#, r#"V"#),
        (r#"𝘞"#, r#"𝘞"#, r#"𝘞"#, r#"W"#, r#"W"#),
        (r#"𝘟"#, r#"𝘟"#, r#"𝘟"#, r#"X"#, r#"X"#),
        (r#"𝘠"#, r#"𝘠"#, r#"𝘠"#, r#"Y"#, r#"Y"#),
        (r#"𝘡"#, r#"𝘡"#, r#"𝘡"#, r#"Z"#, r#"Z"#),
        (r#"𝘢"#, r#"𝘢"#, r#"𝘢"#, r#"a"#, r#"a"#),
        (r#"𝘣"#, r#"𝘣"#, r#"𝘣"#, r#"b"#, r#"b"#),
        (r#"𝘤"#, r#"𝘤"#, r#"𝘤"#, r#"c"#, r#"c"#),
        (r#"𝘥"#, r#"𝘥"#, r#"𝘥"#, r#"d"#, r#"d"#),
        (r#"𝘦"#, r#"𝘦"#, r#"𝘦"#, r#"e"#, r#"e"#),
        (r#"𝘧"#, r#"𝘧"#, r#"𝘧"#, r#"f"#, r#"f"#),
        (r#"𝘨"#, r#"𝘨"#, r#"𝘨"#, r#"g"#, r#"g"#),
        (r#"𝘩"#, r#"𝘩"#, r#"𝘩"#, r#"h"#, r#"h"#),
        (r#"𝘪"#, r#"𝘪"#, r#"𝘪"#, r#"i"#, r#"i"#),
        (r#"𝘫"#, r#"𝘫"#, r#"𝘫"#, r#"j"#, r#"j"#),
        (r#"𝘬"#, r#"𝘬"#, r#"𝘬"#, r#"k"#, r#"k"#),
        (r#"𝘭"#, r#"𝘭"#, r#"𝘭"#, r#"l"#, r#"l"#),
        (r#"𝘮"#, r#"𝘮"#, r#"𝘮"#, r#"m"#, r#"m"#),
        (r#"𝘯"#, r#"𝘯"#, r#"𝘯"#, r#"n"#, r#"n"#),
        (r#"𝘰"#, r#"𝘰"#, r#"𝘰"#, r#"o"#, r#"o"#),
        (r#"𝘱"#, r#"𝘱"#, r#"𝘱"#, r#"p"#, r#"p"#),
        (r#"𝘲"#, r#"𝘲"#, r#"𝘲"#, r#"q"#, r#"q"#),
        (r#"𝘳"#, r#"𝘳"#, r#"𝘳"#, r#"r"#, r#"r"#),
        (r#"𝘴"#, r#"𝘴"#, r#"𝘴"#, r#"s"#, r#"s"#),
        (r#"𝘵"#, r#"𝘵"#, r#"𝘵"#, r#"t"#, r#"t"#),
        (r#"𝘶"#, r#"𝘶"#, r#"𝘶"#, r#"u"#, r#"u"#),
        (r#"𝘷"#, r#"𝘷"#, r#"𝘷"#, r#"v"#, r#"v"#),
        (r#"𝘸"#, r#"𝘸"#, r#"𝘸"#, r#"w"#, r#"w"#),
        (r#"𝘹"#, r#"𝘹"#, r#"𝘹"#, r#"x"#, r#"x"#),
        (r#"𝘺"#, r#"𝘺"#, r#"𝘺"#, r#"y"#, r#"y"#),
        (r#"𝘻"#, r#"𝘻"#, r#"𝘻"#, r#"z"#, r#"z"#),
        (r#"𝘼"#, r#"𝘼"#, r#"𝘼"#, r#"A"#, r#"A"#),
        (r#"𝘽"#, r#"𝘽"#, r#"𝘽"#, r#"B"#, r#"B"#),
        (r#"𝘾"#, r#"𝘾"#, r#"𝘾"#, r#"C"#, r#"C"#),
        (r#"𝘿"#, r#"𝘿"#, r#"𝘿"#, r#"D"#, r#"D"#),
        (r#"𝙀"#, r#"𝙀"#, r#"𝙀"#, r#"E"#, r#"E"#),
        (r#"𝙁"#, r#"𝙁"#, r#"𝙁"#, r#"F"#, r#"F"#),
        (r#"𝙂"#, r#"𝙂"#, r#"𝙂"#, r#"G"#, r#"G"#),
        (r#"𝙃"#, r#"𝙃"#, r#"𝙃"#, r#"H"#, r#"H"#),
        (r#"𝙄"#, r#"𝙄"#, r#"𝙄"#, r#"I"#, r#"I"#),
        (r#"𝙅"#, r#"𝙅"#, r#"𝙅"#, r#"J"#, r#"J"#),
        (r#"𝙆"#, r#"𝙆"#, r#"𝙆"#, r#"K"#, r#"K"#),
        (r#"𝙇"#, r#"𝙇"#, r#"𝙇"#, r#"L"#, r#"L"#),
        (r#"𝙈"#, r#"𝙈"#, r#"𝙈"#, r#"M"#, r#"M"#),
        (r#"𝙉"#, r#"𝙉"#, r#"𝙉"#, r#"N"#, r#"N"#),
        (r#"𝙊"#, r#"𝙊"#, r#"𝙊"#, r#"O"#, r#"O"#),
        (r#"𝙋"#, r#"𝙋"#, r#"𝙋"#, r#"P"#, r#"P"#),
        (r#"𝙌"#, r#"𝙌"#, r#"𝙌"#, r#"Q"#, r#"Q"#),
        (r#"𝙍"#, r#"𝙍"#, r#"𝙍"#, r#"R"#, r#"R"#),
        (r#"𝙎"#, r#"𝙎"#, r#"𝙎"#, r#"S"#, r#"S"#),
        (r#"𝙏"#, r#"𝙏"#, r#"𝙏"#, r#"T"#, r#"T"#),
        (r#"𝙐"#, r#"𝙐"#, r#"𝙐"#, r#"U"#, r#"U"#),
        (r#"𝙑"#, r#"𝙑"#, r#"𝙑"#, r#"V"#, r#"V"#),
        (r#"𝙒"#, r#"𝙒"#, r#"𝙒"#, r#"W"#, r#"W"#),
        (r#"𝙓"#, r#"𝙓"#, r#"𝙓"#, r#"X"#, r#"X"#),
        (r#"𝙔"#, r#"𝙔"#, r#"𝙔"#, r#"Y"#, r#"Y"#),
        (r#"𝙕"#, r#"𝙕"#, r#"𝙕"#, r#"Z"#, r#"Z"#),
        (r#"𝙖"#, r#"𝙖"#, r#"𝙖"#, r#"a"#, r#"a"#),
        (r#"𝙗"#, r#"𝙗"#, r#"𝙗"#, r#"b"#, r#"b"#),
        (r#"𝙘"#, r#"𝙘"#, r#"𝙘"#, r#"c"#, r#"c"#),
        (r#"𝙙"#, r#"𝙙"#, r#"𝙙"#, r#"d"#, r#"d"#),
        (r#"𝙚"#, r#"𝙚"#, r#"𝙚"#, r#"e"#, r#"e"#),
        (r#"𝙛"#, r#"𝙛"#, r#"𝙛"#, r#"f"#, r#"f"#),
        (r#"𝙜"#, r#"𝙜"#, r#"𝙜"#, r#"g"#, r#"g"#),
        (r#"𝙝"#, r#"𝙝"#, r#"𝙝"#, r#"h"#, r#"h"#),
        (r#"𝙞"#, r#"𝙞"#, r#"𝙞"#, r#"i"#, r#"i"#),
        (r#"𝙟"#, r#"𝙟"#, r#"𝙟"#, r#"j"#, r#"j"#),
        (r#"𝙠"#, r#"𝙠"#, r#"𝙠"#, r#"k"#, r#"k"#),
        (r#"𝙡"#, r#"𝙡"#, r#"𝙡"#, r#"l"#, r#"l"#),
        (r#"𝙢"#, r#"𝙢"#, r#"𝙢"#, r#"m"#, r#"m"#),
        (r#"𝙣"#, r#"𝙣"#, r#"𝙣"#, r#"n"#, r#"n"#),
        (r#"𝙤"#, r#"𝙤"#, r#"𝙤"#, r#"o"#, r#"o"#),
        (r#"𝙥"#, r#"𝙥"#, r#"𝙥"#, r#"p"#, r#"p"#),
        (r#"𝙦"#, r#"𝙦"#, r#"𝙦"#, r#"q"#, r#"q"#),
        (r#"𝙧"#, r#"𝙧"#, r#"𝙧"#, r#"r"#, r#"r"#),
        (r#"𝙨"#, r#"𝙨"#, r#"𝙨"#, r#"s"#, r#"s"#),
        (r#"𝙩"#, r#"𝙩"#, r#"𝙩"#, r#"t"#, r#"t"#),
        (r#"𝙪"#, r#"𝙪"#, r#"𝙪"#, r#"u"#, r#"u"#),
        (r#"𝙫"#, r#"𝙫"#, r#"𝙫"#, r#"v"#, r#"v"#),
        (r#"𝙬"#, r#"𝙬"#, r#"𝙬"#, r#"w"#, r#"w"#),
        (r#"𝙭"#, r#"𝙭"#, r#"𝙭"#, r#"x"#, r#"x"#),
        (r#"𝙮"#, r#"𝙮"#, r#"𝙮"#, r#"y"#, r#"y"#),
        (r#"𝙯"#, r#"𝙯"#, r#"𝙯"#, r#"z"#, r#"z"#),
        (r#"𝙰"#, r#"𝙰"#, r#"𝙰"#, r#"A"#, r#"A"#),
        (r#"𝙱"#, r#"𝙱"#, r#"𝙱"#, r#"B"#, r#"B"#),
        (r#"𝙲"#, r#"𝙲"#, r#"𝙲"#, r#"C"#, r#"C"#),
        (r#"𝙳"#, r#"𝙳"#, r#"𝙳"#, r#"D"#, r#"D"#),
        (r#"𝙴"#, r#"𝙴"#, r#"𝙴"#, r#"E"#, r#"E"#),
        (r#"𝙵"#, r#"𝙵"#, r#"𝙵"#, r#"F"#, r#"F"#),
        (r#"𝙶"#, r#"𝙶"#, r#"𝙶"#, r#"G"#, r#"G"#),
        (r#"𝙷"#, r#"𝙷"#, r#"𝙷"#, r#"H"#, r#"H"#),
        (r#"𝙸"#, r#"𝙸"#, r#"𝙸"#, r#"I"#, r#"I"#),
        (r#"𝙹"#, r#"𝙹"#, r#"𝙹"#, r#"J"#, r#"J"#),
        (r#"𝙺"#, r#"𝙺"#, r#"𝙺"#, r#"K"#, r#"K"#),
        (r#"𝙻"#, r#"𝙻"#, r#"𝙻"#, r#"L"#, r#"L"#),
        (r#"𝙼"#, r#"𝙼"#, r#"𝙼"#, r#"M"#, r#"M"#),
        (r#"𝙽"#, r#"𝙽"#, r#"𝙽"#, r#"N"#, r#"N"#),
        (r#"𝙾"#, r#"𝙾"#, r#"𝙾"#, r#"O"#, r#"O"#),
        (r#"𝙿"#, r#"𝙿"#, r#"𝙿"#, r#"P"#, r#"P"#),
        (r#"𝚀"#, r#"𝚀"#, r#"𝚀"#, r#"Q"#, r#"Q"#),
        (r#"𝚁"#, r#"𝚁"#, r#"𝚁"#, r#"R"#, r#"R"#),
        (r#"𝚂"#, r#"𝚂"#, r#"𝚂"#, r#"S"#, r#"S"#),
        (r#"𝚃"#, r#"𝚃"#, r#"𝚃"#, r#"T"#, r#"T"#),
        (r#"𝚄"#, r#"𝚄"#, r#"𝚄"#, r#"U"#, r#"U"#),
        (r#"𝚅"#, r#"𝚅"#, r#"𝚅"#, r#"V"#, r#"V"#),
        (r#"𝚆"#, r#"𝚆"#, r#"𝚆"#, r#"W"#, r#"W"#),
        (r#"𝚇"#, r#"𝚇"#, r#"𝚇"#, r#"X"#, r#"X"#),
        (r#"𝚈"#, r#"𝚈"#, r#"𝚈"#, r#"Y"#, r#"Y"#),
        (r#"𝚉"#, r#"𝚉"#, r#"𝚉"#, r#"Z"#, r#"Z"#),
        (r#"𝚊"#, r#"𝚊"#, r#"𝚊"#, r#"a"#, r#"a"#),
        (r#"𝚋"#, r#"𝚋"#, r#"𝚋"#, r#"b"#, r#"b"#),
        (r#"𝚌"#, r#"𝚌"#, r#"𝚌"#, r#"c"#, r#"c"#),
        (r#"𝚍"#, r#"𝚍"#, r#"𝚍"#, r#"d"#, r#"d"#),
        (r#"𝚎"#, r#"𝚎"#, r#"𝚎"#, r#"e"#, r#"e"#),
        (r#"𝚏"#, r#"𝚏"#, r#"𝚏"#, r#"f"#, r#"f"#),
        (r#"𝚐"#, r#"𝚐"#, r#"𝚐"#, r#"g"#, r#"g"#),
        (r#"𝚑"#, r#"𝚑"#, r#"𝚑"#, r#"h"#, r#"h"#),
        (r#"𝚒"#, r#"𝚒"#, r#"𝚒"#, r#"i"#, r#"i"#),
        (r#"𝚓"#, r#"𝚓"#, r#"𝚓"#, r#"j"#, r#"j"#),
        (r#"𝚔"#, r#"𝚔"#, r#"𝚔"#, r#"k"#, r#"k"#),
        (r#"𝚕"#, r#"𝚕"#, r#"𝚕"#, r#"l"#, r#"l"#),
        (r#"𝚖"#, r#"𝚖"#, r#"𝚖"#, r#"m"#, r#"m"#),
        (r#"𝚗"#, r#"𝚗"#, r#"𝚗"#, r#"n"#, r#"n"#),
        (r#"𝚘"#, r#"𝚘"#, r#"𝚘"#, r#"o"#, r#"o"#),
        (r#"𝚙"#, r#"𝚙"#, r#"𝚙"#, r#"p"#, r#"p"#),
        (r#"𝚚"#, r#"𝚚"#, r#"𝚚"#, r#"q"#, r#"q"#),
        (r#"𝚛"#, r#"𝚛"#, r#"𝚛"#, r#"r"#, r#"r"#),
        (r#"𝚜"#, r#"𝚜"#, r#"𝚜"#, r#"s"#, r#"s"#),
        (r#"𝚝"#, r#"𝚝"#, r#"𝚝"#, r#"t"#, r#"t"#),
        (r#"𝚞"#, r#"𝚞"#, r#"𝚞"#, r#"u"#, r#"u"#),
        (r#"𝚟"#, r#"𝚟"#, r#"𝚟"#, r#"v"#, r#"v"#),
        (r#"𝚠"#, r#"𝚠"#, r#"𝚠"#, r#"w"#, r#"w"#),
        (r#"𝚡"#, r#"𝚡"#, r#"𝚡"#, r#"x"#, r#"x"#),
        (r#"𝚢"#, r#"𝚢"#, r#"𝚢"#, r#"y"#, r#"y"#),
        (r#"𝚣"#, r#"𝚣"#, r#"𝚣"#, r#"z"#, r#"z"#),
        (r#"𝚤"#, r#"𝚤"#, r#"𝚤"#, r#"ı"#, r#"ı"#),
        (r#"𝚥"#, r#"𝚥"#, r#"𝚥"#, r#"ȷ"#, r#"ȷ"#),
        (r#"𝚨"#, r#"𝚨"#, r#"𝚨"#, r#"Α"#, r#"Α"#),
        (r#"𝚩"#, r#"𝚩"#, r#"𝚩"#, r#"Β"#, r#"Β"#),
        (r#"𝚪"#, r#"𝚪"#, r#"𝚪"#, r#"Γ"#, r#"Γ"#),
        (r#"𝚫"#, r#"𝚫"#, r#"𝚫"#, r#"Δ"#, r#"Δ"#),
        (r#"𝚬"#, r#"𝚬"#, r#"𝚬"#, r#"Ε"#, r#"Ε"#),
        (r#"𝚭"#, r#"𝚭"#, r#"𝚭"#, r#"Ζ"#, r#"Ζ"#),
        (r#"𝚮"#, r#"𝚮"#, r#"𝚮"#, r#"Η"#, r#"Η"#),
        (r#"𝚯"#, r#"𝚯"#, r#"𝚯"#, r#"Θ"#, r#"Θ"#),
        (r#"𝚰"#, r#"𝚰"#, r#"𝚰"#, r#"Ι"#, r#"Ι"#),
        (r#"𝚱"#, r#"𝚱"#, r#"𝚱"#, r#"Κ"#, r#"Κ"#),
        (r#"𝚲"#, r#"𝚲"#, r#"𝚲"#, r#"Λ"#, r#"Λ"#),
        (r#"𝚳"#, r#"𝚳"#, r#"𝚳"#, r#"Μ"#, r#"Μ"#),
        (r#"𝚴"#, r#"𝚴"#, r#"𝚴"#, r#"Ν"#, r#"Ν"#),
        (r#"𝚵"#, r#"𝚵"#, r#"𝚵"#, r#"Ξ"#, r#"Ξ"#),
        (r#"𝚶"#, r#"𝚶"#, r#"𝚶"#, r#"Ο"#, r#"Ο"#),
        (r#"𝚷"#, r#"𝚷"#, r#"𝚷"#, r#"Π"#, r#"Π"#),
        (r#"𝚸"#, r#"𝚸"#, r#"𝚸"#, r#"Ρ"#, r#"Ρ"#),
        (r#"𝚹"#, r#"𝚹"#, r#"𝚹"#, r#"Θ"#, r#"Θ"#),
        (r#"𝚺"#, r#"𝚺"#, r#"𝚺"#, r#"Σ"#, r#"Σ"#),
        (r#"𝚻"#, r#"𝚻"#, r#"𝚻"#, r#"Τ"#, r#"Τ"#),
        (r#"𝚼"#, r#"𝚼"#, r#"𝚼"#, r#"Υ"#, r#"Υ"#),
        (r#"𝚽"#, r#"𝚽"#, r#"𝚽"#, r#"Φ"#, r#"Φ"#),
        (r#"𝚾"#, r#"𝚾"#, r#"𝚾"#, r#"Χ"#, r#"Χ"#),
        (r#"𝚿"#, r#"𝚿"#, r#"𝚿"#, r#"Ψ"#, r#"Ψ"#),
        (r#"𝛀"#, r#"𝛀"#, r#"𝛀"#, r#"Ω"#, r#"Ω"#),
        (r#"𝛁"#, r#"𝛁"#, r#"𝛁"#, r#"∇"#, r#"∇"#),
        (r#"𝛂"#, r#"𝛂"#, r#"𝛂"#, r#"α"#, r#"α"#),
        (r#"𝛃"#, r#"𝛃"#, r#"𝛃"#, r#"β"#, r#"β"#),
        (r#"𝛄"#, r#"𝛄"#, r#"𝛄"#, r#"γ"#, r#"γ"#),
        (r#"𝛅"#, r#"𝛅"#, r#"𝛅"#, r#"δ"#, r#"δ"#),
        (r#"𝛆"#, r#"𝛆"#, r#"𝛆"#, r#"ε"#, r#"ε"#),
        (r#"𝛇"#, r#"𝛇"#, r#"𝛇"#, r#"ζ"#, r#"ζ"#),
        (r#"𝛈"#, r#"𝛈"#, r#"𝛈"#, r#"η"#, r#"η"#),
        (r#"𝛉"#, r#"𝛉"#, r#"𝛉"#, r#"θ"#, r#"θ"#),
        (r#"𝛊"#, r#"𝛊"#, r#"𝛊"#, r#"ι"#, r#"ι"#),
        (r#"𝛋"#, r#"𝛋"#, r#"𝛋"#, r#"κ"#, r#"κ"#),
        (r#"𝛌"#, r#"𝛌"#, r#"𝛌"#, r#"λ"#, r#"λ"#),
        (r#"𝛍"#, r#"𝛍"#, r#"𝛍"#, r#"μ"#, r#"μ"#),
        (r#"𝛎"#, r#"𝛎"#, r#"𝛎"#, r#"ν"#, r#"ν"#),
        (r#"𝛏"#, r#"𝛏"#, r#"𝛏"#, r#"ξ"#, r#"ξ"#),
        (r#"𝛐"#, r#"𝛐"#, r#"𝛐"#, r#"ο"#, r#"ο"#),
        (r#"𝛑"#, r#"𝛑"#, r#"𝛑"#, r#"π"#, r#"π"#),
        (r#"𝛒"#, r#"𝛒"#, r#"𝛒"#, r#"ρ"#, r#"ρ"#),
        (r#"𝛓"#, r#"𝛓"#, r#"𝛓"#, r#"ς"#, r#"ς"#),
        (r#"𝛔"#, r#"𝛔"#, r#"𝛔"#, r#"σ"#, r#"σ"#),
        (r#"𝛕"#, r#"𝛕"#, r#"𝛕"#, r#"τ"#, r#"τ"#),
        (r#"𝛖"#, r#"𝛖"#, r#"𝛖"#, r#"υ"#, r#"υ"#),
        (r#"𝛗"#, r#"𝛗"#, r#"𝛗"#, r#"φ"#, r#"φ"#),
        (r#"𝛘"#, r#"𝛘"#, r#"𝛘"#, r#"χ"#, r#"χ"#),
        (r#"𝛙"#, r#"𝛙"#, r#"𝛙"#, r#"ψ"#, r#"ψ"#),
        (r#"𝛚"#, r#"𝛚"#, r#"𝛚"#, r#"ω"#, r#"ω"#),
        (r#"𝛛"#, r#"𝛛"#, r#"𝛛"#, r#"∂"#, r#"∂"#),
        (r#"𝛜"#, r#"𝛜"#, r#"𝛜"#, r#"ε"#, r#"ε"#),
        (r#"𝛝"#, r#"𝛝"#, r#"𝛝"#, r#"θ"#, r#"θ"#),
        (r#"𝛞"#, r#"𝛞"#, r#"𝛞"#, r#"κ"#, r#"κ"#),
        (r#"𝛟"#, r#"𝛟"#, r#"𝛟"#, r#"φ"#, r#"φ"#),
        (r#"𝛠"#, r#"𝛠"#, r#"𝛠"#, r#"ρ"#, r#"ρ"#),
        (r#"𝛡"#, r#"𝛡"#, r#"𝛡"#, r#"π"#, r#"π"#),
        (r#"𝛢"#, r#"𝛢"#, r#"𝛢"#, r#"Α"#, r#"Α"#),
        (r#"𝛣"#, r#"𝛣"#, r#"𝛣"#, r#"Β"#, r#"Β"#),
        (r#"𝛤"#, r#"𝛤"#, r#"𝛤"#, r#"Γ"#, r#"Γ"#),
        (r#"𝛥"#, r#"𝛥"#, r#"𝛥"#, r#"Δ"#, r#"Δ"#),
        (r#"𝛦"#, r#"𝛦"#, r#"𝛦"#, r#"Ε"#, r#"Ε"#),
        (r#"𝛧"#, r#"𝛧"#, r#"𝛧"#, r#"Ζ"#, r#"Ζ"#),
        (r#"𝛨"#, r#"𝛨"#, r#"𝛨"#, r#"Η"#, r#"Η"#),
        (r#"𝛩"#, r#"𝛩"#, r#"𝛩"#, r#"Θ"#, r#"Θ"#),
        (r#"𝛪"#, r#"𝛪"#, r#"𝛪"#, r#"Ι"#, r#"Ι"#),
        (r#"𝛫"#, r#"𝛫"#, r#"𝛫"#, r#"Κ"#, r#"Κ"#),
        (r#"𝛬"#, r#"𝛬"#, r#"𝛬"#, r#"Λ"#, r#"Λ"#),
        (r#"𝛭"#, r#"𝛭"#, r#"𝛭"#, r#"Μ"#, r#"Μ"#),
        (r#"𝛮"#, r#"𝛮"#, r#"𝛮"#, r#"Ν"#, r#"Ν"#),
        (r#"𝛯"#, r#"𝛯"#, r#"𝛯"#, r#"Ξ"#, r#"Ξ"#),
        (r#"𝛰"#, r#"𝛰"#, r#"𝛰"#, r#"Ο"#, r#"Ο"#),
        (r#"𝛱"#, r#"𝛱"#, r#"𝛱"#, r#"Π"#, r#"Π"#),
        (r#"𝛲"#, r#"𝛲"#, r#"𝛲"#, r#"Ρ"#, r#"Ρ"#),
        (r#"𝛳"#, r#"𝛳"#, r#"𝛳"#, r#"Θ"#, r#"Θ"#),
        (r#"𝛴"#, r#"𝛴"#, r#"𝛴"#, r#"Σ"#, r#"Σ"#),
        (r#"𝛵"#, r#"𝛵"#, r#"𝛵"#, r#"Τ"#, r#"Τ"#),
        (r#"𝛶"#, r#"𝛶"#, r#"𝛶"#, r#"Υ"#, r#"Υ"#),
        (r#"𝛷"#, r#"𝛷"#, r#"𝛷"#, r#"Φ"#, r#"Φ"#),
        (r#"𝛸"#, r#"𝛸"#, r#"𝛸"#, r#"Χ"#, r#"Χ"#),
        (r#"𝛹"#, r#"𝛹"#, r#"𝛹"#, r#"Ψ"#, r#"Ψ"#),
        (r#"𝛺"#, r#"𝛺"#, r#"𝛺"#, r#"Ω"#, r#"Ω"#),
        (r#"𝛻"#, r#"𝛻"#, r#"𝛻"#, r#"∇"#, r#"∇"#),
        (r#"𝛼"#, r#"𝛼"#, r#"𝛼"#, r#"α"#, r#"α"#),
        (r#"𝛽"#, r#"𝛽"#, r#"𝛽"#, r#"β"#, r#"β"#),
        (r#"𝛾"#, r#"𝛾"#, r#"𝛾"#, r#"γ"#, r#"γ"#),
        (r#"𝛿"#, r#"𝛿"#, r#"𝛿"#, r#"δ"#, r#"δ"#),
        (r#"𝜀"#, r#"𝜀"#, r#"𝜀"#, r#"ε"#, r#"ε"#),
        (r#"𝜁"#, r#"𝜁"#, r#"𝜁"#, r#"ζ"#, r#"ζ"#),
        (r#"𝜂"#, r#"𝜂"#, r#"𝜂"#, r#"η"#, r#"η"#),
        (r#"𝜃"#, r#"𝜃"#, r#"𝜃"#, r#"θ"#, r#"θ"#),
        (r#"𝜄"#, r#"𝜄"#, r#"𝜄"#, r#"ι"#, r#"ι"#),
        (r#"𝜅"#, r#"𝜅"#, r#"𝜅"#, r#"κ"#, r#"κ"#),
        (r#"𝜆"#, r#"𝜆"#, r#"𝜆"#, r#"λ"#, r#"λ"#),
        (r#"𝜇"#, r#"𝜇"#, r#"𝜇"#, r#"μ"#, r#"μ"#),
        (r#"𝜈"#, r#"𝜈"#, r#"𝜈"#, r#"ν"#, r#"ν"#),
        (r#"𝜉"#, r#"𝜉"#, r#"𝜉"#, r#"ξ"#, r#"ξ"#),
        (r#"𝜊"#, r#"𝜊"#, r#"𝜊"#, r#"ο"#, r#"ο"#),
        (r#"𝜋"#, r#"𝜋"#, r#"𝜋"#, r#"π"#, r#"π"#),
        (r#"𝜌"#, r#"𝜌"#, r#"𝜌"#, r#"ρ"#, r#"ρ"#),
        (r#"𝜍"#, r#"𝜍"#, r#"𝜍"#, r#"ς"#, r#"ς"#),
        (r#"𝜎"#, r#"𝜎"#, r#"𝜎"#, r#"σ"#, r#"σ"#),
        (r#"𝜏"#, r#"𝜏"#, r#"𝜏"#, r#"τ"#, r#"τ"#),
        (r#"𝜐"#, r#"𝜐"#, r#"𝜐"#, r#"υ"#, r#"υ"#),
        (r#"𝜑"#, r#"𝜑"#, r#"𝜑"#, r#"φ"#, r#"φ"#),
        (r#"𝜒"#, r#"𝜒"#, r#"𝜒"#, r#"χ"#, r#"χ"#),
        (r#"𝜓"#, r#"𝜓"#, r#"𝜓"#, r#"ψ"#, r#"ψ"#),
        (r#"𝜔"#, r#"𝜔"#, r#"𝜔"#, r#"ω"#, r#"ω"#),
        (r#"𝜕"#, r#"𝜕"#, r#"𝜕"#, r#"∂"#, r#"∂"#),
        (r#"𝜖"#, r#"𝜖"#, r#"𝜖"#, r#"ε"#, r#"ε"#),
        (r#"𝜗"#, r#"𝜗"#, r#"𝜗"#, r#"θ"#, r#"θ"#),
        (r#"𝜘"#, r#"𝜘"#, r#"𝜘"#, r#"κ"#, r#"κ"#),
        (r#"𝜙"#, r#"𝜙"#, r#"𝜙"#, r#"φ"#, r#"φ"#),
        (r#"𝜚"#, r#"𝜚"#, r#"𝜚"#, r#"ρ"#, r#"ρ"#),
        (r#"𝜛"#, r#"𝜛"#, r#"𝜛"#, r#"π"#, r#"π"#),
        (r#"𝜜"#, r#"𝜜"#, r#"𝜜"#, r#"Α"#, r#"Α"#),
        (r#"𝜝"#, r#"𝜝"#, r#"𝜝"#, r#"Β"#, r#"Β"#),
        (r#"𝜞"#, r#"𝜞"#, r#"𝜞"#, r#"Γ"#, r#"Γ"#),
        (r#"𝜟"#, r#"𝜟"#, r#"𝜟"#, r#"Δ"#, r#"Δ"#),
        (r#"𝜠"#, r#"𝜠"#, r#"𝜠"#, r#"Ε"#, r#"Ε"#),
        (r#"𝜡"#, r#"𝜡"#, r#"𝜡"#, r#"Ζ"#, r#"Ζ"#),
        (r#"𝜢"#, r#"𝜢"#, r#"𝜢"#, r#"Η"#, r#"Η"#),
        (r#"𝜣"#, r#"𝜣"#, r#"𝜣"#, r#"Θ"#, r#"Θ"#),
        (r#"𝜤"#, r#"𝜤"#, r#"𝜤"#, r#"Ι"#, r#"Ι"#),
        (r#"𝜥"#, r#"𝜥"#, r#"𝜥"#, r#"Κ"#, r#"Κ"#),
        (r#"𝜦"#, r#"𝜦"#, r#"𝜦"#, r#"Λ"#, r#"Λ"#),
        (r#"𝜧"#, r#"𝜧"#, r#"𝜧"#, r#"Μ"#, r#"Μ"#),
        (r#"𝜨"#, r#"𝜨"#, r#"𝜨"#, r#"Ν"#, r#"Ν"#),
        (r#"𝜩"#, r#"𝜩"#, r#"𝜩"#, r#"Ξ"#, r#"Ξ"#),
        (r#"𝜪"#, r#"𝜪"#, r#"𝜪"#, r#"Ο"#, r#"Ο"#),
        (r#"𝜫"#, r#"𝜫"#, r#"𝜫"#, r#"Π"#, r#"Π"#),
        (r#"𝜬"#, r#"𝜬"#, r#"𝜬"#, r#"Ρ"#, r#"Ρ"#),
        (r#"𝜭"#, r#"𝜭"#, r#"𝜭"#, r#"Θ"#, r#"Θ"#),
        (r#"𝜮"#, r#"𝜮"#, r#"𝜮"#, r#"Σ"#, r#"Σ"#),
        (r#"𝜯"#, r#"𝜯"#, r#"𝜯"#, r#"Τ"#, r#"Τ"#),
        (r#"𝜰"#, r#"𝜰"#, r#"𝜰"#, r#"Υ"#, r#"Υ"#),
        (r#"𝜱"#, r#"𝜱"#, r#"𝜱"#, r#"Φ"#, r#"Φ"#),
        (r#"𝜲"#, r#"𝜲"#, r#"𝜲"#, r#"Χ"#, r#"Χ"#),
        (r#"𝜳"#, r#"𝜳"#, r#"𝜳"#, r#"Ψ"#, r#"Ψ"#),
        (r#"𝜴"#, r#"𝜴"#, r#"𝜴"#, r#"Ω"#, r#"Ω"#),
        (r#"𝜵"#, r#"𝜵"#, r#"𝜵"#, r#"∇"#, r#"∇"#),
        (r#"𝜶"#, r#"𝜶"#, r#"𝜶"#, r#"α"#, r#"α"#),
        (r#"𝜷"#, r#"𝜷"#, r#"𝜷"#, r#"β"#, r#"β"#),
        (r#"𝜸"#, r#"𝜸"#, r#"𝜸"#, r#"γ"#, r#"γ"#),
        (r#"𝜹"#, r#"𝜹"#, r#"𝜹"#, r#"δ"#, r#"δ"#),
        (r#"𝜺"#, r#"𝜺"#, r#"𝜺"#, r#"ε"#, r#"ε"#),
        (r#"𝜻"#, r#"𝜻"#, r#"𝜻"#, r#"ζ"#, r#"ζ"#),
        (r#"𝜼"#, r#"𝜼"#, r#"𝜼"#, r#"η"#, r#"η"#),
        (r#"𝜽"#, r#"𝜽"#, r#"𝜽"#, r#"θ"#, r#"θ"#),
        (r#"𝜾"#, r#"𝜾"#, r#"𝜾"#, r#"ι"#, r#"ι"#),
        (r#"𝜿"#, r#"𝜿"#, r#"𝜿"#, r#"κ"#, r#"κ"#),
        (r#"𝝀"#, r#"𝝀"#, r#"𝝀"#, r#"λ"#, r#"λ"#),
        (r#"𝝁"#, r#"𝝁"#, r#"𝝁"#, r#"μ"#, r#"μ"#),
        (r#"𝝂"#, r#"𝝂"#, r#"𝝂"#, r#"ν"#, r#"ν"#),
        (r#"𝝃"#, r#"𝝃"#, r#"𝝃"#, r#"ξ"#, r#"ξ"#),
        (r#"𝝄"#, r#"𝝄"#, r#"𝝄"#, r#"ο"#, r#"ο"#),
        (r#"𝝅"#, r#"𝝅"#, r#"𝝅"#, r#"π"#, r#"π"#),
        (r#"𝝆"#, r#"𝝆"#, r#"𝝆"#, r#"ρ"#, r#"ρ"#),
        (r#"𝝇"#, r#"𝝇"#, r#"𝝇"#, r#"ς"#, r#"ς"#),
        (r#"𝝈"#, r#"𝝈"#, r#"𝝈"#, r#"σ"#, r#"σ"#),
        (r#"𝝉"#, r#"𝝉"#, r#"𝝉"#, r#"τ"#, r#"τ"#),
        (r#"𝝊"#, r#"𝝊"#, r#"𝝊"#, r#"υ"#, r#"υ"#),
        (r#"𝝋"#, r#"𝝋"#, r#"𝝋"#, r#"φ"#, r#"φ"#),
        (r#"𝝌"#, r#"𝝌"#, r#"𝝌"#, r#"χ"#, r#"χ"#),
        (r#"𝝍"#, r#"𝝍"#, r#"𝝍"#, r#"ψ"#, r#"ψ"#),
        (r#"𝝎"#, r#"𝝎"#, r#"𝝎"#, r#"ω"#, r#"ω"#),
        (r#"𝝏"#, r#"𝝏"#, r#"𝝏"#, r#"∂"#, r#"∂"#),
        (r#"𝝐"#, r#"𝝐"#, r#"𝝐"#, r#"ε"#, r#"ε"#),
        (r#"𝝑"#, r#"𝝑"#, r#"𝝑"#, r#"θ"#, r#"θ"#),
        (r#"𝝒"#, r#"𝝒"#, r#"𝝒"#, r#"κ"#, r#"κ"#),
        (r#"𝝓"#, r#"𝝓"#, r#"𝝓"#, r#"φ"#, r#"φ"#),
        (r#"𝝔"#, r#"𝝔"#, r#"𝝔"#, r#"ρ"#, r#"ρ"#),
        (r#"𝝕"#, r#"𝝕"#, r#"𝝕"#, r#"π"#, r#"π"#),
        (r#"𝝖"#, r#"𝝖"#, r#"𝝖"#, r#"Α"#, r#"Α"#),
        (r#"𝝗"#, r#"𝝗"#, r#"𝝗"#, r#"Β"#, r#"Β"#),
        (r#"𝝘"#, r#"𝝘"#, r#"𝝘"#, r#"Γ"#, r#"Γ"#),
        (r#"𝝙"#, r#"𝝙"#, r#"𝝙"#, r#"Δ"#, r#"Δ"#),
        (r#"𝝚"#, r#"𝝚"#, r#"𝝚"#, r#"Ε"#, r#"Ε"#),
        (r#"𝝛"#, r#"𝝛"#, r#"𝝛"#, r#"Ζ"#, r#"Ζ"#),
        (r#"𝝜"#, r#"𝝜"#, r#"𝝜"#, r#"Η"#, r#"Η"#),
        (r#"𝝝"#, r#"𝝝"#, r#"𝝝"#, r#"Θ"#, r#"Θ"#),
        (r#"𝝞"#, r#"𝝞"#, r#"𝝞"#, r#"Ι"#, r#"Ι"#),
        (r#"𝝟"#, r#"𝝟"#, r#"𝝟"#, r#"Κ"#, r#"Κ"#),
        (r#"𝝠"#, r#"𝝠"#, r#"𝝠"#, r#"Λ"#, r#"Λ"#),
        (r#"𝝡"#, r#"𝝡"#, r#"𝝡"#, r#"Μ"#, r#"Μ"#),
        (r#"𝝢"#, r#"𝝢"#, r#"𝝢"#, r#"Ν"#, r#"Ν"#),
        (r#"𝝣"#, r#"𝝣"#, r#"𝝣"#, r#"Ξ"#, r#"Ξ"#),
        (r#"𝝤"#, r#"𝝤"#, r#"𝝤"#, r#"Ο"#, r#"Ο"#),
        (r#"𝝥"#, r#"𝝥"#, r#"𝝥"#, r#"Π"#, r#"Π"#),
        (r#"𝝦"#, r#"𝝦"#, r#"𝝦"#, r#"Ρ"#, r#"Ρ"#),
        (r#"𝝧"#, r#"𝝧"#, r#"𝝧"#, r#"Θ"#, r#"Θ"#),
        (r#"𝝨"#, r#"𝝨"#, r#"𝝨"#, r#"Σ"#, r#"Σ"#),
        (r#"𝝩"#, r#"𝝩"#, r#"𝝩"#, r#"Τ"#, r#"Τ"#),
        (r#"𝝪"#, r#"𝝪"#, r#"𝝪"#, r#"Υ"#, r#"Υ"#),
        (r#"𝝫"#, r#"𝝫"#, r#"𝝫"#, r#"Φ"#, r#"Φ"#),
        (r#"𝝬"#, r#"𝝬"#, r#"𝝬"#, r#"Χ"#, r#"Χ"#),
        (r#"𝝭"#, r#"𝝭"#, r#"𝝭"#, r#"Ψ"#, r#"Ψ"#),
        (r#"𝝮"#, r#"𝝮"#, r#"𝝮"#, r#"Ω"#, r#"Ω"#),
        (r#"𝝯"#, r#"𝝯"#, r#"𝝯"#, r#"∇"#, r#"∇"#),
        (r#"𝝰"#, r#"𝝰"#, r#"𝝰"#, r#"α"#, r#"α"#),
        (r#"𝝱"#, r#"𝝱"#, r#"𝝱"#, r#"β"#, r#"β"#),
        (r#"𝝲"#, r#"𝝲"#, r#"𝝲"#, r#"γ"#, r#"γ"#),
        (r#"𝝳"#, r#"𝝳"#, r#"𝝳"#, r#"δ"#, r#"δ"#),
        (r#"𝝴"#, r#"𝝴"#, r#"𝝴"#, r#"ε"#, r#"ε"#),
        (r#"𝝵"#, r#"𝝵"#, r#"𝝵"#, r#"ζ"#, r#"ζ"#),
        (r#"𝝶"#, r#"𝝶"#, r#"𝝶"#, r#"η"#, r#"η"#),
        (r#"𝝷"#, r#"𝝷"#, r#"𝝷"#, r#"θ"#, r#"θ"#),
        (r#"𝝸"#, r#"𝝸"#, r#"𝝸"#, r#"ι"#, r#"ι"#),
        (r#"𝝹"#, r#"𝝹"#, r#"𝝹"#, r#"κ"#, r#"κ"#),
        (r#"𝝺"#, r#"𝝺"#, r#"𝝺"#, r#"λ"#, r#"λ"#),
        (r#"𝝻"#, r#"𝝻"#, r#"𝝻"#, r#"μ"#, r#"μ"#),
        (r#"𝝼"#, r#"𝝼"#, r#"𝝼"#, r#"ν"#, r#"ν"#),
        (r#"𝝽"#, r#"𝝽"#, r#"𝝽"#, r#"ξ"#, r#"ξ"#),
        (r#"𝝾"#, r#"𝝾"#, r#"𝝾"#, r#"ο"#, r#"ο"#),
        (r#"𝝿"#, r#"𝝿"#, r#"𝝿"#, r#"π"#, r#"π"#),
        (r#"𝞀"#, r#"𝞀"#, r#"𝞀"#, r#"ρ"#, r#"ρ"#),
        (r#"𝞁"#, r#"𝞁"#, r#"𝞁"#, r#"ς"#, r#"ς"#),
        (r#"𝞂"#, r#"𝞂"#, r#"𝞂"#, r#"σ"#, r#"σ"#),
        (r#"𝞃"#, r#"𝞃"#, r#"𝞃"#, r#"τ"#, r#"τ"#),
        (r#"𝞄"#, r#"𝞄"#, r#"𝞄"#, r#"υ"#, r#"υ"#),
        (r#"𝞅"#, r#"𝞅"#, r#"𝞅"#, r#"φ"#, r#"φ"#),
        (r#"𝞆"#, r#"𝞆"#, r#"𝞆"#, r#"χ"#, r#"χ"#),
        (r#"𝞇"#, r#"𝞇"#, r#"𝞇"#, r#"ψ"#, r#"ψ"#),
        (r#"𝞈"#, r#"𝞈"#, r#"𝞈"#, r#"ω"#, r#"ω"#),
        (r#"𝞉"#, r#"𝞉"#, r#"𝞉"#, r#"∂"#, r#"∂"#),
        (r#"𝞊"#, r#"𝞊"#, r#"𝞊"#, r#"ε"#, r#"ε"#),
        (r#"𝞋"#, r#"𝞋"#, r#"𝞋"#, r#"θ"#, r#"θ"#),
        (r#"𝞌"#, r#"𝞌"#, r#"𝞌"#, r#"κ"#, r#"κ"#),
        (r#"𝞍"#, r#"𝞍"#, r#"𝞍"#, r#"φ"#, r#"φ"#),
        (r#"𝞎"#, r#"𝞎"#, r#"𝞎"#, r#"ρ"#, r#"ρ"#),
        (r#"𝞏"#, r#"𝞏"#, r#"𝞏"#, r#"π"#, r#"π"#),
        (r#"𝞐"#, r#"𝞐"#, r#"𝞐"#, r#"Α"#, r#"Α"#),
        (r#"𝞑"#, r#"𝞑"#, r#"𝞑"#, r#"Β"#, r#"Β"#),
        (r#"𝞒"#, r#"𝞒"#, r#"𝞒"#, r#"Γ"#, r#"Γ"#),
        (r#"𝞓"#, r#"𝞓"#, r#"𝞓"#, r#"Δ"#, r#"Δ"#),
        (r#"𝞔"#, r#"𝞔"#, r#"𝞔"#, r#"Ε"#, r#"Ε"#),
        (r#"𝞕"#, r#"𝞕"#, r#"𝞕"#, r#"Ζ"#, r#"Ζ"#),
        (r#"𝞖"#, r#"𝞖"#, r#"𝞖"#, r#"Η"#, r#"Η"#),
        (r#"𝞗"#, r#"𝞗"#, r#"𝞗"#, r#"Θ"#, r#"Θ"#),
        (r#"𝞘"#, r#"𝞘"#, r#"𝞘"#, r#"Ι"#, r#"Ι"#),
        (r#"𝞙"#, r#"𝞙"#, r#"𝞙"#, r#"Κ"#, r#"Κ"#),
        (r#"𝞚"#, r#"𝞚"#, r#"𝞚"#, r#"Λ"#, r#"Λ"#),
        (r#"𝞛"#, r#"𝞛"#, r#"𝞛"#, r#"Μ"#, r#"Μ"#),
        (r#"𝞜"#, r#"𝞜"#, r#"𝞜"#, r#"Ν"#, r#"Ν"#),
        (r#"𝞝"#, r#"𝞝"#, r#"𝞝"#, r#"Ξ"#, r#"Ξ"#),
        (r#"𝞞"#, r#"𝞞"#, r#"𝞞"#, r#"Ο"#, r#"Ο"#),
        (r#"𝞟"#, r#"𝞟"#, r#"𝞟"#, r#"Π"#, r#"Π"#),
        (r#"𝞠"#, r#"𝞠"#, r#"𝞠"#, r#"Ρ"#, r#"Ρ"#),
        (r#"𝞡"#, r#"𝞡"#, r#"𝞡"#, r#"Θ"#, r#"Θ"#),
        (r#"𝞢"#, r#"𝞢"#, r#"𝞢"#, r#"Σ"#, r#"Σ"#),
        (r#"𝞣"#, r#"𝞣"#, r#"𝞣"#, r#"Τ"#, r#"Τ"#),
        (r#"𝞤"#, r#"𝞤"#, r#"𝞤"#, r#"Υ"#, r#"Υ"#),
        (r#"𝞥"#, r#"𝞥"#, r#"𝞥"#, r#"Φ"#, r#"Φ"#),
        (r#"𝞦"#, r#"𝞦"#, r#"𝞦"#, r#"Χ"#, r#"Χ"#),
        (r#"𝞧"#, r#"𝞧"#, r#"𝞧"#, r#"Ψ"#, r#"Ψ"#),
        (r#"𝞨"#, r#"𝞨"#, r#"𝞨"#, r#"Ω"#, r#"Ω"#),
        (r#"𝞩"#, r#"𝞩"#, r#"𝞩"#, r#"∇"#, r#"∇"#),
        (r#"𝞪"#, r#"𝞪"#, r#"𝞪"#, r#"α"#, r#"α"#),
        (r#"𝞫"#, r#"𝞫"#, r#"𝞫"#, r#"β"#, r#"β"#),
        (r#"𝞬"#, r#"𝞬"#, r#"𝞬"#, r#"γ"#, r#"γ"#),
        (r#"𝞭"#, r#"𝞭"#, r#"𝞭"#, r#"δ"#, r#"δ"#),
        (r#"𝞮"#, r#"𝞮"#, r#"𝞮"#, r#"ε"#, r#"ε"#),
        (r#"𝞯"#, r#"𝞯"#, r#"𝞯"#, r#"ζ"#, r#"ζ"#),
        (r#"𝞰"#, r#"𝞰"#, r#"𝞰"#, r#"η"#, r#"η"#),
        (r#"𝞱"#, r#"𝞱"#, r#"𝞱"#, r#"θ"#, r#"θ"#),
        (r#"𝞲"#, r#"𝞲"#, r#"𝞲"#, r#"ι"#, r#"ι"#),
        (r#"𝞳"#, r#"𝞳"#, r#"𝞳"#, r#"κ"#, r#"κ"#),
        (r#"𝞴"#, r#"𝞴"#, r#"𝞴"#, r#"λ"#, r#"λ"#),
        (r#"𝞵"#, r#"𝞵"#, r#"𝞵"#, r#"μ"#, r#"μ"#),
        (r#"𝞶"#, r#"𝞶"#, r#"𝞶"#, r#"ν"#, r#"ν"#),
        (r#"𝞷"#, r#"𝞷"#, r#"𝞷"#, r#"ξ"#, r#"ξ"#),
        (r#"𝞸"#, r#"𝞸"#, r#"𝞸"#, r#"ο"#, r#"ο"#),
        (r#"𝞹"#, r#"𝞹"#, r#"𝞹"#, r#"π"#, r#"π"#),
        (r#"𝞺"#, r#"𝞺"#, r#"𝞺"#, r#"ρ"#, r#"ρ"#),
        (r#"𝞻"#, r#"𝞻"#, r#"𝞻"#, r#"ς"#, r#"ς"#),
        (r#"𝞼"#, r#"𝞼"#, r#"𝞼"#, r#"σ"#, r#"σ"#),
        (r#"𝞽"#, r#"𝞽"#, r#"𝞽"#, r#"τ"#, r#"τ"#),
        (r#"𝞾"#, r#"𝞾"#, r#"𝞾"#, r#"υ"#, r#"υ"#),
        (r#"𝞿"#, r#"𝞿"#, r#"𝞿"#, r#"φ"#, r#"φ"#),
        (r#"𝟀"#, r#"𝟀"#, r#"𝟀"#, r#"χ"#, r#"χ"#),
        (r#"𝟁"#, r#"𝟁"#, r#"𝟁"#, r#"ψ"#, r#"ψ"#),
        (r#"𝟂"#, r#"𝟂"#, r#"𝟂"#, r#"ω"#, r#"ω"#),
        (r#"𝟃"#, r#"𝟃"#, r#"𝟃"#, r#"∂"#, r#"∂"#),
        (r#"𝟄"#, r#"𝟄"#, r#"𝟄"#, r#"ε"#, r#"ε"#),
        (r#"𝟅"#, r#"𝟅"#, r#"𝟅"#, r#"θ"#, r#"θ"#),
        (r#"𝟆"#, r#"𝟆"#, r#"𝟆"#, r#"κ"#, r#"κ"#),
        (r#"𝟇"#, r#"𝟇"#, r#"𝟇"#, r#"φ"#, r#"φ"#),
        (r#"𝟈"#, r#"𝟈"#, r#"𝟈"#, r#"ρ"#, r#"ρ"#),
        (r#"𝟉"#, r#"𝟉"#, r#"𝟉"#, r#"π"#, r#"π"#),
        (r#"𝟊"#, r#"𝟊"#, r#"𝟊"#, r#"Ϝ"#, r#"Ϝ"#),
        (r#"𝟋"#, r#"𝟋"#, r#"𝟋"#, r#"ϝ"#, r#"ϝ"#),
        (r#"𝟎"#, r#"𝟎"#, r#"𝟎"#, r#"0"#, r#"0"#),
        (r#"𝟏"#, r#"𝟏"#, r#"𝟏"#, r#"1"#, r#"1"#),
        (r#"𝟐"#, r#"𝟐"#, r#"𝟐"#, r#"2"#, r#"2"#),
        (r#"𝟑"#, r#"𝟑"#, r#"𝟑"#, r#"3"#, r#"3"#),
        (r#"𝟒"#, r#"𝟒"#, r#"𝟒"#, r#"4"#, r#"4"#),
        (r#"𝟓"#, r#"𝟓"#, r#"𝟓"#, r#"5"#, r#"5"#),
        (r#"𝟔"#, r#"𝟔"#, r#"𝟔"#, r#"6"#, r#"6"#),
        (r#"𝟕"#, r#"𝟕"#, r#"𝟕"#, r#"7"#, r#"7"#),
        (r#"𝟖"#, r#"𝟖"#, r#"𝟖"#, r#"8"#, r#"8"#),
        (r#"𝟗"#, r#"𝟗"#, r#"𝟗"#, r#"9"#, r#"9"#),
        (r#"𝟘"#, r#"𝟘"#, r#"𝟘"#, r#"0"#, r#"0"#),
        (r#"𝟙"#, r#"𝟙"#, r#"𝟙"#, r#"1"#, r#"1"#),
        (r#"𝟚"#, r#"𝟚"#, r#"𝟚"#, r#"2"#, r#"2"#),
        (r#"𝟛"#, r#"𝟛"#, r#"𝟛"#, r#"3"#, r#"3"#),
        (r#"𝟜"#, r#"𝟜"#, r#"𝟜"#, r#"4"#, r#"4"#),
        (r#"𝟝"#, r#"𝟝"#, r#"𝟝"#, r#"5"#, r#"5"#),
        (r#"𝟞"#, r#"𝟞"#, r#"𝟞"#, r#"6"#, r#"6"#),
        (r#"𝟟"#, r#"𝟟"#, r#"𝟟"#, r#"7"#, r#"7"#),
        (r#"𝟠"#, r#"𝟠"#, r#"𝟠"#, r#"8"#, r#"8"#),
        (r#"𝟡"#, r#"𝟡"#, r#"𝟡"#, r#"9"#, r#"9"#),
        (r#"𝟢"#, r#"𝟢"#, r#"𝟢"#, r#"0"#, r#"0"#),
        (r#"𝟣"#, r#"𝟣"#, r#"𝟣"#, r#"1"#, r#"1"#),
        (r#"𝟤"#, r#"𝟤"#, r#"𝟤"#, r#"2"#, r#"2"#),
        (r#"𝟥"#, r#"𝟥"#, r#"𝟥"#, r#"3"#, r#"3"#),
        (r#"𝟦"#, r#"𝟦"#, r#"𝟦"#, r#"4"#, r#"4"#),
        (r#"𝟧"#, r#"𝟧"#, r#"𝟧"#, r#"5"#, r#"5"#),
        (r#"𝟨"#, r#"𝟨"#, r#"𝟨"#, r#"6"#, r#"6"#),
        (r#"𝟩"#, r#"𝟩"#, r#"𝟩"#, r#"7"#, r#"7"#),
        (r#"𝟪"#, r#"𝟪"#, r#"𝟪"#, r#"8"#, r#"8"#),
        (r#"𝟫"#, r#"𝟫"#, r#"𝟫"#, r#"9"#, r#"9"#),
        (r#"𝟬"#, r#"𝟬"#, r#"𝟬"#, r#"0"#, r#"0"#),
        (r#"𝟭"#, r#"𝟭"#, r#"𝟭"#, r#"1"#, r#"1"#),
        (r#"𝟮"#, r#"𝟮"#, r#"𝟮"#, r#"2"#, r#"2"#),
        (r#"𝟯"#, r#"𝟯"#, r#"𝟯"#, r#"3"#, r#"3"#),
        (r#"𝟰"#, r#"𝟰"#, r#"𝟰"#, r#"4"#, r#"4"#),
        (r#"𝟱"#, r#"𝟱"#, r#"𝟱"#, r#"5"#, r#"5"#),
        (r#"𝟲"#, r#"𝟲"#, r#"𝟲"#, r#"6"#, r#"6"#),
        (r#"𝟳"#, r#"𝟳"#, r#"𝟳"#, r#"7"#, r#"7"#),
        (r#"𝟴"#, r#"𝟴"#, r#"𝟴"#, r#"8"#, r#"8"#),
        (r#"𝟵"#, r#"𝟵"#, r#"𝟵"#, r#"9"#, r#"9"#),
        (r#"𝟶"#, r#"𝟶"#, r#"𝟶"#, r#"0"#, r#"0"#),
        (r#"𝟷"#, r#"𝟷"#, r#"𝟷"#, r#"1"#, r#"1"#),
        (r#"𝟸"#, r#"𝟸"#, r#"𝟸"#, r#"2"#, r#"2"#),
        (r#"𝟹"#, r#"𝟹"#, r#"𝟹"#, r#"3"#, r#"3"#),
        (r#"𝟺"#, r#"𝟺"#, r#"𝟺"#, r#"4"#, r#"4"#),
        (r#"𝟻"#, r#"𝟻"#, r#"𝟻"#, r#"5"#, r#"5"#),
        (r#"𝟼"#, r#"𝟼"#, r#"𝟼"#, r#"6"#, r#"6"#),
        (r#"𝟽"#, r#"𝟽"#, r#"𝟽"#, r#"7"#, r#"7"#),
        (r#"𝟾"#, r#"𝟾"#, r#"𝟾"#, r#"8"#, r#"8"#),
        (r#"𝟿"#, r#"𝟿"#, r#"𝟿"#, r#"9"#, r#"9"#),
        (r#"𞀰"#, r#"𞀰"#, r#"𞀰"#, r#"а"#, r#"а"#),
        (r#"𞀱"#, r#"𞀱"#, r#"𞀱"#, r#"б"#, r#"б"#),
        (r#"𞀲"#, r#"𞀲"#, r#"𞀲"#, r#"в"#, r#"в"#),
        (r#"𞀳"#, r#"𞀳"#, r#"𞀳"#, r#"г"#, r#"г"#),
        (r#"𞀴"#, r#"𞀴"#, r#"𞀴"#, r#"д"#, r#"д"#),
        (r#"𞀵"#, r#"𞀵"#, r#"𞀵"#, r#"е"#, r#"е"#),
        (r#"𞀶"#, r#"𞀶"#, r#"𞀶"#, r#"ж"#, r#"ж"#),
        (r#"𞀷"#, r#"𞀷"#, r#"𞀷"#, r#"з"#, r#"з"#),
        (r#"𞀸"#, r#"𞀸"#, r#"𞀸"#, r#"и"#, r#"и"#),
        (r#"𞀹"#, r#"𞀹"#, r#"𞀹"#, r#"к"#, r#"к"#),
        (r#"𞀺"#, r#"𞀺"#, r#"𞀺"#, r#"л"#, r#"л"#),
        (r#"𞀻"#, r#"𞀻"#, r#"𞀻"#, r#"м"#, r#"м"#),
        (r#"𞀼"#, r#"𞀼"#, r#"𞀼"#, r#"о"#, r#"о"#),
        (r#"𞀽"#, r#"𞀽"#, r#"𞀽"#, r#"п"#, r#"п"#),
        (r#"𞀾"#, r#"𞀾"#, r#"𞀾"#, r#"р"#, r#"р"#),
        (r#"𞀿"#, r#"𞀿"#, r#"𞀿"#, r#"с"#, r#"с"#),
        (r#"𞁀"#, r#"𞁀"#, r#"𞁀"#, r#"т"#, r#"т"#),
        (r#"𞁁"#, r#"𞁁"#, r#"𞁁"#, r#"у"#, r#"у"#),
        (r#"𞁂"#, r#"𞁂"#, r#"𞁂"#, r#"ф"#, r#"ф"#),
        (r#"𞁃"#, r#"𞁃"#, r#"𞁃"#, r#"х"#, r#"х"#),
        (r#"𞁄"#, r#"𞁄"#, r#"𞁄"#, r#"ц"#, r#"ц"#),
        (r#"𞁅"#, r#"𞁅"#, r#"𞁅"#, r#"ч"#, r#"ч"#),
        (r#"𞁆"#, r#"𞁆"#, r#"𞁆"#, r#"ш"#, r#"ш"#),
        (r#"𞁇"#, r#"𞁇"#, r#"𞁇"#, r#"ы"#, r#"ы"#),
        (r#"𞁈"#, r#"𞁈"#, r#"𞁈"#, r#"э"#, r#"э"#),
        (r#"𞁉"#, r#"𞁉"#, r#"𞁉"#, r#"ю"#, r#"ю"#),
        (r#"𞁊"#, r#"𞁊"#, r#"𞁊"#, r#"ꚉ"#, r#"ꚉ"#),
        (r#"𞁋"#, r#"𞁋"#, r#"𞁋"#, r#"ә"#, r#"ә"#),
        (r#"𞁌"#, r#"𞁌"#, r#"𞁌"#, r#"і"#, r#"і"#),
        (r#"𞁍"#, r#"𞁍"#, r#"𞁍"#, r#"ј"#, r#"ј"#),
        (r#"𞁎"#, r#"𞁎"#, r#"𞁎"#, r#"ө"#, r#"ө"#),
        (r#"𞁏"#, r#"𞁏"#, r#"𞁏"#, r#"ү"#, r#"ү"#),
        (r#"𞁐"#, r#"𞁐"#, r#"𞁐"#, r#"ӏ"#, r#"ӏ"#),
        (r#"𞁑"#, r#"𞁑"#, r#"𞁑"#, r#"а"#, r#"а"#),
        (r#"𞁒"#, r#"𞁒"#, r#"𞁒"#, r#"б"#, r#"б"#),
        (r#"𞁓"#, r#"𞁓"#, r#"𞁓"#, r#"в"#, r#"в"#),
        (r#"𞁔"#, r#"𞁔"#, r#"𞁔"#, r#"г"#, r#"г"#),
        (r#"𞁕"#, r#"𞁕"#, r#"𞁕"#, r#"д"#, r#"д"#),
        (r#"𞁖"#, r#"𞁖"#, r#"𞁖"#, r#"е"#, r#"е"#),
        (r#"𞁗"#, r#"𞁗"#, r#"𞁗"#, r#"ж"#, r#"ж"#),
        (r#"𞁘"#, r#"𞁘"#, r#"𞁘"#, r#"з"#, r#"з"#),
        (r#"𞁙"#, r#"𞁙"#, r#"𞁙"#, r#"и"#, r#"и"#),
        (r#"𞁚"#, r#"𞁚"#, r#"𞁚"#, r#"к"#, r#"к"#),
        (r#"𞁛"#, r#"𞁛"#, r#"𞁛"#, r#"л"#, r#"л"#),
        (r#"𞁜"#, r#"𞁜"#, r#"𞁜"#, r#"о"#, r#"о"#),
        (r#"𞁝"#, r#"𞁝"#, r#"𞁝"#, r#"п"#, r#"п"#),
        (r#"𞁞"#, r#"𞁞"#, r#"𞁞"#, r#"с"#, r#"с"#),
        (r#"𞁟"#, r#"𞁟"#, r#"𞁟"#, r#"у"#, r#"у"#),
        (r#"𞁠"#, r#"𞁠"#, r#"𞁠"#, r#"ф"#, r#"ф"#),
        (r#"𞁡"#, r#"𞁡"#, r#"𞁡"#, r#"х"#, r#"х"#),
        (r#"𞁢"#, r#"𞁢"#, r#"𞁢"#, r#"ц"#, r#"ц"#),
        (r#"𞁣"#, r#"𞁣"#, r#"𞁣"#, r#"ч"#, r#"ч"#),
        (r#"𞁤"#, r#"𞁤"#, r#"𞁤"#, r#"ш"#, r#"ш"#),
        (r#"𞁥"#, r#"𞁥"#, r#"𞁥"#, r#"ъ"#, r#"ъ"#),
        (r#"𞁦"#, r#"𞁦"#, r#"𞁦"#, r#"ы"#, r#"ы"#),
        (r#"𞁧"#, r#"𞁧"#, r#"𞁧"#, r#"ґ"#, r#"ґ"#),
        (r#"𞁨"#, r#"𞁨"#, r#"𞁨"#, r#"і"#, r#"і"#),
        (r#"𞁩"#, r#"𞁩"#, r#"𞁩"#, r#"ѕ"#, r#"ѕ"#),
        (r#"𞁪"#, r#"𞁪"#, r#"𞁪"#, r#"џ"#, r#"џ"#),
        (r#"𞁫"#, r#"𞁫"#, r#"𞁫"#, r#"ҫ"#, r#"ҫ"#),
        (r#"𞁬"#, r#"𞁬"#, r#"𞁬"#, r#"ꙑ"#, r#"ꙑ"#),
        (r#"𞁭"#, r#"𞁭"#, r#"𞁭"#, r#"ұ"#, r#"ұ"#),
        (r#"𞸀"#, r#"𞸀"#, r#"𞸀"#, r#"ا"#, r#"ا"#),
        (r#"𞸁"#, r#"𞸁"#, r#"𞸁"#, r#"ب"#, r#"ب"#),
        (r#"𞸂"#, r#"𞸂"#, r#"𞸂"#, r#"ج"#, r#"ج"#),
        (r#"𞸃"#, r#"𞸃"#, r#"𞸃"#, r#"د"#, r#"د"#),
        (r#"𞸅"#, r#"𞸅"#, r#"𞸅"#, r#"و"#, r#"و"#),
        (r#"𞸆"#, r#"𞸆"#, r#"𞸆"#, r#"ز"#, r#"ز"#),
        (r#"𞸇"#, r#"𞸇"#, r#"𞸇"#, r#"ح"#, r#"ح"#),
        (r#"𞸈"#, r#"𞸈"#, r#"𞸈"#, r#"ط"#, r#"ط"#),
        (r#"𞸉"#, r#"𞸉"#, r#"𞸉"#, r#"ي"#, r#"ي"#),
        (r#"𞸊"#, r#"𞸊"#, r#"𞸊"#, r#"ك"#, r#"ك"#),
        (r#"𞸋"#, r#"𞸋"#, r#"𞸋"#, r#"ل"#, r#"ل"#),
        (r#"𞸌"#, r#"𞸌"#, r#"𞸌"#, r#"م"#, r#"م"#),
        (r#"𞸍"#, r#"𞸍"#, r#"𞸍"#, r#"ن"#, r#"ن"#),
        (r#"𞸎"#, r#"𞸎"#, r#"𞸎"#, r#"س"#, r#"س"#),
        (r#"𞸏"#, r#"𞸏"#, r#"𞸏"#, r#"ع"#, r#"ع"#),
        (r#"𞸐"#, r#"𞸐"#, r#"𞸐"#, r#"ف"#, r#"ف"#),
        (r#"𞸑"#, r#"𞸑"#, r#"𞸑"#, r#"ص"#, r#"ص"#),
        (r#"𞸒"#, r#"𞸒"#, r#"𞸒"#, r#"ق"#, r#"ق"#),
        (r#"𞸓"#, r#"𞸓"#, r#"𞸓"#, r#"ر"#, r#"ر"#),
        (r#"𞸔"#, r#"𞸔"#, r#"𞸔"#, r#"ش"#, r#"ش"#),
        (r#"𞸕"#, r#"𞸕"#, r#"𞸕"#, r#"ت"#, r#"ت"#),
        (r#"𞸖"#, r#"𞸖"#, r#"𞸖"#, r#"ث"#, r#"ث"#),
        (r#"𞸗"#, r#"𞸗"#, r#"𞸗"#, r#"خ"#, r#"خ"#),
        (r#"𞸘"#, r#"𞸘"#, r#"𞸘"#, r#"ذ"#, r#"ذ"#),
        (r#"𞸙"#, r#"𞸙"#, r#"𞸙"#, r#"ض"#, r#"ض"#),
        (r#"𞸚"#, r#"𞸚"#, r#"𞸚"#, r#"ظ"#, r#"ظ"#),
        (r#"𞸛"#, r#"𞸛"#, r#"𞸛"#, r#"غ"#, r#"غ"#),
        (r#"𞸜"#, r#"𞸜"#, r#"𞸜"#, r#"ٮ"#, r#"ٮ"#),
        (r#"𞸝"#, r#"𞸝"#, r#"𞸝"#, r#"ں"#, r#"ں"#),
        (r#"𞸞"#, r#"𞸞"#, r#"𞸞"#, r#"ڡ"#, r#"ڡ"#),
        (r#"𞸟"#, r#"𞸟"#, r#"𞸟"#, r#"ٯ"#, r#"ٯ"#),
        (r#"𞸡"#, r#"𞸡"#, r#"𞸡"#, r#"ب"#, r#"ب"#),
        (r#"𞸢"#, r#"𞸢"#, r#"𞸢"#, r#"ج"#, r#"ج"#),
        (r#"𞸤"#, r#"𞸤"#, r#"𞸤"#, r#"ه"#, r#"ه"#),
        (r#"𞸧"#, r#"𞸧"#, r#"𞸧"#, r#"ح"#, r#"ح"#),
        (r#"𞸩"#, r#"𞸩"#, r#"𞸩"#, r#"ي"#, r#"ي"#),
        (r#"𞸪"#, r#"𞸪"#, r#"𞸪"#, r#"ك"#, r#"ك"#),
        (r#"𞸫"#, r#"𞸫"#, r#"𞸫"#, r#"ل"#, r#"ل"#),
        (r#"𞸬"#, r#"𞸬"#, r#"𞸬"#, r#"م"#, r#"م"#),
        (r#"𞸭"#, r#"𞸭"#, r#"𞸭"#, r#"ن"#, r#"ن"#),
        (r#"𞸮"#, r#"𞸮"#, r#"𞸮"#, r#"س"#, r#"س"#),
        (r#"𞸯"#, r#"𞸯"#, r#"𞸯"#, r#"ع"#, r#"ع"#),
        (r#"𞸰"#, r#"𞸰"#, r#"𞸰"#, r#"ف"#, r#"ف"#),
        (r#"𞸱"#, r#"𞸱"#, r#"𞸱"#, r#"ص"#, r#"ص"#),
        (r#"𞸲"#, r#"𞸲"#, r#"𞸲"#, r#"ق"#, r#"ق"#),
        (r#"𞸴"#, r#"𞸴"#, r#"𞸴"#, r#"ش"#, r#"ش"#),
        (r#"𞸵"#, r#"𞸵"#, r#"𞸵"#, r#"ت"#, r#"ت"#),
        (r#"𞸶"#, r#"𞸶"#, r#"𞸶"#, r#"ث"#, r#"ث"#),
        (r#"𞸷"#, r#"𞸷"#, r#"𞸷"#, r#"خ"#, r#"خ"#),
        (r#"𞸹"#, r#"𞸹"#, r#"𞸹"#, r#"ض"#, r#"ض"#),
        (r#"𞸻"#, r#"𞸻"#, r#"𞸻"#, r#"غ"#, r#"غ"#),
        (r#"𞹂"#, r#"𞹂"#, r#"𞹂"#, r#"ج"#, r#"ج"#),
        (r#"𞹇"#, r#"𞹇"#, r#"𞹇"#, r#"ح"#, r#"ح"#),
        (r#"𞹉"#, r#"𞹉"#, r#"𞹉"#, r#"ي"#, r#"ي"#),
        (r#"𞹋"#, r#"𞹋"#, r#"𞹋"#, r#"ل"#, r#"ل"#),
        (r#"𞹍"#, r#"𞹍"#, r#"𞹍"#, r#"ن"#, r#"ن"#),
        (r#"𞹎"#, r#"𞹎"#, r#"𞹎"#, r#"س"#, r#"س"#),
        (r#"𞹏"#, r#"𞹏"#, r#"𞹏"#, r#"ع"#, r#"ع"#),
        (r#"𞹑"#, r#"𞹑"#, r#"𞹑"#, r#"ص"#, r#"ص"#),
        (r#"𞹒"#, r#"𞹒"#, r#"𞹒"#, r#"ق"#, r#"ق"#),
        (r#"𞹔"#, r#"𞹔"#, r#"𞹔"#, r#"ش"#, r#"ش"#),
        (r#"𞹗"#, r#"𞹗"#, r#"𞹗"#, r#"خ"#, r#"خ"#),
        (r#"𞹙"#, r#"𞹙"#, r#"𞹙"#, r#"ض"#, r#"ض"#),
        (r#"𞹛"#, r#"𞹛"#, r#"𞹛"#, r#"غ"#, r#"غ"#),
        (r#"𞹝"#, r#"𞹝"#, r#"𞹝"#, r#"ں"#, r#"ں"#),
        (r#"𞹟"#, r#"𞹟"#, r#"𞹟"#, r#"ٯ"#, r#"ٯ"#),
        (r#"𞹡"#, r#"𞹡"#, r#"𞹡"#, r#"ب"#, r#"ب"#),
        (r#"𞹢"#, r#"𞹢"#, r#"𞹢"#, r#"ج"#, r#"ج"#),
        (r#"𞹤"#, r#"𞹤"#, r#"𞹤"#, r#"ه"#, r#"ه"#),
        (r#"𞹧"#, r#"𞹧"#, r#"𞹧"#, r#"ح"#, r#"ح"#),
        (r#"𞹨"#, r#"𞹨"#, r#"𞹨"#, r#"ط"#, r#"ط"#),
        (r#"𞹩"#, r#"𞹩"#, r#"𞹩"#, r#"ي"#, r#"ي"#),
        (r#"𞹪"#, r#"𞹪"#, r#"𞹪"#, r#"ك"#, r#"ك"#),
        (r#"𞹬"#, r#"𞹬"#, r#"𞹬"#, r#"م"#, r#"م"#),
        (r#"𞹭"#, r#"𞹭"#, r#"𞹭"#, r#"ن"#, r#"ن"#),
        (r#"𞹮"#, r#"𞹮"#, r#"𞹮"#, r#"س"#, r#"س"#),
        (r#"𞹯"#, r#"𞹯"#, r#"𞹯"#, r#"ع"#, r#"ع"#),
        (r#"𞹰"#, r#"𞹰"#, r#"𞹰"#, r#"ف"#, r#"ف"#),
        (r#"𞹱"#, r#"𞹱"#, r#"𞹱"#, r#"ص"#, r#"ص"#),
        (r#"𞹲"#, r#"𞹲"#, r#"𞹲"#, r#"ق"#, r#"ق"#),
        (r#"𞹴"#, r#"𞹴"#, r#"𞹴"#, r#"ش"#, r#"ش"#),
        (r#"𞹵"#, r#"𞹵"#, r#"𞹵"#, r#"ت"#, r#"ت"#),
        (r#"𞹶"#, r#"𞹶"#, r#"𞹶"#, r#"ث"#, r#"ث"#),
        (r#"𞹷"#, r#"𞹷"#, r#"𞹷"#, r#"خ"#, r#"خ"#),
        (r#"𞹹"#, r#"𞹹"#, r#"𞹹"#, r#"ض"#, r#"ض"#),
        (r#"𞹺"#, r#"𞹺"#, r#"𞹺"#, r#"ظ"#, r#"ظ"#),
        (r#"𞹻"#, r#"𞹻"#, r#"𞹻"#, r#"غ"#, r#"غ"#),
        (r#"𞹼"#, r#"𞹼"#, r#"𞹼"#, r#"ٮ"#, r#"ٮ"#),
        (r#"𞹾"#, r#"𞹾"#, r#"𞹾"#, r#"ڡ"#, r#"ڡ"#),
        (r#"𞺀"#, r#"𞺀"#, r#"𞺀"#, r#"ا"#, r#"ا"#),
        (r#"𞺁"#, r#"𞺁"#, r#"𞺁"#, r#"ب"#, r#"ب"#),
        (r#"𞺂"#, r#"𞺂"#, r#"𞺂"#, r#"ج"#, r#"ج"#),
        (r#"𞺃"#, r#"𞺃"#, r#"𞺃"#, r#"د"#, r#"د"#),
        (r#"𞺄"#, r#"𞺄"#, r#"𞺄"#, r#"ه"#, r#"ه"#),
        (r#"𞺅"#, r#"𞺅"#, r#"𞺅"#, r#"و"#, r#"و"#),
        (r#"𞺆"#, r#"𞺆"#, r#"𞺆"#, r#"ز"#, r#"ز"#),
        (r#"𞺇"#, r#"𞺇"#, r#"𞺇"#, r#"ح"#, r#"ح"#),
        (r#"𞺈"#, r#"𞺈"#, r#"𞺈"#, r#"ط"#, r#"ط"#),
        (r#"𞺉"#, r#"𞺉"#, r#"𞺉"#, r#"ي"#, r#"ي"#),
        (r#"𞺋"#, r#"𞺋"#, r#"𞺋"#, r#"ل"#, r#"ل"#),
        (r#"𞺌"#, r#"𞺌"#, r#"𞺌"#, r#"م"#, r#"م"#),
        (r#"𞺍"#, r#"𞺍"#, r#"𞺍"#, r#"ن"#, r#"ن"#),
        (r#"𞺎"#, r#"𞺎"#, r#"𞺎"#, r#"س"#, r#"س"#),
        (r#"𞺏"#, r#"𞺏"#, r#"𞺏"#, r#"ع"#, r#"ع"#),
        (r#"𞺐"#, r#"𞺐"#, r#"𞺐"#, r#"ف"#, r#"ف"#),
        (r#"𞺑"#, r#"𞺑"#, r#"𞺑"#, r#"ص"#, r#"ص"#),
        (r#"𞺒"#, r#"𞺒"#, r#"𞺒"#, r#"ق"#, r#"ق"#),
        (r#"𞺓"#, r#"𞺓"#, r#"𞺓"#, r#"ر"#, r#"ر"#),
        (r#"𞺔"#, r#"𞺔"#, r#"𞺔"#, r#"ش"#, r#"ش"#),
        (r#"𞺕"#, r#"𞺕"#, r#"𞺕"#, r#"ت"#, r#"ت"#),
        (r#"𞺖"#, r#"𞺖"#, r#"𞺖"#, r#"ث"#, r#"ث"#),
        (r#"𞺗"#, r#"𞺗"#, r#"𞺗"#, r#"خ"#, r#"خ"#),
        (r#"𞺘"#, r#"𞺘"#, r#"𞺘"#, r#"ذ"#, r#"ذ"#),
        (r#"𞺙"#, r#"𞺙"#, r#"𞺙"#, r#"ض"#, r#"ض"#),
        (r#"𞺚"#, r#"𞺚"#, r#"𞺚"#, r#"ظ"#, r#"ظ"#),
        (r#"𞺛"#, r#"𞺛"#, r#"𞺛"#, r#"غ"#, r#"غ"#),
        (r#"𞺡"#, r#"𞺡"#, r#"𞺡"#, r#"ب"#, r#"ب"#),
        (r#"𞺢"#, r#"𞺢"#, r#"𞺢"#, r#"ج"#, r#"ج"#),
        (r#"𞺣"#, r#"𞺣"#, r#"𞺣"#, r#"د"#, r#"د"#),
        (r#"𞺥"#, r#"𞺥"#, r#"𞺥"#, r#"و"#, r#"و"#),
        (r#"𞺦"#, r#"𞺦"#, r#"𞺦"#, r#"ز"#, r#"ز"#),
        (r#"𞺧"#, r#"𞺧"#, r#"𞺧"#, r#"ح"#, r#"ح"#),
        (r#"𞺨"#, r#"𞺨"#, r#"𞺨"#, r#"ط"#, r#"ط"#),
        (r#"𞺩"#, r#"𞺩"#, r#"𞺩"#, r#"ي"#, r#"ي"#),
        (r#"𞺫"#, r#"𞺫"#, r#"𞺫"#, r#"ل"#, r#"ل"#),
        (r#"𞺬"#, r#"𞺬"#, r#"𞺬"#, r#"م"#, r#"م"#),
        (r#"𞺭"#, r#"𞺭"#, r#"𞺭"#, r#"ن"#, r#"ن"#),
        (r#"𞺮"#, r#"𞺮"#, r#"𞺮"#, r#"س"#, r#"س"#),
        (r#"𞺯"#, r#"𞺯"#, r#"𞺯"#, r#"ع"#, r#"ع"#),
        (r#"𞺰"#, r#"𞺰"#, r#"𞺰"#, r#"ف"#, r#"ف"#),
        (r#"𞺱"#, r#"𞺱"#, r#"𞺱"#, r#"ص"#, r#"ص"#),
        (r#"𞺲"#, r#"𞺲"#, r#"𞺲"#, r#"ق"#, r#"ق"#),
        (r#"𞺳"#, r#"𞺳"#, r#"𞺳"#, r#"ر"#, r#"ر"#),
        (r#"𞺴"#, r#"𞺴"#, r#"𞺴"#, r#"ش"#, r#"ش"#),
        (r#"𞺵"#, r#"𞺵"#, r#"𞺵"#, r#"ت"#, r#"ت"#),
        (r#"𞺶"#, r#"𞺶"#, r#"𞺶"#, r#"ث"#, r#"ث"#),
        (r#"𞺷"#, r#"𞺷"#, r#"𞺷"#, r#"خ"#, r#"خ"#),
        (r#"𞺸"#, r#"𞺸"#, r#"𞺸"#, r#"ذ"#, r#"ذ"#),
        (r#"𞺹"#, r#"𞺹"#, r#"𞺹"#, r#"ض"#, r#"ض"#),
        (r#"𞺺"#, r#"𞺺"#, r#"𞺺"#, r#"ظ"#, r#"ظ"#),
        (r#"𞺻"#, r#"𞺻"#, r#"𞺻"#, r#"غ"#, r#"غ"#),
        (r#"🄀"#, r#"🄀"#, r#"🄀"#, r#"0."#, r#"0."#),
        (r#"🄁"#, r#"🄁"#, r#"🄁"#, r#"0,"#, r#"0,"#),
        (r#"🄂"#, r#"🄂"#, r#"🄂"#, r#"1,"#, r#"1,"#),
        (r#"🄃"#, r#"🄃"#, r#"🄃"#, r#"2,"#, r#"2,"#),
        (r#"🄄"#, r#"🄄"#, r#"🄄"#, r#"3,"#, r#"3,"#),
        (r#"🄅"#, r#"🄅"#, r#"🄅"#, r#"4,"#, r#"4,"#),
        (r#"🄆"#, r#"🄆"#, r#"🄆"#, r#"5,"#, r#"5,"#),
        (r#"🄇"#, r#"🄇"#, r#"🄇"#, r#"6,"#, r#"6,"#),
        (r#"🄈"#, r#"🄈"#, r#"🄈"#, r#"7,"#, r#"7,"#),
        (r#"🄉"#, r#"🄉"#, r#"🄉"#, r#"8,"#, r#"8,"#),
        (r#"🄊"#, r#"🄊"#, r#"🄊"#, r#"9,"#, r#"9,"#),
        (r#"🄐"#, r#"🄐"#, r#"🄐"#, r#"(A)"#, r#"(A)"#),
        (r#"🄑"#, r#"🄑"#, r#"🄑"#, r#"(B)"#, r#"(B)"#),
        (r#"🄒"#, r#"🄒"#, r#"🄒"#, r#"(C)"#, r#"(C)"#),
        (r#"🄓"#, r#"🄓"#, r#"🄓"#, r#"(D)"#, r#"(D)"#),
        (r#"🄔"#, r#"🄔"#, r#"🄔"#, r#"(E)"#, r#"(E)"#),
        (r#"🄕"#, r#"🄕"#, r#"🄕"#, r#"(F)"#, r#"(F)"#),
        (r#"🄖"#, r#"🄖"#, r#"🄖"#, r#"(G)"#, r#"(G)"#),
        (r#"🄗"#, r#"🄗"#, r#"🄗"#, r#"(H)"#, r#"(H)"#),
        (r#"🄘"#, r#"🄘"#, r#"🄘"#, r#"(I)"#, r#"(I)"#),
        (r#"🄙"#, r#"🄙"#, r#"🄙"#, r#"(J)"#, r#"(J)"#),
        (r#"🄚"#, r#"🄚"#, r#"🄚"#, r#"(K)"#, r#"(K)"#),
        (r#"🄛"#, r#"🄛"#, r#"🄛"#, r#"(L)"#, r#"(L)"#),
        (r#"🄜"#, r#"🄜"#, r#"🄜"#, r#"(M)"#, r#"(M)"#),
        (r#"🄝"#, r#"🄝"#, r#"🄝"#, r#"(N)"#, r#"(N)"#),
        (r#"🄞"#, r#"🄞"#, r#"🄞"#, r#"(O)"#, r#"(O)"#),
        (r#"🄟"#, r#"🄟"#, r#"🄟"#, r#"(P)"#, r#"(P)"#),
        (r#"🄠"#, r#"🄠"#, r#"🄠"#, r#"(Q)"#, r#"(Q)"#),
        (r#"🄡"#, r#"🄡"#, r#"🄡"#, r#"(R)"#, r#"(R)"#),
        (r#"🄢"#, r#"🄢"#, r#"🄢"#, r#"(S)"#, r#"(S)"#),
        (r#"🄣"#, r#"🄣"#, r#"🄣"#, r#"(T)"#, r#"(T)"#),
        (r#"🄤"#, r#"🄤"#, r#"🄤"#, r#"(U)"#, r#"(U)"#),
        (r#"🄥"#, r#"🄥"#, r#"🄥"#, r#"(V)"#, r#"(V)"#),
        (r#"🄦"#, r#"🄦"#, r#"🄦"#, r#"(W)"#, r#"(W)"#),
        (r#"🄧"#, r#"🄧"#, r#"🄧"#, r#"(X)"#, r#"(X)"#),
        (r#"🄨"#, r#"🄨"#, r#"🄨"#, r#"(Y)"#, r#"(Y)"#),
        (r#"🄩"#, r#"🄩"#, r#"🄩"#, r#"(Z)"#, r#"(Z)"#),
        (r#"🄪"#, r#"🄪"#, r#"🄪"#, r#"〔S〕"#, r#"〔S〕"#),
        (r#"🄫"#, r#"🄫"#, r#"🄫"#, r#"C"#, r#"C"#),
        (r#"🄬"#, r#"🄬"#, r#"🄬"#, r#"R"#, r#"R"#),
        (r#"🄭"#, r#"🄭"#, r#"🄭"#, r#"CD"#, r#"CD"#),
        (r#"🄮"#, r#"🄮"#, r#"🄮"#, r#"WZ"#, r#"WZ"#),
        (r#"🄰"#, r#"🄰"#, r#"🄰"#, r#"A"#, r#"A"#),
        (r#"🄱"#, r#"🄱"#, r#"🄱"#, r#"B"#, r#"B"#),
        (r#"🄲"#, r#"🄲"#, r#"🄲"#, r#"C"#, r#"C"#),
        (r#"🄳"#, r#"🄳"#, r#"🄳"#, r#"D"#, r#"D"#),
        (r#"🄴"#, r#"🄴"#, r#"🄴"#, r#"E"#, r#"E"#),
        (r#"🄵"#, r#"🄵"#, r#"🄵"#, r#"F"#, r#"F"#),
        (r#"🄶"#, r#"🄶"#, r#"🄶"#, r#"G"#, r#"G"#),
        (r#"🄷"#, r#"🄷"#, r#"🄷"#, r#"H"#, r#"H"#),
        (r#"🄸"#, r#"🄸"#, r#"🄸"#, r#"I"#, r#"I"#),
        (r#"🄹"#, r#"🄹"#, r#"🄹"#, r#"J"#, r#"J"#),
        (r#"🄺"#, r#"🄺"#, r#"🄺"#, r#"K"#, r#"K"#),
        (r#"🄻"#, r#"🄻"#, r#"🄻"#, r#"L"#, r#"L"#),
        (r#"🄼"#, r#"🄼"#, r#"🄼"#, r#"M"#, r#"M"#),
        (r#"🄽"#, r#"🄽"#, r#"🄽"#, r#"N"#, r#"N"#),
        (r#"🄾"#, r#"🄾"#, r#"🄾"#, r#"O"#, r#"O"#),
        (r#"🄿"#, r#"🄿"#, r#"🄿"#, r#"P"#, r#"P"#),
        (r#"🅀"#, r#"🅀"#, r#"🅀"#, r#"Q"#, r#"Q"#),
        (r#"🅁"#, r#"🅁"#, r#"🅁"#, r#"R"#, r#"R"#),
        (r#"🅂"#, r#"🅂"#, r#"🅂"#, r#"S"#, r#"S"#),
        (r#"🅃"#, r#"🅃"#, r#"🅃"#, r#"T"#, r#"T"#),
        (r#"🅄"#, r#"🅄"#, r#"🅄"#, r#"U"#, r#"U"#),
        (r#"🅅"#, r#"🅅"#, r#"🅅"#, r#"V"#, r#"V"#),
        (r#"🅆"#, r#"🅆"#, r#"🅆"#, r#"W"#, r#"W"#),
        (r#"🅇"#, r#"🅇"#, r#"🅇"#, r#"X"#, r#"X"#),
        (r#"🅈"#, r#"🅈"#, r#"🅈"#, r#"Y"#, r#"Y"#),
        (r#"🅉"#, r#"🅉"#, r#"🅉"#, r#"Z"#, r#"Z"#),
        (r#"🅊"#, r#"🅊"#, r#"🅊"#, r#"HV"#, r#"HV"#),
        (r#"🅋"#, r#"🅋"#, r#"🅋"#, r#"MV"#, r#"MV"#),
        (r#"🅌"#, r#"🅌"#, r#"🅌"#, r#"SD"#, r#"SD"#),
        (r#"🅍"#, r#"🅍"#, r#"🅍"#, r#"SS"#, r#"SS"#),
        (r#"🅎"#, r#"🅎"#, r#"🅎"#, r#"PPV"#, r#"PPV"#),
        (r#"🅏"#, r#"🅏"#, r#"🅏"#, r#"WC"#, r#"WC"#),
        (r#"🅪"#, r#"🅪"#, r#"🅪"#, r#"MC"#, r#"MC"#),
        (r#"🅫"#, r#"🅫"#, r#"🅫"#, r#"MD"#, r#"MD"#),
        (r#"🅬"#, r#"🅬"#, r#"🅬"#, r#"MR"#, r#"MR"#),
        (r#"🆐"#, r#"🆐"#, r#"🆐"#, r#"DJ"#, r#"DJ"#),
        (r#"🈀"#, r#"🈀"#, r#"🈀"#, r#"ほか"#, r#"ほか"#),
        (r#"🈁"#, r#"🈁"#, r#"🈁"#, r#"ココ"#, r#"ココ"#),
        (r#"🈂"#, r#"🈂"#, r#"🈂"#, r#"サ"#, r#"サ"#),
        (r#"🈐"#, r#"🈐"#, r#"🈐"#, r#"手"#, r#"手"#),
        (r#"🈑"#, r#"🈑"#, r#"🈑"#, r#"字"#, r#"字"#),
        (r#"🈒"#, r#"🈒"#, r#"🈒"#, r#"双"#, r#"双"#),
        (r#"🈓"#, r#"🈓"#, r#"🈓"#, r#"デ"#, r#"デ"#),
        (r#"🈔"#, r#"🈔"#, r#"🈔"#, r#"二"#, r#"二"#),
        (r#"🈕"#, r#"🈕"#, r#"🈕"#, r#"多"#, r#"多"#),
        (r#"🈖"#, r#"🈖"#, r#"🈖"#, r#"解"#, r#"解"#),
        (r#"🈗"#, r#"🈗"#, r#"🈗"#, r#"天"#, r#"天"#),
        (r#"🈘"#, r#"🈘"#, r#"🈘"#, r#"交"#, r#"交"#),
        (r#"🈙"#, r#"🈙"#, r#"🈙"#, r#"映"#, r#"映"#),
        (r#"🈚"#, r#"🈚"#, r#"🈚"#, r#"無"#, r#"無"#),
        (r#"🈛"#, r#"🈛"#, r#"🈛"#, r#"料"#, r#"料"#),
        (r#"🈜"#, r#"🈜"#, r#"🈜"#, r#"前"#, r#"前"#),
        (r#"🈝"#, r#"🈝"#, r#"🈝"#, r#"後"#, r#"後"#),
        (r#"🈞"#, r#"🈞"#, r#"🈞"#, r#"再"#, r#"再"#),
        (r#"🈟"#, r#"🈟"#, r#"🈟"#, r#"新"#, r#"新"#),
        (r#"🈠"#, r#"🈠"#, r#"🈠"#, r#"初"#, r#"初"#),
        (r#"🈡"#, r#"🈡"#, r#"🈡"#, r#"終"#, r#"終"#),
        (r#"🈢"#, r#"🈢"#, r#"🈢"#, r#"生"#, r#"生"#),
        (r#"🈣"#, r#"🈣"#, r#"🈣"#, r#"販"#, r#"販"#),
        (r#"🈤"#, r#"🈤"#, r#"🈤"#, r#"声"#, r#"声"#),
        (r#"🈥"#, r#"🈥"#, r#"🈥"#, r#"吹"#, r#"吹"#),
        (r#"🈦"#, r#"🈦"#, r#"🈦"#, r#"演"#, r#"演"#),
        (r#"🈧"#, r#"🈧"#, r#"🈧"#, r#"投"#, r#"投"#),
        (r#"🈨"#, r#"🈨"#, r#"🈨"#, r#"捕"#, r#"捕"#),
        (r#"🈩"#, r#"🈩"#, r#"🈩"#, r#"一"#, r#"一"#),
        (r#"🈪"#, r#"🈪"#, r#"🈪"#, r#"三"#, r#"三"#),
        (r#"🈫"#, r#"🈫"#, r#"🈫"#, r#"遊"#, r#"遊"#),
        (r#"🈬"#, r#"🈬"#, r#"🈬"#, r#"左"#, r#"左"#),
        (r#"🈭"#, r#"🈭"#, r#"🈭"#, r#"中"#, r#"中"#),
        (r#"🈮"#, r#"🈮"#, r#"🈮"#, r#"右"#, r#"右"#),
        (r#"🈯"#, r#"🈯"#, r#"🈯"#, r#"指"#, r#"指"#),
        (r#"🈰"#, r#"🈰"#, r#"🈰"#, r#"走"#, r#"走"#),
        (r#"🈱"#, r#"🈱"#, r#"🈱"#, r#"打"#, r#"打"#),
        (r#"🈲"#, r#"🈲"#, r#"🈲"#, r#"禁"#, r#"禁"#),
        (r#"🈳"#, r#"🈳"#, r#"🈳"#, r#"空"#, r#"空"#),
        (r#"🈴"#, r#"🈴"#, r#"🈴"#, r#"合"#, r#"合"#),
        (r#"🈵"#, r#"🈵"#, r#"🈵"#, r#"満"#, r#"満"#),
        (r#"🈶"#, r#"🈶"#, r#"🈶"#, r#"有"#, r#"有"#),
        (r#"🈷"#, r#"🈷"#, r#"🈷"#, r#"月"#, r#"月"#),
        (r#"🈸"#, r#"🈸"#, r#"🈸"#, r#"申"#, r#"申"#),
        (r#"🈹"#, r#"🈹"#, r#"🈹"#, r#"割"#, r#"割"#),
        (r#"🈺"#, r#"🈺"#, r#"🈺"#, r#"営"#, r#"営"#),
        (r#"🈻"#, r#"🈻"#, r#"🈻"#, r#"配"#, r#"配"#),
        (r#"🉀"#, r#"🉀"#, r#"🉀"#, r#"〔本〕"#, r#"〔本〕"#),
        (r#"🉁"#, r#"🉁"#, r#"🉁"#, r#"〔三〕"#, r#"〔三〕"#),
        (r#"🉂"#, r#"🉂"#, r#"🉂"#, r#"〔二〕"#, r#"〔二〕"#),
        (r#"🉃"#, r#"🉃"#, r#"🉃"#, r#"〔安〕"#, r#"〔安〕"#),
        (r#"🉄"#, r#"🉄"#, r#"🉄"#, r#"〔点〕"#, r#"〔点〕"#),
        (r#"🉅"#, r#"🉅"#, r#"🉅"#, r#"〔打〕"#, r#"〔打〕"#),
        (r#"🉆"#, r#"🉆"#, r#"🉆"#, r#"〔盗〕"#, r#"〔盗〕"#),
        (r#"🉇"#, r#"🉇"#, r#"🉇"#, r#"〔勝〕"#, r#"〔勝〕"#),
        (r#"🉈"#, r#"🉈"#, r#"🉈"#, r#"〔敗〕"#, r#"〔敗〕"#),
        (r#"🉐"#, r#"🉐"#, r#"🉐"#, r#"得"#, r#"得"#),
        (r#"🉑"#, r#"🉑"#, r#"🉑"#, r#"可"#, r#"可"#),
        (r#"🯰"#, r#"🯰"#, r#"🯰"#, r#"0"#, r#"0"#),
        (r#"🯱"#, r#"🯱"#, r#"🯱"#, r#"1"#, r#"1"#),
        (r#"🯲"#, r#"🯲"#, r#"🯲"#, r#"2"#, r#"2"#),
        (r#"🯳"#, r#"🯳"#, r#"🯳"#, r#"3"#, r#"3"#),
        (r#"🯴"#, r#"🯴"#, r#"🯴"#, r#"4"#, r#"4"#),
        (r#"🯵"#, r#"🯵"#, r#"🯵"#, r#"5"#, r#"5"#),
        (r#"🯶"#, r#"🯶"#, r#"🯶"#, r#"6"#, r#"6"#),
        (r#"🯷"#, r#"🯷"#, r#"🯷"#, r#"7"#, r#"7"#),
        (r#"🯸"#, r#"🯸"#, r#"🯸"#, r#"8"#, r#"8"#),
        (r#"🯹"#, r#"🯹"#, r#"🯹"#, r#"9"#, r#"9"#),
        (r#"丽"#, r#"丽"#, r#"丽"#, r#"丽"#, r#"丽"#),
        (r#"丸"#, r#"丸"#, r#"丸"#, r#"丸"#, r#"丸"#),
        (r#"乁"#, r#"乁"#, r#"乁"#, r#"乁"#, r#"乁"#),
        (r#"𠄢"#, r#"𠄢"#, r#"𠄢"#, r#"𠄢"#, r#"𠄢"#),
        (r#"你"#, r#"你"#, r#"你"#, r#"你"#, r#"你"#),
        (r#"侮"#, r#"侮"#, r#"侮"#, r#"侮"#, r#"侮"#),
        (r#"侻"#, r#"侻"#, r#"侻"#, r#"侻"#, r#"侻"#),
        (r#"倂"#, r#"倂"#, r#"倂"#, r#"倂"#, r#"倂"#),
        (r#"偺"#, r#"偺"#, r#"偺"#, r#"偺"#, r#"偺"#),
        (r#"備"#, r#"備"#, r#"備"#, r#"備"#, r#"備"#),
        (r#"僧"#, r#"僧"#, r#"僧"#, r#"僧"#, r#"僧"#),
        (r#"像"#, r#"像"#, r#"像"#, r#"像"#, r#"像"#),
        (r#"㒞"#, r#"㒞"#, r#"㒞"#, r#"㒞"#, r#"㒞"#),
        (r#"𠘺"#, r#"𠘺"#, r#"𠘺"#, r#"𠘺"#, r#"𠘺"#),
        (r#"免"#, r#"免"#, r#"免"#, r#"免"#, r#"免"#),
        (r#"兔"#, r#"兔"#, r#"兔"#, r#"兔"#, r#"兔"#),
        (r#"兤"#, r#"兤"#, r#"兤"#, r#"兤"#, r#"兤"#),
        (r#"具"#, r#"具"#, r#"具"#, r#"具"#, r#"具"#),
        (r#"𠔜"#, r#"𠔜"#, r#"𠔜"#, r#"𠔜"#, r#"𠔜"#),
        (r#"㒹"#, r#"㒹"#, r#"㒹"#, r#"㒹"#, r#"㒹"#),
        (r#"內"#, r#"內"#, r#"內"#, r#"內"#, r#"內"#),
        (r#"再"#, r#"再"#, r#"再"#, r#"再"#, r#"再"#),
        (r#"𠕋"#, r#"𠕋"#, r#"𠕋"#, r#"𠕋"#, r#"𠕋"#),
        (r#"冗"#, r#"冗"#, r#"冗"#, r#"冗"#, r#"冗"#),
        (r#"冤"#, r#"冤"#, r#"冤"#, r#"冤"#, r#"冤"#),
        (r#"仌"#, r#"仌"#, r#"仌"#, r#"仌"#, r#"仌"#),
        (r#"冬"#, r#"冬"#, r#"冬"#, r#"冬"#, r#"冬"#),
        (r#"况"#, r#"况"#, r#"况"#, r#"况"#, r#"况"#),
        (r#"𩇟"#, r#"𩇟"#, r#"𩇟"#, r#"𩇟"#, r#"𩇟"#),
        (r#"凵"#, r#"凵"#, r#"凵"#, r#"凵"#, r#"凵"#),
        (r#"刃"#, r#"刃"#, r#"刃"#, r#"刃"#, r#"刃"#),
        (r#"㓟"#, r#"㓟"#, r#"㓟"#, r#"㓟"#, r#"㓟"#),
        (r#"刻"#, r#"刻"#, r#"刻"#, r#"刻"#, r#"刻"#),
        (r#"剆"#, r#"剆"#, r#"剆"#, r#"剆"#, r#"剆"#),
        (r#"割"#, r#"割"#, r#"割"#, r#"割"#, r#"割"#),
        (r#"剷"#, r#"剷"#, r#"剷"#, r#"剷"#, r#"剷"#),
        (r#"㔕"#, r#"㔕"#, r#"㔕"#, r#"㔕"#, r#"㔕"#),
        (r#"勇"#, r#"勇"#, r#"勇"#, r#"勇"#, r#"勇"#),
        (r#"勉"#, r#"勉"#, r#"勉"#, r#"勉"#, r#"勉"#),
        (r#"勤"#, r#"勤"#, r#"勤"#, r#"勤"#, r#"勤"#),
        (r#"勺"#, r#"勺"#, r#"勺"#, r#"勺"#, r#"勺"#),
        (r#"包"#, r#"包"#, r#"包"#, r#"包"#, r#"包"#),
        (r#"匆"#, r#"匆"#, r#"匆"#, r#"匆"#, r#"匆"#),
        (r#"北"#, r#"北"#, r#"北"#, r#"北"#, r#"北"#),
        (r#"卉"#, r#"卉"#, r#"卉"#, r#"卉"#, r#"卉"#),
        (r#"卑"#, r#"卑"#, r#"卑"#, r#"卑"#, r#"卑"#),
        (r#"博"#, r#"博"#, r#"博"#, r#"博"#, r#"博"#),
        (r#"即"#, r#"即"#, r#"即"#, r#"即"#, r#"即"#),
        (r#"卽"#, r#"卽"#, r#"卽"#, r#"卽"#, r#"卽"#),
        (r#"卿"#, r#"卿"#, r#"卿"#, r#"卿"#, r#"卿"#),
        (r#"卿"#, r#"卿"#, r#"卿"#, r#"卿"#, r#"卿"#),
        (r#"卿"#, r#"卿"#, r#"卿"#, r#"卿"#, r#"卿"#),
        (r#"𠨬"#, r#"𠨬"#, r#"𠨬"#, r#"𠨬"#, r#"𠨬"#),
        (r#"灰"#, r#"灰"#, r#"灰"#, r#"灰"#, r#"灰"#),
        (r#"及"#, r#"及"#, r#"及"#, r#"及"#, r#"及"#),
        (r#"叟"#, r#"叟"#, r#"叟"#, r#"叟"#, r#"叟"#),
        (r#"𠭣"#, r#"𠭣"#, r#"𠭣"#, r#"𠭣"#, r#"𠭣"#),
        (r#"叫"#, r#"叫"#, r#"叫"#, r#"叫"#, r#"叫"#),
        (r#"叱"#, r#"叱"#, r#"叱"#, r#"叱"#, r#"叱"#),
        (r#"吆"#, r#"吆"#, r#"吆"#, r#"吆"#, r#"吆"#),
        (r#"咞"#, r#"咞"#, r#"咞"#, r#"咞"#, r#"咞"#),
        (r#"吸"#, r#"吸"#, r#"吸"#, r#"吸"#, r#"吸"#),
        (r#"呈"#, r#"呈"#, r#"呈"#, r#"呈"#