// ------------------------------------------------------------------------
//
// SPDX-License-Identifier: LGPL-2.1-or-later
// Copyright (C) 2015 - 2025 by the deal.II authors
//
// This file is part of the deal.II library.
//
// Part of the source code is dual licensed under Apache-2.0 WITH
// LLVM-exception OR LGPL-2.1-or-later. Detailed license information
// governing the source code and code contributions can be found in
// LICENSE.md and CONTRIBUTING.md at the top level directory of deal.II.
//
// ------------------------------------------------------------------------

#ifndef dealii_named_selection_h
#define dealii_named_selection_h

#include <deal.II/base/config.h>

#include <deal.II/algorithms/any_data.h>

#include <deal.II/base/exceptions.h>

#include <string>

DEAL_II_NAMESPACE_OPEN

/**
 * Select data from AnyData corresponding to the attached name.
 *
 * Given a list of names to search for (provided by add()), objects of this
 * class provide an index list of the selected data.
 */
class NamedSelection
{
public:
  /**
   * Add a new name to be searched for in @p data supplied in initialize().
   *
   * @note Names will be added to the end of the current list.
   */
  void
  add(const std::string &name);


  /**
   * Create the index vector pointing into the AnyData object.
   */
  void
  initialize(const AnyData &data);


  /**
   * The number of names in this object. This function may be used whether
   * initialize() was called before or not.
   */
  unsigned int
  size() const;


  /**
   * Return the corresponding index in the AnyData object supplied to the last
   * initialize(). It is an error if initialize() has not been called before.
   *
   * Indices are in the same order as the calls to add().
   */
  unsigned int
  operator()(unsigned int i) const;


private:
  /**
   * The selected names.
   */
  std::vector<std::string> names;

  /**
   * The index map generated by initialize() and accessed by operator().
   */
  std::vector<unsigned int> indices;
};


inline unsigned int
NamedSelection::size() const
{
  return names.size();
}


inline void
NamedSelection::add(const std::string &s)
{
  names.push_back(s);
}


inline unsigned int
NamedSelection::operator()(unsigned int i) const
{
  Assert(indices.size() == names.size(), ExcNotInitialized());

  AssertIndexRange(i, size());

  return indices[i];
}

DEAL_II_NAMESPACE_CLOSE

#endif
