#
#  Copyright (c) 2018 Anderson Toshiyuki Sasaki <ansasaki@redhat.com>
#
#  Redistribution and use is allowed according to the terms of the New
#  BSD license.
#  For details see the accompanying COPYING-CMAKE-SCRIPTS file.
#

#.rst:
# GenerateMap
# -----------
#
# This is a helper script for FindABImap.cmake.
#
# Generates a symbols version script using the abimap tool.
# This script is run in build time to use the correct command depending on the
# existence of the file provided ``CURRENT_MAP``.
#
# If the file exists, the ``abimap update`` subcommand is used to update the
# existing map. Otherwise, the ``abimap new`` subcommand is used to create a new
# map file.
#
# If the file provided in ``CURRENT_MAP`` exists, it is copied to the
# ``OUTPUT_PATH`` before updating.
# This is required because ``abimap`` do not generate output if no symbols were
# changed when updating an existing file.
#
# Expected defined variables
# --------------------------
#
# ``SYMBOLS``:
#   Required file containing the symbols to be used as input. Usually this is
#   the ``OUTPUT`` generated by ``extract_symbols()`` function provided in
#   FindABImap.cmake
#
# ``RELEASE_NAME_VERSION``:
#   Required, expects the library name and version information to be added to
#   the symbols in the format ``library_name_1_2_3``
#
# ``CURRENT_MAP``:
#   Required, expects the path to the current map file (or the path were it
#   should be)
#
# ``OUTPUT_PATH``:
#   Required, expects the output file path.
#
# ``ABIMAP_EXECUTABLE``:
#   Required, expects the path to the ``abimap`` tool.
#
# Optionally defined variables
# ----------------------------
#
# ``FINAL``:
#   If defined, will mark the modified set of symbols in the symbol version
#   script as final, preventing later changes using ``abimap``.
#
# ``BREAK_ABI``:
#   If defined, the build will not fail if symbols were removed.
#   If defined and a symbol is removed, a new release is created containing
#   all symbols from all released versions. This makes an incompatible release.
#

if (NOT DEFINED RELEASE_NAME_VERSION)
    message(SEND_ERROR "RELEASE_NAME_VERSION not defined")
endif()

if (NOT DEFINED SYMBOLS)
    message(SEND_ERROR "SYMBOLS not defined")
endif()

if (NOT DEFINED CURRENT_MAP)
    message(SEND_ERROR "CURRENT_MAP not defined")
endif()

if (NOT DEFINED OUTPUT_PATH)
    message(SEND_ERROR "OUTPUT_PATH not defined")
endif()

if (NOT ABIMAP_EXECUTABLE)
    message(SEND_ERROR "ABIMAP_EXECUTABLE not defined")
endif()

set(ARGS_LIST)

if (FINAL)
    list(APPEND ARGS_LIST "--final")
endif()

if (EXISTS ${CURRENT_MAP})
    if (BREAK_ABI)
        list(APPEND ARGS_LIST "--allow-abi-break")
    endif()

    execute_process(
      COMMAND
        ${CMAKE_COMMAND} -E copy_if_different ${CURRENT_MAP} ${OUTPUT_PATH}
      COMMAND
        ${ABIMAP_EXECUTABLE} update ${ARGS_LIST}
        -r ${RELEASE_NAME_VERSION}
        -i ${SYMBOLS}
        -o ${OUTPUT_PATH}
        ${CURRENT_MAP}
      RESULT_VARIABLE result
    )
else ()
    execute_process(
      COMMAND
        ${ABIMAP_EXECUTABLE} new ${ARGS_LIST}
        -r ${RELEASE_NAME_VERSION}
        -i ${SYMBOLS}
        -o ${OUTPUT_PATH}
      RESULT_VARIABLE result
    )
endif()

if (NOT "${result}" STREQUAL "0")
    message(SEND_ERROR "Map generation failed")
endif()
