/***************************************************************************
 *   Copyright (C) 2005-2007 by Gael de Chalendar (aka Kleag)  *
 *   kleag@free.fr   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
   02110-1301, USA
 ***************************************************************************/

#include "dot2qtconsts.h"

// lib
#include "dotgrammar.h"
#include "kgraphviewerlib_debug.h"

#include <string>

// Helper class for storing raw data in static tables which can be used for QString instance
// creation at runtime without copying/converting to new memalloc'ed memory, as well as avoiding
// that way storing the strings directly as QStrings resulting in non-constexpr init code on
// library loading
struct RawStringData {
    template<std::size_t StringSize>
    constexpr inline RawStringData(const char16_t (&_data)[StringSize])
        : data(_data)
        , size(std::char_traits<char16_t>::length(_data))
    {
    }
    constexpr inline RawStringData(std::nullptr_t)
    {
    }
    constexpr inline RawStringData() = default;

    inline QString toString() const
    {
        if (!data) {
            return QString();
        }

        return Qt::Literals::StringLiterals::operator""_s(data, size);
    }
    inline operator bool() const
    {
        return data != nullptr;
    }

private:
    const char16_t *const data = nullptr;
    const std::size_t size = 0;
};

const Dot2QtConsts Dot2QtConsts::m_componentData;

static const struct {
    const RawStringData input;
    const RawStringData roman;
    const RawStringData italic;
    const RawStringData bold;
    const RawStringData boldItalic;
    const RawStringData light;
    const RawStringData lightItalic;
} postscriptFontNames[] = {{u"arial", u"Arial", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {u"avantgarde", u"AvantGarde-Book", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {u"charter", u"CharterBT-Roman", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {u"garamond", u"Garamond-Regular", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {u"gillsans", u"GillSans", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {u"helvetica", u"Helvetica", u"Helvetica-Oblique", u"Helvetica-Bold", u"Helvetica-BoldOblique", u"Helvetica-Outline", u"Helvetica-OutlineOblique"},
                           {u"palatino", u"Palatino", u"Palatino-Italic", u"Palatino-Bold", u"Palatino-BoldItalic", u"Palatino", u"Palatino-Italic"},
                           {u"times", u"Times-Roman", u"Times-Italic", u"Times-Bold", u"Times-BoldItalic", u"Times-Outline", u"Times-Italic"},
                           {u"new century schoolbook", u"NewCenturySchlbk-Roman", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {u"symbol", u"Symbol", u"Symbol", u"Symbol", u"Symbol", u"Symbol", u"Symbol"},
                           {u"terminal", u"Courier", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {u"times new roman", u"TimesNewRoman", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {u"utopia", u"Utopia-Regular", nullptr, nullptr, nullptr, nullptr, nullptr},
                           {nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr}};

static const struct {
    const RawStringData name;
    unsigned int r;
    unsigned int g;
    unsigned int b;
} color_lib[] = {{u"snow", 255, 250, 250},
                 {u"ghost white", 248, 248, 255},
                 {u"ghostwhite", 248, 248, 255},
                 {u"white smoke", 245, 245, 245},
                 {u"whitesmoke", 245, 245, 245},
                 {u"gainsboro", 220, 220, 220},
                 {u"floral white", 255, 250, 240},
                 {u"floralwhite", 255, 250, 240},
                 {u"old lace", 253, 245, 230},
                 {u"oldlace", 253, 245, 230},
                 {u"linen", 250, 240, 230},
                 {u"antique white", 250, 235, 215},
                 {u"antiquewhite", 250, 235, 215},
                 {u"papaya whip", 255, 239, 213},
                 {u"papayawhip", 255, 239, 213},
                 {u"blanched almond", 255, 235, 205},
                 {u"blanchedalmond", 255, 235, 205},
                 {u"bisque", 255, 228, 196},
                 {u"peach puff", 255, 218, 185},
                 {u"peachpuff", 255, 218, 185},
                 {u"navajo white", 255, 222, 173},
                 {u"navajowhite", 255, 222, 173},
                 {u"moccasin", 255, 228, 181},
                 {u"cornsilk", 255, 248, 220},
                 {u"ivory", 255, 255, 240},
                 {u"lemon chiffon", 255, 250, 205},
                 {u"lemonchiffon", 255, 250, 205},
                 {u"seashell", 255, 245, 238},
                 {u"honeydew", 240, 255, 240},
                 {u"mint cream", 245, 255, 250},
                 {u"mintcream", 245, 255, 250},
                 {u"azure", 240, 255, 255},
                 {u"alice blue", 240, 248, 255},
                 {u"aliceblue", 240, 248, 255},
                 {u"lavender", 230, 230, 250},
                 {u"lavender blush", 255, 240, 245},
                 {u"lavenderblush", 255, 240, 245},
                 {u"misty rose", 255, 228, 225},
                 {u"mistyrose", 255, 228, 225},
                 {u"white", 255, 255, 255},
                 {u"black", 0, 0, 0},
                 {u"dark slate gray", 47, 79, 79},
                 {u"darkslategray", 47, 79, 79},
                 {u"dark slate grey", 47, 79, 79},
                 {u"darkslategrey", 47, 79, 79},
                 {u"dim gray", 105, 105, 105},
                 {u"dimgray", 105, 105, 105},
                 {u"dim grey", 105, 105, 105},
                 {u"dimgrey", 105, 105, 105},
                 {u"slate gray", 112, 128, 144},
                 {u"slategray", 112, 128, 144},
                 {u"slate grey", 112, 128, 144},
                 {u"slategrey", 112, 128, 144},
                 {u"light slate gray", 119, 136, 153},
                 {u"lightslategray", 119, 136, 153},
                 {u"light slate grey", 119, 136, 153},
                 {u"lightslategrey", 119, 136, 153},
                 {u"gray", 190, 190, 190},
                 {u"grey", 190, 190, 190},
                 {u"light grey", 211, 211, 211},
                 {u"lightgrey", 211, 211, 211},
                 {u"light gray", 211, 211, 211},
                 {u"lightgray", 211, 211, 211},
                 {u"midnight blue", 25, 25, 112},
                 {u"midnightblue", 25, 25, 112},
                 {u"navy", 0, 0, 128},
                 {u"navy blue", 0, 0, 128},
                 {u"navyblue", 0, 0, 128},
                 {u"cornflower blue", 100, 149, 237},
                 {u"cornflowerblue", 100, 149, 237},
                 {u"dark slate blue", 72, 61, 139},
                 {u"darkslateblue", 72, 61, 139},
                 {u"slate blue", 106, 90, 205},
                 {u"slateblue", 106, 90, 205},
                 {u"medium slate blue", 123, 104, 238},
                 {u"mediumslateblue", 123, 104, 238},
                 {u"light slate blue", 132, 112, 255},
                 {u"lightslateblue", 132, 112, 255},
                 {u"medium blue", 0, 0, 205},
                 {u"mediumblue", 0, 0, 205},
                 {u"royal blue", 65, 105, 225},
                 {u"royalblue", 65, 105, 225},
                 {u"blue", 0, 0, 255},
                 {u"dodger blue", 30, 144, 255},
                 {u"dodgerblue", 30, 144, 255},
                 {u"deep sky blue", 0, 191, 255},
                 {u"deepskyblue", 0, 191, 255},
                 {u"sky blue", 135, 206, 235},
                 {u"skyblue", 135, 206, 235},
                 {u"light sky blue", 135, 206, 250},
                 {u"lightskyblue", 135, 206, 250},
                 {u"steel blue", 70, 130, 180},
                 {u"steelblue", 70, 130, 180},
                 {u"light steel blue", 176, 196, 222},
                 {u"lightsteelblue", 176, 196, 222},
                 {u"light blue", 173, 216, 230},
                 {u"lightblue", 173, 216, 230},
                 {u"powder blue", 176, 224, 230},
                 {u"powderblue", 176, 224, 230},
                 {u"pale turquoise", 175, 238, 238},
                 {u"paleturquoise", 175, 238, 238},
                 {u"dark turquoise", 0, 206, 209},
                 {u"darkturquoise", 0, 206, 209},
                 {u"medium turquoise", 72, 209, 204},
                 {u"mediumturquoise", 72, 209, 204},
                 {u"turquoise", 64, 224, 208},
                 {u"cyan", 0, 255, 255},
                 {u"light cyan", 224, 255, 255},
                 {u"lightcyan", 224, 255, 255},
                 {u"cadet blue", 95, 158, 160},
                 {u"cadetblue", 95, 158, 160},
                 {u"medium aquamarine", 102, 205, 170},
                 {u"mediumaquamarine", 102, 205, 170},
                 {u"aquamarine", 127, 255, 212},
                 {u"dark green", 0, 100, 0},
                 {u"darkgreen", 0, 100, 0},
                 {u"dark olive green", 85, 107, 47},
                 {u"darkolivegreen", 85, 107, 47},
                 {u"dark sea green", 143, 188, 143},
                 {u"darkseagreen", 143, 188, 143},
                 {u"sea green", 46, 139, 87},
                 {u"seagreen", 46, 139, 87},
                 {u"medium sea green", 60, 179, 113},
                 {u"mediumseagreen", 60, 179, 113},
                 {u"light sea green", 32, 178, 170},
                 {u"lightseagreen", 32, 178, 170},
                 {u"pale green", 152, 251, 152},
                 {u"palegreen", 152, 251, 152},
                 {u"spring green", 0, 255, 127},
                 {u"springgreen", 0, 255, 127},
                 {u"lawn green", 124, 252, 0},
                 {u"lawngreen", 124, 252, 0},
                 {u"green", 0, 255, 0},
                 {u"chartreuse", 127, 255, 0},
                 {u"medium spring green", 0, 250, 154},
                 {u"mediumspringgreen", 0, 250, 154},
                 {u"green yellow", 173, 255, 47},
                 {u"greenyellow", 173, 255, 47},
                 {u"lime green", 50, 205, 50},
                 {u"limegreen", 50, 205, 50},
                 {u"yellow green", 154, 205, 50},
                 {u"yellowgreen", 154, 205, 50},
                 {u"forest green", 34, 139, 34},
                 {u"forestgreen", 34, 139, 34},
                 {u"olive drab", 107, 142, 35},
                 {u"olivedrab", 107, 142, 35},
                 {u"dark khaki", 189, 183, 107},
                 {u"darkkhaki", 189, 183, 107},
                 {u"khaki", 240, 230, 140},
                 {u"pale goldenrod", 238, 232, 170},
                 {u"palegoldenrod", 238, 232, 170},
                 {u"light goldenrod yellow", 250, 250, 210},
                 {u"lightgoldenrodyellow", 250, 250, 210},
                 {u"light yellow", 255, 255, 224},
                 {u"lightyellow", 255, 255, 224},
                 {u"yellow", 255, 255, 0},
                 {u"gold", 255, 215, 0},
                 {u"light goldenrod", 238, 221, 130},
                 {u"lightgoldenrod", 238, 221, 130},
                 {u"goldenrod", 218, 165, 32},
                 {u"dark goldenrod", 184, 134, 11},
                 {u"darkgoldenrod", 184, 134, 11},
                 {u"rosy brown", 188, 143, 143},
                 {u"rosybrown", 188, 143, 143},
                 {u"indian red", 205, 92, 92},
                 {u"indianred", 205, 92, 92},
                 {u"saddle brown", 139, 69, 19},
                 {u"saddlebrown", 139, 69, 19},
                 {u"sienna", 160, 82, 45},
                 {u"peru", 205, 133, 63},
                 {u"burlywood", 222, 184, 135},
                 {u"beige", 245, 245, 220},
                 {u"wheat", 245, 222, 179},
                 {u"sandy brown", 244, 164, 96},
                 {u"sandybrown", 244, 164, 96},
                 {u"tan", 210, 180, 140},
                 {u"chocolate", 210, 105, 30},
                 {u"firebrick", 178, 34, 34},
                 {u"brown", 165, 42, 42},
                 {u"dark salmon", 233, 150, 122},
                 {u"darksalmon", 233, 150, 122},
                 {u"salmon", 250, 128, 114},
                 {u"light salmon", 255, 160, 122},
                 {u"lightsalmon", 255, 160, 122},
                 {u"orange", 255, 165, 0},
                 {u"dark orange", 255, 140, 0},
                 {u"darkorange", 255, 140, 0},
                 {u"coral", 255, 127, 80},
                 {u"light coral", 240, 128, 128},
                 {u"lightcoral", 240, 128, 128},
                 {u"tomato", 255, 99, 71},
                 {u"orange red", 255, 69, 0},
                 {u"orangered", 255, 69, 0},
                 {u"red", 255, 0, 0},
                 {u"hot pink", 255, 105, 180},
                 {u"hotpink", 255, 105, 180},
                 {u"deep pink", 255, 20, 147},
                 {u"deeppink", 255, 20, 147},
                 {u"pink", 255, 192, 203},
                 {u"light pink", 255, 182, 193},
                 {u"lightpink", 255, 182, 193},
                 {u"pale violet red", 219, 112, 147},
                 {u"palevioletred", 219, 112, 147},
                 {u"maroon", 176, 48, 96},
                 {u"medium violet red", 199, 21, 133},
                 {u"mediumvioletred", 199, 21, 133},
                 {u"violet red", 208, 32, 144},
                 {u"violetred", 208, 32, 144},
                 {u"magenta", 255, 0, 255},
                 {u"violet", 238, 130, 238},
                 {u"plum", 221, 160, 221},
                 {u"orchid", 218, 112, 214},
                 {u"medium orchid", 186, 85, 211},
                 {u"mediumorchid", 186, 85, 211},
                 {u"dark orchid", 153, 50, 204},
                 {u"darkorchid", 153, 50, 204},
                 {u"dark violet", 148, 0, 211},
                 {u"darkviolet", 148, 0, 211},
                 {u"blue violet", 138, 43, 226},
                 {u"blueviolet", 138, 43, 226},
                 {u"purple", 160, 32, 240},
                 {u"medium purple", 147, 112, 219},
                 {u"mediumpurple", 147, 112, 219},
                 {u"thistle", 216, 191, 216},
                 {u"snow1", 255, 250, 250},
                 {u"snow2", 238, 233, 233},
                 {u"snow3", 205, 201, 201},
                 {u"snow4", 139, 137, 137},
                 {u"seashell1", 255, 245, 238},
                 {u"seashell2", 238, 229, 222},
                 {u"seashell3", 205, 197, 191},
                 {u"seashell4", 139, 134, 130},
                 {u"antiquewhite1", 255, 239, 219},
                 {u"antiquewhite2", 238, 223, 204},
                 {u"antiquewhite3", 205, 192, 176},
                 {u"antiquewhite4", 139, 131, 120},
                 {u"bisque1", 255, 228, 196},
                 {u"bisque2", 238, 213, 183},
                 {u"bisque3", 205, 183, 158},
                 {u"bisque4", 139, 125, 107},
                 {u"peachpuff1", 255, 218, 185},
                 {u"peachpuff2", 238, 203, 173},
                 {u"peachpuff3", 205, 175, 149},
                 {u"peachpuff4", 139, 119, 101},
                 {u"navajowhite1", 255, 222, 173},
                 {u"navajowhite2", 238, 207, 161},
                 {u"navajowhite3", 205, 179, 139},
                 {u"navajowhite4", 139, 121, 94},
                 {u"lemonchiffon1", 255, 250, 205},
                 {u"lemonchiffon2", 238, 233, 191},
                 {u"lemonchiffon3", 205, 201, 165},
                 {u"lemonchiffon4", 139, 137, 112},
                 {u"cornsilk1", 255, 248, 220},
                 {u"cornsilk2", 238, 232, 205},
                 {u"cornsilk3", 205, 200, 177},
                 {u"cornsilk4", 139, 136, 120},
                 {u"ivory1", 255, 255, 240},
                 {u"ivory2", 238, 238, 224},
                 {u"ivory3", 205, 205, 193},
                 {u"ivory4", 139, 139, 131},
                 {u"honeydew1", 240, 255, 240},
                 {u"honeydew2", 224, 238, 224},
                 {u"honeydew3", 193, 205, 193},
                 {u"honeydew4", 131, 139, 131},
                 {u"lavenderblush1", 255, 240, 245},
                 {u"lavenderblush2", 238, 224, 229},
                 {u"lavenderblush3", 205, 193, 197},
                 {u"lavenderblush4", 139, 131, 134},
                 {u"mistyrose1", 255, 228, 225},
                 {u"mistyrose2", 238, 213, 210},
                 {u"mistyrose3", 205, 183, 181},
                 {u"mistyrose4", 139, 125, 123},
                 {u"azure1", 240, 255, 255},
                 {u"azure2", 224, 238, 238},
                 {u"azure3", 193, 205, 205},
                 {u"azure4", 131, 139, 139},
                 {u"slateblue1", 131, 111, 255},
                 {u"slateblue2", 122, 103, 238},
                 {u"slateblue3", 105, 89, 205},
                 {u"slateblue4", 71, 60, 139},
                 {u"royalblue1", 72, 118, 255},
                 {u"royalblue2", 67, 110, 238},
                 {u"royalblue3", 58, 95, 205},
                 {u"royalblue4", 39, 64, 139},
                 {u"blue1", 0, 0, 255},
                 {u"blue2", 0, 0, 238},
                 {u"blue3", 0, 0, 205},
                 {u"blue4", 0, 0, 139},
                 {u"dodgerblue1", 30, 144, 255},
                 {u"dodgerblue2", 28, 134, 238},
                 {u"dodgerblue3", 24, 116, 205},
                 {u"dodgerblue4", 16, 78, 139},
                 {u"steelblue1", 99, 184, 255},
                 {u"steelblue2", 92, 172, 238},
                 {u"steelblue3", 79, 148, 205},
                 {u"steelblue4", 54, 100, 139},
                 {u"deepskyblue1", 0, 191, 255},
                 {u"deepskyblue2", 0, 178, 238},
                 {u"deepskyblue3", 0, 154, 205},
                 {u"deepskyblue4", 0, 104, 139},
                 {u"skyblue1", 135, 206, 255},
                 {u"skyblue2", 126, 192, 238},
                 {u"skyblue3", 108, 166, 205},
                 {u"skyblue4", 74, 112, 139},
                 {u"lightskyblue1", 176, 226, 255},
                 {u"lightskyblue2", 164, 211, 238},
                 {u"lightskyblue3", 141, 182, 205},
                 {u"lightskyblue4", 96, 123, 139},
                 {u"slategray1", 198, 226, 255},
                 {u"slategray2", 185, 211, 238},
                 {u"slategray3", 159, 182, 205},
                 {u"slategray4", 108, 123, 139},
                 {u"lightsteelblue1", 202, 225, 255},
                 {u"lightsteelblue2", 188, 210, 238},
                 {u"lightsteelblue3", 162, 181, 205},
                 {u"lightsteelblue4", 110, 123, 139},
                 {u"lightblue1", 191, 239, 255},
                 {u"lightblue2", 178, 223, 238},
                 {u"lightblue3", 154, 192, 205},
                 {u"lightblue4", 104, 131, 139},
                 {u"lightcyan1", 224, 255, 255},
                 {u"lightcyan2", 209, 238, 238},
                 {u"lightcyan3", 180, 205, 205},
                 {u"lightcyan4", 122, 139, 139},
                 {u"paleturquoise1", 187, 255, 255},
                 {u"paleturquoise2", 174, 238, 238},
                 {u"paleturquoise3", 150, 205, 205},
                 {u"paleturquoise4", 102, 139, 139},
                 {u"cadetblue1", 152, 245, 255},
                 {u"cadetblue2", 142, 229, 238},
                 {u"cadetblue3", 122, 197, 205},
                 {u"cadetblue4", 83, 134, 139},
                 {u"turquoise1", 0, 245, 255},
                 {u"turquoise2", 0, 229, 238},
                 {u"turquoise3", 0, 197, 205},
                 {u"turquoise4", 0, 134, 139},
                 {u"cyan1", 0, 255, 255},
                 {u"cyan2", 0, 238, 238},
                 {u"cyan3", 0, 205, 205},
                 {u"cyan4", 0, 139, 139},
                 {u"darkslategray1", 151, 255, 255},
                 {u"darkslategray2", 141, 238, 238},
                 {u"darkslategray3", 121, 205, 205},
                 {u"darkslategray4", 82, 139, 139},
                 {u"aquamarine1", 127, 255, 212},
                 {u"aquamarine2", 118, 238, 198},
                 {u"aquamarine3", 102, 205, 170},
                 {u"aquamarine4", 69, 139, 116},
                 {u"darkseagreen1", 193, 255, 193},
                 {u"darkseagreen2", 180, 238, 180},
                 {u"darkseagreen3", 155, 205, 155},
                 {u"darkseagreen4", 105, 139, 105},
                 {u"seagreen1", 84, 255, 159},
                 {u"seagreen2", 78, 238, 148},
                 {u"seagreen3", 67, 205, 128},
                 {u"seagreen4", 46, 139, 87},
                 {u"palegreen1", 154, 255, 154},
                 {u"palegreen2", 144, 238, 144},
                 {u"palegreen3", 124, 205, 124},
                 {u"palegreen4", 84, 139, 84},
                 {u"springgreen1", 0, 255, 127},
                 {u"springgreen2", 0, 238, 118},
                 {u"springgreen3", 0, 205, 102},
                 {u"springgreen4", 0, 139, 69},
                 {u"green1", 0, 255, 0},
                 {u"green2", 0, 238, 0},
                 {u"green3", 0, 205, 0},
                 {u"green4", 0, 139, 0},
                 {u"chartreuse1", 127, 255, 0},
                 {u"chartreuse2", 118, 238, 0},
                 {u"chartreuse3", 102, 205, 0},
                 {u"chartreuse4", 69, 139, 0},
                 {u"olivedrab1", 192, 255, 62},
                 {u"olivedrab2", 179, 238, 58},
                 {u"olivedrab3", 154, 205, 50},
                 {u"olivedrab4", 105, 139, 34},
                 {u"darkolivegreen1", 202, 255, 112},
                 {u"darkolivegreen2", 188, 238, 104},
                 {u"darkolivegreen3", 162, 205, 90},
                 {u"darkolivegreen4", 110, 139, 61},
                 {u"khaki1", 255, 246, 143},
                 {u"khaki2", 238, 230, 133},
                 {u"khaki3", 205, 198, 115},
                 {u"khaki4", 139, 134, 78},
                 {u"lightgoldenrod1", 255, 236, 139},
                 {u"lightgoldenrod2", 238, 220, 130},
                 {u"lightgoldenrod3", 205, 190, 112},
                 {u"lightgoldenrod4", 139, 129, 76},
                 {u"lightyellow1", 255, 255, 224},
                 {u"lightyellow2", 238, 238, 209},
                 {u"lightyellow3", 205, 205, 180},
                 {u"lightyellow4", 139, 139, 122},
                 {u"yellow1", 255, 255, 0},
                 {u"yellow2", 238, 238, 0},
                 {u"yellow3", 205, 205, 0},
                 {u"yellow4", 139, 139, 0},
                 {u"gold1", 255, 215, 0},
                 {u"gold2", 238, 201, 0},
                 {u"gold3", 205, 173, 0},
                 {u"gold4", 139, 117, 0},
                 {u"goldenrod1", 255, 193, 37},
                 {u"goldenrod2", 238, 180, 34},
                 {u"goldenrod3", 205, 155, 29},
                 {u"goldenrod4", 139, 105, 20},
                 {u"darkgoldenrod1", 255, 185, 15},
                 {u"darkgoldenrod2", 238, 173, 14},
                 {u"darkgoldenrod3", 205, 149, 12},
                 {u"darkgoldenrod4", 139, 101, 8},
                 {u"rosybrown1", 255, 193, 193},
                 {u"rosybrown2", 238, 180, 180},
                 {u"rosybrown3", 205, 155, 155},
                 {u"rosybrown4", 139, 105, 105},
                 {u"indianred1", 255, 106, 106},
                 {u"indianred2", 238, 99, 99},
                 {u"indianred3", 205, 85, 85},
                 {u"indianred4", 139, 58, 58},
                 {u"sienna1", 255, 130, 71},
                 {u"sienna2", 238, 121, 66},
                 {u"sienna3", 205, 104, 57},
                 {u"sienna4", 139, 71, 38},
                 {u"burlywood1", 255, 211, 155},
                 {u"burlywood2", 238, 197, 145},
                 {u"burlywood3", 205, 170, 125},
                 {u"burlywood4", 139, 115, 85},
                 {u"wheat1", 255, 231, 186},
                 {u"wheat2", 238, 216, 174},
                 {u"wheat3", 205, 186, 150},
                 {u"wheat4", 139, 126, 102},
                 {u"tan1", 255, 165, 79},
                 {u"tan2", 238, 154, 73},
                 {u"tan3", 205, 133, 63},
                 {u"tan4", 139, 90, 43},
                 {u"chocolate1", 255, 127, 36},
                 {u"chocolate2", 238, 118, 33},
                 {u"chocolate3", 205, 102, 29},
                 {u"chocolate4", 139, 69, 19},
                 {u"firebrick1", 255, 48, 48},
                 {u"firebrick2", 238, 44, 44},
                 {u"firebrick3", 205, 38, 38},
                 {u"firebrick4", 139, 26, 26},
                 {u"brown1", 255, 64, 64},
                 {u"brown2", 238, 59, 59},
                 {u"brown3", 205, 51, 51},
                 {u"brown4", 139, 35, 35},
                 {u"salmon1", 255, 140, 105},
                 {u"salmon2", 238, 130, 98},
                 {u"salmon3", 205, 112, 84},
                 {u"salmon4", 139, 76, 57},
                 {u"lightsalmon1", 255, 160, 122},
                 {u"lightsalmon2", 238, 149, 114},
                 {u"lightsalmon3", 205, 129, 98},
                 {u"lightsalmon4", 139, 87, 66},
                 {u"orange1", 255, 165, 0},
                 {u"orange2", 238, 154, 0},
                 {u"orange3", 205, 133, 0},
                 {u"orange4", 139, 90, 0},
                 {u"darkorange1", 255, 127, 0},
                 {u"darkorange2", 238, 118, 0},
                 {u"darkorange3", 205, 102, 0},
                 {u"darkorange4", 139, 69, 0},
                 {u"coral1", 255, 114, 86},
                 {u"coral2", 238, 106, 80},
                 {u"coral3", 205, 91, 69},
                 {u"coral4", 139, 62, 47},
                 {u"tomato1", 255, 99, 71},
                 {u"tomato2", 238, 92, 66},
                 {u"tomato3", 205, 79, 57},
                 {u"tomato4", 139, 54, 38},
                 {u"orangered1", 255, 69, 0},
                 {u"orangered2", 238, 64, 0},
                 {u"orangered3", 205, 55, 0},
                 {u"orangered4", 139, 37, 0},
                 {u"red1", 255, 0, 0},
                 {u"red2", 238, 0, 0},
                 {u"red3", 205, 0, 0},
                 {u"red4", 139, 0, 0},
                 {u"deeppink1", 255, 20, 147},
                 {u"deeppink2", 238, 18, 137},
                 {u"deeppink3", 205, 16, 118},
                 {u"deeppink4", 139, 10, 80},
                 {u"hotpink1", 255, 110, 180},
                 {u"hotpink2", 238, 106, 167},
                 {u"hotpink3", 205, 96, 144},
                 {u"hotpink4", 139, 58, 98},
                 {u"pink1", 255, 181, 197},
                 {u"pink2", 238, 169, 184},
                 {u"pink3", 205, 145, 158},
                 {u"pink4", 139, 99, 108},
                 {u"lightpink1", 255, 174, 185},
                 {u"lightpink2", 238, 162, 173},
                 {u"lightpink3", 205, 140, 149},
                 {u"lightpink4", 139, 95, 101},
                 {u"palevioletred1", 255, 130, 171},
                 {u"palevioletred2", 238, 121, 159},
                 {u"palevioletred3", 205, 104, 137},
                 {u"palevioletred4", 139, 71, 93},
                 {u"maroon1", 255, 52, 179},
                 {u"maroon2", 238, 48, 167},
                 {u"maroon3", 205, 41, 144},
                 {u"maroon4", 139, 28, 98},
                 {u"violetred1", 255, 62, 150},
                 {u"violetred2", 238, 58, 140},
                 {u"violetred3", 205, 50, 120},
                 {u"violetred4", 139, 34, 82},
                 {u"magenta1", 255, 0, 255},
                 {u"magenta2", 238, 0, 238},
                 {u"magenta3", 205, 0, 205},
                 {u"magenta4", 139, 0, 139},
                 {u"orchid1", 255, 131, 250},
                 {u"orchid2", 238, 122, 233},
                 {u"orchid3", 205, 105, 201},
                 {u"orchid4", 139, 71, 137},
                 {u"plum1", 255, 187, 255},
                 {u"plum2", 238, 174, 238},
                 {u"plum3", 205, 150, 205},
                 {u"plum4", 139, 102, 139},
                 {u"mediumorchid1", 224, 102, 255},
                 {u"mediumorchid2", 209, 95, 238},
                 {u"mediumorchid3", 180, 82, 205},
                 {u"mediumorchid4", 122, 55, 139},
                 {u"darkorchid1", 191, 62, 255},
                 {u"darkorchid2", 178, 58, 238},
                 {u"darkorchid3", 154, 50, 205},
                 {u"darkorchid4", 104, 34, 139},
                 {u"purple1", 155, 48, 255},
                 {u"purple2", 145, 44, 238},
                 {u"purple3", 125, 38, 205},
                 {u"purple4", 85, 26, 139},
                 {u"mediumpurple1", 171, 130, 255},
                 {u"mediumpurple2", 159, 121, 238},
                 {u"mediumpurple3", 137, 104, 205},
                 {u"mediumpurple4", 93, 71, 139},
                 {u"thistle1", 255, 225, 255},
                 {u"thistle2", 238, 210, 238},
                 {u"thistle3", 205, 181, 205},
                 {u"thistle4", 139, 123, 139},
                 {u"gray0", 0, 0, 0},
                 {u"grey0", 0, 0, 0},
                 {u"gray1", 3, 3, 3},
                 {u"grey1", 3, 3, 3},
                 {u"gray2", 5, 5, 5},
                 {u"grey2", 5, 5, 5},
                 {u"gray3", 8, 8, 8},
                 {u"grey3", 8, 8, 8},
                 {u"gray4", 10, 10, 10},
                 {u"grey4", 10, 10, 10},
                 {u"gray5", 13, 13, 13},
                 {u"grey5", 13, 13, 13},
                 {u"gray6", 15, 15, 15},
                 {u"grey6", 15, 15, 15},
                 {u"gray7", 18, 18, 18},
                 {u"grey7", 18, 18, 18},
                 {u"gray8", 20, 20, 20},
                 {u"grey8", 20, 20, 20},
                 {u"gray9", 23, 23, 23},
                 {u"grey9", 23, 23, 23},
                 {u"gray10", 26, 26, 26},
                 {u"grey10", 26, 26, 26},
                 {u"gray11", 28, 28, 28},
                 {u"grey11", 28, 28, 28},
                 {u"gray12", 31, 31, 31},
                 {u"grey12", 31, 31, 31},
                 {u"gray13", 33, 33, 33},
                 {u"grey13", 33, 33, 33},
                 {u"gray14", 36, 36, 36},
                 {u"grey14", 36, 36, 36},
                 {u"gray15", 38, 38, 38},
                 {u"grey15", 38, 38, 38},
                 {u"gray16", 41, 41, 41},
                 {u"grey16", 41, 41, 41},
                 {u"gray17", 43, 43, 43},
                 {u"grey17", 43, 43, 43},
                 {u"gray18", 46, 46, 46},
                 {u"grey18", 46, 46, 46},
                 {u"gray19", 48, 48, 48},
                 {u"grey19", 48, 48, 48},
                 {u"gray20", 51, 51, 51},
                 {u"grey20", 51, 51, 51},
                 {u"gray21", 54, 54, 54},
                 {u"grey21", 54, 54, 54},
                 {u"gray22", 56, 56, 56},
                 {u"grey22", 56, 56, 56},
                 {u"gray23", 59, 59, 59},
                 {u"grey23", 59, 59, 59},
                 {u"gray24", 61, 61, 61},
                 {u"grey24", 61, 61, 61},
                 {u"gray25", 64, 64, 64},
                 {u"grey25", 64, 64, 64},
                 {u"gray26", 66, 66, 66},
                 {u"grey26", 66, 66, 66},
                 {u"gray27", 69, 69, 69},
                 {u"grey27", 69, 69, 69},
                 {u"gray28", 71, 71, 71},
                 {u"grey28", 71, 71, 71},
                 {u"gray29", 74, 74, 74},
                 {u"grey29", 74, 74, 74},
                 {u"gray30", 77, 77, 77},
                 {u"grey30", 77, 77, 77},
                 {u"gray31", 79, 79, 79},
                 {u"grey31", 79, 79, 79},
                 {u"gray32", 82, 82, 82},
                 {u"grey32", 82, 82, 82},
                 {u"gray33", 84, 84, 84},
                 {u"grey33", 84, 84, 84},
                 {u"gray34", 87, 87, 87},
                 {u"grey34", 87, 87, 87},
                 {u"gray35", 89, 89, 89},
                 {u"grey35", 89, 89, 89},
                 {u"gray36", 92, 92, 92},
                 {u"grey36", 92, 92, 92},
                 {u"gray37", 94, 94, 94},
                 {u"grey37", 94, 94, 94},
                 {u"gray38", 97, 97, 97},
                 {u"grey38", 97, 97, 97},
                 {u"gray39", 99, 99, 99},
                 {u"grey39", 99, 99, 99},
                 {u"gray40", 102, 102, 102},
                 {u"grey40", 102, 102, 102},
                 {u"gray41", 105, 105, 105},
                 {u"grey41", 105, 105, 105},
                 {u"gray42", 107, 107, 107},
                 {u"grey42", 107, 107, 107},
                 {u"gray43", 110, 110, 110},
                 {u"grey43", 110, 110, 110},
                 {u"gray44", 112, 112, 112},
                 {u"grey44", 112, 112, 112},
                 {u"gray45", 115, 115, 115},
                 {u"grey45", 115, 115, 115},
                 {u"gray46", 117, 117, 117},
                 {u"grey46", 117, 117, 117},
                 {u"gray47", 120, 120, 120},
                 {u"grey47", 120, 120, 120},
                 {u"gray48", 122, 122, 122},
                 {u"grey48", 122, 122, 122},
                 {u"gray49", 125, 125, 125},
                 {u"grey49", 125, 125, 125},
                 {u"gray50", 127, 127, 127},
                 {u"grey50", 127, 127, 127},
                 {u"gray51", 130, 130, 130},
                 {u"grey51", 130, 130, 130},
                 {u"gray52", 133, 133, 133},
                 {u"grey52", 133, 133, 133},
                 {u"gray53", 135, 135, 135},
                 {u"grey53", 135, 135, 135},
                 {u"gray54", 138, 138, 138},
                 {u"grey54", 138, 138, 138},
                 {u"gray55", 140, 140, 140},
                 {u"grey55", 140, 140, 140},
                 {u"gray56", 143, 143, 143},
                 {u"grey56", 143, 143, 143},
                 {u"gray57", 145, 145, 145},
                 {u"grey57", 145, 145, 145},
                 {u"gray58", 148, 148, 148},
                 {u"grey58", 148, 148, 148},
                 {u"gray59", 150, 150, 150},
                 {u"grey59", 150, 150, 150},
                 {u"gray60", 153, 153, 153},
                 {u"grey60", 153, 153, 153},
                 {u"gray61", 156, 156, 156},
                 {u"grey61", 156, 156, 156},
                 {u"gray62", 158, 158, 158},
                 {u"grey62", 158, 158, 158},
                 {u"gray63", 161, 161, 161},
                 {u"grey63", 161, 161, 161},
                 {u"gray64", 163, 163, 163},
                 {u"grey64", 163, 163, 163},
                 {u"gray65", 166, 166, 166},
                 {u"grey65", 166, 166, 166},
                 {u"gray66", 168, 168, 168},
                 {u"grey66", 168, 168, 168},
                 {u"gray67", 171, 171, 171},
                 {u"grey67", 171, 171, 171},
                 {u"gray68", 173, 173, 173},
                 {u"grey68", 173, 173, 173},
                 {u"gray69", 176, 176, 176},
                 {u"grey69", 176, 176, 176},
                 {u"gray70", 179, 179, 179},
                 {u"grey70", 179, 179, 179},
                 {u"gray71", 181, 181, 181},
                 {u"grey71", 181, 181, 181},
                 {u"gray72", 184, 184, 184},
                 {u"grey72", 184, 184, 184},
                 {u"gray73", 186, 186, 186},
                 {u"grey73", 186, 186, 186},
                 {u"gray74", 189, 189, 189},
                 {u"grey74", 189, 189, 189},
                 {u"gray75", 191, 191, 191},
                 {u"grey75", 191, 191, 191},
                 {u"gray76", 194, 194, 194},
                 {u"grey76", 194, 194, 194},
                 {u"gray77", 196, 196, 196},
                 {u"grey77", 196, 196, 196},
                 {u"gray78", 199, 199, 199},
                 {u"grey78", 199, 199, 199},
                 {u"gray79", 201, 201, 201},
                 {u"grey79", 201, 201, 201},
                 {u"gray80", 204, 204, 204},
                 {u"grey80", 204, 204, 204},
                 {u"gray81", 207, 207, 207},
                 {u"grey81", 207, 207, 207},
                 {u"gray82", 209, 209, 209},
                 {u"grey82", 209, 209, 209},
                 {u"gray83", 212, 212, 212},
                 {u"grey83", 212, 212, 212},
                 {u"gray84", 214, 214, 214},
                 {u"grey84", 214, 214, 214},
                 {u"gray85", 217, 217, 217},
                 {u"grey85", 217, 217, 217},
                 {u"gray86", 219, 219, 219},
                 {u"grey86", 219, 219, 219},
                 {u"gray87", 222, 222, 222},
                 {u"grey87", 222, 222, 222},
                 {u"gray88", 224, 224, 224},
                 {u"grey88", 224, 224, 224},
                 {u"gray89", 227, 227, 227},
                 {u"grey89", 227, 227, 227},
                 {u"gray90", 229, 229, 229},
                 {u"grey90", 229, 229, 229},
                 {u"gray91", 232, 232, 232},
                 {u"grey91", 232, 232, 232},
                 {u"gray92", 235, 235, 235},
                 {u"grey92", 235, 235, 235},
                 {u"gray93", 237, 237, 237},
                 {u"grey93", 237, 237, 237},
                 {u"gray94", 240, 240, 240},
                 {u"grey94", 240, 240, 240},
                 {u"gray95", 242, 242, 242},
                 {u"grey95", 242, 242, 242},
                 {u"gray96", 245, 245, 245},
                 {u"grey96", 245, 245, 245},
                 {u"gray97", 247, 247, 247},
                 {u"grey97", 247, 247, 247},
                 {u"gray98", 250, 250, 250},
                 {u"grey98", 250, 250, 250},
                 {u"gray99", 252, 252, 252},
                 {u"grey99", 252, 252, 252},
                 {u"gray100", 255, 255, 255},
                 {u"grey100", 255, 255, 255},
                 {nullptr, 0, 0, 0}};

Dot2QtConsts::Dot2QtConsts()
{
    m_penStyles = {
        {QStringLiteral("solid"), Qt::SolidLine},
        {QStringLiteral("dashed"), Qt::DashLine},
        {QStringLiteral("dotted"), Qt::DotLine},
        {QStringLiteral("invis"), Qt::NoPen},
    };

    m_colors = {
        {QStringLiteral("crimson"), QStringLiteral("#DC143C")},
        {QStringLiteral("hot_pink"), QStringLiteral("#FF69B4")},
        {QStringLiteral("light_yellow"), QStringLiteral("#FFFFE0")},
        {QStringLiteral("slate_blue"), QStringLiteral("#6A5ACD")},
    };

    uint i = 0;
    while (postscriptFontNames[i].input) {
        QFont font(postscriptFontNames[i].input.toString());
        m_psFonts[postscriptFontNames[i].roman.toString()] = font;
        if (postscriptFontNames[i].italic) {
            QFont italic = font;
            italic.setItalic(true);
            m_psFonts[postscriptFontNames[i].italic.toString()] = italic;
        }
        if (postscriptFontNames[i].bold) {
            QFont bold = font;
            bold.setBold(true);
            m_psFonts[postscriptFontNames[i].bold.toString()] = bold;
        }
        if (postscriptFontNames[i].boldItalic) {
            QFont boldItalic = font;
            boldItalic.setItalic(true);
            boldItalic.setBold(true);
            m_psFonts[postscriptFontNames[i].boldItalic.toString()] = boldItalic;
        }
        if (postscriptFontNames[i].light) {
            QFont light = font;
            light.setWeight(QFont::Light);
            m_psFonts[postscriptFontNames[i].light.toString()] = light;
        }
        if (postscriptFontNames[i].lightItalic) {
            QFont lightItalic = font;
            lightItalic.setWeight(QFont::Light);
            lightItalic.setItalic(true);
            m_psFonts[postscriptFontNames[i].lightItalic.toString()] = lightItalic;
        }
        i++;
    }

    uint j = 0;
    while (color_lib[j].name) {
        //     () << "Adding color " << color_lib[j].name;
        m_qcolors[color_lib[j].name.toString()] = QColor(color_lib[j].r, color_lib[j].g, color_lib[j].b);
        j++;
    }
}

Dot2QtConsts::~Dot2QtConsts()
{
}

QColor Dot2QtConsts::qtColor(const QString &dotColor) const
{
    //   () << "Dot2QtConsts::qtColor";
    QColor color;
    if (parse_numeric_color(qPrintable(dotColor), color)) {
        return color;
    }
    auto qit = m_qcolors.find(dotColor);
    if (qit != m_qcolors.end()) {
        return *qit;
    }
    QColor res(dotColor);
    if (res.isValid()) {
        return res;
    }
    auto it = m_colors.find(dotColor);
    if (it != m_colors.end()) {
        res = QColor(*it);
        if (res.isValid()) {
            return res;
        }

        qCWarning(KGRAPHVIEWERLIB_LOG) << "Unknown stored DOT color '" << dotColor << "'. returning Qt black";
        return Qt::black;
    }

    //           qCWarning(KGRAPHVIEWERLIB_LOG) << "Unknown DOT color '" << dotColor << "'. returning Qt black";
    return Qt::black;
}

Qt::PenStyle Dot2QtConsts::qtPenStyle(const QString &dotLineStyle) const
{
    auto it = m_penStyles.find(dotLineStyle);
    if (it != m_penStyles.end()) {
        return *it;
    }

    if (!dotLineStyle.left(12).isEmpty() && dotLineStyle.left(12) != QLatin1String("setlinewidth")) {
        qCWarning(KGRAPHVIEWERLIB_LOG) << "Unknown DOT line style '" << dotLineStyle << "'. returning Qt solid line";
    }
    return Qt::SolidLine;
}

QFont Dot2QtConsts::qtFont(const QString &dotFont) const
{
    auto it = m_psFonts.find(dotFont);
    if (it != m_psFonts.end()) {
        return *it;
    }

    return QFont(QFont::substitute(dotFont));
}
