/*
 * Copyright 2019-2025 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

/*
 * This file contains self test data required by FIPS 140-3 IG
 * 10.3.A Cryptographic Algorithm Self test Requirements
 *
 * Note that in the 'General CAST requirements': Note33 Allows individual
 * self tests for low level algorithms (such as digests) to be omitted, if
 * they are tested as part of a higher level algorithm (such as HMAC).
 */

/* Macros to build Self test data */
#define ITM(x) ((void *)&x), sizeof(x)
#define ITM_STR(x) ((void *)&x), (sizeof(x) - 1)

#define ST_KAT_PARAM_END() { "", 0, NULL, 0 }
#define ST_KAT_PARAM_BIGNUM(name, data)                                        \
    { name, OSSL_PARAM_UNSIGNED_INTEGER, ITM(data) }
#define ST_KAT_PARAM_OCTET(name, data)                                         \
    { name, OSSL_PARAM_OCTET_STRING, ITM(data) }
#define ST_KAT_PARAM_UTF8STRING(name, data)                                    \
    { name, OSSL_PARAM_UTF8_STRING, ITM_STR(data) }
#define ST_KAT_PARAM_UTF8CHAR(name, data)                                      \
    { name, OSSL_PARAM_UTF8_STRING, ITM(data) }
#define ST_KAT_PARAM_INT(name, i)                                              \
    { name, OSSL_PARAM_INTEGER, ITM(i) }

/* used to store raw parameters for keys and algorithms */
typedef struct st_kat_param_st {
    const char *name;  /* an OSSL_PARAM name */
    size_t type;       /* the type associated with the data */
    const void *data;  /* unsigned char [], or char [] depending on the type */
    size_t data_len;   /* the length of the data */
} ST_KAT_PARAM;

typedef struct st_kat_st {
    const char *desc;
    const char *algorithm;
    const unsigned char *pt;
    size_t pt_len;
    const unsigned char *expected;
    size_t expected_len;
} ST_KAT;

#define CIPHER_MODE_ENCRYPT 1
#define CIPHER_MODE_DECRYPT 2
#define CIPHER_MODE_ALL     (CIPHER_MODE_ENCRYPT | CIPHER_MODE_DECRYPT)

/* FIPS 140-3 only allows DSA verification for legacy purposes */
#define SIGNATURE_MODE_VERIFY_ONLY  1
#define SIGNATURE_MODE_SIGN_ONLY    2
#define SIGNATURE_MODE_DIGESTED     4
#define SIGNATURE_MODE_SIG_DIGESTED 8

typedef ST_KAT ST_KAT_DIGEST;
typedef struct st_kat_cipher_st {
    ST_KAT base;
    int mode;
    const unsigned char *key;
    size_t key_len;
    const unsigned char *iv;
    size_t iv_len;
    const unsigned char *aad;
    size_t aad_len;
    const unsigned char *tag;
    size_t tag_len;
} ST_KAT_CIPHER;

typedef struct st_kat_kdf_st {
    const char *desc;
    const char *algorithm;
    const ST_KAT_PARAM *params;
    const unsigned char *expected;
    size_t expected_len;
} ST_KAT_KDF;

typedef struct st_kat_drbg_st {
    const char *desc;
    const char *algorithm;
    const char *param_name;
    char *param_value;
    const unsigned char *entropyin;
    size_t entropyinlen;
    const unsigned char *nonce;
    size_t noncelen;
    const unsigned char *persstr;
    size_t persstrlen;
    const unsigned char *entropyinpr1;
    size_t entropyinpr1len;
    const unsigned char *entropyinpr2;
    size_t entropyinpr2len;
    const unsigned char *entropyaddin1;
    size_t entropyaddin1len;
    const unsigned char *entropyaddin2;
    size_t entropyaddin2len;
    const unsigned char *expected;
    size_t expectedlen;
} ST_KAT_DRBG;

typedef struct st_kat_kas_st {
    const char *desc;
    const char *algorithm;

    const ST_KAT_PARAM *key_group;
    const ST_KAT_PARAM *key_host_data;
    const ST_KAT_PARAM *key_peer_data;

    const unsigned char *expected;
    size_t expected_len;
} ST_KAT_KAS;

typedef struct st_kat_sign_st {
    const char *desc;
    const char *keytype;
    const char *sigalgorithm;
    int mode;
    const ST_KAT_PARAM *key;
    const unsigned char *msg;
    size_t msg_len;
    const unsigned char *entropy;
    size_t entropy_len;
    const unsigned char *nonce;
    size_t nonce_len;
    const unsigned char *persstr;
    size_t persstr_len;
    const unsigned char *sig_expected; /* Set to NULL if this value changes */
    size_t sig_expected_len;
    const ST_KAT_PARAM *init;
    const ST_KAT_PARAM *verify;
} ST_KAT_SIGN;

typedef struct st_kat_asym_cipher_st {
    const char *desc;
    const char *algorithm;
    int encrypt;
    const ST_KAT_PARAM *key;
    const ST_KAT_PARAM *postinit;
    const unsigned char *in;
    size_t in_len;
    const unsigned char *expected;
    size_t expected_len;
} ST_KAT_ASYM_CIPHER;

typedef struct st_kat_keygen_st {
    const char *desc;
    const char *algorithm;
    const ST_KAT_PARAM *keygen_params;
    const ST_KAT_PARAM *expected_params;
} ST_KAT_ASYM_KEYGEN;

typedef struct st_kat_kem_st {
    const char *desc;
    const char *algorithm;
    const ST_KAT_PARAM *key;
    const unsigned char *cipher_text;
    size_t cipher_text_len;
    const unsigned char *entropy;
    size_t entropy_len;
    const unsigned char *secret;
    size_t secret_len;
    const unsigned char *reject_secret;
} ST_KAT_KEM;

/*- DIGEST SELF TEST DATA */
static const unsigned char sha1_pt[] = "abc";
static const unsigned char sha1_digest[] = {
    0xA9, 0x99, 0x3E, 0x36, 0x47, 0x06, 0x81, 0x6A,
    0xBA, 0x3E, 0x25, 0x71, 0x78, 0x50, 0xC2, 0x6C,
    0x9C, 0xD0, 0xD8, 0x9D
};
static const unsigned char sha512_pt[] = "abc";
static const unsigned char sha512_digest[] = {
    0xDD, 0xAF, 0x35, 0xA1, 0x93, 0x61, 0x7A, 0xBA, 0xCC, 0x41, 0x73, 0x49,
    0xAE, 0x20, 0x41, 0x31, 0x12, 0xE6, 0xFA, 0x4E, 0x89, 0xA9, 0x7E, 0xA2,
    0x0A, 0x9E, 0xEE, 0xE6, 0x4B, 0x55, 0xD3, 0x9A, 0x21, 0x92, 0x99, 0x2A,
    0x27, 0x4F, 0xC1, 0xA8, 0x36, 0xBA, 0x3C, 0x23, 0xA3, 0xFE, 0xEB, 0xBD,
    0x45, 0x4D, 0x44, 0x23, 0x64, 0x3C, 0xE8, 0x0E, 0x2A, 0x9A, 0xC9, 0x4F,
    0xA5, 0x4C, 0xA4, 0x9F
};
static const unsigned char sha3_256_pt[] = { 0xe7, 0x37, 0x21, 0x05 };
static const unsigned char sha3_256_digest[] = {
    0x3a, 0x42, 0xb6, 0x8a, 0xb0, 0x79, 0xf2, 0x8c, 0x4c, 0xa3, 0xc7, 0x52,
    0x29, 0x6f, 0x27, 0x90, 0x06, 0xc4, 0xfe, 0x78, 0xb1, 0xeb, 0x79, 0xd9,
    0x89, 0x77, 0x7f, 0x05, 0x1e, 0x40, 0x46, 0xae
};

/*
 * Note:
 *  SHA256 is tested by higher level algorithms so a
 *  CAST is not needed.
 */
static const ST_KAT_DIGEST st_kat_digest_tests[] =
{
    {
         OSSL_SELF_TEST_DESC_MD_SHA1,
         "SHA1",
         ITM_STR(sha1_pt),
         ITM(sha1_digest),
    },
    {
         OSSL_SELF_TEST_DESC_MD_SHA2,
         "SHA512",
         ITM_STR(sha512_pt),
         ITM(sha512_digest),
    },
    {
         OSSL_SELF_TEST_DESC_MD_SHA3,
         "SHA3-256",
         ITM(sha3_256_pt),
         ITM(sha3_256_digest),
    },
};

/*- CIPHER TEST DATA */

/* AES-256 GCM test data */
static const unsigned char aes_256_gcm_key[] = {
    0x92, 0xe1, 0x1d, 0xcd, 0xaa, 0x86, 0x6f, 0x5c,
    0xe7, 0x90, 0xfd, 0x24, 0x50, 0x1f, 0x92, 0x50,
    0x9a, 0xac, 0xf4, 0xcb, 0x8b, 0x13, 0x39, 0xd5,
    0x0c, 0x9c, 0x12, 0x40, 0x93, 0x5d, 0xd0, 0x8b
};
static const unsigned char aes_256_gcm_iv[] = {
    0xac, 0x93, 0xa1, 0xa6, 0x14, 0x52, 0x99, 0xbd,
    0xe9, 0x02, 0xf2, 0x1a
};
static const unsigned char aes_256_gcm_pt[] = {
    0x2d, 0x71, 0xbc, 0xfa, 0x91, 0x4e, 0x4a, 0xc0,
    0x45, 0xb2, 0xaa, 0x60, 0x95, 0x5f, 0xad, 0x24
};
static const unsigned char aes_256_gcm_aad[] = {
    0x1e, 0x08, 0x89, 0x01, 0x6f, 0x67, 0x60, 0x1c,
    0x8e, 0xbe, 0xa4, 0x94, 0x3b, 0xc2, 0x3a, 0xd6
};
static const unsigned char aes_256_gcm_ct[] = {
    0x89, 0x95, 0xae, 0x2e, 0x6d, 0xf3, 0xdb, 0xf9,
    0x6f, 0xac, 0x7b, 0x71, 0x37, 0xba, 0xe6, 0x7f
};
static const unsigned char aes_256_gcm_tag[] = {
    0xec, 0xa5, 0xaa, 0x77, 0xd5, 0x1d, 0x4a, 0x0a,
    0x14, 0xd9, 0xc5, 0x1e, 0x1d, 0xa4, 0x74, 0xab
};

/* AES-ECB test data */
static const unsigned char aes_128_ecb_key[] = {
    0x10, 0xa5, 0x88, 0x69, 0xd7, 0x4b, 0xe5, 0xa3,
    0x74, 0xcf, 0x86, 0x7c, 0xfb, 0x47, 0x38, 0x59
};
static const unsigned char aes_128_ecb_pt[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static const unsigned char aes_128_ecb_ct[] = {
    0x6d, 0x25, 0x1e, 0x69, 0x44, 0xb0, 0x51, 0xe0,
    0x4e, 0xaa, 0x6f, 0xb4, 0xdb, 0xf7, 0x84, 0x65
};

#ifndef OPENSSL_NO_DES
/*
 * TDES-ECB test data from
 * https://github.com/usnistgov/ACVP-Server/blob/master/gen-val/json-files/ACVP-TDES-ECB-1.0
 * Decrypt
 */
static const unsigned char tdes_key[] = {
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
    0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE
};
static const unsigned char tdes_ct[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
};
static const unsigned char tdes_pt[] = {
    0x4B, 0xAB, 0x3B, 0xE1, 0x50, 0x2E, 0x3B, 0x36
};
#endif

static const ST_KAT_CIPHER st_kat_cipher_tests[] = {
    {
        {
            OSSL_SELF_TEST_DESC_CIPHER_AES_GCM,
            "AES-256-GCM",
            ITM(aes_256_gcm_pt),
            ITM(aes_256_gcm_ct)
        },
        CIPHER_MODE_ENCRYPT | CIPHER_MODE_DECRYPT,
        ITM(aes_256_gcm_key),
        ITM(aes_256_gcm_iv),
        ITM(aes_256_gcm_aad),
        ITM(aes_256_gcm_tag)
    },
    {
        {
            OSSL_SELF_TEST_DESC_CIPHER_AES_ECB,
            "AES-128-ECB",
            ITM(aes_128_ecb_pt),
            ITM(aes_128_ecb_ct)
        },
        CIPHER_MODE_DECRYPT,
        ITM(aes_128_ecb_key)
    },
#ifndef OPENSSL_NO_DES
    {
        {
            OSSL_SELF_TEST_DESC_CIPHER_TDES,
            "DES-EDE3-ECB",
            ITM(tdes_pt),
            ITM(tdes_ct)
        },
        CIPHER_MODE_DECRYPT,
        ITM(tdes_key)
    }
#endif
};

static const char hkdf_digest[] = "SHA256";
/*
 * Input parameters and expected result are from RFC 5869 test case 1, which is
 * with a key-derivation key >= 112 bits required by NIST SP 800-131Ar2
 * section 8.
 */
static const unsigned char hkdf_secret[] = {
    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b
};
static const unsigned char hkdf_salt[] = {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c
};
static const unsigned char hkdf_info[] = {
    0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
    0xf8, 0xf9
};
static const unsigned char hkdf_expected[] = {
    0x3c, 0xb2, 0x5f, 0x25, 0xfa, 0xac, 0xd5, 0x7a,
    0x90, 0x43, 0x4f, 0x64, 0xd0, 0x36, 0x2f, 0x2a,
    0x2d, 0x2d, 0x0a, 0x90, 0xcf, 0x1a, 0x5a, 0x4c,
    0x5d, 0xb0, 0x2d, 0x56, 0xec, 0xc4, 0xc5, 0xbf,
    0x34, 0x00, 0x72, 0x08, 0xd5, 0xb8, 0x87, 0x18,
    0x58, 0x65
};
static const ST_KAT_PARAM hkdf_params[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, hkdf_digest),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_KEY, hkdf_secret),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_SALT, hkdf_salt),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_INFO, hkdf_info),
    ST_KAT_PARAM_END()
};

static const char sskdf_digest[] = "SHA256";
static const unsigned char sskdf_secret[] = {
    0x6d, 0xbd, 0xc2, 0x3f, 0x04, 0x54, 0x88, 0xe4,
    0x06, 0x27, 0x57, 0xb0, 0x6b, 0x9e, 0xba, 0xe1,
    0x83, 0xfc, 0x5a, 0x59, 0x46, 0xd8, 0x0d, 0xb9,
    0x3f, 0xec, 0x6f, 0x62, 0xec, 0x07, 0xe3, 0x72,
    0x7f, 0x01, 0x26, 0xae, 0xd1, 0x2c, 0xe4, 0xb2,
    0x62, 0xf4, 0x7d, 0x48, 0xd5, 0x42, 0x87, 0xf8,
    0x1d, 0x47, 0x4c, 0x7c, 0x3b, 0x18, 0x50, 0xe9
};
static const unsigned char sskdf_otherinfo[] = {
    0xa1, 0xb2, 0xc3, 0xd4, 0xe5, 0x43, 0x41, 0x56,
    0x53, 0x69, 0x64, 0x3c, 0x83, 0x2e, 0x98, 0x49,
    0xdc, 0xdb, 0xa7, 0x1e, 0x9a, 0x31, 0x39, 0xe6,
    0x06, 0xe0, 0x95, 0xde, 0x3c, 0x26, 0x4a, 0x66,
    0xe9, 0x8a, 0x16, 0x58, 0x54, 0xcd, 0x07, 0x98,
    0x9b, 0x1e, 0xe0, 0xec, 0x3f, 0x8d, 0xbe
};
static const unsigned char sskdf_expected[] = {
    0x27, 0xce, 0x57, 0xed, 0xb1, 0x7e, 0x1f, 0xf2,
    0xe4, 0x79, 0x2e, 0x84, 0x8b, 0x04, 0xf1, 0xae
};
static const ST_KAT_PARAM sskdf_params[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, sskdf_digest),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_KEY, sskdf_secret),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_INFO, sskdf_otherinfo),
    ST_KAT_PARAM_END()
};

static const char x942kdf_digest[] = "SHA256";
static const char x942kdf_cekalg[] = "AES-128-WRAP";
static const unsigned char x942kdf_secret[] = {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x10, 0x11, 0x12, 0x13
};
static const unsigned char x942kdf_expected[] = {
    0x79, 0x66, 0xa0, 0x38, 0x22, 0x28, 0x1e, 0xa3,
    0xeb, 0x08, 0xd9, 0xbc, 0x69, 0x5b, 0xd8, 0xff
};
static const ST_KAT_PARAM x942kdf_params[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, x942kdf_digest),
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_CEK_ALG, x942kdf_cekalg),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_KEY, x942kdf_secret),
    ST_KAT_PARAM_END()
};

static const char x963kdf_digest[] = "SHA256";
static const unsigned char x963kdf_otherinfo[] = {
    0x75, 0xee, 0xf8, 0x1a, 0xa3, 0x04, 0x1e, 0x33,
    0xb8, 0x09, 0x71, 0x20, 0x3d, 0x2c, 0x0c, 0x52
};
static const unsigned char x963kdf_secret[] = {
    0x22, 0x51, 0x8b, 0x10, 0xe7, 0x0f, 0x2a, 0x3f,
    0x24, 0x38, 0x10, 0xae, 0x32, 0x54, 0x13, 0x9e,
    0xfb, 0xee, 0x04, 0xaa, 0x57, 0xc7, 0xaf, 0x7d
};
static const unsigned char x963kdf_expected[] = {
    0xc4, 0x98, 0xaf, 0x77, 0x16, 0x1c, 0xc5, 0x9f,
    0x29, 0x62, 0xb9, 0xa7, 0x13, 0xe2, 0xb2, 0x15,
    0x15, 0x2d, 0x13, 0x97, 0x66, 0xce, 0x34, 0xa7,
    0x76, 0xdf, 0x11, 0x86, 0x6a, 0x69, 0xbf, 0x2e
};
static const ST_KAT_PARAM x963kdf_params[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, x963kdf_digest),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_KEY, x963kdf_secret),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_INFO, x963kdf_otherinfo),
    ST_KAT_PARAM_END()
};

static const char pbkdf2_digest[] = "SHA256";
/*
 * Input parameters from RFC 6070, vector 5 (because it is the only one with
 * a salt >= 16 bytes, which NIST SP 800-132 section 5.1 requires). The
 * expected output is taken from
 *   https://github.com/brycx/Test-Vector-Generation/blob/master/PBKDF2/pbkdf2-hmac-sha2-test-vectors.md,
 * which ran these test vectors with SHA-256.
 * Note that the output only uses 2 iterations.
 */
static const unsigned char pbkdf2_password[] = {
    0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x50, 0x41, 0x53, 0x53,
    0x57, 0x4f, 0x52, 0x44, 0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64
};
static const unsigned char pbkdf2_salt[] = {
    0x73, 0x61, 0x6c, 0x74, 0x53, 0x41, 0x4c, 0x54, 0x73, 0x61, 0x6c, 0x74,
    0x53, 0x41, 0x4c, 0x54, 0x73, 0x61, 0x6c, 0x74, 0x53, 0x41, 0x4c, 0x54,
    0x73, 0x61, 0x6c, 0x74, 0x53, 0x41, 0x4c, 0x54, 0x73, 0x61, 0x6c, 0x74
};
static const unsigned char pbkdf2_expected[] = {
    0x13, 0xdc, 0x8a, 0x7c, 0x13, 0xd3, 0x72, 0xc9,
    0x03, 0x82, 0x82, 0x2d, 0x2d, 0xc4, 0x92, 0xf2,
    0xed, 0x52, 0x46, 0x7f, 0xb7, 0x82, 0x8e, 0xa8,
    0x64
};
/*
 * FIPS 140-3 IG 10.3.A.8 allows the iteration count to be smaller
 * so we use the minimum of 2 here.
 */
static int pbkdf2_iterations = 2;
static int pbkdf2_pkcs5 = 1; /* Disable compliance checks so a smaller iteration count can be used */
static const ST_KAT_PARAM pbkdf2_params[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, pbkdf2_digest),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_PASSWORD, pbkdf2_password),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_SALT, pbkdf2_salt),
    ST_KAT_PARAM_INT(OSSL_KDF_PARAM_ITER, pbkdf2_iterations),
    ST_KAT_PARAM_INT(OSSL_KDF_PARAM_PKCS5, pbkdf2_pkcs5),
    ST_KAT_PARAM_END()
};

static const char tls12prf_digest[] = "SHA256";
static const unsigned char tls12prf_secret[] = {
    0x20, 0x2c, 0x88, 0xc0, 0x0f, 0x84, 0xa1, 0x7a,
    0x20, 0x02, 0x70, 0x79, 0x60, 0x47, 0x87, 0x46,
    0x11, 0x76, 0x45, 0x55, 0x39, 0xe7, 0x05, 0xbe,
    0x73, 0x08, 0x90, 0x60, 0x2c, 0x28, 0x9a, 0x50,
    0x01, 0xe3, 0x4e, 0xeb, 0x3a, 0x04, 0x3e, 0x5d,
    0x52, 0xa6, 0x5e, 0x66, 0x12, 0x51, 0x88, 0xbf,
};
static const unsigned char tls12prf_seed[] = {
    'k', 'e', 'y', ' ', 'e', 'x', 'p', 'a', 'n', 's', 'i', 'o', 'n',
    0xae, 0x6c, 0x80, 0x6f, 0x8a, 0xd4, 0xd8, 0x07,
    0x84, 0x54, 0x9d, 0xff, 0x28, 0xa4, 0xb5, 0x8f,
    0xd8, 0x37, 0x68, 0x1a, 0x51, 0xd9, 0x28, 0xc3,
    0xe3, 0x0e, 0xe5, 0xff, 0x14, 0xf3, 0x98, 0x68,
    0x62, 0xe1, 0xfd, 0x91, 0xf2, 0x3f, 0x55, 0x8a,
    0x60, 0x5f, 0x28, 0x47, 0x8c, 0x58, 0xcf, 0x72,
    0x63, 0x7b, 0x89, 0x78, 0x4d, 0x95, 0x9d, 0xf7,
    0xe9, 0x46, 0xd3, 0xf0, 0x7b, 0xd1, 0xb6, 0x16,
 };
static const unsigned char tls12prf_expected[] = {
    0xd0, 0x61, 0x39, 0x88, 0x9f, 0xff, 0xac, 0x1e,
    0x3a, 0x71, 0x86, 0x5f, 0x50, 0x4a, 0xa5, 0xd0,
    0xd2, 0xa2, 0xe8, 0x95, 0x06, 0xc6, 0xf2, 0x27,
    0x9b, 0x67, 0x0c, 0x3e, 0x1b, 0x74, 0xf5, 0x31,
    0x01, 0x6a, 0x25, 0x30, 0xc5, 0x1a, 0x3a, 0x0f,
    0x7e, 0x1d, 0x65, 0x90, 0xd0, 0xf0, 0x56, 0x6b,
    0x2f, 0x38, 0x7f, 0x8d, 0x11, 0xfd, 0x4f, 0x73,
    0x1c, 0xdd, 0x57, 0x2d, 0x2e, 0xae, 0x92, 0x7f,
    0x6f, 0x2f, 0x81, 0x41, 0x0b, 0x25, 0xe6, 0x96,
    0x0b, 0xe6, 0x89, 0x85, 0xad, 0xd6, 0xc3, 0x84,
    0x45, 0xad, 0x9f, 0x8c, 0x64, 0xbf, 0x80, 0x68,
    0xbf, 0x9a, 0x66, 0x79, 0x48, 0x5d, 0x96, 0x6f,
    0x1a, 0xd6, 0xf6, 0x8b, 0x43, 0x49, 0x5b, 0x10,
    0xa6, 0x83, 0x75, 0x5e, 0xa2, 0xb8, 0x58, 0xd7,
    0x0c, 0xca, 0xc7, 0xec, 0x8b, 0x05, 0x3c, 0x6b,
    0xd4, 0x1c, 0xa2, 0x99, 0xd4, 0xe5, 0x19, 0x28,
};
static const ST_KAT_PARAM tls12prf_params[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, tls12prf_digest),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_SECRET, tls12prf_secret),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_SEED, tls12prf_seed),
    ST_KAT_PARAM_END()
};

static const char kbkdf_digest[] = "SHA256";
static const char kbkdf_mac[] = "HMAC";
static const unsigned char kbkdf_salt[] = { 'p', 'r', 'f' };
static const unsigned char kbkdf_prfinput[] = { 't', 'e', 's', 't' };
static unsigned char kbkdf_key[] = {
    0x37, 0x05, 0xD9, 0x60, 0x80, 0xC1, 0x77, 0x28,
    0xA0, 0xE8, 0x00, 0xEA, 0xB6, 0xE0, 0xD2, 0x3C,
};
static unsigned char kbkdf_expected[] = {
    0x9D, 0x18, 0x86, 0x16, 0xF6, 0x38, 0x52, 0xFE,
    0x86, 0x91, 0x5B, 0xB8, 0x40, 0xB4, 0xA8, 0x86,
    0xFF, 0x3E, 0x6B, 0xB0, 0xF8, 0x19, 0xB4, 0x9B,
    0x89, 0x33, 0x93, 0xD3, 0x93, 0x85, 0x42, 0x95,
};
static const ST_KAT_PARAM kbkdf_params[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, kbkdf_digest),
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_MAC, kbkdf_mac),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_KEY, kbkdf_key),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_SALT, kbkdf_salt),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_INFO, kbkdf_prfinput),
    ST_KAT_PARAM_END()
};

static const char kbkdf_kmac_mac[] = "KMAC128";
static unsigned char kbkdf_kmac_label[] = {
    0xB5, 0xB5, 0xF3, 0x71, 0x9F, 0xBE, 0x5B, 0x3D,
    0x7B, 0x8D, 0x05, 0xA1, 0xD3, 0x25, 0x19, 0x50,
};
static unsigned char kbkdf_kmac_context[] = {
    0x36, 0x60, 0x0E, 0xF3, 0xC3, 0x70, 0xB5, 0xEF,
    0x58, 0xBE, 0xF1, 0xBA, 0x1C, 0xF2, 0x74, 0xCB,
};
static unsigned char kbkdf_kmac_key[] = {
    0xB2, 0x51, 0x4C, 0xC1, 0xD5, 0xCD, 0x7B, 0x6B,
    0xA3, 0x3C, 0x90, 0x05, 0xBD, 0xAC, 0x32, 0x2A,
};
static unsigned char kbkdf_kmac_expected[] = {
    0xB1, 0x58, 0xEE, 0xB1, 0x34, 0xA4, 0xDD, 0x9D,
    0xAC, 0x52, 0xBD, 0x9E, 0x30, 0xE8, 0x0D, 0x76,
    0x42, 0x57, 0x01, 0x89, 0x5F, 0x82, 0x74, 0xB9,
    0xEB, 0x3E, 0x84, 0xD8, 0xA5, 0xDE, 0x6E, 0x54,
};
static const ST_KAT_PARAM kbkdf_kmac_params[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_MAC, kbkdf_kmac_mac),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_KEY, kbkdf_kmac_key),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_SALT, kbkdf_kmac_label),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_INFO, kbkdf_kmac_context),
    ST_KAT_PARAM_END()
};

static const char tls13_kdf_digest[] = "SHA256";
static int tls13_kdf_extract_mode = EVP_KDF_HKDF_MODE_EXTRACT_ONLY;
static int tls13_kdf_expand_mode = EVP_KDF_HKDF_MODE_EXPAND_ONLY;
static const unsigned char tls13_kdf_prefix[] = {
    0x74, 0x6C, 0x73, 0x31, 0x33, 0x20          /* "tls13 " */
};
static const unsigned char tls13_kdf_client_early_secret_label[] = {
    0x63, 0x20, 0x65, 0x20, 0x74, 0x72, 0x61, 0x66,
    0x66, 0x69, 0x63                            /* "c e traffic"*/
};
static const unsigned char tls13_kdf_psk[] = {
    0xF8, 0xAF, 0x6A, 0xEA, 0x2D, 0x39, 0x7B, 0xAF,
    0x29, 0x48, 0xA2, 0x5B, 0x28, 0x34, 0x20, 0x06,
    0x92, 0xCF, 0xF1, 0x7E, 0xEE, 0x91, 0x65, 0xE4,
    0xE2, 0x7B, 0xAB, 0xEE, 0x9E, 0xDE, 0xFD, 0x05
};
static const unsigned char tls13_kdf_client_hello_hash[] = {
    0x7c, 0x92, 0xf6, 0x8b, 0xd5, 0xbf, 0x36, 0x38,
    0xea, 0x33, 0x8a, 0x64, 0x94, 0x72, 0x2e, 0x1b,
    0x44, 0x12, 0x7e, 0x1b, 0x7e, 0x8a, 0xad, 0x53,
    0x5f, 0x23, 0x22, 0xa6, 0x44, 0xff, 0x22, 0xb3
};

static const unsigned char tls13_kdf_early_secret[] = {
    0x15, 0x3B, 0x63, 0x94, 0xA9, 0xC0, 0x3C, 0xF3,
    0xF5, 0xAC, 0xCC, 0x6E, 0x45, 0x5A, 0x76, 0x93,
    0x28, 0x11, 0x38, 0xA1, 0xBC, 0xFA, 0x38, 0x03,
    0xC2, 0x67, 0x35, 0xDD, 0x11, 0x94, 0xD2, 0x16
};
static const unsigned char tls13_kdf_client_early_traffic_secret[] = {
    0xC8, 0x05, 0x83, 0xA9, 0x0E, 0x99, 0x5C, 0x48,
    0x96, 0x00, 0x49, 0x2A, 0x5D, 0xA6, 0x42, 0xE6,
    0xB1, 0xF6, 0x79, 0xBA, 0x67, 0x48, 0x28, 0x79,
    0x2D, 0xF0, 0x87, 0xB9, 0x39, 0x63, 0x61, 0x71
};
static const ST_KAT_PARAM tls13_kdf_early_secret_params[] = {
    ST_KAT_PARAM_INT(OSSL_KDF_PARAM_MODE, tls13_kdf_extract_mode),
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, tls13_kdf_digest),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_KEY, tls13_kdf_psk),
    ST_KAT_PARAM_END()
};
static const ST_KAT_PARAM tls13_kdf_client_early_secret_params[] = {
    ST_KAT_PARAM_INT(OSSL_KDF_PARAM_MODE, tls13_kdf_expand_mode),
    ST_KAT_PARAM_UTF8STRING(OSSL_KDF_PARAM_DIGEST, tls13_kdf_digest),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_KEY, tls13_kdf_early_secret),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_DATA, tls13_kdf_client_hello_hash),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_PREFIX, tls13_kdf_prefix),
    ST_KAT_PARAM_OCTET(OSSL_KDF_PARAM_LABEL,
                       tls13_kdf_client_early_secret_label),
    ST_KAT_PARAM_END()
};

/*
 * NOTES:
 * According to FIPS 140-3 10.3.A Note18: SSH KDF is not required, since it is
 * sufficient to self-test the underlying SHA hash functions.
 */
static const ST_KAT_KDF st_kat_kdf_tests[] =
{
    {
        OSSL_SELF_TEST_DESC_KDF_TLS13_EXTRACT,
        OSSL_KDF_NAME_TLS1_3_KDF,
        tls13_kdf_early_secret_params,
        ITM(tls13_kdf_early_secret)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_TLS13_EXPAND,
        OSSL_KDF_NAME_TLS1_3_KDF,
        tls13_kdf_client_early_secret_params,
        ITM(tls13_kdf_client_early_traffic_secret)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_TLS12_PRF,
        OSSL_KDF_NAME_TLS1_PRF,
        tls12prf_params,
        ITM(tls12prf_expected)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_PBKDF2,
        OSSL_KDF_NAME_PBKDF2,
        pbkdf2_params,
        ITM(pbkdf2_expected)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_KBKDF,
        OSSL_KDF_NAME_KBKDF,
        kbkdf_params,
        ITM(kbkdf_expected)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_KBKDF_KMAC,
        OSSL_KDF_NAME_KBKDF,
        kbkdf_kmac_params,
        ITM(kbkdf_kmac_expected)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_HKDF,
        OSSL_KDF_NAME_HKDF,
        hkdf_params,
        ITM(hkdf_expected)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_SSKDF,
        OSSL_KDF_NAME_SSKDF,
        sskdf_params,
        ITM(sskdf_expected)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_X963KDF,
        OSSL_KDF_NAME_X963KDF,
        x963kdf_params,
        ITM(x963kdf_expected)
    },
    {
        OSSL_SELF_TEST_DESC_KDF_X942KDF,
        OSSL_KDF_NAME_X942KDF_ASN1,
        x942kdf_params,
        ITM(x942kdf_expected)
    },
};

/*-
* DRBG test vectors are a small subset of
* https://csrc.nist.rip/groups/STM/cavp/documents/drbg/drbgtestvectors.zip
* Using the folder drbgvectors_pr_true
* Generated for CAVS 14.3.
*/

/*
 * Hash_DRBG.rsp
 *
 * [SHA-256]
 * [PredictionResistance = True]
 * [EntropyInputLen = 256]
 * [NonceLen = 128]
 * [PersonalizationStringLen = 256]
 * [AdditionalInputLen = 256]
 * [ReturnedBitsLen = 1024]
 *
 * COUNT = 14
 */
static const unsigned char drbg_hash_sha256_pr_entropyin[] = {
    0x06, 0x6d, 0xc8, 0xce, 0x75, 0xb2, 0x89, 0x66, 0xa6, 0x85, 0x16, 0x3f,
    0xe2, 0xa4, 0xd4, 0x27, 0xfb, 0xdb, 0x61, 0x66, 0x50, 0x61, 0x6b, 0xa2,
    0x82, 0xfc, 0x33, 0x2b, 0x4e, 0x6f, 0x12, 0x20
};
static const unsigned char drbg_hash_sha256_pr_nonce[] = {
    0x55, 0x9f, 0x7c, 0x64, 0x89, 0x70, 0x83, 0xec, 0x2d, 0x73, 0x70, 0xd9,
    0xf0, 0xe5, 0x07, 0x1f
};
static const unsigned char drbg_hash_sha256_pr_persstr[] = {
    0x88, 0x6f, 0x54, 0x9a, 0xad, 0x1a, 0xc6, 0x3d, 0x18, 0xcb, 0xcc, 0x66,
    0x85, 0xda, 0xa2, 0xc2, 0xf7, 0x9e, 0xb0, 0x89, 0x4c, 0xb4, 0xae, 0xf1,
    0xac, 0x54, 0x4f, 0xce, 0x57, 0xf1, 0x5e, 0x11
};
static const unsigned char drbg_hash_sha256_pr_entropyinpr0[] = {
    0xff, 0x80, 0xb7, 0xd2, 0x6a, 0x05, 0xbc, 0x8a, 0x7a, 0xbe, 0x53, 0x28,
    0x6b, 0x0e, 0xeb, 0x73, 0x3b, 0x71, 0x5a, 0x20, 0x5b, 0xfa, 0x4f, 0xf6,
    0x37, 0x03, 0xde, 0xad, 0xb6, 0xea, 0x0e, 0xf4
};
static const unsigned char drbg_hash_sha256_pr_entropyinpr1[] = {
    0xc7, 0x38, 0x32, 0x53, 0x46, 0x81, 0xed, 0xe3, 0x7e, 0x03, 0x84, 0x6d,
    0x3c, 0x84, 0x17, 0x67, 0x29, 0x7d, 0x24, 0x6c, 0x68, 0x92, 0x41, 0xd2,
    0xe7, 0x75, 0xbe, 0x7e, 0xc9, 0x96, 0x29, 0x3d
};
static const unsigned char drbg_hash_sha256_pr_addin0[] = {
    0xb7, 0x21, 0x5f, 0x14, 0xac, 0x7b, 0xaf, 0xd0, 0xa9, 0x17, 0x72, 0xba,
    0x22, 0xf7, 0x19, 0xaf, 0xbd, 0x20, 0xb3, 0x11, 0x63, 0x6c, 0x2b, 0x1e,
    0x83, 0xe4, 0xa8, 0x23, 0x35, 0x3f, 0xc6, 0xea
};
static const unsigned char drbg_hash_sha256_pr_addin1[] = {
    0xce, 0xd3, 0x1f, 0x7e, 0x0d, 0xae, 0x5b, 0xb5, 0xc0, 0x43, 0xe2, 0x46,
    0xb2, 0x94, 0x73, 0xe2, 0xfd, 0x39, 0x51, 0x2e, 0xad, 0x45, 0x69, 0xee,
    0xe3, 0xe3, 0x80, 0x33, 0x14, 0xab, 0xa7, 0xa3
};
static const unsigned char drbg_hash_sha256_pr_expected[] = {
    0x60, 0xc2, 0x34, 0xcf, 0xaf, 0xb4, 0x68, 0x03, 0x3b, 0xf1, 0x95, 0xe5,
    0x78, 0xce, 0x26, 0x6e, 0x14, 0x65, 0x32, 0x6a, 0x96, 0xa9, 0xe0, 0x3f,
    0x8b, 0x89, 0x36, 0x70, 0xef, 0x62, 0x75, 0x4d, 0x5e, 0x80, 0xd5, 0x53,
    0xa1, 0xf8, 0x49, 0x50, 0x20, 0x8b, 0x93, 0x43, 0x07, 0x9f, 0x2e, 0xf8,
    0x56, 0xe9, 0xc5, 0x70, 0x61, 0x85, 0x97, 0xb5, 0xdc, 0x82, 0xa2, 0xda,
    0xea, 0xa3, 0xfd, 0x9b, 0x2f, 0xd2, 0xa0, 0xd7, 0x1b, 0xc6, 0x29, 0x35,
    0xcc, 0xb8, 0x3d, 0xa0, 0x67, 0x98, 0x05, 0xa0, 0xe3, 0x1e, 0xfe, 0xe4,
    0xf0, 0xe5, 0x13, 0xb0, 0x83, 0x17, 0xfa, 0xca, 0x93, 0x5e, 0x38, 0x29,
    0x48, 0xd2, 0x72, 0xdb, 0x76, 0x3e, 0x6d, 0xf3, 0x25, 0x10, 0xff, 0x1b,
    0x99, 0xff, 0xf8, 0xc6, 0x0e, 0xb0, 0xdd, 0x29, 0x2e, 0xbc, 0xbb, 0xc8,
    0x0a, 0x01, 0x6e, 0xd3, 0xb0, 0x0e, 0x4e, 0xab
};

/*
 * CTR_DRBG.rsp
 *
 * [AES-128 use df]
 * [PredictionResistance = True]
 * [EntropyInputLen = 128]
 * [NonceLen = 64]
 * [PersonalizationStringLen = 128]
 * [AdditionalInputLen = 128]
 * [ReturnedBitsLen = 512]
 *
 * COUNT = 0
 */
static const unsigned char drbg_ctr_aes128_pr_df_entropyin[] = {
    0x92, 0x89, 0x8f, 0x31, 0xfa, 0x1c, 0xff, 0x6d, 0x18, 0x2f, 0x26, 0x06,
    0x43, 0xdf, 0xf8, 0x18
};
static const unsigned char drbg_ctr_aes128_pr_df_nonce[] = {
    0xc2, 0xa4, 0xd9, 0x72, 0xc3, 0xb9, 0xb6, 0x97
};
static const unsigned char drbg_ctr_aes128_pr_df_persstr[] = {
    0xea, 0x65, 0xee, 0x60, 0x26, 0x4e, 0x7e, 0xb6, 0x0e, 0x82, 0x68, 0xc4,
    0x37, 0x3c, 0x5c, 0x0b
};
static const unsigned char drbg_ctr_aes128_pr_df_entropyinpr0[] = {
    0x20, 0x72, 0x8a, 0x06, 0xf8, 0x6f, 0x8d, 0xd4, 0x41, 0xe2, 0x72, 0xb7,
    0xc4, 0x2c, 0xe8, 0x10
};
static const unsigned char drbg_ctr_aes128_pr_df_entropyinpr1[] = {
    0x3d, 0xb0, 0xf0, 0x94, 0xf3, 0x05, 0x50, 0x33, 0x17, 0x86, 0x3e, 0x22,
    0x08, 0xf7, 0xa5, 0x01
};
static const unsigned char drbg_ctr_aes128_pr_df_addin0[] = {
    0x1a, 0x40, 0xfa, 0xe3, 0xcc, 0x6c, 0x7c, 0xa0, 0xf8, 0xda, 0xba, 0x59,
    0x23, 0x6d, 0xad, 0x1d
};
static const unsigned char drbg_ctr_aes128_pr_df_addin1[] = {
    0x9f, 0x72, 0x76, 0x6c, 0xc7, 0x46, 0xe5, 0xed, 0x2e, 0x53, 0x20, 0x12,
    0xbc, 0x59, 0x31, 0x8c
};
static const unsigned char drbg_ctr_aes128_pr_df_expected[] = {
    0x5a, 0x35, 0x39, 0x87, 0x0f, 0x4d, 0x22, 0xa4, 0x09, 0x24, 0xee, 0x71,
    0xc9, 0x6f, 0xac, 0x72, 0x0a, 0xd6, 0xf0, 0x88, 0x82, 0xd0, 0x83, 0x28,
    0x73, 0xec, 0x3f, 0x93, 0xd8, 0xab, 0x45, 0x23, 0xf0, 0x7e, 0xac, 0x45,
    0x14, 0x5e, 0x93, 0x9f, 0xb1, 0xd6, 0x76, 0x43, 0x3d, 0xb6, 0xe8, 0x08,
    0x88, 0xf6, 0xda, 0x89, 0x08, 0x77, 0x42, 0xfe, 0x1a, 0xf4, 0x3f, 0xc4,
    0x23, 0xc5, 0x1f, 0x68
};

/*
 * HMAC_DRBG.rsp
 *
 * [SHA-256]
 * [PredictionResistance = True]
 * [EntropyInputLen = 256]
 * [NonceLen = 128]
 * [PersonalizationStringLen = 256]
 * [AdditionalInputLen = 256]
 * [ReturnedBitsLen = 1024]
 *
 * COUNT = 0
 */
static const unsigned char drbg_hmac_sha2_pr_entropyin[] = {
    0xca, 0x85, 0x19, 0x11, 0x34, 0x93, 0x84, 0xbf,
    0xfe, 0x89, 0xde, 0x1c, 0xbd, 0xc4, 0x6e, 0x68,
    0x31, 0xe4, 0x4d, 0x34, 0xa4, 0xfb, 0x93, 0x5e,
    0xe2, 0x85, 0xdd, 0x14, 0xb7, 0x1a, 0x74, 0x88
};
static const unsigned char drbg_hmac_sha2_pr_nonce[] = {
    0x65, 0x9b, 0xa9, 0x6c, 0x60, 0x1d, 0xc6, 0x9f,
    0xc9, 0x02, 0x94, 0x08, 0x05, 0xec, 0x0c, 0xa8
};
static const unsigned char drbg_hmac_sha2_pr_persstr[] = {
    0xe7, 0x2d, 0xd8, 0x59, 0x0d, 0x4e, 0xd5, 0x29,
    0x55, 0x15, 0xc3, 0x5e, 0xd6, 0x19, 0x9e, 0x9d,
    0x21, 0x1b, 0x8f, 0x06, 0x9b, 0x30, 0x58, 0xca,
    0xa6, 0x67, 0x0b, 0x96, 0xef, 0x12, 0x08, 0xd0
};
static const unsigned char drbg_hmac_sha2_pr_entropyinpr0[] = {
    0x5c, 0xac, 0xc6, 0x81, 0x65, 0xa2, 0xe2, 0xee,
    0x20, 0x81, 0x2f, 0x35, 0xec, 0x73, 0xa7, 0x9d,
    0xbf, 0x30, 0xfd, 0x47, 0x54, 0x76, 0xac, 0x0c,
    0x44, 0xfc, 0x61, 0x74, 0xcd, 0xac, 0x2b, 0x55
};
static const unsigned char drbg_hmac_sha2_pr_entropyinpr1[] = {
    0x8d, 0xf0, 0x13, 0xb4, 0xd1, 0x03, 0x52, 0x30,
    0x73, 0x91, 0x7d, 0xdf, 0x6a, 0x86, 0x97, 0x93,
    0x05, 0x9e, 0x99, 0x43, 0xfc, 0x86, 0x54, 0x54,
    0x9e, 0x7a, 0xb2, 0x2f, 0x7c, 0x29, 0xf1, 0x22
};
static const unsigned char drbg_hmac_sha2_pr_addin0[] = {
    0x79, 0x3a, 0x7e, 0xf8, 0xf6, 0xf0, 0x48, 0x2b,
    0xea, 0xc5, 0x42, 0xbb, 0x78, 0x5c, 0x10, 0xf8,
    0xb7, 0xb4, 0x06, 0xa4, 0xde, 0x92, 0x66, 0x7a,
    0xb1, 0x68, 0xec, 0xc2, 0xcf, 0x75, 0x73, 0xc6
};
static const unsigned char drbg_hmac_sha2_pr_addin1[] = {
    0x22, 0x38, 0xcd, 0xb4, 0xe2, 0x3d, 0x62, 0x9f,
    0xe0, 0xc2, 0xa8, 0x3d, 0xd8, 0xd5, 0x14, 0x4c,
    0xe1, 0xa6, 0x22, 0x9e, 0xf4, 0x1d, 0xab, 0xe2,
    0xa9, 0x9f, 0xf7, 0x22, 0xe5, 0x10, 0xb5, 0x30
};
static const unsigned char drbg_hmac_sha2_pr_expected[] = {
    0xb1, 0xd1, 0x7c, 0x00, 0x2a, 0x7f, 0xeb, 0xd2,
    0x84, 0x12, 0xd8, 0xe5, 0x8a, 0x7f, 0x32, 0x31,
    0x8e, 0x4e, 0xe3, 0x60, 0x5a, 0x99, 0xb0, 0x5b,
    0x05, 0xd5, 0x93, 0x56, 0xd5, 0xf0, 0xc6, 0xb4,
    0x96, 0x0a, 0x4b, 0x8f, 0x96, 0x3b, 0x7e, 0xfa,
    0x55, 0xbb, 0x68, 0x72, 0xfb, 0xea, 0xc7, 0xb9,
    0x9b, 0x78, 0xde, 0xa8, 0xf3, 0x53, 0x19, 0x73,
    0x63, 0x7c, 0x94, 0x6a, 0x9c, 0xab, 0x33, 0x49,
    0x74, 0x4b, 0x24, 0xa0, 0x85, 0x1d, 0xd4, 0x7f,
    0x2b, 0x3b, 0x46, 0x0c, 0x2c, 0x61, 0x84, 0x6e,
    0x91, 0x18, 0x1d, 0x62, 0xd4, 0x2c, 0x60, 0xa4,
    0xef, 0xda, 0x5e, 0xd5, 0x79, 0x02, 0xbf, 0xd7,
    0x02, 0xb3, 0x49, 0xc5, 0x49, 0x52, 0xc7, 0xf6,
    0x44, 0x76, 0x9d, 0x8e, 0xf4, 0x01, 0x5e, 0xcc,
    0x5f, 0x5b, 0xbd, 0x4a, 0xf0, 0x61, 0x34, 0x68,
    0x8e, 0x30, 0x05, 0x0e, 0x04, 0x97, 0xfb, 0x0a
};

static const ST_KAT_DRBG st_kat_drbg_tests[] =
{
    {
        OSSL_SELF_TEST_DESC_DRBG_HASH,
        "HASH-DRBG", "digest", "SHA256",
        ITM(drbg_hash_sha256_pr_entropyin),
        ITM(drbg_hash_sha256_pr_nonce),
        ITM(drbg_hash_sha256_pr_persstr),
        ITM(drbg_hash_sha256_pr_entropyinpr0),
        ITM(drbg_hash_sha256_pr_entropyinpr1),
        ITM(drbg_hash_sha256_pr_addin0),
        ITM(drbg_hash_sha256_pr_addin1),
        ITM(drbg_hash_sha256_pr_expected)
    },
    {
        OSSL_SELF_TEST_DESC_DRBG_CTR,
        "CTR-DRBG", "cipher", "AES-128-CTR",
        ITM(drbg_ctr_aes128_pr_df_entropyin),
        ITM(drbg_ctr_aes128_pr_df_nonce),
        ITM(drbg_ctr_aes128_pr_df_persstr),
        ITM(drbg_ctr_aes128_pr_df_entropyinpr0),
        ITM(drbg_ctr_aes128_pr_df_entropyinpr1),
        ITM(drbg_ctr_aes128_pr_df_addin0),
        ITM(drbg_ctr_aes128_pr_df_addin1),
        ITM(drbg_ctr_aes128_pr_df_expected)
    },
    {
        OSSL_SELF_TEST_DESC_DRBG_HMAC,
        "HMAC-DRBG", "digest", "SHA256",
        ITM(drbg_hmac_sha2_pr_entropyin),
        ITM(drbg_hmac_sha2_pr_nonce),
        ITM(drbg_hmac_sha2_pr_persstr),
        ITM(drbg_hmac_sha2_pr_entropyinpr0),
        ITM(drbg_hmac_sha2_pr_entropyinpr1),
        ITM(drbg_hmac_sha2_pr_addin0),
        ITM(drbg_hmac_sha2_pr_addin1),
        ITM(drbg_hmac_sha2_pr_expected)
    }
};

/* KEY EXCHANGE TEST DATA */

#ifndef OPENSSL_NO_DH
/* DH KAT */
static const unsigned char dh_priv[] = {
    0x14, 0x33, 0xe0, 0xb5, 0xa9, 0x17, 0xb6, 0x0a,
    0x30, 0x23, 0xf2, 0xf8, 0xaa, 0x2c, 0x2d, 0x70,
    0xd2, 0x96, 0x8a, 0xba, 0x9a, 0xea, 0xc8, 0x15,
    0x40, 0xb8, 0xfc, 0xe6
};
static const unsigned char dh_pub[] = {
    0x00, 0x8f, 0x81, 0x67, 0x68, 0xce, 0x97, 0x99,
    0x7e, 0x11, 0x5c, 0xad, 0x5b, 0xe1, 0x0c, 0xd4,
    0x15, 0x44, 0xdf, 0xc2, 0x47, 0xe7, 0x06, 0x27,
    0x5e, 0xf3, 0x9d, 0x5c, 0x4b, 0x2e, 0x35, 0x05,
    0xfd, 0x3c, 0x8f, 0x35, 0x85, 0x1b, 0x82, 0xdd,
    0x49, 0xc9, 0xa8, 0x7e, 0x3a, 0x5f, 0x33, 0xdc,
    0x8f, 0x5e, 0x32, 0x76, 0xe1, 0x52, 0x1b, 0x88,
    0x85, 0xda, 0xa9, 0x1d, 0x5f, 0x1c, 0x05, 0x3a,
    0xd4, 0x8d, 0xbb, 0xe7, 0x46, 0x46, 0x1e, 0x29,
    0x4b, 0x5a, 0x02, 0x88, 0x46, 0x94, 0xd0, 0x68,
    0x7d, 0xb2, 0x9f, 0x3a, 0x3d, 0x82, 0x05, 0xe5,
    0xa7, 0xbe, 0x6c, 0x7e, 0x24, 0x35, 0x25, 0x14,
    0xf3, 0x45, 0x08, 0x90, 0xfc, 0x55, 0x2e, 0xa8,
    0xb8, 0xb1, 0x89, 0x15, 0x94, 0x51, 0x44, 0xa9,
    0x9f, 0x68, 0xcb, 0x90, 0xbc, 0xd3, 0xae, 0x02,
    0x37, 0x26, 0xe4, 0xe9, 0x1a, 0x90, 0x95, 0x7e,
    0x1d, 0xac, 0x0c, 0x91, 0x97, 0x83, 0x24, 0x83,
    0xb9, 0xa1, 0x40, 0x72, 0xac, 0xf0, 0x55, 0x32,
    0x18, 0xab, 0xb8, 0x90, 0xda, 0x13, 0x4a, 0xc8,
    0x4b, 0x7c, 0x18, 0xbc, 0x33, 0xbf, 0x99, 0x85,
    0x39, 0x3e, 0xc6, 0x95, 0x9b, 0x48, 0x8e, 0xbe,
    0x46, 0x59, 0x48, 0x41, 0x0d, 0x37, 0x25, 0x94,
    0xbe, 0x8d, 0xf5, 0x81, 0x52, 0xf6, 0xdc, 0xeb,
    0x98, 0xd7, 0x3b, 0x44, 0x61, 0x6f, 0xa3, 0xef,
    0x7b, 0xfe, 0xbb, 0xc2, 0x8e, 0x46, 0x63, 0xbc,
    0x52, 0x65, 0xf9, 0xf8, 0x85, 0x41, 0xdf, 0x82,
    0x4a, 0x10, 0x2a, 0xe3, 0x0c, 0xb7, 0xad, 0x84,
    0xa6, 0x6f, 0x4e, 0x8e, 0x96, 0x1e, 0x04, 0xf7,
    0x57, 0x39, 0xca, 0x58, 0xd4, 0xef, 0x5a, 0xf1,
    0xf5, 0x69, 0xc2, 0xb1, 0x5c, 0x0a, 0xce, 0xbe,
    0x38, 0x01, 0xb5, 0x3f, 0x07, 0x8a, 0x72, 0x90,
    0x10, 0xac, 0x51, 0x3a, 0x96, 0x43, 0xdf, 0x6f,
    0xea
};
static const unsigned char dh_peer_pub[] = {
    0x1f, 0xc1, 0xda, 0x34, 0x1d, 0x1a, 0x84, 0x6a,
    0x96, 0xb7, 0xbe, 0x24, 0x34, 0x0f, 0x87, 0x7d,
    0xd0, 0x10, 0xaa, 0x03, 0x56, 0xd5, 0xad, 0x58,
    0xaa, 0xe9, 0xc7, 0xb0, 0x8f, 0x74, 0x9a, 0x32,
    0x23, 0x51, 0x10, 0xb5, 0xd8, 0x8e, 0xb5, 0xdb,
    0xfa, 0x97, 0x8d, 0x27, 0xec, 0xc5, 0x30, 0xf0,
    0x2d, 0x31, 0x14, 0x00, 0x5b, 0x64, 0xb1, 0xc0,
    0xe0, 0x24, 0xcb, 0x8a, 0xe2, 0x16, 0x98, 0xbc,
    0xa9, 0xe6, 0x0d, 0x42, 0x80, 0x86, 0x22, 0xf1,
    0x81, 0xc5, 0x6e, 0x1d, 0xe7, 0xa9, 0x6e, 0x6e,
    0xfe, 0xe9, 0xd6, 0x65, 0x67, 0xe9, 0x1b, 0x97,
    0x70, 0x42, 0xc7, 0xe3, 0xd0, 0x44, 0x8f, 0x05,
    0xfb, 0x77, 0xf5, 0x22, 0xb9, 0xbf, 0xc8, 0xd3,
    0x3c, 0xc3, 0xc3, 0x1e, 0xd3, 0xb3, 0x1f, 0x0f,
    0xec, 0xb6, 0xdb, 0x4f, 0x6e, 0xa3, 0x11, 0xe7,
    0x7a, 0xfd, 0xbc, 0xd4, 0x7a, 0xee, 0x1b, 0xb1,
    0x50, 0xf2, 0x16, 0x87, 0x35, 0x78, 0xfb, 0x96,
    0x46, 0x8e, 0x8f, 0x9f, 0x3d, 0xe8, 0xef, 0xbf,
    0xce, 0x75, 0x62, 0x4b, 0x1d, 0xf0, 0x53, 0x22,
    0xa3, 0x4f, 0x14, 0x63, 0xe8, 0x39, 0xe8, 0x98,
    0x4c, 0x4a, 0xd0, 0xa9, 0x6e, 0x1a, 0xc8, 0x42,
    0xe5, 0x31, 0x8c, 0xc2, 0x3c, 0x06, 0x2a, 0x8c,
    0xa1, 0x71, 0xb8, 0xd5, 0x75, 0x98, 0x0d, 0xde,
    0x7f, 0xc5, 0x6f, 0x15, 0x36, 0x52, 0x38, 0x20,
    0xd4, 0x31, 0x92, 0xbf, 0xd5, 0x1e, 0x8e, 0x22,
    0x89, 0x78, 0xac, 0xa5, 0xb9, 0x44, 0x72, 0xf3,
    0x39, 0xca, 0xeb, 0x99, 0x31, 0xb4, 0x2b, 0xe3,
    0x01, 0x26, 0x8b, 0xc9, 0x97, 0x89, 0xc9, 0xb2,
    0x55, 0x71, 0xc3, 0xc0, 0xe4, 0xcb, 0x3f, 0x00,
    0x7f, 0x1a, 0x51, 0x1c, 0xbb, 0x53, 0xc8, 0x51,
    0x9c, 0xdd, 0x13, 0x02, 0xab, 0xca, 0x6c, 0x0f,
    0x34, 0xf9, 0x67, 0x39, 0xf1, 0x7f, 0xf4, 0x8b
};

static const unsigned char dh_secret_expected[256] = {
    0xa0, 0x38, 0x64, 0x37, 0xdf, 0x2d, 0x2c, 0x78,
    0x49, 0xb9, 0xa7, 0x77, 0xfb, 0xc1, 0x69, 0x94,
    0x85, 0xc5, 0x5a, 0xbc, 0x8d, 0x43, 0x32, 0x23,
    0x94, 0xf5, 0xba, 0xb4, 0x5f, 0x22, 0x4b, 0x4e,
    0xc4, 0xfd, 0x89, 0x41, 0x56, 0x41, 0xe8, 0x9f,
    0x2d, 0x0d, 0x26, 0x33, 0x60, 0x13, 0x8a, 0x20,
    0xf1, 0x7e, 0xb3, 0x76, 0x38, 0x03, 0x0e, 0x48,
    0x4f, 0x27, 0x8c, 0x32, 0xdb, 0x66, 0x5c, 0xbf,
    0x7f, 0xc7, 0xeb, 0xc6, 0x2d, 0xfd, 0x00, 0x08,
    0xb0, 0x98, 0x4e, 0xad, 0x68, 0x65, 0xca, 0x9e,
    0x78, 0xe1, 0xaa, 0xb7, 0x8e, 0x08, 0x4d, 0x67,
    0xa6, 0x15, 0x16, 0xbb, 0x41, 0xac, 0x15, 0xb5,
    0x08, 0x92, 0x5d, 0x25, 0x1d, 0x7f, 0xf3, 0x1b,
    0x5c, 0xea, 0x21, 0x6b, 0xe5, 0x00, 0x4d, 0xb6,
    0x8e, 0xae, 0x84, 0xb4, 0xee, 0xf7, 0xcc, 0xdd,
    0x64, 0x19, 0x4e, 0x25, 0xce, 0x37, 0x4f, 0xde,
    0xb6, 0x21, 0xba, 0xd9, 0xc0, 0x7a, 0x87, 0xc7,
    0x90, 0x0a, 0x78, 0x8b, 0xdd, 0xbc, 0x68, 0x77,
    0x2d, 0xa6, 0xdf, 0x4d, 0x2e, 0xca, 0xdc, 0x86,
    0xb6, 0x1e, 0x54, 0x2b, 0x3a, 0xa9, 0x52, 0x67,
    0xf3, 0x1a, 0x35, 0xb7, 0x5a, 0xcd, 0x99, 0x59,
    0xe9, 0x07, 0x6f, 0xd7, 0xd7, 0x96, 0x8a, 0x47,
    0xdf, 0x9f, 0x51, 0x1b, 0x04, 0xa9, 0x45, 0x30,
    0x89, 0x8a, 0x3f, 0x7e, 0xca, 0xfc, 0x05, 0x2d,
    0x18, 0x77, 0x8f, 0x45, 0x25, 0x39, 0xdb, 0xf2,
    0x13, 0x36, 0x31, 0xdb, 0x50, 0x65, 0x63, 0x4a,
    0xae, 0x3e, 0xd1, 0x3e, 0xde, 0xc1, 0x32, 0x4b,
    0x78, 0x19, 0x03, 0x70, 0x0a, 0xc2, 0xa2, 0x6f,
    0x9b, 0xd4, 0xa6, 0x1d, 0x47, 0xf2, 0xa6, 0x91,
    0x61, 0x4a, 0x74, 0xf8, 0x70, 0x39, 0x42, 0x72,
    0xd5, 0x58, 0x7f, 0xcd, 0x16, 0xeb, 0x82, 0x0c,
    0x2c, 0xf4, 0xd0, 0x95, 0x22, 0xf9, 0xbe, 0x99,
};

static const char dh_ffdhe2048[] = "ffdhe2048";
static const ST_KAT_PARAM dh_group[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_PKEY_PARAM_GROUP_NAME, dh_ffdhe2048),
    ST_KAT_PARAM_END()
};

/* The host's private key */
static const ST_KAT_PARAM dh_host_key[] = {
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_PUB_KEY, dh_pub),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_PRIV_KEY, dh_priv),
    ST_KAT_PARAM_END()
};

/* The peer's public key */
static const ST_KAT_PARAM dh_peer_key[] = {
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_PUB_KEY, dh_peer_pub),
    ST_KAT_PARAM_END()
};
#endif /* OPENSSL_NO_DH */


#ifndef OPENSSL_NO_EC
static const char ecdh_curve_name[] = "prime256v1";
static const unsigned char ecdh_privd[] = {
    0x33, 0xd0, 0x43, 0x83, 0xa9, 0x89, 0x56, 0x03,
    0xd2, 0xd7, 0xfe, 0x6b, 0x01, 0x6f, 0xe4, 0x59,
    0xcc, 0x0d, 0x9a, 0x24, 0x6c, 0x86, 0x1b, 0x2e,
    0xdc, 0x4b, 0x4d, 0x35, 0x43, 0xe1, 0x1b, 0xad
};
static const unsigned char ecdh_pub[] = {
    0x04,
    0x1b, 0x93, 0x67, 0x55, 0x1c, 0x55, 0x9f, 0x63,
    0xd1, 0x22, 0xa4, 0xd8, 0xd1, 0x0a, 0x60, 0x6d,
    0x02, 0xa5, 0x77, 0x57, 0xc8, 0xa3, 0x47, 0x73,
    0x3a, 0x6a, 0x08, 0x28, 0x39, 0xbd, 0xc9, 0xd2,
    0x80, 0xec, 0xe9, 0xa7, 0x08, 0x29, 0x71, 0x2f,
    0xc9, 0x56, 0x82, 0xee, 0x9a, 0x85, 0x0f, 0x6d,
    0x7f, 0x59, 0x5f, 0x8c, 0xd1, 0x96, 0x0b, 0xdf,
    0x29, 0x3e, 0x49, 0x07, 0x88, 0x3f, 0x9a, 0x29
};
static const unsigned char ecdh_peer_pub[] = {
    0x04,
    0x1f, 0x72, 0xbd, 0x2a, 0x3e, 0xeb, 0x6c, 0x76,
    0xe5, 0x5d, 0x69, 0x75, 0x24, 0xbf, 0x2f, 0x5b,
    0x96, 0xb2, 0x91, 0x62, 0x06, 0x35, 0xcc, 0xb2,
    0x4b, 0x31, 0x1b, 0x0c, 0x6f, 0x06, 0x9f, 0x86,
    0xcf, 0xc8, 0xac, 0xd5, 0x4f, 0x4d, 0x77, 0xf3,
    0x70, 0x4a, 0x8f, 0x04, 0x9a, 0xb1, 0x03, 0xc7,
    0xeb, 0xd5, 0x94, 0x78, 0x61, 0xab, 0x78, 0x0c,
    0x4a, 0x2d, 0x6b, 0xf3, 0x2f, 0x2e, 0x4a, 0xbc
};

static const ST_KAT_PARAM ecdh_group[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_PKEY_PARAM_GROUP_NAME, ecdh_curve_name),
    ST_KAT_PARAM_END()
};
static const ST_KAT_PARAM ecdh_host_key[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PUB_KEY, ecdh_pub),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_PRIV_KEY, ecdh_privd),
    ST_KAT_PARAM_END()
};
static const ST_KAT_PARAM ecdh_peer_key[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PUB_KEY, ecdh_peer_pub),
    ST_KAT_PARAM_END()
};
static const unsigned char ecdh_secret_expected[] = {
    0x45, 0x2a, 0x2f, 0x0d, 0x24, 0xe6, 0x8d, 0xd0,
    0xda, 0x59, 0x7b, 0x0c, 0xec, 0x9b, 0x4c, 0x38,
    0x41, 0xdd, 0xce, 0xb3, 0xcc, 0xf1, 0x90, 0x8e,
    0x30, 0xdb, 0x5b, 0x5f, 0x97, 0xea, 0xe0, 0xc2
};
#endif /* OPENSSL_NO_EC */

#if !defined(OPENSSL_NO_DH) || !defined(OPENSSL_NO_EC)
static const ST_KAT_KAS st_kat_kas_tests[] =
{
# ifndef OPENSSL_NO_DH
    {
        OSSL_SELF_TEST_DESC_KA_DH,
        "DH",
        dh_group,
        dh_host_key,
        dh_peer_key,
        ITM(dh_secret_expected)
    },
# endif /* OPENSSL_NO_DH */
# ifndef OPENSSL_NO_EC
    {
        OSSL_SELF_TEST_DESC_KA_ECDH,
        "EC",
        ecdh_group,
        ecdh_host_key,
        ecdh_peer_key,
        ITM(ecdh_secret_expected)
    },
# endif /* OPENSSL_NO_EC */
};
#endif /* !defined(OPENSSL_NO_DH) || !defined(OPENSSL_NO_EC) */

/* RSA key data */
static const unsigned char rsa_n[] = {
    0xDB, 0x10, 0x1A, 0xC2, 0xA3, 0xF1, 0xDC, 0xFF,
    0x13, 0x6B, 0xED, 0x44, 0xDF, 0xF0, 0x02, 0x6D,
    0x13, 0xC7, 0x88, 0xDA, 0x70, 0x6B, 0x54, 0xF1,
    0xE8, 0x27, 0xDC, 0xC3, 0x0F, 0x99, 0x6A, 0xFA,
    0xC6, 0x67, 0xFF, 0x1D, 0x1E, 0x3C, 0x1D, 0xC1,
    0xB5, 0x5F, 0x6C, 0xC0, 0xB2, 0x07, 0x3A, 0x6D,
    0x41, 0xE4, 0x25, 0x99, 0xAC, 0xFC, 0xD2, 0x0F,
    0x02, 0xD3, 0xD1, 0x54, 0x06, 0x1A, 0x51, 0x77,
    0xBD, 0xB6, 0xBF, 0xEA, 0xA7, 0x5C, 0x06, 0xA9,
    0x5D, 0x69, 0x84, 0x45, 0xD7, 0xF5, 0x05, 0xBA,
    0x47, 0xF0, 0x1B, 0xD7, 0x2B, 0x24, 0xEC, 0xCB,
    0x9B, 0x1B, 0x10, 0x8D, 0x81, 0xA0, 0xBE, 0xB1,
    0x8C, 0x33, 0xE4, 0x36, 0xB8, 0x43, 0xEB, 0x19,
    0x2A, 0x81, 0x8D, 0xDE, 0x81, 0x0A, 0x99, 0x48,
    0xB6, 0xF6, 0xBC, 0xCD, 0x49, 0x34, 0x3A, 0x8F,
    0x26, 0x94, 0xE3, 0x28, 0x82, 0x1A, 0x7C, 0x8F,
    0x59, 0x9F, 0x45, 0xE8, 0x5D, 0x1A, 0x45, 0x76,
    0x04, 0x56, 0x05, 0xA1, 0xD0, 0x1B, 0x8C, 0x77,
    0x6D, 0xAF, 0x53, 0xFA, 0x71, 0xE2, 0x67, 0xE0,
    0x9A, 0xFE, 0x03, 0xA9, 0x85, 0xD2, 0xC9, 0xAA,
    0xBA, 0x2A, 0xBC, 0xF4, 0xA0, 0x08, 0xF5, 0x13,
    0x98, 0x13, 0x5D, 0xF0, 0xD9, 0x33, 0x34, 0x2A,
    0x61, 0xC3, 0x89, 0x55, 0xF0, 0xAE, 0x1A, 0x9C,
    0x22, 0xEE, 0x19, 0x05, 0x8D, 0x32, 0xFE, 0xEC,
    0x9C, 0x84, 0xBA, 0xB7, 0xF9, 0x6C, 0x3A, 0x4F,
    0x07, 0xFC, 0x45, 0xEB, 0x12, 0xE5, 0x7B, 0xFD,
    0x55, 0xE6, 0x29, 0x69, 0xD1, 0xC2, 0xE8, 0xB9,
    0x78, 0x59, 0xF6, 0x79, 0x10, 0xC6, 0x4E, 0xEB,
    0x6A, 0x5E, 0xB9, 0x9A, 0xC7, 0xC4, 0x5B, 0x63,
    0xDA, 0xA3, 0x3F, 0x5E, 0x92, 0x7A, 0x81, 0x5E,
    0xD6, 0xB0, 0xE2, 0x62, 0x8F, 0x74, 0x26, 0xC2,
    0x0C, 0xD3, 0x9A, 0x17, 0x47, 0xE6, 0x8E, 0xAB
};
static const unsigned char rsa_e[] = { 0x01, 0x00, 0x01 };
static const unsigned char rsa_d[] = {
    0x52, 0x41, 0xF4, 0xDA, 0x7B, 0xB7, 0x59, 0x55,
    0xCA, 0xD4, 0x2F, 0x0F, 0x3A, 0xCB, 0xA4, 0x0D,
    0x93, 0x6C, 0xCC, 0x9D, 0xC1, 0xB2, 0xFB, 0xFD,
    0xAE, 0x40, 0x31, 0xAC, 0x69, 0x52, 0x21, 0x92,
    0xB3, 0x27, 0xDF, 0xEA, 0xEE, 0x2C, 0x82, 0xBB,
    0xF7, 0x40, 0x32, 0xD5, 0x14, 0xC4, 0x94, 0x12,
    0xEC, 0xB8, 0x1F, 0xCA, 0x59, 0xE3, 0xC1, 0x78,
    0xF3, 0x85, 0xD8, 0x47, 0xA5, 0xD7, 0x02, 0x1A,
    0x65, 0x79, 0x97, 0x0D, 0x24, 0xF4, 0xF0, 0x67,
    0x6E, 0x75, 0x2D, 0xBF, 0x10, 0x3D, 0xA8, 0x7D,
    0xEF, 0x7F, 0x60, 0xE4, 0xE6, 0x05, 0x82, 0x89,
    0x5D, 0xDF, 0xC6, 0xD2, 0x6C, 0x07, 0x91, 0x33,
    0x98, 0x42, 0xF0, 0x02, 0x00, 0x25, 0x38, 0xC5,
    0x85, 0x69, 0x8A, 0x7D, 0x2F, 0x95, 0x6C, 0x43,
    0x9A, 0xB8, 0x81, 0xE2, 0xD0, 0x07, 0x35, 0xAA,
    0x05, 0x41, 0xC9, 0x1E, 0xAF, 0xE4, 0x04, 0x3B,
    0x19, 0xB8, 0x73, 0xA2, 0xAC, 0x4B, 0x1E, 0x66,
    0x48, 0xD8, 0x72, 0x1F, 0xAC, 0xF6, 0xCB, 0xBC,
    0x90, 0x09, 0xCA, 0xEC, 0x0C, 0xDC, 0xF9, 0x2C,
    0xD7, 0xEB, 0xAE, 0xA3, 0xA4, 0x47, 0xD7, 0x33,
    0x2F, 0x8A, 0xCA, 0xBC, 0x5E, 0xF0, 0x77, 0xE4,
    0x97, 0x98, 0x97, 0xC7, 0x10, 0x91, 0x7D, 0x2A,
    0xA6, 0xFF, 0x46, 0x83, 0x97, 0xDE, 0xE9, 0xE2,
    0x17, 0x03, 0x06, 0x14, 0xE2, 0xD7, 0xB1, 0x1D,
    0x77, 0xAF, 0x51, 0x27, 0x5B, 0x5E, 0x69, 0xB8,
    0x81, 0xE6, 0x11, 0xC5, 0x43, 0x23, 0x81, 0x04,
    0x62, 0xFF, 0xE9, 0x46, 0xB8, 0xD8, 0x44, 0xDB,
    0xA5, 0xCC, 0x31, 0x54, 0x34, 0xCE, 0x3E, 0x82,
    0xD6, 0xBF, 0x7A, 0x0B, 0x64, 0x21, 0x6D, 0x88,
    0x7E, 0x5B, 0x45, 0x12, 0x1E, 0x63, 0x8D, 0x49,
    0xA7, 0x1D, 0xD9, 0x1E, 0x06, 0xCD, 0xE8, 0xBA,
    0x2C, 0x8C, 0x69, 0x32, 0xEA, 0xBE, 0x60, 0x71
};
static const unsigned char rsa_p[] = {
    0xFA, 0xAC, 0xE1, 0x37, 0x5E, 0x32, 0x11, 0x34,
    0xC6, 0x72, 0x58, 0x2D, 0x91, 0x06, 0x3E, 0x77,
    0xE7, 0x11, 0x21, 0xCD, 0x4A, 0xF8, 0xA4, 0x3F,
    0x0F, 0xEF, 0x31, 0xE3, 0xF3, 0x55, 0xA0, 0xB9,
    0xAC, 0xB6, 0xCB, 0xBB, 0x41, 0xD0, 0x32, 0x81,
    0x9A, 0x8F, 0x7A, 0x99, 0x30, 0x77, 0x6C, 0x68,
    0x27, 0xE2, 0x96, 0xB5, 0x72, 0xC9, 0xC3, 0xD4,
    0x42, 0xAA, 0xAA, 0xCA, 0x95, 0x8F, 0xFF, 0xC9,
    0x9B, 0x52, 0x34, 0x30, 0x1D, 0xCF, 0xFE, 0xCF,
    0x3C, 0x56, 0x68, 0x6E, 0xEF, 0xE7, 0x6C, 0xD7,
    0xFB, 0x99, 0xF5, 0x4A, 0xA5, 0x21, 0x1F, 0x2B,
    0xEA, 0x93, 0xE8, 0x98, 0x26, 0xC4, 0x6E, 0x42,
    0x21, 0x5E, 0xA0, 0xA1, 0x2A, 0x58, 0x35, 0xBB,
    0x10, 0xE7, 0xBA, 0x27, 0x0A, 0x3B, 0xB3, 0xAF,
    0xE2, 0x75, 0x36, 0x04, 0xAC, 0x56, 0xA0, 0xAB,
    0x52, 0xDE, 0xCE, 0xDD, 0x2C, 0x28, 0x77, 0x03
};
static const unsigned char rsa_q[] = {
    0xDF, 0xB7, 0x52, 0xB6, 0xD7, 0xC0, 0xE2, 0x96,
    0xE7, 0xC9, 0xFE, 0x5D, 0x71, 0x5A, 0xC4, 0x40,
    0x96, 0x2F, 0xE5, 0x87, 0xEA, 0xF3, 0xA5, 0x77,
    0x11, 0x67, 0x3C, 0x8D, 0x56, 0x08, 0xA7, 0xB5,
    0x67, 0xFA, 0x37, 0xA8, 0xB8, 0xCF, 0x61, 0xE8,
    0x63, 0xD8, 0x38, 0x06, 0x21, 0x2B, 0x92, 0x09,
    0xA6, 0x39, 0x3A, 0xEA, 0xA8, 0xB4, 0x45, 0x4B,
    0x36, 0x10, 0x4C, 0xE4, 0x00, 0x66, 0x71, 0x65,
    0xF8, 0x0B, 0x94, 0x59, 0x4F, 0x8C, 0xFD, 0xD5,
    0x34, 0xA2, 0xE7, 0x62, 0x84, 0x0A, 0xA7, 0xBB,
    0xDB, 0xD9, 0x8A, 0xCD, 0x05, 0xE1, 0xCC, 0x57,
    0x7B, 0xF1, 0xF1, 0x1F, 0x11, 0x9D, 0xBA, 0x3E,
    0x45, 0x18, 0x99, 0x1B, 0x41, 0x64, 0x43, 0xEE,
    0x97, 0x5D, 0x77, 0x13, 0x5B, 0x74, 0x69, 0x73,
    0x87, 0x95, 0x05, 0x07, 0xBE, 0x45, 0x07, 0x17,
    0x7E, 0x4A, 0x69, 0x22, 0xF3, 0xDB, 0x05, 0x39
};
static const unsigned char rsa_dp[] = {
    0x5E, 0xD8, 0xDC, 0xDA, 0x53, 0x44, 0xC4, 0x67,
    0xE0, 0x92, 0x51, 0x34, 0xE4, 0x83, 0xA5, 0x4D,
    0x3E, 0xDB, 0xA7, 0x9B, 0x82, 0xBB, 0x73, 0x81,
    0xFC, 0xE8, 0x77, 0x4B, 0x15, 0xBE, 0x17, 0x73,
    0x49, 0x9B, 0x5C, 0x98, 0xBC, 0xBD, 0x26, 0xEF,
    0x0C, 0xE9, 0x2E, 0xED, 0x19, 0x7E, 0x86, 0x41,
    0x1E, 0x9E, 0x48, 0x81, 0xDD, 0x2D, 0xE4, 0x6F,
    0xC2, 0xCD, 0xCA, 0x93, 0x9E, 0x65, 0x7E, 0xD5,
    0xEC, 0x73, 0xFD, 0x15, 0x1B, 0xA2, 0xA0, 0x7A,
    0x0F, 0x0D, 0x6E, 0xB4, 0x53, 0x07, 0x90, 0x92,
    0x64, 0x3B, 0x8B, 0xA9, 0x33, 0xB3, 0xC5, 0x94,
    0x9B, 0x4C, 0x5D, 0x9C, 0x7C, 0x46, 0xA4, 0xA5,
    0x56, 0xF4, 0xF3, 0xF8, 0x27, 0x0A, 0x7B, 0x42,
    0x0D, 0x92, 0x70, 0x47, 0xE7, 0x42, 0x51, 0xA9,
    0xC2, 0x18, 0xB1, 0x58, 0xB1, 0x50, 0x91, 0xB8,
    0x61, 0x41, 0xB6, 0xA9, 0xCE, 0xD4, 0x7C, 0xBB
};
static const unsigned char rsa_dq[] = {
    0x54, 0x09, 0x1F, 0x0F, 0x03, 0xD8, 0xB6, 0xC5,
    0x0C, 0xE8, 0xB9, 0x9E, 0x0C, 0x38, 0x96, 0x43,
    0xD4, 0xA6, 0xC5, 0x47, 0xDB, 0x20, 0x0E, 0xE5,
    0xBD, 0x29, 0xD4, 0x7B, 0x1A, 0xF8, 0x41, 0x57,
    0x49, 0x69, 0x9A, 0x82, 0xCC, 0x79, 0x4A, 0x43,
    0xEB, 0x4D, 0x8B, 0x2D, 0xF2, 0x43, 0xD5, 0xA5,
    0xBE, 0x44, 0xFD, 0x36, 0xAC, 0x8C, 0x9B, 0x02,
    0xF7, 0x9A, 0x03, 0xE8, 0x19, 0xA6, 0x61, 0xAE,
    0x76, 0x10, 0x93, 0x77, 0x41, 0x04, 0xAB, 0x4C,
    0xED, 0x6A, 0xCC, 0x14, 0x1B, 0x99, 0x8D, 0x0C,
    0x6A, 0x37, 0x3B, 0x86, 0x6C, 0x51, 0x37, 0x5B,
    0x1D, 0x79, 0xF2, 0xA3, 0x43, 0x10, 0xC6, 0xA7,
    0x21, 0x79, 0x6D, 0xF9, 0xE9, 0x04, 0x6A, 0xE8,
    0x32, 0xFF, 0xAE, 0xFD, 0x1C, 0x7B, 0x8C, 0x29,
    0x13, 0xA3, 0x0C, 0xB2, 0xAD, 0xEC, 0x6C, 0x0F,
    0x8D, 0x27, 0x12, 0x7B, 0x48, 0xB2, 0xDB, 0x31
};
static const unsigned char rsa_qInv[] = {
    0x8D, 0x1B, 0x05, 0xCA, 0x24, 0x1F, 0x0C, 0x53,
    0x19, 0x52, 0x74, 0x63, 0x21, 0xFA, 0x78, 0x46,
    0x79, 0xAF, 0x5C, 0xDE, 0x30, 0xA4, 0x6C, 0x20,
    0x38, 0xE6, 0x97, 0x39, 0xB8, 0x7A, 0x70, 0x0D,
    0x8B, 0x6C, 0x6D, 0x13, 0x74, 0xD5, 0x1C, 0xDE,
    0xA9, 0xF4, 0x60, 0x37, 0xFE, 0x68, 0x77, 0x5E,
    0x0B, 0x4E, 0x5E, 0x03, 0x31, 0x30, 0xDF, 0xD6,
    0xAE, 0x85, 0xD0, 0x81, 0xBB, 0x61, 0xC7, 0xB1,
    0x04, 0x5A, 0xC4, 0x6D, 0x56, 0x1C, 0xD9, 0x64,
    0xE7, 0x85, 0x7F, 0x88, 0x91, 0xC9, 0x60, 0x28,
    0x05, 0xE2, 0xC6, 0x24, 0x8F, 0xDD, 0x61, 0x64,
    0xD8, 0x09, 0xDE, 0x7E, 0xD3, 0x4A, 0x61, 0x1A,
    0xD3, 0x73, 0x58, 0x4B, 0xD8, 0xA0, 0x54, 0x25,
    0x48, 0x83, 0x6F, 0x82, 0x6C, 0xAF, 0x36, 0x51,
    0x2A, 0x5D, 0x14, 0x2F, 0x41, 0x25, 0x00, 0xDD,
    0xF8, 0xF3, 0x95, 0xFE, 0x31, 0x25, 0x50, 0x12
};

static const ST_KAT_PARAM rsa_crt_key[] = {
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_N, rsa_n),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_E, rsa_e),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_D, rsa_d),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_FACTOR1, rsa_p),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_FACTOR2, rsa_q),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_EXPONENT1, rsa_dp),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_EXPONENT2, rsa_dq),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_COEFFICIENT1, rsa_qInv),
    ST_KAT_PARAM_END()
};

static const ST_KAT_PARAM rsa_pub_key[] = {
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_N, rsa_n),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_E, rsa_e),
    ST_KAT_PARAM_END()
};

static const ST_KAT_PARAM rsa_priv_key[] = {
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_N, rsa_n),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_E, rsa_e),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_RSA_D, rsa_d),
    ST_KAT_PARAM_END()
};

static const unsigned char rsa_sig_msg[] = "Hello World!";

static const unsigned char rsa_expected_sig[256] = {
    0xad, 0xbe, 0x2a, 0xaf, 0x16, 0x85, 0xc5, 0x00,
    0x91, 0x3e, 0xd0, 0x49, 0xfb, 0x3a, 0x81, 0xb9,
    0x6c, 0x28, 0xbc, 0xbf, 0xea, 0x96, 0x5f, 0xe4,
    0x9f, 0x99, 0xf7, 0x18, 0x8c, 0xec, 0x60, 0x28,
    0xeb, 0x29, 0x02, 0x49, 0xfc, 0xda, 0xd7, 0x78,
    0x68, 0xf8, 0xe1, 0xe9, 0x4d, 0x20, 0x6d, 0x32,
    0xa6, 0xde, 0xfc, 0xe4, 0xda, 0xcc, 0x6c, 0x75,
    0x36, 0x6b, 0xff, 0x5a, 0xac, 0x01, 0xa8, 0xc2,
    0xa9, 0xe6, 0x8b, 0x18, 0x3e, 0xec, 0xea, 0x4c,
    0x4a, 0x9e, 0x00, 0x09, 0xd1, 0x8a, 0x69, 0x1b,
    0x8b, 0xd9, 0xad, 0x37, 0xe5, 0x7c, 0xff, 0x7d,
    0x59, 0x56, 0x3e, 0xa0, 0xc6, 0x32, 0xd8, 0x35,
    0x2f, 0xff, 0xfb, 0x05, 0x02, 0xcd, 0xd7, 0x19,
    0xb9, 0x00, 0x86, 0x2a, 0xcf, 0xaa, 0x78, 0x16,
    0x4b, 0xf1, 0xa7, 0x59, 0xef, 0x7d, 0xe8, 0x74,
    0x23, 0x5c, 0xb2, 0xd4, 0x8a, 0x99, 0xa5, 0xbc,
    0xfa, 0x63, 0xd8, 0xf7, 0xbd, 0xc6, 0x00, 0x13,
    0x06, 0x02, 0x9a, 0xd4, 0xa7, 0xb4, 0x3d, 0x61,
    0xab, 0xf1, 0xc2, 0x95, 0x59, 0x9b, 0x3d, 0x67,
    0x1f, 0xde, 0x57, 0xb6, 0xb6, 0x9f, 0xb0, 0x87,
    0xd6, 0x51, 0xd5, 0x3e, 0x00, 0xe2, 0xc9, 0xa0,
    0x03, 0x66, 0xbc, 0x01, 0xb3, 0x8e, 0xfa, 0xf1,
    0x15, 0xeb, 0x26, 0xf1, 0x5d, 0x81, 0x90, 0xb4,
    0x1c, 0x00, 0x7c, 0x83, 0x4a, 0xa5, 0xde, 0x64,
    0xae, 0xea, 0x6c, 0x43, 0xc3, 0x20, 0x77, 0x77,
    0x42, 0x12, 0x24, 0xf5, 0xe3, 0x70, 0xdd, 0x59,
    0x48, 0x9c, 0xef, 0xd4, 0x8a, 0x3c, 0x29, 0x6a,
    0x0c, 0x9c, 0xf2, 0x13, 0xa4, 0x1c, 0x2f, 0x49,
    0xcd, 0xb4, 0xaa, 0x28, 0x40, 0x34, 0xc6, 0x75,
    0xba, 0x30, 0xe6, 0xd8, 0x5b, 0x2f, 0x08, 0xd0,
    0x29, 0xa5, 0x39, 0xfb, 0x6e, 0x3b, 0x0f, 0x52,
    0x2c, 0x68, 0xf0, 0x37, 0xa9, 0xd2, 0x56, 0xd6
};

static const unsigned char rsa_asym_plaintext_encrypt[256] = {
   0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
   0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
};
static const unsigned char rsa_asym_expected_encrypt[256] = {
    0x54, 0xac, 0x23, 0x96, 0x1d, 0x82, 0x5d, 0x8b,
    0x8f, 0x36, 0x33, 0xd0, 0xf4, 0x02, 0xa2, 0x61,
    0xb1, 0x13, 0xd4, 0x4a, 0x46, 0x06, 0x37, 0x3c,
    0xbf, 0x40, 0x05, 0x3c, 0xc6, 0x3b, 0x64, 0xdc,
    0x22, 0x22, 0xaf, 0x36, 0x79, 0x62, 0x45, 0xf0,
    0x97, 0x82, 0x22, 0x44, 0x86, 0x4a, 0x7c, 0xfa,
    0xac, 0x03, 0x21, 0x84, 0x3f, 0x31, 0xad, 0x2a,
    0xa4, 0x6e, 0x7a, 0xc5, 0x93, 0xf3, 0x0f, 0xfc,
    0xf1, 0x62, 0xce, 0x82, 0x12, 0x45, 0xc9, 0x35,
    0xb0, 0x7a, 0xcd, 0x99, 0x8c, 0x91, 0x6b, 0x5a,
    0xd3, 0x46, 0xdb, 0xf9, 0x9e, 0x52, 0x49, 0xbd,
    0x1e, 0xe8, 0xda, 0xac, 0x61, 0x47, 0xc2, 0xda,
    0xfc, 0x1e, 0xfb, 0x74, 0xd7, 0xd6, 0xc1, 0x18,
    0x86, 0x3e, 0x20, 0x9c, 0x7a, 0xe1, 0x04, 0xb7,
    0x38, 0x43, 0xb1, 0x4e, 0xa0, 0xd8, 0xc1, 0x39,
    0x4d, 0xe1, 0xd3, 0xb0, 0xb3, 0xf1, 0x82, 0x87,
    0x1f, 0x74, 0xb5, 0x69, 0xfd, 0x33, 0xd6, 0x21,
    0x7c, 0x61, 0x60, 0x28, 0xca, 0x70, 0xdb, 0xa0,
    0xbb, 0xc8, 0x73, 0xa9, 0x82, 0xf8, 0x6b, 0xd8,
    0xf0, 0xc9, 0x7b, 0x20, 0xdf, 0x9d, 0xfb, 0x8c,
    0xd4, 0xa2, 0x89, 0xe1, 0x9b, 0x04, 0xad, 0xaa,
    0x11, 0x6c, 0x8f, 0xce, 0x83, 0x29, 0x56, 0x69,
    0xbb, 0x00, 0x3b, 0xef, 0xca, 0x2d, 0xcd, 0x52,
    0xc8, 0xf1, 0xb3, 0x9b, 0xb4, 0x4f, 0x6d, 0x9c,
    0x3d, 0x69, 0xcc, 0x6d, 0x1f, 0x38, 0x4d, 0xe6,
    0xbb, 0x0c, 0x87, 0xdc, 0x5f, 0xa9, 0x24, 0x93,
    0x03, 0x46, 0xa2, 0x33, 0x6c, 0xf4, 0xd8, 0x5d,
    0x68, 0xf3, 0xd3, 0xe0, 0xf2, 0x30, 0xdb, 0xf5,
    0x4f, 0x0f, 0xad, 0xc7, 0xd0, 0xaa, 0x47, 0xd9,
    0x9f, 0x85, 0x1b, 0x2e, 0x6c, 0x3c, 0x57, 0x04,
    0x29, 0xf4, 0xf5, 0x66, 0x7d, 0x93, 0x4a, 0xaa,
    0x05, 0x52, 0x55, 0xc1, 0xc6, 0x06, 0x90, 0xab,
};

#ifndef OPENSSL_NO_EC
/* ECDSA key data */
static const char ecd_prime_curve_name[] = "secp224r1";
static const unsigned char ecd_prime_priv[] = {
    0x98, 0x1f, 0xb5, 0xf1, 0xfc, 0x87, 0x1d, 0x7d,
    0xde, 0x1e, 0x01, 0x64, 0x09, 0x9b, 0xe7, 0x1b,
    0x9f, 0xad, 0x63, 0xdd, 0x33, 0x01, 0xd1, 0x50,
    0x80, 0x93, 0x50, 0x30
};
static const unsigned char ecd_prime_pub[] = {
    0x04, 0x95, 0x47, 0x99, 0x44, 0x29, 0x8f, 0x51,
    0x39, 0xe2, 0x53, 0xec, 0x79, 0xb0, 0x4d, 0xde,
    0x87, 0x1a, 0x76, 0x54, 0xd5, 0x96, 0xb8, 0x7a,
    0x6d, 0xf4, 0x1c, 0x2c, 0x87, 0x91, 0x5f, 0xd5,
    0x31, 0xdd, 0x24, 0xe5, 0x78, 0xd9, 0x08, 0x24,
    0x8a, 0x49, 0x99, 0xec, 0x55, 0xf2, 0x82, 0xb3,
    0xc4, 0xb7, 0x33, 0x68, 0xe4, 0x24, 0xa9, 0x12,
    0x82
};
static const unsigned char ecdsa_prime_expected_sig[] = {
    0x30, 0x3d, 0x02, 0x1c, 0x48, 0x4f, 0x3c, 0x97,
    0x5b, 0xfa, 0x40, 0x6c, 0xdb, 0xd6, 0x70, 0xb5,
    0xbd, 0x2d, 0xd0, 0xc6, 0x22, 0x93, 0x5a, 0x88,
    0x56, 0xd0, 0xaf, 0x0a, 0x94, 0x92, 0x20, 0x01,
    0x02, 0x1d, 0x00, 0xa4, 0x80, 0xe0, 0x47, 0x88,
    0x8a, 0xef, 0x2a, 0x47, 0x9d, 0x81, 0x9a, 0xbf,
    0x45, 0xc3, 0x6f, 0x9e, 0x2e, 0xc1, 0x44, 0x9f,
    0xfd, 0x79, 0xdb, 0x90, 0x3e, 0xb9, 0xb2
};
static const ST_KAT_PARAM ecdsa_prime_key[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_PKEY_PARAM_GROUP_NAME, ecd_prime_curve_name),
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PUB_KEY, ecd_prime_pub),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_PRIV_KEY, ecd_prime_priv),
    ST_KAT_PARAM_END()
};

# ifndef OPENSSL_NO_EC2M
static const char ecd_bin_curve_name[] = "sect233r1";
static const unsigned char ecd_bin_priv[] = {
    0x00, 0x6d, 0xd6, 0x39, 0x9d, 0x2a, 0xa2, 0xc8,
    0x8c, 0xfc, 0x7b, 0x80, 0x66, 0xaa, 0xe1, 0xaa,
    0xba, 0xee, 0xcb, 0xfd, 0xc9, 0xe5, 0x36, 0x38,
    0x2e, 0xf7, 0x37, 0x6d, 0xd3, 0x20
};
static const unsigned char ecd_bin_pub[] = {
    0x04, 0x00, 0x06, 0xe2, 0x56, 0xf7, 0x37, 0xf9,
    0xea, 0xb6, 0xd1, 0x0f, 0x59, 0xfa, 0x23, 0xc3,
    0x93, 0xa8, 0xb2, 0x26, 0xe2, 0x5c, 0x08, 0xbe,
    0x63, 0x49, 0x26, 0xdc, 0xc7, 0x1e, 0x6f, 0x01,
    0x32, 0x3b, 0xe6, 0x54, 0x8d, 0xc1, 0x13, 0x3e,
    0x54, 0xb2, 0x66, 0x89, 0xb2, 0x82, 0x0a, 0x72,
    0x02, 0xa8, 0xe9, 0x6f, 0x54, 0xfd, 0x3a, 0x6b,
    0x99, 0xb6, 0x8f, 0x80, 0x46
};
static const unsigned char ecdsa_bin_expected_sig[] = {
    0x30, 0x3f, 0x02, 0x1d, 0x58, 0xe9, 0xd0, 0x84,
    0x5c, 0xad, 0x29, 0x03, 0xf6, 0xa6, 0xbc, 0xe0,
    0x24, 0x6d, 0x9e, 0x79, 0x5d, 0x1e, 0xe8, 0x5a,
    0xc3, 0x31, 0x0a, 0xa9, 0xfb, 0xe3, 0x99, 0x54,
    0x11, 0x02, 0x1e, 0x00, 0xa3, 0x44, 0x28, 0xa3,
    0x70, 0x97, 0x98, 0x17, 0xd7, 0xa6, 0xad, 0x91,
    0xaf, 0x41, 0x69, 0xb6, 0x06, 0x99, 0x39, 0xc7,
    0x63, 0xa4, 0x6a, 0x81, 0xe4, 0x9a, 0x9d, 0x15,
    0x8b
};
static const ST_KAT_PARAM ecdsa_bin_key[] = {
    ST_KAT_PARAM_UTF8STRING(OSSL_PKEY_PARAM_GROUP_NAME, ecd_bin_curve_name),
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PUB_KEY, ecd_bin_pub),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_PRIV_KEY, ecd_bin_priv),
    ST_KAT_PARAM_END()
};
# endif /* OPENSSL_NO_EC2M */

# ifndef OPENSSL_NO_ECX
static const unsigned char ecx_sig_msg[] = {
    0x64, 0xa6, 0x5f, 0x3c, 0xde, 0xdc, 0xdd, 0x66,
    0x81, 0x1e, 0x29, 0x15
};
static const unsigned char ed25519_pub[] = {
    0xfc, 0x51, 0xcd, 0x8e, 0x62, 0x18, 0xa1, 0xa3,
    0x8d, 0xa4, 0x7e, 0xd0, 0x02, 0x30, 0xf0, 0x58,
    0x08, 0x16, 0xed, 0x13, 0xba, 0x33, 0x03, 0xac,
    0x5d, 0xeb, 0x91, 0x15, 0x48, 0x90, 0x80, 0x25
};
static const unsigned char ed25519_priv[] = {
    0xc5, 0xaa, 0x8d, 0xf4, 0x3f, 0x9f, 0x83, 0x7b,
    0xed, 0xb7, 0x44, 0x2f, 0x31, 0xdc, 0xb7, 0xb1,
    0x66, 0xd3, 0x85, 0x35, 0x07, 0x6f, 0x09, 0x4b,
    0x85, 0xce, 0x3a, 0x2e, 0x0b, 0x44, 0x58, 0xf7
};
static const ST_KAT_PARAM ed25519_key[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PUB_KEY, ed25519_pub),
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PRIV_KEY, ed25519_priv),
    ST_KAT_PARAM_END()
};
static const unsigned char ed25519_expected_sig[] = {
    0x1e, 0xf4, 0xc5, 0x61, 0xdc, 0x97, 0x9f, 0xaf,
    0x55, 0x6b, 0x46, 0xa1, 0xae, 0xb0, 0x64, 0x13,
    0x1c, 0x98, 0x09, 0x96, 0x88, 0xe0, 0x9d, 0x0e,
    0x4e, 0x7d, 0xc4, 0xa5, 0xa1, 0x91, 0x09, 0xca,
    0xd9, 0x5b, 0x4f, 0x1c, 0x80, 0x82, 0x9f, 0x65,
    0xc1, 0x41, 0xa4, 0xe8, 0x02, 0x05, 0x0c, 0xa6,
    0x7e, 0xa0, 0xfa, 0x01, 0xee, 0xeb, 0xaa, 0x91,
    0x62, 0xfd, 0x0f, 0x25, 0xa0, 0x2d, 0x37, 0x09
};

static const unsigned char ed448_pub[] = {
    0x3b, 0xa1, 0x6d, 0xa0, 0xc6, 0xf2, 0xcc, 0x1f,
    0x30, 0x18, 0x77, 0x40, 0x75, 0x6f, 0x5e, 0x79,
    0x8d, 0x6b, 0xc5, 0xfc, 0x01, 0x5d, 0x7c, 0x63,
    0xcc, 0x95, 0x10, 0xee, 0x3f, 0xd4, 0x4a, 0xdc,
    0x24, 0xd8, 0xe9, 0x68, 0xb6, 0xe4, 0x6e, 0x6f,
    0x94, 0xd1, 0x9b, 0x94, 0x53, 0x61, 0x72, 0x6b,
    0xd7, 0x5e, 0x14, 0x9e, 0xf0, 0x98, 0x17, 0xf5,
    0x80
};
static const unsigned char ed448_priv[] = {
   0x25, 0x8c, 0xdd, 0x4a, 0xda, 0x32, 0xed, 0x9c,
   0x9f, 0xf5, 0x4e, 0x63, 0x75, 0x6a, 0xe5, 0x82,
   0xfb, 0x8f, 0xab, 0x2a, 0xc7, 0x21, 0xf2, 0xc8,
   0xe6, 0x76, 0xa7, 0x27, 0x68, 0x51, 0x3d, 0x93,
   0x9f, 0x63, 0xdd, 0xdb, 0x55, 0x60, 0x91, 0x33,
   0xf2, 0x9a, 0xdf, 0x86, 0xec, 0x99, 0x29, 0xdc,
   0xcb, 0x52, 0xc1, 0xc5, 0xfd, 0x2f, 0xf7, 0xe2,
   0x1b
};
static const ST_KAT_PARAM ed448_key[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PUB_KEY, ed448_pub),
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PRIV_KEY, ed448_priv),
    ST_KAT_PARAM_END()
};
static const unsigned char ed448_expected_sig[] = {
   0x7e, 0xee, 0xab, 0x7c, 0x4e, 0x50, 0xfb, 0x79,
   0x9b, 0x41, 0x8e, 0xe5, 0xe3, 0x19, 0x7f, 0xf6,
   0xbf, 0x15, 0xd4, 0x3a, 0x14, 0xc3, 0x43, 0x89,
   0xb5, 0x9d, 0xd1, 0xa7, 0xb1, 0xb8, 0x5b, 0x4a,
   0xe9, 0x04, 0x38, 0xac, 0xa6, 0x34, 0xbe, 0xa4,
   0x5e, 0x3a, 0x26, 0x95, 0xf1, 0x27, 0x0f, 0x07,
   0xfd, 0xcd, 0xf7, 0xc6, 0x2b, 0x8e, 0xfe, 0xaf,
   0x00, 0xb4, 0x5c, 0x2c, 0x96, 0xba, 0x45, 0x7e,
   0xb1, 0xa8, 0xbf, 0x07, 0x5a, 0x3d, 0xb2, 0x8e,
   0x5c, 0x24, 0xf6, 0xb9, 0x23, 0xed, 0x4a, 0xd7,
   0x47, 0xc3, 0xc9, 0xe0, 0x3c, 0x70, 0x79, 0xef,
   0xb8, 0x7c, 0xb1, 0x10, 0xd3, 0xa9, 0x98, 0x61,
   0xe7, 0x20, 0x03, 0xcb, 0xae, 0x6d, 0x6b, 0x8b,
   0x82, 0x7e, 0x4e, 0x6c, 0x14, 0x30, 0x64, 0xff,
   0x3c, 0x00
};
# endif /* OPENSSL_NO_ECX */
#endif /* OPENSSL_NO_EC */

#ifndef OPENSSL_NO_DSA
/* dsa 2048 */
static const unsigned char dsa_p[] = {
    0xa2, 0x9b, 0x88, 0x72, 0xce, 0x8b, 0x84, 0x23,
    0xb7, 0xd5, 0xd2, 0x1d, 0x4b, 0x02, 0xf5, 0x7e,
    0x03, 0xe9, 0xe6, 0xb8, 0xa2, 0x58, 0xdc, 0x16,
    0x61, 0x1b, 0xa0, 0x98, 0xab, 0x54, 0x34, 0x15,
    0xe4, 0x15, 0xf1, 0x56, 0x99, 0x7a, 0x3e, 0xe2,
    0x36, 0x65, 0x8f, 0xa0, 0x93, 0x26, 0x0d, 0xe3,
    0xad, 0x42, 0x2e, 0x05, 0xe0, 0x46, 0xf9, 0xec,
    0x29, 0x16, 0x1a, 0x37, 0x5f, 0x0e, 0xb4, 0xef,
    0xfc, 0xef, 0x58, 0x28, 0x5c, 0x5d, 0x39, 0xed,
    0x42, 0x5d, 0x7a, 0x62, 0xca, 0x12, 0x89, 0x6c,
    0x4a, 0x92, 0xcb, 0x19, 0x46, 0xf2, 0x95, 0x2a,
    0x48, 0x13, 0x3f, 0x07, 0xda, 0x36, 0x4d, 0x1b,
    0xdf, 0x6b, 0x0f, 0x71, 0x39, 0x98, 0x3e, 0x69,
    0x3c, 0x80, 0x05, 0x9b, 0x0e, 0xac, 0xd1, 0x47,
    0x9b, 0xa9, 0xf2, 0x85, 0x77, 0x54, 0xed, 0xe7,
    0x5f, 0x11, 0x2b, 0x07, 0xeb, 0xbf, 0x35, 0x34,
    0x8b, 0xbf, 0x3e, 0x01, 0xe0, 0x2f, 0x2d, 0x47,
    0x3d, 0xe3, 0x94, 0x53, 0xf9, 0x9d, 0xd2, 0x36,
    0x75, 0x41, 0xca, 0xca, 0x3b, 0xa0, 0x11, 0x66,
    0x34, 0x3d, 0x7b, 0x5b, 0x58, 0xa3, 0x7b, 0xd1,
    0xb7, 0x52, 0x1d, 0xb2, 0xf1, 0x3b, 0x86, 0x70,
    0x71, 0x32, 0xfe, 0x09, 0xf4, 0xcd, 0x09, 0xdc,
    0x16, 0x18, 0xfa, 0x34, 0x01, 0xeb, 0xf9, 0xcc,
    0x7b, 0x19, 0xfa, 0x94, 0xaa, 0x47, 0x20, 0x88,
    0x13, 0x3d, 0x6c, 0xb2, 0xd3, 0x5c, 0x11, 0x79,
    0xc8, 0xc8, 0xff, 0x36, 0x87, 0x58, 0xd5, 0x07,
    0xd9, 0xf9, 0xa1, 0x7d, 0x46, 0xc1, 0x10, 0xfe,
    0x31, 0x44, 0xce, 0x9b, 0x02, 0x2b, 0x42, 0xe4,
    0x19, 0xeb, 0x4f, 0x53, 0x88, 0x61, 0x3b, 0xfc,
    0x3e, 0x26, 0x24, 0x1a, 0x43, 0x2e, 0x87, 0x06,
    0xbc, 0x58, 0xef, 0x76, 0x11, 0x72, 0x78, 0xde,
    0xab, 0x6c, 0xf6, 0x92, 0x61, 0x82, 0x91, 0xb7
};
static const unsigned char dsa_q[] = {
    0xa3, 0xbf, 0xd9, 0xab, 0x78, 0x84, 0x79, 0x4e,
    0x38, 0x34, 0x50, 0xd5, 0x89, 0x1d, 0xc1, 0x8b,
    0x65, 0x15, 0x7b, 0xdc, 0xfc, 0xda, 0xc5, 0x15,
    0x18, 0x90, 0x28, 0x67
};
static const unsigned char dsa_g[] = {
    0x68, 0x19, 0x27, 0x88, 0x69, 0xc7, 0xfd, 0x3d,
    0x2d, 0x7b, 0x77, 0xf7, 0x7e, 0x81, 0x50, 0xd9,
    0xad, 0x43, 0x3b, 0xea, 0x3b, 0xa8, 0x5e, 0xfc,
    0x80, 0x41, 0x5a, 0xa3, 0x54, 0x5f, 0x78, 0xf7,
    0x22, 0x96, 0xf0, 0x6c, 0xb1, 0x9c, 0xed, 0xa0,
    0x6c, 0x94, 0xb0, 0x55, 0x1c, 0xfe, 0x6e, 0x6f,
    0x86, 0x3e, 0x31, 0xd1, 0xde, 0x6e, 0xed, 0x7d,
    0xab, 0x8b, 0x0c, 0x9d, 0xf2, 0x31, 0xe0, 0x84,
    0x34, 0xd1, 0x18, 0x4f, 0x91, 0xd0, 0x33, 0x69,
    0x6b, 0xb3, 0x82, 0xf8, 0x45, 0x5e, 0x98, 0x88,
    0xf5, 0xd3, 0x1d, 0x47, 0x84, 0xec, 0x40, 0x12,
    0x02, 0x46, 0xf4, 0xbe, 0xa6, 0x17, 0x94, 0xbb,
    0xa5, 0x86, 0x6f, 0x09, 0x74, 0x64, 0x63, 0xbd,
    0xf8, 0xe9, 0xe1, 0x08, 0xcd, 0x95, 0x29, 0xc3,
    0xd0, 0xf6, 0xdf, 0x80, 0x31, 0x6e, 0x2e, 0x70,
    0xaa, 0xeb, 0x1b, 0x26, 0xcd, 0xb8, 0xad, 0x97,
    0xbc, 0x3d, 0x28, 0x7e, 0x0b, 0x8d, 0x61, 0x6c,
    0x42, 0xe6, 0x5b, 0x87, 0xdb, 0x20, 0xde, 0xb7,
    0x00, 0x5b, 0xc4, 0x16, 0x74, 0x7a, 0x64, 0x70,
    0x14, 0x7a, 0x68, 0xa7, 0x82, 0x03, 0x88, 0xeb,
    0xf4, 0x4d, 0x52, 0xe0, 0x62, 0x8a, 0xf9, 0xcf,
    0x1b, 0x71, 0x66, 0xd0, 0x34, 0x65, 0xf3, 0x5a,
    0xcc, 0x31, 0xb6, 0x11, 0x0c, 0x43, 0xda, 0xbc,
    0x7c, 0x5d, 0x59, 0x1e, 0x67, 0x1e, 0xaf, 0x7c,
    0x25, 0x2c, 0x1c, 0x14, 0x53, 0x36, 0xa1, 0xa4,
    0xdd, 0xf1, 0x32, 0x44, 0xd5, 0x5e, 0x83, 0x56,
    0x80, 0xca, 0xb2, 0x53, 0x3b, 0x82, 0xdf, 0x2e,
    0xfe, 0x55, 0xec, 0x18, 0xc1, 0xe6, 0xcd, 0x00,
    0x7b, 0xb0, 0x89, 0x75, 0x8b, 0xb1, 0x7c, 0x2c,
    0xbe, 0x14, 0x44, 0x1b, 0xd0, 0x93, 0xae, 0x66,
    0xe5, 0x97, 0x6d, 0x53, 0x73, 0x3f, 0x4f, 0xa3,
    0x26, 0x97, 0x01, 0xd3, 0x1d, 0x23, 0xd4, 0x67
};
static const unsigned char dsa_pub[] = {
    0xa0, 0x12, 0xb3, 0xb1, 0x70, 0xb3, 0x07, 0x22,
    0x79, 0x57, 0xb7, 0xca, 0x20, 0x61, 0xa8, 0x16,
    0xac, 0x7a, 0x2b, 0x3d, 0x9a, 0xe9, 0x95, 0xa5,
    0x11, 0x9c, 0x38, 0x5b, 0x60, 0x3b, 0xf6, 0xf6,
    0xc5, 0xde, 0x4d, 0xc5, 0xec, 0xb5, 0xdf, 0xa4,
    0xa4, 0x1c, 0x68, 0x66, 0x2e, 0xb2, 0x5b, 0x63,
    0x8b, 0x7e, 0x26, 0x20, 0xba, 0x89, 0x8d, 0x07,
    0xda, 0x6c, 0x49, 0x91, 0xe7, 0x6c, 0xc0, 0xec,
    0xd1, 0xad, 0x34, 0x21, 0x07, 0x70, 0x67, 0xe4,
    0x7c, 0x18, 0xf5, 0x8a, 0x92, 0xa7, 0x2a, 0xd4,
    0x31, 0x99, 0xec, 0xb7, 0xbd, 0x84, 0xe7, 0xd3,
    0xaf, 0xb9, 0x01, 0x9f, 0x0e, 0x9d, 0xd0, 0xfb,
    0xaa, 0x48, 0x73, 0x00, 0xb1, 0x30, 0x81, 0xe3,
    0x3c, 0x90, 0x28, 0x76, 0x43, 0x6f, 0x7b, 0x03,
    0xc3, 0x45, 0x52, 0x84, 0x81, 0xd3, 0x62, 0x81,
    0x5e, 0x24, 0xfe, 0x59, 0xda, 0xc5, 0xac, 0x34,
    0x66, 0x0d, 0x4c, 0x8a, 0x76, 0xcb, 0x99, 0xa7,
    0xc7, 0xde, 0x93, 0xeb, 0x95, 0x6c, 0xd6, 0xbc,
    0x88, 0xe5, 0x8d, 0x90, 0x10, 0x34, 0x94, 0x4a,
    0x09, 0x4b, 0x01, 0x80, 0x3a, 0x43, 0xc6, 0x72,
    0xb9, 0x68, 0x8c, 0x0e, 0x01, 0xd8, 0xf4, 0xfc,
    0x91, 0xc6, 0x2a, 0x3f, 0x88, 0x02, 0x1f, 0x7b,
    0xd6, 0xa6, 0x51, 0xb1, 0xa8, 0x8f, 0x43, 0xaa,
    0x4e, 0xf2, 0x76, 0x53, 0xd1, 0x2b, 0xf8, 0xb7,
    0x09, 0x9f, 0xdf, 0x6b, 0x46, 0x10, 0x82, 0xf8,
    0xe9, 0x39, 0x10, 0x7b, 0xfd, 0x2f, 0x72, 0x10,
    0x08, 0x7d, 0x32, 0x6c, 0x37, 0x52, 0x00, 0xf1,
    0xf5, 0x1e, 0x7e, 0x74, 0xa3, 0x41, 0x31, 0x90,
    0x1b, 0xcd, 0x08, 0x63, 0x52, 0x1f, 0xf8, 0xd6,
    0x76, 0xc4, 0x85, 0x81, 0x86, 0x87, 0x36, 0xc5,
    0xe5, 0x1b, 0x16, 0xa4, 0xe3, 0x92, 0x15, 0xea,
    0x0b, 0x17, 0xc4, 0x73, 0x59, 0x74, 0xc5, 0x16
};
static const unsigned char dsa_expected_sig[] = {
    0x30, 0x3c, 0x02, 0x1c, 0x69, 0xc6, 0xd6, 0x9e,
    0x2b, 0x91, 0xea, 0x72, 0xb3, 0x8b, 0x7c, 0x57,
    0x48, 0x75, 0xb7, 0x65, 0xc0, 0xb4, 0xf7, 0xbb,
    0x08, 0xa4, 0x95, 0x77, 0xfc, 0xa7, 0xed, 0x31,
    0x02, 0x1c, 0x4c, 0x2c, 0xff, 0xc6, 0x55, 0xeb,
    0x8f, 0xa7, 0x4f, 0x27, 0xd8, 0xec, 0xfd, 0x62,
    0x73, 0xf2, 0xd1, 0x55, 0xa5, 0xf0, 0x41, 0x68,
    0x34, 0x8d, 0x9e, 0x88, 0x08, 0x06
};

static const ST_KAT_PARAM dsa_key[] = {
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_FFC_P, dsa_p),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_FFC_Q, dsa_q),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_FFC_G, dsa_g),
    ST_KAT_PARAM_BIGNUM(OSSL_PKEY_PARAM_PUB_KEY, dsa_pub),
    ST_KAT_PARAM_END()
};
#endif /* OPENSSL_NO_DSA */

#ifndef OPENSSL_NO_ML_DSA
static const unsigned char ml_dsa_65_pub_key[] = {
    0x3b, 0x5c, 0xb0, 0x79, 0xd2, 0xce, 0x76, 0x2b,
    0x3b, 0x95, 0x7c, 0x26, 0x69, 0x8f, 0xe7, 0x01,
    0xb9, 0x6b, 0x50, 0xa3, 0x2c, 0x73, 0x67, 0xcf,
    0x9e, 0xf4, 0xb8, 0x7d, 0xe3, 0xaf, 0x27, 0x77,
    0xc6, 0x7c, 0x34, 0xb7, 0x30, 0x4d, 0x01, 0xa4,
    0xaa, 0xce, 0x16, 0x7c, 0x13, 0x2b, 0x30, 0x6c,
    0x88, 0x31, 0xe4, 0x90, 0xf5, 0xc2, 0x80, 0xf5,
    0xe8, 0xb9, 0x2d, 0x7a, 0x83, 0x17, 0xfc, 0xbb,
    0x13, 0x6f, 0x18, 0x75, 0x5b, 0x40, 0x39, 0x2d,
    0x47, 0x56, 0x15, 0xc7, 0x1c, 0x6e, 0x9a, 0x95,
    0x7b, 0x6b, 0x77, 0x9f, 0x86, 0x38, 0x0e, 0xee,
    0xc0, 0x82, 0x6c, 0x3c, 0xae, 0xc0, 0xcf, 0x5a,
    0x85, 0x49, 0xb1, 0x0c, 0x2d, 0x0e, 0x51, 0x73,
    0xf0, 0xe7, 0xad, 0x3b, 0xa7, 0x3f, 0xf5, 0x75,
    0xb9, 0xb4, 0x63, 0xb9, 0xf0, 0x0a, 0xf4, 0x29,
    0x90, 0x20, 0x72, 0x46, 0x8c, 0x7a, 0xaa, 0x86,
    0xb7, 0x4e, 0xa7, 0x65, 0x23, 0xef, 0xec, 0x46,
    0x67, 0x02, 0xf1, 0xbb, 0x88, 0xc7, 0xa4, 0xfc,
    0x66, 0x52, 0x07, 0x67, 0x68, 0xa2, 0x72, 0xe8,
    0x8a, 0x53, 0x97, 0xe6, 0x89, 0x96, 0x95, 0x9e,
    0x6a, 0xe9, 0xa4, 0x7d, 0x19, 0x19, 0x5f, 0xb4,
    0x77, 0x52, 0x17, 0xd6, 0xf2, 0xea, 0x7f, 0xfc,
    0x5c, 0xd5, 0x18, 0x16, 0x8c, 0xc2, 0x2e, 0x31,
    0xf8, 0x98, 0x4b, 0x72, 0xa1, 0x80, 0xb6, 0x5c,
    0x32, 0x19, 0x2f, 0xe0, 0xae, 0x74, 0xf4, 0xc4,
    0x0a, 0xe0, 0x54, 0x52, 0x46, 0x9e, 0xf3, 0xb9,
    0x6c, 0x56, 0xd1, 0xe8, 0x99, 0x29, 0x39, 0x95,
    0x30, 0xa2, 0x6a, 0xc6, 0x32, 0x8a, 0xa6, 0x02,
    0x6a, 0x39, 0x2e, 0x13, 0x20, 0xbc, 0xf8, 0x7a,
    0x09, 0xb6, 0xa7, 0xd1, 0x39, 0xa5, 0x12, 0x02,
    0x81, 0x47, 0x8c, 0xc0, 0x1e, 0xfd, 0xf3, 0x28,
    0xe5, 0x34, 0xec, 0xf0, 0xfc, 0x3f, 0x22, 0x16,
    0xd2, 0xfe, 0xf0, 0xca, 0xaa, 0x6f, 0x82, 0xdd,
    0xd6, 0x83, 0xaf, 0xf9, 0xeb, 0x1d, 0xa8, 0x45,
    0x39, 0x63, 0xa5, 0xde, 0xee, 0x7d, 0x91, 0xe3,
    0xaa, 0xcc, 0x07, 0x92, 0xce, 0x50, 0xfd, 0xe4,
    0xa8, 0x50, 0x91, 0xd5, 0xec, 0xc1, 0x1b, 0x57,
    0x92, 0x37, 0x68, 0xf8, 0xd6, 0x32, 0x55, 0xba,
    0x65, 0xae, 0xb6, 0xc3, 0x9f, 0x6c, 0x18, 0xc5,
    0x12, 0x17, 0x9a, 0x04, 0x29, 0xab, 0x14, 0x94,
    0xbb, 0x13, 0x79, 0x5e, 0xb9, 0xf0, 0x62, 0x03,
    0xe0, 0xa4, 0x91, 0xba, 0x81, 0x4b, 0xaa, 0xf1,
    0x82, 0x47, 0x83, 0x43, 0x5f, 0x1e, 0x2d, 0x48,
    0x40, 0x56, 0xe4, 0x29, 0x79, 0x01, 0xd5, 0x89,
    0xaf, 0xa4, 0x56, 0x9e, 0x38, 0x62, 0x03, 0xa0,
    0xe7, 0x9f, 0x08, 0x1f, 0xca, 0xd4, 0x31, 0x48,
    0xf6, 0x68, 0xe0, 0xcc, 0x28, 0xff, 0x06, 0x97,
    0x67, 0x4b, 0x70, 0x78, 0xa0, 0x2a, 0xf9, 0x46,
    0x80, 0x6d, 0x37, 0xfb, 0xb3, 0x17, 0x12, 0xf0,
    0x95, 0xc7, 0xee, 0x31, 0x54, 0x75, 0xdf, 0x2a,
    0xa8, 0x7d, 0xff, 0x97, 0xbb, 0x45, 0x49, 0x55,
    0xd5, 0xac, 0x9c, 0x6f, 0x0e, 0xc3, 0x94, 0x96,
    0xc4, 0x9e, 0x9c, 0x45, 0x31, 0xcb, 0x23, 0xed,
    0x21, 0xf1, 0xfe, 0xe9, 0xf9, 0x8c, 0xb6, 0x8e,
    0x72, 0x6e, 0xdd, 0x37, 0x1c, 0xc7, 0xd6, 0x6b,
    0x36, 0x35, 0xa0, 0x67, 0x54, 0x00, 0x65, 0x2c,
    0xc8, 0xa4, 0xa0, 0x9e, 0x72, 0xd7, 0xc9, 0x3c,
    0x8c, 0x78, 0x1a, 0xf8, 0x80, 0xad, 0xc4, 0x99,
    0xc7, 0x35, 0x4e, 0x89, 0x3b, 0x4f, 0xa6, 0x9d,
    0x5d, 0xce, 0x66, 0x4d, 0x82, 0xef, 0x47, 0x73,
    0xa8, 0xc5, 0x14, 0x20, 0xdd, 0x57, 0x92, 0x10,
    0x95, 0x8a, 0xe1, 0xac, 0x82, 0xec, 0x39, 0xe7,
    0x2c, 0xa6, 0xfd, 0x50, 0x68, 0x0c, 0x3e, 0xcf,
    0xbc, 0xc5, 0x12, 0xfe, 0x30, 0xc8, 0xb7, 0xb4,
    0x84, 0xd8, 0x1d, 0x67, 0x54, 0x9d, 0x20, 0x49,
    0xa5, 0xfd, 0xf9, 0x18, 0xef, 0xc2, 0xd2, 0xcd,
    0xb7, 0x54, 0x2b, 0x31, 0x12, 0xf9, 0xaa, 0x8e,
    0x0a, 0x29, 0x0c, 0x37, 0xd2, 0x7c, 0xcd, 0xc5,
    0x0b, 0x98, 0x25, 0x97, 0x0b, 0x5a, 0xf7, 0x07,
    0x91, 0x98, 0xd1, 0x42, 0xdf, 0xc4, 0xf9, 0x42,
    0x97, 0xda, 0x20, 0xf4, 0x88, 0xe1, 0x6b, 0xd4,
    0x85, 0xf5, 0x1d, 0xca, 0x2a, 0xba, 0x30, 0xc5,
    0xdf, 0x9d, 0x4d, 0xb8, 0xb0, 0x30, 0x54, 0x61,
    0xcf, 0x91, 0x6f, 0x90, 0xa5, 0x25, 0x05, 0x9b,
    0x2b, 0x3e, 0x13, 0xcd, 0xcd, 0x40, 0x59, 0x7c,
    0x92, 0x9b, 0x51, 0x81, 0x0d, 0x58, 0x32, 0x0a,
    0x43, 0xbd, 0x01, 0xb4, 0xb8, 0x0b, 0xd5, 0xee,
    0x0d, 0x67, 0x70, 0x59, 0xd7, 0x47, 0x4f, 0xe5,
    0x84, 0x07, 0x75, 0x3e, 0x54, 0xfb, 0x5d, 0xd1,
    0x3f, 0x72, 0x6e, 0xae, 0xf9, 0x4b, 0x7b, 0x57,
    0xe0, 0xde, 0x8b, 0x8b, 0x12, 0x1b, 0x10, 0x3b,
    0x5e, 0x17, 0xd1, 0x72, 0x18, 0x3d, 0xff, 0xc6,
    0x83, 0xa5, 0xaf, 0xf9, 0x30, 0xae, 0xb7, 0x47,
    0x46, 0x5d, 0xac, 0xba, 0x35, 0x04, 0x35, 0x0b,
    0x42, 0x48, 0x7c, 0xa1, 0x00, 0x1a, 0xea, 0xea,
    0x5f, 0x93, 0x2b, 0xb7, 0xe5, 0x8f, 0x91, 0x3c,
    0x00, 0x98, 0x51, 0x40, 0xee, 0x11, 0x50, 0x70,
    0x40, 0xe4, 0x28, 0xd4, 0x79, 0x2d, 0xcd, 0x82,
    0xaf, 0x3f, 0xb2, 0xfc, 0x96, 0x8d, 0xbe, 0x79,
    0xa3, 0xcd, 0xac, 0x35, 0x4b, 0x5e, 0xb4, 0x81,
    0x0e, 0x6a, 0xde, 0x1f, 0x7e, 0xb0, 0x37, 0x3e,
    0xdc, 0xe0, 0x21, 0xcc, 0x9f, 0x90, 0x26, 0xb6,
    0x8a, 0x1d, 0xb3, 0x1e, 0xec, 0x7a, 0x88, 0x28,
    0x95, 0xe2, 0xc2, 0x1d, 0x07, 0xb1, 0xfa, 0xc6,
    0x21, 0x1b, 0x5e, 0x54, 0x7b, 0x37, 0x0e, 0x63,
    0xff, 0xdd, 0x70, 0xf9, 0xea, 0x2f, 0x2d, 0x98,
    0xe1, 0xbe, 0x37, 0xd0, 0x1f, 0x45, 0x5a, 0x63,
    0xad, 0x44, 0xbc, 0x5f, 0xc6, 0x23, 0x8a, 0xac,
    0x12, 0x71, 0xd5, 0xa2, 0x8b, 0xfc, 0x97, 0xbb,
    0x00, 0x4b, 0xd7, 0x09, 0xa6, 0xaf, 0x40, 0x08,
    0x6c, 0x8d, 0x10, 0x4a, 0x01, 0x34, 0xc1, 0x2c,
    0x92, 0x30, 0x0a, 0x85, 0x8f, 0x3f, 0x08, 0xdd,
    0xff, 0x9c, 0x10, 0xd1, 0x03, 0x03, 0x84, 0x1f,
    0xf8, 0x4e, 0xf2, 0xe3, 0xd1, 0xd3, 0xb9, 0xdf,
    0xfc, 0x97, 0x1c, 0xcf, 0x8a, 0x29, 0xe6, 0x59,
    0x04, 0xe2, 0x87, 0x27, 0xbb, 0xb9, 0x96, 0xd0,
    0x20, 0x2e, 0x91, 0x48, 0xaa, 0xbf, 0x53, 0x4a,
    0x34, 0xb5, 0x0e, 0x11, 0xce, 0xf8, 0x65, 0xa6,
    0x0d, 0x45, 0xda, 0xbf, 0x6a, 0xfb, 0x81, 0xe4,
    0x7c, 0x8c, 0xa0, 0x4b, 0x00, 0x1b, 0xd7, 0x73,
    0x61, 0x80, 0xc0, 0x6b, 0x60, 0xde, 0xf3, 0x32,
    0xae, 0x62, 0x35, 0x66, 0xdd, 0xde, 0x53, 0x61,
    0x86, 0xe9, 0x44, 0xf3, 0x01, 0x7b, 0xaa, 0xe7,
    0x31, 0xd4, 0x5b, 0x06, 0x52, 0x0f, 0xf4, 0x90,
    0x5c, 0x82, 0x3e, 0x12, 0x28, 0x88, 0x7f, 0xfc,
    0xb8, 0xee, 0x17, 0x34, 0x4e, 0xc3, 0x2a, 0xfb,
    0x84, 0x1b, 0x0f, 0xba, 0x51, 0x64, 0x96, 0x22,
    0x0d, 0x88, 0x9b, 0xf2, 0x72, 0x04, 0x55, 0x44,
    0x6a, 0x14, 0x2b, 0xa0, 0xc2, 0xbe, 0x9e, 0x7b,
    0x48, 0x32, 0xa7, 0xf6, 0x11, 0xae, 0x60, 0xfb,
    0xf8, 0x38, 0x67, 0x16, 0xdf, 0xdf, 0x46, 0x96,
    0xd7, 0x6c, 0x39, 0xa2, 0xad, 0xf7, 0xb7, 0x78,
    0x32, 0x2c, 0xba, 0xae, 0x33, 0x5a, 0x88, 0x4b,
    0x40, 0x1f, 0x88, 0xcd, 0xe7, 0x8f, 0x50, 0x5e,
    0xd8, 0x80, 0x82, 0x7b, 0x46, 0xc7, 0x07, 0x71,
    0x4f, 0x3b, 0xca, 0x9d, 0x73, 0x7c, 0xdb, 0xeb,
    0x4c, 0x37, 0xdd, 0xb4, 0xb8, 0x61, 0xf3, 0xdf,
    0xb2, 0xb5, 0x34, 0x27, 0xec, 0xeb, 0xba, 0xcc,
    0xdc, 0xe9, 0xde, 0x47, 0x2e, 0xe9, 0x3b, 0xa1,
    0x36, 0xf1, 0x66, 0xdf, 0xc8, 0x70, 0x7f, 0x39,
    0x82, 0xb3, 0x8c, 0x47, 0x9a, 0x45, 0x59, 0x2e,
    0x30, 0x9b, 0xaf, 0x7c, 0xad, 0x43, 0x38, 0x6e,
    0x05, 0x7d, 0x8b, 0xac, 0x5f, 0x70, 0x63, 0xeb,
    0x85, 0xee, 0xab, 0xa4, 0x57, 0x1d, 0x63, 0xac,
    0x48, 0x45, 0x74, 0xca, 0x0c, 0xa8, 0x65, 0x05,
    0x1a, 0x47, 0xa1, 0x2f, 0x4b, 0x96, 0x26, 0x9e,
    0xee, 0xec, 0x37, 0x57, 0xbf, 0xa0, 0x2b, 0x75,
    0xf5, 0x9b, 0xb5, 0x1d, 0x12, 0x8a, 0x61, 0x9c,
    0x8d, 0x2a, 0x7e, 0xee, 0x05, 0x2b, 0x85, 0x7c,
    0x6f, 0x34, 0xc4, 0xcd, 0xd5, 0xd0, 0xac, 0xf9,
    0x79, 0x24, 0xe7, 0x0f, 0x41, 0x95, 0xe2, 0x9a,
    0x22, 0x32, 0xa5, 0x98, 0x2e, 0x82, 0xc0, 0x07,
    0xf4, 0x74, 0x68, 0x00, 0xf9, 0x35, 0x5e, 0x12,
    0xfe, 0xa2, 0x0e, 0x15, 0x96, 0x83, 0x84, 0x31,
    0xc4, 0x25, 0xda, 0x7a, 0xec, 0x07, 0x15, 0xe4,
    0x7d, 0xc5, 0xf5, 0xe1, 0xc5, 0xba, 0x9a, 0x59,
    0x76, 0xae, 0x4e, 0x54, 0x27, 0x5e, 0xa9, 0x0d,
    0xa0, 0xd3, 0xcd, 0x99, 0x39, 0x76, 0x6d, 0x58,
    0xdf, 0x8a, 0xa9, 0x9e, 0x21, 0x22, 0x48, 0x7c,
    0x0c, 0x13, 0xfa, 0x86, 0x63, 0x74, 0x92, 0xf4,
    0xe5, 0x5d, 0xbf, 0xe4, 0x2d, 0xd7, 0xa1, 0xe3,
    0x0f, 0xc6, 0x3e, 0x82, 0xa0, 0xcc, 0xfa, 0x38,
    0x55, 0x36, 0x9b, 0x22, 0xd0, 0xb8, 0x7f, 0x3f,
    0x0f, 0x35, 0x01, 0xf9, 0x6f, 0xa6, 0x51, 0x77,
    0x21, 0xb7, 0x7a, 0x81, 0xca, 0x83, 0x6b, 0xec,
    0xa7, 0x71, 0x12, 0x6d, 0x22, 0x78, 0xb1, 0xc8,
    0x37, 0xda, 0x1d, 0xd1, 0x9c, 0xa5, 0x6c, 0xfa,
    0xd3, 0x4b, 0x87, 0x39, 0x6f, 0x59, 0xe1, 0xec,
    0x4a, 0xe5, 0x0e, 0x72, 0x2b, 0x31, 0x18, 0xd9,
    0x54, 0x6a, 0x4c, 0xc3, 0xe0, 0x58, 0x3b, 0xd8,
    0xe8, 0x65, 0xc1, 0x98, 0xed, 0x64, 0x7b, 0xb1,
    0xee, 0xa9, 0x54, 0x95, 0x37, 0x98, 0x68, 0xca,
    0x83, 0xef, 0xc5, 0x1b, 0x23, 0x71, 0x5c, 0x1a,
    0xe6, 0xc0, 0xce, 0x2e, 0x16, 0x59, 0x79, 0xf4,
    0x94, 0x43, 0xd8, 0xb1, 0x2e, 0xe3, 0xb9, 0xa0,
    0x95, 0x80, 0x66, 0xdd, 0x1d, 0xdd, 0x0d, 0x78,
    0x9d, 0xc3, 0x91, 0x60, 0x16, 0x8b, 0xc5, 0x39,
    0xad, 0xdb, 0xa3, 0xc1, 0xd2, 0x8d, 0xa0, 0x78,
    0x75, 0x68, 0xa6, 0xb9, 0x15, 0x57, 0x0b, 0x06,
    0x64, 0x55, 0xd7, 0x07, 0x53, 0xf9, 0x8b, 0xd9,
    0x97, 0x46, 0xca, 0x04, 0x95, 0xd4, 0x3c, 0xd3,
    0x8f, 0x0a, 0x53, 0x27, 0xe6, 0xd4, 0xb4, 0x7b,
    0x70, 0x12, 0x12, 0xc5, 0x14, 0xa0, 0x53, 0xe5,
    0xd4, 0x30, 0xaa, 0xcc, 0xec, 0x03, 0x0a, 0x36,
    0x21, 0x9f, 0x81, 0x16, 0x6d, 0x1d, 0x53, 0x6e,
    0x08, 0xae, 0xf7, 0x05, 0xd9, 0x73, 0x5a, 0x45,
    0x3a, 0x52, 0x3e, 0xeb, 0x67, 0x49, 0xe1, 0x1a,
    0x8a, 0x4c, 0xd9, 0x83, 0x64, 0xd8, 0x16, 0x37,
    0x1c, 0x6b, 0x1c, 0x0d, 0x8f, 0x6a, 0xbf, 0x21,
    0xf1, 0x4f, 0x4c, 0x55, 0x6f, 0xe0, 0x5c, 0xa7,
    0xb1, 0x7b, 0x57, 0xa9, 0xa2, 0xb4, 0x9d, 0x53,
    0x7f, 0x0f, 0xb0, 0x21, 0x95, 0x70, 0x3a, 0x0d,
    0xa2, 0xc1, 0x52, 0x26, 0xad, 0xa7, 0x48, 0x66,
    0x2a, 0xfc, 0xaa, 0xaf, 0x25, 0x02, 0x58, 0x80,
    0xbe, 0xe7, 0xe4, 0x2a, 0x50, 0xe5, 0x46, 0x13,
    0xaa, 0x57, 0x0e, 0x6e, 0xee, 0xa9, 0x9e, 0x19,
    0xa3, 0x92, 0x8e, 0xc6, 0x3d, 0x76, 0xbb, 0x12,
    0xe2, 0x78, 0x77, 0x99, 0x6a, 0x06, 0xc4, 0x45,
    0x1f, 0x45, 0xe1, 0xf6, 0x65, 0x1f, 0xba, 0xe4,
    0xf6, 0xca, 0xa9, 0x62, 0xa9, 0x2c, 0x3a, 0x2d,
    0x8d, 0x34, 0xe2, 0x32, 0x6e, 0x4a, 0x52, 0x9f,
    0x3d, 0xcf, 0xab, 0xe8, 0x63, 0x66, 0x3f, 0x06,
    0xa3, 0xaa, 0xd5, 0xcb, 0x5d, 0x41, 0xb8, 0xe7,
    0x0a, 0x6b, 0x45, 0x90, 0x3a, 0xf0, 0xe5, 0x7f,
    0x7e, 0xde, 0x68, 0x20, 0x5e, 0x5a, 0x31, 0x3a,
    0x5c, 0x25, 0xb3, 0x82, 0xc7, 0x1e, 0x7d, 0x4d,
    0xd0, 0x23, 0x9f, 0x22, 0x1a, 0x54, 0x0b, 0xe4,
    0x3c, 0x1c, 0xc5, 0x24, 0x39, 0x4d, 0x96, 0x90,
    0xce, 0xc9, 0x3a, 0x07, 0x45, 0x01, 0x9e, 0xb6,
    0x55, 0x1b, 0xdc, 0xe0, 0x3b, 0xe8, 0x92, 0x57,
    0x43, 0x12, 0x9b, 0x11, 0x3f, 0x81, 0x5a, 0xca,
    0xc5, 0x85, 0x39, 0x25, 0x6d, 0xe4, 0x30, 0xef,
    0x83, 0x21, 0xed, 0x9c, 0xb3, 0xcf, 0x10, 0xe6,
    0xa1, 0x46, 0x10, 0x9a, 0x7b, 0xaf, 0x5f, 0x75,
    0x99, 0x62, 0xc1, 0xf4, 0x2c, 0x28, 0x0a, 0x8a,
    0xc7, 0xd2, 0xb4, 0x75, 0xb9, 0x66, 0x70, 0x9a,
    0xaf, 0xe3, 0xf1, 0x2f, 0xf2, 0xb0, 0x59, 0x9c,
    0x1e, 0x1a, 0xbd, 0xa9, 0x15, 0x55, 0x95, 0x4c,
    0x5b, 0x78, 0x0b, 0x2c, 0x00, 0xbb, 0xf9, 0x8b,
    0xeb, 0x72, 0x4f, 0xfb, 0xa4, 0x3a, 0xa0, 0x04,
    0x45, 0x32, 0x02, 0x6c, 0x16, 0x52, 0x3f, 0x4a,
    0x0a, 0x77, 0x64, 0xf4, 0x74, 0xed, 0x60, 0x6b,
    0x5e, 0x43, 0xa7, 0xe3, 0x84, 0x7e, 0xda, 0xf0,
    0xb1, 0x9e, 0x6d, 0x90, 0x9c, 0x32, 0xae, 0xba,
    0x7c, 0xfb, 0x72, 0x22, 0x27, 0x9c, 0xe1, 0x85,
    0xe1, 0xf6, 0x22, 0x4c, 0x3c, 0x4a, 0xd6, 0xed,
    0x4c, 0xa7, 0x79, 0x59, 0xb5, 0x5b, 0x91, 0x65,
    0x3f, 0x93, 0x97, 0x8d, 0xd7, 0xd0, 0xab, 0x17,
    0x2f, 0x13, 0x74, 0x53, 0x69, 0x74, 0xf8, 0x6b,
    0x39, 0x5c, 0x64, 0x5b, 0x3d, 0x75, 0xca, 0x85,
    0x0e, 0xda, 0x0f, 0x01, 0x34, 0xa3, 0x67, 0x8e,
    0x26, 0x6f, 0x26, 0x3c, 0xd0, 0xd9, 0xae, 0xe8,
    0x52, 0x13, 0x7f, 0xf8, 0x69, 0x62, 0xfc, 0x9a,
    0xc0, 0x0c, 0x66, 0x2e, 0x57, 0x21, 0x75, 0xb0,
    0xb3, 0x8c, 0xf6, 0x97, 0x44, 0x46, 0x65, 0x15,
    0x79, 0xd5, 0x6b, 0x68, 0x96, 0x47, 0xc1, 0xba,
    0x75, 0x46, 0x87, 0x76, 0x7d, 0x2d, 0xac, 0xf3,
    0x16, 0xae, 0xfb, 0x7e, 0x41, 0xe4, 0xae, 0x15,
    0xc2, 0x51, 0x69, 0x71, 0x0b, 0x63, 0x20, 0x6a,
    0xbd, 0xad, 0xce, 0x2a, 0x94, 0xac, 0xcf, 0x15,
    0x4e, 0xdc, 0x8e, 0x2a, 0x48, 0xed, 0xb3, 0x48,
    0x95, 0xf4, 0x41, 0xf3, 0x52, 0xef, 0x62, 0x90,
    0x10, 0x30, 0x42, 0xec, 0xf9, 0x30, 0x25, 0xc3,
    0xc5, 0x47, 0x76, 0xb5, 0x37, 0xeb, 0x9e, 0x87,
    0xbe, 0x5c, 0x24, 0xa5, 0x34, 0xdd, 0x92, 0xa1,
    0x20, 0x7d, 0xa7, 0x94, 0xa2, 0x67, 0x26, 0x70,
    0xfe, 0xc9, 0x3f, 0x21, 0xd1, 0xc2, 0x5a, 0xb1,
    0xaa, 0xac, 0x14, 0x1a, 0xe4, 0xdb, 0x71, 0x7e,
    0x9c, 0xc2, 0x52, 0x59, 0xc9, 0x58, 0xca, 0x88,
    0x71, 0x4f, 0x90, 0xb1, 0xbb, 0xac, 0x80, 0x53,
    0x21, 0xfb, 0xff, 0xfd, 0x1e, 0x9f, 0xc0, 0x59,
    0x8d, 0x80, 0x8e, 0x85, 0xa9, 0x3b, 0xd3, 0x20,
    0x3b, 0x91, 0x16, 0x49, 0xb2, 0x2a, 0xa4, 0xe2,
    0x30, 0xb2, 0x76, 0xf6, 0x92, 0x71, 0x37, 0x1b
};

static const unsigned char ml_dsa_65_priv_key[] = {
    0x3b, 0x5c, 0xb0, 0x79, 0xd2, 0xce, 0x76, 0x2b,
    0x3b, 0x95, 0x7c, 0x26, 0x69, 0x8f, 0xe7, 0x01,
    0xb9, 0x6b, 0x50, 0xa3, 0x2c, 0x73, 0x67, 0xcf,
    0x9e, 0xf4, 0xb8, 0x7d, 0xe3, 0xaf, 0x27, 0x77,
    0x39, 0x3e, 0xac, 0x91, 0xa1, 0xfb, 0xed, 0xf7,
    0x52, 0x0a, 0x4f, 0x77, 0xb2, 0x57, 0x7c, 0x65,
    0x05, 0x6e, 0x76, 0x3d, 0x28, 0xf4, 0x3a, 0x88,
    0x49, 0x33, 0x7d, 0x30, 0x17, 0x21, 0x5a, 0xc0,
    0x4b, 0x03, 0x74, 0x65, 0x6d, 0xbb, 0xfe, 0x8e,
    0x0b, 0x8b, 0xb4, 0x42, 0xce, 0xc1, 0xdd, 0x11,
    0xb1, 0x8f, 0x31, 0x06, 0x0b, 0xeb, 0xdf, 0xb3,
    0x11, 0x12, 0x19, 0x22, 0x34, 0xd7, 0xaa, 0xf2,
    0x17, 0xf0, 0x2e, 0x22, 0xef, 0x68, 0x0c, 0x8f,
    0xc7, 0x42, 0x0b, 0xd6, 0x88, 0xd8, 0x3b, 0xa7,
    0x22, 0xc5, 0x4e, 0x1a, 0xf2, 0x60, 0x85, 0x24,
    0x75, 0x85, 0x49, 0x17, 0x1b, 0xd4, 0x69, 0xd8,
    0x50, 0x68, 0x36, 0x83, 0x51, 0x12, 0x40, 0x24,
    0x06, 0x04, 0x07, 0x47, 0x28, 0x62, 0x86, 0x26,
    0x12, 0x24, 0x40, 0x83, 0x65, 0x44, 0x10, 0x62,
    0x78, 0x58, 0x14, 0x67, 0x20, 0x20, 0x60, 0x53,
    0x28, 0x73, 0x53, 0x34, 0x82, 0x18, 0x42, 0x43,
    0x78, 0x67, 0x65, 0x28, 0x12, 0x65, 0x71, 0x17,
    0x33, 0x06, 0x10, 0x51, 0x38, 0x07, 0x61, 0x42,
    0x72, 0x60, 0x18, 0x52, 0x08, 0x74, 0x58, 0x72,
    0x73, 0x06, 0x65, 0x20, 0x73, 0x20, 0x72, 0x42,
    0x03, 0x86, 0x03, 0x03, 0x50, 0x85, 0x13, 0x43,
    0x24, 0x53, 0x33, 0x82, 0x03, 0x36, 0x82, 0x73,
    0x12, 0x15, 0x78, 0x54, 0x60, 0x64, 0x32, 0x60,
    0x42, 0x47, 0x84, 0x73, 0x51, 0x60, 0x26, 0x03,
    0x35, 0x58, 0x25, 0x28, 0x83, 0x55, 0x64, 0x06,
    0x70, 0x82, 0x75, 0x54, 0x15, 0x70, 0x37, 0x61,
    0x71, 0x38, 0x75, 0x36, 0x15, 0x70, 0x78, 0x17,
    0x62, 0x01, 0x02, 0x13, 0x12, 0x46, 0x12, 0x56,
    0x82, 0x51, 0x32, 0x16, 0x27, 0x00, 0x52, 0x41,
    0x43, 0x86, 0x46, 0x00, 0x61, 0x24, 0x78, 0x33,
    0x40, 0x13, 0x34, 0x47, 0x60, 0x18, 0x52, 0x18,
    0x80, 0x01, 0x46, 0x53, 0x32, 0x74, 0x51, 0x72,
    0x81, 0x21, 0x30, 0x50, 0x28, 0x28, 0x55, 0x53,
    0x71, 0x03, 0x84, 0x87, 0x61, 0x88, 0x45, 0x35,
    0x63, 0x72, 0x05, 0x37, 0x06, 0x37, 0x35, 0x36,
    0x56, 0x71, 0x74, 0x53, 0x07, 0x61, 0x72, 0x65,
    0x78, 0x57, 0x42, 0x41, 0x02, 0x06, 0x61, 0x06,
    0x07, 0x66, 0x64, 0x50, 0x31, 0x24, 0x82, 0x43,
    0x86, 0x31, 0x26, 0x76, 0x47, 0x45, 0x03, 0x66,
    0x44, 0x00, 0x46, 0x04, 0x83, 0x03, 0x83, 0x81,
    0x33, 0x08, 0x77, 0x57, 0x08, 0x40, 0x04, 0x22,
    0x35, 0x17, 0x43, 0x63, 0x78, 0x44, 0x86, 0x24,
    0x40, 0x17, 0x28, 0x67, 0x34, 0x18, 0x75, 0x72,
    0x26, 0x83, 0x68, 0x63, 0x57, 0x47, 0x48, 0x84,
    0x50, 0x16, 0x43, 0x17, 0x86, 0x16, 0x44, 0x17,
    0x18, 0x70, 0x08, 0x51, 0x78, 0x70, 0x15, 0x58,
    0x04, 0x10, 0x16, 0x03, 0x11, 0x04, 0x43, 0x70,
    0x32, 0x77, 0x03, 0x23, 0x12, 0x74, 0x38, 0x26,
    0x65, 0x66, 0x62, 0x04, 0x65, 0x75, 0x54, 0x00,
    0x04, 0x01, 0x43, 0x15, 0x07, 0x47, 0x71, 0x12,
    0x70, 0x41, 0x72, 0x34, 0x05, 0x26, 0x08, 0x11,
    0x55, 0x22, 0x72, 0x83, 0x04, 0x66, 0x88, 0x80,
    0x25, 0x17, 0x68, 0x60, 0x15, 0x37, 0x64, 0x14,
    0x26, 0x64, 0x01, 0x45, 0x54, 0x51, 0x71, 0x38,
    0x62, 0x65, 0x78, 0x43, 0x36, 0x21, 0x02, 0x27,
    0x43, 0x47, 0x83, 0x86, 0x07, 0x05, 0x48, 0x56,
    0x86, 0x31, 0x27, 0x66, 0x34, 0x35, 0x48, 0x81,
    0x08, 0x38, 0x05, 0x10, 0x63, 0x32, 0x61, 0x73,
    0x47, 0x30, 0x00, 0x82, 0x66, 0x54, 0x86, 0x81,
    0x36, 0x06, 0x45, 0x01, 0x45, 0x87, 0x23, 0x36,
    0x83, 0x00, 0x32, 0x81, 0x10, 0x06, 0x57, 0x76,
    0x62, 0x40, 0x61, 0x77, 0x04, 0x17, 0x28, 0x70,
    0x15, 0x07, 0x42, 0x18, 0x61, 0x70, 0x32, 0x42,
    0x68, 0x12, 0x00, 0x18, 0x12, 0x72, 0x60, 0x50,
    0x63, 0x28, 0x51, 0x87, 0x82, 0x03, 0x36, 0x63,
    0x63, 0x25, 0x68, 0x54, 0x56, 0x16, 0x07, 0x58,
    0x77, 0x83, 0x31, 0x85, 0x44, 0x43, 0x60, 0x22,
    0x16, 0x01, 0x41, 0x23, 0x27, 0x11, 0x46, 0x60,
    0x50, 0x84, 0x77, 0x18, 0x81, 0x31, 0x88, 0x47,
    0x36, 0x51, 0x71, 0x56, 0x04, 0x25, 0x17, 0x44,
    0x33, 0x78, 0x28, 0x72, 0x27, 0x87, 0x62, 0x78,
    0x13, 0x83, 0x58, 0x12, 0x36, 0x22, 0x10, 0x25,
    0x13, 0x50, 0x88, 0x10, 0x24, 0x78, 0x15, 0x12,
    0x26, 0x43, 0x58, 0x02, 0x24, 0x17, 0x34, 0x75,
    0x46, 0x15, 0x46, 0x42, 0x84, 0x52, 0x41, 0x75,
    0x48, 0x77, 0x88, 0x33, 0x18, 0x18, 0x84, 0x31,
    0x47, 0x22, 0x35, 0x64, 0x72, 0x35, 0x65, 0x73,
    0x66, 0x88, 0x73, 0x12, 0x25, 0x51, 0x66, 0x57,
    0x05, 0x23, 0x54, 0x74, 0x51, 0x76, 0x52, 0x24,
    0x40, 0x64, 0x68, 0x28, 0x25, 0x65, 0x77, 0x12,
    0x53, 0x35, 0x41, 0x85, 0x18, 0x25, 0x02, 0x25,
    0x47, 0x22, 0x40, 0x37, 0x54, 0x03, 0x53, 0x26,
    0x86, 0x50, 0x70, 0x25, 0x56, 0x27, 0x21, 0x11,
    0x11, 0x72, 0x71, 0x54, 0x47, 0x63, 0x67, 0x22,
    0x75, 0x61, 0x70, 0x05, 0x10, 0x03, 0x41, 0x48,
    0x01, 0x47, 0x14, 0x70, 0x06, 0x64, 0x84, 0x02,
    0x28, 0x06, 0x12, 0x45, 0x13, 0x56, 0x48, 0x65,
    0x00, 0x74, 0x02, 0x10, 0x78, 0x42, 0x80, 0x02,
    0x00, 0x16, 0x37, 0x43, 0x23, 0x56, 0x67, 0x20,
    0x31, 0x86, 0x22, 0x20, 0x35, 0x51, 0x34, 0x58,
    0x10, 0x87, 0x83, 0x43, 0x32, 0x85, 0x47, 0x72,
    0x32, 0x28, 0x38, 0x47, 0x08, 0x73, 0x38, 0x20,
    0x25, 0x14, 0x51, 0x66, 0x88, 0x85, 0x15, 0x88,
    0x86, 0x86, 0x73, 0x46, 0x52, 0x81, 0x32, 0x70,
    0x85, 0x31, 0x20, 0x80, 0x32, 0x42, 0x02, 0x33,
    0x37, 0x68, 0x66, 0x70, 0x11, 0x27, 0x53, 0x75,
    0x48, 0x65, 0x01, 0x25, 0x24, 0x53, 0x68, 0x15,
    0x47, 0x25, 0x42, 0x01, 0x04, 0x47, 0x01, 0x58,
    0x36, 0x17, 0x71, 0x44, 0x21, 0x63, 0x52, 0x61,
    0x56, 0x16, 0x73, 0x17, 0x28, 0x08, 0x07, 0x76,
    0x08, 0x42, 0x58, 0x11, 0x42, 0x18, 0x17, 0x18,
    0x65, 0x63, 0x82, 0x45, 0x74, 0x35, 0x21, 0x01,
    0x35, 0x26, 0x43, 0x56, 0x58, 0x56, 0x64, 0x81,
    0x05, 0x52, 0x77, 0x31, 0x47, 0x27, 0x21, 0x82,
    0x60, 0x05, 0x51, 0x04, 0x15, 0x34, 0x58, 0x00,
    0x40, 0x83, 0x48, 0x21, 0x52, 0x04, 0x62, 0x23,
    0x34, 0x64, 0x85, 0x77, 0x35, 0x81, 0x52, 0x35,
    0x80, 0x24, 0x00, 0x03, 0x84, 0x32, 0x67, 0x76,
    0x78, 0x32, 0x87, 0x50, 0x48, 0x00, 0x64, 0x45,
    0x17, 0x05, 0x56, 0x84, 0x13, 0x02, 0x68, 0x42,
    0x76, 0x81, 0x01, 0x41, 0x18, 0x56, 0x30, 0x58,
    0x43, 0x72, 0x28, 0x71, 0x21, 0x87, 0x14, 0x64,
    0x03, 0x31, 0x58, 0x04, 0x72, 0x11, 0x43, 0x50,
    0x82, 0x57, 0x14, 0x38, 0x53, 0x36, 0x53, 0x44,
    0x37, 0x36, 0x83, 0x47, 0x58, 0x58, 0x52, 0x08,
    0x13, 0x33, 0x43, 0x40, 0x54, 0x26, 0x35, 0x28,
    0x67, 0x02, 0x08, 0x18, 0x34, 0x67, 0x16, 0x17,
    0x73, 0x13, 0x18, 0x02, 0x88, 0x32, 0x61, 0x60,
    0x56, 0x38, 0x05, 0x87, 0x78, 0x52, 0x07, 0x53,
    0x81, 0x06, 0x52, 0x88, 0x07, 0x35, 0x72, 0x72,
    0x44, 0x28, 0x80, 0x86, 0x11, 0x70, 0x40, 0x41,
    0x63, 0x80, 0x46, 0x15, 0x27, 0x62, 0x03, 0x45,
    0x14, 0x33, 0x56, 0x07, 0x78, 0x43, 0x77, 0x62,
    0x06, 0x54, 0x74, 0x43, 0x28, 0x72, 0x16, 0x40,
    0x78, 0x62, 0x44, 0x34, 0x00, 0x25, 0x75, 0x74,
    0x77, 0x37, 0x02, 0x48, 0x67, 0x18, 0x51, 0x67,
    0x77, 0x68, 0x38, 0x72, 0x85, 0x34, 0x42, 0x54,
    0x71, 0x61, 0x47, 0x42, 0x68, 0x55, 0x27, 0x26,
    0x38, 0x85, 0x15, 0x15, 0x43, 0x14, 0x31, 0x24,
    0x81, 0x26, 0x21, 0x17, 0x04, 0x00, 0x27, 0x21,
    0x20, 0x72, 0x85, 0x77, 0x30, 0x17, 0x26, 0x48,
    0x05, 0x48, 0x71, 0x41, 0x48, 0x27, 0x12, 0x43,
    0x35, 0x26, 0x13, 0x00, 0x26, 0x67, 0x28, 0x25,
    0x10, 0x05, 0x74, 0x06, 0x65, 0x16, 0x87, 0x17,
    0x73, 0x37, 0x42, 0x20, 0x16, 0x44, 0x40, 0x68,
    0x27, 0x23, 0x38, 0x28, 0x60, 0x52, 0x64, 0x40,
    0x28, 0x46, 0x00, 0x00, 0x01, 0x43, 0x75, 0x15,
    0x38, 0x12, 0x37, 0x11, 0x11, 0x87, 0x34, 0x43,
    0x35, 0x76, 0x76, 0x50, 0x43, 0x05, 0x00, 0x27,
    0x18, 0x70, 0x68, 0x14, 0x60, 0x26, 0x12, 0x75,
    0x42, 0x06, 0x23, 0x56, 0x74, 0x74, 0x37, 0x17,
    0x46, 0x20, 0x34, 0x72, 0x21, 0x33, 0x82, 0x73,
    0x51, 0x64, 0x72, 0x02, 0x48, 0x82, 0x57, 0x53,
    0x64, 0x74, 0x75, 0x28, 0x64, 0x02, 0x30, 0x22,
    0x06, 0x73, 0x66, 0x72, 0x46, 0x56, 0x30, 0x11,
    0x78, 0x32, 0x47, 0x52, 0x31, 0x02, 0x74, 0x13,
    0x13, 0x41, 0x07, 0x27, 0x72, 0x07, 0x66, 0x63,
    0x34, 0x84, 0x30, 0x66, 0x28, 0x72, 0x50, 0x62,
    0x30, 0x52, 0x82, 0x37, 0x40, 0x78, 0x31, 0x35,
    0x76, 0x83, 0x05, 0x02, 0x37, 0x25, 0x28, 0x85,
    0x62, 0x10, 0x05, 0x38, 0x64, 0x87, 0x21, 0x76,
    0x04, 0x51, 0x12, 0x45, 0x08, 0x50, 0x50, 0x75,
    0x20, 0x86, 0x32, 0x43, 0x13, 0x47, 0x31, 0x24,
    0x26, 0x57, 0x68, 0x64, 0x88, 0x17, 0x02, 0x70,
    0x42, 0x72, 0x37, 0x32, 0x85, 0x87, 0x83, 0x86,
    0x47, 0x70, 0x56, 0x22, 0x24, 0x53, 0x48, 0x85,
    0x02, 0x48, 0x16, 0x01, 0x07, 0x57, 0x73, 0x76,
    0x27, 0x43, 0x00, 0x88, 0x27, 0x38, 0x30, 0x86,
    0x27, 0x18, 0x62, 0x05, 0x30, 0x37, 0x16, 0x80,
    0x31, 0x72, 0x28, 0x37, 0x47, 0x57, 0x62, 0x55,
    0x74, 0x07, 0x41, 0x72, 0x26, 0x58, 0x52, 0x68,
    0x04, 0x80, 0x72, 0x77, 0x35, 0x43, 0x36, 0x47,
    0x38, 0x76, 0x86, 0x60, 0x14, 0x22, 0x77, 0x61,
    0x52, 0x74, 0x17, 0x64, 0x67, 0x85, 0x23, 0x58,
    0x28, 0x23, 0x05, 0x33, 0x66, 0x64, 0x83, 0x71,
    0x14, 0x03, 0x21, 0x70, 0x55, 0x43, 0x35, 0x58,
    0x53, 0x78, 0x22, 0x00, 0x15, 0x00, 0x50, 0x16,
    0x26, 0x84, 0x18, 0x51, 0x24, 0x01, 0x20, 0x51,
    0x06, 0x74, 0x44, 0x83, 0x40, 0x55, 0x70, 0x28,
    0x66, 0x11, 0x31, 0x02, 0x57, 0x55, 0x52, 0x72,
    0x27, 0x84, 0x54, 0x71, 0x40, 0x78, 0x44, 0x75,
    0x24, 0x78, 0x61, 0x21, 0x65, 0x57, 0x43, 0x58,
    0x21, 0x15, 0x11, 0x87, 0x35, 0x82, 0x52, 0x00,
    0x46, 0x11, 0x34, 0x68, 0x60, 0x60, 0x50, 0x15,
    0x25, 0x62, 0x48, 0x08, 0x18, 0x31, 0x36, 0x86,
    0x41, 0x26, 0x80, 0x82, 0x54, 0x68, 0x02, 0x07,
    0x23, 0x22, 0x24, 0x33, 0x27, 0x02, 0x04, 0x40,
    0x22, 0x47, 0x87, 0x62, 0x70, 0x21, 0x81, 0x22,
    0x25, 0x78, 0x87, 0x67, 0x62, 0x38, 0x42, 0x10,
    0x21, 0x58, 0x56, 0x06, 0x20, 0x18, 0x46, 0x18,
    0x22, 0x61, 0x73, 0x45, 0x65, 0x73, 0x58, 0x57,
    0x04, 0x76, 0x72, 0x54, 0x48, 0x00, 0x62, 0x27,
    0x22, 0x25, 0x17, 0x43, 0x55, 0x58, 0x57, 0x64,
    0x04, 0x61, 0x21, 0x24, 0x02, 0x20, 0x17, 0x34,
    0x25, 0x13, 0x35, 0x14, 0x52, 0x76, 0x83, 0x11,
    0x05, 0x05, 0x03, 0x11, 0x28, 0x71, 0x16, 0x40,
    0x56, 0x21, 0x76, 0x18, 0x67, 0x38, 0x02, 0x15,
    0x36, 0x67, 0xf3, 0xc8, 0x13, 0x98, 0x97, 0xb3,
    0x28, 0x88, 0xb7, 0x32, 0x0d, 0xf8, 0xcf, 0x60,
    0x47, 0x25, 0x85, 0xf5, 0x2d, 0x0c, 0x02, 0x62,
    0xa7, 0xb3, 0x22, 0x3c, 0x55, 0xb6, 0x55, 0x9b,
    0x20, 0x35, 0x10, 0x77, 0x87, 0xeb, 0xcf, 0xa0,
    0x11, 0x37, 0xdc, 0x09, 0x0b, 0xdb, 0x94, 0x28,
    0xc5, 0x30, 0x74, 0x7f, 0x73, 0x3a, 0x05, 0x10,
    0xf2, 0x1a, 0x61, 0x88, 0xdb, 0x2e, 0xc9, 0xf1,
    0x14, 0x75, 0x3d, 0xef, 0x09, 0x2f, 0x53, 0x57,
    0x87, 0xb0, 0x78, 0xa2, 0xc1, 0xe3, 0x5b, 0xd4,
    0x4e, 0x77, 0x86, 0x0c, 0x69, 0x06, 0x33, 0x59,
    0x35, 0xdd, 0xbd, 0xa7, 0x9f, 0x3e, 0x2f, 0x0f,
    0x22, 0x4d, 0x17, 0xa6, 0x5c, 0x60, 0x3e, 0xa3,
    0xf1, 0xbf, 0x08, 0x38, 0x9f, 0x14, 0x0b, 0x07,
    0xa4, 0x0d, 0x1e, 0xf8, 0x34, 0x23, 0x01, 0x62,
    0xad, 0x3c, 0x65, 0x49, 0xca, 0x12, 0x01, 0x2b,
    0x2d, 0x17, 0x4c, 0xa1, 0xa7, 0x40, 0xde, 0xdd,
    0x2b, 0x4b, 0x14, 0xf1, 0x21, 0x6a, 0x68, 0xd5,
    0x65, 0xe6, 0x16, 0x42, 0xdc, 0xf3, 0xce, 0x0e,
    0x3a, 0xa4, 0x12, 0x16, 0x72, 0xa5, 0xa4, 0xbb,
    0xa3, 0xb4, 0x08, 0x6d, 0xce, 0x95, 0x73, 0xc6,
    0x4a, 0xe4, 0x5a, 0x14, 0x73, 0xb1, 0x14, 0xab,
    0x65, 0xb7, 0x8b, 0xe4, 0x7c, 0xa1, 0xd2, 0xbb,
    0x6a, 0xf2, 0xf2, 0x7d, 0x64, 0x26, 0x68, 0xec,
    0xc2, 0x51, 0x3c, 0xf0, 0xcc, 0x7a, 0x04, 0x6a,
    0x29, 0x8d, 0x76, 0x7f, 0xe5, 0x8f, 0x95, 0x20,
    0xd1, 0xf7, 0x98, 0x29, 0xf5, 0x35, 0x80, 0x25,
    0xda, 0x54, 0xde, 0xb9, 0x5f, 0x6d, 0x68, 0xcf,
    0x7e, 0x85, 0xcf, 0x38, 0x0d, 0x32, 0x63, 0x37,
    0x77, 0xe1, 0xa5, 0x9c, 0x0a, 0x6c, 0xc5, 0x3f,
    0xde, 0xdb, 0xe6, 0x57, 0x16, 0x4e, 0xbc, 0xb1,
    0x02, 0x7c, 0x58, 0x85, 0x60, 0x1f, 0x0d, 0x0f,
    0x4f, 0x5f, 0xbf, 0xa3, 0xbc, 0x2c, 0xfe, 0x66,
    0xbf, 0xb5, 0x93, 0xf0, 0x13, 0xfa, 0x52, 0x07,
    0x40, 0xdb, 0xbf, 0x6e, 0x6c, 0xca, 0x74, 0x61,
    0x0b, 0xa6, 0xaf, 0x02, 0x60, 0xf1, 0x57, 0x17,
    0x08, 0xf1, 0x09, 0xf7, 0xc4, 0x4e, 0x51, 0xaf,
    0x04, 0xf1, 0x4c, 0xbc, 0xe5, 0x77, 0x8f, 0x0f,
    0x9b, 0xa7, 0xb3, 0xf7, 0xb0, 0x32, 0x14, 0xbc,
    0x1d, 0x40, 0xe7, 0x06, 0x01, 0xd6, 0xf2, 0x7f,
    0x84, 0xab, 0x47, 0x34, 0x15, 0x2c, 0x62, 0xfe,
    0x95, 0x51, 0xaf, 0x51, 0x4a, 0x1d, 0xc4, 0xea,
    0xaf, 0xdc, 0xe4, 0xea, 0xa7, 0x7d, 0x25, 0x2a,
    0x79, 0x24, 0xf8, 0xaa, 0xf2, 0xe9, 0x82, 0x48,
    0x7b, 0x0e, 0x2c, 0x5c, 0x38, 0x89, 0x21, 0xfe,
    0x91, 0x72, 0xed, 0xd9, 0x41, 0xac, 0x0d, 0xdc,
    0x12, 0xf4, 0x3f, 0x49, 0x91, 0x65, 0x2d, 0x39,
    0xea, 0xdf, 0xc6, 0xcd, 0xa8, 0xbe, 0x53, 0x23,
    0x04, 0xbf, 0xd3, 0xf0, 0xb8, 0x76, 0xc8, 0x01,
    0x43, 0x2c, 0x6b, 0x79, 0xae, 0x23, 0x50, 0xe0,
    0x34, 0x09, 0x85, 0xd8, 0xdb, 0xc5, 0x37, 0xb8,
    0xd5, 0x49, 0x26, 0x9c, 0x0b, 0x85, 0x95, 0x5f,
    0x28, 0x17, 0x8d, 0xd3, 0x0d, 0xe0, 0x34, 0x34,
    0x23, 0x07, 0x74, 0x13, 0x55, 0xd7, 0xe5, 0xb4,
    0x83, 0x4c, 0xd8, 0xdc, 0xdb, 0x5a, 0xe9, 0x3c,
    0x36, 0x53, 0xa4, 0x3a, 0x80, 0xde, 0x77, 0xa0,
    0xaa, 0x9a, 0xe7, 0x6e, 0xdd, 0x98, 0xdd, 0x6a,
    0xf4, 0x34, 0xda, 0x83, 0x80, 0x9c, 0xe5, 0x83,
    0xb3, 0xb6, 0xf7, 0x81, 0x39, 0x46, 0xea, 0x93,
    0x85, 0xad, 0xf5, 0xa7, 0x1a, 0xb1, 0x31, 0x6f,
    0xb9, 0x79, 0x11, 0x57, 0x62, 0x47, 0xd9, 0xcc,
    0x67, 0x78, 0xec, 0x32, 0xd3, 0x0c, 0x34, 0xea,
    0x1b, 0xa8, 0x84, 0x54, 0xb1, 0xf3, 0x80, 0x4b,
    0x79, 0x32, 0x82, 0x18, 0x54, 0x2d, 0xdf, 0xbb,
    0x4d, 0xbe, 0x75, 0xa0, 0x8e, 0x53, 0x52, 0xed,
    0x6e, 0x0a, 0x9f, 0xa6, 0x9b, 0x14, 0x32, 0x10,
    0xf1, 0x7e, 0xdd, 0xfa, 0xc5, 0xf5, 0xbd, 0x67,
    0xfb, 0x64, 0xf6, 0xc3, 0xb1, 0xc9, 0x81, 0x17,
    0x60, 0x56, 0x11, 0xdc, 0x63, 0x13, 0x16, 0x4a,
    0xa8, 0xeb, 0x7f, 0x82, 0xd3, 0x90, 0xeb, 0x5e,
    0xe7, 0xb0, 0xd9, 0x3e, 0x13, 0xd3, 0x73, 0x90,
    0x0d, 0x67, 0xc1, 0x18, 0xcc, 0x3b, 0x88, 0xcd,
    0x4d, 0x40, 0xbe, 0xd4, 0x5e, 0xee, 0x73, 0xa4,
    0xee, 0xbd, 0xcb, 0xbc, 0x39, 0x8b, 0x6d, 0x15,
    0x34, 0x18, 0xd2, 0x55, 0x93, 0xc0, 0xf4, 0x95,
    0xd1, 0xe4, 0x7f, 0x37, 0xb1, 0x3a, 0x8c, 0x4b,
    0xbc, 0x7d, 0x76, 0xb4, 0x34, 0xf4, 0x11, 0x5c,
    0x10, 0xeb, 0x34, 0x88, 0x61, 0xd7, 0x89, 0xac,
    0x87, 0x91, 0x6a, 0x80, 0xee, 0x07, 0x0d, 0x26,
    0xfa, 0xd2, 0x75, 0x6b, 0xd6, 0x0b, 0xa5, 0xc6,
    0x71, 0x6d, 0x0a, 0x5b, 0x59, 0x62, 0x32, 0x60,
    0x42, 0x73, 0x75, 0xda, 0x63, 0x36, 0x96, 0xca,
    0xde, 0xe9, 0x60, 0xfe, 0xa2, 0x64, 0xcc, 0x85,
    0x22, 0x4c, 0xc5, 0x2f, 0x0e, 0xe8, 0xf5, 0x93,
    0xdb, 0x16, 0xa7, 0x1f, 0xd9, 0x30, 0xbc, 0xa2,
    0xba, 0x23, 0xca, 0xea, 0xc1, 0x4d, 0x2f, 0xd4,
    0x81, 0x98, 0xff, 0x58, 0xfc, 0x40, 0x39, 0x75,
    0x52, 0x3e, 0x3e, 0xf4, 0x8d, 0x95, 0x55, 0x0a,
    0x95, 0x5f, 0x4f, 0x5d, 0xdd, 0x0d, 0xb8, 0xd4,
    0x63, 0x98, 0x48, 0x6b, 0xec, 0xe8, 0xd9, 0xae,
    0xeb, 0x9d, 0x48, 0xa4, 0x9a, 0x63, 0x03, 0xc4,
    0x57, 0xcd, 0x7d, 0x97, 0xc8, 0x11, 0x7e, 0xcb,
    0x78, 0xe8, 0x5d, 0x6e, 0xe6, 0xdd, 0x13, 0x33,
    0xf2, 0x05, 0xcd, 0x5a, 0x44, 0x18, 0xf9, 0x32,
    0x41, 0x4e, 0x5c, 0xfc, 0x7a, 0xe9, 0xc2, 0x28,
    0xd3, 0x42, 0x56, 0xc8, 0x39, 0x26, 0x78, 0xdd,
    0x92, 0x1c, 0xa6, 0xf4, 0x94, 0xa9, 0xc4, 0x69,
    0xf9, 0x7e, 0xea, 0x9c, 0xd6, 0x92, 0xd6, 0xab,
    0xd5, 0x1d, 0xde, 0x4e, 0x8d, 0x6c, 0xb2, 0xc3,
    0x8a, 0xcb, 0xdb, 0xfc, 0x2d, 0xb6, 0xc6, 0x16,
    0x49, 0xd1, 0x31, 0x51, 0x90, 0x0c, 0x75, 0x6e,
    0x27, 0x3b, 0xda, 0x42, 0x0b, 0xdc, 0xd1, 0xea,
    0xde, 0xd9, 0xbe, 0x02, 0xb8, 0xb0, 0x55, 0xba,
    0xde, 0x0c, 0xcf, 0x68, 0xd2, 0x02, 0x7f, 0x16,
    0x5a, 0x63, 0xcc, 0xd1, 0x42, 0xb3, 0x2f, 0x08,
    0x4a, 0x8c, 0x06, 0xa9, 0x13, 0xa2, 0xca, 0xf0,
    0x72, 0xff, 0xd1, 0x77, 0x58, 0xcd, 0x8d, 0x49,
    0xca, 0x67, 0x9a, 0x21, 0x58, 0x94, 0xa6, 0x80,
    0x57, 0xb4, 0xfb, 0x7a, 0x76, 0x30, 0xe8, 0xa1,
    0xef, 0x45, 0x6f, 0xa9, 0x62, 0xf5, 0x51, 0xfa,
    0x87, 0x56, 0x07, 0x9b, 0xc3, 0xab, 0xd6, 0x3c,
    0xa4, 0x3a, 0x02, 0x1f, 0xcd, 0xaa, 0xce, 0x2c,
    0xc9, 0x49, 0x30, 0x27, 0x3e, 0x27, 0x92, 0x59,
    0xff, 0xb7, 0x35, 0x6e, 0xe7, 0x7e, 0x94, 0x06,
    0xf0, 0x2b, 0x00, 0x58, 0x3f, 0x5e, 0x28, 0x12,
    0x70, 0x1c, 0x6b, 0xef, 0x78, 0x22, 0x12, 0x03,
    0xd7, 0x12, 0xd5, 0x46, 0xd6, 0xf2, 0xea, 0x16,
    0x94, 0x2c, 0x1a, 0xfa, 0x52, 0x05, 0xed, 0xf1,
    0x25, 0xf4, 0x06, 0x8b, 0x85, 0x7c, 0x38, 0x70,
    0x53, 0x54, 0x41, 0x6c, 0x8a, 0x0a, 0x60, 0x83,
    0x59, 0x7d, 0x73, 0xe0, 0xd8, 0xa5, 0x1b, 0x3a,
    0xec, 0xcd, 0x5f, 0x44, 0x0b, 0xeb, 0xcc, 0xe9,
    0x65, 0xd4, 0x68, 0xb0, 0x3d, 0x13, 0xf1, 0xb9,
    0x2a, 0x67, 0xb7, 0xae, 0x4f, 0xf3, 0xa7, 0x2e,
    0x33, 0xf3, 0x80, 0xc1, 0x4f, 0xd1, 0xcd, 0xdd,
    0xd2, 0xab, 0xde, 0xcb, 0x04, 0xc8, 0x1b, 0x26,
    0xd4, 0x86, 0xd8, 0x6f, 0xe1, 0x2b, 0x85, 0xb9,
    0x2c, 0xe3, 0xaf, 0xf5, 0x9f, 0x4b, 0xfb, 0x41,
    0xa9, 0xc9, 0x2e, 0xd0, 0x2d, 0xda, 0xba, 0xa0,
    0xba, 0xfb, 0x14, 0x5d, 0x2e, 0xbf, 0x65, 0xbd,
    0xe3, 0x51, 0x8d, 0xdd, 0xbb, 0x51, 0x35, 0x3a,
    0xf0, 0xbe, 0x74, 0xae, 0x42, 0x8b, 0x5d, 0xd9,
    0xef, 0x67, 0x1c, 0x01, 0xb5, 0x1c, 0xc9, 0x3d,
    0x0c, 0x42, 0x3a, 0xce, 0x53, 0xbb, 0x23, 0xd0,
    0x92, 0x2c, 0x60, 0x0f, 0xf1, 0x73, 0x37, 0xa9,
    0x99, 0x86, 0xd9, 0x7e, 0xfc, 0x8a, 0x56, 0xa2,
    0xf7, 0x17, 0x69, 0x75, 0x49, 0x1a, 0x8e, 0x4c,
    0x4b, 0x87, 0xe0, 0xfb, 0xcc, 0x93, 0x14, 0x7d,
    0x96, 0xdc, 0xcc, 0xc6, 0x3c, 0x8d, 0x71, 0x50,
    0xf2, 0x0f, 0x67, 0xdd, 0xd9, 0x95, 0xb7, 0xc9,
    0xd8, 0xf4, 0x1a, 0xca, 0x0c, 0x90, 0xe4, 0xf5,
    0x0e, 0xf7, 0xfb, 0x8c, 0x4c, 0x92, 0x66, 0x3c,
    0x3d, 0xf2, 0x47, 0x6f, 0x8c, 0x51, 0x71, 0xff,
    0xe1, 0xba, 0x1f, 0x02, 0x64, 0x12, 0x79, 0xa5,
    0xad, 0x6e, 0x7c, 0xab, 0xf1, 0x06, 0x01, 0xc1,
    0x25, 0x4e, 0xd6, 0xb6, 0xc3, 0xe9, 0xbe, 0x64,
    0x52, 0x18, 0x22, 0x01, 0x23, 0xc6, 0xd6, 0xbb,
    0x13, 0xbe, 0x15, 0x40, 0xc4, 0x97, 0x4b, 0x24,
    0xe7, 0xaa, 0xef, 0x3b, 0x24, 0x04, 0x47, 0xcc,
    0x4d, 0xd8, 0xd2, 0x9f, 0xf6, 0x60, 0xb9, 0x04,
    0x27, 0xc0, 0x1c, 0xbc, 0x28, 0xe6, 0x16, 0xa4,
    0x06, 0xa8, 0xab, 0x52, 0x56, 0x58, 0x53, 0x4a,
    0xda, 0x0d, 0xf0, 0xb2, 0x8a, 0x8e, 0x7d, 0xd7,
    0x29, 0x45, 0xe9, 0xc6, 0xc3, 0xf4, 0xaa, 0x16,
    0x01, 0x48, 0x1d, 0x6c, 0x90, 0xbe, 0x49, 0x5c,
    0x28, 0x49, 0x18, 0xd5, 0x52, 0xd5, 0x12, 0x54,
    0x9c, 0x1f, 0xc0, 0xb5, 0x8f, 0x22, 0xa7, 0x2a,
    0xd7, 0x36, 0x41, 0xc2, 0xc1, 0x7d, 0x11, 0x9b,
    0xbf, 0x84, 0x2a, 0x05, 0x9f, 0x41, 0xea, 0x54,
    0xf2, 0xe1, 0x69, 0x4d, 0xd5, 0x63, 0x00, 0x28,
    0x81, 0x75, 0xd3, 0x8c, 0x19, 0x81, 0xd6, 0x0a,
    0x25, 0x62, 0x9e, 0x35, 0xd0, 0x4d, 0xc7, 0x73,
    0x3e, 0xae, 0x39, 0x89, 0xf9, 0xb6, 0x67, 0xf0,
    0x08, 0x98, 0xd7, 0x87, 0xf1, 0xf4, 0x58, 0x12,
    0xf5, 0x22, 0x51, 0xad, 0x9a, 0x4d, 0xed, 0x09,
    0x81, 0x82, 0x99, 0x9b, 0x5f, 0x05, 0xb6, 0x3e,
    0x8e, 0xad, 0x34, 0xa1, 0x2d, 0xb6, 0xae, 0xc7,
    0x7b, 0x7f, 0x39, 0xf9, 0x53, 0xaf, 0xcd, 0x0e,
    0xcc, 0x98, 0x5b, 0x5b, 0x92, 0x8c, 0x75, 0x36,
    0x0a, 0xa9, 0xfc, 0xf5, 0x22, 0x14, 0xf3, 0x44,
    0x5e, 0xc7, 0xf6, 0x7f, 0x3a, 0x92, 0x0b, 0xc1,
    0x68, 0xa3, 0xc2, 0xd8, 0xf3, 0xd1, 0x12, 0x5b,
    0x12, 0xd1, 0x96, 0x18, 0x50, 0x0c, 0xb5, 0x13,
    0x40, 0xcb, 0x38, 0xdd, 0xf7, 0xc2, 0x4f, 0xb9,
    0xda, 0xd2, 0x94, 0x3f, 0xbf, 0x63, 0x4f, 0x0a,
    0xb8, 0x03, 0x67, 0x65, 0xc6, 0x40, 0xff, 0xc8,
    0xd7, 0x8e, 0x06, 0x92, 0x7a, 0xdb, 0xad, 0x7d,
    0x1b, 0x82, 0xd5, 0x19, 0xf1, 0x4a, 0x11, 0x3a,
    0xe4, 0x23, 0x39, 0x8d, 0x5d, 0x0a, 0x63, 0xb2,
    0x67, 0x1f, 0x98, 0xde, 0x5a, 0x44, 0x6a, 0x70,
    0xff, 0x83, 0x34, 0x78, 0xa5, 0x02, 0x0b, 0x51,
    0x4b, 0x0a, 0xab, 0x16, 0x04, 0x2e, 0xce, 0x6a,
    0xf2, 0x72, 0x4c, 0x6f, 0x2f, 0xfa, 0xca, 0xd8,
    0x19, 0x4f, 0x48, 0x9f, 0x1c, 0x7d, 0x60, 0x34,
    0x9c, 0x79, 0x44, 0xd3, 0xaa, 0xea, 0x5c, 0xfe,
    0xa0, 0xa1, 0x6e, 0xff, 0xa8, 0xb6, 0xe3, 0xd9,
    0x60, 0xf2, 0x93, 0x32, 0x77, 0x07, 0xd6, 0xbb,
    0xdf, 0x3a, 0x49, 0x38, 0xad, 0xc4, 0x63, 0xaf,
    0xd0, 0x00, 0x41, 0xdb, 0x96, 0x75, 0xbb, 0xb3,
    0xa8, 0x10, 0x67, 0x7e, 0x47, 0x89, 0x8c, 0xce,
    0xdd, 0xe8, 0x8b, 0xf3, 0xdb, 0xd7, 0xf5, 0x4f,
    0x22, 0xbf, 0x19, 0x61, 0x64, 0xf7, 0xe3, 0xcd,
    0x4e, 0x3d, 0x19, 0x8e, 0x45, 0x8c, 0xce, 0xd3,
    0x65, 0x3e, 0xb0, 0xdc, 0x94, 0x2e, 0xa2, 0xc2,
    0x53, 0xba, 0x46, 0x3a, 0xd1, 0x39, 0xd4, 0x7a,
    0xd8, 0xd6, 0x95, 0xb7, 0xdb, 0xdd, 0x7a, 0x81,
    0x78, 0xac, 0xfe, 0x0a, 0x1c, 0x74, 0x3f, 0x52,
    0xa3, 0x4d, 0xee, 0x4d, 0x37, 0x15, 0xd7, 0x39,
    0x06, 0x59, 0xec, 0xa7, 0x92, 0xb2, 0x65, 0xee,
    0x49, 0x63, 0xf4, 0x01, 0x54, 0xc2, 0xa0, 0xab,
    0xfc, 0x10, 0x03, 0x30, 0x22, 0x63, 0xdb, 0x6e,
    0xed, 0xd9, 0x1c, 0x28, 0xb1, 0x48, 0x09, 0xa8,
    0x1c, 0x97, 0x09, 0xda, 0xaa, 0x3e, 0xd7, 0xc3,
    0x65, 0x2f, 0x81, 0x22, 0x8e, 0x4e, 0x0e, 0x71,
    0xaa, 0x96, 0xe4, 0x23, 0xd7, 0xa1, 0x1f, 0xcb,
    0x8c, 0x9e, 0xa0, 0x39, 0xc4, 0xb1, 0x01, 0x68,
    0x90, 0x61, 0x58, 0xcd, 0x62, 0xa0, 0xa9, 0xeb,
    0xb3, 0x07, 0xb4, 0xf5, 0xf9, 0xb6, 0xdf, 0x05,
    0xc6, 0x03, 0x7a, 0xa5, 0x1e, 0x8c, 0x12, 0x79,
    0x9a, 0xe3, 0x2b, 0x22, 0x25, 0x9a, 0x52, 0x68,
    0x4b, 0x47, 0x71, 0xac, 0x71, 0x3e, 0x1a, 0x14,
    0x7f, 0x34, 0x4d, 0xdc, 0x4e, 0x49, 0xf6, 0x5a,
    0x91, 0x5b, 0x44, 0xc8, 0xef, 0x67, 0x9c, 0xfe,
    0xb3, 0x04, 0xa9, 0x4f, 0xa8, 0xf3, 0x89, 0xdf,
    0x89, 0x83, 0xff, 0xd2, 0x45, 0x95, 0x66, 0x56,
    0x23, 0x84, 0x01, 0x13, 0x90, 0x5a, 0x7a, 0x04,
    0xc3, 0x97, 0x60, 0xf2, 0x86, 0x35, 0x5c, 0x45,
    0x6f, 0xcf, 0x27, 0xbc, 0xbb, 0xc3, 0x52, 0xf6,
    0xb2, 0xd6, 0x5a, 0xe0, 0xbc, 0x14, 0xc8, 0xfd,
    0x57, 0x38, 0x61, 0x8e, 0xc2, 0x58, 0x5e, 0xcc,
    0x0d, 0x86, 0xb6, 0xf3, 0x27, 0x92, 0x4a, 0x44,
    0xde, 0x27, 0x4b, 0xf9, 0xc3, 0x62, 0x5d, 0xe2,
    0x72, 0xfe, 0xae, 0x22, 0x8f, 0x31, 0x52, 0x99,
    0xda, 0x94, 0xd3, 0xaa, 0x73, 0x49, 0xdb, 0x48,
    0x1a, 0x1f, 0x0b, 0x7b, 0x11, 0x3f, 0xcd, 0x57,
    0xc4, 0x4e, 0x7b, 0x3e, 0xc6, 0x82, 0x8a, 0x17,
    0x41, 0xe3, 0x00, 0x55, 0xec, 0x82, 0x9f, 0x65,
    0x30, 0x4d, 0xfa, 0xc1, 0x91, 0x16, 0x89, 0x4d,
    0xec, 0x5a, 0xee, 0x17, 0xa0, 0xaf, 0x03, 0xb4,
    0x04, 0x01, 0x54, 0xa6, 0x13, 0xad, 0x7c, 0x49,
    0x17, 0x75, 0x0c, 0x4a, 0xe6, 0xf8, 0x7c, 0xa2,
    0xf3, 0x53, 0x84, 0xb1, 0xd3, 0x23, 0x53, 0xc7,
    0x35, 0xdf, 0x66, 0xee, 0x0f, 0xbc, 0x8e, 0x2c,
    0x21, 0x43, 0x46, 0x97, 0x89, 0x90, 0x35, 0x18,
    0xcc, 0xd9, 0x3a, 0xc8, 0x58, 0x1c, 0x52, 0x94,
    0x57, 0x4d, 0x5a, 0x37, 0xc3, 0xfb, 0x65, 0xdd,
    0xbe, 0x96, 0x97, 0xdf, 0xda, 0x73, 0x60, 0x9a,
    0x45, 0x1e, 0x18, 0xf4, 0xea, 0x5e, 0xfc, 0x0e,
    0x39, 0xf3, 0x81, 0x88, 0x2a, 0xe5, 0xe1, 0x91,
    0xed, 0x37, 0x32, 0x63, 0x4d, 0xd3, 0x90, 0xcb,
    0x86, 0xc0, 0x5c, 0xbd, 0x04, 0x7e, 0x13, 0x69,
    0xa0, 0xe1, 0x1f, 0xbd, 0x86, 0x33, 0xf4, 0x99,
    0xd1, 0x4e, 0xde, 0x60, 0xbb, 0xd2, 0xd5, 0x9d,
    0x3f, 0x15, 0x01, 0x79, 0x4a, 0x03, 0x5e, 0x39,
    0x23, 0x99, 0xbb, 0x27, 0x8d, 0x93, 0x5c, 0x13,
    0x6e, 0xbf, 0xc0, 0x3a, 0x32, 0xf9, 0xc0, 0xf4,
    0xb4, 0x73, 0x16, 0x63, 0xba, 0x9d, 0x2c, 0xb1,
    0x08, 0x63, 0x69, 0xa4, 0x71, 0x0d, 0x67, 0x06,
    0xd3, 0x48, 0x0e, 0x7a, 0x53, 0xe0, 0xdd, 0x19,
    0xc2, 0xfc, 0x96, 0x07, 0xbc, 0xe6, 0x3d, 0x76,
    0x73, 0xcb, 0x7f, 0x0f, 0xe1, 0x9a, 0x70, 0xd0,
    0x0c, 0xf7, 0xc9, 0x96, 0x04, 0x70, 0x4a, 0xf8,
    0x9a, 0x68, 0x12, 0x1c, 0xfa, 0x2b, 0x96, 0x9c,
    0x3f, 0x45, 0xf9, 0x02, 0x80, 0xf4, 0xcd, 0x60,
    0xb3, 0xb2, 0x03, 0x69, 0x3f, 0x3f, 0x81, 0x2f,
    0xfc, 0x9e, 0x6a, 0x0a, 0x72, 0xdb, 0x65, 0xe0,
    0x20, 0xb9, 0x3e, 0x0f, 0xaa, 0xc7, 0x68, 0x29,
    0xd1, 0xfe, 0xbf, 0x73, 0x32, 0xcb, 0x97, 0xa8,
    0x67, 0x77, 0x08, 0x39, 0x0d, 0x1d, 0x01, 0xec,
    0x5a, 0x7f, 0x60, 0x69, 0x13, 0x47, 0xa1, 0x7f,
    0xfc, 0x0c, 0xee, 0x51, 0x6f, 0x53, 0x16, 0x2f,
    0xd5, 0x50, 0x9d, 0x99, 0xe7, 0xa8, 0xf7, 0xda,
    0xb2, 0x71, 0xcd, 0x3d, 0xaa, 0x40, 0x6d, 0x7a,
    0x51, 0x76, 0xb2, 0xca, 0xc2, 0xc8, 0x88, 0x30,
    0xf6, 0xa9, 0x9c, 0xb4, 0x7e, 0x24, 0xce, 0x20,
    0x0f, 0x10, 0xf2, 0xdb, 0xad, 0x43, 0x5e, 0x0c,
    0x3d, 0x27, 0x6a, 0xed, 0x38, 0x2b, 0xe2, 0xd0,
    0x91, 0xff, 0x5b, 0x7c, 0x0c, 0xed, 0x87, 0xa7,
    0xd5, 0x09, 0x79, 0xfd, 0xe5, 0xeb, 0x26, 0xe0,
    0xa9, 0x71, 0xae, 0xd5, 0xba, 0xc4, 0x8a, 0xdf,
    0x0e, 0xbf, 0x0e, 0x8a, 0x6a, 0x40, 0x3d, 0x48,
    0x06, 0x53, 0x57, 0x43, 0xc6, 0x91, 0x4a, 0x3a,
    0x19, 0xc4, 0x06, 0x85, 0xbb, 0x80, 0x6e, 0x0f,
    0xf4, 0x54, 0xa9, 0xab, 0x2e, 0x85, 0xe8, 0x89,
    0x48, 0x55, 0xe3, 0xca, 0x74, 0xf0, 0x59, 0x47,
    0x01, 0x1d, 0x6b, 0xcd, 0x6c, 0xd3, 0x00, 0x9a,
    0x82, 0xaf, 0x1e, 0x29, 0x82, 0xdb, 0xd3, 0x48,
    0x24, 0x3e, 0x61, 0xf9, 0x40, 0xc7, 0x28, 0xc6,
    0xe3, 0x27, 0x71, 0xf0, 0x37, 0xeb, 0xf4, 0x30,
    0xdd, 0x1e, 0xf1, 0x1f, 0xa6, 0xc0, 0x7d, 0x8c,
    0x83, 0xac, 0xd5, 0x91, 0x3a, 0x9f, 0x8b, 0x03,
    0xe6, 0x7b, 0xef, 0x25, 0x57, 0xe9, 0x37, 0x52,
    0x01, 0xa1, 0xf6, 0xa9, 0xaf, 0xbf, 0x43, 0xf2,
    0x43, 0x79, 0x32, 0xf5, 0x03, 0x70, 0xde, 0x3c,
    0x41, 0x4e, 0x7f, 0xeb, 0x9d, 0x0e, 0x91, 0x45,
    0x0c, 0x98, 0xdd, 0x65, 0xcb, 0x41, 0xde, 0x75,
    0xa8, 0xb6, 0xe0, 0xd6, 0xe9, 0xbe, 0xe7, 0xd9,
    0xb0, 0x01, 0x27, 0x0c, 0xf1, 0xc0, 0x20, 0xed,
    0x7b, 0x17, 0xb2, 0xfa, 0x56, 0x9b, 0x5a, 0xf5,
    0xb1, 0x2e, 0x36, 0xd0, 0x88, 0x2b, 0x00, 0xc5,
    0x02, 0x4a, 0xa0, 0xda, 0x65, 0xb6, 0x7b, 0xa1,
    0x7b, 0x1e, 0x28, 0x72, 0xcf, 0x4f, 0xa3, 0xe6,
    0x21, 0x14, 0x25, 0x40, 0xc4, 0x41, 0xda, 0xde,
    0xd4, 0x10, 0xca, 0x02, 0x95, 0x31, 0xcd, 0xfc,
    0xe9, 0x7b, 0xb8, 0x0f, 0xe3, 0xf1, 0xcb, 0x2e,
    0x8f, 0xd4, 0x3e, 0xfa, 0xdc, 0x5f, 0x4c, 0xc3,
    0xc1, 0x38, 0x8e, 0xbe, 0xf8, 0x21, 0x56, 0xaf,
    0xba, 0x42, 0xb7, 0x76, 0xd8, 0xfc, 0xa3, 0x6c,
    0x07, 0x62, 0xa2, 0xb5, 0x27, 0xe0, 0x18, 0x7c,
    0x57, 0xe3, 0x6d, 0xfe, 0x47, 0x56, 0x61, 0xd5,
    0xda, 0x17, 0x44, 0x7b, 0x47, 0xea, 0x43, 0x62,
    0xcf, 0x11, 0xe4, 0x6b, 0x26, 0x82, 0x24, 0x13,
    0x63, 0xc6, 0x14, 0xbb, 0x00, 0xad, 0xe6, 0xe7,
    0x32, 0x4e, 0x79, 0xae, 0xc0, 0x1a, 0x51, 0x78,
    0xc0, 0xd4, 0x13, 0x02, 0xb1, 0x55, 0xcf, 0xf0,
    0x92, 0xaa, 0xad, 0x8f, 0x89, 0x4d, 0xfc, 0x2e,
    0x3b, 0x1a, 0x22, 0xb2, 0x4e, 0xef, 0xdb, 0x98,
    0x56, 0x0f, 0x7a, 0x0a, 0x5e, 0xf6, 0xde, 0x8e,
    0xd5, 0xdc, 0x84, 0x72, 0xe6, 0xa9, 0xd2, 0x5f,
    0xd5, 0x66, 0xab, 0x1e, 0x8b, 0xf2, 0x28, 0x9b,
    0x77, 0xc5, 0x24, 0x04, 0x93, 0x85, 0x54, 0x5b,
    0xcc, 0x0a, 0x18, 0xdc, 0xa9, 0x27, 0x34, 0x1d
};

static const unsigned char ml_dsa_65_sig[] = {
    0x5e, 0xc1, 0xb0, 0x64, 0x5c, 0x45, 0x6e, 0x22,
    0x9f, 0x7c, 0x49, 0xbf, 0x8e, 0x93, 0xe5, 0x83,
    0xfc, 0xa0, 0xf3, 0xb0, 0x93, 0x6f, 0x47, 0xa9,
    0x11, 0x9e, 0x1b, 0x13, 0xae, 0x05, 0xdd, 0xc6,
    0x55, 0x46, 0x81, 0x4f, 0x5e, 0x0c, 0xa7, 0x75,
    0x6b, 0x5c, 0x13, 0xf2, 0xc7, 0x1f, 0x4d, 0x74,
    0xa5, 0x50, 0xf4, 0xec, 0x3e, 0x75, 0x75, 0xbf,
    0x4c, 0xef, 0x50, 0x02, 0x87, 0x7d, 0x88, 0x05,
    0x40, 0xba, 0xc1, 0x8e, 0xd6, 0x5d, 0xe3, 0x3c,
    0xd9, 0x98, 0x05, 0x76, 0x09, 0xdf, 0xcb, 0xd2,
    0x0e, 0x56, 0x53, 0x88, 0x1e, 0xe1, 0x91, 0xcd,
    0x7d, 0x67, 0x11, 0x89, 0xf8, 0x58, 0xb9, 0x76,
    0xd6, 0x8f, 0x83, 0xca, 0xcc, 0x94, 0xf4, 0x97,
    0x58, 0xf5, 0x3d, 0xc7, 0xaa, 0xa4, 0x3f, 0xe3,
    0xcb, 0xcc, 0x67, 0xac, 0x50, 0x8f, 0xfe, 0xf7,
    0x90, 0x33, 0x04, 0x8b, 0x82, 0x14, 0x26, 0xa7,
    0x21, 0x4e, 0xf2, 0x37, 0x5a, 0x91, 0xe8, 0x7a,
    0x30, 0xb6, 0xce, 0x8f, 0x31, 0xfc, 0xef, 0xbe,
    0x7d, 0x16, 0x81, 0x18, 0xb3, 0x77, 0x0f, 0x4a,
    0xe9, 0x31, 0xf8, 0x51, 0x0e, 0xaa, 0x44, 0x5c,
    0x8b, 0x43, 0x0a, 0x61, 0x02, 0x6c, 0xdf, 0xee,
    0x09, 0x9b, 0x8f, 0x91, 0x5c, 0x75, 0x10, 0xa8,
    0xc6, 0xe3, 0x5d, 0x93, 0x74, 0x63, 0x73, 0x27,
    0xb9, 0x29, 0xc0, 0x9b, 0x62, 0x6a, 0xc6, 0x75,
    0x84, 0x78, 0x41, 0xf3, 0x5c, 0x5d, 0x60, 0xcf,
    0xc5, 0xe4, 0x78, 0x87, 0x53, 0x6c, 0xab, 0xff,
    0xbb, 0x53, 0xe9, 0x54, 0xf7, 0x1e, 0x06, 0xcf,
    0x9e, 0x52, 0xd0, 0x3c, 0x8c, 0x5e, 0x64, 0xba,
    0x28, 0x33, 0xb0, 0x14, 0x84, 0x56, 0x5a, 0xac,
    0xd6, 0xee, 0x1e, 0x5e, 0x39, 0x08, 0xcb, 0x16,
    0x54, 0x21, 0xe4, 0x6c, 0xa7, 0x29, 0xd2, 0x7b,
    0xbe, 0x09, 0xd5, 0x1c, 0xf2, 0xf8, 0x11, 0x1e,
    0x64, 0x6b, 0xca, 0x1a, 0xe4, 0xca, 0xab, 0xf1,
    0x13, 0xdc, 0xa2, 0x1e, 0x78, 0x09, 0x2a, 0x08,
    0xc5, 0xa1, 0x42, 0x62, 0xac, 0x94, 0x46, 0x61,
    0x37, 0x98, 0x0d, 0x93, 0xa6, 0xc2, 0x4d, 0xb8,
    0x00, 0x43, 0x4d, 0x6c, 0x5d, 0x06, 0x30, 0xaf,
    0xd5, 0xb6, 0x32, 0x9e, 0xfe, 0xf2, 0x03, 0x8b,
    0xa6, 0x7c, 0x6d, 0x36, 0x7e, 0xef, 0x99, 0x09,
    0xaf, 0x2c, 0x79, 0x70, 0x26, 0xff, 0xc8, 0x99,
    0x01, 0x93, 0x3b, 0x9c, 0xde, 0x0b, 0x5e, 0xf5,
    0x44, 0x5f, 0x15, 0xb6, 0x49, 0xde, 0x89, 0xdd,
    0x1e, 0x2b, 0x69, 0x7d, 0xb2, 0x88, 0xf2, 0x94,
    0x24, 0x49, 0xeb, 0x23, 0xfc, 0x60, 0x95, 0x09,
    0x89, 0xe5, 0xfd, 0xe3, 0x63, 0xfa, 0x3a, 0xb7,
    0x18, 0x99, 0xe0, 0x5a, 0x25, 0x94, 0xd6, 0x3a,
    0x6a, 0x48, 0x6e, 0xea, 0xe3, 0x61, 0xf8, 0x81,
    0xe8, 0x4a, 0x43, 0xcb, 0x19, 0xfe, 0x2e, 0x16,
    0x8d, 0xca, 0x7e, 0xeb, 0x81, 0x55, 0xc0, 0x8a,
    0xf5, 0x14, 0x5b, 0xb6, 0x06, 0x45, 0x72, 0xa6,
    0x8a, 0xc2, 0x16, 0x8d, 0xe4, 0x31, 0x54, 0x9c,
    0xd2, 0x75, 0x5f, 0x7d, 0xf2, 0x19, 0x19, 0xf4,
    0x4a, 0xfe, 0xaf, 0x18, 0x2d, 0x51, 0x43, 0x2e,
    0x40, 0x02, 0xc0, 0x25, 0x53, 0x10, 0x37, 0x3d,
    0xeb, 0xd6, 0x08, 0x75, 0x6c, 0xb2, 0x38, 0x72,
    0xef, 0x67, 0x80, 0x50, 0xc7, 0x54, 0x4c, 0x91,
    0xcb, 0x68, 0x64, 0x23, 0xa9, 0x4b, 0x0d, 0xcb,
    0x71, 0x9c, 0x27, 0x67, 0x22, 0x98, 0xad, 0x3a,
    0xa0, 0xd8, 0x9f, 0x21, 0x95, 0xdc, 0xb0, 0xa5,
    0xb4, 0x44, 0x3b, 0xc8, 0xcb, 0x5c, 0x50, 0xd1,
    0x38, 0x03, 0x91, 0x79, 0x73, 0x2f, 0x4a, 0xd7,
    0x1f, 0xe5, 0xdc, 0x16, 0xaf, 0xb1, 0x2a, 0x86,
    0x02, 0x0b, 0x31, 0x42, 0x64, 0x2f, 0x8a, 0x83,
    0x7b, 0x13, 0x78, 0xdd, 0x6c, 0x72, 0x46, 0xa2,
    0x84, 0x2f, 0xa8, 0x24, 0x43, 0xb0, 0x69, 0xaf,
    0x52, 0x58, 0x13, 0x7c, 0xb3, 0xf8, 0x2d, 0x74,
    0x87, 0x68, 0x17, 0xd6, 0x0c, 0x50, 0x19, 0xf8,
    0x11, 0x68, 0xd5, 0x1c, 0x56, 0x30, 0x5d, 0x31,
    0x56, 0xe4, 0x9d, 0x80, 0x39, 0x0c, 0xcd, 0x34,
    0xfe, 0x67, 0x2f, 0xfa, 0x0c, 0x10, 0xb2, 0xcd,
    0x66, 0x9e, 0x65, 0xac, 0xc5, 0x45, 0x33, 0x20,
    0x16, 0x79, 0x82, 0x48, 0x26, 0xaa, 0x54, 0xd9,
    0x32, 0x03, 0x99, 0x46, 0x65, 0xf8, 0xa8, 0xe7,
    0x3c, 0xb1, 0xf9, 0xed, 0x76, 0x21, 0x36, 0xab,
    0x98, 0x2d, 0xf2, 0xd1, 0x75, 0x33, 0x58, 0xb5,
    0x7c, 0x4e, 0x3b, 0x78, 0xb6, 0x24, 0x15, 0xd2,
    0xac, 0x41, 0x44, 0x46, 0x4d, 0x7a, 0x5b, 0x86,
    0xce, 0xc2, 0x93, 0x11, 0x8f, 0x68, 0xb5, 0xc7,
    0x83, 0x8f, 0xe1, 0x13, 0x91, 0xef, 0xca, 0x18,
    0xa3, 0xf1, 0x04, 0x28, 0x58, 0xd8, 0x54, 0x12,
    0x03, 0xbe, 0xe1, 0x29, 0xcf, 0x28, 0xa3, 0x43,
    0x78, 0x00, 0xb7, 0x52, 0x00, 0xe3, 0x44, 0x07,
    0x5f, 0x67, 0xb1, 0x78, 0x75, 0x9c, 0xe0, 0xc7,
    0x33, 0x5f, 0x69, 0x27, 0x05, 0x15, 0xc3, 0xd4,
    0x81, 0x09, 0x33, 0x7b, 0x3e, 0xc4, 0x71, 0xc3,
    0xc0, 0x9a, 0x2c, 0xb5, 0xfc, 0x00, 0x79, 0x1d,
    0x12, 0xb4, 0x90, 0xf1, 0xa3, 0xb1, 0xaa, 0xa8,
    0xf9, 0x40, 0x48, 0xb1, 0x8b, 0x3c, 0xa9, 0xa9,
    0x36, 0xe0, 0xe8, 0xd3, 0x43, 0x4f, 0x63, 0xcc,
    0xe1, 0xbf, 0x93, 0xa5, 0xb0, 0x7a, 0xac, 0xd6,
    0x69, 0x42, 0xce, 0x52, 0x8c, 0xd8, 0x30, 0x18,
    0x9f, 0x77, 0xa2, 0xd6, 0xde, 0x5a, 0x3d, 0x59,
    0x28, 0x3a, 0x2a, 0x58, 0x25, 0xcb, 0x95, 0x01,
    0x9d, 0x04, 0x81, 0x1d, 0x63, 0xfb, 0xb1, 0xf3,
    0x0b, 0x3d, 0x68, 0xc1, 0x42, 0x83, 0xc5, 0xc2,
    0x27, 0x1e, 0x58, 0x4f, 0x42, 0x89, 0x45, 0x08,
    0xb2, 0xf2, 0x8d, 0x26, 0xd1, 0xb3, 0x4d, 0xbd,
    0xaf, 0x74, 0xc9, 0x7f, 0xbe, 0x81, 0x46, 0x73,
    0xdb, 0x68, 0x45, 0x41, 0xb1, 0xbe, 0xe7, 0x76,
    0x31, 0x01, 0xc8, 0x8c, 0xb5, 0x26, 0x9e, 0x2e,
    0xde, 0x07, 0x9f, 0x5b, 0x7e, 0x73, 0x36, 0xbf,
    0x51, 0x60, 0x98, 0xe0, 0x6e, 0x83, 0x85, 0x21,
    0x2c, 0xdb, 0x28, 0x2c, 0x8f, 0x4b, 0xf9, 0x15,
    0x08, 0x8f, 0x06, 0xa8, 0xfe, 0x26, 0x2b, 0xea,
    0xaf, 0xb0, 0x97, 0x3c, 0xb9, 0xf8, 0x25, 0xef,
    0xf5, 0x58, 0xd2, 0xf4, 0xbd, 0x1c, 0x1b, 0x84,
    0x29, 0xde, 0xbe, 0xb6, 0xfd, 0x30, 0x47, 0x0c,
    0x63, 0x3a, 0x3e, 0x6d, 0x8b, 0x0b, 0x1e, 0x64,
    0x5d, 0x8f, 0x64, 0xc9, 0x35, 0x7a, 0xb4, 0xb9,
    0xfb, 0x47, 0xe4, 0x50, 0xcd, 0xf6, 0x1a, 0xb1,
    0xf6, 0x22, 0x0a, 0xd5, 0x63, 0xb5, 0x20, 0xed,
    0x13, 0xfb, 0x43, 0x11, 0x67, 0x89, 0x6d, 0x9f,
    0xb1, 0x6f, 0xe8, 0x48, 0x2f, 0x7e, 0xda, 0xbc,
    0x08, 0xc2, 0xde, 0x1c, 0x29, 0xa4, 0x50, 0x7d,
    0x9c, 0x8e, 0xbc, 0x0d, 0x2b, 0xe2, 0xda, 0xb6,
    0x9c, 0x6c, 0x08, 0xd3, 0x3c, 0xf6, 0x71, 0x23,
    0x90, 0x07, 0x8f, 0x99, 0x86, 0x85, 0xf8, 0xaa,
    0xed, 0x65, 0x99, 0xf3, 0xf5, 0x69, 0xe2, 0x6b,
    0x67, 0x47, 0x5c, 0x8d, 0x5d, 0xee, 0x9c, 0xda,
    0xa1, 0x56, 0x73, 0x35, 0xf1, 0xe7, 0x2e, 0xb9,
    0xf9, 0x70, 0xc5, 0xb4, 0xdd, 0x21, 0x9f, 0x25,
    0x23, 0xf4, 0x74, 0x57, 0xc1, 0xfd, 0x09, 0x2e,
    0x80, 0xea, 0x4d, 0x90, 0x93, 0x84, 0x06, 0x43,
    0xc0, 0xbc, 0x26, 0xd2, 0xfd, 0x3b, 0x03, 0x2e,
    0x93, 0x90, 0xb3, 0x4b, 0xd4, 0x58, 0x31, 0x18,
    0xc5, 0xa5, 0xc4, 0x3c, 0xe5, 0xb5, 0x23, 0xfb,
    0xec, 0xb0, 0xb8, 0xe7, 0x7e, 0x1a, 0xb3, 0xf9,
    0xce, 0x5a, 0x25, 0x02, 0x2e, 0xcf, 0xb3, 0x4a,
    0xe3, 0x5e, 0xd5, 0xc8, 0x78, 0x0a, 0x89, 0x7c,
    0x3e, 0x30, 0x8a, 0xd5, 0xb0, 0x60, 0xc1, 0x52,
    0x7c, 0xbc, 0x9e, 0x47, 0x0a, 0xa7, 0xc3, 0xee,
    0x11, 0x06, 0x72, 0x89, 0x23, 0x8b, 0x51, 0xb8,
    0xbb, 0x5d, 0xc1, 0xf0, 0x38, 0x66, 0x8e, 0xa7,
    0x43, 0xad, 0xc1, 0x40, 0x80, 0x58, 0xe4, 0xdf,
    0x0d, 0xb1, 0x2b, 0xa2, 0xd4, 0x58, 0x20, 0x95,
    0xee, 0x51, 0x13, 0x94, 0x1e, 0x8d, 0x1e, 0xd4,
    0x2d, 0x82, 0x73, 0xa6, 0x92, 0x0c, 0xec, 0x74,
    0x12, 0xbe, 0x2a, 0x8c, 0x72, 0xb3, 0x99, 0x7d,
    0x33, 0x50, 0xe4, 0xeb, 0xad, 0x24, 0x7c, 0xef,
    0x1a, 0x41, 0x4b, 0x04, 0xe6, 0x21, 0xa3, 0x1a,
    0xa4, 0x13, 0xf4, 0xdc, 0x70, 0xa2, 0xdc, 0x0d,
    0xc1, 0x02, 0x43, 0xeb, 0xf7, 0x69, 0x82, 0x7a,
    0x99, 0x61, 0x1b, 0x14, 0xaa, 0xce, 0x2b, 0x1b,
    0xd1, 0xaa, 0x3a, 0x99, 0xae, 0x51, 0xae, 0x35,
    0xb9, 0x8c, 0xe0, 0x43, 0x6e, 0x1e, 0x17, 0x36,
    0xf0, 0x9e, 0xf2, 0x83, 0x22, 0x6c, 0x82, 0xee,
    0xfb, 0x72, 0x35, 0x22, 0x45, 0x4a, 0xbc, 0xfa,
    0x65, 0x44, 0xa1, 0xce, 0x6a, 0x6a, 0xe1, 0x3f,
    0x6f, 0xd0, 0x45, 0x72, 0x29, 0xf4, 0xf4, 0xb0,
    0x97, 0x0e, 0x1e, 0xe6, 0xa8, 0xb6, 0xc8, 0x44,
    0x11, 0x99, 0x6e, 0xd9, 0xfd, 0x0b, 0x04, 0xaf,
    0x10, 0x0a, 0xd0, 0xae, 0x5d, 0xff, 0x78, 0x32,
    0x09, 0x39, 0x73, 0x2e, 0xaf, 0x65, 0x70, 0x3b,
    0xd5, 0xfe, 0x90, 0xc0, 0x9c, 0xf5, 0xba, 0xbd,
    0x9d, 0x78, 0xe9, 0x67, 0x87, 0x79, 0xad, 0x1f,
    0x77, 0x12, 0x8e, 0x8d, 0x6e, 0xc3, 0x69, 0xc6,
    0x22, 0x60, 0x2d, 0x4f, 0x49, 0x1b, 0xa7, 0xa3,
    0x8a, 0x14, 0x9d, 0x48, 0xfe, 0x0c, 0x38, 0x9a,
    0x0e, 0xbc, 0x7e, 0x5c, 0x6c, 0x01, 0xc5, 0x68,
    0x39, 0xd1, 0x33, 0xca, 0x23, 0xf6, 0x12, 0x02,
    0x59, 0x3f, 0xe2, 0x11, 0x9d, 0xa0, 0x41, 0x3d,
    0xa5, 0x3b, 0x5a, 0xe1, 0xdc, 0x66, 0x00, 0xcb,
    0x91, 0x39, 0x60, 0xd4, 0xa8, 0xdb, 0x91, 0xf3,
    0x2a, 0x7b, 0xfc, 0xc7, 0x41, 0x66, 0x88, 0x68,
    0x54, 0xff, 0x9f, 0x38, 0x03, 0x8c, 0x93, 0x72,
    0x4c, 0xab, 0xbe, 0xeb, 0x3d, 0x74, 0x74, 0x38,
    0xcf, 0x46, 0x08, 0xfa, 0x72, 0xcf, 0xbf, 0xed,
    0xb6, 0xc1, 0x0c, 0xc5, 0x13, 0xa9, 0x76, 0x67,
    0x6b, 0xe2, 0xad, 0xc4, 0x06, 0xfa, 0x0c, 0x81,
    0xd8, 0xdd, 0x7b, 0x28, 0x3f, 0x15, 0xb7, 0x0d,
    0xb3, 0x56, 0x5a, 0x3c, 0xa6, 0x8f, 0x96, 0x20,
    0xa4, 0x02, 0x6e, 0x0a, 0x7f, 0x8d, 0x97, 0x5a,
    0xb2, 0xf5, 0xa9, 0xd8, 0xb9, 0x0b, 0x72, 0x0c,
    0x6c, 0x6c, 0xcc, 0x31, 0x76, 0x55, 0x1a, 0xde,
    0xf5, 0x17, 0xd9, 0xb0, 0x57, 0xea, 0x39, 0xab,
    0xcd, 0xb7, 0x5f, 0x16, 0x07, 0xdb, 0x36, 0x63,
    0x16, 0x8c, 0x6f, 0x09, 0xe3, 0xc0, 0x43, 0xda,
    0xd0, 0x77, 0x17, 0xb4, 0x72, 0x50, 0xfc, 0x8e,
    0x1d, 0xdd, 0xec, 0x00, 0x2b, 0x21, 0xe4, 0x71,
    0xa4, 0xdd, 0x51, 0xd9, 0xe0, 0xdb, 0x14, 0x21,
    0x42, 0xf2, 0x90, 0x49, 0x25, 0x55, 0x48, 0x47,
    0x58, 0x7b, 0x96, 0x97, 0x42, 0xb2, 0xd5, 0x50,
    0xa9, 0x28, 0xba, 0xec, 0xba, 0x52, 0x6d, 0xd0,
    0xb1, 0x75, 0xd8, 0xd4, 0xda, 0x81, 0x10, 0x0c,
    0xb0, 0x75, 0x57, 0x57, 0x81, 0x7f, 0x8d, 0xf9,
    0x29, 0xaf, 0x27, 0x85, 0x45, 0x4a, 0x84, 0x53,
    0x78, 0xde, 0x09, 0xe9, 0xf5, 0x2c, 0xd4, 0x91,
    0xc3, 0x3a, 0x9c, 0x50, 0xe1, 0xc6, 0x20, 0x66,
    0x26, 0x4a, 0x7b, 0x4f, 0x05, 0x89, 0x97, 0x3b,
    0xe1, 0xb8, 0xbc, 0x28, 0xa4, 0x31, 0xa8, 0x13,
    0xb3, 0xc2, 0xf5, 0xee, 0x10, 0xec, 0x4c, 0xf4,
    0xd6, 0x21, 0xab, 0x9b, 0x58, 0xe8, 0x95, 0x7c,
    0x85, 0x64, 0x57, 0x81, 0x79, 0x1b, 0x8b, 0xb1,
    0x84, 0xd5, 0xe9, 0x86, 0xe1, 0x6f, 0xce, 0x56,
    0xe1, 0x8d, 0x1d, 0xa2, 0x4d, 0x54, 0x97, 0xc3,
    0x2e, 0xc5, 0x5d, 0x6e, 0x9b, 0x4f, 0x43, 0x0c,
    0xf0, 0x36, 0x68, 0x53, 0x9e, 0x2f, 0xbd, 0x5a,
    0x8f, 0x3c, 0x30, 0x9f, 0x38, 0x40, 0x58, 0x34,
    0x27, 0x18, 0xfc, 0x70, 0x54, 0x7e, 0x70, 0x15,
    0xeb, 0x93, 0x29, 0x15, 0x7d, 0xa7, 0x6b, 0xa0,
    0x5b, 0x7d, 0x2b, 0x7a, 0x79, 0x10, 0xb4, 0x8b,
    0xd9, 0xfa, 0x94, 0x5d, 0xc2, 0x1b, 0x0d, 0xf7,
    0x47, 0x18, 0xa9, 0x8f, 0xa2, 0x64, 0x32, 0x17,
    0x15, 0x1d, 0x3b, 0xed, 0x3c, 0x32, 0x8a, 0x69,
    0x2d, 0x14, 0xae, 0x5a, 0x4a, 0xa4, 0x79, 0xdb,
    0x6e, 0x9e, 0x81, 0xa5, 0x32, 0x92, 0xa3, 0x55,
    0x3e, 0xfa, 0x05, 0x21, 0x1e, 0x1d, 0xb8, 0x08,
    0x5e, 0xee, 0x21, 0x14, 0xc4, 0x38, 0x17, 0x6c,
    0x24, 0x1a, 0xdb, 0xbd, 0xf0, 0x37, 0x33, 0xc2,
    0xc5, 0xec, 0xe9, 0x92, 0x6d, 0x1c, 0xc0, 0x3e,
    0x58, 0x26, 0x26, 0xde, 0x87, 0xc2, 0xef, 0x2c,
    0x9d, 0x18, 0x7b, 0x71, 0x05, 0xf7, 0x9f, 0x42,
    0x29, 0x8c, 0x18, 0x9b, 0xf6, 0xe9, 0x34, 0x75,
    0x59, 0x6f, 0x18, 0x8c, 0x43, 0xcc, 0x5c, 0xd7,
    0x45, 0x14, 0xda, 0xa0, 0xcf, 0x11, 0x10, 0xb2,
    0x00, 0xab, 0x94, 0x59, 0x36, 0xd6, 0x77, 0x7b,
    0x8a, 0x84, 0xff, 0x22, 0xd3, 0x60, 0x66, 0x06,
    0x42, 0xa2, 0xe6, 0xdd, 0xc0, 0xe6, 0xb7, 0x4c,
    0xe7, 0xc6, 0x88, 0x71, 0xb9, 0xf2, 0x75, 0x96,
    0x30, 0x85, 0xaf, 0x5f, 0xa7, 0xdb, 0x2b, 0xe7,
    0x17, 0xbc, 0xa6, 0x6c, 0x2b, 0xec, 0xef, 0x53,
    0x26, 0xe5, 0xad, 0xf2, 0x03, 0x02, 0xac, 0x51,
    0xf7, 0xa5, 0x06, 0x8f, 0x80, 0x9f, 0xc6, 0xd0,
    0xed, 0x0f, 0xed, 0xea, 0xf6, 0x5b, 0x61, 0xfc,
    0x2b, 0xb3, 0x44, 0xfd, 0x26, 0x1c, 0xde, 0x45,
    0x9f, 0xdb, 0x58, 0x2a, 0x12, 0x41, 0xdf, 0x9d,
    0xf6, 0xf9, 0xfc, 0x89, 0xa0, 0x86, 0x33, 0xd2,
    0xcf, 0x8b, 0x17, 0x10, 0x84, 0xbe, 0x94, 0x72,
    0x77, 0x28, 0x58, 0x52, 0x9a, 0x24, 0x9c, 0x3a,
    0x55, 0xd5, 0x57, 0x57, 0xb1, 0x3d, 0x03, 0xa6,
    0x0d, 0xb7, 0x89, 0x61, 0xa6, 0x1d, 0xc6, 0x61,
    0xa4, 0x75, 0xc2, 0xa7, 0x0e, 0xc9, 0x4c, 0x95,
    0x2a, 0xa1, 0x65, 0x63, 0x1f, 0x82, 0x2c, 0xf0,
    0xad, 0xcf, 0x24, 0xca, 0x89, 0x64, 0x72, 0x1e,
    0x26, 0x66, 0x37, 0x36, 0x4c, 0x35, 0x1e, 0xe9,
    0xb5, 0x64, 0xd9, 0xe1, 0x8f, 0x42, 0x5e, 0x9d,
    0x1a, 0xd6, 0xfe, 0x20, 0xa9, 0x74, 0xa9, 0x85,
    0x62, 0xc6, 0x17, 0x20, 0x63, 0x1b, 0x8e, 0xb5,
    0x1a, 0x2a, 0x81, 0xc6, 0x21, 0xf9, 0x54, 0x04,
    0x56, 0x26, 0x89, 0x90, 0xa0, 0xab, 0xcb, 0x5f,
    0x86, 0x8f, 0x9e, 0xf1, 0x17, 0x65, 0xa7, 0xb7,
    0x28, 0x63, 0x39, 0x5c, 0x3c, 0xeb, 0x1f, 0x9d,
    0xf9, 0x19, 0xdb, 0xaa, 0x6c, 0x18, 0x8b, 0x57,
    0xac, 0xec, 0x55, 0xf6, 0xb8, 0x2a, 0xc9, 0x24,
    0x04, 0xa7, 0xcc, 0xec, 0x53, 0x28, 0x3d, 0xae,
    0x75, 0x39, 0xaa, 0x66, 0x84, 0xc6, 0x94, 0xd7,
    0xc2, 0xdc, 0x21, 0x12, 0x73, 0x0c, 0x06, 0x2b,
    0x17, 0x40, 0x69, 0x49, 0xfd, 0x78, 0x7b, 0x9a,
    0x13, 0x0f, 0xc4, 0x8a, 0xd2, 0x60, 0x9d, 0xa4,
    0x0b, 0x3a, 0xf4, 0x3a, 0x9b, 0x0a, 0x79, 0x3e,
    0x31, 0xaa, 0x27, 0x63, 0x41, 0x38, 0x4e, 0x46,
    0x14, 0x8c, 0x55, 0x80, 0x7b, 0x75, 0xc8, 0xe9,
    0x15, 0x64, 0xc8, 0xc7, 0xdd, 0xf4, 0x1d, 0xe9,
    0x25, 0x4a, 0x0e, 0x06, 0x4a, 0x1a, 0xc8, 0x88,
    0xec, 0x46, 0xd6, 0xfb, 0x6b, 0xb8, 0x94, 0x28,
    0xaf, 0xbc, 0x2f, 0xe6, 0x28, 0x76, 0xc0, 0x0a,
    0xfc, 0x60, 0x42, 0xef, 0x45, 0xff, 0x76, 0xf8,
    0xe1, 0x50, 0x6f, 0x94, 0xaf, 0xa5, 0xd9, 0x4d,
    0xe9, 0xa9, 0x7c, 0x2d, 0x1d, 0xb3, 0x78, 0x66,
    0xc3, 0x6e, 0x4d, 0x22, 0xd9, 0x0f, 0xcf, 0xf0,
    0x87, 0xa7, 0xdc, 0xec, 0x5a, 0x12, 0x65, 0x55,
    0x3d, 0x9b, 0xc8, 0x33, 0xfd, 0x83, 0xfc, 0x02,
    0xfb, 0x22, 0x0c, 0x1d, 0x7f, 0xff, 0x38, 0xe1,
    0x6b, 0x78, 0x47, 0xa8, 0x54, 0x31, 0x36, 0x69,
    0x44, 0x64, 0xae, 0x1a, 0xfc, 0x8b, 0x3c, 0x3a,
    0x9f, 0x3e, 0xca, 0x85, 0xa1, 0x0d, 0x56, 0x72,
    0x63, 0xb5, 0x7b, 0xfa, 0x4f, 0x18, 0x76, 0x57,
    0x3a, 0xa7, 0xb0, 0x20, 0x24, 0x9b, 0xb1, 0x8d,
    0x01, 0xec, 0x32, 0x9c, 0x84, 0xc1, 0x8d, 0x88,
    0x0d, 0xe4, 0x71, 0xe5, 0x1f, 0xe2, 0x0b, 0xae,
    0x87, 0xcf, 0x5f, 0xbf, 0xde, 0x57, 0x2f, 0x22,
    0x5a, 0x63, 0x5a, 0x9c, 0x85, 0x46, 0xf1, 0x3a,
    0x7e, 0x04, 0xec, 0xce, 0xb4, 0xa0, 0x00, 0x90,
    0x9a, 0x09, 0x11, 0xb4, 0x6b, 0x53, 0x37, 0xd2,
    0x03, 0xd1, 0x59, 0x2f, 0xdb, 0xda, 0xe9, 0xf8,
    0x0a, 0x67, 0x6a, 0x83, 0x30, 0x09, 0x66, 0x82,
    0x11, 0x10, 0xda, 0x52, 0x61, 0x40, 0xca, 0x86,
    0x85, 0xdf, 0x01, 0xc3, 0x0e, 0x68, 0xd0, 0xb7,
    0x0a, 0x37, 0x95, 0x30, 0xc3, 0x0f, 0x37, 0x49,
    0x27, 0x4a, 0x93, 0xd3, 0x6a, 0x1a, 0xaf, 0xa2,
    0x0a, 0xbf, 0xb7, 0xb7, 0x68, 0xe7, 0x8d, 0x4c,
    0x50, 0xf0, 0x9c, 0x10, 0x93, 0x1c, 0xdf, 0x83,
    0x7b, 0xe7, 0xfa, 0xc3, 0x41, 0xbd, 0x6c, 0x06,
    0x10, 0x8d, 0x4c, 0x38, 0x0f, 0x74, 0x04, 0x48,
    0x07, 0xdd, 0xba, 0xbe, 0x6f, 0x92, 0x7b, 0xc6,
    0x87, 0xae, 0x0c, 0xe9, 0x46, 0x52, 0x97, 0xa7,
    0xe6, 0x9a, 0x91, 0xbb, 0x7e, 0xb0, 0x65, 0xac,
    0x22, 0xe0, 0xa3, 0xc7, 0xe8, 0xc5, 0x53, 0x99,
    0x82, 0x1c, 0x2c, 0x54, 0x9b, 0xa0, 0xcf, 0x6b,
    0xd5, 0x9e, 0x18, 0xf9, 0x3c, 0x17, 0x88, 0xb4,
    0xfa, 0x01, 0x46, 0x6e, 0xd6, 0x41, 0xf2, 0xdd,
    0xf1, 0xea, 0xbc, 0x1d, 0x6c, 0xbb, 0xe2, 0xcb,
    0xe6, 0xfc, 0x22, 0xb5, 0x82, 0x9a, 0x89, 0x60,
    0x30, 0xdc, 0x1f, 0x42, 0x20, 0x34, 0x05, 0x02,
    0x70, 0x60, 0xbd, 0x95, 0x22, 0x0e, 0xeb, 0x29,
    0x23, 0x0c, 0x51, 0x95, 0x74, 0x7e, 0xae, 0xed,
    0x3a, 0x5d, 0x83, 0x20, 0xd9, 0x55, 0x80, 0x9b,
    0xe0, 0x50, 0x25, 0xe5, 0xbd, 0xfb, 0x45, 0x43,
    0xce, 0x34, 0x5a, 0x8a, 0xcf, 0x68, 0x2f, 0xde,
    0xdc, 0x95, 0x1d, 0x72, 0x6b, 0x69, 0x7d, 0xe6,
    0xd5, 0x82, 0x2f, 0x65, 0x31, 0x8e, 0xab, 0xbd,
    0x66, 0xb9, 0x39, 0x15, 0x02, 0x0f, 0xa7, 0x62,
    0xfc, 0xdb, 0xc1, 0x50, 0xfa, 0xf1, 0x6a, 0xe6,
    0x88, 0x56, 0x16, 0x3d, 0x1a, 0x6c, 0x1c, 0x95,
    0x6e, 0x93, 0xdc, 0x72, 0xc3, 0x35, 0x4d, 0x22,
    0xe1, 0xd5, 0xa2, 0xa6, 0xcf, 0xe7, 0xaf, 0x2a,
    0x02, 0x02, 0x2b, 0xb4, 0xaa, 0x42, 0x5c, 0x12,
    0xbb, 0x1d, 0x0b, 0xe6, 0x16, 0x9e, 0x2a, 0xc8,
    0xf5, 0xd4, 0x41, 0xf4, 0x72, 0x9b, 0x68, 0xd5,
    0x3a, 0x53, 0xb1, 0x61, 0x7a, 0x5e, 0x74, 0x33,
    0x52, 0xe7, 0x51, 0x62, 0xcb, 0x02, 0xfa, 0x0f,
    0x9d, 0x7d, 0x51, 0xf4, 0x66, 0xeb, 0x80, 0xff,
    0x83, 0xac, 0x1c, 0xde, 0x5d, 0xc6, 0x3a, 0x47,
    0x3e, 0x03, 0x9d, 0xaf, 0x4d, 0x9d, 0x2e, 0x27,
    0xb4, 0xd0, 0x69, 0x4a, 0xdc, 0x66, 0x96, 0xe9,
    0x77, 0x81, 0xfe, 0x13, 0x25, 0xd6, 0x4d, 0xa2,
    0x3e, 0xa5, 0xe3, 0xe0, 0xbb, 0x97, 0x27, 0x49,
    0x92, 0x10, 0x36, 0xdc, 0x31, 0x25, 0x5e, 0xba,
    0xa3, 0x5f, 0xca, 0x74, 0xbe, 0xe6, 0x57, 0xd0,
    0xfa, 0x46, 0x81, 0x2a, 0xe4, 0xf6, 0x9d, 0xd8,
    0xcf, 0xb4, 0x8f, 0x89, 0x75, 0x5e, 0x8b, 0x0e,
    0x10, 0x9e, 0x1a, 0xa5, 0x17, 0xc4, 0x56, 0x97,
    0x9e, 0x38, 0x38, 0xd8, 0xef, 0x4b, 0x82, 0x79,
    0xc4, 0x54, 0x2c, 0x2b, 0xed, 0xe2, 0x6a, 0x83,
    0x4b, 0x12, 0x6a, 0xa7, 0xff, 0x31, 0xe7, 0x8d,
    0x0a, 0x9f, 0xab, 0x49, 0xae, 0xee, 0xc1, 0xa8,
    0x87, 0xdf, 0x2d, 0x11, 0xda, 0xd9, 0xf6, 0xa1,
    0x12, 0x82, 0x96, 0x64, 0x68, 0xa8, 0xe2, 0x13,
    0xf8, 0x25, 0xf3, 0x3d, 0xa0, 0xcd, 0x0e, 0xfb,
    0x1f, 0xbc, 0x27, 0xe3, 0xed, 0x72, 0xa9, 0x31,
    0xee, 0x61, 0xcc, 0xb9, 0xb3, 0x4d, 0x72, 0x41,
    0x5c, 0xb8, 0x2d, 0x6f, 0xae, 0xdf, 0xf7, 0xf4,
    0x07, 0x04, 0xf3, 0xcf, 0x86, 0x30, 0x6f, 0xed,
    0x0c, 0x61, 0x4e, 0x78, 0x56, 0x2e, 0x85, 0xf1,
    0xaf, 0x6a, 0x76, 0x28, 0x0f, 0x4b, 0x14, 0xfa,
    0xf2, 0xda, 0x0c, 0x7f, 0xe1, 0xe3, 0x76, 0x66,
    0xb2, 0xd6, 0xb5, 0xe4, 0x61, 0xec, 0x21, 0xcd,
    0xfa, 0x58, 0x61, 0x96, 0x3b, 0x91, 0x82, 0xd4,
    0xf2, 0xd8, 0xac, 0x91, 0x7e, 0x2f, 0xaa, 0x92,
    0x2e, 0x89, 0x02, 0x19, 0x02, 0xb8, 0xb4, 0x28,
    0xde, 0x58, 0x69, 0xca, 0x8b, 0x31, 0x2d, 0x05,
    0xab, 0x28, 0xdd, 0x82, 0x60, 0x72, 0x98, 0xa1,
    0x50, 0x17, 0x22, 0x64, 0x4a, 0x4f, 0xb9, 0xad,
    0xb7, 0x03, 0x0b, 0x84, 0x32, 0x06, 0xa6, 0xb9,
    0xd2, 0x32, 0x2a, 0xda, 0xb6, 0xdd, 0xad, 0x57,
    0xde, 0xfb, 0x81, 0x1e, 0x91, 0xa1, 0x75, 0xcc,
    0x09, 0x2e, 0x1a, 0x0e, 0x90, 0x74, 0x26, 0xd1,
    0x8f, 0x9e, 0x0a, 0xe6, 0x62, 0x75, 0xca, 0x2c,
    0xc7, 0x85, 0xfa, 0x2b, 0xe2, 0x52, 0xe7, 0xb6,
    0x01, 0xb8, 0x9a, 0x53, 0x71, 0x92, 0x7b, 0x4e,
    0x48, 0x37, 0x2d, 0x18, 0x19, 0xb3, 0x91, 0xf5,
    0x26, 0x66, 0x56, 0x17, 0xb8, 0xb9, 0x4d, 0x38,
    0x7c, 0x03, 0x6e, 0x07, 0x1c, 0x25, 0xe0, 0x9c,
    0x7d, 0xcf, 0xc5, 0xe8, 0x0c, 0x90, 0xa4, 0xad,
    0xa3, 0xad, 0x7c, 0x40, 0x87, 0x4e, 0xe3, 0x9d,
    0x23, 0x4b, 0x72, 0x72, 0x6e, 0xe6, 0xca, 0x6a,
    0x6b, 0x5f, 0x49, 0x8d, 0xe5, 0x09, 0x42, 0x72,
    0x0c, 0x84, 0xdc, 0x59, 0x7e, 0xad, 0x1c, 0x58,
    0xdf, 0x7e, 0x0e, 0x5e, 0x36, 0xb8, 0x31, 0x64,
    0x1c, 0x2e, 0x89, 0xd3, 0x6a, 0x81, 0x7d, 0xce,
    0x04, 0xc9, 0x99, 0x5e, 0x52, 0x11, 0x20, 0x64,
    0xc6, 0xf0, 0x9a, 0x5f, 0x3f, 0xad, 0x8a, 0xec,
    0x26, 0xa2, 0x7b, 0x52, 0x32, 0x5f, 0x36, 0x4b,
    0x28, 0xdc, 0xd3, 0xfb, 0x72, 0x7a, 0x1d, 0xeb,
    0xf9, 0x74, 0xef, 0x66, 0xad, 0x86, 0x84, 0xda,
    0x13, 0x35, 0x7e, 0xe3, 0x1d, 0xb0, 0xb3, 0xc2,
    0x2a, 0xc9, 0x78, 0xed, 0x5f, 0x56, 0xd9, 0x08,
    0x13, 0x84, 0x92, 0x72, 0xd5, 0x1c, 0xfd, 0x90,
    0x5e, 0x11, 0xb5, 0x49, 0x52, 0x07, 0x9d, 0x5e,
    0x4c, 0x43, 0x4f, 0x25, 0x0a, 0x7b, 0xc5, 0xaf,
    0xbd, 0x68, 0x70, 0x1d, 0xe6, 0xe5, 0x79, 0x18,
    0x2c, 0xa3, 0x92, 0x76, 0x59, 0x84, 0x72, 0x10,
    0xf3, 0x6b, 0x8b, 0x15, 0xc1, 0x17, 0x35, 0x1a,
    0x88, 0x35, 0x25, 0x07, 0xaa, 0xd6, 0x02, 0xaf,
    0x03, 0xee, 0x64, 0x42, 0x23, 0x1c, 0x81, 0x01,
    0x45, 0xb8, 0xb5, 0x90, 0xa4, 0x69, 0x58, 0xe4,
    0xaa, 0xd9, 0x45, 0x91, 0x11, 0xed, 0x17, 0x01,
    0x64, 0x1a, 0x87, 0x13, 0x5d, 0x52, 0xa3, 0xca,
    0xa7, 0x50, 0xb7, 0xfa, 0x79, 0x1f, 0xce, 0x20,
    0x58, 0x9c, 0x64, 0x1a, 0x2d, 0x15, 0xc9, 0xfa,
    0x1e, 0x19, 0xcd, 0xcc, 0x11, 0x95, 0xfd, 0x1e,
    0x9b, 0xce, 0xea, 0xbe, 0x33, 0x1c, 0xa5, 0x58,
    0x02, 0x80, 0x8e, 0x95, 0xc5, 0x34, 0xa5, 0x66,
    0x85, 0xf0, 0x45, 0x76, 0xf0, 0x49, 0x6b, 0xea,
    0x0d, 0xaf, 0xb0, 0xa0, 0x7f, 0x8e, 0xe7, 0xfe,
    0x8f, 0x42, 0x0f, 0xc9, 0xaf, 0x56, 0xea, 0xa4,
    0x3e, 0x28, 0xbd, 0x66, 0x9d, 0x9f, 0x68, 0x41,
    0x45, 0xcc, 0xa4, 0x21, 0xa3, 0x40, 0xa7, 0x96,
    0xe1, 0xa3, 0x5d, 0x96, 0x40, 0xde, 0xe1, 0xd7,
    0xb2, 0xbc, 0xc6, 0x01, 0xc5, 0xa0, 0x9f, 0xee,
    0x08, 0x40, 0x29, 0xde, 0x90, 0x7e, 0x8a, 0x3a,
    0x8f, 0xa2, 0x43, 0xd3, 0xd6, 0x19, 0xd2, 0xe6,
    0xc0, 0xc6, 0x02, 0x8f, 0xb2, 0x81, 0x79, 0x2f,
    0x58, 0x0c, 0x75, 0x55, 0xb3, 0x92, 0x23, 0x68,
    0xd0, 0x30, 0x85, 0xcf, 0x95, 0xc5, 0x61, 0x06,
    0x4b, 0x94, 0xa5, 0x36, 0x68, 0x4f, 0x3c, 0xa1,
    0x1d, 0xcd, 0x10, 0xba, 0x1a, 0x5f, 0xa4, 0x52,
    0x77, 0x99, 0xfa, 0x20, 0xb1, 0xf9, 0x66, 0xa4,
    0x86, 0x3f, 0x7a, 0x87, 0x19, 0xdd, 0x4e, 0xb8,
    0x99, 0x6f, 0x12, 0x41, 0x43, 0x0c, 0x08, 0x22,
    0xc0, 0x47, 0x49, 0x71, 0x33, 0x6f, 0xc7, 0xd1,
    0x22, 0x98, 0x15, 0x56, 0x6a, 0xa6, 0x27, 0x73,
    0x03, 0x95, 0x5d, 0xb8, 0xe3, 0x6f, 0x49, 0x7a,
    0xe1, 0x62, 0x20, 0xae, 0x3d, 0x8b, 0x33, 0x6b,
    0x55, 0x2b, 0x3c, 0xcc, 0x0c, 0x13, 0x40, 0x86,
    0xda, 0x79, 0x7c, 0x2a, 0x91, 0xd9, 0xf1, 0xad,
    0x5d, 0x01, 0x7e, 0x84, 0xcc, 0xd9, 0xe0, 0xed,
    0x02, 0x1c, 0x23, 0x6e, 0xa3, 0xa9, 0xbb, 0x02,
    0x20, 0x6c, 0x77, 0x7e, 0x8d, 0xbe, 0xc7, 0xd3,
    0xe4, 0x02, 0x2f, 0x48, 0x4d, 0x78, 0xcb, 0xe0,
    0x23, 0x29, 0x4a, 0x4c, 0x66, 0x8a, 0xb3, 0xdb,
    0xee, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x08, 0x0f, 0x19, 0x20, 0x29
};

/*
 * The message has been specially selected so that:
 *      it fails the z_max rejection test on iteration one
 *      it fails the r0_max rejection test on iteration two
 *      it fails the h_ones rejection test on iteration three
 *      it successfully generates the signature on iteration four
 * Thus, it is an optimal self test in terms of iterations and coverage.
 *
 * Refer to FIPS 140-3 IG 10.3.A.15 for details of the testing requirements.
 */
static const unsigned char ml_dsa_65_msg[] = {
    0x23, 0x37, 0x34, 0x37, 0x36, 0x38, 0x23
};

static const ST_KAT_PARAM ml_dsa_key[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PRIV_KEY, ml_dsa_65_priv_key),
    ST_KAT_PARAM_END()
};

static int ml_dsa_deterministic = 1;
static int ml_dsa_encoding = 0;

static const ST_KAT_PARAM ml_dsa_sig_init[] = {
    ST_KAT_PARAM_INT(OSSL_SIGNATURE_PARAM_DETERMINISTIC, ml_dsa_deterministic),
    ST_KAT_PARAM_INT(OSSL_SIGNATURE_PARAM_MESSAGE_ENCODING, ml_dsa_encoding),
    ST_KAT_PARAM_END()
};
#endif /* OPENSSL_NO_ML_DSA */

#ifndef OPENSSL_NO_SLH_DSA
/*
 * Deterministic SLH_DSA key generation supplies the private key elements and
 * Half of the public key.
 */
static const uint8_t slh_dsa_sha2_128f_keygen_entropy[] = {
    0xae, 0xd6, 0xf6, 0xf5, 0xc5, 0x40, 0x8b, 0xbf, 0xfa, 0x11, 0x36, 0xbc, 0x90, 0x49, 0xa7, 0x01,
    0x4d, 0x4c, 0xe0, 0x71, 0x1e, 0x17, 0x6a, 0x0c, 0x8a, 0x02, 0x35, 0x08, 0xa6, 0x92, 0xc2, 0x07,
    0x74, 0xd9, 0x8d, 0x50, 0x00, 0xaf, 0x53, 0xb9, 0x8f, 0x36, 0x38, 0x9a, 0x12, 0x92, 0xbe, 0xd3
};

/* The expected outputs for the public and private key elements */
static const uint8_t slh_dsa_sha2_128f_keygen_priv_pub[] = {
    0xAE, 0xD6, 0xF6, 0xF5, 0xC5, 0x40, 0x8B, 0xBF, 0xFA, 0x11, 0x36, 0xBC, 0x90, 0x49, 0xA7, 0x01,
    0x4D, 0x4C, 0xE0, 0x71, 0x1E, 0x17, 0x6A, 0x0C, 0x8A, 0x02, 0x35, 0x08, 0xA6, 0x92, 0xC2, 0x07,
    0x74, 0xD9, 0x8D, 0x50, 0x00, 0xAF, 0x53, 0xB9, 0x8F, 0x36, 0x38, 0x9A, 0x12, 0x92, 0xBE, 0xD3,
    0xF4, 0xA6, 0x50, 0xC5, 0x6C, 0x42, 0x6F, 0xCF, 0xDB, 0x88, 0xE3, 0x35, 0x54, 0x59, 0x44, 0x0C
};

static const uint8_t slh_dsa_sha2_128f_keygen_pub[] = {
    0x74, 0xD9, 0x8D, 0x50, 0x00, 0xAF, 0x53, 0xB9, 0x8F, 0x36, 0x38, 0x9A, 0x12, 0x92, 0xBE, 0xD3,
    0xF4, 0xA6, 0x50, 0xC5, 0x6C, 0x42, 0x6F, 0xCF, 0xDB, 0x88, 0xE3, 0x35, 0x54, 0x59, 0x44, 0x0C
};

static const ST_KAT_PARAM slh_dsa_sha2_128f_keygen_init_params[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_SLH_DSA_SEED,
                       slh_dsa_sha2_128f_keygen_entropy),
    ST_KAT_PARAM_END()
};

static const ST_KAT_PARAM slh_dsa_128f_keygen_expected_params[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PRIV_KEY,
                       slh_dsa_sha2_128f_keygen_priv_pub),
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PUB_KEY,
                       slh_dsa_sha2_128f_keygen_pub),
    ST_KAT_PARAM_END()
};

static const unsigned char slh_dsa_sha2_sig_msg[] = {
    0x3f
};

static const unsigned char slh_dsa_shake_sig_msg[] = {
    0x01, 0x02, 0x03, 0x04
};

static int deterministic = 1;
static const ST_KAT_PARAM slh_dsa_sig_params[] = {
    ST_KAT_PARAM_INT(OSSL_SIGNATURE_PARAM_DETERMINISTIC, deterministic),
    ST_KAT_PARAM_END()
};

static const unsigned char slh_dsa_sha2_128f_priv_pub[] = {
    0xd5, 0x21, 0x3b, 0xa4, 0xbb, 0x64, 0x70, 0xf1, 0xb9, 0xed, 0xa8, 0x8c, 0xbc, 0x94, 0xe6, 0x27,
    0x7a, 0x58, 0xa9, 0x51, 0xef, 0x7f, 0x2b, 0x81, 0x46, 0x1d, 0xba, 0xc4, 0x1b, 0x5a, 0x6b, 0x83,
    0xfa, 0x49, 0x5f, 0xb8, 0x34, 0xde, 0xfe, 0xa7, 0xcc, 0x96, 0xa8, 0x13, 0x09, 0x47, 0x91, 0x35,
    0xa6, 0x70, 0x29, 0xe9, 0x06, 0x68, 0xc5, 0xa5, 0x8b, 0x96, 0xe6, 0x01, 0x11, 0x49, 0x1f, 0x3d
};

static const ST_KAT_PARAM slh_dsa_sha2_128f_key_params[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PRIV_KEY, slh_dsa_sha2_128f_priv_pub),
    ST_KAT_PARAM_END()
};

static const unsigned char slh_dsa_shake_128f_priv_pub[] = {
    0xbb, 0xc7, 0x43, 0x06, 0xf7, 0x5d, 0xc2, 0xda, 0xf7, 0x37, 0x2b, 0x3c, 0x98, 0x41, 0xa4, 0xd6,
    0x85, 0x2c, 0x17, 0xb4, 0x59, 0xf1, 0x69, 0x2b, 0x8e, 0x9a, 0x1a, 0x0d, 0xac, 0xe5, 0xba, 0x26,
    0x38, 0x0c, 0x99, 0x30, 0x4a, 0x0d, 0xdd, 0x32, 0xf3, 0x44, 0xb9, 0x51, 0x44, 0xe1, 0xfd, 0xef,
    0x60, 0xbb, 0xc2, 0x34, 0x0e, 0x08, 0x77, 0x0f, 0xb4, 0x1a, 0x80, 0xa7, 0x6c, 0xb0, 0x8e, 0x34,
};

static const ST_KAT_PARAM slh_dsa_shake_128f_key_params[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PRIV_KEY, slh_dsa_shake_128f_priv_pub),
    ST_KAT_PARAM_END()
};

/* This is the SHA256 digest of the signature */
static const unsigned char slh_dsa_sha2_128f_sig_digest[] = {
    0xd3, 0x53, 0x7e, 0x05, 0xae, 0x63, 0x87, 0x6b, 0xf7, 0x80, 0x15, 0xff,
    0x86, 0xcc, 0x9e, 0x28, 0x4f, 0x91, 0xca, 0xbf, 0xac, 0x19, 0x12, 0x98,
    0xfb, 0xaa, 0x37, 0x55, 0x96, 0x35, 0x1d, 0x55
};

static const unsigned char slh_dsa_shake_128f_sig_digest[] = {
    0xb7, 0xeb, 0x1f, 0x00, 0x33, 0x41, 0xff, 0x11,
    0x3f, 0xc7, 0x4d, 0xce, 0x90, 0x6c, 0x55, 0xf7,
    0x4a, 0x54, 0x8b, 0x86, 0xc1, 0xb1, 0x08, 0x48,
    0x89, 0x77, 0x00, 0x72, 0x03, 0x92, 0xd1, 0xa6,
};
#endif /* OPENSSL_NO_SLH_DSA */

/* Hash DRBG inputs for signature KATs */
static const unsigned char sig_kat_entropyin[] = {
    0x06, 0x6d, 0xc8, 0xce, 0x75, 0xb2, 0x89, 0x66, 0xa6, 0x85, 0x16, 0x3f,
    0xe2, 0xa4, 0xd4, 0x27, 0xfb, 0xdb, 0x61, 0x66, 0x50, 0x61, 0x6b, 0xa2,
    0x82, 0xfc, 0x33, 0x2b, 0x4e, 0x6f, 0x12, 0x20
};
static const unsigned char sig_kat_nonce[] = {
    0x55, 0x9f, 0x7c, 0x64, 0x89, 0x70, 0x83, 0xec, 0x2d, 0x73, 0x70, 0xd9,
    0xf0, 0xe5, 0x07, 0x1f
};
static const unsigned char sig_kat_persstr[] = {
    0x88, 0x6f, 0x54, 0x9a, 0xad, 0x1a, 0xc6, 0x3d, 0x18, 0xcb, 0xcc, 0x66,
    0x85, 0xda, 0xa2, 0xc2, 0xf7, 0x9e, 0xb0, 0x89, 0x4c, 0xb4, 0xae, 0xf1,
    0xac, 0x54, 0x4f, 0xce, 0x57, 0xf1, 0x5e, 0x11
};

static const ST_KAT_SIGN st_kat_sign_tests[] = {
    {
        OSSL_SELF_TEST_DESC_SIGN_RSA,
        "RSA", "RSA-SHA256", 0,
        rsa_crt_key,
        ITM_STR(rsa_sig_msg),
        ITM(sig_kat_entropyin),
        ITM(sig_kat_nonce),
        ITM(sig_kat_persstr),
        ITM(rsa_expected_sig)
    },
#ifndef OPENSSL_NO_EC
    {
        OSSL_SELF_TEST_DESC_SIGN_ECDSA,
        "EC", "ECDSA-SHA256", 0,
        ecdsa_prime_key,
        ITM_STR(rsa_sig_msg),
        ITM(sig_kat_entropyin),
        ITM(sig_kat_nonce),
        ITM(sig_kat_persstr),
        ITM(ecdsa_prime_expected_sig)
    },
# ifndef OPENSSL_NO_EC2M
    {
        OSSL_SELF_TEST_DESC_SIGN_ECDSA,
        "EC", "ECDSA-SHA256", 0,
        ecdsa_bin_key,
        ITM_STR(rsa_sig_msg),
        ITM(sig_kat_entropyin),
        ITM(sig_kat_nonce),
        ITM(sig_kat_persstr),
        ITM(ecdsa_bin_expected_sig)
    },
# endif
# ifndef OPENSSL_NO_ECX
    {
        OSSL_SELF_TEST_DESC_SIGN_EDDSA,
        "ED448", "ED448", 0,
        ed448_key,
        ITM(ecx_sig_msg),
        NULL, 0, NULL, 0, NULL, 0,
        ITM(ed448_expected_sig),
    },
    {
        OSSL_SELF_TEST_DESC_SIGN_EDDSA,
        "ED25519", "ED25519", 0,
        ed25519_key,
        ITM(ecx_sig_msg),
        NULL, 0, NULL, 0, NULL, 0,
        ITM(ed25519_expected_sig),
    },
# endif /* OPENSSL_NO_ECX */
#endif /* OPENSSL_NO_EC */
#ifndef OPENSSL_NO_DSA
    {
        OSSL_SELF_TEST_DESC_SIGN_DSA,
        "DSA", "DSA-SHA256", SIGNATURE_MODE_VERIFY_ONLY,
        dsa_key,
        ITM_STR(rsa_sig_msg),
        ITM(sig_kat_entropyin),
        ITM(sig_kat_nonce),
        ITM(sig_kat_persstr),
        ITM(dsa_expected_sig)
    },
#endif /* OPENSSL_NO_DSA */

#ifndef OPENSSL_NO_ML_DSA
    {
        OSSL_SELF_TEST_DESC_SIGN_ML_DSA,
        "ML-DSA-65", "ML-DSA-65", 0,
        ml_dsa_key,
        ITM(ml_dsa_65_msg),
        NULL, 0,
        NULL, 0,
        NULL, 0,
        ITM(ml_dsa_65_sig),
        ml_dsa_sig_init,
        ml_dsa_sig_init
    },
#endif /* OPENSSL_NO_ML_DSA */
#ifndef OPENSSL_NO_SLH_DSA
    /*
     * FIPS 140-3 IG 10.3.A.16 Note 29 says:
     *
     *  It is recommended (but not required) that if the module implements
     *   both "s" and "f" algorithms, the module self-test at least one of
     *   each "s" and "f" algorithm.
     *
     * Because the "s" version is so slow, we only test the "f" versions
     * here.
     */
    {
        OSSL_SELF_TEST_DESC_SIGN_SLH_DSA,
        "SLH-DSA-SHA2-128f", "SLH-DSA-SHA2-128f", SIGNATURE_MODE_SIG_DIGESTED,
        slh_dsa_sha2_128f_key_params,
        ITM(slh_dsa_sha2_sig_msg),
        NULL, 0, NULL, 0, NULL, 0,
        ITM(slh_dsa_sha2_128f_sig_digest),
        slh_dsa_sig_params, slh_dsa_sig_params
    },
    {
        OSSL_SELF_TEST_DESC_SIGN_SLH_DSA,
        "SLH-DSA-SHAKE-128f", "SLH-DSA-SHAKE-128f", SIGNATURE_MODE_SIG_DIGESTED,
        slh_dsa_shake_128f_key_params,
        ITM(slh_dsa_shake_sig_msg),
        NULL, 0, NULL, 0, NULL, 0,
        ITM(slh_dsa_shake_128f_sig_digest),
        slh_dsa_sig_params, slh_dsa_sig_params
    },
#endif /* OPENSSL_NO_SLH_DSA */
};

#if !defined(OPENSSL_NO_ML_DSA)
static const ST_KAT_PARAM ml_dsa_keygen_params[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_ML_DSA_SEED, sig_kat_entropyin),
    ST_KAT_PARAM_END()
};
#endif

/*- KEM SELF TEST DATA */
/*
 * Test case generated via the OpenSSL commands:
 *
 *  openssl genpkey -algorithm ML-KEM-512 -out priv-ml-kem-512.pem
 *  openssl pkey -in priv-ml-kem-512.pem -pubout -out pub-ml-kem-512.pem
 *  openssl pkeyutl -encap -inkey pub-ml-kem-512.pem \
 *          -pkeyopt ikme:00000000000000000000000000000000 -secret good.dat \
 *          -out ctext.dat
 *  openssl pkeyutl -decap -inkey priv-ml-kem-512.pem \
 *          -secret goodcmp.dat -in ctext.dat
 *  apps/openssl pkeyutl -decap -inkey priv-ml-kem-512.pem \
 *          -secret implicit-reject.dat -in <(head -c 768 /dev/zero)
 *  openssl pkey -in priv-ml-kem-512.pem -text |
 *      sed '1,/^priv:/d; /^pub:/,$d' |
 *      tr -d ' :\n' |
 *      xxd -r -p |
 *      tail -c 32 > z.dat
 *  (cat z.dat; head -c 768 /dev/zero) |
 *      openssl dgst -shake256 -xoflen 32 -binary > expected.dat
 */
#ifndef OPENSSL_NO_ML_KEM
static const unsigned char ml_kem_512_cipher_text[] = {
    0x6b, 0xc5, 0x04, 0x00, 0x27, 0x7a, 0xbb, 0x7e,
    0x6b, 0xf9, 0xfb, 0x56, 0x82, 0x01, 0x75, 0xeb,
    0xb7, 0xb9, 0xf4, 0xf2, 0x82, 0x2c, 0x6d, 0x0a,
    0xe0, 0x80, 0xa3, 0x49, 0x92, 0x0f, 0x6d, 0x00,
    0x8e, 0xba, 0x35, 0xb5, 0x42, 0xb9, 0xd7, 0xed,
    0x89, 0xcb, 0xfd, 0x38, 0xd7, 0x9f, 0x55, 0x3b,
    0xf0, 0x8e, 0x63, 0x80, 0x95, 0xcf, 0x0d, 0x4f,
    0x50, 0x40, 0xac, 0x1d, 0x1b, 0xdc, 0x24, 0x84,
    0x3b, 0x18, 0xc2, 0x77, 0x9f, 0x24, 0x11, 0x81,
    0xa6, 0xcc, 0xd8, 0xf4, 0xe8, 0x00, 0xdc, 0x26,
    0x61, 0x36, 0xd1, 0xb6, 0x5d, 0x9c, 0x12, 0x6d,
    0xf5, 0xe7, 0x93, 0xc8, 0x6d, 0xac, 0xaf, 0x3c,
    0x78, 0x3f, 0xa9, 0xc1, 0x00, 0x6f, 0x08, 0x29,
    0x4f, 0x1a, 0x4d, 0x3b, 0xb4, 0x4b, 0x6f, 0xc2,
    0x09, 0x00, 0x4b, 0xc1, 0xf0, 0xe6, 0x7f, 0x48,
    0x48, 0x09, 0x40, 0xf2, 0x0a, 0x86, 0x18, 0xbf,
    0xc6, 0x4e, 0xb2, 0xb0, 0xab, 0xfe, 0x1b, 0xea,
    0x91, 0x58, 0x8c, 0x18, 0x6e, 0x30, 0xe8, 0x33,
    0x87, 0x29, 0x26, 0xef, 0xe9, 0x0c, 0x3b, 0x8b,
    0x0c, 0x99, 0x40, 0x53, 0xb9, 0x30, 0xba, 0x17,
    0xb3, 0x8a, 0x74, 0x59, 0x5d, 0x4c, 0x76, 0x1a,
    0xd3, 0x3f, 0xe4, 0xf7, 0xa3, 0x0f, 0x3a, 0x2c,
    0x85, 0xc1, 0x46, 0xf0, 0xa4, 0x91, 0xa3, 0x12,
    0xb2, 0xa4, 0x4b, 0x88, 0x8d, 0x1c, 0x85, 0xb1,
    0xe4, 0x41, 0x1f, 0x9d, 0xb7, 0x91, 0x2f, 0x60,
    0x98, 0xfd, 0xfc, 0x29, 0x89, 0x11, 0x7d, 0x16,
    0x09, 0x9e, 0x76, 0xf1, 0x9c, 0xb3, 0x50, 0xb9,
    0xd0, 0x02, 0xd2, 0xa2, 0x0d, 0xdd, 0xee, 0x90,
    0x47, 0x0d, 0xb9, 0x4a, 0x53, 0x11, 0xa2, 0x4f,
    0x13, 0x5a, 0x40, 0xdc, 0xc6, 0xfe, 0xd7, 0x28,
    0x36, 0xec, 0xa0, 0x5e, 0xab, 0xc1, 0x7d, 0x19,
    0x33, 0x59, 0xe2, 0xe4, 0xea, 0x26, 0x67, 0x2f,
    0xe5, 0x05, 0xd8, 0x34, 0x6e, 0x3c, 0xab, 0x63,
    0x8b, 0x24, 0x16, 0xc7, 0x1b, 0x2a, 0x9b, 0xe5,
    0x04, 0x78, 0x98, 0x6c, 0x6c, 0x1e, 0x94, 0xe3,
    0x7f, 0x86, 0x52, 0xc0, 0x17, 0x56, 0x8d, 0x01,
    0x7a, 0x28, 0x81, 0x07, 0x3d, 0x61, 0x2a, 0xcd,
    0xc8, 0xb6, 0x7e, 0x5b, 0xad, 0xa8, 0x90, 0xbd,
    0x0c, 0x95, 0xb5, 0x09, 0x9d, 0x7c, 0x34, 0x8c,
    0x74, 0x8f, 0x8e, 0x7c, 0x28, 0x6c, 0xe2, 0x2f,
    0xa2, 0x87, 0x7f, 0x80, 0x43, 0x46, 0x1c, 0xb2,
    0x1c, 0x5a, 0xd2, 0xec, 0xad, 0xf9, 0x55, 0xe3,
    0x6b, 0x19, 0x54, 0x08, 0x84, 0x1a, 0x34, 0x82,
    0xf4, 0x9c, 0xec, 0x3d, 0x65, 0xf9, 0x78, 0x7f,
    0x37, 0x47, 0xcf, 0xf1, 0xcb, 0x15, 0xf2, 0xac,
    0xff, 0x3b, 0x8f, 0xa0, 0x8c, 0x25, 0x88, 0x5c,
    0x38, 0x23, 0x9a, 0x27, 0x16, 0x6a, 0xdf, 0xa3,
    0x98, 0x1d, 0x16, 0x33, 0x4b, 0x4f, 0xfb, 0x83,
    0x85, 0x66, 0x76, 0x03, 0xb9, 0xb5, 0x46, 0x21,
    0xb9, 0xf3, 0xf4, 0xf1, 0x3a, 0x85, 0xec, 0x9e,
    0x56, 0x6a, 0xb6, 0x1d, 0xcc, 0xca, 0xfb, 0x11,
    0xae, 0x47, 0x7d, 0x93, 0xa5, 0xbc, 0x90, 0x32,
    0xde, 0xa1, 0xa5, 0x1e, 0x5d, 0x52, 0x17, 0x98,
    0x0a, 0x8b, 0xc4, 0x1a, 0x28, 0x7c, 0x9c, 0x22,
    0x3e, 0x33, 0x06, 0x40, 0x77, 0xe5, 0x22, 0x49,
    0x86, 0xf9, 0x3c, 0xc5, 0xc1, 0xb9, 0x77, 0x25,
    0x53, 0x66, 0x5a, 0x18, 0x83, 0x5a, 0x2b, 0xbf,
    0xac, 0x04, 0x70, 0x26, 0xe8, 0x2b, 0xb6, 0x0c,
    0xe8, 0x00, 0x95, 0xbb, 0x08, 0x75, 0xf3, 0x37,
    0x31, 0x2e, 0xef, 0x28, 0x8d, 0x58, 0x92, 0xd4,
    0xfd, 0xd7, 0x02, 0xce, 0x8f, 0x11, 0x83, 0x17,
    0x53, 0x19, 0x44, 0xd7, 0xd5, 0x6d, 0x44, 0x04,
    0x3a, 0x0a, 0x01, 0x46, 0xf2, 0xd2, 0xa5, 0x05,
    0x88, 0xa0, 0xd9, 0x0d, 0xe1, 0xa0, 0x7a, 0xf2,
    0x20, 0x2e, 0x5b, 0x05, 0xe4, 0x2b, 0x11, 0x3d,
    0xb3, 0x82, 0x64, 0x3b, 0xef, 0xc1, 0x53, 0xba,
    0x9f, 0x7f, 0x29, 0x59, 0x87, 0x39, 0x52, 0xda,
    0x7b, 0xff, 0xd7, 0xdd, 0xa1, 0xa9, 0x9f, 0xa1,
    0xe2, 0x38, 0x74, 0xb0, 0x94, 0xdc, 0xc5, 0xb5,
    0xf3, 0x61, 0xdf, 0x92, 0x62, 0xe1, 0x96, 0x87,
    0x6d, 0xb4, 0x2c, 0xc7, 0xf0, 0x38, 0xe9, 0x5e,
    0xfc, 0xff, 0x4c, 0x01, 0xc7, 0x59, 0x39, 0xe5,
    0x9b, 0xfb, 0xf5, 0x2b, 0x1b, 0xe5, 0xf8, 0x25,
    0x06, 0x07, 0xc7, 0x82, 0x46, 0x2a, 0x99, 0xd0,
    0xa9, 0x67, 0x81, 0xd7, 0xa2, 0x29, 0x96, 0x1a,
    0x94, 0x8e, 0x7d, 0x51, 0x76, 0x99, 0xad, 0x61,
    0xec, 0xb6, 0xc0, 0x58, 0x8e, 0xd0, 0x9d, 0xff,
    0x58, 0x57, 0x1b, 0x2e, 0xad, 0x65, 0xd8, 0xde,
    0xa5, 0xfa, 0x81, 0x4b, 0x2c, 0x06, 0x1b, 0xfe,
    0x49, 0x20, 0x4d, 0x5e, 0x1b, 0xb7, 0x40, 0x96,
    0xaa, 0x81, 0x25, 0xeb, 0x84, 0xdb, 0xea, 0x5d,
    0x0b, 0xaf, 0xf9, 0x8e, 0x41, 0xa6, 0xdd, 0x91,
    0x3a, 0x68, 0x54, 0xb7, 0x2e, 0xb1, 0x74, 0xff,
    0xf5, 0x0d, 0xa7, 0x3c, 0xc7, 0x30, 0x5b, 0x55,
    0xc6, 0x2d, 0xc8, 0x4c, 0xb4, 0xad, 0xcc, 0xd0,
    0xa1, 0x1b, 0x41, 0xc7, 0x23, 0xe8, 0xda, 0xff,
    0xb7, 0x3a, 0x12, 0xc2, 0xdc, 0x39, 0x7c, 0xf4,
    0xb9, 0x50, 0x00, 0x53, 0x88, 0xc8, 0x77, 0x49,
    0xfd, 0x70, 0x3d, 0xe0, 0xaa, 0x0c, 0x28, 0xd4,
    0xa2, 0xec, 0x82, 0x5d, 0xda, 0xe8, 0x05, 0x2f,
    0xe8, 0x9c, 0x21, 0x39, 0x3a, 0x22, 0x2f, 0x0a,
    0x5c, 0x6d, 0x01, 0xdf, 0xc8, 0x9f, 0x46, 0xf7,
    0x15, 0x02, 0xad, 0x19, 0x35, 0x63, 0x55, 0x58,
    0xf5, 0x7a, 0x46, 0xc9, 0x7a, 0xe6, 0x33, 0x84,
    0x48, 0x2b, 0xf0, 0xe7, 0xc5, 0x72, 0x98, 0x43,
    0x6a, 0x99, 0xba, 0x00, 0x57, 0x4f, 0xaa, 0x0a
};

static const unsigned char ml_kem_512_private_key[] = {
    0x87, 0xca, 0x19, 0x93, 0xb6, 0x4d, 0x89, 0x32,
    0xae, 0x3b, 0x22, 0x52, 0x82, 0xa1, 0xb3, 0xc1,
    0x37, 0x65, 0xdc, 0xc1, 0x22, 0x4c, 0x43, 0x77,
    0x33, 0x0a, 0x04, 0xec, 0xec, 0x0b, 0x25, 0x05,
    0x40, 0x07, 0x53, 0x82, 0xbe, 0x37, 0x52, 0x53,
    0x12, 0x87, 0x7d, 0x77, 0x69, 0xfd, 0x59, 0x4f,
    0xbd, 0x16, 0x42, 0x82, 0x58, 0x9d, 0xee, 0x5c,
    0x0f, 0x2c, 0x14, 0x7f, 0xc6, 0x2a, 0x95, 0x42,
    0x10, 0x3c, 0x08, 0xbc, 0xcc, 0xa0, 0x05, 0x82,
    0xe9, 0xc5, 0x26, 0x81, 0xfa, 0xb8, 0x79, 0x78,
    0x5b, 0x3e, 0x79, 0x49, 0x68, 0x44, 0xb3, 0x7b,
    0xf5, 0x26, 0x62, 0x7a, 0x8a, 0x3c, 0xd8, 0x82,
    0x1f, 0x16, 0x1d, 0x92, 0x99, 0xac, 0xc4, 0xa9,
    0xb9, 0x30, 0x32, 0x6b, 0x6b, 0x67, 0x3d, 0x16,
    0x13, 0x1d, 0xf0, 0x98, 0x94, 0x42, 0x90, 0x68,
    0xfc, 0x65, 0xa3, 0xe5, 0x16, 0x22, 0x09, 0x64,
    0xc0, 0x7d, 0x54, 0x03, 0x47, 0x89, 0xbe, 0xae,
    0x61, 0x4b, 0x13, 0xa1, 0xcd, 0xad, 0xbc, 0x20,
    0x5e, 0x36, 0x36, 0x34, 0x41, 0x1d, 0x5b, 0x3a,
    0x26, 0x91, 0x80, 0x75, 0xe0, 0x63, 0x9d, 0xd6,
    0x35, 0xc9, 0x28, 0x81, 0xa4, 0x6e, 0xfb, 0x95,
    0x01, 0x11, 0x8f, 0xcc, 0x18, 0xb3, 0x9a, 0x91,
    0x66, 0xa6, 0x37, 0x6c, 0xeb, 0x71, 0x42, 0x29,
    0xec, 0x71, 0xa4, 0x99, 0x6d, 0x92, 0x97, 0x9d,
    0x94, 0x64, 0x6e, 0xc3, 0xf0, 0x5d, 0xa5, 0x49,
    0x8f, 0x66, 0xa5, 0x0a, 0x9a, 0xb9, 0xcf, 0x85,
    0x20, 0xa7, 0x28, 0xe1, 0xc2, 0x10, 0x08, 0x72,
    0x58, 0x71, 0x56, 0x3e, 0x7b, 0x47, 0x46, 0x81,
    0x7d, 0x74, 0xfa, 0xb2, 0xb6, 0xf2, 0xa0, 0xe3,
    0x4a, 0x0a, 0x5e, 0x95, 0x2b, 0x32, 0xf1, 0x07,
    0x2c, 0x30, 0x5e, 0x81, 0x84, 0x58, 0xf3, 0x42,
    0x34, 0xdb, 0xaa, 0xc4, 0x06, 0xcb, 0x63, 0x72,
    0xfa, 0x01, 0x86, 0xe8, 0xca, 0xd7, 0x73, 0x1d,
    0xac, 0xd8, 0x64, 0x60, 0x66, 0xb1, 0x19, 0xb1,
    0x59, 0xac, 0x78, 0x21, 0xab, 0x9a, 0x62, 0xbf,
    0xe8, 0x1b, 0xd1, 0xdc, 0x75, 0x08, 0xa4, 0x9e,
    0x22, 0x54, 0xa8, 0x36, 0x68, 0x4b, 0xb7, 0x22,
    0xa3, 0xbc, 0x04, 0x09, 0xe2, 0xe9, 0x4d, 0xce,
    0xf5, 0x46, 0x69, 0x1a, 0x47, 0x80, 0xb2, 0xa5,
    0xa0, 0x24, 0xcf, 0x0d, 0x60, 0x95, 0x99, 0x33,
    0xad, 0x6a, 0x58, 0x7a, 0x56, 0x53, 0x86, 0x44,
    0xa8, 0x3c, 0x1f, 0x92, 0x55, 0x3f, 0x3a, 0x3b,
    0x5f, 0x81, 0xaa, 0x0c, 0xc4, 0x4b, 0x1a, 0xe3,
    0x61, 0x8a, 0xd0, 0x5d, 0x29, 0x87, 0xb6, 0x7d,
    0x1c, 0x85, 0xa5, 0x14, 0xb0, 0xde, 0x1c, 0x8d,
    0x5c, 0xc1, 0x5c, 0x04, 0xfc, 0x77, 0xad, 0x03,
    0x55, 0x96, 0xe0, 0xa7, 0x43, 0xb5, 0x95, 0x9a,
    0xd5, 0x22, 0xef, 0x13, 0x5f, 0x14, 0x9c, 0x0e,
    0x7e, 0x56, 0x89, 0x37, 0x33, 0xc9, 0x00, 0x54,
    0x7e, 0xea, 0x32, 0x4f, 0x02, 0x85, 0xcd, 0xe5,
    0x9c, 0x25, 0x00, 0x3b, 0xad, 0xc2, 0x72, 0x3a,
    0x38, 0x66, 0x95, 0xf9, 0xf4, 0x22, 0x1f, 0x50,
    0x19, 0x20, 0x6a, 0x31, 0x03, 0xf8, 0x97, 0x91,
    0xf0, 0x42, 0xbb, 0xc0, 0x86, 0xde, 0x56, 0x93,
    0xf2, 0x78, 0x9b, 0xc9, 0x98, 0x16, 0x47, 0x83,
    0x67, 0x55, 0x92, 0x4e, 0x5a, 0xfa, 0x5d, 0x88,
    0xf9, 0xc8, 0xef, 0xa8, 0x21, 0x34, 0x58, 0x5f,
    0xca, 0xbb, 0x52, 0x98, 0xbc, 0x5c, 0xf5, 0xa7,
    0xae, 0xc5, 0xfc, 0x78, 0xf9, 0xa5, 0x30, 0x16,
    0x68, 0x9d, 0x62, 0x17, 0x41, 0x7a, 0x95, 0xcf,
    0x27, 0xb6, 0x6d, 0xae, 0x58, 0xa7, 0x28, 0x8c,
    0x8f, 0xc3, 0x28, 0xac, 0x06, 0x79, 0x9d, 0x94,
    0xc4, 0x9d, 0xed, 0xb2, 0x61, 0xf4, 0x44, 0x86,
    0xec, 0x12, 0xc3, 0x13, 0x97, 0xa7, 0x8b, 0x9a,
    0x63, 0x2e, 0xf1, 0x66, 0x08, 0x84, 0x32, 0xf6,
    0x15, 0x3b, 0x91, 0xca, 0xce, 0xf7, 0x40, 0x53,
    0xa9, 0x28, 0x11, 0x63, 0xa0, 0x23, 0x2e, 0xc4,
    0x44, 0x28, 0x05, 0x01, 0x02, 0x74, 0xcf, 0x9c,
    0x3a, 0x1a, 0xbc, 0x93, 0x8a, 0x2c, 0xe8, 0x9a,
    0xca, 0xe1, 0x74, 0x62, 0x03, 0x88, 0xc7, 0x12,
    0x20, 0x96, 0x3c, 0x4d, 0x10, 0x79, 0x28, 0x6f,
    0x7b, 0xa8, 0x1b, 0xfb, 0x5e, 0x57, 0x17, 0xcc,
    0x6d, 0xd0, 0x72, 0x8a, 0xb2, 0x70, 0xa3, 0x0a,
    0x88, 0x03, 0x5b, 0x88, 0x5d, 0x35, 0x12, 0x8e,
    0xac, 0xc1, 0x81, 0xca, 0xb7, 0x2b, 0xb1, 0x96,
    0xf6, 0x35, 0xce, 0xbb, 0x75, 0xad, 0x0d, 0xd0,
    0xba, 0x4e, 0x43, 0x5b, 0x31, 0x08, 0x93, 0x32,
    0x72, 0x63, 0x58, 0xc5, 0xa2, 0x70, 0x95, 0x12,
    0x8f, 0xf2, 0xc1, 0x61, 0xeb, 0x22, 0xe9, 0x4a,
    0x65, 0xa5, 0x48, 0x5d, 0x4c, 0x11, 0x78, 0x69,
    0x0b, 0x1f, 0x39, 0x2b, 0x7f, 0x63, 0x77, 0xad,
    0x96, 0x6b, 0x67, 0x80, 0x90, 0x70, 0x57, 0x2b,
    0xbc, 0x68, 0x9c, 0xaa, 0xb2, 0xd8, 0x3c, 0xbf,
    0xd6, 0x4f, 0xc4, 0x28, 0x65, 0x74, 0x54, 0x84,
    0x43, 0x9a, 0x1c, 0x96, 0x50, 0x02, 0x97, 0xc6,
    0xcf, 0xb0, 0xb1, 0x1d, 0x98, 0x9a, 0x32, 0x94,
    0x73, 0x88, 0x94, 0x78, 0x2d, 0x5f, 0x25, 0x05,
    0x5f, 0xa6, 0x96, 0x7a, 0xc8, 0x3a, 0xdf, 0xa8,
    0x19, 0xb2, 0x53, 0x53, 0x05, 0xf9, 0x31, 0xdc,
    0x58, 0x6c, 0xd1, 0x3a, 0x9b, 0x47, 0x3b, 0x7d,
    0x87, 0xe5, 0xb4, 0xb2, 0xd9, 0x96, 0x2a, 0x59,
    0x90, 0x3c, 0xcc, 0xad, 0xdc, 0xa2, 0x57, 0x87,
    0x71, 0xc6, 0x7e, 0x5a, 0x49, 0x98, 0xc8, 0x94,
    0x29, 0x30, 0x7b, 0x0e, 0x01, 0x97, 0x53, 0x18,
    0x32, 0x50, 0x73, 0x9e, 0x14, 0x47, 0x97, 0xbd,
    0xcc, 0x22, 0xab, 0x02, 0x95, 0xd7, 0xc5, 0x32,
    0xdf, 0x17, 0x84, 0x86, 0x77, 0x41, 0x6e, 0x95,
    0x4d, 0x66, 0xf9, 0xb0, 0x9e, 0x12, 0x81, 0x53,
    0x2a, 0x2e, 0x8f, 0x0c, 0x6a, 0xbe, 0x00, 0x37,
    0xe7, 0xe8, 0x11, 0x90, 0x97, 0xc9, 0xec, 0x84,
    0x5a, 0xa0, 0x69, 0x85, 0xc0, 0x88, 0x55, 0x2c,
    0x41, 0xb6, 0x15, 0x17, 0x36, 0x42, 0xc1, 0x02,
    0x51, 0xc0, 0x6e, 0x91, 0xa2, 0x5c, 0x24, 0x3c,
    0x02, 0x63, 0xb6, 0x75, 0xc7, 0x20, 0x7d, 0x58,
    0x70, 0x1d, 0x13, 0xa5, 0x2e, 0xab, 0x92, 0x56,
    0x5e, 0xf1, 0xa1, 0xde, 0xfb, 0xae, 0xfe, 0x4c,
    0x0b, 0x03, 0xf5, 0x04, 0x44, 0xa3, 0xbe, 0x20,
    0xb1, 0x71, 0x31, 0x0b, 0xa3, 0xf2, 0x08, 0x52,
    0xf3, 0xa2, 0xa1, 0x8e, 0x72, 0x29, 0x40, 0x70,
    0x15, 0x64, 0x60, 0x52, 0xd2, 0xe7, 0x3a, 0xbe,
    0xe3, 0x18, 0xd7, 0x55, 0x89, 0x9c, 0x78, 0x4a,
    0x6f, 0xb5, 0xaa, 0xb1, 0x7c, 0x90, 0x2c, 0xb6,
    0x8a, 0xfa, 0x36, 0x57, 0xb9, 0x01, 0x2e, 0x1b,
    0xb8, 0xb6, 0xaa, 0xc2, 0x68, 0x9a, 0xa4, 0x8c,
    0x2d, 0x42, 0x62, 0xbb, 0x29, 0x63, 0xb0, 0x6a,
    0x24, 0x22, 0x82, 0x54, 0x1c, 0xf6, 0x14, 0x19,
    0xd9, 0x0e, 0x2b, 0xd3, 0x66, 0xbe, 0xe7, 0xa4,
    0x36, 0x9b, 0x72, 0x41, 0xa6, 0x6b, 0xf2, 0x64,
    0x5d, 0x8e, 0x90, 0x4c, 0x3f, 0x7b, 0x30, 0x0c,
    0x04, 0x1d, 0x56, 0x87, 0x39, 0x3d, 0x2c, 0xa9,
    0x29, 0x75, 0xcb, 0xfd, 0xc0, 0x18, 0xdd, 0xc6,
    0x0a, 0xf5, 0x62, 0x00, 0x9e, 0xb0, 0x88, 0x8e,
    0x5a, 0x05, 0x76, 0x24, 0x6a, 0xce, 0x74, 0x5b,
    0xb1, 0x63, 0x40, 0x8b, 0x5a, 0x9a, 0xe3, 0xc1,
    0x94, 0xa8, 0x5c, 0x21, 0x90, 0x7b, 0x37, 0xb1,
    0x62, 0x8e, 0xcd, 0x9a, 0x15, 0xec, 0x20, 0x24,
    0x87, 0x30, 0x27, 0x34, 0x44, 0xb0, 0xa2, 0xf4,
    0x54, 0xf9, 0xb4, 0x73, 0x0f, 0x33, 0x91, 0x50,
    0x47, 0x6e, 0xe0, 0x70, 0x98, 0xf6, 0xbc, 0x1b,
    0x97, 0xcc, 0x1b, 0xd3, 0xb8, 0xc1, 0xa2, 0xeb,
    0x0e, 0x50, 0xa7, 0x82, 0xf2, 0x11, 0x5d, 0xf6,
    0x17, 0x49, 0x6f, 0x6c, 0x6f, 0x8c, 0x09, 0xb0,
    0x5f, 0x88, 0x8d, 0x9e, 0x93, 0x3d, 0x28, 0x77,
    0x46, 0xc6, 0x31, 0xb3, 0x10, 0x87, 0x26, 0xb0,
    0xc4, 0xa7, 0xc8, 0x8b, 0x09, 0xc8, 0x60, 0xac,
    0xd7, 0x52, 0x35, 0x70, 0xc8, 0x02, 0xbd, 0x38,
    0x72, 0x43, 0x16, 0x2d, 0x12, 0x8c, 0xa2, 0x29,
    0x4c, 0x83, 0x43, 0x18, 0xcc, 0x21, 0xfb, 0x14,
    0xd2, 0xab, 0x37, 0x3f, 0x22, 0x4e, 0x3f, 0xd4,
    0x98, 0x43, 0x85, 0x95, 0x09, 0xf4, 0xca, 0x1a,
    0x1a, 0x56, 0x6c, 0x05, 0x67, 0x88, 0xa3, 0xaa,
    0x48, 0x4a, 0xaa, 0xbd, 0xf1, 0xa0, 0x8f, 0x1b,
    0x44, 0xc7, 0x56, 0xab, 0x2a, 0x0c, 0x8b, 0xc5,
    0x85, 0x1e, 0xe2, 0xeb, 0x23, 0x03, 0x4c, 0x2e,
    0xaa, 0x5a, 0xc1, 0x5f, 0x89, 0x75, 0x1b, 0xd5,
    0xca, 0xe1, 0xd5, 0x90, 0x89, 0xd5, 0x00, 0x2b,
    0xb1, 0xb3, 0xbf, 0xca, 0x4a, 0xde, 0x09, 0x82,
    0xad, 0x67, 0x83, 0xe9, 0xd3, 0x7f, 0xb3, 0xe3,
    0x20, 0xac, 0x98, 0x55, 0xdf, 0x66, 0x6b, 0xfd,
    0x7c, 0x6b, 0x87, 0xa9, 0xa4, 0x6e, 0x25, 0x97,
    0x12, 0x56, 0x11, 0x61, 0x50, 0x7e, 0x17, 0x35,
    0x98, 0xf9, 0x88, 0xac, 0xb6, 0xf9, 0xab, 0x3a,
    0x10, 0x92, 0x63, 0x24, 0x46, 0x88, 0xc7, 0x08,
    0x78, 0x75, 0x8f, 0xf4, 0xd4, 0x31, 0x3b, 0x76,
    0x64, 0xf0, 0xf5, 0x10, 0xe9, 0x13, 0xca, 0x01,
    0xc5, 0x2b, 0x3a, 0x1b, 0x46, 0x53, 0x51, 0x44,
    0xe2, 0xdb, 0x0c, 0xac, 0xe8, 0xa6, 0x46, 0x66,
    0x00, 0xcf, 0x2a, 0x87, 0x83, 0x50, 0x07, 0xe1,
    0x6a, 0xa5, 0x07, 0x80, 0x1d, 0x86, 0xb7, 0x38,
    0x5e, 0x66, 0x9b, 0xfa, 0xf6, 0xba, 0x1d, 0xf5,
    0x68, 0x31, 0x63, 0xc2, 0xd5, 0x70, 0x53, 0x2f,
    0xd7, 0x43, 0x90, 0x54, 0x64, 0x0f, 0x24, 0x4c,
    0x52, 0x87, 0x13, 0x59, 0xd2, 0x84, 0x2f, 0xc3,
    0x37, 0xa0, 0x60, 0x03, 0x3a, 0xf0, 0x5e, 0xaa,
    0x00, 0x1c, 0x34, 0xfc, 0x7b, 0xd8, 0xf9, 0x10,
    0x29, 0xe4, 0x6c, 0x29, 0x43, 0x36, 0x27, 0x64,
    0x5d, 0x67, 0x86, 0x64, 0x21, 0xd6, 0x61, 0xaf,
    0x25, 0x74, 0x80, 0x53, 0x2b, 0x88, 0x78, 0x50,
    0xdc, 0x49, 0x9f, 0xfe, 0xd5, 0xb1, 0x40, 0x98,
    0xa0, 0x33, 0x72, 0x5e, 0x82, 0x0a, 0x5b, 0xe1,
    0x40, 0x0a, 0x0c, 0xb7, 0x03, 0x74, 0x1c, 0xa7,
    0x4b, 0x47, 0x86, 0x73, 0xaa, 0xcf, 0x85, 0x16,
    0x6e, 0xe8, 0xa1, 0x84, 0xdb, 0x2c, 0x58, 0x54,
    0x9c, 0x22, 0x40, 0xb8, 0x30, 0x8c, 0x27, 0xbe,
    0xba, 0x40, 0xc3, 0xb9, 0xd7, 0x29, 0xed, 0xb1,
    0x8c, 0xc2, 0x06, 0x8e, 0xcd, 0xb2, 0x7d, 0xa2,
    0x2b, 0x2c, 0xd3, 0xe7, 0xa0, 0xba, 0xa5, 0x30,
    0xc5, 0x19, 0x3c, 0xd3, 0xc8, 0x6b, 0xf8, 0x6a,
    0x44, 0xf0, 0x79, 0x9e, 0x51, 0x55, 0xb0, 0x9b,
    0xe2, 0x50, 0x98, 0x85, 0x23, 0xb3, 0xa7, 0x31,
    0xdd, 0x89, 0x1b, 0xc2, 0x00, 0x60, 0x11, 0xa0,
    0x65, 0xc0, 0xab, 0x57, 0xf1, 0xa6, 0xc1, 0x78,
    0x89, 0x55, 0xe0, 0x13, 0x5f, 0xa5, 0xca, 0x8f,
    0x3e, 0x52, 0xc7, 0x5d, 0x37, 0x16, 0x97, 0x3a,
    0x2f, 0xb2, 0x2c, 0x0e, 0xb1, 0x7c, 0x1b, 0x32,
    0x85, 0x29, 0xd9, 0xfa, 0x76, 0x56, 0xd7, 0x4d,
    0xea, 0x74, 0x0d, 0x9f, 0x07, 0x97, 0x77, 0xc3,
    0x6c, 0x17, 0xa1, 0x9c, 0x19, 0x58, 0x9e, 0x84,
    0xd2, 0xb8, 0xe4, 0xd1, 0xbc, 0x31, 0x07, 0xcb,
    0xd2, 0x78, 0x14, 0x22, 0x48, 0x35, 0x48, 0x44,
    0x6c, 0x89, 0x93, 0x14, 0x77, 0x44, 0xaa, 0x9e,
    0xc1, 0xc5, 0x93, 0xec, 0x2d, 0x5b, 0xaa, 0xc8,
    0x6a, 0x0a, 0xf6, 0x4a, 0x85, 0xe9, 0x09, 0xdf,
    0x8e, 0x28, 0x16, 0x60, 0x5d, 0x20, 0xb4, 0xe3,
    0x82, 0xb3, 0x0b, 0xbb, 0x61, 0xbf, 0x3a, 0x5f,
    0x82, 0x1a, 0x0b, 0x5d, 0xba, 0x9a, 0xd3, 0xe7,
    0xe5, 0xbd, 0x1b, 0x37, 0xa7, 0x5e, 0x0f, 0x09,
    0x29, 0x74, 0xe8, 0x46, 0xe8, 0xc3, 0x7c, 0x45,
    0x48, 0x7d, 0x60, 0x73, 0x9f, 0x99, 0x35, 0x17,
    0x19, 0xa5, 0x39, 0x47, 0x23, 0x26, 0x2b, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char ml_kem_512_secret[] = {
    0x4a, 0xd5, 0x3a, 0x06, 0xb2, 0x9f, 0x12, 0x56,
    0x84, 0x21, 0xa5, 0x52, 0xc0, 0x81, 0x95, 0xb5,
    0x86, 0x73, 0xc8, 0x2f, 0x87, 0x0c, 0xc1, 0xcc,
    0xd6, 0x5a, 0x08, 0xe4, 0x32, 0x5f, 0xeb, 0x27
};

static const unsigned char ml_kem_512_reject_secret[] = {
    0x87, 0xf1, 0xe8, 0x13, 0xb7, 0x2e, 0x04, 0x87,
    0x47, 0x22, 0x56, 0x6e, 0x12, 0x80, 0xd8, 0x9d,
    0x8b, 0x44, 0xb7, 0xfd, 0xff, 0x7d, 0x69, 0x23,
    0x36, 0xd7, 0xb8, 0x7f, 0x38, 0xe7, 0x85, 0xce
};

static const unsigned char ml_kem_512_entropy[ML_KEM_RANDOM_BYTES] = {
    0
};

static const unsigned char ml_kem_512_seed[ML_KEM_SEED_BYTES] = {
    0
};

static const ST_KAT_PARAM ml_kem_keygen_params[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_ML_KEM_SEED, ml_kem_512_seed),
    ST_KAT_PARAM_END()
};

static const ST_KAT_PARAM ml_kem_key[] = {
    ST_KAT_PARAM_OCTET(OSSL_PKEY_PARAM_PRIV_KEY, ml_kem_512_private_key),
    ST_KAT_PARAM_END()
};

static const ST_KAT_KEM st_kat_kem_tests[] = {
    {
        OSSL_SELF_TEST_DESC_KEM,
        "ML-KEM-512",
        ml_kem_key,
        ITM(ml_kem_512_cipher_text),
        ITM(ml_kem_512_entropy),
        ITM(ml_kem_512_secret),
        ml_kem_512_reject_secret    /* No length because same as _secret's */
    },
};
#endif /* OPENSSL_NO_ML_KEM */

#if !defined(OPENSSL_NO_ML_KEM) || !defined(OPENSSL_NO_ML_DSA) || !defined(OPENSSL_NO_SLH_DSA)
static const ST_KAT_ASYM_KEYGEN st_kat_asym_keygen_tests[] = {
# if !defined(OPENSSL_NO_ML_KEM)
    /*
     * FIPS 140-3 IG 10.3.A resolution 14 mandates a CAST for ML-KEM
     * key generation.
     */
    {
        OSSL_SELF_TEST_DESC_KEYGEN_ML_KEM,
        "ML-KEM-512",
        ml_kem_keygen_params,
        ml_kem_key
    },
# endif
# if !defined(OPENSSL_NO_ML_DSA)
    {
        OSSL_SELF_TEST_DESC_KEYGEN_ML_DSA,
        "ML-DSA-65",
        ml_dsa_keygen_params,
        ml_dsa_key
    },
# endif
# if !defined(OPENSSL_NO_SLH_DSA)
    {
        OSSL_SELF_TEST_DESC_KEYGEN_SLH_DSA,
        "SLH-DSA-SHA2-128f",
        slh_dsa_sha2_128f_keygen_init_params,
        slh_dsa_128f_keygen_expected_params
    },
# endif
};
#endif /* !OPENSSL_NO_ML_DSA || !OPENSSL_NO_SLH_DSA */
