// Copyright (C) 2016 Paul Lemire
// Copyright (C) 2014 Klaralvdalens Datakonsult AB (KDAB).
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR LGPL-3.0-only OR GPL-2.0-only OR GPL-3.0-only

#ifndef QT3DRENDER_RENDER_RENDERVIEWINITIALIZERJOB_H
#define QT3DRENDER_RENDER_RENDERVIEWINITIALIZERJOB_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists for the convenience
// of other Qt classes.  This header file may change from version to
// version without notice, or even be removed.
//
// We mean it.
//

#include <Qt3DCore/qaspectjob.h>
#include <Qt3DRender/private/renderlogging_p.h>
#include <Qt3DRender/private/job_common_p.h>
#include <QSize>
#include <QElapsedTimer>

QT_BEGIN_NAMESPACE

namespace Qt3DRender {

namespace Render {

class FrameGraphNode;

template<class RenderView, class Renderer>
class RenderViewInitializerJob : public Qt3DCore::QAspectJob
{
public:
    RenderViewInitializerJob()
        : Qt3DCore::QAspectJob()
    {
        SET_JOB_RUN_STAT_TYPE(this, JobTypes::RenderView, RenderViewInitializerJob::renderViewInstanceCounter++)
    }
    ~RenderViewInitializerJob()
    {
        RenderViewInitializerJob::renderViewInstanceCounter--;
    }

    inline void setRenderer(Renderer *renderer) { m_renderer = renderer; }
    inline RenderView *renderView() const noexcept { return m_renderView; }

    inline void setFrameGraphLeafNode(FrameGraphNode *fgLeaf)
    {
        m_fgLeaf = fgLeaf;
    }

    // Sets the position in the queue of RenderViews that the
    // RenderView generated by this job should be inserted. This is
    // used to ensure that for example a RenderView for creating
    // a shadow map texture is submitted before the RenderView that
    // contains commands making use of the shadow map
    inline void setSubmitOrderIndex(int index) { m_index = index; }
    inline int submitOrderIndex() const { return m_index; }

    void run() override
    {
#if defined(QT3D_RENDER_VIEW_JOB_TIMINGS)
        QElapsedTimer timer;
        timer.start();
        qint64 gatherLightsTime;
        qint64 buildCommandsTime;
#endif

        // Create a RenderView object
        m_renderView = new RenderView;
        m_renderView->setRenderer(m_renderer);

        // Populate the renderview's configuration from the framegraph
        RenderView::setRenderViewConfigFromFrameGraphLeafNode(m_renderView, m_fgLeaf);
#if defined(QT3D_RENDER_VIEW_JOB_TIMINGS)
        qint64 gatherStateTime = timer.nsecsElapsed();
        timer.restart();
#endif
    }

private:
    Renderer *m_renderer = nullptr;
    FrameGraphNode *m_fgLeaf = nullptr;
    RenderView *m_renderView = nullptr;
    int m_index = 0;

    static int renderViewInstanceCounter;
};

template<class RenderView, class Renderer>
int RenderViewInitializerJob<RenderView, Renderer>::renderViewInstanceCounter = 0;

template<class RenderView, class Renderer>
using RenderViewInitializerJobPtr = QSharedPointer<RenderViewInitializerJob<RenderView, Renderer>>;

} // namespace Render
} // namespace Qt3DRender

QT_END_NAMESPACE

#endif // QT3DRENDER_RENDER_RENDERVIEWINITIALIZERJOB_H
