/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This code is made available to you under your choice of the following sets
 * of licensing terms:
 */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */
/* Copyright 2015 Mozilla Contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "pkixgtest.h"

#include "mozpkix/pkixder.h"
#include "mozpkix/pkixnss.h"

using namespace mozilla::pkix;
using namespace mozilla::pkix::der;
using namespace mozilla::pkix::test;

struct VerifySignedDataNSSTestParams
{
  ByteString data;
  DigestAlgorithm digestAlgorithm;
  ByteString signature;
  PublicKeyAlgorithm publicKeyAlgorithm;
  ByteString subjectPublicKeyInfo;
  Result expectedResult;
};

::std::ostream& operator<<(::std::ostream& os,
                           const VerifySignedDataNSSTestParams&)
{
  return os << "TODO (bug 1318770)";
}

#define BS(s) ByteString(s, MOZILLA_PKIX_ARRAY_LENGTH(s))

static const uint8_t kData[] = {
  0x53, 0x70, 0x68, 0x79, 0x6e, 0x78, 0x20, 0x6f, 0x66, 0x20, 0x62, 0x6c,
  0x61, 0x63, 0x6b, 0x20, 0x71, 0x75, 0x61, 0x72, 0x74, 0x7a, 0x2c, 0x20,
  0x6a, 0x75, 0x64, 0x67, 0x65, 0x20, 0x6d, 0x79, 0x20, 0x76, 0x6f, 0x77,
  0x2e
};

static const uint8_t kRsaSubjectPublicKeyInfo[] = {
  0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
  0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00,
  0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xdb, 0x75, 0x02,
  0x7b, 0xeb, 0xf7, 0x3b, 0x31, 0x03, 0x71, 0x77, 0x34, 0x88, 0x8f, 0xb2,
  0x0d, 0xa6, 0xbe, 0x7d, 0xa7, 0xdd, 0xac, 0x0e, 0x99, 0x50, 0x46, 0x69,
  0x90, 0xe6, 0x7c, 0x3a, 0xa6, 0xf9, 0x3e, 0x02, 0x15, 0x3c, 0xf7, 0xb9,
  0xf4, 0xab, 0x3d, 0x54, 0x2c, 0x0d, 0x84, 0x94, 0x37, 0x95, 0xbc, 0x2b,
  0x56, 0x05, 0x00, 0xfa, 0xa2, 0x08, 0xf9, 0xcd, 0xc3, 0x2b, 0x9a, 0x58,
  0x80, 0x11, 0x49, 0xe0, 0x69, 0xf9, 0x81, 0x08, 0x52, 0x75, 0xb4, 0xc1,
  0x94, 0xa2, 0x67, 0x22, 0x5b, 0xfb, 0xe4, 0x74, 0xaa, 0x24, 0xb7, 0xa3,
  0x5e, 0x2c, 0x6b, 0xda, 0x20, 0x09, 0x5a, 0x5e, 0x4f, 0x95, 0xe8, 0x24,
  0x71, 0x64, 0x65, 0x29, 0x2c, 0x44, 0xb5, 0x17, 0xec, 0xe4, 0x68, 0xc3,
  0x69, 0x6b, 0x53, 0x6d, 0xa1, 0xa0, 0xb1, 0x74, 0xe2, 0x28, 0x03, 0xda,
  0x20, 0xca, 0xa4, 0x45, 0x1e, 0xf6, 0xab, 0xc7, 0xe9, 0xcb, 0xe3, 0x9a,
  0x16, 0x34, 0x8f, 0xd7, 0xf3, 0x66, 0x74, 0xea, 0xe7, 0x32, 0xf3, 0xd2,
  0x55, 0x6c, 0x8f, 0x38, 0xb8, 0x1b, 0x38, 0x08, 0x4c, 0x1f, 0x41, 0x74,
  0x35, 0x9e, 0x2d, 0x29, 0xed, 0x72, 0xe3, 0xda, 0x18, 0x01, 0xf4, 0x5f,
  0x8d, 0x9d, 0x72, 0x13, 0x18, 0x09, 0x1f, 0xbe, 0xb0, 0x20, 0x90, 0xc4,
  0x3d, 0x2c, 0x4f, 0xf2, 0xdc, 0x99, 0x8a, 0xae, 0x02, 0xd6, 0xef, 0x5a,
  0x88, 0x08, 0x15, 0x85, 0xdd, 0xaa, 0xce, 0xe4, 0x4b, 0x3f, 0xe9, 0xf4,
  0xfa, 0x54, 0xde, 0xb0, 0x30, 0xdf, 0x8f, 0x14, 0x2c, 0x49, 0x69, 0x24,
  0xe4, 0xa9, 0xeb, 0x62, 0x15, 0xf8, 0x8a, 0xd8, 0xe4, 0x8a, 0x99, 0x2b,
  0xdb, 0x68, 0x8b, 0x2a, 0x61, 0xbd, 0xc0, 0x57, 0xff, 0x5f, 0xee, 0xe9,
  0xac, 0x06, 0x77, 0x13, 0x7b, 0x2e, 0xd1, 0x76, 0x6c, 0xe8, 0x6c, 0x73,
  0x1f, 0x02, 0x03, 0x01, 0x00, 0x01
};

// Use `openssl dgst -binary -sha1` to obtain the hash of the data, and then
// `openssl pkeyutl` with `-pkeyopt rsa_padding_mode:pkcs1` and `-pkeyopt
// digest:sha1` to create the signature.
static const uint8_t kRsaPkcs1Sha1Signature[] = {
  0x6c, 0x11, 0x88, 0xc9, 0x6b, 0x06, 0xf2, 0x55, 0x67, 0xa1, 0x30, 0x3b,
  0x0a, 0xf5, 0x20, 0x84, 0xd7, 0x3c, 0x44, 0xc8, 0x25, 0x67, 0x58, 0x2a,
  0x9d, 0x7b, 0xbc, 0x92, 0x01, 0xbd, 0x4c, 0x11, 0x51, 0x2f, 0x18, 0x85,
  0xd5, 0xef, 0xd5, 0x35, 0x1e, 0x12, 0x89, 0x0e, 0x36, 0xaa, 0x86, 0xb8,
  0xaf, 0x81, 0x5b, 0xf2, 0x65, 0xde, 0x07, 0xb9, 0xbc, 0x1a, 0x22, 0xf1,
  0xac, 0x8e, 0x95, 0x5d, 0x3d, 0x5b, 0x15, 0x92, 0x12, 0x07, 0x24, 0x14,
  0x89, 0x5a, 0x00, 0xfe, 0x0c, 0xf3, 0x99, 0x0b, 0x83, 0x3f, 0x09, 0x2c,
  0x82, 0x06, 0x84, 0x21, 0xd3, 0x79, 0xc7, 0xac, 0x16, 0x89, 0x2e, 0x81,
  0xb4, 0x29, 0x88, 0x08, 0x26, 0x30, 0x79, 0x40, 0x96, 0xa7, 0xd9, 0x5c,
  0x67, 0xdb, 0x94, 0x77, 0x20, 0xcb, 0x07, 0x55, 0xa7, 0x37, 0xb4, 0xf9,
  0xcc, 0x9a, 0x9a, 0x22, 0x42, 0xb5, 0xce, 0xcc, 0x0f, 0x13, 0x54, 0x2f,
  0x1d, 0x7f, 0xa9, 0xb7, 0x7a, 0x06, 0x6f, 0x15, 0x16, 0xe2, 0x30, 0x5b,
  0xe0, 0x84, 0xe6, 0x28, 0x65, 0x53, 0x6a, 0xab, 0x67, 0x85, 0x53, 0x1b,
  0xaf, 0x9f, 0x9b, 0x60, 0x18, 0xb8, 0xf1, 0xd5, 0x53, 0xc9, 0x48, 0x21,
  0xd9, 0x81, 0xbd, 0xeb, 0x44, 0x82, 0x3c, 0x66, 0x91, 0x63, 0xa0, 0x00,
  0x9b, 0x23, 0x0a, 0xef, 0xfe, 0x2d, 0x17, 0xd5, 0x99, 0x64, 0x43, 0x32,
  0x58, 0xae, 0x96, 0x36, 0x5e, 0x80, 0x7c, 0xb7, 0x01, 0x44, 0x06, 0x19,
  0xba, 0x52, 0x30, 0xc9, 0x66, 0xb4, 0x4e, 0x7e, 0xeb, 0xe2, 0x97, 0x37,
  0x0c, 0xbd, 0xa6, 0xe6, 0x20, 0x05, 0x47, 0x79, 0x36, 0x56, 0x48, 0x9d,
  0x91, 0x7e, 0x61, 0x67, 0xbd, 0xc3, 0x12, 0xa8, 0x2a, 0xf0, 0xfc, 0xee,
  0x78, 0x18, 0x59, 0x7e, 0xd6, 0xd1, 0x55, 0xee, 0xf5, 0xaf, 0xb8, 0xbd,
  0x16, 0xd8, 0x01, 0xca
};

// Similarly as above, but with sha256.
static const uint8_t kRsaPkcs1Sha256Signature[] = {
  0x9c, 0xbd, 0x2d, 0x6c, 0x4a, 0xe2, 0x47, 0xc9, 0xe2, 0xd9, 0x48, 0xa6,
  0x3a, 0x3d, 0x43, 0xf5, 0x5e, 0xd0, 0x11, 0x9a, 0xab, 0x79, 0x71, 0xb6,
  0x97, 0x7c, 0x6a, 0xc7, 0x96, 0x29, 0x21, 0xbd, 0xe2, 0xe7, 0xee, 0x01,
  0x1b, 0x0f, 0xd1, 0x31, 0xeb, 0xa6, 0xbe, 0xec, 0xc0, 0x64, 0x63, 0xf4,
  0x71, 0xd4, 0x7b, 0x1b, 0x02, 0xba, 0xae, 0xf3, 0x7a, 0x81, 0x5c, 0x78,
  0xd5, 0x75, 0xee, 0x9a, 0x4a, 0x97, 0x64, 0x6a, 0x75, 0x12, 0xa3, 0xca,
  0xe5, 0x2f, 0x40, 0x43, 0x1e, 0xe0, 0x5c, 0xa2, 0x2f, 0x3e, 0xf0, 0xdd,
  0x5e, 0xdc, 0x57, 0x3a, 0xfc, 0x35, 0x84, 0x1d, 0x79, 0x2e, 0x82, 0xeb,
  0x0d, 0xef, 0xdf, 0x12, 0x96, 0xe0, 0x1f, 0xe8, 0x16, 0xe0, 0x6c, 0xf9,
  0xf4, 0x06, 0x6c, 0x51, 0x40, 0x3d, 0x23, 0xac, 0x24, 0x9c, 0x6c, 0xe6,
  0x1a, 0x84, 0x89, 0x8c, 0x6d, 0xbe, 0xc0, 0x42, 0x70, 0x9c, 0xdf, 0xcb,
  0x70, 0x7e, 0x4f, 0xbc, 0x87, 0x2b, 0xc0, 0xb7, 0xcf, 0x21, 0xfe, 0x1d,
  0x2e, 0x38, 0x00, 0xe4, 0xaf, 0x88, 0x44, 0x16, 0x28, 0x38, 0xf7, 0x4e,
  0x07, 0xf6, 0x82, 0x20, 0x96, 0x4c, 0x33, 0x15, 0xb6, 0xb1, 0x4b, 0x08,
  0x4e, 0x18, 0x3a, 0xaa, 0x2f, 0x69, 0xbe, 0x5c, 0x62, 0x1e, 0xf1, 0x02,
  0xa8, 0xa3, 0x54, 0x85, 0x72, 0xd0, 0x39, 0x4d, 0xa8, 0xec, 0xe2, 0x2e,
  0xb8, 0x4d, 0x93, 0xd6, 0x91, 0x84, 0x26, 0x09, 0xf0, 0x95, 0x37, 0x9f,
  0x28, 0x0e, 0x93, 0xbe, 0x71, 0x4c, 0xb7, 0xf8, 0xc6, 0xa7, 0x40, 0x9e,
  0xf4, 0xe2, 0x6a, 0x1c, 0xd7, 0x87, 0xa5, 0x0a, 0xf1, 0xc4, 0x1b, 0x85,
  0x75, 0x3b, 0x23, 0x69, 0xca, 0x20, 0x14, 0x8f, 0x56, 0x28, 0xa3, 0x9f,
  0xd8, 0xda, 0xd8, 0xff, 0x54, 0x1b, 0x2b, 0x40, 0xa1, 0x04, 0x63, 0xd9,
  0x36, 0x64, 0x98, 0x78
};

// As above.
static const uint8_t kRsaPkcs1Sha384Signature[] = {
  0xc4, 0x63, 0x51, 0x52, 0x13, 0x87, 0x56, 0x43, 0x57, 0x76, 0xf9, 0x19,
  0x0b, 0x12, 0xbc, 0xe3, 0x67, 0x4b, 0x7e, 0xed, 0xb4, 0xb3, 0xc9, 0xe5,
  0xa7, 0xa7, 0x00, 0x73, 0x1a, 0x58, 0x1a, 0xf0, 0x5b, 0xe2, 0xe3, 0x9b,
  0xe4, 0xb1, 0x0a, 0x7b, 0xaf, 0x1d, 0x5a, 0x58, 0x99, 0xa2, 0xa4, 0x61,
  0x92, 0x13, 0x6c, 0xb7, 0x2a, 0xdb, 0xdd, 0x3e, 0xc9, 0xe5, 0x2a, 0x60,
  0x5d, 0xf2, 0x0a, 0x57, 0x5a, 0x65, 0x3d, 0xd1, 0x5e, 0xc1, 0x5b, 0x47,
  0xa9, 0x86, 0x56, 0x08, 0x05, 0xee, 0x08, 0xf9, 0x35, 0x4a, 0x46, 0xa5,
  0x6a, 0x23, 0x01, 0x6d, 0xc3, 0x89, 0x1f, 0x26, 0x2e, 0x44, 0xa8, 0x50,
  0x84, 0x9b, 0x5d, 0x33, 0x5f, 0xf2, 0x6b, 0xc5, 0xdb, 0x5a, 0xcd, 0xec,
  0xd4, 0xd2, 0x16, 0x79, 0x8c, 0x4c, 0x78, 0x9a, 0xf0, 0x66, 0x19, 0xc7,
  0x45, 0x97, 0x15, 0x22, 0x9b, 0xad, 0xe1, 0xda, 0x1a, 0x68, 0x8f, 0xde,
  0x9c, 0x11, 0xb9, 0x63, 0xeb, 0x48, 0x0d, 0xd6, 0xb7, 0x47, 0x61, 0xd4,
  0x19, 0x12, 0x12, 0x23, 0x5c, 0x86, 0x15, 0x87, 0x83, 0x32, 0x11, 0xb3,
  0x8e, 0xdb, 0x64, 0x54, 0x4e, 0xf5, 0x17, 0xf9, 0x2c, 0xa2, 0xb8, 0x85,
  0x3b, 0x60, 0x75, 0xb5, 0x80, 0xb5, 0x75, 0x5a, 0xf3, 0xf2, 0xdd, 0xef,
  0xf0, 0x62, 0x14, 0x7e, 0xba, 0x7b, 0x9b, 0xf6, 0x4a, 0x67, 0x71, 0xa1,
  0x38, 0x05, 0xfb, 0x1d, 0xeb, 0xd7, 0x7c, 0x51, 0xdd, 0xf2, 0x06, 0x1b,
  0x8f, 0xfe, 0x31, 0x9f, 0xe7, 0xf9, 0xf4, 0xd7, 0x73, 0x26, 0x4f, 0xc3,
  0x8e, 0x33, 0x3c, 0x08, 0x5b, 0xfa, 0x40, 0xc5, 0xe6, 0xe9, 0x9c, 0x57,
  0x8b, 0x6f, 0x30, 0xd3, 0x09, 0x5f, 0xe9, 0x87, 0xd6, 0xb6, 0xa1, 0xd6,
  0x36, 0xf1, 0xda, 0x45, 0x8c, 0xc0, 0x4a, 0x7f, 0xb4, 0xe5, 0x74, 0xab,
  0x69, 0x82, 0x19, 0x5f
};

static const uint8_t kRsaPkcs1Sha512Signature[] = {
  0x91, 0x68, 0x2a, 0x91, 0xce, 0xab, 0x57, 0xd2, 0x60, 0x2d, 0x2f, 0xee,
  0x2b, 0x28, 0xc8, 0xa7, 0xe8, 0x2e, 0x41, 0xf8, 0x48, 0x92, 0xb4, 0x1b,
  0xbb, 0x89, 0x54, 0x5d, 0xea, 0x9f, 0xa1, 0x58, 0x23, 0x4d, 0x73, 0xe6,
  0x4f, 0x88, 0x45, 0x7c, 0xa0, 0x06, 0x3e, 0x8e, 0x29, 0xca, 0xdd, 0xc4,
  0x9a, 0x8a, 0x45, 0xbb, 0xad, 0x61, 0x4c, 0x4c, 0x1f, 0xd2, 0x67, 0xb0,
  0x8d, 0x1f, 0x11, 0x3f, 0x74, 0x67, 0x66, 0x3c, 0x65, 0xf5, 0xd8, 0xd8,
  0xcc, 0x2e, 0x66, 0x1f, 0xa9, 0x6f, 0x2e, 0x60, 0x28, 0x74, 0x92, 0xb7,
  0x50, 0x16, 0x8c, 0x96, 0xb2, 0xf4, 0x20, 0xfb, 0xa9, 0x14, 0xd3, 0x12,
  0xcf, 0x35, 0xef, 0x7c, 0x80, 0xf8, 0x70, 0x9f, 0x8d, 0x7f, 0x66, 0x5f,
  0x94, 0xeb, 0xd9, 0xbb, 0xc3, 0x54, 0x3d, 0x9d, 0x54, 0xef, 0xb1, 0x23,
  0xe0, 0x6d, 0x02, 0x96, 0xa9, 0x2b, 0x76, 0xd0, 0x88, 0x6b, 0x64, 0x39,
  0x15, 0x11, 0xba, 0x9a, 0x61, 0x72, 0x16, 0xba, 0x6e, 0x17, 0xf7, 0xc8,
  0xe8, 0xb2, 0x47, 0x31, 0xc5, 0x48, 0x1d, 0x58, 0x80, 0x34, 0x07, 0x30,
  0x8f, 0x05, 0xc4, 0x26, 0x7e, 0x70, 0x78, 0x43, 0xc1, 0x13, 0xe5, 0x09,
  0x4e, 0x68, 0xc1, 0x8d, 0x34, 0xa1, 0xd5, 0x68, 0xa2, 0xfe, 0x73, 0x4a,
  0x54, 0x05, 0x1b, 0xc0, 0xb3, 0x72, 0xe2, 0xab, 0x51, 0x71, 0xa7, 0xaf,
  0x5c, 0x95, 0xfb, 0x65, 0xe0, 0xb0, 0x60, 0x0c, 0xd3, 0x23, 0xa2, 0x0f,
  0x0d, 0xed, 0xd9, 0x87, 0xb4, 0x96, 0x03, 0xd8, 0xfd, 0x80, 0xa2, 0xa1,
  0xc1, 0x36, 0xb3, 0x47, 0x05, 0xd1, 0x6e, 0xd2, 0x81, 0x16, 0xe4, 0x63,
  0x26, 0xf9, 0x96, 0x55, 0x37, 0x18, 0x4f, 0x2b, 0x7e, 0x1c, 0xd9, 0xba,
  0x4f, 0xde, 0x4d, 0x61, 0xc4, 0xd5, 0x7c, 0xe3, 0xa3, 0x21, 0xe8, 0xae,
  0xc2, 0xe9, 0x62, 0x07
};

static const uint8_t kRsaPssSubjectPublicKeyInfo[] = {
  0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
  0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00,
  0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xdb, 0x75, 0x02,
  0x7b, 0xeb, 0xf7, 0x3b, 0x31, 0x03, 0x71, 0x77, 0x34, 0x88, 0x8f, 0xb2,
  0x0d, 0xa6, 0xbe, 0x7d, 0xa7, 0xdd, 0xac, 0x0e, 0x99, 0x50, 0x46, 0x69,
  0x90, 0xe6, 0x7c, 0x3a, 0xa6, 0xf9, 0x3e, 0x02, 0x15, 0x3c, 0xf7, 0xb9,
  0xf4, 0xab, 0x3d, 0x54, 0x2c, 0x0d, 0x84, 0x94, 0x37, 0x95, 0xbc, 0x2b,
  0x56, 0x05, 0x00, 0xfa, 0xa2, 0x08, 0xf9, 0xcd, 0xc3, 0x2b, 0x9a, 0x58,
  0x80, 0x11, 0x49, 0xe0, 0x69, 0xf9, 0x81, 0x08, 0x52, 0x75, 0xb4, 0xc1,
  0x94, 0xa2, 0x67, 0x22, 0x5b, 0xfb, 0xe4, 0x74, 0xaa, 0x24, 0xb7, 0xa3,
  0x5e, 0x2c, 0x6b, 0xda, 0x20, 0x09, 0x5a, 0x5e, 0x4f, 0x95, 0xe8, 0x24,
  0x71, 0x64, 0x65, 0x29, 0x2c, 0x44, 0xb5, 0x17, 0xec, 0xe4, 0x68, 0xc3,
  0x69, 0x6b, 0x53, 0x6d, 0xa1, 0xa0, 0xb1, 0x74, 0xe2, 0x28, 0x03, 0xda,
  0x20, 0xca, 0xa4, 0x45, 0x1e, 0xf6, 0xab, 0xc7, 0xe9, 0xcb, 0xe3, 0x9a,
  0x16, 0x34, 0x8f, 0xd7, 0xf3, 0x66, 0x74, 0xea, 0xe7, 0x32, 0xf3, 0xd2,
  0x55, 0x6c, 0x8f, 0x38, 0xb8, 0x1b, 0x38, 0x08, 0x4c, 0x1f, 0x41, 0x74,
  0x35, 0x9e, 0x2d, 0x29, 0xed, 0x72, 0xe3, 0xda, 0x18, 0x01, 0xf4, 0x5f,
  0x8d, 0x9d, 0x72, 0x13, 0x18, 0x09, 0x1f, 0xbe, 0xb0, 0x20, 0x90, 0xc4,
  0x3d, 0x2c, 0x4f, 0xf2, 0xdc, 0x99, 0x8a, 0xae, 0x02, 0xd6, 0xef, 0x5a,
  0x88, 0x08, 0x15, 0x85, 0xdd, 0xaa, 0xce, 0xe4, 0x4b, 0x3f, 0xe9, 0xf4,
  0xfa, 0x54, 0xde, 0xb0, 0x30, 0xdf, 0x8f, 0x14, 0x2c, 0x49, 0x69, 0x24,
  0xe4, 0xa9, 0xeb, 0x62, 0x15, 0xf8, 0x8a, 0xd8, 0xe4, 0x8a, 0x99, 0x2b,
  0xdb, 0x68, 0x8b, 0x2a, 0x61, 0xbd, 0xc0, 0x57, 0xff, 0x5f, 0xee, 0xe9,
  0xac, 0x06, 0x77, 0x13, 0x7b, 0x2e, 0xd1, 0x76, 0x6c, 0xe8, 0x6c, 0x73,
  0x1f, 0x02, 0x03, 0x01, 0x00, 0x01
};

// Use `openssl dgst -binary -sha1` to obtain the hash of the data, and then
// `openssl pkeyutl` with `-pkeyopt rsa_padding_mode:pss`, `-pkeyopt
// digest:sha1`, and `-pkeyopt rsa_pss_saltlen:-1` to create the signature.
static const uint8_t kRsaPssSha1Signature[] = {
  0xa9, 0xd9, 0x36, 0x19, 0x90, 0x3d, 0x58, 0xa6, 0x66, 0xae, 0x8e, 0xab,
  0xb5, 0x7d, 0xe7, 0x8c, 0xcf, 0x3c, 0x69, 0xd8, 0xba, 0x22, 0xe1, 0x35,
  0x9a, 0x88, 0xfd, 0xf1, 0x01, 0x75, 0x71, 0x05, 0x60, 0xcf, 0x05, 0x76,
  0x18, 0x38, 0xa6, 0x74, 0xaf, 0x36, 0x90, 0xb5, 0xad, 0x3c, 0xde, 0xe3,
  0x5f, 0x86, 0x3b, 0x38, 0x05, 0xa0, 0xae, 0xa2, 0x5c, 0xa9, 0xc8, 0x35,
  0xa1, 0x86, 0x95, 0xcb, 0xb0, 0xd8, 0x91, 0x27, 0x7f, 0x9b, 0x5f, 0xbe,
  0xd7, 0x99, 0x39, 0x88, 0x9a, 0xe8, 0xeb, 0x8a, 0x8b, 0xf2, 0xd3, 0x39,
  0xca, 0xbc, 0x29, 0x9c, 0xf4, 0x8d, 0x2d, 0x86, 0xbc, 0x13, 0xfc, 0xc4,
  0x9f, 0x1f, 0x65, 0x7c, 0x86, 0x10, 0x36, 0x8d, 0x43, 0xf5, 0x55, 0xa4,
  0xd2, 0x5e, 0x0a, 0x8f, 0x6a, 0x2e, 0x9e, 0x2f, 0xe2, 0xb9, 0x32, 0x4a,
  0x00, 0x5a, 0xc6, 0x1a, 0x75, 0x44, 0x59, 0x5f, 0xfc, 0xe3, 0xf6, 0xfb,
  0xc1, 0x95, 0x10, 0x18, 0x2c, 0xed, 0xfd, 0x23, 0x0c, 0x48, 0x14, 0xf9,
  0x5c, 0x04, 0x9b, 0xb7, 0x7a, 0x23, 0xf6, 0x82, 0x78, 0x92, 0xe9, 0xa8,
  0xc6, 0x1b, 0xa9, 0xad, 0xcc, 0x4a, 0xac, 0x56, 0x01, 0x02, 0x81, 0xa9,
  0xa4, 0x34, 0xb2, 0xdd, 0xd3, 0xc4, 0x01, 0x97, 0x84, 0x3f, 0x85, 0x2c,
  0x7e, 0x66, 0x4a, 0x98, 0x1f, 0xf5, 0x1e, 0x2e, 0x62, 0x09, 0x4e, 0x47,
  0xd1, 0x39, 0x93, 0x13, 0x53, 0x48, 0x90, 0xb4, 0xb8, 0x3f, 0x3c, 0x6b,
  0x87, 0x36, 0x24, 0x8f, 0xf5, 0xb8, 0x9d, 0xb2, 0x58, 0x8f, 0x55, 0xbd,
  0x2f, 0xf6, 0xce, 0x05, 0xd3, 0xc5, 0x6b, 0x84, 0x6c, 0x1b, 0x77, 0x00,
  0x7e, 0xfb, 0xa4, 0x01, 0x0f, 0xf5, 0x9d, 0x9a, 0xc7, 0xe3, 0xa4, 0xdb,
  0xac, 0x87, 0x3b, 0x50, 0x66, 0xd2, 0xf5, 0xaf, 0x51, 0x28, 0xb4, 0x5e,
  0x6e, 0xca, 0x74, 0xaa
};

// As above, but with sha256.
static const uint8_t kRsaPssSha256Signature[] = {
  0x34, 0x01, 0x2e, 0x0b, 0xba, 0x00, 0x2a, 0x66, 0x50, 0x5a, 0x6c, 0xf8,
  0x3b, 0x7f, 0xda, 0x43, 0xd1, 0x56, 0x75, 0xc8, 0xa9, 0x7a, 0xa9, 0x53,
  0xc8, 0xb3, 0xca, 0x92, 0x68, 0x8d, 0x68, 0x90, 0x93, 0xf3, 0xfd, 0xb3,
  0x66, 0x3d, 0x81, 0xe4, 0x6e, 0x7d, 0x46, 0x57, 0x86, 0x0d, 0xd5, 0x58,
  0x46, 0x27, 0x4f, 0xae, 0xab, 0x55, 0x1f, 0x25, 0xdd, 0x43, 0xac, 0x72,
  0x50, 0x0c, 0x24, 0x87, 0xbb, 0xf9, 0x08, 0x4a, 0x47, 0x43, 0xff, 0x7a,
  0x40, 0x8a, 0xd6, 0x69, 0x99, 0xe8, 0x65, 0x66, 0x4d, 0x73, 0x9f, 0xc5,
  0x43, 0x60, 0x44, 0xc6, 0xf8, 0x8a, 0xbd, 0xf2, 0x8a, 0xfc, 0x9f, 0x22,
  0xa6, 0x10, 0x4a, 0xa4, 0x8b, 0x70, 0x5b, 0xa1, 0x7c, 0x8c, 0x1a, 0xc3,
  0xf2, 0x84, 0xde, 0x03, 0x41, 0x7d, 0xe1, 0x38, 0x09, 0xa2, 0x78, 0x14,
  0xf4, 0x2c, 0xa6, 0xf6, 0x8a, 0x1e, 0xcc, 0x91, 0xbf, 0x6c, 0x14, 0xe4,
  0xc1, 0x65, 0xc7, 0x74, 0x89, 0x49, 0x24, 0x5c, 0xc6, 0x85, 0x58, 0x35,
  0x76, 0x9b, 0x9c, 0xa1, 0x3e, 0xef, 0xf7, 0x57, 0xc7, 0x18, 0x70, 0x98,
  0x1b, 0xa1, 0x12, 0xae, 0x69, 0x85, 0x1d, 0x0d, 0xa6, 0xec, 0x94, 0x7f,
  0xab, 0x77, 0x08, 0xfc, 0x89, 0x69, 0x9c, 0x51, 0xa1, 0x01, 0xd6, 0xc8,
  0xc2, 0xbf, 0xc7, 0x63, 0xa3, 0xce, 0xca, 0xc9, 0x11, 0xfd, 0x78, 0x62,
  0x5d, 0x80, 0x15, 0x10, 0x75, 0xcd, 0xd8, 0xc9, 0x7f, 0xd2, 0xcb, 0x32,
  0x35, 0x0d, 0xc0, 0x56, 0x0e, 0xa4, 0xa5, 0x2d, 0xe3, 0xeb, 0x1e, 0x30,
  0x60, 0x35, 0x1d, 0xe8, 0xb5, 0x27, 0xa1, 0x46, 0x61, 0x74, 0xb7, 0xf6,
  0x2c, 0x50, 0x4f, 0x23, 0xb3, 0xb0, 0x6b, 0xaf, 0x8d, 0x1a, 0x74, 0x27,
  0x58, 0x58, 0x25, 0xc9, 0x45, 0x84, 0xd8, 0x5d, 0xe8, 0x7e, 0xc6, 0x3e,
  0xd4, 0xf2, 0xb1, 0x3f
};

// As above, but with sha384.
static const uint8_t kRsaPssSha384Signature[] = {
  0x42, 0xd1, 0x90, 0xb2, 0x78, 0x76, 0x97, 0x5f, 0xf1, 0x4c, 0xab, 0x1b,
  0xe0, 0x80, 0xd6, 0x32, 0xac, 0x57, 0x93, 0x2e, 0x03, 0xb7, 0xec, 0x9f,
  0x48, 0x72, 0x91, 0x12, 0xb7, 0x2b, 0xaf, 0x69, 0xc6, 0x2d, 0xde, 0x6a,
  0xdc, 0xcb, 0xaf, 0xf8, 0x84, 0x2b, 0x13, 0xf9, 0xb9, 0x60, 0x76, 0x36,
  0x13, 0x63, 0xcb, 0xdd, 0x22, 0xed, 0xa2, 0x07, 0xac, 0x44, 0xee, 0x40,
  0x01, 0x19, 0x0e, 0xdd, 0x1d, 0xeb, 0x0a, 0xe7, 0xd2, 0x71, 0x1d, 0xea,
  0x66, 0x9f, 0x61, 0x76, 0xee, 0xbf, 0x50, 0xa1, 0x47, 0x17, 0x32, 0x58,
  0x03, 0x32, 0xd5, 0x39, 0x37, 0xee, 0x30, 0x9c, 0x87, 0xed, 0xc3, 0x07,
  0xde, 0x36, 0xc6, 0x65, 0x55, 0xd4, 0xd3, 0xbc, 0x4a, 0x83, 0x70, 0xd3,
  0x8d, 0xd2, 0xe0, 0x65, 0x8c, 0xe1, 0xd8, 0x40, 0xe2, 0x83, 0x52, 0x4f,
  0xf2, 0xe5, 0x16, 0x9a, 0x9b, 0x9b, 0xf2, 0x51, 0x97, 0xb0, 0x64, 0xf9,
  0x78, 0x07, 0xcc, 0x48, 0xe4, 0xe5, 0xeb, 0x69, 0x31, 0xe7, 0x10, 0xb9,
  0xa9, 0xdd, 0x06, 0xb0, 0x9e, 0x06, 0xed, 0xe1, 0x21, 0x3d, 0xcb, 0xfb,
  0x0b, 0x84, 0x78, 0xbb, 0x1c, 0x3c, 0x5e, 0x56, 0x29, 0xd8, 0x85, 0xe1,
  0x6b, 0x14, 0xee, 0xad, 0xf8, 0x8c, 0xb7, 0xfb, 0xdb, 0x48, 0x89, 0xa2,
  0x2f, 0x98, 0x4c, 0xcc, 0x62, 0xbe, 0x1d, 0x5e, 0xe2, 0x59, 0x11, 0x4b,
  0xfd, 0x93, 0xd0, 0xe2, 0x93, 0x9a, 0x8e, 0xc6, 0x97, 0x99, 0x6b, 0x9f,
  0x81, 0x96, 0x76, 0x78, 0x6f, 0xe9, 0xf3, 0x3d, 0xe2, 0xe2, 0xd6, 0x8a,
  0x3e, 0xb3, 0xff, 0x33, 0xfb, 0x1d, 0x54, 0xa9, 0xcd, 0xe8, 0x58, 0x34,
  0xd7, 0x9e, 0x5f, 0x0a, 0xcd, 0x38, 0xbd, 0x67, 0x2c, 0x4e, 0xbd, 0x11,
  0xed, 0x4a, 0xfd, 0x15, 0x35, 0x14, 0x0d, 0xfa, 0xd9, 0xa4, 0xf7, 0x6c,
  0x5f, 0x19, 0xbd, 0x1c
};

// As above, but with sha512.
static const uint8_t kRsaPssSha512Signature[] = {
  0xbf, 0xbc, 0xa6, 0x9d, 0x9b, 0x25, 0xd0, 0x0f, 0x13, 0x2c, 0xa4, 0x36,
  0x5d, 0xab, 0x06, 0xba, 0x7e, 0x0e, 0xc8, 0x30, 0x6e, 0x8b, 0x36, 0x19,
  0x28, 0xdd, 0x41, 0xc7, 0xba, 0x94, 0x0c, 0x0a, 0x46, 0x83, 0xdd, 0xdf,
  0xf0, 0x99, 0xd6, 0x07, 0xa9, 0xc4, 0x93, 0xd9, 0x1e, 0x98, 0xac, 0xa0,
  0xb2, 0x35, 0x95, 0x98, 0x89, 0x4f, 0x6b, 0x81, 0x0e, 0x4d, 0xc1, 0x94,
  0xb8, 0xc2, 0x4c, 0x33, 0xd2, 0x2b, 0xd6, 0xbd, 0xb8, 0x2f, 0x04, 0x24,
  0x64, 0x94, 0xfe, 0x02, 0x36, 0x83, 0xce, 0xe1, 0x47, 0xf3, 0xb3, 0x4a,
  0xc0, 0x7e, 0xe4, 0x6b, 0x3b, 0x0d, 0xcd, 0x83, 0x6e, 0x93, 0xd2, 0x5e,
  0x40, 0xdb, 0xe1, 0xe2, 0x08, 0xd1, 0xc3, 0x7a, 0xfb, 0xdf, 0xe5, 0x91,
  0x01, 0xb2, 0xd9, 0xcd, 0x15, 0xeb, 0x15, 0xcc, 0xf4, 0xa6, 0xdd, 0x87,
  0x31, 0x9b, 0xcb, 0x19, 0x38, 0x00, 0x99, 0xfe, 0xcb, 0xfb, 0x41, 0xda,
  0xce, 0x45, 0xf2, 0x65, 0x5b, 0x3b, 0x9d, 0x34, 0x7e, 0x48, 0x40, 0x63,
  0x67, 0xfa, 0xbe, 0x34, 0xaf, 0xf5, 0x69, 0xa3, 0x40, 0x8d, 0x1b, 0xae,
  0x84, 0x0f, 0x3b, 0x1f, 0xc4, 0x00, 0x8b, 0x8a, 0x64, 0x9f, 0xca, 0xfc,
  0x0c, 0x58, 0x6e, 0xfd, 0x8a, 0xb5, 0x11, 0x3d, 0x2b, 0x7d, 0xf1, 0xdd,
  0x08, 0xe9, 0x1d, 0x27, 0x18, 0x1e, 0x31, 0xc1, 0xec, 0x10, 0xe6, 0x5b,
  0x39, 0xcc, 0x9f, 0x2e, 0x9c, 0x41, 0x4c, 0x47, 0x77, 0x36, 0x5f, 0x2c,
  0x5b, 0x8c, 0x16, 0x96, 0x7f, 0x1f, 0xa8, 0x30, 0x3b, 0x34, 0x2f, 0xa2,
  0x44, 0x41, 0x49, 0x68, 0x3a, 0x75, 0xa5, 0xb4, 0xde, 0xd2, 0x89, 0x72,
  0x7c, 0x6e, 0x83, 0x54, 0x30, 0x9c, 0x1e, 0x9b, 0x58, 0xb0, 0xa3, 0x2c,
  0xdf, 0x3d, 0x69, 0xff, 0xe1, 0x4c, 0x28, 0x4a, 0x19, 0xec, 0x0b, 0x4c,
  0x0a, 0x7e, 0xb7, 0xd4
};

static const uint8_t kEC256SubjectPublicKeyInfo[] = {
  0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
  0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
  0x42, 0x00, 0x04, 0x6d, 0xd3, 0xc0, 0x54, 0xc6, 0x33, 0xd3, 0xff, 0x4a,
  0x43, 0x0c, 0x3a, 0x70, 0xdb, 0x97, 0xf6, 0xc9, 0x68, 0xa0, 0xb1, 0xb9,
  0x6e, 0x15, 0x20, 0x80, 0x50, 0x6f, 0x37, 0x8f, 0xcc, 0xe3, 0x47, 0x96,
  0x2c, 0x5d, 0xb8, 0x76, 0x8e, 0x67, 0x19, 0x1c, 0xc7, 0x64, 0xb4, 0xd5,
  0xda, 0xa6, 0x4d, 0xc5, 0x3d, 0xe6, 0xa7, 0xb3, 0xb0, 0x41, 0xfb, 0x29,
  0xb4, 0x2f, 0xe8, 0x72, 0xeb, 0xfb, 0xfe
};

// Similar to above, but without the RSA padding mode option.
static const uint8_t kEC256Sha1Signature[] = {
  0x30, 0x46, 0x02, 0x21, 0x00, 0xa7, 0x66, 0xd0, 0x2c, 0x3e, 0x2c, 0x06,
  0x28, 0x84, 0x3f, 0x59, 0xe7, 0x21, 0x44, 0x7d, 0x5a, 0xca, 0xd6, 0xd5,
  0xe8, 0x7d, 0x0a, 0xc5, 0x26, 0x99, 0x93, 0x43, 0xf1, 0x7d, 0x07, 0x08,
  0xdb, 0x02, 0x21, 0x00, 0xfc, 0x02, 0xea, 0xcb, 0x3b, 0x7e, 0xd3, 0x2e,
  0x86, 0x09, 0x18, 0x1a, 0x46, 0x4b, 0x98, 0xfa, 0x7c, 0x4c, 0x2f, 0xf8,
  0x6e, 0xf7, 0x42, 0x1e, 0x5d, 0x73, 0x0b, 0x27, 0xbf, 0xd7, 0x43, 0xeb
};

static const uint8_t kEC256Signature[] = {
  0x30, 0x45, 0x02, 0x21, 0x00, 0xfa, 0x47, 0x24, 0x2e, 0xc0, 0x01, 0x6c,
  0x31, 0xca, 0x75, 0x12, 0xa4, 0x02, 0x78, 0x0d, 0xc6, 0x75, 0x30, 0x1f,
  0x31, 0xec, 0xc1, 0xa1, 0x1c, 0xc9, 0x3e, 0xcd, 0xa5, 0x69, 0xe2, 0x06,
  0x61, 0x02, 0x20, 0x7f, 0x94, 0x6e, 0x60, 0x2e, 0xdd, 0x06, 0xa6, 0xd6,
  0x90, 0x74, 0x36, 0x4f, 0xf4, 0xc5, 0x80, 0x8f, 0xf6, 0xf9, 0x3b, 0x6c,
  0xee, 0x28, 0x94, 0x93, 0xce, 0xff, 0x1e, 0xfc, 0x84, 0x10, 0x65
};

static const uint8_t kEC384SubjectPublicKeyInfo[] = {
  0x30, 0x76, 0x30, 0x10, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
  0x01, 0x06, 0x05, 0x2b, 0x81, 0x04, 0x00, 0x22, 0x03, 0x62, 0x00, 0x04,
  0x6c, 0x7a, 0x7b, 0x3f, 0x85, 0x0d, 0x04, 0x8d, 0x8c, 0x90, 0x0a, 0x9e,
  0x42, 0x54, 0xfa, 0x64, 0x78, 0xd3, 0x08, 0x91, 0xe0, 0xaa, 0xc1, 0x03,
  0x0c, 0x6c, 0xf1, 0x8c, 0xa6, 0x32, 0x9d, 0x57, 0x10, 0x38, 0xf2, 0xee,
  0x27, 0x1c, 0xa8, 0x11, 0xd0, 0x1c, 0x50, 0x24, 0xd9, 0x15, 0x17, 0x3c,
  0xe8, 0x29, 0x67, 0x4b, 0x06, 0xe3, 0x89, 0x52, 0x62, 0x0f, 0xd6, 0x12,
  0x95, 0x52, 0x21, 0x3e, 0x5d, 0x7b, 0x5b, 0x04, 0xdc, 0xa1, 0x49, 0x51,
  0xf0, 0xed, 0x35, 0x6b, 0x1c, 0xdd, 0x09, 0x8f, 0xdf, 0x1e, 0x0a, 0x39,
  0xd4, 0x92, 0x38, 0x1b, 0x0a, 0xce, 0xea, 0xf9, 0x6a, 0x0e, 0x37, 0x4d
};

static const uint8_t kEC384Signature[] = {
  0x30, 0x65, 0x02, 0x31, 0x00, 0x8c, 0xdf, 0x30, 0xa4, 0x06, 0x2b, 0x15,
  0x2f, 0x2e, 0x46, 0xbb, 0x3e, 0xd1, 0xa7, 0x87, 0x29, 0x2a, 0x30, 0x74,
  0x29, 0x31, 0x38, 0x2a, 0x05, 0x74, 0x1c, 0xbd, 0xe6, 0xf4, 0x4b, 0x8d,
  0x8e, 0xa9, 0x42, 0xad, 0xea, 0x62, 0x0d, 0x36, 0xa2, 0xde, 0x6e, 0x99,
  0x7f, 0xa5, 0x51, 0x42, 0x1e, 0x02, 0x30, 0x35, 0xad, 0x08, 0x88, 0x52,
  0x40, 0x2e, 0x91, 0x44, 0xf2, 0x07, 0x0a, 0x92, 0x39, 0x35, 0x1d, 0x15,
  0xeb, 0x90, 0xc4, 0xd9, 0xfb, 0x83, 0xae, 0x99, 0xaa, 0x50, 0x4e, 0xc9,
  0x35, 0x1d, 0x3a, 0x8d, 0x40, 0x9a, 0x91, 0x14, 0x6c, 0x0d, 0x02, 0x25,
  0x7e, 0x48, 0x8b, 0x98, 0x9d, 0x3f, 0x14
};

static const uint8_t kEC521SubjectPublicKeyInfo[] = {
  0x30, 0x81, 0x9b, 0x30, 0x10, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d,
  0x02, 0x01, 0x06, 0x05, 0x2b, 0x81, 0x04, 0x00, 0x23, 0x03, 0x81, 0x86,
  0x00, 0x04, 0x01, 0x59, 0x3c, 0x05, 0xb1, 0x74, 0x49, 0x54, 0x72, 0xea,
  0xf5, 0xb0, 0x43, 0x98, 0x39, 0xfe, 0xc6, 0x4e, 0x8a, 0x58, 0x62, 0x83,
  0xe7, 0x61, 0xeb, 0xf6, 0x50, 0x32, 0xa6, 0x1e, 0x7c, 0x6a, 0x7c, 0xa5,
  0x4e, 0x5b, 0x96, 0x77, 0xfd, 0xf5, 0x5c, 0xe8, 0x61, 0x32, 0xc2, 0xd2,
  0xcf, 0x06, 0xef, 0x46, 0xf1, 0x51, 0x50, 0x35, 0xf7, 0xac, 0x97, 0x7a,
  0xfb, 0xe8, 0x9f, 0xa5, 0xc6, 0x32, 0x8a, 0xdf, 0x00, 0x9d, 0x6f, 0xd7,
  0x34, 0xa7, 0xe5, 0x1d, 0x85, 0x4b, 0xc2, 0x26, 0x38, 0x8c, 0x6f, 0x8f,
  0x75, 0xd8, 0x47, 0x08, 0xe7, 0xcb, 0x78, 0x9f, 0x3d, 0xe8, 0x5a, 0xe5,
  0xa1, 0x74, 0x56, 0x80, 0x62, 0x7e, 0x95, 0x2a, 0x2d, 0x6e, 0x27, 0x71,
  0x02, 0xf1, 0xbf, 0x5d, 0xe4, 0x7e, 0xab, 0x94, 0x92, 0x24, 0x0f, 0x31,
  0x51, 0x33, 0x67, 0x6a, 0xfb, 0x29, 0xae, 0xa9, 0xaf, 0xdd, 0x09, 0x77,
  0x08, 0xf8
};

static const uint8_t kEC521Signature[] = {
  0x30, 0x81, 0x87, 0x02, 0x41, 0x56, 0xa2, 0x7e, 0xff, 0x54, 0xe2, 0xb4,
  0x23, 0x08, 0xf8, 0xde, 0x28, 0x63, 0x77, 0x3b, 0x0f, 0x18, 0x7b, 0x5d,
  0xac, 0x2f, 0x57, 0x04, 0xbe, 0x15, 0x7b, 0x04, 0xb1, 0xd2, 0x7f, 0x21,
  0x14, 0x40, 0x18, 0xb2, 0x27, 0xe4, 0xef, 0x14, 0x54, 0x9c, 0xca, 0x90,
  0xd8, 0x95, 0x3a, 0xfe, 0xf8, 0xe4, 0x60, 0x98, 0x45, 0x31, 0x5e, 0x3b,
  0xac, 0x89, 0x48, 0x0b, 0xad, 0xe2, 0xeb, 0xa5, 0xec, 0xae, 0x02, 0x42,
  0x01, 0xc9, 0x16, 0x18, 0x30, 0x1b, 0xdb, 0xbb, 0x99, 0x47, 0x32, 0xa0,
  0x17, 0x02, 0xef, 0x30, 0x72, 0x77, 0x27, 0x06, 0x0c, 0x6d, 0x4a, 0x01,
  0xba, 0xb0, 0x30, 0xec, 0x8b, 0x35, 0x44, 0x9a, 0xe5, 0xd2, 0x15, 0x74,
  0x69, 0x67, 0x2a, 0xe8, 0x5a, 0xd3, 0xa4, 0x91, 0x39, 0x49, 0x60, 0x4f,
  0x90, 0x78, 0xeb, 0xb4, 0xc0, 0x77, 0x53, 0xf7, 0xf5, 0x44, 0xfc, 0x29,
  0x86, 0xaf, 0x95, 0x6f, 0x56, 0xcd
};

static const VerifySignedDataNSSTestParams
  VERIFYSIGNEDDIGESTNSS_TEST_PARAMS[] =
{
  {
    BS(kData),
    DigestAlgorithm::sha1,
    BS(kRsaPkcs1Sha1Signature),
    PublicKeyAlgorithm::RSA_PKCS1,
    BS(kRsaSubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kRsaPkcs1Sha256Signature),
    PublicKeyAlgorithm::RSA_PKCS1,
    BS(kRsaSubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha384,
    BS(kRsaPkcs1Sha384Signature),
    PublicKeyAlgorithm::RSA_PKCS1,
    BS(kRsaSubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha512,
    BS(kRsaPkcs1Sha512Signature),
    PublicKeyAlgorithm::RSA_PKCS1,
    BS(kRsaSubjectPublicKeyInfo),
    Success,
  },
  // SHA-1 is not allowed for RSA-PSS
  {
    BS(kData),
    DigestAlgorithm::sha1,
    BS(kRsaPssSha1Signature),
    PublicKeyAlgorithm::RSA_PSS,
    BS(kRsaPssSubjectPublicKeyInfo),
    Result::ERROR_CERT_SIGNATURE_ALGORITHM_DISABLED,
  },
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kRsaPssSha256Signature),
    PublicKeyAlgorithm::RSA_PSS,
    BS(kRsaPssSubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha384,
    BS(kRsaPssSha384Signature),
    PublicKeyAlgorithm::RSA_PSS,
    BS(kRsaPssSubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha512,
    BS(kRsaPssSha512Signature),
    PublicKeyAlgorithm::RSA_PSS,
    BS(kRsaPssSubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha1,
    BS(kEC256Sha1Signature),
    PublicKeyAlgorithm::ECDSA,
    BS(kEC256SubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kEC256Signature),
    PublicKeyAlgorithm::ECDSA,
    BS(kEC256SubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha384,
    BS(kEC384Signature),
    PublicKeyAlgorithm::ECDSA,
    BS(kEC384SubjectPublicKeyInfo),
    Success,
  },
  {
    BS(kData),
    DigestAlgorithm::sha512,
    BS(kEC521Signature),
    PublicKeyAlgorithm::ECDSA,
    BS(kEC521SubjectPublicKeyInfo),
    Success,
  },
  // Wrong digest algorithm - RSA PKCS#1
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kRsaPkcs1Sha384Signature),
    PublicKeyAlgorithm::RSA_PKCS1,
    BS(kRsaSubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // Wrong digest algorithm - RSA PSS
  {
    BS(kData),
    DigestAlgorithm::sha512,
    BS(kRsaPssSha384Signature),
    PublicKeyAlgorithm::RSA_PSS,
    BS(kRsaPssSubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // Wrong digest algorithm - ECDSA
  {
    BS(kData),
    DigestAlgorithm::sha1,
    BS(kEC256Signature),
    PublicKeyAlgorithm::ECDSA,
    BS(kEC256SubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // ECDSA key for RSA PKCS#1 signature
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kRsaPkcs1Sha256Signature),
    PublicKeyAlgorithm::ECDSA,
    BS(kEC256SubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // RSA PKCS#1 key for ECDSA signature
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kEC256Signature),
    PublicKeyAlgorithm::RSA_PKCS1,
    BS(kRsaSubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // RSA PKCS#1 key for RSA PSS signature
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kRsaPssSha256Signature),
    PublicKeyAlgorithm::RSA_PKCS1,
    BS(kRsaSubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // RSA PSS key for RSA PKCS#1 signature
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kRsaPkcs1Sha256Signature),
    PublicKeyAlgorithm::RSA_PSS,
    BS(kRsaPssSubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // ECDSA key for RSA PSS signature
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kRsaPssSha256Signature),
    PublicKeyAlgorithm::ECDSA,
    BS(kEC256SubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // RSA PSS key for ECDSA signature
  {
    BS(kData),
    DigestAlgorithm::sha256,
    BS(kEC256Signature),
    PublicKeyAlgorithm::RSA_PSS,
    BS(kRsaPssSubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
  // Wrong data.
  {
    BS(kRsaSubjectPublicKeyInfo),
    DigestAlgorithm::sha384,
    BS(kEC384Signature),
    PublicKeyAlgorithm::ECDSA,
    BS(kEC384SubjectPublicKeyInfo),
    Result::ERROR_BAD_SIGNATURE,
  },
};

class pkixnss_VerifySignedDataNSS
  : public ::testing::Test
  , public ::testing::WithParamInterface<VerifySignedDataNSSTestParams>
{
};

void CheckVerifySignedData(PublicKeyAlgorithm publicKeyAlgorithm, Input data,
    DigestAlgorithm digestAlgorithm, Input signature, Input subjectPublicKeyInfo,
    Result expectedResult)
{
  switch (publicKeyAlgorithm) {
    case PublicKeyAlgorithm::RSA_PKCS1:
      ASSERT_EQ(expectedResult,
                VerifyRSAPKCS1SignedDataNSS(data, digestAlgorithm, signature,
                                            subjectPublicKeyInfo, nullptr));
      break;
    case PublicKeyAlgorithm::RSA_PSS:
      ASSERT_EQ(expectedResult,
                VerifyRSAPSSSignedDataNSS(data, digestAlgorithm, signature,
                                          subjectPublicKeyInfo, nullptr));
      break;
    case PublicKeyAlgorithm::ECDSA:
      ASSERT_EQ(expectedResult,
                VerifyECDSASignedDataNSS(data, digestAlgorithm, signature,
                                         subjectPublicKeyInfo, nullptr));
      break;
    default:
      ASSERT_FALSE(true);
  }
}

TEST_P(pkixnss_VerifySignedDataNSS, VerifySignedDataNSS)
{
  const VerifySignedDataNSSTestParams& params(GetParam());

  ByteString data(params.data);
  Input dataInput;
  ASSERT_EQ(Success, dataInput.Init(data.data(), data.length()));
  ByteString signature(params.signature);
  Input signatureInput;
  ASSERT_EQ(Success, signatureInput.Init(signature.data(), signature.length()));

  Input subjectPublicKeyInfo;
  ASSERT_EQ(Success,
            subjectPublicKeyInfo.Init(params.subjectPublicKeyInfo.data(),
                                      params.subjectPublicKeyInfo.length()));
  CheckVerifySignedData(params.publicKeyAlgorithm, dataInput,
      params.digestAlgorithm, signatureInput, subjectPublicKeyInfo,
      params.expectedResult);

  if (params.expectedResult == Success) {
    signature[signature.length() - 4] = ~signature[signature.length() - 4];

    CheckVerifySignedData(params.publicKeyAlgorithm, dataInput,
      params.digestAlgorithm, signatureInput, subjectPublicKeyInfo,
      Result::ERROR_BAD_SIGNATURE);

    signature[signature.length() - 4] = ~signature[signature.length() - 4];
    data[data.length() - 10] = ~data[data.length() - 10];

    CheckVerifySignedData(params.publicKeyAlgorithm, dataInput,
      params.digestAlgorithm, signatureInput, subjectPublicKeyInfo,
      Result::ERROR_BAD_SIGNATURE);
  }
}

INSTANTIATE_TEST_SUITE_P(
  pkixnss_VerifySignedDataNSS, pkixnss_VerifySignedDataNSS,
  testing::ValuesIn(VERIFYSIGNEDDIGESTNSS_TEST_PARAMS));
