/*  shiftoperators.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

/** Commutative Operators OP, OPPROD, OPSUM and the symbolic
 * integrals and equations INTGeneric, IdentityGeneric IdentityGenericList.
 *
 *   OP(i)  INTGeneric(a_1,...,a_i,...a_n)  =  INTGeneric(a_1,...,a_i + 1,...a_n)
 *   OP(-i) INTGeneric(a_1,...,a_i,...a_n)  =  INTGeneric(a_1,...,a_i - 1,...a_n)
 *
 * with a_i from Z   and   INTGeneric: Z^n --> R   and   OP(0) = identity
 */

#ifndef SHIFTOPERATORS_H_
#define SHIFTOPERATORS_H_

#include <ginac/ginac.h>

namespace Reduze {

/// single shift operator OP(i)
class OP {
public:
	/// constructs the unity  OP(0)
	OP() :
		n_(0) {
	}
	/// constructs OP(n)
	explicit OP(int n) :
		n_(n) {
	}
	/// returns the inverse
	OP inverse() const {
		return OP(-n_);
	}

	// access to the index
	int n() const {
		return n_;
	}

	//comparison
	bool operator==(const OP & other) const {
		return (n_ == other.n_);
	}
	bool operator!=(const OP & other) const {
		return (n_ != other.n_);
	}
	bool operator<(const OP & other) const {
		return (n_ < other.n_);
	}

private:
	/// increases/decreases position i of the INTGeneric(a1,....,an)
	// n = +1 increases a1, n = -1 decreases a1, one OP shifts only by one
	int n_;

	friend std::ostream & operator<<(std::ostream& strm, const OP& in);
};

/// product of operators OP(i_1), ... , OP(i_k)
class OPPROD {
public:
	typedef std::list<OP>::const_iterator const_iterator;

	/// standard constructor gives the unity
	OPPROD() :
		prod_(1, OP()) {
	}

	OPPROD(const OP& op) :
		prod_(1, op) {
	}

	const_iterator begin() const {
		return prod_.begin();
	}
	const_iterator end() const {
		return prod_.end();
	}
	size_t size() const {
		return prod_.size();
	}

	bool operator==(const OPPROD& other) const;
	bool operator<(const OPPROD& other) const;

	OPPROD& operator*=(const OPPROD& other);

	OPPROD inverse() const;

private:
	std::list<OP> prod_;

private:
	/// sorts the products in ascending order and removes/replaces terms op*op^1 by the unity
	void unique();
	friend OPPROD operator*(const OPPROD& a, const OPPROD& b);
	friend OPPROD pow(const OPPROD& a, int i);
	friend std::ostream& operator<<(std::ostream& strm, const OPPROD& in);
};

/// sum of instances of OPPROD
class OPSUM {
public:
	typedef std::map<OPPROD, GiNaC::ex>::const_iterator const_iterator;

	/// construct zero operator
	OPSUM() {
	}
	/// construct from a product of operators with coefficient
	OPSUM(const OPPROD& p, const GiNaC::ex coeff);

	/// add an operator product to the sum
	void insert(const OPPROD& i, const GiNaC::ex& coeff);

	const_iterator begin() const {
		return sum_.begin();
	}
	const_iterator end() const {
		return sum_.end();
	}

	size_t size() const {
		return sum_.size();
	}

	void clear() {
		sum_.clear();
	}

	OPSUM& operator+=(const OPSUM& other);
	OPSUM& operator-=(const OPSUM& other);
	OPSUM& operator*=(const OPSUM& other);

private:
	std::map<OPPROD, GiNaC::ex> sum_;

	friend OPSUM operator+(const OPSUM& a, const OPSUM& b);
	friend OPSUM operator-(const OPSUM& a, const OPSUM& b);
	friend OPSUM operator*(const OPSUM& a, const OPSUM& b);
	friend OPSUM pow(const OPSUM& a, int i);
	friend std::ostream& operator<<(std::ostream& strm, const OPSUM& in);
};

} // namespace Reduze

#endif /* SHIFTOPERATORS_H_ */
