/*  propagator.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef PROPAGATOR_H_
#define PROPAGATOR_H_

#ifndef OVERRIDE
#ifdef NEW_GINAC
#define OVERRIDE override
#else
#define OVERRIDE
#endif
#endif

#include <ostream>
#include <ginac/ginac.h>

namespace Reduze {
class IntegralFamily;
class Kinematics;
class print_mma;
}

namespace YAML {
class Node;
class Emitter;
}

/// Real scalar product: commutative bilinear form
class ScalarProduct: public GiNaC::basic {
	GINAC_DECLARE_REGISTERED_CLASS(ScalarProduct, GiNaC::basic)
public:
	ScalarProduct(const GiNaC::ex& left, const GiNaC::ex& right);
	/// returns the canonical form of the ScalarProduct
	/** result: eval() called for operand expressions,
	 ** left and right operands swapped if not in canonical order,
	 ** global numerical factors are pulled out of the ScalarProduct */
#ifdef NEW_GINAC
    GiNaC::ex eval() const OVERRIDE;
#else
    GiNaC::ex eval(int level = 0) const;
#endif
	/// returns the expansion of the ScalarProduct using bilinearity
	GiNaC::ex expand(unsigned options = 0) const OVERRIDE;
	size_t nops() const OVERRIDE;
	GiNaC::ex op(size_t i) const OVERRIDE;
	GiNaC::ex & let_op(size_t i) OVERRIDE;

protected:
	void do_print_dflt(const GiNaC::print_context & c, unsigned level = 0) const;
	void do_print_mma(const Reduze::print_mma & c, unsigned level = 0) const;

private:
	GiNaC::ex left_, right_;
};

DECLARE_FUNCTION_2P (SP)

/// Raw propagator 1/(k1.k2 - m^2) without any numerator
/** note: canonicalization of arguments (e.g. 1/k^2 == 1/(-k)^2) is performed
 ** when converted to GiNaC::ex via eval().
 ** if Propagator is modified via let_op or subs etc., eval() should be called
 ** explicitely afterwards to ensure a canonical form */
class Propagator: public GiNaC::basic {
	GINAC_DECLARE_REGISTERED_CLASS(Propagator, GiNaC::basic)
public:
	Propagator(const GiNaC::ex& momentum, const GiNaC::ex& squaredmass);
	Propagator(const GiNaC::ex& k1, const GiNaC::ex& k2, const GiNaC::ex& sqm);
	virtual ~Propagator();
	/// returns the scalarproduct of the propagator
	ScalarProduct scalarproduct() const {
		return ScalarProduct(momentum1_, momentum2_);
	}
	/// returns true if the two momenta are equal
	bool has_squared_momentum() const;
	/// returns the momentum of the propagator, defined up to +/-1 factor
	/** throws an exception for generalised propagator with different momenta */
	GiNaC::ex momentum() const;
	/// returns momentum1 of the propagator
	GiNaC::ex momentum1() const;
	/// returns momentum2 of the propagator
	GiNaC::ex momentum2() const;
	/// returns the mass^2 of the propagator
	GiNaC::ex squaredmass() const {
		return squaredmass_;
	}
	/// returns (ScalarProduct(momentum1, momentum2) - squaredmass)
	GiNaC::ex inverse_ex() const;
	/// returns (q * d/dk propagator) for two momenta q and k
	/** note: the mass is taken as a fixed (k-independent) parameter ! **/
	GiNaC::ex derivative_contracted(const GiNaC::symbol& q,
			const GiNaC::symbol& k) const;

	GiNaC::ex
	mass_expand(const Reduze::IntegralFamily* fam, const GiNaC::ex& M) const;

	/// returns the canonical form of the Propagator
	/** result: Propagator with eval() called for operand expressions,
	 ** operand signs adjusted to achieve minimal forms in compare sense */
#ifdef NEW_GINAC
    GiNaC::ex eval() const OVERRIDE;
#else
    GiNaC::ex eval(int level = 0) const;
#endif
	size_t nops() const OVERRIDE;
	GiNaC::ex op(size_t i) const OVERRIDE;
	GiNaC::ex & let_op(size_t i) OVERRIDE;

	static Propagator read(const Reduze::Kinematics*,
			const GiNaC::lst& loopmoms, const YAML::Node&);
	using GiNaC::basic::print; // suppress clang warning
	virtual void print(YAML::Emitter&) const;

protected:
	void do_print_dflt(const GiNaC::print_context & c, unsigned level = 0) const;
	void do_print_mma(const Reduze::print_mma & c, unsigned level = 0) const;

private:
	GiNaC::ex momentum1_, momentum2_, squaredmass_;
};

DECLARE_FUNCTION_2P (Prop)
DECLARE_FUNCTION_3P( GProp)

#endif /* PROPAGATOR_H_ */
