/*  equation.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include <sstream>
#include <fstream>

#include <ginac/ginac.h>

#include "integralfamily.h"
#include "sectormappings.h"
#include "int.h"
#include "functions.h"
#include "equation.h"
#include "files.h"
#include "kinematics.h"
#include "sector.h"
#include "ginacutils.h"
#include "identitygenerator.h"
#include "globalsymbols.h"

#ifdef HAVE_FERMAT
#include "fermat.h"
#endif

using namespace std;
using namespace GiNaC;

namespace Reduze {

#ifdef HAVE_FERMAT
GiNaC::ex normal_form(const GiNaC::ex& c) {
	if(Fermat::instance()->is_initialized()) {
		return Fermat::instance()->normalize(c);
	}
	return GiNaC::normal(c);
}
#else //HAVE_FERMAT
GiNaC::ex normal_form(const GiNaC::ex& c) {
	return GiNaC::normal(c);
}
#endif //HAVE_FERMAT

GiNaC::ex factorized_form(const GiNaC::ex& c) {
	GiNaC::ex nd = numer_denom(c);
	return factor(nd.op(0)) / factor(nd.op(1));
}

// LinCombBasic

template<class TI, class TC>
double LinCombBasic<TI, TC>::average_coeff_length() {
	typename terms_type::const_iterator it = terms.begin();
	double length = size();
	double sum = 0.0;
	while (it != terms.end()) {
		stringstream strm;
		strm << it->second;
		string str = strm.str();
		double part = str.size();
		part /= length;
		sum += part;
		++it;
	}
	return sum;
}

template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::insert(const TI& i, const TC& e,
		bool normalize) {
	// looks a little complicated but needs only one map lookup
	typename std::pair<typename terms_type::iterator, bool> info =
			this->terms.insert(std::make_pair(i, e));
	typename terms_type::iterator it = info.first;
	bool inserted = info.second;
	if (!inserted) { // equation contained the integral already, add coeff
		it->second += e;
		if (normalize)
			it->second = normal_form(it->second);
	}
	/* alternatively:
	 typename terms_type::iterator it = this->terms.lower_bound(i);
	 if (it != this->terms.end() && !(this->terms.key_comp()(i, it->first)))
	 it->second += e;
	 else // new integral
	 it = this->terms.insert(it, typename terms_type::value_type(i, e));
	 */

	//if (it->second == 0)
	if (it->second.is_zero())
		this->terms.erase(it);
}

template<>
void LinCombBasic<INT, GiNaC::ex>::insert(const INT & i, const GiNaC::ex & e) {
	// looks a little complicated but needs only one map lookup
	std::pair<terms_type::iterator, bool> info = terms.insert(
			std::make_pair(i, e));
	terms_type::iterator it = info.first;
	bool inserted = info.second;
	if (!inserted) // equation contained the integral already, add coeff
		it->second += e;
}

template<>
void LinCombBasic<INTIndex, GiNaC::ex>::insert(const INTIndex & i,
		const GiNaC::ex & e) {
	// looks a little complicated but needs only one map lookup
	std::pair<terms_type::iterator, bool> info = terms.insert(
			std::make_pair(i, e));
	terms_type::iterator it = info.first;
	bool inserted = info.second;
	if (!inserted) // equation contained the integral already, add coeff
		it->second += e;
}

template<class TI, class TC>
void LinCombBasic<TI, TC>::insert(const TI & i, const TC & e) {
	// looks a little complicated but needs only one map lookup
	typename std::pair<typename terms_type::iterator, bool> info = terms.insert(
			std::make_pair(i, e));
	bool inserted = info.second;
	if (!inserted) // equation contained the integral already
		ABORT("Can't handle multiple occurrence of integral " << i);
}

template<class TI, class TC>
void LinCombBasic<TI, TC>::insert_begin(const TI & i, const TC & e) {
	if (terms.empty() || i < terms.begin()->first)
		terms.insert(terms.begin(), std::make_pair(i, e));
	else
		insert(i, e);
}

template<class TI, class TC>
void LinCombBasic<TI, TC>::insert_end(const TI & i, const TC & e) {
	if (terms.empty() || terms.rbegin()->first < i)
		terms.insert(terms.end(), std::make_pair(i, e));
	else
		insert(i, e);
}

template<class TI, class TC>
bool LinCombBasic<TI, TC>::operator <(
		const LinCombBasic<TI, TC> & other) const {
	// compare integrals of the equations
	typename terms_type::const_reverse_iterator it, oit;
	it = terms.rbegin();
	oit = other.terms.rbegin();
	for (; it != terms.rend() && oit != other.terms.rend(); ++it, ++oit) {
		if (it->first != oit->first)
			return (it->first < oit->first);
	}
	if (it != terms.rend() || oit != other.terms.rend())
		return (it == terms.rend());
	// both equations contain the same integrals, now check coefficients
	it = terms.rbegin();
	oit = other.terms.rbegin();
	ASSERT(size() == other.size());
	for (; it != terms.rend(); ++it, ++oit) {
		int coeffcmp = reduze_compare(it->second, oit->second);
		if (coeffcmp != 0)
			return coeffcmp < 0;
	}
	return false;
}

template<class TI, class TC>
bool LinCombBasic<TI, TC>::operator ==(
		const LinCombBasic<TI, TC> & other) const {
	if (size() != other.size())
		return false;
	// compare the more complicated integrals first
	typename terms_type::const_reverse_iterator l = terms.rbegin();
	typename terms_type::const_reverse_iterator r = other.terms.rbegin();
	for (; l != terms.rend(); ++l, ++r)
		if (l->first != r->first || reduze_compare(l->second, r->second) != 0)
			return false;
	return true;
}

template<class TI, class TC>
bool LinCombBasic<TI, TC>::operator !=(
		const LinCombBasic<TI, TC> & other) const {
	return !(*this == other);
}

// LinCombAlgebraic

template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::insert(const_iterator first,
		const_iterator last) {
	this->terms.insert(first, last);
}

template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::normalize(bool fac) {
	typename terms_type::iterator t;
	for (t = this->terms.begin(); t != this->terms.end();) {
		t->second = fac ? factorized_form(t->second) : normal_form(t->second);
		if (t->second == 0)
			this->terms.erase(t++);
		else
			++t;
	}
}

template<class TI, class TC>
bool LinCombAlgebraic<TI, TC>::has_symbol(const GiNaC::symbol & symb) const {
	typename terms_type::const_iterator it;
	for (it = this->terms.begin(); it != this->terms.end(); ++it)
		if (it->second.has(symb))
			return true;
	return false;
}

template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::series(const GiNaC::symbol& param,
		const GiNaC::ex& point, int order) {
	typename terms_type::iterator t;
	for (t = this->terms.begin(); t != this->terms.end(); ++t)
		if (t->second.has(param))
			t->second = t->second.series(param == point, order + 1);
}

template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::apply(const GiNaC::exmap & m, unsigned options) {
	typename terms_type::iterator it = this->terms.begin();
	for (; it != this->terms.end(); ++it)
		it->second = it->second.subs(m, options);
}
/*
template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::to_negative() {
	typename terms_type::iterator it = this->terms.begin();
	for (; it != this->terms.end(); ++it)
		it->second = -it->second;
}
*/
template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::replace(const EqAlgebraic<TI, TC> & sol,
		bool normalize) {
	if (sol.terms.empty())
		return;
	// find leading term with non-zero coefficient
	typename terms_type::const_iterator st = --sol.terms.end();
	ex lc;
	do {
		lc = normal_form(st->second);
	} while (lc == 0 && st != sol.terms.begin() && (--st, true));
	if (lc == 0)
		return;
	// search if this linear combination contains the leading integral
	typename terms_type::iterator it_found = this->terms.find(st->first);
	if (it_found == this->terms.end())
		return;
	// replace term in this linear combination with the solution
	coeff_type prefac = coeff_type(-1) * it_found->second / lc;
	this->terms.erase(it_found);
	while (st != sol.terms.begin() && (--st, true))
		insert(st->first, st->second * prefac, normalize);
}

// these wrappers could be avoided with a dedicated 'small INT' class with INTIndex rep
template<class TI>
void to_mma_stream(const TI& integral, std::ostream& os) {
	integral.to_mma_stream(os);
}
template<>
void to_mma_stream<INTIndex>(const INTIndex& integral, std::ostream& os) {
	os << integral;
}
template<class TI>
void to_maple_stream(const TI& integral, std::ostream& os) {
	integral.to_maple_stream(os);
}
template<>
void to_maple_stream<INTIndex>(const INTIndex& integral, std::ostream& os) {
	os << integral;
}
template<class TI>
void to_form_stream(const TI& integral, std::ostream& os) {
	integral.to_form_stream(os);
}
template<>
void to_form_stream<INTIndex>(const INTIndex& integral, std::ostream& os) {
	os << integral;
}

template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::to_mma_stream(ostream & out) const {
	out << this->name() << " -> ";
	if (this->empty()) {
		out << "0";
		return;
	}
	typename terms_type::const_reverse_iterator t;
	for (t = this->terms.rbegin(); t != this->terms.rend();) {
		// 6 spaces in front of the prefactor, 2 spaces in front of the integral
		out << "\n  ";
		Reduze::to_mma_stream(t->first, out);
		out << " *\n      (";
		subs_conjugate()(t->second).print(print_mma(out));
		//t->second.print(print_mma(out));
		out << ")";
		++t;
		if (t != this->terms.rend())
			out << " +";
	}
}

template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::to_maple_stream(ostream & out) const {
	out << this->name() << " = ";
	if (this->empty()) {
		out << "0";
		return;
	}
	typename terms_type::const_reverse_iterator t;
	for (t = this->terms.rbegin(); t != this->terms.rend();) {
		// 6 spaces in front of the prefactor, 2 spaces in front of the integral
		out << "\n  ";
		Reduze::to_maple_stream(t->first, out);
		out << " *\n      (";
		t->second.print(print_maple(out));
		out << ")";
		++t;
		if (t != this->terms.rend())
			out << " +";
	}
}

template<class TI, class TC>
void LinCombAlgebraic<TI, TC>::to_form_stream(ostream & out) const {
	out << "id " << this->name() << " = ";
	if (this->empty()) {
		out << "0;\n";
		return;
	}
	typename terms_type::const_reverse_iterator t;
	for (t = this->terms.rbegin(); t != this->terms.rend(); ++t) {
		out << "\n  + ";
		Reduze::to_form_stream(t->first, out);
		out << "\n    * (";
		t->second.print(print_form(out));
		out << ")";
	}
	out << ";\n";
}

// EqAlgebraic

template<class TI, class TC>
void EqAlgebraic<TI, TC>::solve(bool factorize) {
	if (this->terms.empty())
		return;
	typename terms_type::iterator l = --this->terms.end();
	coeff_type div = factorize ? factor(l->second) : normal_form(l->second);
	if (div == 0) {
		this->terms.erase(l);
		return solve(factorize); // recurse
	} else {
		l->second = 1;
		for (typename terms_type::iterator t = this->terms.begin(); t != l;) {
			const ex rat = t->second / div;
			t->second = factorize ? factorized_form(rat) : normal_form(rat);
			if (t->second == 0)
				this->terms.erase(t++);
			else
				++t;
		}
	}
}

template<class TI, class TC>
void EqAlgebraic<TI, TC>::quick_solve() {
	if (this->terms.empty())
		return;
	typename terms_type::iterator l = --this->terms.end();
	coeff_type div = (l->second == 1 ? 1 : normal_form(l->second));
	if (div == 0) {
		this->terms.erase(l);
		return quick_solve(); // recurse
	} else {
		l->second = 1;
		for (typename terms_type::iterator t = this->terms.begin(); t != l;) {
			t->second = t->second / div;
			if (t->second == 0)
				this->terms.erase(t++);
			else
				++t;
		}
	}
}

template<class TI, class TC>
void EqAlgebraic<TI, TC>::to_mma_stream(ostream & out) const {
	typename terms_type::const_reverse_iterator it = this->terms.rbegin();
	if (this->empty()) {
		WARNING("MMA output: Empty equation not printed");
		return;
	}
	if (this->size() == 1) {
		Reduze::to_mma_stream(it->first, out);
		out << " -> 0";
		return;
	}
	if (it->second != 1) {
		WARNING(
				"MMA output: left hand side of the equation still has a prefactor not equal to one!");
		out << "(";
		it->second.print(print_mma(out));
		out << ") * ";
	}
	Reduze::to_mma_stream(it->first, out);
	out << " -> ";
	++it;
	while (it != this->terms.rend()) {
		// 6 spaces in front of the prefactor, 2 spaces in front of the integral
		out << "\n  ";
		Reduze::to_mma_stream(it->first, out);
		out << " *\n      (";
		(-it->second).print(print_mma(out)); // with a minus sign
		out << ")";
		++it;
		if (it != this->terms.rend())
			out << " +";
	}
}

template<class TI, class TC>
void EqAlgebraic<TI, TC>::to_maple_stream(ostream & out) const {
	typename terms_type::const_reverse_iterator it = this->terms.rbegin();
	if (this->empty()) {
		WARNING("Maple output: Empty equation not printed");
		return;
	}
	if (this->size() == 1) {
		Reduze::to_maple_stream(it->first, out);
		out << " = 0";
		return;
	}
	if (it->second != 1) {
		WARNING(
				"Maple output: left hand side of the equation still has a prefactor not equal to one!");
		out << "(";
		it->second.print(print_maple(out));
		out << ") * ";
	}
	Reduze::to_maple_stream(it->first, out);
	out << " = ";
	++it;
	while (it != this->terms.rend()) {
		// 6 spaces in front of the prefactor, 2 spaces in front of the integral
		out << "\n  ";
		Reduze::to_maple_stream(it->first, out);
		out << " * \n      (";
		(-it->second).print(print_maple(out)); // with a minus sign
		out << ")";
		++it;
		if (it != this->terms.rend())
			out << " +";
	}
}

template<class TI, class TC>
void EqAlgebraic<TI, TC>::to_form_stream(ostream & out) const {
	typename terms_type::const_reverse_iterator it = this->terms.rbegin();
	if (this->empty()) {
		WARNING("FORM output: Empty equation not printed");
		return;
	}
	if (this->size() == 1) {
		out << "id ";
		Reduze::to_form_stream(it->first, out);
		out << " = 0;\n";
		return;
	}
	if (it->second != 1) {
		WARNING(
				"FORM output: left hand side of the equation still has a prefactor not equal to one!");
		out << "id (";
		it->second.print(print_form(out));
		out << ") * ";
		Reduze::to_form_stream(it->first, out);
	} else {
		out << "id ";
		Reduze::to_form_stream(it->first, out);
		out << " = ";
	}
	++it;
	while (it != this->terms.rend()) {
		out << "\n  + ";
		Reduze::to_form_stream(it->first, out);
		out << "\n    * (";
		(-it->second).print(print_form(out)); // with a minus sign
		out << ")";
		++it;
	}
	out << ";\n";
}

// helper templates

namespace {
/// returns value for given key, aborts if not found
template<class TKey, class TVal>
TVal translate(const std::map<TKey, TVal>& m, const TKey& k) {
	typename map<TKey, TVal>::const_iterator it = m.find(k);
	if (it == m.end())
		ABORT("no mapping for integral " << k << " found");
	return it->second;
}
}

// EquationLight

EquationLight::EquationLight(const Equation& eq) {
	LinCombBasic<INTIndex, GiNaC::ex>::terms_type::const_iterator it;
	for (it = eq.terms.begin(); it != eq.terms.end(); ++it)
		terms.insert(terms.end(), make_pair(it->first, to_string(it->second)));
}

EquationLight::EquationLight(const Identity& eq,
		const std::map<INT, INTIndex>& m) {
	LinCombBasic<INT, GiNaC::ex>::terms_type::const_iterator it;
	for (it = eq.terms.begin(); it != eq.terms.end(); ++it)
		insert(translate(m, it->first), to_string(it->second));
}

// EquationHLight

//EquationHLight::EquationHLight(const LinCombBasic<INT, string>& lc) {
//	LinCombBasic<INT, string>::terms_type::const_iterator t;
//	for (t = lc.begin(); t != lc.end(); ++t)
//		insert(t->first, t->second);
//	set_name("");
//}

// Equation

Equation::Equation(const EquationLight& eq) {
	const lst& symbols = //
			Files::instance()->kinematics()->kinematic_invariants();
	Equation e(eq, add_lst(symbols, Files::instance()->globalsymbols()->all()));
	e.swap_terms(*this);
}

Equation::Equation(const EquationLight& eq, const GiNaC::lst& symbols) {
	LinCombBasic<INTIndex, std::string>::terms_type::const_iterator it;
	for (it = eq.terms.begin(); it != eq.terms.end(); ++it) {
		const int i = it->first;
		GiNaC::ex coeff(it->second, symbols);
		terms.insert(terms.end(), make_pair(i, coeff));
	}
}

Equation::Equation(const Identity& eq, const std::map<INT, INTIndex>& m) {
	LinCombBasic<INT, GiNaC::ex>::terms_type::const_iterator it;
	for (it = eq.terms.begin(); it != eq.terms.end(); ++it)
		insert(translate(m, it->first), it->second, false);
}

// Identity

Identity::Identity(const EquationHLight& eq) {
	LinCombBasic<INT, std::string>::terms_type::const_iterator t;
	for (t = eq.terms.begin(); t != eq.terms.end(); ++t) {
		const lst& s = //
				t->first.integralfamily()->kinematics()->kinematic_invariants();
		terms.insert(terms.end(),
				make_pair(t->first,
						GiNaC::ex(t->second,
								add_lst(s,
										Files::instance()->globalsymbols()->all()))));
	}
}

Identity::Identity(const EquationLight& eq, const std::map<INTIndex, INT>& m) {
	LinCombBasic<INTIndex, std::string>::terms_type::const_iterator t;
	for (t = eq.terms.begin(); t != eq.terms.end(); ++t) {
		INT integ(translate(m, t->first));
		const lst& s = //
				integ.integralfamily()->kinematics()->kinematic_invariants_and_dimension();
		GiNaC::ex coeff(t->second, s);
		insert(integ, coeff, false);
	}
}

Identity::Identity(const Equation& eq, const std::map<INTIndex, INT>& m) {
	LinCombBasic<INTIndex, GiNaC::ex>::terms_type::const_iterator t;
	for (t = eq.terms.begin(); t != eq.terms.end(); ++t)
		insert(translate(m, t->first), t->second, false);
}

//Identity::Identity(const LinCombAlgebraic<INT, GiNaC::ex>& lc) {
//	LinCombBasic<INT, GiNaC::ex>::terms_type::const_iterator t;
//	for (t = lc.terms.begin(); t != lc.terms.end(); ++t)
//	    insert(t->first, t->second, false);
//	name_.clear();
//}

void Identity::toggle_metric_convention() {
	Identity id;
	for (terms_type::iterator t = terms.begin(); t != terms.end(); ++t) {
		INT i(t->first);
		i.set_metric_mostly_plus(!t->first.uses_metric_mostly_plus());
		int r = static_cast<int>(t->first.r());
		int s = static_cast<int>(t->first.s());
		GiNaC::ex c = (t->second) * pow(ex(-1), r + s);
		id.terms.insert(id.terms.end(), make_pair(i, c));
		//id.insert(i, c, false);
	}
	this->swap_terms(id);
}

/*

 // decomposes terms in a string "int1\ncoeff1\nint2\ncoeff2\n;\n"
 list<pair<string, string> > split_terms_string(const std::string& s) {
 list<pair<string, string> > res;
 size_t from = 0;
 while (true) {
 size_t to = s.find("\n", from);
 string sub1 = s.substr(from, to == string::npos ? to : to - from);
 if (sub1 == ";")
 return res;
 if (to == string::npos || to + 1 >= s.size()) // assert valid next 'from'
 throw runtime_error("incomplete terms in string");
 from = to + 1;
 to = s.find("\n", from);
 sub2 = s.substr(from, to == string::npos ? to : to - from);
 res.insert(make_pair(sub1, sub2));
 if (to == string::npos || to + 1 >= s.size()) // assert valid next 'from'
 throw runtime_error("incomplete terms in string");
 from = to + 1;
 }
 }
 // extracts key/value pairs in string "key1 = val1\nkey2 = val2\nno_more_eq_sign"
 // also returns position at which residual string without '=' sign starts
 map<string, string> extract_settings(const std::string& s, size_t& pos) {
 map<string, string> result;
 while (pos < s.size()) {
 size_t to = s.find("\n", pos);
 string sub = s.substr(pos, to == string::npos ? to : to - pos);
 size_t eq = sub.find("=");
 if (eq == string::npos)
 return result;
 string key = chop_leading_trailing_whitespaces(sub.substr(from, eq));
 string val = chop_leading_trailing_whitespaces(sub.substr(eq + 1, to));
 result[key] = val;
 pos = to + 1;
 }
 pos = string::npos;
 return result;
 }
 */

LinearCombination::LinearCombination(std::istream& is,
		const GiNaC::lst& symbols) {
	string l, r;
	getline(is, l);
	if (l.find("=") != string::npos) { // scan leading name = NAME line
		istringstream ls(l);
		string s1, s2, s3;
		ls >> s1; // name
		ls >> s2; // =
		ls >> s3; // NAME
		if (s1 != "name" || s2 != "=")
			throw runtime_error("missing name for linear combination");
		set_name(s3);
		getline(is, l);
	}
	while (is && chop_whitespaces(l) != ";" && getline(is, r)) {
		insert_begin(INT(l), GiNaC::ex(r, symbols));
		getline(is, l);
	}
	if (!is)
		throw runtime_error("syntax error in linear combination");
}

LinearCombinationGeneric::LinearCombinationGeneric(std::istream& is,
		const GiNaC::lst& symbols) {
	string l, r;
	getline(is, l);
	if (l.find("=") != string::npos) { // scan leading name = NAME line
		istringstream ls(l);
		string s1, s2, s3;
		ls >> s1; // name
		ls >> s2; // =
		ls >> s3; // NAME
		if (s1 != "name" || s2 != "=")
			throw runtime_error("missing name for generic linear combination");
		set_name(s3);
		getline(is, l);
	}
	while (is && chop_whitespaces(l) != ";" && getline(is, r)) {
		INTGeneric in(l);
		ex coeff = ex(r,
				add_lst(symbols,
						in.integralfamily()->propagator_exponent_symbols()));
		insert(in, coeff);
		getline(is, l);
	}
	if (!is)
		throw runtime_error("syntax error for generic linear combination");
}

LinCombHLight::LinCombHLight(std::istream& is, bool discard_coeff) {
	string l, r;
	getline(is, l);
	if (l.find("=") != string::npos) { // scan leading name = NAME line
		istringstream ls(l);
		string s1, s2, s3;
		ls >> s1; // name
		ls >> s2; // =
		ls >> s3; // NAME
		if (s1 != "name" || s2 != "=")
			throw runtime_error("missing name for linear combination");
		set_name(s3);
		getline(is, l);
	}
	while (is && chop_whitespaces(l) != ";" && getline(is, r)) {
		if (discard_coeff)
			insert_begin(INT(l), "");
		else
			insert_begin(INT(l), chop_whitespaces(r));
		getline(is, l);
	}
	if (!is)
		throw runtime_error("syntax error in linear combination");
}

void LinearCombination::toggle_metric_convention() {
	LinearCombination lc;
	for (terms_type::iterator t = terms.begin(); t != terms.end(); ++t) {
		INT i(t->first);
		i.set_metric_mostly_plus(!t->first.uses_metric_mostly_plus());
		int r = static_cast<int>(t->first.r());
		int s = static_cast<int>(t->first.s());
		GiNaC::ex c = (t->second) * pow(ex(-1), r + s);
		lc.terms.insert(lc.terms.end(), make_pair(i, c));
		//lc.insert(i, c, false);
	}
	this->swap_terms(lc);
}

void LinearCombination::remove_zero_integral() {
	for (terms_type::iterator t = terms.begin(); t != terms.end();)
		if (t->first.is_obviously_zero())
			terms.erase(t++);
		else
			++t;
}

void LinearCombination::apply_shifts(bool normalize) {
	set<INT> ints;
	find_INT(ints);
	for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i) {
		Identity res = IdentityGenerator::get_shift_identity(*i, false, false);
		if (!res.empty())
			replace(res, normalize);
	}
}

LinearCombination LinearCombinationGeneric::get_linear_combination(
		const INT& in, bool set_sub_secs_to_zero) const {
	const IntegralFamily* ic = in.integralfamily();
	exmap m;
	const exvector& exponents = ic->propagator_exponents();
	for (size_t j = 0; j < ic->num_propagators(); j++)
		m[exponents[j]] = static_cast<int>(in.v_i(j));
	LinearCombination result;
	for (const_iterator i = begin(); i != end(); ++i) {
		if (*i->first.integralfamily() != *ic)
			ABORT("integral family mismatch");
		ex coeff = i->second.subs(m, subs_options::no_pattern);
		INT integral(i->first, m);
		if (!set_sub_secs_to_zero || !(integral.t() < in.t()))
			result.insert(integral, coeff);
	}
	result.set_name("lin_comb");
	return result;
}

void Identity::reconstruct_symbol_replaced_by_one() {
	using namespace GiNaC;
	const Kinematics* kin = Files::instance()->kinematics();
	const symbol* norm = kin->symbol_to_replace_by_one();
	if (norm == 0) {
		LOGX("No symbol found which was set to one and needs reconstruction");
		return;
	} else if (has_symbol(*norm)) {
		ABORT(
				"Symbol " << *norm << " to reconstruct already present\n" << *this);
	}
	ex dim_norm = kin->find_mass_dimension(*norm);
	exmap recover_invars;
	const lst& invars = kin->kinematic_invariants();
	for (lst::const_iterator s = invars.begin(); s != invars.end(); ++s) {
		if (!is_a < symbol > (*s))
			ABORT("This is no symbol: " << *s);
		ex dim_s = kin->find_mass_dimension(ex_to < symbol > (*s));
		recover_invars[*s] = (*s) * pow(*norm, -dim_s / dim_norm);
	}
	terms_type::iterator t;
	for (t = terms.begin(); t != terms.end(); ++t) {
		const INT& i = t->first;
		ex& c = t->second;
		ex dim_t_rel = pow(*norm, 2 * (i.s() - i.r()) / dim_norm);
		c = normal(c.subs(recover_invars) / dim_t_rel);
	}
}

void Identity::remove_zero_integral() {
	for (terms_type::iterator t = terms.begin(); t != terms.end();)
		if (t->first.is_obviously_zero())
			terms.erase(t++);
		else
			++t;
}
/*
Identity Identity::replace_family_and_coefficients(const IntegralFamily* old_ic,
		const IntegralFamily* new_ic, const GiNaC::exmap& m) const {
	Identity id;
	for (const_iterator it = begin(); it != end(); ++it) {
		VERIFY(*it->first.integralfamily() == *old_ic);
		id.terms.insert(id.terms.end(),
				make_pair(INT(new_ic, it->first.v()), it->second.subs(m)));
	}
	return id;
}
*/
// global functions

// NEEDS an EXPANDED expression,
void propagator_products_to_integrals(const ex& e,
		LinCombAlgebraic<INT, GiNaC::ex>& lincomb, const IntegralFamily* ic) {
	if (is_a < add > (e)) {
		int num = e.nops();
		for (int i = 0; i < num; ++i) {
			INT integral(ic);
			ex coeff;
			if (!ic->match_integrand(e.op(i), integral, coeff)) {
				stringstream ss;
				ss << "can't match to propagators of family " << ic->name()
						<< ": " << e.op(i);
				throw runtime_error(ss.str());
			}
			lincomb.insert(integral.get_equivalent(), coeff, false);
		}
	} else {
		INT integral(ic);
		ex coeff;
		if (!ic->match_integrand(e, integral, coeff)) {
			stringstream ss;
			ss << "can't match to propagators of family " << ic->name() << ": "
					<< e;
			throw runtime_error(ss.str());
		}
		lincomb.insert(integral.get_equivalent(), coeff, false);
	}
}

// explicit instantiations
template class LinCombBasic<INTIndex, std::string> ;
template class LinCombBasic<INT, std::string> ;
template class LinCombBasic<INTIndex, CoeffIndex> ;
template class LinCombBasic<INTIndex, GiNaC::ex> ;
template class LinCombBasic<INT, GiNaC::ex> ;
template class LinCombBasic<INTGeneric, GiNaC::ex> ;
template class LinCombAlgebraic<INTIndex, GiNaC::ex> ;
template class LinCombAlgebraic<INT, GiNaC::ex> ;
template class LinCombAlgebraic<INTGeneric, GiNaC::ex> ;
template class EqAlgebraic<INTIndex, GiNaC::ex> ;
template class EqAlgebraic<INT, GiNaC::ex> ;
template class EqAlgebraic<INTGeneric, GiNaC::ex> ;

} // namespace Reduze
