## "Simple" analysis interface

From Rivet 4.1.0 onwards, a "simplified" API is available for
analysis-writing, particularly aimed at BSM-search analyses where
precise physics-object definitions are not as important as they are in
precision measurements.

While the standard `Analysis` interface makes/lets analyses specify
exactly what their object definitions are -- for example, precisely
which particles are to be used as jet-clustering constituents, or the
exact jet algorithm to use, or how (and which) photons should be
associated with charged leptons -- the `SimpleAnalysis` base class
extends `Analysis` to add default definitions for object
collections. This design is for familiarity with in-experiment
analysis frameworks, and tools like the popular Delphes
fast-simulator, and almost entirely eliminates reference to the
Rivet projection system. For a guide to equivalent constructions between ATLAS'
SimpleAnalysis tool and Rivet's SimpleAnalysis API, see
[this guide](migSimpleAnaToRivet4.md).


### Accessing physics objects

For example, a "normal" Rivet analysis wanting to use direct electrons
would need to define and declare a projection (using
e.g. `DirectFinalState` and/or `LeptonFinder`) in the analysis `init()`
method, and then apply and retrieve it in the `analyze()` method, e.g.

```c++
const Particles elecs = apply<ParticleFinder>(event, "Electrons").particlesByPt();
```

while in the `SimpleAnalysis` interface this already exists and can be
retrieved with simply a call to `electrons()`. Similar methods exist
for `muons()`, `taus()`, `photons()`, `tracks()` and
`jets()`/`bjets()`, as well as missing transverse momentum via
built-in `met()`, `vmet()`, and `metSignf()` methods. In all cases, the
projection mechanism is used in the background to this interface, so there
is no pre-emptive calculation: an analysis unconcerned with taus will never
expend unneccessary CPU on computing unwanted lists of tau objects.

These access methods all have variants accepting `Cut` objects, so for example
a more restricted set of muons can be retrieved with

```c++
muons(Cuts::pT > 30*GeV && Cuts::abseta < 1) .
```


### Object classes and jet definitions

The lepton (including tau) and photon collection accessor methods also
accept an `IDClass` enum to switch to the particles reconstructed via
`IDClass::LOOSE` or `IDClass::TIGHT` definitions rather than the
default `IDClass::MEDIUM`, and a further boolean to return *all*
particles of that species (including from hadron decays) as opposed to
the default directly-produced particles of most interest to BSM searches.

The `jets()` method also accepts an optional string argument to
specify the jet collection to be used, defaulting to `"DEFAULT"` for
anti-kT, $R=0.4$ jets with $p_\perp > 20$ GeV and $|\eta| < 5$. More
jet definitions can be registered in the `init()` phase, using
`setJetDef(jetscheme, jetname)` and `setJetCut(cut, jetname)`
methods. In fact, default cuts can be set in this way for all standard
object types, from a general acceptance cut `setAccCut(cut)` that
applies to all, to specific `setElectronCut()`, etc.


### Detector/reco effects

As these different object classes suggest, `SimpleAnalysis` runs
detector+reconstruction emulation by default, using Rivet's
established projection-based smearing/efficiency mechanism that allows
each analysis to specify its own detector response functions.

A new addition to that mechanism is coherent *families* of
detector-emulation functions, which can be activated all at once with
a single call, e.g. `setDetSmearing(ATLAS_RUN2, MV2C20)` to get the
generic ATLAS Run 2 response functions and the (now rather old) MV2C20
b-tagging performance. Users can overload specific parts of this
family by using `setElectronReco(efffn, smearfn, idclass)` etc.,
`setJetReco(smearfn, btagefffn, jetname)`, and
`setMETReco(smearparamsfn)` methods.

It can be useful to temporarily disable the smearing so you can see
how big an effect it has: to do this and return to perfect-accuracy
truth objects, set the environment variable `RIVET_DISABLE_SMEARING=1`
at runtime.


### Overlap removal

A *basic* overlap-removal (i.e. elimination of double-counting between physics
objects by geometric overlaps) can be run by calling `doSimpleOverlapRemoval()`.
This will first remove jets and b-jets very close to direct leptons, then remove
photons too close to electrons, and finally remove any leptons in the outer parts
of remaining jets. If your analysis has a more complex overlap-removal procedure,
you are still advised to implement this explicitly as it is a key part of the
analysis definition and differences in performance can be significant.


### Overriding object definitions

Finally, note that you are not entirely limited to `SimpleAnalysis`'
default set of physics-object definitions: if you want to override our
electron definition, for example, but keep the rest the same, you can
simply register a projection as usual in the `init()` method and this
will be used instead. For example, to override the tight muon
definition, you could write

```c++
LeptonFinder mymuons(Cuts::abspid == PID::MUON, 0.1);
declare("MuonsPromptMedium", SmearedParticles(mymuons,
    [](const Particle& mu) {
      if (mu.pT() < 7*GeV) return 0.0;
      return 1.0 - exp(-mu.pT()/GeV/5.0);
    }));
```

to use your very own custom definition (or, more simply, use different
standard smearing or efficiency functions from the
`include/Rivet/Tools/*SmearingFunctions.hh` header files). This definition
replaces the default projection for prompt muons at the Medium working
point: substitute `Muons` for `Electrons`, `Photons` or `Taus` for use
with those particles, substitute `Prompt` for `All` to override the
definition without a directness requirement (not from hadron-decays),
and substitute `Medium` for `Loose` or `Tight` to override the other ID
classes.

As this override mechanisms suggests, there is a sliding spectrum
between "blind" use of `SimpleAnalysis` defaults and the full-control
approach of Rivet's historic `Analysis` API: you don't have to
entirely choose one or the other. We hope you find a place on this
spectrum that works for you!
