/// Visual style
/// [parts and states](https://learn.microsoft.com/en-us/windows/win32/controls/parts-and-states).
/// These are originally two separated constant types, but since they form a
/// hierarchy, one single constant type can represent both part and state.
pub struct VS {
	pub(crate) part: i32,
	pub(crate) state: i32,
}

impl std::fmt::Display for VS {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		write!(f, "{}, {}", self.part, self.state)
	}
}
impl std::fmt::Debug for VS {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		write!(f, "Part {}, state {}", self.part, self.state)
	}
}

impl VS {
	/// Creates a new part + style value.
	pub const fn new(part: i32, state: i32) -> VS {
		Self { part, state }
	}

	/// Returns the part of the value.
	pub const fn part(&self) -> i32 {
		self.part
	}

	/// Returns the state of the value.
	pub const fn state(&self) -> i32 {
		self.state
	}
}

macro_rules! impl_pub_const_vs {
	( $( $pubvalname:ident $part:expr, $state:expr )* ) => {
		impl VS {
			$( pub const $pubvalname: Self = Self::new($part, $state); )*
		}
	};
}

impl_pub_const_vs! {
	AEROWIZARD_TITLEBAR_S_TITLEBAR_ACTIVE 1, 1
	AEROWIZARD_TITLEBAR_S_TITLEBAR_INACTIVE 1, 2

	AEROWIZARD_HEADERAREA_S_HEADERAREA_NOMARGIN 2, 1

	AEROWIZARD_CONTENTAREA_S_CONTENTAREA_NOMARGIN 3, 1

	AEROWIZARD_COMMANDAREA 4, 0

	AEROWIZARD_BUTTON 5, 0
}

impl_pub_const_vs! {
	BUTTON_PUSHBUTTON_NORMAL 1, 1
	BUTTON_PUSHBUTTON_HOT 1, 2
	BUTTON_PUSHBUTTON_PRESSED 1, 3
	BUTTON_PUSHBUTTON_DISABLED 1, 4
	BUTTON_PUSHBUTTON_DEFAULTED 1, 5
	BUTTON_PUSHBUTTON_DEFAULTED_ANIMATING 1, 6

	BUTTON_RADIOBUTTON_UNCHECKEDNORMAL 2, 1
	BUTTON_RADIOBUTTON_UNCHECKEDHOT 2, 2
	BUTTON_RADIOBUTTON_UNCHECKEDPRESSED 2, 3
	BUTTON_RADIOBUTTON_UNCHECKEDDISABLED 2, 4
	BUTTON_RADIOBUTTON_CHECKEDNORMAL 2, 5
	BUTTON_RADIOBUTTON_CHECKEDHOT 2, 6
	BUTTON_RADIOBUTTON_CHECKEDPRESSED 2, 7
	BUTTON_RADIOBUTTON_CHECKEDDISABLED 2, 8

	BUTTON_CHECKBOX_UNCHECKEDNORMAL 3, 1
	BUTTON_CHECKBOX_UNCHECKEDHOT 3, 2
	BUTTON_CHECKBOX_UNCHECKEDPRESSED 3, 3
	BUTTON_CHECKBOX_UNCHECKEDDISABLED 3, 4
	BUTTON_CHECKBOX_CHECKEDNORMAL 3, 5
	BUTTON_CHECKBOX_CHECKEDHOT 3, 6
	BUTTON_CHECKBOX_CHECKEDPRESSED 3, 7
	BUTTON_CHECKBOX_CHECKEDDISABLED 3, 8
	BUTTON_CHECKBOX_MIXEDNORMAL 3, 9
	BUTTON_CHECKBOX_MIXEDHOT 3, 10
	BUTTON_CHECKBOX_MIXEDPRESSED 3, 11
	BUTTON_CHECKBOX_MIXEDDISABLED 3, 12
	BUTTON_CHECKBOX_IMPLICITNORMAL 3, 13
	BUTTON_CHECKBOX_IMPLICITHOT 3, 14
	BUTTON_CHECKBOX_IMPLICITPRESSED 3, 15
	BUTTON_CHECKBOX_IMPLICITDISABLED 3, 16
	BUTTON_CHECKBOX_EXCLUDEDNORMAL 3, 17
	BUTTON_CHECKBOX_EXCLUDEDHOT 3, 18
	BUTTON_CHECKBOX_EXCLUDEDPRESSED 3, 19
	BUTTON_CHECKBOX_EXCLUDEDDISABLED 3, 20

	BUTTON_GROUPBOX_NORMAL 4, 1
	BUTTON_GROUPBOX_DISABLED 4, 2

	BUTTON_USERBUTTON 5, 0

	BUTTON_COMMANDLINK_NORMAL 6, 1
	BUTTON_COMMANDLINK_HOT 6, 2
	BUTTON_COMMANDLINK_PRESSED 6, 3
	BUTTON_COMMANDLINK_DISABLED 6, 4
	BUTTON_COMMANDLINK_DEFAULTED 6, 5
	BUTTON_COMMANDLINK_DEFAULTED_ANIMATING 6, 6

	BUTTON_COMMANDLINKGLYPH_NORMAL 7, 1
	BUTTON_COMMANDLINKGLYPH_HOT 7, 2
	BUTTON_COMMANDLINKGLYPH_PRESSED 7, 3
	BUTTON_COMMANDLINKGLYPH_DISABLED 7, 4
	BUTTON_COMMANDLINKGLYPH_DEFAULTED 7, 5

	BUTTON_RADIOBUTTON_HCDISABLED 8, 0

	BUTTON_CHECKBOX_HCDISABLED 9, 0

	BUTTON_GROUPBOX_HCDISABLED 10, 0

	BUTTON_PUSHBUTTONDROPDOWN_NORMAL 11, 1
	BUTTON_PUSHBUTTONDROPDOWN_DISABLED 11, 2
}

impl_pub_const_vs! {
	COMBOBOX_DROPDOWNBUTTON_NORMAL 1, 1
	COMBOBOX_DROPDOWNBUTTON_HOT 1, 2
	COMBOBOX_DROPDOWNBUTTON_PRESSED 1, 3
	COMBOBOX_DROPDOWNBUTTON_DISABLED 1, 4

	COMBOBOX_BACKGROUND 2, 0

	COMBOBOX_TRANSPARENTBACKGROUND_NORMAL 3, 1
	COMBOBOX_TRANSPARENTBACKGROUND_HOT 3, 2
	COMBOBOX_TRANSPARENTBACKGROUND_DISABLED 3, 3
	COMBOBOX_TRANSPARENTBACKGROUND_FOCUSED 3, 4

	COMBOBOX_BORDER_NORMAL 4, 1
	COMBOBOX_BORDER_HOT 4, 2
	COMBOBOX_BORDER_FOCUSED 4, 3
	COMBOBOX_BORDER_DISABLED 4, 4

	COMBOBOX_READONLY_NORMAL 5, 1
	COMBOBOX_READONLY_HOT 5, 2
	COMBOBOX_READONLY_PRESSED 5, 3
	COMBOBOX_READONLY_DISABLED 5, 4

	COMBOBOX_DROPDOWNBUTTONRIGHT_NORMAL 6, 1
	COMBOBOX_DROPDOWNBUTTONRIGHT_HOT 6, 2
	COMBOBOX_DROPDOWNBUTTONRIGHT_PRESSED 6, 3
	COMBOBOX_DROPDOWNBUTTONRIGHT_DISABLED 6, 4

	COMBOBOX_DROPDOWNBUTTONLEFT_NORMAL 7, 1
	COMBOBOX_DROPDOWNBUTTONLEFT_HOT 7, 2
	COMBOBOX_DROPDOWNBUTTONLEFT_PRESSED 7, 3
	COMBOBOX_DROPDOWNBUTTONLEFT_DISABLED 7, 4

	COMBOBOX_CUEBANNER_NORMAL 8, 1
	COMBOBOX_CUEBANNER_HOT 8, 2
	COMBOBOX_CUEBANNER_PRESSED 8, 3
	COMBOBOX_CUEBANNER_DISABLED 8, 4

	COMBOBOX_DROPDOWNITEM_NORMAL 9, 1
	COMBOBOX_DROPDOWNITEM_HIGHLIGHTED 9, 2
}

impl_pub_const_vs! {
	COMMUNICATIONS_TAB_NORMAL 1, 1
	COMMUNICATIONS_TAB_HOT 1, 2
	COMMUNICATIONS_TAB_SELECTED 1, 3
}

impl_pub_const_vs! {
	CONTROLPANEL_NAVIGATIONPANE 1, 0

	CONTROLPANEL_CONTENTPANE 2, 0

	CONTROLPANEL_NAVIGATIONPANELABEL 3, 0

	CONTROLPANEL_CONTENTPANELABEL 4, 0

	CONTROLPANEL_TITLE 5, 0

	CONTROLPANEL_BODYTEXT 6, 0

	CONTROLPANEL_HELPLINK_NORMAL 7, 1
	CONTROLPANEL_HELPLINK_HOT 7, 2
	CONTROLPANEL_HELPLINK_PRESSED 7, 3
	CONTROLPANEL_HELPLINK_DISABLED 7, 4

	CONTROLPANEL_TASKLINK_NORMAL 8, 1
	CONTROLPANEL_TASKLINK_HOT 8, 2
	CONTROLPANEL_TASKLINK_PRESSED 8, 3
	CONTROLPANEL_TASKLINK_DISABLED 8, 4
	CONTROLPANEL_TASKLINK_PAGE 8, 5

	CONTROLPANEL_GROUPTEXT 9, 0

	CONTROLPANEL_CONTENTLINK_NORMAL 10, 1
	CONTROLPANEL_CONTENTLINK_HOT 10, 2
	CONTROLPANEL_CONTENTLINK_PRESSED 10, 3
	CONTROLPANEL_CONTENTLINK_DISABLED 10, 4

	CONTROLPANEL_SECTIONTITLELINK_NORMAL 11, 1
	CONTROLPANEL_SECTIONTITLELINK_HOT 11, 2

	CONTROLPANEL_LARGECOMMANDAREA 12, 0

	CONTROLPANEL_SMALLCOMMANDAREA 13, 0

	CONTROLPANEL_BUTTON 14, 0

	CONTROLPANEL_MESSAGETEXT 15, 0

	CONTROLPANEL_NAVIGATIONPANELINE 16, 0

	CONTROLPANEL_CONTENTPANELINE 17, 0

	CONTROLPANEL_BANNERAREA 18, 0

	CONTROLPANEL_BODYTITLE 19, 0
}

impl_pub_const_vs! {
	DATEPICKER_DATETEXT_NORMAL 1, 1
	DATEPICKER_DATETEXT_DISABLED 1, 2
	DATEPICKER_DATETEXT_SELECTED 1, 3

	DATEPICKER_DATEBORDER_NORMAL 2, 1
	DATEPICKER_DATEBORDER_HOT 2, 2
	DATEPICKER_DATEBORDER_FOCUSED 2, 3
	DATEPICKER_DATEBORDER_DISABLED 2, 4

	DATEPICKER_SHOWCALENDARBUTTONRIGHT_NORMAL 3, 1
	DATEPICKER_SHOWCALENDARBUTTONRIGHT_HOT 3, 2
	DATEPICKER_SHOWCALENDARBUTTONRIGHT_PRESSED 3, 3
	DATEPICKER_SHOWCALENDARBUTTONRIGHT_DISABLED 3, 4
}

impl_pub_const_vs! {
	DRAGDROP_COPY_HIGHLIGHT 1, 1
	DRAGDROP_COPY_NOHIGHLIGHT 1, 2

	DRAGDROP_MOVE_HIGHLIGHT 2, 1
	DRAGDROP_MOVE_NOHIGHLIGHT 2, 2

	DRAGDROP_UPDATEMETADATA_HIGHLIGHT 3, 1
	DRAGDROP_UPDATEMETADATA_NOHIGHLIGHT 3, 2

	DRAGDROP_CREATELINK_HIGHLIGHT 4, 1
	DRAGDROP_CREATELINK_NOHIGHLIGHT 4, 2

	DRAGDROP_WARNING_HIGHLIGHT 5, 1
	DRAGDROP_WARNING_NOHIGHLIGHT 5, 2

	DRAGDROP_NONE_HIGHLIGHT 6, 1
	DRAGDROP_NONE_NOHIGHLIGHT 6, 2

	DRAGDROP_IMAGEBG 7, 0

	DRAGDROP_TEXTBG 8, 0
}

impl_pub_const_vs! {
	EDIT_EDITTEXT_NORMAL 1, 1
	EDIT_EDITTEXT_HOT 1, 2
	EDIT_EDITTEXT_SELECTED 1, 3
	EDIT_EDITTEXT_DISABLED 1, 4
	EDIT_EDITTEXT_FOCUSED 1, 5
	EDIT_EDITTEXT_READONLY 1, 6
	EDIT_EDITTEXT_ASSIST 1, 7
	EDIT_EDITTEXT_CUEBANNER 1, 8

	EDIT_CARET 2, 0

	EDIT_BACKGROUND_NORMAL 3, 1
	EDIT_BACKGROUND_HOT 3, 2
	EDIT_BACKGROUND_DISABLED 3, 3
	EDIT_BACKGROUND_FOCUSED 3, 4
	EDIT_BACKGROUND_READONLY 3, 5
	EDIT_BACKGROUND_ASSIST 3, 6

	EDIT_PASSWORD 4, 0

	EDIT_BACKGROUNDWITHBORDER_NORMAL 5, 1
	EDIT_BACKGROUNDWITHBORDER_HOT 5, 2
	EDIT_BACKGROUNDWITHBORDER_DISABLED 5, 3
	EDIT_BACKGROUNDWITHBORDER_FOCUSED 5, 4

	EDIT_EDITBORDER_NOSCROLL_NORMAL 6, 1
	EDIT_EDITBORDER_NOSCROLL_HOT 6, 2
	EDIT_EDITBORDER_NOSCROLL_FOCUSED 6, 3
	EDIT_EDITBORDER_NOSCROLL_DISABLED 6, 4

	EDIT_EDITBORDER_HSCROLL_NORMAL 7, 1
	EDIT_EDITBORDER_HSCROLL_HOT 7, 2
	EDIT_EDITBORDER_HSCROLL_FOCUSED 7, 3
	EDIT_EDITBORDER_HSCROLL_DISABLED 7, 4

	EDIT_EDITBORDER_VSCROLL_NORMAL 8, 1
	EDIT_EDITBORDER_VSCROLL_HOT 8, 2
	EDIT_EDITBORDER_VSCROLL_FOCUSED 8, 3
	EDIT_EDITBORDER_VSCROLL_DISABLED 8, 4

	EDIT_EDITBORDER_HVSCROLL_NORMAL 9, 1
	EDIT_EDITBORDER_HVSCROLL_HOT 9, 2
	EDIT_EDITBORDER_HVSCROLL_FOCUSED 9, 3
	EDIT_EDITBORDER_HVSCROLL_DISABLED 9, 4
}

impl_pub_const_vs! {
	EXPLORERBAR_HEADERBACKGROUND 1, 0

	EXPLORERBAR_HEADERCLOSE_NORMAL 2, 1
	EXPLORERBAR_HEADERCLOSE_HOT 2, 2
	EXPLORERBAR_HEADERCLOSE_PRESSED 2, 3

	EXPLORERBAR_HEADERPIN_NORMAL 3, 1
	EXPLORERBAR_HEADERPIN_HOT 3, 2
	EXPLORERBAR_HEADERPIN_PRESSED 3, 3
	EXPLORERBAR_HEADERPIN_SELECTEDNORMAL 3, 4
	EXPLORERBAR_HEADERPIN_SELECTEDHOT 3, 5
	EXPLORERBAR_HEADERPIN_SELECTEDPRESSED 3, 6

	EXPLORERBAR_IEBARMENU_NORMAL 4, 1
	EXPLORERBAR_IEBARMENU_HOT 4, 2
	EXPLORERBAR_IEBARMENU_PRESSED 4, 3

	EXPLORERBAR_NORMALGROUPBACKGROUND 5, 0

	EXPLORERBAR_NORMALGROUPCOLLAPSE_NORMAL 6, 1
	EXPLORERBAR_NORMALGROUPCOLLAPSE_HOT 6, 2
	EXPLORERBAR_NORMALGROUPCOLLAPSE_PRESSED 6, 3

	EXPLORERBAR_NORMALGROUPEXPAND_NORMAL 7, 1
	EXPLORERBAR_NORMALGROUPEXPAND_HOT 7, 2
	EXPLORERBAR_NORMALGROUPEXPAND_PRESSED 7, 3

	EXPLORERBAR_NORMALGROUPHEAD 8, 0

	EXPLORERBAR_SPECIALGROUPBACKGROUND 9, 0

	EXPLORERBAR_SPECIALGROUPCOLLAPSE_NORMAL 10, 1
	EXPLORERBAR_SPECIALGROUPCOLLAPSE_HOT 10, 2
	EXPLORERBAR_SPECIALGROUPCOLLAPSE_PRESSED 10, 3

	EXPLORERBAR_SPECIALGROUPEXPAND_NORMAL 11, 1
	EXPLORERBAR_SPECIALGROUPEXPAND_HOT 11, 2
	EXPLORERBAR_SPECIALGROUPEXPAND_PRESSED 11, 3

	EXPLORERBAR_SPECIALGROUPHEAD 12, 0
}

impl_pub_const_vs! {
	FLYOUT_HEADER 1, 0

	FLYOUT_BODY_NORMAL 2, 1
	FLYOUT_BODY_EMPHASIZED 2, 2

	FLYOUT_LABEL_NORMAL 3, 1
	FLYOUT_LABEL_SELECTED 3, 2
	FLYOUT_LABEL_EMPHASIZED 3, 3
	FLYOUT_LABEL_DISABLED 3, 4

	FLYOUT_LINK_NORMAL 4, 1
	FLYOUT_LINK_HOVER 4, 2

	FLYOUT_DIVIDER 5, 0

	FLYOUT_WINDOW 6, 0

	FLYOUT_LINKAREA 7, 0

	FLYOUT_LINKHEADER_NORMAL 8, 1
	FLYOUT_LINKHEADER_HOVER 8, 2
}

impl_pub_const_vs! {
	HEADER_HEADERITEM_NORMAL 1, 1
	HEADER_HEADERITEM_HOT 1, 2
	HEADER_HEADERITEM_PRESSED 1, 3
	HEADER_HEADERITEM_SORTEDNORMAL 1, 4
	HEADER_HEADERITEM_SORTEDHOT 1, 5
	HEADER_HEADERITEM_SORTEDPRESSED 1, 6
	HEADER_HEADERITEM_ICONNORMAL 1, 7
	HEADER_HEADERITEM_ICONHOT 1, 8
	HEADER_HEADERITEM_ICONPRESSED 1, 9
	HEADER_HEADERITEM_ICONSORTEDNORMAL 1, 10
	HEADER_HEADERITEM_ICONSORTEDHOT 1, 11
	HEADER_HEADERITEM_ICONSORTEDPRESSED 1, 12

	HEADER_HEADERITEMLEFT_NORMAL 2, 1
	HEADER_HEADERITEMLEFT_HOT 2, 2
	HEADER_HEADERITEMLEFT_PRESSED 2, 3

	HEADER_HEADERITEMRIGHT_NORMAL 3, 1
	HEADER_HEADERITEMRIGHT_HOT 3, 2
	HEADER_HEADERITEMRIGHT_PRESSED 3, 3

	HEADER_HEADERSORTARROW_SORTEDUP 4, 1
	HEADER_HEADERSORTARROW_SORTEDDOWN 4, 2

	HEADER_HEADERDROPDOWN_NORMAL 5, 1
	HEADER_HEADERDROPDOWN_SOFTHOT 5, 2
	HEADER_HEADERDROPDOWN_HOT 5, 3

	HEADER_HEADERDROPDOWNFILTER_NORMAL 6, 1
	HEADER_HEADERDROPDOWNFILTER_SOFTHOT 6, 2
	HEADER_HEADERDROPDOWNFILTER_HOT 6, 3

	HEADER_HEADEROVERFLOW_NORMAL 7, 1
	HEADER_HEADEROVERFLOW_HOT 7, 2
}

impl_pub_const_vs! {
	LISTBOX_BORDER_HSCROLL_NORMAL 1, 1
	LISTBOX_BORDER_HSCROLL_FOCUSED 1, 2
	LISTBOX_BORDER_HSCROLL_HOT 1, 3
	LISTBOX_BORDER_HSCROLL_DISABLED 1, 4

	LISTBOX_BORDER_HVSCROLL_NORMAL 2, 1
	LISTBOX_BORDER_HVSCROLL_FOCUSED 2, 2
	LISTBOX_BORDER_HVSCROLL_HOT 2, 3
	LISTBOX_BORDER_HVSCROLL_DISABLED 2, 4

	LISTBOX_BORDER_NOSCROLL_NORMAL 3, 1
	LISTBOX_BORDER_NOSCROLL_FOCUSED 3, 2
	LISTBOX_BORDER_NOSCROLL_HOT 3, 3
	LISTBOX_BORDER_NOSCROLL_DISABLED 3, 4

	LISTBOX_BORDER_VSCROLL_NORMAL 4, 1
	LISTBOX_BORDER_VSCROLL_FOCUSED 4, 2
	LISTBOX_BORDER_VSCROLL_HOT 4, 3
	LISTBOX_BORDER_VSCROLL_DISABLED 4, 4

	LISTBOX_ITEM_HOT 5, 1
	LISTBOX_ITEM_HOTSELECTED 5, 2
	LISTBOX_ITEM_SELECTED 5, 3
	LISTBOX_ITEM_SELECTEDNOTFOCUS 5, 4
}

impl_pub_const_vs! {
	LISTVIEW_LISTITEM_NORMAL 1, 1
	LISTVIEW_LISTITEM_HOT 1, 2
	LISTVIEW_LISTITEM_SELECTED 1, 3
	LISTVIEW_LISTITEM_DISABLED 1, 4
	LISTVIEW_LISTITEM_SELECTEDNOTFOCUS 1, 5
	LISTVIEW_LISTITEM_HOTSELECTED 1, 6

	LISTVIEW_LISTGROUP 2, 0

	LISTVIEW_LISTDETAIL 3, 0

	LISTVIEW_LISTSORTEDDETAIL 4, 0

	LISTVIEW_EMPTYTEXT 5, 0

	LISTVIEW_GROUPHEADER_OPEN 6, 1
	LISTVIEW_GROUPHEADER_OPENHOT 6, 2
	LISTVIEW_GROUPHEADER_OPENSELECTED 6, 3
	LISTVIEW_GROUPHEADER_OPENSELECTEDHOT 6, 4
	LISTVIEW_GROUPHEADER_OPENSELECTEDNOTFOCUSED 6, 5
	LISTVIEW_GROUPHEADER_OPENSELECTEDNOTFOCUSEDHOT 6, 6
	LISTVIEW_GROUPHEADER_OPENMIXEDSELECTION 6, 7
	LISTVIEW_GROUPHEADER_OPENMIXEDSELECTIONHOT 6, 8
	LISTVIEW_GROUPHEADER_CLOSE 6, 9
	LISTVIEW_GROUPHEADER_CLOSEHOT 6, 10
	LISTVIEW_GROUPHEADER_CLOSESELECTED 6, 11
	LISTVIEW_GROUPHEADER_CLOSESELECTEDHOT 6, 12
	LISTVIEW_GROUPHEADER_CLOSESELECTEDNOTFOCUSED 6, 13
	LISTVIEW_GROUPHEADER_CLOSESELECTEDNOTFOCUSEDHOT 6, 14
	LISTVIEW_GROUPHEADER_CLOSEMIXEDSELECTION 6, 15
	LISTVIEW_GROUPHEADER_CLOSEMIXEDSELECTIONHOT 6, 16

	LISTVIEW_GROUPHEADERLINE_OPEN 7, 1
	LISTVIEW_GROUPHEADERLINE_OPENHOT 7, 2
	LISTVIEW_GROUPHEADERLINE_OPENSELECTED 7, 3
	LISTVIEW_GROUPHEADERLINE_OPENSELECTEDHOT 7, 4
	LISTVIEW_GROUPHEADERLINE_OPENSELECTEDNOTFOCUSED 7, 5
	LISTVIEW_GROUPHEADERLINE_OPENSELECTEDNOTFOCUSEDHOT 7, 6
	LISTVIEW_GROUPHEADERLINE_OPENMIXEDSELECTION 7, 7
	LISTVIEW_GROUPHEADERLINE_OPENMIXEDSELECTIONHOT 7, 8
	LISTVIEW_GROUPHEADERLINE_CLOSE 7, 9
	LISTVIEW_GROUPHEADERLINE_CLOSEHOT 7, 10
	LISTVIEW_GROUPHEADERLINE_CLOSESELECTED 7, 11
	LISTVIEW_GROUPHEADERLINE_CLOSESELECTEDHOT 7, 12
	LISTVIEW_GROUPHEADERLINE_CLOSESELECTEDNOTFOCUSED 7, 13
	LISTVIEW_GROUPHEADERLINE_CLOSESELECTEDNOTFOCUSEDHOT 7, 14
	LISTVIEW_GROUPHEADERLINE_CLOSEMIXEDSELECTION 7, 15
	LISTVIEW_GROUPHEADERLINE_CLOSEMIXEDSELECTIONHOT 7, 16

	LISTVIEW_EXPANDBUTTON_NORMAL 8, 1
	LISTVIEW_EXPANDBUTTON_HOVER 8, 2
	LISTVIEW_EXPANDBUTTON_PUSHED 8, 3

	LISTVIEW_COLLAPSEBUTTON_NORMAL 9, 1
	LISTVIEW_COLLAPSEBUTTON_HOVER 9, 2
	LISTVIEW_COLLAPSEBUTTON_PUSHED 9, 3

	LISTVIEW_COLUMNDETAIL 10, 0
}

impl_pub_const_vs! {
	MENU_MENUITEM_TMSCHEMA 1, 0

	MENU_MENUDROPDOWN_TMSCHEMA 2, 0

	MENU_MENUBARITEM_TMSCHEMA 3, 0

	MENU_MENUBARDROPDOWN_TMSCHEMA 4, 0

	MENU_CHEVRON_TMSCHEMA 5, 0

	MENU_SEPARATOR_TMSCHEMA 6, 0

	MENU_BARBACKGROUND_ACTIVE 7, 1
	MENU_BARBACKGROUND_INACTIVE 7, 2

	MENU_BARITEM_NORMAL 8, 1
	MENU_BARITEM_HOT 8, 2
	MENU_BARITEM_PUSHED 8, 3
	MENU_BARITEM_DISABLED 8, 4
	MENU_BARITEM_DISABLEDHOT 8, 5
	MENU_BARITEM_DISABLEDPUSHED 8, 6

	MENU_POPUPBACKGROUND 9, 0

	MENU_POPUPBORDERS 10, 0

	MENU_POPUPCHECK_CHECKMARKNORMAL 11, 1
	MENU_POPUPCHECK_CHECKMARKDISABLED 11, 2
	MENU_POPUPCHECK_BULLETNORMAL 11, 3
	MENU_POPUPCHECK_BULLETDISABLED 11, 4

	MENU_POPUPCHECKBACKGROUND_DISABLED 12, 1
	MENU_POPUPCHECKBACKGROUND_NORMAL 12, 2
	MENU_POPUPCHECKBACKGROUND_BITMAP 12, 3

	MENU_POPUPGUTTER 13, 0

	MENU_POPUPITEM_NORMAL 14, 1
	MENU_POPUPITEM_HOT 14, 2
	MENU_POPUPITEM_DISABLED 14, 3
	MENU_POPUPITEM_DISABLEDHOT 14, 4

	MENU_POPUPSEPARATOR 15, 0

	MENU_POPUPSUBMENU_NORMAL 16, 1
	MENU_POPUPSUBMENU_DISABLED 16, 2

	MENU_SYSTEMCLOSE_NORMAL 17, 1
	MENU_SYSTEMCLOSE_DISABLED 17, 2

	MENU_SYSTEMMAXIMIZE_NORMAL 18, 1
	MENU_SYSTEMMAXIMIZE_DISABLED 18, 2

	MENU_SYSTEMMINIMIZE_NORMAL 19, 1
	MENU_SYSTEMMINIMIZE_DISABLED 19, 2

	MENU_SYSTEMRESTORE_NORMAL 20, 1
	MENU_SYSTEMRESTORE_DISABLED 20, 2
}

impl_pub_const_vs! {
	NAVIGATION_BACKBUTTON_NORMAL 1, 1
	NAVIGATION_BACKBUTTON_HOT 1, 2
	NAVIGATION_BACKBUTTON_PRESSED 1, 3
	NAVIGATION_BACKBUTTON_DISABLED 1, 4

	NAVIGATION_FORWARDBUTTON_NORMAL 2, 1
	NAVIGATION_FORWARDBUTTON_HOT 2, 2
	NAVIGATION_FORWARDBUTTON_PRESSED 2, 3
	NAVIGATION_FORWARDBUTTON_DISABLED 2, 4

	NAVIGATION_MENUBUTTON_NORMAL 3, 1
	NAVIGATION_MENUBUTTON_HOT 3, 2
	NAVIGATION_MENUBUTTON_PRESSED 3, 3
	NAVIGATION_MENUBUTTON_DISABLED 3, 4
}

impl_pub_const_vs! {
	PROGRESS_BAR 1, 0

	PROGRESS_BARVERT 2, 0

	PROGRESS_CHUNK 3, 0

	PROGRESS_CHUNKVERT 4, 0

	PROGRESS_FILL_NORMAL 5, 1
	PROGRESS_FILL_ERROR 5, 2
	PROGRESS_FILL_PAUSED 5, 3
	PROGRESS_FILL_PARTIAL 5, 4

	PROGRESS_FILLVERT_NORMAL 6, 1
	PROGRESS_FILLVERT_ERROR 6, 2
	PROGRESS_FILLVERT_PAUSED 6, 3
	PROGRESS_FILLVERT_PARTIAL 6, 4

	PROGRESS_PULSEOVERLAY 7, 0

	PROGRESS_MOVEOVERLAY 8, 0

	PROGRESS_PULSEOVERLAYVERT 9, 0

	PROGRESS_MOVEOVERLAYVERT 10, 0

	PROGRESS_TRANSPARENTBAR_NORMAL 11, 1
	PROGRESS_TRANSPARENTBAR_PARTIAL 11, 2

	PROGRESS_TRANSPARENTBARVERT_NORMAL 12, 1
	PROGRESS_TRANSPARENTBARVERT_PARTIAL 12, 2
}

impl_pub_const_vs! {
	REBAR_GRIPPER 1, 0

	REBAR_GRIPPERVERT 2, 0

	REBAR_BAND 3, 0

	REBAR_CHEVRON_NORMAL 4, 1
	REBAR_CHEVRON_HOT 4, 2
	REBAR_CHEVRON_PRESSED 4, 3

	REBAR_CHEVRONVERT_NORMAL 5, 1
	REBAR_CHEVRONVERT_HOT 5, 2
	REBAR_CHEVRONVERT_PRESSED 5, 3

	REBAR_BACKGROUND 6, 0

	REBAR_SPLITTER_NORMAL 7, 1
	REBAR_SPLITTER_HOT 7, 2
	REBAR_SPLITTER_PRESSED 7, 3

	REBAR_SPLITTERVERT_NORMAL 8, 1
	REBAR_SPLITTERVERT_HOT 8, 2
	REBAR_SPLITTERVERT_PRESSED 8, 3
}

impl_pub_const_vs! {
	SCROLLBAR_ARROWBTN_UPNORMAL 1, 1
	SCROLLBAR_ARROWBTN_UPHOT 1, 2
	SCROLLBAR_ARROWBTN_UPPRESSED 1, 3
	SCROLLBAR_ARROWBTN_UPDISABLED 1, 4
	SCROLLBAR_ARROWBTN_DOWNNORMAL 1, 5
	SCROLLBAR_ARROWBTN_DOWNHOT 1, 6
	SCROLLBAR_ARROWBTN_DOWNPRESSED 1, 7
	SCROLLBAR_ARROWBTN_DOWNDISABLED 1, 8
	SCROLLBAR_ARROWBTN_LEFTNORMAL 1, 9
	SCROLLBAR_ARROWBTN_LEFTHOT 1, 10
	SCROLLBAR_ARROWBTN_LEFTPRESSED 1, 11
	SCROLLBAR_ARROWBTN_LEFTDISABLED 1, 12
	SCROLLBAR_ARROWBTN_RIGHTNORMAL 1, 13
	SCROLLBAR_ARROWBTN_RIGHTHOT 1, 14
	SCROLLBAR_ARROWBTN_RIGHTPRESSED 1, 15
	SCROLLBAR_ARROWBTN_RIGHTDISABLED 1, 16
	SCROLLBAR_ARROWBTN_UPHOVER 1, 17
	SCROLLBAR_ARROWBTN_DOWNHOVER 1, 18
	SCROLLBAR_ARROWBTN_LEFTHOVER 1, 19
	SCROLLBAR_ARROWBTN_RIGHTHOVER 1, 20

	SCROLLBAR_THUMBBTNHORZ_NORMAL 2, 1
	SCROLLBAR_THUMBBTNHORZ_HOT 2, 2
	SCROLLBAR_THUMBBTNHORZ_PRESSED 2, 3
	SCROLLBAR_THUMBBTNHORZ_DISABLED 2, 4
	SCROLLBAR_THUMBBTNHORZ_HOVER 2, 5

	SCROLLBAR_THUMBBTNVERT_NORMAL 3, 1
	SCROLLBAR_THUMBBTNVERT_HOT 3, 2
	SCROLLBAR_THUMBBTNVERT_PRESSED 3, 3
	SCROLLBAR_THUMBBTNVERT_DISABLED 3, 4
	SCROLLBAR_THUMBBTNVERT_HOVER 3, 5

	SCROLLBAR_LOWERTRACKHORZ_NORMAL 4, 1
	SCROLLBAR_LOWERTRACKHORZ_HOT 4, 2
	SCROLLBAR_LOWERTRACKHORZ_PRESSED 4, 3
	SCROLLBAR_LOWERTRACKHORZ_DISABLED 4, 4
	SCROLLBAR_LOWERTRACKHORZ_HOVER 4, 5

	SCROLLBAR_UPPERTRACKHORZ_NORMAL 5, 1
	SCROLLBAR_UPPERTRACKHORZ_HOT 5, 2
	SCROLLBAR_UPPERTRACKHORZ_PRESSED 5, 3
	SCROLLBAR_UPPERTRACKHORZ_DISABLED 5, 4
	SCROLLBAR_UPPERTRACKHORZ_HOVER 5, 5

	SCROLLBAR_LOWERTRACKVERT_NORMAL 6, 1
	SCROLLBAR_LOWERTRACKVERT_HOT 6, 2
	SCROLLBAR_LOWERTRACKVERT_PRESSED 6, 3
	SCROLLBAR_LOWERTRACKVERT_DISABLED 6, 4
	SCROLLBAR_LOWERTRACKVERT_HOVER 6, 5

	SCROLLBAR_UPPERTRACKVERT_NORMAL 7, 1
	SCROLLBAR_UPPERTRACKVERT_HOT 7, 2
	SCROLLBAR_UPPERTRACKVERT_PRESSED 7, 3
	SCROLLBAR_UPPERTRACKVERT_DISABLED 7, 4
	SCROLLBAR_UPPERTRACKVERT_HOVER 7, 5

	SCROLLBAR_GRIPPERHORZ_NORMAL 8, 1
	SCROLLBAR_GRIPPERHORZ_HOT 8, 2
	SCROLLBAR_GRIPPERHORZ_PRESSED 8, 3
	SCROLLBAR_GRIPPERHORZ_DISABLED 8, 4
	SCROLLBAR_GRIPPERHORZ_HOVER 8, 5

	SCROLLBAR_GRIPPERVERT_NORMAL 9, 1
	SCROLLBAR_GRIPPERVERT_HOT 9, 2
	SCROLLBAR_GRIPPERVERT_PRESSED 9, 3
	SCROLLBAR_GRIPPERVERT_DISABLED 9, 4
	SCROLLBAR_GRIPPERVERT_HOVER 9, 5

	SCROLLBAR_SIZEBOX_RIGHTALIGN 10, 1
	SCROLLBAR_SIZEBOX_LEFTALIGN 10, 2
	SCROLLBAR_SIZEBOX_TOPRIGHTALIGN 10, 3
	SCROLLBAR_SIZEBOX_TOPLEFTALIGN 10, 4
	SCROLLBAR_SIZEBOX_HALFBOTTOMRIGHTALIGN 10, 5
	SCROLLBAR_SIZEBOX_HALFBOTTOMLEFTALIGN 10, 6
	SCROLLBAR_SIZEBOX_HALFTOPRIGHTALIGN 10, 7
	SCROLLBAR_SIZEBOX_HALFTOPLEFTALIGN 10, 8

	SCROLLBAR_SIZEBOXBKGND_NORMAL 11, 1
	SCROLLBAR_SIZEBOXBKGND_HOT 11, 2
	SCROLLBAR_SIZEBOXBKGND_PRESSED 11, 3
	SCROLLBAR_SIZEBOXBKGND_DISABLED 11, 4
	SCROLLBAR_SIZEBOXBKGND_HOVER 11, 5
}

impl_pub_const_vs! {
	SPIN_UP_NORMAL 1, 1
	SPIN_UP_HOT 1, 2
	SPIN_UP_PRESSED 1, 3
	SPIN_UP_DISABLED 1, 4

	SPIN_DOWN_NORMAL 2, 1
	SPIN_DOWN_HOT 2, 2
	SPIN_DOWN_PRESSED 2, 3
	SPIN_DOWN_DISABLED 2, 4

	SPIN_UPHORZ_NORMAL 3, 1
	SPIN_UPHORZ_HOT 3, 2
	SPIN_UPHORZ_PRESSED 3, 3
	SPIN_UPHORZ_DISABLED 3, 4

	SPIN_DOWNHORZ_NORMAL 4, 1
	SPIN_DOWNHORZ_HOT 4, 2
	SPIN_DOWNHORZ_PRESSED 4, 3
	SPIN_DOWNHORZ_DISABLED 4, 4
}

impl_pub_const_vs! {
	STATUS_PANE 1, 0

	STATUS_GRIPPERPANE 2, 0

	STATUS_GRIPPER 3, 0
}

impl_pub_const_vs! {
	TAB_TABITEM_NORMAL 1, 1
	TAB_TABITEM_HOT 1, 2
	TAB_TABITEM_SELECTED 1, 3
	TAB_TABITEM_DISABLED 1, 4
	TAB_TABITEM_FOCUSED 1, 5

	TAB_TABITEMLEFTEDGE_NORMAL 2, 1
	TAB_TABITEMLEFTEDGE_HOT 2, 2
	TAB_TABITEMLEFTEDGE_SELECTED 2, 3
	TAB_TABITEMLEFTEDGE_DISABLED 2, 4
	TAB_TABITEMLEFTEDGE_FOCUSED 2, 5

	TAB_TABITEMRIGHTEDGE_NORMAL 3, 1
	TAB_TABITEMRIGHTEDGE_HOT 3, 2
	TAB_TABITEMRIGHTEDGE_SELECTED 3, 3
	TAB_TABITEMRIGHTEDGE_DISABLED 3, 4
	TAB_TABITEMRIGHTEDGE_FOCUSED 3, 5

	TAB_TABITEMBOTHEDGE_NORMAL 4, 1
	TAB_TABITEMBOTHEDGE_HOT 4, 2
	TAB_TABITEMBOTHEDGE_SELECTED 4, 3
	TAB_TABITEMBOTHEDGE_DISABLED 4, 4
	TAB_TABITEMBOTHEDGE_FOCUSED 4, 5

	TAB_TOPTABITEM_NORMAL 5, 1
	TAB_TOPTABITEM_HOT 5, 2
	TAB_TOPTABITEM_SELECTED 5, 3
	TAB_TOPTABITEM_DISABLED 5, 4
	TAB_TOPTABITEM_FOCUSED 5, 5

	TAB_TOPTABITEMLEFTEDGE_NORMAL 6, 1
	TAB_TOPTABITEMLEFTEDGE_HOT 6, 2
	TAB_TOPTABITEMLEFTEDGE_SELECTED 6, 3
	TAB_TOPTABITEMLEFTEDGE_DISABLED 6, 4
	TAB_TOPTABITEMLEFTEDGE_FOCUSED 6, 5

	TAB_TOPTABITEMRIGHTEDGE_NORMAL 7, 1
	TAB_TOPTABITEMRIGHTEDGE_HOT 7, 2
	TAB_TOPTABITEMRIGHTEDGE_SELECTED 7, 3
	TAB_TOPTABITEMRIGHTEDGE_DISABLED 7, 4
	TAB_TOPTABITEMRIGHTEDGE_FOCUSED 7, 5

	TAB_TOPTABITEMBOTHEDGE_NORMAL 8, 1
	TAB_TOPTABITEMBOTHEDGE_HOT 8, 2
	TAB_TOPTABITEMBOTHEDGE_SELECTED 8, 3
	TAB_TOPTABITEMBOTHEDGE_DISABLED 8, 4
	TAB_TOPTABITEMBOTHEDGE_FOCUSED 8, 5

	TAB_PANE 9, 0

	TAB_BODY 10, 0

	TAB_AEROWIZARDBODY 11, 0
}

impl_pub_const_vs! {
	TASKDIALOG_PRIMARYPANEL 1, 0

	TASKDIALOG_MAININSTRUCTIONPANE 2, 0

	TASKDIALOG_MAINICON 3, 0

	TASKDIALOG_CONTENTPANE_STANDALONE 4, 1

	TASKDIALOG_CONTENTICON 5, 0

	TASKDIALOG_EXPANDEDCONTENT 6, 0

	TASKDIALOG_COMMANDLINKPANE 7, 0

	TASKDIALOG_SECONDARYPANEL 8, 0

	TASKDIALOG_CONTROLPANE 9, 0

	TASKDIALOG_BUTTONSECTION 10, 0

	TASKDIALOG_BUTTONWRAPPER 11, 0

	TASKDIALOG_EXPANDOTEXT 12, 0

	TASKDIALOG_EXPANDOBUTTON_NORMAL 13, 1
	TASKDIALOG_EXPANDOBUTTON_HOVER 13, 2
	TASKDIALOG_EXPANDOBUTTON_PRESSED 13, 3
	TASKDIALOG_EXPANDOBUTTON_EXPANDEDNORMAL 13, 4
	TASKDIALOG_EXPANDOBUTTON_EXPANDEDHOVER 13, 5
	TASKDIALOG_EXPANDOBUTTON_EXPANDEDPRESSED 13, 6
	TASKDIALOG_EXPANDOBUTTON_NORMALDISABLED 13, 7
	TASKDIALOG_EXPANDOBUTTON_EXPANDEDDISABLED 13, 8

	TASKDIALOG_VERIFICATIONTEXT 14, 0

	TASKDIALOG_FOOTNOTEPANE 15, 0

	TASKDIALOG_FOOTNOTEAREA 16, 0

	TASKDIALOG_FOOTNOTESEPARATOR 17, 0

	TASKDIALOG_EXPANDEDFOOTERAREA 18, 0

	TASKDIALOG_PROGRESSBAR 19, 0

	TASKDIALOG_IMAGEALIGNMENT 20, 0

	TASKDIALOG_RADIOBUTTONPANE 21, 0
}

impl_pub_const_vs! {
	TEXTSTYLE_MAININSTRUCTION 1, 0

	TEXTSTYLE_INSTRUCTION 2, 0

	TEXTSTYLE_BODYTITLE 3, 0

	TEXTSTYLE_BODYTEXT 4, 0

	TEXTSTYLE_SECONDARYTEXT 5, 0

	TEXTSTYLE_HYPERLINKTEXT_HYPERLINK_NORMAL 6, 1
	TEXTSTYLE_HYPERLINKTEXT_HYPERLINK_HOT 6, 2
	TEXTSTYLE_HYPERLINKTEXT_HYPERLINK_PRESSED 6, 3
	TEXTSTYLE_HYPERLINKTEXT_HYPERLINK_DISABLED 6, 4

	TEXTSTYLE_EXPANDED 7, 0

	TEXTSTYLE_LABEL 8, 0

	TEXTSTYLE_CONTROLLABEL_CONTROLLABEL_NORMAL 9, 1
	TEXTSTYLE_CONTROLLABEL_CONTROLLABEL_DISABLED 9, 2
}

impl_pub_const_vs! {
	TOOLBAR_BUTTON_NORMAL 1, 1
	TOOLBAR_BUTTON_HOT 1, 2
	TOOLBAR_BUTTON_PRESSED 1, 3
	TOOLBAR_BUTTON_DISABLED 1, 4
	TOOLBAR_BUTTON_CHECKED 1, 5
	TOOLBAR_BUTTON_HOTCHECKED 1, 6
	TOOLBAR_BUTTON_NEARHOT 1, 7
	TOOLBAR_BUTTON_OTHERSIDEHOT 1, 8

	TOOLBAR_DROPDOWNBUTTON_NORMAL 2, 1
	TOOLBAR_DROPDOWNBUTTON_HOT 2, 2
	TOOLBAR_DROPDOWNBUTTON_PRESSED 2, 3
	TOOLBAR_DROPDOWNBUTTON_DISABLED 2, 4
	TOOLBAR_DROPDOWNBUTTON_CHECKED 2, 5
	TOOLBAR_DROPDOWNBUTTON_HOTCHECKED 2, 6
	TOOLBAR_DROPDOWNBUTTON_NEARHOT 2, 7
	TOOLBAR_DROPDOWNBUTTON_OTHERSIDEHOT 2, 8

	TOOLBAR_SPLITBUTTON_NORMAL 3, 1
	TOOLBAR_SPLITBUTTON_HOT 3, 2
	TOOLBAR_SPLITBUTTON_PRESSED 3, 3
	TOOLBAR_SPLITBUTTON_DISABLED 3, 4
	TOOLBAR_SPLITBUTTON_CHECKED 3, 5
	TOOLBAR_SPLITBUTTON_HOTCHECKED 3, 6
	TOOLBAR_SPLITBUTTON_NEARHOT 3, 7
	TOOLBAR_SPLITBUTTON_OTHERSIDEHOT 3, 8

	TOOLBAR_SPLITBUTTONDROPDOWN_NORMAL 4, 1
	TOOLBAR_SPLITBUTTONDROPDOWN_HOT 4, 2
	TOOLBAR_SPLITBUTTONDROPDOWN_PRESSED 4, 3
	TOOLBAR_SPLITBUTTONDROPDOWN_DISABLED 4, 4
	TOOLBAR_SPLITBUTTONDROPDOWN_CHECKED 4, 5
	TOOLBAR_SPLITBUTTONDROPDOWN_HOTCHECKED 4, 6
	TOOLBAR_SPLITBUTTONDROPDOWN_NEARHOT 4, 7
	TOOLBAR_SPLITBUTTONDROPDOWN_OTHERSIDEHOT 4, 8

	TOOLBAR_SEPARATOR_NORMAL 5, 1
	TOOLBAR_SEPARATOR_HOT 5, 2
	TOOLBAR_SEPARATOR_PRESSED 5, 3
	TOOLBAR_SEPARATOR_DISABLED 5, 4
	TOOLBAR_SEPARATOR_CHECKED 5, 5
	TOOLBAR_SEPARATOR_HOTCHECKED 5, 6
	TOOLBAR_SEPARATOR_NEARHOT 5, 7
	TOOLBAR_SEPARATOR_OTHERSIDEHOT 5, 8

	TOOLBAR_SEPARATORVERT_NORMAL 6, 1
	TOOLBAR_SEPARATORVERT_HOT 6, 2
	TOOLBAR_SEPARATORVERT_PRESSED 6, 3
	TOOLBAR_SEPARATORVERT_DISABLED 6, 4
	TOOLBAR_SEPARATORVERT_CHECKED 6, 5
	TOOLBAR_SEPARATORVERT_HOTCHECKED 6, 6
	TOOLBAR_SEPARATORVERT_NEARHOT 6, 7
	TOOLBAR_SEPARATORVERT_OTHERSIDEHOT 6, 8

	TOOLBAR_DROPDOWNBUTTONGLYPH_NORMAL 7, 1
	TOOLBAR_DROPDOWNBUTTONGLYPH_HOT 7, 2
	TOOLBAR_DROPDOWNBUTTONGLYPH_PRESSED 7, 3
	TOOLBAR_DROPDOWNBUTTONGLYPH_DISABLED 7, 4
	TOOLBAR_DROPDOWNBUTTONGLYPH_CHECKED 7, 5
	TOOLBAR_DROPDOWNBUTTONGLYPH_HOTCHECKED 7, 6
	TOOLBAR_DROPDOWNBUTTONGLYPH_NEARHOT 7, 7
	TOOLBAR_DROPDOWNBUTTONGLYPH_OTHERSIDEHOT 7, 8
}

impl_pub_const_vs! {
	TOOLTIP_STANDARD_NORMAL 1, 1
	TOOLTIP_STANDARD_LINK 1, 2

	TOOLTIP_STANDARDTITLE 2, 0

	TOOLTIP_BALLOON_NORMAL 3, 1
	TOOLTIP_BALLOON_LINK 3, 2

	TOOLTIP_BALLOONTITLE 4, 0

	TOOLTIP_CLOSE_NORMAL 5, 1
	TOOLTIP_CLOSE_HOT 5, 2
	TOOLTIP_CLOSE_PRESSED 5, 3

	TOOLTIP_BALLOONSTEM_POINTINGUPLEFTWALL 6, 1
	TOOLTIP_BALLOONSTEM_POINTINGUPCENTERED 6, 2
	TOOLTIP_BALLOONSTEM_POINTINGUPRIGHTWALL 6, 3
	TOOLTIP_BALLOONSTEM_POINTINGDOWNRIGHTWALL 6, 4
	TOOLTIP_BALLOONSTEM_POINTINGDOWNCENTERED 6, 5
	TOOLTIP_BALLOONSTEM_POINTINGDOWNLEFTWALL 6, 6

	TOOLTIP_WRENCH_NORMAL 7, 1
	TOOLTIP_WRENCH_HOT 7, 2
	TOOLTIP_WRENCH_PRESSED 7, 3
}

impl_pub_const_vs! {
	TRACKBAR_TRACK_NORMAL 1, 1

	TRACKBAR_TRACKVERT_NORMAL 2, 1

	TRACKBAR_THUMB_NORMAL 3, 1
	TRACKBAR_THUMB_HOT 3, 2
	TRACKBAR_THUMB_PRESSED 3, 3
	TRACKBAR_THUMB_FOCUSED 3, 4
	TRACKBAR_THUMB_DISABLED 3, 5

	TRACKBAR_THUMBBOTTOM_NORMAL 4, 1
	TRACKBAR_THUMBBOTTOM_HOT 4, 2
	TRACKBAR_THUMBBOTTOM_PRESSED 4, 3
	TRACKBAR_THUMBBOTTOM_FOCUSED 4, 4
	TRACKBAR_THUMBBOTTOM_DISABLED 4, 5

	TRACKBAR_THUMBTOP_NORMAL 5, 1
	TRACKBAR_THUMBTOP_HOT 5, 2
	TRACKBAR_THUMBTOP_PRESSED 5, 3
	TRACKBAR_THUMBTOP_FOCUSED 5, 4
	TRACKBAR_THUMBTOP_DISABLED 5, 5

	TRACKBAR_THUMBVERT_NORMAL 6, 1
	TRACKBAR_THUMBVERT_HOT 6, 2
	TRACKBAR_THUMBVERT_PRESSED 6, 3
	TRACKBAR_THUMBVERT_FOCUSED 6, 4
	TRACKBAR_THUMBVERT_DISABLED 6, 5

	TRACKBAR_THUMBLEFT_NORMAL 7, 1
	TRACKBAR_THUMBLEFT_HOT 7, 2
	TRACKBAR_THUMBLEFT_PRESSED 7, 3
	TRACKBAR_THUMBLEFT_FOCUSED 7, 4
	TRACKBAR_THUMBLEFT_DISABLED 7, 5

	TRACKBAR_THUMBRIGHT_NORMAL 8, 1
	TRACKBAR_THUMBRIGHT_HOT 8, 2
	TRACKBAR_THUMBRIGHT_PRESSED 8, 3
	TRACKBAR_THUMBRIGHT_FOCUSED 8, 4
	TRACKBAR_THUMBRIGHT_DISABLED 8, 5

	TRACKBAR_TICS_NORMAL 9, 1

	TRACKBAR_TICSVERT_NORMAL 10, 1
}

impl_pub_const_vs! {
	TREEVIEW_TREEITEM_NORMAL 1, 1
	TREEVIEW_TREEITEM_HOT 1, 2
	TREEVIEW_TREEITEM_SELECTED 1, 3
	TREEVIEW_TREEITEM_DISABLED 1, 4
	TREEVIEW_TREEITEM_SELECTEDNOTFOCUS 1, 5
	TREEVIEW_TREEITEM_HOTSELECTED 1, 6

	TREEVIEW_GLYPH_CLOSED 2, 1
	TREEVIEW_GLYPH_OPENED 2, 2

	TREEVIEW_BRANCH 3, 0

	TREEVIEW_HOTGLYPH_CLOSED 4, 1
	TREEVIEW_HOTGLYPH_OPENED 4, 2
}

impl_pub_const_vs! {
	USERTILE_STROKEBACKGROUND 1, 0

	USERTILE_HOVERBACKGROUND_NORMAL 2, 1
	USERTILE_HOVERBACKGROUND_HOT 2, 2
	USERTILE_HOVERBACKGROUND_PRESSED 2, 3
}

impl_pub_const_vs! {
	TEXTSELECTIONGRIPPER_GRIPPER_NORMAL 1, 1
	TEXTSELECTIONGRIPPER_GRIPPER_CENTERED 1, 2
}

impl_pub_const_vs! {
	WINDOW_CAPTION_ACTIVE 1, 1
	WINDOW_CAPTION_INACTIVE 1, 2
	WINDOW_CAPTION_DISABLED 1, 3

	WINDOW_SMALLCAPTION_ACTIVE 2, 1
	WINDOW_SMALLCAPTION_INACTIVE 2, 2
	WINDOW_SMALLCAPTION_DISABLED 2, 3

	WINDOW_MINCAPTION_ACTIVE 3, 1
	WINDOW_MINCAPTION_INACTIVE 3, 2
	WINDOW_MINCAPTION_DISABLED 3, 3

	WINDOW_SMALLMINCAPTION 4, 0

	WINDOW_MAXCAPTION_ACTIVE 5, 1
	WINDOW_MAXCAPTION_INACTIVE 5, 2
	WINDOW_MAXCAPTION_DISABLED 5, 3

	WINDOW_SMALLMAXCAPTION 6, 0

	WINDOW_FRAMELEFT_ACTIVE 7, 1
	WINDOW_FRAMELEFT_INACTIVE 7, 2

	WINDOW_FRAMERIGHT_ACTIVE 8, 1
	WINDOW_FRAMERIGHT_INACTIVE 8, 2

	WINDOW_FRAMEBOTTOM_ACTIVE 9, 1
	WINDOW_FRAMEBOTTOM_INACTIVE 9, 2

	WINDOW_SMALLFRAMELEFT_ACTIVE 10, 1
	WINDOW_SMALLFRAMELEFT_INACTIVE 10, 2

	WINDOW_SMALLFRAMERIGHT_ACTIVE 11, 1
	WINDOW_SMALLFRAMERIGHT_INACTIVE 11, 2

	WINDOW_SMALLFRAMEBOTTOM_ACTIVE 12, 1
	WINDOW_SMALLFRAMEBOTTOM_INACTIVE 12, 2

	WINDOW_SYSBUTTON_NORMAL 13, 1
	WINDOW_SYSBUTTON_HOT 13, 2
	WINDOW_SYSBUTTON_PUSHED 13, 3
	WINDOW_SYSBUTTON_DISABLED 13, 4

	WINDOW_MDISYSBUTTON 14, 0

	WINDOW_MINBUTTON_NORMAL 15, 1
	WINDOW_MINBUTTON_HOT 15, 2
	WINDOW_MINBUTTON_PUSHED 15, 3
	WINDOW_MINBUTTON_DISABLED 15, 4

	WINDOW_MDIMINBUTTON_NORMAL 16, 1
	WINDOW_MDIMINBUTTON_HOT 16, 2
	WINDOW_MDIMINBUTTON_PUSHED 16, 3
	WINDOW_MDIMINBUTTON_DISABLED 16, 4

	WINDOW_MAXBUTTON_NORMAL 17, 1
	WINDOW_MAXBUTTON_HOT 17, 2
	WINDOW_MAXBUTTON_PUSHED 17, 3
	WINDOW_MAXBUTTON_DISABLED 17, 4

	WINDOW_CLOSEBUTTON_NORMAL 18, 1
	WINDOW_CLOSEBUTTON_HOT 18, 2
	WINDOW_CLOSEBUTTON_PUSHED 18, 3
	WINDOW_CLOSEBUTTON_DISABLED 18, 4

	WINDOW_SMALLCLOSEBUTTON_NORMAL 19, 1
	WINDOW_SMALLCLOSEBUTTON_HOT 19, 2
	WINDOW_SMALLCLOSEBUTTON_PUSHED 19, 3
	WINDOW_SMALLCLOSEBUTTON_DISABLED 19, 4

	WINDOW_MDICLOSEBUTTON_NORMAL 20, 1
	WINDOW_MDICLOSEBUTTON_HOT 20, 2
	WINDOW_MDICLOSEBUTTON_PUSHED 20, 3
	WINDOW_MDICLOSEBUTTON_DISABLED 20, 4

	WINDOW_RESTOREBUTTON_NORMAL 21, 1
	WINDOW_RESTOREBUTTON_HOT 21, 2
	WINDOW_RESTOREBUTTON_PUSHED 21, 3
	WINDOW_RESTOREBUTTON_DISABLED 21, 4

	WINDOW_MDIRESTOREBUTTON_NORMAL 22, 1
	WINDOW_MDIRESTOREBUTTON_HOT 22, 2
	WINDOW_MDIRESTOREBUTTON_PUSHED 22, 3
	WINDOW_MDIRESTOREBUTTON_DISABLED 22, 4

	WINDOW_HELPBUTTON_NORMAL 23, 1
	WINDOW_HELPBUTTON_HOT 23, 2
	WINDOW_HELPBUTTON_PUSHED 23, 3
	WINDOW_HELPBUTTON_DISABLED 23, 4

	WINDOW_MDIHELPBUTTON 24, 0

	WINDOW_HORZSCROLL_NORMAL 25, 1
	WINDOW_HORZSCROLL_HOT 25, 2
	WINDOW_HORZSCROLL_PUSHED 25, 3
	WINDOW_HORZSCROLL_DISABLED 25, 4

	WINDOW_HORZTHUMB_NORMAL 26, 1
	WINDOW_HORZTHUMB_HOT 26, 2
	WINDOW_HORZTHUMB_PUSHED 26, 3
	WINDOW_HORZTHUMB_DISABLED 26, 4

	WINDOW_VERTSCROLL_NORMAL 27, 1
	WINDOW_VERTSCROLL_HOT 27, 2
	WINDOW_VERTSCROLL_PUSHED 27, 3
	WINDOW_VERTSCROLL_DISABLED 27, 4

	WINDOW_VERTTHUMB_NORMAL 28, 1
	WINDOW_VERTTHUMB_HOT 28, 2
	WINDOW_VERTTHUMB_PUSHED 28, 3
	WINDOW_VERTTHUMB_DISABLED 28, 4

	WINDOW_DIALOG 29, 0

	WINDOW_CAPTIONSIZINGTEMPLATE 30, 0

	WINDOW_SMALLCAPTIONSIZINGTEMPLATE 31, 0

	WINDOW_FRAMELEFTSIZINGTEMPLATE 32, 0

	WINDOW_SMALLFRAMELEFTSIZINGTEMPLATE 33, 0

	WINDOW_FRAMERIGHTSIZINGTEMPLATE 34, 0

	WINDOW_SMALLFRAMERIGHTSIZINGTEMPLATE 35, 0

	WINDOW_FRAMEBOTTOMSIZINGTEMPLATE 36, 0

	WINDOW_SMALLFRAMEBOTTOMSIZINGTEMPLATE 37, 0

	WINDOW_FRAME_ACTIVE 38, 1
	WINDOW_FRAME_INACTIVE 38, 2

	WINDOW_BORDER 39, 0
}

impl_pub_const_vs! {
	LINK_HYPERLINK_NORMALTEXT 1, 1
	LINK_HYPERLINK_LINKTEXT 1, 2
}

impl_pub_const_vs! {
	EMPTYMARKUP_MARKUPTEXT_NORMALTEXT 1, 1
	EMPTYMARKUP_MARKUPTEXT_LINKTEXT 1, 2
}

impl_pub_const_vs! {
	STATIC_TEXT 1, 0
}

impl_pub_const_vs! {
	PAGE_UP 1, 0

	PAGE_DOWN 2, 0

	PAGE_UPHORZ 3, 0

	PAGE_DOWNHORZ 4, 0
}

impl_pub_const_vs! {
	MONTHCAL_BACKGROUND 1, 0

	MONTHCAL_BORDERS 2, 0

	MONTHCAL_GRIDBACKGROUND 3, 0

	MONTHCAL_COLHEADERSPLITTER 4, 0

	MONTHCAL_GRIDCELLBACKGROUND_SELECTED 5, 1
	MONTHCAL_GRIDCELLBACKGROUND_HOT 5, 2
	MONTHCAL_GRIDCELLBACKGROUND_SELECTEDHOT 5, 3
	MONTHCAL_GRIDCELLBACKGROUND_SELECTEDNOTFOCUSED 5, 4
	MONTHCAL_GRIDCELLBACKGROUND_TODAY 5, 5
	MONTHCAL_GRIDCELLBACKGROUND_TODAYSELECTED 5, 6

	MONTHCAL_GRIDCELL_HOT 6, 1
	MONTHCAL_GRIDCELL_HASSTATE 6, 2
	MONTHCAL_GRIDCELL_HASSTATEHOT 6, 3
	MONTHCAL_GRIDCELL_TODAY 6, 4
	MONTHCAL_GRIDCELL_TODAYSELECTED 6, 5
	MONTHCAL_GRIDCELL_SELECTED 6, 6
	MONTHCAL_GRIDCELL_SELECTEDHOT 6, 7

	MONTHCAL_GRIDCELLUPPER_HOT 7, 1
	MONTHCAL_GRIDCELLUPPER_HASSTATE 7, 2
	MONTHCAL_GRIDCELLUPPER_HASSTATEHOT 7, 3
	MONTHCAL_GRIDCELLUPPER_SELECTED 7, 4
	MONTHCAL_GRIDCELLUPPER_SELECTEDHOT 7, 5

	MONTHCAL_TRAILINGGRIDCELL_HOT 8, 1
	MONTHCAL_TRAILINGGRIDCELL_HASSTATE 8, 2
	MONTHCAL_TRAILINGGRIDCELL_HASSTATEHOT 8, 3
	MONTHCAL_TRAILINGGRIDCELL_TODAY 8, 4
	MONTHCAL_TRAILINGGRIDCELL_TODAYSELECTED 8, 5
	MONTHCAL_TRAILINGGRIDCELL_SELECTED 8, 6
	MONTHCAL_TRAILINGGRIDCELL_SELECTEDHOT 8, 7

	MONTHCAL_TRAILINGGRIDCELLUPPER_HOT 9, 1
	MONTHCAL_TRAILINGGRIDCELLUPPER_HASSTATE 9, 2
	MONTHCAL_TRAILINGGRIDCELLUPPER_HASSTATEHOT 9, 3
	MONTHCAL_TRAILINGGRIDCELLUPPER_SELECTED 9, 4
	MONTHCAL_TRAILINGGRIDCELLUPPER_SELECTEDHOT 9, 5

	MONTHCAL_NAVNEXT_NORMAL 10, 1
	MONTHCAL_NAVNEXT_HOT 10, 2
	MONTHCAL_NAVNEXT_PRESSED 10, 3
	MONTHCAL_NAVNEXT_DISABLED 10, 4

	MONTHCAL_NAVPREV_NORMAL 11, 1
	MONTHCAL_NAVPREV_HOT 11, 2
	MONTHCAL_NAVPREV_PRESSED 11, 3
	MONTHCAL_NAVPREV_DISABLED 11, 4
}

impl_pub_const_vs! {
	CLOCK_TIME_NORMAL 1, 1
	CLOCK_TIME_HOT 1, 2
	CLOCK_TIME_PRESSED 1, 3
}

impl_pub_const_vs! {
	TRAYNOTIFY_BACKGROUND 1, 0

	TRAYNOTIFY_ANIMBACKGROUND 2, 0
}

impl_pub_const_vs! {
	TASKBAR_BACKGROUNDBOTTOM 1, 0

	TASKBAR_BACKGROUNDRIGHT 2, 0

	TASKBAR_BACKGROUNDTOP 3, 0

	TASKBAR_BACKGROUNDLEFT 4, 0

	TASKBAR_SIZINGBARBOTTOM 5, 0

	TASKBAR_SIZINGBARRIGHT 6, 0

	TASKBAR_SIZINGBARTOP 7, 0

	TASKBAR_SIZINGBARLEFT 8, 0
}

impl_pub_const_vs! {
	TASKBAND_GROUPCOUNT 1, 0

	TASKBAND_FLASHBUTTON 2, 0

	TASKBAND_FLASHBUTTONGROUPMENU 3, 0
}

impl_pub_const_vs! {
	STARTPANEL_USERPANE 1, 0

	STARTPANEL_MOREPROGRAMS 2, 0

	STARTPANEL_MOREPROGRAMSARROW_NORMAL 3, 1
	STARTPANEL_MOREPROGRAMSARROW_HOT 3, 2
	STARTPANEL_MOREPROGRAMSARROW_PRESSED 3, 3

	STARTPANEL_PROGLIST 4, 0

	STARTPANEL_PROGLISTSEPARATOR 5, 0

	STARTPANEL_PLACESLIST 6, 0

	STARTPANEL_PLACESLISTSEPARATOR 7, 0

	STARTPANEL_LOGOFF 8, 0

	STARTPANEL_LOGOFFBUTTONS_NORMAL 9, 1
	STARTPANEL_LOGOFFBUTTONS_HOT 9, 2
	STARTPANEL_LOGOFFBUTTONS_PRESSED 9, 3

	STARTPANEL_USERPICTURE 10, 0

	STARTPANEL_PREVIEW 11, 0

	STARTPANEL_MOREPROGRAMSTAB_NORMAL 12, 1
	STARTPANEL_MOREPROGRAMSTAB_HOT 12, 2
	STARTPANEL_MOREPROGRAMSTAB_SELECTED 12, 3
	STARTPANEL_MOREPROGRAMSTAB_DISABLED 12, 4
	STARTPANEL_MOREPROGRAMSTAB_FOCUSED 12, 5

	STARTPANEL_NSCHOST 13, 0

	STARTPANEL_SOFTWAREEXPLORER_NORMAL 14, 1
	STARTPANEL_SOFTWAREEXPLORER_HOT 14, 2
	STARTPANEL_SOFTWAREEXPLORER_SELECTED 14, 3
	STARTPANEL_SOFTWAREEXPLORER_DISABLED 14, 4
	STARTPANEL_SOFTWAREEXPLORER_FOCUSED 14, 5

	STARTPANEL_OPENBOX_NORMAL 15, 1
	STARTPANEL_OPENBOX_HOT 15, 2
	STARTPANEL_OPENBOX_SELECTED 15, 3
	STARTPANEL_OPENBOX_DISABLED 15, 4
	STARTPANEL_OPENBOX_FOCUSED 15, 5

	STARTPANEL_SEARCHVIEW 16, 0

	STARTPANEL_MOREPROGRAMSARROWBACK_NORMAL 17, 1
	STARTPANEL_MOREPROGRAMSARROWBACK_HOT 17, 2
	STARTPANEL_MOREPROGRAMSARROWBACK_PRESSED 17, 3

	STARTPANEL_TOPMATCH 18, 0

	STARTPANEL_LOGOFFSPLITBUTTONDROPDOWN 19, 0
}

impl_pub_const_vs! {
	MENUBAND_NEWAPPBUTTON_NORMAL 1, 1
	MENUBAND_NEWAPPBUTTON_HOT 1, 2
	MENUBAND_NEWAPPBUTTON_PRESSED 1, 3
	MENUBAND_NEWAPPBUTTON_DISABLED 1, 4
	MENUBAND_NEWAPPBUTTON_CHECKED 1, 5
	MENUBAND_NEWAPPBUTTON_HOTCHECKED 1, 6

	MENUBAND_SEPERATOR 2, 0
}
