// License: GPLv3 Copyright: 2023, Kovid Goyal, <kovid at kovidgoyal.net>

package utils

import (
	"fmt"
	"testing"

	"github.com/google/go-cmp/cmp"
)

var _ = fmt.Print

func TestGettingLoginShell(t *testing.T) {
	entries, err := parse_dscl_data([]byte(all_users_plist))
	if err != nil {
		t.Fatal(err)
	}
	if diff := cmp.Diff(entries[`501`].Shell, "/bin/zsh"); diff != "" {
		t.Fatalf("Failed to parse dscl data for shell. %v %s", entries[`501`], diff)
	}
	if _, e := CurrentUser(); e != nil {
		t.Fatal(e)
	}
	if sh, e := LoginShellForCurrentUser(); e != nil || sh == "" {
		t.Fatalf("Failed to get login shell for current user with error: %s", e)
	}
}

// plist test data {{{
const all_users_plist string = `
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<array>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>278</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/accessoryupdater</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_accessoryupdater</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Accessory Update Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>278</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_accessoryupdater</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>83</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/virusmails</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_amavisd</string>
			<string>amavisd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>AMaViS Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>83</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_amavisd</string>
			<string>amavisd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>263</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/analyticsd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_analyticsd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Analytics Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>263</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_analyticsd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>273</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/appinstalld</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_appinstalld</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>App Install Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>273</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_appinstalld</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>55</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_appleevents</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>AppleEvents Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>55</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_appleevents</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>260</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/applepay</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_applepay</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>applepay Account</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>260</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_applepay</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>87</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_appowner</string>
			<string>appowner</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Application Owner</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>87</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_appowner</string>
			<string>appowner</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>79</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_appserver</string>
			<string>appserver</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Application Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>79</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_appserver</string>
			<string>appserver</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>33</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/appstore</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_appstore</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Mac App Store Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>33</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_appstore</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>67</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_ard</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Apple Remote Desktop</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>67</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_ard</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>235</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_assetcache</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Asset Cache Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>235</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_assetcache</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>245</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/astris</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_astris</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Astris Services</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>245</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_astris</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>97</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_atsserver</string>
			<string>atsserver</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>ATS Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>97</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_atsserver</string>
			<string>atsserver</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_avbdeviced</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Ethernet AVB Device Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>229</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_avbdeviced</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>288</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_avphidbridge</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Apple Virtual Platform HID Bridge</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>288</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_avphidbridge</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>291</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_backgroundassets</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Background Assets Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>291</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_backgroundassets</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>289</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/biome</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_biome</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Biome</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>289</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_biome</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>93</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_calendar</string>
			<string>calendar</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Calendar</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>93</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_calendar</string>
			<string>calendar</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>258</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_captiveagent</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>captiveagent</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>258</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_captiveagent</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>32</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_ces</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Certificate Enrollment Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>32</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_ces</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>82</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/virusmails</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_clamav</string>
			<string>clamav</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>ClamAV Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>82</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_clamav</string>
			<string>clamav</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>262</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/cmiodalassistants</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_cmiodalassistants</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>CoreMedia IO Assistants User</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>262</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_cmiodalassistants</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>202</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_coreaudiod</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Core Audio Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>202</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_coreaudiod</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>236</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_coremediaiod</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Core Media IO Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>236</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_coremediaiod</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>280</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/coreml</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_coreml</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>CoreML Services</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>280</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_coreml</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>259</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_ctkd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>ctkd Account</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>259</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_ctkd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>212</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_cvmsroot</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>CVMS Root</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>212</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_cvmsroot</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>72</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_cvs</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>CVS Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>72</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_cvs</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>6</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/imap</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_cyrus</string>
			<string>cyrusimap</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Cyrus Administrator</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>77</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_cyrus</string>
			<string>cyrusimap</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>284</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/darwindaemon</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_darwindaemon</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Darwin Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>284</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_darwindaemon</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>257</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/datadetectors</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_datadetectors</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>DataDetectors</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>257</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_datadetectors</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>275</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_demod</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Demo Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>275</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_demod</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>59</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_devdocs</string>
			<string>devdocs</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Developer Documentation</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>59</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_devdocs</string>
			<string>devdocs</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>220</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_devicemgr</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Device Management Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>220</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_devicemgr</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>271</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/diskimagesiod</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_diskimagesiod</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>DiskImages IO Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>271</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_diskimagesiod</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>244</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_displaypolicyd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Display Policy Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>244</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_displaypolicyd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>241</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_distnote</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>DistNote</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>241</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_distnote</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>6</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_dovecot</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Dovecot Administrator</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>214</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_dovecot</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>227</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_dovenull</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Dovecot Authentication</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>227</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_dovenull</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>215</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_dpaudio</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>DP Audio</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>215</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_dpaudio</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>270</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_driverkit</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>DriverKit</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>270</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_driverkit</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>71</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_eppc</string>
			<string>eppc</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Apple Events User</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>71</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_eppc</string>
			<string>eppc</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>254</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/findmydevice</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_findmydevice</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Find My Device Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>254</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_findmydevice</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>265</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/fpsd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_fpsd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>FPS Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>265</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_fpsd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_ftp</string>
			<string>ftp</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>FTP Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>98</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_ftp</string>
			<string>ftp</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>247</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_gamecontrollerd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Game Controller Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>247</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_gamecontrollerd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>56</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/geod</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_geod</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Geo Services Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>56</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_geod</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>261</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/hidd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_hidd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>HID Service User</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>261</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_hidd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>240</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_iconservices</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>IconServices</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>240</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_iconservices</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>25</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_installassistant</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Install Assistant</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>25</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_installassistant</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>274</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/installcoordinationd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_installcoordinationd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Install Coordination Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>274</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_installcoordinationd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_installer</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Installer</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>96</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_installer</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>84</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_jabber</string>
			<string>jabber</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Jabber XMPP Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>84</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_jabber</string>
			<string>jabber</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_kadmin_admin</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Kerberos Admin Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>218</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_kadmin_admin</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_kadmin_changepw</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Kerberos Change Password Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>219</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_kadmin_changepw</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>279</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/knowledgegraphd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_knowledgegraphd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Knowledge Graph Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>279</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_knowledgegraphd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_krb_anonymous</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Open Directory Kerberos Anonymous</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>234</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_krb_anonymous</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_krb_changepw</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Open Directory Kerberos Change Password Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>232</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_krb_changepw</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_krb_kadmin</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Open Directory Kerberos Admin Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>231</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_krb_kadmin</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_krb_kerberos</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Open Directory Kerberos</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>233</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_krb_kerberos</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_krb_krbtgt</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Open Directory Kerberos Ticket Granting Ticket</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>230</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_krb_krbtgt</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_krbfast</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Kerberos FAST Account</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>246</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_krbfast</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_krbtgt</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Kerberos Ticket Granting Ticket</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>217</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_krbtgt</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>239</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_launchservicesd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>_launchservicesd</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>239</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_launchservicesd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>211</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_lda</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Local Delivery Agent</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>211</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_lda</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>205</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/locationd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_locationd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Location Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>205</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_locationd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>272</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/diagnostics</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_logd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Log Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>272</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_logd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>26</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/spool/cups</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_lp</string>
			<string>lp</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Printing Services</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>26</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_lp</string>
			<string>lp</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>78</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_mailman</string>
			<string>mailman</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Mailman List Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>78</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_mailman</string>
			<string>mailman</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>248</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/setup</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_mbsetupuser</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Setup User</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/bin/bash</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>248</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_mbsetupuser</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>54</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_mcxalr</string>
			<string>mcxalr</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>MCX AppLaunch</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>54</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_mcxalr</string>
			<string>mcxalr</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>65</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_mdnsresponder</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>mDNSResponder</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>65</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_mdnsresponder</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>283</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/mmaintenanced</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_mmaintenanced</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>mmaintenanced</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>283</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_mmaintenanced</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>253</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/ma</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_mobileasset</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>MobileAsset User</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>253</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_mobileasset</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>74</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_mysql</string>
			<string>mysql</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>MySQL Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>74</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_mysql</string>
			<string>mysql</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>268</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/nearbyd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_nearbyd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Proximity and Ranging Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>268</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_nearbyd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>222</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_netbios</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>NetBIOS</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>222</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_netbios</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>228</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_netstatistics</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Network Statistics Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>228</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_netstatistics</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>24</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/networkd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_networkd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Network Services</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>24</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_networkd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>285</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_notification_proxy</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Notification Proxy</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>285</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_notification_proxy</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>242</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/nsurlsessiond</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_nsurlsessiond</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>NSURLSession Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>242</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_nsurlsessiond</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>243</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/nsurlstoraged</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_nsurlstoraged</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>_nsurlstoraged</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>243</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_nsurlstoraged</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>441</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_oahd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>OAH Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>441</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_oahd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>249</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/ondemand</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_ondemand</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>On Demand Resource Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>249</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_ondemand</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>27</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/spool/postfix</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_postfix</string>
			<string>postfix</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Postfix Mail Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>27</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_postfix</string>
			<string>postfix</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>216</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_postgres</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>PostgreSQL Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>216</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_postgres</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>76</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_qtss</string>
			<string>qtss</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>QuickTime Streaming Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>76</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_qtss</string>
			<string>qtss</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>269</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/reportmemoryexception</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_reportmemoryexception</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>ReportMemoryException</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>269</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_reportmemoryexception</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>277</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/rmd</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_rmd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Remote Management Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>277</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_rmd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>60</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_sandbox</string>
			<string>sandbox</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Seatbelt</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>60</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_sandbox</string>
			<string>sandbox</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>203</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_screensaver</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Screensaver</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>203</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_screensaver</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>31</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_scsd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Service Configuration Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>31</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_scsd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>92</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/securityagent</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_securityagent</string>
			<string>securityagent</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>SecurityAgent</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>92</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_securityagent</string>
			<string>securityagent</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>58</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_serialnumberd</string>
			<string>serialnumberd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>_serialnumberd</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>58</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_serialnumberd</string>
			<string>serialnumberd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>281</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_sntpd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>SNTP Server Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>281</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_sntpd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>200</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/softwareupdate</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_softwareupdate</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Software Update Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>200</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_softwareupdate</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>89</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_spotlight</string>
			<string>spotlight</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Spotlight</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>89</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_spotlight</string>
			<string>spotlight</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>75</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_sshd</string>
			<string>sshd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>sshd Privilege separation</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>75</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_sshd</string>
			<string>sshd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>73</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_svn</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>SVN Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>73</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_svn</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>13</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_taskgated</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Task Gate Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>13</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_taskgated</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>94</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/teamsserver</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_teamsserver</string>
			<string>teamsserver</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>TeamsServer</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>94</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_teamsserver</string>
			<string>teamsserver</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>266</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/timed</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_timed</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Time Sync Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>266</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_timed</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>210</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_timezone</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>AutoTimeZoneDaemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>210</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_timezone</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>91</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_tokend</string>
			<string>tokend</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Token Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>91</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_tokend</string>
			<string>tokend</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>282</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_trustd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>trustd</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>282</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_trustd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>208</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_trustevaluationagent</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Trust Evaluation Agent</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>208</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_trustevaluationagent</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>99</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_unknown</string>
			<string>unknown</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Unknown User</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>99</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_unknown</string>
			<string>unknown</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_update_sharing</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Update Sharing</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>95</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_update_sharing</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>213</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/db/lockdown</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_usbmuxd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>iPhone OS Device Helper</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>213</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_usbmuxd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>4</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/spool/uucp</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_uucp</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Unix to Unix Copy Protocol</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/sbin/uucico</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>4</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_uucp</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>224</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_warmd</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Warm Daemon</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>224</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_warmd</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>221</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_webauthserver</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Web Auth Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>221</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_webauthserver</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>88</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_windowserver</string>
			<string>windowserver</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>WindowServer</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>88</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_windowserver</string>
			<string>windowserver</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>70</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/Library/WebServer</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_www</string>
			<string>www</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>World Wide Web Server</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>70</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_www</string>
			<string>www</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>252</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_wwwproxy</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>WWW Proxy</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>252</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_wwwproxy</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>251</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>_xserverdocs</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>macOS Server Documents Service</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>251</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>_xserverdocs</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>1</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/root</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>daemon</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>System Services</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>1</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>daemon</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>20</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/Users/kovid</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>kovid</string>
			<string>com.apple.idms.appleid.prd.001239-05-db37ae71-1475-47b3-9cd5-d749f085f52d</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Kovid Goyal</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/bin/zsh</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>501</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>kovid</string>
			<string>com.apple.idms.appleid.prd.001239-05-db37ae71-1475-47b3-9cd5-d749f085f52d</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/empty</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>nobody</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>Unprivileged User</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/usr/bin/false</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>-2</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>nobody</string>
		</array>
	</dict>
	<dict>
		<key>dsAttrTypeNative:gid</key>
		<array>
			<string>0</string>
		</array>
		<key>dsAttrTypeNative:home</key>
		<array>
			<string>/var/root</string>
			<string>/private/var/root</string>
		</array>
		<key>dsAttrTypeNative:name</key>
		<array>
			<string>root</string>
			<string>BUILTIN\Local System</string>
		</array>
		<key>dsAttrTypeNative:realname</key>
		<array>
			<string>System Administrator</string>
		</array>
		<key>dsAttrTypeNative:shell</key>
		<array>
			<string>/bin/sh</string>
		</array>
		<key>dsAttrTypeNative:uid</key>
		<array>
			<string>0</string>
		</array>
		<key>dsAttrTypeStandard:RecordName</key>
		<array>
			<string>root</string>
			<string>BUILTIN\Local System</string>
		</array>
	</dict>
</array>
</plist>
` // }}}
