/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once


#include "mongo/base/status.h"
#include "mongo/db/jsobj.h"
#include "mongo/db/matcher/expression.h"
#include "mongo/db/matcher/extensions_callback_noop.h"
#include "mongo/db/pipeline/inner_pipeline_stage_interface.h"
#include "mongo/db/query/collation/collator_interface.h"
#include "mongo/db/query/parsed_find_command.h"
#include "mongo/db/query/projection.h"
#include "mongo/db/query/projection_policies.h"
#include "mongo/db/query/query_request_helper.h"
#include "mongo/db/query/sort_pattern.h"

namespace mongo {

class OperationContext;

class CanonicalQuery {
public:
    // A type that encodes the notion of query shape suitable for use with the plan cache. Encodes
    // the query's match, projection, sort, etc. potentially with some constants removed or replaced
    // with parameter markers.
    typedef std::string QueryShapeString;
    // A second encoding of query shape similar to 'QueryShapeString' above, except designed to work
    // with index filters and the 'planCacheClear' command. A caller can encode a query into an
    // 'PlanCacheCommandKey' in order to look for for matching index filters that should apply to
    // the query, or plan cache entries to which the 'planCacheClear' command should be applied.
    typedef std::string PlanCacheCommandKey;

    /**
     * If parsing succeeds, returns a std::unique_ptr<CanonicalQuery> representing the parsed
     * query (which will never be NULL).  If parsing fails, returns an error Status.
     *
     * 'opCtx' must point to a valid OperationContext, but 'opCtx' does not need to outlive the
     * returned CanonicalQuery.
     */
    static StatusWith<std::unique_ptr<CanonicalQuery>> canonicalize(
        OperationContext* opCtx,
        std::unique_ptr<FindCommandRequest> findCommand,
        bool explain = false,
        const boost::intrusive_ptr<ExpressionContext>& expCtx = nullptr,
        const ExtensionsCallback& extensionsCallback = ExtensionsCallbackNoop(),
        MatchExpressionParser::AllowedFeatureSet allowedFeatures =
            MatchExpressionParser::kDefaultSpecialFeatures,
        const ProjectionPolicies& projectionPolicies = ProjectionPolicies::findProjectionPolicies(),
        std::vector<std::unique_ptr<InnerPipelineStageInterface>> pipeline = {},
        bool isCountLike = false);

    /**
     * Creates a CanonicalQuery from a ParsedFindCommand. Uses 'expCtx->opCtx', which must be valid.
     */
    static StatusWith<std::unique_ptr<CanonicalQuery>> canonicalize(
        boost::intrusive_ptr<ExpressionContext> expCtx,
        std::unique_ptr<ParsedFindCommand> parsedFind,
        bool explain = false,
        std::vector<std::unique_ptr<InnerPipelineStageInterface>> pipeline = {},
        bool isCountLike = false);

    /**
     * For testing or for internal clients to use.
     */

    /**
     * Construct a 'CanonicalQuery' for a subquery of the given query. This function should only be
     * invoked by the subplanner. 'baseQuery' must contain a MatchExpression with rooted $or. This
     * function returns a 'CanonicalQuery' housing a copy of the i'th child of the root.
     */
    static StatusWith<std::unique_ptr<CanonicalQuery>> makeForSubplanner(
        OperationContext* opCtx, const CanonicalQuery& baseQuery, size_t i);

    /**
     * Returns true if "query" describes an exact-match query on _id.
     */
    static bool isSimpleIdQuery(const BSONObj& query);

    /**
     * Perform validation checks on the normalized 'root' which could not be checked before
     * normalization - those should happen in parsed_find_command::isValid().
     */
    static Status isValidNormalized(const MatchExpression* root);

    /**
     * For internal use only - but public for accessibility for make_unique(). You must go through
     * canonicalize to create a CanonicalQuery.
     */
    CanonicalQuery() {}

    NamespaceString nss() const {
        invariant(_findCommand->getNamespaceOrUUID().isNamespaceString());
        return _findCommand->getNamespaceOrUUID().nss();
    }
    StringData ns() const {
        return nss().ns();
    }

    //
    // Accessors for the query
    //
    MatchExpression* root() const {
        return _root.get();
    }
    const BSONObj& getQueryObj() const {
        return _findCommand->getFilter();
    }
    const FindCommandRequest& getFindCommandRequest() const {
        return *_findCommand;
    }

    /**
     * Returns the projection, or nullptr if none.
     */
    const projection_ast::Projection* getProj() const {
        return _proj.get_ptr();
    }

    projection_ast::Projection* getProj() {
        return _proj.get_ptr();
    }

    const boost::optional<SortPattern>& getSortPattern() const {
        return _sortPattern;
    }

    const CollatorInterface* getCollator() const {
        return _expCtx->getCollator();
    }

    /**
     * Returns a bitset indicating what metadata has been requested in the query.
     */
    const QueryMetadataBitSet& metadataDeps() const {
        return _metadataDeps;
    }

    /**
     * Allows callers to request metadata in addition to that needed as part of the query.
     */
    void requestAdditionalMetadata(const QueryMetadataBitSet& additionalDeps) {
        _metadataDeps |= additionalDeps;
    }

    /**
     * Compute the "shape" of this query by encoding the match, projection and sort, and stripping
     * out the appropriate values. Note that different types of PlanCache use different encoding
     * approaches.
     */
    QueryShapeString encodeKey() const;

    /**
     * Similar to 'encodeKey()' above, but intended for use with plan cache commands rather than
     * the plan cache itself.
     */
    PlanCacheCommandKey encodeKeyForPlanCacheCommand() const;

    /**
     * Sets this CanonicalQuery's collator, and sets the collator on this CanonicalQuery's match
     * expression tree.
     *
     * This setter can be used to override the collator that was created from the query request
     * during CanonicalQuery construction.
     */
    void setCollator(std::unique_ptr<CollatorInterface> collator);

    // Debugging
    std::string toString() const;
    std::string toStringShort() const;

    bool getExplain() const {
        return _explain;
    }

    bool getForceClassicEngine() const {
        return _forceClassicEngine;
    }

    void setSbeCompatible(bool sbeCompatible) {
        _sbeCompatible = sbeCompatible;
    }

    bool isSbeCompatible() const {
        return _sbeCompatible;
    }

    void parameterize();

    bool isParameterized() const {
        return !_inputParamIdToExpressionMap.empty();
    }

    const std::vector<const MatchExpression*>& getInputParamIdToMatchExpressionMap() const {
        return _inputParamIdToExpressionMap;
    }

    void setExplain(bool explain) {
        _explain = explain;
    }

    bool getForceGenerateRecordId() const {
        return _forceGenerateRecordId;
    }

    void setForceGenerateRecordId(bool value) {
        _forceGenerateRecordId = value;
    }

    OperationContext* getOpCtx() const {
        tassert(6508300, "'CanonicalQuery' does not have an 'ExpressionContext'", _expCtx);
        return _expCtx->opCtx;
    }

    auto& getExpCtx() const {
        return _expCtx;
    }
    auto getExpCtxRaw() const {
        return _expCtx.get();
    }

    void setPipeline(std::vector<std::unique_ptr<InnerPipelineStageInterface>> pipeline) {
        _pipeline = std::move(pipeline);
    }

    const std::vector<std::unique_ptr<InnerPipelineStageInterface>>& pipeline() const {
        return _pipeline;
    }

    /**
     * Returns true if the query is a count-like query, i.e. has no dependencies on inputs (see
     * DepsTracker::hasNoRequirements()). These queries can be served without accessing the source
     * documents (e.g. {$group: {_id: null, c: {$min: 42}}}) in which case we might be able to avoid
     * scanning the collection and instead use COUNT_SCAN or other optimizations.
     *
     * Note that this applies to the find/non-pipeline portion of the query. If the count-like group
     * is pushed down, later execution stages cannot be treated like a count. In other words, a
     * query with a pushed-down group may be considered a count at the data access layer but not
     * above the canonical query.
     */
    bool isCountLike() const {
        return _isCountLike;
    }

    void optimizeProjection() {
        if (_proj) {
            _proj->optimize();
        }
    }

private:
    Status init(boost::intrusive_ptr<ExpressionContext> expCtx,
                std::unique_ptr<ParsedFindCommand> parsedFind,
                std::vector<std::unique_ptr<InnerPipelineStageInterface>> pipeline,
                bool isCountLike,
                bool optimizeMatchExpression);

    boost::intrusive_ptr<ExpressionContext> _expCtx;

    std::unique_ptr<FindCommandRequest> _findCommand;

    std::unique_ptr<MatchExpression> _root;

    boost::optional<projection_ast::Projection> _proj;

    boost::optional<SortPattern> _sortPattern;

    // A query can include a post-processing pipeline here. Logically it is applied after all the
    // other operations (filter, sort, project, skip, limit).
    std::vector<std::unique_ptr<InnerPipelineStageInterface>> _pipeline;

    // Keeps track of what metadata has been explicitly requested.
    QueryMetadataBitSet _metadataDeps;

    bool _explain = false;

    // Determines whether the classic engine must be used.
    bool _forceClassicEngine = true;

    // True if this query can be executed by the SBE.
    bool _sbeCompatible = false;

    // True if this query must produce a RecordId output in addition to the BSON objects that
    // constitute the result set of the query. Any generated query solution must not discard record
    // ids, even if the optimizer detects that they are not going to be consumed downstream.
    bool _forceGenerateRecordId = false;

    // A map from assigned InputParamId's to parameterised MatchExpression's.
    std::vector<const MatchExpression*> _inputParamIdToExpressionMap;

    // "True" for queries that after doing a scan of an index can produce an empty document and
    // still be correct. For example, this applies to queries like [{$match: {x: 42}}, {$count:
    // "c"}] in presence of index on "x". The stage that follows the index scan doesn't have to be
    // $count but it must have no dependencies on the fields from the prior stages. Note, that
    // [{$match: {x: 42}}, {$group: {_id: "$y"}}, {$count: "c"}]] is _not_ "count like" because
    // the first $group stage needs to access field "y" and this access cannot be incorporated into
    // the index scan.
    bool _isCountLike = false;
};

}  // namespace mongo
