# This code is part of Qiskit.
#
# (C) Copyright IBM 2017, 2023.
#
# This code is licensed under the Apache License, Version 2.0. You may
# obtain a copy of this license in the LICENSE.txt file in the root directory
# of this source tree or at http://www.apache.org/licenses/LICENSE-2.0.
#
# Any modifications or derivative works of this code must retain this
# copyright notice, and modified files need to carry a notice indicating
# that they have been altered from the originals.

"""Dictionary containing commutation relations of unparameterizable standard gates. The dictionary key
is defined by a pair of gates and, optionally, their relative placement to each other, i.e.:

   ┌───┐┌───┐
q: ┤ X ├┤ Z ├    -> standard_gates_commutations["x", "z"] = False
   └───┘└───┘

or
          ┌───┐
q_0: ──■──┤ X ├
     ┌─┴─┐└─┬─┘  -> standard_gates_commutations["cx", "cy"][1, 0] = False
q_1: ┤ Y ├──■──
     └───┘

or

q_0: ──■───────
     ┌─┴─┐
q_1: ┤ X ├──■──  -> standard_gates_commutations['cx', 'cx'][None, 0] = False
     └───┘┌─┴─┐
q_2: ─────┤ X ├
          └───┘
Note that the commutation of a pair of gates is only stored in one specific order, i.e. the commutation
of a pair of cy and cx gates is only stored in standard_gates_commutations["cx", "cy"]. The order of the
gates is derived from the number of qubits in each gate and the value of the integer representation of
the gate name.

The values of standard_gates_commutations can be a Boolean value denoting whether the pair of gates
commute or another dictionary if the relative placement of the pair of gates has an impact on the
commutation. The relative placement is defined by the gate qubit arrangements as q2^{-1}[q1[i]] where
q1[i] is the ith qubit of the first gate and q2^{-1}[q] returns the qubit index of qubit q in the second
gate (possibly 'None'). In the second example, the zeroth qubit of cy overlaps with qubit 1 of cx. Non-
overlapping qubits are specified as 'None' in the dictionary key, e.g. example 3, there is no overlap
on the zeroth qubit of the first cx but the zeroth qubit of the second gate overlaps with qubit 1 of the
first cx.
"""


standard_gates_commutations = {
    ("id", "id"): True,
    ("id", "sx"): True,
    ("id", "cx"): True,
    ("id", "c3sx"): True,
    ("id", "ccx"): True,
    ("id", "dcx"): True,
    ("id", "ch"): True,
    ("id", "cswap"): True,
    ("id", "csx"): True,
    ("id", "cy"): True,
    ("id", "cz"): True,
    ("id", "ccz"): True,
    ("id", "rccx"): True,
    ("id", "rcccx"): True,
    ("id", "ecr"): True,
    ("id", "sdg"): True,
    ("id", "cs"): True,
    ("id", "csdg"): True,
    ("id", "swap"): True,
    ("id", "iswap"): True,
    ("id", "sxdg"): True,
    ("id", "tdg"): True,
    ("id", "rxx"): True,
    ("id", "ryy"): True,
    ("id", "rzz"): True,
    ("id", "rzx"): True,
    ("sx", "sx"): True,
    ("sx", "cx"): {
        (0,): False,
        (1,): True,
    },
    ("sx", "c3sx"): {
        (0,): False,
        (1,): False,
        (2,): False,
        (3,): True,
    },
    ("sx", "ccx"): {
        (0,): False,
        (1,): False,
        (2,): True,
    },
    ("sx", "dcx"): False,
    ("sx", "ch"): False,
    ("sx", "cswap"): False,
    ("sx", "csx"): {
        (0,): False,
        (1,): True,
    },
    ("sx", "cy"): False,
    ("sx", "cz"): False,
    ("sx", "ccz"): False,
    ("sx", "rccx"): False,
    ("sx", "rcccx"): False,
    ("sx", "ecr"): {
        (0,): False,
        (1,): True,
    },
    ("sx", "sdg"): False,
    ("sx", "cs"): False,
    ("sx", "csdg"): False,
    ("sx", "swap"): False,
    ("sx", "iswap"): False,
    ("sx", "sxdg"): True,
    ("sx", "tdg"): False,
    ("sx", "rxx"): True,
    ("sx", "ryy"): False,
    ("sx", "rzz"): False,
    ("sx", "rzx"): {
        (0,): False,
        (1,): True,
    },
    ("x", "id"): True,
    ("x", "sx"): True,
    ("x", "x"): True,
    ("x", "cx"): {
        (0,): False,
        (1,): True,
    },
    ("x", "c3sx"): {
        (0,): False,
        (1,): False,
        (2,): False,
        (3,): True,
    },
    ("x", "ccx"): {
        (0,): False,
        (1,): False,
        (2,): True,
    },
    ("x", "dcx"): False,
    ("x", "ch"): False,
    ("x", "cswap"): False,
    ("x", "csx"): {
        (0,): False,
        (1,): True,
    },
    ("x", "cy"): False,
    ("x", "cz"): False,
    ("x", "ccz"): False,
    ("x", "rccx"): False,
    ("x", "rcccx"): False,
    ("x", "ecr"): {
        (0,): False,
        (1,): True,
    },
    ("x", "sdg"): False,
    ("x", "cs"): False,
    ("x", "csdg"): False,
    ("x", "swap"): False,
    ("x", "iswap"): False,
    ("x", "sxdg"): True,
    ("x", "tdg"): False,
    ("x", "y"): False,
    ("x", "z"): False,
    ("x", "rxx"): True,
    ("x", "ryy"): False,
    ("x", "rzz"): False,
    ("x", "rzx"): {
        (0,): False,
        (1,): True,
    },
    ("cx", "cx"): {
        (0, 1): True,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("cx", "c3sx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): True,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): True,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): True,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): True,
    },
    ("cx", "ccx"): {
        (0, 1): False,
        (0, 2): True,
        (0, None): True,
        (1, 0): False,
        (1, 2): True,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): True,
    },
    ("cx", "dcx"): False,
    ("cx", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("cx", "csx"): {
        (0, 1): True,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("cx", "cy"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("cx", "cz"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("cx", "ccz"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): True,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("cx", "rccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("cx", "rcccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("cx", "ecr"): {
        (0, 1): False,
        (0, None): False,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("cx", "csdg"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("cx", "swap"): False,
    ("cx", "iswap"): False,
    ("cx", "rxx"): {
        (0, 1): False,
        (0, None): False,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): True,
    },
    ("cx", "ryy"): False,
    ("cx", "rzz"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("cx", "rzx"): {
        (0, 1): True,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("c3sx", "c3sx"): {
        (0, 1, 2, 3): True,
        (0, 1, 2, None): True,
        (0, 1, 3, 2): False,
        (0, 1, 3, None): False,
        (0, 1, None, 2): False,
        (0, 1, None, 3): True,
        (0, 1, None, None): True,
        (0, 2, 1, 3): True,
        (0, 2, 1, None): True,
        (0, 2, 3, 1): False,
        (0, 2, 3, None): False,
        (0, 2, None, 1): False,
        (0, 2, None, 3): True,
        (0, 2, None, None): True,
        (0, 3, 1, 2): False,
        (0, 3, 1, None): False,
        (0, 3, 2, 1): False,
        (0, 3, 2, None): False,
        (0, 3, None, 1): False,
        (0, 3, None, 2): False,
        (0, 3, None, None): False,
        (0, None, 1, 2): False,
        (0, None, 1, 3): True,
        (0, None, 1, None): True,
        (0, None, 2, 1): False,
        (0, None, 2, 3): True,
        (0, None, 2, None): True,
        (0, None, 3, 1): False,
        (0, None, 3, 2): False,
        (0, None, 3, None): False,
        (0, None, None, 1): False,
        (0, None, None, 2): False,
        (0, None, None, 3): True,
        (0, None, None, None): True,
        (1, 0, 2, 3): True,
        (1, 0, 2, None): True,
        (1, 0, 3, 2): False,
        (1, 0, 3, None): False,
        (1, 0, None, 2): False,
        (1, 0, None, 3): True,
        (1, 0, None, None): True,
        (1, 2, 0, 3): True,
        (1, 2, 0, None): True,
        (1, 2, 3, 0): False,
        (1, 2, 3, None): False,
        (1, 2, None, 0): False,
        (1, 2, None, 3): True,
        (1, 2, None, None): True,
        (1, 3, 0, 2): False,
        (1, 3, 0, None): False,
        (1, 3, 2, 0): False,
        (1, 3, 2, None): False,
        (1, 3, None, 0): False,
        (1, 3, None, 2): False,
        (1, 3, None, None): False,
        (1, None, 0, 2): False,
        (1, None, 0, 3): True,
        (1, None, 0, None): True,
        (1, None, 2, 0): False,
        (1, None, 2, 3): True,
        (1, None, 2, None): True,
        (1, None, 3, 0): False,
        (1, None, 3, 2): False,
        (1, None, 3, None): False,
        (1, None, None, 0): False,
        (1, None, None, 2): False,
        (1, None, None, 3): True,
        (1, None, None, None): True,
        (2, 0, 1, 3): True,
        (2, 0, 1, None): True,
        (2, 0, 3, 1): False,
        (2, 0, 3, None): False,
        (2, 0, None, 1): False,
        (2, 0, None, 3): True,
        (2, 0, None, None): True,
        (2, 1, 0, 3): True,
        (2, 1, 0, None): True,
        (2, 1, 3, 0): False,
        (2, 1, 3, None): False,
        (2, 1, None, 0): False,
        (2, 1, None, 3): True,
        (2, 1, None, None): True,
        (2, 3, 0, 1): False,
        (2, 3, 0, None): False,
        (2, 3, 1, 0): False,
        (2, 3, 1, None): False,
        (2, 3, None, 0): False,
        (2, 3, None, 1): False,
        (2, 3, None, None): False,
        (2, None, 0, 1): False,
        (2, None, 0, 3): True,
        (2, None, 0, None): True,
        (2, None, 1, 0): False,
        (2, None, 1, 3): True,
        (2, None, 1, None): True,
        (2, None, 3, 0): False,
        (2, None, 3, 1): False,
        (2, None, 3, None): False,
        (2, None, None, 0): False,
        (2, None, None, 1): False,
        (2, None, None, 3): True,
        (2, None, None, None): True,
        (3, 0, 1, 2): False,
        (3, 0, 1, None): False,
        (3, 0, 2, 1): False,
        (3, 0, 2, None): False,
        (3, 0, None, 1): False,
        (3, 0, None, 2): False,
        (3, 0, None, None): False,
        (3, 1, 0, 2): False,
        (3, 1, 0, None): False,
        (3, 1, 2, 0): False,
        (3, 1, 2, None): False,
        (3, 1, None, 0): False,
        (3, 1, None, 2): False,
        (3, 1, None, None): False,
        (3, 2, 0, 1): False,
        (3, 2, 0, None): False,
        (3, 2, 1, 0): False,
        (3, 2, 1, None): False,
        (3, 2, None, 0): False,
        (3, 2, None, 1): False,
        (3, 2, None, None): False,
        (3, None, 0, 1): False,
        (3, None, 0, 2): False,
        (3, None, 0, None): False,
        (3, None, 1, 0): False,
        (3, None, 1, 2): False,
        (3, None, 1, None): False,
        (3, None, 2, 0): False,
        (3, None, 2, 1): False,
        (3, None, 2, None): False,
        (3, None, None, 0): False,
        (3, None, None, 1): False,
        (3, None, None, 2): False,
        (3, None, None, None): False,
        (None, 0, 1, 2): False,
        (None, 0, 1, 3): True,
        (None, 0, 1, None): True,
        (None, 0, 2, 1): False,
        (None, 0, 2, 3): True,
        (None, 0, 2, None): True,
        (None, 0, 3, 1): False,
        (None, 0, 3, 2): False,
        (None, 0, 3, None): False,
        (None, 0, None, 1): False,
        (None, 0, None, 2): False,
        (None, 0, None, 3): True,
        (None, 0, None, None): True,
        (None, 1, 0, 2): False,
        (None, 1, 0, 3): True,
        (None, 1, 0, None): True,
        (None, 1, 2, 0): False,
        (None, 1, 2, 3): True,
        (None, 1, 2, None): True,
        (None, 1, 3, 0): False,
        (None, 1, 3, 2): False,
        (None, 1, 3, None): False,
        (None, 1, None, 0): False,
        (None, 1, None, 2): False,
        (None, 1, None, 3): True,
        (None, 1, None, None): True,
        (None, 2, 0, 1): False,
        (None, 2, 0, 3): True,
        (None, 2, 0, None): True,
        (None, 2, 1, 0): False,
        (None, 2, 1, 3): True,
        (None, 2, 1, None): True,
        (None, 2, 3, 0): False,
        (None, 2, 3, 1): False,
        (None, 2, 3, None): False,
        (None, 2, None, 0): False,
        (None, 2, None, 1): False,
        (None, 2, None, 3): True,
        (None, 2, None, None): True,
        (None, 3, 0, 1): False,
        (None, 3, 0, 2): False,
        (None, 3, 0, None): False,
        (None, 3, 1, 0): False,
        (None, 3, 1, 2): False,
        (None, 3, 1, None): False,
        (None, 3, 2, 0): False,
        (None, 3, 2, 1): False,
        (None, 3, 2, None): False,
        (None, 3, None, 0): False,
        (None, 3, None, 1): False,
        (None, 3, None, 2): False,
        (None, 3, None, None): False,
        (None, None, 0, 1): False,
        (None, None, 0, 2): False,
        (None, None, 0, 3): True,
        (None, None, 0, None): True,
        (None, None, 1, 0): False,
        (None, None, 1, 2): False,
        (None, None, 1, 3): True,
        (None, None, 1, None): True,
        (None, None, 2, 0): False,
        (None, None, 2, 1): False,
        (None, None, 2, 3): True,
        (None, None, 2, None): True,
        (None, None, 3, 0): False,
        (None, None, 3, 1): False,
        (None, None, 3, 2): False,
        (None, None, 3, None): False,
        (None, None, None, 0): False,
        (None, None, None, 1): False,
        (None, None, None, 2): False,
        (None, None, None, 3): True,
    },
    ("c3sx", "rcccx"): {
        (0, 1, 2, 3): False,
        (0, 1, 2, None): True,
        (0, 1, 3, 2): False,
        (0, 1, 3, None): False,
        (0, 1, None, 2): False,
        (0, 1, None, 3): False,
        (0, 1, None, None): True,
        (0, 2, 1, 3): False,
        (0, 2, 1, None): True,
        (0, 2, 3, 1): False,
        (0, 2, 3, None): False,
        (0, 2, None, 1): False,
        (0, 2, None, 3): False,
        (0, 2, None, None): True,
        (0, 3, 1, 2): False,
        (0, 3, 1, None): False,
        (0, 3, 2, 1): False,
        (0, 3, 2, None): False,
        (0, 3, None, 1): False,
        (0, 3, None, 2): False,
        (0, 3, None, None): False,
        (0, None, 1, 2): False,
        (0, None, 1, 3): False,
        (0, None, 1, None): True,
        (0, None, 2, 1): False,
        (0, None, 2, 3): False,
        (0, None, 2, None): True,
        (0, None, 3, 1): False,
        (0, None, 3, 2): False,
        (0, None, 3, None): False,
        (0, None, None, 1): False,
        (0, None, None, 2): False,
        (0, None, None, 3): False,
        (0, None, None, None): True,
        (1, 0, 2, 3): False,
        (1, 0, 2, None): True,
        (1, 0, 3, 2): False,
        (1, 0, 3, None): False,
        (1, 0, None, 2): False,
        (1, 0, None, 3): False,
        (1, 0, None, None): True,
        (1, 2, 0, 3): False,
        (1, 2, 0, None): True,
        (1, 2, 3, 0): False,
        (1, 2, 3, None): False,
        (1, 2, None, 0): False,
        (1, 2, None, 3): False,
        (1, 2, None, None): True,
        (1, 3, 0, 2): False,
        (1, 3, 0, None): False,
        (1, 3, 2, 0): False,
        (1, 3, 2, None): False,
        (1, 3, None, 0): False,
        (1, 3, None, 2): False,
        (1, 3, None, None): False,
        (1, None, 0, 2): False,
        (1, None, 0, 3): False,
        (1, None, 0, None): True,
        (1, None, 2, 0): False,
        (1, None, 2, 3): False,
        (1, None, 2, None): True,
        (1, None, 3, 0): False,
        (1, None, 3, 2): False,
        (1, None, 3, None): False,
        (1, None, None, 0): False,
        (1, None, None, 2): False,
        (1, None, None, 3): False,
        (1, None, None, None): True,
        (2, 0, 1, 3): False,
        (2, 0, 1, None): True,
        (2, 0, 3, 1): False,
        (2, 0, 3, None): False,
        (2, 0, None, 1): False,
        (2, 0, None, 3): False,
        (2, 0, None, None): True,
        (2, 1, 0, 3): False,
        (2, 1, 0, None): True,
        (2, 1, 3, 0): False,
        (2, 1, 3, None): False,
        (2, 1, None, 0): False,
        (2, 1, None, 3): False,
        (2, 1, None, None): True,
        (2, 3, 0, 1): False,
        (2, 3, 0, None): False,
        (2, 3, 1, 0): False,
        (2, 3, 1, None): False,
        (2, 3, None, 0): False,
        (2, 3, None, 1): False,
        (2, 3, None, None): False,
        (2, None, 0, 1): False,
        (2, None, 0, 3): False,
        (2, None, 0, None): True,
        (2, None, 1, 0): False,
        (2, None, 1, 3): False,
        (2, None, 1, None): True,
        (2, None, 3, 0): False,
        (2, None, 3, 1): False,
        (2, None, 3, None): False,
        (2, None, None, 0): False,
        (2, None, None, 1): False,
        (2, None, None, 3): False,
        (2, None, None, None): True,
        (3, 0, 1, 2): False,
        (3, 0, 1, None): False,
        (3, 0, 2, 1): False,
        (3, 0, 2, None): False,
        (3, 0, None, 1): False,
        (3, 0, None, 2): False,
        (3, 0, None, None): False,
        (3, 1, 0, 2): False,
        (3, 1, 0, None): False,
        (3, 1, 2, 0): False,
        (3, 1, 2, None): False,
        (3, 1, None, 0): False,
        (3, 1, None, 2): False,
        (3, 1, None, None): False,
        (3, 2, 0, 1): False,
        (3, 2, 0, None): False,
        (3, 2, 1, 0): False,
        (3, 2, 1, None): False,
        (3, 2, None, 0): False,
        (3, 2, None, 1): False,
        (3, 2, None, None): False,
        (3, None, 0, 1): False,
        (3, None, 0, 2): False,
        (3, None, 0, None): False,
        (3, None, 1, 0): False,
        (3, None, 1, 2): False,
        (3, None, 1, None): False,
        (3, None, 2, 0): False,
        (3, None, 2, 1): False,
        (3, None, 2, None): False,
        (3, None, None, 0): False,
        (3, None, None, 1): False,
        (3, None, None, 2): False,
        (3, None, None, None): False,
        (None, 0, 1, 2): False,
        (None, 0, 1, 3): False,
        (None, 0, 1, None): True,
        (None, 0, 2, 1): False,
        (None, 0, 2, 3): False,
        (None, 0, 2, None): True,
        (None, 0, 3, 1): False,
        (None, 0, 3, 2): False,
        (None, 0, 3, None): False,
        (None, 0, None, 1): False,
        (None, 0, None, 2): False,
        (None, 0, None, 3): False,
        (None, 0, None, None): True,
        (None, 1, 0, 2): False,
        (None, 1, 0, 3): False,
        (None, 1, 0, None): True,
        (None, 1, 2, 0): False,
        (None, 1, 2, 3): False,
        (None, 1, 2, None): True,
        (None, 1, 3, 0): False,
        (None, 1, 3, 2): False,
        (None, 1, 3, None): False,
        (None, 1, None, 0): False,
        (None, 1, None, 2): False,
        (None, 1, None, 3): False,
        (None, 1, None, None): True,
        (None, 2, 0, 1): False,
        (None, 2, 0, 3): False,
        (None, 2, 0, None): True,
        (None, 2, 1, 0): False,
        (None, 2, 1, 3): False,
        (None, 2, 1, None): True,
        (None, 2, 3, 0): False,
        (None, 2, 3, 1): False,
        (None, 2, 3, None): False,
        (None, 2, None, 0): False,
        (None, 2, None, 1): False,
        (None, 2, None, 3): False,
        (None, 2, None, None): True,
        (None, 3, 0, 1): False,
        (None, 3, 0, 2): False,
        (None, 3, 0, None): False,
        (None, 3, 1, 0): False,
        (None, 3, 1, 2): False,
        (None, 3, 1, None): False,
        (None, 3, 2, 0): False,
        (None, 3, 2, 1): False,
        (None, 3, 2, None): False,
        (None, 3, None, 0): False,
        (None, 3, None, 1): False,
        (None, 3, None, 2): False,
        (None, 3, None, None): False,
        (None, None, 0, 1): False,
        (None, None, 0, 2): False,
        (None, None, 0, 3): False,
        (None, None, 0, None): True,
        (None, None, 1, 0): False,
        (None, None, 1, 2): False,
        (None, None, 1, 3): False,
        (None, None, 1, None): True,
        (None, None, 2, 0): False,
        (None, None, 2, 1): False,
        (None, None, 2, 3): False,
        (None, None, 2, None): True,
        (None, None, 3, 0): False,
        (None, None, 3, 1): False,
        (None, None, 3, 2): False,
        (None, None, 3, None): False,
        (None, None, None, 0): False,
        (None, None, None, 1): False,
        (None, None, None, 2): False,
        (None, None, None, 3): False,
    },
    ("ccx", "c3sx"): {
        (0, 1, 2): False,
        (0, 1, 3): True,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, 3): True,
        (0, 2, None): True,
        (0, 3, 1): False,
        (0, 3, 2): False,
        (0, 3, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, 3): True,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, 3): True,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, 3): True,
        (1, 2, None): True,
        (1, 3, 0): False,
        (1, 3, 2): False,
        (1, 3, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, 3): True,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, 3): True,
        (2, 0, None): True,
        (2, 1, 0): False,
        (2, 1, 3): True,
        (2, 1, None): True,
        (2, 3, 0): False,
        (2, 3, 1): False,
        (2, 3, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, 3): True,
        (2, None, None): True,
        (3, 0, 1): False,
        (3, 0, 2): False,
        (3, 0, None): False,
        (3, 1, 0): False,
        (3, 1, 2): False,
        (3, 1, None): False,
        (3, 2, 0): False,
        (3, 2, 1): False,
        (3, 2, None): False,
        (3, None, 0): False,
        (3, None, 1): False,
        (3, None, 2): False,
        (3, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, 3): True,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): False,
        (None, 1, 3): True,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, 3): True,
        (None, 2, None): True,
        (None, 3, 0): False,
        (None, 3, 1): False,
        (None, 3, 2): False,
        (None, 3, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
        (None, None, 3): True,
    },
    ("ccx", "ccx"): {
        (0, 1, 2): True,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, None): False,
        (0, None, 1): False,
        (0, None, 2): True,
        (0, None, None): True,
        (1, 0, 2): True,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, None): False,
        (1, None, 0): False,
        (1, None, 2): True,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, None): False,
        (2, 1, 0): False,
        (2, 1, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): True,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): True,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): True,
    },
    ("ccx", "cswap"): {
        (0, 1, 2): False,
        (0, 1, None): False,
        (0, 2, 1): False,
        (0, 2, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, None): False,
        (1, 2, 0): True,
        (1, 2, None): True,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, None): False,
        (2, 0, 1): False,
        (2, 0, None): False,
        (2, 1, 0): True,
        (2, 1, None): True,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): False,
        (None, 1, None): False,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
    },
    ("ccx", "ccz"): {
        (0, 1, 2): False,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, None): True,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, None): True,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, None): True,
        (2, 1, 0): False,
        (2, 1, None): True,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, None): True,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): False,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, None): True,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
    },
    ("ccx", "rccx"): {
        (0, 1, 2): False,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, None): False,
        (2, 1, 0): False,
        (2, 1, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): False,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
    },
    ("ccx", "rcccx"): {
        (0, 1, 2): False,
        (0, 1, 3): False,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, 3): False,
        (0, 2, None): True,
        (0, 3, 1): False,
        (0, 3, 2): False,
        (0, 3, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, 3): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, 3): False,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, 3): False,
        (1, 2, None): True,
        (1, 3, 0): False,
        (1, 3, 2): False,
        (1, 3, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, 3): False,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, 3): False,
        (2, 0, None): True,
        (2, 1, 0): False,
        (2, 1, 3): False,
        (2, 1, None): True,
        (2, 3, 0): False,
        (2, 3, 1): False,
        (2, 3, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, 3): False,
        (2, None, None): True,
        (3, 0, 1): False,
        (3, 0, 2): False,
        (3, 0, None): False,
        (3, 1, 0): False,
        (3, 1, 2): False,
        (3, 1, None): False,
        (3, 2, 0): False,
        (3, 2, 1): False,
        (3, 2, None): False,
        (3, None, 0): False,
        (3, None, 1): False,
        (3, None, 2): False,
        (3, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, 3): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): False,
        (None, 1, 3): False,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, 3): False,
        (None, 2, None): True,
        (None, 3, 0): False,
        (None, 3, 1): False,
        (None, 3, 2): False,
        (None, 3, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
        (None, None, 3): False,
    },
    ("dcx", "c3sx"): False,
    ("dcx", "ccx"): False,
    ("dcx", "dcx"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("dcx", "cswap"): False,
    ("dcx", "ccz"): False,
    ("dcx", "rccx"): False,
    ("dcx", "rcccx"): False,
    ("dcx", "ecr"): False,
    ("dcx", "csdg"): False,
    ("dcx", "swap"): False,
    ("dcx", "iswap"): False,
    ("dcx", "rxx"): False,
    ("dcx", "ryy"): False,
    ("dcx", "rzz"): False,
    ("dcx", "rzx"): False,
    ("ch", "cx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("ch", "c3sx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("ch", "ccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("ch", "dcx"): False,
    ("ch", "ch"): {
        (0, 1): True,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("ch", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("ch", "csx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("ch", "cy"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("ch", "cz"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("ch", "ccz"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): True,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("ch", "rccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("ch", "rcccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("ch", "ecr"): False,
    ("ch", "cs"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("ch", "csdg"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("ch", "swap"): False,
    ("ch", "iswap"): False,
    ("ch", "rxx"): False,
    ("ch", "ryy"): False,
    ("ch", "rzz"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("ch", "rzx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("cswap", "c3sx"): {
        (0, 1, 2): True,
        (0, 1, 3): False,
        (0, 1, None): False,
        (0, 2, 1): True,
        (0, 2, 3): False,
        (0, 2, None): False,
        (0, 3, 1): False,
        (0, 3, 2): False,
        (0, 3, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, 3): False,
        (0, None, None): True,
        (1, 0, 2): True,
        (1, 0, 3): False,
        (1, 0, None): False,
        (1, 2, 0): True,
        (1, 2, 3): False,
        (1, 2, None): False,
        (1, 3, 0): False,
        (1, 3, 2): False,
        (1, 3, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, 3): False,
        (1, None, None): True,
        (2, 0, 1): True,
        (2, 0, 3): False,
        (2, 0, None): False,
        (2, 1, 0): True,
        (2, 1, 3): False,
        (2, 1, None): False,
        (2, 3, 0): False,
        (2, 3, 1): False,
        (2, 3, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, 3): False,
        (2, None, None): True,
        (3, 0, 1): True,
        (3, 0, 2): True,
        (3, 0, None): False,
        (3, 1, 0): True,
        (3, 1, 2): True,
        (3, 1, None): False,
        (3, 2, 0): True,
        (3, 2, 1): True,
        (3, 2, None): False,
        (3, None, 0): False,
        (3, None, 1): False,
        (3, None, 2): False,
        (3, None, None): False,
        (None, 0, 1): True,
        (None, 0, 2): True,
        (None, 0, 3): False,
        (None, 0, None): False,
        (None, 1, 0): True,
        (None, 1, 2): True,
        (None, 1, 3): False,
        (None, 1, None): False,
        (None, 2, 0): True,
        (None, 2, 1): True,
        (None, 2, 3): False,
        (None, 2, None): False,
        (None, 3, 0): False,
        (None, 3, 1): False,
        (None, 3, 2): False,
        (None, 3, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
        (None, None, 3): False,
    },
    ("cswap", "cswap"): {
        (0, 1, 2): True,
        (0, 1, None): False,
        (0, 2, 1): True,
        (0, 2, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, None): False,
        (1, 2, 0): False,
        (1, 2, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, None): False,
        (2, 0, 1): False,
        (2, 0, None): False,
        (2, 1, 0): False,
        (2, 1, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, None): False,
        (None, 1, 0): False,
        (None, 1, 2): True,
        (None, 1, None): False,
        (None, 2, 0): False,
        (None, 2, 1): True,
        (None, 2, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
    },
    ("cswap", "rcccx"): {
        (0, 1, 2): False,
        (0, 1, 3): False,
        (0, 1, None): False,
        (0, 2, 1): False,
        (0, 2, 3): False,
        (0, 2, None): False,
        (0, 3, 1): False,
        (0, 3, 2): False,
        (0, 3, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, 3): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, 3): False,
        (1, 0, None): False,
        (1, 2, 0): False,
        (1, 2, 3): False,
        (1, 2, None): False,
        (1, 3, 0): False,
        (1, 3, 2): False,
        (1, 3, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, 3): False,
        (1, None, None): True,
        (2, 0, 1): True,
        (2, 0, 3): False,
        (2, 0, None): False,
        (2, 1, 0): True,
        (2, 1, 3): False,
        (2, 1, None): False,
        (2, 3, 0): False,
        (2, 3, 1): False,
        (2, 3, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, 3): False,
        (2, None, None): True,
        (3, 0, 1): True,
        (3, 0, 2): False,
        (3, 0, None): False,
        (3, 1, 0): True,
        (3, 1, 2): False,
        (3, 1, None): False,
        (3, 2, 0): False,
        (3, 2, 1): False,
        (3, 2, None): False,
        (3, None, 0): False,
        (3, None, 1): False,
        (3, None, 2): False,
        (3, None, None): False,
        (None, 0, 1): True,
        (None, 0, 2): False,
        (None, 0, 3): False,
        (None, 0, None): False,
        (None, 1, 0): True,
        (None, 1, 2): False,
        (None, 1, 3): False,
        (None, 1, None): False,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, 3): False,
        (None, 2, None): False,
        (None, 3, 0): False,
        (None, 3, 1): False,
        (None, 3, 2): False,
        (None, 3, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
        (None, None, 3): False,
    },
    ("csx", "c3sx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): True,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): True,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): True,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): True,
    },
    ("csx", "ccx"): {
        (0, 1): False,
        (0, 2): True,
        (0, None): True,
        (1, 0): False,
        (1, 2): True,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): True,
    },
    ("csx", "dcx"): False,
    ("csx", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("csx", "csx"): {
        (0, 1): True,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("csx", "ccz"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): True,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("csx", "rccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("csx", "rcccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("csx", "ecr"): {
        (0, 1): False,
        (0, None): False,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("csx", "csdg"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("csx", "swap"): False,
    ("csx", "iswap"): False,
    ("csx", "rxx"): {
        (0, 1): False,
        (0, None): False,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): True,
    },
    ("csx", "ryy"): False,
    ("csx", "rzz"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("csx", "rzx"): {
        (0, 1): True,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("cy", "c3sx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("cy", "ccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("cy", "dcx"): False,
    ("cy", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("cy", "csx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("cy", "cy"): {
        (0, 1): True,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("cy", "cz"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("cy", "ccz"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): True,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("cy", "rccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): True,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("cy", "rcccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): True,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("cy", "ecr"): False,
    ("cy", "csdg"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("cy", "swap"): False,
    ("cy", "iswap"): False,
    ("cy", "rxx"): False,
    ("cy", "ryy"): {
        (0, 1): False,
        (0, None): False,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): True,
    },
    ("cy", "rzz"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("cy", "rzx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("cz", "c3sx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): True,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): True,
        (None, 3): False,
    },
    ("cz", "ccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): False,
    },
    ("cz", "dcx"): False,
    ("cz", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): True,
        (1, None): False,
        (2, 0): False,
        (2, 1): True,
        (2, None): False,
        (None, 0): True,
        (None, 1): False,
        (None, 2): False,
    },
    ("cz", "csx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): False,
    },
    ("cz", "cz"): True,
    ("cz", "ccz"): True,
    ("cz", "rccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): False,
    },
    ("cz", "rcccx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): True,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): True,
        (None, 3): False,
    },
    ("cz", "ecr"): False,
    ("cz", "csdg"): True,
    ("cz", "swap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("cz", "iswap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("cz", "rxx"): False,
    ("cz", "ryy"): False,
    ("cz", "rzz"): True,
    ("cz", "rzx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): False,
    },
    ("ccz", "c3sx"): {
        (0, 1, 2): True,
        (0, 1, 3): False,
        (0, 1, None): True,
        (0, 2, 1): True,
        (0, 2, 3): False,
        (0, 2, None): True,
        (0, 3, 1): False,
        (0, 3, 2): False,
        (0, 3, None): False,
        (0, None, 1): True,
        (0, None, 2): True,
        (0, None, 3): False,
        (0, None, None): True,
        (1, 0, 2): True,
        (1, 0, 3): False,
        (1, 0, None): True,
        (1, 2, 0): True,
        (1, 2, 3): False,
        (1, 2, None): True,
        (1, 3, 0): False,
        (1, 3, 2): False,
        (1, 3, None): False,
        (1, None, 0): True,
        (1, None, 2): True,
        (1, None, 3): False,
        (1, None, None): True,
        (2, 0, 1): True,
        (2, 0, 3): False,
        (2, 0, None): True,
        (2, 1, 0): True,
        (2, 1, 3): False,
        (2, 1, None): True,
        (2, 3, 0): False,
        (2, 3, 1): False,
        (2, 3, None): False,
        (2, None, 0): True,
        (2, None, 1): True,
        (2, None, 3): False,
        (2, None, None): True,
        (3, 0, 1): False,
        (3, 0, 2): False,
        (3, 0, None): False,
        (3, 1, 0): False,
        (3, 1, 2): False,
        (3, 1, None): False,
        (3, 2, 0): False,
        (3, 2, 1): False,
        (3, 2, None): False,
        (3, None, 0): False,
        (3, None, 1): False,
        (3, None, 2): False,
        (3, None, None): False,
        (None, 0, 1): True,
        (None, 0, 2): True,
        (None, 0, 3): False,
        (None, 0, None): True,
        (None, 1, 0): True,
        (None, 1, 2): True,
        (None, 1, 3): False,
        (None, 1, None): True,
        (None, 2, 0): True,
        (None, 2, 1): True,
        (None, 2, 3): False,
        (None, 2, None): True,
        (None, 3, 0): False,
        (None, 3, 1): False,
        (None, 3, 2): False,
        (None, 3, None): False,
        (None, None, 0): True,
        (None, None, 1): True,
        (None, None, 2): True,
        (None, None, 3): False,
    },
    ("ccz", "cswap"): {
        (0, 1, 2): True,
        (0, 1, None): False,
        (0, 2, 1): True,
        (0, 2, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, None): True,
        (1, 0, 2): True,
        (1, 0, None): False,
        (1, 2, 0): True,
        (1, 2, None): True,
        (1, None, 0): False,
        (1, None, 2): True,
        (1, None, None): False,
        (2, 0, 1): True,
        (2, 0, None): False,
        (2, 1, 0): True,
        (2, 1, None): True,
        (2, None, 0): False,
        (2, None, 1): True,
        (2, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): True,
        (None, 1, None): False,
        (None, 2, 0): False,
        (None, 2, 1): True,
        (None, 2, None): False,
        (None, None, 0): True,
        (None, None, 1): False,
        (None, None, 2): False,
    },
    ("ccz", "ccz"): True,
    ("ccz", "rccx"): {
        (0, 1, 2): False,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, None): False,
        (0, None, 1): True,
        (0, None, 2): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, None): False,
        (1, None, 0): True,
        (1, None, 2): False,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, None): False,
        (2, 1, 0): False,
        (2, 1, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, None): False,
        (None, 0, 1): True,
        (None, 0, 2): False,
        (None, 0, None): True,
        (None, 1, 0): True,
        (None, 1, 2): False,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, None): False,
        (None, None, 0): True,
        (None, None, 1): True,
        (None, None, 2): False,
    },
    ("ccz", "rcccx"): {
        (0, 1, 2): True,
        (0, 1, 3): False,
        (0, 1, None): True,
        (0, 2, 1): True,
        (0, 2, 3): False,
        (0, 2, None): True,
        (0, 3, 1): False,
        (0, 3, 2): False,
        (0, 3, None): False,
        (0, None, 1): True,
        (0, None, 2): True,
        (0, None, 3): False,
        (0, None, None): True,
        (1, 0, 2): True,
        (1, 0, 3): False,
        (1, 0, None): True,
        (1, 2, 0): True,
        (1, 2, 3): False,
        (1, 2, None): True,
        (1, 3, 0): False,
        (1, 3, 2): False,
        (1, 3, None): False,
        (1, None, 0): True,
        (1, None, 2): True,
        (1, None, 3): False,
        (1, None, None): True,
        (2, 0, 1): True,
        (2, 0, 3): False,
        (2, 0, None): True,
        (2, 1, 0): True,
        (2, 1, 3): False,
        (2, 1, None): True,
        (2, 3, 0): False,
        (2, 3, 1): False,
        (2, 3, None): False,
        (2, None, 0): True,
        (2, None, 1): True,
        (2, None, 3): False,
        (2, None, None): True,
        (3, 0, 1): False,
        (3, 0, 2): False,
        (3, 0, None): False,
        (3, 1, 0): False,
        (3, 1, 2): False,
        (3, 1, None): False,
        (3, 2, 0): False,
        (3, 2, 1): False,
        (3, 2, None): False,
        (3, None, 0): False,
        (3, None, 1): False,
        (3, None, 2): False,
        (3, None, None): False,
        (None, 0, 1): True,
        (None, 0, 2): True,
        (None, 0, 3): False,
        (None, 0, None): True,
        (None, 1, 0): True,
        (None, 1, 2): True,
        (None, 1, 3): False,
        (None, 1, None): True,
        (None, 2, 0): True,
        (None, 2, 1): True,
        (None, 2, 3): False,
        (None, 2, None): True,
        (None, 3, 0): False,
        (None, 3, 1): False,
        (None, 3, 2): False,
        (None, 3, None): False,
        (None, None, 0): True,
        (None, None, 1): True,
        (None, None, 2): True,
        (None, None, 3): False,
    },
    ("h", "id"): True,
    ("h", "sx"): False,
    ("h", "x"): False,
    ("h", "cx"): False,
    ("h", "c3sx"): False,
    ("h", "ccx"): False,
    ("h", "dcx"): False,
    ("h", "ch"): {
        (0,): False,
        (1,): True,
    },
    ("h", "cswap"): False,
    ("h", "csx"): False,
    ("h", "cy"): False,
    ("h", "cz"): False,
    ("h", "ccz"): False,
    ("h", "h"): True,
    ("h", "rccx"): False,
    ("h", "rcccx"): False,
    ("h", "ecr"): False,
    ("h", "s"): False,
    ("h", "sdg"): False,
    ("h", "cs"): False,
    ("h", "csdg"): False,
    ("h", "swap"): False,
    ("h", "iswap"): False,
    ("h", "sxdg"): False,
    ("h", "t"): False,
    ("h", "tdg"): False,
    ("h", "y"): False,
    ("h", "z"): False,
    ("h", "rxx"): False,
    ("h", "ryy"): False,
    ("h", "rzz"): False,
    ("h", "rzx"): False,
    ("rccx", "c3sx"): {
        (0, 1, 2): False,
        (0, 1, 3): False,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, 3): False,
        (0, 2, None): True,
        (0, 3, 1): False,
        (0, 3, 2): False,
        (0, 3, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, 3): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, 3): False,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, 3): False,
        (1, 2, None): True,
        (1, 3, 0): False,
        (1, 3, 2): False,
        (1, 3, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, 3): False,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, 3): False,
        (2, 0, None): True,
        (2, 1, 0): False,
        (2, 1, 3): False,
        (2, 1, None): True,
        (2, 3, 0): False,
        (2, 3, 1): False,
        (2, 3, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, 3): False,
        (2, None, None): True,
        (3, 0, 1): False,
        (3, 0, 2): False,
        (3, 0, None): False,
        (3, 1, 0): False,
        (3, 1, 2): False,
        (3, 1, None): False,
        (3, 2, 0): False,
        (3, 2, 1): False,
        (3, 2, None): False,
        (3, None, 0): False,
        (3, None, 1): False,
        (3, None, 2): False,
        (3, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, 3): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): False,
        (None, 1, 3): False,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, 3): False,
        (None, 2, None): True,
        (None, 3, 0): False,
        (None, 3, 1): False,
        (None, 3, 2): False,
        (None, 3, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
        (None, None, 3): False,
    },
    ("rccx", "cswap"): {
        (0, 1, 2): False,
        (0, 1, None): False,
        (0, 2, 1): False,
        (0, 2, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, None): False,
        (1, 2, 0): False,
        (1, 2, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, None): False,
        (2, 0, 1): False,
        (2, 0, None): False,
        (2, 1, 0): False,
        (2, 1, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): False,
        (None, 1, None): False,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
    },
    ("rccx", "rccx"): {
        (0, 1, 2): True,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, None): True,
        (1, 0, 2): True,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, None): False,
        (2, 1, 0): False,
        (2, 1, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): True,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
    },
    ("rccx", "rcccx"): {
        (0, 1, 2): False,
        (0, 1, 3): False,
        (0, 1, None): True,
        (0, 2, 1): False,
        (0, 2, 3): True,
        (0, 2, None): True,
        (0, 3, 1): False,
        (0, 3, 2): False,
        (0, 3, None): False,
        (0, None, 1): False,
        (0, None, 2): False,
        (0, None, 3): False,
        (0, None, None): True,
        (1, 0, 2): False,
        (1, 0, 3): False,
        (1, 0, None): True,
        (1, 2, 0): False,
        (1, 2, 3): True,
        (1, 2, None): True,
        (1, 3, 0): False,
        (1, 3, 2): False,
        (1, 3, None): False,
        (1, None, 0): False,
        (1, None, 2): False,
        (1, None, 3): False,
        (1, None, None): True,
        (2, 0, 1): False,
        (2, 0, 3): True,
        (2, 0, None): True,
        (2, 1, 0): False,
        (2, 1, 3): True,
        (2, 1, None): True,
        (2, 3, 0): False,
        (2, 3, 1): False,
        (2, 3, None): False,
        (2, None, 0): False,
        (2, None, 1): False,
        (2, None, 3): False,
        (2, None, None): True,
        (3, 0, 1): False,
        (3, 0, 2): False,
        (3, 0, None): False,
        (3, 1, 0): False,
        (3, 1, 2): False,
        (3, 1, None): False,
        (3, 2, 0): False,
        (3, 2, 1): False,
        (3, 2, None): False,
        (3, None, 0): False,
        (3, None, 1): False,
        (3, None, 2): False,
        (3, None, None): False,
        (None, 0, 1): False,
        (None, 0, 2): False,
        (None, 0, 3): False,
        (None, 0, None): True,
        (None, 1, 0): False,
        (None, 1, 2): False,
        (None, 1, 3): False,
        (None, 1, None): True,
        (None, 2, 0): False,
        (None, 2, 1): False,
        (None, 2, 3): True,
        (None, 2, None): True,
        (None, 3, 0): False,
        (None, 3, 1): False,
        (None, 3, 2): False,
        (None, 3, None): False,
        (None, None, 0): False,
        (None, None, 1): False,
        (None, None, 2): False,
        (None, None, 3): False,
    },
    ("rcccx", "rcccx"): {
        (0, 1, 2, 3): True,
        (0, 1, 2, None): True,
        (0, 1, 3, 2): False,
        (0, 1, 3, None): False,
        (0, 1, None, 2): False,
        (0, 1, None, 3): False,
        (0, 1, None, None): True,
        (0, 2, 1, 3): True,
        (0, 2, 1, None): True,
        (0, 2, 3, 1): False,
        (0, 2, 3, None): False,
        (0, 2, None, 1): False,
        (0, 2, None, 3): False,
        (0, 2, None, None): True,
        (0, 3, 1, 2): False,
        (0, 3, 1, None): False,
        (0, 3, 2, 1): False,
        (0, 3, 2, None): False,
        (0, 3, None, 1): False,
        (0, 3, None, 2): False,
        (0, 3, None, None): False,
        (0, None, 1, 2): False,
        (0, None, 1, 3): False,
        (0, None, 1, None): True,
        (0, None, 2, 1): False,
        (0, None, 2, 3): True,
        (0, None, 2, None): True,
        (0, None, 3, 1): False,
        (0, None, 3, 2): False,
        (0, None, 3, None): False,
        (0, None, None, 1): False,
        (0, None, None, 2): False,
        (0, None, None, 3): False,
        (0, None, None, None): True,
        (1, 0, 2, 3): True,
        (1, 0, 2, None): True,
        (1, 0, 3, 2): False,
        (1, 0, 3, None): False,
        (1, 0, None, 2): False,
        (1, 0, None, 3): False,
        (1, 0, None, None): True,
        (1, 2, 0, 3): True,
        (1, 2, 0, None): True,
        (1, 2, 3, 0): False,
        (1, 2, 3, None): False,
        (1, 2, None, 0): False,
        (1, 2, None, 3): False,
        (1, 2, None, None): True,
        (1, 3, 0, 2): False,
        (1, 3, 0, None): False,
        (1, 3, 2, 0): False,
        (1, 3, 2, None): False,
        (1, 3, None, 0): False,
        (1, 3, None, 2): False,
        (1, 3, None, None): False,
        (1, None, 0, 2): False,
        (1, None, 0, 3): False,
        (1, None, 0, None): True,
        (1, None, 2, 0): False,
        (1, None, 2, 3): True,
        (1, None, 2, None): True,
        (1, None, 3, 0): False,
        (1, None, 3, 2): False,
        (1, None, 3, None): False,
        (1, None, None, 0): False,
        (1, None, None, 2): False,
        (1, None, None, 3): False,
        (1, None, None, None): True,
        (2, 0, 1, 3): True,
        (2, 0, 1, None): True,
        (2, 0, 3, 1): False,
        (2, 0, 3, None): False,
        (2, 0, None, 1): False,
        (2, 0, None, 3): False,
        (2, 0, None, None): True,
        (2, 1, 0, 3): True,
        (2, 1, 0, None): True,
        (2, 1, 3, 0): False,
        (2, 1, 3, None): False,
        (2, 1, None, 0): False,
        (2, 1, None, 3): False,
        (2, 1, None, None): True,
        (2, 3, 0, 1): False,
        (2, 3, 0, None): False,
        (2, 3, 1, 0): False,
        (2, 3, 1, None): False,
        (2, 3, None, 0): False,
        (2, 3, None, 1): False,
        (2, 3, None, None): False,
        (2, None, 0, 1): False,
        (2, None, 0, 3): True,
        (2, None, 0, None): True,
        (2, None, 1, 0): False,
        (2, None, 1, 3): True,
        (2, None, 1, None): True,
        (2, None, 3, 0): False,
        (2, None, 3, 1): False,
        (2, None, 3, None): False,
        (2, None, None, 0): False,
        (2, None, None, 1): False,
        (2, None, None, 3): False,
        (2, None, None, None): True,
        (3, 0, 1, 2): False,
        (3, 0, 1, None): False,
        (3, 0, 2, 1): False,
        (3, 0, 2, None): False,
        (3, 0, None, 1): False,
        (3, 0, None, 2): False,
        (3, 0, None, None): False,
        (3, 1, 0, 2): False,
        (3, 1, 0, None): False,
        (3, 1, 2, 0): False,
        (3, 1, 2, None): False,
        (3, 1, None, 0): False,
        (3, 1, None, 2): False,
        (3, 1, None, None): False,
        (3, 2, 0, 1): False,
        (3, 2, 0, None): False,
        (3, 2, 1, 0): False,
        (3, 2, 1, None): False,
        (3, 2, None, 0): False,
        (3, 2, None, 1): False,
        (3, 2, None, None): False,
        (3, None, 0, 1): False,
        (3, None, 0, 2): False,
        (3, None, 0, None): False,
        (3, None, 1, 0): False,
        (3, None, 1, 2): False,
        (3, None, 1, None): False,
        (3, None, 2, 0): False,
        (3, None, 2, 1): False,
        (3, None, 2, None): False,
        (3, None, None, 0): False,
        (3, None, None, 1): False,
        (3, None, None, 2): False,
        (3, None, None, None): False,
        (None, 0, 1, 2): False,
        (None, 0, 1, 3): False,
        (None, 0, 1, None): True,
        (None, 0, 2, 1): False,
        (None, 0, 2, 3): True,
        (None, 0, 2, None): True,
        (None, 0, 3, 1): False,
        (None, 0, 3, 2): False,
        (None, 0, 3, None): False,
        (None, 0, None, 1): False,
        (None, 0, None, 2): False,
        (None, 0, None, 3): False,
        (None, 0, None, None): True,
        (None, 1, 0, 2): False,
        (None, 1, 0, 3): False,
        (None, 1, 0, None): True,
        (None, 1, 2, 0): False,
        (None, 1, 2, 3): True,
        (None, 1, 2, None): True,
        (None, 1, 3, 0): False,
        (None, 1, 3, 2): False,
        (None, 1, 3, None): False,
        (None, 1, None, 0): False,
        (None, 1, None, 2): False,
        (None, 1, None, 3): False,
        (None, 1, None, None): True,
        (None, 2, 0, 1): False,
        (None, 2, 0, 3): True,
        (None, 2, 0, None): True,
        (None, 2, 1, 0): False,
        (None, 2, 1, 3): True,
        (None, 2, 1, None): True,
        (None, 2, 3, 0): False,
        (None, 2, 3, 1): False,
        (None, 2, 3, None): False,
        (None, 2, None, 0): False,
        (None, 2, None, 1): False,
        (None, 2, None, 3): False,
        (None, 2, None, None): True,
        (None, 3, 0, 1): False,
        (None, 3, 0, 2): False,
        (None, 3, 0, None): False,
        (None, 3, 1, 0): False,
        (None, 3, 1, 2): False,
        (None, 3, 1, None): False,
        (None, 3, 2, 0): False,
        (None, 3, 2, 1): False,
        (None, 3, 2, None): False,
        (None, 3, None, 0): False,
        (None, 3, None, 1): False,
        (None, 3, None, 2): False,
        (None, 3, None, None): False,
        (None, None, 0, 1): False,
        (None, None, 0, 2): False,
        (None, None, 0, 3): False,
        (None, None, 0, None): True,
        (None, None, 1, 0): False,
        (None, None, 1, 2): False,
        (None, None, 1, 3): False,
        (None, None, 1, None): True,
        (None, None, 2, 0): False,
        (None, None, 2, 1): False,
        (None, None, 2, 3): True,
        (None, None, 2, None): True,
        (None, None, 3, 0): False,
        (None, None, 3, 1): False,
        (None, None, 3, 2): False,
        (None, None, 3, None): False,
        (None, None, None, 0): False,
        (None, None, None, 1): False,
        (None, None, None, 2): False,
        (None, None, None, 3): False,
    },
    ("ecr", "c3sx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): False,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): False,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): True,
    },
    ("ecr", "ccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): False,
        (1, 0): False,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): True,
    },
    ("ecr", "cswap"): False,
    ("ecr", "ccz"): False,
    ("ecr", "rccx"): False,
    ("ecr", "rcccx"): False,
    ("ecr", "ecr"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("ecr", "csdg"): False,
    ("ecr", "swap"): False,
    ("ecr", "iswap"): False,
    ("ecr", "rxx"): {
        (0, 1): False,
        (0, None): False,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): True,
    },
    ("ecr", "ryy"): False,
    ("ecr", "rzz"): False,
    ("ecr", "rzx"): {
        (0, 1): False,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
    ("s", "id"): True,
    ("s", "sx"): False,
    ("s", "x"): False,
    ("s", "cx"): {
        (0,): True,
        (1,): False,
    },
    ("s", "c3sx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("s", "ccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("s", "dcx"): False,
    ("s", "ch"): {
        (0,): True,
        (1,): False,
    },
    ("s", "cswap"): {
        (0,): True,
        (1,): False,
        (2,): False,
    },
    ("s", "csx"): {
        (0,): True,
        (1,): False,
    },
    ("s", "cy"): {
        (0,): True,
        (1,): False,
    },
    ("s", "cz"): True,
    ("s", "ccz"): True,
    ("s", "rccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("s", "rcccx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("s", "ecr"): False,
    ("s", "s"): True,
    ("s", "sdg"): True,
    ("s", "cs"): True,
    ("s", "csdg"): True,
    ("s", "swap"): False,
    ("s", "iswap"): False,
    ("s", "sxdg"): False,
    ("s", "t"): True,
    ("s", "tdg"): True,
    ("s", "y"): False,
    ("s", "z"): True,
    ("s", "rxx"): False,
    ("s", "ryy"): False,
    ("s", "rzz"): True,
    ("s", "rzx"): {
        (0,): True,
        (1,): False,
    },
    ("sdg", "cx"): {
        (0,): True,
        (1,): False,
    },
    ("sdg", "c3sx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("sdg", "ccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("sdg", "dcx"): False,
    ("sdg", "ch"): {
        (0,): True,
        (1,): False,
    },
    ("sdg", "cswap"): {
        (0,): True,
        (1,): False,
        (2,): False,
    },
    ("sdg", "csx"): {
        (0,): True,
        (1,): False,
    },
    ("sdg", "cy"): {
        (0,): True,
        (1,): False,
    },
    ("sdg", "cz"): True,
    ("sdg", "ccz"): True,
    ("sdg", "rccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("sdg", "rcccx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("sdg", "ecr"): False,
    ("sdg", "sdg"): True,
    ("sdg", "cs"): True,
    ("sdg", "csdg"): True,
    ("sdg", "swap"): False,
    ("sdg", "iswap"): False,
    ("sdg", "sxdg"): False,
    ("sdg", "tdg"): True,
    ("sdg", "rxx"): False,
    ("sdg", "ryy"): False,
    ("sdg", "rzz"): True,
    ("sdg", "rzx"): {
        (0,): True,
        (1,): False,
    },
    ("cs", "cx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): False,
    },
    ("cs", "c3sx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): True,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): True,
        (None, 3): False,
    },
    ("cs", "ccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): False,
    },
    ("cs", "dcx"): False,
    ("cs", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): True,
        (1, None): False,
        (2, 0): False,
        (2, 1): True,
        (2, None): False,
        (None, 0): True,
        (None, 1): False,
        (None, 2): False,
    },
    ("cs", "csx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): False,
    },
    ("cs", "cy"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): False,
    },
    ("cs", "cz"): True,
    ("cs", "ccz"): True,
    ("cs", "rccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): False,
    },
    ("cs", "rcccx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): True,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): True,
        (None, 3): False,
    },
    ("cs", "ecr"): False,
    ("cs", "cs"): True,
    ("cs", "csdg"): True,
    ("cs", "swap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("cs", "iswap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("cs", "rxx"): False,
    ("cs", "ryy"): False,
    ("cs", "rzz"): True,
    ("cs", "rzx"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): False,
        (None, 0): True,
        (None, 1): False,
    },
    ("csdg", "c3sx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): True,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): True,
        (None, 3): False,
    },
    ("csdg", "ccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): False,
    },
    ("csdg", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): True,
        (1, None): False,
        (2, 0): False,
        (2, 1): True,
        (2, None): False,
        (None, 0): True,
        (None, 1): False,
        (None, 2): False,
    },
    ("csdg", "ccz"): True,
    ("csdg", "rccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): False,
    },
    ("csdg", "rcccx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): True,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): True,
        (None, 3): False,
    },
    ("csdg", "csdg"): True,
    ("csdg", "swap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("csdg", "iswap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("swap", "c3sx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): False,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): False,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): False,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("swap", "ccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): False,
        (1, 0): True,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("swap", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): False,
        (1, 0): False,
        (1, 2): True,
        (1, None): False,
        (2, 0): False,
        (2, 1): True,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("swap", "ccz"): {
        (0, 1): True,
        (0, 2): True,
        (0, None): False,
        (1, 0): True,
        (1, 2): True,
        (1, None): False,
        (2, 0): True,
        (2, 1): True,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("swap", "rccx"): False,
    ("swap", "rcccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, 3): False,
        (0, None): False,
        (1, 0): True,
        (1, 2): False,
        (1, 3): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): False,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("swap", "swap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("swap", "iswap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("iswap", "c3sx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): False,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): False,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): False,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("iswap", "ccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): False,
        (1, 0): True,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("iswap", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): False,
        (1, 0): False,
        (1, 2): True,
        (1, None): False,
        (2, 0): False,
        (2, 1): True,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("iswap", "ccz"): {
        (0, 1): True,
        (0, 2): True,
        (0, None): False,
        (1, 0): True,
        (1, 2): True,
        (1, None): False,
        (2, 0): True,
        (2, 1): True,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("iswap", "rccx"): False,
    ("iswap", "rcccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, 3): False,
        (0, None): False,
        (1, 0): True,
        (1, 2): False,
        (1, 3): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): False,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("iswap", "iswap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("sxdg", "cx"): {
        (0,): False,
        (1,): True,
    },
    ("sxdg", "c3sx"): {
        (0,): False,
        (1,): False,
        (2,): False,
        (3,): True,
    },
    ("sxdg", "ccx"): {
        (0,): False,
        (1,): False,
        (2,): True,
    },
    ("sxdg", "dcx"): False,
    ("sxdg", "ch"): False,
    ("sxdg", "cswap"): False,
    ("sxdg", "csx"): {
        (0,): False,
        (1,): True,
    },
    ("sxdg", "cy"): False,
    ("sxdg", "cz"): False,
    ("sxdg", "ccz"): False,
    ("sxdg", "rccx"): False,
    ("sxdg", "rcccx"): False,
    ("sxdg", "ecr"): {
        (0,): False,
        (1,): True,
    },
    ("sxdg", "cs"): False,
    ("sxdg", "csdg"): False,
    ("sxdg", "swap"): False,
    ("sxdg", "iswap"): False,
    ("sxdg", "sxdg"): True,
    ("sxdg", "rxx"): True,
    ("sxdg", "ryy"): False,
    ("sxdg", "rzz"): False,
    ("sxdg", "rzx"): {
        (0,): False,
        (1,): True,
    },
    ("t", "id"): True,
    ("t", "sx"): False,
    ("t", "x"): False,
    ("t", "cx"): {
        (0,): True,
        (1,): False,
    },
    ("t", "c3sx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("t", "ccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("t", "dcx"): False,
    ("t", "ch"): {
        (0,): True,
        (1,): False,
    },
    ("t", "cswap"): {
        (0,): True,
        (1,): False,
        (2,): False,
    },
    ("t", "csx"): {
        (0,): True,
        (1,): False,
    },
    ("t", "cy"): {
        (0,): True,
        (1,): False,
    },
    ("t", "cz"): True,
    ("t", "ccz"): True,
    ("t", "rccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("t", "rcccx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("t", "ecr"): False,
    ("t", "sdg"): True,
    ("t", "cs"): True,
    ("t", "csdg"): True,
    ("t", "swap"): False,
    ("t", "iswap"): False,
    ("t", "sxdg"): False,
    ("t", "t"): True,
    ("t", "tdg"): True,
    ("t", "y"): False,
    ("t", "z"): True,
    ("t", "rxx"): False,
    ("t", "ryy"): False,
    ("t", "rzz"): True,
    ("t", "rzx"): {
        (0,): True,
        (1,): False,
    },
    ("tdg", "cx"): {
        (0,): True,
        (1,): False,
    },
    ("tdg", "c3sx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("tdg", "ccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("tdg", "dcx"): False,
    ("tdg", "ch"): {
        (0,): True,
        (1,): False,
    },
    ("tdg", "cswap"): {
        (0,): True,
        (1,): False,
        (2,): False,
    },
    ("tdg", "csx"): {
        (0,): True,
        (1,): False,
    },
    ("tdg", "cy"): {
        (0,): True,
        (1,): False,
    },
    ("tdg", "cz"): True,
    ("tdg", "ccz"): True,
    ("tdg", "rccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("tdg", "rcccx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("tdg", "ecr"): False,
    ("tdg", "cs"): True,
    ("tdg", "csdg"): True,
    ("tdg", "swap"): False,
    ("tdg", "iswap"): False,
    ("tdg", "sxdg"): False,
    ("tdg", "tdg"): True,
    ("tdg", "rxx"): False,
    ("tdg", "ryy"): False,
    ("tdg", "rzz"): True,
    ("tdg", "rzx"): {
        (0,): True,
        (1,): False,
    },
    ("y", "id"): True,
    ("y", "sx"): False,
    ("y", "cx"): False,
    ("y", "c3sx"): False,
    ("y", "ccx"): False,
    ("y", "dcx"): False,
    ("y", "ch"): False,
    ("y", "cswap"): False,
    ("y", "csx"): False,
    ("y", "cy"): {
        (0,): False,
        (1,): True,
    },
    ("y", "cz"): False,
    ("y", "ccz"): False,
    ("y", "rccx"): False,
    ("y", "rcccx"): False,
    ("y", "ecr"): False,
    ("y", "sdg"): False,
    ("y", "cs"): False,
    ("y", "csdg"): False,
    ("y", "swap"): False,
    ("y", "iswap"): False,
    ("y", "sxdg"): False,
    ("y", "tdg"): False,
    ("y", "y"): True,
    ("y", "z"): False,
    ("y", "rxx"): False,
    ("y", "ryy"): True,
    ("y", "rzz"): False,
    ("y", "rzx"): False,
    ("z", "id"): True,
    ("z", "sx"): False,
    ("z", "cx"): {
        (0,): True,
        (1,): False,
    },
    ("z", "c3sx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("z", "ccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("z", "dcx"): False,
    ("z", "ch"): {
        (0,): True,
        (1,): False,
    },
    ("z", "cswap"): {
        (0,): True,
        (1,): False,
        (2,): False,
    },
    ("z", "csx"): {
        (0,): True,
        (1,): False,
    },
    ("z", "cy"): {
        (0,): True,
        (1,): False,
    },
    ("z", "cz"): True,
    ("z", "ccz"): True,
    ("z", "rccx"): {
        (0,): True,
        (1,): True,
        (2,): False,
    },
    ("z", "rcccx"): {
        (0,): True,
        (1,): True,
        (2,): True,
        (3,): False,
    },
    ("z", "ecr"): False,
    ("z", "sdg"): True,
    ("z", "cs"): True,
    ("z", "csdg"): True,
    ("z", "swap"): False,
    ("z", "iswap"): False,
    ("z", "sxdg"): False,
    ("z", "tdg"): True,
    ("z", "z"): True,
    ("z", "rxx"): False,
    ("z", "ryy"): False,
    ("z", "rzz"): True,
    ("z", "rzx"): {
        (0,): True,
        (1,): False,
    },
    ("rxx", "c3sx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): False,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): False,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): True,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): True,
    },
    ("rxx", "ccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): False,
        (1, 0): False,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): True,
        (None, 0): False,
        (None, 1): False,
        (None, 2): True,
    },
    ("rxx", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): False,
        (1, 0): False,
        (1, 2): True,
        (1, None): False,
        (2, 0): False,
        (2, 1): True,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("rxx", "ccz"): False,
    ("rxx", "rccx"): False,
    ("rxx", "rcccx"): False,
    ("rxx", "csdg"): False,
    ("rxx", "swap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("rxx", "iswap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("rxx", "rxx"): True,
    ("rxx", "ryy"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("rxx", "rzz"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("rxx", "rzx"): {
        (0, 1): False,
        (0, None): False,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): True,
    },
    ("ryy", "c3sx"): False,
    ("ryy", "ccx"): False,
    ("ryy", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): False,
        (1, 0): False,
        (1, 2): True,
        (1, None): False,
        (2, 0): False,
        (2, 1): True,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("ryy", "ccz"): False,
    ("ryy", "rccx"): False,
    ("ryy", "rcccx"): False,
    ("ryy", "csdg"): False,
    ("ryy", "swap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("ryy", "iswap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("ryy", "ryy"): True,
    ("ryy", "rzz"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("ryy", "rzx"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("rzz", "c3sx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): True,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): True,
        (None, 3): False,
    },
    ("rzz", "ccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): False,
    },
    ("rzz", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): True,
        (1, None): False,
        (2, 0): False,
        (2, 1): True,
        (2, None): False,
        (None, 0): True,
        (None, 1): False,
        (None, 2): False,
    },
    ("rzz", "ccz"): True,
    ("rzz", "rccx"): {
        (0, 1): True,
        (0, 2): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): False,
    },
    ("rzz", "rcccx"): {
        (0, 1): True,
        (0, 2): True,
        (0, 3): False,
        (0, None): True,
        (1, 0): True,
        (1, 2): True,
        (1, 3): False,
        (1, None): True,
        (2, 0): True,
        (2, 1): True,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): True,
        (None, 1): True,
        (None, 2): True,
        (None, 3): False,
    },
    ("rzz", "csdg"): True,
    ("rzz", "swap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("rzz", "iswap"): {
        (0, 1): True,
        (0, None): False,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): False,
    },
    ("rzz", "rzz"): True,
    ("rzx", "c3sx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): True,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): True,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): True,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): True,
    },
    ("rzx", "ccx"): {
        (0, 1): False,
        (0, 2): True,
        (0, None): True,
        (1, 0): False,
        (1, 2): True,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): True,
    },
    ("rzx", "cswap"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): False,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("rzx", "ccz"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): True,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("rzx", "rccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
    },
    ("rzx", "rcccx"): {
        (0, 1): False,
        (0, 2): False,
        (0, 3): False,
        (0, None): True,
        (1, 0): False,
        (1, 2): False,
        (1, 3): False,
        (1, None): True,
        (2, 0): False,
        (2, 1): False,
        (2, 3): False,
        (2, None): True,
        (3, 0): False,
        (3, 1): False,
        (3, 2): False,
        (3, None): False,
        (None, 0): False,
        (None, 1): False,
        (None, 2): False,
        (None, 3): False,
    },
    ("rzx", "csdg"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("rzx", "swap"): False,
    ("rzx", "iswap"): False,
    ("rzx", "rzz"): {
        (0, 1): False,
        (0, None): True,
        (1, 0): False,
        (1, None): True,
        (None, 0): False,
        (None, 1): False,
    },
    ("rzx", "rzx"): {
        (0, 1): True,
        (0, None): True,
        (1, 0): True,
        (1, None): False,
        (None, 0): False,
        (None, 1): True,
    },
}
