/*
 * WMA 9/3/PRO compatible decoder
 * Copyright (c) 2007 Baptiste Coudurier, Benjamin Larsson, Ulion
 * Copyright (c) 2008 - 2009 Sascha Sommer
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * @brief tables for wmapro decoding
 */

#ifndef AVCODEC_WMAPRODATA_H
#define AVCODEC_WMAPRODATA_H

#include <stddef.h>
#include <stdint.h>

/**
 * @brief frequencies to divide the frequency spectrum into scale factor bands
 */
static const uint16_t critical_freq[] = {
     100,   200,    300,    400,    510,    630,    770,
     920,  1080,   1270,   1480,   1720,   2000,   2320,
    2700,  3150,   3700,   4400,   5300,   6400,   7700,
    9500, 12000,  15500,  20675,  28575,  41375,  63875,
};


/**
 * @name Huffman tables for DPCM-coded scale factors
 * @{
 */
#define HUFF_SCALE_SIZE    121
#define HUFF_SCALE_MAXBITS  19
static const uint8_t scale_table[HUFF_SCALE_SIZE][2] = {
    {  58,  5 }, {  64,  6 }, {  66,  7 }, {  65,  7 }, {  62,  5 },
    {  63,  6 }, {  68,  9 }, {  69, 10 }, {  54, 15 }, {  19, 19 },
    {  20, 19 }, {  21, 19 }, {  22, 19 }, {  23, 19 }, {  24, 19 },
    {  25, 19 }, {  26, 19 }, {  27, 19 }, {  28, 19 }, {  29, 19 },
    {  30, 19 }, {  31, 19 }, {  32, 19 }, {  33, 19 }, {  34, 19 },
    {  17, 19 }, {  36, 19 }, {  37, 19 }, {  38, 19 }, {  39, 19 },
    {  40, 19 }, {  41, 19 }, {  42, 19 }, {  43, 19 }, {  44, 19 },
    {  45, 19 }, {  46, 19 }, {  47, 19 }, {  48, 19 }, {  49, 19 },
    {  50, 19 }, {  51, 19 }, {  52, 19 }, {  15, 19 }, {  16, 19 },
    {  14, 19 }, {  13, 19 }, {  12, 19 }, {  11, 19 }, {  10, 19 },
    {   0, 19 }, {   9, 19 }, {   8, 19 }, {   7, 19 }, {   6, 19 },
    {   5, 19 }, {   4, 19 }, {  55, 13 }, {  70, 13 }, {   3, 19 },
    {   2, 19 }, {   1, 19 }, {  35, 19 }, {  71, 19 }, {  72, 19 },
    {  73, 19 }, {  74, 19 }, {  75, 19 }, {  76, 19 }, {  77, 19 },
    {  78, 19 }, {  79, 19 }, {  80, 19 }, {  81, 19 }, {  82, 19 },
    {  83, 19 }, {  84, 19 }, {  85, 19 }, {  86, 19 }, {  87, 19 },
    {  88, 19 }, {  89, 19 }, {  90, 19 }, {  91, 19 }, {  92, 19 },
    {  93, 19 }, {  94, 19 }, {  95, 19 }, {  96, 19 }, {  97, 19 },
    {  98, 19 }, {  99, 19 }, { 100, 19 }, { 101, 19 }, { 102, 19 },
    { 103, 19 }, { 104, 19 }, { 105, 19 }, { 106, 19 }, { 107, 19 },
    { 108, 19 }, { 109, 19 }, { 110, 19 }, { 111, 19 }, { 112, 19 },
    { 113, 19 }, { 114, 19 }, { 115, 19 }, { 116, 19 }, { 117, 19 },
    { 118, 19 }, { 119, 19 }, { 120, 19 }, {  18, 18 }, {  53, 16 },
    {  56, 11 }, {  57,  8 }, {  67,  7 }, {  61,  3 }, {  59,  2 },
    {  60,  1 },
};
/** @} */


/**
 * @name Huffman, run and level tables for runlevel-coded scale factors
 * @{
 */
#define HUFF_SCALE_RL_SIZE    120
#define HUFF_SCALE_RL_MAXBITS  21
static const uint8_t scale_rl_table[HUFF_SCALE_RL_SIZE][2] = {
    { 103,  7 }, {  80, 11 }, {  60, 11 }, {  18, 10 }, {  56, 10 },
    {  21, 12 }, {  90, 12 }, {  58, 11 }, {  27, 11 }, {  69, 12 },
    {  84, 15 }, {  48, 15 }, {  86, 14 }, {  47, 13 }, {  19, 10 },
    {  32,  9 }, {  78,  6 }, {   5,  5 }, {  28,  4 }, {  53,  5 },
    {   9,  7 }, {  31,  8 }, {  38,  8 }, {  10,  7 }, {  88, 11 },
    {  25, 12 }, { 105, 12 }, { 118, 11 }, {  23, 12 }, {  82, 14 },
    {  98, 16 }, { 110, 16 }, { 108, 15 }, {  93, 13 }, {  68, 10 },
    {  72, 12 }, {  97, 12 }, {  81, 12 }, {  42, 12 }, {  64,  8 },
    {   4,  4 }, {   1,  2 }, {   7,  6 }, {  14,  7 }, {   0,  9 },
    {  55,  9 }, {  61,  9 }, { 117, 10 }, {  24, 12 }, {  44, 12 },
    {  67, 12 }, {  70, 16 }, {  99, 18 }, {  96, 21 }, {  95, 21 },
    {   2, 21 }, {  77, 21 }, {  52, 21 }, { 111, 21 }, { 102, 20 },
    { 101, 17 }, {  46, 15 }, {  73, 15 }, { 109, 15 }, {  51, 14 },
    {  92, 14 }, {  30,  7 }, {  11,  7 }, {  66,  7 }, {  15,  8 },
    {  16,  8 }, { 116,  9 }, {  65,  9 }, {  57, 10 }, {  59, 10 },
    { 115,  9 }, {  12,  7 }, {  35,  9 }, {  17,  9 }, {  41,  9 },
    {  20, 11 }, {  91, 11 }, {  26, 12 }, {  75, 15 }, {  45, 15 },
    { 107, 14 }, {  83, 14 }, { 100, 15 }, {  89, 15 }, {  43, 11 },
    {  62,  9 }, {  37,  9 }, { 104,  8 }, {   6,  5 }, {  39,  8 },
    {  40,  9 }, {  34,  9 }, {  79,  7 }, {   8,  6 }, {  63,  6 },
    {  87, 12 }, {  94, 14 }, {  49, 14 }, {  50, 13 }, {  22, 11 },
    { 119, 10 }, {  33,  9 }, {  36,  9 }, { 113, 11 }, { 106, 12 },
    { 112, 13 }, {  71, 15 }, {  85, 15 }, {  74, 14 }, {  76, 10 },
    { 114,  7 }, {  29,  5 }, {  54,  6 }, {  13,  6 }, {   3,  2 },
};


static const uint8_t scale_rl_run[HUFF_SCALE_RL_SIZE] = {
     0,  0,  0,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9,
    10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24,  0,  1,  2,  3,
     4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22,
    23, 24,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16,
    17, 18, 19, 20, 21, 22, 23, 24,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
     0,  1,  0,  1,  0,  1,
};

static const uint8_t scale_rl_level[HUFF_SCALE_RL_SIZE] = {
     0,  0,  1,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
     2,  2,  2,  2,  2,  2,  2,  2,  2,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
     3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  4,  4,  4,  4,
     4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,
     4,  4,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,
     5,  5,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     7,  7,  8,  8,  9,  9,
};
/** @} */


/**
 * @name Huffman, run and level codes for runlevel-coded coefficients
 * @{
 */
#define HUFF_COEF0_SIZE    272
#define HUFF_COEF0_MAXBITS  21
static const uint8_t coef0_lens[HUFF_COEF0_SIZE] = {
     2,  9, 14, 14, 13, 12, 13, 14, 15, 15, 12, 10, 10, 10, 13, 14, 15, 15, 12,
    11, 13, 14, 14, 13, 15, 15, 14, 12, 12,  8, 10, 10, 15, 15, 14, 13, 14, 14,
    13, 15, 20, 20, 19, 21, 21, 20, 19, 17, 17, 18, 18, 15, 15, 13, 12, 14, 15,
    15, 14, 15, 15, 12, 11,  6,  7,  8,  9, 13, 13, 13, 14, 14, 11, 10,  7,  8,
    14, 14, 14, 14, 12, 13, 13, 12, 12, 12, 11,  9, 13, 14, 14, 12, 11, 11, 11,
     9,  8,  7, 14, 15, 15, 14, 14, 12, 13, 15, 16, 17, 17, 14, 12, 12, 12, 15,
    15, 14, 14, 14, 13, 13,  9,  9, 11, 11, 10,  7,  6, 13, 15, 15, 14, 14, 14,
    13, 14, 15, 15, 13, 14, 14, 14, 14, 10,  9, 10, 10, 11, 11, 10,  8,  9, 13,
    14, 14, 12, 11, 14, 15, 15, 13, 12, 14, 14, 14, 14, 13, 14, 14,  3,  5,  8,
    10, 10, 15, 15, 14, 14, 16, 16, 15, 12, 11, 11, 11,  7,  8,  8,  9, 12, 13,
    13, 12, 14, 15, 15, 13, 10, 11, 11, 13, 14, 14, 13, 14, 14, 11, 10, 13, 15,
    15, 14, 12, 11,  4,  6,  6,  8, 12, 12, 12, 13, 13, 12, 13, 13, 14, 14, 13,
    13, 13,  9,  7,  9, 11, 14, 14, 13, 14, 14, 13, 10,  8,  7,  5,  9, 12, 13,
    14, 15, 15, 12, 12, 10, 14, 14, 13, 12, 13, 14, 14, 12, 13, 13, 12, 12, 12,
     9,  7,  6,  3,  4,  4,
};

static const uint16_t coef0_syms[HUFF_COEF0_SIZE] = {
      2,  25, 111,  94,  69,  58,  87,  93, 136, 135,  59,  37,  34,  36,  82,
    182, 120, 138, 195,  45, 168, 216, 178,  86, 140, 219, 186, 162, 239,  18,
    156,  35, 127, 236, 109,  85, 180, 253,  88, 147, 268, 264, 256, 266, 270,
    262, 260, 248, 246, 252, 258, 137, 189, 230,  64, 179, 146, 208, 101, 118,
    238, 163,  46,   9, 153,   0,  26, 247, 169,  76, 202, 131, 194,  38,  13,
     19, 132, 106, 191,  97,  65, 198,  77,  62,  66, 164,  48,  27,  81, 183,
    102,  60,  47,  49, 159, 227,  20,  14, 112, 263, 144, 217, 104,  63,  79,
    209, 269, 250, 254, 203, 241, 196,  61, 220, 148, 124, 185, 100,  80,  78,
    193,  28,  50, 235,  41,   1,  10, 171, 226, 150, 103, 114, 115, 170, 105,
    211, 149, 249, 108, 188, 107, 255, 231, 155,  42,  40,  55, 160,  39,  21,
     29, 215, 234, 184, 228,  51, 116, 142, 145, 172, 165, 181, 130, 113, 117,
     89, 128, 204,   3,   7, 154, 157,  43, 141, 265, 133, 225, 271, 244, 221,
     74,  54,  56,  52,  15, 222,  22,  30,  83, 199, 173,  73, 123, 210, 143,
    175,  44,  53, 237, 174, 139, 134, 110, 218, 129, 161, 213, 177, 267, 151,
    125,  67, 223,   5,  11, 192,  23, 214, 243, 166, 200, 176,  68, 224, 187,
    257, 261, 232,  96, 251,  31,  16,  32,  57, 207, 121,  91, 126, 119,  99,
    158,  24, 212,   8,  33,  70,  92, 205, 240, 242,  75, 197, 233, 259, 190,
     98,  71, 201, 122, 206,  72,  90,  95,  84, 167, 245, 229,  17,  12,   4,
    152,   6,
};


#define HUFF_COEF1_SIZE    244
#define HUFF_COEF1_MAXBITS  22
static const uint8_t coef1_table[HUFF_COEF1_SIZE][2] = {
    {   2,  2 }, {   3,  3 }, { 102,  3 }, {   4,  4 }, { 148,  6 },
    { 134,  9 }, { 171, 10 }, {  18, 10 }, {  11,  8 }, { 159,  8 },
    {  14,  9 }, { 156, 14 }, { 235, 15 }, {  61, 15 }, {  38, 13 },
    { 153, 13 }, {  48, 14 }, {  49, 14 }, {  23, 11 }, { 203, 13 },
    { 208, 19 }, { 204, 19 }, { 129, 18 }, {  94, 17 }, {  87, 16 },
    {  62, 15 }, { 174, 15 }, { 147, 15 }, {  29, 12 }, { 191, 12 },
    {  64, 15 }, {  65, 15 }, { 146, 14 }, { 164, 13 }, { 142,  5 },
    { 132,  4 }, { 103,  5 }, { 154,  7 }, { 165,  9 }, { 181, 11 },
    { 109, 12 }, {  30, 12 }, {  86, 16 }, {  92, 16 }, { 239, 15 },
    { 138, 14 }, {  39, 13 }, {  50, 14 }, { 115, 15 }, { 238, 21 },
    { 228, 21 }, { 236, 21 }, { 222, 21 }, { 216, 20 }, { 226, 20 },
    { 196, 18 }, { 192, 17 }, { 120, 16 }, { 221, 14 }, {  51, 14 },
    {  24, 11 }, { 143,  8 }, {   7,  6 }, {   9,  7 }, { 152, 10 },
    { 136, 12 }, { 160, 12 }, { 241, 15 }, {  66, 15 }, { 168, 14 },
    { 219, 14 }, { 113, 14 }, { 193, 12 }, {  19, 10 }, { 173, 10 },
    { 105,  8 }, { 149,  9 }, {  15,  9 }, { 205, 13 }, { 207, 13 },
    { 125, 17 }, { 190, 17 }, { 182, 16 }, {  68, 15 }, {  70, 15 },
    {  67, 15 }, { 137, 13 }, {  31, 12 }, { 223, 14 }, { 116, 15 },
    { 210, 19 }, { 220, 19 }, { 198, 18 }, { 126, 17 }, {  88, 16 },
    {  41, 13 }, {  25, 11 }, {  40, 13 }, {  73, 15 }, { 243, 15 },
    {  53, 14 }, { 195, 12 }, { 183, 11 }, { 225, 14 }, {  52, 14 },
    {  71, 15 }, { 121, 16 }, {  89, 16 }, { 170, 14 }, {  55, 14 },
    {  69, 15 }, {  83, 15 }, { 209, 13 }, { 108, 11 }, {  32, 12 },
    {  54, 14 }, { 122, 16 }, { 184, 16 }, { 176, 15 }, {  42, 13 },
    {  12,  8 }, { 161,  8 }, {   6,  5 }, { 167,  9 }, { 106,  9 },
    {  20, 10 }, { 145, 12 }, { 111, 13 }, {  43, 13 }, {  26, 11 },
    { 175, 10 }, { 107, 10 }, {  34, 12 }, {  33, 12 }, { 197, 12 },
    {  74, 15 }, { 128, 17 }, { 232, 20 }, { 212, 20 }, { 224, 19 },
    { 202, 18 }, {  90, 16 }, {  57, 14 }, { 227, 14 }, {  97, 16 },
    {  93, 16 }, { 140, 15 }, { 185, 11 }, {  27, 11 }, {  16,  9 },
    { 158, 11 }, { 211, 13 }, {  56, 14 }, { 117, 15 }, {  72, 15 },
    { 166, 13 }, {  91, 16 }, {  95, 16 }, {  80, 15 }, { 101, 16 },
    { 194, 17 }, { 127, 17 }, {  82, 15 }, {  21, 10 }, { 144, 10 },
    { 177, 10 }, { 151,  6 }, {  10,  7 }, { 157,  7 }, {   8,  6 },
    {   5,  4 }, {  13,  8 }, {   0,  9 }, { 213, 13 }, {  46, 13 },
    { 199, 12 }, {  35, 12 }, { 162, 12 }, { 135, 10 }, { 169,  9 },
    {  45, 13 }, {  59, 14 }, { 114, 14 }, {  44, 13 }, { 188, 16 },
    { 186, 16 }, {  75, 15 }, {  79, 15 }, { 118, 15 }, { 187, 11 },
    { 112, 13 }, { 139, 14 }, { 178, 15 }, {  81, 15 }, { 110, 12 },
    {  28, 11 }, { 163,  8 }, { 133,  6 }, { 104,  6 }, {  17,  9 },
    {  22, 10 }, { 229, 14 }, { 172, 14 }, { 217, 13 }, { 201, 12 },
    {  36, 12 }, { 218, 20 }, { 242, 22 }, { 240, 22 }, { 234, 21 },
    { 230, 19 }, { 206, 18 }, { 200, 18 }, { 214, 18 }, { 130, 17 },
    { 131, 17 }, { 141, 15 }, {  84, 15 }, {  76, 15 }, { 215, 13 },
    {  58, 14 }, { 231, 14 }, { 233, 14 }, { 180, 15 }, {  77, 15 },
    {  37, 12 }, { 189, 11 }, { 179, 10 }, { 155, 10 }, {  47, 13 },
    {  96, 16 }, {  99, 16 }, { 119, 15 }, {  63, 14 }, { 237, 14 },
    {  78, 15 }, {  85, 15 }, {  60, 14 }, {  98, 16 }, { 100, 16 },
    { 124, 16 }, { 123, 16 }, { 150, 11 }, {   1,  7 },
};


static const uint16_t coef0_run[HUFF_COEF0_SIZE] = {
      0,   0,   0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,
     12,  13,  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,
     26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
     40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,
     54,  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,
     68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,
     82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,
     96,  97,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109,
    110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123,
    124, 125, 126, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137,
    138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149,   0,   1,
      2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,
     16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
     30,  31,  32,  33,  34,  35,  36,  37,  38,  39,   0,   1,   2,   3,
      4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16,  17,
     18,  19,   0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   0,   1,
      2,   3,   4,   0,   1,   0,   1,   0,   1,   0,   1,   0,   1,   0,
      1,   0,   1,   0,   1,   0,   1,   0,   1,   0,   1,   0,   1,   0,
      1,   0,   1,   0,   1,   0,   1,   0,   1,   0,   1,   0,   1,   0,
      1,   0,   1,   0,   1,   0,
};

static const float coef0_level[HUFF_COEF0_SIZE] = {
      0,   0,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
      1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   2,   2,
      2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,
      2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,
      2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   3,   3,   3,   3,
      3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,
      3,   3,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   5,   5,
      5,   5,   5,   6,   6,   7,   7,   8,   8,   9,   9,  10,  10,  11,
     11,  12,  12,  13,  13,  14,  14,  15,  15,  16,  16,  17,  17,  18,
     18,  19,  19,  20,  20,  21,  21,  22,  22,  23,  23,  24,  24,  25,
     25,  26,  26,  27,  27,  28,
};


static const uint16_t coef1_run[HUFF_COEF1_SIZE] = {
     0,  0,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33,
    34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69,
    70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87,
    88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,  0,  1,  2,  3,  4,  5,
     6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23,
    24, 25, 26, 27, 28, 29,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  0,  1,
     2,  3,  4,  5,  0,  1,  2,  0,  1,  2,  0,  1,  2,  0,  1,  0,  1,  0,
     1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,
     1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,
     1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,
     1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,
     1,  0,  1,  0,  1,  0,  1,  0,  0,  0,
};

static const float coef1_level[HUFF_COEF1_SIZE] = {
     0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  2,  2,  2,  2,  2,  2,
     2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
     2,  2,  2,  2,  2,  2,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  4,  4,
     4,  4,  4,  4,  5,  5,  5,  6,  6,  6,  7,  7,  7,  8,  8,  9,  9, 10,
    10, 11, 11, 12, 12, 13, 13, 14, 14, 15, 15, 16, 16, 17, 17, 18, 18, 19,
    19, 20, 20, 21, 21, 22, 22, 23, 23, 24, 24, 25, 25, 26, 26, 27, 27, 28,
    28, 29, 29, 30, 30, 31, 31, 32, 32, 33, 33, 34, 34, 35, 35, 36, 36, 37,
    37, 38, 38, 39, 39, 40, 40, 41, 41, 42, 42, 43, 43, 44, 44, 45, 45, 46,
    46, 47, 47, 48, 48, 49, 49, 50, 51, 52,
};
/** @} */


/**
 * @name Huffman and vector lookup tables for vector-coded coefficients
 * @{
 */
#define HUFF_VEC4_SIZE    127
#define HUFF_VEC4_MAXBITS  14
static const uint8_t vec4_lens[HUFF_VEC4_SIZE] = {
     1,  6,  8, 10, 10, 10, 10,  8,  8, 10, 10,  9,  8,  8,  9, 12, 12, 11,
    12, 12, 11,  9,  9,  8,  8,  9,  9,  8,  8,  9,  9, 12, 12, 12, 14, 14,
    13, 11, 11,  9,  8,  9,  9, 11, 11, 10,  9,  8,  6,  6,  6,  6,  6,  6,
    11, 11, 10, 11, 11, 10, 10, 11, 11,  9,  7,  6,  7,  7,  6,  6,  6,  5,
     7, 11, 11, 10,  9,  8,  6,  9,  9, 10, 10,  9,  8,  8,  6,  6,  6,  8,
     8,  9, 12, 12, 11, 10,  8,  8,  8, 10, 10, 10, 10,  9,  9,  8, 10, 11,
    11,  9,  8,  8,  8,  9,  9, 10, 11, 12, 12,  9,  9,  9,  8,  8,  7,  7,
     7,
};

/* The entry in the following table with symbol zero indicates
 * that four further entries are coded explicitly; all other
 * entries encode four numbers in the 0..15 range via
 * the four nibbles of (symbol - 1). */
static const uint16_t vec4_syms[HUFF_VEC4_SIZE] = {
        0,  4370,   275,  8195,  4146, 12545,  8225,   290,  4625,   515,
       20,  8706,  8210,  4355,  4131, 16385,  5121,  8961,   321,  1041,
       51,  4641,   546,  4610,   530,   513,  8451,  4385,  4130,    33,
     8211,     5,    66,  4161,  1281,    81,     6,   801,  8196,  8481,
     8449,  4611,   531,   561,   769, 12290,  8226,    19,  4097,     2,
     4369,   274,  4354,  4114, 12291, 16641, 12305,    49, 12321,   260,
     4100,   516,    21, 12546,  8466,  4353,  4371,  4626,   257,    18,
       17,     1,  4386,  8241,   771,  4865,  8705,  8194,  4098, 12561,
      276,    50,   785,  4116,  8209,  4099,   273,  4113,   258,   259,
     4609,    35,  1026,  1025, 16401,   305,    34,   529,   289,   770,
    12289,     4,  4145,  4356, 12306,  8193, 12801,   261, 16386,  4881,
        3,   514,  4129,   545,   306,    36,  4101,    65, 20481,   786,
     4401,  4866,  8721,   291,  8450,  8465,  4115,
};


#define HUFF_VEC2_SIZE    137
#define HUFF_VEC2_MAXBITS  12
/* The entry in the following table with symbol zero indicates
 * that two further entries are coded explicitly; all other
 * entries encode two numbers in the 0..15 range via
 * (symbol - 1) & 0xF and (symbol - 1) >> 4. */
static const uint8_t vec2_table[HUFF_VEC2_SIZE][2] = {
    {  19,  5 }, { 165, 10 }, { 211, 11 }, {  46, 11 }, {  75, 10 },
    { 177, 11 }, {  12, 11 }, {  86,  8 }, {  83,  7 }, {  38,  7 },
    { 133,  9 }, { 178, 10 }, {  28, 10 }, { 104,  9 }, {  73,  9 },
    {  35,  5 }, {  52,  6 }, { 113,  9 }, {   8,  9 }, { 101,  8 },
    {  69,  7 }, {   0,  3 }, {  71,  8 }, { 119,  9 }, {  91, 10 },
    { 179, 10 }, { 114,  8 }, { 166, 10 }, {  10, 10 }, {  44, 10 },
    { 145, 10 }, {  66,  6 }, {  21,  6 }, {  24,  8 }, { 146,  9 },
    {  26,  9 }, {  65,  7 }, {   5,  7 }, { 226, 11 }, { 225, 12 },
    {  15, 12 }, { 180, 10 }, { 147,  9 }, { 115,  8 }, {  40,  8 },
    {  89,  9 }, { 134,  9 }, {  84,  7 }, {  54,  7 }, {  42,  9 },
    {  60, 10 }, {  31, 11 }, { 193, 11 }, { 181, 10 }, {  76, 10 },
    { 148,  9 }, {  37,  6 }, {  67,  6 }, {  33,  6 }, {   3,  6 },
    {  17,  6 }, {   2,  6 }, { 102,  8 }, {  87,  8 }, { 116,  8 },
    {  56,  8 }, {  50,  5 }, {  20,  5 }, { 120,  9 }, {  58,  9 },
    {  29, 10 }, { 194, 10 }, { 135,  9 }, {  97,  8 }, {   7,  8 },
    { 105,  9 }, {  13, 11 }, { 241, 12 }, {  16, 12 }, {  45, 10 },
    { 149,  9 }, {  74,  9 }, {  98,  7 }, {  23,  7 }, {  85,  7 },
    {  70,  7 }, { 195, 10 }, { 161, 10 }, { 129,  9 }, {  72,  8 },
    {  51,  5 }, {  36,  5 }, { 117,  8 }, {  61, 10 }, {  11, 10 },
    { 162,  9 }, {   1,  7 }, {   4,  6 }, {  49,  6 }, {  68,  6 },
    {   9,  9 }, {  27,  9 }, { 130,  8 }, {  39,  7 }, {  53,  6 },
    {  99,  7 }, {  25,  8 }, { 150,  9 }, {  90,  9 }, { 103,  8 },
    { 163,  9 }, { 196, 10 }, { 210, 10 }, { 136,  9 }, { 121,  9 },
    {  41,  8 }, { 131,  8 }, {  43,  9 }, { 164,  9 }, { 118,  8 },
    {  88,  8 }, {  81,  7 }, {   6,  7 }, {  55,  7 }, {  59,  9 },
    {  30, 10 }, { 209, 11 }, {  14, 11 }, { 151,  9 }, { 106,  9 },
    {  82,  6 }, {  22,  6 }, { 100,  7 }, { 132,  8 }, {  57,  8 },
    {  18,  4 }, {  34,  4 },
};


#define HUFF_VEC1_SIZE    101
#define HUFF_VEC1_MAXBITS  11
static const uint8_t vec1_table[HUFF_VEC1_SIZE][2] = {
    {   7,  5 }, {  32,  8 }, {  59, 10 }, {  60, 10 }, {  83, 11 },
    {  82, 11 }, {  62, 10 }, {  33,  8 }, {  45,  9 }, {  61, 10 },
    {  84, 11 }, {  85, 11 }, {   1,  6 }, {  13,  5 }, {  19,  6 },
    {  25,  7 }, {  34,  8 }, {  46,  9 }, {  47,  9 }, {  14,  5 },
    {   6,  5 }, {  64, 10 }, {  87, 11 }, {  86, 11 }, {  63, 10 },
    {  88, 11 }, {  90, 11 }, {  35,  8 }, {  26,  7 }, {   0,  7 },
    {  48,  9 }, {  65, 10 }, {  66, 10 }, {  36,  8 }, {  15,  5 },
    {  20,  6 }, {  91, 11 }, {  89, 11 }, {  67, 10 }, {  49,  9 },
    {  50,  9 }, {  69, 10 }, {  92, 11 }, {  93, 11 }, {  27,  7 },
    {   5,  5 }, {  37,  8 }, {  68, 10 }, {  71, 10 }, {  51,  9 },
    {  52,  9 }, {  70, 10 }, {  94, 11 }, {  96, 11 }, {  38,  8 },
    {  21,  6 }, {  16,  5 }, {   4,  5 }, {  28,  7 }, {  53,  9 },
    {  95, 11 }, {  97, 11 }, {  73, 10 }, {  39,  8 }, {  29,  7 },
    {  72, 10 }, {  98, 11 }, {  99, 11 }, {  54,  9 }, {  40,  8 },
    {  22,  6 }, {  30,  7 }, {  55,  9 }, {  74, 10 }, {  76, 10 },
    {  56,  9 }, {  75, 10 }, {  77, 10 }, {  17,  5 }, {   3,  5 },
    {  23,  6 }, {  41,  8 }, {  57,  9 }, {  78, 10 }, {  79, 10 },
    {  31,  7 }, {  10,  4 }, {   9,  4 }, { 100,  5 }, {   2,  5 },
    {  11,  4 }, {   8,  4 }, {  18,  5 }, {  42,  8 }, {  58,  9 },
    {  80, 10 }, {  81, 10 }, {  43,  8 }, {  44,  8 }, {  24,  6 },
    {  12,  4 },
};
/** @} */


/**
 * @brief decorrelation matrix for multichannel streams
 **/
static const float default_decorrelation_matrices[] = {
    1.000000,  0.707031, -0.707031,  0.707031,  0.707031,  0.578125,  0.707031,
    0.410156,  0.578125, -0.707031,  0.410156,  0.578125,  0.000000, -0.816406,
    0.500000,  0.652344,  0.500000,  0.269531,  0.500000,  0.269531, -0.500000,
   -0.652344,  0.500000, -0.269531, -0.500000,  0.652344,  0.500000, -0.652344,
    0.500000, -0.269531,  0.445312,  0.601562,  0.511719,  0.371094,  0.195312,
    0.445312,  0.371094, -0.195312, -0.601562, -0.511719,  0.445312,  0.000000,
   -0.632812,  0.000000,  0.632812,  0.445312, -0.371094, -0.195312,  0.601562,
   -0.511719,  0.445312, -0.601562,  0.511719, -0.371094,  0.195312,  0.410156,
    0.558594,  0.500000,  0.410156,  0.289062,  0.148438,  0.410156,  0.410156,
    0.000000, -0.410156, -0.578125, -0.410156,  0.410156,  0.148438, -0.500000,
   -0.410156,  0.289062,  0.558594,  0.410156, -0.148438, -0.500000,  0.410156,
    0.289062, -0.558594,  0.410156, -0.410156,  0.000000,  0.410156, -0.578125,
    0.410156,  0.410156, -0.558594,  0.500000, -0.410156,  0.289062, -0.148438,
};

/**
 * @brief default decorrelation matrix offsets
 */
static const float * const default_decorrelation[] = {
    NULL,
    &default_decorrelation_matrices[0],
    &default_decorrelation_matrices[1],
    &default_decorrelation_matrices[5],
    &default_decorrelation_matrices[14],
    &default_decorrelation_matrices[30],
    &default_decorrelation_matrices[55]
};

#endif /* AVCODEC_WMAPRODATA_H */
