/*******************************************************************************
 * Copyright (c) 2011, 2025 Google, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Google, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.wb.internal.swing.model.property.editor.border.pages;

import org.eclipse.wb.internal.core.utils.ast.AstEditor;
import org.eclipse.wb.internal.swing.model.property.editor.border.BorderDialog;
import org.eclipse.wb.internal.swing.model.property.editor.border.BorderValue;
import org.eclipse.wb.internal.swing.model.property.editor.border.fields.AbstractBorderField;
import org.eclipse.wb.internal.swing.model.property.editor.border.fields.BooleanField;
import org.eclipse.wb.internal.swing.model.property.editor.border.fields.BorderField;
import org.eclipse.wb.internal.swing.model.property.editor.border.fields.ColorField;
import org.eclipse.wb.internal.swing.model.property.editor.border.fields.ComboField;
import org.eclipse.wb.internal.swing.model.property.editor.border.fields.IntegerField;
import org.eclipse.wb.internal.swing.model.property.editor.border.fields.RadioField;
import org.eclipse.wb.internal.swing.model.property.editor.border.fields.TextField;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;

import javax.swing.border.Border;

/**
 * Abstract editor for some {@link Border} type.
 *
 * @author scheglov_ke
 * @coverage swing.property.editor
 */
public abstract class AbstractBorderComposite extends Composite {
	protected static final Map<Class<?>, Predicate<Class<?>>> COMPOSITE_CLASSES = new HashMap<>();
	private final String m_title;
	protected BorderDialog m_borderDialog;

	////////////////////////////////////////////////////////////////////////////
	//
	// Constructor
	//
	////////////////////////////////////////////////////////////////////////////
	public AbstractBorderComposite(Composite parent, String title) {
		super(parent, SWT.NONE);
		m_title = title;
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Access
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Initializes this {@link AbstractBorderComposite}.
	 */
	public void initialize(BorderDialog borderDialog, AstEditor editor) {
		m_borderDialog = borderDialog;
	}

	/**
	 * @return the title to display for user.
	 */
	public final String getTitle() {
		return m_title;
	}

	/**
	 * Sets the {@link BorderValue} to edit.
	 *
	 * @return A {@link CompletableFuture} if this {@link AbstractBorderComposite}
	 *         understands given {@link BorderValue}, otherwise {@code null}.
	 */
	public abstract CompletableFuture<Void> setBorderValue(BorderValue borderValue);

	/**
	 * @return the source for updated {@link Border}.
	 */
	public abstract String getSource();

	/**
	 * Used by the {@link BorderField} for the source code generation.
	 * <i>Important</i> This method should only be called after the
	 * {@link BorderDialog} has been created at least once.
	 *
	 * @return the composite class managing the given border.
	 */
	public static Class<?> getCompositeClass(Class<?> clazz) {
		if (clazz == null) {
			return NoBorderComposite.class;
		}
		for (Map.Entry<Class<?>, Predicate<Class<?>>> entry : COMPOSITE_CLASSES.entrySet()) {
			if (entry.getValue().test(clazz)) {
				return entry.getKey();
			}
		}
		// The generic Swing borders might be of the same type as one that is managed by
		// one of the other composite classes. It must therefore be given the lowest
		// priority.
		if (SwingBorderComposite.contains(clazz)) {
			return SwingBorderComposite.class;
		}
		return null;
	}

	/**
	 * Returns the {@link BorderDialog} containing this composite.
	 */
	public BorderDialog getDialog() {
		return m_borderDialog;
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Components
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Binds given {@link AbstractBorderField}, so that when {@link SWT#Selection} event issued, we
	 * notify {@link BorderDialog}.
	 */
	private void bindField(AbstractBorderField field) {
		field.addListener(SWT.Selection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				m_borderDialog.borderUpdated();
			}
		});
	}

	/**
	 * @return the bound {@link TextField}.
	 */
	protected final TextField createTextField(String label) {
		TextField field = new TextField(this, label);
		bindField(field);
		return field;
	}

	/**
	 * @return the bound {@link IntegerField}.
	 */
	protected final IntegerField createIntegerField(String label) {
		IntegerField field = new IntegerField(this, label);
		bindField(field);
		return field;
	}

	/**
	 * @return the bound {@link RadioField}.
	 */
	protected final RadioField createRadioField(String label,
			Class<?> clazz,
			String[] fields,
			String[] titles) {
		RadioField field = new RadioField(this, label, clazz, fields, titles);
		bindField(field);
		return field;
	}

	/**
	 * @return the bound {@link ComboField}.
	 */
	protected final ComboField createComboField(String label,
			Class<?> clazz,
			String[] fields,
			String[] titles) {
		ComboField field = new ComboField(this, label, clazz, fields, titles);
		bindField(field);
		return field;
	}

	/**
	 * @return the bound {@link BooleanField}.
	 */
	protected final BooleanField createBooleanField(String label, String[] titles) {
		BooleanField field = new BooleanField(this, label, titles);
		bindField(field);
		return field;
	}

	/**
	 * @return the bound {@link ColorField}.
	 */
	protected final ColorField createColorField(String label) {
		ColorField field = new ColorField(this, label);
		bindField(field);
		return field;
	}

	/**
	 * @return the bound {@link BorderField}.
	 */
	protected final BorderField createBorderField(String label, String buttonText) {
		BorderField field = new BorderField(this, label, buttonText);
		bindField(field);
		return field;
	}
}
