/**
 * Copyright (c) 2010-2014, Abel Hegedus, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.addon.querybasedfeatures.tooling;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.viatra.addon.querybasedfeatures.runtime.handler.QueryBasedFeatures;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Annotation;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class SettingDelegateBasedGenerator {
  @Extension
  private QueryBasedFeatureGenerator gen;

  public SettingDelegateBasedGenerator(final QueryBasedFeatureGenerator generator) {
    this.gen = generator;
  }

  protected void updateAnnotations(final Pattern pattern, final Annotation annotation, final boolean generate) {
    try {
      final QueryBasedFeatureParameters parameters = this.gen.processDerivedFeatureAnnotation(pattern, annotation, generate);
      if ((!parameters.resourceWritable)) {
        return;
      }
      if (generate) {
        this.updateEcorePackage(parameters);
        this.updateFeatureAnnotation(parameters);
      } else {
        this.removeFeatureAnnotation(parameters);
      }
      try {
        parameters.ePackage.eResource().save(null);
      } catch (final Throwable _t) {
        if (_t instanceof IOException) {
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof IllegalArgumentException) {
        final IllegalArgumentException e = (IllegalArgumentException)_t;
        if (generate) {
          this.gen.logger.error(e.getMessage(), e);
        }
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }

  private String updateEcorePackage(final QueryBasedFeatureParameters parameters) {
    String _xtrycatchfinallyexpression = null;
    try {
      String _xblockexpression = null;
      {
        final EPackage pckg = parameters.ePackage;
        EList<EAnnotation> _eAnnotations = pckg.getEAnnotations();
        final ArrayList<EAnnotation> annotations = new ArrayList<EAnnotation>(_eAnnotations);
        final Function1<EAnnotation, Boolean> _function = (EAnnotation it) -> {
          String _source = it.getSource();
          return Boolean.valueOf(Objects.equals(_source, QueryBasedFeatures.ECORE_ANNOTATION));
        };
        EAnnotation ecoreAnnotation = IterableExtensions.<EAnnotation>findFirst(annotations, _function);
        if ((ecoreAnnotation == null)) {
          ecoreAnnotation = EcoreFactory.eINSTANCE.createEAnnotation();
          ecoreAnnotation.setSource(QueryBasedFeatures.ECORE_ANNOTATION);
          pckg.getEAnnotations().add(ecoreAnnotation);
        }
        final Function1<Map.Entry<String, String>, Boolean> _function_1 = (Map.Entry<String, String> it) -> {
          String _key = it.getKey();
          return Boolean.valueOf(Objects.equals(_key, QueryBasedFeatures.SETTING_DELEGATES_KEY));
        };
        Map.Entry<String, String> entry = IterableExtensions.<Map.Entry<String, String>>findFirst(ecoreAnnotation.getDetails(), _function_1);
        String _xifexpression = null;
        if ((entry == null)) {
          _xifexpression = ecoreAnnotation.getDetails().put(QueryBasedFeatures.SETTING_DELEGATES_KEY, QueryBasedFeatures.ANNOTATION_SOURCE);
        } else {
          _xifexpression = entry.setValue(QueryBasedFeatures.ANNOTATION_SOURCE);
        }
        _xblockexpression = _xifexpression;
      }
      _xtrycatchfinallyexpression = _xblockexpression;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        this.gen.logger.warn(String.format("Error happened when trying to add QBF annotation to package %s in Ecore!", parameters.ePackage.getName()), e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return _xtrycatchfinallyexpression;
  }

  private String updateFeatureAnnotation(final QueryBasedFeatureParameters parameters) {
    String _xtrycatchfinallyexpression = null;
    try {
      String _xblockexpression = null;
      {
        final EStructuralFeature feat = parameters.feature;
        EList<EAnnotation> _eAnnotations = feat.getEAnnotations();
        final ArrayList<EAnnotation> annotations = new ArrayList<EAnnotation>(_eAnnotations);
        final Consumer<EAnnotation> _function = (EAnnotation it) -> {
          if ((Objects.equals(it.getSource(), QueryBasedFeatures.ANNOTATION_SOURCE) || Objects.equals(it.getSource(), QueryBasedFeatures.LEGACY_ANNOTATION_SOURCE))) {
            feat.getEAnnotations().remove(it);
          }
        };
        annotations.forEach(_function);
        final EAnnotation annotation = EcoreFactory.eINSTANCE.createEAnnotation();
        annotation.setSource(QueryBasedFeatures.ANNOTATION_SOURCE);
        feat.getEAnnotations().add(annotation);
        _xblockexpression = annotation.getDetails().put(QueryBasedFeatures.PATTERN_FQN_KEY, PatternLanguageHelper.getFullyQualifiedName(parameters.pattern));
      }
      _xtrycatchfinallyexpression = _xblockexpression;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        this.gen.logger.warn(String.format("Error happened when trying to add QBF annotation to feature %s in Ecore!", parameters.feature.getName()), e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return _xtrycatchfinallyexpression;
  }

  private void removeFeatureAnnotation(final QueryBasedFeatureParameters parameters) {
    try {
      final EStructuralFeature feat = parameters.feature;
      EList<EAnnotation> _eAnnotations = feat.getEAnnotations();
      final ArrayList<EAnnotation> annotations = new ArrayList<EAnnotation>(_eAnnotations);
      final Consumer<EAnnotation> _function = (EAnnotation it) -> {
        if ((Objects.equals(it.getSource(), QueryBasedFeatures.ANNOTATION_SOURCE) || Objects.equals(it.getSource(), QueryBasedFeatures.LEGACY_ANNOTATION_SOURCE))) {
          feat.getEAnnotations().remove(it);
        }
      };
      annotations.forEach(_function);
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        this.gen.logger.warn(String.format("Error happened when trying to remove annotation to feature %s in Ecore!", parameters.feature), e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
}
