/**
 * Copyright (c) 2017, 2020 Kichwa Coders Ltd. and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j.debug;

import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Properties of a breakpoint passed to the setInstructionBreakpoints request
 */
@SuppressWarnings("all")
public class InstructionBreakpoint {
  /**
   * The instruction reference of the breakpoint.
   * <p>
   * This should be a memory or instruction pointer reference from an EvaluateResponse, Variable, StackFrame,
   * GotoTarget, or Breakpoint.
   */
  @NonNull
  private String instructionReference;

  /**
   * The offset from the instruction reference in bytes.
   * <p>
   * This can be negative.
   * <p>
   * This is an optional property.
   */
  private Integer offset;

  /**
   * An expression for conditional breakpoints.
   * <p>
   * It is only honored by a debug adapter if the corresponding capability
   * {@link Capabilities#getSupportsConditionalBreakpoints} is true.
   * <p>
   * This is an optional property.
   */
  private String condition;

  /**
   * An expression that controls how many hits of the breakpoint are ignored.
   * <p>
   * The debug adapter is expected to interpret the expression as needed.
   * <p>
   * The attribute is only honored by a debug adapter if the corresponding capability
   * {@link Capabilities#getSupportsHitConditionalBreakpoints} is true.
   * <p>
   * This is an optional property.
   */
  private String hitCondition;

  /**
   * The mode of this breakpoint. If defined, this must be one of the `breakpointModes`
   * the debug adapter advertised in its `Capabilities`.
   * <p>
   * This is an optional property.
   * <p>
   * Since 1.65
   */
  private String mode;

  /**
   * The instruction reference of the breakpoint.
   * <p>
   * This should be a memory or instruction pointer reference from an EvaluateResponse, Variable, StackFrame,
   * GotoTarget, or Breakpoint.
   */
  @NonNull
  public String getInstructionReference() {
    return this.instructionReference;
  }

  /**
   * The instruction reference of the breakpoint.
   * <p>
   * This should be a memory or instruction pointer reference from an EvaluateResponse, Variable, StackFrame,
   * GotoTarget, or Breakpoint.
   */
  public void setInstructionReference(@NonNull final String instructionReference) {
    this.instructionReference = Preconditions.checkNotNull(instructionReference, "instructionReference");
  }

  /**
   * The offset from the instruction reference in bytes.
   * <p>
   * This can be negative.
   * <p>
   * This is an optional property.
   */
  public Integer getOffset() {
    return this.offset;
  }

  /**
   * The offset from the instruction reference in bytes.
   * <p>
   * This can be negative.
   * <p>
   * This is an optional property.
   */
  public void setOffset(final Integer offset) {
    this.offset = offset;
  }

  /**
   * An expression for conditional breakpoints.
   * <p>
   * It is only honored by a debug adapter if the corresponding capability
   * {@link Capabilities#getSupportsConditionalBreakpoints} is true.
   * <p>
   * This is an optional property.
   */
  public String getCondition() {
    return this.condition;
  }

  /**
   * An expression for conditional breakpoints.
   * <p>
   * It is only honored by a debug adapter if the corresponding capability
   * {@link Capabilities#getSupportsConditionalBreakpoints} is true.
   * <p>
   * This is an optional property.
   */
  public void setCondition(final String condition) {
    this.condition = condition;
  }

  /**
   * An expression that controls how many hits of the breakpoint are ignored.
   * <p>
   * The debug adapter is expected to interpret the expression as needed.
   * <p>
   * The attribute is only honored by a debug adapter if the corresponding capability
   * {@link Capabilities#getSupportsHitConditionalBreakpoints} is true.
   * <p>
   * This is an optional property.
   */
  public String getHitCondition() {
    return this.hitCondition;
  }

  /**
   * An expression that controls how many hits of the breakpoint are ignored.
   * <p>
   * The debug adapter is expected to interpret the expression as needed.
   * <p>
   * The attribute is only honored by a debug adapter if the corresponding capability
   * {@link Capabilities#getSupportsHitConditionalBreakpoints} is true.
   * <p>
   * This is an optional property.
   */
  public void setHitCondition(final String hitCondition) {
    this.hitCondition = hitCondition;
  }

  /**
   * The mode of this breakpoint. If defined, this must be one of the `breakpointModes`
   * the debug adapter advertised in its `Capabilities`.
   * <p>
   * This is an optional property.
   * <p>
   * Since 1.65
   */
  public String getMode() {
    return this.mode;
  }

  /**
   * The mode of this breakpoint. If defined, this must be one of the `breakpointModes`
   * the debug adapter advertised in its `Capabilities`.
   * <p>
   * This is an optional property.
   * <p>
   * Since 1.65
   */
  public void setMode(final String mode) {
    this.mode = mode;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("instructionReference", this.instructionReference);
    b.add("offset", this.offset);
    b.add("condition", this.condition);
    b.add("hitCondition", this.hitCondition);
    b.add("mode", this.mode);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    InstructionBreakpoint other = (InstructionBreakpoint) obj;
    if (this.instructionReference == null) {
      if (other.instructionReference != null)
        return false;
    } else if (!this.instructionReference.equals(other.instructionReference))
      return false;
    if (this.offset == null) {
      if (other.offset != null)
        return false;
    } else if (!this.offset.equals(other.offset))
      return false;
    if (this.condition == null) {
      if (other.condition != null)
        return false;
    } else if (!this.condition.equals(other.condition))
      return false;
    if (this.hitCondition == null) {
      if (other.hitCondition != null)
        return false;
    } else if (!this.hitCondition.equals(other.hitCondition))
      return false;
    if (this.mode == null) {
      if (other.mode != null)
        return false;
    } else if (!this.mode.equals(other.mode))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.instructionReference== null) ? 0 : this.instructionReference.hashCode());
    result = prime * result + ((this.offset== null) ? 0 : this.offset.hashCode());
    result = prime * result + ((this.condition== null) ? 0 : this.condition.hashCode());
    result = prime * result + ((this.hitCondition== null) ? 0 : this.hitCondition.hashCode());
    return prime * result + ((this.mode== null) ? 0 : this.mode.hashCode());
  }
}
