/*******************************************************************************
 * Copyright (c) 2005, 2017 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Simon Muschel <smuschel@gmx.de> - bug 260549
 *     Simon Scholz <simon.scholz@vogella.com> - Bug 444808
 *******************************************************************************/
package org.eclipse.pde.internal.core.builders;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.osgi.util.NLS;
import org.eclipse.pde.core.plugin.IPluginModelBase;
import org.eclipse.pde.core.plugin.ModelEntry;
import org.eclipse.pde.core.plugin.PluginRegistry;
import org.eclipse.pde.internal.core.ICoreConstants;
import org.eclipse.pde.internal.core.PDECore;
import org.eclipse.pde.internal.core.PDECoreMessages;
import org.eclipse.pde.internal.core.builders.IncrementalErrorReporter.VirtualMarker;
import org.eclipse.pde.internal.core.ifeature.IFeatureModel;
import org.eclipse.pde.internal.core.util.IdUtil;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.NodeList;

public class FeatureErrorReporter extends ManifestErrorReporter {

	static HashSet<String> attrs = new HashSet<>();

	static String[] attrNames = { "id", "version", "include-sources", "label", "provider-name", "image", "os", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$
			"ws", "arch", "nl", "colocation-affinity", "primary", "exclusive", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
			"plugin", "application", "license-feature", "license-feature-version"}; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

	private IProgressMonitor fMonitor;

	public FeatureErrorReporter(IFile file) {
		super(file);
		if (attrs.isEmpty()) {
			attrs.addAll(Arrays.asList(attrNames));
		}
	}

	@Override
	protected void validate(IProgressMonitor monitor) {
		fMonitor = monitor;
		Element element = getDocumentRoot();
		if (element == null) {
			return;
		}
		String elementName = element.getNodeName();
		if (!"feature".equals(elementName)) { //$NON-NLS-1$
			reportIllegalElement(element, CompilerFlags.ERROR);
		} else {
			validateFeatureAttributes(element);
			validateInstallHandler(element);
			validateDescription(element);
			validateLicense(element);
			validateCopyright(element);
			validateURLElement(element);
			validateIncludes(element);
			validateRequires(element);
			validatePlugins(element);
			validateData(element);
		}
	}

	private void validateData(Element parent) {
		NodeList list = getChildrenByName(parent, "data"); //$NON-NLS-1$
		for (int i = 0; i < list.getLength(); i++) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element data = (Element) list.item(i);
			assertAttributeDefined(data, "id", CompilerFlags.ERROR); //$NON-NLS-1$
			NamedNodeMap attributes = data.getAttributes();
			for (int j = 0; j < attributes.getLength(); j++) {
				Attr attr = (Attr) attributes.item(j);
				String name = attr.getName();
				if (!name.equals("id") && !name.equals("os") && !name.equals("ws") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						&& !name.equals("nl") && !name.equals("arch") //$NON-NLS-1$ //$NON-NLS-2$
						&& !name.equals("download-size") && !name.equals("install-size")) { //$NON-NLS-1$ //$NON-NLS-2$
					reportUnknownAttribute(data, name, CompilerFlags.ERROR);
				}
			}
		}
	}

	private void validatePlugins(Element parent) {
		NodeList list = getChildrenByName(parent, "plugin"); //$NON-NLS-1$
		for (int i = 0; i < list.getLength(); i++) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element plugin = (Element) list.item(i);
			assertAttributeDefined(plugin, "id", CompilerFlags.ERROR); //$NON-NLS-1$
			assertAttributeDefined(plugin, "version", CompilerFlags.ERROR); //$NON-NLS-1$
			NamedNodeMap attributes = plugin.getAttributes();
			for (int j = 0; j < attributes.getLength(); j++) {
				Attr attr = (Attr) attributes.item(j);
				String name = attr.getName();
				if (name.equals("id")) { //$NON-NLS-1$
					validatePluginExists(plugin, attr);
				} else if (name.equals("version")) { //$NON-NLS-1$
					validateVersionAttribute(plugin, attr);
					validateVersion(plugin, attr);
				} else if (name.equals("fragment") || name.equals("unpack")) { //$NON-NLS-1$ //$NON-NLS-2$
					validateBoolean(plugin, attr);
				} else if (!name.equals("os") && !name.equals("ws") && !name.equals("nl") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						&& !name.equals("arch") && !name.equals("download-size") //$NON-NLS-1$ //$NON-NLS-2$
						&& !name.equals("install-size") && !name.equals("filter")) { //$NON-NLS-1$ //$NON-NLS-2$
					reportUnknownAttribute(plugin, name, CompilerFlags.ERROR);
				}
			}
		}
	}

	private void validateRequires(Element parent) {
		NodeList list = getChildrenByName(parent, "requires"); //$NON-NLS-1$
		if (list.getLength() > 0) {
			validateImports((Element) list.item(0));
			reportExtraneousElements(list, 1);
		}
	}

	private void validateImports(Element parent) {
		NodeList list = getChildrenByName(parent, "import"); //$NON-NLS-1$
		for (int i = 0; i < list.getLength(); i++) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element element = (Element) list.item(i);
			Attr plugin = element.getAttributeNode("plugin"); //$NON-NLS-1$
			Attr feature = element.getAttributeNode("feature"); //$NON-NLS-1$
			if (plugin == null && feature == null) {
				assertAttributeDefined(element, "plugin", CompilerFlags.ERROR); //$NON-NLS-1$
			} else if (plugin != null && feature != null) {
				reportExclusiveAttributes(element, "plugin", "feature", CompilerFlags.ERROR); //$NON-NLS-1$//$NON-NLS-2$
			} else if (plugin != null) {
				validatePluginExists(element, plugin);
			} else if (feature != null) {
				validateFeatureExists(element, feature);
			}
			NamedNodeMap attributes = element.getAttributes();
			for (int j = 0; j < attributes.getLength(); j++) {
				Attr attr = (Attr) attributes.item(j);
				String name = attr.getName();
				if (name.equals("version")) { //$NON-NLS-1$
					validateVersionAttribute(element, attr);
				} else if (name.equals("match")) { //$NON-NLS-1$
					if (element.getAttributeNode("patch") != null) { //$NON-NLS-1$
						report(NLS.bind(PDECoreMessages.Builders_Feature_patchedMatch, attr.getValue()), getLine(element, attr.getValue()), CompilerFlags.ERROR, PDEMarkerFactory.CAT_FATAL);
					} else {
						validateMatch(element, attr);
					}
				} else if (name.equals("patch")) { //$NON-NLS-1$
					if ("true".equalsIgnoreCase(attr.getValue()) && feature == null) { //$NON-NLS-1$
						report(NLS.bind(PDECoreMessages.Builders_Feature_patchPlugin, attr.getValue()), getLine(element, attr.getValue()), CompilerFlags.ERROR, PDEMarkerFactory.CAT_FATAL);
					} else if ("true".equalsIgnoreCase(attr.getValue()) && element.getAttributeNode("version") == null) { //$NON-NLS-1$ //$NON-NLS-2$
						report(NLS.bind(PDECoreMessages.Builders_Feature_patchedVersion, attr.getValue()), getLine(element, attr.getValue()), CompilerFlags.ERROR, PDEMarkerFactory.CAT_FATAL);
					} else {
						validateBoolean(element, attr);
					}
				} else if (!name.equals("plugin") && !name.equals("feature") && !name.equals("filter")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					reportUnknownAttribute(element, name, CompilerFlags.ERROR);
				}
			}

		}

	}

	private void validateIncludes(Element parent) {
		NodeList list = getChildrenByName(parent, "includes"); //$NON-NLS-1$
		for (int i = 0; i < list.getLength(); i++) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element include = (Element) list.item(i);
			if (assertAttributeDefined(include, "id", CompilerFlags.ERROR) //$NON-NLS-1$
					&& assertAttributeDefined(include, "version", //$NON-NLS-1$
							CompilerFlags.ERROR)) {

				validateFeatureExists(include, include.getAttributeNode("id")); //$NON-NLS-1$
			}
			NamedNodeMap attributes = include.getAttributes();
			for (int j = 0; j < attributes.getLength(); j++) {
				Attr attr = (Attr) attributes.item(j);
				String name = attr.getName();
				if (name.equals("version")) { //$NON-NLS-1$
					validateVersionAttribute(include, attr);
				} else if (name.equals("optional")) { //$NON-NLS-1$
					validateBoolean(include, attr);
				} else if (name.equals("search-location")) { //$NON-NLS-1$
					String value = include.getAttribute("search-location"); //$NON-NLS-1$
					if (!value.equals("root") && !value.equals("self") && !value.equals("both")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						reportIllegalAttributeValue(include, attr);
					}
				} else if (!name.equals("id") && !name.equals("name") && !name.equals("os") && !name.equals("ws") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
						&& !name.equals("nl") && !name.equals("arch") && !name.equals("filter")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					reportUnknownAttribute(include, name, CompilerFlags.ERROR);
				}
			}
		}
	}

	private void validateURLElement(Element parent) {
		NodeList list = getChildrenByName(parent, "url"); //$NON-NLS-1$
		if (list.getLength() > 0) {
			Element url = (Element) list.item(0);
			validateUpdateURL(url);
			validateDiscoveryURL(url);
			reportExtraneousElements(list, 1);
		}
	}

	private void validateUpdateURL(Element parent) {
		NodeList list = getChildrenByName(parent, "update"); //$NON-NLS-1$
		if (list.getLength() > 0) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element update = (Element) list.item(0);
			assertAttributeDefined(update, "url", CompilerFlags.ERROR); //$NON-NLS-1$
			NamedNodeMap attributes = update.getAttributes();
			for (int i = 0; i < attributes.getLength(); i++) {
				String name = attributes.item(i).getNodeName();
				if (name.equals("url")) { //$NON-NLS-1$
					validateURL(update, "url"); //$NON-NLS-1$
				} else if (!name.equals("label")) { //$NON-NLS-1$
					reportUnknownAttribute(update, name, CompilerFlags.ERROR);
				}
			}
			reportExtraneousElements(list, 1);
		}
	}

	private void validateDiscoveryURL(Element parent) {
		NodeList list = getChildrenByName(parent, "discovery"); //$NON-NLS-1$
		if (list.getLength() > 0) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element discovery = (Element) list.item(0);
			assertAttributeDefined(discovery, "url", CompilerFlags.ERROR); //$NON-NLS-1$
			NamedNodeMap attributes = discovery.getAttributes();
			for (int i = 0; i < attributes.getLength(); i++) {
				String name = attributes.item(i).getNodeName();
				if (name.equals("url")) { //$NON-NLS-1$
					validateURL(discovery, "url"); //$NON-NLS-1$
				} else if (name.equals("type")) { //$NON-NLS-1$
					String value = discovery.getAttribute("type"); //$NON-NLS-1$
					if (!value.equals("web") && !value.equals("update")) { //$NON-NLS-1$ //$NON-NLS-2$
						reportIllegalAttributeValue(discovery, (Attr) attributes.item(i));
					}
					reportDeprecatedAttribute(discovery, discovery.getAttributeNode("type")); //$NON-NLS-1$
				} else if (!name.equals("label")) { //$NON-NLS-1$
					reportUnknownAttribute(discovery, name, CompilerFlags.ERROR);
				}
			}
		}
	}

	private void validateCopyright(Element parent) {
		NodeList list = getChildrenByName(parent, "copyright"); //$NON-NLS-1$
		if (list.getLength() > 0) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element element = (Element) list.item(0);
			validateElementWithContent((Element) list.item(0), true);
			NamedNodeMap attributes = element.getAttributes();
			for (int i = 0; i < attributes.getLength(); i++) {
				Attr attr = (Attr) attributes.item(i);
				String name = attr.getName();
				if (name.equals("url")) { //$NON-NLS-1$
					validateURL(element, name);
				} else {
					reportUnknownAttribute(element, name, CompilerFlags.ERROR);
				}
			}
			reportExtraneousElements(list, 1);
		}
	}

	private void validateLicense(Element parent) {
		NodeList list = getChildrenByName(parent, "license"); //$NON-NLS-1$
		if (list.getLength() > 0) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element element = (Element) list.item(0);
			validateElementWithContent((Element) list.item(0), true);
			NamedNodeMap attributes = element.getAttributes();
			for (int i = 0; i < attributes.getLength(); i++) {
				Attr attr = (Attr) attributes.item(i);
				String name = attr.getName();
				if (name.equals("url")) { //$NON-NLS-1$
					validateURL(element, name);
				} else {
					reportUnknownAttribute(element, name, CompilerFlags.ERROR);
				}
			}
			reportExtraneousElements(list, 1);
		}
	}

	private void validateDescription(Element parent) {
		NodeList list = getChildrenByName(parent, "description"); //$NON-NLS-1$
		if (list.getLength() > 0) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element element = (Element) list.item(0);
			validateElementWithContent((Element) list.item(0), true);
			NamedNodeMap attributes = element.getAttributes();
			for (int i = 0; i < attributes.getLength(); i++) {
				Attr attr = (Attr) attributes.item(i);
				String name = attr.getName();
				if (name.equals("url")) { //$NON-NLS-1$
					validateURL(element, name);
				} else {
					reportUnknownAttribute(element, name, CompilerFlags.ERROR);
				}
			}
			reportExtraneousElements(list, 1);
		}
	}

	private void validateInstallHandler(Element element) {
		NodeList elements = getChildrenByName(element, "install-handler"); //$NON-NLS-1$
		if (elements.getLength() > 0) {
			if (fMonitor.isCanceled()) {
				return;
			}
			Element handler = (Element) elements.item(0);
			NamedNodeMap attributes = handler.getAttributes();
			for (int i = 0; i < attributes.getLength(); i++) {
				String name = attributes.item(i).getNodeName();
				if (!name.equals("library") && !name.equals("handler")) { //$NON-NLS-1$ //$NON-NLS-2$
					reportUnknownAttribute(handler, name, CompilerFlags.ERROR);
				}
			}
			reportExtraneousElements(elements, 1);
		}
	}

	private void validateFeatureAttributes(Element element) {
		if (fMonitor.isCanceled()) {
			return;
		}
		assertAttributeDefined(element, "id", CompilerFlags.ERROR); //$NON-NLS-1$
		assertAttributeDefined(element, "version", CompilerFlags.ERROR); //$NON-NLS-1$
		NamedNodeMap attributes = element.getAttributes();
		for (int i = 0; i < attributes.getLength(); i++) {
			String name = attributes.item(i).getNodeName();
			if (!attrs.contains(name)) {
				reportUnknownAttribute(element, name, CompilerFlags.ERROR);
			} else if (name.equals("id")) { //$NON-NLS-1$
				validateFeatureID(element, (Attr) attributes.item(i));
			} else if (name.equals("primary") || name.equals("exclusive")) { //$NON-NLS-1$ //$NON-NLS-2$
				validateBoolean(element, (Attr) attributes.item(i));
			} else if (name.equals("version")) { //$NON-NLS-1$
				validateVersionAttribute(element, (Attr) attributes.item(i));
			}
			if (name.equals("primary")) { //$NON-NLS-1$
				reportDeprecatedAttribute(element, (Attr) attributes.item(i));
			} else if (name.equals("plugin")) { //$NON-NLS-1$
				validatePluginExists(element, (Attr) attributes.item(i));
			}
		}
	}

	/**
	 * Checks whether the given attribute value is a valid feature ID.  If it is not valid, a marker
	 * is created on the element and <code>false</code> is returned. If valid, <code>true</code> is
	 * returned.  Also see {@link #validatePluginID(Element, Attr)}
	 *
	 * @param element element to add the marker to if invalid
	 * @param attr the attribute to check the value of
	 * @return whether the given attribute value is a valid feature ID.
	 */
	protected boolean validateFeatureID(Element element, Attr attr) {
		if (!IdUtil.isValidCompositeID(attr.getValue())) {
			String message = NLS.bind(PDECoreMessages.Builders_Manifest_compositeID, attr.getValue(), attr.getName());
			report(message, getLine(element, attr.getName()), CompilerFlags.WARNING, PDEMarkerFactory.CAT_OTHER);
			return false;
		}
		return true;
	}

	private void validatePluginExists(Element element, Attr attr) {
		String id = attr.getValue();
		int severity = CompilerFlags.getFlag(fProject, CompilerFlags.F_UNRESOLVED_PLUGINS);
		if (severity != CompilerFlags.IGNORE) {
			IPluginModelBase model = PluginRegistry.findModel(id);
			if (model == null || !model.isEnabled()) {
				VirtualMarker marker = report(NLS.bind(PDECoreMessages.Builders_Feature_reference, id), getLine(element, attr.getName()), severity, PDEMarkerFactory.CAT_OTHER);
				addMarkerAttribute(marker, PDEMarkerFactory.compilerKey,  CompilerFlags.F_UNRESOLVED_PLUGINS);
			}
		}
	}

	private void validateFeatureExists(Element element, Attr attr) {
		int severity = CompilerFlags.getFlag(fProject, CompilerFlags.F_UNRESOLVED_FEATURES);
		if (severity != CompilerFlags.IGNORE) {
			List<IFeatureModel> models = PDECore.getDefault().getFeatureModelManager().findFeatureModels(attr.getValue());
			if (models.isEmpty()) {
				VirtualMarker marker = report(NLS.bind(PDECoreMessages.Builders_Feature_freference, attr.getValue()), getLine(element, attr.getName()), severity, PDEMarkerFactory.CAT_OTHER);
				addMarkerAttribute(marker, PDEMarkerFactory.compilerKey,  CompilerFlags.F_UNRESOLVED_FEATURES);
			}
		}
	}

	protected void reportExclusiveAttributes(Element element, String attName1, String attName2, int severity) {
		String message = NLS.bind(PDECoreMessages.Builders_Feature_exclusiveAttributes, (new String[] {attName1, attName2}));
		report(message, getLine(element, attName2), severity, PDEMarkerFactory.CAT_OTHER);
	}

	/**
	 * Validates that the version of the given plug-in is available in the registry.  Adds a
	 * warning if the plug-in could not be found.
	 *
	 * @param plugin xml element describing the plug-in to look for in the registry
	 * @param attr set of element attributes
	 */
	private void validateVersion(Element plugin, Attr attr) {
		String id = plugin.getAttribute("id"); //$NON-NLS-1$
		String version = plugin.getAttribute("version"); //$NON-NLS-1$
		if (id.trim().length() == 0 || version.trim().length() == 0 || version.equals(ICoreConstants.DEFAULT_VERSION)) {
			return;
		}
		ModelEntry entry = PluginRegistry.findEntry(id);
		if (entry != null) {
			IPluginModelBase[] allModels = entry.getActiveModels();
			for (IPluginModelBase model : allModels) {
				if (id.equals(model.getPluginBase().getId())) {
					if (version.equals(model.getPluginBase().getVersion())) {
						return;
					}
				}
			}
		}
		report(NLS.bind(PDECoreMessages.Builders_Feature_mismatchPluginVersion, new String[] {version, id}), getLine(plugin, attr.getName()), CompilerFlags.WARNING, PDEMarkerFactory.CAT_OTHER);
	}

}
