/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.utils;

import static com.google.common.base.Predicates.*;
import static com.google.common.collect.ImmutableList.copyOf;
import static org.apache.commons.lang3.StringUtils.startsWith;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.*;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.logging.aeri.core.ISendOptions;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;

import com.google.common.base.Predicate;
import com.google.common.collect.BiMap;
import com.google.common.collect.Collections2;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Ordering;

public class Servers {

    private static BiMap<EAttribute, EAttribute> SYSTEM_TO_SERVER = HashBiMap.create(ImmutableMap.of(
    // @formatter:off
    USER_SETTINGS__REPORTER_NAME, USER_SETTINGS__REPORTER_NAME,
    USER_SETTINGS__REPORTER_EMAIL, USER_SETTINGS__REPORTER_EMAIL,
    USER_SETTINGS__ANONYMIZE_MESSAGES, USER_SETTINGS__ANONYMIZE_MESSAGES,
    USER_SETTINGS__ANONYMIZE_STACK_TRACES, USER_SETTINGS__ANONYMIZE_STACK_TRACES));
    // @formatter:on

    private static BiMap<EAttribute, EAttribute> SERVER_TO_SEND_OPTIONS = HashBiMap.create(ImmutableMap.of(
            // @formatter:off
            USER_SETTINGS__REPORTER_ID, USER_SETTINGS__REPORTER_ID,
            USER_SETTINGS__REPORTER_NAME, USER_SETTINGS__REPORTER_NAME,
             USER_SETTINGS__REPORTER_EMAIL, USER_SETTINGS__REPORTER_EMAIL,
             USER_SETTINGS__ANONYMIZE_MESSAGES, USER_SETTINGS__ANONYMIZE_MESSAGES,
             USER_SETTINGS__ANONYMIZE_STACK_TRACES, USER_SETTINGS__ANONYMIZE_STACK_TRACES));
    // @formatter:on

    private static BiMap<EAttribute, EAttribute> SYSTEM_TO_SEND_OPTIONS = HashBiMap.create(ImmutableMap.of(
            // @formatter:off
            USER_SETTINGS__REPORTER_ID, USER_SETTINGS__REPORTER_ID,
            USER_SETTINGS__REPORTER_NAME, USER_SETTINGS__REPORTER_NAME,
            USER_SETTINGS__REPORTER_EMAIL, USER_SETTINGS__REPORTER_EMAIL,
            USER_SETTINGS__ANONYMIZE_MESSAGES, USER_SETTINGS__ANONYMIZE_MESSAGES,
            USER_SETTINGS__ANONYMIZE_STACK_TRACES, USER_SETTINGS__ANONYMIZE_STACK_TRACES));
    // @formatter:on

    public static Collection<IServerDescriptor> viewOfConfigured(Collection<IServerDescriptor> unfiltered) {
        return Collections2.filter(unfiltered, CONFIGURED);
    }

    public static Collection<IServerDescriptor> viewOfUnconfigured(Collection<IServerDescriptor> unfiltered) {
        return Collections2.filter(unfiltered, not(CONFIGURED));
    }

    public static Collection<IServerDescriptor> viewOfEnabled(Collection<IServerDescriptor> unfiltered) {
        return Collections2.filter(unfiltered, ENABLED);
    }

    public static Collection<IServerDescriptor> viewOfDisabled(Collection<IServerDescriptor> unfiltered) {
        return Collections2.filter(unfiltered, not(ENABLED));
    }

    public static Collection<IServerDescriptor> viewOfActive(Collection<IServerDescriptor> unfiltered) {
        return Collections2.filter(unfiltered, and(ENABLED, CONFIGURED));
    }

    public static Collection<IServerDescriptor> viewOfInactive(Collection<IServerDescriptor> unfiltered) {
        return Collections2.filter(unfiltered, not(and(ENABLED, CONFIGURED)));
    }

    public static ImmutableList<IServerDescriptor> copyOfConfigured(Collection<IServerDescriptor> unfiltered) {
        return copyOf(viewOfConfigured(unfiltered));
    }

    public static ImmutableList<IServerDescriptor> copyOfUnconfigured(Collection<IServerDescriptor> unfiltered) {
        return copyOf(viewOfUnconfigured(unfiltered));
    }

    public static ImmutableList<IServerDescriptor> copyOfEnabled(Collection<IServerDescriptor> unfiltered) {
        return copyOf(viewOfEnabled(unfiltered));
    }

    public static ImmutableList<IServerDescriptor> copyOfDisabled(Collection<IServerDescriptor> unfiltered) {
        return copyOf(viewOfDisabled(unfiltered));
    }

    public static ImmutableList<IServerDescriptor> copyOfActive(Collection<IServerDescriptor> unfiltered) {
        return copyOf(viewOfActive(unfiltered));
    }

    public static ImmutableList<IServerDescriptor> copyOfInactive(Collection<IServerDescriptor> unfiltered) {
        return copyOf(viewOfInactive(unfiltered));
    }

    public static void copyServerDefaultsIfUnset(IServerDescriptor source, ISendOptions target) {
        copyIfUnset(source, target, SERVER_TO_SEND_OPTIONS);
    }

    public static void copySystemDefaultsIfUnset(ISystemSettings source, ISendOptions target) {
        copyIfUnset(source, target, SYSTEM_TO_SEND_OPTIONS);
    }

    public static void copySystemDefaultsIfUnset(ISystemSettings source, IServerDescriptor target) {
        copyIfUnset(source, target, SYSTEM_TO_SERVER);
    }

    public static void setAsDefaults(IServerDescriptor source, ISystemSettings target) {
        copyIfUnset(source, target, SYSTEM_TO_SERVER.inverse());
    }

    public static List<IServerDescriptor> sort(Iterable<IServerDescriptor> servers) {
        return Ordering.from(new Comparator<IServerDescriptor>() {

            @Override
            public int compare(IServerDescriptor o1, IServerDescriptor o2) {
                String id1 = o1.getId();
                String id2 = o2.getId();

                if (id1.startsWith("org.eclipse.epp.logging.aeri")) { //$NON-NLS-1$
                    return -1;
                } else if (id2.startsWith("org.eclipse.epp.logging.aeri")) { //$NON-NLS-1$
                    return 1;
                } else {
                    return id1.compareTo(id2);
                }
            }
        }).sortedCopy(servers);
    }

    private static void copyIfUnset(EObject source, EObject target, Map<EAttribute, EAttribute> mappings) {
        for (Entry<EAttribute, EAttribute> entry : mappings.entrySet()) {
            EAttribute toAttr = entry.getValue();
            if (!target.eIsSet(toAttr)) {
                EAttribute fromAttr = entry.getKey();
                Object value = source.eGet(fromAttr);
                target.eSet(toAttr, value);
            }
        }
    }

    public static boolean isOnlyEclipseServer(Collection<IServerDescriptor> servers) {
        if (servers.isEmpty()) {
            return false;
        }
        for (IServerDescriptor server : servers) {
            if (!startsWith(server.getId(), "org.eclipse")) { //$NON-NLS-1$
                return false;
            }
        }
        return true;
    }

    private static final Predicate<IServerDescriptor> CONFIGURED = new Predicate<IServerDescriptor>() {

        @Override
        public boolean apply(IServerDescriptor input) {
            return input.isConfigured();
        }
    };

    private static final Predicate<IServerDescriptor> ENABLED = new Predicate<IServerDescriptor>() {

        @Override
        public boolean apply(IServerDescriptor input) {
            return input.isEnabled();
        }
    };
}
