/**
 * Copyright (c) 2016, 2022 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 *   Ansgar Radermacher (CEA LIST) - Larger refactoring
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse;

import com.google.common.collect.Iterables;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.IASTNode;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTCompositeTypeSpecifier;
import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.IDeclaration;
import org.eclipse.cdt.core.model.IEnumeration;
import org.eclipse.cdt.core.model.IField;
import org.eclipse.cdt.core.model.IFunction;
import org.eclipse.cdt.core.model.IInclude;
import org.eclipse.cdt.core.model.IMacro;
import org.eclipse.cdt.core.model.IMethod;
import org.eclipse.cdt.core.model.IMethodDeclaration;
import org.eclipse.cdt.core.model.IParent;
import org.eclipse.cdt.core.model.ISourceReference;
import org.eclipse.cdt.core.model.IStructure;
import org.eclipse.cdt.core.model.IStructureTemplate;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.model.ITypeDef;
import org.eclipse.cdt.core.model.IUsing;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Include;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Template;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Visibility;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.ClassifierTemplateParameter;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.ParameterableElement;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.RedefinableTemplateSignature;
import org.eclipse.uml2.uml.TemplateParameter;
import org.eclipse.uml2.uml.TemplateSignature;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.VisibilityKind;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Pass2: refine classifiers ((attributes and methods) and relationships between them
 * The objective of having two passes is that all type references (set in pass2) and forward declared elements are already
 * available in the UML model
 */
@SuppressWarnings("all")
public class GetOrCreateP2 {
  /**
   * Refine the classifiers within a parent (called initially with a translation unit, then recursively)
   * @param model the model to get/create classifiers in
   * @param parent the CDT model parent
   */
  public static void getOrCreateClassifiers(final IParent parent) throws Exception {
    ICElement[] children = ReverseUtils.safeGetChilds(parent);
    for (int i = 0; (i < children.length); i++) {
      {
        ICElement child = children[i];
        boolean _isActive = ReverseUtils.isActive(child);
        if (_isActive) {
          boolean _matched = false;
          if ((child instanceof IStructure)) {
            _matched=true;
          }
          if (!_matched) {
            if ((child instanceof IEnumeration)) {
              _matched=true;
            }
          }
          if (_matched) {
            GetOrCreateP2.getOrCreateClassifier(child);
          }
          if (!_matched) {
            if ((child instanceof IMethod)) {
              _matched=true;
              if ((((IMethod) child).getTranslationUnit().isHeaderUnit() && (!(child.getParent() instanceof IStructure)))) {
                final IMethod method = ((IMethod) child);
                final Classifier type = ReverseUtils.getTypeOfVariableOrMethod(method, method.getTranslationUnit());
                if ((type != null)) {
                  ITranslationUnit _translationUnit = method.getTranslationUnit();
                  String _elementName = method.getCProject().getElementName();
                  BatchReverseFunctionBody batchReverser = new BatchReverseFunctionBody(_translationUnit, _elementName, type);
                  String name = method.getElementName();
                  IASTNode node = ASTUtils.findEnclosingNode(method);
                  if ((node instanceof IASTFunctionDefinition)) {
                    IASTFunctionDefinition definition = ((IASTFunctionDefinition) node);
                    IASTFunctionDeclarator declarator = definition.getDeclarator();
                    String body = MethodUtils.getBody(method);
                    Operation operation = batchReverser.updateMethod(method, node, 0, parent, name, body, declarator);
                    if ((operation != null)) {
                    }
                  }
                }
              }
            }
          }
          if (!_matched) {
            if ((child instanceof IFunction)) {
              _matched=true;
            }
          }
          if (!_matched) {
            if ((child instanceof IParent)) {
              _matched=true;
              GetOrCreateP2.getOrCreateClassifiers(((IParent) child));
            }
          }
          if (!_matched) {
            if ((child instanceof ITypeDef)) {
              _matched=true;
              GetOrCreateP2.getOrCreateClassifier(child);
            }
          }
          if (!_matched) {
            if ((child instanceof IUsing)) {
              _matched=true;
            }
          }
          if (!_matched) {
            if ((child instanceof IInclude)) {
              _matched=true;
              final ITranslationUnit headerUnit = ReverseUtils.getTranslationUnitFromInclude(((IInclude) child), ReverseData.current.project);
              if (((headerUnit != null) && (ReverseUtils.getSourceFolder(headerUnit) != null))) {
                ReverseCpp2Uml.currentCpp2Uml().reverseHeader(headerUnit);
              }
            }
          }
        }
      }
    }
  }

  /**
   * Create classifier for a declaration, either a (template) class, an enumeration, struct/union or typedefs, using statements
   * The function also creates nested classifiers, if present. For instance a typedef or using statements within a class.
   * It takes care of creating the nested elements at the right moment, e.g. before creating attributes and methods that eventually
   * reference these types.
   * 
   * @param container the owner to create a classifier in, must be a class or package
   * @return the associated UML type
   */
  public static Classifier getOrCreateClassifier(final ICElement ideclaration) {
    try {
      final Map<ICElement, EObject> map = ReverseData.current.map;
      final Map<ICElement, Boolean> analyzeMap = ReverseData.current.analyzeMap;
      if ((((((ideclaration.getElementType() != ICElement.C_CLASS) && (ideclaration.getElementType() != ICElement.C_TEMPLATE_CLASS)) && 
        (ideclaration.getElementType() != ICElement.C_STRUCT)) && 
        (ideclaration.getElementType() != ICElement.C_UNION)) && (ideclaration.getElementType() != ICElement.C_TEMPLATE_STRUCT))) {
        return null;
      }
      Classifier existing = null;
      EObject _get = map.get(ideclaration);
      if ((_get instanceof Classifier)) {
        EObject _get_1 = map.get(ideclaration);
        existing = ((Classifier) _get_1);
      }
      boolean _containsKey = analyzeMap.containsKey(ideclaration);
      if (_containsKey) {
        return existing;
      }
      analyzeMap.put(ideclaration, Boolean.valueOf(true));
      String _elementName = ideclaration.getElementName();
      String _plus = ("Completing type " + _elementName);
      ReverseData.current.monitor.subTask(_plus);
      final IStructure iStructure = ((IStructure) ideclaration);
      if ((existing instanceof org.eclipse.uml2.uml.Class)) {
        org.eclipse.uml2.uml.Class classifier = ((org.eclipse.uml2.uml.Class) existing);
        GetOrCreateP2.addSuperTypes(classifier, iStructure);
        ICElement[] _children = iStructure.getChildren();
        for (final ICElement child : _children) {
          if ((child instanceof IField)) {
            PropertyUtils.createProperty(((IField)child), classifier);
          } else {
            if ((child instanceof IMethodDeclaration)) {
              MethodUtils.createMethod(((IMethodDeclaration)child), classifier);
            }
          }
        }
        int _elementType = iStructure.getElementType();
        boolean _equals = (_elementType == ICElement.C_TEMPLATE_CLASS);
        if (_equals) {
          IStructureTemplate istructureTemplate = ((IStructureTemplate) iStructure);
          final Template template = StereotypeUtil.<Template>applyApp(classifier, Template.class);
          template.setDeclaration(istructureTemplate.getTemplateSignature());
        }
        return classifier;
      } else {
        if ((existing instanceof DataType)) {
          DataType dataType = ((DataType) existing);
          GetOrCreateP2.addSuperTypes(dataType, iStructure);
          GetOrCreateP2.getOrCreateClassifiers(iStructure);
          ICElement[] _children_1 = iStructure.getChildren();
          for (final ICElement child_1 : _children_1) {
            int _elementType_1 = child_1.getElementType();
            boolean _equals_1 = (_elementType_1 == ICElement.C_FIELD);
            if (_equals_1) {
              PropertyUtils.createProperty(((IField) child_1), dataType);
            }
          }
          return dataType;
        }
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Add superTypes to a classifier
   */
  public static void addSuperTypes(final Classifier classifier, final ISourceReference sourceRef) {
    final IASTNode astNode = ASTUtils.findEnclosingNode(sourceRef);
    if ((astNode instanceof ICPPASTCompositeTypeSpecifier)) {
      ICPPASTCompositeTypeSpecifier.ICPPASTBaseSpecifier[] _baseSpecifiers = ((ICPPASTCompositeTypeSpecifier)astNode).getBaseSpecifiers();
      for (final ICPPASTCompositeTypeSpecifier.ICPPASTBaseSpecifier superTypeAST : _baseSpecifiers) {
        {
          final String qName = ASTUtils.getQualifiedNameFromNSpec(superTypeAST.getNameSpecifier());
          final Type superType = ReverseUtils.getUMLType(qName, ((ICElement) sourceRef));
          if ((superType instanceof Classifier)) {
            PkgDependencies.createDependency(classifier, superType);
            Generalization generalization = classifier.createGeneralization(((Classifier)superType));
            final VisibilityKind visibility = ASTUtils.convertVisibility(superTypeAST.getVisibility());
            boolean _notEquals = (!Objects.equals(visibility, VisibilityKind.PUBLIC_LITERAL));
            if (_notEquals) {
              final Visibility visibilitySt = StereotypeUtil.<Visibility>applyApp(generalization, Visibility.class);
              visibilitySt.setValue(visibility.getLiteral().toLowerCase());
            }
          }
        }
      }
    }
  }

  /**
   * Get or create the Include stereotype of a classifier.
   * TODO: currently unused, can remove? (originally getOrCreateTypes), when to call?
   * @param iStructure, a structure or enumeration
   */
  public static List<Type> getOrCreateIncludes(final IDeclaration iStructure, final IParent parent, final Model model) {
    try {
      List<Type> ret = new UniqueEList<Type>();
      String _elementName = iStructure.getElementName();
      String _plus = ("Parsing type " + _elementName);
      ReverseData.current.monitor.subTask(_plus);
      Classifier tempType = null;
      IDeclaration istructure = ((IDeclaration) iStructure);
      String elementName = istructure.getElementName();
      Namespace container = ReverseUtils.getContainer(istructure);
      NamedElement _ownedMember = container.getOwnedMember(elementName);
      tempType = ((Classifier) _ownedMember);
      if ((tempType == null)) {
        Classifier _orCreateClassifier = GetOrCreateP2.getOrCreateClassifier(istructure);
        tempType = ((Classifier) _orCreateClassifier);
      }
      if ((tempType != null)) {
        ret.add(tempType);
        if ((istructure instanceof IParent)) {
          List<ICElement> nestedStructures = new UniqueEList<ICElement>();
          Iterables.<ICElement>addAll(nestedStructures, Iterables.<IDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(((IParent)istructure).getChildren())), IDeclaration.class));
          Iterables.<ICElement>addAll(nestedStructures, Iterables.<ITypeDef>filter(((Iterable<?>)Conversions.doWrapArray(((IParent)istructure).getChildren())), ITypeDef.class));
          int _size = nestedStructures.size();
          ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _size, true);
          for (final Integer i : _doubleDotLessThan) {
            {
              ICElement ns = nestedStructures.get((i).intValue());
              boolean _isEmpty = ns.getElementName().trim().isEmpty();
              boolean _not = (!_isEmpty);
              if (_not) {
                Classifier _orCreateClassifier_1 = GetOrCreateP2.getOrCreateClassifier(ns);
                Type t = ((Type) _orCreateClassifier_1);
                if ((t != null)) {
                  ret.add(t);
                }
              } else {
                if ((ns instanceof IEnumeration)) {
                  boolean isAnonEnumInstance = false;
                  try {
                    int _size_1 = nestedStructures.size();
                    boolean _lessThan = (((i).intValue() + 1) < _size_1);
                    if (_lessThan) {
                      ICElement nextNs = nestedStructures.get(((i).intValue() + 1));
                      if ((nextNs instanceof IField)) {
                        String nsRaw = ASTUtils.findEnclosingNode(((ISourceReference)ns)).getRawSignature();
                        String nextNsRaw = ASTUtils.findEnclosingNode(((ISourceReference)nextNs)).getRawSignature();
                        boolean _contains = nextNsRaw.contains(nsRaw);
                        if (_contains) {
                          isAnonEnumInstance = true;
                        }
                      }
                    }
                  } catch (final Throwable _t) {
                    if (_t instanceof Exception) {
                      final Exception e = (Exception)_t;
                      e.printStackTrace();
                    } else {
                      throw Exceptions.sneakyThrow(_t);
                    }
                  }
                  if ((!isAnonEnumInstance)) {
                    Classifier _orCreateClassifier_2 = GetOrCreateP2.getOrCreateClassifier(ns);
                    Type t_1 = ((Type) _orCreateClassifier_2);
                    if ((t_1 != null)) {
                      ret.add(t_1);
                    }
                  }
                }
              }
            }
          }
        }
      }
      if ((istructure instanceof IStructureTemplate)) {
        IStructureTemplate istructureTemplate = ((IStructureTemplate) istructure);
        for (int ti = 0; (ti < ((List<String>)Conversions.doWrapArray(istructureTemplate.getTemplateParameterTypes())).size()); ti++) {
          GetOrCreateP2.getOrCreateTemplateParameter(tempType, istructureTemplate.getTemplateParameterTypes()[ti], "class");
        }
      }
      int _size_1 = ret.size();
      boolean _greaterThan = (_size_1 > 0);
      if (_greaterThan) {
        if (((!(tempType instanceof Enumeration)) && (!(tempType instanceof PrimitiveType)))) {
          IParent file = parent;
          boolean _isApplied = StereotypeUtil.isApplied(tempType, Include.class);
          boolean _not = (!_isApplied);
          if (_not) {
            StereotypeUtil.apply(tempType, Include.class);
          }
          final Include include = UMLUtil.<Include>getStereotypeApplication(tempType, Include.class);
          if ((include != null)) {
            String header = include.getHeader();
            String escape = "";
            ICElement[] _children = file.getChildren();
            for (final ICElement j : _children) {
              if ((j instanceof IInclude)) {
                boolean _contains = header.contains(((IInclude)j).getElementName());
                boolean _not_1 = (!_contains);
                if (_not_1) {
                  boolean _contains_1 = ((IInclude)j).getElementName().contains("Pkg_");
                  if (_contains_1) {
                    try {
                      String[] includeTokens = ((IInclude)j).getElementName().split("/");
                      String includeTrimmed = "";
                      int _length = includeTokens.length;
                      int _minus = (_length - 1);
                      ExclusiveRange _doubleDotLessThan_1 = new ExclusiveRange(0, _minus, true);
                      for (final Integer i_1 : _doubleDotLessThan_1) {
                        String _includeTrimmed = includeTrimmed;
                        String _get = includeTokens[(i_1).intValue()];
                        includeTrimmed = (_includeTrimmed + _get);
                      }
                      String[] typeQualifiedNameTokens = tempType.getQualifiedName().split("::");
                      String typeQualifiedNameTrimmed = "";
                      int _length_1 = typeQualifiedNameTokens.length;
                      int _minus_1 = (_length_1 - 1);
                      ExclusiveRange _doubleDotLessThan_2 = new ExclusiveRange(1, _minus_1, true);
                      for (final Integer i_2 : _doubleDotLessThan_2) {
                        String _typeQualifiedNameTrimmed = typeQualifiedNameTrimmed;
                        String _get_1 = typeQualifiedNameTokens[(i_2).intValue()];
                        typeQualifiedNameTrimmed = (_typeQualifiedNameTrimmed + _get_1);
                      }
                      boolean _equals = includeTrimmed.equals(typeQualifiedNameTrimmed);
                      boolean _not_2 = (!_equals);
                      if (_not_2) {
                        String _elementName_1 = ((IInclude)j).getElementName();
                        String _plus_1 = ((((header + escape) + "#include ") + "\"") + _elementName_1);
                        String _plus_2 = (_plus_1 + "\"");
                        header = _plus_2;
                        escape = "\n";
                      }
                    } catch (final Throwable _t) {
                      if (_t instanceof Exception) {
                        final Exception e = (Exception)_t;
                        e.printStackTrace();
                        String _elementName_2 = ((IInclude)j).getElementName();
                        String _plus_3 = ((((header + escape) + "#include ") + "\"") + _elementName_2);
                        String _plus_4 = (_plus_3 + "\"");
                        header = _plus_4;
                        escape = "\n";
                      } else {
                        throw Exceptions.sneakyThrow(_t);
                      }
                    }
                  } else {
                    String _elementName_1 = ((IInclude)j).getElementName();
                    String _plus_1 = ((((header + escape) + "#include ") + "\"") + _elementName_1);
                    String _plus_2 = (_plus_1 + "\"");
                    header = _plus_2;
                    escape = "\n";
                  }
                }
              } else {
                if ((j instanceof IMacro)) {
                  boolean _contains_2 = header.contains(((IMacro)j).getElementName());
                  boolean _not_2 = (!_contains_2);
                  if (_not_2) {
                    IASTNode node = ASTUtils.findEnclosingNode(((ISourceReference)j));
                    if ((node != null)) {
                      final String nodeString = node.toString();
                      int _indexOf = nodeString.indexOf("=");
                      int _plus_3 = (_indexOf + 1);
                      String value = nodeString.substring(_plus_3);
                      if (((value == null) || value.equals(""))) {
                        boolean _contains_3 = ((IMacro)j).getElementName().contains(tempType.getName().replaceAll(" ", "").toUpperCase());
                        boolean _not_3 = (!_contains_3);
                        if (_not_3) {
                          String _elementName_2 = ((IMacro)j).getElementName();
                          String _plus_4 = (((header + escape) + "#define ") + _elementName_2);
                          String _plus_5 = (_plus_4 + " ");
                          String _plus_6 = (_plus_5 + value);
                          header = _plus_6;
                          escape = "\n";
                        }
                      } else {
                        String _elementName_3 = ((IMacro)j).getElementName();
                        String _plus_7 = (((header + escape) + "#define ") + _elementName_3);
                        String _plus_8 = (_plus_7 + " ");
                        String _plus_9 = (_plus_8 + value);
                        header = _plus_9;
                        escape = "\n";
                      }
                    }
                  }
                }
              }
            }
            include.setHeader(header);
            if ((file instanceof ITranslationUnit)) {
              try {
                IPath sourcePath = ((ITranslationUnit)file).getPath().removeFileExtension().addFileExtension("cpp");
                ICElement sourceFile = ((ITranslationUnit) file).getCProject().findElement(sourcePath);
                if ((sourceFile instanceof ITranslationUnit)) {
                  BatchReverseFunctionBody.updateCppInclude(((ITranslationUnit)sourceFile), tempType);
                }
              } catch (final Throwable _t) {
                if (_t instanceof Exception) {
                } else {
                  throw Exceptions.sneakyThrow(_t);
                }
              }
            }
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Create a template parameter within a template signature. If the
   * signature does not exist, it will be created.
   * TODO: dead code, only used by getOrCreateIncludes
   */
  public static Classifier getOrCreateTemplateParameter(final NamedElement element, final String parameterTypeName, final String keyWord) {
    Classifier ret = null;
    TemplateSignature templateSignature = null;
    if ((element instanceof Classifier)) {
      Classifier classifier = ((Classifier) element);
      if (((classifier.getOwnedTemplateSignature() == null) || 
        (!(classifier.getOwnedTemplateSignature() instanceof RedefinableTemplateSignature)))) {
        TemplateSignature _createOwnedTemplateSignature = classifier.createOwnedTemplateSignature(
          UMLPackage.Literals.REDEFINABLE_TEMPLATE_SIGNATURE);
        templateSignature = ((RedefinableTemplateSignature) _createOwnedTemplateSignature);
        ((RedefinableTemplateSignature) templateSignature).setName(ReverseUtils.TEMPLATE_PARAMETER_SIGNATURE_NAME);
      }
      TemplateSignature _ownedTemplateSignature = classifier.getOwnedTemplateSignature();
      templateSignature = ((RedefinableTemplateSignature) _ownedTemplateSignature);
    } else {
      if ((element instanceof Operation)) {
        Operation operation = ((Operation) element);
        if (((operation.getOwnedTemplateSignature() == null) || 
          (!(operation.getOwnedTemplateSignature() instanceof TemplateSignature)))) {
          TemplateSignature _createOwnedTemplateSignature_1 = operation.createOwnedTemplateSignature(
            UMLPackage.Literals.TEMPLATE_SIGNATURE);
          templateSignature = ((TemplateSignature) _createOwnedTemplateSignature_1);
        }
        TemplateSignature _ownedTemplateSignature_1 = operation.getOwnedTemplateSignature();
        templateSignature = ((TemplateSignature) _ownedTemplateSignature_1);
      } else {
        return null;
      }
    }
    Iterable<ClassifierTemplateParameter> classifierTemplates = Iterables.<ClassifierTemplateParameter>filter(templateSignature.getOwnedParameters(), ClassifierTemplateParameter.class);
    final Function1<ClassifierTemplateParameter, Boolean> _function = (ClassifierTemplateParameter it) -> {
      ParameterableElement _ownedParameteredElement = it.getOwnedParameteredElement();
      return Boolean.valueOf((_ownedParameteredElement instanceof Classifier));
    };
    Iterable<ClassifierTemplateParameter> classifierTemplatesContainClassifier = IterableExtensions.<ClassifierTemplateParameter>filter(classifierTemplates, _function);
    final Function1<ClassifierTemplateParameter, ParameterableElement> _function_1 = (ClassifierTemplateParameter it) -> {
      return it.getOwnedParameteredElement();
    };
    Iterable<ParameterableElement> containedClassifiers = IterableExtensions.<ClassifierTemplateParameter, ParameterableElement>map(classifierTemplatesContainClassifier, _function_1);
    final Function1<Classifier, Boolean> _function_2 = (Classifier it) -> {
      return Boolean.valueOf(it.getName().equals(parameterTypeName));
    };
    ret = IterableExtensions.<Classifier>head(IterableExtensions.<Classifier>filter(Iterables.<Classifier>filter(containedClassifiers, Classifier.class), _function_2));
    if ((ret == null)) {
      TemplateParameter _createOwnedParameter = templateSignature.createOwnedParameter(
        UMLPackage.Literals.CLASSIFIER_TEMPLATE_PARAMETER);
      ClassifierTemplateParameter classifierTemplate = ((ClassifierTemplateParameter) _createOwnedParameter);
      ParameterableElement _createOwnedParameteredElement = classifierTemplate.createOwnedParameteredElement(UMLPackage.Literals.CLASS);
      ret = ((Classifier) _createOwnedParameteredElement);
      ret.setName(parameterTypeName);
      classifierTemplate.addKeyword(keyWord);
    }
    return ret;
  }

  /**
   * Get an existing classifier, i.e. do not create, if it does not exist yet
   */
  public static Type getClassifier(final ICElement declaration) throws Exception {
    final Map<ICElement, EObject> map = ReverseData.current.map;
    if (((map.get(declaration) != null) && (map.get(declaration) instanceof Type))) {
      EObject _get = map.get(declaration);
      return ((Type) _get);
    }
    Classifier _orCreateClassifier = GetOrCreateP2.getOrCreateClassifier(declaration);
    Type ret = ((Type) _orCreateClassifier);
    return ret;
  }
}
