/*****************************************************************************
 * Copyright (c) 2016 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *
 *****************************************************************************/

package org.eclipse.papyrus.designer.deployment.tools;

import java.util.Iterator;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.Enumerator;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.papyrus.designer.deployment.profile.Deployment.DeploymentPlan;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.PackageUtil;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.InstanceSpecification;
import org.eclipse.uml2.uml.InstanceValue;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.StructuralFeature;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;

public class DepPlanUtils {

	/**
	 * Return the package in which deployment plans are stored. Caveat: needs to be executed within a
	 * transition, since the deployment plan package will be created, if it does not exist yet.
	 *
	 * @param element
	 *            an arbitrary element of the source model (i.e. the model that will
	 *            store the deployment plan
	 * @return package in which deployment plans are stored
	 */
	public static Package getDepPlanRoot(Element element) {
		return ElementUtils.getRoot(element, DeployConstants.depPlanFolder);
	}

	/**
	 * Return all deployment plans
	 *
	 * @param element
	 *            an arbitrary element of the source model (i.e. the model that will
	 *            store the deployment plan
	 * @return a list of deployment plans
	 */
	public static EList<Package> getAllDepPlans(Element element) {
		Package root = PackageUtil.getRootPackage(element);
		Package depPlanRoot = root.getNestedPackage(DeployConstants.depPlanFolder);
		EList<Package> depPlanList = new BasicEList<Package>();
		if (depPlanRoot != null) {
			for (Package pkg : depPlanRoot.getNestedPackages()) {
				if (StereotypeUtil.isApplied(pkg, DeploymentPlan.class)) {
					depPlanList.add(pkg);
				}
			}
		}
		return depPlanList;
	}

	public static void delDepPlan(InstanceSpecification is) {
		Iterator<Slot> slots = is.getSlots().iterator();
		while (slots.hasNext()) {
			Slot slot = slots.next();
			InstanceSpecification subInstance = getInstance(slot);
			if (subInstance != null) {
				delDepPlan(subInstance);
			}
		}
		Element owner = is.getOwner();
		if (owner instanceof Package) {
			((Package) owner).getPackagedElements().remove(is);
		}
	}

	/**
	 * Return the instance that is defined by a slot value
	 *
	 * @param slot
	 * @return the first slot that corresponds to an instance specification
	 */
	public static InstanceSpecification getInstance(Slot slot) {
		Iterator<ValueSpecification> values = slot.getValues().iterator();
		while (values.hasNext()) {
			ValueSpecification value = values.next();
			// instances are accessible via ValueSpecification subclass InstanceValue
			if (value instanceof InstanceValue) {
				return ((InstanceValue) value).getInstance();
			}
		}
		return null;
	}

	/**
	 * create a slot for a given sub-instance specification.
	 *
	 * @param cdp
	 *            a deployment plan
	 * @param is
	 *            an instance specification for a composite class
	 * @param partIS
	 *            the instance specification of a part within the composite
	 * @param part
	 *            the part within the composite
	 */
	public static Slot createSlot(Package cdp, InstanceSpecification is, InstanceSpecification partIS, Property part) {
		// the instance specification of the composite has a slot for each part and it points
		// to the instance specification associated with the part.
		Slot slot = is.createSlot();
		slot.setDefiningFeature(part);

		InstanceValue iv = (InstanceValue)
				slot.createValue(null, null, UMLPackage.eINSTANCE.getInstanceValue());
		iv.setInstance(partIS);
		return slot;

	}

	/**
	 * Automatically choose an implementation, i.e. choose the first implementation
	 * within the component model that implements a given component type.
	 */
	public static Class autoChooseImplementation(Classifier componentType) {
		// choose implementation automatically: get the first one that implements the passed type
		// (problem: further tree expansion might depend on chosen implementation)
		// get reference to component model, then search all classes contained in it.
		Package compModel = ElementUtils.getRoot(componentType, DeployConstants.COMPONENT_MODEL);
		Iterator<Element> elements = compModel.allOwnedElements().iterator();
		while (elements.hasNext()) {
			Element element = elements.next();
			if (element instanceof Class) {
				Class candidate = (Class) element;
				if (candidate.getSuperClass(componentType.getName()) != null) {
					return candidate;
				}
			}
		}
		return null;
	}

	/**
	 * Configure an attribute of an instance specification
	 *
	 * @param instance
	 *            the instance specification
	 * @param property
	 *            An ENamedElement denoting the name of an attribute of a classifier that
	 *            is in the classifier list of the instance specification
	 * @param value
	 *            the string value. An enumeration can be configured via the name of the literal
	 */
	public static void configureProperty(InstanceSpecification instance, ENamedElement property, String value) {
		configureProperty(instance, property.getName(), value);
	}

	/**
	 * Configure an attribute of an instance specification
	 *
	 * @param instance
	 *            the instance specification
	 * @param propertyName
	 *            the name of an attribute of a classifier that is in the classifier list
	 *            of the instance specification
	 * @param value
	 *            the string value. An enumeration can be configured via the name of the literal
	 */
	public static void configureProperty(InstanceSpecification instance, String propertyName, String value) {
		Classifier extension = DepUtils.getClassifier(instance);
		Property attribute = (Property) ElementUtils.getNamedElementFromList(extension.getAllAttributes(), propertyName);
		if (attribute == null) {
			throw new RuntimeException(String.format(Messages.DepPlanUtils_CannotFindAttribute, propertyName, extension.getName()));
		}
		configureProperty(instance, attribute, value);
	}

	/**
	 * Configure an attribute of an instance specification
	 *
	 * @param instance
	 *            the instance specification
	 * @param attribute
	 *            an attribute of a classifier that is in the classifier list of the instance specification
	 * @param value
	 *            the string value. An enumeration can be configured via the name of the literal
	 */
	public static void configureProperty(InstanceSpecification instance, Property attribute, String value) {
		if (attribute.getType() instanceof Enumeration) {
			configureEnumProperty(instance, attribute, value);
		}
		else {
			// create a slot for a string value
			Slot slotStringVal = DepCreation.createSlotForConfigProp(instance, attribute);
			if (slotStringVal.getValues().get(0) instanceof LiteralString) {
				((LiteralString) slotStringVal.getValues().get(0)).setValue(value);
			}
			else {
				// indicates that operation has been called although types do not match
				throw new RuntimeException(String.format(Messages.DepPlanUtils_ConfigOfPropertyFailed, attribute.getName()));
			}
		}
	}

	/**
	 * Configure an attribute of an instance specification
	 *
	 * @param instance
	 *            the instance specification
	 * @param property
	 *            An ENamedElement denoting the name of an attribute of a classifier that
	 *            is in the classifier list of the instance specification
	 * @param value
	 *            the integer value.
	 */
	public static void configureProperty(InstanceSpecification instance, ENamedElement property, int value) {
		configureProperty(instance, property.getName(), value);
	}

	/**
	 * Configure an attribute of an instance specification
	 *
	 * @param instance
	 *            the instance specification
	 * @param propertyName
	 *            the name of an attribute of a classifier that is in the classifier list
	 *            of the instance specification
	 * @param value
	 *            the integer value.
	 */
	public static void configureProperty(InstanceSpecification instance, String propertyName, int value) {
		Classifier extension = DepUtils.getClassifier(instance);
		Property attribute = (Property) ElementUtils.getNamedElementFromList(extension.getAllAttributes(), propertyName);
		if (attribute == null) {
			throw new RuntimeException(String.format(Messages.DepPlanUtils_CannotFindAttribute, propertyName, extension.getName()));
		}
		configureProperty(instance, attribute, value);
	}

	/**
	 * Configure an attribute of an instance specification
	 *
	 * @param instance
	 *            the instance specification
	 * @param attribute
	 *            an attribute of a classifier that is in the classifier list of the instance specification
	 * @param value
	 *            the integer value.
	 */
	public static void configureProperty(InstanceSpecification instance, Property attribute, int value) {
		Slot slotIntVal = instance.createSlot();
		slotIntVal.setDefiningFeature(attribute);
		LiteralInteger intValue = (LiteralInteger)
				slotIntVal.createValue("value for " + attribute.getName(), attribute.getType(), UMLPackage.eINSTANCE.getLiteralInteger()); //$NON-NLS-1$
		intValue.setValue(value);
	}

	/**
	 * Convenience function: allow that an ECore named element is passed instead of a property name. This is useful if the
	 * parameter that should be configured stems from a static profile
	 *
	 * @param instance
	 *            The instance of which an attribute should be configured.
	 * @param property
	 *            The name of the property (denoted by an ENamedElement) that should be configured
	 * @param value
	 *            its value in form of an element of an ECore enumerator value
	 */
	public static void configureProperty(InstanceSpecification instance, ENamedElement property, Enumerator value) {
		configureProperty(instance, property.getName(), value);
	}

	/**
	 * Configure a property for an enumeration. Enumerations are a bit difficult to handle, since the enumeration literal itself
	 * must be created first in form of an instance specification
	 *
	 * @param instance
	 *            The instance of which an attribute should be configured.
	 * @param propertyName
	 *            The name of the property that should be configured
	 * @param value
	 *            its value in form of an element of an ECore enumerator value
	 */
	public static void configureProperty(InstanceSpecification instance, String propertyName, Enumerator value) {
		configureProperty(instance, propertyName, value.getName());
	}

	/**
	 * Configure a property for an enumeration. Enumerations are a bit difficult to handle, since the enumeration literal itself
	 * must be created first in form of an instance specification.
	 *
	 * @param instance
	 *            The instance of which an attribute should be configured.
	 * @param propertyName
	 *            The name of the property that should be configured
	 * @param literalName
	 *            the name of the literal
	 */
	public static void configureEnumProperty(InstanceSpecification instance, String propertyName, String literalName) {
		Classifier extension = DepUtils.getClassifier(instance);
		Property attribute = (Property) ElementUtils.getNamedElementFromList(extension.getAllAttributes(), propertyName);
		if (attribute == null) {
			throw new RuntimeException(String.format(Messages.DepPlanUtils_CannotFindAttribute, propertyName, extension.getName()));
		}
		configureEnumProperty(instance, attribute, literalName);
	}

	public static void configureEnumProperty(InstanceSpecification instance, Property attribute, String literalName) {
		if (attribute.getType() instanceof Enumeration) {
			Enumeration enumeration = (Enumeration) attribute.getType();
			for (EnumerationLiteral enumLiteral : enumeration.getOwnedLiterals()) {
				if (enumLiteral.getLabel().equals(literalName)) {
					Slot slotEnumVal = instance.createSlot();
					slotEnumVal.setDefiningFeature(attribute);
					InstanceValue enumLitValue = (InstanceValue)
							slotEnumVal.createValue("value for " + attribute.getName(), attribute.getType(), UMLPackage.eINSTANCE.getInstanceValue()); //$NON-NLS-1$
					enumLitValue.setInstance(enumLiteral);
					break;
				}
			}
		}
	}

	/**
	 * Update the names of instances within a deployment plan to correspond to changes
	 * in the hierarchy. In particular, flattening of interaction components requires this update.
	 *
	 * @param instance an instance specification
	 * @param name the new name for this instance specification
	 */
	public static void updateInstanceNames(InstanceSpecification instance, String name) {
		instance.setName(name);
		for (Slot slot : instance.getSlots()) {
			InstanceSpecification subInstance = DepUtils.getInstance(slot);
			StructuralFeature sf = slot.getDefiningFeature();
			if ((subInstance != null) && !DepUtils.isShared(slot)) {
				updateInstanceNames(subInstance, name + DeployConstants.SEP_CHAR + sf.getName());
			}
		}
	}
}
