/*******************************************************************************
 * Copyright (c) 2005 - 2006 Joel Cheuoua & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Joel Cheuoua - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor.util;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.codegen.jet.editor.codeassist.JETTemplateCompletionProcessor;
import org.eclipse.emf.codegen.jet.editor.codeassist.JETTemplateDirectiveCompletionProcessor;
import org.eclipse.emf.codegen.jet.editor.presentation.JETReconcilingStrategy;
import org.eclipse.emf.codegen.jet.editor.presentation.JETTextEditor;
import org.eclipse.emf.codegen.jet.editor.scanners.JETTemplateDirectiveScanner;
import org.eclipse.emf.codegen.jet.editor.scanners.JETTemplateExpressionScanner;
import org.eclipse.emf.codegen.jet.editor.scanners.JETTemplateJavaDocScanner;
import org.eclipse.emf.codegen.jet.editor.scanners.JETTemplateJavaMultiLineCommentScanner;
import org.eclipse.emf.codegen.jet.editor.scanners.JETTemplateJavaSingleLineCommentScanner;
import org.eclipse.emf.codegen.jet.editor.scanners.JETTemplateJavaStringScanner;
import org.eclipse.emf.codegen.jet.editor.scanners.JETTemplatePartitionScanner;
import org.eclipse.emf.codegen.jet.editor.scanners.JETTemplateScriptletScanner;
import org.eclipse.emf.codegen.jet.editor.util.hovers.JETTemplateScriptletHover;
import org.eclipse.jdt.internal.ui.text.ContentAssistPreference;
import org.eclipse.jdt.internal.ui.text.java.hover.JavaInformationProvider;
import org.eclipse.jdt.ui.PreferenceConstants;
import org.eclipse.jdt.ui.text.IColorManager;
import org.eclipse.jdt.ui.text.IJavaPartitions;
import org.eclipse.jdt.ui.text.JavaSourceViewerConfiguration;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.DefaultUndoManager;
import org.eclipse.jface.text.IAutoEditStrategy;
import org.eclipse.jface.text.ITextHover;
import org.eclipse.jface.text.IUndoManager;
import org.eclipse.jface.text.contentassist.ContentAssistant;
import org.eclipse.jface.text.contentassist.IContentAssistant;
import org.eclipse.jface.text.information.IInformationPresenter;
import org.eclipse.jface.text.information.IInformationProvider;
import org.eclipse.jface.text.information.InformationPresenter;
import org.eclipse.jface.text.presentation.IPresentationReconciler;
import org.eclipse.jface.text.presentation.PresentationReconciler;
import org.eclipse.jface.text.reconciler.IReconciler;
import org.eclipse.jface.text.reconciler.MonoReconciler;
import org.eclipse.jface.text.rules.DefaultDamagerRepairer;
import org.eclipse.jface.text.rules.RuleBasedScanner;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.ui.texteditor.AbstractDecoratedTextEditorPreferenceConstants;

/**
 * @author jcheuoua
 * 
 * To change the template for this generated type comment go to Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code
 * and Comments
 * @version $Revision: 1.2 $
 */
public class JETTemplateSourceViewerConfiguration extends JavaSourceViewerConfiguration {

  private RuleBasedScanner jetScriptletScanner;

  private RuleBasedScanner jetDirectiveScanner;

  private RuleBasedScanner jetExpressionScanner;

  private RuleBasedScanner javaStringScanner;

  private RuleBasedScanner javaMultiLineCommentScanner;

  private RuleBasedScanner javaSingleLineCommentScanner;

  private RuleBasedScanner javaDocScanner;

  private IPreferenceStore store;

  private JETTextEditor jetEditor;

  /**
   * Constructor for JETTemplateSourceViewerConfiguration.
   * 
   * @param colorManager
   *          IColorManager
   * @param store
   *          IPreferenceStore
   * @param editor
   *          JETTextEditor
   * @param partitioning
   *          String
   */
  public JETTemplateSourceViewerConfiguration(IColorManager colorManager, IPreferenceStore store, JETTextEditor editor,
      String partitioning) {
    super(colorManager, store, editor, partitioning);
    this.store = store;
    this.jetEditor = editor;
  }

  /**
   * Method getConfiguredContentTypes.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @return String[]
   */
  public String[] getConfiguredContentTypes(ISourceViewer sourceViewer) {
    return JETTemplatePartitionScanner.getLegalContentTypes();
  }

  /**
   * Method getJSPExpressionScanner.
   * 
   * @return RuleBasedScanner
   */
  protected RuleBasedScanner getJSPExpressionScanner() {
    if (jetExpressionScanner == null)
      jetExpressionScanner = new JETTemplateExpressionScanner(getColorManager(), store);
    return jetExpressionScanner;
  }

  /**
   * Method getJSPScriptletScanner.
   * 
   * @return RuleBasedScanner
   */
  protected RuleBasedScanner getJSPScriptletScanner() {
    if (jetScriptletScanner == null)
      jetScriptletScanner = new JETTemplateScriptletScanner(getColorManager(), store);
    return jetScriptletScanner;
  }

  /**
   * Method getJSPDirectiveScanner.
   * 
   * @return RuleBasedScanner
   */
  protected RuleBasedScanner getJSPDirectiveScanner() {
    if (jetDirectiveScanner == null)
      jetDirectiveScanner = new JETTemplateDirectiveScanner(getColorManager(), store);
    return jetDirectiveScanner;
  }

  /**
   * Method getJavaStringDirectiveScanner.
   * 
   * @return RuleBasedScanner
   */
  protected RuleBasedScanner getJavaStringDirectiveScanner() {
    if (javaStringScanner == null)
      javaStringScanner = new JETTemplateJavaStringScanner(getColorManager(), store);
    return javaStringScanner;
  }

  /**
   * Method getJavaMultiLineCommentDirectiveScanner.
   * 
   * @return RuleBasedScanner
   */
  protected RuleBasedScanner getJavaMultiLineCommentDirectiveScanner() {
    if (javaMultiLineCommentScanner == null)
      javaMultiLineCommentScanner = new JETTemplateJavaMultiLineCommentScanner(getColorManager(), store);
    return javaMultiLineCommentScanner;
  }

  /**
   * Method getJavaSingleLineCommentDirectiveScanner.
   * 
   * @return RuleBasedScanner
   */
  protected RuleBasedScanner getJavaSingleLineCommentDirectiveScanner() {
    if (javaSingleLineCommentScanner == null)
      javaSingleLineCommentScanner = new JETTemplateJavaSingleLineCommentScanner(getColorManager(), store);
    return javaSingleLineCommentScanner;
  }

  /**
   * Method getJavaDocDirectiveScanner.
   * 
   * @return RuleBasedScanner
   */
  protected RuleBasedScanner getJavaDocDirectiveScanner() {
    if (javaDocScanner == null)
      javaDocScanner = new JETTemplateJavaDocScanner(getColorManager(), store);
    return javaDocScanner;
  }

  /**
   * Method getReconciler.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @return IReconciler
   */
  public IReconciler getReconciler(ISourceViewer sourceViewer) {
    if (jetEditor != null && jetEditor.isEditable()) {

      JETReconcilingStrategy strategy = new JETReconcilingStrategy(sourceViewer, jetEditor);
      MonoReconciler reconciler = new MonoReconciler(strategy, false);      
      reconciler.setProgressMonitor(new NullProgressMonitor());
      reconciler.setDelay(500);

      return reconciler;
    }
    return null;
  }

  /**
   * Method getPresentationReconciler.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @return IPresentationReconciler
   */
  public IPresentationReconciler getPresentationReconciler(ISourceViewer sourceViewer) {
    PresentationReconciler reconciler = (PresentationReconciler) super.getPresentationReconciler(sourceViewer);

    DefaultDamagerRepairer dr = new DefaultDamagerRepairer(getJSPScriptletScanner());
    reconciler.setDamager(dr, JETTemplatePartitionScanner.JET_SCRIPTLET);
    reconciler.setRepairer(dr, JETTemplatePartitionScanner.JET_SCRIPTLET);

    dr = new DefaultDamagerRepairer(getJSPExpressionScanner());
    reconciler.setDamager(dr, JETTemplatePartitionScanner.JET_EXPRESSION);
    reconciler.setRepairer(dr, JETTemplatePartitionScanner.JET_EXPRESSION);

    dr = new DefaultDamagerRepairer(getJSPDirectiveScanner());
    reconciler.setDamager(dr, JETTemplatePartitionScanner.JET_DIRECTIVE);
    reconciler.setRepairer(dr, JETTemplatePartitionScanner.JET_DIRECTIVE);

    dr = new DefaultDamagerRepairer(getJavaStringDirectiveScanner());
    reconciler.setDamager(dr, JETTemplatePartitionScanner.JAVA_STRING);
    reconciler.setRepairer(dr, JETTemplatePartitionScanner.JAVA_STRING);

    dr = new DefaultDamagerRepairer(getJavaStringDirectiveScanner());
    reconciler.setDamager(dr, JETTemplatePartitionScanner.JAVA_CHARACTER);
    reconciler.setRepairer(dr, JETTemplatePartitionScanner.JAVA_CHARACTER);

    dr = new DefaultDamagerRepairer(getJavaMultiLineCommentDirectiveScanner());
    reconciler.setDamager(dr, JETTemplatePartitionScanner.JAVA_MULTI_LINE_COMMENT);
    reconciler.setRepairer(dr, JETTemplatePartitionScanner.JAVA_MULTI_LINE_COMMENT);

    dr = new DefaultDamagerRepairer(getJavaSingleLineCommentDirectiveScanner());
    reconciler.setDamager(dr, JETTemplatePartitionScanner.JAVA_SINGLE_LINE_COMMENT);
    reconciler.setRepairer(dr, JETTemplatePartitionScanner.JAVA_SINGLE_LINE_COMMENT);

    dr = new DefaultDamagerRepairer(getJavaDocDirectiveScanner());
    reconciler.setDamager(dr, JETTemplatePartitionScanner.JAVA_DOC);
    reconciler.setRepairer(dr, JETTemplatePartitionScanner.JAVA_DOC);
    return reconciler;
  }

  /**
   * Method getContentAssistant.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @return IContentAssistant
   */
  public IContentAssistant getContentAssistant(ISourceViewer sourceViewer) {

    char scriptletChars[] = store.getString(PreferenceConstants.CODEASSIST_AUTOACTIVATION_TRIGGERS_JAVA).toCharArray();
    if (getEditor() != null) {

      ContentAssistant assistant = new ContentAssistant();
      assistant.setContentAssistProcessor(new JETTemplateDirectiveCompletionProcessor(),
          JETTemplatePartitionScanner.JET_DIRECTIVE);

      JETTemplateCompletionProcessor jetProcessor = new JETTemplateCompletionProcessor(jetEditor);
      jetProcessor.setCompletionProposalAutoActivationCharacters(scriptletChars);
      assistant.setContentAssistProcessor(jetProcessor, JETTemplatePartitionScanner.JET_SCRIPTLET);
      assistant.setContentAssistProcessor(jetProcessor, JETTemplatePartitionScanner.JET_EXPRESSION);
      assistant.setContentAssistProcessor(jetProcessor, IJavaPartitions.JAVA_DOC);
      assistant.setContentAssistProcessor(jetProcessor, IJavaPartitions.JAVA_STRING);
      assistant.setContentAssistProcessor(jetProcessor, IJavaPartitions.JAVA_MULTI_LINE_COMMENT);
      assistant.setContentAssistProcessor(jetProcessor, IJavaPartitions.JAVA_SINGLE_LINE_COMMENT);
      assistant.setContentAssistProcessor(jetProcessor, IJavaPartitions.JAVA_CHARACTER);

      ContentAssistPreference.configure(assistant, store);

      assistant.setContextInformationPopupOrientation(ContentAssistant.CONTEXT_INFO_ABOVE);
      assistant.setInformationControlCreator(getInformationControlCreator(sourceViewer));

      assistant.enableAutoInsert(false);
      assistant.enableAutoActivation(true);
      return assistant;
    }
    return null;
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.jface.text.source.SourceViewerConfiguration#getAutoIndentStrategy(org.eclipse.jface.text.source.ISourceViewer,
   *      java.lang.String)
   */
  /**
   * Method getAutoEditStrategies.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @param contentType
   *          String
   * @return IAutoEditStrategy[]
   */
  public IAutoEditStrategy[] getAutoEditStrategies(ISourceViewer sourceViewer, String contentType) {
    if (JETTemplatePartitionScanner.JET_DIRECTIVE.equals(contentType)
        || JETTemplatePartitionScanner.JET_EXPRESSION.equals(contentType)
        || JETTemplatePartitionScanner.JET_SCRIPTLET.equals(contentType))
      return new IAutoEditStrategy[] { new JETTemplateAutoEditStrategy() };
    else
      return super.getAutoEditStrategies(sourceViewer, contentType);
  }

  /**
   * Method getInformationPresenter.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @return IInformationPresenter
   */
  public IInformationPresenter getInformationPresenter(ISourceViewer sourceViewer) {
    InformationPresenter presenter = (InformationPresenter) super.getInformationPresenter(sourceViewer);
    IInformationProvider provider = new JavaInformationProvider(jetEditor);
    presenter.setInformationProvider(provider, JETTemplatePartitionScanner.JET_SCRIPTLET);
    return presenter;
  }

  /**
   * Method getTextHover.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @param contentType
   *          String
   * @param stateMask
   *          int
   * @return ITextHover
   */
  public ITextHover getTextHover(ISourceViewer sourceViewer, String contentType, int stateMask) {
    return super.getTextHover(sourceViewer, contentType, stateMask);
  }

  /**
   * Method getTextHover.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @param contentType
   *          String
   * @return ITextHover
   */
  public ITextHover getTextHover(ISourceViewer sourceViewer, String contentType) {
    if (contentType == JETTemplatePartitionScanner.JET_SCRIPTLET) {
      JETTemplateScriptletHover hover = new JETTemplateScriptletHover(jetEditor);
      return hover;
    } else {
      return super.getTextHover(sourceViewer, contentType);
    }
  }

  /*
   * @see org.eclipse.jface.text.source.SourceViewerConfiguration#getUndoManager(org.eclipse.jface.text.source.ISourceViewer)
   * @since 3.1
   */
  public IUndoManager getUndoManager(ISourceViewer sourceViewer) {
    if (fPreferenceStore == null)
      return super.getUndoManager(sourceViewer);
    int undoHistorySize = Math.max(25, fPreferenceStore
        .getInt(AbstractDecoratedTextEditorPreferenceConstants.EDITOR_UNDO_HISTORY_SIZE));
    return new DefaultUndoManager(undoHistorySize);
  }
}