/*******************************************************************************
 * Copyright (c) 2005 - 2006 Joel Cheuoua & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Joel Cheuoua - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor.presentation;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.net.MalformedURLException;
import java.util.HashSet;
import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.codegen.jet.JETException;
import org.eclipse.emf.codegen.jet.JETNature;
import org.eclipse.emf.codegen.jet.JETSkeleton;
import org.eclipse.emf.codegen.jet.editor.JETEditorPlugin;
import org.eclipse.emf.codegen.jet.editor.dyncheck.JETAnnotationModel;
import org.eclipse.emf.codegen.jet.editor.dyncheck.JETProblemAnnotation;
import org.eclipse.emf.codegen.jet.editor.util.JETCompilerExt;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IProblemRequestor;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.WorkingCopyOwner;
import org.eclipse.jdt.core.compiler.IProblem;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.reconciler.DirtyRegion;
import org.eclipse.jface.text.reconciler.IReconcilingStrategy;
import org.eclipse.jface.text.reconciler.IReconcilingStrategyExtension;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;

/**
 */
public class JETReconcilingStrategy implements IReconcilingStrategy, IReconcilingStrategyExtension, IProblemRequestor {
  private JETTextEditor jetEditor;

  private ISourceViewer sourceViewer;

  private IDocument currentDocument;

  private IFile jetFile;

  private JETCompilerExt jetCompiler;

  private ICompilationUnit compilationUnit;

  private HashSet problems;

  /**
   * Constructor for JETReconcilingStrategy.
   * 
   * @param sourceViewer
   *          ISourceViewer
   * @param jetEditor
   *          JETTextEditor
   */
  public JETReconcilingStrategy(ISourceViewer sourceViewer, JETTextEditor jetEditor) {
    this.jetEditor = jetEditor;
    this.sourceViewer = sourceViewer;
  }

  /**
   * Method setDocument.
   * 
   * @param document
   *          IDocument
   * @see org.eclipse.jface.text.reconciler.IReconcilingStrategy#setDocument(IDocument)
   */
  public void setDocument(IDocument document) {
    this.currentDocument = document;
  }

  /**
   * Method reconcile.
   * 
   * @param dirtyRegion
   *          DirtyRegion
   * @param subRegion
   *          IRegion
   * @see org.eclipse.jface.text.reconciler.IReconcilingStrategy#reconcile(DirtyRegion, IRegion)
   */
  public void reconcile(DirtyRegion dirtyRegion, IRegion subRegion) {
    reconcile();
  }

  /**
   * Method reconcile.
   * 
   * @param partition
   *          IRegion
   * @see org.eclipse.jface.text.reconciler.IReconcilingStrategy#reconcile(IRegion)
   */
  public void reconcile(IRegion partition) {
    reconcile();
  }

  private void reconcile() {
    IEditorInput editorInput = jetEditor.getEditorInput();
    if (editorInput != null) {
      jetFile = ((IFileEditorInput) editorInput).getFile();
      IProject project = jetFile.getProject();
      try {
        IJavaProject javaProject = JavaCore.create(project);
        JETNature jetNature = JETNature.getRuntime(project);
        if (javaProject == null || jetNature == null)
          return;
        IContainer parent = jetFile.getParent();
        while (parent != null && !(jetNature.getTemplateContainers().contains(parent))) {
          parent = parent.getParent();
        }
        if (parent == null) {
          jetNature.getTemplateContainers().add(jetFile.getParent());
          jetNature.setTemplateContainers(jetNature.getTemplateContainers());
        }
        if (jetNature.getJavaSourceContainer() == null)
          computeJavaSourceContainer(jetNature, javaProject);
        
        // Compute the corresponding template's java class content
        String uri = jetFile.getLocation().toFile().getAbsoluteFile().toURL().toString();
        String jetSource = currentDocument.get();
        ByteArrayInputStream is = new ByteArrayInputStream(jetSource.getBytes());
        jetCompiler = new JETCompilerExt(uri, is);
        jetCompiler.parse();
        ByteArrayOutputStream arrayOutputStream = new ByteArrayOutputStream();
        jetCompiler.generate(arrayOutputStream);

        JETSkeleton skeleton = jetCompiler.getSkeleton();
        computeCompilationUnit(javaProject, arrayOutputStream, skeleton);
        if (compilationUnit == null)
          return;
        WorkingCopyOwner owner = new WorkingCopyOwner() {/* non shared working copy */};

        ICompilationUnit copy = compilationUnit.getWorkingCopy(owner, this, null);

        JETAnnotationModel annotationModel = (JETAnnotationModel) sourceViewer.getAnnotationModel();
        if (annotationModel != null) {
          annotationModel.setFireChanges(false);
  
          // clean the annotation model
          Iterator it = annotationModel.getAnnotationIterator();
          while (it.hasNext()) {
            Annotation annotation = (Annotation) it.next();
            if (annotation instanceof JETProblemAnnotation) {
              annotationModel.removeAnnotation(annotation);
            }
          }
  
          copy.reconcile(ICompilationUnit.NO_AST, true, owner, null);
  
          annotationModel.setFireChanges(true);
          annotationModel.fireAnnotationModelChanged();
        }
      } catch (CoreException e) {
        JETEditorPlugin.getDefault().log(e);
      } catch (MalformedURLException e) {
        JETEditorPlugin.getDefault().log(e);
      } catch (JETException e) {
        JETEditorPlugin.getDefault().log(e);
      } 
    }
  }

  private void computeCompilationUnit(IJavaProject javaProject, ByteArrayOutputStream arrayOutputStream, JETSkeleton skeleton) throws JavaModelException {
    IPackageFragmentRoot[] roots = javaProject.getPackageFragmentRoots();
    for (int i = 0; i < roots.length; i++) {
      IPackageFragmentRoot root = roots[i];
      IPackageFragment packageFragment = root.getPackageFragment(skeleton.getPackageName());
      if (packageFragment != null && packageFragment.exists()) {
        compilationUnit = packageFragment.createCompilationUnit(skeleton.getClassName() + ".java",
            arrayOutputStream.toString(), true, new NullProgressMonitor());
        break;
      }
    }
  }

  private void computeJavaSourceContainer(JETNature jetNature, IJavaProject javaProject) throws JavaModelException {
    IPackageFragmentRoot[] fragmentRoots = javaProject.getPackageFragmentRoots();
    for (int i = 0; i < fragmentRoots.length; i++) {
      IPackageFragmentRoot fragmentRoot = fragmentRoots[i];
      if (fragmentRoot.getKind() == IPackageFragmentRoot.K_SOURCE) {
        jetNature.setJavaSourceContainer((IContainer) fragmentRoot.getCorrespondingResource());
        break;
      }
    }
  }
  
  /**
   * Method acceptProblem.
   * 
   * @param problem
   *          IProblem
   * @see org.eclipse.jdt.core.IProblemRequestor#acceptProblem(IProblem)
   */
  public void acceptProblem(IProblem problem) {
    problems.add(problem);
  }

  /**
   * Method beginReporting.
   * 
   * @see org.eclipse.jdt.core.IProblemRequestor#beginReporting()
   */
  public void beginReporting() {
    if (problems == null)
      problems = new HashSet();
    problems.clear();
  }

  /**
   * Method endReporting.
   * 
   * @see org.eclipse.jdt.core.IProblemRequestor#endReporting()
   */
  public void endReporting() {
    for (Iterator iter = problems.iterator(); iter.hasNext();) {
      IProblem problem = (IProblem) iter.next();
      IAnnotationModel annotationModel = sourceViewer.getAnnotationModel();
      try {

        // String jetContent = currentDocument.get();
        // String javaSource = generatedSource;

        int offset = problem.getSourceStart();
        int length = problem.getSourceEnd() - problem.getSourceStart() + 1;
        JETCompilerExt.Range javaRange = jetCompiler.getJavaRange(offset);
        JETCompilerExt.Range jetRange = javaRange == null ? null : jetCompiler.getJetRange(javaRange);

        int jetOffset = (jetRange == null) ? 0 : jetRange.start + (offset - javaRange.start);

        JETProblemAnnotation annotation = new JETProblemAnnotation(problem, compilationUnit);
        Position position = new Position(jetOffset, length);
        annotationModel.addAnnotation(annotation, position);
      } catch (Exception e) {
        JETEditorPlugin.getDefault().log(e);
      }
    }
  }

  /**
   * Method isActive.
   * 
   * @return boolean
   * @see org.eclipse.jdt.core.IProblemRequestor#isActive()
   */
  public boolean isActive() {
    return true;
  }

  public void setProgressMonitor(IProgressMonitor monitor) {
    // TODO Auto-generated method stub
    
  }

  public void initialReconcile() {
    reconcile();
  }
}
