/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.core.keystore.crl;

import java.io.Closeable;
import java.io.File;
import java.net.URI;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertStore;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.eclipse.kura.core.keystore.crl.CRLStore;
import org.eclipse.kura.core.keystore.crl.StoredCRL;
import org.eclipse.kura.core.keystore.util.CRLUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CRLManager
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(CRLManager.class);
    private final CRLStore store;
    private final ScheduledExecutorService updateExecutor = Executors.newSingleThreadScheduledExecutor();
    private final ExecutorService downloadExecutor = Executors.newCachedThreadPool();
    private final List<DistributionPointState> referencedDistributionPoints = new ArrayList<DistributionPointState>();
    private final long forceUpdateIntervalNanos;
    private final long periodicReckeckIntervalMs;
    private final CRLVerifier verifier;
    private Optional<ScheduledFuture<?>> updateTask = Optional.empty();
    private Optional<Listener> listener;

    public CRLManager(File storeFile, long storeDelayMs, long periodicRecheckIntervalMs, long forceUpdateIntervalMs, CRLVerifier verifier) {
        this.store = new CRLStore(storeFile, storeDelayMs);
        this.periodicReckeckIntervalMs = periodicRecheckIntervalMs;
        this.forceUpdateIntervalNanos = Duration.ofMillis(forceUpdateIntervalMs).toNanos();
        this.verifier = verifier;
        this.requestUpdate();
    }

    public void setListener(Optional<Listener> listener) {
        this.listener = listener;
    }

    public synchronized boolean addDistributionPoint(Set<URI> uris) {
        logger.info("referencing distribution points: {}", uris);
        Optional<DistributionPointState> existing = this.referencedDistributionPoints.stream().filter(p -> ((DistributionPointState)p).distributionPoints.equals(uris)).findAny();
        if (existing.isPresent()) {
            existing.get().ref();
            return false;
        }
        this.referencedDistributionPoints.add(new DistributionPointState(uris));
        this.requestUpdate();
        return true;
    }

    public synchronized boolean removeDistributionPoint(Set<URI> uris) {
        logger.info("unreferencing distribution points: {}", uris);
        if (this.referencedDistributionPoints.removeIf(p -> ((DistributionPointState)p).distributionPoints.equals(uris) && p.unref() <= 0)) {
            this.requestUpdate();
            return true;
        }
        return false;
    }

    public boolean addTrustedCertificate(X509Certificate entry) {
        Set<URI> distributionPoints;
        try {
            distributionPoints = CRLUtil.getCrlURIs(entry);
        }
        catch (Exception e) {
            logger.warn("failed to get distribution points for {}", (Object)entry.getSubjectX500Principal(), (Object)e);
            return false;
        }
        if (distributionPoints.isEmpty()) {
            logger.info("certificate {} has no CRL distribution points", (Object)entry.getSubjectX500Principal());
            return false;
        }
        return this.addDistributionPoint(distributionPoints);
    }

    public boolean removeTrustedCertificate(X509Certificate entry) {
        try {
            Set<URI> distributionPoints = CRLUtil.getCrlURIs(entry);
            return this.removeDistributionPoint(distributionPoints);
        }
        catch (Exception e) {
            logger.warn("failed to get distribution points for {}", (Object)entry.getSubjectX500Principal(), (Object)e);
            return false;
        }
    }

    public synchronized List<X509CRL> getCrls() {
        return this.store.getCRLs().stream().map(StoredCRL::getCrl).collect(Collectors.toList());
    }

    public CertStore getCertStore() throws InvalidAlgorithmParameterException, NoSuchAlgorithmException {
        return this.store.getCertStore();
    }

    @Override
    public void close() {
        this.listener = Optional.empty();
        this.updateExecutor.shutdown();
        this.downloadExecutor.shutdown();
    }

    private void requestUpdate() {
        if (this.updateTask.isPresent()) {
            this.updateTask.get().cancel(false);
        }
        this.updateTask = Optional.of(this.updateExecutor.scheduleWithFixedDelay(this::update, 5000L, this.periodicReckeckIntervalMs, TimeUnit.MILLISECONDS));
    }

    private synchronized void update() {
        boolean changed = false;
        long now = System.nanoTime();
        for (DistributionPointState state : this.referencedDistributionPoints) {
            Optional<StoredCRL> storedCrl = this.store.getCRLs().stream().filter(c -> c.getDistributionPoints().equals(state.distributionPoints)).findAny();
            if (storedCrl.isPresent() && !storedCrl.get().isExpired() && state.lastDownloadInstantNanos.isPresent() && now - state.lastDownloadInstantNanos.getAsLong() <= this.forceUpdateIntervalNanos) continue;
            CompletableFuture<X509CRL> future = CRLUtil.fetchCRL(state.distributionPoints, this.downloadExecutor);
            try {
                X509CRL crl = future.get(1L, TimeUnit.MINUTES);
                changed |= this.validateAndStoreCRL(now, state, storedCrl, crl);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.warn("failed to download CRL", (Throwable)e);
                future.cancel(true);
            }
            catch (Exception e) {
                logger.warn("failed to download CRL", (Throwable)e);
                future.cancel(true);
            }
        }
        if (changed |= this.store.removeCRLs(c -> this.referencedDistributionPoints.stream().noneMatch(p -> ((DistributionPointState)p).distributionPoints.equals(c.getDistributionPoints())))) {
            this.listener.ifPresent(Listener::onCRLCacheChanged);
        }
    }

    private boolean validateAndStoreCRL(long now, DistributionPointState state, Optional<StoredCRL> storedCrl, X509CRL newCrl) {
        if (storedCrl.isPresent()) {
            X509CRL stored = storedCrl.get().getCrl();
            if (stored.equals(newCrl)) {
                logger.info("current CRL is up to date");
                state.lastDownloadInstantNanos = OptionalLong.of(now);
                return false;
            }
            if (!stored.getIssuerX500Principal().equals(newCrl.getIssuerX500Principal())) {
                logger.warn("CRL issuer differs, not updating CRL");
                return false;
            }
        }
        if (this.verifier.verifyCRL(newCrl)) {
            this.store.storeCRL(new StoredCRL(state.distributionPoints, newCrl));
            state.lastDownloadInstantNanos = OptionalLong.of(now);
            return true;
        }
        logger.warn("CRL verification failed");
        return false;
    }

    public static interface CRLVerifier {
        public boolean verifyCRL(X509CRL var1);
    }

    private class DistributionPointState {
        private int refCnt = 1;
        private OptionalLong lastDownloadInstantNanos = OptionalLong.empty();
        private final Set<URI> distributionPoints;

        public DistributionPointState(Set<URI> distributionPoints) {
            this.distributionPoints = distributionPoints;
        }

        int ref() {
            ++this.refCnt;
            return this.refCnt;
        }

        int unref() {
            --this.refCnt;
            return this.refCnt;
        }
    }

    public static interface Listener {
        public void onCRLCacheChanged();
    }
}

