/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelTypeExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.eclipse.EclipseProjectHelper;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.lib.ModelUtils;
import fr.inria.diverse.melange.metamodel.melange.Inheritance;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.Metamodel;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import fr.inria.diverse.melange.metamodel.melange.ModelingElement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.core.resources.IProject;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@Singleton
@SuppressWarnings("all")
public class EPackageProvider {
  @Inject
  private ModelUtils modelUtils;
  
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;
  
  @Inject
  @Extension
  private ModelTypeExtensions _modelTypeExtensions;
  
  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;
  
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  @Inject
  @Extension
  private EclipseProjectHelper _eclipseProjectHelper;
  
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;
  
  private Map<Resource, SetMultimap<String, EPackage>> dispatchPackages = new HashMap<Resource, SetMultimap<String, EPackage>>();
  
  private Map<Resource, SetMultimap<String, GenModel>> dispatchGenmodels = new HashMap<Resource, SetMultimap<String, GenModel>>();
  
  public Set<EPackage> getPackages(final ModelingElement m) {
    if ((m == null)) {
      return CollectionLiterals.<EPackage>newHashSet();
    }
    final SetMultimap<String, EPackage> packages = this.getPackageRegistry(m.eResource());
    boolean _containsKey = packages.containsKey(this.getFqn(m));
    boolean _not = (!_containsKey);
    if (_not) {
      if ((m instanceof Metamodel)) {
        final IProject project = this._eclipseProjectHelper.getProject(((Metamodel)m).eResource());
        if ((this._languageExtensions.isGeneratedByMelange(((Metamodel)m).getOwningLanguage()) && (project != null))) {
          boolean _exists = project.getFile(this._languageExtensions.getExternalEcorePath(((Metamodel)m).getOwningLanguage())).exists();
          if (_exists) {
            ((Metamodel)m).setEcoreUri(this._languageExtensions.getExternalEcoreUri(((Metamodel)m).getOwningLanguage()));
          }
        }
      }
      boolean _matched = false;
      if (m instanceof ModelType) {
        boolean _isExtracted = this._modelTypeExtensions.isExtracted(((ModelType)m));
        if (_isExtracted) {
          _matched=true;
          packages.putAll(this.getFqn(m), this.getPackages(((ModelType)m).getExtracted().getSyntax()));
        }
      }
      if (!_matched) {
        if (m instanceof Metamodel) {
          boolean _isEmpty = this._languageExtensions.getSuperLanguages(((Metamodel)m).getOwningLanguage()).isEmpty();
          boolean _not_1 = (!_isEmpty);
          if (_not_1) {
            _matched=true;
            final Function1<Inheritance, Iterable<EPackage>> _function = new Function1<Inheritance, Iterable<EPackage>>() {
              @Override
              public Iterable<EPackage> apply(final Inheritance it) {
                final Function1<EPackage, EPackage> _function = new Function1<EPackage, EPackage>() {
                  @Override
                  public EPackage apply(final EPackage it) {
                    final EPackage copy = EcoreUtil.<EPackage>copy(it);
                    copy.setName(it.getName());
                    copy.setNsPrefix(it.getName());
                    StringConcatenation _builder = new StringConcatenation();
                    String _externalPackageUri = EPackageProvider.this._languageExtensions.getExternalPackageUri(((Metamodel)m).getOwningLanguage());
                    _builder.append(_externalPackageUri);
                    String _replace = EPackageProvider.this._ecoreExtensions.getUniqueId(it).replace(".", "/");
                    _builder.append(_replace);
                    _builder.append("/");
                    copy.setNsURI(_builder.toString());
                    return copy;
                  }
                };
                return IterableExtensions.<EPackage, EPackage>map(EPackageProvider.this.getPackages(it.getTargetLanguage().getSyntax()), _function);
              }
            };
            final Iterable<EPackage> pkgsCopy = Iterables.<EPackage>concat(IterableExtensions.<Inheritance, Iterable<EPackage>>map(Iterables.<Inheritance>filter(((Metamodel)m).getOwningLanguage().getOperators(), Inheritance.class), _function));
            packages.putAll(this.getFqn(m), pkgsCopy);
          }
        }
      }
      if (!_matched) {
        String _ecoreUri = m.getEcoreUri();
        boolean _tripleNotEquals = (_ecoreUri != null);
        if (_tripleNotEquals) {
          _matched=true;
          final EPackage root = this.modelUtils.loadPkg(this.toPlatformURI(m.getEcoreUri()));
          if ((root != null)) {
            this.registerPackages(m, root);
          }
        }
      }
    }
    final Set<EPackage> res = packages.get(this.getFqn(m));
    if ((res == null)) {
      return CollectionLiterals.<EPackage>newHashSet();
    } else {
      return res;
    }
  }
  
  public Set<GenModel> getGenModels(final ModelingElement m) {
    if ((m == null)) {
      return CollectionLiterals.<GenModel>newHashSet();
    }
    final SetMultimap<String, GenModel> genmodels = this.getGenmodelRegistry(m.eResource());
    boolean _containsKey = genmodels.containsKey(this.getFqn(m));
    boolean _not = (!_containsKey);
    if (_not) {
      boolean _isXcore = this._modelingElementExtensions.isXcore(m);
      if (_isXcore) {
        final GenModel gm = this.modelUtils.loadGenmodelFromXcore(this.toPlatformURI(m.getEcoreUri()));
        if ((gm != null)) {
          genmodels.put(this.getFqn(m), gm);
        }
      } else {
        if (((m.getGenmodelUris().size() == 0) && (m.getEcoreUri() != null))) {
          EList<String> _genmodelUris = m.getGenmodelUris();
          String _substring = m.getEcoreUri().substring(0, m.getEcoreUri().lastIndexOf("."));
          String _plus = (_substring + ".genmodel");
          _genmodelUris.add(_plus);
        } else {
          if ((m instanceof Metamodel)) {
            final IProject project = this._eclipseProjectHelper.getProject(((Metamodel)m).eResource());
            if ((this._languageExtensions.isGeneratedByMelange(((Metamodel)m).getOwningLanguage()) && (project != null))) {
              boolean _exists = project.getFile(this._languageExtensions.getExternalGenmodelPath(((Metamodel)m).getOwningLanguage())).exists();
              if (_exists) {
                EList<String> _genmodelUris_1 = ((Metamodel)m).getGenmodelUris();
                String _externalGenmodelUri = this._languageExtensions.getExternalGenmodelUri(((Metamodel)m).getOwningLanguage());
                _genmodelUris_1.add(_externalGenmodelUri);
              }
            }
          }
        }
        final Consumer<String> _function = new Consumer<String>() {
          @Override
          public void accept(final String it) {
            final GenModel gm = EPackageProvider.this.modelUtils.loadGenmodel(EPackageProvider.this.toPlatformURI(it));
            if ((gm != null)) {
              genmodels.put(EPackageProvider.this.getFqn(m), gm);
            }
          }
        };
        m.getGenmodelUris().forEach(_function);
        boolean _containsKey_1 = genmodels.containsKey(this.getFqn(m));
        boolean _not_1 = (!_containsKey_1);
        if (_not_1) {
          if ((m instanceof ModelType)) {
            boolean _isExtracted = this._modelTypeExtensions.isExtracted(((ModelType)m));
            if (_isExtracted) {
              genmodels.put(this.getFqn(m), this._modelTypeExtensions.createTransientGenmodel(((ModelType)m)));
            }
          }
        }
      }
    }
    final Set<GenModel> res = genmodels.get(this.getFqn(m));
    if ((res == null)) {
      return CollectionLiterals.<GenModel>newHashSet();
    } else {
      return res;
    }
  }
  
  /**
   * Register {@link root} and its sub EPackages as packages of {@link modElem}
   */
  public void registerPackages(final ModelingElement modElem, final EPackage root) {
    final SetMultimap<String, EPackage> packages = this.getPackageRegistry(modElem.eResource());
    Set<EPackage> _get = packages.get(this.getFqn(modElem));
    EPackage _findFirst = null;
    if (_get!=null) {
      final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
        @Override
        public Boolean apply(final EPackage it) {
          String _uniqueId = EPackageProvider.this._ecoreExtensions.getUniqueId(it);
          String _uniqueId_1 = EPackageProvider.this._ecoreExtensions.getUniqueId(root);
          return Boolean.valueOf(Objects.equal(_uniqueId, _uniqueId_1));
        }
      };
      _findFirst=IterableExtensions.<EPackage>findFirst(_get, _function);
    }
    final EPackage collider = _findFirst;
    if (((collider == null) && (root != null))) {
      final ArrayList<EPackage> pkgs = CollectionLiterals.<EPackage>newArrayList();
      pkgs.add(root);
      final Function1<EPackage, Boolean> _function_1 = new Function1<EPackage, Boolean>() {
        @Override
        public Boolean apply(final EPackage it) {
          final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
            @Override
            public Boolean apply(final EPackage p) {
              String _uniqueId = EPackageProvider.this._ecoreExtensions.getUniqueId(it);
              String _uniqueId_1 = EPackageProvider.this._ecoreExtensions.getUniqueId(p);
              return Boolean.valueOf(Objects.equal(_uniqueId, _uniqueId_1));
            }
          };
          boolean _exists = IterableExtensions.<EPackage>exists(pkgs, _function);
          return Boolean.valueOf((!_exists));
        }
      };
      Iterable<EPackage> _filter = IterableExtensions.<EPackage>filter(this._ecoreExtensions.getReferencedPkgs(root), _function_1);
      Iterables.<EPackage>addAll(pkgs, _filter);
      packages.putAll(this.getFqn(modElem), pkgs);
      final Function1<EPackage, List<EPackage>> _function_2 = new Function1<EPackage, List<EPackage>>() {
        @Override
        public List<EPackage> apply(final EPackage it) {
          return EPackageProvider.this._ecoreExtensions.getAllSubPkgs(it);
        }
      };
      final Function1<EPackage, Boolean> _function_3 = new Function1<EPackage, Boolean>() {
        @Override
        public Boolean apply(final EPackage it) {
          final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
            @Override
            public Boolean apply(final EPackage p) {
              String _uniqueId = EPackageProvider.this._ecoreExtensions.getUniqueId(it);
              String _uniqueId_1 = EPackageProvider.this._ecoreExtensions.getUniqueId(p);
              return Boolean.valueOf(Objects.equal(_uniqueId, _uniqueId_1));
            }
          };
          boolean _exists = IterableExtensions.<EPackage>exists(pkgs, _function);
          return Boolean.valueOf((!_exists));
        }
      };
      packages.putAll(this.getFqn(modElem), IterableExtensions.<EPackage>filter(Iterables.<EPackage>concat(ListExtensions.<EPackage, List<EPackage>>map(pkgs, _function_2)), _function_3));
    }
  }
  
  public String getFqn(final ModelingElement m) {
    String _xifexpression = null;
    if ((m instanceof Metamodel)) {
      EObject _eContainer = ((Metamodel)m).eContainer();
      _xifexpression = this._iQualifiedNameProvider.getFullyQualifiedName(((Language) _eContainer)).toString();
    } else {
      String _xifexpression_1 = null;
      if ((m instanceof ModelType)) {
        _xifexpression_1 = this._iQualifiedNameProvider.getFullyQualifiedName(m).toString();
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  private String toPlatformURI(final String uri) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _startsWith = uri.startsWith("/");
      if (_startsWith) {
        _builder.append("platform:/resource");
      }
    }
    _builder.append(uri);
    return _builder.toString();
  }
  
  private SetMultimap<String, EPackage> getPackageRegistry(final Resource root) {
    SetMultimap<String, EPackage> res = this.dispatchPackages.get(root);
    if ((res == null)) {
      res = HashMultimap.<String, EPackage>create();
      this.dispatchPackages.put(root, res);
    }
    return res;
  }
  
  private SetMultimap<String, GenModel> getGenmodelRegistry(final Resource root) {
    SetMultimap<String, GenModel> res = this.dispatchGenmodels.get(root);
    if ((res == null)) {
      res = HashMultimap.<String, GenModel>create();
      this.dispatchGenmodels.put(root, res);
    }
    return res;
  }
  
  public void resetFor(final Resource r) {
    this.dispatchPackages.put(r, HashMultimap.<String, EPackage>create());
    this.dispatchGenmodels.put(r, HashMultimap.<String, GenModel>create());
  }
}
