/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef AGI_PALETTE_H
#define AGI_PALETTE_H

namespace Agi {

/**
 * 16 color RGB palette.
 * This array contains the 6-bit RGB values of the EGA palette exported
 * to the console drivers.
 */
static const uint8 PALETTE_EGA[16 * 3] = {
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x2a,
	0x00, 0x2a, 0x00,
	0x00, 0x2a, 0x2a,
	0x2a, 0x00, 0x00,
	0x2a, 0x00, 0x2a,
	0x2a, 0x15, 0x00,
	0x2a, 0x2a, 0x2a,
	0x15, 0x15, 0x15,
	0x15, 0x15, 0x3f,
	0x15, 0x3f, 0x15,
	0x15, 0x3f, 0x3f,
	0x3f, 0x15, 0x15,
	0x3f, 0x15, 0x3f,
	0x3f, 0x3f, 0x15,
	0x3f, 0x3f, 0x3f
};

/**
 * 4 color CGA palette.
 */
static const uint8 PALETTE_CGA[4 * 3] = {
	0x00, 0x00, 0x00, // black
	0x55, 0xff, 0xff, // cyan
	0xff, 0x55, 0xff, // magenta
	0xff, 0xff, 0xff
};

/**
 * 2 color Hercules (green) palette. Using 8-bit RGB values.
 */
static const uint8 PALETTE_HERCULES_GREEN[2 * 3] = {
	0x00, 0x00, 0x00, // black
	0x00, 0xdc, 0x28  // green
};

/**
 * 2 color Hercules (amber) palette. Using 8-bit RGB values.
 */
static const uint8 PALETTE_HERCULES_AMBER[2 * 3] = {
	0x00, 0x00, 0x00, // black
	0xdc, 0xb4, 0x00  // amber
};

/**
 * Atari ST AGI palette.
 * Used by all of the tested Atari ST AGI games
 * from Donald Duck's Playground (1986) to Manhunter II (1989).
 * 16 RGB colors. 3 bits per color component.
 */
static const uint8 PALETTE_ATARI_ST[16 * 3] = {
	0x0, 0x0, 0x0,
	0x0, 0x0, 0x7,
	0x0, 0x4, 0x0,
	0x0, 0x5, 0x4,
	0x5, 0x0, 0x0,
	0x5, 0x3, 0x6,
	0x4, 0x3, 0x0,
	0x5, 0x5, 0x5,
	0x3, 0x3, 0x2,
	0x0, 0x5, 0x7,
	0x0, 0x6, 0x0,
	0x0, 0x7, 0x6,
	0x7, 0x2, 0x3,
	0x7, 0x4, 0x7,
	0x7, 0x7, 0x4,
	0x7, 0x7, 0x7
};

/**
 * Second generation Apple IIGS AGI palette.
 * A 16-color, 12-bit RGB palette.
 *
 * Used by at least the following Apple IIGS AGI versions:
 * 1.003 (Leisure Suit Larry I  v1.0E, intro says 1987)
 * 1.005 (AGI Demo 2            1987-06-30?)
 * 1.006 (King's Quest I        v1.0S 1988-02-23)
 * 1.007 (Police Quest I        v2.0B 1988-04-21 8:00am)
 * 1.013 (King's Quest II       v2.0A 1988-06-16 (CE))
 * 1.013 (Mixed-Up Mother Goose v2.0A 1988-05-31 10:00am)
 * 1.014 (King's Quest III      v2.0A 1988-08-28 (CE))
 * 1.014 (Space Quest II        v2.0A, LOGIC.141 says 1988)
 * 2.004 (Manhunter I           v2.0E 1988-10-05 (CE))
 * 2.006 (King's Quest IV       v1.0K 1988-11-22 (CE))
 * 3.001 (Black Cauldron        v1.0O 1989-02-24 (CE))
 * 3.003 (Gold Rush!            v1.0M 1989-02-28 (CE))
 */
// *NOT* identical to Amiga generation 2 palette
static const uint8 PALETTE_APPLE_II_GS[16 * 3] = {
	0x0, 0x0, 0x0,
	0x0, 0x0, 0xF,
	0x0, 0x8, 0x0,
	0x0, 0xD, 0xB,
	0xC, 0x0, 0x0,
	0xB, 0x7, 0xD,
	0x8, 0x5, 0x0,
	0xB, 0xB, 0xB,
	0x7, 0x7, 0x7,
	0x0, 0xB, 0xF,
	0x0, 0xE, 0x0,
	0x0, 0xF, 0xD,
	0xF, 0x9, 0x8,
	0xD, 0x9, 0xF, // difference between Amiga v2 palette and Apple II GS palette, gotten from emulator (SQ2)
	0xE, 0xE, 0x0,
	0xF, 0xF, 0xF
};

// Re-use Amiga v1 palette for Apple IIgs Space Quest 1
#define PALETTE_APPLE_II_GS_SQ1 PALETTE_AMIGA_V1

/**
 * First generation Amiga & Apple IIGS AGI palette.
 * A 16-color, 12-bit RGB palette.
 *
 * Used by at least the following Amiga AGI versions:
 * 2.082 (King's Quest I   v1.0U 1986)
 * 2.082 (Space Quest I    v1.2  1986)
 * 2.090 (King's Quest III v1.01 1986-11-08)
 * 2.107 (King's Quest II  v2.0J 1987-01-29)
 * x.yyy (Black Cauldron   v2.00 1987-06-14)
 * x.yyy (Larry I          v1.05 1987-06-26)
 *
 * Also used by at least the following Apple IIGS AGI versions:
 * 1.002 (Space Quest I, intro says v2.2 1987)
 */
static const uint8 PALETTE_AMIGA_V1[16 * 3] = {
	0x0, 0x0, 0x0,
	0x0, 0x0, 0xF,
	0x0, 0x8, 0x0,
	0x0, 0xD, 0xB,
	0xC, 0x0, 0x0,
	0xB, 0x7, 0xD,
	0x8, 0x5, 0x0,
	0xB, 0xB, 0xB,
	0x7, 0x7, 0x7,
	0x0, 0xB, 0xF,
	0x0, 0xE, 0x0,
	0x0, 0xF, 0xD,
	0xF, 0x9, 0x8,
	0xF, 0x7, 0x0,
	0xE, 0xE, 0x0,
	0xF, 0xF, 0xF
};

/**
 * Second generation Amiga AGI palette.
 * A 16-color, 12-bit RGB palette.
 *
 * Used by at least the following Amiga AGI versions:
 * 2.202 (Space Quest II v2.0F. Intro says 1988. ScummVM 0.10.0 detects as 1986-12-09)
 */
static const uint8 PALETTE_AMIGA_V2[16 * 3] = {
	0x0, 0x0, 0x0,
	0x0, 0x0, 0xF,
	0x0, 0x8, 0x0,
	0x0, 0xD, 0xB,
	0xC, 0x0, 0x0,
	0xB, 0x7, 0xD,
	0x8, 0x5, 0x0,
	0xB, 0xB, 0xB,
	0x7, 0x7, 0x7,
	0x0, 0xB, 0xF,
	0x0, 0xE, 0x0,
	0x0, 0xF, 0xD,
	0xF, 0x9, 0x8,
	0xD, 0x0, 0xF,
	0xE, 0xE, 0x0,
	0xF, 0xF, 0xF
};

/**
 * Third generation Amiga AGI palette.
 * A 16-color, 12-bit RGB palette.
 *
 * Used by at least the following Amiga AGI versions:
 * 2.310 (Police Quest I   v2.0B 1989-02-22)
 * 2.316 (Gold Rush!       v2.05 1989-03-09)
 * x.yyy (Manhunter I      v1.06 1989-03-18)
 * 2.333 (Manhunter II     v3.06 1989-08-17)
 * 2.333 (King's Quest III v2.15 1989-11-15)
 */
static const uint8 PALETTE_AMIGA_V3[16 * 3] = {
	0x0, 0x0, 0x0,
	0x0, 0x0, 0xB,
	0x0, 0xB, 0x0,
	0x0, 0xB, 0xB,
	0xB, 0x0, 0x0,
	0xB, 0x0, 0xB,
	0xC, 0x7, 0x0,
	0xB, 0xB, 0xB,
	0x7, 0x7, 0x7,
	0x0, 0x0, 0xF,
	0x0, 0xF, 0x0,
	0x0, 0xF, 0xF,
	0xF, 0x0, 0x0,
	0xF, 0x0, 0xF,
	0xF, 0xF, 0x0,
	0xF, 0xF, 0xF
};

/**
 * 16 color amiga-ish palette.
 */
static const uint8 PALETTE_AMIGA_ALT[16 * 3] = {
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x3f,
	0x00, 0x2A, 0x00,
	0x00, 0x2A, 0x2A,
	0x33, 0x00, 0x00,
	0x2f, 0x1c, 0x37,
	0x23, 0x14, 0x00,
	0x2f, 0x2f, 0x2f,
	0x15, 0x15, 0x15,
	0x00, 0x2f, 0x3f,
	0x00, 0x33, 0x15,
	0x15, 0x3F, 0x3F,
	0x3f, 0x27, 0x23,
	0x3f, 0x15, 0x3f,
	0x3b, 0x3b, 0x00,
	0x3F, 0x3F, 0x3F
};

/**
 * 16 color Macintosh palette (CLUT format).
 *
 * Used by at least the following Macintosh AGI versions:
 * ?.??? (Police Quest 1   v1.50 23.03.1988)
 * ?.??? (King's Quest 3   v1.52 11.04.1988)
 */
static const uint16 PALETTE_MACINTOSH_CLUT[16 * 3] = {
	0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0xC000,
	0x0000, 0xA800, 0x0000,
	0x0000, 0xA000, 0xA000,
	0xCE50, 0x0000, 0x0000,
	0xC080, 0x0000, 0xFFFF,
	0xD000, 0x6130, 0x32D0,
	0xC000, 0xC000, 0xC000,
	0x6000, 0x6000, 0x6000,
	0x6800, 0x6800, 0xFFFF,
	0x0000, 0xFFFF, 0x0000,
	0x0000, 0xFFFF, 0xFFFF,
	0xFFFF, 0x5390, 0x64B0,
	0xFFFF, 0x8000, 0x0000,
	0xFFFF, 0xFFFF, 0x0000,
	0xFFFF, 0xFFFF, 0xFFFF
};

/**
 * 16 color Macintosh palette (CLUT format).
 *
 * Used by at least the following Macintosh AGI versions:
 * ?.??? (Gold Rush        v1.78 28.07.1989)
 */
static const uint16 PALETTE_MACINTOSH_CLUT2[16 * 3] = {
	0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0xC000,
	0x6524, 0xC2FF, 0x0000,
	0x0000, 0xA000, 0xA000,
	0xDD6B, 0x08C2, 0x06A2,
	0x8000, 0x0000, 0xFFFF,
	0x93FF, 0x281A, 0x12CC,
	0xC000, 0xC000, 0xC000,
	0x8000, 0x8000, 0x8000,
	0x0000, 0x0000, 0xD400,
	0x0000, 0xFFFF, 0x04F1,
	0x0241, 0xAB54, 0xEAFF,
	0xFFFF, 0xC3DC, 0x8160,
	0xFFFF, 0x648A, 0x028C,
	0xFC00, 0xF37D, 0x052F,
	0xFFFF, 0xFFFF, 0xFFFF
};

/**
 * 16 color Macintosh palette (CLUT format).
 *
 * Used by at least the following Macintosh AGI versions:
 * ?.??? (Space Quest 2    v1.51 04.04.1988)
 */
static const uint16 PALETTE_MACINTOSH_CLUT3[16 * 3] = {
	0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0xC000,
	0x0000, 0xA7FF, 0x0000,
	0x0000, 0x9FFF, 0x9FFF,
	0xCE50, 0x0000, 0x0000,
	0xC079, 0x0000, 0xFFFF,
	0xCFFF, 0x6130, 0x32D0,
	0xC000, 0xC000, 0xC000,
	0x6000, 0x6000, 0x6000,
	0x6800, 0x6800, 0xFFFF,
	0x0000, 0xFFFF, 0x0000,
	0x0000, 0xFFFF, 0xFFFF,
	0xFFFF, 0x538C, 0x64B1,
	0xFDCE, 0x1AC0, 0xFFFF,
	0xFFFF, 0xFFFF, 0x0000,
	0xFFFF, 0xFFFF, 0xFFFF,
};

/**
 * 256 color palette used with AGI256 and AGI256-2 games.
 * Uses full 8 bits per color component.
 * This is NOT the standard VGA palette.
 */
static const uint8 PALETTE_VGA[256 * 3] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0xA8, 0x00, 0xA8, 0x00, 0x00, 0xA8, 0xA8,
	0xA8, 0x00, 0x00, 0xA8, 0x00, 0xA8, 0xA8, 0x54, 0x00, 0xA8, 0xA8, 0xA8,
	0x54, 0x54, 0x54, 0x54, 0x54, 0xFC, 0x54, 0xFC, 0x54, 0x54, 0xFC, 0xFC,
	0xFC, 0x54, 0x54, 0xFC, 0x54, 0xFC, 0xFC, 0xFC, 0x54, 0xFC, 0xFC, 0xFC,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x14, 0x20, 0x20, 0x20, 0x2C, 0x2C, 0x2C,
	0x38, 0x38, 0x38, 0x44, 0x44, 0x44, 0x50, 0x50, 0x50, 0x60, 0x60, 0x60,
	0x70, 0x70, 0x70, 0x80, 0x80, 0x80, 0x90, 0x90, 0x90, 0xA0, 0xA0, 0xA0,
	0xB4, 0xB4, 0xB4, 0xC8, 0xC8, 0xC8, 0xE0, 0xE0, 0xE0, 0xFC, 0xFC, 0xFC,
	0x00, 0x00, 0xFC, 0x40, 0x00, 0xFC, 0x7C, 0x00, 0xFC, 0xBC, 0x00, 0xFC,
	0xFC, 0x00, 0xFC, 0xFC, 0x00, 0xBC, 0xFC, 0x00, 0x7C, 0xFC, 0x00, 0x40,
	0xFC, 0x00, 0x00, 0xFC, 0x40, 0x00, 0xFC, 0x7C, 0x00, 0xFC, 0xBC, 0x00,
	0xFC, 0xFC, 0x00, 0xBC, 0xFC, 0x00, 0x7C, 0xFC, 0x00, 0x40, 0xFC, 0x00,
	0x00, 0xFC, 0x00, 0x00, 0xFC, 0x40, 0x00, 0xFC, 0x7C, 0x00, 0xFC, 0xBC,
	0x00, 0xFC, 0xFC, 0x00, 0xBC, 0xFC, 0x00, 0x7C, 0xFC, 0x00, 0x40, 0xFC,
	0x7C, 0x7C, 0xFC, 0x9C, 0x7C, 0xFC, 0xBC, 0x7C, 0xFC, 0xDC, 0x7C, 0xFC,
	0xFC, 0x7C, 0xFC, 0xFC, 0x7C, 0xDC, 0xFC, 0x7C, 0xBC, 0xFC, 0x7C, 0x9C,
	0xFC, 0x7C, 0x7C, 0xFC, 0x9C, 0x7C, 0xFC, 0xBC, 0x7C, 0xFC, 0xDC, 0x7C,
	0xFC, 0xFC, 0x7C, 0xDC, 0xFC, 0x7C, 0xBC, 0xFC, 0x7C, 0x9C, 0xFC, 0x7C,
	0x7C, 0xFC, 0x7C, 0x7C, 0xFC, 0x9C, 0x7C, 0xFC, 0xBC, 0x7C, 0xFC, 0xDC,
	0x7C, 0xFC, 0xFC, 0x7C, 0xDC, 0xFC, 0x7C, 0xBC, 0xFC, 0x7C, 0x9C, 0xFC,
	0xB4, 0xB4, 0xFC, 0xC4, 0xB4, 0xFC, 0xD8, 0xB4, 0xFC, 0xE8, 0xB4, 0xFC,
	0xFC, 0xB4, 0xFC, 0xFC, 0xB4, 0xE8, 0xFC, 0xB4, 0xD8, 0xFC, 0xB4, 0xC4,
	0xFC, 0xB4, 0xB4, 0xFC, 0xC4, 0xB4, 0xFC, 0xD8, 0xB4, 0xFC, 0xE8, 0xB4,
	0xFC, 0xFC, 0xB4, 0xE8, 0xFC, 0xB4, 0xD8, 0xFC, 0xB4, 0xC4, 0xFC, 0xB4,
	0xB4, 0xFC, 0xB4, 0xB4, 0xFC, 0xC4, 0xB4, 0xFC, 0xD8, 0xB4, 0xFC, 0xE8,
	0xB4, 0xFC, 0xFC, 0xB4, 0xE8, 0xFC, 0xB4, 0xD8, 0xFC, 0xB4, 0xC4, 0xFC,
	0x00, 0x00, 0x70, 0x1C, 0x00, 0x70, 0x38, 0x00, 0x70, 0x54, 0x00, 0x70,
	0x70, 0x00, 0x70, 0x70, 0x00, 0x54, 0x70, 0x00, 0x38, 0x70, 0x00, 0x1C,
	0x70, 0x00, 0x00, 0x70, 0x1C, 0x00, 0x70, 0x38, 0x00, 0x70, 0x54, 0x00,
	0x70, 0x70, 0x00, 0x54, 0x70, 0x00, 0x38, 0x70, 0x00, 0x1C, 0x70, 0x00,
	0x00, 0x70, 0x00, 0x00, 0x70, 0x1C, 0x00, 0x70, 0x38, 0x00, 0x70, 0x54,
	0x00, 0x70, 0x70, 0x00, 0x54, 0x70, 0x00, 0x38, 0x70, 0x00, 0x1C, 0x70,
	0x38, 0x38, 0x70, 0x44, 0x38, 0x70, 0x54, 0x38, 0x70, 0x60, 0x38, 0x70,
	0x70, 0x38, 0x70, 0x70, 0x38, 0x60, 0x70, 0x38, 0x54, 0x70, 0x38, 0x44,
	0x70, 0x38, 0x38, 0x70, 0x44, 0x38, 0x70, 0x54, 0x38, 0x70, 0x60, 0x38,
	0x70, 0x70, 0x38, 0x60, 0x70, 0x38, 0x54, 0x70, 0x38, 0x44, 0x70, 0x38,
	0x38, 0x70, 0x38, 0x38, 0x70, 0x44, 0x38, 0x70, 0x54, 0x38, 0x70, 0x60,
	0x38, 0x70, 0x70, 0x38, 0x60, 0x70, 0x38, 0x54, 0x70, 0x38, 0x44, 0x70,
	0x50, 0x50, 0x70, 0x58, 0x50, 0x70, 0x60, 0x50, 0x70, 0x68, 0x50, 0x70,
	0x70, 0x50, 0x70, 0x70, 0x50, 0x68, 0x70, 0x50, 0x60, 0x70, 0x50, 0x58,
	0x70, 0x50, 0x50, 0x70, 0x58, 0x50, 0x70, 0x60, 0x50, 0x70, 0x68, 0x50,
	0x70, 0x70, 0x50, 0x68, 0x70, 0x50, 0x60, 0x70, 0x50, 0x58, 0x70, 0x50,
	0x50, 0x70, 0x50, 0x50, 0x70, 0x58, 0x50, 0x70, 0x60, 0x50, 0x70, 0x68,
	0x50, 0x70, 0x70, 0x50, 0x68, 0x70, 0x50, 0x60, 0x70, 0x50, 0x58, 0x70,
	0x00, 0x00, 0x40, 0x10, 0x00, 0x40, 0x20, 0x00, 0x40, 0x30, 0x00, 0x40,
	0x40, 0x00, 0x40, 0x40, 0x00, 0x30, 0x40, 0x00, 0x20, 0x40, 0x00, 0x10,
	0x40, 0x00, 0x00, 0x40, 0x10, 0x00, 0x40, 0x20, 0x00, 0x40, 0x30, 0x00,
	0x40, 0x40, 0x00, 0x30, 0x40, 0x00, 0x20, 0x40, 0x00, 0x10, 0x40, 0x00,
	0x00, 0x40, 0x00, 0x00, 0x40, 0x10, 0x00, 0x40, 0x20, 0x00, 0x40, 0x30,
	0x00, 0x40, 0x40, 0x00, 0x30, 0x40, 0x00, 0x20, 0x40, 0x00, 0x10, 0x40,
	0x20, 0x20, 0x40, 0x28, 0x20, 0x40, 0x30, 0x20, 0x40, 0x38, 0x20, 0x40,
	0x40, 0x20, 0x40, 0x40, 0x20, 0x38, 0x40, 0x20, 0x30, 0x40, 0x20, 0x28,
	0x40, 0x20, 0x20, 0x40, 0x28, 0x20, 0x40, 0x30, 0x20, 0x40, 0x38, 0x20,
	0x40, 0x40, 0x20, 0x38, 0x40, 0x20, 0x30, 0x40, 0x20, 0x28, 0x40, 0x20,
	0x20, 0x40, 0x20, 0x20, 0x40, 0x28, 0x20, 0x40, 0x30, 0x20, 0x40, 0x38,
	0x20, 0x40, 0x40, 0x20, 0x38, 0x40, 0x20, 0x30, 0x40, 0x20, 0x28, 0x40,
	0x2C, 0x2C, 0x40, 0x30, 0x2C, 0x40, 0x34, 0x2C, 0x40, 0x3C, 0x2C, 0x40,
	0x40, 0x2C, 0x40, 0x40, 0x2C, 0x3C, 0x40, 0x2C, 0x34, 0x40, 0x2C, 0x30,
	0x40, 0x2C, 0x2C, 0x40, 0x30, 0x2C, 0x40, 0x34, 0x2C, 0x40, 0x3C, 0x2C,
	0x40, 0x40, 0x2C, 0x3C, 0x40, 0x2C, 0x34, 0x40, 0x2C, 0x30, 0x40, 0x2C,
	0x2C, 0x40, 0x2C, 0x2C, 0x40, 0x30, 0x2C, 0x40, 0x34, 0x2C, 0x40, 0x3C,
	0x2C, 0x40, 0x40, 0x2C, 0x3C, 0x40, 0x2C, 0x34, 0x40, 0x2C, 0x30, 0x40,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

} // End of namespace Agi

#endif /* AGI_PALETTE_H */
