/*
 * Copyright (C) 2004-2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG	0

#include "build_config.h"
#include "compiler.h"

#include "assert.h"
#include "stdio.h"
#include "kbd.h"
#include "io.h"
#include "cmos.h"
#include "var.h"
#include "const.h"
#include "segment.h"
#include "ptrace.h"
#include "kbd.h"

#define KBD_CONTROL_KBDINT	0x01
#define KBD_CONTROL_AUXINT	0x02
#define KBD_CONTROL_SELFTEST	0x04
#define KBD_CONTROL_KBDDIS	0x10
#define KBD_CONTROL_AUXDIS	0x20
#define KBD_CONTROL_XLATE	0x40

#define KBD_CCMD_WRITE_MODE	0x60
#define KBD_CCMD_MOUSE_DISABLE	0xa7
#define KBD_CCMD_SELF_TEST	0xaa
#define KBD_CCMD_KBD_ENABLE	0xae
#define KBD_CCMD_PLSE		0xf0 /* pulse: bits 3-0 select which bit */

#ifdef INIT_RM
extern void
kbd_led_update(void);
#endif /* INIT_RM */

#ifdef RUNTIME_RM
CODE16;
void
kbd_led_update(void)
{
	uint8_t new;
	uint8_t old;

	asm volatile("sti\n\t");

	new = (var_get(kbd_shift0) >> 4) & 0x7;
	old = var_get(kbd_state3) & 0x7;

	if (new == old) {
		goto done;
	}

	/* Wait while input buffer full. */
	while (inb(0x64) & 0x02) {
		/* Wait... */
	}
	outb(0xed, 0x60); /* Send "Set LED" Command */

	/* Wait for ACK received flag. */
	while (! (var_get(kbd_state3) & 0x10)) {
		/* Wait... */
	}
	/* Reset ACK received flag. */
	var_put(kbd_state3, var_get(kbd_state3) & ~0x10);

	/* Wait while input buffer full. */
	while (inb(0x64) & 0x02) {
		/* Wait... */
	}
	outb(new, 0x60); /* Send "Set LED" Command Data */

	/* Wait for ACK received flag. */
	while (! (var_get(kbd_state3) & 0x10)) {
		/* Wait... */
	}

	/* Reset ACK received flag. */
	var_put(kbd_state3, var_get(kbd_state3) & ~0x10);

	var_put(kbd_state3, (var_get(kbd_state3) & ~0x7) | (new & 0x7));
done:	;
	asm volatile("cli\n\t");
}

/* FIXME */
#define KBD_TAIL	"0x1a"
#define KBD_HEAD	"0x1c"

static ALWAYS_INLINE short
kbd_keyavailable(void)
{
	short ret;

#if 0
	ret = var_get(kbd_head) - var_get(kbd_tail);
#else
	asm volatile (
	"	movw $0x40, %0\n"
	"	movw %0, %%es\n"
	"	movw %%es:" KBD_HEAD ", %0\n"
	"	subw %%es:" KBD_TAIL ", %0\n"
	: "=r" (ret)
	);
#endif
	return ret;
}

static unsigned short
dequeue_key(short removekey)
{
	uint16_t keyval;
	uint16_t tail;
	uint16_t head;

	tail = var_get(kbd_tail);
	head = var_get(kbd_head);

	if (tail == head) {
		return 0;
	}

	keyval = get_word(VAR_SEG, tail);

	if (removekey) {
		tail += 2;
		if (var_get(kbd_bend) <= tail) {
			tail = var_get(kbd_bstart);
		}
		var_put(kbd_tail, tail);
	}

	return keyval;
}

static short
enqueue_key(uint16_t keyval)
{
	unsigned short tail;
	unsigned short tmp;
	unsigned short head;

	tail = var_get(kbd_tail);
	head = var_get(kbd_head);
	tmp = head;
	head += 2;
	if (var_get(kbd_bend) <= head) {
		head = var_get(kbd_bstart);
	}

	if (head == tail) {
		/* Beep */
		/* FIXME VOSSI */
		return 0;
	}

	put_word(VAR_SEG, tmp, keyval);

	var_put(kbd_head, head);

	return 1;
}

void
kbd_system_reset(void)
{
	outb(KBD_CCMD_PLSE | 0x0e, 0x64);
	/* NOT REACHED */
}

/*
 * KBD:
 *
 * Get keycode from buffer.
 * Keycode is removed from buffer.
 *
 * In:	AH	= 00h
 *
 * Out:	AL	= ASCII code
 *	AH	= scan code
 */
static ALWAYS_INLINE void /* Use ALWAYS_INLINE to fix problem with gcc-4.5! */
bios_16_00xx(struct regs *regs)
{
	/* Wait while no key in buffer. */
	while (! kbd_keyavailable()) {
		asm volatile (
			"sti\n\t"
			"hlt\n\t"
			"cli\n\t"
		);
	}

	/*
	 * Remove keystroke from buffer.
	 */
	AX = dequeue_key(1);
}

/*
 * KBD:
 *
 * Check for keycode in buffer.
 * Keycode is *not* removed from buffer.
 *
 * In:	AH	= 01h
 *
 * Out:	AL	= ASCII character
 * 	AH	= scan code
 * 	F
 * 	 zero	= 0: key available; 1: no key available
 */
static ALWAYS_INLINE void
bios_16_01xx(struct regs *regs)
{
	if (! kbd_keyavailable()) {
		/*
		 * No keystroke available.
		 */
		F |= 1 << 6;	/* Set zero flag. */

	} else {
		/*
		 * Peek keystroke.
		 */
		AX = dequeue_key(0);
		F &= ~(1 << 6);	/* Clear zero flag. */
	}
}

/*
 * KBD: Get shift key states.
 *
 * In:	AH	= 02h
 *
 * Out:	AL
 * 	 Bit0	= Right shift key pressed
 * 	 Bit1	= Left shift key pressed
 * 	 Bit2	= Ctrl key pressed
 * 	 Bit3	= Alt key pressed
 * 	 Bit4	= ScrollLock mode active
 * 	 Bit5	= NumLock mode active
 * 	 Bit6	= CapsLock mode active
 * 	 Bit7	= Insert mode active
 */
static void
bios_16_02xx(struct regs *regs)
{
	AL = var_get(kbd_shift0);
}

/*
 * Set keyboard rate.
 */
static void
bios_16_03xx(struct regs *regs)
{
	/* FIXME VOSSI */
}

/*
 * Store key-stroke into buffer.
 */
static void
bios_16_05xx(struct regs *regs)
{
	if (! enqueue_key(CX)) {
		AL = 1;
	} else {
		AL = 0;
	}
}

/*
 * KBD:
 *
 * get keyboard functionality
 *
 * In:	AH = 09h
 *
 * Out:	AL = supported keyboard functions
 *	 Bit0	= INT 16/AX=0300h supported
 *	 Bit1	= INT 16/AX=0304h supported
 *	 Bit2	= INT 16/AX=0305h supported
 *	 Bit3	= INT 16/AX=0306h supported
 *	 Bit4	= INT 16/AH=0Ah supported
 *	 Bit5	= INT 16/AH=10h-12h supported (enhanced keyboard support)
 *	 Bit6	= INT 16/AH=20h-22h supported (122-key keyboard support)
 *	 Bit7	= reserved
 */
static void
bios_16_09xx(struct regs *regs)
{
	AL = 0x20; /* we support 0x10-12 */
}

/*
 * KBD:
 *
 * Get enhanced keycode.
 *
 * In:	AH = 10h
 *
 * Out:	AL	= ASCII code
 *	AH	= scan code
 */
static void
bios_16_10xx(struct regs *regs)
{
	/* FIXME VOSSI */
	bios_16_00xx(regs);
}

/*
 * KBD:
 *
 * Check for enhanced keycode.
 *
 * In:	AH = 11h
 *
 * Out:
 *	AL	= ASCII code
 *	AH	= scan code
 * 	F
 * 	 zero	= 0: key available; 1: no key available
 */
static void
bios_16_11xx(struct regs *regs)
{
	if (! kbd_keyavailable()) {
		/*
		 * No keystroke available.
		 */
		F |= 1 << 6;	/* Set zero flag. */

	} else {
		/*
		 * Peek keystroke.
		 */
		AX = dequeue_key(0);
		F &= ~(1 << 6);	/* Clear zero flag. */
	}
}

/*
 * KBD:
 *
 * Get extended shift states.
 *
 * In:	AH = 12h
 *
 * Out:	AL
 *	 Bit0	= right Shift pressed
 *	 Bit1	= left Shift pressed
 *	 Bit2	= left or right Ctrl pressed
 *	 Bit3	= left or right Alt pressed
 *	 Bit4	= ScrollLock mode active
 *	 Bit5	= NumLock mode active
 *	 Bit6	= CapsLock mode active
 *	 Bit7	= Insert mode active
 * 	AH
 *	 Bit0	= left Ctrl pressed
 *	 Bit1	= left Alt pressed
 *	 Bit2	= right Ctrl pressed
 *	 Bit3	= right Alt pressed
 *	 Bit4	= ScrollLock pressed
 *	 Bit5	= NumLock pressed
 *	 Bit6	= CapsLock pressed
 *	 Bit7	= SysRq pressed
 */
static void
bios_16_12xx(struct regs *regs)
{
	AL = var_get(kbd_shift0);
	AH = 0; /* FIXME MARCEL */
}

C_ENTRY __attribute__((regparm(1))) void
bios_16_xxxx(struct regs *regs)
{
	if (AH == 0x00) {
		bios_16_00xx(regs);
	} else if (AH == 0x01) {
		bios_16_01xx(regs);
	} else if (AH == 0x02) {
		bios_16_02xx(regs);
	} else if (AH == 0x03) {
		bios_16_03xx(regs);
	} else if (AH == 0x05) {
		bios_16_05xx(regs);
	} else if (AH == 0x09) {
		bios_16_09xx(regs);
	} else if (AH == 0x10) {
		bios_16_10xx(regs);
	} else if (AH == 0x11) {
		bios_16_11xx(regs);
	} else if (AH == 0x12) {
		bios_16_12xx(regs);

	} else if (AH == 0x25) {
		/* Keyboard reset. */
		/* Reset mappings to default. */
		/* Do nothing... FIXME VOSSI */

	} else if (AH == 0x92) {
		/* keyboard capability check called by DOS 5.0+ keyb */
		AH = 0x80;	/* function int16 ah=0x10-0x12 supported */

	} else if (AH == 0xA2) {
		/* 122 keys capability check called by DOS 5.0+ keyb */
		/* don't change AH: function int16 ah=0x20-0x22 NOT supported */

	} else {
		dprintf("kbd: unsupported function %02x\n", AH);
	}
}

asm (
"	.code16\n"
"irq16: .globl irq16\n"
#ifdef CONFIG_80286_SUPPORT
	/* FIXME */
#else
"	pushl %eax\n"
"	pushl %ebx\n"
"	pushl %ecx\n"
"	pushl %edx\n"
"	pushw %bp\n"
"	pushw %si\n"
"	pushw %di\n"

"	pushw %ds\n"
"	pushw %es\n"
"	movw %ss, %dx\n"
"	movw %dx, %ds\n"
"	movw %dx, %es\n"

"	movl %esp, %eax\n"
"	pushl %eax\n"
"	andl $0x0000ffff, %esp\n"
"	andl $0x0000ffff, %eax\n"

"	ex_lcall bios_16_xxxx\n" /* Passing argument in %eax. */

"	popl %eax\n"
"	movl %eax, %esp\n"

"	popw %es\n"
"	popw %ds\n"

"	popw %di\n"
"	popw %si\n"
"	popw %bp\n"
"	popl %edx\n"
"	popl %ecx\n"
"	popl %ebx\n"
"	popl %eax\n"
#endif
"	iretw\n"
);

static inline void
kbd_disable(void)
{
	outb(0xad, 0x64);
}

static inline void
kbd_enable(void)
{
	outb(0xae, 0x64);
}

C_ENTRY void
bios_09(void)
{
	/* no special key - translation table */
	static CONST char kbd_lc[] = {
	/* 00 */	0, '\033', '1', '2',
	/* 04 */	'3', '4', '5', '6',
	/* 08 */	'7', '8', '9', '0',
	/* 0c */	'-', '=', '\b', '\t',
	/* 10 */	'q', 'w', 'e', 'r',
	/* 14 */	't', 'y', 'u', 'i',
	/* 18 */	'o', 'p', '[', ']',
	/* 1c */	'\r', -1, 'a', 's',
	/* 20 */	'd', 'f', 'g', 'h',
	/* 24 */	'j', 'k', 'l', ';',
	/* 28 */	'\'', '`', -1, '\\',
	/* 2c */	'z', 'x', 'c', 'v',
	/* 30 */	'b', 'n', 'm', ',',
	/* 34 */	'.', '/', -1, '*',
	/* 38 */	-1, ' ', 0, 0,
	/* 3c */	0, 0, 0, 0,
	/* 40 */	0, 0, 0, 0,
	/* 44 */	0, 0, 0, 0,
	/* 48 */	0, 0, '-', 0,
	/* 4c */	0, 0, '+', 0,
	/* 50 */	0, 0, 0, 0,
	};
	/* shift key - translation table */
	static CONST char kbd_uc[] = {
	/* 00 */	0, 27, '!', '@',
	/* 04 */	'#', '$', 37, 0x5e,
	/* 08 */	'&', '*', '(', ')',
	/* 0c */	'_', '+', 0x08, 0,
	/* 10 */	'Q', 'W', 'E', 'R',
	/* 14 */	'T', 'Y', 'U', 'I',
	/* 18 */	'O', 'P', '{', '}',
	/* 1c */	0x0d, -1, 'A', 'S',
	/* 20 */	'D', 'F', 'G', 'H',
	/* 24 */	'J', 'K', 'L', ':',
	/* 28 */	'"', 0x7e, -1, '|',
	/* 2c */	'Z', 'X', 'C', 'V',
	/* 30 */	'B', 'N', 'M', '<',
	/* 34 */	'>', '?', -1, 0,
	/* 38 */	-1, ' ', -1, 0,
	/* 3c */	0, 0, 0, 0,
	/* 40 */	0, 0, 0, 0,
	/* 44 */	0, 0, 0, 0,
	/* 48 */	0, 0, '-', 0,
	/* 4c */	0, 0, '+', 0,
	/* 50 */	0, 0, 0, 0,
	};
	/* ctrl key - translation table */
	static CONST char kbd_ctrl[] = {
	/* 00 */	0, 0, 0, 0,
	/* 04 */	0, 0, 0, 0,
	/* 08 */	0, 0, 0, 0,
	/* 0c */	0, 0, 0, 0,
	/* 10 */	0x11, 0x17, 0x05, 0x12,
	/* 14 */	0x14, 0x19, 0x15, 0x09,
	/* 18 */	0x0f, 0x10, 0x1b, 0x1d,
	/* 1c */	0, 0, 0x01, 0x13,
	/* 20 */	0x04, 0x06, 0x07, 0x08,
	/* 24 */	0x0a, 0x0b, 0x0c, 0,
	/* 28 */	0, 0, 0, 0,
	/* 2c */	0x1a, 0x18, 0x03, 0x16,
	/* 30 */	0x02, 0x0e, 0x0d, 0,
	/* 34 */	0, 0, 0, 0,
	/* 38 */	0, 0
	};

	unsigned char code;
	unsigned char ascii;
	uint16_t ax;
	uint16_t done;

	kbd_disable();

	code = inb(0x60);

	/* Allow for keyboard intercept. */
	ax = (0x4f << 8) | code;
	asm (
		"stc\n\t"
		"int $0x15\n\t"
		"jnc 1f\n\t"
		"movw $0, %1\n\t"
		"jmp 2f\n\t"
	"1:\n\t"
		"movw $1, %1\n\t"
	"2:\n\t"
		: "=a" (ax), "=r" (done)
		: "0" (ax)
	);

	eoi();

	if (done) {
		goto ret;
	}
	code = ax & 0xff;

	if (code == 0xfa) {
		/* ACK */
		var_put(kbd_state3, var_get(kbd_state3) | 0x10);

	} else if (code == 0xfe) {
		/* RESEND */
		var_put(kbd_state3, var_get(kbd_state3) | 0x20);

	} else if (code == 0xff) {
		/* OVERFLOW */
		/* FIXME */

	} else if (code == 0xe0) {
		var_put(kbd_state2, var_get(kbd_state2) | 0x10 | 0x02);
		goto ret;

	} else if (code == 0xe1) {
		var_put(kbd_state2, var_get(kbd_state2) | 0x10 | 0x01);
		goto ret;

	} else if (code == 0x36) {
		/* Right shift pressed. */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x01);

	} else if (code == (0x36 | 0x80)) {
		/* Right shift released. */
		var_put(kbd_shift0, var_get(kbd_shift0) & ~0x01);

	} else if (code == 0x2a) {
		/* Left shift pressed. */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x02);

	} else if (code == (0x2a | 0x80)) {
		/* Left shift released. */
		var_put(kbd_shift0, var_get(kbd_shift0) & ~0x02);

	} else if (code == 0x1d) {
		if (var_get(kbd_state2) & 0x02) {
			/* Right CTRL pressed. */
			var_put(kbd_state2, var_get(kbd_state2) | 0x04);
		} else {
			/* Left CTRL pressed. */
			var_put(kbd_shift0, var_get(kbd_shift0) | 0x04);
		}

	} else if (code == (0x1d | 0x80)) {
		if (var_get(kbd_state2) & 0x02) {
			/* Right CTRL released. */
			var_put(kbd_state2, var_get(kbd_state2) & ~0x04);
		} else {
			/* Left CTRL released. */
			var_put(kbd_shift0, var_get(kbd_shift0) & ~0x04);
		}

	} else if (code == 0x38) {
		if (var_get(kbd_state2) & 0x02) {
			/* Right ALT pressed. */
			var_put(kbd_state2, var_get(kbd_state2) | 0x08);
		} else {
			/* Left ALT pressed. */
			var_put(kbd_shift0, var_get(kbd_shift0) | 0x08);
		}

	} else if (code == (0x38 | 0x80)) {
		if (var_get(kbd_state2) & 0x02) {
			/* Right ALT released. */
			var_put(kbd_state2, var_get(kbd_state2) & ~0x08);
		} else {
			/* Left ALT released. */
			var_put(kbd_shift0, var_get(kbd_shift0) & ~0x08);
		}

	} else if (code == 0x46) {
		/* scroll lock on */
		var_put(kbd_shift0, var_get(kbd_shift0) ^ 0x10);
		kbd_led_update();

	} else if (code == 0x45) {
		/* num lock on */
		var_put(kbd_shift0, var_get(kbd_shift0) ^ 0x20);
		kbd_led_update();

	} else if (code == 0x3a) {
		/* caps lock on */
		var_put(kbd_shift0, var_get(kbd_shift0) ^ 0x40);
		kbd_led_update();

	} else if (code == 0x52) {
		/* insert on */
		var_put(kbd_shift0, var_get(kbd_shift0) ^ 0x80);

	} else if (code & 0x80) {
		/* Key released. */

	} else {
		/* Normal key pressed. */
		if (/* 0 <= code && */ code < sizeof(kbd_lc)) {
			if ((var_get(kbd_shift0) & 0x08)
			 || (var_get(kbd_state2) & 0x08)) {
				/* ALT */
				ascii = 0;	/* FIXME VOSSI */

			} else if ((var_get(kbd_shift0) & 0x04)
				|| (var_get(kbd_state2) & 0x04)) {
				/* CTRL */
				ascii = const_get(kbd_ctrl[code]);

			} else if (var_get(kbd_shift0) & 0x03) {
				/* Shift */
				ascii = const_get(kbd_uc[code]);

			} else {
				/* No modifier pressed. */
				ascii = const_get(kbd_lc[code]);
			}

		} else {
			/* Just forward returned values and set ascii to 0. */
			/* FIXME MARCEL */
			ascii = 0;
		}

		enqueue_key((code << 8) | ascii);

		/* Signal I/O possible for keyboard. */
		asm volatile (
			"int $0x15\n\t"
			: "=a" (ax)
			: "0" ((0x91 << 8) | 0x02)
		);
	}

	var_put(kbd_state2, var_get(kbd_state2) & ~0x03);

ret:	;
	kbd_enable();
}

#endif /* RUNTIME_RM */
/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

void
kbd_init(void)
{
	/*
	 * Initialize kbd variables in BDA.
	 */
	var_put(kbd_shift0, 0);
	var_put(kbd_shift1, 0);
	var_put(kbd_numpad, 0);
	var_put(ctrlbrk, 0);
	var_put(kbd_state2, 0x10);
	var_put(kbd_state3, 0);

	var_put(kbd_head, (unsigned short) (long) &VAR->kbd_buf[0]);
	var_put(kbd_tail, (unsigned short) (long) &VAR->kbd_buf[0]);
	var_put(kbd_bstart, (unsigned short) (long) &VAR->kbd_buf[0]);
	var_put(kbd_bend, (unsigned short) (long) &VAR->kbd_buf[sizeof(VAR->kbd_buf)]);

	/*
	 * Remove any remaining keycode.
	 */
	(void) inb(0x60);

	/*
	 * Do self test.
	 */
	outb(KBD_CCMD_SELF_TEST, 0x64);
	while (! ((inb(0x60) >> 2) & 1)) {
		/* Wait for self test done. */
	}
	(void) inb(0x60);

	/*
	 * Set keyboard mode to
	 * - keycode translation on.
	 * - mouse interrupts disabled
	 * - keyboard interrupts enabled
	 */
	outb(KBD_CCMD_WRITE_MODE, 0x64);
	outb(KBD_CONTROL_XLATE \
		| KBD_CONTROL_SELFTEST \
		| KBD_CONTROL_KBDINT,
		0x60);

	/*
	 * Enable keyboard.
	 */
	outb(KBD_CCMD_KBD_ENABLE, 0x64);

	/*
	 * Disable mouse.
	 */
	outb(KBD_CCMD_MOUSE_DISABLE, 0x64);

	/*
	 * If set in BIOS setup switch on NumLock status.
	 */
	if (cmos_get(x11) & 0x80) {
		/* Start with NumLock set. */
		var_put(kbd_shift0, var_get(kbd_shift0) ^ 0x20);
		kbd_led_update();
	}
}

#endif /* INIT_RM */
