# How to add dynamic attributes to a device class

```{tags} audience:developers, lang:c++
```

A non-fixed number of attributes is common while working with Input/Output devices
with a variable number of channels (PLCs, DaqBoards, Timers/Counters), when depending
of the operation mode more or less scalar or spectrum attributes are be needed.

## Generating your class with Pogo

`DynAttr` with three types of attributes:

1. StaticAttr: A scalar, Tango::DevShort, READ attribute,
2. LongDynAttr: A scalar, Tango::DevLong, READ_WRITE attribute,
3. DoubleDynAttr: A scalar, Tango::DevDouble, READ_WRITE attribute.

Each device belonging to this class has one `StaticAttr` and a dynamic
list of `LongDynAttr` or `DoubleDynAttr` attributes. This list of dynamic
attributes is defined with a device property named `DynAttrList`.

Therefore, using Pogo, we define a Tango class with three scalar
attributes with the definition given above. We also create the
DynAttrList property for defining dynamic attribute. This is a vector of
strings with a couple of strings for each attribute. The first string is
the dynamic attribute type (`LongDynAttr` or `DoubleDynAttr`) and the second
string is the attribute name.

:::{note}
The dynamic attributes have a light green background
color on the main Pogo window.
:::

:::{figure} how-to-dynamic-attributes-device-class/pogo-dynattr.jpg
:align: center
:::

### Dynamic attribute registration

To register device dynamic attributes, you need to:

1. call the `Tango::DeviceImpl::add_attribute()` method for each device dynamic attribute
2. create the data used with the attribute (for the Attribute::set_value() method)

Pogo generates method(s) named `DynAttr::add_$<DynAttrName>_dynamic_attribute(std::string att_name)` which do
this job for you. You have to call these methods according to your needs in another Pogo generated method
named `DynAttr::add_dynamic_attributes()` which is executed at device creation time.

In our example, in this method we have to:

- analyze the content of the device DynAttrList property and create
  the necessary attributes using the helper method also generated by
  Pogo

The code of the DynAttr::add_dynamic_attributes() method looks like

```{code-block} cpp
:linenos: true

void DynAttr::add_dynamic_attributes()
{
    // Example to add dynamic attribute:
    // Copy inside the following protected area to create instance(s) at startup.
    // add_LongDynAttr_dynamic_attribute("MyLongDynAttrAttribute");
    // add_DoubleDynAttr_dynamic_attribute("MyDoubleDynAttrAttribute");

    /*----- PROTECTED REGION ID(DynAttr::add_dynamic_attributes) ENABLED START -----*/
    /* clang-format on */

    const auto len = dynAttrList.size();

    if(len == 0)
    {
      return;
    }

    if((len % 2) != 0)
    {
      TANGO_THROW_EXCEPTION("DynAttrInvalidSetup", "Expected a multiple of two entries in dynAttrList");
    }

    for(size_t i = 0; i < len; i += 2)
    {
      if(dynAttrList[i] == "LongDynAttr")
      {
        add_LongDynAttr_dynamic_attribute(dynAttrList[i + 1]);
      }
      else if(dynAttrList[i] == "DoubleDynAttr")
      {
        add_DoubleDynAttr_dynamic_attribute(dynAttrList[i + 1]);
      }
      else
      {
        TANGO_THROW_EXCEPTION("DynAttrInvalidSetup", "Unexpected dynamic attribute name");
      }
    }

    /* clang-format off */
    /*----- PROTECTED REGION END -----*/ //  DynAttr::add_dynamic_attributes
}
```

:::{note}
The data associated with all LongDynAttr dynamic attributes
are initialized to 0 and the data associated to all DoubleDynAttr
dynamic attributes are initialized with 0.0
:::

The definition of the DoubleDynAttr attribute is simply to return when
read, the last value which has been written. The code for the
DoubleDynAttr reading/writing is the following

```{code-block} cpp
:linenos: true

void DynAttr::read_DoubleDynAttr(Tango::Attribute &attr)
{
  DEBUG_STREAM << "DynAttr::read_DoubleDynAttr(Tango::Attribute &attr) entering... " << std::endl;
  Tango::DevDouble  *att_value = get_DoubleDynAttr_data_ptr(attr.get_name());
  /*----- PROTECTED REGION ID(DynAttr::read_DoubleDynAttr) ENABLED START -----*/
  /* clang-format on */

  attr.set_value(att_value);

  /* clang-format off */
  /*----- PROTECTED REGION END -----*/  //  DynAttr::read_DoubleDynAttr
}

void DynAttr::write_DoubleDynAttr(Tango::WAttribute &attr)
{
  DEBUG_STREAM << "DynAttr::write_DoubleDynAttr(Tango::WAttribute &attr) entering... " << std::endl;
  //  Retrieve write value
  Tango::DevDouble  w_val;
  attr.get_write_value(w_val);
  /*----- PROTECTED REGION ID(DynAttr::write_DoubleDynAttr) ENABLED START -----*/
  /* clang-format on */

  auto *att_value = get_DoubleDynAttr_data_ptr(attr.get_name());
  *att_value = w_val;

  /* clang-format off */
  /*----- PROTECTED REGION END -----*/  //  DynAttr::write_DoubleDynAttr
}
```

The code of the read method in it's Pogo generated part retrieves a
pointer to the data associated with this attribute with the helper
method named `DynAttr::get_$<DynAttrName>_data_ptr(std::string att_name)`.
The user code simply passes this pointer to the `Tango Attribute::set_value()` method.

The user code of the write method also uses the Pogo generated helper
method to get the attribute data pointer and set this data to the value
sent by the caller.

The definition of the LongDynAttr is a bit more sophisticated. For one
device of this Tango class, we have several dynamic attributes of this
LongDynAttr type. According to which attribute is read or written, we
have to call different method accessing the hardware.

The code for reading/writing the LongDynAttr attribute is given below:

```{code-block} cpp
:linenos: true

void DynAttr::read_LongDynAttr(Tango::Attribute &attr)
{
    DEBUG_STREAM << "DynAttr::read_LongDynAttr(Tango::Attribute &attr) entering... " << endl;
    Tango::DevLong *att_value = get_LongDynAttr_data_ptr(attr.get_name());

    /*----- PROTECTED REGION ID(DynAttr::read_LongDynAttr) ENABLED START -----*/

    std::string &att_name = attr.get_name();
    if(att_name == dynAttrList[1])
    {
      // Access hardware for channel 1 which is the first attribute in the list
      *att_value = read_hardware_channel1();
    }
    else if(att_name == dynAttrList[3])
    {
      // Access hardware for channel 2 which is the second attribute in the list
      *att_value = read_hardware_channel2();
    }
    else
    {
      TANGO_THROW_EXCEPTION("DynAttrInvalidSetup", "Unexpected dynamic attribute name");
    }

    //  Set the attribute value
    attr.set_value(att_value);

    /*----- PROTECTED REGION END -----*/    //  DynAttr::read_LongDynAttr
}

void DynAttr::write_LongDynAttr(Tango::WAttribute &attr)
{
    DEBUG_STREAM << "DynAttr::write_LongDynAttr(Tango::Attribute &attr) entering... " << endl;

    //  Retrieve write value
    Tango::DevLong  w_val;
    attr.get_write_value(w_val);

    /*----- PROTECTED REGION ID(DynAttr::write_LongDynAttr) ENABLED START -----*/

    std::string &att_name = attr.get_name();
    if(att_name == dynAttrList[1])
    {
      // Access hardware for channel 1 which is the first attribute in the list
      write_hardware_channel1(w_val);
    }
    else if(att_name == dynAttrList[3])
    {
      // Access hardware for channel 2 which is the second attribute in the list
      write_hardware_channel2(w_val);
    }
    else
    {
      TANGO_THROW_EXCEPTION("DynAttrInvalidSetup", "Unexpected dynamic attribute name");
    }

    /*----- PROTECTED REGION END -----*/    //  DynAttr::write_LongDynAttr
}
```

## Running the server

A Tango device server process hosting this DynAttr class has been
defined in the database with two device. The dynamic attributes for
these two devices are:

:::{figure} how-to-dynamic-attributes-device-class/jive-dev1.jpg
:align: center
:::

:::{figure} how-to-dynamic-attributes-device-class/prop_dev2.jpg
:align: center
:::

As shown by the Pogo screen-shot in the beginning of this HowTo, the
Tango class also defines a static attribute for each device named
StaticAttr. Running the device server and opening TestDevice panels on
each device displays device attribute list:

:::{figure} how-to-dynamic-attributes-device-class/test-dev1.jpg
:align: center
:::

:::{figure} how-to-dynamic-attributes-device-class/test_device2.jpg
:align: center
:::

This method fully supports restarting device(s) or server using the
device server process admin device.
