use core:io;
use markdown;

// Load a markdown document and add post-processing as required.
Document loadDocument(Node node, Url file) {
	try {
		Document doc = parse(file.readAllText());

		// Add post-processing.
		doc.visit(DocVisitor(node, file));

		return doc;
	} catch (core:lang:CompilerError error) {
		// Don't wrap compiler errors.
		throw error;
	} catch (DocError error) {
		throw error;
	} catch (Exception error) {
		throw DocError(file, error.message);
	}
}


/**
 * Visitor to transform the markdown document to be more suitable for documentation.
 */
class DocVisitor extends markdown:Visitor {
	// Path inside the tree.
	Node node;

	// File.
	Url file;

	// Create.
	init(Node node, Url file) {
		init { node = node; file = file; }
	}

	// Transform elements:
	Element visit(Element element) {
		if (element as CodeBlock) {
			if (element.language == "inlinehtml") {
				return CustomHtml((StrBuf() << join(element.code, "\n")).toS);
			} else {
				// Apply syntax highlighting!
			}
		}

		element;
	}

	// Find and resolve markdown links:
	TextSpan visit(TextSpan span) {
		if (span as Link)
			span.target = resolveLink(span.target);
		span;
	}

	// Resolve a link.
	Str resolveLink(Str link) {
		if (link.startsWith("md:")) {
			Url url = parseMdUrl(link.cut(link.begin + 3));
			unless (target = node.resolveLink(url)) {
				print("WARNING: ${file}: Broken markdown link: ${link}");
				return link;
				// throw DocError(file, "Unable to resolve the markdown link: ${link}");
			}
			return target.relativeOutputPath(node).format();
		}
		link;
	}
}
