% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posixt.R
\name{as_date_time}
\alias{as_date_time}
\alias{as_date_time.POSIXt}
\alias{as_date_time.Date}
\alias{as_date_time.clock_calendar}
\alias{as_date_time.clock_sys_time}
\alias{as_date_time.clock_naive_time}
\alias{as_date_time.clock_zoned_time}
\title{Convert to a date-time}
\usage{
as_date_time(x, ...)

\method{as_date_time}{POSIXt}(x, ...)

\method{as_date_time}{Date}(x, zone, ..., nonexistent = NULL, ambiguous = NULL)

\method{as_date_time}{clock_calendar}(x, zone, ..., nonexistent = NULL, ambiguous = NULL)

\method{as_date_time}{clock_sys_time}(x, zone, ...)

\method{as_date_time}{clock_naive_time}(x, zone, ..., nonexistent = NULL, ambiguous = NULL)

\method{as_date_time}{clock_zoned_time}(x, ...)
}
\arguments{
\item{x}{\verb{[vector]}

A vector.}

\item{...}{These dots are for future extensions and must be empty.}

\item{zone}{\verb{[character(1)]}

The zone to convert to.}

\item{nonexistent}{\verb{[character / NULL]}

One of the following nonexistent time resolution strategies, allowed to be
either length 1, or the same length as the input:
\itemize{
\item \code{"roll-forward"}: The next valid instant in time.
\item \code{"roll-backward"}: The previous valid instant in time.
\item \code{"shift-forward"}: Shift the nonexistent time forward by the size of
the daylight saving time gap.
\item \verb{"shift-backward}: Shift the nonexistent time backward by the size of
the daylight saving time gap.
\item \code{"NA"}: Replace nonexistent times with \code{NA}.
\item \code{"error"}: Error on nonexistent times.
}

Using either \code{"roll-forward"} or \code{"roll-backward"} is generally
recommended over shifting, as these two strategies maintain the
\emph{relative ordering} between elements of the input.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{nonexistent} must be supplied
and cannot be \code{NULL}. This is a convenient way to make production code
robust to nonexistent times.}

\item{ambiguous}{\verb{[character / zoned_time / POSIXct / list(2) / NULL]}

One of the following ambiguous time resolution strategies, allowed to be
either length 1, or the same length as the input:
\itemize{
\item \code{"earliest"}: Of the two possible times, choose the earliest one.
\item \code{"latest"}: Of the two possible times, choose the latest one.
\item \code{"NA"}: Replace ambiguous times with \code{NA}.
\item \code{"error"}: Error on ambiguous times.
}

Alternatively, \code{ambiguous} is allowed to be a zoned_time (or POSIXct) that
is either length 1, or the same length as the input. If an ambiguous time
is encountered, the zoned_time is consulted. If the zoned_time corresponds
to a naive_time that is also ambiguous \emph{and} uses the same daylight saving
time transition point as the original ambiguous time, then the offset of
the zoned_time is used to resolve the ambiguity. If the ambiguity cannot be
resolved by consulting the zoned_time, then this method falls back to
\code{NULL}.

Finally, \code{ambiguous} is allowed to be a list of size 2, where the first
element of the list is a zoned_time (as described above), and the second
element of the list is an ambiguous time resolution strategy to use when
the ambiguous time cannot be resolved by consulting the zoned_time.
Specifying a zoned_time on its own is identical to \verb{list(<zoned_time>, NULL)}.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{ambiguous} must be supplied and
cannot be \code{NULL}. Additionally, \code{ambiguous} cannot be specified as a
zoned_time on its own, as this implies \code{NULL} for ambiguous times that the
zoned_time cannot resolve. Instead, it must be specified as a list
alongside an ambiguous time resolution strategy as described above. This is
a convenient way to make production code robust to ambiguous times.}
}
\value{
A date-time with the same length as \code{x}.
}
\description{
\code{as_date_time()} is a generic function that converts its input to a date-time
(POSIXct).

There are methods for converting dates (Date), calendars, time points, and
zoned-times to date-times.

For converting to a date, see \code{\link[=as_date]{as_date()}}.
}
\details{
Note that clock always assumes that R's Date class is naive, so converting
a Date to a POSIXct will always attempt to retain the printed year, month,
and day. Where possible, the resulting time will be at midnight (\code{00:00:00}),
but in some rare cases this is not possible due to daylight saving time. If
that issue ever arises, an error will be thrown, which can be resolved by
explicitly supplying \code{nonexistent} or \code{ambiguous}.

This is not a drop-in replacement for \code{as.POSIXct()}, as it only converts a
limited set of types to POSIXct. For parsing characters as date-times, see
\code{\link[=date_time_parse]{date_time_parse()}}. For converting numerics to date-times, see
\code{\link[vctrs:new_date]{vctrs::new_datetime()}} or continue to use \code{as.POSIXct()}.
}
\examples{
x <- as.Date("2019-01-01")

# `as.POSIXct()` will always treat Date as UTC, but will show the result
# of the conversion in your system time zone, which can be somewhat confusing
if (rlang::is_installed("withr")) {
  withr::with_timezone("UTC", print(as.POSIXct(x)))
  withr::with_timezone("Europe/Paris", print(as.POSIXct(x)))
  withr::with_timezone("America/New_York", print(as.POSIXct(x)))
}

# `as_date_time()` will treat Date as naive, which means that the original
# printed date will attempt to be kept wherever possible, no matter the
# time zone. The time will be set to midnight.
as_date_time(x, "UTC")
as_date_time(x, "Europe/Paris")
as_date_time(x, "America/New_York")

# In some rare cases, this is not possible.
# For example, in Asia/Beirut, there was a DST gap from
# 2021-03-27 23:59:59 -> 2021-03-28 01:00:00,
# skipping the 0th hour entirely.
x <- as.Date("2021-03-28")
try(as_date_time(x, "Asia/Beirut"))

# To resolve this, set a `nonexistent` time resolution strategy
as_date_time(x, "Asia/Beirut", nonexistent = "roll-forward")


# You can also convert to date-time from other clock types
as_date_time(year_month_day(2019, 2, 3, 03), "America/New_York")
}
