/*
 * dlh - a LaTeX to HTML translator.
 * Copyright (C) 1996 David Mosberger-Tang.
 * This file is part of the dlh package.
 *
 * The dlh translator is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The dlh translator is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with dlh; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <libiberty.h>
#include <limits.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <sys/mman.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "dlh.h"
#include "dlh_math.h"
#include "sh.h"

#ifndef MAP_FILE
# define MAP_FILE		0
#endif

#ifdef __osf__
# define MMAP_CAN_MAP_EXTRA_BYTES	0
#endif

/*
 * Define if it OK to mmap more bytes than the size of the mapped
 * object.  This works on most systems.
 */
#ifndef MMAP_CAN_MAP_EXTRA_BYTES
# define MMAP_CAN_MAP_EXTRA_BYTES	1
#endif

#ifndef PATH_MAX
# define PATH_MAX		1024
#endif

#define NUM_SECTION_LEVELS	7

#undef MIN
#define MIN(a,b)		((a) < (b) ? (a) : (b))

#define WRITE_STR(fd, str)	write((fd), (str), strlen(str))
#define FILENAME_FMT		"doc%03u"

/*
 * Number of dots per inch.  Good monitors can do 100dpi and since we
 * want to be rather on the large than on the small side, we pick
 * something a little bigger than that:
 */
#define DEFAULT_DISPLAY_DPI	120

enum Files {
    OUT_FOOTNOTE, OUT_CONTENTS, OUT_INDEX,
    OUT_MAX
};

struct sh *		counter_tab;	/* counter hash table */
struct sh *		ctab;	/* command hash-table */
struct sh *		etab;	/* environment hash-table */
struct sh *		dtab;	/* dimension hash-table */
struct xref_tab		xtab;	/* cross-reference table */
struct sh *		ltab;	/* label-name -> xref_def hash table */
struct sh *		btab;	/* citation-name -> xref_def hash table */
struct dict_tab		itab;	/* index table */
struct dict_tab		lof;	/* list of figures */
struct dict_tab		lot;	/* list of tables */
char		aux_prefix[PATH_MAX];	/* prefix for auxiliary input files */
const char *	dir_prefix;	/* path of output directory */
const char *	icon_dir = "../icons";
FILE *		f[OUT_MAX];
struct file_stack * out;
unsigned	aux_flags;
int		verbose;
int		counter[CNT_MAX];
int		filenum_bibliography;
int		current_section_level;
int		min_section_level = NUM_SECTION_LEVELS - 1;
int		split_level = 2;
int		appendix_level = -1;
int		current_section_number[NUM_SECTION_LEVELS];
int		display_dpi = DEFAULT_DISPLAY_DPI;
char *		dlhinputs;
char *		texinputs;
const char *	prog_name;
const char *	pos;
const char *	filename = "*INIT*";
unsigned long	linenum;
long		current_font_size;	/* 0=default, <0 smaller, >0 bigger */
long		current_list_type;
const char *	current_default_label = "";
struct env_stack * env;
struct env	anonymous_env = { name: "anonymous" };
unsigned long	section_file_mask;	/* bit i==1 => level i file exists */
const char *	skip_to_string;
off_t		beginning_of_line_offset;	/* used for tabbing */

enum Title_Page_Elements {
    TPE_TITLE,
    TPE_AUTHOR,
    TPE_DATE,
    NUM_TITLE_PAGE_ELEMENTS
};

const char *	title_page[NUM_TITLE_PAGE_ELEMENTS];

const struct option opts[] = {
    {"resolution", required_argument, NULL, 'r'},
    {"force", no_argument, NULL, 'f'},
    {"help", no_argument, NULL, 'h'},
    {"icon-dir", required_argument, NULL, 'i'},
    {"split-level", required_argument, NULL, 'l'},
    {"verbose", no_argument, NULL, 'v'},
    {"version", no_argument, NULL, 'V'},
    {0}
};

#define PT_PER_INCH	72.27	/* number of typesetting points per inch: */
#define PT_PER_EX	10	/* number of points per ex */

struct {
    const char *	name;
    double		length;	/* length in points */
} dim_list[] = {
    {"baselineskip",	10},
    {"bigskipamount",	10},
    {"fill",		PT_PER_INCH},	/* better: 0 with inf stretchability */
    {"medskipamount",	5},
    {"parindent",	40},
    {"smallskipamount",	1},
    {"textheight",	8.5 * PT_PER_INCH},
    {"textwidth",	6.5 * PT_PER_INCH},
};

static void eval (const char * what);


static void
usage (void)
{
    fprintf(stderr,
	    _("usage: %s [-hvV] [-i icon-url]\n"
	      "       [--help] [--icon-dir icon-url] [--verbose] [--version]\n"
	      "       file...\n"), prog_name);
}


/*
 * Parse a LaTeX dimension and return the length in points.
 */
static double
parse_length (const char * str)
{
    double l, unit = 0.0, * dp;
    char * end;

    l = strtod(str, &end);
    if (str == end) {
	l = 1.0;
    }
    switch (end[0]) {
      case '\\':
	  unit = 1.0;
	  dp = sh_lookup(dtab, end + 1);
	  if (!dp) {
	      fprintf(stderr, _("%s:%lu: ignoring unknown length `%s'\n"),
		      filename, linenum, end + 1);
	      break;
	  }
	  l *= *dp;
	  break;

      case 'c':
	  if (end[1] == 'm' && !end[2]) {
	      unit = PT_PER_INCH / 2.54;	/* unit=28.45pt/cm */
	  }
	  break;

      case 'e':
	  if (!end[2]) {
	      switch (end[1]) {
		case 'm':
		    unit = 10;			/* unit=10pt/em */
		    break;

		case 'x':
		    unit = PT_PER_EX;		/* unit=10pt/ex */
		    break;
	      }
	  }
	  break;

      case 'i':
	  if (end[1] == 'n' && !end[2]) {
	      unit = PT_PER_INCH;		/* unit=72.27pt/inch */
	  }
	  break;

      case 'p':
	  if (!end[2]) {
	      if (end[1] == 'c') {
		  unit = 12;			/* unit=12pt/pc */
	      } else if (end[1] == 't') {
		  unit = 1.0;			/* unit=1pt/pt */
	      }
	  }
	  break;

      case 'm':
	  if (end[1] == 'm' && !end[2]) {
	      unit = PT_PER_INCH / 25.4;	/* unit=2.84pt/mm */
	  }
	  break;
    }
    if (unit == 0.0) {
	fprintf(stderr, _("%s:%lu: ignoring unknown unit `%s'\n"),
		filename, linenum, end);
	unit = 1.0;
    }
    return l * unit;
}


static int
locate_and_open (const char * fname, const char * ext,
		 const char * search_path, char * result)
{
    char * copy, * next, * dir;
    int fd = -1;

    if (!search_path) {
	search_path = ".";
    }
    copy = strdup (search_path);
    for (next = copy; (dir = strsep(&next, ":")) != 0; ) {
	if (ext) {
	    sprintf(result, "%s/%s.%s", dir, fname, ext);
	    fd = open(result, O_RDONLY);
	    if (fd >= 0) {
		break;
	    }
	}

	sprintf(result, "%s/%s", dir, fname);
	fd = open(result, O_RDONLY);
	if (fd >= 0) {
	    break;
	}
    }
    free (copy);
    return fd;
}


/*
 * Open and a file and return its contents as an mmap'ed string.  In
 * case of error, 0 is returned.  The file being opened has name FNAME
 * and if EXT is not 0, we first attempt to open FNAME.EXT.  If either
 * lookup fails, the file is searched along the list of directories
 * specified in SEARCH_PATH (if not 0).
 */
static const char *
open_file (const char * fname, const char * ext, char * search_path)
{
    size_t file_size, pgsize;
    unsigned long laddr;
    char buf[PATH_MAX];
    struct stat st;
    char * addr;
    int fd;

    fd = locate_and_open (fname, ext, search_path, buf);
    if (fd < 0) {
	return 0;
    }

    if (fstat(fd, &st) < 0) {
	return 0;
    }
    file_size = st.st_size;
#if MMAP_CAN_MAP_EXTRA_BYTES
    addr = mmap(0, file_size + 1, PROT_READ, MAP_FILE | MAP_PRIVATE, fd, 0);
#else
    addr = mmap(0, file_size, PROT_READ, MAP_FILE | MAP_PRIVATE, fd, 0);
#endif
    if ((long) addr == -1) {
	return 0;
    }
    close(fd);
    pgsize = getpagesize();
    laddr = ((unsigned long) addr + file_size) & ~(pgsize - 1);

#if !MMAP_CAN_MAP_EXTRA_BYTES
    if (laddr == (unsigned long) addr + file_size) {
        fd = open("/dev/zero", O_RDONLY);
        if (fd < 0) {
            return 0;
        }
	addr = mmap((void *) laddr, pgsize, PROT_READ, MAP_PRIVATE, fd, 0);
        if ((unsigned long) addr != laddr) {
            return 0;
        }
    }
#endif
    mprotect((caddr_t) laddr, pgsize, PROT_WRITE);
    addr[file_size] = '\0';
    mprotect((caddr_t) laddr, pgsize, PROT_READ);
    return addr;
}


static const char *
open_file_maybe (const char * fname, const char * ext, char * search_path)
{
    const char * retval;

    retval = open_file (fname, ext, search_path);
    if (!retval) {
	perror(fname);
	return "";
    }
    return retval;
}


static FILE *
open_output (const char * fname)
{
    char buf[1024];
    FILE * fp;

    sprintf(buf, "%s/%s.html", dir_prefix, fname);

    fp = fopen(buf, "w");
    if (!fp) {
	perror(buf);
    }
    return fp;
}


void
push_file (FILE * fp)
{
    struct file_stack * fs;

    fs = xmalloc(sizeof(struct file_stack));
    memset(fs, 0, sizeof(*fs));
    fs->next = out;
    fs->file = fp;
    out = fs;
}


FILE *
pop_file (void)
{
    struct file_stack * fs;
    FILE * fp = 0;

    fs = out;
    if (fs) {
	out = out->next;
	fp = fs->file;
	free(fs);
    }
    return fp;
}


void
push_env (struct env * e)
{
    struct env_stack * new_env;

    new_env = xmalloc(sizeof(struct env_stack));
    memset(new_env, 0, sizeof(*new_env));
    new_env->next = env;
    new_env->e = e;
    env = new_env;
}


void
pop_env (struct env * e)
{
    struct env_stack * next_env;
    int i;

    if (env->e != e) {
	fprintf(stderr, _("%s:%lu: ignoring closing of %s environment "
			  "(expected %s)\n"),
		filename, linenum, e->name, env->e->name);
	return;
    }

    for (i = env->ncleanup - 1; i >= 0; --i) {
	(*env->cleanup[i].op)(env->cleanup[i].arg);
    }
    env->ncleanup = 0;
    next_env = env->next;
    if (next_env) {
	if (env->nalloced) {
	    free(env->cleanup);
	}
	free(env);
	env = next_env;
    }
}


/*
 * Find and return the top-most floating environment (figure or table)
 * or NULL if there is no such environment.
 */
struct env *
env_top_floating (void)
{
    const char * candidate;
    struct env_stack * e;
    size_t len = 0;

    for (e = env; e; e = e->next) {
	candidate = 0;
	switch (e->e->name[0]) {
	  case 'f': candidate = "figure"; len = 6; break;
	  case 't': candidate = "table"; len = 5; break;
	  default: break;
	}
	if (candidate && strncmp(e->e->name, candidate, len) == 0) {
	    int ch = e->e->name[len];

	    if (ch == '\0' || (ch == '*' && e->e->name[len + 1] == '\0')) {
		return e->e;
	    }
	}
    }
    return 0;
}


static void
make_current_section_number_string (char * buf)
{
    int i;

    for (i = min_section_level; i <= current_section_level; ++i) {
	if (i > min_section_level) {
	    *buf++ = '.';
	}
	if (i == appendix_level && current_section_number[i] <= 'Z' - 'A') {
	    sprintf(buf, "%c", 'A' - 1 + current_section_number[i]);
	} else {
	    sprintf(buf, "%d", current_section_number[i]);
	}
	buf += strlen(buf);
    }
}


void
add_cleanup(void (*cleanup_func)(const void *), const void * cleanup_arg)
{
    if (env->ncleanup >= env->nalloced) {
	env->nalloced += 16;
	env->cleanup = xrealloc(env->cleanup,
				env->nalloced * sizeof(*env->cleanup));
    }
    env->cleanup[env->ncleanup].op  = cleanup_func;
    env->cleanup[env->ncleanup].arg = cleanup_arg;
    ++env->ncleanup;
}


void
cmd_emit (struct call * c)
{
    const char * string = c->u.cmd->arg;

    fputs(string, out->file);
}


/*
 * Add to a dictionary.  Used to keep track of index entries and
 * the list of figures and tables.
 */
static struct dict_entry *
add_dict_entry (struct dict_tab * dict, const char * key, int label)
{
    struct dict_entry * de;

    if (dict->len >= dict->alloced) {
	dict->alloced += 512;
	dict->dict = xrealloc(dict->dict,
			      dict->alloced * sizeof(dict->dict[0]));
    }
    de = dict->dict + dict->len++;
    de->key = key;
    de->file_num = counter[CNT_FILE];
    de->label = label;
    return de;
}


static void
cmd_add_to_index (struct call * c)
{
    int label = counter[CNT_INDEX_LABEL]++;

    add_dict_entry(&itab, c->arg_list[0], label);

    fprintf(out->file, "<a name=\"i%d\">", label);
}


static void
add_to_toc (char * toc_entry)
{
    char buf[256];
    int i;

    if (!f[OUT_CONTENTS]) {
	return;
    }

    make_current_section_number_string(buf);

    push_file(f[OUT_CONTENTS]);
    if (current_section_level == min_section_level) {
	fputs("<tr></tr>", out->file);
    }
    fputs("<tr>", out->file);
    for (i = min_section_level; i < current_section_level; ++i) {
	fputs("<td></td>", out->file);
    }
    fputs("<td colspan=20>", out->file);
    fputs(buf, out->file);
    fputs("<tt>   </tt>", out->file);
    fprintf(out->file, "<a href=\"" FILENAME_FMT ".html#s%s\">",
	    counter[CNT_FILE], buf);
    if (current_section_level == min_section_level) {
	fputs("<b>", out->file);
    }
    eval(toc_entry);
    if (current_section_level == min_section_level) {
	fputs("</b>", out->file);
    }
    fputs("</a></td></tr>\n", out->file);
    pop_file();
}


static void
end_of_toc (void)
{
    struct dict_entry * idx;

    push_file(f[OUT_CONTENTS]);

    fputs("</table>", out->file);

    if (aux_flags & AUX_LOF) {
	fputs("\n\n<h1>List of Figures</h1>\n\n<table>\n", out->file);
	for (idx = lof.dict; idx < lof.dict + lof.len; ++idx) {
	    fprintf(out->file, "<tr><td>%d</td><td><a href=\"" FILENAME_FMT
		    ".html#f%d\">",
		    idx->label, idx->file_num, idx->label);
	    eval(idx->key);
	    fputs("</a></td></tr>\n", out->file);
	}
	fputs("</table>\n\n", out->file);
    }


    if (aux_flags & AUX_LOT) {
	fputs("\n\n<h1>List of Tables</h1>\n\n<table>\n", out->file);
	for (idx = lot.dict; idx < lot.dict + lot.len; ++idx) {
	    fprintf(out->file, "<tr><td>%d</td><td><a href=\"" FILENAME_FMT
		    ".html#f%d\">",
		    idx->label, idx->file_num, idx->label);
	    eval(idx->key);
	    fputs("</a></td></tr>\n", out->file);
	}
	fputs("</table>\n\n", out->file);
    }

    pop_file();
}


static void
add_to_footnotes (int num, const char * footnote_entry)
{
    if (!f[OUT_FOOTNOTE]) {
	f[OUT_FOOTNOTE] = open_output("footnotes");
	if (!f[OUT_FOOTNOTE]) {
	    return;
	}
	fputs("<title>Footnotes</title><h1>Footnotes</h1><p>\n\n",
	      f[OUT_FOOTNOTE]);
    }
    push_file(f[OUT_FOOTNOTE]);

    fputs("<hr><p>", out->file);
    fprintf(out->file,
	    "<b><a href=\"" FILENAME_FMT ".html#F%d\">%d</a></b>  ",
	    counter[CNT_FILE], num, num);
    eval(footnote_entry);
    fputc('\n', out->file);

    pop_file();
}


/*
 * \kill
 */
static void
cmd_kill (struct call * c)
{
    fseek(out->file, beginning_of_line_offset, SEEK_SET);
}


/*
 * \> and \= inside tabbing environment:
 */
static void
cmd_tab_move (struct call * c)
{
    fputs("</td>\n<td>", out->file);
}


static void
cmd_line_break (struct call * c)
{
    if (env && env->e && strcmp(env->e->name, "tabbing") == 0) {
	fputs("</td></tr>\n", out->file);
	beginning_of_line_offset = ftell(out->file);
	fputs("<tr valign=bottom><td>", out->file);
    } else {
	fputs("<br>", out->file);
    }
    if (c->arg_list[0]) {
	double length = parse_length(c->arg_list[0]);
	fprintf(out->file, "<spacer type=vertical size=%ld>", (long) length);
    }
}


static void
reset_font (const void * arg)
{
    const char * fontname = arg;

    fprintf(out->file, "</%s>", fontname);
}


/*
 * Switch to font FONTNAME for the duration of the current environment.
 */
void
select_font (const char * fontname)
{
    fprintf(out->file, "<%s>", fontname);
    add_cleanup(reset_font, fontname);
}


void
cmd_set_font (struct call * c)
{
    select_font(c->u.cmd->arg);
}


static void
reset_fontsize (const void * arg)
{
    long size = (long) arg;

    fprintf(out->file, "</font>");
    current_font_size = size;
}


static void
cmd_set_fontsize (struct call * c)
{
    long new_size = (long) c->u.cmd->arg;
    int delta;
    char sign;

    if (new_size != current_font_size) {
	add_cleanup(reset_fontsize, (void *) current_font_size);
	sign = '+';
	delta = new_size - current_font_size;
	if (delta < 0) {
	    sign = '-';
	    delta = -delta;
	}
	fprintf(out->file, "<font size=%c%u>", sign, delta);
	current_font_size = new_size;
    }
}


static void
accented_char (char accent, char letter)
{
    const char * postfix;

    switch (accent) {
      case '`': postfix = "grave"; break;
      case '\'': postfix = "acute"; break;
      case '^': postfix = "circ"; break;
      case '"': postfix = "uml"; break;
      case '~': postfix = "tilde"; break;
      case 'b':
      case 'c': postfix = "cedil"; break;

      case '=':  
      case '.': 
      case 'd':
      case 'H':
      case 'r':
      case 't':
      case 'u':
      case 'v':
      default:
	  postfix = 0;
    }
    if (postfix) {
	fputc('&', out->file);
	fputc(letter, out->file);
	fputs(postfix, out->file);
	fputc(';', out->file);
    } else {
	fprintf(stderr,
		_("%s:%lu:warning: dropped accent %c from letter %c\n"),
		filename, linenum, accent, letter);
	fputc(letter, out->file);
    }
}

static void
htmlify_char (char ch)
{
    const char * repl;

    repl = 0;
    switch (ch) {
      case '<': repl = "&lt;"; break;
      case '>': repl = "&gt;"; break;
      case '&': repl = "&amp;"; break;

      default:
	  break;
    }
    if (repl) {
	fputs(repl, out->file);
    } else {
	fputc(ch, out->file);
    }
}


static void
htmlify (const char * str, size_t len)
{
    size_t i;

    for (i = 0; i < len; ++i) {
	htmlify_char(str[i]);
    }
}


/*
 * Advance POS, scanning for string STR.  Comments are ignored.
 */
static void
stopat (const char * str)
{
    size_t len = strlen(str);
    char ch;

    while ((ch = *pos++)) {
	if (ch == '%' && pos[-2] != '\\') {
	    pos = strchr(pos, '\n');
	    if (!pos) {
		pos = "";
		return;
	    }
	    ++pos;
	    ++linenum;
	} else if (ch == str[0]) {
	    if (strncmp(pos, str + 1, len - 1) == 0) {
		--pos;
		return;
	    }
	}
    }
}


static void
cmd_verb (struct call * c)
{
    char * end;
    char delim = *pos++;

    end = strchr(pos, delim);
    if (!end) {
	fprintf(stderr,
		_("%s:%lu: weird, couldn't find \\verb delimiter `%c'\n"),
		filename, linenum, delim);
	return;
    }
    fputs("<tt>", out->file);
    htmlify(pos, end - pos);
    fputs("</tt>", out->file);
    pos = end + 1;
}


static void
make_footnote (int number, const char * note)
{
    int delta = -2 - current_font_size;
    char sign = '+';

    if (delta < 0) {
	sign = '-';
	delta = -delta;
    }

    fprintf(out->file, "<a name=\"F%d\"><a href=\"footnotes.html#%06d\">"
	    "<sup><fontsize=%c%u>%d</font></sup></a></a>",
	    number, number, sign, delta, number);

    add_to_footnotes(number, note);
}

/*
 * \footnote[number]{note}
 */
static void
cmd_footnote (struct call * c)
{
    int num;

    if (c->arg_list[0]) {
	num = atoi(c->arg_list[0]);
    } else {
	num = ++counter[CNT_FOOTNOTE];
    }
    make_footnote(num, c->arg_list[1]);
}


static void
cmd_maketitle (struct call * c)
{
    if (title_page[TPE_TITLE]) {
	fputs("<center><h1>", out->file);
	push_env(&anonymous_env);
	eval(title_page[TPE_TITLE]);
	pop_env(&anonymous_env);
	fputs("</h1></center>\n", out->file);
    }
    if (title_page[TPE_AUTHOR]) {
	fputs("<center>", out->file);
	push_env(&anonymous_env);
	eval(title_page[TPE_AUTHOR]);
	pop_env(&anonymous_env);
	fputs("</center>\n", out->file);
    }
    if (title_page[TPE_DATE]) {
	fputs("<center>", out->file);
	push_env(&anonymous_env);
	eval(title_page[TPE_DATE]);
	pop_env(&anonymous_env);
	fputs("</center>\n", out->file);
    }
    fputs("\n\n<p>", out->file);
}


static void
cmd_title_page_element (struct call * c)
{
    title_page[(long) c->u.cmd->arg] = c->arg_list[0];
}


static void
cmd_caption (struct call * c)
{
    const char * lst_entry, * nam = 0;
    struct env * e;
    int cnt;

    lst_entry = c->arg_list[1];
    if (c->arg_list[0]) {
	lst_entry = c->arg_list[0];
    }
    
    e = env_top_floating();
    if (!e) {
	fprintf(stderr,
		_("%s:%lu: caption not inside a floating environment\n"),
		filename, linenum);
	return;
    }
    if (e->name[0] == 'f') {
	cnt = counter[CNT_FIGURE];
	nam = "Figure";
	add_dict_entry(&lof, lst_entry, cnt);
    } else {
	cnt = counter[CNT_TABLE];
	nam = "Table";
	add_dict_entry(&lot, lst_entry, cnt);
    }
    fprintf(out->file, "<p><center>%s %d: ", nam, cnt);
    eval(c->arg_list[1]);
    fputs("</center>", out->file);
}


/*
 * Starting at *STR, search for LEFT_DELIM, then search for
 * RIGHT_DELIM, ignoring comments and allowing for properly matched
 * nested occurrences of the LEFT_DELIM/RIGHT_DELIM brackets.  The
 * return value is the character past the first occurrance of
 * LEFT_DELIM.  The matching occurrence of RIGHT_DELIM is replaced by
 * an ASCII NUL.
 */
static char *
parse_arg (const char ** str, char left_delim, char right_delim)
{
    int ch, nesting_level = 0;
    const char * start = 0;
    char * copy;

    while (*str && (ch = *(*str)++)) {
	if (ch == left_delim) {
	    if (nesting_level++ == 0) {
		start = *str;
	    }
	} else if (ch == right_delim) {
	    if (--nesting_level == 0) {
		size_t len = *str - start - 1;
		copy = xmalloc(len + 1);
		memcpy(copy, start, len);
		copy[len] = '\0';
		return copy;
	    }
	} else if (ch == '%' && (*str)[-2] != '\\') {
	    *str = strchr(*str, '\n');
	    if (*str) {
		++linenum;
		++*str;
	    }
	}
    }
    return 0;
}


static char **
parse_args (const char ** str, unsigned pattern)
{
    char ** args;
    int i;

    if (!*str) {
	return 0;
    }

    args = xmalloc(ARG_MAX_SPECS * sizeof(char *));
    memset(args, 0, ARG_MAX_SPECS * sizeof(char *));

    for (i = 0; pattern; ++i, pattern >>= ARG_TYPE_BITS) {
	switch (pattern & ((1 << ARG_TYPE_BITS) - 1)) {
	  case ARG_OPTIONAL:
	      if (**str == '[') {
		  args[i] = parse_arg(str, '[', ']');
	      }
	      break;

	  case ARG_REQUIRED:
	      args[i] = parse_arg(str, '{', '}');
	      if (!args[i]) {
		  free(args);
		  return 0;
	      }
	      break;

	  default:
	      fprintf(stderr, _("%s: bad parse_args() pattern (%x)\n"),
		      prog_name, pattern);
	      break;
	}
    }
    return args;
}


/*
 * A macro call is executed by simply expanding the macro string using
 * the actual arguments and then running eval on it.  Neato.
 */
static void
macro_eval (const char * macro, char ** arg_list)
{
    const char * start, * end;
    size_t size, len;
    char * buf, * dest;
    unsigned argnum;

    size = 1024;
    buf = xmalloc(size);

    dest = buf;
    for (start = macro; (end = strchr(start, '#')); start = end + 2) {
	if ((size_t) ((dest - buf) + (end - start)) > size) {
	    size += (end - start) + 1024;
	    buf = xrealloc(buf, size);
	}
	memcpy(dest, start, end - start);
	dest += end - start;
	argnum = end[1] - '1';
	if (argnum >= ARG_MAX_SPECS || !arg_list || !arg_list[argnum]) {
	    fprintf(stderr, _("%s:%lu: ignoring bad formal macro arg `#%c'\n"),
		    filename, linenum, end[1]);
	    break;
	}
	len = strlen(arg_list[argnum]);
	if ((dest - buf) + len > size) {
	    size += len + 1024;
	    buf = xrealloc(buf, size);
	}
	memcpy(dest, arg_list[argnum], len);
	dest += len;
    }
    len = strlen(start);
    if ((dest - buf) + len >= size) {
	size += len + 1;
	buf = xrealloc(buf, size);
    }
    memcpy(dest, start, len);	/* append last part */
    dest += len;
    *dest++ = '\0';

    eval(buf);
    free(buf);
}


static void
env_floating (struct call * c, int prologue)
{
    fputs("<p>", out->file);

    if (prologue) {
	int * cnt;

	if (c->u.env->name[0] == 'f') {
	    cnt = &counter[CNT_FIGURE];
	} else {
	    cnt = &counter[CNT_TABLE];
	}
	fprintf(out->file, "<a name=\"%c%d\"></a>",
		c->u.env->name[0], ++*cnt);
    }
}


static void
reset_current_default_label (const void * arg)
{
    current_default_label = arg;
}


static void
reset_current_list_type (const void * arg)
{
    long old_list_type = (long) arg;

    current_list_type = old_list_type;
}


/*
 * \item[thislabel]
 */
static void
cmd_item (struct call * c)
{
    switch (current_list_type) {
      case 'i': /* itemize */
      case 'e': /* enumerate */
	  fputs("<li>", out->file);
	  if (c->arg_list[0]) {
	      fprintf(stderr, _("%s:%lu:warning: label in itemize/enumerate "
				"list not properly supported\n"),
		      filename, linenum);
	      eval((char *) c->arg_list[0]);
	      fputc(' ', out->file);
	  }
	  break;

      case 'l': /* list */
	  fputs("<dt>", out->file);
	  if (c->arg_list[0]) {
	      eval(c->arg_list[0]);
	  } else {
	      eval(current_default_label);
	  }
	  break;

      case 't': /* trivlist */
	  fputs("<dd>", out->file);
	  break;

      case 'd': /* description */
	  fputs("<dt>", out->file);
	  if (c->arg_list[0]) {
	      eval(c->arg_list[0]);
	  }
	  fputs("<dd>", out->file);
	  break;

      default:
	  fprintf(stderr,
		  _("%s:%lu: huh, \\item outside of list environment?\n"),
		  filename, linenum);
    }
}


/*
 * A user-defined environment.
 */
static void
env_user (struct call * c, int prologue)
{
    if (prologue) {
	macro_eval(c->u.env->pro, c->arg_list);
    } else {
	eval(c->u.env->epi);
    }
}


/*
 * \begin{rawhtml}
 * \end{rawhtml}
 */
static void
env_rawhtml (struct call * c, int prologue)
{
    char * end;

    if (prologue) {
	end = strstr(pos, "\\end{rawhtml}");
	if (!end) {
	    fprintf(stderr, _("%s:%lu: couldn't find end \\end{rawhtml}\n"),
		    filename, linenum);
	    return;
	}
	while (pos < end)
	    fputc(*pos++, out->file);
    }
}


/*
 * \begin{verbatim}
 * \end{verbatim}
 */
static void
env_verbatim (struct call * c, int prologue)
{
    char * end;

    if (prologue) {
	fputs("<pre>", out->file);

	end = strstr(pos, "\\end{verbatim}");
	if (!end) {
	    fprintf(stderr, _("%s:%lu: couldn't find end \\end{verbatim}\n"),
		    filename, linenum);
	    return;
	}
	htmlify(pos, end - pos);
	pos = end;
    } else {
	fputs("</pre>", out->file);
    }
}


static void
env_theindex (struct call * c, int prologue)
{
    if (prologue) {
	stopat("\\end{theindex}");
    }
}


static void
env_list (struct call * c, int prologue)
{
    if (prologue) {
	fputs(c->u.env->pro, out->file);

	add_cleanup(reset_current_list_type, (void *) current_list_type);
	current_list_type = c->u.env->name[0];

	if (c->u.env->name[0] == 'l') {
	    /* list */
	    add_cleanup(reset_current_default_label, current_default_label);
	    current_default_label = c->arg_list[0];
	}
    } else {
	fputs(c->u.env->epi, out->file);
    }
}


static void
parse_colspec (const char * p, struct table * fmt)
{
    int i, repeat_start, repeat_count, num_repeated;
    char ** arg_list;
    char ch;

    while ((ch = *p++)) {
	switch (ch) {
	  case '*':
	      arg_list = parse_args(&p, ARG_REQ2);
	      if (!arg_list) {
		  fprintf(stderr, _("%s:%lu: skipping bad `*'-spec\n"),
			  filename, linenum);
		  continue;
	      }
	      repeat_count = atoi(arg_list[0]);
	      repeat_start = fmt->num_cols;
	      parse_colspec(arg_list[1], fmt);
	      num_repeated = fmt->num_cols - repeat_start;
	      for (i = 1; i < repeat_count; ++i) {
		  memcpy(fmt->col + fmt->num_cols, fmt->col + repeat_start,
			 num_repeated * sizeof(fmt->col[0]));
		  fmt->num_cols += num_repeated;
	      }
	      break;

	  case '@':
	  case 'p':
	      arg_list = parse_args(&p, ARG_REQ1);
	      if (!arg_list) {
		  fprintf(stderr,
			  _("%s:%lu: bad arg(s) for `%c' table spec\n"),
			  filename, linenum, ch);
	      } else {
		  if (ch == '@') {
		      fmt->col[fmt->num_cols].arg = arg_list[0];
		  }
		  free(arg_list);
	      }
	      /* fall through */
	  case 'l':
	  case 'c':
	  case 'r':
	      fmt->col[fmt->num_cols++].spec = ch;
	      break;

	  case '|':
	      fmt->want_border = 1;
	      break;

	  default:
	      break;	/* ignore everything else... */
	}
    }
}


/*
 * \begin{tabular}[pos]{cols}
 * \end{tabular}
 */
static void
env_tabular (struct call * c, int prologue)
{
    char * cell_end, * cell_start, * p, * align, * wrap, * end;
    int i, colspan, column = 0, done;
    struct table fmt, mcfmt;
    struct cellspec * cell;
    char terminator;
    size_t len;

    if (!prologue) {
	return;
    }

    fmt.num_cols = fmt.want_border = 0;
    parse_colspec(c->arg_list[1], &fmt);

    cell_start = (char *) pos;
    stopat("\\end{tabular}");
    if (!pos) {
	fprintf(stderr, _("%s:%lu: couldn't find \\end{tabular}\n"),
		filename, linenum);
	return;
    }
    len = pos - cell_start;
    p = xmalloc(len + 1);
    memcpy(p, cell_start, len);
    end = p + len;
    *end = '\0';
    cell_start = p;

    if (strstr(cell_start, "\\cline") || strstr(cell_start, "\\hline")) {
	fmt.want_border = 1;
    }

    fputs("<table cellpadding=0 cellspacing=0", out->file);
    if (fmt.want_border) {
	fputs(" border=1" , out->file);
    }
    fputc('>', out->file);

    for (; cell_start < end; cell_start = cell_end) {
	const char * mc;
	char ** mcargs = 0;

	cell = fmt.col + column;
	colspan = 1;
	terminator = '&';
	cell_end = cell_start;
	if (cell->spec != '@') {
	    /* find the end of the cell: */
	    do {
		cell_end = strpbrk(cell_end, "%&\\");
		if (!cell_end) {
		    cell_end = end;
		    break;
		}
		done = 1;
		switch (*cell_end) {
		  case '%':
		      /* comment => skip until end of line */
		      cell_end = strchr(cell_end + 1, '\n');
		      if (cell_end) {
			  done = 0;
		      } else {
			  cell_end = end;
		      }
		      break;

		  case '\\':
		      ++cell_end;
		      if (*cell_end != '\\') {
			  ++cell_end;
			  done = 0;	/* it's not \\ (skip escaped char)*/
		      }
		      break;

		  default:
		      break;
		}
	    } while (!done);
	    terminator = *cell_end;
	    if (terminator == '\\') {
		cell_end[-1] = '\0';
		if (cell_end[1] == '[') {
		    char * str;

		    for (str = cell_end + 2; *str; ++str) {
			if (*str == ']') {
			    cell_end = str + 1;
			}
		    }
		    
		}
	    }
	    *cell_end++ = '\0';

	    /* process cell: */
	    mc = strstr(cell_start, "\\multicolumn");
	    if (mc) {
		mc += strlen("\\multicolumn");
		mcargs = parse_args(&mc, ARG_REQ3);
		if (!mcargs) {
		    fprintf(stderr, _("%s:%lu: bad args for \\multicolumn\n"),
			    filename, linenum);
		    continue;
		}

		mcfmt.num_cols = mcfmt.want_border = 0;
		parse_colspec(mcargs[1], &mcfmt);
		cell_start = mcargs[2];
		cell = &mcfmt.col[0];
		colspan = atoi(mcargs[0]);
		/* multicolumn skips over @{} formats. */
		for (i = 0; column + i < fmt.num_cols && i <= colspan; ++i) {
		    if (fmt.col[column + i].spec == '@') {
			++colspan;
		    }
		}
	    }
	}
	align = "left";
	wrap  = " nowrap";
	switch (cell->spec) {
	  case '@': cell_start = cell->arg; break;
	  case 'p': wrap  = ""; break;
	  case 'c': align = "center"; break;
	  case 'r': align = "right"; break;
	  case 'l':
	  default:
	      break;
	}

	if (column == 0) {
	    fputs("\n<tr valign=top>", out->file);
	}
	fprintf(out->file, "\n<td colspan=%d align=%s%s>",
		colspan, align, wrap);
	push_env(&anonymous_env);
	eval(cell_start);
	pop_env(&anonymous_env);

	fputs("</td>", out->file);
	column += colspan;
	if (terminator != '&') {
	    /* terminator == \ or NUL (end of table) */
	    fputs("</tr>", out->file);
	    column = 0;
	}
	if (mcargs) {
	    free(mcargs);
	}
    }
    fputs("\n</table>", out->file);
}


/*
 * \begin{thebibliography}{widestlabel}
 * \end{thebibliography}
 */
static void
env_thebibliography (struct call * c, int prologue)
{
    char buf[1024];
    FILE * fp;

    if (prologue) {
	/* open file for bibliography: */
	filenum_bibliography = ++counter[CNT_FILE];
	sprintf(buf, FILENAME_FMT, filenum_bibliography);
	fp = open_output(buf);
	if (!fp) {
	    return;
	}
	push_file(fp);

	fputs("<title>", out->file);
	eval("\\DLHbibname");
	fputs("</title>\n<h1>", out->file);
	eval("\\DLHbibname");
	fputs("</h1>\n\n<dl>\n", out->file);
    } else {
	fputs("</dl>\n", out->file);
	fp = pop_file();
	fclose(fp);
    }
}


static void
restore_beginning_of_line_offset (const void * arg)
{
    beginning_of_line_offset = (off_t) arg;
}

/*
 * \begin{tabbing}
 * \end{tabbing}
 * \= \> \kill are meaningful inside tabbing
 */
static void
env_tabbing (struct call * c, int prologue)
{
    if (prologue) {
	add_cleanup (restore_beginning_of_line_offset,
		     (void *) beginning_of_line_offset);
	fputs("\n<table cellspacing=0 cellpadding=0><tr valign=bottom><td>",
	      out->file);
	beginning_of_line_offset = ftell(out->file);
    } else {
	fputs("</td></tr></table>\n", out->file);
    }
}


static struct env environ_list[] = {
    /*
     * This _must_ be the first "environment"---it's used to suppress
     * repeated warnings about unknown environments.
     */
    {},

    /* dlh environments: */
    /*
     * htmlonly is a special environment since the visibility of the
     * definitions in it is not limited to the environment (this is
     * for compatibility with latex2html.).
     */
    {name: "htmlonly"},
    {name: "rawhtml",   op: env_rawhtml},

    /* LaTeX environments: */
    {name: "abstract",
     pro: "<center><b>Abstract</b></center><blockquote>", epi: "</blockquote"},
    {name: "bfseries",                pro: "<b>",  epi: "</b>"},
    {name: "center",		  pro: "<center>", epi: "</center>"},
    {name: "description",
			op: env_list, pro: "<dl>", epi: "</dl>"},
    {name: "document"},
    {name: "enumerate",	op: env_list, pro: "<ol>", epi: "</ol>"},
    {name: "figure",	op: env_floating,		pat: ARG_OPT1},
    {name: "figure*",	op: env_floating,		pat: ARG_OPT1},
    {name: "flushleft",  pro: "<div align=left>", epi: "</div>"},
    {name: "flushright", pro: "<div align=right>", epi: "</div>"},
    {name: "itemize",	op: env_list, pro: "<ul>", epi: "</ul>"},
    {name: "itshape",		      pro: "<i>",  epi: "</i>"},
    {name: "list",	op: env_list, pro: "<dl>", epi: "</dl>", pat: ARG_REQ2},
    {name: "mdseries"},						/* lossage */
    {name: "quote",		  pro: "<blockquote>", epi: "</blockquote>"},
    {name: "quotation",		  pro: "<blockquote>", epi: "</blockquote>"},
    {name: "rmfamily"},						/* lossage */
    {name: "scshape"},						/* lossage */
    {name: "sffamily"},						/* lossage */
    {name: "slshape"},						/* lossage */
    {name: "upshape"},						/* lossage */
    {name: "tabular",	op: env_tabular,			pat: ARG_OPT1_REQ1},
    {name: "tabbing",	op: env_tabbing},
    {name: "thebibliography", op: env_thebibliography, pat: ARG_REQ1},
    {name: "theindex",	op: env_theindex},
    {name: "trivlist",	op: env_list, pro: "<dl>", epi: "</dl>"},
    {name: "ttfamily",		      pro: "<tt>", epi: "</tt>"},
    {name: "verbatim",	op: env_verbatim},
    {name: "table",	op: env_floating,		pat: ARG_OPT1},
    {name: "table*",	op: env_floating,		pat: ARG_OPT1},

    /* math environments: */
    {name: "math",	  op: env_math},
    {name: "displaymath", op: env_displaymath},
    {name: "equation",	  op: env_eqnarray},
    {name: "eqnarray",	  op: env_eqnarray},
    {name: "eqnarray*",	  op: env_eqnarray},
};


/*
 * \begin{env}
 */
static void
cmd_env_begin (struct call * c)
{
    struct call ec;

    ec.u.env = sh_lookup(etab, c->arg_list[0]);
    if (!ec.u.env) {
	fprintf(stderr, _("%s:%lu: ignoring unknown environment `%s'\n"),
		filename, linenum, c->arg_list[0]);
	sh_enter(etab, c->arg_list[0], environ_list);
	return;
    }
    ec.arg_list = 0;
    if (ec.u.env->pat) {
	ec.arg_list = parse_args(&pos, ec.u.env->pat);
	if (!ec.arg_list) {
	    fprintf(stderr, _("%s:%lu: bad args for environment `%s'\n"),
		    filename, linenum, ec.u.env->name);
	    return;
	}
    }
    if (ec.u.env->op) {
	(*ec.u.env->op)(&ec, 1);
    } else if (ec.u.env->pro) {
	fputs(ec.u.env->pro, out->file);
    }
    if (ec.arg_list) {
	free(ec.arg_list);
    }
    push_env(ec.u.env);
}


/*
 * \end{env}
 */
static void
cmd_env_end (struct call * c)
{
    struct env * e = env->e;

    pop_env(e);
    if (e->op) {
	struct call ec;

	memset(&ec, 0, sizeof(ec));
	ec.u.env = e;
	(*e->op)(&ec, 0);
    } else if (e->epi) {
	fputs(e->epi, out->file);
    }
}


/*
 * \psfig{file=path,angle=phi,width=w,height=h}
 */
static void
cmd_psfig (struct call * c)
{
    char * bbox, * p,  * str = c->arg_list[0], * psfile = 0;
    double width = 0.0, height = 0.0, angle = 0.0, bb[4];
    int valid, i, pixel_width, pixel_height, fd;
    double cos_phi, sin_phi, corner[4][2];
    char buf[2048], psfile_path[PATH_MAX];
    const char * file;
    char terminator;
    struct stat st;
    static figure_number = -1;

    ++figure_number;

    for (; str && (p = strpbrk(str, "=,")); str = p) {
	terminator = *p;
	*p++ = '\0';
	valid = 0;
	switch (*str) {
	  case 'a':
	      if (strcmp(str, "angle") == 0) {
		  valid = 1;
		  if (terminator == '=') {
		      str = p;
		      p = strchr(p, ',');
		      if (p) {
			  *p++ = '\0';
		      }
		      angle = strtod(str, 0);
		  }
	      }
	      break;

	  case 'f':
	      if (strcmp(str, "file") == 0) {
		  valid = 1;
		  if (terminator == '=') {
		      psfile = p;
		      p = strchr(p, ',');
		      if (p) {
			  *p++ = '\0';
		      }
		  }
	      }
	      break;

	  case 'w':
	      if (strcmp(str, "width") == 0) {
		  valid = 1;
		  if (terminator == '=') {
		      str = p;
		      p = strchr(p, ',');
		      if (p) {
			  *p++ = '\0';
		      }
		      width = parse_length(str);
		  }
	      }
	      break;

	  case 'h':
	      if (strcmp(str, "height") == 0) {
		  valid = 1;
		  if (terminator == '=') {
		      str = p;
		      p = strchr(p, ',');
		      if (p) {
			  *p++ = '\0';
		      }
		      height = parse_length(str);
		  }
	      }
	      break;
	}
	if (!valid) {
	    fprintf(stderr, _("%s:%lu: ignoring unknown keyword `%s'\n"),
		    filename, linenum, str);
	}
    }
    if (!psfile) {
	fprintf(stderr, _("%s:%lu: huh, no filename in \\psfig??\n"),
		filename, linenum);
	return;
    }

    fd = locate_and_open (psfile, 0, texinputs, psfile_path);
    if (fd < 0) {
	fprintf(stderr, _("%s: %s\n"), psfile, strerror (errno));
	return;
    }
    close (fd);

    file = open_file_maybe(psfile_path, 0, 0);
    bbox = strstr(file, "\n%%BoundingBox:");
    if (!bbox) {
	fprintf(stderr, _("%s: no bounding box found\n"), psfile_path);
	return;
    }
    if (sscanf(bbox, "\n%%%%BoundingBox: %lg %lg %lg %lg\n",
	       bb + 0, bb + 1, bb + 2, bb + 3) != 4)
    {
	fprintf(stderr, _("%s: bad bounding box\n"), psfile_path);
	return;
    }

    /*
     * Rotate the the four corners of the bounding box by ANGLE degrees
     * and determine the new bounding box:
     */
    cos_phi = cos(M_PI * angle / 180.0);
    sin_phi = sin(M_PI * angle / 180.0);

    corner[0][0] = bb[0] * cos_phi - bb[1] * sin_phi;
    corner[0][1] = bb[0] * sin_phi + bb[1] * cos_phi;
    corner[1][0] = bb[2] * cos_phi - bb[3] * sin_phi;
    corner[1][1] = bb[2] * sin_phi + bb[3] * cos_phi;
    corner[2][0] = bb[0] * cos_phi - bb[3] * sin_phi;
    corner[2][1] = bb[0] * sin_phi + bb[3] * cos_phi;
    corner[3][0] = bb[2] * cos_phi - bb[1] * sin_phi;
    corner[3][1] = bb[2] * sin_phi + bb[1] * cos_phi;

    bb[0] = corner[0][0]; bb[1] = corner[0][1];
    bb[2] = corner[0][0]; bb[3] = corner[0][1];
    for (i = 1; i < 4; ++i) {
	if (corner[i][0] < bb[0]) {
	    bb[0] = corner[i][0];
	}
	if (corner[i][1] < bb[1]) {
	    bb[1] = corner[i][1];
	}
	if (corner[i][0] > bb[2]) {
	    bb[2] = corner[i][0];
	}
	if (corner[i][1] > bb[3]) {
	    bb[3] = corner[i][1];
	}
    }

    if (width == 0.0) {
	if (height == 0.0) {
	    width = bb[1] - bb[0];
	} else {
	    width = height / (bb[3] - bb[1]) * (bb[2] - bb[0]);
	}
    }
    if (height == 0.0) {
	height = width / (bb[2] - bb[0]) * (bb[3] - bb[1]);
    }

    pixel_width  = width  / PT_PER_INCH * display_dpi + 0.5;
    pixel_height = height / PT_PER_INCH * display_dpi + 0.5;

    fprintf(out->file, "<img width=%d height=%d src=\"img%03u.gif\">",
	    pixel_width, pixel_height, figure_number);

    sprintf(buf, "%s/img%03u.gif", dir_prefix, figure_number);
    if (!(aux_flags & AUX_FORCE) && stat(buf, &st) == 0) {
	if (verbose) {
	    fprintf(stderr, _("%s.%lu: reusing figure %s for %s\n"),
		    filename, linenum, buf, psfile);
	}
	return;
    }

    if (verbose) {
	fprintf(stderr, _("%s.%lu: translating %s into %s\n"),
		filename, linenum, psfile_path, buf);
    }

    sprintf(buf,
	    "echo '%g %g scale %g neg %g neg translate %g rotate "
	    "(%s) run showpage' | "
	    "gs -q -r%dx%d -g%dx%d -sDEVICE=ppm -sOutputFile=- - | "
	    "ppmtogif -interlace -sort -transparent rgb:f/f/f 2>/dev/null "
	    "> %s/img%03u.gif",
	    /* args to echo: */
	    width / (bb[2] - bb[0]), height / (bb[3] - bb[1]), bb[0], bb[1],
	    angle, psfile_path,
	    /* args to gs: */
	    display_dpi, display_dpi, pixel_width, pixel_height,
	    /* args to ppmtogif */
	    dir_prefix, figure_number);
    system(buf);
}


static void
cmd_today (struct call * c)
{
    char buf[128];
    time_t now;

    time(&now);
    strftime(buf, sizeof(buf), "%B %d, %Y", localtime(&now));
    fputs(buf, out->file);
}


/*
 * \input{filename}
 */
static void
cmd_input (struct call * c)
{
    const char *old_filename = filename;
    unsigned long old_linenum = linenum;

    filename = c->arg_list[0];
    linenum = 1;
    eval(open_file_maybe(filename, "tex", texinputs));
    filename = old_filename;
    linenum = old_linenum;
}


void
cmd_macro (struct call * c)
{
    macro_eval(c->u.cmd->arg, c->arg_list);
}


static void
gen_section_header (char * title, char * filename, int with_toc_entry)
{
    int html_level, appendix_heading = 0;
    char buf[256];

    /*
     * html has h1..h6, LaTeX has \part..\subparagraph (7 levels).  We
     * use h1 for both \part and \chapter.
     */
    html_level = current_section_level;
    if (!html_level) {
	html_level = 1;
    }

    fprintf(out->file, "<h%d>", html_level);

    make_current_section_number_string(buf);
    if (filename) {
	fprintf(out->file, "<a href=\"%s.html\">", filename);
    } else {
	fprintf(out->file, "<a name=\"s%s\">", buf);
	if (with_toc_entry && current_section_level == appendix_level
	    && sh_lookup(ctab, "DLHappendixtitleprefix"))
	{
	    eval("\\DLHappendixtitleprefix");
	}
    }
    if (with_toc_entry) {
	fputs(buf, out->file);
    }
    if (appendix_heading) {
	fprintf(out->file, "</a></h%d>\n<h%d>", html_level, html_level);
    } else {
	fputc(' ', out->file);
    }
    eval(title);
    if (with_toc_entry && !appendix_heading) {
	fputs("</a>", out->file);
    }
    fprintf(out->file, "</h%d>", html_level);
}


/*
 * \section[tocentry]{title}
 * \section*{title}
 */
static void
cmd_section (struct call * c)
{
    int i, level, nsublevels, with_toc_entry;
    char * toc_entry = 0, * title;
    FILE * fp;

    with_toc_entry = (c->u.cmd->name[strlen(c->u.cmd->name) - 1] != '*');
    if (with_toc_entry) {
	title = c->arg_list[1];
	if (c->arg_list[0]) {
	    toc_entry = c->arg_list[0];
	} else {
	    toc_entry = title;
	}
    } else {
	title = c->arg_list[0];
    }

    level = (long) c->u.cmd->arg;

    if (with_toc_entry) {
	++current_section_number[level];

	/* reset numbers of all sublevels: */
	nsublevels = NUM_SECTION_LEVELS - level - 1;
	if (nsublevels > 0) {
	    memset(current_section_number + level + 1, 0,
		   nsublevels * sizeof(current_section_number[0]));
	}
	if (level < min_section_level) {
	    min_section_level = level;
	}
    }

    /* pop back to file at appropriate level: */
    for (i = current_section_level; i >= level; --i) {
	if (section_file_mask & (1 << i)) {
	    section_file_mask &= ~(1 << i);
	    fp = pop_file();
	    fclose(fp);
	    if (!out) {
		fprintf(stderr, _("%s:%lu: internal error: file underflow!\n"),
			filename, linenum);
		exit(1);
	    }
	}
    }
    current_section_level = level;

    if (level <= split_level) {
	char buf[256];

	/* generate header in parent: */
	sprintf(buf, FILENAME_FMT, ++counter[CNT_FILE]);
	gen_section_header(title, buf, with_toc_entry);

	/* open and push new child: */
	fp = open_output(buf);
	if (!fp) {
	    return;
	}
	push_file(fp);
	section_file_mask |= 1 << level;
    }
    /* generate header in child: */
    gen_section_header(title, 0, with_toc_entry);

    /* make <title> tag: */
    fputs("\n<title>", out->file);
    eval(title);
    fputs("</title>\n", out->file);

    if (with_toc_entry) {
	add_to_toc(toc_entry);
    }
}


static void
cmd_appendix (struct call * c)
{
    appendix_level = min_section_level;
    /*
     * Reset the section number of the appendix level, so we start out
     * with appendix A:
     */
    current_section_number[appendix_level] = 0;
}


static int
xref_enter (struct sh * sh, char * name, int must_create)
{
    int i;

    /* see whether a use created an entry already: */
    i = (long) sh_lookup(sh, name) - 1;
    if (i >= 0) {
	if (xtab.def[i].value && must_create) {
	    fprintf(stderr, _("%s:%lu: redefining %s\n"),
		    filename, linenum, name);
	}
    } else {
	if (xtab.ndefs >= xtab.ndefs_alloced) {
	    xtab.ndefs_alloced += 128;
	    xtab.def = xrealloc(xtab.def,
				xtab.ndefs_alloced * sizeof(xtab.def[0]));
	    if (!xtab.def) {
		fprintf(stderr, _("%s:%lu: out of memory\n"),
			filename, linenum);
		return -1;
	    }
	}
	i = xtab.ndefs++;
	memset(xtab.def + i, 0, sizeof(xtab.def[0]));
	xtab.def[i].name = name;
	sh_enter(sh, name, (void *)(long) (i + 1));
    }
    return i;
}


static struct xref_use *
xref_create_use (void)
{
    if (xtab.nuses >= xtab.nuses_alloced) {
	xtab.nuses_alloced += 256;
	xtab.use = xrealloc(xtab.use,
			    xtab.nuses_alloced * sizeof(xtab.use[0]));
	if (!xtab.use) {
	    fprintf(stderr, _("%s:%lu: out of memory\n"),
		    filename, linenum);
	    return 0;
	}
    }
    return xtab.use + xtab.nuses++;
}


/*
 * \label{labelname}
 */
static void
cmd_label_def (struct call * c)
{
    char * name = c->arg_list[0];
    struct xref_def * lbl;
    struct env * e;
    char buf[256];
    int i;

    i = xref_enter(ltab, name, 1);
    if (i < 0) {
	return;
    }
    lbl = xtab.def + i;
    lbl->file_num = counter[CNT_FILE];
    if (aux_flags & AUX_MATH_MODE) {
	/* we're in math mode---use equation number */
	lbl->type = 'e';
	sprintf(buf, "%d", counter[CNT_EQUATION] + 1);
    } else {
	e = env_top_floating();
	if (e) {
	    lbl->type = e->name[0];	/* label refers to table or figure */
	    sprintf(buf, "%d",
		    counter[(e->name[0] == 'f') ? CNT_FIGURE : CNT_TABLE]);
	} else {
	    lbl->type = 's';		/* label refers to a section */
	    make_current_section_number_string(buf);
	}
    }
    lbl->value = strdup(buf);
}


/*
 * \ref{labelname}
 */
static void
cmd_label_use (struct call * c)
{
    char * name = c->arg_list[0];
    struct xref_use * luse;
    int ldef;

    /* lookup definition of label or create an empty def if non-existent: */
    ldef = xref_enter(ltab, name, 0);
    luse = xref_create_use();
    if (ldef < 0 || !luse) {
	return;
    }
    luse->def = ldef;
    luse->pos = ftell(out->file);
    luse->file_num = counter[CNT_FILE];
}


/*
 * \bibitem[label]{citekey}
 *
 * Similar to label_def, but citations use their own name space etc.
 */
static void
cmd_bibitem (struct call * c)
{
    struct xref_def * cdef;
    char * label;
    int i;

    if (c->arg_list[0]) {
	label = c->arg_list[0];
    } else {
	char num[32];

	sprintf(num, "%u", ++counter[CNT_CITATION]);
	label = strdup(num);
    }
    i = xref_enter(btab, c->arg_list[1], 1);
    if (i < 0) {
	return;
    }
    cdef = xtab.def + i;
    cdef->value = label;
    cdef->file_num = counter[CNT_FILE];
    cdef->type = 'b';

    fprintf(out->file, "<dt>[<a name=\"b%s\">%s</a>]\n<dd>", label, label);
}


/*
 * \cite[remark]{key,...}
 */
static void
cmd_cite (struct call * c)
{
    const char * start, * end, * next;
    struct xref_use * cuse;
    int cdef, not_first = 0;
    size_t len;
    char * key;

    fputc('[', out->file);

    start = c->arg_list[1];
    end = start + strlen(start);
    for (; start < end; start = next) {
	if (not_first) {
	    fputs(", ", out->file);
	}
	not_first = 1;

	next = strchr(start, ',');
	if (next) {
	    len = next - start;
	    ++next;
	} else {
	    len = end - start;
	    next = end;
	}
	key = strndup(start, len);
	cdef = xref_enter(btab, key, 0);
	cuse = xref_create_use();
	if (cdef < 0 || !cuse) {
	    return;
	}
	cuse->def = cdef;
	cuse->pos = ftell(out->file);
	cuse->file_num = counter[CNT_FILE];
    }

    if (c->arg_list[0]) {
	if (not_first) {
	    fputs(", ", out->file);
	}
	fputs(c->arg_list[0], out->file);
    }
    fputc(']', out->file);
}


/*
 * \bibliography{file,...}
 *
 * For our purposes, this acts like \input{AUXPREFIX.bbl}.
 */
static void
cmd_bibliography (struct call * c)
{
    const char * old_filename = filename;
    unsigned long old_linenum = linenum;

    linenum = 1;
    filename = aux_prefix;
    eval(open_file_maybe(filename, "bbl", texinputs));
    filename = old_filename;
    linenum = old_linenum;
}


static void
cmd_make_aux (struct call * c)
{
    if (strcmp(c->u.cmd->name, "listoffigures") == 0) {
	aux_flags |= AUX_LOF;
    } else {
	aux_flags |= AUX_LOT;
    }
}


/*
 * \newenvironment{name}[args]{begdef}{enddef}
 * \renewenvironment{name}[args]{begdef}{enddef}
 */
static void
cmd_def_environ (struct call * c)
{
    struct env * env;

    env = xmalloc(sizeof(struct env));
    memset(env, 0, sizeof(*env));

    env->name = c->arg_list[0];
    if (c->arg_list[1]) {
	unsigned i, nargs, pattern = 0;

	nargs = atoi(c->arg_list[1]);
	if (nargs > 9) {
	    fprintf(stderr, _("%s:%lu: can't have more than 9 arguments\n"),
		    filename, linenum);
	    return;
	}
	for (i = 0; i < nargs; ++i) {
	    pattern |= ARG_REQ(i);
	}
	env->pat = pattern;
    }
    env->op  = env_user;
    env->pro = c->arg_list[2];
    env->epi = c->arg_list[3];

    sh_enter(etab, c->arg_list[0], env);
}


/*
 * \newcommand{commandname}[numargs]{body}
 * \renewcommand{commandname}[numargs]{body}
 */
static void
cmd_def_macro (struct call * c)
{
    struct cmd * cmd;
    char * endptr;

    cmd = xmalloc(sizeof(struct cmd));
    memset(cmd, 0, sizeof(*cmd));

    if (c->arg_list[1]) {
	unsigned i, nargs, pattern = 0;

	nargs = strtol(c->arg_list[1], &endptr, 10);
	if (endptr == c->arg_list[1]) {
	    fprintf(stderr,
		    _("%s:%lu: expected argument count, not `%s'\n"),
		    filename, linenum, c->arg_list[1]);
	    return;
	}
	if (nargs > 9) {
	    fprintf(stderr,
		    _("%s:%lu: can't have more than 9 macro arguments\n"),
		    filename, linenum);
	    return;
	}
	for (i = 0; i < nargs; ++i) {
	    pattern |= ARG_REQ(i);
	}
	cmd->pat = pattern;
    }
    cmd->name = c->arg_list[0];
    cmd->op   = cmd_macro;
    cmd->arg  = c->arg_list[2];

    sh_enter(ctab, c->arg_list[0] + 1, cmd);	 /* skip leading `\' */
}


static void
cmd_makeindex (struct call * c)
{
    aux_flags |= AUX_INDEX;
}


/*
 * \mbox{}
 *  or
 * \fbox{}
 * are anonymous environments that also turns off math-mode.
 */
static void
cmd_mbox (struct call * c)
{
    unsigned saved_math_flag = aux_flags & AUX_MATH_MODE;

    push_env(&anonymous_env);
    aux_flags &= ~AUX_MATH_MODE;
    eval(c->arg_list[0]);
    pop_env(&anonymous_env);
    aux_flags |= (aux_flags & ~AUX_MATH_MODE) | saved_math_flag;
}


static void
cmd_centerline (struct call * c)
{
    fputs("<center>", out->file);
    push_env(&anonymous_env);
    eval(c->arg_list[0]);
    pop_env(&anonymous_env);
    fputs("</center>", out->file);
}


static void
cmd_rawhtml (struct call * c)
{
    fputs(c->arg_list[0], out->file);
}


static void
cmd_DLHskipto (struct call * c)
{
    const char * end = strstr (pos, c->arg_list[0]);
    if (end) {
	pos = end;
    } else {
	skip_to_string = strdup (c->arg_list[0]);
	pos = strchr(pos, '\0');
    }
}


static void
cmd_space (struct call * c)
{
    double length = parse_length(c->arg_list[0]);

    fprintf(out->file, "<spacer type=%s size=%ld>",
	    (c->u.cmd->name[0] == 'h') ? "horizontal" : "vertical",
	    (long) length);
}


static void
cmd_newlength (struct call * c)
{
    double * dp;

    dp = xmalloc(sizeof(*dp));
    *dp = 0.0;
    sh_enter(dtab, c->arg_list[0] + 1, dp);	/* skip leading `\' */
}


static void
cmd_setlength (struct call * c)
{
    double * dp;

    dp = sh_lookup(dtab, c->arg_list[0] + 1);
    if (!dp) {
	fprintf(stderr,
		_("%s:%lu: ignoring \\setlength for unknown length `%s'\n"),
		filename, linenum, c->arg_list[0]);
	return;
    }
    *dp = parse_length(c->arg_list[1]);
}


static void
cmd_addtolength (struct call * c)
{
    double * dp;

    dp = sh_lookup(dtab, c->arg_list[0] + 1);
    if (!dp) {
	fprintf(stderr,
		_("%s:%lu: ignoring \\addtolength for unknown length `%s'\n"),
		filename, linenum, c->arg_list[0]);
	return;
    }
    *dp += parse_length(c->arg_list[1]);
}


/*
 * Lossage: we don't know how wide characters are in HTML (depends on
 * browser etc.).  We just guess 10pts/character for now.
 */
static void
cmd_settowidth (struct call * c)
{
    double * dp;
    static warned = 0;

    dp = sh_lookup(dtab, c->arg_list[0] + 1);
    if (!dp) {
	fprintf(stderr,
		_("%s:%lu: ignoring \\settowidth for unknown length `%s'\n"),
		filename, linenum, c->arg_list[0]);
	return;
    }
    if (!warned || verbose) {
	warned = 1;
	fprintf(stderr, _("%s:%lu:warning: "
			  "\\settowidth cannot be translated accurately\n"),
		filename, linenum);
    }
    *dp = strlen(c->arg_list[1]) * 10.0;
}


/*
 * FILELIST is a list of comma-separated filenames.  Each file is
 * opened (using the DLHINPUTS searchpath) and evaluated.  If QUIET is
 * 0, non-existent (or empty) files cause a warning.
 */
static void
input_list (const char * filelist, const char * ext, int quiet)
{
    const char * end, * fname, * str;
    const char * start = filelist;
    size_t len;

    do {
	end = strchr(start, ',');
	if (end) {
	    len = end - start;
	} else {
	    len = strlen(start);
	}
	if (!len) {
	    ++start;
	    continue;
	}
	fname = strndupa(start, len);
	start += len + 1;
	str = open_file(fname, ext, dlhinputs);
	if (!str) {
	    if (!quiet || verbose) {
		fprintf(stderr,
			_("%s.%lu:warning: file %s.%s not found "
			  "along DLHINPUTS (%s)\n"),
			filename, linenum, fname, ext, dlhinputs);
	    }
	    continue;
	}
	eval(str);
    } while (end);
}


/*
 * \documentclass[options]{style}[version]
 */
static void
cmd_documentclass (struct call * c)
{
    input_list(c->arg_list[1], "cls", 0);
    if (c->arg_list[0]) {
	input_list(c->arg_list[0], "sty", 1);
    }
}


/*
 * \documentstyle[options]{style}
 */
static void
cmd_documentstyle (struct call * c)
{
    input_list(c->arg_list[1], "sty", 0);
    if (c->arg_list[0]) {
	input_list(c->arg_list[1], "sty", 1);
    }
}


/*
 * \usepackage[options]{packages}[version]
 */
static void
cmd_usepackage (struct call * c)
{
    const char * packages = c->arg_list[1];

    input_list(packages, "sty", 1);
}


/*
 * Translate a character code from (La-)TeX format into HTML format.
 * The TeX code can be expressed in various bases:
 *	DEC	- decimal number
 *	"HEX	- hexadecimal number
 *	'OCT	- octal number
 *	`ASCII	- ascii character
 *	`\ASCII	- escaped ascii character
 */
static void
translate_char_code (const char * str, const char ** next)
{
    unsigned code, base = 10;

    switch (*str++) {
      case '`':		base =  0; break;
      case '\'':	base =  8; break;
      case '"':		base = 16; break;
      default:		--str; break;
    }
    if (base) {
	code = strtol(str, (char **) next, base);
    } else {
	if (*str == '\\') {
	    ++str;
	}
	code = *str++;
	if (next) {
	    *next = str;
	}
    }
    fprintf(out->file, "&#%u;", code);
}


/*
 * \symbol{DEC}
 * \symbol{"HEX}
 * \symbol{'OCT}
 * \symbol{`ASCII}
 * \symbol{`\ASCII}
 */
static void
cmd_symbol (struct call * c)
{
    translate_char_code(c->arg_list[0], 0);
}


/*
 * Like cmd_symbol() but with TeX syntax (discouraged).
 */
static void
cmd_char (struct call * c)
{
    translate_char_code(pos, &pos);
}


static void
cmd_html_link (struct call * c)
{
    fputs("<a href=\"", out->file);
    eval(c->arg_list[1]);
    fputs("\">", out->file);
    eval(c->arg_list[0]);
    fputs("</a>", out->file);
}


/*
 * \newcounter{countername}[in_counter]
 */
static void
cmd_newcounter (struct call * c)
{
    struct counter * cp;

    if (sh_lookup(counter_tab, c->arg_list[0])) {
	fprintf(stderr, _("%s:%lu: counter %s exists already\n"),
		filename, linenum, c->arg_list[0]);
	return;
    }

    cp = xmalloc(sizeof(*cp));
    memset(cp, 0, sizeof(*cp));

    sh_enter(counter_tab, c->arg_list[0], cp);

    if (c->arg_list[1]) {
	/* this counter gets reset each time arg_list[1] is stepped... */
	struct counter * outer = sh_lookup(counter_tab, c->arg_list[1]);
	cp->next = outer->children;
	outer->children = cp;
    }
}


/*
 * \setcounter{counter}{value}
 *  or
 * \addtocounter{counter}{value}
 */
static void
cmd_inc_counter (struct call * c)
{
    struct counter *cp = sh_lookup(counter_tab, c->arg_list[0]);
    char * end;
    long val;

    if (!cp) {
	fprintf(stderr, _("%s:%lu: counter %s does not exist\n"),
		filename, linenum, c->arg_list[0]);
	return;
    }
    errno = 0;
    val = strtol(c->arg_list[1], &end, 10);
    if (c->arg_list[1] == end || errno == ERANGE) {
	fprintf(stderr, _("%s:%lu: ignoring bad integer value %s\n"),
		filename, linenum, c->arg_list[0], c->arg_list[1]);
	return;
    }
    if (c->u.cmd->name[0] == 'a') {
	cp->value += val;		/* \addtocounter */
    } else {
	cp->value = val;		/* \setcounter */
    }
}


static void
reset_children (struct counter *cp)
{
    while (cp) {
	cp->value = 0;
	if (cp->children) {
	    reset_children(cp->children);
	}
	cp = cp->next;
    }
}


/*
 * \stepcounter{counter}
 *  or
 * \refstepcounter{counter}
 */
static void
cmd_stepcounter (struct call * c)
{
    struct counter *cp = sh_lookup(counter_tab, c->arg_list[0]);

    if (!cp) {
	fprintf(stderr, _("%s:%lu: counter %s does not exist\n"),
		filename, linenum, c->arg_list[0]);
	return;
    }

    ++cp->value;
    if (cp->children) {
	reset_children(cp->children);
    }
}


static void
to_roman (long value, int upcase)
{
    int next, pos = 0;
    struct digit {
	char	code;
	int	value;
    } digit[] = {
	{'m', 1000},
	{'d', 500},
	{'c', 100}, 
	{'l', 50},
	{'x', 10},
	{'v', 5},
	{'i', 1}
    };

    if (value < 0) {
	fputc('-', out->file);
	value = -value;
    }

    pos = 0;
    while (value) {
	next = pos + 1 + (~pos & 1);
	if (value >= digit[pos].value) {
	    fputc(digit[pos].code - (upcase ? 'a' - 'A' : 0), out->file);
	    value -= digit[pos].value;
	} else if (value >= digit[pos].value - digit[next].value) {
	    fputc(digit[next].code - (upcase ? 'a' - 'A' : 0), out->file);
	    fputc(digit[pos ].code - (upcase ? 'a' - 'A' : 0), out->file);
	    value -= digit[pos].value - digit[next].value;
	} else {
	    ++pos;
	}
    }
}


static void
cmd_value (struct call * c)
{
    struct counter *cp = sh_lookup(counter_tab, c->arg_list[0]);
    long value;

    if (!cp) {
	fprintf(stderr, _("%s:%lu: counter %s does not exist\n"),
		filename, linenum, c->arg_list[0]);
	return;
    }
    value = cp->value;

    switch (c->u.cmd->name[0]) {
      case 'A':				/* Alph */
	  if (value >= 1 && value <= 26) {
	      fputc('A' + value - 1, out->file);
	      return;
	  }
	  break;

      case 'a':				/* alph */
	  if (c->u.cmd->name[1] == 'l' && value >= 1 && value <= 26) {
	      fputc('a' + value - 1, out->file);
	      return;
	  }
	  break;

      case 'r':				/* roman */
	  to_roman(value, 0);
	  return;

      case 'R':				/* Roman */
	  to_roman(value, 1);
	  return;

      case 'f':
	  /* use arabic for now */
	  break;
    }
    fprintf(out->file, "%ld", value);	/* default to arabic */
}


static struct cmd cmd_list[] = {
    /*
     * This _must_ be the first "command"---it's used to suppress
     * repeated warnings about unknown commands.
     */
    {},

    /* dlh commands: */
    {name: "DLHbibname",	op: cmd_emit, arg: "References"},
    {name: "DLHskipto",		pat: ARG_REQ1, op: cmd_DLHskipto},
    {name: "rawhtml",		pat: ARG_REQ1, op: cmd_rawhtml},
    {name: "htmladdnormallink",	    pat: ARG_REQ2, op: cmd_html_link},
    {name: "htmladdnormallinkfoot", pat: ARG_REQ2, op: cmd_html_link},

    /* a _few_ TeX commands---the fewer, the better: */
    {name: "char",		op: cmd_char},
    {name: "hfill",		op: cmd_macro, arg: "\\hspace{\\fill}"},

    /* a few macros: */
    {name: "bigskip",	op: cmd_macro, arg: "\\vspace{\\bigskipamount}"},
    {name: "emph",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\em #1}"},
    {name: "medskip",	op: cmd_macro, arg: "\\vspace{\\medskipamount}"},
    {name: "qquad",	op: cmd_macro, arg: "\\hspace{2em}"},
    {name: "quad",	op: cmd_macro, arg: "\\hspace{1em}"},
    {name: "smallskip", op: cmd_macro, arg: "\\vspace{\\smallskipamount}"},
    {name: "textbf",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\bf #1}"},
    {name: "textit",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\it #1}"},
    {name: "textmd",    pat: ARG_REQ1, op: cmd_macro, arg: "#1"},
    {name: "textnormal",pat: ARG_REQ1, op: cmd_macro, arg: "#1"},
    {name: "textrm",    pat: ARG_REQ1, op: cmd_macro, arg: "#1"},
    {name: "textsc",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\sc #1}"},
    {name: "textsf",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\sf #1}"},
    {name: "textsl",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\sl #1}"},
    {name: "texttt",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\tt #1}"},
    {name: "textup",    pat: ARG_REQ1, op: cmd_macro, arg: "#1"},
    /* global commands: */
    {name: "documentclass", pat: ARG_OPT1_REQ1_OPT1, op: cmd_documentclass},
    {name: "documentstyle", pat: ARG_OPT1_REQ1,	     op: cmd_documentstyle},
    {name: "usepackage",    pat: ARG_OPT1_REQ1_OPT1, op: cmd_usepackage},
    {name: "pagestyle",	    pat: ARG_REQ1},
    {name: "appendix",				     op: cmd_appendix},

    /* miscellaneous commands: */
    {name: ","},
    {name: "/"},
    {name: "LaTeX",			       op: cmd_emit, arg: "LaTeX"},
    {name: "P",				       op: cmd_emit, arg: "&#182;"},
    {name: "S",				       op: cmd_emit, arg: "&#167;"},
    {name: "TeX",			       op: cmd_emit, arg: "TeX"},
    {name: "kill",			       op: cmd_kill},
    {name: "=",				       op: cmd_tab_move},
    {name: ">",				       op: cmd_tab_move},
    {name: "\\",		pat: ARG_OPT1, op: cmd_line_break},
    {name: "\\*",		pat: ARG_OPT1, op: cmd_line_break},
    {name: "addvspace",		pat: ARG_REQ1},
    {name: "caption",		pat: ARG_OPT1_REQ1, op: cmd_caption},
    {name: "centerline",	pat: ARG_REQ1, op: cmd_centerline},
    {name: "cline",		pat: ARG_REQ1},
    {name: "copyright",			       op: cmd_emit, arg: "&#169;"},
    {name: "fbox",		pat: ARG_REQ1, op: cmd_mbox},
    {name: "footnote",		pat: ARG_OPT1_REQ1, op: cmd_footnote},
    {name: "hline"},
    {name: "hrule",			       op: cmd_emit, arg: "<hr>"},
    {name: "hspace",		pat: ARG_REQ1, op: cmd_space},
    {name: "hspace*",		pat: ARG_REQ1, op: cmd_space},
    {name: "index",		pat: ARG_REQ1, op: cmd_add_to_index},
    {name: "item",		pat: ARG_OPT1, op: cmd_item},
    {name: "ldots",			       op: cmd_emit, arg: "..."},
    {name: "leavehmode"},
    {name: "leavevmode"},
    {name: "listoffigures",		       op: cmd_make_aux},
    {name: "listoftables",		       op: cmd_make_aux},
    {name: "linebreak",		pat: ARG_OPT1, op: cmd_line_break},
    {name: "makeindex",			       op: cmd_makeindex},
    {name: "mbox",		pat: ARG_REQ1, op: cmd_mbox},
    {name: "noindent"},
    {name: "par",			       op: cmd_line_break},
    {name: "psfig",		pat: ARG_REQ1, op: cmd_psfig},
    {name: "raggedright"},					/* lossage */
    {name: "symbol",		pat: ARG_REQ1, op: cmd_symbol},
    {name: "sloppy"},
    {name: "tableofcontents"},
    {name: "today",			       op: cmd_today},
    {name: "vdots",			       op: cmd_emit, arg: ":"},
    {name: "verb",			       op: cmd_verb},	/* lossage */
    {name: "vspace",		pat: ARG_REQ1, op: cmd_space},
    {name: "vspace*",		pat: ARG_REQ1, op: cmd_space},

    /* counter-related commands */
    {name: "newcounter",	pat: ARG_REQ1_OPT1, op: cmd_newcounter},
    {name: "setcounter",	pat: ARG_REQ2, op: cmd_inc_counter},
    {name: "addtocounter",	pat: ARG_REQ2, op: cmd_inc_counter},
    {name: "stepcounter",	pat: ARG_REQ1, op: cmd_stepcounter},
    {name: "refstepcounter",	pat: ARG_REQ1, op: cmd_stepcounter},
    {name: "value",		pat: ARG_REQ1, op: cmd_value},
    {name: "arabic",		pat: ARG_REQ1, op: cmd_value},
    {name: "Roman",		pat: ARG_REQ1, op: cmd_value},
    {name: "roman",		pat: ARG_REQ1, op: cmd_value},
    {name: "alph",		pat: ARG_REQ1, op: cmd_value},
    {name: "Alph",		pat: ARG_REQ1, op: cmd_value},
    {name: "fnsymbol",		pat: ARG_REQ1, op: cmd_value},

    /* dimension-related commands: */
    {name: "stretch",		pat: ARG_REQ1},
    {name: "newlength",		pat: ARG_REQ1, op: cmd_newlength},
    {name: "setlength",		pat: ARG_REQ2, op: cmd_setlength},
    {name: "addtolength",	pat: ARG_REQ2, op: cmd_addtolength},
    {name: "settowidth",	pat: ARG_REQ2, op: cmd_settowidth},

    /* bibliography commands: */
    {name: "bibliography",	pat: ARG_REQ1, op: cmd_bibliography},
    {name: "bibliographystyle",	pat: ARG_REQ1},
    {name: "bibitem",		pat: ARG_OPT1_REQ1, op: cmd_bibitem},
    {name: "cite",		pat: ARG_OPT1_REQ1, op: cmd_cite},
    {name: "nocite",		pat: ARG_REQ1},
    {name: "newblock"},

    /* meta commands: */
    {name: "newenvironment",	pat: ARG_REQ1_OPT1_REQ2, op: cmd_def_environ},
    {name: "renewenvironment",	pat: ARG_REQ1_OPT1_REQ2, op: cmd_def_environ},
    {name: "newcommand",	pat: ARG_REQ1_OPT1_REQ1, op: cmd_def_macro},
    {name: "renewcommand",	pat: ARG_REQ1_OPT1_REQ1, op: cmd_def_macro},

    /* title-page: */
    {name: "maketitle",	op: cmd_maketitle},
    {name: "thanks",	pat: ARG_REQ1, op: cmd_macro, arg: "\\footnote{#1}"},
    {name: "title",	pat: ARG_REQ1, op: cmd_title_page_element,
     arg: (void *) TPE_TITLE},
    {name: "author",	pat: ARG_REQ1, op: cmd_title_page_element,
     arg: (void *) TPE_AUTHOR},
    {name: "date",	pat: ARG_REQ1, op: cmd_title_page_element,
     arg: (void *) TPE_DATE},

    /* sectioning (starred versions unsupported for now): */
    {name: "part",	 pat: ARG_OPT1_REQ1, op: cmd_section, arg: (void *) 0},
    {name: "part*",	 pat: ARG_REQ1,	     op: cmd_section, arg: (void *) 0},
    {name: "chapter",	 pat: ARG_OPT1_REQ1, op: cmd_section, arg: (void *) 1},
    {name: "chapter*",	 pat: ARG_REQ1,	     op: cmd_section, arg: (void *) 1},
    {name: "section",	 pat: ARG_OPT1_REQ1, op: cmd_section, arg: (void *) 2},
    {name: "section*",	 pat: ARG_REQ1,	     op: cmd_section, arg: (void *) 2},
    {name: "subsection", pat: ARG_OPT1_REQ1, op: cmd_section, arg: (void *) 3},
    {name: "subsection*",
			 pat: ARG_REQ1,      op: cmd_section, arg: (void *) 3},
    {name: "subsubsection",
			 pat: ARG_OPT1_REQ1, op: cmd_section, arg: (void *) 4},
    {name: "subsubsection*",
			 pat: ARG_REQ1,      op: cmd_section, arg: (void *) 4},
    {name: "paragraph",	 pat: ARG_OPT1_REQ1, op: cmd_section, arg: (void *) 5},
    {name: "paragraph*", pat: ARG_REQ1,	     op: cmd_section, arg: (void *) 5},
    {name: "subparagraph",
			 pat: ARG_OPT1_REQ1, op: cmd_section, arg: (void *) 6},
    {name: "subparagraph*",
			 pat: ARG_REQ1,      op: cmd_section, arg: (void *) 6},

    /* cross-referencing: */
    {name: "label",		pat: ARG_REQ1, op: cmd_label_def},
    {name: "ref",		pat: ARG_REQ1, op: cmd_label_use},
    {name: "pageref",		pat: ARG_REQ1, op: cmd_label_use},

    /* environment-related commands: */
    {name: "begin",		pat: ARG_REQ1, op: cmd_env_begin},
    {name: "end",		pat: ARG_REQ1, op: cmd_env_end},

    /* file-related commands: */
    {name: "input",		pat: ARG_REQ1, op: cmd_input},

    /* page-related commands: */
    {name: "thispagestyle",	pat: ARG_REQ1},
    {name: "cleardoublepage"},
    {name: "clearpage"},
    {name: "flushbottom"},
    {name: "newpage"},
    {name: "nopagebreak",	pat: ARG_OPT1},
    {name: "pagebreak",		pat: ARG_OPT1},
    {name: "pagenumbering",	pat: ARG_REQ1},
    {name: "raggedbottom"},
    {name: "samepage"},
 
    /* font changing commands: */
    {name: "rm"},						/* lossage */
    {name: "em",		op: cmd_set_font, arg: "em"},
    {name: "it",		op: cmd_set_font, arg: "i"},
    {name: "bf",		op: cmd_set_font, arg: "b"},
    {name: "tt",		op: cmd_set_font, arg: "tt"},
    {name: "sc"},						/* lossage */
    {name: "sl",		op: cmd_set_font, arg: "it"},	/* lossage */
    {name: "sf"},						/* lossage */
    {name: "tiny",		op: cmd_set_fontsize, arg: (void *) -4},
    {name: "scriptsize",	op: cmd_set_fontsize, arg: (void *) -3},
    {name: "footnotesize",	op: cmd_set_fontsize, arg: (void *) -2},
    {name: "small",		op: cmd_set_fontsize, arg: (void *) -1},
    {name: "normalsize",	op: cmd_set_fontsize, arg: (void *)  0},
    {name: "large",		op: cmd_set_fontsize, arg: (void *)  1},
    {name: "Large",		op: cmd_set_fontsize, arg: (void *)  2},
    {name: "LARGE",		op: cmd_set_fontsize, arg: (void *)  3},
    {name: "huge",		op: cmd_set_fontsize, arg: (void *)  4},
    {name: "Huge",		op: cmd_set_fontsize, arg: (void *)  5},
};


static void
eval (const char * str)
{
    const char * old_pos = pos;
    unsigned long old_linenum = linenum;
    const char * start;
    struct call call;
    int ch, new_lines, is_alpha;

    pos = str;
    while ((ch = *pos++)) {
	switch (ch) {
	  case '\\':
	      start = pos;
	      is_alpha = 0;
	      switch (*pos) {
		case '-': /* soft hyphens have no HTML equivalent */
		    ++pos;
		    continue;

		case '{': case '}': case '_': case '#': case '%': case ' ':
		case '&': case '$':
		    /*
		     * Escaped special characters produce characters
		     * themselves:
		     */
		    fputc(*pos++, out->file);
		    continue;

		case '\\': case '/': case ',': case '>':
		    /* Commands with weird names, such as \\ and \/: */
		    ++pos;
		    break;

		    /* abbreviations for math environments: */
		case '[': ++pos; eval("\\begin{displaymath}"); continue;
		case ']': ++pos; eval("\\end{displaymath}"); continue;
		case '(': ++pos; eval("\\begin{math}"); continue;
		case ')': ++pos; eval("\\end{math}"); continue;

		case '=':
		    if (pos[1] != '{') {
			/* \= is a tabbing command, \={} an accent */
			++pos;
			break;
		    }
		case '`': case '\'': case '.': 
		case '^': case '"': case '~':
		    if (pos[1] != '{') {
			accented_char(pos[0], pos[1]);
			pos += 2;
			continue;
		    }
		    /* fall through */
		case 'b': case 'c': case 'd': case 'H':
		case 'r': case 't': case 'u': case 'v':
		    if (pos[1] == '{' && pos[2] && pos[3] == '}') {
			accented_char(pos[0], pos[2]);
			pos += 3;
			continue;
		    }
		    /* fall through */
		default:
		    is_alpha = 1;
		    while (isalpha(*pos)) ++pos;
		    break;
	      }
	      if (*pos == '*') {
		  ++pos;	/* trailing `*' is part of command name */
	      }
	      /*
	       * If we're in math mode, check first whether its a math
	       * command.  If it is not, check the regular table.
	       */
	      if (!((aux_flags & AUX_MATH_MODE)
		    && (call.u.cmd = sh_nlookup(mtab, start, pos - start))))
	      {
		  call.u.cmd = sh_nlookup(ctab, start, pos - start);
	      }
	      if (is_alpha && (*pos == ' ' || *pos == '\t')) {
		  ++pos;	/* TeX eats the whitespace behind a command */
	      }
	      if (call.u.cmd) {
		  call.arg_list = 0;
		  if (call.u.cmd->pat) {
		      call.arg_list = parse_args(&pos, call.u.cmd->pat);
		      if (!call.arg_list) {
			  fprintf(stderr,
				  _("%s:%lu: bad args for command `\\%s'\n"),
				  filename, linenum, call.u.cmd->name);
			  break;
		      }
		  }
		  if (call.u.cmd->op) {
		      (*call.u.cmd->op)(&call);
		  }
		  if (call.arg_list) {
		      free(call.arg_list);
		  }
	      } else {
		  char * name;

		  name = strndup(start, pos - start);
		  if (!name) {
		      fprintf(stderr, _("%s:%lu: out of memory\n"),
			      filename, linenum);
		      continue;
		  }

		  if (aux_flags & AUX_MATH_MODE) {
		      /*
		       * Emit unknown command literally so there is at
		       * least a chance that the formula will remain
		       * readable.
		       */
		      fputc('\\', out->file); fputs(name, out->file);
		  } else {
		      fprintf(stderr,
			      _("%s:%lu: ignoring unknown command `%s'\n"),
			      filename, linenum, name);
		      sh_enter(ctab, name, cmd_list);
		  }
	      }
	      break;

	  case '$':	/* toggle math mode */
	      if (aux_flags & AUX_MATH_MODE) {
		  eval("\\end{math}");
	      } else {
		  eval("\\begin{math}");
	      }
	      break;

	  case '_':	/* subscript */
	      if (aux_flags & AUX_MATH_MODE) {
		  math_script(0);
	      } else {
		  fprintf(stderr,
			  _("%s:%lu: sorry, _ is for math mode only\n"),
			  filename, linenum);
	      }
	      break;

	  case '^':	/* superscript */
	      if (aux_flags & AUX_MATH_MODE) {
		  math_script(1);
	      } else {
		  fprintf(stderr,
			  _("%s:%lu: sorry, ^ is for math mode only\n"),
			  filename, linenum);
	      }
	      break;

	  case '{':
	      push_env(&anonymous_env);
	      break;

	  case '}':
	      pop_env(&anonymous_env);
	      break;

	  case '~':
	      fputc(' ', out->file);
	      break;

	  case '\n':
	      new_lines = 1;
	      while (*pos == '\n') {
		  ++new_lines;
		  ++pos;
	      }
	      if (new_lines > 1) {
		  fputs("\n\n<p>", out->file);
	      } else {
		  fputc(ch, out->file);
	      }
	      linenum += new_lines;
	      break;

	  case '%':
	      pos = strchr(pos, '\n');
	      if (!pos) {
		  pos = "";
		  break;
	      }
	      ++pos;
	      ++linenum;
	      break;

	  case '&':
	      if (aux_flags & AUX_MATH_MODE) {
		  math_next_column ();
	      } else {
		  fprintf(stderr,
			  _("%s:%lu: found & outside of table or eqnarray\n"),
			  filename, linenum);
	      }
	      break;

	  default:
	      htmlify_char(ch);
	      break;
	}
    }
    pos = old_pos;
    linenum = old_linenum;
    if (skip_to_string) {
	const char * end = strstr (pos, skip_to_string);
	if (end) {
	    free ((void *) skip_to_string);
	    skip_to_string = 0;
	    pos = end;
	}
    }
}


int
cmp_index_entry (const void * left, const void * right)
{
    const struct dict_entry * l = left;
    const struct dict_entry * r = right;

    return strcasecmp(l->key, r->key);
}


void
gen_index (void)
{
    struct dict_entry * idx;
    const char * prev_key = "";
    int prev_label = -1;
    char buf[1024];

    qsort(itab.dict, itab.len, sizeof(itab.dict[0]), cmp_index_entry);

    /* open file for index: */
    sprintf(buf, FILENAME_FMT, ++counter[CNT_FILE]);
    f[OUT_INDEX] = open_output(buf);
    if (!f[OUT_INDEX]) {
	return;
    }
    push_file(f[OUT_INDEX]);

    fputs("<h1>Index</h1>\n\n\n<p><multicol cols=2>", out->file);
    for (idx = itab.dict; idx < itab.dict + itab.len; ++idx) {
	if (tolower(prev_key[0]) != tolower(idx->key[0])) {
	    fputs("\n<p>", out->file);
	}
	if (strcmp(prev_key, idx->key) == 0 && prev_label == idx->label) {
	    continue;	/* skip identical entries */
	}
	fprintf(out->file, "<a href=\"" FILENAME_FMT ".html#i%d\">",
		idx->file_num, idx->label);
	eval(idx->key);
	fputs("</a><br>\n", out->file);
	prev_key = idx->key;
	prev_label = idx->label;
    }
    fputs("</multicol>", out->file);
    pop_file();
}


int
cmp_xref_use (const void * left, const void * right)
{
    const struct xref_use * l = left;
    const struct xref_use * r = right;

    if (l->file_num != r->file_num) {
	return l->file_num - r->file_num;
    }
    return (l->pos > r->pos) - (l->pos < r->pos);
}


static void
gen_navigation_bar (int fd, int file_num)
{
    char buf[2048];

    /* next button: */
    if (file_num < counter[CNT_FILE]) {
	sprintf(buf, "<a href=\"" FILENAME_FMT ".html\">"
		"<img src=%s/next.gif alt=\"Next\"></a>\n",
		file_num + 1, icon_dir);
    } else {
	sprintf(buf,
		"<img src=%s/next_gr.gif border=2 alt=\"Previous\"></a>\n",
		icon_dir);
    }
    WRITE_STR(fd, buf);

    /* up button: */
    if (file_num != 0) {
	sprintf(buf, "<a href=\"" FILENAME_FMT ".html\">"
		"<img src=%s/up.gif alt=\"Up\"></a>\n", 0, icon_dir);
    } else {
	sprintf(buf, "<img src=%s/up_gr.gif border=2 alt=\"Previous\"></a>\n",
		icon_dir);
    }
    WRITE_STR(fd, buf);

    /* previous button: */
    if (file_num > 0) {
	sprintf(buf, "<a href=\"" FILENAME_FMT ".html\">"
		"<img src=%s/previous.gif alt=\"Previous\"></a>\n",
		file_num - 1, icon_dir);
    } else {
	sprintf(buf,
		"<img src=%s/previous_gr.gif border=2 alt=\"Previous\"></a>\n",
		icon_dir);
    }
    WRITE_STR(fd, buf);

    /* contents button */
    if (f[OUT_CONTENTS]) {
	sprintf(buf, "<a href=\"" FILENAME_FMT ".html\">"
		"<img src=%s/contents.gif alt=\"Contents\"></a>\n",
		0, icon_dir);
	WRITE_STR(fd, buf);
    }

    /* references (bibliography) button */
    if (filenum_bibliography) {
	sprintf(buf, "<a href=\"" FILENAME_FMT ".html\">"
		"<img src=%s/references.gif alt=\"References\"></a>\n",
		filenum_bibliography, icon_dir);
	WRITE_STR(fd, buf);
    }

    /* index button */
    if (f[OUT_INDEX]) {
	sprintf(buf, "<a href=\"" FILENAME_FMT ".html\">"
		"<img src=%s/index.gif alt=\"Index\"></a>\n",
		counter[CNT_FILE], icon_dir);
	WRITE_STR(fd, buf);
    }
    WRITE_STR(fd, "<hr>\n");
}


static void
gen_file_header (int fd, int file_num)
{
    WRITE_STR(fd, "<html><body>\n");

    gen_navigation_bar(fd, file_num);
}


static void
gen_file_trailer (int fd, int file_num)
{
    WRITE_STR(fd, "<p><hr>\n");

    gen_navigation_bar(fd, file_num);

    WRITE_STR(fd, "</body></html>\n");
}


static int
reopen_file (char buf[8192], int * ifd, int * ofd, int file_num)
{
    sprintf(buf, "%s/" FILENAME_FMT ".html", dir_prefix, file_num);
    *ifd = open(buf, O_RDONLY);
    if (!*ifd) {
	perror(buf);
	return 0;
    }
    unlink(buf);
    *ofd = open(buf, O_WRONLY | O_EXCL | O_CREAT, 0666);
    if (!*ofd) {
	perror(buf);
	return 0;
    }
    return 1;
}


static void
finish_file (char buf[8192], int ifd, int ofd, int file_num)
{
    size_t n;

    if (ifd < 0) {
	return;
    }

    /* copy the rest of the input file to the output: */
    while (1) {
	n = read(ifd, buf, sizeof(buf));
	if (n == 0) {
	    break;
	} if ((ssize_t) n == -1) {
	    fprintf(stderr, _("%s: error reading file\n"),
		    prog_name);
	    return;
	}
	if (write(ofd, buf, n) != (ssize_t) n) {
	    fprintf(stderr, _("%s: short write\n"), prog_name);
	    return;
	}
    }
    gen_file_trailer(ofd, file_num);
    close(ifd); close(ofd);
}


/*
 * Now make a second pass through all output files, adding navigation
 * bars and fixing cross references.
 */
static void
finalize (void)
{
    int ifd = -1, ofd = -1;
    struct xref_use * xuse;
    struct xref_def * xdef;
    int prev_file_num = -1;
    long prev_pos = 0;
    size_t n, to_copy;
    char buf[8192];

    /* first, sort cross-reference uses according to file and position: */
    qsort(xtab.use, xtab.nuses, sizeof(xtab.use[0]), cmp_xref_use);

    for (xuse = xtab.use; xuse < xtab.use + xtab.nuses; ++xuse) {
	/* open & create file if necessary */
	if (xuse->file_num != prev_file_num) {
	    do {
		finish_file(buf, ifd, ofd, prev_file_num);

		++prev_file_num;
		if (!reopen_file(buf, &ifd, &ofd, prev_file_num)) {
		    return;
		}
		gen_file_header(ofd, prev_file_num);
	    } while (prev_file_num < xuse->file_num);
	    prev_pos = 0;
	}

	/* now copy all bytes up to next label use: */
	to_copy = xuse->pos - prev_pos;
	while (to_copy > 0) {
	    n = to_copy;
	    if (n > sizeof(buf)) {
		n = sizeof(buf);
	    }
	    n = read(ifd, buf, n);
	    if ((ssize_t) n == -1) {
		fprintf(stderr, _("%s: error reading file\n"), prog_name);
		return;
	    }
	    n = write(ofd, buf, n);
	    if ((ssize_t) n == -1 || n == 0) {
		fprintf(stderr, _("%s: error writing file\n"), prog_name);
		return;
	    }
	    to_copy -= n;
	}
	prev_pos = xuse->pos;

	/* now insert the cross-reference: */
	xdef = xtab.def + xuse->def;
	if (xdef->value) {
	    sprintf(buf, "<a href=\"" FILENAME_FMT ".html#%c%s\">%s</a>",
		    xdef->file_num, xdef->type, xdef->value, xdef->value);
	} else {
	    if (!xdef->file_num) {
		fprintf(stderr, _("%s: %s `%s' undefined\n"),
			prog_name, xdef->type == 'b' ? "citation" : "label",
			xdef->name);
	    }
	    xdef->file_num = 1;	/* suppress future warnings */
	    strcpy(buf, "<b><font size=+1>??</font></b>");
	}
	n = strlen(buf);
	if ((size_t) write(ofd, buf, n) != n) {
	    fprintf(stderr, _("%s: short write\n"), prog_name);
	    return;
	}
    }
    finish_file(buf, ifd, ofd, prev_file_num);

    while (++prev_file_num <= counter[CNT_FILE]) {
	if (!reopen_file(buf, &ifd, &ofd, prev_file_num)) {
	    return;
	}
	gen_file_header(ofd, prev_file_num);
	finish_file(buf, ifd, ofd, prev_file_num);
    }
}


int
main (int argc, char ** argv)
{
    char * slash, * dot, buf[1024];
    int i, ch;
    FILE * fp;

    prog_name = strrchr(argv[0], '/');
    if (prog_name) {
	++prog_name;
    } else {
	prog_name = argv[0];
    }
    xmalloc_set_program_name(prog_name);

#if ENABLE_NLS
    textdomain(PACKAGE);
#endif

    ltab = sh_new(1024);	/* create label hash table */
    btab = sh_new(1024);	/* create bibiliography hash table */

    /* enter all pre-defined counters: */
    counter_tab = sh_new(1024);

    /* enter all commands/environments in respective hash-tables: */
    ctab = sh_new(1024);
    for (i = 1; i < NELEMS(cmd_list); ++i) {
	sh_enter(ctab, cmd_list[i].name, cmd_list + i);
    }
    etab = sh_new(1024);
    for (i = 1; i < NELEMS(environ_list); ++i) {
	sh_enter(etab, environ_list[i].name, environ_list + i);
    }

    /* create a few lengths (dimensions): */
    dtab = sh_new(128);
    for (i = 0; i < NELEMS(dim_list); ++i) {
	sh_enter(dtab, dim_list[i].name, (void *) &dim_list[i].length);
    }

    math_init();		/* initialize math-related stuff */

    dlhinputs = getenv("DLHINPUTS");
    if (dlhinputs) {
	size_t len = strlen(dlhinputs);

	/* trailing colon means: append default directory list */
	if (dlhinputs[len - 1] == ':') {
	    void * mem = xmalloc(len + sizeof(DLHINPUTSDIR));
	    memcpy(mem, dlhinputs, len);
	    memcpy(mem + len, DLHINPUTSDIR, sizeof(DLHINPUTSDIR));
	    dlhinputs = mem;
	}
    } else {
	dlhinputs = DLHINPUTSDIR;
    }
    texinputs = getenv("TEXINPUTS");
    if (!texinputs) {
	texinputs = ".";
    }

    while ((ch = getopt_long(argc, argv, "fi:vV", opts, NULL)) != EOF) {
	switch (ch) {
	  case 'f':
	      aux_flags ^= AUX_FORCE;
	      break;

	  case 'h':
	      usage();
	      exit(0);

	  case 'i':
	      icon_dir = optarg;
	      break;

	  case 'l':
	      split_level = atoi (optarg);
	      break;

	  case 'r':
	      display_dpi = atoi (optarg);
	      break;

	  case 'v':
	      verbose ^= 1;
	      break;

	  case 'V':
	      printf("%s version %s\n", PACKAGE, VERSION);
	      exit(0);

	  default:
	      usage();
	      exit(1);
	}
    }

    /* load site-start.sty, if it exists */
    input_list("site-start", "sty", 1);

    while (optind < argc) {
	filename = argv[optind++];
	linenum = 1;
	pos = open_file_maybe(filename, "tex", texinputs);
	if (pos[0]) {
	    /* auxiliary input file prefix is input filename sans .tex: */
	    strcpy(aux_prefix, filename);
	    dot = strrchr(aux_prefix, '.');
	    if (dot && strcmp(dot, ".tex") == 0) {
		*dot = '\0';
	    }

	    /*
	     * Path to output directory is the last component of aux_prefix:
	     */
	    slash = strrchr(aux_prefix, '/');
	    if (slash) {
		dir_prefix = slash + 1;
	    } else {
		dir_prefix = aux_prefix;
	    }
	    if (mkdir(dir_prefix, 0777) < 0) {
		if (errno != EEXIST) {
		    perror(dir_prefix);
		    continue;
		}
		if (verbose) {
		    fprintf(stderr, _("%s: reusing existing directory `%s'\n"),
			    prog_name, dir_prefix);
		}
	    }

	    /* open file for table of contents: */
	    sprintf(buf, FILENAME_FMT, counter[CNT_FILE]);
	    f[OUT_CONTENTS] = open_output(buf);
	    if (!f[OUT_CONTENTS]) {
		continue;
	    }
	    fputs("<title>Contents</title>\n<h1>Contents</h1><p>\n\n<table>\n",
		  f[OUT_CONTENTS]);

	    /* open main file (for abstract and stuff): */
	    sprintf(buf, FILENAME_FMT, ++counter[CNT_FILE]);
	    fp = open_output(buf);
	    if (!fp) {
		continue;
	    }
	    push_file(fp);
	    fprintf(out->file, "<title>%s</title>\n", aux_prefix);

	    eval(pos);

	    end_of_toc();

	    if (aux_flags & AUX_INDEX) {
		gen_index();
	    }

	    while ((fp = pop_file())) {
		fclose(fp);
	    }
	    for (i = 0; i < OUT_MAX; ++i) {
		if (f[i]) {
		    fclose(f[i]);
		}
	    }
	    finalize();

	    /*
	     * Make a symlink from doc000.html to index.html:
	     */
	    sprintf(buf, "%s/index.html", dir_prefix);
	    symlink("doc001.html", buf);
	}
    }
    return 0;
}
