/* tscam.c	- tjh's emulation of "scam"
 *
 * This is a totally independant implementation of something that 
 * emulates "SCAM - Smart Card Analyser and Manipulator".
 *
 * I wrote it so I could use the perl instruction and scanning logic
 * that comes with SCAM in a portable code base. Since then things have
 * moved on so that basically this has evolved into a general purpose
 * infrastructure for working with smartcards.
 *
 * Copyright 1993-1997, Tim Hudson. All rights reserved.
 *
 * You can pretty much do what you like with this code except pretend that 
 * you wrote it provided that any derivative of this code includes the
 * above comments unchanged. If you put this in a product then attribution
 * is mandatory. See the details in the COPYING file.
 *
 * 30-Aug-97 tjh	reworked into this form
 *
 * Tim Hudson
 * tjh@cryptsoft.com
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "platform.h"

#ifdef WINDOWS
#include <windows.h>
#endif

#include "sio.h"
#include "sct0.h"
#include "sc.h"

#define MAX_SCAN_RETRIES 8

static char hexmap[]="0123456789ABCDEF";

static int to_hex(unsigned char c)
{
  return (hexmap[c & 0x0f]);
}

static int from_hex(unsigned char c)
{
  if ((c>='0')&&(c<='9'))
    return(c-'0');
  else if ((c>='A')&&(c<='F'))
    return((c-'A')+10);
  else if ((c>='a')&&(c<='f'))
    return((c-'a')+10);
  else
    return(0);
}

int write_ascii(SIO_INFO *s,unsigned char c)
{
  unsigned char o;
  int ret;

  /* write out the data byte - top first */
  o=to_hex((unsigned char)(c>>4));
  ret=SIO_WriteChar(s,o & 0xff);
  if (ret!=1)
    return(ret);

  /* write out the data byte - now the bottom */
  o=to_hex((unsigned char)(c & 0x0f));
  SIO_WriteChar(s,o & 0xff);
  if (ret!=1)
    return(ret);

  return(1);
}

static int read_non_space(FILE *inf)
{
  int ch;

  do {
    ch=fgetc(inf);
  } while (ch==' ');

  if (feof(inf))
    return(-1);
  else
    return ch;
}

static void talk(SIO_INFO *s,FILE *inf)
{
  unsigned char o;
  char c;
  int state=0;	/* 0=data, 1=waiting on second hex char, 2=done */

  do {
    /* get data from the command stream */
    c=fgetc(inf);
    if (feof(inf))
      break;

    switch(state) {
      case 0: 
	/* getting first character of two-byte hex data */
        if ((c=='.')||(c=='\n')) 
	  state=2;
        else {
	  if (islower(c))
	    c=toupper(c);
          if (isxdigit(c)) {
            if (isdigit(c)) 
	      o=c-'0';
            else 
              o=c-'A'+0xA;
            state=1;
          }
	}
        break;
      case 1: 
	/* getting second character of two-byte hex data */
        if (isxdigit(c)) { 
	  if (islower(c))
	    c=toupper(c);
	  o*=16;
	  if (isdigit(c)) 
	    o+=c-'0';
	  else 
	    o+=c-'A'+0xA;
	  state=1;
        } else {
          if ((c=='.')||(c=='\n')) 
	    state=2; 
	  else 
	    state=0; 
          /* write out the data byte */
	  SIO_WriteChar(s,o & 0xff);
        }
        break;
      default:
        break;
    }
  } while (state!=2);
  return;
}

static void gcr_talk(SIO_INFO *s,FILE *inf)
{
  char buf[256];
  char buf2[256];
  unsigned char o;
  char c;
  int i,l,num;
  unsigned long r;
  char *p;

  /* get until . or newline */
  buf[0]='\0';
  l=0;
  do {
    c=fgetc(inf);
    if (feof(inf))
      break;
    if ((c=='.') || (c=='\n'))
      break;
    buf[l++]=c;
  } while(l<sizeof(buf));
  /* be nice and terminate the string */
  buf[l]='\0';

  /* now turn it into a valid GCR command string */
  num=r=0;

  p=strtok(buf," \t\n");
  while (p!=NULL) {
    l=strtoul(p,NULL,16);
    r^=l;

    c=(unsigned char)(l & 0xff);
    /* gather the data for writing */
    buf2[num++]=c;

    p=strtok(NULL," \t\n");
  }

  /* write out the header ... 0x60 + length */
  write_ascii(s,0x60);
  write_ascii(s,(unsigned char)(num & 0xff));

  /* add in the header to the checksum too */
  r^=0x60;
  r^=(num & 0xff);

  /* write out the data */
  for(i=0;i<num;i++)
    write_ascii(s,buf2[i]);

  /* write out the checksum byte */
  c=(unsigned char)(r & 0xff);
  write_ascii(s,c);
 
  /* EXT - as a raw byte */
  o=0x03;
  SIO_WriteChar(s,o & 0xff);
}

int 
do_scam(SIO_INFO *s,FILE *inf,FILE *outf)
{
  int i,j,c;
  int ret,num,len;
  char buf[1024];
  unsigned char ch;
  char scanres[256];
  char scanins[256];
  int thefile;

  /* switch off output buffering ... otherwise talking on a pipe
   * through to the perl stuff won't work very well 
   */
  fflush(outf);
  setbuf(outf,NULL);

#if 0
  /* the philips DX card needs 50000 as the setting */
  SIO_SetReadTimeout(s,50000);
#endif

  /* flush any data in the serial buffer now ... as the card
   * may have responded to the serial port initialisation with
   * an ATR 
   */
  while ((c=SIO_ReadChar(s))!=-1)
    ;

  /* start with a nop */
  c='\n';

  /* process commands forever */
  do {
    /* between each command we flush the logs so we always know
     * where things are at 
     */
    SIO_FlushLog(s);

    switch(c) {
      case '?':
      case 'h':
        /* help ... a command not in the standard scam logic */
	fprintf(outf,"SCAM commands:\n");
	fprintf(outf,"F - flush, f - fetch data\n");
	fprintf(outf,"r - reset (return ATR)\n");
	fprintf(outf,"t - talk to card and get response (. terminates data)\n");
	fprintf(outf,"c - change iomode (d=direct,i=indirect)\n");
	fprintf(outf,"p - change parity (o=odd,e=even,n=none)\n");
	fprintf(outf,"d - change databits, s - change stopbits\n");
	fprintf(outf,"b - change speed (baud)\n");
	fprintf(outf,"i - io info (current settings)\n");
	fprintf(outf,"E - filter echo (0=off,1=on)\n");
	break;
      case 'F': 
	/* flush data ... read until nothing available */
      	while (SIO_ReadChar(s)!=-1)
	  ;
	break;
      case 'f': 
do_fetch: ;
	/* fetch data ... read until nothing available and do this
	 * twice as some cards "pause" a little during message output
	 */
	for(i=0;i<2;i++) {
	  while ((c=SIO_ReadChar(s))!=-1)
	    fprintf(outf,"%02x ",c);
	}
	fprintf(outf,"\n");
	fflush(outf);
	break;
      case 'r': 
        /* reset card */
        SCT0_Reset(s);
	goto do_fetch;
      case 't': 
        /* talk to card - send data */
        talk(s,inf);

	/* we need to delay before reading to give the reader time
	 * to start responding
	 */
	SIO_Delay(s,100);

	goto do_fetch;

      case 'T': 
        /* talk to Gemplus reader - send data */
        gcr_talk(s,inf);

	/* we need to delay before reading to give the reader time
	 * to start responding
	 */
	SIO_Delay(s,100);

	/* fetch data ... read until nothing available */
        num=0;
	while ((c=SIO_ReadChar(s))!=-1) {
	  fprintf(outf,"%02x ",c);
	  buf[num++]=c;
	}
	buf[num]='\0';
	fprintf(outf,"\n");
	fflush(outf);

	/* now convert from GCR ASCII into bytes */
	fprintf(outf,"DATA: ");
	for(i=0;i<num;i+=2) {
	  /* if we are on the last byte we are done ... 
	   * and the last byte should always be 0x03
	   */
	  if (i==(num-1))
	    break;
	  ch=(from_hex(buf[i])<<4)+from_hex(buf[i+1]);
	  fprintf(outf,"%02x ",ch);
	}
	fprintf(outf,"\n");
	fflush(outf);

        break;

      case 'c': 
        /* change iomode */
	c=read_non_space(inf);
	switch(c) {
	  case 'i': SIO_SetIOMode(s,SIO_IOMODE_INDIRECT); break;
	  case 'd': SIO_SetIOMode(s,SIO_IOMODE_DIRECT); break;
	}
	SIO_WriteSettings(s);
	break;

      case 'p': 
        /* change parity */
	c=read_non_space(inf);
	switch(c) {
	  case 'o': SIO_SetParity(s,SIO_PARITY_ODD); break;
	  case 'e': SIO_SetParity(s,SIO_PARITY_EVEN); break;
	  case 'n': SIO_SetParity(s,SIO_PARITY_NONE); break;
	}
	SIO_WriteSettings(s);
	break;

      case 'd':
        /* change databits */
	fgets(buf,sizeof(buf),inf);
	SIO_SetDataBits(s,atoi(buf+1));
	SIO_WriteSettings(s);
        break;

      case 's':
        /* change stopbits */
	fgets(buf,sizeof(buf),inf);
	SIO_SetStopBits(s,atoi(buf+1));
	SIO_WriteSettings(s);
        break;

      case 'b':
        /* change speed (baud) */
	fgets(buf,sizeof(buf),inf);
	SIO_SetSpeed(s,atoi(buf+1));
	SIO_WriteSettings(s);
        break;

      case 'i':
display_settings: ;
	{
	char *paritystr,*iomodestr;

        /* io information - get settings */
        paritystr=SIO_Parity2String(SIO_GetParity(s));
	if (paritystr==NULL)
	  paritystr="";
    	iomodestr=SIO_IOMode2String(SIO_GetIOMode(s));
	if (iomodestr==NULL)
	  iomodestr="";
	/* the IO mode is an addition (wasn't in SCAM) */
	sprintf(buf,"%ld %d%c%d %c",SIO_GetSpeed(s),SIO_GetDataBits(s),
			paritystr[0],SIO_GetStopBits(s),iomodestr[0]);
	fprintf(outf,"%s\n",buf);
	break;
	}

      case 'D':
        /* change device - ignored */
	fgets(buf,sizeof(buf),inf);
	break;

      case 'P': 
        /* not implemented - I don't know what this was meant to
	 * do so it doesn't return an error and doesn't do anything
	 */
        break;

      case 'E': 
        /* filter echo - TJH addition */
	fgets(buf,sizeof(buf),inf);
	ret=SIO_FilterEcho(s,atoi(buf));
	fprintf(outf,"%d\n",ret);
	break;

      case 'Z': 
	{
	char *tst="0001FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0001020304";
	char *tst2="ec37896b01aed8fbc7c3da0bc4b0ddaa132530391e899d090fba2cdb9eb2a4b0885ad75900ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0100";
	char tmp2[5];
	int len,tch;
	int rev=1;  /* need to reverse the data */
	int snooze=5*100000;

        len=strlen(tst)/2;
        /* send zeros to the card */
	fgets(buf,sizeof(buf),inf);
	i=atoi(buf);
	if (i==0)
	  i=len;
	ret=SIO_WriteChar(s,0xc0);
	ret=SIO_WriteChar(s,0x88);
	ret=SIO_WriteChar(s,0x00);
	ret=SIO_WriteChar(s,0x00);
	ret=SIO_WriteChar(s,0x80);
	SIO_Delay(s,snooze);

#if 0
	if ((tch=SIO_ReadChar(s))!=-1) {
	  fprintf(outf,"(88?) %02x ",tch & 0xff);
	  fflush(outf);
	}
	SIO_Delay(s,snooze);
#endif

	if (i>0) {
	  while (i>0) {

#if 0
	    if ((tch=SIO_ReadChar(s))!=-1) {
	      fprintf(outf,(tch==0x77)?"#":"!"); 
	      fflush(outf);
	    } else {
	      fprintf(outf,"%%"); 
	      fflush(outf);
	    }
#endif
#if 1
	    /* we must get a return from this! */
	    for(j=0;j<50;j++) {
		while ((tch=SIO_ReadChar(s))!=-1) {
		  fprintf(outf,(tch==0x77)?"#":"!"); 
		  fflush(outf);
		  goto wait_over;
		}
		fprintf(outf,"W");
		fflush(outf);
	    }
wait_over:;
#endif

	    if (rev) {
	      /* inverted */
	      memcpy(tmp2,tst+((i-1)*2),2);
	    } else {
	      /* "normal" */
	      memcpy(tmp2,tst+((len-i)*2),2);
	    }
	    tmp2[2]='\0';
	    SC_HexToBuffer(tmp2,&ch,1);
	    fprintf(outf,"%02x ",ch & 0xff);
	    
	    /*
	    ret=SIO_WriteChar(s,0x77);
	    */

	    ret=SIO_WriteChar(s,ch & 0xff);

	    if (ret==-1)
	      break;
	    SIO_Delay(s,snooze);

	    i--;
	    fprintf(outf,"."); 
	    fflush(outf);
	  }
	}
	}
	break;

      case '/': 
	/* get the rest of the command */
        fgets(buf,sizeof(buf)-1,inf);
	len=strlen(buf);
	if (len>0)
	  buf[len-1]=0;

	{
	  static SC_STATE *sc;
	  static SLOG *slog, *sout;
	  static char rspbuf[SC_MAX_HEX_DATA_LEN];
	  static int reader_type=SC_READER_DUMBMOUSE;
	  static int card_type=SC_CARD_CRYPTOFLEX;
	  char *txt;
	  char *cmd;
	  int vars,k;
	  char *tst="0001FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0001020304";

	  if (strncmp(buf,"help",strlen("help"))==0) {
	    fprintf(outf,"Valid cmds: help,init,acmd,cmd,final,reset\n");
	  } else if (strncmp(buf,"init",strlen("init"))==0) {
	    char *p;

	    if (sc!=NULL) {
	      SC_Final(sc);
	      sc=NULL;
	    }

	    /* look for reader or card types as optional args */
	    p=strtok(buf+strlen("init")," \t\n");
	    while (p!=NULL) {
	      j=SC_String2Reader(p);
	      if (j!=SC_READER_UNKNOWN)
		reader_type=j;
	      j=SC_String2Card(p);
	      if (j!=SC_CARD_UNKNOWN)
		card_type=j;
	      p=strtok(NULL," \t\n");
	    }

	    sc=SC_Init(s,reader_type,card_type,1,slog);
	    if (sc==NULL) {
	      fprintf(outf,"SC_Init failed\n");
	    } else {
	      /* display relevant success information */
	      txt=SC_ATR2Text(sc->atr,0);
	      if (txt!=NULL) {
		fprintf(outf,"READER %s CARD %s\n",
		                    SC_Reader2String(reader_type),
		                    SC_Card2String(card_type));
		fprintf(outf,"ATR %s\n",txt);
		free(txt);
	      }
	    }

	    /* also open an output channel that isn't related
	     * to debug tracing of things so that internal cmds
	     * can set output to the user
	     */
	    if (sout==NULL)
	      sout=SLOG_open("-",0);
	    if ((sout!=NULL) && (sc!=NULL))
	      SC_SetOutput(sc,sout);

	  } else if (strncmp(buf,"log",strlen("log"))==0) {
	    /* we handle the logging of the higher level stuff
	     * entirely separate from the low-level debug tracing
	     */
	    if (slog==NULL)
	      slog=SLOG_open("-",0);
	    if ((slog!=NULL) && (sc!=NULL))
	      SC_SetLog(sc,slog);
	  } else if (strncmp(buf,"nolog",strlen("nolog"))==0) {
	    /* close it */
	    slog=NULL;
	    if (sc!=NULL)
	      SC_SetLog(sc,slog);
	  } else if (strncmp(buf,"test",strlen("test"))==0) {
	    if (sc!=NULL)
	      SC_SetTestMode(sc,1);
	  } else if (strncmp(buf,"notest",strlen("notest"))==0) {
	    if (sc!=NULL)
	      SC_SetTestMode(sc,0);
	  } else if (strncmp(buf,"parse",strlen("parse"))==0) {
	    if (sc!=NULL)
	      SC_SetParseMode(sc,1);
	  } else if (strncmp(buf,"noparse",strlen("noparse"))==0) {
	    if (sc!=NULL)
	      SC_SetParseMode(sc,0);
	  } else if (strncmp(buf,"signtest",strlen("signtest"))==0) {
	    char tmp2[5];
	    int len;

	    sprintf(buf,"SignRSA 00 ");

            len=strlen(tst)/2;
	    for(k=len;k>0;k--) {
	    if (1) {
	      /* inverted */
	      memcpy(tmp2,tst+((k-1)*2),2);
	    } else {
	      /* "normal" */
	      memcpy(tmp2,tst+((len-k)*2),2);
	    }
	    tmp2[2]='\0';

	    /* 01020304 PKCS#1 padded for 1024 bit key (128 raw bytes) */
	    /*
	    for(k=strlen(tst);k>0;k--) 
	      sprintf(buf+strlen(buf),"%.2s",tst+k-2);
	      */
	    strcat(buf,tmp2);

	    }
	    cmd=buf;
	    fprintf(outf,"CMD is %s\n",buf);
	    fflush(outf);
	    goto do_exe;
	  } else if (strncmp(buf,"in ",strlen("in "))==0) {
	    /* we have to do a command ... in ascii format */
            if (sc!=NULL) {
	      if (SC_DoAsciiCommand(sc,SC_DIR_IN,buf+strlen("in")+1,
                                                      rspbuf)!=-1) {
		fprintf(outf,"RSP %s\n",rspbuf);
	      } else {
		fprintf(outf,"in failed - error %d - %s\n",
		      sc->error_code,SC_ErrorCode2String(sc->error_code));
	      }
	    }
	  } else if (strcmp(buf,"vars")==0) {
	    vars=SC_GetVarCount(sc);
	    if (vars>0) {
	      for(i=0;i<vars;i++) {
		fprintf(outf,"VAR %s=%s\n",SC_GetVarName(sc,i),
					    SC_GetVarValue(sc,i));
	      }
	    }
	  } else if (strncmp(buf,"out ",strlen("out "))==0) {
	    /* we have to do a command ... in ascii format */
            if (sc!=NULL) {
	      if (SC_DoAsciiCommand(sc,SC_DIR_OUT,buf+strlen("out")+1,
                                                      rspbuf)!=-1) {
		fprintf(outf,"RSP %s\n",rspbuf);
	      } else {
		fprintf(outf,"out failed - error %d - %s\n",
		      sc->error_code,SC_ErrorCode2String(sc->error_code));
	      }
	    }
	  } else if (strncmp(buf,"exe ",strlen("exe "))==0) {
	    cmd=buf+strlen("exe")+1;
do_exe: ;
	    /* we have to do a command ... in ascii format */
            if (sc!=NULL) {
	      if (SC_ExecuteCommand(sc,cmd,rspbuf,1)!=-1) {
		fprintf(outf,"RSP CODE %s STR %s\n",
		        SC_RSPCode2String(sc->last_rsp.rsp_code),rspbuf);
		
		/* display all vars if not an internal command */
		if (cmd[0]!='!') {
		  vars=SC_GetVarCount(sc);
		  if (vars>0) {
		    for(i=0;i<vars;i++) {
		      fprintf(outf,"VAR %s=%s\n",SC_GetVarName(sc,i),
						  SC_GetVarValue(sc,i));
		    }
		  }
		}
	      } else {
		fprintf(outf,"exe failed - error %d - %s\n",
		      sc->error_code,SC_ErrorCode2String(sc->error_code));
	      }
	    }
	  } else if (strncmp(buf,"final",strlen("final"))==0) {
	    if (sc!=NULL) {
	      SC_Final(sc);
	      sc=NULL;
	    }
	  } else {
	    /* assume it is a command to execute and the user has
	     * just ommited the /exe prefix 
	     */
	    cmd=buf;
	    goto do_exe;
	    /*
	    fprintf(outf,"Unknown command \"%s\"\n",buf);
	    */
	  }
	}
	break;

      case 'S':
	{ 
	  int cla,len;
	  char *p;
	  int retry_count;

          retry_count++;    
	  buf[0]='S';
	  fgets(buf+1,sizeof(buf)-1,inf);
	  if (strncmp(buf,"Scan ATR",strlen("Scan ATR"))==0) {
	    len=SCT0_ScanForATR(s,buf,sizeof(buf));
	    if (len>0) {
	      fprintf(outf,"ATR-DATA: ");
	      SIO_DumpBuffer(outf,buf,len,0);
	      fprintf(outf,"\n");
	      fprintf(outf,"ATR-TEXT: ");
	      SIO_DumpBuffer(outf,buf,len,1);
	      fprintf(outf,"\n");
	      fprintf(outf,"SETTINGS: ");
	      goto display_settings;
	    }
	  } else if (strncmp(buf,"Scan a4",strlen("Scan a4"))==0) {
	    char cmd[16];
	    int cmdlen;

	    cmd[0]=0xa4;
	    cmd[1]=0xa4;
	    cmd[2]=0x00;
	    cmd[3]=0x00;
	    cmd[4]=0x02;
	    /*
	    cmd[5]=0x3f;
	    cmd[6]=0x00;
	    cmdlen=7;
	    */
	    cmdlen=5;

	    fprintf(outf,"SCAN START\n");
	    SCT0_ScanForResponse(s,cmd,cmdlen);
	    fprintf(outf,"SCAN FINISH\n");

	  } else if (strncmp(buf,"Scan Cla",strlen("Scan Cla"))==0) {
	    fprintf(outf,"SCAN START\n");
	    if (SCT0_ScanClasses(s,scanres)>0) {
	      for(i=0;i<256;i++) {
		if (scanres[i]==1)
		  fprintf(outf,"%02x ",i);
	      }
	      fprintf(outf,"\n");
	    } else {
	      fprintf(outf,"SCAN failed\n");
	    }
	    fprintf(outf,"SCAN FINISH\n");
	  } else if (strncmp(buf,"Scan Cmd",strlen("Scan Cmd"))==0) {
	    fprintf(outf,"SCAN START\n");
	    /* scan classes and then instructions in each class */
	    if (SCT0_ScanClasses(s,scanres)>0) {
	      for(i=0;i<256;i++) {
		if (scanres[i]==1) {
		  cla=i;
		  if (SCT0_ScanInstructions(s,(unsigned char)(cla & 0xff),scanins)>0) {
		    /* now dump the results */
		    fprintf(outf,"CLA %02x: ",cla);
		    for(j=0;j<256;j++) {
		      if (scanins[j]==1)
			fprintf(outf,"%02x ",j);
		    }
		    fprintf(outf,"\n");
		  }
		}
	      }
	    }
	    fprintf(outf,"SCAN FINISH\n");
	  } else if (strncmp(buf,"Scan Ins",strlen("Scan Ins"))==0) {
	    fprintf(outf,"SCAN START\n");

	    cla=0x00;
	    /* grab the class we are scanning */
	    p=strtok(buf," \t\n");  /* first word = Scan */
	    if (p!=NULL) {
	      p=strtok(NULL," \t\n");  /* second word = Ins */
	      if (p!=NULL) { 
		p=strtok(NULL," \t\n"); /* third work = the class */
		if (p==NULL) {
		  fprintf(outf,"You must include a class to scan - e.g. Scan Ins c0\n");
		  goto no_ins_class;
		} else {
		  cla=strtoul(p,NULL,16);
		}
	      }
	    }

	    if (SCT0_ScanInstructions(s,(unsigned char)(cla & 0xff),scanres)>0) {
	      for(i=0;i<256;i++) {
		if (scanres[i]==1)
		  fprintf(outf,"%02x ",i);
	      }
	      fprintf(outf,"\n");
	    } else {
	      fprintf(outf,"SCAN failed\n");
	    }
no_ins_class: ;
	    fprintf(outf,"SCAN FINISH\n");
	  } else if (strncmp(buf,"Scan File",strlen("Scan File"))==0) {
	    fprintf(outf,"SCAN START\n");

	    cla=0x00;
	    thefile=0x00;
	    /* grab the class we are scanning */
	    p=strtok(buf," \t\n");  /* first word = Scan */
	    if (p!=NULL) {
	      p=strtok(NULL," \t\n");  /* second word = File */
	      if (p!=NULL) { 
		p=strtok(NULL," \t\n"); /* third work = the class */
		if (p==NULL) {
		  fprintf(outf,"You must include a class to scan - e.g. Scan File c0 00\n");
		  goto no_file_class;
		} else {
		  cla=strtoul(p,NULL,16);
		  p=strtok(NULL," \t\n"); /* forth work = the first byte of fileid */
		  if (p==NULL) {
		    fprintf(outf,"You must include the first byte of the file group to scan - e.g. Scan File c0 00\n");
		    goto no_file_class;
		  } else {
		    thefile=strtoul(p,NULL,16);
		  }
		}
	      }
	    }

do_scan_file_again: ;

	    if (SCT0_ScanFiles(s,(unsigned char)(cla & 0xff),
				 (unsigned char)(thefile & 0xff), scanres)>0) {
	      for(i=0;i<256;i++) {
		if (scanres[i]==1)
		  fprintf(outf,"%02x ",i);
	      }
	      fprintf(outf,"\n");
	    } else {
	      fprintf(outf,"SCAN failed\n");
	      if (retry_count++<MAX_SCAN_RETRIES) {
		/* reset card */
		SCT0_Reset(s);

		/* we need to delay before reading to give the reader time
		 * to start responding
		 */
		SIO_Delay(s,100);

		/* swallow ATR */
		while ((c=SIO_ReadChar(s))!=-1)
		    ;
		
		/* try again */
		goto do_scan_file_again;
	      }
	    }
no_file_class: ;
	    fprintf(outf,"SCAN FINISH\n");
	  } else if (strncmp(buf,"Scan FS",strlen("Scan FS"))==0) {
	    fprintf(outf,"SCAN START\n");

	    cla=0x00;
	    /* grab the class we are scanning */
	    p=strtok(buf," \t\n");  /* first word = Scan */
	    if (p!=NULL) {
	      p=strtok(NULL," \t\n");  /* second word = File */
	      if (p!=NULL) { 
		p=strtok(NULL," \t\n"); /* third work = the class */
		if (p==NULL) {
		  fprintf(outf,"You must include a class to scan - e.g. Scan File c0\n");
		  goto no_fs_class;
		} else {
		  cla=strtoul(p,NULL,16);
		}
	      }
	    }

	    for(i=0;i<=255;i++) {
do_scan_fs_again: ;
	      if (SCT0_ScanFiles(s,(unsigned char)(cla & 0xff),
				   (unsigned char)(i & 0xff),scanres)>0) {
		fprintf(outf,"FILE %02x: ",i & 0xff);
		for(j=0;j<256;j++) {
		  if (scanres[j]==1)
		    fprintf(outf,"%02x ",j);
		}
		fprintf(outf,"\n");
	      } else {
		fprintf(outf,"FILE %02x: SCAN failed\n",i & 0xff);

		if (retry_count++<MAX_SCAN_RETRIES) {
		  /* reset card */
		  SCT0_Reset(s);

		  /* we need to delay before reading to give the reader time
		   * to start responding
		   */
		  SIO_Delay(s,100);

		  /* swallow ATR */
		  while ((c=SIO_ReadChar(s))!=-1)
		      ;

		  /* try again */
		  goto do_scan_fs_again;
		}
	      }
	    }
no_fs_class: ;
	    fprintf(outf,"SCAN FINISH\n");
	  } else {
	    fprintf(outf,"Scan ATR|Class|Instruction|Cmd|File|FS\n");
	    fprintf(outf,"Note: FS takes about one hour at 9600 baud\n");
	  }
	}
	break;

      case 'q':
        /* quit - TJH addition - cross platform! */
	goto return_time;

      case '#':
        /* comment - ignored - which makes feeding command files 
	 * to tscam nicer as they can have comments in them so 
	 * that make sense once you have forgotten what you did
	 */
	fgets(buf,sizeof(buf),inf);
	/* echo the comment so we can have annotated command traces */
	fprintf(outf,"%s",buf);
	break;

      case 'A': 
        if ((len=SCT0_GetATR(s,buf,sizeof(buf)))>0) {
	  SC_DumpATR(outf,buf,len,0);
	} else {
	  fprintf(outf,"FAILED GetATR\n");
	}
	break;

      case ' ':
      case '\n':
	/* newline and space (trailing) is okay to ignore */
        break;

      default: 
        fprintf(outf,"unknown command\n");
        break;
    }
    c=fgetc(inf);
  } while (!feof(inf));

return_time: ;

  return(1);
}

