/* scatr.c	- Smartcard ATR related functions
 *
 * Copyright 1993-1997, Tim Hudson. All rights reserved.
 *
 * You can pretty much do what you like with this code except pretend that 
 * you wrote it provided that any derivative of this code includes the
 * above comments unchanged. If you put this in a product then attribution
 * is mandatory. See the details in the COPYING file.
 *
 * Tim Hudson
 * tjh@cryptsoft.com
 *
 */

#include "platform.h"

#ifdef USE_STDIO
#include <stdio.h>
#endif /* USE_STDIO */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "sc.h"

SC_ATR *SC_DecodeATR(char *buf,int len)
{
  SC_ATR atr, *atrp;
  int left,ofs;
  int flags,i;

  /* sanity check parameters */
  if ((buf==NULL)||(len<3))
    goto bad;

  /* start with a blank entry */
  memset(&atr,'\0',sizeof(atr));

  /* more than 33 bytes is not valid according to the spec ... so 
   * we truncate what we look at in the name of sanity preservation
   */
  if (len>33)
    atr.data_count=33;
  else
    atr.data_count=len;

  /* copy the raw data ... always handy to have for
   * simple direct ATR matching (brain-dead but safe logic)
   */
  memcpy(atr.data,buf,atr.data_count);

  ofs=0;
  left=atr.data_count;

  /* first byte is TS */
  atr.TS=atr.data[ofs++];
  left--; 

  /* check the first byte - it can have only two values */
  if (atr.TS==0x3b) {
    atr.direct_convention=1;
  } else if (atr.TS==0x3f) {
    atr.inverse_convention=1;
  } else {
    /* invalid - not a lot more we can do at this point */
    goto bad;
  }

  if (left>0) {
    /* next byte is always T0 */
    atr.T0=atr.data[ofs++];
    left--;

    /* extract flags for T[A-D] and historical count */
    flags=(atr.T0 & 0xf0) >> 4;
    atr.historical_count=(atr.T0 & 0x0f);

    /* now read in the interleaved additional bytes */
    while ((left>0) && (flags!=0)) {
      /* grab the A byte */
      if (flags & 0x01) {
	atr.TA[1+atr.TA_count++]=atr.data[ofs++];
	left--;
	if (left<=0)
	  break;
      }
      if (flags & 0x02) {
	atr.TB[1+atr.TB_count++]=atr.data[ofs++];
	left--;
	if (left<=0)
	  break;
      }
      if (flags & 0x04) {
	atr.TC[1+atr.TC_count++]=atr.data[ofs++];
	left--;
	if (left<=0)
	  break;
      }
      /* we have another TD to start the next block */
      if (flags & 0x08) {
	flags=atr.data[ofs++];
	atr.transmission_protocol=(flags & 0x0f);
	flags=(flags & 0xf0)>>4;
	left--;
	if (left<=0)
	  break;
      } else {
	break;
      }
    }

    /* at this point we should have any historical chars
     * and TCK left to read ... check that we have enough
     * data to cover this and abort if not
     */
    if (left<atr.historical_count) 
      goto bad;

    /* copy historical */
    memcpy(atr.historical,atr.data+ofs,atr.historical_count);
    ofs+=atr.historical_count;
    left-=atr.historical_count;

    /* check to see if we have 9x 00 and in which case it
     * probably isn't the TCK but is the response to the
     * implied select - there doesn't appear to be a nice
     * way of determining if TCK is transmitted or not
     * 
     * for the moment at least this data is considered
     * as part of the ATR ... it is transmitted always on
     * reset but isn't strictly/logically in the ATR
     * but such things we simply have to live with
     */
    if (left>=2) {
      if ( ((atr.data[ofs] & 0x90)==0x90) &&
	    (atr.data[ofs+1]==0x00) ) {
	ofs+=2;
	left-=2;
      }
    }

    /* anything remaining should be the TCK */
    if (left>0) {
      /* now we should just have TCK left */
      atr.TCK=1;
      atr.TCK_sent=atr.data[ofs++];

      /* now we calculate what the TCK should be so it can be checked
       * by user logic if desired
       */
      atr.TCK_calc=0;
      for(i=1;i<(ofs-1);i++)
	atr.TCK_calc ^= (atr.data[i] & 0xff);
      left--;
    }

    /* extra data should be "deleted" from the ATR */
    if (left>0) {
      atr.data_count=-left;
    }
  }

  /* copy the local buffer */
  atrp=(SC_ATR *)malloc(sizeof(SC_ATR));
  if (atrp==NULL)
    goto bad;
  memcpy(atrp,&atr,sizeof(SC_ATR));
  return (atrp);
bad: ;
  return (NULL);
}

char *SC_ATR2Text(SC_ATR *atr,int flag)
{
  char buf[1024];
  char tmpbuf[128];
  int i;

  buf[0]='\0';

  if (atr!=NULL) {
    if (atr->direct_convention)
      strcat(buf,"DIRECT ");
    if (atr->inverse_convention)
      strcat(buf,"INVERSE ");
    sprintf(buf+strlen(buf),"PROTOCOL=%d ",atr->transmission_protocol);
    tmpbuf[0]='\0';
    for(i=1;i<=atr->TA_count;i++)
      sprintf(tmpbuf+strlen(tmpbuf),"TA%d=%02x ",i,atr->TA[i]);
    strcat(buf,tmpbuf);
    tmpbuf[0]='\0';
    for(i=1;i<=atr->TB_count;i++)
      sprintf(tmpbuf+strlen(tmpbuf),"TB%d=%02x ",i,atr->TB[i]);
    strcat(buf,tmpbuf);
    tmpbuf[0]='\0';
    for(i=1;i<=atr->TC_count;i++)
      sprintf(tmpbuf+strlen(tmpbuf),"TC%d=%02x ",i,atr->TC[i]);
    strcat(buf,tmpbuf);
    tmpbuf[0]='\0';

    if (atr->historical_count>0) {
      strcat(buf,"HISTORICAL ");
      for(i=0;i<atr->historical_count;i++)
	sprintf(tmpbuf+strlen(tmpbuf),"%02x ",atr->historical[i] & 0xff);
      strcat(buf,tmpbuf);
      tmpbuf[0]='\0';
    }

    if (atr->TCK)
      sprintf(buf+strlen(buf),"TCK=%02x (%02x)",atr->TCK_sent,atr->TCK_calc);

  }
  return strdup(buf);
}

#ifdef USE_STDIO
/* SC_DumpATR - handy routine for output all information about an
 *                ATR that might be of interest. Intended for debug.
 */
int SC_DumpATR(FILE *outf, char *buf,int len, int flag)
{
  SC_ATR *atr;
  char *txt;
  int ret=0;

  atr=NULL;
  fprintf(outf,"ATR DATA: ");
  SIO_DumpBuffer(outf,buf,len,0);
  fprintf(outf,"\n");
  fprintf(outf,"ATR TEXT: ");
  SIO_DumpBuffer(outf,buf,len,1);
  fprintf(outf,"\n");

  /* now use the higher-level ATR interpretation logic */
  atr=SC_DecodeATR(buf,len);
  if (atr!=NULL) {
    ret=1;
    txt=SC_ATR2Text(atr,0);
    if (txt!=NULL) {
      fprintf(outf,"ATR Decoded: %s\n",txt);
    }
  }

  /* be clean */
  if (atr!=NULL)
    free(atr);

  return(ret);
}
#endif /* USE_STDIO */

int SC_LogATR(SLOG *outf, char *buf,int len, int flag)
{
  SC_ATR *atr;
  char *txt;
  int ret=0;

  atr=NULL;
  SLOG_printf(outf,"ATR DATA: ");
  SLOG_dump(outf,buf,len,0);
  SLOG_printf(outf,"\n");
  SLOG_printf(outf,"ATR TEXT: ");
  SLOG_dump(outf,buf,len,1);
  SLOG_printf(outf,"\n");

  /* now use the higher-level ATR interpretation logic */
  atr=SC_DecodeATR(buf,len);
  if (atr!=NULL) {
    ret=1;
    txt=SC_ATR2Text(atr,0);
    if (txt!=NULL) {
      SLOG_printf(outf,"ATR Decoded: %s\n",txt);
      free(txt);
    }
  }

  /* be clean */
  if (atr!=NULL)
    free(atr);

  return(ret);
}


