/* sc_gsm.c	- Smartcard related functions 
 *
 * CARD: GSM SIM
 *
 * Encryption Algorithms:
 *  A3 - authenticate mobile station (MS) to the network
 *  A8 - generate encryption key
 * (can be combined into A38 within the SIM)
 *
 * COMMS: 8o2 i
 *
 * Copyright 1993-1997, Tim Hudson. All rights reserved.
 *
 * You can pretty much do what you like with this code except pretend that 
 * you wrote it provided that any derivative of this code includes the
 * above comments unchanged. If you put this in a product then attribution
 * is mandatory. See the details in the COPYING file.
 *
 * Tim Hudson
 * tjh@cryptsoft.com
 *
 */

/* Notes:
 *
 * Old Telstra MobileNet GSM SIM card:

/!scan
RSP CODE OK STR 2f01 2fe0 2fe1 2fe2 2fe3 2fe4 2fe5 2fff 3f00

-- hmm ... scanning logic didn't find the dirs after 3f00 ... 

/!scan dir=${DF_TELECOM} all 6f 6f
RSP CODE OK STR 6f3a 6f3c 6f3d 6f3e
/!scan dir=${DF_GSM} all 6f 6f
RSP CODE OK STR 6f02 6f07 6f20 6f30 6f38 6f74 6f78 6f7b 6f7e 6fad

Phone directory via:

/sc_sel /${DF_TELECOM}/${EF_ADN}
RECORD_LEN=20
FILE_SIZE=0E60

 */

#include "platform.h"

#ifdef USE_STDIO
#include <stdio.h>
#endif /* USE_STDIO */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "sio.h"
#include "sct0.h"
#include "sc.h"
#include "scint.h"

SC_CMD_PCONST gsm_pconst[]={
{ "EF_ICCID", "2FE2" },		/* ICC Identification */
{ "DF_GSM", "7F20" },		/* GSM directory - all "standard" things under here */
{ "DF_TELECOM", "7F10" },	/* "private" things */

/* underneath DF_GSM are the following */
{ "EF_LP", "6F05" },		/* LP - Language Preferences */
{ "EF_IMSI", "6F07" },		/* IMSI - International Mobile Subscriber Identity */
{ "EF_Kc", "6F20" },		/* Kc - Ciphering Key */
{ "EF_PLMNsel", "6F30" },	/* PLMNsel - Public Lan Mobile Network selector */
{ "EF_HPLMN", "6F31" },		/* HPLMN - search period */
{ "EF_ACMmax", "6F37" },	/* ACM maximum value */
{ "EF_SST", "6F38" },		/* SIM service table */
{ "EF_ACM", "6F39" },		/* ACM - Accumulated Call Meter */
{ "EF_GID1", "6F3E" },		/* GID1 - Group Identifier Level 1 */
{ "EF_GID2", "6F3F" },		/* GID2 - Group Identifier Level 2 */
{ "EF_SPN", "6F46" },		/* SPN - Service Provider Name */
{ "EF_PUCT", "6F41" },		/* PUCT - Price per unit and currency table */
{ "EF_CBMI", "6F45" },		/* CBMI - Cell Broadcast Message Identifier */

/* skip lots of files ... I got bored :-) */
{ "EF_ADN", "6F3A" },		/* ADN - Abbreviated dialling numbers */	
{ "EF_FDN", "6F3B" },		/* FDN - Fixed dialling numbers */
{ "EF_SMS", "6F3C" },		/* SMS - Short messages */


};
#define _N_gsm_pconst (sizeof(gsm_pconst)/ \
                                    sizeof(gsm_pconst[0]))
int N_gsm_pconst=_N_gsm_pconst;

SC_CMD_PDATA gsm_pdata[]={
/* patterns for command output */
{
  /* select has the following fixed data which is then followed by
   * a variable block of GSM data ... first 10 bytes fixed which are
   * included in this definition ... and then
   * 11 bytes of "optional administrative management" stuff 
   */
  "select_df", 
  "${_RFU1:2}${FREE_SPACE:2}${FILE_ID:2}${FILE_TYPE}${_RFU2:5}${LEN}${FILE_CHARACTERISTICS}${NSD}${NSE}${NCHV}${_RFU3}${CHV1_STATUS}${UNBLOCK_CHV1_STATUS}${CHV2_STATUS}${UNBLOCK_CHV2_STATUS}${OPTIONAL:N}"
},
{
  "select_ef",
  "${_RFU1:2}${FILE_SIZE:2}${FILE_ID:2}${FILE_TYPE}${_RFU2}${AC:3}${FILE_STATUS}${LEN}${FILE_STRUCTURE}${RECORD_LEN}${OPTIONAL:N}"
},

  /* the ID should be 10 bytes ... but could be 20 bytes if the card is 
   * an old one ... and I'm not sure if we will see both these days but
   * the code handles either for the moment
   */
{ "EF_ICCID", "${IDENTIFICATION:N}" },
{ "EF_ISMI", "${LEN}${IMSI:8}" },
{ "EF_Kc", "${KEY:8}${SEQNO}" },
{ "EF_PLMNsel", "${PLMN:3}" },	    /* we have an array of these! */
{ "EF_SST", "${SERVICES:5}" },	    /* 20 services 2 bits each */
/*
"CHV1:CHV1 disable", 
"ADN:Abbreviated Dialing Numbers", 
"FDN:Fixed Dialing Numbers",
"SMS:Short Message Service",
"AoC:Advice of Charge",
"CCP:Capability Configuration Parameters",
"PLMNsel:PLNM selector",
"MSISDN",
"Extension1",
"Extension2",
"SMS Parameters",
"LND:Last Number Dialled",
"CBMI:Cell Broadcast Message Identifier",
"Group Identifier Level 1",
"Group Identifier Level 2",
"Service Provider Name",
"Service18",
"Service19",
"Service20"
*/
{ "EF_SPN", "${DISPLAY_CONDITION}${SERVICE_PROVIDER_NAME:16}" },
{ "EF_PUCT", "${CURRENCY_CODE:3}${PRICE_PER_UNIT:2}" },
{ "EF_CBMI", "${ID:2}" },	/* we have an array of these! */

{ "EF_SMS", "${STATUS}${MESSAGE:175}" }, /* we have multiple records in the file */

};
#define _N_gsm_pdata (sizeof(gsm_pdata)/ \
                                    sizeof(gsm_pdata[0]))
int N_gsm_pdata=_N_gsm_pdata;

SC_CMD_ENT gsm_cmds[]={
  { "SelectFile", SC_DIR_IN, 0xa0, 0xa4, 
                        NULL, "00", "00", NULL, "${FileID:2}",
			"9f", "GetResponse", "${SW2}",
			{ 
			  { 15, "select_ef" }, 
			  { 16, "select_ef" }, 
			  { 0, "select_df" }, 
			} 
  },

  { "GetResponse", SC_DIR_OUT, 0xa0, 0xc0, 
                        NULL, "00", "00", "${Len}", NULL,
			NULL, NULL, NULL },

  { "Status", SC_DIR_OUT, 0xa0, 0xf2, 
                        NULL, "00", "00", "${Len}", NULL,
			NULL, NULL, NULL },

  { "ReadBinary", SC_DIR_OUT, 0xa0, 0xb0, 
                        "${Offset}", NULL, NULL, "${Len}", NULL,
			NULL, NULL, NULL },

  { "UpdateBinary", SC_DIR_IN, 0xa0, 0xd6, 
                        "${Offset}", NULL, NULL, "${Len}", "${Data}",
			NULL, NULL, NULL,
  },

  /* READ_RECORD */
  { "ReadRecord", SC_DIR_OUT, 0xa0, 0xb2, 
                        NULL, "${RecordNumber}", "${ReferenceControl}", 
			"${Len}" , NULL,
			NULL, NULL, NULL,
  },

  { "UpdateRecord", SC_DIR_IN, 0xa0, 0xdc, 
                        NULL, "${RecordNumber}", "${ReferenceControl}", 
			"${Len}" , "${Data}",
			NULL, NULL, NULL,
  },

  /* SEEK */
  /* INCREASE */


/* --- */

  { "VerifyPIN", SC_DIR_IN, 0xa0, 0x20, 
                        NULL, "00", "01", NULL, "${PIN:8}",
			NULL, NULL, NULL },

  { "ChangePIN", SC_DIR_IN, 0xa0, 0x24, 
                        NULL, "00", "01", NULL, "${OldPIN:8}${NewPIN:8}",
			NULL, NULL, NULL },

#if 0
  /* AskRandom returns 8 bytes of (semi-)random data */
  { "AskRandom", SC_DIR_OUT, 0x00, 0x84, 
                        NULL, NULL, NULL, "08", NULL,
			NULL, NULL, NULL },


/*
  { "UpdateBinary2", SC_DIR_IN, 0x00, 0xd0, 
                        "${Offset}", NULL, NULL, "${Len}", "${Data}",
			NULL, NULL, NULL,
  },
*/

  /* AppendBinary - called Create Binary in the doco */
  { "AppendBinary", SC_DIR_IN, 0x00, 0xda, 
                        NULL, NULL, NULL, "${Len}", "${Data}",
			NULL, NULL, NULL,
  },

  { "Invalidate", SC_DIR_OUT, 0x00, 0x04, 
                        NULL, "00", "00", "00", NULL,
			NULL, NULL, NULL 
  },
                        

  /* ComputeRSA either generates or verifies a signature depending 
   * on the file currently selected (public or private key)
   */
  { "ComputeRSA", SC_DIR_IN, 0x00, 0xf6, 
                        NULL, "00", "00", "40", "${Data:64}",
			"61", "GetResponse", "${SW2}",
  },
#endif

/* NOW FOR COMMANDS THAT ARE VARIATIONS ON THE ABOVE OR SIMPLY A DIFFERENT
 * NAME THAT MAKES MORE SENSE TO ME
 */
  { "ReadFirstRecord", SC_DIR_OUT, 0xa0, 0xb2, 
                        NULL, "01", "04",   /* abs with recno=1 */
			"${Len}"
  },

  { "ReadPrevRecord", SC_DIR_OUT, 0xa0, 0xb2, 
                        NULL, "00", "03", 
			"${Len}"
  },

  { "ReadNextRecord", SC_DIR_OUT, 0xa0, 0xb2, 
                        NULL, "00", "02", 
			"${Len}"
  },

  { "ReadAbsRecord", SC_DIR_OUT, 0xa0, 0xb2, 
                        NULL, "${RecordNumber}", "04", 
			"${Len}"
  },

#if 0
  /* do having the following make sense? */
  { "UpdateFirstRecord", SC_DIR_IN, 0xa0, 0xdc, 
                        NULL, "01", "04", 
			"${Len}" , "${Data}",
			NULL, NULL, NULL,
  },

  { "UpdatePrevRecord", SC_DIR_IN, 0xa0, 0xdc, 
                        NULL, "00", "03", 
			"${Len}" , "${Data}",
			NULL, NULL, NULL,
  },

  { "UpdateNextRecord", SC_DIR_IN, 0xa0, 0xdc, 
                        NULL, "00", "02", 
			"${Len}" , "${Data}",
			NULL, NULL, NULL,
  },
#endif

  { "UpdateAbsRecord", SC_DIR_IN, 0xa0, 0xdc, 
                        NULL, "${RecordNumber}", "04", 
			"${Len}" , "${Data}",
			NULL, NULL, NULL,
  },

  { "VerifyCHV", SC_DIR_IN, 0xa0, 0x20, 
                        NULL, "00", "01", NULL, "${CHV:8}",
			NULL, NULL, NULL },

#if 0
  { "SignRSA", SC_DIR_IN, 0x00, 0xf6, 
                        NULL, "00", "00", "40", "${Data:64}",
			"61", "GetResponse", "${SW2}",
  },
  /* UNLOCK -> VerifyPIN */
  { "UNLOCK", SC_DIR_IN, 0x00, 0x20, 
                        NULL, "00", "01", NULL, "${Password:8}",
			NULL, NULL, NULL }
#endif

};
#define _N_gsm_cmds (sizeof(gsm_cmds)/sizeof(gsm_cmds[0]))
int N_gsm_cmds=_N_gsm_cmds;

SC_CMD_ERR_CVT gsm_errs[]={
{ 0x90, 0x00, SC_RSP_OK },
{ 0x94, 0x00, SC_RSP_NOTSELECTED },
{ 0x94, 0x02, SC_RSP_BADADDR },
{ 0x94, 0x04, SC_RSP_NOTFOUND },
{ 0x94, 0x08, SC_RSP_WRONGTYPE },
{ 0x98, 0x40, SC_RSP_BLOCKED },
{ 0x9f, -1, SC_RSP_OK },	  /* worked ... more data via GetResponse */
{ 0x67, -1, SC_RSP_BADLEN },
{ 0x6b, -1, SC_RSP_BADP1P2 },
{ 0x6d, -1, SC_RSP_BADCLA },
{ 0x6e, -1, SC_RSP_BADINS }
};
#define _N_gsm_errs (sizeof(gsm_errs)/sizeof(gsm_errs[0]))

int N_gsm_errs=_N_gsm_errs;

static int gsm_phonelist(SC_STATE *sc,void *pinternalv,char *args,char *uout)
{
  char rspbuf[SC_MAX_HEX_DATA_LEN];
  char buf[SC_MAX_HEX_DATA_LEN];
  char name[20];
  char phone[64];
  char phoneascii[64];
  char *p,*data;
  int record_len;
  int file_size;
  int i,j,k,plen,val;

  uout[0]='\0';
  if (SC_ExecuteCommand(sc,"sc_sel /${DF_TELECOM}/${EF_ADN}",rspbuf,1)) {
    record_len=SC_GetIntVar(sc,"RECORD_LEN");
    file_size=SC_GetIntVar(sc,"FILE_SIZE");
    if ((record_len==-1)||(file_size==-1)) {
      sprintf(uout,"No RECORD_LEN or FILE_SIZE");
      return (0);
    }
    for(i=0;i<file_size/record_len;i++) {
      /*
      sprintf(buf,"%s %02x",(i==0)?"ReadFirstRecord":"ReadNextRecord",
	                        record_len);
      */
      sprintf(buf,"ReadAbsRecord %02x %02x",i+1,record_len);
      if (SC_ExecuteCommand(sc,buf,rspbuf,1)) {
	/* skip any records we fail to read correctly for the moment */
	if (SC_LAST_RSP_CODE(sc)!=SC_RSP_OK) {
	  if (sc->slog)
	    SLOG_printf(sc->slog,"gsm_phonelist: skipped record %d\n",i);
	  continue;
	}

        /* nice var */
	data=sc->last_rsp.data;

	/* we have the data - parse it */
	for(j=0;j<record_len;j++) {
	  if ((unsigned char)(data[j])!=(unsigned char)0xff)
	    break;
	}
	/* if all 0xff then it is a blank record and we have finished
	 * parsing things now
	 */
	if (j==record_len) 
	  break;

	/* parse the data ... */
	memcpy(name,data,18);
	for(j=0;j<18;j++) {
	  if ((unsigned char)name[j]==(unsigned char)0xff) {
	    name[j]='\0';
	    break;
	  }
	}

	plen=record_len-18;
	if (plen>sizeof(phone))
	  plen=sizeof(phone);
	memcpy(phone,data+18,plen);
	for(j=0;j<plen;j++) {
	  if ((unsigned char)phone[j]==(unsigned char)0xff) {
	    phone[j]='\0';
	    plen=j;
	    break;
	  }
	}

        k=0;
	/* see if we have the code that indicates a "+" on the front
	 * that is probably documented somewhere in GSM stuff that I've
	 * not yet found ...
	 */
	if ((unsigned char)phone[1]==(unsigned char)0x91)
	  phoneascii[k++]='+';

	/* inverse the nibbles */
	for(j=2;j<plen;j++) {
	  val=(phone[j] & 0x0f);
	  if (val>9)
	    break;
	  phoneascii[k++]='0'+val;
	  val=((phone[j] >> 4) & 0x0f);
	  if (val>9)
	    break;
	  phoneascii[k++]='0'+val;
	}
	phoneascii[k]='\0';

	/* if we have an output channel ... */
	if (sc->sout) {
	  SLOG_printf(sc->sout,"phone[%d] %s %s\n",i+1,name,phoneascii);
	  SLOG_flush(sc->sout);
	}

	/*
	SC_ExecuteCommand(sc,"!ascii",rspbuf,0);
	*/
      } else {
      }
    }
    /* done */
    sprintf(uout,"ENTRIES %d",i);
    SC_ClearVars(sc);
    return(1);
  } else {
    return(0);
  }
}

SC_CMD_INTERNAL gsm_icmds[]={ 
{ "gsm_phonelist", NULL, gsm_phonelist },
};
#define _N_gsm_icmds (sizeof(gsm_icmds)/sizeof(gsm_icmds[0]))
int N_gsm_icmds=_N_gsm_icmds;

SC_CMD_TABLE gsm_cmd_table=
{ 
	SC_CARD_GSM, "gsm", 
	gsm_cmds, &N_gsm_cmds,
	gsm_icmds, &N_gsm_icmds,
        gsm_errs, &N_gsm_errs,
	gsm_pdata, &N_gsm_pdata
};

